!*********************************************************************************************************************************************************
!>
!>  Calculate XCLASS spectrum
!>
!>  Copyright (C) 2016 - 2025  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine ExpSave:                   returns the exponential function by avoiding Floating-point exception
!>      - subroutine PhenContDescription:       computes phen. continuum description
!>      - subroutine sort:                      sorts an array containing real numbers into ascending numerical order
!>      - subroutine calcxclass:                calculate XCLASS spectrum
!>
!>
!>
!>  Compile with:    f2py -c --f90flags="-O3 -ffree-form -ffree-line-length-none -fbounds-check -frange-check -Wall" -m XCLASS calcxclass.f90
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    2016-04-18  initial version
!>  T. Moeller    2017-11-16  add RRL handling and prepare handling of continuum contributions
!>  T. Moeller    2019-01-23  add background and dust file handling
!>  T. Moeller    2019-09-03  add local-overlap handling
!>
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> function: ExpSave
!>
!> returns the exponential function by avoiding Floating-point exception
!>
!>
!> input variables:     argument of exponential function
!>
!> output variables:    value exponential function for given argument which does not produce an overflow
!>
!>
!> \author Thomas Moeller
!>
!> \date 2017-12-20
!>
subroutine ExpSave(ExpSaveValue, arg)

    implicit none
    real*8 :: arg                                                                           !< argument
    real*8 :: ExpSaveValue                                                                  !< calculated function value


    !< compute function value
    ExpSaveValue = dexp(dmin1(arg, 700.d0))


    !< we're done
    return
end subroutine ExpSave
!*********************************************************************************************************************************************************


!>********************************************************************************************************************************************************
!> subroutine: PhenContDescription
!>
!> computes phen. continuum description
!>
!>
!> input variables:     j_back:                         brightness for background temperature
!>                      freq:                           current frequency
!>                      freq_t:                         current frequency as temperature
!>                      LocalContPhen:                  parameter vector for phen. description
!>
!> output variables:    j_back:                         brightness for background temperature
!>
!>
!> \author Thomas Moeller
!>
!> \date   2018-12-14
!>
subroutine PhenContDescription(j_back, freq, freq_t, LocalContPhen)

    implicit none
    real*8 :: j_back                                                                        !< brightness for background temperature
    real*8 :: freq                                                                          !< current frequency
    real*8 :: freq_t                                                                        !< current frequency as temperature
    real*8 :: ExpSaveValue                                                                  !< calculated function value
    real*8, dimension(6) :: LocalContPhen                                                   !< parameter vector for phen. description

    ! Debug:
    ! print*,"j_back = ", j_back
    ! print*,"freq = ", freq
    ! print*,"LocalContPhen(1) = ", LocalContPhen(1)
    ! print*,"LocalContPhen(2) = ", LocalContPhen(2)
    ! print*,"LocalContPhen(3) = ", LocalContPhen(3)
    ! print*,"LocalContPhen(4) = ", LocalContPhen(4)
    ! print*,"LocalContPhen(5) = ", LocalContPhen(5)
    ! print*,"LocalContPhen(6) = ", LocalContPhen(6)


    !< model standing wave
    if (LocalContPhen(1) == 1.d0) then
        j_back = j_back + (LocalContPhen(2) * dsin((freq * LocalContPhen(3)) + LocalContPhen(4)) &
                        + LocalContPhen(5) * dcos((freq * LocalContPhen(3)) + LocalContPhen(6)))


    !< add black body contribution
    elseif (LocalContPhen(1) == 2.d0) then
        call ExpSave(ExpSaveValue, freq_t / LocalContPhen(2))
        j_back = j_back + (freq_t / (ExpSaveValue - 1.d0))


    !< add parameterized continuum description using T_Back and T_Slope
    elseif (LocalContPhen(1) == 3.d0) then
        if (LocalContPhen(5) == 0.d0 .or. (LocalContPhen(2) <= freq .and. freq <= LocalContPhen(5))) then
            j_back = j_back + (LocalContPhen(3) * (freq / LocalContPhen(2))**LocalContPhen(4))
        endif
    endif

    ! Debug:
    ! print*,"j_back = ", j_back


    !< we're done
    return
end subroutine PhenContDescription
!*********************************************************************************************************************************************************


!>********************************************************************************************************************************************************
!> subroutine: sort
!>
!> sorts an array containing real numbers into ascending numerical order
!>
!>
!> input variables:     n:                  length of array ra
!>                      ra:                 array, which should be sort
!>
!> output variables:    ra:                 sorted array
!>
!>
!> \author Numerical Recipies, modifications by Thomas Moeller
!>
!> \date 2009-09-18
!>
subroutine sort(n, ra)
    !< Sorts an array ra(1:n) into ascending numerical order using the Heapsort algorithm. n is
    !< input; ra is replaced on output by its sorted rearrangement.

    implicit none
    integer :: n                                                                !< length of list ra
    integer :: i, ir, j, l                                                      !< working variables
    real*8 :: rra                                                               !< working array
    real*8, dimension(n) :: ra                                                  !< input/output variable

    if (n < 2) return                                                           !< if array is only one entry: do not sort

    !< The index l will be decremented from its initial value down to 1 during the 'hiring' (heap
    !< creation) phase. Once it reaches 1, the index ir will be decremented from its initial value
    !< down to 1 during the 'retirement-and-promotion' (heap selection) phase.

    l = n/2 + 1
    ir = n
    !10 continue
    Do
        if (l > 1) then                                                         !< Still in hiring phase.
            l = l - 1
            rra = ra(l)
        else                                                                    !< In retirement-and-promotion phase.
            rra = ra(ir)                                                        !< Clear a space at end of array.
            ra(ir) = ra(1)                                                      !< Retire the top of the heap into it.
            ir = ir - 1                                                         !< Decrease the size of the corporation.
            if (ir == 1) then                                                   !< Done with the last promotion.
                ra(1) = rra                                                     !< The least competent worker of all!
                !return
                exit
            endif
        endif
        i = l                                                                   !< Whether in the hiring phase or promotion phase, we here
        j = l + l                                                               !< set up to sift down element rra to its proper level.
     20 if (j <= ir) then                                                       !< 'Do while j<=ir:'
            if (j < ir) then
                if (ra(j) < ra(j + 1)) j = j + 1                                !< Compare to the better underling.
            endif
            if (rra < ra(j)) then                                               !< Demote rra.
                ra(i) = ra(j)
                i = j
                j = j + j
            else                                                                !< This is rras level. Set j to terminate the sift-down.
                j = ir + 1
            endif
            goto 20
        endif
        ra(i) = rra                                                             !< Put rra into its slot.
    end Do
    return
    ! goto 10
end subroutine sort
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> subroutine: calcxclass
!>
!> calculate XCLASS spectrum
!>
!> input variables:     NumEmissionComp:            number of emission (core) components
!>                      LocalRemoveContinuumFlag:   remove continuum flag
!>                      TelescopeSize:              size of telescope
!>                      BMIN:                       BMIN parameter
!>                      BMAJ:                       BMAJ parameter
!>                      BPA:                        BPA parameter
!>                      Inter_Flag:                 inter flag
!>                      GlobalvLSR:                 global v_lsr
!>                      Redshift:                   red shift
!>                      LowFreq:                    lowest frequency of range
!>                      tbFlag:                     flag indicating complete phen. description of continuum
!>                      TBack:                      background temperature
!>                      tSlope:                     temperature slope
!>                      nH:                         hydrogen column density
!>                      beta:                       spectral index
!>                      kappa:                      kappa for dust
!>                      LocalContPhenFuncID:        function index for phen. continuum description
!>                      LocalContPhenFuncParam1:    parameter 1 for phen. continuum description
!>                      LocalContPhenFuncParam2:    parameter 2 for phen. continuum description
!>                      LocalContPhenFuncParam3:    parameter 3 for phen. continuum description
!>                      LocalContPhenFuncParam4:    parameter 4 for phen. continuum description
!>                      LocalContPhenFuncParam5:    parameter 5 for phen. continuum description
!>                      TransFreqList:              (list of) transition frequency(ies)
!>                      EinsteinAList:              Einstein A coefficient(s)
!>                      ElowMinList:                lowest energy(ies)
!>                      gupList:                    upper state degeneracy(ies)
!>                      ScalFactor:                 list of iso ratios
!>                      MolIndexList:               list of molecule indices
!>                      FrequencyList:              list of frequency points
!>                      LocalDustFunc:              array describing dust function taken from corresponding dust file
!>                      LocalBackgroundFunc:        array describing background function taken from corresponding background file
!>                      EmsAbsFunc:                 array describing emission and absorption functions
!>                      LocalActiveCompList:        list of active components
!>                      QTList:                     list of partition function values
!>                      CurrentMolfitParameter:     current molfit parameters
!>                      NumDBParam:                 number of database parameters
!>                      NumFreqPoints:              number of frequency points to calculate
!>                      NumComp:                    number of components
!>                      NumIso:                     number of isotopologues
!>                      NumDist:                    number of distances
!>                      LocalOverlapFlag:           flag indicating if local-overlap is taken into account
!>
!> output variables:    ModeledRangeSpectrum:       array of calculated intensities
!>
!>
!> \author Thomas Moeller
!>
subroutine calcxclass(NumEmissionComp, LocalRemoveContinuumFlag, TelescopeSize, BMIN, BMAJ, BPA, Inter_Flag, GlobalvLSR, &
                      Redshift, LowFreq, tbFlag, TBack, tSlope, nH, beta, kappa, LocalContPhenFuncID, &
                      LocalContPhenFuncParam1, LocalContPhenFuncParam2, LocalContPhenFuncParam3, LocalContPhenFuncParam4, &
                      LocalContPhenFuncParam5, TransFreqList, EinsteinAList, ElowMinList, gupList, ScalFactor, &
                      MolIndexList, FrequencyList, LocalDustFunc, LocalBackgroundFunc, LocalActiveCompList, QTList, &
                      CurrentMolfitParameter, ModeledRangeSpectrum, EmsAbsFunc, NumDBParam, NumFreqPoints, NumComp, &
                      NumIso, NumDist, LocalOverlapFlag)

    implicit none
    !< **************************************************************** input parameters ****************************************************************
    integer :: NumDBParam                                                                   !< INPUT: number of transition frequencies
    !F2PY INTENT(IN) :: NumDBParam
    integer :: NumFreqPoints                                                                !< INPUT: number of frequency points
    !F2PY INTENT(IN) :: NumFreqPoints
    integer :: NumComp                                                                      !< INPUT: number of components
    !F2PY INTENT(IN) :: NumComp
    integer :: NumIso                                                                       !< INPUT: number of isotopologues
    !F2PY INTENT(IN) :: NumIso
    integer :: NumDist                                                                      !< INPUT: number of distances
    !F2PY INTENT(IN) :: NumDist
    integer :: NumEmissionComp                                                              !< INPUT: number of core components
    !F2PY INTENT(IN) :: NumEmissionComp
    integer :: tbFlag                                                                       !< INPUT: t_back_flag
    !F2PY INTENT(IN) :: tbFlag
    integer :: Inter_Flag                                                                   !< INPUT: flag for interferometric observation
    !F2PY INTENT(IN) :: Inter_Flag
    integer :: LocalRemoveContinuumFlag                                                     !< INPUT: flag for removing continuum
    !F2PY INTENT(IN) :: LocalRemoveContinuumFlag
    integer, dimension(NumComp) :: LocalActiveCompList                                      !< INPUT: list of active flags
    !F2PY INTENT(IN) :: LocalActiveCompList
    integer, dimension(NumDBParam) :: MolIndexList                                          !< INPUT: list of molecule indices
    !F2PY INTENT(IN) :: MolIndexList
    real*8 :: TelescopeSize                                                                 !< INPUT: size of telescope
    !F2PY INTENT(IN) :: TelescopeSize
    real*8 :: BMIN                                                                          !< INPUT: BMIN parameter
    !F2PY INTENT(IN) :: BMIN
    real*8 :: BMAJ                                                                          !< INPUT: BMAJ parameter
    !F2PY INTENT(IN) :: BMAJ
    real*8 :: BPA                                                                           !< INPUT: BPA parameter
    !F2PY INTENT(IN) :: BPA
    real*8 :: GlobalvLSR                                                                    !< INPUT: global v_LSR
    !F2PY INTENT(IN) :: GlobalvLSR
    real*8 :: Redshift                                                                      !< INPUT: red shift
    !F2PY INTENT(IN) :: Redshift
    real*8 :: LocalContPhenFuncID                                                           !< INPUT: function index for phen. continuum description
    !F2PY INTENT(IN) :: LocalContPhenFuncID
    real*8 :: LocalContPhenFuncParam1                                                       !< INPUT: parameter 1 for phen. continuum description
    !F2PY INTENT(IN) :: LocalContPhenFuncParam1
    real*8 :: LocalContPhenFuncParam2                                                       !< INPUT: parameter 2 for phen. continuum description
    !F2PY INTENT(IN) :: LocalContPhenFuncParam2
    real*8 :: LocalContPhenFuncParam3                                                       !< INPUT: parameter 3 for phen. continuum description
    !F2PY INTENT(IN) :: LocalContPhenFuncParam3
    real*8 :: LocalContPhenFuncParam4                                                       !< INPUT: parameter 4 for phen. continuum description
    !F2PY INTENT(IN) :: LocalContPhenFuncParam4
    real*8 :: LocalContPhenFuncParam5                                                       !< INPUT: parameter 5 for phen. continuum description
    !F2PY INTENT(IN) :: LocalContPhenFuncParam5
    real*8 :: LowFreq                                                                       !< INPUT: lowest frequency
    !F2PY INTENT(IN) :: LowFreq
    real*8 :: TBack                                                                         !< INPUT: background temperature
    !F2PY INTENT(IN) :: TBack
    real*8 :: tSlope                                                                        !< INPUT: temperature slope
    !F2PY INTENT(IN) :: tSlope
    real*8 :: nH                                                                            !< INPUT: hydrogen column densities
    !F2PY INTENT(IN) :: nH
    real*8 :: beta                                                                          !< INPUT: spectral index
    !F2PY INTENT(IN) :: beta
    real*8 :: kappa                                                                         !< INPUT: kappa
    !F2PY INTENT(IN) :: kappa
    real*8, dimension(NumDBParam) :: TransFreqList                                          !< INPUT: list of transition frequencies
    !F2PY INTENT(IN) :: TransFreqList
    real*8, dimension(NumDBParam) :: EinsteinAList                                          !< INPUT: list Einstein A coefficients
    !F2PY INTENT(IN) :: EinsteinAList
    real*8, dimension(NumDBParam) :: ElowMinList                                            !< INPUT: list of lower energies
    !F2PY INTENT(IN) :: ElowMinList
    real*8, dimension(NumDBParam) :: gupList                                                !< INPUT: list of upper state degeneracies
    !F2PY INTENT(IN) :: gupList
    real*8, dimension(NumFreqPoints) :: FrequencyList                                       !< INPUT: list of observational frequencies
    !F2PY INTENT(IN) :: FrequencyList
    real*8, dimension(NumFreqPoints) :: LocalDustFunc                                       !< INPUT: array describing dust function
    !F2PY INTENT(IN) :: LocalDustFunc
    real*8, dimension(NumFreqPoints) :: LocalBackgroundFunc                                 !< INPUT: array describing background function
    !F2PY INTENT(IN) :: LocalBackgroundFunc
    real*8, dimension(NumFreqPoints, NumDist, 2) :: EmsAbsFunc                              !< INPUT: array describing emission and absorption functions
    !F2PY INTENT(IN) :: EmsAbsFunc
    real*8, dimension(NumDBParam) :: ScalFactor                                             !< INPUT: scale factors for column densities
    !F2PY INTENT(IN) :: ScalFactor
    real*8, dimension(NumComp, NumIso + 1) :: QTList                                        !< INPUT: partition function values for given temperatures
    !F2PY INTENT(IN) :: QTList
    real*8, dimension(NumComp, 9) :: CurrentMolfitParameter                                 !< INPUT: molfit parameters
    !F2PY INTENT(IN) :: CurrentMolfitParameter
    logical :: LocalOverlapFlag                                                             !< flag indicating if local-overlap is taken into account
    !F2PY INTENT(IN) :: LocalOverlapFlag
    !< **************************************************************** output parameters ****************************************************************
    real*8, dimension(NumFreqPoints) :: ModeledRangeSpectrum                                !< OUTPUT: calculated modeled spectrum
    !F2PY INTENT(OUT) :: ModeledRangeSpectrum
    !< **************************************************************** working parameters ***************************************************************
    integer :: i, ccc, CompID, FreqID, TransID, DistID, c2                                  !< loop indices
    integer :: etaMaxIndex                                                                  !< index of core component with biggest source size
    integer :: MolIndex                                                                     !< local molecule index
    integer :: KindOfMolecule                                                               !< kind of molecule
    integer :: NumberDistances                                                              !< number of unequal distances defined in molfit file
    integer :: NumMol                                                                       !< number of molecules / RRLs at current distance
    integer, dimension(NumComp) :: NumCompAtCurrDistance                                    !< number of components per distance
    integer, dimension(NumComp, NumComp) :: DistanceOrderingArray                           !< translation table between pure distances and comp. indices


    !< Physical and astronomical constants (CODATA 2002)
    real*8, parameter :: Tcbg    = 2.725d0                                                  !< temperature of the cosmic background (K)
    real*8, parameter :: hplanck = 6.6260963d-27                                            !< Planck constant (erg/Hz)
    real*8, parameter :: kboltz  = 1.3806505d-16                                            !< Boltzmann constant (erg/K)
    real*8, parameter :: kB      = 1.3806488d-23                                            !< Boltzmann constan (J/K)
    real*8, parameter :: eCharge = 4.8d-10                                                  !< electron charge (statcolumb)
    real*8, parameter :: eMass   = 9.10938291d-28                                           !< electron mass (g)
    real*8, parameter :: ckms    = 299792.458d0                                             !< speed of light (km/s)
    real*8, parameter :: cms     = ckms * 1.d3                                              !< speed of light (m/s)
    real*8, parameter :: MHz2Kelvin = 4.7992433484894915d-5                                 !< MHz to Kelvin conversion factor
                                                                                            !< = (h / k_B)
                                                                                            !< = (6.62606957d-34 / (1.d-6 * 1.3806488d-23) (J s) / (J/K)
                                                                                            !< = 4.7992433484894915e-05
    real*8 :: pi                                                                            !< pi
    real*8 :: EulerConstant                                                                 !< Euler constant
    real*8 :: QT                                                                            !< local partition function value
    real*8 :: SourceSize                                                                    !< local source size
    real*8 :: Temp                                                                          !< local temperature
    real*8 :: Ntot                                                                          !< local column density
    real*8 :: LocalNtot                                                                     !< local column density for each isotopologue
    real*8 :: vwidth                                                                        !< local velocity width
    real*8 :: voff                                                                          !< local velocity offset
    real*8 :: T_d                                                                           !< local dust temperature
    real*8 :: LocalnH                                                                       !< working variable: hydrogen column density for dust
    real*8 :: LocalKappa                                                                    !< working variable: kappa for dust
    real*8 :: LocalBeta                                                                     !< working variable: spectral index for dust
    real*8 :: LocalRefFreq                                                                  !< working variable: rest frequency
    real*8 :: MaxLocalDustFunc                                                              !< working variable: max. of dust file
    real*8 :: TransFreq                                                                     !< local transition frequency
    real*8 :: EinsteinA                                                                     !< local Einstein A coefficient
    real*8 :: ElowMin                                                                       !< local lower energy
    real*8 :: gup                                                                           !< local upper state degeneracy
    real*8 :: ObsFreq                                                                       !< local observation frequency (in MHz)
    real*8 :: freq_t                                                                        !< local observation frequency (in K)
    real*8 :: TelescopeFWHM                                                                 !< local telescope beam full width
    real*8 :: eta                                                                           !< local beam filling factor
    real*8 :: tau_l_em                                                                      !< working variable: sum over all taus
    real*8 :: tau_l_ab                                                                      !< working variable: sum over all taus
    real*8 :: tau_d                                                                         !< local dust opacity
    real*8 :: tau_t                                                                         !< local single line opacity
    real*8 :: tau_total                                                                     !< local total opacity
    real*8 :: vLSR                                                                          !< local v_LSR
    real*8 :: sigma                                                                         !< local sigma
    real*8 :: j_tk                                                                          !< local brightness temperature
    real*8 :: j_td                                                                          !< local dust brightness temperature
    real*8 :: SourceFunc                                                                    !< local source function
    real*8 :: jcb                                                                           !< local brightness temperature for cosmic background
    real*8 :: jBack                                                                         !< local brightness temperature for given temperature
    real*8 :: LocalInt                                                                      !< local intensity
    real*8 :: TotalLocalIntensity                                                           !< local intensity
    real*8 :: LayerIntensity                                                                !< sumed intensity of each layer
    real*8 :: Ibg                                                                           !< local background intensity
    real*8 :: etaMax                                                                        !< local max. beam filling factor
    real*8 :: ExpSaveValue                                                                  !< calculated function value
    real*8 :: PhenContiuumArray                                                             !< working variable: phen. cont.
    real*8 :: LocalDistance                                                                 !< local distance
    real*8 :: val                                                                           !< working value
    real*8, dimension(6) :: LocalContPhen                                                   !< parameter vector for phen. description
    real*8, dimension(NumComp) :: AllDistances                                              !< array describing all distances
    real*8, dimension(NumFreqPoints) :: CopyModeledRangeSpectrum                            !< working variable: calculated modeled spectrum
    logical :: DustFlag                                                                     !< flag for dust continuum contribution
    logical :: PhenContFlag                                                                 !< flag for phen. continuum contribution
    logical :: TauResetFlag                                                                 !< flag indicating that tau_em and tau_abs are cleared
    logical :: UseEmAbsFuncFlag                                                             !< flag indicating usage of emission/absorption file(s)
    logical :: LocalContFlag                                                                !< flag indicating that current componet may contain
                                                                                            !< continuum contribution
    logical :: SameSizeFlag                                                                 !< flag indicating that all components at the current
                                                                                            !< distance have the same source size
    ! Debug:
    ! print*," "
    ! print*,"NumEmissionComp = ", NumEmissionComp
    ! print*,"LocalRemoveContinuumFlag = ", LocalRemoveContinuumFlag
    ! print*,"TelescopeSize = ", TelescopeSize
    ! print*,"BMIN = ", BMIN
    ! print*,"BMAJ = ", BMAJ
    ! print*,"BPA = ", BPA
    ! print*,"Inter_Flag = ", Inter_Flag
    ! print*,"GlobalvLSR = ", GlobalvLSR
    ! print*,"Redshift = ", Redshift
    ! print*,"LowFreq = ", LowFreq
    ! print*,"tbFlag = ", tbFlag
    ! print*,"TBack = ", TBack
    ! print*,"tSlope = ", tSlope
    ! print*,"nH = ", nH
    ! print*,"beta = ", beta
    ! print*,"kappa = ", kappa
    ! print*,"LocalContPhenFuncID = ", LocalContPhenFuncID
    ! print*,"LocalContPhenFuncParam1 = ", LocalContPhenFuncParam1
    ! print*,"LocalContPhenFuncParam2 = ", LocalContPhenFuncParam2
    ! print*,"LocalContPhenFuncParam3 = ", LocalContPhenFuncParam3
    ! print*,"LocalContPhenFuncParam4 = ", LocalContPhenFuncParam4
    ! print*,"LocalContPhenFuncParam5 = ", LocalContPhenFuncParam5
    ! print*,"TransFreqList = ", TransFreqList
    ! print*,"EinsteinAList = ", EinsteinAList
    ! print*,"ElowMinList = ", ElowMinList
    ! print*,"gupList = ", gupList
    ! print*,"ScalFactor = ", ScalFactor
    ! print*,"MolIndexList = ", MolIndexList
    ! print*,"FrequencyList = ", FrequencyList
    ! print*,"LocalDustFunc = ", LocalDustFunc
    ! print*,"LocalBackgroundFunc = ", LocalBackgroundFunc
    ! print*,"EmsAbsFunc = ", EmsAbsFunc
    ! print*,"LocalActiveCompList = ", LocalActiveCompList
    ! print*,"QTList = ", QTList
    ! print*,"CurrentMolfitParameter = ", CurrentMolfitParameter
    ! print*,"NumDBParam = ", NumDBParam
    ! print*,"NumFreqPoints = ", NumFreqPoints
    ! print*,"NumComp = ", NumComp
    ! print*,"NumIso = ", NumIso
    ! print*,"NumDist = ", NumDist
    ! print*,"LocalOverlapFlag = ", LocalOverlapFlag


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< determine some parameters
    pi = 4.d0 * datan(1.d0)                                                                 !< determine pi
    EulerConstant = dexp(0.557d0)                                                           !< Euler constant
    MaxLocalDustFunc = maxval(LocalDustFunc(:))                                             !< get max. value of dust file

    ! Debug:
    ! print*,"MaxLocalDustFunc = ", MaxLocalDustFunc


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< analyze distances defined in molfit file


    !< determine minimal set of distances in molfit file
    AllDistances = 0.d0
    Do CompID = 1, NumComp                                                                  !< loop over all components defined in molfit file
        if (LocalActiveCompList(CompID) == 1) then
            LocalDistance = CurrentMolfitParameter(CompID, 9)
            Do i = 1, NumComp                                                               !< loop over all components
                if (AllDistances(i) == 0.d0) then
                    AllDistances(i) = LocalDistance
                    exit
                elseif (dabs(AllDistances(i) - LocalDistance) < 1.d-9) then
                    exit
                endif
            end Do                                                                          !< i: loop over all components
        endif
    end Do                                                                                  !< CompID: loop over all components

    ! Debug:
    ! print*,"AllDistances = ", AllDistances


    !< sort distances in descending order
    AllDistances = (-1.d0) * AllDistances                                                   !< we need the distances in descending order
    call sort(NumComp, AllDistances)
    AllDistances = dabs(AllDistances)                                                       !< remove negative signs

    ! Debug:
    ! print*,"AllDistances = ", AllDistances


    !< identify components for each distance
    NumberDistances = 0
    DistanceOrderingArray = 0
    NumCompAtCurrDistance = 0
    Do DistID = 1, NumComp                                                                  !< loop over all possible distances
        if (AllDistances(DistID) < tiny(1.d0)) then
            exit
        else
            NumberDistances = NumberDistances + 1                                           !< increase counter for non-zero distances
            i = 0
            Do CompID = 1, NumComp                                                          !< loop over all components
                if (LocalActiveCompList(CompID) == 1) then
                    LocalDistance = CurrentMolfitParameter(CompID, 9)                       !< get current distance
                    if (dabs(AllDistances(DistID) - LocalDistance) < 1.d-9) then
                        i = i + 1
                        DistanceOrderingArray(DistID, i) = CompID
                    endif
                endif
            end Do                                                                          !< CompID: loop over all components
            NumCompAtCurrDistance(DistID) = i
        endif

        ! Debug:
        ! print*,"DistID, DistanceOrderingArray(DistID, :) = ", DistID, DistanceOrderingArray(DistID, :)
        ! print*,"NumCompAtCurrDistance(DistID) = ", NumCompAtCurrDistance(DistID)
    end Do                                                                                  !< DistID: loop over all possible distances

    ! Debug:
    ! print*,"NumberDistances = ", NumberDistances


    !<====================================================================================================================================================
    !< start determine XCLASS spectrum
    ModeledRangeSpectrum = 0.d0
    CopyModeledRangeSpectrum = 0.d0


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< start loop over all frequencies
    Do FreqID = 1, NumFreqPoints                                                            !< loop over frequencies
        ObsFreq = FrequencyList(FreqID)
        freq_t = MHz2Kelvin * ObsFreq                                                       !< convert frequency in temperature (Kelvin)
                                                                                            !< = \frac{h \nu}{k_B}
        ! Debug:
        ! print*,"ObsFreq = ", ObsFreq


        !<------------------------------------------------------------------------------------------------------------------------------------------------
        !< prepare beam filling factor


        !< define size of beam
        if (Inter_Flag == 1) then
            TelescopeSize = dabs(TelescopeSize)
            if (TelescopeSize < tiny(1.d0)) then
                TelescopeFWHM = (BMIN + BMAJ) / 2.d0
            else
                TelescopeFWHM = TelescopeSize
            endif
        else
            TelescopeFWHM = 1.22d-3 * ckms / (ObsFreq * TelescopeSize) * (180.d0 * 3600.d0 / pi)
        endif

        ! Debug:
        ! print*,"TelescopeFWHM = ", TelescopeFWHM


        !<------------------------------------------------------------------------------------------------------------------------------------------------
        !< determie J_CMB
        jcb = freq_t / (dexp(freq_t / Tcbg) - 1.d0)

        ! Debug:
        ! print*,"jcb = ", jcb


        !<------------------------------------------------------------------------------------------------------------------------------------------------
        !< analyze all layers at current frequency
        UseEmAbsFuncFlag = .true.
        Do DistID = 1, NumberDistances                                                      !< loop over all distances

            ! Debug:
            ! if (DistID == 1) print*,"FreqID, DistID, ObsFreq, EmsAbsFunc(FreqID, DistID, 1) = ", FreqID, DistID, ObsFreq, EmsAbsFunc(FreqID, DistID, 1:2)


            !< initialize some parameters
            tau_l_em = EmsAbsFunc(FreqID, DistID, 1) !/ 3
            tau_l_ab = EmsAbsFunc(FreqID, DistID, 2) !/ 3
            tau_d = 0.d0
            j_td = 0.d0
            PhenContiuumArray = 0.d0


            !< determine core component with biggest source size
            etaMax = 0.d0
            etaMaxIndex = 0
            SameSizeFlag = .true.
            NumMol = 0
            val = 0.d0
            Do ccc = 1, NumCompAtCurrDistance(DistID)                                       !< loop over all possible components at current distance
                CompID = DistanceOrderingArray(DistID, ccc)                                 !< get component index
                if (LocalActiveCompList(CompID) == 1) then                                  !< check, if current component is active
                    KindOfMolecule = int(CurrentMolfitParameter(CompID, 1))                 !< get kind of molecule
                    if (KindOfMolecule <= 2) then                                           !< check only for molecules and RRLs
                        SourceSize = CurrentMolfitParameter(CompID, 2)                      !< get source size
                        NumMol = NumMol + 1                                                 !< increase counter for molecules / RRLs


                        !< check, if all components have the same size
                        if (LocalOverlapFlag) then                                          !< check only, if local-overlap is taken into account
                            if (dabs(SourceSize) > tiny(1.d0) .and. SameSizeFlag) then
                                if (dabs(val - SourceSize) < tiny(1.d0) .or. dabs(val) < tiny(1.d0)) then
                                    val = SourceSize
                                else
                                    SameSizeFlag = .false.
                                    exit
                                endif
                            endif


                        !< determine component with largest source size
                        else                                                                !< continue here, if local-overlap is not taken into account
                            eta = SourceSize**2 / (TelescopeFWHM**2 + SourceSize**2)        !< compute beam filling factor
                            if (eta > etaMax) then                                          !< store beam filling factor and corresponding index
                                etaMax = eta
                                etaMaxIndex = CompID
                            endif
                        endif
                    endif
                endif
            end Do                                                                          !< ccc: loop over all possible components at current distance

            ! Debug:
            ! print*,"SameSizeFlag = ", SameSizeFlag
            ! print*,"NumMol = ", NumMol
            ! print*,"etaMaxIndex, etaMax = ", etaMaxIndex, etaMax


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< analyze all components at the current distance
            CopyModeledRangeSpectrum(:) = ModeledRangeSpectrum(:)
            TauResetFlag = .false.                                                          !< initialize reset flag for tau_em and tau_abs
            LayerIntensity = 0.d0                                                           !< reset local intensity
            Do ccc = 1, NumCompAtCurrDistance(DistID)                                       !< loop over all possible components at current distance
                CompID = DistanceOrderingArray(DistID, ccc)                                 !< get component index


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< continue with active components
                if (LocalActiveCompList(CompID) == 1) then                                  !< is component active
                    KindOfMolecule = int(CurrentMolfitParameter(CompID, 1))                 !< <=0: normal LTE molecule
                                                                                            !< =1:  RRL in LTE
                                                                                            !< =3:  dust contiuum
                                                                                            !< =6:  phen. contiuum description
                    SourceSize = CurrentMolfitParameter(CompID, 2)                          !< get source size
                    Temp = CurrentMolfitParameter(CompID, 3)                                !< get temperature

                    ! Debug:
                    ! print*," "
                    ! print*,"DistID, ccc, CompID = ", DistID, ccc, CompID
                    ! print*,"CurrentMolfitParameter(CompID,:) = ", CurrentMolfitParameter(CompID, :)
                    ! print*,"KindOfMolecule = ", KindOfMolecule
                    ! print*,"SourceSize = ", SourceSize
                    ! print*,"Temp = ", Temp


                    !< define reset flag for tau arrays
                    if (KindOfMolecule <= 2) then
                        if (.not. LocalOverlapFlag) then                                    !< reset only if local overlap is not taken into account
                            if (ccc == 1) then                                              !< always reset the arrays for the first component
                                TauResetFlag = .true.
                            elseif (NumCompAtCurrDistance(DistID) > 1 .and. &
                                maxval(DistanceOrderingArray(DistID, 1:NumCompAtCurrDistance(DistID))) > 2) then
                                c2 = DistanceOrderingArray(DistID, ccc - 1)                 !< get previous component index
                                if (int(CurrentMolfitParameter(c2, 1)) <= 2) then           !< if previous component was also a molecule reset tau arrays
                                    TauResetFlag = .true.
                                endif
                            endif
                        endif
                    endif


                    !< if local-overlap is not taken into account, check, if optical depth arrays tau_em and tau_abs have to be reseted
                    if (TauResetFlag) then
                        tau_l_em = 0.d0                                                     !< initialize tau_l_em variable
                        tau_l_ab = 0.d0                                                     !< initialize tau_l_ab variable
                        if (UseEmAbsFuncFlag) then
                            tau_l_em = EmsAbsFunc(FreqID, DistID, 1)
                            tau_l_ab = EmsAbsFunc(FreqID, DistID, 2)
                        endif
                    endif


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< get background intensity
                    if (DistID == 1) then
                        TotalLocalIntensity = LocalBackgroundFunc(FreqID)
                    else
                        TotalLocalIntensity = CopyModeledRangeSpectrum(FreqID)
                    endif

                    ! Debug:
                    ! print*,"TotalLocalIntensity = ", TotalLocalIntensity


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine beam filling (dilution) factor
                    if (LocalOverlapFlag .and. SameSizeFlag) then
                        eta = SourceSize**2 / (TelescopeFWHM**2 + SourceSize**2)
                    elseif (LocalOverlapFlag .or. dabs(SourceSize) < tiny(1.d0)) then
                        eta = 1.d0
                    else
                        eta = SourceSize**2 / (TelescopeFWHM**2 + SourceSize**2)
                    endif

                    ! Debug:
                    ! print*," "
                    ! print*,"LocalOverlapFlag = ", LocalOverlapFlag
                    ! print*,"SameSizeFlag = ", SameSizeFlag
                    ! print*,"eta = ", eta


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< compute brightness temperature
                    j_tk = freq_t / (dexp(freq_t / Temp) - 1.d0)

                    ! Debug:
                    ! print*,"j_tk = ", j_tk


                    !<====================================================================================================================================
                    !< continuum contribution


                    !< define local continuum flag
                    LocalContFlag = .false.
                    if ((LocalOverlapFlag .and. ccc == NumCompAtCurrDistance(DistID)) &
                        .or. ((.not. LocalOverlapFlag ) .and. (CompID == etaMaxIndex))) then
                        LocalContFlag = .true.
                    endif

                    ! Debug:
                    ! print*,"LocalContFlag = ", LocalContFlag


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< dust continuum
                    DustFlag = .false.


                    !< dust continuum is described in molfit file by current component
                    if (KindOfMolecule == 3) then                                           !< we are dealing with dust continuum
                        T_d = Temp
                        LocalnH = CurrentMolfitParameter(CompID, 4)
                        LocalKappa = CurrentMolfitParameter(CompID, 5)
                        LocalBeta = CurrentMolfitParameter(CompID, 6)
                        LocalRefFreq = 2.3d5                                                !< future: LocalRefFreq = CurrentMolfitParameter(CompID, 7)
                        DustFlag = .true.


                    !< dust continuum is defined in obs. xml file
                    ! elseif (LocalContFlag .and. ((kappa > tiny(1.d0) .and. nH > tiny(1.d0)) .or. MaxLocalDustFunc > tiny(1.d0))) then
                    elseif (LocalContFlag .and. DistID == 1 .and. ((kappa > tiny(1.d0) &
                        .and. nH > tiny(1.d0)) .or. MaxLocalDustFunc > tiny(1.d0))) then
                        T_d = Temp
                        LocalnH = nH
                        LocalKappa = kappa
                        LocalBeta = beta
                        LocalRefFreq = 2.3d5                                                !< future: LocalRefFreq = CurrentMolfitParameter(CompID, 7)
                        DustFlag = .true.
                    endif


                    !< consider dust contribution
                    if (DustFlag) then


                        !< compute optical depth of dust
                        LocalKappa = LocalKappa * (2.d0 * 1.66d-24 / 100.d0)
                        tau_d = LocalnH * LocalKappa * (ObsFreq / LocalRefFreq)**LocalBeta + LocalDustFunc(FreqID)


                        !< determine brightness for dust temperature
                        j_td = 0.d0
                        if (T_d /= 0.d0) then
                            call ExpSave(ExpSaveValue, freq_t / T_d)
                            if (dabs((ExpSaveValue - 1.d0)) > tiny(1.d0)) then
                                j_td = freq_t / (ExpSaveValue - 1.d0)
                            endif
                        endif

                        ! Debug:
                        ! print*,"tau_d = ", tau_d
                        ! print*,"j_td = ", j_td


                        !< compute contribution to source function
                        tau_l_em = tau_l_em + tau_d * j_td
                        tau_l_ab = tau_l_ab + tau_d
                    endif


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< phen. continuum
                    PhenContFlag = .false.


                    !< phen. continuum is defined in molfit file by current component
                    if (KindOfMolecule == 6) then
                        LocalContPhen(1:6) = CurrentMolfitParameter(CompID, 3:8)
                        PhenContFlag = .true.


                    !< phen. continuum is defined in obs. xml file
                    elseif (LocalContFlag .and. LocalContPhenFuncID > 0.d0) then
                        LocalContPhen(1) = LocalContPhenFuncID
                        LocalContPhen(2) = LocalContPhenFuncParam1
                        LocalContPhen(3) = LocalContPhenFuncParam2
                        LocalContPhen(4) = LocalContPhenFuncParam3
                        LocalContPhen(5) = LocalContPhenFuncParam4
                        LocalContPhen(6) = LocalContPhenFuncParam5
                        PhenContFlag = .true.
                    endif


                    !< consider phen. continuum description
                    if (PhenContFlag) then
                        jBack = 0.d0
                        call PhenContDescription(jBack, Obsfreq, freq_t, LocalContPhen)
                        PhenContiuumArray = PhenContiuumArray + jBack
                    endif


                    !<====================================================================================================================================
                    !< calculate line opacity for molecules and RRLs
                    if (KindOfMolecule <= 2) then                                           !< continue here for molecules and RRLs
                        Ntot = CurrentMolfitParameter(CompID, 4)
                        vwidth = CurrentMolfitParameter(CompID, 5)
                        voff = CurrentMolfitParameter(CompID, 6) + GlobalvLSR
                        Do TransID = 1, NumDBParam                                          !< loop over transitions
                            MolIndex = MolIndexList(TransID)
                            ! if (abs(KindOfMolecule) == MolIndex .or. KindOfMolecule == 1) then
                            if (1 == 1) then
                                TransFreq = TransFreqList(TransID)
                                EinsteinA = EinsteinAList(TransID)
                                ElowMin = ElowMinList(TransID)
                                gup = gupList(TransID)


                                !< get partition function
                                QT = QTList(CompID, MolIndex + 1)

                                ! Debug:
                                ! print*,"TransID, CompID, MolIndex + 1, QT = ", TransID, CompID, MolIndex + 1, QT


                                !< scale column densities for different isotopologues
                                if (dabs(ScalFactor(TransID)) > 1.d-30) then
                                    LocalNtot = Ntot / ScalFactor(TransID)
                                else
                                    LocalNtot = Ntot
                                endif


                                !< calculate vLSR
                                vLSR = -voff / ckms * TransFreq

                                ! Debug:
                                ! print*,"vLSR = ", vLSR


                                !< calculate sigma
                                sigma = (vwidth / ckms * (TransFreq + vLSR)) / (2.d0 * dsqrt(2.d0 * dlog(2.d0)))

                                ! Debug:
                                ! print*,"sigma = ", sigma


                                !< determine tau_line
                                !< conversion factor: [\nu] = MHz (10^{-6}),
                                !<                    [N_{\rm tot}^{m,c}] = cm-2 (10^{+4}),
                                !<                    [A_{u,l}] = s^(-1)
                                !<  tau = tau * 10^{-6} * 10^{-6} * 10^{-4}
                                !<      = 10^{-8} s^(-1) (phi is missing here)
                                !<      = 10^{-14} s^(-1) (with phi (in MHz^(-1)))
                                if (QT /= 0.d0 .or. KindOfMolecule == 1) then
                                    tau_t = 0.d0
                                    if (KindOfMolecule <= 0) then                           !< normal LTE molecule
                                        tau_t = (cms**2 / (8.0 * pi * TransFreq**2)) * EinsteinA * LocalNtot * gup &
                                                            * (dexp(-ElowMin / Temp) / QT) &
                                                            * (1.d0 - dexp(-(MHz2Kelvin * TransFreq / Temp))) &
                                                            * 1.d-8 &
                                                            * 1.d0 / (dsqrt(2.d0 * pi) * sigma * 1.d6) &
                                                            * dexp(-((ObsFreq - (TransFreq + vLSR))**2) &
                                                            / (2.d0 * sigma**2))
                                    elseif (KindOfMolecule == 1) then                       !< RRLs
                                        tau_t = (1.09911303617d-17 &                        !< (pi*h^3*e^2) / ( (2.0 * pi * me * kB)**(1.5) * me * c)
                                                            * 3.08567758149137d18 &         !< conversion factor: pc^(-1) to cm^(-1)
                                                            * LocalNtot &                   !< EM^{m,c}
                                                            * EinsteinA / Temp**(1.5) &
                                                            * dexp(ElowMin / Temp) &
                                                            * (1.d0 - dexp(-(MHz2Kelvin * TransFreq / Temp))) &
                                                            * 1.d0 &
                                                            * 1.d0 / (dsqrt(2.d0 * pi) * sigma * 1.d6) &
                                                            * dexp(-((ObsFreq - (TransFreq + vLSR))**2) &
                                                            / (2.d0 * sigma**2)))
                                    endif
                                    tau_l_em = tau_l_em + tau_t * j_tk
                                    tau_l_ab = tau_l_ab + tau_t
                                endif
                            endif
                        end Do                                                              !< TransID: loop over transitions
                    endif


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< check, if intensity is calculated now
                    if (ccc == NumCompAtCurrDistance(DistID) .or. (.not. LocalOverlapFlag .and. KindOfMolecule <= 2)) then


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< calculate total opacity
                        tau_total = tau_l_ab

                        ! Debug:
                        ! print*,"tau_total = ", tau_total


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< source function
                        SourceFunc = 0.d0
                        if (tbFlag == 1 .and. (.not. LocalOverlapFlag)) then
                            SourceFunc = j_tk
                        elseif (dabs(tau_l_ab) > tiny(1.d0)) then                           !< prevent division by zero
                            SourceFunc = (tau_l_em / tau_l_ab)
                        endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< beam-averaged continuum background temperature
                        jBack = PhenContiuumArray + jcb
                        if (dabs(TBack) > tiny(1.d0)) then
                            jBack = jBack + TBack * (ObsFreq / LowFreq)**tSlope
                        endif


                        !< define weighting factor for continuum correction
                        if (LocalOverlapFlag) then
                            i = 1
                        else
                            i = NumMol
                        endif
                        i = max0(i, 1)


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< calculate core component contribution
                        ! ExpSave(-tau_total)
                        call ExpSave(ExpSaveValue, -tau_total)
                        if (DistID == 1) then
                            LocalInt = (eta * (SourceFunc * (1.d0 - ExpSaveValue) &
                                        + (jBack + TotalLocalIntensity) * (ExpSaveValue - 1.d0))) &
                                        + (1.d0 / i) * (jBack + TotalLocalIntensity)
                                        ! + (1.d0 / i) * ((jBack - jcb) + TotalLocalIntensity)
                            LayerIntensity = LayerIntensity + LocalInt


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< calculate foreground component contribution
                        else
                            Ibg = (TotalLocalIntensity + PhenContiuumArray) / i
                            LocalInt = SourceFunc * (1.d0 - ExpSaveValue) + Ibg * ExpSaveValue
                            LayerIntensity = LayerIntensity + LocalInt
                        endif

                        ! Debug:
                        ! if (dabs(ObsFreq - 2.220195211000000127e+05) < 1.0 ) then
                        !     print*," "
                        !     print*,"FreqID, ObsFreq = ", FreqID, ObsFreq
                        !     print*,"DistID = ", DistID
                        !     print*,"CompID = ", CompID
                        !     print*,"ccc = ", ccc
                        !     print*,"SourceFunc = ", SourceFunc
                        !     print*,"j_tk = ", j_tk
                        !     print*,"Temp = ", Temp
                        !     print*,"i = ", i
                        !     print*,"jBack = ", jBack
                        !     print*,"tau_l_em = ", tau_l_em
                        !     print*,"tau_l_ab = ", tau_l_ab
                        !     print*,"Ibg = ", Ibg
                        !     print*,"ModeledRangeSpectrum(FreqID) = ", ModeledRangeSpectrum(FreqID)
                        !     print*,"LayerIntensity = ", LayerIntensity
                        !     print*,"CopyModeledRangeSpectrum(FreqID) = ", CopyModeledRangeSpectrum(FreqID)
                        !     print*,"TotalLocalIntensity = ", TotalLocalIntensity
                        !     print*,"PhenContiuumArray = ", PhenContiuumArray
                        !     print*," "
                        ! endif
                    endif
                endif
            end Do                                                                          !< ccc: loop over all possible components at current distance
            ModeledRangeSpectrum(FreqID) = LayerIntensity
        end Do                                                                              !< DistID: loop over all distances
        ModeledRangeSpectrum(FreqID) = ModeledRangeSpectrum(FreqID) - jcb
    end Do                                                                                  !< FreqID: loop over frequency channels


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< remove continuum
    if (LocalRemoveContinuumFlag == 1) then
        Do FreqID = 1, NumFreqPoints                                                        !< loop over frequency channels
            ObsFreq = FrequencyList(FreqID)
            ModeledRangeSpectrum(FreqID) = ModeledRangeSpectrum(FreqID) - TBack * (ObsFreq / LowFreq)**tSlope
        end Do                                                                              !< FreqID: loop over frequency channels
    endif

    ! Debug:
    ! print*,"ModeledRangeSpectrum = ", ModeledRangeSpectrum


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< return to GUI
    return
end subroutine calcxclass
!*********************************************************************************************************************************************************

