!*********************************************************************************************************************************************************
!>  Module: inter
!>
!>
!>  This module contains the core subroutines for the XCLASS package
!>  Copyright (C) 2021 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - function NumberQ:                             function is true, if the string is a number
!>      - subroutine ReadXMLInstanceFile:               read MAGIX instance file containing all required user parameters
!>      - subroutine initxclass:                        initialize XCLASS program, i.e. import input files, import molecular
!>                                                      parameters, partition functions etc.
!>      - subroutine spectrum:                          calculate XCLASS spectrum
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    2021-12-07  initial version
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module inter
    !> declare variables which are used by functions and subroutines of this module

    use myXCLASSCore
    ! use, intrinsic :: iso_c_binding


    implicit none


    !< variables for output
    real*8, allocatable, dimension(:) :: FitParameterVector                                 !< fit parameter vectors
    real*8, allocatable, dimension(:) :: ModelFuncList                                      !< contains synthetic spectrum


    !<====================================================================================================================================================
    !< variables from module Variables
    integer :: InitNumberFreeParameter                                                      !< number of fit parameters
    integer, allocatable, dimension(:) :: Initlengthexpdata                                 !< number of lines of each experimental data
    real*8, allocatable, dimension(:, :, :) :: Initexpdatax                                 !< experimental data x
    real*8, allocatable, dimension(:, :, :) :: Initexpdatay                                 !< experimental data y
    real*8, allocatable, dimension(:, :, :) :: Initexpdatae                                 !< error of the experimental data


    !<====================================================================================================================================================
    !< variables from module FunctionCalling
    ! integer :: LocalErrChannelIndex                                                         !< (no initialization needed) local error channel index
    ! integer :: LocalErrChannel                                                              !< (no initialization needed) local error channel


    !<====================================================================================================================================================
    !< variables from module myXCLASSCore


    !< general variables
    integer, parameter :: MaxNumMol = 1000                                                  !< max. number of molecules
    integer, parameter :: MaxNumRange = 1000                                                !< max. number of freq. ranges
    integer :: NumberProc                                                                   !< number of used cores
    integer :: InitErrChannelIndex                                                          !< loop variable for error handling
    integer, parameter :: InitDebugChannel = 98765                                          !< channel for debugging
    integer, dimension(2) :: InitAllErrChannels = (/0, logchannel/)                         !< list of all error channels
    real*8 :: Inittt1                                                                       !< working variables for debugging: used for timing
    real*8 :: Inittt2                                                                       !< working variables for debugging: used for timing
    real*8, parameter :: Initdebug_freq = 569937.147d0                                      !< frequency for debugging
    logical :: InitUseRegistrationXMLFile                                                   !< use registration XML file
    logical :: InitAllOutputFilesFlag                                                       !< flag for writing intensities and optical depths to file
    logical :: InitEmAbsFlag                                                                !< flag for writing emission and absorption to file
    logical :: InitIntegrationFlag                                                          !< flag indicating integration over channel width
    logical :: InitLocalOverlapFlag                                                         !< flag indicating is taken into account
    logical :: InitNoSubBeamFlag                                                            !< flag for deactivating sub-beam description
    character(len = 10) :: InitParallelizationMethod                                        !< string describing the parallelization method
    character(len = 8192) :: InitXCLASSRootDir                                              !< path of XCLASS root directory


    !< variables for molfit file
    integer :: InitTotalNumberDataPoints                                                    !< total number of data points
    integer :: InitNumberMolecules                                                          !< number of molecules in the current molfit file
    integer :: InitTotalNumberComponents                                                    !< counter for total number of components of all molecules
    integer :: InitTotalNumberOfFrequencyRanges                                             !< total number of frequency ranges
    integer :: InitTotalNumberOfMolecules                                                   !< total number of molecules including isotopologues
    integer :: InitNumberMolfitMolecules                                                    !< number of molecules in molfit file only
    integer :: InitNumberDistances                                                          !< number of unequal distances defined in molfit file
    integer :: InitNumberModelPixelXX                                                       !< number of pixel for sub-beam modeling along x-direction
    integer :: InitNumberModelPixelYY                                                       !< number of pixel for sub-beam modeling along y-direction
    integer, allocatable, dimension(:) :: InitCompMoleculeIndex                             !< molecule index for each component
    integer, allocatable, dimension(:) :: InitSpectrumIndexForFreqRange                     !< store spectrum index for each frequency range
    integer, allocatable, dimension(:) :: InitNumberComponentsPerMolecule                   !< number of components per molecule
    integer, allocatable, dimension(:) :: InitKindOfMolecule                                !< defines kind of molecule, i.e. molecule, rrl, etc.
    integer, allocatable, dimension(:) :: InitGeometryFlag                                  !< defines geometry of each component / layer
    integer, allocatable, dimension(:) :: InitLineProfileFunction                           !< defines line profile function for each component
    integer, allocatable, dimension(:, :) :: InitNumTransFreqPerObsFreq                     !< describes number of trans. freq. for each freq. channel
    integer, allocatable, dimension(:, :) :: InitConversionTableMAGIXmyXCLASSParam          !< conversion table between myXCLASS and MAGIX parameter
    integer, allocatable, dimension(:, :) :: InitDataPointIndexFreqRange                    !< index of data point of first and last freq. in freq. range
    integer, allocatable, dimension(:, :) :: InitDistanceOrderingArray                      !< translation table between pure distances and comp. indices
    integer, allocatable, dimension(:, :) :: InitConfigIndex                                !< configuration index for model map
    integer, allocatable, dimension(:, :) :: InitMoleculeNames                              !< (STRING ARRAY) names of all molecules (including isotopologues)
    real*8 :: InitSizeOfPixel_deg                                                           !< size of pixel
    real*8, allocatable, dimension(:) :: InitStartFrequency                                 !< first frequency for each frequency range
    real*8, allocatable, dimension(:) :: InitEndFrequency                                   !< last frequency for each frequency range
    real*8, allocatable, dimension(:) :: InitStepFrequency                                  !< step size for each frequency range
    real*8, allocatable, dimension(:) :: InitTelescopeSize                                  !< size of telescope for each frequency range
    real*8, allocatable, dimension(:) :: InitTelescopeBMIN                                  !< size of BMIN
    real*8, allocatable, dimension(:) :: InitTelescopeBMAJ                                  !< size of BMAJ
    real*8, allocatable, dimension(:) :: InitTelescopeBPA                                   !< size of BPA
    real*8, allocatable, dimension(:) :: InitGlobalvLSR                                     !< global velocity for each freq. range
    real*8, allocatable, dimension(:) :: InitRedshift_Range                                 !< redshift for each range
    real*8, allocatable, dimension(:) :: InitBackgroundTemperatureRange                     !< T_Back for each frequency range
    real*8, allocatable, dimension(:) :: InitTemperatureSlopeRange                          !< T_Slope for each frequency range
    real*8, allocatable, dimension(:) :: InitHydrogenColumnDensityRange                     !< nH for each frequency range
    real*8, allocatable, dimension(:) :: InitDustBetaRange                                  !< beta for each frequency range
    real*8, allocatable, dimension(:) :: InitKappaRange                                     !< kappa for each frequency range
    real*8, allocatable, dimension(:) :: InitDustTauFromFile                                !< array containing tau_dust from file
    real*8, allocatable, dimension(:) :: InitBackgroundFromFile                             !< array containing background from file
    real*8, allocatable, dimension(:) :: InitSortedDistances                                !< array containing sorted distances
    real*8, allocatable, dimension(:) :: InitPureDistances                                  !< array containing pure distances
    real*8, allocatable, dimension(:, :) :: InitContPhen_Range                              !< parameters for phenomenological continuum description
    real*8, allocatable, dimension(:, :) :: InitmyXCLASSParameter                           !< array containing all molfit parameters for each component
    real*8, allocatable, dimension(:, :) :: InitvWidthLimits                                !< lower and upper limits of velocity widths parameters
    real*8, allocatable, dimension(:, :) :: InitvOffLimits                                  !< lower and upper limits of velocity offsets parameters
    real*8, allocatable, dimension(:, :) :: InitObservationalDataList                       !< list containing all observational data
    real*8, allocatable, dimension(:, :) :: InitGausssianBeamMap                            !< map describing elliptical rotated Gaussian beam
    real*8, allocatable, dimension(:, :, :) :: InitEmsAbsFunc                               !< array describing emission and absorption function
    real*8, allocatable, dimension(:, :, :) :: InitTauEmAbsArray                            !< array containing emission and absorption function
    character(len = 4096) :: InitEmsAbsPATH                                                 !< path of directory containing files describing em. and abs.
    logical :: InitIsoFlag                                                                  !< flag indicating use of isotopologues
    logical :: InitSubBeamDescriptionFlag                                                   !< flag indicating sub-beam description
    logical :: InitUseEmAbsFuncFlag                                                         !< flag indicating usage of emission/absorption file(s)
    logical, allocatable, dimension(:) :: InittbFlag                                        !< flag indicating that T_Back and T_Slope describes
                                                                                            !< continuum completely
    logical, allocatable, dimension(:) :: InitInterFlag                                     !< flag indicating that observation is interferrometric data
    logical, allocatable, dimension(:) :: InittdFlag                                        !< flag for reading T_dOff and T_dSlope from molfit file
    logical, allocatable, dimension(:) :: InitnHFlag                                        !< flag for global setting of nH, kappa and beta
    logical, allocatable, dimension(:) :: InitLTEFlag                                       !< flag for LTE calculation
    logical, allocatable, dimension(:) :: InitThinFlag                                      !< flag indicating optical thin or thick (for Non-LTE RRLs)
    logical, allocatable, dimension(:) :: InitDust_Flag_Range                               !< flag indicating global phen. settings for each range
    logical, allocatable, dimension(:) :: InitPhen_Flag_Range                               !< flag indicating global dust settings for each range
    logical, allocatable, dimension(:, :) :: InitConfigList                                 !< list of unique pixel combination
    logical, allocatable, dimension(:, :, :) :: InitLayerMap                                !< indicates, which components are taken into account


    !< variables for sqlite3 database
    integer :: InitMinNumTransitionsSQL                                                     !< min number of transitions which are considered in SQL query
    integer :: InitMaxNumTransitionsSQL                                                     !< max number of transitions which are considered in SQL query
    integer :: InitOrderTransSQL                                                            !< order or transition considered in SQL query
    integer, allocatable, dimension(:, :) :: InitNumEntriesRanges                           !< number of transitions for each freq. range and molecule
    real*8 :: InitMaxElowSQL                                                                !< highest lower energy of transitions considered in SQL query
    real*8 :: InitMingASQL                                                                  !< min. intensity of transitions considered in SQL query
    character(len = 128) :: InitNameOfPartFuncTable                                         !< name of table for partition function
    character(len = 128) :: InitNameOfRadTransTable                                         !< name of table for rad. trans.
    character(len = 4096) :: InitdbName                                                     !< name of database


    !< variables for partition function
    integer :: InitNumberOfTemperatures                                                     !< number of temperatures
    integer :: InitNumberMoleculePartFunc                                                   !< number of entries in table PartitionFunction
    integer :: InitFirsti, InitLasti, Initstepi                                             !< working variables for interpolation
    integer, allocatable, dimension(:, :) :: InitMolNamePartFunc                            !< (STRING ARRAY) molecule names of table PartitionFunction
    integer, allocatable, dimension(:, :) :: InitColumnNamesPartFunc                        !< (STRING ARRAY) column names of table PartitionFunction
    real*8 :: InitTempLow, InitTempHigh                                                     !< working variables for extrapolation of part. func.
    real*8, allocatable, dimension(:) :: InitTempPartFunc                                   !< temperatures for partition function
    real*8, allocatable, dimension(:, :) :: InitlgQ                                         !< lgQ entries of table PartitionFunction
    character(len = 40) :: InitColumnNameForNamePartFunc                                    !< name of column including the names of molecules


    !< variables for rad-trans parameter
    integer :: InitTotalNumberOfTransitions                                                 !< number of entries in table RadTrans
    integer :: InitTotalNumTransFreq                                                        !< number of doppler-shifted trans. freq. for curr. layer
    integer :: InitColumn300KPartFunc                                                       !< number of column, which contains part. func. @ 300 K
    integer, allocatable, dimension(:, :) :: InitDopplerShiftedTransFreqIndex               !< array containing indices of doppler-shifted trans. freq.
    integer, allocatable, dimension(:, :) :: InitLowerQNString                              !< (STRING ARRAY) quantum number string of lower state
    integer, allocatable, dimension(:, :) :: InitUpperQNString                              !< (STRING ARRAY) quantum number string of upper state
    integer, allocatable, dimension(:, :, :) :: InitMolecularDataIndices                    !< start and end index for each molecule and frequency range
                                                                                            !< within MolecularData variable
    real*8, allocatable, dimension(:, :) :: InitMolecularData                               !< array containing the molecular data for all molecules and
                                                                                            !< all freqency ranges
    real*8, allocatable, dimension(:) :: InitDopplerShiftedTransFreq                        !< array containing all doppler-shifted transitions
    real*8, allocatable, dimension(:, :) :: InitCentralOpticalDepth                         !< optical depths at doppler-shifted trans. frequencies
    character(len = 40) :: InitColumnNameForNameTransitions                                 !< name of column in table transitions for molecule name
    character(len = 40) :: InitColumnNameForFreqTransitions                                 !< name of column in table transitions for frequency
    character(len = 40) :: InitColumnNameForIntTransitions                                  !< name of column in table transitions for intensity
    character(len = 40) :: InitColumnNameForEinsteinATransitions                            !< name of column in table transitions for Einstein A coef.
    character(len = 40) :: InitColumnNameForFreqErrTransitions                              !< name of column in table transitions for error freq.
    character(len = 40) :: InitColumnNameForELowTransitions                                 !< name of column in table transitions for E_low
    character(len = 40) :: InitColumnNameForgUpTransitions                                  !< name of column in table transitions for Upper State degen.
    character(len = 40) :: InitColumnNameForLowerQNTransitions                              !< name of column in table transitions for lower state QN
    character(len = 40) :: InitColumnNameForUpperQNTransitions                              !< name of column in table transitions for upper state QN


    !< variables for iso table file
    integer :: InitNumberOfIsomasters                                                       !< number of iso master molecules
    integer :: InitNumberOfGlobalISORatios                                                  !< number of global iso ratio definitions
    integer, allocatable, dimension(:, :) :: InitIsoMolecule                                !< (STRING ARRAY) names of molecules for iso table
    integer, allocatable, dimension(:, :) :: InitIsoMasters                                 !< (STRING ARRAY) names of isomasters for iso table
    integer, allocatable, dimension(:, :, :) :: InitIsoNfitConversionTable                  !< iso ratios conversion table for free parameter index
    integer, allocatable, dimension(:, :, :) :: InitGlobalIsoRatioParameter                 !< parameters for globally defined iso ratios
    real*8, allocatable, dimension(:, :) :: InitIsoRatio                                    !< iso ratios for all iso master molecules
    real*8, allocatable, dimension(:, :) :: InitIsoRatioConversionTable                     !< table with iso ratios between iso master and molecule
    character(len = 4096) :: InitIsoTableFileName                                           !< path and name of the iso file


    !< import instance xml file
    character(len = 4096) :: InstanceFileName                                               !< path and file name of the instance file
    logical :: iso_flag                                                                     !< flag for using isotopologues or not


    !< variables used for storing opacities and intensities
    real*8, allocatable, dimension(:, :, :) :: InitTauPerCompArray                          !< array containing emission and absorption function
    real*8, allocatable, dimension(:, :, :) :: InitIntPerCompArray                          !< array containing intensities for each comp.


    !< general variables
    integer :: InitErrChannel                                                               !< current error channel


    !<====================================================================================================================================================
    !< subroutines
    contains


        !*************************************************************************************************************************************************
        !> function NumberQ
        !>
        !> This function is true, if the string is a number
        !>
        !>
        !> input variables:     string:         the string which has to be analyzed
        !>
        !> output variables:    NumberQ:        is true, if string is a number
        !>
        !>
        !> \author http://rosettacode.org/wiki/Determine_if_a_string_is_numeric#Fortran
        !>
        !> \date 2015-02-06
        !>
        function NumberQ(string)
            !< returns true if the string is a number

            implicit none
            character(len = *), INTENT(IN) :: string                                        !< the string which has to be analyzed
            logical :: NumberQ                                                              !< the function value
            real :: x                                                                       !< working variable
            integer :: e                                                                    !< working variable


            read(string, *, iostat = e) x
            NumberQ = e == 0
            return
        end function NumberQ


        !>************************************************************************************************************************************************
        !> subroutine: ReadXMLInstanceFile
        !>
        !> read MAGIX instance file containing all required user parameters
        !>
        !> input variables:     ok:                 status variable
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2012-07-09
        !>
        subroutine ReadXMLInstanceFile(ok)

            implicit none
            integer :: ok                                                                   !< status variable
            integer :: NumberOfLines                                                        !< working variable: number of lines in instance file
            integer :: allocstatus, deallocstatus                                           !< working variable: status variables for (de)allocation
            integer :: i                                                                    !< loop variable
            integer :: j, k                                                                 !< working variables
            integer :: io_err                                                               !< status variable for i/o
            integer :: RangeID                                                              !< loop variable for ranges
            real*8 :: freqmin                                                               !< first frequency of the simulated spectrum
            real*8 :: freqmax                                                               !< last frequency of the simulated spectrum
            real*8 :: sim_width                                                             !< step frequency of the simulated spectrum
            real*8 :: LocalvLSR                                                             !< vLSR
            real*8 :: Redshift                                                              !< red shift
            real*8 :: telescope_size                                                        !< size of the telescope
            real*8 :: telescope_BMIN                                                        !< parameter for BMIN (telescope)
            real*8 :: telescope_BMAJ                                                        !< parameter for BMAJ (telescope)
            real*8 :: telescope_BPA                                                         !< parameter for BPA (telescope)
            real*8 :: T_back                                                                !< background temperature
            real*8 :: T_slope                                                               !< temperature slope
            real*8 :: N_H                                                                   !< hydrogen column density
            real*8 :: beta_dust                                                             !< beta for dust
            real*8 :: kappa_1300                                                            !< kappa for dust
            real*8, dimension(6) :: ContPhen                                                !< parameters for phenomenological continuum description
            character(len = 1024) :: helpstring                                             !< working variable
            character(len = 1024) :: line                                                   !< current line of file
            character(len = 4096) :: LocalDustFileName                                      !< path and name of dust file
            character(len = 4096) :: LocalBackgroundFileName                                !< path and name of background file
            logical :: LocalFitFlag                                                         !<


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< open instance file
            ok = 0
            if (printflag) then
                print '("Reading parameters from file .. ", $)'
            endif
            open(4444, file = trim(adjustl(InstanceFileName)), status = 'unknown')


            !< determine number of lines and number of parameters
            parameternumber = 0
            NumberOfLines = 0
            io_err = 0
            NumberFreeParameter = 0
            Do While (io_err == 0)
                read(4444,'(A)', iostat = io_err) line
                if (io_err == 0) then
                    NumberOfLines = NumberOfLines + 1
                    line = trim(adjustl(line))


                    !< get name of parameter
                    j = index(line, "<name>")
                    if (j > 0) then
                        parameternumber = parameternumber + 1
                    endif


                    !< counter number of free parameters
                    j = index(line, "<Parameter fit")
                    if (j > 0) then
                        j = index(line, "true")
                        if (j > 0) then
                            NumberFreeParameter = NumberFreeParameter + 1
                        endif
                    endif
                endif
            end Do
            parameternumber = parameternumber + 38 * InitTotalNumberOfFrequencyRanges

            ! Debug:
            ! print*,'NumberOfLines = ', NumberOfLines
            ! print*,'parameternumber = ', parameternumber
            ! print*,'NumberFreeParameter = ', NumberFreeParameter
            ! stop


            !< (de-)allocate memory
            if (allocated(FitParameterName)) then
                deallocate(FitParameterName, FitParameterValue, paramset, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.ReadXMLInstanceFile!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- FitParameterName")'
                    print '(5x, "- FitParameterValue")'
                    print '(5x, "- paramset")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(FitParameterName(parameternumber), &
                     FitParameterValue(parameternumber), &
                     paramset(4, parameternumber), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.ReadXMLInstanceFile!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- FitParameterName(parameternumber)")'
                print '(5x, "- FitParameterValue(parameternumber)")'
                print '(5x, "- paramset(4, parameternumber)")'
                print '(" ")'
                print '(3x, "Used parameter:")'
                print '(" ")'
                print '(5x, "- parameternumber = ", I8)', parameternumber
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            FitParameterName = ""
            FitParameterValue = ""
            paramset = 0.d0


            !< go back to the beginning of the file
            rewind 4444


            !< analyze instance file
            k = 0
            LocalFitFlag = .false.
            Do i = 1, NumberOfLines
                read(4444, '(A)', iostat = io_err) line
                line = trim(adjustl(line))


                !< check, if parameter is fitted
                j = index(line, "<Parameter fit")
                if (j > 0) then
                    j = index(line, "true")
                    if (j > 0) then
                        LocalFitFlag = .true.
                    else
                        LocalFitFlag = .false.
                    endif
                endif


                !< get name of parameter
                j = index(line, "<name>")
                if (j > 0) then
                    k = k + 1
                    helpstring = line(j + 6:)
                    j = index(helpstring, "</name>")
                    helpstring = helpstring(:j - 1)
                    FitParameterName(k) = trim(adjustl(helpstring))

                    ! Debug:
                    ! print*,'FitParameterName(k) = >>', trim(adjustl(FitParameterName(k))), '<<'
                endif


                !< get value of parameter
                j = index(line, "<value>")
                if (j > 0) then
                    helpstring = line(j + 7:)
                    j = index(helpstring, "</value>")
                    helpstring = helpstring(:j - 1)
                    FitParameterValue(k) = trim(adjustl(helpstring))

                    ! Debug:
                    ! print*,'FitParameterValue(k) = >>', trim(adjustl(FitParameterValue(k))), '<<'


                    !< define paramset variable
                    if (NumberQ(trim(FitParameterValue(k)))) then
                        read(FitParameterValue(k), *) paramset(1, k)
                        if (LocalFitFlag) then
                            paramset(2, k) = 1.0
                        endif

                        ! Debug:
                        ! print*,'FitParameterValue(k) = >>', trim(adjustl(FitParameterValue(k))), '<<'
                        ! print*,'paramset(1, k) = ', paramset(1, k)
                    endif
                endif
            end Do
            close(4444)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< add obs. xml parameters to list of parameters from instance file


            !< loop over ranges
            Do RangeID = 1, InitTotalNumberOfFrequencyRanges


                !< store "MAGIXImportExpFileNames"
                k = k + 1
                FitParameterName(k) = "MAGIXImportExpFileNames"
                FitParameterValue(k) = ""


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< store "MAGIXImportStartFrequency"
                k = k + 1
                FitParameterName(k) = "MAGIXImportStartFrequency"
                freqmin = InitStartFrequency(RangeID)
                write(FitParameterValue(k), '(F25.15)') freqmin
                paramset(1, k) = freqmin


                !< store "MAGIXImportEndFrequency"
                k = k + 1
                FitParameterName(k) = "MAGIXImportEndFrequency"
                freqmax = InitEndFrequency(RangeID)
                write(FitParameterValue(k), '(F25.15)') freqmax
                paramset(1, k) = freqmax


                !< store "MAGIXImportStepFrequency"
                k = k + 1
                FitParameterName(k) = "MAGIXImportStepFrequency"
                sim_width = InitStepFrequency(RangeID)
                write(FitParameterValue(k), '(F25.15)') sim_width
                paramset(1, k) = sim_width


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< store "MAGIXImportTBackFlag"
                k = k + 1
                FitParameterName(k) = "MAGIXImportTBackFlag"
                if (InittbFlag(RangeID)) then
                    paramset(1, k) = 1.d0
                else
                    paramset(1, k) = 0.d0
                endif


                !< store "MAGIXImportBackgroundTemperature"
                k = k + 1
                FitParameterName(k) = "MAGIXImportBackgroundTemperature"
                T_back = InitBackgroundTemperatureRange(RangeID)
                write(FitParameterValue(k), '(F25.15)') T_back
                paramset(1, k) = T_back


                !< store "MAGIXImportTemperatureSlope"
                k = k + 1
                FitParameterName(k) = "MAGIXImportTemperatureSlope"
                T_slope = InitTemperatureSlopeRange(RangeID)
                write(FitParameterValue(k), '(F25.15)') T_slope
                paramset(1, k) = T_slope


                !< store "MAGIXImportBackgroundFileName"
                k = k + 1
                FitParameterName(k) = "MAGIXImportBackgroundFileName"
                LocalBackgroundFileName = ""
                FitParameterValue(k) = trim(adjustl(LocalBackgroundFileName))
                paramset(1, k) = 0.0


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< store parameters for free-free continuum
                if (InitDust_Flag_Range(RangeID)) then


                    !< store "MAGIXImportHydrogenColumnDensity"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportHydrogenColumnDensity"
                    N_H = InitHydrogenColumnDensityRange(RangeID)
                    write(FitParameterValue(k), '(F25.15)') N_H
                    paramset(1, k) = N_H


                    !< store "MAGIXImportDustBeta"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportDustBeta"
                    beta_dust = InitDustBetaRange(RangeID)
                    write(FitParameterValue(k), '(F25.15)') beta_dust
                    paramset(1, k) = beta_dust


                    !< store "MAGIXImportKappa"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportKappa"
                    kappa_1300 = InitKappaRange(RangeID)
                    write(FitParameterValue(k), '(F25.15)') kappa_1300
                    paramset(1, k) = kappa_1300
                endif


                !< store "MAGIXImportDustFileName"
                k = k + 1
                FitParameterName(k) = "MAGIXImportDustFileName"
                LocalDustFileName = ""
                FitParameterValue(k) = trim(adjustl(LocalDustFileName))
                paramset(1, k) = 0.0


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< store parameters for phen. description of continuum
                if (InitPhen_Flag_Range(RangeID)) then


                    !< store "MAGIXImportContPhenID"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportContPhenID"
                    ContPhen(1) = InitContPhen_Range(RangeID, 1)
                    write(FitParameterValue(k), '(F25.15)') ContPhen(1)
                    paramset(1, k) = ContPhen(1)


                    !< store "MAGIXImportContParam1"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportContParam1"
                    ContPhen(2) = InitContPhen_Range(RangeID, 2)
                    write(FitParameterValue(k), '(F25.15)') ContPhen(2)
                    paramset(1, k) = ContPhen(2)


                    !< store "MAGIXImportContParam2"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportContParam2"
                    ContPhen(3) = InitContPhen_Range(RangeID, 3)
                    write(FitParameterValue(k), '(F25.15)') ContPhen(3)
                    paramset(1, k) = ContPhen(3)


                    !< store "MAGIXImportContParam3"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportContParam3"
                    ContPhen(4) = InitContPhen_Range(RangeID, 4)
                    write(FitParameterValue(k), '(F25.15)') ContPhen(4)
                    paramset(1, k) = ContPhen(4)


                    !< store "MAGIXImportContParam4"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportContParam4"
                    ContPhen(5) = InitContPhen_Range(RangeID, 5)
                    write(FitParameterValue(k), '(F25.15)') ContPhen(5)
                    paramset(1, k) = ContPhen(5)


                    !< store "MAGIXImportContParam5"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportContParam5"
                    ContPhen(6) = InitContPhen_Range(RangeID, 6)
                    write(FitParameterValue(k), '(F25.15)') ContPhen(6)
                    paramset(1, k) = ContPhen(6)
                endif


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< store "MAGIXImportGlobalvLSR"
                k = k + 1
                FitParameterName(k) = "MAGIXImportGlobalvLSR"
                LocalvLSR = InitGlobalvLSR(RangeID)
                write(FitParameterValue(k), '(F25.15)') LocalvLSR
                paramset(1, k) = LocalvLSR


                !< store "MAGIXImportRedshift"
                Redshift = InitRedshift_Range(RangeID)
                if (dabs(Redshift) > EPSILON(0.d0)) then
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportRedshift"
                    write(FitParameterValue(k), '(F25.15)') Redshift
                    paramset(1, k) = Redshift
                endif


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< store beam size
                telescope_BMIN = InitTelescopeBMIN(RangeID)
                telescope_BMAJ = InitTelescopeBMAJ(RangeID)
                telescope_BPA = InitTelescopeBPA(RangeID)
                if (dabs(telescope_BMIN) > EPSILON(0.d0) .and. dabs(telescope_BMAJ) > EPSILON(1.d0)) then


                    !< store "MAGIXImportTelescopeBMIN"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportTelescopeBMIN"
                    write(FitParameterValue(k), '(F25.15)') telescope_BMIN
                    paramset(1, k) = telescope_BMIN


                    !< store "MAGIXImportTelescopeBMAJ"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportTelescopeBMAJ"
                    write(FitParameterValue(k), '(F25.15)') telescope_BMAJ
                    paramset(1, k) = telescope_BMAJ


                    !< store "MAGIXImportTelescopeBPA"
                    k = k + 1
                    FitParameterName(k) = "MAGIXImportTelescopeBPA"
                    write(FitParameterValue(k), '(F25.15)') telescope_BPA
                    paramset(1, k) = telescope_BPA
                endif


                !< store "MAGIXImportTelescopeSize"
                k = k + 1
                FitParameterName(k) = "MAGIXImportTelescopeSize"
                telescope_size = InitTelescopeSize(RangeID)
                write(FitParameterValue(k), '(F25.15)') telescope_size
                paramset(1, k) = telescope_size


                !< store "MAGIXImportInterferrometerFlag"
                k = k + 1
                FitParameterName(k) = "MAGIXImportInterferrometerFlag"
                if (InitInterFlag(RangeID)) then
                    paramset(1, k) = 1
                else
                    paramset(1, k) = 0
                endif
            end Do                                                                          !< loop over all ranges


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< store number of model pixels for sub-beam description
            if (InitNumberModelPixelXX /= 0 .and. InitNumberModelPixelYY /= 0) then


                !< store "MAGIXImportNumModelPixelXX"
                k = k + 1
                FitParameterName(k) = "MAGIXImportNumModelPixelXX"
                write(FitParameterValue(k), '(I10)') InitNumberModelPixelXX
                paramset(1, k) = InitNumberModelPixelXX


                !< store "MAGIXImportNumModelPixelYY"
                k = k + 1
                FitParameterName(k) = "MAGIXImportNumModelPixelYY"
                write(FitParameterValue(k), '(I10)') InitNumberModelPixelYY
                paramset(1, k) = InitNumberModelPixelYY
            endif


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< store "LocalOverlapFlag"
            k = k + 1
            FitParameterName(k) = "MAGIXImportLocalOverlapFlag"
            if (InitLocalOverlapFlag) then
                paramset(1, k) = 1
            else
                paramset(1, k) = 0
            endif


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< store "NoSubBeamFlag"
            k = k + 1
            FitParameterName(k) = "MAGIXImportNoSubBeamFlag"
            if (InitNoSubBeamFlag) then
                paramset(1, k) = 1
            else
                paramset(1, k) = 0
            endif


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< store "MAGIXImportEmAbsPATH"
            k = k + 1
            FitParameterName(k) = "MAGIXImportEmAbsPATH"
            FitParameterValue(k) = trim(adjustl(InitEmsAbsPATH))
            paramset(1, k) = 0.0


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< store "MAGIXImportdbFilename"
            k = k + 1
            FitParameterName(k) = "MAGIXImportdbFilename"
            FitParameterValue(k) = trim(adjustl(InitdbName))
            paramset(1, k) = 0.0


            !< we're done
            if (printflag) then
                print '("done!")'
            endif

            ! Debug:
            ! print*,"FitParameterName = ", FitParameterName


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< store parameters for obs. data
            if (allocated(lengthexpdata)) then
                deallocate(lengthexpdata, &
                           expdatax, &
                           expdatay, &
                           expdatae, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.ReadXMLInstanceFile!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variable:")'
                    print '(5x, "- lengthexpdata")'
                    print '(5x, "- expdatax")'
                    print '(5x, "- expdatay")'
                    print '(5x, "- expdatae")'
                    print '(" ")'
                    print '(5x, "- TotalNumberDataPoints = ", I8)', TotalNumberDataPoints
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            TotalNumberDataPoints = sum(Initlengthexpdata)
            i = size(Initlengthexpdata)
            j = maxval(Initlengthexpdata)
            allocate(lengthexpdata(i), expdatax(i, j, 1), expdatay(i, j, 1), &
                     expdatae(i, j, 1), stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.ReadXMLInstanceFile!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- lengthexpdata(i)")'
                print '(5x, "- expdatax(i, j, 1)")'
                print '(5x, "- expdatay(i, j, 1)")'
                print '(5x, "- expdatae(i, j, 1)")'
                print '(" ")'
                print '(3x, "Used parameter:")'
                print '(" ")'
                print '(5x, "- i = ", I8)', i
                print '(5x, "- j = ", I8)', j
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            lengthexpdata = Initlengthexpdata
            expdatax = Initexpdatax
            expdatay = Initexpdatay
            expdatae = Initexpdatae
            NumberExpFiles = i

            ! Debug:
            ! print*,"i = ", i
            ! print*,"j = ", j


            !< we're done
            return
        end subroutine ReadXMLInstanceFile


        !*************************************************************************************************************************************************
        !> subroutine: initxclass
        !>
        !> initialize XCLASS program, i.e. import input files, import molecular parameters, partition functions etc.
        !>
        !> input variables:     None
        !>
        !> output variables:    None
        !>
        !>
        !> \author Thomas Moeller
        !>
        subroutine initxclass()

            implicit none
            integer :: ok                                                                   !< status variable
            integer :: i, j, k, l, m, k1, k2, k3, iii                                       !< working variables
            integer :: MaxNumberOfMolecules                                                 !< working variable: max. number of molecules (incl. isotop.)
            integer :: MaxNumIso                                                            !< working variable max. number of isotopologues per component
            integer :: allocstatus, deallocstatus                                           !< working variable: status variables for (de)allocation
            character(len = 8192) :: EnvironmentVariableValue                               !< working variable: environment variable


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize variables
            printflag = .false.                                                             !< define flag for screen output ( = 0: no screen output)
            ParallezitionMethod = 1                                                         !< define parallelization method used for fftw3
            JobID = 0                                                                       !< unused dummy variable


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< increase omp stack size
            EnvironmentVariableValue = ""
            call get_environment_variable("KMP_STACKSIZE", EnvironmentVariableValue)        !< get environment variable
            call getenv("KMP_STACKSIZE", EnvironmentVariableValue)
            EnvironmentVariableValue = trim(adjustl(EnvironmentVariableValue))              !< remove leading and tailing strings
            if (len_trim(EnvironmentVariableValue) == 0) then
                call get_environment_variable("OMP_STACKSIZE", EnvironmentVariableValue)    !< get environment variable
                call getenv("OMP_STACKSIZE", EnvironmentVariableValue)
                EnvironmentVariableValue = trim(adjustl(EnvironmentVariableValue))          !< remove leading and tailing strings
                if (len_trim(EnvironmentVariableValue) == 0) then
                    call system("export KMP_STACKSIZE='9999M'")
                endif
            endif
            NumberProc = 1
            call omp_set_num_threads(NumberProc)                                            !< set number of threads


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< read in xml file containing user parameters
            call ReadXMLInstanceFile(ok)



            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get partition function and transition parameters from sqlite3 database
            ParallelizationMethod = "NoMAGIX"
            IntegrationFlag = .true.
            LogFlag = .true.
            call myXCLASSInitVar
            LogFlag = .false.



            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< copy variables


            !< general variables
            InitErrChannelIndex = ErrChannelIndex                                           !< copy loop variable for error handling
            InitAllErrChannels = AllErrChannels                                             !< list of all error channels
            Inittt1 = tt1                                                                   !< working variable for debugging: used for timing
            Inittt2 = tt2                                                                   !< working variable for debugging: used for timing
            InitUseRegistrationXMLFile = UseRegistrationXMLFile                             !< use registration XML file
            InitEmAbsFlag = EmAbsFlag                                                       !< flag for writing emission and absorption to file
            InitIntegrationFlag = IntegrationFlag                                           !< flag indicating integration over channel width
            InitLocalOverlapFlag = LocalOverlapFlag                                         !< flag indicating is taken into account
            InitNoSubBeamFlag = NoSubBeamFlag                                               !< flag for deactivating sub-beam description
            InitParallelizationMethod = ParallelizationMethod                               !< string describing the parallelization method
            InitXCLASSRootDir = XCLASSRootDir                                               !< path of XCLASS root directory
            InitNumberFreeParameter = NumberFreeParameter                                   !< define number of fit parameters
            AllOutputFilesFlag = InitAllOutputFilesFlag                                     !< flag for storing intensities and optical depths
                                                                                            !< per component or distance
            ! Debug:
            ! print*,"ErrChannelIndex = ", ErrChannelIndex
            ! print*,"AllErrChannels = ", AllErrChannels
            ! print*,"tt1 = ", tt1
            ! print*,"tt2 = ", tt2
            ! print*,"UseRegistrationXMLFile = ", UseRegistrationXMLFile
            ! print*,"AllOutputFilesFlag = ", AllOutputFilesFlag
            ! print*,"EmAbsFlag = ", EmAbsFlag
            ! print*,"IntegrationFlag = ", IntegrationFlag
            ! print*,"LocalOverlapFlag = ", LocalOverlapFlag
            ! print*,"NoSubBeamFlag = ", NoSubBeamFlag
            ! print*,"ParallelizationMethod = ", ParallelizationMethod
            ! print*,"XCLASSRootDir = ", trim(adjustl(XCLASSRootDir))
            ! print*,"NumberFreeParameter = ", NumberFreeParameter


            !< variables for molfit file
            InitTotalNumberOfFrequencyRanges = TotalNumberOfFrequencyRanges                 !< total number of frequency ranges
            InitNumberDistances = NumberDistances                                           !< number of unequal distances defined in molfit file
            InitNumberModelPixelXX = NumberModelPixelXX                                     !< number of pixel for sub-beam modeling along x-direction
            InitNumberModelPixelYY = NumberModelPixelYY                                     !< number of pixel for sub-beam modeling along y-direction

            ! Debug:
            ! print*,"TotalNumberOfFrequencyRanges = ", TotalNumberOfFrequencyRanges
            ! print*,"NumberDistances = ", NumberDistances
            ! print*,"NumberModelPixelXX = ", NumberModelPixelXX
            ! print*,"NumberModelPixelYY = ", NumberModelPixelYY


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< variables in subroutine myXCLASSInitVar
            InitNumberOfTemperatures = NumberOfTemperatures                                 !< number of temperatures
            if (allocated(InitColumnNamesPartFunc)) then
                deallocate(InitColumnNamesPartFunc, &
                           InitTempPartFunc, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine  inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- InitColumnNamesPartFunc")'
                    print '(5x, "- InitTempPartFunc")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(InitColumnNamesPartFunc(NumberOfTemperatures, 40), &
                     InitTempPartFunc(NumberOfTemperatures), stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.initxclass!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- InitColumnNamesPartFunc(NumberOfTemperatures, 40)")'
                print '(5x, "- InitTempPartFunc(NumberOfTemperatures)")'
                print '(" ")'
                print '(3x, "Used parameter:")'
                print '(" ")'
                print '(5x, "- NumberOfTemperatures = ", I8)', NumberOfTemperatures
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            InitTempPartFunc = TempPartFunc                                                 !< temperatures for partition function


            !< (STRING ARRAY) the following lines are necessary because f2py can not handle string arrays properly
            InitColumnNamesPartFunc = 0
            Do i = 1, NumberOfTemperatures
                Do j = 1, len_trim(ColumnNamesPartFunc(i))
                    k = ichar(ColumnNamesPartFunc(i)(j:j))
                    InitColumnNamesPartFunc(i, j) = k                                       !< column names of table PartitionFunction
                end Do
            end Do

            ! Debug:
            ! print*,"NumberOfTemperatures = ", NumberOfTemperatures
            ! print*,"TempPartFunc = ", TempPartFunc
            ! print*,"ColumnNamesPartFunc = ", ColumnNamesPartFunc


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< variables in subroutine GetmyXCLASSParameter
            InitTotalNumberComponents = TotalNumberComponents                               !< counter for total number of components of all molecules
            if (allocated(InitmyXCLASSParameter)) then
                deallocate(InitmyXCLASSParameter, &
                           InitCompMoleculeIndex, &
                           InitGeometryFlag, &
                           InittdFlag, &
                           InitnHFlag, &
                           InitvWidthLimits, &
                           InitvOffLimits, &
                           InitKindOfMolecule, &
                           InitLineProfileFunction, &
                           InitLTEFlag, &
                           InitThinFlag, &
                           InitSortedDistances, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine  inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- InitmyXCLASSParameter")'
                    print '(5x, "- InitCompMoleculeIndex")'
                    print '(5x, "- InitGeometryFlag")'
                    print '(5x, "- InittdFlag")'
                    print '(5x, "- InitnHFlag")'
                    print '(5x, "- InitvWidthLimits")'
                    print '(5x, "- InitvOffLimits")'
                    print '(5x, "- InitKindOfMolecule")'
                    print '(5x, "- InitLineProfileFunction")'
                    print '(5x, "- InitLTEFlag")'
                    print '(5x, "- InitThinFlag")'
                    print '(5x, "- InitSortedDistances")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(InitmyXCLASSParameter(16, TotalNumberComponents), &
                     InitCompMoleculeIndex(TotalNumberComponents), &
                     InitGeometryFlag(TotalNumberComponents), &
                     InittdFlag(TotalNumberComponents), &
                     InitnHFlag(TotalNumberComponents), &
                     InitvWidthLimits(TotalNumberComponents, 4), &
                     InitvOffLimits(TotalNumberComponents, 3), &
                     InitKindOfMolecule(TotalNumberComponents), &
                     InitLineProfileFunction(TotalNumberComponents), &
                     InitLTEFlag(TotalNumberComponents), &
                     InitThinFlag(TotalNumberComponents), &
                     InitSortedDistances(TotalNumberComponents), stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.initxclass!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- InitmyXCLASSParameter(16, TotalNumberComponents)")'
                print '(5x, "- InitCompMoleculeIndex(TotalNumberComponents)")'
                print '(5x, "- InitGeometryFlag(TotalNumberComponents)")'
                print '(5x, "- InittdFlag(TotalNumberComponents)")'
                print '(5x, "- InitnHFlag(TotalNumberComponents)")'
                print '(5x, "- InitvWidthLimits(TotalNumberComponents, 4)")'
                print '(5x, "- InitvOffLimits(TotalNumberComponents, 3)")'
                print '(5x, "- InitKindOfMolecule(TotalNumberComponents)")'
                print '(5x, "- InitLineProfileFunction(TotalNumberComponents)")'
                print '(5x, "- InitLTEFlag(TotalNumberComponents)")'
                print '(5x, "- InitThinFlag(TotalNumberComponents)")'
                print '(5x, "- InitSortedDistances(TotalNumberComponents)")'
                print '(" ")'
                print '(3x, "Used parameter:")'
                print '(" ")'
                print '(5x, "- TotalNumberComponents = ", I8)', TotalNumberComponents
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            InitmyXCLASSParameter = myXCLASSParameter                                       !< array containing all molfit parameters for each component
            InitCompMoleculeIndex = CompMoleculeIndex                                       !< (int) array molecule index for each component
            InitGeometryFlag = GeometryFlag                                                 !< defines geometry of each component / layer
            InittdFlag = tdFlag                                                             !< flag for reading T_dOff and T_dSlope from molfit file
            InitnHFlag = nHFlag                                                             !< flag for global setting of nH, kappa and beta
            InitvWidthLimits = vWidthLimits                                                 !< lower and upper limits of velocity widths parameters
            InitvOffLimits = vOffLimits                                                     !< lower and upper limits of velocity offsets parameters
            InitKindOfMolecule = KindOfMolecule                                             !< defines kind of molecule, i.e. molecule, rrl, etc.
            InitLineProfileFunction = LineProfileFunction                                   !< defines line profile function for each component
            InitLTEFlag = LTEFlag                                                           !< flag for LTE calculation
            InitThinFlag = ThinFlag                                                         !< flag indicating optical thin or thick (for Non-LTE RRLs)
            InitSortedDistances = SortedDistances                                           !< array containing sorted distances

            ! Debug:
            ! print*,"TotalNumberComponents = ", TotalNumberComponents
            ! print*,"myXCLASSParameter = ", myXCLASSParameter
            ! print*,"CompMoleculeIndex = ", CompMoleculeIndex
            ! print*,"GeometryFlag = ", GeometryFlag
            ! print*,"tdFlag = ", tdFlag
            ! print*,"nHFlag = ", nHFlag
            ! print*,"vWidthLimits = ", vWidthLimits
            ! print*,"vOffLimits = ", vOffLimits
            ! print*,"KindOfMolecule = ", KindOfMolecule
            ! print*,"LineProfileFunction = ", LineProfileFunction
            ! print*,"LTEFlag = ", LTEFlag
            ! print*,"ThinFlag = ", ThinFlag
            ! print*,"SortedDistances = ", SortedDistances


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< variables for iso ratio file
            InitNumberOfIsomasters = NumberOfIsomasters                                     !< number of isomasters
            InitNumberOfGlobalISORatios = NumberOfGlobalISORatios                           !< number of global iso ratio definitions
            InitTotalNumberOfMolecules = TotalNumberOfMolecules                             !< total number of molecules including isotopologues
            InitIsoTableFileName = IsoTableFileName                                         !< path and name of the iso file
            if (allocated(IsoMolecule)) then
                if (allocated(InitIsoRatio)) then
                    deallocate(InitIsoRatio, &
                               InitIsoMolecule, &
                               InitIsoMasters, &
                               stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine  inter.initxclass!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variables:")'
                        print '(5x, "- InitIsoRatio")'
                        print '(5x, "- InitIsoMolecule")'
                        print '(5x, "- InitIsoMasters")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                allocate(InitIsoRatio(NumberOfIsomasters, 2), &
                         InitIsoMolecule(NumberOfIsomasters, 40), &
                         InitIsoMasters(NumberOfIsomasters, 40), stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not allocate the following variables:")'
                    print '(5x, "- InitIsoRatio(NumberOfIsomasters, 2)")'
                    print '(5x, "- InitIsoMolecule(NumberOfIsomasters, 40)")'
                    print '(5x, "- InitIsoMasters(NumberOfIsomasters, 40)")'
                    print '(" ")'
                    print '(3x, "Used parameter:")'
                    print '(" ")'
                    print '(5x, "- NumberOfIsomasters = ", I8)', NumberOfIsomasters
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', allocstatus
                    print '(" ")'
                endif
                InitIsoRatio = IsoRatio                                                     !< iso ratios for all isomasters


                !< the following lines are necessary, because f2py can not handle arrays of string properly
                InitIsoMolecule = 0
                InitIsoMasters = 0
                Do i = 1, InitNumberOfIsomasters
                    Do j = 1, len_trim(IsoMolecule(i))
                        k1 = ichar(IsoMolecule(i)(j:j))
                        InitIsoMolecule(i, j) = k1
                    end Do
                    Do j = 1, len_trim(IsoMasters(i))
                        k2 = ichar(IsoMasters(i)(j:j))
                        InitIsoMasters(i, j) = k2
                    end Do
                end Do

                ! Debug:
                ! print*,"IsoMolecule = ", IsoMolecule
                ! print*,"IsoMasters = ", IsoMasters
                ! print*,"IsoRatio = ", IsoRatio
            endif
            if (allocated(GlobalIsoRatioParameter)) then
                if (allocated(InitGlobalIsoRatioParameter)) then
                    deallocate(InitGlobalIsoRatioParameter, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine  inter.initxclass!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variable:")'
                        print '(5x, "- InitGlobalIsoRatioParameter")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                allocate(InitGlobalIsoRatioParameter(NumberOfIsomasters, NumberOfGlobalISORatios, 2), stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not allocate the following variable:")'
                    print '(5x, "- InitGlobalIsoRatioParameter(NumberOfIsomasters, NumberOfGlobalISORatios, 2)")'
                    print '(" ")'
                    print '(3x, "Used parameter:")'
                    print '(" ")'
                    print '(5x, "- NumberOfIsomasters = ", I8)', NumberOfIsomasters
                    print '(5x, "- NumberOfGlobalISORatios = ", I8)', NumberOfGlobalISORatios
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', allocstatus
                    print '(" ")'
                endif
                InitGlobalIsoRatioParameter = GlobalIsoRatioParameter

                ! Debug:
                ! print*,"GlobalIsoRatioParameter = ", GlobalIsoRatioParameter
            endif
            if (allocated(InitIsoNfitConversionTable)) then
                deallocate(InitIsoNfitConversionTable, &
                           InitIsoRatioConversionTable, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine  inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- InitIsoNfitConversionTable")'
                    print '(5x, "- InitIsoRatioConversionTable")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(InitIsoNfitConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules, 2), &
                     InitIsoRatioConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.initxclass!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- InitIsoNfitConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules, 2)")'
                print '(5x, "- InitIsoRatioConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules)")'
                print '(" ")'
                print '(3x, "Used parameter:")'
                print '(" ")'
                print '(5x, "- TotalNumberOfMolecules = ", I8)', TotalNumberOfMolecules
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            InitIsoNfitConversionTable = IsoNfitConversionTable                             !< iso ratios conversion table for free paramter index
            InitIsoRatioConversionTable = IsoRatioConversionTable                           !< table with iso ratios between iso master and molecule
            InitIsoFlag = IsoFlag                                                           !< flag indicating use of isotopologues

            ! Debug:
            ! print*,"NumberOfIsomasters = ", NumberOfIsomasters
            ! print*,"NumberOfGlobalISORatios = ", NumberOfGlobalISORatios
            ! print*,"TotalNumberOfMolecules = ", TotalNumberOfMolecules
            ! print*,"IsoTableFileName = ", IsoTableFileName
            ! print*,"IsoNfitConversionTable = ", IsoNfitConversionTable
            ! print*,"IsoRatioConversionTable = ", IsoRatioConversionTable
            ! print*,"IsoFlag = ", IsoFlag


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< frequency range parameters
            InitNumberMolecules = NumberMolecules                                           !< number of molecules in the current molfits file
            InitTotalNumberDataPoints = TotalNumberDataPoints                               !< total number of data points
            MaxNumberOfMolecules = NumberMolecules + 2 * NumberOfIsomasters
            iii = max(1, NumberFreeParameter)
            if (allocated(InitBackgroundFromFile)) then
                deallocate(InitBackgroundFromFile, &
                           InitSpectrumIndexForFreqRange, &
                           InitObservationalDataList, &
                           InitDataPointIndexFreqRange, &
                           InitConversionTableMAGIXmyXCLASSParam, &
                           InitDustTauFromFile, &
                           InitMoleculeNames, &
                           InitNumTransFreqPerObsFreq, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine  inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- InitBackgroundFromFile")'
                    print '(5x, "- InitSpectrumIndexForFreqRange")'
                    print '(5x, "- InitObservationalDataList")'
                    print '(5x, "- InitDataPointIndexFreqRange")'
                    print '(5x, "- InitConversionTableMAGIXmyXCLASSParam")'
                    print '(5x, "- InitDustTauFromFile")'
                    print '(5x, "- InitMoleculeNames")'
                    print '(5x, "- InitNumTransFreqPerObsFreq")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(InitBackgroundFromFile(TotalNumberDataPoints), &
                     InitSpectrumIndexForFreqRange(TotalNumberOfFrequencyRanges), &
                     InitObservationalDataList(TotalNumberDataPoints, 3), &
                     InitDataPointIndexFreqRange(TotalNumberOfFrequencyRanges, 2), &
                     InitConversionTableMAGIXmyXCLASSParam(iii, 2), &
                     InitDustTauFromFile(TotalNumberDataPoints), &
                     InitMoleculeNames(MaxNumberOfMolecules, 40), &
                     InitNumTransFreqPerObsFreq(MaxNumberOfMolecules, TotalNumberDataPoints), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.initxclass!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- InitSpectrumIndexForFreqRange(TotalNumberOfFrequencyRanges)")'
                print '(5x, "- InitObservationalDataList(TotalNumberDataPoints, 3)")'
                print '(5x, "- InitDataPointIndexFreqRange(TotalNumberOfFrequencyRanges, 2)")'
                print '(5x, "- InitMoleculeNames(MaxNumberOfMolecules)")'
                print '(5x, "- InitConversionTableMAGIXmyXCLASSParam(iii, 2)")'
                print '(5x, "- InitDustTauFromFile(TotalNumberDataPoints)")'
                print '(5x, "- InitBackgroundFromFile(TotalNumberDataPoints)")'
                print '(5x, "- InitNumTransFreqPerObsFreq(MaxNumberOfMolecules, TotalNumberDataPoints)")'
                print '(" ")'
                print '(3x,"Used parameters:")'
                print '(" ")'
                print '(5x, "- TotalNumberOfFrequencyRanges = ", I8)', TotalNumberOfFrequencyRanges
                print '(5x, "- TotalNumberDataPoints        = ", I8)', TotalNumberDataPoints
                print '(5x, "- MaxNumberOfMolecules         = ", I8)', MaxNumberOfMolecules
                print '(5x, "- NumberFreeParameter          = ", I8)', NumberFreeParameter
                print '(5x, "- iii                          = ", I8)', iii
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            StartFrequency = InitStartFrequency                                             !< first frequency for each frequency range
            EndFrequency = InitEndFrequency                                                 !< last frequency for each frequency range
            StepFrequency = InitStepFrequency                                               !< step size for each frequency range
            tbFlag = InittbFlag                                                             !< flag indicating that T_Back and T_Slope describes
                                                                                            !< continuum completely
            BackgroundTemperatureRange = InitBackgroundTemperatureRange                     !< T_Back for each frequency range
            TemperatureSlopeRange = InitTemperatureSlopeRange                               !< T_Slope for each frequency range
            HydrogenColumnDensityRange = InitHydrogenColumnDensityRange                     !< nH for each frequency range
            DustBetaRange = InitDustBetaRange                                               !< beta for each frequency range
            KappaRange = InitKappaRange                                                     !< kappa for each frequency range
            DustFilenNames = ""
            BackgroundFilenNames = ""
            TelescopeSize = InitTelescopeSize                                               !< size of telescope for each frequency range
            TelescopeBMIN = InitTelescopeBMIN                                               !< size of BMIN
            TelescopeBMAJ = InitTelescopeBMAJ                                               !< size of BMAJ
            TelescopeBPA = InitTelescopeBPA                                                 !< size of BPA
            InterFlag = InitInterFlag                                                       !< flag indicating that observation is interferrometric data
            GlobalvLSR = InitGlobalvLSR                                                     !< global velocity for each freq. range
            Redshift_Range = InitRedshift_Range                                             !< redshift for each range
            InitSpectrumIndexForFreqRange = SpectrumIndexForFreqRange                       !< store spectrum index for each frequency range
            InitObservationalDataList = ObservationalDataList                               !< list containing all observational data
            InitDataPointIndexFreqRange = DataPointIndexFreqRange                           !< index of data point of first and last freq. in freq. range


            !< (STRING ARRAY) the following lines are necessary because f2py can not handle string arrays properly
            InitMoleculeNames = 0
            Do i = 1, MaxNumberOfMolecules
                Do j = 1, len_trim(MoleculeNames(i))
                    k = ichar(MoleculeNames(i)(j:j))
                    InitMoleculeNames(i, j) = k                                             !< names of molecules for iso table
                end Do
            end Do


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< allocate arrays for additional input files
            InitConversionTableMAGIXmyXCLASSParam = ConversionTableMAGIXmyXCLASSParam       !< conversion table between myXCLASS and MAGIX parameter
            InitDustTauFromFile = DustTauFromFile                                           !< array containing tau_dust from file
            InitBackgroundFromFile = BackgroundFromFile                                     !< array containing background from file
            InitNumTransFreqPerObsFreq = NumTransFreqPerObsFreq                             !< describes number of trans. freq. for each freq. channel
            InitNumberMolfitMolecules = NumberMolfitMolecules                               !< number of molecules in molfit file only
            if (allocated(InitNumberComponentsPerMolecule)) then
                deallocate(InitNumberComponentsPerMolecule, &
                           InitPureDistances, &
                           InitDistanceOrderingArray, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine  inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- InitNumberComponentsPerMolecule")'
                    print '(5x, "- InitPureDistances")'
                    print '(5x, "- InitDistanceOrderingArray")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(InitNumberComponentsPerMolecule(0:NumberMolfitMolecules), &
                     InitPureDistances(TotalNumberComponents), &
                     InitDistanceOrderingArray(TotalNumberComponents, TotalNumberComponents), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.initxclass!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- InitNumberComponentsPerMolecule(0:NumberMolfitMolecules)")'
                print '(5x, "- InitPureDistances(TotalNumberComponents)")'
                print '(5x, "- InitDistanceOrderingArray(TotalNumberComponents, TotalNumberComponents)")'
                print '(" ")'
                print '(3x,"Used parameter:")'
                print '(" ")'
                print '(5x, "- NumberMolfitMolecules = ", I8)', NumberMolfitMolecules
                print '(5x, "- TotalNumberComponents = ", I8)', TotalNumberComponents
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            InitNumberComponentsPerMolecule = NumberComponentsPerMolecule                   !< number of components per molecule
            InitPureDistances = PureDistances                                               !< array containing pure distances
            InitDistanceOrderingArray = DistanceOrderingArray                               !< translation table between pure distances and comp. indices
            if (allocated(EmsAbsFunc)) then
                if (allocated(InitEmsAbsFunc)) then
                    deallocate(InitEmsAbsFunc, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine  inter.initxclass!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variable:")'
                        print '(5x, "- InitEmsAbsFunc")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                allocate(InitEmsAbsFunc(TotalNumberDataPoints, NumberDistances, 2), &
                         stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not allocate the following variable:")'
                    print '(5x, "- InitEmsAbsFunc(TotalNumberDataPoints, NumberDistances, 2)")'
                    print '(" ")'
                    print '(3x,"Used parameter:")'
                    print '(" ")'
                    print '(5x, "- TotalNumberDataPoints = ", I8)', TotalNumberDataPoints
                    print '(5x, "- NumberDistances = ", I8)', NumberDistances
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', allocstatus
                    print '(" ")'
                endif
                InitEmsAbsFunc = EmsAbsFunc

                ! Debug:
                ! print*,"EmsAbsFunc = ", EmsAbsFunc
            endif
            InitEmsAbsPATH = EmsAbsPATH                                                     !< path of directory containing files describing em. and abs.
            InitUseEmAbsFuncFlag = UseEmAbsFuncFlag                                         !< flag indicating usage of emission/absorption file(s)

            ! Debug:
            ! print*,"NumberMolecules = ", NumberMolecules
            ! print*,"TotalNumberDataPoints = ", TotalNumberDataPoints
            ! print*,"MaxNumberOfMolecules = ", MaxNumberOfMolecules
            ! print*,"StartFrequency = ", StartFrequency
            ! print*,"EndFrequency = ", EndFrequency
            ! print*,"StepFrequency = ", StepFrequency
            ! print*,"tbFlag = ", tbFlag
            ! print*,"BackgroundTemperatureRange = ", BackgroundTemperatureRange
            ! print*,"TemperatureSlopeRange = ", TemperatureSlopeRange
            ! print*,"BackgroundFromFile = ", BackgroundFromFile
            ! print*,"HydrogenColumnDensityRange = ", HydrogenColumnDensityRange
            ! print*,"DustBetaRange = ", DustBetaRange
            ! print*,"KappaRange = ", KappaRange
            ! print*,"TelescopeSize = ", TelescopeSize
            ! print*,"TelescopeBMIN = ", TelescopeBMIN
            ! print*,"TelescopeBMAJ = ", TelescopeBMAJ
            ! print*,"TelescopeBPA = ", TelescopeBPA
            ! print*,"InterFlag = ", InterFlag
            ! print*,"GlobalvLSR = ", GlobalvLSR
            ! print*,"Redshift_Range = ", Redshift_Range
            ! print*,"SpectrumIndexForFreqRange = ", SpectrumIndexForFreqRange
            ! print*,"ObservationalDataList = ", ObservationalDataList
            ! print*,"DataPointIndexFreqRange = ", DataPointIndexFreqRange
            ! print*,"ConversionTableMAGIXmyXCLASSParam = ", ConversionTableMAGIXmyXCLASSParam
            ! print*,"DustTauFromFile = ", DustTauFromFile
            ! print*,"ContPhen_Range = ", ContPhen_Range
            ! print*,"Dust_Flag_Range = ", Dust_Flag_Range
            ! print*,"Phen_Flag_Range = ", Phen_Flag_Range
            ! print*,"NumTransFreqPerObsFreq = ", NumTransFreqPerObsFreq
            ! print*,"NumberMolfitMolecules = ", NumberMolfitMolecules
            ! print*,"NumberComponentsPerMolecule = ", NumberComponentsPerMolecule
            ! print*,"PureDistances = ", PureDistances
            ! print*,"DistanceOrderingArray = ", DistanceOrderingArray
            ! print*,"EmsAbsFunc = ", EmsAbsFunc
            ! print*,"EmsAbsPATH = ", EmsAbsPATH
            ! print*,"UseEmAbsFuncFlag = ", UseEmAbsFuncFlag


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< allocate variables for emission and absorption function calculation
            if (EmAbsFlag) then
                i = max0(1, TotalNumberDataPoints)
                j = max0(1, TotalNumberComponents)
                if (allocated(InitTauEmAbsArray)) then
                    deallocate(InitTauEmAbsArray, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine inter.initxclass!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variables:")'
                        print '(5x, "- InitTauEmAbsArray")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                allocate(InitTauEmAbsArray(i, j, 2), stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- InitTauEmAbsArray(i, j, 2)")'
                    print '(" ")'
                    print '(3x,"Used parameters:")'
                    print '(" ")'
                    print '(5x, "- i = ", I8)', i
                    print '(5x, "- j = ", I8)', j
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
                InitTauEmAbsArray = 0.d0

                ! Debug:
                ! print*,"InitTauEmAbsArray = ", InitTauEmAbsArray
            endif


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< allocate variables for storing opacities and intensities
            if (AllOutputFilesFlag) then


                !< determine max. number of isotopologues per component
                MaxNumIso = 0
                if (IsoFlag) then
                    Do l = 1, TotalNumberOfMolecules
                        k = 0
                        Do m = 1, TotalNumberOfMolecules
                            if (IsoRatioConversionTable(l, m) /= 0.d0) then                 !< is a iso ratio defined for mth molecule?
                                k = k + 1
                            endif
                        end Do
                        if (MaxNumIso < k) then
                            MaxNumIso = k
                        endif
                    end Do
                endif
                MaxNumIso = max0(1, MaxNumIso)

                ! Debug:
                ! print*,"MaxNumIso = ", MaxNumIso


                !< allocate variables for storing opacities
                i = max0(1, TotalNumberDataPoints)
                j = max0(1, TotalNumberComponents)
                if (allocated(InitIntPerCompArray)) then
                    deallocate(InitIntPerCompArray, InitTauPerCompArray, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine inter.initxclass!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variables:")'
                        print '(5x, "- InitIntPerCompArray")'
                        print '(5x, "- InitTauPerCompArray")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                allocate(InitIntPerCompArray(i, j, MaxNumIso), InitTauPerCompArray(i, j, MaxNumIso), stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not allocate the following variables:")'
                    print '(5x, "- InitIntPerCompArray(i, j, MaxNumIso)")'
                    print '(5x, "- InitTauPerCompArray(i, j, MaxNumIso)")'
                    print '(" ")'
                    print '(3x,"Used parameters:")'
                    print '(" ")'
                    print '(5x, "- i = ", I8)', i
                    print '(5x, "- j = ", I8)', j
                    print '(5x, "- MaxNumIso = ", I8)', MaxNumIso
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', allocstatus
                    print '(" ")'
                endif
                InitIntPerCompArray = 0.d0
                InitTauPerCompArray = 0.d0

                ! Debug:
                ! print*,"InitIntPerCompArray = ", InitIntPerCompArray
                ! print*,"InitTauPerCompArray = ", InitTauPerCompArray
            endif


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< variables for sub-beam description
            if (allocated(ConfigIndex)) then
                if (allocated(InitLayerMap)) then
                    deallocate(InitLayerMap, &
                               InitConfigList, &
                               InitConfigIndex, &
                               InitGausssianBeamMap, &
                               stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine inter.initxclass!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variables:")'
                        print '(5x, "- InitLayerMap")'
                        print '(5x, "- InitConfigList")'
                        print '(5x, "- InitConfigIndex")'
                        print '(5x, "- InitGausssianBeamMap")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                allocate(InitLayerMap(NumberModelPixelXX, NumberModelPixelYY, TotalNumberComponents), &
                         InitConfigList(NumberModelPixelXX * NumberModelPixelYY, TotalNumberComponents), &
                         InitConfigIndex(NumberModelPixelXX, NumberModelPixelYY), &
                         InitGausssianBeamMap(NumberModelPixelXX, NumberModelPixelYY), stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not allocate the following variables:")'
                    print '(5x, "- InitLayerMap(NumberModelPixelXX, NumberModelPixelYY, TotalNumberComponents)")'
                    print '(5x, "- InitConfigList(NumberModelPixelXX * NumberModelPixelYY, TotalNumberComponents)")'
                    print '(5x, "- InitConfigIndex(NumberModelPixelXX, NumberModelPixelYY)")'
                    print '(5x, "- InitGausssianBeamMap(NumberModelPixelXX, NumberModelPixelYY)")'
                    print '(" ")'
                    print '(3x,"Used parameter:")'
                    print '(" ")'
                    print '(5x, "- NumberModelPixelXX = ", I8)', NumberModelPixelXX
                    print '(5x, "- NumberModelPixelYY = ", I8)', NumberModelPixelYY
                    print '(5x, "- TotalNumberComponents = ", I8)', TotalNumberComponents
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', allocstatus
                    print '(" ")'
                endif
                InitLayerMap = LayerMap                                                     !< indicates, which components are taken into account
                InitConfigList = ConfigList                                                 !< list of unique pixel combination
                InitConfigIndex = ConfigIndex                                               !< configuration index for model map
                InitGausssianBeamMap = GausssianBeamMap                                     !< map describing elliptical rotated Gaussian beam

                ! Debug:
                ! print*,"LayerMap = ", LayerMap
                ! print*,"ConfigList = ", ConfigList
                ! print*,"ConfigIndex = ", ConfigIndex
                ! print*,"GausssianBeamMap = ", GausssianBeamMap
            endif
            InitSubBeamDescriptionFlag = SubBeamDescriptionFlag                             !< flag indicating sub-beam description
            InitSizeOfPixel_deg = SizeOfPixel_deg                                           !< size of pixel

            ! Debug:
            ! print*,"SubBeamDescriptionFlag = ", SubBeamDescriptionFlag
            ! print*,"SizeOfPixel_deg = ", SizeOfPixel_deg


            !< variables of subroutine GetPartitionFunction
            InitNumberMoleculePartFunc = NumberMoleculePartFunc                             !< number of entries in table PartitionFunction
            InitFirsti = Firsti                                                             !< working variables for interpolation
            InitLasti = Lasti                                                               !< working variables for interpolation
            Initstepi = stepi                                                               !< working variables for interpolation
            InitTempLow = TempLow                                                           !< working variables for extrapolation of part. func.
            InitTempHigh = TempHigh                                                         !< working variables for extrapolation of part. func.
            InitColumnNameForNamePartFunc = ColumnNameForNamePartFunc                       !< name of column including the names of molecules
            if (allocated(InitMolNamePartFunc)) then
                deallocate(InitMolNamePartFunc, &
                           InitlgQ, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine  inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- InitMolNamePartFunc")'
                    print '(5x, "- InitlgQ")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(InitMolNamePartFunc(NumberMoleculePartFunc, 40), &
                     InitlgQ(NumberOfTemperatures, NumberMoleculePartFunc), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.initxclass!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- InitMolNamePartFunc(NumberMoleculePartFunc, 40)")'
                print '(5x, "- InitlgQ(NumberOfTemperatures, NumberMoleculePartFunc)")'
                print '(" ")'
                print '(3x,"Used parameter:")'
                print '(" ")'
                print '(5x, "- NumberMoleculePartFunc = ", I8)', NumberMoleculePartFunc
                print '(5x, "- NumberOfTemperatures = ", I8)', NumberOfTemperatures
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            InitlgQ = lgQ                                                                   !< lgQ entries of table PartitionFunction


            !< (STRING ARRAY) the following lines are necessary because f2py can not handle string arrays properly
            InitMolNamePartFunc = 0
            Do i = 1, NumberMoleculePartFunc
                Do j = 1, len_trim(MolNamePartFunc(i))
                    k = ichar(MolNamePartFunc(i)(j:j))
                    InitMolNamePartFunc(i, j) = k
                end Do
            end Do

            ! Debug:
            ! print*,"NumberOfTemperatures = ", NumberOfTemperatures
            ! print*,"NumberMoleculePartFunc = ", NumberMoleculePartFunc
            ! print*,"Firsti = ", Firsti
            ! print*,"Lasti = ", Lasti
            ! print*,"stepi = ", stepi
            ! print*,"TempLow = ", TempLow
            ! print*,"TempHigh = ", TempHigh
            ! print*,"TempPartFunc = ", TempPartFunc
            ! print*,"lgQ = ", lgQ
            ! print*,"ColumnNameForNamePartFunc = ", ColumnNameForNamePartFunc
            ! print*,"MolNamePartFunc = ", MolNamePartFunc
            ! print*,"ColumnNamesPartFunc = ", ColumnNamesPartFunc
            ! print*,"MolNamePartFunc = ", MolNamePartFunc
            ! print*,"lgQ = ", lgQ


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< variables of subroutine GetTransitionParameters
            if (allocated(InitNumEntriesRanges)) then
                deallocate(InitNumEntriesRanges, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- InitNumEntriesRanges")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(InitNumEntriesRanges(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.initxclass!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- InitNumEntriesRanges(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules)")'
                print '(" ")'
                print '(3x,"Used parameter:")'
                print '(" ")'
                print '(5x, "- TotalNumberOfFrequencyRanges = ", I8)', TotalNumberOfFrequencyRanges
                print '(5x, "- TotalNumberOfMolecules = ", I8)', TotalNumberOfMolecules
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            InitNumEntriesRanges = NumEntriesRanges                                         !< number of transitions for each freq. range and molecule
            InitTotalNumberOfTransitions = TotalNumberOfTransitions                         !< number of entries in table RadTrans
            if (allocated(InitMolecularData)) then
                deallocate(InitMolecularData, &
                           InitMolecularDataIndices, &
                           InitLowerQNString, &
                           InitUpperQNString, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- InitMolecularData")'
                    print '(5x, "- InitMolecularDataIndices")'
                    print '(5x, "- InitLowerQNString")'
                    print '(5x, "- InitUpperQNString")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            j = max(1, TotalNumberOfTransitions)
            allocate(InitMolecularData(j, 9), &
                     InitMolecularDataIndices(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2), &
                     InitLowerQNString(j, 60), &
                     InitUpperQNString(j, 60), stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.initxclass!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- InitMolecularData(j, 9)")'
                print '(5x, "- InitMolecularDataIndices(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2)")'
                print '(5x, "- InitLowerQNString(j, 60)")'
                print '(5x, "- InitUpperQNString(j, 60)")'
                print '(" ")'
                print '(3x,"Used parameter:")'
                print '(" ")'
                print '(5x, "- TotalNumberOfFrequencyRanges = ", I8)', TotalNumberOfFrequencyRanges
                print '(5x, "- TotalNumberOfMolecules = ", I8)', TotalNumberOfMolecules
                print '(5x, "- j = ", I8)', j
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            InitMolecularData = MolecularData                                               !< array containing the molecular data for all molecules and
                                                                                            !< all freqency ranges
            InitMolecularDataIndices = MolecularDataIndices                                 !< start and end index for each molecule and frequency range
                                                                                            !< within MolecularData variable

            !< (STRING ARRAY) the following lines are necessary because f2py can not handle string arrays properly
            InitMolNamePartFunc = 0
            Do i = 1, j
                Do k = 1, len_trim(LowerQNString(i))
                    k1 = ichar(LowerQNString(i)(k:k))                                       !< quantum number string of lower state
                    InitLowerQNString(i, k) = k1
                end Do
                Do k = 1, len_trim(UpperQNString(i))
                    k2 = ichar(UpperQNString(i)(k:k))                                       !< quantum number string of upper state
                    InitUpperQNString(i, k) = k2
                end Do
            end Do

            ! Debug:
            ! print*,"NumEntriesRanges = ", NumEntriesRanges
            ! print*,"TotalNumberOfTransitions = ", TotalNumberOfTransitions
            ! print*,"j = ", j
            ! print*,"MolecularData = ", MolecularData
            ! print*,"MolecularDataIndices = ", MolecularDataIndices
            ! print*,"LowerQNString = ", LowerQNString
            ! print*,"UpperQNString = ", UpperQNString


            !< variables for sqlite3 database
            InitMinNumTransitionsSQL = MinNumTransitionsSQL                                 !< min number of transitions which are considered in SQL query
            InitMaxNumTransitionsSQL = MaxNumTransitionsSQL                                 !< max number of transitions which are considered in SQL query
            InitOrderTransSQL = OrderTransSQL                                               !< order or transition considered in SQL query
            InitMaxElowSQL = MaxElowSQL                                                     !< highest lower energy of transitions considered in SQL query
            InitMingASQL = MingASQL                                                         !< min. intensity of transitions considered in SQL query
            InitNameOfPartFuncTable = NameOfPartFuncTable                                   !< name of table for partition function
            InitNameOfRadTransTable = NameOfRadTransTable                                   !< name of table for rad. trans.
            InitdbName = dbName                                                             !< name of database

            ! Debug:
            ! print*,"MinNumTransitionsSQL = ", MinNumTransitionsSQL
            ! print*,"MaxNumTransitionsSQL = ", MaxNumTransitionsSQL
            ! print*,"OrderTransSQL = ", OrderTransSQL
            ! print*,"MaxElowSQL = ", MaxElowSQL
            ! print*,"MingASQL = ", MingASQL
            ! print*,"NameOfPartFuncTable = ", NameOfPartFuncTable
            ! print*,"NameOfRadTransTable = ", NameOfRadTransTable
            ! print*,"dbName = ", trim(adjustl(dbName))


            !< variables for rad-trans parameter
            InitTotalNumTransFreq = TotalNumTransFreq                                       !< number of doppler-shifted trans. freq. for curr. layer
            InitColumn300KPartFunc = Column300KPartFunc                                     !< number of column, which contains part. func. @ 300 K
            InitColumnNameForNameTransitions = ColumnNameForNameTransitions                 !< name of column in table transitions for molecule name
            InitColumnNameForFreqTransitions = ColumnNameForFreqTransitions                 !< name of column in table transitions for frequency
            InitColumnNameForIntTransitions = ColumnNameForIntTransitions                   !< name of column in table transitions for intensity
            InitColumnNameForEinsteinATransitions = ColumnNameForEinsteinATransitions       !< name of column in table transitions for Einstein A coef.
            InitColumnNameForFreqErrTransitions = ColumnNameForFreqErrTransitions           !< name of column in table transitions for error freq.
            InitColumnNameForELowTransitions = ColumnNameForELowTransitions                 !< name of column in table transitions for E_low
            InitColumnNameForgUpTransitions = ColumnNameForgUpTransitions                   !< name of column in table transitions for Upper State degen.
            InitColumnNameForLowerQNTransitions = ColumnNameForLowerQNTransitions           !< name of column in table transitions for lower state QN
            InitColumnNameForUpperQNTransitions = ColumnNameForUpperQNTransitions           !< name of column in table transitions for upper state QN
            if (allocated(CentralOpticalDepth)) then
                if (allocated(InitCentralOpticalDepth)) then
                    deallocate(InitCentralOpticalDepth, &
                               InitDopplerShiftedTransFreq, &
                               InitDopplerShiftedTransFreqIndex, &
                               stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine inter.initxclass!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variables:")'
                        print '(5x, "- InitCentralOpticalDepth")'
                        print '(5x, "- InitDopplerShiftedTransFreq")'
                        print '(5x, "- InitDopplerShiftedTransFreqIndex")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                i = max0(1, TotalNumTransFreq)
                allocate(InitCentralOpticalDepth(i, 2), &
                         InitDopplerShiftedTransFreq(i), &
                         InitDopplerShiftedTransFreqIndex(i, 6), stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.initxclass!")'
                    print '(" ")'
                    print '(3x, "Can not allocate the following variables:")'
                    print '(5x, "- InitCentralOpticalDepth(i, 2)")'
                    print '(5x, "- InitDopplerShiftedTransFreq(i)")'
                    print '(5x, "- InitDopplerShiftedTransFreqIndex(i, 6)")'
                    print '(" ")'
                    print '(3x,"Used parameter:")'
                    print '(" ")'
                    print '(5x, "- i = ", I8)', i
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', allocstatus
                    print '(" ")'
                endif
                InitCentralOpticalDepth = CentralOpticalDepth
                InitDopplerShiftedTransFreq = DopplerShiftedTransFreq
                InitDopplerShiftedTransFreqIndex = DopplerShiftedTransFreqIndex

                ! Debug:
                ! print*,"CentralOpticalDepth = ", CentralOpticalDepth
                ! print*,"DopplerShiftedTransFreq = ", DopplerShiftedTransFreq
                ! print*,"DopplerShiftedTransFreqIndex = ", DopplerShiftedTransFreqIndex
            endif

            ! Debug:
            ! print*,"TotalNumTransFreq = ", TotalNumTransFreq
            ! print*,"Column300KPartFunc = ", Column300KPartFunc
            ! print*,"ColumnNameForNameTransitions = ", ColumnNameForNameTransitions
            ! print*,"ColumnNameForFreqTransitions = ", ColumnNameForFreqTransitions
            ! print*,"ColumnNameForIntTransitions = ", ColumnNameForIntTransitions
            ! print*,"ColumnNameForEinsteinATransitions = ", ColumnNameForEinsteinATransitions
            ! print*,"ColumnNameForFreqErrTransitions = ", ColumnNameForFreqErrTransitions
            ! print*,"ColumnNameForELowTransitions = ", ColumnNameForELowTransitions
            ! print*,"ColumnNameForgUpTransitions = ", ColumnNameForgUpTransitions
            ! print*,"ColumnNameForLowerQNTransitions = ", ColumnNameForLowerQNTransitions
            ! print*,"ColumnNameForUpperQNTransitions = ", ColumnNameForUpperQNTransitions


            !< free parameters
            call myXCLASSParamFree(deallocstatus)


            !< we're done
            return
        end subroutine initxclass


        !*************************************************************************************************************************************************
        !> subroutine: spectrum
        !>
        !> calculate XCLASS spectrum
        !>
        !> input variables:     None
        !>
        !> output variables:    ModeledRangeSpectrum:       array of calculated intensities
        !>
        !>
        !> \author Thomas Moeller
        !>
        subroutine spectrum()

            implicit none
            integer :: i, j, k                                                              !< loop indices
            integer :: k1, k2, k3, iii                                                      !< working variables
            integer :: MaxNumberOfMolecules                                                 !< index of core component with biggest source size
            integer :: ThreadNumber                                                         !< working variable: thread number
            integer :: allocstatus, deallocstatus                                           !< working variable: status variables for (de)allocation
            real*8 :: chi2Value                                                             !< unused dummy argument
            logical :: ModelFunctionFlag                                                    !< unused dummy argument


            !< set some parameters
            printflag = .true.                                                              !< define flag for screen output ( = 0: no screen output)
            ParallezitionMethod = 1                                                         !< define parallelization method used for fftw3
            JobID = 0                                                                       !< unused dummy variable


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< copy variables


            !< general variables
            ErrChannelIndex = InitErrChannelIndex                                           !< copy loop variable for error handling
            AllErrChannels = InitAllErrChannels                                             !< list of all error channels
            tt1 = Inittt1                                                                   !< working variable for debugging: used for timing
            tt2 = Inittt2                                                                   !< working variable for debugging: used for timing
            UseRegistrationXMLFile = InitUseRegistrationXMLFile                             !< use registration XML file
            AllOutputFilesFlag = InitAllOutputFilesFlag                                     !< flag for writing intensities and optical depths to file
            EmAbsFlag = InitEmAbsFlag                                                       !< flag for writing emission and absorption to file
            IntegrationFlag = InitIntegrationFlag                                           !< flag indicating integration over channel width
            LocalOverlapFlag = InitLocalOverlapFlag                                         !< flag indicating is taken into account
            NoSubBeamFlag = InitNoSubBeamFlag                                               !< flag for deactivating sub-beam description
            ParallelizationMethod = InitParallelizationMethod                               !< string describing the parallelization method
            XCLASSRootDir = InitXCLASSRootDir                                               !< path of XCLASS root directory

            ! Debug:
            ! print*,"ErrChannelIndex = ", ErrChannelIndex
            ! print*,"AllErrChannels = ", AllErrChannels
            ! print*,"tt1 = ", tt1
            ! print*,"tt2 = ", tt2
            ! print*,"UseRegistrationXMLFile = ", UseRegistrationXMLFile
            ! print*,"AllOutputFilesFlag = ", AllOutputFilesFlag
            ! print*,"EmAbsFlag = ", EmAbsFlag
            ! print*,"IntegrationFlag = ", IntegrationFlag
            ! print*,"LocalOverlapFlag = ", LocalOverlapFlag
            ! print*,"NoSubBeamFlag = ", NoSubBeamFlag
            ! print*,"ParallelizationMethod = ", ParallelizationMethod
            ! print*,"XCLASSRootDir = ", trim(adjustl(XCLASSRootDir))


            !< variables for molfit file
            TotalNumberOfFrequencyRanges = InitTotalNumberOfFrequencyRanges                 !< total number of frequency ranges
            NumberDistances = InitNumberDistances                                           !< number of unequal distances defined in molfit file
            NumberModelPixelXX = InitNumberModelPixelXX                                     !< number of pixel for sub-beam modeling along x-direction
            NumberModelPixelYY = InitNumberModelPixelYY                                     !< number of pixel for sub-beam modeling along y-direction
            NumberFreeParameter = InitNumberFreeParameter

            ! Debug:
            ! print*,"TotalNumberOfFrequencyRanges = ", TotalNumberOfFrequencyRanges
            ! print*,"NumberDistances = ", NumberDistances
            ! print*,"NumberModelPixelXX = ", NumberModelPixelXX
            ! print*,"NumberModelPixelYY = ", NumberModelPixelYY
            ! print*,"NumberFreeParameter = ", NumberFreeParameter
            ! print*,"InitNumberOfTemperatures = ", InitNumberOfTemperatures


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< variables in subroutine myXCLASSInitVar
            NumberOfTemperatures = InitNumberOfTemperatures                                 !< number of temperatures
            if (allocated(ColumnNamesPartFunc)) then
                deallocate(ColumnNamesPartFunc, &
                           TempPartFunc, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- ColumnNamesPartFunc")'
                    print '(5x, "- TempPartFunc")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(ColumnNamesPartFunc(InitNumberOfTemperatures), &
                     TempPartFunc(InitNumberOfTemperatures), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.spectrum!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- ColumnNamesPartFunc(InitNumberOfTemperatures)")'
                print '(5x, "- TempPartFunc(InitNumberOfTemperatures)")'
                print '(" ")'
                print '(3x, "Used parameter:")'
                print '(" ")'
                print '(5x, "- InitNumberOfTemperatures = ", I8)', InitNumberOfTemperatures
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            TempPartFunc = InitTempPartFunc                                                 !< temperatures for partition function


            !< (STRING ARRAY) the following lines are necessary because f2py can not handle string arrays properly
            ColumnNamesPartFunc = ""
            Do i = 1, NumberOfTemperatures
                Do j = 1, 40
                    k = InitColumnNamesPartFunc(i, j)
                    if (k > 0) then
                        ColumnNamesPartFunc(i)(j:j) = char(k)                               !< column names of table PartitionFunction
                    else
                        exit
                    endif
                end Do
            end Do

            ! Debug:
            ! print*,"NumberOfTemperatures = ", NumberOfTemperatures
            ! print*,"size(TempPartFunc) = ", size(TempPartFunc)
            ! print*,"size(InitTempPartFunc) = ", size(InitTempPartFunc)
            ! print*,"size(ColumnNamesPartFunc) = ", size(ColumnNamesPartFunc)
            ! print*,"size(InitColumnNamesPartFunc, 1) = ", size(InitColumnNamesPartFunc, 1)
            ! print*,"size(InitColumnNamesPartFunc, 2) = ", size(InitColumnNamesPartFunc, 2)
            ! print*,"TempPartFunc = ", TempPartFunc
            ! print*,"ColumnNamesPartFunc = ", ColumnNamesPartFunc


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< variables in subroutine GetmyXCLASSParameter
            TotalNumberComponents = InitTotalNumberComponents                               !< counter for total number of components of all molecules
            if (allocated(myXCLASSParameter)) then
                deallocate(myXCLASSParameter, &
                           CompMoleculeIndex, &
                           GeometryFlag, &
                           tdFlag, &
                           nHFlag, &
                           vWidthLimits, &
                           vOffLimits, &
                           KindOfMolecule, &
                           LineProfileFunction, &
                           LTEFlag, &
                           ThinFlag, &
                           SortedDistances, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- myXCLASSParameter")'
                    print '(5x, "- CompMoleculeIndex")'
                    print '(5x, "- GeometryFlag")'
                    print '(5x, "- tdFlag")'
                    print '(5x, "- nHFlag")'
                    print '(5x, "- vWidthLimits")'
                    print '(5x, "- vOffLimits")'
                    print '(5x, "- KindOfMolecule")'
                    print '(5x, "- LineProfileFunction")'
                    print '(5x, "- LTEFlag")'
                    print '(5x, "- ThinFlag")'
                    print '(5x, "- SortedDistances")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(myXCLASSParameter(16, InitTotalNumberComponents), &
                     CompMoleculeIndex(InitTotalNumberComponents), &
                     GeometryFlag(InitTotalNumberComponents), &
                     tdFlag(InitTotalNumberComponents), &
                     nHFlag(InitTotalNumberComponents), &
                     vWidthLimits(InitTotalNumberComponents, 4), &
                     vOffLimits(InitTotalNumberComponents, 3), &
                     KindOfMolecule(InitTotalNumberComponents), &
                     LineProfileFunction(InitTotalNumberComponents), &
                     LTEFlag(InitTotalNumberComponents), &
                     ThinFlag(InitTotalNumberComponents), &
                     SortedDistances(InitTotalNumberComponents), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.spectrum!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- myXCLASSParameter(16, InitTotalNumberComponents)")'
                print '(5x, "- CompMoleculeIndex(InitTotalNumberComponents)")'
                print '(5x, "- GeometryFlag(InitTotalNumberComponents)")'
                print '(5x, "- tdFlag(InitTotalNumberComponents)")'
                print '(5x, "- nHFlag(InitTotalNumberComponents)")'
                print '(5x, "- vWidthLimits(InitTotalNumberComponents, 4)")'
                print '(5x, "- vOffLimits(InitTotalNumberComponents, 3)")'
                print '(5x, "- KindOfMolecule(InitTotalNumberComponents)")'
                print '(5x, "- LineProfileFunction(InitTotalNumberComponents)")'
                print '(5x, "- LTEFlag(InitTotalNumberComponents)")'
                print '(5x, "- ThinFlag(InitTotalNumberComponents)")'
                print '(5x, "- SortedDistances(InitTotalNumberComponents)")'
                print '(" ")'
                print '(3x, "Used parameter:")'
                print '(" ")'
                print '(5x, "- InitTotalNumberComponents = ", I8)', InitTotalNumberComponents
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            myXCLASSParameter = InitmyXCLASSParameter                                       !< array containing all molfit parameters for each component
            CompMoleculeIndex = InitCompMoleculeIndex                                       !< (int) array molecule index for each component
            GeometryFlag = InitGeometryFlag                                                 !< defines geometry of each component / layer
            tdFlag = InittdFlag                                                             !< flag for reading T_dOff and T_dSlope from molfit file
            nHFlag = InitnHFlag                                                             !< flag for global setting of nH, kappa and beta
            vWidthLimits = InitvWidthLimits                                                 !< lower and upper limits of velocity widths parameters
            vOffLimits = InitvOffLimits                                                     !< lower and upper limits of velocity offsets parameters
            KindOfMolecule = InitKindOfMolecule                                             !< defines kind of molecule, i.e. molecule, rrl, etc.
            LineProfileFunction = InitLineProfileFunction                                   !< defines line profile function for each component
            LTEFlag = InitLTEFlag                                                           !< flag for LTE calculation
            ThinFlag = InitThinFlag                                                         !< flag indicating optical thin or thick (for Non-LTE RRLs)
            SortedDistances = InitSortedDistances                                           !< array containing sorted distances

            ! Debug:
            ! print*,"TotalNumberComponents = ", TotalNumberComponents
            ! print*,"size(myXCLASSParameter, 1) = ", size(myXCLASSParameter, 1)
            ! print*,"size(myXCLASSParameter, 2) = ", size(myXCLASSParameter, 2)
            ! print*,"size(InitmyXCLASSParameter, 1) = ", size(InitmyXCLASSParameter, 1)
            ! print*,"size(InitmyXCLASSParameter, 2) = ", size(InitmyXCLASSParameter, 2)
            ! print*,"size(CompMoleculeIndex) = ", size(CompMoleculeIndex)
            ! print*,"size(InitCompMoleculeIndex) = ", size(InitCompMoleculeIndex)
            ! print*,"size(GeometryFlag) = ", size(GeometryFlag)
            ! print*,"size(InitGeometryFlag) = ", size(InitGeometryFlag)
            ! print*,"size(tdFlag) = ", size(tdFlag)
            ! print*,"size(InittdFlag) = ", size(InittdFlag)
            ! print*,"size(nHFlag) = ", size(nHFlag)
            ! print*,"size(InitnHFlag) = ", size(InitnHFlag)
            ! print*,"size(vWidthLimits) = ", size(vWidthLimits)
            ! print*,"size(InitvWidthLimits) = ", size(InitvWidthLimits)
            ! print*,"size(vOffLimits) = ", size(vOffLimits)
            ! print*,"size(InitvOffLimits) = ", size(InitvOffLimits)
            ! print*,"size(KindOfMolecule) = ", size(KindOfMolecule)
            ! print*,"size(InitKindOfMolecule) = ", size(InitKindOfMolecule)
            ! print*,"size(LineProfileFunction) = ", size(LineProfileFunction)
            ! print*,"size(InitLineProfileFunction) = ", size(InitLineProfileFunction)
            ! print*,"size(LTEFlag) = ", size(LTEFlag)
            ! print*,"size(InitLTEFlag) = ", size(InitLTEFlag)
            ! print*,"size(ThinFlag) = ", size(ThinFlag)
            ! print*,"size(InitThinFlag) = ", size(InitThinFlag)
            ! print*,"size(SortedDistances) = ", size(SortedDistances)
            ! print*,"size(InitSortedDistances) = ", size(InitSortedDistances)
            ! print*,"myXCLASSParameter = ", myXCLASSParameter
            ! print*,"CompMoleculeIndex = ", CompMoleculeIndex
            ! print*,"GeometryFlag = ", GeometryFlag
            ! print*,"tdFlag = ", tdFlag
            ! print*,"nHFlag = ", nHFlag
            ! print*,"vWidthLimits = ", vWidthLimits
            ! print*,"vOffLimits = ", vOffLimits
            ! print*,"KindOfMolecule = ", KindOfMolecule
            ! print*,"LineProfileFunction = ", LineProfileFunction
            ! print*,"LTEFlag = ", LTEFlag
            ! print*,"ThinFlag = ", ThinFlag
            ! print*,"SortedDistances = ", SortedDistances


            !< variables for iso ratio file
            NumberOfIsomasters = InitNumberOfIsomasters                                     !< number of isomasters
            NumberOfGlobalISORatios = InitNumberOfGlobalISORatios                           !< number of global iso ratio definitions
            TotalNumberOfMolecules = InitTotalNumberOfMolecules                             !< total number of molecules including isotopologues
            IsoTableFileName = InitIsoTableFileName                                         !< path and name of the iso file
            if (allocated(InitIsoRatio)) then
                if (allocated(IsoMolecule)) then
                    deallocate(IsoMolecule, &
                               IsoMasters, &
                               IsoRatio, &
                               stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine inter.spectrum!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variables:")'
                        print '(5x, "- IsoMolecule")'
                        print '(5x, "- IsoMasters")'
                        print '(5x, "- IsoRatio")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                allocate(IsoMolecule(InitNumberOfIsomasters), &
                         IsoMasters(InitNumberOfIsomasters), &
                         IsoRatio(InitNumberOfIsomasters, 2), &
                         stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not allocate the following variables:")'
                    print '(5x, "- IsoMolecule(InitNumberOfIsomasters)")'
                    print '(5x, "- IsoMasters(InitNumberOfIsomasters)")'
                    print '(5x, "- IsoRatio(InitNumberOfIsomasters, 2)")'
                    print '(" ")'
                    print '(3x, "Used parameter:")'
                    print '(" ")'
                    print '(5x, "- InitNumberOfIsomasters  = ", I8)', InitNumberOfIsomasters
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', allocstatus
                    print '(" ")'
                endif
                IsoRatio = InitIsoRatio                                                     !< iso ratios for all isomasters
                IsoMolecule = ""
                IsoMasters = ""
                Do i = 1, InitNumberOfIsomasters
                    Do j = 1, 40
                        k1 = InitIsoMolecule(i, j)
                        if (k1 > 0) then
                            IsoMolecule(i)(j:j) = char(k1)                                  !< names of molecules for iso table
                        endif
                        k2 = InitIsoMasters(i, j)
                        if (k2 > 0) then
                            IsoMasters(i)(j:j) = char(k2)                                   !< names of isomasters for iso table
                        endif
                        if (k1 == 0 .and. k2 == 0) then
                            exit
                        endif
                    end Do
                end Do

                ! Debug:
                ! print*,"IsoMolecule = ", IsoMolecule
                ! print*,"IsoMasters = ", IsoMasters
                ! print*,"IsoRatio = ", IsoRatio
            endif
            if (allocated(InitGlobalIsoRatioParameter)) then
                if (allocated(GlobalIsoRatioParameter)) then
                    deallocate(GlobalIsoRatioParameter, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine inter.spectrum!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variable:")'
                        print '(5x, "- GlobalIsoRatioParameter")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                allocate(GlobalIsoRatioParameter(InitNumberOfIsomasters, InitNumberOfGlobalISORatios, 2), &
                         stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not allocate the following variable:")'
                    print '(5x, "- GlobalIsoRatioParameter(InitNumberOfIsomasters, InitNumberOfGlobalISORatios, 2)")'
                    print '(" ")'
                    print '(3x, "Used parameter:")'
                    print '(" ")'
                    print '(5x, "- InitNumberOfIsomasters  = ", I8)', InitNumberOfIsomasters
                    print '(5x, "- InitNumberOfGlobalISORatios  = ", I8)', InitNumberOfGlobalISORatios
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', allocstatus
                    print '(" ")'
                endif
                GlobalIsoRatioParameter = InitGlobalIsoRatioParameter

                ! Debug:
                ! print*,"GlobalIsoRatioParameter = ", GlobalIsoRatioParameter
            endif
            if (allocated(IsoNfitConversionTable)) then
                deallocate(IsoNfitConversionTable, &
                           IsoRatioConversionTable, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- IsoNfitConversionTable")'
                    print '(5x, "- IsoRatioConversionTable")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(IsoNfitConversionTable(InitTotalNumberOfMolecules, InitTotalNumberOfMolecules, 2), &
                     IsoRatioConversionTable(InitTotalNumberOfMolecules, InitTotalNumberOfMolecules), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.spectrum!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- IsoNfitConversionTable(InitTotalNumberOfMolecules, InitTotalNumberOfMolecules, 2)")'
                print '(5x, "- IsoRatioConversionTable(InitTotalNumberOfMolecules, InitTotalNumberOfMolecules)")'
                print '(" ")'
                print '(3x, "Used parameter:")'
                print '(" ")'
                print '(5x, "- InitTotalNumberOfMolecules  = ", I8)', InitTotalNumberOfMolecules
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            IsoNfitConversionTable = InitIsoNfitConversionTable                             !< iso ratios conversion table for free paramter index
            IsoRatioConversionTable = InitIsoRatioConversionTable                           !< table with iso ratios between iso master and molecule
            IsoFlag = InitIsoFlag                                                           !< flag indicating use of isotopologues

            ! Debug:
            ! print*,"NumberOfIsomasters = ", NumberOfIsomasters
            ! print*,"NumberOfGlobalISORatios = ", NumberOfGlobalISORatios
            ! print*,"TotalNumberOfMolecules = ", TotalNumberOfMolecules
            ! print*,"IsoTableFileName = ", IsoTableFileName
            ! print*,"size(IsoNfitConversionTable, 1) = ", size(IsoNfitConversionTable, 1)
            ! print*,"size(IsoNfitConversionTable, 2) = ", size(IsoNfitConversionTable, 2)
            ! print*,"size(InitIsoNfitConversionTable, 1) = ", size(InitIsoNfitConversionTable, 1)
            ! print*,"size(InitIsoNfitConversionTable, 2) = ", size(InitIsoNfitConversionTable, 2)
            ! print*,"size(IsoRatioConversionTable, 1) = ", size(IsoRatioConversionTable, 1)
            ! print*,"size(IsoRatioConversionTable, 2) = ", size(IsoRatioConversionTable, 2)
            ! print*,"size(InitIsoRatioConversionTable, 1) = ", size(InitIsoRatioConversionTable, 1)
            ! print*,"size(InitIsoRatioConversionTable, 2) = ", size(InitIsoRatioConversionTable, 2)
            ! print*,"IsoNfitConversionTable = ", IsoNfitConversionTable
            ! print*,"IsoRatioConversionTable = ", IsoRatioConversionTable
            ! print*,"IsoFlag = ", IsoFlag


            !< frequency range parameters
            NumberMolecules = InitNumberMolecules                                           !< number of molecules in the current molfits file
            TotalNumberDataPoints = InitTotalNumberDataPoints                               !< total number of data points
            MaxNumberOfMolecules = NumberMolecules + 2 * NumberOfIsomasters
            iii = max(1, InitNumberFreeParameter)
            if (allocated(StartFrequency)) then
                deallocate(StartFrequency, &
                           EndFrequency, &
                           StepFrequency, &
                           tbFlag, &
                           BackgroundTemperatureRange, &
                           TemperatureSlopeRange, &
                           BackgroundFilenNames, &
                           BackgroundFromFile, &
                           HydrogenColumnDensityRange, &
                           DustBetaRange, &
                           KappaRange, &
                           DustFilenNames, &
                           TelescopeSize, &
                           TelescopeBMIN, &
                           TelescopeBMAJ, &
                           TelescopeBPA, &
                           InterFlag, &
                           GlobalvLSR, &
                           Redshift_Range, &
                           SpectrumIndexForFreqRange, &
                           ObservationalDataList, &
                           DataPointIndexFreqRange, &
                           ConversionTableMAGIXmyXCLASSParam, &
                           DustTauFromFile, &
                           ContPhen_Range, &
                           Dust_Flag_Range, &
                           Phen_Flag_Range, &
                           MoleculeNames, &
                           NumTransFreqPerObsFreq, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- StartFrequency")'
                    print '(5x, "- EndFrequency")'
                    print '(5x, "- StepFrequency")'
                    print '(5x, "- tbFlag")'
                    print '(5x, "- BackgroundTemperatureRange")'
                    print '(5x, "- TemperatureSlopeRange")'
                    print '(5x, "- BackgroundFilenNames")'
                    print '(5x, "- BackgroundFromFile")'
                    print '(5x, "- HydrogenColumnDensityRange")'
                    print '(5x, "- DustBetaRange")'
                    print '(5x, "- KappaRange")'
                    print '(5x, "- DustFilenNames")'
                    print '(5x, "- TelescopeSize")'
                    print '(5x, "- TelescopeBMIN")'
                    print '(5x, "- TelescopeBMAJ")'
                    print '(5x, "- TelescopeBPA")'
                    print '(5x, "- InterFlag")'
                    print '(5x, "- GlobalvLSR")'
                    print '(5x, "- Redshift_Range")'
                    print '(5x, "- SpectrumIndexForFreqRange")'
                    print '(5x, "- ObservationalDataList")'
                    print '(5x, "- DataPointIndexFreqRange")'
                    print '(5x, "- ConversionTableMAGIXmyXCLASSParam")'
                    print '(5x, "- DustTauFromFile")'
                    print '(5x, "- ContPhen_Range")'
                    print '(5x, "- Dust_Flag_Range")'
                    print '(5x, "- Phen_Flag_Range")'
                    print '(5x, "- MoleculeNames")'
                    print '(5x, "- NumTransFreqPerObsFreq")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(StartFrequency(InitTotalNumberOfFrequencyRanges), &
                     EndFrequency(InitTotalNumberOfFrequencyRanges), &
                     StepFrequency(InitTotalNumberOfFrequencyRanges), &
                     tbFlag(InitTotalNumberOfFrequencyRanges), &
                     BackgroundTemperatureRange(InitTotalNumberOfFrequencyRanges), &
                     TemperatureSlopeRange(InitTotalNumberOfFrequencyRanges), &
                     BackgroundFilenNames(InitTotalNumberOfFrequencyRanges), &
                     BackgroundFromFile(InitTotalNumberDataPoints), &
                     HydrogenColumnDensityRange(InitTotalNumberOfFrequencyRanges), &
                     DustBetaRange(InitTotalNumberOfFrequencyRanges), &
                     KappaRange(InitTotalNumberOfFrequencyRanges), &
                     DustFilenNames(InitTotalNumberOfFrequencyRanges), &
                     TelescopeSize(InitTotalNumberOfFrequencyRanges), &
                     TelescopeBMIN(InitTotalNumberOfFrequencyRanges), &
                     TelescopeBMAJ(InitTotalNumberOfFrequencyRanges), &
                     TelescopeBPA(InitTotalNumberOfFrequencyRanges), &
                     InterFlag(InitTotalNumberOfFrequencyRanges), &
                     GlobalvLSR(InitTotalNumberOfFrequencyRanges), &
                     Redshift_Range(InitTotalNumberOfFrequencyRanges), &
                     SpectrumIndexForFreqRange(InitTotalNumberOfFrequencyRanges), &
                     ObservationalDataList(InitTotalNumberDataPoints, 3), &
                     DataPointIndexFreqRange(InitTotalNumberOfFrequencyRanges, 2), &
                     ConversionTableMAGIXmyXCLASSParam(iii, 2), &
                     DustTauFromFile(InitTotalNumberDataPoints), &
                     ContPhen_Range(6, InitTotalNumberOfFrequencyRanges), &
                     Dust_Flag_Range(InitTotalNumberOfFrequencyRanges), &
                     Phen_Flag_Range(InitTotalNumberOfFrequencyRanges), &
                     MoleculeNames(MaxNumberOfMolecules), &
                     NumTransFreqPerObsFreq(MaxNumberOfMolecules, InitTotalNumberDataPoints), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.spectrum!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- StartFrequency(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- EndFrequency(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- StepFrequency(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- tbFlag(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- BackgroundTemperatureRange(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- TemperatureSlopeRange(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- BackgroundFilenNames(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- BackgroundFromFile(InitTotalNumberDataPoints)")'
                print '(5x, "- HydrogenColumnDensityRange(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- DustBetaRange(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- KappaRange(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- DustFilenNames(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- TelescopeSize(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- TelescopeBMIN(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- TelescopeBMAJ(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- TelescopeBPA(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- InterFlag(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- GlobalvLSR(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- Redshift_Range(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- SpectrumIndexForFreqRange(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- ObservationalDataList(InitTotalNumberDataPoints, 3)")'
                print '(5x, "- DataPointIndexFreqRange(InitTotalNumberOfFrequencyRanges, 2)")'
                print '(5x, "- ConversionTableMAGIXmyXCLASSParam(iii, 2)")'
                print '(5x, "- DustTauFromFile(InitTotalNumberDataPoints)")'
                print '(5x, "- ContPhen_Range(6, InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- Dust_Flag_Range(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- Phen_Flag_Range(InitTotalNumberOfFrequencyRanges)")'
                print '(5x, "- MoleculeNames(MaxNumberOfMolecules)")'
                print '(5x, "- NumTransFreqPerObsFreq(MaxNumberOfMolecules, InitTotalNumberDataPoints)")'
                print '(" ")'
                print '(3x, "Used parameter:")'
                print '(" ")'
                print '(5x, "- InitTotalNumberOfFrequencyRanges  = ", I8)', InitTotalNumberOfFrequencyRanges
                print '(5x, "- MaxNumberOfMolecules          = ", I8)', MaxNumberOfMolecules
                print '(5x, "- InitNumberFreeParameter           = ", I8)', InitNumberFreeParameter
                print '(5x, "- InitTotalNumberDataPoints         = ", I8)', InitTotalNumberDataPoints
                print '(5x, "- iii                               = ", I8)', iii
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            StartFrequency = InitStartFrequency                                             !< first frequency for each frequency range
            EndFrequency = InitEndFrequency                                                 !< last frequency for each frequency range
            StepFrequency = InitStepFrequency                                               !< stepsize for each frequency range
            tbFlag = InittbFlag                                                             !< flag indicating that T_Back and T_Slope describes
                                                                                            !< continuum completely
            BackgroundTemperatureRange = InitBackgroundTemperatureRange                     !< T_Back for each frequency range
            TemperatureSlopeRange = InitTemperatureSlopeRange                               !< T_Slope for each frequency range
            BackgroundFilenNames = ""
            DustFilenNames = ""
            HydrogenColumnDensityRange = InitHydrogenColumnDensityRange                     !< nH for each frequency range
            DustBetaRange = InitDustBetaRange                                               !< beta for each frequency range
            KappaRange = InitKappaRange                                                     !< kappa for each frequency range
            TelescopeSize = InitTelescopeSize                                               !< size of telescope for each frequency range
            TelescopeBMIN = InitTelescopeBMIN                                               !< size of BMIN
            TelescopeBMAJ = InitTelescopeBMAJ                                               !< size of BMAJ
            TelescopeBPA = InitTelescopeBPA                                                 !< size of BPA
            InterFlag = InitInterFlag                                                       !< flag indicating that observation is interferrometric data
            GlobalvLSR = InitGlobalvLSR                                                     !< global velocity for each freq. range
            Redshift_Range = InitRedshift_Range                                             !< redshift for each range
            SpectrumIndexForFreqRange = InitSpectrumIndexForFreqRange                       !< store spectrum index for each frequency range
            ObservationalDataList = InitObservationalDataList                               !< list containing all observational data
            DataPointIndexFreqRange = InitDataPointIndexFreqRange                           !< index of data point of first and last freq. in freq. range


            !< (STRING ARRAY) the following lines are necessary because f2py can not handle string arrays properly
            MoleculeNames = ""
            Do i = 1, MaxNumberOfMolecules
                Do j = 1, 40
                    k = InitMoleculeNames(i, j)
                    if (k > 0) then
                        MoleculeNames(i)(j:j) = char(k)                                     !< names of molecules for iso table
                    else
                        exit
                    endif
                end Do
            end Do


            ConversionTableMAGIXmyXCLASSParam = InitConversionTableMAGIXmyXCLASSParam       !< conversion table between myXCLASS and MAGIX parameter
            DustTauFromFile = InitDustTauFromFile                                           !< array containing tau_dust from file
            BackgroundFromFile = InitBackgroundFromFile                                     !< array containing background from file
            ContPhen_Range = InitContPhen_Range                                             !< parameters for phenomenological continuum description
            Dust_Flag_Range = InitDust_Flag_Range                                           !< flag indicating global phen. settings for each range
            Phen_Flag_Range = InitPhen_Flag_Range                                           !< flag indicating global dust settings for each range
            NumTransFreqPerObsFreq = InitNumTransFreqPerObsFreq                             !< describes number of trans. freq. for each freq. channel

            ! Debug:
            ! print*,"NumberMolecules = ", NumberMolecules
            ! print*,"TotalNumberDataPoints = ", TotalNumberDataPoints
            ! print*,"MaxNumberOfMolecules = ", MaxNumberOfMolecules
            ! print*,"TotalNumberDataPoints = ", TotalNumberDataPoints
            ! print*,"size(StartFrequency) = ", size(StartFrequency)
            ! print*,"size(InitStartFrequency) = ", size(InitStartFrequency)
            ! print*,"size(EndFrequency) = ", size(EndFrequency)
            ! print*,"size(InitEndFrequency) = ", size(InitEndFrequency)
            ! print*,"size(StepFrequency) = ", size(StepFrequency)
            ! print*,"size(InitStepFrequency) = ", size(InitStepFrequency)
            ! print*,"size(tbFlag) = ", size(tbFlag)
            ! print*,"size(InittbFlag) = ", size(InittbFlag)
            ! print*,"size(BackgroundTemperatureRange) = ", size(BackgroundTemperatureRange)
            ! print*,"size(InitBackgroundTemperatureRange) = ", size(InitBackgroundTemperatureRange)
            ! print*,"size(TemperatureSlopeRange) = ", size(TemperatureSlopeRange)
            ! print*,"size(InitTemperatureSlopeRange) = ", size(InitTemperatureSlopeRange)
            ! print*,"size(BackgroundFromFile) = ", size(BackgroundFromFile)
            ! print*,"size(InitBackgroundFromFile) = ", size(InitBackgroundFromFile)
            ! print*,"size(HydrogenColumnDensityRange) = ", size(HydrogenColumnDensityRange)
            ! print*,"size(InitHydrogenColumnDensityRange) = ", size(InitHydrogenColumnDensityRange)
            ! print*,"size(DustBetaRange) = ", size(DustBetaRange)
            ! print*,"size(InitDustBetaRange) = ", size(InitDustBetaRange)
            ! print*,"size(KappaRange) = ", size(KappaRange)
            ! print*,"size(InitKappaRange) = ", size(InitKappaRange)
            ! print*,"size(TelescopeSize) = ", size(TelescopeSize)
            ! print*,"size(InitTelescopeSize) = ", size(InitTelescopeSize)
            ! print*,"size(TelescopeBMIN) = ", size(TelescopeBMIN)
            ! print*,"size(InitTelescopeBMIN) = ", size(InitTelescopeBMIN)
            ! print*,"size(TelescopeBMAJ) = ", size(TelescopeBMAJ)
            ! print*,"size(InitTelescopeBMAJ) = ", size(InitTelescopeBMAJ)
            ! print*,"size(TelescopeBPA) = ", size(TelescopeBPA)
            ! print*,"size(InitTelescopeBPA) = ", size(InitTelescopeBPA)
            ! print*,"size(InterFlag) = ", size(InterFlag)
            ! print*,"size(InitInterFlag) = ", size(InitInterFlag)
            ! print*,"size(GlobalvLSR) = ", size(GlobalvLSR)
            ! print*,"size(InitGlobalvLSR) = ", size(InitGlobalvLSR)
            ! print*,"size(Redshift_Range) = ", size(Redshift_Range)
            ! print*,"size(InitRedshift_Range) = ", size(InitRedshift_Range)
            ! print*,"size(SpectrumIndexForFreqRange) = ", size(SpectrumIndexForFreqRange)
            ! print*,"size(InitSpectrumIndexForFreqRange) = ", size(InitSpectrumIndexForFreqRange)
            ! print*,"size(ObservationalDataList, 1) = ", size(ObservationalDataList, 1)
            ! print*,"size(ObservationalDataList, 2) = ", size(ObservationalDataList, 2)
            ! print*,"size(InitObservationalDataList, 1) = ", size(InitObservationalDataList, 1)
            ! print*,"size(InitObservationalDataList, 2) = ", size(InitObservationalDataList, 2)
            ! print*,"size(DataPointIndexFreqRange, 1) = ", size(DataPointIndexFreqRange, 1)
            ! print*,"size(DataPointIndexFreqRange, 2) = ", size(DataPointIndexFreqRange, 2)
            ! print*,"size(InitDataPointIndexFreqRange, 1) = ", size(InitDataPointIndexFreqRange, 1)
            ! print*,"size(InitDataPointIndexFreqRange, 2) = ", size(InitDataPointIndexFreqRange, 2)
            ! print*,"size(ConversionTableMAGIXmyXCLASSParam, 1) = ", size(ConversionTableMAGIXmyXCLASSParam, 1)
            ! print*,"size(ConversionTableMAGIXmyXCLASSParam, 2) = ", size(ConversionTableMAGIXmyXCLASSParam, 2)
            ! print*,"size(InitConversionTableMAGIXmyXCLASSParam, 1) = ", size(InitConversionTableMAGIXmyXCLASSParam, 1)
            ! print*,"size(InitConversionTableMAGIXmyXCLASSParam, 2) = ", size(InitConversionTableMAGIXmyXCLASSParam, 2)
            ! print*,"size(DustTauFromFile) = ", size(DustTauFromFile)
            ! print*,"size(InitDustTauFromFile) = ", size(InitDustTauFromFile)
            ! print*,"size(ContPhen_Range) = ", size(ContPhen_Range)
            ! print*,"size(InitContPhen_Range) = ", size(InitContPhen_Range)
            ! print*,"size(Dust_Flag_Range) = ", size(Dust_Flag_Range)
            ! print*,"size(InitDust_Flag_Range) = ", size(InitDust_Flag_Range)
            ! print*,"size(Phen_Flag_Range) = ", size(Phen_Flag_Range)
            ! print*,"size(InitPhen_Flag_Range) = ", size(InitPhen_Flag_Range)
            ! print*,"size(MoleculeNames) = ", size(MoleculeNames)
            ! print*,"size(InitMoleculeNames) = ", size(InitMoleculeNames)
            ! print*,"size(NumTransFreqPerObsFreq, 1) = ", size(NumTransFreqPerObsFreq, 1)
            ! print*,"size(NumTransFreqPerObsFreq, 2) = ", size(NumTransFreqPerObsFreq, 2)
            ! print*,"size(InitNumTransFreqPerObsFreq, 1) = ", size(InitNumTransFreqPerObsFreq, 1)
            ! print*,"size(InitNumTransFreqPerObsFreq, 2) = ", size(InitNumTransFreqPerObsFreq, 2)
            ! print*,"StartFrequency = ", StartFrequency
            ! print*,"EndFrequency = ", EndFrequency
            ! print*,"StepFrequency = ", StepFrequency
            ! print*,"tbFlag = ", tbFlag
            ! print*,"BackgroundTemperatureRange = ", BackgroundTemperatureRange
            ! print*,"TemperatureSlopeRange = ", TemperatureSlopeRange
            ! print*,"BackgroundFromFile = ", BackgroundFromFile
            ! print*,"InitBackgroundFromFile = ", InitBackgroundFromFile
            ! print*,"HydrogenColumnDensityRange = ", HydrogenColumnDensityRange
            ! print*,"DustBetaRange = ", DustBetaRange
            ! print*,"KappaRange = ", KappaRange
            ! print*,"TelescopeSize = ", TelescopeSize
            ! print*,"TelescopeBMIN = ", TelescopeBMIN
            ! print*,"TelescopeBMAJ = ", TelescopeBMAJ
            ! print*,"TelescopeBPA = ", TelescopeBPA
            ! print*,"InterFlag = ", InterFlag
            ! print*,"GlobalvLSR = ", GlobalvLSR
            ! print*,"Redshift_Range = ", Redshift_Range
            ! print*,"SpectrumIndexForFreqRange = ", SpectrumIndexForFreqRange
            ! print*,"ObservationalDataList = ", ObservationalDataList
            ! print*,"DataPointIndexFreqRange = ", DataPointIndexFreqRange
            ! print*,"ConversionTableMAGIXmyXCLASSParam = ", ConversionTableMAGIXmyXCLASSParam
            ! print*,"iii = ", iii
            ! print*,"DustTauFromFile = ", DustTauFromFile
            ! print*,"ContPhen_Range = ", ContPhen_Range
            ! print*,"Dust_Flag_Range = ", Dust_Flag_Range
            ! print*,"Phen_Flag_Range = ", Phen_Flag_Range
            ! print*,"NumTransFreqPerObsFreq = ", NumTransFreqPerObsFreq


            !< allocate memory for distance parameters
            NumberMolfitMolecules = InitNumberMolfitMolecules                               !< number of molecules in molfit file only
            if (allocated(NumberComponentsPerMolecule)) then
                deallocate(NumberComponentsPerMolecule, &
                           PureDistances, &
                           DistanceOrderingArray, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- NumberComponentsPerMolecule")'
                    print '(5x, "- PureDistances")'
                    print '(5x, "- DistanceOrderingArray")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(NumberComponentsPerMolecule(0:InitNumberMolfitMolecules), &
                     PureDistances(InitTotalNumberComponents), &
                     DistanceOrderingArray(InitTotalNumberComponents, InitTotalNumberComponents), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.spectrum!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- NumberComponentsPerMolecule(0:InitNumberMolfitMolecules)")'
                print '(5x, "- PureDistances(InitTotalNumberComponents)")'
                print '(5x, "- DistanceOrderingArray(InitTotalNumberComponents, InitTotalNumberComponents)")'
                print '(" ")'
                print '(3x,"Used parameter:")'
                print '(" ")'
                print '(5x, "- InitNumberMolfitMolecules = ", I8)', InitNumberMolfitMolecules
                print '(5x, "- InitTotalNumberComponents = ", I8)', InitTotalNumberComponents
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            NumberComponentsPerMolecule = InitNumberComponentsPerMolecule                   !< number of components per molecule
            PureDistances = InitPureDistances                                               !< array containing pure distances
            DistanceOrderingArray = InitDistanceOrderingArray                               !< translation table between pure distances and comp. indices

            ! Debug:
            ! print*,"InitNumberMolfitMolecules = ", InitNumberMolfitMolecules
            ! print*,"InitTotalNumberComponents = ", InitTotalNumberComponents
            ! print*,"size(NumberComponentsPerMolecule) = ", size(NumberComponentsPerMolecule)
            ! print*,"size(InitNumberComponentsPerMolecule) = ", size(InitNumberComponentsPerMolecule)
            ! print*,"size(PureDistances) = ", size(PureDistances)
            ! print*,"size(InitPureDistances) = ", size(InitPureDistances)
            ! print*,"size(DistanceOrderingArray) = ", size(DistanceOrderingArray)
            ! print*,"size(InitDistanceOrderingArray) = ", size(InitDistanceOrderingArray)
            ! print*,"NumberComponentsPerMolecule = ", NumberComponentsPerMolecule
            ! print*,"PureDistances = ", PureDistances
            ! print*,"DistanceOrderingArray = ", DistanceOrderingArray


            !< allocate memory for emission and absorption functions
            if (allocated(InitEmsAbsFunc)) then
                if (allocated(EmsAbsFunc)) then
                    deallocate(EmsAbsFunc, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine inter.spectrum!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variable:")'
                        print '(5x, "- EmsAbsFunc")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                allocate(EmsAbsFunc(InitTotalNumberDataPoints, InitNumberDistances, 2), stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not allocate the following variable:")'
                    print '(5x, "- EmsAbsFunc(InitTotalNumberDataPoints, InitNumberDistances, 2)")'
                    print '(" ")'
                    print '(3x,"Used parameter:")'
                    print '(" ")'
                    print '(5x, "- InitTotalNumberDataPoints = ", I8)', InitTotalNumberDataPoints
                    print '(5x, "- InitNumberDistances = ", I8)', InitNumberDistances
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', allocstatus
                    print '(" ")'
                endif
                EmsAbsFunc = InitEmsAbsFunc

                ! Debug:
                ! print*,"EmsAbsFunc = ", EmsAbsFunc
            endif
            EmsAbsPATH = InitEmsAbsPATH                                                     !< path of directory containing files describing em. and abs.
            UseEmAbsFuncFlag = InitUseEmAbsFuncFlag                                         !< flag indicating usage of emission/absorption file(s)

            ! Debug:
            ! print*,"InitTotalNumberDataPoints = ", InitTotalNumberDataPoints
            ! print*,"InitNumberDistances = ", InitNumberDistances
            ! print*,"size(EmsAbsFunc, 1) = ", size(EmsAbsFunc, 1)
            ! print*,"size(InitEmsAbsFunc, 1) = ", size(InitEmsAbsFunc, 1)
            ! print*,"size(EmsAbsFunc, 2) = ", size(EmsAbsFunc, 2)
            ! print*,"size(InitEmsAbsFunc, 2) = ", size(InitEmsAbsFunc, 2)
            ! print*,"EmsAbsFunc = ", EmsAbsFunc
            ! print*,"EmsAbsPATH = ", EmsAbsPATH
            ! print*,"UseEmAbsFuncFlag = ", UseEmAbsFuncFlag
            ! print*,"allocated(LayerMap) = ", allocated(LayerMap)
            ! print*,"allocated(ConfigList) = ", allocated(ConfigList)
            ! print*,"allocated(ConfigIndex) = ", allocated(ConfigIndex)
            ! print*,"allocated(GausssianBeamMap) = ", allocated(GausssianBeamMap)


            !< variables for sub-beam description
            if (allocated(InitConfigIndex)) then
                if (allocated(LayerMap)) then
                    deallocate(LayerMap, &
                               ConfigList, &
                               ConfigIndex, &
                               GausssianBeamMap, &
                               stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine inter.spectrum!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variables:")'
                        print '(5x, "- LayerMap")'
                        print '(5x, "- ConfigList")'
                        print '(5x, "- ConfigIndex")'
                        print '(5x, "- GausssianBeamMap")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                allocate(LayerMap(NumberModelPixelXX, NumberModelPixelYY, TotalNumberComponents), &
                         ConfigList(NumberModelPixelXX * NumberModelPixelYY, TotalNumberComponents), &
                         ConfigIndex(NumberModelPixelXX, NumberModelPixelYY), &
                         GausssianBeamMap(NumberModelPixelXX, NumberModelPixelYY), &
                         stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not allocate the following variables:")'
                    print '(5x, "- LayerMap(NumberModelPixelXX, NumberModelPixelYY, TotalNumberComponents)")'
                    print '(5x, "- ConfigList(NumberModelPixelXX * NumberModelPixelYY, TotalNumberComponents)")'
                    print '(5x, "- ConfigIndex(NumberModelPixelXX, NumberModelPixelYY)")'
                    print '(5x, "- GausssianBeamMap(NumberModelPixelXX, NumberModelPixelYY)")'
                    print '(" ")'
                    print '(3x,"Used parameter:")'
                    print '(" ")'
                    print '(5x, "- NumberModelPixelXX = ", I8)', NumberModelPixelXX
                    print '(5x, "- NumberModelPixelYY = ", I8)', NumberModelPixelYY
                    print '(5x, "- TotalNumberComponents = ", I8)', TotalNumberComponents
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', allocstatus
                    print '(" ")'
                endif
                LayerMap = InitLayerMap                                                     !< indicates, which components are taken into account
                ConfigList = InitConfigList                                                 !< list of unique pixel combination
                ConfigIndex = InitConfigIndex                                               !< configuration index for model map
                GausssianBeamMap = InitGausssianBeamMap                                     !< map describing elliptical rotated Gaussian beam

                ! Debug:
                ! print*,"LayerMap = ", LayerMap
                ! print*,"ConfigList = ", ConfigList
                ! print*,"ConfigIndex = ", ConfigIndex
                ! print*,"GausssianBeamMap = ", GausssianBeamMap
            endif
            SubBeamDescriptionFlag = InitSubBeamDescriptionFlag                             !< flag indicating sub-beam description
            SizeOfPixel_deg = InitSizeOfPixel_deg                                           !< size of pixel

            ! Debug:
            ! print*,"SubBeamDescriptionFlag = ", SubBeamDescriptionFlag
            ! print*,"SizeOfPixel_deg = ", SizeOfPixel_deg


            !< variables of subroutine GetPartitionFunction
            NumberMoleculePartFunc = InitNumberMoleculePartFunc                             !< number of entries in table PartitionFunction
            Firsti = InitFirsti                                                             !< working variables for interpolation
            Lasti = InitLasti                                                               !< working variables for interpolation
            stepi = Initstepi                                                               !< working variables for interpolation
            TempLow = InitTempLow                                                           !< working variables for extrapolation of part. func.
            TempHigh = InitTempHigh                                                         !< working variables for extrapolation of part. func.
            ColumnNameForNamePartFunc = InitColumnNameForNamePartFunc                       !< name of column including the names of molecules
            if (allocated(MolNamePartFunc)) then
                deallocate(MolNamePartFunc, &
                           lgQ, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- MolNamePartFunc")'
                    print '(5x, "- lgQ")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(MolNamePartFunc(NumberMoleculePartFunc), &
                     lgQ(NumberOfTemperatures, NumberMoleculePartFunc), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.spectrum!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- MolNamePartFunc(NumberMoleculePartFunc, 40)")'
                print '(5x, "- lgQ(NumberOfTemperatures, NumberMoleculePartFunc)")'
                print '(" ")'
                print '(3x,"Used parameter:")'
                print '(" ")'
                print '(5x, "- NumberMoleculePartFunc = ", I8)', NumberMoleculePartFunc
                print '(5x, "- NumberOfTemperatures = ", I8)', NumberOfTemperatures
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            lgQ = InitlgQ                                                                   !< lgQ entries of table PartitionFunction


            !< (STRING ARRAY) the following lines are necessary because f2py can not handle string arrays properly
            MolNamePartFunc = ""
            Do i = 1, NumberMoleculePartFunc
                Do j = 1, 40
                    k = InitMolNamePartFunc(i, j)
                    if (k > 0) then
                        MolNamePartFunc(i)(j:j) = char(k)                                   !< molecule names of table PartitionFunction
                    else
                        exit
                    endif
                end Do
            end Do

            ! Debug:
            ! print*,"NumberOfTemperatures = ", NumberOfTemperatures
            ! print*,"NumberMoleculePartFunc = ", NumberMoleculePartFunc
            ! print*,"size(MolNamePartFunc) = ", size(MolNamePartFunc)
            ! print*,"size(InitMolNamePartFunc, 1) = ", size(InitMolNamePartFunc, 1)
            ! print*,"size(lgQ) = ", size(lgQ)
            ! print*,"size(InitlgQ) = ", size(InitlgQ)
            ! print*,"MolNamePartFunc = ", MolNamePartFunc
            ! print*,"lgQ = ", lgQ
            ! print*,"Firsti = ", Firsti
            ! print*,"Lasti = ", Lasti
            ! print*,"stepi = ", stepi
            ! print*,"TempLow = ", TempLow
            ! print*,"TempHigh = ", TempHigh
            ! print*,"TempPartFunc = ", TempPartFunc
            ! print*,"ColumnNameForNamePartFunc = ", trim(adjustl(ColumnNameForNamePartFunc))
            ! print*,"ColumnNamesPartFunc = ", ColumnNamesPartFunc


            !< variables of subroutine GetTransitionParameters
            if (allocated(NumEntriesRanges)) then
                deallocate(NumEntriesRanges, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- NumEntriesRanges")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(NumEntriesRanges(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.spectrum!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- NumEntriesRanges(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules)")'
                print '(" ")'
                print '(3x,"Used parameter:")'
                print '(" ")'
                print '(5x, "- TotalNumberOfFrequencyRanges = ", I8)', TotalNumberOfFrequencyRanges
                print '(5x, "- TotalNumberOfMolecules = ", I8)', TotalNumberOfMolecules
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            NumEntriesRanges = InitNumEntriesRanges                                         !< number of transitions for each freq. range and molecule

            ! Debug:
            ! print*,"TotalNumberOfFrequencyRanges = ", TotalNumberOfFrequencyRanges
            ! print*,"TotalNumberOfMolecules = ", TotalNumberOfMolecules
            ! print*,"size(NumEntriesRanges, 1) = ", size(NumEntriesRanges, 1)
            ! print*,"size(InitNumEntriesRanges, 1) = ", size(InitNumEntriesRanges, 1)
            ! print*,"size(NumEntriesRanges, 2) = ", size(NumEntriesRanges, 2)
            ! print*,"size(InitNumEntriesRanges, 2) = ", size(InitNumEntriesRanges, 2)
            ! print*,"NumEntriesRanges = ", NumEntriesRanges


            TotalNumberOfTransitions = InitTotalNumberOfTransitions                         !< number of entries in table RadTrans
            if (allocated(MolecularData)) then
                deallocate(MolecularData, &
                           MolecularDataIndices, &
                           LowerQNString, &
                           UpperQNString, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variables:")'
                    print '(5x, "- MolecularData")'
                    print '(5x, "- MolecularDataIndices")'
                    print '(5x, "- LowerQNString")'
                    print '(5x, "- UpperQNString")'
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            j = max(1, TotalNumberOfTransitions)
            allocate(MolecularData(j, 9), &
                     MolecularDataIndices(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2), &
                     LowerQNString(j), &
                     UpperQNString(j), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.spectrum!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- MolecularData(j, 9)")'
                print '(5x, "- MolecularDataIndices(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2)")'
                print '(5x, "- LowerQNString(j, 60)")'
                print '(5x, "- UpperQNString(j, 60)")'
                print '(" ")'
                print '(3x,"Used parameter:")'
                print '(" ")'
                print '(5x, "- TotalNumberOfFrequencyRanges = ", I8)', TotalNumberOfFrequencyRanges
                print '(5x, "- TotalNumberOfMolecules = ", I8)', TotalNumberOfMolecules
                print '(5x, "- j = ", I8)', j
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            MolecularData = InitMolecularData                                               !< array containing the molecular data for all molecules and
                                                                                            !< all freqency ranges
            MolecularDataIndices = InitMolecularDataIndices                                 !< start and end index for each molecule and frequency range
                                                                                            !< within MolecularData variable


            !< (STRING ARRAY) the following lines are necessary because f2py can not handle string arrays properly
            LowerQNString = ""
            UpperQNString = ""
            Do i = 1, j
                Do k = 1, 60
                    k1 = InitLowerQNString(i, k)                                            !< quantum number string of lower state
                    if (k1 > 0) then
                        LowerQNString(i)(k:k) = char(k1)
                    endif
                    k2 = InitUpperQNString(i, k)                                            !< quantum number string of upper state
                    if (k2 > 0) then
                        UpperQNString(i)(k:k) = char(k2)
                    endif
                    if (k1 == 0 .and. k2 == 0) then
                        exit
                    endif
                end Do
            end Do

            ! Debug:
            ! print*,"TotalNumberOfFrequencyRanges = ", TotalNumberOfFrequencyRanges
            ! print*,"TotalNumberOfMolecules = ", TotalNumberOfMolecules
            ! print*,"j = ", j
            ! print*,"size(MolecularData, 1) = ", size(MolecularData, 1)
            ! print*,"size(InitMolecularData, 1) = ", size(InitMolecularData, 1)
            ! print*,"size(MolecularData, 2) = ", size(MolecularData, 2)
            ! print*,"size(InitMolecularData, 2) = ", size(InitMolecularData, 2)
            ! print*,"size(MolecularDataIndices, 1) = ", size(MolecularDataIndices, 1)
            ! print*,"size(InitMolecularDataIndices, 1) = ", size(InitMolecularDataIndices, 1)
            ! print*,"size(MolecularDataIndices, 2) = ", size(MolecularDataIndices, 2)
            ! print*,"size(InitMolecularDataIndices, 2) = ", size(InitMolecularDataIndices, 2)
            ! print*,"size(LowerQNString) = ", size(LowerQNString)
            ! print*,"size(InitLowerQNString) = ", size(InitLowerQNString)
            ! print*,"size(UpperQNString) = ", size(UpperQNString)
            ! print*,"size(InitUpperQNString) = ", size(InitUpperQNString)
            ! print*,"MolecularData = ", MolecularData
            ! print*,"MolecularDataIndices = ", MolecularDataIndices
            ! print*,"LowerQNString = ", LowerQNString
            ! print*,"UpperQNString = ", UpperQNString


            !< variables for sqlite3 database
            MinNumTransitionsSQL = InitMinNumTransitionsSQL                                 !< min number of transitions which are considered in SQL query
            MaxNumTransitionsSQL = InitMaxNumTransitionsSQL                                 !< max number of transitions which are considered in SQL query
            OrderTransSQL = InitOrderTransSQL                                               !< order or transition considered in SQL query
            MaxElowSQL = InitMaxElowSQL                                                     !< highest lower energy of transitions considered in SQL query
            MingASQL = InitMingASQL                                                         !< min. intensity of transitions considered in SQL query
            NameOfPartFuncTable = InitNameOfPartFuncTable                                   !< name of table for partition function
            NameOfRadTransTable = InitNameOfRadTransTable                                   !< name of table for rad. trans.
            dbName = InitdbName                                                             !< name of database

            ! Debug:
            ! print*,"MinNumTransitionsSQL = ", MinNumTransitionsSQL
            ! print*,"MaxNumTransitionsSQL = ", MaxNumTransitionsSQL
            ! print*,"OrderTransSQL = ", OrderTransSQL
            ! print*,"MaxElowSQL = ", MaxElowSQL
            ! print*,"MingASQL = ", MingASQL
            ! print*,"NameOfPartFuncTable = ", trim(adjustl(NameOfPartFuncTable))
            ! print*,"NameOfRadTransTable = ", trim(adjustl(NameOfRadTransTable))
            ! print*,"dbName = ", trim(adjustl(dbName))


            !< variables for rad-trans parameter
            TotalNumTransFreq = InitTotalNumTransFreq                                       !< number of doppler-shifted trans. freq. for curr. layer
            Column300KPartFunc = InitColumn300KPartFunc                                     !< number of column, which contains part. func. @ 300 K
            ColumnNameForNameTransitions = InitColumnNameForNameTransitions                 !< name of column in table transitions for molecule name
            ColumnNameForFreqTransitions = InitColumnNameForFreqTransitions                 !< name of column in table transitions for frequency
            ColumnNameForIntTransitions = InitColumnNameForIntTransitions                   !< name of column in table transitions for intensity
            ColumnNameForEinsteinATransitions = InitColumnNameForEinsteinATransitions       !< name of column in table transitions for Einstein A coef.
            ColumnNameForFreqErrTransitions = InitColumnNameForFreqErrTransitions           !< name of column in table transitions for error freq.
            ColumnNameForELowTransitions = InitColumnNameForELowTransitions                 !< name of column in table transitions for E_low
            ColumnNameForgUpTransitions = InitColumnNameForgUpTransitions                   !< name of column in table transitions for Upper State degen.
            ColumnNameForLowerQNTransitions = InitColumnNameForLowerQNTransitions           !< name of column in table transitions for lower state QN
            ColumnNameForUpperQNTransitions = InitColumnNameForUpperQNTransitions           !< name of column in table transitions for upper state QN
            if (allocated(InitCentralOpticalDepth)) then
                if (allocated(CentralOpticalDepth)) then
                    deallocate(CentralOpticalDepth, &
                               DopplerShiftedTransFreq, &
                               DopplerShiftedTransFreqIndex, &
                               stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "Error in subroutine inter.spectrum!")'
                        print '(" ")'
                        print '(3x, "Can not deallocate the following variables:")'
                        print '(5x, "- CentralOpticalDepth")'
                        print '(5x, "- DopplerShiftedTransFreq")'
                        print '(5x, "- DopplerShiftedTransFreqIndex")'
                        print '(" ")'
                        print '(" ")'
                        print '(3x, "deallocstatus = ", I8)', deallocstatus
                        print '(" ")'
                    endif
                endif
                i = max0(1, TotalNumTransFreq)
                allocate(CentralOpticalDepth(i, 2), &
                         DopplerShiftedTransFreq(i), &
                         DopplerShiftedTransFreqIndex(i, 6), &
                         stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not allocate the following variables:")'
                    print '(5x, "- CentralOpticalDepth(i, 2)")'
                    print '(5x, "- DopplerShiftedTransFreq(i)")'
                    print '(5x, "- DopplerShiftedTransFreqIndex(i, 6)")'
                    print '(" ")'
                    print '(3x,"Used parameter:")'
                    print '(" ")'
                    print '(5x, "- i = ", I8)', i
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "allocstatus = ", I8)', allocstatus
                    print '(" ")'
                endif
                CentralOpticalDepth = InitCentralOpticalDepth
                DopplerShiftedTransFreq = InitDopplerShiftedTransFreq
                DopplerShiftedTransFreqIndex = InitDopplerShiftedTransFreqIndex

                ! Debug:
                ! print*,"CentralOpticalDepth = ", CentralOpticalDepth
                ! print*,"DopplerShiftedTransFreq = ", DopplerShiftedTransFreq
                ! print*,"DopplerShiftedTransFreqIndex = ", DopplerShiftedTransFreqIndex
            endif

            ! Debug:
            ! print*,"TotalNumTransFreq = ", TotalNumTransFreq
            ! print*,"Column300KPartFunc = ", Column300KPartFunc
            ! print*,"ColumnNameForNameTransitions = ", trim(adjustl(ColumnNameForNameTransitions))
            ! print*,"ColumnNameForFreqTransitions = ", trim(adjustl(ColumnNameForFreqTransitions))
            ! print*,"ColumnNameForIntTransitions = ", trim(adjustl(ColumnNameForIntTransitions))
            ! print*,"ColumnNameForEinsteinATransitions = ", trim(adjustl(ColumnNameForEinsteinATransitions))
            ! print*,"ColumnNameForFreqErrTransitions = ", trim(adjustl(ColumnNameForFreqErrTransitions))
            ! print*,"ColumnNameForELowTransitions = ", trim(adjustl(ColumnNameForELowTransitions))
            ! print*,"ColumnNameForgUpTransitions = ", trim(adjustl(ColumnNameForgUpTransitions))
            ! print*,"ColumnNameForLowerQNTransitions = ", trim(adjustl(ColumnNameForLowerQNTransitions))
            ! print*,"ColumnNameForUpperQNTransitions = ", trim(adjustl(ColumnNameForUpperQNTransitions))


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< call XCLASS core routines


            !< store parameters for obs. data
            TotalNumberDataPoints = sum(Initlengthexpdata)
            if (allocated(lengthexpdata)) then
                deallocate(lengthexpdata, &
                           expdatax, &
                           expdatay, &
                           expdatae, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variable:")'
                    print '(5x, "- lengthexpdata")'
                    print '(5x, "- expdatax")'
                    print '(5x, "- expdatay")'
                    print '(5x, "- expdatae")'
                    print '(" ")'
                    print '(5x, "- TotalNumberDataPoints = ", I8)', TotalNumberDataPoints
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            i = size(Initlengthexpdata)
            j = maxval(Initlengthexpdata)
            allocate(lengthexpdata(i), expdatax(i, j, 1), expdatay(i, j, 1), &
                     expdatae(i, j, 1), stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.ReadXMLInstanceFile!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variables:")'
                print '(5x, "- lengthexpdata(i)")'
                print '(5x, "- expdatax(i, j, 1)")'
                print '(5x, "- expdatay(i, j, 1)")'
                print '(5x, "- expdatae(i, j, 1)")'
                print '(" ")'
                print '(3x, "Used parameter:")'
                print '(" ")'
                print '(5x, "- i = ", I8)', i
                print '(5x, "- j = ", I8)', j
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            lengthexpdata = Initlengthexpdata
            expdatax = Initexpdatax
            expdatay = Initexpdatay
            expdatae = Initexpdatae
            NumberExpFiles = i

            ! Debug:
            ! print*,"i = ", i
            ! print*,"j = ", j


            !< allocate some variables variable
            if (allocated(ModelFuncList)) then
                deallocate(ModelFuncList, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "Error in subroutine inter.spectrum!")'
                    print '(" ")'
                    print '(3x, "Can not deallocate the following variable:")'
                    print '(5x, "- ModelFuncList")'
                    print '(" ")'
                    print '(5x, "- TotalNumberDataPoints = ", I8)', TotalNumberDataPoints
                    print '(" ")'
                    print '(" ")'
                    print '(3x, "deallocstatus = ", I8)', deallocstatus
                    print '(" ")'
                endif
            endif
            allocate(ModelFuncList(TotalNumberDataPoints), stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(" ")'
                print '(3x, "Error in subroutine inter.spectrum!")'
                print '(" ")'
                print '(3x, "Can not allocate the following variable:")'
                print '(5x, "- ModelFuncList(TotalNumberDataPoints)")'
                print '(" ")'
                print '(3x,"Used parameter:")'
                print '(" ")'
                print '(5x, "- TotalNumberDataPoints = ", I8)', TotalNumberDataPoints
                print '(" ")'
                print '(" ")'
                print '(3x, "allocstatus = ", I8)', allocstatus
                print '(" ")'
            endif
            ModelFuncList = 0.d0


            !< define some required parameters
            ModelFunctionFlag = .true.
            chi2Value = 0.d0
            ThreadNumber = 0
            LogFlag = .false.                                                                  !< write log information to files

            ! Debug:
            ! print*,"NumberFreeParameter = ", NumberFreeParameter
            ! print*,"FitParameterVector = ", FitParameterVector
            ! print*,"myXCLASSParameter = ", myXCLASSParameter
            ! print*,"IsoRatioConversionTable = ", IsoRatioConversionTable


            !< determine spectrum
            call ModelCalcSpectrum(NumberFreeParameter, FitParameterVector, &
                                   ModelFunctionFlag, TotalNumberComponents, &
                                   chi2Value, CompMoleculeIndex, myXCLASSParameter, &
                                   TotalNumberOfMolecules, IsoRatioConversionTable, &
                                   TotalNumberDataPoints, ModelFuncList, ThreadNumber)


            !< get emission and absorption functions
            if (EmAbsFlag) then
                InitTauEmAbsArray = TauEmAbsArray
            endif


            !< get opacities and intensities
            if (AllOutputFilesFlag) then
                InitIntPerCompArray = IntPerCompArray
                InitTauPerCompArray = TauPerCompArray
            endif


            !< free parameters
            call myXCLASSParamFree(deallocstatus)


            !< we're done
            return
        end subroutine spectrum
end Module inter
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!< main program
program myXCLASS

    implicit none

end program myXCLASS
!*********************************************************************************************************************************************************
