!*********************************************************************************************************************************************************
!>  Module: myXCLASSCore
!>
!>
!>  This module contains the core subroutines for the XCLASS package
!>  Copyright (C) 2015 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine RADEXSPLCOEFF:                     applications of cubic-spline interpolation
!>      - subroutine RADEXsplintrp:                     does numerical quadrature through use of the spline coefficients determined for a set of points
!>                                                      in routine splcoeff
!>      - subroutine CheckMolNames:                     checks, if a molecule name is already known
!>      - subroutine strlen:                            subroutine to determine last non blank and non char(0) character of string st
!>      - subroutine countsubstring:                    count occurrences of a substring
!>      - subroutine ConverStringToLowerCase:           subroutine to convert string to lower case
!>      - subroutine ImportDustFile:                    import dust file and interpolate at given frequency points
!>      - subroutine ImportBackgroundFile:              import background file and interpolate at given frequency points
!>      - subroutine ImportEmsAbsFile:                  import files describing emission and absorption and interpolate at given frequency points
!>      - subroutine CheckCollisionPartner:             check name of collision partner
!>      - subroutine GetmyXCLASSParameter:              reads parameter for myXCLASS from paramset variable
!>      - subroutine GetPartitionFunction:              gets partition function from sqlite3 database
!>      - subroutine GetTransitionParameters:           gets information for transitions from sqlite3 database table linecat
!>      - subroutine RemoveDublicates:                  remove duplicates in a 1d list
!>      - subroutine myXCLASSInitVar:                   initialize myXCLASS variables
!>      - subroutine InterpolateQ:                      interpolates partition function for a given temperature
!>      - function GaussianFunction:                    return a 1-dimensional Gaussian function
!>      - subroutine subroutine InterPol1D:             interpolate a 1D function on an irregular grid
!>      - subroutine subroutine InterPol2D:             interpolate a 2D function on an irregular grid
!>      - subroutine LineProfile:                       calculates the intensity for a given optical depth using the user-defined line profile
!>      - subroutine HeapSort:                          sort an array using heapsort algorithm and apply changes to second array
!>      - subroutine siftdown:                          helper subroutine for heapsort algorithm
!>      - subroutine ComputeCentralOpticalDepthLTE:     calculates the optical depth at the given doppler-shifted transition frequency for
!>                                                      molecules (in LTE) and RRLs (in LTE)
!>      - subroutine ComputeCentralOpticalDepth:        calculates the optical depth at the doppler-shifted transition frequencies
!>      - subroutine DetectionEquation:                 calculates the detection equation for a given frequency
!>      - subroutine ResampleChannel:                   defines additional frequency points which are used for integration of spectrum over channel width
!>      - subroutine CalcLocalOverlapTransFreq:         calculates all Doppler-shifted transition freq. and their central optical depths for current layer
!>      - subroutine DrawEllipse:                       draw an rotated ellipse
!>      - subroutine SetPixel:                          set pixel in model map and check if indices are within boundaries
!>      - subroutine CreatePixelMap:                    determine map of model pixel for each component
!>      - subroutine ComputeGaussianBeamMap:            compute map describing elliptical rotated Gaussian beam
!>      - subroutine ConvolveModelMap:                  convolve model map with beam
!>      - subroutine PhenContDescription:               computes phen. continuum description
!>      - subroutine CalcModelPixelSpectrum:            calculates the entire spectrum for a given model pixel
!>      - subroutine ModelCalcSpectrum:                 calculates the myXCLASS spectrum for a given parameter vector
!>      - subroutine myXCLASSParamFree:                 free memory used by maXCLASS variables
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    2015-02-04  initial version
!>  T. Moeller    2015-06-11  improved dust handling, introduction of dust temperature
!>  T. Moeller    2017-12-01  add Non-LTE description (using RADEX routines)
!>  T. Moeller    2018-02-16  add local overlap description
!>  T. Moeller    2018-03-12  add sub-beam description
!>  T. Moeller    2018-12-27  add background file handling
!>  T. Moeller    2019-12-04  add import and export of emission and absorption function
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module myXCLASSCore
    !> declare variables which are used by functions and subroutines of this module

    use Variables
    use FunctionCalling
    use, intrinsic :: iso_c_binding


    implicit none


    !< general variables
    integer :: ErrChannel                                                                   !< current error channel
    integer :: ErrChannelIndex                                                              !< loop variable for error handling
    integer, parameter :: DebugChannel = 98765                                              !< channel for debugging
    integer, dimension(2) :: AllErrChannels = (/0, logchannel/)                             !< list of all error channels
    real*8 :: tt1, tt2                                                                      !< working variables for debugging: used for timing
    real*8, parameter :: debug_freq = 569937.147d0                                          !< frequency for debugging
    logical :: UseRegistrationXMLFile                                                       !< use registration XML file
    logical :: AllOutputFilesFlag                                                           !< flag for writing intensities and optical depths to file
    logical :: EmAbsFlag                                                                    !< flag for writing emission and absorption to file
    logical :: IntegrationFlag                                                              !< flag indicating integration over channel width
    logical :: LocalOverlapFlag                                                             !< flag indicating is taken into account
    logical :: NoSubBeamFlag                                                                !< flag for deactivating sub-beam description
    character(len = 10) :: ParallelizationMethod                                            !< string describing the parallelization method
    character(len = 8192) :: XCLASSRootDir                                                  !< path of XCLASS root directory


    !< variables for molfit file
    integer :: TotalNumberDataPoints                                                        !< total number of data points
    integer :: NumberMolecules                                                              !< number of molecules in the current molfit file
    integer :: TotalNumberComponents                                                        !< counter for total number of components of all molecules
    integer :: TotalNumberOfFrequencyRanges                                                 !< total number of frequency ranges
    integer :: TotalNumberOfMolecules                                                       !< total number of molecules including isotopologues
    integer :: NumberMolfitMolecules                                                        !< number of molecules in molfit file only
    integer :: NumberDistances                                                              !< number of unequal distances defined in molfit file
    integer :: NumberModelPixelXX                                                           !< number of pixel for sub-beam modeling along x-direction
    integer :: NumberModelPixelYY                                                           !< number of pixel for sub-beam modeling along y-direction
    integer, allocatable, dimension(:) :: CompMoleculeIndex                                 !< molecule index for each component
    integer, allocatable, dimension(:) :: SpectrumIndexForFreqRange                         !< store spectrum index for each frequency range
    integer, allocatable, dimension(:) :: NumberComponentsPerMolecule                       !< number of components per molecule
    integer, allocatable, dimension(:) :: KindOfMolecule                                    !< defines kind of molecule, i.e. molecule, rrl, etc.
    integer, allocatable, dimension(:) :: GeometryFlag                                      !< defines geometry of each component / layer
    integer, allocatable, dimension(:) :: LineProfileFunction                               !< defines line profile function for each component
    integer, allocatable, dimension(:, :) :: NumTransFreqPerObsFreq                         !< describes number of trans. freq. for each freq. channel
    integer, allocatable, dimension(:, :) :: ConversionTableMAGIXmyXCLASSParam              !< conversion table between myXCLASS and MAGIX parameter
    integer, allocatable, dimension(:, :) :: DataPointIndexFreqRange                        !< index of data point of first and last freq. in freq. range
    integer, allocatable, dimension(:, :) :: DistanceOrderingArray                          !< translation table between pure distances and comp. indices
    integer, allocatable, dimension(:, :) :: ConfigIndex                                    !< configuration index for model map
    real*8 :: SizeOfPixel_deg                                                               !< size of pixel
    real*8, allocatable, dimension(:) :: StartFrequency                                     !< first frequency for each frequency range
    real*8, allocatable, dimension(:) :: EndFrequency                                       !< last frequency for each frequency range
    real*8, allocatable, dimension(:) :: StepFrequency                                      !< step size for each frequency range
    real*8, allocatable, dimension(:) :: TelescopeSize                                      !< size of telescope for each frequency range
    real*8, allocatable, dimension(:) :: TelescopeBMIN                                      !< size of BMIN
    real*8, allocatable, dimension(:) :: TelescopeBMAJ                                      !< size of BMAJ
    real*8, allocatable, dimension(:) :: TelescopeBPA                                       !< size of BPA
    real*8, allocatable, dimension(:) :: GlobalvLSR                                         !< global velocity for each freq. range
    real*8, allocatable, dimension(:) :: Redshift_Range                                     !< redshift for each range
    real*8, allocatable, dimension(:) :: BackgroundTemperatureRange                         !< T_Back for each frequency range
    real*8, allocatable, dimension(:) :: TemperatureSlopeRange                              !< T_Slope for each frequency range
    real*8, allocatable, dimension(:) :: HydrogenColumnDensityRange                         !< nH for each frequency range
    real*8, allocatable, dimension(:) :: DustBetaRange                                      !< beta for each frequency range
    real*8, allocatable, dimension(:) :: KappaRange                                         !< kappa for each frequency range
    real*8, allocatable, dimension(:) :: DustTauFromFile                                    !< array containing tau_dust from file
    real*8, allocatable, dimension(:) :: BackgroundFromFile                                 !< array containing background from file
    real*8, allocatable, dimension(:) :: SortedDistances                                    !< array containing sorted distances
    real*8, allocatable, dimension(:) :: PureDistances                                      !< array containing pure distances
    real*8, allocatable, dimension(:, :) :: ContPhen_Range                                  !< parameters for phenomenological continuum description
    real*8, allocatable, dimension(:, :) :: myXCLASSParameter                               !< array containing all molfit parameters for each component
    real*8, allocatable, dimension(:, :) :: vWidthLimits                                    !< lower and upper limits of velocity widths parameters
    real*8, allocatable, dimension(:, :) :: vOffLimits                                      !< lower and upper limits of velocity offsets parameters
    real*8, allocatable, dimension(:, :) :: ObservationalDataList                           !< list containing all observational data
    real*8, allocatable, dimension(:, :) :: GausssianBeamMap                                !< map describing elliptical rotated Gaussian beam
    real*8, allocatable, dimension(:, :, :) :: EmsAbsFunc                                   !< array describing emission and absorption function
    character(len = 4096) :: EmsAbsPATH                                                     !< path of directory containing files describing em. and abs.
    character(len = 40), allocatable, dimension(:) :: MoleculeNames                         !< names of all molecules (including isotopologues)
    character(len = 4096), allocatable, dimension(:) :: DustFilenNames                      !< path and name of the dust files
    character(len = 4096), allocatable, dimension(:) :: BackgroundFilenNames                !< path and name of the background files
    logical :: IsoFlag                                                                      !< flag indicating use of isotopologues
    logical :: SubBeamDescriptionFlag                                                       !< flag indicating sub-beam description
    logical :: UseEmAbsFuncFlag                                                             !< flag indicating usage of emission/absorption file(s)
    logical, allocatable, dimension(:) :: tbFlag                                            !< flag indicating that T_Back and T_Slope describes
                                                                                            !< continuum completely
    logical, allocatable, dimension(:) :: InterFlag                                         !< flag indicating that observation is interferrometric data
    logical, allocatable, dimension(:) :: tdFlag                                            !< flag for reading T_dOff and T_dSlope from molfit file
    logical, allocatable, dimension(:) :: nHFlag                                            !< flag for global setting of nH, kappa and beta
    logical, allocatable, dimension(:) :: LTEFlag                                           !< flag for LTE calculation
    logical, allocatable, dimension(:) :: ThinFlag                                          !< flag indicating optical thin or thick (for Non-LTE RRLs)
    logical, allocatable, dimension(:) :: Dust_Flag_Range                                   !< flag indicating global phen. settings for each range
    logical, allocatable, dimension(:) :: Phen_Flag_Range                                   !< flag indicating global dust settings for each range
    logical, allocatable, dimension(:, :) :: ConfigList                                     !< list of unique pixel combination
    logical, allocatable, dimension(:, :, :) :: LayerMap                                    !< indicates, which components are taken into account


    !< variables for SQLite3 database
    integer :: MinNumTransitionsSQL                                                         !< min number of transitions which are considered in SQL query
    integer :: MaxNumTransitionsSQL                                                         !< max number of transitions which are considered in SQL query
    integer :: OrderTransSQL                                                                !< order or transition considered in SQL query
    integer, allocatable, dimension(:, :) :: NumEntriesRanges                               !< number of transitions for each freq. range and molecule
    real*8 :: MaxElowSQL                                                                    !< highest lower energy of transitions considered in SQL query
    real*8 :: MingASQL                                                                      !< min. intensity of transitions considered in SQL query
    character(len = 128) :: NameOfPartFuncTable                                             !< name of table for partition function
    character(len = 128) :: NameOfRadTransTable                                             !< name of table for rad. trans.
    character(len = 4096) :: dbName                                                         !< name of database


    !< variables for partition function
    integer :: NumberOfTemperatures                                                         !< number of temperatures
    integer :: NumberMoleculePartFunc                                                       !< number of entries in table PartitionFunction
    integer :: Firsti, Lasti, stepi                                                         !< working variables for interpolation
    real*8 :: TempLow, TempHigh                                                             !< working variables for extrapolation of part. func.
    real*8, allocatable, dimension(:) :: TempPartFunc                                       !< temperatures for partition function
    real*8, allocatable, dimension(:, :) :: lgQ                                             !< lgQ entries of table PartitionFunction
    character(len = 40) :: ColumnNameForNamePartFunc                                        !< name of column including the names of molecules
    character(len = 40), allocatable, dimension(:) :: MolNamePartFunc                       !< molecule names of table PartitionFunction
    character(len = 40), allocatable, dimension(:) :: ColumnNamesPartFunc                   !< column names of table PartitionFunction


    !< variables for rad-trans parameter
    integer :: TotalNumberOfTransitions                                                     !< number of entries in table RadTrans
    integer :: TotalNumTransFreq                                                            !< number of doppler-shifted trans. freq. for curr. layer
    integer :: Column300KPartFunc                                                           !< number of column, which contains part. func. @ 300 K
    integer, allocatable, dimension(:, :) :: DopplerShiftedTransFreqIndex                   !< array containing indices of doppler-shifted trans. freq.
    integer, allocatable, dimension(:, :, :) :: MolecularDataIndices                        !< start and end index for each molecule and frequency range
                                                                                            !< within MolecularData variable
    real*8, allocatable, dimension(:, :) :: MolecularData                                   !< array containing the molecular data for all molecules and
                                                                                            !< all freqency ranges
    real*8, allocatable, dimension(:) :: DopplerShiftedTransFreq                            !< array containing all doppler-shifted transitions
    real*8, allocatable, dimension(:, :) :: CentralOpticalDepth                             !< optical depths at doppler-shifted trans. frequencies
    character(len = 40) :: ColumnNameForNameTransitions                                     !< name of column in table transitions for molecule name
    character(len = 40) :: ColumnNameForFreqTransitions                                     !< name of column in table transitions for frequency
    character(len = 40) :: ColumnNameForIntTransitions                                      !< name of column in table transitions for intensity
    character(len = 40) :: ColumnNameForEinsteinATransitions                                !< name of column in table transitions for Einstein A coef.
    character(len = 40) :: ColumnNameForFreqErrTransitions                                  !< name of column in table transitions for error freq.
    character(len = 40) :: ColumnNameForELowTransitions                                     !< name of column in table transitions for E_low
    character(len = 40) :: ColumnNameForgUpTransitions                                      !< name of column in table transitions for Upper State degen.
    character(len = 40) :: ColumnNameForLowerQNTransitions                                  !< name of column in table transitions for lower state QN
    character(len = 40) :: ColumnNameForUpperQNTransitions                                  !< name of column in table transitions for upper state QN
    character(len = 60), allocatable, dimension(:) :: LowerQNString                         !< quantum number string of lower state
    character(len = 60), allocatable, dimension(:) :: UpperQNString                         !< quantum number string of upper state


    !< variables for iso table file
    integer :: NumberOfIsomasters                                                           !< number of iso masters
    integer :: NumberOfGlobalISORatios                                                      !< number of global iso ratio definitions
    integer, allocatable, dimension(:, :, :) :: IsoNfitConversionTable                      !< iso ratios conversion table for free parameter index
    integer, allocatable, dimension(:, :, :) :: GlobalIsoRatioParameter                     !< parameters for globally defined iso ratios
    real*8, allocatable, dimension(:, :) :: IsoRatio                                        !< iso ratios for all iso masters
    real*8, allocatable, dimension(:, :) :: IsoRatioConversionTable                         !< table with iso ratios between iso master and molecule
    character(len = 4096) :: IsoTableFileName                                               !< path and name of the iso file
    character(len = 40), allocatable, dimension(:) :: IsoMolecule                           !< names of molecules for iso table
    character(len = 40), allocatable, dimension(:) :: IsoMasters                            !< names of iso masters for iso table


    !< for myXCLASS function only
    real*8, allocatable, dimension(:) :: LocalIntArray                                      !< working array for intensity per component calculation
    real*8, allocatable, dimension(:) :: TauHelperArrayEM                                   !< working array for intensity per component calculation
    real*8, allocatable, dimension(:) :: TauHelperArrayABS                                  !< working array for intensity per component calculation
    real*8, allocatable, dimension(:) :: TauHelperArray2                                    !< working array for intensity per component calculation
    real*8, allocatable, dimension(:, :) :: TdPerCompArray                                  !< array containing T_d for each comp.
    real*8, allocatable, dimension(:, :) :: IntegHelperArray1                               !< working array for integrated line intensity
    real*8, allocatable, dimension(:, :) :: IntegHelperArray2                               !< working array for integrated line intensity
    real*8, allocatable, dimension(:, :) :: PrevIntPerPeak                                  !< working array for integrated line intensity
    real*8, allocatable, dimension(:, :) :: ModelSpectrumPixel                              !< array containing spectrum per pixel
    real*8, allocatable, dimension(:, :, :) :: TauEmAbsArray                                !< array containing emission and absorption function
    real*8, allocatable, dimension(:, :, :) :: IntPerCompArray                              !< array containing intensities for each comp.
    real*8, allocatable, dimension(:, :, :) :: TauPerCompArray                              !< array containing tau for each comp.
    real*8, allocatable, dimension(:, :, :) :: IntTotalSubBeam                              !< array containing total intensities (sub-beam)
    real*8, allocatable, dimension(:, :, :, :) :: IntensityPerPeak                          !< working array for integrated line intensity
    real*8, allocatable, dimension(:, :, :, :) :: IntPerCompArraySubBeam                    !< array containing intensities for each comp. (sub-beam)
    real*8, allocatable, dimension(:, :, :, :) :: TauPerCompArraySubBeam                    !< array containing tau for each comp. (sub-beam)
    character(len = 4096), allocatable, dimension(:) :: TauCompFileName                     !< store name of file including tau for each component
    character(len = 4096), allocatable, dimension(:) :: IntCompFileName                     !< store name of file including intensity for each component


    !< constants for convolution and fftw3 package
    integer, parameter :: fftw_r2hc = 0
    integer, parameter :: fftw_hc2r = 1
    integer, parameter :: fftw_dht = 2
    integer, parameter :: fftw_redft00 = 3
    integer, parameter :: fftw_redft01 = 4
    integer, parameter :: fftw_redft10 = 5
    integer, parameter :: fftw_redft11 = 6
    integer, parameter :: fftw_rodft00 = 7
    integer, parameter :: fftw_rodft01 = 8
    integer, parameter :: fftw_rodft10 = 9
    integer, parameter :: fftw_rodft11 = 10
    integer, parameter :: fftw_forward = -1
    integer, parameter :: fftw_backward = +1
    integer, parameter :: fftw_measure = 0
    integer, parameter :: fftw_destroy_input = 1
    integer, parameter :: fftw_unaligned = 2
    integer, parameter :: fftw_conserve_memory = 4
    integer, parameter :: fftw_exhaustive = 8
    integer, parameter :: fftw_preserve_input = 16
    integer, parameter :: fftw_patient = 32
    integer, parameter :: fftw_estimate = 64
    integer, parameter :: fftw_estimate_patient = 128
    integer, parameter :: fftw_believe_pcost = 256
    integer, parameter :: fftw_dft_r2hc_icky = 512
    integer, parameter :: fftw_nonthreaded_icky = 1024
    integer, parameter :: fftw_no_buffering = 2048
    integer, parameter :: fftw_no_indirect_op = 4096
    integer, parameter :: fftw_allow_large_generic = 8192
    integer, parameter :: fftw_no_rank_splits = 16384
    integer, parameter :: fftw_no_vrank_splits = 32768
    integer, parameter :: fftw_no_vrecurse = 65536
    integer, parameter :: fftw_no_simd = 131072
    real*8, parameter :: BeamMultiplication = 3.d0                                          !< multiplication of beam


    contains


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !> subroutine: RADEXSPLCOEFF
        !>
        !> This file is part of the RADEX software package to calculate molecular excitation and radiative transfer in a homogeneous medium.
        !> Documentation for the program is posted at
        !>
        !> https://sron.rug.nl/~vdtak/radex/index.shtml
        !>
        !> Although this program has been thoroughly tested, the authors do not claim that it is free of errors and gives correct results in all
        !> situations. Publications using this program should make a reference to our paper: A&A 468, 627 (2007).
        !>
        !>
        !> This package contains several routines for applications of cubic-spline interpolation. A further routine, splinteg, is available on
        !> request (John.Black@chalmers.se) and does numerical quadrature through use of the spline coefficients determined for a set of points in
        !> routine splcoeff. These routines have been written by J. H. Black. The basic spline interpolation routines are based upon the routines of
        !> Numerical Recipes (Ch. 3.3), but are not identical to them.
        !>
        !> N values of a function f(x) are tabulated at points x(i), in order of increasing x(i). fpp(x) is the evaluated second derivative of f(x).
        !> fp1 and fpn are the values of the first derivatives at the beginning and ending points, i=1 and i=N. For natural splines, the second
        !> derivative is set to zero at a boundary. Set fp1 and/or fpn > 1.0D60 for natural splines. Otherwise, the derivatives can be specified and
        !> matched to those of extrapolating functions.
        !>
        !>
        !> input variables:     x:              frequency axis
        !>
        !>                      f:              intensities for each frequency point
        !>
        !>                      n:              number of data points
        !>
        !>                      NumDataPoints:  max. number of data points
        !>
        !>                      fp1:            set fp1 and/or fpn > 1.0D60 for natural splines
        !>
        !>                      fpn:            set fp1 and/or fpn > 1.0D60 for natural splines
        !>
        !>
        !> output variables:    fpp:            array of coefficients
        !>
        !>
        !> \author Van der Tak
        !>
        !> \date 2011-11-30
        !>
        subroutine RADEXSPLCOEFF(x, f, n, NumDataPoints, fp1, fpn, fpp)

            implicit none
            integer :: n                                                                    !< input: number of data points
            integer :: NumDataPoints                                                        !< input: max. number of data points
            integer :: i, k                                                                 !< loop variables
            real*8 :: fp1                                                                   !< input:
            real*8 :: fpn                                                                   !< input:
            real*8, dimension(NumDataPoints) :: x                                           !< input: frequency axis
            real*8, dimension(NumDataPoints) :: f                                           !< input: intensities for each frequency point
            real*8, dimension(NumDataPoints) :: fpp                                         !< output: array of coefficients
            real*8, dimension(NumDataPoints) :: u                                           !< working array
            real*8 :: p, qn, sig, un                                                        !< working variables


            !< Lower boundary condition (use 1e60 for f77/g77, 1e38 for f90):
            if (fp1 > 0.99d38) then
                fpp(1) = 0.d0
                u(1) = 0.d0
            else
                fpp(1) = -0.5d0
                u(1) = (3.d0 / (x(2) - x(1))) * ((f(2) - f(1)) / (x(2) - x(1)) - fp1)
            endif
            Do i = 2, (n - 1)
                sig = (x(i) - x(i - 1)) / (x(i + 1) - x(i - 1))
                p = sig * fpp(i - 1) + 2.d0
                fpp(i) = (sig - 1.d0) / p
                u(i) = (6.d0 * ((f(i + 1) - f(i)) / (x(i + 1) - x(i)) - (f(i) - f(i - 1)) / (x(i) - x(i - 1))) &
                    / (x(i + 1) - x(i - 1)) - sig * u(i - 1)) / p
            end Do


            !< Upper boundary condition (see above):
            if (fpn > 0.99d38) then
                qn = 0.d0
                un = 0.d0
            else
                qn = 0.5d0
                un = (3.d0 / (x(n) - x(n - 1))) * (fpn - (f(n) - f(n - 1)) / (x(n) - x(n - 1)))
            endif

            fpp(n) = (un - qn * u(n - 1)) / (qn * fpp(n - 1) + 1.d0)
            Do k = (n - 1), 1, (-1)
                fpp(k) = fpp(k) * fpp(k + 1) + u(k)
            end Do

            ! Debug:
            ! print*,"n = ", n
            ! print*,"NumDataPoints = ", NumDataPoints
            ! print*,"fp1 = ", fp1
            ! print*,"fpn = ", fpn
            ! print*,"fpp = ", fpp


            !< we're done
            return
        end subroutine RADEXSPLCOEFF


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !> subroutine: RADEXsplintrp
        !>
        !> This file is part of the RADEX software package to calculate molecular excitation and radiative transfer in a homogeneous medium.
        !> Documentation for the program is posted at
        !>
        !> https://sron.rug.nl/~vdtak/radex/index.shtml
        !>
        !> Although this program has been thoroughly tested, the authors do not claim that it is free of errors and gives correct results in all
        !> situations. Publications using this program should make a reference to our paper: A&A 468, 627 (2007).
        !>
        !>
        !> For N tabulated values xin(i) and fin(x) of a function, and the array fppin(x), which is the 2nd derivative of fin(x) delivered by
        !> SPLCOEFF above, an interpolated value of fout is delivered at x. The routine can also, if desired, return the values of the first and
        !> second derivatives of the fitted function, foutp and foutpp.
        !>
        !>
        !> input variables:     xin:            frequency axis
        !>
        !>                      fin:            intensities for each frequency point
        !>
        !>                      fppin:          2nd derivative of fin
        !>
        !>                      N:              number of data points
        !>
        !>                      NumDataPoints:  max. number of data points
        !>
        !>                      x:
        !>
        !>
        !> output variables:    fout:
        !>
        !>
        !> \author Van der Tak
        !>
        !> \date 2011-11-30
        !>
        subroutine RADEXsplintrp(xin, fin, fppin, N, NumDataPoints, x, fout)

            implicit none
            integer :: N                                                                    !< input: number of data points
            integer :: NumDataPoints                                                        !< input: max. number of data points
            integer :: k                                                                    !< working variable
            integer :: khi                                                                  !< working variable
            integer :: klo                                                                  !< working variable
            real*8 :: x                                                                     !< input:
            real*8 :: fout                                                                  !< output:
            real*8 :: a                                                                     !< working variable
            real*8 :: b                                                                     !< working variable
            real*8 :: h                                                                     !< working variable
            real*8 :: foutp                                                                 !< output: first derivatives of the fitted function
            real*8 :: foutpp                                                                !< output: second derivatives of the fitted function
            real*8, dimension(NumDataPoints) :: xin                                         !< input: frequency axis
            real*8, dimension(NumDataPoints) :: fin                                         !< input: intensities for each frequency point
            real*8, dimension(NumDataPoints) :: fppin                                       !< input: 2nd derivative of fin

            klo = 1
            khi = N
          1 if (khi - klo > 1) then
                k = (khi + klo) / 2
                if (xin(k) > x) then
                    khi = k
                else
                    klo = k
                endif
                goto 1
            endif
            h = xin(khi) - xin(klo)
            if (h == 0.d0) then
                print '("Warning: bad xin input in splintrp")'
                print '(" ")'
                h = 1.d0
            endif
            a = (xin(khi) - x) / h
            b = (x - xin(klo)) / h
            fout = a * fin(klo) + b * fin(khi) + ((a**3.d0 - a) * fppin(klo) + (b**3.d0 - b) * fppin(khi)) * (h**2.d0) / 6.d0


            !< first derivative
            foutp = (fin(khi) - fin(klo)) / h - (3.d0 * a * a - 1.d0) * h * fppin(klo) / 6.d0 &
                    + (3.d0 * b * b - 1.d0) * h * fppin(khi) / 6.d0


            !< second derivative
            foutpp = a * fppin(klo) + b * fppin(khi)

            ! Debug:
            ! print*,"fout = ", fout
            ! print*,"foutp = ", foutp
            ! print*,"foutpp = ", foutpp


            !< we're done
            return
        end subroutine RADEXsplintrp


        !>************************************************************************************************************************************************
        !> subroutine: CheckMolNames
        !>
        !> checks, if a molecule name is already known
        !>
        !>
        !> input variables:     TotalNumberOfMoleculesCopy: total number of all already known molecules
        !>                      CurrMol:                current molecule
        !>
        !> output variables:    AlreadyIncludedFlag:    flag indicating if current molecule is already known or not
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2014-07-27
        !>
        subroutine CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMoleculesCopy, CurrMol)

            implicit none
            integer :: i                                                                    !< loop variable
            integer :: TotalNumberOfMoleculesCopy                                           !< total number of all already known molecules
            character(len=40) :: CurrMol                                                    !< current molecule
            logical :: AlreadyIncludedFlag                                                  !< flag indicating if current molecule is already known or not


            AlreadyIncludedFlag = .false.                                                   !< initialize return parameter
            Do i = 1, TotalNumberOfMoleculesCopy
                if (trim(adjustl(MoleculeNames(i))) == trim(adjustl(CurrMol))) then
                    !AlreadyIncludedFlag = .true.
                    exit
                endif
            end Do
            return
        end subroutine CheckMolNames


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !> subroutine to determine last non blank and non char(0) character of string st
        !>
        !>
        !> input variables:     st:                 string which has to be analyzed
        !>
        !> output variables:    val:                position of last non blank and non char(0) character of string st
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2017-12-20
        !>
        subroutine strlen(st)

            implicit none
            integer :: i                                                                    !< working variable
            character :: st*(*)                                                             !< string which has to be analyzed

            ! Debug:
            ! print*,'st = ', st

            Do i = 1, len(st)
                if (st(i:i) == char(0)) then
                    st(i:i) = " "
                endif
            end Do
            return
        end subroutine strlen


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !> subroutine to count occurrences of a substring
        !>
        !>
        !> input variables:     s1:                 string which is analyzed
        !>
        !>                      s2:                 string which is searched
        !>
        !> output variables:    c:                  number of occurrences
        !>
        !>
        !> partly taken from:
        !>      http://rosettacode.org/wiki/Count_occurrences_of_a_substring#Fortran
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2016-01-07
        !>
        subroutine countsubstring(c, s1, s2)

            implicit none
            character(*), intent(in) :: s1, s2
            integer :: c, p, posn

            c = 0
            if (len(s2) == 0) return
            p = 1
            Do
                posn = index(s1(p:), s2)
                if (posn == 0) return
                c = c + 1
                p = p + posn + len(s2)
            end Do
            return
        end subroutine countsubstring


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !> subroutine to convert string to lower case
        !>
        !>
        !> input variables:     StringIn:           string which is analyzed
        !>
        !> output variables:    StringOut:          number of occurrences
        !>
        !>
        !> partly taken from:
        !>      https://stackoverflow.com/questions/10759375/how-can-i-write-a-to-upper-or-to-lower-function-in-f90
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2017-11-22
        !>
        subroutine ConverStringToLowerCase(StringIn)

            implicit none
            integer :: i                                                                    !< loop variable
            integer :: j                                                                    !< working variable: ASCII code of local character
            character(len = *) :: StringIn                                                  !< input string


            !< remove leading and trailing blanks?
            StringIn = trim(adjustl(StringIn))


            !< convert to lower case
            Do i = 1, len_trim(StringIn)
                j = iachar(StringIn(i:i))
                if (j >= iachar("A") .and. j <= iachar("Z") ) then
                    StringIn(i:i) = achar(iachar(StringIn(i:i)) + 32)
                end if
            end do

            ! Debug:
            ! print*,"StringIn = >>", StringIn, "<<"


            !< we're done
            return
        end subroutine ConverStringToLowerCase


        !>************************************************************************************************************************************************
        !> subroutine: ImportDustFile
        !>
        !> import dust file and interpolate at given frequency points
        !>
        !>
        !> input variables:     RangeID
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2017-11-13
        !>
        subroutine ImportDustFile(RangeID)

            implicit none
            integer, parameter :: LocalMaxDataPoints = 10000000                             !< max. number of data points in dust file
            integer :: iii, FreqIndex                                                       !< loop indices
            integer :: RangeID                                                              !< range index
            integer :: NumDustDataPoints                                                    !< number of data points in dust file
            integer :: FirstIndex, LastIndex                                                !< working variables for freq. range index determination
            integer :: io_err                                                               !< error number of i/o traffic
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            real*8, parameter :: huge =1.0e38                                               !< highest allowed by f90 (fdvt 28apr06)
            real*8 :: ObsFreq                                                               !< frequency point in obs. data file
            real*8 :: fout                                                                  !< used for interpolation
            real*8, allocatable, dimension(:) :: fpp                                        !< 2nd derivate of dust tau (helps to calculate splines)
            real*8, allocatable, dimension(:, :) :: ContentDustFile                         !< array for content of dust file
            character(len=4096) :: dummyLine
            character(len=4096) :: LocalDustFileName                                        !< local copy of dust file name


            !< get path and name of dust file
            LocalDustFileName = trim(adjustl(DustFilenNames(RangeID)))


            !< determine number of frequency points in dust file
            NumDustDataPoints = 0
            open(22222, file = trim(adjustl(LocalDustFileName)), status = 'UNKNOWN')
            Do iii = 1, LocalMaxDataPoints
                read(22222, *, iostat = io_err) dummyLine
                if (io_err /= 0) then
                    exit
                else
                    NumDustDataPoints = NumDustDataPoints + 1
                endif
            end Do


            !< allocate memory for content of dust file
            if (allocated(ContentDustFile)) then
                deallocate(ContentDustFile, fpp, stat = dealloc_status)
                if (dealloc_status /= 0) then                                               !< is all ok?
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine ImportDustFile!")')
                        write(ErrChannel, '(3x, "Can not deallocate variables ContentDustFile and fpp!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    close(logchannel)
                    stop 'Program aborted! Please restart the program!'
                endif
            endif
            allocate(ContentDustFile(NumDustDataPoints, 2), fpp(NumDustDataPoints), stat = alloc_status)
            if (alloc_status /= 0) then                                                     !< is all ok?
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine ImportDustFile!")')
                    write(ErrChannel, '(3x, "Can not allocate variables ContentDustFile(NumDustDataPoints, 2) &
                                             &and fpp(NumDustDataPoints)!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameter:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- NumDustDataPoints = ", I8)') NumDustDataPoints
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                close(logchannel)
                stop 'Program aborted! Please restart the program!'
            endif
            ContentDustFile = 0.d0
            fpp = 0.d0


            !< import dust file
            rewind 22222                                                                    !< go to beginning of dust file
            Do iii = 1, NumDustDataPoints
                read(22222, *, iostat = io_err) ContentDustFile(iii, :)
                if (io_err /= 0) then
                    exit
                endif
            end Do
            close(22222)                                                                    !< close parameter file


            !< Set the spline coefficients
            call RADEXSPLCOEFF(ContentDustFile(:, 1), ContentDustFile(:, 2), NumDustDataPoints, NumDustDataPoints, huge, huge, fpp)


            !< interpolate at given frequency points
            FirstIndex = DataPointIndexFreqRange(RangeID, 1)                                !< get index of first freq. point in 'ObservationalDataList'
            LastIndex = DataPointIndexFreqRange(RangeID, 2)                                 !< get index of last freq. point in 'ObservationalDataList'
            Do FreqIndex = FirstIndex, LastIndex                                            !< loop over all frequency points in the current freq. range
                ObsFreq = ObservationalDataList(FreqIndex, 1)                               !< get frequency
                call RADEXsplintrp(ContentDustFile(:, 1), ContentDustFile(:, 2), fpp, NumDustDataPoints, NumDustDataPoints, &
                                   ObsFreq, fout)
                DustTauFromFile(FreqIndex) = fout

                ! Debug:
                ! print*,"FreqIndex, ObsFreq, DustTauFromFile(FreqIndex) = ", FreqIndex, ObsFreq, DustTauFromFile(FreqIndex)
            end Do


            return
        end subroutine ImportDustFile


        !>************************************************************************************************************************************************
        !> subroutine: ImportBackgroundFile
        !>
        !> import background file and interpolate at given frequency points
        !>
        !>
        !> input variables:     RangeID
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2018-12-27
        !>
        subroutine ImportBackgroundFile(RangeID)

            implicit none
            integer, parameter :: LocalMaxDataPoints = 10000000                             !< max. number of data points in background file
            integer :: iii, FreqIndex                                                       !< loop indices
            integer :: RangeID                                                              !< range index
            integer :: NumBackgroundDataPoints                                              !< number of data points in background file
            integer :: FirstIndex, LastIndex                                                !< working variables for freq. range index determination
            integer :: io_err                                                               !< error number of i/o traffic
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            real*8, parameter :: huge =1.0e38                                               !< highest allowed by f90 (fdvt 28apr06)
            real*8 :: ObsFreq                                                               !< frequency point in obs. data file
            real*8 :: fout                                                                  !< used for interpolation
            real*8, allocatable, dimension(:) :: fpp                                        !< 2nd derivate of background tau (helps to calculate splines)
            real*8, allocatable, dimension(:, :) :: ContentBackgroundFile                   !< array for content of background file
            character(len=4096) :: dummyLine
            character(len=4096) :: LocalBackgroundFileName                                  !< local copy of background file name


            !< get path and name of background file
            LocalBackgroundFileName = trim(adjustl(BackgroundFilenNames(RangeID)))


            !< determine number of frequency points in background file
            NumBackgroundDataPoints = 0
            open(22222, file = trim(adjustl(LocalBackgroundFileName)), status = 'UNKNOWN')
            Do iii = 1, LocalMaxDataPoints
                read(22222, *, iostat = io_err) dummyLine
                if (io_err /= 0) then
                    exit
                else
                    NumBackgroundDataPoints = NumBackgroundDataPoints + 1
                endif
            end Do


            !< allocate memory for content of background file
            if (allocated(ContentBackgroundFile)) then
                deallocate(ContentBackgroundFile, fpp, stat = dealloc_status)
                if (dealloc_status /= 0) then                                               !< is all ok?
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine ImportBackgroundFile!")')
                        write(ErrChannel, '(3x, "Can not deallocate variables ContentBackgroundFile and fpp!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    close(logchannel)
                    stop 'Program aborted! Please restart the program!'
                endif
            endif
            allocate(ContentBackgroundFile(NumBackgroundDataPoints, 2), fpp(NumBackgroundDataPoints), stat = alloc_status)
            if (alloc_status /= 0) then                                                     !< is all ok?
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine ImportBackgroundFile!")')
                    write(ErrChannel, '(3x, "Can not allocate variables ContentBackgroundFile(NumBackgroundDataPoints, 2) and")')
                    write(ErrChannel, '(3x, "fpp(NumBackgroundDataPoints)!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameter:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- NumBackgroundDataPoints = ", I8)') NumBackgroundDataPoints
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                close(logchannel)
                stop 'Program aborted! Please restart the program!'
            endif
            ContentBackgroundFile = 0.d0
            fpp = 0.d0


            !< import background file
            rewind 22222                                                                    !< go to beginning of background file
            Do iii = 1, NumBackgroundDataPoints
                read(22222, *, iostat = io_err) ContentBackgroundFile(iii, :)
                if (io_err /= 0) then
                    exit
                endif
            end Do
            close(22222)                                                                    !< close parameter file


            !< Set the spline coefficients
            call RADEXSPLCOEFF(ContentBackgroundFile(:, 1), ContentBackgroundFile(:, 2), NumBackgroundDataPoints, &
                               NumBackgroundDataPoints, huge, huge, fpp)


            !< interpolate at given frequency points
            FirstIndex = DataPointIndexFreqRange(RangeID, 1)                                !< get index of first freq. point in 'ObservationalDataList'
            LastIndex = DataPointIndexFreqRange(RangeID, 2)                                 !< get index of last freq. point in 'ObservationalDataList'
            Do FreqIndex = FirstIndex, LastIndex                                            !< loop over all frequency points in the current freq. range
                ObsFreq = ObservationalDataList(FreqIndex, 1)                               !< get frequency
                call RADEXsplintrp(ContentBackgroundFile(:, 1), ContentBackgroundFile(:, 2), fpp, NumBackgroundDataPoints, &
                                   NumBackgroundDataPoints, ObsFreq, fout)
                BackgroundFromFile(FreqIndex) = fout

                ! Debug:
                ! print*,"FreqIndex, BackgroundFromFile(FreqIndex) = ", FreqIndex, BackgroundFromFile(FreqIndex)
            end Do


            return
        end subroutine ImportBackgroundFile


        !>************************************************************************************************************************************************
        !> subroutine: ImportEmsAbsFile
        !>
        !> import files describing emission and absorption and interpolate at given frequency points
        !>
        !>
        !> input variables:     None
        !>
        !> output variables:    None
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2019-12-04
        !>
        subroutine ImportEmsAbsFile

            implicit none
            integer, parameter :: LocalMaxDataPoints = 10000000                             !< max. number of data points in emission/absorption file
            integer :: c, iii, jjj, FreqIndex, RangeID                                      !< loop indices
            integer :: NumberFiles                                                          !< number of files in EmsAbsPATH
            integer :: NumFreqDataPoints                                                    !< number of data points in emission/absorption file
            integer :: FirstIndex, LastIndex                                                !< working variables for freq. range index determination
            integer :: io_err                                                               !< error number of i/o traffic
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            real*8, parameter :: huge =1.0e38                                               !< highest allowed by f90 (fdvt 28apr06)
            real*8 :: ObsFreq                                                               !< frequency point in obs. data file
            real*8 :: fout                                                                  !< used for interpolation
            real*8 :: LocalMinFreq, LocalMaxFreq                                            !< working variable: min. and max. frequency
            real*8, allocatable, dimension(:) :: fpp                                        !< 2nd derivate of emission/absorption function
            real*8, allocatable, dimension(:, :) :: ContentEmAbsFile                        !< array for content of emission/absorption file(s)
            character(len=30) :: numberstring1                                              !< used for integer number to string conversion
            character(len=4096) :: dummyLine                                                !< working variable used for determine number of lines in file
            character(len=4096) :: LocalFileName                                            !< working variable: local file name
            character(len=8192) :: cmdString                                                !< working variable: call system
            character(len=8192) :: WorkingDir                                               !< containing current working directory
            character(len=4096), allocatable, dimension(:) :: FileListing                   !< list of file names in EmsAbsPATH
            logical :: SplineFlag                                                           !< choose interpolation method


            !< define interpolation method (SplineFlag = .true.: spline interpolation,  SplineFlag = .false.:  nearest-neighboring interpolation)
            SplineFlag = .false.


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< create a local subdirectory within the temp directory for local temp files


            !< create name of current working directory
            WorkingDir = " "                                                                !< working directory is current directory
            write(numberstring1, '(I30)') JobID                                             !< write JobID to string
            WorkingDir = trim(adjustl(EmsAbsPATH)) // "job_" // trim(adjustl(numberstring1)) // "__EmsAbs/"


            !< create working directory
            cmdString = "mkdir -p "// trim(adjustl(WorkingDir))
            call system(trim(adjustl(cmdString)))

            ! Debug:
            ! print*,'-> cmdString = ',trim(adjustl(cmdString)),'<-'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get a list of all files located in directory EmsAbsPATH


            !< print what you do
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(14x, "Import ASCII files describing emission and absorption functions for different distances .. ")'
                print '(" ")'
            endif


            !< use system call to write listing of files into file
            LocalFileName = trim(adjustl(WorkingDir)) // "fileContents.txt"
            cmdString = "(ls " // trim(adjustl(EmsAbsPATH)) // " > " // trim(adjustl(LocalFileName)) // ")"
            call system(trim(adjustl(cmdString)))

            ! Debug:
            ! print*,"EmsAbsPATH = ", trim(adjustl(EmsAbsPATH))
            ! print*,"LocalFileName = ", trim(adjustl(LocalFileName))


            !< import listing file
            open(22222, file = trim(adjustl(LocalFileName)), action = "read", iostat = io_err)
            if (io_err /= 0) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine ImportEmsAbsFile!")')
                    write(ErrChannel, '(3x, "Can not open file fileContents.txt!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '("io_err = ", I5)') io_err
                end Do
                close(logchannel)
                stop 'Program aborted! Please restart the program!'
            endif


            !< determine number of lines in listing file
            NumberFiles = 0
            Do iii = 1, LocalMaxDataPoints
                read(22222, *, iostat = io_err) dummyLine
                if (io_err /= 0) then
                    exit
                else
                    NumberFiles = NumberFiles + 1
                endif
            end Do


            !< allocate memory for file listing
            if (allocated(FileListing)) then
                deallocate(FileListing, stat = dealloc_status)
                if (dealloc_status /= 0) then                                               !< is all ok?
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine ImportEmsAbsFile!")')
                        write(ErrChannel, '(3x, "Can not deallocate variable FileListing!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    close(logchannel)
                    stop 'Program aborted! Please restart the program!'
                endif
            endif
            allocate(FileListing(NumberFiles), stat = alloc_status)
            if (alloc_status /= 0) then                                                     !< is all ok?
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine ImportEmsAbsFile!")')
                    write(ErrChannel, '(3x, "Can not allocate variable FileListing!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameter:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- NumberFiles = ", I8)') NumberFiles
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                close(logchannel)
                stop 'Program aborted! Please restart the program!'
            endif
            FileListing = ""


            !< import file names
            rewind(22222)
            Do iii = 1, NumberFiles                                                         !< loop over all files
                read(22222, '(A)') FileListing(iii)
                FileListing(iii) = trim(adjustl(FileListing(iii)))

                ! Debug:
                ! print*,"iii, FileListing(iii) = ", iii, trim(adjustl(FileListing(iii)))
            end Do                                                                          !< iii: loop over all files
            close(22222, status = "delete")


            !< remove temp-directory
            cmdString = "(rm -rf "// trim(adjustl(WorkingDir)) // " 2>/dev/null)"
            call system(cmdString)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for content of emission/absorption file(s)
            if (allocated(EmsAbsFunc)) then
                deallocate(EmsAbsFunc, stat = dealloc_status)
                if (dealloc_status /= 0) then                                               !< is all ok?
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine ImportEmsAbsFile!")')
                        write(ErrChannel, '(3x, "Can not deallocate variable EmsAbsFunc!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    close(logchannel)
                    stop 'Program aborted! Please restart the program!'
                endif
            endif
            allocate(EmsAbsFunc(TotalNumberDataPoints, NumberDistances, 2), stat = alloc_status)
            if (alloc_status /= 0) then                                                     !< is all ok?
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine ImportEmsAbsFile!")')
                    write(ErrChannel, '(3x, "Can not allocate variable EmsAbsFunc(NumFreqDataPoints, 3)!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameter:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- TotalNumberDataPoints = ", I8)') TotalNumberDataPoints
                    write(ErrChannel, '(5x, "- NumberDistances = ", I8)') NumberDistances
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                close(logchannel)
                stop 'Program aborted! Please restart the program!'
            endif
            EmsAbsFunc = 0.d0


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< import emission and absorption functions from file
            Do c = 1, NumberDistances                                                       !< loop over all layers / distances
                write(numberstring1, '(ES25.15)') PureDistances(c)                          !< convert float to string
                call ConverStringToLowerCase(numberstring1)


                !< find the emission / absorption file which belongs to current distance
                LocalFileName = ""
                Do iii = 1, NumberFiles                                                     !< loop over all files in directory EmsAbsPATH
                    dummyLine = trim(adjustl(FileListing(iii)))
                    call ConverStringToLowerCase(dummyLine)                                 !< convert file name to lower case


                    !< check, if current file belongs to current distance
                    if (dummyLine(1:19) == "emission_absorption" .and. index(dummyLine, "distance__" // &
                        trim(adjustl(numberstring1)) // ".dat") > 0)then
                        LocalFileName = trim(adjustl(EmsAbsPATH)) // trim(adjustl(FileListing(iii)))

                        ! Debug:
                        ! print*,"iii, LocalFileName = ", iii, trim(adjustl(LocalFileName))


                        !< print what you do
                        if (printflag) then
                            print '(16x, "Import ASCII file ", A)', trim(adjustl(LocalFileName))
                        endif


                        !< get number of frequency points in current emission/absorption file (including header lines)
                        NumFreqDataPoints = 0
                        open(22222, file = trim(adjustl(LocalFileName)), status = 'UNKNOWN')
                        Do jjj = 1, LocalMaxDataPoints
                            read(22222, *, iostat = io_err) dummyLine
                            if (io_err /= 0) then
                                exit
                            else
                                NumFreqDataPoints = NumFreqDataPoints + 1
                            endif
                        end Do
                        NumFreqDataPoints = NumFreqDataPoints - 3                           !< remove three header lines


                        !< allocate memory for local content of emission/absorption file
                        if (allocated(ContentEmAbsFile)) then
                            deallocate(ContentEmAbsFile, fpp, stat = dealloc_status)
                            if (dealloc_status /= 0) then                                   !< is all ok?
                                Do ErrChannelIndex = 1, 2
                                    ErrChannel = AllErrChannels(ErrChannelIndex)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x, "Error in subroutine ImportEmsAbsFile!")')
                                    write(ErrChannel, '(3x, "Can not deallocate variables ContentEmAbsFile and fpp!")')
                                    write(ErrChannel, '(" ")')
                                end Do
                                close(logchannel)
                                stop 'Program aborted! Please restart the program!'
                            endif
                        endif
                        allocate(ContentEmAbsFile(NumFreqDataPoints, 3), fpp(NumFreqDataPoints), stat = alloc_status)
                        if (alloc_status /= 0) then                                         !< is all ok?
                            Do ErrChannelIndex = 1, 2
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x, "Error in subroutine ImportEmsAbsFile!")')
                                write(ErrChannel, '(3x, "Can not allocate variables ContentEmAbsFile(NumFreqDataPoints, 2) and")')
                                write(ErrChannel, '(3x, "fpp(NumFreqDataPoints)!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Used parameter:")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(5x, "- NumFreqDataPoints = ", I8)') NumFreqDataPoints
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                            end Do
                            close(logchannel)
                            stop 'Program aborted! Please restart the program!'
                        endif
                        ContentEmAbsFile = 0.d0
                        fpp = 0.d0


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< import emission/absorption file for current layer
                        rewind 22222                                                        !< go to beginning of emission/absorption file
                        Do jjj = 1, NumFreqDataPoints + 3                                   !< loop over all lines including header lines
                            if (jjj <= 3) then                                              !< ignore first three header lines
                                read(22222, *, iostat = io_err) dummyLine
                            else
                                read(22222, *, iostat = io_err) ContentEmAbsFile(jjj - 3, :)    !< import line
                                if (io_err /= 0) then                                       !< exit loop on error
                                    exit
                                endif
                            endif
                        end Do                                                              !< iii: loop over all lines
                        close(22222)                                                        !< close parameter file
                        LocalMinFreq = minval(ContentEmAbsFile(:, 1))
                        LocalMaxFreq = maxval(ContentEmAbsFile(:, 1))

                        ! Debug:
                        ! print*,"LocalMinFreq = ", LocalMinFreq
                        ! print*,"LocalMaxFreq = ", LocalMaxFreq
                        ! print*,"ContentEmAbsFile(:, 1) = ", ContentEmAbsFile(:, 1)


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< interpolate at given frequency points using selected interpolation method


                        !< Set the spline coefficients
                        if (SplineFlag) then
                            call RADEXSPLCOEFF(ContentEmAbsFile(:, 1), ContentEmAbsFile(:, 2), NumFreqDataPoints, &
                                               NumFreqDataPoints, huge, huge, fpp)
                        endif


                        !< interpolate at given frequency points
                        Do RangeID = 1, TotalNumberOfFrequencyRanges                        !< loop over all frequency ranges
                            FirstIndex = DataPointIndexFreqRange(RangeID, 1)                !< get index of first freq. point in 'ObservationalDataList'
                            LastIndex = DataPointIndexFreqRange(RangeID, 2)                 !< get index of last freq. point in 'ObservationalDataList'
                            Do FreqIndex = FirstIndex, LastIndex                            !< loop over all frequency points in the current freq. range
                                ObsFreq = ObservationalDataList(FreqIndex, 1)               !< get frequency
                                if (LocalMinFreq <= ObsFreq .and. ObsFreq <= LocalMaxFreq) then


                                    !< use spline interpolation
                                    if (SplineFlag) then
                                        call RADEXsplintrp(ContentEmAbsFile(:, 1), ContentEmAbsFile(:, 2), fpp, &
                                                           NumFreqDataPoints,NumFreqDataPoints, ObsFreq, fout)
                                        EmsAbsFunc(FreqIndex, c, 1) = fout                  !< define emission function
                                        call RADEXsplintrp(ContentEmAbsFile(:, 1), ContentEmAbsFile(:, 3), fpp, &
                                                           NumFreqDataPoints, NumFreqDataPoints, ObsFreq, fout)
                                        EmsAbsFunc(FreqIndex, c, 2) = fout                  !< define absorption function


                                    !< use nearest-neighboring interpolation
                                    else
                                        jjj = minloc(dabs(ContentEmAbsFile(:, 1) - ObsFreq), dim = 1)
                                        EmsAbsFunc(FreqIndex, c, 1) = ContentEmAbsFile(jjj, 2)
                                        EmsAbsFunc(FreqIndex, c, 2) = ContentEmAbsFile(jjj, 3)
                                    endif

                                    ! Debug:
                                    ! print*,"FreqIndex, EmsAbsFunc(FreqIndex, c, 1:2) = ", FreqIndex, EmsAbsFunc(FreqIndex, c, 1:2)
                                endif
                            end Do                                                          !< FreqIndex: loop over frequencies
                        end Do                                                              !< RangeID: loop over all frequency ranges
                    endif
                end Do                                                                      !< iii: loop over all files
            end Do                                                                          !< c: loop over all layers / distances


            !< we're done
            return
        end subroutine ImportEmsAbsFile


        !>************************************************************************************************************************************************
        !> subroutine: CheckCollisionPartner
        !>
        !> check name of collision partner
        !>
        !>
        !> input variables:     None
        !>
        !> output variables:    None
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2021-08-27
        !>
        subroutine CheckCollisionPartner(k, LocalCollName)

            implicit none
            integer :: k
            character(len=10) :: LocalCollName                                              !< working variable: name of coll. partner


            !< convert name of collision partner to lower case
            LocalCollName = trim(adjustl(LocalCollName))
            call ConverStringToLowerCase(LocalCollName)


            !< check collision partner
            k = 0
            if (trim(adjustl(LocalCollName)) == "h2") then
                k = 1
            elseif (trim(adjustl(LocalCollName)) == "p-h2" .or. trim(adjustl(LocalCollName)) == "p_h2") then
                k = 2
            elseif (trim(adjustl(LocalCollName)) == "o-h2" .or. trim(adjustl(LocalCollName)) == "o_h2") then
                k = 3
            elseif (trim(adjustl(LocalCollName)) == "e") then
                k = 4
            elseif (trim(adjustl(LocalCollName)) == "h") then
                k = 5
            elseif (trim(adjustl(LocalCollName)) == "he") then
                k = 6
            elseif (trim(adjustl(LocalCollName)) == "h+") then
                k = 7
            else
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine CheckCollisionPartner!")')
                    write(ErrChannel, '(3x, "Unknown species. Choose from: H2 p-H2 o-H2 e H He H+!")')
                    write(ErrChannel, '(" ")')
                end Do
                close(logchannel)
            endif

            ! Debug:
            ! print*,"LocalCollName = ", trim(adjustl(LocalCollName))
            ! print*,"k = ", k


            !< we're done
            return
        end subroutine CheckCollisionPartner


        !>************************************************************************************************************************************************
        !> subroutine: GetmyXCLASSParameter
        !>
        !> read parameter for myXCLASS from paramset variable
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2014-07-27
        !>
        subroutine GetmyXCLASSParameter

            implicit none
            integer :: i, j, k, l, m                                                        !< loop variables
            integer :: i1, i2                                                               !< working variable: used for min. beam filling computation
            integer :: odif, odil                                                           !< working variable: used for determine obs. data indices
            integer :: iii                                                                  !< working variable: used for allocation
            integer :: IndexComp                                                            !< index of current component
            integer :: CurrComp                                                             !< counts the current component
            integer :: IndexMolecule                                                        !< index of current molecule
            integer :: IndexFreqRange                                                       !< index of current frequency range
            integer :: IndexSpectrum                                                        !< index of current spectrum
            integer :: IndexCurrIso                                                         !< index of current iso ratio
            integer :: LastIndexFreqRange                                                   !< working variable
            integer :: FirstIndex, LastIndex                                                !< working variables for freq. range index determination
            integer :: MaxNumberOfMolecules                                                 !< working variable: max. number of molecules (incl. isotop.)
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer :: c                                                                    !< working variable: number of occurrences of string s2 in s1
            integer, dimension(1000) :: HelpIntArray                                        !< working variable: used for globally defined iso ratios
            real*8 :: HelpValue                                                             !< working variable: used for counting
            real*8 :: beam_size                                                             !< working variable: beam size (eqn. 2 - 3)
            real*8 :: f1, f2                                                                !< working variable: used for min. beam filling computation
            real*8 :: min_val, max_val                                                      !< working variables: using for determination of pure dist.
            real*8 :: val, val2                                                             !< working value
            real*8, allocatable, dimension(:) :: BeamFillingFactor                          !< working variables: beam filling factor for each comp.
            character(len=1) :: s2                                                          !< working variable: used for subroutine countsubstring
            character(len=10) :: KeyWordString                                              !< keyword of current component
            character(len=40) :: CurrNameMolecule                                           !< working variable: name of current molecule
            character(len=4096) :: ParamName                                                !< working variable: current name of variable without
                                                                                            !< extension
            character(len=4096) :: s1                                                       !< working variable: used for subroutine countsubstring
            character(len=4096), allocatable, dimension(:) :: IsoCollPartnerFileName        !< working variable:
            logical :: AlreadyIncludedFlag                                                  !< working variable: indicates if current molecule name is
                                                                                            !< already known
            logical :: UseCFFLag                                                            !< indicates if CFFLag is used for at least one component
            logical :: UseDistance                                                          !< indicates if distance is defined for at least one component
            logical :: FittedCompShapeParameterFlag                                         !< indicates if source size or offset is fitted at least for
                                                                                            !< one component
            logical :: SameSizeFlag                                                         !< flag indicating that all components at the current
                                                                                            !< distance have the same source size

            !< print what you do
            if (printflag .and. trim(adjustl(ParallelizationMethod)) /= "NoMAGIX") then
                print '(" ")'
                print '(" ")'
                print '(9x, "Using optimized version of MAGIX for myXCLASS (", A, ").")', trim(adjustl(ParallelizationMethod))
                print '(" ")'
                print '(" ")'
                print '(9x, "Initialize myXCLASS program:")'
                print '(" ")'
                print '(" ")'
                print '(14x, "Reading parameter for myXCLASS from paramset variable .. ", $)'
            endif


            !=============================================================================================================================================
            !< analyze parameter list, i.e. get all parameters wich are given in the molfit, iso-ratio, and experimental xml files
            !<      - get name of molecules and the corresponding number of components
            !<      - get parameters for each component
            !<      - get iso master, iso molecule, and iso ratios
            !<      - get frequency ranges, background temperatures and slopes, nH densities, kappa, beta, telescope sizes, and database file


            !< get general parameters from paramset variable
            dbName = "../../../Database/cdms_sqlite.db"                                     !< set default value for db file
            NumberMolecules = 0                                                             !< reset counter for total number of molecules
            TotalNumberOfFrequencyRanges = 0                                                !< total number of frequency ranges
            NumberOfGlobalISORatios = 0
            IsoFlag = .false.
            SubBeamDescriptionFlag = .false.
            UseEmAbsFuncFlag = .false.
            MinNumTransitionsSQL = 0
            MaxNumTransitionsSQL = 0
            MaxElowSQL = 0.0
            MingASQL = 0.0
            OrderTransSQL = 0
            IndexCurrIso = 0
            HelpValue = 0.d0
            LocalOverlapFlag = .true.
            NoSubBeamFlag = .false.
            FittedCompShapeParameterFlag = .false.
            Do i = 1, parameternumber                                                       !< loop over all parameter

                ! Debug:
                ! print*,"FitParameterName(i) = ", trim(adjustl(FitParameterName(i)))


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif

                ! Debug:
                ! print*,"ParamName = ", trim(adjustl(ParamName))
                ! print*,"  paramset(1, i) = ", paramset(1, i)
                ! print*,"  FitParameterValue(i) = trim(adjustl(FitParameterValue(i)))


                !< get XCLASS root directory
                if (ParamName == "XCLASSRootDir") then
                    XCLASSRootDir = trim(adjustl(FitParameterValue(i)))

                    ! Debug:
                    ! print*,"XCLASSRootDir = ", trim(adjustl(XCLASSRootDir))


                !< get number of molecules (without iso molecules)
                elseif (ParamName == "Number_Molecules") then
                    NumberMolecules = int(paramset(1, i))


                !< get number of frequency ranges
                elseif (ParamName == "MAGIXImportStartFrequency") then
                    TotalNumberOfFrequencyRanges = TotalNumberOfFrequencyRanges + 1


                !< get iso flag
                elseif (ParamName == "NumberOfISOLines") then
                    if (paramset(1, i) > 0) then
                        NumberOfIsomasters = int(paramset(1, i))
                        IsoFlag = .true.


                        !< allocate memory for myXCLASSParameter variable
                        if (allocated(IsoMolecule)) then
                            deallocate(IsoMolecule, IsoMasters, IsoRatio, IsoCollPartnerFileName, stat = deallocstatus)
                            if (deallocstatus /= 0) then
                                Do ErrChannelIndex = 1, 2
                                    ErrChannel = AllErrChannels(ErrChannelIndex)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                                    write(ErrChannel, '(3x, "Can not deallocate variables IsoMolecule, IsoMasters, IsoRatio, ", $)')
                                    write(ErrChannel, '("and IsoCollPartnerFileName!")')
                                    write(ErrChannel, '(" ")')
                                end Do
                                close(logchannel)
                                stop 'Program aborted! Please restart the program!'
                            endif
                        endif
                        allocate(IsoMolecule(NumberOfIsomasters), IsoMasters(NumberOfIsomasters), IsoRatio(NumberOfIsomasters, 2), &
                                 IsoCollPartnerFileName(NumberOfIsomasters), stat = allocstatus)
                        if (allocstatus /= 0) then
                            Do ErrChannelIndex = 1, 2
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                                write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                                write(ErrChannel, '(5x, "- IsoMolecule(NumberOfIsomasters)")')
                                write(ErrChannel, '(5x, "- IsoMasters(NumberOfIsomasters)")')
                                write(ErrChannel, '(5x, "- IsoRatio(NumberOfIsomasters, 2)")')
                                write(ErrChannel, '(5x, "- IsoCollPartnerFileName(NumberOfIsomasters)")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Used parameter:")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(5x, "- NumberOfIsomasters = ", I8)') NumberOfIsomasters
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                            end Do
                            close(logchannel)
                            stop 'Program aborted! Please restart the program!'
                        endif
                        IsoMolecule = ""
                        IsoMasters = ""
                        IsoRatio = 0.d0
                        IndexCurrIso = 0
                        IsoCollPartnerFileName = ""                                         !< local variable !!!!
                    endif


                !< get number of global iso ratio definitions
                elseif (ParamName == "NumberOfGlobalISORatios") then
                    if (paramset(1, i) > 0) then
                        NumberOfGlobalISORatios = int(paramset(1, i))

                        ! Debug:
                        ! print*,'NumberOfGlobalISORatios = ', NumberOfGlobalISORatios


                        !< allocate memory for myXCLASSParameter variable
                        if (allocated(GlobalIsoRatioParameter)) then
                            deallocate(GlobalIsoRatioParameter, stat = deallocstatus)
                            if (deallocstatus /= 0) then
                                Do ErrChannelIndex = 1, 2
                                    ErrChannel = AllErrChannels(ErrChannelIndex)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                                    write(ErrChannel, '(3x, "Can not deallocate variable GlobalIsoRatioParameter!")')
                                    write(ErrChannel, '(" ")')
                                end Do
                                stop 'Program aborted! Please restart the program!'
                            endif
                        endif
                        allocate(GlobalIsoRatioParameter(NumberOfIsomasters, NumberOfGlobalISORatios, 2), stat = allocstatus)
                        if (allocstatus /= 0) then
                            Do ErrChannelIndex = 1, 2
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x, "Error in subroutine GetmyXCLASSParameter!")')
                                write(ErrChannel, '(3x, "Can not allocate variable ", $)')
                                write(ErrChannel, '("GlobalIsoRatioParameter(NumberOfIsomasters, NumberOfGlobalISORatios, 2)!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Used parameters:")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(5x, "- NumberOfIsomasters = ", I8)') NumberOfIsomasters
                                write(ErrChannel, '(5x, "- NumberOfGlobalISORatios = ", I8)') NumberOfGlobalISORatios
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                            end Do
                            close(logchannel)
                            stop 'Program aborted! Please restart the program!'
                        endif
                        GlobalIsoRatioParameter = 0
                    endif


                !< get IsoMolecule
                elseif (ParamName == "IsoMolecule" .and. IsoFlag) then
                    IndexCurrIso = IndexCurrIso + 1
                    IsoMolecule(IndexCurrIso) = trim(adjustl(FitParameterValue(i)))


                !< get IsoMasters
                elseif (ParamName == "IsoMasters" .and. IsoFlag) then
                    IsoMasters(IndexCurrIso) = trim(adjustl(FitParameterValue(i)))


                !< get IsoRatio
                elseif (ParamName == "IsoRatio" .and. IsoFlag) then
                    IsoRatio(IndexCurrIso, 1) = paramset(1, i)                              !< store the value of the iso ratio
                    if (paramset(2, i) == 1) then
                        IsoRatio(IndexCurrIso, 2) = sum(paramset(2, 1:i))                   !< store the free parameter index for this iso ratio param.
                    else
                        IsoRatio(IndexCurrIso, 2) = 0
                    endif


                !< get path and name of collision rate file
                elseif (ParamName == "IsoCollPartnerFileName" .and. IsoFlag) then
                    if (trim(adjustl(FitParameterValue(i))) /= "") then
                        IsoCollPartnerFileName(IndexCurrIso) = trim(adjustl(FitParameterValue(i)))
                    endif

                    ! Debug:
                    ! print*,">>> NumberMolColRates = ", NumberMolColRates


                !< get parameters for global iso ratio definitions
                elseif (ParamName == "IsoGlobalParameters" .and. IsoFlag .and. NumberOfGlobalISORatios > 0) then
                    s1 = trim(adjustl(FitParameterValue(i)))
                    s2 = ","
                    c = 0
                    call countsubstring(c, s1, s2)                                          !< determine number of "," character in parameter string
                    if (c > 0) then
                        c = c + 1
                        read(s1, *) HelpIntArray(1:c)
                        c = c / 2
                        k = 0
                        Do j = 1, c
                            k = k + 1
                            GlobalIsoRatioParameter(IndexCurrIso, j, 1) = HelpIntArray(k)
                            k = k + 1
                            GlobalIsoRatioParameter(IndexCurrIso, j, 2) = HelpIntArray(k)

                            ! Debug:
                            ! print *,"GlobalIsoRatioParameter(IndexCurrIso, j, :) = ", GlobalIsoRatioParameter(IndexCurrIso, j, :)
                        end Do
                    endif


                !< get path and name of SQLite3 database file
                elseif (ParamName == "MAGIXImportdbFilename") then
                    if (trim(adjustl(FitParameterValue(i))) /= "") then
                        dbName = trim(adjustl(FitParameterValue(i)))
                    endif
                endif
            end Do

            ! Debug:
            ! print*,"NumberMolecules = ", NumberMolecules
            ! print*,"TotalNumberOfFrequencyRanges = ", TotalNumberOfFrequencyRanges
            ! print*,"dbName = ", trim(adjustl(dbName))
            ! print*,"IsoFlag = ", IsoFlag
            ! if (IsoFlag) then
            !     print*,"NumberOfIsomasters = ", NumberOfIsomasters
            !     Do i = 1, NumberOfIsomasters
            !         print*,"IsoMolecule(i) = ", trim(adjustl(IsoMolecule(i)))
            !         print*,"IsoMasters(i) = ", trim(adjustl(IsoMasters(i)))
            !         print*,"IsoRatio(i, :) = ", IsoRatio(i, :)
            !     end Do
            ! endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for myXCLASSParameter variable
            if (allocated(StartFrequency)) then
                deallocate(StartFrequency, EndFrequency, StepFrequency, tbFlag, BackgroundTemperatureRange, &
                           TemperatureSlopeRange, BackgroundFilenNames, HydrogenColumnDensityRange, DustBetaRange, &
                           KappaRange, DustFilenNames, TelescopeSize, TelescopeBMIN, TelescopeBMAJ, TelescopeBPA, &
                           InterFlag, GlobalvLSR, Redshift_Range, SpectrumIndexForFreqRange, ObservationalDataList, &
                           BackgroundFromFile, DataPointIndexFreqRange, MoleculeNames, ConversionTableMAGIXmyXCLASSParam, &
                           DustTauFromFile, ContPhen_Range, Dust_Flag_Range, Phen_Flag_Range, &
                           NumTransFreqPerObsFreq, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x, "Can not deallocate variables StartFrequency, EndFrequency etc.!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    stop 'Program aborted! Please restart the program!'
                endif
            endif
            TotalNumberDataPoints = sum(lengthexpdata(1:NumberExpFiles))
            MaxNumberOfMolecules = NumberMolecules + 2 * NumberOfIsomasters
            iii = max(1, NumberFreeParameter)
            allocate(StartFrequency(TotalNumberOfFrequencyRanges), EndFrequency(TotalNumberOfFrequencyRanges), &
                     StepFrequency(TotalNumberOfFrequencyRanges), tbFlag(TotalNumberOfFrequencyRanges), &
                     BackgroundTemperatureRange(TotalNumberOfFrequencyRanges), &
                     TemperatureSlopeRange(TotalNumberOfFrequencyRanges), &
                     BackgroundFilenNames(TotalNumberOfFrequencyRanges), BackgroundFromFile(TotalNumberDataPoints), &
                     HydrogenColumnDensityRange(TotalNumberOfFrequencyRanges), DustBetaRange(TotalNumberOfFrequencyRanges), &
                     KappaRange(TotalNumberOfFrequencyRanges), DustFilenNames(TotalNumberOfFrequencyRanges), &
                     TelescopeSize(TotalNumberOfFrequencyRanges), TelescopeBMIN(TotalNumberOfFrequencyRanges), &
                     TelescopeBMAJ(TotalNumberOfFrequencyRanges), TelescopeBPA(TotalNumberOfFrequencyRanges), &
                     InterFlag(TotalNumberOfFrequencyRanges), GlobalvLSR(TotalNumberOfFrequencyRanges), &
                     Redshift_Range(TotalNumberOfFrequencyRanges), SpectrumIndexForFreqRange(TotalNumberOfFrequencyRanges), &
                     ObservationalDataList(TotalNumberDataPoints, 3), DataPointIndexFreqRange(TotalNumberOfFrequencyRanges, 2), &
                     MoleculeNames(MaxNumberOfMolecules), ConversionTableMAGIXmyXCLASSParam(iii, 2), &
                     DustTauFromFile(TotalNumberDataPoints), &
                     ContPhen_Range(6, TotalNumberOfFrequencyRanges), Dust_Flag_Range(TotalNumberOfFrequencyRanges), &
                     Phen_Flag_Range(TotalNumberOfFrequencyRanges), &
                     NumTransFreqPerObsFreq(MaxNumberOfMolecules, TotalNumberDataPoints), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                    write(ErrChannel, '(5x, "- StartFrequency(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- EndFrequency(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- StepFrequency(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- tbFlag(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- BackgroundTemperatureRange(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- TemperatureSlopeRange(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- BackgroundFilenNames(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- HydrogenColumnDensityRange(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- DustBetaRange(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- KappaRange(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- DustFilenNames(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- TelescopeSize(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- TelescopeBMIN(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- TelescopeBMAJ(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- TelescopeBPA(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- InterFlag(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- GlobalvLSR(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- Redshift_Range(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- SpectrumIndexForFreqRange(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- ObservationalDataList(TotalNumberDataPoints, 3)")')
                    write(ErrChannel, '(5x, "- DataPointIndexFreqRange(TotalNumberOfFrequencyRanges, 2)")')
                    write(ErrChannel, '(5x, "- MoleculeNames(MaxNumberOfMolecules)")')
                    write(ErrChannel, '(5x, "- ConversionTableMAGIXmyXCLASSParam(iii, 2)")')
                    write(ErrChannel, '(5x, "- DustTauFromFile(TotalNumberDataPoints)")')
                    write(ErrChannel, '(5x, "- BackgroundFromFile(TotalNumberDataPoints)")')
                    write(ErrChannel, '(5x, "- ContPhen_Range(6, TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- Dust_Flag_Range(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- Phen_Flag_Range(TotalNumberOfFrequencyRanges)")')
                    write(ErrChannel, '(5x, "- NumTransFreqPerObsFreq(MaxNumberOfMolecules, TotalNumberDataPoints)")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameters:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- TotalNumberOfFrequencyRanges = ", I8)') TotalNumberOfFrequencyRanges
                    write(ErrChannel, '(5x, "- TotalNumberDataPoints = ", I8)') TotalNumberDataPoints
                    write(ErrChannel, '(5x, "- MaxNumberOfMolecules = ", I8)') MaxNumberOfMolecules
                    write(ErrChannel, '(5x, "- NumberFreeParameter = ", I8)') NumberFreeParameter
                    write(ErrChannel, '(5x, "- iii = ", I8)') iii
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                close(logchannel)
                stop 'Program aborted! Please restart the program!'
            endif
            StartFrequency = 0.d0
            EndFrequency = 0.d0
            StepFrequency = 0.d0
            tbFlag = .true.
            BackgroundTemperatureRange = 0.d0
            TemperatureSlopeRange = 0.d0
            HydrogenColumnDensityRange = 0.d0
            DustBetaRange = 0.d0
            KappaRange = 0.d0
            ContPhen_Range = 0.d0
            DustFilenNames = ""
            BackgroundFilenNames = ""
            TelescopeSize = 0.d0
            TelescopeBMIN = 0.d0
            TelescopeBMAJ = 0.d0
            TelescopeBPA = 0.d0
            GlobalvLSR = 0.d0
            Redshift_Range = 0.d0
            InterFlag = .false.
            NumberModelPixelXX = 0
            NumberModelPixelYY = 0
            SpectrumIndexForFreqRange = 0
            ObservationalDataList = 0.d0
            NumTransFreqPerObsFreq = 0
            DataPointIndexFreqRange = 0
            MoleculeNames = ""
            ConversionTableMAGIXmyXCLASSParam = 0
            DustTauFromFile = 0.d0
            BackgroundFromFile = 0.d0
            Dust_Flag_Range = .false.
            Phen_Flag_Range = .false.


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< get parameter for each frequency range
            IndexFreqRange = 0                                                              !< reset index for current frequency range
            IndexSpectrum = 0                                                               !< reset index for current spectrum
            LastIndexFreqRange = 0
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif

                ! Debug:
                ! print*," "
                ! print*,"ParamName = ", trim(adjustl(ParamName))
                ! print*,"paramset(1, i) = ", paramset(1, i)
                ! print*,"trim(adjustl(FitParameterValue(i))) = ", trim(adjustl(FitParameterValue(i)))
                ! print*,"IndexFreqRange = ", IndexFreqRange


                !< increase counter for number of obs. data files
                if (ParamName == "MAGIXImportExpFileNames") then
                    IndexSpectrum = IndexSpectrum + 1


                !< get first frequency of each frequency range
                elseif (ParamName == "MAGIXImportStartFrequency") then
                    IndexFreqRange = IndexFreqRange + 1
                    StartFrequency(IndexFreqRange) = paramset(1, i)
                    SpectrumIndexForFreqRange(IndexFreqRange) = IndexSpectrum


                !< get last frequency of each frequency range
                elseif (ParamName == "MAGIXImportEndFrequency") then
                    EndFrequency(IndexFreqRange) = paramset(1, i)


                !< get step size of each frequency range
                elseif (ParamName == "MAGIXImportStepFrequency") then
                    StepFrequency(IndexFreqRange) = paramset(1, i)


                !< get definition for t_back_flag
                elseif (ParamName == "MAGIXImportTBackFlag") then
                    if (paramset(1, i) == 0.d0) then
                        tbFlag(IndexFreqRange) = .false.
                    else
                        tbFlag(IndexFreqRange) = .true.
                    endif


                !< get global definition for background temperature
                elseif (ParamName == "MAGIXImportBackgroundTemperature") then
                    BackgroundTemperatureRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for temperature slope
                elseif (ParamName == "MAGIXImportTemperatureSlope") then
                    TemperatureSlopeRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for background file
                elseif (ParamName == "MAGIXImportBackgroundFileName") then
                    if (trim(adjustl(FitParameterValue(i))) /= "") then
                        BackgroundFilenNames(IndexFreqRange) = trim(adjustl(FitParameterValue(i)))
                    endif


                !< get global definition for n_H
                elseif (ParamName == "MAGIXImportHydrogenColumnDensity") then
                    HydrogenColumnDensityRange(IndexFreqRange) = paramset(1, i)
                    Dust_Flag_Range(IndexFreqRange) = .true.


                !< get global definition for dust beta
                elseif (ParamName == "MAGIXImportDustBeta") then
                    DustBetaRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for kappa
                elseif (ParamName == "MAGIXImportKappa") then
                    KappaRange(IndexFreqRange) = paramset(1, i) * 2.d0 * 1.66d-24 / 100.d0  !< kappa_1300 * M(H2)/M_d_M_g


                !< get global definition for ContPhenID
                elseif (ParamName == "MAGIXImportContPhenID") then
                    ContPhen_Range(1, IndexFreqRange) = paramset(1, i)
                    if (dabs(ContPhen_Range(1, IndexFreqRange)) > tiny(1.d0)) then
                        Phen_Flag_Range(IndexFreqRange) = .true.
                    endif


                !< get global definition for ContParam1
                elseif (ParamName == "MAGIXImportContParam1") then
                    ContPhen_Range(2 ,IndexFreqRange) = paramset(1, i)


                !< get global definition for ContParam2
                elseif (ParamName == "MAGIXImportContParam2") then
                    ContPhen_Range(3, IndexFreqRange) = paramset(1, i)


                !< get global definition for ContParam3
                elseif (ParamName == "MAGIXImportContParam3") then
                    ContPhen_Range(4, IndexFreqRange) = paramset(1, i)


                !< get global definition for ContParam4
                elseif (ParamName == "MAGIXImportContParam4") then
                    ContPhen_Range(5, IndexFreqRange) = paramset(1, i)


                !< get global definition for ContParam5
                elseif (ParamName == "MAGIXImportContParam5") then
                    ContPhen_Range(6, IndexFreqRange) = paramset(1, i)


                !< get global definition for dust file
                elseif (ParamName == "MAGIXImportDustFileName") then
                    if (trim(adjustl(FitParameterValue(i))) /= "") then
                        DustFilenNames(IndexFreqRange) = trim(adjustl(FitParameterValue(i)))
                    endif


                !< get global v_LSR
                elseif (ParamName == "MAGIXImportGlobalvLSR") then
                    GlobalvLSR((LastIndexFreqRange + 1):IndexFreqRange) = paramset(1, i)


                !< get global Redshift
                elseif (ParamName == "MAGIXImportRedshift") then
                    Redshift_Range((LastIndexFreqRange + 1):IndexFreqRange) = paramset(1, i)


                !< get size of telescope
                elseif (ParamName == "MAGIXImportTelescopeSize") then
                    TelescopeSize((LastIndexFreqRange + 1):IndexFreqRange) = paramset(1, i)


                !< get size of telescope BMIN
                elseif (ParamName == "MAGIXImportTelescopeBMIN") then
                    if (paramset(1, i) > -1.d99) then
                        TelescopeBMIN((LastIndexFreqRange + 1):IndexFreqRange) = paramset(1, i)
                    endif


                !< get size of telescope BMAJ
                elseif (ParamName == "MAGIXImportTelescopeBMAJ") then
                    if (paramset(1, i) > -1.d99) then
                        TelescopeBMAJ((LastIndexFreqRange + 1):IndexFreqRange) = paramset(1, i)
                    endif


                !< get size of telescope BPA
                elseif (ParamName == "MAGIXImportTelescopeBPA") then
                    if (paramset(1, i) > -1.d99) then
                        TelescopeBPA((LastIndexFreqRange + 1):IndexFreqRange) = paramset(1, i)
                    endif


                !< get number of pixel for sub-beam modeling along x-direction
                elseif (ParamName == "MAGIXImportNumModelPixelXX") then
                    NumberModelPixelXX = int(paramset(1, i))


                !< get number of pixel for sub-beam modeling along y-direction
                elseif (ParamName == "MAGIXImportNumModelPixelYY") then
                    NumberModelPixelYY = int(paramset(1, i))


                !< get local-overlap flag
                elseif (ParamName == "MAGIXImportLocalOverlapFlag") then
                    if (paramset(1, i) == 0.d0) then
                        LocalOverlapFlag = .false.
                    else
                        LocalOverlapFlag = .true.
                    endif


                !< get NoSubBeam flag
                elseif (ParamName == "MAGIXImportNoSubBeamFlag") then
                    if (paramset(1, i) == 0.d0) then
                        NoSubBeamFlag = .false.
                    else
                        NoSubBeamFlag = .true.
                    endif


                !< get path for emission/absorption function files
                elseif (ParamName == "MAGIXImportEmAbsPATH") then
                    if (trim(adjustl(FitParameterValue(i))) /= "") then
                        EmsAbsPATH = trim(adjustl(FitParameterValue(i)))
                        UseEmAbsFuncFlag = .true.
                    endif


                !< get interferometer flag
                elseif (ParamName == "MAGIXImportInterferrometerFlag") then
                    if (paramset(1, i) == 0.d0) then
                        InterFlag((LastIndexFreqRange + 1):IndexFreqRange) = .false.
                    else
                        InterFlag((LastIndexFreqRange + 1):IndexFreqRange) = .true.
                    endif
                    LastIndexFreqRange = IndexFreqRange                                     !< variable 'LastIndexFreqRange' is necessary, because
                                                                                            !< size of telescope is given only for each spectrum and not
                                                                                            !< for each frequency range
                endif

                ! Debug:
                ! print*,'i, ParamName, paramset(1, i) = ', i, trim(adjustl(ParamName)), paramset(1, i)
            end Do

            ! Debug:
            ! print*,"SpectrumIndexForFreqRange = ", SpectrumIndexForFreqRange(:)
            ! print*,"StartFrequency = ", StartFrequency(:)
            ! print*,"EndFrequency = ", EndFrequency(:)
            ! print*,"StepFrequency = ", StepFrequency(:)
            ! print*,"BackgroundTemperatureRange = ", BackgroundTemperatureRange(:)
            ! print*,"TemperatureSlopeRange = ", TemperatureSlopeRange(:)
            ! print*,"Dust_Flag_Range = ", Dust_Flag_Range(:)
            ! print*,"HydrogenColumnDensityRange = ", HydrogenColumnDensityRange(:)
            ! print*,"DustBetaRange = ", DustBetaRange(:)
            ! print*,"KappaRange = ", KappaRange(:)
            ! print*,"Phen_Flag_Range = ", Phen_Flag_Range(:)
            ! print*,"ContPhen_Range(1, :) = ", ContPhen_Range(1, :)
            ! print*,"ContPhen_Range(2, :) = ", ContPhen_Range(2, :)
            ! print*,"TelescopeSize = ", TelescopeSize(:)
            ! print*,"TelescopeBMIN = ", TelescopeBMIN(:)
            ! print*,"TelescopeBMAJ = ", TelescopeBMAJ(:)
            ! print*,"TelescopeBPA = ", TelescopeBPA(:)
            ! print*,"InterFlag = ", InterFlag
            ! print*,"GlobalvLSR = ", GlobalvLSR(:)
            ! print*,'tbFlag(:) = ', tbFlag(:)
            ! print*,"LocalOverlapFlag = ", LocalOverlapFlag
            ! print*,"NoSubBeamFlag = ", NoSubBeamFlag
            ! print*,"UseEmAbsFuncFlag = ", UseEmAbsFuncFlag
            ! print*,"EmsAbsPATH = ", trim(adjustl(EmsAbsPATH))


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< prepare observational data as one long 2D list
            k = 0
            Do i = 1, IndexSpectrum                                                         !< variable 'IndexSpectrum' is NOT identical to variable
                                                                                            !< 'NumberExpFiles' !!!!!!!!!!!
                Do j = 1, lengthexpdata(i)                                                  !< loop over all points of ith spectrum
                    k = k + 1                                                               !< increase data point counter
                    ObservationalDataList(k, :) = (/ expdatax(i, j, 1), expdatay(i, j, 1), expdatae(i, j, 1) /)
                end Do
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< define indices of first and last frequency (data point) of each frequency range), respectively
            Do l = 1, TotalNumberOfFrequencyRanges                                          !< loop over all frequency ranges
                i = SpectrumIndexForFreqRange(l)                                            !< get spectrum index
                FirstIndex = 0
                LastIndex = 0
                Do j = 1, lengthexpdata(i)                                                  !< loop over all points of ith spectrum
                    if (expdatax(i, j, 1) <= StartFrequency(l)) then
                        FirstIndex = FirstIndex + 1
                    endif
                    if (expdatax(i, j, 1) <= EndFrequency(l)) then
                        LastIndex = LastIndex + 1
                    ! else
                    !     exit
                    endif
                end Do                                                                      !< j: loop over all points of ith spectrum

                ! Debug:
                ! print*,"FirstIndex = ", FirstIndex
                ! print*,"LastIndex = ", LastIndex


                !< check indices
                if (FirstIndex == 0) then
                    FirstIndex = 1
                endif
                if (LastIndex == 0) then
                    LastIndex = 1
                endif
                if (FirstIndex > LastIndex) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"An error occurred by calculating the indices of each frequency range!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    stop 'Program aborted! Please restart the program!'
                else


                    !< determine offset for indices for each frequency range
                    k = 0
                    if (i > 1) then
                        k = sum(lengthexpdata(1:(i - 1)))
                    endif


                    !< define first and  last observational data file index
                    odif = FirstIndex + k                                                   !< first observational data file index
                    odil = LastIndex + k                                                    !< last observational data file index


                    !< check, if first and / or last index was unset before
                    Do j = 1, (l - 1)                                                       !< loop over all previous frequency ranges
                        if (i == SpectrumIndexForFreqRange(j)) then                         !< check, if jth range corresponds to same obs. data file


                            !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                            !< TO DO:       improve the following lines to take all kinds of range overlap into account !!<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
                            !
                            !    if (DataPointIndexFreqRange(j, 1) == odif) then                 !< first obs. data file index was used before
                            !    endif
                            if (DataPointIndexFreqRange(j, 2) == odif) then                 !< last obs. data file index was used before
                                odif = odif + 1
                            endif
                            !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


                        endif
                    end Do                                                                  !< j: loop over all previous frequency ranges


                    !< store observational data file indices
                    DataPointIndexFreqRange(l, :) = (/ odif, odil /)

                    ! Debug:
                    ! print*," "
                    ! print*,"l = ", l
                    ! print*,"StartFrequency(l) = ", StartFrequency(l)
                    ! print*,"EndFrequency(l) = ", EndFrequency(l)
                    ! print*,"FirstIndex = ", FirstIndex
                    ! print*,"LastIndex = ", LastIndex
                    ! print*,"k = ", k
                    ! print*,"i = ", i
                    ! print*, "odif, odil = ", odif, odil
                    ! print*,"ObservationalDataList(odif, 1) = ", ObservationalDataList(odif, 1)
                    ! print*,"ObservationalDataList(odil, 1) = ", ObservationalDataList(odil, 1)
                    ! Do j = 1, lengthexpdata(i)
                    !     print*,j, expdatax(i, j, 1)
                    ! end Do
                endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< import and interpolate dust file to get tau_dust from user-defined file
                if (trim(adjustl(DustFilenNames(l))) /= "") then
                    call ImportDustFile(l)
                endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< import and interpolate background file to get background from user-defined file
                if (trim(adjustl(BackgroundFilenNames(l))) /= "") then
                    call ImportBackgroundFile(l)
                endif
            end Do                                                                          !< l: loop over all frequency ranges

            ! Debug:
            ! print*,"DataPointIndexFreqRange(:, 1) = ", DataPointIndexFreqRange(:, 1)
            ! print*,"DataPointIndexFreqRange(:, 2) = ", DataPointIndexFreqRange(:, 2)
            ! print*,"ObservationalDataList(DataPointIndexFreqRange(1, 1), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 1), :)
            ! print*,"ObservationalDataList(DataPointIndexFreqRange(1, 2), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 2), :)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine list of all molecules which are defined in molfit and iso ratio files
            TotalNumberOfMolecules = 0                                                      !< reset counter for total number of molecules
            NumberMolfitMolecules = 0                                                       !< reset counter for number of molecules in molfit file
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif

                ! Debug:
                ! print*,"i, trim(adjustl(FitParameterName(i))) = ", i, trim(adjustl(FitParameterName(i)))


                !< determine names of molecules
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))
                    TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                    NumberMolfitMolecules = NumberMolfitMolecules + 1
                    MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(CurrNameMolecule))

                    ! Debug:
                    ! print*,"CurrNameMolecule = ", trim(adjustl(CurrNameMolecule))
                    ! print*,"TotalNumberOfMolecules = ", TotalNumberOfMolecules
                endif
            end Do


            !< check, if iso ratios are defined and determine total number of molecules
            if (IsoFlag) then
                k = TotalNumberOfMolecules
                Do i = 1, k
                    CurrNameMolecule = trim(adjustl(MoleculeNames(i)))
                    Do j = 1, NumberOfIsomasters
                        if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMolecule(j)))) then
                            call CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMolecules, IsoMasters(j))
                            if (.not. AlreadyIncludedFlag) then
                                TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                                MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(IsoMasters(j)))
                           endif
                        elseif (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMasters(j)))) then
                            call CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMolecules, IsoMolecule(j))
                            if (.not. AlreadyIncludedFlag) then
                                TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                                MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(IsoMolecule(j)))
                           endif
                        endif
                    end Do
                end Do
            endif

            ! Debug:
            !     print*," "
            !     Do i = 1, TotalNumberOfMolecules
            !         print*,"GetmyXCLASSParameter:    i, MoleculeNames(i) = ", i, MoleculeNames(i)
            !     end Do


            !=============================================================================================================================================
            !< get calculation parameters


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< prepare some variables for isotopologues only


            !< allocate memory for iso conversion tables
            if (allocated(IsoRatioConversionTable)) then
                deallocate(IsoRatioConversionTable, IsoNfitConversionTable, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x, "Can not deallocate variables IsoNfitConversionTable and &
                                                &IsoRatioConversionTable!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    close(logchannel)
                    stop 'Program aborted! Please restart the program!'
                endif
            endif
            allocate(IsoNfitConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules, 2), &
                     IsoRatioConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                    write(ErrChannel, '(5x, "- IsoNfitConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules, 2)")')
                    write(ErrChannel, '(5x, "- IsoRatioConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules)")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameter:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- TotalNumberOfMolecules = ", I8)') TotalNumberOfMolecules
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                close(logchannel)
                stop 'Program aborted! Please restart the program!'
            endif
            IsoNfitConversionTable = 0
            IsoRatioConversionTable = 0.d0


            !< define conversion tables IsoNfitConversionTable and IsoRatioConversionTable
            if (IsoFlag) then
                Do i = 1, parameternumber                                                   !< loop over all parameter


                    !< remove [[..]] expansion of parameter name
                    ParamName = trim(adjustl(FitParameterName(i)))
                    k = index(ParamName, "[[")
                    if (k > 0) then
                        ParamName = ParamName(:(k - 1))
                    endif


                    !< determine name of current molecule
                    if (ParamName == "Molecule_Name") then
                        CurrNameMolecule = trim(adjustl(FitParameterValue(i)))

                        ! Debug:
                        ! print*,"CurrNameMolecule = ", trim(adjustl(CurrNameMolecule))


                    !< determine index for current molecule
                    elseif (ParamName == "Molecule_Inex") then
                        IndexMolecule = int(paramset(1, i))
                        if (IndexMolecule == 0) then
                            print '(" ")'
                            print '(2x, "Error in subroutine GetmyXCLASSParameter:")'
                            print '(4x, "Can not determine molecule index for molecule: ", A, " ")', trim(adjustl(CurrNameMolecule))
                            print '(4x, "Set index to 1!")'
                            print '(" ")'
                            print '(4x,"Known molecules:")'
                            Do k = 1, TotalNumberOfMolecules
                                print '(4x, I5, 3x, A)', k, trim(adjustl(MoleculeNames(k)))
                            end Do
                            IndexMolecule = 1
                        endif

                        ! Debug:
                        ! print*,"IndexMolecule = ", IndexMolecule


                        !< create conversion table between molecules and isotopologues
                        IsoRatioConversionTable(IndexMolecule, IndexMolecule) = 1.d0
                        IsoNfitConversionTable(IndexMolecule, IndexMolecule, :) = 0
                        Do j = 1, NumberOfIsomasters                                        !< loop over all defined isotopologues

                            ! Debug:
                            ! print*," "
                            ! print*," "
                            ! print*,"j = ", j
                            ! print*,"CurrNameMolecule = ", trim(adjustl(CurrNameMolecule))
                            ! print*,"IsoMolecule(j)   = ", trim(adjustl(IsoMolecule(j)))
                            ! print*,"IsoMasters(j)    = ", trim(adjustl(IsoMasters(j)))
                            ! print*,"IsoRatio(j, 1)   = ", IsoRatio(j, 1)


                            if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMolecule(j)))) then
                                l = IndexMolecule
                                m = 0
                                Do k = 1, TotalNumberOfMolecules
                                    if (l == k) then
                                        IsoRatioConversionTable(l, k) = 1.d0
                                        IsoNfitConversionTable(l, k, 1) = 1
                                    elseif (trim(adjustl(IsoMasters(j))) == trim(adjustl(MoleculeNames(k)))) then
                                        m = k
                                        IsoNfitConversionTable(l, m, 1) = int(IsoRatio(j, 2))
                                        IsoNfitConversionTable(l, m, 2) = j
                                        IsoRatioConversionTable(l, m) = IsoRatio(j, 1)
                                        if (dabs(IsoRatio(j, 1)) > 1.e-30) then
                                            IsoRatioConversionTable(m, l) = 1.d0 / IsoRatio(j, 1)
                                        else
                                            IsoRatioConversionTable(m, l) = 1.d0
                                        endif

                                        ! Debug:
                                        ! print*,"m, l, IsoRatioConversionTable(m, l) = ", m, l, IsoRatioConversionTable(m, l)
                                    endif
                                end Do

                                ! Debug:
                                ! print*,"IsoMolecule(j), IsoRatio(j, 1) = ", trim(adjustl(IsoMolecule(j))), "    ", IsoRatio(j, 1)


                            elseif (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMasters(j)))) then
                                l = IndexMolecule
                                m = 0
                                Do k = 1, TotalNumberOfMolecules
                                    if (l == k) then
                                        IsoRatioConversionTable(l, k) = 1.d0
                                        IsoNfitConversionTable(l, k, 1) = 1
                                    elseif (trim(adjustl(IsoMolecule(j))) == trim(adjustl(MoleculeNames(k)))) then
                                        m = k
                                        IsoNfitConversionTable(l, m, 1) = int(IsoRatio(j, 2))
                                        IsoNfitConversionTable(l, m, 2) = -j
                                        if (dabs(IsoRatio(j, 1)) > 1.e-30) then
                                            IsoRatioConversionTable(l, m) = 1.d0 / IsoRatio(j, 1)
                                        else
                                            IsoRatioConversionTable(l, m) = 1.d0
                                        endif
                                        IsoRatioConversionTable(m, l) = IsoRatio(j, 1)
                                    endif
                                end Do

                                ! Debug:
                                ! print*,"IsoMasters(j), 1.d0 / IsoRatio(j, 1) = ", trim(adjustl(IsoMasters(j))), "    ", 1.d0 / IsoRatio(j, 1)
                            endif

                            ! Debug:
                            ! print*,"m, l, IsoRatioConversionTable(m, l) = ", m, l, IsoRatioConversionTable(m, l)
                        end Do
                    endif
                end Do
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine the number of components for each molecule
            if (allocated(NumberComponentsPerMolecule)) then
                deallocate(NumberComponentsPerMolecule, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x, "Can not deallocate variable NumberComponentsPerMolecule!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    close(logchannel)
                    stop 'Program aborted! Please restart the program!'
                endif
            endif
            allocate(NumberComponentsPerMolecule(0:NumberMolfitMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x, "Can not allocate variable NumberComponentsPerMolecule(0:NumberMolfitMolecules)!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameter:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- NumberMolfitMolecules = ", I8)') NumberMolfitMolecules
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                close(logchannel)
                stop 'Program aborted! Please restart the program!'
            endif
            NumberComponentsPerMolecule = 0
            IndexMolecule = 1
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< determine name of current molecule
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))

                    ! Debug:
                    ! print*,"CurrNameMolecule = ", trim(adjustl(CurrNameMolecule))


                !< determine index for current molecule
                elseif (ParamName == "Molecule_Inex") then
                    IndexMolecule = int(paramset(1, i))
                    if (IndexMolecule == 0) then
                        print '(" ")'
                        print '(2x, "Error in subroutine GetmyXCLASSParameter:")'
                        print '(4x, "Can not determine molecule index for molecule: ", A, " ")', trim(adjustl(CurrNameMolecule))
                        print '(4x, "Set index to 1!")'
                        print '(" ")'
                        print '(4x,"Known molecules:")'
                        Do k = 1, TotalNumberOfMolecules
                            print '(4x, I5, 3x, A)', k, trim(adjustl(MoleculeNames(k)))
                        end Do
                        IndexMolecule = 1
                    endif

                    ! Debug:
                    ! print*,"IndexMolecule = ", IndexMolecule


                !< get number of components for current molecule
                elseif (ParamName == "Number_MolLine") then
                    NumberComponentsPerMolecule(IndexMolecule) = int(paramset(1, i))

                    ! Debug:
                    ! print*,"NumberComponentsPerMolecule(IndexMolecule) = ", NumberComponentsPerMolecule(IndexMolecule)
                endif
            end Do


            !< determine total number of components and allocate memory
            TotalNumberComponents = sum(NumberComponentsPerMolecule(:))
            if (allocated(myXCLASSParameter)) then
                deallocate(myXCLASSParameter, CompMoleculeIndex, GeometryFlag, tdFlag, nHFlag, vWidthLimits, &
                           vOffLimits, KindOfMolecule, LineProfileFunction, LTEFlag, ThinFlag, SortedDistances, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x, "Can not deallocate variables myXCLASSParameter, CompMoleculeIndex etc.!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    close(logchannel)
                    stop 'Program aborted! Please restart the program!'
                endif
            endif
            allocate(myXCLASSParameter(16, TotalNumberComponents), CompMoleculeIndex(TotalNumberComponents), &
                     GeometryFlag(TotalNumberComponents), tdFlag(TotalNumberComponents), nHFlag(TotalNumberComponents), &
                     vWidthLimits(TotalNumberComponents, 4), vOffLimits(TotalNumberComponents, 3), &
                     KindOfMolecule(TotalNumberComponents), LineProfileFunction(TotalNumberComponents), &
                     LTEFlag(TotalNumberComponents), ThinFlag(TotalNumberComponents),  &
                     SortedDistances(TotalNumberComponents), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                    write(ErrChannel, '(5x, "- myXCLASSParameter(16, TotalNumberComponents)")')
                    write(ErrChannel, '(5x, "- CompMoleculeIndex(TotalNumberComponents)")')
                    write(ErrChannel, '(5x, "- GeometryFlag(TotalNumberComponents)")')
                    write(ErrChannel, '(5x, "- tdFlag(TotalNumberComponents)")')
                    write(ErrChannel, '(5x, "- nHFlag(TotalNumberComponents)")')
                    write(ErrChannel, '(5x, "- vWidthLimits(TotalNumberComponents, 4)")')
                    write(ErrChannel, '(5x, "- vOffLimits(TotalNumberComponents, 3)")')
                    write(ErrChannel, '(5x, "- KindOfMolecule(TotalNumberComponents)")')
                    write(ErrChannel, '(5x, "- LineProfileFunction(TotalNumberComponents)")')
                    write(ErrChannel, '(5x, "- LTEFlag(TotalNumberComponents)")')
                    write(ErrChannel, '(5x, "- ThinFlag(TotalNumberComponents)")')
                    write(ErrChannel, '(5x, "- SortedDistances(TotalNumberComponents)")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameter:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- TotalNumberComponents = ", I8)') TotalNumberComponents
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                close(logchannel)
                stop 'Program aborted! Please restart the program!'
            endif
            myXCLASSParameter = 0.d0
            vWidthLimits = 0.d0
            vOffLimits = 0.d0
            SortedDistances = 0.d0
            CompMoleculeIndex = 1
            KindOfMolecule = 0
            LineProfileFunction = 1
            GeometryFlag = 0
            tdFlag = .false.
            nHFlag = .false.
            LTEFlag = .true.
            ThinFlag = .false.
            UseCFFLag = .false.
            UseDistance = .false.


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< get calculation parameters
            IndexMolecule = 1                                                               !< reset index for current molecule
            IndexComp = 0                                                                   !< reset index for current component
            CurrComp = 0                                                                    !< reset index for current component of current molecule
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif

                ! Debug:
                ! print*,"ParamName = ", ParamName


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< get name of current molecule
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))

                    ! Debug:
                    ! print*,"CurrNameMolecule = ", trim(adjustl(CurrNameMolecule))


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< database parameters


                !< get min. number of transitions
                elseif (ParamName == "MinNumTransitionsSQL") then
                    MinNumTransitionsSQL = int(paramset(1, i))

                    ! Debug:
                    ! print*,"MinNumTransitionsSQL = ", MinNumTransitionsSQL


                !< get max. number of transitions
                elseif (ParamName == "MaxNumTransitionsSQL") then
                    MaxNumTransitionsSQL = int(paramset(1, i))

                    ! Debug:
                    ! print*,"MaxNumTransitionsSQL = ", MaxNumTransitionsSQL


                !< get highest lower energy of transitions
                elseif (ParamName == "MaxElowSQL") then
                    MaxElowSQL = paramset(1, i)

                    ! Debug:
                    ! print*,"MaxElowSQL = ", MaxElowSQL


                !< get min. intensity of transitions
                elseif (ParamName == "MingASQL") then
                    MingASQL = paramset(1, i)

                    ! Debug:
                    ! print*,"MingASQL = ", MingASQL


                !< get min. intensity of transitions
                elseif (ParamName == "OrderTransSQL") then
                    OrderTransSQL = int(paramset(1, i))

                    ! Debug:
                    ! print*,"OrderTransSQL = ", OrderTransSQL


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< molfit format parameters


                !< determine index for current molecule
                elseif (ParamName == "Molecule_Inex") then
                    IndexMolecule = int(paramset(1, i))
                    if (IndexMolecule == 0) then
                        print '(" ")'
                        print '(2x, "Error in subroutine GetmyXCLASSParameter:")'
                        print '(4x, "Can not determine molecule index for molecule: ", A, " ")', trim(adjustl(CurrNameMolecule))
                        print '(4x, "Set index to 1!")'
                        print '(" ")'
                        print '(4x,"Known molecules:")'
                        Do k = 1, TotalNumberOfMolecules
                            print '(4x, I5, 3x, A)', k, trim(adjustl(MoleculeNames(k)))
                        end Do
                        IndexMolecule = 1
                    endif

                    ! Debug:
                    ! print*,"IndexMolecule = ", IndexMolecule


                !< determine molecule index
                elseif (ParamName == "Number_MolLine") then
                    CurrComp = 0
                    IndexComp = sum(NumberComponentsPerMolecule(:(IndexMolecule - 1)))


                !< get EmissionFlag
                elseif (ParamName == "MolfitFileFormatVersion") then


                    !< increase counter for component
                    IndexComp = IndexComp + 1                                               !< increase indec for current component
                    CurrComp = CurrComp + 1
                    CompMoleculeIndex(IndexComp) = IndexMolecule

                    ! Debug:
                    ! print*,"IndexComp = ", IndexComp
                    ! print*,"CurrComp = ", CurrComp
                    ! print*,"CompMoleculeIndex(IndexComp) = ", CompMoleculeIndex(IndexComp)


                !< get GeometryFlag
                elseif (ParamName == "GeometryFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "NO") then                   !< no geometry of current layer is defined
                        GeometryFlag(IndexComp) = 0
                    elseif (trim(adjustl(FitParameterValue(i))) == "CC") then               !< current layer is a centered circle
                        GeometryFlag(IndexComp) = 1
                    elseif (trim(adjustl(FitParameterValue(i))) == "CO") then               !< current layer is a circle of an offset
                        GeometryFlag(IndexComp) = 2
                    elseif (trim(adjustl(FitParameterValue(i))) == "BC") then               !< current layer is a centered box
                        GeometryFlag(IndexComp) = 3
                    elseif (trim(adjustl(FitParameterValue(i))) == "BO") then               !< current layer is a box of an offset
                        GeometryFlag(IndexComp) = 4
                    endif

                    ! Debug:
                    ! print*,"GeometryFlag(IndexComp) = ", GeometryFlag(IndexComp)


                !< get MoleculeFlag describing the kind of current molecule, i.e. molecule or RRL or continuum etc.
                elseif (ParamName == "MoleculeFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "M") then                    !< current molecule is a "normal" molecule
                        KindOfMolecule(IndexComp) = 0
                    elseif (trim(adjustl(FitParameterValue(i))) == "R") then                !< current molecule is a RRL
                        KindOfMolecule(IndexComp) = 1
                    elseif (trim(adjustl(FitParameterValue(i))) == "D") then                !< current molecule describes dust continuum
                        KindOfMolecule(IndexComp) = 3
                    elseif (trim(adjustl(FitParameterValue(i))) == "P") then                !< current molecule describes phenomenolical continuum
                        KindOfMolecule(IndexComp) = 6
                    endif

                    ! Debug:
                    ! print*,"KindOfMolecule(IndexComp) = ", KindOfMolecule(IndexComp)


                !< get LineWidthFlag
                elseif (ParamName == "LineWidthFlag") then
                    LineProfileFunction(IndexComp) = 1

                    ! Debug:
                    ! print*,"LineProfileFunction(IndexComp) = ", LineProfileFunction(IndexComp)


                !< get tdFlag (inline description of dust temperature offset and slope)
                elseif (ParamName == "tdFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "T") then
                        tdFlag(IndexComp) = .true.
                    else
                        tdFlag(IndexComp) = .false.
                    endif

                    ! Debug:
                    ! print*,"tdFlag(IndexComp) = ", tdFlag(IndexComp)


                    !    !< get nHFlag (inline description of dust parameters)
                    !    elseif (ParamName == "nHFlag") then
                    !        if (trim(adjustl(FitParameterValue(i))) == "T") then
                    !            nHFlag(IndexComp) = .true.
                    !        else
                    !            nHFlag(IndexComp) = .false.
                    !        endif

                    !        ! Debug:
                    !        ! print*,"nHFlag(IndexComp) = ", nHFlag(IndexComp)


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< molfit parameters of current component


                !< get source size for "CC" (circle centered layer) and source radius for "CO" (circled layer with offset)
                elseif (ParamName == "source_size" .or. ParamName == "source_radius") then
                    myXCLASSParameter(14, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        FittedCompShapeParameterFlag = .true.
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 14/)
                    endif


                !< get source_center_x for "CO" (circled layer with offset)
                elseif (ParamName == "source_center_x") then
                    myXCLASSParameter(15, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        FittedCompShapeParameterFlag = .true.
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 15/)
                    endif


                !< get source_center_y for "CO" (circled layer with offset)
                elseif (ParamName == "source_center_y") then
                    myXCLASSParameter(16, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        FittedCompShapeParameterFlag = .true.
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 16/)
                    endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< read in parameters for normal molecule and RRLs
                !<
                !< for normal molecule:
                !<
                !< myXCLASSParameter(1, IndexComp) = T_rot
                !< myXCLASSParameter(2, IndexComp) = N_tot
                !< myXCLASSParameter(3, IndexComp) = -- not used --
                !< myXCLASSParameter(4, IndexComp) = V_width_Gauss
                !< myXCLASSParameter(5, IndexComp) = -
                !< myXCLASSParameter(6, IndexComp) = V_Off
                !<
                !<
                !< for RRLs:
                !<
                !< myXCLASSParameter(1, IndexComp) = T_e
                !< myXCLASSParameter(2, IndexComp) = EM_RRL
                !< myXCLASSParameter(3, IndexComp) = -
                !< myXCLASSParameter(4, IndexComp) = V_width_Gauss
                !< myXCLASSParameter(5, IndexComp) = -
                !< myXCLASSParameter(6, IndexComp) = V_Off
                !<


                !< get T_rot and T_e
                elseif (ParamName == "T_rot" .or. ParamName == "T_e") then
                    myXCLASSParameter(1, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 1/)
                    endif


                    !< check for zero temperature
                    if (myXCLASSParameter(1, IndexComp) == 0.d0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine ReadInmolfitParameter:")')
                            write(ErrChannel, '(4x, "The rotation temperature for component ", I4, " of molecule ", A, " ")') &
                                                    CurrComp, trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "is identical to zero!")')
                            write(ErrChannel, '(4x, "Set rotation temperature for this component to 1.0 &
                                                    &to avoid division by zero!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(1, IndexComp) = 1.d0
                    endif


                !< get N_tot and EM_RRL
                elseif (ParamName == "N_tot" .or. ParamName == "EM_RRL") then
                    myXCLASSParameter(2, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 2/)
                    endif


                    !< check for overflow
                    if (myXCLASSParameter(2, IndexComp) > 300.d0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                            write(ErrChannel, '(4x, "The log10 value of column density for component ", I4, " of molecule ", &
                                                    &A, " ")') CurrComp, trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "causes an overflow (to large number)!")')
                            write(ErrChannel, '(4x, "Set log10 value for this column density to 300!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(2, IndexComp) = 300.d0
                    endif


                    !< convert log10 value back to linear value
                    myXCLASSParameter(2, IndexComp) = 10.d0**myXCLASSParameter(2, IndexComp)

                    ! Debug:
                    ! print*,'i, paramset(1, i) = ', i, paramset(1, i)
                    ! print*,'IndexComp, myXCLASSParameter(2, IndexComp) = ', IndexComp, myXCLASSParameter(2, IndexComp)


                !< get N_e
                elseif (ParamName == "N_e") then
                    myXCLASSParameter(3, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 3/)
                    endif


                    !< check for overflow
                    if (myXCLASSParameter(3, IndexComp) > 300.d0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                            write(ErrChannel, '(4x, "The log10 value of column density for component ", I4, " of molecule ", &
                                                    &A, " ")') CurrComp, trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "causes an overflow (to large number)!")')
                            write(ErrChannel, '(4x, "Set log10 value for this column density to 300!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(3, IndexComp) = 300.d0
                    endif


                    !< convert log10 value back to linear value
                    myXCLASSParameter(3, IndexComp) = 10.d0**myXCLASSParameter(3, IndexComp)

                    ! Debug:
                    ! print*,'i, paramset(1, i) = ', i, paramset(1, i)
                    ! print*,'IndexComp, myXCLASSParameter(3, IndexComp) = ', IndexComp, myXCLASSParameter(3, IndexComp)


                !< get V_width_Gauss
                elseif (ParamName == "V_width_Gauss") then
                    myXCLASSParameter(4, IndexComp) = dmax1(paramset(1, i), 1.d-5)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 4/)
                    endif
                    vWidthLimits(IndexComp, 1:2) = (/dmin1(paramset(3, i), paramset(4, i)), dmax1(paramset(3, i), paramset(4, i))/)
                    !                                                          lower limit,                           upper limit

                !< get V_off
                elseif (ParamName == "V_off") then
                    myXCLASSParameter(6, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 6/)
                    endif
                    vOffLimits(IndexComp, :) = (/paramset(2, i), dmin1(paramset(3, i), paramset(4, i)), &
                                                 dmax1(paramset(3, i), paramset(4, i))/)
                    !                                  fit-flag,                           lower limit,                            upper limit


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< read in parameters for dust continuum
                !< myXCLASSParameter(1, IndexComp) = T_cont_dust
                !< myXCLASSParameter(2, IndexComp) = nHcolumn_cont_dust
                !< myXCLASSParameter(3, IndexComp) = kappa_cont_dust
                !< myXCLASSParameter(4, IndexComp) = beta_cont_dust


                !< get T_cont_dust
                elseif (ParamName == "T_cont_dust") then
                    myXCLASSParameter(1, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 1/)
                    endif


                    !< check for zero temperature
                    if (myXCLASSParameter(1, IndexComp) == 0.d0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                            write(ErrChannel, '(4x, "The rotation temperature for component ", I4, " of molecule ", A, " ")') &
                                                    CurrComp, trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "is identical to zero!")')
                            write(ErrChannel, '(4x, "Set rotation temperature for this component to 1.0 &
                                                    &to avoid division by zero!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(1, IndexComp) = 1.d0
                    endif


                !< get nHcolumn_cont_dust
                elseif (ParamName == "nHcolumn_cont_dust") then
                    myXCLASSParameter(2, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 2/)
                    endif


                    !< check for overflow
                    if (myXCLASSParameter(2, IndexComp) > 300.d0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                            write(ErrChannel, '(4x, "The log10 value of column density for component ", I4, " of molecule ", &
                                                    &A, " ")') CurrComp, trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "causes an overflow (to large number)!")')
                            write(ErrChannel, '(4x, "Set log10 value for this column density to 300!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(2, IndexComp) = 300.d0
                    endif


                    !< convert log10 value back to linear value
                    myXCLASSParameter(2, IndexComp) = 10.d0**myXCLASSParameter(2, IndexComp)

                    ! Debug:
                    ! print*,'i, paramset(1, i) = ', i, paramset(1, i)
                    ! print*,'IndexComp, myXCLASSParameter(2, IndexComp) = ', IndexComp, myXCLASSParameter(2, IndexComp)


                !< get kappa_cont_dust
                elseif (ParamName == "kappa_cont_dust") then
                    myXCLASSParameter(3, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 3/)
                    endif


                !< get beta_cont_dust
                elseif (ParamName == "beta_cont_dust") then
                    myXCLASSParameter(4, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 4/)
                    endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< read in parameters for phenomenological continuum description
                !< myXCLASSParameter(1, IndexComp) = ContFuncID_phen
                !< myXCLASSParameter(2, IndexComp) = ContFuncID_param_1
                !< myXCLASSParameter(3, IndexComp) = ContFuncID_param_2
                !< myXCLASSParameter(4, IndexComp) = ContFuncID_param_3
                !< myXCLASSParameter(5, IndexComp) = ContFuncID_param_4
                !< myXCLASSParameter(6, IndexComp) = ContFuncID_param_5


                !< get ContFuncID_phen
                elseif (ParamName == "ContFuncID_phen") then
                    myXCLASSParameter(1, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 1/)
                    endif


                !< get ContFuncID_param_1
                elseif (ParamName == "ContFuncID_param_1") then
                    myXCLASSParameter(2, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 2/)
                    endif


                !< get ContFuncID_param_2
                elseif (ParamName == "ContFuncID_param_2") then
                    myXCLASSParameter(3, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 3/)
                    endif


                !< get ContFuncID_param_3
                elseif (ParamName == "ContFuncID_param_3") then
                    myXCLASSParameter(4, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 4/)
                    endif


                !< get ContFuncID_param_4
                elseif (ParamName == "ContFuncID_param_4") then
                    myXCLASSParameter(5, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 5/)
                    endif


                !< get ContFuncID_param_5
                elseif (ParamName == "ContFuncID_param_5") then
                    myXCLASSParameter(6, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 6/)
                    endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< read in inline parameters for dust continuum
                !< myXCLASSParameter(7, IndexComp) = T_dOff
                !< myXCLASSParameter(8, IndexComp) = T_dSlope
                !< myXCLASSParameter(9, IndexComp) = nHcolumn
                !< myXCLASSParameter(10, IndexComp) = kappa
                !< myXCLASSParameter(11, IndexComp) = beta


                !< get T_dOff
                elseif (ParamName == "T_dOff") then
                    if (tdFlag(IndexComp)) then
                        myXCLASSParameter(7, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 7/)
                        endif

                        ! Debug:
                        ! print*," "
                        ! print*,"IndexComp, myXCLASSParameter(7, IndexComp) = ", IndexComp, myXCLASSParameter(7, IndexComp)
                    endif


                !< get T_dSlope
                elseif (ParamName == "T_dSlope") then
                    if (tdFlag(IndexComp)) then
                        myXCLASSParameter(8, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 8/)
                        endif

                        ! Debug:
                        ! print*," "
                        ! print*,"IndexComp, myXCLASSParameter(8, IndexComp) = ", IndexComp, myXCLASSParameter(8, IndexComp)
                    endif


                !< get nHcolumn
                elseif (ParamName == "nHcolumn") then
                    nHFlag(IndexComp) = .true.
                    myXCLASSParameter(9, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 9/)
                    endif


                    !< check for overflow
                    if (myXCLASSParameter(9, IndexComp) > 300.d0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                            write(ErrChannel, '(4x, "The log10 value of hydrogen column density for component ", I4, &
                                                    &" of molecule", A, " ")') CurrComp, trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "causes an overflow (to large number)!")')
                            write(ErrChannel, '(4x, "Set log10 value for this column density to 300!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(9, IndexComp) = 300.d0
                    endif


                    !< convert log10 value back to linear value
                    myXCLASSParameter(9, IndexComp) = 10.d0**myXCLASSParameter(9, IndexComp)


                !< get kappa
                elseif (ParamName == "kappa") then
                    if (nHFlag(IndexComp)) then
                        myXCLASSParameter(10, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 10/)
                        endif
                    endif


                !< get beta
                elseif (ParamName == "beta") then
                    if (nHFlag(IndexComp)) then
                        myXCLASSParameter(11, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 11/)
                        endif
                    endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< read in general parameters for stacking
                !< myXCLASSParameter(12, IndexComp) = LayerDistance
                !< myXCLASSParameter(13, IndexComp) = CFFlag


                !< get LayerDistance
                elseif (ParamName == "LayerDistance") then
                    myXCLASSParameter(12, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 12/)
                    endif
                    UseDistance = .true.


                !< get CFFlag
                elseif (ParamName == "CFFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "a" .or. trim(adjustl(FitParameterValue(i))) == "f") then
                        myXCLASSParameter(13, IndexComp) = 2.d0
                    else
                        myXCLASSParameter(13, IndexComp) = 1.d0
                    endif
                    UseCFFLag = .true.


                !< get keyword
                elseif (ParamName == "keyword") then


                    !< convert string to lower case
                    FitParameterValue(i) = trim(adjustl(FitParameterValue(i)))
                    k = min0(len_trim(FitParameterValue(i)), 10)
                    KeyWordString = FitParameterValue(i)(1:k)
                    call ConverStringToLowerCase(KeyWordString)


                    !< check, if optical thin or optical thick is assumed for Non-LTE RRL calculation
                    if (KindOfMolecule(IndexComp) == 2) then
                        k = index(KeyWordString, "thin")
                        if (k > 0) then
                            ThinFlag(IndexComp) = .true.
                        endif
                    endif
                endif
            end Do                                                                          !< i: loop over all parameter

            ! Debug:
            ! print*,"myXCLASSParameter(1, :) = ", myXCLASSParameter(1, :)
            ! print*,"myXCLASSParameter(2, :) = ", myXCLASSParameter(2, :)
            ! print*,"myXCLASSParameter(3, :) = ", myXCLASSParameter(3, :)
            ! print*,"myXCLASSParameter(4, :) = ", myXCLASSParameter(4, :)
            ! print*,"myXCLASSParameter(5, :) = ", myXCLASSParameter(5, :)
            ! print*,"myXCLASSParameter(6, :) = ", myXCLASSParameter(6, :)
            ! print*,"myXCLASSParameter(7, :) = ", myXCLASSParameter(7, :)
            ! print*,"myXCLASSParameter(8, :) = ", myXCLASSParameter(8, :)
            ! print*,"myXCLASSParameter(9, :) = ", myXCLASSParameter(9, :)
            ! print*,"myXCLASSParameter(10, :) = ", myXCLASSParameter(10, :)
            ! print*,"myXCLASSParameter(11, :) = ", myXCLASSParameter(11, :)
            ! print*,"myXCLASSParameter(12, :) = ", myXCLASSParameter(12, :)
            ! print*,"myXCLASSParameter(13, :) = ", myXCLASSParameter(13, :)
            ! print*,"myXCLASSParameter(14, :) = ", myXCLASSParameter(14, :)
            ! print*,"myXCLASSParameter(15, :) = ", myXCLASSParameter(15, :)
            ! print*,"myXCLASSParameter(16, :) = ", myXCLASSParameter(16, :)
            ! if (IsoFlag) then
            !     Do i = 1, TotalNumberOfMolecules
            !         print*,"i, IsoRatioConversionTable(i, :) = ", i, IsoRatioConversionTable(i, :)
            !     end Do
            ! endif


            !< print what you do
            if (printflag .and. trim(adjustl(ParallelizationMethod)) /= "NoMAGIX") then
                print '("done!")'
            endif


            !=============================================================================================================================================
            !< define distance array, describing the order of the different components


            !< check, if CFFlag and distance are defined in molfit file
            if (UseDistance .and. UseCFFLag) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                    write(ErrChannel, '(4x, "The given molfit file contains components where a cf-flag is defined")')
                    write(ErrChannel, '(4x, "and components where a distance is defined!")')
                    write(ErrChannel, '(4x, "Both definitions can not be used simultaneously in the same molfit file!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(4x, "Please correct the molfit file and restart the program!")')
                    write(ErrChannel, '(" ")')
                end Do
                stop 'Program aborted! Please restart the program!'


            !< check, if neither CFFlag nor distance is defined in molfit file
            elseif ((.not. UseDistance) .and. (.not. UseCFFLag)) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                    write(ErrChannel, '(4x, "The given molfit file contains neither a definition of a cf-flag")')
                    write(ErrChannel, '(4x, "nor a definition of a distance!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(4x, "Please correct the molfit file and restart the program!")')
                    write(ErrChannel, '(" ")')
                end Do
                stop 'Program aborted! Please restart the program!'


            !< check, if only distances are defined in molfit file
            elseif (UseDistance .and. (.not. UseCFFLag)) then
                SortedDistances(:) = myXCLASSParameter(12, :)


                !< sort distances in descending order
                SortedDistances = (-1.d0) * SortedDistances                                 !< we need the distances in descending order
                call sort(TotalNumberComponents, SortedDistances)
                SortedDistances = dabs(SortedDistances)                                     !< remove negative signs


            !< check, if only cf-flags are defined in molfit file (no sorting is needed)
            elseif ((.not. UseDistance) .and. UseCFFLag) then
                Do j = 1, TotalNumberComponents
                    if (myXCLASSParameter(13, j) == 1.d0) then
                        SortedDistances(j) = dble(TotalNumberComponents)
                    else
                        SortedDistances(j) = dble(TotalNumberComponents - j + 1)
                    endif
                end Do
            endif

            ! Debug:
            ! print*,"UseDistance = ", UseDistance
            ! print*,"UseCFFLag = ", UseCFFLag
            ! print*,"TotalNumberComponents = ", TotalNumberComponents
            ! print*,"SortedDistances = ", SortedDistances


            !< determine the number of unequal distances
            NumberDistances = 0
            HelpValue = (-1.d0)
            Do j = 1, TotalNumberComponents
                l = 0
                Do k = 1, j - 1
                    if (dabs(SortedDistances(j) - SortedDistances(k)) < 1.d-9) then
                        l = l + 1
                    endif
                end Do
                if (l == 0) then
                    NumberDistances = NumberDistances + 1
                endif
            end Do

            ! Debug:
            ! print*,"NumberDistances = ", NumberDistances


            !< determine total number of components and allocate memory
            if (allocated(PureDistances)) then
                deallocate(PureDistances, DistanceOrderingArray, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x, "Can not deallocate variables PureDistances, DistanceOrderingArray!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    close(logchannel)
                    stop 'Program aborted! Please restart the program!'
                endif
            endif
            i = max0(1, NumberDistances)
            allocate(PureDistances(TotalNumberComponents), DistanceOrderingArray(TotalNumberComponents, TotalNumberComponents), &
                     stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                    write(ErrChannel, '(5x, "- PureDistances(TotalNumberComponents)")')
                    write(ErrChannel, '(5x, "- DistanceOrderingArray(TotalNumberComponents, TotalNumberComponents)")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameter:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- TotalNumberComponents = ", I8)') TotalNumberComponents
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                close(logchannel)
                stop 'Program aborted! Please restart the program!'
            endif
            PureDistances = 0.d0
            DistanceOrderingArray = 0


            !< determine the unequal (pure) distances
            SortedDistances = -SortedDistances
            min_val = minval(SortedDistances)
            max_val = maxval(SortedDistances)
            i = 1
            PureDistances(i) = dabs(min_val)
            Do While (min_val < max_val)
                i = i + 1
                min_val = minval(SortedDistances, mask = SortedDistances > min_val)
                PureDistances(i) = dabs(min_val)
            end Do

            ! Debug:
            ! print*,"i = ", i
            ! print*,"NumberDistances = ", NumberDistances
            ! print*,"PureDistances = ", PureDistances


            !< determine the translation array between component index and reduced distance parameter
            Do i = 1, NumberDistances                                                       !< loop over all distances
                l = 0
                Do j = 1, TotalNumberComponents                                             !< loop over all comp. which correspond to the current dist.
                    if (UseDistance .and. (.not. UseCFFLag)) then                           !< distance is defined in molfit file
                        HelpValue = myXCLASSParameter(12, j)
                    else                                                                    !< cf flag is defined in molfit file
                        if (myXCLASSParameter(13, j) == 1.d0) then
                            HelpValue = dble(TotalNumberComponents)
                        else
                            HelpValue = dble(TotalNumberComponents - j + 1)
                        endif
                    endif
                    if (dabs(PureDistances(i) - HelpValue) <= 1.d-9) then
                        l = l + 1
                        DistanceOrderingArray(i, l) = j
                    endif
                end Do                                                                      !< j: loop over all comp.

                ! Debug:
                ! print*,"i, PureDistances(i), DistanceOrderingArray(i, :) = ", i, PureDistances(i), DistanceOrderingArray(i, :)
            end Do                                                                          !< i: loop over all distances


            !=============================================================================================================================================
            !< define distance array, describing the order of the different components
            if (UseEmAbsFuncFlag) then
                call ImportEmsAbsFile
            endif


            !=============================================================================================================================================
            !< check, if sub-beam description has to be taken into account, i.e. is there a component with a smaller source size which is not closest to
            !< to the continuum


            if (sum(dabs(myXCLASSParameter(15, :))) > tiny(1.d0) .or. sum(dabs(myXCLASSParameter(16, :))) > tiny(1.d0)) then
                                                                                            !< set sub-beam flag, if soure position of at
                SubBeamDescriptionFlag = .true.                                             !< least one comp. is not zero
            elseif (FittedCompShapeParameterFlag .and. TotalNumberComponents > 1) then      !< set sub-beam flag, if soure size or position are fitted
                SubBeamDescriptionFlag = .true.                                             !< for at least one comp.
            elseif (maxval(GeometryFlag(:)) > 2) then
                SubBeamDescriptionFlag = .true.
            else


                !< determine total number of components and allocate memory
                if (allocated(BeamFillingFactor)) then
                    deallocate(BeamFillingFactor, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                            write(ErrChannel, '(3x, "Can not deallocate variable BeamFillingFactor!")')
                            write(ErrChannel, '(" ")')
                        end Do
                        close(logchannel)
                        stop 'Program aborted! Please restart the program!'
                    endif
                endif
                allocate(BeamFillingFactor(TotalNumberComponents), stat = allocstatus)
                if (allocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x, "Can not allocate variable BeamFillingFactor(TotalNumberComponents)!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Used parameter:")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(5x, "- TotalNumberComponents = ", I8)') TotalNumberComponents
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(" ")')
                    end Do
                    close(logchannel)
                    stop 'Program aborted! Please restart the program!'
                endif
                BeamFillingFactor = 0.d0


                !< compute min. beam filling factor for each component
                Do j = 1, TotalNumberComponents                                             !< loop over all compomponents
                    if (dabs(myXCLASSParameter(14, j)) > tiny(1.d0)) then                   !< continue only, if soure size / radius is larger than zero
                        min_val = 1.d99
                        Do l = 1, TotalNumberOfFrequencyRanges                              !< loop over all frequency ranges


                            !< determine beam size for current frequency range
                            if (InterFlag(l)) then
                                beam_size = TelescopeSize(l)
                            else
                                i1 = DataPointIndexFreqRange(l, 1)                          !< get index of first freq. point in 'ObservationalDataList'
                                f1 = ObservationalDataList(i1, 1)
                                i2 = DataPointIndexFreqRange(l, 2)                          !< get index of last freq. point in 'ObservationalDataList'
                                f2 = ObservationalDataList(i2, 1)
                                beam_size = 1.22d-3 * ckms / (dmin1(f1, f2) * TelescopeSize(l)) * (180.d0 / pi) * 3600.d0
                            endif


                            !< compute beam-filling factor
                            HelpValue = myXCLASSParameter(14, k)**2 / (beam_size**2 + myXCLASSParameter(14, k)**2)
                            min_val = dmin1(min_val, HelpValue)
                        end Do                                                              !< l: loop over all frequency ranges
                        BeamFillingFactor(j) = min_val
                    else
                        BeamFillingFactor(j) = 1.d0                                         !< a component without source size is beam filling
                    endif
                end Do                                                                      !< j: loop over all compomponents


                !< analyze beam filling factors
                if (LocalOverlapFlag) then                                                  !< special handling for local overlap
                    val = 0.d0
                    Do i = 1, NumberDistances                                               !< loop over all pure distances
                        SameSizeFlag = .true.
                        val2 = 0.d0
                        Do j = 1, TotalNumberComponents                                     !< loop over all comp. which correspond to the current dist.
                            if (DistanceOrderingArray(i, j) == 0) then                      !< if all components are analyzed exit loop
                                exit
                            else
                                !< check, if all components at current distance have the same source size
                                k = DistanceOrderingArray(i, j)                             !< get component index
                                if (dabs(myXCLASSParameter(14, k)) > tiny(1.d0) .and. SameSizeFlag) then
                                    if (dabs(val2 - myXCLASSParameter(14, k)) < tiny(1.d0) .or. dabs(val2) < tiny(1.d0)) then
                                        val2 = myXCLASSParameter(14, k)
                                    else
                                        SameSizeFlag = .false.
                                        exit
                                    endif
                                endif
                            endif
                        end Do                                                              !< j: loop over all components
                        if (.not. SameSizeFlag) then
                            SubBeamDescriptionFlag = .true.
                            exit
                        elseif (val2 < val) then
                            SubBeamDescriptionFlag = .true.
                            exit
                        else
                            val = val2
                        endif
                    end Do                                                                  !< i: loop over all pure distances
                else
                    Do i = 2, NumberDistances                                               !< loop over all pure distances
                        Do j = 1, TotalNumberComponents                                     !< loop over all comp. which correspond to the current dist.
                            if (DistanceOrderingArray(i, j) == 0) then                      !< if all components are analyzed exit loop
                                exit
                            else
                                k = DistanceOrderingArray(i, j)                             !< get component index
                                if (BeamFillingFactor(k) < 0.99d0 .and. TotalNumberComponents > 1) then
                                    SubBeamDescriptionFlag = .true.
                                    exit
                                endif
                            endif
                        end Do                                                              !< j: loop over all components
                        if (SubBeamDescriptionFlag) exit
                    end Do                                                                  !< i: loop over all pure distances
                endif
            endif
            if (NoSubBeamFlag) then
                SubBeamDescriptionFlag = .false.
            endif
            if (SubBeamDescriptionFlag) then                                                !< sub-beam description has to be used
                if (NumberModelPixelXX <= 0) then
                    NumberModelPixelXX = 100
                endif
                if (NumberModelPixelYY <= 0) then
                    NumberModelPixelYY = 100
                endif
                if (allocated(LayerMap)) then
                    deallocate(LayerMap, ConfigList, ConfigIndex, GausssianBeamMap, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                            write(ErrChannel, '(3x, "Can not deallocate variables LayerMap etc.!")')
                            write(ErrChannel, '(" ")')
                        end Do
                        close(logchannel)
                        stop 'Program aborted! Please restart the program!'
                    endif
                endif
                allocate(LayerMap(NumberModelPixelXX, NumberModelPixelYY, TotalNumberComponents), &
                         ConfigList(NumberModelPixelXX * NumberModelPixelYY, TotalNumberComponents), &
                         ConfigIndex(NumberModelPixelXX, NumberModelPixelYY), &
                         GausssianBeamMap(NumberModelPixelXX, NumberModelPixelYY), stat = allocstatus)
                if (allocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                        write(ErrChannel, '(5x, "- LayerMap(NumberModelPixelXX, NumberModelPixelYY, TotalNumberComponents)")')
                        write(ErrChannel, '(5x, "- ConfigList(NumberModelPixelXX * NumberModelPixelYY, TotalNumberComponents)")')
                        write(ErrChannel, '(5x, "- ConfigIndex(NumberModelPixelXX, NumberModelPixelYY)")')
                        write(ErrChannel, '(5x, "- GausssianBeamMap(NumberModelPixelXX, NumberModelPixelYY)")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Used parameters:")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(5x, "- NumberModelPixelXX = ", I8)') NumberModelPixelXX
                        write(ErrChannel, '(5x, "- NumberModelPixelYY = ", I8)') NumberModelPixelYY
                        write(ErrChannel, '(5x, "- TotalNumberComponents = ", I8)') TotalNumberComponents
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(" ")')
                    end Do
                    close(logchannel)
                    stop 'Program aborted! Please restart the program!'
                endif
                LayerMap = .false.
                ConfigList = .false.
                ConfigIndex = 0
                GausssianBeamMap = 0.d0
            endif


            !< print what you do
            if (printflag) then
                print '(" ")'
                print '(" ")'
                if (.not. LogFlag) then
                    print '("              ", $)'
                endif
                if (SubBeamDescriptionFlag) then
                    print '("Using sub-beam description!")'
                else
                    print '("Sub-beam description is NOT used!")'
                endif
                print '(" ")'
            endif
            if (.not. LogFlag) then
                write(logchannel,'(" ")')
                write(logchannel,'(" ")')
                if (SubBeamDescriptionFlag) then
                    write(logchannel,'(11x, "Using sub-beam description!")')
                else
                    write(logchannel,'(11x, "Sub-beam description is NOT used!")')
                endif
                write(logchannel,'(" ")')
            endif

            ! Debug:
            ! print*,"SubBeamDescriptionFlag = ", SubBeamDescriptionFlag


            !< print if local-overlap is taken into account or not
            if (printflag) then
                print '(" ")'
                if (.not. LogFlag) then
                    print '("              ", $)'
                endif
                if (LocalOverlapFlag) then
                    print '("Take local-overlap into account!")'
                else
                    print '("Do NOT take local-overlap into account!")'
                endif
                print '(" ")'
                print '(" ")'
            endif
            if (.not. LogFlag) then
                write(logchannel,'(" ")')
                if (LocalOverlapFlag) then
                    write(logchannel,'(11x, "Take local-overlap into account!")')
                else
                    write(logchannel,'(11x, "Do NOT take local-overlap into account!")')
                endif
                write(logchannel,'(" ")')
                write(logchannel,'(" ")')
            endif


            !< we're done
            return
        end subroutine GetmyXCLASSParameter


        !>************************************************************************************************************************************************
        !> subroutine: GetPartitionFunction
        !>
        !> get partition function from SQLite3 database
        !>
        !> input variables:     none
        !>
        !> output variables:    ok:             status variable
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2014-07-29
        !>
        subroutine GetPartitionFunction(ok)

            implicit none
            integer :: ok                                                                   !< status variable
            integer :: number_entries                                                       !< number of entries for the PartitionFunction table
            integer :: countlen                                                             !< total length of query
            integer :: i, j, k, l                                                           !< loop/working variables
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            real*8, allocatable, dimension(:, :) :: CopylgQ                                 !< copy of lgQ variable
            character(len=40) :: lower_string                                               !< help variable for string check
            character(len=16384) :: queryString1                                            !< query string (first part)
            character(len=16384) :: queryString2                                            !< query string (second part)
            character(len=4096) :: dbNameCopy                                               !< name of database
            character(len=40) :: ColumnNameForNamePartFuncCopy                              !< copy of name of column including the names of molecules
            character(len=40), dimension(TotalNumberOfMolecules) :: CopyMoleculeNames       !< copy of names of molecules


            !< print what you do
            if (printflag) then
                print '(14x, "Reading partition functions for all molecules from SQLite3 database .. ", $)'
            endif

            ! Debug:
            ! print*," "
            ! print*,"dbName = ", trim(adjustl(dbName))


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define query string


            !< first part
            queryString1 = "SELECT count(*) FROM " // trim(adjustl(NameOfPartFuncTable)) // char(0)
            queryString2 = " " // char(0)


            !< create query string
            ok = 0                                                                          !< reset status variable
            number_entries = 0                                                              !< reset result variable
            j = TotalNumberOfMolecules
            ColumnNameForNamePartFuncCopy = trim(adjustl(ColumnNameForNamePartFunc)) // char(0) !< get name of column for molecule names
            countlen = (len_trim(queryString1) - 1)                                         !< length of string queryString1
            countlen = countlen + 8                                                         !< length plus ' WHERE ('
            CopyMoleculeNames = ""
            Do k = 1, TotalNumberOfMolecules                                                !< loop over all molecules
                CopyMoleculeNames(k) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                countlen = countlen + len_trim(CopyMoleculeNames(k)) - 1                    !< length plus length of name (incl. '""')
                countlen = countlen + len_trim(ColumnNameForNamePartFuncCopy)               !< length plus 'PF_Name'
                countlen = countlen + 7                                                     !< length plus ' or ' and ' = '
            end Do
            countlen = countlen + 2                                                         !< length plus ') '
            countlen = countlen + (len_trim(queryString2) - 1)                              !< length plus length of string queryString2
            countlen = countlen + len_trim(ColumnNameForNamePartFuncCopy) + 2               !< length plus 'PF_Name'
            dbNameCopy = trim(adjustl(dbName)) // char(0)

            ! Debug:
            ! print*,' '
            ! print*,'queryString1 = >', trim(adjustl(queryString1)), '<'
            ! print*,'queryString2 = >', trim(adjustl(queryString2)), '<'
            ! print*,"j = ", j
            ! print*,'countlen = ', countlen


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get number of entries for the partition function table
            call numentr_sqlite3(ok, number_entries, %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), &
                                 %val(j), trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNamePartFuncCopy)), &
                                 CopyMoleculeNames)
            NumberMoleculePartFunc = max(number_entries, TotalNumberOfMolecules)

            ! Debug:
            ! print*,'NumberMoleculePartFunc = ', NumberMoleculePartFunc
            ! print*,'ok = ', ok


            !< allocate variables for partition function
            if (NumberMoleculePartFunc > 0 .and. ok == 0) then
                if (allocated(MolNamePartFunc)) then
                    deallocate(MolNamePartFunc, lgQ, CopylgQ, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x, "Error in subroutine GetPartitionFunction!")')
                            write(ErrChannel, '(3x, "Can not deallocate variables MolNamePartFunc and lgQ!")')
                            write(ErrChannel, '(" ")')
                        end Do
                        close(logchannel)
                        stop 'Program aborted! Please restart the program!'
                    endif
                endif
                allocate(MolNamePartFunc(NumberMoleculePartFunc), lgQ(NumberOfTemperatures, NumberMoleculePartFunc), &
                         CopylgQ(NumberOfTemperatures, NumberMoleculePartFunc), stat = allocstatus)
                if (allocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetPartitionFunction!")')
                        write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                        write(ErrChannel, '(5x, "- MolNamePartFunc(NumberMoleculePartFunc)")')
                        write(ErrChannel, '(5x, "- lgQ(NumberOfTemperatures, NumberMoleculePartFunc)")')
                        write(ErrChannel, '(5x, "- CopylgQ(NumberOfTemperatures, NumberMoleculePartFunc)")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Used parameters:")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(5x, "- NumberMoleculePartFunc = ", I8)') NumberMoleculePartFunc
                        write(ErrChannel, '(5x, "- NumberOfTemperatures = ", I8)') NumberOfTemperatures
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(" ")')
                    end Do
                    close(logchannel)
                    stop 'Program aborted! Please restart the program!'
                endif
                MolNamePartFunc = ""
                lgQ = 0.d0
                CopylgQ = 0.d0


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< construct query string
                ok = 0                                                                      !< reset status variable
                i = NumberMoleculePartFunc
                queryString1 = "SELECT " // trim(adjustl(ColumnNameForNamePartFunc))
                Do j = 1, NumberOfTemperatures
                    queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNamesPartFunc(j)))
                end Do
                queryString1 = trim(adjustl(queryString1)) // " FROM " // trim(adjustl(NameOfPartFuncTable)) // char(0)

                ! Debug:
                ! print*,"queryString1 = ", queryString1


                !< get information from SQLite3 database
                ok = 0                                                                      !< reset status variable
                l = NumberOfTemperatures
                i = NumberMoleculePartFunc
                j = TotalNumberOfMolecules
                countlen = (len_trim(queryString1) - 1)
                CopyMoleculeNames = ""
                Do k = 1, TotalNumberOfMolecules
                    CopyMoleculeNames(k) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                    countlen = countlen + len_trim(CopyMoleculeNames(k)) + 14

                    ! Debug:
                    ! print*,'k, CopyMoleculeNames(k) = ', k, trim(adjustl(CopyMoleculeNames(k)))
                end Do
                countlen = countlen + (len_trim(queryString2) - 1) + 17


                !< bug fix: prevent segmentation fault, if database contains double entries
                if (NumberMoleculePartFunc > TotalNumberOfMolecules) then
                    Do k = TotalNumberOfMolecules, NumberMoleculePartFunc
                        countlen = countlen + 40 + 14
                    end Do
                endif


                !< get information from SQLite3 database
                ! ok,   status
                ! i,    number of entries in the database
                ! l,    number of temperatures
                ! j,    number of molecules in the query string
                dbNameCopy = trim(adjustl(dbName)) // char(0)
                ColumnNameForNamePartFuncCopy = trim(adjustl(ColumnNameForNamePartFunc)) // char(0)
                call ReadPartFunc(ok, %val(l), %val(i), %val(j), %val(countlen), trim(adjustl(queryString1)), &
                                  trim(adjustl(queryString2)), trim(adjustl(dbNameCopy)), &
                                  trim(adjustl(ColumnNameForNamePartFuncCopy)), CopyMoleculeNames, lgQ, &
                                  MolNamePartFunc)

                ! Debug:
                ! print*,'NumberMoleculePartFunc = ', NumberMoleculePartFunc
                ! print*,'lgQ(1:NumberOfTemperatures, 1) = ', lgQ(1:NumberOfTemperatures, 1)
                ! print*,'lgQ(1:NumberOfTemperatures, NumberMoleculePartFunc) = ', lgQ(1:NumberOfTemperatures, NumberMoleculePartFunc)
                ! print*,'MolNamePartFunc(1) = ', MolNamePartFunc(1)
                ! print*,'MolNamePartFunc(NumberMoleculePartFunc) = ', MolNamePartFunc(NumberMoleculePartFunc)
                ! stop
            else
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine GetPartitionFunction!")')
                    write(ErrChannel, '(3x, "There are no entries in the SQLite3 database ", A, "!")') &
                                            trim(adjustl(NameOfPartFuncTable))
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please correct the name of the database and restart the program!")')
                end Do
                ok = 1
                return
            endif


            !< bring partition function in variable lgQ in the same order like MoleculeNames
            CopylgQ(:, :) = lgQ(:, :)
            lgQ = 0.d0
            Do i = 1, TotalNumberOfMolecules
                Do j = 1, NumberMoleculePartFunc

                    ! Debug:
                    ! print*,'----->i, trim(adjustl(MoleculeNames(i))) = ', i, trim(adjustl(MoleculeNames(i)))
                    ! print*,'>>>>>>j, trim(adjustl(MolNamePartFunc(j))) = ', j, trim(adjustl(MolNamePartFunc(j)))

                    if (trim(adjustl(MoleculeNames(i))) // char(0) == trim(adjustl(MolNamePartFunc(j)))) then
                        lgQ(:, i) = CopylgQ(:, j)


                        !< calculate log10 of partition function of the current molecule and check for entries which are equal to zero
                        if (count(lgQ(:, i) == 0) > 0) then


                            !< convert molecule name to lower case
                            lower_string = trim(adjustl(MoleculeNames(i)))
                            call ConverStringToLowerCase(lower_string)
                            lower_string = trim(adjustl(lower_string))


                            !< if molecule name is not a RRL, raise error
                            if (.not. lower_string(1:3) == "rrl") then
                                Do ErrChannelIndex = 1, 2
                                    ErrChannel = AllErrChannels(ErrChannelIndex)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(2x, "Error in subroutine qinterp!")')
                                    write(ErrChannel, '(4x, "The are some entries in the parittion function table ", &
                                                            &A, "for molecule, which are not defined or zero!")') &
                                                            &trim(adjustl(MoleculeNames(i)))
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(4x, "Number of entries which are identical to zero = ", I10)') &
                                                            count(lgQ(:, i) == 0)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(4x, "Set values at these positions to 1!")')
                                    write(ErrChannel, '(" ")')
                                end Do
                            endif


                            !< set values of partition function to 1
                            where (lgQ(:, i) == 0.d0)
                                lgQ(:, i) = 1.d0
                            end where
                        endif


                        !< determine log10 values
                        Do l = 1, NumberOfTemperatures
                            lgQ(l, i) = dlog10(lgQ(l, i))
                        end Do
                        exit
                    endif
                end Do

                ! Debug:
                ! print*,'i, trim(adjustl(MoleculeNames(i))), lgQ(:, i) = ', i, trim(adjustl(MoleculeNames(i))), lgQ(:, i)
            end Do


            !< print some status information to screen
            if (printflag) then
                print '("done!")'
                print '(" ")'
                print '(16x, "Number of entries = ", I10)', NumberMoleculePartFunc
                call strlen(MolNamePartFunc(1))
                if (NumberMoleculePartFunc > 1) then
                    print '(16x, "First Molecule in database: ", A)', trim(adjustl(MolNamePartFunc(1)))
                    call strlen(MolNamePartFunc(NumberMoleculePartFunc))
                    print '(16x, "Last Molecule in database:  ", A)', trim(adjustl(MolNamePartFunc(NumberMoleculePartFunc)))
                else
                    print '(16x, "Molecule in database: ", A)', trim(adjustl(MolNamePartFunc(1)))
                endif
                print '(" ")'
                print '(" ")'
            endif


            ! Debug:
            ! Do i = 1, TotalNumberOfMolecules
            !     print*,"i, MoleculeNames(i), MolNamePartFunc(i) = ", i, trim(adjustl(MoleculeNames(i))), "         ", trim(adjustl(MolNamePartFunc(i)))
            ! end Do


            !< free memory of CopylgQ variable
            if (allocated(CopylgQ)) then
                deallocate(CopylgQ, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetPartitionFunction!")')
                        write(ErrChannel, '(3x, "Can not deallocate variable CopylgQ!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    stop 'Program aborted! Please restart the program!'
                endif
            endif


            !< we're done
            return
        end subroutine GetPartitionFunction


        !>************************************************************************************************************************************************
        !> subroutine: GetTransitionParameters
        !>
        !> get information for transitions from SQLite3 database
        !>
        !> input variables:     none
        !>
        !> output variables:    ok:             status variable
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2014-07-30
        !>
        subroutine GetTransitionParameters(ok)

            implicit none
            integer :: ok                                                                   !< status variable
            integer :: number_entries                                                       !< number of entries for the PartitionFunction table
            integer :: countlen                                                             !< total length of query
            integer :: n                                                                    !< working variable: main quantum number
            integer :: i, j, k                                                              !< loop variables
            integer :: RangeIndex                                                           !< index for frequency ranges
            integer :: MolecularDataCounter                                                 !< overall counter for MolecularData variable
            integer :: LocalTransFreqIndex                                                  !< working variable: local index of transition frequency
            integer :: NumEntriesLocal                                                      !< working variable: number of data within current frequency
                                                                                            !< range and current molecule
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer, allocatable, dimension(:) :: UpperStateDegeneracy                      !< upper state degeneracy
            real*8 :: freqmin                                                               !< working variable lowest frequency of current freq. range
            real*8 :: freqmax                                                               !< working variable highest frequency of current freq. range
            real*8 :: NewFreqMin, NewFreqMax                                                !< working variables for min. and max. freq. of current range
            real*8 :: NewFreqMin1, NewFreqMax1                                              !< working variables for min. and max. freq. of current range
            real*8 :: LocalFreq                                                             !< working variable for Non-LTE transition frequencies
            real*8 :: vLocal                                                                !< working variable local velocity offset frequency
            real*8 :: vwidthLocal                                                           !< working variable local velocity width
            real*8 :: vMin, vMax                                                            !< min. and max. offset velocity of each molecule
            real*8 :: vwidthMax                                                             !< max. velocity width
            real*8, allocatable, dimension(:) :: lFreq                                      !< frequencies of transition
            real*8, allocatable, dimension(:) :: lFreqErr                                   !< uncertainty of frequencies of transition
            real*8, allocatable, dimension(:) :: lElt                                       !< energies of lower states
            real*8, allocatable, dimension(:) :: icatMol                                    !< intensity of the transitions
            real*8, allocatable, dimension(:) :: EinsteinA                                  !< Einstein A coefficients
            character(len=25) :: numberString1, numberString2, numberString3                !< working variables for number to string conversion
            character(len=40) :: LowerCurrNameMolecule                                      !< working variable: name of current molecule (lower case)
            character(len=40) :: ColumnNameForNameTransitionsCopy                           !< copy of name of column including the names of molecules
            character(len=99) :: helpString, helpString2                                    !< working variables for number to string conversion
            character(len=4096) :: dbNameCopy                                               !< name of database
            character(len=8192) :: queryString1                                             !< query string (first part)
            character(len=31384) :: queryString2                                            !< query string (second part)
            character(len=40), dimension(1) :: CopyMoleculeNames                            !< copy of names of molecules
            character(len=60), allocatable, dimension(:) :: MolNameLocal                    !< names of molecules in current freq. range
            character(len=60), allocatable, dimension(:) :: LocalLowerQNString              !< quantum number string of lower state
            character(len=60), allocatable, dimension(:) :: LocalUpperQNString              !< quantum number string of upper state
            logical :: LocalRRLFlag                                                         !< local RRL flag
            logical :: LocalLTEFlag                                                         !< indicates if a molecule has components in LTE
            logical :: LocalFlag                                                            !< working variable


            !< initialize output variable
            ok = 0


            !< print what you do
            if (printflag) then
                print '(14x, "Reading parameters for radiative transitions for all molecules from SQLite3 database .. ", $)'
            endif

            ! Debug:
            ! print*," "
            ! print*,"dbName = ", trim(adjustl(dbName))


            !< allocate memory for variable NumEntriesRanges
            if (allocated(NumEntriesRanges)) then
                deallocate(NumEntriesRanges, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetTransitionParameters!")')
                        write(ErrChannel, '(3x, "Can not deallocate variables NumEntriesRanges!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    ok = 1
                    return
                endif
            endif
            allocate(NumEntriesRanges(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine GetTransitionParameters!")')
                    write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                    write(ErrChannel, '(5x, "- NumEntriesRanges(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules)")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameters:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- TotalNumberOfFrequencyRanges = ", I8)') TotalNumberOfFrequencyRanges
                    write(ErrChannel, '(5x, "- TotalNumberOfMolecules = ", I8)') TotalNumberOfMolecules
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                ok = 1
                return
            endif
            NumEntriesRanges = 0


            !< determine total number of all transitions in all frequency ranges
            TotalNumberOfTransitions = 0
            Do RangeIndex = 1, TotalNumberOfFrequencyRanges                                 !< loop over all frequency ranges
                freqmin = StartFrequency(RangeIndex)
                freqmax = EndFrequency(RangeIndex)
                Do k = 1, TotalNumberOfMolecules                                            !< loop over all molecules
                    LocalLTEFlag = .false.                                                  !< initialize local LTE flag


                    !< determine min. and max. velocity offset
                    NewFreqMin = 0.d0
                    NewFreqMax = 0.d0
                    vMin = 1.d99
                    vMax = -1.d99
                    vwidthMax = -1.d99
                    Do j = 1, TotalNumberComponents                                         !< loop over components
                        if (KindOfMolecule(j) <= 2) then
                            i = CompMoleculeIndex(j)
                            LocalFlag = .false.
                            if (IsoFlag) then
                                if (IsoRatioConversionTable(i, k) /= 0.d0) then
                                    LocalFlag = .true.
                                endif
                            elseif (i == k) then
                                LocalFlag = .true.
                            endif
                            if (LocalFlag) then


                                !< check, if current component is computed in Non-LTE
                                if (LTEFlag(j) .or. KindOfMolecule(j) == 2) then
                                    LocalLTEFlag = .true.
                                endif

                                ! Debug:
                                ! print*," "
                                ! print*,"k, j, i = ", k, j, i
                                ! print*,"trim(adjustl(MoleculeNames(k))) = ", trim(adjustl(MoleculeNames(k)))
                                ! print*,"LocalLTEFlag = ", LocalLTEFlag
                                ! print*,"IsoRatioConversionTable(i, :) = ", IsoRatioConversionTable(i, :)


                                !< for myXCLASS function or not-fitted parameters consider the velocity offset and width parameters directly
                                !< (vOffLimits(j, 1) describes if current parameter is fitted (=1) or not (=0))
                                if (LogFlag .or. vOffLimits(j, 1) == 0) then


                                    !< determine max. velocity width for current molecule
                                    vwidthLocal = maxval(myXCLASSParameter(4:5, j))
                                    if (vwidthLocal > vwidthMax) then
                                        vwidthMax = vwidthLocal
                                    endif


                                    !< determine min. and max. velocity offsets for current molecule
                                    vLocal = myXCLASSParameter(6, j)
                                    if (vLocal < vMin) then
                                        vMin = vLocal
                                    endif
                                    if (vLocal > vMax) then
                                        vMax = vLocal
                                    endif


                                !< for myXCLASSFit function consider the lower and upper limits of velocity offset and width parameters
                                else


                                    !< determine max. velocity width for current molecule
                                    vwidthLocal = vWidthLimits(j, 2)                        !< get upper limit of velocity width
                                    if (vwidthLocal > vwidthMax) then
                                        vwidthMax = vwidthLocal
                                    endif


                                    !< determine min. and max. velocity offsets for current molecule
                                    vLocal = vOffLimits(j, 2)                               !< get lower limit of velocity offset
                                    if (vLocal < vMin) then
                                        vMin = vLocal
                                    endif
                                    vLocal = vOffLimits(j, 3)                               !< get upper limit of velocity offset
                                    if (vLocal > vMax) then
                                        vMax = vLocal
                                    endif
                                endif
                            endif
                        endif
                    end Do                                                                  !< j: loop over components
                    if (vMin == 1.d99) then
                        vMin = 0.d0
                    endif
                    if (vMax == -1.d99) then
                        vMax = 0.d0
                    endif
                    if (vwidthMax == -1.d99) then
                        vwidthMax = 0.d0
                    endif

                    ! Debug:
                    ! print*,"vMin = ", vMin
                    ! print*,"vMax = ", vMax


                    !< define new min. and max. frequencies for current range if user-defined vLSR is unequal zero
                    if (GlobalvLSR(RangeIndex) /= 0.d0) then
                        NewFreqMin = (freqmin * (1.d0 - GlobalvLSR(RangeIndex) / ckms))
                        NewFreqMax = (freqmax * (1.d0 - GlobalvLSR(RangeIndex) / ckms))
                    else
                        NewFreqMin = freqmin
                        NewFreqMax = freqmax
                    endif


                    !< shift the frequency range
                    NewFreqMin1 = (NewFreqMin * (1.d0 - dmax1(vMin, vMax) / ckms))
                    NewFreqMax1 = (NewFreqMax * (1.d0 - dmin1(vMin, vMax) / ckms))
                    if (RangeExpansionFlag) then
                        NewFreqMin = dmin1(NewFreqMin, (NewFreqMin1 * 0.9d0))
                        NewFreqMax = dmax1(NewFreqMax, (NewFreqMax1 * 1.1d0))
                    else
                        NewFreqMin = dmin1(NewFreqMin, NewFreqMin1)
                        NewFreqMax = dmax1(NewFreqMax, NewFreqMax1)
                    endif


                    !< convert new limits to string
                    NewFreqMin1 = dmin1(NewFreqMin, NewFreqMax)
                    NewFreqMax1 = dmax1(NewFreqMin, NewFreqMax)
                    write(numberString1, '(ES25.15)') NewFreqMin1
                    write(numberString2, '(ES25.15)') NewFreqMax1

                    ! Debug:
                    ! print*,"RangeIndex, k = ", RangeIndex, k
                    ! print*,"freqmin = ", freqmin
                    ! print*,"freqmax = ", freqmax
                    ! print*,"freqmax - freqmin = ", freqmax - freqmin
                    ! print*,"NewFreqMin1 = ", NewFreqMin1
                    ! print*,"NewFreqMax1 = ", NewFreqMax1
                    ! print*,"NewFreqMax1 - NewFreqMin1 = ", NewFreqMax1 - NewFreqMin1
                    ! print*,"LocalLTEFlag = ", LocalLTEFlag


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< get number of entries in VAMDC if current molecule has at least one component which is treated in LTE
                    if (LocalLTEFlag) then


                        !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        !< define query
                        ! MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL


                        !< define order of transitions
                        if (OrderTransSQL == 1) then                                        !< order by lower energy
                            helpString = trim(adjustl(ColumnNameForELowTransitions))
                        elseif (OrderTransSQL == 2) then                                    !< order by gA
                            helpString = trim(adjustl(ColumnNameForEinsteinATransitions)) // " * " &
                                        // trim(adjustl(ColumnNameForgUpTransitions))
                        else                                                                !< order by frequency
                            helpString = trim(adjustl(ColumnNameForFreqTransitions))
                        endif


                        !< restrict to transitions with lower energy and minimal intensity
                        helpString2 = ""
                        if (MaxElowSQL > 0.d0) then                                         !< restrict lower energies
                            write(numberString3, '(ES15.5)') MaxElowSQL                     !< convert float to string for freqmax
                            helpString2 = " AND " // trim(adjustl(ColumnNameForELowTransitions)) // " <= " &
                                                  // trim(adjustl(numberString3))
                        endif
                        if (MingASQL > 0.d0) then                                           !< restrict intensity gA
                            write(numberString3, '(ES15.5)') MingASQL                       !< convert float to string for freqmax
                            helpString2 = trim(adjustl(helpString2)) // " AND " &
                                                    // trim(adjustl(ColumnNameForEinsteinATransitions)) &
                                                    // " * "// trim(adjustl(ColumnNameForgUpTransitions)) &
                                                    // " >= " // trim(adjustl(numberString3))
                        endif


                        if (MaxNumTransitionsSQL <= 0) then
                            queryString1 = "SELECT count(*) FROM " // trim(adjustl(NameOfRadTransTable)) // char(0)
                            queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) &
                                            // " >= " // trim(adjustl(numberString1)) &
                                            // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) &
                                            // " <= " // trim(adjustl(numberString2)) &
                                            // " " // trim(adjustl(helpString2)) // ") ORDER by " &
                                            // trim(adjustl(helpString)) // char(0)
                        else
                            write(numberString3, '(I25)') MaxNumTransitionsSQL              !< convert float to string for freqmax
                            queryString1 = "SELECT count(*) FROM (select * from " // trim(adjustl(NameOfRadTransTable)) // char(0)
                            queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) &
                                            // " >= " // trim(adjustl(numberString1)) &
                                            // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) &
                                            // " <= " // trim(adjustl(numberString2)) &
                                            // " " // trim(adjustl(helpString2)) &
                                            // ") ORDER BY " // trim(adjustl(helpString)) &
                                            // " DESC limit " // trim(adjustl(numberString3)) &
                                            // ") ORDER BY " // trim(adjustl(helpString)) // char(0)
                        endif

                        ! Debug:
                        ! print*,' '
                        ! print*,'queryString1 = >', trim(adjustl(queryString1)), '<'
                        ! print*,'queryString2 = >', trim(adjustl(queryString2)), '<'


                        !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        !< get number of entries for the transition table
                        ok = 0                                                              !< reset status variable
                        number_entries = 0                                                  !< reset result variable
                        j = 1
                        countlen = (len_trim(queryString1) - 1)
                        CopyMoleculeNames = ""
                        CopyMoleculeNames(1) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                        countlen = countlen + len_trim(CopyMoleculeNames(1)) + 12
                        countlen = countlen + (len_trim(queryString2) - 1) + 17
                        dbNameCopy = trim(adjustl(dbName)) // char(0)
                        ColumnNameForNameTransitionsCopy = trim(adjustl(ColumnNameForNameTransitions)) // char(0)
                        call numentr_sqlite3(ok, number_entries, %val(countlen), trim(adjustl(queryString1)), &
                                             trim(adjustl(queryString2)), %val(j), &
                                             trim(adjustl(dbNameCopy)), &
                                             trim(adjustl(ColumnNameForNameTransitionsCopy)), &
                                             CopyMoleculeNames)
                        NumEntriesRanges(RangeIndex, k) = number_entries
                        TotalNumberOfTransitions = TotalNumberOfTransitions + max0(1, number_entries)

                        ! Debug:
                        ! print*,"Frequency range index = ", RangeIndex
                        ! print*,"Molecule index = ", k
                        ! print*,"Number of database entries within the current freq. range (number_entries) = ", NumEntriesRanges(RangeIndex, k)
                        ! print*,'TotalNumberOfTransitions = ', TotalNumberOfTransitions
                        ! print*,'ok = ', ok
                        ! print*,"number_entries = ", number_entries
                    endif

                end Do                                                                      !< k: loop over all molecules
            end Do                                                                          !< RangeIndex: loop over all frequency ranges

            ! Debug:
            ! print*,"TotalNumberOfTransitions = ", TotalNumberOfTransitions


            !<============================================================================================================================================
            !< allocate memory for molecular parameters


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< for VAMDC (LTE)


            !< check, if user-defined minimal number of transition is achieved
            if (TotalNumberOfTransitions < MinNumTransitionsSQL) then
                TotalNumberOfTransitions = 0
            endif


            !< allocate memory for variables MolecularData etc.
            if (allocated(MolecularData)) then
                deallocate(MolecularData, MolecularDataIndices, LowerQNString, UpperQNString, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine GetTransitionParameters!")')
                        write(ErrChannel, '(3x, "Can not deallocate variables MolecularData, MolecularDataIndices, &
                                                &LowerQNString, and UpperQNString!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    ok = 1
                    return
                endif
            endif
            j = max(1, TotalNumberOfTransitions)
            allocate(MolecularData(j, 9), MolecularDataIndices(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2), &
                     LowerQNString(j), UpperQNString(j), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine GetTransitionParameters!")')
                    write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                    write(ErrChannel, '(5x, "- MolecularData(j, 9)")')
                    write(ErrChannel, '(5x, "- MolecularDataIndices(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2)")')
                    write(ErrChannel, '(5x, "- LowerQNString(j)")')
                    write(ErrChannel, '(5x, "- UpperQNString(j)")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameters:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- j = ", I8)') j
                    write(ErrChannel, '(5x, "- TotalNumberOfFrequencyRanges = ", I8)') TotalNumberOfFrequencyRanges
                    write(ErrChannel, '(5x, "- TotalNumberOfMolecules = ", I8)') TotalNumberOfMolecules
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                ok = 1
                return
            endif
            MolecularData = 0.d0
            MolecularDataIndices = 0
            LowerQNString = ""
            UpperQNString = ""


            !<============================================================================================================================================
            !< read in molecular parameters
            if ((TotalNumberOfTransitions > 0) .and. ok == 0) then


                !< read in molecular data from database for each molecule
                MolecularDataCounter = 0
                Do RangeIndex = 1, TotalNumberOfFrequencyRanges                             !< loop over all frequency ranges
                    freqmin = StartFrequency(RangeIndex)                                    !< get first frequency of current frequency range
                    freqmax = EndFrequency(RangeIndex)                                      !< get last frequency of current frequency range
                    Do k = 1, TotalNumberOfMolecules                                        !< loop over all molecules


                        !< get name of molecule in lower case
                        LowerCurrNameMolecule = trim(adjustl(MoleculeNames(k)))
                        call ConverStringToLowerCase(LowerCurrNameMolecule)


                        !< get transitions of molecules and RRLs but not continuum contributions
                        if (LowerCurrNameMolecule(1:5) /= "cont-") then                     !< we need molecules or RRLs
                            LocalLTEFlag = .false.                                          !< initialize local LTE flag

                            ! Debug:
                            ! print*,' '
                            ! print*,"RangeIndex, k, MoleculeNames(k) = ", RangeIndex, k, trim(adjustl(MoleculeNames(k)))


                            !< determine min. and max. velocity offset
                            NewFreqMin = 0.d0
                            NewFreqMax = 0.d0
                            vMin = 1.d99
                            vMax = -1.d99
                            vwidthMax = -1.d99
                            Do j = 1, TotalNumberComponents
                                i = CompMoleculeIndex(j)                                    !< get molecule index for current component
                                LocalFlag = .false.
                                if (IsoFlag) then                                           !< check, if current comp. is used for isotopologues as well
                                    if (IsoRatioConversionTable(i, k) /= 0.d0) then
                                        LocalFlag = .true.
                                    endif
                                elseif (i == k) then
                                    LocalFlag = .true.
                                endif
                                if (LocalFlag) then


                                    !< check, if current component is computed in Non-LTE
                                    if (LTEFlag(j) .or. KindOfMolecule(j) == 2) then
                                        LocalLTEFlag = .true.
                                    endif


                                    !< for myXCLASS function or not-fitted parameters consider the velocity offset and width parameters directly
                                    !< (vOffLimits(j, 1) describes if current parameter is fitted (=1) or not (=0))
                                    if (LogFlag .or. vOffLimits(j, 1) == 0) then


                                        !< determine max. velocity width for current molecule
                                        vwidthLocal = dmax1(myXCLASSParameter(4, j), myXCLASSParameter(5, j))
                                        if (vwidthLocal > vwidthMax) then
                                            vwidthMax = vwidthLocal
                                        endif


                                        !< determine min. and max. velocity offsets for current molecule
                                        vLocal = myXCLASSParameter(6, j)
                                        if (vLocal < vMin) then
                                            vMin = vLocal
                                        endif
                                        if (vLocal > vMax) then
                                            vMax = vLocal
                                        endif


                                    !< for myXCLASSFit function consider the lower and upper limits of velocity offset and width parameters
                                    else


                                        !< determine max. velocity width for current molecule
                                        vwidthLocal = vWidthLimits(j, 2)                    !< get upper limit of velocity width
                                        if (vwidthLocal > vwidthMax) then
                                            vwidthMax = vwidthLocal
                                        endif


                                        !< determine min. and max. velocity offsets for current molecule
                                        vLocal = vOffLimits(j, 2)                           !< get lower limit of velocity offset
                                        if (vLocal < vMin) then
                                            vMin = vLocal
                                        endif
                                        vLocal = vOffLimits(j, 3)                           !< get upper limit of velocity offset
                                        if (vLocal > vMax) then
                                            vMax = vLocal
                                        endif
                                    endif
                                endif
                            end Do
                            if (vMin == 1.d99) then
                                vMin = 0.d0
                            endif
                            if (vMax == -1.d99) then
                                vMax = 0.d0
                            endif
                            if (vwidthMax == -1.d99) then
                                vwidthMax = 0.d0
                            endif

                            ! Debug:
                            ! print*,"vMin = ", vMin
                            ! print*,"vMax = ", vMax


                            !< define new min. and max. frequencies for current range if user-defined vLSR is unequal zero
                            if (GlobalvLSR(RangeIndex) /= 0.d0) then
                                NewFreqMin = (freqmin * (1.d0 - GlobalvLSR(RangeIndex) / ckms))
                                NewFreqMax = (freqmax * (1.d0 - GlobalvLSR(RangeIndex) / ckms))
                            else
                                NewFreqMin = freqmin
                                NewFreqMax = freqmax
                            endif


                            !< shift the frequency range
                            NewFreqMin1 = (NewFreqMin * (1.d0 - dmax1(vMin, vMax) / ckms))
                            NewFreqMax1 = (NewFreqMax * (1.d0 - dmin1(vMin, vMax) / ckms))
                            if (RangeExpansionFlag) then
                                NewFreqMin = dmin1(NewFreqMin, (NewFreqMin1 * 0.9d0))
                                NewFreqMax = dmax1(NewFreqMax, (NewFreqMax1 * 1.1d0))
                            else
                                NewFreqMin = dmin1(NewFreqMin, NewFreqMin1)
                                NewFreqMax = dmax1(NewFreqMax, NewFreqMax1)
                            endif


                            !< convert new limits to string
                            NewFreqMin1 = dmin1(NewFreqMin, NewFreqMax)
                            NewFreqMax1 = dmax1(NewFreqMin, NewFreqMax)
                            write(numberString1, '(ES25.15)') NewFreqMin1
                            write(numberString2, '(ES25.15)') NewFreqMax1

                            ! Debug:
                            ! print*,"RangeIndex, k = ", RangeIndex, k
                            ! print*,"freqmin = ", freqmin
                            ! print*,"freqmax = ", freqmax
                            ! print*,"freqmax - freqmin = ", freqmax - freqmin
                            ! print*,"NewFreqMin1 = ", NewFreqMin1
                            ! print*,"NewFreqMax1 = ", NewFreqMax1
                            ! print*,"NewFreqMax1 - NewFreqMin1 = ", NewFreqMax1 - NewFreqMin1
                            ! print*,"LocalLTEFlag = ", LocalLTEFlag
                            ! print*,"CollMolConversionTable = ", CollMolConversionTable


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< get number of entries in VAMDC if current molecule has at least one component which is treated in LTE
                            if (LocalLTEFlag) then


                                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                !< define query
                                ! MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL


                                !< define order of transition
                                if (OrderTransSQL == 1) then                                !< order by lower energy
                                    helpString = trim(adjustl(ColumnNameForELowTransitions))
                                elseif (OrderTransSQL == 2) then                            !< order by gA
                                    helpString = trim(adjustl(ColumnNameForEinsteinATransitions)) // " * " &
                                                    // trim(adjustl(ColumnNameForgUpTransitions))
                                else                                                        !< order by frequency
                                    helpString = trim(adjustl(ColumnNameForFreqTransitions))
                                endif


                                !< restrict to transitions with lower energy and minimal intensity
                                helpString2 = ""
                                if (MaxElowSQL > 0.d0) then                                 !< restrict lower energies
                                    write(numberString3, '(ES15.5)') MaxElowSQL             !< convert float to string for freqmax
                                    helpString2 = "AND " // trim(adjustl(ColumnNameForELowTransitions)) // " <= " &
                                                         // trim(adjustl(numberString3))
                                endif
                                if (MingASQL > 0.d0) then                                   !< restrict intensity gA
                                    write(numberString3, '(ES15.5)') MingASQL               !< convert float to string for freqmax
                                    helpString2 = trim(adjustl(helpString2)) // " AND " &
                                                        // trim(adjustl(ColumnNameForEinsteinATransitions)) &
                                                        // " * "// trim(adjustl(ColumnNameForgUpTransitions)) &
                                                        // " >= " // trim(adjustl(numberString3))
                                endif


                                !< define query
                                if (MaxNumTransitionsSQL <= 0) then
                                    queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) &
                                                    // " >= " // trim(adjustl(numberString1)) &
                                                    // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) &
                                                    // " <= " // trim(adjustl(numberString2)) &
                                                    // " " // trim(adjustl(helpString2)) &
                                                    // ") ORDER by " // trim(adjustl(helpString)) // char(0)
                                else
                                    write(numberString3, '(I25)') MaxNumTransitionsSQL          !< convert float to string for freqmax
                                    queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) &
                                                    // " >= " // trim(adjustl(numberString1)) &
                                                    // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) &
                                                    // " <= " // trim(adjustl(numberString2)) &
                                                    // " " // trim(adjustl(helpString2)) &
                                                    // ") ORDER by " // trim(adjustl(helpString)) // " DESC limit " &
                                                    // trim(adjustl(numberString3)) // ") ORDER BY " // trim(adjustl(helpString)) &
                                                    // char(0)
                                endif

                                ! Debug:
                                ! print*,' '
                                ! print*,"RangeIndex, k, NumEntriesRanges(RangeIndex, k) = ", RangeIndex, k, NumEntriesRanges(RangeIndex, k)


                                ! select * from (select * from transitions where (T_Name = "CH3OH;v=0;" and T_Frequency >= 300000
                                !  and T_Frequency <= 305000) order by T_EinsteinA DESC limit 5) order by T_Frequency


                                !< allocate memory for worker
                                if (allocated(lFreq)) then
                                    deallocate(MolNameLocal, lFreq, lFreqErr, icatMol, EinsteinA, lElt, &
                                               UpperStateDegeneracy, LocalLowerQNString, &
                                               LocalUpperQNString, stat = deallocstatus)
                                    if (deallocstatus /= 0) then
                                        Do ErrChannelIndex = 1, 2
                                            ErrChannel = AllErrChannels(ErrChannelIndex)
                                            write(ErrChannel, '(" ")')
                                            write(ErrChannel, '(3x, "Error in subroutine GetTransitionParameters!")')
                                            write(ErrChannel, '(3x, "Can not deallocate variables MolNameLocal, &
                                                                     &lFreq, lFreqErr, icatMol, ...!")')
                                            write(ErrChannel, '(" ")')
                                        end Do
                                        ok = 1
                                        return
                                    endif
                                endif
                                NumEntriesLocal = max0(1, NumEntriesRanges(RangeIndex, k))
                                allocate(MolNameLocal(NumEntriesLocal), lFreq(NumEntriesLocal), lFreqErr(NumEntriesLocal), &
                                         icatMol(NumEntriesLocal), &
                                         EinsteinA(NumEntriesLocal), lElt(NumEntriesLocal), &
                                         UpperStateDegeneracy(NumEntriesLocal), &
                                         LocalLowerQNString(NumEntriesLocal), LocalUpperQNString(NumEntriesLocal), &
                                         stat = allocstatus)
                                if (allocstatus /= 0) then
                                    Do ErrChannelIndex = 1, 2
                                        ErrChannel = AllErrChannels(ErrChannelIndex)
                                        write(ErrChannel, '(" ")')
                                        write(ErrChannel, '(3x, "Error in subroutine GetTransitionParameters!")')
                                        write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                                        write(ErrChannel, '(5x, "- MolNameLocal(NumEntriesLocal)")')
                                        write(ErrChannel, '(5x, "- lFreq(NumEntriesLocal)")')
                                        write(ErrChannel, '(5x, "- lFreqErr(NumEntriesLocal)")')
                                        write(ErrChannel, '(5x, "- icatMol(NumEntriesLocal)")')
                                        write(ErrChannel, '(5x, "- EinsteinA(NumEntriesLocal)")')
                                        write(ErrChannel, '(5x, "- lElt(NumEntriesLocal)")')
                                        write(ErrChannel, '(5x, "- UpperStateDegeneracy(NumEntriesLocal)")')
                                        write(ErrChannel, '(5x, "- LocalLowerQNString(NumEntriesLocal)")')
                                        write(ErrChannel, '(5x, "- LocalUpperQNString(NumEntriesLocal)")')
                                        write(ErrChannel, '(" ")')
                                        write(ErrChannel, '(3x,"Used parameter:")')
                                        write(ErrChannel, '(" ")')
                                        write(ErrChannel, '(5x, "- NumEntriesLocal = ", I8)') NumEntriesLocal
                                        write(ErrChannel, '(" ")')
                                        write(ErrChannel, '(" ")')
                                    end Do
                                    ok = 1
                                    return
                                endif
                                MolNameLocal = ""
                                lFreq = 0.d0
                                lFreqErr = 0.d0
                                icatMol = 0.d0
                                EinsteinA = 0.d0
                                lElt = 0.d0
                                UpperStateDegeneracy = 0
                                LocalLowerQNString = ""
                                LocalUpperQNString = ""
                                Do j = 1, NumEntriesLocal
                                    MolNameLocal(j) = char(0)
                                    LocalLowerQNString(j) = char(0)
                                    LocalUpperQNString(j) = char(0)
                                end Do


                                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                !< get molecular data
                                if (NumEntriesRanges(RangeIndex, k) > 0) then               !< check, if molecular data are contained in the database
                                                                                            !< for the current frequency range and molecule

                                    !< construct query string
                                    if (MaxNumTransitionsSQL <= 0) then
                                        queryString1 = "SELECT " // trim(adjustl(ColumnNameForNameTransitions))                 !< pos. 1: molecule
                                                                                                                                !< name
                                    else
                                        queryString1 = "SELECT * FROM (SELECT " // trim(adjustl(ColumnNameForNameTransitions))  !< pos. 1: molecule
                                                                                                                                !< name
                                    endif
                                    queryString1 = trim(adjustl(queryString1)) // ", " &
                                                    // trim(adjustl(ColumnNameForFreqTransitions))
                                                                                                                                !< pos. 2: frequency
                                    queryString1 = trim(adjustl(queryString1)) // ", " &
                                                    // trim(adjustl(ColumnNameForFreqErrTransitions))
                                                                                                                                !< pos. 3: error of
                                                                                                                                !< frequency
                                    queryString1 = trim(adjustl(queryString1)) // ", " &
                                                    // trim(adjustl(ColumnNameForIntTransitions))
                                                                                                                                !< pos. 4: intensity
                                    queryString1 = trim(adjustl(queryString1)) // ", " &
                                                    // trim(adjustl(ColumnNameForEinsteinATransitions))
                                                                                                                                !< pos. 5: Einstein A
                                                                                                                                !<         coefficient
                                    queryString1 = trim(adjustl(queryString1)) // ", " &
                                                    // trim(adjustl(ColumnNameForELowTransitions))
                                                                                                                                !< pos. 6: E_lower
                                    queryString1 = trim(adjustl(queryString1)) // ", " &
                                                    // trim(adjustl(ColumnNameForgUpTransitions))
                                                                                                                                !< pos. 7: upper state
                                                                                                                                !<         degeneracy
                                    queryString1 = trim(adjustl(queryString1)) // ", " &
                                                    // trim(adjustl(ColumnNameForUpperQNTransitions))
                                                                                                                                !< pos. 8: upper QN
                                    queryString1 = trim(adjustl(queryString1)) // ", " &
                                                    // trim(adjustl(ColumnNameForLowerQNTransitions))
                                                                                                                                !< pos. 9: lower QN
                                    queryString1 = trim(adjustl(queryString1)) // " FROM " &
                                                    // trim(adjustl(NameOfRadTransTable)) // char(0)

                                    ! Debug:
                                    ! print*," "
                                    ! print*,"queryString1 = -->", trim(adjustl(queryString1)), "<--"


                                    !< get information from SQLite3 database
                                    ok = 0                                                  !< reset status variable
                                    i = TotalNumberOfTransitions
                                    j = 1
                                    countlen = (len_trim(queryString1) - 1)
                                    CopyMoleculeNames = ""
                                    CopyMoleculeNames(1) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                                    countlen = countlen + len_trim(CopyMoleculeNames(1)) + 12
                                    countlen = countlen + (len_trim(queryString2) - 1) + 17
                                    dbNameCopy = trim(adjustl(dbName)) // char(0)
                                    ColumnNameForNameTransitionsCopy = trim(adjustl(ColumnNameForNameTransitions)) // char(0)


                                    !< read information from SQLite3 database
                                    call radtrans(ok, %val(i), %val(countlen), trim(adjustl(queryString1)), &
                                                  trim(adjustl(queryString2)), %val(j), &
                                                  trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNameTransitionsCopy)),&
                                                  CopyMoleculeNames, MolNameLocal, lFreq, lFreqErr, icatMol, EinsteinA, lElt, &
                                                  UpperStateDegeneracy, LocalLowerQNString, LocalUpperQNString)
                                    if (ok /= 0) then
                                        ErrChannel = 6
                                        write(ErrChannel, '(" ")')
                                        write(ErrChannel, '(" ")')
                                        write(ErrChannel, '(" ")')
                                        write(ErrChannel, '(3x, "Error in subroutine GetTransitionParameters!")')
                                        write(ErrChannel, '(3x, "An error occurs while reading entries from SQLite3 database ", $)')
                                        write(ErrChannel, '(A,  " within the given frequency range!")') &
                                                                trim(adjustl(NameOfRadTransTable))
                                        write(ErrChannel, '(" ")')
                                        write(ErrChannel, '(3x,"Molecule:")')
                                        write(ErrChannel, '(3x,A)') trim(adjustl(MoleculeNames(k)))
                                        write(ErrChannel, '(" ")')
                                        write(ErrChannel, '(3x,"Frequency range:")')
                                        write(ErrChannel, '(3x,"Min. frequency = ", A)') trim(adjustl(numberString1))
                                        write(ErrChannel, '(3x,"Max. frequency = ", A)') trim(adjustl(numberString2))
                                        write(ErrChannel, '(" ")')
                                        write(ErrChannel, '(3x,"Please check your inputs and restart the program!")')
                                        ok = 1
                                        return

                                        ! Debug:
                                        ! print*,'MolNameLocal(1) = ', MolNameLocal(1)
                                        ! print*,'MolNameLocal(TotalNumberOfTransitions) = ', MolNameLocal(TotalNumberOfTransitions)
                                        ! print*,'lFreq(1) = ', lFreq(1)
                                        ! print*,'lFreq(TotalNumberOfTransitions) = ', lFreq(TotalNumberOfTransitions)
                                        ! print*,'lFreqErr(1) = ', lFreqErr(1)
                                        ! print*,'lFreqErr(TotalNumberOfTransitions) = ', lFreqErr(TotalNumberOfTransitions)
                                        ! print*,'icatMol(1) = ', icatMol(1)
                                        ! print*,'icatMol(TotalNumberOfTransitions) = ', icatMol(TotalNumberOfTransitions)
                                        ! print*,'lElt(1) = ', lElt(1)
                                        ! print*,'lElt(TotalNumberOfTransitions) = ', lElt(TotalNumberOfTransitions)
                                        ! print*,'LocalLowerQNString(1) = ', LocalLowerQNString(1)
                                        ! print*,'LocalLowerQNString(TotalNumberOfTransitions) = ', LocalLowerQNString(TotalNumberOfTransitions)
                                        ! print*,'LocalUpperQNString(1) = ', LocalUpperQNString(1)
                                        ! print*,'LocalUpperQNString(TotalNumberOfTransitions) = ', LocalUpperQNString(TotalNumberOfTransitions)
                                        ! stop
                                    endif
                                endif


                                !< check if current molecule is RRL
                                LocalRRLFlag = .false.
                                LowerCurrNameMolecule = trim(adjustl(MoleculeNames(k)))
                                call ConverStringToLowerCase(LowerCurrNameMolecule)
                                if (LowerCurrNameMolecule(1:4) == "rrl-") then
                                    LocalRRLFlag = .true.
                                endif


                                !< copy molecular data for current molecule and frequency range to MolecularData variable
                                MolecularDataIndices(RangeIndex, k, 1) = MolecularDataCounter + 1   !< save first index for current freq. range and mol.
                                                                                            !< within MolecularData variable
                                Do j = 1, NumEntriesLocal                                   !< loop over all transitions within the current frequency
                                                                                            !< range and molecule
                                    MolecularDataCounter = MolecularDataCounter + 1
                                    MolecularData(MolecularDataCounter, 1) = lFreq(j)
                                    MolecularData(MolecularDataCounter, 2) = EinsteinA(j)
                                    MolecularData(MolecularDataCounter, 3) = lElt(j) / 1.438769 !< only for debugging, convert K to cm-1?
                                    MolecularData(MolecularDataCounter, 4) = UpperStateDegeneracy(j)
                                    call strlen(LocalLowerQNString(j))
                                    LowerQNString(MolecularDataCounter) = trim(adjustl(LocalLowerQNString(j)))
                                    call strlen( LocalUpperQNString(j))
                                    UpperQNString(MolecularDataCounter) = trim(adjustl(LocalUpperQNString(j)))

                                    ! Debug:
                                    ! print*,"MolecularDataCounter = ", MolecularDataCounter
                                    ! print*,"LowerQNString(MolecularDataCounter) = ", trim(adjustl(LowerQNString(MolecularDataCounter))), '<<'
                                    ! print*,"UpperQNString(MolecularDataCounter) = ", trim(adjustl(UpperQNString(MolecularDataCounter))), '<<'


                                    !< store additional information for RRLs
                                    if (LocalRRLFlag) then


                                        !< determine Z
                                        if (LowerCurrNameMolecule(5:5) == "h") then
                                            MolecularData(MolecularDataCounter, 5) = 1.d0
                                        elseif (LowerCurrNameMolecule(5:5) == "he") then
                                            MolecularData(MolecularDataCounter, 5) = 2.d0
                                        elseif (LowerCurrNameMolecule(5:5) == "c") then
                                            MolecularData(MolecularDataCounter, 5) = 6.d0
                                        elseif (LowerCurrNameMolecule(5:5) == "n") then
                                            MolecularData(MolecularDataCounter, 5) = 7.d0
                                        elseif (LowerCurrNameMolecule(5:5) == "o") then
                                            MolecularData(MolecularDataCounter, 5) = 8.d0
                                        elseif (LowerCurrNameMolecule(5:5) == "s") then
                                            MolecularData(MolecularDataCounter, 5) = 16.d0
                                        else
                                            MolecularData(MolecularDataCounter, 5) = 1.d0
                                        endif


                                        !< store main quantum number of lower and upper states
                                        read(LowerQNString(MolecularDataCounter)(6:), '(I10)') n
                                        MolecularData(MolecularDataCounter, 6) = dble(n)
                                        read(UpperQNString(MolecularDataCounter)(6:), '(I10)') n
                                        MolecularData(MolecularDataCounter, 7) = dble(n)
                                    endif

                                    ! Debug:
                                    ! print*,"k, trim(adjustl(MoleculeNames(k))) = ", k, trim(adjustl(MoleculeNames(k)))
                                    ! print*,"j, MolecularData(j, :) = ", j, MolecularData(j, :)


                                    !< correct entry for lower energy
                                    MolecularData(MolecularDataCounter, 3) = lElt(j)

                                    ! Debug:
                                    ! print*," "
                                    ! print*,"MolecularDataCounter = ", MolecularDataCounter
                                    ! print*,"MolecularData(MolecularDataCounter, :) = ", MolecularData(MolecularDataCounter, :)


                                    !< store number of transition frequency per frequency channel
                                    LocalTransFreqIndex = minloc(dabs(lFreq(j) - ObservationalDataList(:, 1)), dim = 1)
                                    NumTransFreqPerObsFreq(k, LocalTransFreqIndex) &
                                                                        = NumTransFreqPerObsFreq(k, LocalTransFreqIndex) + 1
                                end Do                                                      !< j: loop over all transitions
                                MolecularDataIndices(RangeIndex, k, 2) = MolecularDataCounter   !< save last index for current freq. range and molecule
                                if (NumEntriesRanges(RangeIndex, k) == 0) then
                                    MolecularDataIndices(RangeIndex, k, 1) = 0
                                    MolecularDataIndices(RangeIndex, k, 2) = 0
                                endif
                                                                                            !< within MolecularData variable

                                ! Debug:
                                ! print*,"RangeIndex, k, MolecularDataIndices(RangeIndex, k, :) = ", RangeIndex, k, MolecularDataIndices(RangeIndex, k, :)
                            endif

                        endif                                                               !< continue here, if current "molecule" is continuum
                    end Do                                                                  !< k: loop over all molecules
                end Do                                                                      !< RangeIndex: loop over all frequency ranges
            endif


            !<============================================================================================================================================
            !< print summary to screen


            !< print what you do
            if (printflag) then
                print '("done!")'
            endif


            !< print some information to screen about transitions
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(16x, "Number of transitions for each frequency range and molecule:")'
                Do RangeIndex = 1, TotalNumberOfFrequencyRanges                             !< loop over all frequency ranges
                    freqmin = StartFrequency(RangeIndex)                                    !< get first frequency of current frequency range
                    freqmax = EndFrequency(RangeIndex)                                      !< get last frequency of current frequency range
                    write(numberString1, '(ES25.15)') freqmin
                    write(numberString2, '(ES25.15)') freqmax
                    print '(" ")'
                    print '(16x, "Frequency range: ", A, " MHz - ", A, " MHz:")', trim(adjustl(numberString1)), &
                                                                                  trim(adjustl(numberString2))
                    Do k = 1, TotalNumberOfMolecules                                        !< loop over all molecules
                        LowerCurrNameMolecule = trim(adjustl(MoleculeNames(k)))
                        call ConverStringToLowerCase(LowerCurrNameMolecule)
                        if (LowerCurrNameMolecule(1:5) /= "cont-") then
                            print '(18x, "Number of transitions for molecule ", A, A, A, ": ", T92, I10)', char(34), &
                                                                    trim(adjustl(MoleculeNames(k))), char(34), &
                                                                    (NumEntriesRanges(RangeIndex, k))
                        endif
                    end Do
                end Do
                print '(" ")'
                print '(" ")'
            endif


            !< free memory of local variables
            deallocstatus = 0
            if (allocated(MolNameLocal)) deallocate(MolNameLocal, stat = deallocstatus)
            if (allocated(lFreq) .and. deallocstatus == 0) deallocate(lFreq, stat = deallocstatus)
            if (allocated(lFreqErr) .and. deallocstatus == 0) deallocate(lFreqErr, stat = deallocstatus)
            if (allocated(icatMol) .and. deallocstatus == 0) deallocate(icatMol, stat = deallocstatus)
            if (allocated(EinsteinA) .and. deallocstatus == 0) deallocate(EinsteinA, stat = deallocstatus)
            if (allocated(lElt) .and. deallocstatus == 0) deallocate(lElt, stat = deallocstatus)
            if (allocated(LocalLowerQNString) .and. deallocstatus == 0) deallocate(LocalLowerQNString, stat = deallocstatus)
            if (allocated(LocalUpperQNString) .and. deallocstatus == 0) deallocate(LocalUpperQNString, stat = deallocstatus)
            if (allocated(UpperStateDegeneracy) .and. deallocstatus == 0) deallocate(UpperStateDegeneracy, stat = deallocstatus)
            if (deallocstatus /= 0) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine GetTransitionParameters!")')
                    write(ErrChannel, '(3x, "Can not deallocate variables MolNameLocal etc.!")')
                    write(ErrChannel, '(" ")')
                end Do
                ok = 1
            endif
            return
        end subroutine GetTransitionParameters


        !>************************************************************************************************************************************************
        !> subroutine: Removeduplicates
        !>
        !> remove duplicates in a 1d list
        !>
        !>
        !> source: https://rosettacode.org/wiki/Remove_duplicate_elements#Fortran
        !>
        !>
        !> input variables:     LenIn:          length of input vector
        !>
        !>                      VecIn:          input vector
        !>
        !>
        !> output variables:    LenOut:         length of output vector
        !>
        !>                      VecOut:         output vector
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2017-11-28
        !>
        subroutine Removeduplicates(LenIn, VecIn, LenOut, VecOut)

            implicit none
            integer :: LenIn                                                                !< length of input vector
            integer :: LenOut                                                               !< length of output vector
            real*8, dimension(LenIn) :: VecIn                                               !< input vector
            real*8, dimension(LenIn) :: VecOut                                              !< output vector
            integer :: i, j                                                                 !< loop variables


            LenOut = 1
            VecOut(1) = VecIn(1)
            outer: Do i = 2, LenIn
                Do j = 1, LenOut
                    if (VecOut(j) == VecIn(i)) then                                         !< Found a match so start looking again
                        cycle outer
                    endif
                end Do


                !< No match found so add it to the output
                LenOut = LenOut + 1
                VecOut(LenOut) = VecIn(i)
            end Do outer
        end subroutine Removeduplicates


        !>************************************************************************************************************************************************
        !> subroutine: myXCLASSInitVar
        !>
        !> initialize myXCLASS variables
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2014-07-27
        !>
        subroutine myXCLASSInitVar

            implicit none
            integer :: ok                                                                   !< status variablee: name of parameter
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation


            !< initialize variables
            tt1 = 0.d0                                                                      !< initialize debug parameter tt1
            tt2 = 0.d0                                                                      !< initialize debug parameter tt2


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define variables for SQLite3 database
            NameOfRadTransTable = "transitions"
            NameOfPartFuncTable = "partitionfunctions"
            NumberOfTemperatures = 110


            !< allocate variables for partition function
            if (allocated(ColumnNamesPartFunc)) then
                deallocate(ColumnNamesPartFunc, TempPartFunc, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine myXCLASSInitVar!")')
                        write(ErrChannel, '(3x, "Can not deallocate variables ColumnNamesPartFunc and TempPartFunc!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    stop 'Program aborted! Please restart the program!'
                endif
            endif
            allocate(ColumnNamesPartFunc(NumberOfTemperatures), TempPartFunc(NumberOfTemperatures), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine myXCLASSInitVar!")')
                    write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                    write(ErrChannel, '(5x, "- ColumnNamesPartFunc(NumberOfTemperatures)")')
                    write(ErrChannel, '(5x, "- TempPartFunc(NumberOfTemperatures)")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameter:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- NumberOfTemperatures = ", I8)') NumberOfTemperatures
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                stop 'Program aborted! Please restart the program!'
            endif
            ColumnNamesPartFunc = ""
            TempPartFunc = 0.d0


            !< define column names of table partitionfunctions
            !< Due to a bug in the current gfortran compiler a statement like ColumnNamesPartFunc = (/.../) is not possible
            ColumnNameForNamePartFunc = 'PF_Name'
            ColumnNamesPartFunc(1) = 'PF_1_072'
            ColumnNamesPartFunc(2) = 'PF_1_148'
            ColumnNamesPartFunc(3) = 'PF_1_230'
            ColumnNamesPartFunc(4) = 'PF_1_318'
            ColumnNamesPartFunc(5) = 'PF_1_413'
            ColumnNamesPartFunc(6) = 'PF_1_514'
            ColumnNamesPartFunc(7) = 'PF_1_622'
            ColumnNamesPartFunc(8) = 'PF_1_738'
            ColumnNamesPartFunc(9) = 'PF_1_862'
            ColumnNamesPartFunc(10) = 'PF_1_995'
            ColumnNamesPartFunc(11) = 'PF_2_138'
            ColumnNamesPartFunc(12) = 'PF_2_291'
            ColumnNamesPartFunc(13) = 'PF_2_455'
            ColumnNamesPartFunc(14) = 'PF_2_630'
            ColumnNamesPartFunc(15) = 'PF_2_725'
            ColumnNamesPartFunc(16) = 'PF_2_818'
            ColumnNamesPartFunc(17) = 'PF_3_020'
            ColumnNamesPartFunc(18) = 'PF_3_236'
            ColumnNamesPartFunc(19) = 'PF_3_467'
            ColumnNamesPartFunc(20) = 'PF_3_715'
            ColumnNamesPartFunc(21) = 'PF_3_981'
            ColumnNamesPartFunc(22) = 'PF_4_266'
            ColumnNamesPartFunc(23) = 'PF_4_571'
            ColumnNamesPartFunc(24) = 'PF_4_898'
            ColumnNamesPartFunc(25) = 'PF_5_000'
            ColumnNamesPartFunc(26) = 'PF_5_248'
            ColumnNamesPartFunc(27) = 'PF_5_623'
            ColumnNamesPartFunc(28) = 'PF_6_026'
            ColumnNamesPartFunc(29) = 'PF_6_457'
            ColumnNamesPartFunc(30) = 'PF_6_918'
            ColumnNamesPartFunc(31) = 'PF_7_413'
            ColumnNamesPartFunc(32) = 'PF_7_943'
            ColumnNamesPartFunc(33) = 'PF_8_511'
            ColumnNamesPartFunc(34) = 'PF_9_120'
            ColumnNamesPartFunc(35) = 'PF_9_375'
            ColumnNamesPartFunc(36) = 'PF_9_772'
            ColumnNamesPartFunc(37) = 'PF_10_471'
            ColumnNamesPartFunc(38) = 'PF_11_220'
            ColumnNamesPartFunc(39) = 'PF_12_023'
            ColumnNamesPartFunc(40) = 'PF_12_882'
            ColumnNamesPartFunc(41) = 'PF_13_804'
            ColumnNamesPartFunc(42) = 'PF_14_791'
            ColumnNamesPartFunc(43) = 'PF_15_849'
            ColumnNamesPartFunc(44) = 'PF_16_982'
            ColumnNamesPartFunc(45) = 'PF_18_197'
            ColumnNamesPartFunc(46) = 'PF_18_750'
            ColumnNamesPartFunc(47) = 'PF_19_498'
            ColumnNamesPartFunc(48) = 'PF_20_893'
            ColumnNamesPartFunc(49) = 'PF_22_387'
            ColumnNamesPartFunc(50) = 'PF_23_988'
            ColumnNamesPartFunc(51) = 'PF_25_704'
            ColumnNamesPartFunc(52) = 'PF_27_542'
            ColumnNamesPartFunc(53) = 'PF_29_512'
            ColumnNamesPartFunc(54) = 'PF_31_623'
            ColumnNamesPartFunc(55) = 'PF_33_884'
            ColumnNamesPartFunc(56) = 'PF_36_308'
            ColumnNamesPartFunc(57) = 'PF_37_500'
            ColumnNamesPartFunc(58) = 'PF_38_905'
            ColumnNamesPartFunc(59) = 'PF_41_687'
            ColumnNamesPartFunc(60) = 'PF_44_668'
            ColumnNamesPartFunc(61) = 'PF_47_863'
            ColumnNamesPartFunc(62) = 'PF_51_286'
            ColumnNamesPartFunc(63) = 'PF_54_954'
            ColumnNamesPartFunc(64) = 'PF_58_884'
            ColumnNamesPartFunc(65) = 'PF_63_096'
            ColumnNamesPartFunc(66) = 'PF_67_608'
            ColumnNamesPartFunc(67) = 'PF_72_444'
            ColumnNamesPartFunc(68) = 'PF_75_000'
            ColumnNamesPartFunc(69) = 'PF_77_625'
            ColumnNamesPartFunc(70) = 'PF_83_176'
            ColumnNamesPartFunc(71) = 'PF_89_125'
            ColumnNamesPartFunc(72) = 'PF_95_499'
            ColumnNamesPartFunc(73) = 'PF_102_329'
            ColumnNamesPartFunc(74) = 'PF_109_648'
            ColumnNamesPartFunc(75) = 'PF_117_490'
            ColumnNamesPartFunc(76) = 'PF_125_893'
            ColumnNamesPartFunc(77) = 'PF_134_896'
            ColumnNamesPartFunc(78) = 'PF_144_544'
            ColumnNamesPartFunc(79) = 'PF_150_000'
            ColumnNamesPartFunc(80) = 'PF_154_882'
            ColumnNamesPartFunc(81) = 'PF_165_959'
            ColumnNamesPartFunc(82) = 'PF_177_828'
            ColumnNamesPartFunc(83) = 'PF_190_546'
            ColumnNamesPartFunc(84) = 'PF_204_174'
            ColumnNamesPartFunc(85) = 'PF_218_776'
            ColumnNamesPartFunc(86) = 'PF_225_000'
            ColumnNamesPartFunc(87) = 'PF_234_423'
            ColumnNamesPartFunc(88) = 'PF_251_189'
            ColumnNamesPartFunc(89) = 'PF_269_153'
            ColumnNamesPartFunc(90) = 'PF_288_403'
            ColumnNamesPartFunc(91) = 'PF_300_000'
            ColumnNamesPartFunc(92) = 'PF_309_030'
            ColumnNamesPartFunc(93) = 'PF_331_131'
            ColumnNamesPartFunc(94) = 'PF_354_813'
            ColumnNamesPartFunc(95) = 'PF_380_189'
            ColumnNamesPartFunc(96) = 'PF_407_380'
            ColumnNamesPartFunc(97) = 'PF_436_516'
            ColumnNamesPartFunc(98) = 'PF_467_735'
            ColumnNamesPartFunc(99) = 'PF_500_000'
            ColumnNamesPartFunc(100) = 'PF_501_187'
            ColumnNamesPartFunc(101) = 'PF_537_032'
            ColumnNamesPartFunc(102) = 'PF_575_440'
            ColumnNamesPartFunc(103) = 'PF_616_595'
            ColumnNamesPartFunc(104) = 'PF_660_693'
            ColumnNamesPartFunc(105) = 'PF_707_946'
            ColumnNamesPartFunc(106) = 'PF_758_578'
            ColumnNamesPartFunc(107) = 'PF_812_831'
            ColumnNamesPartFunc(108) = 'PF_870_964'
            ColumnNamesPartFunc(109) = 'PF_933_254'
            ColumnNamesPartFunc(110) = 'PF_1000_000'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define log10 of temperatures for partition function for
            !< T = 1.072, 1.148, 1.23, 1.318, 1.413, 1.514, 1.622, 1.738, 1.862, 1.995, 2.138, 2.291, 2.455, 2.63, 2.725, 2.818, 3.02, 3.236, 3.467,
            !<     3.715, 3.981, 4.266, 4.571, 4.898, 5, 5.248, 5.623, 6.026, 6.457, 6.918, 7.413, 7.943, 8.511, 9.12, 9.375, 9.772, 10.471, 11.22,
            !<     12.023, 12.882, 13.804, 14.791, 15.849, 16.982, 18.197, 18.75, 19.498, 20.893, 22.387, 23.988, 25.704, 27.542, 29.512, 31.623,
            !<     33.884, 36.308, 37.5, 38.905, 41.687, 44.668, 47.863, 51.286, 54.954, 58.884, 63.096, 67.608, 72.444, 75, 77.625, 83.176, 89.125,
            !<     95.499, 102.329, 109.648, 117.49, 125.893, 134.896, 144.544, 150, 154.882, 165.959, 177.828, 190.546, 204.174, 218.776, 225,
            !<     234.423, 251.189, 269.153, 288.403, 300, 309.03, 331.131, 354.813, 380.189, 407.38, 436.516, 467.735, 500, 501.187, 537.032,
            !<     575.44, 616.595, 660.693, 707.946, 758.578, 812.831, 870.964, 933.254, 1000
            TempPartFunc = (/0.030194785356751d0, 0.059941888061955d0, 0.089905111439398d0, 0.119915410257991d0, &
                             0.150142161848559d0, 0.180125875164054d0, 0.210050849875137d0, 0.240049772112648d0, &
                             0.269979676645324d0, 0.299942900022767d0, 0.330007700872759d0, 0.360025089189397d0, &
                             0.390051496458987d0, 0.419955748489758d0, 0.435366506612661d0, 0.449940988773338d0, &
                             0.480006942957151d0, 0.510008512940235d0, 0.539953841656397d0, 0.569958818096594d0, &
                             0.599992177584098d0, 0.630020851113410d0, 0.660011221289331d0, 0.690018780788695d0, &
                             0.698970004336019d0, 0.719993826367604d0, 0.749968083509403d0, 0.780029127337338d0, &
                             0.810030786405839d0, 0.839980557678343d0, 0.869994000121742d0, 0.899984562549391d0, &
                             0.929980590515515d0, 0.959994838328416d0, 0.971971276399756d0, 0.989983458301399d0, &
                             1.019988159591290d0, 1.049992856920140d0, 1.080012847107930d0, 1.109983294819890d0, &
                             1.140004950619450d0, 1.169997537066570d0, 1.200001865406600d0, 1.229988836544810d0, &
                             1.259999795081890d0, 1.273001272063740d0, 1.289990066054320d0, 1.320000804264730d0, &
                             1.349995899262860d0, 1.379994040165740d0, 1.410000712543460d0, 1.439995473953810d0, &
                             1.469998642218750d0, 1.500003068051690d0, 1.529994673069140d0, 1.560002326773370d0, &
                             1.574031267727720d0, 1.590005419651330d0, 1.620000642300070d0, 1.649996507466040d0, &
                             1.679999916229220d0, 1.709998828025190d0, 1.739999309401970d0, 1.769997304028540d0, &
                             1.800001827817970d0, 1.829998088697380d0, 1.860002421901100d0, 1.875061263391700d0, &
                             1.890001613184640d0, 1.919998030964020d0, 1.949999542859920d0, 1.979998823973740d0, &
                             2.009998730048060d0, 2.040000714473320d0, 2.070000903802340d0, 2.100001582801350d0, &
                             2.129999071957430d0, 2.160000068881300d0, 2.176091259055680d0, 2.190000948069660d0, &
                             2.220000809286660d0, 2.250000144081290d0, 2.279999836361080d0, 2.310000437185500d0, &
                             2.339999677628230d0, 2.352182518111360d0, 2.370000219475120d0, 2.400000616976380d0, &
                             2.429999224858320d0, 2.459999773650270d0, 2.477121254719660d0, 2.490000641890950d0, &
                             2.519999840669680d0, 2.549999523574670d0, 2.579999547278480d0, 2.609999703842230d0, &
                             2.640000166906100d0, 2.669999868814090d0, 2.698970004336020d0, 2.699999797554580d0, &
                             2.730000164674160d0, 2.760000047292730d0, 2.789999998688880d0, 2.819999705510490d0, &
                             2.850000132271100d0, 2.880000243300670d0, 2.910000258512930d0, 2.940000204462930d0, &
                             2.969999860022620d0, 3.000000000000000d0/)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine order of partition function in database: ascending or descending order
            TempLow = min(TempPartFunc(1), TempPartFunc(NumberOfTemperatures))
            TempHigh = max(TempPartFunc(1), TempPartFunc(NumberOfTemperatures))
            if (TempPartFunc(1) > TempPartFunc(NumberOfTemperatures)) then
                Firsti = NumberOfTemperatures
                Lasti = 1
                stepi = (-1)
            else
                Firsti = 1
                Lasti = NumberOfTemperatures
                stepi = 1
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define column names of table transitions
            ColumnNameForNameTransitions = "T_Name"
            ColumnNameForFreqTransitions = "T_Frequency"
            ColumnNameForIntTransitions = "T_Intensity"
            ColumnNameForEinsteinATransitions = "T_EinsteinA"
            ColumnNameForFreqErrTransitions = "T_Uncertainty"
            ColumnNameForELowTransitions = "T_EnergyLower"
            ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
            ColumnNameForLowerQNTransitions = "T_LowerStateQuantumNumbers"
            ColumnNameForUpperQNTransitions = "T_UpperStateQuantumNumbers"
            Column300KPartFunc = 91                                                         !< (only used, if use_intensity_flag == .true): define the
                                                                                            !< column number, containing the partition function for 300 K

            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get myXCLASS parameter
            call GetmyXCLASSParameter


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get partition functions for selected molecules and describe partition functions with polynominal
            call GetPartitionFunction(ok)

            ! Debug:
            ! stop 'Subroutine GetPartitionFunction finished!'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get information for each transition in the given frequency ranges
            call GetTransitionParameters(ok)

            ! Debug:
            ! stop 'Subroutine GetTransitionParameters finished!'


            !< we're done
            return
        end subroutine myXCLASSInitVar


        !>************************************************************************************************************************************************
        !> subroutine: InterpolateQ
        !>
        !> interpolates partition function for a given temperature
        !>
        !>
        !> input variables:     Temperature:            temperature
        !>                      MoleculeIndex:          molecule index
        !>
        !> output variable:     QLocal:                 interpolated partition function
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2014-02-17
        !>
        subroutine InterpolateQ(QLocal, Temperature, MoleculeIndex)

            implicit none
            integer :: i                                                                    !< loop variable
            integer :: MoleculeIndex                                                        !< molecule index
            real*8 :: Temperature                                                           !< temperature
            real*8 :: logt, x0, x1, f0, f1                                                  !< working variables for interpolation
            real*8 :: QLocal                                                                !< interpolated partition function


            !< calculate the logarithm of the kinetic temperature tk
            logt = dlog10(Temperature)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< is logt out of range?
            if (logt < TempLow .or. logt > TempHigh) then

                ! Debug:
                ! print*," "
                ! print*," "
                ! print*,"Firsti = ", Firsti
                ! print*,"Lasti = ", Lasti
                ! print*,"Temperature, logt = ", Temperature, logt
                ! print*,"TempHigh = ", TempHigh
                ! print*,"NumberOfTemperatures = ", NumberOfTemperatures


                !< extrapolation for temperatures lower than the lower threshold
                x0 = 0.d0
                x1 = 0.d0
                f0 = 0.d0
                f1 = 0.d0
                if (logt < TempLow) then
                    if (Firsti < Lasti) then                                                !< TempPartFunc(1) describes TempLow
                        x0 = TempPartFunc(1)
                        x1 = TempPartFunc(2)
                        f0 = lgQ(1, MoleculeIndex)
                        f1 = lgQ(2, MoleculeIndex)
                    else                                                                    !< TempPartFunc(1) describes TempHigh
                        x0 = TempPartFunc(NumberOfTemperatures)
                        x1 = TempPartFunc((NumberOfTemperatures - 1))
                        f0 = lgQ(NumberOfTemperatures, MoleculeIndex)
                        f1 = lgQ((NumberOfTemperatures - 1), MoleculeIndex)
                    endif


                !< extrapolation for temperatures higher than the upper threshold
                elseif (logt > TempHigh) then
                    if (Firsti < Lasti) then                                                !< TempPartFunc(1) describes TempLow
                        x0 = TempPartFunc((NumberOfTemperatures - 1))
                        x1 = TempPartFunc(NumberOfTemperatures)
                        f0 = lgQ((NumberOfTemperatures - 1), MoleculeIndex)
                        f1 = lgQ(NumberOfTemperatures, MoleculeIndex)
                    else                                                                    !< TempPartFunc(1) describes TempHigh
                        x0 = TempPartFunc(2)
                        x1 = TempPartFunc(1)
                        f0 = lgQ(2, MoleculeIndex)
                        f1 = lgQ(1, MoleculeIndex)
                    endif
                endif
                QLocal = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)
                QLocal = 10.d0**QLocal

                ! Debug:
                ! print*,' '
                ! print*,'logt > TempPartFunc(1) .or. logt < TempPartFunc(NumberOfTemperatures):'
                ! print*,'x0, f0 = ', x0, f0
                ! print*,'x1, f1 = ', x1, f1
                ! print*,'logt = ',logt
                ! print*,'>QLocal = ', QLocal
                ! print*,'>log10(QLocal) = ', dlog10(QLocal)
                ! stop


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< do normal linear interpolation
            else


                !< start linear interpolation
                QLocal = 0.d0
                Do i = 1, NumberOfTemperatures                                              !< loop over all temperatures
                    if (dabs(TempPartFunc(i) - logt) < 1.d-6) then
                        QLocal = lgQ(i, MoleculeIndex)
                        exit
                    endif


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< linear interpolation: f(x) = f_0 + \frac{f_1 - f_0}{x_1 - x_0} (x - x_0)


                    !< temperature array is in ascending order
                    if (stepi > 0) then
                        if (i == NumberOfTemperatures) then
                            x0 = TempPartFunc(i - 1)
                            x1 = TempPartFunc(i)
                            f0 = lgQ(i - 1, MoleculeIndex)
                            f1 = lgQ(i, MoleculeIndex)
                            QLocal = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)
                            exit


                        elseif (TempPartFunc(i) < logt .and. logt < TempPartFunc(i + 1)) then
                            x0 = TempPartFunc(i)
                            x1 = TempPartFunc(i + 1)
                            f0 = lgQ(i, MoleculeIndex)
                            f1 = lgQ(i + 1, MoleculeIndex)
                            QLocal = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)
                            exit
                        endif


                    !< temperature array is in descending order
                    else
                        if (i == NumberOfTemperatures) then
                            x0 = TempPartFunc(i - 1)
                            x1 = TempPartFunc(i)
                            f0 = lgQ(i - 1, MoleculeIndex)
                            f1 = lgQ(i, MoleculeIndex)
                            QLocal = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)
                            exit


                        elseif (TempPartFunc(i) > logt .and. logt > TempPartFunc(i + 1)) then
                            x0 = TempPartFunc(i)
                            x1 = TempPartFunc(i + 1)
                            f0 = lgQ(i, MoleculeIndex)
                            f1 = lgQ(i + 1, MoleculeIndex)
                            QLocal = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)
                            exit
                        endif
                    endif
                end Do                                                                      !< i: loop over all temperatures
                QLocal = 10.d0**QLocal                                                      !< convert to linear scale

                ! Debug:
                ! print*,' '
                ! print*,' '
                ! print*,"MoleculeNames = ", trim(adjustl(MoleculeNames(MoleculeIndex)))
                ! print*,'Temperature = ', Temperature
                ! print*,'QLocal = ', QLocal
                ! print*,'logt = ', logt
                !    Do i = 1, NumberOfTemperatures
                !        print*,'i = ', i
                !        print*,'10.d0**lgQ(i, MoleculeIndex) = ', 10.d0**lgQ(i, MoleculeIndex), "           ", QLocal
                !        print*,'10.d0**TempPartFunc(i) = ', 10.d0**TempPartFunc(i), "           ",Temperature
                !    end Do
                ! print*,'log10(QLocal) = ', dlog10(QLocal)
            endif


            !< we're done
            return
        end subroutine InterpolateQ


        !>************************************************************************************************************************************************
        !> function: GaussianFunction
        !>
        !> return a 1-dimensional Gaussian function.
        !>
        !>
        !> for asymmetric Gaussian functions or skew normal distributions see http://reference.wolfram.com/language/ref/SkewNormalDistribution.html
        !>
        !>
        !> input variables:     x:                      frequency
        !>                      GaussCenter:            center of Gaussian
        !>                      GaussFWHM:              FWHM of Gaussian
        !>
        !> output variable:     GaussianFunction:       calculated Gaussian
        !>
        !>
        !> \author  Thomas Moeller
        !>
        !> \date    2017-11-17
        !>
        function GaussianFunction(x, GaussCenter, GaussFWHM)

            implicit none
            real*8 :: x                                                                     !< frequency
            real*8 :: GaussCenter                                                           !< center of Gaussian
            real*8 :: GaussFWHM                                                             !< FWHM of Gaussian
            real*8 :: GaussianFunction                                                      !< calculated Gaussian
            real*8 :: GaussSigma                                                            !< variance of Gaussian
            real*8, parameter :: s2pi = dsqrt(2.d0 * pi)                                    !< used for function evaluation


            !< compute Gaussian
            GaussSigma = GaussFWHM / (2.d0 * dsqrt(2.d0 * dlog(2.d0)))
            GaussianFunction = (1.d0 / (s2pi * GaussSigma)) * ExpSave(-(x - GaussCenter)**2 / (2.d0 * GaussSigma**2)) * 1.d-6
        end function GaussianFunction


        !>************************************************************************************************************************************************
        !>  subroutine: InterPol1D
        !>
        !> interpolate a 1D function an irregular 1D grid
        !> Given arrays xa and ya of length N, and given a value x, this routine returns a value y, and an error estimate dy.
        !> If P(x) is the polynomial of degree N - 1 such that P(xa_i) = ya_i, i = 1, ..., N, then the returned value y = P(x).
        !>
        !>
        !> source:              modified NR routine
        !>
        !>
        !> input variables:     xa:                     array containing x points
        !>                      ya:                     array containing corresponding y values
        !>                      x:                      point of interest
        !>
        !> output variable:     y:                      interpolated y value at x
        !>                      dy:                     error estimate
        !>
        !>
        !> \author  Thomas Moeller
        !>
        !> \date    2017-11-29
        !>
        subroutine InterPol1D(xa, ya, x, y, dy)

            implicit none
            real*8, dimension(:), INTENT(IN) :: xa, ya
            real*8, INTENT(IN) :: x
            real*8, INTENT(OUT) :: y, dy
            integer :: n, ns
            integer, dimension(1) :: imin
            real*8 :: f0, f1, x0, x1


            n = size(xa)                                                                    !< get size of xa
            imin = minloc(dabs(x - xa))                                                     !< Find index ns of closest table entry.
            ns = imin(1)
            dy = 0.d0

            ! Debug:
            ! print*,"n = ", n
            ! print*,"ns = ", ns


            !< do linear interpolation
            if (ns == n) then
                x0 = xa(ns - 1)
                x1 = xa(ns)
                f0 = ya(ns - 1)
                f1 = ya(ns)
            else
                x0 = xa(ns)
                x1 = xa(ns + 1)
                f0 = ya(ns)
                f1 = ya(ns + 1)
            endif
            y = f0 + (f1 - f0) / (x1 - x0) * (x - x0)


            !< we're done
            return
        end subroutine InterPol1D


        !>************************************************************************************************************************************************
        !> subroutine: InterPol2D
        !>
        !> interpolate a 2D function on an irregular grid
        !> Given arrays x1a of length M and x2a of length N of independent variables, and an M x N array of function values ya,
        !> tabulated at the grid points defined by x1a and x2a, and given values x1 and x2 of the independent variables, this
        !> routine returns an interpolated function value y, and an accuracy indication dy (based only on the interpolation in the
        !> x1 direction, however).
        !>
        !>
        !> source:              modified NR routine
        !>
        !>
        !> input variables:     x1a:                    array of x1
        !>                      x2a:                    array of x2
        !>                      ya:                     array of function values
        !>                      x1:                     point x1 of interest
        !>                      x2:                     point x2 of interest
        !>
        !> output variable:     y:                      interpolated y value at x1, x2
        !>                      dy:                     error estimate
        !>
        !>
        !> \author  Thomas Moeller
        !>
        !> \date    2017-11-29
        !>
        subroutine InterPol2D(x1a, x2a, ya, x1, x2, y, dy)

            implicit none
            real*8, dimension(:), INTENT(IN) :: x1a, x2a
            real*8, dimension(:, :), INTENT(IN) :: ya
            real*8, INTENT(IN) :: x1, x2
            real*8, INTENT(OUT) :: y, dy
            integer :: j, m, ndum
            real*8, dimension(size(x1a)) :: ymtmp
            real*8, dimension(size(x2a)) :: yntmp

            ! Debug:
            ! print*,"x1, x2 = ", x1, x2
            ! print*," "
            ! print*,"x1a = ", x1a(:)
            ! print*," "
            ! print*,"x2a = ", x2a(:)
            ! print*," "
            !    Do j = 1, size(x1a)
            !        print*,"ya(j, 1:l2) = ", ya(j, 1:size(x2a))
            !    end Do
            ! print*," "


            ! orig. NR:
            ! m=assert_eq(size(x1a),size(ya,1),'polin2: m')
            ! ndum=assert_eq(size(x2a),size(ya,2),'polin2: ndum')
            m = size(x1a)
            ndum = size(x2a)
            Do j = 1, m                                                                     !< Loop over rows.
                yntmp = ya(j, :)                                                            !< Copy row into temporary storage.
                call InterPol1D(x2a, yntmp, x2, ymtmp(j), dy)                               !< Interpolate answer into temporary storage.
            end Do
            call InterPol1D(x1a, ymtmp, x1, y, dy)                                          !< Do the final interpolation.

            ! Debug:
            ! print*,"y, dy = ", y, dy


            !< we're done
            return
        end subroutine InterPol2D


        !>************************************************************************************************************************************************
        !> subroutine: LineProfile
        !>
        !> calculates the intensity for a given optical depth using the user-defined line profile
        !>
        !>
        !> input variables:     l:                      frequency range index
        !>                      c:                      current component index
        !>                      freq:                   current frequency (MHz)
        !>                      nu_t:                   current transition frequency (MHz)
        !>                      v_off:                  current velocity offset (km/s)
        !>                      v_width_1:              velocity width 1 (km/s)
        !>
        !> output variable:     phi_nu:                 calc. intensity at current frequency
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2017-12-08
        !>
        subroutine LineProfile(phi_nu, l, c, freq, nu_t, v_off, v_width_1)

            implicit none
            integer :: l                                                                    !< current freq. range index
            integer :: c                                                                    !< current component index
            real*8 :: phi_nu                                                                !< calc. intensity at current frequency
            real*8 :: freq                                                                  !< current frequency (MHz)
            real*8 :: nu_t                                                                  !< current transition frequency (MHz)
            real*8 :: v_width_1                                                             !< current velocity width 1 (km/s)
            real*8 :: v_off                                                                 !< current velocity offset (km/s)
            real*8 :: FWHM1                                                                 !< full width at half maximum
            real*8 :: nu_doppler                                                            !< doppler shifted trans. frequency


            !<============================================================================================================================================
            !< line profile function
            phi_nu = 1.d0


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< Gaussian line profile
            !<
            !< Furthermore, the profile of a spectral line $t$ is given by a Gaussian
            !< line profile
            !<
            !<     \phi(\nu)^{m,c} = \frac{1}{\sqrt{2 \pi} \, \sigma}
            !<                       \cdot e^{-\frac{\left(\nu-\left(\nu_t + \nu_{\rm LSR}^{m,c} \right) \right)^2}{2 \sigma^2}},
            !<
            !< which is normalized, i.e.\ $\int_0^{\infty} \phi(\nu) \, d\nu$ = 1. The source velocity $\nu_{\rm LSR}^{m,c}$ for
            !< each component $c$ of a molecule $m$ is related to the user defined velocity offset
            !< $\left(v_{\rm offset}^{m,c}\right)$ taken from the aforementioned molfit file, by the following expression
            !<
            !<     \nu_{\rm LSR}^{m,c} = -v_{\rm offset}^{m,c} \cdot \nu_t,
            !<
            !< where $\nu_t$ indicates the frequency of transition $t$ taken from the SQLite3 database mentioned above.
            !< Additionally, the variance $\sigma$ of the profile is defined by the velocity width
            !< $\left(v_{\rm width}^{m,c}\right)$ described in the molfit file for each component $c$ of a molecule $m$:
            !<
            !<     \sigma = \frac{v_{\rm width}^{m,c} \cdot \left(\nu_t + \nu_{\rm LSR}^{m,c} \right)}{2 \, \sqrt{2 \, \ln 2}}.
            !<
            if (LineProfileFunction(c) == 1) then



                !< determine \nu_{\rm LSR}^{m,c}:
                !<
                !<     \nu_{\rm LSR}^{m,c} = -v_{\rm offset}^{m,c} \cdot \nu_t,
                !<
                !vLSR = -(v_off + GlobalvLSR(l)) / ckms * nu_t

                ! Debug:
                !    if (dabs(freq - debug_freq) < 1.d-1) write(DebugChannel, *) 'vLSR = ', vLSR


                !< determine \sigma: (Doppler width)
                !<
                !<     \sigma = \frac{v_{\rm width}^{m,c} \cdot \left(\nu_t + \nu_{\rm LSR}^{m,c}\right)}{2 \, \sqrt{2 \, \ln 2}}.
                !<
                !sigma = (v_width_1 / ckms * (nu_t + vLSR)) / (2.d0 * dsqrt(2.d0 * dlog(2.d0)))

                ! Debug:
                !    if (dabs(freq - debug_freqLocal) < 1.d0) write(DebugChannel, *) 'sigma = ', sigma


                !< determine \phi(\nu)^{m,c}:
                !<
                !<     \phi(\nu)^{m,c} = e^{-\frac{\left(\nu - \left(\nu_t + \nu_{\rm LSR}^{m,c}\right)\right)^2} {2 \sigma^2}},
                !<
                !phi_nu = ExpSave(-((freq - (nu_t + vLSR))**2) / (2.d0 * sigma**2))


                !< normalize profile function if selected
                !<     \phi(\nu)^{m,c} = \frac{1}{\sqrt{2 \pi} \, \sigma} \cdot \phi(\nu)^{m,c}
                !phi_nu = (1.d0 / (dsqrt(2.d0 * pi) * sigma * 1.d6)) * phi_nu


                !< use new Gaussian line profile function
                nu_doppler = nu_t * (1.d0 - (v_off + GlobalvLSR(l)) / ckms)
                if (v_width_1 >= 0.d0) then
                    FWHM1 = v_width_1 / ckms * nu_doppler
                else
                    FWHM1 = dabs(v_width_1)
                endif
                phi_nu = GaussianFunction(freq, nu_doppler, FWHM1)

            endif

            ! Debug:
            ! print*,"phi_nu = ", phi_nu


            !< we're done
            return
        end subroutine LineProfile


        !>************************************************************************************************************************************************
        !> subroutine: HeapSort
        !>
        !> sort an array using heapsort algorithm and apply changes to second array
        !>
        !>
        !> input variables:     n:                  length of array a
        !>                      k:                  length of array CompArray
        !>                      a:                  1st array, which should be sort
        !>                      CompArray:          2nd array, which should be sort
        !>
        !> output variables:    a:                  1st modified array a
        !>                      CompArray:          2nd modified array CompArray
        !>
        !>
        !> \author https://rosettacode.org/wiki/Sorting_algorithms/Heapsort#FreeBASIC, Numerical Recepie, Thomas Moeller
        !>
        !> \date 2018-01-22
        !>
        subroutine HeapSort(n, k, a, CompArray)

            implicit none
            integer, intent(in) :: n                                                        !< length of array a
            integer, intent(in) :: k                                                        !< length of array CompArray
            integer :: i                                                                    !< loop variable
            integer, dimension(k) :: tempInt                                                !< working variable for value swapping
            integer, dimension(n, k), intent(in out) :: CompArray                           !< 1st array, which should be sort
            real*8 :: temp                                                                  !< working variable for value swapping
            real*8, dimension(n), intent(in out) :: a                                       !< 2nd array, which should be sort

            Do i = n / 2, 1, (-1)
                call siftdown(n, k, a, CompArray, i, n)
            end Do

            Do i = n, 2, (-1)
                temp = a(1)
                tempInt(:) = CompArray(1, :)
                a(1) = a(i)
                CompArray(1, :) = CompArray(i, :)
                a(i) = temp
                CompArray(i, :) = tempInt(:)
                call siftdown(n, k, a, CompArray, 1, i - 1)
            end Do
            return
        end subroutine HeapSort


        !>************************************************************************************************************************************************
        !> subroutine: siftdown
        !>
        !> helper subroutine for heapsort algorithm
        !>
        !>
        !> input variables:     n:                  length of array a
        !>                      k:                  length of array CompArray
        !>                      a:                  1st array, which should be sort
        !>                      CompArray:          2nd array, which should be sort
        !>                      l:
        !>                      r:
        !>
        !> output variables:    a:                  1st modified array a
        !>                      CompArray:          2nd array, which should be sort
        !>
        !>
        !> \author https://rosettacode.org/wiki/Sorting_algorithms/Heapsort#FreeBASIC, Numerical Recepie, Thomas Moeller
        !>
        !> \date 2018-01-22
        !>
        subroutine siftdown(n, k, a, CompArray, l, r)

            implicit none
            integer, intent(in) :: n                                                        !< length of array a
            integer, intent(in) :: k                                                        !< length of array CompArray
            integer, intent(in) :: l, r                                                     !<
            integer :: j, jold                                                              !< working parameters
            integer, dimension(k) :: tempInt                                                !< working parameter
            integer, dimension(n, k), intent(in out) :: CompArray                           !< 2nd array, which should be sort
            real*8 :: temp                                                                  !< working parameter
            real*8, dimension(n), intent(in out) :: a                                       !< 1st array, which should be sort

            temp = a(l)
            tempInt(:) = CompArray(l, :)
            jold = l
            j = l + 1
            Do while (j <= r)
                if (j < r) then
                    if (a(j) < a(j + 1)) j = j + 1
                endif
                if (temp >= a(j)) exit
                a(jold) = a(j)
                CompArray(jold, :) = CompArray(j, :)
                jold = j
                j = j + j
            end Do
            a(jold) = temp
            CompArray(jold, :) = tempInt(:)
            return
        end subroutine siftdown


        !>************************************************************************************************************************************************
        !> subroutine: ComputeCentralOpticalDepthLTE
        !>
        !> calculates the optical depth at the given Doppler-shifted transition frequency for molecules (in LTE) and RRLs (in LTE and Non-LTE)
        !>
        !>
        !> input variables:     i:                                  total Doppler-shifted transition frequency counter
        !>                      c:                                  current component index
        !>                      t:                                  current transition index
        !>                      FreqIndex:                          frequency index
        !>                      LocalQ:                             local partition function
        !>                      LocalIsoRatio:                      local iso ratio
        !>                      Temp:                               kinetic temperature (molecules) or electronic temperature (RRLs)
        !>                      Ntot:                               column density (molecules) or emission measure (RRLs)
        !>                      Ne:                                 electron density
        !>
        !> output variable:     tau_t_em:                           optical depth for emission
        !>                      tau_t_ab:                           optical depth for absorption
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2018-01-26
        !>
        subroutine ComputeCentralOpticalDepthLTE(tau_t_em, tau_t_ab, c, t, FreqIndex, LocalQ, LocalIsoRatio, Temp, Ntot, Ne)

            implicit none
            integer :: FreqIndex                                                            !< frequency index
            integer :: c                                                                    !< component index
            integer :: t                                                                    !< transition index
            real*8 :: LocalQ                                                                !< local partition function
            real*8 :: LocalIsoRatio                                                         !< local iso ratio
            real*8 :: Temp                                                                  !< kinetic temperature (molecules) or electronic temp. (RRLs)
            real*8 :: Ntot                                                                  !< column density (molecules) or emission measure (RRLs)
            real*8 :: Ne                                                                    !< electron density
            real*8 :: tau_t_em                                                              !< working variable: tau for each transition
            real*8 :: tau_t_ab                                                              !< working variable: tau for each transition

            ! Debug:
            ! print*,"c = ", c
            ! print*,"t = ", t
            ! print*,"LocalQ = ", LocalQ
            ! print*,"LocalIsoRatio = ", LocalIsoRatio
            ! print*,"Temp = ", Temp
            ! print*,"Ntot = ", Ntot
            ! print*,"Ne = ", Ne


            !< computes optical depth at the Doppler-shifted transition frequencies
            tau_t_em = 0.d0
            tau_t_ab = 0.d0


            !< determine total column density (scale with iso ratio if defined)
            if (IsoFlag) then                                                               !< use isotopologues?
                Ntot = Ntot * LocalIsoRatio
            endif

            ! Debug:
            ! print*,"Temp, Ntot = ", Temp, Ntot


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !<
            !< CASE A:      Normal molecule
            !<
            !<
            !< determine part of eq. (5):
            !<
            !<      \frac{c^3}{8 \pi \nu^3} \, A_{u,l} \, N_{\rm tot}^{m,c}
            !<                              \frac{g_u  e^{-E_l/k_B T_{\rm ex}^{m,c}}}{Q \left(m, T_{\rm ex}^{m,c} \right)}
            !<                              \left(1 - e^{-h \, \nu_{u,l} /k_B \, T_{\rm ex}^{m,c}} \right)
            !<
            !< MolecularData(t, 1) =         \nu_{u,l}          (transition frequency)
            !< MolecularData(t, 2) =         A_{u,l}            (EinsteinA)
            !< MolecularData(t, 3) =         E_low              (energy of lower level in Kelvin)
            !< MolecularData(t, 4) =         g_u                (degeneracy of upper level)
            !< Temp = T_{\rm ex}^{m,c}   (kinetic temperature)
            !< myXCLASSParameter(2, c) =     N_{\rm tot}^{m,c}  (total column density)
            !<
            if (KindOfMolecule(c) == 0 .and. LTEFlag(c)) then                               !< normal molecule in LTE


                !< check for zero partition function
                if (LocalQ /= 0.d0) then
                    tau_t_em = (cms**2 / (8.d0 * pi * MolecularData(t, 1)**2)) * MolecularData(t, 2) * Ntot * MolecularData(t, 4) &
                                                                * (ExpSave(-MolecularData(t, 3) / Temp) / LocalQ) &
                                                                * (1.d0 - ExpSave(-(MHz2Kelvin * MolecularData(t, 1) / Temp))) &
                                                                * 1.d-8                     !<  conversion factor: [\nu] = MHz (10^{-6}),
                                                                                            !<                     [N_{\rm tot}^{m,c}] = cm-2 (10^{+4}),
                                                                                            !<                     [A_{u,l}] = s^(-1)
                                                                                            !<  tau = tau * 10^{-6} * 10^{-6} * 10^{+4}
                                                                                            !<      = 10^{-8} s^(-1) (phi is missing here)
                                                                                            !<      = 10^{-14} s^(-1) (with phi (in MHz^(-1)))
                    tau_t_ab = tau_t_em
                endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !<
            !< CASE B:      RRLs
            !<
            !<
            !< optical depth for recombination lines
            !<
            !<      \tau_l = \int \kappa_{n_1, n_2, \nu}^{\rm ext} \, ds
            !<             = \frac{\pi \, h^3 \, e^2}{(2 \pi \, m_e \, k_B)^{3/2} \, m_e \, c} \cdot {\rm EM}
            !<               \cdot \frac{n_1^2 \, f_{n_1, n_2}}{T_e^{3/2}} \, \exp \left[\frac{Z^2 \, E_{n_1}}{k_B T_e} \right]
            !<               \times \left(1 - e^{-h \nu / k_B T_e} \right) \, \phi_\nu.
            !<
            !< MolecularData(t, 1) =         \nu_{u,l}          (transition frequency)
            !< MolecularData(t, 2) =         n_1 * f_{n_1, n_2} (EinsteinA)
            !< MolecularData(t, 3) =         E_low              (energy of lower level in Kelvin)
            !< MolecularData(t, 4) =         g_u                (degeneracy of upper level)
            !< MolecularData(t, 5) =         Z                  (Z)
            !< Temp = T_{\rm ex}^{m,c}   (electron temperature)
            !< myXCLASSParameter(2, c) = EM^{m,c}           (total column density)
            !< myXCLASSParameter(3, c) = n_e^{m,c}          (electron density)
            !<
            elseif (KindOfMolecule(c) == 1 .or. KindOfMolecule(c) == 2) then


                !< define optical depth
                tau_t_em = 1.09911303617d-17 &                                              !< (pi * h**3 * e**2) / ((2.0 * pi * me * kB)**(1.5) * me * c)
                            * 3.08567758149137d18 &                                         !< conversion factor: pc^(-1) to cm^(-1)
                            * Ntot &                                                        !< EM^{m,c}
                            * MolecularData(t, 2) / Temp**(1.5) &
                            * ExpSave(dabs(MolecularData(t, 3)) / Temp) &
                            * (1.d0 - ExpSave(-(MHz2Kelvin * MolecularData(t, 5) * MolecularData(t, 1) / Temp))) &
                            * 1.d0

                ! Debug:
                ! print*," "
                ! print*,"t = ", t
                ! print*,"Temp = ", Temp
                ! print*,"MolecularData(t, 3) = ", MolecularData(t, 3)
                ! print*,"ExpSave(+dabs(MolecularData(t, 3)) / Temp) = ", ExpSave(+dabs(MolecularData(t, 3)) / Temp)

                tau_t_ab = tau_t_em

                ! Debug:
                ! print*,"tau_t_em = ", tau_t_em
                ! print*,"MolecularData(t, 1) = ", MolecularData(t, 1)
                ! print*,"Temp = ", Temp
            endif


            !< we're done
            return
        end subroutine ComputeCentralOpticalDepthLTE


        !>************************************************************************************************************************************************
        !> subroutine: DetectionEquation
        !>
        !> calculates the detection equation for a given frequency
        !>
        !>
        !> input variables:     tau_l_em:                       tau * J(T_ex)
        !>                      tau_l_ab:                       optical depth
        !>                      j_back:                         background temperature
        !>                      NumComp:                        total number of components
        !>                      freq:                           current frequency
        !>                      FreqIndex:                      index for frequency point
        !>                      l:                              frequency range index
        !>                      CurrDistID:                     distance index
        !>                      MoleculeIndex:                  current molecule index
        !>                      CopymyXCLASSParameter:          local copy of myXCLASSParameterOrig
        !>                      CopyIsoRatioConversionTable:    local copy of IsoRatioConversionTableOrig
        !>                      Q:                              partition function
        !>                      beam_size:                      beam size
        !>                      c:                              current component index
        !>                      ccc:                            component index of current layer
        !>                      TotalLocalIntensity:            total intensity (used for foreground)
        !>                      TotalNumMol:                    total number of molecules
        !>                      ModelFuncList:                  output variable: model function values
        !>                      etaMaxIndex:                    index of largest core component
        !>                      NumCompAtCurrDistance:          number of cores of first distance (i.e. CurrDistID == 1)
        !>                      LocalMaxNumTrans:               max. number of transitions (used for Non-LTE description)
        !>                      FirstFreqIndex:                 first frequency index
        !>                      max_v_width:                    max. width
        !>                      TauResetFlag:                   flag indicating that tau_em and tau_abs are cleared
        !>                      NumberMoleculeComponents:       number of component describing molecules/RRLs
        !>                      SameSizeFlag:                   flag indicating that all components at the current distance have the same source size
        !>
        !> output variable:     LocalIntensity:                 calc. intensity at current frequency
        !>                      tau_l_em:                       tau * J(T_ex)
        !>                      tau_l_ab:                       optical depth
        !>                      j_back:                         background temperature
        !>                      T_d:                            dust temperature
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2014-02-17
        !>
        subroutine DetectionEquation(LocalIntensity, tau_l_em, tau_l_ab, j_back, T_d, NumComp, freq, FreqIndex, l, &
                                     CurrDistID, MoleculeIndex, CopymyXCLASSParameter, CopyIsoRatioConversionTable, Q, &
                                     beam_size, c, ccc, TotalLocalIntensity, TotalNumMol, etaMaxIndex, &
                                     NumCompAtCurrDistance, LocalMaxNumTrans, FirstFreqIndex, &
                                     max_v_width, TauResetFlag, NumberMoleculeComponents, SameSizeFlag)

            implicit none
            integer :: t, MoleculeIndexLoop, wm, ic                                         !< loop variables
            integer :: l                                                                    !< frequency range index
            integer :: i                                                                    !< index for CentralOpticalDepth
            integer :: c, cc, ccc                                                           !< current component index
            integer :: CurrDistID                                                           !< distance index
            integer :: NumComp                                                              !< total number of components
            integer :: nn                                                                   !< local main quantum number for RRLs
            integer :: MoleculeIndex                                                        !< working variable: index for molecule
            integer :: FirstMolecularDataIndex                                              !< working variable: first index for molecular data table
            integer :: LastMolecularDataIndex                                               !< working variable: last index for molecular data table
            integer :: TotalNumMol                                                          !< total number of molecules
            integer :: ll, kk                                                               !< used for handling of isotoplogues
            integer :: tt, t1, t2                                                           !< used for local-overlap
            integer :: FirstIndex                                                           !< index for first freq. data point
            integer :: LastIndex                                                            !< index for last freq. data point
            integer :: FreqIndex                                                            !< index for frequency point
            integer :: FirstFreqIndex                                                       !< first frequency index
            integer :: LocalFreqIndex                                                       !< local index for frequency point
            integer :: IsoCounter                                                           !< working variable: counter for isotopologue
            integer :: etaMaxIndex                                                          !< index of emission comp. with the largest beam fill. factor
            integer :: NumCompAtCurrDistance                                                !< total number of emission components
            integer :: LocalMaxNumTrans                                                     !< max. number of transitions (used for Non-LTE description)
            integer :: NumContComp                                                          !< number of component describing continuum
            integer :: NumberMoleculeComponents                                             !< number of component describing molecules/RRLs
            real*8 :: eta_source                                                            !< working variable: beam filling factor
            real*8 :: freq                                                                  !< working variable: current frequency
            real*8 :: freq_t                                                                !< working variable: frequency as temperature
            real*8 :: beam_size                                                             !< working variable: beam size (eqn. 2 - 3)
            real*8 :: j_tk                                                                  !< working variables: Rayleigh temperatures
            real*8 :: SourceFunc                                                            !< working variable: source function
            real*8 :: Ibg                                                                   !< working variable: background intensity (from emission)
            real*8 :: Ntot                                                                  !< working variable: total column density
            real*8 :: Ne                                                                    !< electron density
            real*8 :: phi_nu                                                                !< working variable: line profile
            real*8 :: nu_t                                                                  !< current transition frequency (MHz)
            real*8 :: v_width_1                                                             !< current velocity width 1 (km/s)
            real*8 :: max_v_width                                                           !< max. width
            real*8 :: v_off                                                                 !< current velocity offset (km/s)
            real*8 :: DeltaFreq                                                             !< frequency delta for trans. freq. selection
            real*8 :: val                                                                   !< working variable: val (for peak integration)
            real*8 :: LocalQ                                                                !< local partition function
            real*8 :: LocalIsoRatio                                                         !< local iso ratio
            real*8 :: j_td                                                                  !< working variable: Planck function for T_d
            real*8 :: T_d                                                                   !< dust temperature
            real*8 :: j_back                                                                !< brightness for background temperature
            real*8 :: j_back_full                                                           !< total brightness for background temperature
            real*8 :: j_cb                                                                  !< brightness for cosmic microwave background temperature
            real*8 :: tau_d                                                                 !< working variable: optical depth dust
            real*8 :: tau_bf                                                                !< working variable: optical depth bound-free contribution
            real*8 :: tau_t_em                                                              !< working variable: tau for each transition
            real*8 :: tau_t_ab                                                              !< working variable: tau for each transition
            real*8 :: tau_l_em                                                              !< working variable: sum over all taus
            real*8 :: tau_l_ab                                                              !< working variable: sum over all taus
            real*8 :: tau_total                                                             !< working variable: total optical depth
            real*8 :: LocalnH                                                               !< working variable: hydrogen column density for dust
            real*8 :: LocalKappa                                                            !< working variable: kappa for dust
            real*8 :: LocalBeta                                                             !< working variable: spectral index for dust
            real*8 :: LocalRefFreq                                                          !< working variable: rest frequency
            real*8 :: Temp                                                                  !< working variable: electron temperature
            real*8 :: LocalIbg                                                              !< working variable: local scaled background intensity
            real*8 :: LocalIntensity                                                        !< working variable: calc. intensity at current data point
            real*8 :: TotalLocalIntensity                                                   !< total intensity (used for foreground)
            real*8, dimension(NumComp, TotalNumMol) :: Q                                    !< partition function at the rotation temperatures
            real*8, dimension(16, NumComp) :: CopymyXCLASSParameter                         !< working variable: local copy of myXCLASSParameterOrig
            real*8, dimension(TotalNumMol, TotalNumMol) :: CopyIsoRatioConversionTable      !< local copy of IsoRatioConversionTableOrig
            logical :: CalcFalg                                                             !< used for calculation of isotopoluges
            logical :: DustFlag                                                             !< flag for dust continuum contribution
            logical :: LocalContFlag                                                        !< flag indicating that current componet may contain
                                                                                            !< continuum contribution
            logical :: TauResetFlag                                                         !< flag indicating that tau_em and tau_abs are cleared
            logical :: SameSizeFlag                                                         !< flag indicating that all components at the current
                                                                                            !< distance have the same source size
            ! Debug:
            ! print*," "
            ! print*,"NumComp = ", NumComp
            ! print*,"freq = ", freq
            ! print*,"FreqIndex = ", FreqIndex
            ! print*,"l = ", l
            ! print*,"CurrDistID = ", CurrDistID
            ! print*,"MoleculeIndex = ", MoleculeIndex
            ! print*,"CopymyXCLASSParameter = ", CopymyXCLASSParameter(:,c)
            ! print*,"CopyIsoRatioConversionTable = ", CopyIsoRatioConversionTable(1,:)
            ! print*,"Q = ", Q(c,:)
            ! print*,"beam_size = ", beam_size
            ! print*,"c = ", c
            ! print*,"ccc = ", ccc
            ! print*,"TotalLocalIntensity = ", TotalLocalIntensity
            ! print*,"TotalNumMol = ", TotalNumMol
            ! print*,"etaMaxIndex = ", etaMaxIndex
            ! print*,"NumCompAtCurrDistance = ", NumCompAtCurrDistance
            ! print*,"FirstFreqIndex = ", FirstFreqIndex
            ! print*,"max_v_width = ", max_v_width
            ! print*,"TauResetFlag = ", TauResetFlag
            ! print*,"SameSizeFlag = ", SameSizeFlag


            !< if molecules or RRLs have no transition within the current frequency range, do nothing
            ! if (KindOfMolecule(c) == 6 .or. (KindOfMolecule(c) <= 2 .and. TotalNumberOfTransitions == 0) then
            if (KindOfMolecule(c) <= 2 .and. TotalNumberOfTransitions == 0) then
                return
            endif


            !< initialize some variables
            LocalIntensity = 0.d0                                                           !< initialize local intensity variable
            DustFlag = .false.                                                              !< initialize DustFlag variable
            tau_d = 0.d0                                                                    !< initialize tau_d variable
            T_d = 0.d0                                                                      !< initialize T_d variable
            j_td = 0.d0                                                                     !< initialize j_td variable
            j_tk = 0.d0                                                                     !< initialize j_tk variable
            LocalRefFreq = 0.d0                                                             !< initialize LocalRefFreq variable
            tau_bf = 0.d0                                                                   !< initialize tau_bf variable
            LocalIbg = 0.d0                                                                 !< initialize LocalIbg variable
            SourceFunc = 0.d0                                                               !< initialize S variable
            Ibg = 0.d0                                                                      !< initialize Ibg variable
            wm = 0                                                                          !< initialize wm variable
            IsoCounter = 0                                                                  !< initialize IsoCounter variable
            eta_source = 0.d0                                                               !< initialize eta_source variable
            NumContComp = 0                                                                 !< initialize NumContComp variable


            !< if local-overlap is not taken into account, check, if optical depth arrays tau_em and tau_abs have to be reseted
            if (TauResetFlag) then
                tau_l_em = 0.d0                                                             !< initialize tau_l_em variable
                tau_l_ab = 0.d0                                                             !< initialize tau_l_ab variable
                if (UseEmAbsFuncFlag) then
                    tau_l_em = EmsAbsFunc(FreqIndex, CurrDistID, 1)
                    tau_l_ab = EmsAbsFunc(FreqIndex, CurrDistID, 2)
                endif
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get some molfit parameters
            !<
            !< CopymyXCLASSParameter(1, c)  T_rot (=0), T_e (=1),    T_e (=2),      T_cont_dust (=3),        - (=4),     - (=5)
            !< CopymyXCLASSParameter(2, c)  N_tot (=0), EM_RRL (=1), EM_RRL (=2),   nHcolumn_cont_dust (=3), - (=4),     - (=5)
            !< CopymyXCLASSParameter(3, c)  - (=0),     - (=1),      N_e (=2),      kappa_cont_dust (=3),    - (=4),     - (=5)
            !< CopymyXCLASSParameter(4, c)  V_width_Gauss,                          beta_cont_dust (=3),     - (=4),     - (=5)
            !< CopymyXCLASSParameter(5, c)  -                                       - (=3),                  - (=4),     - (=5)
            !< CopymyXCLASSParameter(6, c)  V_Off (=0,1,2)                          - (=3),                  - (=4),     - (=5)
            !< CopymyXCLASSParameter(7, c)  T_dOff
            !< CopymyXCLASSParameter(8, c)  T_dSlope
            !< CopymyXCLASSParameter(9, c)  nHcolumn
            !< CopymyXCLASSParameter(10, c) kappa
            !< CopymyXCLASSParameter(11, c) beta
            !< CopymyXCLASSParameter(12, c) LayerDistance (=0,1,2,3,4,5,6)
            !< CopymyXCLASSParameter(13, c) CFFlag (=0,1,2,3,4,5,6)
            !< CopymyXCLASSParameter(14, c) source size, source radius
            !< CopymyXCLASSParameter(15, c) source_center_x
            !< CopymyXCLASSParameter(16, c) source_center_y
            !<
            Temp = CopymyXCLASSParameter(1, c)                                              !< get temperature (K)
            v_width_1 = CopymyXCLASSParameter(4, c)                                         !< get velocity width (km/s) (Gauss)
            v_off = CopymyXCLASSParameter(6, c)                                             !< get velocity offset (km/s)


            !<############################################################################################################################################
            !< Only used for storing opacities and intensities
            !<############################################################################################################################################
            !< initialize working variables
            if (AllOutputFilesFlag) then
                IntegHelperArray1 = 0.d0
                IntegHelperArray2 = 0.d0
                TauHelperArrayEM = 0.d0
                TauHelperArrayABS = 0.d0
                TauHelperArray2 = 0.d0
                LocalIntArray = 0.d0
            endif
            !<############################################################################################################################################
            !<############################################################################################################################################


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine Rayleight-Jeans temperatures
            freq_t = MHz2Kelvin * freq                                                      !< convert freq. in temperature (Kelvin) = \frac{h \nu}{k_B}


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine beam filling (dilution) factor (eq. 2):
            !< \eta(\theta_{m,c}) of molecule m and component c for a source with a Gaussian brightness profile, and a Gaussian
            !< beam is given by
            !<
            !<     \eta(\theta_{m,c}) = \frac{\theta_{m,c}^2}{\theta_{m,c}^2 + \theta_t^2}
            !<
            !< where \theta_{m,c} and \theta_t represents the source and telescope beam FWHM sizes, respectively.
            !<
            if (LocalOverlapFlag .and. SameSizeFlag) then
                eta_source = CopymyXCLASSParameter(14, c)**2 / (beam_size**2 + CopymyXCLASSParameter(14, c)**2)
            ! elseif (LocalOverlapFlag .or. SubBeamDescriptionFlag .or. dabs(CopymyXCLASSParameter(14, c)) < tiny(1.d0)) then
            elseif (SubBeamDescriptionFlag .or. dabs(CopymyXCLASSParameter(14, c)) < tiny(1.d0)) then
                eta_source = 1.d0
                ! eta_source = SizeOfPixel_deg**2 / (beam_size**2 + SizeOfPixel_deg**2)
            else
                eta_source = CopymyXCLASSParameter(14, c)**2 / (beam_size**2 + CopymyXCLASSParameter(14, c)**2)
            endif

            ! Debug:
            ! print*," "
            ! print*,"SubBeamDescriptionFlag = ", SubBeamDescriptionFlag
            ! print*,"LocalOverlapFlag = ", LocalOverlapFlag
            ! print*,"SameSizeFlag = ", SameSizeFlag
            ! print*,"eta_source = ", eta_source


            !<============================================================================================================================================
            !< continuum contribution


            !< define local continuum flag
            LocalContFlag = .false.
            if ((LocalOverlapFlag .and. ccc == NumCompAtCurrDistance) .or. ((.not. LocalOverlapFlag ) &
                .and. (c == etaMaxIndex))) then
                LocalContFlag = .true.
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !<
            !< Compute optical depth for DUST continuum
            !<
            !< CopymyXCLASSParameter(2, c) =    N_H^{m,c}
            !< CopymyXCLASSParameter(3, c) =    \kappa_{\rm 1.3 \, mm}^{m,c}   (contains factor (m_{H_2} * \frac{1}{\zeta_{\rm gas-dust}}) as well)
            !< CopymyXCLASSParameter(4, c) =    \beta^{m,c}
            !< or
            !< CopymyXCLASSParameter(9, c) =   N_H^{m,c}
            !< CopymyXCLASSParameter(10, c) =   \kappa_{\rm 1.3 \, mm}^{m,c}   (contains factor (m_{H_2} * \frac{1}{\zeta_{\rm gas-dust}}) as well)
            !< CopymyXCLASSParameter(11, c) =   \beta^{m,c}
            !<
            DustFlag = .false.


            !< dust continuum is described in molfit file by current component
            if (KindOfMolecule(c) == 3) then                                                !< we are dealing with dust continuum
                T_d = Temp
                LocalnH = CopymyXCLASSParameter(2, c)
                LocalKappa = CopymyXCLASSParameter(3, c)
                LocalBeta = CopymyXCLASSParameter(4, c)
                LocalRefFreq = 2.3d5                                                        !< future release: LocalRefFreq = CopymyXCLASSParameter(5, c)
                DustFlag = .true.


            !< dust continuum is described in molfit file in addition to a molecule component
            elseif (tdFlag(c)) then                                                         !< for in-line dust description
                if (CopymyXCLASSParameter(8, c) < -1.d3) then                               !< for a extreme spectral index, use absolute temp. definition
                    T_d = CopymyXCLASSParameter(7, c)
                else
                    T_d = Temp + (CopymyXCLASSParameter(7, c) * (freq / StartFrequency(l))**CopymyXCLASSParameter(8, c))
                endif
                if (nHFlag(c)) then
                    LocalnH = CopymyXCLASSParameter(9, c)
                    LocalKappa = CopymyXCLASSParameter(10, c)
                    LocalBeta = CopymyXCLASSParameter(11, c)
                else
                    LocalnH = HydrogenColumnDensityRange(l)
                    LocalKappa = KappaRange(l)
                    LocalBeta = DustBetaRange(l)
                endif
                LocalRefFreq = 2.3d5
                DustFlag = .true.


            !< dust continuum is defined in obs. xml file
            ! elseif (Dust_Flag_Range(l) .and. LocalContFlag) then                          !< actually, the correct handling
            elseif (Dust_Flag_Range(l) .and. LocalContFlag .and. CurrDistID == 1) then      !< modification to be backward compatible
                T_d = Temp
                LocalnH = HydrogenColumnDensityRange(l)
                LocalKappa = KappaRange(l)
                LocalBeta = DustBetaRange(l)
                LocalRefFreq = 2.3d5
                DustFlag = .true.
            endif


            !< consider dust contribution
            if (DustFlag) then

                ! Debug:
                ! print*," "
                ! print*," "
                ! print*,"freq = ", freq
                ! print*,"CurrDistID = ", CurrDistID
                ! print*,"c = ", c
                ! print*,"ccc = ", ccc
                ! print*,"T_d = ", T_d
                ! print*,"LocalnH = ", LocalnH
                ! print*,"LocalKappa = ", LocalKappa
                ! print*,"LocalBeta = ", LocalBeta
                ! print*,"FreqIndex, DustTauFromFile(FreqIndex) = ", FreqIndex, DustTauFromFile(FreqIndex)
                ! print*,"Dust_Flag_Range(l) = ", Dust_Flag_Range(l)


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< compute optical depth of dust
                !<
                !<     \tau_d^{m,c} = N_H^{m,c} * \kappa_{\rm 1.3 \, mm}^{m,c} * (\frac{\nu}{\nu_{\rm 1.3 \, mm}} )^\beta^{m,c}
                !<                              * m_{H_2} * \frac{1}{\zeta_{\rm gas-dust}},
                !<
                !< where the hydrogen column density $N_H$, the dust mass opacity $\kappa_{\rm 1.3 \, mm}$, and the exponent $\beta$.
                !< In addition, $\nu_{\rm 1.3 \, mm}$ = 230~GHz indicates the frequency for a wavelength of 1.3~mm, $m_{H_2}$
                !< describes the mass of a hydrogen molecule, and $\zeta_{\rm gas-dust}$ represents the gas to dust mass ratio,
                !< i.e.\ $1 / \zeta_{\rm gas-dust}$ describes the dust to gas ratio and is set to (1/100).
                !<
                LocalKappa = LocalKappa * (2.d0 * 1.66d-24 / 100.d0)
                LocalKappa = LocalKappa * (2.d0 * 1.66d-24 / 100.d0)
                tau_d = LocalnH * LocalKappa * (freq / LocalRefFreq)**LocalBeta + DustTauFromFile(FreqIndex)

                ! Debug:
                ! print*,"tau_d = ", tau_d


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< determine brightness for dust temperature
                j_td = 0.d0
                if (T_d /= 0.d0) then
                    if (dabs((ExpSave(freq_t / T_d) - 1.d0)) > tiny(1.d0)) then
                        j_td = freq_t / (ExpSave(freq_t / T_d) - 1.d0)
                    endif
                endif

                ! Debug:
                ! print*,"j_td = ", j_td


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< compute contribution to source function
                tau_l_em = tau_l_em + tau_d * j_td
                tau_l_ab = tau_l_ab + tau_d


                !<########################################################################################################################################
                !< Only used for storing opacities and intensities
                !<########################################################################################################################################
                !< store emission and absorption function
                if (AllOutputFilesFlag .and. LocalContFlag) then
                    TauHelperArrayEM(1) = TauHelperArrayEM(1) + tau_d * j_td
                    TauHelperArrayABS(1) = TauHelperArrayABS(1) + tau_d
                endif
                !<########################################################################################################################################
                !<########################################################################################################################################
            endif

            ! Debug:
            ! print*,"DustFlag = ", DustFlag


            !<============================================================================================================================================
            !< take local-overlap into account
            if (LocalOverlapFlag .and. ccc == NumCompAtCurrDistance) then


                !< determine delta frequency
                DeltaFreq = 5.d1 * dabs(freq * (1.d0 - max_v_width / ckms) - freq)          !< assume frequency range for trans. frequencies


                !< get optical depth of transition which may contribute at the current frequency
                t1 = minloc(dabs((freq - DeltaFreq) - DopplerShiftedTransFreq(:)), dim = 1)
                t2 = minloc(dabs((freq + DeltaFreq) - DopplerShiftedTransFreq(:)), dim = 1)
                wm = 0                                                                      !< initialize wm variable

                ! Debug:
                ! print*," "
                ! print*,"freq = ", freq
                ! print*,"CurrDistID, DeltaFreq, t1, t2 = ", CurrDistID, DeltaFreq, t1, t2, DopplerShiftedTransFreq(t1:t2)


                !< loop over some Doppler-shifted transition frequencies
                Do tt = t1, t2


                    !< get coefficients from DopplerShiftedTransFreqIndex(i, :) = (/l, MoleculeIndex, MoleculeIndexLoop, c, t, i/)
                    cc = DopplerShiftedTransFreqIndex(tt, 4)                                !< get component index
                    if (cc > 0) then                                                        !< make sure, that comp. index is real
                        ll = DopplerShiftedTransFreqIndex(tt, 1)                            !< get range index
                        t = DopplerShiftedTransFreqIndex(tt, 5)                             !< get transition index
                        i = DopplerShiftedTransFreqIndex(tt, 6)                             !< get index for CentralOpticalDepth
                        wm = wm + 1

                        ! Debug:
                        ! print*," "
                        ! print*,"cc = ", cc
                        ! print*,"ll = ", ll
                        ! print*,"t = ", t
                        ! print*,"i = ", i
                        ! print*,"LTEFlag(cc) = ", LTEFlag(cc)


                        !< get some parameters for current transition
                        nu_t = MolecularData(t, 1)
                        if (nu_t > 0.d0) then
                            v_width_1 = CopymyXCLASSParameter(4, cc)                        !< get velocity width (km/s) (Gauss)
                            v_off = CopymyXCLASSParameter(6, cc)                            !< get velocity offset (km/s)

                            ! Debug:
                            ! print*,"i, t, CentralOpticalDepth(i, :) = ", i, t, CentralOpticalDepth(i, :)


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< line profile function
                            phi_nu = 1.d0
                            call LineProfile(phi_nu, ll, cc, freq, nu_t, v_off, v_width_1)


                            !< apply line profile function to tau
                            tau_t_em = CentralOpticalDepth(i, 1) * phi_nu


                            !< for LTE: CentralOpticalDepth(i, :) = (/tau_t_em, tau_t_ab, MolecularData(t, 1)/)
                            if (LTEFlag(cc)) then
                                Temp = CopymyXCLASSParameter(1, cc)                         !< get kinetic temperature (K)
                                tau_t_ab = CentralOpticalDepth(i, 2) * phi_nu


                            !< for Non-LTE: CentralOpticalDepth(i, :) = (/tau_t_em, tex(c, tt), nu_t/)
                            else
                                Temp = CentralOpticalDepth(i, 2)                            !< get excitation temperature (K)
                                tau_t_ab = tau_t_em
                            endif

                            ! Debug:
                            ! print*," "
                            ! print*,"tt, t1, t2 = ", tt, t1, t2
                            ! print*,"freq = ", freq
                            ! print*,"ccc, DopplerShiftedTransFreqIndex(tt, :) = ", ccc, DopplerShiftedTransFreqIndex(tt, :)
                            ! print*,"CurrDistID = ", CurrDistID
                            ! print*,"i = ", i
                            ! print*,"ll, cc, freq, nu_t, v_off, v_width_1 = ", ll, cc, freq, nu_t, v_off, v_width_1
                            ! print*,"phi_nu = ", phi_nu
                            ! print*,"nu_t = ", nu_t
                            ! print*,"Temp = ", Temp
                            ! print*,"v_width_1 = ", v_width_1
                            ! print*,"v_off = ", v_off
                            ! print*,"CentralOpticalDepth(i, 1) = ", CentralOpticalDepth(i, 1)
                            ! print*,"tau_t_em = ", tau_t_em
                            ! print*,"tau_t_ab = ", tau_t_ab


                            !<------------------------------....------------------------------------------------------------------------------------------
                            !< determine Rayleight-Jeans temperatures
                            j_tk = 1.d0
                            if (dabs(Temp) > 1.d-6) then                                    !< prevent underflow
                                j_tk = freq_t / (ExpSave(freq_t / Temp) - 1.d0)
                            endif

                            ! Debug:
                            ! print*,"j_tk = ", j_tk


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< add tau for current transition to sum of taus
                            tau_l_em = tau_l_em + (tau_t_em * j_tk)
                            tau_l_ab = tau_l_ab + tau_t_ab


                            !<############################################################################################################################
                            !< Only used for storing opacities and intensities
                            !<############################################################################################################################
                            !< store total opacity and emission and absorption function
                            ! if (AllOutputFilesFlag .and. LocalContFlag) then
                            if (AllOutputFilesFlag) then
                                IntegHelperArray1(1, 1) = tau_t_ab
                                TauHelperArrayEM(1) = TauHelperArrayEM(1) + tau_t_em * j_tk
                                TauHelperArrayABS(1) = TauHelperArrayABS(1) + tau_t_ab
                            endif
                            !<############################################################################################################################
                            !<############################################################################################################################
                        endif
                    endif
                end Do                                                                      !< tt: loop over some Doppler-shifted transition frequencies


            !<============================================================================================================================================
            !< do not take local-overlap into account
            elseif (.not. LocalOverlapFlag) then
                j_tk = freq_t / (ExpSave(freq_t / Temp) - 1.d0)


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< determine optical depth $\tau(\nu)^{m,c}$ for each molecule m and component c
                !<
                !<     \tau(\nu)^{m,c} = \sum_t \frac{c^3}{8 \pi \nu^3} \, A_{ul} \, N_{\rm tot}^{m,c}
                !<                              \frac{g_u  e^{-E_l/k_B T_{\rm ex}^{m,c}}}{Q \left(m, T_{\rm ex}^{m,c} \right)}
                !<                              \left(1 - e^{-h \, \nu /k_B \, T_{\rm ex}^{m,c}} \right)
                !<                              \times \phi(\nu)^{m,c},
                !<
                !< where the sum with index $t$ runs over all spectral line transitions of molecule $m$ within the given frequency
                !< range. Additionally, the Einstein $A_{ul}$ coefficient\footnote{The indices $u$ and $l$ represent upper and lower
                !< state, respectively.}, the energy of the lower state $E_l$, the upper state degeneracy $g_u$, and the
                !< partition function $Q \left(m, T_{\rm ex}^{m,c} \right)$ of molecule $m$ are taken from the embedded
                !< SQLite3 database, described in  Sect.~\ref{sec:db}. In addition, the values of the kinetic temperatures
                !< $T_{\rm ex}^{m,c}$ and the column densities $N_{\rm tot}^{m,c}$ for the different components and molecules are
                !< taken from the user defined molfit file.
                !<
                !< if isotopologues are used, each component has to be calculated for all isotopoluges of the current molecule as well
                !< so a further loop is necessary which runs over all molecule indices of the isotopologues related to the current
                !< molecule
                ll = MoleculeIndex                                                          !< get molecule index
                if (IsoFlag) then                                                           !< use isotopologues?
                    kk = TotalNumMol
                else
                    kk = ll
                endif
                Do MoleculeIndexLoop = ll, kk                                               !< loop over all molecule indices (including isotopologues!)
                    CalcFalg = .true.
                    if (IsoFlag) then                                                       !< use isotopologues?
                        if (CopyIsoRatioConversionTable(ll, MoleculeIndexLoop) == 0.d0) then
                            CalcFalg = .false.
                        endif
                    endif
                    if (CalcFalg) then                                                      !< consider current molecule
                        IsoCounter = IsoCounter + 1


                        !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        !< compute optical depth in LTE
                        if (LTEFlag(c)) then


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< perform sum over all transition frequencies
                            wm = 0                                                          !< initialize wm variable
                            FirstMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 1) !< get first mol. data index of current freq. range
                            LastMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 2)  !< get last mol. data index of current freq. range
                            if (FirstMolecularDataIndex > 0) then


                                !< determine delta frequency
                                ! DeltaFreq = 5.d0 * dabs(freq * (1.d0 - max_v_width / ckms) - freq)      !< assume frequency range for trans. frequencies


                                !< get optical depth of transition which may contribute at the current frequency
                                ! t1 = minloc(dabs((freq - DeltaFreq) - MolecularData(FirstMolecularDataIndex:LastMolecularDataIndex, 1)), dim = 1)
                                ! t1 = FirstMolecularDataIndex + t1
                                ! t2 = minloc(dabs((freq + DeltaFreq) - MolecularData(FirstMolecularDataIndex:LastMolecularDataIndex, 1)), dim = 1)
                                ! t2 = FirstMolecularDataIndex + t2

                                ! Debug:
                                ! print*,"FirstMolecularDataIndex, LastMolecularDataIndex = ", FirstMolecularDataIndex, LastMolecularDataIndex
                                ! print*,"DeltaFreq = ", DeltaFreq
                                ! print*,"max_v_width = ", max_v_width
                                ! print*,"t1, t2 = ", t1, t2


                                ! Do t = t1, t2
                                Do t = FirstMolecularDataIndex, LastMolecularDataIndex      !< loop over all transitions of current molecule and
                                    wm = wm + 1
                                    tau_t_em = 0.d0
                                    tau_t_ab = 0.d0

                                    !< MolecularData(t, 1) = lFreq
                                    !< MolecularData(t, 2) = EinsteinA
                                    !< MolecularData(t, 3) = lElt
                                    !< MolecularData(t, 4) = UpperStateDegeneracy


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< compute corresponding optical depth for moleucles (in LTE) and RRLs (in LTE and Non-LTE)
                                    LocalQ = Q(c, MoleculeIndexLoop)                        !< get partition function for molecule
                                    LocalIsoRatio = CopyIsoRatioConversionTable(ll, MoleculeIndexLoop)
                                    Ntot = CopymyXCLASSParameter(2, c)                      !< column density (molecule) or emission measure (RRLs)
                                    Ne = CopymyXCLASSParameter(3, c)                        !< electron density (RRLs)
                                    ic = FreqIndex - FirstFreqIndex + 1                     !< define relative frequency index
                                    call ComputeCentralOpticalDepthLTE(tau_t_em, tau_t_ab, c, t, FreqIndex, LocalQ, &
                                                                       LocalIsoRatio, Temp, Ntot, Ne)

                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< line profile function
                                    phi_nu = 1.d0
                                    nu_t = MolecularData(t, 1)
                                    call LineProfile(phi_nu, l, c, freq, nu_t, v_off, v_width_1)


                                    !< apply line profile function to tau
                                    tau_t_em = tau_t_em * phi_nu
                                    tau_t_ab = tau_t_ab * phi_nu

                                    ! Debug:
                                    ! print*,"tau_t_em = ", tau_t_em


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< add tau for current transition to sum of taus
                                    tau_l_em = tau_l_em + (tau_t_em * j_tk)
                                    tau_l_ab = tau_l_ab + tau_t_ab


                                    !<####################################################################################################################
                                    !< Only used for storing opacities and intensities
                                    !<####################################################################################################################
                                    !< store total opacity and emission and absorption function
                                    ! if (AllOutputFilesFlag .and. LocalContFlag) then
                                    if (AllOutputFilesFlag) then
                                        IntegHelperArray1(IsoCounter, wm) = tau_t_ab
                                        TauHelperArrayEM(IsoCounter) = TauHelperArrayEM(IsoCounter) + tau_t_em * j_tk
                                        TauHelperArrayABS(IsoCounter) = TauHelperArrayABS(IsoCounter) + tau_t_ab
                                    endif
                                    !<####################################################################################################################
                                    !<####################################################################################################################
                                end Do                                                      !< t: loop over transitions
                            endif                                                           !< continue here, if current molecule has no transition

                        endif
                    endif
                end Do                                                                      !< MoleculeIndexLoop: loop over isotopologues
            endif                                                                           !< handling of local overlap


            !<============================================================================================================================================
            !< continue here, only if local-overlap is not taken into account or if last component of current distance is considered
            if (ccc == NumCompAtCurrDistance .or. (.not. LocalOverlapFlag .and. KindOfMolecule(c) <= 2)) then


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< determine background temperature
                !<
                !< The background temperature $J(T_{\rm bg}, \nu)$ is parametrized as
                !<
                !<      I_{\rm bg} (\nu) = T_{\rm bg} * \left(\frac{\nu}{\nu_{\rm min}} \right)^{T_{\rm slope}} + J_\mathrm{CMB}
                !<
                !< to allow the user to define the continuum contribution for each frequency range, individually. Here, $\delta_{T_{\rm bg}, 0}$
                !< represents the Kronecker delta, i.e.\ $I_{\rm bg} (\nu) \equiv J_\mathrm{CMB}$ for $T_{\rm bg} \equiv 0$. Additionally,
                !< $\nu_{\rm min}$ indicates the lowest frequency of a given frequency range. $T_{\rm bg}$ and $T_{\rm slope}$, defined by the
                !< user, describe the background continuum temperature and the temperature slope, respectively. The treatment of the dust is
                !< not entirely self-consistent. To amend that, we would need to define a more precise source model, which we consider to be
                !< outside the scope of this effort.
                !<
                j_cb = freq_t / (ExpSave(freq_t / Tcbg) - 1.d0)
                j_back_full = j_back + j_cb
                if (dabs(BackgroundTemperatureRange(l)) > tiny(1.d0)) then
                    j_back_full = j_back_full + BackgroundTemperatureRange(l) &
                                                * (freq / StartFrequency(l))**TemperatureSlopeRange(l)
                endif

                ! Debug:
                ! print*,"j_cb = ", j_cb
                ! print*,"j_back = ", j_back
                ! print*,"j_back_full = ", j_back_full
                ! print*,"StartFrequency(l) = ", StartFrequency(l)
                ! print*,"BackgroundTemperatureRange(l) = ", BackgroundTemperatureRange(l)
                ! print*,"TemperatureSlopeRange(l) = ", TemperatureSlopeRange(l)


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< define total optical depth \tau_{\rm total}, Eq. (7)
                tau_total = tau_l_ab

                ! Debug:
                ! print*,"freq, tau_total = ", freq, tau_total


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< Calculate source function using eq. (7)
                !<
                !<
                !< The expression $S^{m,c}(\nu)$ describes the source function and is according to Kirchhoff's law of thermal radiation given by
                !<
                !<      S^{m,c}(\nu) &= \frac{\epsilon_l^{m,c}(\nu) + \epsilon_d^{m,c}(\nu)}{\kappa_l^{m,c}(\nu) + \kappa_d^{m,c}(\nu)}\\
                !<                   &= \frac{\kappa_l^{m,c}(\nu) \, J(T_\mathrm{ex}^{m,c}, \nu) + \kappa_d^{m,c}(\nu) \, J(T_d^{m,c}, \nu)}
                !<                           {\kappa_l^{m,c}(\nu) + \kappa_d^{m,c}(\nu)}\\
                !<                   &= \frac{\tau_l^{m,c}(\nu) \, J(T_\mathrm{ex}^{m,c}, \nu) + \gamma \, \tau_d^{m,c}(\nu) \, J(T_d^{m,c}, \nu)}
                !<                           {\tau_l^{m,c}(\nu) + \tau_d^{m,c}(\nu)},
                !<
                !< where $\epsilon_{l,d}(\nu)$ and $\kappa_{l,d}(\nu)$ are the emission and absorption coefficients for line and dust, respectively.
                !< Additionally, the optical depth is given by $\tau(\nu) = \int \kappa(\nu) \, ds = \kappa(\nu) \, s$. This assumes that molecules
                !< and dust are well mixed, i.e.\ it would not be correct if the molecule exists only in part of the cloud, but the dust everywhere.
                !< Since the line form function $\phi$ is part of $\tau_l$, outside the line this reduces to $J(T_d^{m,c}, \nu)$, as it should.
                !< $\tau_l^{m,c}(\nu)$ is actually the sum over all opacities of all lines of molecule $m$ and component $c$ at the given frequency
                !< point $\nu$.
                !<
                SourceFunc = 0.d0
                if (tbFlag(l) .and. (.not. LocalOverlapFlag) .and. LTEFlag(c)) then
                    SourceFunc = j_tk
                elseif (dabs(tau_l_ab) > tiny(1.d0)) then                                   !< prevent division by zero
                    SourceFunc = (tau_l_em / tau_l_ab)
                endif

                ! Debug:
                ! print*," "
                ! print*,"l, tbFlag(l) = ", l, tbFlag(l)
                ! print*,"j_tk = ", j_tk
                ! print*,"tau_l_em = ", tau_l_em
                ! print*,"tau_l_ab = ", tau_l_ab
                ! print*,"freq, SourceFunc = ", freq, SourceFunc


                !< define weighting factor for continuum correction
                if (LocalOverlapFlag) then
                    i = 1
                else
                    i = NumberMoleculeComponents
                endif
                i = max0(i, 1)


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !<
                !< layer closest to background continuum
                !<
                !<----------------------------------------------------------------------------------------------------------------------------------------
                if (CurrDistID == 1) then                                                   !< determine intensity for emission


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< Calculate intensity using eq. (1)
                    !<
                    !<
                    !< We assume that emission components do not interact with each other radiatively, i.e.\ one emission layer is not influenced by
                    !< the others. But the emission layers may overlap to offer the possibility to model sources consisting of several molecules and
                    !< compounds. The solution of the radiative transfer equation for emission layers is
                    !<
                    !<      T_{\rm mb}^{\rm em}(\nu) &= \left[\sum_m \sum_c \left[\eta \left(\theta^{m,c}\right) S^{m,c}(\nu)
                    !<                                  \left(1 - e^{-\tau_{\rm total}^{m,c}(\nu)}\right)
                    !<                                  \left( e^{-\tau_{\rm total}^{m,c}(\nu)} - 1\right) I_{\rm bg} (\nu) \right] \right] \\
                    !<                                  + I_{\rm bg} (\nu) - J_\mathrm{CMB},
                    !<
                    !< where the sums go over the indices $m$ for molecule, and $c$ for (emission) component, respectively. The last term
                    !< $J_\mathrm{CMB}$ describes the OFF position for single dish observations, where we have an intensity caused
                    !< by the cosmic background $J_\mathrm{CMB}$. For interferometric observationss, the contribution of the cosmic background
                    !< is filtered out and has to be subtracted as well.
                    !<
                    LocalIbg = (j_back_full + TotalLocalIntensity)
                    LocalIntensity = (eta_source * (SourceFunc * (1.d0 - ExpSave(-tau_total)) &
                                        + LocalIbg * (ExpSave(-tau_total) - 1.d0)))  + (1.d0 / i) * ((LocalIbg ))
                                                                                  !  + (1.d0 / i) * ((LocalIbg - j_cb))

                    ! Debug:
                    ! print*," "
                    ! print*,"freq = ", freq
                    ! print*,"j_cb = ", j_cb
                    ! print*,"i = ", i
                    ! print*,"j_back_full = ", j_back_full
                    ! print*,"LocalIbg = ", LocalIbg
                    ! print*,"eta_source = ", eta_source
                    ! print*,"SourceFunc = ", SourceFunc
                    ! print*,"TotalLocalIntensity = ", TotalLocalIntensity
                    ! print*,"tau_total = ", tau_total
                    ! print*,"LocalIntensity = ", LocalIntensity


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !<
                !< foreground:
                !<
                !<----------------------------------------------------------------------------------------------------------------------------------------
                else                                                                        !< foreground loop here
                    Ibg = (TotalLocalIntensity + j_back) / i


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< Calculate intensity using eq. (16)
                    !<
                    !<
                    !< In contrast to emission layers, absorbing components may interact with each other, where absorption takes places only, if the
                    !< kinetic temperature for the absorbing layer is lower than the temperature of the background.
                    !<
                    !< foreground components have to be considered in an iterative manner. The solution of the radiative transfer equation for
                    !< foreground layers can be expressed as
                    !<
                    !<      T_{\rm mb}^{\rm abs}(\nu)_{m,c=1} &= \eta \left(\theta^{m,c=1} \right) \\
                    !<                                        &  \Bigg[S^{m,c=1}(\nu) \left(1 - e^{-\tau_{\rm total}^{m,c=1}(\nu)}\right) \\
                    !<                                        & + I_{\rm bg} (\nu) e^{-\tau_{\rm total}^{m,c=1}(\nu)}\Bigg] \\
                    !<                                        & + \left(1 - \eta \left(\theta^{m,c=1} \right)\right) \, I_{\rm bg} (\nu) \\
                    !<      T_{\rm mb}^{\rm abs}(\nu)_{m,c=i} &= \eta \left(\theta^{m,c=i} \right)
                    !<                                        &  \Big[S^{m,c=i}(\nu) \left(1 - e^{-\tau_{\rm total}^{m,c=i}(\nu)}\right) \\
                    !<                                        & + T_{\rm mb}^{\rm abs}(\nu)_{m,c=(i-1)} e^{-\tau_{\rm total}^{m,c=i}(\nu)}\Big]\\
                    !<                                        & + \left(1-\eta \left(\theta^{m,c=i} \right)\right) T_{\rm mb}^{\rm abs}(\nu)_{m,c=(i-1)},
                    !<
                    !< where $m$ indicates the index of the current molecule and $i$ represents an index running over all foreground components of
                    !< all molecules. Additionally, we assume that each absorbing component covers the whole beam, i.e.
                    !< $\eta \left(\theta^{m,c=1} \right) \equiv 1$ for all absorbing layer. So, we achieve
                    !<
                    !<      T_{\rm mb}^{\rm abs}(\nu)_{m,c=1} &= \Big[ S^{m,c=1}(\nu) \left(1 - e^{-\tau_{\rm total}^{m,c=1}(\nu)}\right) \\
                    !<                                        & + I_{\rm bg} (\nu) e^{-\tau_{\rm total}^{m,c=1}(\nu)}\Big] \\
                    !<      T_{\rm mb}^{\rm abs}(\nu)_{m,c=i} &= \Big[S^{m,c=i}(\nu) \left(1 - e^{-\tau_{\rm total}^{m,c=i}(\nu)}\right) \\
                    !<                                        & + T_{\rm mb}^{\rm abs}(\nu)_{m,c=(i-1)} e^{-\tau_{\rm total}^{m,c=i}(\nu)}\Big],
                    !<
                    !< where $I_{\rm bg} (\nu)$ describes the emission spectrum plus continuum. For absorption lines the emission by other components
                    !< is considered by first calculating the emission lines and then use this emission as new continuum for absorption lines
                    !< reflecting the fact that cold absorption layers are often found in front of hotter emission sources. The order of components
                    !< along the line of sight is defined by the occurrence of a certain foreground component in the molfit file.
                    !<
                    !<
                    LocalIntensity = SourceFunc * (1.d0 - ExpSave(-tau_total)) + Ibg * ExpSave(-tau_total)
                endif


                !< remove J_CMB for last component
                if (c == NumComp) then
                    LocalIntensity = LocalIntensity - j_cb
                endif
            endif


            !<############################################################################################################################################
            !< Only used for storing opacities and intensities
            !<############################################################################################################################################
            if (AllOutputFilesFlag) then
                FirstIndex = DataPointIndexFreqRange(l, 1)                                  !< get index of first freq. point in 'ObservationalDataList'
                LastIndex = DataPointIndexFreqRange(l, 2)                                   !< get index of last freq. point in 'ObservationalDataList'
                LocalFreqIndex = minloc(dabs(freq - ObservationalDataList(FirstIndex:LastIndex, 1)), dim = 1)
                if (LocalOverlapFlag) then
                    kk = 1
                    ll = CurrDistID
                else
                    kk = IsoCounter
                    ll = c
                endif
                Do ic = 1, kk                                                               !< loop over isotopologues


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< compute source function for all isotopologues separately


                    !< define arrays for intensity array calculation
                    if (dabs(TauHelperArrayABS(ic)) > tiny(1.d0)) then                      !< prevent division by zero
                        TauHelperArray2(ic) = (TauHelperArrayEM(ic) + tau_d * j_td) / (TauHelperArrayABS(ic) + tau_d)
                    else
                        TauHelperArray2(ic) = 0.d0
                    endif


                    !< define arrays for integrated line calculation
                    !    Do t = 1, wm
                    !        !< IntegHelperArray1(ic, t) = tau_t_em      (for a single transition)
                    !        !< IntegHelperArray2(ic, t) = SourceFunc    (for a single transition)
                    !        if (dabs(IntegHelperArray1(ic, t) + tau_d) > tiny(1.d0)) then   !< prevent division by zero
                    !            IntegHelperArray2(ic, t) = (IntegHelperArray1(ic, t) * j_tk + tau_d * j_td) / (IntegHelperArray1(ic, t) + tau_d)
                    !        else
                    !            IntegHelperArray2(ic, t) = 0.d0
                    !        endif
                    !    end Do


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< layer closest to background continuum
                    if (CurrDistID == 1) then                                               !< determine intensity for emission


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine background temperature
                        j_cb = freq_t / (ExpSave(freq_t / Tcbg) - 1.d0)
                        j_back_full = j_back + j_cb
                        if (dabs(BackgroundTemperatureRange(l)) > tiny(1.d0)) then
                            j_back_full = j_back_full + BackgroundTemperatureRange(l) &
                                                        * (freq / StartFrequency(l))**TemperatureSlopeRange(l)
                        endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< compute intensity by solving the detection equation


                        !< define weighting factor for continuum correction
                        if (LocalOverlapFlag) then
                            i = 1
                        else
                            i = NumberMoleculeComponents
                        endif


                        !< define arrays for intensity array calculation
                        LocalIntArray(ic) = (eta_source * (TauHelperArray2(ic) &
                                                    * (1.d0 - ExpSave(-(TauHelperArrayABS(ic) + tau_d))) &
                                                    + (j_back) * (ExpSave(-(TauHelperArrayABS(ic) + tau_d)) - 1.d0))) &
                                                    + (1.d0 / i) * (j_back - j_cb)


                        !< define arrays for integrated line calculation
                        if (.not. LocalOverlapFlag) then
                            Do t = 1, wm                                                    !< loop over transitions
                                !< IntegHelperArray1(ic, t) = tau_total = tau_l_em  (for a single transition)
                                !< IntegHelperArray2(ic, t) = SourceFunc            (for a single transition)


                                !< calculate intensity
                                IntegHelperArray1(ic, t) = IntegHelperArray1(ic, t) + tau_d
                                val = (eta_source * (IntegHelperArray2(ic, t) * (1.d0 - ExpSave(-IntegHelperArray1(ic, t))) &
                                                                + (j_back) * (ExpSave(-IntegHelperArray1(ic, t)) - 1.d0))) &
                                                                + (1.d0 / i) * (j_back - j_cb)

                                !< integrate peak
                                if (LocalFreqIndex /= FirstIndex) then
                                    IntensityPerPeak(l, ll, ic, t) = IntensityPerPeak(l, ll, ic, t) &
                                                                    + (freq - ObservationalDataList(LocalFreqIndex - 1, 1)) &
                                                                    * (PrevIntPerPeak(ic, t) + val) / 2.d0
                                endif
                                PrevIntPerPeak(ic, t) = val
                            end Do                                                          !< t: loop over transitions
                        endif


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< other layer
                    else
                        Ibg = TotalLocalIntensity


                        !< define arrays for intensity array calculation
                        LocalIntArray(ic) = TauHelperArray2(ic) * (1.d0 - ExpSave(-(TauHelperArrayABS(ic) + tau_d))) &
                                                                        + Ibg * ExpSave(-(TauHelperArrayABS(ic) + tau_d))


                        !< define arrays for integrated line calculation
                        if (.not. LocalOverlapFlag) then
                            Do t = 1, wm                                                    !< loop over transitions
                                !< IntegHelperArray1(ic, t) = tau_total = tau_t_em + tau_d      (for a single transition)
                                !< IntegHelperArray2(ic, t) = SourceFunc                        (for a single transition)


                                !< calculate intensity
                                IntegHelperArray1(ic, t) = IntegHelperArray1(ic, t) + tau_d
                                val = IntegHelperArray2(ic, t) * (1.d0 - ExpSave(-IntegHelperArray1(ic, t))) &
                                                                + Ibg * ExpSave(-IntegHelperArray1(ic, t))


                                !< integrate peak
                                if (LocalFreqIndex /= FirstIndex) then
                                    IntensityPerPeak(l, ll, ic, t) = IntensityPerPeak(l, ll, ic, t) &
                                                                    + (freq - ObservationalDataList(LocalFreqIndex - 1, 1)) &
                                                                    * (PrevIntPerPeak(ic, t) + val) / 2.d0
                                endif
                                PrevIntPerPeak(ic, t) = val
                            end Do                                                          !< t: loop over transitions
                        endif
                    endif
                end Do                                                                      !< ic: loop over isotopologues
            endif
            !<############################################################################################################################################
            !<############################################################################################################################################


            !< we're done
            return
        end subroutine DetectionEquation


        !>************************************************************************************************************************************************
        !> subroutine: ResampleChannel
        !>
        !> defines additional frequency points which are used for integration of spectrum over channel width
        !>
        !>
        !> input variables:     ccc:                            index of current layer component
        !>                      NumCompAtCurrDistance:          max. number of layer components
        !>                      MoleculeIndex:                  index of molcule
        !>                      FreqIndex:                      frequency index
        !>                      FreqChannelFirst:               first frequency of current channel
        !>                      FreqChannelLast:                last frequency of current channel
        !>                      ChannelWidth:                   channel width
        !>                      max_width:                      max. velocity width
        !>                      min_v_width:                    minimal line width
        !>                      v_off:                          velocity offset
        !>                      min_v_off:                      min. velocity offset (in km/s) per layer
        !>                      max_v_off:                      max. velocity offset (in km/s) per layer
        !>                      c:                              current component index
        !>                      l:                              current range index
        !>
        !> output variables:    NumAdditionalFreq:              number of additional frequency points
        !>                      AdditionalFreqList:             list of additional frequency points
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2018-02-16
        !>
        subroutine ResampleChannel(NumAdditionalFreq, AdditionalFreqList, ccc, NumCompAtCurrDistance, MoleculeIndex, FreqIndex, &
                                   FreqChannelFirst, FreqChannelLast, ChannelWidth, ReverseFreqFlag, max_width, min_v_width, &
                                   v_off, min_v_off, max_v_off, c, l)

            implicit none
            integer, parameter :: MaxFreqResampling = 100                                   !< max. number of frequency points for resampling
            integer, parameter :: MaxNumTransFreq = 10                                      !< max. number of trans. freq. which are taken into account
            integer :: ccc                                                                  !< (input) index of current layer component
            integer :: c                                                                    !< (input) current component index
            integer :: l                                                                    !< (input) current range index
            integer :: NumCompAtCurrDistance                                                !< (input) max. number of layer components
            integer :: NumAdditionalFreq                                                    !< (output) number of additional frequency points
            integer :: MoleculeIndex                                                        !< (input) molecule index
            integer :: FreqIndex                                                            !< (input) index of obs. frequency channel
            integer :: LocalTransFreqIndex                                                  !< working variable: local index of transition frequency
            integer :: j                                                                    !< loop variable
            integer :: ShiftIndex1, ShiftIndex2                                             !< working variable: shift index caused by velocity offset
            integer :: sigmaIndex                                                           !< working variable: sigma index caused by line broadening
            integer :: LocalFreqIndex                                                       !< working variable: local obs. freq. index
            integer :: NumNextChan                                                          !< internal parameter: number of neighbouring channels
            real*8 :: FreqChannelFirst                                                      !< (input) first frequency of current channel
            real*8 :: FreqChannelLast                                                       !< (input) last frequency of current channel
            real*8 :: ChannelWidth                                                          !< (input) channel width
            real*8 :: max_width                                                             !< (input) max. velocity width
            real*8 :: min_v_width                                                           !< (input) min. velocity width
            real*8 :: LocalFreq                                                             !< working variable: local obs. freq. (MHz)
            real*8 :: v_off                                                                 !< (input) velocity offset
            real*8 :: min_v_off                                                             !< (input) min. velocity offset
            real*8 :: max_v_off                                                             !< (input) max. velocity offset
            real*8 :: LocalVelocityShift                                                    !< shift due to velocity offset
            real*8 :: DeltaFreq                                                             !< working variable: freq. range for trans. freq. detection
            real*8 :: sigma                                                                 !< working variable: sigma
            real*8, dimension(MaxFreqResampling) :: AdditionalFreqList                      !< (output) list of additional frequency points
            logical :: ReverseFreqFlag                                                      !< flag indicating reversed frequency ordering

            ! Debug:
            ! print*,""
            ! print*,"ccc = ", ccc
            ! print*,"NumCompAtCurrDistance = ", NumCompAtCurrDistance
            ! print*,"MoleculeIndex = ", MoleculeIndex
            ! print*,"FreqIndex = ", FreqIndex
            ! print*,"FreqChannelFirst = ", FreqChannelFirst
            ! print*,"FreqChannelLast = ", FreqChannelLast
            ! print*,"ChannelWidth = ", ChannelWidth
            ! print*,"ReverseFreqFlag = ", ReverseFreqFlag
            ! print*,"max_width = ", max_width
            ! print*,"min_v_width = ", min_v_width
            ! print*,"v_off = ", v_off
            ! print*,"min_v_off = ", min_v_off
            ! print*,"max_v_off = ", max_v_off
            ! print*,"c = ", c
            ! print*,"l = ", l


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize return parameters
            NumAdditionalFreq = 0
            AdditionalFreqList = 0.d0


            !< define internal parameters
            NumNextChan = 2


            !<============================================================================================================================================
            !< prepare integration of spectrum over channel width
            LocalFreq = ObservationalDataList(FreqIndex, 1)                                 !< current obs. frequency


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< take local-overlap into account
            ! if (LocalOverlapFlag .and. ccc == NumCompAtCurrDistance) then
            ! if (LocalOverlapFlag .and. (ccc == NumCompAtCurrDistance .or. (KindOfMolecule(c) == 1 .and. (.not. tbFlag(l))) &
            !     .or. (KindOfMolecule(c) == 2 .and. (.not. tbFlag(l)))) ) then
            if (LocalOverlapFlag .and. (ccc == NumCompAtCurrDistance .or. (.not. tbFlag(l)))) then


                !< get local transition frequency index
                LocalTransFreqIndex = minloc(dabs(FreqChannelFirst - DopplerShiftedTransFreq(:)), dim = 1)


                !< compute shift of NumTransFreqPerObsFreq array due to current velocity offset
                LocalVelocityShift = LocalFreq * ((1.d0 - (min_v_off / ckms)) - 1.d0)       !< compute frequency difference due to velocity offset
                ShiftIndex1 = int(LocalVelocityShift / ChannelWidth)                        !< determine corresponding shift index
                LocalVelocityShift = LocalFreq * ((1.d0 - (max_v_off / ckms)) - 1.d0)       !< compute frequency difference due to velocity offset
                ShiftIndex2 = int(LocalVelocityShift / ChannelWidth)                        !< determine corresponding shift index


                !< compute sigma which correspond to local line width
                sigma = (((min_v_width / ckms) * ObservationalDataList(FreqIndex, 1)) / (2.d0 * dsqrt(2.d0 * dlog(2.d0))))
                sigmaIndex = max(1, NumNextChan * int(ChannelWidth / sigma))                !< determine corresponding sigma index

                ! Debug:
                ! print*," "
                ! print*,"sigmaIndex = ", sigmaIndex


                !< check, if in current obs. freq. +/- min_vel_offset/max_vel_offset
                NumAdditionalFreq = 2
                Do LocalFreqIndex = min(FreqIndex + ShiftIndex1, FreqIndex + ShiftIndex2), &
                                    max(FreqIndex + ShiftIndex1, FreqIndex + ShiftIndex2)
                    if (1 <= LocalFreqIndex .and. LocalFreqIndex <= TotalNumberDataPoints) then

                        ! Debug:
                        ! print*," "
                        ! print*,"FreqIndex = ", FreqIndex
                        ! print*,"LocalFreqIndex = ", LocalFreqIndex


                        !< compute number of additional frequency points
                        if (sum(NumTransFreqPerObsFreq(:, LocalFreqIndex)) > 0) then
                            NumAdditionalFreq = NumAdditionalFreq + sum(NumTransFreqPerObsFreq(:, LocalFreqIndex)) * 3 * sigmaIndex

                            ! Debug:
                            ! print*," "
                            ! print*,"NumAdditionalFreq = ", NumAdditionalFreq
                            ! print*,"sigmaIndex = ", sigmaIndex
                            ! print*,"NumTransFreqPerObsFreq(MoleculeIndex, LocalFreqIndex) = ", NumTransFreqPerObsFreq(MoleculeIndex, LocalFreqIndex)


                            !< check, if neighboring channels contain transition frequencies
                        endif
                    endif
                end Do                                                                      !< LocalFreqIndex: loop over additional frequencies


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< continue here, if local-overlap is not taken into account
            elseif (.not. LocalOverlapFlag) then


                !< compute shift of NumTransFreqPerObsFreq array due to current velocity offset
                LocalVelocityShift = LocalFreq * ((1.d0 - (v_off / ckms)) - 1.d0)           !< compute frequency difference due to velocity offset
                ShiftIndex1 = int(LocalVelocityShift / ChannelWidth)                        !< determine corresponding shift index
                LocalFreqIndex = (FreqIndex + ShiftIndex1)
                if (1 <= LocalFreqIndex .and. LocalFreqIndex <= TotalNumberDataPoints) then

                    ! Debug:
                    ! print*," "
                    ! print*,"FreqIndex = ", FreqIndex
                    ! print*,"LocalFreqIndex = ", LocalFreqIndex


                    !< compute sigma which correspond to local line width
                    sigma = (((max_width / ckms) * ObservationalDataList(FreqIndex, 1)) / (2.d0 * dsqrt(2.d0 * dlog(2.d0))))
                    sigmaIndex = max(1, int(ChannelWidth / sigma))                          !< determine corresponding sigma index

                    ! Debug:
                    ! print*," "
                    ! print*,"sigmaIndex = ", sigmaIndex


                    !< compute number of additional frequency points
                    NumAdditionalFreq = 2
                    if (NumTransFreqPerObsFreq(MoleculeIndex, LocalFreqIndex) > 0) then
                        NumAdditionalFreq = NumAdditionalFreq + NumTransFreqPerObsFreq(MoleculeIndex, LocalFreqIndex) * 3 &
                                                                * sigmaIndex

                        ! Debug:
                        ! print*," "
                        ! print*,"NumAdditionalFreq = ", NumAdditionalFreq
                        ! print*,"sigmaIndex = ", sigmaIndex
                        ! print*,"NumTransFreqPerObsFreq(MoleculeIndex, LocalFreqIndex) = ", NumTransFreqPerObsFreq(MoleculeIndex, LocalFreqIndex)


                        !< check, if neighboring channels contain transition frequencies
                        Do j = (-1) * NumNextChan, NumNextChan
                            if (1 <= (LocalFreqIndex + j) .and. (LocalFreqIndex + j) <= TotalNumberDataPoints .and. j /= 0) then
                                if (NumTransFreqPerObsFreq(MoleculeIndex, LocalFreqIndex + j) > 0) then
                                    NumAdditionalFreq = NumAdditionalFreq + 3 * sigmaIndex
                                endif
                            endif
                        end Do
                    endif
                else
                    NumAdditionalFreq = 2
                endif
            endif


            !< construct additional frequency points
            if (NumAdditionalFreq > 0) then
                NumAdditionalFreq = min(NumAdditionalFreq, MaxNumTransFreq)
                DeltaFreq = (ChannelWidth / NumAdditionalFreq)
                if (ReverseFreqFlag) then
                    Do j = 1, NumAdditionalFreq                                             !< loop over additional frequencies
                        AdditionalFreqList(j) = FreqChannelLast - dabs(DeltaFreq) * (j - 1)
                    end Do                                                                  !< j: loop over additional frequencies
                    AdditionalFreqList(NumAdditionalFreq) = FreqChannelFirst                !< make sure, that last frequency is included
                else
                    Do j = 1, NumAdditionalFreq                                             !< loop over additional frequencies
                        AdditionalFreqList(j) = FreqChannelFirst + DeltaFreq * (j - 1)
                    end Do                                                                  !< j: loop over additional frequencies
                    AdditionalFreqList(NumAdditionalFreq) = FreqChannelLast                 !< make sure, that last frequency is included
                endif
            endif

            ! Debug:
            ! if (FreqIndex == 5) then
            !     print*," "
            !     print*,"l, c, ccc = ", l, c, ccc
            !     print*,"KindOfMolecule(c), tbFlag(l) = ", KindOfMolecule(c), tbFlag(l)
            !     print*,"NumAdditionalFreq, MoleculeIndex, FreqIndex = ", NumAdditionalFreq, MoleculeIndex, FreqIndex
            !     print*,"AdditionalFreqList(1:NumAdditionalFreq) = ", AdditionalFreqList(1:NumAdditionalFreq), FreqIndex
            !     print*,"NumTransFreqPerObsFreq(MoleculeIndex, FreqIndex) = ", NumTransFreqPerObsFreq(MoleculeIndex, FreqIndex)
            ! endif


            !< we're done
            return
        end subroutine ResampleChannel


        !>************************************************************************************************************************************************
        !> subroutine: CalcLocalOverlapTransFreq
        !>
        !> calculates all Doppler-shifted transition frequencies and their central optical depths for current layer
        !>
        !>
        !> input variables:     NumCompAtCurrDistance:          number of cores of first distance (i.e. CurrDistID == 1)
        !>                      NumComp:                        total number of components
        !>                      TotalNumMol:                    total number of molecules
        !>                      CurrDistID:                     index of current distance
        !>                      l:                              current frequency range index
        !>                      CopyCompMoleculeIndex:          local copy of CompMoleculeIndexOrig
        !>                      CopyIsoRatioConversionTable:    local copy of IsoRatioConversionTableOrig
        !>                      CopymyXCLASSParameter:          local copy of myXCLASSParameterOrig
        !>                      Q:                              partition function at the rotation temperatures
        !>                      LocalDistanceOrderingArray:     local copy of DistanceOrderingArray
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2018-02-18
        !>
        subroutine CalcLocalOverlapTransFreq(NumCompAtCurrDistance, NumComp, TotalNumMol, CurrDistID, l, CopyCompMoleculeIndex, &
                                             CopyIsoRatioConversionTable, CopymyXCLASSParameter, Q, LocalDistanceOrderingArray)

            implicit none
            integer :: NumCompAtCurrDistance                                                !< (input) number of comp., which correspond to current dist.
            integer :: NumComp                                                              !< (input) total number of components
            integer :: TotalNumMol                                                          !< (input) total number of molecules
            integer :: CurrDistID                                                           !< (input) index of current distance
            integer :: l                                                                    !< (input) current frequency range index
            integer :: ccc                                                                  !< over-all current component index
            integer :: c                                                                    !< current component index
            integer :: MoleculeIndex                                                        !< index for molecule
            integer :: MoleculeIndexLoop                                                    !< loop index for isotopolouges
            integer :: kk                                                                   !< used for handling of isotoplogues
            integer :: FirstMolecularDataIndex                                              !< first index for molecular data table
            integer :: LastMolecularDataIndex                                               !< last index for molecular data table
            integer :: i                                                                    !< counter for central opacities
            integer :: t                                                                    !< loop variable for different transitions
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer, dimension(NumComp) :: CopyCompMoleculeIndex                            !< local copy of CompMoleculeIndexOrig
            integer, dimension(NumComp, NumComp) :: LocalDistanceOrderingArray              !< (input) local copy of DistanceOrderingArray array
            real*8 :: x0                                                                    !< working variable: Doppler-shifted transition
            real*8 :: LocalQ                                                                !< local partition function
            real*8 :: LocalIsoRatio                                                         !< local iso ratio
            real*8 :: tau_t_em                                                              !< working variable: tau for each transition
            real*8 :: tau_t_ab                                                              !< working variable: tau for each transition
            real*8 :: Temp                                                                  !< working variable: kinetic / electronic temperature
            real*8 :: Ntot                                                                  !< working variable: column density / emission measure
            real*8 :: Ne                                                                    !< electronic density
            real*8, dimension(NumComp, TotalNumMol) :: Q                                    !< partition function at the rotation temperatures
            real*8, dimension(16, NumComp) :: CopymyXCLASSParameter                         !< (input) local copy of myXCLASSParameterOrig
            real*8, dimension(TotalNumMol, TotalNumMol) :: CopyIsoRatioConversionTable      !< (input) local copy of IsoRatioConversionTableOrig
            logical :: CalcFalg                                                             !< used for calculation of isotopoluges


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< compute total number of all Doppler-shifted transitions of all molecules in the current layer
            TotalNumTransFreq = 0                                                           !< reset variable for total number of trans. freq.
            Do ccc = 1, NumCompAtCurrDistance                                               !< loop over all components of current layer
                c = LocalDistanceOrderingArray(CurrDistID, ccc)                             !< get component index
                MoleculeIndex = CopyCompMoleculeIndex(c)                                    !< get molecule index
                if (IsoFlag) then                                                           !< use isotopologues?
                    kk = TotalNumMol
                else
                    kk = MoleculeIndex
                endif
                Do MoleculeIndexLoop = MoleculeIndex, kk                                    !< loop over all molecule indices (including isotopologues!)
                    CalcFalg = .true.
                    if (IsoFlag) then                                                       !< use isotopologues?
                        if (CopyIsoRatioConversionTable(MoleculeIndex, MoleculeIndexLoop) == 0.d0) then
                            CalcFalg = .false.
                        endif
                    endif
                    if (CalcFalg) then                                                      !< consider current molecule


                        !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        !< get transition frequencies for LTE moleclules and RRLs and compute corresponding optical depths
                        if (LTEFlag(c) .or. KindOfMolecule(c) == 2) then
                            FirstMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 1) !< get first mol. data index of freq. range
                            LastMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 2)  !< get last mol. data index of freq. range
                            if (FirstMolecularDataIndex > 0) then
                                TotalNumTransFreq = TotalNumTransFreq + (LastMolecularDataIndex - FirstMolecularDataIndex + 1)
                            endif                                                           !< continue here, if FirstMolecularDataIndex == 0

                        endif                                                               !< continue here, if LTE and Non-LTE calculation is finished
                    endif                                                                   !< continue here, if CalcFalg = .false.
                end Do                                                                      !< MoleculeIndexLoop: loop over all molecule indices
            end Do                                                                          !< ccc: loop over all components of current layer

            ! Debug:
            ! print*,"TotalNumTransFreq = ", TotalNumTransFreq


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for CentralOpticalDepth
            if (allocated(CentralOpticalDepth)) then
                deallocate(CentralOpticalDepth, DopplerShiftedTransFreq, DopplerShiftedTransFreqIndex, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine CalcLocalOverlapTransFreq!")')
                        write(ErrChannel, '(3x, "Can not deallocate variables CentralOpticalDepth, DopplerShiftedTransFreq, ", $)')
                        write(ErrChannel, '("and DopplerShiftedTransFreqIndex!")')
                        write(ErrChannel, '(" ")')
                    end Do
                    return
                endif
            endif
            i = max0(1, TotalNumTransFreq)
            allocate(CentralOpticalDepth(i, 2), DopplerShiftedTransFreq(i), DopplerShiftedTransFreqIndex(i, 6), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 2
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x, "Error in subroutine CalcLocalOverlapTransFreq!")')
                    write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                    write(ErrChannel, '(5x, "- CentralOpticalDepth(i, 2)")')
                    write(ErrChannel, '(5x, "- DopplerShiftedTransFreq(i)")')
                    write(ErrChannel, '(5x, "- DopplerShiftedTransFreqIndex(i, 6)")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Used parameter:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(5x, "- i = ", I8)') i
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                end Do
                return
            endif
            CentralOpticalDepth = 0.d0
            DopplerShiftedTransFreq = 0.d0
            DopplerShiftedTransFreqIndex = 0


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< first of all compute total number of all transitions
            if (TotalNumTransFreq > 0) then
                i = 0                                                                       !< reset counter
                Do ccc = 1, NumCompAtCurrDistance                                           !< loop over all components of current layer
                    c = LocalDistanceOrderingArray(CurrDistID, ccc)                         !< get component index
                    MoleculeIndex = CopyCompMoleculeIndex(c)                                !< get molecule index
                    if (IsoFlag) then                                                       !< use isotopologues?
                        kk = TotalNumMol
                    else
                        kk = MoleculeIndex
                    endif

                    ! Debug:
                    ! print*,"ccc = ", ccc
                    ! print*,"c = ", c
                    ! print*,"MoleculeIndex = ", MoleculeIndex

                    Do MoleculeIndexLoop = MoleculeIndex, kk                                !< loop over all molecule indices (including isotopologues!)
                        CalcFalg = .true.
                        if (IsoFlag) then                                                   !< use isotopologues?
                            if (CopyIsoRatioConversionTable(MoleculeIndex, MoleculeIndexLoop) == 0.d0) then
                                CalcFalg = .false.
                            endif
                        endif
                        if (CalcFalg) then                                                  !< consider current molecule


                            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                            !< get transition frequencies for LTE moleclules and RRLs and compute corresponding optical depths
                            if (LTEFlag(c) .or. KindOfMolecule(c) == 2) then
                                FirstMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 1) !< get first mol. data index of freq. range
                                LastMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 2)  !< get last mol. data index of freq. range
                                if (FirstMolecularDataIndex > 0) then
                                    Do t = FirstMolecularDataIndex, LastMolecularDataIndex  !< loop over all transitions of current molecule and
                                        i = i + 1


                                        !< get transition frequency
                                        x0 = MolecularData(t, 1) * (1.d0 - (CopymyXCLASSParameter(6, c) + GlobalvLSR(l)) / ckms)
                                        DopplerShiftedTransFreq(i) = x0


                                        !< compute corresponding optical depth for moleucles (in LTE) and RRLs (in LTE and Non-LTE)
                                        tau_t_em = 0.d0
                                        tau_t_ab = 0.d0
                                        LocalQ = Q(c, MoleculeIndexLoop)                    !< get partition function for molecule
                                        LocalIsoRatio = CopyIsoRatioConversionTable(MoleculeIndex, MoleculeIndexLoop)
                                        Temp = CopymyXCLASSParameter(1, c)                  !< get kinetic (molecule) or electronic (RRLs) temperature
                                        Ntot = CopymyXCLASSParameter(2, c)                  !< column density (molecule) or emission measure (RRLs)
                                        Ne = CopymyXCLASSParameter(3, c)                    !< electron density (RRLs)
                                        call ComputeCentralOpticalDepthLTE(tau_t_em, tau_t_ab, c, t, 1, LocalQ, &
                                                                           LocalIsoRatio, Temp, Ntot, Ne)

                                        ! Debug:
                                        ! print*," "
                                        ! print*,"l = ", l
                                        ! print*,"c, MoleculeIndexLoop, MoleculeIndex, kk = ", c, MoleculeIndexLoop, MoleculeIndex, kk
                                        ! print*,"t, i = ", t, i
                                        ! print*,"MolecularData(t, :) = ", MolecularData(t, :)
                                        ! print*,"x0 = ", x0
                                        ! print*,"LocalQ = ", LocalQ
                                        ! print*,"LocalIsoRatio = ", LocalIsoRatio
                                        ! print*,"CopymyXCLASSParameter(:, c) = ", CopymyXCLASSParameter(:, c)
                                        ! print*,"tau_t_em = ", tau_t_em


                                        !< store computed optical depth and used indices
                                        DopplerShiftedTransFreqIndex(i, :) = (/l, MoleculeIndex, MoleculeIndexLoop, c, t, i/)


                                        !< store computed optical depths
                                        CentralOpticalDepth(i, :) = (/tau_t_em, tau_t_ab/)
                                    end Do                                                  !< t: loop over transitions
                                endif                                                       !< continue here, if FirstMolecularDataIndex == 0

                            endif                                                           !< continue here, if LTE and Non-LTE calculation is finished
                        endif                                                               !< continue here, if CalcFalg = .false.
                    end Do                                                                  !< MoleculeIndexLoop: loop over all molecule indices
                end Do                                                                      !< ccc: loop over all components of current layer

                ! Debug:
                ! print*," "
                ! print*,"before:"
                ! print*,"DopplerShiftedTransFreq = ", DopplerShiftedTransFreq
                ! print*,"DopplerShiftedTransFreqIndex(:, 5) = ", DopplerShiftedTransFreqIndex(:, 5)


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< create a sorted list of all Doppler-shifted transition frequencies of all molecules
                call HeapSort(TotalNumTransFreq, 6, DopplerShiftedTransFreq, DopplerShiftedTransFreqIndex)

                ! Debug:
                ! print*,"after:"
                ! print*,"sorted DopplerShiftedTransFreq = ", DopplerShiftedTransFreq
                ! print*,"DopplerShiftedTransFreqIndex(:, 5) = ", DopplerShiftedTransFreqIndex(:, 5)
            endif


            !< we're done
            return
        end subroutine CalcLocalOverlapTransFreq


        !>************************************************************************************************************************************************
        !> subroutine: DrawEllipse
        !>
        !> draw an rotated ellipse
        !>
        !>
        !> input variables:     xCen:                           x-coodrinate of center
        !>                      yCen:                           y-coodrinate of center
        !>                      Rx:                             x-radius
        !>                      Ry:                             y-radius
        !>                      alpha:                          rotation angle (in degree)
        !>                      NumX:                           max. number of pixels in x-direction
        !>                      NumY:                           max. number of pixels in y-direction
        !>
        !> output variables:    BeamLayer                       layer for beam description
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2018-03-08
        !>
        subroutine DrawEllipse(xCen, yCen, Rx, Ry, alpha, NumX, NumY, BeamLayer)

            implicit none
            integer :: xCen                                                                 !< x-coodrinate of center
            integer :: yCen                                                                 !< y-coodrinate of center
            integer :: Rx                                                                   !< x-radius
            integer :: Ry                                                                   !< y-radius
            integer :: NumX                                                                 !< max. number of pixels in x-direction
            integer :: NumY                                                                 !< max. number of pixels in y-direction
            integer :: i                                                                    !< loop variable
            integer :: NumDegreeSteps                                                       !< number of step sizes for theta
            integer :: x                                                                    !< working variable: x-coordinates
            integer :: y                                                                    !< working variable: y-coordinates
            real*8 :: alpha                                                                 !< rotation angle (in degree)
            real*8 :: p                                                                     !< working variable
            logical, dimension(NumX, NumY) :: BeamLayer                                     !< layer for beam description

            ! Debug:
            ! print*,"xCen = ", xCen
            ! print*,"yCen = ", yCen
            ! print*,"Rx = ", Rx
            ! print*,"Ry = ", Ry
            ! print*,"alpha = ", alpha
            ! print*,"NumX = ", NumX
            ! print*,"NumY = ", NumY


            !< initialize return parameter
            BeamLayer = .false.


            !< draw ellipse
            NumDegreeSteps = max0(360, max0(NumX, NumY)) * 3                                !< determine number of step sizes for angle iteration
            Do i = 1, NumDegreeSteps
                p = 360.d0 / (NumDegreeSteps - 1) * (i - 1) * (pi/ 180.d0)


                !< coordinates of ellispe taken from
                !< https://math.stackexchange.com/questions/33520/the-locus-of-two-perpendicular-tangents-to-a-given-ellipse/33555#33555
                x = xCen + idnint(Rx * dcos(p) * dcos(alpha) - Ry * dsin(p) * dsin(alpha))
                y = yCen + idnint(Rx * dcos(p) * dsin(alpha) + Ry * dsin(p) * dcos(alpha))


                !< set layer array
                if (1 <= x .and. x <= NumX .and. 1 <= y .and. y <= NumY) then
                    BeamLayer(x, y) = .true.
                endif
            end Do


            !< we're done
            return
        end subroutine DrawEllipse


        !>************************************************************************************************************************************************
        !> subroutine: SetPixel
        !>
        !> set pixel in model map and check if indices are within boundaries
        !>
        !>
        !> input variables:     x1:                             current x-coodrinate
        !>                      y1:                             current y-coodrinate
        !>                      x2:                             current x-coodrinate
        !>                      y2:                             current y-coodrinate
        !>                      c:                              current component index
        !>                      NumX:                           max. number of pixels in x-direction
        !>                      NumY:                           max. number of pixels in y-direction
        !>
        !> output variables:    None
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2018-03-07
        !>
        subroutine SetPixel(x1, x2, y1, y2, c, NumX, NumY)

            implicit none
            integer :: x1, x2, xx1, xx2                                                     !< current x-coodrinates
            integer :: y1, y2, yy1, yy2                                                     !< current y-coodrinates
            integer :: c                                                                    !< current component index
            integer :: NumX                                                                 !< max. number of pixels in x-direction
            integer :: NumY                                                                 !< max. number of pixels in y-direction


            !< make sure that indices are within limits
            xx1 = max0(1, min0(NumX, x1))
            yy1 = max0(1, min0(NumY, y1))


            !< set point
            if (x2 == (-1) .and. y2 == (-1)) then
                if (1 <= x1 .and. x1 <= NumX .and. 1 <= y1 .and. y1 <= NumY) then
                    LayerMap(xx1, yy1, c) = .true.
                endif


            !< set line
            else
                xx2 = max0(1, min0(NumX, x2))
                if (y2 == (-1)) then
                    if (1 <= y1 .and. y1 <= NumY .and. x1 <= NumX .and. x2 > 0) then
                        LayerMap(xx1:xx2, yy1, c) = .true.
                    endif
                else
                    yy2 = max0(1, min0(NumY, y2))
                    LayerMap(xx1:xx2, yy1:yy2, c) = .true.
                endif
            endif


            !< we're done
            return
        end subroutine SetPixel


        !>************************************************************************************************************************************************
        !> subroutine: CreatePixelMap
        !>
        !> determine map of model pixel for each component
        !>
        !>
        !> input variables:     l:                              current frequency range index
        !>                      NumComp:                        total number of components
        !>                      NumX:                           number of model pixels along x-direction
        !>                      NumY:                           number of model pixels along y-direction
        !>                      CopymyXCLASSParameter:          local copy of myXCLASSParameterOrig
        !>                      CopyCompMoleculeIndex:          local copy of CompMoleculeIndexOrig
        !>
        !> output variables:    NumberConfig:                   total number of unique configurations
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2018-03-05
        !>
        subroutine CreatePixelMap(NumberConfig, l, NumComp, NumX, NumY, CopymyXCLASSParameter, CopyCompMoleculeIndex)
            implicit none
            integer :: NumComp                                                              !< total number of components
            integer :: ModelIDy, ModelIDx                                                   !< loop variables for sub-beam description
            integer :: l                                                                    !< frequency range index
            integer :: c                                                                    !< loop variable for component index
            integer :: NumX, NumY                                                           !< number of model pixels along x- and y-direction
            integer :: FirstIndex                                                           !< index for first freq. data point
            integer :: LastIndex                                                            !< index for last freq. data point
            integer :: CenterMapX_phys                                                      !< center of model map (along x direction) (physical)
            integer :: CenterMapY_phys                                                      !< center of model map (along y direction) (physical)
            integer :: CenterX_phys                                                         !< center of model map (along x direction) (physical)
            integer :: CenterY_phys                                                         !< center of model map (along y direction) (physical)
            integer :: Radius_phys                                                          !< radius (physical)
            integer :: BoxHeight_phys                                                       !< height of square (physical)
            integer :: x, y, f, ddf_x, ddf_y                                                !< working variables for midpoint circle algorithm
            integer :: a1, b1                                                               !< working variables for midpoint ellipse algorithm
            integer :: i, j, n, ncomp                                                       !< working variables for log file of myXCLASS function
            integer :: NumberConfig                                                         !< total number of unique configurations
            integer, dimension(NumComp) :: CopyCompMoleculeIndex                            !< local copy of CompMoleculeIndexOrig
            real*8 :: a, b, alpha                                                           !< working variables for beam projection
            real*8 :: f0, f1                                                                !< working variables for single dish beam size
            real*8 :: CenterX_deg                                                           !< center of model map (along x direction) (degree)
            real*8 :: CenterY_deg                                                           !< center of model map (along y direction) (degree)
            real*8 :: Radius_deg                                                            !< radius (degree)
            real*8 :: BoxHeight_deg                                                         !< height of square (degree)
            real*8 :: LocalBeamX_deg                                                        !< size of beam along x-direction
            real*8 :: LocalBeamY_deg                                                        !< size of beam along y-direction
            real*8, dimension(16, NumComp) :: CopymyXCLASSParameter                         !< local copy of myXCLASSParameterOrig
            character(len=30) :: numberstring1                                              !< used for integer number to string conversion
            character(len=40) :: CurrMoleculeName                                           !< name of molecule
            logical :: IncludeFlag                                                          !< include flag
            logical, dimension(NumX, NumY) :: BeamLayer                                     !< layer for beam description (only used by myXCLASS function)

            ! Debug:
            ! print*,"l = ", l
            ! print*,"NumComp = ", NumComp
            ! print*,"NumX = ", NumX
            ! print*,"NumY = ", NumY
            ! print*,"CopymyXCLASSParameter = ", CopymyXCLASSParameter
            ! print*,"CopyCompMoleculeIndex = ", CopyCompMoleculeIndex
            ! print*,"TelescopeSize(l) = ", TelescopeSize(l)
            ! print*,"TelescopeBMIN(l) = ", TelescopeBMIN(l)
            ! print*,"TelescopeBMAJ(l) = ", TelescopeBMAJ(l)
            ! print*,"TelescopeBPA(l) = ", TelescopeBPA(l)


            !< initialize return parameters
            NumberConfig = 0
            ConfigList = .false.
            ConfigIndex = 0


            !< determine max. size of beam along x- and y-direction
            alpha = 0.d0
            if (dabs(TelescopeSize(l)) > tiny(1.d0)) then
                if (InterFlag(l)) then
                    LocalBeamX_deg = TelescopeSize(l)
                else
                    FirstIndex = DataPointIndexFreqRange(l, 1)                              !< get index of first freq. point in 'ObservationalDataList'
                    LastIndex = DataPointIndexFreqRange(l, 2)                               !< get index of last freq. point in 'ObservationalDataList'
                    f0 = ObservationalDataList(FirstIndex, 1)                               !< get first frequency of current freq. range
                    f1 = ObservationalDataList(LastIndex, 1)                                !< get last frequency of current freq. range
                    LocalBeamX_deg = 1.22d-3 * ckms / (dmin1(f0, f1) * TelescopeSize(l)) * (180.d0 / pi) * 3600.d0
                                                                                            !< compute telescope FWHM in degree
                endif
                LocalBeamY_deg = LocalBeamX_deg                                             !< set y-FWHM equal to x-FWHM
                a = LocalBeamX_deg / 2.d0                                                   !< compute x-radius related to telescope FWHM in degree
                b = a                                                                       !< set y-radius equal to x-radius
            elseif (dabs(TelescopeBMIN(l) - TelescopeBMAJ(l)) < tiny(1.d0) .and. dabs(TelescopeBMIN(l)) > tiny(1.d0)) then
                LocalBeamX_deg = TelescopeBMIN(l)
                LocalBeamY_deg = LocalBeamX_deg                                             !< set y-FWHM equal to x-FWHM
                a = LocalBeamX_deg / 2.d0                                                   !< compute x-radius related to telescope FWHM in degree
                b = a                                                                       !< set y-radius equal to x-radius
            else
                alpha = TelescopeBPA(l) * (pi / 180.d0)                                     !< convert rotation angle to radians
                a = TelescopeBMAJ(l) / 2.d0                                                 !< compute x-radius related to telescope x-FWHM in degree
                b = TelescopeBMIN(l) / 2.d0                                                 !< compute x-radius related to telescope y-FWHM in degree


                !< expressions taken from
                !< https://math.stackexchange.com/questions/923503/relation-of-ellipse-semi-axes-with-rotation-angle-and-projection-length and
                !< https://math.stackexchange.com/questions/33520/the-locus-of-two-perpendicular-tangents-to-a-given-ellipse/33555#33555
                LocalBeamX_deg = dsqrt((2.d0 * a * dcos(alpha))**2 + (2.d0 * b * dsin(alpha))**2)
                LocalBeamY_deg = dsqrt((2.d0 * a * dsin(alpha))**2 + (2.d0 * b * dcos(alpha))**2)
            endif
            LocalBeamX_deg = LocalBeamX_deg * BeamMultiplication
            LocalBeamY_deg = LocalBeamY_deg * BeamMultiplication


            ! Debug:
            ! print*," "
            ! print*,"LocalBeamX_deg = ", LocalBeamX_deg
            ! print*,"LocalBeamY_deg = ", LocalBeamY_deg


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< compute size of a pixel in arsec
            SizeOfPixel_deg = 1.d0


            !< NumY > NumX
            if (NumY > NumX .and. LocalBeamX_deg > LocalBeamY_deg) then
                SizeOfPixel_deg = LocalBeamX_deg / NumX

            elseif (NumY > NumX .and. LocalBeamX_deg == LocalBeamY_deg) then
                SizeOfPixel_deg = LocalBeamX_deg / NumX

            elseif (NumY > NumX .and. LocalBeamX_deg < LocalBeamY_deg) then
                SizeOfPixel_deg = LocalBeamY_deg / NumY
                if (SizeOfPixel_deg * NumX > LocalBeamX_deg) then
                    SizeOfPixel_deg = LocalBeamX_deg / NumX
                endif


            !< NumY == NumX
            elseif (NumY == NumX .and. LocalBeamX_deg > LocalBeamY_deg) then
                SizeOfPixel_deg = LocalBeamX_deg / NumY

            elseif (NumY == NumX .and. LocalBeamX_deg == LocalBeamY_deg) then
                SizeOfPixel_deg = LocalBeamX_deg / NumY

            elseif (NumY == NumX .and. LocalBeamX_deg < LocalBeamY_deg) then
                SizeOfPixel_deg = LocalBeamY_deg / NumY


            !< NumY < NumX
            elseif (NumY < NumX .and. LocalBeamX_deg > LocalBeamY_deg) then
                SizeOfPixel_deg = LocalBeamX_deg / NumY
                if (SizeOfPixel_deg * NumY > LocalBeamY_deg) then
                    SizeOfPixel_deg = LocalBeamY_deg / NumY
                endif

            elseif (NumY < NumX .and. LocalBeamX_deg == LocalBeamY_deg) then
                SizeOfPixel_deg = LocalBeamX_deg / NumY

            elseif (NumY < NumX .and. LocalBeamX_deg < LocalBeamY_deg) then
                SizeOfPixel_deg = LocalBeamY_deg / NumY

            endif

            ! Debug:
            ! print*," "
            ! print*,"SizeOfPixel_deg = ", SizeOfPixel_deg


            !< determine the pixel coordinates (physical) of the center of the model map
            CenterMapX_phys = int(NumX / 2.0)
            CenterMapY_phys = int(NumY / 2.0)

            ! Debug:
            ! print*," "
            ! print*,"CenterMapX_phys = ", CenterMapX_phys
            ! print*,"CenterMapY_phys = ", CenterMapY_phys


            !<============================================================================================================================================
            !< determine model map for each component
            Do c = 1, TotalNumberComponents                                                 !< loop over all components


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< no geometry of current layer is defined
                if (GeometryFlag(c) == 0) then
                    LayerMap(:, :, c) = .true.


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< current layer is a circle
                elseif (GeometryFlag(c) == 1 .or. GeometryFlag(c) == 2) then


                    !< get parameter of circle in arcsec
                    Radius_deg = dabs(CopymyXCLASSParameter(14, c)) / 2.d0                   !< get radius of circle in arcsec
                    CenterX_deg = CopymyXCLASSParameter(15, c)                               !< get x-coord. of circle center in arcsec
                    CenterY_deg = CopymyXCLASSParameter(16, c)                               !< get y-coord. of circle center in arcsec


                    !< convert to pixel coordinates
                    Radius_phys = max0(1, idnint(dble(Radius_deg) / dble(SizeOfPixel_deg)))
                    CenterX_phys = CenterMapX_phys + idnint(CenterX_deg / SizeOfPixel_deg)
                    CenterY_phys = CenterMapY_phys + idnint(CenterY_deg / SizeOfPixel_deg)

                    ! Debug:
                    ! print*," "
                    ! print*,"CenterX_phys = ", CenterX_phys
                    ! print*,"CenterY_phys = ", CenterY_phys
                    ! print*,"SizeOfPixel_deg  = ", SizeOfPixel_deg
                    ! print*,"Radius_deg  = ", Radius_deg
                    ! print*,"Radius_phys = ", Radius_phys


                    !< use Bresenham's circle algorithm (ref.: https://en.wikipedia.org/wiki/Midpoint_circle_algorithm)
                    !< to determine all pixels which are covered by filled circle (taken from
                    !< https://rosettacode.org/wiki/Bitmap/Midpoint_circle_algorithm#Fortran
                    f = 1 - Radius_phys
                    ddf_x = 0
                    ddf_y = (-2) * Radius_phys
                    x = 0
                    y = Radius_phys


                    !< uncomment for unfilled circle
                    !    call SetPixel(CenterX_phys, (-1), CenterY_phys + Radius_phys, (-1), c, NumX, NumY)
                    !    call SetPixel(CenterX_phys, (-1), CenterY_phys - Radius_phys, (-1), c, NumX, NumY)
                    !    call SetPixel(CenterX_phys + Radius_phys, (-1), CenterY_phys, (-1), c, NumX, NumY)
                    !    call SetPixel(CenterX_phys - Radius_phys, (-1), CenterY_phys, (-1), c, NumX, NumY)


                    !< filled circle
                    call SetPixel((CenterX_phys - Radius_phys), (CenterX_phys + Radius_phys), CenterY_phys, (-1), c, NumX, NumY)


                    Do while (x < y)
                        if (f >= 0) then
                            y = y - 1
                            ddf_y = ddf_y + 2
                            f = f + ddf_y
                        endif
                        x = x + 1
                        ddf_x = ddf_x + 2
                        f = f + ddf_x + 1


                        !< uncomment for unfilled circle
                        !    call SetPixel(CenterX_phys + x, (-1), CenterY_phys + y, (-1), c, NumX, NumY)
                        !    call SetPixel(CenterX_phys - x, (-1), CenterY_phys + y, (-1), c, NumX, NumY)
                        !    call SetPixel(CenterX_phys + x, (-1), CenterY_phys - y, (-1), c, NumX, NumY)
                        !    call SetPixel(CenterX_phys - x, (-1), CenterY_phys - y, (-1), c, NumX, NumY)
                        !    call SetPixel(CenterX_phys + y, (-1), CenterY_phys + x, (-1), c, NumX, NumY)
                        !    call SetPixel(CenterX_phys - y, (-1), CenterY_phys + x, (-1), c, NumX, NumY)
                        !    call SetPixel(CenterX_phys + y, (-1), CenterY_phys - x, (-1), c, NumX, NumY)
                        !    call SetPixel(CenterX_phys - y, (-1), CenterY_phys - x, (-1), c, NumX, NumY)


                        !< filled circle
                        call SetPixel((CenterX_phys - x), (CenterX_phys + x), (CenterY_phys + y), (-1), c, NumX, NumY)
                        call SetPixel((CenterX_phys - x), (CenterX_phys + x), (CenterY_phys - y), (-1), c, NumX, NumY)
                        call SetPixel((CenterX_phys - y), (CenterX_phys + y), (CenterY_phys + x), (-1), c, NumX, NumY)
                        call SetPixel((CenterX_phys - y), (CenterX_phys + y), (CenterY_phys - x), (-1), c, NumX, NumY)
                    end Do


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< current layer is a square
                elseif (GeometryFlag(c) == 3 .or. GeometryFlag(c) == 4) then


                    !< get parameter of circle in arcsec
                    BoxHeight_deg = CopymyXCLASSParameter(14, c)                             !< get height of square
                    CenterX_deg = CopymyXCLASSParameter(15, c)                               !< get x-coord. of circle center in arcsec
                    CenterY_deg = CopymyXCLASSParameter(16, c)                               !< get y-coord. of circle center in arcsec


                    !< convert to pixel coordinates
                    BoxHeight_phys = max0(1, int(BoxHeight_deg / SizeOfPixel_deg))
                    CenterX_phys = CenterMapX_phys + int(CenterX_deg / SizeOfPixel_deg)
                    CenterY_phys = CenterMapY_phys + int(CenterY_deg / SizeOfPixel_deg)


                    !< determine indices of lower left and upper right corner
                    !    x1 = min(max(0, CenterX_phys + int(BoxHeight_phys / 2.0)), NumX)
                    !    x2 = min(max(0, CenterX_phys - int(BoxHeight_phys / 2.0)), NumX)
                    !    y1 = min(max(0, CenterY_phys + int(BoxHeight_phys / 2.0)), NumY)
                    !    y2 = min(max(0, CenterY_phys - int(BoxHeight_phys / 2.0)), NumY)
                    !    LayerMap(y1:x1, y2:x2, c) = .true.


                    call SetPixel((CenterX_phys - int(BoxHeight_phys / 2.0)), (CenterX_phys + int(BoxHeight_phys / 2.0)), &
                                  (CenterY_phys - int(BoxHeight_phys / 2.0)), (CenterY_phys + int(BoxHeight_phys / 2.0)), &
                                  c, NumX, NumY)


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< current layer is a ?
                !elseif (GeometryFlag(c) == 5 .or. GeometryFlag(c) == 6) then




                endif

                ! Debug:
                ! print*," "
                ! print*,"c = ", c
                ! print*,"LayerMap(:, :, c) = ", LayerMap(:, :, c)


                !<########################################################################################################################################
                !< Only used by myXCLASS function!
                !<########################################################################################################################################
                !< write current model map for current component to log-file
                if (LogFlag) then


                    !< get desciption of beam layer
                    if (c == 1) then
                        a1 = max0(1, int(a / SizeOfPixel_deg))
                        b1 = max0(1, int(b / SizeOfPixel_deg))

                        ! Debug:
                        ! print*," "
                        ! print*,"SizeOfPixel_deg = ", SizeOfPixel_deg
                        ! print*,"a, a1 = ", a, a1
                        ! print*,"b, b1 = ", b, b1


                        !< draw ellipse
                        call DrawEllipse(CenterMapX_phys, CenterMapY_phys, a1, b1, alpha, NumX, NumY, BeamLayer)


                        !< write legend to log file
                        write(2224,'(" ")')
                        write(2224,'(" ")')
                        write(2224,'("Sub-beam description: Model Map")')
                        write(2224,'(" ")')
                        write(2224,'(" ")')
                        write(2224,'(7x, "Legend:")')
                        write(2224,'(" ")')
                        write(2224,'(7x, A1, ".", A1,":  Model pixel is NOT taken into account")') char(34), char(34)
                        write(2224,'(7x, A1, "#", A1,":  Model pixel IS taken into account")') char(34), char(34)
                        write(2224,'(7x, A1, "x", A1,":  Model pixel indicates the beam size (and pixel is NOT taken into &
                                            &account)")') char(34), char(34)
                        write(2224,'(7x, A1, "X", A1,":  Model pixel indicates the beam size (and pixel IS taken into account)")')&
                                            &char(34), char(34)
                    endif


                    !< get some information of current component
                    j = CopyCompMoleculeIndex(c)                                            !< get molecule index for current component
                    CurrMoleculeName = trim(adjustl(MoleculeNames(j)))                      !< get current molecule name


                    !< determine current comp. index and total number of component of current molecule
                    ncomp = 0
                    n = 0
                    Do i = 1, TotalNumberComponents
                        if (CopyCompMoleculeIndex(i) == CopyCompMoleculeIndex(c)) then
                            ncomp = ncomp + 1
                            if (i <= c) then
                                n = n + 1
                            endif
                        endif
                    end Do
                    write(numberstring1, '(I30)') n                                         !< convert integer to string


                    !< write information of current layer to output file
                    write(2224,'(" ")')
                    write(2224,'(" ")')
                    write(2224,'(7x, "Model map for ", A, ", component ", A, ":")') &
                                trim(adjustl(CurrMoleculeName)), trim(adjustl(numberstring1))
                    write(2224,'(" ")')
                    write(2224,'(6x, " ", $)')
                    Do ModelIDy = 1, NumberModelPixelYY                                     !< loop over model pixels along y-direction
                        Do ModelIDx = 1, NumberModelPixelXX                                 !< loop over model pixels along x-direction


                            !< indicate beam size
                            if (BeamLayer(ModelIDx, ModelIDy) .and. (.not. LayerMap(ModelIDx, ModelIDy, c))) then
                                write(2224,'("x", $)')
                            elseif (BeamLayer(ModelIDx, ModelIDy) .and. LayerMap(ModelIDx, ModelIDy, c)) then
                                write(2224,'("X", $)')


                            !< indicate layer
                            else
                                if (LayerMap(ModelIDx, ModelIDy, c)) then
                                    write(2224,'("#", $)')
                                else
                                    write(2224,'(".", $)')
                                endif
                            endif
                        end Do                                                              !< ModelIDx: loop over model pixels along x-direction
                        write(2224,'(" ")')
                        write(2224,'(6x, " ", $)')
                    end Do                                                                  !< ModelIDy: loop over model pixels along y-direction
                    write(2224,'(" ")')
                endif
            end Do                                                                          !< c: loop over all components


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< check which component combinations are really needed
            Do ModelIDy = 1, NumberModelPixelYY                                             !< loop over model pixels along y-direction
                Do ModelIDx = 1, NumberModelPixelXX                                         !< loop over model pixels along x-direction
                    if (any(LayerMap(ModelIDx, ModelIDy, :))) then                          !< ignore pixel which contains no component


                        !< check, if current configuration was used before
                        c = 0
                        IncludeFlag = .true.
                        Do i = 1, NumberConfig                                              !< loop over already known configurations
                            j = 0
                            Do n = 1, TotalNumberComponents                                 !< loop over all layers
                                if (ConfigList(i, n) .EQV. LayerMap(ModelIDx, ModelIDy, n)) then
                                    j = j + 1
                                else
                                    exit
                                endif
                            end Do                                                          !< n: loop over all layers
                            if (j == TotalNumberComponents) then
                                IncludeFlag = .false.
                                c = i                                                       !< store configuration
                                exit
                            endif
                        end Do                                                              !< i: loop over already known configurations
                        if (IncludeFlag) then
                            NumberConfig = NumberConfig + 1
                            ConfigList(NumberConfig, :) = LayerMap(ModelIDx, ModelIDy, :)
                            ConfigIndex(ModelIDx, ModelIDy) = NumberConfig
                        else
                            ConfigIndex(ModelIDx, ModelIDy) = c
                        endif

                        ! Debug:
                        ! print*,"ModelIDy = ", ModelIDy
                        ! print*,"ModelIDx = ", ModelIDx
                        ! print*,"ConfigIndex(ModelIDx, ModelIDy) = ", ConfigIndex(ModelIDx, ModelIDy)
                    endif
                end Do                                                                      !< ModelIDx: loop over model pixels along x-direction

                ! Debug:
                ! write(33333, *) ConfigIndex(:, ModelIDy)
            end Do                                                                          !< ModelIDy: loop over model pixels along y-direction

            ! Debug:
            ! print*," "
            ! print*,"NumberConfig = ", NumberConfig
            !    Do i = 1, NumberConfig
            !        print*,"ConfigList(i, :) = ", ConfigList(i, :)
            !    end Do


            !< we're done
            return
        end subroutine CreatePixelMap


        !>************************************************************************************************************************************************
        !> subroutine: ComputeGaussianBeamMap
        !>
        !> compute map describing elliptical rotated Gaussian beam
        !>
        !>
        !> input variables:     l:                              current frequency range index
        !>                      NumX:                           number of model pixels along x-direction
        !>                      NumY:                           number of model pixels along y-direction
        !>
        !> output variables:    chi2Value:                      chi2 value for parameter vector
        !>                      ModelFuncList:                  model function values
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2018-03-24
        !>
        subroutine ComputeGaussianBeamMap(l, NumX, NumY)

            implicit none
            integer :: l                                                                    !< current range index
            integer :: NumX, NumY                                                           !< number of model pixels along x- and y-direction
            integer :: i, j, nx1, nx2, ny2                                                  !< working variables for FFT
            integer :: nx, ny, ndim                                                         !< working variables for FFT
            integer :: FirstIndex                                                           !< index for first freq. data point
            integer :: LastIndex                                                            !< index for last freq. data point
            integer, dimension(2) :: nn                                                     !< working variables for FFT
            integer*8 :: planForward                                                        !< working variable for FFT
            real*8, parameter :: over = 100.d0                                              !< used for buffer overflow of exp-function
            real*8 :: a, b, c, i1, j1                                                       !< working variables for elliptical rotated Gaussian
            real*8 :: fact, f0, f1, volume, VolumePart                                      !< working variables for FFT
            real*8 :: amaj, amin, bmaj, bmin                                                !< working variables for FFT
            real*8 :: rpa                                                                   !< rotation angle in degree
            real*8, dimension(NumX, NumY) :: beam                                           !< working variables for FFT
            complex*16, dimension(NumX, NumY) :: in                                         !< working variable for FFT (input array)
            integer :: MethodFlag                                                           !< define method of convolution


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< compute parameters for convolution with elliptical, rotated Gaussian


            !< get beam angle and compute related flags
            rpa = TelescopeBPA(l) * pi / 180.d0


            !< get major and minor FWHM and compute related variances
            if (dabs(TelescopeSize(l)) > tiny(1.d0)) then
                if (InterFlag(l)) then
                    bmaj = TelescopeSize(l)
                else
                    FirstIndex = DataPointIndexFreqRange(l, 1)                              !< get index of first freq. point in 'ObservationalDataList'
                    LastIndex = DataPointIndexFreqRange(l, 2)                               !< get index of last freq. point in 'ObservationalDataList'
                    f0 = ObservationalDataList(FirstIndex, 1)                               !< get first frequency of current freq. range
                    f1 = ObservationalDataList(LastIndex, 1)                                !< get last frequency of current freq. range
                    bmaj = 1.22d-3 * ckms / (dmin1(f0, f1) * TelescopeSize(l)) * (180.d0 / pi) * 3600.d0
                                                                                            !< compute telescope FWHM in arcsec
                endif
                bmin = bmaj
            elseif (dabs(TelescopeBMIN(l) - TelescopeBMAJ(l)) < tiny(1.d0) .and. dabs(TelescopeBMIN(l)) > tiny(1.d0)) then
                bmaj = TelescopeBMIN(l)
                bmin = bmaj                                                                 !< set y-FWHM equal to x-FWHM
            else
                bmaj = TelescopeBMAJ(l)
                bmin = TelescopeBMIN(l)
            endif
            amin = bmin / (2.d0 * dsqrt(2.d0 * dlog(2.d0)))                                 !< sigma_x
            amaj = bmaj / (2.d0 * dsqrt(2.d0 * dlog(2.d0)))                                 !< sigma_y

            ! Debug:
            ! print*," "
            ! print*,"bmaj = ", bmaj
            ! print*,"bmin = ", bmin
            ! print*,"amaj = ", amaj
            ! print*,"amin = ", amin
            ! print*,"NumX = ", NumX
            ! print*,"NumY = ", NumY
            ! print*,"rpa = ", rpa
            ! print*,"BeamMultiplication = ", BeamMultiplication


            !< compute parameters for rotated elliptical gaussian
            !< expressions taken from https://en.wikipedia.org/wiki/Gaussian_function#Two-dimensional_Gaussian_function
            a = (dcos(rpa)**2 / (2.d0 * amin**2) + dsin(rpa)**2 / (2.d0 * amaj**2))
            b = (-dsin(2.d0 * rpa) / (4.d0 * amin**2) + dsin(2.d0 * rpa) / (4.d0 * amaj**2))
            c = (dsin(rpa)**2 / (2.d0 * amin**2) + dcos(rpa)**2 / (2.d0 * amaj**2))
            volume = ((2.d0 * pi * amin * amaj))

            ! Debug:
            ! print*," "
            ! print*,"a = ", a
            ! print*,"b = ", b
            ! print*,"c = ", c
            ! print*,"volume = ", volume
            ! print*,"SizeOfPixel_deg = ", SizeOfPixel_deg


            !< compute some abbreations
            nx = NumX
            ny = NumY
            ny2 = max0(1, int(ny / 2))
            nx2 = max0(1, int(nx / 2))
            nx1 = nx2 + 1
            ndim = 2
            nn(1) = nx
            nn(2) = ny


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< compute map of elliptical rotated Gaussian
            GausssianBeamMap = 0.d0
            VolumePart = 0.d0
            MethodFlag = 1


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< Gaussian like GILDAS
            if (MethodFlag == 1) then
                Do j = 1, ny2
                    Do i = 1, nx2
                        j1 = (dble(j - 1) * SizeOfPixel_deg)
                        i1 = (dble(i - 1) * SizeOfPixel_deg)
                        fact = -(a * i1**2 + 2.d0 * b * i1 * j1 + c * j1**2)
                        if (fact <= over) then
                            fact = dexp(fact)
                            GausssianBeamMap(i, j) = fact
                            VolumePart = VolumePart + fact
                        endif
                    end Do
                    Do i = nx1, nx
                        j1 = (dble(j - 1) * SizeOfPixel_deg)
                        i1 = (dble(i - nx - 1) * SizeOfPixel_deg)
                        fact = -(a * i1**2 + 2.d0 * b * i1 * j1 + c * j1**2)
                        if (fact <= over) then
                            fact = dexp(fact)
                            GausssianBeamMap(i, j) = fact
                            VolumePart = VolumePart + fact
                        endif
                   end Do
                end Do
                Do j = (ny2 + 1), ny
                    Do i = 1, nx2
                        j1 = (dble(j - ny - 1) * SizeOfPixel_deg)
                        i1 = (dble(i - 1) * SizeOfPixel_deg)
                        fact = -(a * i1**2 + 2.d0 * b * i1 * j1 + c * j1**2)
                        if (fact <= over) then
                            fact = dexp(fact)
                            GausssianBeamMap(i, j) = fact
                            VolumePart = VolumePart + fact
                        endif
                    end Do
                    Do i = nx1, nx
                        j1 = (dble(j - ny - 1) * SizeOfPixel_deg)
                        i1 = (dble(i - nx - 1) * SizeOfPixel_deg)
                        fact = -(a * i1**2 + 2.d0 * b * i1 * j1 + c * j1**2)
                        if (fact <= over) then
                            fact = dexp(fact)
                            GausssianBeamMap(i, j) = fact
                            VolumePart = VolumePart + fact
                        endif
                    end Do
                end Do


                !< normalize Gaussian map so that sum(GausssianBeamMap(:, :)) = 1
                ! GausssianBeamMap(:, :) = (1.d0 / (VolumePart)) * GausssianBeamMap(:, :)
                GausssianBeamMap(:, :) = (1.d0 / volume) * GausssianBeamMap(:, :)

                ! Debug:
                ! print*,' '
                ! print*,"sum_gauss = ", sum(GausssianBeamMap(:, :))
                ! print*,'planForward = ', planForward
                ! print*,'ndim = ', ndim
                ! print*,'nn = ', nn
                ! print*,'fftw_forward = ', fftw_forward
                ! print*,'FFTW_ESTIMATE = ', FFTW_ESTIMATE
                ! print*,"VolumePart = ", VolumePart
                ! print*,"VolumePart * 0.8 = ", VolumePart * 0.8
                ! print*,"2 pi sigma_x sigma_y = ", 2.d0 * pi * amin * amaj
                !    open(1232, file = "2dGaussianMap.dat")
                !    Do j = 1, ny
                !        write(1232, *) j/dble(ny), GausssianBeamMap(:, j)
                !    end Do
                !    close(1232)
                !    beam = 0.d0
                !    VolumePart = 0.d0
                !    Do j = 1, ny
                !        Do i = 1, nx
                !            j1 = (dble(j - ny2 - 1) * SizeOfPixel_deg)
                !            i1 = (dble(i - nx2 - 1) * SizeOfPixel_deg)
                !            fact = (a * i1**2 + 2.d0 * b * i1 * j1 + c * j1**2)
                !            if (fact <= over) then
                !                fact = dexp(-fact)
                !                beam(i, j) = fact
                !                VolumePart = VolumePart + fact
                !            endif
                !        end Do
                !        write(99999, *) beam(:, j)
                !    end Do
                !    open(99999, file = "Re-Gauss-map.dat")
                !    Do j = 1, ny
                !        write(99999, *) GausssianBeamMap(:, j)
                !    end Do
                !    close(99999)


                !< compute FT of beam
                in(:, :) = dcmplx(GausssianBeamMap(:, :), 0.d0)


                !< OpenMP:
                if (ParallezitionMethod == 1) then
                    !$omp critical
                    call dfftw_plan_dft(planForward, ndim, nn, in, in, fftw_forward, FFTW_ESTIMATE)
                    !$omp end critical
                    call dfftw_execute(planForward)
                    !$omp critical
                    call dfftw_destroy_plan(planForward)
                    !$omp end critical


                !< OpenMPI:
                else
                    call dfftw_plan_dft(planForward, ndim, nn, in, in, fftw_forward, FFTW_ESTIMATE)
                    call dfftw_execute(planForward)
                    call dfftw_destroy_plan(planForward)
                endif
                GausssianBeamMap(:, :) = dreal(in(:, :))

                ! Debug:
                !    Do j = 1, ny
                !        write(88888, *) dreal(in(:, j))
                !        write(99999, *) dimag(in(:, j))
                !    end Do


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< analytical expression
            elseif (MethodFlag == 2) then


                !< compute map with elliptical rotated Gaussian (normal indexing)
                if (dabs(amin - amaj) > tiny(1.d0)) then
                    beam = 0.d0
                    Do j = 1, ny
                        Do i = 1, nx
                            i1 = (dble(i - nx2 - 1) * SizeOfPixel_deg)
                            j1 = (dble(j - ny2 - 1) * SizeOfPixel_deg)


                            !< compute exponent for gaussian
                            if (dabs(b) > tiny(1.d0)) then
                                fact = ((c * i1**2 - 2.d0 * b * i1 * j1 + a * j1**2) / (4.d0 * b**2 - 4.d0 * a * c))
                            else
                                fact = -(c * i1**2 + a * j1**2) / (4.d0 * a * c)
                            endif
                            if (fact <= over) then
                                fact = dexp(fact)


                                !< normalize gaussian
                                if (dabs(a) > tiny(1.d0)) then
                                    if (dabs(b) > tiny(1.d0)) then
                                        beam(i, j) = fact / (4.d0 * dsqrt(a) * dsqrt(-(b**2 / a) + c) * pi * amin * amaj)
                                    else
                                        beam(i, j) = fact / (4.d0 * dsqrt(a) * dsqrt(c) * pi * amin * amaj)
                                    endif
                                else
                                    beam(i, j) = fact / (4.d0 * dsqrt(-(b**2)) * pi * amin * amaj)
                                endif
                            endif
                        end Do
                    end Do


                !< compute map with circle-shaped rotated Gaussian (normal indexing) using analytical expression
                else
                    VolumePart = 0.d0
                    beam = 0.d0
                    Do j = 1, ny
                        Do i = 1, nx
                            i1 = (dble(i - nx2 - 1) * SizeOfPixel_deg)
                            j1 = (dble(j - ny2 - 1) * SizeOfPixel_deg)
                            fact = -((i1**2 + j1**2) * amin * amaj) / 2.d0
                            ! fact = -((i1**2 + j1**2) * pi**2 * amin * amaj)
                            if (fact <= over) then
                                ! beam(i, j) = dexp(fact) * (pi / amin * amaj)
                                beam(i, j) = dexp(fact)         !/ dsqrt(2.d0 * pi)
                                VolumePart = VolumePart + beam(i, j)
                            endif
                        end Do
                    end Do
                endif


                !< need to shift the kernel so that, e.g., [0,0,1,0] -> [1,0,0,0] = unity
                fact = 1.d0 / (VolumePart * SizeOfPixel_deg**2)
                Do j = 1, (ny2 - 1)
                    Do i = 1, (nx2 - 1)
                        GausssianBeamMap(i + nx - nx2 + 1, j + ny - ny2 + 1) = beam(i, j) * fact
                    end Do
                    Do i = nx2, nx
                        GausssianBeamMap(i - nx2 + 1, j + ny - ny2 + 1) = beam(i, j) * fact
                    end Do
                end Do
                Do j = ny2, ny
                    Do i = 1, (nx2 - 1)
                        GausssianBeamMap(i + nx - nx2 + 1, j - ny2 + 1) = beam(i, j) * fact
                    end Do
                    Do i = nx2, nx
                        GausssianBeamMap(i - nx2 + 1, j - ny2 + 1) = beam(i, j) * fact
                    end Do
                end Do


            !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !< standard Gaussian
            elseif (MethodFlag == 3) then


                !< compute map with elliptical rotated Gaussian (normal indexing)
                beam = 0.d0
                VolumePart = 0.d0
                Do j = 1, ny
                    Do i = 1, nx
                        i1 = (dble(i - nx2 - 1) * SizeOfPixel_deg)
                        j1 = (dble(j - ny2 - 1) * SizeOfPixel_deg)
                        fact = (a * i1**2 + 2.d0 * b * i1 * j1 + c * j1**2)
                        if (fact <= over) then
                            fact = dexp(-fact)
                            beam(i, j) = fact
                            VolumePart = VolumePart + fact
                        endif
                    end Do
                end Do


                !< need to shift the kernel so that, e.g., [0,0,1,0] -> [1,0,0,0] = unity
                fact = 1.d0     !(1.d0 / (VolumePart * SizeOfPixel_deg**2))
                Do j = 1, (ny2 - 1)
                    Do i = 1, (nx2 - 1)
                        GausssianBeamMap(i + nx - nx2 + 1, j + ny - ny2 + 1) = beam(i, j) * fact
                    end Do
                    Do i = nx2, nx
                        GausssianBeamMap(i - nx2 + 1, j + ny - ny2 + 1) = beam(i, j) * fact
                    end Do
                end Do
                Do j = ny2, ny
                    Do i = 1, (nx2 - 1)
                        GausssianBeamMap(i + nx - nx2 + 1, j - ny2 + 1) = beam(i, j) * fact
                    end Do
                    Do i = nx2, nx
                        GausssianBeamMap(i - nx2 + 1, j - ny2 + 1) = beam(i, j) * fact
                    end Do
                end Do


                !< normalize Gaussian map so that sum(GausssianBeamMap(:, :)) = 1
                GausssianBeamMap(:, :) = (1.d0 / (VolumePart)) * GausssianBeamMap(:, :)


                !< compute FT of beam
                in(:, :) = dcmplx(GausssianBeamMap(:, :), 0.d0)


                !< OpenMP:
                if (ParallezitionMethod == 1) then
                    !$omp critical
                    call dfftw_plan_dft(planForward, ndim, nn, in, in, fftw_forward, FFTW_ESTIMATE)
                    !$omp end critical
                    call dfftw_execute(planForward)
                    !$omp critical
                    call dfftw_destroy_plan(planForward)
                    !$omp end critical


                !< OpenMPI:
                else
                    call dfftw_plan_dft(planForward, ndim, nn, in, in, fftw_forward, FFTW_ESTIMATE)
                    call dfftw_execute(planForward)
                    call dfftw_destroy_plan(planForward)
                endif
                GausssianBeamMap(:, :) = dreal(in(:, :))
            endif

            ! Debug:
            !    open(99999, file = "FT-Re-Gauss-map.dat")
            !    Do j = 1, ny
            !        write(99999, *) GausssianBeamMap(:, j)
            !    end Do
            !    close(99999)



            !< we're done
            return
        end subroutine ComputeGaussianBeamMap


        !>************************************************************************************************************************************************
        !> subroutine: ConvolveModelMap
        !>
        !> convolve model map with beam
        !>
        !>
        !> input variables:     ModeID:                         convolution mode
        !>                      l:                              current frequency range index
        !>                      NumDataPoints:                  total number of data points
        !>                      LocalNumFreqPoints:             number of data points for current freq. range
        !>                      NumX:                           number of model pixels along x-direction
        !>                      NumY:                           number of model pixels along y-direction
        !>                      LocalModelFunctionFlag:         flag for indicating if model func. values are stored or not
        !>                      MolID:                          molecule index (only used by myXCLASS function)
        !>                      NumberConfig:                   current number of configurations
        !>                      ModelPixelSpectra:              local spectrum
        !>
        !> output variables:    chi2Value:                      chi2 value for parameter vector
        !>                      ModelFuncList:                  model function values
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2018-03-09
        !>
        subroutine ConvolveModelMap(chi2Value, ModelFuncList, ModeID, l, NumDataPoints, LocalNumFreqPoints, NumX, NumY, &
                                    LocalModelFunctionFlag, MolID, NumberConfig, ModelPixelSpectra)

            implicit none
            integer :: ModeID                                                               !< convolution mode
            integer :: l                                                                    !< current range index
            integer :: NumDataPoints                                                        !< total number of data points
            integer :: ModelIDy, ModelIDx                                                   !< loop variables for sub-beam description
            integer :: con                                                                  !< local configuration
            integer :: NumX, NumY                                                           !< number of model pixels along x- and y-direction
            integer :: FirstIndex                                                           !< index for first freq. data point
            integer :: LastIndex                                                            !< index for last freq. data point
            integer :: FreqIndex                                                            !< local freq. index
            integer :: ObsFreqIndex                                                         !< local obs. freq. index
            integer :: LocalNumFreqPoints                                                   !< local number of freq. points
            integer :: MolID                                                                !< molecule index
            integer :: nx1, nx2, ny2                                                        !< working variables for convolution
            integer :: nx, ny, ndim                                                         !< working variables for convolution
            integer :: NumberConfig                                                         !< current number of configurations
            integer, dimension(2) :: nn                                                     !< working variables for convolution
            integer*8 :: planForward, planBackward                                          !< working variable for FFT
            real*8 :: ObsFreq                                                               !< frequency point in obs. data file
            real*8 :: j_back                                                                !< brightness for background temperature
            real*8 :: chi2Value                                                             !< chi2 value for parameter vector
            real*8 :: LocalIntensity                                                        !< working variable: local intensity
            real*8 :: sig2i                                                                 !< sigma factor (for weighting of obs. data)
            real*8, dimension(NumDataPoints) :: ModelFuncList                               !< output variable: model function values
            real*8, dimension(NumberConfig, LocalNumFreqPoints) :: ModelPixelSpectra        !< local spectrum
            complex*16, dimension(NumX, NumY, LocalNumFreqPoints) :: WorkingMap             !< working map
            complex*16, dimension(NumX, NumY) :: FTImage                                    !< working variable for FFT (input array)
            complex*16, dimension(NumX, NumY) :: ConvolvedImage                             !< working variable for convolved image
            character(len=30) :: numberstring1, numberstring3                               !< used for integer number to string conversion
            ! character(len=30) :: numberstring2                                            !< used for integer number to string conversion
            logical :: LocalModelFunctionFlag                                               !< flag for indicating if model func. values are stored or not

            ! Debug:
            ! print*,"ModeID = ", ModeID
            ! print*,"l = ", l
            ! print*,"NumDataPoints = ", NumDataPoints
            ! print*,"LocalNumFreqPoints = ", NumDataPoints
            ! print*,"NumX = ", NumX
            ! print*,"NumY = ", NumY
            ! print*,"SizeOfPixel_deg = ", SizeOfPixel_deg
            ! print*,"LocalModelFunctionFlag = ", LocalModelFunctionFlag
            ! print*,"MolID = ", MolID


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define some parameters for FT
            nx = NumX
            ny = NumY
            ny2 = max0(1, int(ny / 2))
            nx2 = max0(1, int(nx / 2))
            nx1 = nx2 + 1
            ndim = 2
            nn(1) = nx
            nn(2) = ny
            FirstIndex = DataPointIndexFreqRange(l, 1)                                      !< get index of first freq. point in 'ObservationalDataList'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< create model map with intensities of each pixel spectrum at all frequency points of the current range
            WorkingMap = dcmplx(0.d0, 0.d0)
            Do ModelIDy = 1, NumY                                                           !< loop over all pixels of model map along y-direction
                Do ModelIDx = 1, NumX                                                       !< loop over all pixels of model map along x-direction
                    con = ConfigIndex(ModelIDx, ModelIDy)
                    if (con > 0 .and. con <= NumberConfig) then
                        if (ModeID == 0) then                                               !< convole total intensities
                            WorkingMap(ModelIDx, ModelIDy, :) = dcmplx(ModelPixelSpectra(con, :), 0.d0)
                        elseif (ModeID > 0) then                                            !< convole intensities for each component
                            WorkingMap(ModelIDx, ModelIDy, :) = dcmplx(IntPerCompArraySubBeam(con, :, ModeID, MolID), 0.d0)
                        elseif (ModeID < 0) then                                            !< convole optical depths for each component
                            WorkingMap(ModelIDx, ModelIDy, :) = dcmplx(TauPerCompArraySubBeam(con, :, abs(ModeID), MolID), 0.d0)
                        endif
                    else
                        Do FreqIndex = 1, LocalNumFreqPoints
                            ObsFreqIndex = FirstIndex + FreqIndex - 1
                            ObsFreq = ObservationalDataList(ObsFreqIndex, 1)


                            !< add background continuum description
                            j_back = 0.d0
                            if (dabs(BackgroundTemperatureRange(l)) > tiny(1.d0)) then
                                j_back = BackgroundTemperatureRange(l) * (ObsFreq / StartFrequency(l))**TemperatureSlopeRange(l)
                            endif
                            WorkingMap(ModelIDx, ModelIDy, FreqIndex) = dcmplx(j_back, 0.d0)
                        end Do
                    endif
                end Do                                                                      !< ModelIDx: loop over all pixels of model map along x-dir.

                ! Debug:
                ! write(88888,*) dreal(WorkingMap(:, ModelIDy, 637))
                ! write(88888,*) ConfigIndex(:, ModelIDy)
            end Do                                                                          !< ModelIDy: loop over all pixels of model map along y-dir.
            if (printflag .and. LogFlag .and. ModeID == 0) then
                print '(" ")'
            endif


            !< print what you do
            if (printflag .and. LogFlag) then
                if (ModeID == 0) then
                    print '(A, "Compute FFT and convolution ..      ", $)', char(13)
                else
                    write(numberstring3, '(I30)') abs(ModeID)
                    if (ModeID > 0) then
                        print '(A, "Compute FFT and convolution of intensity of component ", A, " .. ", $)', char(13), &
                                trim(adjustl(numberstring3))
                    else
                        print '(A, "Compute FFT and convolution of opt. depth of component ", A, " .. ", $)', char(13), &
                                trim(adjustl(numberstring3))
                    endif
                endif
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< compute intensity
            FirstIndex = DataPointIndexFreqRange(l, 1)                                      !< get index of first freq. point in 'ObservationalDataList'
            LastIndex = DataPointIndexFreqRange(l, 2)                                       !< get index of last freq. point in 'ObservationalDataList'
            Do FreqIndex = 1, LocalNumFreqPoints                                            !< loop over frequency points of current range
                ObsFreqIndex = FirstIndex + FreqIndex - 1                                   !< compute index for ObservationalDataList array


                !< print what you do
                !if (printflag .and. LogFlag) then
                !    write(numberstring1, '(I30)') FreqIndex
                !    write(numberstring2, '(I30)') LocalNumFreqPoints
                !    if (ModeID == 0) then
                !        print '(A, "Compute FFT and convolution for frequency index (", A, "/", A, ") .. ", $)', char(13), &
                !                                                  trim(adjustl(numberstring1)), trim(adjustl(numberstring2))
                !    else
                !        write(numberstring3, '(I30)') abs(ModeID)
                !        if (ModeID > 0) then
                !            print '(A, "Compute FFT and convolution of intensity of component ", A, " for frequency index (", A, "/", A, ") .. ", $)', &
                !                                      char(13), trim(adjustl(numberstring3)), trim(adjustl(numberstring1)), trim(adjustl(numberstring2))
                !        else
                !            print '(A, "Compute FFT and convolution of opt. depth of component ", A, " for frequency index (", A, "/", A, ") .. ", $)', &
                !                                      char(13), trim(adjustl(numberstring3)), trim(adjustl(numberstring1)), trim(adjustl(numberstring2))
                !        endif
                !    endif
                !endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< 2d convolution with arbitrary rotated elliptical gaussian

                ! Debug:
                ! print*,"FreqIndex, sum(WorkingMap(:, :, FreqIndex)) = ", FreqIndex, sum(WorkingMap(:, :, FreqIndex))
                !    if (ModeID == 0) print*, FreqIndex, sum(WorkingMap(:, :, FreqIndex)), dreal(WorkingMap(nx/2, ny/2, FreqIndex))
                !    if (ModeID == 0) then
                !        write(numberstring1, '(I30)') FreqIndex
                !        open(99999, file = "map__" // trim(adjustl(numberstring1)) // ".dat")
                !        Do con = 1, ny
                !            write(99999, *) dreal(WorkingMap(:, con, FreqIndex))
                !        end Do
                !        close(99999)
                !        call system("gzip map__*.dat")
                !    endif


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< apply 2D Fourier transformation
                FTImage(:, :) = WorkingMap(:, :, FreqIndex) * SizeOfPixel_deg**2


                !< OpenMP:
                if (ParallezitionMethod == 1) then
                    if (FreqIndex == 1) then
                        !$omp critical
                        call dfftw_plan_dft(planForward, ndim, nn, FTImage, FTImage, fftw_forward, FFTW_ESTIMATE)
                        !$omp end critical
                    endif
                    call dfftw_execute(planForward)
                    if (FreqIndex == LocalNumFreqPoints) then
                        !$omp critical
                        call dfftw_destroy_plan(planForward)
                        !$omp end critical
                    endif


                !< OpenMPI
                else
                    if (FreqIndex == 1) call dfftw_plan_dft(planForward, ndim, nn, FTImage, FTImage, fftw_forward, FFTW_ESTIMATE)
                    call dfftw_execute(planForward)
                    if (FreqIndex == LocalNumFreqPoints) call dfftw_destroy_plan(planForward)
                endif

                ! Debug:
                !    if (ModeID == 0 .and. ObsFreqIndex == 636) then
                !        FTImage(:, :) = 1.d0 / dsqrt(dble(NumX * NumY)) * FTImage(:, :)
                !        Do con = 1, nx
                !            write(66666, *) con, dreal(FTImage(con, int(ny/2))), dimag(FTImage(con, int(ny/2)))
                !        end Do
                !        write(numberstring1, '(I30)') FreqIndex
                !        open(99999, file = "Re-FT-map__" // trim(adjustl(numberstring1)) // ".dat")
                !        Do con = 1, ny
                !            write(99999, *) dreal(FTImage(:, con))
                !        end Do
                !        close(99999)
                !        open(99999, file = "Im-FT-map__" // trim(adjustl(numberstring1)) // ".dat")
                !        Do con = 1, ny
                !            write(99999, *) dimag(FTImage(:, con))
                !        end Do
                !        close(99999)
                !        call system("gzip *map__*.dat")
                !    endif


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< Convolution: multiply FT of model map with map of elliptical rotated Gaussian
                ConvolvedImage(:, :) = GausssianBeamMap(:, :) * FTImage(:, :)


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< apply inverse 2D Fourier transformation to get intensity of central pixel


                !< OpenMP:
                if (ParallezitionMethod == 1) then
                    if (FreqIndex == 1) then
                        !$omp critical
                        call dfftw_plan_dft(planBackward, ndim, nn, ConvolvedImage, ConvolvedImage, fftw_backward, FFTW_ESTIMATE)
                        !$omp end critical
                    endif
                    call dfftw_execute(planBackward)
                    if (FreqIndex == LocalNumFreqPoints) then
                        !$omp critical
                        call dfftw_destroy_plan(planBackward)
                        !$omp end critical
                    endif


                !< OpenMPI
                else
                    if (FreqIndex == 1) call dfftw_plan_dft(planBackward, ndim, nn, ConvolvedImage, ConvolvedImage, &
                                                            fftw_backward, FFTW_ESTIMATE)
                    call dfftw_execute(planBackward)
                    if (FreqIndex == LocalNumFreqPoints) call dfftw_destroy_plan(planBackward)
                endif
                ConvolvedImage(:, :) = 1.d0 / (dble(NumX * NumY)) * ConvolvedImage(:, :)    !< normalize convolved image


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< get intensity at central pixel
                LocalIntensity = dreal(ConvolvedImage(nx2, ny2))


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< store results


                !< for total intensity store final intensity and compute chi^2 value
                if (ModeID == 0) then

                    ! Debug:
                    ! if (ObsFreqIndex==636) then
                    !     Do con = 1, nx
                    !         write(99999, *) con, dreal(ConvolvedImage(con, int(nx/2)))
                    !     end Do
                    ! endif


                    !< determine sigma (sig2i) factor
                    sig2i = 1.d0
                    if (ObservationalDataList(ObsFreqIndex, 3) /= 0.d0) then                !< do the experimental datas include errors
                        sig2i = 1.d0 / (ObservationalDataList(ObsFreqIndex, 3)**2)
                    endif


                    !< determine chi^2 value using local calculated intensity
                    LocalIntensity = dmin1(LocalIntensity, 1.d20)                           !< prevent overflow in intensity
                    if (LocalModelFunctionFlag) then                                        !< save model function
                        ModelFuncList(ObsFreqIndex) = LocalIntensity
                    endif
                    chi2Value = chi2Value + ((LocalIntensity - ObservationalDataList(ObsFreqIndex, 2))**2 * sig2i)


                    !< store convolved model map
                    if (AllOutputFilesFlag) then
                        IntTotalSubBeam(:, :, FreqIndex) = dreal(ConvolvedImage(1:NumX, 1:NumY))
                    endif


                !< store intensities per component
                elseif (ModeID > 0) then
                    IntPerCompArray(FreqIndex, ModeID, MolID) = LocalIntensity

                    ! Debug:
                    ! print*," "
                    ! print*,"FreqIndex, ModeID, MolID = ", FreqIndex, ModeID, MolID
                    ! print*,"LocalIntensity = ", LocalIntensity


                !< for total intensity store final intensity and compute chi^2 value
                elseif (ModeID < 0) then
                    TauPerCompArray(FreqIndex, abs(ModeID), MolID) = LocalIntensity
                endif
            end Do                                                                          !< FreqIndex: loop over frequency points of current range

            ! Debug:
            ! print*,"chi2Value = ", chi2Value


            !< print what you do
            if (printflag .and. LogFlag) then
                write(numberstring1, '(I30)') abs(ModeID)
                if (ModeID == 0) then
                    print '(A, "Convolution of total intensity is &
                            &finished!                                                            ", $)', char(13)
                elseif (ModeID > 0) then
                    print '(A, "Convolution of intensity of component ", A, " is finished!&
                            &                                             ", $)', char(13), &
                            trim(adjustl(numberstring1))
                elseif (ModeID < 0) then
                    print '(A, "Convolution of optical depth of component ", A, " is finished!&
                            &                                         ", $)', char(13), &
                            trim(adjustl(numberstring1))
                endif
            endif


            !< we're done
            return
        end subroutine ConvolveModelMap


        !>************************************************************************************************************************************************
        !> subroutine: PhenContDescription
        !>
        !> computes phen. continuum description
        !>
        !>
        !> input variables:     j_back:                         brightness for background temperature
        !>                      freq:                           current frequency
        !>                      LocalContPhen:                  parameter vector for phen. description
        !>
        !> output variables:    j_back:                         brightness for background temperature
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2018-12-14
        !>
        subroutine PhenContDescription(j_back, freq, LocalContPhen)

            implicit none
            real*8 :: j_back                                                                !< brightness for background temperature
            real*8 :: freq                                                                  !< current frequency
            real*8 :: freq_t                                                                !< current frequency as temperature
            real*8, dimension(6) :: LocalContPhen                                           !< parameter vector for phen. description

            ! Debug:
            ! print*,"j_back = ", j_back
            ! print*,"freq = ", freq
            ! print*,"LocalContPhen(1) = ", LocalContPhen(1)
            ! print*,"LocalContPhen(2) = ", LocalContPhen(2)
            ! print*,"LocalContPhen(3) = ", LocalContPhen(3)
            ! print*,"LocalContPhen(4) = ", LocalContPhen(4)
            ! print*,"LocalContPhen(5) = ", LocalContPhen(5)
            ! print*,"LocalContPhen(6) = ", LocalContPhen(6)


            !< model "standing wave"
            !< LocalContPhen(2): Amplitude_1 (in K)
            !< LocalContPhen(3): scale parameter
            !< LocalContPhen(4): phase_1
            !< LocalContPhen(5): Amplitude_2 (in K)
            !< LocalContPhen(6): phase_2
            if (LocalContPhen(1) == 1.d0) then
                j_back = j_back + (LocalContPhen(2) * dsin((freq * LocalContPhen(3)) + LocalContPhen(4)) &
                                + LocalContPhen(5) * dcos((freq * LocalContPhen(3)) + LocalContPhen(6)))


            !< model "black body contribution"
            !< LocalContPhen(2): temperature (in K)
            elseif (LocalContPhen(1) == 2.d0) then
                freq_t = MHz2Kelvin * freq                                                  !< convert freq. in temperature (Kelvin) = \frac{h \nu}{k_B}
                j_back = j_back + (freq_t / (ExpSave(freq_t / LocalContPhen(2)) - 1.d0))


            !< model "parameterized continuum description"
            !< LocalContPhen(2): reference frequency (in MHz)
            !< LocalContPhen(3): background temperature (in K)
            !< LocalContPhen(4): temperature slope
            !< LocalContPhen(5): last frequency (in MHz)
                elseif (LocalContPhen(1) == 3.d0) then
                if (LocalContPhen(5) == 0.d0 .or. (LocalContPhen(2) <= freq .and. freq <= LocalContPhen(5))) then
                    j_back = j_back + (LocalContPhen(3) * (freq / LocalContPhen(2))**LocalContPhen(4))
                endif
            endif

            ! Debug:
            ! print*,"j_back = ", j_back


            !< we're done
            return
        end subroutine PhenContDescription


        !>************************************************************************************************************************************************
        !> subroutine: CalcModelPixelSpectrum
        !>
        !> calculates the entire spectrum for a given model pixel
        !>
        !>
        !> input variables:     NumDataPoints:                  total number of data points
        !>                      TotalNumMol:                    total number of molecules
        !>                      NumComp:                        total number of components
        !>                      CopyCompMoleculeIndex:          local copy of CompMoleculeIndexOrig
        !>                      CopyIsoRatioConversionTable:    local copy of IsoRatioConversionTableOrig
        !>                      CopymyXCLASSParameter:          local copy of myXCLASSParameterOrig
        !>                      FirstIndex:                     index for first freq. data point
        !>                      LastIndex:                      index for last freq. data point
        !>                      l:                              current frequency range index
        !>                      LocalModelFunctionFlag:         flag for indicating if model func. values are stored or not
        !>                      LocalMaxNumTrans:               max. number of transitions (used for Non-LTE description)
        !>                      CurrentConfigID:                current unique configuration index (only used for sub-beam description)
        !>                      NumberConfig:                   total number of unique configurations (only used for sub-beam description)
        !>                      LocalPureDistances:             pure distances
        !>                      LocalDistanceOrderingArray:     component indices at each distance
        !>                      LocalNumberDistances:           number of distances
        !>
        !> output variables:    chi2Value:                      chi2 value for parameter vector
        !>                      ModelFuncList:                  model function values
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2018-02-06
        !>
        subroutine CalcModelPixelSpectrum(chi2Value, ModelFuncList, NumDataPoints, TotalNumMol, NumComp, CopyCompMoleculeIndex, &
                                          Q, CopyIsoRatioConversionTable, CopymyXCLASSParameter, FirstIndex, LastIndex, l, &
                                          LocalModelFunctionFlag, LocalMaxNumTrans, CurrentConfigID, NumberConfig, &
                                          LocalPureDistances, LocalDistanceOrderingArray, LocalNumberDistances)

            implicit none
            !< input parameters
            integer, parameter :: MaxFreqResampling = 100                                   !< max. number of frequency points for resampling
            integer :: NumDataPoints                                                        !< total number of data points
            integer :: NumComp                                                              !< total number of components
            integer :: TotalNumMol                                                          !< total number of molecules
            integer :: ccc, c2                                                              !< over-all current component index
            integer :: kk                                                                   !< used for handling of isotoplogues
            integer :: c                                                                    !< current component index
            integer :: l                                                                    !< current frequency range index
            integer :: i, j, m                                                              !< working variables
            integer :: FreqIndex                                                            !< index for frequency point
            integer :: FirstIndex                                                           !< index for first freq. data point
            integer :: LastIndex                                                            !< index for last freq. data point
            integer :: MoleculeIndex                                                        !< index for molecule
            integer :: MoleculeIndexLoop                                                    !< loop index for isotopolouges
            integer :: NumCompAtCurrDistance                                                !< number of components, which correspond to current distance
            integer :: CurrDistID                                                           !< index of current distance
            integer :: etaMaxIndex                                                          !< index of emission comp. with the largest beam fill. factor
            integer :: LocalMaxNumTrans                                                     !< max. number of transitions (used for Non-LTE description)
            integer :: CurrentConfigID                                                      !< current unique configuration index
            integer :: NumberConfig                                                         !< total number of unique configurations
            integer :: AddFreqIndex, fi                                                     !< (for channel int.) loop indices
            integer :: NumAdditionalFreq                                                    !< (for channel int.) number of additional freq. points
            integer :: AddFirst                                                             !< (for channel int.) first additional freq. point
            integer :: LocalNumberDistances                                                 !< (input) number of distances
            integer :: NumberMoleculeComponents                                             !< working variable: number of comp. describing moleceuls/RRLs
            integer, dimension(NumComp) :: CopyCompMoleculeIndex                            !< local copy of CompMoleculeIndexOrig
            integer, dimension(NumComp, 1) :: SortHelperArray2                              !< helper array for sort subroutine
            integer, dimension(NumComp, NumComp) :: LocalDistanceOrderingArray              !< (input) local copy of DistanceOrderingArray array
            real*8 :: NewFreqMin, NewFreqMax                                                !< working variables for min. and max. freq. of current range
            real*8 :: NewFreqMin1, NewFreqMax1                                              !< working variables for min. and max. freq. of current range
            real*8 :: val, val2                                                             !< working value
            real*8 :: Td                                                                    !< dust temperature (for myXCLASS only)
            real*8 :: j_back                                                                !< brightness for background temperature
            real*8 :: tau_l_em                                                              !< working variable: sum over all taus
            real*8 :: tau_l_ab                                                              !< working variable: sum over all taus
            real*8 :: tau_l_em_local, tau_l_ab_local                                        !< (for channel int.) backup of opacities
            real*8 :: tau_l_em_sum, tau_l_ab_sum                                            !< (for channel int.) sum of opacities
            real*8 :: ChannelIntensity                                                      !< calc. intensity for current channel
            real*8 :: chi2Value                                                             !< chi2 value for parameter vector
            real*8 :: CurrentDistance                                                       !< current distance
            real*8 :: Obsfreq                                                               !< observed frequency
            real*8 :: etaMax                                                                !< max. beam filling factor of all emission components
            real*8 :: Temp                                                                  !< kinetic temperature (molecules) or electronic temp. (RRLs)
            real*8 :: Ntot                                                                  !< column density (molecules) or emission measure (RRLs)
            real*8 :: v_width_1                                                             !< current velocity width 1 (km/s)
            real*8 :: v_width                                                               !< velocity width (km/s) for RADEX (km/s)
            real*8 :: min_v_width                                                           !< min. width of all components at a certain distance
            real*8 :: max_v_width                                                           !< max. width of all components at a certain distance
            real*8 :: v_off                                                                 !< current velocity offset (km/s)
            real*8 :: min_v_off, max_v_off                                                  !< min. and max. velocity offset (km/s) per layer
            real*8 :: beam_size, Local_beam_size                                            !< (local) beam size (eqn. 2 - 3)
            real*8 :: LocalIsoRatio                                                         !< local iso ratio
            real*8 :: sig2i                                                                 !< sigma factor (for weighting of obs. data)
            real*8 :: TotalLocalIntensity, BackgroundIntensity                              !< total calc. intensity for current channel
            real*8 :: LayerIntensity                                                        !< sumed intensity of each layer
            real*8 :: LocalIntensity                                                        !< intensities for curret comp.
            real*8 :: ChannelWidth                                                          !< (for channel int.) channel width
            real*8 :: FreqChannelFirst                                                      !< (for channel int.) first frequency of current channel
            real*8 :: FreqChannelLast                                                       !< (for channel int.) last frequency of current channel
            real*8 :: AddFreq                                                               !< (for channel int.) additional frequency
            real*8 :: vLocal                                                                !< working variable local velocity offset frequency
            real*8 :: vMin, vMax                                                            !< min. and max. offset velocity of each molecule
            real*8 :: x0, x1, f0, f1                                                        !< (for channel int.) working variables for integration
            real*8, dimension(6) :: LocalContPhen                                           !< parameter vector for phen. description
            real*8, dimension(NumComp) :: LastIntFreqPoint                                  !< (for channel int.) intensity of the last add. freq. point
            real*8, dimension(NumComp) :: LocalPureDistances                                !< (input) local copy of PureDistances array
            real*8, dimension(NumComp) :: SortHelperArray                                   !< helper array for sort subroutine
            real*8, dimension(MaxFreqResampling) :: AdditionalFreqList                      !< (for channel int.) list of additional freq. points
            real*8, dimension(MaxFreqResampling) :: IntFreqList                             !< (for channel int.) list of intensities
            real*8, dimension(NumDataPoints) :: ModelFuncList                               !< output variable: model function values
            real*8, dimension(16, NumComp) :: CopymyXCLASSParameter                         !< local copy of myXCLASSParameterOrig
            real*8, dimension(NumComp, TotalNumMol) :: Q                                    !< partition function at the rotation temperatures
            real*8, dimension(TotalNumMol, TotalNumMol) :: CopyIsoRatioConversionTable      !< local copy of IsoRatioConversionTableOrig
            character(len=30) :: numberstring1, numberstring2                               !< used for integer number to string conversion
            character(len=30) :: numberstring3, numberstring4                               !< used for integer number to string conversion
            logical :: LocalModelFunctionFlag                                               !< flag for indicating if model func. values are stored or not
            logical :: FirstFrequencyFlag                                                   !< flag indicating if current freq. points is first in range
            logical :: ReverseFreqFlag                                                      !< flag indicating reversed frequency ordering
            logical :: FastFlag                                                             !< flag for reduced screen output
            logical :: LocalFlag                                                            !< working variable
            logical :: TauResetFlag                                                         !< flag indicating that tau_em and tau_abs are cleared
            logical :: SameSizeFlag                                                         !< flag indicating that comp. at given dist. have equal size
            logical, dimension(NumDataPoints) :: FirstDataFlag                              !< indicates if first freq. of channel has to be calculated
                                                                                            !< or not
            ! Debug:
            ! print*," "
            ! print*,"NumDataPoints = ", NumDataPoints
            ! print*,"TotalNumMol = ", TotalNumMol
            ! print*,"NumComp = ", NumComp
            ! print*,"CopyCompMoleculeIndex = ", CopyCompMoleculeIndex
            ! print*,"Q = ", Q(1,:)
            ! print*,"CopyIsoRatioConversionTable = ", CopyIsoRatioConversionTable
            ! print*,"CopymyXCLASSParameter = ", CopymyXCLASSParameter(:,1)
            ! print*,"FirstIndex = ", FirstIndex
            ! print*,"LastIndex = ", LastIndex
            ! print*,"l = ", l
            ! print*,"LocalModelFunctionFlag = ", LocalModelFunctionFlag
            ! print*,"LocalMaxNumTrans = ", LocalMaxNumTrans
            ! print*,"CurrentConfigID = ", CurrentConfigID
            ! print*,"NumberConfig = ", NumberConfig


            !< determine min. and max. velocity offset
            NewFreqMin = 0.d0
            NewFreqMax = 0.d0
            vMin = 1.d99
            vMax = -1.d99
            Do j = 1, TotalNumberComponents                                                 !< loop over all components
                i = CompMoleculeIndex(j)                                                    !< get molecule index for current component
                MoleculeIndex = CopyCompMoleculeIndex(j)                                    !< get molecule index
                LocalFlag = .false.
                if (IsoFlag) then                                                           !< check, if current comp. is used for isotopologues as well
                    if (IsoRatioConversionTable(i, MoleculeIndex) /= 0.d0) then
                        LocalFlag = .true.
                    endif
                elseif (i == MoleculeIndex) then
                    LocalFlag = .true.
                endif
                if (LocalFlag) then


                    !< for myXCLASS function or not-fitted parameters consider the velocity offset and width parameters directly
                    !< (vOffLimits(j, 1) describes if current parameter is fitted (=1) or not (=0))
                    if (LogFlag .or. vOffLimits(j, 1) == 0) then


                        !< determine min. and max. velocity offsets for current molecule
                        vLocal = myXCLASSParameter(6, j)
                        if (vLocal < vMin) then
                            vMin = vLocal
                        endif
                        if (vLocal > vMax) then
                            vMax = vLocal
                        endif


                    !< for myXCLASSFit function consider the lower and upper limits of velocity offset and width parameters
                    else


                        !< determine min. and max. velocity offsets for current molecule
                        vLocal = vOffLimits(j, 2)                                           !< get lower limit of velocity offset
                        if (vLocal < vMin) then
                            vMin = vLocal
                        endif
                        vLocal = vOffLimits(j, 3)                                           !< get upper limit of velocity offset
                        if (vLocal > vMax) then
                            vMax = vLocal
                        endif
                    endif
                endif
            end Do                                                                          !< j: loop over all components
            if (vMin == 1.d99) then
                vMin = 0.d0
            endif
            if (vMax == -1.d99) then
                vMax = 0.d0
            endif

            ! Debug:
            ! print*," "
            ! print*,"vMin = ", vMin
            ! print*,"vMax = ", vMax


            !< define new min. and max. frequencies for current range if user-defined vLSR is unequal zero
            if (GlobalvLSR(l) /= 0.d0) then
                NewFreqMin = (StartFrequency(l) * (1.d0 - GlobalvLSR(l) / ckms))
                NewFreqMax = (EndFrequency(l) * (1.d0 - GlobalvLSR(l) / ckms))
            else
                NewFreqMin = StartFrequency(l)
                NewFreqMax = EndFrequency(l)
            endif


            !< shift the frequency range
            NewFreqMin1 = (NewFreqMin * (1.d0 - dmax1(vMin, vMax) / ckms))
            NewFreqMax1 = (NewFreqMax * (1.d0 - dmin1(vMin, vMax) / ckms))
            if (RangeExpansionFlag) then
                NewFreqMin = dmin1(NewFreqMin, (NewFreqMin1 * 0.9d0))
                NewFreqMax = dmax1(NewFreqMax, (NewFreqMax1 * 1.1d0))
            else
                NewFreqMin = dmin1(NewFreqMin, NewFreqMin1)
                NewFreqMax = dmax1(NewFreqMax, NewFreqMax1)
            endif
            NewFreqMin1 = dmin1(NewFreqMin, NewFreqMax)
            NewFreqMax1 = dmax1(NewFreqMin, NewFreqMax)


            ! Debug:
            ! print*," "
            ! print*,"StartFrequency(l) = ", StartFrequency(l)
            ! print*,"EndFrequency(l) = ", EndFrequency(l)
            ! print*,"EndFrequency(l) - StartFrequency(l) = ", EndFrequency(l) - StartFrequency(l)
            ! print*,"NewFreqMin1 = ", NewFreqMin1
            ! print*,"NewFreqMax1 = ", NewFreqMax1
            ! print*,"NewFreqMax1 - NewFreqMin1 = ", NewFreqMax1 - NewFreqMin1


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< start calculation: loop over all no-equal distances
            ModelFuncList(FirstIndex:LastIndex) = 0.d0
            Do CurrDistID = 1, LocalNumberDistances                                         !< loop over all pure distances
                CurrentDistance = LocalPureDistances(CurrDistID)                            !< store current distance
                NumCompAtCurrDistance = count(LocalDistanceOrderingArray(CurrDistID, :) /= 0)   !< determine the number of components, which correspond
                                                                                            !< to the current distance

                ! Debug:
                ! print*,"CurrDistID = ", CurrDistID
                ! print*,"CurrentDistance = ", CurrentDistance
                ! print*,"NumCompAtCurrDistance = ", NumCompAtCurrDistance


                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                !< if local-overlap is not taken into account and if there are more then one components at current distance and if
                !< at least one component describes a continuum contribution, reorder the components for the current distance so that
                !< components describing continuum contributions are modeled first and that the component with the largest source size
                !< is modeled directly afterwards
                if ((.not. LocalOverlapFlag) .and. NumCompAtCurrDistance > 1 &
                    .and. maxval(LocalDistanceOrderingArray(CurrDistID, :)) > 2) then

                    ! Debug:
                    ! print*," "
                    ! print*,"LocalDistanceOrderingArray(CurrDistID, :) = ", LocalDistanceOrderingArray(CurrDistID, :)


                    !< reorder components so that components describing continuum contributions are modeled first
                    SortHelperArray = 0.d0
                    SortHelperArray2(:, 1) = 0
                    c = 1
                    Do j = 1, NumCompAtCurrDistance                                         !< loop over all comp. which correspond to current dist.
                        m = KindOfMolecule(LocalDistanceOrderingArray(CurrDistID, j))
                        SortHelperArray(j) = (-1.d0) * m                                    !< "-1" to get reversed order
                        if (j > 1 .and. c == 1) then
                            if ((SortHelperArray(j) - SortHelperArray(j - 1)) < 0.d0) then
                                c = 2
                            endif
                        endif
                    end Do                                                                  !< j: loop over all components at current distance
                    if (c == 2) then                                                        !< is it neccessary to sort list of component?
                        SortHelperArray2(:, 1) = LocalDistanceOrderingArray(CurrDistID, :)
                        call HeapSort(NumComp, 1, SortHelperArray(:), SortHelperArray2(:, 1))
                    endif

                    ! Debug:
                    ! print*,"SortHelperArray = ", SortHelperArray
                    ! print*,"LocalDistanceOrderingArray(CurrDistID, :) = ", LocalDistanceOrderingArray(CurrDistID, :)


                    !< find component describing a molecule / RRL which has the largest source size of all molecules at this distance
                    etaMax = 0.d0
                    etaMaxIndex = 0
                    m = 0
                    Do j = 1, NumCompAtCurrDistance                                         !< loop over all comp. which correspond to current dist.
                        c = LocalDistanceOrderingArray(CurrDistID, j)
                        if (KindOfMolecule(c) <= 2) then                                    !< consider only components describing molecules or RRLs


                            !< store first component which describes a molecule / RRL
                            if (m == 0) then
                                m = j
                                if (SubBeamDescriptionFlag) then
                                    etaMaxIndex = j
                                    exit
                                endif
                            endif


                            !< find largest source size
                            if (CopymyXCLASSParameter(14, c) > etaMax) then
                                etaMax = CopymyXCLASSParameter(14, c)
                                etaMaxIndex = j
                            endif
                        endif
                    end Do                                                                  !< j: loop over all components at current distance

                    ! Debug:
                    ! print*," "
                    ! print*," "
                    ! print*,"CurrDistID = ", CurrDistID
                    ! print*,"etaMax = ", etaMax
                    ! print*,"etaMaxIndex = ", etaMaxIndex
                    ! print*,"m = ", m
                    ! print*,"c = ", c
                    ! print*,"LocalDistanceOrderingArray(CurrDistID, :) = ", LocalDistanceOrderingArray(CurrDistID, :)


                    !< move component describing a molecule or RRL with largest source size directly after ther continuum components
                    if (etaMaxIndex /= m) then
                        c = LocalDistanceOrderingArray(CurrDistID, m)
                        LocalDistanceOrderingArray(CurrDistID, m) = LocalDistanceOrderingArray(CurrDistID, etaMaxIndex)
                        LocalDistanceOrderingArray(CurrDistID, etaMaxIndex) = c
                    endif

                    ! Debug:
                    ! print*,">>LocalDistanceOrderingArray(CurrDistID, :) = ", LocalDistanceOrderingArray(CurrDistID, :)
                endif
                NumberMoleculeComponents = NumCompAtCurrDistance                            !< set number of comp. describing molecules


                !<------------------------------------------------------------------------------------------------------------------------------------
                !< add background from user-defined file
                if (CurrDistID == 1) then
                    ModelFuncList(FirstIndex:LastIndex) = BackgroundFromFile(FirstIndex:LastIndex)
                endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< create a sorted list of all Doppler-shifted transition frequencies and their central optical depths for current layer
                if (LocalOverlapFlag) then
                    call CalcLocalOverlapTransFreq(NumCompAtCurrDistance, NumComp, TotalNumMol, &
                                                   CurrDistID, l, CopyCompMoleculeIndex, &
                                                   CopyIsoRatioConversionTable, CopymyXCLASSParameter, Q, &
                                                   LocalDistanceOrderingArray)
                endif


                !< print what you do (reduced screen output)
                FastFlag = .true.
                if (printflag .and. LogFlag .and. FastFlag) then
                    ccc = (LastIndex - FirstIndex + 1)
                    if (SubBeamDescriptionFlag) then
                        write(numberstring3, '(I30)') CurrentConfigID
                        write(numberstring4, '(I30)') NumberConfig
                        print '(A, "Compute pixel spectrum for sub-beam description (", A, "/", A, ") .. ", $)', char(13), &
                                                              trim(adjustl(numberstring3)), trim(adjustl(numberstring4))
                    else
                        print '(A, "Compute intensity .. ", $)', char(13)
                    endif
                endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< define reversed frequency flag
                if (ObservationalDataList(FirstIndex, 1) <= ObservationalDataList(LastIndex, 1)) then
                    ReverseFreqFlag = .false.
                else
                    ReverseFreqFlag = .true.
                endif

                ! Debug:
                ! print*,"ReverseFreqFlag = ", ReverseFreqFlag


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< start loop over frequencies
                LastIntFreqPoint = 0.d0
                FirstDataFlag = .false.
                Do FreqIndex = FirstIndex, LastIndex                                        !< loop over all frequency points in the current freq. range
                    Obsfreq = ObservationalDataList(FreqIndex, 1)                           !< get frequency
                    ChannelIntensity = 0.d0                                                 !< reset channel intensity
                    TotalLocalIntensity = ModelFuncList(FreqIndex)                          !< reset final intensity for current frequency point

                    ! Debug:
                    ! print*," "
                    ! print*," "
                    ! print*,"l = ", l
                    ! print*,"FreqIndex = ", FreqIndex
                    ! print*,"Obsfreq = ", Obsfreq
                    ! print*,"TotalLocalIntensity = ", TotalLocalIntensity


                    !< print what you do (full screen output)
                    if (printflag .and. LogFlag .and. (.not. FastFlag)) then
                        ccc = (LastIndex - FirstIndex + 1)
                        write(numberstring1, '(I30)') (ccc * (CurrDistID - 1) + (FreqIndex - FirstIndex + 1))
                        write(numberstring2, '(I30)') (ccc * LocalNumberDistances)
                        if (SubBeamDescriptionFlag) then
                            write(numberstring3, '(I30)') CurrentConfigID
                            write(numberstring4, '(I30)') NumberConfig
                            print '(A, "Compute pixel spectrum for sub-beam description (", A, "/", A, ")", $)', char(13), &
                                                                  trim(adjustl(numberstring3)), trim(adjustl(numberstring4))
                            print '(" at frequency index (", A, "/", A, ")               ", $)', trim(adjustl(numberstring1)), &
                                                                                                 trim(adjustl(numberstring2))
                        else
                            print '(A, "Compute intensity at frequency index (", A, "/", A, ")                 ", $)', char(13), &
                                                                        trim(adjustl(numberstring1)), trim(adjustl(numberstring2))
                        endif
                    endif


                    !< define first freqeuncy flag
                    if (FreqIndex == FirstIndex) then
                        FirstFrequencyFlag = .true.
                    else
                        FirstFrequencyFlag = .false.
                    endif


                    !<************************************************************************************************************************************
                    !< used for integration over channel width: determine list of frequencies
                    ChannelWidth = 0.d0
                    if (IntegrationFlag) then                                               !< prepare integration over channel width


                        !< determine first and last frequency of current channel
                        FreqChannelFirst = 0.d0
                        FreqChannelLast = 0.d0
                        if (FirstFrequencyFlag) then
                            f0 = dmax1(1.d0, Obsfreq - dabs((ObservationalDataList((FreqIndex + 1), 1) - Obsfreq) / 2.d0))
                            f1 = Obsfreq + dabs((ObservationalDataList((FreqIndex + 1), 1) - Obsfreq) / 2.d0)
                        elseif (FreqIndex == LastIndex) then
                            f0 = dmax1(1.d0, Obsfreq - dabs((Obsfreq - ObservationalDataList((FreqIndex - 1), 1)) / 2.d0))
                            f1 = Obsfreq + dabs((Obsfreq - ObservationalDataList((FreqIndex - 1), 1)) / 2.d0)
                        else
                            f0 = dmax1(1.d0, Obsfreq - dabs((Obsfreq - ObservationalDataList((FreqIndex - 1), 1)) / 2.d0))
                            f1 = Obsfreq + dabs((ObservationalDataList((FreqIndex + 1), 1) - Obsfreq) / 2.d0)
                        endif
                        FreqChannelFirst = dmin1(f0, f1)
                        FreqChannelLast = dmax1(f0, f1)
                        ChannelWidth = (FreqChannelLast - FreqChannelFirst)                 !< determine channel width

                        ! Debug:
                        ! print*,"FreqChannelFirst = ", FreqChannelFirst
                        ! print*,"FreqChannelLast = ", FreqChannelLast
                        ! print*,'Obsfreq, ChannelWidth = ', Obsfreq, ChannelWidth
                    endif
                    !<************************************************************************************************************************************


                    !<====================================================================================================================================
                    !< determine beam size (eq. 3):
                    !<
                    !< for single dish data (InterFlag(l) = .false.):
                    !<
                    !<              telescope beam FWHM size is related to the diameter of the telescope by the diffraction limit:
                    !<
                    !<                  \theta_t = 1.22 \frac{c}{\nu D} * \zeta = 1.22 \frac{c}{\nu D} * (180.d0 / pi) * 3600.d0
                    !<
                    !<              where D describes the diameter of the telescope and c the speed of light.
                    !<
                    !<
                    !< for interferometric data (InterFlag(l) = .true.):
                    !<
                    !<              interferometric beam FWHM size is directly given by the user (and is constant over the whole frequency range!)
                    !<
                    !<                  \theta_t = D
                    !<
                    !<              where D describes the interferometric beam FWHM size given by the user.
                    !<
                    if (InterFlag(l)) then
                        beam_size = TelescopeSize(l)
                    else
                        beam_size = 1.22d-3 * ckms / (Obsfreq * TelescopeSize(l)) * (180.d0 / pi) * 3600.d0
                    endif

                    ! Debug:
                    ! print*,"beam_size = ", beam_size


                    !<====================================================================================================================================
                    !< determine max. beam filling factor and the corresponding component index and check for phen. cont. description
                    etaMax = 0.d0
                    etaMaxIndex = 0
                    j_back = 0.d0                                                           !< reset background temperature
                    kk = 0                                                                  !< set counter for molecule components
                    SameSizeFlag = .true.
                    val2 = 0.d0
                    Do ccc = 1, NumCompAtCurrDistance
                        c2 = LocalDistanceOrderingArray(CurrDistID, ccc)                    !< get component index


                        !< check, if component index is not zero
                        if (c2 == 0) then
                            print*," "
                            print*,"LocalOverlapFlag = ", LocalOverlapFlag
                            print*,"SubBeamDescriptionFlag = ", SubBeamDescriptionFlag
                            print*,"CurrDistID = ", CurrDistID
                            print*,"ccc = ", ccc
                            print*,"c2 = ", c2
                            print*,"LocalDistanceOrderingArray(CurrDistID, :) = ", LocalDistanceOrderingArray(CurrDistID, :)
                        endif


                        !< check, if all components at current distance have the same source size
                        if (dabs(CopymyXCLASSParameter(14, c2)) > tiny(1.d0) .and. SameSizeFlag) then
                            if (dabs(val2 - CopymyXCLASSParameter(14, c2)) < tiny(1.d0) .or. dabs(val2) < tiny(1.d0)) then
                                val2 = CopymyXCLASSParameter(14, c2)
                            else
                                SameSizeFlag = .false.
                            endif
                        endif


                        !< determine max. beam filling factor and the corresponding component index of all emission components
                        if (LocalOverlapFlag .or. SubBeamDescriptionFlag .or. dabs(CopymyXCLASSParameter(14, c2)) < tiny(1.d0)) then
                            val = 1.d0
                        else
                            val = CopymyXCLASSParameter(14, c2)**2 / (beam_size**2 + CopymyXCLASSParameter(14, c2)**2)
                        endif
                        ! if (val > etaMax) then
                        if (val > etaMax .and. KindOfMolecule(c2) <= 2) then
                            etaMax = val
                            etaMaxIndex = c2
                        endif


                        !< phenomenolical description of the continuum is described in molfit file by current component
                        if (KindOfMolecule(c2) == 6) then                                   !< current component defines phen. description of the cont.
                            LocalContPhen(1) = CopymyXCLASSParameter(1, c2)                 !< get function id parameter
                            LocalContPhen(2:6) = CopymyXCLASSParameter(2:6, c2)             !< get 1st - 5th continuum parameter
                            call PhenContDescription(j_back, Obsfreq, LocalContPhen)
                        endif


                        !< count number of components describing molecules / RRLs
                        if (KindOfMolecule(c2) <= 2) then
                            kk = kk + 1
                        endif
                    end Do
                    if (LocalOverlapFlag) then
                        NumberMoleculeComponents = NumCompAtCurrDistance                    !< set number of comp. describing molecules
                    else
                        NumberMoleculeComponents = kk
                    endif


                    !< phenomenolical description of the continuum is defined in obs. xml file
                    if (Phen_Flag_Range(l)) then                                            !< check, if phen. description was defined for current range
                        LocalContPhen(1) = ContPhen_Range(1, l)                             !< get function id parameter
                        LocalContPhen(2:6) = ContPhen_Range(2:6, l)                         !< get 1st - 5th continuum parameter
                        call PhenContDescription(j_back, Obsfreq, LocalContPhen)
                    endif

                    ! Debug:
                    ! print*,"etaMax = ", etaMax
                    ! print*,"etaMaxIndex = ", etaMaxIndex


                    !< initialize variables tau_l_em and tau_l_ab
                    if (UseEmAbsFuncFlag) then
                        tau_l_em = EmsAbsFunc(FreqIndex, CurrDistID, 1)
                        tau_l_ab = EmsAbsFunc(FreqIndex, CurrDistID, 2)
                    else
                        tau_l_em = 0.d0
                        tau_l_ab = 0.d0
                    endif

                    ! Debug:
                    ! print*,"tau_l_em = ", tau_l_em
                    ! print*,"tau_l_ab = ", tau_l_ab


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< loop over all components which belongs to the current distance
                    TauResetFlag = .false.                                                  !< initialize reset flag for tau_em and tau_abs
                    LayerIntensity = 0.d0                                                   !< reset local intensity
                    min_v_width = 1.d99                                                     !< reset minimal width
                    max_v_width = -1.d99                                                    !< reset max. width
                    min_v_off = 1.d99                                                       !< min. velocity offset
                    max_v_off = -1.d99                                                      !< max. velocity offset
                    Do ccc = 1, NumCompAtCurrDistance                                       !< loop over all components
                        c = LocalDistanceOrderingArray(CurrDistID, ccc)                     !< get component index
                        MoleculeIndex = CopyCompMoleculeIndex(c)                            !< get molecule index

                        ! Debug:
                        ! print*,"CurrDistID = ", CurrDistID
                        ! print*,"ccc = ", ccc
                        ! print*,"c = ", c
                        ! print*,"MoleculeIndex = ", MoleculeIndex
                        ! print*,"KindOfMolecule(c) = ", KindOfMolecule(c)
                        ! print*,"LocalDistanceOrderingArray(CurrDistID, :) = ", LocalDistanceOrderingArray(CurrDistID, :)


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< get model parameter for current component
                        if (KindOfMolecule(c) <= 2) then
                            Temp = CopymyXCLASSParameter(1, c)                              !< get kinetic temperature (molecule)


                            !< get velocity width (in km/s) and determine min. width (in km/s)
                            v_width_1 = CopymyXCLASSParameter(4, c)                         !< get velocity width (km/s) (Gauss)
                            v_width = v_width_1
                            if (v_width < min_v_width) then
                                min_v_width = v_width
                            endif
                            if (v_width > max_v_width) then
                                max_v_width = v_width
                            endif


                            !< get velocity offset (in km/s) and determine min. and max. velocity offset (in km/s)
                            v_off = CopymyXCLASSParameter(6, c)                             !< get velocity offset (km/s)
                            if (v_off < min_v_off) then
                                min_v_off = v_off
                            endif
                            if (v_off > max_v_off) then
                                max_v_off = v_off
                            endif

                            ! Debug:
                            ! print*," "
                            ! print*,"Temp = ", Temp
                            ! print*,"v_width = ", v_width
                            ! print*,"min_v_width = ", min_v_width
                            ! print*,"max_v_width = ", max_v_width
                            ! print*,"v_off = ", v_off
                            ! print*,"min_v_off = ", min_v_off
                            ! print*,"max_v_off = ", max_v_off


                            !< define reset flag for tau arrays
                            if (.not. LocalOverlapFlag) then                                !< reset only if local overlap is not taken into account
                                if (ccc == 1) then                                          !< always reset the arrays for the first component
                                    TauResetFlag = .true.
                                elseif (NumCompAtCurrDistance > 1 &
                                    .and. maxval(LocalDistanceOrderingArray(CurrDistID, 1:NumCompAtCurrDistance)) > 2) then
                                    c2 = LocalDistanceOrderingArray(CurrDistID, ccc - 1)    !< get previous component index
                                    if (KindOfMolecule(c2) <= 2) then                       !< if previous component was also a moleucle reset tau arrays
                                        TauResetFlag = .true.
                                    endif
                                endif
                            endif
                        endif


                        !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        !< call RADEX routines to perform Non-LTE calculation (for molecules only)
                        !< Temp = T_{\rm ex}^{m,c}   (kinetic temperature)
                        !< CopymyXCLASSParameter(2, c) = N_{\rm tot}^{m,c}  (total column density)
                        if (FirstFrequencyFlag .and. KindOfMolecule(c) == 0 .and. (.not. LTEFlag(c))) then
                                                                                            !< check, if current component is described in Non-LTE

                            !< get stop index for isotopologues
                            MoleculeIndex = CopyCompMoleculeIndex(c)                        !< get molecule index
                            if (IsoFlag) then                                               !< use isotopologues?
                                kk = TotalNumMol
                            else
                                kk = MoleculeIndex
                            endif


                            !< check, if isotopologues have to be taken into account
                            Do MoleculeIndexLoop = MoleculeIndex, kk                        !< loop over all molecule indices (including isotopologues!)
                                LocalIsoRatio = 1.d0
                                if (IsoFlag) then
                                    LocalIsoRatio = CopyIsoRatioConversionTable(MoleculeIndex, MoleculeIndexLoop)
                                endif
                                if (LocalIsoRatio /= 0.d0) then                             !< current iso ratio is unequal zero
                                    Ntot = LocalIsoRatio * CopymyXCLASSParameter(2, c)      !< scale column density (molecule) with current iso ratio

                                endif                                                       !< continue here, iso ratio is not defined, i.e. equal zero
                            end Do                                                          !< MoleculeIndexLoop: loop over all molecule indices
                        endif                                                               !< continue here for all data points
                        !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


                        !<********************************************************************************************************************************
                        !< integrate spectrum over channel width
                        if (IntegrationFlag .and. KindOfMolecule(c) <= 2) then


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< resample current channel if neccessary
                            call ResampleChannel(NumAdditionalFreq, AdditionalFreqList, ccc, NumCompAtCurrDistance, &
                                                 MoleculeIndex, FreqIndex, FreqChannelFirst, FreqChannelLast, &
                                                 ChannelWidth, ReverseFreqFlag, v_width, min_v_width, v_off, &
                                                 min_v_off, max_v_off, c, l)
                            if (NumAdditionalFreq > 0) then                                 !< make sure, that there is at least one additional frequency


                                !< check, if previous channel was calculated
                                AddFirst = 1
                                if (FirstFrequencyFlag) then
                                    FirstDataFlag(FreqIndex) = .true.
                                elseif (FirstDataFlag(FreqIndex - 1)) then
                                    AddFirst = 2
                                endif

                                ! Debug:
                                ! print*,"FreqIndex, AdditionalFreqList(1:NumAdditionalFreq) = ", FreqIndex, AdditionalFreqList(1:NumAdditionalFreq)


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< compute spectrum at new frequency points
                                !< (after the calculation of intensities at new additional frequency points is finished, tau_l_em is increased by
                                !< opacity at \nu(AddFreqIndex)
                                IntFreqList = 0.d0
                                tau_l_em_sum = 0.d0
                                tau_l_ab_sum = 0.d0
                                Do AddFreqIndex = AddFirst, NumAdditionalFreq               !< loop over additional frequency points
                                    AddFreq = AdditionalFreqList(AddFreqIndex)
                                    tau_l_em_local = tau_l_em                               !< restore tau_l_em variable
                                    tau_l_ab_local = tau_l_ab                               !< restore tau_l_ab variable


                                    !< compute beam size for single dish data at current added frequency
                                    if (InterFlag(l)) then
                                        Local_beam_size = TelescopeSize(l)
                                    else
                                        Local_beam_size = 1.22d-3 * ckms / (AddFreq * TelescopeSize(l)) * (180.d0 / pi) * 3600.d0
                                    endif


                                    !< compute background intensity at current added frequency, by interpolate intensity from higher distance
                                    !< to avoid a jump in intensity at channel edges
                                    !<
                                    !< parabolic interpolation: (taken from http://fourier.eng.hmc.edu/e176/lectures/NM/node25.html)
                                    !<
                                    !< q(x) = f(a) \frac{(x-b)(x-c)}{(a-b)(a-c)} + f(b) \frac{(x-c)(x-a)}{(b-c)(b-a)} + f(c) \frac{(x-a)(x-b)}{(c-a)(c-b)}
                                    !<
                                    !<   if (1 < FreqIndex .and. FreqIndex < LastIndex) then     !< use parabolic interpolation for non-edge channels
                                    !<       x0 = ObservationalDataList(FreqIndex - 1, 1)        !< (= a)
                                    !<       x1 = ObservationalDataList(FreqIndex, 1)            !< (= b)
                                    !<       x2 = ObservationalDataList(FreqIndex + 1, 1)        !< (= c)
                                    !<       f0 = ModelFuncList(FreqIndex - 1)                   !< (= f(a))
                                    !<       f1 = ModelFuncList(FreqIndex)                       !< (= f(b))
                                    !<       f2 = ModelFuncList(FreqIndex + 1)                   !< (= f(c))
                                    !<       BackgroundIntensity =  (f0 * (((AddFreq - x1) * (AddFreq - x2))/ ((x0 - x1) * (x0 - x2))) &
                                    !<                             + f1 * (((AddFreq - x2) * (AddFreq - x0))/ ((x1 - x2) * (x1 - x0))) &
                                    !<                             + f2 * (((AddFreq - x0) * (AddFreq - x1))/ ((x2 - x0) * (x2 - x1))))
                                    !<   elseif (FirstFrequencyFlag) then                        !< use linear interpolation for first freq. channel
                                    !<
                                    !< use picewise linear interpolation
                                    !<
                                    !    if (FreqIndex < LastIndex) then
                                    !        !
                                    !        !    9--8--7--6--5--4--3--2--1
                                    !        !                |
                                    !        !            -1  | +1
                                    !        !            FreqIndex
                                    !        !
                                    !        if (ReverseFreqFlag) then
                                    !            x0 = ObservationalDataList(FreqIndex + 1, 1)
                                    !            x1 = ObservationalDataList(FreqIndex, 1)
                                    !            f0 = ModelFuncList(FreqIndex + 1)
                                    !            f1 = ModelFuncList(FreqIndex)
                                    !        !
                                    !        !    1--2--3--4--5--6--7--8--9
                                    !        !                |
                                    !        !            -1  | +1
                                    !        !            FreqIndex
                                    !        !
                                    !        else
                                    !            if (AddFreq >= Obsfreq .or. FirstFrequencyFlag) then
                                    !                x0 = ObservationalDataList(FreqIndex, 1)
                                    !                x1 = ObservationalDataList(FreqIndex + 1, 1)
                                    !                f0 = ModelFuncList(FreqIndex)
                                    !                f1 = ModelFuncList(FreqIndex + 1)
                                    !            else
                                    !                x0 = ObservationalDataList(FreqIndex - 1, 1)
                                    !                x1 = ObservationalDataList(FreqIndex, 1)
                                    !                f0 = ModelFuncList(FreqIndex - 1)
                                    !                f1 = ModelFuncList(FreqIndex)
                                    !            endif
                                    !        endif
                                    !        BackgroundIntensity = f0 + (AddFreq - x0) * ((f1 - f0) / (x1 - x0))
                                    !    else
                                    !        BackgroundIntensity = TotalLocalIntensity
                                    !    endif
                                    BackgroundIntensity = TotalLocalIntensity


                                    !< solve detection equation at new additional frequency points
                                    call DetectionEquation(LocalIntensity, tau_l_em_local, tau_l_ab_local, j_back, Td, NumComp, &
                                                           AddFreq, FreqIndex, l, CurrDistID, MoleculeIndex, &
                                                           CopymyXCLASSParameter, CopyIsoRatioConversionTable, Q, beam_size, c, &
                                                           ccc, BackgroundIntensity, TotalNumMol, etaMaxIndex, &
                                                           NumCompAtCurrDistance, LocalMaxNumTrans, &
                                                           FirstIndex, max_v_width, TauResetFlag, &
                                                           NumberMoleculeComponents, SameSizeFlag)
                                    IntFreqList(AddFreqIndex) = LocalIntensity


                                    !< compute sum of taus (and source function) to determine average opacities (background temperture)
                                    tau_l_em_sum = tau_l_em_sum + (tau_l_em_local - tau_l_em)
                                    tau_l_ab_sum = tau_l_ab_sum + (tau_l_ab_local - tau_l_ab)
                                end Do                                                      !< AddFreqIndex: loop over additional frequency points


                                !< add calculated intensity of last channel to frequency list
                                FirstDataFlag(FreqIndex) = .true.
                                if (AddFirst == 2) then
                                    if (CurrDistID == 1) then
                                        IntFreqList(1) = LastIntFreqPoint(c)
                                    else
                                        IntFreqList(1) = IntFreqList(2)                     !< bug-fix: avoid shift of modeled spectrum by adding f-comp.
                                    endif
                                endif
                                LastIntFreqPoint(c) = IntFreqList(NumAdditionalFreq)


                                !< compute taus, background temperature, and source function for next component
                                tau_l_em = tau_l_em + (tau_l_em_sum / (NumAdditionalFreq - AddFirst + 1))
                                tau_l_ab = tau_l_ab + (tau_l_ab_sum / (NumAdditionalFreq - AddFirst + 1))


                                !<########################################################################################################################
                                !< Only used for storing opacities and intensities
                                !<########################################################################################################################
                                !< store optical depth from last freq. point NumAdditionalFreq and dust temperature
                                if (AllOutputFilesFlag) then
                                    if (LocalOverlapFlag .and. ccc == NumCompAtCurrDistance) then
                                        TauPerCompArray(FreqIndex, c, :) = tau_l_ab
                                    else
                                        TauPerCompArray(FreqIndex, c, :) = TauHelperArrayABS(:)
                                    endif
                                    TdPerCompArray(FreqIndex, c) = Td
                                endif
                                !<########################################################################################################################
                                !<########################################################################################################################


                                !<########################################################################################################################
                                !< Only used by myXCLASS function!
                                !<########################################################################################################################
                                !< store optical depth from last freq. point NumAdditionalFreq
                                if (EmAbsFlag .and. ccc == NumCompAtCurrDistance) then
                                    TauEmAbsArray(FreqIndex, CurrDistID, 1) = TauEmAbsArray(FreqIndex, CurrDistID, 1) + tau_l_em
                                    TauEmAbsArray(FreqIndex, CurrDistID, 2) = TauEmAbsArray(FreqIndex, CurrDistID, 2) + tau_l_ab
                                endif
                                !<########################################################################################################################


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< integrate spectrum using the trapezoidal rule
                                ChannelIntensity = 0.d0
                                Do fi = 1, (NumAdditionalFreq - 1)
                                    x0 = AdditionalFreqList(fi)
                                    x1 = AdditionalFreqList(fi + 1)
                                    f0 = IntFreqList(fi)
                                    f1 = IntFreqList(fi + 1)
                                    val = dabs(x1 - x0) * (f0 + f1) / 2.d0
                                    ChannelIntensity = ChannelIntensity + val
                                end Do
                                ChannelIntensity = (1.d0 / ChannelWidth) * ChannelIntensity

                                ! Debug:
                                !    write(12342,*) FirstFrequencyFlag, c, FreqChannelFirst, IntFreqList(1), FreqChannelLast, IntFreqList(2), &
                                !                   ChannelIntensity, TotalLocalIntensity
                                !    if (FreqIndex == 1323) then
                                !        print*,">>>>CurrDistID, ccc = ", CurrDistID, ccc
                                !        print*,">>>>NumAdditionalFreq = ", NumAdditionalFreq
                                !        print*,">>>>AdditionalFreqList(1:NumAdditionalFreq) = ", AdditionalFreqList(1:NumAdditionalFreq)
                                !        print*,">>>>IntFreqList(1:NumAdditionalFreq) = ", IntFreqList(1:NumAdditionalFreq)
                                !        print*,"--->ObservationalDataList(FreqIndex, 1:2) = ", ObservationalDataList(FreqIndex, 1:2)
                                !        print*,"===>ChannelWidth = ", ChannelWidth
                                !        print*,"--->ChannelIntensity = ", ChannelIntensity
                                !    endif
                            endif


                        !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        !< solve detection equation without integrating over channel width
                        else
                            call DetectionEquation(LocalIntensity, tau_l_em, tau_l_ab, j_back, Td, NumComp, Obsfreq, &
                                                   FreqIndex, l, CurrDistID, MoleculeIndex, CopymyXCLASSParameter, &
                                                   CopyIsoRatioConversionTable, &
                                                   Q, beam_size, c, ccc, TotalLocalIntensity, TotalNumMol, etaMaxIndex, &
                                                   NumCompAtCurrDistance, &
                                                   LocalMaxNumTrans,  FirstIndex, max_v_width, TauResetFlag,&
                                                   NumberMoleculeComponents, SameSizeFlag)
                            ChannelIntensity = LocalIntensity


                            !< store emission and absorption functions
                            if (EmAbsFlag .and. ccc == NumCompAtCurrDistance) then
                                TauEmAbsArray(FreqIndex, CurrDistID, 1) = TauEmAbsArray(FreqIndex, CurrDistID, 1) + tau_l_em
                                TauEmAbsArray(FreqIndex, CurrDistID, 2) = TauEmAbsArray(FreqIndex, CurrDistID, 2) + tau_l_ab
                            endif


                            !<############################################################################################################################
                            !< Only used for storing opacities and intensities
                            !<############################################################################################################################
                            !< store opacities and dust temperature
                            if (AllOutputFilesFlag) then
                                if (LocalOverlapFlag .and. ccc == NumCompAtCurrDistance) then
                                    TauPerCompArray(FreqIndex, c, :) = tau_l_ab
                                else
                                    TauPerCompArray(FreqIndex, c, :) = TauHelperArrayABS(:)
                                endif
                                TdPerCompArray(FreqIndex, c) = Td
                            endif
                            !<############################################################################################################################
                            !<############################################################################################################################
                        endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine total intensity of all components
                        LayerIntensity = LayerIntensity + ChannelIntensity

                        ! Debug:
                        ! if (FreqIndex == 1323) then
                        !     print*,"**>>>>>>>>>LayerIntensity, ChannelIntensity = ", LayerIntensity, ChannelIntensity
                        ! endif


                        !<################################################################################################################################
                        !< Only used for storing opacities and intensities
                        !<################################################################################################################################
                        !< store intensities
                        if (AllOutputFilesFlag .and. (.not. LocalOverlapFlag &
                            .or. (LocalOverlapFlag .and. ccc == NumCompAtCurrDistance))) then
                            IntPerCompArray(FreqIndex, c, :) = LocalIntArray(:)
                        endif
                        !<################################################################################################################################
                        !<################################################################################################################################
                    end Do                                                                  !< ccc: loop over all components of current layer

                    ! Debug:
                    ! print*,"Obsfreq, LayerIntensity = ", Obsfreq, LayerIntensity


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine chi^2 value using local calculated intensity
                    TotalLocalIntensity = dmin1(LayerIntensity, 1.d20)                      !< avoid to large values for intensity
                    ModelFuncList(FreqIndex) = TotalLocalIntensity
                    if (CurrDistID == LocalNumberDistances) then


                        !< determine sigma (sig2i) factor
                        sig2i = 1.d0
                        if (ObservationalDataList(FreqIndex, 3) /= 0.d0) then               !< do the experimental datas include errors
                            sig2i = 1.d0 / (ObservationalDataList(FreqIndex, 3)**2)
                        endif


                        !< add differnce to chi2 value
                        chi2Value = chi2Value + ((TotalLocalIntensity - ObservationalDataList(FreqIndex, 2))**2 * sig2i)
                    endif
                end Do                                                                      !< FreqIndex: loop over all components
            end Do                                                                          !< CurrDistID: loop over all pure distances

            ! Debug:
            ! print*,"chi2Value = ", chi2Value, minval(ModelFuncList(:)), maxval(ModelFuncList(:))


            !< we're done
            return
        end subroutine CalcModelPixelSpectrum


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcSpectrum
        !>
        !> calculates the myXCLASS spectrum for a given parameter vector
        !>
        !>
        !> input variables:     NumberFreeParameterCopy:    number of free parameters
        !>                      ParameterVector:        parameter vector
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      NumComp:                total number of components
        !>                      CompMoleculeIndex:      molecule index for each component
        !>                      myXCLASSParameterOrig:  array containing all molfit parameters for each component
        !>                      TotalNumMol:            total number of molecules
        !>                      IsoRatioConversionTable: table with iso ratios between iso master and molecule
        !>                      NumDataPoints:          total number of data points
        !>                      ThreadNumber:           current thread number
        !>
        !> output variables:    chi2Value:              chi2 value for parameter vector
        !>                      ModelFuncList:          model function values
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2014-07-31
        !>
        subroutine ModelCalcSpectrum(NumberFreeParameterCopy, ParameterVector, ModelFunctionFlag, NumComp, chi2Value, &
                                     CompMoleculeIndexOrig, myXCLASSParameterOrig, TotalNumMol, IsoRatioConversionTableOrig, &
                                     NumDataPoints, ModelFuncList, ThreadNumber)

            implicit none

            !< input parameters
            integer :: NumberFreeParameterCopy                                              !< number of free parameters
            integer :: NumComp                                                              !< total number of components
            integer :: TotalNumMol                                                          !< total number of molecules
            integer :: NumDataPoints                                                        !< total number of data points
            integer :: ThreadNumber                                                         !< current thread number
            integer, dimension(NumComp) :: CompMoleculeIndexOrig                            !< molecule index for each component
            real*8, dimension(NumberFreeParameterCopy) :: ParameterVector                   !< parameter vector
            real*8, dimension(16, NumComp) :: myXCLASSParameterOrig                         !< array containing all molfit parameters for each component
            real*8, dimension(TotalNumMol, TotalNumMol) :: IsoRatioConversionTableOrig      !< table with iso ratios between iso master and molecule
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model func. values are stored or not

            !< output parameters
            real*8 :: chi2Value                                                             !< chi2 value for parameter vector
            real*8, dimension(NumDataPoints) :: ModelFuncList                               !< output variable: model function values

            !< working variables
            integer, parameter :: MaxFreqResampling = 100                                   !< max. number of frequency points for resampling
            integer :: fitnum, i, j, k, l, m, c, t, fi, ll, kk, MoleculeIndexLoop           !< loop variables
            integer :: ModeID                                                               !< convolution mode
            integer :: IndexComp                                                            !< overall component index
            integer :: varIndex                                                             !< myXCLASS variable index
            integer :: ErrChannelIndexLocal                                                 !< used for error handling
            integer :: ErrChannelLocal                                                      !< used for error handling
            integer :: MoleculeIndex                                                        !< index for molecule
            integer :: FirstMolecularDataIndex                                              !< first index for molecular data table
            integer :: LastMolecularDataIndex                                               !< last index for molecular data table
            integer :: FirstIndex                                                           !< index for first freq. data point
            integer :: LastIndex                                                            !< index for last freq. data point
            integer :: FreqIndex                                                            !< index for frequency point
            integer :: IsoCounter, fff                                                      !< working variable: counter for isotopologue
            integer :: n                                                                    !< current comp. index
            integer :: ncomp                                                                !< total number of comp. of curr. mol.
            integer :: NumberConfig                                                         !< total number of unique configurations
            integer :: LocalMaxNumTrans                                                     !< max. number of transitions (used for Non-LTE description)
            integer :: MaxNumIso                                                            !< working variable max. number of isotopologues per component
            integer :: LocalNumFreqPoints                                                   !< local number of freq. points
            integer :: LocalNumberDistances                                                 !< copy of NumberDistances
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer, dimension(NumComp) :: CopyCompMoleculeIndex                            !< local copy of CompMoleculeIndexOrig
            integer, dimension(NumComp, NumComp) :: LocalDistanceOrderingArray              !< local copy of DistanceOrderingArray
            real*8 :: x0, f0, f1                                                            !< working variables for integration
            real*8 :: Obsfreq                                                               !< observed frequency
            real*8 :: val                                                                   !< value for integration
            real*8 :: j_back                                                                !< brightness for background temperatures
            real*8 :: j_cb                                                                  !< brightness for cosmic microwave background temperature
            real*8 :: Temperature                                                           !< temperature
            real*8 :: beam_size                                                             !< beam size (eqn. 2 - 3)
            real*8, dimension(NumComp) :: LocalPureDistances                                !< local copy of PureDistances
            real*8, dimension(NumComp, TotalNumMol) :: Q                                    !< partition function at the rotation temperatures
            real*8, dimension(16, NumComp) :: CopymyXCLASSParameter                         !< local copy of myXCLASSParameterOrig
            real*8, dimension(TotalNumMol, TotalNumMol) :: CopyIsoRatioConversionTable      !< local copy of IsoRatioConversionTableOrig
            real*8, allocatable, dimension(:, :) :: ModelPixelSpectra                       !< array for model pixel spectra for each unique comp. config.
            character(len = 30) :: numberstring1, numberstring2                             !< used for integer number to string conversion
            character(len = 40) :: CurrMoleculeName                                         !< name of molecule
            character(len = 40) :: ModifiedMoleculeName                                     !< name of optical depth file
            character(len = 4096) :: LocalFileName                                          !< working variable: local file name
            logical :: CalcFalg                                                             !< used for calculation of isotopoluges
            logical :: LocalModelFunctionFlag                                               !< flag for indicating if model func. values are stored or not

            ! Debug:
            ! print*," "
            ! print*,"NumberFreeParameterCopy = ", NumberFreeParameterCopy
            ! print*,"--->ParameterVector = ", ParameterVector
            ! print*,"ModelFunctionFlag = ", ModelFunctionFlag
            ! print*,"NumComp = ", NumComp
            ! print*,"chi2Value = ", chi2Value
            ! print*,"CompMoleculeIndexOrig = ", CompMoleculeIndexOrig
            ! print*,"myXCLASSParameterOrig = ", myXCLASSParameterOrig
            ! print*,"TotalNumMol = ", TotalNumMol
            ! print*,"IsoRatioConversionTableOrig = ", IsoRatioConversionTableOrig
            ! print*,"NumDataPoints = ", NumDataPoints
            ! print*,"ModelFuncList = ", ModelFuncList
            ! print*,"ThreadNumber = ", ThreadNumber


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize some variables
            chi2Value = 0.d0
            ModelFuncList = 0.d0
            MaxNumIso = 1
            LocalModelFunctionFlag = ModelFunctionFlag
            j_back = 0.d0                                                                   !< initialize j_back variable
            j_cb = 0.d0                                                                     !< initialize j_cb variable
            if (ThreadNumber < 0) then
                print '(2x, "Error in subroutine ModelCalcSpectrum:")'
                print '(4x, "Thread number negativ!!")'
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< make private copies of variables CompMoleculeIndexOrig, myXCLASSParameterOrig, and IsoRatioConversionTableOrig
            CopyCompMoleculeIndex = CompMoleculeIndexOrig
            CopymyXCLASSParameter = myXCLASSParameterOrig
            if (IsoFlag) then
                CopyIsoRatioConversionTable = IsoRatioConversionTableOrig
            endif


            !<============================================================================================================================================
            !< update myXCLASS parameter vector with new parameter values
            !< CopymyXCLASSParameter(1, c)  T_rot (=0), T_e (=1),    T_e (=2),      T_cont_dust (=3),        - (=4),     - (=5)
            !< CopymyXCLASSParameter(2, c)  N_tot (=0), EM_RRL (=1), EM_RRL (=2),   nHcolumn_cont_dust (=3), - (=4),     - (=5)
            !< CopymyXCLASSParameter(3, c)  - (=0),     - (=1),      N_e (=2),      kappa_cont_dust (=3),    - (=4),     - (=5)
            !< CopymyXCLASSParameter(4, c)  V_width_Gauss,                          beta_cont_dust (=3),     - (=4),     - (=5)
            !< CopymyXCLASSParameter(5, c)  -                                       - (=3),                  - (=4),     - (=5)
            !< CopymyXCLASSParameter(6, c)  V_Off (=0,1,2)                          - (=3),                  - (=4),     - (=5)
            !< CopymyXCLASSParameter(7, c)  T_dOff
            !< CopymyXCLASSParameter(8, c)  T_dSlope
            !< CopymyXCLASSParameter(9, c)  nHcolumn
            !< CopymyXCLASSParameter(10, c) kappa
            !< CopymyXCLASSParameter(11, c) beta
            !< CopymyXCLASSParameter(12, c) LayerDistance (=0,1,2,3,4,5,6)
            !< CopymyXCLASSParameter(13, c) CFFlag (=0,1,2,3,4,5,6)
            !< CopymyXCLASSParameter(14, c) source size, source radius
            !< CopymyXCLASSParameter(15, c) source_center_x
            !< CopymyXCLASSParameter(16, c) source_center_y
            Do fitnum = 1, NumberFreeParameterCopy                                          !< loop over all free parameters


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< update molfit file parameters
                IndexComp = ConversionTableMAGIXmyXCLASSParam(fitnum, 1)                    !< get current overall component index
                varIndex = ConversionTableMAGIXmyXCLASSParam(fitnum, 2)                     !< get current myXCLASS variable index
                if (IndexComp > 0 .and. varIndex > 0 .and. varIndex <= 16) then             !< check, if current free param. corresponds to molfit param.

                    ! Debug:
                    ! print*,'varIndex, IndexComp, fitnum = ', varIndex, IndexComp, fitnum
                    ! print*,'CopymyXCLASSParameter(:, IndexComp) = ', CopymyXCLASSParameter(:, IndexComp)
                    ! print*,'ParameterVector(fitnum) = ', ParameterVector(fitnum)


                    CopymyXCLASSParameter(varIndex, IndexComp) = ParameterVector(fitnum)


                    !< apply some corrections for some molfit parameters
                    if (KindOfMolecule(IndexComp) <= 4) then


                        !< check N_tot, EM_RRL, nHcolumn_cont_dust, EM_ff, Ne, and nHcolumn for overflow and convert log10 value back to linear scale
                        if (varIndex == 2 .or. (varIndex == 3 .and. KindOfMolecule(IndexComp) == 2) &
                            .or. (varIndex == 9 .and. nHFlag(IndexComp))) then
                            CopymyXCLASSParameter(varIndex, IndexComp) &
                                                = 10.d0**dmin1(300.d0, CopymyXCLASSParameter(varIndex, IndexComp))


                        !< check for V_width_1 < 1.d-5
                        elseif (varIndex == 4 .and. KindOfMolecule(IndexComp) <= 2) then
                            CopymyXCLASSParameter(varIndex, IndexComp) = dmax1(CopymyXCLASSParameter(varIndex, IndexComp), 1.d-5)


                        !< check for V_width_2 < 1.d-5
                        elseif (varIndex == 5 .and. KindOfMolecule(IndexComp) <= 2) then
                            if (CopymyXCLASSParameter(varIndex, IndexComp) >= 0.d0) then
                                CopymyXCLASSParameter(varIndex, IndexComp) &
                                                = dmax1(CopymyXCLASSParameter(varIndex, IndexComp), 1.d-5)
                            endif
                        endif

                    endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< update iso ratio table with new iso ratios
                elseif (IsoFlag) then
                    Do m = 1, TotalNumberOfMolecules
                        Do k = (m + 1), TotalNumberOfMolecules
                            if (IsoNfitConversionTable(m, k, 1) == fitnum) then


                                !< continue here, if iso master and isotopologues are exchanged, i.e. ratio = 1.0 / ratio
                                if (IsoNfitConversionTable(m, k, 2) < 0) then
                                    CopyIsoRatioConversionTable(m, k) = 1.d0 / ParameterVector(fitnum)
                                else
                                    CopyIsoRatioConversionTable(m, k) = ParameterVector(fitnum)
                                endif
                                if (dabs(CopyIsoRatioConversionTable(m, k)) < 1.e-30) then
                                    CopyIsoRatioConversionTable(k, m) = 1.0
                                else
                                    CopyIsoRatioConversionTable(k, m) = 1.d0 / CopyIsoRatioConversionTable(m, k)
                                endif

                                ! Debug:
                                ! print*,' '
                                ! print*,' '
                                ! print*, m, adjustl(MoleculeNames(m))
                                ! print*, k, adjustl(MoleculeNames(k))
                                ! print*, CopyIsoRatioConversionTable(k, m)
                                ! print*, CopyIsoRatioConversionTable(m, k)
                            endif
                        end Do                                                              !< loop over k
                    end Do                                                                  !< loop over m
                endif
            end Do                                                                          !< fitnum: loop over all free parameters


            !<============================================================================================================================================
            !< check, if some iso ratios were globally defined and update corresponding iso ratios
            if (IsoFlag .and. NumberOfGlobalISORatios > 0) then
                Do m = 1, TotalNumberOfMolecules
                    Do k = (m + 1), TotalNumberOfMolecules
                        j = abs(IsoNfitConversionTable(m, k, 2))                            !< get j index for iso ratio

                        ! Debug:
                        ! print*,'m, k, j = ', m, k, j

                        if (j > 0) then


                            !< determine if globally defined iso ratios are involved in the current iso ratio
                            if (GlobalIsoRatioParameter(j, 1, 1) > 0) then
                                val = 1.d0
                                !val = CopyIsoRatioConversionTable(m, k)
                                Do fi = 1, NumberOfGlobalISORatios
                                    l = GlobalIsoRatioParameter(j, fi, 1)                   !< get j index for globally defined iso ratio
                                    if (l == 0) then
                                        exit
                                    endif
                                    t = GlobalIsoRatioParameter(j, fi, 2)                   !< get number of replications
                                    fitnum = int(IsoRatio(l, 2))                            !< get index for fitted variable

                                    ! Debug:
                                    ! print*," "
                                    ! print*," "
                                    ! print*,"fi, l, t, fitnum = ", fi, l, t, fitnum
                                    ! print*,"IsoMolecule(l) = ", trim(adjustl(IsoMolecule(l)))
                                    ! print*,"IsoMasters(l)  = ", trim(adjustl(IsoMasters(l)))
                                    ! print*,"IsoRatio(l, 1) = ", IsoRatio(l, 1)

                                    if (fitnum == 0) then                                   !< is globally defined ratio fitted
                                        val = val * (IsoRatio(l, 1)**float(t))              !< calculated new ratio with non-fitted iso ratio
                                    else
                                        val = val * (ParameterVector(fitnum)**float(t))     !< calculated new ratio with fitted iso ratio

                                        ! Debug:
                                        ! print*,"m, k, fitnum, ParameterVector(fitnum) = ", m, k, fitnum, ParameterVector(fitnum)
                                    endif
                                end Do


                                !< is current iso ratio inverted
                                if (IsoNfitConversionTable(m, k, 2) < 0) then
                                    if (dabs(val) < 1.e-30) then
                                        val = 1.d0
                                    else
                                        val = 1.d0 / val
                                    endif
                                endif
                                val = val * CopyIsoRatioConversionTable(m, k)


                                !< update entry in current working copy of IsoRatioConversionTable
                                CopyIsoRatioConversionTable(m, k) = val
                                if (dabs(val) < 1.e-30) then
                                    CopyIsoRatioConversionTable(k, m) = 1.0
                                else
                                    CopyIsoRatioConversionTable(k, m) = 1.d0 / val
                                endif

                                ! Debug:
                                ! print*,"IsoRatioConversionTableOrig(m, k) = ", IsoRatioConversionTableOrig(m, k)
                                ! print*,"m,k,val = ", val
                            endif
                        endif

                        ! Debug:
                        ! if (CopyIsoRatioConversionTable(m, k) == 0.d0) then
                        !     write(23424, '("     .     ", $)')
                        ! else
                        !     write(23424, '(1x, ES10.2, $)') CopyIsoRatioConversionTable(m, k)
                        ! endif
                        ! if (IsoNfitConversionTable(m, k, 2) == 0.d0) then
                        !     write(23425, '("     .     ", $)')
                        ! else
                        !     write(23425, '(1x, I10, $)') IsoNfitConversionTable(m, k, 2)
                        ! endif
                    end Do                                                                  !< loop over k

                    ! Debug:
                    ! write(23424, '(" ")')
                    ! write(23425, '(" ")')
                end Do                                                                      !< loop over m

                ! Debug:
                ! write(23424, '(" ")')
                ! write(23424, '(" ")')
                ! write(23425, '(" ")')
                ! write(23425, '(" ")')
            endif

            ! Debug:
            ! print*," "
            ! print*," "
            ! Do i = 1, TotalNumberComponents
            !     j = CopyCompMoleculeIndex(i)
            !     print*,"MoleculeNames = ", i, trim(adjustl(MoleculeNames(j)))
            !     print*,"CopymyXCLASSParameter(:, i) = ", CopymyXCLASSParameter(:, i)
            ! end Do
            ! Do j = 1, TotalNumberOfMolecules
            !     print*,j
            !     print*,">>", trim(adjustl(MoleculeNames(j))), "<<"
            ! end Do
            ! Do c = 1, TotalNumberComponents
            !     print*,"c, CopymyXCLASSParameter(:, c) = ", c, CopymyXCLASSParameter(:, c)
            ! end Do


            !<============================================================================================================================================
            !< determine partiton functions for all rotation temperatures and all Doppler-shifted transition frequencies
            Q = 0.d0                                                                        !< reset array for partition function values
            i = 0
            Do c = 1, TotalNumberComponents                                                 !< loop over all components
                if (KindOfMolecule(c) <= 2) then
                    Temperature = CopymyXCLASSParameter(1, c)                               !< get kinetic temperatures

                    ! Debug:
                    ! print*," "
                    ! print*,"c = ", c
                    ! print*,"TotalNumberComponents = ", TotalNumberComponents
                    ! print*,"KindOfMolecule(c) = ", KindOfMolecule(c)
                    ! print*,"Temperature = ", Temperature


                    !< save modeled spectrum for Non-LTE foreground components
                    if (.not. LTEFlag(c) .and. int(CopymyXCLASSParameter(13, c)) == 2) then
                        LocalModelFunctionFlag = .true.
                    endif

                    ! Debug:
                    ! print*,"LTEFlag(c) = ", LTEFlag(c)
                    ! print*,"CopymyXCLASSParameter(13, c) = ", CopymyXCLASSParameter(13, c)
                    ! print*,"LocalModelFunctionFlag = ", LocalModelFunctionFlag


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< check kinetic temperature
                    if (Temperature <= 0.d0) then
                        Temperature = 10.d0**TempLow
                        j = CopyCompMoleculeIndex(c)
                        k = c - sum(NumberComponentsPerMolecule(:(j - 1)))
                        Do ErrChannelIndexLocal = 1, 1
                            ErrChannelLocal = AllErrChannels(ErrChannelIndexLocal)
                            write(ErrChannelLocal, '(" ")')
                            write(ErrChannelLocal, '(" ")')
                            write(ErrChannelLocal, '(2x, "Error in subroutine ModelCalcSpectrum!")')
                            if (j /= 0) then
                                write(ErrChannelLocal, '(4x, "The kinetic temperature for component ", &
                                                &I4, " of molecule ", A, " ")') k, trim(adjustl(MoleculeNames(j)))
                            else
                                write(ErrChannelLocal, '(4x, "The kinetic temperature for component ", I4, " ")') k
                            endif
                            write(ErrChannelLocal, '(4x, "is less than or equal to zero!")')
                            write(ErrChannelLocal, '(" ")')
                            write(ErrChannelLocal, '(" ")')
                            write(ErrChannelLocal, '(4x, "Set temperature to ", F25.15)') Temperature
                            write(ErrChannelLocal, '(" ")')
                        end Do
                    endif

                    ! Debug:
                    ! print*,"CopyCompMoleculeIndex(c) = ", CopyCompMoleculeIndex(c)
                    ! print*,"IsoFlag = ", IsoFlag
                    ! print*,"CopyIsoRatioConversionTable(CopyCompMoleculeIndex(c), :) = ", CopyIsoRatioConversionTable(CopyCompMoleculeIndex(c), :)


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< get partition function at temperature, for molecule and if used for isotopologues as well
                    MoleculeIndex = CopyCompMoleculeIndex(c)                                !< get molecule index
                    l = MoleculeIndex
                    if (IsoFlag) then
                        Do m = l, TotalNumberOfMolecules
                            if (dabs(CopyIsoRatioConversionTable(l, m)) > epsilon(0.d0)) then             !< is a iso ratio defined for mth molecule?
                                call InterpolateQ(Q(c, m), Temperature, m)                  !< get partition function at temperature
                            endif
                        end Do
                        kk = TotalNumMol
                    else
                        kk = MoleculeIndex
                        call InterpolateQ(Q(c, l), Temperature, MoleculeIndex)              !< get partition function at temperature
                    endif

                    ! Debug:
                    ! print*,"l = ", l
                    ! print*,"Q(c, l) = ", Q(c, l)
                endif
            end Do                                                                          !< c: loop over all components


            !<============================================================================================================================================
            !< allocate variables for output variables
            if (EmAbsFlag) then
                i = max0(1, NumDataPoints)
                j = max0(1, TotalNumberComponents)
                if (allocated(TauEmAbsArray)) then
                    deallocate(TauEmAbsArray, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x, "Error in subroutine ModelCalcSpectrum!")')
                            write(ErrChannel, '(3x, "Cannot deallocate variable TauEmAbsArray!")')
                            write(ErrChannel, '(" ")')
                        end Do
                        stop 'Program aborted! Please restart the program!'
                    endif
                endif
                allocate(TauEmAbsArray(i, j, 2), stat = allocstatus)
                if (allocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine ModelCalcSpectrum!")')
                        write(ErrChannel, '(3x, "Can not allocate the following variable:")')
                        write(ErrChannel, '(5x, "- TauEmAbsArray(i, j, 2)")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Used parameters:")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(5x, "- i = ", I8)') i
                        write(ErrChannel, '(5x, "- j = ", I8)') j
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(" ")')
                    end Do
                    stop 'Program aborted! Please restart the program!'
                endif
                TauEmAbsArray = 0.d0
            endif


            !<############################################################################################################################################
            !< Only used for storing opacities and intensities
            !<############################################################################################################################################
            !< allocate variables for output variables
            if (AllOutputFilesFlag) then


                !< determine max. number of isotopologues per component
                MaxNumIso = 0
                if (IsoFlag) then
                    Do l = 1, TotalNumberOfMolecules
                        k = 0
                        Do m = 1, TotalNumberOfMolecules
                            if (CopyIsoRatioConversionTable(l, m) /= 0.d0) then             !< is a iso ratio defined for mth molecule?
                                k = k + 1
                            endif
                        end Do
                        if (MaxNumIso < k) then
                            MaxNumIso = k
                        endif
                    end Do
                endif
                MaxNumIso = max0(1, MaxNumIso)

                ! Debug:
                ! print*,"MaxNumIso = ", MaxNumIso


                !< allocate variables for output variables
                i = max0(1, NumDataPoints)
                j = max0(1, TotalNumberComponents)
                if (allocated(IntPerCompArray)) then
                    deallocate(IntPerCompArray, TauPerCompArray, TdPerCompArray, TauHelperArrayEM, &
                               TauHelperArrayABS, TauHelperArray2, LocalIntArray, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x, "Error in subroutine ModelCalcSpectrum!")')
                            write(ErrChannel, '(3x, "Cannot deallocate variables IntPerCompArray, &
                                                    &TempPartFunc, TdPerCompArray, and TauHelperArray!")')
                            write(ErrChannel, '(" ")')
                        end Do
                        stop 'Program aborted! Please restart the program!'
                    endif
                endif
                allocate(IntPerCompArray(i, j, MaxNumIso), TauPerCompArray(i, j, MaxNumIso), TdPerCompArray(i, j), &
                         TauHelperArrayEM(MaxNumIso), TauHelperArrayABS(MaxNumIso), TauHelperArray2(MaxNumIso), &
                         LocalIntArray(MaxNumIso), stat = allocstatus)
                if (allocstatus /= 0) then
                    Do ErrChannelIndex = 1, 2
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x, "Error in subroutine ModelCalcSpectrum!")')
                        write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                        write(ErrChannel, '(5x, "- IntPerCompArray(i, j, MaxNumIso)")')
                        write(ErrChannel, '(5x, "- TauPerCompArray(i, j, MaxNumIso)")')
                        write(ErrChannel, '(5x, "- TdPerCompArray(i, j)")')
                        write(ErrChannel, '(5x, "- TauHelperArrayEM(MaxNumIso)")')
                        write(ErrChannel, '(5x, "- TauHelperArrayABS(MaxNumIso)")')
                        write(ErrChannel, '(5x, "- TauHelperArray2(MaxNumIso)")')
                        write(ErrChannel, '(5x, "- LocalIntArray(MaxNumIso)")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Used parameters:")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(5x, "- i = ", I8)') i
                        write(ErrChannel, '(5x, "- j = ", I8)') j
                        write(ErrChannel, '(5x, "- MaxNumIso = ", I8)') MaxNumIso
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(" ")')
                    end Do
                    stop 'Program aborted! Please restart the program!'
                endif
                IntPerCompArray = 0.d0
                TauPerCompArray = 0.d0
                TdPerCompArray = 0.d0
                TauHelperArrayEM = 0.d0
                TauHelperArrayABS = 0.d0
                TauHelperArray2 = 0.d0
                LocalIntArray = 0.d0
            endif
            !<############################################################################################################################################
            !<############################################################################################################################################


            !<============================================================================================================================================
            !< start loop over molecules, frequency ranges, components, transitions
            Do l = 1, TotalNumberOfFrequencyRanges                                          !< loop over all frequency ranges
                i = SpectrumIndexForFreqRange(l)                                            !< get spectrum index
                FirstIndex = DataPointIndexFreqRange(l, 1)                                  !< get index of first freq. point in 'ObservationalDataList'
                LastIndex = DataPointIndexFreqRange(l, 2)                                   !< get index of last freq. point in 'ObservationalDataList'
                LocalMaxNumTrans = 1

                ! Debug:
                ! print*,"l = ", l
                ! print*,"StartFrequency(l) = ", StartFrequency(l)
                ! print*,"EndFrequency(l)   = ", EndFrequency(l)
                ! print*,"StepFrequency(l)  = ", StepFrequency(l)
                ! print*,"FirstIndex = ", FirstIndex
                ! print*,"LastIndex  = ", LastIndex


                !<########################################################################################################################################
                !< Only used for storing opacities and intensities
                !<########################################################################################################################################
                !< allocate variables for output variables
                if (AllOutputFilesFlag) then
                    i = NumDataPoints
                    j = TotalNumberComponents
                    k = maxval(NumEntriesRanges(l, :))
                    if (allocated(IntensityPerPeak)) then
                        deallocate(IntensityPerPeak, PrevIntPerPeak, IntegHelperArray1, IntegHelperArray2, stat = deallocstatus)
                        if (deallocstatus /= 0) then
                            Do ErrChannelIndex = 1, 2
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x, "Error in subroutine ModelCalcSpectrum!")')
                                write(ErrChannel, '(3x, "Can not deallocate variables IntensityPerPeak, &
                                                        &PrevIntPerPeak, IntegHelperArray1, and IntegHelperArray2!")')
                                write(ErrChannel, '(" ")')
                            end Do
                            stop 'Program aborted! Please restart the program!'
                        endif
                    endif
                    allocate(IntensityPerPeak(TotalNumberOfFrequencyRanges, j, MaxNumIso, k), PrevIntPerPeak(MaxNumIso, k), &
                                IntegHelperArray1(MaxNumIso, k), IntegHelperArray2(MaxNumIso, k), stat = allocstatus)
                    if (allocstatus /= 0) then
                        Do ErrChannelIndex = 1, 2
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x, "Error in subroutine ModelCalcSpectrum!")')
                            write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                            write(ErrChannel, '(5x, "- IntensityPerPeak(TotalNumberOfFrequencyRanges, j, MaxNumIso, k)")')
                            write(ErrChannel, '(5x, "- PrevIntPerPeak(MaxNumIso, k)")')
                            write(ErrChannel, '(5x, "- IntegHelperArray1(MaxNumIso, k), IntegHelperArray2(MaxNumIso, k)")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x,"Used parameters:")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(5x, "- TotalNumberOfFrequencyRanges = ", I8)') TotalNumberOfFrequencyRanges
                            write(ErrChannel, '(5x, "- j = ", I8)') j
                            write(ErrChannel, '(5x, "- MaxNumIso = ", I8)') MaxNumIso
                            write(ErrChannel, '(5x, "- k = ", I8)') k
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        stop 'Program aborted! Please restart the program!'
                    endif
                    IntensityPerPeak = 0.d0
                    PrevIntPerPeak = 0.d0
                    IntegHelperArray1 = 0.d0
                    IntegHelperArray2 = 0.d0
                endif
                !<########################################################################################################################################
                !<########################################################################################################################################


                !<========================================================================================================================================
                !< use sub-beam description
                NumberConfig = 0
                if (SubBeamDescriptionFlag) then


                    !< for sub-beam description determine which components have to be taken into account for each pixel
                    call CreatePixelMap(NumberConfig, l, TotalNumberComponents, NumberModelPixelXX, NumberModelPixelYY, &
                                        CopymyXCLASSParameter, CopyCompMoleculeIndex)

                    ! Debug:
                    ! print*,"NumberConfig = ", NumberConfig


                    !< allocate memory for model spectra of unique configurations
                    if (NumberConfig > 0) then


                        !< get number of frequency points
                        FirstIndex = DataPointIndexFreqRange(l, 1)                          !< get index of first freq. point in 'ObservationalDataList'
                        LastIndex = DataPointIndexFreqRange(l, 2)                           !< get index of last freq. point in 'ObservationalDataList'
                        LocalNumFreqPoints = LastIndex - FirstIndex + 1                     !< determine number of frequency points for curr. freq. range
                        if (allocated(ModelPixelSpectra)) then
                            deallocate(ModelPixelSpectra, stat = deallocstatus)
                            if (deallocstatus /= 0) then                                    !< is all ok?
                                Do ErrChannelIndex = 1, 2
                                    ErrChannel = AllErrChannels(ErrChannelIndex)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x, "Error in subroutine CreatePixelMap!")')
                                    write(ErrChannel, '(3x, "Can not deallocate variable ModelPixelSpectra!")')
                                    write(ErrChannel, '(" ")')
                                end Do
                                stop 'Program aborted! Please restart the program!'
                            endif
                        endif
                        allocate(ModelPixelSpectra(NumberConfig, LocalNumFreqPoints), stat = allocstatus)
                        if (allocstatus /= 0) then                                          !< is all ok?
                            Do ErrChannelIndex = 1, 2
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x, "Error in subroutine CreatePixelMap!")')
                                write(ErrChannel, '(3x, "Can not allocate variable ModelPixelSpectra(NumberConfig, &
                                                         &LocalNumFreqPoints)!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Used parameters:")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(5x, "- NumberConfig = ", I8)') NumberConfig
                                write(ErrChannel, '(5x, "- LocalNumFreqPoints = ", I8)') LocalNumFreqPoints
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                            end Do
                        endif
                        ModelPixelSpectra = 0.d0


                        !< allocate additional working variables for myXCLASS function
                        if (AllOutputFilesFlag) then


                            !< determine max. number of isotopologues per component
                            MaxNumIso = 0
                            if (IsoFlag) then
                                Do j = 1, TotalNumberOfMolecules
                                    n = 0
                                    Do k = 1, TotalNumberOfMolecules
                                        if (CopyIsoRatioConversionTable(j, k) /= 0.d0) then !< is a iso ratio defined for mth molecule?
                                            n = n + 1
                                        endif
                                    end Do
                                    if (MaxNumIso < n) then
                                        MaxNumIso = n
                                    endif
                                end Do
                            endif
                            MaxNumIso = max0(1, MaxNumIso)
                            j = max0(1, TotalNumberComponents)
                            if (allocated(IntPerCompArraySubBeam)) then
                                deallocate(IntPerCompArraySubBeam, TauPerCompArraySubBeam, IntTotalSubBeam, ModelSpectrumPixel, &
                                           stat = deallocstatus)
                                if (deallocstatus /= 0) then                                !< is all ok?
                                    Do ErrChannelIndex = 1, 2
                                        ErrChannel = AllErrChannels(ErrChannelIndex)
                                        write(ErrChannel, '(" ")')
                                        write(ErrChannel, '(3x, "Error in subroutine CreatePixelMap!")')
                                        write(ErrChannel, '(3x, "Can not deallocate variables IntPerCompArraySubBeam, &
                                                                 &TauPerCompArraySubBeam, ...!")')
                                        write(ErrChannel, '(" ")')
                                    end Do
                                    stop 'Program aborted! Please restart the program!'
                                endif
                            endif
                            allocate(IntPerCompArraySubBeam(NumberConfig, LocalNumFreqPoints, j, MaxNumIso), &
                                     TauPerCompArraySubBeam(NumberConfig, LocalNumFreqPoints, j, MaxNumIso), &
                                     IntTotalSubBeam(NumberModelPixelXX, NumberModelPixelYY, LocalNumFreqPoints), &
                                     ModelSpectrumPixel(NumberConfig, LocalNumFreqPoints), stat = allocstatus)
                            if (allocstatus /= 0) then                                      !< is all ok?
                                Do ErrChannelIndex = 1, 2
                                    ErrChannel = AllErrChannels(ErrChannelIndex)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x, "Error in subroutine CreatePixelMap!")')
                                    write(ErrChannel, '(3x, "Can not allocate the following variables:")')
                                    write(ErrChannel, '(5x, "- IntPerCompArraySubBeam(NumberConfig, LocalNumFreqPoints, j, &
                                                            &MaxNumIso)")')
                                    write(ErrChannel, '(5x, "- TauPerCompArraySubBeam(NumberConfig, LocalNumFreqPoints, j, &
                                                            &MaxNumIso)")')
                                    write(ErrChannel, '(5x, "- IntTotalSubBeam(NumberModelPixelXX, NumberModelPixelYY, &
                                                            &LocalNumFreqPoints)")')
                                    write(ErrChannel, '(5x, "- ModelSpectrumPixel(NumberConfig, LocalNumFreqPoints)")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x,"Used parameters:")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(5x, "- NumberConfig = ", I8)') NumberConfig
                                    write(ErrChannel, '(5x, "- LocalNumFreqPoints = ", I8)') LocalNumFreqPoints
                                    write(ErrChannel, '(5x, "- j = ", I8)') j
                                    write(ErrChannel, '(5x, "- MaxNumIso = ", I8)') MaxNumIso
                                    write(ErrChannel, '(5x, "- NumberModelPixelXX = ", I8)') NumberModelPixelXX
                                    write(ErrChannel, '(5x, "- NumberModelPixelYY = ", I8)') NumberModelPixelYY
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(" ")')
                                end Do
                            endif
                            IntPerCompArraySubBeam = 0.d0
                            TauPerCompArraySubBeam = 0.d0
                            IntTotalSubBeam = 0.d0
                            ModelSpectrumPixel = 0.d0
                        endif


                        !< compute the modelled spectrum for current frequency range for given model pixel
                        Do k = 1, NumberConfig                                              !< loop over unique conifgurations


                            !< determine the distances and corresponding components for current component configuration
                            LocalPureDistances = 0.d0
                            LocalDistanceOrderingArray = 0
                            LocalNumberDistances = 0
                            Do i = 1, NumberDistances                                       !< loop over pure distances
                                kk = 0
                                Do j = 1, TotalNumberComponents                             !< loop over all comp. which correspond to current dist.
                                    if (DistanceOrderingArray(i, j) == 0) then              !< if all components are analyzed exit loop
                                        exit
                                    else
                                        c = DistanceOrderingArray(i, j)                     !< get component index

                                        ! Debug:
                                        ! print*,' '
                                        ! print*,"k, i, j, c = ", k, i, j, c
                                        ! print*,"ConfigList(k, :) = ", ConfigList(k, :)
                                        ! print*,"ConfigList(k, c) = ", ConfigList(k, c)
                                        ! print*,"DistanceOrderingArray(i, j) = ", DistanceOrderingArray(i, j)


                                        if (ConfigList(k, c) .and. c /= 0) then             !< is current component included in the current configuration
                                            kk = kk + 1
                                            if (kk == 1) then
                                                LocalNumberDistances = LocalNumberDistances + 1
                                                LocalPureDistances(LocalNumberDistances) = PureDistances(i)
                                            endif
                                            LocalDistanceOrderingArray(LocalNumberDistances, kk) = c
                                        endif
                                    endif
                                end Do                                                      !< j: loop over all comp. which correspond to current dist.
                            end Do                                                          !< i: loop over pure distances

                            ! Debug:
                            ! print*," "
                            ! print*,"k = ", k
                            ! print*,"LocalNumberDistances = ", LocalNumberDistances
                            ! print*,"LocalPureDistances = ", LocalPureDistances
                            ! print*,"LocalDistanceOrderingArray(1, :) = ", LocalDistanceOrderingArray(1, :)
                            ! print*,"LocalDistanceOrderingArray(2, :) = ", LocalDistanceOrderingArray(2, :)
                            ! print*,"LocalDistanceOrderingArray(3, :) = ", LocalDistanceOrderingArray(3, :)


                            !< compute spectrum for current component configuration
                            call CalcModelPixelSpectrum(chi2Value, ModelFuncList, NumDataPoints, TotalNumMol, NumComp, &
                                                        CopyCompMoleculeIndex, Q, CopyIsoRatioConversionTable, &
                                                        CopymyXCLASSParameter, FirstIndex, LastIndex, l, &
                                                        LocalModelFunctionFlag, LocalMaxNumTrans, k, NumberConfig, &
                                                        LocalPureDistances, LocalDistanceOrderingArray, LocalNumberDistances)
                            ModelPixelSpectra(k, :) = ModelFuncList(FirstIndex:LastIndex)   !< store model pixel spectrum
                            if (AllOutputFilesFlag) then
                                ModelSpectrumPixel(k, :) = ModelFuncList(FirstIndex:LastIndex)
                                IntPerCompArraySubBeam(k, :, :, :) = IntPerCompArray(FirstIndex:LastIndex, :, :)
                                TauPerCompArraySubBeam(k, :, :, :) = TauPerCompArray(FirstIndex:LastIndex, :, :)
                            endif

                            ! Debug:
                            ! write(numberstring1, '(I30)') k
                            ! open(77777, file = "spectrum_for_configuration__" // trim(adjustl(numberstring1)) // ".dat")
                            ! Do c = FirstIndex, LastIndex
                            !     write(77777,*) c, ModelFuncList(c)
                            ! end Do
                            ! close(77777)
                        end Do                                                              !< k: loop over unique conifgurations


                        !< print what you do
                        if (printflag .and. LogFlag) then
                            print '(A, "Computation of pixel spectra finished!&
                                        &                                                       ")', char(13)
                        endif


                        !< compute map describing elliptical rotated Gaussian beam
                        call ComputeGaussianBeamMap(l, NumberModelPixelXX, NumberModelPixelYY)


                        !< convolve model map with beam to get final spectrum
                        ModeID = 0
                        IsoCounter = 0
                        call ConvolveModelMap(chi2Value, ModelFuncList, ModeID, l, NumDataPoints, LocalNumFreqPoints, &
                                              NumberModelPixelXX, NumberModelPixelYY, LocalModelFunctionFlag, IsoCounter, &
                                              NumberConfig, ModelPixelSpectra)
                    endif


                    !<####################################################################################################################################
                    !< Only used for storing opacities and intensities
                    !<####################################################################################################################################
                    !< convolve maps describing intensities and optical depths for each component with beam to get final spectrum
                    if (AllOutputFilesFlag) then
                        Do c = 1, NumComp                                                   !< loop over components
                            MoleculeIndex = CopyCompMoleculeIndex(c)                        !< get molecule index
                            ll = MoleculeIndex                                              !< get molecule index
                            if (IsoFlag) then
                                kk = TotalNumMol
                            else
                                kk = ll
                            endif
                            IsoCounter = 0
                            Do MoleculeIndexLoop = ll, kk                                   !< loop over all molecule indices (including isotopologues!)
                                CalcFalg = .true.
                                if (IsoFlag) then
                                    if (CopyIsoRatioConversionTable(ll, MoleculeIndexLoop) == 0.d0) then
                                        CalcFalg = .false.
                                    endif
                                endif
                                if (CalcFalg) then                                          !< consider current molecule
                                    IsoCounter = IsoCounter + 1


                                    !< convole intensities per component
                                    call ConvolveModelMap(chi2Value, ModelFuncList, c, l, NumDataPoints, LocalNumFreqPoints, &
                                                          NumberModelPixelXX, NumberModelPixelYY, LocalModelFunctionFlag, &
                                                          IsoCounter, NumberConfig, ModelPixelSpectra)


                                    !< convole optical depths per component
                                    call ConvolveModelMap(chi2Value, ModelFuncList, -c, l, NumDataPoints, LocalNumFreqPoints, &
                                                          NumberModelPixelXX, NumberModelPixelYY, LocalModelFunctionFlag, &
                                                          IsoCounter, NumberConfig, ModelPixelSpectra)
                                endif
                            end Do                                                          !< MoleculeIndexLoop: loop over all molecule indices
                        end Do                                                              !< c: loop over components
                        if (printflag .and. LogFlag) then
                            print '(A, "Convolution is finished!&
                                        &                                                               ")', char(13)
                        endif
                    endif
                    !<####################################################################################################################################
                    !<####################################################################################################################################


                !<========================================================================================================================================
                !< do not use sub-beam description
                else
                    ! Debug:
                    ! print*,"NumDataPoints = ", NumDataPoints
                    ! print*,"TotalNumMol = ", TotalNumMol
                    ! print*,"NumComp = ", NumComp
                    ! print*,"CopyCompMoleculeIndex = ", CopyCompMoleculeIndex
                    ! print*,"Q = ", Q
                    ! print*,"CopyIsoRatioConversionTable = ", CopyIsoRatioConversionTable
                    ! print*,"CopymyXCLASSParameter = ", CopymyXCLASSParameter
                    ! print*,"FirstIndex = ", FirstIndex
                    ! print*,"LastIndex = ", LastIndex
                    ! print*,"l = ", l
                    ! print*,"LocalModelFunctionFlag = ", LocalModelFunctionFlag
                    ! print*,"LocalMaxNumTrans = ", LocalMaxNumTrans
                    ! print*,"k = ", k
                    ! print*,"NumberConfig = ", NumberConfig
                    ! print*,"PureDistances = ", PureDistances
                    ! print*,"DistanceOrderingArray = ", DistanceOrderingArray
                    ! print*,"NumberDistances = ", NumberDistances


                    !< compute the modelled spectrum for current frequency range for given model pixel
                    k = 0
                    call CalcModelPixelSpectrum(chi2Value, ModelFuncList, NumDataPoints, TotalNumMol, NumComp, &
                                                CopyCompMoleculeIndex, Q, CopyIsoRatioConversionTable, &
                                                CopymyXCLASSParameter, FirstIndex, LastIndex, l, LocalModelFunctionFlag, &
                                                LocalMaxNumTrans, k, NumberConfig, PureDistances, DistanceOrderingArray, &
                                                NumberDistances)
                endif
            end Do                                                                          !< l: loop over frequency ranges

            ! Debug:
            ! print*,"DataPointIndexFreqRange(:, 1) = ", DataPointIndexFreqRange(:, 1)
            ! print*,"DataPointIndexFreqRange(:, 2) = ", DataPointIndexFreqRange(:, 2)
            ! print*,"ObservationalDataList(DataPointIndexFreqRange(1, 1), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 1), :)
            ! print*,"ObservationalDataList(DataPointIndexFreqRange(1, 2), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 2), :)
            ! print*,'>>>>>>>>>>ThreadNumber, count(isnan(ModelFuncList(:))) = ', ThreadNumber, count(isnan(ModelFuncList(:)))


            !<############################################################################################################################################
            !< Only used by myXCLASS function!
            !<############################################################################################################################################
            !< write output variables to file
            if (LogFlag) then


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< open output file for transition energies
                if (AllOutputFilesFlag) then
                    open(2223, file = 'transition_energies.dat', status = 'unknown')
                    write(2223, '(4x, "% Trans. freq. (MHz):", 4x, "Doppler-shift freq. (MHz):", 16x, "Intensity (K):", $)')
                    write(2223, '(14x, "Integrated Int.:", 20x, "E_low (K):", 25x, "g_up:", 11x, "Einstein A (s^(-1):", $)')
                    write(2223, '(25x, "Quantum numbers of upper level:", 34x, "Quantum numbers of lower level:", 14x, &
                                        &"Molecule name:")')
                endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< write contents of output arrays to file
                Do l = 1, TotalNumberOfFrequencyRanges                                      !< loop over all frequency ranges
                    i = SpectrumIndexForFreqRange(l)                                        !< get spectrum index
                    FirstIndex = DataPointIndexFreqRange(l, 1)                              !< get index of first freq. point in 'ObservationalDataList'
                    LastIndex = DataPointIndexFreqRange(l, 2)                               !< get index of last freq. point in 'ObservationalDataList'


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< open output files for intensities of each component of each molecule
                    if (EmAbsFlag) then
                        Do c = 1, NumberDistances                                           !< loop over all layers / distances
                            write(numberstring2, '(ES25.15)') PureDistances(c)              !< convert float to string
                            LocalFileName = "emission_absorption____distance__" // trim(adjustl(numberstring2)) // ".dat"
                            if (l == 1) then
                                open(19144, file = trim(adjustl(LocalFileName)))
                            else
                                open(19144, file = trim(adjustl(LocalFileName)), position = 'append')
                            endif
                            write(19144, '("current distance = ", I10)') c
                            write(19144, '("total number of distances = ", I10)') NumberDistances
                            write(19144, '(10x, "Frequency [MHz]:", 21x, "Emission:", 19x, "Absorption:")')


                            !< write emission and absorption as function of frequency for current layer to file
                            Do FreqIndex = FirstIndex, LastIndex                            !< loop over all frequency points in the current freq. range
                                Obsfreq = ObservationalDataList(FreqIndex, 1)               !< get frequency
                                write(19144, '(1x, ES25.10, 2(5x, ES25.15E3))') Obsfreq, TauEmAbsArray(FreqIndex, c, 1:2)
                            end Do                                                          !< FreqIndex: loop over frequency points
                            close(19144)                                                    !< close output file for emission/absorption per layer
                        end Do                                                              !< c: loop over all layers / distances
                    endif


                    !< write modeled spectrum to output file
                    Do FreqIndex = FirstIndex, LastIndex                                    !< loop over all frequency points in the current freq. range
                        Obsfreq = ObservationalDataList(FreqIndex, 1)                       !< get frequency
                        write(2222, '(1x, ES25.10, 1x, ES25.16E3)') Obsfreq, ModelFuncList(FreqIndex)
                    end Do
                    if (AllOutputFilesFlag) then


                        !< write telescope beam full width half maximum sizes for \nu_min and \nu_max to log file
                        if (.not. InterFlag(l)) then
                            write(2224,'(" ")')
                            Do t = 1, 2
                                write(2224,'(" ")')
                                if (t == 1) then
                                    x0 = ObservationalDataList(FirstIndex, 1)
                                    write(2224,'("Beginning of frequency range", $)')
                                else
                                    x0 = ObservationalDataList(LastIndex, 1)
                                    write(2224,'("End of frequency range", $)')
                                endif
                                write(numberstring1, '(F25.3)') x0
                                write(2224,'(" (\nu = ", A, " MHz):")') trim(adjustl(numberstring1))
                                write(2224,'(" ")')


                                !< write telescope beam full width half maximum sizes for \nu_min and \nu_max to log file
                                if (InterFlag(l)) then
                                    beam_size = TelescopeSize(l)
                                else
                                    beam_size = 1.22d-3 * ckms / (x0 * TelescopeSize(l)) * (180.d0 / pi) * 3600.d0
                                endif
                                write(numberstring2, '(F18.5)') beam_size
                                write(2224, '(2x, "Telescope beam FWHM size:                       \theta_T (\nu) = ", A, &
                                            &"arcsec")') trim(adjustl(numberstring2))
                                write(2224,'(" ")')
                            end Do
                        endif
                        write(2224,'(" ")')


                        !< write intensities and opacities for each component to files
                        fff = 0
                        Do c = 1, TotalNumberComponents                                     !< loop over all components (i.e. c is global comp. index)
                            MoleculeIndex = CopyCompMoleculeIndex(c)                        !< get molecule index
                            ll = MoleculeIndex                                              !< get molecule index
                            if (IsoFlag) then
                                kk = TotalNumMol
                            else
                                kk = ll
                            endif
                            IsoCounter = 0
                            Do MoleculeIndexLoop = ll, kk                                   !< loop over all molecule indices (including isotopologues!)
                                CalcFalg = .true.
                                if (IsoFlag) then
                                    if (CopyIsoRatioConversionTable(ll, MoleculeIndexLoop) == 0.d0) then
                                        CalcFalg = .false.
                                    endif
                                endif
                                if (CalcFalg) then                                          !< consider current molecule
                                    IsoCounter = IsoCounter + 1
                                    fff = fff + 1
                                    CurrMoleculeName = trim(adjustl(MoleculeNames(MoleculeIndexLoop)))  !< get name of current molecule


                                    !< correct molecule name
                                    ModifiedMoleculeName = CurrMoleculeName
                                    Do n = 1, len_trim(CurrMoleculeName)
                                        if (CurrMoleculeName(n:n) == ";" .or. CurrMoleculeName(n:n) == "(" &
                                            .or. CurrMoleculeName(n:n) == ")" .or. CurrMoleculeName(n:n) == "'" &
                                            .or. CurrMoleculeName(n:n) == ";") then
                                            ModifiedMoleculeName(n:n) = "_"
                                        endif
                                    end Do


                                    !< determine current comp. index and total number of component of current molecule
                                    ncomp = 0
                                    n = 0
                                    Do i = 1, TotalNumberComponents
                                        if (CopyCompMoleculeIndex(i) == CopyCompMoleculeIndex(c)) then
                                            ncomp = ncomp + 1
                                            if (i <= c) then
                                                n = n + 1
                                            endif
                                        endif
                                    end Do
                                    if (LocalOverlapFlag) then
                                        ncomp = 0
                                        Do i = 1, NumberDistances                           !< loop over pure distances
                                            Do j = 1, TotalNumberComponents                 !< loop over all comp. which correspond to current dist.
                                                if (DistanceOrderingArray(i, j) == 0) then  !< if all components are analyzed exit loop
                                                    exit
                                                else
                                                    if (c == DistanceOrderingArray(i, j)) then  !< get component index
                                                        ncomp = i
                                                        exit
                                                    endif
                                                endif
                                            end Do
                                        end Do
                                        write(numberstring2, '(I30)') ncomp                 !< convert integer to string
                                    else
                                        write(numberstring2, '(I30)') n                     !< convert integer to string
                                    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< write dust temperature for each component to log-file
                                    if ((.not. SubBeamDescriptionFlag) .and. (.not. tbFlag(l))) then
                                        write(2224,'(" ")')
                                        write(2224, '("Molecule: ", A, ", component: ", A)') trim(adjustl(CurrMoleculeName)), &
                                                    trim(adjustl(numberstring2))
                                        x0 = TdPerCompArray(FirstIndex, c)
                                        write(2224, '(2x, "Dust temperature (beginning of frequency range):&
                                                    &       T_d (\nu_{min}) = ",5x,F15.5, " K")') x0
                                        x0 = TdPerCompArray(LastIndex, c)
                                        write(2224, '(2x, "Dust temperature (end of frequency range):&
                                                    &             T_d (\nu_{max}) = ",5x,F15.5, " K")') x0
                                    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< open output files for intensities of each component of each molecule
                                    if (LocalOverlapFlag) then
                                        IntCompFileName(c) = "intensity__" // trim(adjustl(ModifiedMoleculeName)) &
                                                            // "__distance__" &
                                                            // trim(adjustl(numberstring2)) // ".dat"
                                        if (l == 1) then
                                            open(19143, file = trim(adjustl(IntCompFileName(c))))
                                        else
                                            open(19143, file = trim(adjustl(IntCompFileName(c))), position = 'append')
                                        endif
                                        write(19143, '("name of current molecule = ", A)') trim(adjustl(CurrMoleculeName))
                                        write(19143, '("current distance = ", I10)') ncomp
                                        write(19143, '("total number of distances = ", I10)') NumberDistances
                                    else
                                        IntCompFileName(c) = "intensity__" // trim(adjustl(ModifiedMoleculeName)) // "__comp__" &
                                                             // trim(adjustl(numberstring2)) // ".dat"
                                        if (l == 1) then
                                            open(19143, file = trim(adjustl(IntCompFileName(c))))
                                        else
                                            open(19143, file = trim(adjustl(IntCompFileName(c))), position = 'append')
                                        endif
                                        write(19143, '("name of current molecule = ", A)') trim(adjustl(CurrMoleculeName))
                                        write(19143, '("current component = ", I10)') n
                                        write(19143, '("total number of components = ", I10)') ncomp
                                    endif
                                    write(19143, '(10x, "Frequency [MHz]:", 21x, "T_mb [K]:")')


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< open output files for optical depths of each component of each molecule
                                    if (LocalOverlapFlag) then
                                        TauCompFileName(c) = "optical_depth__" // trim(adjustl(ModifiedMoleculeName)) &
                                                            // "__distance__" // trim(adjustl(numberstring2)) // ".dat"
                                        if (l == 1) then
                                            open(142, file = trim(adjustl(TauCompFileName(c))))
                                        else
                                            open(142, file = trim(adjustl(TauCompFileName(c))), position = 'append')
                                        endif
                                        write(142, '("name of current molecule = ", A)') trim(adjustl(CurrMoleculeName))
                                        write(142, '("current distance = ", I10)') n
                                        write(142, '("total number of distances = ", I10)') NumberDistances
                                    else
                                        TauCompFileName(c) = "optical_depth__" // trim(adjustl(ModifiedMoleculeName)) &
                                                            // "__comp__" // trim(adjustl(numberstring2)) // ".dat"
                                        if (l == 1) then
                                            open(142, file = trim(adjustl(TauCompFileName(c))))
                                        else
                                            open(142, file = trim(adjustl(TauCompFileName(c))), position = 'append')
                                        endif
                                        write(142, '("name of current molecule = ", A)') trim(adjustl(CurrMoleculeName))
                                        write(142, '("current component = ", I10)') n
                                        write(142, '("total number of components = ", I10)') ncomp
                                    endif
                                    write(142, '(10x, "Frequency [MHz]:", 26x, "tau:")')


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< open output files for optical depths of each component of each molecule
                                    Do FreqIndex = FirstIndex, LastIndex                    !< loop over all frequency points in the current freq. range
                                        Obsfreq = ObservationalDataList(FreqIndex, 1)       !< get frequency


                                        !< write intensities for each component and molecule to file
                                        write(19143, '(1x, ES25.10, 5x, ES25.15E3)') Obsfreq, &
                                                                                     IntPerCompArray(FreqIndex, c, IsoCounter)


                                        !< write optical depth tau for each component and molecule to file
                                        write(142, '(1x, ES25.10, 5x, ES25.15E3)') Obsfreq, &
                                                                                   TauPerCompArray(FreqIndex, c, IsoCounter)
                                    end Do                                                  !< FreqIndex: loop over frequencies


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< write transition energies to output file
                                    FirstMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 1) !< get first mol. index of current freq. range
                                    LastMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 2)  !< get last mol. index of current freq. range
                                    if (FirstMolecularDataIndex > 0) then
                                        Do t = FirstMolecularDataIndex, LastMolecularDataIndex  !< loop over all transitions of current molecule
                                            !    'transition frequency = ', MolecularData(t, 1)
                                            !    'EinsteinA            = ', MolecularData(t, 2)
                                            !    'E_low                = ', MolecularData(t, 3)
                                            !    'g_u                  = ', MolecularData(t, 4)
                                            if (t > 0) then


                                                !< determine Doppler-shifted transition frequencies
                                                x0 = MolecularData(t, 1) &
                                                        * ( 1.d0 - (CopymyXCLASSParameter(6, c) + GlobalvLSR(l)) / ckms)


                                                !< determine modeled spectrum at Doppler-shifted transition frequencies
                                                f0 = 0.d0
                                                FreqIndex = minloc(dabs(x0 - ObservationalDataList(FirstIndex:LastIndex, 1)), &
                                                                   dim = 1)
                                                f0 = ModelFuncList(FreqIndex)


                                                !< write frequencies to output file
                                                if (n == 1) then
                                                    write(2223,'(ES25.15, $)') MolecularData(t, 1)
                                                else
                                                    write(2223,'(24x, " ", $)')
                                                endif
                                                write(2223,'(2(5x, ES25.15), $)') x0, f0


                                                !< write integrated intensities to file (not for sub-beam description)
                                                if (SubBeamDescriptionFlag .or. LocalOverlapFlag) then
                                                    write(2223,'(5x, A25, $)') "-"
                                                else
                                                    f1 = IntensityPerPeak(l, c, IsoCounter, t - FirstMolecularDataIndex + 1)
                                                    write(2223,'(5x, ES25.15, $)') f1
                                                endif


                                                !< add other transition parameter to log file
                                                if (n == 1) then
                                                    write(2223,'(5x, ES25.15, 5x, ES25.15, 5x, ES25.15, $)') MolecularData(t, 3), &
                                                                                                             MolecularData(t, 4), &
                                                                                                             MolecularData(t, 2)
                                                else
                                                    write(2223,'(29x, " ", 29x, " ", 29x, " ", $)')
                                                endif
                                                write(2223,'(5x, A60, 5x, A60, 5x, A)') adjustl(LowerQNString(t)), &
                                                                                        adjustl(UpperQNString(t)), &
                                                                                        adjustl(CurrMoleculeName)
                                            else
                                                exit
                                            endif
                                        end Do                                              !< t: loop over transitions
                                    endif
                                    close(142)                                              !< close output file for optical depths for each component
                                    close(19143)                                            !< close output file for intensities for each component
                                endif                                                       !< CalcFlag
                            end Do                                                          !< MoleculeIndexLoop: loop over molecules
                        end Do                                                              !< c: loop over all components
                    endif
                end Do                                                                      !< l: loop over frequency ranges
                close(2223)                                                                 !< close output file for transition energies


                !< if global iso ratio files are defined, write used iso ratios for all isotopologue to log file
                if (IsoFlag) then
                    if (AllOutputFilesFlag) then
                        write(2224,'(" ")')
                        write(2224,'(" ")')
                        write(2224,'("Used iso ratios:")')
                        write(2224,'(" ")')
                        write(2224,'(4x, "Isotopologue:", 27x, "Iso-master:", 44x, "Iso-ratio:")')
                        write(2224,'(" ")')
                        Do n = 1, TotalNumberOfMolecules
                            Do l = (n + 1), TotalNumberOfMolecules
                                if (CopyIsoRatioConversionTable(l, n) /= 0.d0) then
                                    write(2224,'(4x, A40, 10x, $)') adjustl(MoleculeNames(l))
                                    write(2224,'(A40, 10x, $)') adjustl(MoleculeNames(n))
                                    write(2224,'(ES25.5)') CopyIsoRatioConversionTable(l, n)
                                endif
                            end Do
                        end Do
                    endif
                endif
            endif
            !<############################################################################################################################################
            !<############################################################################################################################################

            ! Debug:
            !    CALL CPU_TIME(tt2)
            !    print '("Total time for subroutine ModelCalcSpectrum: ", F20.3, " sec. = ", F20.3, " min.")', (tt2 - tt1), (tt2 - tt1) / 60.d0
            !    stop '###### stop subroutine ModelCalcSpectrum ######'


            !< we're done
            return
        end subroutine ModelCalcSpectrum


        !>************************************************************************************************************************************************
        !> subroutine: myXCLASSParamFree
        !>
        !> free memory used by myXCLASS variables
        !>
        !>
        !> input variables:     deallocstatus           status of the previous deallocation process
        !>
        !> output variables:    deallocstatus           status of the deallocation process
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date   2014-08-26
        !>
        subroutine myXCLASSParamFree(deallocstatus)

            implicit none
            integer :: deallocstatus                                                        !< status of the deallocation process


            !< deallocate memory of variables of Modules 'Variables' and 'FunctionCalling'
            if (allocated(lengthexpdata)) deallocate(lengthexpdata, stat = deallocstatus)
            if (allocated(NumberXColumns)) deallocate(NumberXColumns, stat = deallocstatus)
            if (allocated(NumberYColumns)) deallocate(NumberYColumns, stat = deallocstatus)
            if (allocated(NumberHeaderLines)) deallocate(NumberHeaderLines, stat = deallocstatus)
            if (allocated(OutputFileFormat)) deallocate(OutputFileFormat, stat = deallocstatus)
            if (allocated(NumberRanges)) deallocate(NumberRanges, stat = deallocstatus)
            if (allocated(ConverterInfit)) deallocate(ConverterInfit, stat = deallocstatus)
            if (allocated(ValueEmptyOutputFile)) deallocate(ValueEmptyOutputFile, stat = deallocstatus)
            if (allocated(LSRAdjustement)) deallocate(LSRAdjustement, stat = deallocstatus)
            if (allocated(chisqValues)) deallocate(chisqValues, stat = deallocstatus)
            if (allocated(BestSitesParamSet)) deallocate(BestSitesParamSet, stat = deallocstatus)
            if (allocated(paramset)) deallocate(paramset, stat = deallocstatus)
            if (allocated(AtOnceFunction)) deallocate(AtOnceFunction, stat = deallocstatus)
            if (allocated(AtOnceGradient)) deallocate(AtOnceGradient, stat = deallocstatus)
            if (allocated(FirstPointExpData)) deallocate(FirstPointExpData, stat = deallocstatus)
            if (allocated(LastPointExpData)) deallocate(LastPointExpData, stat = deallocstatus)
            if (allocated(expdatax)) deallocate(expdatax, stat = deallocstatus)
            if (allocated(expdatay)) deallocate(expdatay, stat = deallocstatus)
            if (allocated(expdatae)) deallocate(expdatae, stat = deallocstatus)
            if (allocated(CalculatedParameterSets)) deallocate(CalculatedParameterSets, stat = deallocstatus)
            if (allocated(MinRange)) deallocate(MinRange, stat = deallocstatus)
            if (allocated(MaxRange)) deallocate(MaxRange, stat = deallocstatus)
            if (allocated(BestSitesModelValues)) deallocate(BestSitesModelValues, stat = deallocstatus)
            if (allocated(BestSitesChi2Values)) deallocate(BestSitesChi2Values, stat = deallocstatus)
            if (allocated(ModelFunction)) deallocate(ModelFunction, stat = deallocstatus)
            if (allocated(FitParameterName)) deallocate(FitParameterName, stat = deallocstatus)
            if (allocated(FitParameterValue)) deallocate(FitParameterValue, stat = deallocstatus)
            if (allocated(CharacterForComments)) deallocate(CharacterForComments, stat = deallocstatus)
            if (allocated(CharacterSeperatingColumns)) deallocate(CharacterSeperatingColumns, stat = deallocstatus)
            if (allocated(ResamplingMethod)) deallocate(ResamplingMethod, stat = deallocstatus)
            if (allocated(InterpolationMethod)) deallocate(InterpolationMethod, stat = deallocstatus)
            if (allocated(OnlyYColumn)) deallocate(OnlyYColumn, stat = deallocstatus)
            if (allocated(LSRAdjustementFitFlag)) deallocate(LSRAdjustementFitFlag, stat = deallocstatus)
            if (allocated(NormalizationFlag)) deallocate(NormalizationFlag, stat = deallocstatus)
            if (allocated(ExpData_reversed_flag)) deallocate(ExpData_reversed_flag, stat = deallocstatus)
            if (allocated(NumberParamPerLine)) deallocate(NumberParamPerLine, stat = deallocstatus)
            if (allocated(ParameterName)) deallocate(ParameterName, stat = deallocstatus)
            if (allocated(ParameterFormat)) deallocate(ParameterFormat, stat = deallocstatus)
            if (allocated(LeadingString)) deallocate(LeadingString, stat = deallocstatus)
            if (allocated(TrailingString)) deallocate(TrailingString, stat = deallocstatus)
            if (allocated(ParamVisible)) deallocate(ParamVisible, stat = deallocstatus)
            if (allocated(FitFktInput)) deallocate(FitFktInput, stat = deallocstatus)
            if (allocated(FitFktOutput)) deallocate(FitFktOutput, stat = deallocstatus)
            if (allocated(valuesModel_output)) deallocate(valuesModel_output, stat = deallocstatus)
            if (allocated(xPointsModel_output)) deallocate(xPointsModel_output, stat = deallocstatus)


            !< deallocate memory of variables of Module 'Model'
            if (allocated(IsoMolecule)) deallocate(IsoMolecule, stat = deallocstatus)
            if (allocated(IsoMasters)) deallocate(IsoMasters, stat = deallocstatus)
            if (allocated(IsoRatio)) deallocate(IsoRatio, stat = deallocstatus)
            if (allocated(StartFrequency)) deallocate(StartFrequency, stat = deallocstatus)
            if (allocated(EndFrequency)) deallocate(EndFrequency, stat = deallocstatus)
            if (allocated(StepFrequency)) deallocate(StepFrequency, stat = deallocstatus)
            if (allocated(BackgroundTemperatureRange)) deallocate(BackgroundTemperatureRange, stat = deallocstatus)
            if (allocated(TemperatureSlopeRange)) deallocate(TemperatureSlopeRange, stat = deallocstatus)
            if (allocated(BackgroundFilenNames)) deallocate(BackgroundFilenNames, stat = deallocstatus)
            if (allocated(BackgroundFromFile)) deallocate(BackgroundFromFile, stat = deallocstatus)
            if (allocated(EmsAbsFunc)) deallocate(EmsAbsFunc, stat = deallocstatus)
            if (allocated(HydrogenColumnDensityRange)) deallocate(HydrogenColumnDensityRange, stat = deallocstatus)
            if (allocated(DustBetaRange)) deallocate(DustBetaRange, stat = deallocstatus)
            if (allocated(KappaRange)) deallocate(KappaRange, stat = deallocstatus)
            if (allocated(DustFilenNames)) deallocate(DustFilenNames, stat = deallocstatus)
            if (allocated(DustTauFromFile)) deallocate(DustTauFromFile, stat = deallocstatus)
            if (allocated(ContPhen_Range)) deallocate(ContPhen_Range, stat = deallocstatus)
            if (allocated(TelescopeSize)) deallocate(TelescopeSize, stat = deallocstatus)
            if (allocated(TelescopeBMIN)) deallocate(TelescopeBMIN, stat = deallocstatus)
            if (allocated(TelescopeBMAJ)) deallocate(TelescopeBMAJ, stat = deallocstatus)
            if (allocated(TelescopeBPA)) deallocate(TelescopeBPA, stat = deallocstatus)
            if (allocated(InterFlag)) deallocate(InterFlag, stat = deallocstatus)
            if (allocated(GlobalvLSR)) deallocate(GlobalvLSR, stat = deallocstatus)
            if (allocated(Redshift_Range)) deallocate(Redshift_Range, stat = deallocstatus)
            if (allocated(SpectrumIndexForFreqRange)) deallocate(SpectrumIndexForFreqRange, stat = deallocstatus)
            if (allocated(ObservationalDataList)) deallocate(ObservationalDataList, stat = deallocstatus)
            if (allocated(NumTransFreqPerObsFreq)) deallocate(NumTransFreqPerObsFreq, stat = deallocstatus)
            if (allocated(DataPointIndexFreqRange)) deallocate(DataPointIndexFreqRange, stat = deallocstatus)
            if (allocated(MoleculeNames)) deallocate(MoleculeNames, stat = deallocstatus)
            if (allocated(KindOfMolecule)) deallocate(KindOfMolecule, stat = deallocstatus)
            if (allocated(LineProfileFunction)) deallocate(LineProfileFunction, stat = deallocstatus)
            if (allocated(ConversionTableMAGIXmyXCLASSParam)) deallocate(ConversionTableMAGIXmyXCLASSParam, stat = deallocstatus)
            if (allocated(IsoRatioConversionTable)) deallocate(IsoRatioConversionTable, stat = deallocstatus)
            if (allocated(IsoNfitConversionTable)) deallocate(IsoNfitConversionTable, stat = deallocstatus)
            if (allocated(GlobalIsoRatioParameter)) deallocate(GlobalIsoRatioParameter, stat = deallocstatus)
            if (allocated(NumberComponentsPerMolecule)) deallocate(NumberComponentsPerMolecule, stat = deallocstatus)
            if (allocated(myXCLASSParameter)) deallocate(myXCLASSParameter, stat = deallocstatus)
            if (allocated(CompMoleculeIndex)) deallocate(CompMoleculeIndex, stat = deallocstatus)
            if (allocated(MolNamePartFunc)) deallocate(MolNamePartFunc, stat = deallocstatus)
            if (allocated(lgQ)) deallocate(lgQ, stat = deallocstatus)
            if (allocated(NumEntriesRanges)) deallocate(NumEntriesRanges, stat = deallocstatus)
            if (allocated(MolecularData)) deallocate(MolecularData, stat = deallocstatus)
            if (allocated(MolecularDataIndices)) deallocate(MolecularDataIndices, stat = deallocstatus)
            if (allocated(LowerQNString)) deallocate(LowerQNString, stat = deallocstatus)
            if (allocated(UpperQNString)) deallocate(UpperQNString, stat = deallocstatus)
            if (allocated(ColumnNamesPartFunc)) deallocate(ColumnNamesPartFunc, stat = deallocstatus)
            if (allocated(TempPartFunc)) deallocate(TempPartFunc, stat = deallocstatus)
            if (allocated(CentralOpticalDepth)) deallocate(CentralOpticalDepth, stat = deallocstatus)
            if (allocated(DopplerShiftedTransFreq)) deallocate(DopplerShiftedTransFreq, stat = deallocstatus)
            if (allocated(DopplerShiftedTransFreqIndex)) deallocate(DopplerShiftedTransFreqIndex, stat = deallocstatus)
            if (allocated(DistanceOrderingArray)) deallocate(DistanceOrderingArray, stat = deallocstatus)
            if (allocated(SortedDistances)) deallocate(SortedDistances, stat = deallocstatus)
            if (allocated(PureDistances)) deallocate(PureDistances, stat = deallocstatus)
            if (allocated(vWidthLimits)) deallocate(vWidthLimits, stat = deallocstatus)
            if (allocated(vOffLimits)) deallocate(vOffLimits, stat = deallocstatus)
            if (allocated(GeometryFlag)) deallocate(GeometryFlag, stat = deallocstatus)
            if (allocated(tbFlag)) deallocate(tbFlag, stat = deallocstatus)
            if (allocated(tdFlag)) deallocate(tdFlag, stat = deallocstatus)
            if (allocated(nHFlag)) deallocate(nHFlag, stat = deallocstatus)
            if (allocated(LTEFlag)) deallocate(LTEFlag, stat = deallocstatus)
            if (allocated(ThinFlag)) deallocate(ThinFlag, stat = deallocstatus)
            if (allocated(Dust_Flag_Range)) deallocate(Dust_Flag_Range, stat = deallocstatus)
            if (allocated(Phen_Flag_Range)) deallocate(Phen_Flag_Range, stat = deallocstatus)


            !< deallocate memory of variables of Module 'Model'
            if (allocated(LayerMap)) deallocate(LayerMap, stat = deallocstatus)
            if (allocated(ConfigList)) deallocate(ConfigList, stat = deallocstatus)
            if (allocated(ConfigIndex)) deallocate(ConfigIndex, stat = deallocstatus)
            if (allocated(GausssianBeamMap)) deallocate(GausssianBeamMap, stat = deallocstatus)


            !< deallocate memory of variable for emission and absorption function
            if (EmAbsFlag) then
                if (allocated(TauEmAbsArray)) deallocate(TauEmAbsArray, stat = deallocstatus)
            endif


            !< deallocate memory of variables for myXCLASS function
            if (AllOutputFilesFlag) then
                if (allocated(IntPerCompArray)) deallocate(IntPerCompArray, stat = deallocstatus)
                if (allocated(TauPerCompArray)) deallocate(TauPerCompArray, stat = deallocstatus)
                if (allocated(TdPerCompArray)) deallocate(TdPerCompArray, stat = deallocstatus)
                if (allocated(TauHelperArrayEM)) deallocate(TauHelperArrayEM, stat = deallocstatus)
                if (allocated(TauHelperArrayABS)) deallocate(TauHelperArrayABS, stat = deallocstatus)
                if (allocated(TauHelperArray2)) deallocate(TauHelperArray2, stat = deallocstatus)
                if (allocated(LocalIntArray)) deallocate(LocalIntArray, stat = deallocstatus)
                if (allocated(TauCompFileName)) deallocate(TauCompFileName, stat = deallocstatus)
                if (allocated(IntCompFileName)) deallocate(IntCompFileName, stat = deallocstatus)
                if (allocated(IntegHelperArray1)) deallocate(IntegHelperArray1, stat = deallocstatus)
                if (allocated(IntegHelperArray2)) deallocate(IntegHelperArray2, stat = deallocstatus)
                if (allocated(PrevIntPerPeak)) deallocate(PrevIntPerPeak, stat = deallocstatus)
                if (allocated(IntensityPerPeak)) deallocate(IntensityPerPeak, stat = deallocstatus)
                if (SubBeamDescriptionFlag) then
                    if (allocated(IntPerCompArraySubBeam)) deallocate(IntPerCompArraySubBeam, stat = deallocstatus)
                    if (allocated(TauPerCompArraySubBeam)) deallocate(TauPerCompArraySubBeam, stat = deallocstatus)
                endif
            endif


            !< we're done
            return
        end subroutine myXCLASSParamFree
end Module myXCLASSCore
!*********************************************************************************************************************************************************

