#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  XCLASS-Viewer to compare up to four different synthetic spectra (compatible with XCLASS version 1.2.6 and higher)
##
##
##  Usage:
##
##      python3 XCLASS-Viewer.py
##
##
##
##
##  Copyright (C) 2016 - 2024  Peter Schilke, Thomas Moeller
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##
##  Class Main:
##  -----------
##
##      - function Main.__init__:                       initialize class Main
##      - function Main.addmpl:                         add widget to GUI
##      - function Main.data_browse:                    manage data browser
##      - function Main.molfit_clear:                   clear molfit file
##      - function Main.molfit_browse:                  manage molfit browser
##      - function Main.get_iso:                        get iso ratio file
##      - function Main.set_center:                     set center
##      - function Main.set_width:                      set width
##      - function Main.get_cont:                       get continuum level
##      - function Main.get_vlsr:                       get v_LSR
##      - function Main.get_size:                       get source size
##      - function Main.get_fmin:                       get min. frequency
##      - function Main.get_fmax:                       get max. frequency
##      - function Main.get_fstep:                      get step size
##      - function Main.rmmpl:                          remove widget from GUI
##      - function Main.exec_fit_1:                     start 1st fit process
##      - function Main.exec_fit_2:                     start 2nd fit process
##      - function Main.exec_fit_3:                     start 3rd fit process
##      - function Main.exec_fit_4:                     start 4th fit process
##      - function Main.exec_fit:                       prepare and start myXCLASSFit function
##      - function Main.first_plot:                     plot function
##      - function Main.re_plot:                        re-plot GUI
##      - function Main.clear_plot:                     clear plot
##
##
##  Main Program:
##  -------------
##
##      - function StartXCLASSViewer:                   start XCLASS-Viewer
##      - function Main:                                start GUI
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  P. Schilke      2016-09-20      initial version
##  T. Moeller      2018-10-31      improved version
##  T. Moeller      2020-01-04      porting to python 3
##  T. Moeller      2020-02-14      porting to PyQt5
##  G. Fuller       2023-09-18      add tickbox to say whether to remove the run directory
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import sys                                                                                  ## import sys package
import os                                                                                   ## import os package
import warnings                                                                             ## import warnings package
from PyQt5 import (QtWidgets, uic)                                                          ## get some PyQt5 functions
# from matplotlib.backends.qt_compat import QtWidgets, uic                                    ## get some PyQt5 functions
from matplotlib.figure import Figure                                                        ## import module Figure from matplotlib package
from matplotlib.backends.backend_qt5agg import (FigureCanvasQTAgg as FigureCanvas, NavigationToolbar2QT as NavigationToolbar)
from xclass import task_LoadASCIIFile
from xclass import task_myXCLASS
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## Main Class
class Main(QtWidgets.QMainWindow):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize main class
    def __init__(self, UiFileName):
        """

    input parameters:
    -----------------

        - UiFileName:       path and name of ui-layout file


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("UiFileName = ", UiFileName)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## initialize GUI parameters
        super(Main, self).__init__()
        uic.loadUi(UiFileName, self)
        self.setWindowTitle("XCLASS Viewer")


        ## define events for menue
        self.dfreq = 0.0
        self.dataBrowse.clicked.connect(self.data_browse)                                   ## when the button is pressed execute data_browse function
        self.molfitBrowse.clicked.connect(self.molfit_browse)                               ## when the button is pressed execute molfit_browse function
        self.molfitClear.clicked.connect(self.molfit_clear)                                 ## when the button is pressed execute molfit_browse function
        self.execFit_1.clicked.connect(self.exec_fit_1)                                     ## When the button is pressed execute exec_fit function
        self.execFit_2.clicked.connect(self.exec_fit_2)                                     ## when the button is pressed execute exec_fit function
        self.execFit_3.clicked.connect(self.exec_fit_3)                                     ## when the button is pressed execute exec_fit function
        self.execFit_4.clicked.connect(self.exec_fit_4)                                     ## when the button is pressed execute exec_fit function
        self.isoSelect.clicked.connect(self.get_iso)
        self.SizeValue.editingFinished.connect(self.get_size)
        self.ContLevel.editingFinished.connect(self.get_cont)
        self.inputvLSR.editingFinished.connect(self.get_vlsr)
        self.FreqMin.editingFinished.connect(self.get_fmin)
        self.FreqMax.editingFinished.connect(self.get_fmax)
        self.FreqStep.editingFinished.connect(self.get_fstep)
        self.plotClear.clicked.connect(self.clear_plot)                                     ## when the button is pressed execute molfit_browse function


        ## set defaults
        self.contLevel = 0
        self.dfmin = 100000
        self.dfmax = 110000
        self.dfstep = 1
        self.sizeValue = 0.5
        self.isoFileName = "none"
        self.vLSR = 0.0
        self.ContLevel.setText(str(self.contLevel))
        self.inputvLSR.setText(str(self.vLSR))
        self.FreqMin.setText(str(self.dfmin))
        self.FreqMax.setText(str(self.dfmax))
        self.FreqStep.setText(str(self.dfstep))
        self.SizeValue.setText(str(self.sizeValue))
        self.isoValue.setText(self.isoFileName)
        self.interButton.isDown()
        self.cleanButton.isDown()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## add widget to GUI
    def addmpl(self, fig):
        """

    input parameters:
    -----------------

        - fig:          figure object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("fig = ", fig)


        ## add widget to GUI
        self.canvas = FigureCanvas(fig)
        self.mplvl.addWidget(self.canvas)
        self.canvas.draw()
        self.toolbar = NavigationToolbar(self.canvas, self.mplwindow, coordinates = True)
        self.mplvl.addWidget(self.toolbar)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## manage data browser
    def data_browse(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## execute getExistingDirectory dialog and set the directory variable to be equal to the user selected directory
        self.DataList.clear()                                                               ## in case there are any existing elements in the list
        self.DataFile = QtWidgets.QFileDialog.getOpenFileNames(self, "Data File")[0]
        if self.DataFile:                                                                   ## if user didn't pick a file don't continue
            DataFileName = str(self.DataFile[0])
            self.DataList.addItem(DataFileName)
            NumHeaderLines = 2
            self.expdata = task_LoadASCIIFile.LoadASCIIFile(DataFileName, NumHeaderLines, 0, 0)
        else:
           print ("No data file selected!")

        self.dfreq = self.expdata[:,0]
        self.data = self.expdata[:,1]
        self.dfmin = self.dfreq.min()
        self.dfmax = self.dfreq.max()
        self.dfwid = self.dfmax-self.dfmin
        self.dfmid = (self.dfmin+self.dfmax)/2
        self.dfstep = abs(self.dfreq[1]-self.dfreq[0])
        self.dmean = self.data.mean()
        self.first_plot()
        self.contLevel = self.dmean
        self.ContLevel.setText(str(self.dmean))
        self.FreqMin.setText(str(int(self.dfmin)))
        self.FreqMax.setText(str(int(self.dfmax)))
        self.FreqStep.setText(str(self.dfstep))


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## clear plot
    def clear_plot(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## in case there are any existing elements in the list
        if hasattr(self,'fig1'):
            if(hasattr(self,'ffreq_1')):
                delattr(self,'ffreq_1')
            if(hasattr(self,'ffreq_2')):
                delattr(self,'ffreq_2')
            if(hasattr(self,'ffreq_3')):
                delattr(self,'ffreq_3')
            if(hasattr(self,'ffreq_4')):
                delattr(self,'ffreq_4')

            self.first_plot()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## clear molfit file
    def molfit_clear(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## in case there are any existing elements in the list
        self.MolfitList.clear()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## manage molfit browser
    def molfit_browse(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        # execute getExistingDirectory dialog and set the directory variable to be equal to the user selected directory
        self.MolfitFile = QtWidgets.QFileDialog.getOpenFileNames(self, "Molfit File")[0]
        if self.MolfitFile: # if user didn't pick a file don't continue
            self.MolFileName = str(self.MolfitFile[0])
            self.MolfitList.addItem(self.MolFileName)
        else:
            self.Status.addItem("No Molfit file selected!")
            self.Status.scrollToBottom()
            print ("No Molfit file selected!")


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get iso ratio file
    def get_iso(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        # execute getExistingDirectory dialog and set the directory variable to be equal to the user selected directory
        self.isoFile = QtWidgets.QFileDialog.getOpenFileNames(self, "Iso File")[0]
        if self.isoFile: # if user didn't pick a file don't continue
            self.isoFileName = str(self.isoFile[0])
            self.isoValue.setText(self.isoFileName)
        else:
            self.Status.addItem("No Molfit file selected!")
            self.Status.scrollToBottom()
            print ("No Molfit file selected!")


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set center
    def set_center(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## add new item to center list
        self.CenterList.clear()
        self.CenterList.addItem(str(self.CenterSlider.value()))


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set width
    def set_width(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## add new width to widthlist
        self.WidthList.clear()
        self.WidthList.addItem(str(self.WidthSlider.value()))


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get continuum level
    def get_cont(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## get continuum level
        self.contLevel = float(self.ContLevel.text())


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get v_LSR
    def get_vlsr(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## get v_LSR
        self.vLSR = float(self.inputvLSR.text())


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get source size
    def get_size(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## get min. frequency
        self.sizeValue = float(self.SizeValue.text())

        # Debug:
        # print (self.contLevel)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get min. frequency
    def get_fmin(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## get min. frequency
        self.dfmin = float(self.FreqMin.text())

        # Debug:
        # print (self.dfmin)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get max. frequency
    def get_fmax(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## get max. frequency
        self.dfmax = float(self.FreqMax.text())

        # Debug:
        # print (self.dfmax)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get step size
    def get_fstep(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## get step size
        self.dfstep = float(self.FreqStep.text())

        # Debug:
        # print (self.dfstep)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## remove widget from widget
    def rmmpl(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## remove widget from GUI
        self.mplvl.removeWidget(self.canvas)
        self.canvas.close()
        self.mplvl.removeWidget(self.toolbar)
        self.toolbar.close()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start 1st fit process
    def exec_fit_1(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## start 1st fit process
        self.exec_fit(1)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start 2nd fit process
    def exec_fit_2(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## start 2nd fit process
        self.exec_fit(2)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start 3rd fit process
    def exec_fit_3(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## start 3rd fit process
        self.exec_fit(3)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start 4th fit process
    def exec_fit_4(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## start 4th fit process
        self.exec_fit(4)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## prepare and start myXCLASSFit function
    def exec_fit(self, n, cleanupFlag = True):
        """

    input parameters:
    -----------------

        - n:                    index of synthetic spectrum

        - cleanupFlag:          (optional) flag indicating clean up of myXCLASS run


    output parameters:
    ------------------

        - None
        """


        # Debug:
        # print ("n = ", n)
        # print ("cleanupFlag = ", cleanupFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write some informations to status widget


        ## check, if iso ratio file is selected and write informations to status widget
        iso_flag = self.isoButton.isChecked()
        if (iso_flag):
            self.Status.addItem("Isotopologues considered from file " + self.isoFileName)
        else:
            self.Status.addItem("No Isotopologues considered")


        ## get telescope parameters and write informations to status widget
        TelescopeSize = self.sizeValue
        InterFlag = self.interButton.isChecked()
        cleanupFlag = self.cleanButton.isChecked()
        if (InterFlag):
            self.Status.addItem("Interferometer mode assumed, resolution " + str(self.sizeValue))
        else:
            self.Status.addItem("Single dish mode assumed, size " +str(self.sizeValue))


        ## write status to status widget
        self.Status.scrollToBottom()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name of current molfit file
        if (n == 1 and self.MolfitList.count() > 0):
            MolFileName = str(self.MolfitList.item(0).text())
        elif (n == 2 and self.MolfitList.count() > 1):
            MolFileName = str(self.MolfitList.item(1).text())
        elif (n == 3 and self.MolfitList.count() > 2):
            MolFileName = str(self.MolfitList.item(2).text())
        elif (n == 4 and self.MolfitList.count() > 3):
            MolFileName = str(self.MolfitList.item(3).text())
        else:
            self.Status.addItem("No Molfit file defined!")
            print ("no Molfit Files defined!")

        # Debug:
        # print (self.MolfitList.count(), n)
        # print (self.MolfitList.item(0).text())
        # print (self.MolfitList.item(1).text())
        # if (n == 1):
        #     print ("n = 1")
        # if (self.MolfitList.count() > 0):
        #     print ("Listcount > 0")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## call myXCLASS function


        ## define some initial myXCLASS parameters
        t_back_flag = True
        tBack = self.contLevel
        tSlope = 0.0
        nH_flag = True
        N_H = 3.0E+22
        beta_dust = 2.0
        kappa_1300 = 0.0
        IsoTableFileName = self.isoFileName
        vLSR = self.vLSR
        RestFreq = (self.dfmax - self.dfmin) / 2.0
        printFlag = True
        LocalOverlapFlag = False
        NoSubBeamFlag = True

        # Debug:
        # print ("\n\nself.dfmin = ", self.dfmin)
        # print ("self.dfmax = ", self.dfmax)
        # print ("self.dfstep = ", self.dfstep)
        # print ("TelescopeSize = ", TelescopeSize)
        # print ("InterFlag = ", InterFlag)
        # print ("t_back_flag = ", t_back_flag)
        # print ("tBack = ", tBack)
        # print ("tSlope = ", tSlope)
        # print ("nH_flag = ", nH_flag)
        # print ("N_H = ", N_H)
        # print ("beta_dust = ", beta_dust)
        # print ("kappa_1300 = ", kappa_1300)
        # print ("MolFileName = ", MolFileName)
        # print ("iso_flag = ", iso_flag)
        # print ("IsoTableFileName = ", IsoTableFileName)
        # print ("RestFreq = ", RestFreq)
        # print ("vLSR = ", vLSR)
        # print ("printFlag = ", printFlag)
        # print ("LocalOverlapFlag = ", LocalOverlapFlag)
        # print ("NoSubBeamFlag = ", NoSubBeamFlag)


        ## call myXCLASS function
        try:
            MolFileName.isalpha()
            modeldata, log, trans, inttau, jobdir = task_myXCLASS.myXCLASSCore(FreqMin = self.dfmin, FreqMax = self.dfmax, FreqStep = self.dfstep, \
                                                                               MolfitsFileName = MolFileName, iso_flag = iso_flag, \
                                                                               IsoTableFileName = IsoTableFileName, TelescopeSize = TelescopeSize, \
                                                                               Inter_Flag = InterFlag, t_back_flag = t_back_flag, \
                                                                               tBack = tBack, tSlope = tSlope, nH_flag = nH_flag, \
                                                                               N_H = N_H, beta_dust = beta_dust, kappa_1300 = kappa_1300, \
                                                                               LocalOverlapFlag = LocalOverlapFlag, RestFreq = RestFreq, \
                                                                               vLSR = vLSR, NoSubBeamFlag = NoSubBeamFlag, \
                                                                               printFlag = printFlag)
            ## clean up?
            if (cleanupFlag):
                cmdString = "rm -rf " + jobdir
                os.system(cmdString)


            ## add synthetic spectrum
            if (n == 1 and self.MolfitList.count() > 0):
                self.ffreq_1 = modeldata[:,0]
                self.fit_1 = modeldata[:,2]

                # Debug:
                # print ("found:",  self.ffreq_1, self.fit_1)

            elif (n == 2 and self.MolfitList.count() > 1):
                self.ffreq_2 = modeldata[:,0]
                self.fit_2 = modeldata[:,2]
            elif (n == 3 and self.MolfitList.count() > 2):
                self.ffreq_3 = modeldata[:,0]
                self.fit_3 = modeldata[:,2]
            elif (n == 4 and self.MolfitList.count() > 3):
                self.ffreq_4 = modeldata[:,0]
                self.fit_4 = modeldata[:,2]
            else:
                pass
        except:
            self.Status.addItem("No Molfit file found!")
            print ("no MolFit File found!")


        ## update GUI
        self.re_plot()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## plot function
    def first_plot(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        try:
            self.rmmpl()
        except:
            pass
        self.fig1 = Figure()
        self.ax1f1 = self.fig1.add_subplot(111)
        self.ax1f1.plot(self.dfreq, self.data, drawstyle = 'steps-mid', label = "Data", lw = 2)
        self.ax1f1.legend()
        self.ax1f1.set_xlabel("Frequency [MHz]")
        self.ax1f1.set_ylabel("Intensity [K]")
        self.ax1f1.set_xlim(self.dfmin, self.dfmax)
        self.ax1f1.grid()
        self.addmpl(self.fig1)
        self.show()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## re-plot GUI
    def re_plot(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        try:
            self.rmmpl()
        except:
            pass
        self.fig1 = Figure()
        self.ax1f1 = self.fig1.add_subplot(111)
        try:
            self.dfreq.real
            self.ax1f1.plot(self.dfreq, self.data, drawstyle = 'steps-mid', label = "Data", lw = 2)
        except:
            pass

        # Debug:
        # print ("replot")
        # print (self.ffreq_1)
        # print (self.fit_1)

        try:
            self.ffreq_1.real
            self.ax1f1.plot(self.ffreq_1, self.fit_1, drawstyle = 'steps-mid', label = "Model 1", lw = 2,color='green')
        except:
            pass
        try:
            self.ffreq_2.real
            self.ax1f1.plot(self.ffreq_2, self.fit_2, drawstyle = 'steps-mid', label = "Model 2", lw = 2,color='orange')
        except:
            pass
        try:
            self.ffreq_3.real
            self.ax1f1.plot(self.ffreq_3, self.fit_3, drawstyle = 'steps-mid', label = "Model 3", lw = 2,color='red')
        except:
            pass
        try:
            self.ffreq_4.real
            self.ax1f1.plot(self.ffreq_4, self.fit_4, drawstyle = 'steps-mid', label = "Model 4", lw = 2,color='black')
        except:
            pass
        self.ax1f1.legend()
        self.ax1f1.set_xlabel("Frequency [MHz]")
        self.ax1f1.set_ylabel("Intensity [K]")
        self.ax1f1.grid()
        self.addmpl(self.fig1)
        self.show()


        ## return to GUI
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start XCLASS-Viewer
##
def StartXCLASSViewer():
    """

input parameters:
-----------------

    - None


output parameters:
------------------

    - None
    """


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## loadUiType function requires a single argument, the name of a Designer UI file, and returns a tuple of the appropriate application classes
    ## the return tuple contains two class definitions. The first is our custom GUI application class, set to the Ui_MainWindow variable. The second
    ## is the base class for our custom GUI application, which in this case is PyQt5.QtGui.QMainWindow. The base class type is defined by the widget
    ## we first created with Designer. Note: These are not instances of these classes, but the class definitions themselves. They are meant to serve
    ## as superclasses to our new application logic class, which we’ll create next.
    XCLASSRootDir = task_myXCLASS.GetXCLASSRootDir()
    XCLASSViewerDir = XCLASSRootDir + "tools/XCLASSViewer/"
    UiFileName = XCLASSViewerDir + 'layout/XCLASSViewer.ui'


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start GUI
    app = QtWidgets.QApplication(sys.argv)
    main = Main(UiFileName)
    main.show()
    sys.exit(app.exec_())


    ## return to main program
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start main program
##
if __name__ == '__main__':


    ## start XCLASS-Viewer
    StartXCLASSViewer()


## finished!
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
