#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module creates a plot showing the (observational) data, the calculated model function, and the transition energies from the database
##  Copyright (C) 2012 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function PlotClass.__init__:                  initialize plotting class
##      - function PlotClass.PlotObsDataFiles:          plot obs. data file
##      - function PlotClass.PlotFitResults:            plot results of myXCLASSFit function
##      - function PlotClass.PlotMoleculeContributions: create plots for each obs. data file including names of molecules
##      - function PlotClass.PlotmyXCLASS:              plot results of myXCLASS function
##      - function InterpretQN:                         interpret quantum number string from database
##      - function ConvertFloatLatexStyle:              convert float number to latex style
##      - function myXCLASSPlotCore:                    core routine of the myXCLASSPlot function
##      - function myXCLASSPlot:                        creates a plot showing the (observational) data, the calculated model function, and the
##                                                      transition energies from the database
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2013-07-25      initial version
##  T. Moeller      2018-10-25      add PlotClass
##  T. Moeller      2020-01-03      porting to python 3.x
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import subprocess                                                                           ## import subprocess package
if (not 'matplotlib' in sys.modules):
    import matplotlib                                                                       ## import matplotlib package
    matplotlib.use("Agg")                                                                   ## avoid display error
else:
    import matplotlib                                                                       ## import matplotlib package
from . import task_MAGIX                                                                    ## import package task_MAGIX
from . import task_myXCLASS                                                                 ## import package task_myXCLASS
from . import task_myXCLASSFit                                                              ## import package task_myXCLASSFit
from . import task_myXCLASSMapFit                                                           ## import package task_myXCLASSMapFit
from . import task_LineIdentification                                                       ## import package task_LineIdentification
from . import task_ListDatabase                                                             ## import package task_ListDatabase
import pylab                                                                                ## import pylab package
import copy                                                                                 ## import copy package
import matplotlib.pyplot as plt                                                             ## import pyplot
from matplotlib.font_manager import fontManager                                             ## import fontManager package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## Class for plotting
class PlotClass():


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize plotting class
    ##
    ## usage:
    ##
    ##     LocalPlotClass = PlotClass(MolfitsFileName, ObsXMLFileName, PlotNum, RemoveContFlag, AddMoleculeNamesToPlotFlag,
    ##                                DefaultDB, AdditionalFiles, JobDir, PlotOption, IsoRatioFileName, BasePath)
    ##     LocalPlotClass.PlotmyXCLASS()                    ## plot results of myXCLASS function
    ##     LocalPlotClass.PlotFitResults()                  ## plot results of myXCLASSFit function
    ##     LocalPlotClass.PlotMoleculeContributions()       ## plot contributions of each molecule
    ##
    def __init__(self, MolfitsFileName, ObsXMLFileName, PlotNum, RemoveContFlag, AddMoleculeNamesToPlotFlag, DefaultDB, AdditionalFiles, \
                 JobDir, PlotOption, NewIsoRatioFileName, BaseResultPath, SubPlotHeight = 17, SubPlotWidth = 10, PDFExportFlag = False, \
                 velLowLimit = -250.0, velUpLimit = 100.0, CleanmyXCLASSFlag = True, vlsr = 0.0, NumberRows = 4, NumberColumns = 4, \
                 TransLabel = "compact", MolecularLabelFontSize = 6, MinNtot = 1.e12, GridFlag = True):
        """

    input parameters:
    -----------------

        - MolfitsFileName:              path and name of molfit file

        - ObsXMLFileName:               path and name of obs. xml-file

        - PlotNum:                      number of plots per obs. file

        - RemoveContFlag:               flag for removing continuum

        - AddMoleculeNamesToPlotFlag:   flag for adding molecule names to plot

        - DefaultDB:                    path and name of default database file

        - AdditionalFiles:              additional files

        - JobDir:                       current job directory

        - PlotOption:                   plot option

        - NewIsoRatioFileName:          path and name of iso ratio file

        - BaseResultPath:               base directory

        - SubPlotHeight:                (optional) height of each subplot (default: 17)

        - SubPlotWidth:                 (optional) width of each subplot (default: 10)

        - PDFExportFlag:                (optional) flag for png and pdf export (default: False, i.e. png only)

        - velLowLimit:                  (optional) lower limit of velocity range (default: -250.0)

        - velUpLimit:                   (optional) upper limit of velocity range (default: 100.0)

        - vlsr:                         (optional) vlsr (default: 0.0)

        - CleanmyXCLASSFlag:            (optional) remove files describing intensities and optical depths
                                        from the myXCLASS job directory (default: True)

        - NumberRows:                   (optional) number of rows for publication plot (default: 4)

        - NumberColumns:                (optional) number of columns for publication plot (default: 4)

        - TransLabel:                   (optional) label of transitions "compact", "rrl", "rrl-red", "full", "detail" (default: "compact")

        - MolecularLabelFontSize:       (optional) font size of molecular transition label (default: 6)

        - MinNtot:                      (optional) min. column density of molecular component (default: 1.e12)

        - GridFlag:                     (optional) flag for plotting grid in transition plots, (default: True)


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("MolfitsFileName = ", MolfitsFileName)
        # print ("ObsXMLFileName = ", ObsXMLFileName)
        # print ("PlotNum = ", PlotNum)
        # print ("RemoveContFlag = ", RemoveContFlag)
        # print ("AddMoleculeNamesToPlotFlag = ", AddMoleculeNamesToPlotFlag)
        # print ("DefaultDB = ", DefaultDB)
        # print ("AdditionalFiles = ", AdditionalFiles)
        # print ("JobDir = ", JobDir)
        # print ("PlotOption = ", PlotOption)
        # print ("NewIsoRatioFileName = ", NewIsoRatioFileName)
        # print ("BaseResultPath = ", BaseResultPath)
        # print ("SubPlotHeight = ", SubPlotHeight)
        # print ("SubPlotWidth = ", SubPlotWidth)
        # print ("PDFExportFlag = ", PDFExportFlag)
        # print ("PDFExportFlag = ", PDFExportFlag)
        # print ("velLowLimit = ", velLowLimit)
        # print ("velUpLimit = ", velUpLimit)
        # print ("vlsr = ", vlsr)
        # print ("CleanmyXCLASSFlag = ", CleanmyXCLASSFlag)
        # print ("NumberRows = ", NumberRows)
        # print ("NumberColumns = ", NumberColumns)
        # print ("TransLabel = ", TransLabel)
        # print ("MolecularLabelFontSize = ", MolecularLabelFontSize)
        # print ("MinNtot = ", MinNtot)
        # print ("GridFlag = ", GridFlag)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## store some input parameters
        self.PlotNum = PlotNum
        self.RemoveContFlag = RemoveContFlag
        self.AddMoleculeNamesToPlotFlag = AddMoleculeNamesToPlotFlag
        self.JobDir = JobDir
        self.SubPlotHeight = SubPlotHeight
        self.SubPlotWidth = SubPlotWidth
        self.PDFExportFlag = PDFExportFlag
        self.velLowLimit = velLowLimit + vlsr
        self.velUpLimit = velUpLimit + vlsr
        self.vlsr = vlsr
        self.CleanmyXCLASSFlag = CleanmyXCLASSFlag
        self.NumberRows = NumberRows
        self.NumberColumns = NumberColumns
        self.TransLabel = TransLabel
        self.MolecularLabelFontSize = MolecularLabelFontSize
        self.MinNtot = MinNtot
        self.GridFlag = GridFlag


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize some parameters
        self.DPIResolution = 100                                                            ## define resolution of figures in dpi
        self.MinNumTransitionsSQL = 1                                                       ## initialize min. number of transitions
        self.MaxNumTransitionsSQL = 1e99                                                    ## initialize max. number of transitions
        self.MaxElowSQL = 1.e6                                                              ## initialize max. value for E_low
        self.MingASQL = 0.0                                                                 ## initialize lower limit for gA
        self.OrderTransSQL = 3                                                              ## initialize order of transitions
        self.BaseResultPath = BaseResultPath                                                ## path and name of base path
        if (not self.BaseResultPath.endswith("/")):
            self.BaseResultPath = self.BaseResultPath + "/"
        self.BaseResultPathOrig = self.BaseResultPath                                       ## store original path
        self.myXCLASSFitFileList = []                                                       ## initialize list of myXCLASSFit files
        self.UsemyXCLASSFlag = False                                                        ## call myXCLASS function
        self.LocalPlotOption = PlotOption                                                   ## local plot option
        self.LocalMoleculeName = ""                                                         ## local molecule name
        self.LocalMolfitFileName = MolfitsFileName                                          ## store path and name of molfit file
        self.NewIsoRatioFileName = NewIsoRatioFileName                                      ## store path and name of iso ratio file
        self.PubPlotFlag = False                                                            ## flag for publication plots
        self.AllMoleculesFlag = False                                                       ## flag for full spectra calculation
        self.FullSpectra = []                                                               ## list for all full spectra
        self.SelectedTransitions = []                                                       ## clear list of selected transitions ("[]" = include all)
        self.OverallPATH = ""                                                               ## clear path of overall fit
        self.pdfgsFlag = True                                                               ## use external program "gs"
        self.pdfUniteFlag = True                                                            ## use external program "pdfunite"
        self.AllowCropPDF = True                                                            ## use external program "pdfcrop"
        self.AdditionalSpectra = None                                                       ## additional spectra added to trans. plots
        self.AdditionalSpectraDict = None                                                   ## dictionary of plotting options for additional spectra


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define current working directory
        self.LocalPath = os.getcwd() + "/"


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze given molfit file


        ## get sql parameters in molfit file
        MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(self.LocalMolfitFileName)
        self.SQLParamArray = [MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL]
        self.MolfitHeaderLinesList = task_myXCLASS.WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)
        self.MinNumTransitionsSQL = MinNumTransitionsSQL
        self.MaxNumTransitionsSQL = MaxNumTransitionsSQL
        self.MaxElowSQL = MaxElowSQL
        self.MingASQL = MingASQL
        self.OrderTransSQL = OrderTransSQL

        # Debug:
        # print ("\n\nself.LocalMolfitFileName = ", self.LocalMolfitFileName)
        # print ("self.MinNumTransitionsSQL = ", self.MinNumTransitionsSQL)
        # print ("self.MaxNumTransitionsSQL = ", self.MaxNumTransitionsSQL)
        # print ("self.MaxElowSQL = ", self.MaxElowSQL)
        # print ("self.MingASQL = ", self.MingASQL)
        # print ("self.OrderTransSQL = ", self.OrderTransSQL)


        ## analyze given molfit file
        if (self.LocalMolfitFileName != ""):
            self.MoleculesInMolfitFile, self.AllParameters, self.MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(self.LocalMolfitFileName)
        else:
            self.MoleculesInMolfitFile = []
            self.AllParameters = []
            self.MolfitFileForEachMolecule = []

        # Debug:
        # print ("\n\nself.MoleculesInMolfitFile = ", self.MoleculesInMolfitFile)
        # print ("self.AllParameters = ", self.AllParameters)
        # print ("self.MolfitFileForEachMolecule = ", self.MolfitFileForEachMolecule)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read in parameters for each freq. range for the current molecule
        self.ListOfSpectraNames = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "FileNamesExpFiles")
        self.NumberExpFiles = len(self.ListOfSpectraNames)
        self.NumberExpRangesList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "NumberExpRanges")
        self.FreqMinList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "MinExpRange")
        self.FreqMaxList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "MaxExpRange")
        self.FreqStepList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "StepFrequency")
        self.t_back_flagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "t_back_flag")
        self.tBackList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "BackgroundTemperature")
        self.tSlopeList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "TemperatureSlope")
        self.N_HList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "HydrogenColumnDensity")
        self.beta_dustList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "DustBeta")
        self.kappa_1300List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "Kappa")
        self.DustFileNameList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "DustFileName")
        self.BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "BackgroundFileName")
        self.ContPhenFuncIDList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncID")
        self.ContPhenFuncParam1List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncParam1")
        self.ContPhenFuncParam2List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncParam2")
        self.ContPhenFuncParam3List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncParam3")
        self.ContPhenFuncParam4List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncParam4")
        self.ContPhenFuncParam5List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncParam5")
        self.NoiseList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "NoiseLevel")
        self.SmoothList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "SmoothValue")
        self.TelescopeSizeList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "TelescopeSize")
        self.BMINList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "BMIN")
        self.BMAJList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "BMAJ")
        self.BPAList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "BPA")
        self.Inter_FlagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "Inter_Flag")
        self.GlobalvLSRList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "GlobalvLSR")
        self.RedshiftList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "Redshift")
        self.ErrorYList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ErrorY")
        self.NumberHeaderLinesList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "NumberHeaderLines")
        self.SeparatorColumnsList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "SeparatorColumns")
        IsotopologuesList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "iso_flag")
        if (IsotopologuesList == []):
            IsotopologuesList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "Isotopologues")
        self.LocalIsoTableFileName = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "IsoTableFileName")
        dbList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "dbFilename")
        if (dbList == []):
            dbFile = task_myXCLASS.GetDefaultDBFile()
            dbList = [dbFile]
        self.dbFile = dbList[0]
        NumModelPixelXXList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "NumModelPixelXX")
        if (NumModelPixelXXList == []):
            self.NumModelPixelXX = 100
        else:
            self.NumModelPixelXX = NumModelPixelXXList[0]
        NumModelPixelYYList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "NumModelPixelYY")
        if (NumModelPixelYYList == []):
            self.NumModelPixelYY = 100
        else:
            self.NumModelPixelYY = NumModelPixelYYList[0]
        LocalOverlapFlagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "LocalOverlap_Flag")
        if (LocalOverlapFlagList == []):
            self.LocalOverlapFlag = False
        else:
            self.LocalOverlapFlag = task_myXCLASSMapFit.CheckBool(LocalOverlapFlagList[0])
        NoSubBeamFlagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "NoSubBeam_Flag")
        if (NoSubBeamFlagList == []):
            self.NoSubBeamFlag = False
        else:
            self.NoSubBeamFlag = task_myXCLASSMapFit.CheckBool(NoSubBeamFlagList[0])
        EmAbsPATHList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "EmAbsPATH")
        if (EmAbsPATHList == []):
            self.EmAbsPATH = None
        else:
            self.EmAbsPATH = EmAbsPATHList[0]


        ## get iso flag
        if (len(IsotopologuesList) > 0):
            self.IsoFlag = task_myXCLASSMapFit.CheckBool(IsotopologuesList[0])
        else:
            self.IsoFlag = False
        if (self.IsoFlag):                                                                  ## set local iso ratio flag
            iso_flag = True
            self.LocalIsoTableFileName = self.NewIsoRatioFileName
        else:
            iso_flag = False
            self.LocalIsoTableFileName = ""


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze iso ratio file
        if (self.IsoFlag):
            dummyName = ""
            if (self.LocalPlotOption == "fit"):
                SplittedName = self.LocalIsoTableFileName.split("/")
                self.LocalIsoTableFileName = self.JobDir + SplittedName[-1].strip()
            self.IsoRatioTable, self.Isotopologues, self.IsoMolecule = task_myXCLASS.ImportIsoRatioFile(self.LocalIsoTableFileName, dummyName)
            if (len(self.IsoRatioTable) == 0):
                self.IsoFlag = False
                self.LocalIsoTableFileName = ""
            else:


                ## analyze, if global iso ratios are defined
                GlobalIsoRatioList = []
                IsotopologuesIndex = (-1)
                for IsoRatioTableLine in self.IsoRatioTable:
                    IsotopologuesIndex += 1
                    LocalIsotopologue = IsoRatioTableLine[0]
                    LocalIsoRatio = IsoRatioTableLine[2]
                    if (LocalIsotopologue.find("GLOBAL__") > (-1)):
                        LocalIsotopologueNew = LocalIsotopologue.replace("GLOBAL__", "")
                        GlobalIsoRatioList.append([LocalIsotopologueNew, IsotopologuesIndex, LocalIsoRatio])

                # Debug:
                # print ("GlobalIsoRatioList = ", GlobalIsoRatioList)


                ## analyze each isotopologue
                NumberOfGlobalISORatios = len(GlobalIsoRatioList)
                if (len(GlobalIsoRatioList) > 0):
                    for IsoID, LocalIsotopologue in enumerate(self.Isotopologues):
                        GlobalIsoRatioParameter = []
                        for GlobalIsoRatio in GlobalIsoRatioList:
                            iii = LocalIsotopologue.count(GlobalIsoRatio[0])
                            if (iii > 0):
                                factor = float(GlobalIsoRatio[2])
                                self.IsoRatioTable[IsoID][2] = float(self.IsoRatioTable[IsoID][2]) * (factor**iii)

                # Debug:
                # print ("self.IsoRatioTable = ", self.IsoRatioTable)
                # sys.exit(0)

        else:
            self.IsoRatioTable = []
            self.Isotopologues = []
            self.IsoMolecule = []

        # Debug:
        # print ("self.IsoRatioTable = ", self.IsoRatioTable)
        # print ("self.Isotopologues = ", self.Isotopologues)
        # print ("self.IsoMolecule = ", self.IsoMolecule)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze additional files
        self.ULines = []
        self.GhostLines = []
        if (AdditionalFiles != []):
            if (AdditionalFiles[0].strip() != ""):
                self.ULines = numpy.loadtxt(AdditionalFiles[0])
            if (AdditionalFiles[1].strip() != ""):
                self.GhostLines = numpy.loadtxt(AdditionalFiles[1])


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## plot obs. data file
    def PlotObsDataFiles(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create plots for each obs. data file
        CounterTransWindow = 0
        TFWList = []                                                                        ## list of all TFW parameters
        SortTFWParamList = []                                                               ## list of TFW parameters used for sorting
        for ObsDataFileIndex in range(len(self.ListOfSpectraNames)):                        ## loop over all obs. data files
            LocalExpFileName = self.ListOfSpectraNames[ObsDataFileIndex][1]
            RestFreq = 0.0
            vLSR = 0.0


            ## get number of header lines and character separating columns from obs. xml file
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                           NumberHeaderLinesListIn = self.NumberHeaderLinesList, \
                                                                           SeparatorColumnsListIn = self.SeparatorColumnsList, \
                                                                           TelescopeSizeListIn = self.TelescopeSizeList, \
                                                                           BMIN_ListIn = self.BMINList, BMAJ_ListIn = self.BMAJList, \
                                                                           BPA_ListIn = self.BPAList, GlobalvLSRListIn = self.GlobalvLSRList, \
                                                                           Redshift_ListIn = self.RedshiftList, \
                                                                           InterFlagListIn = self.Inter_FlagList, \
                                                                           NumberRangeListIn = self.NumberExpRangesList)
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
            NumberHeaderLines = ObsXMLParameterDictFile['NumberHeaderLines']
            if (NumberHeaderLines is None):
                NumberHeaderLines = 0
            SeparatorColumns = ObsXMLParameterDictFile['SeparatorColumns']
            if (SeparatorColumns is None):
                SeparatorColumns = ""
            TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
            BMIN = ObsXMLParameterDictFile['BMIN']
            BMAJ = ObsXMLParameterDictFile['BMAJ']
            BPA = ObsXMLParameterDictFile['BPA']
            InterFlag = ObsXMLParameterDictFile['InterFlag']
            GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
            if (GlobalvLSR is None):
                GlobalvLSR = 0.0
            Redshift = ObsXMLParameterDictFile['Redshift']


            ## initialize tBack and dust flag
            t_back_flag = True
            nH_flag = True


            ## get path and name of corresponding data file
            SpectraNamePure = None
            SpectraFileName = self.ListOfSpectraNames[ObsDataFileIndex][1].strip()
            SpectraName = SpectraFileName
            i = SpectraName.rfind("/")
            if (i > (-1)):
                SpectraNamePure = SpectraName[i+1:]
                if (self.LocalPlotOption == "fit"):
                    SpectraFileName = self.JobDir + SpectraNamePure

            # Debug:
            # print ("SpectraFileName = ", SpectraFileName)
            # print ("SpectraNamePure = ", SpectraNamePure)
            # print ("\tCreate plots for obs. data file " + chr(34) + SpectraNamePure + chr(34) + " ..")


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## import obs. data
            ImportObsDataFlag = True
            try:
                if (SeparatorColumns.strip() == ""):
                    LocalObsSpectrum = numpy.loadtxt(SpectraFileName, skiprows = NumberHeaderLines)
                else:
                    LocalObsSpectrum = numpy.loadtxt(SpectraFileName, skiprows = NumberHeaderLines, delimiter = SeparatorColumns)
            except:
                ImportObsDataFlag = False
            if (ImportObsDataFlag):
                LowestDataPoint = min(LocalObsSpectrum[:,0])
                HighestDataPoint = max(LocalObsSpectrum[:,0])
                FreqArray = LocalObsSpectrum[:, 0]

                # Debug:
                # print ("LowestDataPoint = ", LowestDataPoint)
                # print ("HighestDataPoint = ", HighestDataPoint)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## create figure
                WidthSpectrum = (HighestDataPoint - LowestDataPoint) / float(self.PlotNum)
                fig, layers = pylab.subplots(self.PlotNum, 1, figsize = (self.SubPlotHeight, self.PlotNum * self.SubPlotWidth))
                if (self.PlotNum > 10):
                    pylab.subplots_adjust(bottom = 0.005, top = 0.995, right = 0.98, left = 0.07, hspace = 0.1, wspace = 0.2)
                else:
                    pylab.subplots_adjust(bottom = 0.05, top = 0.90, right = 0.97, left = 0.09, hspace = 0.1, wspace = 0.2)
                # fig.clear()


                ## switch backend
                if ((self.PlotNum * self.SubPlotWidth * self.DPIResolution) > 32768):
                    plotBackend = "svg"
                    pylab.switch_backend('{:s}'.format(plotBackend))
                else:
                    plotBackend = "png"


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## add observational data for each subplot
                ListIntObsMinMax = []
                for PlotID in range(self.PlotNum):


                    ## add title
                    if (PlotID == 0):
                        if (SpectraNamePure is None):
                            TitleStringPart = ""
                        else:
                            TitleStringPart = "Obs. dat file: " + SpectraNamePure + "\n"
                        if (self.LocalPlotOption == "fit"):
                            layers[PlotID].set_title(TitleStringPart + r"E$_{\rm low}$ $\leq$ " + str(self.MaxElowSQL) + " K", y = 1.08)

                        elif (self.LocalPlotOption == "myxclass"):
                            layers[PlotID].set_title(TitleStringPart + r"E$_{\rm low}$ $\leq$ " + str(self.MaxElowSQL) + " K", y = 1.08)

                        elif (self.LocalPlotOption == "molecule"):
                            layers[PlotID].set_title(TitleStringPart + self.LocalMoleculeName + r", E$_{\rm low}$ $\leq$ " + str(self.MaxElowSQL) \
                                                                     + " K", y = 1.08)


                    ## initialize current subplot
                    layers[PlotID].grid(True)
                    layers[PlotID].set_ylabel(r"Brightness Temperature (K)")
                    layers[PlotID].set_xlabel("Rest Frequency (MHz)")


                    ## define boundaries
                    x1min = LowestDataPoint + PlotID * WidthSpectrum
                    x1max = LowestDataPoint + (PlotID + 1) * WidthSpectrum


                    ## add ghost and U-lines
                    if (self.GhostLines != []):
                        for LocalGhostLine in self.GhostLines:
                            if (x1min <= LocalGhostLine and LocalGhostLine <= x1max):
                                layers[PlotID].axvline(x = LocalGhostLine, color = 'cyan', linewidth = 2.0, linestyle = '--')
                    if (self.ULines != []):
                        for LocalULine in self.ULines:
                            if (x1min <= LocalULine and LocalULine <= x1max):
                                layers[PlotID].axvline(x = LocalULine, color = 'orange', linewidth = 2.0, linestyle = '--')


                    ## get min and max indices
                    FreqObsArray = LocalObsSpectrum[(x1min <= LocalObsSpectrum[:, 0]) & (LocalObsSpectrum[:, 0] <= x1max), 0]
                    IntObsArray = LocalObsSpectrum[(x1min <= LocalObsSpectrum[:, 0]) & (LocalObsSpectrum[:, 0] <= x1max), 1]


                    ## get frequencies and intensities
                    if (len(IntObsArray) > 0):
                        IntObsMin = numpy.nanmin(IntObsArray)
                        IntObsMax = numpy.nanmax(IntObsArray)
                        ListIntObsMinMax.append([IntObsMin, IntObsMax])


                        ## add plot to layer
                        layers[PlotID].plot(FreqObsArray, IntObsArray, '-', color = 'black', linewidth = 3.0, label = 'obs. data', \
                                            drawstyle = 'steps-mid')

                        # Debug:
                        # print ("\n\nx1min, x1max = ", x1min, x1max)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## add modeled spectra for each frequency range to subplots
                ModeledSpectraList = []
                MinAllFreq = 1.e99
                MaxAllFreq = -1.e99


                ## get parameters for current frequency range
                for RangeIndex in range(NumberFrequencyRanges):                             ## loop over all range definitions in the whole xml file


                    ## get parameter for current frequency range
                    ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                    FreqMinListIn = self.FreqMinList, \
                                                                                    FreqMaxListIn = self.FreqMaxList, \
                                                                                    FreqStepListIn = self.FreqStepList,  \
                                                                                    tBackFlagListIn = self.t_back_flagList, \
                                                                                    tBackListIn = self.tBackList, tSlopeListIn = self.tSlopeList, \
                                                                                    N_HListIn = self.N_HList, beta_dustListIn = self.beta_dustList, \
                                                                                    kappa_1300ListIn = self.kappa_1300List, \
                                                                                    DustFileNameListIn = self.DustFileNameList, \
                                                                                    BackgroundFileNameListIn = self.BackgroundFileNameList, \
                                                                                    ContPhenFuncID_ListIn = self.ContPhenFuncIDList, \
                                                                                    ContPhenFuncParam1_ListIn = self.ContPhenFuncParam1List, \
                                                                                    ContPhenFuncParam2_ListIn = self.ContPhenFuncParam2List, \
                                                                                    ContPhenFuncParam3_ListIn = self.ContPhenFuncParam3List, \
                                                                                    ContPhenFuncParam4_ListIn = self.ContPhenFuncParam4List, \
                                                                                    ContPhenFuncParam5_ListIn = self.ContPhenFuncParam5List, \
                                                                                    NoiseListIn = self.NoiseList)
                    FreqMin = ObsXMLParameterDictRange['FreqMin']
                    if (FreqMin is not None):
                        CounterTransWindow += 1
                        FreqMax = ObsXMLParameterDictRange['FreqMax']
                        FreqStep = ObsXMLParameterDictRange['FreqStep']
                        t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                        tBack = ObsXMLParameterDictRange['tBack']
                        tSlope = ObsXMLParameterDictRange['tSlope']
                        nH_flag = True
                        N_H = ObsXMLParameterDictRange['N_H']
                        beta_dust = ObsXMLParameterDictRange['beta_dust']
                        kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                        if (N_H is None or beta_dust is None or kappa_1300 is None):
                            nH_flag = False
                            N_H = 0.0
                            beta_dust = 0.0
                            kappa_1300 = 0.0
                        DustFileName = ObsXMLParameterDictRange['DustFileName']
                        if (DustFileName is None):
                            DustFileName = ""
                        BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                        if (BackgroundFileName is None):
                            BackgroundFileName = ""
                        ContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                        ContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                        ContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                        ContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                        ContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                        ContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']
                        Noise = ObsXMLParameterDictRange['NoiseLevel']
                        if (Noise is None):
                            Noise = 0.0

                        # Debug:
                        # print ("self.LocalMolfitFileName = ", self.LocalMolfitFileName)
                        # print ("LocalExpFileName = ", LocalExpFileName)
                        # print ("FreqMin = ", FreqMin)
                        # print ("FreqMax = ", FreqMax)
                        # print ("FreqStep = ", FreqStep)
                        # print ("t_back_flag = ", t_back_flag)
                        # print ("tBack = ", tBack)
                        # print ("tSlope = ", tSlope)
                        # print ("nH_flag = ", nH_flag)
                        # print ("N_H = ", N_H)
                        # print ("beta_dust = ", beta_dust)
                        # print ("kappa_1300 = ", kappa_1300)
                        # print ("TelescopeSize = ", TelescopeSize)
                        # print ("Inter_Flag = ", Inter_Flag)
                        # print ("IsoTableFileName = ", IsoTableFileName)
                        # print ("NumberHeaderLines = ", NumberHeaderLines)
                        # print ("SeparatorColumns = ", SeparatorColumns)


                        ## compute rms
                        # determine index of freqmin and freqmax
                        # LocalThreshold = task_LineIdentification.ComputeRMS(CurrentIntensity)


                        ##================================================================================================================================
                        ## get modeled spectrum for current range
                        ImportModeledSpectrumFlag = True


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## import result of previous myXCLASSFit function
                        if (self.LocalPlotOption == "fit"):
                            try:
                                LocalmyXCLASSFitFileName = self.myXCLASSFitFileList[ObsDataFileIndex].strip()
                                if (LocalmyXCLASSFitFileName != ""):
                                    modeldata = numpy.loadtxt(LocalmyXCLASSFitFileName, skiprows = 0)
                                else:
                                    ImportModeledSpectrumFlag = False
                            except:
                                ImportModeledSpectrumFlag = False


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## start myXCLASS function
                        elif (self.LocalPlotOption in ["myxclass", "molecule"]):


                            ## get transition frequencies for current molecule
                            LocalLowerMoleculeName = self.LocalMoleculeName.lower()
                            if (self.LocalPlotOption == "molecule"):
                                if (self.TransLabel in ["full", "full-i", "detail"] or (self.TransLabel in ["rrl", "rrl-red"] \
                                    and LocalLowerMoleculeName.startswith("rrl-"))):
                                    IncludeQNFlag = True
                                else:
                                    IncludeQNFlag = False
                                RawTransFreqList, RawDBParamList = task_LineIdentification.GetTransFreq(self.LocalMoleculeName, FreqMin, FreqMax, \
                                                                                                        self.SQLParamArray, self.dbFile, \
                                                                                                        self.MaxNumTransitionsSQL, \
                                                                                                        IncludeQNFlag = IncludeQNFlag)
                                TransFreqList = []
                                DBParamList = []
                                for LocalTransFreqID, LocalTransFreq in enumerate(RawTransFreqList):
                                    if (FreqMin <= LocalTransFreq and LocalTransFreq <= FreqMax):
                                        TransFreqList.append(LocalTransFreq)
                                        DBParamList.append(RawDBParamList[LocalTransFreqID])

                                # Debug:
                                # print ("\n\nself.LocalMoleculeName = ", self.LocalMoleculeName)
                                # print ("FreqMin = ", FreqMin)
                                # print ("FreqMax = ", FreqMax)
                                # print ("self.SQLParamArray = ", self.SQLParamArray)
                                # print ("self.dbFile = ", self.dbFile)
                                # print ("TransFreqList = ", TransFreqList)
                                # print ("DBParamList = ", DBParamList)


                            ## start myXCLASS if
                            if (self.LocalMolfitFileName != "" and (self.LocalPlotOption == "myxclass" or TransFreqList != [])):


                                ## check, if spectra are already calculated
                                ComputeSynthSpecFlag = True
                                if (self.OverallPATH != "" and self.LocalPlotOption == "myxclass"):
                                    LocalPath = self.OverallPATH + "/ObsDataFile__" + str(ObsDataFileIndex + 1) + "___range__" + str(RangeIndex + 1) + "/"
                                    LocalPath = os.path.normpath(LocalPath) + "/"
                                    if (os.path.exists(LocalPath)):
                                        try:
                                            modeldata = numpy.loadtxt(LocalPath + "xclass_spectrum_output.dat")
                                            ComputeSynthSpecFlag = False
                                        except:
                                            modeldata = numpy.loadtxt(LocalPath + "xclass_spectrum_output.dat.gz")
                                            ComputeSynthSpecFlag = False

                                # Debug:
                                # print ("ComputeSynthSpecFlag, self.OverallPATH = ", ComputeSynthSpecFlag, self.OverallPATH)
                                # sys.exit(0)


                                ## compute total spectrum ?
                                if (ComputeSynthSpecFlag):


                                    ## if iso ratio file is defined, merge iso ratio file into molfit file
                                    LocalIsoFlag = self.IsoFlag
                                    if (LocalIsoFlag and self.LocalIsoTableFileName != ""):
                                        LocalIsoFlag = False
                                        CurrentMolfitFileName = self.BaseResultPath + "merged_with_iso.molfit"
                                        task_myXCLASS.MergeMolfitAndIsoRatio(self.LocalMolfitFileName, self.LocalIsoTableFileName, \
                                                                             CurrentMolfitFileName, \
                                                                             MinNtot = self.MinNtot)
                                    else:
                                        CurrentMolfitFileName = self.LocalMolfitFileName


                                    ## call myXCLASS function to compute synthetic spectrum of current frequency range
                                    RestFreq = 0.0
                                    vLSR = GlobalvLSR
                                    printflag = False
                                    SpectrumOnlyFlag = True
                                    modeldata, log, TransEnergies, IntOptical, JobDirLocal = task_myXCLASS.myXCLASSCore(FreqMin = FreqMin, \
                                                                                                    FreqMax = FreqMax, FreqStep = FreqStep, \
                                                                                                    MolfitsFileName = CurrentMolfitFileName, \
                                                                                                    iso_flag = LocalIsoFlag, \
                                                                                                    IsoTableFileName = self.LocalIsoTableFileName, \
                                                                                                    TelescopeSize = TelescopeSize, BMIN = BMIN, \
                                                                                                    BMAJ = BMAJ, BPA = BPA, Redshift = Redshift, \
                                                                                                    Inter_Flag = InterFlag, t_back_flag = t_back_flag, \
                                                                                                    tBack = tBack, tSlope = tSlope, nH_flag = nH_flag, \
                                                                                                    N_H = N_H, beta_dust = beta_dust, \
                                                                                                    kappa_1300 = kappa_1300, \
                                                                                                    DustFileName = DustFileName, \
                                                                                                    BackgroundFileName = BackgroundFileName, \
                                                                                                    ContPhenFuncID = ContPhenFuncID, \
                                                                                                    ContPhenFuncParam1 = ContPhenFuncParam1, \
                                                                                                    ContPhenFuncParam2 = ContPhenFuncParam2, \
                                                                                                    ContPhenFuncParam3 = ContPhenFuncParam3, \
                                                                                                    ContPhenFuncParam4 = ContPhenFuncParam4, \
                                                                                                    ContPhenFuncParam5 = ContPhenFuncParam5, \
                                                                                                    NumModelPixelXX = self.NumModelPixelXX, \
                                                                                                    NumModelPixelYY = self.NumModelPixelYY, \
                                                                                                    LocalOverlapFlag = self.LocalOverlapFlag, \
                                                                                                    EmAbsPATH = self.EmAbsPATH, \
                                                                                                    dbFileName = self.dbFile, RestFreq = RestFreq, \
                                                                                                    vLSR = vLSR, NoSubBeamFlag = self.NoSubBeamFlag, \
                                                                                                    CheckMoleculeInDB = False, \
                                                                                                    SpectrumOnlyFlag = SpectrumOnlyFlag, \
                                                                                                    printFlag = printflag)


                                    ## clean up myXCLASS job directory, i.e. remove files describing intensities and optical depths
                                    if (self.CleanmyXCLASSFlag):
                                        cmdString = "rm -rf " + JobDirLocal + "intensity_*; "
                                        cmdString += "rm -rf " + JobDirLocal + "optical_depth_*; "
                                        os.system(cmdString)
                                    if (CurrentMolfitFileName != self.LocalMolfitFileName):
                                        cmdString = "rm -rf " + CurrentMolfitFileName + "; "
                                        os.system(cmdString)


                                    ## save myXCLASS files
                                    RangeDir = self.BaseResultPath + "ObsDataFile__" + str(ObsDataFileIndex + 1) + "___range__" + str(RangeIndex + 1) + "/"
                                    cmdString = "mkdir -p " + RangeDir + "; "
                                    cmdString += "mv " + JobDirLocal + "/*.* " + RangeDir + "; "
                                    cmdString += "gzip -fr " + RangeDir + "; "
                                    cmdString += "rm -rf " + JobDirLocal
                                    os.system(cmdString)


                                ## store full spectra
                                if (self.LocalPlotOption == "myxclass" and self.AllMoleculesFlag):
                                    self.FullSpectra.append(copy.deepcopy(modeldata))

                                    # Debug:
                                    # print ("len(self.FullSpectra) = ", len(self.FullSpectra))


                                ## synthetic spectrum?
                                if (modeldata == []):
                                    ImportModeledSpectrumFlag = False
                                else:


                                    ## store synthetic spectrum for publication plots
                                    if (self.PubPlotFlag):
                                        vLSR = self.vlsr - GlobalvLSR
                                        for LocalTransFreqID, LocalTransFreq in enumerate(TransFreqList):

                                            # Debug:
                                            # print ("LocalTransFreq = ", LocalTransFreq)


                                            ## extract obs. data around current transition frequency
                                            VelocityData = task_myXCLASS.ChangeToVelocitiy(LocalObsSpectrum[:, 0], LocalTransFreq, vLSR)
                                            VelocityData = numpy.asarray(VelocityData)
                                            MinVelIndex, MaxVelIndex = task_myXCLASSFit.GetArrayPart(VelocityData, self.velLowLimit, self.velUpLimit)
                                            CurrentVelocityAxis = VelocityData[MinVelIndex:MaxVelIndex]
                                            CurrentIntensity = LocalObsSpectrum[MinVelIndex:MaxVelIndex, 1]
                                            LocalObsData = numpy.vstack([CurrentVelocityAxis, CurrentIntensity])
                                            LocalObsData = LocalObsData.T

                                            # Debug:
                                            # if (len(LocalObsData) == 0):
                                            #     print ("\n")
                                            #     print ("LocalTransFreq = ", LocalTransFreq)
                                            #     print ("numpy.nanmin(LocalObsSpectrum[:, 0]) = ", numpy.nanmin(LocalObsSpectrum[:, 0]))
                                            #     print ("numpy.nanmax(LocalObsSpectrum[:, 0]) = ", numpy.nanmax(LocalObsSpectrum[:, 0]))
                                            #     print ("numpy.nanmin(VelocityData[:]) = ", numpy.nanmin(VelocityData[:]))
                                            #     print ("numpy.nanmax(VelocityData[:]) = ", numpy.nanmax(VelocityData[:]))
                                            #     print ("MinVelIndex, MaxVelIndex = ", MinVelIndex, MaxVelIndex)
                                            #     print ("self.velLowLimit, self.velUpLimit = ", self.velLowLimit, self.velUpLimit)
                                            #     print ("CurrentVelocityAxis = ", CurrentVelocityAxis)
                                            #     sys.exit(0)


                                            ## extract synthetic spectrum around current transition frequency
                                            VelocityData = task_myXCLASS.ChangeToVelocitiy(modeldata[:, 0], LocalTransFreq, vLSR)
                                            VelocityData = numpy.asarray(VelocityData)
                                            MinVelIndex, MaxVelIndex = task_myXCLASSFit.GetArrayPart(VelocityData, self.velLowLimit, self.velUpLimit)
                                            CurrentVelocityAxis = VelocityData[MinVelIndex:MaxVelIndex]
                                            CurrentIntensity = modeldata[MinVelIndex:MaxVelIndex, 1]
                                            LocalModData = numpy.vstack([CurrentVelocityAxis, CurrentIntensity])
                                            LocalModData = LocalModData.T


                                            ## get additional spectra
                                            LocalAddSpec = None
                                            if (self.AdditionalSpectra is not None):
                                                LocalRangeCounter = (-1)
                                                for AdditionalRangeSpectraID, AdditionalRangeSpectra in enumerate(self.AdditionalSpectra):
                                                    ffmin = numpy.nanmin(AdditionalRangeSpectra[:, 0])
                                                    ffmax = numpy.nanmax(AdditionalRangeSpectra[:, 0])
                                                    if (ffmin <= LocalTransFreq and LocalTransFreq <= ffmax):
                                                        LocalRangeCounter = AdditionalRangeSpectraID
                                                        break
                                                if (LocalRangeCounter >= 0):
                                                    AddSpecVelData = task_myXCLASS.ChangeToVelocitiy(self.AdditionalSpectra[LocalRangeCounter][:, 0], \
                                                                                                     LocalTransFreq, vLSR)
                                                    AddSpecVelData = numpy.asarray(AddSpecVelData)
                                                    MinAddSpecVelIndex, MaxAddSpecVelIndex = task_myXCLASSFit.GetArrayPart(AddSpecVelData, \
                                                                                                                           self.velLowLimit, \
                                                                                                                           self.velUpLimit)
                                                    if (MinAddSpecVelIndex < MaxAddSpecVelIndex or MinAddSpecVelIndex > 0):
                                                        CurrentAddSpecVelAxis = AddSpecVelData[MinAddSpecVelIndex:MaxAddSpecVelIndex]
                                                        CurrentAddSpecInt = self.AdditionalSpectra[LocalRangeCounter][MinAddSpecVelIndex:MaxAddSpecVelIndex, 1]
                                                        LocalAddSpec = numpy.vstack([CurrentAddSpecVelAxis, CurrentAddSpecInt])
                                                        LocalAddSpec = LocalAddSpec.T


                                            ## extra full spectra around current transition frequency
                                            LocalFullData = []
                                            if (CounterTransWindow <= len(self.FullSpectra)):
                                                LocalSpectra = self.FullSpectra[CounterTransWindow - 1]
                                                VelocityData = task_myXCLASS.ChangeToVelocitiy(LocalSpectra[:, 0], LocalTransFreq, vLSR)
                                                VelocityData = numpy.asarray(VelocityData)
                                                MinVelIndex, MaxVelIndex = task_myXCLASSFit.GetArrayPart(VelocityData, self.velLowLimit, self.velUpLimit)
                                                CurrentVelocityAxis = VelocityData[MinVelIndex:MaxVelIndex]
                                                CurrentIntensity = LocalSpectra[MinVelIndex:MaxVelIndex, 1]
                                                LocalFullData = numpy.vstack([CurrentVelocityAxis, CurrentIntensity])
                                                LocalFullData = LocalFullData.T

                                            # Debug:
                                            # print ("\tlen(LocalObsData) = ", len(LocalObsData))
                                            # print ("\tlen(LocalModData) = ", len(LocalModData))
                                            # print ("\tlen(LocalFullData) = ", len(LocalFullData))


                                            ## store results
                                            # if (len(LocalObsData) > 5 or len(LocalModData) > 5):
                                            if (len(LocalObsData) > 5 and len(LocalModData) > 5):


                                                ## take current transition into account?
                                                TFWParameterDict = {}
                                                EinsteinA = DBParamList[LocalTransFreqID][1]
                                                Elow = DBParamList[LocalTransFreqID][2]
                                                gup = DBParamList[LocalTransFreqID][3]
                                                TFWParameterDict['MoleculeName'] = DBParamList[LocalTransFreqID][4].strip()
                                                TFWParameterDict['TransFreq'] = LocalTransFreq
                                                TFWParameterDict['EinsteinA'] = EinsteinA
                                                TFWParameterDict['ElowMin'] = Elow

                                                if (len(DBParamList[LocalTransFreqID]) > 5):
                                                    TFWParameterDict['QNUp'] = DBParamList[LocalTransFreqID][5].strip()
                                                    TFWParameterDict['QNLow'] = DBParamList[LocalTransFreqID][6].strip()
                                                else:
                                                    TFWParameterDict['QNUp'] = None
                                                    TFWParameterDict['QNLow'] = None
                                                TFWParameterDict['gup'] = gup
                                                TFWParameterDict['ObsData'] = LocalObsData
                                                TFWParameterDict['ModData'] = LocalModData
                                                TFWParameterDict['FullData'] = LocalFullData
                                                TFWParameterDict['AddSpec'] = LocalAddSpec
                                                TFWList.append(TFWParameterDict)


                                                ## define sort parameter list depending on given SQL parameter in molfit file
                                                if (self.OrderTransSQL == 1):               ## use lower energy for sorting
                                                    SortParameterValue = Elow
                                                elif (self.OrderTransSQL == 2):             ## use gA for sorting (-1 for reversed sorting)
                                                    SortParameterValue = (-1.0) * (gup * EinsteinA)
                                                elif (self.OrderTransSQL == 3):             ## use weighting factor for sorting (-1 for reversed sorting)
                                                    SortParameterValue = (-1.0) * ((EinsteinA * gup) / (max(0.1, Elow)**2))
                                                else:
                                                    SortParameterValue = LocalTransFreq
                                                SortTFWParamList.append(SortParameterValue)
                            else:
                                modeldata = numpy.copy(LocalObsSpectrum)
                                modeldata[:, 1] = 0.0


                                ## add continuum
                                if (not self.RemoveContFlag):
                                    if (tBack != 0 or tSlope != 0):                         ## add continuum
                                        for i in range(len(modeldata[:, 1])):
                                            modeldata[i, 1] = (abs(tBack) * (modeldata[i, 0] / FreqMin)**tSlope)

                        # Debug:
                        # print ("\n\nModelDataFile = ", ModelDataFile)
                        # print ("ImportModeledSpectrumFlag = ", ImportModeledSpectrumFlag)
                        ##================================================================================================================================


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## create plots
                        if (ImportModeledSpectrumFlag and len(self.SelectedTransitions) == 0):


                            ## remove continuum
                            if (self.RemoveContFlag):
                                if (abs(numpy.nanmin(modeldata[:, 1])) != 0.0 or abs(numpy.nanmax(modeldata[:, 1])) != 0.0):
                                    if (tBack != 0 or tSlope != 0):                         ## apply a baseline correction if one of the input parameter
                                                                                            ## tBack or tSlope are unequal to zero.
                                        for i in range(len(modeldata[:, 1])):               ## loop over all points in the spectrum
                                            modeldata[i, 1] = modeldata[i, 1] - (abs(tBack) * (modeldata[i, 0] / FreqMin)**tSlope)


                            ## get index for FreqMin and FreqMax
                            ModSpectrum = modeldata[(FreqMin <= modeldata[:, 0]) & (modeldata[:, 0] <= FreqMax), :]


                            ## add modeled spectra to figure
                            for PlotID in range(self.PlotNum):


                                ## define boundaries
                                x1min = LowestDataPoint + PlotID * WidthSpectrum
                                x1max = LowestDataPoint + (PlotID + 1) * WidthSpectrum


                                ## get min and max indices
                                FreqModArray = ModSpectrum[(x1min <= ModSpectrum[:, 0]) & (ModSpectrum[:, 0] <= x1max), 0]
                                IntModArray = ModSpectrum[(x1min <= ModSpectrum[:, 0]) & (ModSpectrum[:, 0] <= x1max), 1]


                                ## get frequencies and intensities
                                if (len(IntModArray) > 1):
                                    IntModMin = numpy.nanmin(IntModArray)
                                    IntModMax = numpy.nanmax(IntModArray)


                                    ## add plot to layer
                                    layers[PlotID].plot(FreqModArray, IntModArray, '-', color = 'red', linewidth = 2.0, label = 'fit')

                                    # Debug:
                                    # print ("\n\nPlotID = ", PlotID)
                                    # print ("FreqModArray = ", FreqModArray)
                                    # print ("FreqMin = ", FreqMin)
                                    # print ("x1min = ", x1min)
                                    # print ("FreqMax = ", FreqMax)
                                    # print ("x1max = ", x1max)


                                    ## add transition frequencies and name of molecules to plot
                                    ErrorFlag = False
                                    try:
                                        IntObsMin = ListIntObsMinMax[PlotID][0]
                                    except:
                                        ErrorFlag = True
                                    if (not ErrorFlag):
                                        IntObsMax = ListIntObsMinMax[PlotID][1]
                                        AllMin = min(IntModMin, IntObsMin) * 0.98
                                        AllMax = max(IntModMax, IntObsMax) * 1.07
                                        if (self.AddMoleculeNamesToPlotFlag):
                                            if (self.LocalPlotOption =="molecule"):
                                                MolfitMoleculesList = [self.LocalMoleculeName]
                                            else:
                                                MolfitMoleculesList = copy.deepcopy(self.MoleculesInMolfitFile)
                                            for LocalMolfitMolecules in MolfitMoleculesList:
                                                LocalMoleculeList = [LocalMolfitMolecules]
                                                if (self.IsoFlag):
                                                    for IsoLine in self.IsoRatioTable:
                                                        LocalIsoMaster = IsoLine[1].strip()
                                                        if (LocalIsoMaster == self.LocalMoleculeName):
                                                            LocalIsotopologue = IsoLine[0].strip()
                                                            LocalMoleculeList.append(LocalIsotopologue)
                                                LocalMolCounter = 0
                                                for LocalMolecule in LocalMoleculeList:
                                                    LocalMolCounter += 1
                                                    TransFreqList, DBParamList = task_LineIdentification.GetTransFreq(LocalMolecule, \
                                                                                                                      max(x1min, FreqMin), \
                                                                                                                      min(x1max, FreqMax), \
                                                                                                                      self.SQLParamArray, \
                                                                                                                      self.dbFile, \
                                                                                                                      self.MaxNumTransitionsSQL)
                                                    for SpecFreq in TransFreqList:
                                                        if (LocalMolecule in MolfitMoleculesList):
                                                            layers[PlotID].axvline(x = SpecFreq, color = 'green', linewidth = 1.0, linestyle = '-.')
                                                            y = AllMax - (AllMin * 0.15)
                                                            layers[PlotID].text(SpecFreq, y, LocalMolecule, fontsize = 10, rotation = 90, color = 'g', \
                                                                                zorder = 1)
                                                        else:
                                                            layers[PlotID].axvline(x = SpecFreq, color = 'blue', linewidth = 1.0, linestyle = '-.')
                                                            y = AllMax - (AllMin * 0.15)
                                                            layers[PlotID].text(SpecFreq, y, LocalMolecule, fontsize = 10, rotation = 90, color = 'b', \
                                                                                zorder = 1)


                                    ## define x and y boundaries
                                    layers[PlotID].axvspan(max(x1min, FreqMin), min(x1max, FreqMax), alpha = 0.2, color = 'grey')
                                    layers[PlotID].set_xlim(x1min, x1max)
                                    layers[PlotID].set_ylim(AllMin, AllMax)


                                    ## add legend
                                    if (RangeIndex == 0 and PlotID == 0):
                                        layers[PlotID].legend(bbox_to_anchor = (0.0, 1.02, 1.0, 0.102), loc = 3, ncol = 2, mode = "expand",
                                                              borderaxespad = 0.0)


                ## save plot
                fig.tight_layout()
                pngFile = self.BaseResultPath
                if (SpectraNamePure is None):
                    pngFile += ""
                else:
                    pngFile += SpectraNamePure
                if (self.LocalPlotOption == "molecule"):
                    pngFile += "___" + self.LocalMoleculeFileName
                if (plotBackend == 'svg'):
                    pylab.savefig('{:s}.svg'.format(pngFile, bbox_inches = 'tight', pad_inches = 0.5))
                    os.system('inkscape {:s}.svg --export-pdf={:s}.pdf'.format(pngFile, pngFile))
                else:
                    if (self.PDFExportFlag):
                        pylab.savefig(pngFile + ".pdf")
                pylab.savefig(pngFile + ".png")
                pylab.draw()
                matplotlib.pyplot.close(fig)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create publication plots
        fig = None
        if (self.PubPlotFlag):


            ## get total number of TFW
            if (len(self.SelectedTransitions) > 0):
                TotalNumTFW = len(self.SelectedTransitions)
            else:
                TotalNumTFW = len(TFWList)

            # Debug:
            # print ("TotalNumTFW = ", TotalNumTFW)


            ## generate list of TFW IDs
            if (self.MaxNumTransitionsSQL > 0):
                NumPlots = min(TotalNumTFW, self.MaxNumTransitionsSQL)
            else:
                NumPlots = TotalNumTFW

            # Debug:
            # print ("NumPlots = ", NumPlots)


            ## define local number of rows and columns
            LocalNumberRows = self.NumberRows
            LocalNumberColumns = self.NumberColumns

            # Debug:
            # print ("LocalNumberRows = ", LocalNumberRows)
            # print ("LocalNumberColumns = ", LocalNumberColumns)


            ## sort TFWList
            SortTFWParamList = numpy.asarray(SortTFWParamList)
            if (len(self.SelectedTransitions) == 0):                                        ## take all transitions into account
                SortTFWIndex = numpy.argsort(SortTFWParamList)
            else:
                LocalSortTFWIndex = numpy.argsort(SortTFWParamList)
                SortTFWIndex = []
                for SelectedID in self.SelectedTransitions:
                    for LocalID, Localindex in enumerate(LocalSortTFWIndex):
                        if (SelectedID == (LocalID + 1)):
                            SortTFWIndex.append(Localindex)


                # old
                # for LocalID, Localindex in enumerate(LocalSortTFWIndex):
                #     if ((LocalID + 1) in self.SelectedTransitions):
                #         SortTFWIndex.append(Localindex)

            # Debug:
            # print ("SortTFWParamList = ", SortTFWParamList)
            # print ("SortTFWIndex = ", SortTFWIndex)
            # print ("numpy.argsort(SortTFWParamList) = ", numpy.argsort(SortTFWParamList))
            # print ("self.SelectedTransitions = ", self.SelectedTransitions)


            ## set latex flag for matplotlib
            # LatexFlag = matplotlib.rcParams['text.usetex']
            # matplotlib.rcParams['text.usetex'] = True


            ## set size of figure to describe complete DIN A4 page
            fig_width_cm = 21.0                                                             ## A4 page width (in cm)
            fig_height_cm = 29.7                                                            ## A4 page height (in cm)
            inches_per_cm = 1.0 / 2.58                                                      ## Convert cm to inches
            fig_width = fig_width_cm * inches_per_cm                                        ## width in inches
            fig_height = fig_height_cm * inches_per_cm                                      ## height in inches
            if (LocalNumberRows < self.NumberRows):                                         ## reduce page height for low number of plots
                fig_height = fig_height / (self.NumberRows - LocalNumberRows)
            NumPlotPerFigure = LocalNumberRows * LocalNumberColumns                         ## define max. number of subplots per page
            MaxPageNumber = int(numpy.ceil(NumPlots / NumPlotPerFigure))                    ## get max. number of pages
            PageCounter = 0
            LocalMoleculeLTE = task_LineIdentification.MoleculeFileName(self.LocalMoleculeName)
            TFWIndex = (-1)                                                                 ## initialize TFW index
            for PageID in range(0, NumPlots, NumPlotPerFigure):                             ## how many pages do we need
                PageCounter += 1


                ## print what you do
                print ("\r\tCreate page (" + str(PageCounter) + "/" + str(MaxPageNumber) + ") .. ", end = "", flush = True)


                ## create figures with defined number of subplots
                ## taken from https://stackoverflow.com/questions/13365617/large-number-of-subplots-with-matplotlib
                fig, ax_array = plt.subplots(LocalNumberRows, LocalNumberColumns, squeeze = False)
                fig.set_size_inches(fig_width, fig_height)
                fig.subplots_adjust(bottom = 0.06, top = 0.93, right = 0.96, left = 0.09, hspace = 0.0, wspace = 0.5)
                #fig.suptitle("Molecule: " + chr(34) + LocalMoleculeLTE + chr(34), fontsize = 12)
                ObsDataFlag = False
                BackFlag = False
                LTEFlag = False
                NLTEFlag = False
                lines = []
                labels = []
                CreateSubPlotFlag = False
                for i, ax_row in enumerate(ax_array):
                    for j, axes in enumerate(ax_row):


                        ## define plot index
                        TFWIndex += 1
                        if (TFWIndex < NumPlots):

                            # Debug:
                            # print ("i, j, TFWIndex = ", i, j, TFWIndex)


                            ## get parameters of current TFW
                            CreateSubPlotFlag = True

                            # Debug:
                            # print ("\nTFWIndex = " , TFWIndex)
                            # print ("len(SortTFWIndex) = " , len(SortTFWIndex))
                            # print ("SortTFWIndex[TFWIndex] = " , SortTFWIndex[TFWIndex])
                            # print ("len(TFWList) = " , len(TFWList))


                            TFWParameterDict  = TFWList[SortTFWIndex[TFWIndex]]
                            LocalMoleculeName = TFWParameterDict['MoleculeName']
                            LocalTransFreq = TFWParameterDict['TransFreq']
                            LocalEinsteinA = TFWParameterDict['EinsteinA']
                            LocalElowMin = TFWParameterDict['ElowMin']
                            Localgup = TFWParameterDict['gup']
                            LocalQNUp = TFWParameterDict['QNUp']
                            LocalQNLow = TFWParameterDict['QNLow']
                            LocalObsData = TFWParameterDict['ObsData']
                            LocalModData = TFWParameterDict['ModData']
                            LocalFullData = TFWParameterDict['FullData']
                            LocalAddSpec = TFWParameterDict['AddSpec']


                            ## add vertical line at v_lsr
                            axes.axvline(x = self.vlsr, color = 'grey', linewidth = 1, linestyle = '--')


                            ## add grid?
                            if (self.GridFlag):
                                axes.grid(True)


                            ## add obs. data
                            In1, In2 = task_myXCLASSFit.GetArrayPart(LocalObsData[:, 0], self.velLowLimit, self.velUpLimit)
                            axes.plot(LocalObsData[In1:In2, 0], LocalObsData[In1:In2, 1], '-', linewidth = 1.0, color = 'blue', drawstyle = 'steps-mid', \
                                      label = "obs. data")


                            ## add additional spectrum
                            if (LocalAddSpec is not None):
                                AddSpecLS = task_LineIdentification.GetDicKey(self.AdditionalSpectraDict, "ls", ":")
                                AddSpecLW = task_LineIdentification.GetDicKey(self.AdditionalSpectraDict, "lw", 1.5)
                                AddSpecColor = task_LineIdentification.GetDicKey(self.AdditionalSpectraDict, "color", "orange")
                                AddSpecLabel = task_LineIdentification.GetDicKey(self.AdditionalSpectraDict, "label", "add. spectrum")
                                In1, In2 = task_myXCLASSFit.GetArrayPart(LocalAddSpec[:, 0], self.velLowLimit, self.velUpLimit)
                                axes.plot(LocalAddSpec[In1:In2, 0], LocalAddSpec[In1:In2, 1], AddSpecLS, linewidth = AddSpecLW, \
                                          color = AddSpecColor, label = AddSpecLabel)


                            ## add full spectrum
                            if (len(LocalFullData) > 0):
                                In1, In2 = task_myXCLASSFit.GetArrayPart(LocalFullData[:, 0], self.velLowLimit, self.velUpLimit)
                                axes.plot(LocalFullData[In1:In2, 0], LocalFullData[In1:In2, 1], '--', linewidth = 2.0, color = 'green', label = "total")


                            ## add synthetic spectrum
                            In1, In2 = task_myXCLASSFit.GetArrayPart(LocalModData[:, 0], self.velLowLimit, self.velUpLimit)
                            axes.plot(LocalModData[In1:In2, 0], LocalModData[In1:In2, 1], '-', linewidth = 1.5, color = 'red', label = "fit")


                            ## set x-limits, center line, and grid
                            axes.set_xlim(self.velLowLimit, self.velUpLimit)


                            ## deactivate x-ticks for all plots except last row
                            DeactivateLabelFlag = False
                            if ((not i == (LocalNumberRows - 1)) and PageCounter < MaxPageNumber):
                                DeactivateLabelFlag = True
                            elif (PageCounter == MaxPageNumber and TFWIndex < (NumPlots - LocalNumberColumns)):
                                DeactivateLabelFlag = True
                            if (DeactivateLabelFlag):
                                axes.set_xticklabels([])


                            ## only for the plot in the lower left corner add axes labels
                            if (not DeactivateLabelFlag and j == 0):
                                axes.set_ylabel(r"Brightness Temperature (K)")
                                axes.set_xlabel(r"velocity (km s$^{-1}$)")
                                axes.legend(handles = lines, labels = labels, bbox_to_anchor = (0., 1.02, 2.2, .102), loc = 3, \
                                            ncol = LocalNumberColumns, mode = "expand", borderaxespad = 0, bbox_transform = plt.gcf().transFigure)


                            ## enlarge plots for molecular parameters
                            axes.autoscale(enable = True, axis = 'y')
                            ymin, ymax = axes.get_ylim()
                            Deltay = abs(ymax - ymin)
                            Newymax = ymin + Deltay * 1.36
                            axes.set_ylim(ymin, Newymax)


                            ## add molecular parameters to each plot
                            LocalLatexName = task_ListDatabase.LatexMoleculeName(LocalMoleculeName)
                            LocalTitelString = ""
                            if (len(self.SelectedTransitions) == 0):
                                LocalTitelString += chr(34) + LocalLatexName + chr(34)
                                LocalTitelString += "  (" + str(TFWIndex + 1) + "/" + str(NumPlots) + ")"
                            LocalTitelString += "\n"
                            TransFreqString = "%.1f" % LocalTransFreq
                            LocalTitelString += r"$\nu_{\rm Trans}$ = " + TransFreqString + " MHz,\n"


                            ## special handling of RRLs
                            if (LocalMoleculeName.startswith("RRL-")):
                                if (self.TransLabel in ["rrl-red", "full-i"]):
                                    QNUpID = int(LocalQNUp.replace("n_u =", "").replace("n_l =", "").strip())
                                    QNLowID = int(LocalQNLow.replace("n_l =", "").replace("n_u =", "").strip())
                                    DeltaN = abs(QNUpID - QNLowID)
                                    LocalRRLName = LocalMoleculeName.replace("RRL-", "")
                                    LocalTitelString += "Line: {:s}{:d}".format(LocalRRLName, QNLowID)
                                    DeltaNList = [r"$\alpha$", r"$\beta$", r"$\gamma$", r"$\delta$", r"$\epsilon$", r"$\zeta$", \
                                                  r"$\eta$", r"$\theta$", r"$\iota$", r"$\kappa$", ".."]
                                    DeltaNName = DeltaNList[min(len(DeltaNList), DeltaN - 1)]
                                    LocalTitelString += DeltaNName
                                else:
                                    LocalTitelString += r"E$_{\rm low}$ = " + str(LocalElowMin) + " K"
                                    if (self.TransLabel in ["rrl"]):


                                        ## add name of RRL line, i.e. "Name-of-RRL" + "lower-main-quantum-number" + "Delta_n"
                                        QNUpID = int(LocalQNUp.replace("n_u =", "").replace("n_l =", "").strip())
                                        QNLowID = int(LocalQNLow.replace("n_l =", "").replace("n_u =", "").strip())
                                        DeltaN = abs(QNUpID - QNLowID)
                                        LocalRRLName = LocalMoleculeName.replace("RRL-", "")
                                        LocalTitelString += ",\nLine: {:s}{:d}".format(LocalRRLName, QNLowID)
                                        DeltaNList = [r"$\alpha$", r"$\beta$", r"$\gamma$", r"$\delta$", r"$\epsilon$", r"$\zeta$", \
                                                    r"$\eta$", r"$\theta$", r"$\iota$", r"$\kappa$", ".."]
                                        DeltaNName = DeltaNList[min(len(DeltaNList), DeltaN - 1)]
                                        LocalTitelString += DeltaNName + r" ($\Delta$ n = {:d})".format(DeltaN)


                            ## normal molecules
                            else:
                                EinsteinAString = "${0:.2f} \cdot 10^{{{1:d}}}$".format(*ConvertFloatLatexStyle(LocalEinsteinA))
                                gString = "%.1f" % Localgup
                                LocalgA = ((LocalEinsteinA * Localgup) / (max(0.1, LocalElowMin)**2))
                                gAString = "${0:.2f} \cdot 10^{{{1:d}}}$".format(*ConvertFloatLatexStyle(LocalgA))
                                if (self.TransLabel in ["full", "full-i", "detail"]):
                                    LocalTitelString += r"Einstein A = " + EinsteinAString + r" s$^{-1}$," + "\n"
                                    LocalTitelString += r"$g_{\rm up}$ = " + gString + r"," + "\n"
                                LocalTitelString += r"E$_{\rm low}$ = " + str(LocalElowMin) + " K"
                                if (self.TransLabel in ["compact", "rrl", "rrl-red"]):
                                    LocalTitelString += ",\n" + r"gA/E$_{\rm low}^2$ = " + gAString + r" s$^{-1}$ K$^{-2}$"
                            if (self.TransLabel in ["full"]):
                                LocalTitelString += ",\nQN: " + LocalQNUp + " -- " + LocalQNLow
                            elif (self.TransLabel in ["full-i"] and (not LocalMoleculeName.startswith("RRL-"))):
                                LocalQNUp = InterpretQN(LocalQNUp)
                                LocalQNLow = InterpretQN(LocalQNLow)
                                LocalTitelString += ",\nQN:  " + LocalQNUp + r" $\leftarrow$ " + LocalQNLow
                            axes.text(0.05, 0.95, LocalTitelString, horizontalalignment = 'left', verticalalignment = 'top', \
                                    transform = axes.transAxes, fontsize = self.MolecularLabelFontSize, \
                                    bbox = dict(facecolor = 'white', alpha = 0.5))
                        else:
                            axes.axis('off')


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## save plot to file
                #fig.tight_layout(rect = [0.05, 0.04, 0.99, 0.95])                          ## [left, bottom, right, top] in normalized (0, 1) fig. coord.
                fig.tight_layout()
                if (CreateSubPlotFlag):
                    LocalFileName = self.BaseResultPath + LocalMoleculeLTE + "__page_{:09d}.pdf".format(PageCounter)
                    fig.savefig(LocalFileName, dpi = 300)
                    plt.close(fig)


            ## print what you do
            print ("done!", flush = True)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## combine all pdf files into one big pdf file
            listing = [x for x in os.listdir(self.BaseResultPath) if x.startswith(LocalMoleculeLTE + "__page_")]
            if (len(listing) > 0 and (self.pdfgsFlag or self.pdfUniteFlag)):
                FinalPDFFileName = self.BaseResultPath + LocalMoleculeLTE + "__all.pdf"
                if (self.pdfgsFlag):
                    cmdString = "gs -q -dNOPAUSE -dBATCH -sDEVICE=pdfwrite -sOutputFile=" + FinalPDFFileName + " "
                    cmdString += self.BaseResultPath + LocalMoleculeLTE + "__page_*.pdf ; "
                elif (self.pdfUniteFlag):
                    cmdString = "pdfunite " + self.BaseResultPath + LocalMoleculeLTE + "__page_*.pdf " + FinalPDFFileName + "; "
                cmdString += "rm -rf " + self.BaseResultPath + LocalMoleculeLTE + "__page_*.pdf"
                os.system(cmdString)


                ## if allowed crop blank
                if (self.AllowCropPDF and len(self.SelectedTransitions) > 0):
                    cmdString = "pdfcrop " + FinalPDFFileName + " " + FinalPDFFileName
                    os.system(cmdString)


            ## deactivate matplotlib latex flag
            if (fig is not None):
                matplotlib.pyplot.close(fig)
            # matplotlib.rcParams['text.usetex'] = LatexFlag


        ## return to main routine
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## plot results of myXCLASSFit function
    def PlotFitResults(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print what you do
        print ("\n\nPlot results of myXCLASSFit function for each obs. data file:", flush = True)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create plot subdirectory
        self.BaseResultPath = self.BaseResultPathOrig + "plot_myXCLASSFit/"
        cmdString = "rm -rf " + self.BaseResultPath + "; mkdir " + self.BaseResultPath
        os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine best result


        ## get all filenames in the current job directory
        listing = os.listdir(self.JobDir)

        # Debug:
        # print ("\n\n\nlisting = ", listing)


        ## get number of chi2 log files
        NumOutFiles = 0                                                                     ## determine number of output files
        NumMolfitFiles = 0
        for files in listing:
            if (files.endswith(".log.chi2")):
                NumOutFiles += 1
            if (files.endswith(".out.molfit")):
                NumMolfitFiles += 1
        if (NumOutFiles == 0):                                                              ## check if new input files exsist
            print ("\n\n\n\nError for plot option fit / all:")
            print ("\tCan not find a chi2 log file!")
            print ("\n\tself.JobDir = ", self.JobDir)
            print ("\tlisting = ", listing)
            print ("\n\n\n")
            return


        ## determine the chi2 log file, which corresponds to the best fit
        BestChi2Value = 1.e99
        FileNameBestChi2 = ""
        for files in listing:                                                               ## loop over all files in the current job directory
            if (files.endswith(".log.chi2")):                                               ## we're interested only in the log.chi2 files
                Chi2LogFile = open(self.JobDir + files)                                     ## open log.chi2 file
                dummyline = Chi2LogFile.readline()                                          ## read in first line as dummy
                LogLine = Chi2LogFile.readline()                                            ## read second line with best chi2 value
                Chi2LogFile.close()                                                         ## close log file
                LogLine = LogLine.strip()                                                   ## remove blanks
                SplittedLine = LogLine.split()                                              ## split line into columns
                chi2Value = float(SplittedLine[1])                                          ## get chi2 value
                if (chi2Value <= BestChi2Value):                                            ## check, if current chi2 value is the best one
                    BestChi2Value = chi2Value                                               ## if yes, save chi2 value
                    FileNameBestChi2 = files                                                ## and corresponding file name

                    # Debug:
                    # print ("chi2Value = ", chi2Value)


        ## get phrase identifying the used algorithm for the best chi2 value
        BestAlgorithm = ""
        BestAlgorithm = FileNameBestChi2.replace(".log.chi2", "")
        BestAlgorithm = BestAlgorithm.replace("fit__", "")
        if (NumMolfitFiles == 1 and BestAlgorithm.find("LM") > (-1)):                       ## special handling for LM
            BestAlgorithm = "LM.out"


        ## find model data file
        for CounterRange in range(self.NumberExpFiles):                                     ## loop over all frequency ranges


            ## analyze obs. data file name, we only want the name, not the path
            ExpDataFileName = self.ListOfSpectraNames[CounterRange][1]                      ## get path and name of corresponding obs. data file
            i = ExpDataFileName.rfind("/")
            ExpDataFileName = ExpDataFileName[i + 1:]
            if (ExpDataFileName.endswith(".dat")):
                ExpDataFileName = ExpDataFileName.replace(".dat", "")
            elif (ExpDataFileName.endswith(".fits")):
                ExpDataFileName = ExpDataFileName.replace(".fits", "")


            ## find obs. data file in the current single molecule fit directory
            NewLocalFile = ""
            LocalmyXCLASSFitFileName = ""
            for files in listing:                                                           ## loop over all files in the current job directory
                if (files.startswith(ExpDataFileName) and (files.endswith(BestAlgorithm) or files.endswith(".out.dat"))):
                    i = files.find(BestAlgorithm)
                    if (i > (-1)):
                        NewLocalFile = self.JobDir + files
                        if (not (NewLocalFile in self.myXCLASSFitFileList)):
                            LocalmyXCLASSFitFileName = NewLocalFile
            self.myXCLASSFitFileList.append(NewLocalFile)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create plots
        self.UsemyXCLASSFlag = False
        self.LocalPlotOption = "fit"
        self.PlotObsDataFiles()


        ## return to main routine
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create plots for each obs. data file including names of molecules
    def PlotMoleculeContributions(self, AllMoleculesFlag = False, AddContribution = None, ExcludeMolecules = [], PubPlotFlag = False, \
                                  PlotIsoFlag = True, SelectedMolecule = None, SelectedTransitions = [], IgnoreOtherRRLFlag = False, \
                                  pdfgsFlag = True, pdfUniteFlag = True, AllowCropPDF = True, OverallPATH = "", AdditionalSpectra = None, \
                                  AdditionalSpectraDict = None):
        """

    input parameters:
    -----------------

        - AllMoleculesFlag:             (optional) flag indicating computation of full spectra (default: False)

        - AddContribution:              (optional) additional contribution to the molfit file (default: None)

        - ExcludeMolecules:             (optional) list of excluded molecules (default: []])

        - PubPlotFlag:                  (optional) flag indicating creation of plots for publication (default: False)

        - PlotIsoFlag:                  (optional) flag indicating plot of (if available) isotopologues (default: False)

        - SelectedMolecule:             (optional) restrict contribution plots to given molecule (default: None)

        - SelectedTransitions:          (optional) list of selected transitions (default: [])

        - IgnoreOtherRRLFlag:           (optional) ignore contributions from other RRLs (default: False)

        - pdfgsFlag:                    (optional) allow merging of single pdf files using gs (default: True)

        - pdfUniteFlag:                 (optional) allow merging of single pdf files using pdfunite (default: True)

        - AllowCropPDF:                 (optional) allow crop of final pdf document if single transitions are selected (default: True)

        - OverallPATH:                  (optional) path of folder containing overall fit (default: "")

        - AdditionalSpectra:            (optional) additional spectra added to transition plots (default: None)
                                        The parameter has to contain as many 2d numpy arrays as freq. ranges

        - AdditionalSpectraDict:        (optional) dictionary of plotting options for additional spectra (default: None)

                                        Used keys:
                                        - 'ls':     line style (default: ":")
                                        - 'lw':     line width (default: "1.5")
                                        - 'color':  color (default: "orange")
                                        - 'label':  label (default: "add. spectrum")



    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("AllMoleculesFlag = ", AllMoleculesFlag)
        # print ("AddContribution = ", AddContribution)
        # print ("ExcludeMolecules = ", ExcludeMolecules)
        # print ("PubPlotFlag = ", PubPlotFlag)
        # print ("PlotIsoFlag = ", PlotIsoFlag)
        # print ("SelectedMolecule = ", SelectedMolecule)
        # print ("SelectedTransitions = ", SelectedTransitions)
        # print ("IgnoreOtherRRLFlag = ", IgnoreOtherRRLFlag)
        # print ("pdfgsFlag = ", pdfgsFlag)
        # print ("pdfUniteFlag = ", pdfUniteFlag)
        # print ("AllowCropPDF = ", AllowCropPDF)
        # print ("OverallPATH = ", OverallPATH)
        # print ("AdditionalSpectra = ", AdditionalSpectra)
        # print ("AdditionalSpectraDict = ", AdditionalSpectraDict)


        ## store list of selected transitions
        self.SelectedTransitions = SelectedTransitions
        self.pdfgsFlag = pdfgsFlag
        self.pdfUniteFlag = pdfUniteFlag
        self.AllowCropPDF = AllowCropPDF
        self.OverallPATH = OverallPATH
        self.AdditionalSpectra = AdditionalSpectra
        self.AdditionalSpectraDict = AdditionalSpectraDict


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## compute full spectra
        self.AllMoleculesFlag = AllMoleculesFlag
        if (self.AllMoleculesFlag):
            self.PlotmyXCLASS()


        ## print what you do
        self.PubPlotFlag = PubPlotFlag
        print ("\n\nCreate plots for each molecule:", flush = True)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check if external programs are available
        if (self.PubPlotFlag):
            ListExternalPrograms = ["gs", "pdfunite"]
            if (self.AllowCropPDF):
                ListExternalPrograms.append("pdfcrop")
            for ExternalProgram in ListExternalPrograms:
                rc = subprocess.call(['which', ExternalProgram])
                if (rc != 0):
                    print ("\n\n\nError in the XCLASS package function myXCLASSPlot:")
                    print ("\n\tThe external program {:s} is not available!".format(ExternalProgram))
                    print ("\n\tThe corresponding functionality (gs / pdfunit: merge pdf files, pdfcrop: crop pdf files")
                    print ("\n\tcan not be provided.\n\n")
                    print ("\n\tPlease install {:s}{:s}{:s}\n\n".format(chr(34), ExternalProgram, chr(34)))
                    if (ExternalProgram == "gs"):
                        self.pdfgsFlag = False
                    elif (ExternalProgram == "pdfunite"):
                        self.pdfUniteFlag = False
                    elif (ExternalProgram == "pdfcrop"):
                        self.AllowCropPDF = False


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## loop over all molecules in molfit file
        AlreadyKnownMoleculeList = []
        for MoleculeIndex in range(len(self.MoleculesInMolfitFile)):
            LocalMasterMolecule = self.MoleculesInMolfitFile[MoleculeIndex]


            ## check, if molecule should be considered
            ConsiderFlag = False
            if (SelectedMolecule is not None):
                if (LocalMasterMolecule == SelectedMolecule):
                    ConsiderFlag = True
            else:
                if (not LocalMasterMolecule in ExcludeMolecules):
                    ConsiderFlag = True


            ## check, if molecule has to be taken into account
            if (ConsiderFlag):
                LocalMolfitContents = self.MolfitFileForEachMolecule[MoleculeIndex]
                AlreadyKnownMoleculeList.append(LocalMasterMolecule)


                ## print which molecule is analyzed
                print ("Analzye molecule " + LocalMasterMolecule + " .. ")


                ## create a subdirectory for current molecule
                i = AlreadyKnownMoleculeList.count(LocalMasterMolecule)
                self.LocalMoleculeFileName = task_LineIdentification.MoleculeFileName(LocalMasterMolecule)
                if (i > 1):
                    self.LocalMoleculeFileName += "__" + str(i)


                ## create plot subdirectory
                LocalMoleculePath = self.BaseResultPathOrig + "plot_molecule-contribution/" + self.LocalMoleculeFileName + "/"
                self.BaseResultPath = LocalMoleculePath
                cmdString = "rm -rf " + LocalMoleculePath + "; mkdir -p " + LocalMoleculePath
                os.system(cmdString)


                ## take isotopologues into account as well?
                if (PlotIsoFlag and len(self.IsoRatioTable) > 0):
                    LocalIsoRatioTable = copy.deepcopy(self.IsoRatioTable)
                    self.PlotIsoFlag = True
                else:
                    LocalIsoRatioTable = []
                    self.PlotIsoFlag = False
                LocalIsoRatioTable.append([LocalMasterMolecule, LocalMasterMolecule, "1.0"])  ## add iso master molecule


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## start loop over all isotopologues
                BackupIsoFlag = self.IsoFlag
                for LocalIsoLine in LocalIsoRatioTable:
                    LocalIsoMaster = LocalIsoLine[1].strip()
                    if (LocalIsoMaster == LocalMasterMolecule):                             ## take only those isotopologues into account which belongs
                                                                                            ## to current master molecule
                        LocalIsotopologue = LocalIsoLine[0].strip()
                        self.LocalMoleculeName = LocalIsotopologue
                        LocalRatio = str(LocalIsoLine[2]).strip()


                        ## create new subdirectory?
                        if (self.PlotIsoFlag):
                            i = AlreadyKnownMoleculeList.count(LocalIsotopologue)
                            self.LocalMoleculeFileName = task_LineIdentification.MoleculeFileName(LocalIsotopologue)
                            if (i > 1):
                                self.LocalMoleculeFileName += "__" + str(i)
                            self.BaseResultPath = LocalMoleculePath + self.LocalMoleculeFileName + "/"
                            cmdString = "mkdir -p " + self.BaseResultPath
                            os.system(cmdString)


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## create molfit file for current molecule
                        self.LocalMolfitFileName = self.BaseResultPath + "molecule.molfit"
                        LocalMolfitFile = open(self.LocalMolfitFileName, 'w')


                        ## add SQL parameter definitions
                        for NewLine in self.MolfitHeaderLinesList:
                            LocalMolfitFile.write(NewLine)


                        ## check, if continuum contributions have to be taken into account as well
                        for LocalMoleculeID, LocalMolecule in enumerate(self.MoleculesInMolfitFile):
                            LowerLocalMolecule = LocalMolecule.lower()
                            if (LowerLocalMolecule.startswith("cont-") and MoleculeIndex != LocalMoleculeID):
                                for NewLine in self.MolfitFileForEachMolecule[LocalMoleculeID]:
                                    LocalMolfitFile.write(NewLine.replace("\n", "") + "\n")


                        ## check if additional contribution is defined
                        if (AddContribution is not None):
                            if (type(AddContribution) is str):
                                AddContributionContent = []
                                if (AddContribution.strip() != ""):
                                    f = open(AddContribution)
                                    AddContributionContent = f.readlines()
                                    f.close()
                            else:
                                AddContributionContent = copy.deepcopy(AddContribution)
                            for NewLine in AddContributionContent:
                                LocalMolfitFile.write(NewLine)


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## add molecule parameters


                        ## do we describe isomaster molecule?
                        if (LocalIsotopologue == LocalIsoMaster):
                            for NewLine in LocalMolfitContents:
                                LocalMolfitFile.write(NewLine.replace("\n", "") + "\n")


                        ## create molfit file for isotopologue
                        else:
                            self.IsoFlag = False                                            ## no iso ratio file for isotopologues


                            ## add name of molecule
                            LocalMolfitParameters = self.AllParameters[MoleculeIndex]
                            NewLine = LocalIsotopologue + "   " + str(len(LocalMolfitParameters)) + "\n"
                            LocalMolfitFile.write(NewLine)


                            ## add molecule parameters
                            NewLine = ""
                            for LocalLine in LocalMolfitParameters:

                                # Debug:
                                # print ("LocalLine = ", LocalLine)

                                LocalvOff = 0.0
                                NewComp = " "
                                for col in LocalLine:
                                    Name = col[0]
                                    flag = col[1]
                                    element = col[4]

                                    # Debug:
                                    # print ("col = ", col)
                                    # print ("Name = ", Name)


                                    ## store values for source size
                                    if (Name in ["source_size", "source_radius", "source_center_x", "source_center_y", "T_rot", "T_e", \
                                                "T_cont_dust", "T_dOff", "T_dSlope", "EM_RRL", "nHcolumn_cont_dust", "nHcolumn", \
                                                "V_width_Gauss", \
                                                "V_off", "beta_cont_dust", "kappa_cont_dust", "kappa", "beta", \
                                                "ContFuncID_param_1", "ContFuncID_param_2", "ContFuncID_param_3", \
                                                "ContFuncID_param_4", "ContFuncID_param_5"]):
                                        value = float(element)
                                        NewComp += "  {:12.5e}".format(value)


                                    ## resize column density
                                    if (Name in ["N_tot"]):
                                        # try:
                                        value = float(element) / float(LocalRatio)
                                        # except:
                                        #     value = 1.e1
                                        NewComp += "  {:12.5e}".format(value)


                                    ## store values for EA-Flag
                                    elif (Name in ["LayerDistance", "ContFuncID_phen", "CFFlag", "keyword"]):
                                        NewComp += "    " + element


                                ## add line break
                                NewLine += NewComp + "\n"


                            ## write content to file
                            LocalMolfitFile.write(NewLine)


                        ## close molfit file
                        LocalMolfitFile.close()


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## create plots
                        self.UsemyXCLASSFlag = True
                        self.LocalPlotOption = "molecule"
                        self.PlotObsDataFiles()


                        ## restore original name
                        self.IsoFlag = BackupIsoFlag


        ## define return parrameters
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## plot results of myXCLASS function
    def PlotmyXCLASS(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print what you do
        print ("\n\nCall myXCLASS function for each obs. data file:", flush = True)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create plot subdirectory
        OldBaseResultPath = self.BaseResultPath
        self.BaseResultPath = self.BaseResultPath + "plot_myXCLASS/"
        cmdString = "rm -rf " + self.BaseResultPath + "; mkdir -p " + self.BaseResultPath
        os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create plots
        self.UsemyXCLASSFlag = True
        self.LocalPlotOption = "myxclass"
        self.PlotObsDataFiles()
        self.BaseResultPath = OldBaseResultPath


        ## return to main routine
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## interpret quantum number string from database
##
def InterpretQN(QNString):
    """

input parameters:
-----------------

    - QNString:                 string describing quantum number


output parameters:
------------------

    - NewQNString:              latex string
    """

    # Debug:
    # print ("QNString = ", QNString)


    ## initialize return parameter
    NewQNString = ""


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get quantum numbers
    QNString = QNString.strip()
    SplittedQNString = QNString.split(";")
    QNDict = {}
    for element in SplittedQNString:
        StrippedElement = element.strip()
        LowerStrippedElement = StrippedElement.lower()
        LocalString = LowerStrippedElement.split("=")
        if (len(LocalString) == 2):
            KeyWord = LocalString[0].strip().lower()
            Value = LocalString[1].strip().lower()
            QNDict[KeyWord] = Value
            if (KeyWord in ["v", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12"]):
                try:
                    ListKeyWord = QNDict["vib"]
                except:
                    ListKeyWord = []
                ListKeyWord.append(KeyWord)
                QNDict["vib"] = ListKeyWord
            elif (KeyWord in ["f", "f1", "f2", "f3", "f4", "f5", "f_o"]):
                try:
                    ListKeyWord = QNDict["hyp"]
                except:
                    ListKeyWord = []
                ListKeyWord.append(KeyWord)
                QNDict["hyp"] = ListKeyWord
        else:
            KeyWord = ""
            Value = ""

    # Debug:
    # print ("QNDict = ", QNDict)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## identify kind of molecule
    FormatFlag = False


    ## symmetric rotor
    if (("j" in QNDict.keys()) and ("k" in QNDict.keys()) ):
        NewQNString = r"J$_{{K}}$ = {:s}$_{{{:s}}}$".format(QNDict["j"], QNDict["k"])
        FormatFlag = True


    ## asymmetric rotor
    elif (("j" in QNDict.keys()) and ("ka" in QNDict.keys()) and ("kc" in QNDict.keys())):
        NewQNString = r"J$_{{K_a-K_c}}$ = {:s}$_{{ {:s}-{:s} }}$".format(QNDict["j"], QNDict["ka"], QNDict["kc"])
        FormatFlag = True


    ## no electronic angular momentum, i.e. J = N
    elif (("j" in QNDict.keys()) and ("n" in QNDict.keys()) ):
        NewQNString = r"J$_N$ = {:s}$_{:s}$".format(QNDict["j"], QNDict["n"])
        FormatFlag = True


    ## no electronic angular momentum, i.e. J = N
    elif (("j" in QNDict.keys())):
        NewQNString = r"J = {:s}".format(QNDict["j"])
        FormatFlag = True


    ## add additional attributes
    if (FormatFlag):


        ## add vibrational state
        if ("vib" in QNDict.keys()):
            ListKeyWord = QNDict["vib"]
            for KeyWord in ListKeyWord:
                if (KeyWord in ["v12???"]):
                    LocalKeyWord = r"v$_{12}$"
                else:
                    LocalKeyWord = ""
                if (LocalKeyWord != ""):
                    NewQNStringVib = ", {:s} = {:s}".format(LocalKeyWord, QNDict[KeyWord])
                    if (not NewQNStringVib in [", v = 0", ", v$_{12}$ = 0"]):
                        NewQNString += NewQNStringVib


        ## add hyperfine structure states
        elif ("hyp" in QNDict.keys()):
            ListKeyWord = QNDict["hyp"]
            for KeyWord in ListKeyWord:
                NewQNString += ", {:s} = {:s}".format(KeyWord, QNDict[KeyWord])


        ## add rovibsym state
        elif ("rovibsym" in QNDict.keys()):
            NewQNString += ", {:s}".format(QNDict["rovibsym"].upper())

    # Debug:
    # print ("NewQNString = ", NewQNString)


    ## we're done
    return NewQNString
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert float number to latex style
## taken from: https://stackoverflow.com/questions/13490292/format-number-using-latex-notation-in-python
## use as:    '{0:.2f}^{{{1:d}}}'.format(*ConvertFloatLatexStyle(x))
##
def ConvertFloatLatexStyle(x):
    """

input parameters:
-----------------

    - x:                        float number which is converted to latex style


output parameters:
------------------

    - base:                     base of x

    - exponent:                 exponent of x
    """

    # Debug:
    # print ("x = ", x)


    ## compute exponent
    try:
        exponent = int(numpy.floor(numpy.log10(abs(x))))
    except:
        exponent = 1
        print ("\n\nError in XCLASS package, function myXCLASSPlot.ConvertFloatLatexStyle:")
        print ("\n\tCan not compute exponent of x = ", x)
        print ("\n\tSet exponent to 1.\n\n")


    ## compute base
    try:
        base = x / 10.0**exponent
    except:
        base = 1.0
        print ("\n\nError in XCLASS package, function myXCLASSPlot.ConvertFloatLatexStyle:")
        print ("\n\tCan not compute base of x = ", x)
        print ("\n\tSet base to 1.\n\n")


    ## return to main routine
    return base, exponent
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## core routine of the myXCLASSPlot function
##
def myXCLASSPlotCore(obsdata = None, expdata = None, modeldata = None, TransEnergies = None, RestFreq = 0.0, vLSR = 0.0, \
                     MinIntensity = 0.0, xLowerLimit = None, xUpperLimit = None, yLowerLimit = None, yUpperLimit = None, \
                     UseLatexName = True, PlotTitle = None, LegendFlag = True, SaveFigureFileName = None):
    """

This function creates a plot showing the (observational) data stored in the input parameter
"expdata" as black dots, the calculated model function stored in the parameter "modeldata"
as green line and the transition energies from the table "Transitions" stored in the parameter
"TransEnergies" as blue vertical lines.


input parameters:
-----------------

    - obsdata:              2D numpy array containing the observational data, (default: None)

                            Note, the data has to be given as a function of frequency (in MHz)

    - expdata:              alias for obsdata

    - modeldata:            2D numpy array containing the values of a model function, (default: None)

                            Note, the data has to be given as a function of frequency (in MHz)

    - TransEnergies:        python list containing the transition frequencies (in MHz) from the last myXCLASS run, the Doppler-shifted transition
                            frequencies (in MHz), the corresponding intensities (in K), the energy of the lower level (in K), the upper state
                            degeneracy, the Einstein A coefficient (in s^{-1}), and the molecule names, (default: None)

    - RestFreq:             rest frequency in MHz (default: 0.0). (If this parameter is set to zero, the intensity is plotted against frequency (in MHz)
                            otherwise against velocity (in km/s).

    - vLSR:                 velocity (local standard of rest) in km/s, only used, if RestFreq /= 0. (velocity(Frequency = RestFreq) = vLSR)

    - MinIntensity:         minimal intensity (in K) of a transition for plotting molecule names (default: 0, i.e. plot all names)

    - xLowerLimit:          lower limit (in MHz / km/s) of the frequency/velocity (default: 0). (Depending on the value of the rest frequency: If
                            rest frequency is set to zero, the lower limit has to be given as frequency) If parameter is not given, all data will
                            be plotted (default: None).

    - xUpperLimit:          upper limit (in MHz / km/s) of the frequency/velocity (default: 1.e8). (Depending on the value of the rest frequency:
                            If rest frequency is set to zero, the upper limit has to be given as frequency) If parameter is not given, all data will
                            be plotted (default: None).

    - yUpperLimit:          upper limit of the intensity (y-axis) (default: None). (If value is not given or if yUpperLimit is equal to yLowerLimit then
                            the y-axis range is set automatically.)

    - yLowerLimit:          lower limit of the intensity (y-axis) (default: None). (If value is not given or if yUpperLimit is equal to yLowerLimit then
                            the y-axis range is set automatically.)

    - UseLatexName:         flag indicating usage of latex file name (default: True)

    - PlotTitle:            defines the title of the plot (default: None)

    - LegendFlag:           defines, if legend is plottet (true) or not (false) (default: True)

    - SaveFigureFileName:   defines the path and name of the file to which the current figure is stored. If no file is defined, i.e. "", the figure is
                            shown in a GUI and not saved (default: None).



output parameters:
------------------

    - None


Example:
--------

FreqMin = 580102.0
FreqMax = 580546.5
FreqStep = 0.5
TelescopeSize = 3.5
Inter_Flag = F
t_back_flag = T
tBack = 1.1
tslope = 0.0
nH_flag = T
N_H = 3.0E+24
beta_dust = 2.0
kappa_1300 = 0.02
MolfitsFileName = "demo/myXCLASS/CH3OH__pure.molfit"
iso_flag = T
IsoTableFileName = "demo/myXCLASS/iso_names.txt"
RestFreq = 0.0
vLSR = 0.0
modeldata, log, TransEnergies, IntOptical, jobDir = myXCLASS()

FileName = "demo/myXCLASS/band1b.dat"
NumHeaderLines = 1
expdata = LoadASCIIFile()

MinIntensity = 0.0
xLowerLimit = 580102.0
xUpperLimit = 580546.5
yLowerLimit = 0.8
yUpperLimit = 2.5
PlotTitle = "Example for myXCLASSPlot function"
LegendFlag = T
SaveFigureFileName = ""
myXCLASSPlot()
    """

    # Debug:
    # print ("obsdata = ", obsdata)
    # print ("expdata = ", expdata)
    # print ("modeldata = ", modeldata)
    # print ("TransEnergies = ", TransEnergies)
    # print ("RestFreq = ", RestFreq)
    # print ("vLSR = ", vLSR)
    # print ("MinIntensity = ", MinIntensity)
    # print ("xLowerLimit = ", xLowerLimit)
    # print ("xUpperLimit = ", xUpperLimit)
    # print ("yLowerLimit = ", yLowerLimit)
    # print ("yUpperLimit = ", yUpperLimit)
    # print ("UseLatexName = ", UseLatexName)
    # print ("PlotTitle = ", PlotTitle)
    # print ("LegendFlag = ", LegendFlag)
    # print ("SaveFigureFileName = ", SaveFigureFileName)


    ##====================================================================================================================================================
    ## analyze expdata parameter
    if (obsdata is None and expdata is not None):
        obsdata = copy.deepcopy(expdata)
    expdata = copy.deepcopy(obsdata)
    expdata_flag = ""
    if (obsdata is not None):
        if (type(obsdata).__name__ == 'list'):                                              ## is obsdata a list ?
            expdata_flag = "list"
        else:                                                                               ## is obsdata an array ?
            expdata_flag = "array"

    # Debug:
    # print ("obsdata = ", obsdata)
    # print ("expdata_flag = ", expdata_flag)


    ##====================================================================================================================================================
    ## analyze modeldata parameter
    modeldata_flag = ""
    if (modeldata is not None):
        if (type(modeldata).__name__ == 'list'):                                            ## is modeldata a list ?
            modeldata_flag = "list"
        else:                                                                               ## is modeldata an array ?
            modeldata_flag = "array"

    # Debug:
    # print ("modeldata_flag = ", modeldata_flag)


    ##====================================================================================================================================================
    ## analyze xlimit parameter
    xlimits_flag = False
    if (xLowerLimit is not None and xUpperLimit is not None):
        if (xUpperLimit != xLowerLimit):
            if (xUpperLimit < xLowerLimit):
                print ("\nError in XCLASS package, function myXCLASSPlot:")
                print ("\n\tThe given upper limit of x-axis is smaller than the lower limit of x-axis!")
                print ("\n\tPlease correct the limits of x-axis parameter and start myXCLASSPlot again.\n\n")
                return
            else:
                xlimits_flag = True

    # Debug:
    # print ("xlimits_flag = ", xlimits_flag)


    ##====================================================================================================================================================
    ## analyze ylimit parameter
    ylimits_flag = False
    if (yLowerLimit is not None and yUpperLimit is not None):
        if (yUpperLimit != yLowerLimit):
            if (yUpperLimit < yLowerLimit):
                print ("\nError in XCLASS package, function myXCLASSPlot:")
                print ("\n\tThe given upper limit of y-axis is smaller than the lower limit of y-axis!")
                print ("\n\tPlease correct the limits of y-axis parameter and start myXCLASSPlot again.\n\n")
                return
            else:
                ylimits_flag = True

    # Debug:
    # print ("ylimits_flag = ", ylimits_flag)


    ##====================================================================================================================================================
    ## check input variables


    ## check data array
    PlotObsDataFlag = True
    if (obsdata is None):
        PlotObsDataFlag = False
    elif (len(obsdata[0]) < 2 or len(obsdata) < 3):
        PlotObsDataFlag = False

    # Debug:
    # print ("PlotObsDataFlag = ", PlotObsDataFlag)


    ## check model array
    PlotModelDataFlag = True
    if (modeldata is None):
        PlotModelDataFlag = False
    elif (len(modeldata[0]) < 2 or len(modeldata) < 3):
        PlotModelDataFlag = False

    # Debug:
    # print ("PlotModelDataFlag = ", PlotModelDataFlag)


    ## check transition frequency array
    PlotTransEnergiesFlag = True
    if (TransEnergies is None):
        PlotTransEnergiesFlag = False
    else:
        if (len(TransEnergies) < 1 or RestFreq != 0.0):
            PlotTransEnergiesFlag = False
        elif (len(TransEnergies) == 1):
            if (TransEnergies == 0.0 or TransEnergies == ["0.0"] or TransEnergies == [["0.0"]] or TransEnergies == [["0.0", ""]]):
                PlotTransEnergiesFlag = False
        elif (len(TransEnergies) < 1):
            lotTransEnergiesFlag = False

    # Debug:
    # print ("PlotTransEnergiesFlag = ", PlotTransEnergiesFlag)


    ## extract transition frequencies which fulfill the creteria
    FinalTransitionArray = []
    MolFreqName = []
    if (PlotTransEnergiesFlag):
        TransEn = 0.0
        MaxTransInt = 0.0
        TransAddedFlag = False
        for line in TransEnergies:
            if (line[0].find("%") == (-1)):
                FirstIndex = 0
                if (len(line) > 7):
                    TransEn = float(line[0])
                    MaxTransInt = 0.0
                    FirstIndex = 1
                    TransAddedFlag = False
                if (not TransAddedFlag):
                    LocalInt = float(line[FirstIndex + 1])                                  ## get intensity
                    if (LocalInt > MaxTransInt):
                        MaxTransInt = LocalInt
                    if (MaxTransInt >= MinIntensity):
                        TransAddedFlag = True
                        FinalTransitionArray.append(TransEn)
                        LocalMoleculeName = line[-1].strip()
                        if (UseLatexName):
                            LocalMoleculeName = task_ListDatabase.LatexMoleculeName(LocalMoleculeName)
                        MolFreqName.append(LocalMoleculeName)

    # Debug:
    # print ("PlotObsDataFlag = ", PlotObsDataFlag)
    # print ("PlotModelDataFlag = ", PlotModelDataFlag)
    # print ("PlotTransEnergiesFlag = ", PlotTransEnergiesFlag)
    # print ("FinalTransitionArray = ", FinalTransitionArray)
    # print ("MolFreqName = ", MolFreqName)


    ##====================================================================================================================================================
    ## test if there is something to plot
    if ((not PlotObsDataFlag) and (not PlotModelDataFlag) and (not PlotTransEnergiesFlag)):
        print ("\nError in XCLASS package, function myXCLASSPlot:")
        print ("\n\tNo data to plot!\n\n")
        return


    ##====================================================================================================================================================
    ## clear plot
    if (SaveFigureFileName is not None):
        fig = pylab.figure(figsize=(17, 10))
    else:
        fig = pylab.figure(1)
    fig.clear()


    ##====================================================================================================================================================
    ## create plot
    ax1 = fig.add_subplot(111)
    if (RestFreq != 0.0):
        ax2 = ax1.twiny()
    if (PlotTitle is not None):
        ax1.set_title(r'{:s}'.format(PlotTitle))
    ax1.grid(True)
    # ax1.set_ylabel(r"T$_{\rm MB}$ [K]")
    ax1.set_ylabel(r"Brightness Temperature (K)")

    # Debug:
    # print ("RestFreq = ", RestFreq)
    # print ("PlotObsDataFlag = ", PlotObsDataFlag)
    # print ("PlotModelDataFlag = ", PlotModelDataFlag)
    # print ("PlotTransEnergiesFlag = ", PlotTransEnergiesFlag)


    ## if RestFreq != 0.0, plot data against velocity
    if (RestFreq != 0.0):
        ax1.set_xlabel(r"velocity (km s$^{-1}$)")


        ## convert data points for obsdata array
        if (PlotObsDataFlag):
            if (expdata_flag == "list"):
                FreqData = [(li[0]) for li in obsdata]
            else:
                FreqData = obsdata[:,0]
            expX = task_myXCLASS.ChangeToVelocitiy(FreqData, RestFreq, vLSR)


        ## convert data points for modeldata array
        if (PlotModelDataFlag):
            if (modeldata_flag == "list"):
                FreqData = [(li[0]) for li in modeldata]
            else:
                FreqData = modeldata[:,0]
            modX = task_myXCLASS.ChangeToVelocitiy(FreqData, RestFreq, vLSR)


        ## convert data points for TransEn array
        if (PlotTransEnergiesFlag):
            FreqData = FinalTransitionArray[:]
            transX = task_myXCLASS.ChangeToVelocitiy(FreqData, RestFreq, vLSR)


    ## plot data against frequency
    else:
        ax1.set_xlabel("Frequency (MHz)")
        if (PlotObsDataFlag):
            if (expdata_flag == "list"):
                expX = [(li[0]) for li in obsdata]
            else:
                expX = obsdata[:,0]
        if (PlotModelDataFlag):
            if (modeldata_flag == "list"):
                modX = [(li[0]) for li in modeldata]
            else:
                modX = modeldata[:,0]
        if (PlotTransEnergiesFlag):
            transX = FinalTransitionArray[:]


    ## check limits for exp data
    LowerLimitExpX = 0
    UpperLimitExpX = 0
    if (PlotObsDataFlag):


        ## check Limits
        LowestVal = min(s for s in expX)
        if (not xlimits_flag):
            LowerLimitExpX = LowestVal
        else:
            if (xLowerLimit < LowestVal):
                LowerLimitExpX = LowestVal
            else:
                LowerLimitExpX = xLowerLimit

        HighestVal = max(s for s in expX)
        if (not xlimits_flag):
            UpperLimitExpX = HighestVal
        else:
            if (xUpperLimit > HighestVal):
                UpperLimitExpX = HighestVal
            else:
                UpperLimitExpX = xUpperLimit

        # Debug:
        # print ("LowerLimitExpX = ", LowerLimitExpX)
        # print ("UpperLimitExpX = ", UpperLimitExpX)


    ## check limits for model data
    LowerLimitModX = 0
    UpperLimitModX = 0
    if (PlotModelDataFlag):


        ## check Limits
        LowestVal = min(s for s in modX)
        if (not xlimits_flag):
            LowerLimitModX = LowestVal
        else:
            if (xLowerLimit < LowestVal):
                LowerLimitModX = LowestVal
            else:
                LowerLimitModX = xLowerLimit

        HighestVal = max(s for s in modX)
        if (not xlimits_flag):
            UpperLimitModX = HighestVal
        else:
            if (xUpperLimit > HighestVal):
                UpperLimitModX = HighestVal
            else:
                UpperLimitModX = xUpperLimit


    ## redefine limit values
    if (PlotObsDataFlag and PlotModelDataFlag):
        xLowerLimit = max(LowerLimitExpX, LowerLimitModX)
        xUpperLimit = min(UpperLimitExpX, UpperLimitModX)

    elif (PlotObsDataFlag and (not PlotModelDataFlag)):
        xLowerLimit = LowerLimitExpX
        xUpperLimit = UpperLimitExpX

    elif ((not PlotObsDataFlag) and PlotModelDataFlag):
        xLowerLimit = LowerLimitModX
        xUpperLimit = UpperLimitModX
    ax1.set_xlim(xLowerLimit, xUpperLimit)

    # Debug:
    # print ('xLowerLimit, xUpperLimit = ', xLowerLimit, xUpperLimit)


    ##====================================================================================================================================================
    ## add experimental to plot
    if (PlotObsDataFlag):
        if (expdata_flag == "list"):
            expY = [(li[-1]) for li in obsdata]
        else:
            expY = obsdata[:,-1]


        ## plot experimental data and set ranges
        ax1.plot(expX, expY, '-', color = 'black', linewidth = 2.0, label = 'obs. data', drawstyle = 'steps-mid')
        ax1.set_xlim(xLowerLimit, xUpperLimit)
        if (xlimits_flag):


            ## get index in expX where expX = xLowerLimit and expX = xUpperLimit
            Localexpy = expY[(xLowerLimit <= expX) & (expX <= xUpperLimit)]
            minexp = min(Localexpy)
            maxexp = max(Localexpy)
        else:
            minexp = min(expY)
            maxexp = max(expY)

        # Debug:
        # print ("minexp = ", minexp)
        # print ("maxexp = ", maxexp)


    ##====================================================================================================================================================
    ## add model data to plot
    if (PlotModelDataFlag):
        if (modeldata_flag == "list"):
            modY = [(li[-1]) for li in modeldata]
        else:
            modY = modeldata[:,-1]


        ## plot experimental data and set ranges
        ax1.plot(modX, modY, '-', color = 'red', label = 'model data', drawstyle = 'steps-mid')
        ax1.set_xlim(xLowerLimit, xUpperLimit)
        if (xlimits_flag):


            ## get index in modX where modX = xLowerLimit and modX = xUpperLimit
            LocalmodY = modY[(xLowerLimit <= modX) & (modX <= xUpperLimit)]
            minmodel = min(LocalmodY)
            maxmodel = max(LocalmodY)
        else:
            minmodel = min(modY)
            maxmodel = max(modY)

        # Debug:
        # print ("minmodel = ", minmodel)
        # print ("maxmodel = ", maxmodel)


    ## set y ranges
    if (not ylimits_flag):                                                                  ## continue here, if no user limits are given
        if (PlotObsDataFlag and PlotModelDataFlag and PlotTransEnergiesFlag):               ## T T T
            yLowLim = min(minexp, minmodel)
            yUpLim = max(maxexp, maxmodel)
        elif (PlotObsDataFlag and PlotModelDataFlag and (not PlotTransEnergiesFlag)):       ## T T F
            yLowLim = min(minexp, minmodel)
            yUpLim = max(maxexp, maxmodel)
        elif (PlotObsDataFlag and (not PlotModelDataFlag) and (not PlotTransEnergiesFlag)): ## T F F
            yLowLim = minexp
            yUpLim = maxexp
        elif ((not PlotObsDataFlag) and PlotModelDataFlag and (not PlotTransEnergiesFlag)): ## F T F
            yLowLim = minmodel
            yUpLim = maxmodel
        elif ((not PlotObsDataFlag) and (not PlotModelDataFlag) and PlotTransEnergiesFlag): ## F F T
            yLowLim = 0.0
            yUpLim = 1.0
        elif (PlotObsDataFlag and (not PlotModelDataFlag) and PlotTransEnergiesFlag):       ## T F T
            yLowLim = minexp
            yUpLim = maxexp
        elif ((not PlotObsDataFlag) and PlotModelDataFlag and PlotTransEnergiesFlag):       ## F T T
            yLowLim = minmodel
            yUpLim = maxmodel
    else:
        yLowLim = yLowerLimit
        yUpLim = yUpperLimit
    if (PlotObsDataFlag or PlotModelDataFlag or PlotTransEnergiesFlag):
        ax1.set_ylim(yLowLim, yUpLim)


    ##====================================================================================================================================================
    ## add transition energies to plot
    if (PlotTransEnergiesFlag):


        ## construct array for y positions of ticks
        CounterTransFreq = (-1)
        for lines in transX:
            CounterTransFreq += 1


            #    ## add ticks to plot
            #    if (PlotObsDataFlag and PlotModelDataFlag):
            #        val = min(minexp, minmodel)
            #    elif (PlotObsDataFlag and not PlotModelDataFlag):
            #        val = minexp
            #    elif ((not PlotObsDataFlag) and PlotModelDataFlag):
            #        val = minmodel
            #    else:
            #        val = 1.0
            #    yPos.append(val)


            #    ## add vertical ticks at transition frequencies (velocities) to graph
            #    ax1.plot(transX, yPos, '|', markersize= 10, color='blue', label = 'transition energies')


            ## add vertical ticks at transition frequencies (velocities) to graph
            if (xLowerLimit <= lines and lines <= xUpperLimit):
                ax1.axvline(x = lines, color = 'blue', linewidth = 1, linestyle = '--', zorder = 0)
                ax1.text(lines, 1.0, MolFreqName[CounterTransFreq], verticalalignment = 'top', fontsize = 10, \
                         rotation = 90, color = 'g', zorder = 1, transform = ax1.get_xaxis_transform())


    ## if rest freuqency is unequal zero, add frequency axis on top of the graph
    if (RestFreq != 0.0):
        cLight = 299792458.0                                                                ## speed of light in m/s
        cLight = cLight * 1.e-3                                                             ## speed of light in km/s
        fMin = RestFreq - (xLowerLimit - vLSR) / cLight * RestFreq
        fMax = RestFreq - (xUpperLimit - vLSR) / cLight * RestFreq
        ax2.set_xlim(fMin, fMax)
        ax2.set_xlabel("Rest Frequency (MHz)")
        step = (fMax - fMin) / 5                                                            ## define only 3 ticks
        xtic = numpy.arange(fMin, fMax, step)
        ax2.set_xticks(xtic)
        ax2.xaxis.set_major_formatter(matplotlib.ticker.FormatStrFormatter('%.3e'))


    ## show legend
    if (LegendFlag):
        if (PlotTitle is not None):
            ax1.legend()
        else:
            ax1.legend(bbox_to_anchor = (0.0, 1.02, 1.0, 0.102), loc = 3, ncol = 4, mode = "expand", borderaxespad = 0.0)


    ##====================================================================================================================================================
    ## show or save figure
    if (SaveFigureFileName is not None):
        SaveFigureFileName = SaveFigureFileName.strip()
        if (SaveFigureFileName != ""):
            pylab.savefig(SaveFigureFileName, dpi = 300)
        pylab.draw()
    else:
        pylab.show(block = True)


    ## close figure environment
    pylab.close(fig)


    ## we're done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## A ploting routine
##
def myXCLASSPlot(expdata, modeldata, TransEnergies, RestFreq, vLSR, MinIntensity, xLowerLimit, xUpperLimit, yLowerLimit, yUpperLimit, PlotTitle, \
                 LegendFlag, SaveFigureFile):
    """

This function creates a plot showing the (observational) data stored in the input parameter
"expdata" as black dots, the calculated model function stored in the parameter "modeldata"
as green line and the transition energies from the table "Transitions" stored in the parameter
"TransEnergies" as blue vertical lines.


input parameters:
-----------------

    - expdata:              2D numpy array containing the observational data

                            Note, the data has to be given as a function of frequency (in MHz)

    - modeldata:            2D numpy array containing the values of a model function

                            Note, the data has to be given as a function of frequency (in MHz)

    - TransEnergies:        python list containing the transition frequencies (in MHz) from the last myXCLASS run, the Doppler-shifted transition
                            frequencies (in MHz), the corresponding intensities (in K), the energy of the lower level (in K), the upper state
                            degeneracy, the Einstein A coefficient (in s^{-1}), and the molecule names.

    - RestFreq:             rest frequency in MHz (default: 0). (If this parameter is set to zero, the intensity is plotted against frequency (in MHz)
                            otherwise against velocity (in km/s).

    - vLSR:                 velocity (local standard of rest) in km/s, only used, if RestFreq /= 0. (velocity(Frequency = RestFreq) = vLSR)

    - MinIntensity:         minimal intensity (in K) of a transition for plotting molecule names (default: 0, i.e. plot all names)

    - xLowerLimit:          lower limit (in MHz / km/s) of the frequency/velocity (default: 0). (Depending on the value of the rest frequency: If
                            rest frequency is set to zero, the lower limit has to be given as frequency) If parameter is not given, all data will
                            be plotted (default: 0).

    - xUpperLimit:          upper limit (in MHz / km/s) of the frequency/velocity (default: 1.e8). (Depending on the value of the rest frequency:
                            If rest frequency is set to zero, the upper limit has to be given as frequency) If parameter is not given, all data will
                            be plotted (default: 1.e8).

    - yUpperLimit:          upper limit of the intensity (y-axis) (default: 0). (If value is not given or if yUpperLimit is equal to yLowerLimit then
                            the y-axis range is set automatically.)

    - yLowerLimit:          lower limit of the intensity (y-axis) (default: 0). (If value is not given or if yUpperLimit is equal to yLowerLimit then
                            the y-axis range is set automatically.)

    - PlotTitle:            defines the title of the plot (default: "")

    - LegendFlag:           defines, if legend is plottet (true) or not (false) (default: "T")

    - SaveFigureFile:       defines the path and name of the file to which the current figure is stored. If no file is defined, i.e. "", the figure is
                            shown in a GUI and not saved (default: "").



output parameters:
------------------

    - None


Example:
--------

FreqMin = 580102.0
FreqMax = 580546.5
FreqStep = 0.5
TelescopeSize = 3.5
Inter_Flag = F
t_back_flag = T
tBack = 1.1
tslope = 0.0
nH_flag = T
N_H = 3.0E+24
beta_dust = 2.0
kappa_1300 = 0.02
MolfitsFileName = "demo/myXCLASS/CH3OH__pure.molfit"
iso_flag = T
IsoTableFileName = "demo/myXCLASS/iso_names.txt"
RestFreq = 0.0
vLSR = 0.0
modeldata, log, TransEnergies, IntOptical, jobDir = myXCLASS()

FileName = "demo/myXCLASS/band1b.dat"
NumHeaderLines = 1
expdata = LoadASCIIFile()

MinIntensity = 0.0
xLowerLimit = 580102.0
xUpperLimit = 580546.5
yLowerLimit = 0.8
yUpperLimit = 2.5
PlotTitle = "Example for myXCLASSPlot function"
LegendFlag = T
SaveFigureFile = ""
myXCLASSPlot()
    """

    # Debug:
    # print ("len(expdata) = ", len(expdata))
    # print ("len(modeldata) = ", len(modeldata))
    # print ("len(TransEnergies) = ", len(TransEnergies))
    # print ("TransEnergies = ", TransEnergies)
    # print ("MinIntensity = ", MinIntensity)
    # print ("RestFreq = ", RestFreq)
    # print ("vLSR = ", vLSR)
    # print ("xUpperLimit = ", xUpperLimit)
    # print ("xLowerLimit = ", xLowerLimit)
    # print ("yUpperLimit = ", yUpperLimit)
    # print ("yLowerLimit = ", yLowerLimit)


    ## call core routine of myXCLASSPlot function
    myXCLASSPlotCore(expdata = expdata, modeldata = modeldata, TransEnergies = TransEnergies, RestFreq = RestFreq, vLSR = vLSR, \
                     MinIntensity = MinIntensity, xLowerLimit = xLowerLimit, xUpperLimit = xUpperLimit, yLowerLimit = yLowerLimit, \
                     yUpperLimit = yUpperLimit, PlotTitle = PlotTitle, LegendFlag = LegendFlag, SaveFigureFileName = SaveFigureFile)

    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------
