#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module contains the interface to the myXCLASS program.
##  Copyright (C) 2012 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function GetXCLASSSettings:                   get settings for XCLASS from the init file
##      - function GetXCLASSRootDir:                    get root path of XCLASS package
##      - function GetmyXCLASSRootDir:                  get path of myXCLASS program
##      - function GetDefaultDBFile:                    get path and name of default database file
##      - function GetMAGIXRootDir:                     get absolute path of MAGIX directory
##      - function GetCommandLineArguments:             get command line arguments
##      - function ExtendSysPath:                       extend sys.path variable
##      - function CheckNCopy:                          check if path and name of given input file exists and copy input file to current working directory
##      - function NormNCopyMove:                       normalize path and copy / move files
##      - function ConvertFreq:                         convert frequency for given velocity or redshift
##      - function ChangeToVelocitiy:                   convert frequency points (in MHz) into velocities (in km/s)
##      - function GetParameterLimits:                  get parameter limits from parameter fitflag (valid only for OLD format)
##      - function AnalyzeMolfitFile:                   read in an analyze molfit file
##      - function GetObsXMLFileParameters:             get parameters from obs. xml file
##      - function ConvertParamValue:                   convert parameter value from linear to log value and vise versa
##      - function ConvertNtotToLogScale:               convert column and hydrogen column density (if given for each component) to log scale
##      - function GetFreeFreeParameters:               get parameters for free-free description from the RRL components
##      - function GetUnitParameter:                    determine unit of myXCLASS parameter
##      - function GetLatexParameterName:               determine latex name of myXCLASS parameter
##      - function WriteLineMolfitFile:                 write a line describing component to molfit file
##      - function WriteMolfitFile:                     write molfit parameters to new molfit file
##      - function CreateErrorMolfitFile:               creates a machine readable molfit file containing the best fit values, the left and right errors,
##                                                      the mean value and standard deviation for each free parameter
##      - function GetSQLParameter:                     get SQL parameters from molfit file
##      - function WriteSQLParameter:                   write SQL parameters to molfit file
##      - function ConversionMolfit2XML:                converts a molfit file used by myXCLASS to a xml-file used by MAGIX.
##      - function ImportIsoRatioFile:                  import the iso ratio file
##      - function GetIsoTableParameters:               read iso table parameters from iso table file and include parameters to xml file
##      - function MergeMolfitAndIsoRatio:              merge molfit and iso ratio file into one big molfit file
##      - function CallmyXCLASS:                        call myXCLASS program from command line
##      - function myXCLASSCore:                        calls myXCLASS program
##      - function myXCLASS:                            function defines the interface for the myXCLASS program
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2013-07-25      initial version
##  T. Moeller      2014-04-09      improved version
##  T. Moeller      2015-04-23      add function ImportIsoRatioFile
##  T. Moeller      2020-01-03      porting to python 3.x
##  T. Moeller      2023-09-02      adjustments for new XCLASS structure
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy                                                                                ## import numpy package
import scipy.integrate                                                                      ## import scipy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import copy                                                                                 ## import copy package
import copy                                                                                 ## import package copy
if (not 'matplotlib' in sys.modules):
    import matplotlib                                                                       ## import matplotlib package
    matplotlib.use("Agg")                                                                   ## avoid display error
from . import task_MAGIX                                                                    ## import package MAGIX
from . import task_myXCLASSFit                                                              ## import package myXCLASSFit
from . import task_myXCLASSMapFit                                                           ## import package task_myXCLASSMapFit
from . import task_LineIdentification                                                       ## import package task_LineIdentification
import gzip                                                                                 ## import package gzip
from matplotlib.font_manager import fontManager                                             ## import some special packages from matplotlib
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get settings for XCLASS from the init file
##
def GetXCLASSSettings():
    """

input parameters:
-----------------

    - None


output parameters:
------------------

    - OutDict:          python dictionary containing the following items:

                        * 'XCLASSSystemRootDirectory':  XCLASS root directory
                        * 'DatabaseFileName':           path and name of XCLASS default database file
                        * 'NoMAGIXFlag':                flag indicating usage of MAGIX
"""


    ## initialize return parameter
    OutDict = {}


    ## get home directory
    HomeDir = str(os.environ.get('HOME',''))
    HomeDir = os.path.normpath(HomeDir) + "/"

    # Debug:
    # print ("HomeDir = ", HomeDir)


    ## import init file
    XCLASSHomeDir = "{:s}.xclass/".format(HomeDir)
    InitFileName = "{:s}init.dat".format(XCLASSHomeDir)
    if (not os.path.isfile(InitFileName)):
        print ("\n\n\nError in XCLASS package!")
        print ("\n\tCan not find file describing XCLASS settings located")
        print ("\n\tin the xclass subfolder in the user's home directory, i.e.")
        print ("\n\n\t\t$(HOME)/.xclass/init.dat")
        print ("\n\n\tPlease check your XCLASS installation!\n\n")
        sys.exit(0)
    else:
        InitFile = open(InitFileName)
        InitFileConent = InitFile.readlines()
        InitFile.close()


    ## import XCLASS settings from init file
    XCLASSSystemRootDirectory = ""
    DatabaseFileName = ""
    NoMAGIXFlag = False
    TimeStampFormat = "dd-mm-yyyy"
    for line in InitFileConent:
        StrippedLine = line.strip()


        ## get xclass root directory
        if (StrippedLine.startswith("XCLASSRootDirectory")):
            SplittedLine = StrippedLine.split("=")
            XCLASSSystemRootDirectory = SplittedLine[1].strip()
            if (not XCLASSSystemRootDirectory.endswith("/")):
                XCLASSSystemRootDirectory += "/"


        ## get path and name of database file
        elif (StrippedLine.startswith("DatabaseFileName")):
            SplittedLine = StrippedLine.split("=")
            DatabaseFileName = SplittedLine[1].strip()


        ## get xclass root directory
        elif (StrippedLine.startswith("NoMAGIX")):
            SplittedLine = StrippedLine.split("=")
            NoMAGIXFlag = SplittedLine[1].strip()
            NoMAGIXFlag = task_myXCLASSMapFit.CheckBool(NoMAGIXFlag)


        ## get format of time stamp for job directories
        elif (StrippedLine.startswith("TimeStampFormat")):
            SplittedLine = StrippedLine.split("=")
            TimeStampFormat = SplittedLine[1].strip()

    # Debug:
    # print ("XCLASSSystemRootDirectory = ", XCLASSSystemRootDirectory)
    # print ("DatabaseFileName = ", DatabaseFileName)
    # print ("NoMAGIXFlag = ", NoMAGIXFlag)
    # print ("TimeStampFormat = ", TimeStampFormat)


    ## check path of XCLASS root directory
    if (XCLASSSystemRootDirectory == ""):
        print ("\n\n\nError in XCLASS package!")
        print ("\n\tCan not find the definition of the XCLASS root directory in")
        print ("\n\tthe xclass init file located in the user's home directory, i.e.")
        print ("\n\n\t\t$(HOME)/.xclass/init.dat")
        print ("\n\n\tPlease check your XCLASS installation!\n\n")
        sys.exit(0)


    ## check path of database file
    if (DatabaseFileName == ""):
        print ("\n\n\nError in XCLASS package!")
        print ("\n\tCan not find the definition of the XCLASS database file in")
        print ("\n\tthe xclass init file located in the user's home directory, i.e.")
        print ("\n\n\t\t$(HOME)/.xclass/init.dat")
        print ("\n\n\tPlease check your XCLASS installation!\n\n")
        sys.exit(0)


    ## define return parameter
    OutDict['XCLASSSystemRootDirectory'] = XCLASSSystemRootDirectory
    OutDict['DatabaseFileName'] = DatabaseFileName
    OutDict['NoMAGIXFlag'] = NoMAGIXFlag
    OutDict['TimeStampFormat'] = TimeStampFormat
    return OutDict
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get root path of XCLASS package
##
def GetXCLASSRootDir():
    """

input parameters:
-----------------

    - None


output parameters:
------------------

    - XCLASSSystemRootDirectory:        path of XCLASS package
"""


    ## get XCLASS root directory from init file
    OutDict = GetXCLASSSettings()
    XCLASSSystemRootDirectory = OutDict['XCLASSSystemRootDirectory']


    ## we're done
    return XCLASSSystemRootDirectory
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get path of myXCLASS program
##
def GetmyXCLASSRootDir():
    """

input parameters:
-----------------

    - None


output parameters:
------------------

    - myXCLASSRootDir:      path of myXCLASS program
"""


    ## get XCLASS root directory
    XCLASSRootDir = GetXCLASSRootDir()


    ## define path of myXCLASS program
    myXCLASSRootDir = XCLASSRootDir + "lib/"


    ## define return parameter
    return myXCLASSRootDir
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get path and name of default database file
##
def GetDefaultDBFile():
    """

input parameters:
-----------------

    - None


output parameters:
------------------

    - dbFileName:       path and name of default database file
"""


    ## get XCLASS root directory from init file
    OutDict = GetXCLASSSettings()
    DatabaseFileName = OutDict['DatabaseFileName']


    ## define return parameter
    return DatabaseFileName
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get absolute path of MAGIX directory
##
def GetMAGIXRootDir():
    """

input parameters:
-----------------

    - None


output parameters:
------------------

    - MAGIXRootDir:     absolute path of MAGIX directory
"""


    ## get XCLASS root directory
    XCLASSRootDir = GetXCLASSRootDir()


    ## define path of MAGIX program
    MAGIXRootDir = XCLASSRootDir + "addons/MAGIX/"


    ## define return parameter
    return MAGIXRootDir
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get command line arguments
##
def GetCommandLineArguments(ListOfOptions, NameOfScript, HelpText):
    """

input parameters:
-----------------

    - ListOfOptions:        list of command line options

    - NameOfScript:         name of script

    - HelpText:             text which is printed to screen


output parameters:
------------------

    - ListOptionValues:     list of corresponding user-defined option values
    """

    # Debug:
    # print ("ListOfOptions = ", ListOfOptions)
    # print ("NameOfScript = ", NameOfScript)
    # print ("HelpText = ", HelpText)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    ListOptionValues = []
    for LocalOption in ListOfOptions:
        ListOptionValues.append("")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get parameters from command line
    if (len(sys.argv) > 1):                                                                 ## check, if command line arguments are defined


        ## print help text to screen
        if (sys.argv[1:][0] == "--help"):
            print ("\n\n" + HelpText)
            print ("Start script with")
            print ("\n\t python3 " + NameOfScript + " ", end = ' ')
            for LocalOption in ListOfOptions:
                print ("--" + LocalOption + "= .. ", end = ' ')
            print ("\n\n")
            sys.exit(0)


        ## get command line options
        else:
            for CounterArgument, argument in enumerate(sys.argv[1:]):                       ## loop over all command line arguments


                ## get options
                if (argument.startswith('--')):                                             ## run flags are marked with "--" characters
                    OptionString = argument[2:].strip()                                     ## remove "--" characters and leading and tailing blanks
                    SplittedOptionString = OptionString.split("=")
                    if (len(SplittedOptionString) > 0):                                     ## check, if "=" is included
                        Option = SplittedOptionString[0].strip()                            ## get option
                        OptionValue = "=".join(SplittedOptionString[1:]).strip()            ## get corresponding value


                        ## get command line arguments
                        for LocalOptionID, LocalOption in enumerate(ListOfOptions):
                            if (Option.lower() == LocalOption):
                                ListOptionValues[LocalOptionID] = OptionValue

    # Debug:
    # print ("ListOptionValues = ", ListOptionValues)


    ## define return parameter
    return ListOptionValues
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## extend sys.path variable
##
def ExtendSysPath(ListOfNewPaths):
    """

input parameters:
-----------------

    - ListOfNewPaths:       list of new paths


output parameters:
------------------

    - None:
"""

    # Debug:
    # print ("ListOfNewPaths = ", ListOfNewPaths)


    ## extend sys.path variable
    for NewPath in ListOfNewPaths:
        NormalizedNewPath = os.path.normpath(NewPath)
        if (os.path.exists(NormalizedNewPath)):
            already_included_flag = False
            for entries in sys.path:
                if (entries == NormalizedNewPath):
                    already_included_flag = True
                    break
            if (not already_included_flag):
                sys.path.append(NormalizedNewPath)

    # Debug:
    # print ("sys.path = ", sys.path)
    # os.system("echo $PYTHONPATH")


    ## define return parameter
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check if path and name of given input file exists and copy input file to current working directory
##
def CheckNCopy(InputFileName, NameOfFunction, NameOfFile, CurrentDir, CurrentJobDir, printflag):
    """

input parameters:
-----------------

    - InputFileName:        path and name of input file

    - NameOfFunction:       name of the calling function

    - NameOfFile:           name of input file

    - CurrentDir:           current directory

    - CurrentJobDir:        current job directory

    - printflag:            print flag


output parameters:
------------------

    - NewInputFileName      new path and name of input file
    """

    # Debug:
    # print ("InputFileName = ", InputFileName)
    # print ("NameOfFunction = ", NameOfFunction)
    # print ("NameOfFile = ", NameOfFile)
    # print ("CurrentDir = ", CurrentDir)
    # print ("CurrentJobDir = ", CurrentJobDir)
    # print ("printflag = ", printflag)


    ## print what you do
    if (printflag):
        print ("Copy " + NameOfFile + " file to job directory " + chr(34) + CurrentJobDir + chr(34) + " .. ", end = "", flush = True)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    NewInputFileName = ""


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check, if variable 'InputFileName' is defined and not empty
    NameOfFile = NameOfFile.strip()
    NameOfFunction = NameOfFunction.strip()
    if ((InputFileName is None) or (InputFileName in ["none", "None"])):
        print ("\n\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
        print ("\tNo path and name of the {:s} file is not defined!".format(NameOfFile))
        print ("\n\tPlease enter an existing path and name of a {:s} file and redo {:s} function call!".format(NameOfFile, NameOfFunction))
        return (NewInputFileName)
    elif (InputFileName == ""):
        return (NewInputFileName)
    else:
        InputFileName = InputFileName.strip()
        if (InputFileName == ""):
            return (NewInputFileName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze path of input file


    ## make path of input file absolute path
    if (InputFileName[0] != "/"):
        InputFileName = CurrentDir + InputFileName


    ## does path and file exists?
    if not (os.path.exists(InputFileName) or os.path.isfile(InputFileName)):
        print ("\n\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
        print ("\tThe given path and name of the {:s} file {:s} does not exist!".format(NameOfFile, chr(34) + InputFileName + chr(34)))
        print ("\n\tPlease enter an existing path and name of a {:s} file and redo {:s} function call!".format(NameOfFile, NameOfFunction))
        return (NewInputFileName)


    ## get pure name of input file
    i = InputFileName.rfind("/")
    if (i > 0):
        PureInputFileName = InputFileName[i + 1:]
    else:
        PureInputFileName = InputFileName
    PureInputFileName = PureInputFileName.replace(" ", "_")                                 ## remove blanks in file name


    ## define new path and name of input file
    NewInputFileName = CurrentJobDir + PureInputFileName


    ## copy input file to temp directory
    # NormNCopyMove(InputFileName, NewInputFileName, copyFlag = True)
    if (os.path.exists(NewInputFileName)):
        NewInputFileName = InputFileName
    else:
        NormNCopyMove(InputFileName, NewInputFileName, copyFlag = True)


    ## we're done
    if (printflag):
        print ("done!")

    # Debug:
    # print ("NewInputFileName = ", NewInputFileName)


    ## define return parameter
    return (NewInputFileName)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## normalize path and copy / move files
##
def NormNCopyMove(SourceFileName, DestinationFileName, copyFlag = True):
    """

input parameters:
-----------------

    - SourceFileName:           path and name of source file or directory

    - DestinationFileName:      path and name of destination file

    - copyFlag:                 (optional) flag indicating if file is copied (default: True)


output parameters:
------------------

    - None
"""

    # Debug:
    # print ("\n\nSourceFileName = ", SourceFileName)
    # print ("DestinationFileName = ", DestinationFileName)
    # print ("copyFlag = ", copyFlag)


    ## copy or move
    if (copyFlag):
        cmdString = "cp "
    else:
        cmdString = "mv "


    ## normalize path of files
    SourceFileName = os.path.normpath(SourceFileName).strip()
    DestinationFileName = os.path.normpath(DestinationFileName).strip()


    ## if both file names are not identical, copy / move
    if (SourceFileName != DestinationFileName):
        cmdString += SourceFileName + " " + DestinationFileName
        os.system(cmdString)


    ## define return parameter
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert frequency for given velocity or redshift
## Ref.: http://www.atnf.csiro.au/people/Tobias.Westmeier/tools_hihelpers.php and https://www.iram.fr/IRAMFR/ARN/may95/node4.html
##
def ConvertFreq(freq0, v, z = 0.0, backTrafo = False, MethodConv = "radio"):
    """

input parameters:
-----------------

    - freq0:                original frequency (in MHz)

    - v:                    velocity (in km/s)

    - z:                    (optional) redshift

    - backTrafo:            (optional) flag indicating back transformation

    - MethodConv:           (optional) method of conversion ("radio" or "optical")


output parameters:
------------------

    - freq:                 new frequency (in MHz)
    """

    # Debug:
    # print ("freq0 = ", freq0)
    # print ("v = ", v)
    # print ("z = ", z)
    # print ("backTrafo = ", backTrafo)
    # print ("MethodConv = ", MethodConv)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    freq = freq0


    ## do nothing if redshift and velocity is zero
    if (z == 0.0 and v == 0.0):
        return freq


    ## check for not defined z
    if (z is None):
        z = 0.0


    ## apply backward transformation
    if (backTrafo):
        z = z * (-1.0)
        v = v * (-1.0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## compute redshift for z != 0
    ## => here we assume v == 0
    if (z != 0.0):


        ## use radio convention for given redshift z
        if (MethodConv == "radio"):
            freq = freq0 * (1.0 - (z / (1.0 + z)))


        ## use optical convention for given redshift z
        elif (MethodConv == "optical"):
            freq = freq0 * (1.0 / (1.0 + z))


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## compute Doppler shift for z == 0
    else:


        ## define constants
        cms = 299792458.0                                                                   ## speed of light in m/s
        ckms = cms * 1.e-3                                                                  ## speed of light in km/s


        ## use radio convention for given velocity v
        if (MethodConv == "radio"):
            freq = freq0 * (1.0 - v / ckms)


        ## use optical convention for given velocity v
        elif (MethodConv == "optical"):
            freq = freq0 * (1.0 / (1.0 + v / ckms))


    ## define return parameter
    return freq
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert frequency points (in MHz) into velocities (in km/s)
##
def ChangeToVelocitiy(FreqData, RestFreq, vLSR, conversion = "radio"):
    """

input parameters:
-----------------

    - FreqData:             observational data in MHz

    - RestFreq:             rest frequency in MHz (If this parameter is set to zero, the intensity is plotted against frequency (in MHz)
                            otherwise against velocity (in km/s).

    - vLSR:                 velocity (local standard of rest) in km/s, (velocity(Frequency = RestFreq) = vLSR)

    - conversion:           (optional) conversion used for translation ("radio", "optical", "rel"), (default: "radio")


output parameters:
------------------

    - VelData:              observational data in km/s
    """

    # Debug:
    # print ("FreqData = ", FreqData)
    # print ("RestFreq = ", RestFreq)
    # print ("vLSR = ", vLSR)
    # print ("conversion = ", conversion)


    ## initialize output parameter
    VelData = []


    ## define constants
    cms = 299792458.0
    ckms = cms * 1.e-3


    ## calculate new velocity scale
    ## expressions taken from https://docs.astropy.org/en/stable/units/equivalencies.html
    for freq in FreqData[:]:
        if (conversion == "radio"):
            vel = vLSR + ckms * (RestFreq - freq) / RestFreq
        elif (conversion == "optical"):
            vel = vLSR + ckms * (RestFreq - freq) / freq
        else:
            vel = vLSR + ckms * (RestFreq**2 - freq**2) / (RestFreq**2 + freq**2)
        VelData.append(vel)


    ## define return parameter
    return VelData
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get parameter limits from parameter fitflag (valid only for OLD format)
##
def GetParameterLimits(ParamName, ParameterValue, LimitValues):
    """

input parameters:
-----------------

    - ParamName:            name of parameter

    - ParameterValue:       value of parameter

    - LimitValues:          range of parameter (fitflag)


output parameters:
------------------

    - lowlimit:             lower limit of current parameter

    - uplimit:              upper limit of current parameter
    """

    # Debug:
    # print ("ParamName = ", ParamName)
    # print ("ParameterValue = ", ParameterValue)
    # print ("LimitValues = ", LimitValues)


    ## determine limits for current parameter
    if (ParamName in ["source_size", "source_radius", "N_tot", "EM_RRL", "nHcolumn_cont_dust"]):
        lowlimit = ParameterValue / LimitValues                 ## modification by Tom Bell
        if (lowlimit < 0):
            lowlimit = 0
        uplimit = ParameterValue * LimitValues                  ## modification by Tom Bell
    else:
        lowlimit = ParameterValue - LimitValues


        ## V_off, T_dOff, beta, ContFuncID_param_1, ContFuncID_param_2, ContFuncID_param_3, ContFuncID_param_4, ContFuncID_param_5
        ## could be negativ !!!
        if (lowlimit < 0 and (not ParamName in ["V_off", "T_dOff", "beta", "ContFuncID_param_1", "ContFuncID_param_2", "ContFuncID_param_3", \
           "ContFuncID_param_4", "ContFuncID_param_5"])):
            lowlimit = 0
        uplimit = ParameterValue + LimitValues

    # Debug:
    # print ("lowlimit = ", lowlimit)
    # print ("uplimit = ", uplimit)


    ## define return parameter
    return (lowlimit, uplimit)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## read in an analyze molfit file
##
def AnalyzeMolfitFile(MolfitsFile):
    """

input parameters:
-----------------

    - MolfitsFile:                  path and name of molfit file or content of molfit file


output parameters:
------------------

    - MoleculesInMolfitFile:        list of molecule names in molfit file

    - AllParameters:                list molfit parameters for each molecule

    - MolfitFileForEachMolecule:    molfit files for each molecule
    """

    # Debug:
    # print ("MolfitsFile = ", MolfitsFile)


    ## analyze molfit file
    if (type(MolfitsFile) is str):
        if (MolfitsFile.endswith(".gz")):
            f = gzip.open(MolfitsFile, 'rt')
        else:
            try:
                f = open(MolfitsFile)
            except:
                f = gzip.open(MolfitsFile + ".gz", 'rt')
        MolfitFileContent = f.readlines()
        f.close()
    else:
        MolfitFileContent = copy.deepcopy(MolfitsFile)


    ##====================================================================================================================================================
    ## Molfit file format line coding


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## pure molfit file


    ## dictionary of known options with corresponding parameter names
    ListOfKnownOptions = {"Circle":["source_size"], \
                          "CircleOff":["source_radius", "source_center_x", "source_center_y"], \
                          "Box":["source_size"], \
                          "BoxOff":["source_radius", "source_center_x", "source_center_y"], \
                          "MOLECULE":["T_rot", "N_tot"], \
                          "RRL":["T_e", "EM_RRL"], \
                          "Gauss":["V_width_Gauss"], \
                          "V-Off":["V_off"], \
                          "T-Dust-Inline":["T_dOff", "T_dSlope"], \
                          "nH-Dust-Inline":["nHcolumn", "kappa", "beta"], \
                          "CFFlag":["CFFlag"], \
                          "LayerDistance":["LayerDistance"], \
                          "Keyword":["keyword"], \
                          "DUST":["T_cont_dust", "nHcolumn_cont_dust", "kappa_cont_dust", "beta_cont_dust"], \
                          "PHEN":["ContFuncID_phen", "ContFuncID_param_1", "ContFuncID_param_2", "ContFuncID_param_3", "ContFuncID_param_4", \
                                  "ContFuncID_param_5"]}


    ## list of known shapes
    ListOfKnownShapes = ["", "Circle", "CircleOff", "Box", "BoxOff"]


    ## define a list of fitable parameters
    FitParameterList = ["source_size", "source_radius", "source_center_x", "source_center_y", \
                        "T_rot", "T_e", "T_cont_dust", "T_dOff", "T_dSlope", \
                        "N_tot", "EM_RRL", "nHcolumn_cont_dust", "nHcolumn", \
                        "V_width_Gauss", "V_off", \
                        "beta_cont_dust", "kappa_cont_dust", "kappa", "beta", \
                        "ContFuncID_param_1", "ContFuncID_param_2", "ContFuncID_param_3", "ContFuncID_param_4", "ContFuncID_param_5"]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## list of allowed combinations
    ListOfCombinationsNoShape = [\
                                 ##+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                 ## molecule


                                 ## Gaussian line profile
                                 "MOLECULE_Gauss_V-Off_CFFlag", \
                                 "MOLECULE_Gauss_V-Off_CFFlag_Keyword", \
                                 "MOLECULE_Gauss_V-Off_LayerDistance", \
                                 "MOLECULE_Gauss_V-Off_LayerDistance_Keyword", \
                                 "MOLECULE_Gauss_V-Off_T-Dust-Inline_CFFlag", \
                                 "MOLECULE_Gauss_V-Off_T-Dust-Inline_CFFlag_Keyword", \
                                 "MOLECULE_Gauss_V-Off_T-Dust-Inline_LayerDistance", \
                                 "MOLECULE_Gauss_V-Off_T-Dust-Inline_LayerDistance_Keyword", \
                                 "MOLECULE_Gauss_V-Off_nH-Dust-Inline_CFFlag", \
                                 "MOLECULE_Gauss_V-Off_nH-Dust-Inline_CFFlag_Keyword", \
                                 "MOLECULE_Gauss_V-Off_nH-Dust-Inline_LayerDistance", \
                                 "MOLECULE_Gauss_V-Off_nH-Dust-Inline_LayerDistance_Keyword", \
                                 "MOLECULE_Gauss_V-Off_T-Dust-Inline_nH-Dust-Inline_CFFlag", \
                                 "MOLECULE_Gauss_V-Off_T-Dust-Inline_nH-Dust-Inline_CFFlag_Keyword", \
                                 "MOLECULE_Gauss_V-Off_T-Dust-Inline_nH-Dust-Inline_LayerDistance", \
                                 "MOLECULE_Gauss_V-Off_T-Dust-Inline_nH-Dust-Inline_LayerDistance_Keyword", \


                                 ##+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                 ## RRL


                                 ## Gaussian line profile
                                 "RRL_Gauss_V-Off_CFFlag", \
                                 "RRL_Gauss_V-Off_CFFlag_Keyword", \
                                 "RRL_Gauss_V-Off_LayerDistance", \
                                 "RRL_Gauss_V-Off_LayerDistance_Keyword", \
                                 "RRL_Gauss_V-Off_T-Dust-Inline_CFFlag", \
                                 "RRL_Gauss_V-Off_T-Dust-Inline_CFFlag_Keyword", \
                                 "RRL_Gauss_V-Off_T-Dust-Inline_LayerDistance", \
                                 "RRL_Gauss_V-Off_T-Dust-Inline_LayerDistance_Keyword", \
                                 "RRL_Gauss_V-Off_nH-Dust-Inline_CFFlag", \
                                 "RRL_Gauss_V-Off_nH-Dust-Inline_CFFlag_Keyword", \
                                 "RRL_Gauss_V-Off_nH-Dust-Inline_LayerDistance", \
                                 "RRL_Gauss_V-Off_nH-Dust-Inline_LayerDistance_Keyword", \
                                 "RRL_Gauss_V-Off_T-Dust-Inline_nH-Dust-Inline_CFFlag", \
                                 "RRL_Gauss_V-Off_T-Dust-Inline_nH-Dust-Inline_CFFlag_Keyword", \
                                 "RRL_Gauss_V-Off_T-Dust-Inline_nH-Dust-Inline_LayerDistance", \
                                 "RRL_Gauss_V-Off_T-Dust-Inline_nH-Dust-Inline_LayerDistance_Keyword", \


                                 ##+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                 ## Dust continuum:


                                 "DUST_CFFlag", \
                                 "DUST_CFFlag_Keyword", \
                                 "DUST_LayerDistance", \
                                 "DUST_LayerDistance_Keyword", \


                                 ##+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                 ## phenomenological continuum description:


                                 "PHEN_CFFlag", \
                                 "PHEN_CFFlag_Keyword", \
                                 "PHEN_LayerDistance", \
                                 "PHEN_LayerDistance_Keyword"]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create final list of all allowed combinations
    ListOfAllowedCombinations = []
    for LocalShape in ListOfKnownShapes:
        if (LocalShape == ""):
            AddString = ""
        else:
            AddString = LocalShape + "_"
        for LocalCombination in ListOfCombinationsNoShape:
            ListOfAllowedCombinations.append(AddString + LocalCombination)

    # Debug:
    # print ("ListOfAllowedCombinations = ", ListOfAllowedCombinations)
    # sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create format specifications for one and four column molfit file format
    MolfitFileFormats = {}
    for item in ListOfAllowedCombinations:                                                  ## loop over all existing entries of the dictionary


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## create dictionary entry for pure format
        PureFormatString = "pure_" + item                                                   ## get format string
        PureFormatString = PureFormatString.lower()
        if (PureFormatString.find("_") > (-1)):
            StrippedFormatString = PureFormatString.split("_")
            StrippedFormatString.sort()
            PureFormatString = "_".join(StrippedFormatString)
        PureFormatString = PureFormatString.lower()


        ## create corresponding list of parameter names
        PureNameList = []
        SplittedItem = item.split("_")
        for element in SplittedItem:
            LocalList = ListOfKnownOptions[element]
            PureNameList = PureNameList + LocalList
        MolfitFileFormats[PureFormatString] = PureNameList

        # Debug:
        # print ("\n\nPureFormatString = ", PureFormatString)
        # print ("PureNameList = ", PureNameList)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## create dictionary entry for one column format
        OneColumnFormatString = PureFormatString.replace("pure", "old").lower()
        if (OneColumnFormatString.find("_") > (-1)):
            StrippedFormatString = OneColumnFormatString.split("_")
            StrippedFormatString.sort()
            OneColumnFormatString = "_".join(StrippedFormatString)
        OneColumnFormatString = OneColumnFormatString.lower()
        OneColumnList = []
        for element in PureNameList:
            if (element in  FitParameterList):
                OneColumnList.append(element + "_FitFlag")
            OneColumnList.append(element)
        MolfitFileFormats[OneColumnFormatString] = OneColumnList

        # Debug:
        # print ("MolfitFileFormats = ", MolfitFileFormats)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## create dictionary entry for four column format
        FourColumnFormatString = PureFormatString.replace("pure", "new").lower()
        if (FourColumnFormatString.find("_") > (-1)):
            StrippedFormatString = FourColumnFormatString.split("_")
            StrippedFormatString.sort()
            FourColumnFormatString = "_".join(StrippedFormatString)
        FourColumnFormatString = FourColumnFormatString.lower()
        FourColumnList = []
        for element in PureNameList:
            if (element in  FitParameterList):
                FourColumnList.append(element + "_flag")
                FourColumnList.append(element + "_lowlimit")
                FourColumnList.append(element + "_uplimit")
            FourColumnList.append(element)
        MolfitFileFormats[FourColumnFormatString] = FourColumnList

    # Debug:
    #    for item in MolfitFileFormats.items():
    #        print ("\n\n\n", item[0])
    #    sys.exit(0)


    ##====================================================================================================================================================
    ## define list of known keywords
    ListOfKnownKeyWords = ["nlte", "nonlte", "non-lte", "gauss", "lorentz", "voigt", "horn", "t-dust-inline", "nh-dust-inline", \
                           "circle", "circleoff", "box", "boxoff"]


    ##====================================================================================================================================================
    ## split final molfit file in N small molfit files containing only one molecule
    MoleculesInMolfitFile = []
    MolfitFileForEachMolecule = []
    CounterMolecules = 0
    CounterLines = (-1)
    LinesOfMolfitFile = []
    AllParameters = []
    ParametersPerMolecule = []
    CurrentMolecule = ""
    for line in MolfitFileContent:                                                          ## loop over all lines of the molfit file
        CounterLines += 1                                                                   ## increase counter for lines
        StrippedLine = line.strip()                                                         ## remove leading and trailing blanks


        ## remove comments
        i = StrippedLine.find("%")
        if (i > (-1)):
            StrippedLine = StrippedLine[:i].strip()


        ## analyze current line of molfit file
        if (StrippedLine != ""):


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## analyze line
            SplittedLine = StrippedLine.split()                                             ## split current line into columns
            NumberElementsLine = len(SplittedLine)                                          ## determine number of columns
            if (NumberElementsLine == 2):                                                   ## two columns current line: a molecule name is defined
                if (CounterMolecules > 0):
                    MolfitFileForEachMolecule.append(LinesOfMolfitFile)                     ## add molecule name to output list
                    AllParameters.append(ParametersPerMolecule)
                    ParametersPerMolecule = []
                CounterMolecules += 1                                                       ## increase counter for molecules
                LinesOfMolfitFile = []
                LinesOfMolfitFile.append(line)
                CurrentMolecule = SplittedLine[0]
                MoleculesInMolfitFile.append(CurrentMolecule)
            else:
                LinesOfMolfitFile.append(line)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## determine format of current line


                ## check, which kind of molecule
                NumElementsPure = 0                                                         ## number of elements for pure molfit line (myXCLASS)
                NumElementsOldFit = 0                                                       ## number of elements for old fit molfit line (myXCLASS)
                NumElementsNewFit = 0                                                       ## number of elements for new fit molfit line (myXCLASS)
                AdditionalCommandList = []                                                  ## list of additional commands
                LowerCurrentMolecule = CurrentMolecule.lower()
                if (LowerCurrentMolecule.startswith("rrl-")):                               ## rrl is used
                    LocalMolecule = "rrl"
                    CurrentMolfitFormat = "rrl_circle_gauss_v-off_layerdistance"
                    NumElementsPure = 6
                    NumElementsOldFit = 5 * 2 + 1
                    NumElementsNewFit = 5 * 4 + 1
                elif (LowerCurrentMolecule.startswith("cont-dust")):                        ## cont-dust
                    LocalMolecule = "dust"
                    CurrentMolfitFormat = "dust_circle_layerdistance"
                    NumElementsPure = 6
                    NumElementsOldFit = 5 * 2 + 1
                    NumElementsNewFit = 5 * 4 + 1
                elif (LowerCurrentMolecule.startswith("cont-phen")):                        ## cont-phen
                    LocalMolecule = "phen"
                    CurrentMolfitFormat = "phen_circle_layerdistance"
                    NumElementsPure = 8
                    NumElementsOldFit = 6 * 2 + 1 + 1
                    NumElementsNewFit = 6 * 4 + 1 + 1
                else:
                    LocalMolecule = "molecule"
                    CurrentMolfitFormat = "molecule_circle_gauss_v-off_layerdistance"
                    NumElementsPure = 6
                    NumElementsOldFit = 5 * 2 + 1
                    NumElementsNewFit = 5 * 4 + 1

                # Debug:
                # print ("LowerCurrentMolecule = ", LowerCurrentMolecule)
                # print ("CurrentMolfitFormat = ", CurrentMolfitFormat)
                # print ("SplittedLine = ", SplittedLine)


                ## analyze last element of current line and add CFFlag if necessary
                LastElement = SplittedLine[-1]                                              ## get the last element
                LastElement = LastElement.lower()
                SecondLastElement = SplittedLine[-2]                                        ## get the next to last element
                SecondLastElement = SecondLastElement.lower()


                ## check, if distance of cf-flag definition is used
                if (LastElement in ["a", "f", "e", "c"] or SecondLastElement in ["a", "f", "e", "c"]):
                    AdditionalCommandList.append(["layerdistance", "cfflag"])


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## check, if last element indicates a keyword, i.e. it is neither a number nor "a", "f", "e", and "c"
                try:
                    float(LastElement)
                    FloatFlag = True
                except ValueError:
                    FloatFlag = False
                if (not FloatFlag and (not LastElement in ["a", "f", "e", "c"])):
                    NumElementsPure += 1
                    NumElementsOldFit += 1
                    NumElementsNewFit += 1
                    AdditionalCommandList.append(["", "keyword"])


                    ## determine the number of additional columns where we assume a default configuration of "circle_molecule_gauss_v-off_layerdistance"
                    SplittedCurrentMolfitFormat = LastElement.split("_")
                    LastElement = SecondLastElement
                    for LocalElement in SplittedCurrentMolfitFormat:
                        LocalElement = LocalElement.lower()


                        ## circle off
                        if (LocalElement == "circleoff"):
                            NumAddParam = 2
                            NumElementsPure += NumAddParam
                            NumElementsOldFit += NumAddParam * 2
                            NumElementsNewFit += NumAddParam * 4
                            AdditionalCommandList.append(["circle", LocalElement])          ## default shape-keyword "circle" is replaced by keyword def.


                        ## box
                        elif (LocalElement == "box"):
                            AdditionalCommandList.append(["circle", LocalElement])          ## default shape-keyword "circle" is replaced by keyword def.


                        ## box off
                        elif (LocalElement == "boxoff"):
                            NumAddParam = 2
                            NumElementsPure += NumAddParam
                            NumElementsOldFit += NumAddParam * 2
                            NumElementsNewFit += NumAddParam * 4
                            AdditionalCommandList.append(["circle", LocalElement])          ## default shape-keyword "circle" is replaced by keyword def.


                        # ## other shape definitions
                        # elif (LocalElement == "??"):
                        #     NumAddParam = ???
                        #     NumElementsPure += NumAddParam
                        #     NumElementsOldFit += NumAddParam * 2
                        #     NumElementsNewFit += NumAddParam * 4
                        #     AdditionalCommandList.append(["circle", LocalElement])        ## default shape-keyword "circle" is replaced by keyword def.


                        ## check for special settings for molecules and RRLs
                        if (LocalMolecule in ["molecule", "rrl"]):


                            ## t-dust-inline
                            if (LocalElement == "t-dust-inline"):
                                NumElementsPure += 2
                                NumElementsOldFit += 2 * 2
                                NumElementsNewFit += 2 * 4
                                AdditionalCommandList.append(["", LocalElement])


                            ## nh-dust-inline
                            elif (LocalElement == "nh-dust-inline"):
                                NumElementsPure += 3
                                NumElementsOldFit += 3 * 2
                                NumElementsNewFit += 3 * 4
                                AdditionalCommandList.append(["", LocalElement])


                ## correct format string
                for element in AdditionalCommandList:
                    old = element[0]
                    new = element[1]
                    if (old == ""):
                        CurrentMolfitFormat += "_" + new
                    else:
                        CurrentMolfitFormat = CurrentMolfitFormat.replace(old, new)

                # Debug:
                # print ("\n\nNumElementsPure = ", NumElementsPure)
                # print ("NumElementsOldFit = ", NumElementsOldFit)
                # print ("NumElementsNewFit = ", NumElementsNewFit)
                # print ("NumberElementsLine = ", NumberElementsLine)
                # print ("AdditionalCommandList = ", AdditionalCommandList)
                # print ("CurrentMolfitFormat = ", CurrentMolfitFormat)
                # print ("StrippedLine = >>", StrippedLine, "<<")


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## determine format version (pure, old, new) of current line
                FirstElement = SplittedLine[0].strip()                                      ## get the last element
                FirstElement = FirstElement.lower()
                ColumnFormat = ""
                if (FirstElement in ["y", "n", "t", "f"]):
                    if (NumberElementsLine == NumElementsNewFit):                           ## no modifications
                        CurrentMolfitFormat += "_new"
                        ColumnFormat = "new"
                    elif (NumberElementsLine == NumElementsNewFit - 4):                     ## pure without circle (or other shape definitions)
                        for LocalShape in ListOfKnownShapes:
                            if (LocalShape != ""):
                                LocalShape = LocalShape + "_"
                                LocalShape = LocalShape.lower()
                                CurrentMolfitFormat = CurrentMolfitFormat.replace(LocalShape, "")
                        CurrentMolfitFormat += "_new"
                        ColumnFormat = "new"
                    #    elif (NumberElementsLine == NumElementsNewFit + 2 * 4):                 ## with t-dust-inline
                    #        CurrentMolfitFormat += "_new_t-dust-inline"
                    #        ColumnFormat = "new"
                    #    elif (NumberElementsLine == NumElementsNewFit + 3 * 4):                 ## with nh-dust-inline
                    #        CurrentMolfitFormat += "_new_nh-dust-inline"
                    #        ColumnFormat = "new"
                    #    elif (NumberElementsLine == NumElementsNewFit + 5 * 4):                 ## with t-dust-inline
                    #        CurrentMolfitFormat += "_new_t-dust-inline_nh-dust-inline"
                    #        ColumnFormat = "new"
                    ## missing without circle but with t-dust-inline
                    ## missing without circle but with nh-dust-inline
                    ## missing without circle but with t-dust-inline and nh-dust-inline

                    else:
                        print ("\n\n Error in XCLASS package, function task_myXCLASS.AnalyzeMolfitFile:")
                        print ("\t\t Can not identify the format of the current line")
                        print ("\t\t in molfit file " + MolfitsFile)
                        print ("\n\t\t MolfitFileFormatVersion = ", CurrentMolfitFormat)
                        print ("\t\t NumberElementsLine = ", NumberElementsLine)
                        print ("\t\t NumElementsOldFit = ", NumElementsOldFit)
                        print ("\t\t NumElementsNewFit = ", NumElementsNewFit)
                        print ("\t\t line number: " + str(CounterLines))
                        print ("\t\t line = >>" + line.replace("\n", "") + "<<")
                        print ("\n\n")
                        return (MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule)


                else:
                    if (NumberElementsLine == NumElementsPure):                             ## no modifications
                        CurrentMolfitFormat += "_pure"
                        ColumnFormat = "pure"
                    elif (NumberElementsLine == NumElementsPure - 1):                       ## pure without circle (or other shape definitions)
                        for LocalShape in ListOfKnownShapes:
                            if (LocalShape != ""):
                                LocalShape = LocalShape + "_"
                                LocalShape = LocalShape.lower()
                                CurrentMolfitFormat = CurrentMolfitFormat.replace(LocalShape, "")
                        CurrentMolfitFormat += "_pure"
                        ColumnFormat = "pure"
                    #    elif (NumberElementsLine == NumElementsPure + 2):                       ## with t-dust-inline
                    #        CurrentMolfitFormat += "_pure_t-dust-inline"
                    #        ColumnFormat = "pure"
                    #    elif (NumberElementsLine == NumElementsPure + 3):                       ## with nh-dust-inline
                    #        CurrentMolfitFormat += "_pure_nh-dust-inline"
                    #        ColumnFormat = "pure"
                    #    elif (NumberElementsLine == NumElementsPure + 5):                       ## with t-dust-inline and nh-dust-inline
                    #        CurrentMolfitFormat += "_pure_t-dust-inline_nh-dust-inline"
                    #        ColumnFormat = "pure"
                    ## missing without circle but with t-dust-inline
                    ## missing without circle but with nh-dust-inline
                    ## missing without circle but with t-dust-inline and nh-dust-inline


                    elif (NumberElementsLine == NumElementsOldFit):                         ## no modifications
                        CurrentMolfitFormat += "_old"
                        ColumnFormat = "old"
                    elif (NumberElementsLine == NumElementsOldFit - 2):                     ## pure without circle (or other shape definitions)
                        for LocalShape in ListOfKnownShapes:
                            if (LocalShape != ""):
                                LocalShape = LocalShape + "_"
                                LocalShape = LocalShape.lower()
                                CurrentMolfitFormat = CurrentMolfitFormat.replace(LocalShape, "")
                        CurrentMolfitFormat += "_old"
                        ColumnFormat = "old"
                    #    elif (NumberElementsLine == NumElementsOldFit + 2 * 2):                 ## with t-dust-inline
                    #        CurrentMolfitFormat += "_old_t-dust-inline"
                    #        ColumnFormat = "old"
                    #    elif (NumberElementsLine == NumElementsOldFit + 3 * 2):                 ## with nh-dust-inline
                    #        CurrentMolfitFormat += "_old_nh-dust-inline"
                    #        ColumnFormat = "old"
                    #    elif (NumberElementsLine == NumElementsOldFit + 5 * 2):                 ## with t-dust-inline and nh-dust-inline
                    #        CurrentMolfitFormat += "_old_t-dust-inline_nh-dust-inline"
                    #        ColumnFormat = "old"
                    ## missing without circle but with t-dust-inline
                    ## missing without circle but with nh-dust-inline
                    ## missing without circle but with t-dust-inline and nh-dust-inline


                    else:
                        print ("\n\n Error in XCLASS package, function task_myXCLASS.AnalyzeMolfitFile:")
                        print ("\t\t Can not identify the format of the current line")
                        print ("\t\t in molfit file " + MolfitsFile)
                        print ("\n\t\t MolfitFileFormatVersion = ", CurrentMolfitFormat)
                        print ("\t\t NumberElementsLine = ", NumberElementsLine)
                        print ("\t\t NumElementsOldFit = ", NumElementsOldFit)
                        print ("\t\t NumElementsNewFit = ", NumElementsNewFit)
                        print ("\t\t line number: " + str(CounterLines))
                        print ("\t\t line = >>" + line.replace("\n", "") + "<<")
                        print ("\n\n")
                        return (MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule)

                # Debug:
                # print ("ColumnFormat = ", ColumnFormat)
                # print (">>>CurrentMolfitFormat = ", CurrentMolfitFormat)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## sort elements of format string
                CurrentMolfitFormat = CurrentMolfitFormat.lower()
                if (CurrentMolfitFormat.find("_") > (-1)):
                    StrippedFormatString = CurrentMolfitFormat.split("_")
                    StrippedFormatString.sort()
                    CurrentMolfitFormat = "_".join(StrippedFormatString)
                MolfitFileFormatVersion = CurrentMolfitFormat

                # Debug:
                # print ("####>MolfitFileFormatVersion = ", MolfitFileFormatVersion)


                ## add molfit file format version to output list
                ParameterPerLineList = []
                entry = ["MolfitFileFormatVersion", 0, 0, 0, MolfitFileFormatVersion, 0]
                ParameterPerLineList.append(entry)


                ## store
                for counter, element in enumerate(SplittedLine):                            ## loop over all elements of the current line

                    # Debug:
                    # print ("counter, element = ", counter, element)
                    # print ("MolfitFileFormats[MolfitFileFormatVersion] = ", MolfitFileFormats[MolfitFileFormatVersion])


                    ## get name of parameter which correspond to the current column
                    try:
                        ParamName = MolfitFileFormats[MolfitFileFormatVersion][counter]
                    except:
                        print ("\n\n Error in XCLASS package, function task_myXCLASS.AnalyzeMolfitFile:")
                        print ("\t\t Can not identify the format of the current line")
                        print ("\t\t in molfit file " + MolfitsFile)
                        print ("\n\t\t MolfitFileFormatVersion = ", MolfitFileFormatVersion)
                        print ("\t\t counter = ", counter)
                        print ("\t\t element = ", element)
                        print ("\t\t line number: " + str(CounterLines))
                        print ("\t\t line = >>" + line.replace("\n", "") + "<<")
                        print ("\n\nAbort!")
                        print ("MolfitFileFormats = ", MolfitFileFormats)
                        sys.exit(0)
                        # return (MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule)

                    # Debug:
                    # print ("ParamName = ", ParamName)


                    ## initialize parameter
                    flag = "0"
                    lowlimit = 0.0
                    uplimit = 0.0


                    ## special treatment of keyword and CFFlag
                    if (ParamName in ["ContFuncID_phen", "LayerDistance", "keyword", "CFFlag"]):
                        element = "  " + element
                        flag = "0"


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## determine lower and upper limits for the old format
                    if (ColumnFormat == "pure"):                                            ## line is in PURE format
                        if (not ParamName in ["ContFuncID_phen", "LayerDistance", "keyword", "CFFlag"]):
                                                                                            ## determine lower and upper limits except for e/a flag
                            ParameterValue = (float(element))                               ## get parameter value


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## determine lower and upper limits for the old format
                    elif (ColumnFormat == "old"):                                           ## line is in old format
                        if (ParamName.endswith("_FitFlag")):                                ## get limit range definition
                            LimitValues = abs(float(element))
                            flag = "0"
                            lowlimit = 0.0
                            uplimit = 1.0
                        elif (not ParamName in ["ContFuncID_phen", "LayerDistance", "keyword", "CFFlag"]):
                                                                                            ## determine lower and upper limits except for e/a flag
                            ParameterValue = (float(element))                               ## get parameter value
                            if (LimitValues > 0.0):                                         ## a limit specification unequal zero indicates a free param.
                                flag = "1"
                                lowlimit, uplimit = GetParameterLimits(ParamName, ParameterValue, LimitValues)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## determine lower and upper limits for the old format
                    elif (ColumnFormat == "new"):                                           ## current line is in new format
                        if (ParamName.endswith("_flag")):                                   ## get flag defining if a parameter is fitted or not
                            flag = element
                            if (flag.find("y") > (-1) or flag.find("t") > (-1)):
                                flag = "1"
                            else:
                                flag = "0"
                        elif (ParamName.endswith("_uplimit")):                              ## get upper limit for parameter
                            ParamUplimit = float(element)
                            lowlimit = 0.0
                            uplimit = 1.0
                        elif (ParamName.endswith("_lowlimit")):                             ## get lower limit for parameter
                            ParamLowlimit = float(element)
                            lowlimit = 0.0
                            uplimit = 1.0
                        elif (not ParamName in ["ContFuncID_phen", "LayerDistance", "keyword", "CFFlag"]):
                            lowlimit = ParamLowlimit                                        ## determine lower and upper limits except for CFFlag
                            uplimit = ParamUplimit


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## add parameter name, limits and value to output list
                    entry = [ParamName, flag, lowlimit, uplimit, element, counter]
                    ParameterPerLineList.append(entry)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## add parameters of current line to ParametersPerMolecule array
                ParametersPerMolecule.append(ParameterPerLineList)


    ## add parameter list for last line to global list
    if (CounterMolecules > 0):
        MolfitFileForEachMolecule.append(LinesOfMolfitFile)
        AllParameters.append(ParametersPerMolecule)

    # Debug:
    # print ("MoleculesInMolfitFile = ", MoleculesInMolfitFile)
    # print ("AllParameters = ", AllParameters)
    # print ("MolfitFileForEachMolecule = ", MolfitFileForEachMolecule)
    # sys.exit(0)


    ## define return parameter
    return (MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get parameters from obs. xml file
##
def GetObsXMLFileParameters(ObsDataID, RangeID, ObsDataFileNameListIn = None, TelescopeSizeListIn = None, BMIN_ListIn = None, BMAJ_ListIn = None, \
                            BPA_ListIn = None, GlobalvLSRListIn = None, Redshift_ListIn = None, InterFlagListIn = None, ErrorYFlagListIn = None, \
                            NumberHeaderLinesListIn = None, SeparatorColumnsListIn = None, NumberRangeListIn = None, FreqMinListIn = None, \
                            FreqMaxListIn = None, FreqStepListIn = None, tBackFlagListIn = None, tBackListIn = None, tSlopeListIn = None, \
                            nHFlagListIn = None, N_HListIn = None, beta_dustListIn = None, kappa_1300ListIn = None, DustFileNameListIn = None, \
                            BackgroundFileNameListIn = None, ContPhenFuncID_ListIn = None, ContPhenFuncParam1_ListIn = None, \
                            ContPhenFuncParam2_ListIn = None, ContPhenFuncParam3_ListIn = None, ContPhenFuncParam4_ListIn = None, \
                            ContPhenFuncParam5_ListIn = None, NoiseListIn = None, ThresholdListIn = None, SmoothValueListIn = None):
    """

input parameters:
-----------------

    - ObsDataID:                    obs. data file index

    - RangeID:                      range index

    - ObsDataFileNameListIn:        list of obs. data file names

    - TelescopeSizeListIn:          list of telescope sizes

    - BMIN_ListIn:                  list of telescope BMIN

    - BMAJ_ListIn:                  list of telescope BMAJ

    - BPA_ListIn:                   list of telescope BPA

    - GlobalvLSRListIn:             list of global v_lsr

    - Redshift_ListIn:              list of red shifts

    - InterFlagListIn:              list of inter flag

    - ErrorYFlagListIn:             list of error-y flags

    - NumberHeaderLinesListIn:      list of number of header lines

    - SeparatorColumnsListIn:       list of separator characters

    - NumberRangeListIn:            list of range numbers

    - FreqMinListIn:                list of min. frequency

    - FreqMaxListIn:                list of max. frequency

    - FreqStepListIn:               list of step frequency

    - tBackFlagListIn:              list of t_Back flags

    - tBackListIn:                  list of background temperature

    - tSlopeListIn:                 list of temperature slope

    - nHFlagListIn:                 list of nH flags

    - N_HListIn:                    list of nH column densities

    - beta_dustListIn:              list of betas

    - kappa_1300ListIn:             list of kappas

    - DustFileNameListIn:           list of dust file names

    - BackgroundFileNameListIn:     list of background file names

    - ContPhenFuncID_ListIn:        list of function IDs for phenomenological continuum description

    - ContPhenFuncParam1_ListIn:    list of first parameters for phenomenological continuum description

    - ContPhenFuncParam2_ListIn:    list of second parameters for phenomenological continuum description

    - ContPhenFuncParam3_ListIn:    list of third parameters for phenomenological continuum description

    - ContPhenFuncParam4_ListIn:    list of fourth parameters for phenomenological continuum description

    - ContPhenFuncParam5_ListIn:    list of fifth parameters for phenomenological continuum description

    - NoiseListIn:                  list of noise levels

    - ThresholdListIn:              list of thresholds

    - SmoothValueListIn:            list of smooth values



output parameters:
------------------

    - ObsDataFileName:              path and name of obs. data file

    - TelescopeSize:                size of telescope (beam)

    - BMIN:                         telescope BMIN

    - BMAJ:                         telescope BMAJ

    - BPA:                          telescope BPA

    - GlobalvLSR:                   global v_lsr

    - Redshift:                     red shifts

    - InterFlag:                    interferometer flag

    - ErrorY:                       error-y flag

    - NumberHeaderLines:            number of header lines

    - SeparatorColumns:             character dividing columns

    - FreqMin:                      min. frequency

    - FreqMax:                      max. frequency

    - FreqStep:                     step frequency

    - t_back_flag:                  flag for global background definition

    - tBack:                        background temperature

    - tSlope:                       temperature slope

    - nH_flag:                      flag for global dust definition

    - N_H:                          hydrogen column density

    - beta_dust:                    spectral index

    - kappa_1300:                   kappa

    - DustFileName:                 path and name of dust file

    - BackgroundFileName:           path and name of background file

    - NoiseLevel:                   noise level

    - Threshold:                    threshold

    - SmoothValue:                  smooth value
    """

    # Debug:
    # print ("ObsDataID = ", ObsDataID)
    # print ("RangeID = ", RangeID)
    # print ("ObsDataFileNameListIn = ", ObsDataFileNameListIn)
    # print ("TelescopeSizeListIn = ", TelescopeSizeListIn)
    # print ("BMIN_ListIn = ", BMIN_ListIn)
    # print ("BMAJ_ListIn = ", BMAJ_ListIn)
    # print ("BPA_ListIn = ", BPA_ListIn)
    # print ("GlobalvLSRListIn = ", GlobalvLSRListIn)
    # print ("Redshift_ListIn = ", Redshift_ListIn)
    # print ("InterFlagListIn = ", InterFlagListIn)
    # print ("ErrorYFlagListIn = ", ErrorYFlagListIn)
    # print ("NumberHeaderLinesListIn = ", NumberHeaderLinesListIn)
    # print ("SeparatorColumnsListIn = ", SeparatorColumnsListIn)
    # print ("NumberRangeListIn = ", NumberRangeListIn)
    # print ("FreqMinListIn = ", FreqMinListIn)
    # print ("FreqMaxListIn = ", FreqMaxListIn)
    # print ("FreqStepListIn = ", FreqStepListIn)
    # print ("tBackFlagListIn = ", tBackFlagListIn)
    # print ("tBackListIn = ", tBackListIn)
    # print ("tslopeListIn = ", tslopeListIn)
    # print ("nHFlagListIn = ", nHFlagListIn)
    # print ("N_HListIn = ", N_HListIn)
    # print ("beta_dustListIn = ", beta_dustListIn)
    # print ("kappa_1300ListIn = ", kappa_1300ListIn)
    # print ("DustFileNameListIn = ", DustFileNameListIn)
    # print ("BackgroundFileNameListIn = ", BackgroundFileNameListIn)
    # print ("ContPhenFuncID_ListIn = ", ContPhenFuncID_ListIn)
    # print ("ContPhenFuncParam1_ListIn = ", ContPhenFuncParam1_ListIn)
    # print ("ContPhenFuncParam2_ListIn = ", ContPhenFuncParam2_ListIn)
    # print ("ContPhenFuncParam3_ListIn = ", ContPhenFuncParam3_ListIn)
    # print ("ContPhenFuncParam4_ListIn = ", ContPhenFuncParam4_ListIn)
    # print ("ContPhenFuncParam5_ListIn = ", ContPhenFuncParam5_ListIn)
    # print ("NoiseListIn = ", NoiseListIn)
    # print ("ThresholdListIn = ", ThresholdListIn)
    # print ("SmoothValueListIn = ", SmoothValueListIn)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameters
    ObsXMLParameterDict = {}


    ## define internal list used for analysis
    ##           "Name of input list:"          "Name in dictionary:"       "default value:"     "Format:"
    AllLists = [[ObsDataFileNameListIn,         "ObsDataFileName",          "",                  "s"], \
                [TelescopeSizeListIn,           "TelescopeSize",            0.0,                 "f"], \
                [BMIN_ListIn,                   "BMIN",                     None,                "f"], \
                [BMAJ_ListIn,                   "BMAJ",                     None,                "f"], \
                [BPA_ListIn,                    "BPA",                      None,                "f"], \
                [Redshift_ListIn,               "Redshift",                 None,                "f"], \
                [GlobalvLSRListIn,              "GlobalvLSR",               0.0,                 "f"], \
                [InterFlagListIn,               "InterFlag",                False,               "b"], \
                [ErrorYFlagListIn,              "ErrorY",                   False,               "b"], \
                [NumberHeaderLinesListIn,       "NumberHeaderLines",        0,                   "i"], \
                [SeparatorColumnsListIn,        "SeparatorColumns",         "",                  "s"], \
                [NumberRangeListIn,             "NumberFrequencyRanges",    1,                   "i"], \
                [FreqMinListIn,                 "FreqMin",                  None,                "f"], \
                [FreqMaxListIn,                 "FreqMax",                  None,                "f"], \
                [FreqStepListIn,                "FreqStep",                 None,                "f"], \
                [tBackFlagListIn,               "t_back_flag",              True,                "b"], \
                [tBackListIn,                   "tBack",                    None,                "f"], \
                [tSlopeListIn,                  "tSlope",                   None,                "f"], \
                [nHFlagListIn,                  "nH_flag",                  False,               "b"], \
                [N_HListIn,                     "N_H",                      None,                "f"], \
                [beta_dustListIn,               "beta_dust",                None,                "f"], \
                [kappa_1300ListIn,              "kappa_1300",               None,                "f"], \
                [DustFileNameListIn,            "DustFileName",             None,                "f"], \
                [BackgroundFileNameListIn,      "BackgroundFileName",       None,                "f"], \
                [ContPhenFuncID_ListIn,         "ContPhenFuncID",           None,                "f"], \
                [ContPhenFuncParam1_ListIn,     "ContPhenFuncParam1",       None,                "f"], \
                [ContPhenFuncParam2_ListIn,     "ContPhenFuncParam2",       None,                "f"], \
                [ContPhenFuncParam3_ListIn,     "ContPhenFuncParam3",       None,                "f"], \
                [ContPhenFuncParam4_ListIn,     "ContPhenFuncParam4",       None,                "f"], \
                [ContPhenFuncParam5_ListIn,     "ContPhenFuncParam5",       None,                "f"], \
                [NoiseListIn,                   "NoiseLevel",               None,                "f"], \
                [ThresholdListIn,               "Threshold",                None,                "f"], \
                [SmoothValueListIn,             "SmoothValue",              None,                "f"]]


    ## analyze all lists
    for LocalBlock in AllLists:                                                             ## loop over all lists
        LocalList = LocalBlock[0]                                                           ## get local list
        if (LocalList is not None):                                                             ## is local list unequal None?
            LocalName = LocalBlock[1]                                                       ## get name current variable for dictionary
            NewValue = LocalBlock[2]                                                        ## get default value
            LocalFormat = LocalBlock[3]                                                     ## get format
            for LocalElement in LocalList:                                                  ## loop over all elements of list
                if (len(LocalElement) == 2):
                    if (LocalElement[0] == ObsDataID):                                      ## does the obs. data file ID match?
                        NewValue = LocalElement[1]
                        break
                elif (len(LocalElement) == 3):
                    if (LocalElement[0] == ObsDataID and LocalElement[1] == RangeID):       ## does the frequency range ID match?
                        NewValue = LocalElement[2]
                        break


            ## apply format specification
            if (LocalFormat == "i"):
                try:
                    NewValue = int(NewValue)
                except:
                    pass
            elif (LocalFormat == "f"):
                try:
                    NewValue = float(NewValue)
                except:
                    pass


            ## store value to output dictionary
            ObsXMLParameterDict[LocalName] = NewValue

    # Debug:
    # print ("ObsXMLParameterDict = ", ObsXMLParameterDict)


    ## define return parameter
    return ObsXMLParameterDict
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert parameter value from linear to log value and vise versa
##
def ConvertParamValue(LogLinearFlag, OrigValue):
    """

input parameters:
-----------------

    - LogLinearFlag:        flag indicating linear or log conversion

    - OrigValue:            original parameter value



output parameters:
------------------

    - NewParamValue:        converted parameter value
    """

    # Debug:
    # print ("LogLinearFlag = ", LogLinearFlag)
    # print ("OrigValue = ", OrigValue)


    ## initialize return parameter
    NewParamValue = 0.0


    ## to from linear to log conversion
    if (LogLinearFlag == "log"):
        if (OrigValue == 0.0):
            NewParamValue = numpy.log10(1.e-300)
        else:
            if (OrigValue < 0.E0):
                OrigValue = abs(OrigValue)
            NewParamValue = numpy.log10(OrigValue)
            if (NewParamValue > 307.0):                                                     ## check for overflow
                NewParamValue = 307.0


    ## to from log to linear conversion
    elif (LogLinearFlag == "linear"):
        NewParamValue = 10.E0**OrigValue

    # Debug:
    # print ("NewParamValue = ", NewParamValue)


    ## define return value
    return NewParamValue
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert column and hydrogen column density (if given for each component) to log scale
##
def ConvertNtotToLogScale(MolfitsFileName, LogLinearFlag, ConverSQLElow = False):
    """

input parameters:
-----------------

    - MolfitsFileName:      path and name of the molfit file located in the job directory

    - LogLinearFlag:        indicates, if column density values have to be converted from linear to log. scale (=log) or from log t linear (linear)

    - ConverSQLElow:        (optional) flag indicating that Elow_max has to be converted back


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("LogLinearFlag = ", LogLinearFlag)
    # print ("ConverSQLElow = ", ConverSQLElow)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze molfit file using myXCLASS routine


    ## get sql parameters from molfit file
    MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = GetSQLParameter(MolfitsFileName)
    if (ConverSQLElow):
        MaxElowSQL = MaxElowSQL * 1.42879
    SQLParamArray = [MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL]

    # Debug:
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("SQLParamArray = ", SQLParamArray)


    ## analyze given molfit file
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = AnalyzeMolfitFile(MolfitsFileName)

    # Debug:
    # print ("MoleculesInMolfitFile = ", MoleculesInMolfitFile)


    ## write new molfit file
    NEWFormatFlag = ""
    NewMolfitFile = WriteMolfitFile(MolfitsFileName, MoleculesInMolfitFile, AllParameters, LogLinearFlag, NEWFormatFlag, SQLParamArray)


    ## define return value ?
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get parameters for free-free description from the RRL components
##
def GetFreeFreeParameters(MolfitFileName = None, MoleculesInMolfitFile = None, AllParameters = None):
    """

input parameters:
-----------------

    - MolfitFileName:           (optional) path and name of molfit file, (default: None)

    - MoleculesInMolfitFile:    (optional) list of molecule names used in the moflit file, (default: None)

    - AllParameters:            (optional) list molfit parameters for each molecule, (default: None)



output parameters:
------------------

    - MolfitLines:              lines for a new molfit file

    - FreeFreeParamDict:        list of dictionaries with free-free parameters
    """

    # Debug:
    # print ("MolfitFileName = ", MolfitFileName)
    # print ("AllParameters = ", AllParameters)
    # print ("MoleculesInMolfitFile = ", MoleculesInMolfitFile)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    MolfitLines = []
    FreeFreeParamDict = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    if (MolfitFileName is not None):
        if (MoleculesInMolfitFile is None or AllParameters is None):
            MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = AnalyzeMolfitFile(MolfitFileName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## extract RRL parameters
    if (MoleculesInMolfitFile is not None and AllParameters is not None):


        ## loop over all molecules / RRLs in the molfit file
        for LocalMoleculeIndex, LocalMolecule in enumerate(MoleculesInMolfitFile):
            LowerLocalMolecule = LocalMolecule.lower()


            ## extract parameter from already existing free-free continuum
            if (LowerLocalMolecule.startswith("cont-") or LowerLocalMolecule.startswith("rrl-")):


                ## get parameters
                LocalMolfitParameters = AllParameters[LocalMoleculeIndex]


                ## add molecule parameters
                NewLine = ""
                LocalParamList = {}
                for LocalLine in LocalMolfitParameters:

                    # Debug:
                    # print ("LocalLine = ", LocalLine)

                    LocalvOff = 0.0
                    NewComp = " "
                    for col in LocalLine:
                        Name = col[0]
                        flag = col[1]
                        element = col[4]

                        # Debug:
                        # print ("col = ", col)
                        # print ("Name = ", Name)


                        ## get parameter values describing the source size
                        if (Name in ["source_size", "source_radius", "source_center_x", "source_center_y"]):
                            value = float(element)
                            NewComp += "  {:12.5e}".format(value)
                            LocalParamList[Name] = value


                        ## get parameter values describing the electron temperature
                        elif (Name in ["T_e"]):
                            value = float(element)
                            NewComp += "  {:12.5e}".format(value)
                            LocalParamList[Name] = value


                        ## get parameter values describing the emission measure
                        elif (Name in ["EM_RRL"]):
                            value = float(element)
                            NewComp += "  {:12.5e}".format(value)
                            LocalParamList[Name] = value


                        ## get parameter values describing the stacking
                        elif (Name in ["LayerDistance", "CFFlag"]):
                            NewComp += "    " + element
                            LocalParamList[Name] = value


                        ## get keywords and remove all irrelevant commands
                        elif (Name in ["keyword"]):
                            LocalSplittedElement = element.split("_")
                            CleanedKeyword = []
                            for LocalKeyword in LocalSplittedElement:
                                if (LocalKeyword in ["nlte", "nonlte", "non-lte", "circle", "circleoff", "box", "boxoff"]):
                                    CleanedKeyword.append(LocalKeyword)
                            CleanedKeyword = "_".join(CleanedKeyword)
                            NewComp += "    " + CleanedKeyword
                            LocalParamList[Name] = CleanedKeyword


                    ## add line break
                    NewLine += NewComp + "\n"
                    MolfitLines.append(NewLine)
                    FreeFreeParamDict.append(LocalParamList)


    ## prepend name of free-free continuum
    NewLine = "CONT-FREE-FREE   {:d}\n".format(len(MolfitLines))
    MolfitLines = [NewLine] + MolfitLines


    ## define return values
    return MolfitLines, FreeFreeParamDict
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## determine unit of a myXCLASS parameter
##
def GetUnitParameter(NameParam):
    """

input parameters:
-----------------

    - NameParam:            name of myXCLASS parameter



output parameters:
------------------

    - UnitParm:             unit of selected myXCLASS parameter
    """

    # Debug:
    # print ("NameParam = ", NameParam)


    ## define unit for each parameter
    if (NameParam in ["source_size", "source_radius", "source_center_x", "source_center_y"]):
        UnitParm = "arcsec"
    elif (NameParam in ["T_rot", "T_e", "T_dOff", "T_cont_dust" , "T_Back"]):
        UnitParm = "K"
    elif (NameParam in ["N_tot", "nHcolumn_cont_dust", "nHcolumn"]):
        UnitParm = "cm^(-2)"
    elif (NameParam in ["EM_RRL"]):
        UnitParm = "cm^(-6) pc"
    elif (NameParam in ["V_off"]):
        UnitParm = "km/s"
    elif (NameParam in ["T_Slope", "T_dSlope", "beta_cont_dust", "beta", "ContFuncID_phen", "CFFlag", "keyword"]):
        UnitParm = " "
    elif (NameParam in ["kappa"]):
        UnitParm = "cm^2/g"
    elif (NameParam in ["LayerDistance"]):
        UnitParm = "cm"
    else:
        UnitParm = ""

    # Debug:
    # print ("UnitParm = ", UnitParm)


    ## define return value ?
    return UnitParm
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## determine Latex name of myXCLASS parameter
##
def GetLatexParameterName(NameParam):
    """

input parameters:
-----------------

    - NameParam:            name of myXCLASS parameter



output parameters:
------------------

    - LatexNameParam:       Latex name of myXCLASS parameter
    """

    # Debug:
    # print ("NameParam = ", NameParam)


    ## get name of source size parameters
    if (NameParam in ["source_size", "source_radius"]):
        LatexNameParam = NameParam.replace("_", " ") + " (arcsec)"
    elif (NameParam in ["source_center_x"]):
        LatexNameParam = "source center$_x$ (arcsec)"
    elif (NameParam in ["source_center_y"]):
        LatexNameParam = "source center$_y$ (arcsec)"


    ## get name of temperature parameters
    elif (NameParam in ["T_rot", "T_e", "T_Back"]):
        LatexNameParam = NameParam.replace("_", "$_{") + "}$ (K)"
    elif (NameParam in ["T_Slope"]):
        LatexNameParam = "T$_{Slope}$"
    elif (NameParam in ["T_cont_dust"]):
        LatexNameParam = "T$_{cont}^{dust}$ (K)"
    elif (NameParam in ["T_dOff"]):
        LatexNameParam = "T$_{off}^{dust}$ (K)"
    elif (NameParam in ["T_dSlope"]):
        LatexNameParam = "T$_{Slope}^{dust}$ (K)"


    ## get name of column density parameters
    elif (NameParam in ["N_tot"]):
        LatexNameParam = "N$_{tot}$ (cm$^{-2}$)"
    elif (NameParam in ["nHcolumn"]):
        LatexNameParam = "N$_{H}$ (cm$^{-2}$)"
    elif (NameParam in ["nHcolumn_cont_dust"]):
        LatexNameParam = "N$_{H, cont}^{dust}$ (cm$^{-2}$)"
    elif (NameParam in ["EM_RRL"]):
        LatexNameParam = "EM$^{RRL}$ (cm$^{-6}$ pc)"


    ## get name of line width parameters
    elif (NameParam in ["V_width_Gauss"]):
        LatexNameParam = "v$_{width}^{Gauss}$ (km s$^{-1}$)"


    ## get name of velocity parameter
    elif (NameParam in ["V_off"]):
        LatexNameParam = "v$_{off}$ (km s$^{-1}$)"


    ## get name of other dust parameters
    elif (NameParam in ["beta"]):
        LatexNameParam = "$\\beta$"
    elif (NameParam in ["beta_cont_dust"]):
        LatexNameParam = "$\\beta_{cont}^{dust}$"
    elif (NameParam in ["kappa"]):
        LatexNameParam = "$\\kappa$ (cm$^{2}$ g$^{-1}$)"
    elif (NameParam in ["kappa_cont_dust"]):
        LatexNameParam = "$\\kappa_{cont}^{dust}$ (cm$^{2}$ g$^{-1}$)"


    ## get name of other parameters
    elif (NameParam in ["ContFuncID_phen", "CFFlag", "keyword"]):
        LatexNameParam = NameParam
    elif (NameParam in ["LayerDistance"]):
        LatexNameParam = NameParam + " (cm)"
    else:
        LatexNameParam = ""

    # Debug:
    # print ("LatexNameParam = ", LatexNameParam)


    ## we're done
    return LatexNameParam
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## write a line describing component to molfit file
##
def WriteLineMolfitFile(line):
    """

input parameters:
-----------------

    - line:             line with parameters



output parameters:
------------------

    - NewLine:          new formatted line
    """

    # Debug:
    # print ("line = ", line)


    ## initialize return parameter
    NewLine = ""


    ## write line
    LocalFitFlag = False
    for col in line:
        Name = col[0].strip().lower()
        flag = col[1]
        lowlimit = col[2]
        uplimit = col[3]
        element = col[4]

        # Debug:
        # print ("col = ", col)

        if (Name.endswith("_fitflag") or Name.endswith("_flag")):
            if (float(flag) != 0):
                LocalFitFlag = True
            else:
                LocalFitFlag = False


        ## create new line with fixed parameters
        if (Name != "molfitfileformatversion" and not (Name.endswith("_fitflag") or Name.endswith("_flag") or Name.endswith("_lowlimit") \
            or Name.endswith("_uplimit"))):
            if (Name in ["layerdistance", "contfuncid_phen", "cfflag", "keyword"]):
                NewLine += "  " + element
            else:
                if (float(flag) != 0 or LocalFitFlag):
                    NewLine += "   y  {:12.3e}  {:12.3e}  {:25.15e}".format(float(lowlimit), float(uplimit), float(element))
                else:
                    NewLine += "   n  {:12.3e}  {:12.3e}  {:25.15e}".format(float(lowlimit), float(uplimit), float(element))


    ## define return value
    return NewLine
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## write molfit parameters to new molfit file
##
def WriteMolfitFile(MolfitsFileName, MoleculesInMolfitFile, AllParameters, LogLinearFlag, NEWFormatFlag, SQLParamArray):
    """

input parameters:
-----------------

    - MolfitsFileName:          path and name of the molfit file located in the job directory

    - MoleculesInMolfitFile:    list of all molecules included in the new molfit file

    - AllParameters:            all parameters for the new molfit file

    - LogLinearFlag:            indicates, if column density values have to be converted from linear to log. scale (=log) or from log t linear (linear)

    - NEWFormatFlag:            indicates, if molfit file is written in new format (True) or if current format is used (False)

    - SQLParamArray:            array containing SQL parameters



output parameters:
------------------

    - NewMolfitFile:            array containing the new molfit file, if variable does not define path and name of molfit file
    """

    # Debug:
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("MoleculesInMolfitFile = ", MoleculesInMolfitFile)
    # print ("AllParameters = ", AllParameters)
    # print ("LogLinearFlag = ", LogLinearFlag)
    # print ("NEWFormatFlag = ", NEWFormatFlag)
    # print ("SQLParamArray = ", SQLParamArray)


    ## check, if result is written to file or to python array
    NewMolfitFile = []
    if (type(MolfitsFileName) is str):
        WriteToFileFlag = True
        MolFile = open(MolfitsFileName, 'w')
    else:
        WriteToFileFlag = False


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write header of molfit file including SQL parameters
    MinNumTransitionsSQL = SQLParamArray[0]
    MaxNumTransitionsSQL = SQLParamArray[1]
    MaxElowSQL = SQLParamArray[2]
    MingASQL = SQLParamArray[3]
    OrderTransSQL = SQLParamArray[4]
    MolfitHeaderLinesList = WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)
    for NewLine in MolfitHeaderLinesList:
        if (WriteToFileFlag):
            MolFile.write(NewLine)                                                          ## write new line to file
        else:
            NewMolfitFile.append(NewLine)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write parameter to new molfit file
    for MoleculeIndex in range(len(MoleculesInMolfitFile)):                                 ## loop over all molecules in the molfit file
        LocalMolfitParameters = AllParameters[MoleculeIndex]                                ## get the parameters describing the current molecule

        # Debug:
        # print ("LocalMolfitParameters = ", LocalMolfitParameters)


        ## write name of molecule and number of components
        NewLine = MoleculesInMolfitFile[MoleculeIndex] + "   " + str(len(LocalMolfitParameters)) + "\n"
        if (WriteToFileFlag):
            MolFile.write(NewLine)                                                          ## write new line to file
        else:
            NewMolfitFile.append(NewLine)


        ## write parameter for current molecule to xml file
        for line in LocalMolfitParameters:                                                  ## loop over all components of the current molecule


            ## write each parameter for current molecule to xml file
            NewLine = ""                                                                    ## initialize new line
            for param in line:                                                              ## loop over all parameters

                # Debug:
                # print ("param = ", param)


                ParameterName = param[0].strip()                                            ## get name of parameter
                LimitValues = param[1]                                                      ## get fit-flag of parameter
                LimitValuesTrueFalse = "false"                                              ## define LimitValuesTrueFalse variable
                #if (float(LimitValues) != 0):
                #    LimitValuesTrueFalse = "true"
                lowlimit = param[2]                                                         ## get lower limit
                uplimit = param[3]                                                          ## get upper limit
                ParameterValue = param[4]                                                   ## get value of parameter

                # Debug:
                # print ("param = ", param)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## define molfit file format for each line
                if (ParameterName == "MolfitFileFormatVersion"):
                    SplittedMolfitFileFormatVersion = ParameterValue.split("_")


                    ## write parameter indicating molfit format
                    LocalMolfitFileFormatVersion = "PURE"
                    if ("old" in SplittedMolfitFileFormatVersion):
                        LocalMolfitFileFormatVersion = "OLD"
                    elif ("new" in SplittedMolfitFileFormatVersion):
                        LocalMolfitFileFormatVersion = "NEW"
                else:


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## if LogLinearFlag is defined, convert column densities from linear to log scale or vis versa
                    if (LogLinearFlag != ""):
                        if (ParameterName in ["N_tot", "N_tot_uplimit", "N_tot_lowlimit", \
                                              "EM_RRL", "EM_RRL_uplimit", "EM_RRL_lowlimit", \
                                              "nHcolumn_cont_dust", "nHcolumn_cont_dust_uplimit", "nHcolumn_cont_dust_lowlimit", \
                                              "nHcolumn", "nHcolumn_uplimit", "nHcolumn_lowlimit"]):
                            ParameterValue = float(ParameterValue)
                            NewParamValue = ConvertParamValue(LogLinearFlag, ParameterValue)
                            ParameterValue = "{:25.15e}".format(NewParamValue)

                        # Debug:
                        # print ("ParameterValue = ", ParameterValue)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## convert OLD format to NEW format
                    if (LocalMolfitFileFormatVersion == "OLD" and NEWFormatFlag == "true"):
                        if (ParameterName.endswith("_FitFlag")):
                            ParameterValue = float(ParameterValue)
                            if (ParameterValue == 0.0):                                     ## no range is defined, parameter is kept constant
                                element = " n "
                                lowlimit = 0.0
                                uplimit = 0.0
                            else:                                                           ## a range is defined, determine limits for parameter
                                element = " y "
                                lowlimit, uplimit = GetParameterLimits(ParameterName, ParameterValue, LimitValues)
                            element += "{:12.3f} {:12.3f}".format(lowlimit, uplimit)        ## create new part of current line
                        elif (ParameterName != "LayerDistance" and ParameterName != "ContFuncID_phen" and ParameterName != "CFFlag" \
                            and ParameterName != "keyword"):
                            element = "{:20.10e}".format(float(ParameterValue))
                        else:
                            element = ParameterValue


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## write parameter value with different formats to line
                    else:
                        if (LocalMolfitFileFormatVersion == "PURE"):
                            if (not ParameterName in ["LayerDistance", "ContFuncID_phen", "CFFlag", "keyword"]):
                                element = "{:20.10e}".format(float(ParameterValue))
                            else:
                                element = ParameterValue

                        elif (LocalMolfitFileFormatVersion == "OLD"):
                            if (ParameterName.endswith("_FitFlag")):
                                element = "{:12.2f}".format(float(ParameterValue))
                            elif (not ParameterName in ["LayerDistance", "ContFuncID_phen", "CFFlag", "keyword"]):
                                element = "{:20.10e}".format(float(ParameterValue))
                            else:
                                element = ParameterValue

                        elif (LocalMolfitFileFormatVersion == "NEW"):
                            if (ParameterName.endswith("_uplimit") or ParameterName.endswith("_lowlimit")):
                                if (ParameterName.startswith("N_tot") or ParameterName.startswith("EM_RRL") \
                                    or ParameterName.startswith("nHcolumn_cont_dust") \
                                    or ParameterName.startswith("nHcolumn") ):
                                    element = "{:12.3e}".format(float(ParameterValue))
                                else:
                                    element = "{:12.3f}".format(float(ParameterValue))
                            elif (ParameterName in ["LayerDistance", "ContFuncID_phen", "CFFlag", "keyword"] or ParameterName.endswith("_flag") ):
                                element = ParameterValue
                            else:
                                element = "{:20.10e}".format(float(ParameterValue))


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## create new line
                    NewLine += " " + element


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write new line to file
            NewLine = NewLine + "\n"
            if (WriteToFileFlag):
                MolFile.write(NewLine)
            else:
                NewMolfitFile.append(NewLine)


    ## close xml-file
    if (WriteToFileFlag):
        MolFile.close()

    # Debug:
    # print ("WriteToFileFlag = ", WriteToFileFlag)
    # print ("NewMolfitFile = ", NewMolfitFile)


    ## we've finished
    return NewMolfitFile
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## creates a machine readable molfit file containing the best fit values, the left and right errors, the mean value and standard deviation for each free
## parameter
##
def CreateErrorMolfitFile(InstanceXMLFile, MolfitFileName, printInfoFlag):
    """

input parameters:
-----------------

    - InstanceXMLFile:      path and name of instance xml-file located in the job directory

    - MolfitFileName:       path and name of the molfit file located in the job directory

    - printInfoFlag:        (T/F) print information to screen


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("InstanceXMLFile = ", InstanceXMLFile)
    # print ("MolfitFileName = ", MolfitFileName)
    # print ("printInfoFlag = ", printInfoFlag)


    ## define name of error molfit file
    i = MolfitFileName.rfind("/")                                                           ## get path of molfit file
    if (i > 0):                                                                             ## is a path defined?
        i += 1                                                                              ## include "/" character
        MolfitFileNamePart = MolfitFileName[i:]                                             ## get name of molfit file
        j = MolfitFileNamePart.rfind(".")                                                   ## get name without .molfit extension
        NewErrorMolfitFileName = MolfitFileName[:i] + MolfitFileNamePart[:j] + "__error.molfit"     ## define new name

    # Debug:
    # print ("NewErrorMolfitFileName = ", NewErrorMolfitFileName)


    ## print what you do
    if (printInfoFlag == "true"):
        print ("Create further molfit file " + NewErrorMolfitFileName + " containing error information for each free parameter .. ", end = "")


    ## get total number of parameters in the instance xml file
    TotalParamNumber = task_MAGIX.GetXMLtagNEW(InstanceXMLFile, "NumberParameters")
    TotalParamNumber = int(float(TotalParamNumber[0]))


    ## get names of each parameter
    AllParamNames = task_MAGIX.GetXMLtagNEW(InstanceXMLFile, "name")


    ## get values of each parameter
    AllParamValues = task_MAGIX.GetXMLtagNEW(InstanceXMLFile, "value")


    ## get low limits of each parameter
    AllParamLowLimit = task_MAGIX.GetXMLtagNEW(InstanceXMLFile, "lowlimit")


    ## get upper limits of each parameter
    AllParamUpLimit = task_MAGIX.GetXMLtagNEW(InstanceXMLFile, "uplimit")


    ## get error information of each parameter
    AllParamErrors = task_MAGIX.GetXMLtagNEW(InstanceXMLFile, "error")
    AllParamErrorsCopy = copy.deepcopy(AllParamErrors)
    AllParamErrors = []
    ErrorMethod = "INS"
    for ParamError in AllParamErrorsCopy:
        ParamErrorSplitted = ParamError.split(";")
        if (len(ParamError) < 2):
            NewContent = [0.0, 0.0, 0.0, 0.0, 0.0]
        else:
            NewContent = []
            counterElement = 0
            for element in ParamErrorSplitted:
                counterElement += 1
                if (counterElement > 1):                                                    ## ignore first entry
                    i = element.find("=")
                    if (i > 0):
                        NewContent.append(float(element[i + 1:].strip()))                   ## add error values
            if (len(NewContent) == 2):
                ErrorMethod = "MCMC"
        AllParamErrors.append(NewContent)

    # Debug:
    # print ("\n\nTotalParamNumber = ", TotalParamNumber)
    # print ("AllParamNames = ", AllParamNames)
    # print ("AllParamValues = ", AllParamValues)
    # print ("AllParamErrors = ", AllParamErrors)
    # print ("AllParamLowLimit = ", AllParamLowLimit)
    # print ("AllParamUpLimit = ", AllParamUpLimit)
    # print ("ErrorMethod = ", ErrorMethod)


    ## check consistency of instance xml file
    if (TotalParamNumber != len(AllParamNames) or TotalParamNumber != len(AllParamValues) or TotalParamNumber != len(AllParamErrors)):
        print ("\n\nError in XCLASS package, function task_myXCLASS.CreateErrorMolfitFile:")
        print ("\t The number of parameters does not coincide with the number of parameter names, values, errors etc.")
        print ("\n\t Please check instance xml-file " + InstanceXMLFile)
        print (" ")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write header of error molfit file


    ## open channel for new error molfit file
    NewErrorMolfitFile = open(NewErrorMolfitFileName, 'w')                                  ## open channel for new error molfit file


    ## write first header line
    NumberOfMolecules = AllParamNames.count("Molecule_Name")
    FirstHeaderLine = "% Number of molecules = " + str(NumberOfMolecules) + "\n"
    NewErrorMolfitFile.write(FirstHeaderLine)
    AllNamesList = []


    ## determine source size is defined at least in one component
    SecondHeaderLine = ""
    SourceSizeFlag = False
    countSourceSizeFlag = AllParamNames.count("source_size")
    if (countSourceSizeFlag > 0):
        SourceSizeFlag = True
        AllNamesList.append("source size")


    ## define list of parameter names
    AllNamesList.append("T_rot")
    AllNamesList.append("N_tot")
    AllNamesList.append("V_width_Gauss")
    AllNamesList.append("V_off")


    ## determine if dust parameters are defined for at least one component
    count_T_cont_dust = AllParamNames.count("T_cont_dust")
    if (count_T_cont_dust > 0):
        AllNamesList.append("T_cont_dust")
        AllNamesList.append("nHcolumn_cont_dust")
        AllNamesList.append("kappa_cont_dust")
        AllNamesList.append("beta_cont_dust")


    ## determine if T_doff and T_dslope are defined for at least one component
    count_T_dOff = AllParamNames.count("T_dOff")
    if (count_T_dOff > 0):
        AllNamesList.append("T_back")
        AllNamesList.append("T_slope")


    ## determine if nHcolumn, kappa, and beta are defined for at least one component
    count_nHcolumn = AllParamNames.count("nHcolumn")
    if (count_nHcolumn > 0):
        AllNamesList.append("nH")
        AllNamesList.append("beta")
        AllNamesList.append("kappa")


    ## determine if parameters for phen. continuum description are defined for at least one component
    countContFuncID_phen = AllParamNames.count("ContFuncID_phen")
    if (countContFuncID_phen > 0):
        AllNamesList.append("ContFuncID_phen")
        AllNamesList.append("ContFuncID_param_1")
        AllNamesList.append("ContFuncID_param_2")
        AllNamesList.append("ContFuncID_param_3")
        AllNamesList.append("ContFuncID_param_4")
        AllNamesList.append("ContFuncID_param_5")


    ## write second header line
    SecondHeaderLine = ""
    counter = (-1)
    for ParamName in AllNamesList:
        counter += 1
        if (counter == 0):
            SecondHeaderLine += "%         flag:"
        else:
            SecondHeaderLine += "          flag:"
        SecondHeaderLine += "     low limit:"
        SecondHeaderLine += "      up limit:"
        SecondHeaderLine += "{:>22s}:".format(ParamName)
        SecondHeaderLine += "    error left:"
        SecondHeaderLine += "   error right:"
        if (ErrorMethod == "INS"):
            SecondHeaderLine += "     mean val.:"
            SecondHeaderLine += "      std-dev.:"
            SecondHeaderLine += "    log(evid.):"
    SecondHeaderLine += "        a/e-flag:\n"
    NewErrorMolfitFile.write(SecondHeaderLine)


    ## define names of parameters
    ParameterNameList = ["source_size", "source_radius", "source_center_x", "source_center_y", "T_rot", "T_e", "T_cont_dust", \
                         "T_dOff", "T_dSlope", "N_tot", "EM_RRL", "nHcolumn_cont_dust", "nHcolumn", "V_width_Gauss", "V_off", \
                         "beta_cont_dust", "kappa_cont_dust", "kappa", "beta", \
                         "ContFuncID_param_1", "ContFuncID_param_2", "ContFuncID_param_3", "ContFuncID_param_4", "ContFuncID_param_5"]


    ## write parameters with error information for each line to error molfit file
    InitialitionFlag = True
    NewLine = ""
    for i in range(len(AllParamNames)):
        ParamName = AllParamNames[i]
        ParamValue = AllParamValues[i]
        ParamError = AllParamErrors[i]


        ParamLowLimit = float(AllParamLowLimit[i])
        ParamUpLimit = float(AllParamUpLimit[i])
        if (ParamName == "MolfitFileFormatVersion"):
            NewErrorMolfitFile.write(NewLine + "\n")
            FitFlag = ""
            NewLine = ""


        ## write name of molecule and number of components to file
        elif (ParamName == "Molecule_Name"):                                                ## does current name describes a molecule name?
            if (not InitialitionFlag):
                NewErrorMolfitFile.write(NewLine + "\n")
            else:
                InitialitionFlag = False
            NewLine = ParamValue
        elif (ParamName == "Number_MolLine"):                                               ## is the number of components defined?
            NewLine += "{:>5d}".format(int(float(ParamValue)))


        ## get fit flag
        elif (ParamName.endswith("_FitFlag")):
            ParamValue = float(ParamValue)                                                  ## convert string to float
            if (ParamValue != 0.0):
                FitFlag = "y"
            else:
                FitFlag = "n"
        elif (ParamName.endswith("_flag")):
            FitFlag = ParamValue


        ## write parameters for each component with flag, lower and upper limit, parameter value
        elif (ParamName in ParameterNameList):
            ParamValue = float(ParamValue)                                                  ## convert string to float
            NewLine += "{:>15s}".format(FitFlag)                                            ## add fit flag
            if (FitFlag == "n"):                                                            ## if parameter is not fitted, set limits to zero
                ParamLowLimit = 0.0
                ParamUpLimit = 0.0
            if (ParamName in ["N_tot", "EM_RRL", "nHcolumn_cont_dust", "nHcolumn"]):
                LocalValue = ParamValue
                ParamValue = 10.E0**ParamValue                                              ## convert to linear value
                NewLine += "{:>15.3e}".format(10.E0**ParamLowLimit)                         ## add lower limit to line
                NewLine += "{:>15.3e}".format(10.E0**ParamUpLimit)                          ## add upper limit to line

                LocalLow = float(ParamError[0])                                             ## get left error
                LocalUp = float(ParamError[1])                                              ## get right error
                errorLow = 10.E0**(LocalValue - LocalLow)
                errorUp = 10.E0**(LocalValue + LocalUp)
                ParamValue = 10.E0**LocalValue                                              ## convert to linear value
                ParamError[0] = (ParamValue - errorLow)                                     ## convert error left to linear value
                ParamError[1] = (errorUp - ParamValue)                                      ## convert error right to linear value
            else:
                NewLine += "{:>15.2f}".format(ParamLowLimit)                                ## add lower limit to line
                NewLine += "{:>15.2f}".format(ParamUpLimit)                                 ## add upper limit to line
            NewLine += "{:>23.10e}".format(ParamValue)                                      ## add parameter value to new line of molfit file


            ## add error information of current parameter to line
            x = float(ParamError[0])                                                        ## add left error
            NewLine += "{:>15.5e}".format(x)
            x = float(ParamError[1])                                                        ## add right error
            NewLine += "{:>15.5e}".format(x)
            if (ErrorMethod == "INS"):
                x = float(ParamError[2])                                                    ## add mean value
                NewLine += "{:>15.5e}".format(x)
                x = float(ParamError[3])                                                    ## add standard dev.
                NewLine += "{:>15.5e}".format(x)
                x = float(ParamError[4])                                                    ## add log(evidence)
                NewLine += "{:>15.5e}".format(x)


        ## write parameters for each component with flag, lower and upper limit, parameter value
        elif (ParamName in ["CFFlag", "keyword"]):
            NewLine += "{:>17s}".format(ParamValue)                                         ## add parameter value to new line of molfit file


        ## write parameters for each component with flag, lower and upper limit, parameter value
        elif (ParamName in ["ContFuncID_phen"]):
            NewLine += "{:>17d}".format(int(ParamValue))                                    ## add parameter value to new line of molfit file


        ## write parameters for each component with flag, lower and upper limit, parameter value
        elif (ParamName in ["LayerDistance"]):
            try:
                NewLine += "{:>17.10e}".format(float(ParamValue))                           ## add parameter value to new line of molfit file
            except:
                NewLine += "{:>17s}".format(" ")


    ## write last line to new molfit file
    if (NewLine != ""):
        NewErrorMolfitFile.write(NewLine + "\n")


    ## close channel for new molfit file
    NewErrorMolfitFile.close()


    ## we're done
    if (printInfoFlag == "true"):
        print ("done!")
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Get SQL parameters from molfit file
##
def GetSQLParameter(molfitFilename):
    """

input parameters:
-----------------

    - molfitFilename:           path and name of molfit file


output parameters:
------------------

    - MinNumTransitionsSQL:     min. number of transitions

    - MaxNumTransitionsSQL:     max. number of transitions

    - MaxElowSQL:               max. lower energy for transitions

    - MingASQL:                 minimal intensity (gA) for transitions

    - OrderTransSQL:            order of transitions (=0: frequency, =1: lower energy. =2: gA)
    """

    # Debug:
    # print ("molfitFilename = ", molfitFilename)


    ## initialize return parameter
    MinNumTransitionsSQL = 1
    MaxNumTransitionsSQL = 0
    MaxElowSQL = 1.e6
    MingASQL = 0.0
    OrderTransSQL = 1


    ## read in molfit file
    LocalMolfitFileContents = None
    if (type(molfitFilename) is str):
        if (molfitFilename.strip() != ""):
            if (os.path.isfile(molfitFilename) or os.path.isfile(molfitFilename + ".gz")):
                if (molfitFilename.endswith(".gz")):
                    LocalMolfitFile = gzip.open(molfitFilename, 'rt')
                else:
                    try:
                        LocalMolfitFile = open(molfitFilename)
                    except:
                        LocalMolfitFile = gzip.open(molfitFilename + ".gz", 'rt')
                LocalMolfitFileContents = LocalMolfitFile.readlines()
                LocalMolfitFile.close()
    else:
        LocalMolfitFileContents = copy.deepcopy(molfitFilename)
    if (LocalMolfitFileContents is None):
        return (MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)


    ## get sql parameters
    CommandWordList = ["%minnumtransitionssql", "%maxnumtransitionssql", "%maxelowsql", "%mingasql", "%transordersql", "%ordertranssql"]
    for line in LocalMolfitFileContents:
        StrippedLine = line.strip()
        i = StrippedLine.find("%")
        if (i > (-1)):
            LocalComment = StrippedLine[(i + 1):].strip()
            LocalComment = LocalComment.lower()
            for LocalCommand in CommandWordList:
                j = LocalComment.find(LocalCommand)                                         ## search for command word in molfit file
                if (j > (-1)):
                    LocalComment = LocalComment[(i + len(LocalCommand)):].strip()
                    i = LocalComment.find("=")
                    if (i > (-1)):
                        LocalComment = LocalComment[(i + 1):].strip()
                        i = LocalComment.find("%")
                        if (i > (-1)):
                            LocalComment = LocalComment[:i].strip()
                        if (LocalCommand == CommandWordList[0]):                            ## get parameter for 1st command word in molfit file
                            MinNumTransitionsSQL = int(LocalComment)
                        elif (LocalCommand == CommandWordList[1]):                          ## get parameter for 2nd command word in molfit file
                            MaxNumTransitionsSQL = int(LocalComment)
                        elif (LocalCommand == CommandWordList[2]):                          ## get parameter for 3rd command word in molfit file
                            MaxElowSQL = float(LocalComment)
                        elif (LocalCommand == CommandWordList[3]):                          ## get parameter for 4th command word in molfit file
                            MingASQL = float(LocalComment)
                        elif (LocalCommand == CommandWordList[4] or LocalCommand == CommandWordList[5]):    ## get parameter for 5th command word
                            OrderTransSQL = int(LocalComment)

    # Debug:
    # print ("MinNumTransitionsSQL = ", MinNumTransitionsSQL)
    # print ("MaxNumTransitionsSQL = ", MaxNumTransitionsSQL)
    # print ("MaxElowSQL = ", MaxElowSQL)
    # print ("MingASQL = ", MingASQL)
    # print ("OrderTransSQL = ", OrderTransSQL)


    ## we're done
    return (MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Write SQL parameters to molfit file
##
def WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL, PureHeaderFlag = False):
    """

input parameters:
-----------------

    - MinNumTransitionsSQL:     min. number of transitions

    - MaxNumTransitionsSQL:     max. number of transitions

    - MaxElowSQL:               max. lower energy for transitions

    - MingASQL:                 minimal intensity (gA) for transitions

    - OrderTransSQL:            order of transitions (=0: frequency, =1: lower energy. =2: gA)

    - PureHeaderFlag:           (optional) flag indicating if pure header is created


output parameters:
------------------

    - NewMolfitFileHeaderList:  settings for SQL parameters in molfit file
    """

    # Debug:
    # print ("MinNumTransitionsSQL = ", MinNumTransitionsSQL)
    # print ("MaxNumTransitionsSQL = ", MaxNumTransitionsSQL)
    # print ("MaxElowSQL = ", MaxElowSQL)
    # print ("MingASQL = ", MingASQL)
    # print ("OrderTransSQL = ", OrderTransSQL)
    # print ("PureHeaderFlag = ", PureHeaderFlag)


    ## initialize return parameter
    NewMolfitFileHeaderList = []


    ## add some comments
    LineString = "=" * 153
    LineString = "%" + LineString + "\n"
    NewMolfitFileHeaderList.append(LineString)
    NewMolfitFileHeaderList.append("%\n")
    NewMolfitFileHeaderList.append("% define database parameters:\n")
    NewMolfitFileHeaderList.append("%\n")
    NewMolfitFileHeaderList.append(LineString)


    ## define minimum number of transitions
    NumMinTransString = "{:12d}".format(MinNumTransitionsSQL)
    NumMinTransString = "{:<17s}".format(NumMinTransString.strip())
    NewLine = "%%MinNumTransitionsSQL = " + NumMinTransString + "% (min. number of transitions)\n"
    NewMolfitFileHeaderList.append(NewLine)


    ## define maximum number of transitions
    NumTransString = "{:12d}".format(MaxNumTransitionsSQL)
    NumTransString = "{:<17s}".format(NumTransString.strip())
    NewLine = "%%MaxNumTransitionsSQL = " + NumTransString + "% (max. number of transitions)\n"
    NewMolfitFileHeaderList.append(NewLine)


    ## define max. lower energy
    ElowString = "{:12.3e}".format(MaxElowSQL)
    ElowString = "{:<27s}".format(ElowString.strip())
    NewLine = "%%MaxElowSQL = " + ElowString + "% (max. lower energy, i.e. upper limit of the lower energy of a transition)\n"
    NewMolfitFileHeaderList.append(NewLine)


    ## define minimal intensity
    gAString = "{:12.3e}".format(MingASQL)
    gAString = "{:<29s}".format(gAString.strip())
    NewLine = "%%MingASQL = " + gAString + "% (minimal intensity, i.e. lower limit of gA of a transition)\n"
    NewMolfitFileHeaderList.append(NewLine)


    ## define maximum number of transitions
    OrderString = "{:12d}".format(OrderTransSQL)
    OrderString = "{:<24s}".format(OrderString.strip())
    NewLine = "%%TransOrderSQL = " + OrderString + "% (order of transitions: (=1): by Elow, (=2) by gA, (=3) gA/E_low^2, else trans. freq.)\n"
    NewMolfitFileHeaderList.append(NewLine)


    ## add some comments
    NewMolfitFileHeaderList.append("\n\n")
    if (not PureHeaderFlag):
        NewMolfitFileHeaderList.append(LineString)
        NewMolfitFileHeaderList.append("%\n")
        NewMolfitFileHeaderList.append("% define molecules and their components:\n")
        NewMolfitFileHeaderList.append("%\n")
        NewMolfitFileHeaderList.append(LineString)

    # Debug:
    # print ("NewMolfitFileHeaderList = ", NewMolfitFileHeaderList)


    ## define return parameter
    return NewMolfitFileHeaderList
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## converts a molfit file used by myXCLASS to a xml-file used by MAGIX.
##
def ConversionMolfit2XML(print_flag, molfitFilename, xmlFileName):
    """

input parameters:
-----------------

    - print_flag:               flag if screen output is permitted (=true) or not

    - molfitFilename:           path and name of the molfit file

    - xmlFileName:              path and name of the xml file containing the description of the model


output parameters:
------------------

    - ok:                       status; if everything is ok: ok = 0
    """

    # Debug:
    # print ("print_flag = ", print_flag)
    # print ("molfitFilename = ", molfitFilename)
    # print ("xmlFileName = ", xmlFileName)


    ## initialize return parameter
    ok = 0


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get molfit and xml filename
    if not molfitFilename.endswith(".molfit"):
        print ("\n\nError in XCLASS package, function task_myXCLASS.ConversionMolfit2XML:")
        print ("\t The selected file ", molfitFilename)
        print ("\t is not a molfit file!")
        print ("\n\t Please correct input!\n")
        ok = 1
        return

    if not xmlFileName.endswith(".xml"):
        print ("\n\nError in XCLASS package, function task_myXCLASS.ConversionMolfit2XML:")
        print ("\t The selected file ", xmlFileName)
        print ("\t is not a xml file!")
        print ("\n\t Please correct input!\n")
        ok = 1
        return

    # Debug:
    # print ('molfitFilename = ',molfitFilename)
    # print ('xmlFileName = ',xmlFileName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze molfit file using myXCLASS routine
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = AnalyzeMolfitFile(molfitFilename)

    # Debug:
    # print ("MoleculesInMolfitFile = ", MoleculesInMolfitFile)


    ## get special sql commands from molfit file
    MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = GetSQLParameter(molfitFilename)


    ## determine number of parameters
    NumberParameter = 0


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write parameter.xml file
    NewInstanceFile = open(xmlFileName, 'w')
    line="<?xml version=" + chr(34) + "1.0" + chr(34) + " ?>\n"
    NewInstanceFile.write(line)
    NewInstanceFile.write("<ModelParameters>\n")


    ## write number of parameters
    NewInstanceFile.write("\n")
    NewInstanceFile.write("\n")
    NewInstanceFile.write("    <!-- define number of parameters -->\n")
    NewInstanceFile.write("    <NumberParameters>" + str(NumberParameter + 5) + "</NumberParameters>\n")
    NewInstanceFile.write("\n")
    NewInstanceFile.write("\n")
    star = "=" * 141
    NewInstanceFile.write("    <!-- " + star + " -->\n")
    NewInstanceFile.write("    <!-- first input file -->\n")
    NewInstanceFile.write("\n")
    NewInstanceFile.write("\n")


    ## write min. number of transitions to xml-file
    XCLASSRootDir = GetXCLASSRootDir()
    NewInstanceFile.write("    <!-- define path of XCLASS root directory -->\n")
    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    NewInstanceFile.write("        <name>XCLASSRootDir</name>\n")
    NewInstanceFile.write("        <value>" + XCLASSRootDir.strip() + "</value>\n")
    NewInstanceFile.write("        <error> </error>\n")
    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
    NewInstanceFile.write("        <uplimit>1</uplimit>\n")
    NewInstanceFile.write("    </Parameter>\n")
    NewInstanceFile.write("\n")
    NewInstanceFile.write("\n")


    ## write min. number of transitions to xml-file
    NewInstanceFile.write("    <!-- define min. number of transitions used for SQL query -->\n")
    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    NewInstanceFile.write("        <name>MinNumTransitionsSQL</name>\n")
    NewInstanceFile.write("        <value>" + str(MinNumTransitionsSQL) + "</value>\n")
    NewInstanceFile.write("        <error> </error>\n")
    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
    NewInstanceFile.write("        <uplimit>1</uplimit>\n")
    NewInstanceFile.write("    </Parameter>\n")
    NewInstanceFile.write("\n")
    NewInstanceFile.write("\n")


    ## write max. number of transitions to xml-file
    NewInstanceFile.write("    <!-- define max. number of transitions used for SQL query -->\n")
    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    NewInstanceFile.write("        <name>MaxNumTransitionsSQL</name>\n")
    NewInstanceFile.write("        <value>" + str(MaxNumTransitionsSQL) + "</value>\n")
    NewInstanceFile.write("        <error> </error>\n")
    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
    NewInstanceFile.write("        <uplimit>1</uplimit>\n")
    NewInstanceFile.write("    </Parameter>\n")
    NewInstanceFile.write("\n")
    NewInstanceFile.write("\n")


    ## write highest lower energy of transitions to xml-file
    NewInstanceFile.write("    <!-- define highest lower energy of a transition used for SQL query -->\n")
    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    NewInstanceFile.write("        <name>MaxElowSQL</name>\n")
    h = MaxElowSQL / 1.42879
    NewInstanceFile.write("        <value>" + str(h) + "</value>\n")
    NewInstanceFile.write("        <error> </error>\n")
    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
    NewInstanceFile.write("        <uplimit>1</uplimit>\n")
    NewInstanceFile.write("    </Parameter>\n")
    NewInstanceFile.write("\n")
    NewInstanceFile.write("\n")


    ## write min. intensity of transitions to xml-file
    NewInstanceFile.write("    <!-- define min. intensity of a transition used for SQL query -->\n")
    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    NewInstanceFile.write("        <name>MingASQL</name>\n")
    NewInstanceFile.write("        <value>" + str(MingASQL) + "</value>\n")
    NewInstanceFile.write("        <error> </error>\n")
    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
    NewInstanceFile.write("        <uplimit>1</uplimit>\n")
    NewInstanceFile.write("    </Parameter>\n")
    NewInstanceFile.write("\n")
    NewInstanceFile.write("\n")


    ## write order of transitions to xml-file
    NewInstanceFile.write("    <!-- define order of a transition used for SQL query -->\n")
    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    NewInstanceFile.write("        <name>OrderTransSQL</name>\n")
    NewInstanceFile.write("        <value>" + str(OrderTransSQL) + "</value>\n")
    NewInstanceFile.write("        <error> </error>\n")
    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
    NewInstanceFile.write("        <uplimit>1</uplimit>\n")
    NewInstanceFile.write("    </Parameter>\n")
    NewInstanceFile.write("\n")
    NewInstanceFile.write("\n")


    ## write number of molecules to xml-file
    NewInstanceFile.write("    <!-- define number of molecules -->\n")
    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    NewInstanceFile.write("        <name>Number_Molecules</name>\n")
    NewInstanceFile.write("        <value>" + str(len(MoleculesInMolfitFile)) + "</value>\n")
    NewInstanceFile.write("        <error> </error>\n")
    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
    NewInstanceFile.write("        <uplimit>1</uplimit>\n")
    NewInstanceFile.write("    </Parameter>\n")
    NewInstanceFile.write("\n")
    NewInstanceFile.write("\n")
    NumberParameter = 7

    # Debug:
    # print ("AllParameters = ", AllParameters)


    ## write parameter to xml-file
    MolCounter = 0
    for MoleculeIndex in range(len(MoleculesInMolfitFile)):                                 ## loop over all molecules in the molfit file
        LocalMolfitParameters = AllParameters[MoleculeIndex]                                ## get the parameters describing the current molecule
        MolCounter += 1


        ## write name of molecule
        star = "=" * 59
        NewInstanceFile.write("    <!-- " + star + " describe new molecule " + star + " -->\n")
        NewInstanceFile.write("    <!-- define Molecule_Name -->\n")
        NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
        NewInstanceFile.write("        <name>Molecule_Name</name>\n")
        NewInstanceFile.write("        <value>" + MoleculesInMolfitFile[MoleculeIndex] + "</value>\n")
        NewInstanceFile.write("        <error> </error>\n")
        NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
        NewInstanceFile.write("        <uplimit>1</uplimit>\n")
        NewInstanceFile.write("    </Parameter>\n")
        NewInstanceFile.write("\n")
        NewInstanceFile.write("\n")


        ## write index of molecule
        NewInstanceFile.write("    <!-- define Molecule_Index -->\n")
        NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
        NewInstanceFile.write("        <name>Molecule_Inex</name>\n")
        NewInstanceFile.write("        <value>" + str(MolCounter) + "</value>\n")
        NewInstanceFile.write("        <error> </error>\n")
        NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
        NewInstanceFile.write("        <uplimit>1</uplimit>\n")
        NewInstanceFile.write("    </Parameter>\n")
        NewInstanceFile.write("\n")
        NewInstanceFile.write("\n")


        ## write number of components
        NewInstanceFile.write("    <!-- define Molecule_Name -->\n")
        NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
        NewInstanceFile.write("        <name>Number_MolLine</name>\n")
        NewInstanceFile.write("        <value>" + str(len(LocalMolfitParameters)) + "</value>\n")
        NewInstanceFile.write("        <error> </error>\n")
        NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
        NewInstanceFile.write("        <uplimit>1</uplimit>\n")
        NewInstanceFile.write("    </Parameter>\n")
        NewInstanceFile.write("\n")
        NewInstanceFile.write("\n")
        NumberParameter += 3


        ## write parameter for current molecule to xml file
        NumComp = 0
        for line in LocalMolfitParameters:                                                  ## loop over all components of the current molecule
            NumComp += 1

            # Debug:
            # print ("line = ", line)


            ## write number of components
            if (NumComp < 10):
                j = 53
            elif (NumComp < 100):
                j = 52
            elif (NumComp < 1000):
                j = 51
            else:
                j = 50
            star = "*" * j
            NewInstanceFile.write("    <!-- " + star + " define parameters for component " + str(NumComp) + " " + star + " -->\n")
            NewInstanceFile.write("\n")
            NewInstanceFile.write("\n")


            ## write each parameter for current molecule to xml file
            MolfitFileFormatVersionLocal = ""
            LocalFitFlag = "false"
            for param in line:                                                              ## loop over all parameters

                # Debug:
                # print ("param = ", param)


                ParameterName = param[0].strip()                                            ## get name of parameter
                LimitValues = str(param[1]).strip()                                         ## get fit-flag of parameter
                if (LimitValues == "1"):
                    LimitValuesTrueFalse = "true"
                else:
                    LimitValuesTrueFalse = "false"
                lowlimit = param[2]                                                         ## get lower limit
                uplimit = param[3]                                                          ## get upper limit
                ParameterValue = param[4]                                                   ## get value of parameter

                # Debug:
                # print ("param = ", param)
                # print (LimitValuesTrueFalse)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## define molfit file format for each line
                if (ParameterName.strip() == "MolfitFileFormatVersion"):
                    MolfitFileFormatVersionLocal = ParameterValue
                    SplittedMolfitFileFormatVersion = MolfitFileFormatVersionLocal.split("_")
                    SplittedMolfitFileFormatVersion = [i.lower() for i in SplittedMolfitFileFormatVersion]

                    # Debug:
                    # print ("MolfitFileFormatVersionLocal = ", MolfitFileFormatVersionLocal)
                    # print ("SplittedMolfitFileFormatVersion = ", SplittedMolfitFileFormatVersion)


                    ## write parameter indicating molfit format
                    LocalMolfitFileFormatVersion = "PUR"
                    if ("old" in SplittedMolfitFileFormatVersion):
                        LocalMolfitFileFormatVersion = "OLD"
                    if ("new" in SplittedMolfitFileFormatVersion):
                        LocalMolfitFileFormatVersion = "NEW"
                    NewInstanceFile.write("    <!-- define MolfitFileFormatVersion -->\n")
                    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    NewInstanceFile.write("        <name>MolfitFileFormatVersion</name>\n")
                    NewInstanceFile.write("        <value>" + LocalMolfitFileFormatVersion + "</value>\n")
                    NewInstanceFile.write("        <error> </error>\n")
                    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
                    NewInstanceFile.write("        <uplimit>0</uplimit>\n")
                    NewInstanceFile.write("    </Parameter>\n\n\n")


                    ## write parameter indicating if a absorption component definition contains no source size setting
                    LocaltdFlag = "F"
                    if ("t-dust-inline" in SplittedMolfitFileFormatVersion):
                        LocaltdFlag = "T"
                    NewInstanceFile.write("    <!-- define inline dust temperature parameter -->\n")
                    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    NewInstanceFile.write("        <name>tdFlag</name>\n")
                    NewInstanceFile.write("        <value>" + LocaltdFlag + "</value>\n")
                    NewInstanceFile.write("        <error> </error>\n")
                    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
                    NewInstanceFile.write("        <uplimit>0</uplimit>\n")
                    NewInstanceFile.write("    </Parameter>\n\n\n")


                    ## write parameter nHFlag indicating if column density etc is given in molfit file
                    LocalnHdFlag = "F"
                    if ("nh-dust-inline" in SplittedMolfitFileFormatVersion):
                        LocalnHdFlag = "T"
                    NewInstanceFile.write("    <!-- define nHFlag -->\n")
                    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    NewInstanceFile.write("        <name>nHFlag</name>\n")
                    NewInstanceFile.write("        <value>" + LocalnHdFlag + "</value>\n")
                    NewInstanceFile.write("        <error> </error>\n")
                    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
                    NewInstanceFile.write("        <uplimit>0</uplimit>\n")
                    NewInstanceFile.write("    </Parameter>\n\n\n")


                    ## write parameter GeometryFlag indicating geometry of layer in molfit file
                    LocalGeometryFlag = "NO"
                    if ("circle" in SplittedMolfitFileFormatVersion):
                        LocalGeometryFlag = "CC"
                    elif ("circleoff" in SplittedMolfitFileFormatVersion):
                        LocalGeometryFlag = "CO"
                    elif ("box" in SplittedMolfitFileFormatVersion):
                        LocalGeometryFlag = "BC"
                    elif ("boxoff" in SplittedMolfitFileFormatVersion):
                        LocalGeometryFlag = "BO"
                    ## elif ("?????" in SplittedMolfitFileFormatVersion):
                    #   LocalGeometryFlag = "??"
                    NewInstanceFile.write("    <!-- define geometry flag -->\n")
                    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    NewInstanceFile.write("        <name>GeometryFlag</name>\n")
                    NewInstanceFile.write("        <value>" + LocalGeometryFlag + "</value>\n")
                    NewInstanceFile.write("        <error> </error>\n")
                    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
                    NewInstanceFile.write("        <uplimit>0</uplimit>\n")
                    NewInstanceFile.write("    </Parameter>\n\n\n")


                    ## write parameter MoleculeFlag indicating which "molecule" is given in molfit file
                    LocalMoleculeFlag = "F"
                    LocalRRLNonLTEFlag = "F"
                    if ("molecule" in SplittedMolfitFileFormatVersion):
                        LocalMoleculeFlag = "M"
                    elif ("rrl" in SplittedMolfitFileFormatVersion):
                        LocalMoleculeFlag = "R"
                    elif ("dust" in SplittedMolfitFileFormatVersion):
                        LocalMoleculeFlag = "D"
                    elif ("phen" in SplittedMolfitFileFormatVersion):
                        LocalMoleculeFlag = "P"
                    NewInstanceFile.write("    <!-- define molecule flag -->\n")
                    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    NewInstanceFile.write("        <name>MoleculeFlag</name>\n")
                    NewInstanceFile.write("        <value>" + LocalMoleculeFlag + "</value>\n")
                    NewInstanceFile.write("        <error> </error>\n")
                    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
                    NewInstanceFile.write("        <uplimit>0</uplimit>\n")
                    NewInstanceFile.write("    </Parameter>\n\n\n")


                    ## write parameter RRLNonLTEFlag indicating if RRL is computed in Non-LTE or not (T or F)
                    NewInstanceFile.write("    <!-- define Non-LTE RRL flag -->\n")
                    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    NewInstanceFile.write("        <name>RRLNonLTEFlag</name>\n")
                    NewInstanceFile.write("        <value>" + LocalRRLNonLTEFlag + "</value>\n")
                    NewInstanceFile.write("        <error> </error>\n")
                    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
                    NewInstanceFile.write("        <uplimit>0</uplimit>\n")
                    NewInstanceFile.write("    </Parameter>\n\n\n")


                    ## write parameter LineWidthFlag indicating which line profile is given in molfit file
                    LocalLineWidthFlag = "0"
                    if ("gauss" in SplittedMolfitFileFormatVersion):
                        LocalLineWidthFlag = "G"
                    NewInstanceFile.write("    <!-- define line width flag -->\n")
                    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    NewInstanceFile.write("        <name>LineWidthFlag</name>\n")
                    NewInstanceFile.write("        <value>" + LocalLineWidthFlag + "</value>\n")
                    NewInstanceFile.write("        <error> </error>\n")
                    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
                    NewInstanceFile.write("        <uplimit>0</uplimit>\n")
                    NewInstanceFile.write("    </Parameter>\n\n\n")


                    ## write parameter VelOffFlag indicating if velocity offset is given in molfit file
                    LocalVelOffFlag = "F"
                    if (("molecule" in SplittedMolfitFileFormatVersion) or ("rrl" in SplittedMolfitFileFormatVersion)):
                        LocalVelOffFlag = "T"
                    NewInstanceFile.write("    <!-- define velocity flag -->\n")
                    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    NewInstanceFile.write("        <name>VelOffFlag</name>\n")
                    NewInstanceFile.write("        <value>" + LocalVelOffFlag + "</value>\n")
                    NewInstanceFile.write("        <error> </error>\n")
                    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
                    NewInstanceFile.write("        <uplimit>0</uplimit>\n")
                    NewInstanceFile.write("    </Parameter>\n\n\n")


                    ## write parameter ExtensionFlag indicating if extended format is used in molfit file
                    LocalExtensionFlag = "F"
                    if ("layerdistance" in SplittedMolfitFileFormatVersion):
                        LocalExtensionFlag = "T"
                    elif ("cfflag" in SplittedMolfitFileFormatVersion):
                        LocalExtensionFlag = "F"
                    NewInstanceFile.write("    <!-- define extension flag -->\n")
                    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    NewInstanceFile.write("        <name>ExtensionFlag</name>\n")
                    NewInstanceFile.write("        <value>" + LocalExtensionFlag + "</value>\n")
                    NewInstanceFile.write("        <error> </error>\n")
                    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
                    NewInstanceFile.write("        <uplimit>0</uplimit>\n")
                    NewInstanceFile.write("    </Parameter>\n\n\n")


                    ## write parameter ExtensionFlag indicating if extended format is used in molfit file
                    LocalKeywordFlag = "F"
                    if ("keyword" in SplittedMolfitFileFormatVersion):
                        LocalKeywordFlag = "T"
                    NewInstanceFile.write("    <!-- define extension flag -->\n")
                    NewInstanceFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    NewInstanceFile.write("        <name>KeywordFlag</name>\n")
                    NewInstanceFile.write("        <value>" + LocalKeywordFlag + "</value>\n")
                    NewInstanceFile.write("        <error> </error>\n")
                    NewInstanceFile.write("        <lowlimit>0</lowlimit>\n")
                    NewInstanceFile.write("        <uplimit>0</uplimit>\n")
                    NewInstanceFile.write("    </Parameter>\n\n\n")


                    ## increase parameter counter
                    NumberParameter += 10
                else:


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## for column and hydrogen column density use a log scale for fit
                    if (ParameterName in ["N_tot", "EM_RRL", "nHcolumn_cont_dust", "nHcolumn"]):
                        ParameterValue = float(ParameterValue)

                        # Debug:
                        # print ("ParameterValue = ", ParameterValue)
                        # print ("LimitValues = ", LimitValues)
                        # print ("lowlimit = ", lowlimit)
                        # print ("uplimit = ", uplimit)
                        # sys.exit(0)


                        if (ParameterValue != 0.E0):
                            if (ParameterValue < 0.E0):
                                ParameterValue = abs(ParameterValue)
                            ParameterValue = numpy.log10(ParameterValue)                    ## convert value to log scale
                            if (lowlimit != 0.E0 or uplimit != 0.E0):                       ## continue here, only if limit unequal zero


                                ## check and convert lower limit
                                if (lowlimit == 0.E0):                                      ## is lower limit exactly zero?
                                    lowlimit = 1.e-99                                       ## set to a very small number
                                elif (lowlimit < 0.E0):                                     ## is lower limit less than zero?
                                    lowlimit = abs(lowlimit)                                ## take the absolute value
                                lowlimit = numpy.log10(lowlimit)                            ## convert value to log scale
                                if (lowlimit > 307.0):                                      ## check for overflow
                                    lowlimit = 307.0

                                # Debug:
                                # print (">>lowlimit = ", lowlimit)


                                ## check and convert upper limit
                                if (uplimit == 0.E0):                                       ## is upper limit exactly zero?
                                    uplimit = 1.e-99                                        ## set to a very small number
                                elif (uplimit < 0.E0):                                      ## is upper limit less than zero?
                                    uplimit = abs(uplimit)                                  ## take the absolute value
                                uplimit = numpy.log10(uplimit)                              ## convert value to log scale
                                if (uplimit > 307.0):                                       ## check for overflow
                                    uplimit = 307.0

                                # Debug:
                                # print (">>uplimit = ", uplimit)
                    ##------------------------------------------------------------------------------------------------------------------------------------


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## for column and hydrogen column density use a log scale for fit
                    if (LocalMolfitFileFormatVersion == "NEW"):
                        if (ParameterName.strip() == "N_tot_uplimit" or ParameterName.strip() == "N_tot_lowlimit"):
                            LimitValuesTrueFalse = "false"
                            ParameterValue = float(ParameterValue)
                            if (ParameterValue != 0.E0):
                                if (ParameterValue < 0.E0):
                                    ParameterValue = abs(ParameterValue)
                                ParameterValue = numpy.log10(ParameterValue)                ## convert value to log scale

                        elif (ParameterName.strip() == "EM_RRL_uplimit" or ParameterName.strip() == "EM_RRL_lowlimit"):
                            LimitValuesTrueFalse = "false"
                            ParameterValue = float(ParameterValue)
                            if (ParameterValue != 0.E0):
                                if (ParameterValue < 0.E0):
                                    ParameterValue = abs(ParameterValue)
                                ParameterValue = numpy.log10(ParameterValue)                ## convert value to log scale

                        elif (ParameterName.strip() == "nHcolumn_cont_dust_uplimit" or ParameterName.strip() == "nHcolumn_cont_dust_lowlimit"):
                            LimitValuesTrueFalse = "false"
                            ParameterValue = float(ParameterValue)
                            if (ParameterValue != 0.E0):
                                if (ParameterValue < 0.E0):
                                    ParameterValue = abs(ParameterValue)
                                ParameterValue = numpy.log10(ParameterValue)                ## convert value to log scale

                        elif (ParameterName.strip() == "nHcolumn_uplimit" or ParameterName.strip() == "nHcolumn_lowlimit"):
                            LimitValuesTrueFalse = "false"
                            ParameterValue = float(ParameterValue)
                            if (ParameterValue != 0.E0):
                                if (ParameterValue < 0.E0):
                                    ParameterValue = abs(ParameterValue)
                                ParameterValue = numpy.log10(ParameterValue)                ## convert value to log scale

                        elif (ParameterName.endswith("_flag") or ParameterName.endswith("_uplimit") or ParameterName.endswith("_lowlimit") \
                            or ParameterName == "LayerDistance" or ParameterName == "ContFuncID_phen" or ParameterName == "keyword" \
                            or ParameterName == "CFFlag" or lowlimit == uplimit):
                            if (ParameterName.endswith("_flag")):
                                LocalFitFlag = LimitValuesTrueFalse
                            LimitValuesTrueFalse = "false"
                        else:
                            LimitValuesTrueFalse = LocalFitFlag


                    ## for column and hydrogen column density use a log scale for fit
                    if (LocalMolfitFileFormatVersion == "OLD"):
                        if (ParameterName.endswith("_FitFlag") or ParameterName == "ContFuncID_phen" or ParameterName == "keyword" \
                            or ParameterName == "LayerDistance" or ParameterName == "CFFlag" or lowlimit == uplimit):
                            LimitValuesTrueFalse = "false"

                    # Debug:
                    # print ("LimitValuesTrueFalse = ", LimitValuesTrueFalse)


                    ##------------------------------------------------------------------------------------------------------------------------------------


                    ## write name, value and optimization status for each parameter to xml file
                    NumberParameter += 1
                    NewInstanceFile.write("    <!-- define " + ParameterName + " -->\n")
                    NewInstanceFile.write("    <Parameter fit=" + chr(34) + LimitValuesTrueFalse + chr(34) + ">\n")
                    NewInstanceFile.write("        <name>" + ParameterName + "</name>\n")
                    NewInstanceFile.write("        <value>" + str(ParameterValue).strip() + "</value>\n")
                    NewInstanceFile.write("        <error> </error>\n")


                    ## check order of limit values
                    if (lowlimit > uplimit):                                                ## check, if upper and lower limit values have still
                        h = uplimit                                                         ## correct order
                        uplimit = lowlimit
                        lowlimit = h


                    ## write limits for each parameter to xml-file
                    if (LimitValuesTrueFalse=="false"):
                        NewInstanceFile.write("        <lowlimit>-1e99</lowlimit>\n")
                        NewInstanceFile.write("        <uplimit>1e99</uplimit>\n")
                    else:
                        NewInstanceFile.write("        <lowlimit>" + str(lowlimit) + "</lowlimit>\n")
                        NewInstanceFile.write("        <uplimit>" + str(uplimit) + "</uplimit>\n")
                    NewInstanceFile.write("    </Parameter>\n")
                    NewInstanceFile.write("\n")
                    NewInstanceFile.write("\n")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write end of xml-file
    NewInstanceFile.write("</ModelParameters>\n")


    ## close xml-file
    NewInstanceFile.close()


    ## update number of parameters
    task_MAGIX.WriteXMLtagNEW(xmlFileName, "NumberParameters", [str(NumberParameter)])

    # Debug:
    # print ("ok = ", ok)
    # print ("NumberParameter = ", NumberParameter)
    # sys.exit(0)


    ## define return parameter
    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## import the iso ratio file
##
def ImportIsoRatioFile(IsoRatioFileName, NewIsoTableFileName):
    """

input parameters:
-----------------

    - IsoRatioFileName:     path and name of iso ratio file which is imported

    - NewIsoTableFileName:  path and name of new iso ratio file


output parameters:
------------------

    - IsoRatioTable:        the final table containing molecules, isotopologues and the corresponding ratios

    - Isotopologues:        list containing the isotopologues

    - IsoMolecule:          list containing the corresponding molecules
    """

    # Debug:
    # print ("IsoRatioFileName = ", IsoRatioFileName)
    # print ("NewIsoTableFileName = ", NewIsoTableFileName)


    ## initialize output parameter
    IsoRatioTable = []
    Isotopologues = []
    IsoMolecule = []


    ## read in whole iso ratio file
    IsoRatioFileContents = []
    if (type(IsoRatioFileName) is str):
        IsoRatioFileName = IsoRatioFileName.strip()
        if (IsoRatioFileName != ""):
            if (IsoRatioFileName.endswith(".gz")):
                IsoRatioFile = gzip.open(IsoRatioFileName, 'rt')
            else:
                try:
                    IsoRatioFile = open(IsoRatioFileName)
                except:
                    IsoRatioFile = gzip.open(IsoRatioFileName + ".gz", 'rt')
            IsoRatioFileContents = IsoRatioFile.readlines()
            IsoRatioFile.close()
    else:
        IsoRatioFileContents = copy.deepcopy(IsoRatioFileName)


    ## analyze iso ratio table
    MaxCol = 0
    TotalGlobalIsoList = []
    for line in IsoRatioFileContents:
        StrippedLine = line.strip()


        ## remove comments
        i = StrippedLine.find("%")
        if (i > (-1)):
            StrippedLine = StrippedLine[:i].strip()
        if (StrippedLine != ""):


            ## check, if global ratios are defined in the current line
            LowerLine = StrippedLine.lower()
            if (LowerLine.startswith("global")):
                StrippedLine = StrippedLine.replace("GLOBAL__", "global ")
                StrippedLine = StrippedLine[6:].strip()
                SplittedLine = StrippedLine.split()
                TotalGlobalIsoList.append(SplittedLine[0])
                StrippedLine = "GLOBAL__" + StrippedLine


            ## get isotopologues, iso master, ratio, and lower and upper limits
            SplittedLine = StrippedLine.split()
            if (len(SplittedLine) > 2):
                LocalIsotopologue = SplittedLine[0]
                LocalIsoMaster = SplittedLine[1]
                MaxCol = max(MaxCol, len(SplittedLine))                                     ## determine max. number of columns in current iso ratio file


                ## store local isotopologue, isomaster, iso ratio, and lower and upper limits
                Isotopologues.append(LocalIsotopologue)
                IsoMolecule.append(LocalIsoMaster)
                if (len(SplittedLine) == 3):
                    IsoRatioTable.append([LocalIsotopologue, LocalIsoMaster, SplittedLine[2]])
                elif (len(SplittedLine) == 4):
                    IsoRatioTable.append([LocalIsotopologue, LocalIsoMaster, SplittedLine[2], SplittedLine[3]])
                elif (len(SplittedLine) == 5):
                    IsoRatioTable.append([LocalIsotopologue, LocalIsoMaster, SplittedLine[2], SplittedLine[3], SplittedLine[4]])
                elif (len(SplittedLine) == 6):
                    IsoRatioTable.append([LocalIsotopologue, LocalIsoMaster, SplittedLine[2], SplittedLine[3], SplittedLine[4], SplittedLine[5]])

    # Debug:
    # print ("TotalGlobalIsoList = ", TotalGlobalIsoList)
    # print ("MaxCol = ", MaxCol)


    ## if global defined iso ratios are used, modify limits for iso ratios, whose isotopologues contain the globally defined iso ratios
    if (TotalGlobalIsoList != []):
        NewIsoRatioTable = []
        for LocalLine in IsoRatioTable:
            if (len(LocalLine) == 3):                                                       ## current line of the iso ratio file contians no limit def.
                NewIsoRatioTable.append([LocalLine[0], LocalLine[1], LocalLine[2]])
            elif (len(LocalLine) == 5 or len(LocalLine) == 6):                              ## limits are defined
                LocalIsotopologue = LocalLine[0]
                if (LocalIsotopologue.find("GLOBAL__") > (-1)):
                    if (len(LocalLine) >= 5):
                        NewIsoRatioTable.append([LocalLine[0], LocalLine[1], LocalLine[2], LocalLine[3], LocalLine[4], LocalLine[5]])
                else:
                    IsoLowerLimit = LocalLine[3]
                    IsoUpperLimit = LocalLine[4]
                    NewIsoRatioTable.append([LocalLine[0], LocalLine[1], LocalLine[2], IsoLowerLimit, IsoUpperLimit])
        IsoRatioTable = copy.deepcopy(NewIsoRatioTable)

        # Debug:
        # print ("IsoRatioTable = ", IsoRatioTable)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if new file name is defined
    if (NewIsoTableFileName != ""):
        NewIsoRatioFile = open(NewIsoTableFileName, 'w')                                    ## open file


        ## define header line
        if (MaxCol <= 3):
            NewLine = "{:<45s} {:<45s} {:25s}".format("% isotopologue:", "Iso-master:", "Iso-ratio:")
        elif (MaxCol == 4):
            NewLine = "{:<45s} {:<45s} {:25s}       {:s}".format("% isotopologue:", "Iso-master:", "Iso-ratio:", "Add. file:")
        elif (MaxCol == 5):
            NewLine = "{:<45s} {:<45s} {:25s} {:25s} {:25s}".format("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
        elif (MaxCol >= 6):
            NewLine = "{:<45s} {:<45s} {:25s} {:25s} {:25s}       {:s}".format("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", \
                                                                               "upper-limit:", "Add. file:")
        NewIsoRatioFile.write(NewLine + "\n")


        ## write new formatted iso ratio file
        for IsoLine in IsoRatioTable:
            NewLine = "{:<45s} {:<45s} {:25s}".format(IsoLine[0], IsoLine[1], IsoLine[2])
            if (len(IsoLine) == 4):
                 NewLine += "       {:s}".format(IsoLine[3])
            elif (len(IsoLine) == 5):
                 NewLine += " {:25s} {:25s}".format(IsoLine[3], IsoLine[4])
            elif (len(IsoLine) == 6):
                 NewLine += " {:25s} {:25s}       {:s}".format(IsoLine[3], IsoLine[4], IsoLine[5])
            NewIsoRatioFile.write(NewLine + "\n")
        NewIsoRatioFile.close()

    # Debug:
    # print ("IsoRatioTable = ", IsoRatioTable)
    # print ("Isotopologues = ", Isotopologues)
    # print ("IsoMolecule = ", IsoMolecule)


    ## define output parameters
    return (IsoRatioTable, Isotopologues, IsoMolecule)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## read iso table parameters from iso table file and include parameters to xml file
##
def GetIsoTableParameters(printflag, IsoTableFileName, ParameterFileName):
    """

input parameters:
-----------------

    - print_flag:               flag if screen output is permitted (=true) or not

    - IsoTableFileName:         path and file name of the iso file

    - ParameterFileName:        path and file name of the instance XML file


output parameters:
------------------

    - ok:                       status; if everything is ok: ok = 0
    """

    # Debug:
    # print ("printflag = ", printflag)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("ParameterFileName = ", ParameterFileName)


    ## initialize return parameter
    ok = 0                                                                                  ## reset status variable


    ## read in contents of instance XML file
    ParameterFileName = ParameterFileName.strip()
    if (ParameterFileName != ""):
        if (ParameterFileName.endswith(".gz")):
            ParameterFile = gzip.open(ParameterFileName, 'rt')
        else:
            try:
                ParameterFile = open(ParameterFileName)
            except:
                ParameterFile = gzip.open(ParameterFileName + ".gz", 'rt')
    else:
        ok = 1
        return ok
    ParameterFileContents = ParameterFile.readlines()
    ParameterFile.close()


    ## read in contents of iso file
    counterIsoRatios = 0
    GlobalIsoRatioParameterList = []
    NumberOfGlobalISORatios = 0
    if (IsoTableFileName.strip() != ""):
        NewLocalIsoRatioFileName = ""
        IsoRatioTable, Isotopologues, IsoMolecule = ImportIsoRatioFile(IsoTableFileName, NewLocalIsoRatioFileName)
        counterIsoRatios = len(Isotopologues)


        ## analyze, if global iso ratios are defined
        GlobalIsoRatioList = []
        IsotopologuesIndex = (-1)
        for IsoRatioTableLine in IsoRatioTable:
            IsotopologuesIndex += 1
            LocalIsotopologue = IsoRatioTableLine[0]
            LocalIsoRatio = IsoRatioTableLine[2]
            if (LocalIsotopologue.find("GLOBAL__") > (-1)):
                LocalIsotopologueNew = LocalIsotopologue.replace("GLOBAL__", "")
                GlobalIsoRatioList.append([LocalIsotopologueNew, IsotopologuesIndex, LocalIsoRatio])

        # Debug:
        # print ("GlobalIsoRatioList = ", GlobalIsoRatioList)


        ## analyze each isotopologue
        NumberOfGlobalISORatios = len(GlobalIsoRatioList)
        if (len(GlobalIsoRatioList) > 0):
            for LocalIsotopologue in Isotopologues:
                GlobalIsoRatioParameter = []
                for GlobalIsoRatio in GlobalIsoRatioList:
                    iii = LocalIsotopologue.count(GlobalIsoRatio[0])
                    if (iii > 0):
                        IsotopologuesIndex = GlobalIsoRatio[1] + 1
                        GlobalIsoRatioParameter.append([IsotopologuesIndex, iii])
                GlobalIsoRatioParameterList.append(GlobalIsoRatioParameter)

        # Debug:
        # print ("\nGlobalIsoRatioParameterList = ", GlobalIsoRatioParameterList)


    ## write first part of instance XML file back to instance file
    NewParameterFile = open(ParameterFileName, 'w')
    counterLine = 0
    for line in ParameterFileContents:
        counterLine += 1
        if (counterLine == 6):
            Newline = line.replace("<NumberParameters>", " ")
            Newline = Newline.replace("</NumberParameters>", " ")
            NumParam = int(Newline.strip())

            # Debug:
            # print ("Newline = >>", Newline.strip(), "<<")
            # print ("NumParam = ", NumParam)

            NumParam += (counterIsoRatios * 6) + 2
            NewParameterFile.write("    <NumberParameters>" + str(NumParam) + "</NumberParameters>\n")
        else:
            if (counterLine < len(ParameterFileContents)):
                NewParameterFile.write(line)


    ## write separator
    NewParameterFile.write("\n")
    NewParameterFile.write("\n")
    star = "*" * 141
    NewParameterFile.write("    <!-- " + star + " -->\n")
    NewParameterFile.write("    <!-- third input file -->\n")
    NewParameterFile.write("\n")
    NewParameterFile.write("\n")
    NewParameterFile.write("    <!-- define number of iso ratios -->\n")
    NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    NewParameterFile.write("        <name>NumberOfISOLines</name>\n")
    NewParameterFile.write("        <value>" + str(counterIsoRatios) + "</value>\n")
    NewParameterFile.write("        <error> </error>\n")
    NewParameterFile.write("        <lowlimit>0</lowlimit>\n")
    NewParameterFile.write("        <uplimit>10000</uplimit>\n")
    NewParameterFile.write("    </Parameter>\n")


    ## define number of global iso ratio definitions
    NewParameterFile.write("\n")
    NewParameterFile.write("\n")
    NewParameterFile.write("    <!-- define number of global iso ratio definitions -->\n")
    NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    NewParameterFile.write("        <name>NumberOfGlobalISORatios</name>\n")
    NewParameterFile.write("        <value>" + str(NumberOfGlobalISORatios) + "</value>\n")
    NewParameterFile.write("        <error> </error>\n")
    NewParameterFile.write("        <lowlimit>0</lowlimit>\n")
    NewParameterFile.write("        <uplimit>10000</uplimit>\n")
    NewParameterFile.write("    </Parameter>\n")


    ## analyze iso file
    if (counterIsoRatios > 0):
        counterLines = 0
        for LineIndex in range(len(IsoRatioTable)):
            IsoRatioTableLine = IsoRatioTable[LineIndex]
            counterLines += 1                                                               ## increase counter for lines in iso file

            # Debug:
            # print ("IsoRatioTableLine = ", IsoRatioTableLine)


            ## write comment to xml file
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- line " + str(counterLines) + ":  ratio: " + IsoRatioTableLine[0] + " / " + IsoRatioTableLine[1] + " -->\n")


            ## write name of iso molecule
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoMolecule -->\n")
            NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoMolecule</name>\n")
            NewParameterFile.write("        <value>" + IsoRatioTableLine[0].strip() + "</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
            NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


            ## write name of iso molecule
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoMasters -->\n")
            NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoMasters</name>\n")
            NewParameterFile.write("        <value>" + IsoRatioTableLine[1].strip() + "</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
            NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


            ## check limit specification
            fitFlag = "false"
            if (len(IsoRatioTableLine) >= 5):
                lowerlimit = float(IsoRatioTableLine[3].strip())
                upperlimit = float(IsoRatioTableLine[4].strip())
                if (lowerlimit < upperlimit):
                    fitFlag = "true"


            ## write name of iso ratio
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoMasters -->\n")
            if (fitFlag == "true"):
                NewParameterFile.write("    <Parameter fit=" + chr(34) + "true" + chr(34) + ">\n")
            else:
                NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoRatio</name>\n")
            NewParameterFile.write("        <value>" + IsoRatioTableLine[2].strip() + "</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            if (fitFlag == "true"):
                NewParameterFile.write("        <lowlimit>" + IsoRatioTableLine[3].strip() + "</lowlimit>\n")
                NewParameterFile.write("        <uplimit>" + IsoRatioTableLine[4].strip() + "</uplimit>\n")
            else:
                NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
                NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


            ## write value of lower limit
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoLowerLimit -->\n")
            NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoLowerLimit</name>\n")
            if (fitFlag == "true"):
                NewParameterFile.write("        <value>" + IsoRatioTableLine[3].strip() + "</value>\n")
            else:
                NewParameterFile.write("        <value>-1e9</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
            NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


            ## write value of upper limit
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoUpperLimit -->\n")
            NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoUpperLimit</name>\n")
            if (fitFlag == "true"):
                NewParameterFile.write("        <value>" + IsoRatioTableLine[4].strip() + "</value>\n")
            else:
                NewParameterFile.write("        <value>-1e9</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
            NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


            ## write parameters for global iso ratio definitions
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoGlobalParameters -->\n")
            NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoGlobalParameters</name>\n")
            if (GlobalIsoRatioParameterList == []):
                NewParameterFile.write("        <value> </value>\n")
            else:
                LocalGlobalIsoRatioParameter = GlobalIsoRatioParameterList[LineIndex]
                LocalGlobalIsoRatioParameterString = ""
                iii = 0
                for LocalParam in LocalGlobalIsoRatioParameter:
                    LocalParamString = str(LocalParam[0]) + ", " + str(LocalParam[1])
                    iii += 1
                    if (iii == 1):
                        LocalGlobalIsoRatioParameterString = LocalParamString
                    else:
                        LocalGlobalIsoRatioParameterString += ", " + LocalParamString
                NewParameterFile.write("        <value> " + LocalGlobalIsoRatioParameterString.strip() + "</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
            NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


    ## write end of xml-file
    NewParameterFile.write("</ModelParameters>\n")


    ## close new parameter file
    NewParameterFile.close()

    # Debug:
    # print ("ok = ", ok)


    ## define return parameter
    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## merge molfit and iso ratio file into one big molfit file
##
def MergeMolfitAndIsoRatio(MolfitsFileName, IsoRatioFileName, NewMolfitFileName, MinNtot = 1.e12):
    """

input parameters:
-----------------

    - MolfitsFileName:          path and name of molfit file

    - IsoRatioFileName:         path and name of iso ratio file

    - NewMolfitFileName:        path and name of new molfit file

    - MinNtot:                  (optional) minimal column density for a component (default: 1.e12)


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("IsoRatioFileName = ", IsoRatioFileName)
    # print ("NewMolfitFileName = ", NewMolfitFileName)
    # print ("MinNtot = ", MinNtot)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze molfit file


    ## get sql parameters in molfit file
    MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = GetSQLParameter(MolfitsFileName)
    SQLParamArray = [MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL]
    MolfitHeaderLinesList = WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)

    # Debug:
    # print ("MinNumTransitionsSQL = ", MinNumTransitionsSQL)
    # print ("MaxNumTransitionsSQL = ", MaxNumTransitionsSQL)
    # print ("MaxElowSQL = ", MaxElowSQL)
    # print ("MingASQL = ", MingASQL)
    # print ("OrderTransSQL = ", OrderTransSQL)


    ## analyze given molfit file
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = AnalyzeMolfitFile(MolfitsFileName)

    # Debug:
    # print ("\n\nMoleculesInMolfitFile = ", MoleculesInMolfitFile)
    # print ("AllParameters = ", AllParameters)
    # print ("MolfitFileForEachMolecule = ", MolfitFileForEachMolecule)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze iso ratio file
    dummyName = ""
    IsoRatioTable, Isotopologues, IsoMolecule = ImportIsoRatioFile(IsoRatioFileName, dummyName)

    # Debug:
    # print ("IsoRatioTable = ", IsoRatioTable)


    ## analyze, if global iso ratios are defined
    GlobalIsoRatioList = []
    IsotopologuesIndex = (-1)
    for IsoRatioTableLine in IsoRatioTable:
        IsotopologuesIndex += 1
        LocalIsotopologue = IsoRatioTableLine[0]
        LocalIsoRatio = IsoRatioTableLine[2]
        if (LocalIsotopologue.find("GLOBAL__") > (-1)):
            LocalIsotopologueNew = LocalIsotopologue.replace("GLOBAL__", "")
            GlobalIsoRatioList.append([LocalIsotopologueNew, IsotopologuesIndex, LocalIsoRatio])

    # Debug:
    # print ("GlobalIsoRatioList = ", GlobalIsoRatioList)


    ## analyze each isotopologue
    NumberOfGlobalISORatios = len(GlobalIsoRatioList)
    if (len(GlobalIsoRatioList) > 0):
        for IsoID, LocalIsotopologue in enumerate(Isotopologues):
            for GlobalIsoRatio in GlobalIsoRatioList:
                iii = LocalIsotopologue.count(GlobalIsoRatio[0])
                if (iii > 0):
                    factor = float(GlobalIsoRatio[2])
                    IsoRatioTable[IsoID][2] = float(IsoRatioTable[IsoID][2]) * (factor**iii)

    # Debug:
    # print ("IsoRatioTable = ", IsoRatioTable)
    # print ("Isotopologues = ", Isotopologues)
    # print ("IsoMolecule = ", IsoMolecule)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## open molfit file
    LocalMolfitFile = open(NewMolfitFileName, 'w')


    ## add SQL parameter definitions
    for NewLine in MolfitHeaderLinesList:
        LocalMolfitFile.write(NewLine)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## loop over all molecules in molfit file
    AlreadyKnownMoleculeList = []
    for MoleculeIndex in range(len(MoleculesInMolfitFile)):                                 ## loop over molecules in molfit file
        LocalMoleculeName = MoleculesInMolfitFile[MoleculeIndex]

        # Debug:
        # print ("LocalMoleculeName = ", LocalMoleculeName)
        # print ("IsoMolecule = ", IsoMolecule)
        # print (LocalMoleculeName in IsoMolecule)


        ## add parameters of molfit molecules
        for line in MolfitFileForEachMolecule[MoleculeIndex]:
            LocalMolfitFile.write(line)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## loop over all isotopologues and vibrational excited states related to the current molecule
        for IsoLine in IsoRatioTable:
            LocalIsotopologue = IsoLine[0]
            LocalIsoMaster = IsoLine[1]
            LocalRatio = IsoLine[2]                                                         ## global ratios are NOT taken into account
            if (LocalIsoMaster == LocalMoleculeName):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## create molfit file for current molecule


                ## add name of molecule
                LocalMolfitParameters = AllParameters[MoleculeIndex]
                NewLine = LocalIsotopologue + "   " + str(len(LocalMolfitParameters)) + "\n"
                LocalMolfitFile.write(NewLine)


                ## add molecule parameters
                NewLine = ""
                for LocalLine in LocalMolfitParameters:

                    # Debug:
                    # print ("LocalLine = ", LocalLine)

                    FitFlag = False
                    NewComp = " "
                    for col in LocalLine:
                        Name = col[0]
                        flag = col[1]
                        element = col[4]

                        # Debug:
                        # print ("col = ", col)
                        # print ("Name = ", Name)


                        ## get fit flag
                        if (Name.endswith("_flag") or Name.endswith("_FitFlag")):
                            FitFlag = element.strip().lower()
                            NewComp += "    {:2s}".format(FitFlag)


                        ## ignore limits
                        elif (Name.endswith("_lowlimit") or Name.endswith("_uplimit")):
                            value = float(element)
                            NewComp += "  {:12.5e}".format(value)


                        ## store values for source size
                        if (Name in ["source_size", "source_radius", "source_center_x", "source_center_y", "T_rot", "T_e", "T_cont_dust", \
                                     "T_dOff", "T_dSlope", "EM_RRL", "nHcolumn_cont_dust", "nHcolumn", "V_width_Gauss", "V_off", \
                                     "beta_cont_dust", "kappa_cont_dust", "kappa", "beta", \
                                     "ContFuncID_param_1", "ContFuncID_param_2", "ContFuncID_param_3", "ContFuncID_param_4", "ContFuncID_param_5"]):
                            value = float(element)
                            NewComp += "  {:12.5e}".format(value)


                        ## resize column density
                        if (Name in ["N_tot"]):
                            value = float(element) / float(LocalRatio)
                            NewComp += "  {:12.5e}".format(value)


                        ## store values for EA-Flag
                        elif (Name in ["LayerDistance", "ContFuncID_phen", "CFFlag", "keyword"]):
                            NewComp += "    " + element


                    ## add line break
                    NewLine += NewComp + "\n"


                ## write content to file
                LocalMolfitFile.write(NewLine)


    ## we're done
    LocalMolfitFile.close()


    ## analyze final molfit file
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = AnalyzeMolfitFile(NewMolfitFileName)


    ## bring new molfit file in proper format
    LocalMinColumnDensityEmis = MinNtot
    LocalMinColumnDensityAbs = MinNtot
    task_LineIdentification.CreateMolfitFileInNewFormat(NewMolfitFileName, MolfitFileForEachMolecule, LocalMinColumnDensityAbs, \
                                                        LocalMinColumnDensityEmis, SQLParameters = SQLParamArray)

    ## we're done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## call myXCLASS program from command line
##
def CallmyXCLASS(myXCLASSrootDir, CurrentWorkingDir, FreqMin, FreqMax, FreqStep, TelescopeSize, Inter_Flag, t_back_flag, tBack, tSlope, nH_flag, N_H, \
                 beta_dust, kappa_1300, DustFileName, vLSR, iso_flag, xmlFileName, dbFile, printflag, SpectrumOnlyFlag, \
                 NumModelPixelXX, NumModelPixelYY, LocalOverlapFlag, BMIN, BMAJ, BPA, Redshift, \
                 ContPhenFuncID, ContPhenFuncParam1, ContPhenFuncParam2, ContPhenFuncParam3, ContPhenFuncParam4, ContPhenFuncParam5, \
                 NoSubBeamFlag = False, EmAbsFlag = False, EmAbsPATH = None, BackgroundFileName = None):
    """

input parameters:
-----------------

    - myXCLASSrootDir:          path of myXCLASS directory

    - CurrentWorkingDir:        path of current working directory

    - FreqMin:                  start frequency of simulated spectrum (in K)

    - FreqMax:                  end frequency of simulated spectrum (in K)

    - FreqStep:                 step frequency of simulated spectrum (in K)

    - TelescopeSize:            for single dish observations (Inter_Flag = F): TelescopeSize
                                describes the size of telescope (in m); for
                                interferometric observations (Inter_Flag = T): TelescopeSize
                                describes the interferometric beam FWHM size (in arcsec),

    - Inter_Flag:               defines, if single dish ("F") or interferometric
                                observations ("T") are described.

    - t_back_flag:              ("T"/"F"), defines, if the user defined background
                                temperature Tbg and temperature slope Tslope given
                                by the input parameters tBack and tslope describe the
                                continuum contribution completely (t_back_flag = "T")
                                or not (t_back_flag = "F").

    - tBack:                    background temperature (in K).

    - tslope:                   temperature slope (dimensionless).

    - nH_flag:                  ("T"/"F"), defines, if column density, beta for dust
                                and kappa are given by the molfit file ("F")
                                or if nH_flag is set to "T", the following
                                three parameters define the H density, beta for
                                dust and kappa for all components.

    - N_H:                      (has to be given only if nH_flag is set to "T")
                                Hydrogen column density (in cm^{-2}).

    - beta_dust:                (has to be given only if nH_flag is set to "T")
                                beta for dust (dimensionless).

    - kappa_1300:               (has to be given only if nH_flag is set to "T")
                                kappa (cm^2 g^{-1}).

    - DustFileName:             path and name of file describing dust contribution.

    - MolfitsFileName:          ABSOLUTE path and file name of the molfits file,
                                including the source_size, T_rot (rotation temperature),
                                N_tot (total column density), V_width (velocity
                                width), V_off (velocity offset), CFFlag
                                (core or foreground)

    - iso_flag:                 use isotopologues ("T"/"F"). If iso_flag is set to "F"
                                the ratios are all set to 1.

    - RestFreq:                 rest frequency in MHz (default: 0). (If this parameter
                                is set to zero, the intensity is plotted against
                                frequency (in MHz) otherwise against velocity (in km/s).

    - vLSR:                     velocity (local standard of rest) in km/s, only used,
                                if RestFreq /= 0. (velocity(Frequency = RestFreq) = vLSR)

    - dbFile:                   path and name of database file

    - printflag:                screen output?

    - SpectrumOnlyFlag:         flag indicating that only spectrum is calculated without component spectra

    - NumModelPixelXX:          number in x-direction for sub-beam description

    - NumModelPixelYY:          number in y-direction for sub-beam description

    - LocalOverlapFlag:         take local overlap into account or not

    - BMIN:                     telescope BMIN

    - BMAJ:                     telescope BMAJ

    - BPA:                      telescope BPA

    - Redshift:                 red shifts

    - ContPhenFuncID:           identification number of continuum function

    - ContPhenFuncParam1:       parameter 1 for continuum function

    - ContPhenFuncParam2:       parameter 2 for continuum function

    - ContPhenFuncParam3:       parameter 3 for continuum function

    - ContPhenFuncParam4:       parameter 4 for continuum function

    - ContPhenFuncParam5:       parameter 5 for continuum function

    - NoSubBeamFlag:            (optional) deactivate sub beam description

    - EmAbsFlag:                (optional) flag indicating that emission and absorption
                                as function of frequency are stored (default: False)

    - EmAbsPATH:                (optional) path containing files describing functions
                                emission / absorption (default: None)

    - BackgroundFileName:       (optional) path and name of background file


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("CurrentWorkingDir = ", CurrentWorkingDir)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("FreqStep = ", FreqStep)
    # print ("TelescopeSize = ", TelescopeSize)
    # print ("Inter_Flag = ", Inter_Flag)
    # print ("t_back_flag = ", t_back_flag)
    # print ("tBack = ", tBack)
    # print ("tSlope = ", tSlope)
    # print ("nH_flag = ", nH_flag)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("vLSR = ", vLSR)
    # print ("iso_flag = ", iso_flag)
    # print ("xmlFileName = ", xmlFileName)
    # print ("dbFile = ", dbFile)
    # print ("printflag = ", printflag)
    # print ("SpectrumOnlyFlag = ", SpectrumOnlyFlag)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("BMIN = ", BMIN)
    # print ("BMAJ = ", BMAJ)
    # print ("BPA = ", BPA)
    # print ("Redshift = ", Redshift)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)
    # print ("EmAbsFlag = ", EmAbsFlag)
    # print ("EmAbsPATH = ", EmAbsPATH)
    # print ("BackgroundFileName = ", BackgroundFileName)


    ##====================================================================================================================================================
    ## copy all required files to temp directory
    myXCLASSsrcDir = os.path.normpath(myXCLASSrootDir) + "/"
    myXCLASSexe = "myNewXCLASS.exe"
    command_string = "cp " + myXCLASSsrcDir + myXCLASSexe + " " + CurrentWorkingDir         ## copy myXCLASS exe file to current working directory
    os.system(command_string)


    ##====================================================================================================================================================
    ## construct command to start myXCLASS program


    ## get current directory
    WorkingDirectory = os.getcwd() + "/"


    ## change to current working directory
    myXCLASScommand = "cd " +  CurrentWorkingDir + "; "


    ## start myXCLASS program
    dbgFlag = ""
    # dbgFlag = "prof"
    if (dbgFlag == "db"):
        myXCLASScommand += "gdb --args "
    elif (dbgFlag == "prof"):
        myXCLASScommand += "valgrind --leak-check=full --tool=memcheck ./"
    else:
        myXCLASScommand += "./"
    myXCLASScommand += myXCLASSexe


    ## add min., max., and step frequency
    myXCLASScommand += " --FreqMin=" + str(FreqMin)
    myXCLASScommand += " --FreqMax=" + str(FreqMax)
    myXCLASScommand += " --FreqStep=" + str(FreqStep)


    ## add telescope parameters
    if (TelescopeSize is not None and TelescopeSize != 0.0):
        myXCLASScommand += " --TelescopeSize=" + str(TelescopeSize)
    if (BMIN is not None and BMIN != 0.0):
        myXCLASScommand += " --BMIN=" + str(BMIN)
    if (BMAJ is not None and BMAJ != 0.0):
        myXCLASScommand += " --BMAJ=" + str(BMAJ)
    if (BPA is not None):
        myXCLASScommand += " --BPA=" + str(BPA)


    ## add telescope parameters
    myXCLASScommand += " --Inter_Flag=" + str(Inter_Flag)


    ## red shift
    if (Redshift is not None and Redshift != 0.0):
        myXCLASScommand += " --Redshift=" + str(Redshift)


    ## add phenomenological background parameter
    myXCLASScommand += " --t_back_flag=" + str(t_back_flag) + " --tBack=" + str(tBack) + " --tSlope=" + str(tSlope)


    ## add global dust parameters
    if (N_H is not None and N_H != 0.0):
        myXCLASScommand += " --N_H=" + str(N_H)
    if (beta_dust is not None and kappa_1300 is not None and kappa_1300 != 0.0):
        myXCLASScommand += " --beta_dust=" + str(beta_dust) + " --kappa_1300=" + str(kappa_1300)


    ## add parameter for phenomenological continuum description
    if (ContPhenFuncID is not None):
        myXCLASScommand += " --ContPhenID=" + str(ContPhenFuncID)
        if (ContPhenFuncParam1 is not None):
            myXCLASScommand += " --ContParam1=" + str(ContPhenFuncParam1)
        if (ContPhenFuncParam2 is not None):
            myXCLASScommand += " --ContParam2=" + str(ContPhenFuncParam2)
        if (ContPhenFuncParam3 is not None):
            myXCLASScommand += " --ContParam3=" + str(ContPhenFuncParam3)
        if (ContPhenFuncParam4 is not None):
            myXCLASScommand += " --ContParam4=" + str(ContPhenFuncParam4)
        if (ContPhenFuncParam5 is not None):
            myXCLASScommand += " --ContParam5=" + str(ContPhenFuncParam5)


    ## parameters for sub-beam description
    if (NumModelPixelXX is not None and NumModelPixelXX != 0.0):
        myXCLASScommand += " --NumModelPixelXX=" + str(NumModelPixelXX)


    ## parameters for sub-beam description
    if (NumModelPixelYY is not None and NumModelPixelYY != 0.0):
        myXCLASScommand += " --NumModelPixelYY=" + str(NumModelPixelYY)


    ## add flag for local overlap
    myXCLASScommand += " --LocalOverlapFlag=" + str(LocalOverlapFlag)


    ## add flag for sub-beam description
    if (NoSubBeamFlag):
        myXCLASScommand += " --NoSubBeamFlag=" + str(NoSubBeamFlag)


    ## add flag for sub-beam description
    if (EmAbsFlag):
        myXCLASScommand += " --EmAbsFlag=" + str(EmAbsFlag)


    ## add path and name of dust file
    if (DustFileName is not None):
        DustFileName = DustFileName.strip()
        if (DustFileName != ""):
            PureDustFileName = DustFileName.split("/")
            PureDustFileName = PureDustFileName[-1].strip()
            if (os.path.dirname(DustFileName) + "/" != CurrentWorkingDir):
                cmdString = "cp " + PureDustFileName + " " + CurrentWorkingDir
                os.system(cmdString)
            myXCLASScommand += " --DustFileName=" + PureDustFileName


    ## add path and name of background file
    if (BackgroundFileName is not None):
        BackgroundFileName = BackgroundFileName.strip()
        if (BackgroundFileName != ""):
            PureBackgroundFileName = BackgroundFileName.split("/")
            PureBackgroundFileName = PureBackgroundFileName[-1].strip()
            cmdString = "cp " + BackgroundFileName + " " + CurrentWorkingDir
            os.system(cmdString)
            myXCLASScommand += " --BackgroundFileName=" + PureBackgroundFileName


    ## add global v_lsr
    if (vLSR is not None):
        myXCLASScommand += " --vLSR=" + str(vLSR)


    ## add path and name of parameter xml file
    PureXMLFileName = xmlFileName.split("/")
    PureXMLFileName = PureXMLFileName[-1].strip()
    myXCLASScommand += " --in=" + PureXMLFileName


    ## add for iso_flag
    if (iso_flag is not None):
        myXCLASScommand += " --iso_flag=" + str(iso_flag)


    ## add output parameter (a negative number of processors indicates that all output files are generated)
    if (SpectrumOnlyFlag):
        i = 1
    else:
        i = (-1)
    myXCLASScommand += " --log=" + str(i)


    ## add path of emission/absorption files
    if (EmAbsPATH is not None):
        EmAbsPATH = EmAbsPATH.strip()
        if (EmAbsPATH != ""):
            myXCLASScommand += " --EmAbsPATH=" + EmAbsPATH.strip()


    ## add path and name of used database file
    if (dbFile is not None):
        dbFile = dbFile.strip()
        if (dbFile != ""):
            myXCLASScommand += " --dbFile=" + dbFile.strip()


    ## write screen output to file?
    myXCLASScommand += " "
    if (not printflag):
        myXCLASScommand += "> screen.out 2>&1"


    ## remove myXCLASS program exe file
    myXCLASScommand += "; rm -rf " + myXCLASSexe + "; "
    myXCLASScommand += "cd " + WorkingDirectory

    # Debug:
    # print ("myXCLASScommand = ", myXCLASScommand)


    ##====================================================================================================================================================
    ## start myXCLASS
    os.system(myXCLASScommand)


    ## we're done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
##
## The function defines the interface for the myXCLASS program
##
##
def myXCLASSCore(MolfitsFileName = "", MolfitFileName = None, \
                 ObsXMLFileName = None, \
                 FreqMin = 1.0, FreqMax = 1.e8, FreqStep = 1.0, \
                 TelescopeSize = None, BMIN = None, BMAJ = None, BPA = None, \
                 Redshift = None, Inter_Flag = False, \
                 t_back_flag = True, tBack = 0.0, tSlope = 0.0, \
                 nH_flag = False, N_H = 0.0, beta_dust = 0.0, kappa_1300 = 0.0, DustFileName = "", \
                 BackgroundFileName = "", \
                 ContPhenFuncID = None, ContPhenFuncParam1 = None, ContPhenFuncParam2 = None, \
                 ContPhenFuncParam3 = None, ContPhenFuncParam4 = None, ContPhenFuncParam5 = None, \
                 iso_flag = False, IsoTableFileName = "", \
                 NumModelPixelXX = 100, NumModelPixelYY = 100, \
                 LocalOverlapFlag = False, NoSubBeamFlag = False, \
                 dbFileName = "", CheckMoleculeInDB = True, \
                 RestFreq = 0.0, vLSR = 0.0, \
                 EmAbsFlag = False, EmAbsPATH = None, \
                 printFlag = None, SpectrumOnlyFlag = False):
    """

input parameters:
-----------------

    - MolfitsFileName:          path and file name of the molfits file, (default: "").

    - MolfitFileName:           (optional) alias of MolfitsFileName, (default: None).

    - ObsXMLFileName:           (optional) path and name of an obs. xml file (default: None).

    - FreqMin:                  (optional) start frequency of simulated spectrum (in K),
                                (default: 1).

    - FreqMax:                  (optional) end frequency of simulated spectrum (in K),
                                (default: 1.e8).

    - FreqStep:                 (optional) step frequency of simulated spectrum (in K), (default: 1).

    - TelescopeSize:            for single dish observations (Inter_Flag = False): TelescopeSize
                                describes the size of telescope (in m), (default: 1); for
                                interferometric observations (Inter_Flag = True): TelescopeSize
                                describes the interferometric beam FWHM size (in arcsec),
                                (default: 1).

    - Inter_Flag:               defines, if single dish ("False") or interferometric
                                observations ("True") are described, (default: "False").

    - BMIN:                     (optional) Beam minor axis length (in arcsec) (default: None)

    - BMAJ:                     (optional) Beam major axis length (in arcsec) (default: None)

    - BPA:                      (optional) Beam position angle (in degree) (default: None)

    - Redshift:                 (optional) red shift (dimensionless), (default: None)

    - t_back_flag:              ("True"/"False"), defines, if the user defined background
                                temperature Tbg and temperature slope TSlope given
                                by the input parameters tBack and tSlope describe the
                                continuum contribution completely (t_back_flag = "True")
                                or not (t_back_flag = "False") (default: "True").

    - tBack:                    background temperature (in K), (default: 0.0).

    - tSlope:                   temperature slope (dimensionless), (default: 0.0).

    - N_H:                      (optional) Hydrogen column density (in cm^{-2}), (default: None).

    - beta_dust:                (optional) beta for dust (dimensionless), (default: None).

    - kappa_1300:               (optional) kappa (cm^2 g^{-1}), (default: None).

    - DustFileName:             (optional) path and name of file describing the optical depth
                                of dust (default: None).

    - BackgroundFileName:       (optional) path and name of file describing background (default: None).

    - ContPhenFuncID:           (optional) describes which phenomenological function is used
                                to describe the continuum (default: None).

    - ContPhenFuncParam1:       (optional) first parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam2:       (optional) second parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam3:       (optional) third parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam4:       (optional) fourth parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam5:       (optional) fifth parameter for phenomenological function
                                describing the continuum (default: None).

    - iso_flag:                 use isotopologues ("True"/"False"). If iso_flag is set to "False"
                                the ratios are all set to 1 (default: "False").

    - IsoTableFileName:         (has to be given only if iso_flag is set to "True")
                                ABSOLUTE path and file name of an ASCII file including
                                the iso ratios between certain molecules. The so-called
                                "iso ratio file" defines the iso ratios between
                                molecules. The ASCII file consists of three columns,
                                where the first two columns indicates the molecules,
                                respectively. The third column defines the ratio for
                                both molecules. The columns are separated by blanks or
                                tabs. So, the names of the molecules must not contain
                                blanks. Comments have to be marked with the "%" sign,
                                i.e. all characters on the right side of a "%" sign
                                are ignored.

                                The myXCLASSFit function offers the possibility to
                                optimize the ratios between isotopologues as well. For
                                that purpose, the user has to add two additional columns
                                on the right indicating the lower and the upper limit
                                of a certain ratio, respectively.

    - NumModelPixelXX:          (optional) used for sub-beam modeling, describes the number
                                of pixels used in x-direction (default: None).

    - NumModelPixelYY:          (optional) used for sub-beam modeling, describes the number
                                of pixels used in y-direction (default: None).

    - LocalOverlapFlag:         (optional) take local overlap into account (default: False).

    - NoSubBeamFlag:            (optional) do not use sub-beam description (default: False).

    - SpectrumOnlyFlag:         (optional) flag indicating that only spectrum is calculated
                                (default: False)

    - EmAbsFlag:                (optional) flag indicating that emission and absorption
                                as function of frequency are stored (default: False)

    - EmAbsPATH:                (optional) path containing files describing emission / absorption functions
                                (default: None)

    - CheckMoleculeInDB:        (optional) check, if molecules included in molfit file have transitions
                                within given frequency range (default: True)

    - RestFreq:                 rest frequency in MHz (default: 0). (If this parameter
                                is set to zero, the intensity is plotted against
                                frequency (in MHz) otherwise against velocity (in km/s).

    - vLSR:                     velocity (local standard of rest) in km/s, only used,
                                if RestFreq /= 0. (velocity(Frequency = RestFreq) = vLSR)

    - dbFileName:               path and name of database file

    - printFlag:                (optional) flag for printing messages to screen (True/False)


output parameters:
------------------

    - myXCLASSspectrumList:     this variable contains the calculated myXCLASS spectrum

    - myXCLASSlogList:          this variable contains the corresponding log file

    - myXCLASSTransList:        list of transition energies with the corresponding molecule names within the defined range

    - myXCLASSIntOpticalList    contains intensities and optical depths for each molecule and component


                                The intensities and optical depths are stored as follows:


                                myXCLASSIntOptical[0], contains all information about the intensities

                                    myXCLASSIntOptical[0][i][0]    contains the name of the ith molecule

                                    myXCLASSIntOptical[0][i][1]    contains the index of the component of the ith molecule

                                    myXCLASSIntOptical[0][i][2]    contains the intensities of the ith molecule as a 2D python list

                                    myXCLASSIntOptical[0][i][3]    contains the integrated intensities of the ith molecule


                                myXCLASSIntOptical[1], contains all information about the optical depths.

                                    myXCLASSIntOptical[1][i][0]    contains the name of the ith molecule

                                    myXCLASSIntOptical[1][i][1]    contains the index of the component of the ith molecule

                                    myXCLASSIntOptical[1][i][2]    contains the optical depth of the ith molecule as a 2D python list

    - myXCLASSJobDir:           absolute path of the myXCLASS job directory created for the current run
    """

    # Debug:
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("MolfitFileName = ", MolfitFileName)
    # print ("ObsXMLFileName = ", ObsXMLFileName)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("FreqStep = ", FreqStep)
    # print ("TelescopeSize = ", TelescopeSize)
    # print ("Inter_Flag = ", Inter_Flag)
    # print ("t_back_flag = ", t_back_flag)
    # print ("tBack = ", tBack)
    # print ("tSlope = ", tSlope)
    # print ("nH_flag = ", nH_flag)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("iso_flag = ", iso_flag)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("RestFreq = ", RestFreq)
    # print ("vLSR = ", vLSR)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)
    # print ("SpectrumOnlyFlag = ", SpectrumOnlyFlag)
    # print ("EmAbsFlag = ", EmAbsFlag)
    # print ("EmAbsPATH = ", EmAbsPATH)
    # print ("dbFileName = ", dbFileName)
    # print ("printFlag = ", printFlag)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("BMIN = ", BMIN)
    # print ("BMAJ = ", BMAJ)
    # print ("BPA = ", BPA)
    # print ("Redshift = ", Redshift)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("CheckMoleculeInDB = ", CheckMoleculeInDB)


    ##====================================================================================================================================================
    ## initialize function


    ## define name of function
    NameOfFunction = "myXCLASS"


    ## get current working directory
    CurrentDir = os.getcwd() + "/"


    ## initialize return parameters
    myXCLASSspectrumList = []
    myXCLASSlogList = []
    myXCLASSTransList = []
    myXCLASSIntOpticalList = []
    myXCLASSJobDir = ""


    ##====================================================================================================================================================
    ## create run directory for the current function call
    LocalPrintFlag = printFlag
    myXCLASSrootDir = GetmyXCLASSRootDir()
    myXCLASSJobDir = task_MAGIX.CreateRunDirectory(NameOfFunction, myXCLASSrootDir, LocalPrintFlag)

    # Debug:
    # print ("myXCLASSJobDir = ", myXCLASSJobDir)


    ## extend sys.path
    MAGIXrootDir = GetMAGIXRootDir()
    task_MAGIX.SetMAGIXEnvironment(MAGIXrootDir)


    ##====================================================================================================================================================
    ## molfits file:


    ## get alias for keyword "MolfitFileName"
    MolfitsFileName = MolfitsFileName.strip()
    if (MolfitsFileName == "" and MolfitFileName is not None):
        MolfitsFileName = MolfitFileName.strip()


    ## analyze molfit parameter: check path and name of the molfit xml-file and copy molfit file to temp directory
    FreqMinList = [[0, 0, FreqMin]]
    FreqMaxList = [[0, 0, FreqMax]]
    GlobalvLSRList = [[0, vLSR]]
    RedshiftList = [[0, Redshift]]
    if (dbFileName is None or dbFileName == ""):
        dbFileName = GetDefaultDBFile()
    dbList = [dbFileName]
    if (CheckMoleculeInDB):
        ok, dbFileName, LocalMolfitsFileName, PureNameMolfitsFile = task_myXCLASSFit.CheckMolfitFile(MolfitsFileName, NameOfFunction, CurrentDir, \
                                                                                                     myXCLASSJobDir, FreqMinList, FreqMaxList, \
                                                                                                     GlobalvLSRList, RedshiftList, dbList)
        if (ok == 1):                                                                       ## an error occurred
            return (myXCLASSspectrumList, myXCLASSlogList, myXCLASSTransList, myXCLASSIntOpticalList, myXCLASSJobDir)
    else:
        LocalMolfitsFileName = MolfitsFileName
        PureNameMolfitsFile = os.path.basename(MolfitsFileName)

    # Debug:
    # print ("ok = ", ok)
    # print ("LocalMolfitsFileName = ", LocalMolfitsFileName)
    # print ("PureNameMolfitsFile = ", PureNameMolfitsFile)


    ##====================================================================================================================================================
    ## check, if obs. xml file is given
    if (ObsXMLFileName is not None):


        ## import parameters from obs. xml file
        ExpFileList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "FileNamesExpFiles")          ## get path and name of obs. data file
        NumExpDataFiles = len(ExpFileList)                                                  ## get number of obs. data files
        NumberExpRangesList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "NumberExpRanges")
        FreqMinList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "MinExpRange")
        FreqMaxList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "MaxExpRange")
        FreqStepList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "StepFrequency")
        t_back_flagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "t_back_flag")
        tBackList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "BackgroundTemperature")
        tSlopeList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "TemperatureSlope")
        N_HList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "HydrogenColumnDensity")
        beta_dustList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "DustBeta")
        kappa_1300List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "Kappa")
        DustFileNameList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "DustFileName")
        BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "BackgroundFileName")
        ContPhenFuncIDList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncID")
        ContPhenFuncParam1List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncParam1")
        ContPhenFuncParam2List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncParam2")
        ContPhenFuncParam3List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncParam3")
        ContPhenFuncParam4List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncParam4")
        ContPhenFuncParam5List = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "ContPhenFuncParam5")
        NoiseList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "NoiseLevel")
        SmoothList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "SmoothValue")
        TelescopeSizeList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "TelescopeSize")
        BMINList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "BMIN")
        BMAJList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "BMAJ")
        BPAList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "BPA")
        Inter_FlagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "Inter_Flag")
        GlobalvLSRList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "GlobalvLSR")
        RedshiftList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "Redshift")
        IsotopologuesList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "iso_flag")
        if (IsotopologuesList == []):
            IsotopologuesList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "Isotopologues")
        IsoTableFileNameList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "IsoTableFileName")
        IsoFlag = IsotopologuesList[0].lower()
        IsoFlag = task_myXCLASSMapFit.CheckBool(IsoFlag)
        if (IsoFlag):
            IsoFileName = IsoTableFileNameList[0]
        else:
            IsoFileName = ""
        dbList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "dbFilename")
        if (dbList == []):
            dbFileName = ""
        else:
            dbFileName = dbList[0].strip()
        if (dbFileName == ""):
            dbFileName = GetDefaultDBFile()
        NumModelPixelXXList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "NumModelPixelXX")
        if (NumModelPixelXXList == []):
            NumModelPixelXX = None
        else:
            NumModelPixelXX = NumModelPixelXXList[0]
        NumModelPixelYYList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "NumModelPixelYY")
        if (NumModelPixelYYList == []):
            NumModelPixelYY = None
        else:
            NumModelPixelYY = NumModelPixelYYList[0]
        LocalOverlapFlagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "LocalOverlap_Flag")
        if (LocalOverlapFlagList == []):
            LocalOverlapFlag = None
        else:
            LocalOverlapFlag = LocalOverlapFlagList[0]
        NoSubBeamFlagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "NoSubBeam_Flag")
        if (NoSubBeamFlagList == []):
            NoSubBeamFlag = None
        else:
            NoSubBeamFlag = NoSubBeamFlagList[0]
        EmAbsPATHList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "EmAbsPATH")
        if (EmAbsPATHList == []):
            EmAbsPATH = None
        else:
            EmAbsPATH = EmAbsPATHList[0]

        # Debug:
        # print ("ExpFileList = ", ExpFileList)
        # print ("NumberExpRangesList = ", NumberExpRangesList)
        # print ("FreqMinList = ", FreqMinList)
        # print ("FreqMaxList = ", FreqMaxList)
        # print ("FreqStepList = ", FreqStepList)
        # print ("t_back_flagList = ", t_back_flagList)
        # print ("tBackList = ", tBackList)
        # print ("tSlopeList = ", tSlopeList)
        # print ("N_HList = ", N_HList)
        # print ("beta_dustList = ", beta_dustList)
        # print ("kappa_1300List = ", kappa_1300List)
        # print ("DustFileNameList = ", DustFileNameList)
        # print ("BackgroundFileNameList = ", BackgroundFileNameList)
        # print ("ContPhenFuncIDList = ", ContPhenFuncIDList)
        # print ("ContPhenFuncParam1List = ", ContPhenFuncParam1List)
        # print ("ContPhenFuncParam2List = ", ContPhenFuncParam2List)
        # print ("ContPhenFuncParam3List = ", ContPhenFuncParam3List)
        # print ("ContPhenFuncParam4List = ", ContPhenFuncParam4List)
        # print ("ContPhenFuncParam5List = ", ContPhenFuncParam5List)
        # print ("NoiseList = ", NoiseList)
        # print ("SmoothList = ", SmoothList)
        # print ("GlobalvLSRList = ", GlobalvLSRList)
        # print ("TelescopeSizeList = ", TelescopeSizeList)
        # print ("BMINList = ", BMINList)
        # print ("BMAJList = ", BMAJList)
        # print ("BPAList = ", BPAList)
        # print ("Inter_FlagList = ", Inter_FlagList)
        # print ("RedshiftList = ", RedshiftList)
        # print ("IsotopologuesList = ", IsotopologuesList)
        # print ("dbList = ", dbList)
        # print ("NumModelPixelXXList = ", NumModelPixelXXList)
        # print ("NumModelPixelYYList = ", NumModelPixelYYList)
        # print ("LocalOverlapFlagList = ", LocalOverlapFlagList)
        # print ("NoSubBeamFlagList = ", NoSubBeamFlagList)
        # print ("EmAbsPATH = ", EmAbsPATH)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## create list describing xml parameters for each range
        ObsXMLParameterList = []
        for ObsDataFileIndex in range(NumExpDataFiles):                                     ## loop over all obs. data files


            ## get parameters for current range
            RangeIndex = (-1)
            ObsXMLParameterDictFile = GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberRangeListIn = NumberExpRangesList, \
                                                             TelescopeSizeListIn = TelescopeSizeList, \
                                                             BMIN_ListIn = BMINList, \
                                                             BMAJ_ListIn = BMAJList, \
                                                             BPA_ListIn = BPAList, \
                                                             InterFlagListIn = Inter_FlagList, \
                                                             GlobalvLSRListIn = GlobalvLSRList, \
                                                             Redshift_ListIn = RedshiftList)
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']        ## get number of frequency ranges for current obs. data file
            TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
            BMIN = ObsXMLParameterDictFile['BMIN']
            BMAJ = ObsXMLParameterDictFile['BMAJ']
            BPA = ObsXMLParameterDictFile['BPA']
            InterFlag = ObsXMLParameterDictFile['InterFlag']
            GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
            if (GlobalvLSR is None):
                GlobalvLSR = 0.0
            Redshift = ObsXMLParameterDictFile['Redshift']
            if (Redshift is None):
                Redshift = 0.0


            ## get range parameters
            for RangeIndex in range(NumberFrequencyRanges):                                 ## loop over all range definitions in the whole xml file


                ## get parameters for current range
                ObsXMLParameterDictRange = GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = FreqMinList, \
                                                                  FreqMaxListIn = FreqMaxList, FreqStepListIn = FreqStepList, \
                                                                  tBackFlagListIn = t_back_flagList, tBackListIn = tBackList, \
                                                                  tSlopeListIn = tSlopeList, \
                                                                  N_HListIn = N_HList, \
                                                                  beta_dustListIn = beta_dustList, kappa_1300ListIn = kappa_1300List, \
                                                                  DustFileNameListIn = DustFileNameList, \
                                                                  BackgroundFileNameListIn = BackgroundFileNameList, \
                                                                  ContPhenFuncID_ListIn = ContPhenFuncIDList, \
                                                                  ContPhenFuncParam1_ListIn = ContPhenFuncParam1List, \
                                                                  ContPhenFuncParam2_ListIn = ContPhenFuncParam2List, \
                                                                  ContPhenFuncParam3_ListIn = ContPhenFuncParam3List, \
                                                                  ContPhenFuncParam4_ListIn = ContPhenFuncParam4List, \
                                                                  ContPhenFuncParam5_ListIn = ContPhenFuncParam5List)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None):
                    FreqMax = ObsXMLParameterDictRange['FreqMax']
                    FreqStep = ObsXMLParameterDictRange['FreqStep']
                    t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                    tBack = ObsXMLParameterDictRange['tBack']
                    tSlope = ObsXMLParameterDictRange['tSlope']
                    nH_flag = True
                    N_H = ObsXMLParameterDictRange['N_H']
                    beta_dust = ObsXMLParameterDictRange['beta_dust']
                    kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                    if (N_H is None or beta_dust is None or kappa_1300 is None):
                        nH_flag = False
                        N_H = 0.0
                        beta_dust = 0.0
                        kappa_1300 = 0.0
                    DustFileName = ObsXMLParameterDictRange['DustFileName']
                    BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                    LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                    LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                    LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                    LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                    LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                    LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']


                    ## store parameters to dictionary
                    ObsXMLParameterDict = {}
                    ObsXMLParameterDict['NumModelPixelXX'] = NumModelPixelXX
                    ObsXMLParameterDict['NumModelPixelYY'] = NumModelPixelYY
                    ObsXMLParameterDict['LocalOverlapFlag'] = LocalOverlapFlag
                    ObsXMLParameterDict['NoSubBeamFlag'] = NoSubBeamFlag
                    ObsXMLParameterDict['dbFileName'] = dbFileName
                    ObsXMLParameterDict['ObsDataFileIndex'] = ObsDataFileIndex
                    ObsXMLParameterDict['TelescopeSize'] = TelescopeSize
                    ObsXMLParameterDict['BMIN'] = BMIN
                    ObsXMLParameterDict['BMAJ'] = BMAJ
                    ObsXMLParameterDict['BPA'] = BPA
                    ObsXMLParameterDict['InterFlag'] = InterFlag
                    ObsXMLParameterDict['GlobalvLSR'] = GlobalvLSR
                    ObsXMLParameterDict['Redshift'] = Redshift
                    ObsXMLParameterDict['RangeIndex'] = RangeIndex
                    ObsXMLParameterDict['FreqMin'] = FreqMin
                    ObsXMLParameterDict['FreqMax'] = FreqMax
                    ObsXMLParameterDict['FreqStep'] = FreqStep
                    ObsXMLParameterDict['t_back_flag'] = t_back_flag
                    ObsXMLParameterDict['tBack'] = tBack
                    ObsXMLParameterDict['tSlope'] = tSlope
                    ObsXMLParameterDict['N_H'] = N_H
                    ObsXMLParameterDict['beta_dust'] = beta_dust
                    ObsXMLParameterDict['kappa_1300'] = kappa_1300
                    ObsXMLParameterDict['DustFileName'] = DustFileName
                    ObsXMLParameterDict['BackgroundFileName'] = BackgroundFileName
                    ObsXMLParameterDict['ContPhenFuncID'] = LocalContPhenFuncID
                    ObsXMLParameterDict['ContPhenFuncParam1'] = LocalContPhenFuncParam1
                    ObsXMLParameterDict['ContPhenFuncParam2'] = LocalContPhenFuncParam2
                    ObsXMLParameterDict['ContPhenFuncParam3'] = LocalContPhenFuncParam3
                    ObsXMLParameterDict['ContPhenFuncParam4'] = LocalContPhenFuncParam4
                    ObsXMLParameterDict['ContPhenFuncParam5'] = LocalContPhenFuncParam5
                    ObsXMLParameterDict['IsoFlag'] = IsoFlag
                    ObsXMLParameterDict['IsoTableFileName'] = IsoFileName
                    ObsXMLParameterDict['EmAbsPATH'] = EmAbsPATH
                    ObsXMLParameterList.append(ObsXMLParameterDict)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if no obs. xml file is given create list with dictionary as well
    else:
        ObsXMLParameterDict = {}
        ObsXMLParameterDict['NumModelPixelXX'] = NumModelPixelXX
        ObsXMLParameterDict['NumModelPixelYY'] = NumModelPixelYY
        ObsXMLParameterDict['LocalOverlapFlag'] = LocalOverlapFlag
        ObsXMLParameterDict['NoSubBeamFlag'] = NoSubBeamFlag
        ObsXMLParameterDict['dbFileName'] = dbFileName
        ObsXMLParameterDict['ObsDataFileIndex'] = 0
        ObsXMLParameterDict['TelescopeSize'] = TelescopeSize
        ObsXMLParameterDict['BMIN'] = BMIN
        ObsXMLParameterDict['BMAJ'] = BMAJ
        ObsXMLParameterDict['BPA'] = BPA
        ObsXMLParameterDict['InterFlag'] = Inter_Flag
        ObsXMLParameterDict['GlobalvLSR'] = vLSR
        ObsXMLParameterDict['Redshift'] = Redshift
        ObsXMLParameterDict['RangeIndex'] = 0
        ObsXMLParameterDict['FreqMin'] = FreqMin
        ObsXMLParameterDict['FreqMax'] = FreqMax
        ObsXMLParameterDict['FreqStep'] = FreqStep
        ObsXMLParameterDict['t_back_flag'] = t_back_flag
        ObsXMLParameterDict['tBack'] = tBack
        ObsXMLParameterDict['tSlope'] = tSlope
        ObsXMLParameterDict['N_H'] = N_H
        ObsXMLParameterDict['beta_dust'] = beta_dust
        ObsXMLParameterDict['kappa_1300'] = kappa_1300
        ObsXMLParameterDict['DustFileName'] = DustFileName
        ObsXMLParameterDict['BackgroundFileName'] = BackgroundFileName
        ObsXMLParameterDict['ContPhenFuncID'] = ContPhenFuncID
        ObsXMLParameterDict['ContPhenFuncParam1'] = ContPhenFuncParam1
        ObsXMLParameterDict['ContPhenFuncParam2'] = ContPhenFuncParam2
        ObsXMLParameterDict['ContPhenFuncParam3'] = ContPhenFuncParam3
        ObsXMLParameterDict['ContPhenFuncParam4'] = ContPhenFuncParam4
        ObsXMLParameterDict['ContPhenFuncParam5'] = ContPhenFuncParam5
        ObsXMLParameterDict['IsoFlag'] = task_myXCLASSMapFit.CheckBool(iso_flag)
        ObsXMLParameterDict['IsoTableFileName'] = IsoTableFileName
        ObsXMLParameterList = [ObsXMLParameterDict]
        ObsXMLParameterDict['EmAbsPATH'] = EmAbsPATH

    # Debug:
    # print ("ObsXMLParameterList = ", ObsXMLParameterList)


    ##====================================================================================================================================================
    ## start loop over all ranges
    for ObsXMLParameterDictID, ObsXMLParameterDict in enumerate(ObsXMLParameterList):       ## loop over all freq. ranges and corresponding dictionaries


        ## get parameter for current frequency range
        NumModelPixelXX = ObsXMLParameterDict['NumModelPixelXX']
        NumModelPixelYY = ObsXMLParameterDict['NumModelPixelYY']
        LocalOverlapFlag = ObsXMLParameterDict['LocalOverlapFlag']
        NoSubBeamFlag = ObsXMLParameterDict['NoSubBeamFlag']
        dbFileName = ObsXMLParameterDict['dbFileName']
        ObsDataFileIndex = ObsXMLParameterDict['ObsDataFileIndex']
        TelescopeSize = ObsXMLParameterDict['TelescopeSize']
        BMIN = ObsXMLParameterDict['BMIN']
        BMAJ = ObsXMLParameterDict['BMAJ']
        BPA = ObsXMLParameterDict['BPA']
        Inter_Flag = ObsXMLParameterDict['InterFlag']
        vLSR = ObsXMLParameterDict['GlobalvLSR']
        Redshift = ObsXMLParameterDict['Redshift']
        RangeIndex = ObsXMLParameterDict['RangeIndex']
        FreqMin = ObsXMLParameterDict['FreqMin']
        FreqMax = ObsXMLParameterDict['FreqMax']
        FreqStep = ObsXMLParameterDict['FreqStep']
        t_back_flag = ObsXMLParameterDict['t_back_flag']
        tBack = ObsXMLParameterDict['tBack']
        tSlope = ObsXMLParameterDict['tSlope']
        N_H = ObsXMLParameterDict['N_H']
        beta_dust = ObsXMLParameterDict['beta_dust']
        kappa_1300 = ObsXMLParameterDict['kappa_1300']
        DustFileName = ObsXMLParameterDict['DustFileName']
        BackgroundFileName = ObsXMLParameterDict['BackgroundFileName']
        ContPhenFuncID = ObsXMLParameterDict['ContPhenFuncID']
        ContPhenFuncParam1 = ObsXMLParameterDict['ContPhenFuncParam1']
        ContPhenFuncParam2 = ObsXMLParameterDict['ContPhenFuncParam2']
        ContPhenFuncParam3 = ObsXMLParameterDict['ContPhenFuncParam3']
        ContPhenFuncParam4 = ObsXMLParameterDict['ContPhenFuncParam4']
        ContPhenFuncParam5 = ObsXMLParameterDict['ContPhenFuncParam5']
        EmAbsPATH = ObsXMLParameterDict['EmAbsPATH']
        iso_flag = ObsXMLParameterDict['IsoFlag']
        IsoTableFileName = ObsXMLParameterDict['IsoTableFileName']
        if (IsoTableFileName == ""):
            iso_flag = False


        ## define some internal parameters
        FreqMinList = [[0, 0, FreqMin]]
        FreqMaxList = [[0, 0, FreqMax]]
        GlobalvLSRList = [[0, vLSR]]
        RedshiftList = [[0, Redshift]]

        # Debug:
        # print ("\n\n\nObsXMLParameterDictID = ", ObsXMLParameterDictID)
        # print ("FreqMin = ", FreqMin)
        # print ("FreqMax = ", FreqMax)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, ASCII file describing dust
        if (DustFileName is not None):
            DustFileName = DustFileName.strip()
            if (DustFileName != ""):


                ## check existence of dust file and copy dust file to current working directory
                LocalDustFileName = [[0, 0, DustFileName]]
                ok, DustFileName = task_myXCLASSFit.CheckDustFile(LocalDustFileName, NameOfFunction, CurrentDir, myXCLASSJobDir, FreqMinList, FreqMaxList)
                if (ok == 1):
                    return (myXCLASSspectrumList, myXCLASSlogList, myXCLASSTransList, myXCLASSIntOpticalList, myXCLASSJobDir)
                else:
                    DustFileName = DustFileName[0][2]

                # Debug:
                # print ("DustFileName = ", DustFileName)
        else:
            DustFileName = ""


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, ASCII file describing background
        if (BackgroundFileName is not None):
            BackgroundFileName = BackgroundFileName.strip()
            if (BackgroundFileName != ""):


                ## check existence of background file and copy background file to current working directory
                LocalBackgroundFileName = [[0, 0, BackgroundFileName]]
                ok, BackgroundFileName = task_myXCLASSFit.CheckBackgroundFile(LocalBackgroundFileName, NameOfFunction, CurrentDir, myXCLASSJobDir, \
                                                                              FreqMinList, FreqMaxList)
                if (ok == 1):
                    return (myXCLASSspectrumList, myXCLASSlogList, myXCLASSTransList, myXCLASSIntOpticalList, myXCLASSJobDir)
                else:
                    BackgroundFileName = BackgroundFileName[0][2]
        else:
            BackgroundFileName = ""

        # Debug:
        # print ("BackgroundFileName = ", BackgroundFileName)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, ASCII file describing iso ratios
        if (iso_flag):


            ## analyze iso ratio file parameter: check path and name of the iso ratio file and copy new formatted iso ratio file to job directory
            ok, NewIsoRatioFileName = task_myXCLASSFit.CheckIsoRatioFile(IsoTableFileName, NameOfFunction, CurrentDir, myXCLASSJobDir, FreqMinList, \
                                                                         FreqMaxList, GlobalvLSRList, RedshiftList, [dbFileName], LocalMolfitsFileName)
            if (ok == 1):                                                                   ## an error occurred
                return (myXCLASSspectrumList, myXCLASSlogList, myXCLASSTransList, myXCLASSIntOpticalList, myXCLASSJobDir)

            # Debug:
            # print ("NewIsoRatioFileName = ", NewIsoRatioFileName)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, given path of files describing emission and absorption functions
        if (EmAbsPATH is not None):
            EmAbsPATH = EmAbsPATH.strip()
            if (not EmAbsPATH.startswith("/")):                                             ## make path absolute if given path is relative
                EmAbsPATH = CurrentDir + EmAbsPATH

            # Debug:
            # print ("EmAbsPATH = ", EmAbsPATH)


        ##================================================================================================================================================
        ## convert molfit and iso file to MAGIX xml instance file


        ## convert molfit to xml file
        LocalPrintflag = False
        InstanceXMLFileName = myXCLASSJobDir + "parameter.xml"
        ok = ConversionMolfit2XML(LocalPrintflag, LocalMolfitsFileName, InstanceXMLFileName)
        if (str(iso_flag).lower() == "true"):
            ok = GetIsoTableParameters(LocalPrintflag, NewIsoRatioFileName, InstanceXMLFileName)


        ## convert column and hydrogen column density (if given for each component) to log scale.
        #    LogLinearFlag = "log"
        #    ConvertNtotToLogScale(LocalMolfitsFileName, LogLinearFlag)


        ##================================================================================================================================================
        ## call myXCLASS program
        CallmyXCLASS(myXCLASSrootDir, myXCLASSJobDir, FreqMin, FreqMax, FreqStep, TelescopeSize, Inter_Flag, t_back_flag, tBack, tSlope, nH_flag, N_H, \
                     beta_dust, kappa_1300, DustFileName, vLSR, iso_flag, InstanceXMLFileName, dbFileName, printFlag, \
                     SpectrumOnlyFlag, NumModelPixelXX, NumModelPixelYY, LocalOverlapFlag, BMIN, BMAJ, BPA, Redshift, \
                     ContPhenFuncID, ContPhenFuncParam1, ContPhenFuncParam2, ContPhenFuncParam3, ContPhenFuncParam4, ContPhenFuncParam5,
                     NoSubBeamFlag = NoSubBeamFlag, EmAbsFlag = EmAbsFlag, EmAbsPATH = EmAbsPATH, BackgroundFileName = BackgroundFileName)


        ##================================================================================================================================================
        ## read in calculated spectrum
        myXCLASSoutputFilename = myXCLASSJobDir + "xclass_spectrum_output.dat"

        # Debug:
        # print ("myXCLASSoutputFilename = ", myXCLASSoutputFilename)

        if (printFlag):
            print ("\nImport calculated myXCLASS spectrum .. ", end = "")
        try:
            myXCLASSspectrum = numpy.loadtxt(myXCLASSoutputFilename, skiprows = 0)
            if (printFlag):
                print ("done!")
        except IOError as err:
            myXCLASSspectrumList = []
            print ("\n\n\t Error in XCLASS package, function task_myXCLASS.myXCLASS:")
            print ("\t\t An error occurred while importing the modeled spectrum!\n\n\n\n")
            return (myXCLASSspectrumList, myXCLASSlogList, myXCLASSTransList, myXCLASSIntOpticalList, myXCLASSJobDir)


        ## add velocity axis if RestFreq != 0.0
        if (RestFreq != 0.0):
            if (printFlag):
                print ("Add velocity axis ..", end=' ')
            FreqData = myXCLASSspectrum[:, 0]
            VelocityData = ChangeToVelocitiy(FreqData, RestFreq, vLSR)

            # Debug:
            # print ("FreqData = ", FreqData)
            # print ("VelocityData = ", VelocityData)


            ## add velocity axis to ASCIIdata array
            NewmyXCLASSspectrum = numpy.zeros((len(myXCLASSspectrum), 3), dtype=numpy.float32)
            NewmyXCLASSspectrum[:, 0] = myXCLASSspectrum[:, 0]
            NewmyXCLASSspectrum[:, 1] = VelocityData[:]
            NewmyXCLASSspectrum[:, 2] = myXCLASSspectrum[:, 1]
            myXCLASSspectrum = NewmyXCLASSspectrum


            ## print some information to screen
            if (printFlag):
                print ("done!")


        ## append current spectrum to output list
        myXCLASSspectrumList.append(myXCLASSspectrum)


        ##================================================================================================================================================
        ## read in log file
        myXCLASSlogFilename = myXCLASSJobDir + "xclass_spectrum.log"
        try:
            if (myXCLASSlogFilename.endswith(".gz")):
                f = gzip.open(myXCLASSlogFilename, 'rt')
            else:
                try:
                    f = open(myXCLASSlogFilename)
                except:
                    f = gzip.open(myXCLASSlogFilename + ".gz", 'rt')
            myXCLASSlog = f.readlines()                                                     ## read contents
            f.close()                                                                       ## close file
        except IOError as err:
            myXCLASSlog = []
        myXCLASSlogList.append(myXCLASSlog)

        # Debug:
        # print ("myXCLASSlogFilename = ", myXCLASSlogFilename)
        # print ("myXCLASSlog = ", myXCLASSlog)


        ##================================================================================================================================================
        ## read in transition energies from file
        myXCLASSTransFilename = myXCLASSJobDir + "transition_energies.dat"
        try:
            if (myXCLASSTransFilename.endswith(".gz")):
                f = gzip.open(myXCLASSTransFilename, 'rt')
            else:
                try:
                    f = open(myXCLASSTransFilename)
                except:
                    f = gzip.open(myXCLASSTransFilename + ".gz", 'rt')
            contents = f.readlines()
            f.close()                                                                       ## close file
            myXCLASSTrans = []
            for line in contents:
                StrippedLine = line.strip()
                SplittedLine = StrippedLine.split()
                myXCLASSTrans.append(SplittedLine)
        except IOError as err:
            myXCLASSTrans = []
        myXCLASSTransList.append(myXCLASSTrans)

        # Debug:
        # print ("myXCLASSTransFilename = ", myXCLASSTransFilename)
        # print ("myXCLASSTrans = ", myXCLASSTrans)


        ##================================================================================================================================================
        ## read in intensity from output files
        myXCLASSIntOptical = []
        if (printFlag):
            print ("\nRead in intensities for each molecule and for each component .. ", end = "", flush = True)
        listing = os.listdir(myXCLASSJobDir)


        ## get all optical depths files
        IntensityFiles = []
        for files in listing:
            if (files.startswith("intensity__")):
                IntensityFiles.append(files)
        IntensityFiles = sorted(IntensityFiles)


        ## read in model function values
        IntensitiesPerComponent = []
        for files in IntensityFiles:


            ## read header information
            LocalFileName = myXCLASSJobDir + files
            if (LocalFileName.endswith(".gz")):
                f = gzip.open(LocalFileName, 'rt')
            else:
                try:
                    f = open(LocalFileName)
                except:
                    f = gzip.open(LocalFileName + ".gz", 'rt')
            line = f.readline()
            i = line.find("=")
            molname = line[i + 1:].strip()
            line = f.readline()
            i = line.find("=")
            ncomp = int(line[i + 1:].strip())
            f.close()

            # Debug:
            # print ("\nmolname = ", molname)
            # print ("ncomp = ", ncomp)


            ## read in intensity from file
            CurrentIntensity = numpy.loadtxt(LocalFileName, skiprows = 4)
            if (RestFreq != 0.0):                                                           ## add velocity axis if RestFreq != 0.0
                FreqData = CurrentIntensity[:, 0]
                VelocityData = ChangeToVelocitiy(FreqData, RestFreq, vLSR)


                ## add velocity axis to ASCIIdata array
                NewmCurrentIntensity = numpy.zeros((len(CurrentIntensity), 3), dtype=numpy.float32)
                NewmCurrentIntensity[:, 0] = CurrentIntensity[:, 0]
                NewmCurrentIntensity[:, 1] = VelocityData[:]
                NewmCurrentIntensity[:, 2] = CurrentIntensity[:, 1]
                CurrentIntensity = NewmCurrentIntensity

            # Debug:
            # print ("\n\nCurrentIntensity[:, 0] = ", CurrentIntensity[:, 0])
            # print ("CurrentIntensity[:, -1] = ", CurrentIntensity[:, -1])


            ## integrate intensity
            y = CurrentIntensity[:, -1]
            IntSpectra = scipy.integrate.simps(y, x = CurrentIntensity[:, 0])


            ## add to output array
            IntensitiesPerComponent.append([molname, ncomp, CurrentIntensity, IntSpectra])


        ## sort output array
        # IntensitiesPerComponent.sort()
        myXCLASSIntOptical.append(IntensitiesPerComponent)


        ## we're done
        if (printFlag):
            print ("done!")


        ##================================================================================================================================================
        ## read in optical depth from output files
        if (not SpectrumOnlyFlag):
            if (printFlag):
                print ("\nRead in optical depths for each molecule and for each component .. ", end = "", flush = True)


            ## get all optical depths files
            OpticalDepthsFiles = []
            for files in listing:
                if (files.startswith("optical_depth__")):
                    OpticalDepthsFiles.append(files)
            OpticalDepthsFiles = sorted(OpticalDepthsFiles)


            ## read in optical depth from output files
            OpticalDepthsPerComponent = []
            for files in OpticalDepthsFiles:


                ## read header information
                LocalFileName = myXCLASSJobDir + files
                if (LocalFileName.endswith(".gz")):
                    f = gzip.open(LocalFileName, 'rt')
                else:
                    try:
                        f = open(LocalFileName)
                    except:
                        f = gzip.open(LocalFileName + ".gz", 'rt')
                line = f.readline()
                i = line.find("=")
                molname = line[i + 1:].strip()
                line = f.readline()
                i = line.find("=")
                ncomp = int(line[i + 1:].strip())
                f.close()

                # Debug:
                # print ("\nmolname = ", molname)
                # print ("ncomp = ", ncomp)


                ## read in intensity
                CurrentOpticalDepth = numpy.loadtxt(myXCLASSJobDir + files, skiprows = 4)
                if (RestFreq != 0.0):                                                           ## add velocity axis if RestFreq != 0.0
                    FreqData = CurrentOpticalDepth[:, 0]
                    VelocityData = ChangeToVelocitiy(FreqData, RestFreq, vLSR)


                    ## add velocity axis to ASCIIdata array
                    NewCurrentOpticalDepth = numpy.zeros((len(CurrentOpticalDepth), 3), dtype=numpy.float32)
                    NewCurrentOpticalDepth[:, 0] = CurrentOpticalDepth[:, 0]
                    NewCurrentOpticalDepth[:, 1] = VelocityData[:]
                    NewCurrentOpticalDepth[:, 2] = CurrentOpticalDepth[:, 1]
                    CurrentOpticalDepth = NewCurrentOpticalDepth


                ## add to output array
                OpticalDepthsPerComponent.append([molname, ncomp, CurrentOpticalDepth])


            ## sort output array
            # OpticalDepthsPerComponent.sort()
            myXCLASSIntOptical.append(OpticalDepthsPerComponent)
            myXCLASSIntOpticalList.append(myXCLASSIntOptical)


            ## we're done
            if (printFlag):
                print ("done!")


        ##================================================================================================================================================
        ## if spectrum is calculated for more than one frequency range, move output files from current run to newly create subdirectory within job dir.
        if (len(ObsXMLParameterList) > 1):
            LocalRangeDir = myXCLASSJobDir + "Range__" + str(ObsDataFileIndex + 1) + "__" + str(RangeIndex + 1) + "/"
            cmdString = "mkdir -p " + LocalRangeDir + "; "
            cmdString += "mv " + myXCLASSJobDir + "xclass_spectrum.log " + LocalRangeDir + "; "
            cmdString += "mv " + myXCLASSJobDir + "xclass_spectrum_output.dat " + LocalRangeDir + "; "
            if (not SpectrumOnlyFlag):
                cmdString += "mv " + myXCLASSJobDir + "transition_energies.dat " + LocalRangeDir + "; "
                cmdString += "mv " + myXCLASSJobDir + "optical_depth_*.dat " + LocalRangeDir + "; "
                cmdString += "mv " + myXCLASSJobDir + "intensity_*.dat " + LocalRangeDir + "; "
            if (EmAbsFlag):
                cmdString += "mv " + myXCLASSJobDir + "emission_absorption____distance__*.dat " + LocalRangeDir + "; "
            os.system(cmdString)

            # Debug:
            # print ("cmdString = ", cmdString)


    ##====================================================================================================================================================
    ## print a message to screen
    if (printFlag):
        print ("\nAll files of the current myXCLASS run are stored in the job directory " + myXCLASSJobDir + "!\n\n")

    # Debug:
    # print ("myXCLASSspectrumList = ", myXCLASSspectrumList)
    # print ("myXCLASSlogList = ", myXCLASSlogList)
    # print ("myXCLASSTransList = ", myXCLASSTransList)
    # print ("myXCLASSIntOpticalList = ", myXCLASSIntOpticalList)
    # print ("myXCLASSJobDir = ", myXCLASSJobDir)


    ##====================================================================================================================================================
    ## the following lines are included for backward compatibility
    if (len(myXCLASSspectrumList) == 1):
        myXCLASSspectrumList = myXCLASSspectrumList[0]
        myXCLASSlogList = myXCLASSlogList[0]
        myXCLASSTransList = myXCLASSTransList[0]
        if (not SpectrumOnlyFlag):
            myXCLASSIntOpticalList = myXCLASSIntOpticalList[0]
        else:
            myXCLASSIntOpticalList = []


    ##====================================================================================================================================================
    ## define return variables
    return (myXCLASSspectrumList, myXCLASSlogList, myXCLASSTransList, myXCLASSIntOpticalList, myXCLASSJobDir)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
##
## The function defines the interface for the myXCLASS program
##
##
def myXCLASS(FreqMin, FreqMax, FreqStep, TelescopeSize, BMIN, BMAJ, BPA, Inter_Flag, Redshift, t_back_flag, tBack, tslope, BackgroundFileName, \
             N_H, beta_dust, kappa_1300, DustFileName, ContPhenFuncID, ContPhenFuncParam1, \
             ContPhenFuncParam2, ContPhenFuncParam3, ContPhenFuncParam4, ContPhenFuncParam5, MolfitsFileName, iso_flag, IsoTableFileName, \
             NumModelPixelXX, NumModelPixelYY, LocalOverlapFlag, NoSubBeamFlag, dbFilename, RestFreq, vLSR):
    """

The files produced by a myXCLASS function call are copied to a so-called "job-directory" located in the myXCLASS working directory
"path-of-myXCLASS-CASA-Interface/run/myXCLASS/"! The name of a job directory for a myXCLASS run is made up of four components: The first part
consists of the phrase "job_" whereas the second and third part describes the date (day, month, year) and the time stamp (hours, minutes,
seconds) of the function execution, respectively. The last part indicates a so-called job ID which is composed of the so-called PID followed
by a four digit random integer number to create a really unambiguous job number, e.g.
"path-of-myXCLASS-CASA-Interface/run/myXCLASS/job__25-07-2013__12-02-03__189644932/"


input parameters:
-----------------

    - FreqMin:                  start frequency of simulated spectrum (in K), (default: 1).

    - FreqMax:                  end frequency of simulated spectrum (in K), (default: 1.e8).

    - FreqStep:                 step frequency of simulated spectrum (in K), (default: 1).

    - TelescopeSize:            for single dish observations (Inter_Flag = False): TelescopeSize
                                describes the size of telescope (in m), (default: 1); for
                                interferometric observations (Inter_Flag = True): TelescopeSize
                                describes the interferometric beam FWHM size (in arcsec),
                                (default: 1).

    - BMIN:                     Beam minor axis length (default: None)

    - BMAJ:                     Beam major axis length (default: None)

    - BPA:                      Beam position angle (default: None)

    - Inter_Flag:               defines, if single dish ("False") or interferometric
                                observations ("True") are described, (default: "False").

    - Redshift:                 red shift (dimensionless), (default: None)

    - t_back_flag:              ("True"/"False"), defines, if the user defined background
                                temperature Tbg and temperature slope TSlope given
                                by the input parameters tBack and tSlope describe the
                                continuum contribution completely (t_back_flag = "True")
                                or not (t_back_flag = "False") (default: "True").

    - tBack:                    background temperature (in K), (default: 0.0).

    - tSlope:                   temperature slope (dimensionless), (default: 0.0).

    - BackgroundFileName:       path and name of background file, (default "").

    - N_H:                      Hydrogen column density (in cm^{-2}), (default: None).

    - beta_dust:                beta for dust (dimensionless), (default: None).

    - kappa_1300:               kappa (cm^2 g^{-1}), (default: None).

    - DustFileName:             path and name of file describing the optical depth
                                of dust (default: None).

    - ContPhenFuncID:           describes which phenomenological function is used
                                to describe the continuum (default: None).

    - ContPhenFuncParam1:       first parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam2:       second parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam3:       third parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam4:       fourth parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam5:       fifth parameter for phenomenological function
                                describing the continuum (default: None).

    - MolfitsFileName:          ABSOLUTE path and file name of the molfits file,
                                (default: "").

    - iso_flag:                 use isotopologues ("True"/"False"). If iso_flag is set to "False"
                                the ratios are all set to 1 (default: "False").

    - IsoTableFileName:         (has to be given only if iso_flag is set to "True")
                                ABSOLUTE path and file name of an ASCII file including
                                the iso ratios between certain molecules. The so-called
                                "iso ratio file" defines the iso ratios between
                                molecules. The ASCII file consists of three columns,
                                where the first two columns indicates the molecules,
                                respectively. The third column defines the ratio for
                                both molecules. The columns are separated by blanks or
                                tabs. So, the names of the molecules must not contain
                                blanks. Comments have to be marked with the "%" sign,
                                i.e. all characters on the right side of a "%" sign
                                are ignored.

                                The myXCLASSFit function offers the possibility to
                                optimize the ratios between isotopologues as well. For
                                that purpose, the user has to add two additional columns
                                on the right indicating the lower and the upper limit
                                of a certain ratio, respectively.

    - NumModelPixelXX:          used for sub-beam modeling, describes the number
                                of pixels used in x-direction (default: 100).

    - NumModelPixelYY:          used for sub-beam modeling, describes the number
                                of pixels used in y-direction (default: 100).

    - LocalOverlapFlag:         take local overlap into account (default: False).

    - NoSubBeamFlag:            do not use sub-beam description (default: True).

    - dbFileName:               path and name of database file (default: None, i.e.
                                the default database file is used).

    - RestFreq:                 rest frequency in MHz (default: 0). (If this parameter
                                is set to zero, the intensity is plotted against
                                frequency (in MHz) otherwise against velocity (in km/s).

    - vLSR:                     velocity (local standard of rest) in km/s, only used,
                                if RestFreq /= 0. (velocity(Frequency = RestFreq) = vLSR)


output parameters:
------------------

    - myXCLASSspectrum:         contains the calculated myXCLASS spectrum

    - myXCLASSlog:              contains the corresponding log file

    - myXCLASSTrans:            python list containing the transition frequencies (in MHz) from the last myXCLASS run, the Doppler-shifted transition
                                frequencies (in MHz), the corresponding intensities (in K), the energy of the lower level (in K), the upper state
                                degeneracy, the Einstein A coefficient (in s^{-1}), and the molecule names within the defined range

    - myXCLASSIntOptical        contains intensities and optical depths for each molecule and component


                                The intensities and optical depths are stored as follows:


                                myXCLASSIntOptical[0], contains all information about the intensities

                                    myXCLASSIntOptical[0][i][0]    contains the name of the molecule

                                    myXCLASSIntOptical[0][i][1]    contains the index of the component / layer

                                    myXCLASSIntOptical[0][i][2]    contains the intensities of the ith layer or component as a 2D python list

                                    myXCLASSIntOptical[0][i][3]    contains the integrated intensities of the ith layer or component


                                myXCLASSIntOptical[1], contains all information about the optical depths.

                                    myXCLASSIntOptical[1][i][0]    contains the name of the molecule

                                    myXCLASSIntOptical[1][i][1]    contains the index of the component / layer

                                    myXCLASSIntOptical[1][i][2]    contains the optical depth of the ith layer or component as a 2D python list

    - myXCLASSJobDir:           absolute path of the myXCLASS job directory created for the current run

Note, the user is free to define a different name for the output parameter.



Example:
--------

FreqMin = 578000.5
FreqMax = 582000.5
FreqStep = 0.5
vLSR = 0.0
TelescopeSize = 3.5
Inter_Flag = False
t_back_flag = True
tBack = 1.1
tslope = 0.0
N_H = 3.0E+24
beta_dust = 2.0
kappa_1300 = 0.42
MolfitsFileName = "demo/myXCLASS/CH3OH__pure.molfit"
iso_flag = True
IsoTableFileName = "demo/myXCLASS/iso_names.txt"
RestFreq = 0.0
modeldata, log, TransEnergies, IntOptical, jobDir = myXCLASS()

FileName = "demo/myXCLASS/band1b.dat"
NumHeaderLines = 0
expdata = LoadASCIIFile()

MinIntensity = 0.0
xLowerLimit = 578000.5
xUpperLimit = 582000.5
yLowerLimit = 0.8
yUpperLimit = 2.5
PlotTitle = "Example for myXCLASSPlot function"
LegendFlag = True
SaveFigureFile = ""
myXCLASSPlot()

Please note, the bottom half in the example reads the experimental data from file and plots
the calculated spectrum using the myXCLASSPlot function. Note, the user is free to define
different names for the output parameters. In the example describe above, we use the
name "modeldata" for output parameter myXCLASSspectrum, "log" for output parameter myXCLASSlog,
"TransEnergies" for output parameter myXCLASSTrans, "IntOptical" for output parameter
myXCLASSIntOptical, and "jobDir" for output parameter myXCLASSJobDir.

    """

    # Debug:
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("FreqStep = ", FreqStep)
    # print ("TelescopeSize = ", TelescopeSize)
    # print ("BMIN = ", BMIN)
    # print ("BMAJ = ", BMAJ)
    # print ("BPA = ", BPA)
    # print ("Inter_Flag = ", Inter_Flag)
    # print ("Redshift = ", Redshift)
    # print ("t_back_flag = ", t_back_flag)
    # print ("tBack = ", tBack)
    # print ("tslope = ", tslope)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("iso_flag = ", iso_flag)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)
    # print ("dbFilename = ", dbFilename)
    # print ("RestFreq = ", RestFreq)
    # print ("vLSR = ", vLSR)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get default database file
    if (dbFilename is None):
        dbFilename = GetDefaultDBFile()
    elif (dbFilename.strip() == ""):
        dbFilename = GetDefaultDBFile()


    ## call myXCLASS core function
    printflag = True
    nH_flag = False
    EmAbsFlag = False
    ObsXMLFileName = None
    myXCLASSspectrum, myXCLASSlog, myXCLASSTrans, myXCLASSIntOptical, myXCLASSJobDir = myXCLASSCore(ObsXMLFileName = ObsXMLFileName, \
                                                                                                    FreqMin = FreqMin, FreqMax = FreqMax, \
                                                                                                    FreqStep = FreqStep, \
                                                                                                    MolfitsFileName = MolfitsFileName, \
                                                                                                    iso_flag = iso_flag, \
                                                                                                    IsoTableFileName = IsoTableFileName, \
                                                                                                    TelescopeSize = TelescopeSize, BMIN = BMIN, \
                                                                                                    BMAJ = BMAJ, BPA = BPA, Redshift = Redshift, \
                                                                                                    Inter_Flag = Inter_Flag, t_back_flag = t_back_flag, \
                                                                                                    tBack = tBack, tSlope = tslope, \
                                                                                                    BackgroundFileName = BackgroundFileName, \
                                                                                                    nH_flag = nH_flag, N_H = N_H, beta_dust = beta_dust, \
                                                                                                    kappa_1300 = kappa_1300, DustFileName = DustFileName,\
                                                                                                    ContPhenFuncID = ContPhenFuncID, \
                                                                                                    ContPhenFuncParam1 = ContPhenFuncParam1, \
                                                                                                    ContPhenFuncParam2 = ContPhenFuncParam2, \
                                                                                                    ContPhenFuncParam3 = ContPhenFuncParam3, \
                                                                                                    ContPhenFuncParam4 = ContPhenFuncParam4, \
                                                                                                    ContPhenFuncParam5 = ContPhenFuncParam5, \
                                                                                                    NumModelPixelXX = NumModelPixelXX, \
                                                                                                    NumModelPixelYY = NumModelPixelYY, \
                                                                                                    LocalOverlapFlag = LocalOverlapFlag, \
                                                                                                    NoSubBeamFlag = NoSubBeamFlag, \
                                                                                                    dbFileName = dbFilename, RestFreq = RestFreq, \
                                                                                                    vLSR = vLSR, EmAbsFlag = EmAbsFlag, \
                                                                                                    printFlag = printflag)
    # Debug:
    # print ("myXCLASSspectrum = ", myXCLASSspectrum)
    # print ("myXCLASSlog = ", myXCLASSlog)
    # print ("myXCLASSTrans = ", myXCLASSTrans)
    # print ("myXCLASSIntOptical = ", myXCLASSIntOptical)
    # print ("myXCLASSJobDir = ", myXCLASSJobDir)


    ##====================================================================================================================================================
    ## define return variables
    return (myXCLASSspectrum, myXCLASSlog, myXCLASSTrans, myXCLASSIntOptical, myXCLASSJobDir)
##--------------------------------------------------------------------------------------------------------------------------------------------------------

