#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module contains the interface to MAGIX
##  Copyright (C) 2012 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function GetXMLtag:                           read xml tags with name tagName in xml file GetXMLtag and return contents in ContentsTag
##      - function GetXMLtagNEW:                        read xml tags with name tagName in xml file GetXMLtag and return contents in ContentsTag
##      - function WriteXMLtag:                         write contents in ContentsTag to xml file GetXMLtag xml at tags with name tagName
##      - function WriteXMLtagNEW:                      write contents in ContentsTag to xml file GetXMLtag xml at tags with name tagName
##      - function RemoveXMLtag:                        remove tag definition from xml file
##      - function CreateRunDirectory:                  create a run directory for a function of the XCLASS interface
##      - function SetMAGIXEnvironment:                 set up environment variables for MAGIX call
##      - function StartMAGIX:                          start MAGIX without command line call
##      - function CreateIOXMLFile:                     create i/o XML file
##      - function MAGIX:                               function defines the interface to MAGIX
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2013-07-25      initial version
##  T. Moeller      2020-01-03      porting to python 3.x
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import copy                                                                                 ## import copy package
import random                                                                               ## import random package
import time                                                                                 ## import package date and time
if (not 'matplotlib' in sys.modules):
    import matplotlib                                                                       ## import matplotlib package
    matplotlib.use("Agg")                                                                   ## avoid display error
from . import task_myXCLASS                                                                 ## import package myXCLASS
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## read xml tags with name tagName in xml file GetXMLtag and return contents in ContentsTag
##
def GetXMLtag(xmlFileName, tagName):
    """

input parameters:
-----------------

    - xmlFileName:          path and name of the xml-file

    - tagName:              name of the tag, whose contents is read


output parameters:
------------------

    - ContentsTag:          the contents of the selected tag, always a list, including an entry for each occurance
    """

    # Debug:
    # print ("xmlFileName = ", xmlFileName)
    # print ("tagName = ", tagName)


    ## initialize return parameter
    ContentsTag = []


    ## read in whole xml-file
    xmlFile = open(xmlFileName)
    ContentsXMLFile = xmlFile.readlines()
    xmlFile.close()


    ## analyze contents of xml file
    tagName = tagName.strip()
    CurrentDataFileID = 0
    CurrentRangeID = 0
    for line in ContentsXMLFile:                                                            ## loop over all lines in the xml file
        striplines = line.strip()
        if (not striplines.startswith("<!--")):                                             ## check if current line is not a comment


            ## special handling for some tags
            if (striplines.find("<FileNamesExpFiles>") > (-1)):
                CurrentDataFileID += 1
                CurrentRangeID = 0


            ## special handling for some tags
            elif (striplines.find("<FrequencyRange>") > (-1)):
                CurrentRangeID += 1


            ## get contents of defined tag
            if (striplines.find("<" + tagName + ">") > (-1)):                               ## does the current line described selected tag?
                i = striplines.find(">")
                j = striplines.rfind("<")
                if (i < j):


                    ## special handling for some tags
                    if (tagName in ["MinExpRange", "MaxExpRange", "StepFrequency", "BackgroundTemperature", \
                                    "TemperatureSlope", "HydrogenColumnDensity", "DustBeta", "Kappa", \
                                    "Threshold", "ErrorY", "NumberHeaderLines", "SeparatorColumns", \
                                    "NoiseLevel", "SmoothValue", "DustFileName", "BackgroundFileName"]):
                        ContentsTag.append([CurrentDataFileID, striplines[i + 1:j]])
                    else:
                        ContentsTag.append(striplines[i + 1:j])                             ## save name(s) in list


    ## define return value
    return ContentsTag
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## read xml tags with name tagName in xml file GetXMLtag and return contents in ContentsTag
##
def GetXMLtagNEW(xmlFileName, tagName):
    """

input parameters:
-----------------

    - xmlFileName:          path and name of the xml-file

    - tagName:              name of the tag, whose contents is read


output parameters:
------------------

    - ContentsTag:          the contents of the selected tag, always a list, including an entry for each occurrence
    """

    # Debug:
    # print ("xmlFileName = ", xmlFileName)
    # print ("tagName = ", tagName)



    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    ContentsTag = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define parameter depending on the number of occurrences in the obs. xml file


    ## parameter in obs. xml file which occur only once
    ##
    ## to do: add "HydrogenColumnDensity"
    ##
    ObsXMLParameterListOnlyOnce = ["NumberExpFiles", "iso_flag", "Isotopologues", "IsoTableFileName", "dbFilename", \
                                   "NumModelPixelXX", "NumModelPixelYY", "LocalOverlap_Flag", "NoSubBeam_Flag", "EmAbsPATH"]


    ## parameter in obs. xml file which occur only per obs. data file
    ObsXMLParameterListDataFile = ["FileNamesExpFiles", "ImportFilter", "NumberExpRanges", "TelescopeSize", "TelescopeSizeBMIN", "TelescopeSizeBMAJ", \
                                   "TelescopeSizeBPA", "BMIN", "BMAJ", "BPA", "Inter_Flag", "GlobalvLSR", "ErrorY", "NumberHeaderLines", \
                                   "SeparatorColumns", "Redshift"]


    ## parameter in obs. xml file which (might) occur in every frequency range
    ##
    ## to do: remove "HydrogenColumnDensity"
    ##
    ObsXMLParameterListRange = ["MinExpRange", "MaxExpRange", "StepFrequency", "t_back_flag", "BackgroundTemperature", "TemperatureSlope", \
                                "nH_flag", "HydrogenColumnDensity", "DustBeta", "Kappa", "Threshold", "NoiseLevel", "SmoothValue", "DustFileName", \
                                "BackgroundFileName", "ContPhenFuncID", \
                                "ContPhenFuncParam1", "ContPhenFuncParam2", "ContPhenFuncParam3", "ContPhenFuncParam4", "ContPhenFuncParam5"]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in whole xml-file
    xmlFile = open(xmlFileName)
    ContentsXMLFile = xmlFile.readlines()
    xmlFile.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze contents of xml file
    tagName = tagName.strip()
    CurrentDataFileID = (-1)
    CurrentRangeID = (-1)
    for line in ContentsXMLFile:                                                            ## loop over all lines in the xml file
        striplines = line.strip()
        if (not striplines.startswith("<!--")):                                             ## check if current line is not a comment


            ## special handling for some tags
            if (striplines.find("<file>") > (-1)):
                CurrentDataFileID += 1
                CurrentRangeID = (-1)


            ## special handling for some tags
            elif (striplines.find("<FrequencyRange>") > (-1)):
                CurrentRangeID += 1


            ## get contents of defined tag
            if (striplines.find("<" + tagName + ">") > (-1)):                               ## does the current line described selected tag?
                i = striplines.find(">")
                j = striplines.rfind("<")
                if (i < j):


                    ## special handling for tags which occur per frequency range and obs. data file
                    if (tagName in ObsXMLParameterListRange):
                        ContentsTag.append([CurrentDataFileID, CurrentRangeID, striplines[i + 1:j]])


                    ## special handling for tags which occur only once per obs. data file
                    elif (tagName in ObsXMLParameterListDataFile):
                        ContentsTag.append([CurrentDataFileID, striplines[i + 1:j]])


                    ## special handling for tags which occur only once in the obs. xml file
                    elif (tagName in ObsXMLParameterListOnlyOnce):
                        ContentsTag.append(striplines[i + 1:j])


                    ## general case:
                    else:
                        ContentsTag.append(striplines[i + 1:j])

    # Debug:
    # print ("ContentsTag = ", ContentsTag)


    ## define return value
    return ContentsTag
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## write contents in ContentsTag to xml file
##
def WriteXMLtag(xmlFileName, tagName, ContentsTag):
    """

input parameters:
-----------------

    - xmlFileName:          path and name of the xml-file

    - tagName:              name of the tag, whose contents has to be modified

    - ContentsTag:          the new contents of the tag, which has to be always a list, i.e. a content for each occurrence of the tag



output parameters:
------------------

    - None
    """

    # Debug:
    # print ("xmlFileName = ", xmlFileName)
    # print ("tagName = ", tagName)
    # print ("ContentsTag = ", ContentsTag)


    ## read in whole xml-file
    xmlFile = open(xmlFileName)
    ContentsXMLFile = xmlFile.readlines()
    xmlFile.close()


    ## analyze contents of xml file
    NXF = open(xmlFileName, 'w')                                                            ## open xml file
    counter = (-1)                                                                          ## reset counter
    for line in ContentsXMLFile:                                                            ## loop over all lines in the xml file
        striplines = line.strip()                                                           ## remove leading and trailing blanks
        if (not striplines.startswith("<!--")):                                             ## ignore comments
            if (striplines.find("<" + tagName + ">") > (-1)):                               ## get name of experimental data file
                counter += 1
                i = line.find("<")
                if (i > (-1)):
                    space = line[:i]
                else:
                    space = ""
                if ((len(ContentsTag) - 1) >= counter):
                    NXF.write(space + "<" + tagName + ">" + ContentsTag[counter] + "</" + tagName + ">\n")

                # Debug:
                # print ("tagName = >>", tagName, '<<')
                # print ("counter = ", counter)
                # print ("(len(ContentsTag) - 1) = ", (len(ContentsTag) - 1))
                # print ("ContentsTag[counter] = ", ContentsTag[counter])
            else:
                NXF.write(line)
        else:
            NXF.write(line)
    NXF.close()


    ## special handling for experimental data files for myXCLASS
    if (counter == (-1) or len(ContentsTag) > counter):                                     ## the selected tag was not found or the given tag has more
                                                                                            ## entries than tags in xml file
        if (tagName in ["BackgroundTemperature", "TemperatureSlope", "HydrogenColumnDensity", "DustBeta", \
                        "Kappa", "Threshold", "dbFilename", "ErrorY", "DustFileName", \
                        "BackgroundFileName"]):
                                                                                            ## insert only these tags to xml file


            ## read in contents of xml file
            xmlFile = open(xmlFileName)
            ContentsXMLFile = xmlFile.readlines()
            xmlFile.close()


            ## insert new tag to xml file
            NXF = open(xmlFileName, 'w')                                                    ## open xml file
            for line in ContentsXMLFile:                                                    ## loop over all lines in the xml file
                striplines = line.strip()                                                   ## remove leading and trailing blanks
                if (not striplines.startswith("<!--")):                                     ## ignore comments


                    ## add tag at the end of FrequencyRange in xml file if tag is not found
                    if (striplines == "</FrequencyRange>" and counter == (-1) and tagName != "dbFilename" and tagName != "ErrorY"):
                        counter += 1
                        NXF.write("\n\n")
                        NXF.write("            <!-- " + tagName + " -->\n")
                        NXF.write("            <" + tagName + ">" + ContentsTag[counter] + "</" + tagName + ">\n")
                        NXF.write(line)


                    ## add tag at the end of ExpFiles in xml file if tag is not found
                    elif (striplines == "</ExpFiles>" and counter == (-1) and (tagName == "dbFilename" or tagName == "ErrorY")):
                        counter += 1
                        NXF.write("\n\n")
                        NXF.write("    <!-- " + tagName + " -->\n")
                        NXF.write("    <" + tagName + ">" + ContentsTag[counter] + "</" + tagName + ">\n")
                        NXF.write(line)
                    else:
                        NXF.write(line)
                else:
                    NXF.write(line)
            NXF.close()


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## write contents in ContentsTag to xml file
##
def WriteXMLtagNEW(xmlFileName, tagName, ContentsTag):
    """

input parameters:
-----------------

    - xmlFileName:          path and name of the xml-file

    - tagName:              name of the tag, whose contents has to be modified

    - ContentsTag:          the new contents of the tag, which has to be always a list, i.e. a content for each occurrence of the tag



output parameters:
------------------

    - None
    """

    # Debug:
    # print ("xmlFileName = ", xmlFileName)
    # print ("tagName = ", tagName)
    # print ("ContentsTag = ", ContentsTag)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import in whole xml-file
    xmlFile = open(xmlFileName)
    ContentsXMLFile = xmlFile.readlines()
    xmlFile.close()


    ## define problematic tags
    ## syntax:  ListProblematicTags =[.. ["search for tag", "check tag", "replace tag"] ..]
    FullListProblematicTags = [["Isotopologues", "iso_flag", "iso_flag"], ["iso_flag", "Isotopologues", "iso_flag"]]
    ListProblematicTags = []                                                                ## deactivate problematic tags handling


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze contents of xml file
    for NewTagID, NewTag in enumerate(ContentsTag):                                         ## loop over all new tag values

        # Debug:
        # print ("\ntagName = ", tagName)
        # print ("NewTag = ", NewTag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze new tag values and settings
        if (type(NewTag) in (tuple, list)):                                                 ## check, if current tag is a list
            if (len(NewTag) == 2):                                                          ## current tag contains obs. data file index as well
                SelectedObsDataID = NewTag[0]                                               ## get obs. data file index
                SelectedRangeID = (-1)                                                      ## no range parameter definition
                SelectedValue = str(NewTag[1])                                              ## get tag value
            else:
                SelectedObsDataID = NewTag[0]                                               ## get obs. data file index
                SelectedRangeID = NewTag[1]                                                 ## get range index
                SelectedValue = str(NewTag[2])                                              ## get tag value
        else:                                                                               ## general case, current tag is just one replacement
            SelectedObsDataID = (-1)                                                        ## no obs. data file parameter definition
            SelectedRangeID = (-1)                                                          ## no range parameter definition
            SelectedValue = str(NewTag)                                                     ## get tag value

        # Debug:
        # print ("\nSelectedObsDataID = ", SelectedObsDataID)
        # print ("SelectedRangeID = ", SelectedRangeID)
        # print ("SelectedValue = ", SelectedValue)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## special handling of problematic tags
        if (tagName in ListProblematicTags):                                                ## check, if current tag is problematic


            ## get search, check, and replacement tags
            IndexProblematicTag = ListProblematicTags.index(tagName)
            LocalSearchTag = FullListProblematicTags[IndexProblematicTag][0]                ## get name of problematic tag
            LocalCheckTag = FullListProblematicTags[IndexProblematicTag][1]                 ## get name of tag which has to be checked as well
            LocalReplacementTag = FullListProblematicTags[IndexProblematicTag][2]           ## get name of replacement tag


            ## determine number of occurrences
            ProblematicTagList = GetXMLtagNEW(xmlFileName, LocalSearchTag)                  ## search for problematic tag
            SearchTagList = GetXMLtagNEW(xmlFileName, LocalCheckTag)                        ## search for check tag


            ## define what to do if ..
            if (ProblematicTagList != [] and SearchTagList != []):                          ## check, if problematic tag is included in xml file
                ContentsXMLFile = RemoveXMLtag(ContentsXMLFile, LocalSearchTag)             ## remove tag from xml file
                ContentsXMLFile = RemoveXMLtag(ContentsXMLFile, LocalCheckTag)              ## remove tag from xml file
                tagName = LocalReplacementTag

            elif (ProblematicTagList != [] and SearchTagList == []):                        ## check, if problematic tag is included in xml file
                ContentsXMLFile = RemoveXMLtag(ContentsXMLFile, LocalSearchTag)             ## remove tag from xml file
                tagName = LocalReplacementTag

            elif (ProblematicTagList == [] and SearchTagList != []):                        ## check, if problematic tag is included in xml file
                ContentsXMLFile = RemoveXMLtag(ContentsXMLFile, LocalCheckTag)              ## remove tag from xml file
                tagName = LocalReplacementTag

            elif (ProblematicTagList == [] and SearchTagList == []):                        ## check, if problematic tag is included in xml file
                tagName = LocalReplacementTag


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze xml file content
        CurrentDataFileID = (-1)                                                            ## reset counter for obs. data file
        CurrentRangeID = (-1)                                                               ## reset counter for range
        NewContentsXMLFile = []
        FoundFlag = False
        TagCounter = (-1)
        for line in ContentsXMLFile:                                                        ## loop over all lines in the xml file
            NewLine = line
            StrippedLine = line.strip()                                                     ## remove leading and trailing blanks
            if ((not StrippedLine.startswith("<!--")) and StrippedLine != "" and (not FoundFlag)):      ## ignore comments and empty lines


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## determine obs. data file and range index


                ## determine current obs. data file index
                if (StrippedLine.find("<file>") > (-1)):
                    CurrentDataFileID += 1                                                  ## increase obs. data file index
                    CurrentRangeID = (-1)                                                   ## reset range index for new obs. data file


                ## determine current range index
                elif (StrippedLine.find("<FrequencyRange>") > (-1)):
                    CurrentRangeID += 1                                                     ## increase range index


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## we reached the end of a range definition: check, if new tag definition has to inserted
                elif (StrippedLine.find("</FrequencyRange>") > (-1) and SelectedRangeID > (-1)):
                    if (CurrentDataFileID == SelectedObsDataID and CurrentRangeID == SelectedRangeID):  ## check, if we found the correct range definition
                        FoundFlag = True


                        ## add comment which describes tag
                        NewLine = "\n\n            <!-- " + tagName + " -->\n"
                        NewContentsXMLFile.append(NewLine)


                        ## add new tag value which describes tag
                        NewLine = "            <" + tagName + ">" + SelectedValue + "</" + tagName + ">\n"
                        NewContentsXMLFile.append(NewLine)
                        NewLine = line


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## we reached the end of a obs. data file definition: check, if new tag definition has to inserted
                elif (StrippedLine.find("</file>") > (-1) and SelectedObsDataID > (-1)):
                    if (CurrentDataFileID == SelectedObsDataID):                            ## check, if we found the correct obs. data file definition
                        FoundFlag = True


                        ## add comment which describes tag
                        NewLine = "\n\n        <!-- " + tagName + " -->\n"
                        NewContentsXMLFile.append(NewLine)


                        ## add new tag value which describes tag
                        NewLine = "        <" + tagName + ">" + SelectedValue + "</" + tagName + ">\n"
                        NewContentsXMLFile.append(NewLine)
                        NewLine = line


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## we reached the end of a obs. data file definition: check, if new tag definition has to inserted
                elif (StrippedLine.find("</ExpFiles>") > (-1) and SelectedObsDataID == (-1)):
                    FoundFlag = True

                    # Debug:
                    # print ("\n\ntagName = ", tagName)
                    # print ("SelectedValue = ", SelectedValue)


                    ## add comment which describes tag
                    i = max(10, 2 * len(tagName) + len(SelectedValue) - 4)
                    starLine = i * "*"
                    NewLine = "\n\n    <!-- " + starLine + " -->\n"
                    NewContentsXMLFile.append(NewLine)
                    NewLine = "    <!-- " + tagName + " -->\n"
                    NewContentsXMLFile.append(NewLine)


                    ## add new tag value which describes tag
                    NewLine = "    <" + tagName + ">" + SelectedValue + "</" + tagName + ">\n"
                    NewContentsXMLFile.append(NewLine)
                    NewLine = line


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## do we find the correct tag definition?
                elif (StrippedLine.find("<" + tagName + ">") > (-1)):                       ## does the current line contain the correct tag
                    TagCounter += 1

                    # print ("\n\n\n")
                    # print ("StrippedLine = ", StrippedLine)
                    # print ("CurrentDataFileID = ", CurrentDataFileID)
                    # print ("\nSelectedObsDataID = ", SelectedObsDataID)
                    # print ("SelectedRangeID = ", SelectedRangeID)
                    # print ("TagCounter = ", TagCounter)
                    # print ("NewTagID = ", NewTagID)


                    ## check, if we found the correct obs. data file and range definition
                    DoReplacementFlag = False
                    if (CurrentDataFileID == SelectedObsDataID and CurrentRangeID == SelectedRangeID):
                        DoReplacementFlag = True
                    elif (SelectedObsDataID == (-1) and SelectedRangeID == (-1) and TagCounter == NewTagID):
                        DoReplacementFlag = True


                    elif (CurrentDataFileID == SelectedObsDataID and SelectedRangeID == (-1) and TagCounter == NewTagID):
                        DoReplacementFlag = True


                    if (DoReplacementFlag):
                        FoundFlag = True
                        i = line.find("<")
                        if (i > (-1)):
                            space = line[:i]
                        else:
                            space = ""
                        NewLine = space + "<" + tagName + ">" + SelectedValue + "</" + tagName + ">\n"


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## store new line
            NewContentsXMLFile.append(NewLine)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update content of xml file
        ContentsXMLFile = copy.deepcopy(NewContentsXMLFile)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write new content to xml file
    NXF = open(xmlFileName, 'w')
    for line in NewContentsXMLFile:
        NXF.write(line)
    NXF.close()


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## remove tag definition from xml file
##
def RemoveXMLtag(xmlFile, tagName):
    """

input parameters:
-----------------

    - xmlFile:              path and name of the xml-file

    - tagName:              name of the tag, whose contents has to be modified



output parameters:
------------------

    - ContentsXMLFile:      modified contents of xml file
    """

    # Debug:
    # print ("xmlFile = ", xmlFile)
    # print ("tagName = ", tagName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import in whole xml-file
    if (type(xmlFile).__name__ == 'str'):
        xmlFile = open(xmlFile)
        ContentsXMLFile = xmlFile.readlines()
        xmlFile.close()
    else:
        ContentsXMLFile = copy.deepcopy(xmlFile)


    ## remove tag definition from xml file
    CurrentDataFileID = (-1)                                                                ## reset counter for obs. data file
    CurrentRangeID = (-1)                                                                   ## reset counter for range
    NewContentsXMLFile = []
    for line in ContentsXMLFile:                                                            ## loop over all lines in the xml file
        StrippedLine = line.strip()                                                         ## remove leading and trailing blanks
        AddFlag = True
        if (StrippedLine != ""):                                                            ## ignore empty lines


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## determine obs. data file and range index
            if ((not StrippedLine.startswith("<!--"))):                                     ## ignore comments


                ## determine current obs. data file index
                if (StrippedLine.find("<file>") > (-1)):
                    CurrentDataFileID += 1                                                  ## increase obs. data file index
                    CurrentRangeID = (-1)                                                   ## reset range index for new obs. data file


                ## determine current range index
                elif (StrippedLine.find("<FrequencyRange>") > (-1)):
                    CurrentRangeID += 1                                                     ## increase range index


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## remove tag and try to remove corresponding comment as well
            if (StrippedLine.find("<!-- " + tagName + " -->") > (-1) or StrippedLine.find("<!--" + tagName + "-->") > (-1)):
                AddFlag = False
            elif ((not StrippedLine.startswith("<!--")) and StrippedLine.find("<" + tagName + ">") > (-1)):
                AddFlag = False
        if (AddFlag):
            NewContentsXMLFile.append(line)
    ContentsXMLFile = copy.deepcopy(NewContentsXMLFile)


    ## finished
    return ContentsXMLFile
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## write contents in ContentsTag to xml file GetXMLtag xml at tags with name tagName
##
def CreateRunDirectory(NameOfFunction, myXCLASSrootDir, PrintFlag):
    """

input parameters:
-----------------

    - NameOfFunction:       name of function of the XCLASS interface

    - myXCLASSrootDir:      path and name of root directory

    - PrintFlag:            flag for writing messages to screen



output parameters:
------------------

    - RunDir:               name of run directory for the current function
    """

    # Debug:
    # print ("NameOfFunction = ", NameOfFunction)
    # print ("myXCLASSrootDir = ", myXCLASSrootDir)
    # print ("PrintFlag = ", PrintFlag)


    ## get settings from init file
    OutDict = task_myXCLASS.GetXCLASSSettings()
    XCLASSSystemRootDirectory = OutDict['XCLASSSystemRootDirectory']
    TimeStampFormat = OutDict['TimeStampFormat']

    # Debug:
    # print ("XCLASSSystemRootDirectory = ", XCLASSSystemRootDirectory)
    # print ("TimeStampFormat = ", TimeStampFormat)


    ##====================================================================================================================================================
    ## create temp directory


    ## determine job id, i.e. create a number to identify job unambiguous
    lt = time.localtime()
    jobID = "_"
    if (TimeStampFormat == "dd-mm-yyyy"):
        jobID += time.strftime("%d-%m-%Y", lt)
    elif (TimeStampFormat == "mm-dd-yyyy"):
        jobID += time.strftime("%m-%d-%Y", lt)
    else:
        jobID += time.strftime("%Y-%m-%d", lt)
    jobID += "__" + time.strftime("%H-%M-%S", lt)
    pidRaw = os.getpid()
    pid = "{:05d}".format(pidRaw)
    ranNum = int(random.uniform(1000, 9999))
    # jobID += "__{:s}{:d}".format(pid[:5], ranNum)
    jobID += "__{:d}{:d}".format(pidRaw, ranNum)

    # Debug:
    # print ("jobID = ", jobID)


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## define name of job directory


    ## get path of XCLASS job directories from environment variable
    RunDir = ""
    NamesOfXCLASSJobDirectories = ["XCLASSJobDirectory", "XCLASSRunDirectory", "myXCLASSRunDirectory"]
    for NameOfXCLASSJobDirectory in NamesOfXCLASSJobDirectories:
        NameOfXCLASSJobDirectory = str(os.environ.get(NameOfXCLASSJobDirectory,"")).strip()
        if (NameOfXCLASSJobDirectory != ""):
            if (NameOfXCLASSJobDirectory[0] != "/"):                                        ## make path absolute
                RunDir = myXCLASSrootDir + NameOfXCLASSJobDirectory
            else:
                RunDir = NameOfXCLASSJobDirectory
            break

            # Debug:
            # print ("NameOfXCLASSJobDirectory = ", NameOfXCLASSJobDirectory)


    ## set path of XCLASS job directories to default position using the given XCLASS root directory
    if (RunDir == ""):


        ## get home directory
        HomeDir = str(os.environ.get('HOME',''))
        HomeDir = os.path.normpath(HomeDir) + "/"

        # Debug:
        # print ("HomeDir = ", HomeDir)


        ## set default path for job directories
        RunDir = "{:s}.xclass/run/".format(HomeDir)
        RunDir = os.path.normpath(RunDir) + "/"


        ## make sure, that run directory is already created
        if (not os.path.isdir(RunDir)):
            command_string = "mkdir -p {:s}".format(RunDir)
            os.system(command_string)

    # Debug:
    # print ("RunDir = ", RunDir)


    ## define path of current job directory
    RunDir += "{:s}/job_{:s}/".format(NameOfFunction, jobID)
    RunDir = os.path.normpath(RunDir) + "/"
    if (PrintFlag):
        print ("\nStart function " + NameOfFunction + ":")
        print ("\n\nCreating job directory for function {:s}:".format(NameOfFunction), flush = True)
        print ("\t{:s}".format(RunDir), flush = True)


    ## create temp directory
    command_string = "mkdir -p " + RunDir
    os.system(command_string)
    if (PrintFlag):
        print ("\nAll files of the current {:s} run are stored here!\n\n".format(NameOfFunction))


    ## define return variable
    return RunDir
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## set up environment variables for MAGIX call
##
def SetMAGIXEnvironment(MAGIXrootDir):
    """

input parameters:
-----------------

    - MAGIXrootDir:         MAGIX root directory



output parameters:
------------------

    - None
    """

    # Debug:
    # print ("MAGIXrootDir = ", MAGIXrootDir)


    ## check if sys.path variable is already updated and modify sys path variable if necessary
    ListOfNewPaths = [MAGIXrootDir + "Modules/python",
                      MAGIXrootDir + "Modules/Additional_Packages/bin",
                      MAGIXrootDir + "Modules/INS/bin",
                      MAGIXrootDir + "Modules/Error/bin",
                      MAGIXrootDir + "Modules/MCMC/bin",
                      MAGIXrootDir + "Modules/UltraNest/bin",
                      MAGIXrootDir + "../External_Packages/emcee/version__3.0.2/modified"]
    task_myXCLASS.ExtendSysPath(ListOfNewPaths)

    # Debug:
    # print ("ListOfNewPaths = ", ListOfNewPaths)


    ## check MAGIX temp directory
    MAGIXTempDirectory = str(os.environ.get('MAGIXTempDirectory',''))
    if (MAGIXTempDirectory.strip() == ""):
        os.environ["MAGIXTempDirectory"] = MAGIXrootDir + 'temp/'
        MAGIXTempDirectory = str(os.environ.get('MAGIXTempDirectory','')).strip()
        print ("Set environment variable MAGIXTempDirectory = ", MAGIXTempDirectory)


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start MAGIX without command line call
##
def StartMAGIX(MAGIXrootDir, option, filename_control):
    """

input parameters:
-----------------

    - MAGIXrootDir:         MAGIX root directory

    - option:               run option

    - filename_control:     path and name of io control file



output parameters:
------------------

    - ok:                   status variable
    """

    # Debug:
    # print ("MAGIXrootDir = ", MAGIXrootDir)
    # print ("option = ", option)
    # print ("filename_control = ", filename_control)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read commands from command line
    ok = 0
    printflag = "true"
    plotflag = "true"
    debugflag = "false"
    modelflag = "false"
    if (len(option) > 1):
        OptionList = option.split(",")
        for SingleOption in OptionList:                                                     ## loop over all run flags
            StrippedSingleOption = SingleOption.strip()                                     ## remove leading and tailing blanks
            StrippedSingleOption = StrippedSingleOption.lower()                             ## convert all characters to lower casa


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get start flags
            if (StrippedSingleOption == "debug"):
                debugflag = "true"

            elif (StrippedSingleOption.startswith("model")):
                modelflag = "true"
                SplitModel = StrippedSingleOption.split("=")
                if (len(SplitModel) > 1):
                    modelflag = SplitModel[1]

            elif (StrippedSingleOption == "quiet"):
                plotflag = "saveonly"
                printflag = "false"

            elif (StrippedSingleOption == "noplot"):
                plotflag = "false"

            elif (StrippedSingleOption == "plotsaveonly"):
                plotflag = "saveonly"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check if io-control file exists
    if not(os.path.exists(filename_control)):
        ok = 1
        print ("\nError in io_control.xml file!")
        print ("\t\tThe file ",filename_control)
        print ("\t\tdoes not exists!")
        print ("\n\t\tEnter the path and the name of the")
        print ("\t\tio_control xml-file at the command line:")
        print ("\n\t\tFor example:")
        print ("\t\t\t./magix_start.py run/examples/Levenberg-Marquardt/io_control.xml\n")
        return ok

    # Debug:
    # print ("printflag = ", printflag)
    # print ("plotflag = ", plotflag)
    # print ("debugflag = ", debugflag)
    # print ("modelflag = ", modelflag)
    # print ("filename_control = ", filename_control)


    ## check MAGIX temp directory
    MAGIXTempDirectory = str(os.environ.get('MAGIXTempDirectory',''))
    if (MAGIXTempDirectory.strip() == ""):
        MAGIXTempDirectory = MAGIXrootDir + 'temp/'


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## determine job id, i.e. create a number to identify job unambiguous
    UniqueFlag = False
    while (not UniqueFlag):


        ## generate a random 16 bit integer number
        jobID = int(random.uniform(1, 2**16))


        ## check, if new job id already exists
        NewJobDirectory = MAGIXTempDirectory + "job_" + str(jobID) + "/"
        if (not (os.path.isdir(NewJobDirectory))):
            UniqueFlag = True


    ## convert number to string
    jobID = str(int(jobID))

    # Debug:
    # print ("jobID = ", jobID)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start MAGIX


    ## import first MAGIX module
    from .addons.MAGIX.Modules.python import MainProgram


    ## define path of MAGIX binaries
    LibDir = MAGIXrootDir + "../../lib/"
    LibDir = os.path.normpath(LibDir) + "/"

    # Debug:
    # print ("LibDir = ", LibDir)


    ## start main program of MAGIX
    ok = MainProgram.main(printflag, plotflag, debugflag, modelflag, filename_control, jobID, MAGIXrootDir, LibDir)


    ## define return value
    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## create i/o XML file
##
def CreateIOXMLFile(MAGIXjobDir, NewObsXMLFileName, NewInstanceXMLFileName, NewAlgorithmXMLFileName, MAGIXLogFile, NewRegistrationXMLFileName, \
                    LocalPrintFlag = True):
    """

input parameters:
-----------------

    - MAGIXjobDir:                  current job directory

    - NewObsXMLFileName:            path and name of obs. xml file

    - NewInstanceXMLFileName:       path and name of instance file

    - NewAlgorithmXMLFileName:      path and name of algorithm file

    - MAGIXLogFile:                 path and name of log file

    - NewRegistrationXMLFileName:   path and name of registration file

    - LocalPrintFlag:               (optional) local print flag



output parameters:
------------------

    - None
    """

    # Debug:
    # print ("MAGIXrootDir = ", MAGIXrootDir)
    # print ("NewObsXMLFileName = ", NewObsXMLFileName)
    # print ("NewInstanceXMLFileName = ", NewInstanceXMLFileName)
    # print ("NewAlgorithmXMLFileName = ", NewAlgorithmXMLFileName)
    # print ("MAGIXLogFile = ", MAGIXLogFile)
    # print ("NewRegistrationXMLFileName = ", NewRegistrationXMLFileName)
    # print ("LocalPrintFlag = ", LocalPrintFlag)


    ##====================================================================================================================================================
    ## construct io_control.xml file


    ## print what you do
    if (LocalPrintFlag):
        print ("\nCreate io-control xml-file ..", end = " ")


    ## write i/o control file
    iocontrol = open(MAGIXjobDir + "io_control.xml", 'w')
    iocontrol.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "ISO-8859-1" + chr(34) + "?>\n")
    iocontrol.write("<ioControl>\n")
    iocontrol.write("    <title>io control</title>\n")
    iocontrol.write("    <description>This xml-file contains the paths and the names of all files used by MAGIX</description>\n")
    iocontrol.write("    <PathFilename>\n")
    iocontrol.write("\n")
    iocontrol.write("        <experimental_data>\n")
    iocontrol.write("            <filename>" + NewObsXMLFileName + "</filename>\n")
    iocontrol.write("            <description>path and file name of the experimental file</description>\n")
    iocontrol.write("        </experimental_data>\n")
    iocontrol.write("\n")
    iocontrol.write("        <parameter_file>\n")
    iocontrol.write("            <filename>" + NewInstanceXMLFileName + "</filename>\n")
    iocontrol.write("            <description>path and file name of the xml-file including the start values of each parameter</description>\n")
    iocontrol.write("        </parameter_file>\n")
    iocontrol.write("\n")
    iocontrol.write("        <fit_control>\n")
    iocontrol.write("            <filename>" + NewAlgorithmXMLFileName + "</filename>\n")
    iocontrol.write("            <description>path and file name of the xml file controlling the fitting process</description>\n")
    iocontrol.write("        </fit_control>\n")
    iocontrol.write("\n")
    iocontrol.write("        <fit_log>\n")
    iocontrol.write("            <filename>" + MAGIXLogFile + "</filename>\n")
    iocontrol.write("            <description>path and file name of log file describing the fitting process</description>\n")
    iocontrol.write("        </fit_log>\n")
    iocontrol.write("\n")
    iocontrol.write("        <model_description>\n")
    iocontrol.write("            <filename>" + NewRegistrationXMLFileName + "</filename>\n")
    iocontrol.write("            <description>path and file name of the xml-registration file</description>\n")
    iocontrol.write("        </model_description>\n")
    iocontrol.write("\n")
    iocontrol.write("    </PathFilename>\n")
    iocontrol.write("</ioControl>\n")
    iocontrol.write("\n")
    iocontrol.close()


    ## print what you do
    if (LocalPrintFlag):
        print ("done!\n\n")


    ## define return value
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## The function defines the interface to MAGIX
##
def MAGIX(MAGIXExpXML, MAGIXInstanceXML, MAGIXFitXML, MAGIXRegXML, MAGIXOption):
    """

    The function starts the MAGIX program.

    This function copies the different xml-files (except the registration xml-file)
    to a so-called "job-directory" located in the run working directory
    "path-of-myXCLASS-CASA-Interface/run/MAGIX/"! The name of a job directory for a
    MAGIX run is made up of four components: The first part of the name consists of the
    phrase "job_" whereas the second part describes the date (day, month, year), the
    third part the time stamp (hours, minutes, seconds) of the function execution. The
    last part describes a so-called job ID which is composed of the so-called PID
    followed by a four digit random integer number to create a really unambiguous job
    number, e.g.
    "path-of-myXCLASS-CASA-Interface/run/MAGIX/job__25-07-2013__12-02-03__189644932/"

    All file(s), which are created by the current MAGIX run, are stored in such a job
    directory!

    Additionally, the function copies all xml-files (except the so-called
    registration xml-file) to the current job directory.
    Furthermore, the function copies the experimental data file(s), i.e., the files
    which contains the experimental/observational data, to the current MAGIX job
    directory as well. The path(s) of the experimental data file(s) defined in the
    experimental xml-file are adjusted, so that these path(s) become absolute
    and point to the current job directory. Please note, that all modifications are
    applied to the copies of the xml-files. The original xml-files are not modified.

    As mentioned above, the registration xml-file is neither copied to the job
    directory nor modified! But, if the path of the registration xml-file is
    relative, the path has to be defined relative to the myXCLASS working directory
    "path-of-myXCLASS-CASA-Interface/"

    The io-control file (which is required for each MAGIX run) is created
    automatically!


input parameters:
-----------------

    - MAGIXExpXML:          path and name of the experimental xml-file.

                            Note, if the parameter defines a relative path, this path has to be defined relative to the
                            current working directory!


    - MAGIXInstanceXML:     path and name of the instance xml-file.

                            Note, if the parameter defines a relative path, this path has to be defined relative to the
                            current working directory!


    - MAGIXFitXML:          path and name of the xml file controlling the fitting process.

                            Note, if the parameter defines a relative path, this path has to be defined relative to the
                            current working directory!


    - MAGIXRegXML:          path and name of the so-called registration xml file containing the description of
                            the input and output files of the external model program.

                            Note, if the parameter defines a relative path, this path has to be defined relative to the
                            current working directory!


    - MAGIXOption:          option for the MAGIX run (default is None):

                            - None (default): all information are printed out to the screen
                            - quiet:          no information are printed to the screen except warning and error messages
                            - plotsaveonly:   MAGIX disables the interactive GUI of matplotlib but creates all plots and
                                              saves them into files
                            - debug:          Stop MAGIX after the first function call. This flag can be very helpful to
                                              analyze problems occurring with the call of the external model program.





output parameters:
------------------

    - myXCLASSJobDir:       absolute path of the myXCLASS job directory created for the current run.




Example:
--------

MAGIXExpXML = "demo/MAGIX/TwoOscillators_RefFit_R.xml"
MAGIXInstanceXML = "demo/MAGIX/parameters.xml"
MAGIXFitXML = "demo/MAGIX/Levenberg-Marquardt_Parameters.xml"
MAGIXRegXML = "Fit-Functions/Drude-Lorentz_conv/xml/Conventional_Drude-Lorentz.xml"
MAGIXOption = None

myXCLASSJobDir = MAGIX()
    """

    # Debug:
    # print ("MAGIXExpXML = ", MAGIXExpXML)
    # print ("MAGIXInstanceXML = ", MAGIXInstanceXML)
    # print ("MAGIXFitXML = ", MAGIXFitXML)
    # print ("MAGIXRegXML = ", MAGIXRegXML)
    # print ("MAGIXOption = ", MAGIXOption)


    ##====================================================================================================================================================
    ## initialize function


    ## define name of function
    NameOfFunction = "MAGIX"


    ## get current working directory
    CurrentDir = os.getcwd() + "/"


    ## initialize return parameters
    MAGIXjobDir = ""


    ## check, if MAGIX package is available
    OutDict = task_myXCLASS.GetXCLASSSettings()
    NoMAGIXFlag = OutDict['NoMAGIXFlag']
    if (NoMAGIXFlag):
        print ("\n\nError in XCLASS package, function MAGIX:")
        print ("\n\tThe MAGIX package was not compiled and can not be used!")
        print ("\n\tPlease recompile XCLASS including MAGIX and start again!\n\n")
        sys.exit(0)


    ##====================================================================================================================================================
    ## create run directory for the current function call
    LocalPrintFlag = True
    MAGIXrootDir = task_myXCLASS.GetMAGIXRootDir()
    MAGIXjobDir = CreateRunDirectory("MAGIX", MAGIXrootDir, LocalPrintFlag)

    # Debug:
    # print ("MAGIXjobDir = ", MAGIXjobDir)


    ## extend sys.path to include MAGIX packages
    SetMAGIXEnvironment(MAGIXrootDir)


    ##====================================================================================================================================================
    ## check existence of obs. xml file and copy obs. xml file to current working directory
    LocalPrintFlag = True
    NameOfFile = "obs. xml"
    MAGIXExpXML = MAGIXExpXML.strip()
    NewObsXMLFileName = task_myXCLASS.CheckNCopy(MAGIXExpXML, NameOfFunction, NameOfFile, CurrentDir, MAGIXjobDir, LocalPrintFlag)
    if (NewObsXMLFileName == ""):                                                           ## did an error occurred
        return MAGIXjobDir

    # Debug:
    # print ("NewObsXMLFileName = ", NewObsXMLFileName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get path and names of obs. data file
    ObsDataFileNameList = GetXMLtagNEW(NewObsXMLFileName, "FileNamesExpFiles")


    ## check 'n' copy
    NewObsDataFileNameList = []
    for LocalObsDataFileID, LocalObsDataFile in enumerate(ObsDataFileNameList):
        LocalRangeIndex = (-1)


        ## get path and name of current obs data file
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(LocalObsDataFileID, LocalRangeIndex, ObsDataFileNameListIn = ObsDataFileNameList)
        LocalObsDataFileName = ObsXMLParameterDictFile['ObsDataFileName']
        if (LocalObsDataFileName != "" and LocalObsDataFileName is not None):

            # Debug:
            # print ("LocalObsDataFileName = ", LocalObsDataFileName)


            ## copy obs. data file to job directory
            NameOfFile = "obs. data"
            LocalPrintFlag = False
            NewObsDataFileName = task_myXCLASS.CheckNCopy(LocalObsDataFileName, NameOfFunction, NameOfFile, CurrentDir, MAGIXjobDir, LocalPrintFlag)
            if (NewObsDataFileName == ""):                                                      ## did an error occurred
                return MAGIXjobDir
            NewObsDataFileNameList.append([LocalObsDataFileID, NewObsDataFileName])


    ## modify experimental xml-file in MAGIX temp directory
    WriteXMLtagNEW(NewObsXMLFileName, "FileNamesExpFiles", NewObsDataFileNameList)

    # Debug:
    # print ('NewObsDataFileNameList = ', NewObsDataFileNameList)


    ##====================================================================================================================================================
    ## check existence of instance xml file and copy instance xml file to current working directory
    LocalPrintFlag = True
    NameOfFile = "instance xml"
    MAGIXInstanceXML = MAGIXInstanceXML.strip()
    NewInstanceXMLFileName = task_myXCLASS.CheckNCopy(MAGIXInstanceXML, NameOfFunction, NameOfFile, CurrentDir, MAGIXjobDir, LocalPrintFlag)
    if (NewInstanceXMLFileName == ""):                                                      ## did an error occurred
        return MAGIXjobDir

    # Debug:
    # print ("NewInstanceXMLFileName = ", NewInstanceXMLFileName)


    ##====================================================================================================================================================
    ## check existence of instance xml file and copy instance xml file to current working directory
    LocalPrintFlag = True
    NameOfFile = "algorithm xml"
    MAGIXFitXML = MAGIXFitXML.strip()
    NewAlgorithmXMLFileName = task_myXCLASS.CheckNCopy(MAGIXFitXML, NameOfFunction, NameOfFile, CurrentDir, MAGIXjobDir, LocalPrintFlag)
    if (NewAlgorithmXMLFileName == ""):                                                     ## did an error occurred
        return MAGIXjobDir

    # Debug:
    # print ("NewAlgorithmXMLFileName = ", NewAlgorithmXMLFileName)


    ##====================================================================================================================================================
    ## check existence of instance xml file and copy instance xml file to current working directory
    LocalPrintFlag = True
    NameOfFile = "registration xml"
    MAGIXRegXML = MAGIXRegXML.strip()
    NewRegistrationXMLFileName = task_myXCLASS.CheckNCopy(MAGIXRegXML, NameOfFunction, NameOfFile, CurrentDir, MAGIXjobDir, LocalPrintFlag)
    if (NewRegistrationXMLFileName == ""):                                                  ## did an error occurred
        return MAGIXjobDir

    # Debug:
    # print ("NewRegistrationXMLFileName = ", NewRegistrationXMLFileName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check option for MAGIX run
    MAGIXOptionList = ["noplot", "plotsaveonly", "quiet", "debug"]
    if (MAGIXOption is None or MAGIXOption.strip() == ""):
        MAGIXOption = ""
    elif (not MAGIXOption.lower() in MAGIXOptionList):
        print ("\n\nError in XCLASS package, function MAGIX:")
        print ("\tThe given option for the MAGIX run is neither 'noplot' nor 'plotsaveonly' nor 'quiet' nor 'None' nor 'debug'")
        print ("\n\tPlease enter one of the above given options and redo MAGIX call!")
        return MAGIXjobDir
    else:
        MAGIXOption = MAGIXOption.lower()


    ##====================================================================================================================================================
    ## define name of log file
    MAGIXLogFile = MAGIXjobDir + "fit.log"


    ##====================================================================================================================================================
    ## construct io_control.xml file
    CreateIOXMLFile(MAGIXjobDir, NewObsXMLFileName, NewInstanceXMLFileName, NewAlgorithmXMLFileName, MAGIXLogFile, NewRegistrationXMLFileName)


    ##====================================================================================================================================================
    ## start MAGIX
    ok = StartMAGIX(MAGIXrootDir, MAGIXOption, MAGIXjobDir + "io_control.xml")
    if (ok != 0):
        print ("\n\n\n\t Program MAGIX aborted!\n\n")


    ## import warning:
    else:
        print ("\n\nIMPORTANT:")
        print ("----------")
        print ("\nAll files of the current MAGIX run are stored in the MAGIX job directory: " + MAGIXjobDir)
        print ("\n")


    ##====================================================================================================================================================
    ## define return variables
    return MAGIXjobDir
##--------------------------------------------------------------------------------------------------------------------------------------------------------

