!*********************************************************************************************************************************************************
!>  Module:  GlobalVariables
!>
!>
!>  This module contains some global variables which are used in several other routine
!>  Copyright (C) 2012 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine WriteModelCube:                write model cube to FITS file
!>      - subroutine PrintERROR:                    prints out the descriptive text corresponding to the error status value
!>      - subroutine DeleteFile:                    a simple little routine to delete a FITS file
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    2012-05-30  Initial version
!>  T. Moeller    2018-03-15  add subroutine for writing model cube to FITS file
!>
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module GlobalVariables

    use iso_fortran_env                                                                     !< only available for fortran 2003 and later
    use myXCLASSCore

    implicit none
    real*8 :: telescope_size                                                                !< size of the telescope
    real*8 :: telescope_BMIN                                                                !< parameter for BMIN (telescope)
    real*8 :: telescope_BMAJ                                                                !< parameter for BMAJ (telescope)
    real*8 :: telescope_BPA                                                                 !< parameter for BPA (telescope)

    !< parameters read from molfit file
    integer, parameter :: MaxNumMol = 1000                                                  !< define max. number of molecules
    integer, parameter :: MaxNumComp = 1000                                                 !< define max. number of components
    character(len=4096) :: InstanceFileName                                                 !< path and file name of the instance file
    character(len=40), dimension(MaxNumMol) :: NameMolecule                                 !< name of the current molecule

    !< variables for rad-trans parameter
    integer :: NumberMoleculeRadTrans                                                       !< number of entries in table RadTrans
    integer, allocatable, dimension(:) :: UpperStateDegeneracy                              !< upper state degeneracy
    real*8, allocatable, dimension(:) :: lFreq                                              !< frequencies of transition
    real*8, allocatable, dimension(:) :: lFreqErr                                           !< uncertainty of frequencies of transition
    real*8, allocatable, dimension(:) :: lElt                                               !< energies of lower states
    real*8, allocatable, dimension(:) :: icatMol                                            !< intensity of the transitions
    real*8, allocatable, dimension(:) :: EinsteinA                                          !< Einstein A coefficients
    logical :: use_intensity_flag                                                           !< flag for using intensity instead of Einstein A coefficient
    logical :: nHFlagCommLine                                                               !< flag indicating that dust parameters are globally definied
    logical :: inter_flag                                                                   !< flag for indicating interferometric data


    contains


        !>************************************************************************************************************************************************
        !> subroutine: WriteModelCube
        !>
        !> write model cube to FITS file
        !>
        !> input variables:     ModeID:                         debug mode
        !>                      LocalNumFreqPoints:             number of data points for current freq. range
        !>                      NumX:                           number of model pixels along x-direction
        !>                      NumY:                           number of model pixels along y-direction
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2018-03-15
        !>
        subroutine WriteModelCube(ModeID, LocalNumFreqPoints, NumX, NumY)

            implicit none
            integer :: ModeID                                                               !< debug mode
            integer :: NumX, NumY                                                           !< number of model pixels along x- and y-direction
            integer :: LocalNumFreqPoints                                                   !< local number of freq. points
            integer :: status                                                               !< status parameter
            integer :: unit                                                                 !< logical unit number to use to open the FITS file
            integer :: blocksize                                                            !< historical artifact
            integer :: bitpix                                                               !< bit length of cube numbers
            integer :: group                                                                !< number of elements
            integer :: fpixel                                                               !<
            integer :: ModelIDy, ModelIDx                                                   !< loop variables for sub-beam description
            integer :: con                                                                  !< local configuration
            integer :: l                                                                    !< frequency range index
            integer :: FirstIndex                                                           !< index for first freq. data point
            integer :: LastIndex                                                            !< index for last freq. data point
            integer :: FreqIndex                                                            !< local freq. index
            integer :: ObsFreqIndex                                                         !< local obs. freq. index
            integer :: nelements                                                            !< number of elements
            integer :: naxis                                                                !< number of axis
            integer, dimension(4) :: naxes                                                  !< number of points for each axis
            real*8 :: ObsFreq                                                               !< frequency point in obs. data file
            real*8 :: freq_t                                                                !< working variable: frequency as temperature
            real*8 :: j_back                                                                !< brightness for background temperature
            real*8 :: f0, f1, LocalTelescopeSize                                            !< working variables for single dish beam
            real*8, dimension(NumX, NumY, LocalNumFreqPoints, 1) :: WorkingMap              !< working map
            character(len=4096) :: filename                                                 !< path and name of file
            logical :: simple                                                               !< use simple flag
            logical :: extend                                                               !< use extended header flag

            ! Debug:
            !    print*,"ModeID = ", ModeID
            !    print*,"LocalNumFreqPoints = ", LocalNumFreqPoints
            !    print*,"NumX = ", NumX
            !    print*,"NumY = ", NumY


            !< The STATUS parameter must be initialized before using FITSIO. A positive value of STATUS is returned whenever a serious error occurs.
            !< FITSIO uses an `inherited status' convention, which means that if a subroutine is called with a positive input value of STATUS, then the
            !< subroutine will exit immediately, preserving the status value. For simplicity, this program only checks the status value at the end of
            !< the program, but it is usually better practice to check the status value more frequently.
            status = 0


            !< Name of the FITS file to be created:
            if (ModeID == 0) then                                                           !< get intensity at current frequency point
                filename = "ModelCube__total-intensities.fits"
            elseif (ModeID == 1) then
                filename = "ModelCube__total-intensities__non-convolved.fits"
            endif


            !< Delete the file if it already exists, so we can then recreate it.
            call DeleteFile(filename, status)


            !< Get an unused Logical Unit Number to use to open the FITS file. This routine is not required;  programmers can choose any unused
            !< unit number to open the file.
            call ftgiou(unit, status)


            !< Create the new empty FITS file. The blocksize parameter is a historical artifact and the value is ignored by FITSIO.
            blocksize = 1
            call ftinit(unit, filename, blocksize, status)


            !< Initialize parameters about the FITS image. BITPIX = 16 means that the image pixels will consist of 16-bit integers. The size of
            !< the image is given by the NAXES values. The EXTEND = TRUE parameter indicates that the FITS file may contain extensions following
            !< the primary array.
            simple = .true.
            bitpix = -64
            naxis = 4                                                                       !< we have four axis
            naxes(1) = NumX                                                                 !< number of model pixels along x-direction
            naxes(2) = NumY                                                                 !< number of model pixels along y-direction
            naxes(3) = LocalNumFreqPoints                                                   !< define number of frequency points
            naxes(4) = 1                                                                    !< we have no polarization angles
            extend = .true.                                                                 !< we're using an extended FITS header


            !< Write the required header keywords to the file
            call ftphpr(unit, simple, bitpix, naxis, naxes, 0, 1, extend, status)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< create model cube
            l = 1
            FirstIndex = DataPointIndexFreqRange(l, 1)                                      !< get index of first freq. point in 'ObservationalDataList'
            WorkingMap = 0.d0
            if (ModeID == 0) then                                                           !< get intensity at current frequency point
                WorkingMap(:, :, :, 1) =  IntTotalSubBeam(:, :, :)
            elseif (ModeID == 1) then


                !< create model map with intensities of each pixel spectrum at all frequency points of the current range
                Do ModelIDy = 1, NumY                                                       !< loop over all pixels of model map along y-direction
                    Do ModelIDx = 1, NumX                                                   !< loop over all pixels of model map along x-direction
                        con = ConfigIndex(ModelIDx, ModelIDy)
                        if (con > 0) then
                            WorkingMap(ModelIDx, ModelIDy, :, 1) = ModelSpectrumPixel(con, :)
                        else
                            Do FreqIndex = 1, LocalNumFreqPoints
                                ObsFreqIndex = FirstIndex + FreqIndex - 1
                                ObsFreq = ObservationalDataList(ObsFreqIndex, 1)
                                freq_t = MHz2Kelvin * ObsFreq
                                j_back = freq_t / (ExpSave(freq_t / Tcbg) - 1.d0)
                                if (dabs(BackgroundTemperatureRange(l)) > tiny(1.d0)) then
                                    j_back = j_back + BackgroundTemperatureRange(l) * (ObsFreq / StartFrequency(l))**TemperatureSlopeRange(l)
                                endif
                                WorkingMap(ModelIDx, ModelIDy, FreqIndex, 1) = j_back
                            end Do
                        endif
                    end Do                                                                  !< ModelIDx: loop over all pixels of model map along x-dir.
                end Do                                                                      !< ModelIDy: loop over all pixels of model map along y-dir.
            endif


            !< Write the array to the FITS file. The last letter of the subroutine name defines the datatype of the array argument; in this case
            !< the 'J' indicates that the array has an integer*4 datatype. ('I' = I*2, 'E' = Real*4, 'D' = Real*8). The 2D array is treated as a
            !< single 1-D array with NAXIS1 * NAXIS2 total number of pixels. GROUP is seldom used parameter that should almost always be set = 1.
            group = 1                                                                       !<
            fpixel = 1                                                                      !<
            nelements = product(naxes(:))                                                   !< get the total number of elements
            call ftpprd(unit, group, fpixel, nelements, WorkingMap, status)


            !< Write another optional keyword to the header
            call ftpkyd(unit, "BSCALE", 1.d0, 14, "PHYSICAL = PIXEL * BSCALE + BZERO", status)
            call ftpkyd(unit, "BZERO", 0.d0, 14, " ", status)
            if (dabs(telescope_size) > tiny(1.d0)) then
                if (inter_flag) then
                    LocalTelescopeSize = telescope_size
                else
                    FirstIndex = DataPointIndexFreqRange(l, 1)                              !< get index of first freq. point in 'ObservationalDataList'
                    LastIndex = DataPointIndexFreqRange(l, 2)                               !< get index of last freq. point in 'ObservationalDataList'
                    f0 = ObservationalDataList(FirstIndex, 1)
                    f1 = ObservationalDataList(LastIndex, 1)
                    LocalTelescopeSize = 1.22d-3 * ckms / (dmin1(f0, f1) * telescope_size) * (180.d0 * 3600.d0 / pi)
                endif
                call ftpkyd(unit, "BMAJ", LocalTelescopeSize, 14, " ", status)
                call ftpkyd(unit, "BMIN", LocalTelescopeSize, 14, " ", status)
                call ftpkyd(unit, "BPA", 0.d0, 14, " ", status)
            else
                call ftpkyd(unit, "BMAJ", telescope_BMAJ, 14, " ", status)
                call ftpkyd(unit, "BMIN", telescope_BMIN, 14, " ", status)
                call ftpkyd(unit, "BPA", telescope_BPA, 14, " ", status)
            endif
            call ftpkys(unit, "BTYPE", "Intensity", " ", status)
            call ftpkys(unit, "BUNIT", "KELVIN ", "Brightness (pixel) unit", status)

            call ftpkys(unit, "CTYPE1", "RA---SIN", " ", status)
            call ftpkyd(unit, "CRVAL1", 0.d0, 14, " ", status)
            call ftpkyd(unit, "CDELT1", SizeOfPixel_deg, 14, " ", status)
            call ftpkyj(unit, "CRPIX1", 1, " ", status)
            call ftpkys(unit, "CUNIT1", "deg     ", " ", status)

            call ftpkys(unit, "CTYPE2", "DEC--SIN", " ", status)
            call ftpkyd(unit, "CRVAL2", 0.d0, 14, " ", status)
            call ftpkyd(unit, "CDELT2", SizeOfPixel_deg, 14, " ", status)
            call ftpkyj(unit, "CRPIX2", 1, " ", status)
            call ftpkys(unit, "CUNIT2", "deg     ", " ", status)

            call ftpkys(unit, "CTYPE3", "FREQ    ", " ", status)
            call ftpkyd(unit, "CRVAL3", ObservationalDataList(FirstIndex, 1), 14, " ", status)
            call ftpkyd(unit, "CDELT3", ObservationalDataList(FirstIndex + 1, 1) - ObservationalDataList(FirstIndex, 1), 14, " ", status)
            call ftpkyj(unit, "CRPIX3", 1, " ", status)
            call ftpkys(unit, "CUNIT3", "MHz     ", " ", status)

            call ftpkys(unit, "CTYPE4", "STOKES  ", " ", status)
            call ftpkyd(unit, "CRVAL4", 1.d0, 14, " ", status)
            call ftpkyd(unit, "CDELT4", 1.d0, 14, " ", status)
            call ftpkyd(unit, "CRPIX4", 1.d0, 14, " ", status)
            call ftpkys(unit, "CUNIT4", "        ", " ", status)
            call ftpkyd(unit, "RESTFRQ", 0.d0, 14, "Rest Frequency (Hz)", status)


            !< The FITS file must always be closed before exiting the program. Any unit numbers allocated with FTGIOU must be freed with FTFIOU.
            call ftclos(unit, status)
            call ftfiou(unit, status)


            !< Check for any errors, and if so print out error messages. The PRINTERROR subroutine is listed near the end of this file.
            if (status > 0) call PrintERROR(status)


            !< we're done
            return
        end subroutine WriteModelCube


        !>************************************************************************************************************************************************
        !> subroutine: PrintERROR
        !>
        !> This subroutine prints out the descriptive text corresponding to the error status value and prints out the contents of the internal
        !> error message stack generated by FITSIO whenever an error occurs.
        !>
        !> input variables:     status
        !>
        !> output variables:    none
        !>
        !>
        !> \author CFITSIO cookbook
        !>
        !> \date 2018-03-15
        !>
        subroutine PrintERROR(status)

            implicit none
            integer :: status                                                               !<
            character(len=30) :: errtext                                                    !<
            character(len=80) :: errmessage                                                 !<


            !< Check if status is OK (no error); if so, simply return
            if (status <= 0) return


            !< The FTGERR subroutine returns a descriptive 30-character text string that corresponds to the integer error status number.
            !< A complete list of all the error numbers can be found in the back of the FITSIO User's Guide.
            call ftgerr(status, errtext)
            print '("FITSIO Error Status = ", A, ": ", A)', status, errtext


            !< FITSIO usually generates an internal stack of error messages whenever an error occurs. These messages provide much more information
            !< on the cause of the problem than can be provided by the single integer error status value. The FTGMSG subroutine retrieves the
            !< oldest message from the stack and shifts any remaining messages on the stack down one position. FTGMSG is called repeatedly until
            !< a blank message is returned, which indicates that the stack is empty. Each error message may be up to 80 characters in length.
            !< Another subroutine, called FTCMSG, is available to simply clear the whole error message stack in cases where one is not interested
            !< in the contents.
            call ftgmsg(errmessage)
            Do while (errmessage /= " ")
                print '(A)', errmessage
                call ftgmsg(errmessage)
            end Do


            !< we're done
            return
        end subroutine PrintERROR


        !>************************************************************************************************************************************************
        !> subroutine: DeleteFile
        !>
        !> A simple little routine to delete a FITS file
        !>
        !> input variables:     filename:                       path and name of FITS file
        !>                      status:                         status parameter
        !>
        !> output variables:    none
        !>
        !>
        !> \author CFITSIO cookbook
        !>
        !> \date 2018-03-15
        !>
        subroutine DeleteFile(filename, status)

            implicit none
            integer :: status                                                               !< status parameter
            integer :: unit                                                                 !< logical unit number to use to open the FITS file
            integer :: blocksize                                                            !< historical artifact
            character*(*) filename                                                          !< path and name of FITS file


            !< Simply return if status is greater than zero
            if (status > 0) return


            !< Get an unused Logical Unit Number to use to open the FITS file
            call ftgiou(unit, status)


            !< Try to open the file, to see if it exists
            call ftopen(unit, filename, 1, blocksize, status)
            if (status == 0) then                                                           !< file was opened;  so now delete it
                call ftdelt(unit, status)
            elseif (status == 103) then                                                     !< file doesn't exist, so ..
                status = 0                                                                  !< reset status to zero and ..
                call ftcmsg                                                                 !< clear errors
            else                                                                            !< there was some other error opening the file
                status = 0                                                                  !< reset status to zero and ..
                call ftcmsg                                                                 !< clear errors;
                call ftdelt(unit, status)                                                   !< delete the file anyway
            endif


            !< Free the unit number for later reuse
            call ftfiou(unit, status)


            !< we're done
            return
        end subroutine DeleteFile
end Module GlobalVariables
!*********************************************************************************************************************************************************
