!*********************************************************************************************************************************************************
!>  Module: CommentLineVariables
!>
!>
!>  This module contains variables and subroutines for reading in parameters from the comment line
!>  Copyright (C) 2012 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine getCommandLineArguments:       get command line arguments
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    2012-06-28  initial version
!>  T. Moeller    2017-12-16  improved version
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module CommentLineVariables

    use GlobalVariables

    integer :: PeakShape                                                                    !< shape of peak
    integer :: NumberProc                                                                   !< number of threads
    integer :: NumModelPixelXX                                                              !< number of pixel along x-direction for sub-beam description
    integer :: NumModelPixelYY                                                              !< number of pixel along y-direction for sub-beam description
    real*8 :: freqmin                                                                       !< first frequency of the simulated spectrum
    real*8 :: freqmax                                                                       !< last frequency of the simulated spectrum
    real*8 :: sim_width                                                                     !< step frequency of the simulated spectrum
    real*8 :: LocalvLSR                                                                     !< vLSR
    real*8 :: Redshift                                                                      !< red shift
    real*8 :: T_back                                                                        !< background temperature
    real*8 :: T_slope                                                                       !< temperature slope
    real*8 :: N_H                                                                           !< hydrogen column density
    real*8 :: beta_dust                                                                     !< beta for dust
    real*8 :: kappa_1300                                                                    !< kappa for dust
    real*8 :: Te_ff                                                                         !< electronic temperature for free-free contribution
    real*8 :: EM_ff                                                                         !< EM for free-free contribution
    real*8 :: kappa_sync                                                                    !< kappa for synchrotron contribution
    real*8 :: B_sync                                                                        !< magnetic field for synchrotron contribution
    real*8 :: p_sync                                                                        !< index for synchrotron contribution
    real*8 :: l_sync                                                                        !< thickness of slab for synchrotron contribution
    real*8 :: rms_mod                                                                       !< ??
    real*8, dimension(6) :: ContPhen                                                        !< parameters for phenomenological continuum description
    character(len=4096) :: LocalDustFileName                                                !< path and name of dust file
    character(len=4096) :: LocalBackgroundFileName                                          !< path and name of background file
    character(len=4096) :: LocalEmAbsPATH                                                   !< path of emission/absorption files
    logical :: tback_flag                                                                   !< flag indicating that T_Back and T_Slope describes
                                                                                            !< continuum completely
    logical :: free_free_flag                                                               !< flag for free-free continuum
    logical :: sync_flag                                                                    !< flag for synchrotron continuum
    logical :: phen_flag                                                                    !< flag for phen. description of continuum
    logical :: iso_flag                                                                     !< flag for using isotopologues or not
    logical :: DustFile_Flag                                                                !< flag indicating if dust file is used or not
    logical :: BackgroundFile_Flag                                                          !< flag indicating if background file is used or not
    logical :: FlagLocalOverlap                                                             !< flag for local overlap
    logical :: LocalNoSubBeamFlag                                                           !< flag for supressing sub-beam description
    logical :: verbose                                                                      !<
    logical :: old                                                                          !<


    contains


        !>************************************************************************************************************************************************
        !> subroutine: getCommandLineArguments
        !>
        !> get command line arguments
        !>
        !> input variables:     none
        !>
        !> output variables:    ok:             status variable
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2017-12-16
        !>
        subroutine getCommandLineArguments(ok)

            use GlobalVariables

            implicit none
            integer :: ArgumentID                                                           !< loop variable
            integer :: i                                                                    !< index variable
            integer :: ok                                                                   !< status variable
            integer :: NumberOfCommandLineArguments                                         !< number of command-line arguments
            character(len=30) :: numberString1                                              !< working variable: used for integer to string conversion
            character(len=40) :: CommandWord                                                !< command word
            character(len=4096) :: CommandValue                                             !< value for command word
            character(len=4096) :: CommandLineArgument                                      !< command-line argument
            logical :: LocalFlag                                                            !< local flag


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize return parameter
            ok = 0                                                                          !< reset status variable
            NumberOfCommandLineArguments = command_argument_count()                         !< get number of command-line arguments
            freqmin = 0.d0                                                                  !< initialize parameter for lower frequency
            freqmax = 0.d0                                                                  !< initialize parameter for upper frequency
            sim_width = 0.d0                                                                !< initialize parameter for step frequency
            telescope_size = 0.d0                                                           !< initialize parameter for size of telescope
            telescope_BMIN = 0.d0                                                           !< initialize parameter for BMIN (telescope)
            telescope_BMAJ = 0.d0                                                           !< initialize parameter for BMAJ (telescope)
            telescope_BPA = 0.d0                                                            !< initialize parameter for BPA (telescope)
            inter_flag = .false.                                                            !< initialize parameter for interferrometer flag
            Redshift = 0.d0                                                                 !< initialize parameter redshift
            tback_flag = .false.                                                            !< initialize parameter for t_back flag
            T_back = 0.d0                                                                   !< initialize parameter for background temperature
            T_slope = 0.d0                                                                  !< initialize parameter for temperature slope
            nHFlagCommLine = .false.                                                        !< initialize parameter for nH_flag
            N_H = 0.d0                                                                      !< initialize parameter for hydrogen column density
            beta_dust = 0.d0                                                                !< initialize parameter for spectral index for dust
            kappa_1300 = 0.d0                                                               !< initialize parameter for kappa for dust
            free_free_flag = .false.                                                        !< initialize parameter for free-free flag
            Te_ff = 0.d0                                                                    !< initialize parameter for Te_ff
            EM_ff = 0.d0                                                                    !< initialize parameter for EM_ff
            sync_flag = .false.                                                             !< initialize parameter for synchrotron flag
            kappa_sync = 0.d0                                                               !< initialize parameter for kappa for synchrotron
            B_sync = 0.d0                                                                   !< initialize parameter for magnetic field
            p_sync = 0.d0                                                                   !< initialize parameter for index
            l_sync = 0.d0                                                                   !< initialize parameter for thickness of slab
            phen_flag = .false.                                                             !< initialize flag for phen. description of continuum
            ContPhen = 0.d0                                                                 !< initialize parameter for ContPhen
            NumModelPixelXX = 0                                                             !< initialize parameter for number of pixel along x-direction
            NumModelPixelYY = 0                                                             !< initialize parameter for number of pixel along y-direction
            FlagLocalOverlap = .false.                                                      !< initialize parameter for local overlap
            LocalNoSubBeamFlag = .false.                                                    !< initialize parameter for NoSubBeamFlag
            DustFile_Flag = .false.                                                         !< initialize parameter for dust file flag
            LocalDustFileName = ""                                                          !< initialize parameter for dust file
            BackgroundFile_Flag = .false.                                                   !< initialize parameter for background file flag
            LocalBackgroundFileName = ""                                                    !< initialize parameter for background file
            LocalEmAbsPATH = ""                                                             !< initialize parameter for EmAbs path
            LocalvLSR = 0.d0                                                                !< initialize parameter for v_LSR
            InstanceFileName = ""                                                           !< initialize parameter for input parameter
            iso_flag = .false.                                                              !< initialize parameter iso_flag
            NumberProc = 1                                                                  !< initialize parameter NumberProc
            dbName = "../../../Database/cdms_sqlite.db"                                     !< initialize parameter dbName


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< check, if command line arguements are specified
            if (NumberOfCommandLineArguments < 1) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '("Error in program myXCLASS:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(2x, "No command-line arguments are specified!")')
                    write(ErrChannel, '(2x, "Please start myXCLASS with command-line arguments!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '("Abort myXCLASS")')
                end Do
                ok = 1
                return
            endif

            ! Debug
            !    print*,"NumberOfCommandLineArguments = ", NumberOfCommandLineArguments


            !< print what you do
            if (printflag) then
                print '(" ")'
                print '("Reading command line parameters .. ", $)'
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< analyze command line arguments
            Do ArgumentID = 1, NumberOfCommandLineArguments                                 !< loop over command line arguments


                !< get command line arguments
                call get_command_argument(ArgumentID, CommandLineArgument)
                CommandLineArgument = trim(adjustl(CommandLineArgument))
                if (CommandLineArgument(1:2) == "--") then                                  !< check, if command is defined
                    i = index(CommandLineArgument, "=")                                     !< get separation character between command word and value
                    if (i > (-1)) then                                                      !< separating character was found


                        !< get command word and command value
                        CommandWord = trim(adjustl(CommandLineArgument(3:i - 1)))
                        CommandValue = trim(adjustl(CommandLineArgument(i + 1:)))


                        !< check, if true/false flag was specified for current command value
                        LocalFlag = .false.
                        if (trim(adjustl(CommandValue)) == "t" .or. trim(adjustl(CommandValue)) == "T" .or. trim(adjustl(CommandValue)) == "true" &
                            .or. trim(adjustl(CommandValue)) == "True" .or. trim(adjustl(CommandValue)) == "TRUE" &
                            .or. trim(adjustl(CommandValue)) == "y" .or. trim(adjustl(CommandValue)) == "Y" .or. trim(adjustl(CommandValue)) == "yes" &
                            .or. trim(adjustl(CommandValue)) == "YES" .or. trim(adjustl(CommandValue)) == "Yes") then
                            LocalFlag = .true.
                        endif

                        ! Debug:
                        !    print*," "
                        !    print*,"CommandLineArgument = >>", trim(adjustl(CommandLineArgument)), "<<"
                        !    print*,"CommandWord = >>", trim(adjustl(CommandWord)), "<<"
                        !    print*,"CommandValue = >>", trim(adjustl(CommandValue)), "<<"


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< analyze command word and value


                        !< get lower frequency limit
                        if (trim(adjustl(CommandWord)) == "FreqMin") then
                            read(CommandValue,*) freqmin


                        !< get upper frequency limit
                        elseif (trim(adjustl(CommandWord)) == "FreqMax") then
                            read(CommandValue,*) freqmax


                        !< get step frequency
                        elseif (trim(adjustl(CommandWord)) == "FreqStep") then
                            read(CommandValue,*) sim_width


                        !< get size of telescope
                        elseif (trim(adjustl(CommandWord)) == "TelescopeSize") then
                            read(CommandValue,*) telescope_size


                        !< get size of BMIN
                        elseif (trim(adjustl(CommandWord)) == "BMIN") then
                            read(CommandValue,*) telescope_BMIN


                        !< get size of BMAJ
                        elseif (trim(adjustl(CommandWord)) == "BMAJ") then
                            read(CommandValue,*) telescope_BMAJ


                        !< get size of BPA
                        elseif (trim(adjustl(CommandWord)) == "BPA") then
                            read(CommandValue,*) telescope_BPA


                        !< get inter_flag
                        elseif (trim(adjustl(CommandWord)) == "Inter_Flag" .and. LocalFlag) then
                            inter_flag = .true.


                        !< get vLSR parameter
                        elseif (trim(adjustl(CommandWord)) == "vLSR") then
                            read(CommandValue,*) LocalvLSR


                        !< get Redshift parameter
                        elseif (trim(adjustl(CommandWord)) == "Redshift") then
                            read(CommandValue,*) Redshift


                        !< get t_back flag
                        elseif (trim(adjustl(CommandWord)) == "t_back_flag" .and. LocalFlag) then
                            tback_flag = .true.


                        !< get tBack
                        elseif (trim(adjustl(CommandWord)) == "tBack") then
                            read(CommandValue,*) T_back


                        !< get tslope
                        elseif (trim(adjustl(CommandWord)) == "tSlope") then
                            read(CommandValue,*) T_slope


                        !< get path and name of file describing background
                        elseif (trim(adjustl(CommandWord)) == "BackgroundFileName") then
                            LocalBackgroundFileName = trim(adjustl(CommandValue))
                            BackgroundFile_Flag = .true.


                        !< get N_H
                        elseif (trim(adjustl(CommandWord)) == "N_H") then
                            read(CommandValue,*) N_H
                            nHFlagCommLine = .true.


                        !< get beta_dust
                        elseif (trim(adjustl(CommandWord)) == "beta_dust") then
                            read(CommandValue,*) beta_dust


                        !< get kappa_1300
                        elseif (trim(adjustl(CommandWord)) == "kappa_1300") then
                            read(CommandValue,*) kappa_1300


                        !< get Te_ff
                        elseif (trim(adjustl(CommandWord)) == "Te_ff") then
                            read(CommandValue,*) Te_ff
                            free_free_flag = .true.


                        !< get EM_ff
                        elseif (trim(adjustl(CommandWord)) == "EM_ff") then
                            read(CommandValue,*) EM_ff


                        !< get kappa_sync
                        elseif (trim(adjustl(CommandWord)) == "kappa_sync") then
                            read(CommandValue,*) kappa_sync
                            sync_flag = .true.


                        !< get B_sync
                        elseif (trim(adjustl(CommandWord)) == "B_sync") then
                            read(CommandValue,*) B_sync


                        !< get p_sync
                        elseif (trim(adjustl(CommandWord)) == "p_sync") then
                            read(CommandValue,*) p_sync


                        !< get l_sync
                        elseif (trim(adjustl(CommandWord)) == "l_sync") then
                            read(CommandValue,*) l_sync


                        !< get ContPhenID
                        elseif (trim(adjustl(CommandWord)) == "ContPhenID") then
                            read(CommandValue,*) ContPhen(1)
                            phen_flag = .true.


                        !< get ContParam1
                        elseif (trim(adjustl(CommandWord)) == "ContParam1") then
                            read(CommandValue,*) ContPhen(2)


                        !< get ContParam2
                        elseif (trim(adjustl(CommandWord)) == "ContParam2") then
                            read(CommandValue,*) ContPhen(3)


                        !< get ContParam3
                        elseif (trim(adjustl(CommandWord)) == "ContParam3") then
                            read(CommandValue,*) ContPhen(4)


                        !< get ContParam4
                        elseif (trim(adjustl(CommandWord)) == "ContParam4") then
                            read(CommandValue,*) ContPhen(5)


                        !< get ContParam5
                        elseif (trim(adjustl(CommandWord)) == "ContParam5") then
                            read(CommandValue,*) ContPhen(6)


                        !< get NumModelPixelXX
                        elseif (trim(adjustl(CommandWord)) == "NumModelPixelXX") then
                            read(CommandValue,*) NumModelPixelXX


                        !< get NumModelPixelYY
                        elseif (trim(adjustl(CommandWord)) == "NumModelPixelYY") then
                            read(CommandValue,*) NumModelPixelYY


                        !< get FlagLocalOverlap
                        elseif (trim(adjustl(CommandWord)) == "LocalOverlapFlag" .and. LocalFlag) then
                            FlagLocalOverlap = .true.


                        !< get LocalNoSubBeamFlag
                        elseif (trim(adjustl(CommandWord)) == "NoSubBeamFlag" .and. LocalFlag) then
                            LocalNoSubBeamFlag = .true.


                        !< get EmAbsFlag
                        elseif (trim(adjustl(CommandWord)) == "EmAbsFlag" .and. LocalFlag) then
                            EmAbsFlag = .true.


                        !< get path and name of file describing dust file
                        elseif (trim(adjustl(CommandWord)) == "DustFileName") then
                            LocalDustFileName = trim(adjustl(CommandValue))
                            DustFile_Flag = .true.


                        !< get name (and path) of molfits file
                        elseif (trim(adjustl(CommandWord)) == "in") then
                            InstanceFileName = trim(adjustl(CommandValue))


                        !< get iso flag
                        elseif (trim(adjustl(CommandWord)) == "iso_flag" .and. LocalFlag) then
                            iso_flag = .true.


                        !< get path of folder containing files describing emission and absorption function
                        elseif (trim(adjustl(CommandWord)) == "EmAbsPATH") then
                            LocalEmAbsPATH = trim(adjustl(CommandValue))


                        !< get NumberProc
                        !< determine flag for output files (the following lines are included to avoid introducing a further command line argument
                        elseif (trim(adjustl(CommandWord)) == "log") then
                            read(CommandValue,*) NumberProc
                            if (NumberProc < 0) then
                                AllOutputFilesFlag = .true.                                 !< write intensities and optical depths of each comp. to file
                            endif


                        !< (optional) get path and name of the sqlite database
                        elseif (trim(adjustl(CommandWord)) == "dbFile") then
                            dbName = trim(adjustl(CommandValue))
                        endif
                    endif
                endif
            end Do


            !< print some status information to screen
            if (printflag) then
                print '("done!")'
                print '(" ")'
                print '("Command line arguments:")'
                print '(2x, "First frequency:              ", ES20.10)', freqmin
                print '(2x, "Last frequency:               ", ES20.10)', freqmax
                print '(2x, "Stepsize (resolution):        ", ES20.10)', sim_width
                if (telescope_BMIN /= 0.d0 .and. telescope_BMAJ /= 0.d0) then
                    print '(2x, "BMIN:                         ", ES20.10)', telescope_BMIN
                    print '(2x, "BMAJ:                         ", ES20.10)', telescope_BMAJ
                    print '(2x, "BPA:                          ", ES20.10)', telescope_BPA
                else
                    print '(2x, "Size of telescope:            ", ES20.10)', telescope_size
                endif
                print '(2x, "Interferrometer flag:             ", L1)', inter_flag
                print '(2x, "v_LSR:                        ", ES20.10)', LocalvLSR
                if (Redshift /= 0.d0) then
                    print '(2x, "Redshift:                     ", ES20.10)', Redshift
                endif
                print '(2x, "Background temperature flag:      ", L1)', tback_flag
                print '(2x, "Background Temperature:       ", ES20.10)', T_back
                print '(2x, "Temperature Slope:            ", ES20.10)', T_slope
                if (BackgroundFile_Flag) then
                    print '(2x, "Path and name of background file: ", A)', char(34) // trim(adjustl(LocalBackgroundFileName)) // char(34)
                endif
                if (nHFlagCommLine) then
                    print '(2x, "N_H:                          ", ES20.10)', N_H
                    print '(2x, "beta dust:                    ", ES20.10)', beta_dust
                    print '(2x, "kappa:                        ", ES20.10)', kappa_1300
                endif
                if (free_free_flag) then
                    print '(2x, "Te_ff:                        ", ES20.10)', Te_ff
                    print '(2x, "EM_ff:                        ", ES20.10)', EM_ff
                endif
                if (sync_flag) then
                    print '(2x, "kappa_sync:                   ", ES20.10)', kappa_sync
                    print '(2x, "B_sync:                       ", ES20.10)', B_sync
                    print '(2x, "p_sync:                       ", ES20.10)', p_sync
                    print '(2x, "l_sync:                       ", ES20.10)', l_sync
                endif
                if (phen_flag) then
                    print '(2x, "ContPhen(1:6):                ", 6(ES20.10))', ContPhen(:)
                endif
                if (NumModelPixelXX /= 0 .and. NumModelPixelYY /= 0) then
                    write(numberString1, '(I30)') NumModelPixelXX
                    print '(2x, "NumModelPixelXX:                  ", A)', trim(adjustl(numberString1))
                    write(numberString1, '(I30)') NumModelPixelYY
                    print '(2x, "NumModelPixelYY:                  ", A)', trim(adjustl(numberString1))
                endif
                print '(2x, "Local-overlap flag:               ", L1)', FlagLocalOverlap
                print '(2x, "No sub-beam flag:                 ", L1)', LocalNoSubBeamFlag
                if (EmAbsFlag) then
                    print '(2x, "Em-abs flag:                      ", L1)', EmAbsFlag
                endif
                if (trim(adjustl(LocalEmAbsPATH)) /= "") then
                    print '(2x, "Path of emission/abs. files:      ", A)', char(34) // trim(adjustl(LocalEmAbsPATH)) // char(34)
                endif
                if (DustFile_Flag) then
                    print '(2x, "Path and name of dust file:       ", A)', char(34) // trim(adjustl(LocalDustFileName)) // char(34)
                endif
                print '(2x, "Path and name of instance file:   ", A)', char(34) // trim(adjustl(InstanceFileName)) // char(34)
                print '(2x, "Iso flag:                         ", L1)', iso_flag
                print '(2x, "Path and name of database file:   ", A)', char(34) // trim(adjustl(dbName)) // char(34)
                print '(" ")'
                print '(" ")'
            endif


            !< we're done
            return
        end subroutine getCommandLineArguments
end Module CommentLineVariables
