#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module starts the MAGIX main program
##  Copyright (C) 2009 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2009-06-22      initial version
##  T. Moeller      2012-01-16      improve documentation of source code
##  T. Moeller      2016-06-01      minor changes, include MCMC algorithm and emcee package
##  T. Moeller      2019-12-22      porting to Python 3.x
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************

"""
MAGIX:
------


The modeling of astronomical observations requires specialized numerical codes
and knowledge about how to use them. The program MAGIX provides a framework to
easily interface existing codes to an iterating engine that allows automatic
minimization to constrain model parameters and provide error estimates.

MAGIX can be started by simply typing "./magix_start.py" and the command prompt.
In order to get help type "./magix_start.py --help" or read the readme.txt file.

If no screen output is desired, type "./magix_start.py --quiet" followed by the
path of the i/o control file (see below).

If you do not want to plot the resulting fit function or if you have problems
with the installation of the matplotlib package, type "./magix_start.py --noplot"
followed by the path of the i/o control file (see below).

MAGIX requires the directory "Modules" in the current working directory, which
contains all necessary python and Fortran modules. Do not rename or move this
directory.

MAGIX consists of four modules:

- Read experimental data using import filters for .dat and .fits files
- Read model parameters from xml-file defining model ranges, start values
  and fit attributes. (Fit attributes declare, if a model parameter should be
  optimized by the fit process of hold constant.)
- Start the fit process using several fit algorithm (Levenberg-Marquardt).
- Write optimized parameter to xml-file and plot data.

Detailed descriptions of the different xml files can be found in the manual.
"""

##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import random                                                                               ## import random package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start main program
##
if __name__ == '__main__':


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get current directory
    CurrentDir = os.getcwd() + "/"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## expand sys.path
    LibDir = CurrentDir + "../../lib/"
    modules_path = [LibDir + "Modules/magix-parts_python",
                    LibDir + "Modules/Additional_Packages/bin",
                    LibDir + "Modules/Interval_Nested-Sampling/bin",
                    LibDir + "Modules/ErrEstim_INS/bin",
                    LibDir + "Modules/MCMC/bin",
                    LibDir + "Modules/UltraNest/bin",
                    LibDir + "../External_Packages/emcee",
                    LibDir + "../External_Packages/emcee/version__3.0.2/modified",
                    LibDir + "../External_Packages/emcee/version__3.0.2/modified/backends",
                    LibDir + "../External_Packages/emcee/version__3.0.2/modified/moves"]
    for NewModulesPath in modules_path:                                                     ## loop over all new module paths
        NormalizedNewPath = os.path.normpath(NewModulesPath)
        if (not (NormalizedNewPath in sys.path)):
            sys.path.append(NormalizedNewPath)

    # Debug:
    # print (sys.argv = >>%s<<" % sys.argv)
    # print ("sys.path = >>%s<<" % sys.path)
    # sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read commands from command line
    printflag = "false"
    plotflag = "true"
    debugflag = "false"
    modelflag = "false"
    if (len(sys.argv) > 1):                                                                 ## check, if command line arguments are defined
        for argument in sys.argv[1:]:                                                       ## loop over all command line arguments


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get run flags
            if (argument.startswith('--')):                                                 ## run flags are marked with "--" characters
                option = argument[2:].strip()                                               ## remove "--" characters and leading and tailing blanks
                option = option.lower()


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## write version number
                if (option in ["v", "version"]):
                    print ('Version 2.0.8')


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## print python version
                elif (option == "python"):
                    print ("Using Python version: ", sys.version)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## write help to screen
                elif (option == "help"):
                    from xclass.addons.MAGIX.Modules.python import MainProgram              ## import package for experimental file
                    from xclass.addons.MAGIX.Modules.python import LoadExpFile              ## import package for experimental file
                    from xclass.addons.MAGIX.Modules.python import GetParameterSet          ## import package for reading fit parameters
                    from xclass.addons.MAGIX.Modules.python import FittingEngine            ## import package containing the fitting engine

                    print (__doc__)
                    print (MainProgram.__doc__)
                    print (LoadExpFile.__doc__)
                    print (GetParameterSet.__doc__)
                    print (FittingEngine.__doc__)
                    sys.exit(0)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## write short help to screen
                elif (option == "help_short"):
                    print ("\nMAGIX quick help:")
                    print ("-----------------")
                    print ("\nYou need:")
                    print ("\n(1.) i/o control file,")
                    print ("(2.) xml-file containing import settings for observational data,")
                    print ("(3.) xml-file with model parameters (fit attribute, value, name, upper and lower limit),")
                    print ("(4.) xml-file including all settings for the fit process.")
                    print ("\nFor more help type './magix_start.py --help' or read the pdf-files within the Documentation directory.\n")
                    sys.exit(0)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get debug flag
                elif (option == "debug"):
                    debugflag = "true"


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get model flag
                elif (option.startswith("model")):
                    modelflag = "true"
                    SplitModel = option.split("=")
                    if (len(SplitModel) > 1):
                        modelflag = SplitModel[1]


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get quiet flag
                elif (option == "quiet"):
                    plotflag = "saveonly"
                    printflag = "false"


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get noplot flag
                elif (option == "noplot"):
                    plotflag = "false"


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get plotsaveonly flag
                elif (option == "plotsaveonly"):
                    plotflag = "saveonly"


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## read path and name of the i/o control file
            else:
                filename_control = argument.strip()
                if not(os.path.exists(filename_control)):
                    print ("\n\t  Error in command line input!")
                    print ("\t      The file ",filename_control)
                    print ("\t      does not exists!")
                    print ("\n\t      Please correct your input and")
                    print ("\t      restart the program !\n")
                    sys.exit(0)

    else:
        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set default setting for i/o control file if no argument is specified
        filename_control = "run/io_control.txt"
        printflag = "true"
        plotflag = "true"
        if not(os.path.exists(filename_control)):
            print ("\n\t  Error in io_control.txt file!")
            print ("\t      The file ",filename_control)
            print ("\t      does not exists!")
            print ("\n\t      Enter the path and the file name of the")
            print ("\t      io_control.txt-file at the command line:")
            print ("\t      For example:")
            print ("\t      ./magix_start.py test/io_control.txt\n")
            sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## test, if matplotlib is available
    if (plotflag in ["true", "saveonly"]):
        try:
            import matplotlib
        except:
            plotflag == "false"
        if (plotflag != "false"):
            if (plotflag == "saveonly"):
                # matplotlib.rcParams['backend'] = 'Agg'                                      ## avoid display error
                matplotlib.use("Agg")                                                       ## avoid display error

    # Debug:
    # print (printflag = ", printflag)
    # print (plotflag = ", plotflag)
    # print (debugflag = ", debugflag)
    # print (modelflag = ", modelflag)
    # print (filename_control = ", filename_control)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import magix packages
    from xclass.addons.MAGIX.Modules.python import MainProgram
    from xclass.addons.MAGIX.Modules.python import LoadExpFile
    from xclass.addons.MAGIX.Modules.python import GetParameterSet
    from xclass.addons.MAGIX.Modules.python import FittingEngine


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get path of MAGIX temp directory and make sure that environment variable is defined properly
    MAGIXTempDirectory = str(os.environ.get('MAGIXTempDirectory',''))
    MAGIXTempDirectory = MAGIXTempDirectory.strip()
    if (MAGIXTempDirectory == ""):                                                          ## environment variable is not set
        MAGIXTempDirectory = CurrentDir + "temp/"                                           ## define path of MAGIX temp directory
        os.environ["MAGIXTempDirectory"] = MAGIXTempDirectory                               ## set environment variable 'MAGIXTempDirectory'
    elif (MAGIXTempDirectory[-1] != "/"):                                                   ## check, if variable ends with "/"
        MAGIXTempDirectory = MAGIXTempDirectory + "/"
        os.environ["MAGIXTempDirectory"] = MAGIXTempDirectory                               ## update environment variable 'MAGIXTempDirectory'

    # Debug:
    # print ("MAGIXTempDirectory = ", MAGIXTempDirectory)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create unique job id number
    UniqueFlag = False
    while (not UniqueFlag):


        ## generate a random 16 bit integer number
        jobID = int(random.uniform(1, 2**16))


        ## check, if new job id already exists
        NewJobDirectory = MAGIXTempDirectory + "job_" + str(jobID) + "/"
        if (not (os.path.isdir(NewJobDirectory))):
            UniqueFlag = True


    ## convert number to string
    jobID = str(int(jobID))


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start MAGIX main routine
    MAGIXrootDirectory = CurrentDir
    ok = MainProgram.main(printflag, plotflag, debugflag, modelflag, filename_control, jobID, MAGIXrootDirectory, LibDir)
    if (ok != 0):
        print ("\t Program MAGIX aborted!")

    # Debug:
    # print (MAGIX finished")


##
##
## program finished
##
##
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------

