#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module reads and writes the model parameter to a xml-file.
##  Copyright (C) 2009 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - function XMLRoutines.__init__:                initialize class XMLRoutines
##      - function XMLRoutines.GetTags:                 get list of tags
##      - function XMLRoutines.GetValue:                load contents of selected tag
##      - function XMLRoutines.GetValues:               load contents of XML file
##      - function XMLRoutines.GetSingleTagValue:       get value of selected tag
##      - function XMLRoutines.PrintXMLFileContent:     print contents of XML file
##      - function XMLRoutines.GetSingleObsTagValue:    get value of selected tag from obs. xml file
##      - function XMLRoutines.GetSingleObsTagAttrib:   get attribute of selected tag from obs. xml file
##      - function XMLRoutines.SetSingleObsTagValue:    set value of selected tag
##      - function XMLRoutines.SetSingleObsTagAttrib:   set attribute of selected tag
##      - function XMLRoutines.PrintXMLFileContent:     print contents of XML file
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    2019-12-27   initial version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##******************************************************************** load packages *********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import lxml.etree as ET                                                                     ## import lxml.etree package
from xclass.addons.MAGIX.Modules.python import LoadExpFile
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define class for importing FITS files
##
class XMLRoutines(object):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## initialize class XMLRoutines
    ##
    def __init__(self, XMLFileName):
        """
    input parameters:
    -----------------

        - XMLFileName:              path and name of xml file


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("XMLFileName = ", XMLFileName)


        ## taken from
        self.XMLFileName = XMLFileName
        self.tree = ET.parse(XMLFileName)
        self.root = self.tree.getroot()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## get list of tags
    ##
    def GetTags(self):
        """
    input parameters:       None

    output parameters:      TagList:
                            FullElementList:
        """


        ## get list of all tags and corresponding attributes and values
        FullElementList = [[elem.tag, elem.attrib, elem.text] for elem in self.root.iter()]


        ## get list of all used tag
        TagList = [elem.tag for elem in self.root.iter()]


        ## define return parameter
        return (TagList, FullElementList)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## load contents of selected tag
    ##
    def GetValue(self, XMLString):
        """
    input parameters:       XMLString:      tag tree

    output parameters:       None
        """

        # Debug:
        # print ("XMLString  = ", XMLString)


        ## define return parameter
        return self.root.find(XMLString)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## load contents of XML file
    ##
    def GetValues(self, XMLString):
        """
    input parameters:       XMLString:      tag tree

    output parameters:       None
        """

        # Debug:
        # print ("XMLString  = ", XMLString)


        ## define return parameter
        return self.root.findall(XMLString)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## get value of selected tag
    ##
    def GetSingleTagValue(self, XMLString, FormatString = None, DefaultValue = None):
        """
    input parameters:       XMLString:      tag tree
                            FormatString:   (optional) specifies format
                            DefaultValue:   (optional) default value

    output parameters:       TagValue:       tag value
        """

        # Debug:
        # print ("XMLString  = ", XMLString)
        # print ("FormatString  = ", FormatString)
        # print ("DefaultValue  = ", DefaultValue)


        ## get tag value
        try:
            TagValue = self.GetValue(XMLString).text
        except:
            TagValue = None
        if (TagValue is None):
            TagValue = DefaultValue
        else:
            TagValue = TagValue.strip()


            ## change format
            if (FormatString is not None):
                if (FormatString == "int"):
                    try:
                        TagValue = int(TagValue)
                    except:
                        TagValue = DefaultValue
                elif (FormatString == "float"):
                    try:
                        TagValue = float(TagValue)
                    except:
                        TagValue = DefaultValue
                elif (FormatString == "bool"):
                    TagValue = LoadExpFile.CheckBool(TagValue)
                elif (FormatString == "lower"):
                    try:
                        TagValue = TagValue.lower()
                    except:
                        TagValue = DefaultValue

        # Debug:
        # print ("TagValue = ", TagValue)


        ## define return parameter
        return TagValue
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## get value of selected tag from obs. xml file
    ##
    def GetSingleObsTagValue(self, XMLStringV1, XMLStringV2, TagString, FormatString = None, DefaultValue = None):
        """
    input parameters:       XMLStringV1:    xml tree string version 1
                            XMLStringV2:    xml tree string version 2
                            TagString:      string describing tag
                            FormatString:   (optional) specifies format
                            DefaultValue:   (optional) default value

    output parameters:       TagValue:       tag value
        """

        # Debug:
        # print ("XMLStringV1  = ", XMLStringV1)
        # print ("XMLStringV2  = ", XMLStringV2)
        # print ("TagString  = ", TagString)
        # print ("FormatString  = ", FormatString)
        # print ("DefaultValue  = ", DefaultValue)


        ## get tag value
        try:
            TagValue = self.GetValue(XMLStringV1 + TagString).text
        except:
            try:
                TagValue = self.GetValue(XMLStringV2 + TagString).text
            except:
                TagValue = None
        if (TagValue is None):
            TagValue = DefaultValue
        else:


            ## change format
            if (FormatString is not None):
                if (FormatString != "raw"):
                    TagValue = TagValue.strip()
                    if (FormatString == "int"):
                        try:
                            TagValue = int(float(TagValue))
                        except:
                            TagValue = DefaultValue
                    elif (FormatString == "float"):
                        try:
                            TagValue = float(TagValue)
                        except:
                            TagValue = DefaultValue
                    elif (FormatString == "bool"):
                        TagValue = LoadExpFile.CheckBool(TagValue)
                    elif (FormatString == "lower"):
                        TagValue = TagValue.lower()
            else:
                TagValue = TagValue.strip()

        # Debug:
        # print ("TagValue = ", TagValue)


        ## define return parameter
        return TagValue
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## get attribute of selected tag from obs. xml file
    ##
    def GetSingleObsTagAttrib(self, XMLStringV1, XMLStringV2, TagString):
        """
    input parameters:       XMLStringV1:    xml tree string version 1
                            XMLStringV2:    xml tree string version 2
                            TagString:      string describing tag

    output parameters:       TagAttrib:      tag attribute
        """

        # Debug:
        # print ("XMLStringV1  = ", XMLStringV1)
        # print ("XMLStringV2  = ", XMLStringV2)
        # print ("TagString  = ", TagString)


        ## get tag attribute
        try:
            TagAttrib = self.GetValue(XMLStringV1 + TagString).attrib
        except:
            try:
                TagAttrib = self.GetValue(XMLStringV2 + TagString).attrib
            except:
                TagAttrib = {}

        # Debug:
        # print ("TagAttrib = ", TagAttrib)


        ## define return parameter
        return TagAttrib
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## set value of selected tag
    ##
    def SetSingleObsTagValue(self, XMLStringV1, XMLStringV2, TagString, TagValue):
        """
    input parameters:       XMLStringV1:    xml tree string version 1
                            XMLStringV2:    xml tree string version 2
                            TagString:      string describing tag
                            TagValue:       value of tag

    output parameters:       None
        """

        # Debug:
        # print ("XMLStringV1  = ", XMLStringV1)
        # print ("XMLStringV2  = ", XMLStringV2)
        # print ("TagString  = ", TagString)
        # print ("TagValue  = ", TagValue)


        ## set tag value
        try:
            self.root.find(XMLStringV1 + TagString).text = TagValue
        except:
            self.root.find(XMLStringV2 + TagString).text = TagValue
        self.tree.write(self.XMLFileName, xml_declaration = True, encoding = 'UTF-8')


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## set attribute of selected tag
    ##
    def SetSingleObsTagAttrib(self, XMLStringV1, XMLStringV2, TagString, TagValue):
        """
    input parameters:       XMLStringV1:    xml tree string version 1
                            XMLStringV2:    xml tree string version 2
                            TagString:      string describing tag
                            TagValue:       value of tag

    output parameters:       None
        """

        # Debug:
        # print ("XMLStringV1  = ", XMLStringV1)
        # print ("XMLStringV2  = ", XMLStringV2)
        # print ("TagString  = ", TagString)
        # print ("TagValue  = ", TagValue)


        ## set tag attribute
        try:
            self.root.find(XMLStringV1 + TagString).attrib = TagValue
        except:
            self.root.find(XMLStringV2 + TagString).attrib = TagValue
        self.tree.write(self.XMLFileName, xml_declaration = True, encoding = 'UTF-8')


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## print contents of XML file
    ##
    def PrintXMLFileContent(self):


        ## print contents
        print (ET.tostring(self.root, encoding='utf8').decode('utf8'))


        ## we're done
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------

