#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module starts a deamon which plots the best model function values together with the observational data and the corresponding chi^2 functions
##  to a matplotlib GUI window. This GUI allows to minimize, to enlarge, to zoom and to close the window.
##  The program uses the function "gobject.timeout_add", which calls in a time interval defined in variable "TimeIntervalForUpdate", the function
##  function "update". The time interval has to be given in milliseconds. Probably, this parameter has to be optimized for some model programs.
##  Copyright (C) 2009 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - subroutine update:                            subroutine for plotting model function, exp. data, and chi^2 function ever second
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    2012-09-13   initial version
##  T. Moeller    2020-01-02   porting to python 3, minor improvments
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##******************************************************************** load packages *********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import gobject                                                                              ## load gobject package
import sys                                                                                  ## load sys package
import os                                                                                   ## load os package
import numpy as np                                                                          ## load numpy package as np
import matplotlib                                                                           ## load matplotlib package
matplotlib.use('GTKAgg')                                                                    ## sometimes useful
from pylab import *                                                                         ## load pylab package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define function for updating
##
def update(pathTemp, number_files, xlabel, ylabel, zlabel):
    """
    input parameters:       pathTemp:                   ??
                            number_files:               ??
                            xlabel:                     ??
                            ylabel:                     ??
                            zlabel:                     ??
                            MAGIXrootDirectory:         ??

    output paramters:       None
    """

    # Debug:
    # print ("pathTemp = ", pathTemp)
    # print ("number_files = ", number_files)
    # print ("xlabel = ", xlabel)
    # print ("ylabel = ", ylabel)
    # print ("zlabel = ", zlabel)


    ## check for lock-file
    LockFileName = pathTemp + "lock.txt"
    LockFileName = LockFileName.strip()
    if (os.path.isfile(LockFileName)):
        return True
    else:
        f = open(LockFileName, 'w')                                                         ## create lock-file
        f.write("\n")
        f.close()

    # Debug:
    # print ("LockFileName = ", LockFileName)
    # print (os.path.isfile(LockFileName))


    ## clear figure
    fig.clear()


    ## read in data from file
    exp_data_x = []
    exp_data_y = []
    FitFunctionValues = []
    Chi2Values = []
    countLines = 0
    for NumberFile in range(number_files):
        filename = pathTemp + "plot_data_" + str(NumberFile + 1) + ".dat"

        # Debug:
        # print ("filename = ", filename)
        # os.system("pwd")


        ## abort plot program, if no plot file exsits
        if not(os.path.exists(filename)):
            os.system('rm -rf ' + LockFileName)
            sys.exit(0)


        ## import data from file
        ioerror = "false"
        try:
            #data = np.loadtxt(filename, skiprows = 0)
            inData = open(filename)
            contentsDataFile = inData.readlines()
            inData.close()
        except IOError as ValueError:
            ioerror = "true"
        if (ioerror == "true"):
            break
        else:
            exp_data_x_local = []
            exp_data_y_local = []
            FitFunctionValues_local = []
            Chi2Values_local = []
            for line in contentsDataFile:
                countLines +=1
                linesplit = line.split()
                counter = 0
                for col in linesplit:
                    counter += 1
                    if (counter == 1):
                        exp_data_x_local.append(float(col))
                    elif (counter == 2):
                        exp_data_y_local.append(float(col))
                    elif (counter == 3):
                        FitFunctionValues_local.append(float(col))
                    elif (counter == 4):
                        Chi2Values_local.append(float(col))
            exp_data_x.append(exp_data_x_local)
            exp_data_y.append(exp_data_y_local)
            FitFunctionValues.append(FitFunctionValues_local)
            Chi2Values.append(Chi2Values_local)


            # Debug:
            # print ("exp_data_x_local = ", exp_data_x_local)
            # print ("exp_data_y = ", exp_data_y)
            # print ("FitFunctionValues = ", FitFunctionValues[0][0])
            # print ("Chi2Values = ", Chi2Values)


    ## continue only, if file import was successful
    if (ioerror == "false" and countLines > 0):


        ## get id of currend process and write pid to file
        pid = os.getpid()
        f = open(pathTemp + "pid.txt", 'w')
        f.write(str(pid) + "\n")
        f.close()


        ## combine exp_data_x and exp_data_y for plotting
        xmin = 1e99
        xmax = 0
        counter = 1
        for NumFile in range(number_files):                                                 ## loop over all exp. data files
            exp_data = []
            fit_data = []
            diff_data = []
            exp_dataX = []
            i = -1
            for element in exp_data_x[NumFile]:
                i += 1
                LineExp = []
                LineFit = []
                LineDiff = []

                # Debug:
                # print ("NumFile, i = ", NumFile, i)


                fit_data.append(FitFunctionValues[NumFile][i])
                diff_data.append(Chi2Values[NumFile][i])
                exp_data.append(exp_data_y[NumFile][i])
                element0 = element
                exp_dataX.append(element0)
                if (element0 < xmin):
                    xmin = element0
                if (element0 > xmax):
                    xmax = element0


            # Debug:
            # print ('>>>',NumFile)
            # print ('>',exp_dataX)


            ##============================================================================================================================================
            ## create two plots for exp. data file: left plot: exp. data and fit function, right plot: chi^2 function
            if (number_files == 1):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## plot observation data and fit function together in one diagram
                pl = subplot(1,2,1)
                pl.plot(exp_data_x[NumFile], exp_data, 'b.', label = 'data')                ## add exp. data to left panel
                pl.plot(exp_data_x[NumFile], fit_data, 'g-', label = 'fit')                 ## add model function to left panel
                pl.grid(True)                                                               ## add grid to left panel
                pl.legend()                                                                 ## add legend to left panel


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## define label of axis for left panel
                pl.set_xlabel(xlabel)
                pl.set_ylabel(ylabel)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## set y-ticks of left panel
                ymax_exp = max(exp_data)
                ymax_fit = max(fit_data)
                ymax = max(ymax_exp, ymax_fit)                                              ## set max. y-tick
                ymin_exp = min(exp_data)
                ymin_fit = min(fit_data)
                ymin = min(ymin_exp, ymin_fit)                                              ## set min. y-tick
                if (ymin > 0):                                                              ## if min. y-tick > 0 set to 0
                    ymin = 0
                step = (ymax - ymin)/3                                                      ## define only 5 ticks
                yticks = np.arange(ymin, ymax, step)
                # yticks(yticks)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## set x-ticks of left panel
                xmax = max(exp_data_x[NumFile])                                             ## set max. x-tick
                xmin = min(exp_data_x[NumFile])                                             ## set min. x-tick
                step = (xmax - xmin)/5                                                      ## define only 5 ticks
                xticks = np.arange(xmin, xmax, step)
                pl.set_xticks(xticks)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## plot chi^2 as a function of data points to right panel
                prv = subplot(1,2,2)
                pr = prv.twinx()                                                            ## add second y-axis
                pr.set_xlim(xmin, xmax)
                pr.plot(exp_data_x[NumFile], diff_data, 'r', label = 'chi^2')
                pr.grid(True)
                pr.legend()


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## define label of axis for right panel
                prv.set_xlabel(xlabel)
                pr.set_xlabel(xlabel)
                prv.set_yticklabels([])                                                    ## remove tick labels for the left y-axis
                pr.set_ylabel(ylabel)                                                      ## add y-label to the left y-axis


            ##============================================================================================================================================
            ## create two plots for each exp. data files: left plot: exp. data and fit function, right plot: chi^2 function
            elif (number_files > 1):

                # Debug:
                # print ("number_files, counter = ", number_files, counter)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get min and max of current exp. data file
                xmax = max(exp_data_x[NumFile])                                             ## set max. x-tick
                xmin = min(exp_data_x[NumFile])                                             ## set min. x-tick


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## Left panel: construct plot with exp. data and model function
                if (counter == 1):
                    plv = fig.add_subplot(number_files, 2, counter)
                    pl = plv
                else:
                    plv = fig.add_subplot(number_files, 2, counter)
                    pl = plv
                pl.plot(exp_data_x[NumFile], exp_data, 'b.', label = 'data')                ## add exp. data to left panel
                pl.plot(exp_data_x[NumFile], fit_data, 'g-', linewidth = 2, label = 'fit')  ## add model function to left panel
                pl.grid(True)                                                               ## add grid to left panel
                if (NumFile == 0):
                    pl.legend()                                                             ## add legend to the first left panel
                counter += 1                                                                ## increase counter for all (left and right) panels


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## define label of axis of left panel
                if ((NumFile + 1) == int(number_files / 2)):
                    pl.set_ylabel(ylabel)                                                   ## add y-axis label only to middle panel
                else:
                    pl.set_ylabel("")                                                       ## remove tick labels for the left y-axis
                if ((NumFile + 1) == number_files):
                    pl.set_xlabel(xlabel)                                                   ## add x-axis label only to the lowest panel
                else:
                    pl.set_xlabel("")                                                       ## remove tick labels for the left x-axis


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## set x-ticks of left panel
                step = (xmax - xmin)/4                                                      ## define only 5 ticks
                xtic = np.arange(xmin, xmax, step)
                # pl.set_xticks(xtic)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## set y-ticks of left panel
                ymin, ymax = ylim()
                step = (ymax - ymin)/2                                                      ## define only 3 ticks
                ytic = np.arange(ymin, ymax, step)
                # pl.set_yticks(ytic)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## plot chi^2 as a function of data points to right panel
                prv = fig.add_subplot(number_files, 2, counter)
                prv.plot(exp_data_x[NumFile], diff_data, 'r', label = 'chi^2')              ## add chi^2 "function" to right panel
                pr = prv.twinx()                                                            ## add second y-axis
                prv.set_yticklabels([])                                                     ## remove y-ticks of right axis
                prv.grid(True)                                                              ## add grid to right panel
                if (NumFile == 0):
                    prv.legend()                                                            ## add legend to the first left panel
                counter += 1                                                                ## increase counter for all (left and right) panels


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## define label of axis of right panel
                prv.set_ylabel("")                                                          ## remove y-axis label of left axis of right panel
                if (NumFile + 1 == int(number_files / 2)):
                    pr.set_ylabel(ylabel)                                                   ## add y-axis label only to middle panel
                else:
                    pr.set_ylabel("")                                                       ## remove y-axis labels for the right panel
                if ((NumFile + 1) == number_files):
                    pr.set_xlabel(xlabel)
                    prv.set_xlabel(xlabel)
                else:
                    pr.set_xlabel("")                                                       ## remove x-axis labels for the left x-axis
                    prv.set_xlabel("")                                                      ## remove x-axis label of left axis of right panel


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## set x-ticks of right panel
                step = (xmax - xmin)/4                                                      ## define only 5 ticks
                xtic = np.arange(xmin, xmax, step)
                # pr.set_xticks(xtic)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## set y-ticks of right panel
                ymax = max(diff_data)                                                       ## set max. x-tick
                ymin = min(diff_data)                                                       ## set min. x-tick
                step = (ymax - ymin)/2                                                      ## define only 3 ticks
                ytic = np.arange(ymin, ymax, step)
                # pr.set_yticks(ytic)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## draw panels
            fig.canvas.draw_idle()
            # os.system('kill -19 ' + str(pid))                                             ## send sleep command to current process


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## remove lock file
        os.system('rm -rf ' + LockFileName)


    ## define return paramters
    return True
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##========================================================================================================================================================
## Main Program
if __name__ == '__main__':


    ## read in command line parameters
    pathTemp = ""
    number_files = 0
    if len(sys.argv) > 2:


        ## get path of temp directory
        pathTemp = sys.argv[2]

        # Debug:
        # print ("pathTemp = ", pathTemp)

        if not(os.path.exists(pathTemp)):
            print(" ")
            print("\t  Error in plot.py")
            print("\t      The path ",pathTemp)
            print("\t      does not exists!")
            print(" ")
            print("\t      Please correct your input and")
            print("\t      restart the program !")
            print(" ")
            sys.exit(0)


        ## get number of exp-files
        number_files = int(sys.argv[3])

        # Debug:
        # print ("number_files = ", number_files)


        ## get x and y-labels
        xlabel = sys.argv[4]
        ylabel = sys.argv[5]
        zlabel = sys.argv[6]

        # Debug:
        # print ("xlabel = ", xlabel)
        # print ("ylabel = ", ylabel)
        # print ("zlabel = ", zlabel)


    else:
        print(" ")
        print("\t  Error in plot.py")
        print("\t      No command line parameters specified!")
        print(" ")
        sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize figure window
    fig = figure()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start deamon, which updates the matplotlib window ever "TimeIntervalForUpdate" milliseconds.
    ## The function "gobject.timeout_add" calls every "TimeIntervalForUpdate" milliseconds the function "update" and send the parameters "pathTemp",
    ## "number_files", "xlabel", "ylabel", and "zlabel" to the function "update".


    ## get time interval from environment variable
    TimeIntervalForUpdate = int(os.environ.get('MAGIXTimePlotIter',''))

    # Debug:
    # print ("TimeIntervalForUpdate = ", TimeIntervalForUpdate)


    ## start deamon for plotting
    gobject.timeout_add(TimeIntervalForUpdate, update, pathTemp, number_files, xlabel, ylabel, zlabel)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## show window
    show()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## quit program
    sys.exit(0)
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------

