#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module reads and writes the model parameter to a xml-file.
##  Copyright (C) 2009 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - function CheckBool:                           check given string for boolean
##      - function FITSImport.__init__:                 initialize FITSImport class
##      - function FITSImport.LoadFits:                 load contents of FITS-file
##      - function FITSImport.ConvertUnit:              determine unit conversion factor
##      - function FITSImport.DefineFITSAxisValues:     define values for each axis of a given FITS cube
##      - function FITSImport.GetFITSParameter:         get FITS parameter
##      - function RegionHandling.__init__:             initialize RegionHandling class
##      - function RegionHandling.AnalyzeRegionFile:    analyze given region file
##      - function RegionHandling.CheckHourRange:       Checks that the given value is in the range (-24, 24).
##      - function RegionHandling.CheckMinuteRange:     Checks that the given value is in the range [0,60].
##      - function RegionHandling.CheckSecondRange:     Checks that the given value is in the range [0,60].
##      - function RegionHandling.check_hms_ranges:     Checks that the given hour, minute and second are all within reasonable range.
##      - function RegionHandling.t_SIGN:               get sign of value
##      - function RegionHandling.RAinDegree:           converts right ascension in (hour, minutes, second) to decimal degree (sexagesimal system)
##      - function RegionHandling.DECinDegree:          converts declination to decimal degrees
##      - function RegionHandling.AdjustPlot:           adjust boundaries of figure
##      - function RegionHandling.Image2Degree:         transform a given image pixel to degree
##      - function RegionHandling.TransformFrame:       transform given coordinate to fk5
##      - function RegionHandling.DefineRegionMask:     define region mask
##      - function RegionHandling.GetRegionParameters:  get region parameter
##      - function LoadExp.__init__:                    initialize variables of class LoadExp
##      - function LoadExp.GetMinMaxFitsFile:           gets min. and max. frequency of fits file
##      - function LoadExp.xml:                         load xml-file containing settings for the import of experimental-file(s)
##      - function LoadExp.LoadFile:                    call import routines for different file formats
##      - function LoadExp.LoadDat:                     load contents of ascii-file(s)
##      - function LoadExp.LoadFits:                    load contents of fits-file(s)
##      - function LoadExp.WriteFile:                   call export routines for different file formats
##      - function LoadExp.WriteDat:                    write contents to ascii-file(s)
##      - function LoadExp.ExportFITSFile:              write new header for fits file
##      - function LoadExp.WriteFits:                   write contents to fits-file(s)
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    2009-07-14   initial version
##  T. Moeller    2012-01-16   improve documentation of source code
##  T. Moeller    2016-11-22   include cos correction for FITS file import
##  T. Moeller    2019-12-21   apply changes from XCLASS
##  T. Moeller    2020-01-02   porting to python 3, minor improvements
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************

"""
Read Experimental Data:
-----------------------

The package "LoadExpFile" contains subroutines for loading experimental
files. Only experimental data, which are given in the following file
formats, can be loaded by MAGIX:

- dat-files
- fits-files.

Before the experimental file(s) is (are) loaded, the xml-file containing all
settings required for the import of the experimental data has to be loaded.
Especially, the number of experimental files, the path and the name for
each experimental file has to be defined in the xml-file. If no xml-file is
specified and the given filename points directly toward a fits or a
experimental-file the default settings defined below are used and a warning
message is printed to the screen. By using the xml-descriptions, the user can
specify several (!) ranges of the experimental data which are considered in
the fitting process. Thus, there is no need to load all experimental data.

In the following, the expression "x"-column means the columns of an array
defining the "x" position of the experimental data point. For example: For a
function f(x,y,z) the number of "x"-column is 3, for a function f(x1, x2) the
number of "x" columns is 2. Additionally, for dat-files it may be necessary
to specify the number of "y" columns. This means, that for a given
"x"-position in the experimental data, you can specify several "y"-values. For
example: You measured several spectra under different polarization angles at
the same frequencies. A line in the corresponding dat-file may look like:

100.12,  0.34134,  0.12341,  0.78901,  0.13361

Here, the first column describe the frequency and the other columns the
transmission at different polarization angles. The number of "x" columns
is 1 and the number of "y"-columns is 4.

Note, that for fits-files, the number of "y"-columns is always set to 1!

Declaration of ranges: If only one x-column exists, the min and max of the
range is a simple scalar. But if the number of x-columns is greater 1 the
several indicators of the x-columns have to be separated by the "," character.
For example:

    <NumberColumnsX>3</NumberColumnsX>:
            <MinExpRange>0, 0, 0</MinExpRange>
            <MaxExpRange>2000, 100, 20</MaxExpRange>

In addition, for a dat-file import the user can specify the separator
character separating the columns from each other for each file and the number
of columns starting form the left column which belong to the x point of each
experimental data. Furthermore, the user can specify the number of columns
belonging to the f(x) point of each experimental data.

In case of xml-files, the user has to specify the HDU (header Data Unit) which
should be loaded.


Examples:
---------


Structure of xml-file for fits-file import:
-------------------------------------------

<ExpFiles>
    <NumberExpFiles>2</NumberExpFiles>
    <file>
        <FileNamesExpFiles>one_parameter_free/File3.fits</FileNamesExpFiles>
        <ImportFilter>fits</ImportFilter>
        <NumberHDU>0</NumberHDU>
        <NumberExpRanges>1</NumberExpRanges>
        <MinExpRange>0</MinExpRange>
        <MaxExpRange>1000</MaxExpRange>
    </file>
    <file>
        <FileNamesExpFiles>one_parameter_free/File4.fits</FileNamesExpFiles>
        <ImportFilter>fits</ImportFilter>
        <NumberHDU>0</NumberHDU>
        <NumberExpRanges>2</NumberExpRanges>
        <MinExpRange>0</MinExpRange>
        <MaxExpRange>2000</MaxExpRange>
        <MinExpRange>3130</MinExpRange>
        <MaxExpRange>3200</MaxExpRange>
    </file>
</ExpFiles>



Structure of xml-file for dat-file import:
------------------------------------------

<ExpFiles>
    <NumberExpFiles>2</NumberExpFiles>
    <file>
        <FileNamesExpFiles>one_parameter_free/File1.dat</FileNamesExpFiles>
        <ImportFilter>ascii</ImportFilter>
        <NumberHeaderLines>0</NumberHeaderLines>
        <SeparatorColumns> </SeparatorColumns>
        <NumberColumnsX>1</NumberColumnsX>
        <NumberColumnsY>1</NumberColumnsY>
        <ErrorY>yes</ErrorY>
        <NumberExpRanges>1</NumberExpRanges>
        <MinExpRange>0</MinExpRange>
        <MaxExpRange>1000</MaxExpRange>
    </file>
    <file>
        <FileNamesExpFiles>one_parameter_free/File2.dat</FileNamesExpFiles>
        <ImportFilter>ascii</ImportFilter>
        <NumberHeaderLines>0</NumberHeaderLines>
        <SeparatorColumns> </SeparatorColumns>
        <NumberColumnsX>1</NumberColumnsX>
        <NumberColumnsY>1</NumberColumnsY>
        <ErrorY>no</ErrorY>
        <NumberExpRanges>2</NumberExpRanges>
        <MinExpRange>0</MinExpRange>
        <MaxExpRange>2000</MaxExpRange>
        <MinExpRange>3130</MinExpRange>
        <MaxExpRange>3200</MaxExpRange>
    </file>
</ExpFiles>


IMPORTANT:
----------

- The names of the tags <ExpFiles>, <NumberExpFiles> etc. have
  to be written in the same type as presented in the above example.

- Using the tag <ImportFilter> the user can select a specific
  import filter. Setting the import filter to "automatic", the
  format of the experimental file is chosen by the ending of the
  file name. ".fits" for fits-file, ".dat" for ASCII files.

- Please, do not mix upper and lower case in the name of the tags!!

- The tag <file></file> must not occur more or less than the
  number of experimental files defined in the tag
  <NumberExpFiles></NumberExpFiles> !!!!!

- The number of ranges <NumberExpRanges></NumberExpRanges>
  must always be given! If no range desired (all data should
  be included in the fit process), the number of ranges must
  be set to 0.

- If the number of ranges is set to 0 the tags
  <MinExpRange></MinExpRange> and
  <MaxExpRange></MaxExpRange> need not to be set.
  BUT: All other tags have to be set !!!!

- If the user do not want to consider all data (number of ranges
  set to > 1) the tags <MinExpRange></MinExpRange> and
  <MaxExpRange></MaxExpRange> have to occur as many times as
  given by <NumberExpRanges></NumberExpRanges>

- If the number of "x"-columns is greater than 1, the min and max of
  the "x"-columns of the ranges have to be separated by the "," character.
  For example:

    <NumberColumnsX>1</NumberColumnsX>:
            <MinExpRange>0</MinExpRange>
            <MaxExpRange>2000</MaxExpRange>

    <NumberColumnsX>3</NumberColumnsX>:
            <MinExpRange>0, 0, 0</MinExpRange>
            <MaxExpRange>2000, 100, 20</MaxExpRange>

  Although the <NumberColumnsX></NumberColumnsX> tag is ignored in the
  import of fits-file, the content of this tag is defined by the dimension
  of the fits-file. Thus, the settings for the ranges have to be given
  as described above, if the dimension of the experimental file is larger
  than 1.

- Note, that the xml-file differ by some tags for dat and fits
  file import !!

- (fits-file): For the import of fits-file the user has to specify the
  Header Data Unit (HDU) for each fits file.

- (dat-file): The y-column(s) have to be close to the x-column(s).

  Example:    |<- x-columns ->|<- y-column ->|
                212.124  23.314    1210.643216

- (dat-file): The tag <NumberHeaderLines> defines the number of
  header lines which should be ignored in the import of the
  dat-files.

- (dat-file): The tag <SeparatorColumns> defines the character
  which separate the columns from each other.

- (dat-file): The tag <NumberColumnsX> defines the number of
  columns counted from the left side, which belongs to the
  "x"-points of the experimental data.
  For example: If the user wants to import a dat-file
               containing 3D data, the first 3 columns define
               the x, y, and z position. For that kind of data
               the tag <NumberColumnsX> has to be set to 3.

- (dat-file): The tag <NumberColumnsY> defines the number of
  columns, which belongs to the "y"-points of the experimental
  data. These "y"-columns have to be close to the "x" values!
  For example: If the user wants to import a dat-file which
               contains at every given "x"-point 4 different
               values of the measured function (i.e. that there
               are 4 different dat-files in one dat-file) the
               tag <NumberColumnsY> has to be set to 4.

  Example:    |<- x-columns ->|<- y-column ->|
                212.124  23.314    1210.643216

  Here, <NumberColumnsY> has to be set to 1.

- (dat-file): If the Error tag <ErrorY>yes</ErrorY> is set to
  yes the columns containing the errors have to be close to the
  y-columns. Additionally, the number of these "error" columns
  have to equal to the number of "y"-columns given in the tag
  <NumberColumnsY></NumberColumnsY>

- An example structure of a dat-file is given below:

  Structure of a sample dat-file:
  -------------------------------
                                                                        ONLY RELEVANT IF ErrorY = "YES"
  1           ...    NumberColumnsX   1         ...    NumberColumnsY   1         ...    NumberColumnsY
  100.2313           20.6578          0.5846           1.4218           0.02             0.01
  102.2463           21.7548          0.5947           1.5432           0.03             0.09
  .                  .                .                .                .                .
  .                  .                .                .                .                .
  .                  .                .                .                .                .
"""


##******************************************************************** load packages *********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy                                                                                ## load numpy package
import copy                                                                                 ## load copy package
import os                                                                                   ## load os package
import os.path                                                                              ## load package for file management
import sys                                                                                  ## load sys package
import time                                                                                 ## load time package
import math                                                                                 ## load math package
from xclass.addons.MAGIX.Modules.python import XMLPackage
if (not 'matplotlib' in sys.modules):
    import matplotlib                                                                       ## import matplotlib package
    matplotlib.use("Agg")                                                                   ## avoid display error
import matplotlib.pyplot as plt                                                             ## import pyplot package


## import pyfits package via astropy
try:
    import astropy.io.fits as pyfits
except:
    import pyfits
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check given string for boolean
##
def CheckBool(OptionValue):
    """

    input parameters:
    -----------------

        - OptionValue:          string to analyze


    output parameters:
    ------------------

        - OutFlag:              boolean value
    """

    # Debug:
    # print ("OptionValue = ", OptionValue)


    ## initialize return parameter
    OutFlag = False


    ## define list of known "true" phrases
    ListOfKnownTruePhrases = ['1', 't', 'true', 'y', 'yes', 'yeah']


    ## define return parameter
    OptionValue = str(OptionValue).strip()
    OptionValue = OptionValue.lower()
    if (OptionValue in ListOfKnownTruePhrases):
        OutFlag = True


    ## we're done
    return OutFlag
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define class for importing FITS files
##
class FITSImport(object):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## initialize FITSImport class
    ##
    def __init__(self, FITSFileName, HDU = 0, ConvertFreqAxisFlag = True, printFlag = True, InternalParameterList = []):
        """
    input parameters:
    -----------------

        - FITSFileName:             path and name of FITS file

        - HDU:                      (optional) HDU, (default: 0)

        - ConvertFreqAxisFlag:      (optional) convert frequency axis to MHz

        - printFlag:                (optional) flag for screen output, (default: True)

        - InternalParameterList:    (optional) list of internal parameters, (default: [])


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("FITSFileName = ", FITSFileName)
        # print ("HDU = ", HDU)
        # print ("ConvertFreqAxisFlag = ", ConvertFreqAxisFlag)
        # print ("printFlag = ", printFlag)
        # print ("InternalParameterList = ", InternalParameterList)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## test, if astropy is available
        self.astropyFlag = True
        try:
            import astropy.wcs as wcs
        except:
            self.astropyFlag = False


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## save input parameters
        self.FITSFileName = FITSFileName
        self.HDU = HDU
        self.ConvertFreqAxisFlag = ConvertFreqAxisFlag
        self.printFlag = printFlag
        self.InternalParameterList = InternalParameterList


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## initialize some internal parameters
        self.FITSParameters = {}
        self.NumDimensions = 1
        self.NAXIS = 1
        self.lengthAxis = 1
        self.FITSDataDimension = 1
        self.CTYPEList = []
        self.CRPIXList = []
        self.CRVALList = []
        self.CDELTList = []
        self.CROTList = []
        self.CUNITList = []
        self.ValueOfFirstPointList = []
        self.BSCALE = 1.0
        self.BZERO = 0.0
        self.BUNIT = ""
        self.RESTFRQ = 0.0
        self.AxisValuesList = []


        ## define some constants
        self.cms = 299792458.0                                                              ## speed of light in m/s
        self.ckms = self.cms * 1.e-3                                                        ## speed of light in km/s


        ## define CTYPE indicating frequency axis
        self.CTYPEFreq = ["freq"]
        self.CTYPEVel = ["velo", "vrad", "vopt"]
        self.CTYPEWave = ["wave"]
        self.FrequencyAxisCTYPE = self.CTYPEFreq + self.CTYPEVel + self.CTYPEWave


        ## define lists of unit keywords
        self.KeywordListHz = ["hz", "hertz"]
        self.KeywordListkHz = ["khz", "kilohertz"]
        self.KeywordListMHz = ["mhz", "megahertz"]
        self.KeywordListGHz = ["ghz", "gigahertz"]
        self.KeywordListTHz = ["thz", "terahertz"]
        self.KeywordListWaveNumber = ["cm1", "cm-1", "cm^-1", "cm^(-1)", "cm^{-1}", "cm$^{-1}$", "1/cm"]
        self.KeywordListJoule = ["j", "joule"]
        self.KeywordListKelvin = ["k", "kelvin"]
        self.KeywordListeV = ["ev", "elektronvolt"]
        self.KeywordListVelkms = ["kms", "km/s", "kms-1", "km s-1", "kms1", "km s1", "kms^-1", "km s^-1", "kms^(-1)", "km s^(-1)", "kms$^{-1}$", \
                                  "km s$^{-1}$", "kms^{-1}", "km s^{-1}"]
        self.KeywordListVelms = ["ms", "m/s", "ms-1", "m s-1", "ms1", "m s1", "ms^-1", "m s^-1", "ms^(-1)", "m s^(-1)", "ms$^{-1}$", "m s$^{-1}$", \
                                 "ms^{-1}", "m s^{-1}"]
        self.KeywordListwavelengthm = ["m", "meter"]
        self.KeywordListwavelengthcm = ["cm", "centimeter"]
        self.KeywordListwavelengthmm = ["mm", "millimeter"]
        self.KeywordListwavelengthmum = ["mum", "mikrometer"]
        self.KeywordListwavelengthnm = ["nm", "naometer"]
        self.KeywordListwavelengtha = ["a", "angstrom"]
        self.KeywordListTotal = self.KeywordListHz + self.KeywordListkHz + self.KeywordListMHz + self.KeywordListGHz + self.KeywordListTHz \
                                + self.KeywordListWaveNumber + self.KeywordListJoule + self.KeywordListKelvin + self.KeywordListeV \
                                + self.KeywordListVelkms + self.KeywordListVelms + self.KeywordListwavelengthm + self.KeywordListwavelengthcm \
                                + self.KeywordListwavelengthmm + self.KeywordListwavelengthmum + self.KeywordListwavelengthnm \
                                + self.KeywordListwavelengtha


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## read in FITS file
        self.LoadFits()


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## determine elements for each axis
        self.DefineFITSAxisValues()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## load contents of FITS-file
    ##
    def LoadFits(self):
        """


    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None


    remark:
    -------

        CLASS understands RA--, RA ; DEC-, DEC ; GLON ; GLAT ; TIME, UT.
        For the projection system, CLASS understands
            : P_NONE      = 0 ! Unprojected data
        -TAN: P_GNOMONIC  = 1 ! Radial Tangent plane
        -SIN: P_ORTHO     = 2 ! Dixon Tangent plane
        -ARC: P_AZIMUTHAL = 3 ! Schmidt Tangent plane
        -STG: P_STEREO    = 4 ! Stereographic
        lamb: P_LAMBERT   = 5 ! Lambert equal area
        -ATF: P_AITOFF    = 6 ! Aitoff equal area
        -GLS: P_RADIO     = 7 ! Classic Single dish radio mapping

        Read Representations of celestial coordinates in FITS
        Authors: Mark R. Calabretta, Eric W. Greisen
        (Submitted on 19 Jul 2002)
        arXiv:astro-ph/0207413v1
        http://arxiv.org/abs/astro-ph/0207413
        """


        ## print what you do
        if (self.printFlag):
            print ("\n\t Reading FITS file " + chr(34) + self.FITSFileName + chr(34) + " ..", end = ' ', flush = True)


        ## open fits-file using pyfits
        hdulist = pyfits.open(self.FITSFileName)

        # Debug:
        # print ("\t HDU = ",HDU)
        # print (hdulist.info())


        ## save data
        ## number of columns in scidata is given by len(scidata)
        fitsdata = hdulist[self.HDU].data


        ## verify header, fix if necessary and save header
        # hdulist.verify('fix')
        fitshead = hdulist[self.HDU].header


        ## set array containing dimension of each axis
        self.FITSDataDimension = fitsdata.shape
        self.NumDimensions = len(self.FITSDataDimension)


        ## get wcs object from astropy
        if (self.astropyFlag):
            import astropy.wcs as wcs
            self.w = wcs.WCS(hdulist[self.HDU].header, naxis = self.NumDimensions)          ## definition of naxis is a bug fix to avoid ValueError:
                                                                                            ## When providing two arguments, the array must be of
                                                                                            ## shape (N, ..)
            # Debug:
            # self.w.wcs.print_contents()
            # print (self.w.wcs.restfrq)
            # print (self.w.wcs.naxis)
            # print (self.w.wcs.cunit)
            # print (self.w.wcs.ctype)
            # print ("self.NumDimensions = ", self.NumDimensions)


        ## check dimension of fits file
        if (self.NumDimensions > 4):
            print ("\n\nError in XCLASS package, subroutine LoadFits:")
            print ("\t\tCan not import fits-file with more than 4 dimensions.")
            print ("\t\tPlease select other file and restart program.")
            print ("\n\t\tProgram aborted")
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read header information
        ## get number of axes
        try:
            self.NAXIS = pyfits.getval(self.FITSFileName, 'NAXIS', 0)
        except:
            self.NAXIS = 1

        # Debug:
        # print ("\n\nself.NAXIS = ", self.NAXIS)


        ## get scale parameter BSCALE
        try:
            self.BSCALE = pyfits.getval(self.FITSFileName, 'BSCALE', 0)
        except:
            self.BSCALE = 1

        # Debug:
        # print ("\n\nself.BSCALE = ", self.BSCALE)


        ## get zero parameter BZERO
        try:
            self.BZERO = pyfits.getval(self.FITSFileName, 'BZERO', 0)
        except:
            self.BZERO = 0.0

        # Debug:
        # print ("\n\nself.BZERO = ", self.BZERO)


        ## get unit parameter BUNIT
        try:
            self.BUNIT = pyfits.getval(self.FITSFileName, 'BUNIT', 0)
        except:
            self.BUNIT = ""

        # Debug:
        # print ("\n\nself.BUNIT = ", self.BUNIT)


        ## get rest frequency RESTFRQ
        try:
            self.RESTFRQ = pyfits.getval(self.FITSFileName, 'RESTFRQ', 0)
        except:
            try:
                self.RESTFRQ = pyfits.getval(self.FITSFileName, 'RESTFREQ', 0)
            except:
                self.RESTFRQ = 0.0
        if (abs(self.RESTFRQ) >= 1.e10):
            self.RESTFRQ = self.RESTFRQ * 1.e-6

        # Debug:
        # print ("\n\nself.RESTFRQ = ", self.RESTFRQ)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze header
        self.lengthAxis = []
        self.NumDimensions = 0
        self.FreqAxisID = 2
        for i in range(self.NAXIS):                                                         ## loop over all dimensions
            try:
                lenAxis = pyfits.getval(self.FITSFileName, 'NAXIS' + str(i + 1), 0)         ## read CRPIX
            except:
                lenAxis = 1
            if (lenAxis > 0):                                                               ## read only those entries with more than 1 point
                self.lengthAxis.append(lenAxis)
                self.NumDimensions += 1


                ## get header parameters
                try:
                    CTYPE = pyfits.getval(self.FITSFileName, 'CTYPE' + str(i + 1), 0)       ## indicate the unit of the current axis
                except:
                    CTYPE = ""
                self.CTYPEList.append(CTYPE.strip())
                try:
                    CRPIX = pyfits.getval(self.FITSFileName, 'CRPIX' + str(i + 1), 0)       ## location of a reference point along axis 1
                except:
                    CRPIX = 1
                self.CRPIXList.append(CRPIX)
                try:
                    CRVAL = pyfits.getval(self.FITSFileName, 'CRVAL' + str(i + 1), 0)       ## the value of the coordinate at the reference point CRPIX1
                except:
                    CRVAL = 0
                self.CRVALList.append(CRVAL)
                try:
                    CDELT = pyfits.getval(self.FITSFileName, 'CDELT' + str(i + 1), 0)       ## partial derivative of the coordinate with respect to the
                except:                                                       ## pixel index, evaluated at the reference point CRPIX
                    CDELT = 1
                self.CDELTList.append(CDELT)
                try:
                    CROT = pyfits.getval(self.FITSFileName, 'CROT' + str(i + 1), 0)         ## indicate a rotation from a standard coordinate system
                except:
                    CROT = 0
                self.CROTList.append(CROT)
                try:
                    CUNIT = pyfits.getval(self.FITSFileName, 'CUNIT' + str(i + 1), 0)       ## indicate the unit of the current axis
                except:
                    CUNIT = ""
                self.CUNITList.append(CUNIT)


                ## identify axis describing frequency or velocity
                LocalUNIT = CUNIT.strip()
                LocalUNIT = LocalUNIT.lower()
                if (LocalUNIT in self.KeywordListTotal):
                    self.FreqAxisID = i


                ## calculate value at first pixel
                ValueOfFirstPoint = CRVAL - ((CRPIX - 1) * CDELT)
                self.ValueOfFirstPointList.append(ValueOfFirstPoint)

                # Debug:
                # print ("\n\nCTYPE = ", CTYPE)
                # print ("CRPIX = ", CRPIX)
                # print ("CRVAL = ", CRVAL)
                # print ("CDELT = ", CDELT)
                # print ("CROT = ", CROT)
                # print ("CUNIT = ", CUNIT)
                # print ("ValueOfFirstPoint = ", ValueOfFirstPoint)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add cos correction for R.A.
        if (not self.astropyFlag):
            for AxisID, LocalCTYPE in enumerate(self.CTYPEList):
                LocalCTYPE = LocalCTYPE.strip()
                LocalCTYPE = LocalCTYPE.lower()
                if (LocalCTYPE.startswith("ra--") and self.CRPIXList[AxisID] != 1):
                    CRVAL = self.CRVALList[AxisID]
                    CRPIX = self.CRPIXList[AxisID]
                    CDELT = self.CDELTList[AxisID]
                    DECRefVal = 0.0
                    for AxisIDDEC, LocalCTYPEDEC in enumerate(LocalCTYPE):
                        LocalCTYPEDEC = LocalCTYPEDEC.strip()
                        LocalCTYPEDEC = LocalCTYPEDEC.lower()
                        if (LocalCTYPEDEC.startswith("dec-")):
                            DECRefVal = self.CRVALList[AxisIDDEC]
                            break
                    CDELT = CDELT / math.cos(math.pi / 180.0 * DECRefVal)                   ## add cos correction to delta of R.A.
                    self.CDELTList[AxisID] = CDELT
                    ValueOfFirstPoint = CRVAL - ((CRPIX - 1) * CDELT)                       ## calculate new (corrected) first point
                    self.ValueOfFirstPointList[AxisID] = ValueOfFirstPoint

        # Debug:
        # print ("\n\nself.ValueOfFirstPointList = ", self.ValueOfFirstPointList)


        ## close fits-file
        hdulist.close()


        ## everything is fine
        if (self.printFlag):
            print ("done!")
            print ("\t\t Dimensions = ", self.FITSDataDimension)
            print ("\n", flush = True)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define dictionary containing FITS parameter
        self.FITSParameters['FITSFileName'] = self.FITSFileName
        self.FITSParameters['HDU'] = self.HDU
        self.FITSParameters['NumDimensions'] = self.NumDimensions
        self.FITSParameters['CTYPE'] = self.CTYPEList
        self.FITSParameters['CRPIX'] = self.CRPIXList
        self.FITSParameters['CRVAL'] = self.CRVALList
        self.FITSParameters['CDELT'] = self.CDELTList
        self.FITSParameters['CROT'] = self.CROTList
        self.FITSParameters['CUNIT'] = self.CUNITList
        self.FITSParameters['BSCALE'] = self.BSCALE
        self.FITSParameters['BZERO'] = self.BZERO
        self.FITSParameters['BUNIT'] = self.BUNIT
        self.FITSParameters['RESTFRQ'] = self.RESTFRQ
        self.FITSParameters['ValueOfFirstPoint'] = self.ValueOfFirstPointList
        self.FITSParameters['header'] = fitshead
        self.FITSParameters['FreqAxisID'] = self.FreqAxisID
        self.FITSParameters['data'] = fitsdata
        self.FITSParameters['data.shape'] = fitsdata.shape
        if (self.lengthAxis != self.FITSDataDimension):
            self.FITSParameters['ReverseIndexing'] = True
        else:
            self.FITSParameters['ReverseIndexing'] = False
        if (len(self.lengthAxis) < 2):
            self.lengthAxis.append(1)
        self.FITSParameters['lengthAxis'] = self.lengthAxis

        # Debug:
        # print ("self.FITSParameters = ", self.FITSParameters)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## determine unit conversion factor
    ##
    def ConvertUnit(self, unit, LocalCTYPE):
        """


    input parameters:
    -----------------

        - unit:                     given unit string

        - LocalCTYPE:               ctype of current axis


    output parameters:
    ------------------

        - UnitConversionFactor:     conversion factor
        """

        # Debug:
        # print ("unit = ", unit)
        # print ("LocalCTYPE = ", LocalCTYPE)


        ## initialize return parameter
        UnitConversionFactor = 1.0


        ## remove leading and tailing strings and convert to lower case
        unit = unit.strip()
        unit = unit.lower()
        LocalCTYPE = LocalCTYPE.strip()
        LocalCTYPE = LocalCTYPE.lower()


        ## define what happens if no unit is given and / or no ctype is specified
        if ((unit == "" or self.astropyFlag) and LocalCTYPE in self.CTYPEFreq):             ## if no unit is specified and axis describes frequency
            unit = "hz"                                                                     ## assume unit of "Hz"
                                                                                            ## Note, if astropy is available, we use the all_pix2world
                                                                                            ## function which returns frequency axis always (?!?) in Hz
        elif (unit == "" and LocalCTYPE in self.CTYPEVel):                                  ## if no unit is specified and axis describes velocity
            unit = "m/s"                                                                    ## assume unit of "m/s"
        elif (unit == "" and LocalCTYPE in self.CTYPEWave):                                 ## if no unit is specified and axis describes wavelength
            unit = "mm"                                                                     ## assume unit of "mm"
        elif (unit == "" and LocalCTYPE == ""):                                             ## if no unit and no ctype is specified
            unit = "hz"                                                                     ## assume unit of "Hz"


        ## determine conversion factor
        if (unit in self.KeywordListHz):                                                    ## frequency in Hz
            UnitConversionFactor = 1.e-6
        elif (unit in self.KeywordListkHz):                                                 ## frequency in kHz
            UnitConversionFactor = 1.e-3
        elif (unit in self.KeywordListMHz):                                                 ## frequency in MHz
            UnitConversionFactor = 1.0
        elif (unit in self.KeywordListGHz):                                                 ## frequency in GHz
            UnitConversionFactor = 1.e+3
        elif (unit in self.KeywordListTHz):                                                 ## frequency in THz
            UnitConversionFactor = 1.e+6
        elif (unit in self.KeywordListWaveNumber):                                          ## wave number
            UnitConversionFactor = 1.0 / 3.33565e-11 * 1e-6                                 ## from cm-1 to Hz to MHz
        elif (unit in self.KeywordListKelvin):                                              ## convert Kelvin to MHz
            UnitConversionFactor = 2.08364e+10
        elif (unit in self.KeywordListJoule):                                               ## convert Joule to MHz
            UnitConversionFactor = 1.50930e+33
        elif (unit in self.KeywordListeV):                                                  ## convert eV to MHz
            UnitConversionFactor = 2.41804e+14
        elif (unit in self.KeywordListwavelengthm and LocalCTYPE in self.CTYPEWave):        ## wavelength in m
            UnitConversionFactor = "wavelength m"
        elif (unit in self.KeywordListwavelengthcm and LocalCTYPE in self.CTYPEWave):       ## wavelength in cm
            UnitConversionFactor = "wavelength cm"
        elif (unit in self.KeywordListwavelengthmm and LocalCTYPE in self.CTYPEWave):       ## wavelength in mm
            UnitConversionFactor = "wavelength mm"
        elif (unit in self.KeywordListwavelengthmum and LocalCTYPE in self.CTYPEWave):      ## wavelength in mum
            UnitConversionFactor = "wavelength mum"
        elif (unit in self.KeywordListwavelengthnm and LocalCTYPE in self.CTYPEWave):       ## wavelength in nm
            UnitConversionFactor = "wavelength nm"
        elif (unit in self.KeywordListwavelengtha and LocalCTYPE in self.CTYPEWave):        ## wavelength in Angstrom
            UnitConversionFactor = "wavelength angstrom"
        elif (unit in self.KeywordListVelkms and LocalCTYPE in self.CTYPEVel):              ## velocity in km/s
            UnitConversionFactor = "velocityKMS"
        elif (unit in self.KeywordListVelms or LocalCTYPE in self.CTYPEVel):                ## velocity in m/s
            UnitConversionFactor = "velocityMS"
        else:                                                                               ## define default conversion factor
            UnitConversionFactor = 1.e-6

        # Debug:
        # print ("UnitConversionFactor = ", UnitConversionFactor)
        # if (self.astropyFlag and LocalCTYPE in self.CTYPEFreq):
        #   LocalAxisElement = LocalAxisElement * 1.e-6


        ## define return variable
        return UnitConversionFactor
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## define values for each axis of a given FITS cube
    ##
    def DefineFITSAxisValues(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
    """


        ## initialize return parameters
        cms = 299792458.0                                                                   ## speed of light in m/s
        ckms = cms * 1.e-3                                                                  ## speed of light in km/s


        ## standard pixel
        StandardPixel = [ [0.0], [0.0, 0.0], [0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0] ]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create coordinate map using astropy
        ## taken from https://www.mubdirahman.com/assets/lecture-7---astronomical-specifics.pdf
        ralist = []
        declist = []
        freqlist = []
        stokelist = []
        #    if (self.astropyFlag):                                                              ## continue only, if astropy package is available


        #        ## get RA, Dec values
        #        xpx = numpy.arange(self.FITSDataDimension[1] + 1) - 0.5                         ## Making list of bin edges. Remember there are N + 1 bins.
        #        ypx = numpy.arange(self.FITSDataDimension[0] + 1) - 0.5                         ## The coord. are defined on the centre of the pixel, so
        #                                                                                        ## the first bin edge is at -0.5.
        #        if (self.NumDimensions == 2):                                                   ## we are dealing with an image
        #            xlist, ylist = numpy.meshgrid(xpx, ypx)                                     ## numpy.meshgrid() creates two, 2-D arrays filled with
        #                                                                                        ## the values in the two 1-D arrays you give it
        #            ralist, declist = self.w.all_pix2world(xlist, ylist, 0)                     ## Converting the indices into RA, Dec values for all
        #                                                                                        ## values in the lists.
        #                                                                                        ## Remember that RA traditionally increases to the left,
        #                                                                                        ## so you’ll have to flip the axis manually
        #        elif (self.NumDimensions == 3):                                                 ## we are dealing with an 3d cube
        #            zpx = numpy.arange(self.FITSDataDimension[2] + 1) - 0.5                     ## Making list of bin edges.
        #            xlist, ylist, zlist = numpy.meshgrid(xpx, ypx, zpx)                         ## numpy.meshgrid() creates three, 2-D arrays filled with
        #                                                                                        ## the values in the two 1-D arrays you give it
        #            ralist, declist, freqlist = self.w.all_pix2world(xlist, ylist, zlist, 0)    ## Converting the indices into RA, Dec values for all
        #                                                                                        ## values in the lists.
        #                                                                                        ## Remember that RA traditionally increases to the left,
        #                                                                                        ## so you’ll have to flip the axis manually
        #        elif (self.NumDimensions == 4):                                                 ## we are dealing with an 3d cube
        #            tpx = numpy.arange(self.FITSDataDimension[3] + 1) - 0.5                     ## Making list of bin edges.
        #            zpx = numpy.arange(self.FITSDataDimension[2] + 1) - 0.5                     ## Making list of bin edges.
        #            xlist, ylist, zlist, tlist = numpy.meshgrid(xpx, ypx, zpx, tpx)             ## numpy.meshgrid() creates four, 2-D arrays filled with
        #                                                                                        ## the values in the two 1-D arrays you give it
        #            ralist, declist, freqlist, stokelist = self.w.all_pix2world(xlist, ylist, zlist, tlist, 0)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define elements of each axis
        for LocalDimID in range(self.NumDimensions):


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## convert pixel to world coordinate
            if (ralist == [] or declist == [] or (LocalDimID > 1)):                         ## if astropy package is available use previous results
                LocalList = []


                ## get type of axis
                LocalCTYPE = self.CTYPEList[LocalDimID].lower()
                LocalCTYPE = LocalCTYPE.split("-")
                LocalCTYPE = LocalCTYPE[0].strip()

                # Debug:
                # print ("\n\nLocalCTYPE = ", LocalCTYPE)
                # print ("self.CUNITList[LocalDimID] = ", self.CUNITList[LocalDimID])
                # print ("self.NumDimensions = ", self.NumDimensions)
                # print ("LocalDimID, self.lengthAxis[LocalDimID] = ", LocalDimID, self.lengthAxis[LocalDimID])


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get unit conversion factor
                UnitConversionFactor = 1.0
                if (LocalCTYPE in self.FrequencyAxisCTYPE or LocalCTYPE == ""):
                    FrequencyFlag = True
                    UnitConversionFactor = self.ConvertUnit(self.CUNITList[LocalDimID], LocalCTYPE)

                    # Debug:
                    # print ("UnitConversionFactor = ", UnitConversionFactor


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get "world" coordinates for each axis
                FrequencyFlag = False
                for AxisElementID in range(self.lengthAxis[LocalDimID]):                    ## loop over all frequencies


                    ## use astropy
                    if (self.astropyFlag):                                                  ## if astropy is available, use astropy
                        CurrentPixel = StandardPixel[self.NumDimensions - 1]
                        CurrentPixel[LocalDimID] = AxisElementID + 1
                        pixcrd = numpy.array([CurrentPixel], numpy.float_)
                        world = self.w.all_pix2world(pixcrd, 1)
                        world = world[0]
                        LocalAxisElement = world[LocalDimID]

                        # Debug:
                        # print ("\nLocalDimID = ", LocalDimID)
                        # print ("world = ", world)
                        # print ("LocalAxisElement = ", LocalAxisElement)


                        ## compare with self-written expression
                        #    LocalAxisElementNormal = (self.ValueOfFirstPointList[LocalDimID] + (AxisElementID * self.CDELTList[LocalDimID])) #\
                        #                               #* self.BSCALE + self.BZERO
                        # print ("LocalAxisElement, LocalAxisElementNormal, (LocalAxisElementNormal-LocalAxisElement) = {0:.8f}, {1:.8f}, {2:.8e}" \
                        #           .format(LocalAxisElement, LocalAxisElementNormal, (LocalAxisElementNormal - LocalAxisElement)))


                    ## do not use astropy
                    else:
                        LocalAxisElement = (self.ValueOfFirstPointList[LocalDimID] + (AxisElementID * self.CDELTList[LocalDimID])) * self.BSCALE \
                                            + self.BZERO


                    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                    ## check if units have to be converted
                    if (LocalCTYPE in self.FrequencyAxisCTYPE):
                        FrequencyFlag = True


                        ## convert frequency axis to MHz?
                        if (self.ConvertFreqAxisFlag):


                            ## convert velocities to MHz
                            if (UnitConversionFactor == "velocityKMS"):
                                LocalAxisElement = float(self.RESTFRQ) / ckms * (ckms - LocalAxisElement)
                            elif (UnitConversionFactor == "velocityMS"):
                                LocalAxisElement = float(self.RESTFRQ) / cms * (cms - LocalAxisElement)


                            ## convert wavelengths to MHz
                            elif (UnitConversionFactor == "wavelength m"):
                                LocalAxisElement = max(1.e-20, LocalAxisElement)
                                LocalAxisElement = cms / LocalAxisElement * 1.e-6
                            elif (UnitConversionFactor == "wavelength cm"):
                                LocalAxisElement = max(1.e-20, LocalAxisElement)
                                LocalAxisElement = cms / LocalAxisElement * 1.e-6 * 1.e2
                            elif (UnitConversionFactor == "wavelength mm"):
                                LocalAxisElement = max(1.e-20, LocalAxisElement)
                                LocalAxisElement = cms / LocalAxisElement * 1.e-6 * 1.e3
                            elif (UnitConversionFactor == "wavelength mum"):
                                LocalAxisElement = max(1.e-20, LocalAxisElement)
                                LocalAxisElement = cms / LocalAxisElement * 1.e-6 * 1.e6
                            elif (UnitConversionFactor == "wavelength nm"):
                                LocalAxisElement = max(1.e-20, LocalAxisElement)
                                LocalAxisElement = cms / LocalAxisElement * 1.e-6 * 1.e9
                            elif (UnitConversionFactor == "wavelength angstrom"):
                                LocalAxisElement = max(1.e-20, LocalAxisElement)
                                LocalAxisElement = cms / LocalAxisElement * 1.e-6 * 1.e10


                            ## convert other units to MHz
                            else:
                                LocalAxisElement = LocalAxisElement * UnitConversionFactor


                    ##----------------------------------------------------------------------------------------------------------------------------------------
                    ## store axis element
                    LocalList.append(LocalAxisElement)


                ## convert list to numpy array and store world coordinates
                if (FrequencyFlag):
                    LocalList = numpy.asarray(LocalList, dtype = numpy.float64)
                else:
                    LocalList = numpy.asarray(LocalList, dtype = numpy.float64)
                self.AxisValuesList.append(LocalList)

                # Debug:
                # print ("\n\nLocalDimID = ", LocalDimID)
                # print ("LocalList = ", LocalList)


            ## use previous results
            else:
                if (LocalDimID == 0):
                    self.AxisValuesList.append(ralist)
                else:
                    self.AxisValuesList.append(declist)


        ## special handling for lower-dimensional data (non-cubes)
        if (len(self.AxisValuesList) < 3):
            self.AxisValuesList.append([0.0])


        ## check, if order of axis has to be changed to be consistent with FITS data
        #    if (self.lengthAxis != self.FITSDataDimension):
        #        AxisValuesList_copy = copy.deepcopy(self.AxisValuesList)
        #        self.AxisValuesList = []
        #        for LocalDimID in range(self.NumDimensions):
        #            self.AxisValuesList.append(AxisValuesList_copy[self.NumDimensions - LocalDimID - 1])

        # Debug:
        # print ("\n\nself.AxisValuesList = ", self.AxisValuesList)


        ## store list of axis elements
        self.FITSParameters['AxisValuesList'] = self.AxisValuesList


        # we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## get FITS parameter
    ##
    def GetFITSParameters(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - self.FITSParameters:              FITS parameters
    """


        ## define return parameters
        return self.FITSParameters
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define class for importing and analyzing region files (requires astropy package (version 2.0.8) for full functionality)
##
##    Collection of subroutines and functions to convert sky coordinates to pixel
##    Partly taken from the astropy project!
##    https://github.com/astropy/astropy/blob/master/astropy/coordinates/angle_utilities.py
##
##
##
##
##      TO DO:      - one pixel rotation between fk4 and fk5 / icrs for box regions only !!!!
##      ------
##                  - coordinates in galactic frame have to be given in degree, but box and elliptic regions are rotated by -45°
##
##
class RegionHandling(object):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize RegionHandling class
    def __init__(self, RegionFileName, AllFITSParameters, dbDir = "", AdjustBoundFlag = False, ApplyRACorrection = True):
        """


    input parameters:
    -----------------

        - RegionFileName:           path and name of region file

        - AllFITSParameters:        list of all FITS parameters

        - dbDir:                    (optional) path for debugging, (default: "")

        - AdjustBoundFlag:          (optional) defines, if matplotlib boundaries have to be adjusted, (default: False)

        - ApplyRACorrection:        (optional) apply correction of right ascension, (default: True)


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("RegionFileName = ", RegionFileName)
        # print ("AllFITSParameters = ", AllFITSParameters)
        # print ("dbDir = ", dbDir)
        # print ("AdjustBoundFlag = ", AdjustBoundFlag)
        # print ("ApplyRACorrection = ", ApplyRACorrection)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize return parameter
        self.RegionParameters = {}
        self.RegionPixel = []
        self.RealLength = []
        self.TotalNumberPixels = 0
        self.CentralPixelOfRegionX = None
        self.CentralPixelOfRegionIndexX = None
        self.CentralPixelOfRegionY = None
        self.CentralPixelOfRegionIndexY = None


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store input parameters


        ## store path and name of region file
        self.RegionFileName = RegionFileName.strip()
        self.RegionFileParameters = []


        ## store path for debugging and get adjust boundaries flag
        self.dbDir = dbDir.strip()
        self.AdjustBoundFlag = AdjustBoundFlag


        ## store flag for correction
        self.ApplyRACorrection = ApplyRACorrection


        ## get some parameters for first FITS file
        FITSParameters = AllFITSParameters[0]
        self.length = FITSParameters['lengthAxis']                                          ## number of pixels for each direction
        self.ValueOfFirstPointList = FITSParameters['ValueOfFirstPoint']                    ## list of first pixel for each direction
        self.CDELTList = FITSParameters['CDELT']                                            ## step size for each direction
        self.CRVALList = FITSParameters['CRVAL']                                            ## value of reference pixel
        self.CRPIXList = FITSParameters['CRPIX']                                            ## number of reference pixel
        self.AxisValuesList = FITSParameters['AxisValuesList']                              ## get list of coordinates for each axis


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if no region file is specified
        if (self.RegionFileName == ""):


            ## define RegionMask
            self.RegionMask = numpy.ones((self.length[1], self.length[0]), dtype=numpy.int)


            ## define RegionPixelsX and RegionPixelsY
            self.RegionPixelsX = []
            for x, xPixels in enumerate(self.AxisValuesList[0]):                            ## loop over all pixel along the x direction
                self.RegionPixelsX.append([xPixels, x])
            NumPixelY = self.length[1]
            self.RegionPixelsY = []
            for y, yPixels in enumerate(self.AxisValuesList[1]):                            ## loop over all pixel along the y direction
                self.RegionPixelsY.append([yPixels, y, (NumPixelY - y - 1)])


            ## define center of region
            LeftX = 1
            RightX = self.length[0]
            self.CentralPixelOfRegionIndexX = int((RightX - LeftX + 1) / 2.0) + 1
            self.CentralPixelOfRegionX = self.AxisValuesList[0][self.CentralPixelOfRegionIndexX - 1]
            TopY = 1
            BottomY = self.length[1]
            self.CentralPixelOfRegionIndexY = int((BottomY - TopY + 1) / 2.0) + 1
            self.CentralPixelOfRegionY = self.AxisValuesList[1][self.CentralPixelOfRegionIndexY - 1]

            # Debug:
            # print ("\nself.CentralPixelOfRegionIndexX = ", self.CentralPixelOfRegionIndexX)
            # print ("self.CentralPixelOfRegionX = ", self.CentralPixelOfRegionX)
            # print ("self.CentralPixelOfRegionIndexY = ", self.CentralPixelOfRegionIndexY)
            # print ("self.CentralPixelOfRegionY = ", self.CentralPixelOfRegionY)
            #    sys.exit(0)


            ## we're done
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if astropy is available
        self.astropyFlag = True
        try:
            from astropy.coordinates import SkyCoord
        except:
            self.astropyFlag = False


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define list of known coordinate systems
        self.KnownCoordinateSystemList = ["physical",                                       ## pixel coords of original file using LTM/LTV
                                          "image",                                          ## pixel coords of current file
                                          #"fk4,B1950",                                      ## sky coordinate systems
                                          "fk4",                                            ## sky coordinate systems
                                          #"fk5,J2000",                                      ## sky coordinate systems
                                          "fk5",                                            ## sky coordinate systems
                                          "icrs",                                           ## sky coordinate systems
                                          "galactic",                                       ## sky coordinate systems
                                          # "ecliptic",                                       ## sky coordinate systems (not known by astropy)
                                          "wcs",                                            ## primary WCS
                                          "wcsa",                                           ## secondary WCS
                                          "linear"]                                         ## linear primary WCS


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## use region subpackage of astropy


        ## check, if regions package is available
        RegionsPackageFlag = False


        ## TO DO: continue here, if regions package is available


        ##----------------------------------------------------------------------------------------------------------------------------------------------------
        ## import and analyze region file
        self.AnalyzeRegionFile()


        ##----------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine region parameters
        self.DefineRegionMask()


        ## finished
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## analyze given region file
    ##
    def AnalyzeRegionFile(self):
        """

    input parameters:
    -----------------

        - None



    output parameters:
    ------------------

        - None
        """


        ## define list of known shapes
        KnownShapeList = ["circle",                                                         ## circle x y radius # fill=[0|1]
                          "ellipse",                                                        ## ellipse x y radius radius angle # fill=[0|1]
                          "box",                                                            ## box x y width height angle # fill=[0|1]
                          "polygon"]                                                        ## polygon x1 y1 x2 y2 x3 y3 ... # fill=[0|1]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read in contents of region file
        regionFile = open(self.RegionFileName)
        regionFileContents = regionFile.readlines()
        regionFile.close()


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## analyze region file
        CoordSystem = ""
        self.RegionFileParameters = []
        for line in regionFileContents:


            ## remove comments from current line
            i = line.find("#")
            if (i > (-1)):
                StrippedLine = line[:i].strip()
            else:
                StrippedLine = line.strip()
            if (StrippedLine != "" ):

                # Debug:
                # print ("StrippedLine = ", StrippedLine)


                ## different settings may be separated by ";"
                SplittedLine = StrippedLine.split(";")
                for LocalLine in SplittedLine:
                    StrippedLine = LocalLine.strip()
                    if (StrippedLine != "" and (not StrippedLine.startswith("global"))):
                        LowerLine = StrippedLine.lower()

                        # Debug:
                        # print ("LowerLine = ", LowerLine)


                        ## check for description of coordinate systems
                        if (LowerLine in self.KnownCoordinateSystemList):
                            CoordSystem = LowerLine


                        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        ## check for shape definition
                        else:
                            LowerLine = LowerLine.replace("(", " ")
                            LowerLine = LowerLine.replace(")", " ")
                            LowerLine = LowerLine.replace(",", " ")
                            LocalSplittedLine = LowerLine.split()
                            TestName = LocalSplittedLine[0]
                            TestName = TestName.replace("-", " ")
                            if (TestName in KnownShapeList):
                                LocalDict = {}
                                LocalDict['CoordSystem'] = CoordSystem                      ## store coordinate system
                                LocalDict['ShapeName'] = TestName                           ## store name of shape
                                if (LocalSplittedLine[0].startswith("-")):
                                    LocalDict['InEx'] = "-"                                 ## shape describes excluded region
                                else:
                                    LocalDict['InEx'] = "+"                                 ## shape describes included region


                                ## convert parameter vector to degree
                                ShapeParameterVector = []
                                for elementID, element in enumerate(LocalSplittedLine[1:]):

                                    # Debug:
                                    # print ("elementID, element = ", elementID, element)


                                    ## convert radians to degrees
                                    if (element.find("r") > (-1)):
                                        element = element.replace("r", "")
                                        element = float(element) * 180.0 / math.pi


                                    ## convert minutes to degrees
                                    elif (element.find(chr(39)) > (-1)):
                                        element = element.replace(chr(39), "")
                                        minutes = float(element)
                                        #self.check_hms_ranges(0, minutes, 0)
                                        element = minutes / 60.0


                                    ## convert seconds to degrees
                                    elif (element.find(chr(34)) > (-1)):
                                        element = element.replace(chr(34), "")
                                        seconds = float(element)
                                        #self.check_hms_ranges(0, 0, seconds)
                                        element = seconds / 3600.0


                                    ## convert sexagesimal to degrees
                                    elif (element.find(":") > (-1)):
                                        SplittedElement = element.split(":")
                                        hours = int(SplittedElement[0])
                                        minutes = float(SplittedElement[1])
                                        seconds = float(SplittedElement[2])
                                        #self.check_hms_ranges(hours, minutes, seconds)

                                        # Debug:
                                        # print ("hours, minutes, seconds = ", hours, minutes, seconds)
                                        # print ("elementID = ", elementID)


                                        ## convert values to degrees for coordinate system: fk5
                                        if (elementID % 2 == 0):
                                            element = self.RAinDegree(hours, minutes, seconds)
                                        else:
                                            degree = hours
                                            element = self.DECinDegree(degree, minutes, seconds)


                                    ## store converted degrees
                                    ShapeParameterVector.append(element)


                                ## store shape parameters
                                LocalDict['ShapeParameter'] = ShapeParameterVector

                                # Debug:
                                # print ("\nLocalDict = ", LocalDict)


                                ## store shape parameters
                                self.RegionFileParameters.append(LocalDict)

        # Debug:
        # print ("self.RegionFileParameters = ", self.RegionFileParameters)


        ## finished
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## check hour range (Checks that the given value is in the range (-24, 24).)
    ##
    def CheckHourRange(self, hrs):
        """

    input parameters:
    -----------------

        - hrs:                  hour


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("hrs = ", hrs)


        ## print warning
        if (abs(hrs) == 24.0):
            print ("Treating hour as 24 hr")
        elif (hrs < -24. or hrs > 24.):
            print ("Error: Hours not in range (-24, 24): Hours = ", hrs)


        ## finished
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## check minute range (Checks that the given value is in the range [0,60]. If the value is equal to 60, then a warning is raised.)
    ##
    def CheckMinuteRange(self, m):
        """

    input parameters:
    -----------------

        - m:                    minute


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("m = ", m)


        ## print warning
        if (m == 60.0):
            print ("Treating minute as 0 min, +1 hr/deg")
        elif (m < 0.0 or m > 60.0):
            print ("Error: Minutes not in range [0,60): Minute = ", m)


        ## finished
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## check second range (Checks that the given value is in the range [0,60). If the value is equal to 60, then a warning is raised.)
    ##
    def CheckSecondRange(self, sec):
        """

    input parameters:
    -----------------

        - sec:                  second


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("sec = ", sec)


        ## print warning
        if (sec == 60.0):
            print ("Treating as 0 sec, +1 min")
        elif (sec < 0.0 or sec > 60.0):
            print ("Error: Seconds not in range [0,60): Second = ", sec)


        ## finished
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## check hours, minutes and seconds ranges (Checks that the given hour, minute and second are all within reasonable range.)
    ##
    def check_hms_ranges(self, h, m, s):
        """

    input parameters:
    -----------------

        - h:                    hour

        - m:                    minutes

        - s:                    seconds


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("h = ", h)
        # print ("m = ", m)
        # print ("s = ", s)


        ## check hours, minutes and seconds ranges
        self.CheckHourRange(h)
        self.CheckMinuteRange(m)
        self.CheckSecondRange(s)


        ## finished
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## get sign of value
    ##
    def t_SIGN(self, value):
        """

    input parameters:
    -----------------

        - value:                value


    output parameters:
    ------------------

        - t:                    sign of value
        """

        # Debug:
        # print ("value = ", value)


        ## initialize return parameter
        t = 1.0


        ## compute sign of value
        if (value < 0):
            t = -1.0
        else:
            t = 1.0


        ## define return parameter
        return t
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## convert right ascension to decimal degrees
    ## http://en.wikipedia.org/wiki/Right_ascension, call 15.10.2013
    ## 1 h = 15\circ = math.pi/12 rad
    ## 1 minute = 15' = 1/4\circ = math.pi/720 rad
    ## 1 second = 15" = 1/4' = 1/240\circ = math.pi/43200 rad
    ##
    def RAinDegree(self, h, m, s):
        """

    input parameters:
    -----------------

        - h:                    hour

        - m:                    minutes

        - s:                    seconds


    output parameters:
    ------------------

        - RADegree:             decimal degrees
        """

        # Debug:
        # print ("h = ", h)
        # print ("m = ", m)
        # print ("s = ", s)


        ## initialize return parameter
        RADegree = 0.0


        ## compute right ascension in decimal degrees
        sign = self.t_SIGN(h)
        RADegree = sign * (h * 15.0 + m * 15.0 / 60.0 + s * 15.0 / 3600.0)


        ## define return parameter
        return RADegree
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## convert declination to decimal degrees
    ##
    def DECinDegree(self, degree, minutes, seconds):
        """

    input parameters:
    -----------------

        - degree:               degree

        - minutes:              minutes

        - seconds:              seconds


    output parameters:
    ------------------

        - DecDegree:            decimal degrees
        """

        # Debug:
        # print ("degree = ", degree)
        # print ("minutes = ", minutes)
        # print ("seconds = ", seconds)


        ## initialize return parameter
        DecDegree = 0.0


        ## compute declination in decimal degrees
        sign = self.t_SIGN(degree)
        DecDegree = sign * (abs(degree) + minutes / 60.0 + seconds / 3600.0)


        ## define return parameter
        return DecDegree
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## adjust boundaries of figure
    ##
    def AdjustPlot(self, fig, layer, xLow, xUp, yLow, yUp):
        """

    input parameters:
    -----------------

        - fig:                  figure object

        - layer:                layer object

        - xLow:                 lower limit of x-axis

        - xUp:                  upper limit of x-axis

        - yLow:                 lower limit of y-axis

        - yUp:                  upper limit of y-axis


    output parameters:
    ------------------

        - DecDegree:            decimal degrees
        """

        # Debug:
        # print ("fig = ", fig)
        # print ("layer = ", layer)
        # print ("xLow = ", xLow)
        # print ("xUp = ", xUp)
        # print ("yLow = ", yLow)
        # print ("yUp = ", yUp)


        ## set test markers
        self.RegionMask[:, :] = 0.0
        self.RegionMask[ 0,  0] = 100.0
        self.RegionMask[-1,  0] = 100.0
        self.RegionMask[ 0, -1] = 100.0
        self.RegionMask[-1, -1] = 100.0


        ## define defaults boundaries
        NewLeftBound = 0.0
        NewBottomBound = 0.0
        NewRightBound = 1.0
        NewTopBound = 1.0


        ## adjust boundaries to really include all pixels
        LeftBoundList = [-0.01, 0.000, 0.01]
        BottomBoundList = [-0.01, 0.000, 0.01]
        RightBoundList = [0.999, 1.000, 1.01]
        TopBoundList = [0.999, 1.000, 1.01]
        FoundFlag = False
        i = 0
        for LeftBound in LeftBoundList:
            for BottomBound in BottomBoundList:
                for RightBound in RightBoundList:
                    for TopBound in TopBoundList:
                        i += 1


                        ## clear figure
                        plt.cla()


                        ## define axis
                        ## taken https://stackoverflow.com/questions/45285189/matplotlib-imshow-use-any-vector-as-axis
                        layer.axis([xLow, xUp, yLow, yUp])
                        layer.set_axis_off()


                        ## set new boundaries
                        fig.subplots_adjust(left = LeftBound, bottom = BottomBound, right = RightBound, top = TopBound, wspace = 0.0, hspace = 0.0)


                        ## plot test image
                        layer.imshow(self.RegionMask)


                        ## draw figure
                        fig.canvas.draw()


                        ## save figure to a numpy array
                        ## taken from https://matplotlib.org/gallery/user_interfaces/canvasagg.html
                        s, (width, height) = fig.canvas.print_to_buffer()
                        # X = numpy.fromstring(s, numpy.uint8).reshape((height, width, 4))    ## "4" because of (r, g, b, a)
                        X = numpy.frombuffer(s, numpy.uint8).reshape((height, width, 4))    ## "4" because of (r, g, b, a)
                                                                                            ## rgb(0,0,0) -> black
                                                                                            ## rgb(255,255,255) -> white
                        self.RegionMask = copy.deepcopy(X[:, :, 0])
                        self.RegionMask[self.RegionMask != 255] = 1
                        self.RegionMask[self.RegionMask == 255] = 0


                        ## test if all markers are visible
                        if (self.RegionMask[0, 0] == 1 and self.RegionMask[0, 1] == 0 and self.RegionMask[1, 0] == 0 \
                            and self.RegionMask[0, -1] == 1 and self.RegionMask[1, -1] == 0 and self.RegionMask[0, -2] == 0 \
                            and self.RegionMask[-1, 0] == 1 and self.RegionMask[-2, 0] == 0 and self.RegionMask[-1, 1] == 0 \
                            and self.RegionMask[-1, -1] == 1 and self.RegionMask[-2, -1] == 0 and self.RegionMask[-1, -2] == 0):
                            FoundFlag = True
                            NewLeftBound = LeftBound
                            NewBottomBound = BottomBound
                            NewRightBound = RightBound
                            NewTopBound = TopBound
                            break
                    if (FoundFlag):
                        break
                if (FoundFlag):
                    break
            if (FoundFlag):
                break

        # Debug:
        # print ("\n\nNewLeftBound = ", NewLeftBound)
        # print ("NewBottomBound = ", NewBottomBound)
        # print ("NewRightBound = ", NewRightBound)
        # print ("NewTopBound = ", NewTopBound)
        # print ("self.RegionMask[0, 0] = ", self.RegionMask[0, 0])
        # print ("self.RegionMask[-1, 0] = ", self.RegionMask[-1, 0])
        # print ("self.RegionMask[0, -1] = ", self.RegionMask[0, -1])
        # print ("self.RegionMask[-1, -1] = ", self.RegionMask[-1, -1])


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## clear figure
        fig.subplots_adjust(left = NewLeftBound, bottom = NewBottomBound, right = NewRightBound, top = NewTopBound, wspace = 0.0, hspace = 0.0)
        plt.cla()
        self.RegionMask[:, :] = 0.0


        ## define return parameter
        return fig
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## transform a given image pixel to degree
    ##
    def Image2Degree(self, CoordSystem, pixel, axis = 0):
        """

    input parameters:
    -----------------

        - CoordSystem:          coordinate system

        - pixel:                coordinate


    output parameters:
    ------------------

        - NEWCoord:             coordinate in degree
        """

        # Debug:
        # print ("CoordSystem = ", CoordSystem)
        # print ("pixel = ", pixel)
        # print ("axis = ", axis)


        ## initialize return parameters
        NEWCoord = pixel


        ## just use pixel coordinates
        if (CoordSystem in ["physical", "image"]):
            NEWCoord = self.CDELTList[axis] * pixel


        ## define return parameter
        return NEWCoord
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## transform given coordinate to fk5
    ##
    def TransformFrame(self, CoordSystem, xCoord, yCoord):
        """

    input parameters:
    -----------------

        - CoordSystem:          coordinate system

        - xCoord:               x-coordinate

        - yCoord:               y-coordinate


    output parameters:
    ------------------

        - DecDegree:            decimal degrees
        """

        # Debug:
        # print ("\n\nCoordSystem = ", CoordSystem)
        # print ("xCoord = ", xCoord)
        # print ("yCoord = ", yCoord)


        ## initialize return parameters
        NEWxCoord = xCoord
        NEWyCoord = yCoord


        ## import astropy packages
        if (self.astropyFlag):
            from astropy.coordinates import SkyCoord
            import astropy.units as u


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## transform coordinate system to given reference frame and define axis
        ## taken from http://docs.astropy.org/en/stable/coordinates/skycoord.html
        ## and https://stackoverflow.com/questions/45285189/matplotlib-imshow-use-any-vector-as-axis


        ## just use pixel coordinates
        if (CoordSystem in ["physical", "image"]):
            y0 = (self.AxisValuesList[1][0] - (self.CDELTList[1] / 2.0))
            dy = self.Image2Degree(CoordSystem, (yCoord - 0.5), axis = 1)
            yCoord = y0 + dy
            x0 = (self.AxisValuesList[0][0] - (self.CDELTList[0] / 2.0))
            dx = (self.Image2Degree(CoordSystem, (xCoord - 0.5), axis = 0) / math.cos(math.pi / 180.0 * yCoord))
            xCoord = x0 + dx
            return (xCoord, yCoord)


        ## use astropy to get lower and upper limit for x-and y-axis by taking the current reference frame "fk4", "fk5" etc. into account
        if (self.astropyFlag and (CoordSystem in self.KnownCoordinateSystemList)):
            from astropy import units as u


            ## first convert FITS axis limits to fk5 frame and then to current frame
            if (CoordSystem in ["galactic"]):
                CoordObject = SkyCoord(l = xCoord*u.deg, b = yCoord*u.deg, frame = CoordSystem)
            else:
                CoordObject = SkyCoord(ra = xCoord*u.deg, dec = yCoord*u.deg, frame = CoordSystem)
            FK5CoordObject = CoordObject.transform_to('fk5')


            ## get ra and dec in degree
            NEWxCoord = FK5CoordObject.ra*u.deg
            NEWyCoord = FK5CoordObject.dec*u.deg


            ## get values without unit
            NEWxCoord = NEWxCoord.value
            NEWyCoord = NEWyCoord.value

        # Debug:
        # print ("\n\nCoordObject = ", CoordObject)
        # print ("FK5CoordObject = ", FK5CoordObject)
        # print ("xCoord, NEWxCoord = ", xCoord, NEWxCoord)
        # print ("yCoord, NEWyCoord = ", yCoord, NEWyCoord)


        ## define return parameter
        return (NEWxCoord, NEWyCoord)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## define region mask
    ##
    def DefineRegionMask(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## import matplotlib and astropy package
        from matplotlib.collections import PatchCollection


        ## define number of pixels along each axis
        NumPixelX = self.length[0]
        NumPixelY = self.length[1]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create figure
        plt.close('all')
        plt.clf()
        fig, layer = plt.subplots(nrows = 1, ncols = 1)
        fig.subplots_adjust(left = 0.0, bottom = 0.0, right = 1.0, top = 1.0, wspace = 0.0, hspace = 0.0)


        ## define size of image in pixel and not in inches
        ## taken from https://github.com/matplotlib/matplotlib/issues/2305/
        DPI = fig.get_dpi()
        fig.set_size_inches(NumPixelX / float(DPI), NumPixelY / float(DPI))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create a numpy array with dimension of FITS image
        #RegionMask = numpy.zeros((NumPixelX, NumPixelY), dtype = numpy.int)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define axis
        ## taken https://stackoverflow.com/questions/45285189/matplotlib-imshow-use-any-vector-as-axis


        ## get limits of axis
        x = self.AxisValuesList[0]
        y = self.AxisValuesList[1]
        xLow = x[0]
        xUp = x[-1]
        yLow = y[0]
        yUp = y[-1]


        ## axis coordinates describe center of each pixel, i.e. axis have to be start earlier and end later
        xLow -= (self.CDELTList[0] / 2.0)
        xUp += (self.CDELTList[0] / 2.0)
        yLow -= (self.CDELTList[1] / 2.0)
        yUp += (self.CDELTList[1] / 2.0)


        ## set axis
        # pylab.gca().set_aspect('equal', 'box')
        layer.axis([xLow, xUp, yLow, yUp])
        layer.set_axis_off()

        # Debug:
        # print ("\nself.CDELTList[0] = ", self.CDELTList[0])
        # print ("self.AxisValuesList[0][0] = ", self.AxisValuesList[0][0])
        # print ("self.AxisValuesList[0][-1] = ", self.AxisValuesList[0][-1])
        # print ("xLow = ", xLow)
        # print ("xUp = ", xUp)
        # print ("\nself.CDELTList[1] = ", self.CDELTList[1])
        # print ("self.AxisValuesList[1][0] = ", self.AxisValuesList[1][0])
        # print ("self.AxisValuesList[1][-1] = ", self.AxisValuesList[1][-1])
        # print ("yLow = ", yLow)
        # print ("yUp = ", yUp)
        # print ("layer.get_xlim() = ", layer.get_xlim())
        # print ("layer.get_ylim() = ", layer.get_ylim())


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## adjust boundaries
        if (self.AdjustBoundFlag):
            fig = self.AdjustPlot(fig, layer, xLow, xUp, yLow, yUp)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create regions
        RegionPatches = []
        for LocalRegParam in self.RegionFileParameters:


            ## get parameter
            CoordSystem = LocalRegParam['CoordSystem']                                      ## get coordinate system
            ShapeName = LocalRegParam['ShapeName']                                          ## get name of shape
            InEx = LocalRegParam['InEx']                                                    ## get if region is included or excluded
            if (InEx == "-"):
                DrawColor = 'white'
            else:
                DrawColor = 'black'
            ShapeParameter = LocalRegParam['ShapeParameter']                                ## get additional parameters of region


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## handle boxes
            if (ShapeName == "box"):


                ## get parameter for rectangle
                CenterX = float(ShapeParameter[0])
                CenterY = float(ShapeParameter[1])
                CenterX, CenterY = self.TransformFrame(CoordSystem, CenterX, CenterY)
                Width = float(ShapeParameter[2])
                Width = abs(self.Image2Degree(CoordSystem, Width, axis = 0))
                if (self.ApplyRACorrection):
                    Width = Width / math.cos(math.pi / 180.0 * CenterY)
                Height = float(ShapeParameter[3])
                Height = abs(self.Image2Degree(CoordSystem, Height, axis = 1))
                try:
                    Angle = -float(ShapeParameter[4])
                except:
                    Angle = 0.0

                # Debug:
                # print ("\nCenterX = ", CenterX)
                # print ("CenterY = ", CenterY)
                # print ("Width = ", Width)
                # print ("Height = ", Height)
                # print ("Angle = ", Angle)
                # print ("DrawColor = ", DrawColor)
                # print ("self.CDELTList[0:2] = ", self.CDELTList[0:2])
                # print ("corr_yLow = ", 1.0 / math.cos(math.pi / 180.0 * yLow))
                # print ("corr_yUp = ", 1.0 / math.cos(math.pi / 180.0 * yUp))
                # print ("corr_CenterY = ", 1.0 / math.cos(math.pi / 180.0 * CenterY))


                ## add rectangle to figure
                BottomLeftX = CenterX + self.t_SIGN(self.CDELTList[0]) * (Width / 2.0)
                BottomLeftY = CenterY - self.t_SIGN(self.CDELTList[1]) * (Height / 2.0)
                try:
                    rectangle = matplotlib.patches.Rectangle((BottomLeftX, BottomLeftY), Width, Height, angle = Angle, color = DrawColor, fill = True, \
                                                             antialiased = False)
                except:                                                                     ## CASA uses matplotlib 1.1.0, which has no Angle keyword
                    rectangle = matplotlib.patches.Rectangle((BottomLeftX, BottomLeftY), Width, Height, color = DrawColor, fill = True, \
                                                             antialiased = False)
                RegionPatches.append(rectangle)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## handle circles
            elif (ShapeName == "circle"):


                ## get parameter for circle
                CenterX = float(ShapeParameter[0])
                CenterY = float(ShapeParameter[1])
                CenterX, CenterY = self.TransformFrame(CoordSystem, CenterX, CenterY)
                Radius = float(ShapeParameter[2])
                Radius = abs(self.Image2Degree(CoordSystem, Radius, axis = 0))
                #    if (self.ApplyRACorrection):
                #        Radius = Radius / math.cos(math.pi / 180.0 * CenterY)

                # Debug:
                # print ("\n\nCenterX = ", CenterX)
                # print ("CenterY = ", CenterY)
                # print ("Radius = ", Radius)


                ## add circle to figure
                circle = matplotlib.patches.Circle((CenterX, CenterY), radius = Radius, color = DrawColor, fill = True, antialiased = False)
                RegionPatches.append(circle)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## handle ellipses
            elif (ShapeName == "ellipse"):


                ## get parameter for ellipse
                CenterX = float(ShapeParameter[0])
                CenterY = float(ShapeParameter[1])
                CenterX, CenterY = self.TransformFrame(CoordSystem, CenterX, CenterY)
                Width = 2.0 * float(ShapeParameter[2])                                      ## matplotlib requires the diameter not the radius
                Width = abs(self.Image2Degree(CoordSystem, Width, axis = 0))
                if (self.ApplyRACorrection):
                    Width = Width / math.cos(math.pi / 180.0 * CenterY)
                Height = 2.0 * float(ShapeParameter[3])                                     ## matplotlib requires the diameter not the radius
                Height = abs(self.Image2Degree(CoordSystem, Height, axis = 1))
                try:
                    Angle = -float(ShapeParameter[4])
                except:
                    Angle = 0.0

                # Debug:
                # print ("\n\nCenterX = ", CenterX)
                # print ("CenterY = ", CenterY)
                # print ("Width = ", Width)
                # print ("Height = ", Height)
                # print ("Angle = ", Angle)


                ## add ellipse to figure
                try:
                    ellipse = matplotlib.patches.Ellipse((CenterX, CenterY), Width, Height, angle = Angle, color = DrawColor, fill = True, \
                                                         antialiased = False)
                except:                                                                     ## CASA uses matplotlib 1.1.0, which has no Angle keyword
                    ellipse = matplotlib.patches.Ellipse((CenterX, CenterY), Width, Height, color = DrawColor, fill = True, antialiased = False)
                RegionPatches.append(ellipse)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## handle polygons
            elif (ShapeName == "polygon"):


                ## get parameter for polygon
                ListCoord = []
                LocalPos = []
                i = 0
                for Elements in ShapeParameter[0:]:
                    i += 1
                    LocalPos.append(float(Elements))
                    if (i % 2 == 0):
                        xPos, yPos = self.TransformFrame(CoordSystem, LocalPos[0], LocalPos[1])
                        ListCoord.append([xPos, yPos])
                        LocalPos = []
                ListCoord = numpy.asarray(ListCoord)

                # Debug:
                # print ("ListCoord = ", ListCoord)


                ## add polygon to figure
                polygon = matplotlib.patches.Polygon(ListCoord, closed = True, color = DrawColor, fill = True)
                RegionPatches.append(polygon)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## draw figure
        p = PatchCollection(RegionPatches)
        layer.add_collection(p)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## draw figure
        fig.canvas.draw()


        ## for debugging:
        if (self.dbDir != ""):
            dbFile = self.dbDir + "regions.png"
            plt.savefig(dbFile, dpi = 300)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## save figure to a numpy array
        ## taken from https://matplotlib.org/gallery/user_interfaces/canvasagg.html
        s, (width, height) = fig.canvas.print_to_buffer()
        # X = numpy.fromstring(s, numpy.uint8).reshape((height, width, 4))                  ## "4" because of (r, g, b, a)
        X = numpy.frombuffer(s, numpy.uint8).reshape((height, width, 4))                    ## "4" because of (r, g, b, a)
                                                                                            ## rgb(0, 0, 0) -> black
                                                                                            ## rgb(255, 255, 255) -> white
        self.RegionMask = copy.deepcopy(X[:, :, 0])
        self.RegionMask[self.RegionMask != 255] = 1
        self.RegionMask[self.RegionMask == 255] = 0


        ## for debugging:
        if (self.dbDir != ""):
            dbgFile = open(self.dbDir + "regions.dat", "w")
            for line in self.RegionMask:
                NewLine = ""
                for element in line:
                    element = str(element).replace("0", ".")
                    element = element.replace("[", "")
                    element = element.replace("]", "")
                    element = element.replace("\n", "")
                    element = element.replace("1", "x")
                    element = element.replace(" ", "")
                    NewLine += element
                dbgFile.write(NewLine + "\n")
            dbgFile.close()


        ## close figure
        matplotlib.pyplot.close(fig)

        # Debug:
        # print ("self.RegionMask[0, :] = ", self.RegionMask[0, :])
        # print ("self.RegionMask[-1, :] = ", self.RegionMask[-1, :])
        # print ("self.RegionMask[:, 0] = ", self.RegionMask[:, 0])
        # print ("self.RegionMask[:, -1] = ", self.RegionMask[:, -1])
        # print ("NumPixelX = ", NumPixelX)
        # print ("len(self.RegionMask[0, :]) = ", len(self.RegionMask[0, :]))
        # print ("NumPixelY = ", NumPixelY)
        # print ("len(self.RegionMask[:, 0]) = ", len(self.RegionMask[:, 0]))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## extract region parameters from RegionMask


        ## determine min. and max. index of region(s) along x-axis
        LeftX = (-1)
        RightX = (-1)
        for x in range(NumPixelX):
            if (numpy.sum(self.RegionMask[:, x]) != 0.0 and LeftX == (-1)):
                LeftX = x
            if (numpy.sum(self.RegionMask[:, -x]) != 0.0 and RightX == (-1)):
                RightX = NumPixelX - x
            if (LeftX > (-1) and RightX > (-1)):                                            ## exit loop if left and right edges are identified
                break
        if (LeftX == (-1)):
            LeftX = 0
        if (RightX == (-1)):
            RightX = NumPixelX - 1
        self.RegionPixelsX = []
        for x, xPixels in enumerate(self.AxisValuesList[0]):                                ## loop over all pixel along the x direction
            if (LeftX <= x and x <= RightX):
                self.RegionPixelsX.append([xPixels, x])

        # Debug:
        # print ("\nLeftX = ", LeftX)
        # print ("RightX = ", RightX)
        # print ("self.RegionPixelsX = ", self.RegionPixelsX)


        ## determine min. and max. index of region(s) along y-axis
        TopY = (-1)
        BottomY = (-1)
        for y in range(NumPixelY):
            if (numpy.sum(self.RegionMask[y, :]) != 0.0 and TopY == (-1)):
                TopY = y
            if (numpy.sum(self.RegionMask[-y, :]) != 0.0 and BottomY == (-1)):
                BottomY = NumPixelY - y
            if (TopY > (-1) and BottomY > (-1)):                                            ## exit loop if bottom and top edges are identified
                break
        if (TopY == (-1)):
            TopY = 0
        if (BottomY == (-1)):
            BottomY = NumPixelY - 1
        self.RegionPixelsY = []
        for y, yPixels in enumerate(self.AxisValuesList[1]):                                ## loop over all pixel along the y direction
            if (TopY <= (NumPixelY - y - 1) and (NumPixelY - y - 1) <= BottomY):            ## y-axis indexing is reversed compared to RegionMask
                self.RegionPixelsY.append([yPixels, y, (NumPixelY - y - 1)])                ## FITS file indexing starts in the lower left corner, why
                                                                                            ## a third element is necessary for self.RegionMask indexing
        # Debug:
        # print ("\nTopY = ", TopY)
        # print ("BottomY = ", BottomY)
        # print ("self.RegionPixelsY = ", self.RegionPixelsY)


        ## determine central pixel of region
        self.CentralPixelOfRegionIndexX = int((RightX - LeftX + 1) / 2.0)                   ## (+ LeftX) for the FITS export we need the relative index!
        self.CentralPixelOfRegionX = self.RegionPixelsX[self.CentralPixelOfRegionIndexX][0]
        self.CentralPixelOfRegionIndexY = int((BottomY - TopY + 1) / 2.0)                   ## (+ TopY) for the FITS export we need the relative index!
        self.CentralPixelOfRegionY = self.RegionPixelsY[self.CentralPixelOfRegionIndexY][0]

        # Debug:
        # print ("\nself.CentralPixelOfRegionIndexX = ", self.CentralPixelOfRegionIndexX)
        # print ("self.CentralPixelOfRegionX = ", self.CentralPixelOfRegionX)
        # print ("self.CentralPixelOfRegionIndexY = ", self.CentralPixelOfRegionIndexY)
        # print ("self.CentralPixelOfRegionY = ", self.CentralPixelOfRegionY)
        # sys.exit(0)


        ## define return parameter
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## get region parameter
    ##
    def GetRegionParameters(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - self.RegionParameters:            region parameters
    """


        ## create dictionary containing region parameters
        self.RegionParameters = {}
        self.RegionParameters['TotalNumberPixels'] = numpy.sum(self.RegionMask)             ## total number of fitted pixels
        self.RegionParameters['RegionMask'] = self.RegionMask                               ## region describing which pixel have to be fitted
        self.RegionParameters['RegionPixelsX'] = self.RegionPixelsX                         ## absolute x-coordinates and corresponding absolute index
        NumRegionPixelsX = len(self.RegionPixelsX)
        self.RegionParameters['NumRegionPixelsX'] = NumRegionPixelsX                        ## number of columns of region
        self.RegionParameters['RegionPixelsY'] = self.RegionPixelsY                         ## absolute y-coordinates and corresponding absolute index
        NumRegionPixelsY = len(self.RegionPixelsY)
        self.RegionParameters['NumRegionPixelsY'] = NumRegionPixelsY                        ## number of columns of region
        self.RegionParameters['CentralPixelOfRegionX'] = self.CentralPixelOfRegionX         ## central absolute x-coordinate of region
        self.RegionParameters['CentralPixelOfRegionIndexX'] = self.CentralPixelOfRegionIndexX   ## relative x-index of region center
        self.RegionParameters['CentralPixelOfRegionY'] = self.CentralPixelOfRegionY         ## central absolute y-coordinate of region
        self.RegionParameters['CentralPixelOfRegionIndexY'] = self.CentralPixelOfRegionIndexY   ## relative y-index of region center

        # Debug:
        # print ("\nTotalNumberPixels = ", self.RegionParameters['TotalNumberPixels'])
        # print ("RegionMask = ", self.RegionParameters['RegionMask'])
        # print ("RegionPixelsX = ", self.RegionParameters['RegionPixelsX'])
        # print ("NumRegionPixelsX = ", self.RegionParameters['NumRegionPixelsX'])
        # print ("RegionPixelsY = ", self.RegionParameters['RegionPixelsY'])
        # print ("NumRegionPixelsY = ", self.RegionParameters['NumRegionPixelsY'])
        # print ("CentralPixelOfRegionX = ", self.RegionParameters['CentralPixelOfRegionX'])
        # print ("CentralPixelOfRegionIndexX = ", self.RegionParameters['CentralPixelOfRegionIndexX'])
        # print ("CentralPixelOfRegionY = ", self.RegionParameters['CentralPixelOfRegionY'])
        # print ("CentralPixelOfRegionIndexY = ", self.RegionParameters['CentralPixelOfRegionIndexY'])
        #    sys.exit(0)


        ## define return parameters
        return self.RegionParameters
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
class LoadExp:
    ## class for loading experimental-file
    """

    This class contains the function to import obs. xml-file and the corresponding obs. data-file.

    """


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## initialize class variables
    ##
    def __init__(self, PrintFlag):
        """

    input parameters:
    -----------------

        - PrintFlag:                    flag if screen output is permitted (=true) or not


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("PrintFlag = ", PrintFlag)


        ## initialize some parameters
        self.InputFormat = "fits"                                                           ## format of the experimental file
        self.NumberExpFiles = 1                                                             ## number of exp files
        self.FileNamesExpFiles = []                                                         ## path and file name of each experimental file
        self.NumberHeaderLines = []                                                         ## number of header lines for each experimental-file
        self.NumberHDU = []                                                                 ## number of HDU for each fits-file
        self.SeparatorColumns = []                                                          ## column separator for each experimental-file
        self.NumberColumnsX = []                                                            ## number of columns belonging to the x-positions
        self.NumberColumnsY = []                                                            ## number of columns belonging to the y-positions
        self.ErrorY = []                                                                    ## flag if error of the experimental data is given
        self.NumberExpRanges = []                                                           ## number of selected experimental ranges
        self.MinExpRange = []                                                               ## min of each experimental range
        self.MaxExpRange = []                                                               ## max of each experimental range


        ## store global FITS parameters
        self.AllFITSParameters = []
        self.AllRangeAxis = []


        ## initialize region parameters
        self.TotalNumberPixels = None
        self.RegionMask = None
        self.RegionPixelsX = None
        self.NumRegionPixelsX = None
        self.RegionPixelsY = None
        self.NumRegionPixelsY = None
        self.CentralPixelOfRegionX = None
        self.CentralPixelOfRegionIndexX = None
        self.CentralPixelOfRegionY = None
        self.CentralPixelOfRegionIndexY = None


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## tags used for MAGIX:     "NumberExpFiles", "MinExpRange", "MaxExpRange", "StepFrequency", ..
        ##
        ## special handling:        "dbFilename"
        ##
        ##
        ##
        ##               "Name of MAGIX variable:"               "Name of XML tag:"          "location:"        "default value:"     "Format:"
        ##                                                                                   "r":   range                            "f":   float
        ##                                                                                   "f":   file                             "b":   bool
        ##                                                                                   "o":   once                             "i":   integer
        ##                                                                                                                           "s":   string
        self.TagLists = [["MAGIXImportStartFrequency",           "MinExpRange",              "r",               0.0,                 "float"], \
                         ["MAGIXImportEndFrequency",             "MaxExpRange",              "r",               1.e9,                "float"], \
                         ["MAGIXImportStepFrequency",            "StepFrequency",            "r",               1.0,                 "float"], \
                         ["MAGIXImportTBackFlag",                "t_back_flag",              "r",               False,               "bool"], \
                         ["MAGIXImportBackgroundTemperature",    "BackgroundTemperature",    "r",               0.0,                 "float"], \
                         ["MAGIXImportTemperatureSlope",         "TemperatureSlope",         "r",               0.0,                 "float"], \
                         ["MAGIXImportBackgroundFileName",       "BackgroundFileName",       "r",               "",                  "string"], \
                         ["MAGIXImportHydrogenColumnDensity",    "HydrogenColumnDensity",    "r",               1.e21,               "float"], \
                         ["MAGIXImportDustBeta",                 "DustBeta",                 "r",               0.0,                 "float"], \
                         ["MAGIXImportKappa",                    "Kappa",                    "r",               0.0,                 "float"], \
                         ["MAGIXImportDustFileName",             "DustFileName",             "r",               "",                  "string"], \
                         ["MAGIXImportTeff",                     "Te_ff",                    "r",               0.0,                 "float"], \
                         ["MAGIXImportEMff",                     "EM_ff",                    "r",               0.0,                 "float"], \
                         ["MAGIXImportKappaSync",                "kappa_sync",               "r",               0.0,                 "float"], \
                         ["MAGIXImportBSync",                    "B_sync",                   "r",               0.0,                 "float"], \
                         ["MAGIXImportPSync",                    "p_sync",                   "r",               0.0,                 "float"], \
                         ["MAGIXImportLSync",                    "l_sync",                   "r",               0.0,                 "float"], \
                         ["MAGIXImportContPhenID",               "ContPhenFuncID",           "r",               0,                   "int"], \
                         ["MAGIXImportContParam1",               "ContPhenFuncParam1",       "r",               0.0,                 "float"], \
                         ["MAGIXImportContParam2",               "ContPhenFuncParam2",       "r",               0.0,                 "float"], \
                         ["MAGIXImportContParam3",               "ContPhenFuncParam3",       "r",               0.0,                 "float"], \
                         ["MAGIXImportContParam4",               "ContPhenFuncParam4",       "r",               0.0,                 "float"], \
                         ["MAGIXImportContParam5",               "ContPhenFuncParam5",       "r",               0.0,                 "float"], \
                         ["MAGIXImportGlobalvLSR",               "GlobalvLSR",               "f",               0.0,                 "float"], \
                         ["MAGIXImportRedshift",                 "Redshift",                 "f",               0.0,                 "float"], \
                         ["MAGIXImportTelescopeSize",            "TelescopeSize",            "f",               1.0,                 "float"], \
                         ["MAGIXImportTelescopeBMIN",            "TelescopeBMIN",            "f",               -1.e99,              "float"], \
                         ["MAGIXImportTelescopeBMAJ",            "TelescopeBMAJ",            "f",               -1.e99,              "float"], \
                         ["MAGIXImportTelescopeBPA",             "TelescopeBPA",             "f",               -1.e99,              "float"], \
                         ["MAGIXImportInterferrometerFlag",      "Inter_Flag",               "f",               False,               "bool"], \
                         ["MAGIXImportIsotopologues",            "iso_flag",                 "o",               False,               "bool"], \
                         ["MAGIXImportIsotopologues",            "Isotopologues",            "o",               False,               "bool"], \
                         ["MAGIXImportIsoTableFileName",         "IsoTableFileName",         "o",               "",                  "string"], \
                         ["MAGIXImportdbFilename",               "dbFilename",               "o",               "",                  "string"], \
                         ["MAGIXImportEmAbsPATH",                "EmAbsPATH",                "o",               "",                  "string"], \
                         ["MAGIXImportNumModelPixelXX",          "NumModelPixelXX",          "o",               100,                 "int"], \
                         ["MAGIXImportNumModelPixelYY",          "NumModelPixelYY",          "o",               100,                 "int"], \
                         ["MAGIXImportLocalOverlapFlag",         "LocalOverlap_Flag",        "o",               False,               "bool"], \
                         ["MAGIXImportNoSubBeamFlag",            "NoSubBeamFlag",            "o",               True,                "bool"], \
                         ["MAGIXImportErrorY",                   "ErrorY",                   "f",               False,               "bool"], \
                         ["MAGIXImportNumHeadLines",             "NumberHeaderLines",        "f",               0,                   "int"], \
                         ["MAGIXImportSeparatorColumns",         "SeparatorColumns",         "f",               "",                  "string"]]


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## print what you do
        if (PrintFlag != "false"):
            print ("\nImport observational xml file:")


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## gets min. and max. frequency of fits file
    ##
    def GetMinMaxFitsFile(self, filename, HDU):
        """

        input parameters:       filename:                   path and name of FITS file
                                HDU:                        number of HDU

        output parameters:      LowestDataPoint:            lowest point of the first axis
                                HighestDataPoint:           highest point of the first axis

        working variables:      k:                          loop counter
                                filename:                   temp variable for file name of the ith fits-file
                                hdulist:                    temp variable containing the whole ith fits-file
                                fitsdata:                   temp variable containing the data of the ith fits-file
                                HDU:                        temp variable for number of HDU in the ith fits-file
                                number_columns:             temp variable containing the total number of columns in the ith fits-file
                                dimension:                  temp variable containing the dimension of the data of the ith fits-file
                                NumDimensions:              number of dimensions
                                fitshead:                   header of the ith fits file
        """

        # Debug:
        # print ("filename = ", filename)
        # print ("HDU = ", HDU)


        ## initialize output parameters
        LowestDataPoint = 0
        HighestDataPoint = 0


        ## initialize working variables
        ok = 0
        length = []
        CRPIXList = []
        CRVALList = []
        CDELTList = []
        CROTList = []
        CUNITList = []
        ValueOfFirstPointList = []


        ## open fits-file using pyfits
        hdulist = pyfits.open(filename)

        # Debug:
        # print ("\t HDU = ", HDU)
        # print hdulist.info()


        ## save data (the number of columns in scidata is given by len(scidata))
        fitsdata = hdulist[HDU].data


        ## verify header, fix if necessary and save header
        # hdulist.verify('fix')
        fitshead = hdulist[HDU].header

        # Debug:
        # print ("\t fitsdata.shape = ",fitsdata.shape)


        ## set array containing dimension of each axis
        dimension = fitsdata.shape
        NumDimensions = len(dimension)


        ## transpose fitsdata array
        fitsdata = numpy.transpose(fitsdata)


        ## check dimension of fits file
        if (NumDimensions > 4):
            print ("\n\t Error in subroutine GetMinMaxFitsFile:")
            print ("\t\t Can not import fits-file with more than 4 dimensions.")
            print ("\t\t Please select other file and restart program.", end=' ')
            print ("\n\t\t Program aborted")
            return LowestDataPoint, HighestDataPoint


        ## read header information
        ## get number of axes
        try:
            NAXIS = pyfits.getval(filename, 'NAXIS', 0)
        except KeyError as err:
            NAXIS = 1

        # Debug:
        # print ("\n\nNAXIS = ", NAXIS)


        ## get scale parameter BSCALE
        try:
            BSCALE = pyfits.getval(filename, 'BSCALE', 0)
        except KeyError as err:
            BSCALE = 1

        # Debug:
        # print ("\n\nBSCALE = ", BSCALE)


        ## get zero parameter BZERO
        try:
            BZERO = pyfits.getval(filename, 'BZERO', 0)
        except KeyError as err:
            BZERO = 0.0

        # Debug:
        # print ("\n\nBZERO = ", BZERO)


        ## get unit parameter BUNIT
        try:
            BUNIT = pyfits.getval(filename, 'BUNIT', 0)
        except KeyError as err:
            BUNIT = ""

        # Debug:
        # print ("\n\nBUNIT = ", BUNIT)


        ## get rest frequency RESTFRQ
        try:
            RESTFRQ = pyfits.getval(filename, 'RESTFRQ', 0)
        except KeyError as err:
            RESTFRQ = 0

        # Debug:
        # print ("\n\nRESTFRQ = ", RESTFRQ)


        ## analyze header
        length = []
        NumDimensions = 0
        for i in range(NAXIS):                                                              ## loop over all dimensions
            try:
                lenAxis = pyfits.getval(filename, 'NAXIS' + str(i + 1), 0)                  ## read CRPIX
            except KeyError as err:
                lenAxis = 1
            if (lenAxis > 0 and NumDimensions == 0):                                        ## read only those entries with more than 1 point
                length.append(lenAxis)
                NumDimensions += 1


                ## get header parameters
                try:
                    CRPIX = pyfits.getval(filename, 'CRPIX' + str(i + 1), 0)                ## location of a reference point along axis 1
                except KeyError as err:
                    CRPIX = 1
                CRPIXList.append(CRPIX)
                try:
                    CRVAL = pyfits.getval(filename, 'CRVAL' + str(i + 1), 0)                ## the value of the coordinate at the reference point CRPIX1
                except KeyError as err:
                    CRVAL = 0
                CRVALList.append(CRVAL)
                try:
                    CDELT = pyfits.getval(filename, 'CDELT' + str(i + 1), 0)                ## partial derivative of the coordinate with respect to the
                except KeyError as err:                                                       ## pixel index, evaluated at the reference point CRPIX
                    CDELT = 1
                CDELTList.append(CDELT)
                try:
                    CROT = pyfits.getval(filename, 'CROT' + str(i + 1), 0)                  ## indicate a rotation from a standard coordinate system
                except KeyError as err:
                    CROT = 0
                CROTList.append(CROT)
                try:
                    CUNIT = pyfits.getval(filename, 'CUNIT' + str(i + 1), 0)                ## indicate the unit of the current axis
                except KeyError as err:
                    CUNIT = ""
                CUNITList.append(CUNIT)


                ## calculate value at first pixel
                LowestDataPoint = CRVAL - ((CRPIX - 1) * CDELT)
                HighestDataPoint = LowestDataPoint + ((lenAxis - 1) * CDELT)

                # Debug:
                # print ("CRPIX = ", CRPIX)
                # print ("CRVAL = ", CRVAL)
                # print ("CDELT = ", CDELT)
                # print ("CROT = ", CROT)
                # print ("CUNIT = ", CUNIT)
                # print ("LowestDataPoint = ", LowestDataPoint)
                # print ("HighestDataPoint = ", HighestDataPoint)


        ## close fits-file
        hdulist.close()


        ## define return parameters
        return LowestDataPoint, HighestDataPoint
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## load xml-file containing settings for the import of experimental-file(s)
    ##
    def xml(self, ObsXMLFileName, print_flag, XmlFileInOutput):
        """

        function xml:       load xml-file containing the settings for the import of experimental-file(s)

        input parameter:    ObsXMLFileName:             file name of the xml(experimental)-file including path
                            print_flag:                 flag if screen output is permitted (=true) or not
                            XmlFileInOutput:            path and file name of the xml description of the model
                                                        (necessary for cso to dat-file conversion, myXCLASS required)

        output parameter:   ok:                         status, if everything is ok: ok = 0
                            InFormat:                   format of the experimental files

        required parameter: NumberExpFiles:             number of experimental files
                            FileNamesExpFiles:          path and file name of each experimental file
                            ErrorY:                     error flag for each experimental file
                            NumberExpRanges:            number of selected experimental ranges
                            MinExpRange:                min of each experimental range
                            MaxExpRange:                max of each experimental range

                            NumberHeaderLines:          number of header lines for each dat-file
                            SeparatorColumns:           column separator for each dat-file
                            NumberColumnsX:             number of columns belonging to the x-positions in dat-files
                            NumberColumnsY:             number of columns belonging to the y-positions in dat-files

                            NumberHDU:                  number of HDU for each fits-file
                            FitsFileDimensions:         dimensions for each fit file


        working variables:  i:                          loop counter
                            j:                          loop counter
                            XmlFile:                    contains the whole contents of the xml-file
                            NumHeadLines:               temp variable for reading number of header lines
                            separator:                  temp variable for reading separator character
                            NumberX:                    temp variable for reading number of x columns
                            NumberY:                    temp variable for reading number of y columns
                            error:                      temp variable for reading error flag
                            NumberRange:                temp variable for reading number of ranges
                            MinRange:                   temp variable for reading min of ranges in file i
                            MaxRange:                   temp variable for reading max of ranges in file i
                            MinR:                       temp variable for reading min of range in range j in file i
                            MaxR:                       temp variable for reading max of range in range j in file i
                            printdoneflag:              flag for printing the word done at the end of the function
                            XmlFileCSO:                 path and file name of the xml-description of the model
                            CommandLine:                path and file name of the start script for myXCLASS
                            command_string:             sting containing the command line starting myXCLASS
        """

        # Debug:
        # print ("ObsXMLFileName = ", ObsXMLFileName)
        # print ("print_flag = ", print_flag)
        # print ("XmlFileInOutput = ", XmlFileInOutput)


        ## set status variable
        ok = 0
        printdoneflag = "true"


        ## define default settings
        XCLASSFlag = False                                                                  ## initialize XCLASS flag
        self.ImportFilter = ["automatic"]
        self.NumberExpFiles = 1
        self.ErrorY = ["no"]
        self.NumberExpRanges = [0]
        self.MinExpRange = [[-1]]
        self.MaxExpRange = [[-1]]
        self.NumberHeaderLines = [0]
        self.SeparatorColumns = [" "]
        self.NumberColumnsX = [1]
        self.NumberColumnsY = [1]
        self.NumberHDU = [0]
        self.FitsFileDimensions = [0]
        CommandLineParameters = " "
        ImportParameterSet = []                                                             ## set of all parameters used by the import of the
                                                                                            ## observational data:
                                                                                            ## First dimension describes the parameter names
                                                                                            ## Second dimension describes the parameter values
        ImportParameterNames = []                                                           ## help array
        ImportParameterValues = []                                                          ## help array



        ## is the defined file a xml, a fits or a dat file
        if (not ObsXMLFileName.endswith(".xml")):
            self.FileNamesExpFiles = [ObsXMLFileName]                                       ## define name of exp file


            ## identify format of obs. data file by analyzing the ending
            if (ObsXMLFileName.endswith(".dat") or ObsXMLFileName.endswith(".cso")):        ## obs. data file is an ASCII file or cso file
                InFormat = "dat"
            elif ObsXMLFileName.endswith(".fits"):                                          ## if filename ends with fits
                InFormat = "fits"
            # elif ObsXMLFileName.endswith(".???"):                                         ## if filename ends with ???
            #   InFormat = "???"
            self.ImportFilter = [InFormat]


            ## print summary to screen
            if (print_flag == "true"):
                print ("\n\t WARNING: The selected experimental file is not a xml-file")
                print ("\t\t        But a {0:s}-file. The following default settings for".format(InFormat))
                print ("\t\t        the import of experimental files are used:")
                print (" ")
                print ("\t\t        Number of files        = ",self.NumberExpFiles)
                if (InFormat in ["dat", "cso"]):
                    print ("\t\t        Number of header lines = ",self.NumberHeaderLines[0])
                    print ("\t\t        Column separator       = >>{0:s}<<".format(self.SeparatorColumns[0]))
                    print ("\t\t        Number of x-columns    = ",self.NumberColumnsX[0])
                    print ("\t\t        Number of y-columns    = ",self.NumberColumnsY[0])
                elif (InFormat == "fits"):
                    print ("\t\t        Number of HDU          = ",self.NumberHDU[0])
                # elif (InFormat == "???"):
                #   print ("\t\t        ???                    = ",self.???
                print ("\t\t        Error flag             = ",self.ErrorY[0])
                print ("\t\t        Number of ranges       = ",self.NumberExpRanges[0])
                if (self.NumberExpRanges[0] > 0):
                    print ("\t\t        Min of range           = ",self.MinExpRange[0][0])
                    print ("\t\t        Max of range           = ",self.MaxExpRange[0][0])
                print (" ")
            ok = 2
            return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet


        ## print what you do
        if (print_flag != "false"):
            print ("\t Open xml-file: " + chr(34) + ObsXMLFileName + chr(34))
            print ("\t Import settings .. ", end = "")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## open xml-file file and save content of file to variable XmlFile
        ## old:     XmlFile = MAGIXxmlPACKAGE.XmlObject(fileName = ObsXMLFileName)
        DataFile = XMLPackage.XMLRoutines(ObsXMLFileName)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if xml file contains XCLASS tags
        ObsIDStringV1 = "./file//"
        ObsIDStringV2 = "./Section/SubSection/file//"
        TagString = "TelescopeSize"
        XCLASSTest = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString)
        if (XCLASSTest is not None):
            if (len(XCLASSTest) > 0):
                XCLASSFlag = True

        # Debug:
        # print ("XCLASSFlag = ", XCLASSFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get number of experimental files
        self.NumberExpFiles = 0
        ## old:     self.NumberExpFiles = XmlFile.ExpFiles.NumberExpFiles.getValue()
        ObsIDStringV1 = "./"
        ObsIDStringV2 = "./Section/SubSection/"
        TagString = "NumberExpFiles"
        self.NumberExpFiles = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString)
        self.NumberExpFiles = int(self.NumberExpFiles)
        if (self.NumberExpFiles < 1):
            print ("\n\n\t WARNING:  Number of experimental files < 1:")
            print ("\t\t         Set number of experimental files to 1!")
            printdoneflag = "false"
            self.NumberExpFiles = 1
            ok = 2
            return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet
        ImportParameterNames.append("MAGIXImportNumberExpFiles")                            ## add MAGIXImportNumberExpFiles to list of import parameter
        ImportParameterValues.append(self.NumberExpFiles)                                   ## add number of exp. files to list of import parameter


        ## get path and file name of each experimental file
        self.ImportFilter = []
        self.FileNamesExpFiles = []
        self.NumberHDU = []
        self.FitsFileDimensions = []
        self.NumberHeaderLines = []
        self.FitsHeader = []
        self.SeparatorColumns = []
        self.NumberColumnsX = []
        self.NumberColumnsY = []
        self.ErrorY = []
        self.NumberExpRanges = []
        self.MinExpRange = []
        self.MaxExpRange = []
        csoVariableList =[]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## only for XCLASS: read in parameters which are defined only once
        if (XCLASSFlag):
            for LocalTagElement in self.TagLists:                                           ## loop over all XCLASS tag settings
                CurrentLocation = LocalTagElement[2]
                if (CurrentLocation == "o"):                                                ## select only those settings which are defined only once
                    MAGIXImportName = LocalTagElement[0]
                    XMLTagName = LocalTagElement[1]
                    DefaultValue = LocalTagElement[3]
                    TagFormat = LocalTagElement[4]


                    ## read tag value
                    TagString = XMLTagName
                    TagValue = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString, FormatString = TagFormat, \
                                                             DefaultValue = DefaultValue)


                    ## store tag value
                    if (TagValue is not None):
                        if (TagFormat == "bool"):
                            if (TagValue):
                                TagValue = 1.0
                            else:
                                TagValue = 0.0
                        if (XMLTagName.lower() in ["isotopologues", "iso_flag"]):
                            if (not "MAGIXImportIsotopologues" in ImportParameterNames):
                                ImportParameterNames.append(MAGIXImportName)
                                ImportParameterValues.append(TagValue)
                        else:
                            ImportParameterNames.append(MAGIXImportName)
                            ImportParameterValues.append(TagValue)

                    # Debug:
                    # print ("MAGIXImportName, TagValue = ", MAGIXImportName, TagValue)
                    # print ("ImportParameterNames = ", ImportParameterNames)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## start loop over each exp files
        ##
        ##
        ## the following tags are used for all formats:
        ##
        ## - FileNamesExpFiles
        ## - ImportFilter
        ## - NumberExpRanges
        ##
        ##
        for i in range(self.NumberExpFiles):


            ## define XML string extension for line
            ObsIDStringV1 = "./file[" + str(i + 1) + "]/"
            ObsIDStringV2 = "./Section/SubSection/file[" + str(i + 1) + "]/"


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get path and file name of each experimental-file and set InputFormat variable
            ## old:     filen = XmlFile.ExpFiles.file[i].FileNamesExpFiles.getValue()
            TagString = "FileNamesExpFiles"
            filen = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString)
            if (not(os.path.exists(filen))):
                print ("\n\n\t ERROR in function LoadExpFile.LoadExp.xml:")
                print ("\t\t The experimental-file ",filen)
                print ("\t\t given in the xml-file does not exist!")
                print ("\t\t Please correct xml-file and restart program!\n")
                ok = 1
                return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet
            ImportParameterNames.append("MAGIXImportExpFileNames")                          ## add MAGIXImportExpFileNames to list of import parameter
            ImportParameterValues.append(filen)                                             ## add name of exp. file to list of import parameter

            # Debug:
            # print ("filen = ", filen)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get filter settings for file import
            ## old:     FilterSetting = XmlFile.ExpFiles.file[i].ImportFilter.getValue()
            TagString = "ImportFilter"
            FilterSetting = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString, FormatString = "lower", DefaultValue = "ascii")

            # Debug:
            # print ("FilterSetting = ", FilterSetting)


            ## analyze automatic filter setting
            if (FilterSetting in ["automatic", "auto"]):
                if filen.endswith(".dat"):                                                  ## if filename ends with .dat
                    FilterSetting = "ascii"
                elif filen.endswith(".fits"):                                               ## if filename ends with .fits
                    FilterSetting = "fits"
                elif filen.endswith(".cso"):                                                ## if filename ends with .cso
                    FilterSetting = "cso"
                else:
                    print ("\n\n\t ERROR in function LoadExpFile.LoadExp.xml:")
                    print ("\t\t Can not identify the file format for the experimental-file ",filen)
                    print ("\t\t given in the xml-file!")
                    print ("\t\t Please correct xml-file and restart program!\n")
                    ok = 1
                    return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet


            ## analyze filter setting
            if (FilterSetting == "xclassascii"):
                FilterSetting = "xclassascii"
            elif (FilterSetting == "xclassfits"):
                FilterSetting = "xclassfits"
            elif (FilterSetting == "fits"):
                FilterSetting = "fits"
            elif (FilterSetting in ["dat", "ascii"]):
                FilterSetting = "ascii"
            elif (FilterSetting == "cso"):
                FilterSetting = "cso"
            else:
                FilterSetting = "ascii"
            self.ImportFilter.append(FilterSetting)
            self.FileNamesExpFiles.append(filen)

            # Debug:
            # print ("FilterSetting = ", FilterSetting)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## for xclassfits file read in exp data file to determine min and max frequency
            if (FilterSetting == "xclassfits"):
                LowestDataPoint, HighestDataPoint = self.GetMinMaxFitsFile(filen, 0)

                # Debug:
                # print ("LowestDataPoint, HighestDataPoint = ", LowestDataPoint, HighestDataPoint)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get number of selected ranges
            NumberRange = 1
            ## old:     NumberRange = XmlFile.ExpFiles.file[i].NumberExpRanges.getValue()
            TagString = "NumberExpRanges"
            NumberRange = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString, FormatString = "int", DefaultValue = 0)
            if (NumberRange <= 0):
                if ((FilterSetting in ["cso", "xclassascii", "xclassfits"]) and NumberRange == 0):
                    print ("\n\n\t WARNING:  Number of ranges in the experimental-files <= 0:")
                    print ("\t\t         Set number of ranges to 1 !")
                    printdoneflag = "false"
                    NumberRange = 0
                elif (NumberRange < 0):
                    print ("\n\n\t WARNING:  Number of ranges in the experimental-files < 0:")
                    print ("\t\t         Set number of ranges to 0 !")
                    printdoneflag = "false"
                    NumberRange = 0
            self.NumberExpRanges.append(NumberRange)
            ImportParameterNames.append("MAGIXImportNumberRange")                           ## add MAGIXImportNumberRange to list of import parameter
            ImportParameterValues.append(NumberRange)                                       ## add number of ranges to list of import parameter


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get min and max of each selected range only for ASCII and FITS files
            if (NumberRange > 0):
                MinRange = []
                MaxRange = []
                for LocalRangeID in range(NumberRange):                                     ## loop over frequency ranges


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## non-XCLASS part
                    if (not XCLASSFlag):


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## get min of the LocalRangeID-th range
                        MinR = 0
                        ## old:     MinR = XmlFile.ExpFiles.file[i].MinExpRange[LocalRangeID].getValue()
                        TagString = "MinExpRange[" + str(LocalRangeID + 1) + "]"
                        MinR = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString)
                        line = str(MinR).split(',')
                        MinRange.append(list(map(float, line)))
                        csoVariableList.append(list(map(float, line)))                      ## add start of range to cso variable list
                        ImportParameterNames.append("MAGIXImportMinRange")
                        ImportParameterValues.append(line)


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## get max of the LocalRangeID-th range
                        RangeXMLStringV1 = ObsIDStringV1
                        RangeXMLStringV2 = ObsIDStringV2
                        MaxR = 0
                        ## old:     MaxR = XmlFile.ExpFiles.file[i].MaxExpRange[LocalRangeID].getValue()
                        TagString = "MaxExpRange[" + str(LocalRangeID + 1) + "]"
                        MaxR = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString)
                        line = str(MaxR).split(',')
                        MaxRange.append(list(map(float, line)))
                        csoVariableList.append(list(map(float, line)))                      ## add end of range to cso variable list
                        ImportParameterNames.append("MAGIXImportMaxRange")
                        ImportParameterValues.append(line)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## only for XCLASS: read in parameters which are defined only per obs. data file
                    else:
                        RangeXMLStringV1 = ObsIDStringV1 + "FrequencyRange[" + str(LocalRangeID + 1) + "]"
                        RangeXMLStringV2 = ObsIDStringV2 + "FrequencyRange[" + str(LocalRangeID + 1) + "]"
                        for LocalTagElement in self.TagLists:                               ## loop over all XCLASS tag settings
                            CurrentLocation = LocalTagElement[2]
                            if (CurrentLocation == "r"):                                    ## select tags which are defined for each range
                                MAGIXImportName = LocalTagElement[0]
                                XMLTagName = LocalTagElement[1]
                                DefaultValue = LocalTagElement[3]
                                TagFormat = LocalTagElement[4]

                                # Debug:
                                # print ("\nMAGIXImportName = ", MAGIXImportName)
                                # print ("XMLTagName = ", XMLTagName)
                                # print ("DefaultValue = ", DefaultValue)
                                # print ("TagFormat = ", TagFormat)


                                ## read tag value
                                TagString = XMLTagName
                                TagValue = DataFile.GetSingleObsTagValue(RangeXMLStringV1, RangeXMLStringV2, TagString, FormatString = TagFormat, \
                                                                         DefaultValue = DefaultValue)


                                ## special handling of tags "MinExpRange" and "MaxExpRange"
                                if (MAGIXImportName in ["MAGIXImportStartFrequency", "MAGIXImportEndFrequency"]):
                                    if (FilterSetting == "xclassfits"):
                                        SplittedElement = str(TagValue).split(",")
                                        TagValue = float(SplittedElement[-1])
                                        if (MAGIXImportName == "MAGIXImportStartFrequency"):
                                            MinRange.append(list(map(float, SplittedElement)))
                                        else:
                                            MaxRange.append(list(map(float, SplittedElement)))
                                    else:
                                        if (MAGIXImportName == "MAGIXImportStartFrequency"):
                                            MinRange.append(TagValue)
                                        else:
                                            MaxRange.append(TagValue)

                                    # Debug:
                                    # print ("MinRange = ", MinRange)
                                    # print ("MaxRange = ", MaxRange)


                                ## store tag value
                                if (TagValue is not None):
                                    if (TagFormat == "bool"):
                                        if (TagValue):
                                            TagValue = 1.0
                                        else:
                                            TagValue = 0.0
                                    ImportParameterNames.append(MAGIXImportName)
                                    ImportParameterValues.append(TagValue)

                                    # Debug:
                                    # print ("TagValue = ", TagValue)


                ##------------------------------------------------------------------------------------------------------------------------------------
                ## save to global variable
                self.MinExpRange.append(MinRange)
                self.MaxExpRange.append(MaxRange)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## only for XCLASS: read in parameters which are defined only per obs. data file
            if (XCLASSFlag):
                for LocalTagElement in self.TagLists:                                       ## loop over all XCLASS tag settings
                    CurrentLocation = LocalTagElement[2]
                    if (CurrentLocation == "f"):                                            ## select tags which are defined only once per obs. data file
                        MAGIXImportName = LocalTagElement[0]
                        XMLTagName = LocalTagElement[1]
                        DefaultValue = LocalTagElement[3]
                        TagFormat = LocalTagElement[4]


                        ## read tag value
                        TagString = XMLTagName
                        TagValue = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString, FormatString = TagFormat, \
                                                                 DefaultValue = DefaultValue)

                        ## store tag value
                        if (TagValue is not None):
                            if (TagFormat == "bool"):
                                if (TagValue):
                                    TagValue = 1.0
                                else:
                                    TagValue = 0.0
                            ImportParameterNames.append(MAGIXImportName)
                            ImportParameterValues.append(TagValue)

                        # Debug:
                        # print ("TagValue = ", TagValue)


            ##********************************************************************************************************************************************
            ## for ascii and xclassascii files only
            if (FilterSetting in ["ascii", "xclassascii"]):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get number of columns belonging to the y-positions for each experimental-file
                ## old:     error = XmlFile.ExpFiles.file[i].ErrorY.getValue()
                TagString = "ErrorY"
                LocalErrorFlag = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString)
                LocalErrorFlag = CheckBool(LocalErrorFlag)
                if (LocalErrorFlag):
                    error = "yes"
                else:
                    error = "no"
                self.ErrorY.append(error)
                ImportParameterNames.append("MAGIXImportErrorY")
                ImportParameterValues.append(error)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get number of header lines for each experimental-file
                ## old:     NumberHeaderLines = XmlFile.ExpFiles.file[i].NumberHeaderLines.getValue()
                TagString = "NumberHeaderLines"
                NumHeadLines = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString, FormatString = "int", DefaultValue = 0)
                if (NumHeadLines < 0):
                    print ("\n\n\t WARNING:  Number of header lines in the experimental-files < 0:")
                    print ("\t\t         Set number of header lines to 0 !")
                    printdoneflag = "false"
                    self.NumberHeaderLines = 0
                self.NumberHeaderLines.append(NumHeadLines)
                ImportParameterNames.append("MAGIXImportNumHeadLines")
                ImportParameterValues.append(NumHeadLines)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get separator for each experimental-file
                ## old:     SeparatorColumns = XmlFile.ExpFiles.file[i].SeparatorColumns.getValue()
                TagString = "SeparatorColumns"
                separator = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString, FormatString = "raw", DefaultValue = "")
                self.SeparatorColumns.append(separator)
                ImportParameterNames.append("MAGIXImportSeparatorColumns")
                ImportParameterValues.append(separator)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## the following settings are used only for ASCII files
                if (FilterSetting == "ascii"):


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## get number of columns belonging to the x-positions for each experimental-file
                    ## old:     NumberX = XmlFile.ExpFiles.file[i].NumberColumnsX.getValue()
                    TagString = "NumberColumnsX"
                    NumberX = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString, FormatString = "int", DefaultValue = 1)
                    if (NumberX < 0):
                        print ("\n\n\t WARNING:  Number of x-columns in the experimental-files < 0:")
                        print ("\t\t         Set number of x-columns to 1 !")
                        printdoneflag = "false"
                        NumberX = 1
                    self.NumberColumnsX.append(NumberX)
                    ImportParameterNames.append("MAGIXImportNumberColumnsX")
                    ImportParameterValues.append(NumberX)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## get number of columns belonging to the y-positions for each experimental-file
                    ## old:     NumberY = XmlFile.ExpFiles.file[i].NumberColumnsY.getValue()
                    TagString = "NumberColumnsY"
                    NumberY = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString, FormatString = "int", DefaultValue = 1)
                    if (NumberY < 0):
                        print ("\n\n\t WARNING:  Number of y-columns in the experimental-files < 0:")
                        print ("\t\t         Set number of y-columns to 1 !")
                        printdoneflag = "false"
                        NumberY = 1
                    self.NumberColumnsY.append(NumberY)
                    ImportParameterNames.append("MAGIXImportNumberColumnsY")
                    ImportParameterValues.append(NumberY)
                else:
                    self.NumberColumnsX.append(1)                                           ## set number of x-columns to 1
                    self.NumberColumnsY.append(1)                                           ## set number of y-columns to 1


                ## set default hdu number
                self.NumberHDU.append(0)                                                    ## set hdu to 0


            ##********************************************************************************************************************************************
            ## for fits-files only
            elif (FilterSetting in ["fits", "xclassfits"]):
                self.NumberHeaderLines.append(0)                                            ## set number of header lines to zero
                self.SeparatorColumns.append(" ")                                           ## set separator character to blank
                self.ErrorY.append("no")                                                    ## set error flag to false
                self.NumberColumnsX.append(1)                                               ## set number of x-columns to 1
                self.NumberColumnsY.append(1)                                               ## set number of y-columns to 1


                ## get number of HDU
                ## old:     NumHDU = XmlFile.ExpFiles.file[i].NumberHDU.getValue()
                TagString = "NumberHDU"
                NumHDU = DataFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString)
                try:
                    NumHDU = int(NumHDU)
                except:
                    NumHDU = 1
                self.NumberHDU.append(NumHDU)
                ImportParameterNames.append("MAGIXImportNumHDU")
                ImportParameterValues.append(NumHDU)


            ##********************************************************************************************************************************************
            ## transform cso-file to dat-file (myXCLASS for CLASS is required)
            ## get path of the myXCLASS start script from the xml-description
            elif (FilterSetting == "cso"):
                self.NumberHeaderLines.append(0)                                            ## set number of header lines to zero
                self.SeparatorColumns.append(" ")                                           ## set separator character to blank
                self.ErrorY.append("no")                                                    ## set error flag to false
                self.NumberColumnsX.append(1)                                               ## set number of x-columns to 1
                self.NumberColumnsY.append(1)                                               ## set number of y-columns to 1
                self.NumberHDU.append(0)                                                    ## set hdu to 0
                if not(os.path.exists(XmlFileInOutput)):                                    ## does path and file described in XmlFileInOutput exist?
                    print ("\n\t Error in subroutine read_control")
                    print ("\t\t The conversion of cso to dat-file requires myXCLASS")
                    print ("\t\t and the path to the myXCLASS start script within the")
                    print ("\t\t xml-description of the model does not exist.")
                    print ("\t\t Please correct the command line description in the")
                    print ("\t\t xml-file{0:s}.".format(XmlFileInOutput))
                    print (" ")
                    ok = 1
                    sys.exit(0)


                ## define command string
                ## old:     XmlFileCSO = MAGIXxmlPACKAGE.XmlObject(fileName = XmlFileInOutput)
                DataXCLASSFile = XMLPackage.XMLRoutines(XmlFileInOutput)


                ## get command line of the myXCLASS start script
                ## old:     PathStartScript = XmlFileCSO.InOutputFile.ModelProgramCall.PathStartScript.getValue()
                TagString = "PathStartScript"
                CommandLine = DataXCLASSFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString)


                ## get input line of the myXCLASS start script
                ## old:     InputName = XmlFileCSO.InOutputFile.InputFile[0].InputFileName.getValue()
                TagString = "InputFileName"
                filen = DataXCLASSFile.GetSingleObsTagValue(ObsIDStringV1, ObsIDStringV2, TagString)
                filenew = filen + ".dat"


                ## get pid of current process
                pid = os.getpid()


                ## get current working directory in magix
                CurrentWorkingDir = os.getcwd()


                ## start cso-transformation script
                command_string_org = CommandLine
                csoFilenameRangeAll = []
                for j in range(NumberRange):                                            ## loop of all ranges
                    MinR = MinRange[j]                                                  ## get start frequency
                    MaxR = MaxRange[j]                                                  ## get end frequency
                    stepFre = stepsizeRange[j]                                          ## get resolution (stepsize)


                    ## define new ascii file name if more than one range is selected
                    if (NumberRange > 1):
                        csoFilenameRange = filen + str(j) + ".dat"
                        csoFilenameRangeAll.append(csoFilenameRange)
                    else:
                        csoFilenameRange = filen + ".dat"


                    ## define command string parameters
                    CommandLineParameters = " data " + str(pid) + " " + str(MinR) + " " + str(MaxR) + " " + str(stepFre) + " " + filen + " " \
                                            + csoFilenameRange + " " + CurrentWorkingDir

                    # Debug:
                    # print (">>>>>',CommandLineParameters,'<<<<')


                    ## save new filename
                    #filen = filenew
                    self.FileNamesExpFiles[i] = filenew


                    ## define command sting
                    command_string = command_string_org + CommandLineParameters

                    # Debug:
                    # print ("command_string = ", command_string)
                    # print ("csoFilenameRange = ", csoFilenameRange)


                    ## execute cso-transformation script
                    os.system(command_string)


                    ## check, if file conversion was successfully
                    if not (os.path.exists(csoFilenameRange)):
                        print ("\n\n Error in subroutine LoadExpFile.xml:")
                        print ("\t\t The conversion of the xclass file to ASCII file format was not successfully.")
                        print ("\n\t\t Please take a look at the .output-file and restart MAGIX.\n\n")
                        sys.exit(0)


                ## combine the different ascii-file for each range to one big ascii file
                if (NumberRange > 1):
                    command_string = "rm -f " + filenew
                    os.system(command_string)                                               ## remove old cso-dat file

                    command_string = "cat"
                    for j in range(NumberRange):                                            ## loop of all ranges
                        csoFilenameRange = csoFilenameRangeAll[j]
                        command_string += " " + csoFilenameRange
                    command_string += " >> " + filenew
                    os.system(command_string)                                               ## append all range files to one final ascii-file


                    ## remove range files
                    for j in range(NumberRange):                                            ## loop of all ranges
                        csoFilenameRange = csoFilenameRangeAll[j]
                        command_string = "rm -f " + csoFilenameRange
                        os.system(command_string)                                           ## remove jth range file

        # Debug:
        # print ("ImportParameterNames = ", ImportParameterNames)
        # print ("ImportParameterValues = ", ImportParameterValues)
        # print ("self.InputFormat = ", self.InputFormat)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print results
        if (print_flag != "false"):
            print ("done!")
            print ("\n\t Numberof data files: {:d} ".format(self.NumberExpFiles))
            for i in range(self.NumberExpFiles):
                print ("\n\t Data file number: ", i + 1)
                print ("\t\t Path and name of data file: {:s}".format(chr(34) + self.FileNamesExpFiles[i] + chr(34)))
                print ("\t\t Number of ranges: {:d}".format(self.NumberExpRanges[i]))
                if (self.NumberExpRanges[i] > 0):
                    for j in range(self.NumberExpRanges[i]):
                        print ("\n\t\t\t Start of range {:d}:      {:s}".format(j + 1, str(self.MinExpRange[i][j])))
                        print ("\t\t\t End of range {:d}:        {:s}".format(j + 1, str(self.MaxExpRange[i][j])))
                if (self.ImportFilter[i]  in ["fits", "xclassfits"]):
                    print ("\n\t\t Number of HDU:            {:d}".format(self.NumberHDU[i]))
                elif (self.ImportFilter[i] in ["ascii", "xclassascii", "cso", "dat"]):
                    print ("\n\t\t ErrorY flag:                        {:s}".format(str(self.ErrorY[i])))
                    print ("\t\t Number of header lines:             {:d}".format(self.NumberHeaderLines[i]))
                    print ("\t\t string used to separate columns:  >>{:s}<<".format(self.SeparatorColumns[i]))
                    print ("\t\t Number of x-Columns:                {:d}".format(self.NumberColumnsX[i]))
                    print ("\t\t Number of y-Columns:                {:d}".format(self.NumberColumnsY[i]))
                # elif (self.ImportFilter[i] in ["???"]):
                #   print ("\t\t ")
            print (" ")


        ## combine import parameter names and values to final output parameter set
        ImportParameterSet.append(ImportParameterNames)
        ImportParameterSet.append(ImportParameterValues)


        ## define return parameters
        return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ##
    ##                                              Load data
    ##
    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## load contents of ascii-file(s)
    ##
    def LoadFile(self, print_flag):
        """

        function LoadDat:       load ascii files containing experimental data using the setting given in the xml-file

        input parameters:       print_flag:                 flag for printing message to screen

        required parameters:    NumberExpFiles:             number of ascii-files
                                FileNamesExpFiles:          path and file name of each ascii file
                                NumberHeaderLines:          number of header lines for each ascii-file
                                SeparatorColumns:           column separator for each ascii-file
                                NumberColumnsX:             number of columns belonging to the x-positions
                                NumberColumnsY:             number of columns belonging to the y-positions
                                NumberExpRanges:            number of selected experimental ranges
                                MinExpRange:                min of each experimental range
                                MaxExpRange:                max of each experimental range

        output parameters:      ok:                         status of subroutine
                                NumberFiles:                number of ascii-files
                                LengthExpFileDat:           number of lines in the experimental ascii-file
                                ExpDataX:                   experimental data x column
                                ExpDataY:                   experimental data y column
                                ExpDataError:               error of experimental data y column

        working variables:      i:                          loop counter
                                k:                          loop counter
                                FileFormat:                 format of the current data file


        IMPORTANT:      - The function xml() has to be called before the call of LoadDat()
        """

        # Debug:
        # print ("print_flag = ", print_flag)


        ## set status to 0
        ok = 0


        ## read experimental data of each ascii-file and define output variables
        NumberFiles = self.NumberExpFiles
        LengthExpFileDat = []
        ExpDataX = []
        ExpDataY = []
        ExpDataError = []
        CounterASCIIFile = (-1)
        CounterFitsFile = (-1)
        for i in range(self.NumberExpFiles):


            ## read in ascii files
            FileFormat = self.ImportFilter[i]
            if (FileFormat in ["ascii", "xclassascii", "cso", "dat"]):
                CounterASCIIFile += 1
                ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error = self.LoadDat(i, CounterASCIIFile, print_flag)
            elif (FileFormat in ["fits", "xclassfits"]):
                CounterFitsFile += 1
                ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error = self.LoadFits(i, CounterFitsFile, print_flag)
            else:
                print ("\n\n\t ERROR in subroutine LoadFile:")
                print ("\t\t Can not import the experimental-file ",self.FileNamesExpFiles[i])
                print ("\t\t LoadFilegiven in the xml-file!")
                print ("\n\t\t Unknown file format ", FileFormat)
                print ("\n\t\t Please correct xml-file and restart program!\n")
                ok = 1
                return ok, NumberFiles, LengthExpFileDat, ExpDataX, ExpDataY, ExpDataError

            # Debug:
            # numpy.savetxt("obs_freq__{:d}.dat".format(i + 1), exp_data_x)


            ## copy results to output variables
            ExpDataX.append(exp_data_x)
            ExpDataY.append(exp_data_y)
            ExpDataError.append(exp_data_error)
            LengthExpFileDat.append(lengthexpdata)


        ## print results
        if (print_flag != "false"):
            print ("")
            for i in range(NumberFiles):
                print ("\t Data file Nr.: ", i + 1)
                print ("\t\t Number of data points:  {:d}".format(LengthExpFileDat[i]))
                print ("\t\t x column of first data point: {:s}".format(str(ExpDataX[i][0])))
                print ("\t\t x-column of last data point:  {:s}".format(str(ExpDataX[i][LengthExpFileDat[i]])))
                print ("\t\t y column of first data point: {:s}".format(str(ExpDataY[i][0])))
                print ("\t\t y column of last data point:  {:s}".format(str(ExpDataY[i][LengthExpFileDat[i]])))
                print ("")


        ## define return variables
        return ok, NumberFiles, LengthExpFileDat, ExpDataX, ExpDataY, ExpDataError
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## load contents of ascii-file(s)
    ##
    def LoadDat(self, i, CounterASCIIFile, print_flag):
        """

        function LoadDat:       load ascii files containing experimental data using the setting given in the xml-file

        input parameters:       i:                          index of experimental data file
                                CounterASCIIFile:           counter for ascii files
                                print_flag:                 flag for printing message to screen

        required parameters:    FileNamesExpFiles:          path and file name of each ascii file
                                NumberHeaderLines:          number of header lines for each ascii-file
                                SeparatorColumns:           column separator for each ascii-file
                                NumberColumnsX:             number of columns belonging to the x-positions
                                NumberColumnsY:             number of columns belonging to the y-positions
                                NumberExpRanges:            number of selected experimental ranges
                                MinExpRange:                min of each experimental range
                                MaxExpRange:                max of each experimental range

        output parameters:      ok:                         status of subroutine
                                lengthexpdata:              number of lines in the experimental ascii-file
                                exp_data_x:                 experimental data x column
                                exp_data_y:                 experimental data y column
                                exp_data_error:             error of experimental data y column

        working variables:      k:                          loop counter
                                filename:                   temp variable for file name of the ith ascii-file
                                number_header_lines:        temp variable for number of header lines in the ith ascii-file
                                seperator:                  temp variable for column separator in the ith ascii-file
                                data:                       temp variable containing the whole ith ascii-file
                                length_exp_file:            temp variable containing the total number of lines in the ith ascii-file (without header lines)
                                number_columns:             temp variable containing the total number of columns in the ith ascii-file
                                NumberX:                    temp variable containing the number of x-columns in the ith ascii-file
                                NumberY:                    temp variable containing the number of y-columns in the ith ascii-file
                                error:                      temp variable containing the error flag
                                NumberRange:                temp variable containing the number of ranges in the ith ascii-file
                                xmn:                        array containing all lower limits of the ranges of the ith ascii-file
                                xmx:                        array containing all upper limits of the ranges of the ith ascii-file
                                exp_data_x:                 temp variable containing experimental data x column of the ith ascii-file
                                exp_data_y:                 temp variable containing experimental data y column of the ith ascii-file
                                xcolumn:                    temp variable containing the x-columns of the kth line in the ith ascii-file
                                ycolumn:                    temp variable containing the y-columns of the lth line in the ith ascii-file
                                errorcolumn:                temp variable containing the error-columns of the lth line in the ith ascii-file
                                copyflag:                   flag for extracting experimental data within selected ranges in the ith ascii-file
                                copyflagcolumn:             flag for extracting experimental data within selected ranges in the ith ascii-file
                                element:                    flag for extracting experimental data within selected ranges in the ith ascii-file


        IMPORTANT:      - The function xml() has to be called before the call of LoadDat()
        """

        # Debug:
        # print ("i = ", i)
        # print ("CounterASCIIFile = ", CounterASCIIFile)
        # print ("print_flag = ", print_flag)


        ## set status to 0
        ok = 0


        ## reset output variables
        lengthexpdata = -1
        exp_data_x = []
        exp_data_y = []
        exp_data_error = []


        ## set working variables for the ith ascii-file
        filename = self.FileNamesExpFiles[i]
        number_header_lines = self.NumberHeaderLines[CounterASCIIFile]
        seperator = self.SeparatorColumns[CounterASCIIFile]
        NumberX = self.NumberColumnsX[CounterASCIIFile]
        NumberY = self.NumberColumnsY[CounterASCIIFile]
        error = self.ErrorY[CounterASCIIFile]
        NumberRange = self.NumberExpRanges[i]
        if (NumberRange > 0):
            xmn = self.MinExpRange[i]
            xmx = self.MaxExpRange[i]


        ## print what you do
        if (print_flag != "false"):
            print ("\t Reading ASCII-file " + filename + " ..", end=' ')


        ## import ascii-file
        try:
            data = numpy.loadtxt(filename, delimiter = seperator, skiprows = number_header_lines)
        except:
            data = numpy.loadtxt(filename, skiprows = number_header_lines)

        # Debug:
        # print ("\n\ndata[1] = ",data[1])
        # print ("data.shape = ",data.shape)
        # print ("len(data.shape) = ", len(data.shape))


        ## remove rows which contain at least one NaN
        ## taken from https://stackoverflow.com/questions/11453141/how-to-remove-all-rows-in-a-numpy-ndarray-that-contain-non-numeric-values
        if (len(data.shape) > 1):
            data = data[~numpy.isnan(data).any(axis = 1)]


        ## check if range definition correct
        if (NumberRange > 0):
            if (len(xmn) != NumberRange):
                print ("\n\n\t ERROR in function LoadExpFile.LoadDat:")
                print ("\t\t The dimension of the definition of the content of ")
                print ("\t\t the <MinExpRange></MinExpRange> tag in the xml-file")
                print ("\t\t for the import settings does not coincident with the")
                print ("\t\t dimension of the loaded ascii-file!")
                print ("\t\t Please correct the xml-file and restart program!\n")
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error

            elif (len(xmx) != NumberRange):
                print ("\n\n\t ERROR in function LoadExpFile.LoadDat:")
                print ("\t\t The dimension of the definition of the content of ")
                print ("\t\t the <MaxExpRange></MaxExpRange> tag in the xml-file")
                print ("\t\t for the import settings does not coincident with the")
                print ("\t\t dimension of the loaded ascii-file!")
                print ("\t\t Please correct the xml-file and restart program!\n")
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error


        ## get number of lines and columns of the ith ascii-file and check consistency with xml settings
        length_exp_file = data.shape[0]
        if (len(data.shape) > 1):
            number_columns = data.shape[1]
        else:
            number_columns = 2
        if (NumberX + NumberY > number_columns):
            print ("\n\n\t ERROR in function LoadExpFile.LoadExp.LoadDat:")
            print ("\t\t The number of total number of columns in the")
            print ("\t\t ascii-file ",filename)
            print ("\t\t is less than the number of x-columns plus number of y-columns")
            print ("\t\t Please correct xml-file and restart program!\n")
            ok = 1
            return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error


        ## extract experimental data within ranges if NumberRange > 0
        if (NumberRange > 0 and len(data.shape) > 1):
            for k in range(length_exp_file):
                xcolumn = data[k,0:NumberX]
                copyflag = False                                                            ## set copyflag to false
                for j in range(NumberRange):
                    if (NumberX == 1):                                                      ## is there only one column for the x points?
                        if (xcolumn >= xmn[j]):
                            if (xcolumn <= xmx[j]):
                                copyflag = True
                    else:                                                                   ## no? Continue?
                        copyflagcolumn = 0
                        for klm in range(len(xcolumn)):
                            if (xcolumn[klm] >= xmn[j][klm]):
                                if (xcolumn[klm] <=xmx[j][klm]):
                                    copyflagcolumn += 1
                        if (copyflagcolumn == NumberX):
                            copyflag = True
                    if (copyflag):                                                          ## if experimental x point is within a given range ..
                        lengthexpdata += 1                                                  ## increase length counter
                        exp_data_x.append(xcolumn)                                          ## append current x point
                        exp_data_y.append(data[k,NumberX:NumberX + NumberY])

                        # Debug:
                        # print (">>',k,NumberX,NumberX + NumberY)
                        # print (">>',data[k,NumberX:NumberX + NumberY])
                        #    sys.exit(0)

                        if (error == "yes"):                                                ## if error flag is set to yes get experimental error
                            exp_data_error.append(data[k,NumberX + NumberY:NumberX + NumberY + NumberY])
                        else:                                                               ## otherwise, set error of experimental files to zero
                            exp_data_error.append(data[k,NumberX:NumberX + NumberY] * 0)
                        break


            ## warning if there is no data within the range(s)
            if (lengthexpdata == (-1) and len(data.shape) > 1):
                print ("\n\n\t ERROR in function LoadExpFile.LoadExp.LoadDat:")
                print ("\t\t There is no data within the given ranges!")
                print ("\t\t Please correct xml-file and restart program!\n")
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error

        else:
            lengthexpdata = length_exp_file - 1                                             ## get number of x-columns
            if (len(data.shape) > 1):
                exp_data_x = data[:,0:NumberX]                                              ## get x-columns
                exp_data_y = data[:,NumberX:NumberX + NumberY]                              ## get y-columns
                if (error == "yes"):                                                        ## if error flag is set to yes get experimental error
                    exp_data_error = data[:,NumberX + NumberY:NumberX + NumberY + NumberY]
                else:                                                                       ## otherwise, define error array and set error to zero
                    exp_data_error = data[:,NumberX:NumberX + NumberY] * 0
            else:
                lengthexpdata = 0
                exp_data_x = []
                exp_data_y = []
                exp_data_error = []
                exp_data_x.append(data[0:NumberX])                                          ## get x-columns
                exp_data_y.append(data[NumberX:NumberX + NumberY])                          ## get y-columns
                if (error == "yes"):                                                        ## if error flag is set to yes get experimental error
                    exp_data_error.append(data[NumberX + NumberY:NumberX + NumberY + NumberY])
                else:                                                                       ## otherwise, define error array and set error to zero
                    exp_data_error.append(data[NumberX:NumberX + NumberY] * 0)


            ## warning if there is no data within the range(s)
            if (lengthexpdata == 0 and len(data.shape) > 1):
                print ("\n\n\t ERROR in function LoadExpFile.LoadExp.LoadDat:")
                print ("\t\t There is no data within the given ranges!")
                print ("\t\t Please correct xml-file and restart program!\n")
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error


        # Debug:
        # print ("\n\nlengthexpdata = ",lengthexpdata)
        # print ("exp_data_x[0] = ",exp_data_x[0])
        # print ("exp_data_x[lengthexpdata] = ",exp_data_x[lengthexpdata])
        # print ("exp_data_y[0] = ",exp_data_y[0])
        # print ("exp_data_y[0][lengthexpdata] = ",exp_data_y[lengthexpdata])
        # print (">>>',lengthexpdata,len(exp_data_x))
        #    sys.exit(0)


        ## print that you are finished
        if (print_flag != "false"):
            print ("done!")


        ## define return variables
        return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## load contents of fits-file(s)
    ##
    def LoadFits(self, i, CounterFitsFile, print_flag):
        """

        function fits:          load .fits files containing experimental data using the setting given in the xml-file

        input parameters:       i:                          index of experimental data file
                                CounterFitsFile:            counter for fits files
                                print_flag:                 flag for printing message to screen

        required parameters:    NumberExpFiles:             number of fits-files
                                FileNamesExpFiles:          path and file name of each fits-file
                                NumberHDU:                  number of HDU for each fits-file
                                NumberExpRanges:            number of selected experimental ranges
                                MinExpRange:                min of each experimental range
                                MaxExpRange:                max of each experimental range

        output parameters:      ok:                         status of subroutine
                                lengthexpdata:              number of lines in the experimental ascii-file
                                exp_data_x:                 experimental data x column
                                exp_data_y:                 experimental data y column
                                exp_data_error:             error of experimental data y column

        working variables:      k:                          loop counter
                                filename:                   temp variable for file name of the ith fits-file
                                hdulist:                    temp variable containing the whole ith fits-file
                                fitsdata:                   temp variable containing the data of the ith fits-file
                                HDU:                        temp variable for number of HDU in the ith fits-file
                                length_exp_file:            temp variable containing the total number of lines in the ith fits-file (without header lines)
                                number_columns:             temp variable containing the total number of columns in the ith fits-file
                                dimension:                  temp variable containing the dimension of the data of the ith fits-file
                                NumDimensions:              number of dimensions
                                fitshead:                   header of the ith fits file
                                error:                      temp variable containing the error flag
                                NumberRange:                temp variable containing the number of ranges in the ith fits-file
                                xmn:                        array containing all lower limits of the ranges of the ith fits-file
                                xmx:                        array containing all upper limits of the ranges of the ith fits-file
                                exp_data_x:                 temp variable containing experimental data x column of the ith fits-file
                                exp_data_y:                 temp variable containing experimental data y column of the ith fits-file
                                xcolumn:                    temp variable containing the x-columns of the kth line in the ith fits-file
                                copyflag:                   flag for extracting experimental data within selected ranges in the ith fits-file
                                copyflagcolumn:             flag for extracting experimental data within selected ranges in the ith fits-file
                                element:                    flag for extracting experimental data within selected ranges in the ith fits-file


        IMPORTANT:      - The function xml() has to be called before the call of LoadFits()
        """

        # Debug:
        # print ("i = ", i)
        # print ("CounterFitsFile = ", CounterFitsFile)
        # print ("print_flag = ", print_flag)


        ## set status to 0
        ok = 0


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize return parameters
        exp_data_x = []
        exp_data_y = []
        exp_data_error = []
        lengthexpdata = (-1)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get number of ranges and range definitions
        NumberRange = self.NumberExpRanges[i]
        if (NumberRange > 0):
            xmn = self.MinExpRange[i]
            xmx = self.MaxExpRange[i]

            # Debug:
            # print ("xmn = ", xmn)
            # print ("xmx = ", xmx)
            # print ("\t HDU = ", HDU)
            #    print hdulist.info()


            ## check if range definition correct
            if (len(xmn) != NumberRange):
                print ("\n\n\t ERROR in function LoadExpFile.LoadFits:")
                print ("\t\t The dimension of the definition of the content of ")
                print ("\t\t the <MinExpRange></MinExpRange> tag in the xml-file")
                print ("\t\t for the import settings does not coincident with the")
                print ("\t\t dimension of the loaded fits-file!")
                print ("\t\t Please correct the xml-file and restart program!\n")
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error
            elif (len(xmx) != NumberRange):
                print ("\n\n\t ERROR in function LoadExpFile.LoadFits:")
                print ("\t\t The dimension of the definition of the content of ")
                print ("\t\t the <MaxExpRange></MaxExpRange> tag in the xml-file")
                print ("\t\t for the import settings does not coincident with the")
                print ("\t\t dimension of the loaded fits-file!")
                print ("\t\t Please correct the xml-file and restart program!\n")
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## use myXCLASSMapFit function to import FITS data cube


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## import FITS file
        LocalPrintFlag = CheckBool(print_flag)
        ClassFITSImport = FITSImport(self.FileNamesExpFiles[i], HDU = self.NumberHDU[CounterFitsFile], printFlag = LocalPrintFlag)
        FITSParameters = ClassFITSImport.GetFITSParameters()
        self.AllFITSParameters.append(FITSParameters)


        ## check if datacube contains enough data
        NumDimensions = FITSParameters['NumDimensions']
        if (NumDimensions > 4):
            print ("\n\t ERROR in function LoadExpFile.LoadFits:")
            print ("\t\t Can not import fits-file with more than 4 dimensions.")
            print ("\t\t Please select other file and restart program.")
            print ("\n\t\t Program aborted")
            ok = 1
            return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error


        ## get important parameters
        LocalFITSData = FITSParameters['data']                                              ## get data of FITS file
        ReverseIndexingFlag = FITSParameters['ReverseIndexing']                             ## get reverse indexing flag
        FreqAxisID = FITSParameters['FreqAxisID']                                           ## get reverse indexing flag
        AxisValuesList = FITSParameters['AxisValuesList']                                   ## get list of coordinates for each axis
        try:
            FrequencyAxis = AxisValuesList[FreqAxisID]                                      ## get frequencies
        except:
            FrequencyAxis = None


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## analyze region file and extract parameters
        if (self.RegionMask is None):
            regionFileName = ""
            RegionClass = RegionHandling(regionFileName, self.AllFITSParameters, dbDir = "")
            RegionParameters = RegionClass.GetRegionParameters()


            ## get region parameters from dictionary
            self.TotalNumberPixels = RegionParameters['TotalNumberPixels']
            self.RegionMask = RegionParameters['RegionMask']
            self.RegionPixelsX = RegionParameters['RegionPixelsX']
            self.NumRegionPixelsX = RegionParameters['NumRegionPixelsX']
            self.RegionPixelsY = RegionParameters['RegionPixelsY']
            self.NumRegionPixelsY = RegionParameters['NumRegionPixelsY']
            self.CentralPixelOfRegionX = RegionParameters['CentralPixelOfRegionX']
            self.CentralPixelOfRegionIndexX = RegionParameters['CentralPixelOfRegionIndexX']
            self.CentralPixelOfRegionY = RegionParameters['CentralPixelOfRegionY']
            self.CentralPixelOfRegionIndexY = RegionParameters['CentralPixelOfRegionIndexY']


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## FITS data is stored in transposed order
        NumRange = max(1, self.NumberExpRanges[i])
        RangeAxis = []
        for RangeID in range(NumRange):                                                     ## loop over all frequency ranges



            ## if dimension of FITS cube is larger two consider frequency axis
            if (NumDimensions >= 3):
                DeltaFreq = (FrequencyAxis[1] - FrequencyAxis[0])


                ## check, if frequency ranges are specified
                if (self.NumberExpRanges[i] > 0):
                    f1 = self.MinExpRange[i][RangeID]                                       ## get min. frequency
                    f2 = self.MinExpRange[i][RangeID]                                       ## get max. frequency
                    FreqMin = min(f1, f2)
                    FreqMax = max(f1, f2)


                    ## extract frequencies within given range
                    LocalFrequencyAxis = []
                    for FreqID, Freq in enumerate(FrequencyAxis):
                        if (FreqMin <= Freq and Freq <= FreqMax):
                            LocalFrequencyAxis.append([Freq, FreqID])


                ## continue here, if no frequency ranges are specified
                else:
                    LocalFrequencyAxis = [[Freq, FreqID] for FreqID, Freq in enumerate(FrequencyAxis)]


                ## reverse frequency axis for negative DeltaFreq
                #    if (DeltaFreq < 0.0):
                #        LocalFrequencyAxis.reverse()
            else:
                LocalFrequencyAxis = [[0.0, 0]]


            ## take fourth axis into account
            if (NumDimensions == 4):
                LocalStokesAxis = [[Stoke, StokeID] for StokeID, Stoke in enumerate(AxisValuesList[3])]
            else:
                LocalStokesAxis = [[0.0, 0]]

            # Debug:
            # print ("\n\nReverseIndexingFlag = ", ReverseIndexingFlag)
            # print ("DeltaFreq = ", DeltaFreq)
            # print ("NumDimensions = ", NumDimensions)


            ## store axis
            LocalAxis = {}
            LocalAxis['LocalFrequencyAxis'] = LocalFrequencyAxis
            LocalAxis['LocalStokesAxis'] = LocalStokesAxis
            RangeAxis.append(LocalAxis)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## convert FITS data cube into a list


            ## define which case is used
            Case = 1
            if (ReverseIndexingFlag):
                Case = 4

            # Debug:
            # print ("Case = ", Case)


            ## CASE 1
            if (Case == 1):
                for xPointID, xPoint in enumerate(self.RegionPixelsX):                      ## loop over all columns
                    xPixelCoord = xPoint[0]                                                 ## get current x-pixel coordinate
                    xPixelIndex = xPoint[1]                                                 ## get corresponding index
                    for yPointID, yPoint in enumerate(self.RegionPixelsY):                  ## loop over all rows
                        yPixelCoord = yPoint[0]                                             ## get current y-pixel coordinate
                        yPixelIndex = yPoint[1]                                             ## get corresponding index
                        yi = yPoint[2]                                                      ## get inverted index
                        if (self.RegionMask[yi, xPixelIndex] != 0):                         ## consider only selected pixels
                            for FreqElementID, FreqElement in enumerate(LocalFrequencyAxis):        ## loop over frequency channels
                                Freq = FreqElement[0]                                       ## get frequency
                                FreqIndex = FreqElement[1]                                  ## get frequency index
                                for StokeElementID, StokeElement in enumerate(LocalStokesAxis):     ## loop over Stoke coordinates
                                    StokeCoord = StokeElement[0]                            ## get Stoke coordinate
                                    StokeCoordIndex = StokeElement[1]                       ## get Stoke coordinate index

                                    lengthexpdata += 1
                                    if (NumDimensions == 1):
                                        exp_data_x.append([xPixelCoord])
                                        exp_data_y.append([LocalFITSData[xPixelIndex]])
                                    elif (NumDimensions == 2):
                                        exp_data_x.append([xPixelCoord, yPixelCoord])
                                        exp_data_y.append([LocalFITSData[xPixelIndex, yPixelIndex]])
                                    elif (NumDimensions == 3):
                                        exp_data_x.append([xPixelCoord, yPixelCoord, Freq])
                                        exp_data_y.append([LocalFITSData[xPixelIndex, yPixelIndex, FreqIndex]])
                                    elif (NumDimensions == 4):
                                        exp_data_x.append([xPixelCoord, yPixelCoord, Freq, StokeCoord])
                                        exp_data_y.append([LocalFITSData[xPixelIndex, yPixelIndex, FreqIndex, StokeCoordIndex]])
                                    exp_data_error.append([0])


            ## CASE 2
            elif (Case == 2):
                for xPointID, xPoint in enumerate(self.RegionPixelsX):                      ## loop over all columns
                    xPixelCoord = xPoint[0]                                                 ## get current x-pixel coordinate
                    xPixelIndex = xPoint[1]                                                 ## get corresponding index
                    for yPointID, yPoint in enumerate(self.RegionPixelsY):                  ## loop over all rows
                        yPixelCoord = yPoint[0]                                             ## get current y-pixel coordinate
                        yPixelIndex = yPoint[1]                                             ## get corresponding index
                        yi = yPoint[2]                                                      ## get inverted index
                        if (self.RegionMask[yi, xPixelIndex] != 0):                         ## consider only selected pixels
                            for FreqElementID, FreqElement in enumerate(LocalFrequencyAxis):        ## loop over frequency channels
                                Freq = FreqElement[0]                                       ## get frequency
                                FreqIndex = FreqElement[1]                                  ## get frequency index
                                for StokeElementID, StokeElement in enumerate(LocalStokesAxis):     ## loop over Stoke coordinates
                                    StokeCoord = StokeElement[0]                            ## get Stoke coordinate
                                    StokeCoordIndex = StokeElement[1]                       ## get Stoke coordinate index

                                    lengthexpdata += 1
                                    if (NumDimensions == 1):
                                        exp_data_x.append([xPixelCoord])
                                        exp_data_y.append([LocalFITSData[xPixelIndex]])
                                    elif (NumDimensions == 2):
                                        exp_data_x.append([xPixelCoord, yPixelCoord])
                                        exp_data_y.append([LocalFITSData[yPixelIndex, xPixelIndex]])
                                    elif (NumDimensions == 3):
                                        exp_data_x.append([xPixelCoord, yPixelCoord, Freq])
                                        exp_data_y.append([LocalFITSData[FreqIndex, yPixelIndex, xPixelIndex]])
                                    elif (NumDimensions == 4):
                                        exp_data_x.append([xPixelCoord, yPixelCoord, Freq, StokeCoord])
                                        exp_data_y.append([LocalFITSData[StokeCoordIndex, FreqIndex, yPixelIndex, xPixelIndex]])
                                    exp_data_error.append([0])


            ## CASE 3
            elif (Case == 3):
                for StokeElementID, StokeElement in enumerate(LocalStokesAxis):             ## loop over Stoke coordinates
                    StokeCoord = StokeElement[0]                                            ## get Stoke coordinate
                    StokeCoordIndex = StokeElement[1]                                       ## get Stoke coordinate index
                    for FreqElementID, FreqElement in enumerate(LocalFrequencyAxis):        ## loop over frequency channels
                        Freq = FreqElement[0]                                               ## get frequency
                        FreqIndex = FreqElement[1]                                          ## get frequency index
                        for yPointID, yPoint in enumerate(self.RegionPixelsY):              ## loop over all rows
                            yPixelCoord = yPoint[0]                                         ## get current y-pixel coordinate
                            yPixelIndex = yPoint[1]                                         ## get corresponding index
                            yi = yPoint[2]                                                  ## get inverted index
                            for xPointID, xPoint in enumerate(self.RegionPixelsX):          ## loop over all columns
                                xPixelCoord = xPoint[0]                                     ## get current x-pixel coordinate
                                xPixelIndex = xPoint[1]                                     ## get corresponding index
                                if (self.RegionMask[yi, xPixelIndex] != 0):                 ## consider only selected pixels

                                    lengthexpdata += 1
                                    if (NumDimensions == 1):
                                        exp_data_x.append([xPixelCoord])
                                        exp_data_y.append([LocalFITSData[xPixelIndex]])
                                    elif (NumDimensions == 2):
                                        exp_data_x.append([xPixelCoord, yPixelCoord])
                                        exp_data_y.append([LocalFITSData[xPixelIndex, yPixelIndex]])
                                    elif (NumDimensions == 3):
                                        exp_data_x.append([xPixelCoord, yPixelCoord, Freq])
                                        exp_data_y.append([LocalFITSData[xPixelIndex, yPixelIndex, FreqIndex]])
                                    elif (NumDimensions == 4):
                                        exp_data_x.append([xPixelCoord, yPixelCoord, Freq, StokeCoord])
                                        exp_data_y.append([LocalFITSData[xPixelIndex, yPixelIndex, FreqIndex, StokeCoordIndex]])
                                    exp_data_error.append([0])


            ## CASE 4
            elif (Case == 4):
                for StokeElementID, StokeElement in enumerate(LocalStokesAxis):             ## loop over Stoke coordinates
                    StokeCoord = StokeElement[0]                                            ## get Stoke coordinate
                    StokeCoordIndex = StokeElement[1]                                       ## get Stoke coordinate index
                    for FreqElementID, FreqElement in enumerate(LocalFrequencyAxis):        ## loop over frequency channels
                        Freq = FreqElement[0]                                               ## get frequency
                        FreqIndex = FreqElement[1]                                          ## get frequency index
                        for yPointID, yPoint in enumerate(self.RegionPixelsY):              ## loop over all rows
                            yPixelCoord = yPoint[0]                                         ## get current y-pixel coordinate
                            yPixelIndex = yPoint[1]                                         ## get corresponding index
                            yi = yPoint[2]                                                  ## get inverted index
                            for xPointID, xPoint in enumerate(self.RegionPixelsX):          ## loop over all columns
                                xPixelCoord = xPoint[0]                                     ## get current x-pixel coordinate
                                xPixelIndex = xPoint[1]                                     ## get corresponding index
                                if (self.RegionMask[yi, xPixelIndex] != 0):                 ## consider only selected pixels
                                    lengthexpdata += 1
                                    if (NumDimensions == 1):
                                        exp_data_x.append([xPixelCoord])
                                        exp_data_y.append([LocalFITSData[xPixelIndex]])
                                    elif (NumDimensions == 2):
                                        exp_data_x.append([xPixelCoord, yPixelCoord])
                                        exp_data_y.append([LocalFITSData[yPixelIndex, xPixelIndex]])
                                    elif (NumDimensions == 3):
                                        exp_data_x.append([xPixelCoord, yPixelCoord, Freq])
                                        exp_data_y.append([LocalFITSData[FreqIndex, yPixelIndex, xPixelIndex]])
                                    elif (NumDimensions == 4):
                                        exp_data_x.append([xPixelCoord, yPixelCoord, Freq, StokeCoord])
                                        exp_data_y.append([LocalFITSData[StokeCoordIndex, FreqIndex, yPixelIndex, xPixelIndex]])
                                    exp_data_error.append([0])

        ## store range axis
        self.AllRangeAxis.append(RangeAxis)

        # Debug:
        # print ("\n\nok = ", ok)
        # print ("lengthexpdata = ", lengthexpdata)
        # print ("exp_data_x = ", exp_data_x)
        # print ("exp_data_y = ", exp_data_y)
        # print ("exp_data_error = ", exp_data_error)


        ## define return parameters
        return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ##
    ##                                              Write data
    ##
    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## write contents to ascii-file(s)
    ##
    def WriteFile(self, ExpInputFormat, print_flag, NumberExpFiles, LengthExpRange, ColumnY, ExpDataX, FitFunctionValues, Chi2SaveFlag, Chi2Values, \
                  FilenameExtension):
        """

        function WriteFile:     write values of the fit function within the ranges to file

        input parameters:       ExpInputFormat:             defines if output file has ASCII or fits format
                                print_flag:                 flag for printing message to screen
                                alldata:                    if all == true the x and y values are written to file
                                NumberExpFiles:             number of dat-files
                                LengthExpRange:             number of lines in the experimental dat-file
                                ColumnY:                    number of "y"-columns
                                ExpDataX:                   experimental data x column
                                FitFunctionValues:          values of the fit function
                                Chi2SaveFlag:               flag for saving chi^2 values
                                Chi2Values:                 values of chi^2
                                FilenameExtension:          extension of the parameter file name

        output parameters:      ok:                         status of subroutine
        """

        # Debug:
        # print ("ExpInputFormat = ", ExpInputFormat)
        # print ("print_flag = ", print_flag)
        # print ("NumberExpFiles = ", NumberExpFiles)
        # print ("LengthExpRange = ", LengthExpRange)
        # print ("ColumnY = ", ColumnY)
        # print ("ExpDataX = ", ExpDataX)
        # print ("FitFunctionValues = ", FitFunctionValues)
        # print ("Chi2SaveFlag = ", Chi2SaveFlag)
        # print ("Chi2Values = ", Chi2Values)
        # print ("FilenameExtension = ", FilenameExtension)


        ## set status to 0
        ok = 0


        ## print what you do
        if (print_flag != "false"):
            print ("Write values of the fit function to files.")


        ## read experimental data of each ascii-file and define output variables
        NumberFiles = self.NumberExpFiles
        CounterASCIIFile = (-1)
        CounterFitsFile = (-1)
        for i in range(NumberFiles):
            FileFormat = ExpInputFormat[i].lower()

            # Debug:
            # print ("i, FileFormat = ", i, FileFormat)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write values of the fit function to file
            ## load experimental data using several import functions depending on the file format
            if (FileFormat in ["dat", "ascii", "xclassascii", "cso", "xclass"]):
                                                                                            ## if ascii file selected, use ascii-file function to
                                                                                            ##   import experimental data
                CounterASCIIFile += 1
                alldata = "true"                                                            ## write all data x and y values to file
                ok = self.WriteDat(i, CounterASCIIFile, print_flag, alldata, NumberExpFiles, LengthExpRange, ColumnY, ExpDataX, FitFunctionValues, \
                                   Chi2SaveFlag, Chi2Values, FilenameExtension)
            elif (FileFormat in ["fits", "xclassfits"]):                                    ## if fits file selected, use fits-file function
                                                                                            ##   to import experimental data
                CounterFitsFile += 1
                ok = self.WriteFits(i, CounterFitsFile, print_flag, NumberExpFiles, LengthExpRange, ColumnY, ExpDataX, FitFunctionValues, \
                                    Chi2SaveFlag, Chi2Values, FilenameExtension)


            ## does an error occur
            if (ok != 0):
                return ok
            if (print_flag != "false"):
                print ()
                print ()


        ## define return parameters
        return ok
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## write contents to ascii-file(s)
    ##
    def WriteDat(self, ObsDataFileIndex, CounterASCIIFile, print_flag, alldata, number_files, LengthExpRange, ColumnY, ExpDataX, FitFunctionValues, \
                 Chi2SaveFlag, Chi2Values, FilenameExtension):
        """

        function WriteDat:      write values of the fit function within the ranges to ascii-file

        input parameters:       ObsDataFileIndex:           index of experimental data file
                                CounterASCIIFile:           counter for ascii files
                                print_flag:                 flag for printing message to screen
                                alldata:                    if all == true the x and y values are written to file
                                number_files:               number of ascii-files
                                LengthExpRange:             number of lines in the experimental ascii-file
                                ColumnY:                    number of "y"-columns
                                ExpDataX:                   experimental data x column
                                FitFunctionValues:          values of the fit function
                                Chi2SaveFlag:               flag for saving chi^2 values
                                Chi2Values:                 values of chi^2
                                FilenameExtension:          extention of the file name

        required parameters:    NumberExpFiles:             number of ascii-files
                                FileNamesExpFiles:          path and file name of each ascii file
                                NumberHeaderLines:          number of header lines for each ascii-file
                                SeparatorColumns:           column separator for each ascii-file
                                NumberColumnsX:             number of columns belonging to the x-positions
                                NumberColumnsY:             number of columns belonging to the y-positions

        output parameters:      ok:                         status of subroutine

        working variables:      j:                          loop counter
                                filename:                   temp variable for file name of the ith ascii-file
                                number_header_lines:        temp variable for number of header lines in the ith
                                                            ascii-file
                                seperator:                  temp variable for column separator in the ith
                                                            ascii-file
                                data:                       temp variable containing the whole ith ascii-file
                                length_exp_file:            temp variable containing the total number of lines
                                                            in the ith ascii-file (without header lines)
                                error:                      temp variable containing the error flag
                                LengthFile:                 temp variable containing the number of lines in the
                                                            ith ascii-file
                                column:                     temp variable containing the number of "y"-columns
                                                            in the ith ascii-file
                                line:                       jth line of the ith ascii file


        IMPORTANT:      - The function xml() has to be called before the call of WriteDat()
        """

        # Debug:
        # print ("ObsDataFileIndex = ", ObsDataFileIndex)
        # print ("CounterASCIIFile = ", CounterASCIIFile)
        # print ("print_flag = ", print_flag)
        # print ("alldata = ", alldata)
        # print ("number_files = ", number_files)
        # print ("LengthExpRange = ", LengthExpRange)
        # print ("ColumnY = ", ColumnY)
        # print ("ExpDataX = ", ExpDataX)
        # print ("FitFunctionValues = ", FitFunctionValues)
        # print ("Chi2SaveFlag = ", Chi2SaveFlag)
        # print ("Chi2Values = ", Chi2Values)
        # print ("FilenameExtension = ", FilenameExtension)


        ## set status to 0
        ok = 0


        ## set working variables for the ith ascii-file
        filename = self.FileNamesExpFiles[ObsDataFileIndex]
        number_header_lines = self.NumberHeaderLines[CounterASCIIFile]
        seperator = self.SeparatorColumns[CounterASCIIFile]
        NumberX = self.NumberColumnsX[CounterASCIIFile]
        NumberY = self.NumberColumnsY[CounterASCIIFile]
        error = self.ErrorY[CounterASCIIFile]
        LengthFile = LengthExpRange[ObsDataFileIndex]
        column = ColumnY[CounterASCIIFile]


        ## warning if there is no data within the range(s)
        if (LengthFile == 0):
            print ("\n\n\t ERROR in function LoadExpFile.LoadExp.WriteDat:")
            print ("\t\t There is no data within the given ranges!")
            print ("\t\t Please correct xml-file and restart program!\n")
            ok = 1
            return ok
        else:


            ## modify filename
            if filename.endswith(".fits"):
                filename = filename.replace(".fits", ".dat")
            else:
                fileExt = "." + FilenameExtension + ".dat"
                if filename.endswith(".dat"):
                    filename = filename.replace(".dat", fileExt)
                else:
                    filename = filename.strip()
                    filename = filename + fileExt


            ## print what you do
            if (print_flag != "false"):
                print ("\t Writing data to ASCII file " + filename + " ..", end=' ')

            # Debug:
            # print ("\nNumberFiles = ", NumberFiles)
            # print ("LengthFile = ", LengthFile)
            # print ("column = ", column)
            # print ("FitFunctionValues = ",  FitFunctionValues)
            # print ("column = ", column)
            # print ("LengthFile = ", LengthFile)


            ## construct output array
            outdat = []
            for j in range(LengthFile):
                line = []
                if (alldata == "true"):
                    for k in range(len(ExpDataX[ObsDataFileIndex][j])):
                        line.append(ExpDataX[ObsDataFileIndex][j][k])
                for k in range(column):
                    line.append(FitFunctionValues[ObsDataFileIndex][j][k])
                outdat.append(line)


            ## write ascii-file
            outdatFile = open(filename, 'w')
            for line in outdat:
                finalLine = ""
                for columns in line:
                    # finalLine = finalLine + seperator + str(columns)
                    finalLine = finalLine + " " + str(columns)
                outdatFile.write(finalLine + "\n")
            outdatFile.close()

            # Debug:
            # print ("\n\ndata[1] = ", data[1])
            # print ("data.shape = ", data.shape)


            ## save values of chi^2
            if (Chi2SaveFlag == "true"):

                ## modify filename
                fileExt = "." + FilenameExtension + ".dat"
                fileExtOut = "." + FilenameExtension + ".chi2.dat"
                filename = filename.replace(fileExt, fileExtOut)


                ## print what you do
                if (print_flag != "false"):
                    print ("done!")
                    print ("\t Writing chi^2 values to file " + filename + " ..", end=' ')


                ## construct output array
                outdat = []
                for j in range(LengthFile):
                    line = []
                    if (alldata == "true"):
                        for k in range(len(ExpDataX[ObsDataFileIndex][j])):
                            line.append(ExpDataX[ObsDataFileIndex][j][k])
                    for k in range(column):
                        line.append(Chi2Values[ObsDataFileIndex][j][k])
                    outdat.append(line)


                ## write ascii-file
                outdatFile = open(filename, 'w')
                for line in outdat:
                    finalLine = ""
                    for columns in line:
                        # finalLine = finalLine + seperator + str(columns)
                        finalLine = finalLine + " " + str(columns)
                    outdatFile.write(finalLine + "\n")
                outdatFile.close()


            ## print that you are finished
            if (print_flag != "false"):
                print ("done!")


        ## define return parameters
        return ok
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## export new FITS file
    ##
    def ExportFITSFile(self, NewHDU, NameOfFunction, FITSParameters, ImageFlag, CentralPixelOfRegionX, CentralPixelOfRegionIndexX, \
                       CentralPixelOfRegionY, CentralPixelOfRegionIndexY, NumXPixels, NumYPixels, NewBUNIT, NewFITSFileName, RefFrequency = None, \
                       UseCentralPixelFlag = True, ApplyCenterCorrectionFlag = False):
        """

    input parameters:
    -----------------

        - NewHDU:                       HDU for new FITS file

        - NameOfFunction:               name of function

        - FITSParameters:               FITS parameters

        - ImageFlag:                    store image or cube

        - CentralPixelOfRegionX:        x-coordinate of region center

        - CentralPixelOfRegionIndexX:   index of x-coordinate of region center

        - CentralPixelOfRegionY:        y-coordinate of region center

        - CentralPixelOfRegionIndexY:   index of y-coordinate of region center

        - NumXPixels:                   number of pixels in x direction

        - NumYPixels:                   number of pixels in y direction

        - NewBUNIT:                     new unit

        - NewFITSFileName:              path and name of new FITS file

        - RefFrequency:                 (optional) reference frequency point (only used for FrequencyFlag == true)

        - UseCentralPixelFlag:          (optional) use central pixel as reference pixel (not for frequency axis, where first pixel is used as reference)

        - ApplyCenterCorrectionFlag:    (optional) apply correction of center position


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("NewHDU = ", NewHDU)
        # print ("NameOfFunction = ", NameOfFunction)
        # print ("FITSParameters = ", FITSParameters)
        # print ("ImageFlag = ", ImageFlag)
        # print ("CentralPixelOfRegionX = ", CentralPixelOfRegionX)
        # print ("CentralPixelOfRegionIndexX = ", CentralPixelOfRegionIndexX)
        # print ("CentralPixelOfRegionY = ", CentralPixelOfRegionY)
        # print ("CentralPixelOfRegionIndexY = ", CentralPixelOfRegionIndexY)
        # print ("NumXPixels = ", NumXPixels)
        # print ("NumYPixels = ", NumYPixels)
        # print ("NewBUNIT = ", NewBUNIT)
        # print ("NewFITSFileName = ", NewFITSFileName)
        # print ("RefFrequency = ", RefFrequency)
        # print ("ApplyCenterCorrectionFlag = ", ApplyCenterCorrectionFlag)


        ## get some parameters from first FITS file
        HDU = FITSParameters['HDU']
        NumDimensions = FITSParameters['NumDimensions']
        lengthAxis = FITSParameters['lengthAxis']
        CTYPEList = FITSParameters['CTYPE']
        CRPIXList = FITSParameters['CRPIX']
        CRVALList = FITSParameters['CRVAL']
        CDELTList = FITSParameters['CDELT']
        CROTList = FITSParameters['CROT']
        CUNITList = FITSParameters['CUNIT']
        BSCALE = FITSParameters['BSCALE']
        BZERO = FITSParameters['BZERO']
        BUNIT = FITSParameters['BUNIT']
        RESTFRQ = FITSParameters['RESTFRQ']
        ValueOfFirstPointList = FITSParameters['ValueOfFirstPoint']
        FITSHeader = FITSParameters['header']
        ReverseIndexing = FITSParameters['ReverseIndexing']
        AxisValuesList = FITSParameters['AxisValuesList']                                   ## get list of coordinates for each axis
        if (not ImageFlag):
            FreqAxisID = FITSParameters['FreqAxisID']
            FrequencyAxis = AxisValuesList[FreqAxisID]                                      ## get frequencies


        ## select FITS command for header update
        try:
            import astropy.io.fits as pyfits
            OldFITSCommandFlag = False
        except ImportError:
            OldFITSCommandFlag = True

        # Debug:
        # print ("\n\n\nNewHDU.header = ", NewHDU.header)
        # print ("HDU = ", HDU)
        # print ("NumDimensions = ", NumDimensions)
        # print ("lengthAxis = ", lengthAxis)
        # print ("CTYPEList = ", CTYPEList)
        # print ("CRPIXList = ", CRPIXList)
        # print ("CRVALList = ", CRVALList)
        # print ("CDELTList = ", CDELTList)
        # print ("CROTList = ", CROTList)
        # print ("CUNITList = ", CUNITList)
        # print ("BSCALE = ", BSCALE)
        # print ("BZERO = ", BZERO)
        # print ("BUNIT = ", BUNIT)
        # print ("RESTFRQ = ", RESTFRQ)
        # print ("ValueOfFirstPointList = ", ValueOfFirstPointList)
        # print ("FITSHeader = ", FITSHeader)
        # print ("ReverseIndexing = ", ReverseIndexing)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## import header parameters from FITS file


        ## define list of forbidden command words
        ForbiddenCommandWordList = ["simple", "bitpix", "naxis", "naxis1", "naxis2", "naxis3", "naxis4", "history", "date", "object", "datamin", \
                                    "datamax", "bscale", "bzero", "bunit", "btype", "restfrq", "comment", "observer", "date-obs", "origin", "checksum", \
                                    "checkver", "creator", "datamode", "datasum", "date-end", "exposure", "exptime", "obs_id", "obs_mode", "ontime", \
                                    "detnam", "elaptime", "time-end", "time-obs", "program", \
                                    "dec", "dec_nom", "dec_obj", "dec_pnt", "dec_scx", "dec_scy", "dec_scz", \
                                    "ra", "ra_nom", "ra_obj", "ra_pnt", "ra_scx", "ra_scy", "ra_scz", \
                                    "cdelt1", "crpix1", "crval1", "ctype1", "cunit1", "crot1", \
                                    "cdelt2", "crpix2", "crval2", "ctype2", "cunit2", "crot2", \
                                    "cdelt3", "crpix3", "crval3", "ctype3", "cunit3", "crot3", \
                                    "cdelt4", "crpix4", "crval4", "ctype4", "cunit4", "crot4"]


        ## get a list of all command words in the FITS header
        ListOfCommandWords = list(FITSHeader.keys())
        for LocalCommandWord in ListOfCommandWords:
            LowerLocalCommandWord = LocalCommandWord.lower()
            LowerLocalCommandWord = LowerLocalCommandWord.strip()
            if (LowerLocalCommandWord != "" and (not LowerLocalCommandWord in ForbiddenCommandWordList)):


                ## get value(s) for current command word
                LocalValueRaw = str(FITSHeader[LocalCommandWord])
                try:
                    LocalValue = float(LocalValueRaw)
                    if (LowerLocalCommandWord == "blank"):                                  ## 'BLANK' keyword must be an integer
                        LocalValue = int(LocalValue)
                except:
                    LocalValue = LocalValueRaw

                # Debug:
                # print ("\n\nLocalCommandWord = ", LocalCommandWord)
                # print ("LocalValue = ", LocalValue)


                ## update header
                if (OldFITSCommandFlag):
                    NewHDU.header.update(LocalCommandWord, LocalValue, '')
                else:
                    NewHDU.header[LocalCommandWord] = LocalValue


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update general header parameters


        ## define/update OBJECT command word
        if (OldFITSCommandFlag):
            NewHDU.header.update('BITPIX', -32, '')
        else:
            NewHDU.header['BITPIX'] = (-32, '')


        ## define/update OBJECT command word
        if (OldFITSCommandFlag):
            NewHDU.header.update('OBJECT', '        ', '')
        else:
            NewHDU.header['OBJECT'] = ('        ', '')


        ## define/update BSCALE command word
        if (OldFITSCommandFlag):
            NewHDU.header.update('BSCALE', BSCALE, 'PHYSICAL = PIXEL * BSCALE + BZERO')
        else:
            NewHDU.header['BSCALE'] = (BSCALE, 'PHYSICAL = PIXEL * BSCALE + BZERO')


        ## define/update BZERO command word
        if (OldFITSCommandFlag):
            NewHDU.header.update('BZERO', BZERO, '')
        else:
            NewHDU.header['BZERO'] = (BZERO, '')


        ## define/update BUNIT command word
        if (OldFITSCommandFlag):
            NewHDU.header.update('BUNIT', NewBUNIT[0], NewBUNIT[1])
        else:
            NewHDU.header['BUNIT'] = (NewBUNIT[0], NewBUNIT[1])


        ## define/update RESTFRQ command word
        if (OldFITSCommandFlag):
            NewHDU.header.update('RESTFRQ', RESTFRQ, '')
        else:
            NewHDU.header['RESTFRQ'] = (RESTFRQ, '')


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update parameters for each axis
        if (ImageFlag):
            NAXIS = 2
        else:
            NAXIS = min(len(NewHDU.data.shape), NumDimensions)
        for AxisID in range(NAXIS):                                                         ## loop over all axis
            ctype = CTYPEList[AxisID]
            crpix = CRPIXList[AxisID]
            crval = CRVALList[AxisID]
            cdelt = CDELTList[AxisID]
            cunit = CUNITList[AxisID]
            crot = (CROTList[AxisID])


            ##********************************************************************************************************************************************
            ## use central pixel as reference point
            if (UseCentralPixelFlag):


                ##========================================================================================================================================
                ## define parameters for axis 1
                if (AxisID == 0):


                    ## define/update crval1
                    if (ApplyCenterCorrectionFlag):
                        crval = (CentralPixelOfRegionX + CDELTList[AxisID] / 2.0)
                    else:
                        crval = CentralPixelOfRegionX


                    ## define/update crpix1
                    crpix = CentralPixelOfRegionIndexX


                    ## define/update cdelt AxisID
                    #    if (ApplyCenterCorrectionFlag):
                    #        crval2 = (CentralPixelOfRegionY + CDELTList[1] / 2.0)
                    #    else:
                    #        crval2 = CentralPixelOfRegionY
                    #    cdelt = cdelt * math.cos(math.pi / 180.0 * crval2)


                ##========================================================================================================================================
                ## define parameters for axis 2
                elif (AxisID == 1):


                    ## define/update crval
                    if (ApplyCenterCorrectionFlag):
                        crval = (CentralPixelOfRegionY + CDELTList[AxisID] / 2.0)
                    else:
                        crval = CentralPixelOfRegionY


                    ## define/update crpix2
                    crpix = CentralPixelOfRegionIndexY


                ##========================================================================================================================================
                ## define parameters for axis 3
                elif (AxisID == 2):


                    ## define/update crval3
                    # crval = (RefFrequency + CDELTList[AxisID] / 2.0)
                    crval = RefFrequency


                    ## define/update crpix3
                    # crpix = (NumFrequencies / 2.0 + 1.0)
                    crpix = 1                                                               ## define first frequency as reference point


            ##********************************************************************************************************************************************
            ## the header uses always the first pixel as reference point
            else:
                crpix = 1
                if (AxisID == 0):
                    crval2 = CRVALList[1]
                    #cdelt = cdelt * math.cos(math.pi / 180.0 * crval2)


            ##********************************************************************************************************************************************
            ## special settings for frequency axis
            if (not ImageFlag):
                if (AxisID == FreqAxisID):
                    ctype = 'FREQ    '
                    cunit = 'MHz     '
                    crpix = 1
                    crval = RefFrequency
                    cdelt = (FrequencyAxis[1] - FrequencyAxis[0])


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## update command words


            ## define/update CTYPE command word
            if (OldFITSCommandFlag):
                NewHDU.header.update('CTYPE' + str(AxisID + 1), ctype, '')
            else:
                NewHDU.header['CTYPE' + str(AxisID + 1)] = (ctype, '')


            ## define/update CUNIT command word
            if (OldFITSCommandFlag):
                NewHDU.header.update('CUNIT' + str(AxisID + 1), cunit, '')
            else:
                NewHDU.header['CUNIT' + str(AxisID + 1)] = (cunit, '')


            ## define/update CRPIX command word
            if (OldFITSCommandFlag):
                NewHDU.header.update('CRPIX' + str(AxisID + 1), crpix, '')
            else:
                NewHDU.header['CRPIX' + str(AxisID + 1)] = (crpix, '')


            ## define/update CRVAL command word
            if (OldFITSCommandFlag):
                NewHDU.header.update('CRVAL' + str(AxisID + 1), crval, '')
            else:
                NewHDU.header['CRVAL' + str(AxisID + 1)] = (crval, '')


            ## define/update CDELT command word
            if (OldFITSCommandFlag):
                NewHDU.header.update('CDELT' + str(AxisID + 1), cdelt, '')
            else:
                NewHDU.header['CDELT' + str(AxisID + 1)] = (cdelt, '')


            ## define/update CROT command word
            if (OldFITSCommandFlag):
                NewHDU.header.update('CROT' + str(AxisID + 1), crot, '')
            else:
                NewHDU.header['CROT' + str(AxisID + 1)] = (crot, '')


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add data and time stamp to header
        lt = time.localtime()
        TimeStamp = time.strftime("%d-%m-%Y", lt) + "__" + time.strftime("%H-%M-%S", lt)
        if (OldFITSCommandFlag):
            NewHDU.header.update('DATE', TimeStamp, 'Date FITS file was written')
        else:
            NewHDU.header['DATE'] = (TimeStamp, 'Date FITS file was written')


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add software label
        if (OldFITSCommandFlag):
            NewHDU.header.update('software', NameOfFunction, 'Created by MAGIX')
        else:
            NewHDU.header['software'] = (NameOfFunction, 'Created by MAGIX')


        ## convert FITS data array to 32 bit float and transpose if necessary
        LocalFITSDataArray = numpy.float32(NewHDU.data)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write new FITS file to hard disk
        try:
            pyfits.writeto(NewFITSFileName, LocalFITSDataArray, NewHDU.header, overwrite = True)
        except:
            pyfits.writeto(NewFITSFileName, LocalFITSDataArray, NewHDU.header, clobber = True)


        ## finished
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## write contents to fits-file(s)
    ##
    def WriteFits(self, ObsDataFileIndex, CounterFitsFile, print_flag, number_files, LengthExpRange, ColumnY, ExpDataX, FitFunctionValues, \
                  Chi2SaveFlag, Chi2Values, FilenameExtension):
        """

        function WriteFits:     write values of the fit function within the ranges to fits-file

        input parameters:       ObsDataFileIndex:           index of experimental data file
                                CounterFitsFile:            counter for fits files
                                print_flag:                 flag for printing message to screen
                                number_files:               number of fits-files
                                LengthExpRange:             number of lines in the experimental fits-file
                                ColumnY:                    number of "y"-columns
                                ExpDataX:                   experimental data x column
                                FitFunctionValues:          values of the fit function
                                Chi2SaveFlag:               flag for saving chi^2 values
                                Chi2Values:                 values of chi^2
                                FilenameExtension:          extention of the parameter file name

        required parameters:    NumberExpFiles:             number of fits-files
                                FileNamesExpFiles:          path and file name of each fits file
                                FitsHeader:                 header of each fits file
                                HDU:                        HDU number

        output parameters:      ok:                         status of the subroutine

        working variables:      j:                          loop counter
                                filename:                   temp variable for file name of the ith fits-file
                                number_header_lines:        temp variable for number of header lines in the ith fits-file
                                seperator:                  temp variable for column separator in the ith fits-file
                                data:                       temp variable containing the whole ith fits-file
                                length_exp_file:            temp variable containing the total number of lines in the ith fits-file (without header lines)
                                LengthFile:                 temp variable containing the number of lines in the ith fits-file
                                column:                     temp variable containing the number of "y"-columns in the ith fits-file
                                line:                       jth line of the ith fits file
                                HDUNumber:                  HDU number


        IMPORTANT:      - The function xml() has to be called before the call of WriteDat()
        """

        # Debug:
        # print ("ObsDataFileIndex = ", ObsDataFileIndex)
        # print ("CounterFitsFile = ", CounterFitsFile)
        # print ("print_flag = ", print_flag)
        # print ("number_files = ", number_files)
        # print ("LengthExpRange = ", LengthExpRange)
        # print ("ColumnY = ", ColumnY)
        # print ("ExpDataX = ", ExpDataX)
        # print ("FitFunctionValues = ", FitFunctionValues)
        # print ("Chi2SaveFlag = ", Chi2SaveFlag)
        # print ("Chi2Values = ", Chi2Values)
        # print ("FilenameExtension = ", FilenameExtension)


        ## set status to 0
        ok = 0


        ## set working variables for the ith fits-file
        LocalOutputFileName = self.FileNamesExpFiles[ObsDataFileIndex]
        HDU = self.NumberHDU[CounterFitsFile]
        RangeAxis = self.AllRangeAxis[CounterFitsFile]
        LocalPrintFlag = CheckBool(print_flag)
        LocalChi2SaveFlag = CheckBool(Chi2SaveFlag)
        LengthFile = LengthExpRange[ObsDataFileIndex]
        column = ColumnY[CounterFitsFile]
        NumberRange = self.NumberExpRanges[ObsDataFileIndex]                                ## read number of ranges
        if (NumberRange > 0):
            xmn = self.MinExpRange[ObsDataFileIndex]
            xmx = self.MaxExpRange[ObsDataFileIndex]

        # Debug:
        # print ("\n\n")
        # print ("ObsDataFileIndex = ", ObsDataFileIndex)
        # print ("xmn = ", xmn)
        # print ("xmx = ", xmx)


        ## warning if there is no data within the range(s)
        if (LengthFile == 0):
            print ("\n\n\t ERROR in function LoadExpFile.LoadExp.WriteFits:")
            print ("\t\t There is no data within the given ranges!")
            print ("\t\t Please correct xml-file and restart program!\n")
            ok = 1
            return ok


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get default settings from current cube
        FITSParameters = self.AllFITSParameters[ObsDataFileIndex]                           ## get parameter from first FITS file
        ReverseIndexingFlag = FITSParameters['ReverseIndexing']                             ## get reverse indexing flag
        UnitAixs = FITSParameters['BUNIT']                                                  ## get unit of each axis
        OrigNameOfCube = FITSParameters['FITSFileName']                                     ## get path and name of FITS file
        FreqAxisID = FITSParameters['FreqAxisID']                                           ## get reverse indexing flag
        AxisValuesList = FITSParameters['AxisValuesList']                                   ## get list of coordinates for each axis
        FrequencyAxis = AxisValuesList[FreqAxisID]                                          ## get frequencies
        NumDimensions = FITSParameters['NumDimensions']                                     ## get list of coordinates for each axis


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create FITS images for each data range and obs. data file
        NumRan = max(1, NumberRange)
        for RangeID in range(NumRan):                                                       ## loop over all ranges in the current obs. xml file


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## define new output file name
            if (NumberRange < 2):
                if LocalOutputFileName.endswith(".dat"):
                    LocalOutputFileNameNew = LocalOutputFileName.replace(".dat", ".fits")
                    if (LocalChi2SaveFlag):
                        Chi2FileName = LocalOutputFileName.replace(".dat", ".out.chi2.fits")
                else:
                    fileExt = "." + FilenameExtension + ".fits"
                    LocalOutputFileNameNew = LocalOutputFileName.replace(".fits", fileExt)
                    if (LocalChi2SaveFlag):
                        Chi2FileName = LocalOutputFileNameNew.replace(fileExt, "." + FilenameExtension + ".chi2.fits")
            else:
                # Debug:
                # print ("\n\n\nran, xmn[RangeID] = ", RangeID, xmn[RangeID])
                # print ("LocalOutputFileName = ", LocalOutputFileName)


                ## check if range limits are lists
                if (isinstance(xmn[RangeID], list)):
                    xmnString = ""
                    xmxString = ""
                    for DimID in range(len(xmn[RangeID])):
                        xmnString += "_" + str(xmn[RangeID][DimID])
                        xmxString += "_" + str(xmx[RangeID][DimID])
                else:
                    xmnString = "_" + str(xmn[RangeID])
                    xmxString = "_" + str(xmx[RangeID])
                RangeExten = "_" + str(xmnString) + "_-" + str(xmxString)

                # Debug:
                # print ("RangeExten = ", RangeExten)


                if LocalOutputFileName.endswith(".dat"):
                    LocalOutputFileNameNew = LocalOutputFileName.replace(".dat", RangeExten + ".fits")
                    if (LocalChi2SaveFlag):
                        Chi2FileName = LocalOutputFileName.replace(".dat", RangeExten + ".out.chi2.fits")
                else:
                    fileExt = "." + FilenameExtension + "_" + RangeExten + ".fits"
                    LocalOutputFileNameNew = LocalOutputFileName.replace(".fits", fileExt)
                    if (LocalChi2SaveFlag):
                        Chi2FileName = LocalOutputFileNameNew.replace(fileExt, "." + FilenameExtension + RangeExten + ".chi2.fits")


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## delete old output file
            if (os.path.exists(LocalOutputFileNameNew)):
                os.remove(LocalOutputFileNameNew)
            if (os.path.exists(Chi2FileName)):
                os.remove(Chi2FileName)


            ## print what you do
            if (print_flag != "false"):
                print ("\t Writing fits-file " + LocalOutputFileNameNew + " ..", end=' ')


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## construct output arrays


            ## get additional axis
            LocalAxis = RangeAxis[RangeID]
            LocalFrequencyAxis = LocalAxis['LocalFrequencyAxis']
            LocalStokesAxis = LocalAxis['LocalStokesAxis']

            # Debug:
            # print ("LocalFrequencyAxis = ", LocalFrequencyAxis)
            # print ("LocalStokesAxis = ", LocalStokesAxis)


            ## new FITS data and chi^2 array
            if (NumDimensions == 1):
                DimensionTuple = (len(self.RegionPixelsX),)
            elif (NumDimensions == 2):
                DimensionTuple = (len(self.RegionPixelsX), len(self.RegionPixelsY) )
            elif (NumDimensions == 3):
                DimensionTuple = (len(self.RegionPixelsX), len(self.RegionPixelsY), len(LocalFrequencyAxis))
            elif (NumDimensions == 4):
                DimensionTuple = (len(self.RegionPixelsX), len(self.RegionPixelsY), len(LocalFrequencyAxis), len(LocalStokesAxis))
            if (ReverseIndexingFlag):
                DimensionTuple = DimensionTuple[::-1]
            NewFITSCube = numpy.zeros(DimensionTuple, dtype = numpy.float32)
            if (LocalChi2SaveFlag):
                NewChi2Cube = numpy.zeros(DimensionTuple, dtype = numpy.float32)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## construct numpy arrays
            npixels = (-1)


            ## store FITS and chi^2 cubes in normal order
            if (ReverseIndexingFlag):
                for StokeElementID, StokeElement in enumerate(LocalStokesAxis):             ## loop over Stoke coordinates
                    StokeCoord = StokeElement[0]                                            ## get Stoke coordinate
                    StokeCoordIndex = StokeElement[1]                                       ## get Stoke coordinate index
                    for FreqElementID, FreqElement in enumerate(LocalFrequencyAxis):        ## loop over frequency channels
                        Freq = FreqElement[0]                                               ## get frequency
                        FreqIndex = FreqElement[1]                                          ## get frequency index
                        for yPointID, yPoint in enumerate(self.RegionPixelsY):              ## loop over all rows
                            yPixelCoord = yPoint[0]                                         ## get current y-pixel coordinate
                            yPixelIndex = yPoint[1]                                         ## get corresponding index
                            yi = yPoint[2]                                                  ## get inverted index
                            for xPointID, xPoint in enumerate(self.RegionPixelsX):          ## loop over all columns
                                xPixelCoord = xPoint[0]                                     ## get current x-pixel coordinate
                                xPixelIndex = xPoint[1]                                     ## get corresponding index
                                if (self.RegionMask[yi, xPixelIndex] != 0):                 ## consider only selected pixels


                                    ## get intensity
                                    npixels += 1
                                    LocalInt = FitFunctionValues[ObsDataFileIndex][npixels][0]
                                    if (LocalChi2SaveFlag):
                                        LocalChi2Value = Chi2Values[ObsDataFileIndex][npixels][0]


                                    ## store intensity and chi^2 value
                                    if (NumDimensions == 1):
                                        NewFITSCube[xPixelIndex] = LocalInt
                                        if (LocalChi2SaveFlag):
                                            NewChi2Cube[xPixelIndex] = LocalChi2Value
                                    elif (NumDimensions == 2):
                                        NewFITSCube[yPixelIndex, xPixelIndex] = LocalInt
                                        if (LocalChi2SaveFlag):
                                            NewChi2Cube[yPixelIndex, xPixelIndex] = LocalChi2Value
                                    elif (NumDimensions == 3):
                                        NewFITSCube[FreqIndex, yPixelIndex, xPixelIndex] = LocalInt
                                        if (LocalChi2SaveFlag):
                                            NewChi2Cube[FreqIndex, yPixelIndex, xPixelIndex] = LocalChi2Value
                                    elif (NumDimensions == 4):
                                        NewFITSCube[StokeCoordIndex, FreqIndex, yPixelIndex, xPixelIndex] = LocalInt
                                        if (LocalChi2SaveFlag):
                                            NewChi2Cube[StokeCoordIndex, FreqIndex, yPixelIndex, xPixelIndex] = LocalChi2Value


            ## store FITS and chi^2 cubes in normal order
            else:
                for xPointID, xPoint in enumerate(self.RegionPixelsX):                      ## loop over all columns
                    xPixelCoord = xPoint[0]                                                 ## get current x-pixel coordinate
                    xPixelIndex = xPoint[1]                                                 ## get corresponding index
                    for yPointID, yPoint in enumerate(self.RegionPixelsY):                  ## loop over all rows
                        yPixelCoord = yPoint[0]                                             ## get current y-pixel coordinate
                        yPixelIndex = yPoint[1]                                             ## get corresponding index
                        yi = yPoint[2]                                                      ## get inverted index
                        if (self.RegionMask[yi, xPixelIndex] != 0):                         ## consider only selected pixels
                            for FreqElementID, FreqElement in enumerate(LocalFrequencyAxis):    ## loop over frequency channels
                                Freq = FreqElement[0]                                       ## get frequency
                                FreqIndex = FreqElement[1]                                  ## get frequency index
                                for StokeElementID, StokeElement in enumerate(LocalStokesAxis): ## loop over Stoke coordinates
                                    StokeCoord = StokeElement[0]                            ## get Stoke coordinate
                                    StokeCoordIndex = StokeElement[1]                       ## get Stoke coordinate index


                                    ## get intensity
                                    npixels += 1
                                    LocalInt = FitFunctionValues[ObsDataFileIndex][npixels][0]
                                    if (LocalChi2SaveFlag):
                                        LocalChi2Value = Chi2Values[ObsDataFileIndex][npixels][0]


                                    ## store intensity and chi^2 value
                                    if (NumDimensions == 1):
                                        NewFITSCube[xPixelIndex] = LocalInt
                                        if (LocalChi2SaveFlag):
                                            NewChi2Cube[xPixelIndex] = LocalChi2Value
                                    elif (NumDimensions == 2):
                                        NewFITSCube[xPixelIndex, yPixelIndex] = LocalInt
                                        if (LocalChi2SaveFlag):
                                            NewChi2Cube[xPixelIndex, yPixelIndex] = LocalChi2Value
                                    elif (NumDimensions == 3):
                                        NewFITSCube[xPixelIndex, yPixelIndex, FreqIndex] = LocalInt
                                        if (LocalChi2SaveFlag):
                                            NewChi2Cube[xPixelIndex, yPixelIndex, FreqIndex] = LocalChi2Value
                                    elif (NumDimensions == 4):
                                        NewFITSCube[xPixelIndex, yPixelIndex, FreqIndex, StokeCoordIndex] = LocalInt
                                        if (LocalChi2SaveFlag):
                                            NewChi2Cube[xPixelIndex, yPixelIndex, FreqIndex, StokeCoordIndex] = LocalChi2Value
            # Debug:
            # print ("\n\nnpixels = ", npixels)
            # print ("LengthExpRange[ObsDataFileIndex] = ", LengthExpRange[ObsDataFileIndex])


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write new FITS cube to file


            ## create hdu
            NewHDU = pyfits.PrimaryHDU(NewFITSCube)


            ## define reference frequency
            PureFrequencyAxis = [Freq[0] for Freq in LocalFrequencyAxis]
            LocalDelta = PureFrequencyAxis[1] - PureFrequencyAxis[0]
            if (NumberRange > 0):
                if (LocalDelta < 0.0):
                    f1 = numpy.abs(PureFrequencyAxis[:] - xmx[RangeID]).argmin()
                else:
                    f1 = numpy.abs(PureFrequencyAxis[:] - xmn[RangeID]).argmin()
            else:
                f1 = 0
            RefFrequency = PureFrequencyAxis[f1]


            ## export FITS image describing parameter map
            NameOfFunction = "LoadExpFile.LoadExp.WriteFits"
            ImageFlag = False
            BUNITINFO = [UnitAixs, 'Brightness (pixel) unit']
            LocalAxisValuesList = FITSParameters['AxisValuesList']                          ## get list of coordinates for each axis
            LocalAxisValuesList[FreqAxisID] = PureFrequencyAxis
            FITSParameters['AxisValuesList'] = LocalAxisValuesList                          ## update frequency axis
            self.ExportFITSFile(NewHDU, NameOfFunction, FITSParameters, ImageFlag, self.CentralPixelOfRegionX, \
                                self.CentralPixelOfRegionIndexX, self.CentralPixelOfRegionY, self.CentralPixelOfRegionIndexY, \
                                self.NumRegionPixelsX, self.NumRegionPixelsY, BUNITINFO, LocalOutputFileNameNew, \
                                RefFrequency = RefFrequency)
            FITSParameters['AxisValuesList'] = AxisValuesList


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write chi2 cube to file
            if (LocalChi2SaveFlag):

                ## create hdu
                NewHDU = pyfits.PrimaryHDU(NewChi2Cube)


                ## export FITS image describing parameter map
                NameOfFunction = "LoadExpFile.LoadExp.WriteFits"
                ImageFlag = False
                BUNITINFO = [UnitAixs, 'Brightness (pixel) unit']
                LocalAxisValuesList = FITSParameters['AxisValuesList']                      ## get list of coordinates for each axis
                LocalAxisValuesList[FreqAxisID] = PureFrequencyAxis
                FITSParameters['AxisValuesList'] = LocalAxisValuesList                      ## update frequency axis
                self.ExportFITSFile(NewHDU, NameOfFunction, FITSParameters, ImageFlag, self.CentralPixelOfRegionX, \
                                    self.CentralPixelOfRegionIndexX, self.CentralPixelOfRegionY, self.CentralPixelOfRegionIndexY, \
                                    self.NumRegionPixelsX, self.NumRegionPixelsY, BUNITINFO, Chi2FileName, \
                                    RefFrequency = RefFrequency)
                FITSParameters['AxisValuesList'] = AxisValuesList

            ## print that you are finished
            if (print_flag != "false"):
                print ("done!")


        ## we're done
        return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------

