!*********************************************************************************************************************************************************
!>  Package: MPI starter program
!>
!>
!>  This module contains the subroutines for MPI main program
!>  Copyright (C) 2009 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    2014-10-06  Initial version
!>  T. Moeller    2014-11-07  Updated version
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!*********************************************************************************************************************************************************



!>********************************************************************************************************************************************************
!> subroutine: getArgument
!>
!> save value of the command-line argument
!>
!> input variables:     NumArg:                 index of the command-line argument
!>
!> output variables:    CommandLineArgument:    contents of the NumArg-th argument
!>
!>
!> \author Thomas Moeller
!>
!> \date 2012-05-30
!>
subroutine getArgument(NumArg, CommandLineArgument)

    ! use iso_fortran_env                                                                     !< only available for fortran 2003 and later

    implicit none
    integer :: NumArg                                                                       !< output variable: index of the command-line argument
    character(len=8192) :: CommandLineArgument                                              !< command-line argument

    CommandLineArgument = ""
    if (COMMAND_ARGUMENT_COUNT() >= NumArg) then                                            !< is the number of command line arguments big enough?
        call getarg(NumArg, CommandLineArgument)
    endif
    return
end subroutine getArgument


!*********************************************************************************************************************************************************
!> program Starter
!>
!> main subroutine which starts the algorithm main subroutine
!>
!>
program Starter

    use Algorithm

    implicit none
    ! ********** input variables **********
    integer :: parameternum                                                                 !< number of model parameter
    integer :: NumberOfFitAlgorithms                                                        !< total number of all algorithms in the chain
    integer :: numiter                                                                      !< max. number of iterations
    integer :: NumFileOrg                                                                   !< number of experimental files
    integer, allocatable, dimension(:) :: lengthexpdataorg                                  !< number of lines in experimental data
    integer, allocatable, dimension(:) :: NumberXColumnsOrg                                 !< number of x-columns for each experimental file
    integer, allocatable, dimension(:) :: NumberYColumnsOrg                                 !< number of y-columns for each experimental file
    integer, allocatable, dimension(:) :: NumberRangesOrg                                   !< number of y-columns for each experimental file
    integer :: MaxColXOrg                                                                   !< number of columns concerning to the experimental x side
    integer :: MaxColYOrg                                                                   !< number of columns concerning to the experimental y side
    integer :: MaxLength                                                                    !< max length of experimental data
    integer :: LastAlgorithmNum                                                             !< flag for screen output 1 (=yes) or 0 (=no)
    integer :: printflagNum                                                                 !< flag for screen output 1 (=yes) or 0 (=no)
    integer :: SortFortranNum                                                               !< flag indicating if chi2 log file is sorted by fortran
                                                                                            !< yes (=1) or not (=0)
    integer :: DeterminationChi2                                                            !< method being used for the determination of chi^2
    integer :: PlotIteration                                                                !< plot model func. for each iteration set 1 (=yes) or 0 (=no)
    integer :: PlotType                                                                     !< get type of plot
    integer :: NumberInputFilesorg                                                          !< number of input files for the external model program
    integer :: NumberOutputFilesOrg                                                         !< number of output files for the external model program
    integer :: ParallelizationFlagorg                                                       !< contains the number of processors used for parallelization
    integer :: JobIDorg                                                                     !< job identification number
    integer :: MaxInputLinesOrg                                                             !< max number of lines in an input file
    integer :: MaxParameterOrg                                                              !< max number of parameters in a line of an input file
    integer :: RenormalizedChi2Org                                                          !< flag for using renormalized chi**2
    integer :: AlgorithmCounter                                                             !< counts number of calls
    integer :: MaxRangeNumber                                                               !< max. number of ranges
    integer :: NumberSites                                                                  !< number of sites
    real*8 :: chilm                                                                         !< user defined abort criteria for chi**2
    real*8, dimension(15) :: GeneralAlgorithmSettings                                       !< array containing special algorithm settings
    real*8, allocatable, dimension(:, :, :) :: expdataxorg                                  !< array containing the experimental x side
    real*8, allocatable, dimension(:, :, :) :: expdatayorg                                  !< array containing the experimental y side
    real*8, allocatable, dimension(:, :, :) :: expdataerrororg                              !< array containing the experimental error of the y side
    real*8, allocatable, dimension(:, :, :) :: MinRangeOrg                                  !< array containing the minimal exp. ranges
    real*8, allocatable, dimension(:, :, :) :: MaxRangeOrg                                  !< array containing the maximal exp. ranges
    character(len=8192) :: fitlog                                                           !< path for log-file containing the current values of chi**2
    character(len=256) :: xAxisLabel                                                        !< label of the x-axis (for plot)
    character(len=256) :: yAxisLabel                                                        !< label of the y-axis (for plot)
    character(len=256) :: zAxisLabel                                                        !< label of the z-axis (for plot)
    character(len=20) :: CalculationMethodOrg                                               !< method of computation
    character(len=8192) :: PathStartScriptOrg                                               !< command for calling model function
    character(len=8192) :: ExeCommandStartScriptOrg                                         !< command for calling model function
    character(len=8192) :: currentpathorg                                                   !< path of the working directory
    character(len=512), allocatable, dimension(:) :: FitParameterNameLocal                  !< array containing the names of the model parameters
    character(len=512), allocatable, dimension(:) :: FitParameterValueLocal                 !< array containing the values of the model param. as string


    ! ********** in/output variables **********
    real*8, allocatable, dimension(:, :) :: parametersetorg                                 !< the following line is necessary for f2py


    ! ********** output variables **********
    integer :: calstatus                                                                    !< the following line is necessary for f2py
    real*8, allocatable, dimension(:, :) :: FinalParameterSet                               !< array containing the optimized parameter set
    real*8, allocatable, dimension(:, :, :, :) :: FitFunctionOut                            !< values of the model function at the calculated points
    real*8, allocatable, dimension(:, :, :, :) :: Chi2Values                                !< values of the model function at the calculated points


    ! ********** working variabels **********
    integer :: i, j, k, l                                                                   !< working variables
    integer :: allocstatus, deallocstatus                                                   !< working variables for allocation/deallocation
    character(len=100) :: HelpString                                                        !< working variable


    !< for command line arguments
    integer, parameter :: debug = 1                                                         !< for debugging
    integer :: NumberOfCommandLineArguments                                                 !< number of command-line arguments
    character(len=8192) :: CommandLineArgument                                              !< command-line argument


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< Initializes the MPI execution environment
    !< MPI_INIT(IERROR)
    !<  INTEGER :: IERROR
    !<
    !<  IERROR:      (output)    Fortran only: Error status (integer).
    !<
    !< Description:  This routine, or MPI_Init_thread, must be called before any other MPI routine (apart from MPI_Initialized) is called. MPI can be
    !<               initialized at most once; subsequent calls to MPI_Init or MPI_Init_thread are erroneous. All MPI programs must contain a call to
    !<               MPI_Init or MPI_Init_thread.
    !<
    !< Error:        Almost all MPI routines return an error value; C routines as the value of the function and Fortran routines in the last argument.
    !<               C++ functions Do not return errors. If the default error handler is set to MPI::ERRORS_THROW_EXCEPTIONS, then on error the
    !<               C++ exception mechanism will be used to throw an MPI::Exception object.
    !<               Before the error value is returned, the current MPI error handler is called. By default, this error handler aborts the MPI job,
    !<               except for I/O function errors. The error handler may be changed with MPI_Comm_set_errhandler; the predefined error handler
    !<               MPI_ERRORS_RETURN may be used to cause error values to be returned. Note that MPI does not guarantee that an MPI program can
    !<               continue past an error.
    !<               See the MPI man page for a full list of MPI error codes.
    call MPI_Init(ierr)
    if (ierr /= MPI_SUCCESS) then
        print '(" ")'
        print '("Error in program Starter:")'
        print '(2x,"Error occurred starting MPI program!")'
        print '(" ")'
        print '(2x, "Program aborted!")'
        print '(" ")'


        !< Terminates MPI execution environment.
        !< MPI_ABORT(COMM, ERRORCODE, IERROR)
        !<  INTEGER :: COMM, ERRORCODE, IERROR
        !<
        !<  comm:        (input)     Communicator of tasks to abort.
        !<  errorcode:   (input)     Error code to return to invoking environment.
        !<  IERROR:      (output)    Fortran only: Error status (integer).
        !<
        !< Description:  This routine makes a "best attempt" to abort all tasks in the group of comm. This function does not require that the
        !<               invoking environment take any action with the error code. However, a UNIX or POSIX environment should handle this as a
        !<               return errorcode from the main program or an abort (errorcode).
        !<               The long-term goal of the Open MPI implementation is to terminate all processes in all tasks that contain a process in
        !<               comm, and the error code is not returned to the invoking environment. At the moment, this isn’t fully implemented and
        !<               MPI_Abort will terminate the entire job. The Java implementation currently does not allow specification of a
        !<               communicator and aborts the entire job - this will be updated to support the long-term implementation at a later date.
        !<               Note: All associated processes are sent a SIGTERM.
        !<
        !< Error:        Almost all MPI routines return an error value; C routines as the value of the function and Fortran routines in the last
        !<               argument. C++ and Java functions Do not return errors. If the default error handler is set to MPI::ERRORS_THROW_EXCEPTIONS,
        !<               then on error the C++ exception mechanism will be used to throw an MPI::Exception object - similar behavior is followed
        !<               by Java.
        !<               Before the error value is returned, the current MPI error handler is called. By default, this error handler aborts the
        !<               MPI job, except for I/O function errors. The error handler may be changed with MPI_Comm_set_errhandler; the predefined
        !<               error handler MPI_ERRORS_RETURN may be used to cause error values to be returned. Note that MPI does not guarantee that
        !<               an MPI program can continue past an error.
        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
    endif


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< Returns the size of the group associated with a communicator, i.e. get the number of processors this job is using:
    !< MPI_COMM_SIZE(COMM, SIZE, IERROR)
    !<  INTEGER :: COMM, SIZE, IERROR
    !<
    !<  comm:        (input)     Communicator (handle).
    !<  size:        (output)    Number of processes in the group of comm (integer).
    !<  IERROR:      (output)    Fortran only: Error status (integer).
    !<
    !< Description:  This function indicates the number of processes involved in a communicator. For MPI_COMM_WORLD, it indicates the total number of
    !<               processes available. This function is equivalent to accessing the communicator’s group with MPI_Comm_group, computing the size
    !<               using MPI_Group_size, and then freeing the temporary group via MPI_Group_free. If the communicator is an inter-communicator
    !<               (enables communication between two groups), this function returns the size of the local group. To return the size of the remote
    !<               group, use the MPI_Comm_remote_size function.
    !<               This call is often used with MPI_Comm_rank to determine the amount of concurrency available for a specific library or program.
    !<               MPI_Comm_rank indicates the rank of the process that calls it in the range from 0 . . . size-1, where size is the return value
    !<               of MPI_Comm_size.
    !<
    !< Error:        Almost all MPI routines return an error value; C routines as the value of the function and Fortran routines in the last argument.
    !<               C++ functions Do not return errors. If the default error handler is set to MPI::ERRORS_THROW_EXCEPTIONS, then on error the C++
    !<               exception mechanism will be used to throw an MPI::Exception object.
    !<               Before the error value is returned, the current MPI error handler is called. By default, this error handler aborts the MPI job,
    !<               except for I/O function errors. The error handler may be changed with MPI_Comm_set_errhandler; the predefined error handler
    !<               MPI_ERRORS_RETURN may be used to cause error values to be returned. Note that MPI does not guarantee that an MPI program can
    !<               continue past an error.
    call MPI_Comm_size(MPI_COMM_WORLD, nsize, ierr)


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< Determines the rank of the calling process in the communicator, i.e. get the rank of the processor this thread is running on. (Each processor
    !< has a unique rank.)
    !< MPI_COMM_RANK(COMM, RANK, IERROR)
    !<  INTEGER :: COMM, RANK, IERROR
    !<
    !<  comm:        (input)     Communicator (handle).
    !<  rank:        (output)    Rank of the calling process in group of comm (integer).
    !<  IERROR:      (output)    Fortran only: Error status (integer).
    !<
    !< Description:  This function gives the rank of the process in the particular communicator’s group. It is equivalent to accessing the
    !<               communicator’s group with MPI_Comm_group, computing the rank using MPI_Group_rank, and then freeing the temporary group via
    !<               MPI_Group_free.
    !<               Many programs will be written with the master-slave model, where one process (such as the rank-zero process) will play a
    !<               supervisory role, and the other processes will serve as compute nodes. In this framework, MPI_Comm_size and MPI_Comm_rank are
    !<               useful for determining the roles of the various processes of a communicator.
    !<
    !< Error:        Almost all MPI routines return an error value; C routines as the value of the function and Fortran routines in the last
    !<               argument. C++ functions Do not return errors. If the default error handler is set to MPI::ERRORS_THROW_EXCEPTIONS, then on
    !<               error the C++ exception mechanism will be used to throw an MPI:Exception object.
    !<               Before the error value is returned, the current MPI error handler is called. By default, this error handler aborts the MPI job,
    !<               except for I/O function errors. The error handler may be changed with MPI_Comm_set_errhandler; the predefined error handler
    !<               MPI_ERRORS_RETURN may be used to cause error values to be returned. Note that MPI does not guarantee that an MPI program can
    !<               continue past an error.
    call MPI_Comm_rank(MPI_COMM_WORLD, myrank, ierr)


    !< send all processes except master to sleep
    if (myrank /= 0) then
        call SlaveLoop


    !< the next lines are executed only by the master thread
    else


        !-------------------------------------------------------------------------------------------------------------------------------------------------
        !< set temp-directory
        TempDirectory = " "
        CALL GetEnv('MAGIXTempDirectory', TempDirectory)


        !-------------------------------------------------------------------------------------------------------------------------------------------------
        !< get dimensions from command line
        NumberOfCommandLineArguments = COMMAND_ARGUMENT_COUNT()                             !< get number of command-line arguments
        if (NumberOfCommandLineArguments < 1) then                                          !< continue if
            print '(" ")'
            print '("Error in program Starter:")'
            print '(2x,"There are no comment line arguments defined.")'
            print '(" ")'
            stop ' Program aborted!'
        else


            !< get NumFileOrg (python: NumberExpFiles)
            NumFileOrg = 0
            call getArgument(1, CommandLineArgument)
            read(CommandLineArgument,*) NumFileOrg


            !< get MaxLength (python: MaxLength)
            MaxLength = 0
            call getArgument(2, CommandLineArgument)
            read(CommandLineArgument,*) MaxLength


            !< get MaxColXOrg (python: MaxColX)
            MaxColXOrg = 0
            call getArgument(3, CommandLineArgument)
            read(CommandLineArgument,*) MaxColXOrg


            !< get MaxColYOrg (python: MaxColY)
            MaxColYOrg = 0
            call getArgument(4, CommandLineArgument)
            read(CommandLineArgument,*) MaxColYOrg


            !< get MaxRangeNumber (python: MaxRangeNumber)
            MaxRangeNumber = 0
            call getArgument(5, CommandLineArgument)
            read(CommandLineArgument,*) MaxRangeNumber


            !< get parameternum (python: NumberParameter)
            parameternum = 0
            call getArgument(6, CommandLineArgument)
            read(CommandLineArgument,*) parameternum


            !< get parameternum (python: ParamSetCounter)
            NumberSites = 0
            call getArgument(7, CommandLineArgument)
            read(CommandLineArgument,*) NumberSites


            !< get job ID (python: JobID)
            JobIDorg = 0
            call getArgument(8, CommandLineArgument)
            read(CommandLineArgument,*) JobIDorg
        endif

        ! Debug:
        !    print*,'NumFileOrg = ', NumFileOrg
        !    print*,'MaxLength = ', MaxLength
        !    print*,'MaxColXOrg = ', MaxColXOrg
        !    print*,'MaxColYOrg = ', MaxColYOrg
        !    print*,'MaxRangeNumber = ', MaxRangeNumber
        !    print*,'parameternum = ', parameternum
        !    print*,'NumberSites = ', NumberSites
        !    print*,'JobIDorg = ', JobIDorg


        !<------------------------------------------------------------------------------------------------------------------------------------------------
        !< (de)allocate memory
        if (allocated(lengthexpdataorg)) then
            deallocate(lengthexpdataorg, NumberXColumnsOrg, NumberYColumnsOrg, NumberRangesOrg, expdataxorg, expdatayorg, expdataerrororg, MinRangeOrg, &
                       MaxRangeOrg, parametersetorg, FinalParameterSet, FitFunctionOut, Chi2Values, FitParameterNameLocal, FitParameterValueLocal, &
                       stat = deallocstatus)
            if (deallocstatus /= 0) then                                                    !< is all ok?
                print '(" ")'
                print '("Error in program Starter:")'
                print '(2x,"Can not deallocate variables lengthexpdataorg etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("deallocstatus = ",I4)',deallocstatus
                print '(" ")'


                !< Terminates MPI execution environment.
                !< MPI_ABORT(COMM, ERRORCODE, IERROR)
                !<  INTEGER :: COMM, ERRORCODE, IERROR
                !<
                !<  comm:        (input)     Communicator of tasks to abort.
                !<  errorcode:   (input)     Error code to return to invoking environment.
                !<  IERROR:      (output)    Fortran only: Error status (integer).
                !<
                !< Description:  This routine makes a "best attempt" to abort all tasks in the group of comm. This function does not require that the
                !<               invoking environment take any action with the error code. However, a UNIX or POSIX environment should handle this as a
                !<               return errorcode from the main program or an abort (errorcode).
                !<               The long-term goal of the Open MPI implementation is to terminate all processes in all tasks that contain a process in
                !<               comm, and the error code is not returned to the invoking environment. At the moment, this isn’t fully implemented and
                !<               MPI_Abort will terminate the entire job. The Java implementation currently does not allow specification of a
                !<               communicator and aborts the entire job - this will be updated to support the long-term implementation at a later date.
                !<               Note: All associated processes are sent a SIGTERM.
                !<
                !< Error:        Almost all MPI routines return an error value; C routines as the value of the function and Fortran routines in the last
                !<               argument. C++ and Java functions Do not return errors. If the default error handler is set to
                !<               MPI::ERRORS_THROW_EXCEPTIONS, then on error the C++ exception mechanism will be used to throw an MPI::Exception object
                !<               - similar behavior is followed by Java.
                !<               Before the error value is returned, the current MPI error handler is called. By default, this error handler aborts the
                !<               MPI job, except for I/O function errors. The error handler may be changed with MPI_Comm_set_errhandler; the predefined
                !<               error handler MPI_ERRORS_RETURN may be used to cause error values to be returned. Note that MPI does not guarantee that
                !<               an MPI program can continue past an error.
                ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                stop ' Program aborted!'
            endif
        endif
        allocate(lengthexpdataorg(NumFileOrg), NumberXColumnsOrg(NumFileOrg), NumberYColumnsOrg(NumFileOrg), NumberRangesOrg(NumFileOrg), &
                 expdataxorg(NumFileOrg, MaxLength, MaxColXOrg), expdatayorg(NumFileOrg, MaxLength, MaxColYOrg), &
                 expdataerrororg(NumFileOrg, MaxLength, MaxColYOrg), MinRangeOrg(NumFileOrg, MaxRangeNumber, MaxColXOrg), &
                 MaxRangeOrg(NumFileOrg, MaxRangeNumber, MaxColXOrg), parametersetorg(4, parameternum), FinalParameterSet(NumberSites, parameternum), &
                 FitParameterNameLocal(parameternum), FitParameterValueLocal(parameternum), &
                 FitFunctionOut(NumberSites, NumFileOrg, MaxLength, MaxColYOrg), Chi2Values(NumberSites, NumFileOrg, MaxLength, MaxColYOrg), &
                 stat = allocstatus)
        if (allocstatus /= 0) then                                                          !< is all ok?
            print '(" ")'
            print '("Error in program Starter:")'
            print '(2x,"Can not allocate variables lengthexpdataorg etc.")'
            print '(2x,"Please close all other programs and restart the program!")'
            print '(" ")'
            print '("allocstatus = ",I4)',allocstatus
            print '(" ")'


            !< Terminates MPI execution environment.
            !< MPI_ABORT(COMM, ERRORCODE, IERROR)
            !<  INTEGER :: COMM, ERRORCODE, IERROR
            !<
            !<  comm:        (input)     Communicator of tasks to abort.
            !<  errorcode:   (input)     Error code to return to invoking environment.
            !<  IERROR:      (output)    Fortran only: Error status (integer).
            !<
            !< Description:  This routine makes a "best attempt" to abort all tasks in the group of comm. This function does not require that the
            !<               invoking environment take any action with the error code. However, a UNIX or POSIX environment should handle this as a
            !<               return errorcode from the main program or an abort (errorcode).
            !<               The long-term goal of the Open MPI implementation is to terminate all processes in all tasks that contain a process in
            !<               comm, and the error code is not returned to the invoking environment. At the moment, this isn’t fully implemented and
            !<               MPI_Abort will terminate the entire job. The Java implementation currently does not allow specification of a
            !<               communicator and aborts the entire job - this will be updated to support the long-term implementation at a later date.
            !<               Note: All associated processes are sent a SIGTERM.
            !<
            !< Error:        Almost all MPI routines return an error value; C routines as the value of the function and Fortran routines in the last
            !<               argument. C++ and Java functions Do not return errors. If the default error handler is set to MPI::ERRORS_THROW_EXCEPTIONS,
            !<               then on error the C++ exception mechanism will be used to throw an MPI::Exception object - similar behavior is followed
            !<               by Java.
            !<               Before the error value is returned, the current MPI error handler is called. By default, this error handler aborts the
            !<               MPI job, except for I/O function errors. The error handler may be changed with MPI_Comm_set_errhandler; the predefined
            !<               error handler MPI_ERRORS_RETURN may be used to cause error values to be returned. Note that MPI does not guarantee that
            !<               an MPI program can continue past an error.
            ! call MPI_Abort(MPI_COMM_WORLD, ierr)
            call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
            stop ' Program aborted!'
        endif
        lengthexpdataorg = 0
        NumberXColumnsOrg = 0
        NumberYColumnsOrg = 0
        NumberRangesOrg = 0
        expdataxorg = 0.d0
        expdatayorg = 0.d0
        expdataerrororg = 0.d0
        MinRangeOrg = 0.d0
        MaxRangeOrg = 0.d0
        parametersetorg = 0.d0
        FinalParameterSet = 0.d0
        FitFunctionOut = 0.d0
        Chi2Values = 0.d0
        FitParameterNameLocal = ""
        FitParameterValueLocal = ""


        !<------------------------------------------------------------------------------------------------------------------------------------------------
        !< read in parameter values from scratch file
        write(HelpString,'(I100)') JobIDorg                                                 !< write JobID to string


        !<================================================================================================================================================
        !< read python variables from temp file
        !<================================================================================================================================================


        !<------------------------------------------------------------------------------------------------------------------------------------------------
        !< read general input variables
        open(11111, file = trim(adjustl(TempDirectory)) // "/job_" // trim(adjustl(HelpString)) // "/general-variables__input.dat", status = 'UNKNOWN')


        !< general INTEGER variables
        read(11111, *) printflagNum
        read(11111, *) NumberOfFitAlgorithms
        read(11111, *) NumberInputFilesorg
        read(11111, *) NumberOutputFilesOrg
        read(11111, *) MaxInputLinesOrg
        read(11111, *) MaxParameterOrg
        Do i = 1, NumFileOrg
            read(11111, *) lengthexpdataorg(i), NumberXColumnsOrg(i), NumberYColumnsOrg(i), NumberRangesOrg(i)
        end Do


        !< general REAL*8 variables
        Do i = 1, NumFileOrg
            Do j = 1, MaxLength
                Do k = 1, MaxColXOrg
                    read(11111, *) expdataxorg(i, j, k)
                end Do
                Do k = 1, MaxColYOrg
                    read(11111, *) expdatayorg(i, j, k), expdataerrororg(i, j, k)
                end Do
            end Do
            Do j = 1, MaxRangeNumber
                Do k = 1, MaxColXOrg
                    read(11111, *) MinRangeOrg(i, j, k), MaxRangeOrg(i, j, k)
                end Do
            end Do
        end Do


        !< general CHARACTER variables
        read(11111, '(A)') fitlog
        read(11111, '(A)') CalculationMethodOrg
        read(11111, '(A)') PathStartScriptOrg
        read(11111, '(A)') ExeCommandStartScriptOrg
        read(11111, '(A)') currentpathorg
        Do k = 1, parameternum
            read(11111, '(A)') FitParameterNameLocal(k)
            read(11111, '(A)') FitParameterValueLocal(k)
        end Do
        close(11111)

        ! Debug:
        !    print*,'(FORTRAN): General input variables:'
        !    print*,'printflagNum = ', printflagNum
        !    print*,'NumberOfFitAlgorithms = ', NumberOfFitAlgorithms
        !    print*,'NumberInputFilesorg = ', NumberInputFilesorg
        !    print*,'NumberOutputFilesOrg = ', NumberOutputFilesOrg
        !    print*,'MaxInputLinesOrg = ', MaxInputLinesOrg
        !    print*,'MaxParameterOrg = ', MaxParameterOrg
        !    print*,'lengthexpdataorg(:) = ', lengthexpdataorg(:)
        !    print*,'NumberXColumnsOrg(:) = ', NumberXColumnsOrg(:)
        !    print*,'NumberYColumnsOrg(:) = ', NumberYColumnsOrg(:)
        !    print*,'NumberRangesOrg(:) = ', NumberRangesOrg(:)
        !    print*,'expdataxorg(1, 1, 1) = ', expdataxorg(1, 1, 1)
        !    print*,'expdataxorg(1, MaxLength, 1) = ', expdataxorg(1, MaxLength, 1)
        !    print*,'expdatayorg(1, 1, 1) = ', expdatayorg(1, 1, 1)
        !    print*,'expdatayorg(1, MaxLength, 1) = ', expdatayorg(1, MaxLength, 1)
        !    print*,'expdataerrororg(1, 1, 1) = ', expdataerrororg(1, 1, 1)
        !    print*,'expdataerrororg(1, MaxLength, 1) = ', expdataerrororg(1, MaxLength, 1)
        !    print*,'MinRangeOrg(:, :, :) = ', MinRangeOrg(:, :, :)
        !    print*,'MaxRangeOrg(:, :, :) = ', MaxRangeOrg(:, :, :)
        !    print*,'fitlog = >', trim(adjustl(fitlog)), '<'
        !    print*,'CalculationMethodOrg = >', trim(adjustl(CalculationMethodOrg)), '<'
        !    print*,'PathStartScriptOrg = >', trim(adjustl(PathStartScriptOrg)), '<'
        !    print*,'ExeCommandStartScriptOrg = >', trim(adjustl(ExeCommandStartScriptOrg)), '<'
        !    print*,'currentpathorg = >', trim(adjustl(currentpathorg)), '<'
        !    print*,'FitParameterNameLocal(1) = >', trim(adjustl(FitParameterNameLocal(1))), '<'
        !    print*,'FitParameterNameLocal(parameternum) = >', trim(adjustl(FitParameterNameLocal(parameternum))), '<'
        !    print*,'FitParameterValueLocal(1) = >', trim(adjustl(FitParameterValueLocal(1))), '<'
        !    print*,'FitParameterValueLocal(parameternum) = >', trim(adjustl(FitParameterValueLocal(parameternum))), '<'
        !    print*,' '
        !    print*,' '
        !    print*,' '


        !<------------------------------------------------------------------------------------------------------------------------------------------------
        !< read algorithm specific input variables
        open(11111, file = trim(adjustl(TempDirectory)) // "/job_" // trim(adjustl(HelpString)) // "/specific-alg-variables__input.dat", &
                                                                                                                       status = 'UNKNOWN')

        !< algorithm specific INTEGER variables
        read(11111, *) LastAlgorithmNum
        read(11111, *) numiter
        read(11111, *) ParallelizationFlagorg
        read(11111, *) DeterminationChi2
        read(11111, *) PlotIteration
        read(11111, *) PlotType
        read(11111, *) RenormalizedChi2Org
        read(11111, *) AlgorithmCounter
        read(11111, *) SortFortranNum


        !< algorithm specific REAL*8 variables
        read(11111, *) chilm
        read(11111, *) (GeneralAlgorithmSettings(i), i = 1, 15)
        Do i = 1, parameternum
            read(11111, *) (parametersetorg(j, i), j = 1, 4)
        end Do


        !< algorithm specific CHARACTER variables
        read(11111, '(A)') xAxisLabel
        read(11111, '(A)') yAxisLabel
        read(11111, '(A)') zAxisLabel
        close(11111, status = 'delete')

        ! Debug:
        !    print*,'(FORTRAN): Algorithm specific input variables:'
        !    print*,'LastAlgorithmNum = ', LastAlgorithmNum
        !    print*,'numiter = ', numiter
        !    print*,'ParallelizationFlagorg = ', ParallelizationFlagorg
        !    print*,'DeterminationChi2 = ', DeterminationChi2
        !    print*,'PlotIteration = ', PlotIteration
        !    print*,'PlotType = ', PlotType
        !    print*,'RenormalizedChi2Org = ', RenormalizedChi2Org
        !    print*,'AlgorithmCounter = ', AlgorithmCounter
        !    print*,'SortFortranNum = ', SortFortranNum
        !    print*,'chilm = ', chilm
        !    print*,'GeneralAlgorithmSettings(:) = ', GeneralAlgorithmSettings(:)
        !    print*,'parametersetorg(:, 1) = ', parametersetorg(:, 1)
        !    print*,'parametersetorg(:, parameternum) = ', parametersetorg(:, parameternum)
        !    print*,'xAxisLabel = >', trim(adjustl(xAxisLabel)), '<'
        !    print*,'yAxisLabel = >', trim(adjustl(yAxisLabel)), '<'
        !    print*,'zAxisLabel = >', trim(adjustl(zAxisLabel)), '<'


        !<================================================================================================================================================
        !< call main program of algorithm package
        !<================================================================================================================================================
        call MainAlg(printflagNum, LastAlgorithmNum, calstatus, FitFunctionOut, Chi2Values, chilm, NumberOfFitAlgorithms, numiter, AlgorithmCounter, &
                     NumberSites, GeneralAlgorithmSettings, DeterminationChi2, PlotIteration, PlotType, fitlog, NumberInputFilesorg, &
                     NumberOutputFilesOrg, ParallelizationFlagorg, JobIDorg, MaxInputLinesOrg, MaxParameterOrg, RenormalizedChi2Org, currentpathorg, &
                     FitParameterNameLocal, FitParameterValueLocal, CalculationMethodOrg, xAxisLabel, yAxisLabel, zAxisLabel, PathStartScriptOrg, &
                     ExeCommandStartScriptOrg, parametersetorg, FinalParameterSet, expdataxorg, expdatayorg, expdataerrororg, NumberRangesOrg, &
                     MinRangeOrg, MaxRangeOrg, NumberXColumnsOrg, NumberYColumnsOrg, lengthexpdataorg, MaxRangeNumber, NumFileOrg, MaxLength, &
                     MaxColXOrg, MaxColYOrg, parameternum, SortFortranNum)

        ! Debug:
        !    print*,'(FORTRAN): Result output variables:'
        !    print*,'calstatus = ', calstatus
        !    print*,'FinalParameterSet = ', FinalParameterSet
        !    print*,'FitFunctionOut(1, 1, 1, 1) = ', FitFunctionOut(1, 1, 1, 1)
        !    print*,'FitFunctionOut(1, 1, MaxLength, 1) = ', FitFunctionOut(1, 1, MaxLength, 1)
        !    print*,'Chi2Values(1, 1, 1, 1) = ', Chi2Values(1, 1, 1, 1)
        !    print*,'Chi2Values(1, 1, MaxLength, 1) = ', Chi2Values(1, 1, MaxLength, 1)
        !    print*,'parametersetorg(:, 1) = ', parametersetorg(:, 1)
        !    print*,'parametersetorg(:, parameternum) = ', parametersetorg(:, parameternum)


        !<================================================================================================================================================
        !< write return parameters to (un)formatted scratch file
        !<================================================================================================================================================
        open(11111, file = trim(adjustl(TempDirectory)) // "/job_" // trim(adjustl(HelpString)) // "/result-variables__output.dat", status = 'UNKNOWN')
        write(11111, '(I30)') calstatus
        Do i = 1, NumberSites
            Do j = 1, parameternum
                write(11111, '(ES27.15E3)') FinalParameterSet(i, j)
            end Do
            Do j = 1, NumFileOrg
                Do k = 1, MaxLength
                    Do l = 1, MaxColYOrg
                        write(11111, '(2(1x, ES27.15E3))') FitFunctionOut(i, j, k, l), Chi2Values(i, j, k, l)
                    end Do
                end Do
            end Do
        end Do
        Do i = 1, parameternum
            write(11111, '(4(1x, ES27.15E3))') (parametersetorg(j, i), j = 1, 4)
        end Do
        close(11111)


        !<================================================================================================================================================
        !< free memory of variables used by starter program
        !<================================================================================================================================================
        if (allocated(lengthexpdataorg)) deallocate(lengthexpdataorg, stat = deallocstatus)
        if (allocated(NumberXColumnsOrg)) deallocate(NumberXColumnsOrg, stat = deallocstatus)
        if (allocated(NumberYColumnsOrg)) deallocate(NumberYColumnsOrg, stat = deallocstatus)
        if (allocated(NumberRangesOrg)) deallocate(NumberRangesOrg, stat = deallocstatus)
        if (allocated(expdataxorg)) deallocate(expdataxorg, stat = deallocstatus)
        if (allocated(expdatayorg)) deallocate(expdatayorg, stat = deallocstatus)
        if (allocated(expdataerrororg)) deallocate(expdataerrororg, stat = deallocstatus)
        if (allocated(MinRangeOrg)) deallocate(MinRangeOrg, stat = deallocstatus)
        if (allocated(MaxRangeOrg)) deallocate(MaxRangeOrg, stat = deallocstatus)
        if (allocated(parametersetorg)) deallocate(parametersetorg, stat = deallocstatus)
        if (allocated(FinalParameterSet)) deallocate(FinalParameterSet, stat = deallocstatus)
        if (allocated(FitFunctionOut)) deallocate(FitFunctionOut, stat = deallocstatus)
        if (allocated(Chi2Values)) deallocate(Chi2Values, stat = deallocstatus)
        if (deallocstatus /= 0) then
            write(logchannel,*)
            write(logchannel,'("Error in program Starter:")')
            write(logchannel,'(2x,"Can not deallocate variabes lengthexpdataorg etc.")')
            write(logchannel,*)
            write(logchannel,'("deallocstatus = ",I4)') deallocstatus
            write(logchannel,'(" ")')
            write(logchannel,'("Program aborted!")')

            print '(" ")'
            print '("Error in program Starter:")'
            print '(2x,"Can not deallocate variables lengthexpdataorg etc.")'
            print '(" ")'
            print '("deallocstatus = ",I4)', deallocstatus
            print '(" ")'


            !< Terminates MPI execution environment.
            !< MPI_ABORT(COMM, ERRORCODE, IERROR)
            !<  INTEGER :: COMM, ERRORCODE, IERROR
            !<
            !<  comm:        (input)     Communicator of tasks to abort.
            !<  errorcode:   (input)     Error code to return to invoking environment.
            !<  IERROR:      (output)    Fortran only: Error status (integer).
            !<
            !< Description:  This routine makes a "best attempt" to abort all tasks in the group of comm. This function does not require that the
            !<               invoking environment take any action with the error code. However, a UNIX or POSIX environment should handle this as a
            !<               return errorcode from the main program or an abort (errorcode).
            !<               The long-term goal of the Open MPI implementation is to terminate all processes in all tasks that contain a process in
            !<               comm, and the error code is not returned to the invoking environment. At the moment, this isn’t fully implemented and
            !<               MPI_Abort will terminate the entire job. The Java implementation currently does not allow specification of a
            !<               communicator and aborts the entire job - this will be updated to support the long-term implementation at a later date.
            !<               Note: All associated processes are sent a SIGTERM.
            !<
            !< Error:        Almost all MPI routines return an error value; C routines as the value of the function and Fortran routines in the last
            !<               argument. C++ and Java functions Do not return errors. If the default error handler is set to MPI::ERRORS_THROW_EXCEPTIONS,
            !<               then on error the C++ exception mechanism will be used to throw an MPI::Exception object - similar behavior is followed
            !<               by Java.
            !<               Before the error value is returned, the current MPI error handler is called. By default, this error handler aborts the
            !<               MPI job, except for I/O function errors. The error handler may be changed with MPI_Comm_set_errhandler; the predefined
            !<               error handler MPI_ERRORS_RETURN may be used to cause error values to be returned. Note that MPI does not guarantee that
            !<               an MPI program can continue past an error.
            ! call MPI_Abort(MPI_COMM_WORLD, ierr)
            call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
            stop ' Program aborted!'
        endif
    endif


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< Terminates MPI execution environment.
    !< MPI_FINALIZE(IERROR)
    !<  INTEGER :: IERROR
    !<
    !<  IERROR:      (output)    Fortran only: Error status (integer).
    !<
    !< Description:  This routine cleans up all MPI states. Once this routine is called, no MPI routine (not even MPI_Init) may be called, except for
    !<               MPI_Get_version, MPI_Initialized, and MPI_Finalized. Unless there has been a call to MPI_Abort, you must ensure that all pending
    !<               communications involving a process are complete before the process calls MPI_Finalize. If the call returns, each process may
    !<               either continue local computations or exit without participating in further communication with other processes. At the moment when
    !<               the last process calls MPI_Finalize, all pending sends must be matched by a receive, and all pending receives must be matched by
    !<               a send.
    !<               MPI_Finalize is collective over all connected processes. If no processes were spawned, accepted, or connected, then this means it
    !<               is collective over MPI_COMM_WORLD. Otherwise, it is collective over the union of all processes that have been and continue to
    !<               be connected.
    !<
    !< Error:        Almost all MPI routines return an error value; C routines as the value of the function and Fortran routines in the last argument.
    !<               C++ functions Do not return errors. If the default error handler is set to MPI::ERRORS_THROW_EXCEPTIONS, then on error the
    !<               C++ exception mechanism will be used to throw an MPI::Exception object.
    !<               Before the error value is returned, the current MPI error handler is called. By default, this error handler aborts the MPI job,
    !<               except for I/O function errors. The error handler may be changed with MPI_Comm_set_errhandler; the predefined error handler
    !<               MPI_ERRORS_RETURN may be used to cause error values to be returned. Note that MPI does not guarantee that an MPI program can
    !<               continue past an error.
    call MPI_Finalize(ierr)
end program Starter
!---------------------------------------------------------------------------------------------------------------------------------------------------------

