!*********************************************************************************************************************************************************
!>  Module: Model
!>
!>
!>  This module contains subroutines for the myXCLASS program for application in MPI version of MAGIX
!>  Copyright (C) 2009 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine SlaveWork:                     wake up MPI slave threads and do something or fall asleep
!>      - subroutine SlaveLoop:                     manage subroutine for slave threads
!>      - subroutine broadcast_scalar_int:          broadcasts a scalar integer variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_scalar_dbl:          broadcasts a scalar real variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_scalar_log:          broadcasts a scalar logical variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_scalar_char:         broadcasts a scalar character variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_int_1d:        broadcasts an 1d-array integer variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_dbl_1d:        broadcasts an 1d-array real variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_log_1d:        broadcasts an 1d-array logical variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_char_1d:       broadcasts an 1d-array character variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_int_2d:        broadcasts an 2d-array integer variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_dbl_2d:        broadcasts an 2d-array real variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_log_2d:        broadcasts an 2d-array logical variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_char_2d:       broadcasts an 2d-array character variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_int_3d:        broadcasts an 3d-array integer variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_dbl_3d:        broadcasts an 3d-array real variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_log_3d:        broadcasts an 3d-array logical variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_char_3d:       broadcasts an 3d-array character variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_int_4d:        broadcasts an 4d-array integer variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_dbl_4d:        broadcasts an 4d-array real variable from one processor (root_tn) to all other processors
!>      - subroutine broadcast_array_dbl_5d:        broadcasts an 5d-array real variable from one processor (root_tn) to all other processors
!>      - subroutine BroadCastMPIVariables:         broadcast MPI variables
!>      - subroutine ModelInit:                     initialize and broadcast module variables
!>      - subroutine PrepareInputFiles:             prepare input files for MPI call of external model programs using MPI_Spawn_Multiple function
!>      - subroutine ModelCalcChiFunctionLM:        calculates the chi2 values for the Levenberg-Marquard algorithm
!>      - subroutine ModelCalcChiFunctionGeneral:   calculates the chi2 values for several given parameter vector sets
!>      - subroutine ModelCalcChiFunction:          prepare call of subroutine 'ModelCalcCore'
!>      - subroutine ModelCalcCore:                 calculates the chi2 values for a given set of parameter vectors
!>      - subroutine ModelParamFree:                free memory used by variables of the Module Model
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    2014-09-12  Initial version
!>  T. Moeller    2021-01-05  improved broadcast handling
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module Model
    !> declare variables which are used by functions and subroutines of this module

    use mpi
    use myXCLASSCore

    implicit none


    !< MPI variables
    ! include 'mpif.h'
    integer, parameter :: root_tn = 0                                                       !< thread number of root thread
    integer :: ierr                                                                         !< MPI error variable
    integer :: errorcode                                                                    !< MPI error variable
    integer :: myrank                                                                       !< current thread number
    integer :: nsize                                                                        !< MPI variables
    integer :: nproc_for_mpi                                                                !< MPI variables
    integer :: icode0                                                                       !< MPI variables
    integer :: ireq0                                                                        !< MPI variables
    integer :: itag0                                                                        !< MPI variables
    integer :: TotalNumParamVectors                                                         !< used for subroutine ModelCalcCore for exchanging data
    integer, dimension(MPI_STATUS_SIZE) :: istatus0                                         !< MPI variables
    real*8, allocatable, dimension(:) :: LocalChi2ValueVector                               !< used for subroutine ModelCalcCore for exchanging data
    real*8, allocatable, dimension(:, :) :: GlobalParamVectorSet                            !< used for subroutine ModelCalcCore for exchanging data
    logical, parameter :: dbflag = .false.


    contains


        !*************************************************************************************************************************************************
        !> subroutine: SlaveWork
        !>
        !> wake up MPI slave threads and do something or fall asleep
        !>
        !>
        !> input variables:     SlaveControlVar:        command which is send to the slave threads
        !>
        !>
        !> output parameters:   none
        !>
        !>
        !> \author Stefan Borowski, Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine SlaveWork(SlaveControlVar)

            implicit none
            integer :: SlaveControlVar                                                      !< command which is send to the slave threads
            integer :: i                                                                    !< loop variable

            ! Debug:
            ! print*,'MASTER: SlaveControlVar = ', SlaveControlVar


            Do i = 1, (TotalNumberOfThreads - 1)
                call MPI_Send(SlaveControlVar, 1, MPI_INTEGER, i, 0, MPI_COMM_WORLD, ierr)
            end Do
            return
        end subroutine SlaveWork


        !>************************************************************************************************************************************************
        !> subroutine: SlaveLoop
        !>
        !> manage subroutine for slave threads
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine SlaveLoop

            implicit none
            integer :: i
            integer :: SlaveControlVar                                                      !< command which is send to the slave threads
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: deallocstatus                                                        !< status of the deallocation process
            logical :: go                                                                   !< working variable
            logical :: ModelFunctionFlag                                                    !< do we need model function values?


            !< start a loop which is executed by all slave threads
            SlaveControlVar = 0
            Do
                call MPI_Irecv(SlaveControlVar, 1, MPI_INTEGER, 0, 0, MPI_COMM_WORLD, ireq0, ierr)

                ! Debug:
                ! print*,'SLAVE: myrank, SlaveControlVar = ', myrank, SlaveControlVar


                !< wait for jobs
                Do
                    Do i = 1, 10000
                    end Do
                    call MPI_Test(ireq0, go, istatus0, ierr)
                    if (go) exit
                end Do


                !< exit slave loop if (SlaveControlVar == 1)
                if (SlaveControlVar == 1) then
                    call ModelParamFree(deallocstatus)                                      !< free memory before exit
                    exit


                !< initialize variables (SlaveControlVar == 2)
                elseif (SlaveControlVar == 2) then

                    ! Debug:
                    ! print*,'********************************************************************************************>slave, myrank = ', myrank

                    call BroadCastMPIVariables


                !< determine model functions variables (SlaveControlVar == 3)
                elseif (SlaveControlVar == 3) then


                    !< determine chi2 value and model function values for different parameter vectors
                    ma = parameternumber
                    colx = MaxColX
                    NumFile = NumberExpFiles
                    MaxL = MaxExpLength
                    MaxCol = MaxColY
                    call ModelCalcCore(NumberFreeParameter, TotalNumberDataPoints, ModelFunctionFlag, ma, colx, NumFile, MaxL, MaxCol)
                endif
            end Do

            return
        end subroutine SlaveLoop


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_scalar_int
        !>
        !> Broadcasts a scalar integer variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     scalar:         scalar to be broadcast
        !>                      ParamName:      name of variable
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_scalar_int(scalar, ParamName)

            implicit none
            integer, intent(inout) :: scalar                                                !< scalar to be broadcast
            character(*) :: ParamName                                                       !< name of parameter


            !< broadcast scalar integer variable to all other threads
            call MPI_Bcast(scalar, 1, MPI_INTEGER, root_tn, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of scalar integer variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_scalar_int


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_scalar_dbl
        !>
        !> Broadcasts a scalar dbl variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     scalar:         scalar to be broadcast
        !>                      ParamName:      name of variable
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_scalar_dbl(scalar, ParamName)

            implicit none
            real*8, intent(inout) :: scalar                                                 !< scalar to be broadcast
            character(*) :: ParamName                                                       !< name of parameter


            !< broadcast scalar dbl variable to all other threads
            call MPI_Bcast(scalar, 1, MPI_DOUBLE_PRECISION, root_tn, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of scalar real variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif

            return
        end subroutine broadcast_scalar_dbl


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_scalar_log
        !>
        !> Broadcasts a scalar logical variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     scalar:         scalar to be broadcast
        !>                      ParamName:      name of variable
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_scalar_log(scalar, ParamName)

            implicit none
            logical, intent(inout) :: scalar                                                !< scalar to be broadcast
            character(*) :: ParamName                                                       !< name of parameter


            !< broadcast scalar logical variable to all other threads
            call MPI_Bcast(scalar, 1, MPI_LOGICAL, root_tn, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of scalar logical variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_scalar_log


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_scalar_char
        !>
        !> Broadcasts a scalar character variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     scalar:         scalar to be broadcast
        !>                      ParamName:      name of variable
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_scalar_char(scalar, ParamName)

            implicit none
            integer :: clength                                                              !< length of character
            character(*), intent(inout) :: scalar                                           !< scalar to be broadcast
            character(*) :: ParamName                                                       !< name of parameter


            !< broadcast scalar character variable to all other threads
            clength = len(scalar)
            call MPI_BCAST(scalar, clength, MPI_CHARACTER, root_tn, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print*,"clength = ", clength


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of scalar character variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_scalar_char


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_int_1d
        !>
        !> Broadcasts an 1d-array integer variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_int_1d(array, xl, xu, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: nelements                                                            !< size of array to be broadcast
            integer, dimension(xl:xu), intent(inout) :: array                               !< array to be broadcast
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu


            !< broadcast array integer variable to all other threads
            nelements = size(array)
            call MPI_Bcast(array, nelements, MPI_INTEGER, root_tn, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print*,"nelements = ", nelements


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of ineger 1d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_int_1d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_dbl_1d
        !>
        !> Broadcasts an 1d-array dbl variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_dbl_1d(array, xl, xu, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: nelements                                                            !< size of array to be broadcast
            real*8, dimension(xl:xu), intent(inout) :: array                                !< array to be broadcast
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu


            !< broadcast array dbl variable to all other threads
            nelements = size(array)
            call MPI_Bcast(array, nelements, MPI_DOUBLE_PRECISION, root_tn, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print*,"nelements = ", nelements


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of real 1d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_dbl_1d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_log_1d
        !>
        !> Broadcasts an 1d-array logical variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_log_1d(array, xl, xu, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: nelements                                                            !< size of array to be broadcast
            logical, dimension(xl:xu), intent(inout) :: array                               !< array to be broadcast
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu


            !< broadcast array logical variable to all other threads
            nelements = size(array)
            call MPI_Bcast(array, nelements, MPI_LOGICAL, root_tn, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print*,"nelements = ", nelements


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of logical 1d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_log_1d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_char_1d
        !>
        !> Broadcasts an 1d-array character variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      clength:        length of character
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_char_1d(array, xl, xu, clength, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: clength                                                              !< length of character
            integer :: nelements                                                            !< size of array to be broadcast
            character(len = clength), dimension(xl:xu), intent(inout) :: array              !< array to be broadcast
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"clength = ", clength


            !< broadcast array character variable to all other threads
            nelements = size(array) * clength
            call MPI_BCAST(array, nelements, MPI_CHARACTER, root_tn, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print*,"nelements = ", nelements


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of character 1d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_char_1d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_int_2d
        !>
        !> Broadcasts an 2d-array integer variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      yl:             lower index of dimesnion 2
        !>                      yu:             upper index of dimesnion 2
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_int_2d(array, xl, xu, yl, yu, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: yl                                                                   !< lower index of dimesnion 2
            integer :: yu                                                                   !< upper index of dimesnion 2
            integer*8 :: xs, ys                                                             !< size of each dimension
            integer :: nelements                                                            !< size of array to be broadcast
            integer :: i                                                                    !< loop variable
            integer, dimension(xl:xu, yl:yu), intent(inout) :: array                        !< array to be broadcast
            integer, dimension(yl:yu) :: TempArray                                          !< help array to broadcast large arrays
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"yl = ", yl
            ! print*,"yu = ", yu


            !< compute size of variable
            xs = abs(xu - xl + 1)
            ys = abs(yu - yl + 1)

            ! Debug:
            ! print*,"huge(0) = ", huge(0)
            ! print*,"xs * ys = ", xs * ys


            !< broadcast array integer variable to all other threads
            if ((xs * ys) <= huge(0)) then                                                  !< check, if size of variable is describable by 16 bit integer
                nelements = size(array)
                call MPI_Bcast(array, nelements, MPI_INTEGER, root_tn, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)

                ! Debug:
                ! print*,"nelements = ", nelements
            else
                Do i = xl, xu
                    TempArray = array(i, :)
                    nelements = size(TempArray)
                    call MPI_BCAST(TempArray, nelements, MPI_INTEGER, root_tn, MPI_COMM_WORLD, ierr)
                    call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                    if (myrank /= root_tn) then
                        array(i, :) = TempArray
                    endif
                end Do
            endif


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of integer 2d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_int_2d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_dbl_2d
        !>
        !> Broadcasts an 2d-array dbl variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      yl:             lower index of dimesnion 2
        !>                      yu:             upper index of dimesnion 2
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_dbl_2d(array, xl, xu, yl, yu, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: yl                                                                   !< lower index of dimesnion 2
            integer :: yu                                                                   !< upper index of dimesnion 2
            integer*8 :: xs, ys                                                             !< size of each dimension
            integer :: nelements                                                            !< size of array to be broadcast
            integer :: i                                                                    !< loop variable
            real*8, dimension(xl:xu, yl:yu), intent(inout) :: array                         !< array to be broadcast
            real*8, dimension(yl:yu) :: TempArray                                           !< help array to broadcast large arrays
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"yl = ", yl
            ! print*,"yu = ", yu


            !< compute size of variable
            xs = abs(xu - xl + 1)
            ys = abs(yu - yl + 1)

            ! Debug:
            ! print*,"huge(0) = ", huge(0)
            ! print*,"xs * ys = ", xs * ys


            !< broadcast array dbl variable to all other threads
            if ((xs * ys) <= huge(0)) then                                                  !< check, if size of variable is describable by 16 bit integer
                nelements = size(array)
                call MPI_Bcast(array, nelements, MPI_DOUBLE_PRECISION, root_tn, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)

                ! Debug:
                ! print*,"nelements = ", nelements
            else
                Do i = xl, xu
                    TempArray = array(i, :)
                    nelements = size(TempArray)
                    call MPI_BCAST(TempArray, nelements, MPI_DOUBLE_PRECISION, root_tn, MPI_COMM_WORLD, ierr)
                    call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                    if (myrank /= root_tn) then
                        array(i, :) = TempArray
                    endif
                end Do
            endif


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of real 2d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_dbl_2d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_log_2d
        !>
        !> Broadcasts an 2d-array logical variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      yl:             lower index of dimesnion 2
        !>                      yu:             upper index of dimesnion 2
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_log_2d(array, xl, xu, yl, yu, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: yl                                                                   !< lower index of dimesnion 2
            integer :: yu                                                                   !< upper index of dimesnion 2
            integer*8 :: xs, ys                                                             !< size of each dimension
            integer :: nelements                                                            !< size of array to be broadcast
            integer :: i                                                                    !< loop variable
            logical, dimension(xl:xu, yl:yu), intent(inout) :: array                        !< array to be broadcast
            logical, dimension(yl:yu) :: TempArray                                          !< help array to broadcast large arrays
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"yl = ", yl
            ! print*,"yu = ", yu


            !< compute size of variable
            xs = abs(xu - xl + 1)
            ys = abs(yu - yl + 1)

            ! Debug:
            ! print*,"huge(0) = ", huge(0)
            ! print*,"xs * ys = ", xs * ys


            !< broadcast array logical variable to all other threads
            if ((xs * ys) <= huge(0)) then                                                  !< check, if size of variable is describable by 16 bit integer
                nelements = size(array)
                call MPI_Bcast(array, nelements, MPI_LOGICAL, root_tn, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)

                ! Debug:
                ! print*,"nelements = ", nelements
            else
                Do i = xl, xu
                    TempArray = array(i, :)
                    nelements = size(TempArray)
                    call MPI_BCAST(TempArray, nelements, MPI_LOGICAL, root_tn, MPI_COMM_WORLD, ierr)
                    call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                    if (myrank /= root_tn) then
                        array(i, :) = TempArray
                    endif
                end Do
            endif


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of logical 2d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_log_2d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_char_2d
        !>
        !> Broadcasts an 2d-array character variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      yl:             lower index of dimesnion 2
        !>                      yu:             upper index of dimesnion 2
        !>                      clength:        length of character
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_char_2d(array, xl, xu, yl, yu, clength, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: yl                                                                   !< lower index of dimesnion 2
            integer :: yu                                                                   !< upper index of dimesnion 2
            integer*8 :: xs, ys                                                             !< size of each dimension
            integer :: clength                                                              !< length of character
            integer :: nelements                                                            !< size of array to be broadcast
            integer :: i                                                                    !< loop variable
            character(len = clength), dimension(xl:xu, yl:yu), intent(inout) :: array       !< array to be broadcast
            character(len = clength), dimension(yl:yu) :: TempArray                         !< help array to broadcast large arrays
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"yl = ", yl
            ! print*,"yu = ", yu
            ! print*,"clength = ", clength


            !< compute size of variable
            xs = abs(xu - xl + 1)
            ys = abs(yu - yl + 1)

            ! Debug:
            ! print*,"huge(0) = ", huge(0)
            ! print*,"xs * ys * clength = ", xs * ys * clength


            !< broadcast array character variable to all other threads
            if ((xs * ys * clength) <= huge(0)) then                                        !< check, if size of variable is describable by 16 bit integer
                nelements = size(array) * clength
                call MPI_BCAST(array, nelements, MPI_CHARACTER, root_tn, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)

                ! Debug:
                ! print*,"nelements = ", nelements
            else
                Do i = xl, xu
                    TempArray = array(i, :)
                    nelements = size(TempArray) * clength
                    call MPI_BCAST(TempArray, nelements, MPI_CHARACTER, root_tn, MPI_COMM_WORLD, ierr)
                    call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                    if (myrank /= root_tn) then
                        array(i, :) = TempArray
                    endif
                end Do
            endif


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of character 2d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_char_2d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_int_3d
        !>
        !> Broadcasts an 3d-array integer variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      yl:             lower index of dimesnion 2
        !>                      yu:             upper index of dimesnion 2
        !>                      zl:             lower index of dimesnion 3
        !>                      zu:             upper index of dimesnion 3
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_int_3d(array, xl, xu, yl, yu, zl, zu, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: yl                                                                   !< lower index of dimesnion 2
            integer :: yu                                                                   !< upper index of dimesnion 2
            integer :: zl                                                                   !< lower index of dimesnion 3
            integer :: zu                                                                   !< upper index of dimesnion 3
            integer :: nelements                                                            !< size of array to be broadcast
            integer*8 :: xs, ys, zs                                                         !< size of each dimension
            integer :: i, j                                                                 !< loop variables
            integer, dimension(xl:xu, yl:yu, zl:zu), intent(inout) :: array                 !< array to be broadcast
            integer, dimension(zl:zu) :: TempArray                                          !< help array to broadcast large arrays
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"yl = ", yl
            ! print*,"yu = ", yu
            ! print*,"zl = ", zl
            ! print*,"zu = ", zu


            !< compute size of variable
            xs = abs(xu - xl + 1)
            ys = abs(yu - yl + 1)
            zs = abs(zu - zl + 1)

            ! Debug:
            ! print*,"huge(0) = ", huge(0)
            ! print*,"xs * ys * zs = ", xs * ys * zs


            !< broadcast array integer variable to all other threads
            if ((xs * ys * zs) <= huge(0)) then                                             !< check, if size of variable is describable by 16 bit integer
                nelements = size(array)
                call MPI_Bcast(array, nelements, MPI_INTEGER, root_tn, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)

                ! Debug:
                ! print*,"nelements = ", nelements
            else
                Do i = xl, xu
                    Do j = yl, yu
                        TempArray = array(i, j, :)
                        nelements = size(TempArray)
                        call MPI_BCAST(TempArray, nelements, MPI_INTEGER, root_tn, MPI_COMM_WORLD, ierr)
                        call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                        if (myrank /= root_tn) then
                            array(i, j, :) = TempArray
                        endif
                    end Do
                end Do
            endif


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of integer 3d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_int_3d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_dbl_3d
        !>
        !> Broadcasts an 3d-array dbl variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      yl:             lower index of dimesnion 2
        !>                      yu:             upper index of dimesnion 2
        !>                      zl:             lower index of dimesnion 3
        !>                      zu:             upper index of dimesnion 3
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_dbl_3d(array, xl, xu, yl, yu, zl, zu, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: yl                                                                   !< lower index of dimesnion 2
            integer :: yu                                                                   !< upper index of dimesnion 2
            integer :: zl                                                                   !< lower index of dimesnion 3
            integer :: zu                                                                   !< upper index of dimesnion 3
            integer*8 :: xs, ys, zs                                                         !< size of each dimension
            integer :: nelements                                                            !< size of array to be broadcast
            integer :: i, j                                                                 !< loop variables
            real*8, dimension(xl:xu, yl:yu, zl:zu), intent(inout) :: array                  !< array to be broadcast
            real*8, dimension(zl:zu) :: TempArray                                           !< help array to broadcast large arrays
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"yl = ", yl
            ! print*,"yu = ", yu
            ! print*,"zl = ", zl
            ! print*,"zu = ", zu


            !< compute size of variable
            xs = abs(xu - xl + 1)
            ys = abs(yu - yl + 1)
            zs = abs(zu - zl + 1)

            ! Debug:
            ! print*,"huge(0) = ", huge(0)
            ! print*,"xs * ys * zs = ", xs * ys * zs


            !< broadcast array dbl variable to all other threads
            if ((xs * ys * zs) <= huge(0)) then                                             !< check, if size of variable is describable by 16 bit integer
                nelements = size(array)
                call MPI_Bcast(array, nelements, MPI_DOUBLE_PRECISION, root_tn, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)

                ! Debug:
                ! print*,"nelements = ", nelements
            else
                Do i = xl, xu
                    Do j = yl, yu
                        TempArray = array(i, j, :)
                        nelements = size(TempArray)
                        call MPI_BCAST(TempArray, nelements, MPI_DOUBLE_PRECISION, root_tn, MPI_COMM_WORLD, ierr)
                        call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                        if (myrank /= root_tn) then
                            array(i, j, :) = TempArray
                        endif
                    end Do
                end Do
            endif


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of real 3d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_dbl_3d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_log_3d
        !>
        !> Broadcasts an 3d-array logical variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      yl:             lower index of dimesnion 2
        !>                      yu:             upper index of dimesnion 2
        !>                      zl:             lower index of dimesnion 3
        !>                      zu:             upper index of dimesnion 3
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_log_3d(array, xl, xu, yl, yu, zl, zu, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: yl                                                                   !< lower index of dimesnion 2
            integer :: yu                                                                   !< upper index of dimesnion 2
            integer :: zl                                                                   !< lower index of dimesnion 3
            integer :: zu                                                                   !< upper index of dimesnion 3
            integer*8 :: xs, ys, zs                                                         !< size of each dimension
            integer :: nelements                                                            !< size of array to be broadcast
            integer :: i, j                                                                 !< loop variables
            logical, dimension(xl:xu, yl:yu, zl:zu), intent(inout) :: array                 !< array to be broadcast
            logical, dimension(zl:zu) :: TempArray                                          !< help array to broadcast large arrays
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"yl = ", yl
            ! print*,"yu = ", yu
            ! print*,"zl = ", zl
            ! print*,"zu = ", zu


            !< compute size of variable
            xs = abs(xu - xl + 1)
            ys = abs(yu - yl + 1)
            zs = abs(zu - zl + 1)

            ! Debug:
            ! print*,"huge(0) = ", huge(0)
            ! print*,"xs * ys * zs = ", xs * ys * zs


            !< broadcast array logical variable to all other threads
            if ((xs * ys * zs) <= huge(0)) then                                             !< check, if size of variable is describable by 16 bit integer
                nelements = size(array)
                call MPI_Bcast(array, nelements, MPI_LOGICAL, root_tn, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)

                ! Debug:
                ! print*,"nelements = ", nelements
            else
                Do i = xl, xu
                    Do j = yl, yu
                        TempArray = array(i, j, :)
                        nelements = size(TempArray)
                        call MPI_BCAST(TempArray, nelements, MPI_LOGICAL, root_tn, MPI_COMM_WORLD, ierr)
                        call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                        if (myrank /= root_tn) then
                            array(i, j, :) = TempArray
                        endif
                    end Do
                end Do
            endif


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of logical 3d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_log_3d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_char_3d
        !>
        !> Broadcasts an 3d-array character variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      yl:             lower index of dimesnion 2
        !>                      yu:             upper index of dimesnion 2
        !>                      zl:             lower index of dimesnion 3
        !>                      zu:             upper index of dimesnion 3
        !>                      clength:        length of character
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_char_3d(array, xl, xu, yl, yu, zl, zu, clength, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: yl                                                                   !< lower index of dimesnion 2
            integer :: yu                                                                   !< upper index of dimesnion 2
            integer :: zl                                                                   !< lower index of dimesnion 3
            integer :: zu                                                                   !< upper index of dimesnion 3
            integer :: clength                                                              !< length of character
            integer*8 :: xs, ys, zs                                                         !< size of each dimension
            integer :: nelements                                                            !< size of array to be broadcast
            integer :: i, j                                                                 !< loop variables
            character(len = clength), dimension(xl:xu, yl:yu, zl:zu), intent(inout) :: array    !< array to be broadcast
            character(len = clength), dimension(zl:zu) :: TempArray                         !< temp array for broadcasting large arrays
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"yl = ", yl
            ! print*,"yu = ", yu
            ! print*,"zl = ", zl
            ! print*,"zu = ", zu
            ! print*,"clength = ", clength


            !< compute size of variable
            xs = abs(xu - xl + 1)
            ys = abs(yu - yl + 1)
            zs = abs(zu - zl + 1)

            ! Debug:
            ! print*,"huge(0) = ", huge(0)
            ! print*,"xs * ys * zs * clength = ", xs * ys * zs * clength


            !< broadcast array character variable to all other threads
            if ((xs * ys * zs * clength) <= huge(0)) then                                   !< check, if size of variable is describable by 16 bit integer
                nelements = size(array) * clength
                call MPI_BCAST(array, nelements, MPI_CHARACTER, root_tn, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            else
                Do i = xl, xu
                    Do j = yl, yu
                        TempArray = array(i, j, :)
                        nelements = size(TempArray) * clength
                        call MPI_BCAST(TempArray, nelements, MPI_CHARACTER, root_tn, MPI_COMM_WORLD, ierr)
                        call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                        if (myrank /= root_tn) then
                            array(i,j, :) = TempArray
                        endif
                    end Do
                end Do
            endif

            ! Debug:
            ! print*,"nelements = ", nelements


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of character 3d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_char_3d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_int_4d
        !>
        !> Broadcasts an 4d-array integer variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      yl:             lower index of dimesnion 2
        !>                      yu:             upper index of dimesnion 2
        !>                      zl:             lower index of dimesnion 3
        !>                      zu:             upper index of dimesnion 3
        !>                      tl:             lower index of dimesnion 4
        !>                      tu:             upper index of dimesnion 4
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_int_4d(array, xl, xu, yl, yu, zl, zu, tl, tu, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: yl                                                                   !< lower index of dimesnion 2
            integer :: yu                                                                   !< upper index of dimesnion 2
            integer :: zl                                                                   !< lower index of dimesnion 3
            integer :: zu                                                                   !< upper index of dimesnion 3
            integer :: tl                                                                   !< lower index of dimesnion 4
            integer :: tu                                                                   !< upper index of dimesnion 4
            integer*8 :: xs, ys, zs, ts                                                     !< size of each dimension
            integer :: nelements                                                            !< size of array to be broadcast
            integer :: i, j, k                                                              !< loop variables
            integer, dimension(xl:xu, yl:yu, zl:zu, tl:tu), intent(inout) :: array          !< array to be broadcast
            integer, dimension(tl:tu) :: TempArray                                          !< temp array for broadcasting large arrays
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"yl = ", yl
            ! print*,"yu = ", yu
            ! print*,"zl = ", zl
            ! print*,"zu = ", zu
            ! print*,"tl = ", tl
            ! print*,"tu = ", tu


            !< compute size of variable
            xs = abs(xu - xl + 1)
            ys = abs(yu - yl + 1)
            zs = abs(zu - zl + 1)
            ts = abs(tu - tl + 1)

            ! Debug:
            ! print*,"huge(0) = ", huge(0)
            ! print*,"xs * ys * zs * ts = ", xs * ys * zs * ts


            !< broadcast array character variable to all other threads
            if ((xs * ys * zs * ts) <= huge(0)) then                                        !< check, if size of variable is describable by 16 bit integer
                nelements = size(array)
                call MPI_BCAST(array, nelements, MPI_INTEGER, root_tn, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)

                ! Debug:
                ! print*,"nelements = ", nelements
            else
                Do i = xl, xu
                    Do j = yl, yu
                        Do k = zl, zu
                            TempArray = array(i, j, k, :)
                            nelements = size(TempArray)
                            call MPI_BCAST(TempArray, nelements, MPI_INTEGER, root_tn, MPI_COMM_WORLD, ierr)
                            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                            if (myrank /= root_tn) then
                                array(i, j, k, :) = TempArray
                            endif
                        end Do
                    end Do
                end Do
            endif


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of integer 4d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_int_4d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_dbl_4d
        !>
        !> Broadcasts an 4d-array dbl variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      yl:             lower index of dimesnion 2
        !>                      yu:             upper index of dimesnion 2
        !>                      zl:             lower index of dimesnion 3
        !>                      zu:             upper index of dimesnion 3
        !>                      tl:             lower index of dimesnion 4
        !>                      tu:             upper index of dimesnion 4
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_dbl_4d(array, xl, xu, yl, yu, zl, zu, tl, tu, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: yl                                                                   !< lower index of dimesnion 2
            integer :: yu                                                                   !< upper index of dimesnion 2
            integer :: zl                                                                   !< lower index of dimesnion 3
            integer :: zu                                                                   !< upper index of dimesnion 3
            integer :: tl                                                                   !< lower index of dimesnion 4
            integer :: tu                                                                   !< upper index of dimesnion 4
            integer*8 :: xs, ys, zs, ts                                                     !< size of each dimension
            integer :: nelements                                                            !< size of array to be broadcast
            integer :: i, j, k                                                              !< loop variables
            real*8, dimension(xl:xu, yl:yu, zl:zu, tl:tu), intent(inout) :: array           !< array to be broadcast
            real*8, dimension(tl:tu) :: TempArray                                           !< temp array for broadcasting large arrays
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"yl = ", yl
            ! print*,"yu = ", yu
            ! print*,"zl = ", zl
            ! print*,"zu = ", zu
            ! print*,"tl = ", tl
            ! print*,"tu = ", tu


            !< compute size of variable
            xs = abs(xu - xl + 1)
            ys = abs(yu - yl + 1)
            zs = abs(zu - zl + 1)
            ts = abs(tu - tl + 1)

            ! Debug:
            ! print*,"huge(0) = ", huge(0)
            ! print*,"xs * ys * zs * ts = ", xs * ys * zs * ts


            !< broadcast array dbl variable to all other threads
            if ((xs * ys * zs * ts) <= huge(0)) then                                        !< check, if size of variable is describable by 16 bit integer
                nelements = size(array)
                call MPI_Bcast(array, nelements, MPI_DOUBLE_PRECISION, root_tn, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)

                ! Debug:
                ! print*,"nelements = ", nelements
            else
                Do i = xl, xu
                    Do j = yl, yu
                        Do k = zl, zu
                            TempArray = array(i, j, k, :)
                            nelements = size(TempArray)
                            call MPI_BCAST(TempArray, nelements, MPI_DOUBLE_PRECISION, root_tn, MPI_COMM_WORLD, ierr)
                            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                            if (myrank /= root_tn) then
                                array(i, j, k, :) = TempArray
                            endif
                        end Do
                    end Do
                end Do
            endif


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of real 4d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_dbl_4d


        !>************************************************************************************************************************************************
        !> subroutine: broadcast_array_dbl_5d
        !>
        !> Broadcasts an 5d-array dbl variable from one processor (root_tn) to all other processors.
        !>
        !>
        !> input variables:     array:          array to be broadcast
        !>                      xl:             lower index of dimesnion 1
        !>                      xu:             upper index of dimesnion 1
        !>                      yl:             lower index of dimesnion 2
        !>                      yu:             upper index of dimesnion 2
        !>                      zl:             lower index of dimesnion 3
        !>                      zu:             upper index of dimesnion 3
        !>                      tl:             lower index of dimesnion 4
        !>                      tu:             upper index of dimesnion 4
        !>                      ol:             lower index of dimesnion 4
        !>                      ou:             upper index of dimesnion 4
        !>                      ParamName:      name of variable
        !>
        !> output variables:    array:          array to be broadcast
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 2021-01-05
        !>
        subroutine broadcast_array_dbl_5d(array, xl, xu, yl, yu, zl, zu, tl, tu, ol, ou, ParamName)

            implicit none
            integer :: xl                                                                   !< lower index of dimesnion 1
            integer :: xu                                                                   !< upper index of dimesnion 1
            integer :: yl                                                                   !< lower index of dimesnion 2
            integer :: yu                                                                   !< upper index of dimesnion 2
            integer :: zl                                                                   !< lower index of dimesnion 3
            integer :: zu                                                                   !< upper index of dimesnion 3
            integer :: tl                                                                   !< lower index of dimesnion 4
            integer :: tu                                                                   !< upper index of dimesnion 4
            integer :: ol                                                                   !< lower index of dimesnion 5
            integer :: ou                                                                   !< upper index of dimesnion 5
            integer*8 :: xs, ys, zs, ts, os                                                 !< size of each dimension
            integer :: nelements                                                            !< size of array to be broadcast
            integer :: i, j, k, l                                                           !< loop variables
            real*8, dimension(xl:xu, yl:yu, zl:zu, tl:tu, ol:ou), intent(inout) :: array    !< array to be broadcast
            real*8, dimension(ol:ou) :: TempArray                                           !< help array to broadcast large arrays
            character(*) :: ParamName                                                       !< name of parameter

            ! Debug:
            ! print*,"xl = ", xl
            ! print*,"xu = ", xu
            ! print*,"yl = ", yl
            ! print*,"yu = ", yu
            ! print*,"zl = ", zl
            ! print*,"zu = ", zu
            ! print*,"tl = ", tl
            ! print*,"tu = ", tu
            ! print*,"ol = ", ol
            ! print*,"ou = ", ou


            !< compute size of variable
            xs = abs(xu - xl + 1)
            ys = abs(yu - yl + 1)
            zs = abs(zu - zl + 1)
            ts = abs(tu - tl + 1)
            os = abs(ou - ol + 1)

            ! Debug:
            ! print*,"huge(0) = ", huge(0)
            ! print*,"xs * ys * zs * ts * os = ", xs * ys * zs * ts * os


            !< broadcast array dbl variable to all other threads
            if ((xs * ys * zs * ts * os) <= huge(0)) then                                   !< check, if size of variable is describable by 16 bit integer
                nelements = size(array)
                call MPI_Bcast(array, nelements, MPI_DOUBLE_PRECISION, root_tn, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)

                ! Debug:
                ! print*,"nelements = ", nelements
            else
                Do i = xl, xu
                    Do j = yl, yu
                        Do k = zl, zu
                            Do l = tl, tu
                                TempArray = array(i, j, k, l, :)
                                nelements = size(TempArray)
                                call MPI_Bcast(TempArray, nelements, MPI_DOUBLE_PRECISION, root_tn, MPI_COMM_WORLD, ierr)
                                call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                                if (myrank /= root_tn) then
                                    array(i, j, k, l, :) = TempArray
                                endif
                            end Do
                        end Do
                    end Do
                end Do
            endif


            !< for debugging
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Broadcast of real 5d-array variable ", A, " done!")', char(34) // trim(adjustl(ParamName)) // char(34)
                flush(6)
            endif


            return
        end subroutine broadcast_array_dbl_5d


        !>************************************************************************************************************************************************
        !> subroutine: BroadCastMPIVariables
        !>
        !> broadcast MPI variables between master and slave threads
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine BroadCastMPIVariables


            implicit none
            integer :: k, l                                                                 !< working variables: used for allocation
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            integer :: MaxNumberOfMolecules                                                 !< working variable: max. number of molecules (incl. isotop.)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< define some global variables
            DontStoreModelFuncValuesFlag = .true.                                           !< model function values are stored for each function call
            ParallezitionMethod = 2                                                         !< set parallelization method to MPI (=2)
            myrankOfMPI = myrank                                                            !< copy thread number to global variable
            call GetEnv('MAGIXTempDirectory', TempDirectory)                                !< get temp directory for current thread


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< broadcast scalar variables for Module 'Variables'
            call broadcast_scalar_int(QualityLimit, "QualityLimit")
            ! call MPI_Bcast(QualityLimit, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberExpFiles, "NumberExpFiles")
            ! call MPI_Bcast(NumberExpFiles, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(modelnumber, "modelnumber")
            ! call MPI_Bcast(modelnumber, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberLinesChi2, "NumberLinesChi2")
            ! call MPI_Bcast(NumberLinesChi2, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(MaxColX, "MaxColX")
            ! call MPI_Bcast(MaxColX, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(MaxColY, "MaxColY")
            ! call MPI_Bcast(MaxColY, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(parameternumber, "parameternumber")
            ! call MPI_Bcast(parameternumber, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberInputFiles, "NumberInputFiles")
            ! call MPI_Bcast(NumberInputFiles, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberOutputFiles, "NumberOutputFiles")
            ! call MPI_Bcast(NumberOutputFiles, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(JobID, "JobID")
            ! call MPI_Bcast(JobID, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(MaxInputLines, "MaxInputLines")
            ! call MPI_Bcast(MaxInputLines, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(MaxParameter, "MaxParameter")
            ! call MPI_Bcast(MaxParameter, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(ExternalThreadNumber, "ExternalThreadNumber")
            ! call MPI_Bcast(ExternalThreadNumber, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(CurrentNumberLinesCalcReduction, "CurrentNumberLinesCalcReduction")
            ! call MPI_Bcast(CurrentNumberLinesCalcReduction, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(CurrentExpFile, "CurrentExpFile")
            ! call MPI_Bcast(CurrentExpFile, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(CurrentYColumn, "CurrentYColumn")
            ! call MPI_Bcast(CurrentYColumn, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberFreeParameter, "NumberFreeParameter")
            ! call MPI_Bcast(NumberFreeParameter, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(MaxExpLength, "MaxExpLength")
            ! call MPI_Bcast(MaxExpLength, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(MaxNumberRanges, "MaxNumberRanges")
            ! call MPI_Bcast(MaxNumberRanges, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(ParallezitionMethod, "ParallezitionMethod")
            ! call MPI_Bcast(ParallezitionMethod, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(TotalNumberOfThreads, "TotalNumberOfThreads")
            ! call MPI_Bcast(TotalNumberOfThreads, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(ParallelizationFlag, "ParallelizationFlag")
            ! call MPI_Bcast(ParallelizationFlag, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_dbl(ochisq, "ochisq")
            ! call MPI_Bcast(ochisq, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(NaNReplaceString, "NaNReplaceString")
            ! call MPI_BCAST(NaNReplaceString, 10, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(currentpath, "currentpath")
            ! call MPI_BCAST(currentpath, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(StandardWorkingDirectory, "StandardWorkingDirectory")
            ! call MPI_BCAST(StandardWorkingDirectory, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(TempDirectory, "TempDirectory")
            ! call MPI_BCAST(TempDirectory, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(RenormalizedChi2, "RenormalizedChi2")
            ! call MPI_BCAST(RenormalizedChi2, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(initflag, "initflag")
            ! call MPI_BCAST(initflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(printflag, "printflag")
            ! call MPI_BCAST(printflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(Gradientflag, "Gradientflag")
            ! call MPI_BCAST(Gradientflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(PlotIterationFlag, "PlotIterationFlag")
            ! call MPI_BCAST(PlotIterationFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(WriteChi2Flag, "WriteChi2Flag")
            ! call MPI_BCAST(WriteChi2Flag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(UseCalculationReduction, "UseCalculationReduction")
            ! call MPI_BCAST(UseCalculationReduction, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(LastAlgorithmFlag, "LastAlgorithmFlag")
            ! call MPI_BCAST(LastAlgorithmFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(AllInOneOutputFile, "AllInOneOutputFile")
            ! call MPI_BCAST(AllInOneOutputFile, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(NumOutputFilesEqualNumExpFiles, "NumOutputFilesEqualNumExpFiles")
            ! call MPI_BCAST(NumOutputFilesEqualNumExpFiles, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(DontStoreModelFuncValuesFlag, "DontStoreModelFuncValuesFlag")
            ! call MPI_BCAST(DontStoreModelFuncValuesFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< variables for Module 'Variables'


            !< allocate memory for variables for Module 'Variables'
            if (myrank /= root_tn) then                                                           !< only slave threads (de)allocate memory
                if (allocated(lengthexpdata)) then
                    deallocate(lengthexpdata, NumberXColumns, NumberYColumns, NumberHeaderLines, OutputFileFormat, NumberRanges, ConverterInfit, &
                               ValueEmptyOutputFile, LSRAdjustement, chisqValues, BestSitesParamSet, paramset, AtOnceFunction, AtOnceGradient, &
                               FirstPointExpData, LastPointExpData, expdatax, expdatay, expdatae, CalculatedParameterSets, MinRange, MaxRange, &
                               BestSitesModelValues, BestSitesChi2Values, ModelFunction, FitParameterName, FitParameterValue, CharacterForComments, &
                               CharacterSeperatingColumns, ResamplingMethod, InterpolationMethod, OnlyYColumn, LSRAdjustementFitFlag, NormalizationFlag, &
                               ExpData_reversed_flag, stat = dealloc_status)
                    if (dealloc_status /= 0) then                                           !< is all ok?
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                        write(logchannel,'(2x,"Can not deallocate variables for Module Variables for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine BroadCastMPIVariables:")'
                        print '(2x,"Can not deallocate variables for Module Variables for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("dealloc_status = ",I4)',dealloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                endif
                allocate(lengthexpdata(NumberExpFiles), &
                         NumberXColumns(NumberExpFiles), &
                         NumberYColumns(NumberExpFiles), &
                         NumberHeaderLines(NumberOutputFiles), &
                         OutputFileFormat(NumberOutputFiles), &
                         NumberRanges(NumberExpFiles), &
                         ConverterInfit(NumberFreeParameter), &
                         ValueEmptyOutputFile(NumberOutputFiles), &
                         LSRAdjustement(NumberOutputFiles), &
                         chisqValues(0:0), &
                         BestSitesParamSet(QualityLimit, NumberFreeParameter + 1), &
                         paramset(4, parameternumber), &
                         AtOnceFunction(0:0, NumberExpFiles, MaxColY, MaxExpLength), &
                         FirstPointExpData(NumberExpFiles, MaxColX), &
                         LastPointExpData(NumberExpFiles, MaxColX), &
                         expdatax(NumberExpFiles, MaxExpLength, MaxColX), &
                         expdatay(NumberExpFiles, MaxExpLength, MaxColY), &
                         expdatae(NumberExpFiles, MaxExpLength, MaxColY), &
                         CalculatedParameterSets(0:ParallelizationFlag, CurrentNumberLinesCalcReduction, NumberFreeParameter + 1), &
                         MinRange(NumberExpFiles, MaxNumberRanges, MaxColX), &
                         MaxRange(NumberExpFiles, MaxNumberRanges, MaxColX), &
                         BestSitesModelValues(QualityLimit, NumberExpFiles, MaxExpLength, MaxColY), &
                         BestSitesChi2Values(QualityLimit, NumberExpFiles, MaxExpLength, MaxColY), &
                         ModelFunction(1, NumberExpFiles, MaxColY, MaxExpLength), &
                         FitParameterName(parameternumber), &
                         FitParameterValue(parameternumber), &
                         CharacterForComments(NumberOutputFiles), &
                         CharacterSeperatingColumns(NumberOutputFiles), &
                         ResamplingMethod(NumberOutputFiles), &
                         InterpolationMethod(NumberOutputFiles), &
                         OnlyYColumn(NumberOutputFiles), &
                         LSRAdjustementFitFlag(NumberOutputFiles), &
                         NormalizationFlag(NumberOutputFiles), &
                         ExpData_reversed_flag(NumberExpFiles), &
                         stat = alloc_status)
                if (alloc_status /= 0) then                                                 !< is all ok?
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                    write(logchannel,'(2x,"Can not allocate variables for Module Variables for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("alloc_status = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine BroadCastMPIVariables:")'
                    print '(2x,"Can not allocate variables for Module Variables for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("alloc_status = ",I4)',alloc_status
                    print '(" ")'
                    ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                    call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                    stop
                endif
                lengthexpdata = 0
                NumberXColumns = 0
                NumberYColumns = 0
                NumberHeaderLines = 0
                OutputFileFormat = 0
                NumberRanges = 0
                ConverterInfit = 0
                ValueEmptyOutputFile = 0.d0
                LSRAdjustement = 0.d0
                chisqValues = 0.d0
                BestSitesParamSet = 0.d0
                paramset = 0.d0
                AtOnceFunction = 0.d0
                FirstPointExpData = 0.d0
                LastPointExpData = 0.d0
                expdatax = 0.d0
                expdatay = 0.d0
                expdatae = 0.d0
                CalculatedParameterSets = 0.d0
                MinRange = 0.d0
                MaxRange = 0.d0
                BestSitesModelValues = 0.d0
                BestSitesChi2Values = 0.d0
                ModelFunction = 0.d0
                FitParameterName = ""
                FitParameterValue = ""
                CharacterForComments = ""
                CharacterSeperatingColumns = ""
                ResamplingMethod = ""
                InterpolationMethod = ""
                OnlyYColumn = .false.
                LSRAdjustementFitFlag = .false.
                NormalizationFlag = .false.
                ExpData_reversed_flag = .false.
            endif
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Allocation of variables of Module >>Variables<< done!")'
                flush(6)
            endif


            !< broadcast scalar variables for Module 'Variables'
            !< exclude variables: AtOnceFunction, BestSitesModelValues, BestSitesChi2Values, ModelFunction

            call broadcast_array_int_1d(lengthexpdata, 1, NumberExpFiles, "lengthexpdata")
            ! nelements = size(lengthexpdata)
            ! call MPI_Bcast(lengthexpdata, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_1d(NumberXColumns, 1, NumberExpFiles, "NumberXColumns")
            ! nelements = size(NumberXColumns)
            ! call MPI_Bcast(NumberXColumns, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_1d(NumberYColumns, 1, NumberExpFiles, "NumberYColumns")
            ! nelements = size(NumberYColumns)
            ! call MPI_Bcast(NumberYColumns, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_1d(NumberHeaderLines, 1, NumberOutputFiles, "NumberHeaderLines")
            ! nelements = size(NumberHeaderLines)
            ! call MPI_Bcast(NumberHeaderLines, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_1d(OutputFileFormat, 1, NumberOutputFiles, "OutputFileFormat")
            ! nelements = size(OutputFileFormat)
            ! call MPI_Bcast(OutputFileFormat, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_1d(NumberRanges, 1, NumberExpFiles, "NumberRanges")
            ! nelements = size(NumberRanges)
            ! call MPI_Bcast(NumberRanges, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_1d(ConverterInfit, 1, NumberFreeParameter, "ConverterInfit")
            ! nelements = size(ConverterInfit)
            ! call MPI_Bcast(ConverterInfit, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(ValueEmptyOutputFile, 1, NumberOutputFiles, "ValueEmptyOutputFile")
            ! nelements = size(ValueEmptyOutputFile)
            ! call MPI_Bcast(ValueEmptyOutputFile, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(LSRAdjustement, 1, NumberOutputFiles, "LSRAdjustement")
            ! nelements = size(LSRAdjustement)
            ! call MPI_Bcast(LSRAdjustement, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(chisqValues, 0, 0, "chisqValues")
            ! nelements = size(chisqValues)
            ! call MPI_Bcast(chisqValues, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(BestSitesParamSet, 1, QualityLimit, 1, NumberFreeParameter + 1, "BestSitesParamSet")
            ! nelements = size(BestSitesParamSet)
            ! call MPI_Bcast(BestSitesParamSet, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(paramset, 1, 4, 1, parameternumber, "paramset")
            ! nelements = size(paramset)
            ! call MPI_Bcast(paramset, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(FirstPointExpData, 1, NumberExpFiles, 1, MaxColX, "FirstPointExpData")
            ! nelements = size(FirstPointExpData)
            ! call MPI_Bcast(FirstPointExpData, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(LastPointExpData, 1, NumberExpFiles, 1, MaxColX, "LastPointExpData")
            ! nelements = size(LastPointExpData)
            ! call MPI_Bcast(LastPointExpData, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_3d(expdatax, 1, NumberExpFiles, 1, MaxExpLength, 1, MaxColX, "expdatax")
            ! nelements = size(expdatax)
            ! call MPI_Bcast(expdatax, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_3d(expdatay, 1, NumberExpFiles, 1, MaxExpLength, 1, MaxColY, "expdatay")
            ! nelements = size(expdatay)
            ! call MPI_Bcast(expdatay, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_3d(expdatae, 1, NumberExpFiles, 1, MaxExpLength, 1, MaxColY, "expdatae")
            ! nelements = size(expdatae)
            ! call MPI_Bcast(expdatae, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_3d(MinRange, 1, NumberExpFiles, 1, MaxNumberRanges, 1, MaxColX, "MinRange")
            ! nelements = size(MinRange)
            ! call MPI_Bcast(MinRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_3d(MaxRange, 1, NumberExpFiles, 1, MaxNumberRanges, 1, MaxColX, "MaxRange")
            ! nelements = size(MaxRange)
            ! call MPI_Bcast(MaxRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(FitParameterName, 1, parameternumber, 512, "FitParameterName")
            ! nelements = size(FitParameterName) * 512
            ! call MPI_Bcast(FitParameterName, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(FitParameterValue, 1, parameternumber, 512, "FitParameterValue")
            ! nelements = size(FitParameterValue) * 512
            ! call MPI_Bcast(FitParameterValue, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(CharacterForComments, 1, NumberOutputFiles, 512, "CharacterForComments")
            ! nelements = size(CharacterForComments) * 512
            ! call MPI_Bcast(CharacterForComments, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(CharacterSeperatingColumns, 1, NumberOutputFiles, 512, "CharacterSeperatingColumns")
            ! nelements = size(CharacterSeperatingColumns) * 512
            ! call MPI_Bcast(CharacterSeperatingColumns, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(ResamplingMethod, 1, NumberOutputFiles, 512, "ResamplingMethod")
            ! nelements = size(ResamplingMethod) * 512
            ! call MPI_Bcast(ResamplingMethod, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(InterpolationMethod, 1, NumberOutputFiles, 512, "InterpolationMethod")
            ! nelements = size(InterpolationMethod) * 512
            ! call MPI_Bcast(InterpolationMethod, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(OnlyYColumn, 1, NumberOutputFiles, "OnlyYColumn")
            ! nelements = size(OnlyYColumn)
            ! call MPI_Bcast(OnlyYColumn, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(LSRAdjustementFitFlag, 1, NumberOutputFiles, "LSRAdjustementFitFlag")
            ! nelements = size(LSRAdjustementFitFlag)
            ! call MPI_Bcast(LSRAdjustementFitFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(NormalizationFlag, 1, NumberOutputFiles, "NormalizationFlag")
            ! nelements = size(NormalizationFlag)
            ! call MPI_Bcast(NormalizationFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(ExpData_reversed_flag, 1, NumberExpFiles, "ExpData_reversed_flag")
            ! nelements = size(ExpData_reversed_flag)
            ! call MPI_Bcast(ExpData_reversed_flag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< variables for Module 'FunctionCalling'
            call broadcast_scalar_int(NumberOfUsedThreads, "NumberOfUsedThreads")
            ! call MPI_Bcast(NumberOfUsedThreads, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberLinesOutput, "NumberLinesOutput")
            ! call MPI_Bcast(NumberLinesOutput, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberColumnsBegin, "NumberColumnsBegin")
            ! call MPI_Bcast(NumberColumnsBegin, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberColumnsEnd, "NumberColumnsEnd")
            ! call MPI_Bcast(NumberColumnsEnd, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(DetChi2, "DetChi2")
            ! call MPI_Bcast(DetChi2, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(idum, "idum")
            ! call MPI_Bcast(idum, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(GradientMethod, "GradientMethod")
            ! call MPI_Bcast(GradientMethod, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_dbl(GradientVariationValue, "GradientVariationValue")
            ! call MPI_Bcast(GradientVariationValue, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(NaNReplaceString, "NaNReplaceString")
            ! call MPI_BCAST(NaNReplaceString, 10, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(PlotPID, "PlotPID")
            ! call MPI_BCAST(PlotPID, 100, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(CalculationMethod, "CalculationMethod")
            ! call MPI_BCAST(CalculationMethod, 20, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(PathStartScript, "PathStartScript")
            ! call MPI_BCAST(PathStartScript, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(ExeCommandStartScript, "ExeCommandStartScript")
            ! call MPI_BCAST(ExeCommandStartScript, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(CommandWordOutput, "CommandWordOutput")
            ! call MPI_BCAST(CommandWordOutput, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(InputDataPath, "InputDataPath")
            ! call MPI_BCAST(InputDataPath, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(debugflag, "debugflag")
            ! call MPI_BCAST(debugflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)


            !< allocate memory for variables for Module 'FunctionCalling'
            if (myrank /= root_tn) then                                                           !< only slave threads (de)allocate memory
                if (allocated(NumberParamPerLine)) then
                    deallocate(NumberParamPerLine, ParameterName, ParameterFormat, LeadingString, TrailingString, ParamVisible, FitFktInput, &
                               FitFktOutput, stat = dealloc_status)
                    if (dealloc_status /= 0) then                                           !< is all ok?
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                        write(logchannel,'(2x,"Can not deallocate variables for Module FunctionCalling for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine FunctionCalling:")'
                        print '(2x,"Can not deallocate variables for Module FunctionCalling for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("dealloc_status = ",I4)',dealloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                endif
                allocate(NumberParamPerLine(NumberInputFiles, MaxInputLines), &
                         ParameterName(NumberInputFiles, MaxInputLines, MaxParameter), &
                         ParameterFormat(0:NumberInputFiles, 0:MaxInputLines, 0:MaxParameter), &
                         LeadingString(NumberInputFiles, MaxInputLines, MaxParameter), &
                         TrailingString(NumberInputFiles, MaxInputLines, MaxParameter), &
                         ParamVisible(NumberInputFiles, MaxInputLines, MaxParameter), &
                         FitFktInput(NumberInputFiles), &
                         FitFktOutput(NumberOutputFiles), &
                         stat = alloc_status)
                if (alloc_status /= 0) then                                                 !< is all ok?
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                    write(logchannel,'(2x,"Can not allocate variables for Module FunctionCalling for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("alloc_status = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine BroadCastMPIVariables:")'
                    print '(2x,"Can not allocate variables for Module FunctionCalling for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("alloc_status = ",I4)',alloc_status
                    print '(" ")'
                    ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                    call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                    stop
                endif
                NumberParamPerLine = 0
                ParameterName = ""
                ParameterFormat = ""
                LeadingString = ""
                TrailingString = ""
                ParamVisible = ""
                FitFktInput = ""
                FitFktOutput = ""
            endif
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Allocation of variables of Module >>FunctionCalling<< done!")'
                flush(6)
            endif


            !< broadcast scalar variables for Module 'FunctionCalling'
            call broadcast_array_int_2d(NumberParamPerLine, 1, NumberInputFiles, 1, MaxInputLines, "NumberParamPerLine")
            ! nelements = size(NumberParamPerLine)
            ! call MPI_Bcast(NumberParamPerLine, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_3d(ParameterName, 1, NumberInputFiles, 1, MaxInputLines, 1, MaxParameter, 8192, "ParameterName")
            ! nelements = size(ParameterName) * 8192
            ! call MPI_BCAST(ParameterName, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_3d(ParameterFormat, 0, NumberInputFiles, 0, MaxInputLines, 0, MaxParameter, 10, "ParameterFormat")
            ! nelements = size(ParameterFormat) * 10
            ! call MPI_BCAST(ParameterFormat, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_3d(LeadingString, 1, NumberInputFiles, 1, MaxInputLines, 1, MaxParameter, 8192, "LeadingString")
            ! nelements = size(LeadingString) * 8192
            ! call MPI_BCAST(LeadingString, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_3d(TrailingString, 1, NumberInputFiles, 1, MaxInputLines, 1, MaxParameter, 8192, "TrailingString")
            ! nelements = size(TrailingString) * 8192
            ! call MPI_BCAST(TrailingString, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_3d(ParamVisible, 1, NumberInputFiles, 1, MaxInputLines, 1, MaxParameter, 8192, "ParamVisible")
            ! nelements = size(ParamVisible) * 8192
            ! call MPI_BCAST(ParamVisible, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(FitFktInput, 1, NumberInputFiles, 8192, "FitFktInput")
            ! nelements = size(FitFktInput) * 8192
            ! call MPI_BCAST(FitFktInput, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(FitFktOutput, 1, NumberOutputFiles, 8192, "FitFktOutput")
            ! nelements = size(FitFktOutput) * 8192
            ! call MPI_BCAST(FitFktOutput, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< broadcast scalar variables for Module 'Model'
            call broadcast_scalar_int(ErrChannelIndex, "ErrChannelIndex")
            ! call MPI_Bcast(ErrChannelIndex, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(TotalNumberDataPoints, "TotalNumberDataPoints")
            ! call MPI_Bcast(TotalNumberDataPoints, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberMolecules, "NumberMolecules")
            ! call MPI_Bcast(NumberMolecules, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(TotalNumberComponents, "TotalNumberComponents")
            ! call MPI_Bcast(TotalNumberComponents, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(TotalNumberOfFrequencyRanges, "TotalNumberOfFrequencyRanges")
            ! call MPI_Bcast(TotalNumberOfFrequencyRanges, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberDistances, "NumberDistances")
            ! call MPI_Bcast(NumberDistances, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(TotalNumberOfMolecules, "TotalNumberOfMolecules")
            ! call MPI_Bcast(TotalNumberOfMolecules, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberOfTemperatures, "NumberOfTemperatures")
            ! call MPI_Bcast(NumberOfTemperatures, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberMoleculePartFunc, "NumberMoleculePartFunc")
            ! call MPI_Bcast(NumberMoleculePartFunc, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(Firsti, "Firsti")
            ! call MPI_Bcast(Firsti, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(Lasti, "Lasti")
            ! call MPI_Bcast(Lasti, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(stepi, "stepi")
            ! call MPI_Bcast(stepi, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(TotalNumberOfTransitions, "TotalNumberOfTransitions")
            ! call MPI_Bcast(TotalNumberOfTransitions, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(Column300KPartFunc, "Column300KPartFunc")
            ! call MPI_Bcast(Column300KPartFunc, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberOfIsomasters, "NumberOfIsomasters")
            ! call MPI_Bcast(NumberOfIsomasters, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberOfGlobalISORatios, "NumberOfGlobalISORatios")
            ! call MPI_Bcast(NumberOfGlobalISORatios, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberModelPixelXX, "NumberModelPixelXX")
            ! call MPI_Bcast(NumberModelPixelXX, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(NumberModelPixelYY, "NumberModelPixelYY")
            ! call MPI_Bcast(NumberModelPixelYY, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_dbl(SizeOfPixel_deg, "SizeOfPixel_deg")
            ! call MPI_Bcast(SizeOfPixel_deg, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_dbl(TempLow, "TempLow")
            ! call MPI_Bcast(TempLow, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_dbl(TempHigh, "TempHigh")
            ! call MPI_Bcast(TempHigh, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(XCLASSRootDir, "XCLASSRootDir")
            ! call MPI_Bcast(XCLASSRootDir, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(ColumnNameForNamePartFunc, "ColumnNameForNamePartFunc")
            ! call MPI_Bcast(ColumnNameForNamePartFunc, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(ColumnNameForNameTransitions, "ColumnNameForNameTransitions")
            ! call MPI_Bcast(ColumnNameForNameTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(ColumnNameForFreqTransitions, "ColumnNameForFreqTransitions")
            ! call MPI_Bcast(ColumnNameForFreqTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(ColumnNameForIntTransitions, "ColumnNameForIntTransitions")
            ! call MPI_Bcast(ColumnNameForIntTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(ColumnNameForEinsteinATransitions, "ColumnNameForEinsteinATransitions")
            ! call MPI_Bcast(ColumnNameForEinsteinATransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(ColumnNameForFreqErrTransitions, "ColumnNameForFreqErrTransitions")
            ! call MPI_Bcast(ColumnNameForFreqErrTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(ColumnNameForELowTransitions, "ColumnNameForELowTransitions")
            ! call MPI_Bcast(ColumnNameForELowTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(ColumnNameForgUpTransitions, "ColumnNameForgUpTransitions")
            ! call MPI_Bcast(ColumnNameForgUpTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(NameOfPartFuncTable, "NameOfPartFuncTable")
            ! call MPI_Bcast(NameOfPartFuncTable, 128, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(NameOfRadTransTable, "NameOfRadTransTable")
            ! call MPI_Bcast(NameOfRadTransTable, 128, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(dbName, "dbName")
            ! call MPI_Bcast(dbName, 4096, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(IsoTableFileName, "IsoTableFileName")
            ! call MPI_Bcast(IsoTableFileName, 4096, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(UseRegistrationXMLFile, "UseRegistrationXMLFile")
            ! call MPI_Bcast(UseRegistrationXMLFile, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(IsoFlag, "IsoFlag")
            ! call MPI_Bcast(IsoFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(SubBeamDescriptionFlag, "SubBeamDescriptionFlag")
            ! call MPI_Bcast(SubBeamDescriptionFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(LocalOverlapFlag, "LocalOverlapFlag")
            ! call MPI_Bcast(LocalOverlapFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)


            !< allocate memory for variables for Module 'Model'
            MaxNumberOfMolecules = NumberMolecules + 2 * NumberOfIsomasters
            l = max0(1, TotalNumberOfTransitions)
            k = max0(1, NumberDistances)
            if (myrank /= root_tn) then                                                           !< only slave threads (de)allocate memory
                if (allocated(CompMoleculeIndex)) then
                    deallocate(CompMoleculeIndex, SpectrumIndexForFreqRange, NumberComponentsPerMolecule, IsoNfitConversionTable, &
                               ConversionTableMAGIXmyXCLASSParam, DataPointIndexFreqRange, NumEntriesRanges, &
                               MolecularData, MolecularDataIndices, Redshift_Range, &
                               TempPartFunc, StartFrequency, EndFrequency, StepFrequency, tbFlag, BackgroundTemperatureRange, TemperatureSlopeRange, &
                               BackgroundFilenNames, BackgroundFromFile, HydrogenColumnDensityRange, DustBetaRange, KappaRange, DustFilenNames, &
                               GlobalvLSR, TelescopeSize, TelescopeBMIN, TelescopeBMAJ, TelescopeBPA, InterFlag, IsoRatio, IsoRatioConversionTable, &
                               lgQ, myXCLASSParameter, GeometryFlag, tdFlag, nHFlag, LTEFlag, ThinFlag, vWidthLimits, vOffLimits, &
                               ObservationalDataList, MolNamePartFunc, ColumnNamesPartFunc, MoleculeNames, IsoMolecule, IsoMasters, DustTauFromFile, &
                               LowerQNString, UpperQNString, KindOfMolecule, LineProfileFunction, &
                               ContPhen_Range, Dust_Flag_Range, &
                               Phen_Flag_Range, SortedDistances, PureDistances, DistanceOrderingArray, NumTransFreqPerObsFreq, stat = dealloc_status)
                    if (dealloc_status /= 0) then                                           !< is all ok?
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                        write(logchannel,'(2x,"Can not deallocate variables for Module Model for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine FunctionCalling:")'
                        print '(2x,"Can not deallocate variables for Module Model for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("dealloc_status = ",I4)',dealloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                endif
                allocate(CompMoleculeIndex(TotalNumberComponents), &
                         SpectrumIndexForFreqRange(TotalNumberOfFrequencyRanges), &
                         NumberComponentsPerMolecule(0:TotalNumberOfMolecules), &
                         IsoNfitConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules, 2), &
                         ConversionTableMAGIXmyXCLASSParam(NumberFreeParameter, 2), &
                         DataPointIndexFreqRange(TotalNumberOfFrequencyRanges, 2), &
                         NumEntriesRanges(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules), &
                         MolecularDataIndices(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2), &
                         MolecularData(l, 9), &
                         TempPartFunc(NumberOfTemperatures), &
                         StartFrequency(TotalNumberOfFrequencyRanges), &
                         EndFrequency(TotalNumberOfFrequencyRanges), &
                         StepFrequency(TotalNumberOfFrequencyRanges), &
                         tbFlag(TotalNumberOfFrequencyRanges), &
                         BackgroundTemperatureRange(TotalNumberOfFrequencyRanges), &
                         TemperatureSlopeRange(TotalNumberOfFrequencyRanges), &
                         BackgroundFilenNames(TotalNumberOfFrequencyRanges), &
                         BackgroundFromFile(TotalNumberDataPoints), &
                         HydrogenColumnDensityRange(TotalNumberOfFrequencyRanges), &
                         DustBetaRange(TotalNumberOfFrequencyRanges), &
                         KappaRange(TotalNumberOfFrequencyRanges), &
                         DustFilenNames(TotalNumberOfFrequencyRanges), &
                         GlobalvLSR(TotalNumberOfFrequencyRanges), &
                         TelescopeSize(TotalNumberOfFrequencyRanges), &
                         TelescopeBMIN(TotalNumberOfFrequencyRanges), &
                         TelescopeBMAJ(TotalNumberOfFrequencyRanges), &
                         TelescopeBPA(TotalNumberOfFrequencyRanges), &
                         InterFlag(TotalNumberOfFrequencyRanges), &
                         IsoRatioConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules), &
                         lgQ(NumberOfTemperatures, NumberMoleculePartFunc), &
                         MolNamePartFunc(NumberMoleculePartFunc), &
                         myXCLASSParameter(16, TotalNumberComponents), &
                         GeometryFlag(TotalNumberComponents), &
                         tdFlag(TotalNumberComponents), &
                         nHFlag(TotalNumberComponents), &
                         LTEFlag(TotalNumberComponents), &
                         ThinFlag(TotalNumberComponents), &
                         vWidthLimits(TotalNumberComponents, 4), &
                         vOffLimits(TotalNumberComponents, 3), &
                         ObservationalDataList(TotalNumberDataPoints, 3), &
                         ColumnNamesPartFunc(NumberOfTemperatures), &
                         MoleculeNames(MaxNumberOfMolecules), &
                         DustTauFromFile(TotalNumberDataPoints), &
                         LowerQNString(l), &
                         UpperQNString(l), &
                         KindOfMolecule(TotalNumberComponents), &
                         LineProfileFunction(TotalNumberComponents), &
                         ContPhen_Range(6, TotalNumberOfFrequencyRanges), &
                         Dust_Flag_Range(TotalNumberOfFrequencyRanges), &
                         Phen_Flag_Range(TotalNumberOfFrequencyRanges), &
                         SortedDistances(TotalNumberComponents), &
                         PureDistances(TotalNumberComponents), &
                         Redshift_Range(TotalNumberOfFrequencyRanges), &
                         DistanceOrderingArray(TotalNumberComponents, TotalNumberComponents), &
                         NumTransFreqPerObsFreq(MaxNumberOfMolecules, TotalNumberDataPoints), &
                         stat = alloc_status)
                if (alloc_status /= 0) then                                                 !< is all ok?
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                    write(logchannel,'(2x,"Can not allocate variables for Module Model for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("alloc_status = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine BroadCastMPIVariables:")'
                    print '(2x,"Can not allocate variables for Module Model for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("alloc_status = ",I4)',alloc_status
                    print '(" ")'
                    ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                    call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                    stop
                endif
                CompMoleculeIndex = 0
                SpectrumIndexForFreqRange = 0
                NumberComponentsPerMolecule = 0
                IsoNfitConversionTable = 0
                ConversionTableMAGIXmyXCLASSParam = 0
                DataPointIndexFreqRange = 0
                DistanceOrderingArray = 0
                NumEntriesRanges = 0
                MolecularDataIndices = 0
                TempPartFunc = 0.d0
                StartFrequency = 0.d0
                EndFrequency = 0.d0
                StepFrequency = 0.d0
                tbFlag = .true.
                BackgroundTemperatureRange = 0.d0
                TemperatureSlopeRange = 0.d0
                BackgroundFilenNames = ""
                BackgroundFromFile = 0.d0
                HydrogenColumnDensityRange = 0.d0
                DustBetaRange = 0.d0
                KappaRange = 0.d0
                ContPhen_Range = 0.d0
                Dust_Flag_Range = .false.
                Phen_Flag_Range = .false.
                DustFilenNames = ""
                GlobalvLSR = 0.d0
                TelescopeSize = 0.d0
                TelescopeBMIN = 0.d0
                TelescopeBMAJ = 0.d0
                TelescopeBPA = 0.d0
                InterFlag = .true.
                IsoRatioConversionTable = 0.d0
                MolecularData = 0.d0
                lgQ = 0.d0
                KindOfMolecule = 0
                LineProfileFunction = 1
                myXCLASSParameter = 0.d0
                GeometryFlag = 0
                tdFlag = .false.
                nHFlag = .false.
                LTEFlag = .true.
                ThinFlag = .false.
                vWidthLimits = 0.d0
                vOffLimits = 0.d0
                SortedDistances = 0.d0
                ObservationalDataList = 0.d0
                NumTransFreqPerObsFreq = 0
                Redshift_Range = 0.d0
                PureDistances = 0.d0
                DustTauFromFile = 0.d0
                MolNamePartFunc = ""
                ColumnNamesPartFunc = ""
                MoleculeNames = ""
                LowerQNString = ""
                UpperQNString = ""
            endif
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            if (dbflag .and. myrank == root_tn) then
                print '(11x, "Allocation of variables of Module >>Model<< done!")'
                flush(6)
            endif


            call broadcast_array_int_1d(CompMoleculeIndex, 1, TotalNumberComponents, "CompMoleculeIndex")
            ! nelements = size(CompMoleculeIndex)
            ! call MPI_Bcast(CompMoleculeIndex, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_1d(SpectrumIndexForFreqRange, 1, TotalNumberOfFrequencyRanges, "SpectrumIndexForFreqRange")
            ! nelements = size(SpectrumIndexForFreqRange)
            ! call MPI_Bcast(SpectrumIndexForFreqRange, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_1d(NumberComponentsPerMolecule, 0, TotalNumberOfMolecules, "NumberComponentsPerMolecule")
            ! nelements = size(NumberComponentsPerMolecule)
            ! call MPI_Bcast(NumberComponentsPerMolecule, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_3d(IsoNfitConversionTable, 1, TotalNumberOfMolecules, 1, TotalNumberOfMolecules, 1, 2, "IsoNfitConversionTable")
            ! nelements = size(IsoNfitConversionTable)
            ! call MPI_Bcast(IsoNfitConversionTable, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_2d(ConversionTableMAGIXmyXCLASSParam, 1, NumberFreeParameter, 1, 2, "ConversionTableMAGIXmyXCLASSParam")
            ! nelements = size(ConversionTableMAGIXmyXCLASSParam)
            ! call MPI_Bcast(ConversionTableMAGIXmyXCLASSParam, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_2d(DataPointIndexFreqRange, 1, TotalNumberOfFrequencyRanges, 1, 2, "DataPointIndexFreqRange")
            ! nelements = size(DataPointIndexFreqRange)
            ! call MPI_Bcast(DataPointIndexFreqRange, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_2d(DistanceOrderingArray, 1, TotalNumberComponents, 1, TotalNumberComponents, "DistanceOrderingArray")
            ! nelements = size(DistanceOrderingArray)
            ! call MPI_Bcast(DistanceOrderingArray, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_2d(NumEntriesRanges, 1, TotalNumberOfFrequencyRanges, 1, TotalNumberOfMolecules, "NumEntriesRanges")
            ! nelements = size(NumEntriesRanges)
            ! call MPI_Bcast(NumEntriesRanges, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_3d(MolecularDataIndices, 1, TotalNumberOfFrequencyRanges, 1, TotalNumberOfMolecules, 1, 2, "MolecularDataIndices")
            ! nelements = size(MolecularDataIndices)
            ! call MPI_Bcast(MolecularDataIndices, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(TempPartFunc, 1, NumberOfTemperatures, "TempPartFunc")
            ! nelements = size(TempPartFunc)
            ! call MPI_Bcast(TempPartFunc, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(StartFrequency, 1, TotalNumberOfFrequencyRanges, "StartFrequency")
            ! nelements = size(StartFrequency)
            ! call MPI_Bcast(StartFrequency, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(EndFrequency, 1, TotalNumberOfFrequencyRanges, "EndFrequency")
            ! nelements = size(EndFrequency)
            ! call MPI_Bcast(EndFrequency, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(StepFrequency, 1, TotalNumberOfFrequencyRanges, "StepFrequency")
            ! nelements = size(StepFrequency)
            ! call MPI_Bcast(StepFrequency, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(tbFlag, 1, TotalNumberOfFrequencyRanges, "tbFlag")
            ! nelements = size(tbFlag)
            ! call MPI_Bcast(tbFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(BackgroundTemperatureRange, 1, TotalNumberOfFrequencyRanges, "BackgroundTemperatureRange")
            ! nelements = size(BackgroundTemperatureRange)
            ! call MPI_Bcast(BackgroundTemperatureRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(TemperatureSlopeRange, 1, TotalNumberOfFrequencyRanges, "TemperatureSlopeRange")
            ! nelements = size(TemperatureSlopeRange)
            ! call MPI_Bcast(TemperatureSlopeRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(BackgroundFilenNames, 1, TotalNumberOfFrequencyRanges, 4096, "BackgroundFilenNames")
            ! nelements = size(BackgroundFilenNames) * 4096
            ! call MPI_Bcast(BackgroundFilenNames, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(BackgroundFromFile, 1, TotalNumberDataPoints, "BackgroundFromFile")
            ! nelements = size(BackgroundFromFile)
            ! call MPI_Bcast(BackgroundFromFile, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(HydrogenColumnDensityRange, 1, TotalNumberOfFrequencyRanges, "HydrogenColumnDensityRange")
            ! nelements = size(HydrogenColumnDensityRange)
            ! call MPI_Bcast(HydrogenColumnDensityRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(DustBetaRange, 1, TotalNumberOfFrequencyRanges, "DustBetaRange")
            ! nelements = size(DustBetaRange)
            ! call MPI_Bcast(DustBetaRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(KappaRange, 1, TotalNumberOfFrequencyRanges, "KappaRange")
            ! nelements = size(KappaRange)
            ! call MPI_Bcast(KappaRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(ContPhen_Range, 1, 6, 1, TotalNumberOfFrequencyRanges, "ContPhen_Range")
            ! nelements = size(ContPhen_Range)
            ! call MPI_Bcast(ContPhen_Range, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(DustFilenNames, 1, TotalNumberOfFrequencyRanges, 4096, "DustFilenNames")
            ! nelements = size(DustFilenNames) * 4096
            ! call MPI_Bcast(DustFilenNames, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(GlobalvLSR, 1, TotalNumberOfFrequencyRanges, "GlobalvLSR")
            ! nelements = size(GlobalvLSR)
            ! call MPI_Bcast(GlobalvLSR, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(Redshift_Range, 1, TotalNumberOfFrequencyRanges, "Redshift_Range")
            ! nelements = size(Redshift_Range)
            ! call MPI_Bcast(Redshift_Range, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(TelescopeSize, 1, TotalNumberOfFrequencyRanges, "TelescopeSize")
            ! nelements = size(TelescopeSize)
            ! call MPI_Bcast(TelescopeSize, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(TelescopeBMIN, 1, TotalNumberOfFrequencyRanges, "TelescopeBMIN")
            ! nelements = size(TelescopeBMIN)
            ! call MPI_Bcast(TelescopeBMIN, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(TelescopeBMAJ, 1, TotalNumberOfFrequencyRanges, "TelescopeBMAJ")
            ! nelements = size(TelescopeBMAJ)
            ! call MPI_Bcast(TelescopeBMAJ, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(TelescopeBPA, 1, TotalNumberOfFrequencyRanges, "TelescopeBPA")
            ! nelements = size(TelescopeBPA)
            ! call MPI_Bcast(TelescopeBPA, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(InterFlag, 1, TotalNumberOfFrequencyRanges, "InterFlag")
            ! nelements = size(InterFlag)
            ! call MPI_Bcast(InterFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(IsoRatioConversionTable, 1, TotalNumberOfMolecules, 1, TotalNumberOfMolecules, "IsoRatioConversionTable")
            ! nelements = size(IsoRatioConversionTable)
            ! call MPI_Bcast(IsoRatioConversionTable, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(MolecularData, 1, l, 1, 9, "MolecularData")
            ! nelements = size(MolecularData)
            ! call MPI_Bcast(MolecularData, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(LowerQNString, 1, l, 60, "LowerQNString")
            ! nelements = size(LowerQNString) * 60
            ! call MPI_Bcast(LowerQNString, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(UpperQNString, 1, l, 60, "UpperQNString")
            ! nelements = size(UpperQNString) * 60
            ! call MPI_Bcast(UpperQNString, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(lgQ, 1, NumberOfTemperatures, 1, NumberMoleculePartFunc, "lgQ")
            ! nelements = size(lgQ)
            ! call MPI_Bcast(lgQ, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(myXCLASSParameter, 1, 16, 1, TotalNumberComponents, "myXCLASSParameter")
            ! nelements = size(myXCLASSParameter)
            ! call MPI_Bcast(myXCLASSParameter, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_1d(KindOfMolecule, 1, TotalNumberComponents, "KindOfMolecule")
            ! nelements = size(KindOfMolecule)
            ! call MPI_Bcast(KindOfMolecule, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_1d(LineProfileFunction, 1, TotalNumberComponents, "LineProfileFunction")
            ! nelements = size(LineProfileFunction)
            ! call MPI_Bcast(LineProfileFunction, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_int_1d(GeometryFlag, 1, TotalNumberComponents, "GeometryFlag")
            ! nelements = size(GeometryFlag)
            ! call MPI_Bcast(GeometryFlag, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(tdFlag, 1, TotalNumberComponents, "tdFlag")
            ! nelements = size(tdFlag)
            ! call MPI_Bcast(tdFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(LTEFlag, 1, TotalNumberComponents, "LTEFlag")
            ! nelements = size(LTEFlag)
            ! call MPI_Bcast(LTEFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(ThinFlag, 1, TotalNumberComponents, "ThinFlag")
            ! nelements = size(ThinFlag)
            ! call MPI_Bcast(ThinFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(nHFlag, 1, TotalNumberComponents, "nHFlag")
            ! nelements = size(nHFlag)
            ! call MPI_Bcast(nHFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(vWidthLimits, 1, TotalNumberComponents, 1, 4, "vWidthLimits")
            ! nelements = size(vWidthLimits)
            ! call MPI_Bcast(vWidthLimits, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(vOffLimits, 1, TotalNumberComponents, 1, 3, "vOffLimits")
            ! nelements = size(vOffLimits)
            ! call MPI_Bcast(vOffLimits, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(SortedDistances, 1, TotalNumberComponents, "SortedDistances")
            ! nelements = size(SortedDistances)
            ! call MPI_Bcast(SortedDistances, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_2d(ObservationalDataList, 1, TotalNumberDataPoints, 1, 3, "ObservationalDataList")
            ! nelements = size(ObservationalDataList)
            ! call MPI_Bcast(ObservationalDataList, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(PureDistances, 1, TotalNumberComponents, "PureDistances")
            ! nelements = size(PureDistances)
            ! call MPI_Bcast(PureDistances, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_dbl_1d(DustTauFromFile, 1, TotalNumberDataPoints, "DustTauFromFile")
            ! nelements = size(DustTauFromFile)
            ! call MPI_Bcast(DustTauFromFile, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(MolNamePartFunc, 1, NumberMoleculePartFunc, 40, "MolNamePartFunc")
            ! nelements = size(MolNamePartFunc) * 40
            ! call MPI_Bcast(MolNamePartFunc, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(ColumnNamesPartFunc, 1, NumberOfTemperatures, 40, "ColumnNamesPartFunc")
            ! nelements = size(ColumnNamesPartFunc) * 40
            ! call MPI_Bcast(ColumnNamesPartFunc, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_char_1d(MoleculeNames, 1, MaxNumberOfMolecules, 40, "MoleculeNames")
            ! nelements = size(MoleculeNames) * 40
            ! call MPI_Bcast(MoleculeNames, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(Dust_Flag_Range, 1, TotalNumberOfFrequencyRanges, "Dust_Flag_Range")
            ! nelements = size(Dust_Flag_Range)
            ! call MPI_Bcast(Dust_Flag_Range, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_array_log_1d(Phen_Flag_Range, 1, TotalNumberOfFrequencyRanges, "Phen_Flag_Range")
            ! nelements = size(Phen_Flag_Range)
            ! call MPI_Bcast(Phen_Flag_Range, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_char(EmsAbsPATH, "EmsAbsPATH")
            ! call MPI_Bcast(EmsAbsPATH, 4096, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_log(UseEmAbsFuncFlag, "UseEmAbsFuncFlag")
            ! call MPI_Bcast(UseEmAbsFuncFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)


            !< allocate memory for variable for sub-beam description
            if (UseEmAbsFuncFlag) then
                if (myrank /= root_tn) then                                                       !< only slave threads (de)allocate memory
                    if (allocated(EmsAbsFunc)) then
                        deallocate(EmsAbsFunc, stat = dealloc_status)
                        if (dealloc_status /= 0) then                                       !< is all ok?
                            write(logchannel,*)
                            write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                            write(logchannel,'(2x,"Can not deallocate variable EmsAbsFunc for thread = ", I5, ".")') myrank
                            write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                            write(logchannel,*)
                            write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                            write(logchannel,'(" ")')
                            write(logchannel,'("Program aborted!")')
                            close(logchannel)

                            print '(" ")'
                            print '("Error in subroutine FunctionCalling:")'
                            print '(2x,"Can not deallocate variable EmsAbsFunc for thread = ", I5, ".")', myrank
                            print '(2x,"Please close all other programs and restart the program!")'
                            print '(" ")'
                            print '("dealloc_status = ",I4)', dealloc_status
                            print '(" ")'
                            ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                            call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                            stop
                        endif
                    endif
                    allocate(EmsAbsFunc(TotalNumberDataPoints, NumberDistances, 2), stat = alloc_status)
                    if (alloc_status /= 0) then                                                 !< is all ok?
                        write(logchannel,'(" ")')
                        write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                        write(logchannel,'(2x,"Can not allocate variable EmsAbsFunc for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,'(" ")')
                        write(logchannel,'("alloc_status = ",I4)') alloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine BroadCastMPIVariables:")'
                        print '(2x,"Can not allocate variable EmsAbsFunc for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("alloc_status = ",I4)',alloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                    EmsAbsFunc = 0.d0
                endif
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)

                call broadcast_array_dbl_3d(EmsAbsFunc, 1, TotalNumberDataPoints, 1, NumberDistances, 1, 2, "EmsAbsFunc")
                ! nelements = size(EmsAbsFunc)
                ! call MPI_Bcast(EmsAbsFunc, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for variables for sub-beam description
            if (SubBeamDescriptionFlag) then
                if (myrank /= root_tn) then                                                           !< only slave threads (de)allocate memory
                    if (allocated(LayerMap)) then
                        deallocate(LayerMap, ConfigList, ConfigIndex, GausssianBeamMap, stat = dealloc_status)
                        if (dealloc_status /= 0) then                                           !< is all ok?
                            write(logchannel,*)
                            write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                            write(logchannel,'(2x,"Can not deallocate variables for Module Model for thread = ", I5, ".")') myrank
                            write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                            write(logchannel,*)
                            write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                            write(logchannel,'(" ")')
                            write(logchannel,'("Program aborted!")')
                            close(logchannel)

                            print '(" ")'
                            print '("Error in subroutine FunctionCalling:")'
                            print '(2x,"Can not deallocate variables for Module Model for thread = ", I5, ".")', myrank
                            print '(2x,"Please close all other programs and restart the program!")'
                            print '(" ")'
                            print '("dealloc_status = ",I4)',dealloc_status
                            print '(" ")'
                            ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                            call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                            stop
                        endif
                    endif
                    allocate(LayerMap(NumberModelPixelXX, NumberModelPixelYY, TotalNumberComponents), &
                            ConfigList(NumberModelPixelXX * NumberModelPixelYY, TotalNumberComponents), &
                            ConfigIndex(NumberModelPixelXX, NumberModelPixelYY), &
                            GausssianBeamMap(NumberModelPixelXX, NumberModelPixelYY), stat = alloc_status)
                    if (alloc_status /= 0) then                                                 !< is all ok?
                        write(logchannel,'(" ")')
                        write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                        write(logchannel,'(2x,"Can not allocate variables for Module Model for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,'(" ")')
                        write(logchannel,'("alloc_status = ",I4)') alloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine BroadCastMPIVariables:")'
                        print '(2x,"Can not allocate variables for Module Model for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("alloc_status = ",I4)',alloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                    LayerMap = .false.
                    ConfigList = .false.
                    ConfigIndex = 0
                    GausssianBeamMap = 0.d0
                endif
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)


                call broadcast_array_log_3d(LayerMap, 1, NumberModelPixelXX, 1, NumberModelPixelYY, 1, TotalNumberComponents, "LayerMap")
                ! nelements = size(LayerMap)
                ! call MPI_Bcast(LayerMap, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

                call broadcast_array_log_2d(ConfigList, 1, NumberModelPixelXX * NumberModelPixelYY, 1, TotalNumberComponents, "ConfigList")
                ! nelements = size(ConfigList)
                ! call MPI_Bcast(ConfigList, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

                call broadcast_array_int_2d(ConfigIndex, 1, NumberModelPixelXX, 1, NumberModelPixelYY, "iupp")
                ! nelements = size(ConfigIndex)
                ! call MPI_Bcast(ConfigIndex, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

                call broadcast_array_dbl_2d(GausssianBeamMap, 1, NumberModelPixelXX, 1, NumberModelPixelYY, "GausssianBeamMap")
                ! nelements = size(GausssianBeamMap)
                ! call MPI_Bcast(GausssianBeamMap, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            endif
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< variables for iso ratio file
            if (IsoFlag) then
                if (myrank /= root_tn) then                                                       !< only slave threads (de)allocate memory
                    if (allocated(IsoMolecule)) then
                        deallocate(IsoMolecule, IsoMasters, IsoRatio, stat = dealloc_status)
                        if (dealloc_status /= 0) then                                           !< is all ok?
                            write(logchannel,*)
                            write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                            write(logchannel,'(2x,"Can not deallocate variables IsoMolecule, IsoMasters, and IsoRatio for thread = ", I5, ".")') myrank
                            write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                            write(logchannel,*)
                            write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                            write(logchannel,'(" ")')
                            write(logchannel,'("Program aborted!")')
                            close(logchannel)

                            print '(" ")'
                            print '("Error in subroutine FunctionCalling:")'
                            print '(2x,"Can not deallocate variables IsoMolecule, IsoMasters, and IsoRatio for thread = ", I5, ".")', myrank
                            print '(2x,"Please close all other programs and restart the program!")'
                            print '(" ")'
                            print '("dealloc_status = ",I4)', dealloc_status
                            print '(" ")'
                            ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                            call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                            stop
                        endif
                    endif
                    allocate(IsoMolecule(NumberOfIsomasters), IsoRatio(NumberOfIsomasters, 2), IsoMasters(NumberOfIsomasters), stat = alloc_status)
                    if (alloc_status /= 0) then                                                 !< is all ok?
                        write(logchannel,'(" ")')
                        write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                        write(logchannel,'(2x,"Can not allocate variables IsoMolecule, IsoMasters, and IsoRatio for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,'(" ")')
                        write(logchannel,'("alloc_status = ",I4)') alloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine BroadCastMPIVariables:")'
                        print '(2x,"Can not allocate variables IsoMolecule, IsoMasters, and IsoRatio for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("alloc_status = ",I4)', alloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                    IsoMolecule = ""
                    IsoMasters = ""
                    IsoRatio = 0.d0


                    if (NumberOfGlobalISORatios > 0) then
                        if (allocated(GlobalIsoRatioParameter)) then
                            deallocate(GlobalIsoRatioParameter, stat = dealloc_status)
                            if (dealloc_status /= 0) then                                           !< is all ok?
                                write(logchannel,*)
                                write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                                write(logchannel,'(2x,"Can not deallocate variable GlobalIsoRatioParameter for thread = ", I5, ".")') myrank
                                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                                write(logchannel,*)
                                write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                                write(logchannel,'(" ")')
                                write(logchannel,'("Program aborted!")')
                                close(logchannel)

                                print '(" ")'
                                print '("Error in subroutine FunctionCalling:")'
                                print '(2x,"Can not deallocate variable GlobalIsoRatioParameter for thread = ", I5, ".")', myrank
                                print '(2x,"Please close all other programs and restart the program!")'
                                print '(" ")'
                                print '("dealloc_status = ",I4)', dealloc_status
                                print '(" ")'
                                ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                                call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                                stop
                            endif
                        endif
                        allocate(GlobalIsoRatioParameter(NumberOfIsomasters, NumberOfGlobalISORatios, 2), stat = alloc_status)
                        if (alloc_status /= 0) then                                                 !< is all ok?
                            write(logchannel,'(" ")')
                            write(logchannel,'("Error in subroutine BroadCastMPIVariables:")')
                            write(logchannel,'(2x,"Can not allocate variable GlobalIsoRatioParameter for thread = ", I5, ".")') myrank
                            write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                            write(logchannel,'(" ")')
                            write(logchannel,'("alloc_status = ",I4)') alloc_status
                            write(logchannel,'(" ")')
                            write(logchannel,'("Program aborted!")')
                            close(logchannel)

                            print '(" ")'
                            print '("Error in subroutine BroadCastMPIVariables:")'
                            print '(2x,"Can not allocate variable GlobalIsoRatioParameter for thread = ", I5, ".")', myrank
                            print '(2x,"Please close all other programs and restart the program!")'
                            print '(" ")'
                            print '("alloc_status = ",I4)', alloc_status
                            print '(" ")'
                            ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                            call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                            stop
                        endif
                        GlobalIsoRatioParameter = 0
                    endif
                endif

                call broadcast_array_char_1d(IsoMolecule, 1, NumberOfIsomasters, 40, "IsoMolecule")
                ! nelements = size(IsoMolecule) * 40
                ! call MPI_Bcast(IsoMolecule, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

                call broadcast_array_char_1d(IsoMasters, 1, NumberOfIsomasters, 40, "IsoMasters")
                ! nelements = size(IsoMasters) * 40
                ! call MPI_Bcast(IsoMasters, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)

                call broadcast_array_dbl_2d(IsoRatio, 1, NumberOfIsomasters, 1, 2, "IsoRatio")
                ! nelements = size(IsoRatio)
                ! call MPI_Bcast(IsoRatio, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

                if (NumberOfGlobalISORatios > 0) then
                    call broadcast_array_int_3d(GlobalIsoRatioParameter, 1, NumberOfIsomasters, 1, NumberOfGlobalISORatios, 1, 2, &
                                                "GlobalIsoRatioParameter")
                    ! nelements = size(GlobalIsoRatioParameter)
                    ! call MPI_Bcast(GlobalIsoRatioParameter, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
                endif
            endif
            return
        end subroutine BroadCastMPIVariables


        !>************************************************************************************************************************************************
        !> subroutine: ModelInit
        !>
        !> initialize and broadcast module variables
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 26.08.2014
        !>
        subroutine ModelInit

            implicit none
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation


            !< initalize myXCLASS variables
            ParallelizationMethod = "MPI"
            IntegrationFlag = .true.
            LogFlag = .false.
            AllOutputFilesFlag = .false.
            call myXCLASSInitVar


            !< initialize MPI environment
            if (printflag) then
                print '(9x, "Initialize MPI environment .. ", $)'
            endif


            !< resize AtOnceFunction variable (only for MASTER thread)
            !< this section is neccessary to avoid changes in the Algorithm core routine
            if (allocated(AtOnceFunction)) then
                deallocate(AtOnceFunction, stat = deallocstatus)
                if (deallocstatus /= 0) then                                                !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine ModelInit:")')
                    write(logchannel,'(2x,"Can not deallocate variable AtOnceFunction for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ", I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine ModelInit:")'
                    print '(2x,"Can not deallocate variable AtOnceFunction for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ", I4)', deallocstatus
                    print '(" ")'
                    ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                    call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                    stop
                endif
            endif
            allocate(AtOnceFunction(0:0, NumberExpFiles, MaxColY, MaxExpLength), stat = allocstatus)
            if (allocstatus /= 0) then                                                 !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine ModelInit:")')
                write(logchannel,'(2x,"Can not allocate variable AtOnceFunction for thread = ", I5, ".")') myrank
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')
                close(logchannel)

                print '(" ")'
                print '("Error in subroutine ModelInit:")'
                print '(2x,"Can not allocate variable AtOnceFunction for thread = ", I5, ".")', myrank
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                stop
            endif
            AtOnceFunction = 0.d0


            !< redefine variable 'ParallelizationFlag'
            TotalNumberOfThreads = ParallelizationFlag
            NumberOfUsedThreads = 1                                                         !< in Module 'FunctionCalling' we need only one thread


            !< wake up slave threads
            call SlaveWork(2)


            !< broadcast variables
            call BroadCastMPIVariables


            !< we're finished
            if (printflag) then
                print '("done!")'
            endif


            !< sleep little slave threads
            call SlaveWork(4)
            return
        end subroutine ModelInit


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionLM
        !>
        !> calculates the chi2 values for the Levenberg-Marquard algorithm
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      a:                  array containing the parameter set
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      NumberFreeParameterCopy:    number of fitted parameters
        !>                      fitparam:           parameter which have to be optimized
        !>                      colx:               number of columns in experimental x data
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>                      alpha:              matrix alpha (only used for Levenberg-Marquardt algorithm)
        !>                      beta2:              beta2 array (only used for Levenberg-Marquardt algorithm)
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 31.07.2014
        !>
        subroutine ModelCalcChiFunctionLM(ma, a, ia, NumberFreeParameterCopy, fitparam, colx, NumFile, MaxL, MaxCol, FitFunctionOut, Chi2Values, &
                                          alpha, beta2)

            implicit none
            integer :: i, j, k, l, m, n, fitnum, NumberFile                                 !< loop variables
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: counter                                                              !< counter for ModelFunction
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            integer :: NumParameterVectors                                                  !< number of parameter vectors in ParameterVectorSet
            real*8 :: dy, sig2i, wt, ymod                                                   !< working variables
            real*8 :: value                                                                 !< calculated value of the fit function
            real*8 :: variation, d1, d2                                                     !< working variables
            real*8, dimension(ma) :: steph                                                  !< working variable
            real*8, dimension(ma) :: modelparam                                             !< array containing the parameter set
            real*8, dimension(ma) :: modelparamcopy                                         !< copy of modelparam
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(ma, ma) :: alpha                                              !< matrix alpha
            real*8, dimension(NumberFreeParameterCopy) :: beta2                             !< beta2 array
            real*8, dimension(NumberFreeParameterCopy) :: fitparam                          !< parameter which have to be optimized
            real*8, dimension(NumFile, MaxL, MaxCol) :: FitFunctionOut                      !< values of the fit function at all calculated points
            real*8, dimension(NumFile, MaxL, MaxCol) :: Chi2Values                          !< values of the fit function at all calculated points
            real*8, dimension(NumberFreeParameterCopy, NumFile, MaxCol, MaxL) :: GradientHelp
            real*8, dimension(ma) :: dyda                                                   !< gradient of the fit function
            real*8, dimension(NumberFreeParameterCopy + 1) :: chi2ValuesVector              !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            real*8, dimension(1, NumberFreeParameterCopy) :: ParameterVectorSetLocal        !< local copy of ParameterVectorSet
            real*8, dimension(NumberFreeParameterCopy + 1, NumberFreeParameterCopy) :: ParameterVectorSet
                                                                                            !< set of parameter vectors calculated by model module
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file


            character(len=50) :: valueh, valuel                                             !< working variable for determine gradient
            logical :: equal_flag                                                           !< required for string comparison
            logical, dimension(ma) :: ChangeSign_Flag                                       !< change sign flag
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize values for fit function calculation
            Do j = 1, NumberFreeParameterCopy                                               !< Initialize (symmetric) alpha, beta2.
                Do k = 1, j
                    alpha(j,k) = 0.d0                                                       !< clear alpha partially
                end Do
                beta2(j) = 0.d0                                                             !< clear beta2
            end Do


            !< update the parameter set with the current values of the fit parameters some algorithms optimize only the parameters in the array fitparam
            !< all other parameter in the parameter set a are not included in these algorithms but are required for the calculation of the fit function
            k = 0
            Do i = 1, ma
                if (ia(i)) then
                    k = k + 1
                    a(i) = fitparam(k)
                endif
            end Do
            ParameterVectorSet(1, :) = fitparam(:)                                          !< first entry of ParameterVectorSet contains the current
                                                                                            !< parameter vector
            ! Debug:
            ! print*,'>>>>>>>>>>>>>fitparam = ', fitparam(:)


            !< check, if gradient is necessary (for Levenberg-Marquardt it is always necessary!)
            NumParameterVectors = 1
            if (Gradientflag) then
                NumParameterVectors = NumberFreeParameterCopy + 1


                !< initialize some working parameter
                steph = 0.d0
                ChangeSign_Flag = .false.
                modelparamcopy = a                                                          !< dublicate array containing the parameter set


                !< calculate parameter vector for gradient calculation

                !$omp parallel default(shared) &
                !$omp shared(OutputFileFormat, NumberRanges, ValueEmptyOutputFile, LSRAdjustement, FirstPointExpData, LastPointExpData, idum) &
                !$omp shared(MinRange, MaxRange, NaNReplaceString, StandardWorkingDirectory, CharacterForComments, CharacterSeperatingColumns) &
                !$omp shared(ResamplingMethod, InterpolationMethod, RenormalizedChi2, OnlyYColumn, LSRAdjustementFitFlag, NormalizationFlag) &
                !$omp shared(AtOnceGradient, ia, paramset, modelparamcopy, ConverterInfit) &
                !$omp shared(ParameterName, ParameterFormat, LeadingString, TrailingString, ParamVisible, NumberLinesOutput) &
                !$omp shared(printflag, chisqValues, NumberExpFiles, modelnumber, lengthexpdata, currentpath, NumberHeaderLines, QualityLimit) &
                !$omp shared(FitParameterName, FitParameterValue, NumberColumnsBegin, NumberColumnsEnd, NumberParamPerLine, CalculationMethod) &
                !$omp shared(CommandWordOutput, DetChi2, MaxParameter, ExternalThreadNumber, expdatax, expdatay, expdatae, TempDirectory) &
                !$omp shared(NumberXColumns, NumberYColumns, MaxColX, MaxColY, parameternumber, NumberInputFiles, ParallelizationFlag, JobID) &
                !$omp shared(MaxInputLines, UseCalculationReduction, WriteChi2Flag, Gradientflag,  initflag, CalculatedParameterSets) &
                !$omp shared(LastAlgorithmFlag, ChangeSign_Flag, steph, NumberFreeParameterCopy, ParameterVectorSet, fitparam) &
                !$omp shared(CurrentNumberLinesCalcReduction, ochisq, NumberLinesChi2, InputDataPath, FitFktInput, ModelFunction, FitFktOutput) &
                !$omp shared(ExeCommandStartScript, NumberOutputFiles, CurrentExpFile, GradientMethod, PathStartScript, CurrentYColumn) &
                !$omp shared(ExpData_reversed_flag, BestSitesModelValues, BestSitesChi2Values, BestSitesParamSet, GradientVariationValue) &
                !$omp private(fitnum, i, j, k, modelparam, d1, d2, sig2i, NumberFile, variation, value, valueh, valuel) &
                !$omp private(IntegerTrue, NumInputFile_index, i_index, j_index, equal_flag)
                !$omp do

                Do fitnum = 1, NumberFreeParameterCopy                                      !< loop over all free parameter
                    !$omp critical
                    ParameterVectorSet(fitnum + 1, :) = fitparam(:)
                    !$omp end critical
                    modelparam = modelparamcopy                                             !< load unmodified parameter values
                    i = ConverterInfit(fitnum)                                              !< get appropriate parameter index within parameter set
                    call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)

                    ! Debug:
                    ! print*,'fitnum, NumInputFile_index, i_index, j_index = ', fitnum, NumInputFile_index, i_index, j_index


                    !< check if parameter is within parameter limits
                    if (modelparamcopy(i) < paramset(3, i) .or. modelparamcopy(i) > paramset(4, i)) then
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        write(logchannel,'(2x,"The parameter ",A," is out of limits.")') &
                                          trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        write(logchannel,'(2x,"Upper limit = ", ES25.15)') paramset(4, i)
                        write(logchannel,'(2x,"Lower limit = ", ES25.15)') paramset(3, i)
                        write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        print '(13x,"The parameter ",A," is out of limits.")', trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        print '(13x,"Upper limit = ", ES25.15)', paramset(4, i)
                        print '(13x,"Lower limit = ", ES25.15)', paramset(3, i)
                        print '(13x,"Value of parameter = ", ES25.15)',modelparamcopy(i)
                        print '(" ")'
                        print '(13x,"Program aborted!")'
                        stop
                    endif


                    !< determine strength of variation
                    variation = GradientVariationValue                                      !< variation of the parameter in percent/100
                    !$omp critical
                    steph(i) = dabs(modelparamcopy(i) * variation)                          !< define stepsize for foating point numbers
                    if (modelparamcopy(i) == 0.d0) then
                        if (IntegerTrue) then                                               !< is parameter an integer ??
                            steph(i) = 1.d0                                                 !< variation of the parameter in percent/100
                        else
                            steph(i) = variation                                            !< variation of the parameter in percent/100
                        endif
                    elseif (IntegerTrue .and. steph(i) < 1.d0) then
                        steph(i) = 1.d0
                    endif
                    !$omp end critical

                    ! Debug:
                    ! print*,'i = ', i
                    ! print*,'modelparamcopy(i) = ', modelparamcopy(i)
                    ! print*,'steph(i) = ',steph(i)
                    ! print*,'modelparamcopy(i) + steph(i) = ', modelparamcopy(i) + steph(i)
                    ! print*,'paramset(3, i) = ', paramset(3, i)
                    ! print*,'paramset(4, i) = ', paramset(4, i)
                    ! print*,'modelparamcopy(i) - steph(i) = ', modelparamcopy(i) - steph(i)


                    !< test, if we can accelerate the calculation
                    value = modelparamcopy(i) + steph(i)
                    if (value < paramset(3, i) .or. paramset(4, i) < value) then
                        if (value > paramset(4, i)) then                                    !< is f(x_i + h) > upper limit ?
                            value = modelparamcopy(i) - steph(i)                            !< use f(x_i - h)
                            if (value < paramset(3, i)) then
                                if (dabs(modelparamcopy(i) - paramset(4, i)) < 1.d-12) then
                                    write(logchannel,*)
                                    write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                                    write(logchannel,'(2x,"The gradient for parameter ",A," cannot be calculated.")') &
                                                      trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    write(logchannel,'(2x,"The variation runs out of limits.")')
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")')
                                    write(logchannel,*)
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Upper limit = ",ES25.15)') paramset(4, i)
                                    write(logchannel,'(2x,"Lower limit = ",ES25.15)') paramset(3, i)
                                    write(logchannel,'(2x,"value of variation = ", ES25.15)') variation
                                    write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                                    write(logchannel,'(" ")')
                                    write(logchannel,'("Program aborted!")')

                                    print '(" ")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '("Error in subroutine ModelCalcChiFunctionLM:")'
                                    print '(2x,"The gradient for parameter ",A," cannot be calculated.")', &
                                                      trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    print '(2x,"The variation runs out of limits.")'
                                    print '(" ")'
                                    print '(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '(2x,"Upper limit = ", ES25.15)', paramset(4, i)
                                    print '(2x,"Lower limit = ", ES25.15)', paramset(3, i)
                                    print '(2x,"value of variation = ", ES25.15)', variation
                                    print '(2x,"Value of parameter = ", ES25.15)', modelparamcopy(i)
                                    print '(" ")'
                                    print '("Program aborted!")'
                                    stop
                                endif
                                value = paramset(4, i)
                            else
                                !$omp critical
                                ChangeSign_Flag(i) = .true.
                                !$omp end critical
                            endif
                        endif
                    endif

                    ! Debug:
                    ! print*,'> value, modelparamcopy(i), steph(i) = ',value, modelparamcopy(i), steph(i)


                    !< check, if the variation leads in combination with the Format to a variation in the current parameter
                    if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                        .or. index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                        write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) int(value)
                        write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) int(modelparamcopy(i))
                    else
                        write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) value
                        write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) modelparamcopy(i)
                    endif
                    equal_flag = .true.
                    valueh = adjustl(valueh)
                    valuel = adjustl(valuel)
                    if (len_trim(valueh) == len_trim(valuel)) then
                        Do j = 1, len_trim(valueh)
                            if (valueh(j:j) /= valuel(j:j)) then
                                equal_flag = .false.
                                exit
                            endif
                        end Do
                    else
                        equal_flag = .false.
                    endif
                    if (equal_flag) then                                                    !< both expressions are equal
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        write(logchannel,'(2x,"The format specification of the parameter ",A)') &
                                trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        write(logchannel,'(2x,"prevents the variation of the current parameter.")')
                        write(logchannel,'(2x," ")')
                        write(logchannel,'(2x,"The gradient entry for this parameter is set to zero. Therefore")')
                        write(logchannel,'(2x,"no variation of this parameter in the current iteration is done")')
                        write(logchannel,'(" ")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        print '(13x,"The format specification of the parameter ",A)', &
                               trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        print '(13x,"prevents the variation of the current parameter.")'
                        print '(" ")'
                        print '(13x,"The gradient entry for this parameter is set to zero. Therefore")'
                        print '(13x,"no variation of this parameter in the current iteration is done")'
                        print '(" ")'

                        ! Debug:
                        !    print*," "
                        !    print*," "
                        !    print*,"valueh = >>", trim(adjustl(valueh)), "<<"
                        !    print*,"valuel = >>", trim(adjustl(valuel)), "<<"
                        !    print*,"NumInputFile_index, i_index, j_index = ", NumInputFile_index, i_index, j_index
                        !    print*,"ParameterName(NumInputFile_index, i_index, j_index) = ", &
                        !            trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        !    print*,"ParameterFormat(NumInputFile_index, i_index, j_index) = ", ParameterFormat(NumInputFile_index, i_index, j_index)
                        !    print*,"i, value, modelparamcopy(i) = ", i, value, modelparamcopy(i)
                        !    print*," "
                        !    print*," "
                        !    print*," "
                    endif


                    !< modify the ith parameter
                    !$omp critical
                    modelparam(i) = value                                                   !< modify value of the ith parameter
                    ParameterVectorSet(fitnum + 1, fitnum) = value
                    !$omp end critical

                end Do

                !$omp end do
                !$omp end parallel                                                          !< end of parallel environment

                modelparam = modelparamcopy                                                 !< restore old paramter values
            endif

            ! Debug:
            !    Do l = 2, NumParameterVectors
            !        Do k = 1, NumberFreeParameterCopy
            !            if (dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k)) > 1.d-6) then
            !                print*,'l, k, ParameterVectorSet(l, k) = ', l, k, ParameterVectorSet(l, k), ParameterVectorSet(1, k), &
            !                                                            dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k))
            !            endif
            !        end Do
            !    end Do
            !    stop


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .true.
            if (Gradientflag) then
                call ModelCalcChiFunction(NumberFreeParameterCopy, NumParameterVectors, ParameterVectorSet(:, :), ModelFunctionFlag, &
                                          ma, a, ia, colx, NumFile, MaxL, MaxCol, chi2ValuesVector)
            else
                ParameterVectorSetLocal(1, :) = ParameterVectorSet(1, :)
                call ModelCalcChiFunction(NumberFreeParameterCopy, NumParameterVectors, ParameterVectorSetLocal(:, :), ModelFunctionFlag, &
                                          ma, a, ia, colx, NumFile, MaxL, MaxCol, chi2ValuesVector)
            endif

            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< start loop for determine chi**2
            GradientHelp = 0.d0
            FitFunctionOut = 0.d0                                                           !< reset array containing the values of the fit function
            Chi2Values = 0.d0
            chisqValues = 0.d0
            counter = 0
            Do NumberFile = 1, NumberExpFiles                                               !< loop over exp. data files
                CurrentExpFile = NumberFile


                !< start loop over all lines(expdata)
                Do n = 1, NumberYColumns(NumberFile)                                        !< loop over y-columns
                    Do i = 1, lengthexpdata(NumberFile)                                     !< loop over all line of current exp. data file
                        CurrentYColumn = n
                        counter = counter + 1


                        !< get fit function
                        ymod = ModelFunction(1, NumberFile, n, i)

                        ! Debug:
                        !    print*,'>> NumberFile, n, i, counter, ymod = ', NumberFile, n, i, counter, ymod
                        !    print*,'>> ModelFunction(1, NumberFile, n, i) = ', ModelFunction(1, NumberFile, n, i)


                        !< determine sigma (sig2i) factor
                        sig2i = 1.d0
                        if (expdatae(NumberFile, i, n) /= 0.d0) then                        !< do the experimental datas include errors
                            sig2i = 1.d0 / (expdatae(NumberFile, i, n) * expdatae(NumberFile, i, n))
                        endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< get gradient
                        if (Gradientflag) then
                            dyda = 0.d0
                            Do l = 1, NumberFreeParameterCopy
                                j = ConverterInfit(l)                                       !< get appropriate parameter index within parameter set

                                ! order new: NumberFile, n, i, l,      j
                                ! order old: NumberFile, n, i, l,      j
                                ! order org: NumberFile, k, j, fitnum, i

                                ! Debug:
                                !    print*,'l, j = ', l, j
                                !    print*,'ParameterVectorSet(l + 1, l) = ', ParameterVectorSet(l + 1, l)
                                !    print*,'steph(j) = ', steph(j)
                                !    print*,'ChangeSign_Flag(j) = ', ChangeSign_Flag(j)
                                !    print*,'ModelFunction(l + 1, NumberFile, n, i) = ', ModelFunction(l + 1, NumberFile, n, i)
                                !    print*,'GradientMethod = ', GradientMethod


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine gradient for numerical recepies version
                                if (GradientMethod <= 1) then


                                    !< determine gradient of model function, see Numerical Recepies, Sect. 15.5
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        dyda(j) = (ModelFunction(1, NumberFile, n, i) - ModelFunction(l + 1, NumberFile, n, i)) / steph(j)
                                    else
                                        dyda(j) = (ModelFunction(l + 1, NumberFile, n, i) - ModelFunction(1, NumberFile, n, i)) / steph(j)
                                    endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine gradient for minpack version
                                elseif (GradientMethod == 2) then
                                    d1 = (expdatay(NumberFile, i, n) - ModelFunction(1, NumberFile, n, i))**2 * sig2i
                                    d2 = (expdatay(NumberFile, i, n) - ModelFunction(l + 1, NumberFile, n, i))**2 * sig2i
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        GradientHelp(l, NumberFile, n, i) = (d1 - d2) / steph(j)
                                    else
                                        GradientHelp(l, NumberFile, n, i) = (d2 - d1) / steph(j)
                                    endif


                                    !< copy gradient to dyda array
                                    dyda(j) = GradientHelp(l, NumberFile, n, i)
                                endif
                            end Do

                            ! Debug:
                            !    print*,'ymod = ', ymod
                            !    print*,'dyda(:) = ', dyda(:)
                            !    print*,'a = ', a
                            !    print*,'############################################################################'
                            !    stop
                        endif
                        FitFunctionOut(NumberFile, i, n) = ymod                             !< save value of fit function

                        ! Debug:
                        !    print*,' '
                        !    print*,' '
                        !    print*,'NumberExpFiles = ',NumberExpFiles
                        !    print*,'lengthexpdata(NumberFile) = ',lengthexpdata(NumberFile)
                        !    print*,'NumberYColumns(NumberFile) = ',NumberYColumns(NumberFile)
                        !    print*,'NumberFile = ',NumberFile
                        !    print*,'i = ',i
                        !    print*,'n = ',n
                        !    print*,'lenposdatexp = ',lenposdatexp
                        !    print*,'posdatexp = ',posdatexp
                        !    print*,'expdatay(NumberFile,i,n) = ',expdatay(NumberFile,i,n)
                        !    print*,'ymod = ',ymod
                        !    print*,'dyda = ',dyda(1:NumberFreeParameterCopy)
                        !    if (i==3) stop


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                        if (abs(DetChi2) == 1) then
                            dy = (expdatay(NumberFile, i, n) - ymod)                        !< define distance between fit and data
                            Chi2Values(NumberFile, i, n) = dy * dy * sig2i                  !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * dy * sig2i               !< And find chi^2.


                            !< if gradient is calculated, determine alpha and beta as well
                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j,k) = alpha(j,k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)**2
                        elseif (abs(DetChi2) == 2) then
                            dy = (expdatay(NumberFile, i, n)**2 - ymod**2)
                            Chi2Values(NumberFile, i, n) = dy * sig2i                       !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * sig2i                    !< And find chi^2.


                            !< if gradient is calculated, determine alpha and beta as well
                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j, k) = alpha(j, k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< add other chi2 methods here!!!


                        endif
                    end Do                                                                  !< loop over all line of current exp. data file
                end Do                                                                      !< loop over y-columns
            end Do                                                                          !< loop over exp. data files

            ! Debug:
            ! print*,'chisqValues(0) = ', chisqValues(0)


            !< only used for MINPACK version of Levenberg-Marquardt algorithm
            if (GradientMethod == 2) then
                Do l = 1, NumberFreeParameterCopy
                    ModelFunction(l + 1, :, :, :) = GradientHelp(l, :, :, :)
                end Do
            endif


            !< Fill in the symmetric side of alpha
            Do j = 2, NumberFreeParameterCopy
                Do k = 1, (j - 1)
                   alpha(k, j) = alpha(j, k)
                end Do

                ! Debug:
                !    print*,'j = ', j
                !    print*,'alpha(j,:) = ', alpha(j,:NumberFreeParameterCopy)
                !    print*,'beta2(j) = ', beta2(j)
            end Do


            !< writing current value of chi**2 and corresponding values of parameters to file
            if (WriteChi2Flag) then
                NumberLinesChi2 = NumberLinesChi2 + 1
                k = 0
                FormattedParmValues(:)(:) = ""
                Do i = 1, ma
                    if (ia(i)) then
                        k = k + 1
                        a(i) = fitparam(k)

                        ! Debug:
                        !   print*,'fitparam(k) = ',k,fitparam(k)


                        !< build list with fit parameters
                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                        if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                            .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) int(a(i))
                            endif
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) a(i)
                            endif
                        endif
                        FormattedParmValues(k) = trim(adjustl(HelpString))
                    endif
                end Do


                !< write line of formatted parameter values to log file
                write(Chi2Channel,'(ES25.15,$)') chisqValues(0)
                Do i = 1, NumberFreeParameterCopy
                    write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                end Do
                write(Chi2Channel,*)
            endif


            return
        end subroutine ModelCalcChiFunctionLM


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionGeneral
        !>
        !> calculates the chi2 values for several given parameter vector sets
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      a:                  array containing the parameter set
        !>                      NumberParamVectors:         number of parameter vectors
        !>                      NumberFreeParameterCopy:    number of fitted parameters
        !>                      ParameterVectorSet:
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 19.08.2014
        !>
        subroutine ModelCalcChiFunctionGeneral(ma, ia, a, NumberParamVectors, NumberFreeParameterCopy, NumFile, MaxL, MaxCol, ParameterVectorSet, &
                                               chi2ValuesVector)

            implicit none
            integer :: ma                                                                   !< total number of parameters
            integer :: NumberParamVectors                                                   !< number of parameter vectors
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: colx                                                                 !< max. number of x columns
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(NumberParamVectors, NumberFreeParameterCopy) :: ParameterVectorSet   !< set of parameter vectors calculated by model module
            real*8, dimension(NumberParamVectors) :: chi2ValuesVector                       !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            !< working variables
            integer :: i, k, l                                                              !< loop variables
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .false.
            chi2ValuesVector = 0.d0
            colx = MaxColX
            call ModelCalcChiFunction(NumberFreeParameterCopy, NumberParamVectors, ParameterVectorSet, ModelFunctionFlag, ma, a, ia, colx, NumFile, &
                                      MaxL, MaxCol, chi2ValuesVector)

            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !< writing current value of chi**2 and corresponding values of parameters to file
            Do l = 1, NumberParamVectors
                if (WriteChi2Flag) then
                    NumberLinesChi2 = NumberLinesChi2 + 1
                    k = 0
                    FormattedParmValues(:)(:) = ""

                    Do i = 1, ma
                        if (ia(i)) then
                            k = k + 1
                            a(i) = ParameterVectorSet(l, k)


                            !< build list with fit parameters
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                            if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(a(i))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) a(i)
                                endif
                            endif
                            FormattedParmValues(k) = trim(adjustl(HelpString))
                        endif
                    end Do


                    !< write line of formatted parameter values to log file
                    write(Chi2Channel,'(ES25.15,$)') chi2ValuesVector(l)
                    Do i = 1, NumberFreeParameterCopy
                        write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                    end Do
                    write(Chi2Channel,*)
                endif
            end Do
            return
        end subroutine ModelCalcChiFunctionGeneral


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunction
        !>
        !> prepare call of subroutine 'ModelCalcCore'
        !>
        !>
        !> input variables:     NumberFreeParam:        number of free parameters
        !>                      NumParamVectors:        number of parameter vectors
        !>                      ParameterVectorSet:     parameter vector
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      ma:                     total number of parameters
        !>                      a:                      array containing the parameter set
        !>                      ia:                     flags for including/excluding parameter in the fit
        !>                      colx:                   number of columns in experimental x data
        !>                      NumFile:                number of experimental files
        !>                      MaxL:                   max number of lines of all experimental files
        !>                      MaxCol:                 max number of columns of all experimental files
        !>
        !> output variables:    chi2ValuesVector:       chi2 value for parameter vector
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 15.08.2014
        !>
        subroutine ModelCalcChiFunction(NumberFreeParam, NumParamVectors, ParameterVectorSet, ModelFunctionFlag, ma, a, ia, colx, NumFile, &
                                        MaxL, MaxCol, chi2ValuesVector)

            use omp_lib

            implicit none
            integer :: NumberFile                                                           !< loop variables
            integer :: NumberFreeParam                                                      !< number of free parameters
            integer :: NumParamVectors                                                      !< number of parameter vectors
            integer :: length                                                               !< total length of AtOnceFunction
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            real*8, dimension(ma) :: a, acopy                                               !< array containing the parameter set
            real*8, dimension(NumParamVectors) :: chi2ValuesVector                          !< chi2 value for parameter vector
            real*8, dimension(NumParamVectors, NumberFreeParam) :: ParameterVectorSet       !< parameter vector set
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function values are stored or
                                                                                            !< not
            logical :: AllocateFlag                                                         !< used to avoid multiple allocation of memory
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            ! Debug:
            ! print*,'NumFile, MaxL, MaxCol = ', NumFile, MaxL, MaxCol
            ! print*,'ParallezitionMethod = ', ParallezitionMethod


            !< reset output variables
            chi2ValuesVector = 0.d0


            !< determine total length of AtOnceFunction
            length = 0
            Do NumberFile = 1, NumberExpFiles
                length = length + (NumberYColumns(NumberFile) * lengthexpdata(NumberFile))
            end Do


            !< make a copy of a
            acopy = a


            !< make a copy of NumParamVectors
            TotalNumParamVectors = NumParamVectors


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< (de)allocate memory for array containing all parameter vectors and chi2 vector
            !< the following lines are included to aviod reallocation for each call of this subroutine
            AllocateFlag = .false.
            if (.not. allocated(GlobalParamVectorSet)) then
                AllocateFlag = .true.
            else
                if (size(GlobalParamVectorSet) /= (NumParamVectors * NumberFreeParameter)) then
                    AllocateFlag = .true.
                endif
            endif

            ! Debug:
            ! print*,'>>>>>>>>>>>>>>>>>>myrank, AllocateFlag, TotalNumParamVectors = ', myrank ,AllocateFlag, TotalNumParamVectors


            if (AllocateFlag) then
                if (allocated(GlobalParamVectorSet)) then
                    deallocate(GlobalParamVectorSet, LocalChi2ValueVector, stat = dealloc_status)
                    if (dealloc_status /= 0) then                                           !< is all ok?
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                        write(logchannel,'(2x,"Can not deallocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine ModelCalcChiFunction:")'
                        print '(2x,"Can not deallocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("dealloc_status = ",I4)', dealloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                endif
                allocate(GlobalParamVectorSet(NumParamVectors, NumberFreeParameter), LocalChi2ValueVector(NumParamVectors), stat = alloc_status)
                if (alloc_status /= 0) then                                                 !< is all ok?
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                    write(logchannel,'(2x,"Can not allocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("alloc_status = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine ModelCalcChiFunction:")'
                    print '(2x,"Can not allocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("alloc_status = ",I4)', alloc_status
                    print '(" ")'
                    ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                    call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                    stop
                endif
            endif
            GlobalParamVectorSet(:, :) = ParameterVectorSet(:, :)
            LocalChi2ValueVector = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< weak up slave threads and compute model function
            call SlaveWork(3)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine chi2 value and model function values for different parameter vectors
            call ModelCalcCore(NumberFreeParam, TotalNumberDataPoints, ModelFunctionFlag, ma, colx, NumFile, MaxL, MaxCol)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< sleep little slave threads
            call SlaveWork(4)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< get result of model function calculation
            chi2ValuesVector(:) = LocalChi2ValueVector(:)

            ! Debug:
            !    print*,'chi2ValuesVector = ', chi2ValuesVector
            !    ! call MPI_Abort(MPI_COMM_WORLD, ierr)
            !    call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
            !    stop
            return
        end subroutine ModelCalcChiFunction


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcCore
        !>
        !> calculates the chi2 values for a given set of parameter vectors
        !>
        !>
        !> input variables:     NumberFreeParam:        number of free parameters
        !>                      TotalNumDataPts:        total number of data points
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      ma:                     total number of parameters
        !>                      a:                      array containing the parameter set
        !>                      ia:                     flags for including/excluding parameter in the fit
        !>                      colx:                   number of columns in experimental x data
        !>                      NumFile:                number of experimental files
        !>                      MaxL:                   max number of lines of all experimental files
        !>                      MaxCol:                 max number of columns of all experimental files
        !>
        !> output variables:    chi2Value:              chi2 value
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine ModelCalcCore(NumberFreeParam, TotalNumDataPts, ModelFunctionFlag, ma, colx, NumFile, MaxL, MaxCol)

            implicit none
            integer :: m, l, j, n, i, NumberFile                                            !< loop variables
            integer :: k                                                                    !< counter variable
            integer :: ParamIndex                                                           !< working variable: index of parameter vector
            integer :: NumberFreeParam                                                      !< number of free parameters
            integer :: nelements                                                            !< working variable: size of an array to send
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: CurrentNumberWorkers                                                 !< the number of workers
            integer :: LastWorker                                                           !< number of last worker
            integer :: NumWorkers                                                           !< total number of workers
            integer :: TotalNumDataPts                                                      !< total number of data points
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            real*8 :: chi2Value                                                             !< working variable for chi2 value
            real*8 :: sig2i, ymod, val                                                      !< working variables
            real*8, dimension(NumberFreeParam) :: LocalParameterVector                      !< local copy of part of ParameterVectorSet
            real*8, dimension(NumberFreeParam) :: LocalParameterVector2                     !< further local copy of part of ParameterVectorSet
            real*8, dimension(TotalNumDataPts) :: ModelFunctionListLocal                    !< model function values
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function values are stored or
                                                                                            !< not
            logical :: NoCalFlag                                                            !< flag for not calculating model function for a specific
                                                                                            !< parameter vector
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit
            logical :: OutOfRangeFlag                                                       !< indicates, if a parameter is out of allowed range
            logical :: StoredBefore_Flag                                                    !< flag indicating if parameter vector is stored before or not
            logical :: AllocateFlag                                                         !< used to avoid multiple allocation of memory
            logical :: LocalModelFunctionFlag                                               !< in contrast to the GPU version we can store the model
                                                                                            !< function for all parameter vectors


            !< broadcast some variables between master and slave threads
            call broadcast_scalar_log(ModelFunctionFlag, "ModelFunctionFlag")
            ! call MPI_Bcast(ModelFunctionFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)

            call broadcast_scalar_int(TotalNumParamVectors, "TotalNumParamVectors")
            ! call MPI_Bcast(TotalNumParamVectors, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)


            !< (de)allocate memory for array containing all parameter vectors and chi2 vector
            !< the following lines are included to aviod reallocation for each call of this subroutine
            if (myrank /= root_tn) then

                ! Debug:
                ! print*,'-------------------myrank = ', myrank


                AllocateFlag = .false.
                if (.not. allocated(GlobalParamVectorSet)) then
                    AllocateFlag = .true.
                else
                    if (size(GlobalParamVectorSet) /= (TotalNumParamVectors * NumberFreeParameter)) then
                        AllocateFlag = .true.
                    endif
                endif

                ! Debug:
                ! print*,'-------------------myrank, AllocateFlag, TotalNumParamVectors = ', myrank ,AllocateFlag, TotalNumParamVectors


                if (AllocateFlag) then
                    if (allocated(GlobalParamVectorSet)) then
                        deallocate(GlobalParamVectorSet, stat = dealloc_status)
                        if (dealloc_status /= 0) then                                           !< is all ok?
                            write(logchannel,*)
                            write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                            write(logchannel,'(2x,"Can not deallocate variable GlobalParamVectorSet for thread = ", I5, ".")') &
                                                                                                                                                    myrank
                            write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                            write(logchannel,*)
                            write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                            write(logchannel,'(" ")')
                            write(logchannel,'("Program aborted!")')
                            close(logchannel)

                            print '(" ")'
                            print '("Error in subroutine ModelCalcChiFunction:")'
                            print '(2x,"Can not deallocate variable GlobalParamVectorSet for thread = ", I5, ".")', myrank
                            print '(2x,"Please close all other programs and restart the program!")'
                            print '(" ")'
                            print '("dealloc_status = ",I4)',dealloc_status
                            print '(" ")'
                            ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                            call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                            stop
                        endif
                    endif
                    allocate(GlobalParamVectorSet(TotalNumParamVectors, NumberFreeParameter), stat = alloc_status)
                    if (alloc_status /= 0) then                                                 !< is all ok?
                        write(logchannel,'(" ")')
                        write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                        write(logchannel,'(2x,"Can not allocate variable GlobalParamVectorSet for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,'(" ")')
                        write(logchannel,'("alloc_status = ",I4)') alloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine ModelCalcChiFunction:")'
                        print '(2x,"Can not allocate variable GlobalParamVectorSet for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("alloc_status = ",I4)',alloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                endif
                GlobalParamVectorSet = 0.d0
            endif


            !< broadcast some variables between master and slave threads
            call broadcast_array_dbl_2d(GlobalParamVectorSet, 1, TotalNumParamVectors, 1, NumberFreeParameter, "GlobalParamVectorSet")
            ! nelements = size(GlobalParamVectorSet)
            ! call MPI_Bcast(GlobalParamVectorSet, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

            ! Debug:
            !    print*,'myrank, ModelFunctionFlag = ', myrank, ModelFunctionFlag
            !    print*,'myrank, colx = ', myrank, colx
            !    print*,'myrank, NumFile = ', myrank, NumFile
            !    print*,'myrank, MaxL = ', myrank, MaxL
            !    print*,'myrank, MaxCol = ', myrank, MaxCol
            !    print*,'myrank, TotalNumParamVectors = ', myrank, TotalNumParamVectors
            !    print*,'myrank, GlobalParamVectorSet(myrank + 1, :) = ', myrank, GlobalParamVectorSet(myrank + 1, :)
            !    print*,'>>>myrank, paramset(1, :) = ', myrank, paramset(1, :)


            !< reset output variables
            if (myrank == root_tn) then
                LocalChi2ValueVector = 0.d0
            endif
            ModelFunction = 0.d0


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            Do l = 1, TotalNumParamVectors, TotalNumberOfThreads                                 !< loop over all parameter vectors

                ! Debug:
                ! print*,'l + myrank, TotalNumParamVectors, TotalNumberOfThreads = ', l + myrank, TotalNumParamVectors, TotalNumberOfThreads


                !< determine max. number of workers
                LastWorker = min(TotalNumParamVectors, (l + TotalNumberOfThreads - 1))
                NumWorkers = LastWorker - l + 1
                CurrentNumberWorkers = NumWorkers

                ! Debug:
                ! print*,'myrank, CurrentNumberWorkers = ', myrank, CurrentNumberWorkers


                !< do nothing if no more parameter vector is left
                if (l + myrank > TotalNumParamVectors) then
                    exit
                endif


                !< select a parameter vector for each thread (master and slave)
                LocalParameterVector(:) = GlobalParamVectorSet(l + myrank, :)


                !< initialize result variables
                chi2Value = 1.d99
                ModelFunctionListLocal = 0.d0


                !< is calculation reduction wanted (to be done soon ........... !)
                NoCalFlag = .false.


                !< is a calculation still neccessary
                if (.not. NoCalFlag) then


                    !< make copy of current parameter vector
                    OutOfRangeFlag = .false.
                    k = 0
                    ia = .false.
                    Do i = 1, ma
                        if (paramset(2, i) == 1.d0) then
                            ia(i) = .true.
                            k = k + 1
                            if (LocalParameterVector(k) < paramset(3, i) .or. paramset(4, i) < LocalParameterVector(k)) then
                                OutOfRangeFlag = .true.
                                exit
                            endif
                        endif
                    end Do

                    ! Debug:
                    ! print*,'myrank, LocalParameterVector = ', myrank, LocalParameterVector
                    ! print*,'myrank, paramset(1, :) = ', myrank, paramset(1, :)
                    ! print*,'myrank, NumberFreeParam = ', myrank, NumberFreeParam
                    ! print*,'myrank, OutOfRangeFlag = ', myrank, OutOfRangeFlag



                    !-------------------------------------------------------------------------------------------------------------------------------------
                    !< calculate model function for current parameter vector
                    if (.not. OutOfRangeFlag) then


                        !< call subroutine to calculate model function
                        LocalModelFunctionFlag = .true.
                        call ModelCalcSpectrum(NumberFreeParam, LocalParameterVector, LocalModelFunctionFlag, TotalNumberComponents, chi2Value, &
                                               CompMoleculeIndex, myXCLASSParameter, TotalNumberOfMolecules, IsoRatioConversionTable, &
                                               TotalNumberDataPoints, ModelFunctionListLocal, myrank)

                        ! Debug:
                        ! print*,'myrank, ModelFunctionListLocal(100) = ', myrank, ModelFunctionListLocal(100)
                    endif
                endif


                !< send results back to master thread
                if (myrank /= root_tn) then

                    ! Debug:
                    ! print*,'SLAVE: myrank, chi2Value = ', myrank, chi2Value


                    !< send some other parameters back to master thread
                    call MPI_Send(chi2Value, 1, MPI_DOUBLE_PRECISION, 0, 0, MPI_COMM_WORLD, ierr)
                    nelements = size(ModelFunctionListLocal)
                    call MPI_Send(ModelFunctionListLocal, nelements, MPI_DOUBLE_PRECISION, 0, 0, MPI_COMM_WORLD, ierr)


                !< MASTER: recieve results from slave threads
                else
                    Do j = 0, (CurrentNumberWorkers - 1)
                        ParamIndex = l + j


                        !< get chi2 value
                        if (j > 0) then
                            call MPI_Recv(chi2Value, 1, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, istatus0, ierr)

                            ! Debug:
                            ! print*,'MPI_Recv(chi2Value): myrank, ierr = ', myrank, ierr
                        endif
                        LocalChi2ValueVector(ParamIndex) = chi2Value

                        ! Debug:
                        ! print*,'myrank, j, chi2Value = ', myrank, j, chi2Value


                        !< get model function values
                        if (j > 0) then
                            nelements = size(ModelFunctionListLocal)
                            call MPI_Recv(ModelFunctionListLocal, nelements, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, istatus0, ierr)

                            ! Debug:
                            ! print*,'MPI_Recv(ModelFunctionListLocal): myrank, ierr = ', myrank, ierr
                        endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< do we need to determine the best fit parameter vector
                        if (ModelFunctionFlag) then
                            k = 0
                            Do NumberFile = 1, NumberExpFiles                               !< loop over exp. data files
                                Do n = 1, NumberYColumns(NumberFile)                        !< loop over y-columns
                                    Do i = 1, lengthexpdata(NumberFile)                     !< loop over all line of current exp. data file
                                        k = k + 1                                           !< increase data point counter
                                        ModelFunction(ParamIndex, NumberFile, n, i) = ModelFunctionListLocal(k)
                                    end Do
                                end Do
                            end Do
                        else
                            LocalParameterVector(:) = GlobalParamVectorSet(ParamIndex, :)
                            Do k = 1, QualityLimit
                                if (chi2Value < BestSitesParamSet(k, 1)) then
                                    StoredBefore_Flag = .false.
                                    if (QualityLimit > 1) then
                                        Do m = 1, QualityLimit
                                            if (chi2Value == BestSitesParamSet(m, 1)) then
                                                LocalParameterVector2(:) = BestSitesParamSet(m, 2:)
                                                call CompareTwoParameterSets(StoredBefore_Flag, ma, NumberFreeParameter, ia, LocalParameterVector2(:), &
                                                                             LocalParameterVector(:))
                                                if (StoredBefore_Flag) then
                                                    exit
                                                endif
                                            endif
                                        end Do
                                    endif
                                    if (.not. StoredBefore_Flag) then
                                        if (k < QualityLimit) then
                                            Do m = QualityLimit, (k + 1), (-1)
                                                BestSitesParamSet(m, :) = BestSitesParamSet((m - 1), :)
                                                BestSitesModelValues(m, :, :, :) = BestSitesModelValues((m - 1), :, :, :)
                                                BestSitesChi2Values(m, :, :, :) = BestSitesChi2Values((m - 1), :, :, :)
                                            end Do
                                        endif
                                        BestSitesParamSet(k, 1) = chi2Value
                                        BestSitesParamSet(k, 2:) = LocalParameterVector(:)
                                        m = 0
                                        Do NumberFile = 1, NumberExpFiles                   !< loop over exp. data files
                                            Do n = 1, NumberYColumns(NumberFile)            !< loop over y-columns
                                                Do i = 1, lengthexpdata(NumberFile)         !< loop over all line of current exp. data file
                                                    m = m + 1                               !< increase data point counter
                                                    ymod = ModelFunctionListLocal(m)
                                                    sig2i = 1.d0
                                                    if (expdatae(NumberFile, i, n) /= 0.d0) then    !< do the experimental datas include errors
                                                        sig2i = 1.d0/(expdatae(NumberFile, i, n) * expdatae(NumberFile, i, n))
                                                    endif


                                                    !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                                                    val = 0.d0
                                                    if (abs(DetChi2) == 1) then
                                                        val = (expdatay(NumberFile, i, n) - ymod)**2


                                                    !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)
                                                    elseif (abs(DetChi2) == 2) then
                                                        val = (expdatay(NumberFile, i, n)**2 - ymod**2)
                                                    endif
                                                    BestSitesChi2Values(k, NumberFile, i, n) = val * sig2i
                                                    BestSitesModelValues(k, NumberFile, i, n) = ymod
                                                    ModelFunction(1, NumberFile, n, i) = ymod
                                                end Do
                                            end Do
                                        end Do
                                    endif
                                endif
                            end Do
                        endif
                    end Do
                endif

                ! Debug:
                ! print*,'myrank = ', myrank
                ! if (myrank == root_tn) print*,'#######################################################################################'
                ! print '(".", $)'
                ! ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                ! call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                ! stop 'lllklklkklkl'

            end Do
            return
        end subroutine ModelCalcCore


        !>************************************************************************************************************************************************
        !> subroutine: ModelParamFree
        !>
        !> free memory used by variables of the Module Model
        !>
        !>
        !> input variables:     deallocstatus           status of the previous deallocation process
        !>
        !> output variables:    deallocstatus           status of the deallocation process
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 26.08.2014
        !>
        subroutine ModelParamFree(deallocstatus)

            implicit none
            integer :: deallocstatus                                                        !< status of the deallocation process


            !< free memory of slave threads and kill slave threads
            if (myrank == root_tn) then
                call SlaveWork(1)
            endif


            !< deallocate memory of myXCLASS variables
            call myXCLASSParamFree(deallocstatus)
            return
        end subroutine ModelParamFree
end Module Model
!*********************************************************************************************************************************************************
