!*********************************************************************************************************************************************************
!>  Module: Model
!>
!>
!>  This module contains subroutines for the default MAGIX version where no external model program is included
!>  Copyright (C) 2009 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine ModelInit:                     initializes myXCLASS program
!>      - subroutine ModelCalcChiFunctionLM:        calculates the chi2 values for the Levenberg-Marquard algorithm
!>      - subroutine ModelCalcChiFunctionGeneral:   calculates the chi2 values for several given parameter vector sets
!>      - subroutine ModelCalcChiFunction:          calculates the chi2 values for a given set of parameter vectors
!>      - subroutine ModelParamFree:                free memory used by variables of the Module Model
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    2014-08-26  Initial version
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module Model
    !> declare variables which are used by functions and subroutines of this module

    use Variables
    use FunctionCalling

    implicit none
    character(len=3) :: ParallelizationMethod                                               !< string describing the parallelization method

    contains


        !>************************************************************************************************************************************************
        !> subroutine: ModelInit
        !>
        !> initialize default module
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 26.08.2014
        !>
        subroutine ModelInit

            implicit none


            !< initialize some variables
            DontStoreModelFuncValuesFlag = .false.                                          !< model function values are stored for each function call
            ParallezitionMethod = 1                                                         !< set parallelization method to OpenMP (=1)
            myrankOfMPI = 0                                                                 !< copy thread number to global variable
            NumberOfUsedThreads = ParallelizationFlag                                       !< in Module 'FunctionCalling' we need only all thread
            ParallelizationMethod = "SMP"


            !< set number of threads
            if (ParallelizationFlag /= 0) then                                              !< call openMP commands for more than 1 processors
                call omp_set_num_threads(ParallelizationFlag)                               !< set number of threads

                 ! Debug:
                 ! print*,'ParallelizationFlag = ', ParallelizationFlag
            endif


            return
        end subroutine ModelInit


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionLM
        !>
        !> calculates the chi2 values for the Levenberg-Marquard algorithm
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      a:                  array containing the parameter set
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      NumberFreeParameterCopy:             number of fitted parameters
        !>                      fitparam:           parameter which have to be optimized
        !>                      colx:               number of columns in experimental x data
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>                      alpha:              matrix alpha (only used for Levenberg-Marquardt algorithm)
        !>                      beta2:              beta2 array (only used for Levenberg-Marquardt algorithm)
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 31.07.2014
        !>
        subroutine ModelCalcChiFunctionLM(ma, a, ia, NumberFreeParameterCopy, fitparam, colx, NumFile, MaxL, MaxCol, FitFunctionOut, Chi2Values, &
                                          alpha, beta2)

            use omp_lib

            implicit none
            integer :: i, j, k, l, m, n, fitnum, NumberFile                                 !< loop variables
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: counter                                                              !< counter for ModelFunction
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            integer :: NumParameterVectors                                                  !< number of parameter vectors in ParameterVectorSet
            real*8 :: dy, sig2i, wt, ymod                                                   !< working variables
            real*8 :: value                                                                 !< calculated value of the fit function
            real*8 :: variation, d1, d2                                                     !< working variables
            real*8, dimension(ma) :: steph                                                  !< working variable
            real*8, dimension(ma) :: modelparam                                             !< array containing the parameter set
            real*8, dimension(ma) :: modelparamcopy                                         !< copy of modelparam
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(ma, ma) :: alpha                                              !< matrix alpha
            real*8, dimension(NumberFreeParameterCopy) :: beta2                             !< beta2 array
            real*8, dimension(NumberFreeParameterCopy) :: fitparam                          !< parameter which have to be optimized
            real*8, dimension(NumFile, MaxL, MaxCol) :: FitFunctionOut                      !< values of the fit function at all calculated points
            real*8, dimension(NumberFreeParameterCopy, NumFile, MaxCol, MaxL) :: GradientHelp
            real*8, dimension(NumFile, MaxL, MaxCol) :: Chi2Values                          !< values of the fit function at all calculated points
            real*8, dimension(ma) :: dyda                                                   !< gradient of the fit function
            real*8, dimension(NumberFreeParameterCopy + 1) :: chi2ValuesVector              !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            real*8, dimension(NumberFreeParameterCopy + 1, NumberFreeParameterCopy) :: ParameterVectorSet
                                                                                            !< set of parameter vectors calculated by model module
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file


            character(len=50) :: valueh, valuel                                             !< working variable for determine gradient
            logical :: equal_flag                                                           !< required for string comparison
            logical, dimension(ma) :: ChangeSign_Flag                                       !< change sign flag
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize values for fit function calculation
            Do j = 1, NumberFreeParameterCopy                                               !< Initialize (symmetric) alpha, beta2.
                Do k = 1, j
                    alpha(j,k) = 0.d0                                                       !< clear alpha partially
                end Do
                beta2(j) = 0.d0                                                             !< clear beta2
            end Do


            !< update the parameter set with the current values of the fit parameters some algorithms optimize only the parameters in the array fitparam
            !< all other parameter in the parameter set a are not included in these algorithms but are required for the calculation of the fit function
            k = 0
            Do i = 1, ma
                if (ia(i)) then
                    k = k + 1
                    a(i) = fitparam(k)
                endif
            end Do
            ParameterVectorSet = 0.d0
            ParameterVectorSet(1, :) = fitparam(:)                                          !< first entry of ParameterVectorSet contains the current
                                                                                            !< parameter vector
            ! Debug:
            ! print*,'>>>>>>>>>>>>>fitparam = ', fitparam(:)


            !< check, if gradient is necessary (for Levenberg-Marquardt it is always necessary!)
            NumParameterVectors = 1
            if (Gradientflag) then
                NumParameterVectors = NumberFreeParameterCopy + 1


                !< initialize some working parameter
                steph = 0.d0
                ChangeSign_Flag = .false.
                modelparamcopy = a                                                          !< dublicate array containing the parameter set


                !< calculate parameter vector for gradient calculation

                !$omp parallel default(shared) &
                !$omp shared(OutputFileFormat, NumberRanges, ValueEmptyOutputFile, LSRAdjustement, FirstPointExpData, LastPointExpData, idum) &
                !$omp shared(MinRange, MaxRange, NaNReplaceString, StandardWorkingDirectory, CharacterForComments, CharacterSeperatingColumns) &
                !$omp shared(ResamplingMethod, InterpolationMethod, RenormalizedChi2, OnlyYColumn, LSRAdjustementFitFlag, NormalizationFlag) &
                !$omp shared(AtOnceGradient, ia, paramset, modelparamcopy, ConverterInfit) &
                !$omp shared(ParameterName, ParameterFormat, LeadingString, TrailingString, ParamVisible, NumberLinesOutput) &
                !$omp shared(printflag, chisqValues, NumberExpFiles, modelnumber, lengthexpdata, currentpath, NumberHeaderLines, QualityLimit) &
                !$omp shared(FitParameterName, FitParameterValue, NumberColumnsBegin, NumberColumnsEnd, NumberParamPerLine, CalculationMethod) &
                !$omp shared(CommandWordOutput, DetChi2, MaxParameter, ExternalThreadNumber, expdatax, expdatay, expdatae, TempDirectory) &
                !$omp shared(NumberXColumns, NumberYColumns, MaxColX, MaxColY, parameternumber, NumberInputFiles, ParallelizationFlag, JobID) &
                !$omp shared(MaxInputLines, UseCalculationReduction, WriteChi2Flag, Gradientflag,  initflag, CalculatedParameterSets) &
                !$omp shared(LastAlgorithmFlag, ChangeSign_Flag, steph, NumberFreeParameterCopy, ParameterVectorSet, fitparam) &
                !$omp shared(CurrentNumberLinesCalcReduction, ochisq, NumberLinesChi2, InputDataPath, FitFktInput, ModelFunction, FitFktOutput) &
                !$omp shared(ExeCommandStartScript, NumberOutputFiles, CurrentExpFile, GradientMethod, PathStartScript, CurrentYColumn) &
                !$omp shared(ExpData_reversed_flag, BestSitesModelValues, BestSitesChi2Values, BestSitesParamSet, GradientVariationValue) &
                !$omp private(fitnum, i, j, k, modelparam, d1, d2, sig2i, NumberFile, variation, value, valueh, valuel) &
                !$omp private(IntegerTrue, NumInputFile_index, i_index, j_index, equal_flag)
                !$omp do

                Do fitnum = 1, NumberFreeParameterCopy                                      !< loop over all free parameter
                    !$omp critical
                    ParameterVectorSet(fitnum + 1, :) = fitparam(:)
                    !$omp end critical
                    modelparam = modelparamcopy                                             !< load unmodified parameter values
                    i = ConverterInfit(fitnum)                                              !< get appropriate parameter index within parameter set
                    call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)

                    ! Debug:
                    ! print*,'fitnum, NumInputFile_index, i_index, j_index = ', fitnum, NumInputFile_index, i_index, j_index


                    !< check if parameter is within parameter limits
                    if (modelparamcopy(i) < paramset(3, i) .or. modelparamcopy(i) > paramset(4, i)) then
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        write(logchannel,'(2x,"The parameter ",A," is out of limits.")') &
                                          trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        write(logchannel,'(2x,"Upper limit = ", ES25.15)') paramset(4, i)
                        write(logchannel,'(2x,"Lower limit = ", ES25.15)') paramset(3, i)
                        write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        print '(13x,"The parameter ",A," is out of limits.")', trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        print '(13x,"Upper limit = ", ES25.15)', paramset(4, i)
                        print '(13x,"Lower limit = ", ES25.15)', paramset(3, i)
                        print '(13x,"Value of parameter = ", ES25.15)',modelparamcopy(i)
                        print '(" ")'
                        print '(13x,"Program aborted!")'
                        stop
                    endif


                    !< determine strength of variation
                    variation = GradientVariationValue                                      !< variation of the parameter in percent/100
                    !$omp critical
                    steph(i) = dabs(modelparamcopy(i) * variation)                          !< define stepsize for foating point numbers
                    if (modelparamcopy(i) == 0.d0) then
                        if (IntegerTrue) then                                               !< is parameter an integer ??
                            steph(i) = 1.d0                                                 !< variation of the parameter in percent/100
                        else
                            steph(i) = variation                                            !< variation of the parameter in percent/100
                        endif
                    elseif (IntegerTrue .and. steph(i) < 1.d0) then
                        steph(i) = 1.d0
                    endif
                    !$omp end critical

                    ! Debug:
                    ! print*,'i = ', i
                    ! print*,'modelparamcopy(i) = ', modelparamcopy(i)
                    ! print*,'steph(i) = ',steph(i)
                    ! print*,'modelparamcopy(i) + steph(i) = ', modelparamcopy(i) + steph(i)
                    ! print*,'paramset(3, i) = ', paramset(3, i)
                    ! print*,'paramset(4, i) = ', paramset(4, i)
                    ! print*,'modelparamcopy(i) - steph(i) = ', modelparamcopy(i) - steph(i)


                    !< test, if we can accelerate the calculation
                    value = modelparamcopy(i) + steph(i)
                    if (value < paramset(3, i) .or. paramset(4, i) < value) then
                        if (value > paramset(4, i)) then                                    !< is f(x_i + h) > upper limit ?
                            value = modelparamcopy(i) - steph(i)                            !< use f(x_i - h)
                            if (value < paramset(3, i)) then
                                if (dabs(modelparamcopy(i) - paramset(4, i)) < 1.d-12) then
                                    write(logchannel,*)
                                    write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                                    write(logchannel,'(2x,"The gradient for parameter ",A," cannot be calculated.")') &
                                                      trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    write(logchannel,'(2x,"The variation runs out of limits.")')
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")')
                                    write(logchannel,*)
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Upper limit = ",ES25.15)') paramset(4, i)
                                    write(logchannel,'(2x,"Lower limit = ",ES25.15)') paramset(3, i)
                                    write(logchannel,'(2x,"value of variation = ", ES25.15)') variation
                                    write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                                    write(logchannel,'(" ")')
                                    write(logchannel,'("Program aborted!")')

                                    print '(" ")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '("Error in subroutine ModelCalcChiFunctionLM:")'
                                    print '(2x,"The gradient for parameter ",A," cannot be calculated.")', &
                                                      trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    print '(2x,"The variation runs out of limits.")'
                                    print '(" ")'
                                    print '(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '(2x,"Upper limit = ", ES25.15)', paramset(4, i)
                                    print '(2x,"Lower limit = ", ES25.15)', paramset(3, i)
                                    print '(2x,"value of variation = ", ES25.15)', variation
                                    print '(2x,"Value of parameter = ", ES25.15)', modelparamcopy(i)
                                    print '(" ")'
                                    print '("Program aborted!")'
                                    stop
                                endif
                                value = paramset(4, i)
                            else
                                !$omp critical
                                ChangeSign_Flag(i) = .true.
                                !$omp end critical
                            endif
                        endif
                    endif

                    ! Debug:
                    ! print*,'> value, modelparamcopy(i), steph(i) = ',value, modelparamcopy(i), steph(i)


                    !< check, if the variation leads in combination with the Format to a variation in the current parameter
                    if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                        .or. index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                        write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) int(value)
                        write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) int(modelparamcopy(i))
                    else
                        write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) value
                        write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) modelparamcopy(i)
                    endif
                    equal_flag = .true.
                    valueh = adjustl(valueh)
                    valuel = adjustl(valuel)
                    if (len_trim(valueh) == len_trim(valuel)) then
                        Do j = 1, len_trim(valueh)
                            if (valueh(j:j) /= valuel(j:j)) then
                                equal_flag = .false.
                                exit
                            endif
                        end Do
                    else
                        equal_flag = .false.
                    endif
                    if (equal_flag) then                                                    !< both expressions are equal
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        write(logchannel,'(2x,"The format specification of the parameter ",A)') &
                                trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        write(logchannel,'(2x,"prevents the variation of the current parameter.")')
                        write(logchannel,'(2x," ")')
                        write(logchannel,'(2x,"The gradient entry for this parameter is set to zero. Therefore")')
                        write(logchannel,'(2x,"no variation of this parameter in the current iteration is done")')
                        write(logchannel,'(" ")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        print '(13x,"The format specification of the parameter ",A)', &
                               trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        print '(13x,"prevents the variation of the current parameter.")'
                        print '(" ")'
                        print '(13x,"The gradient entry for this parameter is set to zero. Therefore")'
                        print '(13x,"no variation of this parameter in the current iteration is done")'
                        print '(" ")'
                    endif


                    !< modify the ith parameter
                    !$omp critical
                    modelparam(i) = value                                                   !< modify value of the ith parameter
                    ParameterVectorSet(fitnum + 1, fitnum) = value
                    !$omp end critical

                end Do

                !$omp end do
                !$omp end parallel                                                          !< end of parallel environment

                modelparam = modelparamcopy                                                 !< restore old paramter values
            endif

            ! Debug:
            !    Do l = 2, NumParameterVectors
            !        Do k = 1, NumberFreeParameterCopy
            !            if (dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k)) > 1.d-6) then
            !                print*,'l, k, ParameterVectorSet(l, k) = ', l, k, ParameterVectorSet(l, k), ParameterVectorSet(1, k), &
            !                                                            dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k))
            !            endif
            !        end Do
            !    end Do
            !    stop
            !    Do l = 1, NumParameterVectors
            !        print*,'>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>l, ParameterVectorSet(l, :) = ', l, ParameterVectorSet(l, :)
            !    end Do



            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .true.
            l = NumberFreeParameterCopy + 1
            chi2ValuesVector = 0.d0
            call ModelCalcChiFunction(NumberFreeParameterCopy, l, ParameterVectorSet, ModelFunctionFlag, ma, a, ia, colx, NumFile, MaxL, MaxCol, &
                                      chi2ValuesVector)

            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< start loop for determine chi**2
            GradientHelp = 0.d0
            FitFunctionOut = 0.d0                                                           !< reset array containing the values of the fit function
            Chi2Values = 0.d0
            chisqValues = 0.d0
            counter = 0
            Do NumberFile = 1, NumberExpFiles                                               !< loop over exp. data files
                CurrentExpFile = NumberFile


                !< start loop over all lines(expdata)
                Do n = 1, NumberYColumns(NumberFile)                                        !< loop over y-columns
                    Do i = 1, lengthexpdata(NumberFile)                                     !< loop over all line of current exp. data file
                        CurrentYColumn = n
                        counter = counter + 1


                        !< get fit function
                        ymod = ModelFunction(1, NumberFile, n, i)

                        ! Debug:
                        !    print*,'>> NumberFile, n, i, counter, ymod = ', NumberFile, n, i, counter, ymod
                        !    print*,'>> ModelFunction(1, NumberFile, n, i) = ', ModelFunction(1, NumberFile, n, i)


                        !< determine sigma (sig2i) factor
                        sig2i = 1.d0
                        if (expdatae(NumberFile, i, n) /= 0.d0) then                        !< do the experimental datas include errors
                            sig2i = 1.d0 / (expdatae(NumberFile, i, n) * expdatae(NumberFile, i, n))
                        endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< get gradient
                        if (Gradientflag) then
                            dyda = 0.d0
                            Do l = 1, NumberFreeParameterCopy
                                j = ConverterInfit(l)                                       !< get appropriate parameter index within parameter set

                                ! order new: NumberFile, n, i, l,      j
                                ! order old: NumberFile, n, i, l,      j
                                ! order org: NumberFile, k, j, fitnum, i

                                ! Debug:
                                !    print*,'l, j = ', l, j
                                !    print*,'ParameterVectorSet(l + 1, l) = ', ParameterVectorSet(l + 1, l)
                                !    print*,'steph(j) = ', steph(j)
                                !    print*,'ChangeSign_Flag(j) = ', ChangeSign_Flag(j)
                                !    print*,'ModelFunction(l + 1, NumberFile, n, i) = ', ModelFunction(l + 1, NumberFile, n, i)
                                !    print*,'GradientMethod = ', GradientMethod


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine gradient for numerical recepies version
                                if (GradientMethod <= 1) then


                                    !< determine gradient of model function, see Numerical Recepies, Sect. 15.5
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        dyda(j) = (ModelFunction(1, NumberFile, n, i) - ModelFunction(l + 1, NumberFile, n, i)) / steph(j)
                                    else
                                        dyda(j) = (ModelFunction(l + 1, NumberFile, n, i) - ModelFunction(1, NumberFile, n, i)) / steph(j)
                                    endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine gradient for minpack version
                                elseif (GradientMethod == 2) then
                                    d1 = (expdatay(NumberFile, i, n) - ModelFunction(1, NumberFile, n, i))**2 * sig2i
                                    d2 = (expdatay(NumberFile, i, n) - ModelFunction(l + 1, NumberFile, n, i))**2 * sig2i
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        GradientHelp(l, NumberFile, n, i) = (d1 - d2) / steph(j)
                                    else
                                        GradientHelp(l, NumberFile, n, i) = (d2 - d1) / steph(j)
                                    endif


                                    !< copy gradient to dyda array
                                    dyda(j) = GradientHelp(l, NumberFile, n, i)
                                endif
                            end Do

                            ! Debug:
                            !    print*,'ymod = ', ymod
                            !    print*,'dyda(:) = ', dyda(:)
                            !    print*,'a = ', a
                            !    print*,'############################################################################'
                            !    stop
                        endif
                        FitFunctionOut(NumberFile, i, n) = ymod                             !< save value of fit function

                        ! Debug:
                        !    print*,' '
                        !    print*,' '
                        !    print*,'NumberExpFiles = ',NumberExpFiles
                        !    print*,'lengthexpdata(NumberFile) = ',lengthexpdata(NumberFile)
                        !    print*,'NumberYColumns(NumberFile) = ',NumberYColumns(NumberFile)
                        !    print*,'NumberFile = ',NumberFile
                        !    print*,'i = ',i
                        !    print*,'n = ',n
                        !    print*,'lenposdatexp = ',lenposdatexp
                        !    print*,'posdatexp = ',posdatexp
                        !    print*,'expdatay(NumberFile,i,n) = ',expdatay(NumberFile,i,n)
                        !    print*,'ymod = ',ymod
                        !    print*,'dyda = ',dyda(1:NumberFreeParameterCopy)
                        !    if (i==3) stop


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                        if (abs(DetChi2) == 1) then
                            dy = (expdatay(NumberFile, i, n) - ymod)                        !< define distance between fit and data
                            Chi2Values(NumberFile, i, n) = dy * dy * sig2i                  !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * dy * sig2i               !< And find chi^2.


                            !< if gradient is calculated, determine alpha and beta as well
                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j,k) = alpha(j,k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)**2
                        elseif (abs(DetChi2) == 2) then
                            dy = (expdatay(NumberFile, i, n)**2 - ymod**2)
                            Chi2Values(NumberFile, i, n) = dy * sig2i                       !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * sig2i                    !< And find chi^2.


                            !< if gradient is calculated, determine alpha and beta as well
                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j, k) = alpha(j, k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< add other chi2 methods here!!!


                        endif
                    end Do                                                                  !< loop over all line of current exp. data file
                end Do                                                                      !< loop over y-columns
            end Do                                                                          !< loop over exp. data files

            ! Debug:
            ! print*,'chisqValues(0) = ', chisqValues(0)


            !< only used for MINPACK version of Levenberg-Marquardt algorithm
            if (GradientMethod == 2) then
                Do l = 1, NumberFreeParameterCopy
                    ModelFunction(l + 1, :, :, :) = GradientHelp(l, :, :, :)
                end Do
            endif


            !< Fill in the symmetric side of alpha
            Do j = 2, NumberFreeParameterCopy
                Do k = 1, (j - 1)
                   alpha(k, j) = alpha(j, k)
                end Do

                ! Debug:
                !    print*,'j = ', j
                !    print*,'alpha(j,:) = ', alpha(j,:NumberFreeParameterCopy)
                !    print*,'beta2(j) = ', beta2(j)
            end Do


            !< writing current value of chi**2 and corresponding values of parameters to file
            if (WriteChi2Flag) then
                NumberLinesChi2 = NumberLinesChi2 + 1
                k = 0
                FormattedParmValues(:)(:) = ""
                Do i = 1, ma
                    if (ia(i)) then
                        k = k + 1
                        a(i) = fitparam(k)

                        ! Debug:
                        !   print*,'fitparam(k) = ',k,fitparam(k)


                        !< build list with fit parameters
                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                        if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                            .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) int(a(i))
                            endif
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) a(i)
                            endif
                        endif
                        FormattedParmValues(k) = trim(adjustl(HelpString))
                    endif
                end Do


                !< write line of formatted parameter values to log file
                write(Chi2Channel,'(ES25.15,$)') chisqValues(0)
                Do i = 1, NumberFreeParameterCopy
                    write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                end Do
                write(Chi2Channel,*)
            endif
            return
        end subroutine ModelCalcChiFunctionLM


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionGeneral
        !>
        !> calculates the chi2 values for several given parameter vector sets
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      a:                  array containing the parameter set
        !>                      NumberParamVectors:         number of parameter vectors
        !>                      NumberFreeParameterCopy:    number of fitted parameters
        !>                      ParameterVectorSet:
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 19.08.2014
        !>
        subroutine ModelCalcChiFunctionGeneral(ma, ia, a, NumberParamVectors, NumberFreeParameterCopy, NumFile, MaxL, MaxCol, ParameterVectorSet, &
                                               chi2ValuesVector)

            use omp_lib

            implicit none
            integer :: ma                                                                   !< total number of parameters
            integer :: NumberParamVectors                                                   !< number of parameter vectors
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: colx                                                                 !< max. number of x columns
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(NumberParamVectors, NumberFreeParameterCopy) :: ParameterVectorSet   !< set of parameter vectors calculated by model module
            real*8, dimension(NumberParamVectors) :: chi2ValuesVector                       !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            !< working variables
            integer :: i, k, l                                                              !< loop variables
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .false.
            chi2ValuesVector = 0.d0
            colx = MaxColX
            call ModelCalcChiFunction(NumberFreeParameterCopy, NumberParamVectors, ParameterVectorSet, ModelFunctionFlag, ma, a, ia, colx, NumFile, &
                                      MaxL, MaxCol, chi2ValuesVector)

            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !< writing current value of chi**2 and corresponding values of parameters to file
            Do l = 1, NumberParamVectors
                if (WriteChi2Flag) then
                    NumberLinesChi2 = NumberLinesChi2 + 1
                    k = 0
                    FormattedParmValues(:)(:) = ""
                    Do i = 1, ma
                        if (ia(i)) then
                            k = k + 1
                            a(i) = ParameterVectorSet(l, k)


                            !< build list with fit parameters
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                            if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(a(i))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) a(i)
                                endif
                            endif
                            FormattedParmValues(k) = trim(adjustl(HelpString))
                        endif
                    end Do


                    !< write line of formatted parameter values to log file
                    write(Chi2Channel,'(ES25.15,$)') chi2ValuesVector(l)
                    Do i = 1, NumberFreeParameterCopy
                        write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                    end Do
                    write(Chi2Channel,*)
                endif
            end Do
            return
        end subroutine ModelCalcChiFunctionGeneral


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunction
        !>
        !> calculates the chi2 values for a given set of parameter vectors
        !>
        !>
        !> input variables:     NumberFreeParam:        number of free parameters
        !>                      NumParamVectors:        number of parameter vectors
        !>                      ParameterVectorSet:     parameter vector
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      ma:                     total number of parameters
        !>                      a:                      array containing the parameter set
        !>                      ia:                     flags for including/excluding parameter in the fit
        !>                      colx:                   number of columns in experimental x data
        !>                      NumFile:                number of experimental files
        !>                      MaxL:                   max number of lines of all experimental files
        !>                      MaxCol:                 max number of columns of all experimental files
        !>
        !> output variables:    chi2ValuesVector:       chi2 value for parameter vector
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 15.08.2014
        !>
        subroutine ModelCalcChiFunction(NumberFreeParam, NumParamVectors, ParameterVectorSet, ModelFunctionFlag, ma, a, ia, colx, NumFile, &
                                        MaxL, MaxCol, chi2ValuesVector)

            use omp_lib

            implicit none
            integer :: ok                                                                   !< status variable (needed for MPI version)
            integer :: l, m, n, i, NumberFile                                               !< loop variables
            integer :: k                                                                    !< counter variable
            integer :: num                                                                  !< working variable for model function value array
            integer :: NumberFreeParam                                                      !< number of free parameters
            integer :: NumParamVectors                                                      !< number of parameter vectors
            integer :: ThreadNumber                                                         !< thread number
            integer :: length                                                               !< total length of AtOnceFunction
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            real*8, dimension(ma) :: a, acopy                                               !< array containing the parameter set
            real*8 :: chi2Value                                                             !< working variable for chi2 value
            real*8 :: dy, sig2i, ymod                                                       !< working variables
            real*8, dimension(NumParamVectors) :: chi2ValuesVector                          !< chi2 value for parameter vector
            real*8, dimension(NumParamVectors, NumberFreeParam) :: ParameterVectorSet       !< parameter vector set
            real*8, dimension(NumFile, MaxL, MaxCol) :: LocalModelfunctionValues            !< local copy of the model func. values of lth param. vectors
            real*8, dimension(NumFile, MaxL, MaxCol) :: LocalChi2Values                     !< local copy of chi2 values of lth param. vectors
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function values are stored or
                                                                                            !< not
            logical :: EqualParam_Flag                                                      !< flag for parameter comparison
            logical :: NoCalFlag                                                            !< flag for not calculating model function for a specific
                                                                                            !< parameter vector
            logical :: StoredBefore_Flag                                                    !< flag indicating if parameter vector is stored before or not
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit
            logical :: OutOfRangeFlag                                                       !< indicates, if a parameter is out of allowed range


            !< reset output variables
            chi2ValuesVector = 0.d0
            ModelFunction = 0.d0


            !< determine total length of AtOnceFunction
            length = 0
            Do NumberFile = 1, NumberExpFiles
                length = length + (NumberYColumns(NumberFile) * lengthexpdata(NumberFile))
            end Do


            !< make a copy of a
            acopy = a


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'

            !$omp parallel default(shared) &
            !$omp shared(NumParamVectors, NumberFreeParam, ParameterVectorSet, ModelFunctionFlag, num, chi2ValuesVector) &
            !$omp shared(NumberYColumns, lengthexpdata, QualityLimit, BestSitesParamSet, BestSitesModelValues, BestSitesChi2Values) &
            !$omp shared(ParallelizationFlag, ModelFunction, ConverterInfit, length, ma, ia, NumFile, MaxL, MaxCol, colx) &
            !$omp private(l, chi2Value, k, NumberFile, n, i, m, ThreadNumber, a, NoCalFlag, ymod, LocalModelfunctionValues, LocalChi2Values) &
            !$omp private(EqualParam_Flag, StoredBefore_Flag, sig2i, dy, OutOfRangeFlag, ok)
            !$omp do

            Do l = 1, NumParamVectors


                !< determine current thread number
                ThreadNumber = 0
                if (ParallelizationFlag > 1) then
                    ThreadNumber = omp_get_thread_num()
                endif


                !< is calculation reduction wanted
                NoCalFlag = .false.
                if (UseCalculationReduction) then
                    Do i = 1, CurrentNumberLinesCalcReduction
                        EqualParam_Flag = .false.
                        call CompareTwoParameterSets(EqualParam_Flag, ma, NumberFreeParam, ia, CalculatedParameterSets(1, i, 2:), ParameterVectorSet(l,:))


                        !< if all parameter are identical set flag QualityFlag
                        if (EqualParam_Flag) then
                            chi2ValuesVector(l) = CalculatedParameterSets(1, i, 1)
                            NoCalFlag = .true.
                        endif
                    end Do
                endif


                !< is a calculation still neccessary
                if (.not. NoCalFlag) then


                    !< make copy of current parameter vector
                    OutOfRangeFlag = .false.
                    a = acopy
                    k = 0
                    Do i = 1, ma
                        if (ia(i)) then
                            k = k + 1
                            a(i) = ParameterVectorSet(l, k)
                            if (a(i) < paramset(3, i) .or. paramset(4, i) < a(i)) then
                                OutOfRangeFlag = .true.
                                exit
                            endif
                        else
                            a(i) = paramset(1, i)
                        endif
                    end Do


                    !-------------------------------------------------------------------------------------------------------------------------------------
                    !< calculate model function for current parameter vector
                    if (.not. OutOfRangeFlag) then
                        call FitFunctionAtOnce(ok, ma, a, NumFile, MaxL, MaxCol, colx)
                        if (ok == 1) then
                            stop
                        endif


                        !---------------------------------------------------------------------------------------------------------------------------------
                        !< check, if all model func. values are stored, or if only the model function values for the overall lowest chi2 value is stored
                        if (ModelFunctionFlag) then
                            ModelFunction(l, :, :, :) = AtOnceFunction(ThreadNumber, :, :, :)


                        !---------------------------------------------------------------------------------------------------------------------------------
                        !< calculate chi2 value for the current parameter vector and save model function values only for the lowest chi2 value
                        else
                            LocalModelfunctionValues = 0.d0
                            LocalChi2Values = 0.d0
                            Do NumberFile = 1, NumberExpFiles                               !< loop over exp. data files
                                Do n = 1, NumberYColumns(NumberFile)                        !< loop over y-columns
                                    Do i = 1, lengthexpdata(NumberFile)                     !< loop over all line of current exp. data file
                                        ymod = AtOnceFunction(ThreadNumber, NumberFile, n, i)
                                        LocalModelfunctionValues(NumberFile, i, n) = ymod   !< save model function


                                        !<----------------------------------------------------------------------------------------------------------------
                                        !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                                        if (abs(DetChi2) == 1) then

                                            sig2i = 1.d0
                                            if (expdatae(NumberFile,i,n) /= 0.d0) then      !< do the experimental datas include errors
                                                sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                                            endif
                                            dy = (expdatay(NumberFile, i, n) - ymod)        !< define distance between fit and data


                                        !<----------------------------------------------------------------------------------------------------------------
                                        !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)
                                        elseif (abs(DetChi2) == 2) then

                                            sig2i = 1.d0
                                            if (expdatae(NumberFile,i,n) /= 0.d0) then      !< do the experimental datas include errors
                                                sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                                            endif
                                            dy = (expdatay(NumberFile, i, n)**2 - ymod**2)
                                        endif
                                        LocalChi2Values(NumberFile, i, n) = dy * dy * sig2i
                                        !$omp critical
                                        chi2ValuesVector(l) = chi2ValuesVector(l) + LocalChi2Values(NumberFile, i, n)
                                        !$omp end critical
                                    end Do
                                end Do
                            end Do



                            !< store model function values and correpsonding chi2 values for the best fitness
                            !$omp critical
                            Do k = 1, QualityLimit
                                if (chi2ValuesVector(l) < BestSitesParamSet(k, 1)) then
                                    StoredBefore_Flag = .false.
                                    if (QualityLimit > 1) then
                                        Do m = 1, QualityLimit
                                            if (chi2ValuesVector(l) == BestSitesParamSet(m, 1)) then
                                                call CompareTwoParameterSets(StoredBefore_Flag, ma, NumberFreeParameter, ia, BestSitesParamSet(m, 2:), &
                                                                             ParameterVectorSet(l, :))
                                                if (StoredBefore_Flag) then
                                                    exit
                                                endif
                                            endif
                                        end Do
                                    endif
                                    if (.not. StoredBefore_Flag) then
                                        if (k < QualityLimit) then
                                            Do m = QualityLimit, (k + 1), (-1)
                                                BestSitesParamSet(m, :) = BestSitesParamSet((m - 1), :)
                                                BestSitesModelValues(m, :, :, :) = BestSitesModelValues((m - 1), :, :, :)
                                                BestSitesChi2Values(m, :, :, :) = BestSitesChi2Values((m - 1), :, :, :)
                                            end Do
                                        endif
                                        BestSitesParamSet(k, 1) = chi2ValuesVector(l)
                                        BestSitesParamSet(k, 2:) = ParameterVectorSet(l, :)
                                        BestSitesChi2Values(k, :, :, :) = LocalChi2Values(:, :, :)
                                        BestSitesModelValues(k, :, :, :) = LocalModelfunctionValues(:, :, :)
                                    endif
                                endif
                            end Do
                            !$omp end critical
                        endif
                    else
                        !$omp critical
                        if (ModelFunctionFlag) then
                            AtOnceFunction(ThreadNumber, :, :, :) = 0.d0
                            ModelFunction(l, :, :, :) = AtOnceFunction(ThreadNumber, :, :, :)
                        endif
                        chi2ValuesVector(l) = 1.d99
                        !$omp end critical
                    endif
                endif
            end Do

            !$omp end do
            !$omp end parallel

            ! Debug:
            !   print*,' '
            !   print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            !   print*,'ModelFunction(1, 1, 1, :) = ', ModelFunction(1, 1, 1, :)
            !   print*,'minval(ModelFunction(1, 1, 1, :)) = ', minval(ModelFunction(1, 1, 1, :))
            !   print*,'maxval(ModelFunction(1, 1, 1, :)) = ', maxval(ModelFunction(1, 1, 1, :))
            !   print*,'count(isnan(ModelFunction(1, 1, 1, :))) = ', count(isnan(ModelFunction(1, 1, 1, :)))
            !   print*,'##########################################'
            !   print*,'ModelFunction(1, 2, 1, :) = ', ModelFunction(1, 2, 1, :)
            !   print*,'minval(ModelFunction(1, 2, 1, :)) = ', minval(ModelFunction(1, 2, 1, :))
            !   print*,'maxval(ModelFunction(1, 2, 1, :)) = ', maxval(ModelFunction(1, 2, 1, :))
            !   print*,'count(isnan(ModelFunction(1, 2, 1, :))) = ', count(isnan(ModelFunction(1, 2, 1, :)))
            !   print*,'ParameterVectorSet(1, :) = ', ParameterVectorSet(1, :)
            !   stop 'Test LM-algorithm up to here!'


            return
        end subroutine ModelCalcChiFunction


        !>************************************************************************************************************************************************
        !> subroutine: ModelParamFree
        !>
        !> free memory used by variables of the Module Model
        !>
        !>
        !> input variables:     deallocstatus           status of the previous deallocation process
        !>
        !> output variables:    deallocstatus           status of the deallocation process
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 26.08.2014
        !>
        subroutine ModelParamFree(deallocstatus)

            implicit none
            integer :: deallocstatus                                                        !< status of the deallocation process


            !< deallocate memory of variables of Modules 'Variables' and 'FunctionCalling'
            if (allocated(lengthexpdata)) deallocate(lengthexpdata, stat = deallocstatus)
            if (allocated(NumberXColumns)) deallocate(NumberXColumns, stat = deallocstatus)
            if (allocated(NumberYColumns)) deallocate(NumberYColumns, stat = deallocstatus)
            if (allocated(NumberHeaderLines)) deallocate(NumberHeaderLines, stat = deallocstatus)
            if (allocated(OutputFileFormat)) deallocate(OutputFileFormat, stat = deallocstatus)
            if (allocated(NumberRanges)) deallocate(NumberRanges, stat = deallocstatus)
            if (allocated(ConverterInfit)) deallocate(ConverterInfit, stat = deallocstatus)
            if (allocated(ValueEmptyOutputFile)) deallocate(ValueEmptyOutputFile, stat = deallocstatus)
            if (allocated(LSRAdjustement)) deallocate(LSRAdjustement, stat = deallocstatus)
            if (allocated(chisqValues)) deallocate(chisqValues, stat = deallocstatus)
            if (allocated(BestSitesParamSet)) deallocate(BestSitesParamSet, stat = deallocstatus)
            if (allocated(paramset)) deallocate(paramset, stat = deallocstatus)
            if (allocated(AtOnceFunction)) deallocate(AtOnceFunction, stat = deallocstatus)
            if (allocated(AtOnceGradient)) deallocate(AtOnceGradient, stat = deallocstatus)
            if (allocated(FirstPointExpData)) deallocate(FirstPointExpData, stat = deallocstatus)
            if (allocated(LastPointExpData)) deallocate(LastPointExpData, stat = deallocstatus)
            if (allocated(expdatax)) deallocate(expdatax, stat = deallocstatus)
            if (allocated(expdatay)) deallocate(expdatay, stat = deallocstatus)
            if (allocated(expdatae)) deallocate(expdatae, stat = deallocstatus)
            if (allocated(CalculatedParameterSets)) deallocate(CalculatedParameterSets, stat = deallocstatus)
            if (allocated(MinRange)) deallocate(MinRange, stat = deallocstatus)
            if (allocated(MaxRange)) deallocate(MaxRange, stat = deallocstatus)
            if (allocated(BestSitesModelValues)) deallocate(BestSitesModelValues, stat = deallocstatus)
            if (allocated(BestSitesChi2Values)) deallocate(BestSitesChi2Values, stat = deallocstatus)
            if (allocated(ModelFunction)) deallocate(ModelFunction, stat = deallocstatus)
            if (allocated(FitParameterName)) deallocate(FitParameterName, stat = deallocstatus)
            if (allocated(FitParameterValue)) deallocate(FitParameterValue, stat = deallocstatus)
            if (allocated(CharacterForComments)) deallocate(CharacterForComments, stat = deallocstatus)
            if (allocated(CharacterSeperatingColumns)) deallocate(CharacterSeperatingColumns, stat = deallocstatus)
            if (allocated(ResamplingMethod)) deallocate(ResamplingMethod, stat = deallocstatus)
            if (allocated(InterpolationMethod)) deallocate(InterpolationMethod, stat = deallocstatus)
            if (allocated(OnlyYColumn)) deallocate(OnlyYColumn, stat = deallocstatus)
            if (allocated(LSRAdjustementFitFlag)) deallocate(LSRAdjustementFitFlag, stat = deallocstatus)
            if (allocated(NormalizationFlag)) deallocate(NormalizationFlag, stat = deallocstatus)
            if (allocated(ExpData_reversed_flag)) deallocate(ExpData_reversed_flag, stat = deallocstatus)
            if (allocated(NumberParamPerLine)) deallocate(NumberParamPerLine, stat = deallocstatus)
            if (allocated(ParameterName)) deallocate(ParameterName, stat = deallocstatus)
            if (allocated(ParameterFormat)) deallocate(ParameterFormat, stat = deallocstatus)
            if (allocated(LeadingString)) deallocate(LeadingString, stat = deallocstatus)
            if (allocated(TrailingString)) deallocate(TrailingString, stat = deallocstatus)
            if (allocated(ParamVisible)) deallocate(ParamVisible, stat = deallocstatus)
            if (allocated(FitFktInput)) deallocate(FitFktInput, stat = deallocstatus)
            if (allocated(FitFktOutput)) deallocate(FitFktOutput, stat = deallocstatus)
            if (allocated(valuesModel_output)) deallocate(valuesModel_output, stat = deallocstatus)
            if (allocated(xPointsModel_output)) deallocate(xPointsModel_output, stat = deallocstatus)
            return
        end subroutine ModelParamFree
end Module Model
!*********************************************************************************************************************************************************

