!*********************************************************************************************************************************************************
!>  Module: Model
!>
!>
!>  This module contains subroutines for the default MAGIX version where no external model program is included
!>  Copyright (C) 2009 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine SlaveWork:                     wake up MPI slave threads and do something or fall asleep
!>      - subroutine SlaveLoop:                     manage subroutine for slave threads
!>      - subroutine BroadCastMPIVariables:         broadcast MPI variables
!>      - subroutine ModelInit:                     initialize and broadcast module variables
!>      - subroutine PrepareInputFiles:             prepare input files for MPI call of external model programs using MPI_Spawn_Multiple function
!>      - subroutine ModelCalcChiFunctionLM:        calculates the chi2 values for the Levenberg-Marquard algorithm
!>      - subroutine ModelCalcChiFunctionGeneral:   calculates the chi2 values for several given parameter vector sets
!>      - subroutine ModelCalcChiFunction:          prepare call of subroutine 'ModelCalcCore'
!>      - subroutine ModelCalcCore:                 calculates the chi2 values for a given set of parameter vectors
!>      - subroutine ModelParamFree:                free memory used by variables of the Module Model
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    2014-09-12  Initial version
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module Model
    !> declare variables which are used by functions and subroutines of this module

    use mpi
    use Variables
    use FunctionCalling

    implicit none
    ! include 'mpif.h'
    integer :: ierr                                                                         !< MPI error variable
    integer :: errorcode                                                                    !< MPI error variable
    integer :: myrank                                                                       !< current thread number
    integer :: nsize                                                                        !< MPI variables
    integer :: nproc_for_mpi                                                                !< MPI variables
    integer :: icode0                                                                       !< MPI variables
    integer :: ireq0                                                                        !< MPI variables
    integer :: itag0                                                                        !< MPI variables
    integer :: TotalNumParamVectors                                                         !< used for subroutine ModelCalcCore for exchanging data
    integer, dimension(MPI_STATUS_SIZE) :: istatus0                                         !< MPI variables
    real*8, allocatable, dimension(:) :: LocalChi2ValueVector                               !< used for subroutine ModelCalcCore for exchanging data
    real*8, allocatable, dimension(:, :) :: GlobalParamVectorSet                            !< used for subroutine ModelCalcCore for exchanging data
    character(len=3) :: ParallelizationMethod                                               !< string describing the parallelization method

    contains


        !*************************************************************************************************************************************************
        !> subroutine: SlaveWork
        !>
        !> wake up MPI slave threads and do something or fall asleep
        !>
        !>
        !> input variables:     SlaveControlVar:        command which is send to the slave threads
        !>
        !>
        !> output parameters:   none
        !>
        !>
        !> \author Stefan Borowski, Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine SlaveWork(SlaveControlVar)

            implicit none
            integer :: SlaveControlVar                                                      !< command which is send to the slave threads
            integer :: i                                                                    !< loop variable

            ! Debug:
            ! print*,'MASTER: SlaveControlVar = ', SlaveControlVar


            Do i = 1, (TotalNumberOfThreads - 1)
                call MPI_Send(SlaveControlVar, 1, MPI_INTEGER, i, 0, MPI_COMM_WORLD, ierr)
            end Do
            return
        end subroutine SlaveWork


        !>************************************************************************************************************************************************
        !> subroutine: SlaveLoop
        !>
        !> manage subroutine for slave threads
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine SlaveLoop

            implicit none
            integer :: i
            integer :: SlaveControlVar                                                      !< command which is send to the slave threads
            integer :: length                                                               !< total length of AtOnceFunction
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: deallocstatus                                                        !< status of the deallocation process
            logical :: go                                                                   !< working variable
            logical :: ModelFunctionFlag                                                    !< do we need model function values?


            !< start a loop which is executed by all slave threads
            SlaveControlVar = 0
            Do
                call MPI_Irecv(SlaveControlVar, 1, MPI_INTEGER, 0, 0, MPI_COMM_WORLD, ireq0, ierr)

                ! Debug:
                ! print*,'SLAVE: myrank, SlaveControlVar = ', myrank, SlaveControlVar


                !< wait for jobs
                Do
                    Do i = 1, 10000
                    end Do
                    call MPI_Test(ireq0, go, istatus0, ierr)
                    if (go) exit
                end Do


                !< exit slave loop if (SlaveControlVar == 1)
                if (SlaveControlVar == 1) then
                    call ModelParamFree(deallocstatus)                                      !< free memory before exit
                    exit


                !< initialize variables (SlaveControlVar == 2)
                elseif (SlaveControlVar == 2) then

                    ! Debug:
                    ! print*,'********************************************************************************************>slave, myrank = ', myrank

                    call BroadCastMPIVariables


                !< determine model functions variables (SlaveControlVar == 3)
                elseif (SlaveControlVar == 3) then


                    !< determine chi2 value and model function values for different parameter vectors
                    ma = parameternumber
                    colx = MaxColX
                    NumFile = NumberExpFiles
                    MaxL = MaxExpLength
                    MaxCol = MaxColY
                    call ModelCalcCore(NumberFreeParameter, ModelFunctionFlag, length, ma, colx, NumFile, MaxL, MaxCol)
                endif
            end Do

            return
        end subroutine SlaveLoop


        !>************************************************************************************************************************************************
        !> subroutine: BroadCastMPIVariables
        !>
        !> broadcast MPI variables
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine BroadCastMPIVariables


            implicit none
            integer :: nelements                                                            !< working variable: size of array to be broadcast
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< define some global variables
            DontStoreModelFuncValuesFlag = .false.                                          !< model function values are stored for each function call
            ParallezitionMethod = 2                                                         !< set parallelization method to MPI (=2)
            myrankOfMPI = myrank                                                            !< copy thread number to global variable
            call GetEnv('MAGIXTempDirectory', TempDirectory)                                !< get temp directory for current thread


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< broadcast variables between master and slave threads


            !< broadcast scalar variables for Module 'Variables'
            call MPI_Bcast(QualityLimit, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberExpFiles, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(modelnumber, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberLinesChi2, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxColX, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxColY, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(parameternumber, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberInputFiles, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberOutputFiles, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(JobID, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxInputLines, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxParameter, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ExternalThreadNumber, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(CurrentNumberLinesCalcReduction, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(CurrentExpFile, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(CurrentYColumn, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberFreeParameter, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxExpLength, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxNumberRanges, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ParallezitionMethod, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(TotalNumberOfThreads, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ParallelizationFlag, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ochisq, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(NaNReplaceString, 10, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(currentpath, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(StandardWorkingDirectory, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(TempDirectory, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(RenormalizedChi2, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(initflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(printflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(Gradientflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(PlotIterationFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(WriteChi2Flag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(UseCalculationReduction, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(LastAlgorithmFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(AllInOneOutputFile, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(NumOutputFilesEqualNumExpFiles, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(DontStoreModelFuncValuesFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Broadcast of scalar variables of Module >>Variables<< done!")'


            !< allocate memory for variables for Module 'Variables'
            if (myrank /= 0) then                                                           !< only slave threads (de)allocate memory
                if (allocated(lengthexpdata)) then
                    deallocate(lengthexpdata, NumberXColumns, NumberYColumns, NumberHeaderLines, OutputFileFormat, NumberRanges, ConverterInfit, &
                               ValueEmptyOutputFile, LSRAdjustement, chisqValues, BestSitesParamSet, paramset, AtOnceFunction, AtOnceGradient, &
                               FirstPointExpData, LastPointExpData, expdatax, expdatay, expdatae, CalculatedParameterSets, MinRange, MaxRange, &
                               BestSitesModelValues, BestSitesChi2Values, ModelFunction, FitParameterName, FitParameterValue, CharacterForComments, &
                               CharacterSeperatingColumns, ResamplingMethod, InterpolationMethod, OnlyYColumn, LSRAdjustementFitFlag, NormalizationFlag, &
                               ExpData_reversed_flag, stat = dealloc_status)
                    if (dealloc_status /= 0) then                                           !< is all ok?
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelInit:")')
                        write(logchannel,'(2x,"Can not deallocate variables for Module Variables for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine ModelInit:")'
                        print '(2x,"Can not deallocate variables for Module Variables for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("dealloc_status = ",I4)',dealloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                endif
                allocate(lengthexpdata(NumberExpFiles), NumberXColumns(NumberExpFiles), NumberYColumns(NumberExpFiles), &
                         NumberHeaderLines(NumberOutputFiles), OutputFileFormat(NumberOutputFiles), NumberRanges(NumberExpFiles), &
                         ConverterInfit(NumberFreeParameter), ValueEmptyOutputFile(NumberOutputFiles), LSRAdjustement(NumberOutputFiles), &
                         chisqValues(0:0), BestSitesParamSet(QualityLimit, NumberFreeParameter + 1), paramset(4, parameternumber), &
                         AtOnceFunction(0:0, NumberExpFiles, MaxColY, MaxExpLength), FirstPointExpData(NumberExpFiles, MaxColX), &
                         LastPointExpData(NumberExpFiles, MaxColX), expdatax(NumberExpFiles, MaxExpLength, MaxColX), &
                         expdatay(NumberExpFiles, MaxExpLength, MaxColY), expdatae(NumberExpFiles, MaxExpLength, MaxColY), &
                         CalculatedParameterSets(ParallelizationFlag, CurrentNumberLinesCalcReduction, NumberFreeParameter + 1), &
                         MinRange(NumberExpFiles, MaxNumberRanges, MaxColX), MaxRange(NumberExpFiles, MaxNumberRanges, MaxColX), &
                         BestSitesModelValues(QualityLimit, NumberExpFiles, MaxExpLength, MaxColY), &
                         BestSitesChi2Values(QualityLimit, NumberExpFiles, MaxExpLength, MaxColY), &
                         ModelFunction(1, NumberExpFiles, MaxColY, MaxExpLength), FitParameterName(parameternumber), FitParameterValue(parameternumber), &
                         CharacterForComments(NumberOutputFiles), CharacterSeperatingColumns(NumberOutputFiles), ResamplingMethod(NumberOutputFiles), &
                         InterpolationMethod(NumberOutputFiles), OnlyYColumn(NumberOutputFiles), LSRAdjustementFitFlag(NumberOutputFiles), &
                         NormalizationFlag(NumberOutputFiles), ExpData_reversed_flag(NumberExpFiles), stat = alloc_status)
                if (alloc_status /= 0) then                                                 !< is all ok?
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine ModelInit:")')
                    write(logchannel,'(2x,"Can not allocate variables for Module Variables for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("alloc_status = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine ModelInit:")'
                    print '(2x,"Can not allocate variables for Module Variables for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("alloc_status = ",I4)',alloc_status
                    print '(" ")'
                    ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                    call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                    stop
                endif
                lengthexpdata = 0
                NumberXColumns = 0
                NumberYColumns = 0
                NumberHeaderLines = 0
                OutputFileFormat = 0
                NumberRanges = 0
                ConverterInfit = 0
                ValueEmptyOutputFile = 0.d0
                LSRAdjustement = 0.d0
                chisqValues = 0.d0
                BestSitesParamSet = 0.d0
                paramset = 0.d0
                AtOnceFunction = 0.d0
                FirstPointExpData = 0.d0
                LastPointExpData = 0.d0
                expdatax = 0.d0
                expdatay = 0.d0
                expdatae = 0.d0
                CalculatedParameterSets = 0.d0
                MinRange = 0.d0
                MaxRange = 0.d0
                BestSitesModelValues = 0.d0
                BestSitesChi2Values = 0.d0
                ModelFunction = 0.d0
                FitParameterName = ""
                FitParameterValue = ""
                CharacterForComments = ""
                CharacterSeperatingColumns = ""
                ResamplingMethod = ""
                InterpolationMethod = ""
                OnlyYColumn = .false.
                LSRAdjustementFitFlag = .false.
                NormalizationFlag = .false.
                ExpData_reversed_flag = .false.
            endif
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Allocation of variables of Module >>Variables<< done!")'


            !< broadcast scalar variables for Module 'Variables'
            !< exclude variables: AtOnceFunction, BestSitesModelValues, BestSitesChi2Values, ModelFunction
            nelements = size(lengthexpdata)
            call MPI_Bcast(lengthexpdata, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(NumberXColumns)
            call MPI_Bcast(NumberXColumns, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(NumberYColumns)
            call MPI_Bcast(NumberYColumns, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(NumberHeaderLines)
            call MPI_Bcast(NumberHeaderLines, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(OutputFileFormat)
            call MPI_Bcast(OutputFileFormat, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(NumberRanges)
            call MPI_Bcast(NumberRanges, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ConverterInfit)
            call MPI_Bcast(ConverterInfit, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ValueEmptyOutputFile)
            call MPI_Bcast(ValueEmptyOutputFile, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(LSRAdjustement)
            call MPI_Bcast(LSRAdjustement, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(chisqValues)
            call MPI_Bcast(chisqValues, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(BestSitesParamSet)
            call MPI_Bcast(BestSitesParamSet, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(paramset)
            call MPI_Bcast(paramset, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(FirstPointExpData)
            call MPI_Bcast(FirstPointExpData, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(LastPointExpData)
            call MPI_Bcast(LastPointExpData, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(expdatax)
            call MPI_Bcast(expdatax, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(expdatay)
            call MPI_Bcast(expdatay, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(expdatae)
            call MPI_Bcast(expdatae, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(MinRange)
            call MPI_Bcast(MinRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(MaxRange)
            call MPI_Bcast(MaxRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(FitParameterName) * 512
            call MPI_Bcast(FitParameterName, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(FitParameterValue) * 512
            call MPI_Bcast(FitParameterValue, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(CharacterForComments) * 512
            call MPI_Bcast(CharacterForComments, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(CharacterSeperatingColumns) * 512
            call MPI_Bcast(CharacterSeperatingColumns, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ResamplingMethod) * 512
            call MPI_Bcast(ResamplingMethod, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(InterpolationMethod) * 512
            call MPI_Bcast(InterpolationMethod, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(OnlyYColumn)
            call MPI_Bcast(OnlyYColumn, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(LSRAdjustementFitFlag)
            call MPI_Bcast(LSRAdjustementFitFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(NormalizationFlag)
            call MPI_Bcast(NormalizationFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ExpData_reversed_flag)
            call MPI_Bcast(ExpData_reversed_flag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Broadcast of array variables of Module >>Variables<< done!")'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< broadcast scalar variables for Module 'FunctionCalling'
            call MPI_Bcast(NumberOfUsedThreads, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberLinesOutput, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberColumnsBegin, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberColumnsEnd, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(DetChi2, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(idum, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(GradientMethod, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(GradientVariationValue, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(NaNReplaceString, 10, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(PlotPID, 100, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(CalculationMethod, 20, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(PathStartScript, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(ExeCommandStartScript, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(CommandWordOutput, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(InputDataPath, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(debugflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Broadcast of scalar variables of Module >>FunctionCalling<< done!")'


            !< allocate memory for variables for Module 'FunctionCalling'
            if (myrank /= 0) then                                                           !< only slave threads (de)allocate memory
                if (allocated(NumberParamPerLine)) then
                    deallocate(NumberParamPerLine, ParameterName, ParameterFormat, LeadingString, TrailingString, ParamVisible, FitFktInput, &
                               FitFktOutput, stat = dealloc_status)
                    if (dealloc_status /= 0) then                                           !< is all ok?
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelInit:")')
                        write(logchannel,'(2x,"Can not deallocate variables for Module FunctionCalling for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine FunctionCalling:")'
                        print '(2x,"Can not deallocate variables for Module FunctionCalling for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("dealloc_status = ",I4)',dealloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                endif
                allocate(NumberParamPerLine(NumberInputFiles, MaxInputLines), ParameterName(NumberInputFiles, MaxInputLines, MaxParameter), &
                         ParameterFormat(0:NumberInputFiles, 0:MaxInputLines, 0:MaxParameter), &
                         LeadingString(NumberInputFiles, MaxInputLines, MaxParameter), TrailingString(NumberInputFiles, MaxInputLines, MaxParameter), &
                         ParamVisible(NumberInputFiles, MaxInputLines, MaxParameter), FitFktInput(NumberInputFiles), FitFktOutput(NumberOutputFiles), &
                         stat = alloc_status)
                if (alloc_status /= 0) then                                                 !< is all ok?
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine ModelInit:")')
                    write(logchannel,'(2x,"Can not allocate variables for Module FunctionCalling for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("alloc_status = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine ModelInit:")'
                    print '(2x,"Can not allocate variables for Module FunctionCalling for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("alloc_status = ",I4)',alloc_status
                    print '(" ")'
                    ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                    call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                    stop
                endif
                NumberParamPerLine = 0
                ParameterName = ""
                ParameterFormat = ""
                LeadingString = ""
                TrailingString = ""
                ParamVisible = ""
                FitFktInput = ""
                FitFktOutput = ""
            endif
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Allocation of variables of Module >>Variables<< done!")'


            !< broadcast scalar variables for Module 'FunctionCalling'
            !< exclude variables:
            nelements = size(NumberParamPerLine)
            call MPI_Bcast(NumberParamPerLine, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ParameterName) * 8192
            call MPI_BCAST(ParameterName, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ParameterFormat) * 10
            call MPI_BCAST(ParameterFormat, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(LeadingString) * 8192
            call MPI_BCAST(LeadingString, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(TrailingString) * 8192
            call MPI_BCAST(TrailingString, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ParamVisible) * 8192
            call MPI_BCAST(ParamVisible, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(FitFktInput) * 8192
            call MPI_BCAST(FitFktInput, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(FitFktOutput) * 8192
            call MPI_BCAST(FitFktOutput, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Broadcast of array variables of Module >>FunctionCalling<< done!")'

            return
        end subroutine BroadCastMPIVariables


        !>************************************************************************************************************************************************
        !> subroutine: ModelInit
        !>
        !> initialize and broadcast module variables
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 26.08.2014
        !>
        subroutine ModelInit

            implicit none
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation


            !< initialize MPI environment
            if (printflag) then
                print '(9x, "Initialize MPI environment .. ", $)'
            endif


            !< resize AtOnceFunction variable (only for MASTER thread)
            !< this section is neccessary to avoid changes in the Algorithm core routine
            if (allocated(AtOnceFunction)) then
                deallocate(AtOnceFunction, stat = dealloc_status)
                if (dealloc_status /= 0) then                                           !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine ModelInit:")')
                    write(logchannel,'(2x,"Can not deallocate variable AtOnceFunction for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine ModelInit:")'
                    print '(2x,"Can not deallocate variable AtOnceFunction for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("dealloc_status = ",I4)',dealloc_status
                    print '(" ")'
                    ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                    call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                    stop
                endif
            endif
            allocate(AtOnceFunction(0:0, NumberExpFiles, MaxColY, MaxExpLength), stat = alloc_status)
            if (alloc_status /= 0) then                                                 !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine ModelInit:")')
                write(logchannel,'(2x,"Can not allocate variable AtOnceFunction for thread = ", I5, ".")') myrank
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("alloc_status = ",I4)') alloc_status
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')
                close(logchannel)

                print '(" ")'
                print '("Error in subroutine ModelInit:")'
                print '(2x,"Can not allocate variable AtOnceFunction for thread = ", I5, ".")', myrank
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("alloc_status = ",I4)',alloc_status
                print '(" ")'
                ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                stop
            endif
            AtOnceFunction = 0.d0


            !< redefine variable 'ParallelizationFlag'
            TotalNumberOfThreads = ParallelizationFlag
            NumberOfUsedThreads = 1                                                         !< in Module 'FunctionCalling' we need only one thread
            ParallelizationMethod = "MPI"


            !< wake up slave threads
            call SlaveWork(2)


            !< broadcast variables
            call BroadCastMPIVariables


            !< we've finished
            if (printflag) then
                print '("done!")'
            endif


            !< sleep little slave threads
            call SlaveWork(4)
            return
        end subroutine ModelInit


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionLM
        !>
        !> calculates the chi2 values for the Levenberg-Marquard algorithm
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      a:                  array containing the parameter set
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      NumberFreeParameterCopy:             number of fitted parameters
        !>                      fitparam:           parameter which have to be optimized
        !>                      colx:               number of columns in experimental x data
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>                      alpha:              matrix alpha (only used for Levenberg-Marquardt algorithm)
        !>                      beta2:              beta2 array (only used for Levenberg-Marquardt algorithm)
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 31.07.2014
        !>
        subroutine ModelCalcChiFunctionLM(ma, a, ia, NumberFreeParameterCopy, fitparam, colx, NumFile, MaxL, MaxCol, FitFunctionOut, Chi2Values, &
                                          alpha, beta2)

            implicit none
            integer :: i, j, k, l, m, n, fitnum, NumberFile                                 !< loop variables
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: counter                                                              !< counter for ModelFunction
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            integer :: NumParameterVectors                                                  !< number of parameter vectors in ParameterVectorSet
            real*8 :: dy, sig2i, wt, ymod                                                   !< working variables
            real*8 :: value                                                                 !< calculated value of the fit function
            real*8 :: variation, d1, d2                                                     !< working variables
            real*8, dimension(ma) :: steph                                                  !< working variable
            real*8, dimension(ma) :: modelparam                                             !< array containing the parameter set
            real*8, dimension(ma) :: modelparamcopy                                         !< copy of modelparam
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(ma, ma) :: alpha                                              !< matrix alpha
            real*8, dimension(NumberFreeParameterCopy) :: beta2                             !< beta2 array
            real*8, dimension(NumberFreeParameterCopy) :: fitparam                          !< parameter which have to be optimized
            real*8, dimension(NumFile, MaxL, MaxCol) :: FitFunctionOut                      !< values of the fit function at all calculated points
            real*8, dimension(NumFile, MaxL, MaxCol) :: Chi2Values                          !< values of the fit function at all calculated points
            real*8, dimension(NumberFreeParameterCopy, NumFile, MaxCol, MaxL) :: GradientHelp
            real*8, dimension(ma) :: dyda                                                   !< gradient of the fit function
            real*8, dimension(NumberFreeParameterCopy + 1) :: chi2ValuesVector              !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            real*8, dimension(1, NumberFreeParameterCopy) :: ParameterVectorSetLocal        !< local copy of ParameterVectorSet
            real*8, dimension(NumberFreeParameterCopy + 1, NumberFreeParameterCopy) :: ParameterVectorSet
                                                                                            !< set of parameter vectors calculated by model module
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file


            character(len=50) :: valueh, valuel                                             !< working variable for determine gradient
            logical :: equal_flag                                                           !< required for string comparison
            logical, dimension(ma) :: ChangeSign_Flag                                       !< change sign flag
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize values for fit function calculation
            Do j = 1, NumberFreeParameterCopy                                               !< Initialize (symmetric) alpha, beta2.
                Do k = 1, j
                    alpha(j,k) = 0.d0                                                       !< clear alpha partially
                end Do
                beta2(j) = 0.d0                                                             !< clear beta2
            end Do


            !< update the parameter set with the current values of the fit parameters some algorithms optimize only the parameters in the array fitparam
            !< all other parameter in the parameter set a are not included in these algorithms but are required for the calculation of the fit function
            k = 0
            Do i = 1, ma
                if (ia(i)) then
                    k = k + 1
                    a(i) = fitparam(k)
                endif
            end Do
            ParameterVectorSet(1, :) = fitparam(:)                                          !< first entry of ParameterVectorSet contains the current
                                                                                            !< parameter vector
            ! Debug:
            ! print*,'>>>>>>>>>>>>>fitparam = ', fitparam(:)


            !< check, if gradient is necessary (for Levenberg-Marquardt it is always necessary!)
            NumParameterVectors = 1
            if (Gradientflag) then
                NumParameterVectors = NumberFreeParameterCopy + 1


                !< initialize some working parameter
                steph = 0.d0
                ChangeSign_Flag = .false.
                modelparamcopy = a                                                          !< dublicate array containing the parameter set


                !< calculate parameter vector for gradient calculation

                !$omp parallel default(shared) &
                !$omp shared(OutputFileFormat, NumberRanges, ValueEmptyOutputFile, LSRAdjustement, FirstPointExpData, LastPointExpData, idum) &
                !$omp shared(MinRange, MaxRange, NaNReplaceString, StandardWorkingDirectory, CharacterForComments, CharacterSeperatingColumns) &
                !$omp shared(ResamplingMethod, InterpolationMethod, RenormalizedChi2, OnlyYColumn, LSRAdjustementFitFlag, NormalizationFlag) &
                !$omp shared(AtOnceGradient, ia, paramset, modelparamcopy, ConverterInfit) &
                !$omp shared(ParameterName, ParameterFormat, LeadingString, TrailingString, ParamVisible, NumberLinesOutput) &
                !$omp shared(printflag, chisqValues, NumberExpFiles, modelnumber, lengthexpdata, currentpath, NumberHeaderLines, QualityLimit) &
                !$omp shared(FitParameterName, FitParameterValue, NumberColumnsBegin, NumberColumnsEnd, NumberParamPerLine, CalculationMethod) &
                !$omp shared(CommandWordOutput, DetChi2, MaxParameter, ExternalThreadNumber, expdatax, expdatay, expdatae, TempDirectory) &
                !$omp shared(NumberXColumns, NumberYColumns, MaxColX, MaxColY, parameternumber, NumberInputFiles, ParallelizationFlag, JobID) &
                !$omp shared(MaxInputLines, UseCalculationReduction, WriteChi2Flag, Gradientflag,  initflag, CalculatedParameterSets) &
                !$omp shared(LastAlgorithmFlag, ChangeSign_Flag, steph, NumberFreeParameterCopy, ParameterVectorSet, fitparam) &
                !$omp shared(CurrentNumberLinesCalcReduction, ochisq, NumberLinesChi2, InputDataPath, FitFktInput, ModelFunction, FitFktOutput) &
                !$omp shared(ExeCommandStartScript, NumberOutputFiles, CurrentExpFile, GradientMethod, PathStartScript, CurrentYColumn) &
                !$omp shared(ExpData_reversed_flag, BestSitesModelValues, BestSitesChi2Values, BestSitesParamSet, GradientVariationValue) &
                !$omp private(fitnum, i, j, k, modelparam, d1, d2, sig2i, NumberFile, variation, value, valueh, valuel) &
                !$omp private(IntegerTrue, NumInputFile_index, i_index, j_index, equal_flag)
                !$omp do

                Do fitnum = 1, NumberFreeParameterCopy                                      !< loop over all free parameter
                    !$omp critical
                    ParameterVectorSet(fitnum + 1, :) = fitparam(:)
                    !$omp end critical
                    modelparam = modelparamcopy                                             !< load unmodified parameter values
                    i = ConverterInfit(fitnum)                                              !< get appropriate parameter index within parameter set
                    call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)

                    ! Debug:
                    ! print*,'fitnum, NumInputFile_index, i_index, j_index = ', fitnum, NumInputFile_index, i_index, j_index


                    !< check if parameter is within parameter limits
                    if (modelparamcopy(i) < paramset(3, i) .or. modelparamcopy(i) > paramset(4, i)) then
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        write(logchannel,'(2x,"The parameter ",A," is out of limits.")') &
                                          trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        write(logchannel,'(2x,"Upper limit = ", ES25.15)') paramset(4, i)
                        write(logchannel,'(2x,"Lower limit = ", ES25.15)') paramset(3, i)
                        write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        print '(13x,"The parameter ",A," is out of limits.")', trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        print '(13x,"Upper limit = ", ES25.15)', paramset(4, i)
                        print '(13x,"Lower limit = ", ES25.15)', paramset(3, i)
                        print '(13x,"Value of parameter = ", ES25.15)',modelparamcopy(i)
                        print '(" ")'
                        print '(13x,"Program aborted!")'
                        stop
                    endif


                    !< determine strength of variation
                    variation = GradientVariationValue                                      !< variation of the parameter in percent/100
                    !$omp critical
                    steph(i) = dabs(modelparamcopy(i) * variation)                          !< define stepsize for foating point numbers
                    if (modelparamcopy(i) == 0.d0) then
                        if (IntegerTrue) then                                               !< is parameter an integer ??
                            steph(i) = 1.d0                                                 !< variation of the parameter in percent/100
                        else
                            steph(i) = variation                                            !< variation of the parameter in percent/100
                        endif
                    elseif (IntegerTrue .and. steph(i) < 1.d0) then
                        steph(i) = 1.d0
                    endif
                    !$omp end critical

                    ! Debug:
                    ! print*,'i = ', i
                    ! print*,'modelparamcopy(i) = ', modelparamcopy(i)
                    ! print*,'steph(i) = ',steph(i)
                    ! print*,'modelparamcopy(i) + steph(i) = ', modelparamcopy(i) + steph(i)
                    ! print*,'paramset(3, i) = ', paramset(3, i)
                    ! print*,'paramset(4, i) = ', paramset(4, i)
                    ! print*,'modelparamcopy(i) - steph(i) = ', modelparamcopy(i) - steph(i)


                    !< test, if we can accelerate the calculation
                    value = modelparamcopy(i) + steph(i)
                    if (value < paramset(3, i) .or. paramset(4, i) < value) then
                        if (value > paramset(4, i)) then                                    !< is f(x_i + h) > upper limit ?
                            value = modelparamcopy(i) - steph(i)                            !< use f(x_i - h)
                            if (value < paramset(3, i)) then
                                if (dabs(modelparamcopy(i) - paramset(4, i)) < 1.d-12) then
                                    write(logchannel,*)
                                    write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                                    write(logchannel,'(2x,"The gradient for parameter ",A," cannot be calculated.")') &
                                                      trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    write(logchannel,'(2x,"The variation runs out of limits.")')
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")')
                                    write(logchannel,*)
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Upper limit = ",ES25.15)') paramset(4, i)
                                    write(logchannel,'(2x,"Lower limit = ",ES25.15)') paramset(3, i)
                                    write(logchannel,'(2x,"value of variation = ", ES25.15)') variation
                                    write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                                    write(logchannel,'(" ")')
                                    write(logchannel,'("Program aborted!")')

                                    print '(" ")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '("Error in subroutine ModelCalcChiFunctionLM:")'
                                    print '(2x,"The gradient for parameter ",A," cannot be calculated.")', &
                                                      trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    print '(2x,"The variation runs out of limits.")'
                                    print '(" ")'
                                    print '(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '(2x,"Upper limit = ", ES25.15)', paramset(4, i)
                                    print '(2x,"Lower limit = ", ES25.15)', paramset(3, i)
                                    print '(2x,"value of variation = ", ES25.15)', variation
                                    print '(2x,"Value of parameter = ", ES25.15)', modelparamcopy(i)
                                    print '(" ")'
                                    print '("Program aborted!")'
                                    stop
                                endif
                                value = paramset(4, i)
                            else
                                !$omp critical
                                ChangeSign_Flag(i) = .true.
                                !$omp end critical
                            endif
                        endif
                    endif

                    ! Debug:
                    ! print*,'> value, modelparamcopy(i), steph(i) = ',value, modelparamcopy(i), steph(i)


                    !< check, if the variation leads in combination with the Format to a variation in the current parameter
                    if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                        .or. index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                        write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) int(value)
                        write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) int(modelparamcopy(i))
                    else
                        write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) value
                        write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) modelparamcopy(i)
                    endif
                    equal_flag = .true.
                    valueh = adjustl(valueh)
                    valuel = adjustl(valuel)
                    if (len_trim(valueh) == len_trim(valuel)) then
                        Do j = 1, len_trim(valueh)
                            if (valueh(j:j) /= valuel(j:j)) then
                                equal_flag = .false.
                                exit
                            endif
                        end Do
                    else
                        equal_flag = .false.
                    endif
                    if (equal_flag) then                                                    !< both expressions are equal
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        write(logchannel,'(2x,"The format specification of the parameter ",A)') &
                                trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        write(logchannel,'(2x,"prevents the variation of the current parameter.")')
                        write(logchannel,'(2x," ")')
                        write(logchannel,'(2x,"The gradient entry for this parameter is set to zero. Therefore")')
                        write(logchannel,'(2x,"no variation of this parameter in the current iteration is done")')
                        write(logchannel,'(" ")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        print '(13x,"The format specification of the parameter ",A)', &
                               trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        print '(13x,"prevents the variation of the current parameter.")'
                        print '(" ")'
                        print '(13x,"The gradient entry for this parameter is set to zero. Therefore")'
                        print '(13x,"no variation of this parameter in the current iteration is done")'
                        print '(" ")'
                    endif


                    !< modify the ith parameter
                    !$omp critical
                    modelparam(i) = value                                                   !< modify value of the ith parameter
                    ParameterVectorSet(fitnum + 1, fitnum) = value
                    !$omp end critical

                end Do

                !$omp end do
                !$omp end parallel                                                          !< end of parallel environment

                modelparam = modelparamcopy                                                 !< restore old paramter values
            endif

            ! Debug:
            !    Do l = 2, NumParameterVectors
            !        Do k = 1, NumberFreeParameterCopy
            !            if (dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k)) > 1.d-6) then
            !                print*,'l, k, ParameterVectorSet(l, k) = ', l, k, ParameterVectorSet(l, k), ParameterVectorSet(1, k), &
            !                                                            dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k))
            !            endif
            !        end Do
            !    end Do
            !    stop


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .true.
            if (Gradientflag) then
                call ModelCalcChiFunction(NumberFreeParameterCopy, NumParameterVectors, ParameterVectorSet(:, :), ModelFunctionFlag, &
                                          ma, a, ia, colx, NumFile, MaxL, MaxCol, chi2ValuesVector)
            else
                ParameterVectorSetLocal(1, :) = ParameterVectorSet(1, :)
                call ModelCalcChiFunction(NumberFreeParameterCopy, NumParameterVectors, ParameterVectorSetLocal(:, :), ModelFunctionFlag, &
                                          ma, a, ia, colx, NumFile, MaxL, MaxCol, chi2ValuesVector)
            endif

            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< start loop for determine chi**2
            GradientHelp = 0.d0
            FitFunctionOut = 0.d0                                                           !< reset array containing the values of the fit function
            Chi2Values = 0.d0
            chisqValues = 0.d0
            counter = 0
            Do NumberFile = 1, NumberExpFiles                                               !< loop over exp. data files
                CurrentExpFile = NumberFile


                !< start loop over all lines(expdata)
                Do n = 1, NumberYColumns(NumberFile)                                        !< loop over y-columns
                    Do i = 1, lengthexpdata(NumberFile)                                     !< loop over all line of current exp. data file
                        CurrentYColumn = n
                        counter = counter + 1


                        !< get fit function
                        ymod = ModelFunction(1, NumberFile, n, i)

                        ! Debug:
                        !    print*,'>> NumberFile, n, i, counter, ymod = ', NumberFile, n, i, counter, ymod
                        !    print*,'>> ModelFunction(1, NumberFile, n, i) = ', ModelFunction(1, NumberFile, n, i)


                        !< determine sigma (sig2i) factor
                        sig2i = 1.d0
                        if (expdatae(NumberFile, i, n) /= 0.d0) then                        !< do the experimental datas include errors
                            sig2i = 1.d0 / (expdatae(NumberFile, i, n) * expdatae(NumberFile, i, n))
                        endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< get gradient
                        if (Gradientflag) then
                            dyda = 0.d0
                            Do l = 1, NumberFreeParameterCopy
                                j = ConverterInfit(l)                                       !< get appropriate parameter index within parameter set

                                ! order new: NumberFile, n, i, l,      j
                                ! order old: NumberFile, n, i, l,      j
                                ! order org: NumberFile, k, j, fitnum, i

                                ! Debug:
                                !    print*,'l, j = ', l, j
                                !    print*,'ParameterVectorSet(l + 1, l) = ', ParameterVectorSet(l + 1, l)
                                !    print*,'steph(j) = ', steph(j)
                                !    print*,'ChangeSign_Flag(j) = ', ChangeSign_Flag(j)
                                !    print*,'ModelFunction(l + 1, NumberFile, n, i) = ', ModelFunction(l + 1, NumberFile, n, i)
                                !    print*,'GradientMethod = ', GradientMethod


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine gradient for numerical recepies version
                                if (GradientMethod <= 1) then


                                    !< determine gradient of model function, see Numerical Recepies, Sect. 15.5
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        dyda(j) = (ModelFunction(1, NumberFile, n, i) - ModelFunction(l + 1, NumberFile, n, i)) / steph(j)
                                    else
                                        dyda(j) = (ModelFunction(l + 1, NumberFile, n, i) - ModelFunction(1, NumberFile, n, i)) / steph(j)
                                    endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine gradient for minpack version
                                elseif (GradientMethod == 2) then
                                    d1 = (expdatay(NumberFile, i, n) - ModelFunction(1, NumberFile, n, i))**2 * sig2i
                                    d2 = (expdatay(NumberFile, i, n) - ModelFunction(l + 1, NumberFile, n, i))**2 * sig2i
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        GradientHelp(l, NumberFile, n, i) = (d1 - d2) / steph(j)
                                    else
                                        GradientHelp(l, NumberFile, n, i) = (d2 - d1) / steph(j)
                                    endif


                                    !< copy gradient to dyda array
                                    dyda(j) = GradientHelp(l, NumberFile, n, i)
                                endif
                            end Do

                            ! Debug:
                            !    print*,'ymod = ', ymod
                            !    print*,'dyda(:) = ', dyda(:)
                            !    print*,'a = ', a
                            !    print*,'############################################################################'
                            !    stop
                        endif
                        FitFunctionOut(NumberFile, i, n) = ymod                             !< save value of fit function

                        ! Debug:
                        !    print*,' '
                        !    print*,' '
                        !    print*,'NumberExpFiles = ',NumberExpFiles
                        !    print*,'lengthexpdata(NumberFile) = ',lengthexpdata(NumberFile)
                        !    print*,'NumberYColumns(NumberFile) = ',NumberYColumns(NumberFile)
                        !    print*,'NumberFile = ',NumberFile
                        !    print*,'i = ',i
                        !    print*,'n = ',n
                        !    print*,'lenposdatexp = ',lenposdatexp
                        !    print*,'posdatexp = ',posdatexp
                        !    print*,'expdatay(NumberFile,i,n) = ',expdatay(NumberFile,i,n)
                        !    print*,'ymod = ',ymod
                        !    print*,'dyda = ',dyda(1:NumberFreeParameterCopy)
                        !    if (i==3) stop


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                        if (abs(DetChi2) == 1) then
                            dy = (expdatay(NumberFile, i, n) - ymod)                        !< define distance between fit and data
                            Chi2Values(NumberFile, i, n) = dy * dy * sig2i                  !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * dy * sig2i               !< And find chi^2.


                            !< if gradient is calculated, determine alpha and beta as well
                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j,k) = alpha(j,k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)**2
                        elseif (abs(DetChi2) == 2) then
                            dy = (expdatay(NumberFile, i, n)**2 - ymod**2)
                            Chi2Values(NumberFile, i, n) = dy * sig2i                       !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * sig2i                    !< And find chi^2.


                            !< if gradient is calculated, determine alpha and beta as well
                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j, k) = alpha(j, k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< add other chi2 methods here!!!


                        endif
                    end Do                                                                  !< loop over all line of current exp. data file
                end Do                                                                      !< loop over y-columns
            end Do                                                                          !< loop over exp. data files

            ! Debug:
            ! print*,'chisqValues(0) = ', chisqValues(0)


            !< only used for MINPACK version of Levenberg-Marquardt algorithm
            if (GradientMethod == 2) then
                Do l = 1, NumberFreeParameterCopy
                    ModelFunction(l + 1, :, :, :) = GradientHelp(l, :, :, :)
                end Do
            endif


            !< Fill in the symmetric side of alpha
            Do j = 2, NumberFreeParameterCopy
                Do k = 1, (j - 1)
                   alpha(k, j) = alpha(j, k)
                end Do

                ! Debug:
                !    print*,'j = ', j
                !    print*,'alpha(j,:) = ', alpha(j,:NumberFreeParameterCopy)
                !    print*,'beta2(j) = ', beta2(j)
            end Do


            !< writing current value of chi**2 and corresponding values of parameters to file
            if (WriteChi2Flag) then
                NumberLinesChi2 = NumberLinesChi2 + 1
                k = 0
                FormattedParmValues(:)(:) = ""
                Do i = 1, ma
                    if (ia(i)) then
                        k = k + 1
                        a(i) = fitparam(k)

                        ! Debug:
                        !   print*,'fitparam(k) = ',k,fitparam(k)


                        !< build list with fit parameters
                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                        if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                            .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) int(a(i))
                            endif
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) a(i)
                            endif
                        endif
                        FormattedParmValues(k) = trim(adjustl(HelpString))
                    endif
                end Do


                !< write line of formatted parameter values to log file
                write(Chi2Channel,'(ES25.15,$)') chisqValues(0)
                Do i = 1, NumberFreeParameterCopy
                    write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                end Do
                write(Chi2Channel,*)
            endif


            return
        end subroutine ModelCalcChiFunctionLM


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionGeneral
        !>
        !> calculates the chi2 values for several given parameter vector sets
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      a:                  array containing the parameter set
        !>                      NumberParamVectors:         number of parameter vectors
        !>                      NumberFreeParameterCopy:    number of fitted parameters
        !>                      ParameterVectorSet:
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 19.08.2014
        !>
        subroutine ModelCalcChiFunctionGeneral(ma, ia, a, NumberParamVectors, NumberFreeParameterCopy, NumFile, MaxL, MaxCol, ParameterVectorSet, &
                                               chi2ValuesVector)

            implicit none
            integer :: ma                                                                   !< total number of parameters
            integer :: NumberParamVectors                                                   !< number of parameter vectors
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: colx                                                                 !< max. number of x columns
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(NumberParamVectors, NumberFreeParameterCopy) :: ParameterVectorSet   !< set of parameter vectors calculated by model module
            real*8, dimension(NumberParamVectors) :: chi2ValuesVector                       !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            !< working variables
            integer :: i, k, l                                                              !< loop variables
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .false.
            chi2ValuesVector = 0.d0
            colx = MaxColX
            call ModelCalcChiFunction(NumberFreeParameterCopy, NumberParamVectors, ParameterVectorSet, ModelFunctionFlag, ma, a, ia, colx, NumFile, &
                                      MaxL, MaxCol, chi2ValuesVector)

            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !< writing current value of chi**2 and corresponding values of parameters to file
            Do l = 1, NumberParamVectors
                if (WriteChi2Flag) then
                    NumberLinesChi2 = NumberLinesChi2 + 1
                    k = 0
                    FormattedParmValues(:)(:) = ""

                    Do i = 1, ma
                        if (ia(i)) then
                            k = k + 1
                            a(i) = ParameterVectorSet(l, k)


                            !< build list with fit parameters
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                            if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(a(i))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) a(i)
                                endif
                            endif
                            FormattedParmValues(k) = trim(adjustl(HelpString))
                        endif
                    end Do


                    !< write line of formatted parameter values to log file
                    write(Chi2Channel,'(ES25.15,$)') chi2ValuesVector(l)
                    Do i = 1, NumberFreeParameterCopy
                        write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                    end Do
                    write(Chi2Channel,*)
                endif
            end Do
            return
        end subroutine ModelCalcChiFunctionGeneral


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunction
        !>
        !> prepare call of subroutine 'ModelCalcCore'
        !>
        !>
        !> input variables:     NumberFreeParam:        number of free parameters
        !>                      NumParamVectors:        number of parameter vectors
        !>                      ParameterVectorSet:     parameter vector
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      ma:                     total number of parameters
        !>                      a:                      array containing the parameter set
        !>                      ia:                     flags for including/excluding parameter in the fit
        !>                      colx:                   number of columns in experimental x data
        !>                      NumFile:                number of experimental files
        !>                      MaxL:                   max number of lines of all experimental files
        !>                      MaxCol:                 max number of columns of all experimental files
        !>
        !> output variables:    chi2ValuesVector:       chi2 value for parameter vector
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 15.08.2014
        !>
        subroutine ModelCalcChiFunction(NumberFreeParam, NumParamVectors, ParameterVectorSet, ModelFunctionFlag, ma, a, ia, colx, NumFile, &
                                        MaxL, MaxCol, chi2ValuesVector)

            use omp_lib

            implicit none
            integer :: NumberFile                                                           !< loop variables
            integer :: NumberFreeParam                                                      !< number of free parameters
            integer :: NumParamVectors                                                      !< number of parameter vectors
            integer :: length                                                               !< total length of AtOnceFunction
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            real*8, dimension(ma) :: a, acopy                                               !< array containing the parameter set
            real*8, dimension(NumParamVectors) :: chi2ValuesVector                          !< chi2 value for parameter vector
            real*8, dimension(NumParamVectors, NumberFreeParam) :: ParameterVectorSet       !< parameter vector set
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function values are stored or
                                                                                            !< not
            logical :: AllocateFlag                                                         !< used to avoid multiple allocation of memory
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            ! Debug:
            ! print*,'NumFile, MaxL, MaxCol = ', NumFile, MaxL, MaxCol
            ! print*,'ParallezitionMethod = ', ParallezitionMethod


            !< reset output variables
            chi2ValuesVector = 0.d0
            ModelFunction = 0.d0


            !< determine total length of AtOnceFunction
            length = 0
            Do NumberFile = 1, NumberExpFiles
                length = length + (NumberYColumns(NumberFile) * lengthexpdata(NumberFile))
            end Do


            !< make a copy of a
            acopy = a


            !< make a copy of NumParamVectors
            TotalNumParamVectors = NumParamVectors


            !< (de)allocate memory for array containing all parameter vectors and chi2 vector
            !< the following lines are included to aviod reallocation for each call of this subroutine
            AllocateFlag = .false.
            if (.not. allocated(GlobalParamVectorSet)) then
                AllocateFlag = .true.
            else
                if (size(GlobalParamVectorSet) /= (NumParamVectors * NumberFreeParameter)) then
                    AllocateFlag = .true.
                endif
            endif

            ! Debug:
            ! print*,'myrank, AllocateFlag = ', myrank, AllocateFlag


            if (AllocateFlag) then
                if (allocated(GlobalParamVectorSet)) then
                    deallocate(GlobalParamVectorSet, LocalChi2ValueVector, stat = dealloc_status)
                    if (dealloc_status /= 0) then                                           !< is all ok?
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                        write(logchannel,'(2x,"Can not deallocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5,".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine ModelCalcChiFunction:")'
                        print '(2x,"Can not deallocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("dealloc_status = ",I4)', dealloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                endif
                allocate(GlobalParamVectorSet(NumParamVectors, NumberFreeParameter), LocalChi2ValueVector(NumParamVectors), stat = alloc_status)
                if (alloc_status /= 0) then                                                 !< is all ok?
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                    write(logchannel,'(2x,"Can not allocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("alloc_status = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine ModelCalcChiFunction:")'
                    print '(2x,"Can not allocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("alloc_status = ",I4)', alloc_status
                    print '(" ")'
                    ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                    call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                    stop
                endif
            endif
            GlobalParamVectorSet(:, :) = ParameterVectorSet(:, :)
            LocalChi2ValueVector = 0.d0


            !< weak up slave threads and compute model function
            call SlaveWork(3)


            !< determine chi2 value and model function values for different parameter vectors
            call ModelCalcCore(NumberFreeParam, ModelFunctionFlag, length, ma, colx, NumFile, MaxL, MaxCol)


            !< sleep little slave threads
            call SlaveWork(4)


            !< get result of model function calculation
            chi2ValuesVector(:) = LocalChi2ValueVector(:)

            ! Debug:
            !    print*,'chi2ValuesVector = ', chi2ValuesVector
            !    ! call MPI_Abort(MPI_COMM_WORLD, ierr)
            !    call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
            !    stop
            return
        end subroutine ModelCalcChiFunction


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcCore
        !>
        !> calculates the chi2 values for a given set of parameter vectors
        !>
        !>
        !> input variables:     NumberFreeParam:        number of free parameters
        !>                      counter:                index of first parameter vector in local parameter set
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      length:                 total length of AtOnceFunction
        !>                      ma:                     total number of parameters
        !>                      a:                      array containing the parameter set
        !>                      ia:                     flags for including/excluding parameter in the fit
        !>                      colx:                   number of columns in experimental x data
        !>                      NumFile:                number of experimental files
        !>                      MaxL:                   max number of lines of all experimental files
        !>                      MaxCol:                 max number of columns of all experimental files
        !>
        !> output variables:    chi2Value:              chi2 value
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine ModelCalcCore(NumberFreeParam, ModelFunctionFlag, length, ma, colx, NumFile, MaxL, MaxCol)

            implicit none
            integer :: ok                                                                   !< status variable (needed for MPI version)
            integer :: m, l, j, n, i, NumberFile                                            !< loop variables
            integer :: k                                                                    !< counter variable
            integer :: NumberFreeParam                                                      !< number of free parameters
            integer :: nelements                                                            !< working variable: size of an array to send
            integer :: length                                                               !< total length of AtOnceFunction
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: CurrentNumberWorkers                                                 !< the number of workers
            integer :: LastWorker                                                           !< number of last worker
            integer :: NumWorkers                                                           !< total number of workers
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8 :: chi2Value                                                             !< working variable for chi2 value
            real*8 :: dy, sig2i, ymod                                                       !< working variables
            real*8, dimension(NumberFreeParam) :: LocalParamVector                          !< parameter vector
            real*8, dimension(NumFile, MaxL, MaxCol) :: LocalModelFunctionValuesChangedOrder
            real*8, dimension(NumFile, MaxL, MaxCol) :: LocalChi2ValuesChangedOrder         !< local copy of chi2 values of lth param. vectors
            real*8, dimension(NumberFreeParam) :: LocalParameterVector                      !< local copy of part of ParameterVectorSet
            real*8, dimension(NumberFreeParam) :: LocalParameterVector2                     !< further local copy of part of ParameterVectorSet
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function values are stored or
                                                                                            !< not
            logical :: EqualParam_Flag                                                      !< flag for parameter comparison
            logical :: NoCalFlag                                                            !< flag for not calculating model function for a specific
                                                                                            !< parameter vector
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit
            logical :: OutOfRangeFlag                                                       !< indicates, if a parameter is out of allowed range
            logical :: StoredBefore_Flag                                                    !< flag indicating if parameter vector is stored before or not
            logical :: AllocateFlag                                                         !< used to avoid multiple allocation of memory


            !< define ia array
            ia = .false.
            Do i = 1, ma
                if (paramset(2, i) == 1.d0) then
                    ia(i) = .true.
                endif
            end Do


            !< broadcast some variables between master and slave threads
            call MPI_Bcast(length, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ModelFunctionFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(ia, ma, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(a, ma, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(TotalNumParamVectors, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)



            !< (de)allocate memory for array containing all parameter vectors and chi2 vector
            !< the following lines are included to aviod reallocation for each call of this subroutine
            if (myrank /= 0) then

                ! Debug:
                ! print*,'-------------------myrank = ', myrank


                AllocateFlag = .false.
                if (.not. allocated(GlobalParamVectorSet)) then
                    AllocateFlag = .true.
                else
                    if (size(GlobalParamVectorSet) /= (TotalNumParamVectors * NumberFreeParameter)) then
                        AllocateFlag = .true.
                    endif
                endif

                ! Debug:
                ! print*,'-------------------myrank, AllocateFlag = ', myrank ,AllocateFlag


                if (AllocateFlag) then
                    if (allocated(GlobalParamVectorSet)) then
                        deallocate(GlobalParamVectorSet, stat = dealloc_status)
                        if (dealloc_status /= 0) then                                           !< is all ok?
                            write(logchannel,*)
                            write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                            write(logchannel,'(2x,"Can not deallocate variable GlobalParamVectorSet for thread = ", I5, ".")') myrank
                            write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                            write(logchannel,*)
                            write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                            write(logchannel,'(" ")')
                            write(logchannel,'("Program aborted!")')
                            close(logchannel)

                            print '(" ")'
                            print '("Error in subroutine ModelCalcChiFunction:")'
                            print '(2x,"Can not deallocate variable GlobalParamVectorSet for thread = ", I5, ".")', myrank
                            print '(2x,"Please close all other programs and restart the program!")'
                            print '(" ")'
                            print '("dealloc_status = ",I4)',dealloc_status
                            print '(" ")'
                            ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                            call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                            stop
                        endif
                    endif
                    allocate(GlobalParamVectorSet(TotalNumParamVectors, NumberFreeParameter), stat = alloc_status)
                    if (alloc_status /= 0) then                                                 !< is all ok?
                        write(logchannel,'(" ")')
                        write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                        write(logchannel,'(2x,"Can not allocate variable GlobalParamVectorSet for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,'(" ")')
                        write(logchannel,'("alloc_status = ",I4)') alloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine ModelCalcChiFunction:")'
                        print '(2x,"Can not allocate variable GlobalParamVectorSet for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("alloc_status = ",I4)',alloc_status
                        print '(" ")'
                        ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                        call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                        stop
                    endif
                endif
                GlobalParamVectorSet = 0.d0
            endif


            !< broadcast some variables between master and slave threads
            nelements = size(GlobalParamVectorSet)
            call MPI_Bcast(GlobalParamVectorSet, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(paramset)
            call MPI_Bcast(paramset, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            !    print*,'myrank, length = ', myrank, length
            !    print*,'myrank, ModelFunctionFlag = ', myrank, ModelFunctionFlag
            !    print*,'myrank, colx = ', myrank, colx
            !    print*,'myrank, NumFile = ', myrank, NumFile
            !    print*,'myrank, MaxL = ', myrank, MaxL
            !    print*,'myrank, MaxCol = ', myrank, MaxCol
            !    print*,'myrank, TotalNumParamVectors = ', myrank, TotalNumParamVectors
            !    print*,'myrank, GlobalParamVectorSet(myrank + 1, :) = ', myrank, GlobalParamVectorSet(myrank + 1, :)
            !    print*,'>>>myrank, paramset(1, :) = ', myrank, paramset(1, :)


            !< reset output variables
            chi2Value = 0.d0
            ModelFunction = 0.d0


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            Do l = 1, TotalNumParamVectors, TotalNumberOfThreads                                 !< loop over all parameter vectors

                ! Debug:
                ! print*,'l + myrank, TotalNumParamVectors, TotalNumberOfThreads = ', l + myrank, TotalNumParamVectors, TotalNumberOfThreads


                !< determine max. number of workers
                LastWorker = min(TotalNumParamVectors, (l + TotalNumberOfThreads - 1))
                NumWorkers = LastWorker - l + 1
                CurrentNumberWorkers = NumWorkers


                !< do nothing if no more parameter vector is left
                if (l + myrank > TotalNumParamVectors) then
                    exit
                endif


                !< select a parameter vector for each thread (master and slave)
                LocalParameterVector(:) = GlobalParamVectorSet(l + myrank, :)


                !< is calculation reduction wanted
                NoCalFlag = .false.
                if (UseCalculationReduction) then
                    Do i = 1, CurrentNumberLinesCalcReduction
                        EqualParam_Flag = .false.
                        call CompareTwoParameterSets(EqualParam_Flag, ma, NumberFreeParam, ia, CalculatedParameterSets(1, i, 2:), LocalParamVector(:))


                        !< if all parameter are identical set flag QualityFlag
                        if (EqualParam_Flag) then
                            chi2Value = CalculatedParameterSets(1, i, 1)
                            NoCalFlag = .true.
                        endif
                    end Do
                endif


                !< is a calculation still neccessary
                if (.not. NoCalFlag) then


                    !< make copy of current parameter vector
                    OutOfRangeFlag = .false.
                    k = 0
                    ia = .false.
                    a = 0.d0
                    Do i = 1, ma
                        if (paramset(2, i) == 1.d0) then
                            ia(i) = .true.
                            k = k + 1
                            a(i) = LocalParameterVector(k)
                            if (a(i) < paramset(3, i) .or. paramset(4, i) < a(i)) then
                                OutOfRangeFlag = .true.
                                exit
                            endif
                        else
                            a(i) = paramset(1, i)
                        endif
                    end Do

                    ! Debug:
                    ! print*,'myrank, LocalParameterVector = ', myrank, LocalParameterVector
                    ! print*,'myrank, paramset(1, :) = ', myrank, paramset(1, :)
                    ! print*,'myrank, NumberFreeParam, a = ', myrank, NumberFreeParam, a
                    ! print*,'myrank, OutOfRangeFlag = ', myrank, OutOfRangeFlag



                    !-------------------------------------------------------------------------------------------------------------------------------------
                    !< calculate model function for current parameter vector
                    if (.not. OutOfRangeFlag) then
                        call FitFunctionAtOnce(ok, ma, a, NumFile, MaxL, MaxCol, colx)
                        if (ok == 1) then
                            ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                            call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                            stop
                        endif

                        ! Debug:
                        ! print*,'myrank, AtOnceFunction(0, 1, 1, 100) = ', myrank, AtOnceFunction(0, 1, 1, 100)


                        !< calculate chi2 value for each thread
                        chi2Value = 0.d0
                        LocalModelFunctionValuesChangedOrder = 0.d0
                        LocalChi2ValuesChangedOrder = 0.d0
                        Do NumberFile = 1, NumberExpFiles                               !< loop over exp. data files
                            Do n = 1, NumberYColumns(NumberFile)                        !< loop over y-columns
                                Do i = 1, lengthexpdata(NumberFile)                     !< loop over all line of current exp. data file
                                    ymod = AtOnceFunction(0, NumberFile, n, i)
                                    LocalModelFunctionValuesChangedOrder(NumberFile, i, n) = ymod       !< save model function
                                    dy = 0.d0
                                    sig2i = 0.d0


                                    !<----------------------------------------------------------------------------------------------------------------
                                    !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                                    if (abs(DetChi2) == 1) then

                                        sig2i = 1.d0
                                        if (expdatae(NumberFile,i,n) /= 0.d0) then      !< do the experimental datas include errors
                                            sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                                        endif
                                        dy = (expdatay(NumberFile, i, n) - ymod)        !< define distance between fit and data


                                    !<----------------------------------------------------------------------------------------------------------------
                                    !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)
                                    elseif (abs(DetChi2) == 2) then

                                        sig2i = 1.d0
                                        if (expdatae(NumberFile,i,n) /= 0.d0) then      !< do the experimental datas include errors
                                            sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                                        endif
                                        dy = (expdatay(NumberFile, i, n)**2 - ymod**2)
                                    endif
                                    LocalChi2ValuesChangedOrder(NumberFile, i, n) = dy * dy * sig2i
                                    chi2Value = chi2Value + LocalChi2ValuesChangedOrder(NumberFile, i, n)
                                end Do
                            end Do
                        end Do

                    else
                        AtOnceFunction(0, :, :, :) = 0.d0
                        chi2Value = 1.d99
                        LocalModelFunctionValuesChangedOrder = 0.d0
                        LocalChi2ValuesChangedOrder = 0.d0
                    endif
                endif


                !< send results back to master thread
                if (myrank /= 0) then

                    ! Debug:
                    ! print*,'SLAVE: myrank, chi2Value = ', myrank, chi2Value


                    !< send some other parameters back to master thread
                    call MPI_Send(chi2Value, 1, MPI_DOUBLE_PRECISION, 0, 0, MPI_COMM_WORLD, ierr)
                    nelements = size(AtOnceFunction(:, :, :, :))
                    call MPI_Send(AtOnceFunction(:, :, :, :), nelements, MPI_DOUBLE_PRECISION, 0, 0, MPI_COMM_WORLD, ierr)
                    nelements = size(LocalModelFunctionValuesChangedOrder(:, :, :))
                    call MPI_Send(LocalModelFunctionValuesChangedOrder(:, :, :), nelements, MPI_DOUBLE_PRECISION, 0, 0, MPI_COMM_WORLD, ierr)
                    nelements = size(LocalChi2ValuesChangedOrder(:, :, :))
                    call MPI_Send(LocalChi2ValuesChangedOrder(:, :, :), nelements, MPI_DOUBLE_PRECISION, 0, 0, MPI_COMM_WORLD, ierr)


                !< MASTER: recieve results from slave threads
                else
                    Do j = 0, (CurrentNumberWorkers - 1)
                        m = l + j


                        !< get chi2 value
                        if (j > 0) then
!                            call MPI_Recv(chi2Value, 1, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, ireq0, ierr)
                            call MPI_Recv(chi2Value, 1, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, istatus0, ierr)
                        endif
                        LocalChi2ValueVector(m) = chi2Value

                        ! Debug:
                        ! print*,'myrank, m, chi2Value = ', myrank, m, chi2Value


                        !< get model function values
                        if (j > 0) then
                            nelements = size(AtOnceFunction)
!                            call MPI_Recv(AtOnceFunction, nelements, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, ireq0, ierr)
                            call MPI_Recv(AtOnceFunction, nelements, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, istatus0, ierr)

                            nelements = size(LocalModelFunctionValuesChangedOrder)
!                            call MPI_Recv(LocalModelFunctionValuesChangedOrder, nelements, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, ireq0, ierr)
                            call MPI_Recv(LocalModelFunctionValuesChangedOrder, nelements, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, istatus0, ierr)

                            nelements = size(LocalChi2ValuesChangedOrder)
!                            call MPI_Recv(LocalChi2ValuesChangedOrder, nelements, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, ireq0, ierr)
                            call MPI_Recv(LocalChi2ValuesChangedOrder, nelements, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, istatus0, ierr)
                        endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< do we need the model function values only?
                        if (ModelFunctionFlag) then
                            ModelFunction(m, 1:NumFile, 1:MaxCol, 1:MaxL) = AtOnceFunction(0, 1:NumFile, 1:MaxCol, 1:MaxL)


                        !---------------------------------------------------------------------------------------------------------------------------------
                        !< store model function values and correpsonding chi2 values for the best fitness
                        else
                            LocalParameterVector(:) = GlobalParamVectorSet(l + j, :)
                            Do k = 1, QualityLimit
                                if (chi2Value < BestSitesParamSet(k, 1)) then
                                    StoredBefore_Flag = .false.
                                    if (QualityLimit > 1) then
                                        Do m = 1, QualityLimit
                                            if (chi2Value == BestSitesParamSet(m, 1)) then
                                                LocalParameterVector2(:) = BestSitesParamSet(m, 2:)
                                                call CompareTwoParameterSets(StoredBefore_Flag, ma, NumberFreeParameter, ia, LocalParameterVector2(:), &
                                                                             LocalParameterVector(:))
                                                if (StoredBefore_Flag) then
                                                    exit
                                                endif
                                            endif
                                        end Do
                                    endif
                                    if (.not. StoredBefore_Flag) then
                                        if (k < QualityLimit) then
                                            Do m = QualityLimit, (k + 1), (-1)
                                                BestSitesParamSet(m, :) = BestSitesParamSet((m - 1), :)
                                                BestSitesModelValues(m, :, :, :) = BestSitesModelValues((m - 1), :, :, :)
                                                BestSitesChi2Values(m, :, :, :) = BestSitesChi2Values((m - 1), :, :, :)
                                            end Do
                                        endif
                                        BestSitesParamSet(k, 1) = chi2Value
                                        BestSitesParamSet(k, 2:) = LocalParameterVector(:)
                                        BestSitesChi2Values(k, :, :, :) = LocalChi2ValuesChangedOrder(:, :, :)
                                        BestSitesModelValues(k, :, :, :) = LocalModelFunctionValuesChangedOrder(:, :, :)
                                    endif
                                endif
                            end Do
                        endif
                    end Do
                endif

                ! Debug:
                ! print*,'myrank = ', myrank
                ! if (myrank == 0) print*,'#######################################################################################'
                ! print '(".", $)'
                ! call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                ! ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                ! call MPI_Abort(MPI_COMM_WORLD, errorcode, ierr)
                ! stop

            end Do
            return
        end subroutine ModelCalcCore


        !>************************************************************************************************************************************************
        !> subroutine: ModelParamFree
        !>
        !> free memory used by variables of the Module Model
        !>
        !>
        !> input variables:     deallocstatus           status of the previous deallocation process
        !>
        !> output variables:    deallocstatus           status of the deallocation process
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 26.08.2014
        !>
        subroutine ModelParamFree(deallocstatus)

            implicit none
            integer :: deallocstatus                                                        !< status of the deallocation process


            !< free memory of slave threads and kill slave threads
            if (myrank == 0) then
                call SlaveWork(1)
            endif


            !< deallocate memory of variables of Modules 'Variables' and 'FunctionCalling'
            if (allocated(lengthexpdata)) deallocate(lengthexpdata, stat = deallocstatus)
            if (allocated(NumberXColumns)) deallocate(NumberXColumns, stat = deallocstatus)
            if (allocated(NumberYColumns)) deallocate(NumberYColumns, stat = deallocstatus)
            if (allocated(NumberHeaderLines)) deallocate(NumberHeaderLines, stat = deallocstatus)
            if (allocated(OutputFileFormat)) deallocate(OutputFileFormat, stat = deallocstatus)
            if (allocated(NumberRanges)) deallocate(NumberRanges, stat = deallocstatus)
            if (allocated(ConverterInfit)) deallocate(ConverterInfit, stat = deallocstatus)
            if (allocated(ValueEmptyOutputFile)) deallocate(ValueEmptyOutputFile, stat = deallocstatus)
            if (allocated(LSRAdjustement)) deallocate(LSRAdjustement, stat = deallocstatus)
            if (allocated(chisqValues)) deallocate(chisqValues, stat = deallocstatus)
            if (allocated(BestSitesParamSet)) deallocate(BestSitesParamSet, stat = deallocstatus)
            if (allocated(paramset)) deallocate(paramset, stat = deallocstatus)
            if (allocated(AtOnceFunction)) deallocate(AtOnceFunction, stat = deallocstatus)
            if (allocated(AtOnceGradient)) deallocate(AtOnceGradient, stat = deallocstatus)
            if (allocated(FirstPointExpData)) deallocate(FirstPointExpData, stat = deallocstatus)
            if (allocated(LastPointExpData)) deallocate(LastPointExpData, stat = deallocstatus)
            if (allocated(expdatax)) deallocate(expdatax, stat = deallocstatus)
            if (allocated(expdatay)) deallocate(expdatay, stat = deallocstatus)
            if (allocated(expdatae)) deallocate(expdatae, stat = deallocstatus)
            if (allocated(CalculatedParameterSets)) deallocate(CalculatedParameterSets, stat = deallocstatus)
            if (allocated(MinRange)) deallocate(MinRange, stat = deallocstatus)
            if (allocated(MaxRange)) deallocate(MaxRange, stat = deallocstatus)
            if (allocated(BestSitesModelValues)) deallocate(BestSitesModelValues, stat = deallocstatus)
            if (allocated(BestSitesChi2Values)) deallocate(BestSitesChi2Values, stat = deallocstatus)
            if (allocated(ModelFunction)) deallocate(ModelFunction, stat = deallocstatus)
            if (allocated(FitParameterName)) deallocate(FitParameterName, stat = deallocstatus)
            if (allocated(FitParameterValue)) deallocate(FitParameterValue, stat = deallocstatus)
            if (allocated(CharacterForComments)) deallocate(CharacterForComments, stat = deallocstatus)
            if (allocated(CharacterSeperatingColumns)) deallocate(CharacterSeperatingColumns, stat = deallocstatus)
            if (allocated(ResamplingMethod)) deallocate(ResamplingMethod, stat = deallocstatus)
            if (allocated(InterpolationMethod)) deallocate(InterpolationMethod, stat = deallocstatus)
            if (allocated(OnlyYColumn)) deallocate(OnlyYColumn, stat = deallocstatus)
            if (allocated(LSRAdjustementFitFlag)) deallocate(LSRAdjustementFitFlag, stat = deallocstatus)
            if (allocated(NormalizationFlag)) deallocate(NormalizationFlag, stat = deallocstatus)
            if (allocated(ExpData_reversed_flag)) deallocate(ExpData_reversed_flag, stat = deallocstatus)
            if (allocated(NumberParamPerLine)) deallocate(NumberParamPerLine, stat = deallocstatus)
            if (allocated(ParameterName)) deallocate(ParameterName, stat = deallocstatus)
            if (allocated(ParameterFormat)) deallocate(ParameterFormat, stat = deallocstatus)
            if (allocated(LeadingString)) deallocate(LeadingString, stat = deallocstatus)
            if (allocated(TrailingString)) deallocate(TrailingString, stat = deallocstatus)
            if (allocated(ParamVisible)) deallocate(ParamVisible, stat = deallocstatus)
            if (allocated(FitFktInput)) deallocate(FitFktInput, stat = deallocstatus)
            if (allocated(FitFktOutput)) deallocate(FitFktOutput, stat = deallocstatus)
            if (allocated(valuesModel_output)) deallocate(valuesModel_output, stat = deallocstatus)
            if (allocated(xPointsModel_output)) deallocate(xPointsModel_output, stat = deallocstatus)
            return
        end subroutine ModelParamFree
end Module Model
!*********************************************************************************************************************************************************

