#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
## Package: UltraNest algorithm
##
##
##  This module contains the subroutines for UltraNest algorithm
##  Copyright (C) 2023 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - subroutine ModelFunctionCallClass.__init__:                       initialize class ModelFunctionCallClass
##      - subroutine ModelFunctionCallClass.logLhood:                       call of model-function
##      - subroutine ModelFunctionCallClass.GetBestResult:                  get best result
##      - subroutine ModelFunctionCallClass.GetBestFunctionValues:          get function values of best result
##      - subroutine ModelFunctionCallClass.WriteLogLines:                  write informatons of current iteration to log files
##      - subroutine ModelFunctionCallClass.my_prior_transform:             prior transform
##      - subroutine UltraNest:                                             main subroutine for UltraNest
##      - subroutine start:                                                 subroutine to start UltraNest algorithm
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2023-06-23      initial version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##******************************************************************** load packages *********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import copy                                                                                 ## import random copy
import numpy                                                                                ## load numpy package
import time                                                                                 ## load time package
import os                                                                                   ## load os package
import sys                                                                                  ## load sys package
from xclass.addons.MAGIX.Modules.python import FittingEngine                                ## import package containing the fitting engine
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## This class contains all required parameters for the ModelFunctionCall package
class ModelFunctionCallClass:


    ##****************************************************************************************************************************************************
    ## initialize class variables
    def __init__(self, printflagNumIn, GeneralAlgorithmSettingsIn, LastAlgorithmNumIn, DeterminationChi2In, PlotIterationIn, \
                 PlotTypeIn, fit_logIn, NumberInputFilesIn, NumberOutputFilesIn, ParallelizationFlagIn, JobIDIn, MaxInputLinesIn, MaxParameterIn, \
                 RenormalizedChi2In, currentpathIn, CalculationMethodIn, xAxisLabelIn, yAxisLabelIn, zAxisLabelIn, PathStartScriptIn, \
                 ExeCommandStartScriptIn, parameter_setIn, ExpDataXIn, ExpDataYIn, ExpDataErrorIn, NumberRangesIn, MinRangeIn, MaxRangeIn, \
                 NumberXColumnsIn, NumberYColumnsIn, LengthExpRangeIn, MaxRangeNumberIn, NumberExpFilesIn, MaxLengthIn, MaxColXIn, MaxColYIn, \
                 NumberParameterIn, MPIFlagIn, MAGIXrootDirectoryIn, JobDirIn, SpecialAlgorithmSettingsIn, StarterExecutable, Chi2Channel, logchannel, \
                 paramchannel, UltraNestBestSiteCounter, SortFortranNum):
        self.printflagNum = printflagNumIn
        self.GeneralAlgorithmSettings = GeneralAlgorithmSettingsIn
        self.LastAlgorithmNum = LastAlgorithmNumIn
        self.DeterminationChi2 = DeterminationChi2In
        self.PlotIteration = PlotIterationIn
        self.PlotType = PlotTypeIn
        self.fitlog = fit_logIn
        self.NumberInputFiles = NumberInputFilesIn
        self.NumberOutputFiles = NumberOutputFilesIn
        self.ParallelizationFlag = ParallelizationFlagIn
        self.JobID = JobIDIn
        self.MaxInputLines = MaxInputLinesIn
        self.MaxParameter = MaxParameterIn
        self.RenormalizedChi2 = RenormalizedChi2In
        self.currentpath = currentpathIn
        self.CalculationMethod = CalculationMethodIn
        self.xAxisLabel = xAxisLabelIn
        self.yAxisLabel = yAxisLabelIn
        self.zAxisLabel = zAxisLabelIn
        self.PathStartScript = PathStartScriptIn
        self.ExeCommandStartScript = ExeCommandStartScriptIn
        self.parameter_set = parameter_setIn
        self.ExpDataX = ExpDataXIn
        self.ExpDataY = ExpDataYIn
        self.ExpDataError = ExpDataErrorIn
        self.NumberRanges = NumberRangesIn
        self.MinRange = MinRangeIn
        self.MaxRange = MaxRangeIn
        self.NumberXColumns = NumberXColumnsIn
        self.NumberYColumns = NumberYColumnsIn
        self.LengthExpRange = LengthExpRangeIn
        self.MaxRangeNumber = MaxRangeNumberIn
        self.NumberExpFiles = NumberExpFilesIn
        self.MaxLength = MaxLengthIn
        self.MaxColX = MaxColXIn
        self.MaxColY = MaxColYIn
        self.NumberParameter = NumberParameterIn
        self.MPIFlag = MPIFlagIn
        self.MAGIXrootDirectory = MAGIXrootDirectoryIn
        self.JobDir = JobDirIn
        self.SpecialAlgorithmSettings = SpecialAlgorithmSettingsIn
        self.StarterExecutable = StarterExecutable
        self.Chi2Channel = Chi2Channel
        self.logchannel = logchannel
        self.paramchannel = paramchannel
        self.UltraNestBestSiteCounter = int(UltraNestBestSiteCounter)
        self.BestChi2 = numpy.ones(self.UltraNestBestSiteCounter) * 1.e99
        self.BestFunctionValues = []
        self.BestChi2Values = []
        self.BestParameter = []
        self.BestInputFiles = []
        for i in range(self.UltraNestBestSiteCounter):
            self.BestParameter.append(0.0)
            self.BestFunctionValues.append(0.0)
            self.BestChi2Values.append(0.0)
            self.BestInputFiles.append(0.0)
        self.BestLogLine = ""
        self.FuncCallCounter = 0
        self.TempDir = str(os.environ.get('MAGIXTempDirectory',''))
        self.CurrentParameterIndex = (-1)
        self.Chi2DistributionParam = []
        self.Chi2DistributionChi2 = []
        self.CurrIter = (-1)                                                                ## turn initialization flag on
        self.Points = []
        self.SortFortranNum = SortFortranNum
        self.FuncCounter = 1

        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## call of logLhood
    def logLhood(self, parameter_vector):
        """
        input variables:    parameter_vector:       parameter vector

        output variables:   chi2value:              value of chi^2
        """

        # Debug:
        # print ("parameter_vector = ", parameter_vector)


        ## reset output value
        chi2value = 0.0


        ## get number of parameter vectors
        if (len(parameter_vector.shape) == 1):
            number_param_set = 1
            SingleVectorFlag = True
        else:
            number_param_set = len(parameter_vector)
            SingleVectorFlag = False

        # Debug:
        # print ("number_param_set = ", number_param_set)


        ## print what you do
        i = self.FuncCounter + number_param_set - 1
        print ("\rCompute points {:d} - {:d} ..                                        \r".format(self.FuncCounter, i), end = "", flush = True)
        self.FuncCounter = i + 1


        ## make parameter_vector always has the same length
        # if (number_param_set == 1):
        #     parameter_vector = [parameter_vector]


        ## update parameter_set array and write new parameter vectors to file
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        NewParamFile = open(WorkingDirectory.strip() + "new-parameters.dat", 'w')
        for k in range(number_param_set):
            j = (-1)
            line = ""
            for i in range(self.NumberParameter):
                if (self.parameter_set[1][i] == 1):
                    j += 1
                    if (SingleVectorFlag):
                        line += "   " + str(parameter_vector[j])
                    else:
                        line += "   " + str(parameter_vector[k][j])
                else:
                    line += "   " + str(self.parameter_set[0][i])
            NewParamFile.write(line + "\n")
        NewParamFile.close()

        # Debug:
        # print ("ModelFunctionCall.startcall.__doc__ = ", ModelFunctionCall.startcall.__doc__)
        # print ("self.printflagNum = ", self.printflagNum)
        # print ("number_param_set = ", number_param_set)
        # print ("self.DeterminationChi2 = ", self.DeterminationChi2)
        # print ("self.PlotIteration = ", self.PlotIteration)
        # print ("self.PlotType = ", self.PlotType)
        # print ("self.NumberInputFiles = ", self.NumberInputFiles)
        # print ("self.NumberOutputFiles = ", self.NumberOutputFiles)
        # print ("self.ParallelizationFlag = ", self.ParallelizationFlag)
        # print ("self.JobID = ", self.JobID)
        # print ("self.MaxInputLines = ", self.MaxInputLines)
        # print ("self.MaxParameter = ", self.MaxParameter)
        # print ("self.RenormalizedChi2 = ", self.RenormalizedChi2)


        ## define dummy arguments
        chilm = 1.0
        AlgCounter = 1

        # Debug:
        # print ("self.StarterExecutable = ", self.StarterExecutable)
        # print ("self.MPIFlag = ", self.MPIFlag)
        # print ("self.MAGIXrootDirectory = ", self.MAGIXrootDirectory)
        # print ("self.JobDir = ", self.JobDir)
        # print ("self.JobID = ", self.JobID)
        # print ("self.NumberExpFiles = ", self.NumberExpFiles)
        # print ("self.MaxLength = ", self.MaxLength)
        # print ("self.MaxColX = ", self.MaxColX)
        # print ("self.MaxColY = ", self.MaxColY)
        # print ("self.MaxRangeNumber = ", self.MaxRangeNumber)
        # print ("self.NumberParameter = ", self.NumberParameter)
        # print ("self.LastAlgorithmNum = ", self.LastAlgorithmNum)
        # print ("self.ParallelizationFlag = ", self.ParallelizationFlag)
        # print ("self.DeterminationChi2 = ", self.DeterminationChi2)
        # print ("self.PlotIteration = ", self.PlotIteration)
        # print ("self.PlotType = ", self.PlotType)
        # print ("self.RenormalizedChi2 = ", self.RenormalizedChi2)
        # print ("AlgCounter = ", AlgCounter)
        # print ("chilm = ", chilm)
        # print ("self.GeneralAlgorithmSettings = ", self.GeneralAlgorithmSettings)
        # print ("self.SpecialAlgorithmSettings = ", self.SpecialAlgorithmSettings)
        # print ("self.parameter_set = ", self.parameter_set)
        # print ("self.xAxisLabel = ", self.xAxisLabel)
        # print ("self.yAxisLabel = ", self.yAxisLabel)
        # print ("self.zAxisLabel = ", self.zAxisLabel)
        # print ("self.SortFortranNum = ", self.SortFortranNum)


        ## call model function package
        ParamSetCounter = number_param_set
        numiter = self.CurrIter
        NameOfAlgorithm = "Model-Function_Call"
        calstatus, FitFunctionOut, Chi2Values, parmCopy, FinalParameterSet = FittingEngine.StartAlgPackage(NameOfAlgorithm, self.StarterExecutable, \
                                                                                            self.MPIFlag, self.MAGIXrootDirectory, self.JobDir, \
                                                                                            self.JobID, self.NumberExpFiles, self.MaxLength, \
                                                                                            self.MaxColX, self.MaxColY, self.MaxRangeNumber, \
                                                                                            self.NumberParameter, ParamSetCounter, \
                                                                                            self.LastAlgorithmNum, numiter, self.ParallelizationFlag, \
                                                                                            self.DeterminationChi2, self.PlotIteration, self.PlotType, \
                                                                                            self.RenormalizedChi2, AlgCounter, chilm, \
                                                                                            self.GeneralAlgorithmSettings, \
                                                                                            self.SpecialAlgorithmSettings, self.parameter_set, \
                                                                                            self.xAxisLabel, self.yAxisLabel, self.zAxisLabel, \
                                                                                            self.SortFortranNum)
        self.CurrIter = 0                                                                   ## turn initialization flag off


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get calculated chi2 values from temp file
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        ResultFile = open(WorkingDirectory.strip() + "chi2Values.dat", 'r')
        ResultFileContents = ResultFile.readlines()
        ResultFile.close()
        ValuesOfChi2 = []
        for val in ResultFileContents:
            ValuesOfChi2.append(float(val))

        # Debug:
        # print (">>>>ValuesOfChi2 = ", ValuesOfChi2)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check if parameter vector is within given range
        for i in range(number_param_set):                                                  ## loop over parameter sets
            k = (-1)
            for j in range(self.NumberParameter):                                          ## loop over all parameter
                if (self.parameter_set[1][j] == 1):
                    k += 1
                    if (SingleVectorFlag):
                        value = float(parameter_vector[k])
                    else:
                        value = float(parameter_vector[i][k])
                    if (value < float(self.parameter_set[2][j]) or value > float(self.parameter_set[3][j])):
                        ValuesOfChi2[i] = 1.e99
                        FitFunctionOut[i] = 0.e0
                        Chi2Values[i] = 0.e0

        # Debug:
        # print ("parameter_vector = ", parameter_vector)
        # print (">>>>ValuesOfChi2 = ", ValuesOfChi2)
        # print ("self.CurrentParameterIndex = ", self.CurrentParameterIndex)
        # sys.exit(0)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## save chi2 value and corresponding error parameter to arrays
        if (self.CurrentParameterIndex > (-1)):
            for k, chi2value in enumerate(ValuesOfChi2):
                self.Chi2DistributionParam.append(parameter_vector[k][self.CurrentParameterIndex])
                self.Chi2DistributionChi2.append(chi2value)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define complete list of chi2 values
        AllChi2Values = []
        RealNumParamVec = len(ValuesOfChi2)
        ParamVector = []
        for k, chi2value in enumerate(ValuesOfChi2):
            if (chi2value != numpy.inf):
                if (SingleVectorFlag):
                    lll = [parameter_vector[k]]
                else:
                    lll = [x for x in parameter_vector[k]]
                if (not (lll in ParamVector)):
                    AllChi2Values.append([chi2value, k])
                    ParamVector.append(lll)
        for k, chi2value in enumerate(self.BestChi2):
            AllChi2Values.append([chi2value, RealNumParamVec + k])

        # Debug:
        # print ("\nAllChi2Values = ", AllChi2Values)


        ## sort list of chi2 values
        AllChi2Values = sorted(AllChi2Values, key=lambda l:l[0])

        # Debug:
        # print ("Sorted: AllChi2Values = ", AllChi2Values


        ## make a copy of the current best site variables
        OldBestParameter = copy.deepcopy(self.BestParameter)
        OldBestFunctionValues = copy.deepcopy(self.BestFunctionValues)
        OldBestChi2Values = copy.deepcopy(self.BestChi2Values)
        OldBestInputFiles = copy.deepcopy(self.BestInputFiles)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store parameter vectors and function values for best sites
        for i in range(self.UltraNestBestSiteCounter):
            LocalChi2Value = AllChi2Values[i][0]
            LocalChi2Index = AllChi2Values[i][1]

            # Debug:
            # print ("\nLocalChi2Value = ", LocalChi2Value)
            # print ("LocalChi2Index = ", LocalChi2Index)
            # print ("number_param_set = ", RealNumParamVec)
            # print ("LocalChi2Index - RealNumParamVec = ", (LocalChi2Index - RealNumParamVec))


            ## identify best chi2 values
            if (LocalChi2Index < RealNumParamVec):
                self.BestChi2[i] = LocalChi2Value
                self.BestParameter[i] = copy.deepcopy(parameter_vector[LocalChi2Index])
                self.BestFunctionValues[i] = copy.deepcopy(FitFunctionOut[LocalChi2Index])
                self.BestChi2Values[i] = copy.deepcopy(Chi2Values[LocalChi2Index])


                ## get corresponding line with formatted parameters
                if (i == 0):
                    f = open(WorkingDirectory.strip() + "log_lines__single_call__" + str(LocalChi2Index + 1) + ".dat", 'r')
                    self.BestLogLine = f.readline()
                    f.close()
                    self.BestLogLine = self.BestLogLine[0:len(self.BestLogLine)-1]


                ## write best model function values to temp file
                for j in range(self.NumberExpFiles):
                    f = open(WorkingDirectory.strip() + "best_model_function_values_call__" + str(i + 1) + "__" + str(j + 1) + ".dat", 'w')
                    for lines in self.BestFunctionValues[i][j]:
                        f.write(str(lines[0]) + "\n")
                    f.close()


                ## write chi2 values to temp file
                for j in range(self.NumberExpFiles):
                    f = open(WorkingDirectory.strip() + "best_chi2_values_call__" + str(i + 1) + "__" + str(j + 1) + ".dat", 'w')
                    for lines in self.BestChi2Values[i][j]:
                        f.write(str(lines[0]) + "\n")
                    f.close()


                ## get corresponding contents of the input file(s)
                f = open(WorkingDirectory.strip() + "log_input-files__single_call__" + str(LocalChi2Index + 1) + ".dat", 'r')
                fcontentLocal = f.readlines()
                f.close()
                self.BestInputFiles[i] = fcontentLocal
            else:
                k = LocalChi2Index - RealNumParamVec
                self.BestChi2[i] = LocalChi2Value
                self.BestParameter[i] = copy.deepcopy(OldBestParameter[k])
                self.BestFunctionValues[i] = copy.deepcopy(OldBestFunctionValues[k])
                self.BestChi2Values[i] = copy.deepcopy(OldBestChi2Values[k])
                self.BestInputFiles[i] = copy.deepcopy(OldBestInputFiles[k])


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## append content of chi2-log file to total list of all calculated chi^2 values
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"


        ## read in local chi2 file
        f = open(WorkingDirectory.strip() + "log_chi2_single_call.dat", 'r')
        contentChi2LogFile = f.readlines()
        f.close()


        ## append content of local chi2 file to global chi2 file
        for line in contentChi2LogFile:
            self.Chi2Channel.write(line)
            self.Chi2Channel.flush()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print what you do
        self.FuncCallCounter += 1                                                           ## increase counter for model function calls


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store chi2 values
        for k in range(number_param_set):
            if (ValuesOfChi2[k] != numpy.inf):
                NewArray = []
                if (SingleVectorFlag):
                    NewArray.append(parameter_vector[k])
                else:
                    for param in parameter_vector[k]:
                        NewArray.append(param)
                NewArray.append(ValuesOfChi2[k])
                self.Points.append(NewArray)

        # Debug:
        # print (">>self.BestChi2 = ", self.BestChi2)
        # print ("self.BestLogLine = ", self.BestLogLine)
        # print ("self.BestInputFiles = ", self.BestInputFiles)
        # print ("self.BestParameter = ", self.BestParameter)


        ## we're done
        print ("\r                                                                                                \r", end = "", flush = True)


        ## define return variable
        ## for definition of logL see e.g. https://johannesbuchner.github.io/UltraNest/example-sine-highd.html
        if (SingleVectorFlag):
            ValuesOfChi2[:] = [(-0.5) * ValuesOfChi2[0]]
        else:
            ValuesOfChi2[:] = [x * (-0.5) for x in ValuesOfChi2]
        ValuesOfChi2 = numpy.asarray(ValuesOfChi2)
        return ValuesOfChi2
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get best result
    def GetBestResult(self):
        """
        input variables:    None

        output variables:   self.BestChi2:              chi2 value for best result
                            self.BestParameter:         parameter vector for best result
                            self.BestLogLine:           line in log file for best result
                            self.BestInputFiles:        content of input file for best result
        """


        ## define return parameters
        return self.BestChi2, self.BestParameter, self.BestLogLine, self.BestInputFiles
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get function values of best result
    def GetBestFunctionValues(self):
        """
        input variables:    None

        output variables:   self.BestParameter:         parameter vector of best result
                            self.BestFunctionValues:    model function values of best result
                            self.BestChi2Values:        chi2 function values of best result
        """


        ## define return parameters
        return self.BestParameter, self.BestFunctionValues, self.BestChi2Values
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## write informatons of current iteration to log files
    def WriteLogLines(self, CurrentIteration):
        """
        input variables:    CurrentIteration:       number of current iteration

        output variables:   None
        """

        # Debug:
        # print ("CurrentIteration = ", CurrentIteration)


        ## print what you do
        bestchi, bestparam, bestline, bestinput = self.GetBestResult()
        chisq = bestchi[0]
        if (self.printflagNum == 1):
            print ("\r                {:5d}{:s}".format(CurrentIteration, bestline))


        ## wirte information to param-log file
        self.paramchannel.write("\n")
        self.paramchannel.write("\n")
        outputstring = "*" * 122
        self.paramchannel.write(outputstring + "\n")
        outputstring = "Iteration: {:5d},  chi^2 = {:25.15e}".format(CurrentIteration, chisq)
        self.paramchannel.write(outputstring + "\n")
        self.paramchannel.write("\n")
        self.paramchannel.write("\n")
        self.paramchannel.write("\n")
        self.paramchannel.write("Parameters: " + str(bestparam[0]) + "\n")
        outputstring = "-" * 122
        self.paramchannel.write(outputstring + "\n")
        self.paramchannel.write("\n")
        self.paramchannel.flush()


        ## write contents of input file(s) to param-log file
        for lineinput in bestinput[0]:
            self.paramchannel.write(lineinput)
            self.paramchannel.flush()
        self.paramchannel.flush()


        ## wirte information to log file
        outputstring = "                {:5d}{:s}".format(CurrentIteration, bestline)
        self.logchannel.write(outputstring + "\n")
        self.logchannel.flush()


        ## define return parameters
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## prior transform
    def my_prior_transform(self, cube):
        """
        input variables:            cube:                   current parameter vector

        output variables:           params:                 status flag of calculation (= 0: all ok)
        """

        # Debug:
        # print ("cube = ", cube)
        # print ("cube.shape = ", cube.shape)
        # print ("len(cube.shape) = ", len(cube.shape))


        ## make a copy of current parameter vector
        params = cube.copy()
        if (len(cube.shape) == 1):
            number_param_set = 1
            SingleVectorFlag = True
        else:
            number_param_set = len(cube)
            SingleVectorFlag = False


        ## reproject new parameter vector
        for ParamVecID in range(number_param_set):
            j = (-1)
            for i in range(self.NumberParameter):
                if (self.parameter_set[1][i] == 1):
                    j += 1
                    lo = float(self.parameter_set[2][i])
                    hi = float(self.parameter_set[3][i])
                    if (SingleVectorFlag):
                        params[j] = cube[j] * (hi - lo) + lo
                    else:
                        params[ParamVecID][j] = cube[ParamVecID][j] * (hi - lo) + lo

                    # Debug:
                    # print ("\n\n")
                    # print ("ParamVecID = ", ParamVecID)
                    # print ("j = ", j)
                    # print ("lo = ", lo)
                    # print ("hi = ", hi)
                    # print ("cube[ParamVecID][j] = ", cube[ParamVecID][j])
                    # print ("params[ParamVecID][j] = ", params[ParamVecID][j])


        ## define return parameters
        return params
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Main subroutine to call UltraNest
def UltraNest(printflag, ModelCallClass, logchannel, num_par, max_iter, UltraNestCounter, NumberUltraNestSampler, \
              NumberBurnInIter, UltraNestDicString, JobDir, LocalJobDir, UltraNestBackendFileName, param_name):
    """
    input variables:    printflag:                  flag indicating if messages are printed to screen or not
                        ModelCallClass:             class for model function call
                        logchannel:                 channel for main log file
                        num_par:                    number of parameter
                        max_iter:                   max. number of iterations
                        UltraNestCounter:           counter for UltraNest algorithm call
                        NumberUltraNestSampler:     number of samplers
                        NumberBurnInIter:           number of iterations for burn-in phase
                        UltraNestDicString:         string describing dictionary with parameters
                                                    the remainder. Set to a low number (1e-2 … 1e-5) to make
                                                    sure peaks are discovered. Set to a higher number (0.5)
                                                    if you know the posterior is simple.
                        JobDir:                     path and name of current tmp directory
                        LocalJobDir:                path and name of current job directory
                        UltraNestBackendFileName:   path and name of backend file
                        param_name:                 list of parameter names

    output variables:   calstatus:                  status flag of calculation (= 0: all ok)
                        AddInfo:                    best chi2 value and corresponding parameter vector etc.
    """

    # Debug:
    # print ("printflag = ", printflag)
    # print ("ModelCallClass = ", ModelCallClass)
    # print ("logchannel = ", logchannel)
    # print ("num_par = ", num_par)
    # print ("max_iter = ", max_iter)
    # print ("UltraNestCounter = ", UltraNestCounter)
    # print ("NumberUltraNestSampler = ", NumberUltraNestSampler)
    # print ("NumberBurnInIter = ", NumberBurnInIter)
    # print ("UltraNestDicString = ", UltraNestDicString)
    # print ("JobDir = ", JobDir)
    # print ("LocalJobDir = ", LocalJobDir)
    # print ("UltraNestBackendFileName = ", UltraNestBackendFileName)
    # print ("param_name = ", param_name)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import modified emcee package
    try:
        import ultranest
    except:
        print ("Can not import {:s} package!".format(chr(34) + "ultranest" + chr(34)))


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    calstatus = 0
    AddInfo = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check user input


    ## check user-defined number of iterations for burn-in phase
    NumberBurnInIter = int(NumberBurnInIter)

    # Debug:
    # print ("\n\n\n")
    # print ("num_par = ", num_par)
    # print ("NumberUltraNestSampler = ", NumberUltraNestSampler)
    # print ("NumberBurnInIter = ", NumberBurnInIter)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define names of parameters
    ## param_name = [LocalParamName, LocalMoleculeName, LocalComponentIndex]
    param_names = []
    for i in range(num_par):
        LocalParamName = param_name[i][0]
        LocalMoleculeName = param_name[i][1]
        LocalComponentIndex = param_name[i][2]
        param_names.append(LocalParamName + "__" + LocalMoleculeName + "__comp_" + str(LocalComponentIndex))

    # Debug:
    # print ("param_names = ", param_names)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set parameters for run_iter
    update_interval_volume_fraction = 0.8
    update_interval_ncall = None
    log_interval = max_iter
    dlogz = 0.5
    dKL = 0.5
    frac_remain = 0.01
    Lepsilon = 0.001
    min_ess = 400
    max_ncalls = None
    max_num_improvement_loops = -1
    cluster_num_live_points = 40
    show_status = False
    viz_callback = 'auto'
    insertion_test_window = 10000
    insertion_test_zscore_threshold = 2
    # region_class = MLFriends
    # widen_before_initial_plateau_num_warn = 10000
    # widen_before_initial_plateau_num_max = 50000


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get parameters from dictionary
    UltraNestDicString = UltraNestDicString.strip()
    if (UltraNestDicString != ""):
        try:
            LocalDic = eval(UltraNestDicString.replace("'", "\""))
        except:
            LocalDic = None
        if (LocalDic is not None):
            for LocalKey in LocalDic.keys():
                if (LocalKey == "update_interval_volume_fraction"):
                    update_interval_volume_fraction = LocalDic[LocalKey]
                elif (LocalKey == "update_interval_ncall"):
                    update_interval_ncall = LocalDic[LocalKey]
                elif (LocalKey == "log_interval"):
                    log_interval = LocalDic[LocalKey]
                elif (LocalKey == "dlogz"):
                    dlogz = LocalDic[LocalKey]
                elif (LocalKey == "dKL"):
                    dKL = LocalDic[LocalKey]
                elif (LocalKey == "frac_remain"):
                    frac_remain = LocalDic[LocalKey]
                elif (LocalKey == "Lepsilon"):
                    Lepsilon = LocalDic[LocalKey]
                elif (LocalKey == "min_ess"):
                    min_ess = LocalDic[LocalKey]
                elif (LocalKey == "max_ncalls"):
                    max_ncalls = LocalDic[LocalKey]
                elif (LocalKey == "max_num_improvement_loops"):
                    max_num_improvement_loops = LocalDic[LocalKey]
                elif (LocalKey == "cluster_num_live_points"):
                    cluster_num_live_points = LocalDic[LocalKey]
                elif (LocalKey == "show_status"):
                    show_status = LocalDic[LocalKey]
                elif (LocalKey == "viz_callback"):
                    viz_callback = LocalDic[LocalKey]
                elif (LocalKey == "insertion_test_window"):
                    insertion_test_window = LocalDic[LocalKey]
                elif (LocalKey == "insertion_test_zscore_threshold"):
                    insertion_test_zscore_threshold = LocalDic[LocalKey]
                # elif (LocalKey == "region_class"):
                #     region_class = LocalDic[LocalKey]
                # elif (LocalKey == "widen_before_initial_plateau_num_warn"):
                #     widen_before_initial_plateau_num_warn = LocalDic[LocalKey]
                # elif (LocalKey == "widen_before_initial_plateau_num_max"):
                #     widen_before_initial_plateau_num_max = LocalDic[LocalKey]

    # Debug:
    # print ("update_interval_volume_fraction = ", update_interval_volume_fraction)
    # print ("update_interval_ncall = ", update_interval_ncall)
    # print ("log_interval = ", log_interval)
    # print ("dlogz = ", dlogz)
    # print ("dKL = ", dKL)
    # print ("frac_remain = ", frac_remain)
    # print ("Lepsilon  = ", Lepsilon )
    # print ("min_ess   = ", min_ess  )
    # print ("max_ncalls = ", max_ncalls)
    # print ("max_num_improvement_loops = ", max_num_improvement_loops)
    # print ("cluster_num_live_points = ", cluster_num_live_points)
    # print ("show_status = ", show_status)
    # print ("viz_callback = ", viz_callback)
    # print ("insertion_test_window = ", insertion_test_window)
    # print ("insertion_test_zscore_threshold = ", insertion_test_zscore_threshold)
    # print ("widen_before_initial_plateau_num_warn = ", widen_before_initial_plateau_num_warn)
    # print ("widen_before_initial_plateau_num_max = ", widen_before_initial_plateau_num_max)
    # print ("region_class = ", region_class)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## restore from a previous run
    UltraNestBackendFileName = UltraNestBackendFileName.strip()
    UseBackendFileFlag = False
    if (UltraNestBackendFileName != ""):
        if (os.path.isfile(UltraNestBackendFileName)):
            UseBackendFileFlag = True


            ## print what you do
            print ("\n\n           Resume and reuse an existing run:", flush = True)


            ## define accelerated likelihood and prior transform
            aux_pnames, aux_logL, aux_prior_trans, vectorized = ultranest.integrator.warmstart_from_similar_file(UltraNestBackendFileName, \
                                                                                                                 param_names, \
                                                                                                                 ModelCallClass.logLhood, \
                                                                                                                 ModelCallClass.my_prior_transform, \
                                                                                                                 vectorized = True)
            ## call ReactiveNestedSampler class
            sampler = ultranest.ReactiveNestedSampler(aux_pnames, aux_logL, aux_prior_trans, vectorized = vectorized, resume = "resume-similar")

            # Debug:
            # print ("\n")
            # print ("param_names = ", param_names)
            # print ("aux_pnames = ", aux_pnames)
            # print ("aux_logL = ", aux_logL)
            # print ("aux_prior_trans = ", aux_prior_trans)
            # print ("vectorized = ", vectorized)
            # print ("sampler = ", sampler)
            # sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## normal run: call ReactiveNestedSampler class
    if (not UseBackendFileFlag):
        sampler = ultranest.ReactiveNestedSampler(param_names, \
                                                  ModelCallClass.logLhood, \
                                                  ModelCallClass.my_prior_transform, \
                                                  vectorized = True, \
                                                  log_dir = JobDir, \
                                                  resume = True, \
                                                  storage_backend = 'hdf5')
    # Debug:
    # print ("sampler = ", sampler)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## perform burn-in iterations
    # if (NumberBurnInIter > 0):


    #     ##------------------------------------------------------------------------------------------------------------------------------------------------
    #     ## print what you do to log file (and screen)
    #     OutputString1 = "\r                                                                            "
    #     OutputString1 += "\n\n           Calculate burn-in steps:"
    #     logchannel.write(OutputString1 + "\n")
    #     OutputString2 = "\n\n           Iteration:                    chi^2:     Parameter:"
    #     logchannel.write(OutputString2 + "\n")
    #     logchannel.flush()
    #     if (printflag):
    #         print (OutputString1, flush = True)
    #         print (OutputString2, flush = True)


    #     ##-----------------------------------------------------------------------------------------------------------------------------------------------
    #     ## Burn-In


    #     ## create step sampler:
    #     import ultranest.stepsampler
    #     sampler.stepsampler = ultranest.stepsampler.SliceSampler(nsteps = NumberBurnInIter,
    #                                                              generate_direction = ultranest.stepsampler.generate_mixture_random_direction,
    #                                                             # adaptive_nsteps=False,
    #                                                             # max_nsteps=400
    #                                                             )


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print what you do to log file (and screen)
    PosteriorDic = None
    result = None
    Local_max_iter = 1
    if (Local_max_iter > 0):
        OutputString1 = "\n\n           Calculate final UltraNest run:"
        logchannel.write(OutputString1 + "\n")
        OutputString2 = "\n\n           Iteration:                    chi^2:     Parameter:"
        logchannel.write(OutputString2 + "\n")
        logchannel.flush()
        if (printflag):
            print (OutputString1, flush = True)
            print (OutputString2, flush = True)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## explore and plot result at each iteration
        # result = sampler.run(min_num_live_points = 400, max_iters = max_iter)
        CurrentIteration = 0
        for result in sampler.run_iter(max_iters = max_iter,
                                       min_num_live_points = NumberUltraNestSampler,
                                       update_interval_volume_fraction = update_interval_volume_fraction,
                                       update_interval_ncall = update_interval_ncall,
                                       log_interval = log_interval,
                                       dlogz = dlogz,
                                       dKL = dKL,
                                       frac_remain = frac_remain,
                                       Lepsilon = Lepsilon,
                                       min_ess = min_ess,
                                       max_ncalls = max_ncalls,
                                       max_num_improvement_loops = max_num_improvement_loops,
                                       cluster_num_live_points = cluster_num_live_points,
                                       show_status = show_status,
                                       viz_callback = viz_callback,
                                       insertion_test_window = insertion_test_window,
                                       insertion_test_zscore_threshold = insertion_test_zscore_threshold,
                                       # region_class = region_class,
                                       # widen_before_initial_plateau_num_max = widen_before_initial_plateau_num_max,
                                       # widen_before_initial_plateau_num_warn = widen_before_initial_plateau_num_warn,
                                       ):
            # print ('lnZ = %(logz).2f +- %(logzerr).2f' % result)


            ## print out status
            CurrentIteration += 1
            ModelCallClass.WriteLogLines(CurrentIteration)


            ## get errors
            PosteriorDic = result["posterior"]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print results
        sampler.print_results()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## plot run (stored as run.pdf: diagnostic plot showing integration progress)
        ## - Visualises how the number of live points, likelihood and posterior weight evolved through the nested sampling run.
        ## - Visualises the evidence integration and its uncertainty.
        print ("\n\n\t Create run plot .. ", end = "", flush = True)
        sampler.plot_run()
        cmdString = "mv {:s}plots/run.pdf {:s}UltraNest__run__call_{:d}.pdf".format(JobDir, LocalJobDir, abs(UltraNestCounter))
        os.system(cmdString)
        print ("done!")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## plot trace (stored as trace.pdf: diagnostic plot showing problem structure)
        ## - Visualises how each parameter’s range was reduced as the nested sampling proceeds.
        ## - Color indicates where the bulk of the posterior lies.
        ## - Useful to understand the structure of the inference problem, and which parameters are learned first.
        print ("\n\n\t Create trace plot .. ", end = "", flush = True)
        sampler.plot_trace()
        cmdString = "mv {:s}plots/trace.pdf {:s}UltraNest__trace__call_{:d}.pdf".format(JobDir, LocalJobDir, abs(UltraNestCounter))
        os.system(cmdString)
        print ("done!")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## move backend file (stored in "results/points.hdf5") to user-defined destination
        if (UltraNestBackendFileName != ""):
            print ("\n\n\t Save backend file .. ", end = "", flush = True)
            # cmdString = "mv {:s}results/points.hdf5 {:s}".format(JobDir, UltraNestBackendFileName)
            cmdString = "mv {:s}chains/weighted_post_untransformed.txt {:s}".format(JobDir, UltraNestBackendFileName)
            os.system(cmdString)
            print ("done!")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## save debug log
        cmdString = "cp {:s}debug.log {:s}UltraNest__debug__call_{:d}.log ".format(JobDir, LocalJobDir, abs(UltraNestCounter))
        os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## clean up job directory
        print ("\n\n\t Clean up job directory .. ", end = "", flush = True)
        cmdString = "rm -rf {:s}chains/ ; ".format(JobDir)
        cmdString += "rm -rf {:s}extra/ ; ".format(JobDir)
        cmdString += "rm -rf {:s}info/ ; ".format(JobDir)
        cmdString += "rm -rf {:s}plots/ ; ".format(JobDir)
        cmdString += "rm -rf {:s}results/ ; ".format(JobDir)
        os.system(cmdString)
        print ("done!\n\n")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get results
    bestchi, bestparam, bestline, bestinput = ModelCallClass.GetBestResult()
    prob = None
    AddInfo = [0.0, prob, bestinput, sampler, bestchi, bestparam, PosteriorDic, result]

    # Debug:
    # print ("bestparam = ", bestparam)


    ## define return parameters
    return (calstatus, AddInfo)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## main routinte for UltraNest algorithm
def Start(SpecialAlgorithmSettings, GeneralAlgorithmSettings, printflagNum, LastAlgorithmNum, ParamSetCounter, chilm, numiter, UltraNestCounter, \
          DeterminationChi2, PlotIteration, PlotType, fit_log, NumberInputFiles, NumberOutputFiles, ParallelizationFlag, JobID, MaxInputLines, \
          MaxParameter, RenormalizedChi2, currentpath, CalculationMethod, xAxisLabel, yAxisLabel, zAxisLabel, PathStartScript, \
          ExeCommandStartScript, parameter_set, ExpDataX, ExpDataY, ExpDataError, NumberRanges, MinRange, MaxRange, NumberXColumns, \
          NumberYColumns, LengthExpRange, MaxRangeNumber, NumberExpFiles, MaxLength, MaxColX, MaxColY, NumberParameter, plotflag, modelflag, \
          MPIFlag, MAGIXrootDirectory, JobDir, SortFortranNum, ParameterValuesOriginal, FitParameterName):
    """
    input variables:            SpecialAlgorithmSettings:   algorithm user setttings
                                GeneralAlgorithmSettings:   special algorithm settings
                                printflagNum:               flag for screen output 1 (=yes) or 0 (=no)
                                LastAlgorithmNum:           number of last algorithm
                                ParamSetCounter:            number of best sites
                                chilm:                      user defined abort criteria for chi**2
                                numiter:                    max. number of iterations
                                UltraNestCounter:           counts number of calls
                                DeterminationChi2:          method being used for the determination of chi^2
                                PlotIteration:              plot model function for each iteration set 1(=yes) or 0(=no)
                                PlotType:                   get type of plot
                                fit_log:                    path for log-file containing the current values of chi**2
                                NumberInputFiles:           number of input files for the external model program
                                NumberOutputFiles:          number of output files for the external model program
                                ParallelizationFlag:        contains the number of processors used for parallelization
                                JobID:                      job identification number
                                MaxInputLines:              max number of lines in an input file
                                MaxParameter:               max number of parameters in a line of an input file
                                RenormalizedChi2:           flag for using renormalized chi**2
                                currentpath:                path of the working directory
                                CalculationMethod:          method of computation (at once or point-to-point)
                                xAxisLabel:                 label of the x-axis (for plot)
                                yAxisLabel:                 label of the y-axis (for plot)
                                zAxisLabel:                 label of the z-axis (for plot)
                                PathStartScript:            path and name of the start script for calling model function
                                ExeCommandStartScript:      command for calling model function
                                parameter_set:              the complete set of paramters (incl. flags and limits)
                                ExpDataX:                   array containing the experimental x side
                                ExpDataY:                   array containing the experimental y side
                                ExpDataError:               array containing the experimental error of the y side
                                NumberRanges:               number of ranges
                                MinRange:                   array containing lower limit of each range for each file
                                MaxRange:                   array containing upper limit of each range for each file
                                NumberXColumns:             number x-columns
                                NumberYColumns:             number y-columns
                                LengthExpRange:             total number of data points
                                MaxRangeNumber:             max number of ranges over all files
                                NumberExpFiles:             number of exp. data files
                                MaxLength:                  max. length of experimental data
                                MaxColX:                    number of columns concerning to the experimental x side
                                MaxColY:                    number of columns concerning to the experimental y side
                                NumberParameter:            number of model parameter
                                plotflag:                   flag for plotting histograms
                                modelflag:                  flag to use optimized MAGIX packages for a certain model or not
                                MPIFlag:                    mpi flag
                                MAGIXrootDirectory:         MAGIX root directory
                                JobDir:                     job directory
                                SortFortranNum:             flag for sorting chi2 log file
                                ParameterValuesOriginal:    parameter values incl. string parameters
                                FitParameterName:           names of parameters

    output variables:           FinalParameterSet:          final parameter vector
                                FitFunctionAll:             final synthetic spectra
                                Chi2ValuesAll:              final chi^2 function
    """

    # Debug:
    # print ("SpecialAlgorithmSettings = ", SpecialAlgorithmSettings)
    # print ("GeneralAlgorithmSettings = ", GeneralAlgorithmSettings)
    # print ("printflagNum = ", printflagNum)
    # print ("LastAlgorithmNum = ", LastAlgorithmNum)
    # print ("ParamSetCounter = ", ParamSetCounter)
    # print ("chilm = ", chilm)
    # print ("numiter = ", numiter)
    # print ("UltraNestCounter = ", UltraNestCounter)
    # print ("DeterminationChi2 = ", DeterminationChi2)
    # print ("PlotIteration = ", PlotIteration)
    # print ("PlotType = ", PlotType)
    # print ("fit_log = ", fit_log)
    # print ("NumberInputFiles = ", NumberInputFiles)
    # print ("NumberOutputFiles = ", NumberOutputFiles)
    # print ("ParallelizationFlag = ", ParallelizationFlag)
    # print ("JobID = ", JobID)
    # print ("MaxInputLines = ", MaxInputLines)
    # print ("MaxParameter = ", MaxParameter)
    # print ("RenormalizedChi2 = ", RenormalizedChi2)
    # print ("currentpath = ", currentpath)
    # print ("CalculationMethod = ", CalculationMethod)
    # print ("xAxisLabel = ", xAxisLabel)
    # print ("yAxisLabel = ", yAxisLabel)
    # print ("zAxisLabel = ", zAxisLabel)
    # print ("PathStartScript = ", PathStartScript)
    # print ("ExeCommandStartScript = ", ExeCommandStartScript)
    # print ("parameter_set = ", parameter_set)
    # print ("ExpDataX = ", ExpDataX)
    # print ("ExpDataY = ", ExpDataY)
    # print ("ExpDataError = ", ExpDataError)
    # print ("NumberRanges = ", NumberRanges)
    # print ("MinRange = ", MinRange)
    # print ("MaxRange = ", MaxRange)
    # print ("NumberXColumns = ", NumberXColumns)
    # print ("NumberYColumns = ", NumberYColumns)
    # print ("LengthExpRange = ", LengthExpRange)
    # print ("LengthExpRange = ", LengthExpRange)
    # print ("NumberExpFiles = ", NumberExpFiles)
    # print ("MaxLength = ", MaxLength)
    # print ("MaxColX = ", MaxColX)
    # print ("MaxColY = ", MaxColY)
    # print ("NumberParameter = ", NumberParameter)
    # print ("plotflag = ", plotflag)
    # print ("modelflag = ", modelflag)
    # print ("MPIFlag = ", MPIFlag)
    # print ("MAGIXrootDirectory = ", MAGIXrootDirectory)
    # print ("JobDir = ", JobDir)
    # print ("SortFortranNum =" , SortFortranNum)
    # print ("ParameterValuesOriginal = ", ParameterValuesOriginal)
    # print ("FitParameterName = ", FitParameterName)


    ## initialize return parameters
    FinalParameterSet = []
    FitFunctionAll = []
    Chi2ValuesAll = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get special settings
    NumberUltraNestSampler = SpecialAlgorithmSettings[0]                                    ## get number of samplers (only used for UltraNest)
    NumberBurnInIter = SpecialAlgorithmSettings[1]                                          ## get number of iter. for burn-in (only used for UltraNest)
    UltraNestBestSiteCounter = SpecialAlgorithmSettings[2]                                  ## get number of best sites
    UltraNestBackendFileName = SpecialAlgorithmSettings[3]                                  ## get path and name of backend file
    UltraNestDicString = SpecialAlgorithmSettings[4]                                        ## string describing dictionary with parameters
    max_iter = numiter                                                                      ## get max. number of iterations

    # Debug:
    # print ("NumberUltraNestSampler = ", NumberUltraNestSampler)
    # print ("NumberBurnInIter = ", NumberBurnInIter)
    # print ("UltraNestBestSiteCounter = ", UltraNestBestSiteCounter)
    # print ("UltraNestBackendFileName = ", UltraNestBackendFileName)
    # print ("UltraNestDicString = ", UltraNestDicString)
    # print ("max_iter = ", max_iter)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define num_par, min_point and param_up_down from parameter_set
    for i in range(NumberParameter):
        parameter_set[1][i] = float(parameter_set[1][i])
    num_par = int(sum(parameter_set[1]))                                                    ## determine num_par
    min_point = numpy.zeros(num_par, dtype = 'float')                                       ## determine minimum point
    param_up_down = numpy.zeros((num_par, 2), dtype = 'float')                              ## define param_up_down
    param_name = []
    j = (-1)
    for i in range(NumberParameter):


        ## for XCLASS: get current molecule and component
        LocalParamName = FitParameterName[i].strip()
        LocalParamName = LocalParamName.split("[[")
        LocalParamName = LocalParamName[0]
        if (LocalParamName in ["Molecule_Name"]):
            LocalMoleculeName = ParameterValuesOriginal[i].strip()
            LocalComponentIndex = 0
        elif (LocalParamName in ["MolfitFileFormatVersion"]):
            LocalComponentIndex += 1


        ## store limits and name for fit parameters
        parameter_set[1][i] = float(parameter_set[1][i])
        if (parameter_set[1][i] == 1):
            j += 1
            min_point[j] = parameter_set[0][i]
            param_up_down[j, 0] = parameter_set[2][i]
            param_up_down[j, 1] = parameter_set[3][i]
            param_name.append([LocalParamName, LocalMoleculeName, LocalComponentIndex])

    # Debug:
    # print ("param_name = ", param_name)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get current job directory
    LocalJobDir = os.path.dirname(fit_log) + "/"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print what you do
    printflag = True
    if (printflagNum == 1):
        printflag = True
    else:
        printflag = False
    if (printflag):
        print ("\n         temporary files are stored in: " + str(os.environ.get('MAGIXTempDirectory','')).strip() + "job_" + str(JobID).strip() + "/")
        print ("\n\n         Start UltraNest algorithm ..")


    ## modify file names for log-files
    fit_log = fit_log.strip()
    i = fit_log.rfind("/")
    j = fit_log.rfind(".")
    if (j > i):
        fit_log = fit_log[:j] + "__UltraNest__call_" + str(abs(UltraNestCounter)).strip() + ".log"
    else:
        fit_log = fit_log + "__UltraNest__call_" + str(abs(UltraNestCounter)).strip() + ".log"

    # Debug:
    # print ("fit_log = ", fit_log)


    ## open param-log file
    fitlogparam = fit_log.strip() + ".param"                                                ## define file name for param-log file
    paramchannel = open(fitlogparam.strip(), 'w')
    paramchannel.write("\n")
    LogString = "log-file containing the actual values of the parameters used in the UltraNest algorithm:"
    paramchannel.write(LogString + "\n")
    LogString = "-" * len(LogString)
    paramchannel.write(LogString + "\n")
    paramchannel.write("\n")
    paramchannel.flush()


    ## open log file and get current time and date
    fitlogparam = fit_log.strip()                                                           ## define file name for param-log file
    logchannel = open(fitlogparam.strip(), 'w')
    logchannel.write("\n")
    LogString = "log-file for UltraNest algorithm:"
    logchannel.write(LogString + "\n")
    LogString = "-" * len(LogString)
    logchannel.write(LogString + "\n")
    logchannel.write("\n")
    logchannel.write("\n")


    ## get local time
    lt = time.localtime()
    datestring = time.strftime("algorithm starts at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)
    logchannel.write(str(datestring) + "\n")
    logchannel.write("\n")
    logchannel.write("\n")
    logchannel.flush()


    ## open log file and get current time and date
    fitlogChi2 = fit_log.strip() + ".chi2"                                                  ## define file name for param-log file
    Chi2Channel = open(fitlogChi2.strip(), 'w')


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import MAGIX package ModelFunctionCall ..


    ## define name of MAGIX executable
    if (modelflag == "true" or modelflag.lower() == "myxclass"):                            ## .. optimized for myXCLASS model
        StarterExecutable = "Starter__myXCLASS.exe"
    else:                                                                                   ## .. default package
        StarterExecutable = "Starter.exe"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define parameters for class
    ModelCallClass = ModelFunctionCallClass(printflagNum, GeneralAlgorithmSettings, LastAlgorithmNum, DeterminationChi2, PlotIteration, PlotType, \
                                            fit_log, NumberInputFiles, NumberOutputFiles, ParallelizationFlag, JobID, MaxInputLines, MaxParameter, \
                                            RenormalizedChi2, currentpath, CalculationMethod, xAxisLabel, yAxisLabel, zAxisLabel, PathStartScript, \
                                            ExeCommandStartScript, parameter_set, ExpDataX, ExpDataY, ExpDataError, NumberRanges, MinRange, MaxRange, \
                                            NumberXColumns, NumberYColumns, LengthExpRange, MaxRangeNumber, NumberExpFiles, MaxLength, MaxColX, \
                                            MaxColY, NumberParameter, MPIFlag, MAGIXrootDirectory, JobDir, SpecialAlgorithmSettings, StarterExecutable, \
                                            Chi2Channel, logchannel, paramchannel, UltraNestBestSiteCounter, SortFortranNum)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start UltraNest algorithm
    calstatus, AddInfo = UltraNest(printflag, ModelCallClass, logchannel, num_par, max_iter, UltraNestCounter, \
                                   NumberUltraNestSampler, NumberBurnInIter, UltraNestDicString, JobDir, \
                                   LocalJobDir, UltraNestBackendFileName, param_name)
    Bestinput = AddInfo[2]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## close param-log file
    Chi2Channel.close()
    paramchannel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in contents of log file
    Chi2Channel = open(fitlogChi2.strip(), 'r')
    contens = Chi2Channel.readlines()                                                       ## read in contents of input file
    Chi2Channel.close()


    ##--------------Chi2Channel---------------------------------------------------------------------------------------------------------------------------
    ## sort log.chi^2 file
    sortflag = True                                                                         ## define if log file is sorted or not
    if (sortflag):                                                                          ## !!! sorting does not work properly !!!!
        Chi2Contents = []
        for inputline in contens:                                                           ## loop over all lines of the input file
            line = inputline.split()
            Currentline = []
            for count, elements in enumerate(line):
                if (count == 0):
                    elements = float(elements)
                Currentline.append(elements)
            Chi2Contents.append(Currentline)


        ## sort belonging to chi^2 values
        sortchi2 = sorted(Chi2Contents)

        # Debug:
        # print ("Chi2Contents[0] = ", Chi2Contents[0])
        # print ("sortchi2[0] = ", sortchi2[0])


        ## write sorted chi^2 parameter vectors
        Chi2Channel = open(fitlogChi2.strip(), 'w')
        Chi2Channel.write("    Nr.:                   chi**2:  Parameters:\n")
        lastline = ""
        countLine = 0
        for inputline in sortchi2:
            formatline = ""
            for count, elements in enumerate(inputline):
                if (count == 0):
                    formatline = "{:27.15e}        ".format(float(elements))
                else:
                    k = len(elements)
                    if (k < 20):
                        formatelements = " {:20s} ".format(elements)
                    else:
                        formatelements = " " + elements + " "
                    formatline += formatelements
            if (lastline != formatline):
                lastline = formatline
                countLine += 1
                countLineString = "{:8d}".format(countLine)
                Chi2Channel.write(countLineString + " " + str(formatline) + "\n")
        Chi2Channel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if log file is not sorted just write a header line at the beginning of the log file
    else:
        Chi2Channel = open(fitlogChi2.strip(), 'w')
        Chi2Channel.write("    Nr.:                   chi**2:  Parameters:\n")
        for countLine, line in enumerate(contens):
            countLineString = "{:8d}".format(countLine + 1)
            Chi2Channel.write(countLineString + " " + line)
        Chi2Channel.close()


    ## close log file
    logchannel.write("\n")
    lt = time.localtime()
    datestring = time.strftime("algorithm ends at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)
    logchannel.write(str(datestring) + "\n")
    logchannel.write("\n")
    helpString = "-" * 150
    logchannel.write(helpString + "\n")
    logchannel.flush()
    logchannel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create final input files


    ## define base directory, i.e. the path where the final input file(s) are written to
    BaseDir = fit_log.strip()
    i = BaseDir.rfind("/")
    if (i > (-1)):
        BaseDir = BaseDir[:i + 1]
    else:
        BaseDir = ""


    ## extract informations from Bestinput variable
    NameInputFile = ""
    WriteFlag = False
    for LocalBestSite in range(UltraNestBestSiteCounter):
        for line in Bestinput[LocalBestSite]:
            StrippedLine = line.strip()


            ## get name of input file
            if (StrippedLine.startswith("Input-File ")):
                k = StrippedLine.find(":  , file:")
                if (k > (-1)):
                    NameInputFile = StrippedLine[k + 10:].strip()


                    ## modify input file name
                    j = NameInputFile.rfind(".")
                    if (j > (-1)):
                        NameInputFile = NameInputFile[:j] + "__UltraNest__call_" + str(abs(UltraNestCounter))
                    else:
                        NameInputFile = NameInputFile + "__UltraNest__call_" + str(abs(UltraNestCounter))
                    if (UltraNestBestSiteCounter > 1):
                        NameInputFile += "__site_" + str(LocalBestSite + 1)
                    NameInputFile += ".out.input"


                    ## open channel for final input file
                    NameInputFile = BaseDir + "/" + NameInputFile
                    InputFile = open(NameInputFile, 'w')


            ## identify begin of contents of final input file(s)
            elif (StrippedLine.startswith("-start_input-file-")):
                WriteFlag = True


            ## identify end of contents of final input file(s)
            elif (StrippedLine.startswith("-end_input-file-")):
                try:
                    InputFile.close()
                except AttributeError:
                    NameInputFile = ""
                WriteFlag = False


            ## write contents of final input file(s) to current input file
            elif (WriteFlag == True and NameInputFile != ""):
                InputFile.write(line)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get parameters and function values for best sites
    FinalParameter, FitFunctionAll, Chi2ValuesAll = ModelCallClass.GetBestFunctionValues()


    ## define new parameter set
    FinalParameterSet = []
    for LocalBestSite in range(UltraNestBestSiteCounter):
        j = (-1)
        LocalParameterSet = []
        for i in range(NumberParameter):
            if (parameter_set[1][i] == 1):
                j += 1
                LocalParameterSet.append(FinalParameter[LocalBestSite][j])
            else:
                LocalParameterSet.append(parameter_set[0][i])
        FinalParameterSet.append(LocalParameterSet)

    # Debug:
    # print ("NumberParameter = ", NumberParameter)
    # print ("FinalParameter = ", FinalParameter)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define return variables
    calstatus = 0
    return (calstatus, FinalParameterSet, FitFunctionAll, Chi2ValuesAll)
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------

