!*********************************************************************************************************************************************************
!>  Package: Simulated-Annealing algorithm
!>
!>
!>  This module contains the subroutines for Simulated-Annealing algorithm taken from the book "Numerical Recepies Fortran" and modified
!>  Copyright (C) 2009 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - Module SimulatedAnnealing_Variables:      module contains global variables for all Simulated-Annealing subroutines
!>      - Module SimulatedAnnealing_SCIPY:          module containing variables and subroutines for the Simulated-Annealing algorithm (scipy version)
!>      - subroutine getstart_temp:                 (scipy) Find a matching starting temperature and starting parameters vector i.e. find x0 such that
!>                                                  func(x0) = T0.
!>      - subroutine accept_test:                   (scipy) accept new guess or not
!>      - subroutine fast_sa_init:                  (scipy) determine c constant
!>      - subroutine fast_sa_update_guess:          (scipy) determine new parameter vector for fast schedule
!>      - subroutine fast_sa_update_temp:           (scipy) update SA_T and SA_k
!>      - subroutine cauchy_sa_update_guess:        (scipy) determine new parameter vector for cauchy schedule
!>      - subroutine cauchy_sa_update_temp:         (scipy) update SA_T and SA_k
!>      - subroutine boltzmann_sa_update_guess:     (scipy) determine new parameter vector for boltzmann schedule
!>      - subroutine boltzmann_sa_update_temp:      (scipy) update SA_T and SA_k
!>      - subroutine call_scipy:                    (scipy) Minimize a function using simulated annealing.
! ############################################################ Begin: NR version only ####################################################################
!>      - Module SimulatedAnnealing_NR:             module containing global variables and subroutines for the Simulated-Annealing algorithm (NR version)
!>      - subroutine call_amebsa:                   (NR) prepare the call of amebsa
!>      - subroutine signum:                        (NR) determine signum function
!>      - subroutine amebsa:                        (NR) subroutine amebsa include simplex-Simulated Annealing
!>      - subroutine amotsa:                        (NR) extrapolates a point an checks it
! ############################################################ End: NR version only   ####################################################################
!>      - Module Algorithm:                         module contains the main subroutine used to start the different versions of the SA algorithm
!>      - subroutine MainAlg:                       main subroutine which starts the SimulatedAnnealing algorithm
!>
!>
!>
!>
!>  Versions of the program:
!>
!>  Who             When        What
!>
!>  Travis Oliphant 2002        Initial version (scipy version)
!>  Tim Leslie      2006        Bug-fixes (scipy version)
!>  T. Moeller      2010-06-14  Initial version
!>  T. Moeller      2012-01-13  Updated version
!>  T. Moeller      2012-03-06  Translation to fortran, parallelization (scipy version)
!>  T. Moeller      2014-08-20  myXCLASS (model) optimized version
!>  T. Moeller      2014-09-01  modified and restructured for GPU
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: SimulatedAnnealing_Variables
!>
!>         Module contains global variables for all Simulated-Annealing subroutines
!>
!>
!> \author Thomas Moeller
!>
!> \date 2014-08-06
!>
Module SimulatedAnnealing_Variables

    use Model

    implicit none
    integer :: NumberOfReductions                                                           !< number of reduction for temperature
    integer :: MaxNumberOfReheatingPhases                                                   !< max number of reheating phases
    real*8 :: Temperature                                                                   !< temperature for algorithm
    real*8 :: TemperatureReductionKoeff                                                     !< coefficients for temperature reduction

end Module SimulatedAnnealing_Variables
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: SimulatedAnnealing_SCIPY
!>
!>         Module containing variables and subroutines for the Simulated-Annealing algorithm (scipy version)
!>
!>
!> \author Travis Oliphant 2002        Initial version
!>         Tim Leslie      2006        Bug-fixes
!>         T. Moeller      06/03/2012  Translation to fortran, parallelization
!>
Module SimulatedAnnealing_SCIPY

    use FunctionCalling
    use SimulatedAnnealing_Variables

    implicit none
    real*8, parameter :: double_min = tiny(0.d0)                                            !< set smallest number
    real*8, parameter :: double_max = huge(0.d0)                                            !< set biggest number
    integer :: ScheduleSA                                                                   !< schedule for scipy version
    integer :: dwell                                                                        !< The number of times to search the space at each temperature.
    integer :: Ninit                                                                        !<
    integer :: accepted                                                                     !< Number of tests accepted.
    integer :: tests                                                                        !<
    integer :: feval                                                                        !< Number of function evaluations.
    real*8 :: learn_rate                                                                    !< Scale constant for adjusting guesses.
    real*8 :: T0                                                                            !< current temperature
    real*8 :: SA_k                                                                          !<
    real*8 :: SA_T                                                                          !< Final temperature.
    real*8, allocatable, dimension(:) :: x0, x0_orig                                        !< Initial guess.
    real*8, allocatable, dimension(:) :: lower, upper                                       !< lower and upper bound


    !< variables for fast sa
    real*8 :: fast_sa_c                                                                     !<
    real*8 :: fast_sa_m                                                                     !<
    real*8 :: fast_sa_n                                                                     !<
    real*8 :: fast_sa_quench                                                                !<
    real*8 :: boltzmann                                                                     !<

    contains


        !*************************************************************************************************************************************************
        !> subroutine: getstart_temp
        !>
        !> Find a matching starting temperature and starting parameters vector i.e. find x0 such that func(x0) = T0.
        !>
        !>
        !> input variables:     best_state:         A _state object to store the function value and x0 found.
        !>
        !>
        !> output variables:    best_state.x:       The starting parameters vector.
        !>
        !>
        !> \author scipy and Thomas Moeller
        !>
        !> \date 06.03.2012
        !>
        subroutine getstart_temp(nfit, best_state, ma, a, ia, colx, NumFile, MaxL, MaxCol)

            use Variables

            implicit none
            integer :: n                                                                    !< loop variable
            integer :: nfit                                                                 !< number of free parameters
            integer :: NumFile                                                              !< Number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: idum                                                                 !< used for random generator routine
            real*8 :: fmin, fmax, value, val1, dummy                                        !< working variables
            real*8, dimension(nfit + 1) :: best_state                                       !< best parameter vector with corresponding chi^2 value
            real*8, dimension(nfit, nfit) :: x0local                                        !< local copy of param. vector used within OpenMP environment
            real*8, dimension(nfit) :: chi2ValuesVector                                     !< not used here
            real*8, dimension(ma) :: a                                                      !< parameter vector with all parameter
            character(len=10) :: Number1, Number2                                           !< used for number to string conversion
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            fmax = 1.d99
            fmin = 1.d-99
            write(Number2, '(I10)') nfit
            BestSitesParamSet(1, 1) = 1.d99


            !< initialize ran1
            idum = (-1)
            call ran1(dummy, idum)


            !< determine new parameter vectors
            x0local = 0.d0
            Do n = 1, nfit
                x0local(n, :) = x0_orig(:)


                !< original scipy version: take random value within the full parameter range
                !x0local(n, n) = RandomWithLimits(lower(n), upper(n))


                !< modified version
                value = 0.d0
                call ran1(val1, idum)
                value = 1.d0
                if ((0.5d0 - val1) < 0.d0) then
                    value = -1.d0
                endif
                call ran1(val1, idum)
                if (x0local(n, n) == 0.d0) then
                    x0local(n, n) = x0_orig(n) + value * 0.2d0 * val1
                else
                    x0local(n, n) = x0_orig(n) + value * 0.2d0 * val1 * x0_orig(n)
                endif


                !< check, if parameter is within limits
                if (x0local(n, n) < lower(n)) then
                    x0local(n, n) = lower(n)
                elseif (x0local(n, n) > upper(n)) then
                    x0local(n, n) = upper(n)
                endif
            end Do


            !< print what you do ..
            if (printflag) then
                write(Number1, '(I10)') n
                print '(A,11x,"Initialize model function for parameter-sets ..                    ",A,$)', char(13), char(13)
            endif


            !< determine chi2 values for all parameter vectors
            call ModelCalcChiFunctionGeneral(ma, ia, a, nfit, nfit, NumFile, MaxL, MaxCol, x0local, chi2ValuesVector)


            !< redefine temperature
            if (dabs((fmax - fmin) * 1.5d0) < T0) then
                T0 = (fmax - fmin) * 1.5d0
            endif

            ! Debug:
            ! print*,'T0 = ',T0


            !< define output array
            best_state(1:nfit) = BestSitesParamSet(1, 2:)
            best_state(nfit + 1) = BestSitesParamSet(1, 1)


            !< we're done
            return
        end subroutine getstart_temp


        !*************************************************************************************************************************************************
        !> subroutine: accept_test
        !>
        !> accept new guess or not
        !>
        !>
        !> input variables:     dE:                 delta E
        !>
        !>
        !> output variables:    returnValue:        return value
        !>
        !>
        !> \author scipy and Thomas Moeller
        !>
        !> \date 06.03.2012
        !>
        subroutine accept_test(returnValue, dE)

            implicit none
            integer :: returnValue                                                          !< return value
            real*8 :: p                                                                     !< working variables
            real*8 :: dE                                                                    !< delta E

            returnValue = 0
            tests = tests + 1
            if (dE < 0) then
                accepted = accepted + 1
                returnValue = 1
                return
            endif
            p = dexp(-dE * 1.d0 / boltzmann / SA_T)
            if (p > RandomWithLimits(0.d0, 1.d0)) then
                accepted = accepted + 1
                returnValue = 1
                return
            endif


            !< we're done
            return
        end subroutine accept_test


        !*************************************************************************************************************************************************
        !> subroutine: fast_sa_init
        !>
        !> determine c constant
        !>
        !>
        !> input variables:     dE:                 delta E
        !>
        !>
        !> output variables:    returnValue:        return value
        !>
        !>
        !> \author scipy and Thomas Moeller
        !>
        !> \date 06.03.2012
        !>
        !<class fast_sa(base_schedule):
        subroutine fast_sa_init

            implicit none
            fast_sa_c = fast_sa_m * dexp(-fast_sa_n * fast_sa_quench)

            ! Debug:
            ! print*,"fast_sa_c = ", fast_sa_c


            !< we're done
            return
        end subroutine fast_sa_init


        !*************************************************************************************************************************************************
        !> subroutine: fast_sa_update_guess
        !>
        !> determine new parameter vector for fast schedule
        !>
        !>
        !> input variables:     nfit:               number of free parameters
        !>                      x0:                 initial parameter vector
        !>
        !> output variables:    xnew:               new and updated parameter vector
        !>
        !>
        !> \author scipy and Thomas Moeller
        !>
        !> \date 06.03.2012
        !>
        subroutine fast_sa_update_guess(nfit, xnew, x0, ma, a, ia, colx, NumFile, MaxL, MaxCol)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: i                                                                    !< loop variable
            integer :: NumFile                                                              !< Number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            real*8 :: u, y, xc, signum                                                      !< working variables
            real*8, dimension(nfit + 1) :: xnew                                             !< new and updated parameter vector
            real*8, dimension(nfit + 1) :: x0                                               !< initial parameter vector
            real*8, dimension(ma) :: a                                                      !< parameter vector containing all parameter
            real*8, dimension(1, nfit + 1) :: xnewvec                                       !< parameter vector of xnew
            real*8, dimension(1) :: chi2ValuesVector                                        !< here, we have only one param. vector
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            Do i = 1, nfit                                                                  !< loop over all free parameters
                u = RandomWithLimits(0.d0, 1.d0)
                signum = 1.d0
                if ((u - 0.5d0) < 0.d0) then
                    signum = -1.d0
                endif
                y = signum * SA_T * ((1.d0 + 1.d0 / SA_T)**abs(2.d0 * u - 1.d0) - 1.d0)
                xc = y * (upper(i) - lower(i)) * u
                xnew(i) = x0(i) + xc                                                        !< determine new parameter value
                if (xnew(i) > upper(i)) then
                    xnew(i) = x0(i) - xc
                    if (xnew(i) < lower(i)) then
                        xnew(i) = upper(i)
                    endif
                elseif (xnew(i) < lower(i)) then
                    xnew(i) = x0(i) - xc
                    if (xnew(i) > upper(i)) then
                        xnew(i) = lower(i)
                    endif
                endif
            end Do


            !< determine chi2 values for all parameter vectors
            xnewvec(1, :) = xnew
            chi2ValuesVector = 0.d0
            call ModelCalcChiFunctionGeneral(ma, ia, a, 1, nfit, NumFile, MaxL, MaxCol, xnewvec, chi2ValuesVector)
            xnew(nfit + 1) = chi2ValuesVector(1)


            !< we're done
            return
        end subroutine fast_sa_update_guess


        !*************************************************************************************************************************************************
        !> subroutine: fast_sa_update_temp
        !>
        !> update SA_T and SA_k
        !>
        !>
        !> input variables:     - None
        !>
        !>
        !> output variables:    - SA_T, SA_k
        !>
        !>
        !> \author scipy and Thomas Moeller
        !>
        !> \date 06.03.2012
        !>
        subroutine fast_sa_update_temp

            implicit none


            !< compute SA_T and SA_k
            SA_T = T0 * dexp(-fast_sa_c * SA_k**(fast_sa_quench))
            SA_k = SA_k + 1.d0

            ! Debug:
            ! print*,"SA_T, SA_k = ", SA_T, SA_k


            !< we're done
            return
        end subroutine fast_sa_update_temp


        !*************************************************************************************************************************************************
        !> subroutine: cauchy_sa_update_guess
        !>
        !> determine new parameter vector for chauchy schedule
        !>
        !>
        !> input variables:     nfit:               number of free parameters
        !>                      x0:                 initial parameter vector
        !>
        !> output variables:    xnew:               new and updated parameter vector
        !>
        !>
        !> \author scipy and Thomas Moeller
        !>
        !> \date 06.03.2012
        !>
        subroutine cauchy_sa_update_guess(nfit, xnew, x0, ma, a, ia, colx, NumFile, MaxL, MaxCol)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: i                                                                    !< loop variable
            integer :: NumFile                                                              !< Number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            !real*8 :: pi = 3.14159265358979                                                 !< pi
            real*8 :: numbers, xc                                                           !< working variable
            real*8, dimension(nfit + 1) :: xnew                                             !< new and updated parameter vector
            real*8, dimension(nfit + 1) :: x0                                               !< initial parameter vector
            real*8, dimension(ma) :: a                                                      !< parameter vector containing all parameter
            real*8, dimension(1, nfit + 1) :: xnewvec                                       !< parameter vector of xnew
            real*8, dimension(1) :: chi2ValuesVector                                        !< here, we have only one param. vector
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            Do i = 1, nfit                                                                  !< loop over all free parameters
                numbers = RandomWithLimits(-pi/2, pi/2)
                xc = learn_rate * SA_T * dtan(numbers)
                xnew(i) = x0(i) + xc                                                        !< determine new parameter value
                if (xnew(i) > upper(i)) then
                    xnew(i) = x0(i) - xc
                    if (xnew(i) < lower(i)) then
                        xnew(i) = upper(i)
                    endif
                elseif (xnew(i) < lower(i)) then
                    xnew(i) = x0(i) - xc
                    if (xnew(i) > upper(i)) then
                        xnew(i) = lower(i)
                    endif
                endif
            end Do


            !< determine chi2 values for all parameter vectors
            xnewvec(1, :) = xnew
            chi2ValuesVector = 0.d0
            call ModelCalcChiFunctionGeneral(ma, ia, a, 1, nfit, NumFile, MaxL, MaxCol, xnewvec, chi2ValuesVector)
            xnew(nfit + 1) = chi2ValuesVector(1)


            !< we're done
            return
        end subroutine cauchy_sa_update_guess


        !*************************************************************************************************************************************************
        !> subroutine: cauchy_sa_update_temp
        !>
        !> update SA_T and SA_k
        !>
        !>
        !> input variables:     -
        !>
        !>
        !> output variables:    -
        !>
        !>
        !> \author scipy and Thomas Moeller
        !>
        !> \date 06.03.2012
        !>
        subroutine cauchy_sa_update_temp

            implicit none
            SA_T = T0 / (1.d0 + SA_k)
            SA_k = SA_k + 1.d0

            ! Debug:
            ! print*,"SA_T, SA_k = ", SA_T, SA_k


            !< we're done
            return
        end subroutine cauchy_sa_update_temp


        !*************************************************************************************************************************************************
        !> subroutine: boltzmann_sa_update_guess
        !>
        !> determine new parameter vector for boltzmann schedule
        !>
        !>
        !> input variables:     nfit:               number of free parameters
        !>                      x0:                 initial parameter vector
        !>
        !> output variables:    xnew:               new and updated parameter vector
        !>
        !>
        !> \author scipy and Thomas Moeller
        !>
        !> \date 06.03.2012
        !>
        subroutine boltzmann_sa_update_guess(nfit, xnew, x0, ma, a, ia, colx, NumFile, MaxL, MaxCol)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: i                                                                    !< loop variable
            integer :: NumFile                                                              !< Number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            real*8 :: std, xc                                                               !< working variables
            real*8, dimension(nfit + 1) :: xnew                                             !< new and updated parameter vector
            real*8, dimension(nfit + 1) :: x0                                               !< initial parameter vector
            real*8, dimension(ma) :: a                                                      !< parameter vector containing all parameter
            real*8, dimension(1, nfit + 1) :: xnewvec                                       !< parameter vector of xnew
            real*8, dimension(1) :: chi2ValuesVector                                        !< here, we have only one param. vector
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            Do i = 1, nfit                                                                  !< loop over all free parameters
                std = dmin1(dsqrt(SA_T), (upper(i) - lower(i))/3.d0/learn_rate)
                xc = RandomWithLimits(0.d0, 1.d0)
                xnew(i) = x0(i) + xc * std * learn_rate                                     !< determine new parameter value
                if (xnew(i) > upper(i)) then
                    xnew(i) = x0(i) - (xc * std * learn_rate)
                    if (xnew(i) < lower(i)) then
                        xnew(i) = upper(i)
                    endif
                elseif (xnew(i) < lower(i)) then
                    xnew(i) = x0(i) - (xc * std * learn_rate)
                    if (xnew(i) > upper(i)) then
                        xnew(i) = lower(i)
                    endif
                endif
            end Do


            !< determine chi2 values for all parameter vectors
            xnewvec(1, :) = xnew
            chi2ValuesVector = 0.d0
            call ModelCalcChiFunctionGeneral(ma, ia, a, 1, nfit, NumFile, MaxL, MaxCol, xnewvec, chi2ValuesVector)
            xnew(nfit + 1) = chi2ValuesVector(1)


            !< we're done
            return
        end subroutine boltzmann_sa_update_guess


        !*************************************************************************************************************************************************
        !> subroutine: boltzmann_sa_update_temp
        !>
        !> update SA_T and SA_k
        !>
        !>
        !> input variables:     -
        !>
        !>
        !> output variables:    -
        !>
        !>
        !> \author scipy and Thomas Moeller
        !>
        !> \date 06.03.2012
        !>
        subroutine boltzmann_sa_update_temp

            implicit none
            SA_k = SA_k + 1.d0
            SA_T = T0 / dlog(SA_k + 1.d0)


            !< we're done
            return
        end subroutine boltzmann_sa_update_temp


        !*************************************************************************************************************************************************
        !> subroutine: call_scipy
        !>
        !>  Minimize a function using simulated annealing.
        !>
        !>  Schedule is a schedule class implementing the annealing schedule.
        !>  Available ones are 'fast', 'cauchy', 'boltzmann'
        !>
        !>  Parameters
        !>  ----------
        !>  func : callable f(x, *args)
        !>      Function to be optimized.
        !>  x0 : ndarray
        !>      Initial guess.
        !>  args : tuple
        !>      Extra parameters to `func`.
        !>  schedule : base_schedule
        !>      Annealing schedule to use (a class).
        !>  full_output : bool
        !>      Whether to return optional outputs.
        !>  T0 : float
        !>      Initial Temperature (estimated as 1.2 times the largest
        !>      cost-function deviation over random points in the range).
        !>  Tf : float
        !>      Final goal temperature.
        !>  maxeval : int
        !>      Maximum function evaluations.
        !>  maxaccept : int
        !>      Maximum changes to accept.
        !>  maxiter : int
        !>      Maximum cooling iterations.
        !>  learn_rate : float
        !>      Scale constant for adjusting guesses.
        !>  boltzmann : float
        !>      Boltzmann constant in acceptance test
        !>      (increase for less stringent test at each temperature).
        !>  feps : float
        !>      Stopping relative error tolerance for the function value in
        !>      last four coolings.
        !>  quench, m, n : float
        !>      Parameters to alter fast_sa schedule.
        !>  lower, upper : float or ndarray
        !>      Lower and upper bounds on `x`.
        !>  dwell : int
        !>      The number of times to search the space at each temperature.
        !>
        !>  Returns
        !>  -------
        !>  xmin : ndarray
        !>      Point giving smallest value found.
        !>  Jmin : float
        !>      Minimum value of function found.
        !>  SA_T : float
        !>      Final temperature.
        !>  feval : int
        !>      Number of function evaluations.
        !>  iters : int
        !>      Number of cooling iterations.
        !>  accept : int
        !>      Number of tests accepted.
        !>  retval : int
        !>      Flag indicating stopping condition::
        !>
        !>              0 : Points no longer changing
        !>              1 : Cooled to final temperature
        !>              2 : Maximum function evaluations
        !>              3 : Maximum cooling iterations reached
        !>              4 : Maximum accepted query locations reached
        !>              5 : Final point not the minimum amongst encountered points
        !>
        !>  Notes
        !>  -----
        !>  Simulated annealing is a random algorithm which uses no derivative
        !>  information from the function being optimized. In practice it has
        !>  been more useful in discrete optimization than continuous
        !>  optimization, as there are usually better algorithms for continuous
        !>  optimization problems.
        !>
        !>  Some experimentation by trying the difference temperature
        !>  schedules and altering their parameters is likely required to
        !>  obtain good performance.
        !>
        !>  The randomness in the algorithm comes from random sampling in numpy.
        !>  To obtain the same results you can call numpy.random.seed with the
        !>  same seed immediately before calling scipy.optimize.anneal.
        !>
        !>  We give a brief description of how the three temperature schedules
        !>  generate new points and vary their temperature. Temperatures are
        !>  only updated with iterations in the outer loop. The inner loop is
        !>  over loop over xrange(dwell), and new points are generated for
        !>  every iteration in the inner loop. (Though whether the proposed
        !>  new points are accepted is probabilistic.)
        !>
        !>  For readability, let d denote the dimension of the inputs to func.
        !>  Also, let x_old denote the previous state, and SA_k denote the
        !>  iteration number of the outer loop. All other variables not
        !>  defined below are input variables to scipy.optimize.anneal itself.
        !>
        !>  In the 'fast' schedule the updates are ::
        !>
        !>      u ~ Uniform(0, 1, size = d)
        !>      y = sgn(u - 0.5) * SA_T * ((1 + 1 / SA_T)**abs(2 u - 1) - 1.0)
        !>      xc = y * (upper - lower)
        !>      x_new = x_old + xc
        !>
        !>      c = n * exp(-n * quench)
        !>      T_new = T0 * exp(-c * SA_k**quench)
        !>
        !>
        !>  In the 'cauchy' schedule the updates are ::
        !>
        !>      u ~ Uniform(-pi/2, pi/2, size = d)
        !>      xc = learn_rate * SA_T * tan(u)
        !>      x_new = x_old + xc
        !>
        !>      T_new = T0 / (1 + SA_k)
        !>
        !>  In the 'boltzmann' schedule the updates are ::
        !>
        !>      std = minimum( sqrt(SA_T) * ones(d), (upper-lower) / (3*learn_rate) )
        !>      y ~ Normal(0, std, size = d)
        !>      x_new = x_old + learn_rate * y
        !>
        !>      T_new = T0 / log(1 + SA_k)
        !>
        !>
        !> input variables:     mp:                 ndim+1
        !>                      np:                 number of free parameters
        !>                      ndim:               number of free parameters
        !>                      nfit:               number of free parameters
        !>                      ftol:               limit of chi^2
        !>                      iiter:              max number of iterations
        !>                      ma:                 total number of parameters
        !>                      a:                  array containing the parameter set
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      colx:               number of columns in experimental x data
        !>                      NumFile:            Number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      PlotIteration:      flag for plotting the model function for each step
        !>                      PlotType  :         get type of plot
        !>                      xAxisLabel:         label of the x-axis (for plot)
        !>                      yAxisLabel:         label of the y-axis (for plot)
        !>                      zAxisLabel:         label of the z-axis (for plot)
        !>                      fitlog:             path for log files
        !>
        !> output variables:    None
        !>
        !>
        !> \author scipy and Thomas Moeller
        !>
        !> \date 06.03.2012
        !>
        subroutine call_scipy(mp, np, ndim, nfit, ftol, iiter, ma, a, ia, colx, NumFile, MaxL, MaxCol, PlotIteration, PlotType, xAxisLabel, yAxisLabel, &
                              zAxisLabel, fitlog, MaxNumberOfReheatingPhases)

            use Variables

            implicit none
            integer :: i, j, m, n, k, iters                                                 !< loop variables
            integer :: mp                                                                   !<
            integer :: ndim                                                                 !< number of free parameters
            integer :: np                                                                   !<
            integer :: mm                                                                   !< working variable
            integer :: MaxNumberOfReheatingPhases                                           !< number of reheating phases
            integer :: NumInputFile_index, i_index, j_index                                 !< working variables
            integer :: CurrentIter                                                          !< working variables
            integer :: iiter, maxiter                                                       !< max number of iterations
            integer :: nfit                                                                 !< number of free parameters
            integer :: NumFile                                                              !< Number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: NumInputFiles                                                        !< needed for loop over input files
            integer :: PlotIteration                                                        !< flag for plotting the model function for each step
            integer :: PlotType                                                             !< get type of plot
            integer :: HeatingCycles_Counter                                                !< counts the number of reheating cycles
            integer :: allocstatus, deallocstatus                                           !< variables for allocation/deallocation
            integer,dimension(nfit) :: EffectiveArray                                       !< used for parallelization
            real*8 :: dummy                                                                 !< dummy
            real*8 :: dE                                                                    !< delta E
            real*8 :: Temperature                                                           !< Temperature
            real*8 :: ftol, feps                                                            !< limit of chi^2
            real*8 :: fval                                                                  !< working variables
            real*8 :: chisq                                                                 !< value of chisq
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(nfit + 1) :: best_state, last_state, current_state            !< best site and corresponding chi^2 value and copy
            real*8, dimension(colx) :: posdatexp                                            !< working variable
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=5196) :: ListParamFormated                                        !< working variable
            character(len=8192) :: fitlog                                                   !< path for log files
            character(len=100) :: HelpString                                                !< working variable
            character(len=10) :: Number1, Number2                                           !< used for number to string conversion
            character(len=20) :: LongNumber                                                 !< working variable
            character(len=25) :: LongNumber1, LongNumber2                                   !< working variables
            logical,dimension(ma) :: ia                                                     !< flags for including/excluding parameter in the fit
            logical :: exitflag                                                             !< working variable
            logical :: InitPlotFlag                                                         !< flag for saving plot
            logical :: IntegerTrue                                                          !< flag for indicating integer numbers
            logical :: errorflag                                                            !< flag for including errors

            ! Debug:
            !    print*,' '
            !    print*,'	 Parameters:'
            !    print*,' '
            !    print*,'	 nfit = ',nfit
            !    Do i=1,mp
            !        print*,'	 p(i) = ',i,p(i,:)
            !    end Do
            !    print*,'	 ##################################################################'
            !   stop


            !< print what you do
            if (ScheduleSA == 1) then
                HelpString = "Fast"
            elseif (ScheduleSA == 2) then
                HelpString = "Cauchy"
            else
                HelpString = "Boltzmann"
            endif
            write(logchannel,'(11x,"Using scipy version with ",A," schedule!")') trim(adjustl(HelpString))
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')
            if (printflag) then
                print '(11x,"Using scipy version with ",A," schedule!")', trim(adjustl(HelpString))
                print '(" ")'
                print '(" ")'
            endif


            !< initialize ran1
            idum = (-1)
            call ran1(dummy, idum)


            !< print what you do
            if (printflag) then
                print '(11x,"Iteration:",20x,"chi^2:",8x,"Temperature:",5x,"Parameter:")'
            endif
            write(logchannel,'(11x,"Iteration:",20x,"chi^2:",8x,"Temperature:",5x,"Parameter:")')


            !< set SA internal parameters
            Ninit = 50
            accepted = 0
            tests = 0
            feval = 0
            SA_k = 1.d0
            SA_T = 0.d0
            feps = ftol
            maxiter = iiter


            !< set initial values for schedule fast
            if (ScheduleSA == 1) then
                fast_sa_c = 0.d0
                fast_sa_m = 1.d0
                fast_sa_n = 1.d0
                fast_sa_quench = 1.d0
                call fast_sa_init
            endif
            boltzmann = 1.d0


            !< deallocate/allocate memory for some working variables, clear contents of the variables and print error message if necessary
            if (allocated(x0)) then
                deallocate(x0, x0_orig, lower, upper, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine call_scipy:")')
                    write(logchannel,'(2x,"Can not deallocate variables x0 etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine call_scipy:")'
                    print '(2x,"Can not deallocate variables x0 etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif
            allocate(x0(nfit), x0_orig(nfit), lower(nfit), upper(nfit), stat = allocstatus)
            if (allocstatus /= 0) then
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine call_scipy:")')
                write(logchannel,'(2x,"Can not allocate variables x0 etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine call_scipy:")'
                print '(2x,"Can not allocate variables x0 etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            x0 = 0.d0
            x0_orig = 0.d0
            lower = 0.d0
            upper = 0.d0


            !< determine x0, lower and upper array
            m = 0
            Do i = 1, ma
                if (ia(i)) then
                    m = m + 1
                    x0(m) = a(i)
                    lower(m) = paramset(3, i)
                    upper(m) = paramset(4, i)
                    EffectiveArray(m) = i
                endif
            end Do
            x0_orig = x0


            !< initialize
            Temperature = T0
            call getstart_temp(nfit, best_state, ma, a, ia, colx, NumFile, MaxL, MaxCol)
            x0 = best_state(1:nfit)
            fval = best_state(nfit + 1)

            ! Debug:
            ! print*,'fval, x0, T0 = ',fval, x0, T0


            chisq = fval
            last_state(:) = best_state(:)
            SA_T = T0
            feval = feval + 1
            CurrentIter = 0
            HeatingCycles_Counter = 0
            Do iters = 1, maxiter                                                           !< loop over all iterations
                CurrentIter = CurrentIter + 1
                Do n = 1, dwell                                                             !< loop over


                    !< call amebsa subroutine
                    if (printflag) then
                        write(Number1, '(I10)') n
                        write(Number2, '(I10)') dwell
                        write(LongNumber, '(ES20.10)') SA_T
                        print '(A,11x,"Calculate model function (",A,"/",A,") for temperature: ",A," ..   ",A1,$ )', char(13), &
                                trim(adjustl(Number1)), trim(adjustl(Number2)), trim(adjustl(LongNumber)), char(13)
                    endif


                    !< update parameter vector
                    if (ScheduleSA == 1) then
                        call fast_sa_update_guess(nfit, current_state, last_state, ma, a, ia, colx, NumFile, MaxL, MaxCol)
                    elseif (ScheduleSA == 2) then
                        call cauchy_sa_update_guess(nfit, current_state, last_state, ma, a, ia, colx, NumFile, MaxL, MaxCol)
                    elseif (ScheduleSA == 3) then
                        call boltzmann_sa_update_guess(nfit, current_state, last_state, ma, a, ia, colx, NumFile, MaxL, MaxCol)
                    endif
                    feval = feval + 1


                    !< determine delta E
                    dE = (current_state(nfit + 1) - last_state(nfit + 1))
                    call accept_test(mm, dE)
                    if (mm == 1) then
                        last_state(:) = current_state(:)
                        if (last_state(nfit + 1) < best_state(nfit + 1)) then
                            best_state(:) = last_state(:)
                            chisq = BestSitesParamSet(1, 1)
                            k = 0
                            Do i = 1, ma
                                if (ia(i)) then
                                    k = k + 1
                                    a(i) = BestSitesParamSet(1, k + 1)
                                endif
                            end Do


                            !< check if the values of the current parameters are within the given upper and lower limits
                            call parameterlimitcheck(errorflag, ma, a)
                        endif
                    endif
                end Do


                !< build list with fit parameters
                k = 0
                ListParamFormated = ""
                Do j = 1, parameternumber
                    if (ia(j)) then
                        k = k + 1
                        a(j) = BestSitesParamSet(1, k + 1)
                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                        if (IntegerTrue) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(j))
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) int(a(j))
                            endif
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(j)
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) a(j)
                            endif
                        endif
                        if (k == 1) then
                            ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                        else
                            ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                        endif
                    endif
                end Do


                !< print status of iteration process ..
                if (printflag) then
                    print '(11x,I10,ES26.15,ES20.10,5x,A)',iters, chisq, SA_T, trim(adjustl(ListParamFormated))
                endif
                write(paramchannel,'("  ")')
                write(paramchannel,'("  ")')
                write(paramchannel,'(123("*"))')
                write(paramchannel,'("Iteration: ",I5,",  chi^2 = ",ES25.15,",   Temp = ",ES20.10)') iters, chisq, SA_T
                write(logchannel,'(11x,I10,ES26.15,ES20.10,5x,A)') iters, chisq, SA_T, trim(adjustl(ListParamFormated))


                !< write actual parameters to files
                write(paramchannel,'("  ")')
                write(paramchannel,'("  ")')
                write(paramchannel,'("Parameters: ",A)') trim(adjustl(ListParamFormated))
                write(paramchannel,'(123("-"))')
                write(paramchannel,'("  ")')


                !< save current experimental x point of the first experimental file to variable posdatexp
                posdatexp(1:colx) = 0.d0


                !< call subroutine to write current values of the parameter to file
                write(paramchannel,'("-",61(" -"))')
                Do NumInputFiles = 1, NumberInputFiles
                    write(paramchannel,'("Input-File ",I5,":  , file: ",A)') NumInputFiles, trim(adjustl(FitFktInput(NumInputFiles)))
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("-start_input-file",106("-"))')
                    call WriteParameter(paramchannel, .true., colx, posdatexp, ma, a, NumInputFiles)
                    write(paramchannel,'("-end_input-file",108("-"))')
                end Do


                !< plot experimental data, model function, and chi**2
                if (PlotIteration == 0) then
                    if (iters == 1) then
                        InitPlotFlag = .true.
                    else
                        InitPlotFlag = .false.
                    endif
                    call PlotFitFunction(InitPlotFlag, xAxisLabel, yAxisLabel, zAxisLabel)
                endif


                !< print stop criteria
                if (chisq < ftol) then
                    write(paramchannel,'("  ")')
                    write(paramchannel,'(123("="))')
                    write(paramchannel,'("  ")')
                    exitflag = .true.
                    write(LongNumber1,'(ES25.15)') chisq
                    write(LongNumber2,'(ES25.15)') ftol
                    if (printflag) then
                        print '(11x,100(" "))'
                        print '(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)', trim(adjustl(LongNumber1)), &
                                                                                                        trim(adjustl(LongNumber2))
                    endif
                    write(logchannel,'("  ")')
                    write(logchannel,'(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)') trim(adjustl(LongNumber1)), &
                                                                                                        trim(adjustl(LongNumber2))
                    exit
                endif


                !< update temperature
                if (ScheduleSA == 1) then
                    call fast_sa_update_temp
                elseif (ScheduleSA == 2) then
                    call cauchy_sa_update_temp
                elseif (ScheduleSA == 3) then
                    call boltzmann_sa_update_temp
                endif


                !< is temperature below limit?
                if (SA_T < 1.d-5) then                                                      !< temperature dropped below 10^(-5)
                    if (HeatingCycles_Counter == MaxNumberOfReheatingPhases) then
                        write(paramchannel,'("  ")')
                        write(paramchannel,'(123("="))')
                        write(paramchannel,'("  ")')
                        exitflag = .true.
                        if (printflag) then
                            print '(11x,100(" "))'
                            print '(11x,"Iteration stopped. Number of reheating phases is equal to ",I4,".")', MaxNumberOfReheatingPhases
                        endif
                        write(logchannel,'("  ")')
                        write(logchannel,'(11x,"Iteration stopped. Number of reheating phases is equal to ",I4,".")') MaxNumberOfReheatingPhases
                        exit
                    else
                        exitflag = .false.
                        HeatingCycles_Counter = HeatingCycles_Counter + 1
                        T0 = Temperature
                        SA_T = T0
                        SA_k = 1.d0

                        ! Debug:
                        ! print*,'Temperature reset! ---------------------------------------'

                        if (printflag) then
                            print '(A1,11x,"Heat again.",A1,$)', char(13), char(13)
                        endif
                    endif
                endif
            end Do                                                                          !< end loop over iteration


            !< get best result
            chisq = BestSitesParamSet(1, 1)
            k = 0
            Do i = 1, ma
                if (ia(i)) then
                    k = k + 1
                    a(i) = BestSitesParamSet(1, k + 1)
                endif
            end Do


            !< print stop criteria
            if (chisq > ftol .and. CurrentIter >= maxiter) then
                write(paramchannel,'("  ")')
                write(paramchannel,'(123("="))')
                write(paramchannel,'("  ")')
                exitflag = .true.
                if (printflag) then
                    print '(11x,100(" "))'
                    print '(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ",I6)',maxiter
                endif
                write(logchannel,'("  ")')
                write(logchannel,'(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ",I6)') maxiter
            endif


            !< clear screen massage
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
            endif


            !< we're done
            return
        end subroutine call_scipy
end Module SimulatedAnnealing_SCIPY


! ############################################################ Begin: NR version only ####################################################################
!*********************************************************************************************************************************************************
!> Module: SimulatedAnnealing_NR
!>
!>         Module containing global variables and subroutines for the Simulated-Annealing algorithm (NR version)
!>
!>
!> \author Thomas Moeller
!>
!> \date 2010-06-09
!>
Module SimulatedAnnealing_NR

    use SimulatedAnnealing_Variables

    implicit none

    contains


        !*************************************************************************************************************************************************
        !> subroutine: call_amebsa
        !>
        !> prepare the call of amebsa
        !>
        !>
        !> input variables:     mp:                 ndim+1
        !>                      np:                 number of free parameters
        !>                      ndim:               number of free parameters
        !>                      nfit:               number of free parameters
        !>                      ftol:               limit of chi^2
        !>                      iiter:              max number of iterations
        !>                      ma:                 total number of parameters
        !>                      a:                  array containing the parameter set
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      colx:               number of columns in experimental x data
        !>                      NumFile:            Number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      PlotIteration:      flag for plotting the model function for each step
        !>                      PlotType  :         get type of plot
        !>                      xAxisLabel:         label of the x-axis (for plot)
        !>                      yAxisLabel:         label of the y-axis (for plot)
        !>                      zAxisLabel:         label of the z-axis (for plot)
        !>                      fitlog:             path for log files
        !>
        !> output variables:    None
        !>
        !>
        !> \author Numercial Recipies and Thomas Moeller
        !>
        !> \date 18.09.2009
        !>
        subroutine call_amebsa(mp, np, ndim, nfit, ftol, iiter, ma, a, ia, colx, NumFile, MaxL, MaxCol, PlotIteration, PlotType, xAxisLabel, yAxisLabel, &
                               zAxisLabel, fitlog)

            use Variables
            use FunctionCalling

            implicit none
            integer :: i, j, k                                                              !< loop variables
            integer :: mp                                                                   !<
            integer :: ndim                                                                 !< number of free parameters
            integer :: np                                                                   !<
            integer :: NumInputFile_index, i_index, j_index                                 !< working variables
            integer :: iter                                                                 !< working variables
            integer :: CurrentIter, jiter                                                   !< working variables
            integer :: iiter                                                                !< max number of iterations
            integer :: MaxIteration, LastIteration, iter_best                               !< working variables
            integer :: nfit                                                                 !< number of free parameters
            integer :: NumFile                                                              !< Number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: NumInputFiles                                                        !< needed for loop over input files
            integer :: PlotIteration                                                        !< flag for plotting the model function for each step
            integer :: PlotType                                                             !< get type of plot
            integer :: HeatingCycles_Counter                                                !< counts the number of reheating cycles
            integer,dimension(nfit) :: EffectiveArray                                       !< used for parallelization
            real*8 :: dummy                                                                 !< dummy
            real*8 :: ftol                                                                  !< limit of chi^2
            real*8 :: temptr, yb, ybb, value, val1, val2                                    !< working variables
            real*8 :: chisq, chisq_best                                                     !< value of chisq
            real*8, dimension(mp) :: y                                                      !< vector y(1:ndim+1), whose components must be pre-initialized
                                                                                            !<  to the values of funk evaluated at the ndim+1 vertices (rows)
                                                                                            !<  of p
            real*8, dimension(np) :: pb, pb_best                                            !< working variable
            real*8, dimension(mp,np) :: p                                                   !< input matrix p(1..ndim+1,1..ndim) has ndim+1 rows, each an
                                                                                            !<  ndim-dimensional vector which is a vertex of the
                                                                                            !<  starting simplex
            real*8, dimension(ma) :: a, a_orig                                              !< array containing the parameter set
            real*8, dimension(nfit) :: ptry, ptry_orig, param                               !< dummy
            real*8, dimension(mp) :: chi2ValuesVector                                       !< chi2 value vector
            real*8, dimension(colx) :: posdatexp                                            !< working variable
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=5196) :: ListParamFormated                                        !< working variable
            character(len=8192) :: fitlog                                                   !< path for log files
            character(len=100) :: HelpString                                                !< working variable
            character(len=10) :: Number2                                                    !< used for number to string conversion
            character(len=25) :: LongNumber1, LongNumber2                                   !< working variables
            logical,dimension(ma) :: ia                                                     !< flags for including/excluding parameter in the fit
            logical :: exitflag                                                             !< working variable
            logical :: InitPlotFlag                                                         !< flag for saving plot
            logical :: IntegerTrue                                                          !< flag for indicating integer numbers
            logical :: errorflag                                                            !< flag for including errors

            ! Debug:
            !    print*,' '
            !    print*,'	 Parameters:'
            !    print*,' '
            !    print*,'	 nfit = ',nfit
            !    Do i=1,mp
            !        print*,'	 p(i) = ',i,p(i,:)
            !    end Do
            !    print*,'	 ##################################################################'
            !   stop


            !< initialize ran1
            idum = (-1)
            call ran1(dummy, idum)


            !< print some comments to screen
            if (printflag) then
                print '(11x,"Using Numerical Recepies (NR) version!")'
                print '(" ")'
                print '(" ")'
            endif
            write(logchannel,'(11x,"Using Numerical Recepies (NR) version!")')
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')


            !< print what you do
            if (printflag) then
                print '(11x,"Iteration:",20x,"chi^2:",8x,"Temperature:",5x,"Parameter:")'
            endif
            write(logchannel,'(11x,"Iteration:",20x,"chi^2:",8x,"Temperature:",5x,"Parameter:")')


            !< start iteration
            temptr = Temperature
            exitflag = .false.
            CurrentIter = 0
            MaxIteration = iiter
            LastIteration = 0
            chisq = 1.d30
            chisq_best = 1.d99
            pb_best = 0.d0
            iter_best = 0
            a_orig = a
            yb = 1.d30
            ybb = 1.d20
            HeatingCycles_Counter = 0
            Do While (.not.exitflag)


                !< determine ptry array and EffectiveArray array for parallelization
                EffectiveArray = 0
                ptry = 0.d0
                k = 0
                Do i = 1, ma
                    if (ia(i)) then
                        k = k + 1
                        ptry(k) = a(i)
                        EffectiveArray(k) = i
                    endif
                end Do
                if (printflag) then
                    print '(A,120(" "),A,$)',char(13),char(13)
                    print '(11x,"Initialize model function ..",20(" "),A1,$ )',char(13)
                endif

                ! Debug:
                ! print*,'ParallelizationFlag = ',ParallelizationFlag


                !< determine p and y array
                k = 0
                ptry_orig = ptry                                                            !< save original parameter set
                p = 0.d0
                p(1,:) = ptry                                                               !< first column represent starting parameter vector
                Do k = 1, nfit
                    i = EffectiveArray(k)
                    ptry(:) = ptry_orig(:)
                    value = 0.d0
                    call ran1(val1, idum)
                    call signum(val2, 0.5d0 - val1)


                    !< old statement
                    !call ran1(val1, idum)
                    !if (ptry_orig(k) == 0.d0) then
                    !    ptry(k) = ptry_orig(k) + val2 * 0.2d0 * val1
                    !else
                    !    ptry(k) = ptry_orig(k) + val2 * 0.2d0 * val1 * ptry_orig(k)
                    !endif


                    !< determine new parameter value
                    !value = dmin1(dsqrt(dabs(temptr)), (paramset(4,i) - paramset(3,i))/3.d0/TemperatureReductionKoeff)
                    !ptry(k) = ptry_orig(k) + val2 * val1 * value * TemperatureReductionKoeff


                    !< new
                    if (ptry_orig(k) == 0.d0) then
                        ptry(k) = ptry_orig(k) + val2 * temptr
                    else
                        ptry(k) = ptry_orig(k) + val2 * temptr * ptry_orig(k)
                    endif


                    !< check if parameters are within limits
                    if (ptry(k) < paramset(3,i) .or. ptry(k) > paramset(4,i)) then
                        ptry(k) = ptry_orig(k)
                    endif
                    p(k + 1,:) = ptry(:)
                end Do

                ! Debug:
                ! print*,'ptry(:) = ',ptry(:)


                if (printflag) then
                    print '(A,120(" "),A,$)',char(13),char(13)
                    write(Number2,'(I10)') nfit
                endif


                !< determine chi2 values for all parameter vectors
                if (printflag) then
                    print '(A,11x,"Initialize model function for parameter-set ..                    ",A,$)', char(13), char(13)
                endif
                chi2ValuesVector = 0.d0
                call ModelCalcChiFunctionGeneral(ma, ia, a, nfit + 1, nfit, NumFile, MaxL, MaxCol, p, chi2ValuesVector)
                y(:) = chi2ValuesVector(:)
                yb = BestSitesParamSet(1, 1)
                param(:) = BestSitesParamSet(1, 2:)
                ptry(:) = param(:)                                                          !< ptry contains best parameter set
                pb = ptry
                Do jiter = 1, NumberOfReductions
                    iter = 1 + 0 * iiter
                    errorflag = .false.
                    call amebsa(p, y, mp, np, ndim, pb, yb, ftol, iter, temptr, ma, a, ia, colx, NumFile, MaxL, MaxCol)

                    ! Debug:
                    ! print*,'>',temptr, jiter, pb,yb
                    ! print*,'>>>',temptr,yb,ybb

                    yb = BestSitesParamSet(1, 1)
                    ybb = yb
                    chisq = yb
                    pb = BestSitesParamSet(1, 2:)
                    CurrentIter = CurrentIter + 1
                    k = 0
                    Do i = 1, ma
                        if (ia(i)) then
                            k = k + 1
                            a(i) = pb(k)
                        endif
                    end Do


                    !< check if the values of the current parameters are within the given upper and lower limits
                    call parameterlimitcheck(errorflag, ma, a)


                    !< build list with fit parameters
                    k = 0
                    ListParamFormated = ""
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                            if (IntegerTrue) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(j))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(a(j))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(j)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) a(j)
                                endif
                            endif
                            if (k == 1) then
                                ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                            else
                                ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                            endif
                        endif
                    end Do


                    !< print status of iteration process ..
                    if (printflag) then
                        print '(11x,I10,ES26.15,ES20.10,5x,A)',CurrentIter, chisq, temptr, trim(adjustl(ListParamFormated))
                    endif
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("  ")')
                    write(paramchannel,'(123("*"))')
                    write(paramchannel,'("Iteration: ",I5,",  chi^2 = ",ES25.15,",   temptr = ",ES20.10)') CurrentIter, chisq, temptr
                    write(logchannel,'(11x,I10,ES26.15,ES20.10,5x,A)') CurrentIter, chisq, temptr, trim(adjustl(ListParamFormated))


                    !< write actual parameters to files
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("Parameters: ",A)') trim(adjustl(ListParamFormated))
                    write(paramchannel,'(123("-"))')
                    write(paramchannel,'("  ")')


                    !< save current experimental x point of the first experimental file to variable posdatexp
                    posdatexp(1:colx) = 0.d0


                    !< call subroutine to write current values of the parameter to file
                    write(paramchannel,'("-",61(" -"))')
                    Do NumInputFiles = 1, NumberInputFiles
                        write(paramchannel,'("Input-File ",I5,":  , file: ",A)') NumInputFiles, trim(adjustl(FitFktInput(NumInputFiles)))
                        write(paramchannel,'("  ")')
                        write(paramchannel,'("-start_input-file",106("-"))')
                        call WriteParameter(paramchannel, .true., colx, posdatexp, ma, a, NumInputFiles)
                        write(paramchannel,'("-end_input-file",108("-"))')
                    end Do


                    !< plot experimental data, model function, and chi**2
                    if (PlotIteration == 0) then
                        if (CurrentIter == 1) then
                            InitPlotFlag = .true.
                        else
                            InitPlotFlag = .false.
                        endif
                        call PlotFitFunction(InitPlotFlag, xAxisLabel, yAxisLabel, zAxisLabel)
                    endif
                    temptr = temptr * TemperatureReductionKoeff
                    if (CurrentIter >= MaxIteration .or. chisq < ftol) then
                        exit
                    endif
                end Do
                if (chisq < ftol) then
                    write(paramchannel,'("  ")')
                    write(paramchannel,'(123("="))')
                    write(paramchannel,'("  ")')
                    exitflag = .true.
                    write(LongNumber1,'(ES25.15)') chisq
                    write(LongNumber2,'(ES25.15)') ftol
                    if (printflag) then
                        print '(11x,100(" "))'
                        print '(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)', trim(adjustl(LongNumber1)), &
                                                                                                        trim(adjustl(LongNumber2))
                    endif
                    write(logchannel,'("  ")')
                    write(logchannel,'(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)') trim(adjustl(LongNumber1)), &
                                                                                                        trim(adjustl(LongNumber2))

                elseif (CurrentIter >= MaxIteration) then
                    write(paramchannel,'("  ")')
                    write(paramchannel,'(123("="))')
                    write(paramchannel,'("  ")')
                    exitflag = .true.
                    if (printflag) then
                        print '(11x,100(" "))'
                        print '(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ",I6)',MaxIteration
                    endif
                    write(logchannel,'("  ")')
                    write(logchannel,'(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ",I6)') MaxIteration

                else
                    !if (temptr < 1.d-5) then                                                !< temperature dropped below 10^(-5)
                    HeatingCycles_Counter = HeatingCycles_Counter + 1
                    if (HeatingCycles_Counter == MaxNumberOfReheatingPhases) then
                        write(paramchannel,'("  ")')
                        write(paramchannel,'(123("="))')
                        write(paramchannel,'("  ")')
                        exitflag = .true.
                        if (printflag) then
                            print '(11x,100(" "))'
                            print '(11x,"Iteration stopped. Number of reheating phases is equal to ",I4,".")', MaxNumberOfReheatingPhases
                        endif
                        write(logchannel,'("  ")')
                        write(logchannel,'(11x,"Iteration stopped. Number of reheating phases is equal to ",I4,".")') MaxNumberOfReheatingPhases
                    else
                        exitflag = .false.
                        temptr = Temperature
                        BestSitesParamSet(1, :) = 0.d0
                        BestSitesParamSet(1, 1) = 1.d99
                        yb = 1.d30
                        ybb = 1.d20

                        ! Debug:
                        ! print*,'Temperature reset! ---------------------------------------'

                        if (printflag) then
                            print '(A1,11x,"Heat again.",A1,$)', char(13), char(13)
                        endif
                    endif
                    !endif
                endif


                !< save best result
                if (chisq_best > chisq) then
                    chisq_best = chisq
                    pb_best = pb
                    iter_best = CurrentIter

                elseif (CurrentIter > 1.and.CurrentIter < MaxIteration) then
                    chisq = chisq_best
                    yb = chisq_best
                    pb = pb_best

                    ! Debug:
                    ! print*,'>',temptr,CurrentIter,pb
                endif


                !< is max. iteration reached
                if (CurrentIter >= MaxIteration) then
                    exitflag = .true.
                endif
                LastIteration = CurrentIter
            end Do

            ! Debug:
            !    print*,' '
            !    print*,'yb = ',yb
            !    print*,'pb = ',pb(:)
            !    print*,'a  = ',a
            !    print*,'##################################################################'
            !    stop

            !< if best result was not reached in the last iteration cycle than copy best result to working variables
            if (LastIteration /= iter_best) then


                !< copy values of the model function
                chisq = chisq_best
                pb = pb_best


                !< save final result to array a
                k = 0
                Do i = 1, ma
                    if (ia(i)) then
                        k = k + 1
                        a(i) = pb(k)
                    endif
                end Do


                !< build list with fit parameters
                k = 0
                ListParamFormated = ""
                Do j = 1, parameternumber
                    if (ia(j)) then
                        k = k + 1
                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                        if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                            .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(pb(k))
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) int(pb(k))
                            endif
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) pb(k)
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) pb(k)
                            endif
                        endif
                        if (k == 1) then
                            ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                        else
                            ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                        endif
                    endif
                end Do


                !< print best results to screen and to file
                if (printflag) then
                    print '(" ")'
                    print '(" ")'
                    print '(11x,"Best results:")'
                    print '(13x,"counter = ",I4,", chi**2 = ",ES20.10,",  Parameterset = ",A)', iter_best, chisq, trim(adjustl(ListParamFormated))
                endif
                write(logchannel,'(" ")')
                write(logchannel,'(" ")')
                write(logchannel,'(11x,"Best results:")')
                write(logchannel,'(13x,"counter = ",I4,", chi**2 = ",ES20.10,",  Parameterset = ",A)') iter_best, chisq, trim(adjustl(ListParamFormated))
            endif


            !< clear screen massage
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
            endif


            return
        end subroutine call_amebsa


        !*************************************************************************************************************************************************
        !> subroutine: signum
        !>
        !> determine signum function
        !>
        !>
        !> input variables:     x:                  argument
        !>
        !> output variables:    y:                  (-1) or +1 on exit depending on argument x
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 27.05.2010
        !>
        subroutine signum(y,x)
            implicit none
            real*8 :: x
            real*8 :: y

            y = 0.d0
            if (x >= 0.d0) then
                y = +1.d0
            else
                y = -1.d0
            endif
            return
        end subroutine signum


        !*************************************************************************************************************************************************
        !> subroutine: amebsa
        !>
        !> subroutine amebsa include simplex-Simulated Annealing
        !>
        !>
        !> input variables:     p:                  matrix n+1 by n contains the initial simplex
        !>                      y:                  array, n+1, contains the cost function for each point in simplex
        !>                                          func function the cost function
        !>                      mp:                 ndim+1
        !>                      np:                 number of free parameters
        !>                      ndim:               number of free parameters
        !>                      pb:                 good parameter set
        !>                      yb:                 corresponding chi^2 value
        !>                      ftol:               real the desired tolerance
        !>                      iter:               integer number of iterations done
        !>                      temptr:             the temperature
        !>                      ma:                 total number of parameters
        !>                      a:                  array containing the parameter set
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      colx:               number of columns in experimental x data
        !>                      NumFile:            Number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>
        !> output variables:    None
        !>
        !>
        !>
        !> \author Numercial Recipies and Thomas Moeller
        !>
        !> \date 18.09.2009
        !>
        subroutine amebsa(p, y, mp, np, ndim, pb, yb, ftol, iter, temptr, ma, a, ia, colx, NumFile, MaxL, MaxCol)
            !<
            !<   uses ran1, funk
            !<   Multidimensional minimization of the function funk(x) where x(1:ndim) is a vector in
            !<   dimensions, by simulated annealing combined with the downhill simplex method of
            !<   Nelder and Mead. The input matrix p(1..ndim+1,1..ndim) has ndim+1 rows, each an
            !<   ndim-dimensional vector which is a vertex of the starting simplex. Also input is the vector
            !<   y(1:ndim+1), whose components must be pre-initialized to the values of funk evaluated at
            !<   the ndim+1 vertices (rows) of p; ftol, the fractional convergence tolerance to be achieved
            !<   in the function value for an early return; iter, and temptr. The routine makes iter
            !<   function evaluations at an annealing temperature temptr, then returns. You should then
            !<   decrease temptr according to your annealing schedule, reset iter, and call the routine
            !<   again (leaving other arguments unaltered between calls). If iter is returned with a positive
            !<   value, then early convergence and return occurred. If you initialize yb to a very large value
            !<   on the first call, then yb and pb(1:ndim) will subsequently return the best function value
            !<   and point ever encountered (even if it is no longer a point in the simplex).
            !<
            !<   the function which is minimized, is chi^2 !!!!!!
            !<   use modified mrqcof subroutine to determine FUNCTION funk
            !<   replace function funk by subroutine ModelCalcChiFunctionGeneral

            use Variables
            use FunctionCalling

            implicit none
            integer :: iter                                                                 !< integer number of iterations done
            integer :: ndim                                                                 !< number of free parameters
            integer :: mp                                                                   !< ndim+1
            integer :: np                                                                   !< number of free parameters
            integer :: NumFile                                                              !< Number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer, dimension(ndim) :: EffectiveArray                                      !< used for parallelization
            integer, dimension(ndim) :: ConversionTable                                     !< need for parallelization
            real*8 :: ftol                                                                  !< real the desired tolerance
            real*8 :: temptr                                                                !< the temperature
            real*8 :: yb                                                                    !< corresponding chi^2 value to pb
            real*8 :: val1                                                                  !< working variables
            real*8, dimension(mp) :: y                                                      !< array, n+1, contains the cost function for each point in
                                                                                            !< simplex func function the cost function
            real*8, dimension(np) :: pb                                                     !< good parameter set
            real*8, dimension(mp,np) :: p                                                   !< matrix n+1 by n contains the initial simplex
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit

            !<-- working variables
            integer :: i, j, k, m, n                                                        !< loop variables
            integer :: ihi, ilo, ii                                                         !< working variables
            real*8 :: rtol, summe, swap, tt, yhi, ylo                                       !< working variables
            real*8 :: ynhi, ysave, yt, ytry                                                 !< working variables
            real*8, dimension(np) :: psum                                                   !< modified declaration
            real*8, dimension(np, np) :: pp                                                 !< effective parameter vectors
            real*8, dimension(np) :: chi2ValuesVector                                       !< chi2 value vector


            pp = 0.d0
            tt = -temptr
        1   Do n = 1,ndim                                                                   !< Enter here when starting or after overall contraction.
                summe = 0.d0                                                                !< Recompute psum.
                Do m = 1, (ndim + 1)
                    summe = summe + p(m,n)
                end Do
                psum(n) = summe
            end Do
        2   ilo = 1                                                                         !< Enter here after changing a single point. Find which point
            ihi = 2                                                                         !< is the highest (worst), next - highest, and lowest (best).
            call ran1(val1, idum)
            ylo = y(1) + tt * dlog(val1)                                                    !< Whenever we "look at" a vertex, it gets a random thermal
            ynhi = ylo                                                                      !< fluctuation.
            call ran1(val1, idum)
            yhi = y(2) + tt * dlog(val1)
            if (ylo > yhi) then
                ihi = 1
                ilo = 2
                ynhi = yhi
                yhi = ylo
                ylo = ynhi
            endif
            Do i = 3, (ndim + 1)                                                            !< Loop over the points in the simplex.
                call ran1(val1, idum)
                yt = y(i) + tt * dlog(val1)                                                 !< More thermal uctuations.
                if (yt <= ylo) then
                    ilo = i
                    ylo = yt
                endif
                if (yt > yhi) then
                    ynhi = yhi
                    ihi = i
                    yhi = yt
                elseif (yt > ynhi) then
                    ynhi = yt
                endif
            end Do
            rtol = 2.d0 * dabs(yhi - ylo)/(dabs(yhi) + dabs(ylo))                           !< Compute the fractional range from highest to lowest and
                                                                                            !< return if satisfactory.
            if (rtol < ftol .or. iter < 0) then                                             !< If returning, put best point and value in slot 1.
                swap = y(1)
                y(1) = y(ilo)
                y(ilo) = swap
                Do n = 1,ndim
                    swap = p(1, n)
                    p(1, n) = p(ilo, n)
                    p(ilo, n) = swap
                end Do
                return
            endif
            iter = iter - 2


            !< Begin a new iteration. First extrapolate by a factor -1 through the face of the simplex across
            !< from the high point, i.e., reflect the simplex from the high point.
            call amotsa(ytry, p, y, psum, mp, np, ndim, pb, yb, ihi, yhi, (-1.d0), tt, ma, a, ia, colx, NumFile, MaxL, MaxCol)
            if (ytry <= ylo) then


                !< Gives a result better than the best point, so try an additional extrapolation by a factor 2.
                call amotsa(ytry, p, y, psum, mp, np, ndim, pb, yb, ihi, yhi, 2.d0, tt, ma, a, ia, colx, NumFile, MaxL, MaxCol)


            elseif (ytry >= ynhi) then


                !< The reflected point is worse than the second-highest, so look for an intermediate lower point, i.e. do a one-dimensional contraction.
                ysave = yhi
                call amotsa(ytry, p, y, psum, mp, np, ndim, pb, yb, ihi, yhi, 0.5d0, tt, ma, a, ia, colx, NumFile, MaxL, MaxCol)
                if (ytry >= ysave) then                                                     !< Can't seem to get rid of that high point.
                    EffectiveArray = 0
                    k = 0
                    Do i = 1, (ndim + 1)
                        if (i /= ilo) then
                            k = k + 1
                            EffectiveArray(k) = i
                        endif
                    end Do
                    ConversionTable = 0
                    k = 0
                    Do i = 1, ma
                        if (ia(i)) then
                            k = k + 1
                            ConversionTable(k) = i
                        endif
                    end Do


                    !< update p array
                    Do k = 1, ndim                                                          !< Better contract around the lowest (best) point.
                        i = EffectiveArray(k)
                        Do j = 1, ndim
                            ii = ConversionTable(j)
                            psum(j) = 0.5d0 * (p(i,j) + p(ilo,j))


                            !< check if parameters are within limits
                            if (psum(j) < paramset(3, ii) .or. psum(j) > paramset(4, ii)) then
                                if (p(i,j) >= paramset(3, ii) .and. p(i, j) <= paramset(4, ii)) then
                                    psum(j) = p(i,j)
                                else
                                    if (psum(j) < paramset(3, ii)) then
                                        psum(j) = paramset(3, ii)
                                    elseif (psum(j) > paramset(4, ii)) then
                                        psum(j) = paramset(4, ii)
                                    endif
                                endif

                            endif
                            p(i, j) = psum(j)
                            pp(k, j) = psum(j)
                        end Do
                    end Do

                    ! Debug:
                    ! print*,'ParallelizationFlag = ',ParallelizationFlag
                    ! call system("ulimit -s")


                    !< determine chi2 values for all parameter vectors
                    chi2ValuesVector = 0.d0
                    call ModelCalcChiFunctionGeneral(ma, ia, a, ndim, ndim, NumFile, MaxL, MaxCol, pp, chi2ValuesVector)
                    y(:) = chi2ValuesVector(:)


                    iter = iter - ndim
                    goto 1
                endif
            else
                iter = iter + 1                                                             !< Correct the evaluation count.
            endif
            goto 2
        end subroutine amebsa


        !*************************************************************************************************************************************************
        !> subroutine: amotsa
        !>
        !> subroutine amotsa extrapolates a point an checks it
        !>
        !>
        !> input variables:     p:                  matrix n+1 by n contains the initial simplex
        !>                      y:                  array, n+1, contains the cost function for each point in simplex
        !>                                          func function the cost function
        !>                      psum:               parameters for simplex (?)
        !>                      mp:                 ndim+1
        !>                      np:                 number of free parameters
        !>                      ndim:               number of free parameters
        !>                      pb:                 good parameter set
        !>                      yb:                 corresponding chi^2 value
        !>                      ihi:                for finding highest point (in vertex ?)
        !>                      yhi:                for finding highest point (in vertex ?)
        !>                      fac:                factor fac for extrapolation
        !>                      tt:                 negative value of temperature
        !>                      ma:                 total number of parameters
        !>                      a:                  array containing the parameter set
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      colx:               number of columns in experimental x data
        !>                      NumFile:            Number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>
        !>
        !> output variables:    results:            final result of amotsa
        !>
        !>
        !> \author Numercial Recipies and Thomas Moeller
        !>
        !> \date 18.09.2009
        !>
        subroutine amotsa(results, p, y, psum, mp, np, ndim, pb, yb, ihi, yhi, fac, tt, ma, a, ia, colx, NumFile, MaxL, MaxCol)
            !<
            !<  USES funk, ran1
            !<  Extrapolates by a factor fac through the face of the simplex across from the high point,
            !<  tries it, and replaces the high point if the new point is better.

            use Variables
            use FunctionCalling

            implicit none
            integer :: ndim                                                                 !< number of free parameters
            integer :: mp                                                                   !< ndim+1
            integer :: np                                                                   !< number of free parameters
            integer :: ihi                                                                  !< for finding highest point (in vertex ?)
            integer :: NumFile                                                              !< Number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            real*8 :: results                                                               !< final result of amotsa
            real*8 :: fac                                                                   !< factor fac for extrapolation
            real*8 :: yb                                                                    !< corresponding chi^2 value to pb
            real*8 :: yhi                                                                   !< for finding highest point (in vertex ?)
            real*8 :: tt                                                                    !< negative value of temperature
            real*8, dimension(mp) :: y                                                      !< array, n+1, contains the cost function for each point in
                                                                                            !> simplex func function the cost function
            real*8, dimension(np) :: pb                                                     !< good parameter set
            real*8, dimension(np) :: psum                                                   !< parameters for simplex (?)
            real*8, dimension(mp,np) :: p                                                   !< matrix n+1 by n contains the initial simplex
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(1) :: chi2ValuesVector                                        !< here only one chi2 value
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit

            !< working variables
            integer :: i, j                                                                 !< loop variables
            real*8 :: fac1, fac2, yflu, ytry, val1                                          !< working variables
            real*8, dimension(ndim) :: ptry                                                 !< working variables


            !< calculate new parameter vector
            ptry = 0.d0
            fac1 = (1.d0 - fac)/ndim
            fac2 = fac1 - fac
            j = 0
            Do i = 1, ma
                if (ia(i)) then
                    j = j + 1
                    ptry(j) = psum(j) * fac1 - p(ihi,j) * fac2


                    !< check if parameters are within limits
                    if (ptry(j) < paramset(3,i) .or. ptry(j) > paramset(4,i)) then
                        if (psum(j) >= paramset(3,i) .and. psum(j) <= paramset(4,i)) then
                            ptry(j) = psum(j)
                        else
                            if (ptry(j) < paramset(3,i)) then
                                ptry(j) = paramset(3,i)
                            elseif (ptry(j) > paramset(4,i)) then
                                ptry(j) = paramset(4,i)
                            endif
                        endif

                    endif
                endif
            end Do


            !< determine chi2 values for all parameter vectors
            chi2ValuesVector = 0.d0
            call ModelCalcChiFunctionGeneral(ma, ia, a, 1, ndim, NumFile, MaxL, MaxCol, ptry, chi2ValuesVector)
            ytry = chi2ValuesVector(1)

            ! Debug:
            !   print*,'tt = ', tt
            !   print*,'ytry = ', ytry
            !   print*,'ptry = ', ptry


            if (ytry <= yb) then                                                            !< Save the best - ever.
                Do j = 1, ndim
                    pb(j) = ptry(j)
                end Do
                yb = ytry
            endif
            call ran1(val1, idum)
            yflu = ytry - tt * dlog(val1)                                                   !< We added a thermal fluctuation to all the current
            if (yflu < yhi) then                                                            !<   vertices, but we subtract it here, so as to give
                y(ihi) = ytry                                                               !<   the simplex a thermal Brownian motion: It likes
                yhi = yflu                                                                  !<   to accept any suggested change.
                Do j = 1, ndim
                    psum(j) = psum(j) - p(ihi,j) + ptry(j)
                    p(ihi,j) = ptry(j)
                end Do
            endif
            results = yflu
            return
        end subroutine amotsa
end Module SimulatedAnnealing_NR
!*********************************************************************************************************************************************************
! ############################################################ End: NR version only   ####################################################################


!*********************************************************************************************************************************************************
!> Module: Algorithm
!>
!>         Module contains the main subroutine used to start the different versions of the SimulatedAnnealing algorithm
!>
!>
!> \author Thomas Moeller
!>
!> \date 01.09.2014
!>
Module Algorithm

    use Variables
    use SimulatedAnnealing_Variables
    use SimulatedAnnealing_SCIPY
    ! ############################################################ Begin: NR version only ################################################################
    use SimulatedAnnealing_NR
    ! ############################################################ End: NR version only   ################################################################

    implicit none

    contains


        !*************************************************************************************************************************************************
        !> subroutine: MainAlg
        !>
        !> main subroutine which starts the SimulatedAnnealing algorithm
        !>
        !>
        !> input variables:         printflagNum:           flag for screen output 1 (=yes) or 0 (=no)
        !>                          LastAlgorithmNum:       number of last algorithm
        !>                          chilm:                  user defined abort criteria for chi**2
        !>                          NumberOfFitAlgorithms:  total number of all algorithms in the chain
        !>                          numiter:                max. number of iterations
        !>                          Temperatureorg:         temperature for "heating"
        !>                          SimulatedAnnealingCounter:      counts number of calls
        !>                          TemperatureReductionKoeffOrg:   coefficient for temperature reduction
        !>                          NumberOfReductionsOrg:  number of reductions
        !>                          MaxNumberOfReheatingPhasesOrg:   max. number of reheating phases
        !>                          DeterminationChi2:      method being used for the determination of chi^2
        !>                          PlotIterationOrg:       plot model function for each iteration set 1(=yes) or 0(=no)
        !>                          PlotTypeOrg:            get type of plot
        !>                          fitlog:                 path for log-file containing the current values of chi**2
        !>                          NumberInputFilesorg:    number of input files for the external model program
        !>                          NumberOutputFilesOrg:   number of output files for the external model program
        !>                          ParallelizationFlagorg: contains the number of processors used for parallelization
        !>                          JobIDorg:               job identification number
        !>                          MaxInputLinesOrg:       max number of lines in an input file
        !>                          MaxParameterOrg:        max number of parameters in a line of an input file
        !>                          RenormalizedChi2Org:    flag for using renormalized chi**2
        !>                          currentpathorg:         path of the working directory
        !>                          FitParameterNameOrg:    array containing the names of the model parameters
        !>                          FitParameterValueLineOrg:   array containing the values of the model parameters as string
        !>                          CalculationMethodOrg:   method of computation (at once or point-to-point)
        !>                          xAxisLabel:             label of the x-axis (for plot)
        !>                          yAxisLabel:             label of the y-axis (for plot)
        !>                          zAxisLabel:             label of the z-axis (for plot)
        !>                          PathStartScriptOrg:     path and name of the start script for calling model function
        !>                          ExeCommandStartScriptOrg:   command for calling model function
        !>                          parametersetorg:        the complete set of paramters (incl. flags and limits)
        !>                          expdataxorg:            array containing the experimental x side
        !>                          expdatayorg:            array containing the experimental y side
        !>                          expdataerrororg:        array containing the experimental error of the y side
        !>                          NumberRangesOrg:        number of y-columns for each experimental file
        !>                          MinRangeOrg:            array containing the minimal exp. ranges
        !>                          MaxRangeOrg:            array containing the maximal exp. ranges
        !>                          NumberXColumnsOrg:      number of x-columns for each experimental file
        !>                          NumberYColumnsOrg:      number of y-columns for each experimental file
        !>                          lengthexpdataorg:       number of lines in experimental data
        !>                          MaxRangeNumber:         max. number of ranges
        !>                          NumFileOrg:             number of experimental files
        !>                          MaxLengthOrg:           max length of experimental data
        !>                          MaxColXOrg:             number of columns concerning to the experimental x side
        !>                          MaxColYOrg:             number of columns concerning to the experimental y side
        !>                          parameternum:           number of model parameter
        !>                          SortFortranNum:         sort chi^2 log file by fortran
        !>
        !> output variables:        calstatus:              status flag of calculation (= 0: all ok)
        !>                          FitFunctionOut:         values of the model function at the calculated points
        !>                          Chi2Values:             values of the chi^2 function at the calculated points
        !>                          FinalParameterSet:      the complete set of paramters (incl. flags and limits)
        !>
        subroutine MainAlg(printflagNum, LastAlgorithmNum, calstatus, FitFunctionOut, Chi2Values, chilm, NumberOfFitAlgorithms, numiter, &
                           SimulatedAnnealingCounter, NumberSites, GeneralAlgorithmSettings, DeterminationChi2, PlotIteration, PlotType, fitlog, &
                           NumberInputFilesorg, NumberOutputFilesOrg, ParallelizationFlagorg, JobIDorg, MaxInputLinesOrg, MaxParameterOrg, &
                           RenormalizedChi2Org, currentpathorg, FitParameterNameLocal, FitParameterValueLocal, CalculationMethodOrg, xAxisLabel, &
                           yAxisLabel, zAxisLabel, PathStartScriptOrg, ExeCommandStartScriptOrg, parametersetorg, FinalParameterSet, expdataxorg, &
                           expdatayorg, expdataerrororg, NumberRangesOrg, MinRangeOrg, MaxRangeOrg, NumberXColumnsOrg, NumberYColumnsOrg, &
                           lengthexpdataorg, MaxRangeNumber, NumFileOrg, MaxLength, MaxColXOrg, MaxColYOrg, parameternum, SortFortranNum)

            implicit none
            ! ********** input variables **********
            integer :: parameternum                                                         !< number of model parameter
            integer :: NumberOfFitAlgorithms                                                !< total number of all algorithms in the chain
            integer :: numiter                                                              !< max. number of iterations
            integer :: MethodSA                                                             !< version of SA algorithm (NR or scipy)
            integer :: ScheduleSAorg                                                        !< schedule for scipy version
            integer :: NumFileOrg                                                           !< number of experimental files
            integer, dimension(NumFileOrg) :: lengthexpdataorg                              !< number of lines in experimental data
            integer, dimension(NumFileOrg) :: NumberXColumnsOrg                             !< number of x-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberYColumnsOrg                             !< number of y-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberRangesOrg                               !< number of y-columns for each experimental file
            integer :: MaxColXOrg                                                           !< number of columns concerning to the experimental x side
            integer :: MaxColYOrg                                                           !< number of columns concerning to the experimental y side
            integer :: MaxLength                                                            !< max length of experimental data
            integer :: printflagNum                                                         !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: LastAlgorithmNum                                                     !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: SortFortranNum                                                       !< flag indicating if chi2 log file is sorted by fortran
                                                                                            !< yes (=1) or not (=0)
            integer :: DeterminationChi2                                                    !< method being used for the determination of chi^2
            integer :: PlotIteration                                                        !< plot model func. for each iteration set 1 (=yes) or 0 (=no)
            integer :: PlotType                                                             !< get type of plot
            integer :: NumberInputFilesorg                                                  !< number of input files for the external model program
            integer :: NumberOutputFilesOrg                                                 !< number of output files for the external model program
            integer :: ParallelizationFlagorg                                               !< contains the number of processors used for parallelization
            integer :: JobIDorg                                                             !< job identification number
            integer :: MaxInputLinesOrg                                                     !< max number of lines in an input file
            integer :: MaxParameterOrg                                                      !< max number of parameters in a line of an input file
            integer :: RenormalizedChi2Org                                                  !< flag for using renormalized chi**2
            integer :: SimulatedAnnealingCounter                                            !< counts number of calls
            integer :: NumberOfReductionsOrg                                                !< counts the number of temperature reductions
            integer :: MaxNumberOfReheatingPhasesOrg                                        !< max number of reheating phases
            integer :: NumberSites                                                          !< number of best sites
            integer :: MaxRangeNumber                                                       !< max. number of ranges
            real*8 :: TemperatureReductionKoeffOrg                                          !< temperature reduction coefficient
            real*8 :: Temperatureorg                                                        !< temperature for algorithm
            real*8 :: chilm                                                                 !< user defined abort criteria for chi**2
            real*8, dimension(15) :: GeneralAlgorithmSettings                               !< special algorithm settings
            real*8, dimension(NumFileOrg, MaxLength, MaxColXOrg) :: expdataxorg             !< array containing the experimental x side
            real*8, dimension(NumFileOrg, MaxLength, MaxColYOrg) :: expdatayorg             !< array containing the experimental y side
            real*8, dimension(NumFileOrg, MaxLength, MaxColYOrg) :: expdataerrororg         !< array containing the experimental error of the y side
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MinRangeOrg        !< array containing the minimal exp. ranges
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MaxRangeOrg        !< array containing the maximal exp. ranges
            character(len=8192) :: fitlog                                                   !< path for log-file containing the current values of chi**2
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=20) :: CalculationMethodOrg                                       !< method of computation
            character(len=8192) :: PathStartScriptOrg                                       !< command for calling model function
            character(len=8192) :: ExeCommandStartScriptOrg                                 !< command for calling model function
            character(len=8192) :: currentpathorg                                           !< path of the working directory
            character(len=512), dimension(parameternum) :: FitParameterNameLocal            !< array containing the names of the model parameters
            character(len=512), dimension(parameternum) :: FitParameterValueLocal           !< array containing the values of the model parameters as
                                                                                            !< string

            ! ********** in/output variables **********
            real*8, dimension(4, parameternum) :: parametersetorg                           !< the non-optimized (initial parameter set)


            ! ********** output variables **********
            integer :: calstatus                                                            !< the following line is necessary for f2py
            real*8, dimension(NumberSites, parameternum) :: FinalParameterSet               !< array containing the optimized parameter set
            real*8, dimension(NumberSites, NumFileOrg, MaxLength, MaxColYOrg) :: FitFunctionOut !< values of the model function at the calculated points
            real*8, dimension(NumberSites, NumFileOrg, MaxLength, MaxColYOrg) :: Chi2Values     !< values of the model function at the calculated points


            ! ********** working variabels **********
            integer :: i, j, k, ii, jj                                                      !< working variables
            integer :: nfit                                                                 !< (= effective parameter number) number of model
                                                                                            !<      parameters which are optimized
            integer :: ok                                                                   !< status of calculation
            integer :: actualiteration                                                      !< contains the current iteration within the iteration loop
            integer :: flag                                                                 !< working variable used within the iteration loop
            integer :: NumInputFiles                                                        !< need for loop over input files
            integer :: allocstatus, deallocstatus                                           !< working variables for allocation/deallocation
            integer, dimension(8) :: VALUES                                                 !< value for the date_and_time subroutine
            real*8 :: chilim                                                                !< lower limit of chi**2 normalized to the total number of
                                                                                            !< all data points
            real*8, dimension(1) :: chi2ValuesVector                                        !< here only one chi2 value
            real*8, allocatable, dimension(:) :: currentparmval, currentparmvalold          !< array containing the current parameter set within the
                                                                                            !< iteration loop
            character(len=512) :: fitlogparam                                               !< path for log-file containing the current parameter values
            character(len=512) :: fitlogChi2                                                !< path for log-file containing chi**2 and the corresponding
                                                                                            !< parameter values
            character(len=10) :: Number1                                                    !< variable for number to string converting
            character(len=8) :: DATE                                                        !< variable for the date_and_time subroutine
            character(len=10) :: TIME                                                       !< variable for the date_and_time subroutine
            character(len=5) :: ZONE                                                        !< variable for the date_and_time subroutine
            logical, allocatable, dimension(:) :: ia                                        !< array indicating if parameter should be optimized
                                                                                            !< (true) or not (false)
            character(len=8192) :: SiteExt, NumExt, BaseDir, FuncCallExt                    !< working variables for final input file name


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< set print flag and last algorithm flag
            calstatus = 0                                                                   !< set calculation status to 0 = everything is ok
            if (printflagNum == 1) then                                                     !< set printflag
                printflag = .true.
            else
                printflag = .false.
            endif
            if (LastAlgorithmNum == 1) then
                LastAlgorithmFlag = .true.
            else
                LastAlgorithmFlag = .false.
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy contents of some input variables to module variables
            NumberExpFiles = NumFileOrg                                                     !< copy number of experimental files to global variable
            currentpath = trim(adjustl(currentpathorg))                                     !< copy path of working directory to module variable without
                                                                                            !< trailing and leading blanks
            MaxColX = MaxColXOrg                                                            !< copy number of columns of the experimental x data to
                                                                                            !< module variable
            MaxColY = MaxColYOrg                                                            !< copy number of columns of the experimental y data to
                                                                                            !< module variable
            MaxExpLength = MaxLength                                                        !< copy max. number of exp. data points
            MaxNumberRanges = MaxRangeNumber                                                !< copy of max. number of data ranges in a exp. data file
            parameternumber = parameternum                                                  !< copy input variable containing the number of parameters
                                                                                            !< to module variable
            DetChi2 = DeterminationChi2                                                     !< copy flag for determination of chi**2
            NumberInputFiles = NumberInputFilesorg                                          !< copy number of input files for the external program to
                                                                                            !< global variable
            NumberOutputFiles = NumberOutputFilesOrg                                        !< copy number of output files for the external program to
                                                                                            !< global variable
            ParallelizationFlag = ParallelizationFlagorg                                    !< copy number of used processors to global variable
            JobID = JobIDorg                                                                !< copy job-ID number to global variable
            MaxInputLines = MaxInputLinesOrg                                                !< copy max number of input lines in an input file
            MaxParameter = MaxParameterOrg                                                  !< copy max number of parameters in a line of an input file
            RenormalizedChi2 = .true.                                                       !< define flag for using renormalized chi**2
            QualityLimit = 1                                                                !< set number of best sites to 1
            if (RenormalizedChi2Org /= 1) then
                RenormalizedChi2 = .false.
            endif
            PlotIterationFlag = .false.
            if (PlotIteration == 0) PlotIterationFlag = .true.


            !< get special algorithm settings
            MethodSA = int(GeneralAlgorithmSettings(2))
            ScheduleSAorg = GeneralAlgorithmSettings(3)
            Temperatureorg = GeneralAlgorithmSettings(4)
            TemperatureReductionKoeffOrg = GeneralAlgorithmSettings(5)
            NumberOfReductionsOrg = GeneralAlgorithmSettings(6)
            MaxNumberOfReheatingPhasesOrg = GeneralAlgorithmSettings(7)

            ScheduleSA = ScheduleSAorg                                                      !< schedule for scipy version
            Temperature = Temperatureorg                                                    !< copy temperature for algorithm
            T0 = Temperatureorg                                                             !< copy temperature for algorithm
            dwell = NumberOfReductionsOrg                                                   !< copy number of reductions
            NumberOfReductions = NumberOfReductionsOrg                                      !< copy number of reductions
            MaxNumberOfReheatingPhases = MaxNumberOfReheatingPhasesOrg                      !< copy max number of reheating phases
            TemperatureReductionKoeff = TemperatureReductionKoeffOrg                        !< copy temperature reduction coefficient
            learn_rate = TemperatureReductionKoeffOrg                                       !< copy temperature reduction coefficient

            ! Debug:
            !    print*,'MethodSA = ', MethodSA
            !    print*,'ScheduleSAorg = ', ScheduleSAorg
            !    print*,'Temperatureorg = ', Temperatureorg
            !    print*,'TemperatureReductionKoeffOrg = ', TemperatureReductionKoeffOrg
            !    print*,'NumberOfReductionsOrg = ', NumberOfReductionsOrg
            !    print*,'MaxNumberOfReheatingPhasesOrg = ', MaxNumberOfReheatingPhasesOrg
            !    print*,'ExeCommandStartScriptOrg = ',trim(ExeCommandStartScriptOrg)
            !    print*,'FitFktInputOrg = ',trim(FitFktInputOrg)
            !    print*,'MaxNumberParameter = ',MaxNumberParameter
            !    print*,'NumFileOrg = ',NumFileOrg
            !    print*,'MaxColXOrg = ',MaxColXOrg
            !    print*,'MaxColYOrg = ',MaxColYOrg
            !    Do i=1,NumFileOrg
            !        print*,'    Experimental file: i = ',i
            !        print*,'    lengthexpdataorg(i) = ',lengthexpdataorg(i)
            !        print*,'    NumberYColumnsOrg(i) = ',NumberYColumnsOrg(i)
            !        print*,'    expdataxorg(i,1:5,1) = ',expdataxorg(i,1:5,1)
            !        print*,'    expdatayorg(i,1:5,1) = ',expdatayorg(i,1:5,1)
            !        print*,'    expdataerrororg(i,1:5,1) = ',expdataerrororg(i,1:5,1)
            !    end Do
            !    print*,'chilm = ',chilm
            !    print*,'numiter = ',numiter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'currentpathorg = ',trim(adjustl(currentpathorg))
            !    print*,'len(FitParameterNameOrg) = ',len(FitParameterNameOrg
            !    print*,'FitParameterNameOrg = ',FitParameterNameOrg
            !    print*,'FitParameterValueLineOrg = ',FitParameterValueLineOrg
            !    print*,"parametersetorg(1,:) = ",parametersetorg(1,:)
            !    print*,"parametersetorg(2,:) = ",parametersetorg(2,:)
            !    print*,"parametersetorg(3,:) = ",parametersetorg(3,:)
            !    print*,"parametersetorg(4,:) = ",parametersetorg(4,:)
            !    print*,"RenormalizedChi2 = ",RenormalizedChi2
            !    return


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< set temp-directory
            TempDirectory = " "
            CALL GetEnv('MAGIXTempDirectory', TempDirectory)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< modify name of log file
            i = index(fitlog, "/", back = .true.)
            j = index(fitlog, ".", back = .true.)
            Number1 = "          "
            write(Number1,'(I10)') abs(SimulatedAnnealingCounter)
            if (j > i) then
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__SA__call_" // trim(adjustl(Number1)) // trim(adjustl(fitlog(j:)))
                else
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__SA" // trim(adjustl(fitlog(j:)))
                endif
            else
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog)) // "__SA__call_" // trim(adjustl(Number1)) // ".log"
                else
                    fitlog = trim(adjustl(fitlog)) // "__SA.log"
                endif
            endif

            ! Debug:
            !   print*,'>',trim(adjustl(fitlog)),'<'


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open log file and write header
            open(logchannel, file = trim(fitlog), status = 'replace')
            write(logchannel,'(" ")')
            write(logchannel,'("log-file for Simulated-Annealing algorithm:")')
            write(logchannel,'(43("-"))')
            write(logchannel,'(" ")')


            !< get current local time and date and write to log-file
            call date_and_time(DATE, TIME, ZONE, VALUES)
            write(logchannel,'(" ")')
            write(logchannel,'("algorithm starts at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                                  TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open log-file for the parameter and write header
            fitlogparam = trim(fitlog) // ".param"
            open(paramchannel,file = trim(fitlogparam), status = 'replace')
            write(paramchannel,'(" ")')
            write(paramchannel,'("log-file containing the actual values of the parameters used in the Simulated-Annealing algorithm:")')
            write(paramchannel,'(98("-"))')
            write(paramchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open file containing the values of chi**2 and the corresponding values of the parameters
            WriteChi2Flag = .true.
            NumberLinesChi2 = 0
            fitlogChi2 = trim(fitlog) // ".chi2"
            open(Chi2Channel,file = trim(fitlogChi2), status = 'replace')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate if necessary and print error message if necessay
            if (allocated(expdatax)) then
                deallocate(expdatax, expdatay, expdatae, lengthexpdata, NumberXColumns, NumberYColumns, FirstPointExpData, LastPointExpData, &
                           NumberRanges, MinRange, MaxRange, ExpData_reversed_flag, stat = deallocstatus)
                if (deallocstatus /= 0) then                                                    !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine MainAlg:")')
                    write(logchannel,'(2x,"Can not deallocate variables expdatax etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine MainAlg:")'
                    print '(2x,"Can not deallocate variables expdatax etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for variables, clear content of the variables and print error message if necessay
            allocate(expdatax(NumberExpFiles, MaxLength, MaxColX), expdatay(NumberExpFiles, MaxLength, MaxColY), &
                     expdatae(NumberExpFiles, MaxLength, MaxColY), lengthexpdata(NumberExpFiles), NumberXColumns(NumberExpFiles), &
                     NumberYColumns(NumberExpFiles), FirstPointExpData(NumberExpFiles, MaxColX), LastPointExpData(NumberExpFiles, MaxColX), &
                     NumberRanges(NumberExpFiles), MinRange(NumberExpFiles, MaxRangeNumber, MaxColX), MaxRange(NumberExpFiles, MaxRangeNumber, MaxColX), &
                     ExpData_reversed_flag(NumberExpFiles), stat = allocstatus)
            if (allocstatus /= 0) then                                                      !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not allocate variables expdatax etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not allocate variables expdatax,expdatay etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            MaxRangeNumber = MaxRangeNumber - 1                                             !< get real value
            expdatax = 0.d0
            expdatay = 0.d0
            expdatae = 0.d0
            lengthexpdata = 0
            NumberXColumns = 0
            NumberYColumns = 0
            FirstPointExpData = 1.d99
            LastPointExpData = -1.d99
            NumberRanges = 0
            MinRange = 0.d0
            MaxRange = 0.d0
            ExpData_reversed_flag = .false.


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy input variables to module variabels
            expdatax(:,:,:) = expdataxorg(:,:,:)
            expdatay(:,:,:) = expdatayorg(:,:,:)
            expdatae(:,:,:) = expdataerrororg(:,:,:)
            lengthexpdata = lengthexpdataorg
            NumberXColumns = NumberXColumnsOrg
            NumberYColumns = NumberYColumnsOrg
            CalculationMethod = CalculationMethodOrg
            PathStartScript = PathStartScriptOrg
            ExeCommandStartScript = ExeCommandStartScriptOrg
            NumberRanges = NumberRangesOrg
            MinRange = MinRangeOrg
            MaxRange = MaxRangeOrg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine first and last point of each exp. data file
            Do i = 1, NumberExpFiles
                Do j = 1, lengthexpdata(i)
                    ii = 0
                    jj = 0
                    Do k = 1,NumberXColumns(i)
                        if (expdatax(i, j, k) <= FirstPointExpData(i, k)) then
                            ii = ii + 1
                        endif
                        if (expdatax(i, j, k) >= LastPointExpData(i, k)) then
                            jj = jj + 1
                        endif
                    end Do
                    if (ii == NumberXColumns(i)) then
                        FirstPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                    if (jj == NumberXColumns(i)) then
                        LastPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                end Do


                !< check output file starts with the highest x-column value interchange FirstPointOutputFile and LastPointOutputFile
                ii = 0
                Do k = 1, NumberXColumns(i)
                    if (expdatax(i, 1, k) >= expdatax(i, lengthexpdata(i), k)) then
                        ii = ii + 1
                    endif
                end Do
                if (ii == NumberXColumns(i)) then
                    ExpData_reversed_flag(i) = .true.
                endif

                ! Debug:
                ! print*,' '
                ! print*,'File = ',i
                ! print*,'FirstPointExpData(i, 1:NumberXColumns(i)) = ', FirstPointExpData(i, 1:NumberXColumns(i))
                ! print*,'LastPointExpData(i, 1:NumberXColumns(i)) = ', LastPointExpData(i, 1:NumberXColumns(i))
                ! print*,'##########################################'
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy number of parameters required for the model function to working variable
            nfit = sum(parametersetorg(2, :))                                               !< number of parameters which should be optimized


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< dimension of the arrays 'ModelFunction'
            i = 0
            Do k = 1, NumberExpFiles
                i = i + (NumberYColumns(k) * lengthexpdata(k))
            end Do
            j = nfit


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate/allocate memory for some working variables, clear contents of the variables and print error message if necessary
            if (allocated(paramset)) then
                deallocate(paramset, currentparmval, ia, currentparmvalold, FitParameterName, FitParameterValue, ModelFunction, chisqValues, &
                           BestSitesParamSet, BestSitesModelValues, BestSitesChi2Values, ConverterInfit, AtOnceFunction, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine MainAlg:")')
                    write(logchannel,'(2x,"Can not deallocate variables paramset etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine MainAlg:")'
                    print '(2x,"Can not deallocate variables paramset etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif
            allocate(paramset(4, parameternumber), currentparmval(parameternumber), currentparmvalold(parameternumber), ConverterInfit(nfit), &
                     ia(parameternumber), FitParameterName(parameternumber), FitParameterValue(parameternumber), &
                     ModelFunction(1, NumberExpFiles, MaxColY, MaxLength), chisqValues(0:0), &
                     AtOnceFunction(0:ParallelizationFlag - 1, NumberExpFiles, MaxColY, MaxLength), &
                     BestSitesParamSet(QualityLimit, nfit + 1), BestSitesModelValues(QualityLimit, NumberExpFiles, MaxLength, MaxColY), &
                     BestSitesChi2Values(QualityLimit, NumberExpFiles, MaxLength, MaxColY), stat = allocstatus)
            if (allocstatus /= 0) then
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not allocate variables paramset etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not allocate variables paramset etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            ia = .false.
            ConverterInfit = 0
            FitParameterName = FitParameterNameLocal
            FitParameterValue = FitParameterValueLocal
            currentparmval = 0.d0
            currentparmvalold = 0.d0
            ModelFunction = 0.d0
            AtOnceFunction = 0.d0
            chisqValues = 0.d0
            BestSitesParamSet = 0.d0
            BestSitesParamSet(:, 1) = 1.d99
            BestSitesModelValues = 0.d0
            BestSitesChi2Values = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy parameter set to module variable
            paramset = parametersetorg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write registration mask for the fit model
            if (printflag) print '(9x, "Writing registration mask for the fit model .. ", $)'
            call RegistrationMask(ok)
            if (ok /= 0) then
                return
            endif
            if (printflag) print '("done!")'


            !< define ia variable
            ConverterInfit = 0
            ia = .false.
            k = 0
            Do i = 1, parameternumber
                if (paramset(2, i) == 1) then
                    k = k + 1
                    ia(i) = .true.
                    ConverterInfit(k) = i
                endif
            end Do
            NumberFreeParameter = sum(paramset(2, :))                                       !< determine number of free parameter


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize model program
            call ModelInit


            !< write on the screen what you Do ..
            if (printflag) then
                print '(" ")'
                print '(" ")'
                write(Number1,'(I10)') JobID                                                !< write JobID to string
                print '(9x,"Temporary files are stored in: ",A)', trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(Number1)) // "/"
                print '(" ")'
                print '(" ")'
                print '(9x,"Start Simulated-Annealing algorithm (", A, " version) ..")', trim(adjustl(ParallelizationMethod))
                print '(" ")'
            endif
            write(logchannel,'(11x,"Start Simulated-Annealing algorithm (", A, " version) ..")') trim(adjustl(ParallelizationMethod))
            write(logchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< lower limit of chi^2 for stopping condition normalized to the number of calculation points
            chilim = 0.d0
            if (RenormalizedChi2) then
                Do i = 1, NumberExpFiles
                    chilim = chilim + (NumberYColumns(i) * lengthexpdata(i) - parameternumber) * dabs(chilm)
                end Do
                chilim = dabs(chilim)
                write(logchannel,'(11x,"Renormalized limit for chi^2 = ", ES25.15)') chilim
                write(logchannel,'(" ")')
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Renormalized limit for chi^2 = ",ES25.15)', chilim
                    print '(" ")'
                    print '(" ")'
                endif
            else
                chilim = dabs(chilim)
                write(logchannel,'(11x,"Limit for chi^2 = ", ES25.15)') chilim
                write(logchannel,'(" ")')
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Limit for chi^2 = ",ES25.15)', chilim
                    print '(" ")'
                    print '(" ")'
                endif
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< start iteration ..
            Gradientflag = .false.                                                          !< we do not need the gradient of the function here
            flag = 0                                                                        !< set flag variable to 0
            actualiteration = 0                                                             !< set working variable containing the current
                                                                                            !<      iteration number to 0
            currentparmval = paramset(1,1:parameternumber)                                  !< copy the values of the parameters to another array
            FitFunctionOut = 0.d0                                                           !< reset FitFunctionOut array
            Chi2Values = 0.d0                                                               !< reset Chi2Values array
            UseCalculationReduction = .false.                                               !< no calculation reduction
            CurrentNumberLinesCalcReduction = 0                                             !< reset CurrentNumberLinesCalcReduction variable


            !< call modified subroutine amebsa
            currentparmvalold = currentparmval

            ! ############################################################ Begin: NR version only ########################################################
            if (MethodSA == 1) then
            ! ############################################################ End: NR version only   ########################################################
                call call_scipy(nfit+1, nfit, nfit, nfit, chilim, numiter, parameternumber, currentparmval, ia, MaxColX, NumberExpFiles, MaxLength, &
                                MaxColY, PlotIteration, PlotType, xAxisLabel, yAxisLabel, zAxisLabel, fitlog, &
                                MaxNumberOfReheatingPhases)
            ! ############################################################ Begin: NR version only ########################################################
            else
                call call_amebsa(nfit+1, nfit, nfit, nfit, chilim, numiter, parameternumber, currentparmval, ia, MaxColX, NumberExpFiles, MaxLength, &
                                MaxColY, PlotIteration, PlotType, xAxisLabel, yAxisLabel, zAxisLabel, fitlog)
            endif
            ! ############################################################ End: NR version only   ########################################################


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine model function values if plot for each iteration option is not selected
            if (.not. PlotIterationFlag) then
                PlotIterationFlag = .true.
                chi2ValuesVector = 0.d0
                call ModelCalcChiFunctionGeneral(parameternumber, ia, paramset(1, :), 1, nfit, NumberExpFiles, MaxLength, MaxColY, &
                                                 BestSitesParamSet(1, 2:), chi2ValuesVector)
            endif
            FitFunctionOut(1, :, :, :) = BestSitesModelValues(1, :, :, :)
            Chi2Values(1, :, :, :) = BestSitesChi2Values(1, :, :, :)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< close file
            if (SortFortranNum == 1) call SortChi2File(nfit, parameternumber, ia, currentparmval)
            close(Chi2Channel)
            close(paramchannel)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< write final parameter sets for each site to final input files


            !< define base directory, i.e. path where final input file is written to
            k = index(fitlog, '/', back = .true.)
            if (k == 0) then
                BaseDir = ""
            else
                BaseDir = trim(adjustl(fitlog(:k)))
            endif


            !< define file name extension for number of algorithm call
            write(Number1,'(I10)') abs(SimulatedAnnealingCounter)
            FuncCallExt = "__SA__call_" //trim(adjustl(Number1))


            !< write files
            SiteExt = ".out"


            !< write parameter sets to file
            Do NumInputFiles = 1, NumberInputFiles                                          !< loop over all input files
                NumExt = trim(adjustl(FitFktInput(NumInputFiles)))
                k = index(NumExt, '/', back = .true.)
                if (k > 0) then                                                             !< we only need the file name
                    NumExt = trim(adjustl(NumExt(k:)))
                endif
                j = index(NumExt, '.', back = .true.)
                if (j > 1) then                                                             !< remove file name extension
                    NumExt = trim(adjustl(NumExt(:j - 1)))
                endif

                ! Debug:
                ! print*,"Site = ", i
                ! print*,"Nr. Final input file = ", NumInputFiles
                ! print*,"Final input file = ", trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt)) // trim(adjustl(SiteExt)) &
                !                               // ".input"


                !< write parameter sets to file
                open(235,file = trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt)) // trim(adjustl(SiteExt)) // ".input")
                call WriteParameter(235, .false., MaxColX, expdataxorg, parameternumber, currentparmval, NumInputFiles)
                close(235)
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy current values of the model parameter to variable paramset
            paramset(1,1:parameternumber) = currentparmval
            parametersetorg = paramset
            FinalParameterSet(1, :) = parametersetorg(1, :)
            parametersetorg(2, :) = 0.d0                                                        !< set errors to zero (no errors determined)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write end of log-file and print message to screen
            call date_and_time(DATE, TIME, ZONE, VALUES)
            write(logchannel,*)
            write(logchannel,'("algorithm ends at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                                  TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel,'(" ")')
            write(logchannel,'(150("-"))')

            ! Debug:
            !    print*,'lengthexpdata = ',lengthexpdata
            !    print*,'MaxColX = ',MaxColX
            !    print*,'MaxColY = ',MaxColY
            !    print*,'expdataxorg(1:5,1) = ',expdataxorg(1:5,1)
            !    print*,'expdatayorg(1:5,1) = ',expdatayorg(1:5,1)
            !    print*,'expdataerrororg(1:5,1) = ',expdataerrororg(1:5,1)
            !    print*,'chilm = ',chilm
            !    print*,'numiter = ',numiter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'model = ',trim(model)
            !    print*,'currentpathorg = ',currentpathorg
            !    print*,'parametersetorg(1,:) = ',parametersetorg(1,:)
            !    print*,'parametersetorg(2,:) = ',parametersetorg(2,:)
            !    print*,'parametersetorg(3,:) = ',parametersetorg(3,:)
            !    print*,'parametersetorg(4,:) = ',parametersetorg(4,:)


            !< send plot program the kill signal
            if (PlotIteration == 0) then
                call system("kill -9 " // trim(adjustl(PlotPID)) // " 1>/dev/null 2>&1")
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< free memory of general Simulated Annealing variables
            if (allocated(currentparmval)) deallocate(currentparmval, stat = deallocstatus)
            if (allocated(currentparmvalold)) deallocate(currentparmvalold, stat = deallocstatus)
            if (allocated(ia)) deallocate(ia, stat = deallocstatus)
            if (allocated(x0)) deallocate(x0, stat = deallocstatus)
            if (allocated(x0_orig)) deallocate(x0_orig, stat = deallocstatus)
            if (allocated(lower)) deallocate(lower, stat = deallocstatus)
            if (allocated(upper)) deallocate(upper, stat = deallocstatus)


            !< free memory of model variables
            call ModelParamFree(deallocstatus)
            if (deallocstatus /= 0) then
                write(logchannel,*)
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not deallocate expdatax etc.")')
                write(logchannel,*)
                write(logchannel,'("deallocstatus = ", I4)') deallocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not deallocate variables expdatax etc.")'
                print '(" ")'
                print '("deallocstatus = ", I4)', deallocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< close log file
            close(logchannel)
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(9x,"Finished Simulated-Annealing algorithm!")'
                print '(" ")'
                print '(" ")'
            endif


            !< we're done
            return
        end subroutine MainAlg
end Module Algorithm
!*********************************************************************************************************************************************************


