!*********************************************************************************************************************************************************
!> Module: OpticalProperties
!>
!>
!>  This module contains the subroutines for calculating the epsilon tensor using the generalized Drude-Lorentz model
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine epsTensor:                     calculates the epsilon tensor within the ac-plane
!>
!>
!> Copyright (C) 2009 - 2024
!>
!> I. Physikalisches Institut, Universitaet zu Koeln
!>
!> Produced for the CATS project and MnWO4 Paper
!>
!>
!> fortran module containing the subroutine for "generalized Drude-Lorentz model"
!>
!> Who           When        What
!>
!> T. Moeller    2009-07-07  Initial version
!>
!*********************************************************************************************************************************************************
Module OpticalProperties

    implicit none
    integer :: number_osc                                                                   !< number of osc. for the generalized Drude-Lorentz model
    real*8 :: eps_inf_xx                                                                    !< epsilon_\infty for epsilon_xx matrix element
    real*8 :: eps_inf_xz                                                                    !< epsilon_\infty for epsilon_xz matrix element
    real*8 :: eps_inf_zz                                                                    !< epsilon_\infty for epsilon_zz matrix element
    real*8, allocatable, dimension(:) :: w0, wp, G, theta                                   !< eigenfrequencies, plasma frequencies, damping, and angle
                                                                                            !< for the generalized Drude-Lorentz model

    contains

        !>************************************************************************************************************************************************
        !> subroutine: epsTensor
        !>
        !> calculate the epsilon tensor within the ac-plane
        !>
        !>
        !> input variables:     w:                  frequency
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine epsTensor(w)

            implicit none
            integer :: i, k                                                                 !< loop variables
            real*8 :: w                                                                     !< frequency
            real*8 :: value                                                                 !< calculated reflectance
            real*8 :: theta_rad                                                             !< theta is in radians
            real*8 :: pi                                                                    !< pi
            complex(8) :: eps11, eps12, eps21, eps22                                        !< working variables: components of the epsilon tensor
            complex(8) :: DrudeLorentz                                                      !< working variables: argument of the sum for one oscillator


            !< reset output variable
            value = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate matrix elements of the epsilon tensor
            eps11 = eps_inf_xx
            eps12 = eps_inf_xz
            eps22 = eps_inf_zz

            pi = 4.d0 * datan(1.d0)                                                         !< pi
            k = 4
            Do i = 1, number_osc                                                            !< loop over all oscillators
                DrudeLorentz = ((wp(i)**2) / (w0(i)**2 - w**2 - dcmplx(0.d0,1.d0) * G(i) * w))
                theta_rad = theta(i) * (pi/180.d0)                                          !< convert degree to rad
                eps11 = eps11 + DrudeLorentz * (dcos(theta_rad)**2)
                eps12 = eps12 + DrudeLorentz * (dsin(theta_rad) * dcos(theta_rad))
                eps22 = eps22 + DrudeLorentz * (dsin(theta_rad)**2)
                k = k + 4
            end Do
            eps21 = eps12


            !< write epsilon tensor to file
            write(80,*) w, dreal(eps11)
            write(81,*) w, dimag(eps11)
            write(82,*) w, dreal(eps12)
            write(83,*) w, dimag(eps12)
            write(84,*) w, dreal(eps22)
            write(85,*) w, dimag(eps22)
            return
        end subroutine epsTensor
end Module OpticalProperties
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!>
!>
!> Main Program
!>
!>
program DrudeLorentzGeneral
    !< calculation of the refelctivity using the generalized Drude-Lorentz model

    use OpticalProperties

    implicit none
    integer :: i                                                                            !< loop variables
    integer :: NumberXValues                                                                !< number exp. data points
    integer :: alloc_status, dealloc_status                                                 !< status variables for (de-)allocation
    real*8 :: w                                                                             !< working variable: current frequency w


    !< open files
    open(21,file = "in.txt")                                                                !< open parameter file
    open(22,file = "DataIn.dat")                                                            !< open experimental-point file


    !< read parameter from file
    read(21,*) NumberXValues                                                                !< read number of exp. data points
    read(21,*) eps_inf_xx                                                                   !< read epsilon_xx
    read(21,*) eps_inf_xz                                                                   !< read epsilon_xz
    read(21,*) eps_inf_zz                                                                   !< read epsilon_zz
    read(21,*) number_osc                                                                   !< read number of oscillators


    ! deallocate/allocate memory
    if (allocated(w0)) then
        deallocate(w0, wp, G, theta, stat = dealloc_status)
        if (dealloc_status /= 0) then
            print '(" ")'
            print '(2x,"Error in module DrudeLorentzGeneral:")'
            print '(4x,"Cannot deallocate variable w0, wp, G, theta. ")'
            print '(" ")'
            stop '  Program aborted!'
        endif
    endif
    allocate(w0(number_osc), wp(number_osc), G(number_osc), theta(number_osc), stat = alloc_status)
    if (alloc_status /= 0) then
        print '(" ")'
        print '(2x,"Error in module DrudeLorentzGeneral:")'
        print '(4x,"Cannot allocate variable w0, wp, G, theta. ")'
        print '(" ")'
        stop '  Program aborted!'
    endif
    w0 = 0.d0
    wp = 0.d0
    G = 0.d0
    theta = 0.d0


    !< read parameters for each oscillators from file
    Do i = 1, number_osc                                                                    !< loop over all oscillators
        read(21,*) w0(i), wp(i), G(i), theta(i)
    end Do
    close(21, status = 'delete')                                                            !< close and delete parameter file

    ! Debug:
    !    print*,'DrudeLorentzGeneral:'
    !    print*,'w = ',w
    !    print*,'angle_chi = ',angle_chi
    !    print*,'eps_inf_xx = ',eps_inf_xx
    !    print*,'eps_inf_xz = ',eps_inf_xz
    !    print*,'eps_inf_zz = ',eps_inf_zz
    !    print*,'number_osc = ',number_osc
    !    print*,'w0(1:number_osc) = ',w0(1:number_osc)
    !    print*,'wp(1:number_osc) = ',wp(1:number_osc)
    !    print*,'G(1:number_osc) = ',G(1:number_osc)
    !    print*,'theta(1:number_osc) = ',theta(1:number_osc)
    !    print*,' '


    !< open output files for final epsilon component functions
    open(80, file = "epsilon-tensor__Re-eps11.dat")
    open(81, file = "epsilon-tensor__Im-eps11.dat")
    open(82, file = "epsilon-tensor__Re-eps12.dat")
    open(83, file = "epsilon-tensor__Im-eps12.dat")
    open(84, file = "epsilon-tensor__Re-eps22.dat")
    open(85, file = "epsilon-tensor__Im-eps22.dat")


    !< calculate optical properties at any frequency w
    Do i = 1, NumberXValues                                                                 !< loop over all frequency points
        read(22,*) w                                                                        !< read frequency and polarization angle form file


        !< call subroutine for determine the epsilon tensor
        call epsTensor(w)
    end Do


    !< close files
    close(22, status = 'delete')                                                            !< close and delete experimental-point file
    close(80)
    close(81)
    close(82)
    close(83)
    close(84)
    close(85)
end program DrudeLorentzGeneral
!---------------------------------------------------------------------------------------------------------------------------------------------------------

