!*********************************************************************************************************************************************************
!> Module: OpticalProperties
!>
!>
!>  This module contains the subroutines for calculating the epsilon tensor using the generalized Drude-Lorentz model
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine DetermineEpsTensor:            calculate epsilon tensor
!>      - subroutine reflectance:                   calculate the reflectance within the ac-plane
!>      - subroutine Rp_reflectance:                calculate the p-polarized reflectance
!>      - subroutine InverseMatrix:                 calculate the inverse of a 2x2 complex matrix
!>
!>
!> Copyright (C) 2009 - 2024
!>
!> I. Physikalisches Institut, Universitaet zu Koeln
!>
!> Produced for the CATS project and MnWO4 Paper
!>
!>
!> fortran module containing the subroutine for "generalized Drude-Lorentz model"
!>
!> Who           When        What
!>
!> T. Moeller    2009-07-07  Initial version
!>
!*********************************************************************************************************************************************************
Module OpticalProperties

    implicit none
    real*8 :: BasisRotAngle
    real*8 :: InfallAngle
    real*8 :: ReRotAngle
    real*8 :: ImRotAngle
    real*8 :: ReEPS11
    real*8 :: ImEPS11
    real*8 :: ReEPS22
    real*8 :: ImEPS22
    real*8 :: pi

    contains


        !>************************************************************************************************************************************************
        !> subroutine: DetermineEpsTensor
        !>
        !> calculate epsilon tensor
        !>
        !>
        !> input variables:     None
        !>
        !> output variables:    epsilon_tensor:     calculated epsilon tensor
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine DetermineEpsTensor(epsilon_tensor)

            implicit none
            real*8, dimension(2, 2) :: ReEpsTensor                                          !< matrix for the real part of the epsilon tensor
            real*8, dimension(2, 2) :: ImEpsTensor                                          !< matrix for the imaginary part of the epsilon tensor
            real*8, dimension(2, 2) :: ReDiagEpsTensor                                      !< matrix for the diagonalized real part of the epsilon tensor
            real*8, dimension(2, 2) :: ImDiagEpsTensor                                      !< matrix for the diagonalized imag part of the epsilon tensor
            real*8, dimension(2, 2) :: ReRotMatrix, ReInverseRotMatrix                      !< rotation and inverse of rotation matrix of real part
            real*8, dimension(2, 2) :: ImRotMatrix, ImInverseRotMatrix                      !< rotation and inverse of rotation matrix of imag part
            complex(8), dimension(2, 2) :: epsilon_tensor                                   !< output variable: epsilon tensor


            !< construct matrices
            ReDiagEpsTensor = 0.d0
            ReDiagEpsTensor(1, 1) = ReEPS11
            ReDiagEpsTensor(2, 2) = ReEPS22
            ImDiagEpsTensor = 0.d0
            ImDiagEpsTensor(1, 1) = ImEPS11
            ImDiagEpsTensor(2, 2) = ImEPS22


            !< construct rotation matrix for real part
            ReRotMatrix = 0.d0
            ReRotMatrix(1, 1) = dcos(ReRotAngle)
            ReRotMatrix(1, 2) = -dsin(ReRotAngle)
            ReRotMatrix(2, 1) = dsin(ReRotAngle)
            ReRotMatrix(2, 2) = dcos(ReRotAngle)
            ReInverseRotMatrix = 0.d0
            ReInverseRotMatrix(1, 1) = dcos(ReRotAngle)
            ReInverseRotMatrix(1, 2) = dsin(ReRotAngle)
            ReInverseRotMatrix(2, 1) = -dsin(ReRotAngle)
            ReInverseRotMatrix(2, 2) = dcos(ReRotAngle)


            !< construct rotation matrix for imaginary part
            ImRotMatrix = 0.d0
            ImRotMatrix(1, 1) = dcos(ImRotAngle)
            ImRotMatrix(1, 2) = -dsin(ImRotAngle)
            ImRotMatrix(2, 1) = dsin(ImRotAngle)
            ImRotMatrix(2, 2) = dcos(ImRotAngle)
            ImInverseRotMatrix = 0.d0
            ImInverseRotMatrix(1, 1) = dcos(ImRotAngle)
            ImInverseRotMatrix(1, 2) = dsin(ImRotAngle)
            ImInverseRotMatrix(2, 1) = -dsin(ImRotAngle)
            ImInverseRotMatrix(2, 2) = dcos(ImRotAngle)


            !< construct non-diagonal form of real part of epsilon tensor
            ReEpsTensor = matmul(matmul(ReRotMatrix, ReDiagEpsTensor), ReInverseRotMatrix)
            ImEpsTensor = matmul(matmul(ImRotMatrix, ImDiagEpsTensor), ImInverseRotMatrix)


            !< define epsilon tensor
            epsilon_tensor = dcmplx(0.d0, 0.d0)
            epsilon_tensor(1, 1) = dcmplx(ReEpsTensor(1, 1), ImEpsTensor(1, 1))
            epsilon_tensor(1, 2) = dcmplx(ReEpsTensor(1, 2), ImEpsTensor(1, 2))
            epsilon_tensor(2, 1) = dcmplx(ReEpsTensor(2, 1), ImEpsTensor(2, 1))
            epsilon_tensor(2, 2) = dcmplx(ReEpsTensor(2, 2), ImEpsTensor(2, 2))

            return
        end subroutine DetermineEpsTensor


        !>************************************************************************************************************************************************
        !> subroutine: reflectance
        !>
        !> calculate the reflectance within the ac-plane
        !>
        !>
        !> input variables:     angle:              polarization angle
        !>
        !> output variables:    value:              calculated reflectance
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine reflectance(angle, value)

            implicit none
            real*8 :: angle                                                                 !< input variable: polarization angle
            real*8 :: value                                                                 !< output variable: calculated reflectance
            real*8, dimension(2, 2) :: Unit_matrix                                          !< unit matrix
            complex(8) :: eps11, eps12, eps21, eps22                                        !< components of the epsilon tensor
            complex(8), dimension(2) :: dir_vector, help_vector                             !< working variables: help vectors
            complex(8), dimension(2) :: ev1, ev2                                            !< eigenvectors
            complex(8), dimension(2, 2) :: epsilon_tensor                                   !< epsilon tensor
            complex(8), dimension(2, 2) :: Inverse_epsilon_tensor                           !< inverse of epsilon tensor
            complex(8), dimension(2, 2) :: Eigenvectors_epsilon_tensor                      !< eigenvector matrix
            complex(8), dimension(2, 2) :: diagonal_matrix                                  !< diagonal form of the epsilon tensor
            complex(8), dimension(2, 2) :: sqrt_diagonal_matrix                             !< squart-root of diagonal form of the epsilon tensor
            complex(8), dimension(2, 2) :: new_diagonal_matrix                              !< new diagonal form of the epsilon tensor
            complex(8), dimension(2, 2) :: upper_refl_matrix                                !< upper part of the epsilon tensor
            complex(8), dimension(2, 2) :: lower_refl_matrix                                !< lower part of the epsilon tensor
            complex(8), dimension(2, 2) :: Inverse_lower_refl_matrix                        !< inverse of the lower part of the epsilon tensor


            !< reset output variable
            value = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine complex epsilon tensor
            call DetermineEpsTensor(epsilon_tensor)
            eps11 = epsilon_tensor(1, 1)
            eps12 = epsilon_tensor(1, 2)
            eps21 = epsilon_tensor(2, 1)
            eps22 = epsilon_tensor(2, 2)

            ! Debug:
            !   print*,'frequency = ',w
            !   print*, epsilon_tensor(1, 1), epsilon_tensor(1, 2)
            !   print*, epsilon_tensor(2, 1), epsilon_tensor(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine eigenvectors of epsilon tensor (expression taken from Mathematica)


            !< EV1
            ev1(1) = -((-eps11 + eps22 + cdsqrt(eps11**2 + 4 * eps12**2 - 2 * eps11 * eps22 + eps22**2))/(2.d0 *eps12))
            ev1(2) = 1.d0
            ev1(:) = (1.d0/cdsqrt(dot_product(ev1(:), ev1(:)))) * ev1(:)


            !< EV2
            ev2(1) = -((-eps11 + eps22 - cdsqrt(eps11**2 + 4 * eps12**2 - 2 * eps11 * eps22 + eps22**2))/(2.d0 *eps12))
            ev2(2) = 1.d0
            ev2(:) = (1.d0/cdsqrt(dot_product(ev2(:), ev2(:)))) * ev2(:)


            !< construct eigenvector matrix
            Eigenvectors_epsilon_tensor = dcmplx(0.d0, 0.d0)
            Eigenvectors_epsilon_tensor(1, 1) = ev1(1)
            Eigenvectors_epsilon_tensor(2, 1) = ev1(2)
            Eigenvectors_epsilon_tensor(1, 2) = ev2(1)
            Eigenvectors_epsilon_tensor(2, 2) = ev2(2)

            ! Debug:
            !	print*,Eigenvectors_epsilon_tensor(1,1), Eigenvectors_epsilon_tensor(1,2)
            !	print*,Eigenvectors_epsilon_tensor(2,1), Eigenvectors_epsilon_tensor(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine inverse of eigenvector matrix
            Inverse_epsilon_tensor = dcmplx(0.d0, 0.d0)
            call InverseMatrix(Eigenvectors_epsilon_tensor, Inverse_epsilon_tensor)

            ! Debug:
            !	print*, Inverse_epsilon_tensor(1, 1), Inverse_epsilon_tensor(1, 2)
            !	print*, Inverse_epsilon_tensor(2, 1), Inverse_epsilon_tensor(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< bring epsilon tensor on diagonal form
            diagonal_matrix = dcmplx(0.d0, 0.d0)
            diagonal_matrix = matmul(matmul(Inverse_epsilon_tensor, epsilon_tensor), Eigenvectors_epsilon_tensor)

            ! Debug:
            !	print*,diagonal_matrix(1,1), diagonal_matrix(1,2)
            !	print*,diagonal_matrix(2,1), diagonal_matrix(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< take square root of diagonal elements
            sqrt_diagonal_matrix = dcmplx(0.d0, 0.d0)
            sqrt_diagonal_matrix = diagonal_matrix
            sqrt_diagonal_matrix(1, 1) = cdsqrt(sqrt_diagonal_matrix(1, 1))
            sqrt_diagonal_matrix(2, 2) = cdsqrt(sqrt_diagonal_matrix(2, 2))

            ! Debug:
            !	print*, sqrt_diagonal_matrix(1, 1), sqrt_diagonal_matrix(1, 2)
            !	print*, sqrt_diagonal_matrix(2, 1), sqrt_diagonal_matrix(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< rotate matrix back
            new_diagonal_matrix = dcmplx(0.d0, 0.d0)
            new_diagonal_matrix = matmul(matmul(Eigenvectors_epsilon_tensor, sqrt_diagonal_matrix), Inverse_epsilon_tensor)

            ! Debug:
            !	print*, new_diagonal_matrix(1, 1), new_diagonal_matrix(1, 2)
            !	print*, new_diagonal_matrix(2, 1), new_diagonal_matrix(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine upper and lower part of the reflectance matrix


            !< define unity matrix
            Unit_matrix = 0.d0
            Unit_matrix(1, 1) = 1.d0
            Unit_matrix(2, 2) = 1.d0


            !< define upper matrix
            upper_refl_matrix = dcmplx(0.d0,0.d0)
            upper_refl_matrix = Unit_matrix - new_diagonal_matrix


            !< define lower matrix
            lower_refl_matrix = dcmplx(0.d0,0.d0)
            lower_refl_matrix = Unit_matrix + new_diagonal_matrix


            !< determine inverse of lower_refl_matrix
            Inverse_lower_refl_matrix = dcmplx(0.d0,0.d0)
            call InverseMatrix(lower_refl_matrix, Inverse_lower_refl_matrix)

            ! Debug:
            !	print*,upper_refl_matrix(1,1),upper_refl_matrix(1,2)
            !	print*,upper_refl_matrix(2,1),upper_refl_matrix(2,2)
            !	print*,lower_refl_matrix(1,1),lower_refl_matrix(1,2)
            !	print*,lower_refl_matrix(2,1),lower_refl_matrix(2,2)

            !	print*,Inverse_lower_refl_matrix(1,1),Inverse_lower_refl_matrix(1,2)
            !	print*,Inverse_lower_refl_matrix(2,1),Inverse_lower_refl_matrix(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< calulate reflectance
            dir_vector = (/dcmplx(dcos(angle * (pi/180.d0)), 0.d0), dcmplx(dsin(angle * (pi/180.d0)), 0.d0)/)
            help_vector = matmul(matmul(upper_refl_matrix, Inverse_lower_refl_matrix), dir_vector)
            value = dreal(dot_product(help_vector, help_vector))
            return
        end subroutine reflectance


        !>************************************************************************************************************************************************
        !> subroutine: Rp_reflectance
        !>
        !> calculate the p-polarized reflectance
        !>
        !>
        !> input variables:     None
        !>
        !> output variables:    value:              calculated reflectance
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine Rp_reflectance(value)

            implicit none
            real*8 :: value                                                                 !< output variable: calculated reflectance
            real*8, dimension(2, 2) :: RotMatrix, InverseRotMatrix                          !< rotation and inverse of rotation matrix
            complex(8) :: epsxx, epsxz, epszz                                               !< components of the epsilon tensor
            complex(8) :: const                                                             !< working variable: value of the Drude-Lorentz model
            complex(8), dimension(2, 2) :: epsilon_tensor                                   !< epsilon tensor


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine complex epsilon tensor
            call DetermineEpsTensor(epsilon_tensor)

            ! Debug:
            ! print*,'w = ', w
            ! print*,'epsilon_tensor(1, 1) = ', epsilon_tensor(1, 1)
            ! print*,'epsilon_tensor(1, 2) = ', epsilon_tensor(1, 2)
            ! print*,'epsilon_tensor(2, 1) = ', epsilon_tensor(2, 1)
            ! print*,'epsilon_tensor(2, 2) = ', epsilon_tensor(2, 2)


            !< write components of epsilon tensor to files
            ! write(90,*) w, dreal(epsilon_tensor(1, 1))
            ! write(91,*) w, dimag(epsilon_tensor(1, 1))
            ! write(92,*) w, dreal(epsilon_tensor(1, 2))
            ! write(93,*) w, dimag(epsilon_tensor(1, 2))
            ! write(94,*) w, dreal(epsilon_tensor(2, 2))
            ! write(95,*) w, dimag(epsilon_tensor(2, 2))


            !< rotate basis of epsilon tensor
            RotMatrix(1, 1) = dcos(BasisRotAngle)
            RotMatrix(1, 2) = -dsin(BasisRotAngle)
            RotMatrix(2, 1) = dsin(BasisRotAngle)
            RotMatrix(2, 2) = dcos(BasisRotAngle)
            InverseRotMatrix(1, 1) = dcos(BasisRotAngle)
            InverseRotMatrix(1, 2) = dsin(BasisRotAngle)
            InverseRotMatrix(2, 1) = -dsin(BasisRotAngle)
            InverseRotMatrix(2, 2) = dcos(BasisRotAngle)
            epsilon_tensor = matmul(matmul(RotMatrix, epsilon_tensor), InverseRotMatrix)

            epsxx = epsilon_tensor(1, 1)
            epsxz = epsilon_tensor(1, 2)
            epszz = epsilon_tensor(2, 2)


            !< define constant
            const = cdsqrt(epsxx * epszz - epsxz**2)

            ! Debug:
            ! print*,'BasisRotAngle = ', BasisRotAngle
            ! print*,'BasisRotAngle [°] = ', BasisRotAngle * 180.d0/pi
            ! print*,'InfallAngle = ', InfallAngle
            ! print*,'InfallAngle [°] = ', InfallAngle * 180.d0/pi
            ! print*,'epsxx = ', epsxx
            ! print*,'epsxz = ', epsxz
            ! print*,'epszz = ', epszz
            ! print*,'const = ', const


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate Koch model (eq. 23, Chem. Phys. 3, 362 (1974)
            value = cdabs((dcos(InfallAngle) - cdsqrt(epszz - dsin(InfallAngle)**2) * 1.d0/(const)) &
                    / (dcos(InfallAngle) + cdsqrt(epszz - dsin(InfallAngle)**2) * 1.d0/(const)))**2

            if (value > 1.d0) value = 1.d0 / value

            ! Debug:
            ! print*,'value = ', value
            ! stop
            return
        end subroutine Rp_reflectance


        !>************************************************************************************************************************************************
        !> subroutine: InverseMatrix
        !>
        !> calculate the inverse of a 2x2 symmetric complex matrix using analytic functions determined by Mathematica
        !>
        !>
        !> input variables:     matrix:             matrix which has to inverted
        !>
        !> output variables:    Inverse_tensor:     inverted matrix
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine InverseMatrix(matrix, Inverse_tensor)

            implicit none
            complex(8) :: eps11, eps12, eps21, eps22                                        !< working variables: matrix elements
            complex(8), dimension(2, 2) :: matrix                                           !< input variable: matrix
            complex(8), dimension(2, 2) :: Inverse_tensor                                   !< output variable: inverse of matrix

            eps11 = matrix(1, 1)
            eps12 = matrix(1, 2)
            eps21 = matrix(2, 1)
            eps22 = matrix(2, 2)

            Inverse_tensor = dcmplx(0.d0, 0.d0)
            Inverse_tensor(1, 1) =   eps22/(-eps12 * eps21 + eps11 * eps22)
            Inverse_tensor(1, 2) = -(eps12/(-eps12 * eps21 + eps11 * eps22))
            Inverse_tensor(2, 1) = -(eps21/(-eps12 * eps21 + eps11 * eps22))
            Inverse_tensor(2, 2) =   eps11/(-eps12 * eps21 + eps11 * eps22)
            return
        end subroutine InverseMatrix
end Module OpticalProperties
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!>
!>
!> Main Program
!>
!>
program epsTensor
    !< calculation of the refelctivity using the generalized Drude-Lorentz model

    use OpticalProperties

    implicit none
    integer :: i                                                                            !< loop variables
    integer :: NumberXValues                                                                !< number exp. data points
    real*8 :: w                                                                             !< working variable: current frequency w
    real*8 :: refl                                                                          !< working variable: final reflectance for frequency w
    real*8 :: angle_chi                                                                     !< working variable: current polarization angle
    ! character(len=4096) :: PathEpsFiles


    pi = 4.d0 * datan(1.d0)                                                                 !< pi


    !< open files
    open(21,file = "in.txt")                                                                !< open parameter file
    open(22,file = "PolAngle.dat")                                                          !< open experimental-point file
    open(23,file = "output.dat")                                                            !< open file for fit function values


    !< read parameter from file
    read(21,*) NumberXValues                                                                !< read number of exp. data points
    read(21,*) ReRotAngle                                                                   !< read in angle for rotation of real part
    read(21,*) ImRotAngle                                                                   !< read in angle for rotation of imaginary part
    read(21,*) ReEPS11
    read(21,*) ImEPS11
    read(21,*) ReEPS22
    read(21,*) ImEPS22
    read(21,*) BasisRotAngle                                                                !< read in angle for basis rotation
    read(21,*) InfallAngle                                                                  !< read in angle of incidence
    close(21)

    ! Debug:
    ! print*,'eps-Tensor:'
    ! print*,'NumberXValues = ', NumberXValues
    ! print*,'ReRotAngle = ', ReRotAngle
    ! print*,'ImRotAngle = ', ImRotAngle
    ! print*,'ReEPS11 = ', ReEPS11
    ! print*,'ImEPS11 = ', ImEPS11
    ! print*,'ReEPS22 = ', ReEPS22
    ! print*,'ImEPS22 = ', ImEPS22
    ! print*,'BasisRotAngle = ', BasisRotAngle
    ! print*,'InfallAngle = ', InfallAngle
    ! print*,' '


    !< convert angles from degree in radiant
    ReRotAngle = ReRotAngle * (pi/180.d0)
    ImRotAngle = ImRotAngle * (pi/180.d0)
    BasisRotAngle = BasisRotAngle * (pi/180.d0)
    InfallAngle = InfallAngle * (pi/180.d0)


    !< calculate optical properties at any frequency w
    Do i = 1, NumberXValues                                                                 !< loop over all frequency points
        read(22,*) w, angle_chi                                                             !< read frequency and polarization angle form file


        ! Debug:
        ! if (i == 1) then
        !     PathEpsFiles = "/home/tom/ph1/CATS/magix/run/general_Drude-Lorentz-model/MnWO4/10K/eps-Tensor/100-139/"
        !     if (w == 100.03932388000000) then
        !        open(90,file = trim(adjustl(PathEpsFiles)) // "Rp-ReEps11.dat")
        !        open(91,file = trim(adjustl(PathEpsFiles)) // "Rp-ImEps11.dat")
        !        open(92,file = trim(adjustl(PathEpsFiles)) // "Rp-ReEps12.dat")
        !        open(93,file = trim(adjustl(PathEpsFiles)) // "Rp-ImEps12.dat")
        !        open(94,file = trim(adjustl(PathEpsFiles)) // "Rp-ReEps22.dat")
        !        open(95,file = trim(adjustl(PathEpsFiles)) // "Rp-ImEps22.dat")
        !    else
        !        open(90,file = trim(adjustl(PathEpsFiles)) // "Rp-ReEps11.dat", status = 'old', action = 'write', position = 'append')
        !        open(91,file = trim(adjustl(PathEpsFiles)) // "Rp-ImEps11.dat", status = 'old', action = 'write', position = 'append')
        !        open(92,file = trim(adjustl(PathEpsFiles)) // "Rp-ReEps12.dat", status = 'old', action = 'write', position = 'append')
        !        open(93,file = trim(adjustl(PathEpsFiles)) // "Rp-ImEps12.dat", status = 'old', action = 'write', position = 'append')
        !        open(94,file = trim(adjustl(PathEpsFiles)) // "Rp-ReEps22.dat", status = 'old', action = 'write', position = 'append')
        !        open(95,file = trim(adjustl(PathEpsFiles)) // "Rp-ImEps22.dat", status = 'old', action = 'write', position = 'append')
        !    endif
        ! endif


        !< call subroutine for determine reflectance
        if (angle_chi == -1000.d0) then
            call Rp_reflectance(refl)
        else
            call reflectance(angle_chi, refl)                                               !< calculate reflectance
        endif


        !< write fit function value to file
        write(23,*) refl                                                                    !< write reflection to output file
    end Do


    !< close files
    close(22, status = 'delete')                                                            !< close and delete experimental-point file
    close(23)                                                                               !< close output file


    ! Debug:
    ! close(90)
    ! close(91)
    ! close(92)
    ! close(93)
    ! close(94)
    ! close(95)
end program epsTensor
!---------------------------------------------------------------------------------------------------------------------------------------------------------

