# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##
##  Script for installing the XCLASS package
##  Copyright (C) 2012 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function CustomizeXCLASS:                     customize XCLASS installation
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    2013-08-08   initial version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##--------------------------------------------------------------------------------------------------------------------------------------------------------
import os
from . import (task_DatabaseQuery,
               task_GetTransitions,
               task_LineIdentification,
               task_ListDatabase,
               task_LoadASCIIFile,
               task_MAGIX,
               task_myXCLASS,
               task_myXCLASSFit,
               task_myXCLASSMapFit,
               task_myXCLASSMapRedoFit,
               task_myXCLASSPlot,
               task_UpdateDatabase)

__all__ = ['task_DatabaseQuery',
           'task_GetTransitions',
           'task_LineIdentification',
           'task_ListDatabase',
           'task_LoadASCIIFile',
           'task_MAGIX',
           'task_myXCLASS',
           'task_myXCLASSFit',
           'task_myXCLASSMapFit',
           'task_myXCLASSMapRedoFit',
           'task_myXCLASSPlot',
           'task_UpdateDatabase']


## make version available
from .version import __version__


## the following lines (taken from
## https://stackoverflow.com/questions/61939952/mp-set-start-methodspawn-triggered-an-error-saying-the-context-is-already-be)
## are necessary to avoid the error message
##      "TypeError: cannot pickle '_io.TextIOWrapper' object"
## within the myXCLASSMapFit function. Apparently, this error comes
## from a change in Python 3.8 in the multiprocessing library:
## On macOS, the spawn start method is now the default. The fork
## start method should be considered unsafe as it can lead to crashes
## of the subprocess. Taken from
## https://docs.python.org/3.11/library/multiprocessing.html
import multiprocessing as mp
try:
   mp.set_start_method('fork', force = True)
except RuntimeError:
   pass
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## customize XCLASS installation
##
def CustomizeXCLASS(XCLASSRootDir, NoMAGIXFlag, HomeDir):
    """

input parameters:
-----------------

    - XCLASSRootDir:            XCLASS root directory

    - NoMAGIXFlag:              no MAGIX flag

    - HomeDir:                  user home directory



output parameters:
------------------

    - None
    """

    # Debug:
    # print ("XCLASSRootDir = ", XCLASSRootDir)
    # print ("NoMAGIXFlag = ", NoMAGIXFlag)
    # print ("HomeDir = ", HomeDir)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## modify path definitions in various xml files for MAGIX
    if (not NoMAGIXFlag):


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## modify the registration xml file for myXCLASS
        MAGIXRegXML = XCLASSRootDir + "addons/MAGIX/Fit-Functions/myXCLASS/xml/myNewXCLASS.xml"

        # Debug:
        # print ("MAGIXRegXML = ", MAGIXRegXML)


        ## write new path to registration XML file
        PATHExecutable = XCLASSRootDir + "lib/start_myNewXCLASS.py"
        task_MAGIX.WriteXMLtag(MAGIXRegXML, "PathStartScript", [PATHExecutable])


        ## write path of current directory to the execution tag for myXCLASS program
        ExeCommandStartScriptTag = ["python3 start_myNewXCLASS.py " + XCLASSRootDir]
        task_MAGIX.WriteXMLtag(MAGIXRegXML, "ExeCommandStartScript", ExeCommandStartScriptTag)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## modify the registration xml file for conventional Drude Lorentz test model
        MAGIXRegXML = XCLASSRootDir + "addons/MAGIX/Fit-Functions/Drude-Lorentz_conv/"
        MAGIXRegXML += "xml/Conventional_Drude-Lorentz.xml"

        # Debug:
        # print ("MAGIXRegXML = ", MAGIXRegXML)


        ## write new path to registration XML file
        PATHExecutable = XCLASSRootDir + "lib/DrudeLorentzConv.exe"
        task_MAGIX.WriteXMLtag(MAGIXRegXML, "PathStartScript", [PATHExecutable])


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## modify the registration xml file for generalized Drude Lorentz test model
        MAGIXRegXML = XCLASSRootDir + "addons/MAGIX/Fit-Functions/Drude-Lorentz_general/"
        MAGIXRegXML += "xml/Generalized_Drude-Lorentz__sym__freq-damping+Rp.xml"


        ## write new path to registration XML file
        PATHExecutable = XCLASSRootDir
        PATHExecutable += "lib/DrudeLorentzGeneral__sym__freq-damping+Rp.exe"
        task_MAGIX.WriteXMLtag(MAGIXRegXML, "PathStartScript", PATHExecutable)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write XCLASS settings to init.dat file


    ## use appdirs package if available
    XCLASSHomeDir = "{:s}.xclass/".format(HomeDir)


    ## create subfolder in user directory
    os.makedirs(XCLASSHomeDir, exist_ok = True)
    InitFileName = "{:s}init.dat".format(XCLASSHomeDir)


    ## check, if file "init.dat" already exists and import some settings
    TimeStampFormat = "dd-mm-yyyy"
    DatabaseFileName = "{:s}db/cdms_sqlite.db".format(str(XCLASSHomeDir))
    if (os.path.isfile(InitFileName)):
        InitFile = open(InitFileName)
        InitFileConent = InitFile.readlines()
        InitFile.close()
        for line in InitFileConent:
            StrippedLine = line.strip()
            if (StrippedLine.startswith("TimeStampFormat")):
                SplittedLine = StrippedLine.split("=")
                TimeStampFormat = SplittedLine[1].strip()
            # elif (StrippedLine.startswith("DatabaseFileName")):
            #     SplittedLine = StrippedLine.split("=")
            #     DatabaseFileName = SplittedLine[1].strip()


    ## get path for temp files
    MAGIXTempDirectory = str(os.environ.get('MAGIXTempDirectory',''))
    MAGIXTempDirectory = MAGIXTempDirectory.strip()
    if (MAGIXTempDirectory == ""):
        MAGIXTempDirectory = "{:s}temp/".format(XCLASSRootDir)
    elif (MAGIXTempDirectory[-1] != "/"):
        MAGIXTempDirectory = MAGIXTempDirectory + "/"
    os.environ["MAGIXTempDirectory"] = MAGIXTempDirectory


    ## write content init file
    with open(InitFileName, 'w') as InitFile:
        NewLine =  "XCLASS settings:\n"
        NewLine += "----------------\n"
        NewLine += "XCLASSRootDirectory = {:s}\n".format(XCLASSRootDir)
        NewLine += "DatabaseFileName    = {:s}\n".format(DatabaseFileName)
        NewLine += "NoMAGIX             = {:s}\n".format(str(NoMAGIXFlag))
        NewLine += "TimeStampFormat     = {:s}\n".format(TimeStampFormat)
        NewLine += "MAGIXTempDirectory  = {:s}\n".format(MAGIXTempDirectory)
        InitFile.write(NewLine)


    ## define return value
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


## get home directory
HomeDir = os.path.expanduser('~')
HomeDir = os.path.normpath(HomeDir) + "/"

# Debug:
# print ("HomeDir = ", HomeDir)


## get the path of the current package
import xclass
XCLASSRootDir = xclass.__file__
XCLASSRootDir = os.path.dirname(XCLASSRootDir) + "/../xclass/"
XCLASSRootDir = os.path.normpath(XCLASSRootDir) + "/"

# Debug:
# print ("XCLASSRootDir = ", XCLASSRootDir)


NoMAGIXFlag = False


##========================================================================================================================================================
## customize XCLASS installation
CustomizeXCLASS(XCLASSRootDir, NoMAGIXFlag, HomeDir)


## we're done
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
