#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##
##  Script for installing the XCLASS package
##  Copyright (C) 2012 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##  Usage:
##
##      python3 -m pip install --upgrade xclasspip/ -vvv
##
##
##  Further reading:
##
##      https://stackoverflow.com/questions/41169711/python-setuptools-distutils-custom-build-for-the-extra-package-with-makefile
##
##      https://stackoverflow.com/questions/64950460/link-f2py-generated-so-file-in-a-python-package-using-setuptools
##
##
##
##  The following functions are included in this module:
##
##      - function CheckExtLibraries:                   check if external libraries are available
##      - function CompileXCLASS:                       compile XCLASS packages
##      - function CustomizeXCLASS:                     customize XCLASS installation
##      - function Main:                                main program
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    2013-08-08   initial version
##  T. Moeller    2023-09-16   current version 1.4.3
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##--------------------------------------------------------------------------------------------------------------------------------------------------------
import os
import re
import io
import subprocess
from setuptools import setup, find_namespace_packages
from setuptools.command.install import install
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Custom install command that runs CustomizeXCLASS after installation.
##
class XCLASSInstall(install):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## Custom install command that runs CustomizeXCLASS after installation.
    ##
    def run(self):
        """

    input parameters:
    -----------------

        - None



    output parameters:
    ------------------

        - None
        """



        ## First, run normal install steps (copies files etc.)
        super().run()


        ##************************************************************************************************************************************************
        ## Determine the installation root directory of the xclass package
        try:
            import xclass
            XCLASSRootDir = os.path.dirname(xclass.__file__)
        except ImportError:


            ## fallback: use install_lib (where setuptools installed your package)
            XCLASSRootDir = os.path.join(self.install_lib, "xclass")


        ##************************************************************************************************************************************************
        ## Determine user home directory
        HomeDir = os.path.expanduser("~")


        ##************************************************************************************************************************************************
        ## set no-MAGIX flag
        NoMAGIXFlag = False                                                                 ## initialize MAGIX flag
        try:
            EnvNoMAGIXFlag = str(os.environ.get('XCLASS_PIP_INSTALL_NO_MAGIX',''))
        except:
            EnvNoMAGIXFlag = None
        if (EnvNoMAGIXFlag is not None):
            EnvNoMAGIXFlag = EnvNoMAGIXFlag.lower()
            if (EnvNoMAGIXFlag in ["true", "t", "y"]):
                NoMAGIXFlag = True
            else:
                NoMAGIXFlag = False

        # Debug:
        # print ("EnvNoMAGIXFlag = ", EnvNoMAGIXFlag)
        # print ("NoMAGIXFlag = ", NoMAGIXFlag)


        ##************************************************************************************************************************************************
        ## Execute the customization
        CustomizeXCLASS(XCLASSRootDir, NoMAGIXFlag = NoMAGIXFlag, HomeDir = HomeDir)


        ## we're done
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check if external libraries are available
##
def CheckExtLibraries():
    """

input parameters:
-----------------

    - None



output parameters:
------------------

    - StatusFlag:               status of libraries (!=0, something is wrong)
    """


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## initialize status flag of this function
    StatusFlag = 0


    ## define list of external libraries
    ## "libm":          library of basic mathematical functions
    ##                  (required by XCLASS-Interface, myXCLASS, Bees, FunctionCall, Genetic, LM, NS, PSO, SA)
    ## "libz":          dynamic library libz.so.x.y.z (library for data compression, required by cfitsio)
    ##                  (required by XCLASS-Interface, myXCLASS, Bees, FunctionCall, Genetic, LM, NS, PSO, SA)
    ## "libdl":         dynamic linking library
    ##                  (required by XCLASS-Interface, myXCLASS, Bees, FunctionCall, Genetic, LM, NS, PSO, SA)
    ## "libcurl":       dynamic linking library (the multiprotocol file transfer library)
    ##                  (required by CFITSIO)
    ## "libpthread":    POSIX threads library
    ##                  (required by XCLASS-Interface, myXCLASS, Bees, FunctionCall, Genetic, LM, NS, PSO, SA)
    ## "libgomp":       GCC OpenMP shared support library
    ##                  (required by XCLASS-Interface, myXCLASS, Bees, FunctionCall, Genetic, LM, NS, PSO, SA)
    ListOfRequiredLibraries = ["libm",
                               "libz",
                               "libdl",
                               "libcurl",
                               "libpthread",
                               "libgomp"]


    ## print what you do
    print ("Check if external libraries {:s} are available .. ".format(chr(34) + ", ".join(ListOfRequiredLibraries) + chr(34)), \
           end = "", flush = True)


    ## create dummy c program
    DummyCFileName = "test.c"
    DummyCFile = open(DummyCFileName, 'w')
    DummyCFile.write("int main(void) {\n")
    DummyCFile.write("    return 0;\n")
    DummyCFile.write("}\n")
    DummyCFile.close()


    ## check, if library is available
    ## method taken from https://serverfault.com/questions/54736/how-to-check-if-a-library-is-installed
    AllAvailFlag = True
    NotAvailLib = []
    ScreenFileName = "screen.out"
    for LocalLib in ListOfRequiredLibraries:


        ## compile dummy c program
        cmdString = "rm -rf {:s} ; ".format(ScreenFileName)
        cmdString += "gcc {:s} -{:s} > {:s} 2>&1 ; ".format(DummyCFileName, LocalLib.replace("lib", "l"), ScreenFileName)
        subprocess.call(cmdString, shell = True)


        ## import screen output
        ScreenFile = open(ScreenFileName)
        ScreenFileContent = ScreenFile.readlines()
        ScreenFile.close()
        for line in ScreenFileContent:
            SearchString1 = ": -lcurl"
            SearchString2 = "cannot find"
            if (line.find(SearchString1) > (-1) or line.find(SearchString2) > (-1)):
                AllAvailFlag = False
                NotAvailLib.append(LocalLib)
                break


    ## remove dummy c and compiled file
    cmdString = "rm -rf {:s} {:s} a.out; ".format(DummyCFileName, ScreenFileName)
    subprocess.call(cmdString, shell = True)


    ## print error message
    if (not AllAvailFlag):
        print ("\n\n\n\nError in setup.py:")
        if (len(NotAvailLib) == 1):
            print ("\n\tThe external library")
            print ("\n\t\t{:s}".format(NotAvailLib[0]))
            print ("\n\tis not available!")
            print ("\n\tPlease make sure that the library is installed and that the path is included in the LD_LIBRARY_PATH variable.\n\n\n")
        else:
            print ("\n\tThe external libraries")
            for LocalLib in NotAvailLib:
                print ("\n\t\t{:s}".format(LocalLib))
            print ("\n\tare not available!")
            print ("\n\tPlease make sure that the libraries are installed and that the paths are included in the LD_LIBRARY_PATH variable.\n\n\n")
        StatusFlag = 1
        return StatusFlag
    else:
        print ("done!\n")


    ## define return value
    return StatusFlag
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## compile XCLASS packages
##
def CompileXCLASS(XCLASSRootDir, LibPATH, CompileFlag, MAGIXCompilationFlag, NoMAGIXFlag, NoExtLIBFlag, ToSCREENFlag, static_lib = True):
    """

input parameters:
-----------------

    - XCLASSRootDir:            current directory

    - LibPATH:                  path of subdirectory used for compiled binaries

    - CompileFlag:              flag indicating compilation

    - MAGIXCompilationFlag:     parallelization option

    - NoMAGIXFlag:              no MAGIX flag

    - NoExtLIBFlag:             flag indicating compilation of external libraries

    - ToSCREENFlag:             flag indicating

    - static_lib:               (optional) flag for compiling static libraries (default: False)



output parameters:
------------------

    - StatusFlag:               status of compilation
    """

    # Debug:
    # print ("XCLASSRootDir = ", XCLASSRootDir)
    # print ("LibPATH = ", LibPATH)
    # print ("CompileFlag = ", CompileFlag)
    # print ("MAGIXCompilationFlag = ", MAGIXCompilationFlag)
    # print ("NoMAGIXFlag = ", NoMAGIXFlag)
    # print ("NoExtLIBFlag = ", NoExtLIBFlag)
    # print ("ToSCREENFlag = ", ToSCREENFlag)
    # print ("static_lib = ", static_lib)


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## initialize status flag of this function
    StatusFlag = 0


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## create path for compiled
    if (not os.path.exists(LibPATH)):
        cmdString = "rm -rf {:s} ; ".format(LibPATH)
        cmdString += "mkdir -p {:s} ; ".format(LibPATH)
        subprocess.call(cmdString, shell = True)

    # Debug:
    # print ("LibPATH = ", LibPATH)


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## check, if external libraries (fftw3 and cfitsio have to be compiled as well)
    if (not NoExtLIBFlag):


        ##************************************************************************************************************************************************
        ## check if compilers and mpi is available


        ## get command for C- and FORTRAN compiler
        gcc_var = os.getenv('CC')
        if (gcc_var is None):
            gcc_var = "gcc"
        fortran_var = os.getenv('FC')
        if (fortran_var is None):
            fortran_var = "gfortran"

        # Debug:
        # print ("gcc_var = ", gcc_var)
        # print ("fortran_var = ", fortran_var)


        ## check, if gcc and gfortran compilers are available
        print ("Check, if gcc and gfortran compilers are available .. ", end = "")
        ListOfWarnings = []
        ListOfCommands = ["{:s} -v".format(gcc_var), "{:s} -v".format(fortran_var)]
        for CompilerID, LocalCommand in enumerate(ListOfCommands):
            cmdString = LocalCommand + " > tmp.out 2>&1"
            os.system(cmdString)
            tmpFile = open("tmp.out")
            CompilerInfo = tmpFile.readlines()
            tmpFile.close()
            os.remove("tmp.out")
            for line in CompilerInfo:
                if (line.find("not found") > (-1)):
                    if (CompilerID == 0):
                        ListOfWarnings.append("gcc")
                    else:
                        ListOfWarnings.append("gfortran")
        if (ListOfWarnings != []):
            print ("\n\n\nError in the installation script for XCLASS package!")
            print ("\n\tCan not find " + ListOfWarnings[0] + "", end = ' ')
            if (len(ListOfWarnings) == 2):
                print ("and " + ListOfWarnings[1] + " ", end = ' ')
            print ("compiler!")
            print ("\n\tPlease install compiler and re-execute XCLASS installation script!\n\n")
            StatusFlag = 1
            return StatusFlag
        else:
            print ("done!")


        ## check, if gfortran version 10 or newer is available
        print ("Check, if gfortran version 10 or newer is available .. ", end = "")
        cmdString = "gfortran -fallow-argument-mismatch > tmp.out 2>&1"
        os.system(cmdString)
        tmpFile = open("tmp.out")
        CompilerInfo = tmpFile.readlines()
        tmpFile.close()
        os.remove("tmp.out")
        XCLASS_EXTRA_FORTRAN_FLAG = "{:s}-fallow-argument-mismatch{:s}".format(chr(34), chr(34))
        for line in CompilerInfo:
            if (line.find("unrecognized command line option") > (-1) \
                or line.find("Fehler: nicht erkannte Kommandozeilenoption") > (-1)):
                XCLASS_EXTRA_FORTRAN_FLAG = "{:s}{:s}".format(chr(34), chr(34))
                break
        print ("done!")


        ## check, if OpenMPI is available if --mpi is selected
        if (MAGIXCompilationFlag in ["mpi"] and not NoMAGIXFlag):
            print ("Check, if OpenMPI package is available .. ", end = "")


            ## use ompi_info to test OpenMPI installation
            cmdString = "ompi_info > tmp.out 2>&1"
            os.system(cmdString)
            tmpFile = open("tmp.out")
            CompilerInfo = tmpFile.readlines()
            tmpFile.close()
            os.remove("tmp.out")
            LocalErrorFlag = False
            for line in CompilerInfo:
                if (line.find("not found") > (-1)):
                    print ("\n\n\nError in the installation script for XCLASS package!")
                    print ("\n\tOpenMPI package is not available!")
                    print ("\n\tPlease install OpenMPI and re-execute XCLASS installation script!\n\n")
                    LocalErrorFlag = True
                    break
            if (LocalErrorFlag):
                StatusFlag = 1
                return StatusFlag


            ## test mpif90 command
            cmdString = "mpif90 -v > tmp.out 2>&1"
            os.system(cmdString)
            tmpFile = open("tmp.out")
            CompilerInfo = tmpFile.readlines()
            tmpFile.close()
            os.remove("tmp.out")
            LocalErrorFlag = False
            for line in CompilerInfo:
                if (line.find("not found") > (-1)):
                    print ("\n\n\nError in the installation script for XCLASS package!")
                    print ("\n\tCan not find OpenMPI compiler!")
                    print ("\n\tPlease install compiler and re-execute XCLASS installation script!\n\n")
                    LocalErrorFlag = True
                    break
            if (LocalErrorFlag):
                StatusFlag = 1
                return StatusFlag


            ## we're done
            print ("done!")


        ##************************************************************************************************************************************************
        ## define path and name of external libraries


        ## set name of cfitsio library file
        cfitsioPath = XCLASSRootDir + "xclass/external/CFITSIO/src/"
        if (static_lib):
            cfitsio_lib_pure_file_name = "libcfitsio.a"
        else:
            cfitsio_lib_pure_file_name = "libcfitsio.so.10.4.3.0"
        cfitsioLIBFileName = os.path.normpath(cfitsioPath + "../bin/{:s}".format(cfitsio_lib_pure_file_name))

        # Debug:
        # print ("cfitsio_lib_pure_file_name = ", cfitsio_lib_pure_file_name)
        # print ("cfitsioLIBFileName = ", cfitsioLIBFileName)


        ## set names of fftw3 library file
        fftw3Path = XCLASSRootDir + "xclass/external/FFTW/src/"
        if (static_lib):
            fftw3_lib_pure_file_name = "libfftw3.a"
        else:
            fftw3_lib_pure_file_name = "libfftw3.so.3.6.10"
        fftw3LIBFileName = os.path.normpath(fftw3Path + "../bin/{:s}".format(fftw3_lib_pure_file_name))

        # Debug:
        # print ("fftw3_lib_pure_file_name = ", fftw3_lib_pure_file_name)
        # print ("fftw3LIBFileName = ", fftw3LIBFileName)


        ## set names of SQLite3 library file
        sqlite3Path = XCLASSRootDir + "xclass/external/SQLite/"
        if (static_lib):
            sqlite3_lib_pure_file_name = "libsqlitex.a"
        else:
            sqlite3_lib_pure_file_name = "libsqlitex.so"
        sqlite3LIBFileName = os.path.normpath(sqlite3Path + "bin/{:s}".format(sqlite3_lib_pure_file_name))

        # Debug:
        # print ("sqlite3_lib_pure_file_name = ", sqlite3_lib_pure_file_name)
        # print ("sqlite3LIBFileName = ", sqlite3LIBFileName)


        ##************************************************************************************************************************************************
        ## cfitsio library
        if (CompileFlag):


            ## print what you do
            print ("\nCompile external library {:s} .. ".format(chr(34) + "cfitsio" + chr(34)), flush = True)


            ## unzip file
            ExtLibcmdStringCFITSIO = "cd {:s}xclass/external/ ; ".format(XCLASSRootDir)
            ExtLibcmdStringCFITSIO += "rm -rf CFITSIO/ ; "
            ExtLibcmdStringCFITSIO += "mkdir -p CFITSIO/bin/ ; "
            ExtLibcmdStringCFITSIO += "mkdir -p CFITSIO/src/ ; "
            ExtLibcmdStringCFITSIO += "tar xzf cfitsio-4.3.0.tar.gz ; "
            ExtLibcmdStringCFITSIO += "cp -r cfitsio-4.3.0/* CFITSIO/src/ ; "
            ExtLibcmdStringCFITSIO += "rm -rf cfitsio-4.3.0/ ; "
            ExtLibcmdStringCFITSIO += "cd {:s} ; ".format(XCLASSRootDir)
            if (CompileFlag):
                subprocess.call(ExtLibcmdStringCFITSIO, shell = True)


            ## compile cfitsio library
            if (ToSCREENFlag):
                OutputString = ""
                cmdString = ""
            else:
                OutputString = " >> {:s}cfitsio.out 2>&1".format(XCLASSRootDir)
                cmdString = "rm -rf {:s}cfitsio.out; ".format(XCLASSRootDir)
            cmdString += "rm -rf {:s} ; ".format(cfitsioLIBFileName)
            cmdString += "cd {:s} ; ".format(cfitsioPath)
            # cmdString += "./configure --disable-curl --prefix={:s} {:s} ; ".format(cfitsioPath, OutputString)
            cmdString += "./configure --prefix={:s} {:s} ; ".format(cfitsioPath, OutputString)
            subprocess.call(cmdString, shell = True)
            cmdString = "cd {:s} ; ".format(cfitsioPath)
            cmdString += "make {:s} ; ".format(OutputString)
            subprocess.call(cmdString, shell = True)
            cmdString = "cd {:s} ; ".format(cfitsioPath)
            cmdString += "make install {:s} ; ".format(OutputString)
            subprocess.call(cmdString, shell = True)
            LocalLib64 = "{:s}lib64/{:s}".format(cfitsioPath, cfitsio_lib_pure_file_name)
            LocalLib32 = "{:s}lib/{:s}".format(cfitsioPath, cfitsio_lib_pure_file_name)
            cmdString = "cd {:s} ; ".format(cfitsioPath)
            if (os.path.isfile(LocalLib64)):
                cmdString += "mv {:s} {:s} ; ".format(LocalLib64, cfitsioLIBFileName)
            elif (os.path.isfile(LocalLib32)):
                cmdString += "mv {:s} {:s} ; ".format(LocalLib32, cfitsioLIBFileName)
            cmdString += "make clean {:s} ; ".format(OutputString)
            cmdString += "cd {:s}".format(XCLASSRootDir)
            subprocess.call(cmdString, shell = True)


            ## check, if compilation of cfitsio library was successfully
            if (not os.path.isfile(cfitsioLIBFileName)):                                    ## check, if library file is created
                print ("\n\n\nError in the installation script for XCLASS package!")
                print ("\n\tAn error occurred in the compilation of the cfitsio library!")
                print ("\n\tPlease check the screen output of the compilation", end = "", flush = True)
                if (not ToSCREENFlag):
                    print (" which was written to file cfitsio.out", end = "", flush = True)
                print ("!\n\n")
                StatusFlag = 1
                return StatusFlag
            else:
                if (not ToSCREENFlag):
                    cmdString = "rm -rf {:s}cfitsio.out".format(XCLASSRootDir)
                    subprocess.call(cmdString, shell = True)


            ## create symbolic link
            if (not static_lib):
                cmdString = "cd {:s}../bin/ ; ".format(cfitsioPath)
                cmdString += "ln -s {:s} libcfitsio.so ; ".format(cfitsioLIBFileName)
                cmdString += "ln -s {:s} libcfitsio.so.6 ; ".format(cfitsioLIBFileName)     ## DON'T ASK WHY THIS IS NECESSARY !!!!
                cmdString += "cd {:s}".format(XCLASSRootDir)
                subprocess.call(cmdString, shell = True)


            # ## copy compiled library file to lib directory
            # cmdString = "chmod u+x {:s} ; ".format(cfitsioLIBFileName)
            # cmdString += "cp {:s} {:s} ; ".format(cfitsioLIBFileName, LibPATH)
            # subprocess.call(cmdString, shell = True)


            ## create symbolic link
            if (not static_lib):
                cmdString = "cd {:s}../bin/ ; ".format(cfitsioPath)
                cmdString += "ln -s {:s} libcfitsio.so ; ".format(cfitsioLIBFileName)
                cmdString += "ln -s {:s} libcfitsio.so.6 ; ".format(cfitsioLIBFileName)     ## DON'T ASK WHY THIS IS NECESSARY !!!!
                cmdString += "cd {:s}".format(XCLASSRootDir)
                subprocess.call(cmdString, shell = True)


            # ## copy compiled library file to lib directory
            # cmdString = "chmod u+x {:s} ; ".format(cfitsioLIBFileName)
            # cmdString += "cp {:s} {:s} ; ".format(cfitsioLIBFileName, LibPATH)
            # subprocess.call(cmdString, shell = True)


        ##************************************************************************************************************************************************
        ## fftw3 library
        if (CompileFlag):


            ## print what you do
            print ("Compile external library {:s} .. ".format(chr(34) + "fftw3" + chr(34)), flush = True)


            ## unzip file
            ExtLibcmdStringFFTW3 = "cd {:s}xclass/external/ ; ".format(XCLASSRootDir)
            ExtLibcmdStringFFTW3 += "rm -rf FFTW/ ; "
            ExtLibcmdStringFFTW3 += "mkdir -p FFTW/bin/ ; "
            ExtLibcmdStringFFTW3 += "mkdir -p FFTW/src/ ; "
            ExtLibcmdStringFFTW3 += "tar xzf fftw-3.3.10.tar.gz ; "
            ExtLibcmdStringFFTW3 += "cp -r fftw-3.3.10/* FFTW/src/ ; "
            ExtLibcmdStringFFTW3 += "rm -rf fftw-3.3.10/ ; "
            ExtLibcmdStringFFTW3 += "cd {:s} ; ".format(XCLASSRootDir)
            subprocess.call(ExtLibcmdStringFFTW3, shell = True)


            ## compile fftw3 library
            if (ToSCREENFlag):
                OutputString = ""
                cmdString = ""
            else:
                OutputString = " >> {:s}fftw3.out 2>&1".format(XCLASSRootDir)
                cmdString = "rm -rf {:s} {:s}fftw3.out; ".format(fftw3LIBFileName, XCLASSRootDir)
            cmdString += "cd {:s} ; ".format(fftw3Path)
            if (static_lib):
                cmdString += "./configure --enable-static --disable-shared --prefix={:s} {:s}; ".format(fftw3Path, OutputString)
            else:
                cmdString += "./configure --enable-shared --prefix={:s} {:s}; ".format(fftw3Path, OutputString)
            cmdString += "make CFLAGS=-fPIC {:s} ; ".format(OutputString)
            cmdString += "make install {:s} ; ".format(OutputString)
            subprocess.call(cmdString, shell = True)


            ## move compiled library file to final destination
            if (os.path.isfile("{:s}lib/{:s}".format(fftw3Path, fftw3_lib_pure_file_name))):
                cmdString = "mv {:s}lib/{:s} {:s} ; ".format(fftw3Path, fftw3_lib_pure_file_name, fftw3LIBFileName)
                subprocess.call(cmdString, shell = True)
            elif (os.path.isfile("{:s}lib64/{:s}".format(fftw3Path, fftw3_lib_pure_file_name))):
                cmdString = "mv {:s}lib64/{:s} {:s} ; ".format(fftw3Path, fftw3_lib_pure_file_name, fftw3LIBFileName)
                subprocess.call(cmdString, shell = True)


            ## clean up
            cmdString = ""
            # cmdString += "make clean {:s} ; ".format(OutputString)
            cmdString += "cd {:s}".format(XCLASSRootDir)
            subprocess.call(cmdString, shell = True)


            ## check, if compilation of fftw3 library was successfully
            if (not os.path.isfile(fftw3LIBFileName)):                                      ## check, if library file is created
                print ("\n\n\nError in the installation script for XCLASS package!")
                print ("\n\tAn error occurred in the compilation of the fftw3 library!")
                print ("\n\tPlease check the screen output of the compilation", end = "", flush = True)
                if (not ToSCREENFlag):
                    print (" which was written to file fftw3.out", end = "", flush = True)
                print ("!\n\n")
                StatusFlag = 1
                return StatusFlag
            else:
                if (not ToSCREENFlag):
                    cmdString = "rm -rf {:s}fftw3.out".format(XCLASSRootDir)
                    subprocess.call(cmdString, shell = True)


            ## create symbolic link
            if (not static_lib):
                cmdString = "cd {:s}../bin/ ; ".format(fftw3Path)
                cmdString += "ln -s {:s} libfftw3.so ; ".format(fftw3LIBFileName)
                cmdString += "cd {:s}".format(XCLASSRootDir)
                subprocess.call(cmdString, shell = True)


            ## copy compiled library file to lib directory
            # cmdString = "chmod u+x {:s} ; ".format(fftw3LIBFileName)
            # cmdString += "cp {:s} {:s} ; ".format(fftw3LIBFileName, LibPATH)
            # subprocess.call(cmdString, shell = True)


        ##************************************************************************************************************************************************
        ## SQLite3 library
        if (CompileFlag):


            ## print what you do
            print ("Compile external library {:s} .. ".format(chr(34) + "SQLite3" + chr(34)), flush = True)


            ## compile SQLite3 library
            if (ToSCREENFlag):
                OutputString = ""
                cmdString = ""
            else:
                OutputString = " > {:s}sqlite3.out 2>&1".format(XCLASSRootDir)
                cmdString = "rm -rf {:s} {:s}sqlite3.out; ".format(sqlite3LIBFileName, XCLASSRootDir)
            cmdString += "cd {:s} ; ".format(sqlite3Path)
            cmdString += "rm -rf bin/ ; "
            cmdString += "mkdir bin/ ; "
            if (static_lib):
                cmdString += "make all {:s} ; ".format(OutputString)
            else:
                cmdString += "make allso {:s} ; ".format(OutputString)
            cmdString += "cd {:s}".format(XCLASSRootDir)
            subprocess.call(cmdString, shell = True)


            ## check, if compilation of SQLite3 library was successfully
            if (not os.path.isfile(sqlite3LIBFileName)):                                    ## check, if library file is created
                print ("\n\n\nError in the installation script for XCLASS package!")
                print ("\n\tAn error occurred in the compilation of the SQLite3 library!")
                print ("\n\tPlease check the screen output of the compilation", end = "", flush = True)
                if (not ToSCREENFlag):
                    print (" which was written to file sqlite3.out", end = "", flush = True)
                print ("!\n\n")
                StatusFlag = 1
                return StatusFlag
            else:
                if (not ToSCREENFlag):
                    cmdString = "rm -rf {:s}sqlite3.out".format(XCLASSRootDir)
                    subprocess.call(cmdString, shell = True)
                # print ("done!\n")


            ## copy compiled library file to lib directory
            # cmdString = "chmod u+x {:s} ; ".format(sqlite3LIBFileName)
            # cmdString += "cp {:s} {:s} ; ".format(sqlite3LIBFileName, LibPATH)
            # subprocess.call(cmdString, shell = True)


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## XCLASS core


    ## compile XCLASS core routines
    XCLASSCorePath = "{:s}xclass/xclass/core/".format(XCLASSRootDir)
    XCLASSCoreObjPath = "{:s}obj/".format(XCLASSCorePath)
    XCLASSCoreLIBFileName = "{:s}bin/libxclass.a".format(XCLASSCorePath)
    if (CompileFlag):
        print ("Compile XCLASS core routines .. ", end = "", flush = True)
        if (ToSCREENFlag):
            OutputString = ""
            cmdString = ""
        else:
            OutputString = " > {:s}xclasscore.out 2>&1".format(XCLASSRootDir)
            cmdString = "rm -rf {:s}xclasscore.out ; ".format(XCLASSRootDir)
        cmdString += "rm -rf {:s}* ; ".format(XCLASSCoreObjPath)
        cmdString += "rm -rf {:s} ; ".format(XCLASSCoreLIBFileName)
        cmdString += "cd {:s} ; ".format(XCLASSCorePath)
        cmdString += "rm -rf obj/ ; "
        cmdString += "mkdir obj/ ; "
        cmdString += "export XCLASS_EXTRA_FORTRAN_FLAG={:s} ; ".format(XCLASS_EXTRA_FORTRAN_FLAG)
        cmdString += "make build {:s} ; ".format(OutputString)
        cmdString += "cd {:s}".format(XCLASSRootDir)
        subprocess.call(cmdString, shell = True)


        ## check, if compilation of XCLASS core routines was successfully
        XCLASSTestFileName = "{:s}Module_myXCLASS__Core.o".format(XCLASSCoreObjPath)
        if (not os.path.isfile(XCLASSTestFileName)):
            print ("\n\n\nError in the installation script for XCLASS package!")
            print ("\n\tAn error occurred in the compilation of the XCLASS core routines!")
            print ("\n\tPlease check the screen output of the compilation", end = "", flush = True)
            if (not ToSCREENFlag):
                print (" which was written to file xclasscore.out", end = "", flush = True)
            print ("!\n\n")
            StatusFlag = 1
            return StatusFlag
        else:
            if (not ToSCREENFlag):
                cmdString = "rm -rf {:s}xclasscore.out".format(XCLASSRootDir)
                subprocess.call(cmdString, shell = True)
            print ("done!\n")


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## XCLASS interface


    ## compile source files of XCLASS interface
    XCLASSInterfacePath = "{:s}xclass/xclass/interface/".format(XCLASSRootDir)
    XCLASSInterfaceLIBFileName = "{:s}bin/xclassinterface.so".format(XCLASSInterfacePath)
    if (CompileFlag):
        print ("Compile source files of XCLASS interface .. ", end = "", flush = True)
        if (ToSCREENFlag):
            OutputString = ""
            cmdString = ""
        else:
            OutputString = " > {:s}xclass_interface.out 2>&1".format(XCLASSRootDir)
            cmdString = "rm -rf {:s}xclass_interface.out ; ".format(XCLASSRootDir)
        cmdString += "rm -rf {:s} ; ".format(XCLASSInterfaceLIBFileName)
        cmdString += "cd {:s} ; ".format(XCLASSInterfacePath)
        cmdString += "rm -rf bin/ ; "
        cmdString += "mkdir bin/ ; "
        cmdString += "export XCLASS_EXTRA_FORTRAN_FLAG={:s} ; ".format(XCLASS_EXTRA_FORTRAN_FLAG)
        # cmdString += "make all {:s} ; ".format(OutputString)
        cmdString += "make new {:s} ; ".format(OutputString)
        cmdString += "cp {:s} {:s} ; ".format(XCLASSInterfaceLIBFileName, LibPATH)
        cmdString += "cd {:s}".format(XCLASSRootDir)
        subprocess.call(cmdString, shell = True)


        ## check, if compilation of XCLASS interface was successfully
        if (not os.path.isfile(XCLASSInterfaceLIBFileName)):
            print ("\n\n\nError in the installation script for XCLASS package!")
            print ("\n\tAn error occurred in the compilation of the XCLASS interface!")
            print ("\n\tPlease check the screen output of the compilation", end = "", flush = True)
            if (not ToSCREENFlag):
                print (" which was written to file xclass_interface.out", end = "", flush = True)
            print ("!\n\n")
            StatusFlag = 1
            return StatusFlag
        else:
            if (not ToSCREENFlag):
                cmdString = "rm -rf {:s}xclass_interface.out".format(XCLASSRootDir)
                subprocess.call(cmdString, shell = True)
            print ("done!\n")


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## XCLASS lite interface


    ## compile source files of XCLASS lite interface
    XCLASSLitePath = "{:s}xclass/xclass/lite/".format(XCLASSRootDir)
    XCLASSLiteLIBFileName = "{:s}bin/xclasslite.so".format(XCLASSLitePath)
    if (CompileFlag):
        print ("Compile source files of XCLASS lite interface .. ", end = "", flush = True)
        if (ToSCREENFlag):
            OutputString = ""
            cmdString = ""
        else:
            OutputString = " > {:s}xclass_lite.out 2>&1".format(XCLASSRootDir)
            cmdString = "rm -rf {:s}xclass_lite.out ; ".format(XCLASSRootDir)
        cmdString += "rm -rf {:s} ; ".format(XCLASSLiteLIBFileName)
        cmdString += "cd {:s} ; ".format(XCLASSLitePath)
        cmdString += "rm -rf bin/ ; "
        cmdString += "mkdir bin/ ; "
        cmdString += "export XCLASS_EXTRA_FORTRAN_FLAG={:s} ; ".format(XCLASS_EXTRA_FORTRAN_FLAG)
        cmdString += "make all {:s} ; ".format(OutputString)
        cmdString += "cp {:s} {:s} ; ".format(XCLASSLiteLIBFileName, LibPATH)
        cmdString += "cd {:s}".format(XCLASSRootDir)
        subprocess.call(cmdString, shell = True)


        ## check, if compilation of XCLASS lite interface was successfully
        if (not os.path.isfile(XCLASSLiteLIBFileName)):
            print ("\n\n\nError in the installation script for XCLASS package!")
            print ("\n\tAn error occurred in the compilation of the XCLASS lite interface!")
            print ("\n\tPlease check the screen output of the compilation", end = "", flush = True)
            if (not ToSCREENFlag):
                print (" which was written to file xclass_lite.out", end = "", flush = True)
            print ("!\n\n")
            StatusFlag = 1
            return StatusFlag
        else:
            if (not ToSCREENFlag):
                cmdString = "rm -rf {:s}xclass_lite.out".format(XCLASSRootDir)
                subprocess.call(cmdString, shell = True)
            print ("done!\n")


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## myXCLASS


    ## compile source files of myXCLASS
    myXCLASSPath = "{:s}xclass/addons/myXCLASS/".format(XCLASSRootDir)
    myXCLASSExeFileName = "{:s}bin/myNewXCLASS.exe".format(myXCLASSPath)
    if (CompileFlag):
        print ("Compile source files of myXCLASS .. ", end = "", flush = True)
        if (ToSCREENFlag):
            OutputString = ""
            cmdString = ""
        else:
            OutputString = " >> {:s}myXCLASS.out 2>&1".format(XCLASSRootDir)
            cmdString = "rm -rf {:s}myXCLASS.out ; ".format(XCLASSRootDir)
        cmdString += "rm -rf {:s} ; ".format(myXCLASSExeFileName)
        cmdString += "cd {:s} ; ".format(myXCLASSPath)
        cmdString += "rm -rf bin/ ; "
        cmdString += "mkdir bin/ ; "
        cmdString += "export XCLASS_EXTRA_FORTRAN_FLAG={:s} ; ".format(XCLASS_EXTRA_FORTRAN_FLAG)
        cmdString += "make all {:s} ; ".format(OutputString)
        cmdString += "cp {:s} {:s} ; ".format(myXCLASSExeFileName, LibPATH)
        cmdString += "cd {:s}".format(XCLASSRootDir)
        subprocess.call(cmdString, shell = True)


        ## check, if compilation of myXCLASS was successfully
        if (not os.path.isfile(myXCLASSExeFileName)):
            print ("\n\n\nError in the installation script for XCLASS package!")
            print ("\n\tAn error occurred in the compilation of the myXCLASS package!")
            print ("\n\tPlease check the screen output of the compilation", end = "", flush = True)
            if (not ToSCREENFlag):
                print (" which was written to file myXCLASS.out", end = "", flush = True)
            print ("!\n\n")
            StatusFlag = 1
            return StatusFlag
        else:
            if (not ToSCREENFlag):
                cmdString = "rm -rf {:s}myXCLASS.out".format(XCLASSRootDir)
                subprocess.call(cmdString, shell = True)
            print ("done!\n")


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## compile MAGIX and external test programs
    if (CompileFlag and (not NoMAGIXFlag)):


        ## compile source files of MAGIX
        print ("Compile MAGIX .. ", end = "", flush = True)
        MAGIXPath = "{:s}xclass/addons/MAGIX/".format(XCLASSRootDir)
        if (MAGIXCompilationFlag in ["mpi"]):
            MAGIXExeFileNames = ["{:s}Modules/Bees/bin/Bees__MPI-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/Bees/bin/Bees__MPI-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/Genetic/bin/Genetic__MPI-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/Genetic/bin/Genetic__MPI-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/LM/bin/LM__MPI-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/LM/bin/LM__MPI-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/FunctionCall/bin/FunctionCall__MPI-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/FunctionCall/bin/FunctionCall__MPI-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/NS/bin/NS__MPI-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/NS/bin/NS__MPI-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/PSO/bin/PSO__MPI-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/PSO/bin/PSO__MPI-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/SA/bin/SA__MPI-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/SA/bin/SA__MPI-Starter__myXCLASS.exe".format(MAGIXPath)]
        else:
            MAGIXExeFileNames = ["{:s}Modules/Bees/bin/Bees__SMP-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/Bees/bin/Bees__SMP-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/Genetic/bin/Genetic__SMP-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/Genetic/bin/Genetic__SMP-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/LM/bin/LM__SMP-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/LM/bin/LM__SMP-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/FunctionCall/bin/FunctionCall__SMP-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/FunctionCall/bin/FunctionCall__SMP-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/NS/bin/NS__SMP-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/NS/bin/NS__SMP-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/PSO/bin/PSO__SMP-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/PSO/bin/PSO__SMP-Starter__myXCLASS.exe".format(MAGIXPath),
                                 "{:s}Modules/SA/bin/SA__SMP-Starter.exe".format(MAGIXPath),
                                 "{:s}Modules/SA/bin/SA__SMP-Starter__myXCLASS.exe".format(MAGIXPath)]
        MAGIXExeFileNames = MAGIXExeFileNames + ["{:s}Fit-Functions/Drude-Lorentz_conv/bin/DrudeLorentzConv.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__asym.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__asym__Koch-model.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__asym__var-dielectr-Fkt.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/Drude-Lorentz_general__eps-Tensor.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/Drude-Lorentz_general__eps-Tensor__final-eps.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__extended.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__extended__new.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__factor.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__imag.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__Menzel.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__sym.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__sym_all.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__sym_eps-fit.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__sym__freq-damping.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__sym__freq-damping__eps-Func.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__sym__freq-damping+Rp.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/Drude-Lorentz_general__sym__freq-damping+var-dielectr-Fkt.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__sym__Koch-model.exe".format(MAGIXPath),
                                                 "{:s}Fit-Functions/Drude-Lorentz_general/bin/DrudeLorentzGeneral__sym__var-dielectr-Fkt.exe".format(MAGIXPath)]
        if (ToSCREENFlag):
            OutputString = ""
            cmdString = ""
        else:
            OutputString = " > {:s}magix.out 2>&1".format(XCLASSRootDir)
            cmdString = "rm -rf {:s}magix.out ; ".format(XCLASSRootDir)
        for LocalMAGIXExeFileName in MAGIXExeFileNames:
            cmdString += "rm -rf {:s} ; ".format(LocalMAGIXExeFileName)
        cmdString += "cd {:s} ; ".format(MAGIXPath)
        cmdString += "export XCLASS_EXTRA_FORTRAN_FLAG={:s} ; ".format(XCLASS_EXTRA_FORTRAN_FLAG)
        cmdString += "sh install.sh --{:s} {:s} ; ".format(MAGIXCompilationFlag, OutputString)
        cmdString += "cd {:s}".format(XCLASSRootDir)
        subprocess.call(cmdString, shell = True)


        ## check, if compilation of MAGIX was successfully
        ErrorList = []
        for LocalMAGIXExeFileName in MAGIXExeFileNames:
            if (not os.path.isfile(LocalMAGIXExeFileName)):
                ErrorList.append(LocalMAGIXExeFileName)
            else:
                cmdString = "cp {:s} {:s} ; ".format(LocalMAGIXExeFileName, LibPATH)
                subprocess.call(cmdString, shell = True)
        if (len(ErrorList) > 0):
            print ("\n\n\nError in the installation script for XCLASS package!")
            print ("\n\tAn error occurred in the compilation of the MAGIX package!")
            print ("\n\n\tErrorList = ", ErrorList)
            print ("\n\tPlease check the screen output of the compilation", end = "", flush = True)
            if (not ToSCREENFlag):
                print (" which was written to file magix.out", end = "", flush = True)
            print ("!\n\n")
            StatusFlag = 1
            return StatusFlag
        else:
            if (not ToSCREENFlag):
                cmdString = "rm -rf {:s}magix.out".format(XCLASSRootDir)
                subprocess.call(cmdString, shell = True)
            print ("done!\n")


    ## define return value
    return StatusFlag
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## customize XCLASS installation
##
def CustomizeXCLASS(XCLASSRootDir, NoMAGIXFlag, HomeDir):
    """

input parameters:
-----------------

    - XCLASSRootDir:            XCLASS root directory

    - NoMAGIXFlag:              no MAGIX flag

    - HomeDir:                  user home directory



output parameters:
------------------

    - None
    """

    # Debug:
    # print ("XCLASSRootDir = ", XCLASSRootDir)
    # print ("NoMAGIXFlag = ", NoMAGIXFlag)
    # print ("HomeDir = ", HomeDir)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write XCLASS settings to init.dat file


    ## print what you do
    print ("Write XCLASS settings to init.dat file .. ", end = "", flush = True)


    ## use appdirs package if available
    XCLASSHomeDir = "{:s}.xclass/".format(HomeDir)


    ## create subfolder in user directory
    os.makedirs(XCLASSHomeDir, exist_ok = True)
    InitFileName = "{:s}init.dat".format(XCLASSHomeDir)


    ## check, if file "init.dat" already exists and import some settings
    TimeStampFormat = "dd-mm-yyyy"
    DatabaseFileName = "{:s}db/cdms_sqlite.db".format(str(XCLASSHomeDir))
    if (os.path.isfile(InitFileName)):
        InitFile = open(InitFileName)
        InitFileConent = InitFile.readlines()
        InitFile.close()
        for line in InitFileConent:
            StrippedLine = line.strip()
            if (StrippedLine.startswith("TimeStampFormat")):
                SplittedLine = StrippedLine.split("=")
                TimeStampFormat = SplittedLine[1].strip()


    ## get path for temp files
    MAGIXTempDirectory = str(os.environ.get('MAGIXTempDirectory',''))
    MAGIXTempDirectory = MAGIXTempDirectory.strip()
    if (MAGIXTempDirectory == ""):
        MAGIXTempDirectory = "{:s}temp/".format(XCLASSRootDir)
    if (MAGIXTempDirectory[-1] != "/"):
        MAGIXTempDirectory = MAGIXTempDirectory + "/"
    os.environ["MAGIXTempDirectory"] = MAGIXTempDirectory


    ## write content init file
    with open(InitFileName, 'w') as InitFile:
        NewLine =  "XCLASS settings:\n"
        NewLine += "----------------\n"
        NewLine += "XCLASSRootDirectory = {:s}\n".format(XCLASSRootDir)
        NewLine += "DatabaseFileName    = {:s}\n".format(DatabaseFileName)
        NewLine += "NoMAGIX             = {:s}\n".format(str(NoMAGIXFlag))
        NewLine += "TimeStampFormat     = {:s}\n".format(TimeStampFormat)
        NewLine += "MAGIXTempDirectory  = {:s}\n".format(MAGIXTempDirectory)
        InitFile.write(NewLine)


    ## we're done
    print ("done!\n")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## download initial database file to home directory


    ## get database file
    DatabasePATH = "{:s}/db/".format(XCLASSHomeDir)
    if (not os.path.exists(DatabasePATH)):


        ## print what you do
        print ("Download database file to .xclass directory .. ", end = "", flush = True)


        ## create subdirectory for database files, if it is not existing
        cmdString = "mkdir -p {:s}".format(DatabasePATH)
        subprocess.call(cmdString, shell = True)


        ## download the latest version of the sqlite3 database file from the CDMS server
        urlDB = "https://cdms.astro.uni-koeln.de/cdms/portal/cdms_lite.db.gz"
        global ZipFileName                                                                  ## global variable to be used in dlProgress
        ZipFileName = DatabasePATH + "/cdms_lite.db.gz"
        urllib.request.urlretrieve(urlDB, ZipFileName, reporthook = dlProgress)
        sys.stdout.write("\rDownloading sqlite3 database file for XCLASS from CDMS Server .. done!                        ")


        ## unzip file
        print ("\nUnzip the downloaded database file ..", end = " ", flush = True)
        cmdString = "cd {:s} ; ".format(DatabasePATH)
        cmdString += "gunzip cdms_lite.db.gz ; "
        cmdString += "mv cdms_lite.db cdms_sqlite.db ; "
        subprocess.call(cmdString, shell = True)


        ## we're done
        print ("done!\n")


    ## define return value
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##========================================================================================================================================================
## start installation
if __name__ == "__main__":


    ##====================================================================================================================================================
    ## initialize some parameters


    ## get home directory
    # HomeDir = str(os.environ.get('HOME',''))
    HomeDir = os.path.expanduser('~')
    HomeDir = os.path.normpath(HomeDir) + "/"

    # Debug:
    # print ("HomeDir = ", HomeDir)


    ## get the path of the current package
    XCLASSRootDir = os.path.dirname(os.path.abspath(__file__))
    XCLASSRootDir += "/"

    # Debug:
    # print ("XCLASSRootDir = ", XCLASSRootDir)


    ## set path and name of lib subdirectory
    LibPATH = "{:s}xclass/lib/".format(XCLASSRootDir)

    # Debug:
    # print ("LibPATH = ", LibPATH)


    ##====================================================================================================================================================
    ## initialize some parameters
    MAGIXCompilationFlag = "mpi"                                                            ## initialize parallelization option
    NoExtLIBFlag = False                                                                    ## initialize external libraries flag


    ##****************************************************************************************************************************************************
    ## set compile flag
    CompileFlag = True                                                                      ## initialize compile flag
    try:
        EnvCompileFlag = str(os.environ.get('XCLASS_PIP_INSTALL_COMPILE',''))
    except:
        EnvCompileFlag = None
    if (EnvCompileFlag is not None):
        EnvCompileFlag = EnvCompileFlag.strip().lower()
        if (EnvCompileFlag != ""):
            if (not (EnvCompileFlag in ["true", "t", "y"])):
                CompileFlag = False

    # Debug:
    # print ("EnvCompileFlag = ", EnvCompileFlag)
    # print ("CompileFlag = ", CompileFlag)


    ##****************************************************************************************************************************************************
    ## set no-MAGIX flag
    NoMAGIXFlag = False                                                                     ## initialize MAGIX flag
    try:
        EnvNoMAGIXFlag = str(os.environ.get('XCLASS_PIP_INSTALL_NO_MAGIX',''))
    except:
        EnvNoMAGIXFlag = None
    if (EnvNoMAGIXFlag is not None):
        EnvNoMAGIXFlag = EnvNoMAGIXFlag.lower()
        if (EnvNoMAGIXFlag in ["true", "t", "y"]):
            NoMAGIXFlag = True
        else:
            NoMAGIXFlag = False

    # Debug:
    # print ("EnvNoMAGIXFlag = ", EnvNoMAGIXFlag)
    # print ("NoMAGIXFlag = ", NoMAGIXFlag)


    ##****************************************************************************************************************************************************
    ## set to screen flag
    ToSCREENFlag = False                                                                    ## send everything to the screen
    try:
        EnvToSCREENFlag = str(os.environ.get('XCLASS_PIP_INSTALL_TO_SCREEN',''))
    except:
        EnvToSCREENFlag = None
    if (EnvToSCREENFlag is not None):
        EnvToSCREENFlag = EnvToSCREENFlag.lower()
        if (EnvToSCREENFlag in ["true", "t", "y"]):
            ToSCREENFlag = True
        else:
            ToSCREENFlag = False

    # Debug:
    # print ("EnvToSCREENFlag = ", EnvToSCREENFlag)
    # print ("ToSCREENFlag = ", ToSCREENFlag)


    ##====================================================================================================================================================
    ## clean up
    print ("\nRemove old installation directories .. ", end = "", flush = True)
    cmdString = "rm -rf {:s}build/ ; ".format(XCLASSRootDir)
    cmdString += "rm -rf {:s}xclass.egg-info/ ; ".format(XCLASSRootDir)
    subprocess.call(cmdString, shell = True)
    print ("done!\n", flush = True)


    ##====================================================================================================================================================
    ## check availability of required external libraries
    StatusFlag = CheckExtLibraries()
    if (StatusFlag != 0):
        print ("\n\nXCLASS installation FAILED!\n\n\n")
    else:


        ##================================================================================================================================================
        ## compile XCLASS packages
        StatusFlag = CompileXCLASS(XCLASSRootDir, LibPATH, CompileFlag, MAGIXCompilationFlag, NoMAGIXFlag, NoExtLIBFlag, ToSCREENFlag)
        if (StatusFlag != 0):
            print ("\n\nXCLASS installation FAILED!\n\n\n")
        else:


            ##============================================================================================================================================
            ## start setup


            ## Get version string without importing xclass
            version_file = os.path.join(os.path.dirname(__file__), "xclass", "version.py")
            with io.open(version_file, "r", encoding="utf-8") as f:
                content = f.read()
            match = re.search(r"__version__\s*=\s*['\"]([^'\"]+)['\"]", content)
            if match:
                version_str = match.group(1)
            else:
                version_str = "0.0.0"

            # Debug:
            # print ("version.__version__ = ", version.__version__)


            ## setup without using a setup.cfg file
            setup(name = "xclass",
                version = version_str,
                author = "Thomas Möller",
                author_email = "moeller@ph1.uni-koeln.de",
                description = "A package for modeling interferometric and single dish data.",
                long_description = "file: README.md",
                url = "https://git.ph1.uni-koeln.de/moeller/xclass",
                license = "BSD",
                classifiers =["Development Status :: 3 - Alpha",
                                "Intended Audience :: Developers",
                                "License :: OSI Approved :: MIT License",
                                "Programming Language :: Python :: 3",
                                "Operating System :: OS Independent"],
                install_requires = ["numpy",
                                    "scipy",
                                    "matplotlib",
                                    "astropy",
                                    "spectral_cube",
                                    "regions",
                                    "aplpy",
                                    "lmfit",
                                    "h5py",
                                    "emcee",
                                    "ultranest",
                                    "meson",
                                    "ninja",
                                    "lxml",
                                    "tqdm",
                                    "setuptools",
                                    "PyInstaller",
                                    "PyQt5",
                                    "sphinx",
                                    "sphinx_rtd_theme"],
                entry_points = {"console_scripts": ["XCLASS-GUI = xclass.tools.XCLASSGUI.XCLASS_GUI:StartXCLASSGUI",
                                                    "XCLASS-Viewer = xclass.tools.XCLASSViewer.XCLASS_Viewer:StartXCLASSViewer"]},
                packages = find_namespace_packages(exclude = ["examples.*", \
                                                                "docs"]),
                package_data = {"xclass": ["lib/*.exe", \
                                            "lib/*.so", \
                                            "lib/*.a", \
                                            "addons/MAGIX/Fit-Functions/Drude-Lorentz_conv/xml/*.xml", \
                                            "addons/MAGIX/Fit-Functions/Drude-Lorentz_general/xml/*.xml", \
                                            "addons/MAGIX/Fit-Functions/myXCLASS/xml/*.xml", \
                                            "tools/XCLASSGUI/gui/layout/*.ui", \
                                            "tools/XCLASSViewer/layout/*.ui"]},
                cmdclass={"build_ext": XCLASSInstall})


## we're done
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
