#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  Create GUI where frequency range and continuum parameters can be defined
##
##  Copyright (C) 2016 - 2024  Thomas Moeller
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function RangeSelectionGUI.__init__:              initialize class RangeSelectionGUI
##      - function RangeSelectionGUI.keyPressEvent:         redefine return-press event
##      - function RangeSelectionGUI.ButtonWidget:          handle button event of widget
##      - function RangeSelectionGUI.ComboWidget:           handle combo box event of widget
##      - function RangeSelectionGUI.CheckBoxWidget:        handle check box event of widget
##      - function RangeSelectionGUI.LineEditWidget:        handle line edit event of widget
##      - function RangeSelectionGUI.SliderWidget:          handle slider event of widget
##      - function RangeSelectionGUI.connectSpecPlot:       connect all the stored connection ids
##      - function RangeSelectionGUI.dispose:               quit gui
##      - function RangeSelectionGUI.disconnect:            disconnect all the stored connection ids
##      - function RangeSelectionGUI.replotGUI:             replot GUI
##      - function RangeSelectionGUI.OnKeySpectrum:         defines what happens on key event
##      - function RangeSelectionGUI.MouseMoveSpectrum:     define what to do when mouse is moved in spectrum plot
##      - function RangeSelectionGUI.MouseMoveMap:          define what to do when mouse is moved in map plot
##      - function RangeSelectionGUI.getParam:              get parameter
##      - function RangeSelectionGUI.ChangeFreqRange:       modify frequency range
##      - function RangeSelectionGUI.onclickSpectrum:       defines what happens on mouse click
##      - function RangeSelectionGUI.onclickMap:            define what happens on mouse click in map plot
##      - function RangeSelectionGUI.FITSImport:            import fits file
##      - function RangeSelectionGUI.FITSPixelSpecImport:   import pixel spectrum from fits file
##      - function SelectRange:                             plot transition frequency window spectra
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2016-10-06      initial version
##  T. Moeller      2017-02-08      add multiple range definition
##  T. Moeller      2020-01-08      porting to python 3
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
from copy import deepcopy                                                                   ## import module deepcopy from package copy
from PyQt5.uic import loadUiType                                                            ## import module loadUiType from PyQt5 package
from PyQt5 import (QtCore, QtGui, QtWidgets)
from matplotlib.figure import Figure                                                        ## import module Figure from matplotlib package
from matplotlib.backends.backend_qt5agg import (FigureCanvasQTAgg as FigureCanvas, NavigationToolbar2QT as NavigationToolbar)
import matplotlib.pyplot as plt                                                             ## import plt package from matplotlib
import matplotlib.patches as patches                                                        ## import patches package from matplotlib
from matplotlib.widgets import SpanSelector                                                 ## import spanselector from matplotlib
from spectral_cube import SpectralCube                                                      ## import SpectralCube package
from spectral_cube.utils import SpectralCubeWarning, WCSMismatchWarning, SliceWarning, StokesWarning, WCSMismatchWarning
import astropy.units as u                                                                   ## import units package from astropy
from astropy.io import fits                                                                 ## import fits from astropy
from astropy.wcs import WCS                                                                 ## import wcs from astropy
from regions import Regions                                                                 ## import regions package
import pylab                                                                                ## import pylab package
import warnings                                                                             ## import warnings package
from xclass import task_LineIdentification                                                  ## import myXCLASSMapFit function from XCLASS package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define routines for mouse handling
##
# class RangeSelectionGUI(QtGui.QWidget):
class RangeSelectionGUI(QtWidgets.QDialog):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize
    returnPressed = QtCore.pyqtSignal()

    def __init__(self, parent = None, ObsDataFileNameIn = "", ObsDataFileIn = [], LocalNumRangeListIn = 1, LocalFreqMinListIn = [0.0], \
                 LocalFreqMaxListIn = [0.0], LocalFreqStepListIn = [0.0], TbgFlagListIn = [True], TbgListIn = [0.0], TSlopeListIn = [0.0], \
                 nHFlagListIn = [False], nHListIn = [0.0], betaListIn = [0.0], kappaListIn = [0.0], DustFileIn = "", BackgroundFileIn = "", \
                 ContFuncIDListIn = [None], ContParam1ListIn = [None], ContParam2ListIn = [None], ContParam3ListIn = [None], ContParam4ListIn = [None], \
                 ContParam5ListIn = [None], TbgFlagDefaultIn = True, TbgDefaultIn = 0.0, TslopeDefaultIn = 0.0, nHFlagDefaultIn = False, \
                 nHDefaultIn = 0.0, kappaDefaultIn = 0.0, betaDefaultIn = 0.0, DustFileNamesDefaultIn = "", BackgroundFileNamesDefaultIn = "", \
                 ContFuncIDDefaultIn = None, ContParam1DefaultIn = None, ContParam2DefaultIn = None, \
                 ContParam3DefaultIn = None, ContParam4DefaultIn = None, ContParam5DefaultIn = None, ObsDataFileIDIn = 0, \
                 SizeTelescopeIn = 1.0, BMINIn = None, BMAJIn = None, BPAIn = None, InterferoFlagIn = False, GlobalvLSRIn = 0.0, RedShiftIn = None, \
                 FITSSubDirIN = "", NumObsDataFileIN = 1, EstContIntDefaultIn = 20):
        """

    input parameters:
    -----------------

        - ObsDataFileNameIn:            path and name of obs. data file

        - ObsDataFileIn:                array containing data

        - LocalNumRangeListIn:          number of frequency ranges

        - LocalFreqMinListIn:           lowest frequency of data file

        - LocalFreqMaxListIn:           highest frequency of data file

        - LocalFreqStepListIn:          step frequency

        - TbgFlagListIn:                list of TbgFlag

        - TbgListIn:                    initial guess for T_bg

        - TSlopeListIn:                 initial guess for T_slope

        - nHFlagListIn:                 list of nH flags

        - nHListIn:                     list of nH

        - betaListIn:                   list of beta

        - kappaListIn:                  list of kappa

        - DustFileIn:                   path and name of dust files

        - BackgroundFileIn:             path and name of background files

        - ContFuncIDListIn:             function index for phen. continuum description

        - ContParam1ListIn:             parameter 1 for phen. continuum description

        - ContParam2ListIn:             parameter 2 for phen. continuum description

        - ContParam3ListIn:             parameter 3 for phen. continuum description

        - ContParam4ListIn:             parameter 4 for phen. continuum description

        - ContParam5ListIn:             parameter 5 for phen. continuum description

        - TbgFlagDefaultIn:             default T_Back flag

        - TbgDefaultIn:                 default background temperature

        - TslopeDefaultIn:              default temperature slope

        - nHFlagDefaultIn:              default n_H flag

        - nHDefaultIn:                  default hydrogen column density

        - kappaDefaultIn:               default kappa value

        - betaDefaultIn:                default beta value

        - DustFileNamesDefaultIn:       default path and name of dust file

        - BackgroundFileNamesDefaultIn: default path and name of background file

        - ContFuncIDDefaultIn:          default value for cont. function index

        - ContParam1DefaultIn:          default value for cont. parameter 1

        - ContParam2DefaultIn:          default value for cont. parameter 2

        - ContParam3DefaultIn:          default value for cont. parameter 3

        - ContParam4DefaultIn:          default value for cont. parameter 4

        - ContParam5DefaultIn:          default value for cont. parameter 5

        - ObsDataFileIDIn:              current obs. data file id

        - SizeTelescopeIn:              user input for size of telescope

        - BMINIn:                       value for BMIN

        - BMAJIn:                       value for BMAJ

        - BPAIn:                        value for BPA

        - InterferoFlagIn:              user input for interferometer flag

        - GlobalvLSRIn:                 user input for global v_LSR

        - RedShiftIn:                   value for redshift

        - FITSSubDirIN:                 path of subdirectory for FITS files

        - NumObsDataFileIN:             total number of obs. data files

        - EstContIntDefaultIn:          default value for number of intervals (cont. estimation)


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("ObsDataFileNameIn = ", ObsDataFileNameIn)
        # print ("ObsDataFileIn = ", ObsDataFileIn)
        # print ("LocalNumRangeListIn = ", LocalNumRangeListIn)
        # print ("LocalFreqMinListIn = ", LocalFreqMinListIn)
        # print ("LocalFreqMaxListIn = ", LocalFreqMaxListIn)
        # print ("LocalFreqStepListIn = ", LocalFreqStepListIn)
        # print ("TbgFlagListIn = ", TbgFlagListIn)
        # print ("TbgListIn = ", TbgListIn)
        # print ("TSlopeListIn = ", TSlopeListIn)
        # print ("nHFlagListIn = ", nHFlagListIn)
        # print ("nHListIn = ", nHListIn)
        # print ("betaListIn = ", betaListIn)
        # print ("kappaListIn = ", kappaListIn)
        # print ("DustFileIn = ", DustFileIn)
        # print ("BackgroundFileIn = ", BackgroundFileIn)
        # print ("ContFuncIDListIn = ", ContFuncIDListIn)
        # print ("ContParam1ListIn = ", ContParam1ListIn)
        # print ("ContParam2ListIn = ", ContParam2ListIn)
        # print ("ContParam3ListIn = ", ContParam3ListIn)
        # print ("ContParam4ListIn = ", ContParam4ListIn)
        # print ("ContParam5ListIn = ", ContParam5ListIn)
        # print ("TbgFlagDefaultIn = ", TbgFlagDefaultIn)
        # print ("TbgDefaultIn = ", TbgDefaultIn)
        # print ("TslopeDefaultIn = ", TslopeDefaultIn)
        # print ("nHFlagDefaultIn = ", nHFlagDefaultIn)
        # print ("nHDefaultIn = ", nHDefaultIn)
        # print ("kappaDefaultIn = ", kappaDefaultIn)
        # print ("betaDefaultIn = ", betaDefaultIn)
        # print ("DustFileNamesDefaultIn = ", DustFileNamesDefaultIn)
        # print ("BackgroundFileNamesDefaultIn = ", BackgroundFileNamesDefaultIn)
        # print ("ContFuncIDDefaultIn = ", ContFuncIDDefaultIn)
        # print ("ContParam1DefaultIn = ", ContParam1DefaultIn)
        # print ("ContParam2DefaultIn = ", ContParam2DefaultIn)
        # print ("ContParam3DefaultIn = ", ContParam3DefaultIn)
        # print ("ContParam4DefaultIn = ", ContParam4DefaultIn)
        # print ("ContParam5DefaultIn = ", ContParam5DefaultIn)
        # print ("ObsDataFileIDIn = ", ObsDataFileIDIn)
        # print ("SizeTelescopeIn = ", SizeTelescopeIn)
        # print ("BMINIn = ", BMINIn)
        # print ("BMAJIn = ", BMAJIn)
        # print ("BPAIn = ", BPAIn)
        # print ("InterferoFlagIn = ", InterferoFlagIn)
        # print ("GlobalvLSRIn = ", GlobalvLSRIn)
        # print ("RedShiftIn = ", RedShiftIn)
        # print ("FITSSubDirIN = ", FITSSubDirIN)
        # print ("NumObsDataFileIN = ", NumObsDataFileIN)
        # print ("EstContIntDefaultIn = ", EstContIntDefaultIn)


        ## create widget
        super(RangeSelectionGUI, self).__init__(parent)


        ## copy input parameters to class variables
        self.ObsDataFileID = ObsDataFileIDIn                                                ## current obs. data file id
        self.UserObsDataFileName = ObsDataFileNameIn                                        ## get path and name of obs. data file
        self.OnceFlag = True                                                                ## set once flag to add colorbar
        self.InitFlag = True                                                                ## define initialization flag
        self.InitNumDataPtsFreqRange = 100                                                  ## initial number of data points per freq. range
        self.RefreshMapOnlyFlag = False                                                     ## flag indicating that only map is refreshed
        self.AllMinInt = numpy.nan                                                          ## reset global minimum intensity
        self.AllMaxInt = numpy.nan                                                          ## reset global maximum intensity
        self.ok = 0                                                                         ## set status variable
        self.FITSCube = []                                                                  ## fits data cube
        self.lockElements = True                                                            ## flag for locking elements
        self.ChannelIndex = 0                                                               ## current channel index
        self.NumDimFitsFile = 4                                                             ## dimension of fits file
        self.ReverseFlag = True                                                             ## reverse flag for fits data indexing
        self.xPixelID = None                                                                ## x-pixel index of current pixel spectrum
        self.yPixelID = None                                                                ## y-pixel index of current pixel spectrum
        self.FrequencyAxis = []                                                             ## list of all frequency of map
        self.MapCurrFreq = 1.0                                                              ## frequency of the current channel map
        self.MapFreqMin = 1.0                                                               ## lowest frequency of map
        self.MapFreqMax = 1.e9                                                              ## highest frequency of map
        self.MapFreqStep = 1.0                                                              ## step frequency of map
        self.ListSpecCoord = []                                                             ## list of coordinates of selected spectra
        self.ListSpecData = []                                                              ## list of data of selected spectra
        self.DecAxis = []                                                                   ## ticks for axis
        self.DecAxisDelta = 0.0                                                             ## initialize delta of Dec axis
        self.RAAxis = []                                                                    ## ticks for axis
        self.RAAxisDelta = 0.0                                                              ## initialize delta of RA axis
        self.LastXRange = []                                                                ## last x limits
        self.LastYRange = []                                                                ## last y limits
        self.AllRangeFlag = False                                                           ## all range flag
        self.DefineNewRangeFlag = False                                                     ## define new range flag
        self.ObsDataFile = None                                                             ## spectra of selected regions
        self.SpectrumSelectedFlag = False                                                   ## flag indicating if spectrum is selected or not


        self.FreqStepList = deepcopy(LocalFreqStepListIn)
        self.TbgFlagList = deepcopy(TbgFlagListIn)
        self.TbgList = deepcopy(TbgListIn)                                                  ## list containing T_bg for each range
        self.TSlopeList = deepcopy(TSlopeListIn)                                            ## list containing T_Slope for each range
        self.EstContInt = [EstContIntDefaultIn for x in TSlopeListIn]
        self.nHFlagList = deepcopy(nHFlagListIn)
        self.nHList = deepcopy(nHListIn)
        self.betaList = deepcopy(betaListIn)
        self.kappaList = deepcopy(kappaListIn)
        self.DustFileNameList = deepcopy(DustFileIn)
        self.BackgroundFileNameList = deepcopy(BackgroundFileIn)
        self.ContFuncIDList = deepcopy(ContFuncIDListIn)
        self.ContParam1List = deepcopy(ContParam1ListIn)
        self.ContParam2List = deepcopy(ContParam2ListIn)
        self.ContParam3List = deepcopy(ContParam3ListIn)
        self.ContParam4List = deepcopy(ContParam4ListIn)
        self.ContParam5List = deepcopy(ContParam5ListIn)
        self.GlobalEstContInt = EstContIntDefaultIn
        self.GlobalTbg = TbgDefaultIn
        self.GlobalTSlope = TslopeDefaultIn
        self.TbgFlagDefault = TbgFlagDefaultIn
        self.nHFlagDefault = nHFlagDefaultIn
        self.nHDefault = nHDefaultIn
        self.kappaDefault = kappaDefaultIn
        self.betaDefaultIn = betaDefaultIn
        self.DustFileNamesDefaultIn = DustFileNamesDefaultIn
        self.BackgroundFileNamesDefaultIn = BackgroundFileNamesDefaultIn
        self.ContFuncIDDefaultIn = ContFuncIDDefaultIn
        self.ContParam1DefaultIn = ContParam1DefaultIn
        self.ContParam2DefaultIn = ContParam2DefaultIn
        self.ContParam3DefaultIn = ContParam3DefaultIn
        self.ContParam4DefaultIn = ContParam4DefaultIn
        self.ContParam5DefaultIn = ContParam5DefaultIn
        self.SizeTelescope = SizeTelescopeIn
        self.BMINIn = BMINIn
        self.BMAJIn = BMAJIn
        self.BPAIn = BPAIn
        self.InterFlagPerSpectrum = InterferoFlagIn
        self.GlobalvLSR = GlobalvLSRIn
        self.RedShiftIn = RedShiftIn
        self.FITSSubDir = FITSSubDirIN

        # Debug:
        # print ("\nself.RangeList = ", self.RangeList)
        # print ("RangeListString = ", self.RangeListString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if FITS file is selected
        if (self.UserObsDataFileName.endswith(".fits")):
            self.FITSFlag = True
            self.ChannelIndex = 0
            self.LowestFreq = 1.e5                                                          ## set default min. frequency
            self.HighestFreq = 5.e5                                                         ## set default max. frequency
            self.FITSImport()                                                               ## import FITS cube
            self.MapCurrFreq = self.MapFreqMin


            ## check, if this data file was considered before
            NewObsDataFileName = self.FITSSubDir + "FitsSpectrum__" + str(self.ObsDataFileID + 1) + ".dat"
            if (not (os.path.isfile(NewObsDataFileName))):                                  ## if no pixel were selected for this FITS file before,
                NewObsDataFileName = ""                                                     ## use coordinate from other FITS file
                for ObsID in range(NumObsDataFileIN):
                    NewObsDataFileName = self.FITSSubDir + "FitsSpectrum__" + str(ObsID + 1) + ".dat"
                    if (os.path.isfile(NewObsDataFileName)):
                        break
                    else:
                        NewObsDataFileName = ""
            if (NewObsDataFileName != ""):
                with open(NewObsDataFileName, 'r') as f:
                    FirstHeaderLine = f.readline()


                ## extract list of pixel coordinates indices
                FirstHeaderLine = FirstHeaderLine.strip()
                FirstHeaderLine = FirstHeaderLine.replace("# List of pixel coordinate indices:", "")
                FirstHeaderLine = FirstHeaderLine.strip()
                FirstHeaderLine = FirstHeaderLine.split("],")
                for LocalCoordString in FirstHeaderLine:
                    LocalCoord = LocalCoordString.replace("[", "")
                    LocalCoord = LocalCoord.replace("]", "").strip()
                    LocalCoord = LocalCoord.split(",")
                    x = int(LocalCoord[0])
                    y = int(LocalCoord[1])
                    if (not ([x, y] in self.ListSpecCoord)):
                        self.xPixelID = x
                        self.yPixelID = y
                        self.FITSPixelSpecImport()
                        self.ListSpecData.append(deepcopy(self.ObsDataFile))
                        self.ListSpecCoord.append([x, y])

                # Debug:
                # print ("FirstHeaderLine = ", FirstHeaderLine)
                # print ("self.ListSpecCoord = ", self.ListSpecCoord)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## continue here, if no FITS file was selected
        else:
            self.FITSFlag = False
            self.ObsDataFile = deepcopy(ObsDataFileIn)                                      ## get data as array
            self.LowestFreq = numpy.nanmin(ObsDataFileIn[:, 0])                             ## get lowest frequency of data
            self.HighestFreq = numpy.nanmax(ObsDataFileIn[:, 0])                            ## get highest frequency of data
        self.FreqStep = self.HighestFreq - self.LowestFreq                                  ## step size of data files
        self.RangeID = 0                                                                    ## current frequency range
        self.RangeList = []                                                                 ## list of frequency ranges
        self.RangeListString = ["add new range .."]                                         ## list of range string for combo box
        for RangeID in range(len(LocalFreqMinListIn)):
            self.RangeList.append([LocalFreqMinListIn[RangeID], LocalFreqMaxListIn[RangeID]])
            MinString = "{:.2f}".format(float(LocalFreqMinListIn[RangeID]))
            MaxString = "{:.2f}".format(float(LocalFreqMaxListIn[RangeID]))
            self.RangeListString.append(MinString + " MHz - " + MaxString + " MHz")


        ##================================================================================================================================================
        ## create plot window for spectrum (and map)
        if (self.FITSFlag):
            self.figSpectrum = pylab.figure(figsize=(15, 10))
            self.figSpectrum.clear()
            self.figSpectrum.subplots_adjust(bottom = 0.12, top = 0.88, right = 0.98, left = 0.15, hspace = 0.0, wspace = 0.0)
            # self.figSpectrum.subplots_adjust(bottom = 0.02, top = 0.901, right = 0.921, left = 0.133, hspace = 0.0, wspace = 0.0)
            self.PlotSpec = pylab.subplot(1, 1, 1)

            self.figMap = pylab.figure(figsize=(15, 10))
            self.figMap.clear()
            self.figMap.subplots_adjust(bottom = 0.09, top = 0.966, right = 0.981, left = 0.14, hspace = 0.0, wspace = 0.0)
            self.PlotMap = pylab.subplot(1, 1, 1, projection = self.FITSCube[0, :, :].wcs)
        else:
            self.figSpectrum = pylab.figure(figsize=(15, 10))
            self.figSpectrum.clear()
            self.figSpectrum.subplots_adjust(bottom = 0.09, top = 0.95, right = 0.98, left = 0.05, hspace = 0.0, wspace = 0.0)
            self.PlotSpec = pylab.subplot(1, 1, 1)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add toolbar(s) to figure(s)
        self.canvasSpectrum = FigureCanvas(self.figSpectrum)
        self.canvasSpectrum.draw()
        self.toolbarSpectrum = NavigationToolbar(self.canvasSpectrum, self)
        self.canvasSpectrum.draw()
        if (self.FITSFlag):
            self.canvasMap = FigureCanvas(self.figMap)
            self.canvasMap.draw()
            self.toolbarMap = NavigationToolbar(self.canvasMap, self)
            self.canvasMap.draw()


        ## check, if statcont is available
        STATCONTAvailableFlag = True
        try:
            from external.statcont import cont_finding
        except:
            STATCONTAvailableFlag = False


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define elements
        if (self.FITSFlag):


            ## add line edit element for entering min. freq.
            self.LabelChannel = QtWidgets.QLabel("Frequency:")
            self.LabelChannel.setFixedWidth(80)

            self.LineEditChannel = QtWidgets.QLineEdit()
            self.LineEditChannel.setObjectName("EditChannel")
            self.LineEditChannel.setFixedWidth(350)
            self.LineEditChannel.setToolTip('Enter frequency.')
            self.LineEditChannel.setValidator(QtGui.QDoubleValidator(self.MapFreqMin, self.MapFreqMax, 3))
            self.LineEditChannel.editingFinished.connect(self.LineEditWidget)
            self.LineEditChannel.setEnabled(True)
            self.LineEditChannel.setFixedWidth(250)
            self.LineEditChannel.setText("{:.3f}".format(self.MapCurrFreq))


            ## add slider element for entering min. freq.
            self.SliderChannel = QtWidgets.QSlider()
            self.SliderChannel.setOrientation(QtCore.Qt.Horizontal)
            self.SliderChannel.setMinimum(0)
            self.SliderChannel.setMaximum(len(self.FrequencyAxis) - 1)
            self.SliderChannel.setTickInterval(1)
            self.SliderChannel.setValue(int(self.MapCurrFreq))
            self.SliderChannel.setObjectName("SliderChannel")
            self.SliderChannel.setFixedWidth(350)
            self.SliderChannel.setToolTip('Enter frequency.')
            self.SliderChannel.valueChanged.connect(self.SliderWidget)
            self.SliderChannel.setEnabled(True)


        ## add line edit element for entering min. freq.
        self.LabelComboRange = QtWidgets.QLabel("Range:")
        self.ComboRangeSelect = QtWidgets.QComboBox()
        self.ComboRangeSelect.setObjectName("ComboRange")
        self.ComboRangeSelect.setToolTip('Chose frequency range.')
        self.ComboRangeSelect.currentIndexChanged.connect(self.ComboWidget)
        self.ComboRangeSelect.setEnabled(True)
        self.ComboRangeSelect.addItems(self.RangeListString)
        self.ComboRangeSelect.setCurrentIndex(1)


        ## add line edit element for entering max. freq.
        self.LabelRangeFlag = QtWidgets.QLabel("    All ranges:")
        self.FlagAllRangesDescription = QtWidgets.QCheckBox()
        self.FlagAllRangesDescription.setToolTip('Use T_bg + T_Slope for all ranges.')
        self.FlagAllRangesDescription.clicked.connect(self.CheckBoxWidget)
        self.FlagAllRangesDescription.setEnabled(True)
        self.FlagAllRangesDescription.setChecked(self.AllRangeFlag)


        ## add line edit element for entering background temperature
        if (STATCONTAvailableFlag):
            self.LabelTbg = QtWidgets.QLabel("T_bg (K):")
        else:
            self.LabelTbg = QtWidgets.QLabel("T_bg (K):")
        self.LineEditTbg = QtWidgets.QLineEdit()
        # self.LineEditTbg.setFixedWidth(100)
        self.LineEditTbg.setObjectName("EditTbg")
        self.LineEditTbg.setToolTip('Enter value for background temperature.')
        self.LineEditTbg.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))
        self.LineEditTbg.editingFinished.connect(self.LineEditWidget)
        self.LineEditTbg.setEnabled(True)
        TbgString = "{:.4f}".format(self.TbgList[self.RangeID])
        self.LineEditTbg.setText(TbgString)


        ## add line edit element for entering temperature slope
        self.LabelTSlope = QtWidgets.QLabel("  T_Slope:")
        self.LineEditTSlope = QtWidgets.QLineEdit()
        self.LineEditTSlope.setObjectName("EditTSlope")
        # self.LineEditTSlope.setFixedWidth(100)
        self.LineEditTSlope.setToolTip('Enter value for temperature slope.')
        self.LineEditTSlope.setValidator(QtGui.QDoubleValidator(-10.00, 10.00, 6))
        self.LineEditTSlope.editingFinished.connect(self.LineEditWidget)
        TSlopeString = "{:.2f}".format(self.TSlopeList[self.RangeID])
        self.LineEditTSlope.setText(TSlopeString)


        # add button "Add Range"
        self.buttonAddRange = QtWidgets.QPushButton('Add Range')
        self.buttonAddRange.setShortcut("+")
        self.buttonAddRange.setToolTip('Add frequency range')
        self.buttonAddRange.setObjectName("buttonAddRange")
        self.buttonAddRange.clicked.connect(self.ButtonWidget)


        # add button "Remove Range"
        self.buttonRemoveRange = QtWidgets.QPushButton('Remove Range')
        self.buttonRemoveRange.setShortcut("DEL")
        self.buttonRemoveRange.setToolTip('Remove current frequency range')
        self.buttonRemoveRange.setObjectName("buttonRemoveRange")
        self.buttonRemoveRange.clicked.connect(self.ButtonWidget)


        # add button "Estimate Continuum"
        self.buttonEstCont = QtWidgets.QPushButton('Estimate Continuum')
        self.buttonEstCont.setShortcut("e")
        self.buttonEstCont.setToolTip('Estimate Continuum automatically')
        self.buttonEstCont.setObjectName("buttonEstCont")
        self.buttonEstCont.clicked.connect(self.ButtonWidget)


        ## add line edit element for number of intervals
        self.LabelEstContInt = QtWidgets.QLabel("  Number Intervals:")
        self.LineEditEstContInt = QtWidgets.QLineEdit()
        self.LineEditEstContInt.setObjectName("EditEstContInt")
        # self.LineEditEstContInt.setFixedWidth(100)
        self.LineEditEstContInt.setToolTip('Enter number of intervals for continuum estimation.')
        self.LineEditEstContInt.setValidator(QtGui.QDoubleValidator(0, 999999, 0))
        self.LineEditEstContInt.editingFinished.connect(self.LineEditWidget)
        EstContIntString = "{:d}".format(self.EstContInt[self.RangeID])
        self.LineEditEstContInt.setText(EstContIntString)


        ## add button "|<"
        self.buttonFSC = QtWidgets.QPushButton('|<')
        self.buttonFSC.setShortcut("0")
        self.buttonFSC.setToolTip('first spectral channel')
        self.buttonFSC.setObjectName("SpecChannelFitst")
        self.buttonFSC.setFixedWidth(80)
        self.buttonFSC.clicked.connect(self.ButtonWidget)


        ## add button "<"
        self.buttonPSC = QtWidgets.QPushButton('<')
        self.buttonPSC.setShortcut("-")
        self.buttonPSC.setToolTip('previous spectral channel')
        self.buttonPSC.setObjectName("SpecChannelm1")
        self.buttonPSC.setFixedWidth(80)
        self.buttonPSC.clicked.connect(self.ButtonWidget)


        ## add button ">"
        self.buttonNSC = QtWidgets.QPushButton('>')
        self.buttonNSC.setShortcut("+")
        self.buttonNSC.setToolTip('next spectral channel')
        self.buttonNSC.setObjectName("SpecChannelp1")
        self.buttonNSC.setFixedWidth(80)
        self.buttonNSC.clicked.connect(self.ButtonWidget)


        ## add button ">|"
        self.buttonLSC = QtWidgets.QPushButton('>|')
        self.buttonLSC.setShortcut("9")
        self.buttonLSC.setToolTip('last spectral channel')
        self.buttonLSC.setObjectName("SpecChannelLast")
        self.buttonLSC.setFixedWidth(80)
        self.buttonLSC.clicked.connect(self.ButtonWidget)


        ## add button "Cancel"
        self.buttonCancel = QtWidgets.QPushButton('Cancel')
        self.buttonCancel.setShortcut("ESC")
        self.buttonCancel.setToolTip('Cancel')
        self.buttonCancel.setObjectName("ButtonCancel")
        self.buttonCancel.clicked.connect(self.ButtonWidget)


        ## add button "OK"
        self.buttonOK = QtWidgets.QPushButton('OK')
        self.buttonOK.setShortcut("q")
        self.buttonOK.setToolTip('Quit GUI.')
        self.buttonOK.setObjectName("ButtonOK")
        self.buttonOK.clicked.connect(self.ButtonWidget)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## connect events to current plot
        self.connectSpecPlot()
        if (self.FITSFlag):
            self.connectMap()


        ##================================================================================================================================================
        ## box layout
        layout = QtWidgets.QGridLayout()


        ## select pixel in map
        if (self.FITSFlag):
            layout.addWidget(self.canvasMap, 0, 0, 3, 5)
            layout.addWidget(self.toolbarMap, 3, 0, 1, 5)
            layout.addWidget(self.LabelChannel, 4, 0)
            layout.addWidget(self.LineEditChannel, 4, 1, 1, 3)
            layout.addWidget(self.SliderChannel, 4, 4, 1, 2)
            layout.addWidget(self.canvasSpectrum, 0, 5, 1, 12)
            layout.addWidget(self.toolbarSpectrum, 2, 5, 1, 12)
            layout.addWidget(self.LabelComboRange, 3, 10)
            layout.addWidget(self.ComboRangeSelect, 3, 11, 1, 2)
            layout.addWidget(self.buttonRemoveRange, 3, 13)
            layout.addWidget(self.LabelRangeFlag, 3, 14)
            layout.addWidget(self.FlagAllRangesDescription, 3, 15)
            layout.addWidget(self.LabelTbg, 4, 10)
            layout.addWidget(self.LineEditTbg, 4, 11)
            layout.addWidget(self.LabelTSlope, 4, 12)
            layout.addWidget(self.LineEditTSlope, 4, 13, 1, 3)
            layout.addWidget(self.buttonFSC, 5, 0)
            layout.addWidget(self.buttonPSC, 5, 1)
            layout.addWidget(self.buttonNSC, 5, 2)
            layout.addWidget(self.buttonLSC, 5, 3)
            layout.addWidget(self.buttonCancel, 5, 14)
            layout.addWidget(self.buttonOK, 5, 15)


         ## normal spectrum plot without FITS map
        else:
            if (not STATCONTAvailableFlag):
                layout.addWidget(self.canvasSpectrum, 0, 0, 1, 12)
                layout.addWidget(self.toolbarSpectrum, 2, 0, 1, 12)
                layout.addWidget(self.LabelComboRange, 3, 0)
                layout.addWidget(self.ComboRangeSelect, 3, 1)
                layout.addWidget(self.buttonAddRange, 3, 2)
                layout.addWidget(self.buttonRemoveRange, 3, 3)
                layout.addWidget(self.LabelRangeFlag, 3, 4)
                layout.addWidget(self.FlagAllRangesDescription, 3, 5)
                layout.addWidget(self.LabelTbg, 3, 6)
                layout.addWidget(self.LineEditTbg, 3, 7)
                layout.addWidget(self.LabelTSlope, 3, 8)
                layout.addWidget(self.LineEditTSlope, 3, 9)
                layout.addWidget(self.buttonCancel, 3, 10)
                layout.addWidget(self.buttonOK, 3, 11)
            else:
                layout.addWidget(self.canvasSpectrum, 0, 0, 1, 11)

                layout.addWidget(self.toolbarSpectrum, 2, 0, 1, 11)

                layout.addWidget(self.LabelComboRange, 3, 0)
                layout.addWidget(self.ComboRangeSelect, 3, 1)
                layout.addWidget(self.buttonAddRange, 3, 3)
                layout.addWidget(self.buttonRemoveRange, 3, 4)

                layout.addWidget(self.LabelTbg, 4, 0)
                layout.addWidget(self.LineEditTbg, 4, 1)
                layout.addWidget(self.LabelTSlope, 4, 2)
                layout.addWidget(self.LineEditTSlope, 4, 3)
                layout.addWidget(self.buttonEstCont, 4, 4)
                layout.addWidget(self.LabelEstContInt, 4, 5)
                layout.addWidget(self.LineEditEstContInt, 4, 6)
                layout.addWidget(self.LabelRangeFlag, 4, 7)
                layout.addWidget(self.FlagAllRangesDescription, 4, 8)
                layout.addWidget(self.buttonCancel, 4, 9)
                layout.addWidget(self.buttonOK, 4, 10)


        ## set layout and window title
        self.setLayout(layout)
        head, LocalObsName = os.path.split(self.UserObsDataFileName)
        LocalTitelString = "Define spectrum, frequency range and continuum for data file " + chr(34) + LocalObsName.strip() + chr(34)
        self.setWindowTitle(LocalTitelString)

        # Debug:
        # print ("self.LowestFreq, self.HighestFreq = ", self.LowestFreq, self.HighestFreq)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set limits for plots
        self.PlotSpec.set_xlim(self.LowestFreq, self.HighestFreq)
        self.LowestFreq = numpy.nanmin(self.ObsDataFile[:, 0])
        self.HighestFreq = numpy.nanmax(self.ObsDataFile[:, 0])
        self.FreqStep = (self.ObsDataFile[1, 0] - self.ObsDataFile[0, 0])
        self.InitNumDataPtsFreqRange = max(10, int(len(self.ObsDataFile[:, 0]) * 0.1))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## plot spectra
        self.lockElements = False
        self.InitFlag = True
        self.replotGUI()
        self.InitFlag = False


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## show spectra
        self.show()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## redefine return-press event
    def keyPressEvent(self, event):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("event = ", event)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## redefine return-press event
        if (event.key() == QtCore.Qt.Key_Return):
            self.returnPressed.emit(event)
        # elif (event.key() == QtCore.Qt.Key_Control):
        else:
            event.accept()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle button event of widget
    def ButtonWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## is element locked
        if (self.lockElements):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()
        name = sender.text()

        # Debug:
        # print ("objectName = ", objectName)
        # print ("name = ", name)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## identify calling button


        # cancel GUI
        if (objectName == "ButtonCancel"):
            self.ok = 1
            self.disconnect()


        # quit GUI
        elif (objectName == "ButtonOK"):
            self.disconnect()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## (only for FITS map): control spectral channel


        # go to first spectral channel
        elif (objectName == "SpecChannelFitst"):
            self.ChannelIndex = 0


            ## update GUI
            self.replotGUI()


        # go to first spectral channel
        elif (objectName == "SpecChannelm1"):
            self.ChannelIndex = max(0, self.ChannelIndex - 1)


            ## update GUI
            self.replotGUI()


        # go to first spectral channel
        elif (objectName == "SpecChannelp1"):
            self.ChannelIndex = min(len(self.FrequencyAxis) - 1, self.ChannelIndex + 1)


            ## update GUI
            self.replotGUI()


        # go to first spectral channel
        elif (objectName == "SpecChannelLast"):
            self.ChannelIndex = len(self.FrequencyAxis) - 1


            ## update GUI
            self.replotGUI()



        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add range on right double click
        elif (objectName == "buttonAddRange"):
            self.DefineNewRangeFlag = True


            ## update GUI
            self.replotGUI()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## remove current range on right double click
        elif (objectName == "buttonRemoveRange" and (not self.DefineNewRangeFlag)):
            del self.RangeList[self.RangeID]
            del self.FreqStepList[self.RangeID]
            del self.TbgFlagList[self.RangeID]
            del self.TbgList[self.RangeID]
            del self.TSlopeList[self.RangeID]
            del self.nHFlagList[self.RangeID]
            del self.nHList[self.RangeID]
            del self.betaList[self.RangeID]
            del self.kappaList[self.RangeID]
            del self.DustFileNameList[self.RangeID]
            del self.BackgroundFileNameList[self.RangeID]
            del self.ContFuncIDList[self.RangeID]
            del self.ContParam1List[self.RangeID]
            del self.ContParam2List[self.RangeID]
            del self.ContParam3List[self.RangeID]
            del self.ContParam4List[self.RangeID]
            del self.ContParam5List[self.RangeID]
            if (self.RangeList == []):
                self.RangeList = [[self.LowestFreq, self.HighestFreq]]
                self.FreqStepList = [self.FreqStep]
                self.TbgFlagList = [self.TbgFlagDefault]
                self.TbgList = [self.GlobalTbg]
                self.TSlopeList = [self.GlobalTSlope]
                self.nHFlagList = [self.nHFlagDefault]
                self.nHList = [self.nHDefault]
                self.betaList = [self.betaDefaultIn]
                self.kappaList = [self.kappaDefault]
                self.DustFileNameList = [self.DustFileNamesDefaultIn]
                self.BackgroundFileNameList = [self.BackgroundFileNamesDefaultIn]
                self.ContFuncIDList = [self.ContFuncIDDefaultIn]
                self.ContParam1List = [self.ContParam1DefaultIn]
                self.ContParam2List = [self.ContParam2DefaultIn]
                self.ContParam3List = [self.ContParam3DefaultIn]
                self.ContParam4List = [self.ContParam4DefaultIn]
                self.ContParam5List = [self.ContParam5DefaultIn]
                self.RangeID = 0
            self.RangeID = min(len(self.RangeList) - 1, self.RangeID)


            ## update GUI
            self.replotGUI()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## estimate continuum
        elif (objectName == "buttonEstCont"):


            ## get number of intervals
            if (self.AllRangeFlag):
                NumInt = self.GlobalEstContInt
            else:
                NumInt = self.EstContInt[self.RangeID]


            ## prepare data array
            DataArray = None
            if (self.AllRangeFlag):
                for LocalRange in self.RangeList:
                    LocalFreqMin = float(LocalRange[0])
                    LocalFreqMax = float(LocalRange[1])
                    LocalDataArray = self.ObsDataFile[(self.ObsDataFile[:, 0] >= LocalFreqMin) & (self.ObsDataFile[:, 0] <= LocalFreqMax), :]
                    if (DataArray is None):
                        DataArray = deepcopy(LocalDataArray)
                    else:
                        DataArray = numpy.concatenate((DataArray, LocalDataArray), axis = 0)
            else:
                LocalFreqMin = self.RangeList[self.RangeID][0]
                LocalFreqMax = self.RangeList[self.RangeID][1]
                DataArray = self.ObsDataFile[(self.ObsDataFile[:, 0] >= LocalFreqMin) & (self.ObsDataFile[:, 0] <= LocalFreqMax), :]

            # Debug:
            # print ("DataArray = ", DataArray)


            ## estimate continuum
            try:
                DataNoContArray, NewTBack, NewTSlope = task_LineIdentification.EstimateContinuum(DataArray, NumParts = NumInt)
            except:
                NewTBack = None
                NewTSlope = None

            # Debug:
            # print ("NewTBack = ", NewTBack)
            # print ("NewTSlope = ", NewTSlope)


            ## set T_Back and T_Slope
            if (NewTBack is not None and NewTSlope is not None):
                if (self.AllRangeFlag):
                    self.GlobalTbg = NewTBack
                    self.GlobalTSlope = NewTSlope
                else:
                    self.TbgList[self.RangeID] = NewTBack
                    self.TSlopeList[self.RangeID] = NewTSlope


            ## update GUI
            self.replotGUI()


        ## return to main GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle check box event of widget
    def CheckBoxWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is element locked
        if (self.lockElements):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name, value and flag
        sender = self.sender()
        flag = sender.isChecked()
        self.AllRangeFlag = flag

        # Debug:
        # print ("flag = ", flag)


        ## update GUI
        if (not self.InitFlag):
            self.replotGUI()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle combo box event of widget
    def ComboWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is element locked
        if (self.lockElements):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()

        # Debug:
        # print ("objectName = ", objectName)


        ## get index of selected frequency range
        LocalComboIndex = max(0, self.ComboRangeSelect.currentIndex())
        self.DefineNewRangeFlag = False
        self.RangeID = LocalComboIndex


        ## update GUI
        self.replotGUI()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle line edit event of widget
    def LineEditWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is element locked
        if (self.lockElements):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()
        val = str(sender.text())
        self.CurrCursorPos = sender.cursorPosition()

        # Debug:
        # print ("objectName = ", objectName)
        # print ("val = ", val)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if value is number which is greater or equal zero
        IsNumber = True
        try:
            i = int(val)
        except ValueError:
            try:
                i = float(val)
            except ValueError:
                IsNumber = False
        if (not IsNumber):
            if (str(IsNumber).strip() == ""):
                return
        if (IsNumber):                                                                      ## continue if input is a number
            val = float(val)


            ## get value of background temperature
            if (objectName == "EditTbg"):
                if (self.AllRangeFlag):
                    self.GlobalTbg = val
                else:
                    self.TbgList[self.RangeID] = val
                self.replotGUI()                                                            ## replot spectrum


            ## get value of temperature slope
            elif (objectName == "EditTSlope"):
                if (self.AllRangeFlag):
                    self.GlobalTSlope = val
                else:
                    self.TSlopeList[self.RangeID] = val
                self.replotGUI()                                                            ## replot spectrum


            ## get number of intervals
            elif (objectName == "EditEstContInt"):
                if (self.AllRangeFlag):
                    self.GlobalEstContInt = val
                else:
                    self.EstContInt[self.RangeID] = val
                self.replotGUI()                                                            ## replot spectrum


            ## get channel index
            elif (objectName == "EditChannel"):
                self.MapCurrFreq = val
                self.ChannelIndex = max(0, (numpy.abs(self.FrequencyAxis - val)).argmin() - 1)
                self.RefreshMapOnlyFlag = True
                self.replotGUI()                                                            ## replot spectrum
                self.RefreshMapOnlyFlag = False

        # Debug:
        # print ("val = ", val)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle slider event of widget
    def SliderWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is element locked
        if (self.lockElements):
            return


        ## get name of slider
        sender = self.sender()
        objectName = sender.objectName()                                                    ## get name of object
        val = sender.value()                                                                ## get value from slider
        if (objectName == "SliderChannel"):
            self.ChannelIndex = val
            self.MapCurrFreq = self.FrequencyAxis[val]

            # Debug:
            # print ("self.MapCurrFreq = ", self.MapCurrFreq)
            # print ("self.ChannelIndex = ", self.ChannelIndex)
            # print ("val = ", val)
            # print ("numpy.abs(self.FrequencyAxis - val) = ", numpy.abs(self.FrequencyAxis - val))
            # print ("(numpy.abs(self.FrequencyAxis - val)).argmin() - 1 = ", (numpy.abs(self.FrequencyAxis - val)).argmin() - 1)


            ## replot spectrum
            self.RefreshMapOnlyFlag = True
            self.replotGUI()                                                                ## replot spectrum
            self.RefreshMapOnlyFlag = False


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## connect all the stored connection ids
    def connectSpecPlot(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## define events
        self.cidquitSpectrum = self.canvasSpectrum.mpl_connect('close_event', self.dispose)
        self.cidpressSpectrum = self.canvasSpectrum.mpl_connect('button_press_event', self.onclickSpectrum)
        self.cidmotionSpectrum = self.canvasSpectrum.mpl_connect('motion_notify_event', self.MouseMoveSpectrum)
        # self.cidkeySpectrum = self.canvasSpectrum.mpl_connect('key_press_event', self.OnKeySpectrum)


        ## use tight layout
        # self.figSpectrum.tight_layout()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## connect events to
    def connectMap(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## define events
        self.cidquitMap = self.canvasMap.mpl_connect('close_event', self.dispose)
        self.cidpressMap = self.canvasMap.mpl_connect('button_press_event', self.onclickMap)
        self.cidmotionMap = self.canvasMap.mpl_connect('motion_notify_event', self.MouseMoveMap)
        # self.cidkeyMap = self.canvasMap.mpl_connect('key_press_event', self.OnKeyMap)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## quit gui
    def dispose(self, event):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("event = ", event)


        ## stop loop
        self.disconnect()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## disconnect all the stored connection ids and close gui
    def disconnect(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## disconnect events
        self.canvasSpectrum.mpl_disconnect(self.cidquitSpectrum)
        self.canvasSpectrum.mpl_disconnect(self.cidpressSpectrum)
        self.canvasSpectrum.mpl_disconnect(self.cidmotionSpectrum)
        # self.canvasSpectrum.mpl_disconnect(self.cidkeySpectrum)
        if (self.FITSFlag):
            self.canvasMap.mpl_disconnect(self.cidquitMap)
            self.canvasMap.mpl_disconnect(self.cidpressMap)
            self.canvasMap.mpl_disconnect(self.cidmotionMap)
            # self.canvasMap.mpl_disconnect(self.cidkeyMap)
        self.close()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## replot window
    def replotGUI(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("\n\nself.LowestFreq = ", self.LowestFreq)
        # print ("self.HighestFreq = ", self.HighestFreq)
        # print ("self.GlobalTbg = ", self.GlobalTbg)
        # print ("self.GlobalTSlope = ", self.GlobalTSlope)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## lock element
        self.lockElements = True


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get ranges
        if (self.LastXRange != []):
            self.LastXRange = self.PlotMap.get_xlim()
            self.LastYRange = self.PlotMap.get_ylim()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create map plot
        if (self.FITSFlag):
            self.PlotMap.grid(True)
            self.PlotMap.set_xlabel("RA ({:s})".format(self.RAAxisUNIT))
            self.PlotMap.set_ylabel("Dec ({:s})".format(self.DecAxisUNIT))

            # Debug:
            # print ("self.NumDimFitsFile = ", self.NumDimFitsFile)
            # print ("self.ReverseFlag = ", self.ReverseFlag)
            # print ("self.ChannelIndex = ", self.ChannelIndex)
            # print ("max(int(self.ChannelIndex - 1) , 0) = ", max(int(self.ChannelIndex - 1) , 0))


            ## plot FITS image
            FITSImage = self.FITSCube[max(int(self.ChannelIndex - 1) , 0), :, :]

            # Debug:
            # print ("FITSImage = ", FITSImage)


            ## add FITS image to figure
            im = self.PlotMap.imshow(FITSImage.value, cmap = 'gray', origin = 'lower')


            ## add colorbar
            if (self.OnceFlag):
                self.figMap.colorbar(im)
                self.OnceFlag = False

            # Debug:
            # print ("\n\nself.RAAxis = ", self.RAAxis)
            # print ("self.DecAxis = ", self.DecAxis)


            ## plot lines for cursor cursor
            self.xPosMap = self.PlotMap.axhline(linewidth = 1, color = 'green')             ## the horiz line
            self.yPosMap = self.PlotMap.axvline(linewidth = 1, color = 'green')             ## the vert line


            ## plot marker to indicate position of current spectrum
            if (self.ListSpecCoord == []):
                self.PlotMap.axvline(x = self.xPixelID, color = 'red', linewidth = 1, linestyle = '--')
                self.PlotMap.axhline(y = self.yPixelID, color = 'red', linewidth = 1, linestyle = '--')
            else:
                for PixelCoord in self.ListSpecCoord:
                    xLocal = PixelCoord[0]
                    yLocal = PixelCoord[1]
                    if (len(self.ListSpecCoord) == 1):
                        self.xPixelID = xLocal
                        self.yPixelID = yLocal
                        self.PlotMap.axvline(x = self.xPixelID, color = 'red', linewidth = 1, linestyle = '--')
                        self.PlotMap.axhline(y = self.yPixelID, color = 'red', linewidth = 1, linestyle = '--')
                    else:
                        self.PlotMap.add_patch(patches.Rectangle( (xLocal - 0.5, yLocal - 0.5), 1.0, 1.0, color='green'))


            ## set ranges for map
            if (self.LastXRange == []):
                self.LastXRange = self.PlotMap.get_xlim()
                self.LastYRange = self.PlotMap.get_ylim()
            else:
                self.PlotMap.set_xlim(self.LastXRange[0], self.LastXRange[1])
                self.PlotMap.set_ylim(self.LastYRange[0], self.LastYRange[1])


            ## update map plot of gui
            self.canvasMap.draw()


            ## use tight layout
            #self.figMap.tight_layout()


            ## if channel index is changed stop here
            if (self.RefreshMapOnlyFlag):
                MapCurrFreqString = "{:.2f}".format(self.MapCurrFreq)
                self.LineEditChannel.setText("{:<s}".format(MapCurrFreqString))
                self.SliderChannel.setValue(self.ChannelIndex)
                self.lockElements = False
                return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create spectrum plot


        ## check, if range defined correctly
        f1 = self.LowestFreq
        f2 = self.HighestFreq
        self.LowestFreq = min(f1, f2)
        self.HighestFreq = max(f1, f2)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## sort frequency range list


        ## create list of min. frequencies
        LocalFreqMinList = []
        for LocalRange in self.RangeList:
            LocalFreqMinList.append(LocalRange[0])
        OldLocalFreqMinList = deepcopy(LocalFreqMinList)

        # Debug:
        # print ("self.RangeID = ", self.RangeID)


        ## make coy of range lists
        OldRange = deepcopy(self.RangeList)
        OldFreqStepList = deepcopy(self.FreqStepList)
        OldTbgFlagList = deepcopy(self.TbgFlagList)
        OldTbgList = deepcopy(self.TbgList)
        OldTSlopeList = deepcopy(self.TSlopeList)
        OldnHFlagList = deepcopy(self.nHFlagList)
        OldnHList = deepcopy(self.nHList)
        OldbetaList = deepcopy(self.betaList)
        OldkappaList = deepcopy(self.kappaList)
        OldDustFileNameList = deepcopy(self.DustFileNameList)
        OldBackgroundFileNameList = deepcopy(self.BackgroundFileNameList)
        OldContFuncIDList = deepcopy(self.ContFuncIDList)
        OldContParam1List = deepcopy(self.ContParam1List)
        OldContParam2List = deepcopy(self.ContParam2List)
        OldContParam3List = deepcopy(self.ContParam3List)
        OldContParam4List = deepcopy(self.ContParam4List)
        OldContParam5List = deepcopy(self.ContParam5List)
        OldRangeIDFreq = LocalFreqMinList[self.RangeID]


        ## sort list with min. frequencies
        LocalFreqMinList.sort()


        ## define new range lists
        self.RangeID = LocalFreqMinList.index(OldRangeIDFreq)
        self.RangeList = []
        self.FreqStepList = []
        self.TbgFlagList = []
        self.TbgList = []
        self.TSlopeList = []
        self.nHFlagList = []
        self.nHList = []
        self.betaList = []
        self.kappaList = []
        self.DustFileNameList = []
        self.BackgroundFileNameList = []
        self.ContFuncIDList = []
        self.ContParam1List = []
        self.ContParam2List = []
        self.ContParam3List = []
        self.ContParam4List = []
        self.ContParam5List = []
        for LocalFreqMin in LocalFreqMinList:
            OldIndex = OldLocalFreqMinList.index(LocalFreqMin)
            self.RangeList.append(OldRange[OldIndex])
            self.FreqStepList.append(OldFreqStepList[OldIndex])
            self.TbgFlagList.append(OldTbgFlagList[OldIndex])
            self.TbgList.append(OldTbgList[OldIndex])
            self.TSlopeList.append(OldTSlopeList[OldIndex])
            self.nHFlagList.append(OldnHFlagList[OldIndex])
            self.nHList.append(OldnHList[OldIndex])
            self.betaList.append(OldbetaList[OldIndex])
            self.kappaList.append(OldkappaList[OldIndex])
            self.DustFileNameList.append(OldDustFileNameList[OldIndex])
            self.BackgroundFileNameList.append(OldBackgroundFileNameList[OldIndex])
            self.ContFuncIDList.append(OldContFuncIDList[OldIndex])
            self.ContParam1List.append(OldContParam1List[OldIndex])
            self.ContParam2List.append(OldContParam2List[OldIndex])
            self.ContParam3List.append(OldContParam3List[OldIndex])
            self.ContParam4List.append(OldContParam4List[OldIndex])
            self.ContParam5List.append(OldContParam5List[OldIndex])


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## calculate background continuum
        if (self.AllRangeFlag):
            xData = self.ObsDataFile[:, 0]
            yContFit = numpy.zeros(len(xData))
            for FreqIndex, Freq in enumerate(xData):
                yContFit[int(FreqIndex)] = float(self.GlobalTbg) * (Freq/self.LowestFreq)**float(self.GlobalTSlope)
        else:
            xData = []
            yContFit = []
            for LocalRangeID, LocalRange in enumerate(self.RangeList):


                ## get min. and max. frequency and corresponding indices
                LocalMinFreq = float(LocalRange[0])
                LocalMaxFreq = float(LocalRange[1])
                f1 = max(0, (numpy.abs(self.ObsDataFile[:, 0] - LocalMinFreq)).argmin() - 1)
                f2 = min(len(self.ObsDataFile[:, 0]) - 1, (numpy.abs(self.ObsDataFile[:, 0] - LocalMaxFreq)).argmin() + 1)
                LocalMinFreqIndex = min(f1, f2)
                LocalMaxFreqIndex = max(f1, f2)
                for FreqIndex in range(LocalMinFreqIndex, LocalMaxFreqIndex + 1):
                    Freq = self.ObsDataFile[FreqIndex, 0]
                    xData.append(Freq)
                    yCont = float(self.TbgList[LocalRangeID]) * (Freq/LocalMinFreq)**float(self.TSlopeList[LocalRangeID])
                    yContFit.append(yCont)
            xData = numpy.asarray(xData)
            yContFit = numpy.asarray(yContFit)

        # Debug:
        # print ("xData = ", xData)
        # print ("yContFit = ", yContFit)
        # print ("self.ObsDataFile[:, -1] = ", self.ObsDataFile[:, -1])


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set y-limits
        if (self.InitFlag):
            self.PlotSpec.clear()
            if (not (numpy.isnan(self.LowestFreq) and numpy.isnan(self.HighestFreq))):
                self.PlotSpec.set_xlim(self.LowestFreq, self.HighestFreq)
            if (not (numpy.isnan(self.AllMinInt) and numpy.isnan(self.AllMaxInt))):
                self.PlotSpec.set_ylim(self.AllMinInt * 0.9, self.AllMaxInt * 1.1)
        else:
            xlimits = self.PlotSpec.get_xlim()
            ylimits = self.PlotSpec.get_ylim()
            self.PlotSpec.clear()
            if (xlimits == (0.0, 1.0)):
                if (not (numpy.isnan(self.LowestFreq) and numpy.isnan(self.HighestFreq))):
                    self.PlotSpec.set_xlim(self.LowestFreq, self.HighestFreq)
                if (not (numpy.isnan(self.AllMinInt) and numpy.isnan(self.AllMaxInt))):
                    self.PlotSpec.set_ylim(self.AllMinInt * 0.9, self.AllMaxInt * 1.1)
            else:
                self.PlotSpec.set_xlim(xlimits[0], xlimits[1])
                self.PlotSpec.set_ylim(ylimits[0], ylimits[1])
        if (self.InitFlag and numpy.nanmin(xData) < numpy.nanmax(xData)):
            self.PlotSpec.set_xlim(numpy.nanmin(xData), numpy.nanmax(xData))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set title
        if (self.FITSFlag):
            if (len(self.ListSpecCoord) < 2):
                xCoord = self.RAAxis[self.xPixelID]
                yCoord = self.DecAxis[self.yPixelID]
                LocalTitelString = "Define frequency range(s) and continuum for \npixel = ({:.4f}, {:.4f})".format(yCoord, xCoord)
            else:
                xCoord = self.RAAxis[self.ListSpecCoord[0][0]]
                yCoord = self.DecAxis[self.ListSpecCoord[0][1]]
                LocalTitelString = "Define frequency range(s) and continuum for \npixels = ({:.4f}, {:.4f}) .. ".format(yCoord, xCoord)
        else:
            if (self.DefineNewRangeFlag):
                LocalTitelString = "Select new min. frequency with left mouse button and new max. frequency with right button"
            else:
                LocalTitelString = "Define frequency range(s) and continuum"
        self.PlotSpec.set_title(LocalTitelString)


        ## define grids and axis labels
        self.PlotSpec.grid(True)
        self.PlotSpec.set_ylabel(r"Brightness Temperature [K]")
        self.PlotSpec.set_xlabel("Rest frequency [MHz]")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## mark range
        for LocalRangeID, FreqRange in enumerate(self.RangeList):
            f1 = FreqRange[0]
            f2 = FreqRange[1]
            if (self.RangeID == LocalRangeID and (not self.DefineNewRangeFlag)):
                self.PlotSpec.axvspan(min(f1, f2), max(f1, f2), alpha = 0.2, color = 'blue')
            else:
                self.PlotSpec.axvspan(min(f1, f2), max(f1, f2), alpha = 0.5, color = 'grey')


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add obs. data
        self.PlotSpec.plot(self.ObsDataFile[:, 0], self.ObsDataFile[:, -1], '-', color = 'black', linewidth = 1.0, drawstyle = 'steps-mid')
        self.AllMinInt = numpy.nanmin(self.ObsDataFile[:, -1])
        self.AllMaxInt = numpy.nanmax(self.ObsDataFile[:, -1])
        if (self.InitFlag):
            offset = abs(self.AllMaxInt - self.AllMinInt) * 0.05
            if (self.AllMinInt - offset < self.AllMaxInt + offset):
                self.PlotSpec.set_ylim(self.AllMinInt - offset, self.AllMaxInt + offset)



        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## add overall-fit to plot
        #    DataDir = "/home/tom/ph1/ALMA/myXCLASS-CASA-Interface/NR-version/Linux/developer/real-fits/LineID/SgrB2/HIFI/SgrB2m/overall-fit/runs/"
        #    DataDir += "job__02-02-2017__10-19-54__1004767044/"
        #    NameList = ["SgrB2m_hifi_band1a_5MHz.LM.out.dat", "SgrB2m_hifi_band1b_5MHz.LM.out.dat", "SgrB2m_hifi_band2a_5MHz.LM.out.dat", \
        #                "SgrB2m_hifi_band2b_5MHz.LM.out.dat"]
        #    LocalNoiseList = [0.1, 0.2, 0.25, 0.3]
        #    LocalNoise = LocalNoiseList[self.ObsDataFileID]
        #    FitData = numpy.loadtxt(DataDir + NameList[self.ObsDataFileID])
        #    self.PlotSpec.plot(FitData[:, 0], FitData[:, -1], '-', color='green', linewidth=2.0)
        #    self.PlotSpec.plot(xData, yContFit + LocalNoise, '--', color="blue", linewidth=1.0)
        #    self.PlotSpec.plot(xData, yContFit - LocalNoise, '--', color="blue", linewidth=1.0)
        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++





        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add continuum fit
        self.PlotSpec.plot(xData, yContFit, '-', color = "red", linewidth = 2.0)

        # Debug:
        # print ("self.AllMinInt = ", self.AllMinInt)
        # print ("self.AllMaxInt = ", self.AllMaxInt)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## plot lines for cursor cursor
        self.yPosSpectrum = self.PlotSpec.axhline(color = 'k')                              ## the horiz line
        self.xPosSpectrum = self.PlotSpec.axvline(color = 'k')                              ## the vert line


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add position text location in axes coords
        # self.txt = self.PlotSpec.text(0.5, 0.9, '', transform=self.PlotSpec.transAxes)
        # if (self.InitFlag):
        #     self.txt.set_text( 'Freq.={:1.2f} MHz, Int.={:1.2f} K'.format(0.0, 0.0) )
        self.txt = self.PlotSpec.text(0.2, 0.6, '', transform = self.PlotSpec.transAxes)
        if (not self.SpectrumSelectedFlag):
            self.txt.set_text("Select at least one pixel position!")
        else:
            self.txt.set_text("")

        # Debug:
        # print ("self.xPixelID = ", self.xPixelID)
        # print ("self.yPixelID = ", self.yPixelID)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## reformat labeling
        self.PlotSpec.xaxis.set_major_formatter(pylab.matplotlib.ticker.FormatStrFormatter('%.1f'))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update spectrum plot of gui
        self.canvasSpectrum.draw()


        ## use tight layout
        # self.figSpectrum.tight_layout()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update values in lineEdit boxes
        self.FlagAllRangesDescription.setChecked(self.AllRangeFlag)
        self.ComboRangeSelect.clear()
        self.RangeListString = []
        if (self.DefineNewRangeFlag):
            AddNewRangeString = "Add range: Define new min. freq."
            self.RangeListString.append(AddNewRangeString)
        for LocalRange in self.RangeList:
            MinString = "{:.2f}".format(float(LocalRange[0]))
            MaxString = "{:.2f}".format(float(LocalRange[1]))
            self.RangeListString.append(MinString + " MHz - " + MaxString + " MHz")
        self.ComboRangeSelect.addItems(self.RangeListString)
        if (self.DefineNewRangeFlag):
            self.ComboRangeSelect.setCurrentIndex(0)
        else:
            self.ComboRangeSelect.setCurrentIndex(self.RangeID)
        if (self.AllRangeFlag):
            GlobalTbgString = "{:.4f}".format(self.GlobalTbg)
            self.LineEditTbg.setText(GlobalTbgString)
            GlobalTSlopeString = "{:.2f}".format(self.GlobalTSlope)
            self.LineEditTSlope.setText(GlobalTSlopeString)
        else:
            TbgString = "{:.4f}".format(self.TbgList[self.RangeID])
            self.LineEditTbg.setText(TbgString)
            TSlopeString = "{:.2f}".format(self.TSlopeList[self.RangeID])
            self.LineEditTSlope.setText(TSlopeString)
        if (self.FITSFlag):
            MapCurrFreqString = "{:.2f}".format(self.MapCurrFreq)
            self.LineEditChannel.setText("{:<s}".format(MapCurrFreqString))
            self.SliderChannel.setValue(self.ChannelIndex)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## unlock elements
        self.lockElements = False

        # Debug:
        # print ("self.RangeID = ", self.RangeID)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what happens on key event
    def OnKeySpectrum(self, event):
        """

    input parameters:
    -----------------

        - event:            evenet object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ('you pressed', event.key, event.xdata, event.ydata)


        ## define event on "-" key:  remove selected frequency range
        if (event.key == "-"):
            x, y = event.xdata, event.ydata                                                 ## get position of mouse cursor
            LocalRangeID, NearestMinFreqRangeID, NearestMaxFreqRangeID = self.ChangeFreqRange(x)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## remove range if current mouse position defines range
            if (LocalRangeID > (-1)):
                del self.RangeList[LocalRangeID]
                if (self.RangeList == []):
                    self.RangeList = [[self.LowestFreq, self.HighestFreq]]
                    self.RangeID = 0


                ## update GUI
                self.replotGUI()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what to do when mouse is moved in spectrum plot
    def MouseMoveSpectrum(self, eventSpec):
        """

    input parameters:
    -----------------

        - eventSpec:         evenet object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("eventSpec = ", eventSpec)


        ## check, if mouse cursor position is in allowed range
        if (not eventSpec.inaxes):
            return


        ## get position of mouse cursor
        x, y = eventSpec.xdata, eventSpec.ydata


        ## update the line positions
        # self.yPosSpectrum.set_ydata(y )
        # self.xPosSpectrum.set_xdata(x )
        # self.txt.set_text( 'Freq.={:1.2f} MHz, Int.={:1.2f} K'.format(x, y) )
        # self.canvasSpectrum.draw()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what to do when mouse is moved in map plot
    def MouseMoveMap(self, eventMap):
        """

    input parameters:
    -----------------

        - eventMap:            evenet object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("eventMap = ", eventMap)


        ## get position of mouse cursor
        x, y = eventMap.xdata, eventMap.ydata

        # Debug:
        # print ("y, x = ", y, x, eventMap.inaxes)


        ## update the line positions
        self.yPosMap.set_ydata(y)
        self.xPosMap.set_xdata(x)


        ## set title
        if (eventMap.inaxes):
            xCoord = self.RAAxis[0] + self.RAAxisDelta * x
            yCoord = self.DecAxis[0] + self.DecAxisDelta * y
            self.PlotMap.format_coord = lambda x, y: 'x=%.5f,y=%.5f' % (yCoord, xCoord)         ## show cursor position in toolbar
        else:
            self.PlotMap.set_title(" ")


        ## update GUI
        self.canvasMap.draw()


        ## use tight layout
        # self.figMap.tight_layout()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get parameter
    def getParam(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## define lists of min. and max. freq.
        LocalFreqMinListOut = []
        LocalFreqMaxListOut = []
        TbgListOut = []
        TSlopeListOut = []
        for LocalRangeID, LocalRange in enumerate(self.RangeList):
            LocalFreqMinListOut.append(LocalRange[0])
            LocalFreqMaxListOut.append(LocalRange[1])
            if (self.AllRangeFlag):
                tslope = self.GlobalTSlope
                tback = float(self.GlobalTbg) * (LocalRange[0] / self.LowestFreq)**float(tslope)
            else:
                tback = self.TbgList[LocalRangeID]
                tslope = self.TSlopeList[LocalRangeID]
            TbgListOut.append(tback)
            TSlopeListOut.append(tslope)


        ## make pixel coordinates available
        if (self.ListSpecCoord == []):
            self.ListSpecCoord = [[self.xPixelID, self.yPixelID]]


        ## we're done
        NewRangeParametersDict = {}
        NewRangeParametersDict['FreqMin'] = LocalFreqMinListOut
        NewRangeParametersDict['FreqMax'] = LocalFreqMaxListOut
        NewRangeParametersDict['FreqStep'] = self.FreqStepList
        NewRangeParametersDict['TbgFlag'] = self.TbgFlagList
        NewRangeParametersDict['Tbg'] = TbgListOut
        NewRangeParametersDict['TSlope'] = TSlopeListOut
        NewRangeParametersDict['nHFlag'] = self.nHFlagList
        NewRangeParametersDict['nH'] = self.nHList
        NewRangeParametersDict['beta'] = self.betaList
        NewRangeParametersDict['kappa'] = self.kappaList
        NewRangeParametersDict['DustFile'] = self.DustFileNameList
        NewRangeParametersDict['BackgroundFile'] = self.BackgroundFileNameList
        NewRangeParametersDict['ContFuncID'] = self.ContFuncIDList
        NewRangeParametersDict['ContParam1'] = self.ContParam1List
        NewRangeParametersDict['ContParam2'] = self.ContParam2List
        NewRangeParametersDict['ContParam3'] = self.ContParam3List
        NewRangeParametersDict['ContParam4'] = self.ContParam4List
        NewRangeParametersDict['ContParam5'] = self.ContParam5List


        ## we're done
        return (self.ok, NewRangeParametersDict, self.ObsDataFile, self.ListSpecCoord)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## modify frequency range
    def ChangeFreqRange(self, SelectedFreq):
        """

    input parameters:
    -----------------

        - SelectedFreq:     selected frequency


    output parameters:
    ------------------

        - LocalRangeID:     id of selected range

        - NearestMinFreqRangeID:    id of nearest left range

        - NearestMaxFreqRangeID:    id of nearest right range
        """

        # Debug:
        # print ("SelectedFreq = ", SelectedFreq)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine selected frequency range id
        LocalRangeID = (-1)
        LowestDistMin = 1.e99
        NearestMinFreq = 0.0
        NearestMinFreqRangeID = (-1)
        LowestDistMax = 1.e99
        NearestMaxFreq = 0.0
        NearestMaxFreqRangeID = (-1)
        for RangeID, FreqRange in enumerate(self.RangeList):
            LocalMinFreq = FreqRange[0]
            LocalMaxFreq = FreqRange[1]


            ## determine minimal distance and corresponding range id for min. freq.
            LocalDistMin = (SelectedFreq - LocalMinFreq)
            if (LocalDistMin < LowestDistMin and LocalDistMin > 0.0):
                LowestDistMin = LocalDistMin
                NearestMinFreq = LocalMinFreq
                NearestMinFreqRangeID = RangeID


            ## determine minimal distance and corresponding range id for max. freq.
            LocalDistMax = (LocalMaxFreq - SelectedFreq)
            if (LocalDistMax < LowestDistMax and LocalDistMax > 0.0):
                LowestDistMax = LocalDistMax
                NearestMaxFreq = LocalMaxFreq
                NearestMaxFreqRangeID = RangeID


            ## check, if selected frequency is located in existing freq. range
            if (LocalMinFreq <= SelectedFreq and SelectedFreq <= LocalMaxFreq):
                LocalRangeID = RangeID
                break


        ## we're done
        return LocalRangeID, NearestMinFreqRangeID, NearestMaxFreqRangeID
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what happens on mouse click (NOT defined at the moment)
    def onclickSpectrum(self, event):
        """

    input parameters:
    -----------------

        - event:            evenet object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ('events', event.key, event.xdata, event.ydata, event.inaxes, event.button, event.dblclick)


        ## check mouse buttons
        # if (event.button == 1):
        #    print ("You pressed the left mouse button!")
        # elif (event.button == 2):
        #    print ("You pressed the middle mouse button!")
        # elif (event.button == 3):
        #    print ("You pressed the right mouse button!")


        ## check, if mouse cursor position is in allowed range
        if (not event.inaxes):
            return


        ## define event on left mouse button pressed
        if (event.dblclick and (event.button == 1 or event.button == 3)):
            x, y = event.xdata, event.ydata                                                 ## get position of mouse cursor
            LocalRangeID, NearestMinFreqRangeID, NearestMaxFreqRangeID = self.ChangeFreqRange(x)

            # Debug:
            # print ("\n\nLocalRangeID = ", LocalRangeID)
            # print ("NearestMinFreqRangeID = ", NearestMinFreqRangeID)
            # print ("NearestMaxFreqRangeID = ", NearestMaxFreqRangeID)
            # print ("self.RangeID = ", self.RangeID)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## change limits of existing range
            if (not self.DefineNewRangeFlag):
                if (LocalRangeID > (-1)):
                    LocalFreqMin = self.RangeList[LocalRangeID][0]
                    LocalFreqMax = self.RangeList[LocalRangeID][1]
                    if (event.button == 1):
                        LocalFreqMin = x
                    elif (event.button == 3):
                        LocalFreqMax = x
                    self.RangeList[LocalRangeID] = [LocalFreqMin, LocalFreqMax]
                else:


                    ## define new min. freq.
                    if (event.button == 1):
                        LocalFreqMin = x
                        CurrentFreqMax = self.RangeList[self.RangeID][1]
                        if (self.RangeID > 0):
                            LeftFreqMax = self.RangeList[self.RangeID - 1][1]
                        else:
                            LeftFreqMax = self.LowestFreq


                        ## make sure that new min. freq. is properly defined
                        if (x < CurrentFreqMax and x > LeftFreqMax):
                            self.RangeList[self.RangeID] = [LocalFreqMin, CurrentFreqMax]


                    ## define new max. freq.
                    elif (event.button == 3):
                        LocalFreqMax = x
                        CurrentFreqMin = self.RangeList[self.RangeID][0]
                        if (self.RangeID < len(self.RangeList) - 1):
                            RightFreqMin = self.RangeList[self.RangeID + 1][0]
                        else:
                            RightFreqMin = self.HighestFreq

                        # Debug:
                        # print ("\n\nself.RangeID = ", self.RangeID)
                        # print ("CurrentFreqMin, LocalFreqMax = ", CurrentFreqMin, LocalFreqMax)
                        # print ("len(self.RangeList) = ", len(self.RangeList))
                        # print ("self.RangeList = ", self.RangeList)
                        # print ("RightFreqMin = ", RightFreqMin)


                        ## make sure that new max. freq. is properly defined
                        if (x > CurrentFreqMin and x < RightFreqMin):
                            self.RangeList[self.RangeID] = [CurrentFreqMin, LocalFreqMax]


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## define limits of new range
            elif (LocalRangeID == (-1)):                                                    ## make sure that not an already existing ranges is selected
                self.DefineNewRangeFlag = False

                # Debug:
                # print ("\n\nNearestMinFreqRangeID = ", NearestMinFreqRangeID)
                # print ("NearestMaxFreqRangeID = ", NearestMaxFreqRangeID)


                ## add range if no range is already defined at current mouse position
                if (NearestMinFreqRangeID > (-1)):
                    NearestFreqMax = min(self.RangeList[NearestMinFreqRangeID][1] - (self.FreqStep * 5), self.HighestFreq)
                else:
                    NearestFreqMax = self.LowestFreq


                if (NearestMaxFreqRangeID > (-1)):
                    NearestFreqMin = max(self.RangeList[NearestMaxFreqRangeID][0] + (self.FreqStep * 5), self.LowestFreq)
                else:
                    NearestFreqMin = self.HighestFreq


                ## define new min. and max. freq. for new freq. range
                if (event.button == 1):
                    NewMinFreq = x
                    NewMaxFreq = min(NearestFreqMin, x + (self.FreqStep * self.InitNumDataPtsFreqRange))
                else:
                    NewMinFreq = max(NearestFreqMax, x - (self.FreqStep * self.InitNumDataPtsFreqRange))
                    NewMaxFreq = x

                # Debug:
                # print ("NewMinFreq = ", NewMinFreq)
                # print ("NewMaxFreq = ", NewMaxFreq)


                ## update range list
                NewID = None
                for RangeID, LocalRange in enumerate(self.RangeList):
                    LocalMinFreq = LocalRange[0]
                    if (LocalMinFreq > NewMaxFreq):
                        NewID = RangeID
                        break

                # Debug:
                # print ("NewID = ", NewID)


                if (NewID is None):
                    self.RangeList.append([NewMinFreq, NewMaxFreq])
                    self.FreqStepList.append(self.FreqStep)
                    self.TbgFlagList.append(self.TbgFlagDefault)
                    self.TbgList.append(self.GlobalTbg)
                    self.TSlopeList.append(self.GlobalTSlope)
                    self.nHFlagList.append(self.nHFlagDefault)
                    self.nHList.append(self.nHDefault)
                    self.betaList.append(self.betaDefaultIn)
                    self.kappaList.append(self.kappaDefault)
                    self.DustFileNameList.append(self.DustFileNamesDefaultIn)
                    self.BackgroundFileNameList.append(self.BackgroundFileNamesDefaultIn)

                    self.ContFuncIDList.append(self.ContFuncIDDefaultIn)
                    self.ContParam1List.append(self.ContParam1DefaultIn)
                    self.ContParam2List.append(self.ContParam2DefaultIn)
                    self.ContParam3List.append(self.ContParam3DefaultIn)
                    self.ContParam4List.append(self.ContParam4DefaultIn)
                    self.ContParam5List.append(self.ContParam5DefaultIn)
                    self.RangeID = len(self.RangeList) - 1
                else:
                    self.RangeList.insert(NewID, [NewMinFreq, NewMaxFreq])
                    self.FreqStepList.insert(NewID, self.FreqStep)
                    self.TbgFlagList.insert(NewID, self.TbgFlagDefault)
                    self.TbgList.insert(NewID, self.GlobalTbg)
                    self.TSlopeList.insert(NewID, self.GlobalTSlope)
                    self.nHFlagList.insert(NewID, self.nHFlagDefault)
                    self.nHList.insert(NewID, self.nHDefault)
                    self.betaList.insert(NewID, self.betaDefaultIn)
                    self.kappaList.insert(NewID, self.kappaDefault)
                    self.DustFileNameList.insert(NewID, self.DustFileNamesDefaultIn)
                    self.BackgroundFileNameList.insert(NewID, self.BackgroundFileNamesDefaultIn)
                    self.ContFuncIDList.insert(NewID, self.ContFuncIDDefaultIn)
                    self.ContParam1List.insert(NewID, self.ContParam1DefaultIn)
                    self.ContParam2List.insert(NewID, self.ContParam2DefaultIn)
                    self.ContParam3List.insert(NewID, self.ContParam3DefaultIn)
                    self.ContParam4List.insert(NewID, self.ContParam4DefaultIn)
                    self.ContParam5List.insert(NewID, self.ContParam5DefaultIn)
                    self.RangeID = NewID

                    # Debug:
                    # print ("NewID = ", NewID)
                    # print ("self.RangeList = ", self.RangeList)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## refresh GUI
            self.replotGUI()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what happens on mouse click in map plot
    def onclickMap(self, event):
        """

    input parameters:
    -----------------

        - event:            evenet object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ('button={:d}, x={:d}, y={:d}, xdata={:.2f}, ydata={:.2f}'.format(event.button, event.x, event.y, event.xdata, event.ydata))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if mouse cursor position is in allowed range
        if (not event.inaxes):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set title
        x, y = event.xdata, event.ydata                                                     ## get position of mouse cursor


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define event on left mouse button pressed
        if (event.button == 1 and event.dblclick):
            self.SpectrumSelectedFlag = True
            self.xPixelID = int(round(x))
            self.yPixelID = int(round(y))
            self.FITSPixelSpecImport()

            # Debug:
            # print ("self.xPixelID = ", self.xPixelID)
            # print ("self.yPixelID = ", self.yPixelID)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## what happens on double-click
            self.ListSpecCoord = []
            self.ListSpecData = []


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## replot map plot
            self.InitFlag = True
            self.replotGUI()
            self.InitFlag = False


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## on single click with right mouse button average over selected spectra
        if (event.button == 3):
            self.SpectrumSelectedFlag = True
            self.xPixelID = int(round(x))
            self.yPixelID = int(round(y))
            NewCoordinate = [self.xPixelID, self.yPixelID]
            if (NewCoordinate in self.ListSpecCoord):                                       ## remove entry
                if (len(self.ListSpecCoord) > 1):                                           ## check, if at least two pixels are defined
                    i = self.ListSpecCoord.index(NewCoordinate)
                    del self.ListSpecCoord[i]
                    del self.ListSpecData[i]
                    self.xPixelID = int(round(self.ListSpecCoord[-1][0]))
                    self.yPixelID = int(round(self.ListSpecCoord[-1][1]))
                else:
                    return
            else:
                self.ListSpecCoord.append(NewCoordinate)                                    ## add current coordinate to list of pixel coordinates
                self.FITSPixelSpecImport()
                self.ListSpecData.append(deepcopy(self.ObsDataFile))


            ## determine avarage of all selected spectra
            avSpec = deepcopy(self.ObsDataFile[:, 1])
            NumSpec = len(self.ListSpecData)
            for dataID, data in enumerate(self.ListSpecData):
                avSpec += deepcopy(data[:, 1])
            self.ObsDataFile[:, 1] = deepcopy(avSpec) / float(NumSpec)

            # Debug:
            # print ("\n\nNumSpec = ", NumSpec)
            #   for data in self.ListSpecData:
            #       print data[0])
            # print ("self.ObsDataFile[0] = ", self.ObsDataFile[0])


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## replot map plot
            self.AllMinInt = 1.e99                                                      ## reset global minimum intensity
            self.AllMaxInt = -1.e99                                                     ## reset global maximum intensity
            self.InitFlag = True
            self.replotGUI()
            self.InitFlag = False


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import fits file
    def FITSImport(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## import FITS file
        self.FITSCube = SpectralCube.read(self.UserObsDataFileName)
        self.FITSCube = self.FITSCube.with_spectral_unit(u.MHz)


        ## allow huge operations
        self.FITSCube.allow_huge_operations = True


        ## deactivate silencing warnings
        warnings.filterwarnings(action = 'ignore', category = SpectralCubeWarning, append = True)
        warnings.filterwarnings(action = 'ignore', category = WCSMismatchWarning, append = True)
        warnings.filterwarnings(action = 'ignore', category = SliceWarning, append = True)
        warnings.filterwarnings(action = 'ignore', category = StokesWarning, append = True)
        warnings.simplefilter('ignore')


        ## define parameters for RA axis
        velo, dec, ra = self.FITSCube.world[:]
        self.DecAxis = dec[0, :, 0].value
        self.DecAxisUNIT = dec[0, :, 0].unit
        self.RAAxis = ra[0, 0, :].value
        self.RAAxisUNIT = ra[0, 0, :].unit


        ## define parameters for Dec axis
        self.DecAxisDelta = self.DecAxis[1] - self.DecAxis[0]
        self.yPixelID = len(self.DecAxis) / 2.0

        # Debug:
        # print ("self.DecAxisDelta = ", self.DecAxisDelta)
        # print ("self.yPixelID = ", self.yPixelID)


        ## define parameters for RA axis
        self.RAAxisDelta = self.RAAxis[1] - self.RAAxis[0]
        self.xPixelID = len(self.RAAxis) / 2.0

        # Debug:
        # print ("self.RAAxis = ", self.RAAxis)
        # print ("self.RAAxisDelta = ", self.RAAxisDelta)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define parameters for freq. axis


        ## get lowest and highest frequency point
        self.FrequencyAxis = self.FITSCube.spectral_axis.value
        self.MapFreqMin = numpy.nanmin(self.FrequencyAxis)
        self.MapFreqMax = numpy.nanmax(self.FrequencyAxis)
        self.MapFreqStep = self.FrequencyAxis[1] - self.FrequencyAxis[0]

        # Debug:
        # print ("self.FrequencyAxis = ", self.FrequencyAxis)
        # print ("self.MapFreqMin = ", self.MapFreqMin)
        # print ("self.MapFreqMax = ", self.MapFreqMax)
        # print ("self.MapFreqStep = ", self.MapFreqStep)


        ## get spectrum from central pixel
        self.xPixelID = 0
        self.yPixelID = 0
        self.FITSPixelSpecImport()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import pixel spectrum from fits file
    def FITSPixelSpecImport(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## extract spectrum at current position
        self.ObsDataFile = self.FITSCube[:, self.yPixelID, self.xPixelID].value
        self.ObsDataFile = numpy.vstack((self.FrequencyAxis, self.ObsDataFile)).T


        ## define lower and upper frequency limit
        f0 = numpy.nanmin(self.ObsDataFile[:, 0])
        f1 = numpy.nanmax(self.ObsDataFile[:, 0])
        if (self.LowestFreq == self.HighestFreq):
            self.LowestFreq = f0
            self.HighestFreq = f1
        else:
            if (self.LowestFreq < f0):
                self.LowestFreq = f0
            if (f1 < self.HighestFreq):
                self.HighestFreq = f1

        # Debug:
        # print ("f0 = ", f0)
        # print ("f1 = ", f1)
        # print ("self.LowestFreq = ", self.LowestFreq)
        # print ("self.HighestFreq = ", self.HighestFreq)
        # print ("self.ObsDataFile[:, 0] = ", self.ObsDataFile[:, 0])
        # print ("self.ObsDataFile[:, 1] = ", self.ObsDataFile[:, 1])


        ## we're done
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define frequency range and continuum description
##
def SelectRange(ObsDataFileID, ObsDataFileNameIn, ObsDataFileIn, LocalNumberExpRangesIn, LocalFreqMinListIn, LocalFreqMaxListIn, LocalFreqStepIn, \
                TbgFlagListIn, TbgListIn, TSlopeListIn, nHFlagListIn, nHListIn, betaListIn, kappaListIn, DustFileNamesIn, BackgroundFileNamesIn, \
                ContFuncIDListIn, ContParam1ListIn, ContParam2ListIn, \
                ContParam3ListIn, ContParam4ListIn, ContParam5ListIn, TbgFlagDefaultIn, TbgDefaultIn, TslopeDefaultIn, nHFlagDefaultIn, nHDefaultIn, \
                kappaDefaultIn, betaDefaultIn, DustFileNamesDefaultIn, BackgroundFileNamesDefaultIn, \
                ContFuncIDDefaultListIn, ContParam1DefaultListIn, \
                ContParam2DefaultListIn, ContParam3DefaultListIn, ContParam4DefaultListIn, ContParam5DefaultListIn, SizeTelescopeIn, BMINIn, BMAJIn, \
                BPAIn, InterferoFlagIn, vLSRIn, RedShiftIn, ObsDataFileIDIn, FITSSubDirIN, NumObsDataFileIN):
    """

input parameters:
-----------------

    - ObsDataFileID:                obs. data file index

    - ObsDataFileNameIn:            path and name of obs. data file

    - ObsDataFileIn:                array containing data

    - LocalNumberExpRangesIn:       number of frequency ranges

    - LocalFreqMinInListIn:         lowest frequency of data file

    - LocalFreqMaxListIn:           highest frequency of data file

    - LocalFreqStepIn:              step frequency

    - TbgFlagListIn:                temperature background flag

    - TbgListIn:                    initial guess for T_bg

    - TSlopeListIn:                 initial guess for T_slope

    - nHFlagListIn:                 global dust flag

    - nHListIn:                     hydrogen column density

    - betaListIn:                   spectral index

    - kappaListIn:                  kappa

    - DustFileNamesIn:              path and name of dust file

    - BackgroundFileNamesIn:        path and name of background file

    - ContFuncIDListIn:             function index for phen. continuum description

    - ContParam1ListIn:             parameter 1 for phen. continuum description

    - ContParam2ListIn:             parameter 2 for phen. continuum description

    - ContParam3ListIn:             parameter 3 for phen. continuum description

    - ContParam4ListIn:             parameter 4 for phen. continuum description

    - ContParam5ListIn:             parameter 5 for phen. continuum description

    - TbgFlagDefaultIn:             default value for TbgFlag

    - TbgDefaultIn:                 default value for Tbg

    - TslopeDefaultIn:              default value for Tslope

    - nHFlagDefaultIn:              default value for nHFlag

    - nHDefaultIn:                  default value for nH

    - kappaDefaultIn:               default value for kappa

    - betaDefaultIn:                default value for beta

    - DustFileNamesDefaultIn:       default value for path and name of dust file

    - BackgroundFileNamesDefaultIn: default value for path and name of background file

    - ContFuncIDDefaultListIn:      default value for cont. function index

    - ContParam1DefaultListIn:      default value for cont. parameter 1

    - ContParam2DefaultListIn:      default value for cont. parameter 2

    - ContParam3DefaultListIn:      default value for cont. parameter 3

    - ContParam4DefaultListIn:      default value for cont. parameter 4

    - ContParam5DefaultListIn:      default value for cont. parameter 5

    - SizeTelescopeIn:              value for size of telescope

    - BMINIn:                       value for BMIN

    - BMAJIn:                       value for BMAJ

    - BPAIn:                        value for BPA

    - InterferoFlagIn:              value for interferometer flag

    - vLSRIn:                       value for global v_LSR

    - RedShiftIn:                   value for redshift

    - ObsDataFileIDIn:              current obs. data file id

    - FITSSubDirIN:                 path and name of FITS subdirectory

    - NumObsDataFileIN:             total number of obs. data files


output parameters:
------------------

    - ok:                           status flag

    - LocalFreqMinList:             new lower frequency

    - LocalFreqMaxList:             new upper frequency

    - TbgList:                      new background temperature

    - TSlopeList:                   new temperature slope
    """

    # Debug:
    # print ("ObsDataFileID = ", ObsDataFileID)
    # print ("ObsDataFileIn = ", ObsDataFileIn)
    # print ("LocalNumberExpRangesIn = ", LocalNumberExpRangesIn)
    # print ("LocalFreqMinInListIn = ", LocalFreqMinInListIn)
    # print ("LocalFreqStepIn = ", LocalFreqStepIn)
    # print ("LocalFreqMaxListIn = ", LocalFreqMaxListIn)
    # print ("TbgFlagListIn = ", TbgFlagListIn)
    # print ("TbgListIn = ", TbgListIn)
    # print ("TSlopeListIn = ", TSlopeListIn)
    # print ("nHFlagListIn = ", nHFlagListIn)
    # print ("nHListIn = ", nHListIn)
    # print ("betaListIn = ", betaListIn)
    # print ("kappaListIn = ", kappaListIn)
    # print ("DustFileNamesIn = ", DustFileNamesIn)
    # print ("BackgroundFileNamesIn = ", BackgroundFileNamesIn)
    # print ("ContFuncIDListIn = ", ContFuncIDListIn)
    # print ("ContParam1ListIn = ", ContParam1ListIn)
    # print ("ContParam2ListIn = ", ContParam2ListIn)
    # print ("ContParam3ListIn = ", ContParam3ListIn)
    # print ("ContParam4ListIn = ", ContParam4ListIn)
    # print ("ContParam5ListIn = ", ContParam5ListIn)
    # print ("ContFuncIDDefaultListIn = ", ContFuncIDDefaultListIn)
    # print ("ContParam1DefaultListIn = ", ContParam1DefaultListIn)
    # print ("ContParam2DefaultListIn = ", ContParam2DefaultListIn)
    # print ("ContParam3DefaultListIn = ", ContParam3DefaultListIn)
    # print ("ContParam4DefaultListIn = ", ContParam4DefaultListIn)
    # print ("ContParam5DefaultListIn = ", ContParam5DefaultListIn)
    # print ("ObsDataFileIDIn = ", ObsDataFileIDIn)
    # print ("ObsDataFileNameIn = ", ObsDataFileNameIn)
    # print ("SizeTelescopeIn = ", SizeTelescopeIn)
    # print ("BMINIn = ", BMINIn)
    # print ("BMAJIn  = ", BMAJIn)
    # print ("BPAIn = ", BPAIn)
    # print ("InterferoFlagIn = ", InterferoFlagIn)
    # print ("vLSRIn = ", vLSRIn)
    # print ("RedShiftIn = ", RedShiftIn)
    # print ("ObsDataFileIDIn = ", ObsDataFileIDIn)
    # print ("FITSSubDirIN = ", FITSSubDirIN)
    # print ("NumObsDataFileIN = ", NumObsDataFileIN)
    # print ("DustFileNamesDefaultIn = ", DustFileNamesDefaultIn)
    # print ("BackgroundFileNamesDefaultIn = ", BackgroundFileNamesDefaultIn)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## extract parameters from parameter lists which correspond to current obs. data file
    LocalRangeListNumRange       = [x[-1] for x in LocalNumberExpRangesIn if x[0] == ObsDataFileID]
    LocalRangeListFreqMin        = []
    for x in LocalFreqMinListIn:
        if (x[0] == ObsDataFileID):
            val = x[-1]
            try:
                val = float(val)
            except:
                val = 0.0
        LocalRangeListFreqMin.append(val)
    LocalRangeListFreqMax        = []
    for x in LocalFreqMaxListIn:
        if (x[0] == ObsDataFileID):
            val = x[-1]
            try:
                val = float(val)
            except:
                val = 0.0
        LocalRangeListFreqMax.append(val)

    LocalRangeListFreqStep       = [x[-1] for x in LocalFreqStepIn        if x[0] == ObsDataFileID]
    LocalRangeListTbgFlag        = [x[-1] for x in TbgFlagListIn          if x[0] == ObsDataFileID]


    LocalRangeListTbg        = []
    for x in TbgListIn:
        if (x[0] == ObsDataFileID):
            val = x[-1]
            try:
                val = float(val)
            except:
                val = 0.0
        LocalRangeListTbg.append(val)

    LocalRangeListTSlope        = []
    for x in TSlopeListIn:
        if (x[0] == ObsDataFileID):
            val = x[-1]
            try:
                val = float(val)
            except:
                val = 0.0
        LocalRangeListTSlope.append(val)

    LocalRangeListnHFlag         = [x[-1] for x in nHFlagListIn           if x[0] == ObsDataFileID]
    LocalRangeListnH             = [x[-1] for x in nHListIn               if x[0] == ObsDataFileID]
    LocalRangeListbeta           = [x[-1] for x in betaListIn             if x[0] == ObsDataFileID]
    LocalRangeListkappa          = [x[-1] for x in kappaListIn            if x[0] == ObsDataFileID]
    LocalRangeListDustFile       = [x[-1] for x in DustFileNamesIn        if x[0] == ObsDataFileID]
    LocalRangeListBackgroundFile = [x[-1] for x in BackgroundFileNamesIn  if x[0] == ObsDataFileID]
    LocalRangeListContFuncID     = [x[-1] for x in ContFuncIDListIn       if x[0] == ObsDataFileID]
    LocalRangeListContParam1     = [x[-1] for x in ContParam1ListIn       if x[0] == ObsDataFileID]
    LocalRangeListContParam2     = [x[-1] for x in ContParam2ListIn       if x[0] == ObsDataFileID]
    LocalRangeListContParam3     = [x[-1] for x in ContParam3ListIn       if x[0] == ObsDataFileID]
    LocalRangeListContParam4     = [x[-1] for x in ContParam4ListIn       if x[0] == ObsDataFileID]
    LocalRangeListContParam5     = [x[-1] for x in ContParam5ListIn       if x[0] == ObsDataFileID]
    LocalRangeListSizeTele       = [x[-1] for x in SizeTelescopeIn        if x[0] == ObsDataFileID]
    LocalRangeListBMIN           = [x[-1] for x in BMINIn                 if x[0] == ObsDataFileID]
    LocalRangeListBMAJ           = [x[-1] for x in BMAJIn                 if x[0] == ObsDataFileID]
    LocalRangeListBPA            = [x[-1] for x in BPAIn                  if x[0] == ObsDataFileID]
    LocalRangeListInter          = [x[-1] for x in InterferoFlagIn        if x[0] == ObsDataFileID]
    LocalRangeListvLSR           = [x[-1] for x in vLSRIn                 if x[0] == ObsDataFileID]
    LocalRangeListRedShift       = [x[-1] for x in RedShiftIn             if x[0] == ObsDataFileID]

    # Debug:
    # print ("LocalRangeListNumRange       = ", LocalRangeListNumRange)
    # print ("LocalRangeListFreqMin        = ", LocalRangeListFreqMin)
    # print ("LocalRangeListFreqMax        = ", LocalRangeListFreqMax)
    # print ("LocalRangeListFreqStep       = ", LocalRangeListFreqStep)
    # print ("LocalRangeListTbgFlag        = ", LocalRangeListTbgFlag)
    # print ("LocalRangeListTbg            = ", LocalRangeListTbg)
    # print ("LocalRangeListTSlope         = ", LocalRangeListTSlope)
    # print ("LocalRangeListnHFlag         = ", LocalRangeListnHFlag)
    # print ("LocalRangeListnH             = ", LocalRangeListnH)
    # print ("LocalRangeListbeta           = ", LocalRangeListbeta)
    # print ("LocalRangeListkappa          = ", LocalRangeListkappa)
    # print ("LocalRangeListDustFile       = ", LocalRangeListDustFile)
    # print ("LocalRangeListBackgroundFile = ", LocalRangeListBackgroundFile)
    # print ("LocalRangeListContFuncID     = ", LocalRangeListContFuncID)
    # print ("LocalRangeListContParam1     = ", LocalRangeListContParam1)
    # print ("LocalRangeListContParam2     = ", LocalRangeListContParam2)
    # print ("LocalRangeListContParam3     = ", LocalRangeListContParam3)
    # print ("LocalRangeListContParam4     = ", LocalRangeListContParam4)
    # print ("LocalRangeListContParam5     = ", LocalRangeListContParam5)
    # print ("LocalRangeListSizeTele       = ", LocalRangeListSizeTele)
    # print ("LocalRangeListInter          = ", LocalRangeListInter)
    # print ("LocalRangeListBMIN           = ", LocalRangeListBMIN)
    # print ("LocalRangeListBMAJ           = ", LocalRangeListBMAJ)
    # print ("LocalRangeListBPA            = ", LocalRangeListBPA)
    # print ("LocalRangeListvLSR           = ", LocalRangeListvLSR)
    # print ("LocalRangeListRedShift       = ", LocalRangeListRedShift)


    ## suppress warnings
    warnings.simplefilter('once', UserWarning)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    ok = 0
    NewRangeParametersDict = {}


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check, if fits file is defined
    LocalFITSFlag = False
    if (ObsDataFileNameIn.endswith(".fits")):
        LocalFITSFlag = True
        try:
            import astropy
            from astropy.io import fits
        except ImportError:
            ok = 1
            return (ok, NewRangeParametersDict)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start GUI
    RangeSelection = RangeSelectionGUI(ObsDataFileNameIn = ObsDataFileNameIn, ObsDataFileIn = ObsDataFileIn, \
                                       LocalNumRangeListIn = LocalRangeListNumRange, LocalFreqMinListIn = LocalRangeListFreqMin, \
                                       LocalFreqMaxListIn = LocalRangeListFreqMax, LocalFreqStepListIn = LocalRangeListFreqStep, \
                                       TbgFlagListIn = LocalRangeListTbgFlag, TbgListIn = LocalRangeListTbg, TSlopeListIn = LocalRangeListTSlope, \
                                       nHFlagListIn = LocalRangeListnHFlag, nHListIn = LocalRangeListnH, betaListIn = LocalRangeListbeta, \
                                       kappaListIn = LocalRangeListkappa, DustFileIn = LocalRangeListDustFile, \
                                       BackgroundFileIn = LocalRangeListBackgroundFile, \
                                       ContFuncIDListIn = LocalRangeListContFuncID, ContParam1ListIn = LocalRangeListContParam1, \
                                       ContParam2ListIn = LocalRangeListContParam2, ContParam3ListIn = LocalRangeListContParam3, \
                                       ContParam4ListIn = LocalRangeListContParam4, ContParam5ListIn = LocalRangeListContParam5, \
                                       TbgFlagDefaultIn = TbgFlagDefaultIn, TbgDefaultIn = TbgDefaultIn, TslopeDefaultIn = TslopeDefaultIn, \
                                       nHFlagDefaultIn = nHFlagDefaultIn, nHDefaultIn = nHDefaultIn, kappaDefaultIn = kappaDefaultIn, \
                                       betaDefaultIn = betaDefaultIn, DustFileNamesDefaultIn = DustFileNamesDefaultIn, \
                                       BackgroundFileNamesDefaultIn = BackgroundFileNamesDefaultIn, \
                                       ContFuncIDDefaultIn = ContFuncIDDefaultListIn, ContParam1DefaultIn = ContParam1DefaultListIn, \
                                       ContParam2DefaultIn = ContParam2DefaultListIn, ContParam3DefaultIn = ContParam3DefaultListIn, \
                                       ContParam4DefaultIn = ContParam4DefaultListIn, ContParam5DefaultIn = ContParam5DefaultListIn, \
                                       ObsDataFileIDIn = ObsDataFileIDIn, SizeTelescopeIn = LocalRangeListSizeTele, BMINIn = LocalRangeListBMIN, \
                                       BMAJIn = LocalRangeListBMAJ, BPAIn = LocalRangeListBPA, InterferoFlagIn = LocalRangeListInter, \
                                       GlobalvLSRIn = LocalRangeListvLSR, RedShiftIn = LocalRangeListRedShift, FITSSubDirIN = FITSSubDirIN, \
                                       NumObsDataFileIN = NumObsDataFileIN)
    # RangeSelection.show()
    RangeSelection.exec_()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get parameter from GUI
    ok, NewRangeParametersDict, ObsData, ListCoord = RangeSelection.getParam()

    # Debug:
    # print ("\n\n\nok = ", ok)
    # print ("NewRangeParametersDict = ", NewRangeParametersDict)
    # print ("ObsData = ", ObsData)
    # print ("ListCoord = ", ListCoord)


    ## if obs. data file is a fits file save extracted spectrum (spectra) to FITS subdirectory
    if (LocalFITSFlag and ok == 0):
        HeaderString = "List of pixel coordinate indices:  " + str(ListCoord) + "\n"
        HeaderString += "Name of FITS file:  " + chr(34) + ObsDataFileNameIn + chr(34) + "\n"
        NewObsDataFileName = FITSSubDirIN + "FitsSpectrum__" + str(ObsDataFileIDIn + 1) + ".dat"
        numpy.savetxt(NewObsDataFileName, ObsData, header=HeaderString)


    ## define return parameters
    return (ok, NewRangeParametersDict)
##--------------------------------------------------------------------------------------------------------------------------------------------------------

