#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  Create GUI where algorithm parameters can be defined
##
##  Copyright (C) 2017 - 2024  Thomas Moeller
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function AlgorithmGUI.__init__:                   initialize class AlgorithmGUI
##      - function AlgorithmGUI.GetAlgXMLtag:               read tags from the algorithm xml file and return contents in ContentsTag
##      - function AlgorithmGUI.AllTabChanged:              define what happens if tab was changed
##      - function AlgorithmGUI.ButtonWidget:               handle button event of widget
##      - function AlgorithmGUI.CheckBoxWidget:             handle check box event of widget
##      - function AlgorithmGUI.RadioButtonWidget:          handle radio button event of widget
##      - function AlgorithmGUI.ComboWidget:                handle combo box event of widget
##      - function AlgorithmGUI.LineEditWidget:             handle line edit event of widget
##      - function AlgorithmGUI.disconnect:                 disconnect all the stored connection ids
##      - function AlgorithmGUI.replotGUI:                  replot GUI
##      - function AlgorithmGUI.CreateEntries:              create entries for each tab depending on selected algorithm
##      - function AlgorithmGUI.WriteAlgXMLFile:            write algorithm settings to xml file
##      - function AlgorithmGUI.getNewAlgorithmXMLFileName: get path and name of new algorithm xml file
##      - SelectAlgorithm:                                  select algorithm and edit settings
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2017-09-25      initial version
##  T. Moeller      2020-01-08      porting to python 3
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy                                                                                ## import numpy package
import sys                                                                                  ## import sys package
import warnings                                                                             ## import warnings package
from copy import deepcopy                                                                   ## import module deepcopy from package copy
from PyQt5.uic import loadUiType                                                            ## import module loadUiType from PyQt5 package
from PyQt5 import (QtCore, QtGui, QtWidgets)
#from PyQt5.QtGui import (QMessageBox)
from PyQt5.QtWidgets import QMessageBox
from xclass import task_MAGIX
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define routines for mouse handling
##
# class AlgorithmGUI(QtGui.QTabWidget):
class AlgorithmGUI(QtWidgets.QDialog):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize
    def __init__(self, parent = None, GUIJobDirIn = "", UserMaxIterLMIn = 10, AlgorithmXMLFileNameIn = ""):
        """

    input parameters:
    -----------------

        - GUIJobDirIn:              path of job directory

        - UserMaxIterLMIn:          max. number of iterations for LM

        - AlgorithmXMLFileNameIn:   path and name of algorithm xml file


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("parent = ", parent)
        # print ("GUIJobDirIn = ", GUIJobDirIn)
        # print ("UserMaxIterLMIn = ", UserMaxIterLMIn)
        # print ("AlgorithmXMLFileNameIn = ", AlgorithmXMLFileNameIn)


        ## create widget
        super(AlgorithmGUI, self).__init__(parent)


        ## copy input parameters to class variables
        self.AlgorithmXMLFileName = AlgorithmXMLFileNameIn                                  ## get path and name of algorithm xml file
        self.NewAlgXMLFileName = AlgorithmXMLFileNameIn                                     ## the new path and name of the algorithm xml file
        if (UserMaxIterLMIn is None):
            self.MaxIterationsLM = 10
        else:
            self.MaxIterationsLM = UserMaxIterLMIn                                          ## get max. number of iterations
        self.GUIJobDir = GUIJobDirIn                                                        ## get path of working directory

        # Debug:
        # print ("\nself.GUIJobDir = ", self.GUIJobDir)
        # print ("self.MaxIterationsLM = ", self.MaxIterationsLM)
        # print ("self.AlgorithmXMLFileName = ", self.AlgorithmXMLFileName)


        ## define some internal parameters
        self.ok = 0
        self.InitFlag = True
        self.ActiveTab = 0
        self.FirstCallFlag = True
        self.SetListStrings = ["y", "yes", "t", "true"]
        self.KnwonAlgorithms = ["Nested-Sampling", "Levenberg-Marquardt", "Simulated-Annealing", "PSO", "Bees", "Genetic", "MCMC", "INS", \
                                "AdditionalPackages", "ErrorEstim"]
        self.KnwonAlgorithmsLower = [x.lower() for x in self.KnwonAlgorithms]
        self.MethodChi2LocalAlgList = ["default"]
        self.APLocalAlgList = ["fmin", "fmin_powell", "fmin_cg", "fmin_bfgs", "fmin_ncg", "fmin_l_bfgs_b", "fmin_tnc", "anneal", "brute"]
        self.ErrorEstimLocalmethodList = ["MCMC", "Fisher", "INS"]
        self.ChooseImplementationFlag = True
        self.ShowGraphicOptions = True
        self.ShowChi2Methods = True

        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define default values and ranges


        ## general
        self.DefaultAlgorithmName = "Levenberg-Marquardt"
        self.DefaultNumberProcessors = 4
        self.MaxProcessors = 1e9
        self.DefaultMPIHostFileName = ""
        self.DefaultMaxIter = 10
        self.TotalMaxIterations = 1e9
        self.DefaultLimitOfChi2 = 1.e-7
        self.MaxLimitChi2 = 1e99
        self.DefaultRenormalizedChi2Flag = True
        self.DefaultDeterminationChi2 = 0
        self.DefaultSaveChi2 = True
        self.DefaultPlotAxisX = "Rest Frequency [MHz]"
        self.DefaultPlotAxisY = "T_mb [K]"
        self.DefaultPlotIteration = False


        ## LM
        self.DefaultLMVarVal = 1.e-4
        self.MinLMVarVal = 1.e-99
        self.MaxLMVarVal = 1.e99
        self.LMImplementation = "NR"


        ## SA
        self.DefaultSATemp = 1
        self.MaxTempSA = 1.e99
        self.MinTempSA = 1.e-99
        self.DefaultSARedCoef = 0.8
        self.MinSARedCoef = 1.e-15
        self.MaxSARedCoef = 1.0
        self.DefaultSARedCycles = 90
        self.MinSARedCycles = 1
        self.MaxSARedCycles = 1e99
        self.DefaultSAReheat = 1
        self.MinSAReheat = 1
        self.MaxSAReheat = 1e99
        self.SAImplementation = "NR"


        ## Bees
        self.DefaultBeesNum = (-1)
        self.DefaultBeesBS = 5
        self.TotalMinParticles = 1
        self.TotalMaxParticles = 1e99


        ## PSO
        self.DefaultPSOBS = 4


        ## Genetic
        self.DefaultGeneticNum = (-1)
        self.DefaultGeneticBS = 1
        self.DefaultGeneticNewRange = False


        ## NS
        self.DefaultNSNum = 100
        self.DefaultNSBS = 5


        ## INS
        self.DefaultINSNum = 100
        self.DefaultINSBS = 5


        ## MCMC
        self.DefaultMCMCNum = 47
        self.DefaultMCMCBS = 1
        self.DefaultMCMCBI = 10


        ## AP
        self.DefaultAPAlg = "fmin"


        ## ErrorMCMC
        self.DefaultErrorMethod = "MCMC"
        self.DefaultErrorMCMCNum = 47
        self.DefaultErrorMCMCBS = 1
        self.DefaultErrorMCMCBI = 10
        self.DefaultErrorFisherVarVal = 1.e-3


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## extract informations from xml file, if defined
        self.GlobalAlgSettings = []
        self.GlobalActiveAlgSettingsID = []
        if (self.AlgorithmXMLFileName != ""):


            ## extract informations from xml file
            self.NumberOfAlgorithms = int(task_MAGIX.GetXMLtag(self.AlgorithmXMLFileName, "NumberOfFitAlgorithms")[0])
            AlgorithmList = task_MAGIX.GetXMLtag(self.AlgorithmXMLFileName, "FitAlgorithm")
            NumIterationsList = task_MAGIX.GetXMLtag(self.AlgorithmXMLFileName, "number_iterations")
            NumberProcessorsList = task_MAGIX.GetXMLtag(self.AlgorithmXMLFileName, "NumberProcessors")
            LimitOfChi2List = task_MAGIX.GetXMLtag(self.AlgorithmXMLFileName, "limit_of_chi2")


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## make sanity check
            EverythinsIsFineFlag = True
            ErrorTextString = ""
            if (self.NumberOfAlgorithms != len(AlgorithmList)):
                ErrorTextString += "The number of algorithms is unequal to tag <NumberOfFitAlgorithms>\n\n"
            if (self.NumberOfAlgorithms != len(NumIterationsList)):
                ErrorTextString += "The number of itertations is not defined for each algorithm\n\n"
            if (self.NumberOfAlgorithms != len(NumberProcessorsList)):
                ErrorTextString += "The number of processors is not defined for each algorithm\n\n"
            if (self.NumberOfAlgorithms != len(LimitOfChi2List)):
                ErrorTextString += "The lower limit of chi^2 is not defined for each algorithm\n\n"


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## prepare informations in correct order
            if (not EverythinsIsFineFlag):


                ## print warning message and return to main GUI
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText(ErrorTextString)
                ret = msgBox.exec_()


                ## retrun to GUI
                self.ok = 1
                return
            else:
                for LocalAlgorithmID, LocalAlgorithm in enumerate(AlgorithmList):
                    LocalAlgorithmLower = LocalAlgorithm.lower()
                    if (LocalAlgorithmLower == "interval-ns"):
                        LocalAlgorithmLower = "ins"
                    elif (LocalAlgorithmLower == "errorestim_ins"):
                        LocalAlgorithmLower = "errorestim"


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## get special settings
                    LocalSpecialSettings = []


                    ## LM
                    if (LocalAlgorithmLower == "levenberg-marquardt"):
                        LMVarVal = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "VariationValue", LocalAlgorithmLower, LocalAlgorithmID)
                        LMMethod = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "MethodLM", LocalAlgorithmLower, LocalAlgorithmID)
                        LocalSpecialSettings.append(LMVarVal)
                        LocalSpecialSettings.append(LMMethod)


                    ## SA
                    elif (LocalAlgorithmLower == "simulated-annealing"):
                        SATemperature = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "Temperature", LocalAlgorithmLower, LocalAlgorithmID)
                        SATemperatureReductionKoeff = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "TemperatureReductionKoeff", LocalAlgorithmLower, \
                                                                   LocalAlgorithmID)
                        SANumberOfReduction = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "NumberOfReduction", LocalAlgorithmLower, LocalAlgorithmID)
                        SANumberOfReheatingPhases = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "NumberOfReheatingPhases", LocalAlgorithmLower, \
                                                                 LocalAlgorithmID)
                        SASchedule = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "ScheduleSA", LocalAlgorithmLower, LocalAlgorithmID)
                        LocalSpecialSettings.append(SATemperature)
                        LocalSpecialSettings.append(SATemperatureReductionKoeff)
                        LocalSpecialSettings.append(SANumberOfReduction)
                        LocalSpecialSettings.append(SANumberOfReheatingPhases)
                        LocalSpecialSettings.append(SASchedule)


                    ## Bees
                    elif (LocalAlgorithmLower == "bees"):
                        BeesBestSiteCounter = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "BestSiteCounter", LocalAlgorithmLower, LocalAlgorithmID)
                        BeesNumber = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "NumberBees", LocalAlgorithmLower, LocalAlgorithmID)
                        LocalSpecialSettings.append(BeesBestSiteCounter)
                        LocalSpecialSettings.append(BeesNumber)


                    ## PSO
                    elif (LocalAlgorithmLower == "pso"):
                        PSOBestSiteCounter = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "BestSiteCounter", LocalAlgorithmLower, LocalAlgorithmID)
                        LocalSpecialSettings.append(PSOBestSiteCounter)


                    ## Genetic
                    elif (LocalAlgorithmLower == "genetic"):
                        GeneticBestSiteCounter = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "BestSiteCounter", LocalAlgorithmLower, LocalAlgorithmID)
                        try:
                            GeneticBestSiteCounter = GeneticBestSiteCounter.strip()
                        except:
                            pass
                        if (GeneticBestSiteCounter == ""):
                            GeneticBestSiteCounter = 1
                        LocalSpecialSettings.append(GeneticBestSiteCounter)

                        GeneticNumber = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "NumberChromosomes", LocalAlgorithmLower, LocalAlgorithmID)
                        GeneticUseNewRange = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "UseNewRange", LocalAlgorithmLower, LocalAlgorithmID).lower()
                        LocalSpecialSettings.append(GeneticNumber)

                        if (GeneticUseNewRange in self.SetListStrings):
                            GeneticUseNewRange = True
                        else:
                            GeneticUseNewRange = False
                        LocalSpecialSettings.append(GeneticUseNewRange)

                    ## NS
                    elif (LocalAlgorithmLower == "nested-sampling"):
                        NSBestSiteCounter = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "BestSiteCounter", LocalAlgorithmLower, LocalAlgorithmID)
                        NSNumber = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "NumberObjects", LocalAlgorithmLower, LocalAlgorithmID)
                        LocalSpecialSettings.append(NSBestSiteCounter)
                        LocalSpecialSettings.append(NSNumber)


                    ## INS
                    elif (LocalAlgorithmLower == "interval-ns" or LocalAlgorithmLower == "ins"):
                        INSBestSiteCounter = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "BestSiteCounter", LocalAlgorithmLower, LocalAlgorithmID)
                        INSvol_bound = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "vol_bound", LocalAlgorithmLower, LocalAlgorithmID)
                        INSdelta_incl = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "delta_incl", LocalAlgorithmLower, LocalAlgorithmID)
                        LocalSpecialSettings.append(INSBestSiteCounter)
                        LocalSpecialSettings.append(INSvol_bound)
                        LocalSpecialSettings.append(INSdelta_incl)


                    ## MCMC
                    elif (LocalAlgorithmLower == "mcmc"):
                        MCMCBestSiteCounter = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "BestSiteCounter", LocalAlgorithmLower, LocalAlgorithmID)
                        MCMCNumber = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "NumberMCMCSampler", LocalAlgorithmLower, LocalAlgorithmID)
                        MCMCNumberBurnInIter = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "NumberBurnInIter", LocalAlgorithmLower, LocalAlgorithmID)
                        LocalSpecialSettings.append(MCMCBestSiteCounter)
                        LocalSpecialSettings.append(MCMCNumber)
                        LocalSpecialSettings.append(MCMCNumberBurnInIter)


                    ## AP
                    elif (LocalAlgorithmLower == "additionalpackages"):
                        APAlgorithm = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "minAlgorithm", LocalAlgorithmLower, LocalAlgorithmID)
                        LocalSpecialSettings.append(APAlgorithm)


                    ## Error Estimation
                    elif (LocalAlgorithmLower == "errorestim" or LocalAlgorithmLower == "errorestim_ins"):


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## get error method
                        ErrorEstimMethod = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "ErrorMethod", LocalAlgorithmLower, LocalAlgorithmID)


                        ## MCMC
                        if (ErrorEstimMethod.lower() == "mcmc"):
                            LocalSpecialSettings.append(ErrorEstimMethod)


                            ## get number of MCMC sampler
                            ErrorEstimNumberMCMCSampler = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "NumberMCMCSampler", LocalAlgorithmLower, \
                                                                            LocalAlgorithmID)
                            LocalSpecialSettings.append(ErrorEstimNumberMCMCSampler)


                            ## get number of MCMC burn in iterations
                            ErrorEstimNumberBurnInIter = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "NumberBurnInIter", LocalAlgorithmLower, \
                                                                            LocalAlgorithmID)
                            LocalSpecialSettings.append(ErrorEstimNumberBurnInIter)


                            ## get number of MCMC sigma multiplyier
                            ErrorEstimMultiplicitySigma = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "MultiplicitySigma", LocalAlgorithmLower, \
                                                                            LocalAlgorithmID)
                            LocalSpecialSettings.append(ErrorEstimMultiplicitySigma)


                            ## get number of MCMC use previous results flag
                            ErrorEstimUsePrevResults = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "UsePrevResults", LocalAlgorithmLower, \
                                                                         LocalAlgorithmID)
                            LocalSpecialSettings.append(ErrorEstimUsePrevResults)


                        ## Fisher
                        elif (ErrorEstimMethod.lower() == "fisher"):
                            LocalSpecialSettings.append(ErrorEstimMethod)


                            ## get number of MCMC sampler
                            ErrorEstimVariationValue = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "VariationValue", LocalAlgorithmLower, \
                                                                         LocalAlgorithmID)
                            LocalSpecialSettings.append(ErrorEstimVariationValue)


                        ## INS
                        else:
                            ErrorEstimMethod = "INS"
                            LocalSpecialSettings.append(ErrorEstimMethod)

                        # Debug:
                        # print ("LocalSpecialSettings = ", LocalSpecialSettings)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## get general settings


                    ## get max. number of iterations
                    try:
                        LocalNumIterations = int(NumIterationsList[LocalAlgorithmID])
                    except:
                        LocalNumIterations = 1


                    ## get max. number of processors
                    try:
                        LocalNumberProcessors = int(NumberProcessorsList[LocalAlgorithmID])
                    except:
                        LocalNumberProcessors = 4


                    ## get path and name of host file
                    LocalMPIHostFileName = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "MPIHostFileName", LocalAlgorithmLower, LocalAlgorithmID)


                    ## get lower limit of chi^2
                    try:
                        LocalLimitOfChi2 = float(LimitOfChi2List[LocalAlgorithmID])
                    except:
                        LocalLimitOfChi2 = 1.e-9


                    ## get flag for using renormalized chi^2
                    LocalRenormalizedChi2 = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "RenormalizedChi2", LocalAlgorithmLower, LocalAlgorithmID)
                    LocalRenormalizedChi2 = LocalRenormalizedChi2.lower()
                    if (LocalRenormalizedChi2 in self.SetListStrings):
                        LocalRenormalizedChi2 = True
                    else:
                        LocalRenormalizedChi2 = False


                    ## get flag for using renormalized chi^2
                    LocalDeterminationChi2 = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "DeterminationChi2", LocalAlgorithmLower, LocalAlgorithmID)
                    LocalDeterminationChi2 = LocalDeterminationChi2.lower()
                    if (LocalDeterminationChi2 == "default"):
                        LocalDeterminationChi2 = 0
                    else:
                        LocalDeterminationChi2 = 1


                    ## get flag for saving chi^2
                    LocalSaveChi2 = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "SaveChi2", LocalAlgorithmLower, LocalAlgorithmID).lower()
                    if (LocalSaveChi2 in self.SetListStrings):
                        LocalSaveChi2 = True
                    else:
                        LocalSaveChi2 = False


                    ## get max. number of iterations
                    LocalPlotAxisX = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "PlotAxisX", LocalAlgorithmLower, LocalAlgorithmID)
                    LocalPlotAxisY = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "PlotAxisY", LocalAlgorithmLower, LocalAlgorithmID)


                    ## get flag for creating plot for each iteration
                    LocalPlotIterationFlag = self.GetAlgXMLtag(self.AlgorithmXMLFileName, "PlotIteration", LocalAlgorithmLower, LocalAlgorithmID).lower()
                    if (LocalPlotIterationFlag in self.SetListStrings):
                        LocalPlotIterationFlag = True
                    else:
                        LocalPlotIterationFlag = False


                    ## define list of settings
                    LocalAlgSettings = [[LocalAlgorithmLower, LocalSpecialSettings, LocalNumIterations, LocalNumberProcessors, LocalMPIHostFileName, \
                                        LocalLimitOfChi2, LocalRenormalizedChi2, LocalDeterminationChi2, LocalSaveChi2, LocalPlotAxisX, LocalPlotAxisY, \
                                        LocalPlotIterationFlag]]
                    self.GlobalAlgSettings.append(LocalAlgSettings)
                    self.GlobalActiveAlgSettingsID.append(0)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## if no xml file is defined, use default values
        else:
            SpecialSettings = [self.DefaultLMVarVal]                                        ## define a default variation value
            LocalAlgSettings = [[self.DefaultAlgorithmName, SpecialSettings, self.MaxIterationsLM, self.DefaultNumberProcessors, \
                                self.DefaultMPIHostFileName, self.DefaultLimitOfChi2, self.DefaultRenormalizedChi2Flag, self.DefaultDeterminationChi2, \
                                self.DefaultSaveChi2, self.DefaultPlotAxisX, self.DefaultPlotAxisY, self.DefaultPlotIteration]]
            self.GlobalAlgSettings.append(LocalAlgSettings)
            self.GlobalActiveAlgSettingsID.append(0)
        self.NumberOfAlgorithms = len(self.GlobalAlgSettings)

        # Debug:
        # print ("\n\nself.self.NumberOfAlgorithms = ", self.self.NumberOfAlgorithms)
        # print ("\n\nself.GlobalAlgSettings = ", self.GlobalAlgSettings)


        ## plot GUI
        self.replotGUI()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read tags from the algorithm xml file and return contents in ContentsTag
    def GetAlgXMLtag(self, xmlFileName, tagName, UsedAlgorithm, AlgID):
        """

    input parameters:
    -----------------

        - xmlFileName:          path and name of the xml-file

        - tagName:              name of the tag, whose contents is read

        - UsedAlgorithm:        current algorithm

        - AlgID:                algorithm counter


    output parameters:
    ------------------

        - ContentsTag:          the contents of the selected tag, always a list, including an entry for each occurance
        """

        # Debug:
        # print ("\n\n\nxmlFileName = ", xmlFileName)
        # print ("tagName = ", tagName)
        # print ("UsedAlgorithm = ", UsedAlgorithm)
        # print ("AlgID = ", AlgID)


        ## initialize return parameter
        ContentsTag = ""


        ## read in whole xml-file
        xmlFile = open(xmlFileName)
        ContentsXMLFile = xmlFile.readlines()
        xmlFile.close()


        ## analyze contents of xml file
        tagName = tagName.strip()
        CurrentDataFileID = 0
        LocalAlg = ""
        LocalAlgID = (-1)
        for line in ContentsXMLFile:                                                        ## loop over all lines in the xml file
            striplines = line.strip()
            if (not striplines.startswith("<!--")):                                         ## check if current line is not a comment


                ## special handling for some tags
                if (striplines.find("<FitAlgorithm>") > (-1)):
                    i = striplines.find(">")
                    j = striplines.rfind("<")
                    if (i < j):
                        LocalAlg = striplines[i + 1:j].lower().strip()                      ## store current algorithm
                        if (LocalAlg == "interval-ns" or LocalAlg == "ins"):
                            LocalAlg = "ins"
                        elif (LocalAlg == "errorestim" or LocalAlg == "errorestim_ins"):
                            LocalAlg = "errorestim"
                        LocalAlgID += 1


                ## get contents of defined tag
                elif (striplines.find("<" + tagName + ">") > (-1)):                         ## does the current line described selected tag?
                    i = striplines.find(">")
                    j = striplines.rfind("<")
                    if (i < j and LocalAlg == UsedAlgorithm and LocalAlgID == AlgID):
                        ContentsTag = striplines[i + 1:j]
                        break

        # Debug:
        # print ("\nContentsTag = ", ContentsTag)


        ## define return value
        # ContentsTag = ContentsTag.lower()
        return ContentsTag
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what happens if tab was changed
    def AllTabChanged(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None

        """


        ## set active tab
        self.ActiveTab = self.TabGUI.currentIndex()

        # Debug:
        # print ("self.ActiveTab = ", self.ActiveTab)


        ## update GUI
        if (not self.InitFlag):
            self.replotGUI()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle button event of widget
    def ButtonWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None

        """

        ## is element locked
        if (self.lockElements):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()
        name = sender.text()

        # Debug:
        # print ("objectName = ", objectName)
        # print ("name = ", name)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## identify calling button


        # fit continuum
        if (objectName == "ButtonCancel"):
            self.ok = 1
            self.disconnect()


        # quit plot
        elif (objectName == "ButtonOK"):


            ## write settings to algorithm xml file
            self.WriteAlgXMLFile()


            ## close gui
            self.disconnect()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add algorithm after current algorithm
        elif (objectName == "buttonAddAlg"):
            LocalAlgID = self.TabGUI.currentIndex()


            ## insert default algorithm at current position
            SpecialSettings = [1.e-4, "NR"]                                                 ## define a default variation value
            LocalAlgSettings = [[self.DefaultAlgorithmName, SpecialSettings, self.DefaultMaxIter, self.DefaultNumberProcessors, \
                                self.DefaultMPIHostFileName, self.DefaultLimitOfChi2, self.DefaultRenormalizedChi2Flag, self.DefaultDeterminationChi2, \
                                self.DefaultSaveChi2, self.DefaultPlotAxisX, self.DefaultPlotAxisY, self.DefaultPlotIteration]]
            self.GlobalAlgSettings.insert(self.ActiveTab, LocalAlgSettings)
            self.GlobalActiveAlgSettingsID.insert(self.ActiveTab, 0)
            self.NumberOfAlgorithms = len(self.GlobalAlgSettings)


            ## update GUI
            self.InitFlag = True
            self.replotGUI()
            self.TabGUI.setCurrentIndex(self.ActiveTab)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## remove current range on right double click
        elif (objectName == "buttonRemoveAlg" and len(self.GlobalAlgSettings) > 1):
            LocalAlgID = self.TabGUI.currentIndex()
            del self.GlobalAlgSettings[LocalAlgID]
            del self.GlobalActiveAlgSettingsID[LocalAlgID]
            self.NumberOfAlgorithms = len(self.GlobalAlgSettings)


            ## update GUI
            self.InitFlag = True
            self.replotGUI()


            ## switch to next tab
            if (LocalAlgID > len(self.GlobalAlgSettings)):
                self.ActiveTab = (LocalAlgID - 1)
                self.TabGUI.setCurrentIndex(self.ActiveTab)


        ## return to main GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle check box event of widget
    def CheckBoxWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None

        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is element locked
        if (self.lockElements):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name, value and flag
        sender = self.sender()
        flag = sender.isChecked()
        objectName = sender.objectName()

        # Debug:
        # print ("flag = ", flag)


        ## get settings for current algorithm
        ## LocalAlgSettings= [LocalAlg.lower(), LocalSpecialSettings, self.DefaultMaxIter, self.DefaultNumberProcessors, \
        ##                    self.DefaultMPIHostFileName, self.DefaultLimitOfChi2, self.DefaultRenormalizedChi2Flag, \
        ##                    self.DefaultDeterminationChi2, self.DefaultSaveChi2, self.DefaultPlotAxisX, self.DefaultPlotAxisY, \
        ##                    self.DefaultPlotIteration]
        LocalAlgSettings = self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]]


        ## set corresponding flag
        if (objectName == "NewRangeFlag"):
            self.GeneticNewRangeFlag = flag
            LocalAlgSettings[1][2] = self.GeneticNewRangeFlag


        ## set renormalized chi^2 flag
        elif (objectName == "RenormalizedChi2"):
            self.LocalRenormalizedChi2Flag = flag
            LocalAlgSettings[6] = self.LocalRenormalizedChi2Flag


        ## set save chi^2 flag
        elif (objectName == "SaveChi2"):
            self.LocalSaveChi2Flag = flag
            LocalAlgSettings[8] = self.LocalSaveChi2Flag


        ## set plot on iteration flag
        elif (objectName == "PlotOnIteration"):
            self.LocalPlotOnIterFlag = flag
            LocalAlgSettings[11] = self.LocalPlotOnIterFlag

        # Debug:
        # print ("LocalAlgSettings = ", LocalAlgSettings)


        ## save changes
        self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


        ## update GUI
        if (not self.InitFlag):
            self.replotGUI()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle radio button event of widget
    def RadioButtonWidget(self, RadioElement):
        """

    input parameters:
    -----------------

        - RadioElement:         radio button element


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is element locked
        if (self.lockElements):
            return

        # Debug:
        # print ("\n\nRadioElement.text() = ", RadioElement.text())
        # print ("RadioElement.isChecked() = ", RadioElement.isChecked())
        # print ("self.ActiveAlg = ", self.ActiveAlg)


        ## get settings for current algorithm
        LocalAlgSettings = self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]]


        ## set NR option
        if (RadioElement.text() == "NR"):
            if (RadioElement.isChecked() == True):
                if (self.ActiveAlg == "levenberg-marquardt"):
                    self.LocalLMMethod = "nr"
                    LocalAlgSettings[1][1] = self.LocalLMMethod
                else:
                    self.LocalSAMethod = "nr"
                    LocalAlgSettings[1][1] = self.LocalSAMethod
            else:
                if (self.ActiveAlg == "levenberg-marquardt"):
                    self.LocalLMMethod = "minpack"
                    LocalAlgSettings[1][1] = self.LocalLMMethod
                else:
                    self.LocalSAMethod = "scipy"
                    LocalAlgSettings[1][1] = self.LocalSAMethod


        ## set MINPACK option
        elif (RadioElement.text() == "MINPACK"):
            if (RadioElement.isChecked() == True):
                self.LocalLMMethod = "minpack"
            else:
                self.LocalLMMethod = "nr"
            LocalAlgSettings[1][1] = self.LocalLMMethod


        ## set SCIPY option
        elif (RadioElement.text() == "SCIPY"):
            if (RadioElement.isChecked() == True):
                self.LocalSAMethod = "scipy"
            else:
                self.LocalSAMethod = "nr"
            LocalAlgSettings[1][1] = self.LocalSAMethod

        # Debug:
        # print ("LocalAlgSettings = ", LocalAlgSettings)


        ## save changes
        self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


        ## update GUI
        if (not self.InitFlag):
            self.replotGUI()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle combo box event of widget
    def ComboWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is element locked
        if (self.lockElements):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()

        # Debug:
        # print ("objectName = ", objectName)


        ## get local algorithm of AP package
        if (objectName == "ComboCurrAlgorithm"):
            LocalComboIndex = max(0, self.ComboCurrAlgorithm[self.ActiveTab].currentIndex())
            LocalAlg = self.KnwonAlgorithms[LocalComboIndex].lower()


            ## check, if selected algorithm was used before
            FoundAlgFlag = False
            for LocalAlgParamID, LocalAlgParam in enumerate(self.GlobalAlgSettings[self.ActiveTab]):
                InterAlg = LocalAlgParam[0]
                if (InterAlg == LocalAlg):
                    self.GlobalActiveAlgSettingsID[self.ActiveTab] = LocalAlgParamID
                    FoundAlgFlag = True
                    break


            ## if selected algorithm is not already used, create new entry
            if (not FoundAlgFlag):
                LocalAlgSettings = self.GlobalAlgSettings[self.ActiveTab]


                ## LM
                if (LocalAlg == "levenberg-marquardt"):
                    LocalSpecialSettings = [self.DefaultLMVarVal, self.LMImplementation]


                ## SA
                elif (LocalAlg == "simulated-annealing"):
                    LocalSpecialSettings = [self.DefaultSATemp, self.DefaultSARedCoef, self.DefaultSARedCycles, self.DefaultSAReheat, \
                                            self.SAImplementation]


                ## Bees
                elif (LocalAlg == "bees"):
                    LocalSpecialSettings = [self.DefaultBeesBS, self.DefaultBeesNum]


                ## PSO
                elif (LocalAlg == "pso"):
                    LocalSpecialSettings = [self.DefaultPSOBS]


                ## Genetic
                elif (LocalAlg == "genetic"):
                    LocalSpecialSettings = [self.DefaultGeneticBS, self.DefaultGeneticNum, self.DefaultGeneticNewRange]



                ## NS
                elif (LocalAlg == "nested-sampling"):
                    LocalSpecialSettings = [self.DefaultNSBS, self.DefaultNSNum]


                ## INS
                elif (LocalAlg == "ins"):
                    LocalSpecialSettings = [self.DefaultINSBS, self.DefaultINSNum]



                ## MCMC
                elif (LocalAlg == "mcmc"):
                    LocalSpecialSettings = [self.DefaultMCMCBS, self.DefaultMCMCNum, self.DefaultMCMCBI]



                ## AP
                elif (LocalAlg == "additionalpackages"):
                    LocalSpecialSettings = [self.DefaultAPAlg]


                ## Error
                elif (LocalAlg == "errorestim"):
                    LocalSpecialSettings = [self.DefaultErrorMethod, self.DefaultErrorMCMCNum, self.DefaultErrorMCMCBS, self.DefaultErrorMCMCBI, \
                                            self.DefaultErrorFisherVarVal]


                ## create new entry
                LocalAlgSettings.append([LocalAlg.lower(), LocalSpecialSettings, self.DefaultMaxIter, self.DefaultNumberProcessors, \
                                         self.DefaultMPIHostFileName, self.DefaultLimitOfChi2, self.DefaultRenormalizedChi2Flag, \
                                         self.DefaultDeterminationChi2, self.DefaultSaveChi2, self.DefaultPlotAxisX, self.DefaultPlotAxisY, \
                                         self.DefaultPlotIteration])
                self.GlobalAlgSettings[self.ActiveTab] = LocalAlgSettings
                i = len(LocalAlgSettings) - 1
                self.GlobalActiveAlgSettingsID[self.ActiveTab] = i

                # Debug:
                # print (">>>>>>>>>LocalAlgSettings = ", LocalAlgSettings)


            ## update GUI
            self.InitFlag = True
            self.replotGUI()


        ## get local algorithm of AP package
        if (objectName == "ComboAPAlg"):
            LocalComboIndex = max(0, self.APComboAlgSelect[self.ActiveTab].currentIndex())
            self.LocalAPAlgID = LocalComboIndex
            LocalAlgSettings = self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]]
            LocalAlgSettings[1][0] = self.APLocalAlgList[LocalComboIndex]
            self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


            ## update GUI
            self.replotGUI()


        ## get method for error estimation
        elif (objectName == "ComboErrorEstimMethod"):
            LocalComboIndex = max(0, (self.ComboErrorMethod[self.ActiveTab]).currentIndex())
            self.LocalErrorEstimMethodID = LocalComboIndex
            LocalAlgSettings = self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]]
            LocalErrorMethod = self.ErrorEstimLocalmethodList[LocalComboIndex]
            if (LocalErrorMethod.lower() == "mcmc" and len(LocalAlgSettings[1]) != 4):
                LocalAlgSettings[1] = [LocalErrorMethod, self.DefaultErrorMCMCNum, self.DefaultErrorMCMCBI, self.DefaultErrorMCMCBS]
            elif (LocalErrorMethod.lower() == "fisher" and len(LocalAlgSettings[1]) != 2):
                LocalAlgSettings[1] = [LocalErrorMethod, self.DefaultErrorFisherVarVal]
            else:
                LocalAlgSettings[1][0] = LocalErrorMethod
            self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


            ## update GUI
            self.InitFlag = True
            self.replotGUI()
            self.TabGUI.setCurrentIndex(self.ActiveTab)


        ## get method for chi^2 determination
        elif (objectName == "ComboChi2Det"):
            LocalComboIndex = max(0, self.GeneralMethodChi2[self.ActiveTab].currentIndex())
            self.LocalDeterminationChi2 = LocalComboIndex
            LocalAlgSettings = self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]]
            LocalAlgSettings[1][0] = self.MethodChi2LocalAlgList[LocalComboIndex]
            self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


            ## update GUI
            self.replotGUI()

        # Debug:
        # print ("\nself.ActiveTab = ", self.ActiveTab)
        # print ("self.GlobalActiveAlgSettingsID[self.ActiveTab] = ", self.GlobalActiveAlgSettingsID[self.ActiveTab])
        # print ("self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = ", \
        #           self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]])


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle line edit event of widget
    def LineEditWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is element locked
        if (self.lockElements):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()
        val = str(sender.text())
        self.CurrCursorPos = sender.cursorPosition()

        # Debug:
        # print ("objectName = ", objectName)
        # print ("val = ", val)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if value is number which is greater or equal zero
        IsNumber = True
        try:
            NumVal = int(val)
        except ValueError:
            try:
                NumVal = float(val)
            except ValueError:
                IsNumber = False

        # Debug:
        # print ("IsNumber = ", IsNumber)


        ## get parameter of current algorithm
        ## LocalAlgSettings= [LocalAlg.lower(), LocalSpecialSettings, self.DefaultMaxIter, self.DefaultNumberProcessors, \
        ##                    self.DefaultMPIHostFileName, self.DefaultLimitOfChi2, self.DefaultRenormalizedChi2Flag, \
        ##                    self.DefaultDeterminationChi2, self.DefaultSaveChi2, self.DefaultPlotAxisX, self.DefaultPlotAxisY, \
        ##                    self.DefaultPlotIteration]
        LocalAlgSettings = self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze user input


        ## LM


        if (objectName == "LMVarVal"):
            if (IsNumber):
                self.LocalLMVarValue = NumVal
                LocalAlgSettings[1][0] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## SA


        ## analyze user input
        elif (objectName == "TempSA"):
            if (IsNumber):
                self.LocalSATemp = NumVal
                LocalAlgSettings[1][0] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "TempRedCoefSA"):
            if (IsNumber):
                self.LocalSATempRedCoef = NumVal
                LocalAlgSettings[1][1] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "SARedCycle"):
            if (IsNumber):
                self.LocalSANumRed = NumVal
                LocalAlgSettings[1][2] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "SAReheatPhases"):
            if (IsNumber):
                self.LocalSANumReHeat = NumVal
                LocalAlgSettings[1][3] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## Bees


        ## analyze user input
        elif (objectName == "BeesNumBestSites"):
            if (IsNumber):
                self.LocalBeesNumBestSites = NumVal
                LocalAlgSettings[1][0] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "BeesNumParticles"):
            if (IsNumber):
                self.LocalBeesNumParticles = NumVal
                LocalAlgSettings[1][1] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## PSO


        ## analyze user input
        elif (objectName == "PSONumBestSites"):
            if (IsNumber):
                self.LocalPSONumBestSites = NumVal
                LocalAlgSettings[1][0] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## Genetic


        ## analyze user input
        elif (objectName == "GeneticNumBestSites"):
            if (IsNumber):
                self.LocalGeneticNumBestSites = NumVal
                LocalAlgSettings[1][0] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "GeneticNumParticles"):
            if (IsNumber):
                self.LocalGeneticNumParticles = NumVal
                LocalAlgSettings[1][1] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## NS


        ## analyze user input
        elif (objectName == "NSNumBestSites"):
            if (IsNumber):
                self.LocalNSNumBestSites = NumVal
                LocalAlgSettings[1][0] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "NSNumParticles"):
            if (IsNumber):
                self.LocalNSNumParticles = NumVal
                LocalAlgSettings[1][1] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## MCMC


        ## analyze user input
        elif (objectName == "MCMCNumBestSites"):
            if (IsNumber):
                self.LocalMCMCNumBestSites = NumVal
                LocalAlgSettings[1][0] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "MCMCNumParticles"):
            if (IsNumber):
                self.LocalMCMCNumParticles = NumVal
                LocalAlgSettings[1][1] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "MCMCNumBurnInIter"):
            if (IsNumber):
                self.LocalMCMCBurnInIter = NumVal
                LocalAlgSettings[1][2] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## INS


        ## analyze user input
        elif (objectName == "INSNumBestSites"):
            if (IsNumber):
                self.LocalINSNumBestSites = NumVal
                LocalAlgSettings[1][0] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "INSvolBound"):
            if (IsNumber):
                self.LocalINSVolBond = NumVal
                LocalAlgSettings[1][1] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "INSDeltaIncl"):
            if (IsNumber):
                self.LocalINSDeltaIncl = NumVal
                LocalAlgSettings[1][2] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## Error Estimation


        ## analyze user input
        elif (objectName == "ErrorNumParticles"):
            if (IsNumber):
                self.LocalErrorNumParticles = NumVal
                LocalAlgSettings[1][1] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "ErrorNumBurnInIter"):
            if (IsNumber):
                self.LocalErrorMCMCBurnInIter = NumVal
                LocalAlgSettings[1][2] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "ErrorLMVarVal"):
            if (IsNumber):
                self.LocalErrorLMVarValue = NumVal
                LocalAlgSettings[1][1] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## general settings


        ## analyze user input
        elif (objectName == "MaxIter"):
            if (IsNumber):
                self.LocalMaxIter = NumVal
                LocalAlgSettings[2] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "NumberProc"):
            if (IsNumber):
                self.LoaclNumProc = NumVal
                LocalAlgSettings[3] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "EnterHostFile"):
            self.LocalHostFileName = val
            LocalAlgSettings[4] = val
            self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


            ## replot spectrum
            self.replotGUI()


        ## analyze user input
        elif (objectName == "Limitchi2"):
            if (IsNumber):
                self.LocalLimitChi2 = NumVal
                LocalAlgSettings[5] = NumVal
                self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


                ## replot spectrum
                self.replotGUI()


            else:
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please enter a number!")
                ret = msgBox.exec_()


        ## analyze user input
        elif (objectName == "EnterXAxis"):
            self.LocalTitleXAxis = val
            LocalAlgSettings[9] = val
            self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


            ## replot spectrum
            self.replotGUI()



        ## analyze user input
        elif (objectName == "EnterYAxis"):
            self.LocalTitleYAxis = val
            LocalAlgSettings[10] = val
            self.GlobalAlgSettings[self.ActiveTab][self.GlobalActiveAlgSettingsID[self.ActiveTab]] = LocalAlgSettings


            ## replot spectrum
            self.replotGUI()

        # Debug:
        # print ("\nself.ActiveTab = ", self.ActiveTab)
        # print ("self.GlobalActiveAlgSettingsID[self.ActiveTab] = ", self.GlobalActiveAlgSettingsID[self.ActiveTab])
        # print ("self.GlobalAlgSettings[self.ActiveTab] = ", self.GlobalAlgSettings[self.ActiveTab])


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## disconnect all the stored connection ids and close gui
    def disconnect(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## disconnect events
        self.close()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## replot window
    def replotGUI(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("\n\nself.InitFlag = ", self.InitFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## lock element
        self.lockElements = True


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define elements for each tab
        if (self.InitFlag):


            ## initialize elements
            self.LineEditGeneralMaxIter = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditGeneralNumProc = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditGeneralHostFile = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditGeneralLimitChi2 = [x for x in range(self.NumberOfAlgorithms)]
            self.GeneralFlagRenormalizedChi2 = [x for x in range(self.NumberOfAlgorithms)]
            if (self.ShowChi2Methods):
                self.GeneralMethodChi2 = [x for x in range(self.NumberOfAlgorithms)]
            self.GeneralFlagSaveChi2 = [x for x in range(self.NumberOfAlgorithms)]
            if (self.ShowGraphicOptions):
                self.LineEditGeneralXAxis = [x for x in range(self.NumberOfAlgorithms)]
                self.LineEditGeneralYAxis = [x for x in range(self.NumberOfAlgorithms)]
                self.GeneralPlotOnIteration = [x for x in range(self.NumberOfAlgorithms)]
            self.ComboCurrAlgorithm = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditLMVarVal = [x for x in range(self.NumberOfAlgorithms)]
            self.QRadioButtonLMMethodNR = [x for x in range(self.NumberOfAlgorithms)]
            self.QRadioButtonLMMethodMIN = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditSATemp = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditSATempRedCoef = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditSARedCycle = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditSAReheatings = [x for x in range(self.NumberOfAlgorithms)]
            self.QRadioButtonSAMethodNR = [x for x in range(self.NumberOfAlgorithms)]
            self.QRadioButtonSAMethodMIN = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditBeesBestSites = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditBeesNumPart = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditPSONumBestSites = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditGeneticNumBestSites = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditGeneticNumPart = [x for x in range(self.NumberOfAlgorithms)]
            self.FlagNewRangeGenetic = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditNSNumBestSites = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditNSNumPart = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditMCMCNumBestSites = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditMCMCNumPart = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditMCMCNumBurnIInter = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditINSNumBestSites = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditINSvolBond = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditINSDeltaIncl = [x for x in range(self.NumberOfAlgorithms)]
            self.APComboAlgSelect = [x for x in range(self.NumberOfAlgorithms)]
            self.ComboErrorMethod = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditErrorMCMCNumPart = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditErrorMCMCNumBurnIInter = [x for x in range(self.NumberOfAlgorithms)]
            self.LineEditErrorLMVarVal = [x for x in range(self.NumberOfAlgorithms)]


            ## initialize tab widget
            OldActiveTab = self.ActiveTab
            try:
                for i in range(self.mainLayout.count()): self.mainLayout.itemAt(i).widget().close()
            except:
                pass
            self.TabGUI = QtWidgets.QTabWidget()
            self.TabGUI.setFixedWidth(500)
            self.TabGUI.currentChanged.connect(self.AllTabChanged)
            self.ActiveTab = OldActiveTab
            self.TabGUI.setCurrentIndex(self.ActiveTab)


            ## add elements for each element
            for LocalAlgSettingsID, InterAlgSettings in enumerate(self.GlobalAlgSettings):
                LocalAlgSettings = InterAlgSettings[self.GlobalActiveAlgSettingsID[LocalAlgSettingsID]]


                ## add title to GUI
                if (LocalAlgSettingsID == 0):
                    self.setWindowTitle("Edit Algorithms")


                ## get parameters for current algorithm
                LocalAlg = LocalAlgSettings[0].lower()                                      ## get name of current algorithm
                LocalSpecialSettings = LocalAlgSettings[1]                                  ## get special settings for current algorithm
                self.LocalMaxIter = LocalAlgSettings[2]                                     ## get max. number of iterations
                self.LoaclNumProc = LocalAlgSettings[3]                                     ## get number of cores
                self.LocalHostFileName = LocalAlgSettings[4]                                ## get path and name of host file
                self.LocalLimitChi2 = LocalAlgSettings[5]                                   ## get lower limit of chi^2
                self.LocalDeterminationChi2 = LocalAlgSettings[6]                           ## get method for chi^2 determination
                self.LocalRenormalizedChi2Flag = LocalAlgSettings[7]                        ## get flag for usage of renormalized chi^2
                self.LocalSaveChi2Flag = LocalAlgSettings[8]                                ## get flag for saving chi^2 values
                self.LocalPlotOnIterFlag = LocalAlgSettings[9]                              ## get flag for create plot for each iteration
                self.LocalTitleXAxis = LocalAlgSettings[10]                                 ## get title for x axis
                self.LocalTitleYAxis = LocalAlgSettings[11]                                 ## get title for y axis


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## create tab
                try:
                    for i in range(self.LocalTabLayer.count()): self.LocalTabLayer.itemAt(i).widget().close()
                except:
                    pass
                self.LocalTabLayer = QtWidgets.QWidget()
                layout = QtWidgets.QGridLayout()


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## add buttons to add and remove algorithms


                ## add button "Remove Range"
                self.buttonAddAlg = QtWidgets.QPushButton('Add algorithm')
                self.buttonAddAlg.setShortcut("+")
                self.buttonAddAlg.setToolTip('Add algorithm')
                self.buttonAddAlg.setObjectName("buttonAddAlg")
                self.buttonAddAlg.setFixedWidth(200)
                self.buttonAddAlg.clicked.connect(self.ButtonWidget)
                layout.addWidget(self.buttonAddAlg, 0, 0)


                ## add button "Remove Range"
                self.buttonRemoveAlg = QtWidgets.QPushButton('Remove algorithm')
                self.buttonRemoveAlg.setShortcut("DEL")
                self.buttonRemoveAlg.setToolTip('Remove current current')
                self.buttonRemoveAlg.setObjectName("buttonRemoveAlg")
                self.buttonRemoveAlg.setFixedWidth(200)
                self.buttonRemoveAlg.clicked.connect(self.ButtonWidget)
                if (len(self.GlobalAlgSettings) == 1):
                    self.buttonRemoveAlg.setEnabled(False)
                else:
                    self.buttonRemoveAlg.setEnabled(True)
                layout.addWidget(self.buttonRemoveAlg, 0, 1)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## add elements for current algorithm
                NumberElements = self.CreateEntries(layout, LocalAlgSettingsID, LocalAlg, LocalSpecialSettings)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## add layer to tab
                self.LocalTabLayer.setLayout(layout)
                self.TabGUI.addTab(self.LocalTabLayer, "Algorithm " + str(LocalAlgSettingsID + 1))


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## define underlying layout for tabs and standard buttons
            if (self.FirstCallFlag):
                self.mainLayout = QtWidgets.QGridLayout()
            self.mainLayout.addWidget(self.TabGUI, 0, 0, 1, 2)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## add standard buttons


            ## add button "Cancel"
            self.buttonCancel = QtWidgets.QPushButton('Cancel')
            self.buttonCancel.setShortcut("ESC")
            self.buttonCancel.setToolTip('Cancel')
            self.buttonCancel.setObjectName("ButtonCancel")
            self.buttonCancel.clicked.connect(self.ButtonWidget)
            self.mainLayout.addWidget(self.buttonCancel, 1, 0)


            ## add button "OK"
            self.buttonOK = QtWidgets.QPushButton('OK')
            self.buttonOK.setShortcut("q")
            self.buttonOK.setToolTip('Quit GUI.')
            self.buttonOK.setObjectName("ButtonOK")
            self.buttonOK.clicked.connect(self.ButtonWidget)
            self.mainLayout.addWidget(self.buttonOK, 1, 1)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## set main layout
            self.setLayout(self.mainLayout)
            self.InitFlag = False
            self.FirstCallFlag = False
            self.ActiveTab = OldActiveTab
            self.TabGUI.setCurrentIndex(self.ActiveTab)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## fill values to elements
        for LocalAlgSettingsID, InterAlgSettings in enumerate(self.GlobalAlgSettings):
            LocalAlgSettings = InterAlgSettings[self.GlobalActiveAlgSettingsID[LocalAlgSettingsID]]


            ## fill only element of active tab
            if (self.TabGUI.currentIndex() == LocalAlgSettingsID):


                ## get parameters for current algorithm
                LocalAlg = LocalAlgSettings[0].lower()                                      ## get name of current algorithm
                self.ActiveAlg = LocalAlg
                LocalSpecialSettings = LocalAlgSettings[1]                                  ## get special settings for current algorithm
                self.LocalMaxIter = int(LocalAlgSettings[2])                                ## get max. number of iterations
                self.LoaclNumProc = LocalAlgSettings[3]                                     ## get number of cores
                self.LocalHostFileName = LocalAlgSettings[4]                                ## get path and name of host file
                self.LocalLimitChi2 = LocalAlgSettings[5]                                   ## get lower limit of chi^2
                self.LocalRenormalizedChi2Flag = LocalAlgSettings[6]                        ## get flag for usage of renormalized chi^2
                self.LocalDeterminationChi2 = LocalAlgSettings[7]                           ## get method for chi^2 determination
                self.LocalSaveChi2Flag = LocalAlgSettings[8]                                ## get flag for saving chi^2 values
                self.LocalTitleXAxis = LocalAlgSettings[9]                                  ## get title for x axis
                self.LocalTitleYAxis = LocalAlgSettings[10]                                 ## get title for y axis
                self.LocalPlotOnIterFlag = LocalAlgSettings[11]                             ## get flag for create plot for each iteration


                ## set entry in algorithm combo list
                i = self.KnwonAlgorithmsLower.index(LocalAlg)
                self.ComboCurrAlgorithm[LocalAlgSettingsID].setCurrentIndex(i)


                ## Levenberg-Marquardt
                if (LocalAlg == "levenberg-marquardt"):
                    self.LocalLMVarValue = LocalSpecialSettings[0]
                    self.LineEditLMVarVal[LocalAlgSettingsID].setText(str(self.LocalLMVarValue))
                    if (self.ChooseImplementationFlag):
                        try:
                            self.LocalLMMethod = LocalSpecialSettings[1]
                        except:
                            self.LocalLMMethod = "NR"
                        if (self.LocalLMMethod.lower() == "nr"):
                            self.QRadioButtonLMMethodNR[LocalAlgSettingsID].setChecked(True)
                            self.QRadioButtonLMMethodMIN[LocalAlgSettingsID].setChecked(False)
                        else:
                            self.QRadioButtonLMMethodNR[LocalAlgSettingsID].setChecked(False)
                            self.QRadioButtonLMMethodMIN[LocalAlgSettingsID].setChecked(True)


                ## Simulated-Annealing
                elif (LocalAlg == "simulated-annealing"):
                    self.LocalSATemp = LocalSpecialSettings[0]
                    self.LineEditSATemp[LocalAlgSettingsID].setText(str(self.LocalSATemp))
                    self.LocalSATempRedCoef = LocalSpecialSettings[1]
                    self.LineEditSATempRedCoef[LocalAlgSettingsID].setText(str(self.LocalSATempRedCoef))
                    self.LocalSANumRed = LocalSpecialSettings[2]
                    self.LineEditSARedCycle[LocalAlgSettingsID].setText(str(self.LocalSANumRed))
                    self.LocalSANumReHeat = LocalSpecialSettings[3]
                    self.LineEditSAReheatings[LocalAlgSettingsID].setText(str(self.LocalSANumReHeat))
                    if (self.ChooseImplementationFlag):
                        try:
                            self.LocalSAMethod = LocalSpecialSettings[4]
                        except:
                            self.LocalSAMethod = "NR"
                        if (self.LocalSAMethod.lower() == "nr"):
                            self.QRadioButtonSAMethodNR[LocalAlgSettingsID].setChecked(True)
                            self.QRadioButtonSAMethodMIN[LocalAlgSettingsID].setChecked(False)
                        else:
                            self.QRadioButtonSAMethodNR[LocalAlgSettingsID].setChecked(False)
                            self.QRadioButtonSAMethodMIN[LocalAlgSettingsID].setChecked(True)


                ## Bees
                elif (LocalAlg == "bees"):
                    self.LocalBeesNumBestSites = LocalSpecialSettings[0]
                    self.LineEditBeesBestSites[LocalAlgSettingsID].setText(str(self.LocalBeesNumBestSites))
                    self.LocalBeesNumParticles = LocalSpecialSettings[1]
                    if (self.LocalBeesNumParticles == (-1)):
                        self.LineEditBeesNumPart[LocalAlgSettingsID].setText("automatic")
                    else:
                        self.LineEditBeesNumPart[LocalAlgSettingsID].setText(str(self.LocalBeesNumParticles))


                ## PSO
                elif (LocalAlg == "pso"):
                    self.LocalPSONumBestSites = LocalSpecialSettings[0]
                    self.LineEditPSONumBestSites[LocalAlgSettingsID].setText(str(self.LocalPSONumBestSites))


                ## Genetic
                elif (LocalAlg == "genetic"):
                    self.LocalGeneticNumBestSites = LocalSpecialSettings[0]
                    try:
                        self.LocalGeneticNumBestSites = self.LocalGeneticNumBestSites.strip()
                    except:
                        pass
                    if (self.LocalGeneticNumBestSites != ""):
                        self.LineEditGeneticNumBestSites[LocalAlgSettingsID].setText(str(self.LocalGeneticNumBestSites))
                    else:
                        self.LineEditGeneticNumBestSites[LocalAlgSettingsID].setText("1")
                    self.LocalGeneticNumParticles = LocalSpecialSettings[1]
                    if (self.LocalGeneticNumParticles == (-1)):
                        self.LineEditGeneticNumPart[LocalAlgSettingsID].setText("automatic")
                    else:
                        self.LineEditGeneticNumPart[LocalAlgSettingsID].setText(str(self.LocalGeneticNumParticles))
                    self.GeneticNewRangeFlag = LocalSpecialSettings[2]
                    self.FlagNewRangeGenetic[LocalAlgSettingsID].setChecked(self.GeneticNewRangeFlag)


                ## Nested-Sampling
                elif (LocalAlg == "nested-sampling"):
                    self.LocalNSNumBestSites = LocalSpecialSettings[0]
                    self.LineEditNSNumBestSites[LocalAlgSettingsID].setText(str(self.LocalNSNumBestSites))
                    self.LocalNSNumParticles = LocalSpecialSettings[1]
                    if (self.LocalNSNumParticles == (-1)):
                        self.LineEditNSNumPart[LocalAlgSettingsID].setText("automatic")
                    else:
                        self.LineEditNSNumPart[LocalAlgSettingsID].setText(str(self.LocalNSNumParticles))


                ## MCMC
                elif (LocalAlg == "mcmc"):
                    self.LocalMCMCNumBestSites = LocalSpecialSettings[0]
                    self.LineEditMCMCNumBestSites[LocalAlgSettingsID].setText(str(self.LocalMCMCNumBestSites))
                    self.LocalMCMCNumParticles = LocalSpecialSettings[1]
                    if (self.LocalMCMCNumParticles == (-1)):
                        self.LineEditMCMCNumPart[LocalAlgSettingsID].setText("automatic")
                    else:
                        self.LineEditMCMCNumPart[LocalAlgSettingsID].setText(str(self.LocalMCMCNumParticles))
                    self.LocalMCMCBurnInIter = LocalSpecialSettings[2]
                    self.LineEditMCMCNumBurnIInter[LocalAlgSettingsID].setText(str(self.LocalMCMCBurnInIter))


                ## Interval-NS
                elif (LocalAlg == "interval-ns" or LocalAlg == "ins"):
                    self.LocalINSNumBestSites = LocalSpecialSettings[0]
                    self.LineEditINSNumBestSites[LocalAlgSettingsID].setText(str(self.LocalINSNumBestSites))
                    self.LocalINSVolBond = LocalSpecialSettings[1]
                    self.LineEditINSvolBond[LocalAlgSettingsID].setText(str(self.LocalINSVolBond))
                    self.LocalINSDeltaIncl = LocalSpecialSettings[2]
                    self.LineEditINSDeltaIncl[LocalAlgSettingsID].setText(str(self.LocalINSDeltaIncl))


                ## AdditionalPackages
                elif (LocalAlg == "additionalpackages"):
                    self.LocalAPAlgID = self.APLocalAlgList.index(LocalSpecialSettings[0])
                    self.APComboAlgSelect[LocalAlgSettingsID].setCurrentIndex(self.LocalAPAlgID)


                ## Error estimation
                elif (LocalAlg == "errorestim" or LocalAlg == "errorestim_ins"):
                    self.LocalErrorEstimMethodID = self.ErrorEstimLocalmethodList.index(LocalSpecialSettings[0])
                    self.ComboErrorMethod[LocalAlgSettingsID].setCurrentIndex(self.LocalErrorEstimMethodID)


                    ## add additional elements for each method
                    if (self.LocalErrorEstimMethodID == 0):                                 ## MCMC
                        self.LocalErrorNumParticles = LocalSpecialSettings[1]
                        if (self.LocalErrorNumParticles == (-1)):
                            self.LineEditErrorMCMCNumPart[LocalAlgSettingsID].setText("automatic")
                        else:
                            self.LineEditErrorMCMCNumPart[LocalAlgSettingsID].setText(str(self.LocalErrorNumParticles))
                        self.LocalErrorMCMCBurnInIter = LocalSpecialSettings[2]
                        self.LineEditErrorMCMCNumBurnIInter[LocalAlgSettingsID].setText(str(self.LocalErrorMCMCBurnInIter))
                    elif (self.LocalErrorEstimMethodID == 1):                               ## Fisher
                        self.LocalErrorLMVarValue = LocalSpecialSettings[1]
                        self.LineEditErrorLMVarVal[LocalAlgSettingsID].setText(str(self.LocalErrorLMVarValue))


                ## general settings
                self.LineEditGeneralMaxIter[LocalAlgSettingsID].setText(str(self.LocalMaxIter))
                self.LineEditGeneralNumProc[LocalAlgSettingsID].setText(str(self.LoaclNumProc))
                self.LineEditGeneralHostFile[LocalAlgSettingsID].setText(str(self.LocalHostFileName))
                self.LineEditGeneralLimitChi2[LocalAlgSettingsID].setText(str(self.LocalLimitChi2))
                self.GeneralFlagRenormalizedChi2[LocalAlgSettingsID].setChecked(self.LocalRenormalizedChi2Flag)
                if (self.ShowChi2Methods):
                    self.GeneralMethodChi2[LocalAlgSettingsID].setCurrentIndex(self.LocalDeterminationChi2)
                self.GeneralFlagSaveChi2[LocalAlgSettingsID].setChecked(self.LocalSaveChi2Flag)
                if (self.ShowGraphicOptions):
                    self.LineEditGeneralXAxis[LocalAlgSettingsID].setText(str(self.LocalTitleXAxis))
                    self.LineEditGeneralYAxis[LocalAlgSettingsID].setText(str(self.LocalTitleYAxis))
                    self.GeneralPlotOnIteration[LocalAlgSettingsID].setChecked(self.LocalPlotOnIterFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## unlock elements
        self.lockElements = False


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create entries for each tab depending on selected algorithm
    def CreateEntries(self, layout, AlgID, LocalAlg, LocalSpecialSettings):
        """

    input parameters:
    -----------------

        - layout:                   layout object

        - AlgID:                    algorithm id

        - LocalAlg:                 current algorithm

        - LocalSpecialSettings:     special settings for current algorithm


    output parameters:
    ------------------

        - NumberElements:   number of used elements
        """

        # Debug:
        # print ("\n\nAlgID = ", AlgID)
        # print ("LocalAlg = ", LocalAlg)


        ## initialize return parameter
        NumberElements = 1


        ##================================================================================================================================================
        ## Levenberg-Marquardt
        self.LabelErrorMethod = QtWidgets.QLabel("Algorithm:")
        layout.addWidget(self.LabelErrorMethod, NumberElements, 0)
        self.ComboCurrAlgorithm[AlgID] = QtWidgets.QComboBox()
        self.ComboCurrAlgorithm[AlgID].setObjectName("ComboCurrAlgorithm")
        self.ComboCurrAlgorithm[AlgID].setToolTip('Choose algorithm.')
        self.ComboCurrAlgorithm[AlgID].currentIndexChanged.connect(self.ComboWidget)
        self.ComboCurrAlgorithm[AlgID].setEnabled(True)
        self.ComboCurrAlgorithm[AlgID].addItems(self.KnwonAlgorithms)
        layout.addWidget(self.ComboCurrAlgorithm[AlgID], NumberElements, 1)
        NumberElements += 1


        ##================================================================================================================================================
        ## special settings for each algorithms


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Levenberg-Marquardt
        if (LocalAlg == "levenberg-marquardt"):


            ## add line edit element for entering value of variation
            self.LabelLMVarVal = QtWidgets.QLabel("Value of the variation:")
            layout.addWidget(self.LabelLMVarVal, NumberElements, 0)
            self.LineEditLMVarVal[AlgID] = QtWidgets.QLineEdit()
            self.LineEditLMVarVal[AlgID].setObjectName("LMVarVal")
            # self.LineEditLMVarVal[AlgID].setFixedWidth(350)
            self.LineEditLMVarVal[AlgID].setToolTip('Value of variation used for determine the gradient of the chi^2 function.')
            self.LineEditLMVarVal[AlgID].setValidator(QtGui.QDoubleValidator(self.MinLMVarVal, self.MaxLMVarVal, 6))
            self.LineEditLMVarVal[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditLMVarVal[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditLMVarVal[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add radio button for selecting implementation
            if (self.ChooseImplementationFlag):
                self.QRadioButtonLMMethodNR[AlgID] = QtWidgets.QRadioButton("NR")
                self.QRadioButtonLMMethodNR[AlgID].toggled.connect(lambda:self.RadioButtonWidget(self.QRadioButtonLMMethodNR[AlgID]))
                self.QRadioButtonLMMethodNR[AlgID].setEnabled(True)
                layout.addWidget(self.QRadioButtonLMMethodNR[AlgID], NumberElements, 0)
                self.QRadioButtonLMMethodMIN[AlgID] = QtWidgets.QRadioButton("MINPACK")
                self.QRadioButtonLMMethodMIN[AlgID].toggled.connect(lambda:self.RadioButtonWidget(self.QRadioButtonLMMethodMIN[AlgID]))
                self.QRadioButtonLMMethodMIN[AlgID].setEnabled(True)
                layout.addWidget(self.QRadioButtonLMMethodMIN[AlgID], NumberElements, 1)
                NumberElements += 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Simulated-Annealing
        elif (LocalAlg == "simulated-annealing"):


            ## add line edit element for entering temperature
            self.LabelSATemp = QtWidgets.QLabel("Temperature:")
            layout.addWidget(self.LabelSATemp, NumberElements, 0)
            self.LineEditSATemp[AlgID] = QtWidgets.QLineEdit()
            self.LineEditSATemp[AlgID].setObjectName("TempSA")
            # self.LineEditSATemp[AlgID].setFixedWidth(350)
            self.LineEditSATemp[AlgID].setToolTip('Temperature (value of variation).')
            self.LineEditSATemp[AlgID].setValidator(QtGui.QDoubleValidator(self.MinTempSA, self.MaxTempSA, 6))
            self.LineEditSATemp[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditSATemp[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditSATemp[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for entering temperature reduction coefficient
            self.LabelSATempRedCoef = QtWidgets.QLabel("Temperature red. coeff.:")
            layout.addWidget(self.LabelSATempRedCoef, NumberElements, 0)
            self.LineEditSATempRedCoef[AlgID] = QtWidgets.QLineEdit()
            self.LineEditSATempRedCoef[AlgID].setObjectName("TempRedCoefSA")
            # self.LineEditSATempRedCoef[AlgID].setFixedWidth(350)
            self.LineEditSATempRedCoef[AlgID].setToolTip('Enter reduction coefficient for temperature.')
            self.LineEditSATempRedCoef[AlgID].setValidator(QtGui.QDoubleValidator(self.MinSARedCoef, self.MaxSARedCoef, 6))
            self.LineEditSATempRedCoef[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditSATempRedCoef[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditSATempRedCoef[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for entering number of reduction cycles
            self.LabelSARedCycle = QtWidgets.QLabel("Number of reduction cycles:")
            layout.addWidget(self.LabelSARedCycle, NumberElements, 0)
            self.LineEditSARedCycle[AlgID] = QtWidgets.QLineEdit()
            self.LineEditSARedCycle[AlgID].setObjectName("SARedCycle")
            # self.LineEditSARedCycle[AlgID].setFixedWidth(350)
            self.LineEditSARedCycle[AlgID].setToolTip('Enter number of reduction cycles.')
            self.LineEditSARedCycle[AlgID].setValidator(QtGui.QDoubleValidator(self.MinSARedCycles, self.MaxSARedCycles, 6))
            self.LineEditSARedCycle[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditSARedCycle[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditSARedCycle[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for entering number of reheating phases
            self.LabelSAReheatings = QtWidgets.QLabel("Number of reheating phases:")
            layout.addWidget(self.LabelSAReheatings, NumberElements, 0)
            self.LineEditSAReheatings[AlgID] = QtWidgets.QLineEdit()
            self.LineEditSAReheatings[AlgID].setObjectName("SAReheatPhases")
            # self.LineEditSAReheatings[AlgID].setFixedWidth(350)
            self.LineEditSAReheatings[AlgID].setToolTip('Enter number of reheating phases.')
            self.LineEditSAReheatings[AlgID].setValidator(QtGui.QDoubleValidator(self.MinSAReheat, self.MaxSAReheat, 6))
            self.LineEditSAReheatings[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditSAReheatings[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditSAReheatings[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add radio button for selecting implementation
            if (self.ChooseImplementationFlag):
                self.QRadioButtonSAMethodNR[AlgID] = QtWidgets.QRadioButton("NR")
                self.QRadioButtonSAMethodNR[AlgID].toggled.connect(lambda:self.RadioButtonWidget(self.QRadioButtonSAMethodNR[AlgID]))
                self.QRadioButtonSAMethodNR[AlgID].setEnabled(True)
                layout.addWidget(self.QRadioButtonSAMethodNR[AlgID], NumberElements, 0)
                self.QRadioButtonSAMethodMIN[AlgID] = QtWidgets.QRadioButton("SCIPY")
                self.QRadioButtonSAMethodMIN[AlgID].toggled.connect(lambda:self.RadioButtonWidget(self.QRadioButtonSAMethodMIN[AlgID]))
                self.QRadioButtonSAMethodMIN[AlgID].setEnabled(True)
                layout.addWidget(self.QRadioButtonSAMethodMIN[AlgID], NumberElements, 1)
                NumberElements += 1


            ## add line edit element for entering schedule for Simulated-Annealing algorithm (only valid for scipy version)
            #    if ((not ChooseImplementationFlag) or self.SAImplementation != "NR"):
            #        self.LocalSASchedule = "fast"


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Bees
        elif (LocalAlg == "bees"):


            ## add line edit element for entering number of best sites
            self.LabelBeesBestSites = QtWidgets.QLabel("Number of best sites:")
            layout.addWidget(self.LabelBeesBestSites, NumberElements, 0)
            self.LineEditBeesBestSites[AlgID] = QtWidgets.QLineEdit()
            self.LineEditBeesBestSites[AlgID].setObjectName("BeesNumBestSites")
            # self.LineEditBeesBestSites[AlgID].setFixedWidth(350)
            self.LineEditBeesBestSites[AlgID].setToolTip('Enter the number of best sites.')
            self.LineEditBeesBestSites[AlgID].setValidator(QtGui.QDoubleValidator(1, self.LocalMaxIter, 6))
            self.LineEditBeesBestSites[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditBeesBestSites[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditBeesBestSites[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for entering number of particles
            self.LabelBeesNumPart = QtWidgets.QLabel("Number of particles:")
            layout.addWidget(self.LabelBeesNumPart, NumberElements, 0)
            self.LineEditBeesNumPart[AlgID] = QtWidgets.QLineEdit()
            self.LineEditBeesNumPart[AlgID].setObjectName("BeesNumParticles")
            # self.LineEditBeesNumPart[AlgID].setFixedWidth(350)
            self.LineEditBeesNumPart[AlgID].setToolTip('Enter the number of particles.')
            self.LineEditBeesNumPart[AlgID].setValidator(QtGui.QDoubleValidator(self.TotalMinParticles, self.TotalMaxParticles, 6))
            self.LineEditBeesNumPart[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditBeesNumPart[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditBeesNumPart[AlgID], NumberElements, 1)
            NumberElements += 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## PSO
        elif (LocalAlg == "pso"):


            ## add line edit element for entering number of best sites
            self.LabelPSONumBestSites = QtWidgets.QLabel("Number of best sites:")
            layout.addWidget(self.LabelPSONumBestSites, NumberElements, 0)
            self.LineEditPSONumBestSites[AlgID] = QtWidgets.QLineEdit()
            self.LineEditPSONumBestSites[AlgID].setObjectName("PSONumBestSites")
            # self.LineEditPSONumBestSites[AlgID].setFixedWidth(350)
            self.LineEditPSONumBestSites[AlgID].setToolTip('Enter the number of best sites.')
            self.LineEditPSONumBestSites[AlgID].setValidator(QtGui.QDoubleValidator(1, self.LocalMaxIter, 6))
            self.LineEditPSONumBestSites[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditPSONumBestSites[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditPSONumBestSites[AlgID], NumberElements, 1)
            NumberElements += 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Genetic
        elif (LocalAlg == "genetic"):


            ## add line edit element for entering number of best sites
            self.LabelGeneticNumBestSites = QtWidgets.QLabel("Number of best sites:")
            layout.addWidget(self.LabelGeneticNumBestSites, NumberElements, 0)
            self.LineEditGeneticNumBestSites[AlgID] = QtWidgets.QLineEdit()
            self.LineEditGeneticNumBestSites[AlgID].setObjectName("GeneticNumBestSites")
            # self.LineEditGeneticNumBestSites[AlgID].setFixedWidth(350)
            self.LineEditGeneticNumBestSites[AlgID].setToolTip('Enter the number of best sites.')
            self.LineEditGeneticNumBestSites[AlgID].setValidator(QtGui.QDoubleValidator(1, self.LocalMaxIter, 6))
            self.LineEditGeneticNumBestSites[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditGeneticNumBestSites[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditGeneticNumBestSites[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for entering number of particles
            self.LabelGeneticNumPart = QtWidgets.QLabel("Number of particles:")
            layout.addWidget(self.LabelGeneticNumPart, NumberElements, 0)
            self.LineEditGeneticNumPart[AlgID] = QtWidgets.QLineEdit()
            self.LineEditGeneticNumPart[AlgID].setObjectName("GeneticNumParticles")
            # self.LineEditGeneticNumPart[AlgID].setFixedWidth(350)
            self.LineEditGeneticNumPart[AlgID].setToolTip('Enter the number of particles.')
            self.LineEditGeneticNumPart[AlgID].setValidator(QtGui.QDoubleValidator(self.TotalMinParticles, self.TotalMaxParticles, 6))
            self.LineEditGeneticNumPart[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditGeneticNumPart[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditGeneticNumPart[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for selecting if new ranges are computed or not
            self.LabelFlagNewRangeGenetic = QtWidgets.QLabel("Use new ranges:")
            layout.addWidget(self.LabelFlagNewRangeGenetic, NumberElements, 0)
            self.FlagNewRangeGenetic[AlgID] = QtWidgets.QCheckBox()
            self.FlagNewRangeGenetic[AlgID].setObjectName("NewRangeFlag")
            #    self.FlagNewRangeGenetic[AlgID].setFixedWidth(100)
            self.FlagNewRangeGenetic[AlgID].setToolTip('Compute new parameter ranges?')
            self.FlagNewRangeGenetic[AlgID].clicked.connect(self.CheckBoxWidget)
            self.FlagNewRangeGenetic[AlgID].setEnabled(True)
            layout.addWidget(self.FlagNewRangeGenetic[AlgID], NumberElements, 1)
            NumberElements += 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Nested-Sampling
        elif (LocalAlg == "nested-sampling"):


            ## add line edit element for entering number of best sites
            self.LabelNSNumBestSites = QtWidgets.QLabel("Number of best sites:")
            layout.addWidget(self.LabelNSNumBestSites, NumberElements, 0)
            self.LineEditNSNumBestSites[AlgID] = QtWidgets.QLineEdit()
            self.LineEditNSNumBestSites[AlgID].setObjectName("NSNumBestSites")
            # self.LineEditNSNumBestSites[AlgID].setFixedWidth(350)
            self.LineEditNSNumBestSites[AlgID].setToolTip('Enter the number of best sites.')
            self.LineEditNSNumBestSites[AlgID].setValidator(QtGui.QDoubleValidator(1, self.LocalMaxIter, 6))
            self.LineEditNSNumBestSites[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditNSNumBestSites[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditNSNumBestSites[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for entering number of particles
            self.LabelNSNumPart = QtWidgets.QLabel("Number of particles:")
            layout.addWidget(self.LabelNSNumPart, NumberElements, 0)
            self.LineEditNSNumPart[AlgID] = QtWidgets.QLineEdit()
            self.LineEditNSNumPart[AlgID].setObjectName("NSNumParticles")
            # self.LineEditNSNumPart[AlgID].setFixedWidth(350)
            self.LineEditNSNumPart[AlgID].setToolTip('Enter the number of particles.')
            self.LineEditNSNumPart[AlgID].setValidator(QtGui.QDoubleValidator(self.TotalMinParticles, self.TotalMaxParticles, 6))
            self.LineEditNSNumPart[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditNSNumPart[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditNSNumBestSites[AlgID], NumberElements, 1)
            NumberElements += 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## MCMC
        elif (LocalAlg == "mcmc"):


            ## add line edit element for entering number of best sites
            self.LabelMCMCNumBestSites = QtWidgets.QLabel("Number of best sites:")
            layout.addWidget(self.LabelMCMCNumBestSites, NumberElements, 0)
            self.LineEditMCMCNumBestSites[AlgID] = QtWidgets.QLineEdit()
            self.LineEditMCMCNumBestSites[AlgID].setObjectName("MCMCNumBestSites")
            # self.LineEditMCMCNumBestSites[AlgID].setFixedWidth(350)
            self.LineEditMCMCNumBestSites[AlgID].setToolTip('Enter the number of best sites.')
            self.LineEditMCMCNumBestSites[AlgID].setValidator(QtGui.QDoubleValidator(1, self.LocalMaxIter, 6))
            self.LineEditMCMCNumBestSites[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditMCMCNumBestSites[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditMCMCNumBestSites[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for entering number of particles
            self.LabelMCMCNumPart = QtWidgets.QLabel("Number of particles:")
            layout.addWidget(self.LabelMCMCNumPart, NumberElements, 0)
            self.LineEditMCMCNumPart[AlgID] = QtWidgets.QLineEdit()
            self.LineEditMCMCNumPart[AlgID].setObjectName("MCMCNumParticles")
            # self.LineEditMCMCNumPart[AlgID].setFixedWidth(350)
            self.LineEditMCMCNumPart[AlgID].setToolTip('Enter the number of particles.')
            self.LineEditMCMCNumPart[AlgID].setValidator(QtGui.QDoubleValidator(self.TotalMinParticles, self.TotalMaxParticles, 6))
            self.LineEditMCMCNumPart[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditMCMCNumPart[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditMCMCNumPart[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for entering number of iterations for burn-in phase
            self.LabelMCMCNumBurnIInter = QtWidgets.QLabel("Burn-In phase: Number of iterations:")
            layout.addWidget(self.LabelMCMCNumBurnIInter, NumberElements, 0)
            self.LineEditMCMCNumBurnIInter[AlgID] = QtWidgets.QLineEdit()
            self.LineEditMCMCNumBurnIInter[AlgID].setObjectName("MCMCNumBurnInIter")
            # self.LineEditMCMCNumBurnIInter[AlgID].setFixedWidth(350)
            self.LineEditMCMCNumBurnIInter[AlgID].setToolTip('Enter the number of iterations for burn-in phase.')
            self.LineEditMCMCNumBurnIInter[AlgID].setValidator(QtGui.QDoubleValidator(0, self.LocalMaxIter, 6))
            self.LineEditMCMCNumBurnIInter[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditMCMCNumBurnIInter[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditMCMCNumBurnIInter[AlgID], NumberElements, 1)
            NumberElements += 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Interval-NS
        elif (LocalAlg == "interval-ns" or LocalAlg == "ins"):


            ## add line edit element for entering number of best sites
            self.LabelINSNumBestSites = QtWidgets.QLabel("Number of best sites:")
            layout.addWidget(self.LabelINSNumBestSites, NumberElements, 0)
            self.LineEditINSNumBestSites[AlgID] = QtWidgets.QLineEdit()
            self.LineEditINSNumBestSites[AlgID].setObjectName("INSNumBestSites")
            # self.LineEditINSNumBestSites[AlgID].setFixedWidth(350)
            self.LineEditINSNumBestSites[AlgID].setToolTip('Enter the number of best sites.')
            self.LineEditINSNumBestSites[AlgID].setValidator(QtGui.QDoubleValidator(1, self.LocalMaxIter, 6))
            self.LineEditINSNumBestSites[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditINSNumBestSites[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditINSNumBestSites[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for entering value for vol_bond
            self.LabelINSvolBond = QtWidgets.QLabel("vol_bound:")
            layout.addWidget(self.LabelINSvolBond, NumberElements, 0)
            self.LineEditINSvolBond[AlgID] = QtWidgets.QLineEdit()
            self.LineEditINSvolBond[AlgID].setObjectName("INSvolBound")
            # self.LineEditINSvolBond[AlgID].setFixedWidth(350)
            self.LineEditINSvolBond[AlgID].setToolTip('Enter vaue for vol_bound.')
            self.LineEditINSvolBond[AlgID].setValidator(QtGui.QDoubleValidator(1, self.LocalMaxIter, 6))
            self.LineEditINSvolBond[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditINSvolBond[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditINSvolBond[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for entering value for delta_incl
            self.LabelINSDeltaIncl = QtWidgets.QLabel("delta_incl:")
            layout.addWidget(self.LabelINSDeltaIncl, NumberElements, 0)
            self.LineEditINSDeltaIncl[AlgID] = QtWidgets.QLineEdit()
            self.LineEditINSDeltaIncl[AlgID].setObjectName("INSDeltaIncl")
            # self.LineEditINSDeltaIncl[AlgID].setFixedWidth(350)
            self.LineEditINSDeltaIncl[AlgID].setToolTip('Enter vaue for delta_incl.')
            self.LineEditINSDeltaIncl[AlgID].setValidator(QtGui.QDoubleValidator(1, self.LocalMaxIter, 6))
            self.LineEditINSDeltaIncl[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditINSDeltaIncl[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditINSDeltaIncl[AlgID], NumberElements, 1)
            NumberElements += 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## AdditionalPackages
        elif (LocalAlg == "additionalpackages"):
            self.LabelAPComboAlgSelect = QtWidgets.QLabel("Choose algorithm:")
            layout.addWidget(self.LabelAPComboAlgSelect, NumberElements, 0)
            self.APComboAlgSelect[AlgID] = QtWidgets.QComboBox()
            self.APComboAlgSelect[AlgID].setObjectName("ComboAPAlg")
            self.APComboAlgSelect[AlgID].setToolTip('Choose algorithm for AP.')
            self.APComboAlgSelect[AlgID].currentIndexChanged.connect(self.ComboWidget)
            self.APComboAlgSelect[AlgID].setEnabled(True)
            self.APComboAlgSelect[AlgID].addItems(self.APLocalAlgList)
            layout.addWidget(self.APComboAlgSelect[AlgID], NumberElements, 1)
            NumberElements += 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Error estimation
        elif (LocalAlg == "errorestim" or LocalAlg == "errorestim_ins"):
            self.LabelErrorMethod = QtWidgets.QLabel("Choose method:")
            layout.addWidget(self.LabelErrorMethod, NumberElements, 0)
            self.ComboErrorMethod[AlgID] = QtWidgets.QComboBox()
            self.ComboErrorMethod[AlgID].setObjectName("ComboErrorEstimMethod")
            self.ComboErrorMethod[AlgID].setToolTip('Choose method for error estimation.')
            self.ComboErrorMethod[AlgID].currentIndexChanged.connect(self.ComboWidget)
            self.ComboErrorMethod[AlgID].setEnabled(True)
            self.ComboErrorMethod[AlgID].addItems(self.ErrorEstimLocalmethodList)
            layout.addWidget(self.ComboErrorMethod[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add additional elements for each method
            self.LocalErrorEstimMethodID = self.ErrorEstimLocalmethodList.index(LocalSpecialSettings[0])
            if (self.LocalErrorEstimMethodID == 0):                                         ## MCMC


                ## add line edit element for entering number of particles
                self.LabelErrorMCMCNumPart = QtWidgets.QLabel("Number of particles:")
                layout.addWidget(self.LabelErrorMCMCNumPart, NumberElements, 0)
                self.LineEditErrorMCMCNumPart[AlgID] = QtWidgets.QLineEdit()
                self.LineEditErrorMCMCNumPart[AlgID].setObjectName("ErrorNumParticles")
                # self.LineEditErrorMCMCNumPart[AlgID].setFixedWidth(350)
                self.LineEditErrorMCMCNumPart[AlgID].setToolTip('Enter the number of particles.')
                self.LineEditErrorMCMCNumPart[AlgID].setValidator(QtGui.QDoubleValidator(self.TotalMinParticles, self.TotalMaxParticles, 6))
                self.LineEditErrorMCMCNumPart[AlgID].textChanged.connect(self.LineEditWidget)
                self.LineEditErrorMCMCNumPart[AlgID].setEnabled(True)
                self.LocalErrorNumParticles = LocalSpecialSettings[1]
                if (self.LocalErrorNumParticles == (-1)):
                    self.LineEditErrorMCMCNumPart[AlgID].setText("automatic")
                else:
                    self.LineEditErrorMCMCNumPart[AlgID].setText(str(self.LocalErrorNumParticles))
                layout.addWidget(self.LineEditErrorMCMCNumPart[AlgID], NumberElements, 1)
                NumberElements += 1


                ## add line edit element for entering number of iterations for burn-in phase
                self.LabelErrorMCMCNumBurnIInter = QtWidgets.QLabel("Burn-In phase: Number of iterations:")
                layout.addWidget(self.LabelErrorMCMCNumBurnIInter, NumberElements, 0)
                self.LineEditErrorMCMCNumBurnIInter[AlgID] = QtWidgets.QLineEdit()
                self.LineEditErrorMCMCNumBurnIInter[AlgID].setObjectName("ErrorNumBurnInIter")
                # self.LineEditErrorMCMCNumBurnIInter[AlgID].setFixedWidth(350)
                self.LineEditErrorMCMCNumBurnIInter[AlgID].setToolTip('Enter the number of iterations for burn-in phase.')
                self.LineEditErrorMCMCNumBurnIInter[AlgID].setValidator(QtGui.QDoubleValidator(0, self.LocalMaxIter, 6))
                self.LineEditErrorMCMCNumBurnIInter[AlgID].textChanged.connect(self.LineEditWidget)
                self.LineEditErrorMCMCNumBurnIInter[AlgID].setEnabled(True)
                self.LocalMCMCBurnInIter = LocalSpecialSettings[2]
                self.LineEditErrorMCMCNumBurnIInter[AlgID].setText(str(self.LocalMCMCBurnInIter))
                layout.addWidget(self.LineEditErrorMCMCNumBurnIInter[AlgID], NumberElements, 1)
                NumberElements += 1


            ## add Fisher elements
            elif (self.LocalErrorEstimMethodID == 1):                                       ## Fisher


                ## add line edit element for entering value of variation
                self.LabelErrorLMVarVal = QtWidgets.QLabel("Value of the variation:")
                layout.addWidget(self.LabelErrorLMVarVal, NumberElements, 0)
                self.LineEditErrorLMVarVal[AlgID] = QtWidgets.QLineEdit()
                self.LineEditErrorLMVarVal[AlgID].setObjectName("ErrorLMVarVal")
                # self.LineEditErrorLMVarVal.setFixedWidth(350)
                self.LineEditErrorLMVarVal[AlgID].setToolTip('Value of variation used for determine the gradient of the chi^2 function.')
                self.LineEditErrorLMVarVal[AlgID].setValidator(QtGui.QDoubleValidator(self.MinLMVarVal, self.MaxLMVarVal, 6))
                self.LineEditErrorLMVarVal[AlgID].textChanged.connect(self.LineEditWidget)
                self.LineEditErrorLMVarVal[AlgID].setEnabled(True)
                self.LocalErrorLMVarValue = LocalSpecialSettings[1]
                self.LineEditErrorLMVarVal[AlgID].setText(str(self.LocalErrorLMVarValue))
                layout.addWidget(self.LineEditErrorLMVarVal[AlgID], NumberElements, 1)
                NumberElements += 1


        ##================================================================================================================================================
        ## general settings


        ## add line edit element for entering max. number of iterations
        self.LabelGeneralMaxIter = QtWidgets.QLabel("Max. number of iterations:")
        layout.addWidget(self.LabelGeneralMaxIter, NumberElements, 0)
        self.LineEditGeneralMaxIter[AlgID] = QtWidgets.QLineEdit()
        self.LineEditGeneralMaxIter[AlgID].setObjectName("MaxIter")
        # self.LineEditGeneralMaxIter[AlgID].setFixedWidth(350)
        self.LineEditGeneralMaxIter[AlgID].setToolTip('Enter max. number of iterations.')
        self.LineEditGeneralMaxIter[AlgID].setValidator(QtGui.QDoubleValidator(0, self.TotalMaxIterations, 6))
        self.LineEditGeneralMaxIter[AlgID].textChanged.connect(self.LineEditWidget)
        self.LineEditGeneralMaxIter[AlgID].setEnabled(True)
        layout.addWidget(self.LineEditGeneralMaxIter[AlgID], NumberElements, 1)
        NumberElements += 1


        ## add line edit element for entering number of processors
        self.LabelGeneralNumProc = QtWidgets.QLabel("Number of processors:")
        layout.addWidget(self.LabelGeneralNumProc, NumberElements, 0)
        self.LineEditGeneralNumProc[AlgID] = QtWidgets.QLineEdit()
        self.LineEditGeneralNumProc[AlgID].setObjectName("NumberProc")
        # self.LineEditGeneralNumProc[AlgID].setFixedWidth(350)
        self.LineEditGeneralNumProc[AlgID].setToolTip('Enter number of processors used for this algorithm.')
        self.LineEditGeneralNumProc[AlgID].setValidator(QtGui.QDoubleValidator(0, self.MaxProcessors, 6))
        self.LineEditGeneralNumProc[AlgID].textChanged.connect(self.LineEditWidget)
        self.LineEditGeneralNumProc[AlgID].setEnabled(True)
        layout.addWidget(self.LineEditGeneralNumProc[AlgID], NumberElements, 1)
        NumberElements += 1


        ## add line edit element for entering number of processors
        self.LabelGeneralHostFile = QtWidgets.QLabel("Name of host file:")
        layout.addWidget(self.LabelGeneralHostFile, NumberElements, 0)
        self.LineEditGeneralHostFile[AlgID] = QtWidgets.QLineEdit()
        self.LineEditGeneralHostFile[AlgID].setObjectName("EnterHostFile")
        # self.LineEditGeneralHostFile[AlgID].setFixedWidth(350)
        self.LineEditGeneralHostFile[AlgID].setToolTip('Enter path and name of host file.')
        # self.LineEditGeneralHostFile[AlgID].setValidator(QtGui.QDoubleValidator(0, self.MaxProcessors, 6))
        self.LineEditGeneralHostFile[AlgID].textChanged.connect(self.LineEditWidget)
        self.LineEditGeneralHostFile[AlgID].setEnabled(True)
        layout.addWidget(self.LineEditGeneralHostFile[AlgID], NumberElements, 1)
        NumberElements += 1


        ## add line edit element for entering limit of chi^2
        self.LabelGeneralLimitChi2 = QtWidgets.QLabel("Limit of chi^2:")
        layout.addWidget(self.LabelGeneralLimitChi2, NumberElements, 0)
        self.LineEditGeneralLimitChi2[AlgID] = QtWidgets.QLineEdit()
        self.LineEditGeneralLimitChi2[AlgID].setObjectName("Limitchi2")
        # self.LineEditGeneralLimitChi2[AlgID].setFixedWidth(350)
        self.LineEditGeneralLimitChi2[AlgID].setToolTip('Enter abort creteria, if chi^2 value is lower than this limit stop iteration.')
        self.LineEditGeneralLimitChi2[AlgID].setValidator(QtGui.QDoubleValidator(0, self.MaxLimitChi2, 6))
        self.LineEditGeneralLimitChi2[AlgID].textChanged.connect(self.LineEditWidget)
        self.LineEditGeneralLimitChi2[AlgID].setEnabled(True)
        layout.addWidget(self.LineEditGeneralLimitChi2[AlgID], NumberElements, 1)
        NumberElements += 1


        ## add check box element for selecting usage of renormalized chi^2
        self.LabelGeneralFlagRenormalizedChi2 = QtWidgets.QLabel("Use renormalized chi^2:")
        layout.addWidget(self.LabelGeneralFlagRenormalizedChi2, NumberElements, 0)
        self.GeneralFlagRenormalizedChi2[AlgID] = QtWidgets.QCheckBox()
        self.GeneralFlagRenormalizedChi2[AlgID].setObjectName("RenormalizedChi2")
        #    self.GeneralFlagRenormalizedChi2[AlgID].setFixedWidth(100)
        self.GeneralFlagRenormalizedChi2[AlgID].setToolTip('Use renormalized chi^2 or not?')
        self.GeneralFlagRenormalizedChi2[AlgID].clicked.connect(self.CheckBoxWidget)
        self.GeneralFlagRenormalizedChi2[AlgID].setEnabled(True)
        layout.addWidget(self.GeneralFlagRenormalizedChi2[AlgID], NumberElements, 1)
        NumberElements += 1


        ## method of chi^2 determination
        if (self.ShowChi2Methods):
            self.LabelAPGeneralMethodChi2 = QtWidgets.QLabel("Method of chi^2 determination:")
            layout.addWidget(self.LabelAPGeneralMethodChi2, NumberElements, 0)
            self.GeneralMethodChi2[AlgID] = QtWidgets.QComboBox()
            self.GeneralMethodChi2[AlgID].setObjectName("ComboChi2Det")
            self.GeneralMethodChi2[AlgID].setToolTip('Choose method of determine chi^2.')
            self.GeneralMethodChi2[AlgID].currentIndexChanged.connect(self.ComboWidget)
            self.GeneralMethodChi2[AlgID].setEnabled(True)
            self.GeneralMethodChi2[AlgID].addItems(self.MethodChi2LocalAlgList)
            layout.addWidget(self.GeneralMethodChi2[AlgID], NumberElements, 1)
            NumberElements += 1


        ## add check box element for selecting if chi^2 values are saved as well
        self.LabelGeneralFlagSaveChi2 = QtWidgets.QLabel("Save chi^2:")
        layout.addWidget(self.LabelGeneralFlagSaveChi2, NumberElements, 0)
        self.GeneralFlagSaveChi2[AlgID] = QtWidgets.QCheckBox()
        self.GeneralFlagSaveChi2[AlgID].setObjectName("SaveChi2")
        #    self.GeneralFlagSaveChi2[AlgID].setFixedWidth(100)
        self.GeneralFlagSaveChi2[AlgID].setToolTip('Save chi^2 values or not?')
        self.GeneralFlagSaveChi2[AlgID].clicked.connect(self.CheckBoxWidget)
        self.GeneralFlagSaveChi2[AlgID].setEnabled(True)
        layout.addWidget(self.GeneralFlagSaveChi2[AlgID], NumberElements, 1)
        NumberElements += 1


        ## graphic options
        if (self.ShowGraphicOptions):


            ## add line edit element for entering title for x-axis
            self.LabelGeneralXAxis = QtWidgets.QLabel("Title for x-axis:")
            layout.addWidget(self.LabelGeneralXAxis, NumberElements, 0)
            self.LineEditGeneralXAxis[AlgID] = QtWidgets.QLineEdit()
            self.LineEditGeneralXAxis[AlgID].setObjectName("EnterXAxis")
            # self.LineEditGeneralXAxis[AlgID].setFixedWidth(350)
            self.LineEditGeneralXAxis[AlgID].setToolTip('Enter a title for x-axis in the final plot.')
            # self.LineEditGeneralXAxis[AlgID].setValidator(QtGui.QDoubleValidator(0, self.MaxProcessors, 6))
            self.LineEditGeneralXAxis[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditGeneralXAxis[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditGeneralXAxis[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add line edit element for entering title for y-axis
            self.LabelGeneralYAxis = QtWidgets.QLabel("Title for y-axis:")
            layout.addWidget(self.LabelGeneralYAxis, NumberElements, 0)
            self.LineEditGeneralYAxis[AlgID] = QtWidgets.QLineEdit()
            self.LineEditGeneralYAxis[AlgID].setObjectName("EnterYAxis")
            # self.LineEditGeneralYAxis[AlgID].setFixedWidth(350)
            self.LineEditGeneralYAxis[AlgID].setToolTip('Enter a title for y-axis in the final plot.')
            # self.LineEditGeneralYAxis[AlgID].setValidator(QtGui.QDoubleValidator(0, self.MaxProcessors, 6))
            self.LineEditGeneralYAxis[AlgID].textChanged.connect(self.LineEditWidget)
            self.LineEditGeneralYAxis[AlgID].setEnabled(True)
            layout.addWidget(self.LineEditGeneralYAxis[AlgID], NumberElements, 1)
            NumberElements += 1


            ## add check box element for selecting if a plot is created for each iteration
            self.LabelGeneralPlotOnIteration = QtWidgets.QLabel("Plot each iteration:")
            layout.addWidget(self.LabelGeneralPlotOnIteration, NumberElements, 0)
            self.GeneralPlotOnIteration[AlgID] = QtWidgets.QCheckBox()
            self.GeneralPlotOnIteration[AlgID].setObjectName("PlotOnIteration")
            # self.GeneralPlotOnIteration[AlgID].setFixedWidth(100)
            self.GeneralPlotOnIteration[AlgID].setToolTip('Create plot for each iteration?')
            self.GeneralPlotOnIteration[AlgID].clicked.connect(self.CheckBoxWidget)
            self.GeneralPlotOnIteration[AlgID].setEnabled(True)
            layout.addWidget(self.GeneralPlotOnIteration[AlgID], NumberElements, 1)
            NumberElements += 1
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write algorithm settings to xml file
    def WriteAlgXMLFile(self):
        """

    input parameters:
    -----------------

        - None:


    output parameters:
    ------------------

        - None:
        """


        ## define new xml file name
        if (self.AlgorithmXMLFileName.strip() != ""):
            self.NewAlgXMLFileName = self.GUIJobDir + self.AlgorithmXMLFileName.split("/")[-1]
        else:
            self.NewAlgXMLFileName = self.GUIJobDir + "algorithm.xml"

        # Debug:
        # print ("self.NewAlgXMLFileName = ", self.NewAlgXMLFileName)


        ## open file
        NewAlgXMLFile = open(self.NewAlgXMLFileName, 'w')
        NewAlgXMLFile.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
        NewAlgXMLFile.write("<FitControl>\n")
        NewAlgXMLFile.write("    <!-- settings for fit process -->\n\n\n")
        NewAlgXMLFile.write("    <!-- set number of used algorithms -->\n")
        NewAlgXMLFile.write("    <NumberOfFitAlgorithms>" + str(len(self.GlobalAlgSettings)) + "</NumberOfFitAlgorithms>\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## fill values to elements
        for LocalAlgSettingsID, InterAlgSettings in enumerate(self.GlobalAlgSettings):
            LocalAlgSettings = InterAlgSettings[self.GlobalActiveAlgSettingsID[LocalAlgSettingsID]]


            ## get parameters for current algorithm
            LocalAlg = LocalAlgSettings[0].lower()                                          ## get name of current algorithm
            LocalSpecialSettings = LocalAlgSettings[1]                                      ## get special settings for current algorithm
            self.LocalMaxIter = int(LocalAlgSettings[2])                                    ## get max. number of iterations
            self.LoaclNumProc = LocalAlgSettings[3]                                         ## get number of cores
            self.LocalHostFileName = LocalAlgSettings[4]                                    ## get path and name of host file
            self.LocalLimitChi2 = LocalAlgSettings[5]                                       ## get lower limit of chi^2
            self.LocalRenormalizedChi2Flag = LocalAlgSettings[6]                            ## get flag for usage of renormalized chi^2
            self.LocalDeterminationChi2 = LocalAlgSettings[7]                               ## get method for chi^2 determination
            self.LocalSaveChi2Flag = LocalAlgSettings[8]                                    ## get flag for saving chi^2 values
            self.LocalTitleXAxis = LocalAlgSettings[9]                                      ## get title for x axis
            self.LocalTitleYAxis = LocalAlgSettings[10]                                     ## get title for y axis
            self.LocalPlotOnIterFlag = LocalAlgSettings[11]                                 ## get flag for create plot for each iteration


            ## enter name of algorithm
            LineSting = "=" * 141
            NewAlgXMLFile.write("\n\n    <!-- " + LineSting + " -->\n")
            NewAlgXMLFile.write("    <!-- describe algorithm " + str(LocalAlgSettingsID + 1) + " -->\n")
            NewAlgXMLFile.write("    <algorithm>\n")
            NewAlgXMLFile.write("        <FitAlgorithm>" + LocalAlgSettings[0].strip() + "</FitAlgorithm>\n\n\n")


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## include special settings for algorithm here
            LineSting = "*" * 137


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## Levenberg-Marquardt
            if (LocalAlg == "levenberg-marquardt"):
                NewAlgXMLFile.write("        <!-- " + LineSting + " -->\n")
                NewAlgXMLFile.write("        <!-- special settings for Levenberg-Marquardt algorithm -->\n\n\n")


                ## variation value
                self.LocalLMVarValue = LocalSpecialSettings[0]
                NewAlgXMLFile.write("        <!-- define value of the variation -->\n")
                self.LocalLMVarValue = str(self.LocalLMVarValue).strip()
                if (self.LocalLMVarValue != ""):
                    NewAlgXMLFile.write("        <VariationValue>" + self.LocalLMVarValue + "</VariationValue>\n\n\n")
                else:
                    NewAlgXMLFile.write("        <VariationValue>" + str(self.DefaultLMVarVal) + "</VariationValue>\n\n\n")


                ## implementation
                if (self.ChooseImplementationFlag):
                    try:
                        self.LocalLMMethod = LocalSpecialSettings[1]
                    except:
                        self.LocalLMMethod = "NR"
                    NewAlgXMLFile.write("        <!-- define method used for Levenberg-Marquardt -->\n")
                    NewAlgXMLFile.write("        <MethodLM>nr</MethodLM>\n\n\n")


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## Simulated-Annealing
            elif (LocalAlg == "simulated-annealing"):
                NewAlgXMLFile.write("        <!-- " + LineSting + " -->\n")
                NewAlgXMLFile.write("        <!-- special settings for Simulated-Annealing algorithm -->\n\n\n")


                ## temperature
                self.LocalSATemp = LocalSpecialSettings[0]
                NewAlgXMLFile.write("        <!-- define temperature (value of variation) > 0 -->\n")
                NewAlgXMLFile.write("        <Temperature>" + str(self.LocalSATemp) + "</Temperature>\n\n\n")


                ## reduction coef.
                self.LocalSATempRedCoef = LocalSpecialSettings[1]
                NewAlgXMLFile.write("        <!-- define TemperatureReductionKoeff (reduction coef. for the temp. for each reduction cycle) -->\n")
                NewAlgXMLFile.write("        <TemperatureReductionKoeff>" + str(self.LocalSATempRedCoef) + "</TemperatureReductionKoeff>\n\n\n")


                ## number of reduction
                self.LocalSANumRed = LocalSpecialSettings[2]
                NewAlgXMLFile.write("        <!-- define NumberOfReduction (number of reduction cycle) > 0 -->\n")
                NewAlgXMLFile.write("        <NumberOfReduction>" + str(self.LocalSANumRed) + "</NumberOfReduction>\n\n\n")


                ## number of reheating phases
                self.LocalSANumReHeat = LocalSpecialSettings[3]
                NewAlgXMLFile.write("        <!-- define NumberOfReheatingPhases (number of reheating phases) > 0 -->\n")
                NewAlgXMLFile.write("        <NumberOfReheatingPhases>" + str(self.LocalSANumReHeat) + "</NumberOfReheatingPhases>\n\n\n")


                ## implementation
                if (self.ChooseImplementationFlag):
                    try:
                        self.LocalSAMethod = LocalSpecialSettings[4]
                    except:
                        self.LocalSAMethod = "NR"
                    NewAlgXMLFile.write("        <!-- define method used for Simulated-Annealing algorithm -->\n")
                    NewAlgXMLFile.write("        <MethodSA>" + self.LocalSAMethod + "</MethodSA>\n\n\n")


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## Bees
            elif (LocalAlg == "bees"):
                NewAlgXMLFile.write("        <!-- " + LineSting + " -->\n")
                NewAlgXMLFile.write("        <!-- special settings for Bees algorithm -->\n\n\n")


                ## best sites
                NewAlgXMLFile.write("        <!-- define number of best sites > 0 -->\n")
                self.LocalBeesNumBestSites = LocalSpecialSettings[0]
                NewAlgXMLFile.write("        <BestSiteCounter>" + str(self.LocalBeesNumBestSites) + "</BestSiteCounter>\n\n\n")


                ## number of bees
                NewAlgXMLFile.write("        <!-- define NumberBees: number of bees: automatic (default), > 0 -->\n")
                self.LocalBeesNumParticles = LocalSpecialSettings[1]
                if (self.LocalBeesNumParticles == (-1)):
                    NewAlgXMLFile.write("        <NumberBees>automatic</NumberBees>\n\n\n")
                else:
                    NewAlgXMLFile.write("        <NumberBees>" + str(self.LocalSATempRedCoef) + "</NumberBees>\n\n\n")


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## PSO
            elif (LocalAlg == "pso"):
                NewAlgXMLFile.write("        <!-- " + LineSting + " -->\n")
                NewAlgXMLFile.write("        <!-- special settings for PSO algorithm -->\n\n\n")


                ## best sites
                self.LocalPSONumBestSites = LocalSpecialSettings[0]
                NewAlgXMLFile.write("        <!-- define number of best sites > 0 -->\n")
                NewAlgXMLFile.write("        <BestSiteCounter>" + str(self.LocalPSONumBestSites) + "</BestSiteCounter>\n\n\n")


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## Genetic
            elif (LocalAlg == "genetic"):
                NewAlgXMLFile.write("        <!-- " + LineSting + " -->\n")
                NewAlgXMLFile.write("        <!-- special settings for Genetic algorithm -->\n\n\n")


                ## best sites
                NewAlgXMLFile.write("        <!-- define number of best sites > 0 -->\n")
                self.LocalGeneticNumBestSites = LocalSpecialSettings[0]
                try:
                    self.LocalGeneticNumBestSites = self.LocalGeneticNumBestSites.strip()
                except:
                    pass
                if (self.LocalGeneticNumBestSites != ""):
                    NewAlgXMLFile.write("        <BestSiteCounter>" + str(self.LocalGeneticNumBestSites) + "</BestSiteCounter>\n\n\n")
                else:
                    NewAlgXMLFile.write("        <BestSiteCounter>1</BestSiteCounter>\n\n\n")


                ## number of chromosomes
                NewAlgXMLFile.write("        <!-- define number of chromosomes -->\n")
                self.LocalGeneticNumParticles = LocalSpecialSettings[1]
                if (self.LocalGeneticNumParticles == (-1)):
                    NewAlgXMLFile.write("        <NumberChromosomes>automatic</NumberChromosomes>\n\n\n")
                else:
                    NewAlgXMLFile.write("        <NumberChromosomes>" + str(self.LocalGeneticNumParticles) + "</NumberChromosomes>\n\n\n")


                ## new range flag
                NewAlgXMLFile.write("        <!-- use new ranges: True, False (default) -->\n")
                self.GeneticNewRangeFlag = LocalSpecialSettings[2]
                if (self.GeneticNewRangeFlag):
                    NewAlgXMLFile.write("        <UseNewRange>True</UseNewRange>\n\n\n")
                else:
                    NewAlgXMLFile.write("        <UseNewRange>False</UseNewRange>\n\n\n")


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## Nested-Sampling
            elif (LocalAlg == "nested-sampling"):
                NewAlgXMLFile.write("        <!-- " + LineSting + " -->\n")
                NewAlgXMLFile.write("        <!-- special settings for Nested-Sampling algorithm -->\n\n\n")


                ## best sites
                NewAlgXMLFile.write("        <!-- define number of best sites > 0 -->\n")
                self.LocalNSNumBestSites = LocalSpecialSettings[0]
                if (self.LocalNSNumBestSites != ""):
                    NewAlgXMLFile.write("        <BestSiteCounter>" + str(self.LocalNSNumBestSites) + "</BestSiteCounter>\n\n\n")
                else:
                    NewAlgXMLFile.write("        <BestSiteCounter>1</BestSiteCounter>\n\n\n")


                ## number of objects
                NewAlgXMLFile.write("        <!-- define number of objects -->\n")
                self.LocalNSNumParticles = LocalSpecialSettings[1]
                if (self.LocalNSNumParticles == (-1)):
                    NewAlgXMLFile.write("        <NumberObjects>automatic</NumberObjects>\n\n\n")
                else:
                    NewAlgXMLFile.write("        <NumberObjects>" + str(self.LocalGeneticNumParticles) + "</NumberObjects>\n\n\n")


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## MCMC
            elif (LocalAlg == "mcmc"):
                NewAlgXMLFile.write("        <!-- " + LineSting + " -->\n")
                NewAlgXMLFile.write("        <!-- special settings for MCMC algorithm -->\n\n\n")


                ## best sites
                NewAlgXMLFile.write("        <!-- define number of best sites > 0 -->\n")
                self.LocalMCMCNumBestSites = LocalSpecialSettings[0]
                if (self.LocalMCMCNumBestSites != ""):
                    NewAlgXMLFile.write("        <BestSiteCounter>" + str(self.LocalMCMCNumBestSites) + "</BestSiteCounter>\n\n\n")
                else:
                    NewAlgXMLFile.write("        <BestSiteCounter>1</BestSiteCounter>\n\n\n")


                ## number of samplers
                NewAlgXMLFile.write("        <!-- define number of samplers -->\n")
                self.LocalMCMCNumParticles = LocalSpecialSettings[1]
                if (self.LocalMCMCNumParticles == (-1)):
                    NewAlgXMLFile.write("        <NumberMCMCSampler>automatic</NumberMCMCSampler>\n\n\n")
                else:
                    NewAlgXMLFile.write("        <NumberMCMCSampler>" + str(self.LocalMCMCNumParticles) + "</NumberMCMCSampler>\n\n\n")


                ## number of iterations within burn-in phase
                NewAlgXMLFile.write("        <!-- define number of iterations within burn-in phase -->\n")
                self.LocalMCMCBurnInIter = LocalSpecialSettings[2]
                if (self.LocalMCMCBurnInIter != ""):
                    NewAlgXMLFile.write("        <NumberBurnInIter>" + str(self.LocalMCMCBurnInIter) + "</NumberBurnInIter>\n\n\n")
                else:
                    NewAlgXMLFile.write("        <NumberBurnInIter>1</NumberBurnInIter>\n\n\n")


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## Interval-NS
            elif (LocalAlg == "interval-ns" or LocalAlg == "ins"):
                NewAlgXMLFile.write("        <!-- " + LineSting + " -->\n")
                NewAlgXMLFile.write("        <!-- special settings for INS algorithm -->\n\n\n")


                ## best sites
                NewAlgXMLFile.write("        <!-- define number of best sites > 0 -->\n")
                self.LocalINSNumBestSites = LocalSpecialSettings[0]
                if (self.LocalINSNumBestSites != ""):
                    NewAlgXMLFile.write("        <BestSiteCounter>" + str(self.LocalINSNumBestSites) + "</BestSiteCounter>\n\n\n")
                else:
                    NewAlgXMLFile.write("        <BestSiteCounter>1</BestSiteCounter>\n\n\n")


                ## vol_bound
                NewAlgXMLFile.write("        <!-- define vol_bound = 1e-5, .., 0.1 -->\n")
                self.LocalINSVolBond = LocalSpecialSettings[1]
                NewAlgXMLFile.write("        <vol_bound>" + str(self.LocalMCMCBurnInIter) + "</vol_bound>\n\n\n")


                ## delta_incl
                NewAlgXMLFile.write("        <!-- define delta_incl = 1e-5, .., 0.5 (default 0.001 -->\n")
                self.LocalINSDeltaIncl = LocalSpecialSettings[2]
                NewAlgXMLFile.write("        <delta_incl>" + str(self.LocalMCMCBurnInIter) + "</delta_incl>\n\n\n")


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## AdditionalPackages
            elif (LocalAlg == "additionalpackages"):
                NewAlgXMLFile.write("        <!-- " + LineSting + " -->\n")
                NewAlgXMLFile.write("        <!-- special settings for AP algorithm -->\n\n\n")


                ## get scipy algorithm
                NewAlgXMLFile.write("        <!-- define scipy algorithm -->\n")
                NewAlgXMLFile.write("        <minAlgorithm>" + LocalSpecialSettings[0] + "</minAlgorithm>\n\n\n")


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## Error estimation
            elif (LocalAlg == "errorestim" or LocalAlg == "errorestim_ins"):
                NewAlgXMLFile.write("        <!-- " + LineSting + " -->\n")
                NewAlgXMLFile.write("        <!-- special settings for error estimation algorithm -->\n\n\n")


                ## method for error estimation
                NewAlgXMLFile.write("        <!-- define method for error estimation -->\n")
                NewAlgXMLFile.write("        <minAlgorithm>" + LocalSpecialSettings[0] + "</minAlgorithm>\n\n\n")
                self.LocalErrorEstimMethodID = self.ErrorEstimLocalmethodList.index(LocalSpecialSettings[0])


                ## add additional elements for MCMC
                if (self.LocalErrorEstimMethodID == 0):                                 ## MCMC


                    ## number of samplers
                    NewAlgXMLFile.write("        <!-- define number of samplers used for error estimation -->\n")
                    NewAlgXMLFile.write("        <NumberMCMCSampler>" + str(LocalSpecialSettings[1]) + "</NumberMCMCSampler>\n\n\n")


                    ## number of samplers
                    NewAlgXMLFile.write("        <!-- define number of iterations within burn-in phase -->\n")
                    NewAlgXMLFile.write("        <NumberBurnInIter>" + str(LocalSpecialSettings[2]) + "</NumberBurnInIter>\n\n\n")


                    ## multiplicity of standard deviation
                    if (len(LocalSpecialSettings) > 2):
                        NewAlgXMLFile.write("        <!-- define multiplicity of standard deviation (1sigma, 2sigma, 3sigma, etc.) -->\n")
                        NewAlgXMLFile.write("        <MultiplicitySigma>" + str(LocalSpecialSettings[3]) + "</MultiplicitySigma>\n\n\n")


                    ## multiplicity of standard deviation
                    if (len(LocalSpecialSettings) > 3):
                        NewAlgXMLFile.write("        <!-- define if previous calculated parameter vectors are used or not -->\n")
                        LocalFlag = str(LocalSpecialSettings[4]).strip()
                        if (LocalFlag in self.SetListStrings):
                            NewAlgXMLFile.write("        <UsePrevResults>True</UsePrevResults>\n\n\n")
                        else:
                            NewAlgXMLFile.write("        <UsePrevResults>False</UsePrevResults>\n\n\n")


                ## add additional elements for Fisher
                elif (self.LocalErrorEstimMethodID == 1):                               ## Fisher


                    ## number of samplers
                    NewAlgXMLFile.write("        <!-- define number of samplers used for error estimation -->\n")
                    NewAlgXMLFile.write("        <VariationValue>" + str(LocalSpecialSettings[1]) + "</VariationValue>\n\n\n")


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## define general settings
            NewAlgXMLFile.write("        <!-- " + LineSting + " -->\n")
            NewAlgXMLFile.write("        <!-- general settings -->\n\n\n")
            NewAlgXMLFile.write("        <!-- set max. number of iterations -->\n")
            NewAlgXMLFile.write("        <number_iterations>" + str(self.LocalMaxIter) + "</number_iterations>\n\n\n")
            NewAlgXMLFile.write("        <!-- set max. number of processors -->\n")
            NewAlgXMLFile.write("        <NumberProcessors>" + str(self.LoaclNumProc) + "</NumberProcessors>\n\n\n")
            NewAlgXMLFile.write("        <!-- define path and name of host file -->\n")
            NewAlgXMLFile.write("        <MPIHostFileName>" + self.LocalHostFileName + "</MPIHostFileName>\n\n\n")
            NewAlgXMLFile.write("        <!-- settings for chi^2 -->\n")
            NewAlgXMLFile.write("        <limit_of_chi2>" + str(self.LocalLimitChi2) + "</limit_of_chi2>\n")
            if (self.LocalRenormalizedChi2Flag):
                NewAlgXMLFile.write("        <RenormalizedChi2>True</RenormalizedChi2>\n")
            else:
                NewAlgXMLFile.write("        <RenormalizedChi2>False</RenormalizedChi2>\n")
            if (self.ShowChi2Methods):
                NewAlgXMLFile.write("        <DeterminationChi2>" + self.MethodChi2LocalAlgList[self.LocalDeterminationChi2] + "</DeterminationChi2>\n")
            if (self.LocalSaveChi2Flag):
                NewAlgXMLFile.write("        <SaveChi2>True</SaveChi2>\n")
            else:
                NewAlgXMLFile.write("        <SaveChi2>False</SaveChi2>\n")
            if (self.ShowGraphicOptions):
                NewAlgXMLFile.write("\n\n        <!-- set plot options -->\n")
                NewAlgXMLFile.write("        <PlotAxisX>" + self.LocalTitleXAxis + "</PlotAxisX>\n")
                NewAlgXMLFile.write("        <PlotAxisY>" + self.LocalTitleYAxis + "</PlotAxisY>\n")
                if (self.LocalPlotOnIterFlag):
                    NewAlgXMLFile.write("        <PlotIteration>True</PlotIteration>\n")
                else:
                    NewAlgXMLFile.write("        <PlotIteration>False</PlotIteration>\n")
            NewAlgXMLFile.write("    </algorithm>\n")
        NewAlgXMLFile.write("</FitControl>\n")


        ## close algorithm xml file
        NewAlgXMLFile.close()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get path and name of new algorithm xml file
    def getNewAlgorithmXMLFileName(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - self.AlgXMLFileName:      path and name of new algorithm xml file
        """


        ## we're done
        return self.NewAlgXMLFileName
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## select algorithm and edit settings
##
def SelectAlgorithm(GUIJobDir, UserMaxIterLM, AlgXMLFileName):
    """

input parameters:
-----------------

    - GUIJobDir:            path of job directory

    - UserMaxIterLM:        max. number of iterations for Levenberg-Marquardt algorithm

    - AlgXMLFileName:       path and name of algorithm xml file


output parameters:
------------------

    - ok:                   status flag

    - NewMaxIteration:      new max. iteration
    """

    # Debug:
    # print ("GUIJobDir = ", GUIJobDir)
    # print ("UserMaxIterLM = ", UserMaxIterLM)
    # print ("AlgXMLFileName = ", AlgXMLFileName)


    ## suppress warnings
    warnings.simplefilter('once', UserWarning)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    ok = 0
    NewMaxIteration = (-1)



    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start GUI
    AlgorithmSelectionClass = AlgorithmGUI(GUIJobDirIn = GUIJobDir, UserMaxIterLMIn = UserMaxIterLM, AlgorithmXMLFileNameIn = AlgXMLFileName)


    ## start GUI
    AlgorithmSelectionClass.exec_()
    # AlgorithmSelectionClass.show()
    # sys.exit(AlgorithmSelectionClass.exec_())


    ## get path and name of new algorithm xml file
    AlgXMLFileName = AlgorithmSelectionClass.getNewAlgorithmXMLFileName()


    ## define return parameters
    return AlgXMLFileName
##--------------------------------------------------------------------------------------------------------------------------------------------------------

