#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  GUI for manual fit parameter estimation (compatible with XCLASS version 1.2.6 and higher)
##
##
##      PUBLIC VERSION
##
##
##
##  Usage:
##
##      python3 XCLASS-GUI.py --molfit=. --obs=. --alg=.
##
##
##  Command linie parameter:
##
##      - "molfit":             defines path and name of molfit file
##
##      - "obs":                defines path and name of obs. xml file
##
##      - "alg":                defines path and name of algorithm xml file
##
##
##
##
##  Copyright (C) 2016 - 2024  Thomas Moeller
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included:
##
##
##  Class Main:
##  -----------
##
##      - function Main.__init__:                       initialize class Main
##
##      - function Main.ContinuumFunction:              continuum function
##      - function Main.EstimateContinuum:              estimate continuum
##      - function Main.UpdateGeneralParameters:        update general parameter arrays
##      - function Main.QuitGUI:                        quit GUI
##      - function Main.PrevMolecule:                   go to previous molecule
##      - function Main.NextMolecule:                   go to next molecule
##      - function Main.buttonClicked:                  subroutine for check box events
##      - function Main.FitCore:                        start myXCLASSFit
##      - function Main.AllTabChanged:                  define what happens if tab was changed
##      - function Main.UpdateParamList:                update value in parameter list
##      - function Main.GetPartitionFunction:           calculate partition function for T = T_rot
##      - function Main.GetShiftedFreq:                 compute shifted frequency ranges
##
##      - function Main.MenuSaveCurrMolfitFile:         save current molfit file
##      - function Main.MenuLoadIsoRatioFile:           reload iso ratio file
##      - function Main.MenuSaveIsoRatioFile:           save current iso ratio file
##      - function Main.MenuAnalyzeMolfitFile:          analyze molfit file
##      - function Main.MenuAddMolecule:                add molecule
##      - function Main.MenuRemoveMolecule:             remove current molecule
##      - function Main.MenueTFW:                       call TFW GUI
##      - function Main.MenuExternalWidget:             call external widgets
##      - function Main.MenuSortSpectra:                sort transition frequency spectra by different criteria
##      - function Main.MenuCreateObsXMLFile:           create obs. xml file for current fit
##      - function Main.MenuFit:                        fit current molecule
##      - function Main.MenuSnapshot:                   create snapshots of each transition window spectrum and save plot to file
##      - function Main.MenuHelpMe:                     display some help information
##      - function Main.MenuAboutXCLASSGUI:             print some general informations about XCLASS GUI
##
##      - function Main.Tab1GetMoleculesFromDatabase:   get list of all molecules
##      - function Main.Tab1setDisEnabled:              enables or disables elements of tab 1
##      - function Main.Tab1ImportObsXMLFile:           get parameters from obs. xml file
##      - function Main.Tab1selectFile:                 subroutine for file dialog of tab 1
##      - function Main.Tab1LineEdit:                   subroutine for LineEdit text events
##      - function Main.Tab1SetValues:                  set all values in tab 1
##      - function Main.Tab1checkBox:                   subroutine for check box events of tab 1
##      - function Main.Tab1comboSelected:              subroutine for combo box event of tab 1
##      - function Main.Tab1selectFreqRange:            subroutine for selecting frequency range in tab 1
##      - function Main.Tab1SelectAlgorithm:            subroutine for selecting algorithm in tab 1
##      - function Main.Tab1CreateUserExpXMLFile:       creates the experimental xml file in the current job directory
##
##      - function Main.Tab2ImportFile:                 import ASCII file
##      - function Main.Tab2setDisEnabled:              enables or disables elements of tab 2
##      - function Main.Tab2ChangeTab:                  check, if all parameters from tab 1 are ok and analyze molfit and obs. xml file
##      - function Main.Tab2CreateXMLFile:              create new xml file
##      - function Main.Tab2NewXMLFile:                 create new xml file and determine transition frequency window spectra
##      - function Main.Tab2SetValuesForTab:            set all values in tab 2
##      - function Main.Tab2GetParameterFromMolfitFile: get molfit parameter from file
##      - function Main.Tab2CreateLineForMolfitFile:    create new line for molfit file from parameter list
##      - function Main.Tab2CreateMolfitFile:           create molfit file from parameter list
##      - function Main.Tab2CreateIsoRatioFileContents: create contents of iso ratio file
##      - function Main.Tab2CreatePlots:                create plots
##      - function Main.Tab2CalculateXCLASS:            calculate XCLASS spectra
##      - function Main.Tab2UpdateHistoryArray:         update history arrays
##      - function Main.Tab2molfitListEntrySelected:    get current component
##      - function Main.Tab2radioSelected:              subroutine for radio box events
##      - function Main.Tab2checkBoxSelected:           subroutine for check box events
##      - function Main.Tab2checkActive:                subroutine for active check box event
##      - function Main.Tab2checkBackground:            compute background spectrum
##      - function Main.Tab2GetEmAbsFunc:               get emission / absorption function
##      - function Main.Tab2checkrmCont:                subroutine for remove continuum check box event
##      - function Main.Tab2checkUseIso:                subroutine for (de-)activate usage of isotopologues
##      - function Main.Tab2comboSelected:              subroutine for usage of combo box
##      - function Main.Tab2SliderValueChange:          subroutine for slider events
##      - function Main.Tab2EditSelected:               subroutine for text edit events
##      - function Main.Tab2GetIsoRatiosFromFile:       get iso ratios from file
##      - function Main.Tab2SetCurrIsoIndex:            set current iso index for iso combo box
##
##      - function Main.Tab3ChangeTab:                  define what happens if tab 3 is selected
##      - function Main.Tab3setDisEnabled:              enables or disables elements of tab 3
##      - function Main.Tab3SetValuesForTab:            set all values in tab 3
##      - function Main.Tab3comboSelected:              subroutine for usage of combo box
##      - function Main.Tab3molfitListEntrySelected:    get current component
##      - function Main.Tab3checkBoxSelected:           check box event for tab 3
##      - function Main.Tab3FitAll:                     fit all molecules and isotopologues
##
##
##  Class ProgressBar:
##  ------------------
##
##      - function ProgressBar.__init__:                initialize class ProgressBar
##      - function ProgressBar.handleButton:            define button action
##      - function ProgressBar.setValue:                set value of progress bar
##      - function ProgressBar.setBodyText              set text of body
##
##
##  Class ExternalWidget:
##  ---------------------
##
##      - function ExternalWidget.__init__:             initialize class ExternalWidget
##      - function ExternalWidget.ButtonWidget:         handle button event of widget
##      - function ExternalWidget.LineEditWidget:       handle lineEdit event of widget
##      - function ExternalWidget.getReturnparamWidget: get return parameters of widget
##
##
##  Main Program:
##  -------------
##
##      - function StartXCLASSGUI:                      start XCLASS-GUI
##      - function Main:                                start GUI
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2016-04-15      initial version
##  T. Moeller      2016-09-24      improved version
##  T. Moeller      2017-02-10      improved Tab1 and RangeSelection module
##  T. Moeller      2017-02-21      improved command line argument handling
##  T. Moeller      2017-11-07      applied changes related to XCLASS release 1.2.5
##  T. Moeller      2018-01-07      add new continuum contributions
##  T. Moeller      2018-05-01      applied changes related to XCLASS release 1.2.6
##  T. Moeller      2018-10-31      modified handling of layout file
##  T. Moeller      2019-09-16      add handling of contiuum contributions and layer stacking
##  T. Moeller      2019-12-12      add handling of emission / absorption function
##  T. Moeller      2020-01-08      porting to python 3.x
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import sys                                                                                  ## import sys package
import os                                                                                   ## import os package
import math                                                                                 ## import math package
import numpy as np                                                                          ## import numpy package
import time                                                                                 ## import time package
import sqlite3                                                                              ## import sqlite3 package
import warnings                                                                             ## import warnings package
import copy                                                                                 ## import module copy
from PyQt5.uic import loadUiType                                                            ## import module loadUiType from PyQt5 package
from PyQt5 import (QtCore, QtGui, QtWidgets, uic)
from matplotlib.figure import Figure                                                        ## import module Figure from matplotlib package
from matplotlib.backends.backend_qt5agg import (FigureCanvasQTAgg as FigureCanvas, NavigationToolbar2QT as NavigationToolbar)
import matplotlib.pyplot as plt                                                             ## import plt package
import scipy                                                                                ## import from scipy interpolate package
import pylab                                                                                ## import pylab package
warnings.filterwarnings("ignore")
from xclass import task_MAGIX
from xclass import task_myXCLASS
from xclass import task_myXCLASSFit
from xclass import task_myXCLASSMapFit
from xclass import task_LineIdentification
from xclass.tools.XCLASSGUI.gui.extensions import AlgorithmGUI
from xclass.tools.XCLASSGUI.gui.extensions import RangeSelection
from xclass.lib import xclasslite
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## Main Class
class Main(QtWidgets.QMainWindow):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize main class
    def __init__(self, UiFileName, XCLASSRootDir, UserMolfitsFileName, UserObsXMLFilename, UserAlgXMLFile, devFlag):
        """

    input parameters:
    -----------------

        - UiFileName:           path and name of ui-layout file

        - XCLASSRootDir:        path of XCLASS package

        - UserMolfitsFileName:  path and name of user defined molfit file

        - UserObsXMLFilename:   path and name of user defined obs. xml file

        - UserAlgXMLFile:       path and name of user defined algorithm xml file

        - devFlag:              flag indicating developer version


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("UiFileName = ", UiFileName)
        # print ("XCLASSRootDir = ", XCLASSRootDir)
        # print ("UserMolfitsFileName = ", UserMolfitsFileName)
        # print ("UserObsXMLFilename = ", UserObsXMLFilename)
        # print ("UserAlgXMLFile = ", UserAlgXMLFile)
        # print ("devFlag = ", devFlag)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## initialize GUI parameters
        super(Main, self).__init__()
        uic.loadUi(UiFileName, self)
        self.tabWidget.setCurrentIndex(0)                                                   ## set current tab
        self.XCLASSRootDir = XCLASSRootDir                                                  ## save XCLASS root directory
        self.DefaultDB = task_myXCLASS.GetDefaultDBFile()                                   ## define path and name of default database file
        self.GUITitle = "XCLASS-GUI"                                                        ## set title of GUI window
        self.setWindowTitle(self.GUITitle)
        self.devFlag = devFlag


        ## set default parameter values
        self.LastEvent = ""                                                                 ## define last event
        self.SpectrumIndexList = []                                                         ## reset variable SpectrumIndexList
        self.canvas = []                                                                    ## reset list for canvas elements
        self.ReturnParameter = 0                                                            ## define return parameter for GUI
        self.startFlag = True                                                               ## define start flag as true
        self.lock = 0                                                                       ## used to lock tab
        self.InitFlag = True                                                                ## set initialize flag
        self.dbgFlag = False                                                                ## set debug flag
        self.OnceFlag = True                                                                ## set "once" flag
        self.SnapshotPlotFlag = True                                                        ## set "snapshot" flag
        self.BackgroundFlag = False                                                         ## set "background" flag
        self.CFUsageFlag = True                                                             ## set "CFUsage" flag
        self.AllInOneWindowFlag = True                                                      ## set "all in one window" flag
        self.CompNumChangeFlag = False                                                      ## set "component has changed" flag
        self.MaxNumberSpectrumDefault = 6                                                   ## set default number of plots in main GUI
        self.MinNumTransitionsSQL = 1                                                       ## define min. number of transitions
        self.MaxNumTransInFit = 10000                                                       ## define max. number of transitions
        self.ElowMax = 5.e9                                                                 ## define default value for max. lower energy for main GUI
        self.IMin = 1.e-8                                                                   ## define default value for min. intensity for main GUI
        self.velLowLimit = -250.0                                                           ## define default lower limit for vel. range of spectra
        self.velUpLimit = 100.0                                                             ## define default upper limit for vel. range of spectra
        self.velLowLimitWeight = -250.0                                                     ## define default lower limit for vel. range of main GUI
        self.velUpLimitWeight = 100.0                                                       ## define default upper limit for vel. range of main GUI
        self.VelBinExtGUI = 1                                                               ## define default value for vel. resolution (for TWF GUI)
        self.NoiseLevelExtGUI = 5.e-2                                                       ## define default value for noise level (for TWF GUI)
        self.RemoveContFlagGUI = True                                                       ## define default value for "remove cont." flag (for TWF GUI)
        self.SmoothValue = 0.0                                                              ## define default value for smooth flag for main GUI
        self.SmoothedFlagGUI = False                                                        ## define default value for smooth flag (for TWF GUI)
        self.SmoothValueGUI = 0.1                                                           ## define default value for smooth factor (for TWF GUI)
        self.SmoothMethod = "hanning"                                                       ## define method for smoothing (for TWF GUI),
                                                                                            ## ("hanning" or "savgol")

        ## parameters for obs. xml file
        self.ExpFileList = []                                                               ## reset list of path and names of obs. data files
        self.ExpFileNameList = []                                                           ## reset list of obs. data file names
        self.NumberExpRangesList = []                                                       ## reset list of number of frequency ranges
        self.FreqMinList = []                                                               ## reset list of lower frequencies of each range
        self.FreqMaxList = []                                                               ## reset list of upper frequencies of each range
        self.FreqStepList = []                                                              ## reset list of step frequencies of each range
        self.t_back_flagList = []                                                           ## reset list of t_back flags of each range
        self.tBackList = []                                                                 ## reset list of background temperatures of each range
        self.tSlopeList = []                                                                ## reset list of temperatures slopes of each range
        self.N_HList = []                                                                   ## reset list of n_H densities of each range
        self.beta_dustList = []                                                             ## reset list of betas of each range
        self.kappa_1300List = []                                                            ## reset list of kappas of each range
        self.DustFileNameList = []                                                          ## reset list of dust file names
        self.BackgroundFileNameList = []                                                    ## reset list of background file names
        self.ContFuncIDPhenList = []                                                        ## reset list of function ids for phen. cont. description
        self.ContParam1PhenList = []                                                        ## reset list of parameter 1 for phen. cont. description
        self.ContParam2PhenList = []                                                        ## reset list of parameter 2 for phen. cont. description
        self.ContParam3PhenList = []                                                        ## reset list of parameter 3 for phen. cont. description
        self.ContParam4PhenList = []                                                        ## reset list of parameter 4 for phen. cont. description
        self.ContParam5PhenList = []                                                        ## reset list of parameter 5 for phen. cont. description
        self.NoiseList = []                                                                 ## reset list of noise levels of each range
        self.GlobalvLSRList = []                                                            ## reset list of global v_LSR of each data file
        self.TelescopeSizeList = []                                                         ## reset list of telescope sizes of each data file
        self.BMINList = []                                                                  ## reset list of BMIN parameters
        self.BMAJList = []                                                                  ## reset list of BMAJ parameters
        self.BPAList = []                                                                   ## reset list of BPA parameters
        self.RedshiftList = []                                                              ## reset list of redshifts
        self.Inter_FlagList = []                                                            ## reset list of interfer. flags of each data file
        self.ErrorYList = []                                                                ## reset list of error flags of each data file
        self.NumberHeaderLinesList = []                                                     ## reset list of number of header lines of each data file
        self.SeparatorColumnsList = []                                                      ## reset list of separator characters of each data file
        self.IsotopologuesList = []                                                         ## reset list of iso flag
        self.IsoFlag = False                                                                ## set default value of iso flag
        self.IsoTableFileNameList = []                                                      ## reset list of path and name of iso ratio file
        self.dbList = [self.DefaultDB]                                                      ## reset list of path and name of database file
        self.NumModelPixelXXList = []                                                       ## reset list of model pixels along x-direction
        self.NumModelPixelYYList = []                                                       ## reset list of model pixels along y-direction
        self.LocalOverlapFlagList = []                                                      ## reset list of local-overlap flag
        self.UserLocalOverlapFlag = False                                                   ## set local overlap flag
        self.UserLocalOverlapFlagDefault = False                                            ## set default value for local-overlap flag
        self.NoSubBeamFlagList = []                                                         ## reset list of no-sub-beam flags
        self.EmAbsFlag = True                                                               ## set flag for em and abs. function usage
        self.EmAbsPATHList = []                                                             ## reset list of emission / absorption function directory
        self.UserEmAbsPATH = None                                                           ## set EmAbsPATH
        self.UserEmAbsPATHDefault = None                                                    ## set default value for EmAbsPATH
        self.UserNoSubBeamFlag = None                                                       ## set no-sub-beam flag
        self.UserNoSubBeamFlagDefault = False                                               ## set default value for no-sub-beam flag


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## initialize parameters for tab 1
        self.MolfitFileName = UserMolfitsFileName.strip()                                   ## set path and name of molfit file
        self.ObsXMLFileName = UserObsXMLFilename.strip()                                    ## set path and name of obs. xml file
        self.AlgXMLFileName = UserAlgXMLFile.strip()                                        ## set path and name of algorithm xml file
        self.UserCurrentMolecule = ""                                                       ## reset name of current molecule
        self.UserCurrentMoleculeID = 0                                                      ## reset index of current molecule
        self.ObsDataFileID = 0                                                              ## id of current obs. data file
        self.DataFileNameList = []                                                          ## list of obs. data file names
        self.FrequencyRangeID = 0                                                           ## set current id of frequency range
        self.FrequencyRange = []                                                            ## frequency range string for combo box
        self.UserObsDataFileName = [""]                                                     ## reset path and name of obs. data file
        self.UserNumberExpRanges = [[0, 1]]                                                 ## reset user input for number frequency ranges
        self.UserNumberExpRangesDefault = 1                                                 ## set default value for number of frequency ranges
        self.UserFreqMin = [[0, 0, None]]                                                   ## reset user input for lower frequency
        self.UserFreqMinDefault = 1.0                                                       ## set default value for user input for lower frequency
        self.UserFreqMax = [[0, 0, None]]                                                   ## reset user input for upper frequency
        self.UserFreqMaxDefault = 1.e10                                                     ## set default value for user input for upper frequency
        self.UserFreqStep = [[0, 0, None]]                                                  ## reset user input for step frequency
        self.UserFreqStepDefault = 1.0                                                      ## set default value for user input for step frequency
        self.UserTbgFlag = [[0, 0, None]]                                                   ## reset user input of T_Back flag
        self.UserTbgFlagDefault = True                                                      ## set default value for user input for T_Back flag
        self.UserTbg = [[0, 0, None]]                                                       ## reset user input for background temperature
        self.UserTbgDefault = 0.0                                                           ## set default value for user input background temperature
        self.UserTslope = [[0, 0, None]]                                                    ## reset user input for temperature slope
        self.UserTslopeDefault = 0.0                                                        ## set default value for user input temperature slope
        self.UsernHFlag = [[0, 0, None]]                                                    ## reset user input of nH flag
        self.UsernHFlagDefault = True                                                       ## set default value for user input for nH flag
        self.UsernH = [[0, 0, None]]                                                        ## reset user input for nH
        self.UsernHDefault = 1.e24                                                          ## set default value for user input nH
        self.Userkappa = [[0, 0, None]]                                                     ## reset user input for kappa
        self.UserkappaDefault = 0.42                                                        ## set default value for user input kappa
        self.Userbeta = [[0, 0, None]]                                                      ## reset user input for spectral index
        self.UserbetaDefault = 2.0                                                          ## set default value for user input for spectral index
        self.UserDustFileName = [[0, 0, None]]                                              ## reset user input for dust file
        self.UserDustFileNameDefault = ""                                                   ## set default value for user input for dust file
        self.UserBackgroundFileName = [[0, 0, None]]                                        ## reset user input for dust file
        self.UserBackgroundFileNameDefault = ""                                             ## set default value for user input for dust file
        self.UserContFuncID = [[0, 0, None]]                                                ## reset user input for ContFuncID
        self.UserContFuncIDDefault = None                                                   ## set default value for user input for ContFuncID
        self.UserContParam1 = [[0, 0, None]]                                                ## reset user input for ContParam1
        self.UserContParam1Default = None                                                   ## set default value for user input for ContParam1
        self.UserContParam2 = [[0, 0, None]]                                                ## reset user input for ContParam2
        self.UserContParam2Default = None                                                   ## set default value for user input for ContParam2
        self.UserContParam3 = [[0, 0, None]]                                                ## reset user input for ContParam3
        self.UserContParam3Default = None                                                   ## set default value for user input for ContParam3
        self.UserContParam4 = [[0, 0, None]]                                                ## reset user input for ContParam4
        self.UserContParam4Default = None                                                   ## set default value for user input for ContParam4
        self.UserContParam5 = [[0, 0, None]]                                                ## reset user input for ContParam5
        self.UserContParam5Default = None                                                   ## set default value for user input for ContParam5
        self.UserInterferoFlag = [[0, False]]                                               ## reset user input for interferometer flag
        self.UserInterferoFlagDefault = False                                               ## set default value for user input for interferometer flag
        self.UserSizeTelescope = [[0, None]]                                                ## reset user input for size of telescope
        self.UserSizeTelescopeDefault = None                                                ## set default value for user input for size of telescope
        self.UserBMIN = [[0, None]]                                                         ## reset user input for BMIN
        self.UserBMINDefault = None                                                         ## set default value for user input for BMIN
        self.UserBMAJ= [[0, None]]                                                          ## reset user input for BMAJ
        self.UserBMAJDefault = None                                                         ## set default value for user input for BMAJ
        self.UserBPA = [[0, None]]                                                          ## reset user input for BPA
        self.UserBPADefault = None                                                          ## set default value for user input for BPA
        self.UservLSR = [[0, None]]                                                         ## reset user input for v_LSR
        self.UservLSRDefault = 0.0                                                          ## set default value for user input for v_LSR
        self.UserRedShift = [[0, None]]                                                     ## reset user input for redshift
        self.UserRedShiftDefault = None                                                     ## set default value for user input for redshift
        self.UserUseIsoFlag = None                                                          ## reset user input for usage of iso ratio file
        self.UserUseIsoFlagDefault = False                                                  ## set default value for user input for iso flag
        self.UserIsoRatioFileName = ""                                                      ## reset user input for path and name of iso ratio file
        self.UserMaxIterLM = None                                                           ## reset user input for max. iterations
        self.MolfitFileChangeFlag = True                                                    ## set molfit change flag
        self.ObsXMLFileChangeFlag = True                                                    ## set obs. xml file change flag
        self.AlgXMLFileChangeFlag = True                                                    ## set algorithm xml file change flag
        self.FITSFlag = False                                                               ## FITS cube was imported
        self.FITSSubDir = ""                                                                ## subdirectory used to store files related to FITS import
        self.UserNumModelPixelXX = None                                                     ## reset user input for number of model pixels along x-dir.
        self.UserNumModelPixelXXDefault = None                                              ## set default value for user input for NumModelPixelXX
        self.UserNumModelPixelYY = None                                                     ## reset user input for number of model pixels along y-dir.
        self.UserNumModelPixelYYDefault = None                                              ## set default value for user input for NumModelPixelYY
        self.LocalOverlapFlag = False                                                       ## reset user input for local-overlap flag
        self.LocalOverlapFlagDefault = False                                                ## set default value for local-overlap flag
        self.NoSubBeamFlag = False                                                          ## reset user input for no-sub-beam flag
        self.NoSubBeamFlagDefault = False                                                   ## set default value for no-sub-beam flag


        self.lineEdit_Redshift.setEnabled(False)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## parameters related to molfit file
        self.LocalMolfitFile = []                                                           ## contents of molfit file
        self.MoleculesInMolfitFile = []                                                     ## list of molecules included in molfit file
        self.AllMoleculesMolfitFiles = []                                                   ## list of all molecule molfit parts
        self.MolfitFileAllParameters = []                                                   ## parameters for each molecule in molfit file
        self.CurrentMolecule = ""                                                           ## current molecule displayed in tab 2
        self.CurrentMoleculeID = 0                                                          ## index of current molecule in self.MoleculesInMolfitFile
        self.CurrentMolfitFileContents = []                                                 ## part of molfit file which correspond to current molecule
        self.CurrentMolfitParameter = []                                                    ## parameters for current molecule
        self.CurrentMoleculeDir = ""                                                        ## path and name of current molecule job directory
        self.CurrentComponentIndex = 0                                                      ## index of current component
        self.CurrentMolfitIndex = 1                                                         ## describes current line in self.CurrentMolfitFileContents
                                                                                            ## default: 1, i.e. first line after molecule definition
        self.ActiveCompList = []                                                            ## list of active components of current molecule
        self.MolfitContinuumParameters = []                                                 ## list of contiuum parameters defined in molfit file
        self.DefaultMolfitParameter = [["n",    0.5,   1.e3,  5.e2],
                                       ["y",   2.73, 1000.0,  10.0],
                                       ["y",  1.e12,  1.e19, 1.e15],
                                       ["y",    0.5,   30.0,   2.0],
                                       ["y", -250.0,  100.0,   0.0],
                                       ["c"]]


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## parameters related to iso ratio file
        self.UseIsoFlag = False                                                             ## flag indicating if current molecule has isotopologue
        self.CurrentIsoParam = []                                                           ## list of isotopologues related to the current molecule
        self.AllIsotopologueList = []                                                       ## list of isotopologue parameters for all molecules
        self.CurrentIsoID = 0                                                               ## index of current isotopologue in self.CurrentIsoParam


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## initialize parameters for tab 2
        self.InitTab2Flag = True                                                            ## initialization flag for tab 2
        self.PureMolNameVector = []                                                         ## list of molecule names without dublicates
        self.AllSpectra = []                                                                ## obs. data of transition frequency windows
        self.AllSpectraCopy = None
        self.IndexSortConversionListCopy = None
        self.NewParameterRanges = []
        self.RemoveContinuumFlag = False                                                    ## flag for removing continuum or not
        self.NewExpXMLFileName = ""
        self.SortIndex = 0                                                                  ## defining sort criteria for transition plots
        self.IndexSortConversionList = [[], [], [], []]                                     ## corresponding conversion table


        ## initialize parameters for undo/redo function
        self.UndoFlag = True
        self.OldLocalMolfitFile = []
        self.OldMoleculesInMolfitFile = []
        self.OldAllMoleculesMolfitFiles = []
        self.OldMolfitFileAllParameters = []
        self.OldCurrentMolecule = []
        self.OldCurrentMoleculeID = []
        self.OldParamVectorCounter = 0
        self.OldCurrentMoleculeDir = []
        self.OldMolfitParameter = []
        self.OldAllIsotopologueList = []
        self.OldIsoRatioParameter = []
        self.OldActiveCompList = []


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## initialize parameters for tab 3
        self.OnceFlagTab3 = True                                                            ## once flag
        self.PlotFlagTab3 = True                                                            ## flag for replotting tab 2 starting from tab 1 + 3
        self.InitFlagTab3 = True                                                            ## initiazlization flag for tab 3
        self.LockTab3 = True                                                                ## flag for locking elements in tab 3
        self.ShowAllFlag = False                                                            ## flag indicating that all transitions where shown
        self.CallmyXCLASSFunctionFlag = False                                               ## flag indicating if myXCLASS func. has to be called or not
        self.CurrentObsDataID = 0                                                           ## define index of current obs. data file index
        self.CurrentFreqRangeID = 0                                                         ## define index of current freq. range index
        self.CurrentAllMolCompIndex = 0                                                     ## reset component index of all-molfit file
        self.AllMolCurrentIsoID = 0                                                         ## iso index for all-molfit file
        self.CurrentRangeList = []                                                          ## list of current frequency ranges
        self.AlreadyCalculatedRanges = []                                                   ## list of already calculated obs. data files and ranges
        self.ObsDataList = []                                                               ## list containing obs. data
        self.AllMolfitFileContent = []                                                      ## current content of all molecule molfit file
        self.CurrentAllMolfitIndex = 0                                                      ## position within all molfit file
        self.LastMolfitFileAllParameters = []                                               ## last molfit parameters
        self.LastMoleculesInMolfitFile = []                                                 ## last list of molecules
        self.LastMolfitFile = []                                                            ## contents of last molfit file
        self.LastIsoRatioFileContent = []                                                   ## contents of last iso ratio file
        self.LocalIsotopologuelist = []                                                     ## list of isotopologues for the current molecule
        self.myXCLASSSpectraList = []                                                       ## list containing calcuated spectra for plot
        self.AllTransFreqList = []                                                          ## list of transition frequencies
        self.AllMolNameList = []                                                            ## list of molecule names
        self.AllTransParamList = []                                                         ## list of transition parameters
        self.AllMinFreq = 1.e99                                                             ## lower frequency of plot
        self.AllMaxFreq = -1.e99                                                            ## upper frequency of plot


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create job directory for current run
        LocalPrintFlag = False
        myXCLASSDir = self.XCLASSRootDir + "xclass/addons/myXCLASS/"
        self.GUIJobDir = task_MAGIX.CreateRunDirectory("XCLASSGUI", myXCLASSDir, LocalPrintFlag)
        self.CurrentMolfitFileName = self.GUIJobDir + "/New_molfit-file.molfit"             ## define path and name of new molfit file
        self.IsoRatioFileName = self.GUIJobDir + "/New_iso-ratio-file.molfit"               ## define path and name of new iso ratio file


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## message
        #    title = "Unofficial part of XCLASS"
        #    bodyText = "\nYou are using a prototype of the XCLASS-GUI.\n\nUsage only on a collaboration-level!\n\n\nContinue?"
        #    answer = QtGui.QMessageBox.question(self, title, bodyText, QtGui.QMessageBox.Yes | QtGui.QMessageBox.No, QtGui.QMessageBox.No)
        #    if (answer == QtGui.QMessageBox.No):
        #        self.ReturnParameter = 100
        #        if (self.tabWidget.currentWidget() == 1):
        #            plt.close(self.fig1)
        #            if (self.AllInOneWindowFlag):
        #                self.canvas.close()
        #                self.toolbar.close()
        #        self.close()
        #        sys.exit(0)
        #        return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get sql parameters from molfit file
        if (self.MolfitFileName != ""):
            MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(self.MolfitFileName)
            if (MinNumTransitionsSQL != 1):
                self.MinNumTransitionsSQL = MinNumTransitionsSQL
            if (MaxNumTransitionsSQL != 0.0):
                self.MaxNumTransInFit = MaxNumTransitionsSQL
            if (MaxElowSQL != 1.e6):
                self.ElowMax = MaxElowSQL
            if (MingASQL != 0.0):
                self.IMin = MingASQL
            if (OrderTransSQL != 1):
                self.SortIndex = OrderTransSQL


        ## define SQL parameter array
        self.SQLParamArray = [self.MinNumTransitionsSQL, self.MaxNumTransInFit, self.ElowMax, self.IMin, self.SortIndex]


        ## define list of all known molecules used in combo box of tab 1
        self.UserMoleculelist = []
        QueryString = "select PF_Name from Partitionfunctions"
        AlreadyUsedMolecules = []
        ListOfAllFreqMin = [1.0]
        ListOfAllFreqMax = [1.e9]
        if (self.MolfitFileName == ""):
            print ("\tDetermine all molecules in the database ..", end = ' ', flush = True)
            FastFlag = True
            self.UserMoleculelist = self.Tab1GetMoleculesFromDatabase(AlreadyUsedMolecules, self.DefaultDB, ListOfAllFreqMin, ListOfAllFreqMax, \
                                                                      self.SQLParamArray, FastFlag)
            print ("done!", flush = True)
        self.UserMoleculelist = ["Select molecule .."] + self.UserMoleculelist


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## center GUI
        # screen = QtWidgets.QDesktopWidget().screenGeometry()
        # rect = self.geometry()
        # screen.moveTo(int((screen.width()-rect.width()) / 2.0), int((screen.height() - rect.height()) / 2.0))

        qr = self.frameGeometry()
        cp = QtWidgets.QDesktopWidget().availableGeometry().center()
        qr.moveCenter(cp)
        self.move(qr.topLeft())


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## disable some menu elements
        self.actionSave_current_molfit_file.setEnabled(False)
        self.actionLoad_new_iso_ratio_file.setEnabled(False)
        self.actionSave_new_iso_ratio_file.setEnabled(False)
        self.actionAnalyzeMolfit.setEnabled(False)
        self.actionAdd_molecule.setEnabled(False)
        self.actionRemove_current_molecule.setEnabled(False)
        self.actionGo_to_previous_molecule.setEnabled(False)
        self.actionGo_to_next_molecule.setEnabled(False)
        self.actionShow_TFW.setEnabled(False)
        self.actionFit_molecule.setEnabled(False)
        self.actionFit_all_molecules.setEnabled(False)
        self.actionMake_a_snapshot.setEnabled(False)
        self.actionIgnore_em_and_abs_functions.setEnabled(False)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define events for menue
        self.ExternalWidgetName = ""
        self.actionSave_current_molfit_file.triggered.connect(self.MenuSaveCurrMolfitFile)  ## event for menu "File" -> "Save current molfit file"
        self.actionLoad_new_iso_ratio_file.triggered.connect(self.MenuLoadIsoRatioFile)     ## event for menu "File" -> "Load new iso ratio file"
        self.actionSave_new_iso_ratio_file.triggered.connect(self.MenuSaveIsoRatioFile)     ## event for menu "File" -> "Save current iso ratio file"
        self.actionAnalyzeMolfit.triggered.connect(self.MenuAnalyzeMolfitFile)              ## event for menu "File" -> "Analyze molfit file"
        self.actionQuit.triggered.connect(self.QuitGUI)                                     ## event for menu "File" -> "Quit"
        self.actionAdd_molecule.triggered.connect(self.MenuAddMolecule)                     ## event for menu "Molecule" -> "Add molecule"
        self.actionRemove_current_molecule.triggered.connect(self.MenuRemoveMolecule)       ## event for menu "Molecule" -> "Remove current molecule"
        self.actionGo_to_previous_molecule.triggered.connect(self.PrevMolecule)             ## event for menu "Molecule" -> "Go to previous molecule"
        self.actionGo_to_next_molecule.triggered.connect(self.NextMolecule)                 ## event for menu "Molecule" -> "Go to next molecule"
        self.actionShow_TFW.triggered.connect(self.MenueTFW)                                ## event for menu "Components" -> "Show transition frequency"
        self.actionSet_Elow.triggered.connect(self.MenuExternalWidget)                      ## event for menu "Settings" -> "Set new E_low"
        self.actionSet_IMin.triggered.connect(self.MenuExternalWidget)                      ## event for menu "Settings" -> "Set new I_min"
        self.actionSet_VelRange.triggered.connect(self.MenuExternalWidget)                  ## event for menu "Settings" -> "Set new velocity range"
        self.actionSet_MaxNumTrans.triggered.connect(self.MenuExternalWidget)               ## event for menu "Settings" -> "Set max. number of trans"
        self.actionSort_plots_by_frequency.triggered.connect(self.MenuSortSpectra)          ## event for menu "Settings" -> "Sort plots by frequency .."
        self.actionSort_plots_by_E_low.triggered.connect(self.MenuSortSpectra)              ## event for menu "Settings" -> "Sort plots by E_low .."
        self.actionSort_plots_by_gA.triggered.connect(self.MenuSortSpectra)                 ## event for menu "Settings" -> "Sort plots by gA .."
        self.actionSort_plots_by_E_low_and_gA.triggered.connect(self.MenuSortSpectra)       ## event for menu "Settings" -> "Sort plots by E_low and gA ."
        self.actionSet_subplots_per_figure.triggered.connect(self.MenuExternalWidget)       ## event for menu "Settings" -> "Set subplots per figures"
        self.actionLocal_overlap.triggered.connect(self.MenuExternalWidget)                 ## event for menu "Settings" -> "Local overlap"
        self.actionCF_distance_ordering.triggered.connect(self.MenuExternalWidget)          ## event for menu "Settings" -> "CF <-> distance ordering"
        self.actionIgnore_em_and_abs_functions.triggered.connect(self.MenuExternalWidget)   ## event for menu "Settings" -> "Ignore em and abs functions"
        self.actionFit_molecule.triggered.connect(self.MenuFit)                             ## event for menu "Fit" -> "Fit current molecule"
        self.actionFit_all_molecules.triggered.connect(self.Tab3FitAll)                     ## event for menu "Fit" -> "Fit all molecules"
        self.actionMake_a_snapshot.triggered.connect(self.MenuSnapshot)                     ## event for menu "Snapshot" -> "Make a snapshot"
        self.actionHelp_me.triggered.connect(self.MenuHelpMe)                               ## event for menu "Help" -> "Help me"
        self.actionAbout_XCLASS_GUI.triggered.connect(self.MenuAboutXCLASSGUI)              ## event for menu "Help" -> "About XCLASS GUI"


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## define events for Tab 1 of GUI


        ## limit line edit events
        self.lineEdit_MinFreq.setValidator(QtGui.QDoubleValidator(0.0, 1.e15, 6))           ## set range for lineEdit element lineEdit_MinFreq
        self.lineEdit_MaxFreq.setValidator(QtGui.QDoubleValidator(0.0, 1.e15, 6))           ## set range for lineEdit element lineEdit_MaxFreq
        self.lineEdit_SizeTele.setValidator(QtGui.QDoubleValidator(0.0, 1.e6, 6))           ## set range for lineEdit element lineEdit_SizeTele
        self.lineEdit_Tbg.setValidator(QtGui.QDoubleValidator(-1.e5, 1.e5, 6))              ## set range for lineEdit element lineEdit_Tbg
        self.lineEdit_TSlope.setValidator(QtGui.QDoubleValidator(0.0, 1.e5, 6))             ## set range for lineEdit element lineEdit_TSlope
        self.lineEdit_beta.setValidator(QtGui.QDoubleValidator(-1.e5, 1.e5, 6))             ## set range for lineEdit element lineEdit_beta
        self.lineEdit_kappa.setValidator(QtGui.QDoubleValidator(0.0, 1.e99, 6))             ## set range for lineEdit element lineEdit_kappa
        self.lineEdit_nH.setValidator(QtGui.QDoubleValidator(0.0, 1.e99, 6))                ## set range for lineEdit element lineEdit_nH
        self.lineEdit_vLSR.setValidator(QtGui.QDoubleValidator(-1.e9, 1.e9, 6))             ## set range for lineEdit element lineEdit_vLSR
        #self.lineEdit_Redshift.setValidator(QtGui.QDoubleValidator(-1.e9, 1.e9, 6))         ## set range for lineEdit element lineEdit_Redshift
        self.lineEdit_MaxIterLM.setValidator(QtGui.QIntValidator(1, 65535))                 ## set range for lineEdit element lineEdit_MaxIterLM


        ## define events for tab 1
        self.tabWidget.currentChanged.connect(self.AllTabChanged)                           ## event for tab change
        self.pushButton_SelectMolfit_Tab1.clicked.connect(self.Tab1selectFile)              ## event for button "Select ..." of molfit file
        self.lineEdit_PathMolfit_Tab1.editingFinished.connect(self.Tab1LineEdit)            ## event for button "Select ..." of molfit file
        self.pushButton_SelectObsXML_Tab1.clicked.connect(self.Tab1selectFile)              ## event for button "Select ..." of obs. xml file
        self.lineEdit_PathObsXML_Tab1.editingFinished.connect(self.Tab1LineEdit)            ## event for button "Select ..." of obs. xml file
        self.pushButton_SelectAlgXML_Tab1.clicked.connect(self.Tab1selectFile)              ## event for button "Select ..." of alg. xml file
        self.lineEdit_PathAlgXML_Tab1.editingFinished.connect(self.Tab1LineEdit)            ## event for button "Select ..." of alg. xml file
        self.comboBox_MoleculeInput.currentIndexChanged.connect(self.Tab1comboSelected)     ## event for combo box event of tab 1
        self.pushButton_AddObsFile.clicked.connect(self.Tab1selectFile)                     ## event for button "Select ..." of obs. data file
        self.pushButton_RemoveObsFile.clicked.connect(self.Tab1selectFile)                  ## event for button "Select ..." of iso ratio file
        self.checkBox_Interfer.clicked.connect(self.Tab1checkBox)                           ## event for check box Interfer of tab 1
        self.checkBox_UseIso.clicked.connect(self.Tab1checkBox)                             ## event for check box UseIso of tab 1
        self.comboBox_DataFileSelection.currentIndexChanged.connect(self.Tab1comboSelected) ## event for combo box event of tab 1
        self.comboBox_RangeSelection.currentIndexChanged.connect(self.Tab1comboSelected)    ## event for combo box event of tab 1
        self.lineEdit_MinFreq.editingFinished.connect(self.Tab1LineEdit)                    ## event for line edit of min. freq. of tab 1
        self.lineEdit_MaxFreq.editingFinished.connect(self.Tab1LineEdit)                    ## event for line edit of max. freq. of tab 1
        self.pushButton_SelectFreqRange.clicked.connect(self.Tab1selectFreqRange)           ## event for button "Select Frequency .. "
        self.lineEdit_SizeTele.editingFinished.connect(self.Tab1LineEdit)                   ## event for line edit of size of telescope of tab 1
        self.lineEdit_Tbg.editingFinished.connect(self.Tab1LineEdit)                        ## event for line edit of T_back of tab 1
        self.lineEdit_TSlope.editingFinished.connect(self.Tab1LineEdit)                     ## event for line edit of T_slope. of tab 1
        self.lineEdit_MaxIterLM.editingFinished.connect(self.Tab1LineEdit)                  ## event for line edit of max. iter. for LM of tab 1
        self.pushButton_SelectAlgorithm.clicked.connect(self.Tab1SelectAlgorithm)           ## event for button "Select ..." of algorithm xml file
        self.lineEdit_beta.editingFinished.connect(self.Tab1LineEdit)                       ## event for line edit of beta of tab 1
        self.lineEdit_kappa.editingFinished.connect(self.Tab1LineEdit)                      ## event for line edit of kappa of tab 1
        self.lineEdit_nH.editingFinished.connect(self.Tab1LineEdit)                         ## event for line edit of nH. of tab 1
        self.lineEdit_vLSR.editingFinished.connect(self.Tab1LineEdit)                       ## event for line edit of vLSR of tab 1
        #self.lineEdit_Redshift.editingFinished.connect(self.Tab1LineEdit)                   ## event for line edit of vLSR of tab 1
        self.pushButton_SelectIso.clicked.connect(self.Tab1selectFile)                      ## event for button "Select ..." of iso ratio file
        self.lineEdit_PathIsoFile.editingFinished.connect(self.Tab1LineEdit)                ## event for button "Select ..." of iso ratio file
        self.pushButton_SelectObsData.clicked.connect(self.Tab1selectFile)                  ## event for button "Select ..." of obs. data file
        self.lineEdit_PathObsData.editingFinished.connect(self.Tab1LineEdit)                ## event for button "Select ..." of obs. data file
        self.checkBox_Tbg_Flag.clicked.connect(self.Tab1checkBox)                           ## event for check box Tbg_Flag
        #    self.checkBox_nH_Flag.clicked.connect(self.Tab1checkBox)                       ## event for check box nH_Flag
        self.pushButton_Cancel_Tab1.clicked.connect(self.buttonClicked)                     ## event for button "Cancel"
        self.pushButton_SMF_Tab1.clicked.connect(self.buttonClicked)                        ## event for button "Single Molecule Fits >>"


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## define events for Tab 2 of GUI


        ## limit line edit events
        self.Edit_Low_ss.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))             ## set range for lineEdit element Edit_Low_ss
        self.Edit_Up_ss.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))              ## set range for lineEdit element Edit_Up_ss
        self.Edit_Val_ss.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))             ## set range for lineEdit element Edit_Val_ss
        self.Edit_Low_Temp.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))           ## set range for lineEdit element Edit_Low_Temp
        self.Edit_Up_Temp.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))            ## set range for lineEdit element Edit_Up_Temp
        self.Edit_Val_Temp.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))           ## set range for lineEdit element Edit_Val_Temp
        self.Edit_Low_Ntot.setValidator(QtGui.QDoubleValidator(0.0, 1.e45, 6))              ## set range for lineEdit element Edit_Low_Ntot
        self.Edit_Up_Ntot.setValidator(QtGui.QDoubleValidator(0.0, 1.e45, 6))               ## set range for lineEdit element Edit_Up_Ntot
        self.Edit_Val_Ntot.setValidator(QtGui.QDoubleValidator(0.0, 1.e45, 6))              ## set range for lineEdit element Edit_Val_Ntot
        self.Edit_Low_vwidth.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))         ## set range for lineEdit element Edit_Low_vwidth
        self.Edit_Up_vwidth.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))          ## set range for lineEdit element Edit_Up_vwidth
        self.Edit_Val_vwidth.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))         ## set range for lineEdit element Edit_Val_vwidth
        self.Edit_Low_voff.setValidator(QtGui.QDoubleValidator(-9999.99, 9999.99, 6))       ## set range for lineEdit element Edit_Low_voff
        self.Edit_Up_voff.setValidator(QtGui.QDoubleValidator(-9999.99, 9999.99, 6))        ## set range for lineEdit element Edit_Up_voff
        self.Edit_Val_voff.setValidator(QtGui.QDoubleValidator(-9999.99, 9999.99, 6))       ## set range for lineEdit element Edit_Val_voff
        self.Edit_Low_IsoRatio.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))       ## set range for lineEdit element Edit_Low_IsoRatio
        self.Edit_Up_IsoRatio.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))        ## set range for lineEdit element Edit_Up_IsoRatio
        self.Edit_Val_IsoRatio.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))       ## set range for lineEdit element Edit_Val_IsoRatio
        self.Edit_Val_Distance.setValidator(QtGui.QDoubleValidator(0.0, 99999.99, 6))       ## set range for lineEdit element Edit_Val_Distance


        ## define events for tab 2
        ListParam = ["ss", "Temp", "Ntot", "vwidth", "voff"]
        for Param in ListParam:
            exec("self.checkBox_FF_" + Param + ".clicked.connect(self.Tab2checkBoxSelected)")   ## get FitFlag check boxes
            exec("self.Slider__" + Param + ".valueChanged.connect(self.Tab2SliderValueChange)") ## get slider values
            exec("self.Edit_Low_" + Param + ".editingFinished.connect(self.Tab2EditSelected)")  ## event for text changed in Edit_Low_*
            exec("self.Edit_Up_" + Param + ".editingFinished.connect(self.Tab2EditSelected)")   ## event for text changed in Edit_Up_*
            exec("self.Edit_Val_" + Param + ".editingFinished.connect(self.Tab2EditSelected)")  ## event for text changed in Edit_Val_*
        self.Edit_Low_IsoRatio.editingFinished.connect(self.Tab2EditSelected)               ## event for text changed in Edit_Low_*
        self.Edit_Up_IsoRatio.editingFinished.connect(self.Tab2EditSelected)                ## event for text changed in Edit_Up_*
        self.Edit_Val_IsoRatio.editingFinished.connect(self.Tab2EditSelected)               ## event for text changed in Edit_Val_*
        self.Edit_Val_Distance.editingFinished.connect(self.Tab2EditSelected)               ## event for text changed in Edit_Val_Distance
        self.checkBox_FF_iso.clicked.connect(self.Tab2checkBoxSelected)                     ## get FitFlag check boxes
        self.comboBox_iso.currentIndexChanged.connect(self.Tab2comboSelected)               ## event for combo box isotopologue
        self.Slider__IsoRatio.valueChanged.connect(self.Tab2SliderValueChange)              ## get slider values
        self.comboBox_molecule.currentIndexChanged.connect(self.Tab2comboSelected)          ## event for combo box molecule
        self.pushButton_AddComp.clicked.connect(self.buttonClicked)                         ## event for button AddComp
        self.pushButton_rmComp.clicked.connect(self.buttonClicked)                          ## event for button Remove Comp
        self.pushButton_left.clicked.connect(self.buttonClicked)                            ## event for button left
        self.pushButton_left_big.clicked.connect(self.buttonClicked)                        ## event for button left
        self.pushButton_right.clicked.connect(self.buttonClicked)                           ## event for button right
        self.pushButton_right_big.clicked.connect(self.buttonClicked)                       ## event for button right
        self.pushButton_up.clicked.connect(self.buttonClicked)                              ## event for button up
        self.pushButton_down.clicked.connect(self.buttonClicked)                            ## event for button down
        self.pushButton_Cancel.clicked.connect(self.buttonClicked)                          ## event for button Cancel
        self.pushButton_Undo.clicked.connect(self.buttonClicked)                            ## event for button Undo
        self.pushButton_Redo.clicked.connect(self.buttonClicked)                            ## event for button Redo
        self.pushButton_PrevMol.clicked.connect(self.buttonClicked)                         ## event for button Prev. Mol.
        self.pushButton_NextMol.clicked.connect(self.buttonClicked)                         ## event for button Next Mol.
        self.list_molfit.clicked.connect(self.Tab2molfitListEntrySelected)                  ## event for selected molfit line
        self.radioButton_Core.clicked.connect(self.Tab2radioSelected)                       ## event for radioButton_Core
        self.radioButton_Fore.clicked.connect(self.Tab2radioSelected)                       ## event for radioButton_Fore
        self.checkBox_active.clicked.connect(self.Tab2checkActive)                          ## event for checkBox_active
        self.checkBox_background.clicked.connect(self.Tab2checkBackground)                  ## event for checkBox_background
        self.checkBox_rm_cont.clicked.connect(self.Tab2checkrmCont)                         ## event for checkBox_rm_cont
        self.checkBox_useIso.clicked.connect(self.Tab2checkUseIso)                          ## event for checkBox_useIso
        self.pushButton_Undo.setEnabled(False)                                              ## disable undo button
        self.pushButton_Redo.setEnabled(False)                                              ## disable redo button
        if (self.CFUsageFlag):
            self.label_Dist.setVisible(False)
        else:
            self.label_Dist.setVisible(True)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## define events for Tab 3 of GUI


        ## define comboBox events
        self.comboBox_ObsDataFile_Tab3.currentIndexChanged.connect(self.Tab3comboSelected)  ## event for combo box ObsDataFile_Tab3
        self.comboBox_FreqRange_Tab3.currentIndexChanged.connect(self.Tab3comboSelected)    ## event for combo box FreqRange_Tab3
        self.comboBox_Isotopologues_Tab3.currentIndexChanged.connect(self.Tab3comboSelected) ## event for combo box Isotopologues_Tab3


        ## define list_molfit_Tab3 event
        self.list_molfit_Tab3.clicked.connect(self.Tab3molfitListEntrySelected)             ## event for selected molfit line in Tab 3


        ## define checkBox_ShowAll_Tab3 event
        self.checkBox_ShowAll_Tab3.clicked.connect(self.Tab3checkBoxSelected)               ## event for check box checkBox_ShowAll_Tab3


        ## define pushButton events for Tab 3
        self.pushButton_SMF_Tab3.clicked.connect(self.buttonClicked)                        ## event for button "<< Single Molecule Fits"
        self.pushButton_FitAll_Tab3.clicked.connect(self.buttonClicked)                     ## event for button "Fit all"
        self.pushButton_Cancel_Tab3.clicked.connect(self.buttonClicked)                     ## event for button "Cancel"
        self.pushButton_Quit_Tab3.clicked.connect(self.buttonClicked)                       ## event for button "Quit"


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize tabs


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## initialize tab 1
        if (self.tabWidget.currentIndex() == 0):
            self.Tab1SetValues()


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## initialize tab 2 and create plots for GUI
        if (self.tabWidget.currentIndex() == 1):
            self.Tab2SetValuesForTab()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##====================================================================================================================================================
    ##
    ##  general subroutines for all tabs
    ##
    ##====================================================================================================================================================


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## continuum function
    ##
    def ContinuumFunction(self, nu, Tbg, TSlope, nuMin):
        """

    input parameters:
    -----------------


        - nu:                   frequency

        - Tbg:                  background temperature

        - TSlope:               temperature slope

        - nuMin:                min. frequency


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("nu = ", nu)
        # print ("Tbg = ", Tbg)
        # print ("TSlope = ", TSlope)
        # print ("nuMin = ", nuMin)


        ## return to main program
        return Tbg * (nu / nuMin)**TSlope
    ##--------------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## estimate continuum
    def EstimateContinuum(self, DataArray, Method = "statcont", NumParts = 20):
        """

    input parameters:
    -----------------

        - DataArray:        data array

        - Method:           (optional) method used to estimate continuum, (default: "statcont")

        - NumParts:         (optional) number of subclasses, (default: 20)


    output parameters:
    ------------------

        - DataNoContArray:  data array without continuum
        """

        # Debug:
        # print ("\nDataArray = ", DataArray)
        # print ("Method = ", Method)
        # print ("NumParts = ", NumParts)


        ## initialize return parameter
        DataNoContArray = np.zeros((len(DataArray)), dtype = np.float32)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## use statcont to remove continuum
        if (Method in ["statcont"]):


            ## try to import statcont
            STATCONTAvailableFlag = True
            try:
                from external.statcont import cont_finding
            except:
                STATCONTAvailableFlag = False


            ## remove continuum using statcont
            if (STATCONTAvailableFlag):


                ## compute rms
                rms = np.sqrt(np.mean(DataArray[:, 1]**2))

                # Debug:
                # print ("rms = ", rms)


                ## split spectrum into smaller parts
                l = len(DataArray)
                q = float(l) / float(NumParts)
                if (q == float(int(q))):
                    SplittedLocalData = np.vsplit(DataArray, NumParts)
                else:
                    ll = int(q) * NumParts
                    SplittedLocalData = np.vsplit(DataArray[:ll, :], NumParts)

                # Debug:
                # print ("l = ", l)
                # print ("NumParts = ", NumParts)
                # print ("q = ", q)


                ## apply statcont to each part
                nuList = []
                contLevelList = []
                for LocalPartDataID, LocalPartData in enumerate(SplittedLocalData):


                    ## write the intensity of a given pixel for all the channels into the array flux and the frequencies in the array freqs
                    rms_noise = rms
                    freq_axis = 0
                    freq = LocalPartData[:, 0]
                    if (len(freq) > 0):
                        MinFreq = np.min(freq)
                        MaxFreq = np.max(freq)
                        CenterFreq = MinFreq + ((MaxFreq - MinFreq) / 2.0)
                        flux = LocalPartData[:, 1]
                        nchan = len(flux)
                        nuList.append(CenterFreq)


                        ## call sigma clipping subroutine
                        sigmaclip_flux_prev, sigmaclip_flux, sigmaclip_noise, filtered_data = statcont.cont_finding.c_sigmaclip(flux, rms_noise,freq_axis)
                        contLevelList.append(sigmaclip_flux)


                ## fit background function to list of continuum levels
                contLevelList = np.asarray(contLevelList)
                nuList = np.asarray(nuList)
                Tbg = DataArray[0, 1]
                TSlope = 0.0
                nuMin= DataArray[0, 0]
                p0 = scipy.array([Tbg, TSlope, nuMin])
                try:
                    popt, pcov = scipy.optimize.curve_fit(self.ContinuumFunction, nuList, contLevelList, p0)
                except:
                    popt = None

                # Debug:
                # print ("popt = ", popt)


                ## define array of fit function
                if (popt is not None):
                    DataNoContArray[:] = DataArray[:, 1] - self.ContinuumFunction(DataArray[:, 0], *popt)


        ## return to GUI
        return DataNoContArray
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## update general parameter arrays
    def UpdateGeneralParameters(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("\n\nself.CurrentMoleculeID = ", self.CurrentMoleculeID)
        # print ("self.CurrentMolecule = ", self.CurrentMolecule)
        # print ("self.CurrentMolfitFileContents = ", self.CurrentMolfitFileContents)
        # print ("self.IsoFlag = ", self.IsoFlag)
        # print ("len(self.AllMoleculesMolfitFiles) = ", len(self.AllMoleculesMolfitFiles))
        # print ("self.AllMoleculesMolfitFiles = ", self.AllMoleculesMolfitFiles)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store local molfit and local iso ratio file
        self.AllMoleculesMolfitFiles[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentMolfitFileContents)
        if (self.IsoFlag):
            self.AllIsotopologueList[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentIsoParam)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update global isotopologue parameters for all other molecules as well


        ## is a global ratio is defined in the current iso param. list
        GlobalIsoList = []
        for LocalIsoParam in self.CurrentIsoParam:
            Isotopologue = LocalIsoParam[0]
            if (Isotopologue.lower().find("global_") > (-1)):
                GlobalIsoList.append(LocalIsoParam)


        ## if yes, update entries for all other molecules
        if (GlobalIsoList != []):
            for GlobalIsoEntry in GlobalIsoList:
                GlobalIsoIso = GlobalIsoEntry[0]
                GlobalIsoMaster = GlobalIsoEntry[1]
                CopyAllIsotopologueList = copy.deepcopy(self.AllIsotopologueList)
                for LocalIsoParamMoleculeIndex, LocalIsoParamMolecule in enumerate(CopyAllIsotopologueList):
                    for LocalIsoParamIndex, LocalIsoParam in enumerate(LocalIsoParamMolecule):
                        if (GlobalIsoIso == LocalIsoParam[0] and GlobalIsoMaster == LocalIsoParam[1]):
                            self.AllIsotopologueList[LocalIsoParamMoleculeIndex][LocalIsoParamIndex] = copy.deepcopy(GlobalIsoEntry)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## quit GUI
    def QuitGUI(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## quit GUI
        self.ReturnParameter = 100
        if (self.tabWidget.currentWidget() == 1):
            plt.close(self.fig1)
            if (self.AllInOneWindowFlag):
                #self.canvas.close()
                self.toolbar.close()
        self.close()


        ## define return variables
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## go to previous molecule
    def PrevMolecule(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check if there is a previous molecule
        if (self.CurrentMoleculeID <= 0):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## disable elements of tab 2
        self.Tab2setDisEnabled(False)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update general parameter arrays
        self.UpdateGeneralParameters()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get index of selected molecule
        LastCurrentMoleculeID = self.CurrentMoleculeID
        self.CurrentMoleculeID -= 1
        self.CurrentMolecule = self.MoleculesInMolfitFile[self.CurrentMoleculeID]
        self.CurrentMolfitIndex = 1
        self.CurrentIsoID = 0

        # Debug:
        # print ("self.CurrentIsoID = ", self.CurrentIsoID)
        # print ("self.CurrentMolecule = ", self.CurrentMolecule)
        # print ("self.MoleculesInMolfitFile = ", self.MoleculesInMolfitFile)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## prepare subdirectories for current molecule
        OldBackgroundFileNameList = self.BackgroundFileNameList
        LocalMoleculeFileName = task_LineIdentification.MoleculeFileName(self.CurrentMolecule)
        self.CurrentMoleculeDir = self.GUIJobDir + LocalMoleculeFileName + "/"
        if (not os.path.exists(self.CurrentMoleculeDir)):
            cmdString = "rm -rf {:s} ; ".format(self.CurrentMoleculeDir)
            cmdString += "mkdir {:s} ; ".format(self.CurrentMoleculeDir)
            os.system(cmdString)


        ## check, if a background file is defined in the already existing obs. xml file
        else:
            LocalObsXMLFile = self.CurrentMoleculeDir + "obs.xml"
            if (os.path.isfile(LocalObsXMLFile)):
                self.BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFile, "BackgroundFileName")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update GUI
        self.CompNumChangeFlag = True
        self.InitTab2Flag = True
        okStatus = self.Tab2NewXMLFile()
        self.BackgroundFileNameList = OldBackgroundFileNameList
        if (str(okStatus) == "1"):
            self.CurrentMoleculeID = LastCurrentMoleculeID
            self.CurrentMolecule = self.MoleculesInMolfitFile[self.CurrentMoleculeID]
            self.CurrentMolfitIndex = 1
            self.CurrentIsoID = 0
            self.comboBox_molecule.setCurrentIndex(self.CurrentMoleculeID)
            self.ObsXMLFileChangeFlag = True
        self.Tab2setDisEnabled(True)


        ## define return variables
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## go to next molecule
    def NextMolecule(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check if there is a next molecule
        if (self.CurrentMoleculeID >= (len(self.MoleculesInMolfitFile) - 1)):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## disable elements of tab 2
        self.Tab2setDisEnabled(False)


        ## set max. number of plots
        self.MaxNumberSpectrum = self.MaxNumberSpectrumDefault
        self.InitTab2Flag = True


        ## store local molfit and local iso ratio file
        self.AllMoleculesMolfitFiles[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentMolfitFileContents)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update general parameter arrays
        self.UpdateGeneralParameters()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get index of selected molecule
        LastCurrentMoleculeID = self.CurrentMoleculeID
        self.CurrentMoleculeID += 1
        self.CurrentMolecule = self.MoleculesInMolfitFile[self.CurrentMoleculeID]
        self.CurrentMolfitIndex = 1
        self.CurrentIsoID = 0

        # Debug:
        # print ("self.CurrentIsoID = ", self.CurrentIsoID)
        # print ("self.CurrentMolecule = ", self.CurrentMolecule)
        # print ("self.MoleculesInMolfitFile = ", self.MoleculesInMolfitFile)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## prepare subdirectories for current molecule
        OldBackgroundFileNameList = self.BackgroundFileNameList
        LocalMoleculeFileName = task_LineIdentification.MoleculeFileName(self.CurrentMolecule)
        self.CurrentMoleculeDir = self.GUIJobDir + LocalMoleculeFileName + "/"
        if (not os.path.exists(self.CurrentMoleculeDir)):
            cmdString = "rm -rf " + self.CurrentMoleculeDir + "; mkdir " + self.CurrentMoleculeDir
            os.system(cmdString)


        ## check, if a background file is defined in the already existing obs. xml file
        else:
            LocalObsXMLFile = self.CurrentMoleculeDir + "obs.xml"
            if (os.path.isfile(LocalObsXMLFile)):
                self.BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFile, "BackgroundFileName")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update GUI
        self.CompNumChangeFlag = True
        self.InitTab2Flag = True
        okStatus = self.Tab2NewXMLFile()
        self.BackgroundFileNameList = OldBackgroundFileNameList
        if (str(okStatus) == "1"):
            self.CurrentMoleculeID = LastCurrentMoleculeID
            self.CurrentMolecule = self.MoleculesInMolfitFile[self.CurrentMoleculeID]
            self.CurrentMolfitIndex = 1
            self.CurrentIsoID = 0
            self.comboBox_molecule.setCurrentIndex(self.CurrentMoleculeID)
            self.ObsXMLFileChangeFlag = True
        self.Tab2setDisEnabled(True)


        ## update history arrays
        self.Tab2UpdateHistoryArray()


        ## define return variables
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for pressed button events
    def buttonClicked(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## is GUI locked?
        if (self.lock == 1):
            return


        ## get name of button
        sender = self.sender()
        objectName = sender.objectName()
        name = sender.text()

        # Debug:
        # print ("sender = ", sender)
        # print ("objectName = ", objectName)
        # print ("name = ", name)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ##
        ## buttons of tab 1
        ##
        ##------------------------------------------------------------------------------------------------------------------------------------------------


        ## "Cancel" button of tab 1 and "Cancel" button of tab 3 and "Quit" button of tab 3
        if (objectName == "pushButton_Cancel_Tab1" or objectName == "pushButton_Cancel_Tab3" or objectName == "pushButton_Quit_Tab3"):
            self.QuitGUI()


        ## "Single Molecule Fits >>" button of tab 1 and "<< Single Molecule Fits" button of tab 3
        if (objectName == "pushButton_SMF_Tab1" or objectName == "pushButton_SMF_Tab3"):
            self.tabWidget.setCurrentIndex(1)
            # self.InitTab2Flag = True
            # self.Tab2ChangeTab()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ##
        ## buttons of tab 2
        ##
        ##------------------------------------------------------------------------------------------------------------------------------------------------


        ## "Add Component" button
        elif (name == "Add Component"):
            self.CompNumChangeFlag = True
            LocalDefaultParameter = copy.deepcopy(self.DefaultMolfitParameter)
            if (not self.CFUsageFlag):
                LocalDefaultParameter[5][0] = "1.0"
            self.CurrentMolfitParameter.insert(self.CurrentComponentIndex, LocalDefaultParameter)
            self.ActiveCompList.insert(self.CurrentComponentIndex, True)


            ## update history arrays
            self.Tab2UpdateHistoryArray()


            ## update GUI
            self.Tab2SetValuesForTab()


        ## "Remove Component" button
        elif (name == "Remove Component"):
            if (len(self.CurrentMolfitParameter) > 1):
                self.CompNumChangeFlag = True
                del self.CurrentMolfitFileContents[self.CurrentMolfitIndex]
                del self.CurrentMolfitParameter[self.CurrentComponentIndex]
                del self.ActiveCompList[self.CurrentComponentIndex]
                if (self.CurrentMolfitIndex > 1):
                    self.CurrentMolfitIndex = max(1, self.CurrentMolfitIndex - 1)
                    self.CurrentComponentIndex -= 1


                ## update history arrays
                self.Tab2UpdateHistoryArray()


                ## update GUI
                self.Tab2SetValuesForTab()


        ## "<<" button
        elif (name == "<<"):
            ReplotFlag = False
            MinSpecIndex = min(self.SpectrumIndexList)
            MaxSpecIndex = max(self.SpectrumIndexList)
            if (MinSpecIndex > 0 and MaxSpecIndex <= (self.TotalNumberSpectra - 1)):
                NewMin = max(MinSpecIndex - self.MaxNumberSpectrum, 0)
                NewMax = MinSpecIndex
                self.SpectrumIndexList = list(range(NewMin, NewMax))
                self.ChangeSpectrumWindowFlag = True
                ReplotFlag = True


            ## if you reach first spectrum, go to last one
            elif (MinSpecIndex == 0 and self.MaxNumberSpectrum <= (self.TotalNumberSpectra - 1)):
                i = int(self.TotalNumberSpectra / self.MaxNumberSpectrum)
                h = self.TotalNumberSpectra - (i * self.MaxNumberSpectrum)
                if (h == 0):
                    h = self.MaxNumberSpectrum
                NewMin = (self.TotalNumberSpectra - h)
                NewMax = self.TotalNumberSpectra
                self.SpectrumIndexList = list(range(NewMin, NewMax))
                self.ChangeSpectrumWindowFlag = True
                ReplotFlag = True


            ## replot GUI
            if (ReplotFlag):


                # Debug:
                if (self.dbgFlag):
                    print ("\nbuttonClicked:")
                    print ("name = ", name)
                    print ("NewMin = ", NewMin)
                    print ("NewMax = ", NewMax)
                    print ("<<self.SpectrumIndexList = ", self.SpectrumIndexList)


                ## update GUI
                self.Tab2SetValuesForTab()


        ## ">>" button
        elif (name == ">>"):
            ReplotFlag = False
            MinSpecIndex = min(self.SpectrumIndexList)
            MaxSpecIndex = max(self.SpectrumIndexList)
            if (MinSpecIndex >= 0 and MaxSpecIndex < (self.TotalNumberSpectra - 1)):
                NewMin = (MaxSpecIndex + 1)
                NewMax = min((MaxSpecIndex + self.MaxNumberSpectrum + 1), self.TotalNumberSpectra)
                self.SpectrumIndexList = list(range(NewMin, NewMax))
                self.ChangeSpectrumWindowFlag = True
                ReplotFlag = True


            ## if you reach last spectrum, go to first one
            elif (MaxSpecIndex == (self.TotalNumberSpectra - 1) and self.MaxNumberSpectrum <= (self.TotalNumberSpectra - 1)):
                NewMin = 0
                NewMax = self.MaxNumberSpectrum
                self.SpectrumIndexList = list(range(NewMin, NewMax))
                self.ChangeSpectrumWindowFlag = True
                ReplotFlag = True

            # Debug:
            # print ("\nMinSpecIndex = ", MinSpecIndex)
            # print ("MaxSpecIndex = ", MaxSpecIndex)
            # print ("self.TotalNumberSpectra = ", self.TotalNumberSpectra)
            # print ("self.MaxNumberSpectrum = ", self.MaxNumberSpectrum)


            ## replot GUI
            if (ReplotFlag):


                # Debug:
                if (self.dbgFlag):
                    print ("\nbuttonClicked:")
                    print ("name = ", name)
                    print (">>self.SpectrumIndexList = ", self.SpectrumIndexList)


                ## update GUI
                self.Tab2SetValuesForTab()


        ## "↑" button
        elif (objectName == "pushButton_up"):
            if (self.CurrentComponentIndex >= 1):
                self.CompNumChangeFlag = True
                NewCompIndex = self.CurrentComponentIndex - 1


                ## self.CurrentMolfitParameter
                LineCopy = self.CurrentMolfitParameter[NewCompIndex]
                self.CurrentMolfitParameter[NewCompIndex] = self.CurrentMolfitParameter[self.CurrentComponentIndex]
                self.CurrentMolfitParameter[self.CurrentComponentIndex] = LineCopy


                ## self.CurrentMolfitFileContents
                LineCopy = self.CurrentMolfitFileContents[NewCompIndex]
                self.CurrentMolfitFileContents[NewCompIndex] = self.CurrentMolfitFileContents[self.CurrentComponentIndex]
                self.CurrentMolfitFileContents[self.CurrentComponentIndex] = LineCopy


                ## self.ActiveCompList
                LineCopy = self.ActiveCompList[NewCompIndex]
                self.ActiveCompList[NewCompIndex] = self.ActiveCompList[self.CurrentComponentIndex]
                self.ActiveCompList[self.CurrentComponentIndex] = LineCopy


                ## update indices
                self.CurrentComponentIndex -= 1
                self.CurrentMolfitIndex = self.CurrentComponentIndex + 1


                ## update history arrays
                self.Tab2UpdateHistoryArray()


                ## update GUI
                self.Tab2SetValuesForTab()


        ## "↓" button
        elif (objectName == "pushButton_down"):
            if ((self.CurrentComponentIndex + 1) < len(self.CurrentMolfitParameter)):
                self.CompNumChangeFlag = True
                NewCompIndex = self.CurrentComponentIndex + 1


                ## self.CurrentMolfitParameter
                LineCopy = self.CurrentMolfitParameter[NewCompIndex]
                self.CurrentMolfitParameter[NewCompIndex] = self.CurrentMolfitParameter[self.CurrentComponentIndex]
                self.CurrentMolfitParameter[self.CurrentComponentIndex] = LineCopy


                ## self.CurrentMolfitFileContents
                LineCopy = self.CurrentMolfitFileContents[NewCompIndex]
                self.CurrentMolfitFileContents[NewCompIndex] = self.CurrentMolfitFileContents[self.CurrentComponentIndex]
                self.CurrentMolfitFileContents[self.CurrentComponentIndex] = LineCopy


                ## self.ActiveCompList
                LineCopy = self.ActiveCompList[NewCompIndex]
                self.ActiveCompList[NewCompIndex] = self.ActiveCompList[self.CurrentComponentIndex]
                self.ActiveCompList[self.CurrentComponentIndex] = LineCopy


                ## update indices
                self.CurrentComponentIndex += 1
                self.CurrentMolfitIndex = self.CurrentComponentIndex + 1


                ## update history arrays
                self.Tab2UpdateHistoryArray()


                ## update GUI
                self.Tab2SetValuesForTab()


        ## "Cancel" button
        elif (name == "Cancel"):
            self.QuitGUI()


        ## "Undo" button
        elif (name == "Undo"):


            ## restore old parameter
            if (self.OldParamVectorCounter > 0):
                self.OldParamVectorCounter -= 1


            ## check, if molecule changes
            self.CurrentMolecule = copy.deepcopy(self.OldCurrentMolecule[self.OldParamVectorCounter])
            self.CurrentMoleculeDir = copy.deepcopy(self.OldCurrentMoleculeDir[self.OldParamVectorCounter])
            self.AllMoleculesMolfitFiles = copy.deepcopy(self.OldAllMoleculesMolfitFiles[self.OldParamVectorCounter])
            self.AllIsotopologueList = copy.deepcopy(self.OldAllIsotopologueList[self.OldParamVectorCounter])
            self.CurrentMolfitParameter = copy.deepcopy(self.OldMolfitParameter[self.OldParamVectorCounter])
            self.ActiveCompList = copy.deepcopy(self.OldActiveCompList[self.OldParamVectorCounter])
            self.CurrentIsoParam = copy.deepcopy(self.OldIsoRatioParameter[self.OldParamVectorCounter])
            NewMolID = copy.deepcopy(self.OldCurrentMoleculeID[self.OldParamVectorCounter])
            # self.CurrentMolfitFileContents = copy.deepcopy(self.AllMoleculesMolfitFiles[NewMolID])
            if (self.CurrentMoleculeID != NewMolID):
                self.CurrentMoleculeID = NewMolID                                           ## update molecule id
                self.Tab2setDisEnabled(False)                                               ## disable elements of tab 2
                self.UpdateGeneralParameters()                                              ## update general parameter arrays


                ## update GUI
                self.CompNumChangeFlag = True
                self.InitTab2Flag = True
                self.UndoFlag = False
                okStatus = self.Tab2NewXMLFile()
                self.UndoFlag = True
                self.Tab2setDisEnabled(True)                                                ## enable elements of tab 2
            else:
                self.CurrentMoleculeID = NewMolID                                           ## update molecule id


            ## get parameters
            self.LocalMolfitFile = copy.deepcopy(self.OldLocalMolfitFile[self.OldParamVectorCounter])
            self.MoleculesInMolfitFile = copy.deepcopy(self.OldMoleculesInMolfitFile[self.OldParamVectorCounter])
            self.MolfitFileAllParameters = copy.deepcopy(self.OldMolfitFileAllParameters[self.OldParamVectorCounter])
            self.ActiveCompList = copy.deepcopy(self.OldActiveCompList[self.OldParamVectorCounter])

            # Debug:
            # print ("\n\nUndo:"
            # print ("self.OldParamVectorCounter = ", self.OldParamVectorCounter)
            # print ("len(self.OldMolfitParameter) = ", len(self.OldMolfitParameter)
            # print ("self.OldMolfitParameter = ", self.OldMolfitParameter)


            ## update all parameters
            self.CompNumChangeFlag = True
            self.Tab2SetValuesForTab()


        ## "Redo" button
        elif (name == "Redo"):


            ## restore old parameter
            if (self.OldParamVectorCounter < len(self.OldMolfitParameter)):
                self.OldParamVectorCounter += 1


            ## check, if molecule changes
            self.CurrentMolecule = copy.deepcopy(self.OldCurrentMolecule[self.OldParamVectorCounter])
            self.CurrentMoleculeDir = copy.deepcopy(self.OldCurrentMoleculeDir[self.OldParamVectorCounter])
            self.AllMoleculesMolfitFiles = copy.deepcopy(self.OldAllMoleculesMolfitFiles[self.OldParamVectorCounter])
            self.AllIsotopologueList = copy.deepcopy(self.OldAllIsotopologueList[self.OldParamVectorCounter])
            self.CurrentMolfitParameter = copy.deepcopy(self.OldMolfitParameter[self.OldParamVectorCounter])
            self.CurrentIsoParam = copy.deepcopy(self.OldIsoRatioParameter[self.OldParamVectorCounter])
            self.ActiveCompList = copy.deepcopy(self.OldActiveCompList[self.OldParamVectorCounter])
            NewMolID = copy.deepcopy(self.OldCurrentMoleculeID[self.OldParamVectorCounter])
            # self.CurrentMolfitFileContents = copy.deepcopy(self.AllMoleculesMolfitFiles[NewMolID])
            if (self.CurrentMoleculeID != NewMolID):
                self.CurrentMoleculeID = NewMolID                                           ## update molecule id
                self.Tab2setDisEnabled(False)                                               ## disable elements of tab 2
                self.UpdateGeneralParameters()                                              ## update general parameter arrays


                ## update GUI
                self.CompNumChangeFlag = True
                self.InitTab2Flag = True
                self.UndoFlag = False
                okStatus = self.Tab2NewXMLFile()
                self.UndoFlag = True
                self.Tab2setDisEnabled(True)                                                ## enable elements of tab 2
            else:
                self.CurrentMoleculeID = NewMolID                                           ## update molecule id


            ## get parameters
            self.LocalMolfitFile = copy.deepcopy(self.OldLocalMolfitFile[self.OldParamVectorCounter])
            self.MoleculesInMolfitFile = copy.deepcopy(self.OldMoleculesInMolfitFile[self.OldParamVectorCounter])
            self.MolfitFileAllParameters = copy.deepcopy(self.OldMolfitFileAllParameters[self.OldParamVectorCounter])
            self.ActiveCompList = copy.deepcopy(self.OldActiveCompList[self.OldParamVectorCounter])

            # Debug:
            # print ("\n\nRedo:"
            # print ("self.OldParamVectorCounter = ", self.OldParamVectorCounter)
            # print ("len(self.OldMolfitParameter) = ", len(self.OldMolfitParameter)
            # print ("self.OldMolfitParameter = ", self.OldMolfitParameter)


            ## update all parameters
            self.Tab2SetValuesForTab()


        ## "Prev. Mol." button
        elif (objectName == "pushButton_PrevMol"):
            self.CurrentComponentIndex = 0
            self.PrevMolecule()


            ## update history arrays
            self.Tab2UpdateHistoryArray()


        ## "Next Mol." button
        elif (objectName == "pushButton_NextMol"):
            self.CurrentComponentIndex = 0
            self.NextMolecule()


            ## update history arrays
            self.Tab2UpdateHistoryArray()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ##
        ## buttons of tab 3
        ##
        ##------------------------------------------------------------------------------------------------------------------------------------------------


        ## "Fit All" button
        elif (objectName == "pushButton_FitAll_Tab3"):
            self.Tab3FitAll()


        ## set current event
        if (self.tabWidget.currentIndex() == 1):
            self.LastEvent = objectName


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start myXCLASSFit
    def FitCore(self, LocalFitDir, LocalMolfitFileName, LocalObsXMLFile):
        """

    input parameters:
    -----------------

        - LocalFitDir:              path of fit directory

        - LocalMolfitFileName:      path and name of local molfit file

        - LocalObsXMLFile:          path and name of local obs. xml file


    output parameters:
    ------------------

        - BestMolfitFile:           content of optimized molfit file

        - IsoRatioFileContent:      content of optimized iso ratio file
        """

        # Debug:
        # print ("LocalFitDir = ", LocalFitDir)
        # print ("LocalMolfitFileName = ", LocalMolfitFileName)
        # print ("LocalObsXMLFile = ", LocalObsXMLFile)


        ## deactivate tag describing emission and absorption function if selected
        LocalEmAbsPATHList = []
        if (not self.EmAbsFlag):
            LocalEmAbsPATHList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFile, "EmAbsPATH")
            if (len(LocalEmAbsPATHList) > 0):
                LocalEmAbsPATHListCleaned = [""]
                task_MAGIX.WriteXMLtagNEW(LocalObsXMLFile, "EmAbsPATH", LocalEmAbsPATHListCleaned)


        ## start fit
        newmolfit, modeldata, JobDir = task_myXCLASSFit.myXCLASSFitCore(MolfitsFileName = LocalMolfitFileName, \
                                                                        AlgorithmXMLFileName = self.AlgXMLFileName, \
                                                                        experimentalData = LocalObsXMLFile, \
                                                                        LocalOverlapFlag = self.UserLocalOverlapFlag)


        ## read in parameters for each freq. range for the current molecule
        ListOfSpectraNames = task_MAGIX.GetXMLtagNEW(LocalObsXMLFile, "FileNamesExpFiles")
        ListNumberExpRanges = task_MAGIX.GetXMLtagNEW(LocalObsXMLFile, "NumberExpRanges")
        ListOfAllFreqMax = task_MAGIX.GetXMLtagNEW(LocalObsXMLFile, "MaxExpRange")
        Localt_back_flagList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFile, "t_back_flag")
        LocaltBackList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFile, "BackgroundTemperature")
        LocaltSlopeList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFile, "TemperatureSlope")
        ConvertLogLinFlag = "False"
        ListOfAllFreqMin = []
        BestMolfitFile, IsoRatioFileContent, dummy = task_LineIdentification.GetBestResult(JobDir, ConvertLogLinFlag, ListOfSpectraNames, \
                                                                                           ListNumberExpRanges, ListOfAllFreqMin, ListOfAllFreqMax, \
                                                                                           LocaltBackList, LocaltSlopeList)
        ## analyze best molfit file and remove continuum contributions from molfit file
        LocalMoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(BestMolfitFile)
        molfitFileWithoutContinuum = []
        for LocalMoleculeID, LocalMolecule in enumerate(LocalMoleculesInMolfitFile):
            LowerLocalMolecule = LocalMolecule.lower()
            if (not LowerLocalMolecule.startswith("cont")):
                molfitFileWithoutContinuum = molfitFileWithoutContinuum + MolfitFileForEachMolecule[LocalMoleculeID]


        ## re-activate tag describing emission and absorption function if selected
        if (len(LocalEmAbsPATHList) > 0):
            task_MAGIX.WriteXMLtagNEW(LocalObsXMLFile, "EmAbsPATH", LocalEmAbsPATHList)


        ## copy contents of current jobdirectory to local directory
        cmdString = "mv " + JobDir + "/* " + LocalFitDir + "; rm -rf " + JobDir
        os.system(cmdString)


        ## remove MAGIX plot
        cmdString = "rm -f " + LocalFitDir  + "/final_plot.*.png"
        os.system(cmdString)


        ## return to GUI
        return (molfitFileWithoutContinuum, IsoRatioFileContent)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what happens if tab was changed
    def AllTabChanged(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## define what happens if tab was changed
        if (self.tabWidget.currentIndex() == 0):
            return
        elif (self.tabWidget.currentIndex() == 1):
            try:
                self.canvas.close()
                self.toolbar.close()
            except:
                pass
            self.OnceFlag = True
            self.InitTab2Flag = True
            self.Tab2ChangeTab()
        elif (self.tabWidget.currentIndex() == 2):
            try:
                self.canvas.close()
                self.toolbar.close()
            except:
                pass
            self.OnceFlagTab3 = True
            self.InitFlagTab3 = True
            self.Tab3ChangeTab()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## update value in parameter list
    def UpdateParamList(self, LocalObsDataFileIndex, LocalRangeIndex, LocalParamList, value, actionItem):
        """

    input parameters:
    -----------------

        - LocalObsDataFileIndex:    local obs. data file index

        - LocalRangeIndex:          local freq. range index

        - LocalParamList:           local parameter list

        - value:                    new value

        - actionItem:               define what to do


    output parameters:
    ------------------

        - LocalParamList:           updated parameter list
        """

        # Debug:
        # print ("LocalObsDataFileIndex = ", LocalObsDataFileIndex)
        # print ("LocalRangeIndex = ", LocalRangeIndex)
        # print ("LocalParamList = ", LocalParamList)
        # print ("value = ", value)
        # print ("actionItem = ", actionItem)


        ## make a copy of parameter list
        CopyLocalParamList = copy.deepcopy(LocalParamList)


        ## insert elements
        if (actionItem == "add"):
            LocalParamList = []
            for LocalElementID, LocalElement in enumerate(CopyLocalParamList):
                ObsDataID = LocalElement[0]

                if (ObsDataID < LocalObsDataFileIndex):
                    LocalParamList.append(LocalElement)

                elif (ObsDataID >= LocalObsDataFileIndex):
                    ObsDataID = ObsDataID + 1
                    LocalValue = LocalElement[-1]
                    if (len(LocalElement) == 2):
                        LocalParamList.append([ObsDataID, LocalValue])
                    else:
                        RangeID = LocalElement[1]
                        LocalParamList.append([ObsDataID, RangeID, LocalValue])


            ## add new entry
            for LocalValueID, LocalValue in enumerate(value):
                if (LocalRangeIndex == (-1)):
                    LocalParamList.append([LocalObsDataFileIndex, LocalValue])
                else:
                    LocalParamList.append([LocalObsDataFileIndex, LocalValueID, LocalValue])


        ## delete elements from list
        elif (actionItem == "delete"):
            LocalParamList = []
            for LocalElementID, LocalElement in enumerate(CopyLocalParamList):
                ObsDataID = LocalElement[0]
                if (LocalObsDataFileIndex != ObsDataID):
                    if (ObsDataID <= LocalObsDataFileIndex):
                        LocalParamList.append(LocalElement)
                    else:
                        ObsDataID = ObsDataID - 1
                        LocalValue = LocalElement[-1]
                        if (len(LocalElement) == 2):
                            LocalParamList.append([ObsDataID, LocalValue])
                        else:
                            RangeID = LocalElement[1]
                            LocalParamList.append([ObsDataID, RangeID, LocalValue])


        ## update entry in parameter list
        elif (actionItem == "update"):
            if (LocalRangeIndex == (-1)):
                for LocalElementID, LocalElement in enumerate(CopyLocalParamList):
                    ObsDataID = LocalElement[0]
                    if (ObsDataID == LocalObsDataFileIndex):
                        LocalParamList[LocalElementID][1] = value
            else:
                for LocalElementID, LocalElement in enumerate(CopyLocalParamList):
                    ObsDataID = LocalElement[0]
                    if (ObsDataID == LocalObsDataFileIndex):
                        RangeID = LocalElement[1]
                        if (RangeID == LocalRangeIndex):
                            LocalParamList[LocalElementID][2] = value


        ## unknown option
        else:
            print ("\n\n\n\nUnknown action item for function UpdateParamList.")
            print ("actionItem = ", actionItem)
            sys.exit(0)


        ## sort list
        ## use lambda to sort by "x[0]"-> first element of the sublists or x[1] -> second element, if its a tie
        LocalParamList = sorted(LocalParamList,key = lambda x: (x[0], x[1]))


        ## return to GUI
        return LocalParamList
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## calculate partition function for T = T_rot
    def GetPartitionFunction(self, MolName, Trot, dbList):
        """

    input parameters:
    -----------------

        - MolName:                  name of molecule

        - Trot:                     rotation temperture

        - dbList:                   path and name of database file


    output parameters:
    ------------------

        - QT:                       partition function for T = Trot
        """

        # Debug:
        # print ("MolName = ", MolName)
        # print ("Trot = ", Trot)
        # print ("dbList = ", dbList)


        ## initialize return parameter
        QT = 1.0


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define query string
        QueryString = "select * from Partitionfunctions where PF_Name == " + chr(34) + MolName + chr(34)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get partition function values from database
        dbFile = dbList[0]
        try:
            conn = sqlite3.connect(dbFile)
        except sqlite3.Error as e:
            print ("\n\nErrot in subroutine GetPartitionFunction:")
            print ("\t\tCan not connect to sqlite3 databse %s." % dbFile)
            print ("\t\tError: %d: %s" % (e.args[0], e.args[1]))
            sys.exit(1)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read data from database
        cursor = conn.cursor()
        cursor.execute(QueryString)
        conn.commit()
        rows = cursor.fetchall()
        Contents = rows
        conn.close()

        # Debug:
        # print ("rows = ", rows)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## extract partition function values
        TempArray = [1.072, 1.148, 1.23, 1.318, 1.413, 1.514, 1.622, 1.738, 1.862, 1.995, 2.138, 2.291, 2.455, 2.63, 2.725, 2.818, 3.02, 3.236, 3.467, \
                     3.715, 3.981, 4.266, 4.571, 4.898, 5, 5.248, 5.623, 6.026, 6.457, 6.918, 7.413, 7.943, 8.511, 9.12, 9.375, 9.772, 10.471, 11.22, \
                     12.023, 12.882, 13.804, 14.791, 15.849, 16.982, 18.197, 18.75, 19.498, 20.893, 22.387, 23.988, 25.704, 27.542, 29.512, 31.623, \
                     33.884, 36.308, 37.5, 38.905, 41.687, 44.668, 47.863, 51.286, 54.954, 58.884, 63.096, 67.608, 72.444, 75, 77.625, 83.176, 89.125, \
                     95.499, 102.329, 109.648, 117.49, 125.893, 134.896, 144.544, 150, 154.882, 165.959, 177.828, 190.546, 204.174, 218.776, 225, \
                     234.423, 251.189, 269.153, 288.403, 300, 309.03, 331.131, 354.813, 380.189, 407.38, 436.516, 467.735, 500, 501.187, 537.032, \
                     575.44, 616.595, 660.693, 707.946, 758.578, 812.831, 870.964, 933.254, 1000.0]
        QTArray = rows[0][5:115]

        # Debug:
        # print ("TempArray = ", TempArray)
        # print ("len(TempArray) = ", len(TempArray))
        # print ("len(QTArray) = ", len(QTArray))


        ## interpolate QT array to get partition function at T = T_rot
        QT = np.interp(Trot, TempArray, QTArray)

        # Debug:
        # print ("\nTrot = ", Trot)
        # print ("QT = ", QT)


        ## define return parameter
        return QT
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## compute shifted frequency ranges
    def GetShiftedFreq(self, NumExpDataFiles, NumberExpRangesList, GlobalvLSRList, RedshiftList, FreqMinList, FreqMaxList):
        """

    input parameters:
    -----------------

        - NumExpDataFiles:          total number of obs. data files

        - NumberExpRangesList:      list of frequency range numbers

        - GlobalvLSRList:           list of global v_lsr

        - RedshiftList:             list of redshifts

        - FreqMinList:              list of unshifted lower frequencies

        - FreqMaxList:              list of unshifted upper frequencies


    output parameters:
    ------------------

        - ShiftedFreqMinList:       shifted lower frequencies

        - ShiftedFreqMaxList:       shifted upper frequencies
        """

        # Debug:
        # print ("NumExpDataFiles = ", NumExpDataFiles)
        # print ("NumberExpRangesList = ", NumberExpRangesList)
        # print ("GlobalvLSRList = ", GlobalvLSRList)
        # print ("RedshiftList = ", RedshiftList)
        # print ("FreqMinList = ", FreqMinList)
        # print ("FreqMaxList = ", FreqMaxList)


        ## initialize return parameters
        ShiftedFreqMinList = []
        ShiftedFreqMaxList = []


        ## get frequency limits
        for ObsDataFileIndex in range(NumExpDataFiles):                                     ## loop over all obs. data files


            ## get parameters for current range
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberRangeListIn = NumberExpRangesList, \
                                                                           GlobalvLSRListIn = GlobalvLSRList, Redshift_ListIn = RedshiftList)
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']        ## get number of frequency ranges for current obs. data file
            GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
            if (GlobalvLSR is None):
                GlobalvLSR = 0.0
            Redshift = ObsXMLParameterDictFile['Redshift']
            if (Redshift is None):
                Redshift = 0.0


            ## get range parameters
            for RangeIndex in range(NumberFrequencyRanges):                                 ## loop over all range definitions in the whole xml file


                ## get parameters for current range
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = FreqMinList, \
                                                                                FreqMaxListIn = FreqMaxList)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None):
                    FreqMax = ObsXMLParameterDictRange['FreqMax']

                    # Debug:
                    # print ("\n\n\n\n\n\n\n\n\n\n\n")
                    # print ("ObsDataFileIndex, RangeIndex = ", ObsDataFileIndex, RangeIndex)
                    # print ("FreqMin = ", FreqMin)
                    # print ("FreqMax = ", FreqMax)


                    ## shift min. and max. frequency of each range by v_LSR
                    NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift)
                    NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift)
                    ShiftedFreqMin = min(NewFreqMin, NewFreqMax)
                    ShiftedFreqMax = max(NewFreqMin, NewFreqMax)
                    ShiftedFreqMinList.append([ObsDataFileIndex, RangeIndex, ShiftedFreqMin])
                    ShiftedFreqMaxList.append([ObsDataFileIndex, RangeIndex, ShiftedFreqMax])

            # Debug:
            # print ("ShiftedFreqMinList = ", ShiftedFreqMinList)
            # print ("ShiftedFreqMaxList = ", ShiftedFreqMaxList)


        ## define return parameter
        return (ShiftedFreqMinList, ShiftedFreqMaxList)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check, if string describes number
    def IsNumber(self, n):
        """

    input parameters:
    -----------------

        - NumExpDataFiles:          total number of obs. data files


    output parameters:
    ------------------

        - is_number:                flag indicating if string is a number (True) or not (False)
        """

        # Debug:
        # print ("n = ", n)


        is_number = True
        try:
            num = float(n)


            ## check for "nan" floats
            is_number = num == num                                                          ## or use `math.isnan(num)`
        except ValueError:
            is_number = False


        ## define return parameter
        return is_number
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##====================================================================================================================================================
    ##
    ##  subroutines for menu
    ##
    ##====================================================================================================================================================


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## save current molfit file
    def MenuSaveCurrMolfitFile(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None

        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if self.AllMoleculesMolfitFiles is already defined
        if (self.AllMoleculesMolfitFiles == []):
            if (self.MolfitFileName.strip() != ""):
                self.MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(self.MolfitFileName.strip())
                self.AllMoleculesMolfitFiles = copy.deepcopy(MolfitFileForEachMolecule)
            else:
                return
        else:

            ## update self.AllMoleculesMolfitFiles variable with latest parameter status
            self.AllMoleculesMolfitFiles[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentMolfitFileContents)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write molfit files for each molecule to file
        CurrentMolfitFile = open(self.CurrentMolfitFileName, "w")


        ## get sql parameters
        MinNumTransitionsSQL = self.SQLParamArray[0]
        MaxNumTransitionsSQL = self.SQLParamArray[1]
        MaxElowSQL = self.SQLParamArray[2]
        MingASQL = self.SQLParamArray[3]
        OrderTransSQL = self.SQLParamArray[4]
        MolfitHeaderLinesList = task_myXCLASS.WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)


        ## write parts of molfit file
        for line in MolfitHeaderLinesList:                                                  ## write header of molfit file
            CurrentMolfitFile.write(line)


        ## add continuum contributions
        if (self.MolfitContinuumParameters != []):
            for CurrentContinuumParameter in self.MolfitContinuumParameters:
                LocalMolfitFileForEachMolecule = CurrentContinuumParameter[2]
                for line in LocalMolfitFileForEachMolecule:                                 ## write continuum contributions of molfit file
                    CurrentMolfitFile.write(line)


        ## add molecule description
        for LocalMoleculeMolfitFile in self.AllMoleculesMolfitFiles:
            for LocalLine in LocalMoleculeMolfitFile:
                NewLine = LocalLine + "\n"
                NewLine = NewLine. replace("\n\n", "\n")
                CurrentMolfitFile.write(NewLine)
        CurrentMolfitFile.close()


        ## set title of GUI window
        self.GUITitle = "XCLASS-GUI"
        if (self.tabWidget.currentIndex() == 1):
            self.GUITitle += "   ({:d}/{:d})".format(self.CurrentMoleculeID + 1, len(self.MoleculesInMolfitFile))
        self.setWindowTitle(self.GUITitle)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## reload iso ratio file
    def MenuLoadIsoRatioFile(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if iso ratio file is defined
        LocalIsoFlag = str(self.IsotopologuesList[0]).lower()
        LocalIsoFlag = task_myXCLASSMapFit.CheckBool(LocalIsoFlag)
        if (LocalIsoFlag):


            ## shift frequency range limits by global v_lsr
            NumberExpFiles = len(self.ExpFileList)
            ShiftedFreqMinList, ShiftedFreqMaxList = self.GetShiftedFreq(NumberExpFiles, self.NumberExpRangesList, self.GlobalvLSRList, \
                                                                         self.RedshiftList, self.FreqMinList, self.FreqMaxList)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## reset list of iso ratio file parameter
            self.AllIsotopologueList = []
            LocaldbFile = self.dbList[0]
            SelectIsoRatioTable = task_myXCLASSFit.CheckMolecules(self.LocalMolfitFile, self.IsoTableFileNameList[0], ShiftedFreqMinList, \
                                                                  ShiftedFreqMaxList, [LocaldbFile])
            for LocalMolIndex, LocalMol in enumerate(self.MoleculesInMolfitFile):
                LocalIsoParam = self.Tab2GetIsoRatiosFromFile(LocalMol, SelectIsoRatioTable)
                self.AllIsotopologueList.append(copy.deepcopy(LocalIsoParam))
                if (LocalMolIndex == self.CurrentMoleculeID):
                    self.CurrentIsoParam = copy.deepcopy(LocalIsoParam)
                    if (self.CurrentIsoParam != []):                                        ## check if at least one iso. is defined for current mol.
                        self.UseIsoFlag = True
                        if (self.CurrentIsoID > (len(self.CurrentIsoParam) - 1)):
                            self.CurrentIsoID = (len(self.CurrentIsoParam) - 1)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## determine transition frequency windows and refresh Tab 2
            if (self.tabWidget.currentIndex() == 1):
                okStatus = self.Tab2NewXMLFile()
                self.Tab2setDisEnabled(True)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## save current iso ratio file
    def MenuSaveIsoRatioFile(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("self.CurrentIsoParam = ", self.CurrentIsoParam)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## if no isotopologues are used return to GUI
        if (not self.IsoFlag):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store local molfit and local iso ratio file
        if (self.AllIsotopologueList != []):
            self.AllIsotopologueList[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentIsoParam)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create list of all isotopologues parameter for all molecules
        LocalAllIsotopologueList = []
        IDList = []
        for LocalParamIso in self.AllIsotopologueList:
            for LocalParam in LocalParamIso:
                Isotopologue = LocalParam[0]
                IsoMaster = LocalParam[1]
                LocalIDList = [Isotopologue, IsoMaster]
                if (not LocalIDList in IDList):
                    IDList.append(LocalIDList)
                    LocalAllIsotopologueList.append(LocalParam)

                    # Debug:
                    # print ("\nLocalParam = ", LocalParam)
                    # print ("IDList = ", IDList)

        # Debug:
        # print ("LocalAllIsotopologueList = ", LocalAllIsotopologueList)
        # print ("self.AllIsotopologueList = ", self.AllIsotopologueList)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write isotopologues parameter of all molecules to file
        WrittenHeaderFlag = False
        if (LocalAllIsotopologueList != []):
            LocalIsoRatioFileContents = self.Tab2CreateIsoRatioFileContents(LocalAllIsotopologueList)
            self.IsoTableFileNameList = [self.IsoRatioFileName]
            LocalIsoRatioFile = open(self.IsoRatioFileName, "w")
            for IsoLine in LocalIsoRatioFileContents:
                NewLine = IsoLine + "\n"
                NewLine = NewLine.replace("\n\n", "\n")
                LocalIsoRatioFile.write(NewLine)
            LocalIsoRatioFile.close()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze molfit file
    def MenuAnalyzeMolfitFile(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if developer version is used:
        if (not self.devFlag):


            ## Not available
            title = "Not available"
            bodyText = "\nStill under development!" + "\n"
            QtWidgets.QMessageBox.warning(self, title, bodyText, QtWidgets.QMessageBox.Close)


        ## for developing usage only
        else:


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## check, if self.AllMoleculesMolfitFiles is already defined
            if (self.AllMoleculesMolfitFiles == []):
                if (self.MolfitFileName.strip() != ""):
                    self.MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(self.MolfitFileName.strip())
                    self.AllMoleculesMolfitFiles = copy.deepcopy(MolfitFileForEachMolecule)
                else:
                    return
            else:

                ## update self.AllMoleculesMolfitFiles variable with latest parameter status
                self.AllMoleculesMolfitFiles[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentMolfitFileContents)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write molfit files for each molecule to file
            CurrentMolfitFile = []
            for LocalMoleculeMolfitFile in self.AllMoleculesMolfitFiles:
                for LocalLine in LocalMoleculeMolfitFile:
                    NewLine = LocalLine + "\n"
                    NewLine = NewLine. replace("\n\n", "\n")
                    CurrentMolfitFile.append(NewLine)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call GUI for analyzing molfit file
            LocalIsoFlag = str(self.IsotopologuesList[0]).lower()
            LocalIsoFlag = task_myXCLASSMapFit.CheckBool(LocalIsoFlag)
            if (LocalIsoFlag):
                IsoRatioFileName = self.IsoTableFileNameList[0]
            else:
                IsoRatioFileName = ""
            ResBin = 50.0
            UpperIsoRatioLimit = 1000.0
            AnalyzeMolfitFile.Analyze(CurrentMolfitFile, IsoRatioFileName, ResBin, UpperIsoRatioLimit, self.GUIJobDir)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## add molecule
    def MenuAddMolecule(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## start progress bar and center dialog window on screen
        self.Tab2setDisEnabled(False)
        TotalCalls = 5
        bar = ProgressBar(total=TotalCalls, widgetTitle = "Add molecule", bodyText = "\nCollecting molecules. Please wait ..\n")
        # bar.resize(400, 100)
        desktopSize = QtWidgets.QDesktopWidget().screenGeometry()
        top = int((desktopSize.height() / 2.0) - 50)
        left = int((desktopSize.width() / 2.0) - 200)
        bar.move(left, top)
        bar.show()


        ## update progress bar
        ProgressBarValue = 2
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            self.Tab2setDisEnabled(True)
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine list of all molecules which have at least on transition in one of the given frequency ranges
        PossibleMolecules = []
        if (self.dbList == []):
            PathFileNameDB = task_myXCLASS.GetDefaultDBFile()
        else:
            PathFileNameDB = self.dbList[0]
        AlreadyUsedMolecules = copy.deepcopy(self.MoleculesInMolfitFile)

        # Debug:
        # print ("PathFileNameDB = ", PathFileNameDB)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create list of all isotopologues parameter for all molecules
        if (self.IsoFlag):


            ## store local molfit and local iso ratio file
            if (self.AllIsotopologueList != []):
                if (self.CurrentMoleculeID > len(self.AllIsotopologueList)):
                    self.AllIsotopologueList.append(self.CurrentIsoParam)
                else:
                    self.AllIsotopologueList[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentIsoParam)


            IDList = []
            for LocalParamIso in self.AllIsotopologueList:
                for LocalParam in LocalParamIso:
                    Isotopologue = LocalParam[0]
                    IsoMaster = LocalParam[1]
                    LocalIDList = [Isotopologue, IsoMaster]
                    if (not LocalIDList in IDList):
                        IDList.append(LocalIDList)
                        if (not Isotopologue in AlreadyUsedMolecules):
                            AlreadyUsedMolecules.append(Isotopologue)
                        if (not IsoMaster in AlreadyUsedMolecules):
                            AlreadyUsedMolecules.append(IsoMaster)

        # Debug:
        # print ("AlreadyUsedMolecules = ", AlreadyUsedMolecules)


        ## update progress bar
        ProgressBarValue = 3
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            self.Tab2setDisEnabled(True)
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create list of lower and upper frequencies
        ListOfAllFreqMin = []
        ListOfAllFreqMax = []
        for ObsDataFileIndex in range(len(self.ExpFileList)):                               ## loop over all obs. data files


            ## get number of header lines and character separating columns from obs. xml file
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberRangeListIn = self.NumberExpRangesList)
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']        ## get number of frequency ranges for current obs. data file


            ## get parameters for current frequency range
            for RangeIndex in range(NumberFrequencyRanges):                                 ## loop over all range definitions in the whole xml file


                ## get parameter for current frequency range
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = self.FreqMinList, \
                                                                                 FreqMaxListIn = self.FreqMaxList)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None):
                    FreqMax = ObsXMLParameterDictRange['FreqMax']
                    ListOfAllFreqMin.append(min(FreqMin, FreqMax))
                    ListOfAllFreqMax.append(max(FreqMin, FreqMax))

        # Debug:
        # print ("ListOfAllFreqMin = ", ListOfAllFreqMin)
        # print ("ListOfAllFreqMax = ", ListOfAllFreqMax)


        ## update progress bar
        ProgressBarValue = 4
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            self.Tab2setDisEnabled(True)
            return

        # Debug:
        # print ("AlreadyUsedMolecules = ", AlreadyUsedMolecules)
        # print ("PathFileNameDB = ", PathFileNameDB)
        # print ("ListOfAllFreqMin = ", ListOfAllFreqMin)
        # print ("ListOfAllFreqMax = ", ListOfAllFreqMax)
        # print ("self.SQLParamArray = ", self.SQLParamArray)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get all molecules which have at least one transition within given frequency ranges
        FastFlag = False
        PossibleMolecules = self.Tab1GetMoleculesFromDatabase(AlreadyUsedMolecules, PathFileNameDB, ListOfAllFreqMin, ListOfAllFreqMax, \
                                                              self.SQLParamArray, FastFlag)
        # Debug:
        # print ("self.SQLParamArray = ", self.SQLParamArray)
        # print ("PossibleMolecules = ", PossibleMolecules)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## call qdialog to select molecule
        bar.close()
        if (PossibleMolecules == []):
            msgBox = QtWidgets.QMessageBox()
            msgBox.setWindowTitle("WARNING")
            LocalLogText = "There is no molecule which has at least one transition within given frequency range(s).\n\n"
            LocalLogText += "Please define another molecule or expand range definition(s)!\n"
            msgBox.setText(LocalLogText)
            ret = msgBox.exec_()
        else:


            ## call combo box to select molecule
            widgetTitle = "Add new molecule"
            bodyText = "Please select a molecule:"
            LocalSelectedMolecule, ok = QtWidgets.QInputDialog.getItem(self, widgetTitle, bodyText, PossibleMolecules, 0, False)
            LocalSelectedMolecule = str(LocalSelectedMolecule)

            # Debug:
            # print ("ok = ", ok)
            # print ("LocalSelectedMolecule = ", LocalSelectedMolecule)


            ## is "OK" button pressed?
            if (ok):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## update general parameter arrays
                self.UpdateGeneralParameters()


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## add molecule
                self.MoleculesInMolfitFile.append(LocalSelectedMolecule)
                self.CurrentMolecule = LocalSelectedMolecule
                self.CurrentMoleculeID = (len(self.MoleculesInMolfitFile) - 1)
                self.CurrentMolfitParameter = []
                self.CurrentMolfitParameter.append(copy.deepcopy(self.DefaultMolfitParameter))
                self.ActiveCompList.append([True])
                self.CurrentComponentIndex = 0
                self.CompNumChangeFlag = True


                ## prepare subdirectories for current molecule
                LocalMoleculeFileName = task_LineIdentification.MoleculeFileName(self.CurrentMolecule)
                self.CurrentMoleculeDir = self.GUIJobDir + LocalMoleculeFileName + "/"
                os.system("rm -rf " + self.CurrentMoleculeDir + "; mkdir " + self.CurrentMoleculeDir)


                ## update current line of the molfit file
                self.Tab2CreateMolfitFile()
                self.AllMoleculesMolfitFiles.append(copy.deepcopy(self.CurrentMolfitFileContents))

                # Debug:
                # print ("self.CurrentMolfitFileContents = ", self.CurrentMolfitFileContents)


                ## get molfit parameter
                LocalMolInMolfitFile, LocalMolParam, LocalMolfitForEachMol = task_myXCLASS.AnalyzeMolfitFile(self.CurrentMolfitFileContents)
                LocalMolfitParameters = LocalMolParam[0]
                NewRangeFlag = False
                self.CurrentMolfitParameter = self.Tab2GetParameterFromMolfitFile(LocalMolfitParameters, NewRangeFlag)
                self.MolfitFileAllParameters.append(copy.deepcopy(self.CurrentMolfitParameter))


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## check, if iso ratio file is defined
                self.CurrentIsoParam = []
                self.CurrentIsoID = 0
                LocalIsoFlag = str(self.IsotopologuesList[0]).lower()
                LocalIsoFlag = task_myXCLASSMapFit.CheckBool(LocalIsoFlag)
                if (LocalIsoFlag):


                    ## shift frequency range limits by global v_lsr
                    NumberExpFiles = len(self.ExpFileList)
                    ShiftedFreqMinList, ShiftedFreqMaxList = self.GetShiftedFreq(NumberExpFiles, self.NumberExpRangesList, self.GlobalvLSRList, \
                                                                                 self.RedshiftList, self.FreqMinList, self.FreqMaxList)

                    ## reset list of iso ratio file parameter
                    LocaldbFile = self.dbList[0]
                    SelectIsoRatioTable = task_myXCLASSFit.CheckMolecules(self.LocalMolfitFile, self.IsoTableFileNameList[0], ShiftedFreqMinList, \
                                                                          ShiftedFreqMaxList, [LocaldbFile])
                    LocalIsoParam = self.Tab2GetIsoRatiosFromFile(self.CurrentMolecule, SelectIsoRatioTable)
                    self.AllIsotopologueList.append(copy.deepcopy(LocalIsoParam))
                    self.CurrentIsoParam = copy.deepcopy(LocalIsoParam)
                    if (self.CurrentIsoParam != []):                                        ## check if at least one iso. is defined for current mol.
                        self.UseIsoFlag = True
                self.AllIsotopologueList.append(copy.deepcopy(self.CurrentIsoParam))

                # Debug:
                # print ("self.CurrentIsoParam = ", self.CurrentIsoParam)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## update GUI
                if (self.tabWidget.currentIndex() == 1):
                    self.InitTab2Flag = True
                    okStatus = self.Tab2NewXMLFile()
                else:
                    self.ObsXMLFileChangeFlag = True
            else:
                self.Tab2setDisEnabled(True)
                return


        ## return to GUI
        self.Tab2setDisEnabled(True)
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## remove current molecule
    def MenuRemoveMolecule(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is only one molecule contained in current molfit file
        if (len(self.MoleculesInMolfitFile) == 1):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## are you sure?
        title = "Remove current molecule?"
        bodyText = "Are you sure you want to remove this molecule?"
        answer = QtWidgets.QMessageBox.question(self, title, bodyText, QtWidgets.QMessageBox.Yes | QtWidgets.QMessageBox.No, QtWidgets.QMessageBox.No)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update general parameter arrays
        if (answer == QtWidgets.QMessageBox.Yes):


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## disable elements of tab 2
            self.Tab2setDisEnabled(False)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## update general parameter arrays
            self.UpdateGeneralParameters()


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## remove current molecule
            del self.AllMoleculesMolfitFiles[self.CurrentMoleculeID]
            if (self.IsoFlag):
                del self.AllIsotopologueList[self.CurrentMoleculeID]
            del self.MoleculesInMolfitFile[self.CurrentMoleculeID]
            del self.MolfitFileAllParameters[self.CurrentMoleculeID]


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## update pointer
            if (self.CurrentMoleculeID >= (len(self.MoleculesInMolfitFile) - 1)):
                self.CurrentMoleculeID = (len(self.MoleculesInMolfitFile) - 1)
            self.CurrentMolecule = self.MoleculesInMolfitFile[self.CurrentMoleculeID]
            self.CurrentMolfitIndex = 1
            self.CurrentComponentIndex = 0
            self.CurrentIsoID = 0

            # Debug:
            # print ("self.CurrentIsoID = ", self.CurrentIsoID)
            # print ("self.CurrentMolecule = ", self.CurrentMolecule)
            # print ("self.MoleculesInMolfitFile = ", self.MoleculesInMolfitFile)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## update GUI
            self.CompNumChangeFlag = True
            self.InitTab2Flag = True
            okStatus = self.Tab2NewXMLFile()
            self.Tab2setDisEnabled(True)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## call TFW GUI
    def MenueTFW(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if developer version is used:
        if (not self.devFlag):


            ## Not available
            title = "Not available"
            bodyText = "\nStill under development!" + "\n"
            QtWidgets.QMessageBox.warning(self, title, bodyText, QtWidgets.QMessageBox.Close)


        ## for developing usage only
        else:


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call transition frequency window GUI
            LocalSortIndex = self.SQLParamArray[4]
            StatusGUI, EmPosList, AbsPosList = TFW_Plot.PlotTFWS(self.CurrentMolecule, self.CurrentMoleculeDir, \
                                                                 self.velLowLimitWeight, self.velUpLimitWeight, self.AllSpectra, \
                                                                 self.IndexSortConversionList, LocalSortIndex, self.VelBinExtGUI, \
                                                                 self.NoiseLevelExtGUI, self.RemoveContFlagGUI, self.SmoothedFlagGUI, \
                                                                 self.SmoothValueGUI)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## call widgets
    def MenuExternalWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name of button
        sender = self.sender()
        objectName = sender.objectName()
        name = sender.text()

        # Debug:
        # print ("objectName = ", objectName)
        # print ("name = ", name)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## open QInputDialog for option "Set max. E_low"
        if (objectName == "actionSet_Elow"):
            widgetTitle = "Set max. E_low"
            bodyText = "Set new upper limit for E_low (K):"
            LocalElowMax = self.SQLParamArray[2]
            LocalElowMax, ok = QtWidgets.QInputDialog.getDouble(self, widgetTitle, bodyText, value = LocalElowMax, min = 0.0, max = 1.e5)

            # Debug:
            # print ("ok = ", ok)
            # print ("LocalElowMax = ", LocalElowMax)


            ## is "OK" button pressed?
            if (ok):
                OldValue = self.ElowMax
                self.ElowMax = LocalElowMax
                self.SQLParamArray[2] = self.ElowMax
                if (self.tabWidget.currentIndex() == 1):
                    self.CompNumChangeFlag = True
                    self.InitTab2Flag = True
                    okStatus = self.Tab2NewXMLFile()
                    self.Tab2setDisEnabled(True)
                else:
                    self.ObsXMLFileChangeFlag = True
                if (LocalElowMax != OldValue):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files
            else:
                return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## open QInputDialog for option "Set min. Intensity"
        elif (objectName == "actionSet_IMin"):
            widgetTitle = "Set min. Intensity"
            bodyText = "Set new lower limit for gA (s-1):"
            LocalIMin = self.SQLParamArray[3]
            LocalIMin, ok = QtWidgets.QInputDialog.getDouble(self, widgetTitle, bodyText, value = LocalIMin, decimals = 5, min = 0.0, max = 1.e5)

            # Debug:
            # print ("\n\nok = ", ok)
            # print ("LocalIMin = ", LocalIMin)


            ## is "OK" button pressed?
            if (ok):
                OldValue = self.IMin
                self.IMin = LocalIMin
                self.SQLParamArray[3] = LocalIMin
                if (self.tabWidget.currentIndex() == 1):
                    self.CompNumChangeFlag = True
                    self.InitTab2Flag = True
                    okStatus = self.Tab2NewXMLFile()
                    self.Tab2setDisEnabled(True)
                else:
                    self.ObsXMLFileChangeFlag = True
                if (LocalIMin != OldValue):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files
            else:
                return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## open QInputDialog for option "Limit number of transitions"
        elif (objectName == "actionSet_MaxNumTrans"):
            widgetTitle = "Limit number of transitions"
            bodyText = "Set max. number of transitions:"
            LocalTransNum, ok = QtWidgets.QInputDialog.getInt(self, widgetTitle, bodyText, value = self.MaxNumTransInFit, min = 1, max = 1.e5)

            # Debug:
            # print ("ok = ", ok)
            # print ("LocalTransNum = ", LocalTransNum)


            ## is "OK" button pressed?
            if (ok):
                OldValue = self.MaxNumTransInFit
                self.MaxNumTransInFit = LocalTransNum
                if (self.tabWidget.currentIndex() == 1):
                    self.InitTab2Flag = True
                    okStatus = self.Tab2NewXMLFile()
                    self.Tab2setDisEnabled(True)
                else:
                    self.ObsXMLFileChangeFlag = True
                if (LocalTransNum != OldValue):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files
            else:
                return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## open QDialog for option "Set new vel range"
        elif (objectName == "actionSet_VelRange"):
            ExternalWidgetName = "NewVelRange"
            OrigValue = [self.velLowLimitWeight, self.velUpLimitWeight]


            ## call external widget
            LoopFlag = True
            while LoopFlag:
                wid = ExternalWidget(self, ExternalWidgetName=ExternalWidgetName, OrigValue=OrigValue)
                wid.resize(400, 150)
                desktopSize = QtWidgets.QDesktopWidget().screenGeometry()
                top = int((desktopSize.height() / 2.0) - 75)
                left = int((desktopSize.width() / 2.0) - 200)
                wid.move(left, top)
                wid.exec_()


                ## get return parameter
                ok, NewRange = wid.getReturnparamWidget()

                # Debug:
                # print ("ok = ", ok)
                # print ("NewRange = ", NewRange)


                ## is "OK" button pressed?
                if (not ok):
                    LoopFlag = False
                    break
                else:


                    ## check, user input
                    LocalLog = []
                    try:
                        LocalLowLimit = float(NewRange[0])
                    except:
                        LocalLog.append("Lower limit is not a number!")
                    try:
                        LocalUpLimit = float(NewRange[1])
                    except:
                        LocalLog.append("Upper limit is not a number!")
                    if (LocalLog != []):
                        title = "Wrong user input"
                        bodyText = ""
                        for LocalLogLine in LocalLog:
                            bodyText += LocalLogLine + "\n"
                        bodyText += "\nPlease correct your input!" + "\n"
                        QtWidgets.QMessageBox.warning(self, title, bodyText, QtWidgets.QMessageBox.Close)


                    else:
                        if (LocalLowLimit > LocalUpLimit):
                            title = "Wrong user input"
                            bodyText = "The lower limit is higher than the upper limit!\n"
                            bodyText += "\nPlease correct your input!" + "\n"
                            QtWidgets.QMessageBox.warning(self, title, bodyText, QtWidgets.QMessageBox.Close)

                        elif (LocalLowLimit == LocalUpLimit):
                            title = "Wrong user input"
                            bodyText = "The lower limit is equal to the upper limit!\n"
                            bodyText += "\nPlease correct your input!" + "\n"
                            QtWidgets.QMessageBox.warning(self, title, bodyText, QtWidgets.QMessageBox.Close)

                        else:
                            self.velLowLimitWeight = LocalLowLimit
                            self.velUpLimitWeight = LocalUpLimit


                            ## determine new transition frequency windows
                            if (self.velLowLimitWeight < self.velLowLimit or self.velUpLimitWeight > self.velUpLimit):
                                self.velLowLimit = min(self.velLowLimitWeight, self.velLowLimit)
                                self.velUpLimit = max(self.velUpLimitWeight, self.velUpLimit)
                                if (self.tabWidget.currentIndex() == 1):
                                    okStatus = self.Tab2NewXMLFile()
                                    self.Tab2setDisEnabled(True)
                                else:
                                    self.ObsXMLFileChangeFlag = True
                            else:


                                ## update all parameters
                                if (self.tabWidget.currentIndex() == 1):
                                    self.Tab2SetValuesForTab()
                                else:
                                    self.ObsXMLFileChangeFlag = True


                            ## exit loop
                            LoopFlag = False
                            break


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## open QDialog for option "actionSet_subplots_per_figure"
        elif (objectName == "actionSet_subplots_per_figure"):
            widgetTitle = "Set number of subplots per figure"
            bodyText = "Set number of subplots per figure:"
            LocalMaxNumberSpectrumDefault, ok = QtWidgets.QInputDialog.getInt(self, widgetTitle, bodyText, value = self.MaxNumberSpectrumDefault, \
                                                                          min = 1, max = 99)
            # Debug:
            # print ("ok = ", ok)
            # print ("LocalMaxNumberSpectrumDefault = ", LocalMaxNumberSpectrumDefault)


            ## is "OK" button pressed?
            if (ok):
                OldValue = self.MaxNumberSpectrumDefault
                self.MaxNumberSpectrumDefault = LocalMaxNumberSpectrumDefault


                ## initialize parameterrs for plot
                self.TotalNumberSpectra = len(self.AllSpectra)
                self.MaxNumberSpectrum = min(self.MaxNumberSpectrumDefault, self.TotalNumberSpectra)
                self.SpectrumIndexList = list(range(0, self.MaxNumberSpectrum))
                self.ChangeSpectrumWindowFlag = True


                ## update tab 2
                if (self.tabWidget.currentIndex() == 1):
                    self.Tab2SetValuesForTab()

                if (LocalMaxNumberSpectrumDefault != OldValue):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files
            else:
                return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set local-overlap flag to true or false
        elif (objectName == "actionLocal_overlap"):
            if (self.UserLocalOverlapFlag):
                self.UserLocalOverlapFlag = False
            else:
                self.UserLocalOverlapFlag = True
            if (self.UserLocalOverlapFlag):
                self.actionLocal_overlap.setChecked(True)
            else:
                self.actionLocal_overlap.setChecked(False)


            ## update tab 2
            if (self.tabWidget.currentIndex() == 1):
                self.Tab2SetValuesForTab()
            self.Tab2RemoveBackgroundFiles()                                            ## remove background files


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set local-overlap flag to true or false
        elif (objectName == "actionCF_distance_ordering"):
            ParamChangeFlag = True


            ## check, if current molfit contains pure CF formalism
            PureCF = True
            DistanceList = []
            for CounterComp, Compparam in enumerate(self.CurrentMolfitParameter):
                CFFlag = Compparam[5][0]
                if (self.IsNumber(CFFlag)):
                    CFFlag = float(CFFlag)
                    PureCF = False
                DistanceList.append(CFFlag)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## convert CF to distance formalism
            if (PureCF):
                self.CFUsageFlag = False
                NumCompCont = 0
                for CurrentContinuumParameter in self.MolfitContinuumParameters:
                    LocalMoleculeParameters = CurrentContinuumParameter[1]
                    NumCompCont += len(LocalMoleculeParameters)
                NumComp = len(self.CurrentMolfitParameter) + NumCompCont
                HelperArray = copy.deepcopy(self.CurrentMolfitParameter)
                for CounterComp, Compparam in enumerate(HelperArray):
                    CFFlag = Compparam[5][0]
                    CFFlag = CFFlag.strip()
                    CFFlag = CFFlag.lower()
                    if (CFFlag in ["c", "e"]):
                        CFFlag = NumComp + 1
                    else:
                        CFFlag = ((NumComp + 1) - (NumCompCont + CounterComp + 1))
                    self.CurrentMolfitParameter[CounterComp][5][0] = CFFlag


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## convert distance to CF formalism
            else:


                ## check, if conversion is possible
                MaxDist = max(DistanceList)
                CopyDistanceList = copy.deepcopy(DistanceList)
                while MaxDist in CopyDistanceList: CopyDistanceList.remove(MaxDist)
                DoubleFlag = False
                for i in CopyDistanceList:
                    if (CopyDistanceList.count(i) > 1):
                        DoubleFlag = True
                        break
                if (DoubleFlag):
                    msgBox = QtWidgets.QMessageBox()
                    msgBox.setWindowTitle("WARNING")
                    LocalLogText = "The conversion to CF formalism is not possible because one or more new foreground components"
                    LocalLogText += " occur at the same distance.\n\n"
                    LocalLogText += "Please check the distanct definition and retry!\n"
                    msgBox.setText(LocalLogText)
                    ret = msgBox.exec_()
                    ParamChangeFlag = False


                ## convert distance to CF formalism
                else:
                    self.CFUsageFlag = True


                    ## sort molfit parameters according to distance parameter
                    self.CurrentMolfitParameter = sorted(self.CurrentMolfitParameter, key = lambda x: x[5][0], reverse = True)


                    ## convert CF flag
                    HelperArray = copy.deepcopy(self.CurrentMolfitParameter)
                    for CounterComp, Compparam in enumerate(HelperArray):
                        CFFlag = float(Compparam[5][0])
                        if (CFFlag == MaxDist):
                            CFFlag = "c"
                        else:
                            CFFlag = "f"
                        self.CurrentMolfitParameter[CounterComp][5][0] = CFFlag

                # Debug:
                # print ("self.CurrentMolfitParameter = ", self.CurrentMolfitParameter)


            ## update tab 2
            if (ParamChangeFlag):
                self.CompNumChangeFlag = True
                self.Tab2setDisEnabled(True)
                self.Tab2SetValuesForTab()
                self.Tab2RemoveBackgroundFiles()                                            ## remove background files


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set "ignore em. and abs. funcitons" to true or false
        elif (objectName == "actionIgnore_em_and_abs_functions"):
            if (self.EmAbsFlag):
                self.EmAbsFlag = False
            else:
                self.EmAbsFlag = True
            if (self.EmAbsFlag):
                self.actionIgnore_em_and_abs_functions.setChecked(False)
            else:
                self.actionIgnore_em_and_abs_functions.setChecked(True)


            ## update tab 2
            if (self.tabWidget.currentIndex() == 1):
                self.Tab2SetValuesForTab()
            self.Tab2RemoveBackgroundFiles()                                            ## remove background files


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## sort transition frequency spectra by different criteria
    def MenuSortSpectra(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name of button
        sender = self.sender()
        objectName = sender.objectName()
        name = sender.text()

        # Debug:
        # print ("objectName = ", objectName)
        # print ("name = ", name)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define sort index and conversion table for chosen sort criteria
        if (objectName == "actionSort_plots_by_frequency"):                                 ## menu "Sort plots by frequency .."
            self.SortIndex = 0

        elif (objectName ==  "actionSort_plots_by_E_low"):                                  ## menu "Sort plots by E_low .."
            self.SortIndex = 1

        elif (objectName == "actionSort_plots_by_gA"):                                      ## menu "Sort plots by gA .."
            self.SortIndex = 2

        elif (objectName == "actionSort_plots_by_E_low_and_gA"):                            ## menu "Sort plots by E_low and gA .."
            self.SortIndex = 3
        self.SQLParamArray[4] = self.SortIndex


        ## initialize parameterrs for plot
        self.TotalNumberSpectra = len(self.AllSpectra)
        self.MaxNumberSpectrum = min(self.MaxNumberSpectrumDefault, self.TotalNumberSpectra)
        self.SpectrumIndexList = list(range(0, self.MaxNumberSpectrum))
        self.ChangeSpectrumWindowFlag = True


        ## update tab 2
        if (self.tabWidget.currentIndex() == 1):
            self.Tab2SetValuesForTab()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create obs. xml file for current fit
    def MenuCreateObsXMLFile(self, NewObsXMLFileName):
        """

    input parameters:
    -----------------

        - NewObsXMLFileName:        path and name of new obs. xml file


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("NewObsXMLFileName = ", NewObsXMLFileName)


        ## get sort index
        LocalSortIndex = self.SQLParamArray[4]

        # Debug:
        # print ("LocalSortIndex = ", LocalSortIndex)


        ## get list of tfw
        LocalSpectrumIndexList = list(range(0, min(self.MaxNumTransInFit, self.TotalNumberSpectra)))

        # Debug:
        # print ("\n\n\nLocalSpectrumIndexList = ", LocalSpectrumIndexList)
        # print ("self.MaxNumTransInFit = ", self.MaxNumTransInFit)


        ## create figure
        RangeList = []
        for OrigSpectrumIndex in LocalSpectrumIndexList:


            ## depending on the sorting criteria, chose correct spectrum index
            SpectrumIndex = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][0]
            SpecTransFreq = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][1]
            SpecElow = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][2]
            Specgup = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][5]
            SpecEinsteinA = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][6]
            SpecMoleculeName = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][7]


            # self.AllSpectra[i]:   [LocalObsDataTransWindow, SmoothedData, DBParam, RangeParam, IntWithoutContinuum[In1:In2], RestFreq]


            ## get range parameters
            LocalTransWin = self.AllSpectra[SpectrumIndex]
            RangeParam = LocalTransWin[3]                                                   ## get range parameters
            TelescopeSize = RangeParam[0]
            BMIN = RangeParam[1]
            BMAJ = RangeParam[2]
            BPA = RangeParam[3]
            InterFlag = RangeParam[4]
            GlobalvLSR = RangeParam[5]
            Redshift = RangeParam[6]
            NewTBack = RangeParam[7]
            tSlope = RangeParam[8]
            LowFreq = RangeParam[9]
            UpFreq = RangeParam[10]
            nH = RangeParam[11]
            beta = RangeParam[12]
            kappa = RangeParam[13]
            LocalContPhenFuncID = RangeParam[14]
            LocalContPhenFuncParam1 = RangeParam[15]
            LocalContPhenFuncParam2 = RangeParam[16]
            LocalContPhenFuncParam3 = RangeParam[17]
            LocalContPhenFuncParam4 = RangeParam[18]
            LocalContPhenFuncParam5 = RangeParam[19]
            BackgroundFileName = RangeParam[20]
            DustFileName = RangeParam[21]
            ObsDataFileIndex = RangeParam[22]
            RangeIndex = RangeParam[23]
            LocalExpFileName = RangeParam[24]
            NumberHeaderLines = RangeParam[25]
            SeparatorColumns = RangeParam[26]
            FreqStep = RangeParam[27]
            DustFileName = RangeParam[28]
            BackgroundFileName = RangeParam[29]
            RangeList.append(RangeParam)


        ## sort range list according to min. and max. frequencies
        RangeList = sorted(RangeList, key = lambda x: (x[9], x[10]))


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## combine overlapping ranges


        ## check, if part of current freq. range is already covered by a previous range
        TotalNumberRanges = len(RangeList)
        NotIncludeRangesList = []
        NewRangeList = []
        for RangeParamID, RangeParam in enumerate(RangeList):
            if (not RangeParamID in NotIncludeRangesList):
                lf = RangeParam[9]
                uf = RangeParam[10]


                ## search for overlapping ranges
                for i in range(RangeParamID + 1, TotalNumberRanges):
                    if (not i in NotIncludeRangesList):
                        LowFreq = RangeList[i][9]
                        HighFreq = RangeList[i][10]


                        ## exit loop?
                        if (uf < LowFreq):
                            break


                        ## check case:  overlapping range definitons
                        ##  lf|                 |uf
                        ##          LowFreq|            |HighFreq
                        if (LowFreq <= uf and uf <= HighFreq):
                            RangeParam[10] = HighFreq
                            NotIncludeRangesList.append(i)


                        ## check case:  overlapping range definitons
                        ##  lf|                             |uf
                        ##          LowFreq|    |HighFreq
                        elif (lf <= LowFreq and HighFreq <= uf):
                            NotIncludeRangesList.append(i)


                        ## check case:  overlapping range definitons
                        ##          lf|                     |uf
                        ##  LowFreq|            |HighFreq
                        elif (LowFreq <= lf and HighFreq <= uf):
                            RangeParam[9] = LowFreq
                            tBack = RangeParam[7]
                            tSlope = RangeParam[8]
                            NewTBack = tBack * (LowFreq / lf)**tSlope
                            RangeParam[7] = NewTBack
                            NotIncludeRangesList.append(i)


                        ## check case:  overlapping range definitons
                        ##          lf|          |uf
                        ## LowFreq|                      |HighFreq
                        elif (LowFreq <= lf and uf <= HighFreq):
                            RangeParam[9] = LowFreq
                            RangeParam[10] = HighFreq
                            tBack = RangeParam[7]
                            tSlope = RangeParam[8]
                            NewTBack = tBack * (LowFreq / lf)**tSlope
                            RangeParam[7] = NewTBack
                            NotIncludeRangesList.append(i)


                ## add new range if range is not covered before
                if (abs(RangeParam[10] - RangeParam[9]) >= 2.0 * FreqStep):
                    NewRangeList.append(RangeParam)
                    NotIncludeRangesList.append(RangeParamID)


        ## sort range list according to obs. data file index, min. and max. frequencies
        NewRangeList = sorted(NewRangeList, key = lambda x: (x[22], x[9], x[10]))
        # NewRangeList = sorted(RangeList, key = lambda x: (x[28], x[9], x[10]))


        ## create new obs. xml file
        LocalListObsDataFileNames = []
        LocalNumberExpRanges = []
        LocalFreqMin = []
        LocalFreqMax = []
        LocalFreqStep = []
        LocalTbgFlag = []
        LocalTbg = []
        LocalTslope = []
        LocalnHFlag = []
        LocalnH = []
        Localbeta = []
        Localkappa = []
        LocalDustFileName = []
        LocalBackgroundFileName = []
        LocalContFuncID = []
        LocalContParam1 = []
        LocalContParam2 = []
        LocalContParam3 = []
        LocalContParam4 = []
        LocalContParam5 = []
        ListOfThresholds = []
        ListOfNoiseLevel = []
        LocalSizeTelescope = []
        LocalBMIN = []
        LocalBMAJ = []
        LocalBPA = []
        LocalInterferoFlag = []
        LocalvLSR = []
        LocalRedShift = []
        ErrorYList = []
        NumberHeaderLinesList = []
        SeparatorColumnsList = []
        ObsDataFileIndexList = []
        for x in NewRangeList:
            if (not x[22] in ObsDataFileIndexList):
                ObsDataFileIndexList.append(x[22])
        ObsDataFileIndexList.sort()
        for RangeParamID, RangeParam in enumerate(NewRangeList):
            TelescopeSize = RangeParam[0]
            BMIN = RangeParam[1]
            BMAJ = RangeParam[2]
            BPA = RangeParam[3]
            InterFlag = RangeParam[4]
            GlobalvLSR = RangeParam[5]
            Redshift = RangeParam[6]
            NewTBack = RangeParam[7]
            tSlope = RangeParam[8]
            LowFreq = RangeParam[9]
            UpFreq = RangeParam[10]
            nH = RangeParam[11]
            beta = RangeParam[12]
            kappa = RangeParam[13]
            LocalContPhenFuncID = RangeParam[14]
            LocalContPhenFuncParam1 = RangeParam[15]
            LocalContPhenFuncParam2 = RangeParam[16]
            LocalContPhenFuncParam3 = RangeParam[17]
            LocalContPhenFuncParam4 = RangeParam[18]
            LocalContPhenFuncParam5 = RangeParam[19]
            #    BackgroundFileName = RangeParam[20]
            #    DustFileName = RangeParam[21]
            OrigObsDataFileIndex = RangeParam[22]
            ObsDataFileIndex = ObsDataFileIndexList.index(OrigObsDataFileIndex)
            # RangeIndex = RangeParam[23]
            RangeIndex = [x[0] for x in LocalFreqMin].count(ObsDataFileIndex)
            LocalExpFileName = os.path.normpath(RangeParam[24])
            NumberHeaderLines = RangeParam[25]
            SeparatorColumns = RangeParam[26]
            FreqStep = RangeParam[27]
            DustFileName = RangeParam[28]
            BackgroundFileName = RangeParam[29]


            ## re-correct frequency limits
            LowFreq = task_myXCLASS.ConvertFreq(LowFreq, GlobalvLSR, z = Redshift)
            UpFreq = task_myXCLASS.ConvertFreq(UpFreq, GlobalvLSR, z = Redshift)


            ## create lists for subroutine
            element = [ObsDataFileIndex, LocalExpFileName]
            if (not element in LocalListObsDataFileNames):
                LocalListObsDataFileNames.append(element)


            element = [ObsDataFileIndex, TelescopeSize]
            if (not element in LocalSizeTelescope):
                LocalSizeTelescope.append(element)


            element = [ObsDataFileIndex, BMIN]
            if (not element in LocalBMIN):
                LocalBMIN.append(element)


            element = [ObsDataFileIndex, BMAJ]
            if (not element in LocalBMAJ):
                LocalBMAJ.append(element)


            element = [ObsDataFileIndex, BPA]
            if (not element in LocalBPA):
                LocalBPA.append(element)


            element = [ObsDataFileIndex, InterFlag]
            if (not element in LocalInterferoFlag):
                LocalInterferoFlag.append(element)


            element = [ObsDataFileIndex, GlobalvLSR]
            if (not element in LocalvLSR):
                LocalvLSR.append(element)


            element = [ObsDataFileIndex, Redshift]
            if (not element in LocalRedShift):
                LocalRedShift.append(element)


            element = [ObsDataFileIndex, NumberHeaderLines]
            if (not element in NumberHeaderLinesList):
                NumberHeaderLinesList.append(element)


            element = [ObsDataFileIndex, SeparatorColumns]
            if (not element in SeparatorColumnsList):
                SeparatorColumnsList.append(element)


            element = [ObsDataFileIndex, "no"]
            if (not element in ErrorYList):
                ErrorYList.append(element)


            ## define range lists
            if (RangeIndex >= 0):
                LocalFreqMin.append([ObsDataFileIndex, RangeIndex, LowFreq])
                LocalFreqMax.append([ObsDataFileIndex, RangeIndex, UpFreq])
                LocalFreqStep.append([ObsDataFileIndex, RangeIndex, FreqStep])
                LocalTbgFlag.append([ObsDataFileIndex, RangeIndex, True])
                LocalTbg.append([ObsDataFileIndex, RangeIndex, NewTBack])
                LocalTslope.append([ObsDataFileIndex, RangeIndex, tSlope])
                LocalnHFlag.append([ObsDataFileIndex, RangeIndex, False])
                LocalnH.append([ObsDataFileIndex, RangeIndex, nH])
                Localbeta.append([ObsDataFileIndex, RangeIndex, beta])
                Localkappa.append([ObsDataFileIndex, RangeIndex, kappa])
                LocalDustFileName.append([ObsDataFileIndex, RangeIndex, DustFileName])
                LocalBackgroundFileName.append([ObsDataFileIndex, RangeIndex, BackgroundFileName])
                LocalContFuncID.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncID])
                LocalContParam1.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncParam1])
                LocalContParam2.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncParam2])
                LocalContParam3.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncParam3])
                LocalContParam4.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncParam4])
                LocalContParam5.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncParam5])
                ListOfThresholds.append([ObsDataFileIndex, RangeIndex, 0.0])
                ListOfNoiseLevel.append([ObsDataFileIndex, RangeIndex, 0.0])


        for i in range(len(LocalListObsDataFileNames)):
            j = [x[0] for x in LocalFreqMin].count(i)
            element = [i, j]
            if (not element in LocalNumberExpRanges):
                LocalNumberExpRanges.append(element)

        # Debug:
        # print ("LocalFreqMin = ", LocalFreqMin)


        ## write obs. xml file
        LocalIsoRatioFileName = []
        task_myXCLASSFit.CreateExpXMLFile(NewObsXMLFileName, LocalListObsDataFileNames, LocalNumberExpRanges, LocalFreqMin, LocalFreqMax, \
                                          LocalFreqStep, LocalTbgFlag, LocalTbg, LocalTslope, LocalnHFlag, LocalnH, Localbeta, Localkappa, \
                                          LocalDustFileName, LocalBackgroundFileName, \
                                          LocalContFuncID, LocalContParam1, LocalContParam2, LocalContParam3, \
                                          LocalContParam4, LocalContParam5, ListOfThresholds, ListOfNoiseLevel, LocalSizeTelescope, \
                                          LocalBMIN, LocalBMAJ, LocalBPA, LocalInterferoFlag, LocalvLSR, LocalRedShift, ErrorYList, \
                                          NumberHeaderLinesList, SeparatorColumnsList, self.UseIsoFlag, LocalIsoRatioFileName, \
                                          self.dbList, self.NumModelPixelXXList, self.NumModelPixelYYList, \
                                          self.LocalOverlapFlagList, self.NoSubBeamFlagList, self.EmAbsPATHList)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## fit current molecule
    def MenuFit(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## disable elements of tab 2
        self.Tab2setDisEnabled(False)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create subdirectory for fit
        lt = time.localtime()
        LocalFitDir = self.CurrentMoleculeDir + "single-molecule-fit____" + time.strftime("%Y_%m_%d", lt) + "___" + time.strftime("%H-%M-%S", lt) + "/"
        cmdString = "mkdir -p " + LocalFitDir
        os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if moflit parameter values are within defined limits
        ParamChangeFlag = False
        ListParam = ["ss", "Temp", "Ntot", "vwidth", "voff"]
        for CompIndex in range(len(self.CurrentMolfitParameter)):
            for CounterParameter, Param in enumerate(ListParam):
                LocalFitFlag = self.CurrentMolfitParameter[CompIndex][CounterParameter][0]
                LocalLowerLimit = float(self.CurrentMolfitParameter[CompIndex][CounterParameter][1])
                LocalUpperLimit = float(self.CurrentMolfitParameter[CompIndex][CounterParameter][2])
                LocalParamVal = float(self.CurrentMolfitParameter[CompIndex][CounterParameter][3])

                # Debug:
                # print ("\n\nLocalFitFlag = ", LocalFitFlag)
                # print ("LocalLowerLimit = ", LocalLowerLimit)
                # print ("LocalUpperLimit = ", LocalUpperLimit)
                # print ("LocalParamVal = ", LocalParamVal)


                ## check lower limit
                if (LocalParamVal < LocalLowerLimit):
                    ParamChangeFlag = True
                    if (LocalParamVal != 0.0):
                        LocalLowerLimit = LocalParamVal - abs(LocalParamVal) * 0.1
                    # LocalLowerLimit = LocalParamVal
                    self.CurrentMolfitParameter[CompIndex][CounterParameter][1] = LocalLowerLimit


                ## check upper limit
                if (LocalParamVal > LocalUpperLimit):
                    ParamChangeFlag = True
                    if (LocalParamVal != 0.0):
                        LocalUpperLimit = LocalParamVal + abs(LocalParamVal) * 0.1
                    # LocalUpperLimit = LocalParamVal
                    self.CurrentMolfitParameter[CompIndex][CounterParameter][2] = LocalUpperLimit

                # Debug:
                # print ("self.CurrentMolfitParameter[CompIndex][CounterParameter][0:4] = ", self.CurrentMolfitParameter[CompIndex][CounterParameter][0:4])


        ## create new molfit file
        if (ParamChangeFlag):
            self.Tab2CreateMolfitFile()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if iso ratio parameter values are within defined limits
        if (self.UseIsoFlag):
            for LocalIsoID in range(len(self.CurrentIsoParam)):
                LocalLowerLimit = self.CurrentIsoParam[LocalIsoID][3]
                LocalUpperLimit = self.CurrentIsoParam[LocalIsoID][4]
                LocalParamVal = self.CurrentIsoParam[LocalIsoID][2]


                ## check lower limit
                if (LocalParamVal < LocalLowerLimit):
                    if (LocalParamVal == 0):
                        NewLowerLimit = -1.0
                    else:
                        NewLowerLimit = LocalParamVal - abs(LocalParamVal) * 0.1
                    self.CurrentIsoParam[LocalIsoID][3] = NewLowerLimit


                ## check upper limit
                if (LocalParamVal > LocalUpperLimit):
                    if (LocalParamVal == 0):
                        NewUpperLimit = 1.0
                    else:
                        NewUpperLimit = LocalParamVal + abs(LocalParamVal) * 0.1
                    self.CurrentIsoParam[LocalIsoID][4] = NewUpperLimit


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write current molfit file contents to file


        ## get sql parameters
        MinNumTransitionsSQL = self.SQLParamArray[0]
        MaxNumTransitionsSQL = self.SQLParamArray[1]
        # MaxNumTransitionsSQL = self.MaxNumTransInFit
        MaxElowSQL = self.SQLParamArray[2]
        # MaxElowSQL = self.ElowMax
        MingASQL = self.SQLParamArray[3]
        # MingASQL = self.IMin
        OrderTransSQL = self.SQLParamArray[4]
        # OrderTransSQL = self.SortIndex
        MolfitHeaderLinesList = task_myXCLASS.WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)

        # Debug:
        # print ("MinNumTransitionsSQL = ", MinNumTransitionsSQL)
        # print ("MaxNumTransitionsSQL = ", MaxNumTransitionsSQL)
        # print ("MaxElowSQL = ", MaxElowSQL)
        # print ("OrderTransSQL = ", OrderTransSQL)


        ## write parts of molfit file
        LocalMolfitFileName = LocalFitDir + "current.molfit"
        LocalMolfitFile = open(LocalMolfitFileName, "w")
        for line in MolfitHeaderLinesList:                                                  ## write header of molfit file
            LocalMolfitFile.write(line)
        for CurrentContinuumParameter in self.MolfitContinuumParameters:
            LocalMolfitFileForEachMolecule = CurrentContinuumParameter[2]
            for line in LocalMolfitFileForEachMolecule:                                     ## write continuum contributions of molfit file
                LocalMolfitFile.write(line + "\n")
        for line in self.CurrentMolfitFileContents:                                         ## write content of molfit file
            LocalMolfitFile.write(line + "\n")
        LocalMolfitFile.close()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## make a copy of obs. xml file
        LocalObsXMLFile = LocalFitDir + "SMF_obs.xml"
        self.MenuCreateObsXMLFile(LocalObsXMLFile)

        # cmdString = "cp " +  self.NewExpXMLFileName + " " + LocalObsXMLFile
        # os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check if iso ratio file is defined
        if (self.UseIsoFlag):


            ## write iso ratio file
            IsoRatioFile = self.Tab2CreateIsoRatioFileContents(self.CurrentIsoParam)
            if (IsoRatioFile is not None):
                LocalIsoRatioFileName = LocalFitDir + "iso-ratio.dat"
                LocalIsoRatioFile = open(LocalIsoRatioFileName, "w")
                for line in IsoRatioFile:
                    NewLine = line + "\n"
                    NewLine = NewLine.replace("\n\n", "\n")
                    LocalIsoRatioFile.write(NewLine)
                LocalIsoRatioFile.close()

                # Debug:
                # print ("LocalObsXMLFile = ", LocalObsXMLFile)


                ## update name and path of iso ratio file
                task_MAGIX.WriteXMLtagNEW(LocalObsXMLFile, "iso_flag", ["True"])
                task_MAGIX.WriteXMLtagNEW(LocalObsXMLFile, "Isotopologues", ["True"])
                task_MAGIX.WriteXMLtagNEW(LocalObsXMLFile, "IsoTableFileName", [LocalIsoRatioFileName])
            else:
                task_MAGIX.WriteXMLtagNEW(LocalObsXMLFile, "iso_flag", ["False"])
                task_MAGIX.WriteXMLtagNEW(LocalObsXMLFile, "Isotopologues", ["True"])


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## call fit routine to start myXCLASSFit function
        BestMolfitFile, IsoRatioFileContent = self.FitCore(LocalFitDir, LocalMolfitFileName, LocalObsXMLFile)

        # Debug:
        # print ("\n\nIsoRatioFileContent = ", IsoRatioFileContent)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get optimized molfit parameters
        self.CurrentMolfitFileContents = []
        for line in BestMolfitFile:
            self.CurrentMolfitFileContents.append(line.replace("\n", ""))
        MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(self.CurrentMolfitFileContents)
        LocalMolfitParameters = AllParameters[0]
        NewRangeFlag = False
        self.CurrentMolfitParameter = self.Tab2GetParameterFromMolfitFile(LocalMolfitParameters, NewRangeFlag)
        self.MolfitFileAllParameters[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentMolfitParameter)
        #self.AllMoleculesMolfitFiles[self.CurrentMoleculeID] = copy.deepcopy(MolfitFileForEachMolecule[0])


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get optimized iso ratios
        NewIsoRatioFileContent = []
        if (IsoRatioFileContent != []):
            for line in IsoRatioFileContent:
                StrippedLine = line.strip()
                i  = StrippedLine.find("%")
                if (i > (-1)):
                    StrippedLine = StrippedLine[:i].strip()
                if (StrippedLine != ""):
                    SplittedLine = StrippedLine.split()
                    if (len(SplittedLine) < 4):
                        LowLimit = 1.0
                        UpLimit = 1.0
                    else:
                        LowLimit = float(SplittedLine[3])
                        UpLimit = float(SplittedLine[4])
                    NewIsoRatioFileContent.append([SplittedLine[0], SplittedLine[1], SplittedLine[2], LowLimit, UpLimit])
            NewIsoParam = self.Tab2GetIsoRatiosFromFile(self.CurrentMolecule, NewIsoRatioFileContent)
            for IsoParamID, LocalIsotopologue in enumerate(self.CurrentIsoParam):
                Isotopologue = LocalIsotopologue[0].strip()
                IsoMaster = LocalIsotopologue[1].strip()
                NewEntry = []
                for NewIsotopologueLine in NewIsoParam:
                    NewIsotopologue = NewIsotopologueLine[0].strip()
                    NewIsoMaster = NewIsotopologueLine[1].strip()
                    if (Isotopologue == NewIsotopologue and IsoMaster == NewIsoMaster):
                        NewEntry = copy.deepcopy(NewIsotopologueLine)
                        break
                if (NewEntry != []):
                    self.CurrentIsoParam[IsoParamID] = copy.deepcopy(NewEntry)
            # self.CurrentIsoID = max(0, len(self.CurrentIsoParam) - 1)

            # Debug:
            # print ("\nIsoRatioFileContent = ", IsoRatioFileContent)
            # print ("self.CurrentIsoID = ", self.CurrentIsoID)
            # print ("self.CurrentIsoParam = ", self.CurrentIsoParam)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## enable elements of tab 2
        self.Tab2setDisEnabled(True)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update history arrays
        self.Tab2UpdateHistoryArray()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update GUI
        self.CompNumChangeFlag = True
        self.Tab2SetValuesForTab()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create snapshots of each transition window spectrum and save plot to file
    def MenuSnapshot(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None

        """


        ## call progress bar
        NumSpec = len(self.AllSpectra)
        TotalCalls = NumSpec + 1
        bar = ProgressBar(total=TotalCalls, widgetTitle = "Prepare snapshot for " + self.CurrentMolecule, bodyText = "Create Spectrum ..")
        bar.resize(400, 100)
        desktopSize = QtWidgets.QDesktopWidget().screenGeometry()
        top = int((desktopSize.height() / 2.0) - 50)
        left = int((desktopSize.width() / 2.0) - 200)
        bar.move(left, top)
        bar.show()


        ## update progress bar
        ProgressBarValue = 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return


        ## create subdirectory for snapshots
        lt = time.localtime()
        LocalSnapDir = self.CurrentMoleculeDir + "/snapshot____" + time.strftime("%Y_%m_%d", lt) + "___" + time.strftime("%H-%M-%S", lt) + "/"
        cmdString = "mkdir -p " + LocalSnapDir
        os.system(cmdString)


        ## create single plot for each plot
        if (self.SnapshotPlotFlag):


            ## get sort index
            LocalSortIndex = self.SQLParamArray[4]

            # Debug:
            # print ("\n\n\nself.SpectrumIndexList = ", self.SpectrumIndexList)
            # print ("LocalSortIndex = ", LocalSortIndex)
            # print ("self.IndexSortConversionList[LocalSortIndex] = ", self.IndexSortConversionList[LocalSortIndex])


            ## get list of tfw
            LocalSpectrumIndexList = list(range(0, min(self.MaxNumTransInFit, self.TotalNumberSpectra)))

            # Debug:
            # print ("\n\n\nLocalSpectrumIndexList = ", LocalSpectrumIndexList)
            # print ("self.MaxNumTransInFit = ", self.MaxNumTransInFit)


            ## create figure
            CounterSpectrum = 0
            for OrigSpectrumIndex in LocalSpectrumIndexList:


                ## depending on the sorting criteria, chose correct spectrum index
                SpectrumIndex = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][0]
                SpecTransFreq = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][1]
                SpecElow = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][2]
                Specgup = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][5]
                SpecEinsteinA = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][6]
                SpecMoleculeName = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][7]


                # self.AllSpectra[i]:   [LocalObsDataTransWindow, SmoothedData, DBParam, RangeParam, IntWithoutContinuum[In1:In2], RestFreq]


                ## print what you do
                print ("\r\tCreate Spectrum (" + str(OrigSpectrumIndex + 1) + "/" + str(NumSpec) + ") ..           ", end = ' ', flush = True)


                ## update progress bar
                ProgressBarValue += 1
                ok = bar.setValue(ProgressBarValue)
                if (not ok):
                    print ("\\n\\nAbort")
                    self.lock = 0
                    return


                ## create spectrum
                LocalTransWin = self.AllSpectra[SpectrumIndex]
                LocalTransWinSpectrum = LocalTransWin[0].copy()                             ## get spectrum with continuum
                if (self.RemoveContinuumFlag):
                    LocalTransWinSpectrum[:, 2] = LocalTransWin[4].copy()                   ## get spectrum without continuum
                RestFreq = LocalTransWin[5]
                LocalMolecularParameterList = LocalTransWin[7]
                RangeParam = LocalTransWin[3]                                               ## get range parameters
                LocalBackgroundFileFunction = RangeParam[21]
                TransFreq = []
                EinsteinA = []
                ElowMin = []
                gup = []
                ScalFactor = []
                MolIndexList = []
                if (self.UseIsoFlag):
                    # NumMolPerRange = int(LocalTransWin[6]) + 1
                    NumMolPerRange = int(len(self.CurrentIsoParam)) + 1
                else:
                    NumMolPerRange = 1
                LocalIsotopologue = self.PureMolNameVector[0]
                for MolIndex in range(NumMolPerRange):
                    LocalRatio = 1.0
                    if (MolIndex > 0):
                        LocalIsoParameters = self.CurrentIsoParam[MolIndex - 1]
                        LocalIsotopologue = LocalIsoParameters[0]
                        LocalIsoMaster = LocalIsoParameters[1]
                        LocalRatio = LocalIsoParameters[2]
                        # if (LocalIsotopologue.lower().find("global") == (-1)):
                        if (LocalIsoMaster == self.PureMolNameVector[0]):
                            GlobalIndexList = LocalIsoParameters[-1]
                            if (GlobalIndexList != []):
                                for CurrGlobalIndex in GlobalIndexList:
                                    LocalRatio = LocalRatio * self.CurrentIsoParam[CurrGlobalIndex[0]][2]**CurrGlobalIndex[1]

                                    # Debug:
                                    # print ("\n\n\nLocalIsotopologue = ", LocalIsotopologue)
                                    # print ("MolIndex = ", MolIndex)
                                    # print ("CurrGlobalIndex = ", CurrGlobalIndex)
                                    # print ("self.CurrentIsoParam[CurrGlobalIndex][2] = ", self.CurrentIsoParam[CurrGlobalIndex[0]][2])
                                    # print ("LocalRatio = ", LocalRatio)


                    ## extract molecular parameters for current range
                    ## LocalMolecularParameterList = [TransFreq, EinsteinA, Elow, gup, Mol-index]
                    for LocaldbParam in LocalMolecularParameterList:
                        LocalMolIndex = int(LocaldbParam[4])
                        if (self.PureMolNameVector[LocalMolIndex] == LocalIsotopologue):
                            TransFreq.append(LocaldbParam[0])
                            EinsteinA.append(LocaldbParam[1])
                            ElowMin.append(LocaldbParam[2])
                            gup.append(LocaldbParam[3])
                            ScalFactor.append(LocalRatio)
                            MolIndexList.append(MolIndex)
                    LocalIsotopologue = ""


                ## create figure
                figureSnap = plt.figure(figsize = (15, 10))
                figureSnap.clear()
                pylab.subplots_adjust(bottom = 0.06, top = 0.90, right = 0.98, left = 0.07, hspace = 0.1, wspace = 0.2)
                layerSnap = pylab.subplot(1, 1, 1)


                ## add plots to GUI
                layerSnap.plot(LocalTransWinSpectrum[:, 1], LocalTransWinSpectrum[:, 2], '-', linewidth = 2.0, drawstyle = 'steps-mid', label = "data")


                ## add background spectrum to GUI
                if (LocalBackgroundFileFunction is not None):
                    LocalBackgroundFunc = LocalBackgroundFileFunction(LocalTransWinSpectrum[:, 0])
                    layerSnap.plot(LocalTransWinSpectrum[:, 1], LocalBackgroundFunc[:], color = 'green', linestyle = 'dashed', linewidth = 2.0, \
                                   label = "background spectrum")


                ## determine XCLASS spectra
                LocalIntArray = self.Tab2CalculateXCLASS(RangeParam, TransFreq, EinsteinA, ElowMin, gup, LocalTransWinSpectrum, ScalFactor, MolIndexList)
                layerSnap.plot(LocalTransWinSpectrum[:, 1], LocalIntArray, 'r-', linewidth = 2.0, label = "fit")


                ## set y ticks
                layerSnap.set_xlim(self.velLowLimitWeight, self.velUpLimitWeight)
                layerSnap.set_ylabel(r"T$_{mb}$ [K]")
                layerSnap.set_xlabel("velocity [km/s]")
                layerSnap.grid(True)
                layerSnap.legend()


                ## add vertical line
                layerSnap.axvline(x = 0.0, color = 'orange', linewidth = 2, linestyle = '--')


                ## get quantum numbers
                LocalFreqMinList = SpecTransFreq - 1.0
                LocalFreqMaxList = SpecTransFreq + 1.0
                MinNumTransitionsSQL = 1
                MaxNumTransitionsSQL = 0
                MaxElowSQL = 1.e99
                MingASQL = 0.0
                OrderTransSQL = self.SQLParamArray[4]
                LocalSQLParamArray = [MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL]
                LocaldbFile = self.dbList[0]
                LocalMaxNumTransInFit = 1000
                GlobalTransFreqList, GlobalDBParamList = task_LineIdentification.GetTransFreq([SpecMoleculeName], LocalFreqMinList, LocalFreqMaxList, \
                                                                                              LocalSQLParamArray, LocaldbFile, LocalMaxNumTransInFit, \
                                                                                              IncludeQNFlag = True)
                QNUp = ""
                QNLow = ""
                for line in GlobalDBParamList:
                    LocalTransFreq = line[0]
                    LocalEinsteinA = line[1]
                    LocalElow = line[2]
                    Localgup = line[3]
                    if (abs(LocalTransFreq - SpecTransFreq) < 1.e-1 and abs(Localgup - Specgup) < 1):
                        QNUp = line[5].strip()
                        QNLow = line[6].strip()
                        break

                # Debug:
                # print ("GlobalDBParamList = ", GlobalDBParamList)
                # print ("QNUp = ", QNUp)
                # print ("QNLow = ", QNLow)


                ## add title
                TransFreqString = "%.3f" % SpecTransFreq
                EinsteinAString = "%.2e" % SpecEinsteinA
                gString = "%.2e" % Specgup
                gAString = "%.2e" % ((SpecEinsteinA * Specgup) / (max(0.1, SpecElow)**2))
                LocalTitelString = "Molecule: " + chr(34) + SpecMoleculeName + chr(34) + "\n"
                LocalTitelString += r"$\nu_{\rm Trans}$ = " + TransFreqString + " MHz,  "
                LocalTitelString += r"Einstein A = " + EinsteinAString + r" s$^{-1}$,  "
                LocalTitelString += r"$g_{\rm up}$ = " + gString + r" s$^{-1}$,  "
                LocalTitelString += r"E$_{\rm low}$ = " + str(SpecElow) + " K,  "
                LocalTitelString += r"gA/E$_{\rm low}^2$ = " + gAString + r" s$^{-1}$"
                if (QNUp != "" and QNLow != ""):
                    LocalTitelString += "\nupper state: " + chr(34) + QNUp + chr(34) + ",     lower state: " + chr(34) + QNLow + chr(34)
                layerSnap.set_title(LocalTitelString)


                ## save plot to file
                LocalFileName = LocalSnapDir + "TW-Spectrum___vt__" + TransFreqString + "_MHz___gAE-2__" + gAString + "_s-1.png"
                figureSnap.savefig(LocalFileName, dpi=300)
                plt.close(figureSnap)


        ## get sql parameters
        MinNumTransitionsSQL = self.SQLParamArray[0]
        MaxNumTransitionsSQL = self.SQLParamArray[1]
        MaxElowSQL = self.SQLParamArray[2]
        MingASQL = self.SQLParamArray[3]
        OrderTransSQL = self.SQLParamArray[4]
        MolfitHeaderLinesList = task_myXCLASS.WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)


        ## write current molfit file contents to file
        LocalMolfitFile = open(LocalSnapDir + "current.molfit", "w")
        for line in MolfitHeaderLinesList:                                                  ## write header of molfit file
            LocalMolfitFile.write(line)
        for line in self.CurrentMolfitFileContents:
            LocalMolfitFile.write(line + "\n")
        LocalMolfitFile.close()


        ## write current iso ratio file contents to file
        if (self.UseIsoFlag):
            IsoRatioFileContents = self.Tab2CreateIsoRatioFileContents(self.CurrentIsoParam)
            LocalIsoRatioFile = open(LocalSnapDir + "iso-ratio.dat", "w")
            NewLine = "%-45s %-45s %25s %25s %25s" % ("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
            LocalIsoRatioFile.write(NewLine + "\n")
            for IsoLine in IsoRatioFileContents:
                LocalIsoRatioFile.write(IsoLine + "\n")
            LocalIsoRatioFile.close()


        ## we're done
        bar.close()
        print ("\r\t                                                                         ", flush = True)


        ## start eog
        # cmdString = "eog " + LocalFileName
        # os.system(cmdString)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## display some help information
    def MenuHelpMe(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None

        """


        ## update history arrays
        title = "Help me"
        bodyText = "\n\nNot yet implemented!\n\n"
        msgBox = QtWidgets.QMessageBox()
        msgBox.setWindowTitle(title)
        msgBox.setText(bodyText)
        ret = msgBox.exec_()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print some general informations about XCLASS GUI
    def MenuAboutXCLASSGUI(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None

        """


        ## The QWidget widget is the base class of all user interface objects in PyQt4.
        w = QtWidgets.QWidget()


        ## print some general informations about XCLASS GUI
        title = "About XCLASS GUI"
        bodyText = "<center>XCLASS GUI<br/>beta-version 1.0<br/><br/>(C) 2016 - 2024<br/><br/>Author:<br/>Thomas M&ouml;ller<br/><br/></center>"
        msgBox = QtWidgets.QMessageBox()
        msgBox.setWindowTitle(title)
        msgBox.setText(bodyText)
        ret = msgBox.exec_()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##====================================================================================================================================================
    ##
    ##  subroutines for tab 1
    ##
    ##====================================================================================================================================================


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get list of all molecules
    def Tab1GetMoleculesFromDatabase(self, AlreadyUsedMolecules, dbFilename, ListOfAllFreqMin, ListOfAllFreqMax, SQLParamArray, FastFlag):
        """

    input parameters:
    -----------------

        - AlreadyUsedMolecules: list of already used molecules

        - QueryString:          query string send to database

        - ListOfAllFreqMin:     list of lower frequencies

        - ListOfAllFreqMax:     list of upper frequencies

        - SQLParamArray:        sql parameter vector

        - FastFlag:             flag for fast query



    output parameters:
    ------------------

        - LocalContentsList:    list of all molecules in database
        """

        # Debug:
        # print ("AlreadyUsedMolecules = ", AlreadyUsedMolecules)
        # print ("dbFilename = ", dbFilename)
        # print ("ListOfAllFreqMin = ", ListOfAllFreqMin)
        # print ("ListOfAllFreqMax = ", ListOfAllFreqMax)
        # print ("SQLParamArray = ", SQLParamArray)
        # print ("FastFlag = ", FastFlag)


        ## define some parameters for database
        NameOfRadTransTable = "transitions"
        ColumnNameForNameTransitions = "T_Name"
        ColumnNameForFreqTransitions = "T_Frequency"
        ColumnNameForIntTransitions = "T_Intensity"
        ColumnNameForEinsteinATransitions = "T_EinsteinA"
        ColumnNameForFreqErrTransitions = "T_Uncertainty"
        ColumnNameForELowTransitions = "T_EnergyLower"
        ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## connect to database
        try:
            conn = sqlite3.connect(dbFilename)

            # Debug:
            # print ("\nConnection to sqlite3 database %s established." % dbFilename)
        except sqlite3.Error as e:
            print ("\nCan not connect to sqlite3 databse %s." % dbFilename)
            print ("Error: %d: %s" % (e.args[0], e.args[1]))
            sys.exit(1)


        ## get sql parameters
        ElowMax = SQLParamArray[2]
        IMin = SQLParamArray[3]
        if (IMin == 0.0):
            IMin = 1.e-9


        ## construct query string for transition table
        if (FastFlag):
            query_string = "select PF_Name from Partitionfunctions"
        else:
            query_string = "SELECT DISTINCT " + ColumnNameForNameTransitions + " FROM " + NameOfRadTransTable + " WHERE ("
            NumberRanges = len(ListOfAllFreqMin)
            for RangeIndex in range(NumberRanges):
                FreqMin = ListOfAllFreqMin[RangeIndex]
                FreqMax = ListOfAllFreqMax[RangeIndex]
                query_string += "(" + ColumnNameForFreqTransitions + " >= " + str(FreqMin)
                query_string += " AND " + ColumnNameForFreqTransitions + " <= " + str(FreqMax)
                query_string += " AND " + ColumnNameForELowTransitions + " <= " + str(ElowMax / 1.42879)
                query_string += " AND (" + ColumnNameForgUpTransitions + " * " + ColumnNameForEinsteinATransitions + ") >= " + str(IMin) + ")"
                if (RangeIndex < (NumberRanges - 1)):
                    query_string += " OR "
            query_string += ") ORDER by " + ColumnNameForNameTransitions

        # Debug:
        # print ("query_string = ", query_string)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read data from database
        cursor = conn.cursor()
        cursor.execute(query_string)
        conn.commit()
        rows = cursor.fetchall()
        Contents = rows
        conn.close()


        ## sort molecule list
        LocalContentsList = []
        for line in Contents:
            # LocalLine = line[0].encode('ascii','ignore').strip()
            LocalLine = line[0].strip()
            if (LocalLine != ""):
                if (not LocalLine in AlreadyUsedMolecules):
                    LocalContentsList.append(LocalLine)
        LocalContentsList.sort()


        ## define return variables
        return LocalContentsList
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## enables or disables elements of tab 1
    def Tab1setDisEnabled(self, DisEnabledFlag):
        """

    input parameters:
    -----------------

        - DisEnabledFlag:       flag indicating if elements are enabled (True) or disabled (False)


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("DisEnabledFlag = ", DisEnabledFlag)
        # print ("self.UseIsoFlag = ", self.UseIsoFlag)
        # print ("self.CurrentIsoParam = ", self.CurrentIsoParam)


        ## enable / disable elements of tab 1 (molfit file)
        self.lineEdit_PathMolfit_Tab1.setEnabled(DisEnabledFlag)
        self.pushButton_SelectMolfit_Tab1.setEnabled(DisEnabledFlag)


        ## enable / disable elements of tab 1 (algorithm file)
        self.comboBox_MoleculeInput.setEnabled(DisEnabledFlag)
        self.lineEdit_PathAlgXML_Tab1.setEnabled(DisEnabledFlag)
        self.pushButton_SelectAlgXML_Tab1.setEnabled(DisEnabledFlag)
        self.pushButton_SelectAlgorithm.setEnabled(DisEnabledFlag)


        ## enable / disable elements of tab 1 (obs. xml file)
        self.lineEdit_PathObsXML_Tab1.setEnabled(DisEnabledFlag)
        self.pushButton_SelectObsXML_Tab1.setEnabled(DisEnabledFlag)
        self.comboBox_DataFileSelection.setEnabled(DisEnabledFlag)
        self.pushButton_AddObsFile.setEnabled(DisEnabledFlag)
        self.pushButton_RemoveObsFile.setEnabled(DisEnabledFlag)
        self.lineEdit_PathObsData.setEnabled(DisEnabledFlag)
        self.pushButton_SelectObsData.setEnabled(DisEnabledFlag)
        self.lineEdit_SizeTele.setEnabled(DisEnabledFlag)
        self.checkBox_Interfer.setEnabled(DisEnabledFlag)
        self.lineEdit_vLSR.setEnabled(DisEnabledFlag)
        self.lineEdit_Redshift.setEnabled(DisEnabledFlag)
        self.comboBox_RangeSelection.setEnabled(DisEnabledFlag)
        self.pushButton_SelectFreqRange.setEnabled(DisEnabledFlag)
        self.lineEdit_MinFreq.setEnabled(DisEnabledFlag)
        self.lineEdit_MaxFreq.setEnabled(DisEnabledFlag)
        self.lineEdit_Tbg.setEnabled(DisEnabledFlag)
        self.lineEdit_TSlope.setEnabled(DisEnabledFlag)
        self.checkBox_Tbg_Flag.setEnabled(DisEnabledFlag)
        self.lineEdit_nH.setEnabled(DisEnabledFlag)
        self.lineEdit_kappa.setEnabled(DisEnabledFlag)
        self.lineEdit_beta.setEnabled(DisEnabledFlag)
        self.lineEdit_PathIsoFile.setEnabled(DisEnabledFlag)
        self.pushButton_SelectIso.setEnabled(DisEnabledFlag)


        ## define return variables
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set all values in tab 1
    def Tab1SetValues(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## activate lock
        self.lock = 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set path and name of molfit file (if selected)
        if (self.MolfitFileName.strip() != ""):
            self.lineEdit_PathMolfit_Tab1.setEnabled(True)
            self.lineEdit_PathMolfit_Tab1.setText(self.MolfitFileName)
            self.comboBox_MoleculeInput.setEnabled(False)
            self.comboBox_MoleculeInput.clear()
            self.comboBox_MoleculeInput.addItems([])


        ## if path and name of molfit file is not defined, enable molecule combo box
        elif (self.MolfitFileName.strip() == ""):
            self.comboBox_MoleculeInput.setEnabled(True)
            self.comboBox_MoleculeInput.clear()


            ## define list of all known molecules used in combo box of tab 1
            if (self.UserMoleculelist == ["Select molecule .."]):
                self.UserMoleculelist = []
                QueryString = "select PF_Name from Partitionfunctions"
                AlreadyUsedMolecules = []
                ListOfAllFreqMin = [1.0]
                ListOfAllFreqMax = [1.e9]
                if (self.MolfitFileName.strip() == ""):
                    FastFlag = True
                    self.UserMoleculelist = self.Tab1GetMoleculesFromDatabase(AlreadyUsedMolecules, self.DefaultDB, ListOfAllFreqMin, ListOfAllFreqMax, \
                                                                              self.SQLParamArray, FastFlag)
                self.UserMoleculelist = ["Select molecule .."] + self.UserMoleculelist


            ## add list of molecules
            self.comboBox_MoleculeInput.addItems(self.UserMoleculelist)
            if (self.UserMoleculelist != [] and self.UserCurrentMolecule.strip() != ""):
                self.UserCurrentMoleculeID = self.UserMoleculelist.index(self.UserCurrentMolecule)
                self.comboBox_MoleculeInput.setCurrentIndex(self.UserCurrentMoleculeID)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print settings for obs. data import


        ## import xml settings
        if (self.ObsXMLFileName.strip() != "" and self.UserFreqMin == [[0, 0, None]]):
            self.Tab1ImportObsXMLFile()


        ## path and name of obs. xml file
        self.lineEdit_PathObsXML_Tab1.setEnabled(True)
        self.lineEdit_PathObsXML_Tab1.setText(self.ObsXMLFileName)


        ## obs. data file import
        self.lineEdit_PathObsData.setEnabled(True)
        self.lineEdit_PathObsData.setText(self.UserObsDataFileName[self.ObsDataFileID])
        self.pushButton_SelectObsData.setEnabled(True)
        self.pushButton_AddObsFile.setEnabled(True)
        self.pushButton_SelectFreqRange.setEnabled(True)


        ## define content of comboBox comboBox_DataFileSelection
        self.comboBox_DataFileSelection.setEnabled(True)
        self.comboBox_DataFileSelection.clear()
        # self.DataFileNameList = ["add obs. data file .."]
        self.DataFileNameList = []
        for LocalDataFileID, LocalDataFileName in enumerate(self.UserObsDataFileName):
            if (LocalDataFileName is None or LocalDataFileName.strip() == ""):
                self.DataFileNameList.append("                 ")
            else:
                self.DataFileNameList.append("Data file " + str(LocalDataFileID + 1) + ": " + LocalDataFileName.split("/")[-1])
        self.comboBox_DataFileSelection.addItems(self.DataFileNameList)
        self.comboBox_DataFileSelection.setCurrentIndex(self.ObsDataFileID)


        ## deactivate remove button if only one data file is defined
        if (len(self.UserObsDataFileName) > 1):
            self.pushButton_RemoveObsFile.setEnabled(True)
        else:
            self.pushButton_RemoveObsFile.setEnabled(False)


        ## define content of comboBox RangeSelection
        self.comboBox_RangeSelection.setEnabled(True)
        self.comboBox_RangeSelection.clear()
        self.FrequencyRange = []


        ## get number of header lines and character separating columns from obs. xml file
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, RangeIndex, NumberRangeListIn = self.UserNumberExpRanges)
        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']            ## get number of frequency ranges for current obs. data file
        for RangeIndex in range(NumberFrequencyRanges):


            ## get parameter for current frequency range
            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, RangeIndex, FreqMinListIn = self.UserFreqMin, \
                                                                             FreqMaxListIn = self.UserFreqMax)
            FreqMin = ObsXMLParameterDictRange['FreqMin']
            if (FreqMin is not None):
                FreqMax = ObsXMLParameterDictRange['FreqMax']
                MinRangeString = "%.2f" % float(FreqMin)
                MaxRangeString = "%.2f" % float(FreqMax)
                self.FrequencyRange.append("Range " + str(RangeIndex + 1) + ": " + MinRangeString + " MHz - " + MaxRangeString + " MHz")
            else:
                self.FrequencyRange.append("                 ")
        self.comboBox_RangeSelection.addItems(self.FrequencyRange)
        self.comboBox_RangeSelection.setCurrentIndex(self.FrequencyRangeID)


        ## set content of line edit box "lineEdit_MinFreq"
        self.lineEdit_MinFreq.setEnabled(True)
        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, self.FrequencyRangeID, FreqMinListIn = self.UserFreqMin)
        Local_FreqMin = ObsXMLParameterDictRange['FreqMin']
        if (Local_FreqMin is None):
            Local_FreqMin = 0.0
        LineEditString = "%.2f" % float(Local_FreqMin)
        self.lineEdit_MinFreq.setText(LineEditString)


        ## set content of line edit box "lineEdit_MaxFreq"
        self.lineEdit_MaxFreq.setEnabled(True)
        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, self.FrequencyRangeID, FreqMaxListIn = self.UserFreqMax)
        Local_FreqMax = ObsXMLParameterDictRange['FreqMax']
        if (Local_FreqMax is None):
            Local_FreqMax = 0.0
        LineEditString = "%.2f" % float(Local_FreqMax)
        self.lineEdit_MaxFreq.setText(LineEditString)


        ## set content of line edit box "lineEdit_SizeTele"
        self.lineEdit_SizeTele.setEnabled(True)
        RangeIndex = (-1)


        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, RangeIndex, TelescopeSizeListIn = self.UserSizeTelescope)
        Local_TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
        if (Local_TelescopeSize is None):
            Local_TelescopeSize = 1.0
        LineEditString = "%.2f" % float(Local_TelescopeSize)
        self.lineEdit_SizeTele.setText(LineEditString)


        ## set content of check box "checkBox_Interfer"
        self.checkBox_Interfer.setEnabled(True)
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, RangeIndex, InterFlagListIn = self.UserInterferoFlag)
        Local_InterFlag = ObsXMLParameterDictFile['InterFlag']
        if (Local_InterFlag is not None):
            self.checkBox_Interfer.setChecked(eval(str(Local_InterFlag)))
        else:
            self.checkBox_Interfer.setChecked(False)


        ## set content of line edit box "lineEdit_vLSR"
        self.lineEdit_vLSR.setEnabled(True)
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, RangeIndex, GlobalvLSRListIn = self.UservLSR)
        Local_GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
        if (Local_GlobalvLSR is None):
            Local_GlobalvLSR = 0.0
        LineEditString = "%.4f" % float(Local_GlobalvLSR)
        self.lineEdit_vLSR.setText(LineEditString)


        ## set content of line edit box "lineEdit_Redshift"
        #    self.lineEdit_Redshift.setEnabled(True)
        #    RangeIndex = (-1)
        #    ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, RangeIndex, Redshift_ListIn = self.RedshiftList)
        #    Local_Redshift = ObsXMLParameterDictFile['Redshift']
        #    if (Local_Redshift is None):
        #        Local_Redshift = 0.0
        #    LineEditString = "%.4f" % float(Local_Redshift)
        #    self.lineEdit_Redshift.setText(LineEditString)


        ## set content of check box "checkBox_Tbg_Flag"
        self.checkBox_Tbg_Flag.setEnabled(True)
        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, self.FrequencyRangeID, tBackFlagListIn = self.UserTbgFlag)
        Local_t_back_flag = ObsXMLParameterDictRange['t_back_flag']
        if (Local_t_back_flag is not None):
            self.checkBox_Tbg_Flag.setChecked(eval(str(Local_t_back_flag)))
        else:
            self.checkBox_Tbg_Flag.setChecked(False)


        ## set content of line edit box "lineEdit_Tbg"
        self.lineEdit_Tbg.setEnabled(True)
        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, self.FrequencyRangeID, tBackListIn = self.UserTbg)
        Local_tBack = ObsXMLParameterDictRange['tBack']
        if (Local_tBack is not None):
            LineEditString = "%.2f" % float(Local_tBack)
            self.lineEdit_Tbg.setText(LineEditString)


        ## set content of line edit box "lineEdit_TSlope"
        self.lineEdit_TSlope.setEnabled(True)
        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, self.FrequencyRangeID, tSlopeListIn = self.UserTslope)
        Local_tSlope = ObsXMLParameterDictRange['tSlope']
        if (Local_tSlope is not None):
            LineEditString = "%.2f" % float(Local_tSlope)
            self.lineEdit_TSlope.setText(LineEditString)


        ## set content of line edit box "lineEdit_nH"
        self.lineEdit_nH.setEnabled(True)
        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, self.FrequencyRangeID, N_HListIn = self.UsernH)
        Local_N_H = ObsXMLParameterDictRange['N_H']
        if (Local_N_H is not None):
            LineEditString = "%.2e" % float(Local_N_H)
            self.lineEdit_nH.setText(LineEditString)


        ## set content of line edit box "Userkappa"
        self.lineEdit_kappa.setEnabled(True)
        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, self.FrequencyRangeID, kappa_1300ListIn = self.Userkappa)
        Local_kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
        if (Local_kappa_1300 is not None):
            LineEditString = "%.4f" % float(Local_kappa_1300)
            self.lineEdit_kappa.setText(LineEditString)


        ## set content of line edit box "lineEdit_beta"
        self.lineEdit_beta.setEnabled(True)
        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, self.FrequencyRangeID, beta_dustListIn = self.Userbeta)
        Local_beta_dust = ObsXMLParameterDictRange['beta_dust']
        if (Local_beta_dust is not None):
            LineEditString = "%.4f" % float(Local_beta_dust)
            self.lineEdit_beta.setText(LineEditString)


        ## set content of check box "checkBox_nH_Flag"
        #    self.checkBox_nH_Flag.setEnabled(True)
        #    ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, self.FrequencyRangeID, nHFlagListIn = self.UsernHFlag)
        #    Local_nH_flag = ObsXMLParameterDictRange['nH_flag']
        #    if (Local_nH_flag is None or Local_N_H is None or Local_kappa_1300 is None or Local_beta_dust is None):
        #        self.checkBox_nH_Flag.setChecked(False)
        #    else:
        #        self.checkBox_nH_Flag.setChecked(eval(str(Local_nH_flag)))




        ##!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        ##
        ##
        ##  to do:
        ##  ------
        ##
        ##
        ##  7.)  add lineEdit elements for ContFuncID
        ##
        ##  8.)  add lineEdit elements for ContParam1
        ##
        ##  9.)  add lineEdit elements for ContParam2
        ##
        ##  10.) add lineEdit elements for ContParam3
        ##
        ##  11.) add lineEdit elements for ContParam4
        ##
        ##  12.) add lineEdit elements for ContParam5
        ##
        ##  13.) add lineEdit elements for redshift
        ##
        ##  14.) add lineEdit elements for NumModelPixelXX
        ##
        ##  15.) add lineEdit elements for NumModelPixelYY
        ##
        ##  16.) add checkBox for LocalOverlapFlag
        ##
        ##  17.) add checkBox for NoSubBeamFlagList
        ##
        ##
        ##!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!




        ## set content of check box "checkBox_UseIso"
        self.checkBox_UseIso.setEnabled(True)
        if (self.UserUseIsoFlag is not None):
            self.checkBox_UseIso.setChecked(self.UserUseIsoFlag)
        else:
            self.checkBox_UseIso.setChecked(False)


        ## set content of line edit box "lineEdit_PathIsoFile"
        self.lineEdit_PathIsoFile.setEnabled(True)
        if (self.UserUseIsoFlag):
            self.lineEdit_PathIsoFile.setText(self.UserIsoRatioFileName)
        else:
            self.lineEdit_PathIsoFile.setText("")
        self.pushButton_SelectIso.setEnabled(True)


        ## if path and name of obs. xml file is not defined, enable elements for user input
        if (self.ObsXMLFileName.strip() == ""):
            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, self.FrequencyRangeID, tBackListIn = self.UserTbg)
            Local_tBack = ObsXMLParameterDictRange['tBack']
            if (Local_tBack is None):
                Local_tBack = 0.0
            LineEditString = "%.4f" % float(Local_tBack)
            self.ErrorYList = [[0, "no"]]
            self.NumberHeaderLinesList = [[0, "0"]]
            self.SeparatorColumnsList = [[0, ""]]



        ## create (well formatted) obs. xml file
        self.ObsXMLFileName = self.GUIJobDir + "obs.xml"
        ListOfThresholds = 0.0
        ListOfNoiseLevel = 0.0
        LocalListObsDataFileNames = []
        for LocalFileID, LocalFile in enumerate(self.UserObsDataFileName):
            LocalListObsDataFileNames.append([LocalFileID, LocalFile])
        task_myXCLASSFit.CreateExpXMLFile(self.ObsXMLFileName, LocalListObsDataFileNames, self.UserNumberExpRanges, self.UserFreqMin, self.UserFreqMax, \
                                          self.UserFreqStep, self.UserTbgFlag, self.UserTbg, self.UserTslope, self.UsernHFlag, self.UsernH, \
                                          self.Userbeta, self.Userkappa, self.UserDustFileName, self.UserBackgroundFileName, \
                                          self.UserContFuncID, self.UserContParam1, \
                                          self.UserContParam2, self.UserContParam3, self.UserContParam4, self.UserContParam5, ListOfThresholds, \
                                          ListOfNoiseLevel, self.UserSizeTelescope, self.UserBMIN, self.UserBMAJ, self.UserBPA, self.UserInterferoFlag, \
                                          self.UservLSR, self.UserRedShift, self.ErrorYList, self.NumberHeaderLinesList, self.SeparatorColumnsList, \
                                          self.UserUseIsoFlag, self.UserIsoRatioFileName, self.dbList, \
                                          [self.UserNumModelPixelXX], [self.UserNumModelPixelYY], [self.UserLocalOverlapFlag], [self.UserNoSubBeamFlag],
                                          [self.UserEmAbsPATH])
        # Debug:
        # print ("\n\nself.ObsXMLFileName = ", self.ObsXMLFileName)
        # print ("LocalListObsDataFileNames = ", LocalListObsDataFileNames)
        # print ("self.UserNumberExpRanges = ", self.UserNumberExpRanges)
        # print ("self.UserFreqMin = ", self.UserFreqMin)
        # print ("self.UserFreqMax = ", self.UserFreqMax)
        # print ("self.UserFreqStep = ", self.UserFreqStep)
        # print ("self.UserTbgFlag = ", self.UserTbgFlag)
        # print ("self.UserTbg = ", self.UserTbg)
        # print ("self.UserTslope = ", self.UserTslope)
        # print ("self.UsernHFlag = ", self.UsernHFlag)
        # print ("self.UsernH = ", self.UsernH)
        # print ("self.Userbeta = ", self.Userbeta)
        # print ("self.Userkappa = ", self.Userkappa)
        # print ("self.UserDustFileName = ", self.UserDustFileName)
        # print ("self.UserBackgroundFileName = ", self.UserBackgroundFileName)
        # print ("self.UserContFuncID = ", self.UserContFuncID)
        # print ("self.UserContParam1 = ", self.UserContParam1)
        # print ("self.UserContParam2 = ", self.UserContParam2)
        # print ("self.UserContParam3 = ", self.UserContParam3)
        # print ("self.UserContParam4 = ", self.UserContParam4)
        # print ("self.UserContParam5 = ", self.UserContParam5)
        # print ("ListOfThresholds = ", ListOfThresholds)
        # print ("ListOfNoiseLevel = ", ListOfNoiseLevel)
        # print ("self.UserSizeTelescope = ", self.UserSizeTelescope)
        # print ("self.UserBMIN = ", self.UserBMIN)
        # print ("self.UserBMAJ = ", self.UserBMAJ)
        # print ("self.UserBPA = ", self.UserBPA)
        # print ("self.UserInterferoFlag = ", self.UserInterferoFlag)
        # print ("self.UservLSR = ", self.UservLSR)
        # print ("self.UserRedShift = ", self.UserRedShift)
        # print ("self.ErrorYList = ", self.ErrorYList)
        # print ("self.NumberHeaderLinesList = ", self.NumberHeaderLinesList)
        # print ("self.SeparatorColumnsList = ", self.SeparatorColumnsList)
        # print ("self.UserUseIsoFlag = ", self.UserUseIsoFlag)
        # print ("self.UserIsoRatioFileName = ", self.UserIsoRatioFileName)
        # print ("self.dbList = ", self.dbList)
        # print ("[self.UserNumModelPixelXX] = ", self.UserNumModelPixelXX)
        # print ("[self.UserNumModelPixelYY] = ", self.UserNumModelPixelYY)
        # print ("[self.UserLocalOverlapFlag] = ", self.UserLocalOverlapFlag)
        # print ("[self.UserNoSubBeamFlag] = ", self.UserNoSubBeamFlag)
        # print ("[self.UserEmAbsPATH] = ", self.UserEmAbsPATH)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set path and name of alg. xml file (if selected)
        if (self.AlgXMLFileName.strip() != ""):
            self.lineEdit_PathAlgXML_Tab1.setEnabled(True)
            self.lineEdit_PathAlgXML_Tab1.setText(self.AlgXMLFileName)
            self.lineEdit_MaxIterLM.setEnabled(False)
            self.lineEdit_MaxIterLM.setText("")


        ## if path and name of alg. xml file is not defined, enable elements for user input
        elif (self.AlgXMLFileName.strip() == ""):
            # self.lineEdit_PathAlgXML_Tab1.setEnabled(False)
            self.lineEdit_MaxIterLM.setEnabled(True)
            if (self.UserMaxIterLM is not None):
                self.lineEdit_MaxIterLM.setText(str(self.UserMaxIterLM))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## deactivate tab 2 and tab 3 if not all required parameters are defined
        if ((self.MolfitFileName.strip() == "" and self.UserCurrentMolecule.strip() == "") \
            or (self.ObsXMLFileName.strip() == "" and self.UserObsDataFileName[self.ObsDataFileID].strip() == "") \
            or (self.AlgXMLFileName.strip() == "" and self.UserMaxIterLM is None)):
            self.pushButton_SMF_Tab1.setEnabled(False)
        else:
            self.pushButton_SMF_Tab1.setEnabled(True)


        ## return to GUI
        self.lock = 0
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get parameters from obs. xml file
    def Tab1ImportObsXMLFile(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None

        """


        ## is obs. xml file defined?
        if (self.ObsXMLFileName.strip() == ""):
            return


        ## get parameters
        LocalExpFileList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "FileNamesExpFiles")
        LocalNumberExpFiles = len(LocalExpFileList)
        LocalNumberExpRangesList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "NumberExpRanges")
        LocalFreqMinList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "MinExpRange")
        if (LocalFreqMinList == []):
            return
        LocalFreqMaxList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "MaxExpRange")
        LocalFreqStepList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "StepFrequency")
        Localt_back_flagList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "t_back_flag")
        LocaltBackList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "BackgroundTemperature")
        LocaltSlopeList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "TemperatureSlope")
        LocalN_HList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "HydrogenColumnDensity")
        Localbeta_dustList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "DustBeta")
        Localkappa_1300List = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "Kappa")
        LocalDustFileNameList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "DustFileName")
        LocalBackgroundFileNameList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "BackgroundFileName")
        LocalContPhenFuncIDList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "ContPhenFuncID")
        LocalContPhenFuncParam1List = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "ContPhenFuncParam1")
        LocalContPhenFuncParam2List = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "ContPhenFuncParam2")
        LocalContPhenFuncParam3List = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "ContPhenFuncParam3")
        LocalContPhenFuncParam4List = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "ContPhenFuncParam4")
        LocalContPhenFuncParam5List = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "ContPhenFuncParam5")
        LocalNoiseList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "NoiseLevel")
        LocalTelescopeSizeList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "TelescopeSize")
        LocalBMINList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "BMIN")
        LocalBMAJList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "BMAJ")
        LocalBPAList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "BPA")
        LocalInter_FlagList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "Inter_Flag")
        LocalGlobalvLSRList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "GlobalvLSR")
        LocalRedshiftList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "Redshift")
        LocalIsotopologuesList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "iso_flag")
        if (LocalIsotopologuesList == []):
            LocalIsotopologuesList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "Isotopologues")
        LocalIsoTableFileNameList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "IsoTableFileName")
        self.ErrorYList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "ErrorY")
        self.NumberHeaderLinesList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "NumberHeaderLines")
        self.SeparatorColumnsList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "SeparatorColumns")
        LocalNumModelPixelXX = None
        LocalNumModelPixelXXList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "NumModelPixelXX")
        if (LocalNumModelPixelXXList != []):
            LocalNumModelPixelXX = LocalNumModelPixelXXList[0]
        LocalNumModelPixelYY = None
        LocalNumModelPixelYYList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "NumModelPixelYY")
        if (LocalNumModelPixelYYList != []):
            LocalNumModelPixelYY = LocalNumModelPixelYYList[0]
        LocalLocalOverlapFlag = None
        LocalLocalLocalOverlapFlagList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "LocalOverlap_Flag")
        if (LocalLocalLocalOverlapFlagList != []):
            LocalLocalOverlapFlag = LocalLocalLocalOverlapFlagList[0]
        LocalNoSubBeamFlag = None
        LocalLocalNoSubBeamFlagList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "NoSubBeam_Flag")
        if (LocalLocalNoSubBeamFlagList != []):
            LocalNoSubBeamFlag = LocalLocalNoSubBeamFlagList[0]
        LocalEmAbsPATH = None
        LocalEmAbsPATHList = task_MAGIX.GetXMLtagNEW(self.ObsXMLFileName, "EmAbsPATH")
        if (LocalEmAbsPATHList != []):
            LocalEmAbsPATH = LocalEmAbsPATHList[0]


        ## construct arrays for GUI
        self.UserObsDataFileName = []
        self.UserNumberExpRanges = []
        self.UserFreqMin = []
        self.UserFreqMax = []
        self.UserFreqStep = []
        self.UserTbgFlag = []
        self.UserTbg = []
        self.UserTslope = []
        self.UsernHFlag = []
        self.UsernH = []
        self.Userkappa = []
        self.Userbeta = []
        self.UserDustFileName = []
        self.UserBackgroundFileName = []
        self.UserTeff = []
        self.UserContFuncID = []
        self.UserContParam1 = []
        self.UserContParam2 = []
        self.UserContParam3 = []
        self.UserContParam4 = []
        self.UserContParam5 = []
        self.UserSizeTelescope = []
        self.UserBMIN = []
        self.UserBMAJ= []
        self.UserBPA = []
        self.UserInterferoFlag = []
        self.UservLSR = []
        self.UserRedShift = []
        self.UserNumModelPixelXX = []
        self.UserNumModelPixelYY = []
        self.UserLocalOverlapFlag = []
        self.UserNoSubBeamFlag = []
        self.UserEmAbsPATH = []
        for ObsDataFileIndex in range(LocalNumberExpFiles):                                 ## loop over all obs. data file


            ## get number of header lines and character separating columns from obs. xml file
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberRangeListIn = LocalNumberExpRangesList, \
                                                                           TelescopeSizeListIn = LocalTelescopeSizeList, BMIN_ListIn = LocalBMINList, \
                                                                           BMAJ_ListIn = LocalBMAJList, BPA_ListIn = LocalBPAList, \
                                                                           InterFlagListIn = LocalInter_FlagList, \
                                                                           GlobalvLSRListIn = LocalGlobalvLSRList, Redshift_ListIn = LocalRedshiftList, \
                                                                           NumberHeaderLinesListIn = self.NumberHeaderLinesList)

            ## get number of frequency ranges for current obs. data file
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
            self.UserNumberExpRanges.append([ObsDataFileIndex, NumberFrequencyRanges])


            ## get size of telescope
            TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
            self.UserSizeTelescope.append([ObsDataFileIndex, float(TelescopeSize)])


            ## get BMIN
            BMIN = ObsXMLParameterDictFile['BMIN']
            self.UserBMIN.append([ObsDataFileIndex, BMIN])


            ## get BMAJ
            BMAJ = ObsXMLParameterDictFile['BMAJ']
            self.UserBMAJ.append([ObsDataFileIndex, BMAJ])


            ## get BPA
            BPA = ObsXMLParameterDictFile['BPA']
            self.UserBPA.append([ObsDataFileIndex, BPA])


            ## get interferometer flag
            InterF = ObsXMLParameterDictFile['InterFlag']
            InterF = task_myXCLASSMapFit.CheckBool(InterF)
            if (InterF):
                self.UserInterferoFlag.append([ObsDataFileIndex, True])
            else:
                self.UserInterferoFlag.append([ObsDataFileIndex, False])


            ## get global v_LSR
            GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
            if (GlobalvLSR is None):
                GlobalvLSR = 0.0
            self.UservLSR.append([ObsDataFileIndex, GlobalvLSR])


            ## get redshift
            Redshift = ObsXMLParameterDictFile['Redshift']
            if (Redshift is None):
                Redshift = 0.0
            self.UserRedShift.append([ObsDataFileIndex, Redshift])


            ## get number of header lines
            LocalNumHeaderLine = ObsXMLParameterDictFile['NumberHeaderLines']
            if (LocalNumHeaderLine is None):
                LocalNumHeaderLine = 0.0


            ## get parameter for each obs. data file
            LocalObsDataFileName = LocalExpFileList[ObsDataFileIndex][1].strip()
            self.UserObsDataFileName.append(LocalObsDataFileName)
            if (not LocalObsDataFileName.endswith(".fits")):                                ## import obs. data file if it's not a fits cube
                ObsDataFile = np.loadtxt(LocalExpFileList[self.ObsDataFileID][1], skiprows = LocalNumHeaderLine)


            ## get parameters for each range
            for RangeIndex in range(NumberFrequencyRanges):                                 ## loop over all frequency ranges


                ## get parameter for current frequency range
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = LocalFreqMinList, \
                                                                                FreqMaxListIn = LocalFreqMaxList, FreqStepListIn = LocalFreqStepList, \
                                                                                tBackFlagListIn = Localt_back_flagList, tBackListIn = LocaltBackList, \
                                                                                tSlopeListIn = LocaltSlopeList, N_HListIn = LocalN_HList, \
                                                                                beta_dustListIn = Localbeta_dustList, \
                                                                                kappa_1300ListIn = Localkappa_1300List, \
                                                                                DustFileNameListIn = LocalDustFileNameList, \
                                                                                BackgroundFileNameListIn = LocalBackgroundFileNameList, \
                                                                                ContPhenFuncID_ListIn = LocalContPhenFuncIDList, \
                                                                                ContPhenFuncParam1_ListIn = LocalContPhenFuncParam1List, \
                                                                                ContPhenFuncParam2_ListIn = LocalContPhenFuncParam2List, \
                                                                                ContPhenFuncParam3_ListIn = LocalContPhenFuncParam3List, \
                                                                                ContPhenFuncParam4_ListIn = LocalContPhenFuncParam4List, \
                                                                                ContPhenFuncParam5_ListIn = LocalContPhenFuncParam5List, \
                                                                                NoiseListIn = LocalNoiseList)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None):
                    self.UserFreqMin.append([ObsDataFileIndex, RangeIndex, FreqMin])


                    ## get FreqMax
                    FreqMax = ObsXMLParameterDictRange['FreqMax']
                    self.UserFreqMax.append([ObsDataFileIndex, RangeIndex, FreqMax])


                    ## get FreqStep
                    FreqStep = ObsXMLParameterDictRange['FreqStep']
                    self.UserFreqStep.append([ObsDataFileIndex, RangeIndex, FreqStep])


                    ## get T_Back Flag
                    t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                    if (t_back_flag is None):
                        t_back_flag = False
                    self.UserTbgFlag.append([ObsDataFileIndex, RangeIndex, t_back_flag])


                    ## get T_Back
                    tBack = ObsXMLParameterDictRange['tBack']
                    if (tBack is None):
                        tBack = 0.0
                    self.UserTbg.append([ObsDataFileIndex, RangeIndex, tBack])


                    ## get T_Slope
                    tSlope = ObsXMLParameterDictRange['tSlope']
                    if (tSlope is None):
                        tSlope = 0.0
                    self.UserTslope.append([ObsDataFileIndex, RangeIndex, tSlope])


                    ## get nH Flag, nH, beta, kappa
                    nH_flag = True
                    N_H = ObsXMLParameterDictRange['N_H']
                    beta_dust = ObsXMLParameterDictRange['beta_dust']
                    kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                    if (N_H is None or beta_dust is None or kappa_1300 is None):
                        nH_flag = False
                        N_H = 0.0
                        beta_dust = 0.0
                        kappa_1300 = 0.0
                    self.UsernHFlag.append([ObsDataFileIndex, RangeIndex, nH_flag])
                    self.UsernH.append([ObsDataFileIndex, RangeIndex, N_H])
                    self.Userkappa.append([ObsDataFileIndex, RangeIndex, kappa_1300])
                    self.Userbeta.append([ObsDataFileIndex, RangeIndex, beta_dust])


                    ## get path and name of dust file
                    DustFileName = ObsXMLParameterDictRange['DustFileName']
                    if (DustFileName is None):
                        DustFileName = ""
                    self.UserDustFileName.append([ObsDataFileIndex, RangeIndex, DustFileName])


                    ## get path and name of background file
                    BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                    if (BackgroundFileName is None):
                        BackgroundFileName = ""
                    self.UserBackgroundFileName.append([ObsDataFileIndex, RangeIndex, BackgroundFileName])


                    ## get ContPhenFuncID
                    LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                    self.UserContFuncID.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncID])


                    ## get ContPhenFuncParam1
                    LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                    self.UserContParam1.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncParam1])


                    ## get ContPhenFuncParam2
                    LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                    self.UserContParam2.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncParam2])


                    ## get ContPhenFuncParam3
                    LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                    self.UserContParam3.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncParam3])


                    ## get ContPhenFuncParam4
                    LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                    self.UserContParam4.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncParam4])


                    ## get ContPhenFuncParam5
                    LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']
                    self.UserContParam5.append([ObsDataFileIndex, RangeIndex, LocalContPhenFuncParam5])


        ## isotopologues
        IsoTableFileName = ""
        LocalIsoFlag = str(LocalIsotopologuesList[0]).lower()
        LocalIsoFlag = task_myXCLASSMapFit.CheckBool(LocalIsoFlag)
        if (LocalIsoFlag):
            self.UserUseIsoFlag = True
        else:
            self.UserUseIsoFlag = False
        if (self.UserUseIsoFlag):
            IsoTableFileName = LocalIsoTableFileNameList[0]
        self.UserIsoRatioFileName = IsoTableFileName


        ## initialize other parameters
        self.UserNoSubBeamFlag = LocalNoSubBeamFlag
        self.UserNumModelPixelXX = LocalNumModelPixelXX
        self.UserNumModelPixelYY = LocalNumModelPixelYY
        self.UserEmAbsPATH = LocalEmAbsPATH
        self.UserLocalOverlapFlag = LocalLocalOverlapFlag
        if (self.UserLocalOverlapFlag):
            self.actionLocal_overlap.setChecked(True)
        else:
            self.actionLocal_overlap.setChecked(False)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get molfit parameter from file
    def Tab1selectFile(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## get name of button
        sender = self.sender()
        objectName = sender.objectName()
        name = sender.text()

        # Debug:
        # print ("\n\nsender = ", sender)
        # print ("objectName = ", objectName)
        # print ("name = ", name)


        ## get path and name of selected file
        if (objectName != "pushButton_RemoveObsFile"):
            SelectedPathOfFile = QtWidgets.QFileDialog.getOpenFileName(self)[0]
            SelectedPathOfFile = str(SelectedPathOfFile).strip()

        # Debug:
        # print ("SelectedPathOfFile = ", SelectedPathOfFile)


        ## store local path to required variable
        if (objectName == "pushButton_SelectMolfit_Tab1"):
            if (not SelectedPathOfFile.endswith(".molfit")):
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please choose a molfit file!")
                ret = msgBox.exec_()
            else:
                if (self.MolfitFileName.strip() == SelectedPathOfFile.strip()):
                    self.MolfitFileChangeFlag = False
                else:
                    self.MolfitFileChangeFlag = True
                    self.ObsXMLFileChangeFlag = True
                    self.CurrentMolecule = ""
                    self.UserCurrentMoleculeID = 0
                    self.CurrentMolfitFileContents = []
                    self.CurrentMolfitParameter = []
                    self.CurrentMoleculeDir = ""
                    self.CurrentComponentIndex = 0
                    self.CurrentMolfitIndex = 1
                    self.ActiveCompList = []
                    self.UseIsoFlag = False
                    self.CurrentIsoParam = []
                    self.AllIsotopologueList = []
                    self.CurrentIsoID = 0
                self.MolfitFileName = SelectedPathOfFile
                self.lineEdit_PathMolfit_Tab1.setText(self.MolfitFileName)

        elif (objectName == "pushButton_SelectObsXML_Tab1"):
            if (not SelectedPathOfFile.endswith(".xml")):
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please choose a xml file!")
                ret = msgBox.exec_()
            else:
                if (self.ObsXMLFileName.strip() == SelectedPathOfFile.strip()):
                    self.ObsXMLFileChangeFlag = False
                else:
                    self.ObsXMLFileChangeFlag = True
                self.ObsXMLFileName = SelectedPathOfFile
                self.lineEdit_PathObsXML_Tab1.setText(self.ObsXMLFileName)
                self.Tab1ImportObsXMLFile()                                                 ## import parameters from obs. xml file

        elif (objectName == "pushButton_SelectAlgXML_Tab1"):
            if (not SelectedPathOfFile.endswith(".xml")):
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("Error")
                msgBox.setText("Please choose a xml file!")
                ret = msgBox.exec_()
            else:
                if (self.AlgXMLFileName.strip() == SelectedPathOfFile.strip()):
                    self.AlgXMLFileChangeFlag = False
                else:
                    self.AlgXMLFileChangeFlag = True
                self.AlgXMLFileName = SelectedPathOfFile
                self.lineEdit_PathAlgXML_Tab1.setText(self.AlgXMLFileName)


        ## button "pushButton_SelectObsData" or "pushButton_AddObsFile"
        elif (objectName == "pushButton_SelectObsData" or objectName == "pushButton_AddObsFile"):
            LocalFlag = False
            if (objectName == "pushButton_AddObsFile"):
                self.ObsDataFileID += 1
                self.FrequencyRangeID = 0
                if (len(self.UserObsDataFileName) == 1):
                    if (self.UserObsDataFileName[0].strip() == ""):
                        LocalFlag = True
                if (LocalFlag):
                    self.UserObsDataFileName = [SelectedPathOfFile.strip()]
                    self.ObsDataFileID = 0
                else:
                    self.UserObsDataFileName.append(SelectedPathOfFile.strip())
            else:
                self.UserObsDataFileName[self.ObsDataFileID] = SelectedPathOfFile.strip()
            self.lineEdit_PathObsData.setText(self.UserObsDataFileName[self.ObsDataFileID])
            if (self.UserObsDataFileName[self.ObsDataFileID].endswith(".fits")):
                self.Tab1selectFreqRange()
            elif (self.UserObsDataFileName[self.ObsDataFileID].strip() != ""):
                if (not self.UserObsDataFileName[self.ObsDataFileID].endswith(".fits")):
                    RangeIndex = (-1)
                    ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, RangeIndex, \
                                                                                   NumberHeaderLinesListIn = self.NumberHeaderLinesList)
                    LocalNumHeaderLine = ObsXMLParameterDictFile['NumberHeaderLines']
                    if (LocalNumHeaderLine is None):
                        LocalNumHeaderLine = 0.0
                    ObsDataFile = np.loadtxt(self.UserObsDataFileName[self.ObsDataFileID], skiprows = LocalNumHeaderLine)
                if (objectName == "pushButton_AddObsFile" and not LocalFlag):
                    self.UserNumberExpRanges.append([self.ObsDataFileID, self.UserNumberExpRangesDefault])
                    self.UserFreqMin.append([self.ObsDataFileID, self.FrequencyRangeID, np.min(ObsDataFile[:, 0])])
                    self.UserFreqMax.append([self.ObsDataFileID, self.FrequencyRangeID, np.max(ObsDataFile[:, 0])])
                    self.UserFreqStep.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserFreqStepDefault])
                    self.UserTbgFlag.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserTbgFlagDefault])
                    self.UserTbg.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserTbgDefault])
                    self.UserTslope.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserTslopeDefault])
                    self.UsernHFlag.append([self.ObsDataFileID, self.FrequencyRangeID, self.UsernHFlagDefault])
                    self.UsernH.append([self.ObsDataFileID, self.FrequencyRangeID, self.UsernHDefault])
                    self.Userkappa.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserkappaDefault])
                    self.Userbeta.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserbetaDefault])
                    self.UserDustFileName.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserDustFileNameDefault])
                    self.UserBackgroundFileName.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserBackgroundFileNameDefault])
                    self.UserContFuncID.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserContFuncIDDefault])
                    self.UserContParam1.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam1Default])
                    self.UserContParam2.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam2Default])
                    self.UserContParam3.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam3Default])
                    self.UserContParam4.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam4Default])
                    self.UserContParam5.append([self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam5Default])
                    self.UserSizeTelescope.append([self.ObsDataFileID, self.UserSizeTelescopeDefault])
                    self.UserBMIN.append([self.ObsDataFileID, self.UserBMINDefault])
                    self.UserBMAJ.append([self.ObsDataFileID, self.UserBMAJDefault])
                    self.UserBPA.append([self.ObsDataFileID, self.UserBPADefault])
                    self.UserInterferoFlag.append([self.ObsDataFileID, self.UserInterferoFlagDefault])
                    self.UservLSR.append([self.ObsDataFileID, self.UservLSRDefault])
                    self.UserRedShift.append([self.ObsDataFileID, self.UserRedShiftDefault])
                    self.UserUseIsoFlag = self.UserUseIsoFlagDefault
                    self.UserNumModelPixelXX = self.UserNumModelPixelXXDefault
                    self.UserNumModelPixelYY = self.UserNumModelPixelYYDefault
                    self.UserLocalOverlapFlag = self.UserLocalOverlapFlagDefault
                    self.UserNoSubBeamFlag = self.UserNoSubBeamFlagDefault
                    self.UserEmAbsPATH = self.UserEmAbsPATHDefault
                else:
                    self.UserNumberExpRanges = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserNumberExpRanges, self.UserNumberExpRangesDefault, \
                                                                    "update")
                    self.UserFreqMin = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserFreqMin, np.min(ObsDataFile[:, 0]), \
                                                            "update")
                    self.UserFreqMax = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserFreqMax, np.max(ObsDataFile[:, 0]), \
                                                            "update")
                    self.UserFreqStep = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserFreqStep, self.UserFreqStepDefault, \
                                                             "update")
                    self.UserTbgFlag = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserTbgFlag, self.UserTbgFlagDefault, \
                                                            "update")
                    self.UserTbg = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserTbg, self.UserTbgDefault, "update")
                    self.UserTslope = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserTslope, self.UserTslopeDefault, "update")
                    self.UsernHFlag = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UsernHFlag, self.UsernHFlagDefault, "update")
                    self.UsernH = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UsernH, self.UsernHDefault, "update")
                    self.Userkappa = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.Userkappa, self.UserkappaDefault, "update")
                    self.Userbeta = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.Userbeta, self.UserbetaDefault, "update")
                    self.UserDustFileName = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserDustFileName, \
                                                                  self.UserDustFileNameDefault, "update")
                    self.UserBackgroundFileName = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserBackgroundFileName, \
                                                                       self.UserBackgroundFileNameDefault, "update")
                    self.UserContFuncID = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContFuncID, \
                                                               self.UserContFuncIDDefault, "update")
                    self.UserContParam1 = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam1, \
                                                               self.UserContParam1Default, "update")
                    self.UserContParam2 = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam2, \
                                                               self.UserContParam2Default, "update")
                    self.UserContParam3 = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam3, \
                                                               self.UserContParam3Default, "update")
                    self.UserContParam4 = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam4, \
                                                               self.UserContParam4Default, "update")
                    self.UserContParam5 = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam5, \
                                                               self.UserContParam5Default, "update")
                    self.UserSizeTelescope = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserSizeTelescope, self.UserSizeTelescopeDefault, \
                                                                  "update")
                    self.UserBMIN = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserBMIN, self.UserBMINDefault, "update")
                    self.UserBMAJ = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserBMAJ, self.UserBMAJDefault, "update")
                    self.UserBPA = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserBPA, self.UserBPADefault, "update")
                    self.UserInterferoFlag = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserInterferoFlag, self.UserInterferoFlagDefault, \
                                                                  "update")
                    self.UservLSR = self.UpdateParamList(self.ObsDataFileID, (-1), self.UservLSR, self.UservLSRDefault, "update")
                    self.UserRedShift = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserRedShift, self.UserRedShiftDefault, "update")
                    self.UserUseIsoFlag = self.UserUseIsoFlagDefault
                    self.UserNumModelPixelXX = self.UserNumModelPixelXXDefault
                    self.UserNumModelPixelYY = self.UserNumModelPixelYYDefault
                    self.UserLocalOverlapFlag = self.UserLocalOverlapFlagDefault
                    self.UserNoSubBeamFlag = self.UserNoSubBeamFlagDefault
                    self.UserEmAbsPATH = self.UserEmAbsPATHDefault

            # Debug:
            # print ("self.UserFreqMin = ", self.UserFreqMin)
            # print ("self.UserFreqMax = ", self.UserFreqMax)


        ## button "pushButton_SelectIso"
        elif (objectName == "pushButton_SelectIso"):
            self.UserIsoRatioFileName = SelectedPathOfFile
            self.lineEdit_PathIsoFile.setText(self.UserIsoRatioFileName)


        ## button "pushButton_RemoveObsFile"
        elif (objectName == "pushButton_RemoveObsFile"):
            if (len(self.UserObsDataFileName) > 1):


                ## make sure everything is ok.
                reply = QtWidgets.QMessageBox.question(self, 'Message', "Are you sure to remove current obs. data file setting?", \
                                                   QtWidgets.QMessageBox.Yes | QtWidgets.QMessageBox.No, QtWidgets.QMessageBox.No)
                if (reply == QtWidgets.QMessageBox.Yes):

                    # Debug:
                    # print ("->self.UserFreqMin = ", self.UserFreqMin)


                    ## remove elements from parameter lists, which have current obs. data file ID
                    del self.UserObsDataFileName[self.ObsDataFileID]
                    self.UserNumberExpRanges = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserNumberExpRanges, "", "delete")
                    self.UserFreqMin = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserFreqMin, "", "delete")
                    self.UserFreqMax = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserFreqMax, "", "delete")
                    self.UserFreqStep = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserFreqStep, "", "delete")
                    self.UserTbgFlag = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserTbgFlag, "", "delete")
                    self.UserTbg = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserTbg, "", "delete")
                    self.UserTslope = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserTslope, "", "delete")
                    self.UsernHFlag = self.UpdateParamList(self.ObsDataFileID, (-1), self.UsernHFlag, "", "delete")
                    self.UsernH = self.UpdateParamList(self.ObsDataFileID, (-1), self.UsernH, "", "delete")
                    self.Userkappa = self.UpdateParamList(self.ObsDataFileID, (-1), self.Userkappa, "", "delete")
                    self.Userbeta = self.UpdateParamList(self.ObsDataFileID, (-1), self.Userbeta, "", "delete")
                    self.UserDustFileName = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserDustFileName, "", "delete")
                    self.UserBackgroundFileName = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserBackgroundFileName, "", "delete")
                    self.UserContFuncID = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContFuncID, "", "delete")
                    self.UserContParam1 = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContParam1, "", "delete")
                    self.UserContParam2 = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContParam2, "", "delete")
                    self.UserContParam3 = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContParam3, "", "delete")
                    self.UserContParam4 = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContParam4, "", "delete")
                    self.UserContParam5 = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContParam5, "", "delete")
                    self.UserSizeTelescope = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserSizeTelescope, "", "delete")
                    self.UserBMIN = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserBMIN, "", "delete")
                    self.UserBMAJ = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserBMAJ, "", "delete")
                    self.UserBPA = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserBPA, "", "delete")
                    self.UserInterferoFlag = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserInterferoFlag, "", "delete")
                    self.UservLSR = self.UpdateParamList(self.ObsDataFileID, (-1), self.UservLSR, "", "delete")
                    self.UserRedShift = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserRedShift, "", "delete")

                    # Debug:
                    # print (">>self.UserFreqMin = ", self.UserFreqMin)


                    ## remove ASCII files containing extracted spectrum (spectra) of of selected FITS file
                    LocalObsDataFileName = self.FITSSubDir + "FitsSpectrum__" + str(self.ObsDataFileID + 1) + ".dat"
                    if (os.path.isfile(LocalObsDataFileName)):
                        cmdString = "rm -rf " + LocalObsDataFileName
                        os.system(cmdString)


                    ## rename ASCII files containing extracted spectrum (spectra) of each FITS file
                    TotalNumObsDataFiles = len(self.UserObsDataFileName)
                    for LoaclObsID in range(self.ObsDataFileID, TotalNumObsDataFiles):
                        if (LoaclObsID < TotalNumObsDataFiles - 2):
                            LocalObsDataFileName = self.FITSSubDir + "FitsSpectrum__" + str(LoaclObsID + 2) + ".dat"
                            if (os.path.isfile(LocalObsDataFileName)):
                                LocalObsDataFileNameNew = self.FITSSubDir + "FitsSpectrum__" + str(LoaclObsID + 1) + ".dat"
                                cmdString = "mv " + LocalObsDataFileName + " " + LocalObsDataFileNameNew
                                os.system(cmdString)


                    ## redefine id for obs. data file
                    self.ObsDataFileID = min(TotalNumObsDataFiles - 1, self.ObsDataFileID)


        ## update all parameters
        self.Tab1SetValues()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for LineEdit text events
    def Tab1LineEdit(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## GUI locked
        if (self.lock == 1):
            return


        ## get name of edit box and the corresponding value
        sender = self.sender()
        if (str(sender).find("QLineEdit") > (-1)):
            objectName = sender.objectName()
            val = str(sender.text())
            self.CurrCursorPos = sender.cursorPosition()
        else:
            try:
                val = str(sender.toPlainText())
                objectName = str(sender.objectName())                                       ## get name of object
                cursor = sender.textCursor()
                self.CurrCursorPos = cursor.position()
            except:
                return

        # Debug:
        if (self.dbgFlag):
            print ("self.CurrCursorPos = ", self.CurrCursorPos)


        ## check, if value is number
        IsNumber = True
        try:
            i = int(val)
        except ValueError:
            try:
                i = float(val)
            except ValueError:
                IsNumber = False
        if (IsNumber):                                                                      ## continue if input is a number
            # val = float(val)


            ## get column of text edit
            if (objectName == "lineEdit_MaxIterLM"):
                self.UserMaxIterLM = val
            elif (objectName == "lineEdit_MinFreq"):
                OldValue = self.UserFreqMin
                self.UserFreqMin = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserFreqMin, val, "update")
                if (OldValue != self.UserFreqMin):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_MaxFreq"):
                OldValue = self.UserFreqMax
                self.UserFreqMax = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserFreqMax, val, "update")
                if (OldValue != self.UserFreqMax):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_Tbg"):
                OldValue = self.UserTbg
                self.UserTbg = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserTbg, val, "update")
                if (OldValue != self.UserTbg):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_TSlope"):
                OldValue = self.UserTslope
                self.UserTslope = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserTslope, val, "update")
                if (OldValue != self.UserTslope):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_nH"):
                OldValue = self.UsernH
                self.UsernH = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UsernH, val, "update")
                if (OldValue != self.UsernH):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_kappa"):
                OldValue = self.Userkappa
                self.Userkappa = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.Userkappa, val, "update")
                if (OldValue != self.Userkappa):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_beta"):
                OldValue = self.Userbeta
                self.Userbeta = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.Userbeta, val, "update")
                if (OldValue != self.Userbeta):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_SizeTele"):
                OldValue = self.UserSizeTelescope
                self.UserSizeTelescope = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserSizeTelescope, val, "update")
                if (OldValue != self.UserSizeTelescope):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_vLSR"):
                OldValue = self.UservLSR
                self.UservLSR = self.UpdateParamList(self.ObsDataFileID, (-1), self.UservLSR, val, "update")
                if (OldValue != self.UservLSR):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_RedShift"):
                OldValue = self.UserRedShift
                self.UserRedShift = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserRedShift, val, "update")
                if (OldValue != self.UserRedShift):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files


            ##!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
            ## TO DO:       add line edit element "ContFuncID"
            ##              add line edit element "ContParam1"
            ##              add line edit element "ContParam2"
            ##              add line edit element "ContParam3"
            ##              add line edit element "ContParam4"
            ##              add line edit element "ContParam5"
            ##              add line edit element "RedShift"
            ##              add line edit element "BMIN"
            ##              add line edit element "BMAJ"
            ##              add line edit element "BPA"
            ##              add line edit element "NumModelPixelXX"
            ##              add line edit element "NumModelPixelYY"
            ##              add checkBox element "LocalOverlapFlag"
            ##              add checkBox element "NoSubBeamFlag"
            elif (objectName == "lineEdit_ContFuncID"):
                OldValue = self.UserContFuncID
                self.UserContFuncID = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContFuncID, val, "update")
                if (OldValue != self.UserContFuncID):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_ContParam1"):
                OldValue = self.UserContParam1
                self.UserContParam1 = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam1, val, "update")
                if (OldValue != self.UserContParam1):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_ContParam2"):
                OldValue = self.UserContParam2
                self.UserContParam2 = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam2, val, "update")
                if (OldValue != self.UserContParam2):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_ContParam3"):
                OldValue = self.UserContParam3
                self.UserContParam3 = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam3, val, "update")
                if (OldValue != self.UserContParam3):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_ContParam4"):
                OldValue = self.UserContParam4
                self.UserContParam4 = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam4, val, "update")
                if (OldValue != self.UserContParam4):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_ContParam5"):
                OldValue = self.UserContParam5
                self.UserContParam5 = self.UpdateParamList(self.ObsDataFileID, self.FrequencyRangeID, self.UserContParam5, val, "update")
                if (OldValue != self.UserContParam5):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_BMIN"):
                OldValue = self.UserBMIN
                self.UserBMIN = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserBMIN, val, "update")
                if (OldValue != self.UserBMIN):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_BMAJ"):
                OldValue = self.UserBMAJ
                self.UserBMAJ = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserBMAJ, val, "update")
                if (OldValue != self.UserBMAJ):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_BPA"):
                OldValue = self.UserBPA
                self.UserBPA = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserBPA, val, "update")
                if (OldValue != self.UserBPA):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_NumModelPixelXX"):
                OldValue = self.UserNumModelPixelXX
                self.UserNumModelPixelXX = val
                if (OldValue != self.UserNumModelPixelXX):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_NumModelPixelYY"):
                OldValue = self.UserNumModelPixelYY
                self.UserNumModelPixelYY = val
                if (OldValue != self.UserNumModelPixelYY):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files


        else:
            val = val.strip()
            if (objectName == "lineEdit_PathMolfit_Tab1"):
                #if (not val.endswith(".molfit")):
                #    msgBox = QtGui.QMessageBox()
                #    msgBox.setWindowTitle("Error")
                #    msgBox.setText("Please choose a molfit file!")
                #    ret = msgBox.exec_()
                #else:
                self.MolfitFileName = val

            elif (objectName == "lineEdit_PathObsXML_Tab1"):
                #if (not val.endswith(".xml")):
                #    msgBox = QtGui.QMessageBox()
                #    msgBox.setWindowTitle("Error")
                #    msgBox.setText("Please choose a xml file!")
                #    ret = msgBox.exec_()
                #else:
                OldValue = self.ObsXMLFileName
                self.ObsXMLFileName = val
                self.Tab1ImportObsXMLFile()                                                 ## import parameters from obs. xml file
                if (OldValue != self.ObsXMLFileName):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_PathAlgXML_Tab1"):
                #if (not val.endswith(".xml")):
                #    msgBox = QtGui.QMessageBox()
                #    msgBox.setWindowTitle("Error")
                #    msgBox.setText("Please choose a xml file!")
                #    ret = msgBox.exec_()
                #else:
                self.AlgXMLFileName = val

            elif (objectName == "lineEdit_PathObsData"):
                OldValue = self.UserObsDataFileName[self.ObsDataFileID]
                self.UserObsDataFileName[self.ObsDataFileID] = val.strip()
                if (self.UserObsDataFileName[self.ObsDataFileID].endswith(".fits")):
                    self.Tab1selectFreqRange()
                if (OldValue != self.UserObsDataFileName[self.ObsDataFileID]):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files

            elif (objectName == "lineEdit_PathIsoFile"):
                OldValue = self.UserIsoRatioFileName
                self.UserIsoRatioFileName = val
                if (OldValue != self.UserIsoRatioFileName):
                    self.Tab2RemoveBackgroundFiles()                                        ## remove background files


        ## set text cursor
        if (str(sender).find("QLineEdit") > (-1)):
            sender.setCursorPosition(self.CurrCursorPos)
        else:
            cursor.setPosition(self.CurrCursorPos)
            sender.setTextCursor(cursor)


        ## update all parameters
        self.Tab1SetValues()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for check box events of tab 1
    def Tab1checkBox(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## get name, flag, and object name of selected check box
        sender = self.sender()
        flag = sender.isChecked()
        objectName = sender.objectName()


        ## set corresponding flag
        if (objectName == "checkBox_Interfer"):
            for LocalIFFlagID, LocalIFFlag in enumerate(self.UserInterferoFlag):
                ObsDataID = LocalIFFlag[0]
                if (ObsDataID == self.ObsDataFileID):
                    self.UserInterferoFlag[LocalIFFlagID][1] = flag

        elif (objectName == "checkBox_UseIso"):
            self.IsoFlag = flag
            self.UserUseIsoFlag = flag
            if (flag and self.UserIsoRatioFileName.strip() == ""):
                SelectedPathOfFile = QtWidgets.QFileDialog.getOpenFileName(self)[0]
                self.UserIsoRatioFileName = str(SelectedPathOfFile).strip()

        elif (objectName == "checkBox_Tbg_Flag"):
            for LocalTBFlagID, LocalTBFlag in enumerate(self.UserTbgFlag):
                ObsDataID = LocalTBFlag[0]
                if (ObsDataID == self.ObsDataFileID):
                    RangeID = LocalTBFlag[1]
                    if (RangeID == self.FrequencyRangeID):
                        self.UserTbgFlag[LocalTBFlagID][2] = flag

        #    elif (objectName == "checkBox_nH_Flag"):
        #        for LocalnHFlagID, LocalnHFlag in enumerate(self.UsernHFlag):
        #            ObsDataID = LocalnHFlag[0]
        #            if (ObsDataID == self.ObsDataFileID):
        #                RangeID = LocalnHFlag[1]
        #                if (RangeID == self.FrequencyRangeID):
        #                    self.UsernHFlag[LocalnHFlagID][2] = flag


        ## update all parameters
        self.Tab1SetValues()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for usage of combo box event of tab 1
    def Tab1comboSelected(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is GUI locked ?
        if (self.lock == 1):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## combo box for molecules
        if (objectName == "comboBox_MoleculeInput"):


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## disable elements of tab 2
            self.Tab2setDisEnabled(False)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## store local molfit and local iso ratio file
            if (self.AllMoleculesMolfitFiles != []):
                self.AllMoleculesMolfitFiles[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentMolfitFileContents)
            if (self.AllIsotopologueList != []):
                self.AllIsotopologueList[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentIsoParam)
            self.CurrentIsoID = 0


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## update global isotopologue parameters for all other molecules as well


            ## is a global ratio is defined in the current iso param. list
            GlobalIsoList = []
            for LocalIsoParam in self.CurrentIsoParam:
                Isotopologue = LocalIsoParam[0]
                if (Isotopologue.lower().find("global_") > (-1)):
                    GlobalIsoList.append(LocalIsoParam)


            ## if yes, update entries for all other molecules
            if (GlobalIsoList != []):
                for GlobalIsoEntry in GlobalIsoList:
                    GlobalIsoIso = GlobalIsoEntry[0]
                    GlobalIsoMaster = GlobalIsoEntry[1]
                    CopyAllIsotopologueList = copy.deepcopy(self.AllIsotopologueList)
                    for LocalIsoParamMoleculeIndex, LocalIsoParamMolecule in enumerate(CopyAllIsotopologueList):
                        for LocalIsoParamIndex, LocalIsoParam in enumerate(LocalIsoParamMolecule):
                            if (GlobalIsoIso == LocalIsoParam[0] and GlobalIsoMaster == LocalIsoParam[1]):
                                self.AllIsotopologueList[LocalIsoParamMoleculeIndex][LocalIsoParamIndex] = copy.deepcopy(GlobalIsoEntry)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get index of selected isotopologue
            self.MolfitFileChangeFlag = True
            self.CurrentMolecule = ""
            self.UserCurrentMoleculeID = 0
            self.CurrentMolfitFileContents = []
            self.CurrentMolfitParameter = []
            self.CurrentMoleculeDir = ""
            self.CurrentComponentIndex = 0
            self.CurrentMolfitIndex = 1
            self.ActiveCompList = []
            self.UseIsoFlag = False
            self.CurrentIsoParam = []
            self.AllIsotopologueList = []
            self.CurrentIsoID = 0
            self.ObsXMLFileChangeFlag = True
            self.UserCurrentMoleculeID = max(0, self.comboBox_MoleculeInput.currentIndex())
            self.UserCurrentMolecule = self.UserMoleculelist[self.UserCurrentMoleculeID]

            # Debug:
            if (self.dbgFlag):
                print ("\nTab1comboSelected:")
                print ("self.UserCurrentMoleculeID = ", self.UserCurrentMoleculeID)
                print ("self.UserMoleculelist = ", self.UserMoleculelist)
                print ("self.UserCurrentMolecule = ", self.UserCurrentMolecule)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define events for combobox DataFileSelection
        elif (objectName == "comboBox_DataFileSelection"):
            self.ObsDataFileID = max(0, self.comboBox_DataFileSelection.currentIndex())

            # Debug:
            # print ("\n\nself.ObsDataFileID = ", self.ObsDataFileID)


            ## update all parameters
            self.Tab1SetValues()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define events for combobox RangeSelection
        elif (objectName == "comboBox_RangeSelection"):
            self.FrequencyRangeID = max(0, self.comboBox_RangeSelection.currentIndex())

            # Debug:
            # print ("\n\nself.FrequencyRangeID = ", self.FrequencyRangeID)


            ## update all parameters
            self.Tab1SetValues()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for selecting frequency range in tab 1
    def Tab1selectFreqRange(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## disable elements of tab 1
        # self.Tab1setDisEnabled(False)


        ## load obs data file
        NumObsDataFile = len(self.UserObsDataFileName)
        self.FITSFlag = False
        if (self.UserObsDataFileName[self.ObsDataFileID].endswith(".fits")):
            self.FITSFlag = True
            ObsDataFile = []
        else:


            ## get number of header lines and character separating columns from obs. xml file
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, RangeIndex, \
                                                                           NumberHeaderLinesListIn = self.NumberHeaderLinesList)


            ## get number of header lines
            LocalNumHeaderLine = ObsXMLParameterDictFile['NumberHeaderLines']
            if (LocalNumHeaderLine is None):
                LocalNumHeaderLine = 0.0


            ## import ASCII file
            ObsDataFile = np.loadtxt(self.UserObsDataFileName[self.ObsDataFileID], skiprows = LocalNumHeaderLine)


        ## if obs. data file is a FITS file, create subdirectory within current job directory to store extracted spectra and pixel coordinates
        if (self.FITSFlag):
            self.FITSSubDir = self.GUIJobDir + "FITS/"
            cmdString = "mkdir -p " + self.FITSSubDir
            os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## call GUI to select range
        ok, NewRangeParametersDict = RangeSelection.SelectRange(self.ObsDataFileID, \
                                                               self.UserObsDataFileName[self.ObsDataFileID], \
                                                               ObsDataFile, \
                                                               self.UserNumberExpRanges, \
                                                               self.UserFreqMin, \
                                                               self.UserFreqMax, \
                                                               self.UserFreqStep, \
                                                               self.UserTbgFlag, \
                                                               self.UserTbg, \
                                                               self.UserTslope, \
                                                               self.UsernHFlag, \
                                                               self.UsernH, \
                                                               self.Userbeta, \
                                                               self.Userkappa, \
                                                               self.UserDustFileName, \
                                                               self.UserBackgroundFileName, \
                                                               self.UserContFuncID, \
                                                               self.UserContParam1, \
                                                               self.UserContParam2, \
                                                               self.UserContParam3, \
                                                               self.UserContParam4, \
                                                               self.UserContParam5, \
                                                               self.UserTbgFlagDefault, \
                                                               self.UserTbgDefault, \
                                                               self.UserTslopeDefault, \
                                                               self.UsernHFlagDefault, \
                                                               self.UsernHDefault, \
                                                               self.UserkappaDefault, \
                                                               self.UserbetaDefault, \
                                                               self.UserDustFileNameDefault, \
                                                               self.UserBackgroundFileNameDefault, \
                                                               self.UserContFuncIDDefault, \
                                                               self.UserContParam1Default, \
                                                               self.UserContParam2Default, \
                                                               self.UserContParam3Default, \
                                                               self.UserContParam4Default, \
                                                               self.UserContParam5Default, \
                                                               self.UserSizeTelescope, \
                                                               self.UserBMIN, \
                                                               self.UserBMAJ, \
                                                               self.UserBPA, \
                                                               self.UserInterferoFlag, \
                                                               self.UservLSR, \
                                                               self.UserRedShift, \
                                                               self.ObsDataFileID, \
                                                               self.FITSSubDir, NumObsDataFile)


        ## check if everything is ok
        if (ok == 0):

            # Debug:
            # print ("\n\n\nself.ObsDataFileID = ", self.ObsDataFileID)
            # print (">>>self.UserNumberExpRanges = ", self.UserNumberExpRanges)
            # print (">>>self.UserFreqMin = ", self.UserFreqMin)
            # print (">>>self.UserFreqMax = ", self.UserFreqMax)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## update parameter lists for current obs. data file


            ## get new range parameters
            LocalListFreqMin = NewRangeParametersDict['FreqMin']
            LocalNumFreqRanges = len(LocalListFreqMin)
            LocalListFreqMax = NewRangeParametersDict['FreqMax']
            LocalListFreqStep = NewRangeParametersDict['FreqStep']
            LocalListTbgFlag = NewRangeParametersDict['TbgFlag']
            LocalListTbg = NewRangeParametersDict['Tbg']
            LocalListTSlope = NewRangeParametersDict['TSlope']
            LocalListnHFlag = NewRangeParametersDict['nHFlag']
            LocalListnH = NewRangeParametersDict['nH']
            LocalListbeta = NewRangeParametersDict['beta']
            LocalListkappa = NewRangeParametersDict['kappa']
            LocalListDustFile = NewRangeParametersDict['DustFile']
            LocalListBackgroundFile = NewRangeParametersDict['BackgroundFile']
            LocalListContFuncID = NewRangeParametersDict['ContFuncID']
            LocalListContParam1 = NewRangeParametersDict['ContParam1']
            LocalListContParam2 = NewRangeParametersDict['ContParam2']
            LocalListContParam3 = NewRangeParametersDict['ContParam3']
            LocalListContParam4 = NewRangeParametersDict['ContParam4']
            LocalListContParam5 = NewRangeParametersDict['ContParam5']

            # Debug:
            # print ("\n\nLocalNumFreqRanges = ", LocalNumFreqRanges)
            # print ("LocalListFreqMin = ", LocalListFreqMin)
            # print ("LocalListFreqMax = ", LocalListFreqMax)
            # print ("LocalListFreqStep = ", LocalListFreqStep)
            # print ("LocalListTbgFlag = ", LocalListTbgFlag)
            # print ("LocalListTbg = ", LocalListTbg)
            # print ("LocalListTSlope = ", LocalListTSlope)
            # print ("LocalListnHFlag = ", LocalListnHFlag)
            # print ("LocalListnH = ", LocalListnH)
            # print ("LocalListbeta = ", LocalListbeta)
            # print ("LocalListkappa = ", LocalListkappa)
            # print ("LocalListDustFile = ", LocalListDustFile)
            # print ("LocalListBackgroundFile = ", LocalListBackgroundFile)


            ## remove old range specifications for current obs. data file
            self.UserFreqMin = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserFreqMin, "", "delete")
            self.UserFreqMax = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserFreqMax, "", "delete")
            self.UserFreqStep = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserFreqStep, "", "delete")
            self.UserTbgFlag = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserTbgFlag, "", "delete")
            self.UserTbg = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserTbg, "", "delete")
            self.UserTslope = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserTslope, "", "delete")
            self.UsernHFlag = self.UpdateParamList(self.ObsDataFileID, (-1), self.UsernHFlag, "", "delete")
            self.UsernH = self.UpdateParamList(self.ObsDataFileID, (-1), self.UsernH, "", "delete")
            self.Userkappa = self.UpdateParamList(self.ObsDataFileID, (-1), self.Userkappa, "", "delete")
            self.Userbeta = self.UpdateParamList(self.ObsDataFileID, (-1), self.Userbeta, "", "delete")
            self.UserDustFileName = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserDustFileName, "", "delete")
            self.UserBackgroundFileName = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserBackgroundFileName, "", "delete")
            self.UserContFuncID = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContFuncID, "", "delete")
            self.UserContParam1 = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContParam1, "", "delete")
            self.UserContParam2 = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContParam2, "", "delete")
            self.UserContParam3 = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContParam3, "", "delete")
            self.UserContParam4 = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContParam4, "", "delete")
            self.UserContParam5 = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserContParam5, "", "delete")

            # Debug:
            # print ("-->self.UserFreqMin = ", self.UserFreqMin)


            ## insert new parameters to parameter lists
            self.UserNumberExpRanges = self.UpdateParamList(self.ObsDataFileID, (-1), self.UserNumberExpRanges, LocalNumFreqRanges, "update")
            for RangeID in range(len(LocalListFreqMin)):
                self.UserFreqMin = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserFreqMin, LocalListFreqMin, "add")
                self.UserFreqMax = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserFreqMax, LocalListFreqMax, "add")
                self.UserFreqStep = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserFreqStep, LocalListFreqStep, "add")
                self.UserTbgFlag = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserTbgFlag, LocalListTbgFlag, "add")
                self.UserTbg = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserTbg, LocalListTbg, "add")
                self.UserTslope = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserTslope, LocalListTSlope, "add")
                self.UsernHFlag = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UsernHFlag, LocalListnHFlag, "add")
                self.UsernH = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UsernH, LocalListnH, "add")
                self.Userkappa = self.UpdateParamList(self.ObsDataFileID, RangeID, self.Userkappa, LocalListkappa, "add")
                self.Userbeta = self.UpdateParamList(self.ObsDataFileID, RangeID, self.Userbeta, LocalListbeta, "add")
                self.UserDustFileName = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserDustFileName, LocalListDustFile, "add")
                self.UserBackgroundFileName = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserBackgroundFileName, LocalListBackgroundFile, \
                                                                   "add")
                self.UserContFuncID = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserContFuncID, LocalListContFuncID, "add")
                self.UserContParam1 = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserContParam1, LocalListContParam1, "add")
                self.UserContParam2 = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserContParam2, LocalListContParam2, "add")
                self.UserContParam3 = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserContParam3, LocalListContParam3, "add")
                self.UserContParam4 = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserContParam4, LocalListContParam4, "add")
                self.UserContParam5 = self.UpdateParamList(self.ObsDataFileID, RangeID, self.UserContParam5, LocalListContParam5, "add")

            # Debug:
            # print ("\n\nok = ", ok)
            # print ("++>self.UserNumberExpRanges = ", self.UserNumberExpRanges)
            # print ("++>self.UserFreqMin = ", self.UserFreqMin)
            # print ("++>self.UserFreqMax = ", self.UserFreqMax)
            # print ("\n\nself.ObsDataFileID = ", self.ObsDataFileID)
            # print ("self.UserFreqMin = ", self.UserFreqMin)
            # print ("self.UserFreqMin[self.ObsDataFileID] = ", self.UserFreqMin[self.ObsDataFileID])


            ## update all parameters
            self.Tab1SetValues()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for selecting algorithms in tab 1
    def Tab1SelectAlgorithm(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## disable elements of tab 1
        # self.Tab1setDisEnabled(False)


        ## call GUI to select range
        NewAlgXMLFileName = AlgorithmGUI.SelectAlgorithm(self.GUIJobDir, self.UserMaxIterLM, self.AlgXMLFileName)


        ## check if everything is ok
        if (NewAlgXMLFileName != self.AlgXMLFileName):
            self.AlgXMLFileName = NewAlgXMLFileName

            # Debug:
            # print ("self.AlgXMLFileName = ", self.AlgXMLFileName)


            ## update all parameters
            self.Tab1SetValues()


        ## disable elements of tab 1
        #self.Tab1setDisEnabled(True)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## creates the experimental xml file in the current job directory
    def Tab1CreateUserExpXMLFile(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## define content of obs. xml file
        EXPcontrol = open(self.GUIJobDir + "obs.xml", 'w')
        EXPcontrol.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
        EXPcontrol.write("<ExpFiles>\n")
        EXPcontrol.write("\n\n    <!-- set number of obs. data files -->\n")
        EXPcontrol.write("    <NumberExpFiles>" + str(len(self.UserObsDataFileName)) + "</NumberExpFiles>\n")
        for LocalObsID, LocalObsDataFileName in enumerate(self.UserObsDataFileName):
            EXPcontrol.write("\n\n    <!-- define setting for import of obs. data file " + str(LocalObsID) + " -->\n")
            EXPcontrol.write("    <file>\n")
            EXPcontrol.write("\n\n        <!-- define path and name of obs. data file and set import filter -->\n")
            EXPcontrol.write("        <FileNamesExpFiles>" + LocalObsDataFileName + "</FileNamesExpFiles>\n")
            EXPcontrol.write("        <ImportFilter>xclassASCII</ImportFilter>\n")
            EXPcontrol.write("\n\n        <!-- define number of frequency ranges -->\n")
            EXPcontrol.write("        <NumberExpRanges>" + str(len(self.UserFreqMin[LocalObsID])) + "</NumberExpRanges>\n")
            for LocalRangeID in range(len(self.UserFreqMin[LocalObsID])):
                EXPcontrol.write("\n\n        <!-- define parameters for frequency range " + str(LocalRangeID) + " -->\n")
                EXPcontrol.write("        <FrequencyRange>\n")
                EXPcontrol.write("            <MinExpRange>" + str(self.UserFreqMin[LocalObsID][LocalRangeID]) + "</MinExpRange>\n")
                EXPcontrol.write("            <MaxExpRange>" + str(self.UserFreqMax[LocalObsID][LocalRangeID]) + "</MaxExpRange>\n")
                EXPcontrol.write("            <StepFrequency>" + str(self.UserFreqStepDefault) + "</StepFrequency>\n")
                EXPcontrol.write("\n\n            <!-- define background temperature and slope for frequency range " + str([LocalRangeID]) + " -->\n")
                LocalTbgF = self.UserTbgFlag[LocalObsID][LocalRangeID]
                LocalTbgF = task_myXCLASSMapFit.CheckBool(LocalTbgF)
                if (LocalTbgF):
                    EXPcontrol.write("            <t_back_flag>True</t_back_flag>\n")
                else:
                    EXPcontrol.write("            <t_back_flag>False</t_back_flag>\n")
                EXPcontrol.write("            <BackgroundTemperature>" + str(self.UserTbg[LocalObsID][LocalRangeID]) + "</BackgroundTemperature>\n")
                EXPcontrol.write("            <TemperatureSlope>" + str(self.UserTslope[LocalObsID][LocalRangeID]) + "</TemperatureSlope>\n")
                LocalBackgroundFileName = self.UserBackgroundFileName[LocalObsID][LocalRangeID]
                if (LocalBackgroundFileName is not None):
                    LocalBackgroundFileName = LocalBackgroundFileName.strip()
                    if (LocalBackgroundFileName != ""):
                        EXPcontrol.write("\n\n            <!-- define path and name of file describing background -->\n")
                        EXPcontrol.write("            <BackgroundFileName>" + LocalBackgroundFileName + "</BackgroundFileName>\n")


                ## add dust parameter
                LocalnHF = self.UsernHFlag[LocalObsID][LocalRangeID]
                LocalnHF = task_myXCLASSMapFit.CheckBool(LocalnHF)
                if (LocalnHF):
                    EXPcontrol.write("\n\n            <!-- define hydrogen column density etc. for frequency range " + str([LocalRangeID]) + " -->\n")
                    LocalnH = self.UsernH[LocalObsID][LocalRangeID]
                    if (LocalnH is not None):
                        EXPcontrol.write("            <HydrogenColumnDensity>" + str(LocalnH) + "</HydrogenColumnDensity>\n")
                    EXPcontrol.write("            <DustBeta>" + str(self.Userbeta[LocalObsID][LocalRangeID]) + "</DustBeta>\n")
                    EXPcontrol.write("            <Kappa>" + str(self.Userkappa[LocalObsID][LocalRangeID]) + "</Kappa>\n")
                LocalDustFileName = self.UserDustFileName[LocalObsID][LocalRangeID]
                if (LocalDustFileName is not None):
                    LocalDustFileName = LocalDustFileName.strip()
                    if (LocalDustFileName != ""):
                        EXPcontrol.write("\n\n            <!-- define path and name of file describing dust contribution -->\n")
                        EXPcontrol.write("            <DustFileName>" + LocalDustFileName + "</DustFileName>\n")


                ## define parameters for phenomenological continuum description
                LocalContPhenFuncID = self.UserContFuncID[LocalObsID][LocalRangeID]
                LocalContPhenFuncParam1 = self.UserContParam1[LocalObsID][LocalRangeID]
                LocalContPhenFuncParam2 = self.UserContParam2[LocalObsID][LocalRangeID]
                LocalContPhenFuncParam3 = self.UserContParam3[LocalObsID][LocalRangeID]
                LocalContPhenFuncParam4 = self.UserContParam4[LocalObsID][LocalRangeID]
                LocalContPhenFuncParam5 = self.UserContParam5[LocalObsID][LocalRangeID]
                if (LocalContPhenFuncID is not None):
                    EXPcontrol.write("\n\n            <!-- define parameters for phenomenological continuum description -->\n")
                    EXPcontrol.write("            <ContPhenFuncID>" + str(LocalContPhenFuncID) + "</ContPhenFuncID>\n")
                    if (LocalContPhenFuncParam1 is not None):
                        EXPcontrol.write("            <ContPhenFuncParam1>" + str(LocalContPhenFuncParam1) + "</ContPhenFuncParam1>\n")
                    if (LocalContPhenFuncParam2 is not None):
                        EXPcontrol.write("            <ContPhenFuncParam2>" + str(LocalContPhenFuncParam2) + "</ContPhenFuncParam2>\n")
                    if (LocalContPhenFuncParam3 is not None):
                        EXPcontrol.write("            <ContPhenFuncParam3>" + str(LocalContPhenFuncParam3) + "</ContPhenFuncParam3>\n")
                    if (LocalContPhenFuncParam4 is not None):
                        EXPcontrol.write("            <ContPhenFuncParam4>" + str(LocalContPhenFuncParam4) + "</ContPhenFuncParam4>\n")
                    if (LocalContPhenFuncParam5 is not None):
                        EXPcontrol.write("            <ContPhenFuncParam5>" + str(LocalContPhenFuncParam5) + "</ContPhenFuncParam5>\n")


                ## close range description
                EXPcontrol.write("        </FrequencyRange>\n")


            ## define parameters for current obs. data file
            LocalTelescopeSize = self.UserSizeTelescope[LocalObsID]
            if (LocalTelescopeSize is not None):
                EXPcontrol.write("\n\n        <!-- define size of telescope -->\n")
                EXPcontrol.write("        <TelescopeSize>" + str(LocalTelescopeSize) + "</TelescopeSize>\n")


            ## define BMIN parameter
            LocalBMIN = self.UserBMIN[LocalObsID]
            if (LocalBMIN is not None):
                EXPcontrol.write("\n\n        <!-- define BMIN parameters -->\n")
                EXPcontrol.write("        <BMIN>" + str(LocalBMIN) + "</BMIN>\n")


            ## define BMAJ parameter
            LocalBMAJ = self.UserBMAJ[LocalObsID]
            if (LocalBMAJ is not None):
                EXPcontrol.write("\n\n        <!-- define BMAJ parameters -->\n")
                EXPcontrol.write("        <BMAJ>" + str(LocalBMAJ) + "</BMAJ>\n")


            ## define BPA parameter
            LocalBPA = self.UserBPA[LocalObsID]
            if (LocalBPA is not None):
                EXPcontrol.write("\n\n        <!-- define BPA parameters -->\n")
                EXPcontrol.write("        <BPA>" + str(LocalBPA) + "</BPA>\n")


            ## define interferometer flag
            EXPcontrol.write("\n\n        <!-- define if interferrometric observation is modeled -->\n")
            Inter_Flag = self.UserInterferoFlag[LocalObsID]
            Inter_Flag = task_myXCLASSMapFit.CheckBool(Inter_Flag)
            if (Inter_Flag):
                EXPcontrol.write("        <Inter_Flag>True</Inter_Flag>\n")
            else:
                EXPcontrol.write("        <Inter_Flag>False</Inter_Flag>\n")


            ## define vLSR parameter
            LocalvLSR = self.UservLSR[LocalObsID]
            if (LocalvLSR is not None):
                EXPcontrol.write("\n\n        <!-- define local standard of rest (vLSR) -->\n")
                EXPcontrol.write("        <GlobalvLSR>" + str(LocalvLSR) + "</GlobalvLSR>\n")


            ## define redshift parameter
            LocalRedshift = self.UserRedShift[LocalObsID]
            if (LocalRedshift is not None):
                EXPcontrol.write("\n\n        <!-- define redshift -->\n")
                EXPcontrol.write("        <Redshift>" + str(LocalRedshift) + "</Redshift>\n")


            ## define parameters for import of obs. data file
            EXPcontrol.write("\n\n        <!-- define parameters for ASCII file import -->\n")
            if (self.ErrorYList != [] or self.NumberHeaderLinesList != [] or self.SeparatorColumnsList != []):
                EXPcontrol.write("        <!-- define parameters for ASCII file import -->\n")
            if (self.ErrorYList != []):
                DataFileIndex = int(self.ErrorYList[LocalObsID][0])
                if (DataFileIndex == (LocalObsID + 1)):
                    EXPcontrol.write("        <ErrorY>" + self.ErrorYList[LocalObsID][1] + "</ErrorY>\n")
            if (self.NumberHeaderLinesList != []):
                DataFileIndex = int(self.NumberHeaderLinesList[LocalObsID][0])
                if (DataFileIndex == (LocalObsID + 1)):
                    EXPcontrol.write("        <NumberHeaderLines>" + self.NumberHeaderLinesList[LocalObsID][1] + "</NumberHeaderLines>\n")
            if (self.SeparatorColumnsList != []):
                DataFileIndex = int(self.SeparatorColumnsList[LocalObsID][0])
                if (DataFileIndex == (LocalObsID + 1)):
                    EXPcontrol.write("        <SeparatorColumns>" + self.SeparatorColumnsList[LocalObsID][1] + "</SeparatorColumns>\n")
            EXPcontrol.write("    </file>\n")


        ## define parameters for isotopologues
        EXPcontrol.write("\n\n    <!-- parameters for isotopologues -->\n")
        iso_flag = self.UserUseIsoFlag
        iso_flag = task_myXCLASSMapFit.CheckBool(iso_flag)
        if (iso_flag):
            EXPcontrol.write("    <iso_flag>True</iso_flag>\n")
            EXPcontrol.write("    <IsoTableFileName>" + self.UserIsoRatioFileName + "</IsoTableFileName>\n")
        else:
            EXPcontrol.write("    <iso_flag>False</iso_flag>\n")


        ## define path and name of database file
        EXPcontrol.write("\n\n    <!-- define path and name of database file -->\n")
        EXPcontrol.write("    <dbFilename>" + self.DefaultDB + "</dbFilename>\n")


        ## define number of model pixels along x-and y-direction
        if (self.UserNumModelPixelXX is not None and self.UserNumModelPixelYY is not None):
            star = "*" * 54
            EXPcontrol.write("\n\n    <!-- " + star + " -->\n")
            EXPcontrol.write("    <!-- define number of model pixels along x- and y-direction -->\n")
            EXPcontrol.write("    <NumModelPixelXX>" + str(self.UserNumModelPixelXX) + "</NumModelPixelXX>\n")
            EXPcontrol.write("    <NumModelPixelYY>" + str(self.UserNumModelPixelYY) + "</NumModelPixelYY>\n")


        ## add local-overlap flag
        if (self.UserLocalOverlapFlag is not None):
            star = "*" * 54
            EXPcontrol.write("\n\n    <!-- " + star + " -->\n")
            EXPcontrol.write("    <!-- take local-overlap into account -->\n")
            EXPcontrol.write("    <LocalOverlap_Flag>" + str(self.UserLocalOverlapFlag) + "</LocalOverlap_Flag>\n")


        ## add no-sub-beam flag
        if (self.UserNoSubBeamFlag is not None):
            star = "*" * 54
            EXPcontrol.write("\n\n    <!-- " + star + " -->\n")
            EXPcontrol.write("    <!-- do not use sub-beam description -->\n")
            EXPcontrol.write("    <NoSubBeam_Flag>" + str(self.UserNoSubBeamFlag) + "</NoSubBeam_Flag>\n")


        ## add no-sub-beam flag
        if (self.UserEmAbsPATH is not None):
            star = "*" * 65
            EXPcontrol.write("\n\n    <!-- " + star + " -->\n")
            EXPcontrol.write("    <!-- define path of files describing emission and absorption functions -->\n")
            EXPcontrol.write("    <EmAbsPATH>" + self.UserEmAbsPATH.strip() + "</EmAbsPATH>\n")


        ## close xml file
        EXPcontrol.write("</ExpFiles>\n")
        EXPcontrol.write("\n")
        EXPcontrol.close()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##====================================================================================================================================================
    ##
    ##  subroutines for tab 2
    ##
    ##====================================================================================================================================================


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import ASCII file
    def Tab2ImportFile(self, ASCIIFileName):
        """

    input parameters:
    -----------------

        - ASCIIFileName:            path and name of ASCII file


    output parameters:
    ------------------

        - InterpolASCIIFunc:        interpolation function to compute background function on a new frequency axis
        """

        # Debug:
        # print ("ASCIIFileName = ", ASCIIFileName)


        ## initialize return variable
        InterpolASCIIFunc = None


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check path and name of ASCII file
        if (ASCIIFileName is None):
            ASCIIFileName = ""
        if (ASCIIFileName.strip() == ""):
            return
        if (not ASCIIFileName.startswith("/")):
            LocalPath = os.getcwd() + "/"
            ASCIIFileName = LocalPath + ASCIIFileName
        if (not os.path.exists(ASCIIFileName)):
            msgBox = QtWidgets.QMessageBox()
            msgBox.setWindowTitle("Error")
            LocalLogText = "The file " + ASCIIFileName + "\n"
            LocalLogText += "defined in the obs.xml file does not exist!\n\n"
            LocalLogText += "Please correct the xml file and start again!\n"
            msgBox.setText(LocalLogText)
            ret = msgBox.exec_()
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## import ASCII file
        FileContent = np.loadtxt(ASCIIFileName)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get ASCII values at frequency points
        InterpolASCIIFunc = scipy.interpolate.interp1d(FileContent[:, 0], FileContent[:, 1], bounds_error = False, fill_value = "extrapolate")

        # Debug:
        # print ("InterpolASCIIFunc = ", InterpolASCIIFunc)


        ## we're done
        return InterpolASCIIFunc
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## enables or disables elements of tab 2
    def Tab2setDisEnabled(self, DisEnabledFlag):
        """

    input parameters:
    -----------------

        - DisEnabledFlag:       flag indicating if elements are enabled (True) or disabled (False)


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("DisEnabledFlag = ", DisEnabledFlag)
        # print ("self.UseIsoFlag = ", self.UseIsoFlag)
        # print ("self.CurrentIsoParam = ", self.CurrentIsoParam)


        if (DisEnabledFlag and self.UseIsoFlag):
            self.Edit_Low_IsoRatio.setEnabled(True)
            self.Edit_Up_IsoRatio.setEnabled(True)
            self.Edit_Val_IsoRatio.setEnabled(True)
            self.checkBox_FF_iso.setEnabled(True)
            self.Slider__IsoRatio.setEnabled(True)
            self.comboBox_iso.setEnabled(True)
            self.checkBox_useIso.setEnabled(True)
            self.Tab2SetCurrIsoIndex()
        else:
            self.comboBox_iso.setEnabled(False)
            # self.comboBox_iso.clear()
            self.Edit_Low_IsoRatio.setEnabled(False)
            self.Edit_Up_IsoRatio.setEnabled(False)
            self.Edit_Val_IsoRatio.setEnabled(False)
            self.checkBox_FF_iso.setEnabled(False)
            self.Slider__IsoRatio.setEnabled(False)
            self.checkBox_useIso.setEnabled(False)

        ListParam = ["ss", "Temp", "Ntot", "vwidth", "voff"]
        for Param in ListParam:
            exec("self.checkBox_FF_" + Param + ".setEnabled(DisEnabledFlag)")
            exec("self.Slider__" + Param + ".setEnabled(DisEnabledFlag)")
            exec("self.Edit_Low_" + Param + ".setEnabled(DisEnabledFlag)")
            exec("self.Edit_Up_" + Param + ".setEnabled(DisEnabledFlag)")
            exec("self.Edit_Val_" + Param + ".setEnabled(DisEnabledFlag)")
        self.comboBox_molecule.setEnabled(DisEnabledFlag)
        self.pushButton_AddComp.setEnabled(DisEnabledFlag)
        if (len(self.CurrentMolfitParameter) > 1):
            self.pushButton_rmComp.setEnabled(DisEnabledFlag)
        else:
            self.pushButton_rmComp.setEnabled(False)
        self.pushButton_left.setEnabled(DisEnabledFlag)
        self.pushButton_left_big.setEnabled(DisEnabledFlag)
        self.pushButton_right.setEnabled(DisEnabledFlag)
        self.pushButton_right_big.setEnabled(DisEnabledFlag)
        self.pushButton_up.setEnabled(DisEnabledFlag)
        self.pushButton_down.setEnabled(DisEnabledFlag)
        self.pushButton_Cancel.setEnabled(DisEnabledFlag)

        # Debug:
        # print ("\n\n>>>self.OldParamVectorCounter = ", self.OldParamVectorCounter)
        # print ("(len(self.OldMolfitParameter) - 1) = ", (len(self.OldMolfitParameter) - 1))


        if (self.OldParamVectorCounter == 0 or (self.OldMolfitParameter == [] and self.OldIsoRatioParameter == [])):
            self.pushButton_Undo.setEnabled(False)
        else:
            self.pushButton_Undo.setEnabled(DisEnabledFlag)
        if (self.OldParamVectorCounter < (len(self.OldCurrentMoleculeID) - 1) and (self.OldMolfitParameter != [] or self.OldIsoRatioParameter != [])):
            self.pushButton_Redo.setEnabled(DisEnabledFlag)
        else:
            self.pushButton_Redo.setEnabled(False)

        self.list_molfit.setEnabled(DisEnabledFlag)
        if (self.CFUsageFlag):
            self.label_Dist.setVisible(False)
            self.Edit_Val_Distance.setVisible(False)
            self.radioButton_Core.setVisible(True)
            self.radioButton_Core.setEnabled(DisEnabledFlag)
            self.radioButton_Fore.setVisible(True)
            self.radioButton_Fore.setEnabled(DisEnabledFlag)
        else:
            self.label_Dist.setVisible(True)
            self.label_Dist.setEnabled(DisEnabledFlag)
            self.Edit_Val_Distance.setVisible(True)
            self.Edit_Val_Distance.setEnabled(DisEnabledFlag)
            self.radioButton_Core.setVisible(False)
            self.radioButton_Fore.setVisible(False)
        self.checkBox_active.setEnabled(DisEnabledFlag)
        if (len(self.MoleculesInMolfitFile) == 1):
            self.checkBox_background.setEnabled(False)
        else:
            self.checkBox_background.setEnabled(DisEnabledFlag)
        self.checkBox_rm_cont.setEnabled(DisEnabledFlag)
        self.label_1.setEnabled(DisEnabledFlag)
        self.label_2.setEnabled(DisEnabledFlag)
        self.label_3.setEnabled(DisEnabledFlag)
        self.label_4.setEnabled(DisEnabledFlag)
        self.label_5.setEnabled(DisEnabledFlag)
        if (DisEnabledFlag and self.CurrentMoleculeID > 0):
            self.pushButton_PrevMol.setEnabled(True)
        else:
            self.pushButton_PrevMol.setEnabled(False)
        if (DisEnabledFlag and self.CurrentMoleculeID < (len(self.MoleculesInMolfitFile) - 1)):
            self.pushButton_NextMol.setEnabled(True)
        else:
            self.pushButton_NextMol.setEnabled(False)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check, if all parameters from tab 1 are ok and analyze molfit and obs. xml file
    def Tab2ChangeTab(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("self.MolfitFileChangeFlag = ", self.MolfitFileChangeFlag)
        # print ("self.ObsXMLFileChangeFlag = ", self.ObsXMLFileChangeFlag)
        # print ("self.AlgXMLFileChangeFlag = ", self.AlgXMLFileChangeFlag)
        # print ("self.tabWidget.currentIndex() = ", self.tabWidget.currentIndex())


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if some of the obs. data files are FITS files and if pixel coordinates for all FITS files were selected
        FITSImportFailed = False
        MissedSelectedFIles = []
        for LocalObsDataFileID, LocalObsDataFile in enumerate(self.UserObsDataFileName):
            if (LocalObsDataFile.lower().endswith(".fits")):
                NewObsDataFileName = self.FITSSubDir + "FitsSpectrum__" + str(LocalObsDataFileID + 1) + ".dat"
                if (not (os.path.isfile(NewObsDataFileName))):
                    FITSImportFailed = True
                    MissedSelectedFIles.append(LocalObsDataFileID)
        if (FITSImportFailed):


            ## set active tab index
            self.tabWidget.setCurrentIndex(0)


            ## warning message
            msgBox = QtWidgets.QMessageBox()
            msgBox.setWindowTitle("Error")
            MessageString = "Pixel selections are missing for the following FITS file(s):\n\n"
            for MissedIDCounter, MissedID in enumerate(MissedSelectedFIles):
                LocalFITSFileName = self.UserObsDataFileName[MissedID].split("/")
                if (len(LocalFITSFileName) == 0):
                    LocalFITSFileName = [LocalFITSFileName]
                MessageString += "%3i.)   %s\n" % ((MissedIDCounter + 1), LocalFITSFileName[-1])
            msgBox.setText(MessageString)
            ret = msgBox.exec_()
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if tab 2 can be displayed
        if ((self.MolfitFileName.strip() == "" and self.UserCurrentMolecule.strip() == "") \
            or (self.ObsXMLFileName.strip() == "" and self.UserObsDataFileName[self.ObsDataFileID].strip() == "") \
            or (self.AlgXMLFileName.strip() == "" and self.UserMaxIterLM is None)):
            self.tabWidget.setCurrentIndex(0)
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## disable elements in tab 2
        self.Tab2setDisEnabled(False)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set max. number of plots
        self.MaxNumberSpectrum = self.MaxNumberSpectrumDefault


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze molfit file
        if (self.MolfitFileChangeFlag):


            ## is path for molfit file defined?
            if (self.MolfitFileName.strip() != ""):
                self.LocalMolfitFile = self.MolfitFileName.strip()


            ## path for molfit file is not defined
            elif (self.MolfitFileName.strip() == ""):
                if (self.UserCurrentMolecule.strip() == ""):
                    msgBox = QtWidgets.QMessageBox()
                    msgBox.setWindowTitle("Error")
                    msgBox.setText("Please choose a molecule or define path and name of a molfit file!")
                    ret = msgBox.exec_()
                    self.Tab2setDisEnabled(True)
                    self.tabWidget.setCurrentIndex(0)
                    return
                else:
                    self.LocalMolfitFile = []
                    NewLine = self.UserCurrentMolecule.strip() + "   1\n"
                    self.LocalMolfitFile.append(NewLine)
                    NewLine = ""
                    for LocalLineIndex, LocalLine in enumerate(self.DefaultMolfitParameter):
                        if (LocalLineIndex == 5):
                            NewLine += "    " + LocalLine[0] + "\n"
                            self.LocalMolfitFile.append(NewLine)
                        elif (LocalLineIndex != 2):
                            NewLine += "    %s  %.2f  %.2f   %.2f" % (LocalLine[0], LocalLine[1], LocalLine[2], LocalLine[3])
                        else:
                            NewLine += "    %s  %.2e  %.2e   %.2e" % (LocalLine[0], LocalLine[1], LocalLine[2], LocalLine[3])

                    # Debug:
                    # print ("self.LocalMolfitFile = ", self.LocalMolfitFile)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## analyze full molfit file
            LocalMoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(self.LocalMolfitFile)
            NewRangeFlag = False
            molfitFileWithoutContinuum = []
            self.MolfitContinuumParameters = []
            for LocalMoleculeID, LocalMolecule in enumerate(LocalMoleculesInMolfitFile):
                LowerLocalMolecule = LocalMolecule.lower()


                ## get continuum parameters
                if (LowerLocalMolecule.startswith("cont")):
                    LocalMolfitParameters = self.Tab2GetParameterFromMolfitFile(AllParameters[LocalMoleculeID], NewRangeFlag)
                    self.MolfitContinuumParameters.append([LowerLocalMolecule, LocalMolfitParameters, MolfitFileForEachMolecule[LocalMoleculeID]])
                else:
                    molfitFileWithoutContinuum = molfitFileWithoutContinuum + MolfitFileForEachMolecule[LocalMoleculeID]


            ## analyze molfit file
            self.AllMoleculesMolfitFiles = []
            self.CurrentMoleculeID = 0
            self.MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(molfitFileWithoutContinuum)
            self.CurrentMolecule = self.MoleculesInMolfitFile[self.CurrentMoleculeID]
            self.MolfitFileAllParameters = copy.deepcopy(AllParameters)
            if (self.AllMoleculesMolfitFiles == []):
                self.AllMoleculesMolfitFiles = copy.deepcopy(MolfitFileForEachMolecule)
            for LocalMolfitParametersID, LocalOrigMolfitParameters in enumerate(self.MolfitFileAllParameters):
                LocalMolfitParameters = self.Tab2GetParameterFromMolfitFile(LocalOrigMolfitParameters, NewRangeFlag)
                self.MolfitFileAllParameters[LocalMolfitParametersID] = copy.deepcopy(LocalMolfitParameters)
                if (self.CurrentMoleculeID == LocalMolfitParametersID):
                    self.CurrentMolfitParameter = copy.deepcopy(LocalMolfitParameters)


            ## prepare subdirectories for current molecule
            LocalMoleculeFileName = task_LineIdentification.MoleculeFileName(self.CurrentMolecule)
            self.CurrentMoleculeDir = self.GUIJobDir + LocalMoleculeFileName + "/"
            os.system("rm -rf " + self.CurrentMoleculeDir + "; mkdir " + self.CurrentMoleculeDir)


            ## set self.MolfitFileChangeFlag to false
            self.MolfitFileChangeFlag = False


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze obs. xml file
        if (self.ObsXMLFileChangeFlag):


            ## is path for obs. xml file defined?
            if (self.ObsXMLFileName.strip() != ""):
                UnmodifiedExpXML = self.ObsXMLFileName.strip()
                self.ExpFileList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "FileNamesExpFiles")
                self.ExpFileNameList = []
                for LocalObsFile in self.ExpFileList:
                    LocalPath, LocalFileName = os.path.split(LocalObsFile[1])
                    self.ExpFileNameList = [LocalFileName]
                NumberExpFiles = len(self.ExpFileList)
                self.NumberExpRangesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumberExpRanges")
                self.FreqMinList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "MinExpRange")
                self.FreqMaxList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "MaxExpRange")
                self.FreqStepList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "StepFrequency")
                self.t_back_flagList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "t_back_flag")
                self.tBackList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BackgroundTemperature")
                self.tSlopeList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "TemperatureSlope")
                self.N_HList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "HydrogenColumnDensity")
                self.beta_dustList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "DustBeta")
                self.kappa_1300List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Kappa")
                self.DustFileNameList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "DustFileName")
                self.BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BackgroundFileName")
                self.ContFuncIDPhenList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncID")
                self.ContParam1PhenList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam1")
                self.ContParam2PhenList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam2")
                self.ContParam3PhenList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam3")
                self.ContParam4PhenList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam4")
                self.ContParam5PhenList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam5")
                self.NoiseList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NoiseLevel")
                self.TelescopeSizeList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "TelescopeSize")
                self.BMINList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BMIN")
                self.BMAJList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BMAJ")
                self.BPAList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BPA")
                self.Inter_FlagList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Inter_Flag")
                self.GlobalvLSRList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "GlobalvLSR")
                self.RedshiftList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Redshift")
                self.ErrorYList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ErrorY")
                self.NumberHeaderLinesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumberHeaderLines")
                self.SeparatorColumnsList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "SeparatorColumns")
                self.IsotopologuesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "iso_flag")
                if (self.IsotopologuesList == []):
                    self.IsotopologuesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Isotopologues")
                self.IsoTableFileNameList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "IsoTableFileName")
                self.dbList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "dbFilename")
                if (self.dbList == []):
                    self.dbList = task_myXCLASS.GetDefaultDBFile()
                    self.dbList = [self.dbList]
                self.NumModelPixelXXList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumModelPixelXX")
                self.NumModelPixelYYList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumModelPixelYY")
                self.LocalOverlapFlagList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "LocalOverlap_Flag")
                self.NoSubBeamFlagList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NoSubBeam_Flag")
                self.EmAbsPATHList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "EmAbsPATH")
                if (self.EmAbsPATHList is not None):
                    if (len(self.EmAbsPATHList) > 0):
                        self.actionIgnore_em_and_abs_functions.setEnabled(True)


            ## path for obs. xml file is not defined
            elif (self.ObsXMLFileName.strip() == ""):
                LocalLog = []


                ## get path and name of obs. data file
                if (self.UserObsDataFileName[self.ObsDataFileID].strip() == ""):
                    LocalLog.append("enter a valid path and name of an obs. data file")
                else:
                    LocalObsFile = self.UserObsDataFileName[self.ObsDataFileID].strip()
                    self.ExpFileList = [LocalObsFile]
                    LocalPath, LocalFileName = os.path.split(LocalObsFile)
                    self.ExpFileNameList = [0, LocalFileName]


                ## get parameters for current obs. data file
                RangeIndex = (-1)
                ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, RangeIndex, \
                                                                               NumberRangeListIn = self.UserNumberExpRanges, \
                                                                               TelescopeSizeListIn = self.UserSizeTelescope, \
                                                                               BMIN_ListIn = self.UserBMIN, \
                                                                               BMAJ_ListIn = self.UserBMAJ, \
                                                                               BPA_ListIn = self.UserBPA, \
                                                                               InterFlagListIn = self.UserInterferoFlag, \
                                                                               GlobalvLSRListIn = self.UservLSR, \
                                                                               Redshift_ListIn = self.UserRedShift)


                ## get parameters for current range
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.ObsDataFileID, self.FrequencyRangeID, \
                                                                                FreqMinListIn = self.UserFreqMin, \
                                                                                FreqMaxListIn = self.UserFreqMax, \
                                                                                FreqStepListIn = self.UserFreqStep, \
                                                                                tBackFlagListIn = self.UserTbgFlag, \
                                                                                tBackListIn = self.UserTbg, \
                                                                                tSlopeListIn = self.UserTslope, \
                                                                                N_HListIn = self.UsernH, \
                                                                                beta_dustListIn = self.Userbeta, \
                                                                                kappa_1300ListIn = self.Userkappa, \
                                                                                DustFileNameListIn = self.DustFileNameList, \
                                                                                BackgroundFileNameListIn = self.BackgroundFileNameList, \
                                                                                ContPhenFuncID_ListIn = self.UserContFuncID, \
                                                                                ContPhenFuncParam1_ListIn = self.UserContParam1, \
                                                                                ContPhenFuncParam2_ListIn = self.UserContParam2, \
                                                                                ContPhenFuncParam3_ListIn = self.UserContParam3, \
                                                                                ContPhenFuncParam4_ListIn = self.UserContParam4, \
                                                                                ContPhenFuncParam5_ListIn = self.UserContParam5)
                ## FreqMin
                FreqMinLocal = ObsXMLParameterDictRange['FreqMin']
                if (FreqMinLocal is None):
                    LocalLog.append("enter a valid lower frequency")
                else:
                    self.FreqMinList = [[0, 0, FreqMinLocal]]


                ## FreqMax
                FreqMaxLocal = ObsXMLParameterDictRange['FreqMax']
                if (FreqMaxLocal is None):
                    LocalLog.append("enter a valid upper frequency")
                else:
                    self.FreqMaxList = [[0, 0, FreqMaxLocal]]


                ## check FreqMin and FreqMax
                if (FreqMinLocal is not None and FreqMinLocal >= FreqMaxLocal):
                    LocalLog.append("lower frequency is greater or equal upper frequency")


                ## FreqStep
                FreqStepLocal = ObsXMLParameterDictRange['FreqStep']
                if (FreqStepLocal is None):
                    LocalLog.append("enter a valid step frequency")
                else:
                    self.FreqStepList = [[0, 0, FreqStepLocal]]


                ## t_back_flag
                t_back_flagLocal = ObsXMLParameterDictRange['t_back_flag']
                if (t_back_flagLocal is None):
                    self.t_back_flagList = [0, 0, "False"]
                else:
                    self.t_back_flagList = [0, 0, str(t_back_flagLocal)]


                ## tBack
                tBackLocal = ObsXMLParameterDictRange['tBack']
                if (tBackLocal is None):
                    LocalLog.append("enter a valid background temperature")
                else:
                    self.tBackList = [[0, 0, tBackLocal]]


                ## tSlope
                tSlopeLocal = ObsXMLParameterDictRange['tSlope']
                if (tSlopeLocal is None):
                    LocalLog.append("enter a valid temperature slope")
                else:
                    self.tSlopeList = [[0, 0, tSlopeLocal]]


                ## nH_flagList
                #if (self.UsernHFlag[self.ObsDataFileID][self.FrequencyRangeID] is None):
                #    self.nH_flagList = [0, 0, "False"]
                #else:
                #    self.nH_flagList = [0, 0, str(self.UsernHFlag[self.ObsDataFileID][self.FrequencyRangeID])]


                ## nH
                nHLocal = ObsXMLParameterDictRange['N_H']
                if (nHLocal is None):
                    LocalLog.append("enter a valid nH density")
                else:
                    self.N_HList = [[0, 0, nHLocal]]


                ## kappa
                kappaLocal = ObsXMLParameterDictRange['kappa_1300']
                if (kappaLocal is None):
                    LocalLog.append("enter a valid kappa")
                else:
                    self.kappa_1300List = [[0, 0, kappaLocal]]


                ## beta
                betaLocal = ObsXMLParameterDictRange['beta_dust']
                if (betaLocal is None):
                    LocalLog.append("enter a valid beta")
                else:
                    self.beta_dustList = [[0, 0, betaLocal]]


                ## get path and name of dust file
                DustFileName = ObsXMLParameterDictRange['DustFileName']
                if (DustFileName is None):
                    DustFileName = ""
                self.DustFileNameList.append([0, 0, DustFileName])


                ## get path and name of dust file
                BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                if (BackgroundFileName is None):
                    BackgroundFileName = ""
                self.BackgroundFileNameList.append([0, 0, BackgroundFileName])


                ## get ContPhenFuncID
                LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                self.ContFuncIDPhenList.append([0, 0, LocalContPhenFuncID])


                ## get ContPhenFuncParam1
                LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                self.ContParam1PhenList.append([0, 0, LocalContPhenFuncParam1])


                ## get ContPhenFuncParam2
                LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                self.ContParam2PhenList.append([0, 0, LocalContPhenFuncParam2])


                ## get ContPhenFuncParam3
                LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                self.ContParam3PhenList.append([0, 0, LocalContPhenFuncParam3])


                ## get ContPhenFuncParam4
                LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                self.ContParam4PhenList.append([0, 0, LocalContPhenFuncParam4])


                ## get ContPhenFuncParam5
                LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']
                self.ContParam5PhenList.append([0, 0, LocalContPhenFuncParam5])


                ## SizeTelescope
                TelescopeSizeLocal = ObsXMLParameterDictFile['TelescopeSize']
                if (TelescopeSizeLocal is None):
                    LocalLog.append("enter a valid size of telescope")
                else:
                    self.TelescopeSizeList = [0, str(TelescopeSizeLocal)]


                ## get BMIN
                BMIN = ObsXMLParameterDictFile['BMIN']
                self.BMINList.append([0, BMIN])


                ## get BMAJ
                BMAJ = ObsXMLParameterDictFile['BMAJ']
                self.BMAJList.append([0, BMAJ])


                ## get BPA
                BPA = ObsXMLParameterDictFile['BPA']
                self.BPAList.append([0, BPA])


                ## Inter_Flag
                Inter_FlagLocal = ObsXMLParameterDictFile['InterFlag']
                if (Inter_FlagLocal is None):
                    self.Inter_FlagList = [0, "False"]
                else:
                    self.Inter_FlagList = [0, str(Inter_FlagLocal)]


                ## global v_LSR
                GlobalvLSRLocal = ObsXMLParameterDictFile['GlobalvLSR']
                if (GlobalvLSRLocal is None):
                    LocalLog.append("enter a valid v_LSR velocity")
                else:
                    self.GlobalvLSRList = [0, str(GlobalvLSRLocal)]


                ## get redshift
                Redshift = ObsXMLParameterDictFile['Redshift']
                if (Redshift is None):
                    Redshift = 0.0
                self.RedshiftList.append([0, Redshift])


                ## iso flag
                if (self.UserUseIsoFlag is None):
                    self.IsotopologuesList = ["False"]
                else:
                    self.IsotopologuesList = [str(self.UserUseIsoFlag)]


                ## path and name of iso ratio file
                if (self.UserIsoRatioFileName.strip() == "" and self.IsoFlag):
                    LocalLog.append("enter a valid path and name of an iso ratio file")
                else:
                    self.IsoTableFileNameList = [self.UserIsoRatioFileName.strip()]


                ## did an error occur
                if (LocalLog != []):
                    msgBox = QtWidgets.QMessageBox()
                    msgBox.setWindowTitle("Error")
                    LocalLogText = "Please "
                    for LocalLogLineIndex, LocalLogLine in enumerate(LocalLog):
                        if (LocalLogLineIndex > 0):
                            LocalLogText += "and "
                        LocalLogText += LocalLogLine + "\n"
                    LocalLogText += "\nor define a valid path and name of an obs. xml file!\n"
                    msgBox.setText(LocalLogText)
                    ret = msgBox.exec_()
                    self.Tab2setDisEnabled(True)
                    self.tabWidget.setCurrentIndex(0)
                    return


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## check, if iso ratio file is defined
            if (self.IsotopologuesList != []):
                LocalIsoFlag = str(self.IsotopologuesList[0]).lower()
                LocalIsoFlag = task_myXCLASSMapFit.CheckBool(LocalIsoFlag)
                if (LocalIsoFlag):


                    ## shift frequency range limits by global v_lsr
                    NumberExpFiles = len(self.ExpFileList)
                    ShiftedFreqMinList, ShiftedFreqMaxList = self.GetShiftedFreq(NumberExpFiles, self.NumberExpRangesList, self.GlobalvLSRList, \
                                                                                 self.RedshiftList, self.FreqMinList, self.FreqMaxList)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## reset list of iso ratio file parameter
                    self.UseIsoFlag = False
                    self.AllIsotopologueList = []
                    LocaldbFile = self.dbList[0]
                    LocalUserIsoRatioFileName = self.UserIsoRatioFileName.strip()
                    if (not LocalUserIsoRatioFileName.startswith("/")):
                        LocalUserIsoRatioFileName = os.getcwd() + "/" + LocalUserIsoRatioFileName
                    self.IsoTableFileNameList = [LocalUserIsoRatioFileName]
                    SelectIsoRatioTable = task_myXCLASSFit.CheckMolecules(self.LocalMolfitFile, self.IsoTableFileNameList[0], ShiftedFreqMinList, \
                                                                          ShiftedFreqMaxList, [LocaldbFile])
                    for LocalMolIndex, LocalMol in enumerate(self.MoleculesInMolfitFile):
                        LocalIsoParam = self.Tab2GetIsoRatiosFromFile(LocalMol, SelectIsoRatioTable)
                        if (LocalIsoParam != []):
                            self.AllIsotopologueList.append(copy.deepcopy(LocalIsoParam))
                            for CurrentIsoParam2 in LocalIsoParam:
                                LocalIsoMaster = CurrentIsoParam2[1]
                                if (LocalMolIndex == self.CurrentMoleculeID):
                                    self.CurrentIsoParam = copy.deepcopy(CurrentIsoParam2)
                                    if (self.CurrentIsoParam != []):                        ## check if at least one iso. is defined for current mol.
                                        self.UseIsoFlag = True
                                        if (self.CurrentIsoID > (len(self.CurrentIsoParam) - 1)):
                                            self.CurrentIsoID = (len(self.CurrentIsoParam) - 1)

                    # Debug:
                    # print (">>self.UseIsoFlag = ", self.UseIsoFlag)


            ## everything is fine, create transition frequency windows
            okStatus = self.Tab2NewXMLFile()
            if (str(okStatus) == "1"):
                self.Tab2setDisEnabled(False)
                self.tabWidget.setCurrentIndex(0)
                self.ObsXMLFileChangeFlag = True
                return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze alg. xml file
        if (self.AlgXMLFileChangeFlag):


            ## path of obs. xml file is not defined
            if (self.AlgXMLFileName.strip() == ""):
                if (self.UserMaxIterLM is None):
                    LocalLog = ["Please enter a valid max. number of iterations!"]
                    msgBox = QtWidgets.QMessageBox()
                    msgBox.setWindowTitle("Error")
                    msgBox.setText("Please enter a valid max. number of iterations\n\nor define a valid path and name of an alg. xml file!")
                    ret = msgBox.exec_()
                    self.Tab2setDisEnabled(True)
                    self.tabWidget.setCurrentIndex(0)
                    return


            ## set self.AlgXMLFileChangeFlag to false
            self.AlgXMLFileChangeFlag = False


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## enable elements of menu
        if (self.startFlag):
            self.actionSave_current_molfit_file.setEnabled(True)
            self.actionLoad_new_iso_ratio_file.setEnabled(True)
            self.actionSave_new_iso_ratio_file.setEnabled(True)
            self.actionAnalyzeMolfit.setEnabled(True)
            self.actionAdd_molecule.setEnabled(True)
            self.actionRemove_current_molecule.setEnabled(True)
            self.actionGo_to_previous_molecule.setEnabled(True)
            self.actionGo_to_next_molecule.setEnabled(True)
            self.actionShow_TFW.setEnabled(True)
            self.actionFit_molecule.setEnabled(True)
            self.actionFit_all_molecules.setEnabled(True)
            self.actionMake_a_snapshot.setEnabled(True)
            self.startFlag = False


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## replot elements of tab 2?
        if (self.PlotFlagTab3):
            self.Tab2SetValuesForTab()
            self.PlotFlagTab3 = False


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## enable elements of tab 2
        self.Tab2setDisEnabled(True)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create new xml file
    def Tab2CreateXMLFile(self, LocalExpFileList, RestrictedMolecularParameterList):
        """

    input parameters:
    -----------------


        - LocalExpFileList:         path and names of obs. data files

        - RestrictedMolecularParameterList:     restricted list of molecular parameters


    output parameters:
    ------------------

        - ok:                       status variable
        """

        # Debug:
        # print ("LocalExpFileList = ", LocalExpFileList)
        # print ("RestrictedMolecularParameterList = ", RestrictedMolecularParameterList)


        ## get numpy list of restricted (!!!!) transition frequencies
        TransFreqNPList = np.asarray(RestrictedMolecularParameterList[:, 0])

        # Debug:
        # print ("TransFreqNPList = ", TransFreqNPList)
        # print ("len(TransFreqNPList) = ", len(TransFreqNPList))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create new xml file for current xml file


        ## open xml file and write header
        NewExpXMLFile = open(self.NewExpXMLFileName, 'w')
        NewExpXMLFile.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
        NewExpXMLFile.write("<ExpFiles>\n\n\n")
        NewExpXMLFile.write("    <!-- define number of observation files -->\n")


        ## write parameters for each observational data file
        CounterTransitions = abs(self.MaxNumTransInFit)
        SmoothList = []
        EffectiveObsDataFileCounter = 0
        LinesForXMLFile = []
        NumExpDataFiles = len(LocalExpFileList)
        for ObsDataFileIndex in range(NumExpDataFiles):                                     ## loop over all obs. data files
            helpString = "*" * 141
            LinesForObsFile = ["\n\n    <!-- " + helpString + " -->\n"]
            LinesForObsFile.append("    <!-- define parameters for observation file " + chr(34)
                                         + os.path.basename(LocalExpFileList[ObsDataFileIndex][1]) + chr(34) + " -->\n")
            LinesForObsFile.append("    <file>\n")
            LinesForObsFile.append("        <FileNamesExpFiles>" + LocalExpFileList[ObsDataFileIndex][1] + "</FileNamesExpFiles>\n")
            LinesForObsFile.append("\n\n        <!-- define import filter -->\n")
            LinesForObsFile.append("        <ImportFilter>xclassASCII</ImportFilter>\n")


            ## get number of header lines and character separating columns from obs. xml file
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                           NumberRangeListIn = self.NumberExpRangesList, \
                                                                           TelescopeSizeListIn = self.TelescopeSizeList, \
                                                                           BMIN_ListIn = self.BMINList, \
                                                                           BMAJ_ListIn = self.BMAJList, \
                                                                           BPA_ListIn = self.BPAList, \
                                                                           InterFlagListIn = self.Inter_FlagList, \
                                                                           GlobalvLSRListIn = self.GlobalvLSRList, \
                                                                           Redshift_ListIn = self.RedshiftList, \
                                                                           ErrorYFlagListIn = self.ErrorYList, \
                                                                           NumberHeaderLinesListIn = self.NumberHeaderLinesList, \
                                                                           SeparatorColumnsListIn = self.SeparatorColumnsList)
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']        ## get number of frequency ranges for current obs. data file
            TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
            LocalBMIN = ObsXMLParameterDictFile['BMIN']
            LocalBMAJ = ObsXMLParameterDictFile['BMAJ']
            LocalBPA = ObsXMLParameterDictFile['BPA']
            InterFlag = ObsXMLParameterDictFile['InterFlag']
            GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
            if (GlobalvLSR is None):
                GlobalvLSR = 0.0
            LocalRedshift = ObsXMLParameterDictFile['Redshift']
            if (LocalRedshift is None):
                LocalRedshift = 0.0
            ErrorY = ObsXMLParameterDictFile['ErrorY']
            NumberHeaderLines = ObsXMLParameterDictFile['NumberHeaderLines']
            LocalNumberHeaderLines = NumberHeaderLines
            SeparatorColumns = ObsXMLParameterDictFile['SeparatorColumns']


            ## determine lowest and highest data point
            if (LocalNumberHeaderLines is None):
                LocalNumberHeaderLines = 0
            data = np.loadtxt(LocalExpFileList[ObsDataFileIndex][1], skiprows = int(LocalNumberHeaderLines))
            LowestDataPoint = min(data[:,0])
            HighestDataPoint = max(data[:,0])

            # Debug:
            # print ("LowestDataPoint = ", LowestDataPoint)
            # print ("HighestDataPoint = ", HighestDataPoint)


            ## determine new frequency range definitions and write to xml file
            TotalNumberFreqRanges = 0
            for RangeIndex in range(NumberFrequencyRanges):                                 ## loop over all frequency ranges


                ## get parameter for current frequency range
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = self.FreqMinList, \
                                                                                FreqMaxListIn = self.FreqMaxList, FreqStepListIn = self.FreqStepList, \
                                                                                tBackFlagListIn = self.t_back_flagList, tBackListIn = self.tBackList, \
                                                                                tSlopeListIn = self.tSlopeList, N_HListIn = self.N_HList, \
                                                                                beta_dustListIn = self.beta_dustList, \
                                                                                kappa_1300ListIn = self.kappa_1300List, \
                                                                                DustFileNameListIn = self.DustFileNameList, \
                                                                                BackgroundFileNameListIn = self.BackgroundFileNameList, \
                                                                                ContPhenFuncID_ListIn = self.ContFuncIDPhenList, \
                                                                                ContPhenFuncParam1_ListIn = self.ContParam1PhenList, \
                                                                                ContPhenFuncParam2_ListIn = self.ContParam2PhenList, \
                                                                                ContPhenFuncParam3_ListIn = self.ContParam3PhenList, \
                                                                                ContPhenFuncParam4_ListIn = self.ContParam4PhenList, \
                                                                                ContPhenFuncParam5_ListIn = self.ContParam5PhenList, \
                                                                                NoiseListIn = self.NoiseList, SmoothValueListIn = SmoothList)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None):
                    FreqMax = ObsXMLParameterDictRange['FreqMax']
                    FreqStep = ObsXMLParameterDictRange['FreqStep']
                    t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                    tBack = ObsXMLParameterDictRange['tBack']
                    tSlope = ObsXMLParameterDictRange['tSlope']
                    nH_flag = True
                    N_H = ObsXMLParameterDictRange['N_H']
                    beta_dust = ObsXMLParameterDictRange['beta_dust']
                    kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                    if (N_H is None or beta_dust is None or kappa_1300 is None):
                        nH_flag = False
                        N_H = 0.0
                        beta_dust = 0.0
                        kappa_1300 = 0.0
                    DustFileName = ObsXMLParameterDictRange['DustFileName']
                    BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                    LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                    LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                    LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                    LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                    LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                    LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']
                    Noise = ObsXMLParameterDictRange['NoiseLevel']
                    if (Noise is None):
                        Noise = 0.0
                    SmoothValue = ObsXMLParameterDictRange['SmoothValue']


                    ## shift min. and max. frequency of each range by v_LSR or redshift z
                    FreqMinOrig = FreqMin
                    NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = LocalRedshift, backTrafo = True)
                    NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = LocalRedshift, backTrafo = True)
                    ShiftedFreqMin = min(NewFreqMin, NewFreqMax)
                    ShiftedFreqMax = max(NewFreqMin, NewFreqMax)


                    ## shift min. and max. frequency of obs. data file by v_LSR or redshift z
                    LocalLowestDataPoint1 = task_myXCLASS.ConvertFreq(LowestDataPoint, GlobalvLSR, z = LocalRedshift, backTrafo = True)
                    LocalHighestDataPoint1 = task_myXCLASS.ConvertFreq(HighestDataPoint, GlobalvLSR, z = LocalRedshift, backTrafo = True)
                    LocalLowestDataPoint = min(LocalLowestDataPoint1, LocalHighestDataPoint1)
                    LocalHighestDataPoint = max(LocalLowestDataPoint1, LocalHighestDataPoint1)


                    ## determine number of transition frequencies
                    idx = np.where((TransFreqNPList >= ShiftedFreqMin) & (TransFreqNPList <= ShiftedFreqMax))
                    CounterTransitions = len(TransFreqNPList[idx])
                    if (CounterTransitions > 0):


                        ## define new parameter ranges
                        dbFile = self.dbList[0]
                        LocalListOfMolecules = []
                        MaxWidth = 100.0
                        SQLParamArray = []
                        CounterTransitions = 100000000
                        NewFreqRanges = task_LineIdentification.DefineNewFreqRanges(LocalLowestDataPoint, LocalHighestDataPoint, LocalListOfMolecules, \
                                                                                    ShiftedFreqMin, ShiftedFreqMax, FreqStep, t_back_flag, tBack, \
                                                                                    tSlope, nH_flag, N_H, beta_dust, kappa_1300, DustFileName, \
                                                                                    BackgroundFileName, LocalContPhenFuncID, \
                                                                                    LocalContPhenFuncParam1, LocalContPhenFuncParam2, \
                                                                                    LocalContPhenFuncParam3, LocalContPhenFuncParam4, \
                                                                                    LocalContPhenFuncParam5, Noise, self.velLowLimit, self.velUpLimit, \
                                                                                    MaxWidth, SQLParamArray, dbFile, CounterTransitions, FreqMinOrig, \
                                                                                    TransFreqList = TransFreqNPList, MinimalDist = 0.1)
                        NumberFreqRanges = len(NewFreqRanges)
                        TotalNumberFreqRanges += NumberFreqRanges

                        # Debug:
                        # print ("\n\n>>>FreqMin = ", FreqMin)
                        # print (">>>FreqMax = ", FreqMax)
                        # print (">>>self.CurrentMolecule = ", self.CurrentMolecule)
                        # print (">>>ShiftedFreqMin = ", ShiftedFreqMin)
                        # print (">>>ShiftedFreqMax = ", ShiftedFreqMax)
                        # print (">>>SQLParamArray = ", SQLParamArray)
                        # print (">>>NumberFreqRanges = ", NumberFreqRanges)
                        # print (">>>NewFreqRanges = ", NewFreqRanges)


                        ## write parameters for new freq range(s) to xml file
                        for FreqRange in range(NumberFreqRanges):
                            MinExpRange = float(NewFreqRanges[FreqRange][0])
                            MaxExpRange = float(NewFreqRanges[FreqRange][1])
                            StepFrequency = NewFreqRanges[FreqRange][2]
                            t_back_flag = NewFreqRanges[FreqRange][3]
                            BackgroundTemperature = NewFreqRanges[FreqRange][4]
                            TemperatureSlope = NewFreqRanges[FreqRange][5]
                            nH_flag = NewFreqRanges[FreqRange][6]
                            HydrogenColumnDensity = NewFreqRanges[FreqRange][7]
                            DustBeta = NewFreqRanges[FreqRange][8]
                            Kappa = NewFreqRanges[FreqRange][9]
                            DustFileName = NewFreqRanges[FreqRange][10]
                            BackgroundFileName = NewFreqRanges[FreqRange][11]
                            ContPhenFuncID = NewFreqRanges[FreqRange][12]
                            ContPhenFuncParam1 = NewFreqRanges[FreqRange][13]
                            ContPhenFuncParam2 = NewFreqRanges[FreqRange][14]
                            ContPhenFuncParam3 = NewFreqRanges[FreqRange][15]
                            ContPhenFuncParam4 = NewFreqRanges[FreqRange][16]
                            ContPhenFuncParam5 = NewFreqRanges[FreqRange][17]
                            NoiseLevel = NewFreqRanges[FreqRange][18]

                            # Debug:
                            # print ("\n\nFreqRange = ", FreqRange)
                            # print ("MinExpRange = ", MinExpRange)
                            # print ("MaxExpRange = ", MaxExpRange)
                            # print ("GlobalvLSR = ", GlobalvLSR)


                            ## shift min. and max. frequency of each range back by v_LSR
                            f1 = task_myXCLASS.ConvertFreq(MinExpRange, GlobalvLSR, z = LocalRedshift)
                            f2 = task_myXCLASS.ConvertFreq(MaxExpRange, GlobalvLSR, z = LocalRedshift)
                            MinExpRange = min(f1, f2)
                            MaxExpRange = max(f1, f2)


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## write parameter for current range to xml file
                            LinesForObsFile.append("\n\n        <!-- define parameters for each data range -->\n")
                            LinesForObsFile.append("        <FrequencyRange>\n")
                            LinesForObsFile.append("            <MinExpRange>" + str(MinExpRange) + "</MinExpRange>\n")
                            LinesForObsFile.append("            <MaxExpRange>" + str(MaxExpRange) + "</MaxExpRange>\n")
                            LinesForObsFile.append("            <StepFrequency>" + str(StepFrequency) + "</StepFrequency>\n")


                            ## define parameters for background continuum
                            LinesForObsFile.append("\n\n            <!-- define background temperature and temperature slope -->\n")
                            LinesForObsFile.append("            <t_back_flag>" + t_back_flag + "</t_back_flag>\n")
                            LinesForObsFile.append("            <BackgroundTemperature>" + str(BackgroundTemperature) + "</BackgroundTemperature>\n")
                            LinesForObsFile.append("            <TemperatureSlope>" + str(TemperatureSlope) + "</TemperatureSlope>\n")
                            if (BackgroundFileName is not None):
                                BackgroundFileName = BackgroundFileName.strip()
                                if (BackgroundFileName != ""):
                                    LinesForObsFile.append("\n\n            <!-- define path and name of background file -->\n")
                                    LinesForObsFile.append("            <BackgroundFileName>" + BackgroundFileName + "</BackgroundFileName>\n")


                            ## define parameters for dust continuum
                            if (nH_flag.lower() == "true"):
                                LinesForObsFile.append("\n\n            <!-- define hydrogen column density, beta for dust, and kappa -->\n")
                                LinesForObsFile.append("            <HydrogenColumnDensity>" + str(HydrogenColumnDensity) + "</HydrogenColumnDensity>\n")
                                LinesForObsFile.append("            <DustBeta>" + str(DustBeta) + "</DustBeta>\n")
                                LinesForObsFile.append("            <Kappa>" + str(Kappa) + "</Kappa>\n")
                            if (DustFileName is not None):
                                DustFileName = DustFileName.strip()
                                if (DustFileName != ""):
                                    LinesForObsFile.append("\n\n            <!-- define path and name of dust file -->\n")
                                    LinesForObsFile.append("            <DustFileName>" + DustFileName + "</DustFileName>\n")


                            ## define function ID for phenomenological continuum description
                            if (ContPhenFuncID != "None"):
                                LinesForObsFile.append("\n\n            <!-- define parameters for phenomenological continuum description -->\n")
                                LinesForObsFile.append("            <ContPhenFuncID>" + ContPhenFuncID + "</ContPhenFuncID>\n")
                                if (ContPhenFuncParam1 != "None"):
                                    LinesForObsFile.append("            <ContPhenFuncParam1>" + ContPhenFuncParam1 + "</ContPhenFuncParam1>\n")
                                if (ContPhenFuncParam2 != "None"):
                                    LinesForObsFile.append("            <ContPhenFuncParam2>" + ContPhenFuncParam2 + "</ContPhenFuncParam2>\n")
                                if (ContPhenFuncParam3 != "None"):
                                    LinesForObsFile.append("            <ContPhenFuncParam3>" + ContPhenFuncParam3 + "</ContPhenFuncParam3>\n")
                                if (ContPhenFuncParam4 != "None"):
                                    LinesForObsFile.append("            <ContPhenFuncParam4>" + ContPhenFuncParam4 + "</ContPhenFuncParam4>\n")
                                if (ContPhenFuncParam5 != "None"):
                                    LinesForObsFile.append("            <ContPhenFuncParam5>" + ContPhenFuncParam5 + "</ContPhenFuncParam5>\n")


                            ## define noise level
                            LinesForObsFile.append("\n\n            <!-- define noise level for current frequency range -->\n")
                            LinesForObsFile.append("            <NoiseLevel>" + str(NoiseLevel) + "</NoiseLevel>\n")


                            ## define smooth values
                            if (SmoothValue is not None):
                                LinesForObsFile.append("\n\n            <!-- define smooth value for current frequency range -->\n")
                                LinesForObsFile.append("            <SmoothValue>" + str(SmoothValue) + "</SmoothValue>\n")
                            LinesForObsFile.append("        </FrequencyRange>\n")


            ## add number of frequency ranges
            LinesForObsFile.append("\n\n        <!-- define number of data ranges -->\n")
            LinesForObsFile.append("        <NumberExpRanges>" + str(TotalNumberFreqRanges) + "</NumberExpRanges>\n")


            ## write rest settings of frequency range definition
            if (TelescopeSize is not None):
                LinesForObsFile.append("\n\n        <!-- define size of telescope -->\n")
                LinesForObsFile.append("        <TelescopeSize>" + str(TelescopeSize) + "</TelescopeSize>\n")
            if (LocalBMIN is not None):
                LinesForObsFile.append("\n\n        <!-- define BMIN parameters -->\n")
                LinesForObsFile.append("        <BMIN>" + str(LocalBMIN) + "</BMIN>\n")
            if (LocalBMAJ is not None):
                LinesForObsFile.append("\n\n        <!-- define BMAJ parameters -->\n")
                LinesForObsFile.append("        <BMAJ>" + str(LocalBMAJ) + "</BMAJ>\n")
            if (LocalBPA is not None):
                LinesForObsFile.append("\n\n        <!-- define BPA parameters -->\n")
                LinesForObsFile.append("        <BPA>" + str(LocalBPA) + "</BPA>\n")
            LinesForObsFile.append("\n\n        <!-- define if interferrometric observation is modeled -->\n")
            LinesForObsFile.append("        <Inter_Flag>" + str(InterFlag) + "</Inter_Flag>\n")
            LinesForObsFile.append("\n\n        <!-- define local standard of rest (vLSR) -->\n")
            LinesForObsFile.append("        <GlobalvLSR>" + str(GlobalvLSR) + "</GlobalvLSR>\n")
            if (LocalRedshift is not None):
                LinesForObsFile.append("\n\n        <!-- define redshift parameter -->\n")
                LinesForObsFile.append("        <Redshift>" + str(LocalRedshift) + "</Redshift>\n")


            ## write parameters for import of obs. data file
            LinesForObsFile.append("\n\n        <!-- define parameters for ASCII file import -->\n")
            LinesForObsFile.append("        <ErrorY>" + str(ErrorY) + "</ErrorY>\n")
            LinesForObsFile.append("        <NumberHeaderLines>" + str(NumberHeaderLines) + "</NumberHeaderLines>\n")
            LinesForObsFile.append("        <SeparatorColumns>" + SeparatorColumns + "</SeparatorColumns>\n")
            LinesForObsFile.append("    </file>\n")


            ## write parameters for current obs. data file to global output array, if at least one frequency array is defined
            if (TotalNumberFreqRanges > 0):
                EffectiveObsDataFileCounter += 1
                for line in LinesForObsFile:
                    LinesForXMLFile.append(line)


        ## write effective total number of obs. data file and parameters for obs. data file(s) to xml file
        NewExpXMLFile.write("    <NumberExpFiles>" + str(EffectiveObsDataFileCounter)  + "</NumberExpFiles>\n")
        for line in LinesForXMLFile:
            NewExpXMLFile.write(line)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write rest of xml file


        ## define iso flag and ratio file
        if (self.IsoTableFileNameList != []):
            star = "*" * (len(self.IsoTableFileNameList[0].strip()) + 41 - 13)
        else:
            star = "*" * 7
        NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
        NewExpXMLFile.write("    <!-- parameters for isotopologues -->\n")
        iso_flag = str(self.IsotopologuesList[0])
        iso_flag = iso_flag.lower()
        iso_flag = task_myXCLASSMapFit.CheckBool(iso_flag)
        if (iso_flag):
            NewExpXMLFile.write("    <iso_flag>True</iso_flag>\n")
            NewExpXMLFile.write("    <IsoTableFileName>" + self.IsoTableFileNameList[0].strip() + "</IsoTableFileName>\n")
        else:
            NewExpXMLFile.write("    <iso_flag>False</iso_flag>\n")
            NewExpXMLFile.write("    <IsoTableFileName></IsoTableFileName>\n")


        ## define path and name of database file
        if (self.dbList != []):
            dbFilename = self.dbList[0].strip()
            if (dbFilename != ""):
                star = "*" * (len(dbFilename) + 29 - 13)
                NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
                NewExpXMLFile.write("    <!-- define path and name of database file -->\n")
                NewExpXMLFile.write("    <dbFilename>" + dbFilename + "</dbFilename>\n")


        ## define number of model pixels
        if (self.NumModelPixelXXList is not None and self.NumModelPixelYYList is not None):
            if (self.NumModelPixelXXList != [] and self.NumModelPixelYYList != []):
                if (self.NumModelPixelXXList[0] is not None and self.NumModelPixelYYList[0]):
                    star = "*" * 54
                    NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
                    NewExpXMLFile.write("    <!-- define number of model pixels along x- and y-direction -->\n")
                    NewExpXMLFile.write("    <NumModelPixelXX>" + str(self.NumModelPixelXXList[0]) + "</NumModelPixelXX>\n")
                    NewExpXMLFile.write("    <NumModelPixelYY>" + str(self.NumModelPixelYYList[0]) + "</NumModelPixelYY>\n")


        ## add local-overlap flag
        if (self.LocalOverlapFlagList is not None):
            if (self.LocalOverlapFlagList != []):
                if (self.LocalOverlapFlagList[0] is not None):
                    star = "*" * 54
                    NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
                    NewExpXMLFile.write("    <!-- take local-overlap into account -->\n")
                    NewExpXMLFile.write("    <LocalOverlap_Flag>" + str(self.LocalOverlapFlagList[0]) + "</LocalOverlap_Flag>\n")


        ## add no-sub-beam flag
        if (self.NoSubBeamFlagList is not None):
            if (self.NoSubBeamFlagList != []):
                if (self.NoSubBeamFlagList[0] is not None):
                    star = "*" * 54
                    NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
                    NewExpXMLFile.write("    <!-- do not use sub-beam description -->\n")
                    NewExpXMLFile.write("    <NoSubBeam_Flag>" + str(self.NoSubBeamFlagList[0]) + "</NoSubBeam_Flag>\n")


        ## add no-sub-beam flag
        if (self.EmAbsPATHList is not None):
            if (self.EmAbsPATHList != []):
                if (self.EmAbsPATHList[0] is not None):
                    star = "*" * 65
                    NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
                    NewExpXMLFile.write("    <!-- define path of files describing emission and absorption functions -->\n")
                    NewExpXMLFile.write("    <EmAbsPATH>" + str(self.EmAbsPATHList[0]) + "</EmAbsPATH>\n")


        ## write footer of obs. xml file
        NewExpXMLFile.write("</ExpFiles>\n")
        NewExpXMLFile.close()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create new xml file and determine transition frequency window spectra
    def Tab2NewXMLFile(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - ok:                       status variable
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define some constants
        self.lock = 1
        ok = 0


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## start progress bar and center dialog window on screen
        TotalCalls = len(self.ExpFileList) + 13
        bar = ProgressBar(total = TotalCalls, widgetTitle = "Prepare Tab Single Molecule Fit for " + self.CurrentMolecule, \
                          bodyText = "Prepare modified XML file for " + chr(34) + self.CurrentMolecule + chr(34) + "..")
        bar.resize(400, 100)
        desktopSize = QtWidgets.QDesktopWidget().screenGeometry()
        top = int((desktopSize.height() / 2.0) - 50)
        left = int((desktopSize.width() / 2.0) - 200)
        bar.move(left, top)
        bar.show()


        ## update progress bar
        ProgressBarValue = 2
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return ok

        # Debug:
        # print ("\n\n\nself.CurrentMolecule = ", self.CurrentMolecule)
        # print ("self.velLowLimit = ", self.velLowLimit)
        # print ("self.velUpLimit = ", self.velUpLimit)
        # print ("self.ExpFileList = ", self.ExpFileList)
        # print ("self.FreqMinList = ", self.FreqMinList)
        # print ("self.FreqMaxList = ", self.FreqMaxList)
        # print ("self.FreqStepList = ", self.FreqStepList)
        # print ("self.t_back_flagList = ", self.t_back_flagList)
        # print ("self.tBackList = ", self.tBackList)
        # print ("self.tSlopeList = ", self.tSlopeList)
        # print ("self.N_HList = ", self.N_HList)
        # print ("self.beta_dustList = ", self.beta_dustList)
        # print ("self.kappa_1300List = ", self.kappa_1300List)
        # print ("self.DustFileNameList = ", self.DustFileNameList)
        # print ("self.BackgroundFileNameList = ", self.BackgroundFileNameList)
        # print ("self.NoiseList = ", self.NoiseList)
        # print ("self.GlobalvLSRList = ", self.GlobalvLSRList)
        # print ("self.TelescopeSizeList = ", self.TelescopeSizeList)
        # print ("self.Inter_FlagList = ", self.Inter_FlagList)
        # print ("self.ErrorYList = ", self.ErrorYList)
        # print ("self.NumberHeaderLinesList = ", self.NumberHeaderLinesList)
        # print ("self.SeparatorColumnsList = ", self.SeparatorColumnsList)
        # print ("self.IsotopologuesList = ", self.IsotopologuesList)
        # print ("self.IsoTableFileNameList = ", self.IsoTableFileNameList)
        # print ("self.dbList = ", self.dbList)
        # print ("self.SQLParamArray = ", self.SQLParamArray)
        # print ("self.MaxNumTransInFit = ", self.MaxNumTransInFit)
        # print ("TotalCalls = ", TotalCalls)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create a subdirectory within job directory for current molecule if directory does not already exist
        if (not os.path.isdir(self.CurrentMoleculeDir)):
            cmdString = "mkdir -p " + self.CurrentMoleculeDir
            os.system(cmdString)

            # Debug:
            # print ("cmdString = ", cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## adjust the path definition for obs. data and obs. xml files
        LocalExpFileList = []
        for LocalExpFileID, LocalExpFile in enumerate(self.ExpFileList):
            if (LocalExpFile[1].lower().endswith(".fits")):
                NewObsDataFile = self.FITSSubDir + "FitsSpectrum__" + str(LocalExpFileID + 1) + ".dat"
                LocalExpFileList.append([LocalExpFileID, NewObsDataFile])
            else:
                LocalExpFileList.append([LocalExpFileID, LocalExpFile[1]])
        self.NewExpXMLFileName = self.CurrentMoleculeDir + "obs.xml"
        MaxWidth = 100.0

        # Debug:
        # print ("LocalExpFileList = ", LocalExpFileList)
        # print ("self.NewExpXMLFileName = ", self.NewExpXMLFileName)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get a list of all transition frequencies in all selected frequency ranges


        ## update progress bar
        ProgressBarValue += 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return ok
        print ("Get a list of all transition frequencies in all selected frequency ranges ..")


        ## define doppler-shifted frequency ranges
        NumExpDataFiles = len(LocalExpFileList)                                             ## get number of obs. data files
        LocalFreqMinList = []
        PureFreqMinList = []
        LocalFreqMaxList = []
        PureFreqMaxList = []
        for ObsDataFileIndex in range(NumExpDataFiles):                                     ## loop over all obs. data files


            ## get number of header lines and character separating columns etc. from obs. xml file
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                           NumberRangeListIn = self.NumberExpRangesList, \
                                                                           GlobalvLSRListIn = self.GlobalvLSRList, \
                                                                           Redshift_ListIn = self.RedshiftList)
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']        ## get number of frequency ranges for current obs. data file
            GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
            if (GlobalvLSR is None):
                GlobalvLSR = 0.0
            Redshift = ObsXMLParameterDictFile['Redshift']
            if (Redshift is None):
                Redshift = 0.0


            ## analyze frequency ranges
            for RangeIndex in range(NumberFrequencyRanges):                                 ## loop over all frequency ranges


                ## get parameter for current frequency range
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                FreqMinListIn = self.FreqMinList, \
                                                                                FreqMaxListIn = self.FreqMaxList)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None):
                    FreqMax = ObsXMLParameterDictRange['FreqMax']


                    ## determine doppler shifted min. and max. frequencies
                    NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
                    NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
                    ShiftedFreqMin = min(NewFreqMin, NewFreqMax)
                    ShiftedFreqMax = max(NewFreqMin, NewFreqMax)
                    LocalFreqMinList.append([ObsDataFileIndex, RangeIndex, ShiftedFreqMin])
                    PureFreqMinList.append(ShiftedFreqMin)
                    LocalFreqMaxList.append([ObsDataFileIndex, RangeIndex, ShiftedFreqMax])
                    PureFreqMaxList.append(ShiftedFreqMax)

                    # Debug:
                    # print ("\n\nGlobalvLSR = ", GlobalvLSR)
                    # print ("Redshift = ", Redshift)
                    # print ("FreqMin = ", FreqMin)
                    # print ("ShiftedFreqMin = ", ShiftedFreqMin)
                    # print ("FreqMax = ", FreqMax)
                    # print ("ShiftedFreqMax = ", ShiftedFreqMax)

        # Debug:
        # print ("\n\n\nself.CurrentMolecule = ", self.CurrentMolecule)
        # print ("LocalFreqMinList = ", LocalFreqMinList)
        # print ("PureFreqMinList  = ", PureFreqMinList)
        # print ("LocalFreqMaxList = ", LocalFreqMaxList)
        # print ("PureFreqMaxList  = ", PureFreqMaxList)
        # print ("self.MaxNumTransInFit = ", self.MaxNumTransInFit)


        ## update progress bar
        ProgressBarValue += 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return ok


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## get isotopologues (if defined) for current molecule which have at least one transitions in one or more unshrinked frequency ranges
        ListOfMoleculesAndIso = [self.CurrentMolecule]
        self.IsoFlag = task_myXCLASSMapFit.CheckBool(self.IsotopologuesList[0])
        # if (self.UseIsoFlag and self.IsoFlag):
        if (self.IsoFlag):
            LocaldbFile = self.dbList[0]
            IsoFileName = self.IsoTableFileNameList[0]
            SelectIsoRatioTable = task_myXCLASSFit.CheckMolecules(self.LocalMolfitFile, IsoFileName, LocalFreqMinList, LocalFreqMaxList, [LocaldbFile])
            LocalCurrentIsoParam = self.Tab2GetIsoRatiosFromFile(self.CurrentMolecule, SelectIsoRatioTable)


            ## get all isotopologues which correspond to the current molecule
            for i, LocalIsotopologue in enumerate(LocalCurrentIsoParam):
                if (LocalIsotopologue[0] != self.CurrentMolecule and LocalIsotopologue[1] == self.CurrentMolecule):
                    if (not LocalIsotopologue[0] in ListOfMoleculesAndIso):
                        ListOfMoleculesAndIso.append(LocalIsotopologue[0])

        # Debug:
        # print ("\n\nself.IsoFlag = ", self.IsoFlag)
        # print ("self.IsotopologuesList = ", self.IsotopologuesList)
        # print ("\n\n>>ListOfMoleculesAndIso = ", ListOfMoleculesAndIso)
        # print ("LocalFreqMinList = ", LocalFreqMinList)
        # print ("LocalFreqMaxList = ", LocalFreqMaxList)
        # print ("self.SQLParamArray = ", self.SQLParamArray)


        ## define local list of SQL parameters
        MinNumTransitionsSQL = self.SQLParamArray[0]
        MaxNumTransitionsSQL = 0
        MaxElowSQL = 1.e99
        #        MaxElowSQL = self.SQLParamArray[3]
        MingASQL = 0.0
        OrderTransSQL = self.SQLParamArray[4]
        LocalSQLParamArray = [MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL]

        # Debug:
        # print ("LocalSQLParamArray = ", LocalSQLParamArray)
        # print ("OrderTransSQL = ", OrderTransSQL)


        ## update progress bar
        ProgressBarValue += 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return ok
        print ("Get transition parameters from database for all selected doppler-shifted frequency ranges ..")


        ##************************************************************************************************************************************************
        ##************************************************************************************************************************************************
        ## get transition parameters from database for all selected doppler-shifted frequency ranges
        LocaldbFile = self.dbList[0]
        LocalMaxNumTransInFit = 1e99
        GlobalTransFreqList, GlobalDBParamList = task_LineIdentification.GetTransFreq(ListOfMoleculesAndIso, PureFreqMinList, PureFreqMaxList, \
                                                                                      LocalSQLParamArray, LocaldbFile, LocalMaxNumTransInFit)
        # Debug:
        # print ("ListOfMoleculesAndIso = ", ListOfMoleculesAndIso)
        # print ("PureFreqMinList = ", PureFreqMinList)
        # print ("PureFreqMaxList = ", PureFreqMaxList)
        # print ("LocalSQLParamArray = ", LocalSQLParamArray)
        # print ("LocaldbFile = ", LocaldbFile)
        # print ("LocalMaxNumTransInFit = ", LocalMaxNumTransInFit)
        # print ("\n\nGlobalTransFreqList = ", GlobalTransFreqList)
        # print ("len(GlobalTransFreqList) = ", len(GlobalTransFreqList))
        # print ("GlobalDBParamList = ", GlobalDBParamList)


        ## update progress bar
        if (GlobalDBParamList == []):
            print ("\n\nNo transitions found for ", ListOfMoleculesAndIso , "!")
            ok = 1
            self.lock = 0
            return ok


        ## update progress bar
        ProgressBarValue += 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\n\nAbort")
            self.lock = 0
            return ok
        print ("Sort transition frequencies in ascending order ..")


        ## sort GlobalDBParamList by transition frequencies in ascending order
        ## DBParamList.append([TransFreq, EinsteinA, Elow, gup, LocalMoleculeName])
        GlobalDBParamNPList = np.asarray(GlobalDBParamList)
        GlobalDBParamNPList = GlobalDBParamNPList[GlobalDBParamNPList[:, 0].argsort()]

        # Debug:
        # print ("GlobalDBParamNPList[:, 0] = ", GlobalDBParamNPList[:, 0])
        # print ("GlobalDBParamNPList[:, 1] = ", GlobalDBParamNPList[:, 1])
        # print ("GlobalDBParamNPList[:, 2] = ", GlobalDBParamNPList[:, 2])
        # print ("GlobalDBParamNPList[:, 3] = ", GlobalDBParamNPList[:, 3])
        # print ("GlobalDBParamNPList[:, 4] = ", GlobalDBParamNPList[:, 4])


        ## define helper vectors
        TransFreqVector = np.asarray(GlobalDBParamNPList[:, 0], dtype='float64')
        EinsteinAVector = np.asarray(GlobalDBParamNPList[:, 1], dtype='float64')
        ELowVector = np.asarray(GlobalDBParamNPList[:, 2], dtype='float64')
        gupVector = np.asarray(GlobalDBParamNPList[:, 3], dtype='float64')
        NameVector = GlobalDBParamNPList[:, 4]
        self.PureMolNameVector = list(dict.fromkeys(NameVector))
        self.PureMolNameVector.remove(self.CurrentMolecule)                                 ## make sure, that current molecule is always
        self.PureMolNameVector.insert(0, self.CurrentMolecule)                              ## at the first element in list self.PureMolNameVector
        MolNameIndexVector = [self.PureMolNameVector.index(x) for x in NameVector]

        # Debug:
        # print ("self.PureMolNameVector = ", self.PureMolNameVector)
        # print ("MolNameIndexVector = ", MolNameIndexVector)
        # print ("len(ELowVector) = ", len(ELowVector))
        # print ("np.min(ELowVector) = ", np.min(ELowVector))


        ## update progress bar
        ProgressBarValue += 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return ok
        print ("Create final unrestricted list of molecular parameters ordered by increasing transition frequency ..")


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## create final unrestricted list of molecular parameters ordered by increasing transition frequency
        MolecularParameterList = np.zeros( (len(GlobalTransFreqList), 7), dtype = np.float32)
        MolecularParameterList[:, 0] = TransFreqVector                                      ## first column: trans. freq
        MolecularParameterList[:, 1] = ELowVector                                           ## second column: Elow
        MolecularParameterList[:, 2] = EinsteinAVector * gupVector                          ## third column: (gup * Einstein A)
        MolecularParameterList[:, 2] = np.where((MolecularParameterList[:, 2] != 0.0), 1.0 / MolecularParameterList[:, 2], 0.0)
        MolecularParameterList[:, 3] = (EinsteinAVector * gupVector) / (ELowVector**2)      ## fourth column: (gup * Einstein A) / Elow**2
        MolecularParameterList[:, 3] = np.where((MolecularParameterList[:, 3] != 0.0), 1.0 / MolecularParameterList[:, 3], 0.0)
        MolecularParameterList[:, 4] = EinsteinAVector                                      ## fifth column: Einstein A coefficients
        MolecularParameterList[:, 5] = gupVector                                            ## sixth column: upper state degeneracy
        MolecularParameterList[:, 6] = np.asarray(MolNameIndexVector)                       ## seventh column: LocalMoleculeName

        # Debug:
        # print ("MolecularParameterList[:, 0] = ", MolecularParameterList[:, 0])
        # print ("MolecularParameterList[:, 1] = ", MolecularParameterList[:, 1])
        # print ("MolecularParameterList[:, 2] = ", MolecularParameterList[:, 2])
        # print ("MolecularParameterList[:, 3] = ", MolecularParameterList[:, 3])
        # print ("MolecularParameterList[:, 4] = ", MolecularParameterList[:, 4])
        # print ("MolecularParameterList[:, 5] = ", MolecularParameterList[:, 5])
        # print ("MolecularParameterList[:, 6] = ", MolecularParameterList[:, 6])


        ## sort unrestricted list by given order index
        MolecularParameterList = MolecularParameterList[MolecularParameterList[:, OrderTransSQL].argsort()]


        ## update progress bar
        ProgressBarValue += 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return ok
        print ("Created restricted list of molecular parameters according SQL parameters ..")


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## created restricted list of molecular parameters according SQL parameters
        RestrictedMolecularParameterList = MolecularParameterList[(MolecularParameterList[:, 1] <= self.SQLParamArray[2]) \
                                                                & (MingASQL <= MolecularParameterList[:, 3]), :]
        # Debug:
        # print ("self.SQLParamArray[2] = ", self.SQLParamArray[2])
        # print ("self.SQLParamArray[3] = ", self.SQLParamArray[3])
        # print ("MingASQL = ", MingASQL)
        # print ("GlobalDBParamNPList[:, 2] = ", GlobalDBParamNPList[:, 2])
        # print ("np.min(MolecularParameterList[:, 1]) = ", np.min(MolecularParameterList[:, 1]))
        # print ("np.min(RestrictedMolecularParameterList[:, 1]) = ", np.min(RestrictedMolecularParameterList[:, 1]))
        # print ("np.max(RestrictedMolecularParameterList[:, 3]) = ", np.max(RestrictedMolecularParameterList[:, 3]))
        #    sys.exit(0)


        ## truncate restricted list at self.MaxNumTransInFit
        LocalMaxNumTrans = max(1, self.MaxNumTransInFit)
        LocalMaxNumTrans = min(LocalMaxNumTrans, len(RestrictedMolecularParameterList))
        RestrictedMolecularParameterList = RestrictedMolecularParameterList[:LocalMaxNumTrans, :]

        # Debug:
        # print ("LocalMaxNumTrans = ", LocalMaxNumTrans)
        # print ("RestrictedMolecularParameterList = ", RestrictedMolecularParameterList)

        if (len(RestrictedMolecularParameterList) == 0):
            ProgressBarValue = TotalCalls
            ok = bar.setValue(ProgressBarValue)

            # Debug:
            # print ("cancel!\n\n")


            msgBox = QtWidgets.QMessageBox()
            msgBox.setWindowTitle("Error")
            LocalLogText = "The selected molecule has no transition within given frequency range(s).\n\n"
            LocalLogText += "Please define another molecule or expand range definition(s)!\n"
            msgBox.setText(LocalLogText)
            ret = msgBox.exec_()

            self.PlotFlagTab3 = False
            self.MolfitFileChangeFlag = True
            self.ObsXMLFileChangeFlag = True
            self.CurrentMolecule = ""
            self.CurrentMoleculeID = 0
            self.UserCurrentMoleculeID = 0
            self.CurrentMolfitFileContents = []
            self.CurrentMolfitParameter = []
            self.CurrentMoleculeDir = ""
            self.CurrentComponentIndex = 0
            self.CurrentMolfitIndex = 1
            self.ActiveCompList = []
            self.UseIsoFlag = False
            self.CurrentIsoParam = []
            self.AllIsotopologueList = []
            self.CurrentIsoID = 0
            ok = 1
            self.lock = 0
            return ok


        ## update progress bar
        ProgressBarValue += 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return ok
        print ("Extract only the relevant molecular parameters for spectrum modeling ..")


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## extract only the relevant molecular parameters for spectrum modeling
        col_idx = [0, 4, 1, 5, 6]                                                           ## we need only TransFreq, EinsteinA, Elow, gup, Mol-index
        RelevantMolecularParameterList = MolecularParameterList[:, col_idx]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create new xml file for current xml file
        self.Tab2CreateXMLFile(LocalExpFileList, RestrictedMolecularParameterList)


        ## update progress bar
        ProgressBarValue += 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return ok
        print ("Get parameters from shrinked observational xml file ..")


        ## get path and names of observational data files from shrinked xml file
        NewExpFileList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "FileNamesExpFiles")
        NewNumberExpFiles = len(NewExpFileList)
        NewNumberExpRangesList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "NumberExpRanges")
        NewFreqMinList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "MinExpRange")
        NewFreqMaxList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "MaxExpRange")
        NewFreqStepList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "StepFrequency")
        Newt_back_flagList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "t_back_flag")
        NewtBackList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "BackgroundTemperature")
        NewtSlopeList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "TemperatureSlope")
        NewN_HList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "HydrogenColumnDensity")
        Newbeta_dustList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "DustBeta")
        Newkappa_1300List = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "Kappa")
        NewDustFileNameList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "DustFileName")
        NewBackgroundFileNameList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "BackgroundFileName")
        NewContFuncIDPhenList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "ContPhenFuncID")
        NewContParam1PhenList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "ContPhenFuncParam1")
        NewContParam2PhenList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "ContPhenFuncParam2")
        NewContParam3PhenList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "ContPhenFuncParam3")
        NewContParam4PhenList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "ContPhenFuncParam4")
        NewContParam5PhenList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "ContPhenFuncParam5")
        NewNoiseList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "NoiseLevel")
        NewTelescopeSizeList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "TelescopeSize")
        NewBMINList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "BMIN")
        NewBMAJList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "BMAJ")
        NewBPAList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "BPA")
        NewInter_FlagList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "Inter_Flag")
        NewGlobalvLSRList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "GlobalvLSR")
        NewRedshiftList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "Redshift")
        NewErrorYFlagList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "ErrorY")
        NewNumberHeaderLinesList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "NumberHeaderLines")
        NewSeparatorColumnsList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "SeparatorColumns")
        NewIsotopologuesList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "iso_flag")
        if (NewIsotopologuesList == []):
            NewIsotopologuesList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "Isotopologues")
        NewIsoTableFileNameList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "IsoTableFileName")
        LocaldbFile = self.dbList[0]
        NewEmAbsPATHList = task_MAGIX.GetXMLtagNEW(self.NewExpXMLFileName, "EmAbsPATH")
        if (NewEmAbsPATHList != []):
            print ("Import emission and absorption functions from files ..")
            LocalEmAbsPATH = NewEmAbsPATHList[0]
            EmsAbsFileInterDict = self.Tab2GetEmAbsFunc(LocalEmAbsPATH)
        else:
            LocalEmAbsPATH = None
            EmsAbsFileInterDict = None

        # Debug:
        # print ("\n\nNewFreqMinList = ", NewFreqMinList)
        # print ("NewFreqMaxList = ", NewFreqMaxList)
        # print ("NewNumberExpFiles = ", NewNumberExpFiles)
        # print ("EmsAbsFileInterDict = ", EmsAbsFileInterDict)


        ## update progress bar
        ProgressBarValue += 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return ok


        ## get iso flag
        LocalIsoFlag = str(NewIsotopologuesList[0]).lower()
        self.IsoFlag = task_myXCLASSMapFit.CheckBool(LocalIsoFlag)

        # Debug:
        # print ("\nLocaldbFile = ", LocaldbFile)
        # print ("self.IsoFlag = ", self.IsoFlag)
        # print ("self.CurrentMolecule = ", self.CurrentMolecule)
        # print ("self.MaxNumTransInFit = ", self.MaxNumTransInFit)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## construct global parameter self.AllSpectra
        self.AllSpectra = []
        dbListForSorting = []
        LowFreqList = []
        OrigLowFreqList = []
        UpFreqList = []
        OrigUpFreqList = []
        CounterTransWindow = 0
        AllSpectraIndex = (-1)
        DBParamList = []
        for LocalMol in ListOfMoleculesAndIso:
            DBParamList.append([])
        for ObsDataFileIndex in range(NewNumberExpFiles):


            ## get number of header lines and character separating columns etc. from obs. xml file
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                           NumberRangeListIn = NewNumberExpRangesList, \
                                                                           NumberHeaderLinesListIn = NewNumberHeaderLinesList, \
                                                                           SeparatorColumnsListIn = NewSeparatorColumnsList, \
                                                                           TelescopeSizeListIn = NewTelescopeSizeList, \
                                                                           BMIN_ListIn = NewBMINList, \
                                                                           BMAJ_ListIn = NewBMAJList, \
                                                                           BPA_ListIn = NewBPAList, \
                                                                           InterFlagListIn = NewInter_FlagList, \
                                                                           GlobalvLSRListIn = NewGlobalvLSRList, \
                                                                           Redshift_ListIn = NewRedshiftList)
            LocalExpFileName = NewExpFileList[ObsDataFileIndex][1]
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']        ## get number of frequency ranges for current obs. data file
            NumberHeaderLines = ObsXMLParameterDictFile['NumberHeaderLines']
            SeparatorColumns = ObsXMLParameterDictFile['SeparatorColumns']
            TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
            BMIN = ObsXMLParameterDictFile['BMIN']
            BMAJ = ObsXMLParameterDictFile['BMAJ']
            BPA = ObsXMLParameterDictFile['BPA']
            Inter_Flag = ObsXMLParameterDictFile['InterFlag']
            GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
            if (GlobalvLSR is None):
                GlobalvLSR = 0.0
            Redshift = ObsXMLParameterDictFile['Redshift']
            if (Redshift is None):
                Redshift = 0.0

            # Debug:
            # print ("LocalExpFileName = ", LocalExpFileName)
            # print ("NumberHeaderLines = ", NumberHeaderLines)
            # print ("SeparatorColumns = ", SeparatorColumns)
            # print ("TelescopeSize = ", TelescopeSize)
            # print ("BMIN = ", BMIN)
            # print ("BMAJ = ", BMAJ)
            # print ("BPA = ", BPA)
            # print ("Inter_Flag = ", Inter_Flag)
            # print ("GlobalvLSR = ", GlobalvLSR)
            # print ("Redshift = ", Redshift)


            ## import obs. data
            if (self.FITSFlag):
                NewObsDataFile = self.FITSSubDir + "FitsSpectrum__" + str(ObsDataFileIndex + 1) + ".dat"
                ImportExpData = np.loadtxt(NewObsDataFile, skiprows = 3)
            else:
                if (SeparatorColumns.strip() == ""):
                    ImportExpData = np.loadtxt(LocalExpFileName, skiprows = NumberHeaderLines)
                else:
                    ImportExpData = np.loadtxt(LocalExpFileName, skiprows = NumberHeaderLines, delimiter = SeparatorColumns)


            ## analyze frequency ranges
            for RangeIndex in range(NumberFrequencyRanges):                                 ## loop over all frequency ranges


                ## get parameter for current frequency range
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = NewFreqMinList, \
                                                                                FreqMaxListIn = NewFreqMaxList, FreqStepListIn = NewFreqStepList, \
                                                                                tBackFlagListIn = Newt_back_flagList, tBackListIn = NewtBackList, \
                                                                                tSlopeListIn = NewtSlopeList, N_HListIn = NewN_HList, \
                                                                                beta_dustListIn = Newbeta_dustList, \
                                                                                kappa_1300ListIn = Newkappa_1300List, \
                                                                                DustFileNameListIn = NewDustFileNameList, \
                                                                                BackgroundFileNameListIn = NewBackgroundFileNameList, \
                                                                                ContPhenFuncID_ListIn = NewContFuncIDPhenList, \
                                                                                ContPhenFuncParam1_ListIn = NewContParam1PhenList, \
                                                                                ContPhenFuncParam2_ListIn = NewContParam2PhenList, \
                                                                                ContPhenFuncParam3_ListIn = NewContParam3PhenList, \
                                                                                ContPhenFuncParam4_ListIn = NewContParam4PhenList, \
                                                                                ContPhenFuncParam5_ListIn = NewContParam5PhenList, \
                                                                                NoiseListIn = NewNoiseList)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None):
                    CounterTransWindow += 1
                    FreqMax = ObsXMLParameterDictRange['FreqMax']
                    FreqStep = ObsXMLParameterDictRange['FreqStep']
                    Noise = ObsXMLParameterDictRange['NoiseLevel']
                    if (Noise is None):
                        Noise = 0.0


                    ## get background continuum parameters
                    t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                    t_back_flag = task_myXCLASSMapFit.CheckBool(t_back_flag)
                    if (not t_back_flag):
                        self.RemoveContinuumFlag = False
                    tBack = ObsXMLParameterDictRange['tBack']
                    tSlope = ObsXMLParameterDictRange['tSlope']
                    BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                    BackgroundFile = self.Tab2ImportFile(BackgroundFileName)


                    ## get dust parameters
                    nH_flag = True
                    nH = ObsXMLParameterDictRange['N_H']
                    beta = ObsXMLParameterDictRange['beta_dust']
                    kappa = ObsXMLParameterDictRange['kappa_1300']
                    if (nH is None or beta is None or kappa is None):
                        nH_flag = False
                        nH = 0.0
                        beta = 0.0
                        kappa = 0.0
                    DustFileName = ObsXMLParameterDictRange['DustFileName']
                    DustFile = self.Tab2ImportFile(DustFileName)


                    ## get parameters for phenomenological description of continuum
                    LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                    LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                    LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                    LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                    LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                    LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']

                    # Debug:
                    # print ("\n\n>>FreqMin, FreqMax = ", FreqMin, FreqMax)


                    ## define part for range array
                    RangeParamPart = [nH, beta, kappa, \
                                      LocalContPhenFuncID, LocalContPhenFuncParam1, LocalContPhenFuncParam2, LocalContPhenFuncParam3, \
                                      LocalContPhenFuncParam4, LocalContPhenFuncParam5, DustFile, BackgroundFile, \
                                      ObsDataFileIndex, RangeIndex, LocalExpFileName, NumberHeaderLines, SeparatorColumns, FreqStep, \
                                      DustFileName, BackgroundFileName, EmsAbsFileInterDict]


                    ## determine doppler shifted min. and max. frequencies
                    NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
                    NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
                    ShiftedFreqMin = min(NewFreqMin, NewFreqMax)
                    ShiftedFreqMax = max(NewFreqMin, NewFreqMax)

                    # Debug:
                    # print ("\n\nObsDataFileIndex, RangeIndex = ", ObsDataFileIndex, RangeIndex)
                    # print ("FreqMin, FreqMax = ", FreqMin, FreqMax)
                    # print ("->ShiftedFreqMin, ShiftedFreqMax = ", ShiftedFreqMin, ShiftedFreqMax)


                    ## determine min and max. freq. index of obs. data and extract obs. data
                    f1 = max(0, (np.abs(ImportExpData[:, 0] - FreqMin)).argmin() - 1)
                    f2 = min(len(ImportExpData[:, 0]) - 1, (np.abs(ImportExpData[:, 0] - FreqMax)).argmin() + 1)
                    MinFreqIndexExp = min(f1, f2)
                    MaxFreqIndexExp = max(f1, f2)
                    LocalObsData = ImportExpData[MinFreqIndexExp:MaxFreqIndexExp, :]


                    ## define new array
                    NewASCIIdata = np.zeros((len(LocalObsData), 4), dtype=np.float32)
                    NewASCIIdata[:, 0] = LocalObsData[:, 0]                                 ## frequencies
                    NewASCIIdata[:, 1] = 0.0                                                ## velocities (will be added later)
                    NewASCIIdata[:, 2] = LocalObsData[:, 1]                                 ## intensities
                    NewASCIIdata[:, 3] = 0.0                                                ## weights


                    ## remove continuum
                    if (abs(tBack) >= 0.0):
                        IntWithoutContinuum = np.zeros((len(LocalObsData)), dtype=np.float32)
                        for mm in range(len(LocalObsData[:, 0])):
                            val = NewASCIIdata[mm, 2] - abs(tBack) * (LocalObsData[mm, 0] / FreqMin)**tSlope
                            IntWithoutContinuum[mm] = val
                            if (self.RemoveContinuumFlag):
                                NewASCIIdata[mm, 2] = val
                    else:
                        IntWithoutContinuum = self.EstimateContinuum(LocalObsData)

                    # Debug:
                    # print ("\n\nFreqMin = ", FreqMin)
                    # print ("FreqMax = ", FreqMax)
                    # print ("GlobalTransFreqNPList = ", GlobalTransFreqNPList)


                    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                    ## get a local list of restricted molecular parameters
                    ## MolecularParameterList = [TransFreq, ELow, (EinsteinA * gup)^(-1), (EinsteinA * gup) / (ELow**2), EinsteinA, gup, MolNameIndex]
                    LocalRestrictedMolecularParameterList = RestrictedMolecularParameterList[(ShiftedFreqMin <= RestrictedMolecularParameterList[:, 0]) \
                                                                                          & (RestrictedMolecularParameterList[:, 0] <= ShiftedFreqMax), :]
                    for LocalTransition in LocalRestrictedMolecularParameterList:


                        ## define list DBParam = [TransFreq, EinsteinA, ELow, gup, MolName]
                        TransFreq = LocalTransition[0]                                      ## get transition frequency
                        ELow = LocalTransition[1]                                           ## get E_low
                        gAm1 = LocalTransition[2]                                           ## get (gA)
                        E2gAEm1 = LocalTransition[3]                                        ## get E^2/gA
                        EinsteinA = LocalTransition[4]                                      ## get Einstein A
                        gup = LocalTransition[5]                                            ## get g_up
                        MolNameIndex = int(LocalTransition[6])                              ## get index of molecule name
                        LocalMoleculeName = self.PureMolNameVector[MolNameIndex]
                        DBParam = [TransFreq, EinsteinA, ELow, gup, LocalMoleculeName]

                        # Debug:
                        # print ("DBParam = ", DBParam)


                        ## make a local copy of obs. data
                        LocalNewASCIIdata = np.zeros((len(LocalObsData), 4), dtype=np.float32)
                        LocalNewASCIIdata[:, 2] = NewASCIIdata[:, 2]


                        ## shift frequency axis
                        if (GlobalvLSR != 0.0):
                            for mm in range(len(NewASCIIdata[:, 0])):
                                LocalNewASCIIdata[mm, 0] = task_myXCLASS.ConvertFreq(NewASCIIdata[mm, 0], GlobalvLSR, z = Redshift, backTrafo = True)
                        else:
                            LocalNewASCIIdata[:, 0] = NewASCIIdata[:, 0]

                        # Debug:
                        # print ("LocalNewASCIIdata[:, 0] = ", NewASCIIdata[:, 0])


                        ## add velocity axis
                        vLSR = 0.0                                                      ## current trans. freq. is at v = 0
                        RestFreq = TransFreq                                            ## define rest frequency
                        VelocityData = task_myXCLASS.ChangeToVelocitiy(LocalNewASCIIdata[:, 0], RestFreq, vLSR)
                        LocalNewASCIIdata[:, 1] = np.asarray(VelocityData[:])

                        # Debug:
                        # print ("\nTransFreq = ", TransFreq)
                        # print ("VelocityData = ", VelocityData)
                        # print ("DBParam = ", DBParam)


                        ## extract velocity range around transition frequency
                        v1 = task_myXCLASS.ConvertFreq(TransFreq, self.velLowLimitWeight)
                        v2 = task_myXCLASS.ConvertFreq(TransFreq, self.velUpLimitWeight)
                        v1FreqIndex = max(0, (np.abs(LocalNewASCIIdata[:, 0] - v1)).argmin() - 1)
                        v2FreqIndex = min(len(LocalNewASCIIdata[:, 0]) - 1, (np.abs(LocalNewASCIIdata[:, 0] - v2)).argmin() + 1)
                        In1 = int(min(v1FreqIndex, v2FreqIndex))
                        In2 = int(max(v1FreqIndex, v2FreqIndex))
                        LocalObsDataTransWindow = LocalNewASCIIdata[In1:In2, :]


                        ## save range parameters
                        LowFreq = np.min(LocalObsDataTransWindow[:, 0])
                        UpFreq = np.max(LocalObsDataTransWindow[:, 0])
                        NewTBack = tBack * (LowFreq / FreqMin)**tSlope
                        LowFreqList.append(LowFreq)
                        OrigLowFreqList.append([1, 0, LowFreq])
                        UpFreqList.append(UpFreq)
                        OrigUpFreqList.append([1, 0, UpFreq])


                        ## define range parameter array
                        # TelescopeSize = RangeParam[0]
                        # BMIN = RangeParam[1]
                        # BMAJ = RangeParam[2]
                        # BPA = RangeParam[3]
                        # Inter_Flag = RangeParam[4]
                        # GlobalvLSR = RangeParam[5]
                        # Redshift = RangeParam[6]
                        # NewTBack = RangeParam[7]
                        # tSlope = RangeParam[8]
                        # LowFreq = RangeParam[9]
                        # UpFreq = RangeParam[10]
                        # nH = RangeParam[11]
                        # beta = RangeParam[12]
                        # kappa = RangeParam[13]
                        # LocalContPhenFuncID = RangeParam[14]
                        # LocalContPhenFuncParam1 = RangeParam[15]
                        # LocalContPhenFuncParam2 = RangeParam[16]
                        # LocalContPhenFuncParam3 = RangeParam[17]
                        # LocalContPhenFuncParam4 = RangeParam[18]
                        # LocalContPhenFuncParam5 = RangeParam[19]
                        # DustFile = RangeParam[20]
                        # BackgroundFile = RangeParam[21]
                        # ObsDataFileIndex = RangeParam[22]
                        # RangeIndex = RangeParam[23]
                        # LocalExpFileName = RangeParam[24]
                        # NumberHeaderLines = RangeParam[25]
                        # SeparatorColumns = RangeParam[26]
                        # FreqStep = RangeParam[27]
                        # DustFileName = RangeParam[28]
                        # BackgroundFileName = RangeParam[29]
                        # EmsAbsFileInterDict = RangeParam[30]
                        RangeParam = [TelescopeSize, BMIN, BMAJ, BPA, Inter_Flag, GlobalvLSR, Redshift, NewTBack, tSlope, LowFreq, UpFreq] \
                                     + RangeParamPart


                        ## smooth data
                        SmoothedData = []
                        if (self.SmoothValue > 0.0):
                            SmoothedData = LocalObsDataTransWindow[:, 2].copy()
                            y = task_LineIdentification.SmoothObsData(SmoothedData, self.SmoothValue, self.SmoothMethod)
                            xy = int(min(len(y), len(SmoothedData)))
                            SmoothedData[:xy] = y[:xy]

                            # Debug:
                            # print ("self.SmoothValue = ", self.SmoothValue)


                        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        ## get molecular parameters for other transitions and isotopologues within current TWF
                        LocalMolecularParameterList = RelevantMolecularParameterList[(ShiftedFreqMin <= MolecularParameterList[:, 0]) \
                                                                                   & (MolecularParameterList[:, 0] <= ShiftedFreqMax), :]
                        NumMolecules = np.max(LocalMolecularParameterList[:, -1])           ## get total number of molecules + isotopologues

                        # Debug:
                        # print ("\n\nLocalMolecularParameterList = ", LocalMolecularParameterList)
                        # print ("NumMolecules = ", NumMolecules)


                        ##********************************************************************************************************************************
                        ##********************************************************************************************************************************
                        ## store all TFW parameters
                        self.AllSpectra.append([LocalObsDataTransWindow, SmoothedData, DBParam, RangeParam, IntWithoutContinuum[In1:In2], RestFreq, \
                                                NumMolecules, LocalMolecularParameterList])
                        ##********************************************************************************************************************************
                        ##********************************************************************************************************************************

                        # Debug:
                        # print ("\n\nself.AllSpectra[-1] = ", self.AllSpectra[-1])


                        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        ## define parameters for plot sorting
                        AllSpectraIndex += 1
                        dbListForSorting.append([AllSpectraIndex, TransFreq, ELow, gAm1, E2gAEm1, gup, EinsteinA, LocalMoleculeName])


            ## update progress bar
            ProgressBarValue += 1
            ok = bar.setValue(ProgressBarValue)
            if (not ok):
                print ("\\n\\nAbort")
                self.lock = 0
                return ok

        # Debug:
        # print ("len(self.AllSpectra) = ", len(self.AllSpectra))
        # print ("self.AllSpectra[0] = ", self.AllSpectra[0])
        # print ("self.AllSpectra[1] = ", self.AllSpectra[1])
        # print ("self.AllSpectra = ", self.AllSpectra)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## sort list of molecular parameters depending on user-defined sort criterium
        for LocalSortIndex in range(4):
            LocalCopydbListForSorting = copy.deepcopy(dbListForSorting)
            self.IndexSortConversionList[LocalSortIndex] = copy.deepcopy(sorted(LocalCopydbListForSorting, key = lambda x:(x[LocalSortIndex + 1])))

        # Debug:
        # print ("\n\n\ndbListForSorting = ", dbListForSorting)
        # print ("self.IndexSortConversionList[0] = ", self.IndexSortConversionList[0])
        # print ("self.IndexSortConversionList[1] = ", self.IndexSortConversionList[1])
        # print ("self.IndexSortConversionList[2] = ", self.IndexSortConversionList[2])
        # print ("self.IndexSortConversionList[3] = ", self.IndexSortConversionList[3])


        ## update progress bar
        ProgressBarValue += 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return ok


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if iso-ratio file is defined
        LocaldbFile = self.dbList[0]
        self.CurrentIsoParam = []
        if (self.IsoFlag):


            ## import parameters from iso ratio file for all isotopologues which have at least on transition in one of the TFW
            IsoTableFileName = self.IsoTableFileNameList[0]
            SelectIsoRatioTable = task_myXCLASSFit.CheckMolecules(self.LocalMolfitFile, IsoTableFileName, OrigLowFreqList, OrigUpFreqList, \
                                                                  [LocaldbFile])
            self.CurrentIsoParam = self.Tab2GetIsoRatiosFromFile(self.CurrentMolecule, SelectIsoRatioTable)
            if (self.CurrentMoleculeID < len(self.AllIsotopologueList)):
                self.AllIsotopologueList[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentIsoParam)
            else:
                self.AllIsotopologueList.append(copy.deepcopy(self.CurrentIsoParam))
            if (self.CurrentIsoParam != []):                                                ## check if at least one iso. is defined for current mol.
                self.UseIsoFlag = True
                if (self.CurrentIsoID > (len(self.CurrentIsoParam) - 1)):
                    self.CurrentIsoID = (len(self.CurrentIsoParam) - 1)
            else:
                self.UseIsoFlag = False

            # Debug:
            # print ("\n\nself.UseIsoFlag = ", self.UseIsoFlag)
            # print ("self.IsoFlag = ", self.IsoFlag)
            # print ("self.CurrentIsoParam = ", self.CurrentIsoParam)
            # print ("self.CurrentIsoID = ", self.CurrentIsoID)
            # print ("self.CurrentMoleculeID = ", self.CurrentMoleculeID)
            # print ("self.AllIsotopologueList = ", self.AllIsotopologueList)
            # print ("SelectIsoRatioTable = ", SelectIsoRatioTable)
            # print ("self.LocalMolfitFile = ", self.LocalMolfitFile)
            # print ("IsoTableFileName = ", IsoTableFileName)
            # print ("OrigLowFreqList = ", OrigLowFreqList)
            # print ("OrigUpFreqList = ", OrigUpFreqList)
            # print ("LocaldbFile = ", LocaldbFile)

        else:
            IsoTableFileName = ""

        # Debug:
        # print ("\nlen(ListLocalIsotopologue) = ", len(ListLocalIsotopologue))
        # print ("ListLocalIsotopologue = ", ListLocalIsotopologue)
        # print ("self.AllIsotopologueList = ", self.AllIsotopologueList)


        ## update progress bar
        ProgressBarValue += 1
        ok = bar.setValue(ProgressBarValue)
        if (not ok):
            print ("\\n\\nAbort")
            self.lock = 0
            return ok


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine molfit parameter
        self.CurrentMolfitParameter = self.MolfitFileAllParameters[self.CurrentMoleculeID]
        self.ActiveCompList = []
        for i in range(len(self.CurrentMolfitParameter)):
            self.ActiveCompList.append(True)
        NewRangeFlag = False
        if (self.CurrentMolfitParameter == []):
            self.CurrentMolfitParameter.append(copy.deepcopy(self.DefaultMolfitParameter))
        if (self.UndoFlag):
            self.OldMolfitParameter.append(copy.deepcopy(self.CurrentMolfitParameter))
            self.OldIsoRatioParameter.append(copy.deepcopy(self.CurrentIsoParam))
            self.OldLocalMolfitFile.append(copy.deepcopy(self.LocalMolfitFile))
            self.OldMoleculesInMolfitFile.append(copy.deepcopy(self.MoleculesInMolfitFile))
            self.OldAllMoleculesMolfitFiles.append(copy.deepcopy(self.AllMoleculesMolfitFiles))
            self.OldMolfitFileAllParameters.append(copy.deepcopy(self.MolfitFileAllParameters))
            self.OldCurrentMolecule.append(copy.deepcopy(self.CurrentMolecule))
            self.OldCurrentMoleculeID.append(copy.deepcopy(self.CurrentMoleculeID))
            self.OldCurrentMoleculeDir.append(copy.deepcopy(self.CurrentMoleculeDir))
            self.OldAllIsotopologueList.append(copy.deepcopy(self.AllIsotopologueList))
            self.OldActiveCompList.append(copy.deepcopy(self.ActiveCompList))


        ## close progress bar
        bar.close()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update all parameters
        self.Tab2SetValuesForTab()


        ## return to GUI
        self.lock = 0
        return ok
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set all values in tab 2
    def Tab2SetValuesForTab(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## activate lock
        if (self.lock == 1):
            AlreadyLockedFlag = True
        else:
            AlreadyLockedFlag = False
            self.lock = 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize tab2
        if (self.InitTab2Flag):
            self.InitTab2Flag = False


            ## create molfit file from parameter list
            self.Tab2CreateMolfitFile()


            MaxNumTransitionsSQL = self.SQLParamArray[1]
            self.TotalNumberSpectra = min(MaxNumTransitionsSQL, len(self.AllSpectra))
            self.MaxNumberSpectrum = min(self.MaxNumberSpectrumDefault, self.TotalNumberSpectra)
            self.SpectrumIndexList = list(range(0, self.MaxNumberSpectrum))
            self.ChangeSpectrumWindowFlag = True


            ## set combo box molecule
            self.comboBox_molecule.clear()
            self.comboBox_molecule.addItems(self.MoleculesInMolfitFile)
            self.comboBox_molecule.setCurrentIndex(self.CurrentMoleculeID)


            ## set title of GUI window
            self.GUITitle = "XCLASS-GUI"
            if (self.tabWidget.currentIndex() == 1):
                self.GUITitle += "   ({:d}/{:d})".format(self.CurrentMoleculeID + 1, len(self.MoleculesInMolfitFile))
            self.setWindowTitle(self.GUITitle)


            ## enable elements which are connected with iso ratio file
            if (self.UseIsoFlag and self.CurrentIsoParam != []):
                self.Edit_Low_IsoRatio.setEnabled(True)
                self.Edit_Up_IsoRatio.setEnabled(True)
                self.Edit_Val_IsoRatio.setEnabled(True)
                self.comboBox_iso.setEnabled(True)
                self.checkBox_FF_iso.setEnabled(True)
                self.Slider__IsoRatio.setEnabled(True)
                self.checkBox_useIso.setEnabled(True)
                self.Tab2SetCurrIsoIndex()
            else:
                self.comboBox_iso.clear()
                self.Edit_Low_IsoRatio.setEnabled(False)
                self.Edit_Up_IsoRatio.setEnabled(False)
                self.Edit_Val_IsoRatio.setEnabled(False)
                self.comboBox_iso.setEnabled(False)
                self.checkBox_FF_iso.setEnabled(False)
                self.Slider__IsoRatio.setEnabled(False)
                self.checkBox_useIso.setEnabled(False)


            ## set all values
            self.CurrCursorPos = 0
            molfitParameters = []


        ## define list of parameters
        ListParam = ["ss", "Temp", "Ntot", "vwidth", "voff"]

        # Debug:
        # print ("self.CurrentMolfitParameter = ", self.CurrentMolfitParameter)
        # print ("self.CurrentComponentIndex = ", self.CurrentComponentIndex)


        ## get CF-Flag
        try:
            CFFlag = self.CurrentMolfitParameter[self.CurrentComponentIndex][5][0]
        except:
            print ("\n\n\nself.CurrentComponentIndex = ", self.CurrentComponentIndex)
            print ("self.CurrentMolfitParameter = ", self.CurrentMolfitParameter)
            self.CurrentComponentIndex = len(self.CurrentMolfitParameter[:]) - 1
            CFFlag = self.CurrentMolfitParameter[self.CurrentComponentIndex][5][0]
            # sys.exit(0)
        if (self.IsNumber(CFFlag)):
            CFStringFlag = False
            CFFlag = float(CFFlag)
            self.CFUsageFlag = False
        else:
            CFFlag = CFFlag.strip()
            CFStringFlag = True


        ## edit boxes
        for CounterParameter, Param in enumerate(ListParam):
            if (CounterParameter < 5):


                ## get lower, upper and parameter values
                LocalFitFlag = self.CurrentMolfitParameter[self.CurrentComponentIndex][CounterParameter][0]
                LocalLowerLimit = float(self.CurrentMolfitParameter[self.CurrentComponentIndex][CounterParameter][1])
                LocalUpperLimit = float(self.CurrentMolfitParameter[self.CurrentComponentIndex][CounterParameter][2])
                LocalParamVal = float(self.CurrentMolfitParameter[self.CurrentComponentIndex][CounterParameter][3])


                ## set fit flag
                if (CFStringFlag and (CFFlag in ["f", "a"]) and Param == "ss"):
                    exec("self.checkBox_FF_" + Param + ".setEnabled(False)")
                else:
                    exec("self.checkBox_FF_" + Param + ".setEnabled(True)")
                    if (LocalFitFlag == "y"):
                        exec("self.checkBox_FF_" + Param + ".setChecked(True)")
                    else:
                        exec("self.checkBox_FF_" + Param + ".setChecked(False)")


                ## add values to edit boxes
                if (CFStringFlag and (CFFlag in ["f", "a"]) and Param == "ss"):
                    exec("self.Edit_Low_" + Param + ".setText(" + chr(34) + " " + chr(34) + ")")
                    exec("self.Edit_Up_" + Param + ".setText(" + chr(34) + " " + chr(34) + ")")
                    exec("self.Edit_Val_" + Param + ".setText(" + chr(34) + " " + chr(34) + ")")
                    exec("self.Edit_Low_" + Param + ".setEnabled(False)")
                    exec("self.Edit_Up_" + Param + ".setEnabled(False)")
                    exec("self.Edit_Val_" + Param + ".setEnabled(False)")
                else:
                    exec("self.Edit_Low_" + Param + ".setEnabled(True)")
                    exec("self.Edit_Up_" + Param + ".setEnabled(True)")
                    exec("self.Edit_Val_" + Param + ".setEnabled(True)")
                    if (Param == "Ntot"):
                        LocalLowerLimitString = "%.2e" % LocalLowerLimit
                        LocalUpperLimitString = "%.2e" % LocalUpperLimit
                        LocalParamValString = "%.3e" % LocalParamVal
                    else:
                        LocalLowerLimitString = "%.2f" % LocalLowerLimit
                        LocalUpperLimitString = "%.2f" % LocalUpperLimit
                        LocalParamValString = "%.3f" % LocalParamVal
                    exec("self.Edit_Low_" + Param + ".setText(" + chr(34) + LocalLowerLimitString + chr(34) + ")")
                    exec("self.Edit_Up_" + Param + ".setText(" + chr(34) + LocalUpperLimitString + chr(34) + ")")
                    exec("self.Edit_Val_" + Param + ".setText(" + chr(34) + LocalParamValString + chr(34) + ")")


                ## set slider range etc.
                if (CFStringFlag and (CFFlag in ["f", "a"]) and Param == "ss"):
                    exec("self.Slider__" + Param + ".setEnabled(False)")
                else:
                    exec("self.Slider__" + Param + ".setEnabled(True)")
                    LowLimit = 0
                    UpLimit = 100
                    LocalInterval = 1
                    exec("self.Slider__" + Param + ".setMinimum(" + str(LowLimit) + ")")
                    exec("self.Slider__" + Param + ".setMaximum(" + str(UpLimit) + ")")
                    exec("self.Slider__" + Param + ".setTickInterval(" + str(LocalInterval) + ")")
                    if (LocalUpperLimit > LocalLowerLimit and LocalLowerLimit <= LocalParamVal and LocalParamVal <= LocalUpperLimit):
                        OldMin = LocalLowerLimit
                        OldMax = LocalUpperLimit
                        OldValue = LocalParamVal
                        NewMin = LowLimit
                        NewMax = UpLimit
                        OldRange = (OldMax - OldMin)
                        NewRange = (NewMax - NewMin)
                        val = (((OldValue - OldMin) * NewRange) / OldRange) + NewMin
                    else:
                        val = LowLimit
                    exec("self.Slider__" + Param + ".setValue(" + str(int(val)) + ")")


        ## update current line of the molfit file
        self.Tab2CreateMolfitFile()


        ## write contents of molfit file to list box
        self.list_molfit.clear()
        for lineID, line in enumerate(self.CurrentMolfitFileContents):
            if (lineID == 0 or self.ActiveCompList[lineID - 1]):
                self.list_molfit.addItem(line)
            else:
                self.list_molfit.addItem("D" + line)
        self.CurrentMolfitIndex = max(1, self.CurrentMolfitIndex)
        self.list_molfit.setCurrentRow(self.CurrentMolfitIndex)


        ## set radio button for current component
        if (CFStringFlag):
            Param = self.CurrentMolfitParameter[self.CurrentComponentIndex][-1][0].strip()
            if (Param in ["c", "e"]):
                self.radioButton_Core.setChecked(True)
                self.radioButton_Fore.setChecked(False)
            else:
                self.radioButton_Core.setChecked(False)
                self.radioButton_Fore.setChecked(True)
            self.label_Dist.setVisible(False)
            self.Edit_Val_Distance.setVisible(False)
        else:
            self.radioButton_Core.setVisible(False)
            self.radioButton_Fore.setVisible(False)
            self.label_Dist.setVisible(True)
            self.Edit_Val_Distance.setVisible(True)
            self.Edit_Val_Distance.setText(str(CFFlag))


        ## set active check box
        self.checkBox_active.setChecked(self.ActiveCompList[self.CurrentComponentIndex])
        self.checkBox_active.setText("active comp. " + str(self.CurrentComponentIndex + 1))


        ## set background check box
        if (len(self.BackgroundFileNameList) > 0):
            self.checkBox_background.setChecked(True)
        else:
            self.checkBox_background.setChecked(False)
        self.checkBox_background.setText("Background")


        ## set "remove continuum" check box
        self.checkBox_rm_cont.setChecked(self.RemoveContinuumFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## settings for iso ratio file


        ## set use iso ratio flag
        self.checkBox_useIso.setChecked(self.UseIsoFlag)

        # Debug:
        # print ("self.CurrentIsoID = ", self.CurrentIsoID)
        # print ("self.UseIsoFlag = ", self.UseIsoFlag)
        # print ("self.CurrentIsoParam = ", self.CurrentIsoParam)


        ## set values for edit boxes, fit flag check box, slider, combo box
        if (self.UseIsoFlag and self.CurrentIsoParam != []):
            self.Edit_Low_IsoRatio.setEnabled(True)
            self.Edit_Up_IsoRatio.setEnabled(True)
            self.Edit_Val_IsoRatio.setEnabled(True)
            self.comboBox_iso.setEnabled(True)
            self.checkBox_FF_iso.setEnabled(True)
            flag = self.CurrentIsoParam[self.CurrentIsoID][5]
            self.checkBox_FF_iso.setChecked(flag)
            self.Slider__IsoRatio.setEnabled(True)
            self.Tab2SetCurrIsoIndex()


            ## ratio lower and upper limit
            LocalLowerLimit = self.CurrentIsoParam[self.CurrentIsoID][3]
            LocalUpperLimit = self.CurrentIsoParam[self.CurrentIsoID][4]
            LocalParamVal = self.CurrentIsoParam[self.CurrentIsoID][2]


            ## slider
            LowLimit = 0
            UpLimit = 100
            LocalInterval = 1
            exec("self.Slider__IsoRatio.setMinimum(" + str(LowLimit) + ")")
            exec("self.Slider__IsoRatio.setMaximum(" + str(UpLimit) + ")")
            exec("self.Slider__IsoRatio.setTickInterval(" + str(LocalInterval) + ")")
            if (LocalUpperLimit > LocalLowerLimit and LocalLowerLimit <= LocalParamVal and LocalParamVal <= LocalUpperLimit):
                OldMin = LocalLowerLimit
                OldMax = LocalUpperLimit
                OldValue = LocalParamVal
                NewMin = LowLimit
                NewMax = UpLimit
                OldRange = (OldMax - OldMin)
                NewRange = (NewMax - NewMin)
                val = (((OldValue - OldMin) * NewRange) / OldRange) + NewMin
            else:
                val = LowLimit
            exec("self.Slider__IsoRatio.setValue(" + str(val) + ")")


            ## edit boxes
            LocalLowerLimitString = "%.2e" % LocalLowerLimit
            LocalUpperLimitString = "%.2e" % LocalUpperLimit
            LocalParamValString = "%.3e" % LocalParamVal
            self.Edit_Low_IsoRatio.setText(LocalLowerLimitString)
            self.Edit_Up_IsoRatio.setText(LocalUpperLimitString)
            self.Edit_Val_IsoRatio.setText(LocalParamValString)
        else:
            LocalLowerLimitString = " "
            LocalUpperLimitString = " "
            LocalParamValString = " "
            self.Edit_Low_IsoRatio.setText(LocalLowerLimitString)
            self.Edit_Up_IsoRatio.setText(LocalUpperLimitString)
            self.Edit_Val_IsoRatio.setText(LocalParamValString)
            self.Edit_Low_IsoRatio.setEnabled(False)
            self.Edit_Up_IsoRatio.setEnabled(False)
            self.Edit_Val_IsoRatio.setEnabled(False)
            self.comboBox_iso.clear()
            self.comboBox_iso.setEnabled(False)
            self.checkBox_FF_iso.setEnabled(False)
            self.Slider__IsoRatio.setEnabled(False)

        # Debug:
        # print ("\n\nself.OldParamVectorCounter = ", self.OldParamVectorCounter)
        # print ("(len(self.OldMolfitParameter) - 1) = ", (len(self.OldMolfitParameter) - 1))


        ## (de-)activate undo and redo button
        if (self.OldParamVectorCounter == 0 or (self.OldMolfitParameter == [] and self.OldIsoRatioParameter == [])):
            self.pushButton_Undo.setEnabled(False)
        else:
            self.pushButton_Undo.setEnabled(True)
        if (self.OldParamVectorCounter < (len(self.OldCurrentMoleculeID) - 1) and (self.OldMolfitParameter != [] or self.OldIsoRatioParameter != [])):
            self.pushButton_Redo.setEnabled(True)
        else:
            self.pushButton_Redo.setEnabled(False)


        ## (de-)activate the "Prev. Mol." button
        if (self.CurrentMoleculeID <= 0):
            self.pushButton_PrevMol.setEnabled(False)
            self.actionGo_to_previous_molecule.setEnabled(False)
        else:
            self.pushButton_PrevMol.setEnabled(True)
            self.actionGo_to_previous_molecule.setEnabled(True)


        ## (de-)activate the "Next Mol." button
        if (self.CurrentMoleculeID >= (len(self.MoleculesInMolfitFile) - 1)):
            self.pushButton_NextMol.setEnabled(False)
            self.actionGo_to_next_molecule.setEnabled(False)
        else:
            self.pushButton_NextMol.setEnabled(True)
            self.actionGo_to_next_molecule.setEnabled(True)


        ## update plots
        self.Tab2CreatePlots()


        ## (de)activate remove comp. button
        if (len(self.CurrentMolfitParameter) > 1):
            self.pushButton_rmComp.setEnabled(True)
        else:
            self.pushButton_rmComp.setEnabled(False)


        ## return to GUI
        if (not AlreadyLockedFlag):
            self.lock = 0
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get molfit parameter from file
    def Tab2GetParameterFromMolfitFile(self, LocalMolfitParameters, NewRangeFlag):
        """

    input parameters:
    -----------------

        - LocalMolfitParameters:    new molfit parameters

        - NewRangeFlag:         flag indicating new parameter ranges


    output parameters:
    ------------------

        - MolfitParameter:      molfit parameter from file
        """

        # Debug:
        # print ("\nLocalMolfitParameters = ", LocalMolfitParameters)
        # print ("NewRangeFlag = ", NewRangeFlag)


        ## generate molfit file for current parameter
        MolfitParameter = []
        CounterComp = (-1)
        for LocalLine in LocalMolfitParameters:
            CounterComp += 1
            MolfitParameterLine = []
            LocalFitsFlag = "n"
            for col in LocalLine:
                Name = col[0]
                try:
                    flag = int(col[1])
                except:
                    flag = 0
                lowlimit = min(col[2], col[3])
                uplimit = max(col[2], col[3])
                element = col[4]
                AddFlag = False

                # Debug:
                # print ("col = ", col)


                ## get fits flag
                if (Name.endswith("_flag")):
                    if (flag == 0):
                        LocalFitsFlag = "n"
                    else:
                        LocalFitsFlag = "y"


                ## store values for source size
                elif (Name == "source_size"):
                    value = float(element)
                    if (NewRangeFlag):
                        lowlimit = min(value * 0.9, self.NewParameterRanges[0][0])
                        uplimit = max(value * 1.1, self.NewParameterRanges[0][1])
                    if (flag == 0):
                        FitFlag = "n"
                    else:
                        FitFlag = "y"
                    AddFlag = True


                ## store values for T_rot
                elif (Name in ["T_rot", "T_e", "T_dOff", "T_cont_dust", "T_Back"]):
                    value = float(element)
                    if (NewRangeFlag):
                        lowlimit = min(value * 0.9, self.NewParameterRanges[1][0])
                        uplimit = max(value * 1.1, self.NewParameterRanges[1][1])
                    if (flag == 0):
                        FitFlag = "n"
                    else:
                        FitFlag = "y"
                    AddFlag = True


                ## store values for N_tot
                elif (Name in ["N_tot", "nHcolumn_cont_dust", "nHcolumn", "EM_RRL"]):
                    value = float(element)
                    if (NewRangeFlag):
                        lowlimit = min(value * 0.9, self.NewParameterRanges[2][0])
                        uplimit = max(value * 1.1, self.NewParameterRanges[2][1])
                    if (flag == 0):
                        FitFlag = "n"
                    else:
                        FitFlag = "y"
                    AddFlag = True


                ## store values for V_width andV_width_Gauss
                elif (Name in ["V_width", "V_width_Gauss"]):
                    value = float(element)
                    if (NewRangeFlag):
                        lowlimit = min(value * 0.9, self.NewParameterRanges[3][0])
                        uplimit = max(value * 1.1, self.NewParameterRanges[3][1])
                    if (flag == 0):
                        FitFlag = "n"
                    else:
                        FitFlag = "y"
                    AddFlag = True


                ## store values for V_off
                elif (Name == "V_off"):
                    value = float(element)
                    if (NewRangeFlag):
                        lowlimit = min(value * 0.9, self.NewParameterRanges[4][0])
                        uplimit = max(value * 1.1, self.NewParameterRanges[4][1])
                    if (flag == 0):
                        FitFlag = "n"
                    else:
                        FitFlag = "y"
                    AddFlag = True


                ## store values for continuum descriptions
                elif (Name in ["kappa_cont_dust", "beta_cont_dust", "ContFuncID_phen", \
                      "ContFuncID_param_1", "ContFuncID_param_2", "ContFuncID_param_3", "ContFuncID_param_4", "ContFuncID_param_5"]):
                    value = float(element)
                    if (NewRangeFlag):
                        lowlimit = min(value * 0.9, self.NewParameterRanges[4][0])
                        uplimit = max(value * 1.1, self.NewParameterRanges[4][1])
                    if (flag == 0):
                        FitFlag = "n"
                    else:
                        FitFlag = "y"
                    AddFlag = True


                ## store values for EA-Flag
                elif (Name == "CFFlag"):
                    AddFlag = False
                    MolfitParameterLine.append([element])
                    if (element.strip() in ["f", "a"]):
                        if (len(MolfitParameterLine) < 6):
                            MolfitParameterLine[:0] = [["n", 0.0, 0.0, 0.0]]


                ## store values for ordering parameter
                elif (Name == "LayerDistance"):
                    AddFlag = False
                    value = float(element)
                    MolfitParameterLine.append([value])
                    if (element.strip() in ["f", "a"]):
                        if (len(MolfitParameterLine) < 6):
                            MolfitParameterLine[:0] = [["n", 0.0, 0.0, 0.0]]


                ## create new line
                if (AddFlag):
                    MolfitParameterLine.append([LocalFitsFlag, lowlimit, uplimit, value])


            ## write new line to file
            MolfitParameter.append(MolfitParameterLine)

        # Debug:
        # print ("MolfitParameter = ", MolfitParameter)


        ## return to GUI
        return MolfitParameter
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create new line for molfit file from parameter list
    def Tab2CreateLineForMolfitFile(self, CompIndex):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - NewLine:              new line for molfit file
        """

        # Debug:
        # print ("\nCompIndex = " , CompIndex)
        # print ("self.CurrentMolfitParameter[CompIndex] = ", self.CurrentMolfitParameter[CompIndex])


        ## create new line
        CFFlag = self.CurrentMolfitParameter[CompIndex][5][0]
        if (self.IsNumber(CFFlag)):
            CFStringFlag = False
            CFFlag = float(CFFlag)
        else:
            CFFlag = CFFlag.strip()
            CFStringFlag = True
        NewLine = ""
        ListParam = ["ss", "Temp", "Ntot", "vwidth", "voff"]
        for CounterParameter, Param in enumerate(ListParam):
            if (CounterParameter < 5):


                ## get lower, upper and parameter values
                LocalFitFlag = self.CurrentMolfitParameter[CompIndex][CounterParameter][0]
                LocalLowerLimit = self.CurrentMolfitParameter[CompIndex][CounterParameter][1]
                LocalUpperLimit = self.CurrentMolfitParameter[CompIndex][CounterParameter][2]
                LocalParamVal = self.CurrentMolfitParameter[CompIndex][CounterParameter][3]


                ## define new line for molfit file
                if (Param == "Ntot"):
                    NewLine += "    {:1s}{:10.2e}{:10.2e} {:10.3e}".format(LocalFitFlag, float(LocalLowerLimit), float(LocalUpperLimit), \
                                                                           float(LocalParamVal))
                elif ((CFFlag in ["f", "a"]) and Param == "ss"):
                    helpstring = " " * 36
                    NewLine = helpstring + NewLine
                else:
                    NewLine += "    {:1s}{:10.2f}{:10.2f} {:10.3f}".format(LocalFitFlag, float(LocalLowerLimit), float(LocalUpperLimit),  \
                                                                           float(LocalParamVal))
        if (CFStringFlag):
            NewLine += "   " + CFFlag
        else:
            NewLine += "   " + str(CFFlag)


        ## return to GUI
        return NewLine
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create molfit file from parameter list
    def Tab2CreateMolfitFile(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## create molfit file
        if (self.CurrentComponentIndex == 0 or self.CompNumChangeFlag):
            self.CurrentMolfitFileContents = []
            NumComp = len(self.CurrentMolfitParameter)
            NewLine = self.CurrentMolecule + "   "  + str(NumComp)
            self.CurrentMolfitFileContents.append(NewLine)
            for CompIndex in range(NumComp):
                NewLine = self.Tab2CreateLineForMolfitFile(CompIndex)


                ## add new line
                self.CurrentMolfitFileContents.append(NewLine)


            ## set current component
            if (self.CurrentComponentIndex <= 0):
                self.CurrentComponentIndex = 0
        else:
            NewLine = self.Tab2CreateLineForMolfitFile(self.CurrentComponentIndex)
            self.CurrentMolfitFileContents[self.CurrentComponentIndex + 1] = NewLine
        self.CompNumChangeFlag = False


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create contents of iso ratio file
    def Tab2CreateIsoRatioFileContents(self, LocalIsoRatioParam):
        """

    input parameters:
    -----------------

        - LocalIsoRatioParam:   local iso ratio parameters


    output parameters:
    ------------------

        - IsoRatioFile:         contents of iso ratio file

        """

        # Debug:
        # print ("LocalIsoRatioParam = ", LocalIsoRatioParam)


        ## check is list is not empty
        if (LocalIsoRatioParam == []):
            return


        ## create content of iso ratio file
        IsoRatioFile = []
        NewLine = "%-45s %-45s %25s %25s %25s" % ("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
        IsoRatioFile.append(NewLine + "\n")
        for IsoLine in LocalIsoRatioParam:
            if (len(IsoLine) == 3):
                NewLine = "%-45s %-45s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2])
            else:
                if (not IsoLine[5]):
                    NewLine = "%-45s %-45s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2])
                else:
                    NewLine = "%-45s %-45s %25s %25s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2], IsoLine[3], IsoLine[4])
            IsoRatioFile.append(NewLine + "\n")


        ## return to GUI
        return IsoRatioFile
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create plots
    def Tab2CreatePlots(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("self.OnceFlag = ", self.OnceFlag)


        ## setup figure
        if (self.OnceFlag):
            self.fig1 = plt.figure()
        self.fig1.clf()                                                                     ## clear figure


        ## define number of rows and columns of subplots
        NumSubPlots = len(self.SpectrumIndexList)
        colNum = 2
        if (NumSubPlots == 1):
            rowNum = 1
            colNum = 1
        else:
            if (10 < NumSubPlots <= 20):
                colNum = 4
            elif (20 < NumSubPlots <= 30):
                colNum = 5
            elif (30 < NumSubPlots <= 42):
                colNum = 6
            elif (42 < NumSubPlots <= 49):
                colNum = 7
            elif (49 < NumSubPlots <= 64):
                colNum = 8
            elif (64 < NumSubPlots):
                colNum = 9
            rowNum = math.ceil(NumSubPlots / colNum)


        ## get sort index
        LocalSortIndex = self.SQLParamArray[4]

        # Debug:
        # print ("\n\n\nself.SpectrumIndexList = ", self.SpectrumIndexList)
        # print ("LocalSortIndex = ", LocalSortIndex)
        # print ("self.IndexSortConversionList[LocalSortIndex] = ", self.IndexSortConversionList[LocalSortIndex])


        ## create figure
        CounterSpectrum = 0
        for OrigSpectrumIndex in self.SpectrumIndexList:
            i = (OrigSpectrumIndex + 1)
            LocalRowNum = math.ceil(i / colNum)
            LocalColNum = i - int(i / colNum) * colNum

            # Debug:
            # print ("\nOrigSpectrumIndex = ", OrigSpectrumIndex)
            # print ("rowNum, colNum = ", rowNum, colNum)
            # print ("LocalRowNum, LocalColNum = ", LocalRowNum, LocalColNum)


            ## depending on the sorting criteria, chose correct spectrum index
            SpectrumIndex = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][0]
            SpecTransFreq = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][1]
            SpecElow = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][2]
            Specgup = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][5]
            SpecEinsteinA = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][6]
            SpecMoleculeName = self.IndexSortConversionList[LocalSortIndex][OrigSpectrumIndex][7]


            # self.AllSpectra[i]:   [LocalObsDataTransWindow, SmoothedData, DBParam, RangeParam, IntWithoutContinuum[In1:In2], RestFreq, \
            #                        NumMolecules, LocalMolecularParameterList]


            ## plot data
            CounterSpectrum += 1
            LocalTransWin = self.AllSpectra[SpectrumIndex]
            LocalTransWinSpectrum = copy.deepcopy(LocalTransWin[0])                         ## get spectrum with continuum
            if (self.RemoveContinuumFlag):
                LocalTransWinSpectrum[:, 2] = copy.deepcopy(LocalTransWin[4])               ## get spectrum without continuum
            RestFreq = LocalTransWin[5]
            LocalMolecularParameterList = LocalTransWin[7]

            # Debug:
            # print ("\n\nNumMolecules = ", int(LocalTransWin[6]) + 1)
            # print ("LocalMolecularParameterList = ", LocalMolecularParameterList)
            # print ("LocalTransWin = ", LocalTransWin)
            # print ("self.UseIsoFlag = ", self.UseIsoFlag)


            ## determine transition frequency, lower energy, upper state degeneracy, Einstein A coefficients, iso ratio
            ## for main molecule and corresponding isotopologues
            TransFreq = []
            EinsteinA = []
            ElowMin = []
            gup = []
            ScalFactor = []
            MolIndexList = []
            if (self.UseIsoFlag):
                # NumMolPerRange = int(LocalTransWin[6]) + 1
                NumMolPerRange = int(len(self.CurrentIsoParam)) + 1
            else:
                NumMolPerRange = 1
            LocalIsotopologue = self.PureMolNameVector[0]
            for MolIndex in range(NumMolPerRange):
                LocalRatio = 1.0
                if (MolIndex > 0):
                    LocalIsoParameters = self.CurrentIsoParam[MolIndex - 1]
                    LocalIsoMaster = LocalIsoParameters[1]
                    # if (LocalIsotopologue.lower().find("global") == (-1)):
                    if (LocalIsoMaster == self.PureMolNameVector[0]):
                        LocalIsotopologue = LocalIsoParameters[0]
                        LocalRatio = LocalIsoParameters[2]
                        GlobalIndexList = LocalIsoParameters[-1]
                        if (GlobalIndexList != []):
                            for CurrGlobalIndex in GlobalIndexList:
                                LocalRatio = LocalRatio * self.CurrentIsoParam[CurrGlobalIndex[0]][2]**CurrGlobalIndex[1]

                                # Debug:
                                # print ("\n\n\n>>>>>>>LocalIsotopologue = ", LocalIsotopologue)
                                # print ("MolIndex = ", MolIndex)
                                # print ("CurrGlobalIndex = ", CurrGlobalIndex)
                                # print ("self.CurrentIsoParam[CurrGlobalIndex][2] = ", self.CurrentIsoParam[CurrGlobalIndex[0]][2])
                                # print ("LocalRatio = ", LocalRatio)


                ## extract molecular parameters for current range
                ## LocalMolecularParameterList = [TransFreq, EinsteinA, Elow, gup, Mol-index]
                for LocaldbParam in LocalMolecularParameterList:
                    LocalMolIndex = int(LocaldbParam[4])
                    if (self.PureMolNameVector[LocalMolIndex] == LocalIsotopologue):
                        TransFreq.append(LocaldbParam[0])
                        EinsteinA.append(LocaldbParam[1])
                        ElowMin.append(LocaldbParam[2])
                        gup.append(LocaldbParam[3])
                        ScalFactor.append(LocalRatio)
                        MolIndexList.append(MolIndex)
                LocalIsotopologue = ""

            # Debug:
            # print ("\n\nTransFreq = ", TransFreq)
            # print ("EinsteinA = ", EinsteinA)
            # print ("ElowMin = ", ElowMin)
            # print ("gup = ", gup)
            # print ("ScalFactor = ", ScalFactor)
            # print ("MolIndexList = ", MolIndexList)


            #---------------------------------------------------------------------------------------------------------------------------------------------
            # RangeParam:
            #
            # TelescopeSize = RangeParam[0]
            # BMIN = RangeParam[1]
            # BMAJ = RangeParam[2]
            # BPA = RangeParam[3]
            # Inter_Flag = RangeParam[4]
            # GlobalvLSR = RangeParam[5]
            # Redshift = RangeParam[6]
            # NewTBack = RangeParam[7]
            # tSlope = RangeParam[8]
            # LowFreq = RangeParam[9]
            # UpFreq = RangeParam[10]
            # nH = RangeParam[11]
            # beta = RangeParam[12]
            # kappa = RangeParam[13]
            # LocalContPhenFuncID = RangeParam[14]
            # LocalContPhenFuncParam1 = RangeParam[15]
            # LocalContPhenFuncParam2 = RangeParam[16]
            # LocalContPhenFuncParam3 = RangeParam[17]
            # LocalContPhenFuncParam4 = RangeParam[18]
            # LocalContPhenFuncParam5 = RangeParam[19]
            # BackgroundFileName = RangeParam[20]
            # DustFileName = RangeParam[21]

            RangeParam = LocalTransWin[3]                                                   ## get range parameters
            TelescopeSize = RangeParam[0]
            BMIN = RangeParam[1]
            BMAJ = RangeParam[2]
            BPA = RangeParam[3]
            InterFlag = RangeParam[4]
            GlobalvLSR = RangeParam[5]
            Redshift = RangeParam[6]
            NewTBack = RangeParam[7]
            tSlope = RangeParam[8]
            LowFreq = RangeParam[9]
            UpFreq = RangeParam[10]
            nH = RangeParam[11]
            beta = RangeParam[12]
            kappa = RangeParam[13] * (2.0 * 1.66e-24 / 100.0)
            LocalContPhenFuncID = RangeParam[14]
            LocalContPhenFuncParam1 = RangeParam[15]
            LocalContPhenFuncParam2 = RangeParam[16]
            LocalContPhenFuncParam3 = RangeParam[17]
            LocalContPhenFuncParam4 = RangeParam[18]
            LocalContPhenFuncParam5 = RangeParam[19]
            DustFileName = RangeParam[20]
            LocalBackgroundFileFunction = RangeParam[21]

            # Debug:
            # print ("\nSpectrumIndex = ", SpectrumIndex)
            # print ("TransFreq = ", TransFreq)
            # print ("RangeParam = ", RangeParam)
            # print ("LowFreq, UpFreq = ", LowFreq, UpFreq)
            # print ("LocalBackgroundFileFunction = ", LocalBackgroundFileFunction)


            ## add and adjust subplot to figure
            if (colNum < 3):
                self.fig1.subplots_adjust(top = 0.97, bottom = 0.07, left = 0.060, right = 0.99, hspace = 0.0, wspace = 0.15)
            else:
                self.fig1.subplots_adjust(top = 0.97, bottom = 0.07, left = 0.015, right = 0.99, hspace = 0.0, wspace = 0.04)
            if (CounterSpectrum == 1):
                layer = self.fig1.add_subplot(rowNum, colNum, CounterSpectrum)
                layer1 = layer
            else:
                layer = self.fig1.add_subplot(rowNum, colNum, CounterSpectrum, sharex = layer1)


            ## x labels are visible only for the lowest row
            # if (CounterSpectrum < ((rowNum * colNum) - 1) ):
            # if ((OrigSpectrumIndex + 1) <= (NumSubPlots - colNum)):
            if (CounterSpectrum <= (NumSubPlots - colNum)):
                plt.setp(layer.get_xticklabels(), visible = False)
            if (colNum > 2):
                plt.setp(layer.get_yticklabels(), visible = False)


            ## add plots to GUI
            layer.plot(LocalTransWinSpectrum[:, 1], LocalTransWinSpectrum[:, 2], 'b-', linewidth = 1.0, drawstyle = 'steps-mid')


            ## add background spectrum to GUI
            # if (LocalBackgroundFileFunction is not None and not self.RemoveContinuumFlag):
            #     LocalBackgroundFunc = LocalBackgroundFileFunction(LocalTransWinSpectrum[:, 0])
            #     layer.plot(LocalTransWinSpectrum[:, 1], LocalBackgroundFunc[:], color = 'green', linestyle = 'dashed', linewidth = 2.0)


            ## determine XCLASS spectra
            if (TransFreq != []):
                LocalIntArray = self.Tab2CalculateXCLASS(RangeParam, TransFreq, EinsteinA, ElowMin, gup, LocalTransWinSpectrum, ScalFactor, MolIndexList)
            else:
                LocalIntArray = copy.deepcopy(LocalTransWinSpectrum[:, 1])
                for FreqID, Freq in enumerate(LocalTransWinSpectrum[:, 0]):
                    LocalIntArray[FreqID] = NewTBack * (Freq / LowFreq)**tSlope
            layer.plot(LocalTransWinSpectrum[:, 1], LocalIntArray, 'r-', linewidth = 1.0)


            ## set y ticks
            yminObs = np.min(LocalTransWinSpectrum[:, 2])
            ymaxObs = np.max(LocalTransWinSpectrum[:, 2])
            yminMod = np.min(LocalIntArray)
            ymaxMod = np.max(LocalIntArray)
            ymin = min(yminObs, yminMod)
            ymax = max(ymaxObs, ymaxMod)
            ystep = (ymax - ymin) / 3.0
            if (colNum < 3):
                try:
                    ticksY = np.arange(ymin, ymax, ystep)
                    layer.set_yticks(ticksY)
                except:
                    pass
            layer.set_xlim(self.velLowLimitWeight, self.velUpLimitWeight)
            layer.grid(True)
            layer.yaxis.set_major_formatter(pylab.matplotlib.ticker.FormatStrFormatter('%.3e'))


            ## add vertical line
            layer.axvline(x = 0.0, color = 'green', linewidth = 2, linestyle = '--')
            if (colNum < 3):
                self.txt = layer.text(0.01, 0.85, '', transform = layer.transAxes, clip_on = True)


            ## add text
            if (colNum < 3):
                if (self.IsoFlag and self.UseIsoFlag):
                    self.txt.set_text(r"$\nu_{\rm trans}$=%1.2f MHz, gA=%.2e s-1, E$_{\rm low}$=%.1f K, Trans.(%i/%i), %s" % (SpecTransFreq, \
                                       (SpecEinsteinA * Specgup), SpecElow, (OrigSpectrumIndex + 1), self.TotalNumberSpectra, SpecMoleculeName))
                else:
                    self.txt.set_text(r"$\nu_{\rm trans}$=%1.2f MHz, gA=%.2e s-1, E$_{\rm low}$=%.1f K, Trans.(%i/%i)" % (SpecTransFreq, \
                                       (SpecEinsteinA * Specgup), SpecElow, (OrigSpectrumIndex + 1), self.TotalNumberSpectra))


        ## add figure only once to the GUI
        if (self.OnceFlag):
            self.OnceFlag = False
            if (self.AllInOneWindowFlag):
                self.canvas = FigureCanvas(self.fig1)
                self.gridLayout.addWidget(self.canvas)
                self.canvas.draw()
                self.toolbar = NavigationToolbar(self.canvas, self, coordinates = True)
                self.gridLayout.addWidget(self.toolbar)
                self.canvas.draw()
            else:
                plt.show(block = False)
        else:
            if (self.AllInOneWindowFlag):
                self.canvas.draw()
            else:
                plt.draw()
                plt.pause(0.0001)
        self.InitFlag = False
        self.ChangeSpectrumWindowFlag = False

        # Debug:
        # print ("done!")


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## calculate XCLASS spectra
    def Tab2CalculateXCLASS(self, RangeParam, TransFreqList, EinsteinAList, ElowMinList, gupList, LocalTransWinSpectrum, ScalFactor, MolIndexList):
        """

    input parameters:
    -----------------

        - RangeParam:       range parameters

        - TransFreqList:    (list of) transition frequency(ies)

        - EinsteinAList:    Einstein A coefficient(s)

        - ElowMinList:      lowest energy(ies)

        - gupList:          upper state degeneracy(ies)

        - LocalTransWinSpectrum:    frequency range

        - ScalFactor:       list of scaling factors

        - MolIndexList:     list of molecule indices


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("\nRangeParam = ", RangeParam)
        # print ("TransFreqList = ", TransFreqList)
        # print ("EinsteinAList = ", EinsteinAList)
        # print ("ElowMinList = ", ElowMinList)
        # print ("gupList = ", gupList)
        # print ("ScalFactor = ", ScalFactor)
        # print ("MolIndexList = ", MolIndexList)


        ## define frequency axis for current transition frequency window
        LocalFreqAxis = LocalTransWinSpectrum[:, 0]
        NumFreqPoints = len(LocalFreqAxis)


        ## get range parameters
        TelescopeSize = RangeParam[0]
        BMIN = RangeParam[1]
        if (BMIN is None):
            BMIN = 0.0
        BMAJ = RangeParam[2]
        if (BMAJ is None):
            BMAJ = 0.0
        BPA = RangeParam[3]
        if (BPA is None):
            BPA = 0.0
        InterFlag = task_myXCLASSMapFit.CheckBool(RangeParam[4])
        GlobalvLSR = RangeParam[5]
        if (GlobalvLSR is None):
            GlobalvLSR = 0.0
        Redshift = RangeParam[6]
        if (Redshift is None):
            Redshift = 0.0
        t_back_flag = False                                                                  ## CHECK !!!!!!!!!!!!! <=====================
        TBack = RangeParam[7]
        tSlope = RangeParam[8]
        LowFreq = RangeParam[9]
        UpFreq = RangeParam[10]
        nH = RangeParam[11]
        if (nH is None):
            nH = 0.0
        beta = RangeParam[12]
        if (beta is None):
            beta = 0.0
        kappa = RangeParam[13] * (2.0 * 1.66e-24 / 100.0)
        if (kappa is None):
            kappa = 0.0
        LocalContPhenFuncID = RangeParam[14]
        if (LocalContPhenFuncID is None):
            LocalContPhenFuncID = 0.0
        LocalContPhenFuncParam1 = RangeParam[15]
        if (LocalContPhenFuncParam1 is None):
            LocalContPhenFuncParam1 = 0.0
        LocalContPhenFuncParam2 = RangeParam[16]
        if (LocalContPhenFuncParam2 is None):
            LocalContPhenFuncParam2 = 0.0
        LocalContPhenFuncParam3 = RangeParam[17]
        if (LocalContPhenFuncParam3 is None):
            LocalContPhenFuncParam3 = 0.0
        LocalContPhenFuncParam4 = RangeParam[18]
        if (LocalContPhenFuncParam4 is None):
            LocalContPhenFuncParam4 = 0.0
        LocalContPhenFuncParam5 = RangeParam[19]
        if (LocalContPhenFuncParam5 is None):
            LocalContPhenFuncParam5 = 0.0
        LocalDustFileFunction = RangeParam[20]
        if (LocalDustFileFunction is None):
            LocalDustFunc = np.zeros((NumFreqPoints, 1))
        else:
            LocalDustFunc = LocalDustFileFunction(LocalFreqAxis)
        LocalBackgroundFileFunction = RangeParam[21]
        if (LocalBackgroundFileFunction is None):
            LocalBackgroundFunc = np.zeros((NumFreqPoints, 1))
        else:
            LocalBackgroundFunc = LocalBackgroundFileFunction(LocalFreqAxis)
        LocalEmissionAbsorptionFunction = RangeParam[30]

        # Debug:
        # print ("RangeParam = ", RangeParam)
        # print ("LocalDustFunc = ", LocalDustFunc)
        # print ("LocalBackgroundFunc = ", LocalBackgroundFunc)
        # print ("LocalEmissionAbsorptionFunction = ", LocalEmissionAbsorptionFunction)


        ## determine parameters for Fortran routine
        if (InterFlag):
            InterFlagint = 1
        else:
            InterFlagint = 0
        if (self.RemoveContinuumFlag):
            LocalRemoveContinuumFlag = 1
        else:
            LocalRemoveContinuumFlag = 0
        if (self.UserLocalOverlapFlag):
            CurrentLocalOverlapFlag = 1
        else:
            CurrentLocalOverlapFlag = 0


        ## get RRL flag
        LocalMolName = self.CurrentMolecule.strip()
        LocalMolName = LocalMolName.lower()
        KindOfMolecule = 0.0
        if (LocalMolName.startswith("rrl-")):
            KindOfMolecule = 1.0

        # Debug:
        # print ("KindOfMolecule = ", KindOfMolecule)


        ## determine partition function values for different temperatures
        NumCompCont = 0
        for CurrentContinuumParameter in self.MolfitContinuumParameters:
            LocalMoleculeParameters = CurrentContinuumParameter[1]
            NumCompCont += len(LocalMoleculeParameters)
        NumComp = len(self.CurrentMolfitParameter) + NumCompCont
        NumIso = len(self.CurrentIsoParam)
        QTList = np.zeros((NumComp, NumIso + 1), dtype=np.float32)
        LocalActiveCompList = []
        self.NumEmissionComp = 0
        LocalMolfitParam = []
        ListDistances = []
        for CounterComp, Compparam in enumerate(self.CurrentMolfitParameter):
            ss = Compparam[0][3]
            Temp = Compparam[1][3]
            Ntot = Compparam[2][3]
            vwidth = Compparam[3][3]
            voff = Compparam[4][3]
            CFFlag = Compparam[5][0]
            if (self.IsNumber(CFFlag)):
                CFFlag = float(CFFlag)
            else:
                CFFlag = CFFlag.strip()
                CFFlag = CFFlag.lower()
                if (CFFlag in ["c", "e"]):
                    CFFlag = NumComp + 1
                else:
                    CFFlag = ((NumComp + 1) - (NumCompCont + CounterComp + 1))
            if (not CFFlag in ListDistances):
                ListDistances.append(CFFlag)
            LocalMolfitParam.append([KindOfMolecule, ss, Temp, Ntot, vwidth, voff, 0.0, 0.0, CFFlag])


            ## determine number of core components
            if (self.ActiveCompList[CounterComp] and CFFlag == 1):
                self.NumEmissionComp += 1


            ## get active flags
            if (self.ActiveCompList[CounterComp]):
                flag = 1
            else:
                flag = 2
            LocalActiveCompList.append(flag)


            ## get partition function
            QT = task_LineIdentification.GetPartitionFunction(self.CurrentMolecule, Temp, self.dbList)
            QTList[CounterComp, 0] = QT
            if (self.CurrentIsoParam != []):
                for IsoID in range(NumIso):
                    LocalIsoParameters = self.CurrentIsoParam[IsoID]
                    Isotopologue = LocalIsoParameters[0]
                    if (Isotopologue.lower().find("global") == (-1)):
                        QT = task_LineIdentification.GetPartitionFunction(Isotopologue, Temp, self.dbList)
                        QTList[CounterComp, IsoID + 1] = QT
                    else:
                        QTList[CounterComp, IsoID + 1] = 1.0

            # Debug:
            # print ("CounterComp, QTList[CounterComp, :] = ", CounterComp, QTList[CounterComp, :])


        ## add continuum parameters
        LocalNumEmissionComp = self.NumEmissionComp
        CounterComp = 0
        for CurrentContinuumParameter in self.MolfitContinuumParameters:


            ## get kind of contiuum
            LowerLocalMoleculeName = CurrentContinuumParameter[0]
            if (LowerLocalMoleculeName.startswith("cont-dust")):                            ## cont-dust
                KindOfMolecule = 3
            elif (LowerLocalMoleculeName.startswith("cont-phen")):                          ## cont-phen
                KindOfMolecule = 6


            ## get corresponding component parameters
            LocalMoleculeParameters = CurrentContinuumParameter[1]
            for Compparam in LocalMoleculeParameters:
                CounterComp += 1
                ss = Compparam[0][3]
                LocalParameterVector = [KindOfMolecule, ss]
                if (KindOfMolecule == 3):                                                   ## dust
                    T_d = Compparam[1][3]
                    CurrentnH = Compparam[2][3]
                    CurrentKappa = Compparam[3][3]
                    CurrentBeta = Compparam[4][3]
                    CFFlag = Compparam[5][0]
                    LocalParameterVector = LocalParameterVector + [T_d, CurrentnH, CurrentKappa, CurrentBeta, 0.0, 0.0]
                elif (KindOfMolecule == 6):                                                 ## phenomenical
                    CurrentContPhenFuncID = Compparam[1][3]
                    CurrentContPhenFuncParam1 = Compparam[2][3]
                    CurrentContPhenFuncParam2 = Compparam[3][3]
                    CurrentContPhenFuncParam3 = Compparam[4][3]
                    CurrentContPhenFuncParam4 = Compparam[5][3]
                    CurrentContPhenFuncParam5 = Compparam[6][3]
                    CFFlag = Compparam[7][0]
                    LocalParameterVector = LocalParameterVector + [CurrentContPhenFuncID, CurrentContPhenFuncParam1, CurrentContPhenFuncParam2, \
                                                                   CurrentContPhenFuncParam3, CurrentContPhenFuncParam4, CurrentContPhenFuncParam5]

                ## post-process CF-Flag
                if (self.IsNumber(CFFlag)):
                    CFFlag = float(CFFlag)
                else:
                    CFFlag = CFFlag.strip()
                    CFFlag = CFFlag.lower()
                    if (CFFlag in ["c", "e"]):
                        CFFlag = NumComp + 1
                        LocalNumEmissionComp += 1
                    else:
                        CFFlag = ((NumComp + 1) - (CounterComp + 1)) + 1
                if (not CFFlag in ListDistances):                                           ## identify all distances in molfit file
                    ListDistances.append(CFFlag)
                LocalParameterVector.append(CFFlag)
                LocalMolfitParam.append(LocalParameterVector)
                LocalActiveCompList.append(1)                                               ## contiuum components are always active


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## prepare emission and absorption files (if defined)


        ## get list of distances defined in molfit file
        ListDistances.sort(reverse = True)                                                  ## sort list of molfit distances
        NumDistances = len(ListDistances)                                                   ## get number of distances defined in molfit file

        # Debug:
        # print ("ListDistances = ", ListDistances)
        # print ("NumDistances = ", NumDistances)
        # print ("LocalEmissionAbsorptionFunction = ", LocalEmissionAbsorptionFunction)


        ## initialize array for emission / absorption funciton
        EmsAbsFunc = np.zeros((NumFreqPoints, NumDistances, 2))


        ## check, if emission and absorption functions were defined
        if (LocalEmissionAbsorptionFunction is not None and self.EmAbsFlag):                ## are emission / absorption functions defined?
            EmsAbsFileContentDir = LocalEmissionAbsorptionFunction                          ## get interpolation functions for each distance


            ## get a list of all distances defined in the emission / absorption files
            DistanceList = [[float(x), x] for x in list(EmsAbsFileContentDir.keys())]
            DistanceList = sorted(DistanceList, key = lambda l:l[0], reverse = True)

            # Debug:
            # print ("Dict-DistanceList = ", DistanceList)


            ## define emission / absorption funcitons for each distance
            for LocalDistance in DistanceList:                                              ## loop over all distances in emission / absorption files
                LocalDistanceNum = LocalDistance[0]                                         ## get distance value
                if (LocalDistanceNum in ListDistances):                                     ## check, if current dist. is defined in molfit file as well
                    c = ListDistances.index(LocalDistanceNum)                               ## get index of current distance
                    LocalDistanceString = LocalDistance[1]                                  ## get directory key
                    LocalFunc = LocalEmissionAbsorptionFunction[LocalDistanceString]
                    EmissionFunc = LocalFunc[0]
                    AbsorptionFunc = LocalFunc[1]
                    EmsAbsFunc[:, c, 0] = EmissionFunc(LocalFreqAxis)
                    EmsAbsFunc[:, c, 1] = AbsorptionFunc(LocalFreqAxis)

                    # Debug:
                    # print ("c, EmsAbsFunc[:, c, 0] = ", c, EmsAbsFunc[:, c, 0])


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## convert lists to numpy arrays
        NumDBParam = len(TransFreqList)
        TransFreqList = np.asarray(TransFreqList)
        EinsteinAList = np.asarray(EinsteinAList)
        ElowMinList = np.asarray(ElowMinList)
        gupList = np.asarray(gupList)
        ScalFactor = np.asarray(ScalFactor)
        MolIndexList = np.asarray(MolIndexList)

        # Debug:
        # print ("NumDBParam = ", NumDBParam)
        # print ("NumComp = ", NumComp)
        # print ("TransFreqList = ", TransFreqList)
        # print ("LocalMolfitParam = ", LocalMolfitParam
        # print xclasslite.calcxclass.__doc__)
        # print ("InterFlagint = ", InterFlagint)
        # print ("np.min(LocalFreqAxis[:]) = ", np.min(LocalFreqAxis[:]))
        # print ("np.max(LocalFreqAxis[:]) = ", np.max(LocalFreqAxis[:]))
        # print ("LocalNumEmissionComp = ", LocalNumEmissionComp)


        ## start fortran module
        ModeledRangeSpectrum = xclasslite.calcxclass(LocalNumEmissionComp, LocalRemoveContinuumFlag, TelescopeSize, BMIN, BMAJ, BPA, \
                                                     InterFlagint, GlobalvLSR, Redshift, LowFreq, t_back_flag, TBack, tSlope, nH, beta, kappa, \
                                                     LocalContPhenFuncID, LocalContPhenFuncParam1, LocalContPhenFuncParam2, LocalContPhenFuncParam3, \
                                                     LocalContPhenFuncParam4, LocalContPhenFuncParam5, TransFreqList, EinsteinAList, ElowMinList, \
                                                     gupList, ScalFactor, MolIndexList, LocalFreqAxis, LocalDustFunc, LocalBackgroundFunc, \
                                                     LocalActiveCompList, QTList, LocalMolfitParam, EmsAbsFunc, CurrentLocalOverlapFlag)
                                                     # NumDBParam, NumFreqPoints, NumComp, NumIso, NumDist, \
        # Debug:
        # print ("ModeledRangeSpectrum = ", ModeledRangeSpectrum)


        ## return to GUI
        return ModeledRangeSpectrum
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## update history arrays
    def Tab2UpdateHistoryArray(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("\nself.OldParamVectorCounter = ", self.OldParamVectorCounter)
        # print ("len(self.CurrentMolfitParameter) = ", len(self.CurrentMolfitParameter))


        ## update history parameters
        if (self.OldParamVectorCounter < (len(self.OldIsoRatioParameter) - 1)):
            del self.OldMolfitParameter[self.OldParamVectorCounter + 1:]
            del self.OldIsoRatioParameter[self.OldParamVectorCounter + 1:]
            del self.OldLocalMolfitFile[self.OldParamVectorCounter + 1:]
            del self.OldMoleculesInMolfitFile[self.OldParamVectorCounter + 1:]
            del self.OldAllMoleculesMolfitFiles[self.OldParamVectorCounter + 1:]
            del self.OldMolfitFileAllParameters[self.OldParamVectorCounter + 1:]
            del self.OldCurrentMolecule[self.OldParamVectorCounter + 1:]
            del self.OldCurrentMoleculeID[self.OldParamVectorCounter + 1:]
            del self.OldCurrentMoleculeDir[self.OldParamVectorCounter + 1:]
            del self.OldAllIsotopologueList[self.OldParamVectorCounter + 1:]
            del self.OldActiveCompList[self.OldParamVectorCounter + 1:]
        self.OldParamVectorCounter += 1
        self.OldMolfitParameter.append(copy.deepcopy(self.CurrentMolfitParameter))
        self.OldIsoRatioParameter.append(copy.deepcopy(self.CurrentIsoParam))
        self.OldLocalMolfitFile.append(copy.deepcopy(self.LocalMolfitFile))
        self.OldMoleculesInMolfitFile.append(copy.deepcopy(self.MoleculesInMolfitFile))
        self.OldAllMoleculesMolfitFiles.append(copy.deepcopy(self.AllMoleculesMolfitFiles))
        self.OldMolfitFileAllParameters.append(copy.deepcopy(self.MolfitFileAllParameters))
        self.OldCurrentMolecule.append(copy.deepcopy(self.CurrentMolecule))
        self.OldCurrentMoleculeID.append(copy.deepcopy(self.CurrentMoleculeID))
        self.OldCurrentMoleculeDir.append(copy.deepcopy(self.CurrentMoleculeDir))
        self.OldAllIsotopologueList.append(copy.deepcopy(self.AllIsotopologueList))
        self.OldActiveCompList.append(copy.deepcopy(self.ActiveCompList))

        # Debug:
        # print ("\nself.OldParamVectorCounter = ", self.OldParamVectorCounter)
        # print ("self.OldMolfitParameter = ", self.OldMolfitParameter)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get current component
    def Tab2molfitListEntrySelected(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## is GUI locked?
        if (self.lock == 1):
            return


        ## get selected line
        StrippedLine = self.list_molfit.currentItem().text()
        StrippedLine = str(StrippedLine).strip()
        self.CurrentMolfitIndex = self.list_molfit.currentRow()
        if (self.CurrentMolfitIndex == 0):
            self.CurrentMolfitIndex = 1
            self.list_molfit_Tab3.setCurrentRow(self.CurrentMolfitIndex)
        self.CurrentComponentIndex = self.CurrentMolfitIndex - 1
        self.Tab2SetValuesForTab()

        # Debug:
        if (self.dbgFlag):
            print ("\nmolfitListEntrySelected:")
            print ("self.CurrentComponentIndex = ", self.CurrentComponentIndex)
            print ("self.CurrentMolfitIndex = ", self.CurrentMolfitIndex)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##====================================================================================================================================================
    ##
    ## event handling
    ##
    ##====================================================================================================================================================


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for radio box events
    def Tab2radioSelected(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## GUI locked?
        if (self.lock == 1):
            return


        ## get name and flag of radio button
        sender = self.sender()
        flag = sender.isChecked()
        objectName = sender.objectName()


        ## set flag
        if (objectName == "radioButton_Core" and flag):
            self.CurrentMolfitParameter[self.CurrentComponentIndex][-1][0] = "c"
            self.CurrentMolfitParameter[self.CurrentComponentIndex][0][0] = "n"
            self.CurrentMolfitParameter[self.CurrentComponentIndex][0][1] = 1.0
            self.CurrentMolfitParameter[self.CurrentComponentIndex][0][2] = 1.e4
            self.CurrentMolfitParameter[self.CurrentComponentIndex][0][3] = 1.e2
        elif (objectName == "radioButton_Fore" and flag):
            self.CurrentMolfitParameter[self.CurrentComponentIndex][-1][0] = "f"

        # Debug:
        if (self.dbgFlag):
            print ("\nradioSelected:")
            print ("objectName = ", objectName)
            print ("flag = ", flag)


        ## update history arrays
        self.Tab2UpdateHistoryArray()


        ## update GUI
        self.Tab2SetValuesForTab()


        ## set current event
        self.LastEvent = objectName


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for check box events
    def Tab2checkBoxSelected(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## is GUI locked
        if (self.lock == 1):
            return


        ## get name, flag, and object name of selected check box
        sender = self.sender()
        flag = sender.isChecked()
        objectName = sender.objectName()
        if (objectName == "checkBox_FF_ss"):
            ParamNumber = 0
        elif (objectName == "checkBox_FF_Temp"):
            ParamNumber = 1
        elif (objectName == "checkBox_FF_Ntot"):
            ParamNumber = 2
        elif (objectName == "checkBox_FF_vwidth"):
            ParamNumber = 3
        elif (objectName == "checkBox_FF_voff"):
            ParamNumber = 4
        elif (objectName == "checkBox_FF_iso"):
            ParamNumber = 5


        ## set new parameter
        if (ParamNumber < 5):
            if (flag):
                self.CurrentMolfitParameter[self.CurrentComponentIndex][ParamNumber][0] = "y"
            else:
                self.CurrentMolfitParameter[self.CurrentComponentIndex][ParamNumber][0] = "n"
        else:
            self.CurrentIsoParam[self.CurrentIsoID][5] = flag

        # Debug:
        if (self.dbgFlag):
            print ("\ncheckBoxSelected:")
            print ("ParamNumber = ", ParamNumber)
            print ("flag = ", flag)


        ## update history arrays
        self.Tab2UpdateHistoryArray()


        ## set title of GUI window
        self.GUITitle = "XCLASS-GUI*"
        if (self.tabWidget.currentIndex() == 1):
            self.GUITitle += "   ({:d}/{:d})".format(self.CurrentMoleculeID + 1, len(self.MoleculesInMolfitFile))
        self.setWindowTitle(self.GUITitle)


        ## update GUI
        self.Tab2SetValuesForTab()


        ## set current event
        self.LastEvent = objectName


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for active check box event
    def Tab2checkActive(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## is GUI locked
        if (self.lock == 1):
            return


        ## set current component active or deactive
        if (self.ActiveCompList[self.CurrentComponentIndex]):
            self.ActiveCompList[self.CurrentComponentIndex] = False
        else:
            self.ActiveCompList[self.CurrentComponentIndex] = True

        # Debug:
        if (self.dbgFlag):
            print ("\ncheckActive:")
            print ("self.CurrentComponentIndex = ", self.CurrentComponentIndex)
            print ("self.ActiveCompList[self.CurrentComponentIndex] = ", self.ActiveCompList[self.CurrentComponentIndex])


        ## update history arrays
        self.Tab2UpdateHistoryArray()


        ## update GUI
        self.Tab2SetValuesForTab()


        ## set current event
        self.LastEvent = "Tab2checkActive"


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## remove background files
    def Tab2RemoveBackgroundFiles(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## get all filenames in the current job directory
        cmdString = "rm -rf " + self.CurrentMoleculeDir + "background-spectrum___data-file_*.dat"
        os.system(cmdString)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## compute background spectrum
    def Tab2checkBackground(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is only one molecule contained in current molfit file
        if (len(self.MoleculesInMolfitFile) == 1):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if background flag is set
        if (self.BackgroundFlag):
            self.BackgroundFlag = False
            self.BackgroundFileNameList = []


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## background flag was not set before
        else:
            self.BackgroundFlag = True                                                      ## set background file flag


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## determine background file names
            OldBackgroundFileNameList = copy.deepcopy(self.BackgroundFileNameList)
            self.BackgroundFileNameList = []
            DirListing = os.listdir(self.CurrentMoleculeDir)                                ## get all filenames in the current job directory
            for LocalFile in DirListing:
                if (LocalFile.startswith("background-spectrum___data-file_")):
                    WorkingString = LocalFile.replace("background-spectrum___data-file_", "")
                    WorkingString = WorkingString.replace(".dat", "")
                    WorkingString = WorkingString.split("_")
                    ObsDataFileIndex = int(WorkingString[0]) - 1
                    RangeIndex = int(WorkingString[-1]) - 1
                    LocalFileName = self.CurrentMoleculeDir + LocalFile
                    self.BackgroundFileNameList.append([ObsDataFileIndex, RangeIndex, LocalFileName])


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## check, if new background file have to be computed
            if (self.BackgroundFileNameList == []):


                ## disable elements of tab 2
                self.Tab2setDisEnabled(False)


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## create molfit file without current molecule


                ## get sql parameters
                MinNumTransitionsSQL = self.SQLParamArray[0]
                MaxNumTransitionsSQL = self.SQLParamArray[1]
                MaxElowSQL = self.SQLParamArray[2]
                MingASQL = self.SQLParamArray[3]
                OrderTransSQL = self.SQLParamArray[4]
                MolfitHeaderLinesList = task_myXCLASS.WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)

                # Debug:
                # print ("self.AllMoleculesMolfitFiles = ", self.AllMoleculesMolfitFiles)
                # print ("self.MolfitFileName = ", self.MolfitFileName)


                ## check, if self.AllMoleculesMolfitFiles is already defined
                if (self.AllMoleculesMolfitFiles == []):
                    if (self.MolfitFileName.strip() != ""):
                        LocalMolInMolfitFile, LocalMolParam, LocalMolfitForEachMol = task_myXCLASS.AnalyzeMolfitFile(self.MolfitFileName.strip())
                    else:
                        return
                else:
                    LocalMolfitFileContent = []
                    for LocalMoleculeMolfitFile in self.AllMoleculesMolfitFiles:
                        for LocalLine in LocalMoleculeMolfitFile:
                            NewLine = LocalLine + "\n"
                            NewLine = NewLine. replace("\n\n", "\n")
                            LocalMolfitFileContent.append(NewLine)
                    LocalMolInMolfitFile, LocalMolParam, LocalMolfitForEachMol = task_myXCLASS.AnalyzeMolfitFile(LocalMolfitFileContent)

                # Debug:
                # print ("LocalMolInMolfitFile = ", LocalMolInMolfitFile)
                # print ("LocalMolParam = ", LocalMolParam)
                # print ("LocalMolfitForEachMol = ", LocalMolfitForEachMol)
                # print ("self.CurrentMolecule = ", self.CurrentMolecule)


                ## write current molfit file contents to file
                LocalMolfitFileName = self.CurrentMoleculeDir + "background-spectrum.molfit"
                LocalMolfitFile = open(LocalMolfitFileName, "w")
                for line in MolfitHeaderLinesList:                                          ## write header of molfit file
                    LocalMolfitFile.write(line)
                for LocalMoleculeID, LocalMolecule in enumerate(LocalMolInMolfitFile):      ## loop over all molecules
                    if (LocalMolecule.strip() != self.CurrentMolecule.strip()):             ## include all molecules except current molecule
                        for line in LocalMolfitForEachMol[LocalMoleculeID]:                 ## write molfit parameters to file
                            LocalMolfitFile.write(line)
                LocalMolfitFile.close()


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## write current iso ratio file contents to file
                LocalIsoRatioFileName = ""
                if (self.UseIsoFlag):
                    IsoRatioFileContents = self.Tab2CreateIsoRatioFileContents(self.CurrentIsoParam)
                    LocalIsoRatioFileName = self.CurrentMoleculeDir + "iso-ratio.dat"
                    LocalIsoRatioFile = open(LocalIsoRatioFileName, "w")
                    NewLine = "%-45s %-45s %25s %25s %25s" % ("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
                    LocalIsoRatioFile.write(NewLine + "\n")
                    for IsoLine in IsoRatioFileContents:
                        LocalIsoRatioFile.write(IsoLine + "\n")
                    LocalIsoRatioFile.close()


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## get additional settings


                ## get path and name of database file
                LocalDBFileName = ""
                if (self.dbList != []):
                    LocalDBFileName = self.dbList[0]

                # Debug:
                # print ("LocalDBFileName = ", LocalDBFileName)


                ## get number of pixels
                LocalNumModelPixelXX = None
                if (self.NumModelPixelXXList != []):
                    LocalNumModelPixelXX = self.NumModelPixelXXList[0]
                LocalNumModelPixelYY = None
                if (self.NumModelPixelYYList != []):
                    LocalNumModelPixelYY = self.NumModelPixelYYList[0]
                LocalLocalOverlapFlag = None
                if (self.LocalOverlapFlagList != []):
                    LocalLocalOverlapFlag = self.LocalOverlapFlagList[0]
                LocalNoSubBeamFlag = None
                if (self.NoSubBeamFlagList != []):
                    LocalNoSubBeamFlag = self.NoSubBeamFlagList[0]
                LocalEmAbsPATH = None
                if (self.EmAbsPATHList != []):
                    LocalEmAbsPATH = self.EmAbsPATHList[0]

                # Debug:
                # print ("LocalNumModelPixelXX = ", LocalNumModelPixelXX)
                # print ("LocalNumModelPixelYY = ", LocalNumModelPixelYY)


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## compute spectrum for each frequency range of each obs. data file


                ## make a copy of curent obs. data file
                CurrentObsXMLFileName = self.CurrentMoleculeDir + "obs.xml"
                cmdString = "cp " + CurrentObsXMLFileName + " " + self.CurrentMoleculeDir + "exp_background.xml"
                os.system(cmdString)


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## determine number of myXCLASS calls
                NumExpFiles = len(self.ExpFileList)                                         ## get number of obs. data files
                NumCalls = 0
                LocalFreqCounter = (-1)                                                     ## reset counter for freq. range
                for ObsDataFileIndex in range(NumExpFiles):                                 ## loop over all spectra
                    if (self.CurrentObsDataID == (-1) or self.CurrentObsDataID == ObsDataFileIndex):


                        ## get number of freq. ranges from obs. xml file
                        RangeIndex = (-1)
                        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                       NumberRangeListIn = self.NumberExpRangesList)
                        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges'] ## get number of freq. ranges for current obs. data file


                        ## get parameters for current frequency range
                        for RangeIndex in range(NumberFrequencyRanges):                     ## loop over all range definitions in the whole xml file
                            LocalFreqCounter += 1                                           ## increase counter for freq. range
                            if ((self.CurrentFreqRangeID - 1) != (-1) or (self.CurrentFreqRangeID - 1) != LocalFreqCounter):
                                NumCalls += 1


                ## call progress bar dialog
                NumCalls += 2
                widgetTitleString = "Calculate background spectra for selected frequency ranges"
                bodyTextString = "Call myXCLASS to calculate background spectra .."
                bar = ProgressBar(total = NumCalls, widgetTitle = widgetTitleString, bodyText = bodyTextString)
                bar.resize(500, 100)
                desktopSize = QtWidgets.QDesktopWidget().screenGeometry()
                top = int((desktopSize.height() / 2.0) - 50)
                left = int((desktopSize.width() / 2.0) - 250)
                bar.move(left, top)
                bar.show()


                ## update progress bar
                ProgressBarValue = 1
                ok = bar.setValue(ProgressBarValue)
                if (not ok):
                    print ("\\n\\nAbort")
                    self.lock = 0
                    return

                # Debug:
                # print ("\nNumExpFiles = ", NumExpFiles)
                # print ("self.FreqMinList = ", self.FreqMinList)
                # print ("self.FreqMaxList = ", self.FreqMaxList)


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## get calculation parameters from obs. xml file
                LocalFreqCounter = (-1)                                                     ## reset counter for freq. range
                LocalBackgroundFileNameList = []
                for ObsDataFileIndex in range(NumExpFiles):                                 ## loop over all spectra
                    if (self.CurrentObsDataID == (-1) or self.CurrentObsDataID == ObsDataFileIndex):


                        ## get number of freq. ranges etc. from obs. xml file
                        RangeIndex = (-1)
                        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                       NumberRangeListIn = self.NumberExpRangesList, \
                                                                                       TelescopeSizeListIn = self.TelescopeSizeList, \
                                                                                       BMIN_ListIn = self.BMINList, \
                                                                                       BMAJ_ListIn = self.BMAJList, \
                                                                                       BPA_ListIn = self.BPAList, \
                                                                                       InterFlagListIn = self.Inter_FlagList, \
                                                                                       GlobalvLSRListIn = self.GlobalvLSRList, \
                                                                                       Redshift_ListIn = self.RedshiftList)
                        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
                        TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
                        if (TelescopeSize is None):
                            TelescopeSize = 0.0
                        BMIN = ObsXMLParameterDictFile['BMIN']
                        BMAJ = ObsXMLParameterDictFile['BMAJ']
                        BPA = ObsXMLParameterDictFile['BPA']
                        InterFlag = ObsXMLParameterDictFile['InterFlag']
                        if (InterFlag is None):
                            InterFlag = False
                        GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
                        if (GlobalvLSR is None):
                            GlobalvLSR = 0.0
                        Redshift = ObsXMLParameterDictFile['Redshift']
                        if (Redshift is None):
                            Redshift = 0.0


                        ## determine new frequency range definitions and write to xml file
                        TotalNumberFreqRanges = 0
                        for RangeIndex in range(NumberFrequencyRanges):                     ## loop over all frequency ranges
                            LocalFreqCounter += 1
                            if ((self.CurrentFreqRangeID - 1) == (-1) or (self.CurrentFreqRangeID - 1) == LocalFreqCounter):


                                ## get parameter for current frequency range
                                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                                FreqMinListIn = self.FreqMinList, \
                                                                                                FreqMaxListIn = self.FreqMaxList, \
                                                                                                FreqStepListIn = self.FreqStepList, \
                                                                                                tBackFlagListIn = self.t_back_flagList, \
                                                                                                tBackListIn = self.tBackList, \
                                                                                                tSlopeListIn = self.tSlopeList, \
                                                                                                N_HListIn = self.N_HList, \
                                                                                                beta_dustListIn = self.beta_dustList, \
                                                                                                kappa_1300ListIn = self.kappa_1300List, \
                                                                                                DustFileNameListIn = self.DustFileNameList, \
                                                                                                BackgroundFileNameListIn = OldBackgroundFileNameList, \
                                                                                                ContPhenFuncID_ListIn = self.ContFuncIDPhenList, \
                                                                                                ContPhenFuncParam1_ListIn = self.ContParam1PhenList, \
                                                                                                ContPhenFuncParam2_ListIn = self.ContParam2PhenList, \
                                                                                                ContPhenFuncParam3_ListIn = self.ContParam3PhenList, \
                                                                                                ContPhenFuncParam4_ListIn = self.ContParam4PhenList, \
                                                                                                ContPhenFuncParam5_ListIn = self.ContParam5PhenList)
                                FreqMin = ObsXMLParameterDictRange['FreqMin']
                                if (FreqMin is not None):
                                    FreqMax = ObsXMLParameterDictRange['FreqMax']
                                    FreqStep = ObsXMLParameterDictRange['FreqStep']
                                    t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                                    t_back_flag = task_myXCLASSMapFit.CheckBool(t_back_flag)
                                    tBack = ObsXMLParameterDictRange['tBack']
                                    tSlope = ObsXMLParameterDictRange['tSlope']
                                    nH_flag = True
                                    N_H = ObsXMLParameterDictRange['N_H']
                                    beta_dust = ObsXMLParameterDictRange['beta_dust']
                                    kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                                    DustFileName = ObsXMLParameterDictRange['DustFileName']
                                    BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']


                                    ## check background parameters
                                    if ((not t_back_flag) or tBack is None or tSlope is None):
                                        tBack = 0.0                                         ## set background temperature
                                        tSlope = 0.0                                        ## set temperature slope


                                    ## check dust parameters
                                    nH_flag = True
                                    if (N_H is None or beta_dust is None or kappa_1300 is None):
                                        nH_flag = False                                     ## set nH flag to False
                                        N_H = 0.0                                           ## set nH density
                                        beta_dust = 0.0                                     ## set beta
                                        kappa_1300 = 0.0                                    ## set kappa


                                    ## check dust file
                                    if (DustFileName is None):
                                        DustFileName = ""


                                    ## check background file
                                    if (BackgroundFileName is None):
                                        BackgroundFileName = ""


                                    ## get parameters for phenomenological description of continuum
                                    LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                                    LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                                    LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                                    LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                                    LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                                    LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']


                                    ## update progress bar
                                    ProgressBarValue += 1
                                    ok = bar.setValue(ProgressBarValue)
                                    if (not ok):
                                        print ("\\n\\nAbort")
                                        self.lock = 0
                                        return

                                    # Debug:
                                    # print ("\n\nFreqMin = ", FreqMin)
                                    # print ("FreqMax = ", FreqMax)
                                    # print ("FreqStep = ", FreqStep)
                                    # print ("MolfitsFileName = ", LocalMolfitFileName)
                                    # print ("iso_flag = ", self.IsoFlag)
                                    # print ("IsoTableFileName = ", IsoTableFileName)
                                    # print ("TelescopeSize = ", TelescopeSize)
                                    # print ("BMIN = ", BMIN)
                                    # print ("BMAJ = ", BMAJ)
                                    # print ("BPA = ", BPA)
                                    # print ("Inter_Flag = ", InterFlag)
                                    # print ("vLSR = ", GlobalvLSR)
                                    # print ("Redshift = ", Redshift)
                                    # print ("t_back_flag = ", t_back_flag)
                                    # print ("tBack = ", tBack)
                                    # print ("tSlope = ", tSlope)
                                    # print ("nH_flag = ", nH_flag)
                                    # print ("N_H = ", N_H)
                                    # print ("beta_dust = ", beta_dust)
                                    # print ("kappa_1300 = ", kappa_1300)
                                    # print ("DustFileName = ", DustFileName)
                                    # print ("BackgroundFileName = ", BackgroundFileName)
                                    # print ("ContPhenFuncID = ", LocalContPhenFuncID)
                                    # print ("ContPhenFuncParam1 = ", LocalContPhenFuncParam1)
                                    # print ("ContPhenFuncParam2 = ", LocalContPhenFuncParam2)
                                    # print ("ContPhenFuncParam3 = ", LocalContPhenFuncParam3)
                                    # print ("ContPhenFuncParam4 = ", LocalContPhenFuncParam4)
                                    # print ("ContPhenFuncParam5 = ", LocalContPhenFuncParam5)
                                    # print ("dbFileName = ", LocalDBFileName)
                                    # print ("NumModelPixelXX = ", LocalNumModelPixelXX)
                                    # print ("NumModelPixelYY = ", LocalNumModelPixelYY)
                                    # print ("LocalLocalOverlapFlag = ", LocalLocalOverlapFlag)
                                    # print ("LocalNoSubBeamFlag = ", LocalNoSubBeamFlag)
                                    # print ("LocalEmAbsPATH = ", LocalEmAbsPATH)


                                    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                    ## call myXCLASS function
                                    printflag = True
                                    modeldata, log, TransEnergies, IntOpt, JobDir = task_myXCLASS.myXCLASSCore(FreqMin = FreqMin, FreqMax = FreqMax, \
                                                                                                           FreqStep = FreqStep, \
                                                                                                           MolfitsFileName = LocalMolfitFileName, \
                                                                                                           iso_flag = self.IsoFlag, \
                                                                                                           IsoTableFileName = LocalIsoRatioFileName, \
                                                                                                           TelescopeSize = TelescopeSize, \
                                                                                                           BMIN = BMIN, BMAJ = BMAJ, BPA = BPA, \
                                                                                                           Inter_Flag = InterFlag, \
                                                                                                           vLSR = GlobalvLSR, Redshift = Redshift, \
                                                                                                           t_back_flag = t_back_flag, tBack = tBack, \
                                                                                                           tSlope = tSlope, nH_flag = nH_flag, \
                                                                                                           N_H = N_H, beta_dust = beta_dust, \
                                                                                                           kappa_1300 = kappa_1300, \
                                                                                                           DustFileName = DustFileName, \
                                                                                                           BackgroundFileName = BackgroundFileName, \
                                                                                                           ContPhenFuncID = LocalContPhenFuncID, \
                                                                                                           ContPhenFuncParam1 = LocalContPhenFuncParam1, \
                                                                                                           ContPhenFuncParam2 = LocalContPhenFuncParam2, \
                                                                                                           ContPhenFuncParam3 = LocalContPhenFuncParam3, \
                                                                                                           ContPhenFuncParam4 = LocalContPhenFuncParam4, \
                                                                                                           ContPhenFuncParam5 = LocalContPhenFuncParam5, \
                                                                                                           dbFileName = LocalDBFileName, \
                                                                                                           NumModelPixelXX = LocalNumModelPixelXX, \
                                                                                                           NumModelPixelYY = LocalNumModelPixelYY, \
                                                                                                           LocalOverlapFlag = LocalLocalOverlapFlag, \
                                                                                                           NoSubBeamFlag = LocalNoSubBeamFlag, \
                                                                                                           EmAbsPATH = LocalEmAbsPATH, \
                                                                                                           printFlag = printflag)
                                    ## clean up job directory for myXCLASS function call
                                    cmdString = "rm -rf " + JobDir
                                    os.system(cmdString)


                                    ## remove continuum from background spectrum
                                    if (tBack != 0.0):
                                        for FreqID, Freq in enumerate(modeldata[:, 0]):
                                            modeldata[FreqID, -1] = modeldata[FreqID, -1] - abs(tBack) * (Freq / FreqMin)**tSlope


                                    ## store background spectrum for current frequency range
                                    LocalSpectrumFileName = self.CurrentMoleculeDir + "background-spectrum___data-file_"
                                    LocalSpectrumFileName += str(ObsDataFileIndex + 1) + "__range_" + str(RangeIndex + 1) + ".dat"
                                    np.savetxt(LocalSpectrumFileName, modeldata)
                                    LocalBackgroundFileNameList.append([ObsDataFileIndex, RangeIndex, LocalSpectrumFileName])


                # Debug:
                # print ("LocalBackgroundFileNameList = ", LocalBackgroundFileNameList)


                ## update entry in obs. xml file
                self.BackgroundFileNameList = copy.deepcopy(LocalBackgroundFileNameList)


                ## update progress bar
                ProgressBarValue += 1
                ok = bar.setValue(ProgressBarValue)
                if (not ok):
                    print ("\\n\\nAbort")
                    self.lock = 0
                    return
                bar.close()


                ## we're done
                print ("\r\t                                                                         ", flush = True)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update GUI
        self.CompNumChangeFlag = True
        self.InitTab2Flag = True
        okStatus = self.Tab2NewXMLFile()
        self.Tab2setDisEnabled(True)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get emission / absorption function
    def Tab2GetEmAbsFunc(self, LocalEmAbsPATH):
        """

    input parameters:
    -----------------

        - LocalEmAbsPATH:               path containing ASCII files describing emission / absorption functions


    output parameters:
    ------------------

        - EmsAbsFileInterDict:          directory containing the interpolation function objects for each distance
        """

        # Debug:
        # print ("LocalEmAbsPATH = ", LocalEmAbsPATH)


        ## get name of folder corresponding to current molecule
        LocalMoleculeFileName = task_LineIdentification.MoleculeFileName(self.CurrentMolecule)
        LocalMoleculeFileName = LocalMoleculeFileName + "/"

        # Debug:
        # print ("LocalMoleculeFileName = ", LocalMoleculeFileName)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is only one molecule contained in current molfit file
        EmsAbsFileContentDir = {}
        ListOfPATHs = [LocalEmAbsPATH, LocalEmAbsPATH + "/" + LocalMoleculeFileName]
        for LocalPATH in ListOfPATHs:
            try:
                listing = os.listdir(LocalPATH)
            except:
                listing = None
                msgBox = QtWidgets.QMessageBox()
                msgBox.setWindowTitle("WARNING")
                LocalLogText = "Can not find emission and absorption files for current molecule.\n\n"
                LocalLogText += "Modeling of current molecule can be done only without ems. and abs. function!\n"
                msgBox.setText(LocalLogText)
                ret = msgBox.exec_()
            if (listing is not None):
                for LocalFileName in listing:
                    LowerLocalFileName = LocalFileName.lower()

                    # Debug:
                    # print ("LowerLocalFileName = ", LowerLocalFileName)


                    ## check, if emission / absorption files are included in the given directory
                    if (LowerLocalFileName.startswith("emission_absorption____")):
                        LocalEmAbsFileName = LocalPATH + "/" + LocalFileName


                        ## determine distance from file name
                        Distance = LowerLocalFileName.split("__")
                        Distance = Distance[-1]
                        Distance = Distance.replace(".dat", "")
                        Distance = Distance.replace(".gz", "")
                        DistanceNum = float(Distance)

                        # Debug:
                        # print ("DistanceNum = ", DistanceNum)


                        ## import content of ASCII file and store to python dictionary
                        LocalEmsAbsFileContent = np.loadtxt(LocalEmAbsFileName, skiprows = 3)
                        try:
                            l = EmsAbsFileContentDir[Distance]
                            EmsAbsFileContentDir[Distance] = np.concatenate((l, LocalEmsAbsFileContent), axis = 0)
                        except:
                            EmsAbsFileContentDir[Distance] = LocalEmsAbsFileContent


                ## exit loop?
                if (EmsAbsFileContentDir != {}):
                    break

        # Debug:
        # print ("EmsAbsFileContentDir = ", EmsAbsFileContentDir)
        # print ("EmsAbsFileContentDir.keys() = ", EmsAbsFileContentDir.keys())


        ## sort merged arrays
        EmsAbsFileInterDict = {}
        for LocalDistance in list(EmsAbsFileContentDir.keys()):
            LocalArray = EmsAbsFileContentDir[LocalDistance]
            LocalArray = LocalArray[LocalArray[:, 0].argsort()]

            # Debug:
            # np.savetxt(LocalDistance + ".dat", LocalArray)


            ## interpolate emission and absorption functions
            EmInterpolFunc = scipy.interpolate.interp1d(LocalArray[:, 0], LocalArray[:, 1], bounds_error = False, fill_value = "extrapolate")
            AbsInterpolFunc = scipy.interpolate.interp1d(LocalArray[:, 0], LocalArray[:, 2], bounds_error = False, fill_value = "extrapolate")
            EmsAbsFileInterDict[LocalDistance] = [EmInterpolFunc, AbsInterpolFunc]

        # Debug:
        # print ("EmsAbsFileInterDict = ", EmsAbsFileInterDict)


        ## return to GUI
        return EmsAbsFileInterDict
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for remove contiuum check box event
    def Tab2checkrmCont(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## is GUI locked
        if (self.lock == 1):
            return


        ## invert remove contiuum flag
        if (self.RemoveContinuumFlag):
            self.RemoveContinuumFlag = False
        else:
            self.RemoveContinuumFlag = True


        ## update GUI
        self.Tab2SetValuesForTab()


        # Debug:
        if (self.dbgFlag):
            print ("\ncheckrmCont:")
            print ("self.RemoveContinuumFlag = ", self.RemoveContinuumFlag)


        ## set current event
        self.LastEvent = "Tab2checkrmCont"


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for (de-)activate usage of isotopologues
    def Tab2checkUseIso(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## is GUI locked
        if (self.lock == 1):
            return


        ## set global iso ratio flag??
        if (self.UseIsoFlag):
            self.UseIsoFlag = False
        else:
            self.UseIsoFlag = True


        ## update history arrays
        self.Tab2UpdateHistoryArray()


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## modify list of transition window parameters to exclude isotopologues
        if (not self.UseIsoFlag):


            ## save full list of transition window parameters
            self.AllSpectraCopy = copy.deepcopy(self.AllSpectra)
            self.IndexSortConversionListCopy = copy.deepcopy(self.IndexSortConversionList)


            ## define list of isotopologues
            LocalIsotopologuelist = []
            for line in self.CurrentIsoParam:
                LocalIso = line[0].strip()
                LocalIsotopologuelist.append(LocalIso)

            # Debug:
            # print ("LocalIsotopologuelist = ", LocalIsotopologuelist)


            ## create new list of transition window parameters
            # self.AllSpectra[i]:   [LocalObsDataTransWindow, SmoothedData, DBParam, RangeParam, IntWithoutContinuum[In1:In2], RestFreq, \
            #                        NumMolecules, LocalMolecularParameterList]
            self.AllSpectra = []
            dbListForSorting = []
            AllSpectraIndex = (-1)
            for LocalElement in self.AllSpectraCopy:
                DBParam = LocalElement[2]
                LocalMoleculeName = DBParam[4].strip()

                # Debug:
                # print ("LocalMoleculeName = ", LocalMoleculeName)


                ## check if current molecule is not an isotopologue
                if (not LocalMoleculeName in LocalIsotopologuelist):


                    ## get other transition window parameters
                    LocalObsDataTransWindow = LocalElement[0]
                    SmoothedData = LocalElement[1]
                    RangeParam = LocalElement[3]
                    IntWithoutContinuum = LocalElement[4]
                    RestFreq = LocalElement[5]
                    NumMolecules = LocalElement[6]
                    LocalMolecularParameterList = LocalElement[7]


                    ## DBParam = [TransFreq, EinsteinA, ELow, gup, LocalMoleculeName]
                    TransFreq = DBParam[0]
                    EinsteinA = DBParam[1]
                    ELow = DBParam[2]
                    gup = DBParam[3]
                    gAm1 = EinsteinA * gup
                    if (gAm1 != 0.0):
                        gAm1 = 1.0 / gAm1
                    else:
                        gAm1 = 0.0
                    if (ELow != 0.0):
                        E2gAEm1 = (EinsteinA * gup) / (ELow**2)
                    else:
                        E2gAEm1 = 0.0


                    ## store all TFW parameters
                    self.AllSpectra.append([LocalObsDataTransWindow, SmoothedData, DBParam, RangeParam, IntWithoutContinuum, \
                                            RestFreq, NumMolecules, LocalMolecularParameterList])


                    ## define parameters for plot sorting
                    AllSpectraIndex += 1
                    dbListForSorting.append([AllSpectraIndex, TransFreq, ELow, gAm1, E2gAEm1, gup, EinsteinA, LocalMoleculeName])

            # Debug:
            # print ("self.AllSpectra = ", self.AllSpectra)
            # print ("len(self.AllSpectra) = ", len(self.AllSpectra))


            ## sort list of molecular parameters depending on user-defined sort criterium
            self.IndexSortConversionList = [[], [], [], []]
            for LocalSortIndex in range(4):
                LocalCopydbListForSorting = copy.deepcopy(dbListForSorting)
                self.IndexSortConversionList[LocalSortIndex] = copy.deepcopy(sorted(LocalCopydbListForSorting, key = lambda x:(x[LocalSortIndex + 1])))


            ## set constants
            MaxNumTransitionsSQL = self.SQLParamArray[1]
            self.TotalNumberSpectra = min(MaxNumTransitionsSQL, len(self.AllSpectra))
            self.MaxNumberSpectrum = min(self.MaxNumberSpectrumDefault, self.TotalNumberSpectra)
            self.SpectrumIndexList = list(range(0, self.MaxNumberSpectrum))
            self.ChangeSpectrumWindowFlag = True


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## restore original parameters
        elif (self.AllSpectraCopy is not None):
            self.AllSpectra = copy.deepcopy(self.AllSpectraCopy)
            self.IndexSortConversionList = copy.deepcopy(self.IndexSortConversionListCopy)


            ## set constants
            MaxNumTransitionsSQL = self.SQLParamArray[1]
            self.TotalNumberSpectra = min(MaxNumTransitionsSQL, len(self.AllSpectra))
            self.MaxNumberSpectrum = min(self.MaxNumberSpectrumDefault, self.TotalNumberSpectra)
            self.SpectrumIndexList = list(range(0, self.MaxNumberSpectrum))
            self.ChangeSpectrumWindowFlag = True


        ## update tab 2
        if (self.tabWidget.currentIndex() == 1):
            self.Tab2SetValuesForTab()


        # Debug:
        if (self.dbgFlag):
            print ("\ncheckUseIso:")
            print ("self.UseIsoFlag = ", self.UseIsoFlag)


        ## set current event
        self.LastEvent = "Tab2checkUseIso"


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for usage of combo box
    def Tab2comboSelected(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## is subroutine locked?
        if (self.lock == 1):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## combo box for molecules
        if (objectName == "comboBox_molecule"):


            ## disable elements of tab 2
            self.Tab2setDisEnabled(False)


            ## set max. number of plots
            self.MaxNumberSpectrum = self.MaxNumberSpectrumDefault
            self.InitTab2Flag = True


            ## store local molfit and local iso ratio file
            self.AllMoleculesMolfitFiles[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentMolfitFileContents)


            ## get index of selected molecule
            LastCurrentMoleculeID = self.CurrentMoleculeID
            self.CurrentMoleculeID = self.comboBox_molecule.currentIndex()
            self.CurrentMolecule = self.MoleculesInMolfitFile[self.CurrentMoleculeID]
            self.CurrentMolfitIndex = 1
            self.CurrentIsoID = 0

            # Debug:
            # print (">---self.CurrentMoleculeID = ", self.CurrentMoleculeID)
            # print ("self.CurrentMolecule = ", self.CurrentMolecule)
            # print ("self.MoleculesInMolfitFile = ", self.MoleculesInMolfitFile)


            ## prepare subdirectories for current molecule
            OldBackgroundFileNameList = self.BackgroundFileNameList
            LocalMoleculeFileName = task_LineIdentification.MoleculeFileName(self.CurrentMolecule)
            self.CurrentMoleculeDir = self.GUIJobDir + LocalMoleculeFileName + "/"
            if (not os.path.exists(self.CurrentMoleculeDir)):
                cmdString = "rm -rf " + self.CurrentMoleculeDir + "; mkdir " + self.CurrentMoleculeDir
                os.system(cmdString)


            ## check, if a background file is defined in the already existing obs. xml file
            else:
                LocalObsXMLFile = self.CurrentMoleculeDir + "obs.xml"
                if (os.path.isfile(LocalObsXMLFile)):
                    self.BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFile, "BackgroundFileName")


            ## update GUI
            self.CompNumChangeFlag = True
            self.InitTab2Flag = True
            okStatus = self.Tab2NewXMLFile()
            self.BackgroundFileNameList = OldBackgroundFileNameList
            if (str(okStatus) == "1"):
                self.CurrentMoleculeID = LastCurrentMoleculeID
                self.CurrentMolecule = self.MoleculesInMolfitFile[self.CurrentMoleculeID]
                self.CurrentMolfitIndex = 1
                self.CurrentIsoID = 0
                self.comboBox_molecule.setCurrentIndex(self.CurrentMoleculeID)
                self.ObsXMLFileChangeFlag = True
            self.Tab2setDisEnabled(True)


            ## update history arrays
            self.Tab2UpdateHistoryArray()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## combo box for isotopologues
        elif (objectName == "comboBox_iso"):


            ## get index of selected isotopologue
            self.CurrentIsoID = max(0, self.comboBox_iso.currentIndex())


            ## update GUI
            self.Tab2SetValuesForTab()

            # Debug:
            if (self.dbgFlag):
                print ("\nTab2comboSelected:")
                print ("self.CurrentIsoID = ", self.CurrentIsoID)


            ## set current event
            self.LastEvent = "Tab2comboSelected:"


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for slider events
    def Tab2SliderValueChange(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## is GUI locked
        if (self.lock == 1):
            return


        ## get name of slider
        sender = self.sender()


        ## determine parameter index
        objectName = sender.objectName()                                                    ## get name of object
        if (objectName == "Slider__ss"):
            ParamNumber = 0
            ParamName = "ss"
        elif (objectName == "Slider__Temp"):
            ParamNumber = 1
            ParamName = "Temp"
        elif (objectName == "Slider__Ntot"):
            ParamNumber = 2
            ParamName = "Ntot"
        elif (objectName == "Slider__vwidth"):
            ParamNumber = 3
            ParamName = "vwidth"
        elif (objectName == "Slider__voff"):
            ParamNumber = 4
            ParamName = "voff"
        elif (objectName == "Slider__IsoRatio"):
            ParamNumber = 5
            ParamName = "IsoRatio"


        ## convert slider value to parameter value
        OldValue = sender.value()                                                           ## get value from slider
        if (ParamNumber < 5):
            LowLimit = self.CurrentMolfitParameter[self.CurrentComponentIndex][ParamNumber][1]
            UpLimit = self.CurrentMolfitParameter[self.CurrentComponentIndex][ParamNumber][2]
        else:
            LowLimit = self.CurrentIsoParam[self.CurrentIsoID][3]
            UpLimit = self.CurrentIsoParam[self.CurrentIsoID][4]
        LocalInterval = 1
        OldMin = 0
        OldMax = 100
        NewMin = LowLimit
        NewMax = UpLimit
        OldRange = (OldMax - OldMin)
        NewRange = (NewMax - NewMin)
        val = (((OldValue - OldMin) * NewRange) / OldRange) + NewMin
        if (ParamNumber < 5):
            OldParamValue = self.CurrentMolfitParameter[self.CurrentComponentIndex][ParamNumber][3]
            if (OldParamValue != val):
                self.CurrentMolfitParameter[self.CurrentComponentIndex][ParamNumber][3] = val
                self.Tab2UpdateHistoryArray()                                               ## update history arrays
        else:
            OldParamValue = self.CurrentIsoParam[self.CurrentIsoID][2]
            if (OldParamValue != val):
                self.CurrentIsoParam[self.CurrentIsoID][2] = val
                self.Tab2UpdateHistoryArray()                                               ## update history arrays

        # Debug:
        if (self.dbgFlag):
            print ("\nSliderValueChange:")
            print ("val = ", val)
            print ("ParamName = ", ParamName)
            print ("ParamNumber = ", ParamNumber)


        ## set title of GUI window
        self.GUITitle = "XCLASS-GUI*"
        if (self.tabWidget.currentIndex() == 1):
            self.GUITitle += "   ({:d}/{:d})".format(self.CurrentMoleculeID + 1, len(self.MoleculesInMolfitFile))
        self.setWindowTitle(self.GUITitle)


        ## update GUI
        self.Tab2SetValuesForTab()


        ## set current event
        self.LastEvent = "Tab2SliderValueChange"


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for text edit events
    def Tab2EditSelected(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## is GUI locked?
        if (self.lock == 1):
            return


        ## get name of edit box and the corresponding value
        sender = self.sender()
        objectName = str(sender.objectName())                                           ## get name of object
        val = str(sender.text())
        self.CurrCursorPos = sender.cursorPosition()

        # Debug:
        if (self.dbgFlag):
            print ("self.CurrCursorPos = ", self.CurrCursorPos)


        ## check for enter character
        ValString = val
        UpdateFlag = False
        if (val.find("\n") > (-1)):
            val = val.replace("\n", "")
            UpdateFlag = True


        ## check, if value is number
        IsNumber = True
        try:
            i = int(val)
        except ValueError:
            try:
                i = float(val)
            except ValueError:
                IsNumber = False
        if (IsNumber):                                                                      ## continue if input is a number
            val = float(val)


            ## get column of text edit
            ColumnFlag = ""
            if (objectName.find("_Low_") > (-1)):
                ColumnFlag = "Low"
                ColumnFlagIndex = 1
            elif (objectName.find("_Up_") > (-1)):
                ColumnFlag = "Up"
                ColumnFlagIndex = 2
            elif (objectName.find("_Val_") > (-1)):
                ColumnFlag = "Val"
                ColumnFlagIndex = 3


            ## determine parameter index
            if (objectName.find("_ss") > (-1)):
                ParamNumber = 0
                ParamName = "ss"
            elif (objectName.find("_Temp") > (-1)):
                ParamNumber = 1
                ParamName = "Temp"
            elif (objectName.find("_Ntot") > (-1)):
                ParamNumber = 2
                ParamName = "Ntot"
            elif (objectName.find("_vwidth") > (-1)):
                ParamNumber = 3
                ParamName = "vwidth"
            elif (objectName.find("_voff") > (-1)):
                ParamNumber = 4
                ParamName = "voff"
            elif (objectName.find("_Distance") > (-1)):
                ParamNumber = 5
                ParamName = "Distance"
                ColumnFlagIndex = 0
            elif (objectName.find("_IsoRatio") > (-1)):
                ParamNumber = 6
                ParamName = "IsoRatio"


            ## set new parameter
            if (ParamNumber < 6):
                OldParamValue = self.CurrentMolfitParameter[self.CurrentComponentIndex][ParamNumber][ColumnFlagIndex]
                if (OldParamValue != val):
                    self.CurrentMolfitParameter[self.CurrentComponentIndex][ParamNumber][ColumnFlagIndex] = val
                    self.Tab2UpdateHistoryArray()                                           ## update history arrays

                # Debug:
                # print ("\nOldParamValue = ", OldParamValue)
                # print ("val = ", val)
                # print ("OldParamValue != val = ", (OldParamValue != val))

            else:
                if (ColumnFlagIndex == 1):
                    OldParamValue = self.CurrentIsoParam[self.CurrentIsoID][3]
                    if (OldParamValue != val):
                        self.CurrentIsoParam[self.CurrentIsoID][3] = val
                        self.Tab2UpdateHistoryArray()                                       ## update history arrays

                elif (ColumnFlagIndex == 2):
                    OldParamValue = self.CurrentIsoParam[self.CurrentIsoID][4]
                    if (OldParamValue != val):
                        self.CurrentIsoParam[self.CurrentIsoID][4] = val
                        self.Tab2UpdateHistoryArray()                                       ## update history arrays

                elif (ColumnFlagIndex == 3):
                    OldParamValue = self.CurrentIsoParam[self.CurrentIsoID][2]
                    if (OldParamValue != val):
                        self.CurrentIsoParam[self.CurrentIsoID][2] = val
                        self.Tab2UpdateHistoryArray()                                       ## update history arrays

                    # Debug:
                    # print ("\nOldParamValue = ", OldParamValue)
                    # print ("val = ", val)

            # Debug:
            if (self.dbgFlag):
                print ("\nEditSelected:")
                print ("objectName = ", objectName)
                print ("ColumnFlag = ", ColumnFlag)
                print ("ParamNumber = ", ParamNumber)
                print ("ParamName = ", ParamName)
                print ("IsNumber = ", IsNumber)
                print ("val = ", val)


            ## set title of GUI window
            self.GUITitle = "XCLASS-GUI*"
            if (self.tabWidget.currentIndex() == 1):
                self.GUITitle += "   ({:d}/{:d})".format(self.CurrentMoleculeID + 1, len(self.MoleculesInMolfitFile))
            self.setWindowTitle(self.GUITitle)


            ## update GUI
            self.Tab2SetValuesForTab()


            ## set text cursor
            #if (self.CurrCursorPos == len(ValString) and (ParamNumber == 2 or ParamNumber == 5)):
            #    cursor.setPosition(self.CurrCursorPos + 1)
            #else:
            #    cursor.setPosition(self.CurrCursorPos)
            #sender.setTextCursor(cursor)


        ## set current event
        self.LastEvent = objectName


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get iso ratios from file
    def Tab2GetIsoRatiosFromFile(self, LocalMoleculeName, IsoTableFileName):
        """

    input parameters:
    -----------------

        - LocalMoleculeName:        name of current molecule

        - IsoTableFileName:         path and name of iso ratio file


    output parameters:
    ------------------

        - ListLocalIsotopologue:    iso ratios from file
        """

        # Debug:
        # print ("LocalMoleculeName = ", LocalMoleculeName)
        # print ("IsoTableFileName = ", IsoTableFileName)


        ## analyze iso ratio file
        if (type(IsoTableFileName) is str):
            NewLocalIsoRatioFileName = ""
            IsoRatioTable, Isotopologues, IsoMolecule = task_myXCLASS.ImportIsoRatioFile(IsoTableFileName, NewLocalIsoRatioFileName)
        else:
            IsoRatioTable = IsoTableFileName

        # Debug:
        # print ("IsoRatioTable = ", IsoRatioTable)


        ## search for globally defined iso ratios
        GlobalList = []
        for IsoIndex in range(len(IsoRatioTable)):
            IsoRatioTableLine = IsoRatioTable[IsoIndex]
            LocalIsotopologue = IsoRatioTableLine[0].strip()
            if (LocalIsotopologue.lower().find("global") > (-1)):
                LocalIsotopologue = LocalIsotopologue.replace("Global", "")
                LocalIsotopologue = LocalIsotopologue.replace("GLOBAL", "")
                LocalIsotopologue = LocalIsotopologue.replace("global", "")
                LocalIsotopologue = LocalIsotopologue.replace("___", "")
                LocalIsotopologue = LocalIsotopologue.replace("__", "")
                GlobalList.append([LocalIsotopologue, IsoIndex])

        # Debug:
        # print ("GlobalList = ", GlobalList)


        ## read in iso ratio file
        ListLocalIsotopologue = []
        for IsoIndex in range(len(IsoRatioTable)):
            IsoRatioTableLine = IsoRatioTable[IsoIndex]
            LocalIsotopologue = IsoRatioTableLine[0].strip()
            IsoMaster = IsoRatioTableLine[1].strip()
            if (IsoMaster == LocalMoleculeName or LocalIsotopologue.lower().find("global") > (-1)):
                RatioValue = float(IsoRatioTableLine[2])
                LocalFitFlag = False
                if (len(IsoRatioTableLine) == 3):
                    LowerLimitIsoRatio = 1.0
                    UpperLimitIsoRatio = 1.0
                    LocalFitFlag = False
                else:
                    l1 = float(IsoRatioTableLine[3])
                    l2 = float(IsoRatioTableLine[4])
                    LowerLimitIsoRatio = min(l1, l2)
                    UpperLimitIsoRatio = max(l1, l2)
                    if (LowerLimitIsoRatio != UpperLimitIsoRatio):
                        LocalFitFlag = True


                ## for not globally defined ratios, search for globally defined isotopologues
                ListIndex = []
                if (LocalIsotopologue.lower().find("global") == (-1) and GlobalList != []):
                    for GlobalElement in GlobalList:
                        GlobalIso = GlobalElement[0]
                        GlobalIndex = GlobalElement[1]
                        if (LocalIsotopologue.find(GlobalIso) > (-1)):
                            Multiplicator = LocalIsotopologue.count(GlobalIso)
                            ListIndex.append([GlobalIndex, Multiplicator])
                ListLocalIsotopologue.append([LocalIsotopologue, IsoMaster, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag, ListIndex])


        ## return to GUI
        return ListLocalIsotopologue
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set current iso index for iso combo box
    def Tab2SetCurrIsoIndex(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## set current iso index for iso combo box
        if (self.lock == 1):
            AlreadyLockedFlag = True
        else:
            self.lock = 1
            AlreadyLockedFlag = False
        self.comboBox_iso.clear()
        Isotopologuelist = []
        GlobalIsoEntryCounter = 0
        for line in self.CurrentIsoParam:
            LocalIso = line[0].strip()
            Isotopologuelist.append(LocalIso)
            LocalIso = LocalIso.lower()
            if (LocalIso.find("global_") > (-1)):
                GlobalIsoEntryCounter += 1
        if (self.CurrentIsoParam != []):
            self.comboBox_iso.addItems(Isotopologuelist)
            self.comboBox_iso.setCurrentIndex(self.CurrentIsoID)
        if (not AlreadyLockedFlag):
            self.lock = 0

        # Debug:
        # print ("\n\nself.CurrentIsoID = ", self.CurrentIsoID)
        # print ("self.CurrentIsoParam = ", self.CurrentIsoParam)
        # print ("self.UseIsoFlag = ", self.UseIsoFlag)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##====================================================================================================================================================
    ##
    ##  subroutines for tab 3
    ##
    ##====================================================================================================================================================


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what happens if tab 3 is selected
    def Tab3ChangeTab(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if parameter already defined
        if (self.FreqMinList == [] or self.CurrentMolfitParameter == []):
            self.tabWidget.setCurrentIndex(0)
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## lock elements of tab 3
        self.LockTab3 = True


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## replot element of tab 2
        self.PlotFlagTab3 = True


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## import obs. data from file
        if (self.ObsDataList == []):
            NumExpFiles = len(self.ExpFileList)
            for ExpDataFileIndex in range(NumExpFiles):
                LocalExpFileName = self.ExpFileList[ExpDataFileIndex][1]
                NumberHeaderLines = int(self.NumberHeaderLinesList[ExpDataFileIndex][1])
                SeparatorColumns = self.SeparatorColumnsList[ExpDataFileIndex][1]
                if (self.FITSFlag):
                    NewObsDataFile = self.FITSSubDir + "FitsSpectrum__" + str(ExpDataFileIndex + 1) + ".dat"
                    ImportExpData = np.loadtxt(NewObsDataFile, skiprows = 3)
                else:
                    if (SeparatorColumns.strip() == ""):
                        ImportExpData = np.loadtxt(LocalExpFileName, skiprows = NumberHeaderLines)
                    else:
                        ImportExpData = np.loadtxt(LocalExpFileName, skiprows = NumberHeaderLines, delimiter = SeparatorColumns)
                self.ObsDataList.append(ImportExpData)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine all frequency ranges for the current obs. data file
        self.CurrentFreqRangeID = 0
        self.CurrentRangeList = ["All ranges"]


        ## get number of freq. ranges from obs. xml file
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(self.CurrentObsDataID, RangeIndex, NumberRangeListIn = self.NumberExpRangesList)
        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']            ## get number of frequency ranges for current obs. data file


        ## get parameters for current frequency range
        for RangeIndex in range(NumberFrequencyRanges):                                     ## loop over all range definitions in the whole xml file


            ## get parameter for current frequency range
            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.CurrentObsDataID, RangeIndex, FreqMinListIn = self.FreqMinList, \
                                                                            FreqMaxListIn = self.FreqMaxList)
            FreqMin = ObsXMLParameterDictRange['FreqMin']
            if (FreqMin is not None):
                FreqMax = ObsXMLParameterDictRange['FreqMax']
                self.CurrentRangeList.append(str(FreqMin) + " - " + str(FreqMax) + " MHz")
        self.comboBox_FreqRange_Tab3.clear()
        self.comboBox_FreqRange_Tab3.addItems(self.CurrentRangeList)
        self.comboBox_FreqRange_Tab3.setCurrentIndex(self.CurrentFreqRangeID)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine list of isotopologues for the current molecule
        if (self.IsoFlag):
            self.AllIsotopologueList[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentIsoParam)
            self.AllMolCurrentIsoID = 0
            self.LocalIsotopologuelist = ["All"]
            for line in self.CurrentIsoParam:
                self.LocalIsotopologuelist.append(line[0].strip())
            self.comboBox_Isotopologues_Tab3.clear()
            if (self.ShowAllFlag):
                self.comboBox_Isotopologues_Tab3.addItems([])
            else:
                self.comboBox_Isotopologues_Tab3.addItems(self.LocalIsotopologuelist)
                self.comboBox_Isotopologues_Tab3.setCurrentIndex(self.AllMolCurrentIsoID)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store current molfit parameters
        self.Tab2CreateMolfitFile()                                                         ## update current line of the molfit file
        LocalMolInMolfitFile, LocalMolParam, LocalMolfitForEachMol = task_myXCLASS.AnalyzeMolfitFile(self.CurrentMolfitFileContents)
        NewRangeFlag = False
        LocalMolfitParameters = self.Tab2GetParameterFromMolfitFile(LocalMolParam[0], NewRangeFlag)
        self.MolfitFileAllParameters[self.CurrentMoleculeID] = copy.deepcopy(LocalMolfitParameters)
        ii = 0
        for LocalMolfitMolIndex in range(self.CurrentMoleculeID):
            ii += len(self.MolfitFileAllParameters[LocalMolfitMolIndex])
        self.CurrentAllMolfitIndex = ii + self.CurrentComponentIndex + 1

        # Debug:
        # print ("self.CurrentAllMolfitIndex = ", self.CurrentAllMolfitIndex)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set values for tab 3
        self.InitFlagTab3 = True
        self.Tab3SetValuesForTab()
        self.InitFlagTab3 = False


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## enables or disables elements of tab 3
    def Tab3setDisEnabled(self, DisEnabledFlag):
        """

    input parameters:
    -----------------

        - DisEnabledFlag:       flag indicating if elements are enabled (True) or disabled (False)


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("DisEnabledFlag = ", DisEnabledFlag)


        self.label_FreqRange_Tab3.setEnabled(DisEnabledFlag)
        self.label_ObsDataFile_Tab3.setEnabled(DisEnabledFlag)
        self.label_Isotopologues_Tab3.setEnabled(DisEnabledFlag)
        self.comboBox_ObsDataFile_Tab3.clear()
        self.comboBox_FreqRange_Tab3.clear()
        self.comboBox_Isotopologues_Tab3.clear()
        self.comboBox_ObsDataFile_Tab3.setEnabled(DisEnabledFlag)
        self.comboBox_FreqRange_Tab3.setEnabled(DisEnabledFlag)
        self.comboBox_Isotopologues_Tab3.setEnabled(False)
        self.checkBox_ShowAll_Tab3.setEnabled(DisEnabledFlag)
        self.pushButton_SMF_Tab3.setEnabled(DisEnabledFlag)
        self.pushButton_FitAll_Tab3.setEnabled(DisEnabledFlag)
        self.pushButton_Cancel_Tab3.setEnabled(DisEnabledFlag)
        self.pushButton_Quit_Tab3.setEnabled(DisEnabledFlag)
        if (DisEnabledFlag):


            ## add names of all obs. data files to combo box and select current obs. data file
            LocalFileNameList = []
            for LocalObsFile in self.ExpFileList:
                LocalPath, LocalFileName = os.path.split(LocalObsFile[1])
                LocalFileNameList.append(LocalFileName)
            self.comboBox_ObsDataFile_Tab3.addItems(LocalFileNameList)
            self.comboBox_ObsDataFile_Tab3.setCurrentIndex(self.CurrentObsDataID)


            ## add names of all frequency ranges to combo box and select current frequency range
            self.comboBox_FreqRange_Tab3.addItems(self.CurrentRangeList)
            self.comboBox_FreqRange_Tab3.setCurrentIndex(self.CurrentFreqRangeID)


            ## add names of isotopolgues to combo box and select current isotopologue
            if (self.IsoFlag and not self.ShowAllFlag):
                self.comboBox_Isotopologues_Tab3.setEnabled(DisEnabledFlag)
                self.comboBox_Isotopologues_Tab3.addItems(self.LocalIsotopologuelist)
                self.comboBox_Isotopologues_Tab3.setCurrentIndex(self.AllMolCurrentIsoID)
            else:
                self.comboBox_Isotopologues_Tab3.addItems([])
        else:
            self.comboBox_ObsDataFile_Tab3.addItems([])
            self.comboBox_FreqRange_Tab3.addItems([])
            self.comboBox_Isotopologues_Tab3.addItems([])
        if (DisEnabledFlag):
            if (self.ShowAllFlag):
                self.list_molfit_Tab3.setEnabled(False)
            else:
                self.list_molfit_Tab3.setEnabled(DisEnabledFlag)
                self.list_molfit_Tab3.setCurrentRow(self.CurrentAllMolfitIndex)
        else:
            self.list_molfit_Tab3.setEnabled(DisEnabledFlag)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set all values in tab 3
    def Tab3SetValuesForTab(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## lock elements of tab 3
        self.LockTab3 = True


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## disable elements of tab 3
        self.Tab3setDisEnabled(False)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## save current molfit and iso ratio file


        ## save current molfit file
        OldMolfitFileName = self.CurrentMolfitFileName
        self.CurrentMolfitFileName = self.GUIJobDir + "myXCLASS.molfit"
        LocalMolfitFileName = self.CurrentMolfitFileName
        self.MenuSaveCurrMolfitFile()
        MolfitFile = open(self.CurrentMolfitFileName)
        self.AllMolfitFileContent = MolfitFile.readlines()
        MolfitFile.close()
        self.CurrentMolfitFileName = OldMolfitFileName


        ## write contents of molfit file to list box
        self.list_molfit_Tab3.clear()
        for LocalLineIndex, LocalLine in enumerate(self.AllMolfitFileContent):
            StrippedLine = LocalLine.strip()
            self.list_molfit_Tab3.addItem(LocalLine.replace("\n", ""))
        self.list_molfit_Tab3.setCurrentRow(self.CurrentAllMolfitIndex)


        ## save current iso ratio file
        IsoRatioFileContent = []
        IsoTableFileName = ""
        if (self.IsoFlag):
            OldIsoRatioFileName = self.IsoRatioFileName
            self.IsoRatioFileName = self.GUIJobDir + "myXCLASS_iso-ratio.dat"
            IsoTableFileName = self.IsoRatioFileName
            self.MenuSaveIsoRatioFile()
            IsoRatioFile = open(self.IsoRatioFileName)
            IsoRatioFileContent = IsoRatioFile.readlines()
            IsoRatioFile.close()
            self.IsoRatioFileName = OldIsoRatioFileName

        # Debug:
        # print ("IsoTableFileName = ", IsoTableFileName)


        ## get path and name of database file
        LocalDBFileName = ""
        if (self.dbList != []):
            LocalDBFileName = self.dbList[0]

        # Debug:
        # print ("LocalDBFileName = ", LocalDBFileName)


        ## get number of pixels
        LocalNumModelPixelXX = None
        if (self.NumModelPixelXXList != []):
            LocalNumModelPixelXX = self.NumModelPixelXXList[0]
        LocalNumModelPixelYY = None
        if (self.NumModelPixelYYList != []):
            LocalNumModelPixelYY = self.NumModelPixelYYList[0]
        LocalLocalOverlapFlag = None
        if (self.LocalOverlapFlagList != []):
            LocalLocalOverlapFlag = self.LocalOverlapFlagList[0]
        LocalNoSubBeamFlag = None
        if (self.NoSubBeamFlagList != []):
            LocalNoSubBeamFlag = self.NoSubBeamFlagList[0]
        LocalEmAbsPATH = None
        if (self.EmAbsPATHList != []):
            LocalEmAbsPATH = self.EmAbsPATHList[0]

        # Debug:
        # print ("LocalNumModelPixelXX = ", LocalNumModelPixelXX)
        # print ("LocalNumModelPixelYY = ", LocalNumModelPixelYY)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if myXCLASS function has to be called


        ## compare current molfit and iso ratio file with last molfit and iso ratio file, if they differ call myXCLASS function to determine
        if (not self.CallmyXCLASSFunctionFlag):
            if (len(self.LastMolfitFile) != len(self.AllMolfitFileContent) or len(self.LastIsoRatioFileContent) != len(IsoRatioFileContent)):
                self.CallmyXCLASSFunctionFlag = True

                # Debug:
                # print ("\n\nself.CallmyXCLASSFunctionFlag = ", self.CallmyXCLASSFunctionFlag)
                # print ("len(self.LastMolfitFile), len(self.AllMolfitFileContent) = ", len(self.LastMolfitFile), len(self.AllMolfitFileContent))
                # print ("len(self.LastIsoRatioFileContent), len(IsoRatioFileContent) = ", len(self.LastIsoRatioFileContent), len(IsoRatioFileContent))

            else:
                #    for LocalLineIndex, LocalLine in enumerate(self.LastMolfitFile):            ## molfit files are not identical
                #        if (LocalLine.strip().replace(" ", "") != self.AllMolfitFileContent[LocalLineIndex].strip().replace(" ", "")):
                if (self.LastMolfitFileAllParameters != self.MolfitFileAllParameters or self.LastMoleculesInMolfitFile != self.MoleculesInMolfitFile):
                    self.CallmyXCLASSFunctionFlag = True

                    # Debug:
                    # print ("\n\nself.CallmyXCLASSFunctionFlag = ", self.CallmyXCLASSFunctionFlag)
                    # print ("self.LastMolfitFileAllParameters = ", self.LastMolfitFileAllParameters)
                    # print ("self.MolfitFileAllParameters = ", self.MolfitFileAllParameters)
                    # print ("\nself.LastMoleculesInMolfitFile = ", self.LastMoleculesInMolfitFile)
                    # print ("self.MoleculesInMolfitFile = ", self.MoleculesInMolfitFile)


                if (not self.CallmyXCLASSFunctionFlag):
                    for LocalLineIndex, LocalLine in enumerate(self.LastIsoRatioFileContent):
                        if (LocalLine.strip() != IsoRatioFileContent[LocalLineIndex].strip()):
                            self.CallmyXCLASSFunctionFlag = True

                            # Debug:
                            # print ("\n\nself.CallmyXCLASSFunctionFlag = ", self.CallmyXCLASSFunctionFlag)
                            # print ("LocalLine.strip() = ", LocalLine.strip())
                            # print ("IsoRatioFileContent[LocalLineIndex] = ", IsoRatioFileContent[LocalLineIndex])

                            break


        ## check if current frequency range is calculated before
        ObsRangeID = [self.CurrentObsDataID, self.CurrentFreqRangeID]
        if (not ObsRangeID in self.AlreadyCalculatedRanges):

            # Debug:
            # print ("\n\nself.CallmyXCLASSFunctionFlag = ", self.CallmyXCLASSFunctionFlag)
            # print ("self.AlreadyCalculatedRanges = ", self.AlreadyCalculatedRanges)
            # print ("ObsRangeID = ", ObsRangeID)

            if (len(self.CurrentRangeList) == 2):
                ObsRangeID = [self.CurrentObsDataID, (1 - self.CurrentFreqRangeID)]
                if (not ObsRangeID in self.AlreadyCalculatedRanges):
                    self.CallmyXCLASSFunctionFlag = True
                    self.AlreadyCalculatedRanges.append(ObsRangeID)
            else:
                self.CallmyXCLASSFunctionFlag = True
                self.AlreadyCalculatedRanges.append(ObsRangeID)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## calculate myXCLASS function
        NumExpFiles = len(self.ExpFileList)                                                 ## get number of obs. data files
        FinalTransitionArray = []                                                           ## reset list of transition energies
        MolFreqName = []                                                                    ## reset list of molecule names
        if (self.CallmyXCLASSFunctionFlag):                                                 ## call myXCLASS function
            self.myXCLASSSpectraList = []
            self.AllMinFreq = 1.e99
            self.AllMaxFreq = -1.e99
            self.AllTransFreqList = []
            self.AllMolNameList = []
            self.AllTransParamList = []

            # Debug:
            # print ("\n\nself.CurrentObsDataID = ", self.CurrentObsDataID)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## determine number of myXCLASS calls
            NumCalls = 0
            LocalFreqCounter = (-1)                                                         ## reset counter for freq. range
            for ObsDataFileIndex in range(NumExpFiles):                                     ## loop over all spectra
                if (self.CurrentObsDataID == (-1) or self.CurrentObsDataID == ObsDataFileIndex):


                    ## get number of freq. ranges from obs. xml file
                    RangeIndex = (-1)
                    ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                   NumberRangeListIn = self.NumberExpRangesList)
                    NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges'] ## get number of frequency ranges for current obs. data file


                    ## get parameters for current frequency range
                    for RangeIndex in range(NumberFrequencyRanges):                         ## loop over all range definitions in the whole xml file
                        LocalFreqCounter += 1                                               ## increase counter for freq. range
                        if ((self.CurrentFreqRangeID - 1) != (-1) or (self.CurrentFreqRangeID - 1) != LocalFreqCounter):
                            NumCalls += 1


            ## call progress bar dialog
            NumCalls += 2
            bar = ProgressBar(total = NumCalls, widgetTitle = "Calculate spectra for selected frequency ranges", \
                              bodyText = "Call myXCLASS to calculate spectra ..")
            bar.resize(500, 100)
            desktopSize = QtWidgets.QDesktopWidget().screenGeometry()
            top = int((desktopSize.height() / 2.0) - 50)
            left = int((desktopSize.width() / 2.0) - 250)
            bar.move(left, top)
            bar.show()


            ## update progress bar
            ProgressBarValue = 1
            ok = bar.setValue(ProgressBarValue)
            if (not ok):
                print ("\\n\\nAbort")
                self.lock = 0
                return

            # Debug:
            # print ("\nNumExpFiles = ", NumExpFiles)
            # print ("self.FreqMinList = ", self.FreqMinList)
            # print ("self.FreqMaxList = ", self.FreqMaxList)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## get calculation parameters from obs. xml file
            self.CallmyXCLASSFunctionFlag = False                                           ## do not call myXCLASS function for next time
            LocalFreqCounter = (-1)                                                         ## reset counter for freq. range
            for ObsDataFileIndex in range(NumExpFiles):                                     ## loop over all spectra
                if (self.CurrentObsDataID == (-1) or self.CurrentObsDataID == ObsDataFileIndex):


                    ## get number of freq. ranges etc. from obs. xml file
                    RangeIndex = (-1)
                    ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                   NumberRangeListIn = self.NumberExpRangesList, \
                                                                                   TelescopeSizeListIn = self.TelescopeSizeList, \
                                                                                   BMIN_ListIn = self.BMINList, \
                                                                                   BMAJ_ListIn = self.BMAJList, \
                                                                                   BPA_ListIn = self.BPAList, \
                                                                                   InterFlagListIn = self.Inter_FlagList, \
                                                                                   GlobalvLSRListIn = self.GlobalvLSRList, \
                                                                                   Redshift_ListIn = self.RedshiftList)
                    NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
                    TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
                    if (TelescopeSize is None):
                        TelescopeSize = 0.0
                    BMIN = ObsXMLParameterDictFile['BMIN']
                    BMAJ = ObsXMLParameterDictFile['BMAJ']
                    BPA = ObsXMLParameterDictFile['BPA']
                    InterFlag = ObsXMLParameterDictFile['InterFlag']
                    if (InterFlag is None):
                        InterFlag = False
                    GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
                    if (GlobalvLSR is None):
                        GlobalvLSR = 0.0
                    Redshift = ObsXMLParameterDictFile['Redshift']
                    if (Redshift is None):
                        Redshift = 0.0


                    ## determine new frequency range definitions and write to xml file
                    TotalNumberFreqRanges = 0
                    for RangeIndex in range(NumberFrequencyRanges):                         ## loop over all frequency ranges
                        LocalFreqCounter += 1
                        if ((self.CurrentFreqRangeID - 1) == (-1) or (self.CurrentFreqRangeID - 1) == LocalFreqCounter):


                            ## get parameter for current frequency range
                            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                            FreqMinListIn = self.FreqMinList, \
                                                                                            FreqMaxListIn = self.FreqMaxList, \
                                                                                            FreqStepListIn = self.FreqStepList, \
                                                                                            tBackFlagListIn = self.t_back_flagList, \
                                                                                            tBackListIn = self.tBackList, \
                                                                                            tSlopeListIn = self.tSlopeList, \
                                                                                            N_HListIn = self.N_HList, \
                                                                                            beta_dustListIn = self.beta_dustList, \
                                                                                            kappa_1300ListIn = self.kappa_1300List, \
                                                                                            DustFileNameListIn = self.DustFileNameList, \
                                                                                            BackgroundFileNameListIn = self.BackgroundFileNameList, \
                                                                                            ContPhenFuncID_ListIn = self.ContFuncIDPhenList, \
                                                                                            ContPhenFuncParam1_ListIn = self.ContParam1PhenList, \
                                                                                            ContPhenFuncParam2_ListIn = self.ContParam2PhenList, \
                                                                                            ContPhenFuncParam3_ListIn = self.ContParam3PhenList, \
                                                                                            ContPhenFuncParam4_ListIn = self.ContParam4PhenList, \
                                                                                            ContPhenFuncParam5_ListIn = self.ContParam5PhenList)
                            FreqMin = ObsXMLParameterDictRange['FreqMin']
                            if (FreqMin is not None):
                                FreqMax = ObsXMLParameterDictRange['FreqMax']
                                FreqStep = ObsXMLParameterDictRange['FreqStep']
                                t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                                t_back_flag = task_myXCLASSMapFit.CheckBool(t_back_flag)
                                tBack = ObsXMLParameterDictRange['tBack']
                                tSlope = ObsXMLParameterDictRange['tSlope']
                                nH_flag = True
                                N_H = ObsXMLParameterDictRange['N_H']
                                beta_dust = ObsXMLParameterDictRange['beta_dust']
                                kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                                DustFileName = ObsXMLParameterDictRange['DustFileName']
                                BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']


                                ## determine lowest and highest frequency
                                if (FreqMin < self.AllMinFreq):                             ## determine lowest frequency
                                    self.AllMinFreq = FreqMin
                                if (FreqMax > self.AllMaxFreq):                             ## determine highest frequency
                                    self.AllMaxFreq = FreqMax


                                ## check background parameters
                                if ((not t_back_flag) or tBack is None or tSlope is None):
                                    tBack = 0.0                                             ## set background temperature
                                    tSlope = 0.0                                            ## set temperature slope


                                ## check dust parameters
                                nH_flag = True
                                if (N_H is None or beta_dust is None or kappa_1300 is None):
                                    nH_flag = False                                         ## set nH flag to False
                                    N_H = 0.0                                               ## set nH density
                                    beta_dust = 0.0                                         ## set beta
                                    kappa_1300 = 0.0                                        ## set kappa


                                ## check dust file
                                if (DustFileName is None):
                                    DustFileName = ""


                                ## check background file
                                if (BackgroundFileName is None):
                                    BackgroundFileName = ""


                                ## get parameters for phenomenological description of continuum
                                LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                                LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                                LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                                LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                                LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                                LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']


                                ## update progress bar
                                ProgressBarValue += 1
                                ok = bar.setValue(ProgressBarValue)
                                if (not ok):
                                    print ("\\n\\nAbort")
                                    self.lock = 0
                                    return

                                # Debug:
                                # print ("\n\nFreqMin = ", FreqMin)
                                # print ("FreqMax = ", FreqMax)
                                # print ("FreqStep = ", FreqStep)
                                # print ("MolfitsFileName = ", LocalMolfitFileName)
                                # print ("iso_flag = ", self.IsoFlag)
                                # print ("IsoTableFileName = ", IsoTableFileName)
                                # print ("TelescopeSize = ", TelescopeSize)
                                # print ("BMIN = ", BMIN)
                                # print ("BMAJ = ", BMAJ)
                                # print ("BPA = ", BPA)
                                # print ("Inter_Flag = ", InterFlag)
                                # print ("vLSR = ", GlobalvLSR)
                                # print ("Redshift = ", Redshift)
                                # print ("t_back_flag = ", t_back_flag)
                                # print ("tBack = ", tBack)
                                # print ("tSlope = ", tSlope)
                                # print ("nH_flag = ", nH_flag)
                                # print ("N_H = ", N_H)
                                # print ("beta_dust = ", beta_dust)
                                # print ("kappa_1300 = ", kappa_1300)
                                # print ("DustFileName = ", DustFileName)
                                # print ("BackgroundFileName = ", BackgroundFileName)
                                # print ("ContPhenFuncID = ", LocalContPhenFuncID)
                                # print ("ContPhenFuncParam1 = ", LocalContPhenFuncParam1)
                                # print ("ContPhenFuncParam2 = ", LocalContPhenFuncParam2)
                                # print ("ContPhenFuncParam3 = ", LocalContPhenFuncParam3)
                                # print ("ContPhenFuncParam4 = ", LocalContPhenFuncParam4)
                                # print ("ContPhenFuncParam5 = ", LocalContPhenFuncParam5)
                                # print ("dbFileName = ", LocalDBFileName)
                                # print ("NumModelPixelXX = ", LocalNumModelPixelXX)
                                # print ("NumModelPixelYY = ", LocalNumModelPixelYY)
                                # print ("LocalLocalOverlapFlag = ", LocalLocalOverlapFlag)
                                # print ("LocalNoSubBeamFlag = ", LocalNoSubBeamFlag)
                                # print ("LocalEmAbsPATH = ", LocalEmAbsPATH)


                                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                ## call myXCLASS function
                                printflag = True
                                modeldata, log, TransEnergies, IntOpt, JobDir = task_myXCLASS.myXCLASSCore(FreqMin = FreqMin, FreqMax = FreqMax, \
                                                                                                           FreqStep = FreqStep, \
                                                                                                           MolfitsFileName = LocalMolfitFileName, \
                                                                                                           iso_flag = self.IsoFlag, \
                                                                                                           IsoTableFileName = IsoTableFileName, \
                                                                                                           TelescopeSize = TelescopeSize, \
                                                                                                           BMIN = BMIN, BMAJ = BMAJ, BPA = BPA, \
                                                                                                           Inter_Flag = InterFlag, \
                                                                                                           vLSR = GlobalvLSR, Redshift = Redshift, \
                                                                                                           t_back_flag = t_back_flag, tBack = tBack, \
                                                                                                           tSlope = tSlope, nH_flag = nH_flag, \
                                                                                                           N_H = N_H, beta_dust = beta_dust, \
                                                                                                           kappa_1300 = kappa_1300, \
                                                                                                           DustFileName = DustFileName, \
                                                                                                           BackgroundFileName = BackgroundFileName, \
                                                                                                           ContPhenFuncID = LocalContPhenFuncID, \
                                                                                                           ContPhenFuncParam1 = LocalContPhenFuncParam1, \
                                                                                                           ContPhenFuncParam2 = LocalContPhenFuncParam2, \
                                                                                                           ContPhenFuncParam3 = LocalContPhenFuncParam3, \
                                                                                                           ContPhenFuncParam4 = LocalContPhenFuncParam4, \
                                                                                                           ContPhenFuncParam5 = LocalContPhenFuncParam5, \
                                                                                                           dbFileName = LocalDBFileName, \
                                                                                                           NumModelPixelXX = LocalNumModelPixelXX, \
                                                                                                           NumModelPixelYY = LocalNumModelPixelYY, \
                                                                                                           LocalOverlapFlag = LocalLocalOverlapFlag, \
                                                                                                           NoSubBeamFlag = LocalNoSubBeamFlag, \
                                                                                                           EmAbsPATH = LocalEmAbsPATH, \
                                                                                                           printFlag = printflag)

                                self.myXCLASSSpectraList.append(modeldata)                  ## store calculated spectrum
                                # os.system("rm -rf " + JobDir)                               ## clean up job directory


                                ## extract transition frequencies which fulfill the creteria
                                TransFreqList = []
                                MolNameList = []
                                TransParamList = []
                                for line in TransEnergies:
                                    if (len(line) == 8):
                                        TransFreqList.append(float(line[0]))
                                        MolNameList.append(line[-1])
                                        TransParamList.append(line)
                                self.AllTransFreqList.append(TransFreqList)
                                self.AllMolNameList.append(MolNameList)
                                self.AllTransParamList.append(TransParamList)


            ## update progress bar
            bar.close()
            #    ProgressBarValue += 1
            #    ok = bar.setValue(ProgressBarValue)
            #    if (not ok):
            #        print ("\\n\\nAbort")
            #        self.lock = 0
            #        return


        ##================================================================================================================================================
        ## create plot


        ## setup figure
        if (self.OnceFlagTab3):
            self.figTab3Top = plt.figure()                                                  ## set up figure
        self.figTab3Top.clf()                                                               ## clear figure


        ## add and adjust subplot to figure
        self.figTab3Top.subplots_adjust(bottom = 0.16, top = 0.97, right = 0.98, left = 0.07, hspace = 0.0, wspace = 0.13)
        layerTab3Top = self.figTab3Top.add_subplot(1, 1, 1)
        layerTab3Top.set_xlim(self.AllMinFreq, self.AllMaxFreq)
        layerTab3Top.set_ylabel(r"T$_{mb}$ [K]")
        layerTab3Top.set_xlabel("Frequency [MHz]")
        layerTab3Top.grid(True)
        layerTab3Top.xaxis.set_major_formatter(pylab.matplotlib.ticker.FormatStrFormatter('%.1e'))
        layerTab3Top.yaxis.set_major_formatter(pylab.matplotlib.ticker.FormatStrFormatter('%.3e'))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add obs. data and myXCLASS spectrum
        ObsData = self.ObsDataList[self.CurrentObsDataID]
        layerTab3Top.plot(ObsData[:, 0], ObsData[:, -1], 'b-', linewidth=1.0, drawstyle='steps-mid', label = "obs. data")
        for LocalmyXCLASSSpectraIndex, LocalmyXCLASSSpectra in enumerate(self.myXCLASSSpectraList):
            if (LocalmyXCLASSSpectraIndex == 0):
                layerTab3Top.plot(LocalmyXCLASSSpectra[:, 0], LocalmyXCLASSSpectra[:, -1], 'r-', linewidth=1.0, label = "fit")
            else:
                layerTab3Top.plot(LocalmyXCLASSSpectra[:, 0], LocalmyXCLASSSpectra[:, -1], 'r-', linewidth=1.0)
        ylimits = layerTab3Top.get_ylim()

        # Debug:
        # print ("\n\nself.CurrentObsDataID = ", self.CurrentObsDataID)
        # print ("self.CurrentFreqRangeID = ", self.CurrentFreqRangeID)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add lines at the shifted transition frequencies
        iii = (-1)
        LocalFreqCounter = 0
        for ObsDataFileIndex in range(NumExpFiles):                                         ## loop over all spectra
            if (self.CurrentObsDataID == (-1) or self.CurrentObsDataID == ObsDataFileIndex):


                ## get number of freq. ranges etc. from obs. xml file
                RangeIndex = (-1)
                ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                               NumberRangeListIn = self.NumberExpRangesList, \
                                                                               GlobalvLSRListIn = self.GlobalvLSRList)
                NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
                GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
                if (GlobalvLSR is None):
                    GlobalvLSR = 0.0


                ## determine new frequency range definitions and write to xml file
                TotalNumberFreqRanges = 0
                for RangeIndex in range(NumberFrequencyRanges):                             ## loop over all frequency ranges
                    LocalFreqCounter += 1
                    if ((self.CurrentFreqRangeID - 1) == (-1) or (self.CurrentFreqRangeID - 1) == LocalFreqCounter):


                        ## get parameter for current frequency range
                        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                        FreqMinListIn = self.FreqMinList, \
                                                                                        FreqMaxListIn = self.FreqMaxList)
                        FreqMin = ObsXMLParameterDictRange['FreqMin']
                        if (FreqMin is not None):
                            FreqMax = ObsXMLParameterDictRange['FreqMax']
                            iii += 1


                            ## get transition parameters
                            TransFreqList = self.AllTransFreqList[iii]
                            MolNameList = self.AllMolNameList[iii]
                            TransParamList = self.AllTransParamList[iii]


                            ## add vertical lines
                            for LocalMolNameIndex, LocalMolName in enumerate(MolNameList):
                                LocalFlag = False
                                if (LocalMolName == self.CurrentMolecule or self.ShowAllFlag):
                                    LocalFlag = True

                                elif (self.IsoFlag):
                                    if (LocalMolName == self.LocalIsotopologuelist[self.AllMolCurrentIsoID]):
                                        LocalFlag = True
                                    elif (self.LocalIsotopologuelist[self.AllMolCurrentIsoID] == "All"):
                                        if (LocalMolName in self.LocalIsotopologuelist):
                                            LocalFlag = True

                                if (LocalFlag):
                                    TransParam = TransParamList[LocalMolNameIndex]
                                    TransFreq = TransParam[0]
                                    for CompID, Compparam in enumerate(self.CurrentMolfitParameter):
                                        if (self.CurrentAllMolCompIndex == CompID or self.CurrentAllMolCompIndex == (-1)):
                                            voff = float(Compparam[4][3]) + float(GlobalvLSR)
                                            xFreq = task_myXCLASS.ConvertFreq(float(TransFreq), voff)
                                            if (self.AllMinFreq <= xFreq and xFreq <= self.AllMaxFreq):
                                                layerTab3Top.axvline(x=xFreq, color='blue', linewidth=1, linestyle='--', zorder=0)
                                                y = ylimits[1] - (ylimits[1] * 0.02)
                                                layerTab3Top.text(xFreq, y, LocalMolName, fontsize=10, rotation=90, color = 'g', zorder=1)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add figure only once to the GUI
        layerTab3Top.legend()
        if (self.OnceFlagTab3):
            self.OnceFlagTab3 = False
            self.canvas = FigureCanvas(self.figTab3Top)
            self.gridLayout_AllPlot_Tab3.addWidget(self.canvas)
            self.canvas.draw()
            self.toolbar = NavigationToolbar(self.canvas, self, coordinates=True)
            self.gridLayout_AllPlot_Tab3.addWidget(self.toolbar)
            self.canvas.draw()
        else:
            self.canvas.draw()


        ## store current molfit and iso ratio file
        self.LastMolfitFile = copy.deepcopy(self.AllMolfitFileContent)
        self.LastIsoRatioFileContent = copy.deepcopy(IsoRatioFileContent)
        self.LastMolfitFileAllParameters = self.MolfitFileAllParameters
        self.LastMoleculesInMolfitFile = self.MoleculesInMolfitFile


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## disable elements of tab 3
        self.Tab3setDisEnabled(True)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## release lock
        self.LockTab3 = False


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for usage of combo box
    def Tab3comboSelected(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## are elements of tab 3 locked?
        if (self.LockTab3):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## combo box for obs. data file
        if (objectName == "comboBox_ObsDataFile_Tab3"):


            ## get index of selected data file
            self.CurrentObsDataID = max(0, self.comboBox_ObsDataFile_Tab3.currentIndex())

            # Debug:
            # print ("self.CurrentObsDataID = ", self.CurrentObsDataID)


            ## determine all frequency ranges for the current obs. data file
            self.CurrentFreqRangeID = 0
            self.CurrentRangeList = ["All ranges"]


            ## get number of freq. ranges from obs. xml file
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(self.CurrentObsDataID, RangeIndex, \
                                                                           NumberRangeListIn = self.NumberExpRangesList)
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']        ## get number of frequency ranges for current obs. data file

            # Debug:
            # print ("NumberFrequencyRanges = ", NumberFrequencyRanges


            ## get parameters for current frequency range
            for RangeIndex in range(NumberFrequencyRanges):                                 ## loop over all range definitions in the whole xml file


                ## get parameter for current frequency range
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(self.CurrentObsDataID, RangeIndex, FreqMinListIn = self.FreqMinList, \
                                                                                FreqMaxListIn = self.FreqMaxList)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None):
                    FreqMax = ObsXMLParameterDictRange['FreqMax']
                    self.CurrentRangeList.append(str(FreqMin) + " - " + str(FreqMax) + " MHz")
            self.comboBox_FreqRange_Tab3.clear()
            self.comboBox_FreqRange_Tab3.addItems(self.CurrentRangeList)
            self.comboBox_FreqRange_Tab3.setCurrentIndex(self.CurrentFreqRangeID)

            # Debug:
            # print ("self.CurrentRangeList = ", self.CurrentRangeList)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## combo box for frequency range
        elif (objectName == "comboBox_FreqRange_Tab3"):


            ## get index of selected frequency range
            self.CurrentFreqRangeID = max(0, self.comboBox_FreqRange_Tab3.currentIndex())

            # Debug:
            # print ("self.CurrentFreqRangeID = ", self.CurrentFreqRangeID)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## combo box for molecule
        elif (objectName == "comboBox_Isotopologues_Tab3"):


            ## get index of selected isotopologue
            self.AllMolCurrentIsoID = max(0, self.comboBox_Isotopologues_Tab3.currentIndex())
            self.CurrentIsoID = max(0, self.AllMolCurrentIsoID - 1)

            # Debug:
            # print ("self.AllMolCurrentIsoID = ", self.AllMolCurrentIsoID)
            # print ("self.CurrentIsoID = ", self.CurrentIsoID)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update GUI
        self.Tab3SetValuesForTab()

        # Debug:
        if (self.dbgFlag):
            print ("\nTab3comboSelected:")


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subroutine for usage of combo box
    def Tab3molfitListEntrySelected(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## are elements of tab 3 locked?
        if (self.LockTab3):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get selected line
        StrippedLine = self.list_molfit_Tab3.currentItem().text()                                   ## get selected line of all-molfit file
        StrippedLine = str(StrippedLine).strip()                                                    ## remove leading and tailing blanks
        self.CurrentAllMolfitIndex = self.list_molfit_Tab3.currentRow()                             ## get number of selected row


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine molecule and component id
        self.CurrentAllMolCompIndex = (-1)                                                          ## reset component index for all-molfit file
        for LocalLineIndex in range(self.CurrentAllMolfitIndex, -1, -1):                            ## loop over all lines in all-molfit file
            StrippedLine = self.AllMolfitFileContent[LocalLineIndex].strip()                        ## remove leading and tailing blanks
            SplittedLine = StrippedLine.split()                                                     ## split line
            if (len(SplittedLine) == 2):                                                            ## does current line define molecule?
                self.CurrentMolecule = SplittedLine[0].strip()                                      ## get current molecule
                self.CurrentMoleculeID = self.MoleculesInMolfitFile.index(self.CurrentMolecule)     ## WARNING: May lead to wrong results if molecule
                                                                                                    ##          occur more than once in all-molfit file
                self.CurrentMolfitParameter = self.MolfitFileAllParameters[self.CurrentMoleculeID]
                self.CompNumChangeFlag = True
                self.Tab2CreateMolfitFile()
                self.AllMoleculesMolfitFiles[self.CurrentMoleculeID] = copy.deepcopy(self.CurrentMolfitFileContents)


                ## prepare subdirectories for current molecule
                LocalMoleculeFileName = task_LineIdentification.MoleculeFileName(self.CurrentMolecule)
                self.CurrentMoleculeDir = self.GUIJobDir + LocalMoleculeFileName + "/"
                if (not os.path.exists(self.CurrentMoleculeDir)):
                    os.system("rm -rf " + self.CurrentMoleculeDir + "; mkdir " + self.CurrentMoleculeDir)


                ## exit loop
                break
            else:
                self.CurrentAllMolCompIndex += 1                                                    ## increase comp. index


        ## set component index
        if (self.CurrentAllMolCompIndex != (-1)):
            self.CurrentComponentIndex = self.CurrentAllMolCompIndex
        else:
            self.CurrentComponentIndex = 0

        # Debug:
        # print ("\n\nself.CurrentAllMolCompIndex = ", self.CurrentAllMolCompIndex)
        # print ("self.AllMolfitFileContent = ", self.AllMolfitFileContent)
        # print ("self.CurrentMolecule = ", self.CurrentMolecule)
        # print ("self.CurrentMoleculeID = ", self.CurrentMoleculeID)
        # print ("\nself.CurrentMolfitParameter = ", self.CurrentMolfitParameter)
        # print ("self.CurrentComponentIndex = ", self.CurrentComponentIndex)
        # print ("self.MolfitFileAllParameters = ", self.MolfitFileAllParameters)
        # print ("self.CurrentMolfitFileContents = ", self.CurrentMolfitFileContents)
        # print ("self.CompNumChangeFlag = ", self.CompNumChangeFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get index of selected molecule
        if (self.IsoFlag):
            self.CurrentIsoParam = copy.deepcopy(self.AllIsotopologueList[self.CurrentMoleculeID])
            self.AllMolCurrentIsoID = 0
            self.LocalIsotopologuelist = ["All"]
            for line in self.CurrentIsoParam:
                self.LocalIsotopologuelist.append(line[0].strip())
        else:
            self.CurrentIsoParam = []
            self.AllMolCurrentIsoID = 0
            self.LocalIsotopologuelist = []


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set values for tab 3
        self.Tab3SetValuesForTab()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check box event for tab 3
    def Tab3checkBoxSelected(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## are elements of tab 3 locked?
        if (self.LockTab3):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get flag and name of calling object
        sender = self.sender()
        flag = sender.isChecked()
        objectName = sender.objectName()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define ShowAllFlag
        self.ShowAllFlag = flag


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set values for tab 3
        self.Tab3SetValuesForTab()


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## fit all molecules and isotopologues
    def Tab3FitAll(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## are elements of tab 3 locked?
        if (self.LockTab3):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update history arrays
        # self.Tab2UpdateHistoryArray()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## disable elements of tab 3
        self.Tab3setDisEnabled(False)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create subdirectory for fit
        lt = time.localtime()
        LocalFitDir = self.GUIJobDir + "all-molecule-fit____" + time.strftime("%Y_%m_%d", lt) + "___" + time.strftime("%H-%M-%S", lt) + "/"
        cmdString = "mkdir -p " + LocalFitDir
        os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## make a copy of obs. xml file
        LocalObsXMLFile = LocalFitDir + "/all_obs.xml"
        cmdString = "cp " + self.GUIJobDir + "obs.xml " + LocalObsXMLFile
        os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write current molfit file contents to file
        LocalMolfitFileName = LocalFitDir + "all.molfit"
        OldMolfitFileName = self.CurrentMolfitFileName
        self.CurrentMolfitFileName = LocalMolfitFileName
        self.MenuSaveCurrMolfitFile()
        self.CurrentMolfitFileName = OldMolfitFileName


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write current iso ratio file contents to file
        if (self.IsoFlag):
            LocalIsoRatioFileName = LocalFitDir + "myXCLASS_iso-ratio.dat"
            OldIsoRatioFileName = self.IsoRatioFileName
            self.IsoRatioFileName = LocalIsoRatioFileName
            self.MenuSaveIsoRatioFile()
            self.IsoRatioFileName = OldIsoRatioFileName


            ## update name and path of iso ratio file
            self.IsotopologuesList = ["True"]
            task_MAGIX.WriteXMLtagNEW(LocalObsXMLFile, "iso_flag", self.IsotopologuesList)
            self.IsoTableFileNameList = [LocalIsoRatioFileName]
            task_MAGIX.WriteXMLtagNEW(LocalObsXMLFile, "Isotopologues", self.IsoTableFileNameList)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## call fit routine to start myXCLASSFit function
        BestMolfitFile, IsoRatioFileContent = self.FitCore(LocalFitDir, LocalMolfitFileName, LocalObsXMLFile)

        # Debug:
        # print ("\n\nBestMolfitFile = ", BestMolfitFile)
        # print ("IsoRatioFileContent = ", IsoRatioFileContent)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get optimized molfit parameters
        MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(BestMolfitFile)
        self.CurrentMolecule = self.MoleculesInMolfitFile[self.CurrentMoleculeID]
        self.CurrentMolfitFileContents = copy.deepcopy(self.AllMoleculesMolfitFiles[self.CurrentMoleculeID])
        self.MolfitFileAllParameters = []
        NewRangeFlag = False
        for LocalMolfitParametersID, LocalOrigMolfitParameters in enumerate(AllParameters):
            LocalMolfitParameters = self.Tab2GetParameterFromMolfitFile(LocalOrigMolfitParameters, NewRangeFlag)
            self.MolfitFileAllParameters.append(copy.deepcopy(LocalMolfitParameters))
            if (self.CurrentMoleculeID == LocalMolfitParametersID):
                self.CurrentMolfitParameter = copy.deepcopy(LocalMolfitParameters)
        self.AllMoleculesMolfitFiles = copy.deepcopy(MolfitFileForEachMolecule)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get optimized iso ratios
        if (self.IsoFlag):
            for LocalMoleculeID, LocalMolecule in enumerate(self.MoleculesInMolfitFile):
                NewIsoRatioFileContent = []
                for line in IsoRatioFileContent:
                    StrippedLine = line.strip()
                    i  = StrippedLine.find("%")
                    if (i > (-1)):
                        StrippedLine = StrippedLine[:i].strip()
                    if (StrippedLine != ""):
                        SplittedLine = StrippedLine.split()
                        if (len(SplittedLine) < 4):
                            LowLimit = 1.0
                            UpLimit = 1.0
                        else:
                            LowLimit = float(SplittedLine[3])
                            UpLimit = float(SplittedLine[4])
                        NewIsoRatioFileContent.append([SplittedLine[0], SplittedLine[1], SplittedLine[2], LowLimit, UpLimit])
                LocalIsoParam = self.Tab2GetIsoRatiosFromFile(LocalMolecule, NewIsoRatioFileContent)
                self.AllIsotopologueList[LocalMoleculeID] = copy.deepcopy(LocalIsoParam)
                if (LocalMoleculeID == self.CurrentMoleculeID):
                    self.CurrentIsoParam = copy.deepcopy(LocalIsoParam)

            # Debug:
            # print ("\nIsoRatioFileContent = ", IsoRatioFileContent)
            # print ("self.CurrentIsoParam = ", self.CurrentIsoParam)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## enable elements of tab 2
        self.Tab2setDisEnabled(True)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update GUI
        self.CompNumChangeFlag = True
        if (self.tabWidget.currentIndex() == 1):
            self.Tab2SetValuesForTab()
        elif (self.tabWidget.currentIndex() == 2):
            self.Tab3ChangeTab()


        ## return to GUI
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## taken from http://stackoverflow.com/questions/13269936/python-qt-progressbar
class ProgressBar(QtWidgets.QDialog):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize class
    def __init__(self, parent=None, total=20, widgetTitle="Progress", bodyText=""):
        """

        input parameters:
        -----------------

            - None


        output parameters:
        ------------------

            - None

        """


        super(ProgressBar, self).__init__(parent)
        self.setAbortFlag = False
        self.progressbar = QtWidgets.QProgressBar()
        self.progressbar.setMinimum(1)
        self.progressbar.setMaximum(total)
        self.label = QtWidgets.QLabel()
        self.button = QtWidgets.QPushButton('Cancel')
        self.button.setToolTip('Click to cancel!')
        self.button.clicked.connect(self.handleButton)
        main_layout = QtWidgets.QGridLayout()
        main_layout.addWidget(self.label, 0, 0)
        main_layout.addWidget(self.progressbar, 1, 0)
        # main_layout.addWidget(self.button, 2, 1)
        self.label.setText(bodyText)
        self.setLayout(main_layout)
        self.setWindowTitle(widgetTitle)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle button
    def handleButton(self):
        """

        input parameters:
        -----------------

            - None


        output parameters:
        ------------------

            - None
        """

        ## set abort flag
        self.setAbortFlag = True


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set value for progress bar
    def setValue(self, value):
        """

        input parameters:
        -----------------

            - value:        counter for progress bar


        output parameters:
        ------------------

            - None
        """

        # Debug:
        # print ("value = ", value)


        ## set value
        self.progressbar.setValue(value)
        QtWidgets.QApplication.processEvents()


        ## return to GUI
        ok = True
        if (self.setAbortFlag):
            ok = False
        return ok
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set value for progress bar
    def setBodyText(self, bodyText):
        """

        input parameters:
        -----------------

            - bodyText:         text which is displayed in the widget


        output parameters:
        ------------------

            - None
        """

        # Debug:
        # print ("bodyText = ", bodyText)


        ## set value for progress bar
        self.label.setText(bodyText)


        ## return to GUI
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## class ExternalWidget (used for external widgets)
class ExternalWidget(QtWidgets.QDialog):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize class ExternalWidget
    def __init__(self, parent = None, ExternalWidgetName = "", OrigValue = 0.0):
        """

        input parameters:
        -----------------

            - None


        output parameters:
        ------------------

            - None
        """

        # Debug:
        # print ("parent = ", parent)
        # print ("ExternalWidgetName = ", ExternalWidgetName)
        # print ("OrigValue = ", OrigValue)


        ## create widget
        super(ExternalWidget, self).__init__(parent)


        ## initialize return parameters
        self.OKFlag = False
        self.ExternalWidgetName = ExternalWidgetName
        self.returnParam = [0.0, 0.0]


        ## define title and body text
        if (ExternalWidgetName == "NewVelRange"):
            widgetTitle = "Set velocity ranges"
            bodyText = "Set new ranges for velocity (km/s):"
            lineEdit1Label = "Min. velocity (km/s):"
            lineEdit1Value = str(OrigValue[0])
            lineEdit2Label = "Max. velocity (km/s):"
            lineEdit2Value = str(OrigValue[1])
            self.returnParam = [lineEdit1Value, lineEdit2Value]


            ## create elements for widget
            self.label = QtWidgets.QLabel()
            self.label.setText(bodyText)
            if (ExternalWidgetName == "NewVelRange"):
                self.label1 = QtWidgets.QLabel()
                self.label1.setText(lineEdit1Label)
                self.lineEdit1 = QtWidgets.QLineEdit()
                self.lineEdit1.setText(lineEdit1Value)
                self.lineEdit1.setObjectName("Edit1")
                self.lineEdit1.editingFinished.connect(self.LineEditWidget)

                self.label2 = QtWidgets.QLabel()
                self.label2.setText(lineEdit2Label)
                self.lineEdit2 = QtWidgets.QLineEdit()
                self.lineEdit2.setText(lineEdit2Value)
                self.lineEdit2.setObjectName("Edit2")
                self.lineEdit2.editingFinished.connect(self.LineEditWidget)
            self.buttonOK = QtWidgets.QPushButton('OK')
            self.buttonOK.setShortcut("Return")
            self.buttonOK.clicked.connect(self.ButtonWidget)
            self.buttonCancel = QtWidgets.QPushButton('Cancel')
            self.buttonCancel.setShortcut("ESC")
            self.buttonCancel.clicked.connect(self.ButtonWidget)


            ## box layout
            layout = QtWidgets.QGridLayout()
            layout.addWidget(self.label, 0, 0, 1, 0)
            layout.addWidget(self.label1, 1, 0)
            layout.addWidget(self.lineEdit1, 1, 1)
            layout.addWidget(self.label2, 2, 0)
            layout.addWidget(self.lineEdit2, 2, 1)
            layout.addWidget(self.buttonCancel, 3, 0)
            layout.addWidget(self.buttonOK, 3, 1)
            self.setLayout(layout)
            self.setWindowTitle(widgetTitle)


        ## return to GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle button event of widget
    def ButtonWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()
        name = sender.text()

        # Debug:
        # print ("objectName = ", objectName)
        # print ("name = ", name)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## identify calling button
        if (name == "Cancel"):
            self.OKFlag = False
        else:
            self.OKFlag = True


        ## return to main GUI
        self.close()
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle lineEdit event of widget
    def LineEditWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()
        val = str(sender.text())

        # Debug:
        # print ("objectName = ", objectName)
        # print ("val = ", val)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## identify calling lineEdit
        if (objectName == "Edit1"):
            self.returnParam[0] = val
        elif (objectName == "Edit2"):
            self.returnParam[1] = val


        ## return to main GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get return parameters of widget
    def getReturnparamWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## return to GUI
        return self.OKFlag, self.returnParam
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start XCLASS-GUI
##
def StartXCLASSGUI():
    """

input parameters:
-----------------

    - None


output parameters:
------------------

    - None
    """


    # ## define path of XCLASS-GUI
    devFlag = False

    # Debug:
    # print ("devFlag = ", devFlag)


    ## get XCLASS root directory
    XCLASSRootDir = task_myXCLASS.GetXCLASSRootDir()

    # Debug:
    # print ("XCLASSRootDir = ", XCLASSRootDir)


    ## set path of XCLASS-GUI
    XCLASSGUIDir = XCLASSRootDir + "tools/XCLASSGUI/"
    XCLASSGUIDir = os.path.normpath(XCLASSGUIDir) + "/"

    # Debug:
    print ("XCLASSGUIDir = ", XCLASSGUIDir)


    ##====================================================================================================================================================
    ## initialize user parameter
    UserMolfitsFileName = ""
    UserObsXMLFilename = ""
    UserAlgXMLFile = ""


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get command line arguments
    ListOfOptions = ["molfit", "obs", "exp", "alg"]
    NameOfScript = "XCLASS-GUI.py"
    HelpText = "GUI for manual fit parameter estimation."
    ListOptionValues = task_myXCLASS.GetCommandLineArguments(ListOfOptions, NameOfScript, HelpText)
    for LocalOptionID, Option in enumerate(ListOfOptions):
        OptionValue = ListOptionValues[LocalOptionID].strip()
        if (OptionValue != ""):


            ## get path and name of molfit file
            if (Option == "molfit"):
                UserMolfitsFileName = OptionValue


            ## get path and name of obs. xml file
            elif (Option in ["obs", "exp"]):
                UserObsXMLFilename = OptionValue


            ## get path and name of alg. xml file
            elif (Option == "alg"):
                UserAlgXMLFile = OptionValue

    # Debug:
    # print ("\n\nMolfitsFileName = ", MolfitsFileName)
    # print ("UserObsXMLFilename = ", UserObsXMLFilename)
    # print ("UserAlgXMLFile = ", UserAlgXMLFile)
    # sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## loadUiType function requires a single argument, the name of a Designer UI file, and returns a tuple of the appropriate application classes
    ## the return tuple contains two class definitions. The first is our custom GUI application class, set to the Ui_MainWindow variable. The second
    ## is the base class for our custom GUI application, which in this case is PyQt4.QtGui.QMainWindow. The base class type is defined by the widget
    ## we first created with Designer. Note: These are not instances of these classes, but the class definitions themselves. They are meant to serve
    ## as superclasses to our new application logic class, which we’ll create next.
    UiFileName = XCLASSGUIDir + 'gui/layout/XCLASS-GUI.ui'


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start GUI
    global app
    app = QtWidgets.QApplication.instance()
    if (app is None):
        app = QtWidgets.QApplication(sys.argv)
    main = Main(UiFileName, XCLASSRootDir, UserMolfitsFileName, UserObsXMLFilename, UserAlgXMLFile, devFlag)
    main.show()
    app.exit(app.exec_())


    ## return to main program
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start main program
##
if __name__ == '__main__':


    ## start XCLASS-GUI
    StartXCLASSGUI()


## finished!
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------

