#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module contains all subroutines to perform the myXCLASSMapRedoFit function which enables to improve a previous myXCLASSMapFit function run.
##  Copyright (C) 2012 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function CopyNDecompress:                     copy and decompress a file or directory
##      - function myXCLASSMapRedoFitCore:              core myXCLASSMapFit function
##      - function myXCLASSMapRedoFit:                  redo one or more pixel fits of a previous myXCLASSMapFit function run
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2014-10-01      initial version
##  T. Moeller      2020-01-03      porting to python 3.x
##  T. Moeller      2021-11-30      modification of new myXCLASSMapFit function
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import os                                                                                   ## import os package
os.environ['OPENBLAS_NUM_THREADS'] = '1'
import numpy                                                                                ## import numpy package
import sys                                                                                  ## import sys package
import time                                                                                 ## import package date and time
if (not 'matplotlib' in sys.modules):
    import matplotlib                                                                       ## import matplotlib package
    matplotlib.use("Agg")                                                                   ## avoid display error
from spectral_cube import SpectralCube, Slice                                               ## import SpectralCube package
from spectral_cube.utils import WCSMismatchWarning
import astropy.units as u                                                                   ## import units package from astropy
from astropy.io import fits                                                                 ## import fits from astropy
from astropy.wcs import WCS                                                                 ## import wcs from astropy
from . import task_myXCLASS                                                                 ## import package myXCLASS
from . import task_myXCLASSMapFit                                                           ## import package myXCLASSFit
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## copy and decompress a file
##
def CopyNDecompress(SourceFileName, DestinationFileName):
    """

input parameters:
-----------------

    - SourceFileName:           path and name of source file or directory

    - DestinationFileName:      path and name of destination file


output parameters:
------------------

    - None
"""

    # Debug:
    # print ("SourceFileName = ", SourceFileName)
    # print ("DestinationFileName = ", DestinationFileName)


    ## copy file
    SourceFileName = os.path.normpath(SourceFileName)
    DestinationFileName = os.path.normpath(DestinationFileName)
    cmdString = "cp " + SourceFileName + " " + DestinationFileName + "; "
    if (DestinationFileName.endswith(".gz")):
        cmdString += "gzip -dr " + DestinationFileName


    ## execute shell commands
    os.system(cmdString)


    ## define return parameter
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Simplified CASA interface for MAGIX with myXCLASS program to fit a complete data cube instead of a single spectrum
##
def myXCLASSMapRedoFitCore(JobNumber = None, PixelList = [], JobPath = "", MaskFileName = "", Threshold = None,
                           MolfitsFileName = "", MolfitFileName = None, \
                           EstimateContinuumFlag = False, EstimateConNumParts = 20, EstimateConNumEntireCubeFlag = True, DataSmoothValue = 0.0, \
                           NumberIteration = 50, NumberProcessors = 2, AlgorithmXMLFileName = "", AlgorithmXMLFile = None, clusterdef = "", \
                           FastFitFlag = True, FullFitFlag = False, ChannelIntegrationFlag = False, daskFlag = False, \
                           FreqMin = None, FreqMax = None, t_back_flag = True, tBack = 0.0, tSlope = 0.0, nH_flag = False, \
                           N_H = 0.0, beta_dust = 0.0, kappa_1300 = 0.0, DustFileName = "", BackgroundFileName = "", \
                           ContPhenFuncID = None, ContPhenFuncParam1 = None, ContPhenFuncParam2 = None, ContPhenFuncParam3 = None, \
                           ContPhenFuncParam4 = None, ContPhenFuncParam5 = None, iso_flag = False, \
                           IsoTableFileName = "", NumModelPixelXX = 100, NumModelPixelYY = 100, LocalOverlapFlag = False, \
                           NoSubBeamFlag = True, EmAbsPATH = None):
    """

This function offers the possibility to redo one or more so called pixel fits of a
previous myXCLASSMapFit run. The function performs fits for selected pixels and
recreates the different parameter maps using the new optimized parameter values. The
user has to define the pixel coordinates, which should be re-fitted, the job number
of the previous myXCLASSMapFit and the new molfit and algorithm XML file.


input parameters:
-----------------

    - JobNumber:                    Job number of a previous myXCLASSMapFit run which should
                                    be improved.

    - PixelList:                    list of all pixel coordinates which should be re-fitted.
                                    Each coordinate consists of two numbers separated by a
                                    comma, where the first number corresponds to the right
                                    ascension and second to the declination. Different pixel
                                    coordinates are enclosed by squared brackets and separated
                                    by commas.

    - JobPath:                      (optional) path of job directory describing myXCLASSMapFit
                                    application, (default: "").

    - EstimateContinuumFlag:        (optional) flag indicating continuum estimation, (default: False)

    - EstimateConNumParts:          (optional) number of sub parts used for continuum estimation,
                                    (default: 20)

    - EstimateConNumEntireCubeFlag: (optional) flag indicating that continuum estimation is done
                                    for the entire spectral range (default: True)

    - DataSmoothValue:              (optional) smooth factor (=0, no smoothing is performed) used for
                                    smoothing obs. data (default: 0.0).

    - NumberIteration:              (optional, only used if no alg. xml file is defined)
                                    max. number of iterations (default: 50).

    - NumberProcessors:             (optional, only used if no alg. xml file is defined)
                                    number of cores (default: 2)

    - AlgorithmXMLFileName:         (optional) only necessary, if the user wants to use another fit
                                    algorithm (than Levenberg-Marquardt) for fitting. Therefore,
                                    the path and name of a MAGIX xml-file defining settings
                                    for an algorithm or algorithm chain has to be given. (A
                                    relative path has to be defined relative to the current
                                    working directory!), (default: "")

    - AlgorithmXMLFile:             (optional) alias for AlgorithmXMLFileName

    - clusterdef:                   (optional) path and name of file describing a cluster (default: "")

    - FastFitFlag:                  (optional) flag indicating usage of fast-fitting method, (default: True)

    - FullFitFlag:                  (optional) full fit flag (default: False).

    - ChannelIntegrationFlag:       (optional, for full fitting method only) flag indicating channel
                                    integration (default: False)

    - daskFlag:                     (optional) flag indicating usage of dask package (default: False)

    - MolfitsFileName:              path and name of the extended molfit file, including the
                                    source size, the rotation temperature the column density,
                                    the velocity width, the velocity offset and the flag
                                    indicating if the current component is considered for
                                    core "c" or foreground "f".

                                    In contrast to the format of the molfit file the extended
                                    molfit file required by the myXCLASSMapFit function contains
                                    one (three) additional column(s) for each parameter of
                                    each component.

                                    A detailed description of the extended molfit file required
                                    by the myXCLASSMapFit function can be found in the manual.

                                    NOTE, a relative path has to be defined relative to the
                                    current working directory!

    - MolfitFileName:               (optional) alias of MolfitsFileName, (default: None).

    - Threshold:                    (optional) defines a threshold for a pixel. If the spectrum of a pixel
                                    has a intensity lower than the value defined by this parameter
                                    the pixel is not fitted (ignored), (default: None)

    - MaskFileName:                 (optional) path and name of FITS image describing selected pixel
                                    (default: "")


    - FreqMin:                      start frequency of simulated spectrum (in MHz), (default: 1.e99).

                                    Please note, if no start frequency is given, or if start frequency is
                                    lower or equal to the end frequency, the myXCLASSMapRedoFit  function will
                                    use the max. frequency range defined in the FITS header. Additionally,
                                    the step size, i.e. the difference between two frequencies is taken
                                    from the FITS header.

    - FreqMax:                      end frequency of simulated spectrum (in MHz), (default: 1.e99).

    - t_back_flag:                  ("True"/"False"), defines, if the user defined background
                                    temperature 'Tbg' and temperature slope 'Tslope' given
                                    by the input parameters tBack and tSlope describe the
                                    continuum contribution completely (t_back_flag = "True")
                                    or not (t_back_flag = "False") (default: "True").

    - tBack:                        background temperature (in K), (default: None).

    - tSlope:                       temperature slope (dimensionless), (default: None).

    - N_H:                          Hydrogen column density (cm^{-2}), (default: None).

    - beta_dust:                    beta for dust (dimensionless), (default: None).

    - kappa_1300:                   kappa (cm^2 g^{-1}), (default: None).

    - DustFileName:                 path and name of file describing dust contribution
                                    (default: "").

    - BackgroundFileName:           (optional) path and name of file describing background
                                    (default: "").

    - ContPhenFuncID:               (optional) describes which phenomenologicalal function is used
                                    to describe the continuum (default: None).

    - ContPhenFuncParam1:           (optional) first parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam2:           (optional) second parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam3:           (optional) third parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam4:           (optional) fourth parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam5:           (optional) fifth parameter for phenomenological function
                                    describing the continuum (default: None).

    - iso_flag:                     use isotopologues ("True"/"False"). If iso_flag is set to
                                    "False" the ratios are all set to 1 (default: "False").

    - IsoTableFileName:             path and name of an ASCII file including the iso ratios
                                    between certain molecules. The so-called "iso ratio file"
                                    defines the iso ratios between molecules. The ASCII file
                                    consists of three columns, where the first two columns
                                    indicates the molecules, respectively. The third column
                                    defines the ratio for both molecules. Here, the first
                                    41 characters belong to the first column, the next
                                    41 characters to the second column. The further character
                                    belongs to the third column.

                                    If no path and name is specified (default), the so-called iso-flag
                                    is set to "false".

    - NumModelPixelXX:              (optional) used for sub-beam modeling, describes the number
                                    of pixels used in x-direction (default: 100).

    - NumModelPixelYY:              (optional) used for sub-beam modeling, describes the number
                                    of pixels used in y-direction (default: 100).

    - LocalOverlapFlag:             (optional) flag indicates if local-overlap description is
                                    used or not (default: False).

    - NoSubBeamFlag:                (optional) prevent sub beam description (default: True).

    - EmAbsPATH:                    (optional) path containing files describing functions
                                    emission / absorption (default: None)


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("JobNumber = ", JobNumber)
    # print ("PixelList = ", PixelList)
    # print ("JobPath = ", JobPath)
    # print ("EstimateContinuumFlag = ", EstimateContinuumFlag)
    # print ("EstimateConNumParts = ", EstimateConNumParts)
    # print ("EstimateConNumEntireCubeFlag = ", EstimateConNumEntireCubeFlag)
    # print ("DataSmoothValue = ", DataSmoothValue)
    # print ("NumberIteration = ", NumberIteration)
    # print ("NumberProcessors = ", NumberProcessors)
    # print ("AlgorithmXMLFileName = ", AlgorithmXMLFileName)
    # print ("AlgorithmXMLFile = ", AlgorithmXMLFile)
    # print ("clusterdef = ", clusterdef)
    # print ("FastFitFlag = ", FastFitFlag)
    # print ("FullFitFlag = ", FullFitFlag)
    # print ("ChannelIntegrationFlag = ", ChannelIntegrationFlag)
    # print ("daskFlag = ", daskFlag)
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("MolfitFileName = ", MolfitFileName)
    # print ("Threshold = ", Threshold)
    # print ("MaskFileName = ", MaskFileName)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("t_back_flag = ", t_back_flag)
    # print ("tBack = ", tBack)
    # print ("tSlope = ", tSlope)
    # print ("nH_flag = ", nH_flag)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("iso_flag = ", iso_flag)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)
    # print ("EmAbsPATH = ", EmAbsPATH)


    ##====================================================================================================================================================
    ## initialize function


    ## define name of function
    NameOfFunction = "myXCLASSMapRedoFit"


    ## get default database file
    DefaultDBFileName = task_myXCLASS.GetDefaultDBFile()


    ##====================================================================================================================================================
    ## define MAGIX and myXCLASS root directory
    MAGIXrootDir = task_myXCLASS.GetMAGIXRootDir()
    myXCLASSrootDir = task_myXCLASS.GetmyXCLASSRootDir()

    # Debug:
    # print ("MAGIXrootDir = ", MAGIXrootDir)
    # print ("myXCLASSrootDir = ", myXCLASSrootDir)


    ##====================================================================================================================================================
    ## analyze input parameters


    ## determine XCLASS working directory
    if (JobPath != ""):
        MapFitJobDir = JobPath
    else:
        myXCLASSRunDirectory = str(os.environ.get('myXCLASSRunDirectory','')).strip()       ## get value of environment variable
        if (myXCLASSRunDirectory == ""):                                                    ## is environment variable defined?
            RunDir = os.path.abspath(myXCLASSrootDir) + "/run/"
        else:
            if (myXCLASSRunDirectory[0] != "/"):                                            ## make path absolute
                myXCLASSRunDirectory = myXCLASSrootDir + myXCLASSRunDirectory
            RunDir = myXCLASSRunDirectory + "/"                                             ## if environment variable is defined, apply user setting
        MapFitJobDir = os.path.normpath(RunDir + "/myXCLASSMapFit/") + "/"                  ## define path of directory for myXCLASSMapFit function runs

    # Debug:
    # print ("\n\nRunDir = ", RunDir)
    # print ("MapFitJobDir = ", MapFitJobDir)


    ## get alias for keyword "MolfitFileName"
    MolfitsFileName = MolfitsFileName.strip()
    if (MolfitsFileName == "" and MolfitFileName is not None):
        MolfitsFileName = MolfitFileName.strip()


    ## check alias of AlgorithmXMLFileName
    AlgorithmXMLFileName = AlgorithmXMLFileName.strip()
    if (AlgorithmXMLFileName == "" and AlgorithmXMLFile is not None):
        AlgorithmXMLFileName = AlgorithmXMLFile.strip()


    ## read in all job directories in the myXCLASSMapFit run directory
    MapFitFileListing = os.listdir(MapFitJobDir)


    ## check, if a job directory with the given job ID exists
    FoundFlag = False
    for JobDir in MapFitFileListing:

        # Debug:
        # print ("JobNumber, JobDir = ", JobNumber, JobDir)

        if (JobDir.find("_" + str(JobNumber)) > (-1)):
            MapFitJobDir = MapFitJobDir + JobDir + "/"
            FoundFlag = True
            break
    if (not FoundFlag):
        print ("\n\n\nError in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:")
        print ("\tCan not find a directory with the give job number.")
        print ("\n\tJobNumber = ", JobNumber)
        print ("\tCurrent myXCLASS working directory = ", MapFitJobDir)
        print ("\n\tPlease enter an existing job number and redo myXCLASSMapRedoFit function call!")
        return


    ## check, if the user defined pixels exists
    print ("\n\nCheck user input .. ", end = "", flush = True)
    if (len(PixelList) == 0 and MaskFileName == ""):
        print ("\n\n\nError in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:")
        print ("\tNor pixels neither a FITS image used for masking is defined.")
        print ("\n\tPlease correct the input and start myXCLASSMapRedoFit function again!")
        return

    # Debug:
    # print (">>MapFitJobDir = ", MapFitJobDir)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze masking parameters


    ## get name of FITS image describing chi2 values
    Chi2FileName = None
    ObsXMLFileName = None
    MapFitFileListing = os.listdir(MapFitJobDir)
    for LocalFileName in MapFitFileListing:
        if (LocalFileName == "BestResult___chi2values.fits"):
            Chi2FileName = MapFitJobDir + LocalFileName
        elif (LocalFileName == "obs.xml"):
            ObsXMLFileName = MapFitJobDir + LocalFileName
    if (Chi2FileName is None):
        print ("\n\n\nError in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:")
        print ("\n\t\tCan not find FITS image describing chi2 values in the given job directory!")
        print ("\n\t\tAbort myXCLASSMapRedoFit function!")
        print ("\n\n")
        return
    if (ObsXMLFileName is None):
        print ("\n\n\nError in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:")
        print ("\n\t\tCan not find obs. xml file in the given job directory!")
        print ("\n\t\tAbort myXCLASSMapRedoFit function!")
        print ("\n\n")
        return
    else:


        ## if no FITS image for masking is defined, create FITS image to describe selected pixels
        if (MaskFileName == ""):
            MaskFileName = ""


            ## get dimension, wcs, and header from chi^2 map
            NumX = None
            NumY = None
            ImageHeader = None
            ImageWCS = None
            LocalImage = fits.getdata(Chi2FileName, ext = 0)
            ImageHeader = fits.getheader(Chi2FileName)
            ImageWCS = WCS(ImageHeader)
            NumX = LocalImage.shape[0]
            NumY = LocalImage.shape[1]

            # Debug:
            # print ("NumX = ", NumX)
            # print ("NumY = ", NumY)
            # print ("ImageHeader = ", ImageHeader)
            # print ("ImageWCS = ", ImageWCS)


            ## define mask
            if (NumX is not None and NumY is not None):
                MaskArray = numpy.ones((NumX, NumY)) * numpy.nan
                for LocalCoord in PixelList:
                    xID = LocalCoord[0]
                    yID = LocalCoord[1]
                    MaskArray[xID, yID] = True


                ## write FITS image
                LocalMaskImage = Slice(value = MaskArray, wcs = ImageWCS, header = ImageHeader)
                MaskFileName = MapFitJobDir + "Masking-FITS-image.fits"
                LocalMaskImage.write(MaskFileName)


        ## if masking FITS image is given, make sure, that image covers all pixels of the previous run
        else:


            ## import FITS image describing chi2 values
            CubeDict = task_myXCLASSMapFit.FITSImport(Chi2FileName)
            Chi2FITSImage = CubeDict['Image']
            DecAxis = CubeDict['DecAxis']
            DecAxisUNIT = CubeDict['DecAxisUNIT']
            RAAxis = CubeDict['RAAxis']
            RAAxisUNIT = CubeDict['RAAxisUNIT']


            ## import masking FITS image
            MaskingFITSImage = fits.getdata(MaskFileName, ext = 0)
            MaskingFITSImageHeader = fits.getheader(MaskFileName)
            MaskingFITSImageWCS = WCS(MaskingFITSImageHeader)
            MaskingFITSImageObject = Slice(data = MaskingFITSImage, wcs = MaskingFITSImageWCS)

            # Debug:
            # print ("MaskingFITSImage.shape = ", MaskingFITSImage.shape)
            # print ("MaskingFITSImageHeader = ", MaskingFITSImageHeader)
            # print ("MaskingFITSImageWCS = ", MaskingFITSImageWCS)
            # print ("MaskingFITSImageObject.shape = ", MaskingFITSImageObject.shape)


            ## restrict masking image to coordinates of chi2 map
            MaskingFITSImageObject = MaskingFITSImageObject.subimage(xlo = RAAxis[0] * u.Unit(RAAxisUNIT), \
                                                                     xhi = RAAxis[-1] * u.Unit(RAAxisUNIT), \
                                                                     ylo = DecAxis[0] * u.Unit(DecAxisUNIT), \
                                                                     yhi = DecAxis[-1] * u.Unit(DecAxisUNIT))

            ## check, if new mask is smaller that chi2 map
            if (MaskingFITSImageObject.shape[0] != Chi2FITSImage.shape[0] or MaskingFITSImageObject.shape[1] != Chi2FITSImage.shape[1]):
                print ("\n\n\nError in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:")
                print ("\n\t\tThe FITS image used to select pixel does not!")
                print ("\n\t\tcover the whole region of the previous myXCLASSMapFit run")
                print ("\n\t\tAbort myXCLASSMapRedoFit function!")
                print ("\n\n")
                return


            ## write new masking FITS image to file, if dimension of mask is changed!
            elif (MaskingFITSImageObject.shape[0] != MaskingFITSImage.shape[0] or MaskingFITSImageObject.shape[1] != MaskingFITSImage.shape[1]):
                MaskFileName = MapFitJobDir + "Masking-FITS-image.fits"
                MaskingFITSImageObject.write(MaskFileName)
                LocalMaskImage = MaskingFITSImageObject


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## call myXCLASSMapFit function with new input parameters
    NewJobDir = task_myXCLASSMapFit.myXCLASSMapFitCore(MolfitsFileName = MolfitsFileName, \
                                                       FastFitFlag = FastFitFlag, \
                                                       FullFitFlag = FullFitFlag, \
                                                       ChannelIntegrationFlag = ChannelIntegrationFlag, \
                                                       daskFlag = daskFlag, \
                                                       ObsXMLFileName = ObsXMLFileName, \
                                                       AlgorithmXMLFileName = AlgorithmXMLFileName, \
                                                       NumberIteration = NumberIteration, \
                                                       NumberProcessors = NumberProcessors, \
                                                       clusterdef = clusterdef, \
                                                       FITSImageMaskFileName = MaskFileName, \
                                                       Threshold = Threshold, \
                                                       EstimateContinuumFlag = EstimateContinuumFlag, \
                                                       EstimateConNumParts = EstimateConNumParts, \
                                                       EstimateConNumEntireCubeFlag = EstimateConNumEntireCubeFlag, \
                                                       DataSmoothValue = DataSmoothValue, \
                                                       FreqMin = FreqMin, \
                                                       FreqMax = FreqMax, \
                                                       t_back_flag = t_back_flag, \
                                                       tBack = tBack, \
                                                       tSlope = tSlope, \
                                                       nH_flag = nH_flag, \
                                                       N_H = N_H, \
                                                       beta_dust = beta_dust, \
                                                       kappa_1300 = kappa_1300, \
                                                       DustFileName = DustFileName, \
                                                       BackgroundFileName = BackgroundFileName, \
                                                       ContPhenFuncID = ContPhenFuncID, \
                                                       ContPhenFuncParam1 = ContPhenFuncParam1, \
                                                       ContPhenFuncParam2 = ContPhenFuncParam2, \
                                                       ContPhenFuncParam3 = ContPhenFuncParam3, \
                                                       ContPhenFuncParam4 = ContPhenFuncParam4, \
                                                       ContPhenFuncParam5 = ContPhenFuncParam5, \
                                                       iso_flag = iso_flag, \
                                                       IsoTableFileName = IsoTableFileName, \
                                                       NumModelPixelXX = NumModelPixelXX, \
                                                       NumModelPixelYY = NumModelPixelYY, \
                                                       LocalOverlapFlag = LocalOverlapFlag, \
                                                       NoSubBeamFlag = NoSubBeamFlag, \
                                                       EmAbsPATH = EmAbsPATH)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if fast-fitting method was not used, copy pixel fit directories as well
    if (not FastFitFlag):


        ## check, if non-fast-fitting method was already used
        if ("Pixel-Fits" in MapFitFileListing):


            ## get list of pixel directories in new and old job directories
            OldPixelMapFitFileListing = os.listdir(MapFitJobDir + "Pixel-Fits/")
            NewPixelMapFitFileListing = os.listdir(NewJobDir + "Pixel-Fits/")


            ## find pixel directories which have the same pixel indices
            for NewLocalPixelDir in NewPixelMapFitFileListing:
                NewLocalPixelCood = NewLocalPixelDir.split("___")
                NewLocalPixelCood = NewLocalPixelCood[-1].strip()
                for OldLocalPixelDir in OldPixelMapFitFileListing:
                    OldLocalPixelCood = OldLocalPixelDir.split("___")
                    OldLocalPixelCood = OldLocalPixelCood[-1].strip()


                    ## WARNING: Maybe it is not sufficient to compare pixel indices only, i.e.
                    ## if (NewLocalPixelDir == OldLocalPixelDir):
                    if (NewLocalPixelCood == OldLocalPixelCood):


                        ## compress and move content of old pixel directory
                        OldLocalPixelDirFull = MapFitJobDir + "Pixel-Fits/" + OldLocalPixelDir + "/"
                        lt = time.localtime()
                        CurrentBackupDir = os.path.normpath(OldLocalPixelDirFull + "backup_for_RedoFit__" \
                                                            + time.strftime("%d-%m-%Y", lt) + "__" \
                                                            + time.strftime("%H-%M-%S", lt)) + "/"
                        cmdString = "mkdir " + CurrentBackupDir + "; "
                        cmdString += "mv " + OldLocalPixelDirFull + "*.* " + CurrentBackupDir + "; "
                        cmdString += "gzip -dr " + CurrentBackupDir
                        os.system(cmdString)

                        # Debug:
                        # print ("CurrentBackupDir = ", CurrentBackupDir)


                        ## copy content of new pixel directory to current old pixel directory
                        NewLocalPixelDirFull = NewJobDir + "Pixel-Fits/" + OldLocalPixelDir + "/"
                        cmdString = "mv " + NewLocalPixelDirFull + "*.* " + OldLocalPixelDirFull + "; "
                        os.system(cmdString)


        ## if not, copy pixel directories to old job directory
        else:
            cmdString = "mv " + NewJobDir + "Pixel-Fits/ " + MapFitJobDir + "; "
            os.system(cmdString)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## merge new myXCLASSMapFit run with previous run
    RemoveNewMapFitRunFlag = True


    ## get list of all fits file in the new MapFit run
    NewMapFitFileListing = os.listdir(NewJobDir)
    ListOfNewFITSFileNames = []
    for LocalFileName in NewMapFitFileListing:
        LowerLocalFileName = LocalFileName.lower()
        if (LowerLocalFileName.endswith(".fits")):
            ListOfNewFITSFileNames.append(LocalFileName)

    # Debug:
    # print ("ListOfNewFITSFileNames = ", ListOfNewFITSFileNames)


    ## rename old FITS files which were refitted and merge FITS files
    for LocalFileName in MapFitFileListing:
        LowerLocalFileName = LocalFileName.lower()
        if (LowerLocalFileName.endswith(".fits")):
            if (LocalFileName in ListOfNewFITSFileNames):


                ## print what you do
                print ("\nUpdate FITS file {:s} .. ".format(LocalFileName), end = "")


                ## rename existing FITS files with "REDO" phrase
                OldFITSFileName = LocalFileName.replace(".fits", "__REDO-OLD.fits")
                cmdString = "mv " + MapFitJobDir + LocalFileName + " " + MapFitJobDir + OldFITSFileName
                os.system(cmdString)


                ## merge parameter maps
                if (LowerLocalFileName.startswith("bestresult___")):


                    ## import FITS images
                    OldFITSImage = fits.getdata(MapFitJobDir + OldFITSFileName, ext = 0)
                    NewFITSImage = fits.getdata(NewJobDir + LocalFileName, ext = 0)


                    ## merge FITS images
                    NewFITSImage = numpy.where(numpy.isnan(NewFITSImage), OldFITSImage, NewFITSImage)


                    ## write new FITS image to file
                    OldFITSImageHeader = fits.getheader(MapFitJobDir + OldFITSFileName)
                    OldFITSImageWCS = WCS(OldFITSImageHeader)
                    NewFITSImageObject = Slice(value = NewFITSImage, wcs = OldFITSImageWCS)
                    NewFITSImageObject.write(MapFitJobDir + LocalFileName, overwrite = True)


                ## merge cubes
                else:
                    OldFITSCube = SpectralCube.read(MapFitJobDir + OldFITSFileName)
                    NewFITSCube = SpectralCube.read(NewJobDir + LocalFileName)


                    ## check, if both cubes have the same spatial dimensions
                    if (OldFITSCube.shape[1] != NewFITSCube.shape[1] or OldFITSCube.shape[2] != NewFITSCube.shape[2]):
                        print ("\n\n\t\tWARNING:  The new obs. data FITS cube {:s}".format(LocalFileName))
                        print ("\t\t          has a different number of pixels along the dec-direction and / or ra-direction")
                        print ("\t\t          compared to the old FITS cube!")
                        print ("\t\t          Shape of new obs. data cube: ", NewFITSCube[0, :, :].shape)
                        print ("\t\t          Shape of old obs. data cube: ", OldFITSCube[0, :, :].shape)
                        print ("\n\t\t          FITS cubes can not be merged!")
                        print ("\n\n")
                        RemoveNewMapFitRunFlag = False


                    ## check, if spectral axis is different
                    else:
                        NewSpectralAxis = NewFITSCube.spectral_axis.value
                        OldSpectralAxis = OldFITSCube.spectral_axis.value
                        CommonSpectralAxis = numpy.intersect1d(NewSpectralAxis, OldSpectralAxis)
                        if (len(CommonSpectralAxis) == 0):
                            print ("\n\n\nError in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:")
                            print ("\n\t\tThe new and the old FITS cubes {:s}".format(LocalFileName))
                            print ("\n\t\thave different spectral axis, which do not overlap")
                            print ("\n\n")
                            return
                        else:


                            ## merge FITS cubes
                            nt = OldFITSCube.shape[0]
                            ny = OldFITSCube.shape[1]
                            nx = OldFITSCube.shape[2]
                            NewArray = numpy.zeros((nt, ny, nx))
                            LocalUnit = OldFITSCube.unit
                            # RefittedPixelCoord = numpy.argwhere(LocalMaskImage)


                            ## if both cubes have the same spectral axis
                            if (OldFITSCube.shape[0] == NewFITSCube.shape[0]):
                                for x in range(nx):
                                    for y in range(ny):
                                        if (numpy.isnan(LocalMaskImage[y, x])):
                                            NewArray[:, y, x] = OldFITSCube[:, y, x].value
                                        else:
                                            NewArray[:, y, x] = NewFITSCube[:, y, x].value
                                OldFITSCube = SpectralCube(data = NewArray * LocalUnit, wcs = OldFITSCube.wcs, header = OldFITSCube.header)


                            ## continue here, if spectral axis are different
                            else:
                                for x in range(nx):
                                    for y in range(ny):
                                        if (numpy.isnan(LocalMaskImage[y, x])):
                                            NewArray[:, y, x] = OldFITSCube[:, y, x].value
                                        else:
                                            for Freq in CommonSpectralAxis:
                                                i1 = numpy.argmin(numpy.abs(NewSpectralAxis - Freq))
                                                i2 = numpy.argmin(numpy.abs(OldSpectralAxis - Freq))
                                                NewArray[i2, y, x] = NewFITSCube[i1, y, x].value
                                OldFITSCube = SpectralCube(data = NewArray, wcs = OldFITSCube.wcs, header = OldFITSCube.header)


                            ## write new FITS cube to file
                            OldFITSCube.write(MapFitJobDir + LocalFileName)


                ## print what you do
                print ("done!")


    ##====================================================================================================================================================
    ## remove job directory of new myXCLASSMapFit fun
    if (RemoveNewMapFitRunFlag):
        cmdString = "rm -rf " + NewJobDir + "; "
        cmdString += "rm -rf " + MapFitJobDir + "Masking-FITS-image.fits"
        os.system(cmdString)


    ##====================================================================================================================================================
    ## print some information to screen
    print ("\n\nPlease note, the new created files are stored in the given")
    print ("myXCLASSMapRedoFit job directory " + chr(34) + MapFitJobDir + chr(34) + "!\n\n")


    ##====================================================================================================================================================
    ## define return variables
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Simplified CASA interface for MAGIX with myXCLASS program to fit a complete data cube instead of a single spectrum
##
def myXCLASSMapRedoFit(JobNumber, PixelList, NumberIteration, AlgorithmXMLFile, MolfitsFileName, experimentalData, Threshold, FreqMin, FreqMax, \
                       t_back_flag, tBack, tslope, BackgroundFileName, N_H, beta_dust, kappa_1300, DustFileName, \
                       ContPhenFuncID, ContPhenFuncParam1, ContPhenFuncParam2, ContPhenFuncParam3, ContPhenFuncParam4, \
                       ContPhenFuncParam5, iso_flag, IsoTableFileName, NumModelPixelXX, NumModelPixelYY, LocalOverlapFlag, \
                       NoSubBeamFlag):
    """

This function offers the possibility to redo one or more so called pixel fits of a
previous myXCLASSMapFit run. The function performs fits for selected pixels and
recreates the different parameter maps using the new optimized parameter values. The
user has to define the pixel coordinates, which should be re-fitted, the job number
of the previous myXCLASSMapFit and the new molfit and algorithm XML file.


input parameters:
-----------------

    - JobNumber:            Job number of a previous myXCLASSMapFit run which should
                            be improved.

    - PixelList:            list of all pixel coordinates which should be re-fitted.
                            Each coordinate consists of two numbers separated by a
                            comma, where the first number corresponds to the right
                            ascension and second to the declination. Different pixel
                            coordinates are enclosed by squared brackets and separated
                            by commas.

    - NumberIteration:      max. number of iterations (default: 50).

    - AlgorithmXMLFile:     only necessary, if the user wants to use another fit
                            algorithm (than Levenberg-Marquardt) for fitting. Therefore,
                            the path and name of a MAGIX xml-file defining settings
                            for an algorithm or algorithm chain has to be given. (A
                            relative path has to be defined relative to the current
                            working directory!)

                            NOTE, if the user specify a xml file, the number of
                            iterations given by the parameter "NumberIteration" is
                            ignored. The number of iteration is then given by the xml
                            file. In order to use the implemented fit algorithm
                            (Levenberg-Marquardt) clear the AlgorithmXMLFile parameter,
                            i.e. AlgorithmXMLFile = "", and define the max. number of
                            iterations by using parameter "NumberIteration".

    - MolfitsFileName:      path and name of the extended molfit file, including the
                            source size, the rotation temperature the column density,
                            the velocity width, the velocity offset and the flag
                            indicating if the current component is considered for
                            core "c" or foreground "f".

                            In contrast to the format of the molfit file the extended
                            molfit file required by the myXCLASSMapFit function contains
                            one (three) additional column(s) for each parameter of
                            each component.

                            A detailed description of the extended molfit file required
                            by the myXCLASSMapFit function can be found in the manual.

                            NOTE, a relative path has to be defined relative to the
                            current working directory!

    - Threshold:            defines a threshold for a pixel. If the spectrum of a pixel
                            has a intensity lower than the value defined by this parameter
                            the pixel is not fitted (ignored).

    - FreqMin:              start frequency of simulated spectrum (in MHz), (default: 1.e99).

                            Please note, if no start frequency is given, or if start frequency is
                            lower or equal to the end frequency, the myXCLASSMapRedoFit  function will
                            use the max. frequency range defined in the FITS header. Additionally,
                            the step size, i.e. the difference between two frequencies is taken
                            from the FITS header.

    - FreqMax:              end frequency of simulated spectrum (in MHz), (default: 1.e99).

    - t_back_flag:          ("True"/"False"), defines, if the user defined background
                            temperature Tbg and temperature slope Tslope given
                            by the input parameters tBack and tslope describe the
                            continuum contribution completely (t_back_flag = "True")
                            or not (t_back_flag = "False") (default: "True").

    - tBack:                background temperature (in K), (default: 1.e99).

    - tslope:               temperature slope (dimensionless), (default: 1.e99).

    - BackgroundFileName:   path and name of file describing background (default: "").

    - N_H:                  Hydrogen column density (in cm^{-2}), (default: 0.0).

    - beta_dust:            beta for dust (dimensionless), (default: 0.0).

    - kappa_1300:           kappa (cm^2 g^{-1}), (default: 0.0).

    - DustFileName:         path and name of file describing the optical depth
                            of dust (default: None).

    - ContPhenFuncID:       describes which phenomenological function is used
                            to describe the continuum (default: None).

    - ContPhenFuncParam1:   first parameter for phenomenological function
                            describing the continuum (default: None).

    - ContPhenFuncParam2:   second parameter for phenomenological function
                            describing the continuum (default: None).

    - ContPhenFuncParam3:   third parameter for phenomenological function
                            describing the continuum (default: None).

    - ContPhenFuncParam4:   fourth parameter for phenomenological function
                            describing the continuum (default: None).

    - ContPhenFuncParam5:   fifth parameter for phenomenological function
                            describing the continuum (default: None).

    - iso_flag:             use isotopologues ("True"/"False"). If iso_flag is set to
                            "False" the ratios are all set to 1 (default: "False").

    - IsoTableFileName:     path and name of an ASCII file including the iso ratios
                            between certain molecules. The so-called "iso ratio file"
                            defines the iso ratios between molecules. The ASCII file
                            consists of three columns, where the first two columns
                            indicates the molecules, respectively. The third column
                            defines the ratio for both molecules. Here, the first
                            41 characters belong to the first column, the next
                            41 characters to the second column. The further character
                            belongs to the third column.

                            If no path and name is specified (default), the so-called iso-flag
                            is set to "False".

    - NumModelPixelXX:      used for sub-beam modeling, describes the number
                            of pixels used in x-direction (default: 100).

    - NumModelPixelYY:      used for sub-beam modeling, describes the number
                            of pixels used in y-direction (default: 100).

    - LocalOverlapFlag:     flag indicates if local-overlap description is
                            used or not (default: False).

    - NoSubBeamFlag:        do not use sub-beam description (default: True).


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("JobNumber = ", JobNumber)
    # print ("PixelList = ", PixelList)
    # print ("NumberIteration = ", NumberIteration)
    # print ("AlgorithmXMLFile = ", AlgorithmXMLFile)
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("Threshold = ", Threshold)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("t_back_flag = ", t_back_flag)
    # print ("tBack = ", tBack)
    # print ("tslope = ", tslope)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("iso_flag = ", iso_flag)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)


    ## for later releases


    ##====================================================================================================================================================
    ## initialize working variables
    nH_flag = False
    myXCLASSMapRedoFitCore(JobNumber = JobNumber, PixelList = PixelList, NumberIteration = NumberIteration, AlgorithmXMLFile = AlgorithmXMLFile, \
                           MolfitsFileName = MolfitsFileName, Threshold = Threshold, FreqMin = FreqMin, \
                           FreqMax = FreqMax, t_back_flag = t_back_flag, tBack = tBack, tSlope = tslope, BackgroundFileName = BackgroundFileName, \
                           nH_flag = nH_flag, N_H = N_H, beta_dust = beta_dust, kappa_1300 = kappa_1300, DustFileName = DustFileName, \
                           ContPhenFuncID = ContPhenFuncID, ContPhenFuncParam1 = ContPhenFuncParam1, ContPhenFuncParam2 = ContPhenFuncParam2, \
                           ContPhenFuncParam3 = ContPhenFuncParam3, ContPhenFuncParam4 = ContPhenFuncParam4, ContPhenFuncParam5 = ContPhenFuncParam5, \
                           iso_flag = iso_flag, IsoTableFileName = IsoTableFileName, \
                           NumModelPixelXX = NumModelPixelXX, NumModelPixelYY = NumModelPixelYY, LocalOverlapFlag = LocalOverlapFlag, \
                           NoSubBeamFlag = NoSubBeamFlag)



    ##====================================================================================================================================================
    ## define return variables
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------

