#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module contains all subroutines to fit a complete data cube instead of single spectra
##  Copyright (C) 2012 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function FastMapFit.__init__:                     initialize class FastMapFit
##      - function FastMapFit.InitializeCluster:            initialize dask cluster
##      - function FastMapFit.GetParameterMapsFromFile:     get parameter maps from file
##      - function FastMapFit.ImportAlgSettings:            (fast-fitting only): analyze algorithm xml file
##      - function FastMapFit.PrepareFit:                   prepare map fit
##      - function FastMapFit.SplitString:                  split a string into fixed length chunks
##      - function FastMapFit.InitializeXCLASS:             (full fitting only): initialize XCLASS package
##      - function FastMapFit.PrepareXCLASS:                (full fitting only): copy stored XCLASS variables to XCLASS interface
##      - function FastMapFit.UpdateMolfitFile:             (normal fitting only): update molfit file with parameter maps
##      - function FastMapFit.UpdateIsoRatioFile:           (normal fitting only): update iso ratio file
##      - function FastMapFit.UpdateParameter:              (normal fitting only): update parameter value
##      - function FastMapFit.StartFastFits:                (fast-fitting only): start fast fitting
##      - function FastMapFit.CallbackOptAlg:               (fast-fitting only): callback function for some optimization algorithms
##      - function FastMapFit.CallOptimizationAlgorithms:   (fast-fitting only): call selected optimization algorithms
##      - function FastMapFit.brute_dict_configs:           (non-MapFit, brute only): used for brute-force, generate parameter grid
##      - function FastMapFit.CallBruteForce:               (non-MapFit, brute only): call brute force algorithm
##      - function FastMapFit.log_prior:                    (non-MapFit, emcee and UltraNest only): used for emcee, compute the log-prior
##      - function FastMapFit.log_probability:              (non-MapFit, emcee and UltraNest only): used for emcee, compute the full log-prob. func.
##      - function FastMapFit.ComputeMode:                  (non-MapFit, emcee and UltraNest only): calculate mode(s) of list
##      - function FastMapFit.ComputeHPD:                   (non-MapFit, emcee and UltraNest only): compute highest prob. density region given
##      - function FastMapFit.make_corner_plot:             (non-MapFit, emcee and UltraNest only): compute errors and create corner plot
##      - function FastMapFit.Callemcee:                    (non-MapFit, emcee only): call emcee package
##      - function FastMapFit.ultranest_prior_transform:    (non-MapFit, UltraNest only): prior transform
##      - function FastMapFit.CallUltraNest:                (non-MapFit, UltraNest only): call UltraNest package
##      - function FastMapFit.pygad_fitness_func:           (non-MapFit, PyGAD only): fitness function for PyGAD
##      - function FastMapFit.pygad_on_generation:          (non-MapFit, PyGAD only): helper function for PyGAD
##      - function FastMapFit.CallPyGAD:                    (non-MapFit, PyGAD only): call PyGAD package
##      - function FastMapFit.EntireFITSCubeFit:            (non-MapFit only): call different optimization algorithms for CubeFit function
##      - function FastMapFit.NonMapFitWorkers:             (non-MapFit only): start worker for the non-MapFit fitting
##      - function FastMapFit.GradientJAC:                  compute the Jacobian matrix for optimization algorithms which require a gradient
##      - function FastMapFit.StartWorker:                  start worker using multiprocessing package
##      - function FastMapFit.PixelFitFunction:             (fast-fitting only): function called for each pixel
##      - function FastMapFit.FastFitFuncMin:               (fast-fitting only): fit function for scipy optimization function
##      - function FastMapFit.FullFitFuncMin:               (full fitting only): fit function for scipy optimization function
##      - function FastMapFit.StartNormalFitting:           (normal fitting only): start normal fitting
##      - function FastMapFit.NormalPixelFitFunction:       (normal fitting only): function called MAGIX for each pixel
##      - function FastMapFit.GetMolfitFileOfBestResult:    (normal fitting only): determines the best fit result from a previous pixel fit
##      - function FastMapFit.ReadParmXMLFile:              (normal fitting only): read parameter values from xml file
##      - function FastMapFit.ExportResults:                write results to job directory
##      - function FastMapFit.SmoothParameterMaps:          smooth parameter maps
##      - function PlottingGUI.__init__:                    initialize class PlottingGUI
##      - function PlottingGUI.keyPressEvent:               redefine return-press event
##      - function PlottingGUI.ButtonWidget:                handle button event of widget
##      - function PlottingGUI.ComboWidget:                 handle combo box event of widget
##      - function PlottingGUI.LineEditWidget:              handle line edit event of widget
##      - function PlottingGUI.SliderWidget:                handle slider event of widget
##      - function PlottingGUI.connectSpecPlot:             connect all the stored connection ids
##      - function PlottingGUI.dispose:                     quit gui
##      - function PlottingGUI.disconnect:                  disconnect all the stored connection ids
##      - function PlottingGUI.replotGUI:                   replot GUI
##      - function PlottingGUI.OnKeySpectrum:               defines what happens on key event
##      - function PlottingGUI.MouseMoveSpectrum:           define what to do when mouse is moved in spectrum plot
##      - function PlottingGUI.MouseMoveMap:                define what to do when mouse is moved in map plot
##      - function PlottingGUI.getParam:                    get parameter
##      - function PlottingGUI.onclickSpectrum:             defines what happens on mouse click
##      - function PlottingGUI.onclickMap:                  define what happens on mouse click in map plot
##      - function PlottingGUI.FITSPixelSpecImport:         import pixel spectrum from fits file
##      - function StartPlottingGUI:                        start GUI to inspect results of myXCLASSMapFit function
##      - function GetPartitionFunctionFunc:                determine interpolation object for partition function
##      - function CheckBool:                               check given string for boolean
##      - function CheckNumber:                             check given string for number
##      - function AnaylzeObsXMLFile:                       import and analyze obs. data file(s) and extract parameters from obs. xml file
##      - function local_subcube_from_regions:              local version of the spectral_cube function "subcube_from_regions"
##      - function local_regionlist_to_single_region:       recursively merge a region list into a single compound region,
##                                                          required by function "local_subcube_from_regions"
##      - function SubCubeFromCube:                         extracts a subcube from a cube using a masking cube
##      - function Subimage_from_regions:                   extracts a subimage from an image using ds9 region
##      - function FITSImport:                              import fits file
##      - function GetHelperCubes:                          import additional cubes, e.g. cubes describing the background spectrum
##      - function EstimateContinuumCube:                   estimate continuum level for all pixels of given FITS cube
##      - function ContinuumFunction:                       continuum function
##      - function EstimateContinuumPixel:                  estimate continuum for each pixel
##      - function JanskyPerBeamInKelvin:                   converts Jansky in Kelvin and vis versa
##      - function ReadClusterdefFile:                      reads in file that contains cluster definition
##      - function myXCLASSMapFitCore:                      core myXCLASSMapFit function
##      - function myXCLASSMapFit:                          Simplified CASA interface for MAGIX with myXCLASS program to fit a
##                                                          complete data cube instead of a single spectrum
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2013-07-25      initial version
##  T. Moeller      2014-04-09      improved version
##  T. Moeller      2019-02-26      improved version
##  T. Moeller      2020-01-03      porting to python 3.x
##  T. Moeller      2021-11-17      added fast map fitting procedure and GUI
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import os                                                                                   ## import os package
os.environ['OPENBLAS_NUM_THREADS'] = '1'
import sys                                                                                  ## import sys package
import time                                                                                 ## import package date and time
import datetime                                                                             ## import datetime package
import copy                                                                                 ## import copy package
import warnings                                                                             ## import warnings package
import numpy                                                                                ## import numpy package
from itertools import product
if (not 'matplotlib' in sys.modules):                                                       ## is matplotlib already loaded?
    import matplotlib                                                                       ## import matplotlib package
    matplotlib.use("Agg")                                                                   ## avoid display error
else:
    import matplotlib                                                                       ## import matplotlib package
from matplotlib.backends.backend_qt5agg import (FigureCanvasQTAgg as FigureCanvas, NavigationToolbar2QT as NavigationToolbar)
from matplotlib.backends.qt_compat import QtWidgets, QtCore, QtGui                          ## get some PyQt5 functions
# from PyQt5 import (QtWidgets, QtCore, QtGui)
try:
    from matplotlib.cm import register_cmap
except ImportError:
    from matplotlib import colormaps
from matplotlib.colors import LogNorm                                                       ## needed for log. colorbars
from matplotlib.lines import Line2D                                                         ## used for corner plots
import socket                                                                               ## import socket package
import pickle                                                                               ## load pickle package
import multiprocessing                                                                      ## import multiprocessing package
from threading import current_thread                                                        ## used for PyGAD
from scipy.interpolate import interp1d                                                      ## import interp1d package from scipy
from scipy.special import ndtr                                                              ## load scipy.special package
from scipy.optimize import brute, basinhopping, minimize, least_squares, fmin               ## import optimization functions form scipy
from scipy.optimize import differential_evolution, shgo, dual_annealing, curve_fit          ## import optimization functions form scipy
from scipy.ndimage.filters import gaussian_filter                                           ## import smoothing functions from scipy
from scipy.ndimage import uniform_filter, median_filter, convolve1d                         ## import smoothing functions from scipy
from spectral_cube import BooleanArrayMask
from spectral_cube import SpectralCube, Slice                                               ## import SpectralCube package
from spectral_cube.utils import SpectralCubeWarning, WCSMismatchWarning, SliceWarning, StokesWarning, WCSMismatchWarning
import astropy.units as u                                                                   ## import units package from astropy
from astropy.io import fits                                                                 ## import fits from astropy
from astropy.wcs import WCS                                                                 ## import wcs from astropy
from astropy.convolution import Gaussian1DKernel                                            ## import Gaussian1DKernel from astropy
from astropy.coordinates import Angle                                                       ## needed for coordinate description
from regions import Regions                                                                 ## import regions package
import pylab                                                                                ## import pylab package
import sqlite3                                                                              ## import sqlite3 package
from . import task_MAGIX                                                                    ## import package MAGIX
from . import task_myXCLASS                                                                 ## import package myXCLASS
from . import task_myXCLASSFit                                                              ## import package myXCLASSFit
from . import task_LineIdentification                                                       ## import package task_LineIdentification
import matplotlib.pyplot as plt                                                             ## import pyplot from matplotlib
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## Class FastMapFit
class FastMapFit(object):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize main class
    def __init__(self, ParameterDict, RemoveContinuumFlag = False, PrintFitStatusFlag = True, dbgFlag = False):
        """

    input parameters:
    -----------------

        - ParameterDict:            dictionary describing parameters

        - RemoveContinuumFlag:      (optinal) flag for removing continuum, (default: False)

        - PrintFitStatusFlag:       (optinal) flag indicating output of fit status message, (default: True)

        - dbgFlag:                  (optinal) flag for debugging, (default: False)


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("ParameterDict = ", ParameterDict)
        # print ("RemoveContinuumFlag = ", RemoveContinuumFlag)
        # print ("PrintFitStatusFlag = ", PrintFitStatusFlag)
        # print ("dbgFlag = ", dbgFlag)


        ## initialize global variables in this class
        self.NameOfFunction = None                                                          ## name of function
        self.myXCLASSMapFitJobDir = None                                                    ## path and name of current job directory
        self.MAGIXrootDir = None                                                            ## path and name of MAGIX root directory
        self.FastFitFlag = None                                                             ## flag indicating fast fitting
        self.daskFlag = None                                                                ## flag indicating usage of dask package
        self.ParallelMethod = None                                                          ## method usd for parallelization
        self.DataFileFormat = "fits"                                                        ## format of obs. data file
        self.NoFITSCubeFlag = False                                                         ## flag indicating fitting of single spectra
        self.OrigUnits = None                                                               ## list of units of obs. data cubes
        self.OrigRestFreq = None                                                            ## list of rest frequencies
        self.TotalNumObsDataSpecChannels = None                                             ## total number of all spectral channels
        self.Fit_Algorithm = None                                                           ## used fit algorithm
        self.FitGradientFlag = None                                                         ## gradient flag
        self.FitAlgSettings = None                                                          ## fit algorithm settings
        self.Fit_LSFlag = None                                                              ## flag indicating least-squares algorithm (=0),
                                                                                            ## global algorithm (=1), and blank (=3)
        self.Fit_Variation = None                                                           ## used fit variation
        self.Fit_limit = None                                                               ## limit of chi^2
        self.Fit_NumIter = None                                                             ## number of iterations
        self.Fit_NumProc = None                                                             ## number of cores
        self.Fit_Host = None                                                                ## path and name of host file
        self.Fit_Chi2 = None                                                                ## store chi2 function
        self.Fit_Chi2LogFlag = None                                                         ## use .chi2.log file
        self.pygad_last_fitness = None                                                      ## used for PyGAD: last fitness
        self.GradientVector = None                                                          ## gradient vector
        self.VectorizedFlag = None                                                          ## flag indicating list of param. vectors for likelihood func.
        self.UseCallbackFuncFlag = None                                                     ## flag indicating usage of callback function
        self.IsoFlag = None                                                                 ## flag indicating usage of iso ratio file
        self.ListOfInitialMaps = None                                                       ## list of parameters of initial maps
        self.SQLParamArray = None                                                           ## sql parameters from the molfit file
        self.MolfitsFileName = None                                                         ## path and name of molfit file
        self.MolfitHeaderLinesList = None                                                   ## sql header of molfit file
        self.MolfitCoord = None                                                             ## parameters to find molfit param. (used for param maps)
        self.MolNameList = None                                                             ## list of all molecules in molfit and iso ratio file
        self.IsoRatioFileName = None                                                        ## path and name of iso ratio file
        self.NumIsoMaster = None                                                            ## number of iso master molecules and isotopologues
        self.GlobalIsoParam = None                                                          ## global ratio parameters
        self.IsoParameters = None                                                           ## parameters of iso ratio file
        self.IsoFitParameters = None                                                        ## fitted iso ratios with limits
        self.IsotopologuesIn = None                                                         ## list of isotopologues
        self.IsoMoleculeIn = None                                                           ## list of iso master molecules
        self.PartFunc = None                                                                ## interpolated partition function objects for all molecules
        self.MolfitFitParameters = None                                                     ## list of fit parameters in molfit file
        self.FitParameterNames = []                                                         ## list of the fit parameter names
        self.AllMolfitParameters = None                                                     ## numpy array describing all molfit parameters
        self.ListDistances = None                                                           ## list of used distances in the molfit file
        self.TrotIndices = None                                                             ## list of indices describing positions of temperatures
        self.NumEmissionComp = None                                                         ## number of emission components
        self.MolIndexPerComp = None                                                         ## indices of molecules in transition paramters
        self.pBLow = None                                                                   ## lower limits of fit parameters
        self.pBUp = None                                                                    ## upper limits of fit parameters
        self.RangeList = None                                                               ## list of parameter ranges
        self.RangeTupleList = None                                                          ## list of parameter ranges as tuples
        self.dbFile = None                                                                  ## path and name of database file
        self.LogFile = None                                                                 ## for non-MapFit usage: log file
        self.LogChi2File = None                                                             ## for non-MapFit usage: chi2 log file
        self.FuncCallCounter = None                                                         ## for non-MapFit usage: counter for func. calls
        self.ListOfCubes = None                                                             ## list of FITS cubes for each freq. range
        self.ListOfSpectralAxis = None                                                      ## list of spectral axis for each freq. range
        self.RangeParam = None                                                              ## list of range parameters
        self.PixelMask = None                                                               ## map describing pixels which are fitted
        self.NumYPixels = None                                                              ## max. number of fitted pixels in y-direction
        self.NumXPixels = None                                                              ## max. number of fitted pixels in x-direction
        self.dec = None                                                                     ## list of dec coordinates for the selected region
        self.ra = None                                                                      ## list of ra coordinates for the selected region
        self.TotalNumPixel = None                                                           ## max. total number of fitted pixels
        self.ListOfSelectedPixels = None                                                    ## list of indices of selected pixels
        self.PrintFitStatusFlag = None                                                      ## flag indicating output of fit status message
        self.NumFitParameters = None                                                        ## total number of pixels
        self.FitParameterMaps = None                                                        ## maps describing fitted parameter values for each pixel
        self.ParameterMapHeader = None                                                      ## FITS header for parameter maps
        self.ParameterMapWCS = None                                                         ## WCS for parameter maps
        self.Chi2Map = None                                                                 ## map describing chi2 values for each pixel
        self.SyntheticSpectraCubeList = None                                                ## cube describing synthetic spectra for each pixel
        self.manager = None                                                                 ## object for parallelization method "process ""
        self.return_dict = None                                                             ## list containing return values used for method "process"
        self.BigCubeIndexList = None                                                        ## list of indices within big cube
        self.Chi2SpectraCubeList = None                                                     ## cube describing squared residual for each pixel
        self.LimitIndices = None                                                            ## list of indices describing limits of each freq. range
        self.NumParamMapIterations = None                                                   ## total number of iterations for parameter map smoothing
        self.ParamSmoothMethodParam = None                                                  ## parameters for smoothing algorithm
        self.ParamSmoothMethod = None                                                       ## method for smoothing
        self.FullFitFlag = None                                                             ## full fit flag
        self.ChannelIntegrationFlag = None                                                  ## flag for channel integration
        self.ParameterNameLookUpDict = None                                                 ## look-up directory for parameter names
        self.CurrentFitIteration = 0                                                        ## current iteration step
        self.BestFitResultsDict = None                                                      ## dictionary with best fit results
        self.MolfitTranslationTable = None                                                  ## translation table between content of molfit file and
                                                                                            ## myXCLASS parameter array
        self.dbgFlag = False                                                                ## debug flag
        self.eps = numpy.finfo(numpy.float32).tiny                                          ## machine limit for 32-bit floating point number


        ## deactivate silencing warnings
        warnings.filterwarnings(action = 'ignore', category = SpectralCubeWarning, append = True)


        ## get flag indicating output of fit status message
        self.PrintFitStatusFlag = PrintFitStatusFlag


        ## get flag indicating debugging
        self.dbgFlag = dbgFlag


        ## use dask package?
        self.daskFlag = ParameterDict['daskFlag']


        ## get method used for parallelization
        self.ParallelMethod = ParameterDict['ParallelMethod']
        if (self.dbgFlag):
            self.ParallelMethod = "dbg"


        ## get MAGIX root directory
        self.MAGIXrootDir = task_myXCLASS.GetMAGIXRootDir()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get fast fitting flag
        self.FastFitFlag = ParameterDict['FastFitFlag']
        self.FullFitFlag = ParameterDict['FullFitFlag']
        if (self.FastFitFlag and (not self.FullFitFlag)):
            print ("Use fast-fitting method!")


        ## get full-fit flag
        if (self.FullFitFlag):
            self.ChannelIntegrationFlag = ParameterDict['ChannelIntegrationFlag']


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## prepare map fit
        self.PrepareFit(ParameterDict, RemoveContinuumFlag)


        ## print some informations to screen
        if (not self.NoFITSCubeFlag):
            print ("\n\n\nNumber of pixels in DEC-direction: {:d}".format(self.NumYPixels))
            print ("Number of pixels in RA-direction:  {:d}\n\n".format(self.NumXPixels))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## prepare numpy array for fit parameters
        self.FitParameterMaps = numpy.zeros((self.NumYPixels, self.NumXPixels, self.NumFitParameters))

        # Debug:
        # print ("\n\nself.NumFitParameters = ", self.NumFitParameters)
        # print ("self.MolfitFitParameters = ", self.MolfitFitParameters)
        # print ("self.IsoFitParameters = ", self.IsoFitParameters)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## prepare molfit fit parameters maps
        ## self.MolfitFitParameters.append([CompCounter, ParameterCounter, LogFlag, LowerLimit, UpperLimit, value, Name])
        ParamCounter = (-1)
        for FitParameter in self.MolfitFitParameters:
            ParamCounter += 1

            # Debug:
            # print ("\n\n\nFitParameter = ", FitParameter)
            # print ("ParamCounter = ", ParamCounter)


            ## initialize fit parameter map
            value = FitParameter[5]
            self.FitParameterMaps[:, :, ParamCounter] = value


            ## add parameters from initial parameter map
            if (len(self.ListOfInitialMaps) > 0):
                CompCounter = FitParameter[0]
                if (self.FullFitFlag):
                    NameParam = FitParameter[6]
                    try:
                        ParameterCounter = self.ParameterNameLookUpDict[NameParam.lower()]
                    except:
                        ParameterCounter = None
                else:
                    # try:
                    #     ParamCounter = ListOfParamNames.index(NameParam.lower())
                    #     break
                    # except:
                    #     ParamCounter = None
                    ParameterCounter = (FitParameter[1] - 1)
                LocalKey = "{:s}|{:d}|{:d}".format("molfit", CompCounter, ParameterCounter)
                try:
                    LocalMap = self.ListOfInitialMaps[LocalKey]
                except:
                    LocalMap = None
                if (LocalMap is not None):
                    LogFlag = FitParameter[2]
                    LowerLimit = FitParameter[3]                                        ## lower limit of current fit parameter
                    UpperLimit = FitParameter[4]                                        ## upper limit of current fit parameter
                    self.FitParameterMaps[:, :, ParamCounter] = LocalMap

                    # Debug:
                    # print ("LocalKey = ", LocalKey)
                    # print ("LocalMap[47, 49] = ", LocalMap[47, 49])
                    # print ("LogFlag = ", LogFlag)


                    ## convert to log scale?
                    if (LogFlag):
                        self.FitParameterMaps[:, :, ParamCounter] = numpy.where(self.FitParameterMaps[:, :, ParamCounter] <= 0.0, \
                                                                                0.0, \
                                                                                numpy.log10(self.FitParameterMaps[:, :, ParamCounter]))
                    ## replace nan by molfit value
                    self.FitParameterMaps[:, :, ParamCounter] = numpy.where(numpy.isnan(self.FitParameterMaps[:, :, ParamCounter]),
                                                                            numpy.log10(value), \
                                                                            self.FitParameterMaps[:, :, ParamCounter])
                    ## check, lower limit
                    self.FitParameterMaps[:, :, ParamCounter] = numpy.where(self.FitParameterMaps[:, :, ParamCounter] < LowerLimit, \
                                                                            LowerLimit, \
                                                                            self.FitParameterMaps[:, :, ParamCounter])
                    ## check, upper limit
                    self.FitParameterMaps[:, :, ParamCounter] = numpy.where(self.FitParameterMaps[:, :, ParamCounter] > UpperLimit, \
                                                                            UpperLimit, \
                                                                            self.FitParameterMaps[:, :, ParamCounter])
            ## apply mask to parameter map
            self.FitParameterMaps[:, :, ParamCounter] = numpy.where(self.PixelMask, self.FitParameterMaps[:, :, ParamCounter], numpy.nan)

            # Debug:
            # print ("\nMolfit-Parameters:")
            # print ("len(self.ListOfInitialMaps) = ", len(self.ListOfInitialMaps))
            # print ("ParamCounter, self.FitParameterMaps[0, 0, ParamCounter] = ", ParamCounter, self.FitParameterMaps[0, 0, ParamCounter])


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## self.IsoFitParameters[i] = [IsoIndex, LowerLimitIsoRatio, UpperLimitIsoRatio, RatioValue]
        for IsoParameter in self.IsoFitParameters:
            ParamCounter += 1

            # Debug:
            # print ("IsoParameter = ", IsoParameter)


            ## initialize fit parameter map
            RatioValue = IsoParameter[3]
            self.FitParameterMaps[:, :, ParamCounter] = RatioValue


            ## add parameters from initial parameter map
            ## HERE:
            if (len(self.ListOfInitialMaps) > 0):
                IsoIndex = IsoParameter[0]                                              ## iso index of isotopologue
                if (IsoIndex < 0):
                    IsoIndex = abs(IsoIndex) - 1
                    LocalKey = "{:s}|{:d}".format("globalisoratio", IsoIndex)
                else:
                    LocalKey = "{:s}|{:d}".format("isoratio", IsoIndex)
                try:
                    LocalMap = self.ListOfInitialMaps[LocalKey]
                except:
                    LocalMap = None
                if (LocalMap is not None):
                    LowerLimit = IsoParameter[1]                                        ## lower limit of current fit parameter
                    UpperLimit = IsoParameter[2]                                        ## upper limit of current fit parameter
                    self.FitParameterMaps[:, :, ParamCounter] = LocalMap

                    # Debug:
                    # print ("\nLocalKey = ", LocalKey)
                    # print ("LocalMap = ", LocalMap)


                    ## replace nan by molfit value
                    self.FitParameterMaps[:, :, ParamCounter] = numpy.where(numpy.isnan(self.FitParameterMaps[:, :, ParamCounter]),
                                                                            RatioValue, self.FitParameterMaps[:, :, ParamCounter])

                    ## check, lower limit
                    self.FitParameterMaps[:, :, ParamCounter] = numpy.where(self.FitParameterMaps[:, :, ParamCounter] < LowerLimit, \
                                                                            LowerLimit, \
                                                                            self.FitParameterMaps[:, :, ParamCounter])
                    ## check, upper limit
                    self.FitParameterMaps[:, :, ParamCounter] = numpy.where(self.FitParameterMaps[:, :, ParamCounter] > UpperLimit, \
                                                                            UpperLimit, \
                                                                            self.FitParameterMaps[:, :, ParamCounter])
            ## apply mask to parameter map
            self.FitParameterMaps[:, :, ParamCounter] = numpy.where(self.PixelMask, self.FitParameterMaps[:, :, ParamCounter], numpy.nan)

            # Debug:
            # print ("\nIso-Parameters:")
            # print ("len(self.ListOfInitialMaps) = ", len(self.ListOfInitialMaps))
            # print ("ParamCounter, self.FitParameterMaps[:, :, ParamCounter] = ", ParamCounter, self.FitParameterMaps[:, :, ParamCounter])


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## prepare numpy array for chi2 values
        self.Chi2Map = numpy.ones((self.NumYPixels, self.NumXPixels)) * numpy.nan


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## prepare numpy array for chi2 values
        IndexRange = numpy.arange(self.TotalNumPixel)

        # Debug:
        # print ("\nIndexRange[:] = ", IndexRange[:])


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## prepare FITS cubes for synthetic spectra
        self.SyntheticSpectraCubeList = []
        self.Chi2SpectraCubeList = []
        self.LimitIndices = []
        FullSpectralAxis = None
        FreqMinIndex = 0
        for LocalRangeID, ObsXMLParameterList in enumerate(self.RangeParam):
            LocalSpectralAxis = self.ListOfSpectralAxis[LocalRangeID].value
            LocalSyntCopy = numpy.zeros((len(LocalSpectralAxis), self.NumYPixels, self.NumXPixels))
            self.SyntheticSpectraCubeList.append(LocalSyntCopy)
            LocalChi2CubeCopy = numpy.zeros((len(LocalSpectralAxis), self.NumYPixels, self.NumXPixels))
            self.Chi2SpectraCubeList.append(LocalChi2CubeCopy)


            ## determine the indices of range limits within total frequency array
            if (FullSpectralAxis is None):
                FullSpectralAxis = copy.deepcopy(LocalSpectralAxis)
            else:
                FullSpectralAxis = numpy.concatenate((FullSpectralAxis, LocalSpectralAxis), axis = None)
            FreqMaxIndex = len(FullSpectralAxis)
            self.LimitIndices.append([FreqMinIndex, FreqMaxIndex])
            FreqMinIndex = FreqMaxIndex

            # Debug:
            # print ("\nLocalRangeID = ", LocalRangeID)
            # print ("self.LimitIndices[-1][0] = ", self.LimitIndices[-1][0])
            # print ("self.LimitIndices[-1][1] = ", self.LimitIndices[-1][1])


        ##================================================================================================================================================
        ## start loop over different fit algorithms
        self.NumParamMapIterations = int(self.NumParamMapIterations)
        for LocalParamMapIter in range(self.NumParamMapIterations):                         ## perform user-defined number of smoothing iterations
            self.LocalParamMapIter = LocalParamMapIter


            ## print what you do
            if (self.NumParamMapIterations > 1):
                print ("\n\n\n\n\n----------------------------------------------\nPerform map fit for iteration: " + str(LocalParamMapIter + 1) + ":\n\n")


            ## start fast fitting
            if (self.FastFitFlag):
                self.StartFastFits(IndexRange)


            ## start normal fitting
            else:
                self.Fit_LSFlag = 1
                self.StartNormalFitting(IndexRange)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## smooth parameter maps
            self.SmoothParameterMaps()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize cluster
    def InitializeCluster(self, ClusterServerList, ClusterMode = "local"):
        """

    input parameters:
    -----------------

        - ClusterServerList:            list of nodes with number of workers

        - ClusterMode:                  (optional) method used for cluster ("local", "ssh", "slurm")
                                        (default: "local")


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("ClusterServerList = ", ClusterServerList)
        # print ("ClusterMode = ", ClusterMode)


        ## for normal fitting extract the number of cores, which are available for the current computer from the cluster file
        LocalServer = socket.gethostname()
        SumLocalCores = 0
        LocalClusterFlag = True
        ListNodes = []
        for LocalServerSettings in ClusterServerList:
            Server = LocalServerSettings[0]
            ListNodes.append(Server)
            NumCores = LocalServerSettings[1]
            if (Server.lower() in ["localhost", LocalServer]):
                SumLocalCores += NumCores
            else:
                LocalClusterFlag = False

            # Debug:
            # print ("Server = ", Server)
            # print ("NumCores = ", NumCores)

        # Debug:
        # print ("LocalServer = ", LocalServer)
        # print ("LocalClusterFlag = ", LocalClusterFlag)
        # print ("SumLocalCores = ", SumLocalCores)
        # print ("ListNodes = ", ListNodes)


        ##********************************************************************************************************************************************
        ## local cluster
        if (LocalClusterFlag):


            ## using local computer with dask
            if (ClusterMode in ["process"]):
                self.manager = multiprocessing.Manager()


            ## using local computer with dask
            elif (ClusterMode in ["dask"]):


                ## import required dask packages
                from dask.distributed import Client, LocalCluster


                ## print what you do
                print ("\n\nUsing dask package on a local cluster with {:d} workers!\n\n".format(SumLocalCores))


                # ## connect cluster to client
                cluster = LocalCluster(n_workers = SumLocalCores)           # , processes = False
                self.client = Client(cluster)


        ##********************************************************************************************************************************************
        ## use dask and ssh
        ## taken from https://docs.dask.org/en/latest/how-to/deploy-dask/ssh.html
        elif (ClusterMode in ["dask"]):


            ## import required dask packages
            from dask.distributed import Client, SSHCluster


            ## define slurm parameters
            cluster = SSHCluster(ListNodes,
                                 connect_options = {"known_hosts": None},
                                 worker_options = {"nthreads": NumCores})


            ## connect cluster to client
            self.client = Client(cluster)



        ##********************************************************************************************************************************************
        ## use pathos and ssh
        elif (ClusterMode in ["pathos"]):
            pass


            ## import required packages
            # from pathos.core import connect


        ##********************************************************************************************************************************************
        ## define worker (for SLURM cluster)
        ## to be done ..
        elif (ClusterMode in ["slurm"]):
            pass


            # ## import required packages
            # from dask_jobqueue import SLURMCluster


            # ## define slurm parameters
            # cluster = SLURMCluster(project='glab',
            #                        cores=24,
            #                        processes=2,
            #                        memory="128GB",
            #                        shebang="#!/usr/bin/env bash",
            #                        walltime="00:30:00",
            #                        death_timeout="15s",
            #                        interface="ib0"
            #                        )


            # ## connect cluster to client
            # self.client = Client(cluster)


            # ## add workers to the cluster (maybe not necessary)
            # # self.cluster.scale(8)


            # ## apply configurations
            # self.cluster = SLURMCluster()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import parameter maps from file
    def GetParameterMapsFromFile(self, ParameterMapDir, CompRegions):
        """

    input parameters:
    -----------------

        - ParameterMapDir:              path of directory containing FITS images describing parameter maps

        - CompRegions:                  combined regions


    output parameters:
    ------------------

        - ListOfMaps:                   list of maps
        """

        # Debug:
        # print ("ParameterMapDir = ", ParameterMapDir)
        # print ("CompRegions = ", CompRegions)
        # print ("self.ParameterNameLookUpDict = ", self.ParameterNameLookUpDict)


        ## initialize return parameter
        ListOfMaps = {}


        ## continue if path is defined
        if (ParameterMapDir is not None):
            if (ParameterMapDir != ""):
                ParameterMapDir = ParameterMapDir.strip()
                if (not ParameterMapDir.endswith("/")):
                    ParameterMapDir += "/"


                ## print what you do
                print ("\n\nImport parameter map(s) from file .. ", end = "", flush = True)


                ## get all filenames in the given directory
                listing = os.listdir(ParameterMapDir)
                for LocalFileName in listing:
                    LowerLocalFileName = LocalFileName.lower()
                    if (LowerLocalFileName.endswith(".fits")):
                        LowerPureFITSFileName = LowerLocalFileName.replace(".fits", "")
                        SplittedName = LowerPureFITSFileName.split("___")
                        NameParam = ""
                        MoleculeName = ""
                        SecondMolecue = None
                        NumberComp = None
                        if (len(SplittedName) > 0):                                         ## make sure, that we found a parameter map file
                            for element in SplittedName:


                                ## get name of parameter
                                if (element.find("parameter") > (-1)):
                                    LocalNameParam = element.replace("parameter", "")
                                    LocalNameParam = LocalNameParam.replace("__", "")
                                    NameParam = LocalNameParam.strip("_")
                                    NameParam = NameParam.strip()


                                ## get name of molecule
                                elif (element.find("molecule") > (-1)):
                                    MoleculeName = element.replace("molecule", "")
                                    MoleculeName = MoleculeName.replace("__", "")
                                    MoleculeName = MoleculeName.strip("_")
                                    MoleculeName = MoleculeName.strip()


                                ## get component id
                                elif (element.find("component") > (-1)):
                                    NumberCompString = element.replace("component", "")
                                    NumberCompString = NumberCompString.replace("__", "")
                                    NumberCompString = NumberCompString.strip("_")
                                    try:
                                        NumberComp = int(NumberCompString)
                                    except:
                                        NumberComp = None

                            # Debug:
                            # print ("\n\nNameParam = ", NameParam)
                            # print ("MoleculeName = ", MoleculeName)
                            # print ("SecondMolecue = ", SecondMolecue)
                            # print ("NumberComp = ", NumberComp)
                            # print ("LocalFileName = ", LocalFileName)


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## import FITS file and get parameter values at correct coordinates
                            if (NameParam != "" and MoleculeName != "" and (NumberComp is not None or SecondMolecue is not None \
                                or NameParam == "isoratio")):
                                LocalImage = Subimage_from_regions(ParameterMapDir + LocalFileName, CompRegions)


                                ## reproject FITS image to make sure, that parameter values are taken at the correct postions
                                LocalImage = LocalImage.reproject(self.ParameterMapHeader, order = 'bilinear')

                                # Debug:
                                # print ("numpy.nanmin(LocalImage[:,:]), numpy.nanmax(LocalImage[:,:]) = ", \
                                #         numpy.nanmin(LocalImage[:,:]), numpy.nanmax(LocalImage[:,:]))


                                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                ## store parameter maps for iso ratio file parameters
                                LocalKey = None
                                if (NameParam in ["isoratio"]):
                                    LowerMoleculeName = MoleculeName.lower()


                                    ## special handling for global defined iso ratios
                                    ## HERE:  self.ListOfInitialMaps[globalisoratio + "|" + IsoIndex] or self.ListOfInitialMaps[isoratio + "|" + IsoIndex]
                                    ## self.GlobalIsoParam[i] = [LocalIsotopologue, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag]
                                    ## self.IsoParameters[i] = [LocalIsotopologue, IsoMaster, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio,
                                    ##                          LocalFitFlag, ListIndex]
                                    if (LowerMoleculeName.startswith("global")):
                                        IsoIndex = None
                                        if (self.FullFitFlag):
                                            IsoMoleculeXCLASS = self.XCLASSDict['IsoMolecule']
                                            for LocalGlobalIsoParamLineID, LocalGlobalIsoParamLine in enumerate(self.GlobalIsoParam):
                                                LocalGlobalIsoParamLineMoleculeName = IsoMoleculeXCLASS.replace("__", "")
                                                LocalGlobalIsoParamLineMoleculeName = LocalGlobalIsoParamLineMoleculeName.lower()
                                                if (LocalGlobalIsoParamLineMoleculeName == LowerMoleculeName):
                                                    IsoIndex = LocalGlobalIsoParamLineID
                                                    break
                                        else:
                                            for LocalGlobalIsoParamLineID, LocalGlobalIsoParamLine in enumerate(self.GlobalIsoParam):
                                                LocalGlobalIsoParamLineMoleculeName = LocalGlobalIsoParamLine[0].replace("__", "")
                                                LocalGlobalIsoParamLineMoleculeName = LocalGlobalIsoParamLineMoleculeName.lower()
                                                if (LocalGlobalIsoParamLineMoleculeName == LowerMoleculeName):
                                                    IsoIndex = LocalGlobalIsoParamLineID
                                                    break
                                        if (IsoIndex is not None):
                                            LocalKey = "{:s}|{:d}".format("globalisoratio", IsoIndex)
                                            ListOfMaps[LocalKey] = LocalImage


                                    ## continue here for normally defined iso ratios
                                    else:
                                        IsoIndex = None
                                        if (self.FullFitFlag):
                                            IsoMoleculeXCLASS = self.XCLASSDict['IsoMolecule']
                                        else:
                                            IsoMoleculeXCLASS = self.IsoParameters
                                        for LocalIsoParametersLineID, LocalIsoParametersLine in enumerate(IsoMoleculeXCLASS):
                                            LocalGlobalIsoParamLineMoleculeName = task_LineIdentification.MoleculeFileName(LocalIsoParametersLine[0])
                                            LocalGlobalIsoParamLineMoleculeName = LocalGlobalIsoParamLineMoleculeName.strip("_")
                                            LocalGlobalIsoParamLineMoleculeName = LocalGlobalIsoParamLineMoleculeName.lower()
                                            if (LocalGlobalIsoParamLineMoleculeName == LowerMoleculeName):
                                                IsoIndex = LocalIsoParametersLineID
                                                break
                                        if (IsoIndex is not None):
                                            LocalKey = "{:s}|{:d}".format(NameParam, IsoIndex)
                                            ListOfMaps[LocalKey] = LocalImage


                                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                ## store parameter maps for molfit parameters
                                else:


                                    ## store parameter map with different key depending on fitting method
                                    if (self.FastFitFlag):


                                        ## determine position of current map parameter within molfit file
                                        ## self.MolfitCoord = [LocalMoleculeNameFileName, LocalLineID, [List of parameter names per component]]
                                        TotalCompCounter = None
                                        for LocalMolfitCoordID, LocalMolfitCoord in enumerate(self.MolfitCoord):
                                            LocalMoleculeNameFileName = LocalMolfitCoord[0].lower()
                                            if (LocalMoleculeNameFileName == MoleculeName):
                                                LocalLineID = LocalMolfitCoord[1]
                                                if (LocalLineID == (NumberComp - 1)):
                                                    TotalCompCounter = LocalMolfitCoordID
                                                    ListOfParamNames = LocalMolfitCoord[2]
                                                    if (self.FullFitFlag):
                                                        try:
                                                            ParamCounter = self.ParameterNameLookUpDict[NameParam.lower()]
                                                            break
                                                        except:
                                                            ParamCounter = None
                                                    else:
                                                        try:
                                                            ParamCounter = ListOfParamNames.index(NameParam.lower())
                                                            break
                                                        except:
                                                            ParamCounter = None

                                        # Debug:
                                        # print ("NameParam, TotalCompCounter, ParamCounter = ", NameParam, TotalCompCounter, ParamCounter)


                                        ## create key for dictionary
                                        if (TotalCompCounter is not None and ParamCounter is not None):
                                            LocalKey = "{:s}|{:d}|{:d}".format("molfit", TotalCompCounter, ParamCounter)
                                            ListOfMaps[LocalKey] = LocalImage


                                    ## for normal fitting, store parameter maps with a different key
                                    else:
                                        LocalKey = "{:s}|{:s}|{:d}|{:s}".format("molfit", MoleculeName, NumberComp - 1, NameParam)
                                        ListOfMaps[LocalKey] = LocalImage

                                # Debug:
                                # if (LocalKey is not None):
                                #     print ("\nLocalImage.shape = ", LocalImage.shape)
                                #     print ("LocalKey = ", LocalKey)
                                #     print ("LocalImage = ", LocalImage)


                ## we're done
                print ("done!")

            # Debug:
            # print ("self.FastFitFlag = ", self.FastFitFlag)
            # print ("ListOfMaps = ", ListOfMaps)
            # sys.exit(0)


        ## we're done
        return ListOfMaps
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze algorithm xml file
    def ImportAlgSettings(self, AlgorithmXMLFileName, NumberIteration, NumberProcessors):
        """

    input parameters:
    -----------------

        - AlgorithmXMLFileName:         path and name of alg. xml file

        - NumberIteration:              max. number of iterations

        - NumberProcessors:             number of cores


    output parameters:
    ------------------

        - FitDict:                      dictionary describing setting for fitting
        """

        # Debug:
        # print ("AlgorithmXMLFileName = ", AlgorithmXMLFileName)
        # print ("NumberIteration = ", NumberIteration)
        # print ("NumberProcessors = ", NumberProcessors)


        ## imitialize return parameter
        FitDict = []


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define algorithms


        ## define forbidden algorithms
        ForbiddenAlgorithms = ["simulated-annealing", "additionalpackages"]


        ## define list of knwon algorithms
        KnwonAlgorithms = ["trf", "dogbox", "lm", \
                           "nelder-mead", "powell", "cobyla", \
                           "cg", "bfgs", "l-bfgs-b", "tnc", "slsqp", \
                           "basinhopping", "brute", "bruteforce", "brute-force", \
                           "differential_evolution", "dual_annealing", "direct", \
                           "error-estimation", "errorestim_ins", \
                           "mcmc", "emcee", "ultranest", "blank"]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## if no algorithm xml file is defined, set default values
        if (AlgorithmXMLFileName == ""):
            LocalAlgDict = {}
            LocalAlgDict["FitAlgorithm"] = "trf"
            LocalAlgDict["VariationValue"] = 1.e-3
            LocalAlgDict['NumberSampler'] = 20
            LocalAlgDict["SampleMethod"] = "local"
            LocalAlgDict["SigmaMultiplicity"] = 2.0
            LocalAlgDict["ErrorType"] = "hpd"
            LocalAlgDict["NumberBurnInIter"] = 0
            LocalAlgDict["BackendFileName"] = ""
            LocalAlgDict["Dictionary"] = ""
            LocalAlgDict['BruteSteps'] = 2
            LocalAlgDict["PlotFileFormat"] = "png"
            LocalAlgDict["Chi2Limit"] = 1.e-8
            LocalAlgDict["Chi2RenormalizedFlag"] = True
            LocalAlgDict["Chi2LogFlag"] = True
            LocalAlgDict["Chi2StoreFlag"] = True
            LocalAlgDict["NumberIteration"] = NumberIteration
            LocalAlgDict["NumberProcessors"] = min(multiprocessing.cpu_count() - 1, NumberProcessors)
            LocalAlgDict["MPIHostFileName"] = ""
            LocalAlgDict["PlotFlag"] = True
            FitDict = [LocalAlgDict]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze alg. xml file
        else:
            from xclass.addons.MAGIX.Modules.python import XMLPackage


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## open xml-file file and save content of file to variable XmlFile
            AlgFile = XMLPackage.XMLRoutines(AlgorithmXMLFileName)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## read flag for using heuristic
            XMLString = "./NumberOfFitAlgorithms"
            NumberOfFitAlgorithms = AlgFile.GetSingleTagValue(XMLString, FormatString = "int", DefaultValue = 1)
            if (not CheckNumber(NumberOfFitAlgorithms)):                                    ## is the number of fit algorithms a number ?
                print ("\n\n\n\n")
                print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                print ("\t\t\t The selected number of fit algorithms is not a number !")
                print ("\t\t\t Please correct the .xml-file and restart XCLASS.")
                print ("\n\n\n\n")
                return FitDict
            elif (NumberOfFitAlgorithms < 1):                                               ## is the number of fit algorithms in a meaningful range?
                print ("\n\n\n\n")
                print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings!")
                print ("\t\t\t The number of fit algoirthms is negative!")
                print ("\t\t\t Please correct the .xml-file and restart XCLASS.")
                print ("\n\n\n\n")
                return FitDict


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## loop over all algorithms
            for algorithm in range(NumberOfFitAlgorithms):                                  ## loop over all algorithms
                LocalAlgDict = {}


                ## define stat string
                AlgXMLStartString = "./algorithm[" + str(algorithm + 1) + "]/"


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## read type of algorithm
                XMLString = AlgXMLStartString + "FitAlgorithm"
                FitAlgorithm = AlgFile.GetSingleTagValue(XMLString, FormatString = "lower")
                if (FitAlgorithm in ForbiddenAlgorithms):                                   ## is the chosen algorithm known?
                    print ("\n\n\n\n")
                    print ("\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                    print ("\n\t\t The selected algorithm {:s} can not be used in combination with the scipy / MAGIX-lite package!" \
                           .format(chr(34) + FitAlgorithm + chr(34)))
                    print ("\n\t\t Please use one of the following algorithms:")
                    for i in range(0, len(KnwonAlgorithms), 5):
                        print ("\t\t  {:s} ".format(", ".join("{:s}".format(chr(34) + LocaAlg + chr(34)) for LocaAlg in KnwonAlgorithms[i:i+5])))
                    print ("\n\t\t Please correct the algorithm xml-file and restart XCLASS.")
                    print ("\n\n\n\n")
                    sys.exit(0)
                if (not FitAlgorithm in KnwonAlgorithms):                                   ## is the chosen algorithm known?
                    print ("\n\n\n\n")
                    print ("\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                    print ("\n\t\t The selected algorithm {:s} is not known or implemented !".format(chr(34) + FitAlgorithm + chr(34)))
                    print ("\n\t\t The following algorithms are known:\n")
                    for i in range(0, len(KnwonAlgorithms), 5):
                        print ("\t\t  {:s} ".format(", ".join("{:s}".format(chr(34) + LocaAlg + chr(34)) for LocaAlg in KnwonAlgorithms[i:i+5])))
                    print ("\n\t\t Please correct the algorithm xml-file and restart XCLASS.")
                    print ("\n\n\n\n")
                    sys.exit(0)
                LocalAlgDict['FitAlgorithm'] = FitAlgorithm

                # Debug:
                # print ("FitAlgorithm = ", FitAlgorithm)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## error estimation: read method of error estimatoin of algorithm
                if (FitAlgorithm in ["errorestim_ins", "error-estimation"]):
                    XMLString = AlgXMLStartString + "ErrorMethod"
                    ErrorMethod = AlgFile.GetSingleTagValue(XMLString, FormatString = "lower")
                    if (ErrorMethod in ["ins", "mcmc", "ultranest"]):
                        LocalAlgDict['FitAlgorithm'] = ErrorMethod

                        # Debug:
                        # print ("ErrorMethod = ", ErrorMethod)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get value of variation (in percent) for the calculation of the gradient
                XMLString = AlgXMLStartString + "VariationValue"
                VariationValue = AlgFile.GetSingleTagValue(XMLString, FormatString = "float", DefaultValue = 1.e-3)
                if (not str(VariationValue)):                                               ## is the number of ranges a number ?
                    print ("\n\n\n\n")
                    print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings!")
                    print ("\t\t\t The selected variation value is not a number !")
                    print ("\t\t\t Please correct the .xml-file and restart XCLASS.")
                    print ("\n\n\n\n")
                    return FitDict
                LocalAlgDict['VariationValue'] = VariationValue

                # Debug:
                # print ("VariationValue = ", VariationValue)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get parameters "vol_bound" for INS algorithm
                XMLString = AlgXMLStartString + "vol_bound"
                VolBoundValue = AlgFile.GetSingleTagValue(XMLString, DefaultValue = "")
                LocalAlgDict['vol_bound'] = VolBoundValue

                # Debug:
                # print ("VolBoundValue = ", VolBoundValue)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get parameters "delta_incl" for INS algorithm
                XMLString = AlgXMLStartString + "delta_incl"
                DeltaInclValue = AlgFile.GetSingleTagValue(XMLString, DefaultValue = "")
                LocalAlgDict['delta_incl'] = DeltaInclValue

                # Debug:
                # print ("DeltaInclValue = ", DeltaInclValue)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## emcee and UltraNest only: get number of sampler
                ListTagNames = ["NumberMCMCSampler", "NumberSampler", "NumberUltraNestSampler"]
                for LocalTagName in ListTagNames:
                    XMLString = AlgXMLStartString + LocalTagName
                    NumberSampler = AlgFile.GetSingleTagValue(XMLString, FormatString = "int", DefaultValue = None)
                    if (NumberSampler is not None):
                        break
                if (NumberSampler is None):
                    NumberSampler = 20
                if (not CheckNumber(NumberSampler)):                                        ## is the number of ranges a number ?
                    print ("\n\n\n\n")
                    print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                    print ("\t\t\t The selected number of sampler is not a number !")
                    print ("\t\t\t Please correct the .xml-file and restart XCLASS.")
                    print ("\n\n\n\n")
                    return FitDict
                elif (NumberSampler < 0):                                                   ## is the number of fit algorithms in a meaningful range?
                    print ("\n\n\n\n")
                    print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                    print ("\t\t\t The selected number of MCMC sampler is lower than 0.")
                    print ("\t\t\t Please correct this value to a positive value and restart XCLASS.")
                    print ("\n\n\n\n")
                    return FitDict
                LocalAlgDict['NumberSampler'] = NumberSampler

                # Debug:
                # print ("NumberSampler = ", NumberSampler)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## emcee only: get sampling method
                XMLString = AlgXMLStartString + "SampleMethod"
                SampleMethod = AlgFile.GetSingleTagValue(XMLString, FormatString = "lower", DefaultValue = "local")
                if (SampleMethod in ["global"]):
                    SampleMethod = "global"
                else:
                    SampleMethod = "local"
                LocalAlgDict["SampleMethod"] = SampleMethod

                # Debug:
                # print ("SampleMethod = ", SampleMethod)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## emcee and UltraNest only: get multiplicity of standard deviation (1-sigma, 2-sigma, 3-sigma, etc.)
                XMLString = AlgXMLStartString + "MultiplicitySigma"
                SigmaMultiplicity = AlgFile.GetSingleTagValue(XMLString, FormatString = "float", DefaultValue = 2.0)
                if (not str(SigmaMultiplicity)):                                            ## describes the parameter a number ?
                    print ("\n\n\n\n")
                    print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings!")
                    print ("\t\t\t The selected sigma multiplicity is not a number !")
                    print ("\t\t\t Please correct the .xml-file and restart XCLASS.")
                    print ("\n\n\n\n")
                    return FitDict
                LocalAlgDict["SigmaMultiplicity"] = SigmaMultiplicity

                # Debug:
                # print ("SigmaMultiplicity = ", SigmaMultiplicity)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## emcee and UltraNest only: get type of error range: ["Gaussian", "Percentile", "HPD", "UltraNest"]
                XMLString = AlgXMLStartString + "ErrorType"
                ErrorType = AlgFile.GetSingleTagValue(XMLString, FormatString = "str", DefaultValue = "hpd")
                ErrorType = ErrorType.lower()
                if (ErrorType in ["gaussian"]):
                    ErrorType = "gaussian"
                elif (ErrorType in ["percentile"]):
                    ErrorType = "percentile"
                elif (ErrorType in ["hpd"]):
                    ErrorType = "hpd"
                elif (ErrorType in ["ultranest"]):
                    ErrorType = "ultranest"
                else:
                    ErrorType = ""
                LocalAlgDict["ErrorType"] = ErrorType

                # Debug:
                # print ("ErrorType = ", ErrorType)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## emcee and UltraNest only: get number of iterations for burn-in phase
                XMLString = AlgXMLStartString + "NumberBurnInIter"
                NumberBurnInIter = AlgFile.GetSingleTagValue(XMLString, FormatString = "int", DefaultValue = 0)
                LocalAlgDict["NumberBurnInIter"] = NumberBurnInIter

                # Debug:
                # print ("NumberBurnInIter = ", NumberBurnInIter)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## emcee and UltraNest only: get path and name of pickle file
                XMLString = AlgXMLStartString + "BackendFileName"
                BackendFileName = AlgFile.GetSingleTagValue(XMLString, DefaultValue = "")
                LocalAlgDict["BackendFileName"] = BackendFileName

                # Debug:
                # print ("BackendFileName = ", BackendFileName)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get dictionary describing additional parameters (for e.g. Ultranest)
                ListTagNames = ["dictionary", "Dictionary"]
                DictionaryString = None
                for LocalTagName in ListTagNames:
                    XMLString = AlgXMLStartString + LocalTagName
                    DictionaryString = AlgFile.GetSingleTagValue(XMLString, FormatString = "str", DefaultValue = None)
                    if (DictionaryString is not None):
                        break
                if (DictionaryString is None):
                    DictionaryString = ""
                LocalAlgDict["Dictionary"] = DictionaryString

                # Debug:
                # print ("DictionaryString = ", DictionaryString)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## brute force only: get number of grid points along each parameter axis
                XMLString = AlgXMLStartString + "BruteSteps"
                brute_steps = AlgFile.GetSingleTagValue(XMLString, FormatString = "lower")
                if (brute_steps is not None):
                    brute_steps = brute_steps.strip()                                       ## remove leading / tailing blanks
                    brute_steps = brute_steps.strip('[]')                                   ## remove squared brackets
                    BruteStepsElements = brute_steps.split(",")
                    if (len(BruteStepsElements) > 0):
                        brute_steps = [int(x) for x in BruteStepsElements]
                        if (self.NumFitParameters != len(brute_steps)):
                            print ("\n\n\n\n")
                            print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                            print ("\t\t\t The selected number of grid points (for brute force algorithm) has to be an integer number !")
                            print ("\t\t\t or a list of integer numbers which length is identical to the number of free parameters !")
                            print ("\t\t\t Please correct the xml-file and restart XCLASS.")
                            print ("\n\n\n\n")
                            return FitDict
                    else:
                        if (CheckNumber(brute_steps)):                                      ## is the number of ranges a number ?
                            if (NumberIteration < 1):                                       ## is number of iterations less than 1
                                print ("\n\n\n\n")
                                print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                                print ("\t\t\t The selected number of grid points (for brute force algorithm) is less than 1 !")
                                print ("\t\t\t Please correct the xml-file and restart XCLASS.")
                                print ("\n\n\n\n")
                                return FitDict
                            else:
                                brute_steps = [int(brute_steps) for x in range(self.NumFitParameters)]
                        else:
                            print ("\n\n\n\n")
                            print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                            print ("\t\t\t The selected number of grid points (for brute force algorithm) has to be an integer number !")
                            print ("\t\t\t or a list of integer numbers which length is identical to the number of free parameters !")
                            print ("\t\t\t Please correct the xml-file and restart XCLASS.")
                            print ("\n\n\n\n")
                            return FitDict
                LocalAlgDict['BruteSteps'] = brute_steps

                # Debug:
                # print ("brute_steps = ", brute_steps)
                # print ("brute_steps[0] = ", brute_steps[0])
                # print ("brute_steps[0] * 3 = ", brute_steps[0] * 3)
                # sys.exit(0)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## format of plot files
                XMLString = AlgXMLStartString + "PlotFileFormat"
                PlotFileFormat = AlgFile.GetSingleTagValue(XMLString, DefaultValue = "png")
                LocalAlgDict["PlotFileFormat"] = PlotFileFormat

                # Debug:
                # print ("PlotFileFormat = ", PlotFileFormat)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get limit for chi**2 from control file
                ListTagNames = ["limit_of_chi2", "LimitOfChi2", "LimitChi2"]
                for LocalTagName in ListTagNames:
                    XMLString = AlgXMLStartString + LocalTagName
                    Chi2Limit = AlgFile.GetSingleTagValue(XMLString, FormatString = "float", DefaultValue = None)
                    if (Chi2Limit is not None):
                        break
                if (Chi2Limit is None):
                    Chi2Limit = 1.e-8
                if (not CheckNumber(Chi2Limit)):                                            ## is the number of ranges a number ?
                    print ("\n\n\n\n")
                    print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                    print ("\t\t\t The selected limit for chi**2 is not a number !")
                    print ("\t\t\t Please correct the .xml-file and restart XCLASS.")
                    print ("\n\n\n\n")
                    return FitDict
                elif (Chi2Limit < 0):                                                       ## is the number of fit algorithms in a meaningful range?
                    print ("\n\n\n\n")
                    print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                    print ("\t\t\t The selected limit for chi**2 is lower than 0.")
                    print ("\t\t\t Please correct this value to a positive value and restart XCLASS.")
                    print ("\n\n\n\n")
                    return FitDict
                LocalAlgDict["Chi2Limit"] = Chi2Limit

                # Debug:
                # print ("Chi2Limit = ", Chi2Limit)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get flag for usage of renormalized chi**2
                XMLString = AlgXMLStartString + "RenormalizedChi2"
                Chi2RenormalizedFlag = AlgFile.GetSingleTagValue(XMLString, FormatString = "bool", DefaultValue = True)
                Chi2RenormalizedFlag = CheckBool(Chi2RenormalizedFlag)
                LocalAlgDict["Chi2RenormalizedFlag"] = Chi2RenormalizedFlag

                # Debug:
                # print ("Chi2RenormalizedFlag = ", Chi2RenormalizedFlag)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get flag for chi2-log file
                XMLString = AlgXMLStartString + "LogChi2"
                Chi2LogFlag = AlgFile.GetSingleTagValue(XMLString, FormatString = "bool", DefaultValue = True)
                Chi2LogFlag = CheckBool(Chi2LogFlag)
                LocalAlgDict["Chi2LogFlag"] = Chi2LogFlag

                # Debug:
                # print ("Chi2LogFlag = ", Chi2LogFlag)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get flag for saving chi**2 function (residual)
                XMLString = AlgXMLStartString + "SaveChi2"
                Chi2StoreFlag = AlgFile.GetSingleTagValue(XMLString, FormatString = "bool", DefaultValue = True)
                Chi2StoreFlag = CheckBool(Chi2StoreFlag)
                LocalAlgDict["Chi2StoreFlag"] = Chi2StoreFlag

                # Debug:
                # print ("Chi2StoreFlag = ", Chi2StoreFlag)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get maximum number of iterations from control file
                ListTagNames = ["number_iterations", "NumberIterations"]
                for LocalTagName in ListTagNames:
                    XMLString = AlgXMLStartString + LocalTagName
                    NumberIteration = AlgFile.GetSingleTagValue(XMLString, FormatString = "int", DefaultValue = None)
                    if (NumberIteration is not None):
                        break
                if (NumberIteration is None):
                    NumberIteration = 10
                if (not CheckNumber(NumberIteration)):                                      ## is the number of ranges a number ?
                    print ("\n\n\n\n")
                    print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings!")
                    print ("\t\t\t The selected maximum number of iterations is not a number !")
                    print ("\t\t\t Please correct the .xml-file and restart XCLASS.")
                    print ("\n\n\n\n")
                    return FitDict
                elif (NumberIteration < 1):                                                 ## is number of iterations less than 1
                    print ("\n\n\n\n")
                    print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                    print ("\t\t\t The selected maximum number of iterations is less than 1 !")
                    print ("\t\t\t Please correct the xml-file and restart XCLASS.")
                    print ("\n\n\n\n")
                    return FitDict
                LocalAlgDict['NumberIteration'] = NumberIteration

                # Debug:
                # print ("numiter = ", numiter)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get number of cores
                XMLString = AlgXMLStartString + "NumberProcessors"
                NumberProcessors = AlgFile.GetSingleTagValue(XMLString, FormatString = "int", DefaultValue = 2)
                if (not CheckNumber(NumberProcessors)):                                     ## is the number of cores a number ?
                    print ("\n\n\n\n")
                    print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                    print ("\t\t\t The selected number of cores is not a number !")
                    print ("\t\t\t Please correct the xml-file and restart XCLASS.")
                    print ("\n\n\n\n")
                    return FitDict
                elif (NumberProcessors < 1):                                                ## is number of cores less than 1
                    print ("\n\n\n\n")
                    print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                    print ("\t\t\t The selected number of cores is less than 1 !")
                    print ("\t\t\t Please correct the xml-file and restart XCLASS.")
                    print ("\n\n\n\n")
                    return FitDict
                NumberAvailProc = multiprocessing.cpu_count()
                if (NumberProcessors > NumberAvailProc):
                    print ("\n\t WARNING:")
                    print ("\t\tThe defined number of cores ({:d}) ".format(NumberProcessors))
                    print ("\t\tis larger than the number of available cores ({:d}).".format(NumberAvailProc))
                    print ("\t\tReduce number of cores to the number of available cores minus one.\n")
                    NumberProcessors = NumberAvailProc - 1
                LocalAlgDict['NumberProcessors'] = NumberProcessors

                # Debug:
                # print ("NumberProcessors = ", NumberProcessors)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get path and name of MPI host file
                XMLString = AlgXMLStartString + "MPIHostFileName"
                MPIHostFileName = AlgFile.GetSingleTagValue(XMLString, DefaultValue = "")
                if (MPIHostFileName != "" and MPIHostFileName != "MPI_HOSTS"):


                    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                    ## check MPI host file
                    if (MPIHostFileName != ""):


                        ## check existence of host file only for MPI version
                        if (not (os.path.exists(MPIHostFileName))):
                            print ("\n\n\n\n")
                            print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                            print ("\t\t\t The host file defined in the fit control xml file does not exists!")
                            print ("\t\t\t Please correct the xml-file and restart XCLASS.")
                            print ("\n\n\n\n")
                            return FitDict


                        ## read in host file
                        MPIHostFile = open(MPIHostFileName)
                        MPIHostFileContent = MPIHostFile.readlines()
                        MPIHostFile.close()
                        if (MPIHostFileContent == []):
                            print ("\n\n\n\n")
                            print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                            print ("\t\t\t The host file defined in the fit control xml file is empty!")
                            print ("\t\t\t Please correct the xml-file and restart XCLASS.")
                            print ("\n\n\n\n")
                            return FitDict


                        ## get name of current machine (system hostname)
                        try:
                            SystemHostname = socket.gethostname()
                        except ImportError:
                            SystemHostname = "localhost"


                        ## check, if first entry in host file includes current host
                        ChangeFlag = True
                        NewHostFile = []
                        CounterLines = 0
                        CurrentHostLine = ""
                        for line in MPIHostFileContent:
                            i = line.find("#")
                            if (i > (-1)):
                                StrippedLine = line[:i].strip()
                            else:
                                StrippedLine = line.strip()
                            if (StrippedLine != ""):
                                CounterLines += 1
                                if (CounterLines == 1):
                                    if (StrippedLine.find(SystemHostname) > (-1) or StrippedLine.find("localhost") > (-1)):
                                        ChangeFlag = False
                                        break
                                if (StrippedLine.find(SystemHostname) > (-1)):
                                    CurrentHostLine = StrippedLine
                                else:
                                    NewHostFile.append(StrippedLine)
                        if (ChangeFlag):                                                    ## first line in host file does not describe current host


                            ## check, if current host is included in host file
                            if (CurrentHostLine == ""):
                                print ("\n\n\n\n")
                                print ("\t\t ERROR in subroutine myXCLASSMapFit.FastMapFit.ImportAlgSettings !")
                                print ("\t\t\t The host file does not contain settings for the current host (" + SystemHostname + ")!")
                                print ("\t\t\t Please correct host file and restart XCLASS.")
                                print ("\n\n\n\n")

                                # Debug:
                                # print ("MPIHostFileName = ", MPIHostFileName)
                                # print ("MPIHostFileContent = ", MPIHostFileContent)
                                # print ("\n\n\n")
                            else:
                                MPIHostFile = open(MPIHostFileName, 'w')
                                MPIHostFile.write(CurrentHostLine + "\n")
                                for line in NewHostFile:
                                    MPIHostFile.write(line + "\n")
                                MPIHostFile.close()
                LocalAlgDict['MPIHostFileName'] = MPIHostFileName

                # Debug:
                # print ("MPIHostFileName = ", MPIHostFileName)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get flag for creating final plot
                XMLString = AlgXMLStartString + "PlotFlag"
                PlotFlag = AlgFile.GetSingleTagValue(XMLString, FormatString = "bool", DefaultValue = True)
                PlotFlag = CheckBool(PlotFlag)
                LocalAlgDict["PlotFlag"] = PlotFlag

                # Debug:
                # print ("PlotFlag = ", PlotFlag)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## store dictionary
                FitDict.append(LocalAlgDict)

        # Debug:
        # print ("FitDict = ", FitDict)
        # sys.exit(0)


        ## we're done
        return FitDict
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## prepare map fit
    def PrepareFit(self, ParameterDict, RemoveContinuumFlag):
        """

    input parameters:
    -----------------

        - ParameterDict:                dictionary with parameters

        - RemoveContinuumFlag:          flag indicating removing of continuum


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("ParameterDict = ", ParameterDict)
        # print ("RemoveContinuumFlag = ", RemoveContinuumFlag)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## get parameters from dictionary
        self.NameOfFunction = ParameterDict['NameOfFunction']
        self.myXCLASSMapFitJobDir = ParameterDict['myXCLASSMapFitJobDir']
        EstimateContinuumFlag = ParameterDict['EstimateContinuumFlag']
        EstimateConNumParts = ParameterDict['EstimateConNumParts']
        EstimateConNumEntireCubeFlag = ParameterDict['EstimateConNumEntireCubeFlag']
        CurrentDir = ParameterDict['CurrentDir']
        ParameterMapDir = ParameterDict['ParameterMapDir']
        self.NumParamMapIterations = ParameterDict['ParamMapIterations']
        self.ParamSmoothMethodParam = ParameterDict['ParamSmoothMethodParam']
        self.ParamSmoothMethod = ParameterDict['ParamSmoothMethod']
        RegionFileName = ParameterDict['RegionFileName']
        FITSImageMaskFileName = ParameterDict['FITSImageMaskFileName']
        GlobalThreshold = ParameterDict['Threshold']
        LocalObsXMLFileName = ParameterDict['LocalObsXMLFileName']
        ClusterServerList = ParameterDict['ClusterServerList']
        AlgorithmXMLFileName = ParameterDict['AlgorithmXMLFileName']
        NumberIteration = ParameterDict['NumberIteration']
        NumberProcessors = ParameterDict['NumberProcessors']
        self.MolfitsFileName = ParameterDict['LocalMolfitsFileName']
        self.IsoFlag = ParameterDict['IsoFlag']
        IsoRatioFileName = ParameterDict['IsoRatioFileName']
        self.IsoRatioFileName = ParameterDict['IsoRatioFileName']
        # DoParameterEstimationFlag = ParameterDict['DoParameterEstimationFlag']
        # InternalParameterList = ParameterDict['InternalParameterList']
        dbList = ParameterDict['dbList']


        ## determine parameters for Fortran routine
        if (RemoveContinuumFlag):
            RemoveContinuumFlag = 1
        else:
            RemoveContinuumFlag = 0


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## import ds9 region file
        CompRegions = None
        if (RegionFileName != ""):
            region_list = Regions.read(RegionFileName, format = 'ds9')


            ## prepare region if only one region was selected
            if (len(region_list) == 1):
                CompRegions = copy.deepcopy(region_list)


            ## prepare region if only many regions were selected
            else:
                for region in region_list:
                    if (CompRegions is None):
                        CompRegions = region
                    else:
                        CompRegions = CompRegions | region

                    # Debug:
                    # print ("\n\nregion = ", region)


                ## prepare comp. variable for spectral cube
                CompRegions = [CompRegions]

        # Debug:
        # print ("CompRegions = ", CompRegions)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## import FITS image used for masking
        FITSImageMask = None
        FITSImageMaskValues = None
        if (FITSImageMaskFileName != ""):
            FITSImageMaskFileName = FITSImageMaskFileName.strip()
            FITSImageMaskValues = Subimage_from_regions(FITSImageMaskFileName, CompRegions)
            FITSImageMask = numpy.isnan(FITSImageMaskValues)

        # Debug:
        # print ("FITSImageMask = ", FITSImageMask)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## initialize dask cluster
        if (self.ParallelMethod in ["dask", "pathos", "process"]):
            self.InitializeCluster(ClusterServerList, ClusterMode = self.ParallelMethod)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## analyze molfit file


        ## get sql parameters from molfit file
        MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(self.MolfitsFileName)
        self.SQLParamArray = [MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL]

        # Debug:
        # print ("self.SQLParamArray = ", self.SQLParamArray)


        ## prepare call of parameter estimation algorithm
        self.AllMoleculesInMolfitFile, self.AllParameters, self.AllMolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(self.MolfitsFileName)
        self.MolNameList = copy.deepcopy(self.AllMoleculesInMolfitFile)

        # Debug:
        # print ("self.AllMoleculesInMolfitFile = ", self.AllMoleculesInMolfitFile)
        # print ("self.AllParameters = ", self.AllParameters)
        # print ("self.AllMolfitFileForEachMolecule = ", self.AllMolfitFileForEachMolecule)


        ## for normal fit get SQL header for molfit file
        self.MolfitHeaderLinesList = task_myXCLASS.WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)


        ## determine total number of components
        NumComp = 0
        NumCompCont = 0
        for LocalMoleculeID in range(len(self.AllMoleculesInMolfitFile)):
            NumComp += len(self.AllParameters[LocalMoleculeID])

        # Debug:
        # print ("NumComp = ", NumComp)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create molfit file array for fitting


        ## loop over molecules
        self.MolfitFitParameters = []
        self.FitParameterNames = []
        self.AllMolfitParameters = []
        self.ListDistances = []
        self.TrotIndices = []
        self.NumEmissionComp = 0
        self.MolIndexPerComp = []
        CompCounter = (-1)
        self.MolfitCoord = []
        for LocalMoleculeID, LocalMolecule in enumerate(self.AllMoleculesInMolfitFile):


            # ## get RRL flag
            # LocalMolName = LocalMolecule.lower()
            # KindOfMolecule = 0.0
            # if (LocalMolName.startswith("rrl-")):
            #     KindOfMolecule = 1.0
            # elif (not LocalMolName.startswith("cont-")):
            #     try:
            #         i = self.MolNameList.index(LocalMolecule)
            #     except:
            #         i = 1000
            #     KindOfMolecule = i * (-1)


            LocalMolName = LocalMolecule.lower()
            KindOfMolecule = 0.0
            if (LocalMolName.startswith("rrl-")):
                KindOfMolecule = 1.0
            elif (LocalMolName.startswith("cont-dust")):
                KindOfMolecule = 3.0
            elif (LocalMolName.startswith("cont-phen")):
                KindOfMolecule = 6.0
            elif (not LocalMolName.startswith("cont-")):
                try:
                    i = self.MolNameList.index(LocalMolecule)
                except:
                    i = 1000
                KindOfMolecule = i * (-1)

                # Debug:
                # print ("KindOfMolecule = ", KindOfMolecule)


            ## loop over the parameters
            LocalMolfitParameters = self.AllParameters[LocalMoleculeID]
            for LocalLineID, LocalLine in enumerate(LocalMolfitParameters):
                CompCounter += 1
                self.MolIndexPerComp.append(LocalMolecule)

                # Debug:
                # print ("LocalLine = ", LocalLine)


                ## store name of molecule and current component (used for parameter maps)
                LocalMoleculeNameFileName = task_LineIdentification.MoleculeFileName(LocalMolecule)
                LocalMoleculeNameFileName = LocalMoleculeNameFileName.strip("_")
                LocalMolfitCoord = [LocalMoleculeNameFileName, LocalLineID]


                ## initialize paramter array for current line
                LocalLineParameter = numpy.zeros(9, dtype = numpy.float32)
                LocalLineParameter[0] = KindOfMolecule


                ## get molfit paramters
                flag = 0
                LowerLimit = None
                UpperLimit = None
                FitFlag = False
                Distance = None
                OldFormatValue = 0.0
                ParameterCounter = 0
                LocalTrotIndices = (-1)
                ListParamLine = []
                for col in LocalLine:
                    Name = col[0]
                    element = col[4]
                    AddFlag = False

                    # Debug:
                    # print ("col = ", col)
                    # print ("Name = ", Name)


                    ## get fit flag
                    if (Name.endswith("_flag")):
                        FitFlag = element.strip().lower()
                        FitFlag = CheckBool(FitFlag)
                        OldFormatValue = 0.0


                    ## get fit flag
                    elif (Name.endswith("_FitFlag")):
                        try:
                            val = float(element)
                        except:
                            val = 0.0
                        if (val != 0.0):
                            OldFormatValue = val
                            LowerLimit = 0.0
                            UpperLimit = 0.0
                            FitFlag = True
                        else:
                            LowerLimit = 0.0
                            UpperLimit = 0.0
                            FitFlag = False


                    ## get lower limit
                    elif (Name.endswith("_lowlimit")):
                        LowerLimit = float(element)


                    ## get upper limit
                    elif (Name.endswith("_uplimit")):
                        UpperLimit = float(element)


                    ## store values for source size
                    elif (Name in ["source_size", "source_radius", "source_center_x", "source_center_y"]):
                        value = float(element)
                        LogFlag = False
                        AddFlag = True
                        if (OldFormatValue != 0.0):
                            LowerLimit = max(0.0001, value - OldFormatValue)
                            UpperLimit = value + OldFormatValue


                    ## store values for T_rot
                    elif (Name in ["T_rot", "T_e", "T_dOff", "T_cont_dust", "T_Back"]):
                        value = float(element)
                        LogFlag = False
                        AddFlag = True
                        if (OldFormatValue != 0.0):
                            LowerLimit = max(1.072, value - OldFormatValue)
                            UpperLimit = value + OldFormatValue


                    ## store values for N_tot
                    elif (Name in ["N_tot", "EM_RRL", "nHcolumn_cont_dust", "nHcolumn"]):
                        value = float(element)
                        LogFlag = True
                        AddFlag = True
                        if (OldFormatValue != 0.0):
                            LowerLimit = max(10.0**(numpy.log10(value) - OldFormatValue), 10.0)
                            UpperLimit = 10.0**(numpy.log10(value) + OldFormatValue)


                    ## store values for V_width
                    elif (Name in ["V_width_Gauss", "V_width"]):
                        value = float(element)
                        LogFlag = False
                        AddFlag = True
                        if (OldFormatValue != 0.0):
                            LowerLimit = max(0.00001, value - OldFormatValue)
                            UpperLimit = value + OldFormatValue


                    ## store values for V_off
                    elif (Name == "V_off"):
                        value = float(element)
                        LogFlag = False
                        AddFlag = True
                        if (OldFormatValue != 0.0):
                            LowerLimit = value - OldFormatValue
                            UpperLimit = value + OldFormatValue


                    ## store values for continuum descriptions
                    # "beta",
                    elif (Name in ["kappa_cont_dust"]):
                        value = float(element)
                        LogFlag = True
                        AddFlag = True
                        if (OldFormatValue != 0.0):
                            LowerLimit = value - OldFormatValue
                            UpperLimit = value + OldFormatValue


                    ## store values for continuum descriptions
                    elif (Name in ["beta_cont_dust", "beta", "ContFuncID_phen", \
                                   "ContFuncID_param_1", "ContFuncID_param_2", "ContFuncID_param_3", \
                                   "ContFuncID_param_4", "ContFuncID_param_5"]):
                        value = float(element)
                        LogFlag = False
                        AddFlag = True
                        if (OldFormatValue != 0.0):
                            LowerLimit = value - OldFormatValue
                            UpperLimit = value + OldFormatValue


                    ## store values for CF-Flag
                    elif (Name in ["CFFlag"]):
                        LogFlag = False
                        CFFlag = element.strip()
                        CFFlag = CFFlag.lower()
                        if (CFFlag in ["c", "e"]):
                            Distance = NumComp + 1
                            self.NumEmissionComp += 1
                        else:
                            Distance = ((NumComp + 1) - (NumCompCont + CompCounter + 1))
                        if (not Distance in self.ListDistances):
                            self.ListDistances.append(Distance)


                    ## store values for distance
                    elif (Name in ["LayerDistance"]):
                        Distance = float(element)
                        if (not Distance in self.ListDistances):
                            self.ListDistances.append(Distance)


                    ## store paramters
                    if (AddFlag):
                        ParameterCounter += 1
                        LocalLineParameter[ParameterCounter] = value
                        ListParamLine.append(Name.lower())
                        if (Name in ["T_rot"]):
                            LocalTrotIndices = ParameterCounter


                        ## store fit parameter
                        if (FitFlag):
                            self.FitParameterNames.append([LocalMolecule, LocalLineID, Name, LogFlag])
                            if (LogFlag):
                                LogValue = numpy.log10(value)
                                LogLowerLimit = numpy.log10(LowerLimit)
                                LogUpperLimit = numpy.log10(UpperLimit)
                                self.MolfitFitParameters.append([CompCounter, ParameterCounter, LogFlag, min(LogLowerLimit, LogUpperLimit), \
                                                                 max(LogLowerLimit, LogUpperLimit), LogValue, Name])
                            else:
                                self.MolfitFitParameters.append([CompCounter, ParameterCounter, LogFlag, min(LowerLimit, UpperLimit), \
                                                                 max(LowerLimit, UpperLimit), value, Name])

                        # Debug:
                        # print ("\n\nFitFlag = ", FitFlag)
                        # print ("value = ", value)
                        # print ("LowerLimit = ", LowerLimit)
                        # print ("UpperLimit = ", UpperLimit)


                ## store fit parameters with limits
                LocalLineParameter[8] = Distance
                self.AllMolfitParameters.append(LocalLineParameter)
                self.TrotIndices.append(LocalTrotIndices)
                LocalMolfitCoord.append(ListParamLine)
                self.MolfitCoord.append(LocalMolfitCoord)

                # Debug:
                # print ("LocalLineParameter = ", LocalLineParameter)
                # print ("LocalTrotIndices = ", LocalTrotIndices)


        ## convert molfit parameter list and temperture index list to numpy arrays
        self.AllMolfitParameters = numpy.asarray(self.AllMolfitParameters)
        self.TrotIndices = numpy.asarray(self.TrotIndices)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## analyze iso ratio file and get list of all molecules
        self.NumIsoMaster = len(self.MolNameList)
        self.GlobalIsoParam = []
        self.IsoParameters = []
        self.IsoFitParameters = []
        self.IsoRatioTable = []
        self.IsotopologuesIn = []
        self.IsoMoleculeIn = []
        if (self.IsoFlag):


            ## read in iso ratio file
            NewLocalIsoRatioFileName = ""
            self.IsoRatioTable, self.IsotopologuesIn, self.IsoMoleculeIn = task_myXCLASS.ImportIsoRatioFile(self.IsoRatioFileName, \
                                                                                                            NewLocalIsoRatioFileName)


            ## add isotpologues to global list of molecules
            LocalMolNameListCopy = copy.deepcopy(self.MolNameList)
            for LocalMoleculeName in LocalMolNameListCopy:
                for LocalIsotopologuesID, LocalIsotopologues in enumerate(self.IsotopologuesIn):    ## loop over all self.IsotopologuesIn
                    if (self.IsoMoleculeIn[LocalIsotopologuesID] == LocalMoleculeName):     ## check if isotopologue corresponds to current molecule
                        self.MolNameList.append(LocalIsotopologues.strip())


            ## search for globally defined iso ratios
            GlobalList = []
            for IsoIndex in range(len(self.IsoRatioTable)):
                IsoRatioTableLine = self.IsoRatioTable[IsoIndex]
                LocalIsotopologue = IsoRatioTableLine[0].strip()
                LowerLocalIsotopologue = LocalIsotopologue.lower()
                if (LowerLocalIsotopologue.startswith("global")):
                    LocalIsotopologue = LocalIsotopologue.replace("Global", "")
                    LocalIsotopologue = LocalIsotopologue.replace("GLOBAL", "")
                    LocalIsotopologue = LocalIsotopologue.replace("global", "")
                    LocalIsotopologue = LocalIsotopologue.replace("___", "")
                    LocalIsotopologue = LocalIsotopologue.replace("__", "")
                    GlobalList.append([LocalIsotopologue, IsoIndex])
                    RatioValue = float(IsoRatioTableLine[2])
                    LocalFitFlag = False
                    if (len(IsoRatioTableLine) == 3):
                        LowerLimitIsoRatio = 1.0
                        UpperLimitIsoRatio = 1.0
                    else:
                        l1 = float(IsoRatioTableLine[3])
                        l2 = float(IsoRatioTableLine[4])
                        LowerLimitIsoRatio = min(l1, l2)
                        UpperLimitIsoRatio = max(l1, l2)
                        if (LowerLimitIsoRatio != UpperLimitIsoRatio):
                            LocalFitFlag = True
                            self.IsoFitParameters.append([(-1) * (len(self.GlobalIsoParam) + 1), LowerLimitIsoRatio, UpperLimitIsoRatio, RatioValue])
                    self.GlobalIsoParam.append([IsoRatioTableLine[0], RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag])

            # Debug:
            # print ("GlobalList = ", GlobalList)


            ## analyze iso ratio file and update ratios
            for LocalMoleculeName in self.MolNameList:


                ## define iso parameter list
                for IsoIndex in range(len(self.IsoRatioTable)):
                    IsoRatioTableLine = self.IsoRatioTable[IsoIndex]
                    LocalIsotopologue = IsoRatioTableLine[0].strip()
                    LowerLocalIsotopologue = LocalIsotopologue.lower()
                    IsoMaster = IsoRatioTableLine[1].strip()
                    if (IsoMaster == LocalMoleculeName and not LowerLocalIsotopologue.startswith("global")):
                        RatioValue = float(IsoRatioTableLine[2])
                        LocalFitFlag = False


                        ## store fit parameters
                        if (len(IsoRatioTableLine) == 3):
                            LowerLimitIsoRatio = 1.0
                            UpperLimitIsoRatio = 1.0
                        else:
                            l1 = float(IsoRatioTableLine[3])
                            l2 = float(IsoRatioTableLine[4])
                            LowerLimitIsoRatio = min(l1, l2)
                            UpperLimitIsoRatio = max(l1, l2)


                        ## for not globally defined ratios, search for globally defined isotopologues
                        ListIndex = []
                        if ((not LowerLocalIsotopologue.startswith("global")) and GlobalList != []):
                            for GlobalElement in GlobalList:
                                GlobalIso = GlobalElement[0]
                                GlobalIndex = GlobalElement[1]
                                if (LocalIsotopologue.find(GlobalIso) > (-1)):
                                    Multiplicator = LocalIsotopologue.count(GlobalIso)
                                    ListIndex.append([GlobalIndex, Multiplicator])
                        self.IsoParameters.append([LocalIsotopologue, IsoMaster, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, \
                                                   LocalFitFlag, ListIndex])


                        ## store fit parameters
                        if (len(IsoRatioTableLine) > 3 and LowerLimitIsoRatio != UpperLimitIsoRatio):
                            LocalFitFlag = True
                            self.IsoFitParameters.append([len(self.IsoParameters) - 1, LowerLimitIsoRatio, UpperLimitIsoRatio, RatioValue])
                            self.FitParameterNames.append([LocalIsotopologue, (-1), "IsoRatio", False])

        # Debug:
        # print ("self.MolNameList = ", self.MolNameList)
        # print ("self.GlobalIsoParam = ", self.GlobalIsoParam)
        # print ("self.IsoParameters = ", self.IsoParameters)
        # print ("self.IsoFitParameters = ", self.IsoFitParameters)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## define parameter and bounding vectors
        self.pBLow = []
        self.pBUp = []
        self.RangeList = []


        ## get ranges for molfit parameters
        ## self.MolfitFitParameters.append([CompCounter, ParameterCounter, LogFlag, LowerLimit, UpperLimit, value, Name])
        for line in self.MolfitFitParameters:
            LowerLimit = min(line[3], line[4])
            UpperLimit = max(line[3], line[4])
            self.pBLow.append(LowerLimit)
            self.pBUp.append(UpperLimit)
            self.RangeList.append([LowerLimit, UpperLimit])


        ## get ranges for iso ratio parameters
        ## self.IsoFitParameters[i] = [IsoIndex, LowerLimitIsoRatio, UpperLimitIsoRatio, RatioValue]
        ## self.GlobalIsoParam[i] = [IsoRatioTableLine[0], RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag]
        ## self.IsoParameters[i] = [LocalIsotopologue, IsoMaster, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag, ListIndex]
        for line in self.IsoFitParameters:
            IsoIndex = line[0]
            LowerLimit = min(line[1], line[2])
            UpperLimit = max(line[1], line[2])
            self.pBLow.append(LowerLimit)
            self.pBUp.append(UpperLimit)
            self.RangeList.append([LowerLimit, UpperLimit])


            ## store names of isotopologues as well
            if (IsoIndex >= 0):
                LocalIsotopologue = self.IsoParameters[IsoIndex][0].strip()
            else:
                IsoIndex = abs(IsoIndex) - 1
                LocalIsotopologue = self.GlobalIsoParam[abs(IsoIndex)][0].strip()
            # self.FitParameterNames.append([LocalIsotopologue, (-1), "IsoRatio", False])

        # Debug:
        # print ("self.pBLow = ", self.pBLow)
        # print ("self.pBUp = ", self.pBUp)
        # print ("self.RangeList = ", self.RangeList)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## some algorithms require a different storage of parameter limits
        self.RangeTupleList = [tuple(x) for x in self.RangeList]


        ## get list of distances defined in molfit file
        self.ListDistances.sort(reverse = True)                                             ## sort list of molfit distances
        self.NumDistances = len(self.ListDistances)                                         ## get number of distances defined in molfit file

        # Debug:
        # print ("self.ListDistances = ", self.ListDistances)
        # print ("self.NumDistances = ", self.NumDistances)


        ## define total number of fit parameters
        self.NumFitParameters = len(self.MolfitFitParameters) + len(self.IsoFitParameters)

        # Debug:
        # print ("len(self.MolfitFitParameters) = ", len(self.MolfitFitParameters))
        # print ("len(self.IsoFitParameters) = ", len(self.IsoFitParameters))
        # print ("self.NumFitParameters = ", self.NumFitParameters)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## compute interpolation object for partition function
        if (self.FastFitFlag and (not self.FullFitFlag)):
            self.PartFunc = {}
            for LocalMoleculeName in self.MolNameList:
                LowerLocalMolecule = LocalMoleculeName.strip()
                LowerLocalMolecule = LowerLocalMolecule.lower()
                if (not (LowerLocalMolecule.startswith("cont-") or LowerLocalMolecule.startswith("rrl-"))):
                    LocalPartFunc = GetPartitionFunctionFunc(LocalMoleculeName, dbList)
                    self.PartFunc[LocalMoleculeName] = LocalPartFunc
                else:
                    self.PartFunc[LocalMoleculeName] = None
            # for IsoLine in self.IsoParameters:
            #     LocalIsotopologue = IsoLine[0]
            #     if (not (LocalIsotopologue.startswith("cont-") or LocalIsotopologue.startswith("rrl-"))):
            #         LocalPartFunc = GetPartitionFunctionFunc(LocalIsotopologue, dbList)
            #         self.PartFunc[LocalIsotopologue] = LocalPartFunc
            #     else:
            #         self.PartFunc[LocalIsotopologue] = None

            # Debug:
            # print ("self.PartFunc = ", self.PartFunc)
            # sys.exit(0)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## get algorithm settings
        if (self.FastFitFlag):
            self.FitAlgSettings = self.ImportAlgSettings(AlgorithmXMLFileName, NumberIteration, NumberProcessors)
        else:


            ## for normal fitting extract the number of cores, which are available for the current computer from the cluster file
            self.Fit_NumProc = NumberProcessors
            LocalServer = socket.gethostname()
            for LocalServerSettings in ClusterServerList:
                Server = LocalServerSettings[0].lower()
                NumCores = LocalServerSettings[1]
                if (Server in ["localhost", LocalServer]):
                    try:
                        self.Fit_NumProc = int(NumCores)
                    except:
                        pass
                    break

                # Debug:
                # print ("Server = ", Server)
                # print ("NumCores = ", NumCores)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## import and (if necessary) convert obs. data file(s) and get parameters from obs. xml file


        ## prepare parameters for import of molecular parameters
        TransFlag = True
        TransDict = {}
        TransDict['MolNameList'] = self.MolNameList
        TransDict['SQLParamArray'] = self.SQLParamArray


        ## import data files and get xml paramters
        ParamDict = AnaylzeObsXMLFile(LocalObsXMLFileName, RegionFileName, FITSImageMaskFileName, \
                                     self.NameOfFunction, GlobalThreshold = GlobalThreshold, \
                                     TransFlag = TransFlag, TransDict = TransDict, FastFitFlag = True, \
                                     FullFitFlag = self.FullFitFlag, \
                                     RemoveContinuumFlag = RemoveContinuumFlag, daskFlag = self.daskFlag, \
                                     EstimateContinuumFlag = EstimateContinuumFlag, \
                                     EstimateConNumEntireCubeFlag = EstimateConNumEntireCubeFlag, \
                                     EstimateConNumParts = EstimateConNumParts, \
                                     NumberCoresEstCont = self.Fit_NumProc, CompRegions = CompRegions, \
                                     FITSImageMask = FITSImageMask)


        ## get parameters from dictionary
        self.DataFileFormat = ParamDict['DataFileFormat']
        if (self.DataFileFormat != "fits"):
            self.NoFITSCubeFlag = True
        self.CompRegions = ParamDict['CompRegions']
        self.FITSImageMask = ParamDict['FITSImageMask']
        self.FITSImageMaskValues = ParamDict['FITSImageMaskValues']
        self.ExpFileList = ParamDict['ExpFileList']
        self.dbFile = ParamDict['dbFileName']
        self.ListOfCubes = ParamDict['ListOfCubes']
        self.ListOfFullCubes = ParamDict['ListOfFullCubes']
        self.ListOfSpectralAxis = ParamDict['ListOfSpectralAxis']
        self.ListOfFullSpectralAxis = ParamDict['ListOfFullSpectralAxis']
        self.OrigRestFreq = ParamDict['OrigRestFreq']
        self.OrigUnits = ParamDict['OrigUnits']
        self.RangeParam = ParamDict['RangeParam']
        self.PixelMask = ParamDict['PixelMask']
        self.ParameterMapHeader = ParamDict['ParameterMapHeader']
        self.ParameterMapWCS = ParamDict['ParameterMapWCS']
        self.NumYPixels = ParamDict['NumYPixels']
        self.NumXPixels = ParamDict['NumXPixels']
        self.TotalNumPixel = ParamDict['TotalNumPixel']
        self.ListOfSelectedPixels = ParamDict['ListOfSelectedPixels']
        self.dec = ParamDict['DECAxis']
        self.ra = ParamDict['RAAxis']

        # Debug:
        # print ("ParamDict = ", ParamDict)
        # print ("self.FullFitFlag = ", self.FullFitFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## only for full-fit method: initialize XCLASS package
        if (self.FullFitFlag):
            self.InitializeXCLASS(ParameterDict)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## get parameter maps from file
        self.ListOfInitialMaps = self.GetParameterMapsFromFile(ParameterMapDir, CompRegions)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## prepare normal fitting
        if (not self.FastFitFlag):
            self.Fit_NumProc = min(self.Fit_NumProc, self.TotalNumPixel)
            self.LocalParamMapIter = (-1)                                                   ## indicates initialization


            ## some parameter from obs. xml file
            NumModelPixelXXList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "NumModelPixelXX")
            NumModelPixelYYList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "NumModelPixelYY")
            LocalOverlapFlagList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "LocalOverlap_Flag")
            NoSubBeamFlagList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "NoSubBeam_Flag")
            LocalEmAbsPATH = None
            EmAbsPATHList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "EmAbsPATH")
            if (EmAbsPATHList != []):
                print ("Import emission and absorption functions from files ..")
                LocalEmAbsPATH = EmAbsPATHList[0]
                EmsAbsFileInterDict = task_LineIdentification.GetEmAbsFunc(LocalEmAbsPATH)
            else:
                LocalEmAbsPATH = None
                EmsAbsFileInterDict = None


            ## start loop over all pixels
            counterPixels = 0
            for ypos in range(self.NumYPixels):
                for xpos in range(self.NumXPixels):
                    counterPixels += 1
                    if (self.PixelMask[ypos, xpos]):


                        ##********************************************************************************************************************************
                        ## prepare pixel directory
                        xPixelCoord = self.ra[xpos].value
                        yPixelCoord = self.dec[ypos].value
                        xString = "{:.10f}".format(xPixelCoord)
                        yString = "{:.10f}".format(yPixelCoord)
                        PixelDirectory = self.myXCLASSMapFitJobDir + "Pixel-Fits/" + xString + "__" + yString + "___"
                        PixelDirectory += str(xpos + 1) + "_-_" + str(ypos + 1)
                        PixelDirectory = os.path.normpath(PixelDirectory) + "/"


                        ## print current pixel
                        print_string = "\rPrepare files for fit (" + str(counterPixels) + "/" + str(self.TotalNumPixel)
                        print_string += ") for pixel (" + str(xPixelCoord) + "/" + str(yPixelCoord) + ") ..{:30s}".format(" ")


                        ## create pixel directory
                        cmdstring = "mkdir -p " + PixelDirectory
                        os.system(cmdstring)


                        ##********************************************************************************************************************************
                        ## prepare molfit array and update fit parameter
                        LocalMolfitFileName = PixelDirectory + "all-molecules.molfit"
                        self.UpdateMolfitFile(LocalMolfitFileName, ypos, xpos)


                        ##********************************************************************************************************************************
                        ## write iso ratio to local directory
                        IsoTableFileName = ""
                        if (self.IsoFlag):
                            IsoTableFileName = PixelDirectory + "iso.dat"
                            self.UpdateIsoRatioFile(IsoTableFileName, ypos, xpos)


                        ##********************************************************************************************************************************
                        ## construct fit.xml file, if no path for another algorithm xml file is given


                        ## adjust algorithm xml file (if defined)
                        AlgorithmXMLFileName = AlgorithmXMLFileName.strip()
                        if (AlgorithmXMLFileName != ""):


                            ## check existence of algorithm xml file and copy algorithm xml file to current working directory
                            LocalPrintFlag = False
                            NameOfFile = "algorithm xml"
                            LocalAlgorithmXMLFileName = task_myXCLASS.CheckNCopy(AlgorithmXMLFileName, self.NameOfFunction, NameOfFile, \
                                                                                 CurrentDir, PixelDirectory, LocalPrintFlag)
                            # Debug:
                            # print ("\nLocalAlgorithmXMLFileName = ", LocalAlgorithmXMLFileName)


                            ## rename algorithm xml
                            NewAlgorithmXMLFileName = PixelDirectory + "algorithm_control.xml"
                            task_myXCLASS.NormNCopyMove(LocalAlgorithmXMLFileName, NewAlgorithmXMLFileName, copyFlag = False)


                        ## if no algorithm xml file is defined, create one
                        else:
                            NewAlgorithmXMLFileName = os.path.normpath(PixelDirectory + "algorithm_control.xml")
                            task_myXCLASSFit.CreateLMControlXMLFile(PixelDirectory, NumberIteration)


                        ##********************************************************************************************************************************
                        ## construct obs. xml file
                        ListASCIIDataFileName = []
                        NumberFrequencyRanges = []
                        TelescopeSizeList = []
                        BMINList = []
                        BMAJList = []
                        BPAList = []
                        InterFlagList = []
                        GlobalvLSRList = []
                        RedshiftList = []
                        RangeIndex = []
                        FreqMinList = []
                        FreqMaxList = []
                        FreqStepList = []
                        t_back_flagList = []
                        tBackList = []
                        tSlopeList = []
                        BackgroundFileNameList = []
                        N_HList = []
                        beta_dustList = []
                        kappa_1300List = []
                        DustFileNameList = []
                        LocalContPhenFuncIDList = []
                        LocalContPhenFuncParam1List = []
                        LocalContPhenFuncParam2List = []
                        LocalContPhenFuncParam3List = []
                        LocalContPhenFuncParam4List = []
                        LocalContPhenFuncParam5List = []
                        for RangeID, ObsXMLParameterList in enumerate(self.RangeParam):


                            ## get obs. xml parameters
                            NumberFrequencyRanges.append([RangeID, 1])
                            TelescopeSizeList.append([RangeID, ObsXMLParameterList['TelescopeSize']])
                            BMINList.append([RangeID, ObsXMLParameterList['BMIN']])
                            BMAJList.append([RangeID, ObsXMLParameterList['BMAJ']])
                            BPAList.append([RangeID, ObsXMLParameterList['BPA']])
                            InterFlagList.append([RangeID, CheckBool(ObsXMLParameterList['InterFlag'])])
                            GlobalvLSRList.append([RangeID, ObsXMLParameterList['GlobalvLSR']])
                            RedshiftList.append([RangeID, ObsXMLParameterList['Redshift']])
                            RangeIndex = ObsXMLParameterList['RangeIndex']
                            FreqMinList.append([RangeID, 0, ObsXMLParameterList['FreqMin']])
                            FreqMaxList.append([RangeID, 0, ObsXMLParameterList['FreqMax']])
                            FreqStepList.append([RangeID, 0, ObsXMLParameterList['FreqStep']])
                            t_back_flagList.append([RangeID, 0, CheckBool(ObsXMLParameterList['t_back_flag'])])
                            tBack = ObsXMLParameterList['tBack']
                            if (tBack is None):                                             ## check, if continuum was estimated
                                TBackImage = ObsXMLParameterList['TBackImage']
                                tBack = TBackImage[ypos, xpos]
                                TSlopeImage = ObsXMLParameterList['TSlopeImage']
                                tSlope = TSlopeImage[ypos, xpos]
                            else:
                                tSlope = ObsXMLParameterList['tSlope']
                            tBackList.append([RangeID, 0, tBack])
                            tSlopeList.append([RangeID, 0,tSlope])
                            N_HList.append([RangeID, 0, ObsXMLParameterList['N_H']])
                            beta_dustList.append([RangeID, 0, ObsXMLParameterList['beta_dust']])
                            kappa_1300 = ObsXMLParameterList['kappa_1300']
                            kappa_1300 /= (2.0 * 1.66e-24 / 100.0)
                            kappa_1300List.append([RangeID, 0, kappa_1300])
                            LocalContPhenFuncIDList.append([RangeID, 0, ObsXMLParameterList['ContPhenFuncID']])
                            LocalContPhenFuncParam1List.append([RangeID, 0, ObsXMLParameterList['ContPhenFuncParam1']])
                            LocalContPhenFuncParam2List.append([RangeID, 0, ObsXMLParameterList['ContPhenFuncParam2']])
                            LocalContPhenFuncParam3List.append([RangeID, 0, ObsXMLParameterList['ContPhenFuncParam3']])
                            LocalContPhenFuncParam4List.append([RangeID, 0, ObsXMLParameterList['ContPhenFuncParam4']])
                            LocalContPhenFuncParam5List.append([RangeID, 0, ObsXMLParameterList['ContPhenFuncParam5']])
                            LocalBackgroundCube = ObsXMLParameterList['BackgroundCube']
                            LocalDustCube = ObsXMLParameterList['DustCube']


                            ## store local obs. spectrum to file
                            LocalASCIIDataFileName = PixelDirectory + "obs-spectrum__{:d}.dat".format(RangeID + 1)
                            LocalSpectrum = self.ListOfCubes[RangeID][:, ypos, xpos].value
                            LocalSpectrum = numpy.vstack((self.ListOfSpectralAxis[RangeID].value, LocalSpectrum)).T
                            numpy.savetxt(LocalASCIIDataFileName, LocalSpectrum)
                            ListASCIIDataFileName.append([RangeID, LocalASCIIDataFileName])


                            ## store local background spectrum to file
                            if (LocalBackgroundCube is not None):
                                LocalBackgroundDataFileName = PixelDirectory + "background-spectrum__{:d}.dat".format(RangeID + 1)
                                LocalSpectrum = LocalBackgroundCube[:, ypos, xpos].value
                                LocalSpectrum = numpy.nan_to_num(LocalSpectrum)
                                LocalSpectrum = numpy.vstack((self.ListOfSpectralAxis[RangeID].value, LocalSpectrum)).T
                                numpy.savetxt(LocalBackgroundDataFileName, LocalSpectrum)
                                BackgroundFileNameList.append(LocalBackgroundDataFileName)
                            else:
                                BackgroundFileNameList.append([])


                            ## store local dust opacity to file
                            if (LocalDustCube is not None):
                                LocalDustFileName = PixelDirectory + "dust-opacity__{:d}.dat".format(RangeID + 1)
                                LocalOpacity = LocalDustCube[:, ypos, xpos].value
                                LocalOpacity = numpy.nan_to_num(LocalOpacity)
                                LocalOpacity = numpy.vstack((self.ListOfSpectralAxis[RangeID].value, LocalSpectrum)).T
                                numpy.savetxt(LocalDustFileName, LocalOpacity)
                                DustFileNameList.append(LocalDustFileName)
                            else:
                                DustFileNameList.append([])


                        ## create obs. xml file
                        nH_flagList = []
                        ErrorYFlag = []
                        NumberHeaderLines = []
                        SeparatorColumns = []
                        Threshold = []
                        NoiseLevel = []
                        ObsXMLFileName = PixelDirectory + "obs.xml"
                        task_myXCLASSFit.CreateExpXMLFile(ObsXMLFileName, ListASCIIDataFileName, NumberFrequencyRanges, FreqMinList, FreqMaxList, \
                                                          FreqStepList, t_back_flagList, tBackList, tSlopeList, nH_flagList, N_HList, beta_dustList, \
                                                          kappa_1300List, DustFileNameList, BackgroundFileNameList, \
                                                          LocalContPhenFuncIDList, LocalContPhenFuncParam1List, LocalContPhenFuncParam2List,\
                                                          LocalContPhenFuncParam3List, LocalContPhenFuncParam4List, LocalContPhenFuncParam5List, \
                                                          Threshold, NoiseLevel, TelescopeSizeList, BMINList, BMAJList, BPAList, InterFlagList, \
                                                          GlobalvLSRList, RedshiftList, ErrorYFlag, NumberHeaderLines, SeparatorColumns, self.IsoFlag, \
                                                          IsoTableFileName, self.dbFile, NumModelPixelXXList, \
                                                          NumModelPixelYYList, LocalOverlapFlagList, NoSubBeamFlagList, EmAbsPATHList)


                        ##********************************************************************************************************************************
                        ## construct i/o xml file
                        task_myXCLASSFit.CreateIOControlXMLFile(PixelDirectory, ObsXMLFileName, LocalMolfitFileName, self.MAGIXrootDir)

        # Debug:
        # print ("self.RangeList = ", self.RangeList)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## split a string into fixed length chunks
    ## taken from https://stackoverflow.com/questions/18854620/whats-the-best-way-to-split-a-string-into-fixed-length-chunks-and-work-with-the/18854817
    ##
    def SplitString(self, FullString, length):
        """

    input parameters:
    -----------------

        - FullString:               full string

        - length:                   length of chunks


    output parameters:
    ------------------

        - list_of_strings:          list of chunks
        """

        # Debug:
        # print ("FullString = ", FullString)
        # print ("length = ", length)


        list_of_strings = []
        for i in range(0, len(FullString), length):
            list_of_strings.append(FullString[i:length+i])


        ## we're done
        return list_of_strings
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## FULL XCLASS: initialize XCLASS package
    ##
    def InitializeXCLASS(self, ParameterDict):
        """

    input parameters:
    -----------------

        - ParameterDict:                dictionary with parameters


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("ParameterDict = ", ParameterDict)


        ## print what you do
        print ("\n\nInitialize XCLASS interface .. ", end = "", flush = True)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## create look-up directory for molfit paramters
        ##
        ## myXCLASSParameter(1, c)  T_rot (=0), T_e (=1),    T_e (=2),      T_cont_dust (=3),        - (=4),     - (=5)
        ## myXCLASSParameter(2, c)  N_tot (=0), EM_RRL (=1), EM_RRL (=2),   nHcolumn_cont_dust (=3), - (=4),     - (=5)
        ## myXCLASSParameter(3, c)  - (=0),     - (=1),      -,             kappa_cont_dust (=3),    - (=4),     - (=5)
        ## myXCLASSParameter(4, c)  V_width_Gauss,                          beta_cont_dust (=3),     - (=4),     - (=5)
        ## myXCLASSParameter(5, c)  -                                       - (=3),                  - (=4),     - (=5)
        ## myXCLASSParameter(6, c)  V_Off (=0,1,2)                          - (=3),                  - (=4),     - (=5)
        ## myXCLASSParameter(7, c)  T_dOff
        ## myXCLASSParameter(8, c)  T_dSlope
        ## myXCLASSParameter(9, c)  nHcolumn
        ## myXCLASSParameter(10, c) kappa
        ## myXCLASSParameter(11, c) beta
        ## myXCLASSParameter(12, c) LayerDistance (=0,1,2,3,4,5,6)
        ## myXCLASSParameter(13, c) CFFlag (=0,1,2,3,4,5,6)
        ## myXCLASSParameter(14, c) source size, source radius
        ## myXCLASSParameter(15, c) source_center_x
        ## myXCLASSParameter(16, c) source_center_y


        ## initialize dictionary
        self.ParameterNameLookUpDict = {}


        ## parameters of column 1
        self.ParameterNameLookUpDict['t_rot'] = 0
        self.ParameterNameLookUpDict['t_e'] = 0
        self.ParameterNameLookUpDict['t_cont_dust'] = 0


        ## parameters of column 2
        self.ParameterNameLookUpDict['n_tot'] = 1
        self.ParameterNameLookUpDict['em_rrl'] = 1
        self.ParameterNameLookUpDict['nhcolumn_cont_dust'] = 1


        ## parameters of column 3
        self.ParameterNameLookUpDict['n_e'] = 2
        self.ParameterNameLookUpDict['kappa_cont_dust'] = 2


        ## parameters of column 4
        self.ParameterNameLookUpDict['v_width_gauss'] = 3
        self.ParameterNameLookUpDict['beta_cont_dust'] = 3


        ## parameters of column 5
        # -


        ## parameters of column 6
        self.ParameterNameLookUpDict['v_off'] = 5


        ## parameters of column 7
        self.ParameterNameLookUpDict['t_doff'] = 6


        ## parameters of column 8
        self.ParameterNameLookUpDict['t_dslope'] = 7


        ## parameters of column 9
        self.ParameterNameLookUpDict['nhcolumn'] = 8


        ## parameters of column 10
        self.ParameterNameLookUpDict['kappa'] = 9


        ## parameters of column 11
        self.ParameterNameLookUpDict['beta'] = 10


        ## parameters of column 12
        self.ParameterNameLookUpDict['layerdistance'] = 11


        ## parameters of column 13
        self.ParameterNameLookUpDict['cfflag'] = 12


        ## parameters of column 14
        self.ParameterNameLookUpDict['source_size'] = 13
        self.ParameterNameLookUpDict['source_radius'] = 13


        ## parameters of column 15
        self.ParameterNameLookUpDict['source_center_x'] = 14


        ## parameters of column 16
        self.ParameterNameLookUpDict['source_center_y'] = 15


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## get some parameters from dictionary
        LocalObsXMLFileName = ParameterDict['LocalObsXMLFileName']

        # Debug:
        # print ("LocalObsXMLFileName = ", LocalObsXMLFileName)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## create instance xml file


        ## combine molfit and iso ratio files to xml file
        LocalPrintflag = False
        InstanceXMLFileName = self.myXCLASSMapFitJobDir + "instance.xml"
        ok = task_myXCLASS.ConversionMolfit2XML(LocalPrintflag, self.MolfitsFileName, InstanceXMLFileName)
        if (self.IsoFlag):
            ok = task_myXCLASS.GetIsoTableParameters(LocalPrintflag, self.IsoRatioFileName, InstanceXMLFileName)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## import xclass interface
        from xclass.lib import xclassinterface


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store some parameters to XCLASS interface
        lengthexpdata = [len(x) for x in self.ListOfSpectralAxis]
        expdatax = numpy.zeros( (len(self.ListOfSpectralAxis), max(lengthexpdata), 1) )
        expdatay = copy.deepcopy(expdatax)
        expdatae = copy.deepcopy(expdatax)
        ListTelescopeSize = []
        ListBMIN = []
        ListBMAJ = []
        ListBPA = []
        ListInterFlag = []
        ListGlobalvLSR = []
        ListRedshift = []
        ListFreqMin = []
        ListFreqMax = []
        ListFreqStep = []
        Listt_back_flag = []
        ListtBack = []
        ListtSlope = []
        ListDust_Flag_Range = []
        ListN_H = []
        Listbeta_dust = []
        Listkappa_1300 = []
        ListPhen_Flag_Range = []
        ListInitContPhen_Range = []
        LocalOverlapFlag = False
        NoSubBeamFlag = True
        NumModelPixelXX = 100
        NumModelPixelYY = 100
        for RangeID, ObsXMLParameterList in enumerate(self.RangeParam):


            ## get flag for local-overlap
            LocalOverlapFlag = ObsXMLParameterList['LocalOverlapFlag']


            ## get source velocity
            BMIN = ObsXMLParameterList['BMIN']
            if (BMIN is None):
                BMIN_val = 0.0
            else:
                BMIN_val = BMIN
            ListBMIN.append(BMIN_val)


            ## get source velocity
            BMAJ = ObsXMLParameterList['BMAJ']
            if (BMAJ is None):
                BMAJ_val = 0.0
            else:
                BMAJ_val = BMAJ
            ListBMAJ.append(BMAJ_val)


            ## get source velocity
            BPA = ObsXMLParameterList['BPA']
            if (BPA is None):
                BPA_val = 0.0
            else:
                BPA_val = BPA
            ListBPA.append(BPA_val)


            ## get size of telescope or circular beam
            TelescopeSize = ObsXMLParameterList['TelescopeSize']
            if (TelescopeSize is None or TelescopeSize == 0.0):
                TelescopeSize_val = 0.0
                if (BMIN is not None and BMAJ is not None):
                    TelescopeSize_val = (BMIN_val + BMAJ_val) / 2.0
            else:
                TelescopeSize_val = TelescopeSize
            ListTelescopeSize.append(TelescopeSize_val)


            ## get interferometer flag
            InterFlag = ObsXMLParameterList['InterFlag']
            if (InterFlag):
                InterFlag = 1
            else:
                InterFlag = 0
            ListInterFlag.append(InterFlag)


            ## get source velocity
            GlobalvLSR = ObsXMLParameterList['GlobalvLSR']
            if (GlobalvLSR is None):
                GlobalvLSR_val = 0.0
            else:
                GlobalvLSR_val = GlobalvLSR
            ListGlobalvLSR.append(GlobalvLSR_val)


            ## get redshift
            Redshift = ObsXMLParameterList['Redshift']
            if (Redshift is None):
                Redshift_val = 0.0
            else:
                Redshift_val = Redshift
            ListRedshift.append(Redshift_val)


            ## get min. frequency
            FreqMin = ObsXMLParameterList['FreqMin']
            if (FreqMin is None):
                FreqMin_val = 1.0
            else:
                FreqMin_val = FreqMin
            ListFreqMin.append(FreqMin_val)


            ## get max. frequency
            FreqMax = ObsXMLParameterList['FreqMax']
            if (FreqMax is None):
                FreqMax_val = 1.0
            else:
                FreqMax_val = FreqMax
            ListFreqMax.append(FreqMax_val)


            ## get step size of spectral axis
            FreqStep = ObsXMLParameterList['FreqStep']
            if (FreqStep is None):
                FreqStep_val = 1.0
            else:
                FreqStep_val = FreqStep
            ListFreqStep.append(FreqStep_val)


            ## get continuum flag
            t_back_flag = ObsXMLParameterList['t_back_flag']
            if (t_back_flag):
                t_back_flag = 1
            else:
                t_back_flag = 0
            Listt_back_flag.append(t_back_flag)


            ## get temperature slope
            tBack = ObsXMLParameterList['tBack']
            if (tBack is None):
                tBack_val = 0.0
            else:
                tBack_val = tBack
            ListtBack.append(tBack_val)


            ## get temperature slope
            tSlope = ObsXMLParameterList['tSlope']
            if (tSlope is None):
                tSlope_val = 0.0
            else:
                tSlope_val = tSlope
            ListtSlope.append(tSlope_val)


            ## get hydrogen column density
            N_H = ObsXMLParameterList['N_H']
            if (N_H is None):
                N_H_val = 0.0
            else:
                N_H_val = N_H
            ListN_H.append(N_H_val)


            ## get spectral index of dust
            beta_dust = ObsXMLParameterList['beta_dust']
            if (beta_dust is None):
                beta_dust_val = 0.0
            else:
                beta_dust_val = beta_dust
            Listbeta_dust.append(beta_dust_val)


            ## get dust opacity
            kappa_1300 = ObsXMLParameterList['kappa_1300']
            if (kappa_1300 is None):
                kappa_1300_val = 0.0
            else:
                kappa_1300_val = kappa_1300
            Listkappa_1300.append(kappa_1300_val)


            ## set dust flag for current range
            if (N_H is not None or beta_dust is not None or kappa_1300 is not None):
                Dust_Flag_Range = 1
            else:
                Dust_Flag_Range = 0
            ListDust_Flag_Range.append(Dust_Flag_Range)


            ## get function id of phen. continuum description
            LocalContList = []
            LocalContPhenFuncID = ObsXMLParameterList['ContPhenFuncID']
            if (LocalContPhenFuncID is None):
                LocalContPhenFuncID_val = 0.0
            else:
                LocalContPhenFuncID_val = LocalContPhenFuncID
            LocalContList.append(LocalContPhenFuncID_val)


            ## get first parameter for phen. continuum description
            LocalContPhenFuncParam1 = ObsXMLParameterList['ContPhenFuncParam1']
            if (LocalContPhenFuncParam1 is None):
                LocalContPhenFuncParam1_val = 0.0
            else:
                LocalContPhenFuncParam1_val = LocalContPhenFuncParam1
            LocalContList.append(LocalContPhenFuncParam1_val)


            ## get second parameter for phen. continuum description
            LocalContPhenFuncParam2 = ObsXMLParameterList['ContPhenFuncParam2']
            if (LocalContPhenFuncParam2 is None):
                LocalContPhenFuncParam2_val = 0.0
            else:
                LocalContPhenFuncParam2_val = LocalContPhenFuncParam1
            LocalContList.append(LocalContPhenFuncParam2_val)


            ## get third parameter for phen. continuum description
            LocalContPhenFuncParam3 = ObsXMLParameterList['ContPhenFuncParam3']
            if (LocalContPhenFuncParam3 is None):
                LocalContPhenFuncParam3_val = 0.0
            else:
                LocalContPhenFuncParam3_val = LocalContPhenFuncParam3
            LocalContList.append(LocalContPhenFuncParam3_val)


            ## get fourth parameter for phen. continuum description
            LocalContPhenFuncParam4 = ObsXMLParameterList['ContPhenFuncParam4']
            if (LocalContPhenFuncParam4 is None):
                LocalContPhenFuncParam4_val = 0.0
            else:
                LocalContPhenFuncParam4_val = LocalContPhenFuncParam4
            LocalContList.append(LocalContPhenFuncParam4_val)


            ## get fifth parameter for phen. continuum description
            LocalContPhenFuncParam5 = ObsXMLParameterList['ContPhenFuncParam5']
            if (LocalContPhenFuncParam5 is None):
                LocalContPhenFuncParam5_val = 0.0
            else:
                LocalContPhenFuncParam5_val = LocalContPhenFuncParam5
            LocalContList.append(LocalContPhenFuncParam5_val)
            ListInitContPhen_Range.append(LocalContList)


            ## set phen. cont. flag for current range
            if (LocalContPhenFuncID is not None or LocalContPhenFuncParam1 is not None \
                or LocalContPhenFuncParam2 is not None or LocalContPhenFuncParam3 is not None \
                or LocalContPhenFuncParam4 is not None or LocalContPhenFuncParam5 is not None):
                Phen_Flag_Range = 1
            else:
                Phen_Flag_Range = 0
            ListPhen_Flag_Range.append(Phen_Flag_Range)


            ## get number of model pixel for sub-beam description in x-direction
            NumModelPixelXX = ObsXMLParameterList['NumModelPixelXX']
            if (NumModelPixelXX is None):
                NumModelPixelXX = 0.0


            ## get number of model pixel for sub-beam description in y-direction
            NumModelPixelYY = ObsXMLParameterList['NumModelPixelYY']
            if (NumModelPixelYY is None):
                NumModelPixelYY = 0.0


            ## get flag for sub-beam description
            NoSubBeamFlag = ObsXMLParameterList['NoSubBeamFlag']
            if (NoSubBeamFlag):
                NoSubBeamFlag = 1
            else:
                NoSubBeamFlag = 0


            ## get path and name of iso ratio file
            IsoRatioFileName = ObsXMLParameterList['IsoRatioFileName']
            if (IsoRatioFileName is None):
                IsoRatioFileName = ""


            ## get path and name of database file
            dbFileName = ObsXMLParameterList['dbFileName']
            if (dbFileName is None):
                dbFileName = ""


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## copy frequency axis
            LocalFreqAxis = self.ListOfSpectralAxis[RangeID].value
            DeltaFreq = LocalFreqAxis[1] - LocalFreqAxis[0]                                 ## in order to avoid modification on the XCLASS core functions
            if (DeltaFreq < 0.0):                                                           ## flip frequency axis to get positive freq. step sizes
                expdatax[RangeID, 0:len(self.ListOfSpectralAxis[RangeID]), 0] = numpy.flip(LocalFreqAxis)
            else:
                expdatax[RangeID, 0:len(self.ListOfSpectralAxis[RangeID]), 0] = LocalFreqAxis

        # Debug:
        # print ("expdatax.shape = ", expdatax.shape)


        ## store frequencies to XCLASS interface
        xclassinterface.inter.initexpdatax = expdatax
        xclassinterface.inter.initexpdatay = expdatay
        xclassinterface.inter.initexpdatae = expdatae
        xclassinterface.inter.initlengthexpdata = lengthexpdata


        ## pass variables to fortran module
        xclassinterface.inter.inittotalnumberoffrequencyranges = len(ListFreqMin)
        xclassinterface.inter.initstartfrequency = ListFreqMin
        xclassinterface.inter.initendfrequency = ListFreqMax
        xclassinterface.inter.initstepfrequency = ListFreqStep
        xclassinterface.inter.inittelescopesize = ListTelescopeSize
        xclassinterface.inter.inittelescopebmin = ListBMIN
        xclassinterface.inter.inittelescopebmaj = ListBMAJ
        xclassinterface.inter.inittelescopebpa = ListBPA
        xclassinterface.inter.initinterflag = ListInterFlag
        xclassinterface.inter.initglobalvlsr = ListGlobalvLSR
        xclassinterface.inter.initredshift_range = ListRedshift
        xclassinterface.inter.inittbflag = Listt_back_flag
        xclassinterface.inter.initbackgroundtemperaturerange = ListtBack
        xclassinterface.inter.inittemperaturesloperange = ListtSlope
        xclassinterface.inter.initdust_flag_range = ListDust_Flag_Range
        xclassinterface.inter.inithydrogencolumndensityrange = ListN_H
        xclassinterface.inter.initdustbetarange = Listbeta_dust
        xclassinterface.inter.initkapparange = Listkappa_1300
        xclassinterface.inter.initphen_flag_range = ListPhen_Flag_Range
        xclassinterface.inter.initcontphen_range = ListInitContPhen_Range
        xclassinterface.inter.initnosubbeamflag = NoSubBeamFlag
        xclassinterface.inter.initlocaloverlapflag = LocalOverlapFlag
        xclassinterface.inter.initnumbermodelpixelxx = NumModelPixelXX
        xclassinterface.inter.initnumbermodelpixelyy = NumModelPixelYY
        xclassinterface.inter.initdbname = "{:4096s}".format(dbFileName)
        xclassinterface.inter.initemsabspath = "{:4096s}".format("")            ## <--------------------- add ems/abs path here or at function calling ??


        ## store instance xml file
        xclassinterface.inter.instancefilename = "{:4096s}".format(InstanceXMLFileName)

        # Debug:
        # print ("InstanceXMLFileName = ", InstanceXMLFileName)
        # print (xclassinterface.inter.__doc__)
        # print ("dbFileName = ", dbFileName)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize XCLASS package
        xclassinterface.inter.initxclass()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize directory for XCLASS
        self.XCLASSDict = {}


        ## general variables
        self.XCLASSDict['ErrChannelIndex'] = xclassinterface.inter.initerrchannelindex
        self.XCLASSDict['DebugChannel'] = xclassinterface.inter.initdebugchannel
        self.XCLASSDict['AllErrChannels'] = xclassinterface.inter.initallerrchannels
        self.XCLASSDict['tt1'] = xclassinterface.inter.inittt1
        self.XCLASSDict['tt2'] = xclassinterface.inter.inittt2
        self.XCLASSDict['debug_freq'] = xclassinterface.inter.initdebug_freq
        self.XCLASSDict['UseRegistrationXMLFile'] = xclassinterface.inter.inituseregistrationxmlfile
        self.XCLASSDict['AllOutputFilesFlag'] = xclassinterface.inter.initalloutputfilesflag
        self.XCLASSDict['EmAbsFlag'] = xclassinterface.inter.initemabsflag
        self.XCLASSDict['IntegrationFlag'] = xclassinterface.inter.initintegrationflag
        self.XCLASSDict['LocalOverlapFlag'] = xclassinterface.inter.initlocaloverlapflag
        self.XCLASSDict['NoSubBeamFlag'] = xclassinterface.inter.initnosubbeamflag
        self.XCLASSDict['ParallelizationMethod'] = str(xclassinterface.inter.initparallelizationmethod.astype(str))
        self.XCLASSDict['XCLASSRootDir'] = str(xclassinterface.inter.initxclassrootdir.astype(str))


        ## variables for molfit file
        self.XCLASSDict['TotalNumberDataPoints'] = xclassinterface.inter.inittotalnumberdatapoints
        self.XCLASSDict['NumberMolecules'] = xclassinterface.inter.initnumbermolecules
        self.XCLASSDict['TotalNumberComponents'] = xclassinterface.inter.inittotalnumbercomponents
        self.XCLASSDict['TotalNumberOfFrequencyRanges'] = xclassinterface.inter.inittotalnumberoffrequencyranges
        self.XCLASSDict['TotalNumberOfMolecules'] = xclassinterface.inter.inittotalnumberofmolecules
        self.XCLASSDict['NumberMolfitMolecules'] = xclassinterface.inter.initnumbermolfitmolecules
        self.XCLASSDict['NumberDistances'] = xclassinterface.inter.initnumberdistances
        self.XCLASSDict['NumberModelPixelXX'] = xclassinterface.inter.initnumbermodelpixelxx
        self.XCLASSDict['NumberModelPixelYY'] = xclassinterface.inter.initnumbermodelpixelyy
        self.XCLASSDict['CompMoleculeIndex'] = xclassinterface.inter.initcompmoleculeindex
        self.XCLASSDict['SpectrumIndexForFreqRange'] = xclassinterface.inter.initspectrumindexforfreqrange
        self.XCLASSDict['NumberComponentsPerMolecule'] = xclassinterface.inter.initnumbercomponentspermolecule
        self.XCLASSDict['KindOfMolecule'] = xclassinterface.inter.initkindofmolecule
        self.XCLASSDict['GeometryFlag'] = xclassinterface.inter.initgeometryflag
        self.XCLASSDict['LineProfileFunction'] = xclassinterface.inter.initlineprofilefunction
        self.XCLASSDict['NumTransFreqPerObsFreq'] = xclassinterface.inter.initnumtransfreqperobsfreq
        self.XCLASSDict['ConversionTableMAGIXmyXCLASSParam'] = xclassinterface.inter.initconversiontablemagixmyxclassparam
        self.XCLASSDict['DataPointIndexFreqRange'] = xclassinterface.inter.initdatapointindexfreqrange
        self.XCLASSDict['DistanceOrderingArray'] = xclassinterface.inter.initdistanceorderingarray
        self.XCLASSDict['ConfigIndex'] = xclassinterface.inter.initconfigindex
        self.XCLASSDict['SizeOfPixel_deg'] = xclassinterface.inter.initsizeofpixel_deg
        self.XCLASSDict['StartFrequency'] = xclassinterface.inter.initstartfrequency
        self.XCLASSDict['EndFrequency'] = xclassinterface.inter.initendfrequency
        self.XCLASSDict['StepFrequency'] = xclassinterface.inter.initstepfrequency
        self.XCLASSDict['TelescopeSize'] = xclassinterface.inter.inittelescopesize
        self.XCLASSDict['TelescopeBMIN'] = xclassinterface.inter.inittelescopebmin
        self.XCLASSDict['TelescopeBMAJ'] = xclassinterface.inter.inittelescopebmaj
        self.XCLASSDict['TelescopeBPA'] = xclassinterface.inter.inittelescopebpa
        self.XCLASSDict['GlobalvLSR'] = xclassinterface.inter.initglobalvlsr
        self.XCLASSDict['Redshift_Range'] = xclassinterface.inter.initredshift_range
        self.XCLASSDict['BackgroundTemperatureRange'] = xclassinterface.inter.initbackgroundtemperaturerange
        self.XCLASSDict['TemperatureSlopeRange'] = xclassinterface.inter.inittemperaturesloperange
        self.XCLASSDict['HydrogenColumnDensityRange'] = xclassinterface.inter.inithydrogencolumndensityrange
        self.XCLASSDict['DustBetaRange'] = xclassinterface.inter.initdustbetarange
        self.XCLASSDict['KappaRange'] = xclassinterface.inter.initkapparange
        self.XCLASSDict['DustTauFromFile'] = xclassinterface.inter.initdusttaufromfile
        self.XCLASSDict['BackgroundFromFile'] = xclassinterface.inter.initbackgroundfromfile
        self.XCLASSDict['SortedDistances'] = xclassinterface.inter.initsorteddistances
        self.XCLASSDict['PureDistances'] = xclassinterface.inter.initpuredistances
        self.XCLASSDict['ContPhen_Range'] = xclassinterface.inter.initcontphen_range
        self.XCLASSDict['myXCLASSParameter'] = xclassinterface.inter.initmyxclassparameter
        self.XCLASSDict['vWidthLimits'] = xclassinterface.inter.initvwidthlimits
        self.XCLASSDict['vOffLimits'] = xclassinterface.inter.initvofflimits
        self.XCLASSDict['ObservationalDataList'] = xclassinterface.inter.initobservationaldatalist
        self.XCLASSDict['GausssianBeamMap'] = xclassinterface.inter.initgausssianbeammap
        self.XCLASSDict['EmsAbsFunc'] = xclassinterface.inter.initemsabsfunc
        self.XCLASSDict['EmsAbsPATH'] = "{:4096s}".format(str(xclassinterface.inter.initemsabspath.astype(str)))
        self.XCLASSDict['MoleculeNames'] = xclassinterface.inter.initmoleculenames
        self.XCLASSDict['IsoFlag'] = xclassinterface.inter.initisoflag
        self.XCLASSDict['SubBeamDescriptionFlag'] = xclassinterface.inter.initsubbeamdescriptionflag
        self.XCLASSDict['UseEmAbsFuncFlag'] = xclassinterface.inter.inituseemabsfuncflag
        self.XCLASSDict['tbFlag'] = xclassinterface.inter.inittbflag
        self.XCLASSDict['InterFlag'] = xclassinterface.inter.initinterflag
        self.XCLASSDict['tdFlag'] = xclassinterface.inter.inittdflag
        self.XCLASSDict['nHFlag'] = xclassinterface.inter.initnhflag
        self.XCLASSDict['LTEFlag'] = xclassinterface.inter.initlteflag
        self.XCLASSDict['ThinFlag'] = xclassinterface.inter.initthinflag
        self.XCLASSDict['Dust_Flag_Range'] = xclassinterface.inter.initdust_flag_range
        self.XCLASSDict['Phen_Flag_Range'] = xclassinterface.inter.initphen_flag_range
        self.XCLASSDict['ConfigList'] = xclassinterface.inter.initconfiglist
        self.XCLASSDict['LayerMap'] = xclassinterface.inter.initlayermap


        ## variables for sqlite3 database
        self.XCLASSDict['MinNumTransitionsSQL'] = xclassinterface.inter.initminnumtransitionssql
        self.XCLASSDict['MaxNumTransitionsSQL'] = xclassinterface.inter.initmaxnumtransitionssql
        self.XCLASSDict['OrderTransSQL'] = xclassinterface.inter.initordertranssql
        self.XCLASSDict['NumEntriesRanges'] = xclassinterface.inter.initnumentriesranges
        self.XCLASSDict['MaxElowSQL'] = xclassinterface.inter.initmaxelowsql
        self.XCLASSDict['MingASQL'] = xclassinterface.inter.initmingasql
        self.XCLASSDict['NameOfPartFuncTable'] = str(xclassinterface.inter.initnameofpartfunctable.astype(str))
        self.XCLASSDict['NameOfRadTransTable'] = str(xclassinterface.inter.initnameofradtranstable.astype(str))
        self.XCLASSDict['dbName'] = str(xclassinterface.inter.initdbname.astype(str))


        ## variables for partition function
        self.XCLASSDict['NumberOfTemperatures'] = xclassinterface.inter.initnumberoftemperatures
        self.XCLASSDict['NumberMoleculePartFunc'] = xclassinterface.inter.initnumbermoleculepartfunc
        self.XCLASSDict['Firsti'] = xclassinterface.inter.initfirsti
        self.XCLASSDict['Lasti'] = xclassinterface.inter.initlasti
        self.XCLASSDict['stepi'] = xclassinterface.inter.initstepi
        self.XCLASSDict['TempLow'] = xclassinterface.inter.inittemplow
        self.XCLASSDict['TempHigh'] = xclassinterface.inter.inittemphigh
        self.XCLASSDict['TempPartFunc'] = xclassinterface.inter.inittemppartfunc
        self.XCLASSDict['lgQ'] = xclassinterface.inter.initlgq
        self.XCLASSDict['ColumnNameForNamePartFunc'] = str(xclassinterface.inter.initcolumnnamefornamepartfunc.astype(str))
        # self.XCLASSDict['MolNamePartFunc'] = xclassinterface.inter.initmolnamepartfunc
        # self.XCLASSDict['ColumnNamesPartFunc'] = xclassinterface.inter.initcolumnnamespartfunc


        ## variables for rad-trans parameter
        self.XCLASSDict['TotalNumberOfTransitions'] = xclassinterface.inter.inittotalnumberoftransitions
        self.XCLASSDict['TotalNumTransFreq'] = xclassinterface.inter.inittotalnumtransfreq
        self.XCLASSDict['Column300KPartFunc'] = xclassinterface.inter.initcolumn300kpartfunc
        self.XCLASSDict['DopplerShiftedTransFreqIndex'] = xclassinterface.inter.initdopplershiftedtransfreqindex
        self.XCLASSDict['MolecularDataIndices'] = xclassinterface.inter.initmoleculardataindices
        self.XCLASSDict['MolecularData'] = xclassinterface.inter.initmoleculardata
        self.XCLASSDict['DopplerShiftedTransFreq'] = xclassinterface.inter.initdopplershiftedtransfreq
        self.XCLASSDict['CentralOpticalDepth'] = xclassinterface.inter.initcentralopticaldepth
        self.XCLASSDict['ColumnNameForNameTransitions'] = str(xclassinterface.inter.initcolumnnamefornametransitions.astype(str))
        self.XCLASSDict['ColumnNameForFreqTransitions'] = str(xclassinterface.inter.initcolumnnameforfreqtransitions.astype(str))
        self.XCLASSDict['ColumnNameForIntTransitions'] = str(xclassinterface.inter.initcolumnnameforinttransitions.astype(str))
        self.XCLASSDict['ColumnNameForEinsteinATransitions'] = str(xclassinterface.inter.initcolumnnameforeinsteinatransitions.astype(str))
        self.XCLASSDict['ColumnNameForFreqErrTransitions'] = str(xclassinterface.inter.initcolumnnameforfreqerrtransitions.astype(str))
        self.XCLASSDict['ColumnNameForELowTransitions'] = str(xclassinterface.inter.initcolumnnameforelowtransitions.astype(str))
        self.XCLASSDict['ColumnNameForgUpTransitions'] = str(xclassinterface.inter.initcolumnnameforguptransitions.astype(str))
        self.XCLASSDict['ColumnNameForLowerQNTransitions'] = str(xclassinterface.inter.initcolumnnameforlowerqntransitions.astype(str))
        self.XCLASSDict['ColumnNameForUpperQNTransitions'] = str(xclassinterface.inter.initcolumnnameforupperqntransitions.astype(str))
        # self.XCLASSDict['LowerQNString'] = xclassinterface.inter.initlowerqnstring
        # self.XCLASSDict['UpperQNString'] = xclassinterface.inter.initupperqnstring


        # MolecularData = [\nu_Trans, Einstein-A, E_low, g_up]
        # print ("\n\n\n\n")
        # MolOut = []
        # for line in self.XCLASSDict['MolecularData']:
        #     MolOut.append(line[:4])
        #     print (" ".join("{:.8e}".format(x) for x in line[:4]))
        # MolOut = numpy.asarray(MolOut)
        # numpy.savetxt("mol_full.dat", MolOut)


        ## variables for iso table file
        self.XCLASSDict['NumberOfIsomasters'] = xclassinterface.inter.initnumberofisomasters
        self.XCLASSDict['NumberOfGlobalISORatios'] = xclassinterface.inter.initnumberofglobalisoratios
        self.XCLASSDict['IsoNfitConversionTable'] = xclassinterface.inter.initisonfitconversiontable
        self.XCLASSDict['GlobalIsoRatioParameter'] = xclassinterface.inter.initglobalisoratioparameter
        self.XCLASSDict['IsoRatio'] = xclassinterface.inter.initisoratio
        self.XCLASSDict['IsoRatioConversionTable'] = xclassinterface.inter.initisoratioconversiontable
        self.XCLASSDict['IsoTableFileName'] = "{:4096s}".format(str(xclassinterface.inter.initisotablefilename.astype(str)))
        self.XCLASSDict['IsoMolecule'] = xclassinterface.inter.initisomolecule
        self.XCLASSDict['IsoMasters'] = xclassinterface.inter.initisomasters

        # Debug:
        # print ("self.XCLASSDict['IsoMolecule'] = ", self.XCLASSDict['IsoMolecule'])
        # print ("self.XCLASSDict['IsoMasters'] = ", self.XCLASSDict['IsoMasters'])
        # sys.exit(0)


        ## remove instance xml file
        cmdString = "rm -rf " + InstanceXMLFileName
        os.system(cmdString)


        ## we're done
        print ("done!")

        # Debug:
        # print ("\n\nThe size of dictionary {:s} is {:d} bytes".format(chr(34) + "XCLASSDict" + chr(34), sys.getsizeof(self.XCLASSDict)))
        # for LocalKey in self.XCLASSDict.keys():
        #     print ("self.XCLASSDict[" + LocalKey + "] = ", self.XCLASSDict[LocalKey])
        # sys.exit(0)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## FULL: copy stored XCLASS variables to XCLASS interface
    ##
    def PrepareXCLASS(self, xclassinterface):
        """

    input parameters:
    -----------------

        - xclassinterface:          xclassinterface object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("xclassinterface = ", xclassinterface)
        # print ("self.XCLASSDict = ", self.XCLASSDict)


        ## general variables
        xclassinterface.inter.initerrchannelindex = self.XCLASSDict['ErrChannelIndex']
        # xclassinterface.inter.initdebugchannel = self.XCLASSDict['DebugChannel']
        xclassinterface.inter.initallerrchannels = self.XCLASSDict['AllErrChannels']
        xclassinterface.inter.inittt1 = self.XCLASSDict['tt1']
        xclassinterface.inter.inittt2 = self.XCLASSDict['tt2']
        # xclassinterface.inter.initdebug_freq = self.XCLASSDict['debug_freq']
        xclassinterface.inter.inituseregistrationxmlfile = self.XCLASSDict['UseRegistrationXMLFile']
        xclassinterface.inter.initalloutputfilesflag = self.XCLASSDict['AllOutputFilesFlag']
        xclassinterface.inter.initemabsflag = self.XCLASSDict['EmAbsFlag']
        xclassinterface.inter.initintegrationflag = self.XCLASSDict['IntegrationFlag']
        xclassinterface.inter.initlocaloverlapflag = self.XCLASSDict['LocalOverlapFlag']
        xclassinterface.inter.initnosubbeamflag = self.XCLASSDict['NoSubBeamFlag']
        xclassinterface.inter.initparallelizationmethod = self.XCLASSDict['ParallelizationMethod']
        xclassinterface.inter.initxclassrootdir = self.XCLASSDict['XCLASSRootDir']

        # Debug:
        # print ("\n\nIntegrationFlag = ", self.XCLASSDict['IntegrationFlag'])
        # print ("LocalOverlapFlag = ", self.XCLASSDict['LocalOverlapFlag'])
        # print ("NoSubBeamFlag = ", self.XCLASSDict['NoSubBeamFlag'])
        # print ("ParallelizationMethod = ", self.XCLASSDict['ParallelizationMethod'])
        # print ("XCLASSRootDir = ", self.XCLASSDict['XCLASSRootDir'].strip())
        # print ("TotalNumberDataPoints = ", self.XCLASSDict['TotalNumberDataPoints'])
        # print ("NumberMolecules = ", self.XCLASSDict['NumberMolecules'])
        # print ("TotalNumberComponents = ", self.XCLASSDict['TotalNumberComponents'])
        # print ("TotalNumberOfFrequencyRanges = ", self.XCLASSDict['TotalNumberOfFrequencyRanges'])
        # print ("TotalNumberOfMolecules = ", self.XCLASSDict['TotalNumberOfMolecules'])
        # print ("NumberMolfitMolecules = ", self.XCLASSDict['NumberMolfitMolecules'])
        # print ("NumberDistances = ", self.XCLASSDict['NumberDistances'])
        # print ("NumberModelPixelXX = ", self.XCLASSDict['NumberModelPixelXX'])
        # print ("NumberModelPixelYY = ", self.XCLASSDict['NumberModelPixelYY'])
        # print ("CompMoleculeIndex = ", self.XCLASSDict['CompMoleculeIndex'])
        # print ("SpectrumIndexForFreqRange = ", self.XCLASSDict['SpectrumIndexForFreqRange'])
        # print ("NumberComponentsPerMolecule = ", self.XCLASSDict['NumberComponentsPerMolecule'])
        # print ("KindOfMolecule = ", self.XCLASSDict['KindOfMolecule'])
        # print ("GeometryFlag = ", self.XCLASSDict['GeometryFlag'])
        # print ("LineProfileFunction = ", self.XCLASSDict['LineProfileFunction'])
        # print ("NumTransFreqPerObsFreq = ", self.XCLASSDict['NumTransFreqPerObsFreq'])
        # print ("ConversionTableMAGIXmyXCLASSParam = ", self.XCLASSDict['ConversionTableMAGIXmyXCLASSParam'])
        # print ("DataPointIndexFreqRange = ", self.XCLASSDict['DataPointIndexFreqRange'])
        # print ("DistanceOrderingArray = ", self.XCLASSDict['DistanceOrderingArray'])
        # print ("ConfigIndex = ", self.XCLASSDict['ConfigIndex'])
        # print ("SizeOfPixel_deg = ", self.XCLASSDict['SizeOfPixel_deg'])
        # print ("StartFrequency = ", self.XCLASSDict['StartFrequency'])
        # print ("EndFrequency = ", self.XCLASSDict['EndFrequency'])
        # print ("StepFrequency = ", self.XCLASSDict['StepFrequency'])
        # print ("TelescopeSize = ", self.XCLASSDict['TelescopeSize'])
        # print ("TelescopeBMIN = ", self.XCLASSDict['TelescopeBMIN'])
        # print ("TelescopeBMAJ = ", self.XCLASSDict['TelescopeBMAJ'])
        # print ("TelescopeBPA = ", self.XCLASSDict['TelescopeBPA'])
        # print ("GlobalvLSR = ", self.XCLASSDict['GlobalvLSR'])
        # print ("Redshift_Range = ", self.XCLASSDict['Redshift_Range'])
        # print ("BackgroundTemperatureRange = ", self.XCLASSDict['BackgroundTemperatureRange'])
        # print ("TemperatureSlopeRange = ", self.XCLASSDict['TemperatureSlopeRange'])
        # print ("HydrogenColumnDensityRange = ", self.XCLASSDict['HydrogenColumnDensityRange'])
        # print ("DustBetaRange = ", self.XCLASSDict['DustBetaRange'])
        # print ("KappaRange = ", self.XCLASSDict['KappaRange'])
        # print ("DustTauFromFile = ", self.XCLASSDict['DustTauFromFile'])
        # print ("BackgroundFromFile = ", self.XCLASSDict['BackgroundFromFile'])
        # print ("SortedDistances = ", self.XCLASSDict['SortedDistances'])
        # print ("PureDistances = ", self.XCLASSDict['PureDistances'])
        # print ("ContPhen_Range = ", self.XCLASSDict['ContPhen_Range'])
        # print ("myXCLASSParameter = ", self.XCLASSDict['myXCLASSParameter'])
        # print ("vWidthLimits = ", self.XCLASSDict['vWidthLimits'])
        # print ("vOffLimits = ", self.XCLASSDict['vOffLimits'])
        # print ("ObservationalDataList = ", self.XCLASSDict['ObservationalDataList'])
        # print ("GausssianBeamMap = ", self.XCLASSDict['GausssianBeamMap'])
        # print ("EmsAbsFunc = ", self.XCLASSDict['EmsAbsFunc'])
        # print ("EmsAbsPATH = ", self.XCLASSDict['EmsAbsPATH'].strip())
        # print ("MoleculeNames = ", self.XCLASSDict['MoleculeNames'])
        # print ("IsoFlag = ", self.XCLASSDict['IsoFlag'])
        # print ("SubBeamDescriptionFlag = ", self.XCLASSDict['SubBeamDescriptionFlag'])
        # print ("UseEmAbsFuncFlag = ", self.XCLASSDict['UseEmAbsFuncFlag'])
        # print ("tbFlag = ", self.XCLASSDict['tbFlag'])
        # print ("InterFlag = ", self.XCLASSDict['InterFlag'])
        # print ("tdFlag = ", self.XCLASSDict['tdFlag'])
        # print ("nHFlag = ", self.XCLASSDict['nHFlag'])
        # print ("LTEFlag = ", self.XCLASSDict['LTEFlag'])
        # print ("ThinFlag = ", self.XCLASSDict['ThinFlag'])
        # print ("Dust_Flag_Range = ", self.XCLASSDict['Dust_Flag_Range'])
        # print ("Phen_Flag_Range = ", self.XCLASSDict['Phen_Flag_Range'])
        # print ("ConfigList = ", self.XCLASSDict['ConfigList'])
        # print ("LayerMap = ", self.XCLASSDict['LayerMap'])
        # print ("MinNumTransitionsSQL = ", self.XCLASSDict['MinNumTransitionsSQL'])
        # print ("MaxNumTransitionsSQL = ", self.XCLASSDict['MaxNumTransitionsSQL'])
        # print ("OrderTransSQL = ", self.XCLASSDict['OrderTransSQL'])
        # print ("NumEntriesRanges = ", self.XCLASSDict['NumEntriesRanges'])
        # print ("MaxElowSQL = ", self.XCLASSDict['MaxElowSQL'])
        # print ("MingASQL = ", self.XCLASSDict['MingASQL'])
        # print ("NameOfPartFuncTable = ", self.XCLASSDict['NameOfPartFuncTable'])
        # print ("NameOfRadTransTable = ", self.XCLASSDict['NameOfRadTransTable'])
        # print ("NumberOfIsomasters = ", self.XCLASSDict['NumberOfIsomasters'])
        # print ("NumberOfGlobalISORatios = ", self.XCLASSDict['NumberOfGlobalISORatios'])
        # print ("IsoNfitConversionTable = ", self.XCLASSDict['IsoNfitConversionTable'])
        # print ("GlobalIsoRatioParameter = ", self.XCLASSDict['GlobalIsoRatioParameter'])
        # print ("IsoRatio = ", self.XCLASSDict['IsoRatio'])
        # print ("IsoRatioConversionTable = ", self.XCLASSDict['IsoRatioConversionTable'])
        # print ("IsoTableFileName = ", self.XCLASSDict['IsoTableFileName'].strip())
        # print ("IsoMolecule = ", self.XCLASSDict['IsoMolecule'])
        # print ("IsoMasters = ", self.XCLASSDict['IsoMasters'])


        ## variables for molfit file
        xclassinterface.inter.inittotalnumberdatapoints = self.XCLASSDict['TotalNumberDataPoints']
        xclassinterface.inter.initnumbermolecules = self.XCLASSDict['NumberMolecules']
        xclassinterface.inter.inittotalnumbercomponents = self.XCLASSDict['TotalNumberComponents']
        xclassinterface.inter.inittotalnumberoffrequencyranges = self.XCLASSDict['TotalNumberOfFrequencyRanges']
        xclassinterface.inter.inittotalnumberofmolecules = self.XCLASSDict['TotalNumberOfMolecules']
        xclassinterface.inter.initnumbermolfitmolecules = self.XCLASSDict['NumberMolfitMolecules']
        xclassinterface.inter.initnumberdistances = self.XCLASSDict['NumberDistances']
        xclassinterface.inter.initnumbermodelpixelxx = self.XCLASSDict['NumberModelPixelXX']
        xclassinterface.inter.initnumbermodelpixelyy = self.XCLASSDict['NumberModelPixelYY']
        xclassinterface.inter.initcompmoleculeindex = self.XCLASSDict['CompMoleculeIndex']
        xclassinterface.inter.initspectrumindexforfreqrange = self.XCLASSDict['SpectrumIndexForFreqRange']
        xclassinterface.inter.initnumbercomponentspermolecule = self.XCLASSDict['NumberComponentsPerMolecule']
        xclassinterface.inter.initkindofmolecule = self.XCLASSDict['KindOfMolecule']
        xclassinterface.inter.initgeometryflag = self.XCLASSDict['GeometryFlag']
        xclassinterface.inter.initlineprofilefunction = self.XCLASSDict['LineProfileFunction']
        xclassinterface.inter.initnumtransfreqperobsfreq = self.XCLASSDict['NumTransFreqPerObsFreq']
        xclassinterface.inter.initconversiontablemagixmyxclassparam = self.XCLASSDict['ConversionTableMAGIXmyXCLASSParam']
        xclassinterface.inter.initdatapointindexfreqrange = self.XCLASSDict['DataPointIndexFreqRange']
        xclassinterface.inter.initdistanceorderingarray = self.XCLASSDict['DistanceOrderingArray']
        xclassinterface.inter.initconfigindex = self.XCLASSDict['ConfigIndex']
        xclassinterface.inter.initsizeofpixel_deg = self.XCLASSDict['SizeOfPixel_deg']
        xclassinterface.inter.initstartfrequency = self.XCLASSDict['StartFrequency']
        xclassinterface.inter.initendfrequency = self.XCLASSDict['EndFrequency']
        xclassinterface.inter.initstepfrequency = self.XCLASSDict['StepFrequency']
        xclassinterface.inter.inittelescopesize = self.XCLASSDict['TelescopeSize']
        xclassinterface.inter.inittelescopebmin = self.XCLASSDict['TelescopeBMIN']
        xclassinterface.inter.inittelescopebmaj = self.XCLASSDict['TelescopeBMAJ']
        xclassinterface.inter.inittelescopebpa = self.XCLASSDict['TelescopeBPA']
        xclassinterface.inter.initglobalvlsr = self.XCLASSDict['GlobalvLSR']
        xclassinterface.inter.initredshift_range = self.XCLASSDict['Redshift_Range']
        xclassinterface.inter.initbackgroundtemperaturerange = self.XCLASSDict['BackgroundTemperatureRange']
        xclassinterface.inter.inittemperaturesloperange = self.XCLASSDict['TemperatureSlopeRange']
        xclassinterface.inter.inithydrogencolumndensityrange = self.XCLASSDict['HydrogenColumnDensityRange']
        xclassinterface.inter.initdustbetarange = self.XCLASSDict['DustBetaRange']
        xclassinterface.inter.initkapparange = self.XCLASSDict['KappaRange']
        xclassinterface.inter.initdusttaufromfile = self.XCLASSDict['DustTauFromFile']
        xclassinterface.inter.initbackgroundfromfile = self.XCLASSDict['BackgroundFromFile']
        xclassinterface.inter.initsorteddistances = self.XCLASSDict['SortedDistances']
        xclassinterface.inter.initpuredistances = self.XCLASSDict['PureDistances']
        xclassinterface.inter.initcontphen_range = self.XCLASSDict['ContPhen_Range']
        xclassinterface.inter.initmyxclassparameter = self.XCLASSDict['myXCLASSParameter']
        xclassinterface.inter.initvwidthlimits = self.XCLASSDict['vWidthLimits']
        xclassinterface.inter.initvofflimits = self.XCLASSDict['vOffLimits']
        xclassinterface.inter.initobservationaldatalist = self.XCLASSDict['ObservationalDataList']
        xclassinterface.inter.initgausssianbeammap = self.XCLASSDict['GausssianBeamMap']
        xclassinterface.inter.initemsabsfunc = self.XCLASSDict['EmsAbsFunc']
        xclassinterface.inter.initemsabspath = self.XCLASSDict['EmsAbsPATH']
        xclassinterface.inter.initmoleculenames = self.XCLASSDict['MoleculeNames']
        xclassinterface.inter.initisoflag = self.XCLASSDict['IsoFlag']
        xclassinterface.inter.initsubbeamdescriptionflag = self.XCLASSDict['SubBeamDescriptionFlag']
        xclassinterface.inter.inituseemabsfuncflag = self.XCLASSDict['UseEmAbsFuncFlag']
        xclassinterface.inter.inittbflag = self.XCLASSDict['tbFlag']
        xclassinterface.inter.initinterflag = self.XCLASSDict['InterFlag']
        xclassinterface.inter.inittdflag = self.XCLASSDict['tdFlag']
        xclassinterface.inter.initnhflag = self.XCLASSDict['nHFlag']
        xclassinterface.inter.initlteflag = self.XCLASSDict['LTEFlag']
        xclassinterface.inter.initthinflag = self.XCLASSDict['ThinFlag']
        xclassinterface.inter.initdust_flag_range = self.XCLASSDict['Dust_Flag_Range']
        xclassinterface.inter.initphen_flag_range = self.XCLASSDict['Phen_Flag_Range']
        xclassinterface.inter.initconfiglist = self.XCLASSDict['ConfigList']
        xclassinterface.inter.initlayermap = self.XCLASSDict['LayerMap']


        ## variables for sqlite3 database
        xclassinterface.inter.initminnumtransitionssql = self.XCLASSDict['MinNumTransitionsSQL']
        xclassinterface.inter.initmaxnumtransitionssql = self.XCLASSDict['MaxNumTransitionsSQL']
        xclassinterface.inter.initordertranssql = self.XCLASSDict['OrderTransSQL']
        xclassinterface.inter.initnumentriesranges = self.XCLASSDict['NumEntriesRanges']
        xclassinterface.inter.initmaxelowsql = self.XCLASSDict['MaxElowSQL']
        xclassinterface.inter.initmingasql = self.XCLASSDict['MingASQL']
        xclassinterface.inter.initnameofpartfunctable = self.XCLASSDict['NameOfPartFuncTable']
        xclassinterface.inter.initnameofradtranstable = self.XCLASSDict['NameOfRadTransTable']
        xclassinterface.inter.initdbname = self.XCLASSDict['dbName']


        ## variables for partition function
        xclassinterface.inter.initnumberoftemperatures = self.XCLASSDict['NumberOfTemperatures']
        xclassinterface.inter.initnumbermoleculepartfunc = self.XCLASSDict['NumberMoleculePartFunc']
        xclassinterface.inter.initfirsti = self.XCLASSDict['Firsti']
        xclassinterface.inter.initlasti = self.XCLASSDict['Lasti']
        xclassinterface.inter.initstepi = self.XCLASSDict['stepi']
        xclassinterface.inter.inittemplow = self.XCLASSDict['TempLow']
        xclassinterface.inter.inittemphigh = self.XCLASSDict['TempHigh']
        xclassinterface.inter.inittemppartfunc = self.XCLASSDict['TempPartFunc']
        xclassinterface.inter.initlgq = self.XCLASSDict['lgQ']
        xclassinterface.inter.initcolumnnamefornamepartfunc = self.XCLASSDict['ColumnNameForNamePartFunc']
        # xclassinterface.inter.initmolnamepartfunc = self.XCLASSDict['MolNamePartFunc']
        # xclassinterface.inter.initcolumnnamespartfunc = self.XCLASSDict['ColumnNamesPartFunc']


        ## variables for rad-trans parameter
        xclassinterface.inter.inittotalnumberoftransitions = self.XCLASSDict['TotalNumberOfTransitions']
        xclassinterface.inter.inittotalnumtransfreq = self.XCLASSDict['TotalNumTransFreq']
        xclassinterface.inter.initcolumn300kpartfunc = self.XCLASSDict['Column300KPartFunc']
        xclassinterface.inter.initdopplershiftedtransfreqindex = self.XCLASSDict['DopplerShiftedTransFreqIndex']
        xclassinterface.inter.initmoleculardataindices = self.XCLASSDict['MolecularDataIndices']
        xclassinterface.inter.initmoleculardata = self.XCLASSDict['MolecularData']
        xclassinterface.inter.initdopplershiftedtransfreq = self.XCLASSDict['DopplerShiftedTransFreq']
        xclassinterface.inter.initcentralopticaldepth = self.XCLASSDict['CentralOpticalDepth']
        xclassinterface.inter.initcolumnnamefornametransitions = self.XCLASSDict['ColumnNameForNameTransitions']
        xclassinterface.inter.initcolumnnameforfreqtransitions = self.XCLASSDict['ColumnNameForFreqTransitions']
        xclassinterface.inter.initcolumnnameforinttransitions = self.XCLASSDict['ColumnNameForIntTransitions']
        xclassinterface.inter.initcolumnnameforeinsteinatransitions = self.XCLASSDict['ColumnNameForEinsteinATransitions']
        xclassinterface.inter.initcolumnnameforfreqerrtransitions = self.XCLASSDict['ColumnNameForFreqErrTransitions']
        xclassinterface.inter.initcolumnnameforelowtransitions = self.XCLASSDict['ColumnNameForELowTransitions']
        xclassinterface.inter.initcolumnnameforguptransitions = self.XCLASSDict['ColumnNameForgUpTransitions']
        xclassinterface.inter.initcolumnnameforlowerqntransitions = self.XCLASSDict['ColumnNameForLowerQNTransitions']
        xclassinterface.inter.initcolumnnameforupperqntransitions = self.XCLASSDict['ColumnNameForUpperQNTransitions']
        # xclassinterface.inter.initlowerqnstring = self.XCLASSDict['LowerQNString']
        # xclassinterface.inter.initupperqnstring = self.XCLASSDict['UpperQNString']


        ## variables for iso table file
        xclassinterface.inter.initnumberofisomasters = self.XCLASSDict['NumberOfIsomasters']
        xclassinterface.inter.initnumberofglobalisoratios = self.XCLASSDict['NumberOfGlobalISORatios']
        xclassinterface.inter.initisonfitconversiontable = self.XCLASSDict['IsoNfitConversionTable']
        xclassinterface.inter.initglobalisoratioparameter = self.XCLASSDict['GlobalIsoRatioParameter']
        xclassinterface.inter.initisoratio = self.XCLASSDict['IsoRatio']
        xclassinterface.inter.initisoratioconversiontable = self.XCLASSDict['IsoRatioConversionTable']
        xclassinterface.inter.initisotablefilename = self.XCLASSDict['IsoTableFileName']
        if (self.XCLASSDict['IsoMolecule'] is not None):
            xclassinterface.inter.initisomolecule = self.XCLASSDict['IsoMolecule']
            xclassinterface.inter.initisomasters = self.XCLASSDict['IsoMasters']


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## (normal fitting only): update molfit file with parameter maps
    ##
    def UpdateMolfitFile(self, LocalMolfitFileName, ypos, xpos, ExportFlag = False):
        """

    input parameters:
    -----------------

        - LocalMolfitFileName:  path and name of local pixel molfit file

        - ypos:                 y-coordinate index

        - xpos:                 x-coordinate index

        - ExportFlag:           (optional) flag indicating export of fitted parameters to molfit file (default: False)


    output parameters:
    ------------------

        - FitPixelFlag:         fit current pixel
        """

        # Debug:
        # print ("LocalMolfitFileName = ", LocalMolfitFileName)
        # print ("ypos = ", ypos)
        # print ("xpos = ", xpos)
        # print ("ExportFlag = ", ExportFlag)


        ## initial return parameter
        FitPixelFlag = True


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## convert list of maps
        ## self.ListOfInitialMaps = [NameParam, MoleculeName, SecondMolecue, CurrCompMap, LocalImage]
        LocalParameterList = []
        if (self.LocalParamMapIter == (-1) and (not ExportFlag)):
            for LocalKey in self.ListOfInitialMaps:
                SplittedLocalKey = LocalKey.split("|")
                if (SplittedLocalKey[0] in ["molfit"]):
                    MoleculeName = SplittedLocalKey[1]
                    NumberComp = int(SplittedLocalKey[2])
                    NameParam = SplittedLocalKey[3]
                    LocalMap = self.ListOfInitialMaps[LocalKey]
                    LocalElement = LocalMap[ypos, xpos].value
                    LocalParameterList.append([NameParam, MoleculeName, NumberComp, LocalElement])

                    # Debug:
                    # print ("\nNameParam = ", NameParam)
                    # print ("MoleculeName = ", MoleculeName)
                    # print ("NumberComp = ", NumberComp)
                    # print ("LocalElement = {:.3e}".format(LocalElement))
                    # print ("ypos, xpos = ", ypos, xpos)


        ## self.MolfitFitParameters.append([CompCounter, ParameterCounter, LogFlag, LowerLimit, UpperLimit, value, Name])
        ## self.FitParameterNames[i] = [LocalMolecule, LocalLineID, Name, LogFlag]
        else:
            for FitParameterID, LocalList in enumerate(self.FitParameterNames):             ## loop over all parameter maps
                LocalMolecule = LocalList[0]
                LocalMolecule = task_LineIdentification.MoleculeFileName(LocalMolecule)
                try:
                    LocalLineID = int(LocalList[1])
                except:
                    LocalLineID = None
                if (LocalLineID is not None):
                    if (LocalLineID > (-1)):
                        NameParam = LocalList[2]
                        LogFlag = LocalList[3]
                        LocalElement = self.FitParameterMaps[ypos, xpos, FitParameterID]    ## <- apply new fit parameter for current position
                        if (LogFlag):
                            if (ExportFlag):
                                LocalElement = 10.0**(LocalElement)
                            else:
                                LocalElement = numpy.log10(LocalElement)
                        LocalParameterList.append([NameParam, LocalMolecule, LocalLineID, LocalElement])

                        # Debug:
                        # print ("\nNameParam = ", NameParam)
                        # print ("LocalMolecule = ", LocalMolecule)
                        # print ("LogFlag = ", LogFlag)
                        # print ("LocalLineID = ", LocalLineID)
                        # print ("LocalElement = {:.3e}".format(LocalElement))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## open new molfit file
        CurrentMolfitFile = open(LocalMolfitFileName, 'w')
        for line in self.MolfitHeaderLinesList:
            CurrentMolfitFile.write(line)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update molfit file parameters using parameter
        CompCounter = 0
        CountMolecules = 0
        for MoleculeIndex, LocalMolfitFileContent in enumerate(self.AllMolfitFileForEachMolecule):   ## loop over all molecules in the given molfit file
            LocalMoleculeNameOrig = self.AllMoleculesInMolfitFile[MoleculeIndex]            ## get name of current molecule
            LocalMoleculeName = task_LineIdentification.MoleculeFileName(LocalMoleculeNameOrig)
            LocalMoleculeName = LocalMoleculeName.strip("_")
            LocalMolfitParameters = self.AllParameters[MoleculeIndex]
            NewMoleculeContent = []
            for line in LocalMolfitFileContent:                                             ## loop over all lines corresponding to current molecule


                ## remove comments from current molfit line
                i = line.find("%")
                if (i > (-1)):
                    StrippedLine = line[:i].strip()
                else:
                    StrippedLine = line.strip()
                SplittedLine = StrippedLine.split()


                ## current line describes molecule and number of components
                if (len(SplittedLine) == 2):
                    CompCounter = 0

                    # Debug:
                    # print ("line = ", line)


                ## current line describes component
                elif (len(SplittedLine) > 2):
                    AddFlag = False
                    CompCounter += 1
                    NewLine = ""
                    LocalLine = LocalMolfitParameters[CompCounter - 1]

                    # Debug:
                    # print ("LocalLine = ", LocalLine)


                    ## analyze and update current component
                    LocalvOff = 0.0
                    FitFlag = False
                    NewComp = " "
                    LocalIncludeFlag = True
                    for col in LocalLine:
                        ParamName = col[0]
                        flag = col[1]
                        lowlimit = min(float(col[2]), float(col[3]))
                        uplimit = max(float(col[2]), float(col[3]))
                        element = col[4]
                        AddFlag = False
                        if (not LocalIncludeFlag):
                            break
                        else:

                            # Debug:
                            # print ("\ncol = ", col)
                            # print ("element = ", element)
                            # print ("lowlimit = ", lowlimit)
                            # print ("uplimit = ", uplimit)
                            # print ("ParamName = ", ParamName)
                            # print ("LocalMoleculeName = ", LocalMoleculeName)
                            # print ("CompCounter = ", CompCounter)


                            ## get "_flag" flag
                            if (ParamName.endswith("_flag")):
                                FitFlag = element.strip().lower()
                                AddFlag = False


                            ## get "_FitFlag" flag
                            if (ParamName.endswith("_FitFlag")):
                                FitFlag = float(element)
                                if (abs(FitFlag) > 0.0):
                                    FitFlag = "y"
                                else:
                                    FitFlag = "n"
                                AddFlag = False


                            ## ignore limits
                            elif (ParamName.endswith("_lowlimit") or ParamName.endswith("_uplimit")):
                                AddFlag = False


                            ## store values for "source_size", "source_radius", "source_center_x", "source_center_y"
                            elif (ParamName in ["s_size", "source_size", "source_radius", "source_center_x", "s_off_x", \
                                                "source_center_y", "s_off_x"]):
                                value = float(element)
                                value, lowlimit, uplimit = self.UpdateParameter(value, lowlimit, uplimit, ParamName, LocalMoleculeName, \
                                                                                CompCounter, LocalParameterList)
                                AddFlag = True


                            ## store values for "T_rot", "T_e", "T_cont_dust", "T_dOff", "T_dSlope"
                            elif (ParamName in ["T_rot", "T_e", "T_cont_dust", "T_dOff", "T_dSlope"]):
                                value = float(element)
                                value, lowlimit, uplimit = self.UpdateParameter(value, lowlimit, uplimit, ParamName, LocalMoleculeName, \
                                                                                CompCounter, LocalParameterList)
                                AddFlag = True


                            ## store values for "N_tot", "EM_RRL", "nHcolumn_cont_dust", "nHcolumn"
                            elif (ParamName in ["N_tot", "EM_RRL", "nHcolumn_cont_dust", "nHcolumn"]):
                                value = float(element)
                                value, lowlimit, uplimit = self.UpdateParameter(value, lowlimit, uplimit, ParamName, LocalMoleculeName, \
                                                                                CompCounter, LocalParameterList)
                                AddFlag = False
                                if (numpy.isnan(value)):
                                    LocalIncludeFlag = False
                                else:
                                    NewComp += "    {:1s}  {:10.2e}  {:10.2e}   {:15.5e}".format(FitFlag, lowlimit, uplimit, value)


                            ## store values for "V_width_Gauss", "V_width"
                            elif (ParamName in ["V_width_Gauss", "V_width"]):
                                value = float(element)
                                value, lowlimit, uplimit = self.UpdateParameter(value, lowlimit, uplimit, ParamName, LocalMoleculeName, \
                                                                                CompCounter, LocalParameterList)
                                AddFlag = True


                            ## store values for "V_off"
                            elif (ParamName == "V_off"):
                                value = float(element)
                                value, lowlimit, uplimit = self.UpdateParameter(value, lowlimit, uplimit, ParamName, LocalMoleculeName, \
                                                                                CompCounter, LocalParameterList)
                                AddFlag = True


                            ## store values for "beta_cont_dust", "kappa_cont_dust", "kappa", "beta"
                            elif (ParamName in ["beta_cont_dust", "kappa_cont_dust", "kappa", "beta"]):
                                value = float(element)
                                value, lowlimit, uplimit = self.UpdateParameter(value, lowlimit, uplimit, ParamName, LocalMoleculeName, \
                                                                                CompCounter, LocalParameterList)
                                AddFlag = True


                            ## store values for "ContFuncID_param_1", "ContFuncID_param_2", "ContFuncID_param_3",
                            ##                  "ContFuncID_param_4", "ContFuncID_param_5"
                            elif (ParamName in ["ContFuncID_param_1", "ContPhenFuncParam1", "ContFuncID_param_2", "ContPhenFuncParam2", \
                                                "ContFuncID_param_3", "ContPhenFuncParam3", "ContFuncID_param_4", "ContPhenFuncParam4", \
                                                "ContFuncID_param_5", "ContPhenFuncParam5"]):
                                value = float(element)
                                value, lowlimit, uplimit = self.UpdateParameter(value, lowlimit, uplimit, ParamName, LocalMoleculeName, \
                                                                                CompCounter, LocalParameterList)
                                AddFlag = True


                            ## store values for EA-Flag
                            elif (ParamName in ["ContFuncID_phen", "LayerDistance", "CFFlag", "keyword"]):
                                AddFlag = False
                                NewComp += "    " + element


                            ## create new line
                            if (AddFlag and (lowlimit is not None) and (uplimit is not None) and (value is not None)):
                                NewComp += "    {:s}  {:10.2f}  {:10.2f}   {:15.5f}".format(FitFlag, lowlimit, uplimit, value)


                    ## write new lines to file
                    if (LocalIncludeFlag):
                        NewMoleculeContent.append(NewComp + "\n")

                    # Debug:
                    # print ("NewComp = ", NewComp)


            ## write local molecule content to molfit file
            if (NewMoleculeContent != []):
                CountMolecules += 1
                NewLine = LocalMoleculeNameOrig + "  " + str(len(NewMoleculeContent)) + "\n"
                CurrentMolfitFile.write(NewLine)
                for NewLine in NewMoleculeContent:
                    CurrentMolfitFile.write(NewLine)


        ## close molfit file(s)
        CurrentMolfitFile.close()


        ## check if molfit file is not empty
        if (CountMolecules == 0):
            FitPixelFlag = False


        ## we're done
        if (ExportFlag):
            return
        else:
            return FitPixelFlag
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## (normal fitting only): update iso ratio file
    ##
    def UpdateIsoRatioFile(self, IsoTableFileName, ypos, xpos, ExportFlag = False):
        """

    input parameters:
    -----------------

        - IsoTableFileName:     path and name of local iso ratio file

        - ypos:                 y-coordinate index

        - xpos:                 x-coordinate index

        - ExportFlag:           (optional) flag indicating export of fitted parameters to molfit file (default: False)


    output parameters:
    ------------------

        - FitPixelFlag:         fit current pixel
        """

        # Debug:
        # print ("IsoTableFileName = ", IsoTableFileName)
        # print ("ypos = ", ypos)
        # print ("xpos = ", xpos)
        # print ("ExportFlag = ", ExportFlag)


        ## write new iso ratio file
        LocalIsoRatioFile = open(IsoTableFileName, 'w')
        NewLine = "{:<45s} {:<45s} {:>25s} {:>25s} {:>25s}\n".format("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
        LocalIsoRatioFile.write(NewLine)


        ## add new values of iso fit parameters to array describing molfit parameters
        LocalGlobalIsoParam = copy.deepcopy(self.GlobalIsoParam)
        LocalIsoParametersList = copy.deepcopy(self.IsoParameters)


        ## update global and normal iso ratio definitions from initial map
        ## HERE:    self.ListOfInitialMaps[globalisoratio + "|" + IsoIndex] or self.ListOfInitialMaps[isoratio + "|" + IsoIndex]
        ## self.GlobalIsoParam[i] = [LocalIsotopologue, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag]
        ## self.IsoParameters[i] = [LocalIsotopologue, IsoMaster, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag, ListIndex]
        if (self.LocalParamMapIter == (-1) and (not ExportFlag)):
            if (len(self.ListOfInitialMaps) > 0):
                for LocalKey in self.ListOfInitialMaps:
                    SplittedLocalKey = LocalKey.split("|")


                    ## globally defined iso ratios
                    if (SplittedLocalKey[0] in ["globalisoratio"]):
                        IsoIndex = int(SplittedLocalKey[1])
                        LocalMap = self.ListOfInitialMaps[LocalKey]
                        LocalGlobalIsoParam[IsoIndex][1] = LocalMap[ypos, xpos]


                    ## normally defined iso ratios
                    elif (SplittedLocalKey[0] in ["isoratio"]):
                        IsoIndex = int(SplittedLocalKey[1])
                        LocalMap = self.ListOfInitialMaps[LocalKey]
                        LocalIsoParametersList[IsoIndex][2] = LocalMap[ypos, xpos]

                    # Debug:
                    # print ("ypos, xpos, LocalMap = ", ypos, xpos, LocalMap[ypos, xpos])


        ## update fit parameters
        ##
        ## for global iso ratio definitions:
        ##      self.IsoFitParameters[i] = [(-1) * (len(self.GlobalIsoParam) + 1), LowerLimitIsoRatio, UpperLimitIsoRatio, RatioValue]
        ##
        ## for normal iso definitions:
        ##      self.IsoFitParameters[i] = [IsoIndex,                              LowerLimitIsoRatio, UpperLimitIsoRatio, RatioValue]
        ##
        ## self.IsoParameters[i] = [LocalIsotopologue, IsoMaster, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag, ListIndex]
        else:


            ## update fit parameters
            p0IndexOffset = len(self.MolfitFitParameters)
            for IsoFitParameterID, IsoFitParameter in enumerate(self.IsoFitParameters):
                IsoIndex = IsoFitParameter[0]
                LowerLimitIsoRatio = IsoFitParameter[1]
                UpperLimitIsoRatio = IsoFitParameter[2]
                RatioValue = IsoFitParameter[3]


                ## get new value and check if parameter is within limits
                ## self.GlobalIsoParam[i] = [LocalIsotopologue, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag]
                FitParameterID = p0IndexOffset + IsoFitParameterID
                if (IsoIndex >= 0):
                    LocalIsoParametersList[IsoIndex][2] = self.FitParameterMaps[ypos, xpos, FitParameterID]  ## <- set new fit param. for current position
                else:
                    IsoIndex = abs(IsoIndex) - 1
                    LocalGlobalIsoParam[IsoIndex][1] = self.FitParameterMaps[ypos, xpos, FitParameterID]     ## <- set new fit param. for current position


        ## create array describing iso parameters
        ## self.IsoParameters[i] = [LocalIsotopologue, IsoMaster, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag, ListIndex]
        ## with
        ## ListIndex[i] = [GlobalIndex, Multiplicator]
        ## and
        ## GlobalIsoParam = [LocalIsotopologue, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag]
        for line in LocalGlobalIsoParam:
            LocalIsotopologue = line[0]
            IsoMaster = LocalIsotopologue.replace("GLOBAL", "")
            IsoMaster = IsoMaster.replace("global", "")
            IsoMaster = IsoMaster.replace("Global", "")
            IsoMaster = IsoMaster.replace("___", "")
            IsoMaster = IsoMaster.replace("__", "")
            IsoMaster = IsoMaster.strip("_")
            RatioValue = line[1]
            LowerLimitIsoRatio = line[2]
            UpperLimitIsoRatio = line[3]
            NewLine = "{:<45s} {:<45s} {:25.2f} {:25.2f} {:25.2f}".format(LocalIsotopologue, IsoMaster, RatioValue, \
                                                                          LowerLimitIsoRatio, UpperLimitIsoRatio)
            LocalIsoRatioFile.write(NewLine + "\n")
        for line in LocalIsoParametersList:
            LocalIsotopologue = line[0]
            IsoMaster = line[1]
            RatioValue = line[2]
            LowerLimitIsoRatio = line[3]
            UpperLimitIsoRatio = line[4]
            NewLine = "{:<45s} {:<45s} {:25.2f} {:25.2f} {:25.2f}".format(LocalIsotopologue, IsoMaster, RatioValue, \
                                                                          LowerLimitIsoRatio, UpperLimitIsoRatio)

            ## check, if path and name of file describing rates has to be added as well?
            for IsoRatioTableLine in self.IsoRatioTable:
                LocalTestIsotopologue = IsoRatioTableLine[0]
                if (LocalIsotopologue == LocalTestIsotopologue and len(IsoRatioTableLine) == 6):
                    NewLine += "   {:s}".format(IsoRatioTableLine[5].strip())
            LocalIsoRatioFile.write(NewLine + "\n")
        LocalIsoRatioFile.close()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## (normal fitting only): update parameter value
    ##
    def UpdateParameter(self, ParameterValue, ParameterLowLimit, ParameterUpLimit, ParameterName, CurrMoleculeName, CompCounter, LocalParameterList):
        """

    input parameters:
    -----------------

        - ParameterValue:       current parameter value

        - ParameterLowLimit:    lower limit for current parameter

        - ParameterUpLimit:     upper limit for current parameter

        - ParameterName:        name of current parameter

        - CurrMoleculeName:     name of current molecule

        - CompCounter:          current component ID

        - LocalParameterList:   list of new parameter values


    output parameters:
    ------------------

        - NewParameterValue:    new parameter value

        - NewParameterLowLimit: new lower limit

        - NewParameterUpLimit:  new upper limit

        """

        # Debug:
        # print ("\n\nParameterValue = ", ParameterValue)
        # print ("ParameterLowLimit = ", ParameterLowLimit)
        # print ("ParameterUpLimit = ", ParameterUpLimit)
        # print ("ParameterName = ", ParameterName)
        # print ("CurrMoleculeName = ", CurrMoleculeName)
        # print ("CompCounter = ", CompCounter)
        # print ("LocalParameterList = ", LocalParameterList)


        ## initialize return parameters
        NewParameterValue = ParameterValue
        NewParameterLowLimit = ParameterLowLimit
        NewParameterUpLimit = ParameterUpLimit


        ## convert input parameter to lower case
        ParameterName = ParameterName.lower()
        CurrMoleculeName = CurrMoleculeName.lower().strip("_")

        # Debug:
        # print ("\n\n\nParameterName = ", ParameterName)
        # print ("CurrMoleculeName = ", CurrMoleculeName)
        # print ("CompCounter = ", CompCounter)


        ## check, if current paramter has to be fitted
        for LocalParm in LocalParameterList:                                                ## loop over all parameter maps
            NameParam = LocalParm[0].lower()                                                ## get name of parameter
            MoleculeName = LocalParm[1].lower().strip("_")                                  ## get name of molecule
            NumberComp = int(LocalParm[2])                                                  ## get component ID
            NewParameterMapValue = LocalParm[3]                                             ## get new parameter value

            # Debug:
            # print ("\n\nNameParam, ParameterName = ", NameParam, ParameterName, NameParam == ParameterName)
            # print ("MoleculeName, CurrMoleculeName = ", MoleculeName, CurrMoleculeName, MoleculeName == CurrMoleculeName)
            # print ("NumberComp, (CompCounter - 1) = ", NumberComp, (CompCounter - 1), NumberComp == (CompCounter - 1))
            # print ("NewParameterMapValue = ", NewParameterMapValue)


            ## do we have the correct entry?
            if (NameParam == ParameterName and MoleculeName == CurrMoleculeName and NumberComp == (CompCounter - 1) and NewParameterMapValue is not None):


                ## check, if parameter value is None or NaN or inf
                if (numpy.isinf(NewParameterMapValue)):
                    NewParameterMapValue = numpy.nan


                ## if new parameter value is not None, define new lower and upper limits
                if (not numpy.isnan(NewParameterMapValue)):
                    NewParameterValue = NewParameterMapValue
                    if (NewParameterMapValue <= ParameterLowLimit):
                        NewParameterLowLimit = NewParameterMapValue - abs(NewParameterMapValue) * 0.1
                    if (NewParameterMapValue >= ParameterUpLimit):
                        NewParameterUpLimit = NewParameterMapValue + abs(NewParameterMapValue) * 0.1
                else:


                    ## if current parameter does not describe a column density (or something like that) restore initial value if new value is None
                    if (ParameterName not in ["n_tot", "em_rrl", "nhcolumn_cont_dust", "nhcolumn"]):
                        NewParameterValue = ParameterValue
                    else:
                        NewParameterValue = NewParameterMapValue

                # Debug:
                # print ("\n\nNameParam = ", NameParam)
                # print ("MoleculeName = ", MoleculeName)
                # print ("NumberComp = ", NumberComp)
                # print ("\t\tNewParameterValue = ", NewParameterValue)


                ## we're found a new value and exit loop
                break

        # Debug:
        # print ("\n\nNewParameterValue = ", NewParameterValue)
        # print ("NewParameterLowLimit = ", NewParameterLowLimit)
        # print ("NewParameterUpLimit = ", NewParameterUpLimit)


        ## define return variable
        return (NewParameterValue, NewParameterLowLimit, NewParameterUpLimit)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (fast-fitting only): start fast fitting
    def StartFastFits(self, IndexRange):
        """

    input parameters:
    -----------------

        - IndexRange:                   list of indices


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("IndexRange = ", IndexRange)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## In order to apply the "apply_function_parallel_spectral" in combination with dask, it is necessary
        ## to create one big cube with all obs. data cubes and parameters included.
        ## The first points along the spectral axis describes the diffferent obs. data cubes
        ## thereafter the spectral axis describes the fit parameters (number stored in self.NumFitParameters)
        ## then one spectral channel will describe the chi^2 values for all fitted pixels
        ## and finally the last spectral channel describes an index parameter, which is used to determine
        ## the x- and y-position of each pixel


        ##************************************************************************************************************************************************
        ## determine total number of spectral channels


        ## first, get total number of all spectral channels of the obs. data cubes
        TotalNumSpecChannels = 0
        for LocalRangeID in range(len(self.RangeParam)):
            TotalNumSpecChannels += len(self.ListOfSpectralAxis[LocalRangeID].value)
        self.TotalNumObsDataSpecChannels = TotalNumSpecChannels


        ## now, add number of fit paramters plus one channel for chi^2 values, plus one channel for indexing
        TotalNumSpecChannels += self.NumFitParameters + 2

        # Debug:
        # print ("TotalNumSpecChannels = ", TotalNumSpecChannels)


        ##************************************************************************************************************************************************
        ## (only for parallelization method: "dask" and "sc": Initialize big cube and fill cube
        TempFileName = None
        if (self.ParallelMethod in ["dask", "sc"]):


            ## print what you do
            print ("\nInitialize numpy array for spectral-cube's {:s} .. ".format(chr(34) + "apply_function_parallel_spectral" + chr(34)), end = "")


            ## initialize numpy array
            BigCube = numpy.ones((TotalNumSpecChannels, self.NumYPixels, self.NumXPixels)) * numpy.nan


            ## add obs. data
            self.BigCubeIndexList = []
            LastIndex = 0
            NumSpecChannels = (-1)
            for LocalRangeID in range(len(self.RangeParam)):
                NumSpecChannels += len(self.ListOfSpectralAxis[LocalRangeID].value)
                self.BigCubeIndexList.append([LastIndex, (NumSpecChannels + 1)])

                # Debug:
                # print ("\nLocalRangeID = ", LocalRangeID)
                # print ("len(self.ListOfSpectralAxis[LocalRangeID].value) = ", len(self.ListOfSpectralAxis[LocalRangeID].value))
                # print ("NumSpecChannels = ", NumSpecChannels)
                # print ("LastIndex = ", LastIndex)
                # print ("self.ListOfSpectralAxis[LocalRangeID].value = ", self.ListOfSpectralAxis[LocalRangeID].value)

                BigCube[LastIndex:(NumSpecChannels + 1), :, :] = self.ListOfCubes[LocalRangeID][:, :, :]

                # Debug:
                # BigCube[LastIndex:(NumSpecChannels + 1), 0, 0] = self.ListOfSpectralAxis[LocalRangeID].value
                # print ("\n\nBigCube[:, 0, 0] = ", BigCube[:, 0, 0])

                LastIndex = NumSpecChannels + 1


            ## add fit parameters
            for i in range(self.NumFitParameters):
                BigCube[LastIndex + i, :, :] = self.FitParameterMaps[:, :, i]
            NumSpecChannels += self.NumFitParameters
            self.BigCubeIndexList.append([LastIndex, (NumSpecChannels + 1)])

            # Debug:
            # print ("\nself.NumFitParameters = ", self.NumFitParameters)
            # print ("\n\nBigCube[:, 0, 0] = ", BigCube[:, 0, 0])


            ## add index parameter (leave one channel for chi^2 values)
            NumSpecChannels += 1
            self.BigCubeIndexList.append([NumSpecChannels, NumSpecChannels + 1])
            NumSpecChannels += 1
            self.BigCubeIndexList.append([NumSpecChannels, NumSpecChannels + 1])
            BigCube[NumSpecChannels, :, :] = IndexRange.reshape(self.NumYPixels, self.NumXPixels)

            # Debug:
            # print ("\n\nBigCube[:, 0, 0] = ", BigCube[:, 0, 0])
            # print ("self.BigCubeIndexList = ", self.BigCubeIndexList)


            ##************************************************************************************************************************************************
            ## Initialize spectral-cube streaming object


            ## get header and wcs from first obs. data cube
            BigDataCubeHeader = self.ListOfCubes[0].header
            BigDataCubeHeaderCopy = BigDataCubeHeader.copy()
            BigDataCubeWCS = self.ListOfCubes[0].wcs
            BigDataCubeWCSCopy = BigDataCubeWCS.copy()


            ## modify header to take new spectral axis into account (the spectral axis of the big cube is not used anywhere)
            ## modify FITS header
            # BigDataCubeHeaderCopy["NAXIS3"] = 92


            ## creating a SpectralCube instance
            self.BigCubeObject = SpectralCube(data = BigCube, wcs = BigDataCubeWCSCopy, header = BigDataCubeHeaderCopy, use_dask = self.daskFlag)
            self.BigCubeObject = self.BigCubeObject * u.K


            ## the last two lines seems to be necessary to avoid a spectral cube error
            TempFileName = self.myXCLASSMapFitJobDir + "temp.fits"
            self.BigCubeObject.write(TempFileName, overwrite = True)
            if (self.ParallelMethod in ["dask"]):
                self.BigCubeObject = SpectralCube.read(TempFileName, use_dask = True)
            else:
                self.BigCubeObject = SpectralCube.read(TempFileName, use_dask = self.daskFlag)


            ## we're done
            print ("done!")


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## apply algorithms to cube(s)
        for LocalAlgDict in self.FitAlgSettings:                                             ## start loop over algorithms in the alg. xml file


            ## get parametes of current fit algorithm
            self.Fit_AlgDict = LocalAlgDict
            self.Fit_Algorithm = LocalAlgDict["FitAlgorithm"]
            self.Fit_limit = LocalAlgDict["Chi2Limit"]
            self.Fit_NumIter = LocalAlgDict["NumberIteration"]
            if (self.Fit_Algorithm in ["blank"]):
                self.Fit_NumIter = 1
            self.Fit_NumProc = LocalAlgDict["NumberProcessors"]
            self.Fit_Host = LocalAlgDict["MPIHostFileName"]
            self.Fit_Chi2 = LocalAlgDict["Chi2StoreFlag"]
            self.Fit_Chi2LogFlag = LocalAlgDict["Chi2LogFlag"]


            ## compute gradient for the following algorithms
            if (self.Fit_Algorithm in ["trf", "dogbox", "lm", "levenberg-marquardt", "cg", "bfgs", "l-bfgs-b", "tnc", "slsqp"]):
                self.FitGradientFlag = True
                self.Fit_Variation = LocalAlgDict['VariationValue']
            else:
                self.FitGradientFlag = False
                self.Fit_Variation = None


            ## use "blank" algortihm, if no fit parameter is defined
            if (self.NumFitParameters == 0):
                self.Fit_Algorithm = "blank"
                self.FitGradientFlag = False


            ## define number of used processors
            if (self.NoFITSCubeFlag):
                if (self.FitGradientFlag):
                    self.Fit_NumProc = min(self.Fit_NumProc, self.TotalNumPixel * (self.NumFitParameters + 1))
            else:
                self.Fit_NumProc = min(self.Fit_NumProc, self.TotalNumPixel)
            self.Fit_AlgDict["NumberProcessors"] = self.Fit_NumProc


            ## set vectroization flag for following algorithms (only for non-MapFit applications)
            ## differential_evolution algorithm is available in parallel from version 1.9.0
            self.VectorizedFlag = False
            if (self.NoFITSCubeFlag):
                if (self.Fit_NumProc > 1 and self.Fit_Algorithm in ["brute", "bruteforce", "brute-force", \
                                                                    "mcmc", "emcee", "ultranest"]):
                    self.VectorizedFlag = True


            ## set flag describing usage of least-squares algorithm
            self.UseCallbackFuncFlag = False
            self.Fit_FuncEvalFlag = False
            if (self.Fit_Algorithm in ["trf", "dogbox", "lm", "levenberg-marquardt"]):
                self.Fit_LSFlag = 0
            elif (self.Fit_Algorithm in ["blank"]):
                self.Fit_LSFlag = 1
            else:
                self.Fit_LSFlag = 2

            # Debug:
            # print ("\nself.Fit_Algorithm = ", self.Fit_Algorithm)
            # print ("self.Fit_Variation = ", self.Fit_Variation)
            # print ("self.Fit_limit = ", self.Fit_limit)
            # print ("self.Fit_NumIter = ", self.Fit_NumIter)
            # print ("self.Fit_NumProc = ", self.Fit_NumProc)
            # print ("self.Fit_Host = ", self.Fit_Host)
            # print ("self.Fit_Chi2 = ", self.Fit_Chi2)
            # print ("self.Fit_LSFlag = ", self.Fit_LSFlag)
            # print ("self.ParallelMethod  = ", self.ParallelMethod)
            # print ("IndexRange = ", IndexRange)


            ## print name of algorithm only, if more than one fit algorithm was selected
            print ("\n\nUse {:s} algorithm to fit data ..\n\n".format(chr(34) + self.Fit_Algorithm + chr(34)))


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## non-MapFit fitting: write header of log files and call optimization algorithm
            self.CurrentFitIteration = 0
            self.BestFitResultsDict = {}
            if (self.NoFITSCubeFlag):


                ## open file for log entries and write some informations about algorithm settings to log file and screen
                self.LogFile = open(self.myXCLASSMapFitJobDir + "fit__{:s}.log".format(self.Fit_Algorithm), "w")
                if (self.Fit_Chi2LogFlag):
                    self.LogChi2File = open(self.myXCLASSMapFitJobDir + "fit__{:s}.log.chi2".format(self.Fit_Algorithm), "w")
                OutputLine = "\tAlgorithm settings:\n\n"
                OutputLine += "\t    Algorithm:                  {:s}\n".format(chr(34) + self.Fit_Algorithm + chr(34))
                if (self.Fit_Variation is not None and self.Fit_Algorithm in ["trf", "dogbox", "lm", "levenberg-marquardt"]):
                    OutputLine += "\t    Variation value:            {:.3e}\n".format(self.Fit_Variation)
                if (self.Fit_AlgDict["BruteSteps"] is not None and self.Fit_Algorithm in ["brute", "bruteforce", "brute-force"]):
                    OutputLine += "\t    brute-force steps:          {:s}\n".format(str(self.Fit_AlgDict["BruteSteps"]))
                if (self.Fit_AlgDict["NumberSampler"] is not None and self.Fit_Algorithm in ["shgo", "mcmc", "emcee", "ultranest", "bees"]):
                    OutputLine += "\t    Number of samples:          {:d}\n".format(self.Fit_AlgDict["NumberSampler"])
                if (self.Fit_AlgDict["NumberBurnInIter"] is not None and self.Fit_Algorithm in ["mcmc", "emcee", "ultranest"]):
                    OutputLine += "\t    Number of burn-in steps:    {:d}\n".format(self.Fit_AlgDict["NumberBurnInIter"])
                if (self.Fit_AlgDict["BackendFileName"] != "" and self.Fit_Algorithm in ["mcmc", "emcee", "ultranest"]):
                    OutputLine += "\t    Backend file name:          {:s}\n".format(chr(34) + self.Fit_AlgDict["BackendFileName"] + chr(34))
                if (self.Fit_AlgDict["Dictionary"] != "" and self.Fit_Algorithm in ["ultranest"]):
                    OutputLine += "\t    Dictionary:                 {:s}\n".format(chr(34) + self.Fit_AlgDict["Dictionary"] + chr(34))
                if (self.Fit_AlgDict["SampleMethod"] is not None and self.Fit_Algorithm in ["mcmc", "emcee"]):
                    OutputLine += "\t    Sample method:              {:s}\n".format(chr(34) + self.Fit_AlgDict["SampleMethod"] + chr(34))
                if (self.Fit_AlgDict["SigmaMultiplicity"] is not None and self.Fit_Algorithm in ["mcmc", "emcee", "ultranest"]):
                    OutputLine += "\t    Multiplicity of sigma:      {:.1f}\n".format(self.Fit_AlgDict["SigmaMultiplicity"])
                if (self.Fit_AlgDict["ErrorType"] is not None and self.Fit_Algorithm in ["mcmc", "emcee", "ultranest"]):
                    OutputLine += "\t    Type of error ranges:       {:s}\n".format(chr(34) + self.Fit_AlgDict["ErrorType"] + chr(34))
                if (self.Fit_AlgDict["PlotFileFormat"] is not None):
                    OutputLine += "\t    File format of plot-files:  {:s}\n".format(chr(34) + self.Fit_AlgDict["PlotFileFormat"] + chr(34))
                if (not self.Fit_Algorithm in ["blank"]):
                    OutputLine += "\t    Limit of chi^2:             {:.5e}\n".format(self.Fit_limit)
                    if (self.Fit_AlgDict["Chi2RenormalizedFlag"] is not None):
                        OutputLine += "\t    Use renormalized chi^2:     {:s}\n".format(chr(34) + str(self.Fit_AlgDict["Chi2RenormalizedFlag"]) + chr(34))
                OutputLine += "\t    Store chi^2 function:       {:s}\n".format(chr(34) + str(self.Fit_Chi2) + chr(34))
                OutputLine += "\t    Use a .chi2.log file:       {:s}\n".format(chr(34) + str(self.Fit_Chi2LogFlag) + chr(34))
                if (not self.Fit_Algorithm in ["blank"]):
                    OutputLine += "\t    Max. number of iterations:  {:d}\n".format(self.Fit_NumIter)
                OutputLine += "\t    Number of processors:       {:d}\n".format(self.Fit_NumProc)
                if (self.NoFITSCubeFlag):
                    OutputLine += "\t    Create final plot:          {:s}\n".format(chr(34) + str(self.Fit_AlgDict["PlotFlag"]) + chr(34))
                OutputLine += "\n"
                print (OutputLine)
                lt = time.localtime()
                OutputLine += "\n\tFitting starts at date: {:s},  time: {:s}\n\n\n".format(time.strftime("%Y-%m-%d", lt), \
                                                                                           time.strftime("%H:%M:%S", lt))
                self.LogFile.write(OutputLine.replace("\t", ""))


                ## print header line of iteration step
                if (self.Fit_Algorithm in ["levenberg-marquardt"]):
                    OutputLine = "{:>10s}".format("Iteration:")
                    OutputLine += "{:>19s}".format("chi^2:")
                    OutputLine += "{:>13s}".format("alamda:")
                    OutputLine += "     {:<s}".format("Parameter vector:")
                    print ("\t" + OutputLine)
                    self.LogFile.write(OutputLine + "\n")
                elif (not self.Fit_Algorithm in ["mcmc", "emcee", "ultranest"]):
                    OutputLine = "{:>10s}   {:>16s}     {:<s}".format("Iteration:", "chi^2:", "Parameter vector:")
                    print ("\t" + OutputLine)
                    self.LogFile.write(OutputLine + "\n")
                self.FuncCallCounter = 0
                if (self.Fit_Chi2LogFlag):
                    OutputLine = "{:>10s}   {:>16s}     {:<s}".format("Nr.:", "chi^2:", "Parameter vector:")
                    self.LogChi2File.write(OutputLine + "\n")


                ## define input dictionary for optimization algorithms
                OptAlgParamDict = {}
                OptAlgParamDict["args"] = (IndexRange,)
                OptAlgParamDict["verbose"] = 1
                OptAlgParamDict["GradientJAC"] = self.GradientJAC
                OptAlgParamDict["callback"] = self.CallbackOptAlg
                OptAlgParamDict["Fit_LSFlag"] = self.Fit_LSFlag


                ## call optimization algorithm
                p0 = self.FitParameterMaps[0, 0, :]
                self.EntireFITSCubeFit(self.NonMapFitWorkers, p0, OptAlgParamDict)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## MapFit: start pool of worker for myXCLASSMapFit function
            else:
                if (self.FastFitFlag and self.ParallelMethod in ["sc", "dask"]):
                    self.StartWorker(self.PixelFitFunction, IndexRange, method = self.ParallelMethod)
                elif (self.ParallelMethod in ["process"]):
                    self.StartWorker(self.PixelFitFunction, IndexRange, method = "process")
                elif (self.ParallelMethod in ["dbg", "debug"]):
                    self.StartWorker(self.PixelFitFunction, IndexRange, method = "dbg")
                else:
                    self.StartWorker(self.PixelFitFunction, IndexRange, method = "pool")


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## write results to job directory
            if (self.NumParamMapIterations > 1):
                self.ExportResults(self.LocalParamMapIter)
            else:
                self.ExportResults((-1))


            ## for non-MapFit fittings, close log file
            if (self.NoFITSCubeFlag):
                lt = time.localtime()
                OutputLine = "\n\nFitting ends at date: {:s},  time: {:s}\n".format(time.strftime("%Y-%m-%d", lt), time.strftime("%H:%M:%S", lt))
                self.LogFile.write(OutputLine)
                self.LogFile.close()
                if (self.Fit_Chi2LogFlag):
                    self.LogChi2File.close()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## remove temp file
        if (TempFileName is not None):
            cmdString = "rm -rf " + TempFileName
            os.system(cmdString)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## callback function for some optimization algorithms
    def CallbackOptAlg(self, *args, **kwargs):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## increase counter for iteration
        self.CurrentFitIteration += 1


        ## write line to log file and screen
        OutputLine = "{:10d}".format(self.CurrentFitIteration)
        OutputLine += "   {:16.8e}".format(self.BestFitResultsDict["BestChi2Value"])
        OutputLine += "   {:s}".format(self.BestFitResultsDict["BestParamVecString"])
        self.LogFile.write(OutputLine + "\n")
        print ("\t" + OutputLine)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## call selected optimization algorithms
    def CallOptimizationAlgorithms(self, LocalFitFunction, p0, OptAlgParamDict):
        """

    input parameters:
    -----------------

        - LocalFitFunction:             ojective function

        - p0:                           fit parameters

        - OptAlgParamDict:              dictionary with parameters for optimization algorithms


    output parameters:
    ------------------

        - OutDict:                      ouptut directory
        """

        # Debug:
        # print ("LocalFitFunction = ", LocalFitFunction)
        # print ("p0 = ", p0)
        # print ("len(p0) = ", len(p0))
        # print ("OptAlgParamDict = ", OptAlgParamDict)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get the tuple with additional parameters
        args = OptAlgParamDict["args"]
        AddParamDict = args[0]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Nonlinear least-squares


        ## use scipy's least_squares algorithm with method "trf" ("Trust Region Reflective" algorithm)
        OutDict = {}
        if (self.Fit_Algorithm in ["trf"]):
            jac = OptAlgParamDict["GradientJAC"]
            verbose = OptAlgParamDict["verbose"]
            res = least_squares(LocalFitFunction, p0, args = args, bounds = (self.pBLow, self.pBUp), method = 'trf', \
                                jac = jac, diff_step = self.Fit_Variation, max_nfev = self.Fit_NumIter, gtol = None, \
                                ftol = self.Fit_limit, verbose = verbose)                   # tr_solver = "lsmr",
            OutDict['chi2'] = res.cost                                                      ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters
            OutDict['residuals'] = res.fun                                                  ## get residual


        ## use scipy's least_squares algorithm with method "dogbox" ("dogleg" algorithm with rectangular trust regions)
        elif (self.Fit_Algorithm in ["dogbox"]):
            jac = OptAlgParamDict["GradientJAC"]
            verbose = OptAlgParamDict["verbose"]
            res = least_squares(LocalFitFunction, p0, args = args, bounds = (self.pBLow, self.pBUp), method = 'dogbox', \
                                jac = jac, diff_step = self.Fit_Variation, max_nfev = self.Fit_NumIter, \
                                ftol = self.Fit_limit, verbose = verbose)
            OutDict['chi2'] = res.cost                                                      ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters
            OutDict['residuals'] = res.fun                                                  ## get residual


        ## use scipy's least_squares algorithm with method "Levenberg-Marquardt" algorithm as implemented in MINPACK
        elif (self.Fit_Algorithm in ["lm"]):
            jac = OptAlgParamDict["GradientJAC"]
            verbose = OptAlgParamDict["verbose"]
            res = least_squares(LocalFitFunction, p0, args = args, jac = jac, method = 'lm', \
                                diff_step = self.Fit_Variation, max_nfev = self.Fit_NumIter, \
                                ftol = self.Fit_limit, verbose = verbose)
            OutDict['chi2'] = res.cost                                                      ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters
            OutDict['residuals'] = res.fun                                                  ## get residual


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Global optimization


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## Optimization algorithms from the scipy.optimize.minimize package that do not require a Jacobian or Hessian matrix


        ## use scipy's minimize method in conjunction with Nelder-Mead algorithm
        elif (self.Fit_Algorithm in ["nelder-mead"]):
            res = minimize(LocalFitFunction, p0, args = args, bounds = self.RangeList, method = 'Nelder-Mead', \
                           options = {"maxiter": self.Fit_NumIter, "maxfev": self.Fit_NumIter, "disp": True, \
                           "fatol": self.Fit_limit}, tol = self.Fit_limit)
            OutDict['chi2'] = res.fun                                                       ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters

            # Debug:
            # print ("res = ", res)


        ## use scipy's minimize method in conjunction with the modified Powell algorithm.
        elif (self.Fit_Algorithm in ["powell"]):
            res = minimize(LocalFitFunction, p0, args = args, bounds = self.RangeList, method = 'Powell', \
                           options = {"maxiter": self.Fit_NumIter, "maxfev": self.Fit_NumIter, "disp": True, \
                           "ftol": self.Fit_limit}, tol = self.Fit_limit)
            OutDict['chi2'] = res.fun                                                       ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters

            # Debug:
            # print ("res = ", res)


        ## use scipy's minimize method in conjunction with Constrained Optimization BY Linear Approximation (COBYLA) algorithm.
        elif (self.Fit_Algorithm in ["cobyla"]):
            res = minimize(LocalFitFunction, p0, args = args, bounds = self.RangeList, method = 'COBYLA', \
                           options = {"maxiter": self.Fit_NumIter, "maxfev": self.Fit_NumIter, "disp": True, \
                           "ftol": self.Fit_limit}, tol = self.Fit_limit)
            OutDict['chi2'] = res.fun                                                       ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters

            # Debug:
            # print ("res = ", res)


        ## use scipy's basinhopping algorithm
        elif (self.Fit_Algorithm in ["basinhopping"]):
            minimizer_kwargs = {"method": "BFGS", "args": args[0]}
            res = basinhopping(LocalFitFunction, p0, minimizer_kwargs = minimizer_kwargs, niter = self.Fit_NumIter, disp = True)
            OutDict['params'] = res.x                                                       ## get parameters

            # Debug:
            # print ("res = ", res)


        ## use scipy's differential_evolution algorithm
        ## option "vectorized = True" is available from version 1.9.0
        elif (self.Fit_Algorithm in ["differential_evolution"]):
            # self.UseCallbackFuncFlag = True
            # LocalCallback = OptAlgParamDict["callback"]
            res = differential_evolution(LocalFitFunction, self.RangeList, args = args, \
                                         maxiter = self.Fit_NumIter, tol = self.Fit_limit, \
                                         atol = self.Fit_limit)                             # , callback = LocalCallback
            OutDict['chi2'] = res.fun                                                       ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters

            # Debug:
            # print ("res = ", res)


        ## use scipy's shgo algorithm (not used because requires Hessian matrix)
        # elif (self.Fit_Algorithm in ["shgo"]):
        #     self.UseCallbackFuncFlag = True
        #     LocalCallback = OptAlgParamDict["callback"]
        #     shgo_options = {"f_tol" : self.Fit_limit, "maxiter" : self.Fit_NumIter}
        #     res = shgo(LocalFitFunction, self.RangeTupleList, args = args, options = shgo_options, \
        #                n = self.Fit_AlgDict["NumberSampler"], callback = LocalCallback)
        #     OutDict['chi2'] = res.fun                                                      ## get value of the cost function at the solution
        #     OutDict['params'] = res.x                                                      ## get parameters


        ## use scipy's dual_annealing algorithm
        elif (self.Fit_Algorithm in ["dual_annealing"]):
            res = dual_annealing(LocalFitFunction, self.RangeList, args = args, maxiter = self.Fit_NumIter)
            OutDict['chi2'] = res.fun                                                       ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters

            # Debug:
            # print ("res = ", res)


        ## use scipy's direct algorithm
        # elif (self.Fit_Algorithm in ["direct"]):
        #     args = OptAlgParamDict["args"]
        #     res = direct(LocalFitFunction, self.RangeList, args = args, maxiter = self.Fit_NumIter, \
        #                  maxfun = self.Fit_NumIter, f_min = self.Fit_limit)
        #     OutDict['chi2'] = res.fun                                                     ## get value of the cost function at the solution
        #     OutDict['params'] = res.x                                                     ## get parameters


        ## apply brute force method, using self-written brute force method
        elif (self.Fit_Algorithm in ["brute", "bruteforce", "brute-force"]):
            ReturnDict = self.CallBruteForce(LocalFitFunction, p0, args[0])
            OutDict['chi2'] = ReturnDict['chi2']                                            ## get value of the cost function at the solution
            OutDict['params'] = ReturnDict['params']                                        ## get parameters


        ## call PyGAD package
        elif (self.Fit_Algorithm in ["pygad"] and (self.NoFITSCubeFlag)):
            ReturnDict = self.CallPyGAD(LocalFitFunction, p0, args[0])
            OutDict['chi2'] = ReturnDict['chi2']                                            ## get value of the cost function at the solution
            OutDict['params'] = ReturnDict['params']                                        ## get parameters
            OutDict['residuals'] = ReturnDict['residuals']                                  ## get residual

        # Debug:
        # print ("OutDict = ", OutDict)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## Optimization algorithms from the scipy.optimize.minimize package that do not require a Hessian matrix but a Jacobian matrix


        ## use scipy's minimize method in conjunction with the conjugate gradient algorithm
        elif (self.Fit_Algorithm in ["cg"]):
            jac = OptAlgParamDict["GradientJAC"]
            res = minimize(LocalFitFunction, p0, args = args, bounds = self.RangeList, jac = jac, method = 'CG', \
                           options = {"maxiter": self.Fit_NumIter, "disp": True, }, tol = self.Fit_limit)
            OutDict['chi2'] = res.fun                                                       ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters


        ## use scipy's minimize method in conjunction with the BFGS algorithm
        elif (self.Fit_Algorithm in ["bfgs"]):
            jac = OptAlgParamDict["GradientJAC"]
            res = minimize(LocalFitFunction, p0, args = args, bounds = self.RangeList, jac = jac, method = 'BFGS', \
                           options = {"maxiter": self.Fit_NumIter, "disp": True, }, tol = self.Fit_limit)
            OutDict['chi2'] = res.fun                                                       ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters


        ## use scipy's minimize method in conjunction with the L-BFGS-B algorithm
        elif (self.Fit_Algorithm in ["l-bfgs-b"]):
            jac = OptAlgParamDict["GradientJAC"]
            res = minimize(LocalFitFunction, p0, args = args[0], bounds = self.RangeList, jac = jac, method = 'L-BFGS-B', \
                           options = {"maxiter": self.Fit_NumIter, "maxfun": self.Fit_NumIter, "disp": True, \
                                      "ftol": self.Fit_limit}, tol = self.Fit_limit)
            OutDict['chi2'] = res.fun                                                       ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters


        ## use scipy's minimize method in conjunction with a truncated Newton (TNC) algorithm
        elif (self.Fit_Algorithm in ["tnc"]):
            jac = OptAlgParamDict["GradientJAC"]
            res = minimize(LocalFitFunction, p0, args = args, bounds = self.RangeList, jac = jac, method = 'TNC', \
                           options = {"maxiter": self.Fit_NumIter, "disp": True, }, tol = self.Fit_limit)
            OutDict['chi2'] = res.fun                                                       ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters


        ## use scipy's minimize method in conjunction with Sequential Least Squares Programming (SLSQP)
        elif (self.Fit_Algorithm in ["slsqp"]):
            jac = OptAlgParamDict["GradientJAC"]
            res = minimize(LocalFitFunction, p0, args = args, bounds = self.RangeList, jac = jac, method = 'SLSQP', \
                           options = {"maxiter": self.Fit_NumIter, "disp": True, "ftol": self.Fit_limit}, tol = self.Fit_limit)
            OutDict['chi2'] = res.fun                                                       ## get value of the cost function at the solution
            OutDict['params'] = res.x                                                       ## get parameters


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## apply no optimization algorithm, just compute the objective function


        ## blank call of objective function
        elif (self.Fit_Algorithm in ["blank"]):
            OutDict['chi2'] = 0.0                                                           ## get value of the cost function at the solution
            OutDict['params'] = p0                                                          ## get parameters


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## apply emcee and ultranest algorithms


        ## call emcee
        elif (self.Fit_Algorithm in ["mcmc", "emcee"] and (self.NoFITSCubeFlag)):
            ReturnDict = self.Callemcee(LocalFitFunction, p0, args[0])
            OutDict['chi2'] = ReturnDict['chi2']                                            ## get value of the cost function at the solution
            OutDict['params'] = ReturnDict['params']                                        ## get parameters
            OutDict['residuals'] = ReturnDict['residuals']                                  ## get residual

        # Debug:
        # print ("OutDict = ", OutDict)


        ## call ultranest
        elif (self.Fit_Algorithm in ["ultranest"] and (self.NoFITSCubeFlag)):
            ReturnDict = self.CallUltraNest(LocalFitFunction, p0, args[0])
            OutDict['chi2'] = ReturnDict['chi2']                                            ## get value of the cost function at the solution
            OutDict['params'] = ReturnDict['params']                                        ## get parameters
            OutDict['residuals'] = ReturnDict['residuals']                                  ## get residual

        # Debug:
        # print ("OutDict = ", OutDict)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## in addition to the "blank" algorithm,  some algorithms do not return the synthetic spectrum
        ## for the optimized parameter vector, so we need to call the optimization function again
        if (self.Fit_Algorithm in ["blank", "nelder-mead", "powell", "cobyla", "cg", "bfgs", \
                                   "l-bfgs-b", "tnc", "slsqp", "basinhopping", "differential_evolution", \
                                   "shgo", "dual_annealing", "direct", "brute", "bruteforce", "brute-force"]):


            ## for the MapFit function and the blank algorithm, really compute the synthetic spectrum
            if (self.Fit_Algorithm in ["blank"] or (not self.NoFITSCubeFlag)):
                if (not self.Fit_Algorithm in ["blank"]):
                    p0 = OutDict['params']
                fun = LocalFitFunction(p0, AddParamDict)
                OutDict['residuals'] = fun


            ## for the single spectra fit, just copy the spectrum from the corresponding parameter
            elif (self.NoFITSCubeFlag):
                OutDict['residuals'] = self.BestFitResultsDict["BestParamFunc"]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define return parameter
        if (self.ParallelMethod in ["dask", "sc"]):
            IndexMap = OptAlgParamDict["IndexMap"]
            LocalBigCubeSpectralAxis = numpy.concatenate((OutDict['residuals'], OutDict['params'], [OutDict['chi2']], [IndexMap]))
            return LocalBigCubeSpectralAxis
        else:
            return OutDict
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, brute only): generate parameter grid
    ## taken from https://stackoverflow.com/questions/65392737/python-how-to-create-a-parameter-grid-with-dynamic-number-of-parameters
    def brute_dict_configs(self, d):
        """

    input parameters:
    -----------------

        - d:                        dictionary with 1D-arrays for each parameter


    output parameters:
    ------------------

        - ParamGrid:                dictionary with parameter vectors
        """

        # Debug:
        # print ("d = ", d)


        for vcomb in product(*d.values()):
            yield dict(zip(d.keys(), vcomb))
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, brute only): call brute force algorithm
    def CallBruteForce(self, LocalFitFunction, p0, args):
        """

    input parameters:
    -----------------

        - LocalFitFunction:             fitting function

        - p0:                           initial parameter vector

        - args:                         additional argument


    output parameters:
    ------------------

        - OutDict:                      dictionary with output parameters:

                                        - OutDict["chi2"]:       best chi^2 value
                                        - OutDict["params"]:     parameter vector with lowest chi^2 value
                                        - OutDict["residuals"]:  synthetic spectrum with lowest chi^2 value
        """

        # Debug:
        # print ("p0 = ", p0)


        ## define parameter grid points
        number_brute_steps = self.Fit_AlgDict['BruteSteps']
        ParamGridDict = {}
        for LocalRangeID, LocalRange in enumerate(self.RangeList):
            MinParam = min(LocalRange[0], LocalRange[1])
            MaxParam = max(LocalRange[0], LocalRange[1])
            LocalLoop = numpy.linspace(MinParam, MaxParam, number_brute_steps[LocalRangeID], endpoint = True)
            ParamGridDict[LocalRangeID] = LocalLoop

        # Debug:
        # print ("ParamGridDict = ", ParamGridDict)


        ## generate parameter grid
        AllParamVector = [p0]
        for config in self.brute_dict_configs(ParamGridDict):
            LocalParamVec = []
            for LocalKey in config.keys():
                LocalParamVec.append(config[LocalKey])
            AllParamVector.append(LocalParamVec)

        # Debug:
        # print ("AllParamVector = ", AllParamVector)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## compute chi^2 values for each parameter vector


        ## compute chi^2 for the MapFit function
        if (not self.NoFITSCubeFlag):
            AddParamDict = args
            AddParamDict["Fit_LSFlag"] = 2
            BestChi2 = 1.e99
            p0Best = None
            for p0 in AllParamVector:
                LocalChi2 = LocalFitFunction(p0, args)
                if (LocalChi2 < BestChi2):
                    p0Best = p0
                    BestChi2 = LocalChi2


            ## define return parameter
            OutDict = {}
            OutDict["chi2"] = BestChi2
            OutDict["params"] = p0Best


        ## compute chi^2 for the CuebFit function and single spectra fits
        else:
            ListChi2 = LocalFitFunction(AllParamVector, args)


            ## define return parameter
            OutDict = {}
            OutDict["chi2"] = self.BestFitResultsDict["BestChi2Value"]
            OutDict["params"] = self.BestFitResultsDict["BestParamVec"]


        ## we're done
        return OutDict
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, emcee only): compute the log-prior
    ## partly taken from https://emcee.readthedocs.io/en/stable/tutorials/line/
    def log_prior(self, p0List):
        """

    input parameters:
    -----------------

        - p0List:                       list of parameter vectors


    output parameters:
    ------------------

        - log_prior:                    0.0 if all parameter values are within their limits, -numpy.inf otherwise
        """

        # Debug:
        # print ("p0List = ", p0List)


        ## check, if at least one parmaeter is out of its limit
        if (p0List.ndim == 1):
            p0List = [p0List]
        log_prior_list = []
        for p0 in p0List:
            OutOfBoundFlag = False
            for i, pi in enumerate(p0):
                if (pi < self.pBLow[i] or self.pBUp[i] < pi):
                    OutOfBoundFlag = True
                    break
            if (OutOfBoundFlag):
                log_prior_list.append(-numpy.inf)
            else:
                log_prior_list.append(0.0)
        return log_prior_list
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, emcee and UltraNest only): compute the full log-probability function
    ## partly taken from https://emcee.readthedocs.io/en/stable/tutorials/line/
    def log_probability(self, p0List, LocalFitFunction, args):
        """

    input parameters:
    -----------------

        - p0List:                       list of parameter vectors

        - args:                         additional arguments


    output parameters:
    ------------------

        - log_probability:              log_likelihood if all parameter values are within their limits, -numpy.inf otherwise
        """

        # Debug:
        # print ("p0List = ", p0List)
        # print ("args = ", args)


        ## compute the log-prior and define list of parameter vectors for likelihood determination
        lpList = self.log_prior(p0List)
        log_probability_list = []
        Localp0List = []
        for lpID, lp in enumerate(lpList):
            if (not numpy.isfinite(lp)):
                log_probability_list.append(-numpy.inf)
            else:
                Localp0List.append(p0List[lpID])
                log_probability_list.append(lp)

        # Debug:
        # print ("Localp0List = ", Localp0List)
        # print ("lpList = ", lpList)
        # print ("len(Localp0List) = ", len(Localp0List))
        # print ("len(lpList) = ", len(lpList))


        ## compute likelihood
        log_list = LocalFitFunction(Localp0List, args)

        # Debug:
        # print ("log_list = ", log_list)
        # print ("len(log_list) = ", len(log_list))
        # sys.exit(0)


        ## add likelihood values to list of log_probability
        i = (-1)
        for lpID, lp in enumerate(lpList):
            if (numpy.isfinite(lp)):
                i += 1
                log_probability_list[lpID] = log_list[i]

        # Debug:
        # print ("log_probability_list = ", log_probability_list)


        ## we're done
        return log_probability_list
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, emcee only): calculate mode(s) of list
    ## taken from http://stackoverflow.com/questions/10797819/finding-the-mode-of-a-list-in-python, and modified
    def ComputeMode(self, arr, LocalLeftLimit, LocalRightLimit, BestFitPoint):
        """

    input parameters:
    -----------------

        - arr:                          list which is analyzed

        - LocalLeftLimit:               lower limit of list elements

        - LocalRightLimit:              upper limit of list elements

        - BestFitPoint:                 parameter value of best fit


    output parameters:
    ------------------

        - LocalMode:                    calculated list of mode(s)
        """

        # Debug:
        # print ("arr = ", arr)
        # print ("LocalLeftLimit = ", LocalLeftLimit)
        # print ("LocalRightLimit = ", LocalRightLimit)
        # print ("BestFitPoint = ", BestFitPoint)


        ## initialize return parameter
        LocalMode = []


        ## select list elements within given limits
        LocalList = []
        for a in arr:
            if (LocalLeftLimit <= a and a <= LocalRightLimit):
                LocalList.append(a)


        ## calculate mode(s)
        if (LocalList != []):

            # Debug:
            # print ("old LocalMode = ", LocalMode)


            ## compute histogram for selected list elements
            hist, bin_edges = numpy.histogram(LocalList, bins = 20)
            BinDist = numpy.diff(bin_edges)[0]
            ii = numpy.argmax(hist)
            LocalMode = bin_edges[ii]
            if (abs(BestFitPoint - LocalMode) <= BinDist):                                  ## if mode is close to best fit value, use best fit value
                LocalMode = max(LocalLeftLimit, BestFitPoint)
                LocalMode = min(BestFitPoint, LocalRightLimit)

            # Debug:
            # print ("new LocalMode = ", LocalMode)
            # print ("BestFitPoint = ", BestFitPoint)


        ## define return parameters
        return LocalMode
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, emcee only): compute highest probability density region given by a set of samples.
    def ComputeHPD(self, trace, mass_frac):
        """

        taken from
        http://bebi103.caltech.edu/2015/tutorials/l06_credible_regions.html

        Computation of the HPD is a little trickier. The function below will compute the HPD interval. The idea is that we rank-order the MCMC trace.
        We know that the number of samples that are included in the HPD is 0.95 times the total number of MCMC sample. We then consider all intervals
        that contain that many samples and find the shortest one.


        Parameters
        ----------
        trace : array
            1D array of MCMC samples for a single variable
        mass_frac : float with 0 < mass_frac <= 1
            The fraction of the probability to be included in
            the HPD.  For example, `massfrac` = 0.95 gives a
            95% HPD.

        Returns
        -------
        output : array, shape (2,)
            The bounds of the HPD
        """

        # Debug:
        # print ("trace = ", trace)
        # print ("mass_frac = ", mass_frac)


        ## Get sorted list
        d = numpy.sort(numpy.copy(trace))

        # Debug:
        # print ("d = ", d)
        # print ("d[0] = ", d[0])


        ## Number of total samples taken
        n = len(trace)

        # Debug:
        # print ("n = ", n)


        ## Get number of samples that should be included in HPD
        n_samples = numpy.floor(mass_frac * n).astype(int)

        # Debug:
        # print ("n_samples = ", n_samples)


        ## Get width (in units of data) of all intervals with n_samples samples
        int_width = d[n_samples:] - d[:n-n_samples]

        # Debug:
        # print ("int_width = ", int_width)


        ## Pick out minimal interval
        min_int = numpy.argmin(int_width)
        dMin = max(0, min(min_int, (n - 1)))
        dMax = max(0, min(min_int + n_samples, (n - 1)))

        # Debug:
        # print ("min_int = ", min_int)
        # print ("dMin = ", dMin)
        # print ("dMax = ", dMax)


        ## Return interval
        return numpy.array([d[dMin], d[dMax]])
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, emcee and UltraNest only): compute errors and create corner plot
    ## partly taken from https://emcee.readthedocs.io/en/stable/tutorials/line/
    def make_corner_plot(self, NameAlg, FinalSamples, SamplesRaw, FinalSamplesOrig, ndim, \
                         bestparam, LocalOutputDirectory, UltraNestDict = None):
        """

    input parameters:
    -----------------

        - NameAlg:                      name of calling algortihm

        - FinalSamples:                 ?

        - SamplesRaw:                   numpy array with parameter evolution values

        - FinalSamplesOrig:             ?

        - ndim:                         number of fit parameters

        - bestparame:                   parameter vector for best result

        - LocalOutputDirectory:         path and name of subdirectory for output

        - UltraNestDict:                (optional) UltraNest dictionary


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("NameAlg = ", NameAlg)
        # print ("FinalSamples = ", FinalSamples)
        # print ("SamplesRaw = ", SamplesRaw)
        # print ("FinalSamplesOrig = ", FinalSamplesOrig)
        # print ("ndim = ", ndim)
        # print ("bestparam = ", bestparam)
        # print ("LocalOutputDirectory = ", LocalOutputDirectory)
        # print ("UltraNestDict = ", UltraNestDict)


        ## get some algorithm parameters
        SigmaMultiplicity = self.Fit_AlgDict["SigmaMultiplicity"]
        ErrorType = self.Fit_AlgDict["ErrorType"]
        CornerPlotType = "new"
        LocalPlotFormat = self.Fit_AlgDict["PlotFileFormat"]


        ## define error bounds using scipy.special.ndtr (Gaussian cumulative distribution
        ## function, returns the area under the standard Gaussian probability
        ## density function, integrated from minus infinity to x:
        ## 1/math.sqrt(2 * math.pi) * integral(numpy.exp(-t**2 / 2), t=-inf..x)
        sigma = ndtr(SigmaMultiplicity) - ndtr(-SigmaMultiplicity)
        sigmaLimits = [50 - (sigma * 1.e2 / 2.0), 50, 50 + (sigma * 1.e2 / 2.0)]

        # Debug:
        # print ("sigmaLimits = ", sigmaLimits)


        ## create plots for each parameter describing the histogram for each parameter
        binVal = 400
        RefPointList = []
        LocalErrorLeftList = []
        LocalErrorRightList = []
        LabelList = []
        Error_left = numpy.array(list(range(ndim)), dtype = 'float')
        Error_right = numpy.array(list(range(ndim)), dtype = 'float')
        error_tag_content = []
        for ParamIndex, xs in enumerate(FinalSamples):

            # Debug:
            # print ("ParamIndex, xs = ", ParamIndex, xs)


            ## define list of histogram labels
            ## get some informations for current parameter
            ## self.FitParameterNames[i] = [LocalMolecule, LocalLineID, Name, LogFlag]
            ## self.FitParameterNames[i] = [LocalIsotopologue, (-1), "IsoRatio", False]
            try:
                LocalParamName = self.FitParameterNames[ParamIndex][2]
            except:
                LocalParamName = None
            if (LocalParamName is None):
                CurrParamName = "free param. {:d}".format(ParamIndex + 1)
            else:
                LocalLatexParamName = task_myXCLASS.GetLatexParameterName(LocalParamName)
                LocalMoleculeName = self.FitParameterNames[ParamIndex][0]
                LocalComponentIndex = self.FitParameterNames[ParamIndex][1] + 1
                if (LocalParamName in ["IsoRatio"]):
                    CurrParamName = LocalParamName
                elif (LocalMoleculeName is not None and LocalComponentIndex is not None):
                    CurrParamName = "{:s},\ncomp. {:d}".format(chr(34) + LocalMoleculeName + chr(34), \
                                                                LocalComponentIndex)
                    if (LocalLatexParamName != ""):
                        CurrParamName += ", {:s}".format(LocalLatexParamName)
                else:
                    CurrParamName = LocalParamName
            LabelList.append(CurrParamName)

            # Debug:
            # print ("LabelList = ", LabelList)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## create plots for each parameter describing the parameter values for each iteration
            if (NameAlg == "emcee"):
                LocalSamples = SamplesRaw[:, :, ParamIndex]
                NumIterations = len(LocalSamples[:, 0])
                fig = pylab.figure(figsize = (15, 10))
                fig.clear()
                layer = pylab.subplot(1, 1, 1)
                pylab.grid(True)
                pylab.xlabel("Iteration steps")
                pylab.ylabel("Parameter values for free parameter {:d}:\n{:s}".format((ParamIndex + 1), CurrParamName.replace("\n", "")))
                layer.xaxis.set_major_formatter(matplotlib.ticker.FormatStrFormatter('%i'))
                layer.plot(LocalSamples, color = "k", alpha = 0.4)
                layer.set_xlim(0, len(SamplesRaw))
                OutputFileName = "{:s}{:s}__parameter-evolution_of_free-parameter_{:d}".format(LocalOutputDirectory, NameAlg, (ParamIndex + 1))
                pylab.savefig(OutputFileName + ".{:s}".format(LocalPlotFormat), bbox_inches = 'tight')
                pylab.draw()
                matplotlib.pyplot.close(fig)


                ## save results of parameter evolution as well
                XColumn = numpy.arange(1, NumIterations + 1, 1)
                LocalSamples = numpy.insert(LocalSamples, 0, XColumn, axis = 1)             ## add x-column describing iteration step
                numpy.savetxt(OutputFileName + ".dat", LocalSamples)


            ##----------------------------------------------------------------------------------------------------------------------------------------
            ## create histogramm for each free parameter including different error bars
            fig = pylab.figure(figsize = (15, 10))
            fig.clear()
            layer = pylab.subplot(1, 1, 1)
            pylab.grid(True)
            pylab.xlabel("Free parameter {:d}: {:s}".format((ParamIndex + 1), CurrParamName.replace("\n", "")))
            pylab.ylabel("Probability density")
            if hasattr(xs, "compressed"):
                xs = xs.compressed()
            n, b, p = layer.hist(xs, histtype = 'step', density = True, bins = binVal, lw = 1, fill = True)

            # Debug:
            # print ("xs[0] = ", xs[0])
            # print ("xs = ", xs)


            ##----------------------------------------------------------------------------------------------------------------------------------------
            ## compute quantile credible regions
            means = xs.mean()                                                           ## compute mean
            stds = xs.std()                                                             ## compute std.
            LocalErrorLeftGauss = means - (1.0 + sigma) * stds
            LocalErrorRightGauss = means + (1.0 + sigma) * stds
            RefGauss = means

            # Debug:
            # print ("LocalErrorLeftGauss = ", LocalErrorLeftGauss)
            # print ("LocalErrorRightGauss = ", LocalErrorRightGauss)


            ##----------------------------------------------------------------------------------------------------------------------------------------
            ## compute quantile credible regions (use numpy.percentile)
            perctiles = numpy.percentile(xs, sigmaLimits)                               ## compute perctiles for corresponding sigma multiplicity
            LocalErrorLeftQuant = min(perctiles[0], perctiles[2])                       ## get lower limit of credible interval
            LocalErrorRightQuant = max(perctiles[0], perctiles[2])                      ## get upper limit of credible interval
            RefQuantile = perctiles[1]                                                  ## get median of xs

            # Debug:
            # print ("LocalErrorLeftQuant = ", LocalErrorLeftQuant)
            # print ("LocalErrorRightQuant = ", LocalErrorRightQuant)


            ##----------------------------------------------------------------------------------------------------------------------------------------
            ## compute HPD credible regions
            hpds = self.ComputeHPD(xs, sigma)                                           ## use caltech routine to compute HPD interval
            LocalErrorLeftHPD = min(hpds[0], hpds[1])                                   ## get lower limit of HPD interval
            LocalErrorRightHPD = max(hpds[0], hpds[1])                                  ## get upper limit of HPD interval
            RefHPD = self.ComputeMode(xs, LocalErrorLeftHPD, LocalErrorRightHPD, bestparam[ParamIndex]) ## determine mode

            # Debug:
            # print ("hpds = ", hpds)
            # print ("hpds2 = ", hpds2)
            # print ("RefHPD = ", RefHPD)
            # print ("LocalErrorLeftHPD = ", LocalErrorLeftHPD)
            # print ("LocalErrorRightHPD = ", LocalErrorRightHPD)


            ##----------------------------------------------------------------------------------------------------------------------------------------
            ## get error parameters for UltraNest
            if (UltraNestDict is not None):
                LocalErrorLeftUltraNest = UltraNestDict["errlo"][ParamIndex]
                LocalErrorRightUltraNest = UltraNestDict["errup"][ParamIndex]
                RefUltraNest = UltraNestDict["median"][ParamIndex]
            else:
                LocalErrorLeftUltraNest = None
                LocalErrorRightUltraNest = None
                RefUltraNest = None


            ## check if mode lies within HPD interval
            # if (modes < LocalErrorLeftHPD or LocalErrorRightHPD < modes):
            # print (" ")


            ## store calculated errors
            if (ErrorType == "gauss"):
                RefPoint = RefGauss
                LocalErrorLeft = LocalErrorLeftGauss
                LocalErrorRight = LocalErrorRightGauss
            elif (ErrorType == "quantile"):
                RefPoint = RefQuantile
                LocalErrorLeft = LocalErrorLeftQuant
                LocalErrorRight = LocalErrorRightQuant
            elif (ErrorType == "hpd"):
                RefPoint = RefHPD
                LocalErrorLeft = LocalErrorLeftHPD
                LocalErrorRight = LocalErrorRightHPD
            elif (ErrorType == "ultranest"):
                RefPoint = RefUltraNest
                LocalErrorLeft = LocalErrorLeftUltraNest
                LocalErrorRight = LocalErrorRightUltraNest
            Error_left[ParamIndex] = RefPoint - abs(LocalErrorLeft)
            Error_right[ParamIndex] = abs(LocalErrorRight) - RefPoint
            tag_content =[abs(Error_left[ParamIndex]),  abs(Error_right[ParamIndex])]
            error_tag_content.append(tag_content)
            RefPointList.append(RefPoint)
            LocalErrorLeftList.append(LocalErrorLeft)
            LocalErrorRightList.append(LocalErrorRight)


            ## print errors to screen and log file
            LogLine = ("\n\n\t Errors for free param. {:d}:\n".format(ParamIndex + 1))
            LogLine += ("\t Gaussian:  = [{:.10e}, {:.10e}]\n".format(LocalErrorLeftGauss, LocalErrorRightGauss))
            LogLine += ("\t Percentile = [{:.10e}, {:.10e}]\n".format(LocalErrorLeftQuant, LocalErrorRightQuant))
            LogLine += ("\t HPD        = [{:.10e}, {:.10e}]\n".format(LocalErrorLeftHPD, LocalErrorRightHPD))
            if (LocalErrorLeftUltraNest is not None and LocalErrorRightUltraNest is not None):
                LogLine += ("\t UltraNest  = [{:.10e}, {:.10e}]\n".format(LocalErrorLeftUltraNest, LocalErrorRightUltraNest))
            print (LogLine)
            self.LogFile.write(LogLine.replace("\t", ""))


            ##----------------------------------------------------------------------------------------------------------------------------------------
            ## plot error levels
            layer.axvline(x = RefPoint, color = 'k', linewidth = 1, linestyle = '--')
            layer.axvline(x = LocalErrorLeft, color = 'r', linewidth = 1, linestyle = '--')
            layer.axvline(x = LocalErrorRight, color = 'r', linewidth = 1, linestyle = '--')


            ##----------------------------------------------------------------------------------------------------------------------------------------
            ## save plots
            ylims = layer.get_ylim()                                                    ## get y-limits
            layer.set_ylim(ylims[0], ylims[1] * 1.07)
            layer.set_xlim(min(self.pBLow[ParamIndex], self.pBUp[ParamIndex]), max(self.pBLow[ParamIndex], self.pBUp[ParamIndex]))
            OutputFileName = "{:s}{:s}__histogram_of_free-parameter_{:d}" \
                              .format(LocalOutputDirectory, NameAlg, (ParamIndex + 1))
            pylab.savefig(OutputFileName + ".{:s}".format(LocalPlotFormat), bbox_inches = 'tight')
            pylab.draw()
            matplotlib.pyplot.close(fig)


            ## construct ascii file for histogram data
            histascii_file = open(OutputFileName + ".dat", 'w')
            NewLine = "{:25s}  {:25s}\n".format("bin:", "prob.:")
            histascii_file.write(NewLine)
            for count, val in enumerate(n):
                NewLine = "{:25.15e}  {:25.15e}\n".format(b[count], val)
                histascii_file.write(NewLine)
            histascii_file.close()


        ##--------------------------------------------------------------------------------------------------------------------------------------------
        ## do corner plots for models with a few parameters
        if (ndim < 1e99):
            print ("\n\n\n\t Create corner plots .. ", end = "", flush = True)


            ## define error marks
            ErrorMarks = []
            if (ErrorType in ["gauss", "hpd", "ultranest"]):
                for i, LocalRefPoint in enumerate(RefPointList):
                    ErrorMarks.append([LocalErrorLeftList[i], LocalRefPoint, LocalErrorRightList[i]])


            ## define type of error
            if (ErrorType == "gauss"):
                LocalLevels = (1.0 - numpy.exp(-0.5 * SigmaMultiplicity**2),)
            elif (ErrorType == "quantile"):
                ErrorMarks = [[sigmaLimits[0] * 1.e-2, sigmaLimits[1] * 1.e-2, sigmaLimits[2] * 1.e-2]]
                LocalLevels = (1.0 - numpy.exp(-0.5 * SigmaMultiplicity**2),)
            elif (ErrorType == "hpd"):
                LocalLevels = (1.0 - numpy.exp(-0.5 * SigmaMultiplicity**2),)
            elif (ErrorType == "ultranest"):
                LocalLevels = (1.0 - numpy.exp(-0.5 * SigmaMultiplicity**2),)


            ## store parameters for corner plot into pickle file
            CornerDict = {}
            CornerDict['xs'] = FinalSamplesOrig
            CornerDict['bins'] = binVal
            CornerDict['labels'] = LabelList
            CornerDict['truths'] = bestparam
            CornerDict['quantiles'] = ErrorMarks
            CornerDict['levels'] = LocalLevels
            CornerDict['ErrorTyp'] = ErrorType
            # OutputFileName = LocalOutputDirectory + "corner-parameter-dictionary__call_{:d}.pkl".format(abs(ErrINSCounter))
            OutputFileName = LocalOutputDirectory + "corner-parameter-dictionary.pkl"
            f = open(OutputFileName, "wb")
            pickle.dump(CornerDict, f)
            f.close()


            ## import modified corner plot
            from xclass.external.emcee import emcee__plotting


            ## make traditional corner package
            if (CornerPlotType in ["old"]):
                fig = emcee__plotting.corner(FinalSamplesOrig, bin = binVal, labels = LabelList, truths = bestparam, quantiles = ErrorMarks, \
                                             show_titles = True, title_kwargs = {"fontsize": 10}, title_fmt = ".4e", levels = LocalLevels, \
                                             ErrorTyp = ErrorType)
                fig.tight_layout(w_pad = 0.0, h_pad = 0.0)
                OutputFileName = "{:s}{:s}__corner-plot.{:s}".format(LocalOutputDirectory, NameAlg, LocalPlotFormat)
                fig.savefig(OutputFileName)


            ## make corner package in Peter's style
            else:
                quantile = numpy.quantile(FinalSamplesOrig, 0.5, axis = 0)
                ndim = FinalSamplesOrig.shape[1]


                ## add median to list of quantiles
                ModErrorMarks = copy.deepcopy(ErrorMarks)
                for NumParam, LocalQuantile in enumerate(quantile):
                    ModErrorMarks[NumParam][1] = LocalQuantile

                # Debug:
                # print ("ModErrorMarks = ", ModErrorMarks)


                ## create corner plot
                figure = emcee__plotting.corner(FinalSamplesOrig, bin = binVal, labels = LabelList, \
                                                plot_datapoints = False, quantiles = ModErrorMarks, \
                                                show_titles = True, title_kwargs = {"fontsize": 12}, \
                                                title_fmt = ".4e", levels = LocalLevels, \
                                                ErrorTyp = ErrorType, norm_height_hist = True, \
                                                hist_kwargs = {"density" : False}, \
                                                add_label_hist = True)


                # Extract the axes
                axes = numpy.array(figure.axes).reshape((ndim, ndim))
                custom_lines = [Line2D([0], [0], color = 'cornflowerblue', lw = 4),
                                Line2D([0], [0], color = 'orange', lw = 4, ls = '--')]


                # Loop over the diagonal
                for i in range(ndim):
                    ax = axes[i, i]
                    t = ax.get_title()
                    nt = t.replace(" = ", ":\n")
                    ax.set_title(nt, y = 1.15)
                    ax.axvline(quantile[i], color = "cornflowerblue")
                    ax.axvline(bestparam[i], color = "orange", ls = '--')
                    bst = f'Min: {bestparam[i]:.2f}'
                    # ax.legend(custom_lines, ['Median', bst], loc = 'lower left')
                    ax.legend(custom_lines, ['Median', bst], bbox_to_anchor = (0.0, 1.02, 1.0, 0.102), \
                                loc = 3, ncol = 2, mode = "expand", borderaxespad = 0.0, fontsize = 'small')


                # Loop over the histograms
                for yi in range(ndim):
                    for xi in range(yi):
                        ax = axes[yi, xi]
                        ax.axvline(quantile[xi], color = "cornflowerblue")
                        ax.axhline(quantile[yi], color = "cornflowerblue")
                        ax.plot(quantile[xi], quantile[yi], "s", color = 'cornflowerblue')
                        ax.axvline(bestparam[xi], color = "orange", ls = '--')
                        ax.axhline(bestparam[yi], color = "orange", ls = '--')
                        ax.plot(bestparam[xi], bestparam[yi], "s", color = 'orange', ls = '--')


                ## save figure
                OutputFileName = "{:s}{:s}__corner-plot.{:s}".format(LocalOutputDirectory, NameAlg, LocalPlotFormat)
                figure.savefig(OutputFileName, bbox_inches = 'tight')


            ## we're done
            print ("done!")


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, emcee only): call emcee package
    ## partly taken from https://emcee.readthedocs.io/en/stable/tutorials/line/
    def Callemcee(self, LocalFitFunction, p0, addargs):
        """

    input parameters:
    -----------------

        - LocalFitFunction:             objective function

        - p0:                           current parameter vector

        - addargs:                      additional arguments


    output parameters:
    ------------------

        - OutDict:                      dictionary with output parameters:

                                        - OutDict["chi2"]:       best chi^2 value
                                        - OutDict["params"]:     parameter vector with lowest chi^2 value
                                        - OutDict["residuals"]:  synthetic spectrum with lowest chi^2 value
        """

        # Debug:
        # print ("LocalFitFunction = ", LocalFitFunction)
        # print ("p0 = ", p0)
        # print ("addargs = ", addargs)


        ## try to import emcee package
        try:
            import emcee
        except:
            print ("Emcee package is not available!")
            return


        ## avoid problems
        os.environ["OMP_NUM_THREADS"] = "1"


        ## get some parameters for emcee from alg. xml file
        NumberSampler = self.Fit_AlgDict["NumberSampler"]
        NumberBurnInIter = self.Fit_AlgDict["NumberBurnInIter"]
        BackendFileName = self.Fit_AlgDict["BackendFileName"]
        SampleMethod = self.Fit_AlgDict["SampleMethod"]
        LocalPlotFormat = self.Fit_AlgDict["PlotFileFormat"]
        LocalOutputDirectory = self.myXCLASSMapFitJobDir + "emcee/"
        os.mkdir(LocalOutputDirectory)
        ndim = len(p0)

        # Debug:
        # print ("NumberSampler = ", NumberSampler)
        # print ("BackendFileName = ", BackendFileName)
        # print ("SampleMethod = ", SampleMethod)
        # print ("ndim = ", ndim)
        # sys.exit(0)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine initial positions
        p0Init = []
        for i in range(NumberSampler):
            NewVector = []
            for j in range(ndim):
                a = self.pBLow[j]
                b = self.pBUp[j]


                ## distribute walkers in the entire parameter space
                if (SampleMethod == "global"):
                    NewValue = (b - a) * numpy.random.random_sample() + a


                ## distribute walkers in the neighbourhood of min. point
                elif (SampleMethod == "local"):
                    MinVal = p0[j]
                    range_ratio = 0.001
                    NewValue = (-1.0)**numpy.random.random_integers(2) * (b - a) * numpy.random.random_sample() * range_ratio / 2 + MinVal


                    ## Should still catch the case when MinVal is too close to the edges.
                    if (NewValue < a):
                        NewValue = a
                    elif (NewValue > b):
                        NewValue = b
                NewVector.append(NewValue)
            p0Init.append(NewVector)

        # Debug:
        # for ppi in p0Init:
        #     print (ppi)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get emcee backends
        backend = None
        if (BackendFileName is not None):
            if (BackendFileName.strip() != ""):
                # import backends
                # backend = backends.HDFBackend(BackendFileName)
                backend = emcee.backends.HDFBackend(BackendFileName)
                # backend.reset(nwalkers, ndim)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ##  initializing the walkers
        sampler = emcee.EnsembleSampler(NumberSampler, ndim, self.log_probability, \
                                        args = (LocalFitFunction, addargs), \
                                        vectorize = self.VectorizedFlag, backend = backend)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## restore parameter from previous h5py file
        Local_max_iter = self.Fit_NumIter
        Local_NumberBurnInIter = NumberBurnInIter
        res = None
        if (backend is not None):
            if (backend.iteration > 0):
                pp0 = sampler.get_last_sample()
                pos = pp0
                prob = sampler.get_log_prob()
                # burnin = sampler.burnin
                burnin = False
                DoneIterations = sampler.iteration
                self.CurrentFitIteration = DoneIterations


                ## print what you do
                OutputString1 = "\n\n\t Restoring parameters from backend file .."
                self.LogFile.write(OutputString1.replace("\t", "") + "\n")
                print (OutputString1)
                if (burnin):
                    OutputString2 = "\t Found {:d} iteration(s) from burn-in phase.\n\n".format(DoneIterations)
                else:
                    OutputString2 = "\t Found {:d} iteration(s) from previous emcee run.\n\n".format(DoneIterations)
                self.LogFile.write(OutputString2.replace("\t", ""))
                print (OutputString2)


                ## compute samples
                if (burnin):
                    Local_NumberBurnInIter = max(0, NumberBurnInIter - DoneIterations)
                    for _ in sampler.sample(pp0, iterations = Local_NumberBurnInIter, store = True):
                        continue
                else:
                    Local_NumberBurnInIter = 0
                    Local_max_iter = max(0, self.Fit_NumIter - DoneIterations)
                    # for _ in sampler.sample(pp0, iterations = 1, store = True):
                    #     continue


                    ## hack to get the correct results for each iteration
                    ## original: res = sampler.run_mcmc(pp0, Local_max_iter)
                    results = None
                    for results in sampler.sample(pp0, iterations = Local_max_iter, progress = True):
                        self.CurrentFitIteration += 1
                        OutputLine = "{:10d}".format(self.CurrentFitIteration)
                        OutputLine += "   {:16.8e}".format(self.BestFitResultsDict["BestChi2Value"])
                        OutputLine += "   {:s}".format(self.BestFitResultsDict["BestParamVecString"])
                        self.LogFile.write(OutputLine + "\n")

                # Debug:
                # print ("\nsampler = ", sampler)
                # print ("pp0 = ", pp0)
                # print ("pos = ", pos)
                # print ("prob = ", prob)
                # print ("burnin = ", burnin)
                # print ("backend.iteration = ", backend.iteration)
                # print ("sampler.iteration = ", sampler.iteration)
                # print ("DoneIterations = ", DoneIterations)

        # Debug:
        # print ("\nNumberBurnInIter = ", NumberBurnInIter)
        # print ("Local_NumberBurnInIter = ", Local_NumberBurnInIter)
        # print ("Local_max_iter = ", Local_max_iter)
        # print ("pp0 = ", pp0)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## perform burn-in iterations
        if (Local_NumberBurnInIter > 0):


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## print what you do to log file (and screen)
            OutputString1 = "\n\n\t Calculate burn-in steps:\n"
            self.LogFile.write(OutputString1 + "\n")
            # OutputString2 = "\n\n\t Iteration:                    chi^2:     Parameter:"
            # self.LogFile.write(OutputString2 + "\n")
            self.LogFile.flush()
            print (OutputString1, flush = True)
            # print (OutputString2, flush = True)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## Burn-In: Run NumberBurnInIter steps as a burn-in.
            ## hack to get the correct results for each iteration
            ## original: res = sampler.run_mcmc(p0Init, Local_NumberBurnInIter)
            results = None
            for results in sampler.sample(p0Init, iterations = Local_NumberBurnInIter, progress = True):
                self.CurrentFitIteration += 1
                OutputLine = "{:10d}".format(self.CurrentFitIteration)
                OutputLine += "   {:16.8e}".format(self.BestFitResultsDict["BestChi2Value"])
                OutputLine += "   {:s}".format(self.BestFitResultsDict["BestParamVecString"])
                self.LogFile.write(OutputLine + "\n")
            pos = copy.deepcopy(p0Init)


            ## print what you do
            print ("\n\n\t Main iteration loop:\n")


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## Reset the chain to remove the burn-in samples.
            # sampler.reset()
        else:
            pos = copy.deepcopy(p0Init)

        # Debug:
        # print ("\npos = ", pos)
        # print ("len(pos) = ", len(pos))
        # print ("Local_max_iter = ", Local_max_iter)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## make final iterations


        ## hack to get the correct results for each iteration
        ## original: res = sampler.run_mcmc(pos, max(1, Local_max_iter), progress = True)
        results = None
        for results in sampler.sample(pos, iterations = max(1, Local_max_iter), progress = True):
            self.CurrentFitIteration += 1
            OutputLine = "{:10d}".format(self.CurrentFitIteration)
            OutputLine += "   {:16.8e}".format(self.BestFitResultsDict["BestChi2Value"])
            OutputLine += "   {:s}".format(self.BestFitResultsDict["BestParamVecString"])
            self.LogFile.write(OutputLine + "\n")

        # Debug:
        # print ("res = ", res)
        # print ("\npos = ", pos)
        # print ("\n\n")
        # for i in xrange(len(pos)):
        #     print (prob[i], pos[i])


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## post processing
        if (sampler is not None):


            ## get some parameters
            FinalSamplesOrig = sampler.chain[:, :, :].reshape((-1, ndim))
            FinalSamples = numpy.atleast_1d(FinalSamplesOrig)
            bestparam = self.BestFitResultsDict["BestParamVec"]
            SamplesRaw = sampler.get_chain()
            if len(FinalSamples.shape) == 1:
                FinalSamples = numpy.atleast_2d(FinalSamples)
            else:
                assert len(FinalSamples.shape) == 2, "The input sample array must be 1- or 2-D."
                FinalSamples = FinalSamples.T


            ## make corner plot
            NameAlg = "emcee"
            self.make_corner_plot(NameAlg, FinalSamples, SamplesRaw, FinalSamplesOrig, ndim, bestparam, LocalOutputDirectory)


            ## Print out the mean acceptance fraction. In general, acceptance_fraction has an entry for each walker so, in this case,
            ## it is a ndim-dimensional vector.
            LogLine = "\n\n\t Mean acceptance fraction:  {:s}\n\n".format(str(numpy.mean(sampler.acceptance_fraction)))
            print (LogLine)
            self.LogFile.write(LogLine.replace("\t", ""))


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## Estimate the integrated autocorrelation time for the time series in each parameter.
            try:
                AutoCorrTime = sampler.get_autocorr_time()
            except:
                AutoCorrTime = None
                print ("\t WARNING:")
                print ("\t\t XCLASS can not determine the integrated autocorrelation time !")
                print ("\t\t The chain is shorter than 50 times the integrated autocorrelation time for {:d} parameter(s).".format(ndim))
                print ("\t\t Run a longer chain!")
                print ("\n\n\n\n")
            if (AutoCorrTime is not None):
                LogLine = "\n\n\t Autocorrelation time for each free parameter:"
                for i in range(ndim):
                    LogLine += "\t\t Autocorrelation time for parameter {:d}:  {:.4e}\n".format((i + 1), AutoCorrTime[i])
                LogLine += "\n\n"
                print (LogLine)
                self.LogFile.write(LogLine.replace("\t", ""))


                ## get autocorrelation time:
                ## taken from
                ## https://python.hotexamples.com/de/examples/chainconsumer/ChainConsumer/add_chain/python-chainconsumer-add_chain-method-examples.html
                def next_pow_two(n):
                    i = 1
                    while i < n:
                        i = i << 1
                    return i

                def autocorr_func_1d(x, norm=True):
                    x = numpy.atleast_1d(x)
                    if len(x.shape) != 1:
                        raise ValueError(
                            "invalid dimensions for 1D autocorrelation function")
                    n = next_pow_two(len(x))

                    # Compute the FFT and then (from that) the auto-correlation function
                    f = numpy.fft.fft(x - numpy.mean(x), n=2 * n)
                    acf = numpy.fft.ifft(f * numpy.conjugate(f))[:len(x)].real
                    acf /= 4 * n

                    # Optionally normalize
                    if norm:
                        acf /= acf[0]

                    return acf

                # Automated windowing procedure following Sokal (1989)
                def auto_window(taus, c):
                    m = numpy.arange(len(taus)) < c * taus
                    if numpy.any(m):
                        return numpy.argmin(m)
                    return len(taus) - 1

                # Following the suggestion from Goodman & Weare (2010)
                def autocorr_gw2010(y, c = 5.0):
                    f = autocorr_func_1d(numpy.mean(y, axis = 0))
                    taus = 2.0 * numpy.cumsum(f) - 1.0
                    window = auto_window(taus, c)
                    return taus[window]

                def autocorr_new(y, c = 5.0):
                    f = numpy.zeros(y.shape[1])
                    for yy in y:
                        f += autocorr_func_1d(yy)
                    f /= len(y)
                    taus = 2.0 * numpy.cumsum(f) - 1.0
                    window = auto_window(taus, c)
                    return taus[window]


                # Compute the estimators for a few different chain lengths
                chain = sampler.get_chain()[:, :, 0].T
                N = numpy.exp(numpy.linspace(numpy.log(100), numpy.log(chain.shape[1]), 10)).astype(int)
                gw2010 = numpy.empty(len(N))
                new = numpy.empty(len(N))
                for i, n in enumerate(N):
                    gw2010[i] = autocorr_gw2010(chain[:, :n])
                    new[i] = autocorr_new(chain[:, :n])


                # Plot the comparisons
                fig = pylab.figure(figsize = (15, 10))
                fig.clear()
                layer = pylab.subplot(1, 1, 1)
                pylab.grid(True)
                layer.loglog(N, gw2010, "o-", label = "G&W 2010")
                layer.loglog(N, new, "o-", label = "new")
                ylim = plt.gca().get_ylim()
                layer.plot(N, N / 50.0, "--k", label=r"$\tau = N/50$")
                layer.set_ylim(ylim)
                layer.set_xlabel("number of samples, $N$")
                layer.set_ylabel(r"$\tau$ estimates")
                layer.legend(fontsize = 14)
                # OutputFileName = "{:s}emcee__autocorrelation_time__call_{:d}.{:s}".format(LocalOutputDirectory, abs(ErrINSCounter), LocalPlotFormat)
                OutputFileName = "{:s}emcee__autocorrelation_time.{:s}".format(LocalOutputDirectory, LocalPlotFormat)
                fig.savefig(OutputFileName, bbox_inches = 'tight')


        ## define return parameter
        OutDict = {}
        OutDict["chi2"] = self.BestFitResultsDict["BestChi2Value"]
        OutDict["params"] = self.BestFitResultsDict["BestParamVec"]
        OutDict["residuals"] = self.BestFitResultsDict["BestParamFunc"]
        return OutDict
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, UltraNest only): prior transform
    def ultranest_prior_transform(self, p0):
        """

    input parameters:
    -----------------

        - p0:                           current parameter vector


    output parameters:
    ------------------

        - params:                       dictionary with lowest chi^2 value and corresponding parameter vector and spectra
        """

        # Debug:
        # print ("p0 = ", p0)
        # print ("p0.shape = ", p0.shape)
        # print ("len(p0.shape) = ", len(p0.shape))


        ## make a copy of current parameter vector
        params = p0.copy()
        if (len(p0.shape) == 1):
            number_param_set = 1
            NumberParameter = len(p0)
            SingleVectorFlag = True
        else:
            number_param_set = len(p0)
            NumberParameter = len(p0[0])
            SingleVectorFlag = False


        ## reproject new parameter vector
        for ParamVecID in range(number_param_set):
            j = (-1)
            for i in range(NumberParameter):
                j += 1
                lo = self.pBLow[i]
                hi = self.pBUp[i]
                if (SingleVectorFlag):
                    params[j] = p0[j] * (hi - lo) + lo
                else:
                    params[ParamVecID][j] = p0[ParamVecID][j] * (hi - lo) + lo

                # Debug:
                # print ("\n\n")
                # print ("ParamVecID = ", ParamVecID)
                # print ("j = ", j)
                # print ("lo = ", lo)
                # print ("hi = ", hi)
                # print ("p0[ParamVecID][j] = ", p0[ParamVecID][j])
                # print ("params[ParamVecID][j] = ", params[ParamVecID][j])


        ## define return parameters
        return params
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, UltraNest only): call UltraNest package
    ## partly taken from https://emcee.readthedocs.io/en/stable/tutorials/line/
    def CallUltraNest(self, LocalFitFunction, p0, addargs):
        """

    input parameters:
    -----------------

        - LocalFitFunction:             objective function

        - p0:                           current parameter vector

        - addargs:                      additional arguments


    output parameters:
    ------------------

        - OutDict:                      dictionary with lowest chi^2 value and corresponding parameter vector and spectra
        """

        # Debug:
        # print ("LocalFitFunction = ", LocalFitFunction)
        # print ("p0 = ", p0)
        # print ("addargs = ", addargs)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## import modified emcee package
        try:
            import ultranest
        except:
            print ("Can not import {:s} package!".format(chr(34) + "ultranest" + chr(34)))


        ## get some parameters for emcee from alg. xml file
        NumberSampler = self.Fit_AlgDict["NumberSampler"]
        NumberBurnInIter = self.Fit_AlgDict["NumberBurnInIter"]
        BackendFileName = self.Fit_AlgDict["BackendFileName"]
        UltraNestDictString = self.Fit_AlgDict["Dictionary"]
        LocalOutputDirectory = self.myXCLASSMapFitJobDir + "UltraNest/"
        os.mkdir(LocalOutputDirectory)
        ndim = len(p0)

        # Debug:
        # print ("NumberSampler = ", NumberSampler)
        # print ("BackendFileName = ", BackendFileName)
        # print ("UltraNestDictString = ", UltraNestDictString)
        # print ("SampleMethod = ", SampleMethod)
        # print ("ndim = ", ndim)
        # print ("self.VectorizedFlag = ", self.VectorizedFlag)
        # sys.exit(0)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set parameters for run_iter
        update_interval_volume_fraction = 0.8
        update_interval_ncall = None
        log_interval = self.Fit_NumIter
        dlogz = 0.5
        dKL = 0.5
        frac_remain = 0.01
        Lepsilon = 0.001
        min_ess = 400
        max_ncalls = None
        max_num_improvement_loops = -1
        cluster_num_live_points = 40
        show_status = False
        viz_callback = 'auto'
        insertion_test_window = 10000
        insertion_test_zscore_threshold = 2
        # region_class = MLFriends
        # widen_before_initial_plateau_num_warn = 10000
        # widen_before_initial_plateau_num_max = 50000


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get parameters from dictionary
        UltraNestDictString = UltraNestDictString.strip()
        if (UltraNestDictString != ""):
            try:
                LocalDict = eval(UltraNestDictString.replace("'", "\""))
            except:
                LocalDict = None
            if (LocalDict is not None):
                for LocalKey in LocalDict.keys():
                    if (LocalKey == "update_interval_volume_fraction"):
                        update_interval_volume_fraction = LocalDict[LocalKey]
                    elif (LocalKey == "update_interval_ncall"):
                        update_interval_ncall = LocalDict[LocalKey]
                    elif (LocalKey == "log_interval"):
                        log_interval = LocalDict[LocalKey]
                    elif (LocalKey == "dlogz"):
                        dlogz = LocalDict[LocalKey]
                    elif (LocalKey == "dKL"):
                        dKL = LocalDict[LocalKey]
                    elif (LocalKey == "frac_remain"):
                        frac_remain = LocalDict[LocalKey]
                    elif (LocalKey == "Lepsilon"):
                        Lepsilon = LocalDict[LocalKey]
                    elif (LocalKey == "min_ess"):
                        min_ess = LocalDict[LocalKey]
                    elif (LocalKey == "max_ncalls"):
                        max_ncalls = LocalDict[LocalKey]
                    elif (LocalKey == "max_num_improvement_loops"):
                        max_num_improvement_loops = LocalDict[LocalKey]
                    elif (LocalKey == "cluster_num_live_points"):
                        cluster_num_live_points = LocalDict[LocalKey]
                    elif (LocalKey == "show_status"):
                        show_status = LocalDict[LocalKey]
                    elif (LocalKey == "viz_callback"):
                        viz_callback = LocalDict[LocalKey]
                    elif (LocalKey == "insertion_test_window"):
                        insertion_test_window = LocalDict[LocalKey]
                    elif (LocalKey == "insertion_test_zscore_threshold"):
                        insertion_test_zscore_threshold = LocalDict[LocalKey]
                    # elif (LocalKey == "region_class"):
                    #     region_class = LocalDict[LocalKey]
                    # elif (LocalKey == "widen_before_initial_plateau_num_warn"):
                    #     widen_before_initial_plateau_num_warn = LocalDict[LocalKey]
                    # elif (LocalKey == "widen_before_initial_plateau_num_max"):
                    #     widen_before_initial_plateau_num_max = LocalDict[LocalKey]

        # Debug:
        # print ("update_interval_volume_fraction = ", update_interval_volume_fraction)
        # print ("update_interval_ncall = ", update_interval_ncall)
        # print ("log_interval = ", log_interval)
        # print ("dlogz = ", dlogz)
        # print ("dKL = ", dKL)
        # print ("frac_remain = ", frac_remain)
        # print ("Lepsilon  = ", Lepsilon )
        # print ("min_ess   = ", min_ess  )
        # print ("max_ncalls = ", max_ncalls)
        # print ("max_num_improvement_loops = ", max_num_improvement_loops)
        # print ("cluster_num_live_points = ", cluster_num_live_points)
        # print ("show_status = ", show_status)
        # print ("viz_callback = ", viz_callback)
        # print ("insertion_test_window = ", insertion_test_window)
        # print ("insertion_test_zscore_threshold = ", insertion_test_zscore_threshold)
        # # print ("widen_before_initial_plateau_num_warn = ", widen_before_initial_plateau_num_warn)
        # # print ("widen_before_initial_plateau_num_max = ", widen_before_initial_plateau_num_max)
        # # print ("region_class = ", region_class)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define names of parameters
        ## param_name = [LocalParamName, LocalMoleculeName, LocalComponentIndex]
        param_names = []
        for ParamIndex in range(ndim):

            LocalParamName = self.FitParameterNames[ParamIndex][2]
            LocalMoleculeName = self.FitParameterNames[ParamIndex][0]
            LocalComponentIndex = self.FitParameterNames[ParamIndex][1] + 1
            if (LocalParamName is not None):
                if (LocalParamName in ["IsoRatio"]):
                    CurrParamName = LocalParamName
                elif (LocalMoleculeName is not None and LocalComponentIndex is not None):
                    LocalLatexParamName = task_myXCLASS.GetLatexParameterName(LocalParamName)
                    CurrParamName = "{:s},\ncomp. {:d}, {:s}".format(chr(34) + LocalMoleculeName + chr(34), \
                                                                        LocalComponentIndex, LocalLatexParamName)
                else:
                    CurrParamName = LocalParamName
            else:
                CurrParamName = "free param. {:d}".format(ParamIndex + 1)
            param_names.append(LocalParamName + "__" + LocalMoleculeName + "__comp_" + str(LocalComponentIndex))

        # Debug:
        # print ("param_names = ", param_names)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## local wrapper for log-likelihood function
        ## hack is required, because UltraNest does not provide passing additional arguments to log-likelihood function
        def logpropwrapper(p0):
            log_probability_list = self.log_probability(p0, LocalFitFunction, addargs)
            log_probability_list = numpy.asarray(log_probability_list)
            return log_probability_list


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize start of UltraNest algorithm


        ## (warm start): restore from a previous run
        BackendFileUsageFlag = False
        if (BackendFileName != ""):
            if (os.path.isfile(BackendFileName)):
                BackendFileUsageFlag = True


                ## print what you do
                print ("\n\n           Resume and reuse an existing run:", flush = True)


                ## define accelerated likelihood and prior transform
                aux_pnames, aux_logL, aux_prior_trans, v = ultranest.integrator.warmstart_from_similar_file(BackendFileName, \
                                                                                                            param_names, \
                                                                                                            logpropwrapper, \
                                                                                                            self.ultranest_prior_transform, \
                                                                                                            vectorized = self.VectorizedFlag)
                ## call ReactiveNestedSampler class
                sampler = ultranest.ReactiveNestedSampler(aux_pnames, \
                                                          aux_logL, \
                                                          aux_prior_trans, \
                                                          vectorized = self.VectorizedFlag, \
                                                          resume = "resume-similar")
                # Debug:
                # print ("\n")
                # print ("param_names = ", param_names)
                # print ("aux_pnames = ", aux_pnames)
                # print ("aux_logL = ", aux_logL)
                # print ("aux_prior_trans = ", aux_prior_trans)
                # print ("v = ", v)
                # print ("sampler = ", sampler)
                # sys.exit(0)


        ## (normal start): call UltraNest package
        if (not BackendFileUsageFlag):
            sampler = ultranest.ReactiveNestedSampler(param_names, \
                                                    logpropwrapper, \
                                                    self.ultranest_prior_transform, \
                                                    vectorized = self.VectorizedFlag, \
                                                    storage_backend = 'hdf5', \
                                                    resume = True, \
                                                    log_dir = LocalOutputDirectory)


        ##--------------------------------------------------------------------------------------------------------------------------------------------
        ## print what you do to log file (and screen)
        OutputLine = "{:10s}".format("Iteration:")
        OutputLine += "   {:>16s}".format("chi^2:")
        OutputLine += "     {:s}".format("Parameter:")
        self.LogFile.write("\t" + OutputLine + "\n")
        self.LogFile.flush()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## start iterations
        LocalIteration = 0
        for results in sampler.run_iter(max_iters = self.Fit_NumIter, \
                                        update_interval_volume_fraction = update_interval_volume_fraction, \
                                        update_interval_ncall = update_interval_ncall, \
                                        log_interval = log_interval, \
                                        dlogz = dlogz, \
                                        dKL = numpy.inf, \
                                        frac_remain = frac_remain, \
                                        Lepsilon = Lepsilon, \
                                        min_ess = NumberSampler, \
                                        max_ncalls = max_ncalls, \
                                        max_num_improvement_loops = max_num_improvement_loops, \
                                        min_num_live_points = NumberSampler, \
                                        cluster_num_live_points = cluster_num_live_points, \
                                        show_status = show_status, \
                                        viz_callback = viz_callback, \
                                        insertion_test_window = insertion_test_window, \
                                        insertion_test_zscore_threshold = insertion_test_zscore_threshold):


            ## write line to log file and screen
            LocalIteration += 1
            OutputLine = "{:10d}".format(LocalIteration)
            OutputLine += "   {:16.8e}".format(self.BestFitResultsDict["BestChi2Value"])
            OutputLine += "   {:s}".format(self.BestFitResultsDict["BestParamVecString"])
            self.LogFile.write("\t" + OutputLine + "\n")


            ## get errors
            PosteriorDict = results["posterior"]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print results
        sampler.print_results()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## plot run (stored as run.pdf: diagnostic plot showing integration progress)
        ## - Visualises how the number of live points, likelihood and posterior weight evolved through the nested sampling run.
        ## - Visualises the evidence integration and its uncertainty.
        print ("\n\n\t Create run plot .. ", end = "", flush = True)
        sampler.plot_run()
        cmdString = "mv {:s}plots/run.pdf {:s}UltraNest__run.pdf".format(LocalOutputDirectory, LocalOutputDirectory)
        os.system(cmdString)
        print ("done!")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## plot trace (stored as trace.pdf: diagnostic plot showing problem structure)
        ## - Visualises how each parameter’s range was reduced as the nested sampling proceeds.
        ## - Color indicates where the bulk of the posterior lies.
        ## - Useful to understand the structure of the inference problem, and which parameters are learned first.
        print ("\n\n\t Create trace plot .. ", end = "", flush = True)
        sampler.plot_trace()
        cmdString = "mv {:s}plots/trace.pdf {:s}UltraNest__trace.pdf".format(LocalOutputDirectory, LocalOutputDirectory)
        os.system(cmdString)
        print ("done!")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## plot corner plot
        from ultranest.plot import cornerplot
        print ("\n\n\t Create ultranest corner-plot .. ", end = "", flush = True)
        sampler.plot_corner()
        cmdString = "mv {:s}plots/corner.pdf {:s}UltraNest__corner.pdf".format(LocalOutputDirectory, LocalOutputDirectory)
        os.system(cmdString)
        print ("done!")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## move backend file (stored in "results/points.hdf5") to user-defined destination
        if (BackendFileName != ""):
            print ("\n\n\t Save backend file .. ", end = "", flush = True)
            cmdString = "mv {:s}chains/weighted_post_untransformed.txt {:s}".format(LocalOutputDirectory, BackendFileName)
            os.system(cmdString)
            print ("done!")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## save debug log
        cmdString = "cp {:s}debug.log {:s}UltraNest__debug.log ".format(LocalOutputDirectory, LocalOutputDirectory)
        os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## clean up job directory
        print ("\n\n\t Clean up job directory .. ", end = "", flush = True)
        cmdString = "rm -rf {:s}chains/ ; ".format(LocalOutputDirectory)
        cmdString += "rm -rf {:s}extra/ ; ".format(LocalOutputDirectory)
        cmdString += "rm -rf {:s}info/ ; ".format(LocalOutputDirectory)
        cmdString += "rm -rf {:s}plots/ ; ".format(LocalOutputDirectory)
        cmdString += "rm -rf {:s}results/ ; ".format(LocalOutputDirectory)
        os.system(cmdString)
        print ("done!\n\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## compute errors and create corner plot


        ## make UltraNest parameters comparable to emcee parameters
        paramnames = results['paramnames']
        data = numpy.array(results['weighted_samples']['points'])
        weights = numpy.array(results['weighted_samples']['weights'])
        cumsumweights = numpy.cumsum(weights)
        mask = cumsumweights > 1e-4
        FinalSamplesOrig = data[mask,:]


        ## hack to get rid of extra parameter
        if ("aux_logweight" in paramnames):
            FinalSamplesOrig = FinalSamplesOrig[:, :-1]


        ## continue
        FinalSamples = numpy.atleast_1d(FinalSamplesOrig)
        if len(FinalSamples.shape) == 1:
            FinalSamples = numpy.atleast_2d(FinalSamples)
        else:
            assert len(FinalSamples.shape) == 2, "The input sample array must be 1- or 2-D."
            FinalSamples = FinalSamples.T


        ## make corner plot
        NameAlg = "ultranest"
        bestparam = self.BestFitResultsDict["BestParamVec"]
        self.make_corner_plot(NameAlg, FinalSamples, FinalSamples, FinalSamplesOrig, ndim, bestparam, \
                              LocalOutputDirectory, UltraNestDict = PosteriorDict)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define return parameter
        OutDict = {}
        OutDict["chi2"] = self.BestFitResultsDict["BestChi2Value"]
        OutDict["params"] = self.BestFitResultsDict["BestParamVec"]
        OutDict["residuals"] = self.BestFitResultsDict["BestParamFunc"]
        return OutDict
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, PyGAD only): fitness function for PyGAD
    ## partly taken from https://stackoverflow.com/questions/72647791/genetic-algorithm-with-pygad
    def pygad_fitness_func(self, ga_instance, solution, solution_idx):
        """

    input parameters:
    -----------------

        - ga_instance:                  GA instance


    output parameters:
    ------------------

        - fitness:                      new fitness
        """

        # Debug:
        # print ("ga_instance = ", ga_instance)
        # print ("solution = ", solution)
        # print ("solution_idx = ", solution_idx)


        ## compute log-probability function
        fitness = self.pygad_LocalFitFunction(solution, self.pygad_args)

        # Debug:
        # print ("fitness = ", fitness)


        ## we're done
        return fitness
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, PyGAD only): helper function for PyGAD
    ## partly taken from https://stackoverflow.com/questions/72647791/genetic-algorithm-with-pygad
    def pygad_on_generation(self, ga_instance):
        """

    input parameters:
    -----------------

        - ga_instance:                  GA instance


    output parameters:
    ------------------

        - OutDict:                      dictionary with lowest chi^2 value and corresponding parameter vector and spectra
        """

        # Debug:
        # print ("ga_instance = ", ga_instance)


        ## print some informations
        # print("Generation = {generation}".format(generation = ga_instance.generations_completed))
        # print("Fitness    = {fitness}".format(fitness = ga_instance.best_solution(pop_fitness = ga_instance.last_generation_fitness)[1]))
        # print("Change     = {change}".format(change = ga_instance.best_solution(pop_fitness = ga_instance.last_generation_fitness)[1] - self.pygad_last_fitness))


        ## get last fitness
        self.pygad_last_fitness = ga_instance.best_solution(pop_fitness = ga_instance.last_generation_fitness)[1]


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (non-MapFit, PyGAD only): call PyGAD package
    ## partly taken from https://stackoverflow.com/questions/72647791/genetic-algorithm-with-pygad
    def CallPyGAD(self, LocalFitFunction, p0, addargs):
        """

    input parameters:
    -----------------

        - LocalFitFunction:             objective function

        - p0:                           current parameter vector

        - addargs:                      additional arguments


    output parameters:
    ------------------

        - OutDict:                      dictionary with lowest chi^2 value and corresponding parameter vector and spectra
        """

        # Debug:
        # print ("LocalFitFunction = ", LocalFitFunction)
        # print ("p0 = ", p0)
        # print ("addargs = ", addargs)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## import modified emcee package
        try:
            import pygad
        except:
            print ("Can not import {:s} package!".format(chr(34) + "PyGad" + chr(34)))
            sys.exit(0)


        ## get some parameters for emcee from alg. xml file
        NumberSampler = self.Fit_AlgDict["NumberSampler"]
        LocalPlotFormat = self.Fit_AlgDict["PlotFileFormat"]
        LocalOutputDirectory = self.myXCLASSMapFitJobDir + "PyGAD/"
        os.mkdir(LocalOutputDirectory)
        ndim = len(p0)

        # Debug:
        # print ("NumberSampler = ", NumberSampler)
        # print ("LocalPlotFormat = ", LocalPlotFormat)
        # print ("ndim = ", ndim)
        # sys.exit(0)


        ## define ranges for each parameter
        gene_space = []
        for i in range(len(p0)):
            LocalDict = {}
            LocalDict["low"] = self.pBLow[i]
            LocalDict["high"] = self.pBUp[i]
            gene_space.append(LocalDict)

        # Debug:
        # print ("gene_space = ", gene_space)


        ## define instance for pygad
        self.pygad_last_fitness = 0
        self.pygad_LocalFitFunction = LocalFitFunction
        self.pygad_args = addargs
        OldFit_NumProc = self.Fit_NumProc
        self.Fit_NumProc = 1
        # self.ParallelMethod = "threading"
        ga_instance = pygad.GA(num_generations = self.Fit_NumIter,
                               num_parents_mating = 5,
                               sol_per_pop = NumberSampler,
                               num_genes = ndim,
                               gene_space = gene_space,
                               mutation_by_replacement = True,
                               fitness_func = self.pygad_fitness_func,
                               parallel_processing = 1,
                               on_generation = self.pygad_on_generation)

                            #   self.Fit_NumProc,
                            #   parallel_processing = ["process", self.Fit_NumProc],


        ## start genetic algorithm
        ga_instance.run()


        ## plot results
        ga_instance.plot_fitness()


        ## get results
        solution, solution_fitness, solution_idx = ga_instance.best_solution(ga_instance.last_generation_fitness)


        ## print results
        # print("Solution", solution)
        # print("Fitness value of the best solution = {solution_fitness}".format(solution_fitness=solution_fitness))


        ## restore old setting
        self.Fit_NumProc = OldFit_NumProc
        # self.ParallelMethod = "process"

        # Debug:
        # print ("self.BestFitResultsDict = ", self.BestFitResultsDict)


        ## define return parameter
        OutDict = {}
        OutDict["chi2"] = self.BestFitResultsDict["BestChi2Value"]
        OutDict["params"] = self.BestFitResultsDict["BestParamVec"]
        OutDict["residuals"] = self.BestFitResultsDict["BestParamFunc"]
        return OutDict
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## no MapFit: fit entire cube / spectra at once instead of fitting each pixel individually
    def EntireFITSCubeFit(self, LocalFitFunction, p0, OptAlgParamDict):
        """

    input parameters:
    -----------------

        - LocalFitFunction:             ojective function

        - p0:                           fit parameters

        - OptAlgParamDict:              dictionary with parameters for optimization algorithms


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("LocalFitFunction = ", LocalFitFunction)
        # print ("p0 = ", p0)
        # print ("OptAlgParamDict = ", OptAlgParamDict)


        ##--------------------------------------------------------------------------------------------------------------------------------------------
        ## call optimization algorithm
        OutputParameter = self.CallOptimizationAlgorithms(LocalFitFunction, p0, OptAlgParamDict)
        if (self.ParallelMethod in ["dask", "sc"]):
            LocalBigCubeSpectralAxis = OutputParameter
        else:
            OutDict = OutputParameter


        ##--------------------------------------------------------------------------------------------------------------------------------------------
        ## print what you do
        if (self.PrintFitStatusFlag):
            print ("\n\nCollect results and write to file ..", end = "", flush = True)


        ##--------------------------------------------------------------------------------------------------------------------------------------------
        ## collect results
        if (self.ParallelMethod in ["dask", "sc"]):
            print ("\n\nDask / sc parallelization not implemented yet !!", flush = True)
        else:
            if (len(OutDict) > 0):

                # Debug:
                # print ("OutDict = ", OutDict)


                ## get total number of frequency points
                NumFreqPointsList = [len(x) for x in self.ListOfSpectralAxis]
                TotalFreqPoints = sum(NumFreqPointsList)


                ## get final fit parameters
                if (self.NoFITSCubeFlag):
                    self.FitParameterMaps[0, 0, :] = OutDict['params']


                ## get full results of optimization algorithm
                AllPixelSpectra = OutDict['residuals']


                ## get list of pixel indices
                args = OptAlgParamDict["args"]
                IndexMap = args[0]


                ## start loop over all pixels
                for SingleMapIndexID, SingleMapIndex in enumerate(IndexMap):


                    ## write status
                    if (self.PrintFitStatusFlag):
                        print ("\rCollect results and write to file ({:d}/{:d}) done!".format(SingleMapIndex + 1, len(IndexMap)), \
                                end = "", flush = True)


                    ## get indices of current pixel
                    # ypos = int(SingleMapIndex / self.NumXPixels)
                    # xpos = SingleMapIndex - (ypos * self.NumXPixels)

                    ypos = self.ListOfSelectedPixels[SingleMapIndex][0]
                    xpos = self.ListOfSelectedPixels[SingleMapIndex][1]

                    i1 = SingleMapIndexID * TotalFreqPoints
                    i2 = (SingleMapIndexID + 1) * TotalFreqPoints
                    LocalSpectra = AllPixelSpectra[i1:i2]
                    LocalChi2 = 0
                    TotalWeightingArray = numpy.ones((TotalFreqPoints, ))
                    for LocalRangeID, ObsXMLParameterList in enumerate(self.RangeParam):
                        if (self.DataFileFormat == "fits"):
                            LocalSubCube = self.ListOfCubes[LocalRangeID]
                            LocalObsIntArray = LocalSubCube[:, ypos, xpos].value
                            FreqMinIndex = self.LimitIndices[LocalRangeID][0]
                            FreqMaxIndex = self.LimitIndices[LocalRangeID][1]
                            LocalModelSpectrum = LocalSpectra[FreqMinIndex:FreqMaxIndex]
                            LocalWeightingArray = numpy.ones((len(LocalObsIntArray), ))
                        elif (self.DataFileFormat == "ascii"):
                            LocalASCIIData = self.ListOfCubes[LocalRangeID]
                            LocalObsIntArray = LocalASCIIData[:, 1]
                            ErrorY = ObsXMLParameterList['ErrorY']                          ## required for weighting
                            if (ErrorY):                                                    ## add weighting
                                LocalWeightingArray = LocalASCIIData[:, 2]
                            else:
                                LocalWeightingArray = numpy.ones((len(LocalObsIntArray), ))
                            FreqMinIndex = self.LimitIndices[LocalRangeID][0]
                            FreqMaxIndex = self.LimitIndices[LocalRangeID][1]
                            LocalModelSpectrum = LocalSpectra[FreqMinIndex:FreqMaxIndex]

                        # Debug:
                        # print ("\nLocalRangeID = ", LocalRangeID)
                        # print ("LocalObsIntArray.shape = ", LocalObsIntArray.shape)
                        # print ("LocalObsIntArray = ", LocalObsIntArray)
                        # print ("i1, i2 = ", i1, i2)
                        # print ("AllPixelSpectra[:].shape = ", AllPixelSpectra[:].shape)
                        # print ("FreqMinIndex, FreqMaxIndex = ", FreqMinIndex, FreqMaxIndex)
                        # print ("LocalModelSpectrum[:].shape = ", LocalModelSpectrum[:].shape)
                        # print ("LocalSpectra[:].shape = ", LocalSpectra[:].shape)
                        # print ("LocalModelSpectrum[:] = ", LocalModelSpectrum[:])


                        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        ## store spectra


                        ## flip model spectrum?
                        ReverseFlag = ObsXMLParameterList['ReverseFlag']
                        if (ReverseFlag and self.FullFitFlag):                              ## (only for full fit method): for neg. freq. step sizes
                            LocalModelSpectrum = numpy.flip(LocalModelSpectrum)             ## flip model spectrum
                            LocalWeightingArray = numpy.flip(LocalWeightingArray)           ## flip model spectrum


                        ## store model spectrum for current position
                        if (self.Fit_LSFlag == 0):
                            self.SyntheticSpectraCubeList[LocalRangeID][:, ypos, xpos] = (-1.0) * (LocalModelSpectrum[:] * LocalWeightingArray[:] - LocalObsIntArray[:])
                            if (self.Fit_Chi2):
                                self.Chi2SpectraCubeList[LocalRangeID][:, ypos, xpos] = (LocalModelSpectrum[:])**2
                            LocalChi2 += numpy.sum(LocalModelSpectrum[:]**2)
                        else:
                            self.SyntheticSpectraCubeList[LocalRangeID][:, ypos, xpos] = LocalModelSpectrum[:]
                            if (self.Fit_Chi2):
                                self.Chi2SpectraCubeList[LocalRangeID][:, ypos, xpos] = (LocalModelSpectrum[:] - LocalObsIntArray)**2
                            LocalChi2 += numpy.sum((LocalModelSpectrum[:] - LocalObsIntArray)**2)


                    ## store chi^2 value
                    self.Chi2Map[ypos, xpos] = LocalChi2

                # Debug:
                # print ("LocalParams = ", LocalParams)


            ## write status
            if (self.PrintFitStatusFlag):
                print ("\rCollect results and write to file done!                                                ", flush = True)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start worker
    def NonMapFitWorkers(self, p0, *args):
        """

    input parameters:
    -----------------

        - p0:                           fit parameters

        - args:                         list of indices


    output parameters:
    ------------------

        - LocalSpectrum:                return value for optimization algorithm
        """

        # Debug:
        # print ("\np0 = ", p0)
        # print ("\nlen(p0) = ", len(p0))
        # print ("args = ", args)


        ## initialize return parameter
        LocalSpectrum = None
        LocalResidual = None


        ## get current parameters
        IndexRange = args[0]

        # Debug:
        # print ("IndexRange = ", IndexRange)


        ## define list of parameter vectors
        AllParamVector = []
        GradientStepsVector = []
        if (self.VectorizedFlag):
            AllParamVector = copy.deepcopy(p0)
        else:
            AllParamVector.append(copy.deepcopy(p0))


        ##================================================================================================================================================
        ## Gradient: compute parameter values for gradient
        if (self.FitGradientFlag and (not self.Fit_FuncEvalFlag)):
            for iID, xi in enumerate(p0):
                LocalParamVector = copy.deepcopy(p0)


                ## scipy implementation:
                # ## by default we use rel_step
                # if abs_step is None:
                #     h = _compute_absolute_step(rel_step, x0, f0, method)
                # else:
                #     # user specifies an absolute step
                #     sign_x0 = float(x0 >= 0) * 2 - 1
                #     h = abs_step

                # # cannot have a zero step. This might happen if x0 is very large
                # # or small. In which case fall back to relative step.
                # dx = ((x0 + h) - x0)
                # h = numpy.where(dx == 0,
                #             _eps_for_method(x0.dtype, f0.dtype, method) *
                #             sign_x0 * numpy.maximum(1.0, numpy.abs(x0)),
                #             h)


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## compute parameter for gradient


                ## compute step size and current parameter value for gradient
                if (abs(xi) <= self.eps):
                    steph = self.Fit_Variation                                              ## variation of the parameter in percent/100
                else:
                    steph = abs(xi * self.Fit_Variation)                                    ## define stepsize for foating point numbers
                dx = steph
                value = (xi + steph)


                ## make sure that gradient point is within the parameter limits
                if (value <  self.pBLow[iID]):
                    value = (xi - steph)
                    if (value >  self.pBUp[iID]):
                        value = xi
                elif (value >  self.pBUp[iID]):
                    value = (xi - steph)
                    if (value <  self.pBLow[iID]):
                        value = xi


                ## store parameter value, step size and new parameter vector
                LocalParamVector[iID] = value
                GradientStepsVector.append(dx)
                AllParamVector.append(LocalParamVector)

            # Debug:
            # print ("AllParamVector = ", AllParamVector)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## prepare start of workers
        NumParamVec = len(AllParamVector)
        self.GradientVector = [None for x in range(NumParamVec - 1)]
        self.return_dict = self.manager.dict()
        TotalNumSpectra = NumParamVec * self.TotalNumPixel

        # Debug:
        # print ("NumParamVec = ", NumParamVec)
        # print ("self.TotalNumPixel = ", self.TotalNumPixel)
        # print ("TotalNumSpectra = ", TotalNumSpectra)
        # print ("self.Fit_NumProc = ", self.Fit_NumProc)
        # print ("self.FitGradientFlag = ", self.FitGradientFlag)
        # print ("self.VectorizedFlag = ", self.VectorizedFlag)
        # print ("AllParamVector.shape() = ", numpy.asarray(AllParamVector).shape)
        # print ("AllParamVector = ", AllParamVector)
        # sys.exit(0)


        LocalChi2Value = 0.0
        ListOfChi2Values = []
        for SpectrumRangeID in range(0, TotalNumSpectra, self.Fit_NumProc):


            ## print what you do
            if (not self.Fit_Algorithm in ["mcmc", "emcee", "ultranest"]):
                print ("\rCompute spectrum ({:d}/{:d}) .. {:>20s}".format((SpectrumRangeID + 1), TotalNumSpectra, " "), \
                      end = "", flush = True)


            ## start processes
            thread_list = []
            for ProcessorID in range(self.Fit_NumProc):
                TotalCounter = (SpectrumRangeID + ProcessorID)


                ## start the threads
                PixelID = numpy.mod(TotalCounter, self.TotalNumPixel)
                if (PixelID < len(IndexRange)):


                    ## initialize local thread dictionary
                    LocalThreadDict = {}


                    ## get parameter vector and corresponding ID
                    ParamVectorID = int(TotalCounter / self.TotalNumPixel)
                    if (ParamVectorID < NumParamVec):
                        ParamVector = AllParamVector[ParamVectorID]
                        LocalThreadDict["ParamVector"] = ParamVector


                        ## set local gradient flag
                        if (ParamVectorID == 0):
                            LocalGradientFlag = False
                        else:
                            LocalGradientFlag = True
                        LocalThreadDict["LocalGradientFlag"] = LocalGradientFlag


                        ## call process
                        LocalThreadDict["IndexRange"] = IndexRange[PixelID]
                        t = multiprocessing.Process(name = str(ProcessorID), target = self.PixelFitFunction, args = (LocalThreadDict, ))


                        ## Sticks the thread in a list so that it remains accessible
                        thread_list.append(t)

                        # Debug:
                        # print ("\n\n")
                        # print ("SpectrumRangeID = ", SpectrumRangeID)
                        # print ("ProcessorID = ", ProcessorID)
                        # print ("PixelID = ", PixelID)
                        # print ("ParamVectorID = ", ParamVectorID)
                        # print ("self.Fit_NumProc = ", self.Fit_NumProc)
                        # print ("self.TotalNumPixel = ", self.TotalNumPixel)
                        # print ("TotalNumSpectra = ", TotalNumSpectra)
                        # print ("NumParamVec = ", NumParamVec)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## Start all threads (workers)
            for thread in thread_list:
                thread.start()


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## This blocks the calling thread until the thread whose join() method is called is terminated.
            ## From http://docs.python.org/2/library/threading.html#thread-objects
            for thread in thread_list:
                thread.join()


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get return values
            for ProcessorID in range(self.Fit_NumProc):
                TotalCounter = (SpectrumRangeID + ProcessorID)
                PixelID = numpy.mod(TotalCounter, self.TotalNumPixel)
                if (PixelID < len(IndexRange)):


                    ## get parameter vector ID
                    ParamVectorID = int(TotalCounter / self.TotalNumPixel)
                    if (ParamVectorID < NumParamVec):
                        if (self.ParallelMethod in ["process"]):
                            LocalProcessorID = ProcessorID
                        elif (self.ParallelMethod in ["threading"]):
                            thread = current_thread()
                            ThreadName = thread.name
                            ThreadName = ThreadName.replace("ThreadPoolExecutor-0_", "")
                            LocalProcessorID = int(ThreadName)
                        try:
                            LocalDict = self.return_dict[LocalProcessorID]
                        except:
                            print ("\n\nLocalProcessorID = ", LocalProcessorID)
                            print ("PixelID = ", PixelID)
                            print ("len(IndexRange) = ", len(IndexRange))
                            print ("SpectrumRangeID = ", SpectrumRangeID)
                            print ("TotalCounter = ", TotalCounter)
                            print ("self.TotalNumPixel = ", self.TotalNumPixel)
                            print ("self.return_dict = ", self.return_dict)
                            sys.exit(0)
                        if (len(LocalDict) > 0):
                            LocalChi2Value += LocalDict['chi2']
                            LocalSpectrumPart = LocalDict['spectrum']
                            LocalChi2Part = LocalDict['residual']

                            # Debug:
                            # print ("PixelID, LocalDict = ", PixelID, LocalDict)
                            # print ("\nPixelID, LocalChi2Value = ", PixelID, LocalChi2Value)
                            # print ("LocalSpectrumPart = ", LocalSpectrumPart)
                            # print ("LocalSpectrumPart.shape = ", LocalSpectrumPart.shape)
                            # print ("numpy.nanmin(LocalSpectrumPart) = ", numpy.nanmin(LocalSpectrumPart))
                            # print ("numpy.nanmax(LocalSpectrumPart) = ", numpy.nanmax(LocalSpectrumPart))
                            # print ("LocalChi2Part = ", LocalChi2Part)
                            # print ("LocalChi2Part.shape = ", LocalChi2Part.shape)


                            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                            ## compute parameter values for each parameter vector
                            if (self.VectorizedFlag):

                                # Debug:
                                # print ("TotalCounter, TotalNumSpectra = ", TotalCounter, TotalNumSpectra)


                                ## determine chi^2 value and create spectrum
                                if (LocalSpectrum is None):
                                    LocalSpectrum = copy.deepcopy(LocalSpectrumPart)
                                else:
                                    LocalSpectrum = numpy.concatenate((LocalSpectrum, LocalSpectrumPart))

                                # Debug:
                                # print (PixelID + 1, len(IndexRange))
                                # print ("len(LocalSpectrum) = ", len(LocalSpectrum))


                                ##************************************************************************************************************************
                                ## if new parameter vector is calculated, check, if last parameter vector has the lowest chi^2 value
                                if ((PixelID + 1) == len(IndexRange)):
                                    Localp0 = AllParamVector[ParamVectorID]


                                    ## compute reduced chi^2 value
                                    if (self.Fit_AlgDict["Chi2RenormalizedFlag"]):
                                        # LocalChi2Value = abs(len(LocalSpectrum) - len(Localp0)) * LocalChi2Value / len(LocalSpectrum)
                                        if (len(Localp0) > 0):
                                            LocalChi2Value /= len(Localp0)


                                    ## create current parameter vector
                                    ParamVecString = ", ".join("{:16.8e}".format(pi) for pi in Localp0)

                                    # Debug:
                                    # print ("\r{:18d} {:18.10e}   {:s}".format(ParamVectorID + 1, LocalChi2Value, ParamVecString))


                                    ## store best parameter vector and chi^2 value
                                    if (len(self.BestFitResultsDict.keys()) == 0):
                                        self.BestFitResultsDict["BestChi2Value"] = LocalChi2Value
                                        self.BestFitResultsDict["BestParamVec"] = Localp0
                                        self.BestFitResultsDict["BestParamVecString"] = ParamVecString
                                        self.BestFitResultsDict["BestParamFunc"] = LocalSpectrum
                                    else:
                                        BestChi2Value = self.BestFitResultsDict["BestChi2Value"]
                                        if (LocalChi2Value < BestChi2Value):
                                            self.BestFitResultsDict["BestChi2Value"] = LocalChi2Value
                                            self.BestFitResultsDict["BestParamVec"] = Localp0
                                            self.BestFitResultsDict["BestParamVecString"] = ParamVecString
                                            self.BestFitResultsDict["BestParamFunc"] = LocalSpectrum

                                    # Debug:
                                    # print ("LocalSpectrum = ", LocalSpectrum)
                                    # print ("len(LocalSpectrum) = ", len(LocalSpectrum))
                                    # print ("(TotalCounter + 1), TotalNumSpectra = ", (TotalCounter + 1), TotalNumSpectra)


                                    ## write line to log file and screen
                                    if ((not self.UseCallbackFuncFlag) and (not self.Fit_Algorithm in ["mcmc", "emcee", "ultranest"])):
                                        LogLine = "{:18d} {:18.10e}   {:s}".format(ParamVectorID + 1, \
                                                                                   self.BestFitResultsDict["BestChi2Value"], \
                                                                                   self.BestFitResultsDict["BestParamVecString"])
                                        self.LogFile.write(LogLine + "\n")
                                        print ("\r{:s}".format(LogLine))


                                    ## write chi2 and parameter vector to log.chi2 file
                                    self.FuncCallCounter += 1
                                    if (self.Fit_Chi2LogFlag):
                                        OutputLine = "{:10d}   {:16.8e}   ".format(self.FuncCallCounter, LocalChi2Value)
                                        OutputLine += ParamVecString
                                        self.LogChi2File.write(OutputLine + "\n")


                                    ## if call of likelihood function was vectorized, store all spectra
                                    if (self.Fit_Algorithm in ["mcmc", "emcee", "ultranest"]):
                                        if (numpy.all((Localp0 >= self.pBLow) & (Localp0 <= self.pBUp))):
                                            if (self.Fit_Algorithm in ["mcmc", "emcee", "ultranest"]):
                                                LocalChi2Value = (-0.5) * LocalChi2Value
                                        else:
                                            LocalChi2Value = numpy.inf
                                        ListOfChi2Values.append(LocalChi2Value)


                                    ## reset variables
                                    if (TotalCounter + 1 < TotalNumSpectra):
                                        LocalChi2Value = 0.0
                                        LocalSpectrum = None


                            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                            ## store results for initial parameter vector
                            elif (ParamVectorID == 0):


                                ## construct final spectrum
                                if (LocalSpectrum is None):
                                    LocalSpectrum = copy.deepcopy(LocalSpectrumPart)
                                else:
                                    LocalSpectrum = numpy.concatenate((LocalSpectrum, LocalSpectrumPart))


                                ## construct final spectrum
                                if (self.Fit_LSFlag == 0):
                                    if (LocalResidual is None):
                                        LocalResidual = copy.deepcopy(LocalChi2Part)
                                    else:
                                        LocalResidual = numpy.concatenate((LocalResidual, LocalChi2Part))


                            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                            ## for the gradient calculation, store spectrum for ith gradient entry
                            elif (self.FitGradientFlag):
                                if (self.Fit_LSFlag == 0):
                                    if (self.GradientVector[ParamVectorID - 1] is None):
                                        self.GradientVector[ParamVectorID - 1] = copy.deepcopy(LocalChi2Part)
                                    else:
                                        self.GradientVector[ParamVectorID - 1] = numpy.concatenate((self.GradientVector[ParamVectorID - 1], \
                                                                                                    LocalChi2Part))
                                elif (self.Fit_LSFlag == 1):
                                    if (self.GradientVector[ParamVectorID - 1] is None):
                                        self.GradientVector[ParamVectorID - 1] = copy.deepcopy(LocalSpectrumPart)
                                    else:
                                        self.GradientVector[ParamVectorID - 1] = numpy.concatenate((self.GradientVector[ParamVectorID - 1], \
                                                                                                    LocalSpectrumPart))
                                else:
                                    xx = LocalChi2Value
                                    if (self.Fit_AlgDict["Chi2RenormalizedFlag"]):
                                        # xx = abs(len(LocalSpectrum) - len(p0)) * xx / len(LocalSpectrum)
                                        if (len(p0) > 0):
                                            xx /= len(p0)
                                    self.GradientVector[ParamVectorID - 1] = xx

                                # Debug:
                                # sys.exit(0)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## compute gradient
        if (self.FitGradientFlag and not self.Fit_FuncEvalFlag):
            for ParamVectorID in range(1, NumParamVec):

                # Debug:
                # print ("self.Fit_LSFlag = ", self.Fit_LSFlag)
                # print ("ParamVectorID = ", ParamVectorID)
                # print ("dx = ", GradientStepsVector[ParamVectorID])
                # print ("\nself.GradientVector[ParamVectorID - 1] = ", self.GradientVector[ParamVectorID - 1])
                # print ("LocalChi2Value = ", LocalChi2Value)
                # print ("LocalSpectrum = ", LocalSpectrum)


                ## print what you do
                print ("\rCompute gradient ({:d}/{:d}) ..                                   ".format(ParamVectorID, NumParamVec), end = "", flush = True)


                ## compute gradient
                dx = GradientStepsVector[ParamVectorID - 1]
                if (self.Fit_LSFlag == 0):
                    self.GradientVector[ParamVectorID - 1] = (self.GradientVector[ParamVectorID - 1] - LocalResidual) / dx
                elif (self.Fit_LSFlag == 1):
                    self.GradientVector[ParamVectorID - 1] = (self.GradientVector[ParamVectorID - 1] - LocalSpectrum) / dx
                else:
                    xx = LocalChi2Value
                    if (self.Fit_AlgDict["Chi2RenormalizedFlag"]):
                        # xx = abs(len(LocalSpectrum) - len(p0)) * xx / len(LocalSpectrum)
                        if (len(p0) > 0):
                            xx /= len(p0)
                    self.GradientVector[ParamVectorID - 1] = (xx - self.GradientVector[ParamVectorID - 1]) / dx

            # Debug:
            # print ("GradientStepsVector = ", GradientStepsVector)
            # print ("self.GradientVector = ", self.GradientVector)


        ## done
        if (not self.Fit_Algorithm in ["mcmc", "emcee", "ultranest"]):
            print ("\r                                                                      \r", end = "", flush = True)

        # Debug:
        # print ("p0 = ", p0)
        # print ("LocalSpectrum = ", LocalSpectrum)
        # print ("numpy.nanmin(LocalSpectrum) = ", numpy.nanmin(LocalSpectrum))
        # print ("numpy.nanmax(LocalSpectrum) = ", numpy.nanmax(LocalSpectrum))
        # print ("self.GradientVector = ", self.GradientVector)
        # sys.exit(0)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print status


        ## print status (for vectorized function calls)
        if (self.VectorizedFlag):
            if ((not self.Fit_Algorithm in ["mcmc", "emcee"]) \
                and (not self.Fit_FuncEvalFlag)):


                ## increase counter for iteration
                self.CurrentFitIteration += 1


                ## write line to log file and screen
                OutputLine = "{:10d}".format(self.CurrentFitIteration)
                OutputLine += "   {:16.8e}".format(self.BestFitResultsDict["BestChi2Value"])
                OutputLine += "   {:s}".format(self.BestFitResultsDict["BestParamVecString"])
                # self.LogFile.write(OutputLine + "\n")


        ## print status
        else:


            ## create current parameter vector
            ParamVecString = ", ".join("{:16.8e}".format(pi) for pi in p0)


            ## compute reduced chi^2 value
            if (self.Fit_AlgDict["Chi2RenormalizedFlag"]):
                # LocalChi2Value = abs(len(LocalSpectrum) - len(p0)) * LocalChi2Value / len(LocalSpectrum)
                if (len(p0) > 0):
                    LocalChi2Value /= len(p0)


            ## get best (lowest) chi^2 value and corresponding parameter vector
            if (len(self.BestFitResultsDict.keys()) == 0):
                self.BestFitResultsDict["BestChi2Value"] = LocalChi2Value
                self.BestFitResultsDict["BestParamVec"] = p0
                self.BestFitResultsDict["BestParamVecString"] = ParamVecString
                self.BestFitResultsDict["BestParamFunc"] = LocalSpectrum
            else:
                BestChi2Value = self.BestFitResultsDict["BestChi2Value"]
                if (BestChi2Value <= LocalChi2Value):
                    LocalChi2Value = BestChi2Value
                    ParamVecString = self.BestFitResultsDict["BestParamVecString"]
                else:
                    self.BestFitResultsDict["BestChi2Value"] = LocalChi2Value
                    self.BestFitResultsDict["BestParamVec"] = p0
                    self.BestFitResultsDict["BestParamVecString"] = ParamVecString
                    self.BestFitResultsDict["BestParamFunc"] = LocalSpectrum


            ## print results
            if (not self.Fit_FuncEvalFlag):


                ## write line to log file and screen
                if ((not self.UseCallbackFuncFlag) and (not self.Fit_Algorithm in ["levenberg-marquardt"])):
                    self.CurrentFitIteration += 1
                    OutputLine = "{:10d}   {:16.8e}   ".format(self.CurrentFitIteration, LocalChi2Value)
                    OutputLine += ParamVecString
                    print ("\t" + OutputLine)
                    self.LogFile.write(OutputLine + "\n")


                ## write chi2 and parameter vector to log.chi2 file
                self.FuncCallCounter += 1
                if (self.Fit_Chi2LogFlag):
                    OutputLine = "{:10d}   {:16.8e}   ".format(self.FuncCallCounter, LocalChi2Value)
                    OutputLine += ParamVecString
                    self.LogChi2File.write(OutputLine + "\n")

        # Debug:
        # print ("\n\nListOfChi2Values = ", ListOfChi2Values)
        # print ("LocalChi2Value = ", LocalChi2Value)
        # print ("LocalSpectrum = ", LocalSpectrum)
        # print ("LocalSpectrum.shape = ", LocalSpectrum.shape)
        # print ("LocalResidual = ", LocalResidual)
        # print ("LocalResidual.shape = ", LocalResidual.shape)
        # print ("numpy.all(numpy.isfinite(LocalSpectrum)) = ", numpy.all(numpy.isfinite(LocalSpectrum)))
        # print ("numpy.all(numpy.isfinite(LocalResidual)) = ", numpy.all(numpy.isfinite(LocalResidual)))


        ## we're done
        if (self.VectorizedFlag):
            return ListOfChi2Values
        elif (self.Fit_LSFlag == 0):                                                    ## for least square algorithms
            if (not (numpy.all((p0 >= self.pBLow) & (p0 <= self.pBUp)))):
                LocalResidual = LocalResidual * numpy.inf
            return LocalResidual
        elif (self.Fit_LSFlag == 1):                                                    ## for blank application
            if (not (numpy.all((p0 >= self.pBLow) & (p0 <= self.pBUp)))):
                LocalSpectrum = LocalSpectrum * numpy.inf
            return LocalSpectrum
        else:                                                                           ## for global optimization algorithms
            if (not (numpy.all((p0 >= self.pBLow) & (p0 <= self.pBUp)))):
                LocalChi2Value = LocalChi2Value * numpy.inf
            return LocalChi2Value
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## compute the Jacobian matrix for optimization algorithms which require a gradient
    def GradientJAC(self, dummyp0, *args):
        """

    input parameters:
    -----------------

        - dummyp0:                  the scipy functions require a function for the Jacobian matrix which has the current
                                    parameter vector as input, but we compute the gradient in the initial function call
                                    and use this function for returning the already calculated gradient function only


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("dummyp0 = ", dummyp0)
        # print ("args = ", args)
        # print ("numpy.nanmin(self.GradientVector) = ", numpy.nanmin(self.GradientVector))
        # print ("numpy.nanmax(self.GradientVector) = ", numpy.nanmax(self.GradientVector))
        # print ("self.GradientVector = ", self.GradientVector)
        # print ("numpy.all(numpy.isfinite(self.GradientVector)) = ", numpy.all(numpy.isfinite(self.GradientVector)))


        if (self.Fit_Algorithm in ["trf", "dogbox", "lm", "levenberg-marquardt", "cg", "bfgs", "l-bfgs-b", "tnc", "slsqp"]):
            return numpy.transpose(self.GradientVector)
        else:
            return self.GradientVector


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start worker using multiprocessing package
    def StartWorker(self, ObjFunction, IndexRange, method = "process"):
        """

    input parameters:
    -----------------

        - ObjFunction:                  objective function

        - IndexRange:                   list of indices

        - method:                       (optional) use pool or process method for parallelization ("pool", "process", "dask", "sc")
                                        (default: "pool")


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("ObjFunction = ", ObjFunction)
        # print ("IndexRange = ", IndexRange)
        # print ("method = ", method)
        # print ("self.Fit_NumProc = ", self.Fit_NumProc)


        ## save start time
        start = time.time()


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## for debugging only
        if (method in ["dbg", "debug"]):
            SelectedPixel = None


            ## print warning message
            print ("\n\nDEBUGGING MODE !!!!\n\n")


            ## select certain pixel
            ## in order to select a certain pixel at position [x,y] use
            # SelectedPixel = (y * len(y-pixels)) + x


            ## call model function for all selected pixels
            results = []
            if (SelectedPixel is None):
                for i in IndexRange:
                    out = ObjFunction(i)
                    results.append(out)

            else:
                for i in range(SelectedPixel, SelectedPixel + 1):
                    out = ObjFunction(i)
                    results.append(out)


            ## STOP !!!!
            sys.exit(0)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## start pool of workers
        ## for problem see https://stackoverflow.com/questions/47776486/python-struct-error-i-format-requires-2147483648-number-2147483647?rq=1
        elif (method in ["pool"]):
            pool = multiprocessing.Pool(processes = self.Fit_NumProc)
            results = pool.map(ObjFunction, IndexRange)
            pool.close()
            pool.join()


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## start process of workers
        ## for problem see https://stackoverflow.com/questions/47776486/python-struct-error-i-format-requires-2147483648-number-2147483647?rq=1
        elif (method in ["process"]):

            # Debug:
            # print ("\nself.TotalNumPixel = ", self.TotalNumPixel)
            # print ("self.Fit_NumProc = ", self.Fit_NumProc)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## prepare start of workers
            results = []
            self.return_dict = self.manager.dict()
            for i in range(0, self.TotalNumPixel, self.Fit_NumProc):
                thread_list = []
                for j in range(self.Fit_NumProc):


                    ## Instatiates the thread
                    k = i + j
                    if (k < len(IndexRange)):
                        t = multiprocessing.Process(name = str(j), target = ObjFunction, args = (IndexRange[k], ))


                        ## Sticks the thread in a list so that it remains accessible
                        thread_list.append(t)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                # Start all threads (workers)
                for thread in thread_list:
                    thread.start()


                ##----------------------------------------------------------------------------------------------------------------------------------------
                # This blocks the calling thread until the thread whose join() method is called is terminated.
                # From http://docs.python.org/2/library/threading.html#thread-objects
                for thread in thread_list:
                    thread.join()


                ##----------------------------------------------------------------------------------------------------------------------------------------
                # get return values
                for j in range(self.Fit_NumProc):
                    k = i + j
                    if (k < len(IndexRange)):
                        try:
                            LocalDict = self.return_dict[j]
                        except:
                            print ("\n\nj = ", j)
                            print ("k = ", k)
                            print ("len(IndexRange) = ", len(IndexRange))
                            print ("i = ", i)
                            print ("self.TotalNumPixel = ", self.TotalNumPixel)
                            print ("self.return_dict = ", self.return_dict)
                            sys.exit(0)
                        results.append(LocalDict)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## for future: start workers using MPI
        ## self.Fit_Host = ""
        elif (method in ["dask", "sc"]):
            results = []

            # Debug:
            # print ("self.BigCubeObject[:, 0, 0] = ", self.BigCubeObject[:, 0, 0])
            # print ("self.BigCubeObject = ", self.BigCubeObject)
            # print ("self.BigCubeObject.unit = ", self.BigCubeObject.unit)
            # print ("self.BigCubeObject.spectral_axis = ", self.BigCubeObject.spectral_axis)
            # print ("\nself.BigCubeObject.wcs = ", self.BigCubeObject.wcs)
            # print ("self.BigCubeObject.header = ", self.BigCubeObject.header)


            ##********************************************************************************************************************************************
            ## do the fits


            ## use dask in combination with spectral_cube
            if (method in ["dask"]):


                ## print where we are
                print ("\nUse spectral cube's function {:s} with dask!".format(chr(34) + "apply_function_parallel_spectral" + chr(34)))


                ## use spectral_cube and dask (in a furture version of spectral_cube add "save_to_tmp_dir = True" to the apply-function)
                with self.BigCubeObject.use_dask_scheduler('threads', num_workers = 50):
                    ResultBigCubeObject = self.BigCubeObject.apply_function_parallel_spectral(ObjFunction)


            ## use spectral_cube only
            else:


                ## print where we are
                print ("\nUse spectral cube's function {:s}!".format(chr(34) + "apply_function_parallel_spectral" + chr(34)))


                ## use spectral_cube only (in a furture version of spectral_cube add "save_to_tmp_dir = True" to the apply-function)
                ResultBigCubeObject = self.BigCubeObject.apply_function_parallel_spectral(ObjFunction, num_cores = self.Fit_NumProc)


            ##********************************************************************************************************************************************
            ## get results


            ## print time for fitting
            if (self.PrintFitStatusFlag):
                print ("\r                                                                       ", end = "", flush = True)
                end = time.time()
                TotalSeconds = end - start
                TimeString = str(datetime.timedelta(seconds = TotalSeconds))
                TimeString = TimeString.strip()
                # print ("\rUsed time for fitting: {:s}".format(TimeString))
                print ("\rUsed time for fitting: {:.1f} sec".format(TotalSeconds))


            ## print what you do
            if (self.PrintFitStatusFlag):
                print ("\n\nCollect results .. ", end = "", flush = True)


            ## get results
            NumObsDataCubes = len(self.RangeParam)
            ParamIndex = (-1)
            for PartID, PartBigCubeIndexList in enumerate(self.BigCubeIndexList):


                ## get synthetic spectra and chi^2 functions for each fitted pixel
                if (PartID < NumObsDataCubes):
                    i1 = PartBigCubeIndexList[0]
                    i2 = PartBigCubeIndexList[1]
                    LocalModelCube = ResultBigCubeObject[i1:i2, :, :]


                    ## flip model spectra?
                    ObsXMLParameterList = self.RangeParam[PartID]
                    ReverseFlag = ObsXMLParameterList['ReverseFlag']
                    if (ReverseFlag and self.FullFitFlag):                              ## (only for full fit method): for neg. freq. step sizes
                        LocalModelCube = LocalModelCube.apply_numpy_function(numpy.flip, axis = 0) ## flip model spectrum


                    if (self.Fit_LSFlag == 0):
                        self.SyntheticSpectraCubeList[PartID][:, :, :] = LocalModelCube[:, :, :] + self.BigCubeObject[i1:i2, :, :]
                        if (self.Fit_Chi2):
                            self.Chi2SpectraCubeList[PartID][:, :, :] = (LocalModelCube[:, :, :])**2
                    else:
                        self.SyntheticSpectraCubeList[PartID][:, :, :] = LocalModelCube[:, :, :]
                        if (self.Fit_Chi2):
                            self.Chi2SpectraCubeList[PartID][:, :, :] = (LocalModelCube[:, :, :] - self.BigCubeObject[i1:i2, :, :])**2


                ## get parameter values
                elif (PartID < len(self.BigCubeIndexList) - 2):
                    ParamIndex += 1
                    self.FitParameterMaps[:, :, ParamIndex] = ResultBigCubeObject[self.TotalNumObsDataSpecChannels + ParamIndex, :, :]


            ##********************************************************************************************************************************************
            ## store chi^2 value
            self.Chi2Map[:, :] = ResultBigCubeObject[-2, :, :]


            ## write status
            if (self.PrintFitStatusFlag):
                print ("done!")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store results back to parameter and chi2 maps
        if (not method in ["dask", "sc"]):


            ## print run time
            ## see https://stackoverflow.com/questions/775049/how-do-i-convert-seconds-to-hours-minutes-and-seconds
            if (self.PrintFitStatusFlag):
                print ("\r                                                                       ", end = "", flush = True)
                end = time.time()
                TotalSeconds = end - start
                TimeString = str(datetime.timedelta(seconds = TotalSeconds))
                TimeString = TimeString.strip()
                print ("\rUsed time for fitting: {:.1f} sec".format(TotalSeconds))


            ## print what you do
            if (self.PrintFitStatusFlag):
                print ("\n\nCollect results and write to file ..", end = "", flush = True)


            ## collect results
            for IndexMap, OutDict in enumerate(results):


                ## write status
                if (self.PrintFitStatusFlag):
                    print ("\rCollect results and write to file ({:d}/{:d}) done!".format(IndexMap + 1, len(results)), end = "", flush = True)


                ## get indices of current pixel
                # ypos = int(IndexMap / self.NumXPixels)
                # xpos = IndexMap - (ypos * self.NumXPixels)

                ypos = self.ListOfSelectedPixels[IndexMap][0]
                xpos = self.ListOfSelectedPixels[IndexMap][1]

                if (len(OutDict) > 0):
                    LocalChi2 = OutDict['chi2']
                    LocalParams = OutDict['params']
                    LocalSpectra = OutDict['residuals']
                    self.FitParameterMaps[ypos, xpos, :] = LocalParams
                    for LocalRangeID, ObsXMLParameterList in enumerate(self.RangeParam):
                        if (self.DataFileFormat == "fits"):
                            LocalSubCube = self.ListOfCubes[LocalRangeID]
                            LocalObsIntArray = LocalSubCube[:, ypos, xpos].value
                        elif (self.DataFileFormat == "ascii"):
                            LocalASCIIData = self.ListOfCubes[LocalRangeID]
                            LocalObsIntArray = LocalASCIIData[:, 1]
                        FreqMinIndex = self.LimitIndices[LocalRangeID][0]
                        FreqMaxIndex = self.LimitIndices[LocalRangeID][1]
                        LocalModelSpectrum = LocalSpectra[FreqMinIndex:FreqMaxIndex]

                        # Debug:
                        # print ("\nLocalRangeID = ", LocalRangeID)
                        # print ("ypos, xpos = ", ypos, xpos)
                        # print ("LocalObsIntArray.shape = ", LocalObsIntArray.shape)
                        # print ("LocalObsIntArray = ", LocalObsIntArray)
                        # print ("LocalModelSpectrum[:].shape = ", LocalModelSpectrum[:].shape)
                        # print ("LocalModelSpectrum[:] = ", LocalModelSpectrum[:])
                        # sys.exit(0)


                        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        ## store spectra


                        ## flip model spectrum?
                        ReverseFlag = ObsXMLParameterList['ReverseFlag']
                        if (ReverseFlag and self.FullFitFlag):                              ## (only for full fit method): for neg. freq. step sizes
                            LocalModelSpectrum = numpy.flip(LocalModelSpectrum)             ## flip model spectrum


                        ## store model spectrum for current position
                        if (self.Fit_LSFlag == 0):
                            self.SyntheticSpectraCubeList[LocalRangeID][:, ypos, xpos] = (-1.0) * (LocalModelSpectrum[:] - LocalObsIntArray)
                            if (self.Fit_Chi2):
                                self.Chi2SpectraCubeList[LocalRangeID][:, ypos, xpos] = (LocalModelSpectrum[:])**2
                        else:
                            self.SyntheticSpectraCubeList[LocalRangeID][:, ypos, xpos] = LocalModelSpectrum[:]
                            if (self.Fit_Chi2):
                                self.Chi2SpectraCubeList[LocalRangeID][:, ypos, xpos] = (LocalModelSpectrum[:] - LocalObsIntArray)**2
                                LocalChi2 += numpy.sum(self.Chi2SpectraCubeList[LocalRangeID][:, ypos, xpos])
                            else:
                                LocalChi2 += numpy.sum((LocalModelSpectrum[:] - LocalObsIntArray)**2)


                    ## store chi^2 value
                    self.Chi2Map[ypos, xpos] = LocalChi2

                # Debug:
                # print ("LocalParams = ", LocalParams)


            ## write status
            if (self.PrintFitStatusFlag):
                print ("\rCollect results and write to file done!                                                ", flush = True)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (fast-fitting only with spectral cube's function ): function called for each pixel
    def PixelFitFunction(self, InputParameter):
        """

    input parameters:
    -----------------

        - InputParameter:               input parameter with different meaning depending on the parallelization method which is used


    output parameters:
    ------------------

        - OutDict:                      directory {'chi2', 'params', 'residuals'} containing the fit results
        """

        # Debug:
        # print ("InputParameter = ", InputParameter)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize return parameter
        OutDict = {}


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get input parameters from dictionary


        ## get input parameters
        LocalGradientFlag = None
        if (self.NoFITSCubeFlag):
            ParamVector = InputParameter["ParamVector"]
            LocalGradientFlag = InputParameter["LocalGradientFlag"]
            IndexMap = InputParameter["IndexRange"]

            # Debug:
            # print ("\nParamVector = ", ParamVector)
            # print ("LocalGradientFlag = ", LocalGradientFlag)
            # print ("IndexMap = ", IndexMap)


        ## get input parameters for MapFit function
        else:
            LocalBigCubeSpectralAxis = None
            if (self.ParallelMethod in ["dask", "sc"]):
                LocalBigCubeSpectralAxis = InputParameter
                IndexMap = int(LocalBigCubeSpectralAxis[-1])
            else:
                IndexMap = InputParameter

        # Debug:
        # print ("IndexMap = ", IndexMap)
        # print ("self.NumXPixels = ", self.NumXPixels)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get indices of current pixel position
        ypos = self.ListOfSelectedPixels[IndexMap][0]
        xpos = self.ListOfSelectedPixels[IndexMap][1]

        # Debug:
        # print ("xpos = ", xpos)
        # print ("ypos = ", ypos)
        # print ("self.Fit_Variation = ", self.Fit_Variation)
        # print ("self.Fit_NumIter = ", self.Fit_NumIter)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get thread ID
        ThreadID = None
        if (self.ParallelMethod in ["process"]):
            ProcessName = multiprocessing.current_process().name
            ThreadID = int(ProcessName)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## continue only for pixels, which are fitted
        if (self.PixelMask[ypos, xpos]):


            ## print where you are
            if (self.PrintFitStatusFlag and (not self.NoFITSCubeFlag) and ThreadID is not None):
                if (ThreadID == 0):
                    print ("\rPerform fit of pixel ({:d}/{:d}) .. done!".format((IndexMap + 1), len(self.ListOfSelectedPixels)), end = "", flush = True)


            ## get initial parameter vector
            if (self.NoFITSCubeFlag):
                p0 = ParamVector
            else:
                p0 = self.FitParameterMaps[ypos, xpos, :]

            # Debug:
            # print ("\np0 = ", p0)
            # print ("self.pBLow = ", self.pBLow)
            # print ("self.pBUp = ", self.pBUp)


            ## define dictionary with additional parameters
            AddParamDict = {}
            AddParamDict["ypos"] = ypos
            AddParamDict["xpos"] = xpos
            AddParamDict["GradientFlag"] = LocalGradientFlag
            AddParamDict["Fit_LSFlag"] = self.Fit_LSFlag
            args = (AddParamDict,)

            # Debug:
            # print ("AddParamDict = ", AddParamDict)
            # print ("args = ", args)


            ## define, which fit function is used
            if (self.FullFitFlag):
                LocalFitFunction = self.FullFitFuncMin
            else:
                LocalFitFunction = self.FastFitFuncMin

            # Debug:
            # print ("LocalFitFunction = ", LocalFitFunction)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## start fitting process


            ## for the myXCLASSFit function only, just compute the spectra at the given pixel position
            if (self.NoFITSCubeFlag):
                OutDict = LocalFitFunction(p0, AddParamDict)


            ## continue with the normal MapFit part
            else:


                ## define parameter dictionary
                OptAlgParamDict = {}
                OptAlgParamDict["args"] = args
                OptAlgParamDict["verbose"] = 0
                OptAlgParamDict["GradientJAC"] = '2-point'
                OptAlgParamDict["callback"] = None


                ## call optimization algorithm
                OutputParameter = self.CallOptimizationAlgorithms(LocalFitFunction, p0, OptAlgParamDict)
                if (self.ParallelMethod in ["dask", "sc"]):
                    LocalBigCubeSpectralAxis = OutputParameter
                else:
                    OutDict = OutputParameter


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define return value for parallelization method "process"
        if (self.ParallelMethod in ["process"] and ThreadID is not None):
            self.return_dict[ThreadID] = OutDict


        ## define return parameter
        OutputParameter = None
        if (self.ParallelMethod in ["dask", "sc"]):
            OutputParameter = LocalBigCubeSpectralAxis
        else:
            OutputParameter = OutDict
        return OutputParameter
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (fast-fitting only): fit function for scipy optimization function
    def FastFitFuncMin(self, p0, *args):
        """

    input parameters:
    -----------------

        - p0:                       vector with fit parameters

        - args:                     additional arguments


    output parameters:
    ------------------

        - ReturnValue:              for least square fit algorithm: residual, others: model function
        """

        # Debug:
        # print ("p0 = ", p0)
        # print ("args = ", args)


        ## initialize return parameter
        ReturnValue = []


        ## get additional arguments
        AddParamDict = args[0]
        ypos = AddParamDict["ypos"]
        xpos = AddParamDict["xpos"]
        # LocalGradientFlag = AddParamDict["GradientFlag"]
        LocalFit_LSFlag = AddParamDict["Fit_LSFlag"]

        # Debug:
        # print ("self.NoFITSCubeFlag = ", self.NoFITSCubeFlag)
        # print ("ypos = ", ypos)
        # print ("xpos = ", xpos)
        # print ("LocalGradientFlag = ", LocalGradientFlag)
        # print ("LocalFit_LSFlag = ", LocalFit_LSFlag)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## prepare molfit array and update fit parameter
        LocalMolfitParameters = copy.deepcopy(self.AllMolfitParameters)

        # Debug:
        # print ("LocalMolfitParameters = ", LocalMolfitParameters)


        ## add parameters from initial parameter map
        ## self.ListOfInitialMaps = [NameParam, MoleculeName, SecondMolecue, CurrCompMap, LocalImage]
        if (len(self.ListOfInitialMaps) > 0):
            for LocalKey in self.ListOfInitialMaps:
                SplittedLocalKey = LocalKey.split("|")
                if (SplittedLocalKey[0] in ["molfit"]):
                    TotalCompCounter = int(SplittedLocalKey[1])
                    ParamCounter = int(SplittedLocalKey[2]) + 1
                    LocalMap = self.ListOfInitialMaps[LocalKey]
                    LocalMolfitParameters[TotalCompCounter, ParamCounter] = LocalMap[ypos, xpos]

                    # Debug:
                    # print ("\n\nLocalKey = ", LocalKey)
                    # print ("SplittedLocalKey = ", SplittedLocalKey)
                    # print ("TotalCompCounter = ", TotalCompCounter)
                    # print ("ParamCounter = ", ParamCounter)
                    # print ("ypos, xpos, LocalMap = ", ypos, xpos, LocalInitialMap[ypos, xpos])


        ## add new values of fit parameters to array describing molfit parameters
        OutOfRangeFlag = False
        LocalIsoParameters = []
        LocalRatioList = []
        for FitParameterID, FitParameter in enumerate(self.MolfitFitParameters):
            CompCounter = FitParameter[0]
            ParameterCounter = FitParameter[1]
            LogFlag = FitParameter[2]
            LowerLimit = FitParameter[3]
            UpperLimit = FitParameter[4]


            ## get new value and check if parameter is within limits
            NewFitParameterValue = p0[FitParameterID]

            # Debug:
            # print ("NewFitParameterValue = ", NewFitParameterValue)

            if (not (LowerLimit <= NewFitParameterValue and NewFitParameterValue <= UpperLimit)):
                OutOfRangeFlag = True
                break
            else:
                if (LogFlag):
                    NewFitParameterValue = 10.0**NewFitParameterValue
                LocalMolfitParameters[CompCounter, ParameterCounter] = NewFitParameterValue

                # Debug:
                # print ("LocalMolfitParameters[CompCounter, :] = ", LocalMolfitParameters[CompCounter, :])


        ## continue only, if fit parameter is within range
        if (not OutOfRangeFlag):


            ## add new values of iso fit parameters to array describing molfit parameters
            ##
            ## for global iso ratio definitions:
            ##      self.IsoFitParameters[i] = [(-1) * (len(self.GlobalIsoParam) + 1), LowerLimitIsoRatio, UpperLimitIsoRatio, RatioValue]
            ##
            ## for normal iso definitions:
            ##      self.IsoFitParameters[i] = [IsoIndex,                              LowerLimitIsoRatio, UpperLimitIsoRatio, RatioValue]
            ##
            p0IndexOffset = len(self.MolfitFitParameters)
            LocalGlobalIsoParam = copy.deepcopy(self.GlobalIsoParam)
            LocalIsoParametersList = copy.deepcopy(self.IsoParameters)


            ## update global and normal iso ratio definitions from initial map
            ## HERE:    self.ListOfInitialMaps[globalisoratio + "|" + IsoIndex] or self.ListOfInitialMaps[isoratio + "|" + IsoIndex]
            ## self.GlobalIsoParam[i] = [LocalIsotopologue, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag]
            ## self.IsoParameters[i] = [LocalIsotopologue, IsoMaster, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag, ListIndex]
            if (len(self.ListOfInitialMaps) > 0):
                for LocalKey in self.ListOfInitialMaps:
                    SplittedLocalKey = LocalKey.split("|")


                    ## globally defined iso ratios
                    if (SplittedLocalKey[0] in ["globalisoratio"]):
                        IsoIndex = int(SplittedLocalKey[1])
                        LocalMap = self.ListOfInitialMaps[LocalKey]
                        LocalGlobalIsoParam[IsoIndex][1] = LocalMap[ypos, xpos]


                    ## normally defined iso ratios
                    elif (SplittedLocalKey[0] in ["isoratio"]):
                        IsoIndex = int(SplittedLocalKey[1])
                        LocalMap = self.ListOfInitialMaps[LocalKey]
                        LocalIsoParametersList[IsoIndex][2] = LocalMap[ypos, xpos]

                    # Debug:
                    # print ("ypos, xpos, LocalMap = ", ypos, xpos, LocalMap[ypos, xpos])


            ## update fit parameters
            for IsoFitParameterID, IsoFitParameter in enumerate(self.IsoFitParameters):
                IsoIndex = IsoFitParameter[0]
                LowerLimitIsoRatio = IsoFitParameter[1]
                UpperLimitIsoRatio = IsoFitParameter[2]
                RatioValue = IsoFitParameter[3]


                ## get new value and check if parameter is within limits
                NewFitParameterValue = p0[p0IndexOffset + IsoFitParameterID]
                if (not (LowerLimitIsoRatio <= NewFitParameterValue and NewFitParameterValue <= UpperLimitIsoRatio \
                         and LowerLimitIsoRatio != UpperLimitIsoRatio)):
                    OutOfRangeFlag = True
                    break
                else:


                    ## update global iso ratios
                    ## self.GlobalIsoParam[i] = [LocalIsotopologue, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag]
                    if (IsoIndex < 0):
                        IsoIndex = abs(IsoIndex) - 1
                        LocalGlobalIsoParam[IsoIndex][1] = NewFitParameterValue


                    ## update normal iso ratios
                    else:
                        LocalIsoParametersList[IsoIndex][2] = NewFitParameterValue

            # Debug:
            # print ("LocalIsoParametersList = ", LocalIsoParametersList)
            # print ("OutOfRangeFlag = ", OutOfRangeFlag)


            ## create final iso ratio array
            if (not OutOfRangeFlag):


                ## create array describing iso parameters
                ## self.IsoParameters[i] = [LocalIsotopologue, IsoMaster, RatioValue, LowerLimitIsoRatio, UpperLimitIsoRatio, LocalFitFlag, ListIndex]
                ## with
                ## ListIndex[i] = [GlobalIndex, Multiplicator]
                NumIsoPerMolecule = {}
                for line in LocalIsoParametersList:
                    LocalIsotopologue = line[0]
                    IsoMaster = line[1]
                    RatioValue = line[2]
                    LowerLimitIsoRatio = line[3]
                    UpperLimitIsoRatio = line[4]
                    LocalFitFlag = line[5]
                    ListIndex = line[6]
                    if (len(ListIndex) > 0):
                        for LocalListIndex in ListIndex:
                            GlobalIndex = LocalListIndex[0]
                            Multiplicator = LocalListIndex[1]
                            RatioValue = RatioValue * (LocalGlobalIsoParam[GlobalIndex][1]**Multiplicator)
                    try:
                        i = self.MolNameList.index(LocalIsotopologue)
                    except:
                        i = (-1)
                    if (i > (-1)):
                        LocalIsoParameters.append([LocalIsotopologue, IsoMaster])
                        LocalRatioList.append([i, RatioValue])
                LocalRatioList = numpy.asarray(LocalRatioList)

                # Debug:
                # print ("LocalRatioList = ", LocalRatioList)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## determine partition function values for given temperatures
        NumComp = len(LocalMolfitParameters[:, 0])
        MaxNumIso = len(self.MolNameList)
        QTList = numpy.zeros((NumComp, MaxNumIso + 1), dtype = numpy.float32)
        for CompID, LocalTrotIndex in enumerate(self.TrotIndices):
            if (LocalTrotIndex > 0):
                LocalTrot = LocalMolfitParameters[CompID, LocalTrotIndex]
                LocalTrot = numpy.arange(1) + LocalTrot
                LocalMolecule = self.MolIndexPerComp[CompID]
                LocalPartFunc = self.PartFunc[LocalMolecule]
                if (LocalPartFunc is not None):
                    try:
                        i = self.MolNameList.index(LocalMolecule)
                    except:
                        i = (-1)
                    if (i > (-1)):
                        QTList[CompID, i] = LocalPartFunc(LocalTrot)


                        ## get partition functions for isotopologue as well
                        ## LocalIsoParameters[i] = [LocalIsotopologue, IsoMaster]
                        for IsoLine in LocalIsoParameters:
                            IsoMaster = IsoLine[1]
                            if (IsoMaster == LocalMolecule):
                                LocalIsotopologue = IsoLine[0]
                                LocalPartFunc = self.PartFunc[LocalIsotopologue]
                                try:
                                    i = self.MolNameList.index(LocalIsotopologue)
                                except:
                                    i = (-1)
                                if (i > (-1)):
                                    if (LocalPartFunc is not None):
                                        QTList[CompID, i] = LocalPartFunc(LocalTrot)

                        # Debug:
                        # print ("CompID, QTList[CompID, :] = ", CompID, QTList[CompID, :])

        # Debug:
        # print ("NumComp = ", NumComp)
        # print ("MaxNumIso = ", MaxNumIso)
        # print ("self.TrotIndices = ", self.TrotIndices)
        # print ("QTList = ", QTList)
        # print ("self.MolNameList = ", self.MolNameList)
        # print ("len(self.MolNameList) = ", len(self.MolNameList))
        # sys.exit(0)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## loop over all ranges
        yModInt = []
        yObsInt = []
        Weighting = []
        for LocalRangeID, ObsXMLParameterList in enumerate(self.RangeParam):


            ## get cube and corresponding spectral axis for current range
            LocalSubCube = self.ListOfCubes[LocalRangeID]
            LocalSpectralAxis = self.ListOfSpectralAxis[LocalRangeID]
            NumFreqPoints = len(LocalSpectralAxis)
            LocalWeightingArray = None
            if (self.DataFileFormat == "fits"):
                LocalObsIntArray = LocalSubCube[:, ypos, xpos].value
            elif (self.DataFileFormat == "ascii"):
                LocalASCIIData = self.ListOfCubes[LocalRangeID]
                LocalObsIntArray = LocalASCIIData[:, 1]
                ErrorY = ObsXMLParameterList['ErrorY']                                      ## required for weighting
                if (ErrorY):                                                                ## add weighting
                    LocalWeightingArray = LocalASCIIData[:, 2]
            LocalObsIntArray = numpy.nan_to_num(LocalObsIntArray)
            if (LocalWeightingArray is None):
                LocalWeightingArray = numpy.ones( (len(LocalObsIntArray[:]), ))


            ## get obs. xml parameter for current range
            RemoveContinuumFlag = ObsXMLParameterList['RemoveContinuumFlag']
            LocalOverlapFlag = ObsXMLParameterList['LocalOverlapFlag']
            TelescopeSize = ObsXMLParameterList['TelescopeSize']
            BMIN = ObsXMLParameterList['BMIN']
            BMAJ = ObsXMLParameterList['BMAJ']
            BPA = ObsXMLParameterList['BPA']
            InterFlag = ObsXMLParameterList['InterFlag']
            GlobalvLSR = ObsXMLParameterList['GlobalvLSR']
            Redshift = ObsXMLParameterList['Redshift']
            FreqMin = ObsXMLParameterList['FreqMin']
            FreqMax = ObsXMLParameterList['FreqMax']
            t_back_flag = ObsXMLParameterList['t_back_flag']
            tBack = ObsXMLParameterList['tBack']
            if (tBack is None):                                                             ## check, if continuum was estimated
                TBackImage = ObsXMLParameterList['TBackImage']
                tBack = TBackImage[ypos, xpos]
                TSlopeImage = ObsXMLParameterList['TSlopeImage']
                tSlope = TSlopeImage[ypos, xpos]
            else:
                tSlope = ObsXMLParameterList['tSlope']
            nH = ObsXMLParameterList['N_H']
            beta_dust = ObsXMLParameterList['beta_dust']
            kappa_dust = ObsXMLParameterList['kappa_1300']
            LocalContPhenFuncID = ObsXMLParameterList['ContPhenFuncID']
            LocalContPhenFuncParam1 = ObsXMLParameterList['ContPhenFuncParam1']
            LocalContPhenFuncParam2 = ObsXMLParameterList['ContPhenFuncParam2']
            LocalContPhenFuncParam3 = ObsXMLParameterList['ContPhenFuncParam3']
            LocalContPhenFuncParam4 = ObsXMLParameterList['ContPhenFuncParam4']
            LocalContPhenFuncParam5 = ObsXMLParameterList['ContPhenFuncParam5']
            LocalSmoothValue = ObsXMLParameterList['SmoothValue']
            LocalBackgroundCube = ObsXMLParameterList['BackgroundCube']
            if (LocalBackgroundCube is not None):
                if (self.DataFileFormat == "fits"):
                    try:
                        LocalBackgroundFunc = LocalBackgroundCube[:, ypos, xpos].vallue
                        LocalBackgroundFunc = numpy.nan_to_num(LocalBackgroundFunc)
                    except:
                        LocalBackgroundFunc = numpy.zeros((NumFreqPoints, 1))
                if (self.DataFileFormat == "ascii"):
                    LocalBackgroundFunc = copy.deepcopy(LocalBackgroundCube)
                    LocalBackgroundFunc = numpy.nan_to_num(LocalBackgroundFunc)
            else:
                LocalBackgroundFunc = numpy.zeros((NumFreqPoints, 1))
            LocalDustFileFunction = ObsXMLParameterList['DustCube']
            if (LocalDustFileFunction is not None):
                if (self.DataFileFormat == "fits"):
                    try:
                        LocalDustFunc = LocalDustFileFunction[:, ypos, xpos].vallue
                        LocalDustFunc = numpy.nan_to_num(LocalDustFunc)
                    except:
                        LocalDustFunc = numpy.zeros((NumFreqPoints, 1))
                if (self.DataFileFormat == "ascii"):
                    LocalDustFunc = copy.deepcopy(LocalDustFileFunction)
                    LocalDustFunc = numpy.nan_to_num(LocalDustFunc)
            else:
                LocalDustFunc = numpy.zeros((NumFreqPoints, 1))
            LocalEmissionAbsorptionFunction = ObsXMLParameterList['EmAbsPATH']
            DBParamList = ObsXMLParameterList['DBParamList']
            if (len(DBParamList) > 0 and not OutOfRangeFlag):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## prepare emission and absorption files (if defined)


                ## initialize array for emission / absorption funciton
                EmsAbsFunc = numpy.zeros((NumFreqPoints, self.NumDistances, 2))


                ## check, if emission and absorption functions were defined
                if (LocalEmissionAbsorptionFunction is not None):                           ## are emission / absorption functions defined?
                    EmsAbsFileContentDir = LocalEmissionAbsorptionFunction                  ## get interpolation functions for each distance


                    ## get a list of all distances defined in the emission / absorption files
                    DistanceList = [[float(x), x] for x in list(EmsAbsFileContentDir.keys())]
                    DistanceList = sorted(DistanceList, key = lambda l:l[0], reverse = True)

                    # Debug:
                    # print ("Dict-DistanceList = ", DistanceList)


                    ## define emission / absorption funcitons for each distance
                    for LocalDistance in DistanceList:                                      ## loop over all distances in emission / absorption files
                        LocalDistanceNum = LocalDistance[0]                                 ## get distance value
                        if (LocalDistanceNum in self.ListDistances):                        ## check, if current dist. is defined in molfit file as well
                            c = self.ListDistances.index(LocalDistanceNum)                  ## get index of current distance
                            LocalDistanceString = LocalDistance[1]                          ## get directory key
                            LocalFunc = LocalEmissionAbsorptionFunction[LocalDistanceString]
                            EmissionFunc = LocalFunc[0]
                            AbsorptionFunc = LocalFunc[1]
                            EmsAbsFunc[:, c, 0] = EmissionFunc(LocalSpectralAxis)
                            EmsAbsFunc[:, c, 1] = AbsorptionFunc(LocalSpectralAxis)

                            # Debug:
                            # print ("c, EmsAbsFunc[:, c, 0:10] = ", c, EmsAbsFunc[:, c, 0:10])


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get transition parameters
                ## DBParamList[i] = [TransFreq, EinsteinA, Elow, gup, MoleculeNameIndex]
                NumDBParam = len(DBParamList[:, 0])
                TransFreqList = DBParamList[:, 0]
                EinsteinAList = DBParamList[:, 1]
                ElowMinList = DBParamList[:, 2]
                gupList = DBParamList[:, 3]
                MolIndexList = DBParamList[:, 4]
                NumDist = self.NumDistances
                LocalActiveCompList = numpy.ones(NumComp)


                ## iso ratios for each transition!!!!!!
                ScalFactor = numpy.zeros(NumDBParam)
                for TransID, MolIndex in enumerate(MolIndexList):
                    if ((MolIndex + 1) > self.NumIsoMaster):
                        i = numpy.argwhere(LocalRatioList[:, 0] == MolIndex)
                        ScalFactor[TransID] = LocalRatioList[i[0], 1]

                # Debug:
                # print ("\n\n")
                # print ("self.NumEmissionComp = ", self.NumEmissionComp)
                # print ("RemoveContinuumFlag = ", RemoveContinuumFlag)
                # print ("TelescopeSize = ", TelescopeSize)
                # print ("BMIN = ", BMIN)
                # print ("BMAJ = ", BMAJ)
                # print ("BPA = ", BPA)
                # print ("InterFlag = ", InterFlag)
                # print ("GlobalvLSR = ", GlobalvLSR)
                # print ("Redshift = ", Redshift)
                # print ("FreqMin = ", FreqMin)
                # print ("t_back_flag = ", t_back_flag)
                # print ("tBack = ", tBack)
                # print ("tSlope = ", tSlope)
                # print ("nH = ", nH)
                # print ("beta_dust = ", beta_dust)
                # print ("kappa_dust = ", kappa_dust)
                # print ("LocalContPhenFuncID = ", LocalContPhenFuncID)
                # print ("LocalContPhenFuncParam1 = ", LocalContPhenFuncParam1)
                # print ("LocalContPhenFuncParam2 = ", LocalContPhenFuncParam2)
                # print ("LocalContPhenFuncParam3 = ", LocalContPhenFuncParam3)
                # print ("LocalContPhenFuncParam4 = ", LocalContPhenFuncParam4)
                # print ("LocalContPhenFuncParam5 = ", LocalContPhenFuncParam5)
                # print ("len(TransFreqList), TransFreqList = ", len(TransFreqList), TransFreqList)
                # print ("len(EinsteinAList), EinsteinAList = ", len(EinsteinAList), EinsteinAList)
                # print ("len(ElowMinList), ElowMinList = ", len(ElowMinList), ElowMinList)
                # print ("len(gupList), gupList = ", len(gupList), gupList)
                # print ("len(ScalFactor), ScalFactor = ", len(ScalFactor), ScalFactor)
                # print ("len(MolIndexList), MolIndexList = ", len(MolIndexList), MolIndexList)
                # print ("self.NumIsoMaster = ", self.NumIsoMaster)
                # print ("LocalRatioList[:, 0] = ", LocalRatioList[:, 0])
                # print ("len(LocalSpectralAxis), LocalSpectralAxis = ", len(LocalSpectralAxis), LocalSpectralAxis)
                # print ("len(LocalDustFunc), LocalDustFunc = ", len(LocalDustFunc), LocalDustFunc)
                # print ("len(LocalBackgroundFunc), LocalBackgroundFunc = ", len(LocalBackgroundFunc), LocalBackgroundFunc)
                # print ("len(LocalActiveCompList), LocalActiveCompList = ", len(LocalActiveCompList), LocalActiveCompList)
                # print ("QTList.shape, QTList = ", QTList.shape, QTList)
                # print ("LocalMolfitParameters.shape, LocalMolfitParameters = ", LocalMolfitParameters.shape, LocalMolfitParameters)
                # print ("len(EmsAbsFunc), EmsAbsFunc = ", len(EmsAbsFunc), EmsAbsFunc)
                # print ("NumDBParam = ", NumDBParam)
                # print ("NumFreqPoints = ", NumFreqPoints)
                # print ("NumComp = ", NumComp)
                # print ("MaxNumIso = ", MaxNumIso)
                # print ("NumDist = ", NumDist)
                # print ("LocalOverlapFlag = ", LocalOverlapFlag)


                ##****************************************************************************************************************************************
                ## call fortran module TinyXCLASS to compute synthetic spectrum for current range
                ## Note: SubBeam Flag is not require, because this Fortran module can not handle sub-beam description
                from xclass.lib import xclasslite
                ModeledRangeSpectrum = xclasslite.calcxclass(self.NumEmissionComp, RemoveContinuumFlag, TelescopeSize, BMIN, BMAJ, BPA, InterFlag, \
                                                                 GlobalvLSR, Redshift, FreqMin, t_back_flag, tBack, tSlope, nH, beta_dust, kappa_dust, \
                                                                 LocalContPhenFuncID, LocalContPhenFuncParam1, LocalContPhenFuncParam2, \
                                                                 LocalContPhenFuncParam3, LocalContPhenFuncParam4, LocalContPhenFuncParam5, \
                                                                 TransFreqList, EinsteinAList, ElowMinList, gupList, ScalFactor, MolIndexList, \
                                                                 LocalSpectralAxis, LocalDustFunc, LocalBackgroundFunc, LocalActiveCompList, QTList, \
                                                                 LocalMolfitParameters, EmsAbsFunc, LocalOverlapFlag)
                                                                 # NumDBParam, NumFreqPoints, NumComp, MaxNumIso, NumDist, \
                # Debug:
                # print ("xclasslite.calcxclass.__doc__ = ", xclasslite.calcxclass.__doc__)
                # print ("ModeledRangeSpectrum = ", ModeledRangeSpectrum)
                # print ("LocalObsIntArray = ", LocalObsIntArray)

                # plt.figure(figsize=(15, 10))
                # plt.plot(LocalSpectralAxis, LocalObsIntArray, color = 'black', lw = 2, label = "data")
                # plt.plot(LocalSpectralAxis, ModeledRangeSpectrum, color = 'red', lw = 1, label = "model")
                # plt.legend()
                # plt.savefig("model-spectrum.png", bbox_inches = 'tight')
                # sys.exit(0)


            ## continue here, if no transition is included in the current frequency range
            # LocalSpectralAxis
            else:
                ModeledRangeSpectrum = task_LineIdentification.ContinuumFunction(LocalSpectralAxis.value, tBack, tSlope, FreqMin)

                # Debug:
                # print ("ModeledRangeSpectrum = ", ModeledRangeSpectrum)


            ##********************************************************************************************************************************************
            ## remove NaNs and infinity values
            ModeledRangeSpectrum = numpy.nan_to_num(ModeledRangeSpectrum)

            # Debug:
            # print ("numpy.isnan(ModeledRangeSpectrum).any() = ", numpy.isnan(ModeledRangeSpectrum).any())
            # print ("numpy.isinf(abs(ModeledRangeSpectrum)).any() = ", numpy.isinf(abs(ModeledRangeSpectrum)).any())


            ##********************************************************************************************************************************************
            ## for minimazation compute chi2


            ## for least square algorithms
            if (LocalFit_LSFlag == 0 or self.NoFITSCubeFlag):
                if (LocalRangeID == 0):
                    yModInt = copy.deepcopy(ModeledRangeSpectrum)
                    yObsInt = copy.deepcopy(LocalObsIntArray)
                    Weighting = copy.deepcopy(LocalWeightingArray)
                else:
                    yModInt = numpy.concatenate((yModInt, ModeledRangeSpectrum), axis = None)
                    yObsInt = numpy.concatenate((yObsInt, LocalObsIntArray), axis = None)
                    Weighting = numpy.concatenate((Weighting, LocalWeightingArray), axis = None)


            ## for blank application
            elif (LocalFit_LSFlag == 1):
                if (LocalRangeID == 0):
                    yModInt = copy.deepcopy(ModeledRangeSpectrum)
                else:
                    yModInt = numpy.concatenate((yModInt, ModeledRangeSpectrum), axis = None)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store results for


        ## non-MapFit
        if (self.NoFITSCubeFlag):
            ReturnValue = {}
            ReturnValue['chi2'] = numpy.sum( ((yObsInt - yModInt)**2) / (Weighting**2) )
            ReturnValue['spectrum'] = yModInt
            ReturnValue['residual'] = (yObsInt - yModInt) * Weighting**(-1)

            # Debug:
            # print ("\n\nyObsInt = ", yObsInt)
            # print ("yModInt = ", yModInt)
            # print ("Weighting = ", Weighting)


        ## MapFit
        else:
            if (LocalFit_LSFlag == 0):                                                      ## for least square algorithms
                ReturnValue = (yObsInt - yModInt)
                ReturnValue = numpy.nan_to_num(ReturnValue)
                ReturnValue = numpy.clip(ReturnValue, -1.e20, 1.e20)
            elif (LocalFit_LSFlag == 1):                                                    ## for blank application
                ReturnValue = yModInt
            else:                                                                           ## for global optimization algorithms
                ReturnValue = numpy.sum((yObsInt - yModInt)**2)

        # Debug:
        # print ("ReturnValue = ", ReturnValue)
        # print ("numpy.isfinite(abs(ReturnValue)).any() = ", numpy.isfinite(abs(ReturnValue)).any())


        ## we're done
        return ReturnValue
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (fast-fitting only): fit function for scipy optimization function
    def FullFitFuncMin(self, p0, *args):
        """

    input parameters:
    -----------------

        - p0:                           vector with fit parameters

        - args:                         additional arguments


    output parameters:
    ------------------

        - ReturnValue:                  for least square fit algorithm: residual, others: model function



    used class variables:
    ---------------------

        - self.Fit_Algorithm:           current fit algorithm
        - self.NoFITSCubeFlag:          single spectra flag
        - self.ListOfInitialMaps:       list of initial parameter maps
        - self.XCLASSDict:              dictionary with parameters for XCLASS core routines
        - self.ra:                      list of ra coordinates
        - self.dec:                     list of dec coordinates
        - self.IsoFlag:                 flag for usage of iso ratio file
        - self.IsotopologuesIn:         list of isotopologues
        - self.IsoMoleculeIn:           list of iso master molecules
        - self.ListOfSpectralAxis:      list of frequency points
        - self.NumDistances:            number of used distances
        - self.RangeParam:              additional parameters for each frequency range


    used class functions:
    ---------------------

        - xclassinterface:              fortran object with XCLASS core functions
        - self.PrepareXCLASS:           prepare call of XCLASS core functions
        """

        # Debug:
        # print ("p0 = ", p0)
        # print ("args = ", args)


        ## initialize return parameter
        ReturnValue = []


        ## get additional arguments
        AddParamDict = args[0]
        ypos = AddParamDict["ypos"]
        xpos = AddParamDict["xpos"]
        LocalGradientFlag = AddParamDict["GradientFlag"]
        LocalFit_LSFlag = AddParamDict["Fit_LSFlag"]

        # Debug:
        # print ("self.NoFITSCubeFlag = ", self.NoFITSCubeFlag)
        # print ("ypos = ", ypos)
        # print ("xpos = ", xpos)
        # print ("LocalGradientFlag = ", LocalGradientFlag)
        # print ("LocalFit_LSFlag = ", LocalFit_LSFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## prepare XCLASS


        ## import XCLASS interface package
        from xclass.lib import xclassinterface


        ## prepare call of XCLASS core functions
        self.PrepareXCLASS(xclassinterface)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update with initial parameter maps
        LocalInitmyXCLASSParameter = None
        if (len(self.ListOfInitialMaps) > 0):
            LocalInitmyXCLASSParameter = copy.deepcopy(self.XCLASSDict['myXCLASSParameter'])
            # try:
            #     LocalInitGlobalIsoRatioParameter = copy.deepcopy(self.XCLASSDict['GlobalIsoRatioParameter'])
            # except:
            #     LocalInitGlobalIsoRatioParameter = None
            try:
                LocalInitIsoRatio = copy.deepcopy(self.XCLASSDict['IsoRatio'])
            except:
                LocalInitIsoRatio = None
            for LocalKey in self.ListOfInitialMaps:
                SplittedLocalKey = LocalKey.split("|")

                # Debug:
                # print ("SplittedLocalKey = ", SplittedLocalKey)
                # print ("ypos, xpos = ", ypos, xpos)


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## add parameters from initial parameter map
                ## LocalInitmyXCLASSParameter = myXCLASSParameter(16, TotalNumberComponents)
                if (SplittedLocalKey[0] in ["molfit"]):
                    TotalCompCounter = int(SplittedLocalKey[1])
                    ParamCounter = int(SplittedLocalKey[2])
                    LocalMap = self.ListOfInitialMaps[LocalKey]
                    LocalInitmyXCLASSParameter[ParamCounter, TotalCompCounter] = LocalMap[ypos, xpos]

                    # Debug:
                    # print ("\n\nTotalCompCounter = ", TotalCompCounter)
                    # print ("ParamCounter = ", ParamCounter)
                    # print ("ypos, xpos, LocalMap = ", ypos, xpos, LocalMap[ypos, xpos])
                    # print ("LocalInitmyXCLASSParameter = ", LocalInitmyXCLASSParameter)


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## update global and normal iso ratio definitions from initial map


                ## globally defined iso ratios
                # elif (SplittedLocalKey[0] in ["globalisoratio"] and LocalInitGlobalIsoRatioParameter is not None):
                #     IsoIndex = int(SplittedLocalKey[1])
                #     LocalMap = self.ListOfInitialMaps[LocalKey]
                #     LocalInitGlobalIsoRatioParameter[IsoIndex] = LocalMap[ypos, xpos]

                    # Debug:
                    # print ("ypos, xpos, LocalMap = ", ypos, xpos, LocalMap[ypos, xpos])


                ## normally defined iso ratios
                elif (SplittedLocalKey[0] in ["isoratio"] and LocalInitIsoRatio is not None):
                    IsoIndex = int(SplittedLocalKey[1])
                    LocalMap = self.ListOfInitialMaps[LocalKey]
                    LocalInitIsoRatio[IsoIndex, 1] = LocalMap[ypos, xpos]

                    # Debug:
                    # print ("ypos, xpos, LocalMap = ", ypos, xpos, LocalMap[ypos, xpos])


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## update parameter arrays

            # Debug:
            # print ("\nLocalInitmyXCLASSParameter = ", LocalInitmyXCLASSParameter)
            # print ("\nLocalInitIsoRatio = ", LocalInitIsoRatio)


            ## update molfit parameters
            xclassinterface.inter.initmyxclassparameter = LocalInitmyXCLASSParameter


            ## update iso ratios
            if (LocalInitIsoRatio is not None):
                xclassinterface.inter.initisoratio = LocalInitIsoRatio
            # if (LocalInitGlobalIsoRatioParameter is not None):
            #     xclassinterface.inter.initglobalisoratioparameter = LocalInitGlobalIsoRatioParameter


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update fit parameters
        ## "fitparametervector" instead of Initfitparametervector has to be used here, because it is not directly defined
        if (len(p0) > 0):
            xclassinterface.inter.fitparametervector = p0
        else:
            xclassinterface.inter.fitparametervector = numpy.ones((2,))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## loop over all ranges


        ## prepare numpy arrays for obs. data, background spectrum, dust opacity and emisssion / absorption functions
        NumFreqPointsList = [len(x) for x in self.ListOfSpectralAxis]
        TotalFreqPoints = sum(NumFreqPointsList)
        TotalObsIntArray = numpy.zeros((TotalFreqPoints, ))
        TotalWeightingArray = numpy.ones((TotalFreqPoints, ))
        TotalBackgroundFunc = numpy.zeros((TotalFreqPoints, ))
        TotalDustFunc = numpy.zeros((TotalFreqPoints, ))
        TotalEmsAbsFunc = numpy.zeros((TotalFreqPoints, self.NumDistances, 2))
        NewtBackList = []
        NewtSlopeList = []
        UseEmAbsFuncFlag = False
        startindex = 0
        endindex = 0
        for LocalRangeID, ObsXMLParameterList in enumerate(self.RangeParam):
            LocalSpectralAxis = self.ListOfSpectralAxis[LocalRangeID]
            NumFreqPoints = len(LocalSpectralAxis)
            endindex += NumFreqPoints

            # Debug:
            # print ("\nLocalRangeID = ", LocalRangeID)
            # print ("NumFreqPoints = ", NumFreqPoints)
            # print ("startindex = ", startindex)
            # print ("endindex = ", endindex)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## get cube and corresponding spectral axis for current range
            LocalWeightingArray = None
            if (self.DataFileFormat == "fits"):
                LocalSubCube = self.ListOfCubes[LocalRangeID]
                LocalObsIntArray = LocalSubCube[:, ypos, xpos].value
            elif (self.DataFileFormat == "ascii"):
                LocalASCIIData = self.ListOfCubes[LocalRangeID]
                LocalObsIntArray = LocalASCIIData[:, 1]
                ErrorY = ObsXMLParameterList['ErrorY']                                      ## required for weighting
                if (ErrorY):                                                                ## add weighting
                    LocalWeightingArray = LocalASCIIData[:, 2]
            LocalObsIntArray = numpy.nan_to_num(LocalObsIntArray)
            if (LocalSpectralAxis[1] - LocalSpectralAxis[0] < 0.0 and self.Fit_LSFlag != 1):
                LocalObsIntArray = numpy.flip(LocalObsIntArray)
                if (LocalWeightingArray is not None):
                    LocalWeightingArray = numpy.flip(LocalWeightingArray)
            TotalObsIntArray[startindex:endindex] = LocalObsIntArray
            if (LocalWeightingArray is not None):
                TotalWeightingArray[startindex:endindex] = LocalWeightingArray

            # Debug:
            # print ("TotalObsIntArray = ", TotalObsIntArray)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## get T_Back and T_Slope
            tBack = ObsXMLParameterList['tBack']
            if (tBack is None):
                TBackImage = ObsXMLParameterList['TBackImage']
                tBack = TBackImage[ypos, xpos]
                TSlopeImage = ObsXMLParameterList['TSlopeImage']
                tSlope = TSlopeImage[ypos, xpos]
            else:
                tSlope = ObsXMLParameterList['tSlope']
            NewtBackList.append(tBack)
            NewtSlopeList.append(tSlope)

            # Debug:
            # print ("tBack = ", tBack)
            # print ("tSlope = ", tSlope)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## get background function
            LocalBackgroundCube = ObsXMLParameterList['BackgroundCube']
            if (LocalBackgroundCube is not None):
                if (self.DataFileFormat == "fits"):
                    try:
                        LocalBackgroundFunc = LocalBackgroundCube[:, ypos, xpos].vallue
                        TotalBackgroundFunc[startindex:endindex] = numpy.nan_to_num(LocalBackgroundFunc)
                    except:
                        TotalBackgroundFunc[startindex:endindex] = 0.0
                elif (self.DataFileFormat == "ascii"):
                    TotalBackgroundFunc[startindex:endindex] = numpy.nan_to_num(LocalBackgroundCube)

            # Debug:
            # print ("LocalBackgroundCube = ", LocalBackgroundCube)
            # print ("TotalBackgroundFunc = ", TotalBackgroundFunc)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## get dust opacity
            LocalDustFileFunction = ObsXMLParameterList['DustCube']
            if (LocalDustFileFunction is not None):
                if (self.DataFileFormat == "fits"):
                    try:
                        LocalDustFunc = LocalDustFileFunction[:, ypos, xpos]
                        TotalDustFunc[startindex:endindex] = numpy.nan_to_num(LocalDustFunc)
                    except:
                        TotalDustFunc[startindex:endindex] = 0.0
                elif (self.DataFileFormat == "ascii"):
                    TotalDustFunc[startindex:endindex] = numpy.nan_to_num(LocalDustFileFunction)

            # Debug:
            # print ("LocalDustFileFunction = ", LocalDustFileFunction)
            # print ("TotalDustFunc = ", TotalDustFunc)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## prepare emission and absorption files (if defined)
            LocalEmissionAbsorptionFunction = ObsXMLParameterList['EmAbsPATH']
            if (LocalEmissionAbsorptionFunction is not None):
                EmsAbsFileContentDir = LocalEmissionAbsorptionFunction                      ## get interpolation functions for each distance


                ## get a list of all distances defined in the emission / absorption files
                DistanceList = [[float(x), x] for x in list(EmsAbsFileContentDir.keys())]
                DistanceList = sorted(DistanceList, key = lambda l:l[0], reverse = True)

                # Debug:
                # print ("Dict-DistanceList = ", DistanceList)


                ## define emission / absorption funcitons for each distance
                for LocalDistance in DistanceList:                                          ## loop over all distances in emission / absorption files
                    LocalDistanceNum = LocalDistance[0]                                     ## get distance value
                    if (LocalDistanceNum in self.ListDistances):                            ## check, if current dist. is defined in molfit file as well
                        UseEmAbsFuncFlag = True
                        c = self.ListDistances.index(LocalDistanceNum)                      ## get index of current distance
                        LocalDistanceString = LocalDistance[1]                              ## get directory key
                        LocalFunc = LocalEmissionAbsorptionFunction[LocalDistanceString]
                        EmissionFunc = LocalFunc[0]
                        AbsorptionFunc = LocalFunc[1]
                        TotalEmsAbsFunc[startindex:endindex, c, 0] = EmissionFunc(LocalSpectralAxis)
                        TotalEmsAbsFunc[startindex:endindex, c, 1] = AbsorptionFunc(LocalSpectralAxis)

                        # Debug:
                        # print ("c, EmsAbsFunc[:, c, 0:10] = ", c, EmsAbsFunc[:, c, 0:10])


            ## increase index limit
            startindex += NumFreqPoints

        # Debug:
        # print ("TotalObsIntArray = ", TotalObsIntArray)
        # print ("TotalBackgroundFunc = ", TotalBackgroundFunc)
        # print ("TotalDustFunc = ", TotalDustFunc)
        # print ("c, EmsAbsFunc[:, c, 0:10] = ", c, EmsAbsFunc[:, c, 0:10])


        ## update some parameters in XCLASS interface
        xclassinterface.inter.initbackgroundtemperaturerange = NewtBackList                 ## update new T_Back parameters
        xclassinterface.inter.inittemperaturesloperange = NewtSlopeList                     ## update new T_Slope parameters
        xclassinterface.inter.initbackgroundfromfile = TotalBackgroundFunc.flatten()        ## update background function
        xclassinterface.inter.initdusttaufromfile = TotalDustFunc.flatten()                 ## update dust opacities
        if (self.ChannelIntegrationFlag):
            xclassinterface.inter.initintegrationflag = 1                                   ## apply channel integration
        else:
            xclassinterface.inter.initintegrationflag = 0                                   ## do not apply channel integration
        xclassinterface.inter.initalloutputfilesflag = 0                                    ## do not write intensities and optical depths to file
        if (UseEmAbsFuncFlag):                                                              ## set flag indicating usage of ems. / abs. func.
            xclassinterface.inter.inituseemabsfuncflag = 1
        else:
            xclassinterface.inter.inituseemabsfuncflag = 0
        xclassinterface.inter.initemsabsfunc = TotalEmsAbsFunc                              ## update em. and abs. functions


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## call XCLASS core functions to compute synthetic spectrum for all ranges
        xclassinterface.inter.spectrum()
        try:
            ModeledRangeSpectrum = xclassinterface.inter.modelfunclist
        except:
            ModeledRangeSpectrum = copy.deepcopy(TotalObsIntArray) * 0.0

        # Debug:
        # print ("xclassinterface.inter.__doc__ = ", xclassinterface.inter.__doc__)
        # print ("ModeledRangeSpectrum = ", ModeledRangeSpectrum)
        # print ("LocalObsIntArray = ", LocalObsIntArray)
        # if (ypos == 0 and xpos == 0):
        #     print ("p0 = ", p0)
        #     print ("xclassinterface.inter.initmyxclassparameter = ", xclassinterface.inter.initmyxclassparameter)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## create pixel plots (for debugging only)
        if (self.dbgFlag):
            # if (not LocalGradientFlag):


            ## create pixel plots
            i1 = 0
            i2 = 0
            i = len(self.ListOfSpectralAxis)
            fig, axs = plt.subplots(i)
            for id, LocalSpectralAxis in enumerate(self.ListOfSpectralAxis):
                if (i == 1):
                    ax = axs
                else:
                    ax = axs[id]
                i2 += len(LocalSpectralAxis)
                if (id == 0):
                    ax.plot(LocalSpectralAxis, TotalObsIntArray[i1:i2], color = 'black', lw = 2, label = "data")
                    ax.plot(LocalSpectralAxis, ModeledRangeSpectrum, color = 'red', lw = 1, label = "model")
                    ax.set_xlabel("Frequency (MHz)")
                    ax.set_ylabel("Brightness Temperature (K)")
                    ax.legend()
                else:
                    ax.plot(LocalSpectralAxis, TotalObsIntArray[i1:i2], color = 'black', lw = 2)
                    ax.plot(LocalSpectralAxis, ModeledRangeSpectrum, color = 'red', lw = 1)
                i1 = i2
            plt.savefig("model-spectrum___{:d}_-_{:d}.png".format(ypos, xpos), bbox_inches = 'tight')


            ## write spectrum to file
            numpy.savetxt("model-spectrum___{:d}_-_{:d}.dat".format(ypos, xpos), \
                          numpy.vstack((self.ListOfSpectralAxis[0].value, ModeledRangeSpectrum)).T)


            ## write spectrum to file
            numpy.savetxt("obs-spectrum___{:d}_-_{:d}.dat".format(ypos, xpos), \
                          numpy.vstack((self.ListOfSpectralAxis[0].value, TotalObsIntArray)).T)
            sys.exit(0)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store results for

        ## non-MapFit
        if (self.NoFITSCubeFlag):
            ReturnValue = {}
            ReturnValue['chi2'] = numpy.sum( ((TotalObsIntArray[:] - ModeledRangeSpectrum[:])**2)  * TotalWeightingArray[:]**(-2) )
            ReturnValue['spectrum'] = ModeledRangeSpectrum
            ReturnValue['residual'] = (TotalObsIntArray - ModeledRangeSpectrum) * TotalWeightingArray[:]**(-1)


        ## MapFit
        else:
            if (LocalFit_LSFlag == 0):                                                      ## for least square algorithms
                ReturnValue = (TotalObsIntArray - ModeledRangeSpectrum)
            elif (LocalFit_LSFlag == 1):                                                    ## for blank application
                ReturnValue = ModeledRangeSpectrum
            else:                                                                           ## for global optimization algorithms
                ReturnValue = numpy.sum((TotalObsIntArray[:] - ModeledRangeSpectrum[:])**2)

        # Debug:
        # print ("ReturnValue = ", ReturnValue)


        ## we're done
        return ReturnValue
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (normal fitting only): start normal fitting
    def StartNormalFitting(self, IndexRange):
        """

    input parameters:
    -----------------

        - IndexRange:                   list of indices


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("IndexRange = ", IndexRange)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## start pool of worker
        self.StartWorker(self.NormalPixelFitFunction, IndexRange, method = "process")


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## write results to job directory
        if (self.NumParamMapIterations > 1):
            self.ExportResults(self.LocalParamMapIter)
        else:
            self.ExportResults((-1))


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (normal fitting only): function called MAGIX for each pixel
    def NormalPixelFitFunction(self, IndexMap):
        """

    input parameters:
    -----------------

        - IndexMap:                     map describing indices of each pixel


    output parameters:
    ------------------

        - OutDict:                      directory {'chi2', 'params', 'residuals'} containing the fit results
        """

        # Debug:
        # print ("\nIndexMap = ", IndexMap)


        ## get indices of current pixel
        # ypos = int(IndexMap / self.NumXPixels)
        # xpos = IndexMap - (ypos * self.NumXPixels)

        ypos = self.ListOfSelectedPixels[IndexMap][0]
        xpos = self.ListOfSelectedPixels[IndexMap][1]

        # Debug:
        # print ("xpos = ", xpos)
        # print ("ypos = ", ypos)
        # print ("self.NumXPixels = ", self.NumXPixels)


        ## print where you are
        if (self.PrintFitStatusFlag):
            if (xpos == 0):
                print ("\rPerform fit ({:d}/{:d}) .. done!".format((ypos + 1), self.NumYPixels), end = "", flush = True)


        ## continue only, if indices are not None
        OutDict = {}
        if (self.PixelMask[ypos, xpos]):
            p0 = self.FitParameterMaps[ypos, xpos, :]
            args = (ypos, xpos)

            # Debug:
            # print ("\np0 = ", p0)
            # print ("self.pBLow = ", self.pBLow)
            # print ("self.pBUp = ", self.pBUp)


            ## get path of current pixel
            xPixelCoord = self.ra[xpos].value
            yPixelCoord = self.dec[ypos].value
            xString = "{:.10f}".format(xPixelCoord)
            yString = "{:.10f}".format(yPixelCoord)
            PixelDirectory = self.myXCLASSMapFitJobDir + "Pixel-Fits/" + xString + "__" + yString + "___"
            PixelDirectory += str(xpos + 1) + "_-_" + str(ypos + 1)
            PixelDirectory = os.path.normpath(PixelDirectory) + "/"


            ##================================================================================================================================================
            ## update input file(s) for map iteration > 1
            if (self.LocalParamMapIter > 0):


                ## update molfit file
                LocalMolfitFileName = PixelDirectory + "all-molecules.molfit"
                self.UpdateMolfitFile(LocalMolfitFileName, ypos, xpos)


                ## update iso ratio file
                if (self.IsoFlag):
                    IsoTableFileName = PixelDirectory + "iso.dat"
                    self.UpdateIsoRatioFile(IsoTableFileName, ypos, xpos)


            ##================================================================================================================================================
            ## start MAGIX


            ## on local computer only
            cmdString = "cd " + self.MAGIXrootDir + "; "
            cmdString += "python3 magix_start.py --plotsaveonly --model=myxclass " + PixelDirectory + "io_control.xml > "
            cmdString += PixelDirectory + "screen__magix.out"


            ## via ssh on other nodes
            ## ssh charger 'cd ~/magma; nohup nice /usr/bin/magma magma.script > /var/tmp/magma.out' &
            # cmdString = "ssh " + Node + " " + chr(34) + "cd " + PixelDirectory + "; "
            # cmdString += "nohup python3 magix_start.py --plotsaveonly --model=myxclass " + PixelDirectory + "io_control.xml > "
            # cmdString += PixelDirectory + "screen__magix.out" + chr(34)


            ## execute command string
            os.system(cmdString)


            ##================================================================================================================================================
            ## read in ouput files
            ConvertLogLinFlag = True
            if (self.NameOfFunction == "myXCLASSMapRedoFit"):
                ConvertLogLinFlag = False
            OutDict = self.GetMolfitFileOfBestResult(PixelDirectory, ConvertLogLinFlag)

        # Debug
        # print ("OutDict = ", OutDict)


        ## define return value for parallelization method "process"
        if (self.ParallelMethod in ["process"]):
            ProcessName = multiprocessing.current_process().name
            ThreadID = int(ProcessName)
            self.return_dict[ThreadID] = OutDict

        elif (self.ParallelMethod in ["threading"]):
            thread = current_thread()
            ThreadName = thread.name
            ThreadName = ThreadName.replace("ThreadPoolExecutor-0_", "")
            LocalThreadID = int(ThreadName)
            self.return_dict[LocalThreadID] = OutDict

            # Debug:
            # print ("LocalThreadID = ", LocalThreadID)


        ## we're done
        return OutDict
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## (normal fitting only): determines the best fit result from a previous pixel fit
    ##
    def GetMolfitFileOfBestResult(self, JobDir, ConvertLogLinFlag, SQLParameters = None, LocalMinColumnDensityAbs = 1.0, LocalMinColumnDensityEmis = 1.0):
        """

    input parameters:
    -----------------

        - JobDir:                       path of job directory

        - ConvertLogLinFlag:            defines if column densities in the fitted molfit file(s) are converted back to linear values

        - SQLParameters:                (optional): SQL parameters, (default: None)

        - LocalMinColumnDensityAbs:     (optional): min. column density of absorption component, (default: 1.0)

        - LocalMinColumnDensityEmis:    (optional): min. column density of emission component, (default: 1.0)


    output parameters:
    ------------------

        - BestResultsDict:              dictionary containing best results
        """

        # Debug:
        # print ("JobDir = ", JobDir)
        # print ("ConvertLogLinFlag = ", ConvertLogLinFlag)
        # print ("SQLParameters = ", SQLParameters)
        # print ("LocalMinColumnDensityAbs = ", LocalMinColumnDensityAbs)
        # print ("LocalMinColumnDensityEmis = ", LocalMinColumnDensityEmis)


        ## initialize internal parameters
        BestResultsDict = {}
        BestChi2Value = 1.e99
        p0 = []
        yModInt = []


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get all filenames in the current job directory
        listing = os.listdir(JobDir)

        # Debug:
        # print ("JobDir = ", JobDir)
        # print ("listing = ", listing)


        ## get number of chi2 log files
        NumOutFiles = 0                                                                     ## determine number of output files
        NumMolfitFiles = 0
        for files in listing:
            if (files.endswith(".log.chi2")):
                NumOutFiles += 1
            if (files.endswith(".out.molfit")):
                NumMolfitFiles += 1
        if (NumOutFiles == 0):                                                              ## check if new input files exsist
            print ("\n\nError in XCLASS package, subroutine task_myXCLASSMapFit.GetBestResult:")
            print ("\tCan not find a chi2 log file!")
            print ("\n\tJobDir = ", JobDir)
            print ("\tlisting = ", listing)
            print ("\n\n\n")
            BestResultsDict['chi2'] = BestChi2Value
            BestResultsDict['params'] = p0
            BestResultsDict['residuals'] = yModInt
            return BestResultsDict


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine the chi2 log file, which corresponds to the best fit
        FileNameBestChi2 = ""
        for files in listing:                                                               ## loop over all files in the current job directory
            if (files.endswith(".log.chi2")):                                               ## we're interested only in the log.chi2 files
                Chi2LogFile = open(JobDir + files)                                          ## open log.chi2 file
                dummyline = Chi2LogFile.readline()                                          ## read in first line as dummy
                LogLine = Chi2LogFile.readline()                                            ## read second line with best chi2 value
                Chi2LogFile.close()                                                         ## close log file
                LogLine = LogLine.strip()                                                   ## remove blanks
                SplittedLine = LogLine.split()                                              ## split line into columns
                try:
                    chi2Value = float(SplittedLine[1])                                      ## get chi2 value
                except IndexError:
                    chi2Value = numpy.nan
                if (not numpy.isnan(chi2Value)):
                    if (chi2Value <= BestChi2Value):                                        ## check, if current chi2 value is the best one
                        BestChi2Value = chi2Value                                           ## if yes, save chi2 value
                        FileNameBestChi2 = files                                            ## and corresponding file name

                        # Debug:
                        # print ("chi2Value = ", chi2Value)
                        # print ("FileNameBestChi2 = ", FileNameBestChi2)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get phrase identifying the used algorithm for the best chi2 value
        BestAlgorithm = ""
        BestAlgorithm = FileNameBestChi2.replace(".log.chi2", "")
        BestAlgorithm = BestAlgorithm.replace("fit__", "")                                  ## bug fix???

        # Debug:
        # print ("BestAlgorithm = ", BestAlgorithm)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## find molfit file
        for files in listing:                                                               ## loop over all files in the current job directory
            if (files.endswith(".out.molfit")):                                             ## we're interested only in the molfit files
                i = files.find(BestAlgorithm)
                if (i > (-1)):


                    ## convert column and hydrogen column density (if given for each component) back to linear scale
                    if (ConvertLogLinFlag):
                        LogLinearFlag = "linear"
                        task_myXCLASS.ConvertNtotToLogScale(JobDir + files, LogLinearFlag, ConverSQLElow = True)


                    ## read in molfit file and check, if fitted parameter values are within ranges
                    LocalMolfitFileName = JobDir + files
                    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(LocalMolfitFileName)

                    # Debug:
                    # print ("\n\nLocalMolfitFileName = ", LocalMolfitFileName)
                    # print ("MolfitFileForEachMolecule = ", MolfitFileForEachMolecule)


                    ## analyze molfit file content
                    task_LineIdentification.CreateMolfitFileInNewFormat(LocalMolfitFileName, MolfitFileForEachMolecule, LocalMinColumnDensityAbs, \
                                                                        LocalMinColumnDensityEmis, SQLParameters = SQLParameters)


                    ## read in new output files
                    MolfitFile = open(LocalMolfitFileName)
                    BestMolfitFileContent = MolfitFile.readlines()
                    MolfitFile.close()

                    # Debug:
                    # print ("\n\nfiles = ", files)
                    # print ("BestMolfitFileContent = ", BestMolfitFileContent)


                    ## exit loop
                    break


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## find optimized iso ratio file
        for files in listing:                                                               ## loop over all files in the current job directory
            if (files.startswith("myXClass_isoratios") and files.endswith(".out.input")):
                i = files.find(BestAlgorithm)
                if (i > (-1)):
                    IsoRatioFile = open(JobDir + files)
                    BestIsoRatioFileContent = IsoRatioFile.readlines()
                    IsoRatioFile.close()
                    break


        ## special handling for LM
        if (NumMolfitFiles == 1 and BestAlgorithm.find("LM") > (-1)):
            BestAlgorithm = "LM.out"

        # Debug:
        # print ("BestAlgorithm = ", BestAlgorithm)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## find xml file which corresponds to molfit file
        ListOfParamXMLFilesNames = []
        for files in listing:                                                               ## loop over all files in the current job directory
            if (files.endswith(".out.xml")):                                                ## we're interested only in the molfit files
                i = files.find(BestAlgorithm)
                if (i > (-1)):
                    ListOfParamXMLFilesNames.append(files)
        p0 = self.ReadParmXMLFile(JobDir, ListOfParamXMLFilesNames)

        # Debug:
        # print ("p0 = ", p0)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## find model data file(s)
        yModInt = []
        for RangeID in range(len(self.RangeParam)):
            LocalASCIIDataFileName = ""
            for LocalFileName in listing:
                if (LocalFileName.startswith("obs-spectrum__{:d}.{:s}".format(RangeID + 1, BestAlgorithm))):
                    LocalASCIIDataFileName = JobDir + LocalFileName
                    break
            ModeledRangeSpectrum = numpy.loadtxt(LocalASCIIDataFileName)
            if (RangeID == 0):
                yModInt = copy.deepcopy(ModeledRangeSpectrum[:, 1])
            else:
                yModInt = numpy.concatenate((yModInt, ModeledRangeSpectrum[:, 1]), axis = None)

        # Debug:
        # print ("\nBestChi2Value = ", BestChi2Value)
        # print ("p0 = ", p0)
        # print ("ListOfParamXMLFilesNames = ", ListOfParamXMLFilesNames)
        # sys.exit(0)


        ## define return dictionary
        BestResultsDict['chi2'] = BestChi2Value
        BestResultsDict['params'] = p0
        BestResultsDict['residuals'] = yModInt
        return BestResultsDict
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## (normal fitting only): read parameter values from xml file
    ##
    def ReadParmXMLFile(self, PixelDirectory, ListOfParamXMLFiles, IncludeAllParamFlag = False):
        """

    input parameters:
    -----------------

        - PixelDirectory:       current pixel directory

        - ListOfParamXMLFiles:  list of xml files

        - IncludeAllParamFlag:  (optional) flag indicating if fitted and non-fitted parameters are stored


    output parameters:
    ------------------

        - ParameterList:        list of fitted parameters with param. names

        """

        # Debug:
        # print ("\nPixelDirectory =", PixelDirectory)
        # print ("ListOfParamXMLFiles = ", ListOfParamXMLFiles)
        # print ("IncludeAllParamFlag = ", IncludeAllParamFlag)


        ## initialize return parameter
        p0 = []


        ## initialize some internal parameters
        LocalKnownParameterNames = []
        IsoMolecule = ""
        IsoMaster = ""


        ## check, if a xml file is given
        ParameterList = []
        if (len(ListOfParamXMLFiles) == 0):
            return p0
        else:


            ## read total contents of xml file
            output_file = open(PixelDirectory + ListOfParamXMLFiles[0])
            output_file_contents = output_file.readlines()
            output_file.close()


            ## get value of fitted parameter
            FittedParameter = False
            LogLinearFlag = False
            CurrentNumberFittedParameter = 0
            CurrentMolecule = ""
            ComponentCounter = 0
            NameFlag = False
            ParameterName = None
            for line in output_file_contents:
                StrippedLine = line.strip()

                # Debug:
                # print ("StrippedLine = ", StrippedLine)


                ## determine name of parameter
                i = StrippedLine.find("<name>")
                if (i > (-1)):
                    j = StrippedLine.find("</name>")
                    ParameterName = StrippedLine[i + 6:j].strip()
                    if (ParameterName in ["Molecule_Name"]):
                        ComponentCounter = 0
                        NameFlag = True
                    else:
                        NameFlag = False
                    if (ParameterName in ["nHFlag"]):                                       ## the occurrence of parameter "nHFlag" is used to
                        ComponentCounter += 1                                               ## determine the index of the current component and molecule


                ## get value of parameter
                i = StrippedLine.find("<value>")
                if (i > (-1)):
                    j = StrippedLine.find("</value>")


                    ## check, if name of molecule is defined
                    if (NameFlag):
                        CurrentMolecule = StrippedLine[i + 7:j].strip()
                        # CurrentMolecule = CurrentMolecule.replace(";", "_")
                        # CurrentMolecule = CurrentMolecule.replace(",", "_")
                        # CurrentMolecule = CurrentMolecule.replace("'", "_")
                        # CurrentMolecule = CurrentMolecule.replace("(", "_")
                        # CurrentMolecule = CurrentMolecule.replace(")", "_")
                        # CurrentMolecule = CurrentMolecule.strip("_")


                    ## check, if name of isotopologue or iso master is defined
                    elif (ParameterName in ["IsoMolecule", "IsoMasters"]):
                        LocalTestName = StrippedLine[i + 7:j].strip()
                        # LocalTestName = LocalTestName.replace(";", "_")
                        # LocalTestName = LocalTestName.replace(",", "_")
                        # LocalTestName = LocalTestName.replace("'", "_")
                        # LocalTestName = LocalTestName.replace("(", "_")
                        # LocalTestName = LocalTestName.replace(")", "_")
                        # LocalTestName = LocalTestName.strip("_")
                        if (ParameterName in ["IsoMolecule"]):
                            IsoMolecule = LocalTestName.strip()
                        elif (ParameterName in ["IsoMasters"]):
                            IsoMaster = LocalTestName.strip()


                ## check, if current line describes a fit parameter
                i = StrippedLine.find("Parameter fit=" + chr(34) + "true" + chr(34))
                j = StrippedLine.find("Parameter fit=" + chr(34) + "false" + chr(34))
                if (i > (-1) or (j > (-1) and IncludeAllParamFlag)):
                    FittedParameter = True
                    CurrentNumberFittedParameter += 1
                else:


                    ## continue here, if current parameter block descibes a fit parameter
                    if (FittedParameter):


                        ## convert column densities to linear values
                        ## check, if parameter value is given
                        i = StrippedLine.find("<name>")
                        if (i > (-1)):
                            j = StrippedLine.find("</name>")
                            ParameterName = StrippedLine[i + 6:j]
                            ParameterName = ParameterName.strip()


                            ## modify name of non-standard XCLASS parameters
                            # if (ParameterName == "IsoRatio"):
                            #     countOccurrence = 1
                            #     for name in LocalKnownParameterNames:
                            #         if (name.strip() == ParameterName):
                            #             countOccurrence += 1
                            #     LocalKnownParameterNames.append(ParameterName)
                            #     ParameterName = ParameterName + "__" + IsoMolecule + "__" + IsoMaster + "__" + str(countOccurrence)
                            # else:
                            #     LocalKnownParameterNames.append(ParameterName)


                            ## check new parameter name
                            if (ParameterName in ["N_tot", "nHcolumn_cont_dust", "nHcolumn", "EM_RRL"]):
                                LogLinearFlag = True
                            else:
                                LogLinearFlag = False


                        ## check, if parameter value is given
                        i = StrippedLine.find("<value>")
                        if (i > (-1)):
                            j = StrippedLine.find("</value>")
                            value = StrippedLine[i + 7:j].strip()
                            if (LogLinearFlag):
                                value = 10.0**float(value)

                            # Debug:
                            # print ("value = ", value)


                            ## append parameter values of current pixel to parameter list
                            if (ParameterName in ["IsoRatio"]):
                                ParameterList.append([IsoMolecule, (-1), ParameterName, value])


                        ## check, if parameter block is closed
                        i = StrippedLine.find("</Parameter>")
                        if (i > (-1)):
                            FittedParameter = False

        # Debug:
        # print ("ParameterList = ", ParameterList)


        ## bring fit parameters in correct order
        p0 = []
        for ParamID in range(self.NumFitParameters):

            # Debug:
            # print ("ParamID, self.FitParameterNames[ParamID] = ", ParamID, self.FitParameterNames[ParamID])


            ## get some informations for current parameter
            ## self.FitParameterNames[i] = [LocalMolecule, LocalLineID, Name, LogFlag]
            ## self.FitParameterNames[i] = [LocalIsotopologue, (-1), "IsoRatio", False]
            NameOfParameter = self.FitParameterNames[ParamID][2]
            NameOfMolecule = self.FitParameterNames[ParamID][0]
            CurrentComponent = self.FitParameterNames[ParamID][1] + 1

            # Debug:
            # print ("\n\nParamID = ", ParamID)
            # print ("NameOfMolecule = ", NameOfMolecule)
            # print ("CurrentComponent = ", CurrentComponent)


            ## ParameterList.append([CurrentMolecule, ComponentCounter, ParameterName, value])
            for LocalParameterList in ParameterList:
                ParameterNameXML = LocalParameterList[2]
                CurrentMoleculeXML = LocalParameterList[0]
                ComponentCounterXML = LocalParameterList[1]
                if (NameOfMolecule == CurrentMoleculeXML and NameOfParameter == ParameterNameXML):
                    if (CurrentComponent == ComponentCounterXML or CurrentComponent == 0):
                        p0.append(LocalParameterList[3])
                        break

        # Debug:
        # print ("p0 = ", p0)


        ## define return variable
        return p0
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write results to job directory
    def ExportResults(self, LocalParamMapIter, IncludeAlwaysAlgFlag = True):
        """

    input parameters:
    -----------------

        - LocalParamMapIter:            current smooth iteration

        - IncludeAlwaysAlgFlag:         (optional) flag indicating, that phrase describing used algorithm
                                        is always included in the names of the output files (default: True)

    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("LocalParamMapIter = ", LocalParamMapIter)
        # print ("IncludeAlwaysAlgFlag = ", IncludeAlwaysAlgFlag)


        ## defines extensions of output file nams describing current fit algorithm
        FitExtOutputFileName = ""
        if (self.FitAlgSettings is not None and self.FastFitFlag):
            if (len(self.FitAlgSettings) > 1):
                FitExtOutputFileName = "__" + self.Fit_Algorithm + "__"
        if (LocalParamMapIter > (-1)):
            FitExtOutputFileName += "___Iter__" + str(LocalParamMapIter + 1)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write chi2 map(s) as FITS image to file


        ## modify header
        if (not self.NoFITSCubeFlag):
            Chi2HeaderNew = self.ParameterMapHeader.copy()
            Chi2HeaderNew["software"] = (self.NameOfFunction, 'Produced with XCLASS')


            ## create new slice
            Chi2MapObject = Slice(value = self.Chi2Map[:, :], wcs = self.ParameterMapWCS, header = Chi2HeaderNew)


            ## add unit to slice instance
            Chi2MapObject = Chi2MapObject * u.K**2


            ## create new slice and write it to file
            OutputChi2MapFileName = self.myXCLASSMapFitJobDir + "BestResult___chi2values" + FitExtOutputFileName
            Chi2MapObject.write(OutputChi2MapFileName + ".fits")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write parameter maps to file


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## only for sinlge spectra fitting procedures: export results to single output files
        if (self.NoFITSCubeFlag):
            Localypos = 0
            Localxpos = 0


            ## write results to new molfit file
            OutputMolfitFileName = self.myXCLASSMapFitJobDir + os.path.basename(self.MolfitsFileName)
            OutputMolfitFileName = OutputMolfitFileName.replace(".molfit", "")
            if (len(self.FitAlgSettings) > 1 or IncludeAlwaysAlgFlag):
                OutputMolfitFileName += "__{:s}".format(self.Fit_Algorithm)
            self.UpdateMolfitFile(OutputMolfitFileName + ".out.molfit", Localypos, Localxpos, ExportFlag = True)


            ## write results to new iso ratio file
            if (self.IsoFlag):
                OutputIsoRatioFileName = self.myXCLASSMapFitJobDir + os.path.basename(self.IsoRatioFileName)
                OutputIsoRatioFileName = OutputIsoRatioFileName.replace(".dat", "")
                if (len(self.FitAlgSettings) > 1 or IncludeAlwaysAlgFlag):
                    OutputIsoRatioFileName += "__{:s}".format(self.Fit_Algorithm)
                self.UpdateIsoRatioFile(OutputIsoRatioFileName + ".out.dat", Localypos, Localxpos, ExportFlag = True)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## MapFit only: write parameter maps to job directory
        else:
            for ParamID in range(self.NumFitParameters):


                ## get parameters for MapFit function
                ## get some informations for current parameter
                ## self.FitParameterNames[i] = [LocalMolecule, LocalLineID, Name, LogFlag]
                ## self.FitParameterNames[i] = [LocalIsotopologue, (-1), "IsoRatio", False]
                NameOfMolecule = self.FitParameterNames[ParamID][0]
                NameOfMolecule = task_LineIdentification.MoleculeFileName(NameOfMolecule)
                CurrentComponent = self.FitParameterNames[ParamID][1]
                NameOfParameter = self.FitParameterNames[ParamID][2]
                LogFlag = self.FitParameterNames[ParamID][3]


                ## modify header
                M1HeaderParam = self.ParameterMapHeader.copy()
                # del M1HeaderParam['SLICE']
                del M1HeaderParam['COMMENT']
                M1HeaderParam["software"] = (self.NameOfFunction, 'Produced with XCLASS')


                ## create new slice
                if (LogFlag and self.FastFitFlag):
                    ParamMapObject = Slice(value = 10.0**self.FitParameterMaps[:, :, ParamID], wcs = self.ParameterMapWCS, header = M1HeaderParam)
                else:
                    ParamMapObject = Slice(value = self.FitParameterMaps[:, :, ParamID], wcs = self.ParameterMapWCS, header = M1HeaderParam)


                ## add unit to slice instance
                UnitParm = task_myXCLASS.GetUnitParameter(NameOfParameter)
                ParamMapObject = ParamMapObject * u.Unit(UnitParm)


                ## create file name for current fit parameter map
                OutputParameterMapFileName = self.myXCLASSMapFitJobDir + "BestResult___parameter__" + NameOfParameter
                OutputParameterMapFileName += "___molecule__{:s}".format(NameOfMolecule)
                if (CurrentComponent > (-1)):
                    OutputParameterMapFileName += "___component__{:d}".format(CurrentComponent + 1)
                if (self.NameOfFunction == "myXCLASSMapRedoFit"):
                    OutputParameterMapFileName += "__RedoFit"
                OutputParameterMapFileName += FitExtOutputFileName

                # Debug:
                # print ("\nOutputParameterMapFileName = ", OutputParameterMapFileName)
                # print ("self.FitParameterMaps[:, :, ParamID] = ", self.FitParameterMaps[:, :, ParamID])
                # print ("self.ParameterMapWCS = ", self.ParameterMapWCS)
                # print ("NameOfParameter, UnitParm = ", NameOfParameter, UnitParm)


                ## write parameter map to file
                ParamMapObject.write(OutputParameterMapFileName + ".fits")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write cubes with synthetic spectra and chi2 functions to file
        SingleSpectraDict = {}
        SingleChi2Dict = {}
        for LocalRangeID, ObsXMLParameterList in enumerate(self.RangeParam):


            ## get dictionary with parameters for the current range
            ObsDataFileIndex = ObsXMLParameterList['ObsDataFileIndex']


            ## get the original obs. data FITS cube with the corresponding units for intensity and spectral axis
            LocalObsDataCube = self.ListOfCubes[LocalRangeID]
            OrigIntUnit = self.OrigUnits[ObsDataFileIndex][0]


            ##********************************************************************************************************************************************
            ## handle non-cube export as well
            if (self.NoFITSCubeFlag):


                ## get spectrum of curent range
                LocalSpectrum = self.SyntheticSpectraCubeList[LocalRangeID][:, 0, 0]
                LocalSpectrum = numpy.vstack((LocalObsDataCube[:, 0], LocalSpectrum)).T
                if (self.Fit_Chi2):
                    LocalChi2Spectrum = self.Chi2SpectraCubeList[LocalRangeID][:, 0, 0]
                    LocalChi2Spectrum = numpy.vstack((LocalObsDataCube[:, 0], LocalChi2Spectrum)).T
                if (not ObsDataFileIndex in SingleSpectraDict.keys()):
                    SingleSpectraDict[ObsDataFileIndex] = LocalSpectrum
                    if (self.Fit_Chi2):
                        SingleChi2Dict[ObsDataFileIndex] = LocalChi2Spectrum
                else:
                    # SingleSpectraDict[ObsDataFileIndex] = numpy.concatenate((SingleSpectraDict[ObsDataFileIndex], LocalSpectrum), axis = 1)
                    SingleSpectraDict[ObsDataFileIndex] = numpy.vstack((SingleSpectraDict[ObsDataFileIndex], LocalSpectrum))
                    if (self.Fit_Chi2):
                        # SingleChi2Dict[ObsDataFileIndex] = numpy.concatenate((SingleChi2Dict[ObsDataFileIndex], LocalChi2Spectrum), axis = 1)
                        SingleChi2Dict[ObsDataFileIndex] = numpy.vstack((SingleChi2Dict[ObsDataFileIndex], LocalChi2Spectrum))


            ##********************************************************************************************************************************************
            ## handle cube export
            else:


                ## get spatial coordinates and header from first FITS cube
                LocalObsDataCubeHeader = LocalObsDataCube.header
                LocalObsDataCubeHeaderCopy = LocalObsDataCubeHeader.copy()
                LocalObsDataCubeWCS = LocalObsDataCube.wcs

                # Debug:
                # print ("LocalObsDataCubeHeader = ", LocalObsDataCubeHeader)
                # print ("LocalObsDataCubeWCS = ", LocalObsDataCubeWCS)


                ## convert synthetic cubes (spectra + chi^2) to original units and write to file
                LocalDataCubeRAW = self.SyntheticSpectraCubeList[LocalRangeID]


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## convert Kelvin to Jy / beam, if necessary
                JyBeamFlag = False
                LowerOrigIntUnit = str(OrigIntUnit).lower()
                LowerOrigIntUnit = LowerOrigIntUnit.replace(" ", "").strip()
                if (LowerOrigIntUnit in ["jy/beam", "beam-1jy"]):
                    LocalTelescopeSize = ObsXMLParameterList['TelescopeSize']
                    LocalBMIN = ObsXMLParameterList['BMIN']
                    LocalBMAJ = ObsXMLParameterList['BMAJ']
                    LocalInterFlag = ObsXMLParameterList['InterFlag']


                    ## for interferometer get correct minar and major axis
                    InterFlag = False
                    if (LocalInterFlag == 1):
                        InterFlag = True
                        LocalBMIN = LocalObsDataCube.beam.major.to(u.arcsec).value
                        LocalBMAJ = LocalObsDataCube.beam.minor.to(u.arcsec).value


                    ## compute conversion factors from Kelvin to Jy/beam
                    JyBeamFlag = True
                    ConvertJyPerBeam2Kelvin, ConvertKelvin2JyPerBeam = JanskyPerBeamInKelvin(LocalTelescopeSize, LocalBMAJ, LocalBMIN, \
                                                                                            InterFlag, LocalObsDataCube.spectral_axis.value)
                    for i, LocalConvertKelvin2JyPerBeam in enumerate(ConvertKelvin2JyPerBeam):
                        LocalDataCubeRAW[i, :, :] = LocalDataCubeRAW[i, :, :] * LocalConvertKelvin2JyPerBeam


                ## check, if intensities are given in another unit than Kelvin or Jy/beam
                ListOfDataCubes = [LocalDataCubeRAW]
                if (self.Fit_Chi2):
                    ListOfDataCubes.append(self.Chi2SpectraCubeList[LocalRangeID])
                for LocalDataCubeID, LocalDataCubeRAW in enumerate(ListOfDataCubes):

                    # Debug:
                    # print ("\n\nLocalDataCubeID = ", LocalDataCubeID)
                    # print ("LocalDataCube = ", LocalDataCube)


                    ## modify FITS header
                    LocalObsDataCubeHeaderCopy["software"] = (self.NameOfFunction, 'Produced with XCLASS')


                    ## creating a SpectralCube instance
                    SpecCubeObject = SpectralCube(data = LocalDataCubeRAW, wcs = LocalObsDataCubeWCS, header = LocalObsDataCubeHeaderCopy, \
                                                  use_dask = self.daskFlag)
                    SpecCubeObject.allow_huge_operations = True


                    ## add current units to SpectralCube instance
                    if (LocalDataCubeID == 0):
                        if (JyBeamFlag):
                            SpecCubeObject = SpecCubeObject * (u.Jy / u.beam)
                        else:
                            SpecCubeObject = SpecCubeObject * u.K
                    else:
                        SpecCubeObject = SpecCubeObject * u.K**2

                    # Debug:
                    # print ("\nRAW:    SpecCubeObject = ", SpecCubeObject)
                    # print ("OrigIntUnit = ", OrigIntUnit)
                    # print ("LocalObsDataCubeHeaderCopy = ", LocalObsDataCubeHeaderCopy)
                    # print ("LocalObsDataCubeWCS = ", LocalObsDataCubeWCS)
                    # print ("SpecCubeObject.header = ", SpecCubeObject.header)
                    # print ("SpecCubeObject.unit = ", SpecCubeObject.unit)
                    # print ("SpecCubeObject.spectral_axis = ", SpecCubeObject.spectral_axis)
                    # print (SpecCubeObject)


                    ## convert intensities back to original units
                    ## Here: LocalDataCubeID == 0 indicates the cube with synth. spectra,  LocalDataCubeID == 1 represents the chi2 cube
                    if (LocalDataCubeID == 0 and (not OrigIntUnit in ["K"])):
                        SpecCubeObject = SpecCubeObject.to(u.Unit(OrigIntUnit))

                    # Debug:
                    # print ("\nNew BUNIT:  SpecCubeObject = ", SpecCubeObject)


                    ## convert intensities back to original units
                    ## restfreq = [restfreq, velConv]
                    OrigspecUnit = self.OrigUnits[ObsDataFileIndex][1]
                    if (not OrigspecUnit in ["MHz"]):
                        OrigspecUnitLower = str(OrigspecUnit).strip().lower()
                        if (OrigspecUnitLower in ["m / s", "m/s", "km / s", "km/s"]):
                            restfreq = self.OrigRestFreq[LocalRangeID][0]
                            velConv = self.OrigRestFreq[LocalRangeID][1]
                            SpecCubeObject = SpecCubeObject.with_spectral_unit(u.Unit(OrigspecUnit), velocity_convention = velConv, \
                                                                               rest_value = restfreq * u.Hz)
                        else:
                            SpecCubeObject = SpecCubeObject.with_spectral_unit(u.Unit(OrigspecUnit))

                    # Debug:
                    # print ("\nNew CUNIT3:  SpecCubeObject = ", SpecCubeObject)


                    ## get some obs. xml parameter for current range and prepare output file name
                    NumberFrequencyRanges = ObsXMLParameterList['NumberFrequencyRanges']
                    if (NumberFrequencyRanges > 1):
                        FreqMin = ObsXMLParameterList['FreqMin']
                        FreqMax = ObsXMLParameterList['FreqMax']
                        ExtString = "__{:.4f}_-_{:.4f}__MHz".format(FreqMin, FreqMax)
                    else:
                        ExtString = ""
                    LocalObsDataFileName = self.ExpFileList[ObsDataFileIndex][1]
                    OutputObsDataFileName = os.path.basename(LocalObsDataFileName)
                    OutputObsDataFileName = OutputObsDataFileName.replace(".fits", "")
                    OutputObsDataFileName = OutputObsDataFileName.replace(".FITS", "")
                    OutputObsDataFileName = OutputObsDataFileName.replace(".Fits", "")
                    OutputObsDataFileName = self.myXCLASSMapFitJobDir + OutputObsDataFileName + ExtString + FitExtOutputFileName

                    # Debug:
                    # print ("OutputObsDataFileName = ", OutputObsDataFileName)


                    ## write cube describing pixel spectra to file
                    if (LocalDataCubeID == 0):
                        SpecCubeObject.write(OutputObsDataFileName + "__model.out.fits")
                    else:
                        SpecCubeObject.write(OutputObsDataFileName + "__model.chi2.out.fits")


        ##************************************************************************************************************************************************
        ## export single spectra
        if (self.NoFITSCubeFlag):
            for ObsDataFileIndex in SingleSpectraDict.keys():


                ## get sepctrum
                LocalSpectrum = SingleSpectraDict[ObsDataFileIndex]


                ## define name of output file
                LocalObsDataFileName = self.ExpFileList[ObsDataFileIndex][1]
                OutputObsDataFileName = self.myXCLASSMapFitJobDir + os.path.basename(LocalObsDataFileName)
                OutputObsDataFileName = OutputObsDataFileName.replace(".dat", "")
                if (len(self.FitAlgSettings) > 1 or IncludeAlwaysAlgFlag):
                    OutputObsDataFileName += "__{:s}".format(self.Fit_Algorithm)


                ## export spectrum
                numpy.savetxt(OutputObsDataFileName + ".out.dat", LocalSpectrum)
                if (self.Fit_Chi2):
                    LocalChi2Spectrum = SingleChi2Dict[ObsDataFileIndex]
                    numpy.savetxt(OutputObsDataFileName + ".chi2.out.dat", LocalChi2Spectrum)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## create plot
            if (self.NoFITSCubeFlag and self.Fit_AlgDict["PlotFlag"]):


                ## initialize figure
                NumObsDataFiles = len(self.ExpFileList)
                pylab.subplots_adjust(hspace = 0.45, wspace = 0.03, bottom = 0.02, top = 0.98)
                fig, layers = plt.subplots(NumObsDataFiles, 2, constrained_layout = True)
                fig.set_figwidth(15)
                fig.set_figheight(10)
                for ObsDataFileIndex in SingleSpectraDict.keys():
                    if (NumObsDataFiles == 1):
                        LeftLayer = layers[0]
                        RightLayer = layers[1]
                    else:
                        LeftLayer = layers[ObsDataFileIndex, 0]
                        RightLayer = layers[ObsDataFileIndex, 1]


                    # ListRanges = []
                    # for LocalRangeID, ObsXMLParameterList in enumerate(self.RangeParam):
                    #     LocalObsDataFileIndex = ObsXMLParameterList['ObsDataFileIndex']
                    #     if (LocalObsDataFileIndex == ObsDataFileIndex):
                    #         LocalObsDataCube = self.ListOfCubes[LocalRangeID]
                    #         ListRanges.append([numpy.nanmin(LocalObsDataCube[:, 0]), numpy.nanmax(LocalObsDataCube[:, 0])])


                    ## add obs. data to plot
                    xoffset = 0
                    LastObsDataFileIndex = (-1)
                    for LocalRangeID, ObsXMLParameterList in enumerate(self.RangeParam):
                        LocalObsDataFileIndex = ObsXMLParameterList['ObsDataFileIndex']
                        if (LocalObsDataFileIndex == ObsDataFileIndex):
                            LocalObsDataCube = self.ListOfCubes[LocalRangeID]
                            xpoints = LocalObsDataCube[:, 0]
                            # i = len(xpoints)
                            # xpoints = numpy.arange(0 + xoffset, i + xoffset, 1)
                            # LeftLayer.axvline(x = i + xoffset)
                            # xoffset += i
                            if (LastObsDataFileIndex != LocalObsDataFileIndex):
                                LastObsDataFileIndex = LocalObsDataFileIndex
                                LeftLayer.plot(xpoints, LocalObsDataCube[:, 1], '-', color = 'black', \
                                            linewidth = 1.0, drawstyle = 'steps-mid', label = "data")
                            else:
                                LeftLayer.plot(xpoints, LocalObsDataCube[:, 1], '-', color = 'black', \
                                            linewidth = 1.0, drawstyle = 'steps-mid')


                    ## plot observation data and fit function together in one diagram
                    LocalSpectrum = SingleSpectraDict[ObsDataFileIndex]
                    xpoints = LocalSpectrum[:, 0]
                    # i = len(xpoints)
                    # xpoints = numpy.arange(0, i, 1)
                    LeftLayer.plot(xpoints, LocalSpectrum[:, 1], '-', color = 'red', \
                                   linewidth = 1.0, label = 'fit')
                    LeftLayer.grid(True)                                                    ## add grid to left panel
                    LeftLayer.set_xlabel("Frequency (MHz)")
                    LeftLayer.set_ylabel("Brightness Temperature (K)")
                    # LeftLayer.set_xlim(numpy.nanmin(LocalSpectrum[:, 0]), numpy.nanmax(LocalSpectrum[:, 0]))
                    # LeftLayer.xaxis.set_major_formatter(pylab.matplotlib.ticker.FormatStrFormatter('%.5e'))
                    LeftLayer.set_xlim(numpy.nanmin(LocalSpectrum[:, 0]), numpy.nanmax(LocalSpectrum[:, 0]))
                    LeftLayer.legend()                                                      ## add legend to left panel


                    ## plot chi^2 function
                    LocalChi2Spectrum = SingleChi2Dict[ObsDataFileIndex]
                    RightLayer.plot(LocalChi2Spectrum[:, 0], LocalChi2Spectrum[:, 1], '-', color = 'blue', \
                                    linewidth = 1.0, label = r'$\chi^2$')
                    RightLayer.grid(True)                                                   ## add grid to left panel
                    RightLayer.set_xlabel("Frequency (MHz)")
                    RightLayer.set_ylabel(r"$\chi^2$ (K$^2$)")
                    RightLayer.yaxis.set_label_position("right")
                    RightLayer.yaxis.set_ticks_position("right")
                    RightLayer.set_xlim(numpy.nanmin(LocalChi2Spectrum[:, 0]), numpy.nanmax(LocalChi2Spectrum[:, 0]))
                    # RightLayer.xaxis.set_major_formatter(pylab.matplotlib.ticker.FormatStrFormatter('%.5e'))
                    RightLayer.legend()                                                     ## add legend to left panel


                ## save figure
                OutputFileName = "{:s}final-plot.out".format(self.myXCLASSMapFitJobDir)
                if (len(self.FitAlgSettings) > 1 or IncludeAlwaysAlgFlag):
                    OutputFileName += "__{:s}".format(self.Fit_Algorithm)
                pylab.savefig(OutputFileName + ".{:s}".format(self.Fit_AlgDict["PlotFileFormat"]), bbox_inches = 'tight')
                pylab.draw()
                matplotlib.pyplot.close(fig)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## smooth parameter maps
    def SmoothParameterMaps(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## write parameter maps to job directory
        for ParamID in range(self.NumFitParameters):


            ## replace NaN in LocalMap with arithmetic mean of arithmetic means along x-and y-axis, respectively
            LocalMap = self.FitParameterMaps[:, :, ParamID]
            xm = numpy.nanmean(LocalMap)                                                    ## compute the arithmetic mean along y-axis, ignoring NaNs
            nans = numpy.isnan(LocalMap)
            LocalMapNoNaN = numpy.where(nans, xm, LocalMap)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## smooth current parameter map


            ## use scipy's multidimensional Gaussian filter
            if (self.ParamSmoothMethod == "gaussian"):
                sigma = [self.ParamSmoothMethodParam, self.ParamSmoothMethodParam]
                LocalMap = gaussian_filter(LocalMapNoNaN, sigma)


            ## use scipy's multidimensional uniform filter
            elif (self.ParamSmoothMethod == "uniform"):
                LocalMap = uniform_filter(LocalMapNoNaN, size = self.ParamSmoothMethodParam)


            ## use scipy's multidimensional median filter
            elif (self.ParamSmoothMethod == "median"):
                LocalMap = median_filter(LocalMapNoNaN, size = self.ParamSmoothMethodParam)

            # Debug:
            # print ("\n\nParamID = ", ParamID)
            # print ("LocalMap.shape = ", LocalMap.shape)
            # print ("LocalMap = ", LocalMap)
            # print ("self.RangeList[ParamID] = ", self.RangeList[ParamID])


            ## replace nan with mean values
            # xm = numpy.nanmean(LocalMap)                                                    ## compute the arithmetic mean along y-axis, ignoring NaNs
            # nans = numpy.isnan(LocalMap)
            # LocalMap = numpy.where(nans, xm, LocalMap)


            ## check, that pixel values are within given limits
            LocalMap[:, :] = numpy.where(LocalMap < self.RangeList[ParamID][0], self.RangeList[ParamID][0], LocalMap)
            LocalMap[:, :] = numpy.where(LocalMap > self.RangeList[ParamID][1], self.RangeList[ParamID][1], LocalMap)


            ## store smoothed parameter maps back to variable
            self.FitParameterMaps[:, :, ParamID] = LocalMap


        ## we're done
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define routines for mouse handling
##
class PlottingGUI(QtWidgets.QMainWindow):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize
    def __init__(self, JobDir):
        """

    input parameters:
    -----------------

        - JobDir:                       path of job directory containing cubes


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("JobDir = ", JobDir)


        ## hack
        super(PlottingGUI, self).__init__()


        ## copy input parameters to class variables
        self.JobDir = JobDir                                                                ## path of job directory containing cubes
        self.JobDirFileListing = None                                                       ## list of all FITS files in the selected job directory
        self.ListObsDataFileNames = None                                                    ## list of all obs. data files described in
                                                                                            ## obs. xml file with corresponding model files
        self.FoundObsDataFileFlag = None                                                    ## flag indicating, if obs. data files are available
        self.FITSImageFlag = False                                                          ## flag indicating that current FITS file is an image
        self.CurrentFITSFileName = ""                                                       ## name of current FITS file
        self.CurrentModelFITSFileName = ""                                                  ## name of current model FITS file
        self.FITSObsCube = []                                                               ## fits data cube
        self.FITSChi2Cube = None                                                            ## FITS cubes for chi2 spectra
        self.xPixelID = None                                                                ## x-pixel index of current pixel for displayed spectrum
        self.yPixelID = None                                                                ## y-pixel index of current pixel for displayed spectrum
        self.xPixelCoord = None                                                             ## x-pixel coordinate of current pixel for displayed spectrum
        self.yPixelCoord = None                                                             ## y-pixel coordinate of current pixel for displayed spectrum
        self.xCursorPixelID = 0                                                             ## y-pixel index of current map cursor position
        self.yCursorPixelID = 0                                                             ## x-pixel index of current map cursor position
        self.xFixedCursorPixelID = None                                                     ## y-pixel index of current map cursor position
        self.yFixedCursorPixelID = None                                                     ## x-pixel index of current map cursor position
        self.MapCMAP = "black"                                                              ## define color of map cursor
        self.MapCMAPDefault = "winter"
        self.ParameterName = None                                                           ## name of parameter map
        self.LogFlag = False                                                                ## flag for log. scale
        self.xPosMap = None                                                                 ## x-pos. of map cursor
        self.yPosMap = None                                                                 ## y-pos. of map cursor
        self.FrequencyAxis = []                                                             ## list of all frequency of map
        self.MapCurrFreq = 1.0                                                              ## frequency of the current channel map
        self.MapFreqMin = 1.0                                                               ## lowest frequency of map
        self.MapFreqMax = 1.e9                                                              ## highest frequency of map
        self.MapFreqStep = 1.0                                                              ## step frequency of map
        self.DecAxis = []                                                                   ## ticks for axis
        self.DecAxisDelta = 0.0                                                             ## initialize delta of Dec axis
        self.RAAxis = []                                                                    ## ticks for axis
        self.RAAxisDelta = 0.0                                                              ## initialize delta of RA axis
        self.ObsSpectrum = None                                                             ## current obs. spectrum
        self.ComboList = None                                                               ## list for combo box
        self.colorbar = None                                                                ## colorbar object
        self.OnceFlag = True                                                                ## set once flag to add colorbar
        self.InitFlag = True                                                                ## define initialization flag
        self.RefreshMapOnlyFlag = False                                                     ## flag indicating that only map is refreshed
        self.AllMinInt = numpy.nan                                                          ## reset global minimum intensity
        self.AllMaxInt = numpy.nan                                                          ## reset global maximum intensity
        self.LockCursorPosFlag = False                                                      ## lock spectrum at cursor position or not
        self.ok = 0                                                                         ## set status variable
        self.lockElements = True                                                            ## flag for locking elements
        self.ChannelIndex = 0                                                               ## current channel index
        self.ListSpecCoord = []                                                             ## list of coordinates of selected spectra
        self.ListSpecData = []                                                              ## list of data of selected spectra
        self.SpectrumObsFlag = False                                                        ## flag indicating that obs. spectrum is displayed
        self.SpectrumModFlag = False                                                        ## flag indicating that model spectrum is displayed
        self.SpectrumChi2Flag = False                                                       ## flag indicating that chi2 spectrum is displayed

        # Debug:
        # print ("\nself.RangeList = ", self.RangeList)
        # print ("RangeListString = ", self.RangeListString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define new colormap "b" from ds9
        ds9b = {'red': lambda v : 4 * v - 1,
                'green': lambda v : 4 * v - 2,
                'blue': lambda v : numpy.select([v < 0.25, v < 0.5, v < 0.75, v <= 1], [4 * v, -4 * v + 2, 0, 4 * v - 3])}
        try:
            register_cmap('ds9b', data = ds9b)
            self.MapCMAPDefault = "ds9b"
        except:
            self.MapCMAPDefault = "winter"
        self.MapCMAP = self.MapCMAPDefault


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize some parameters
        self.ChannelIndex = 0
        self.LowestFreq = 1.e5                                                              ## set default min. frequency
        self.HighestFreq = 5.e5                                                             ## set default max. frequency


        ## if no job directory is given, open file dialog and make sure, that path of job directory ends with "/"
        self.JobDir = self.JobDir.strip()
        if (self.JobDir == ""):
            self.JobDir = QtWidgets.QFileDialog.getExistingDirectory(self, 'Select a job directory', )
            if (os.path.isdir(self.JobDir)):
                self.JobDir = self.JobDir.strip()
            else:
                self.close()
                sys.exit()
                return
        if (not self.JobDir.endswith("/")):
            self.JobDir += "/"


        ## print what you do
        print ("\n\nShow results of {:s}\n\n".format(chr(34) + self.JobDir + chr(34)))


        ## get list of FITS files in the job directory and add path and names of obs. data files from obs. xml file
        self.JobDirFileListing = []
        self.CurrentFITSFileName = ""
        self.CurrentModelFITSFileName = ""
        self.FoundObsDataFileFlag = False
        JobDirListing = os.listdir(self.JobDir)
        FirstObsDataFileName = ""
        ListPureObsDataFileNames = []
        for LocalFileName in JobDirListing:
            LowerLocalFileName = LocalFileName.strip()
            LowerLocalFileName = LowerLocalFileName.lower()
            if (LowerLocalFileName.endswith(".fits")):
                self.JobDirFileListing.append(self.JobDir + LocalFileName)
            elif (LowerLocalFileName.endswith(".xml")):
                ObsXMLFileName = LocalFileName
                ListObsDataFileNameRAW = task_MAGIX.GetXMLtagNEW(self.JobDir + ObsXMLFileName, "FileNamesExpFiles")
                for LocalObsDataFileName in ListObsDataFileNameRAW:
                    ListPureObsDataFileNames.append(LocalObsDataFileName[1])
                    self.JobDirFileListing.append(LocalObsDataFileName[1])
        self.JobDirFileListing.sort()                                                       ## sort list of files

        # Debug:
        # print ("self.JobDirFileListing = ", self.JobDirFileListing)


        ## find for each obs. data file, the corresponding model cubes
        if (len(ListPureObsDataFileNames) > 0):
            self.FoundObsDataFileFlag = True
            self.ExtOfBestModel = ""
            self.ListObsDataFileNames = []
            self.ListObsDataFileParams = []
            for LocalObsDataFileNameID, LocalObsDataFileName in enumerate(ListPureObsDataFileNames):
                PureObsDataFileName = os.path.basename(LocalObsDataFileName)
                LowerPureObsDataFileName = PureObsDataFileName.lower()
                LowerPureObsDataFileName = LowerPureObsDataFileName.strip()
                LowerPureObsDataFileName = LowerPureObsDataFileName.replace(".fits", "")

                # Debug:
                # print ("\n\nPureObsDataFileName = ", PureObsDataFileName)


                ## find model cubes for current obs. data files
                LocalListModelFiles = []
                for LocalFileNameID, LocalFileName in enumerate(self.JobDirFileListing):
                    PureLocalFileName = os.path.basename(LocalFileName)
                    PureLocalFileName = PureLocalFileName.strip()
                    LowerPureLocalFileName = PureLocalFileName.lower()

                    # Debug:
                    # print ("LowerPureLocalFileName = ", LowerPureLocalFileName)


                    ## get index of current obs. data file
                    if (PureObsDataFileName == PureLocalFileName and LocalObsDataFileNameID == 0):
                        self.ComboIndex = LocalFileNameID
                        self.CurrentFITSFileName = LocalFileName
                        self.FoundObsDataFileFlag = True


                    ## search for cubes describing chi^2
                    if (LowerPureLocalFileName.endswith(".chi2.out.fits")):
                        LowerPureLocalFileName = LowerPureLocalFileName.replace("__model.chi2.out.fits", "")
                        LowerPureLocalFileName = LowerPureLocalFileName.replace(".out.fits", "")
                        LowerPureLocalFileName = LowerPureLocalFileName.replace(".fits", "")
                        LowerPureLocalFileName = LowerPureLocalFileName.split("___iter__")
                        LowerPureLocalFileName = LowerPureLocalFileName[0].strip("_").strip()
                        if (LowerPureLocalFileName.find(LowerPureObsDataFileName) > (-1)):
                            LocalListModelFiles.append(LocalFileName)


                    ## search for cubes describing model function
                    elif (LowerPureLocalFileName.endswith(".out.fits")):
                        LowerPureLocalFileName = LowerPureLocalFileName.replace("___model-function-values.fits", "")
                        LowerPureLocalFileName = LowerPureLocalFileName.replace("__model.out.fits", "")
                        LowerPureLocalFileName = LowerPureLocalFileName.replace(".out.fits", "")
                        LowerPureLocalFileName = LowerPureLocalFileName.replace(".fits", "")
                        LowerPureLocalFileName = LowerPureLocalFileName.split("___iter__")
                        LowerPureLocalFileName = LowerPureLocalFileName[0].strip("_").strip()
                        if (LowerPureLocalFileName.find(LowerPureObsDataFileName) > (-1)):
                            LocalListModelFiles.append(LocalFileName)

                # Debug:
                # print ("LocalListModelFiles = ", LocalListModelFiles)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## find model file which describes last model
                MaxIter = None
                AllFreqMin = None
                BestModelID = None
                for LocalModelFileID, LocalModelFile in enumerate(LocalListModelFiles):
                    PureLocalFileName = os.path.basename(LocalModelFile)
                    PureLocalFileName = PureLocalFileName.strip()
                    LowerPureLocalFileName = PureLocalFileName.lower()
                    if (not LowerPureLocalFileName.endswith(".chi2.out.fits")):


                        ## find latest file from map iteration
                        if (LowerPureLocalFileName.find("___iter__") > (-1)):
                            SplittedLowerPureLocalFileName = LowerPureLocalFileName.split("__")
                            for ElementID, element in enumerate(SplittedLowerPureLocalFileName):
                                if (element.strip("_") == "iter" and ElementID < len(SplittedLowerPureLocalFileName)):
                                    CurrIter = int(SplittedLowerPureLocalFileName[ElementID + 1])
                                    if (MaxIter is None or CurrIter > MaxIter):
                                        MaxIter = CurrIter
                                        BestModelID = LocalModelFileID
                                        self.ExtOfBestModel = "___Iter__{:d}__model".format(CurrIter)
                                        break


                        ## find latest file from map iteration
                        elif (LowerPureLocalFileName.find("__mhz__") > (-1)):
                            LocalLowerPureLocalFileName = LowerPureLocalFileName.replace(LowerPureObsDataFileName, "")
                            LocalLowerPureLocalFileName = LocalLowerPureLocalFileName.replace("___model-function-values.fits", "")
                            LocalLowerPureLocalFileName = LocalLowerPureLocalFileName.replace("__model.out.fits", "")
                            LocalLowerPureLocalFileName = LocalLowerPureLocalFileName.replace(".out.fits", "")
                            LocalLowerPureLocalFileName = LocalLowerPureLocalFileName.replace(".fits", "")
                            LocalLowerPureLocalFileName = LocalLowerPureLocalFileName.split("___iter__")
                            LocalLowerPureLocalFileName = LocalLowerPureLocalFileName[0].strip("_").strip()
                            LocalExtOfBestModel = LocalLowerPureLocalFileName
                            LocalLowerPureLocalFileName = LocalLowerPureLocalFileName.replace("__mhz", "")
                            LocalLowerPureLocalFileName = LocalLowerPureLocalFileName.split("_-_")
                            try:
                                FreqMin = float(LocalLowerPureLocalFileName[0])
                            except:
                                FreqMin = None
                            try:
                                FreqMax = float(LocalLowerPureLocalFileName[1])
                            except:
                                FreqMax = None
                            if (AllFreqMin is None):
                                self.ExtOfBestModel = LocalExtOfBestModel.replace("__mhz", "__MHz")
                                BestModelID = LocalModelFileID
                            elif (AllFreqMin > FreqMin):
                                self.ExtOfBestModel = LocalExtOfBestModel.replace("__mhz", "__MHz")
                                BestModelID = LocalModelFileID


                        ## find latest file from map iteration
                        else:
                            self.ExtOfBestModel = "__model"
                            BestModelID = LocalModelFileID


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## store list of model files related to current obs. data file
                self.ListObsDataFileNames.append(LocalObsDataFileName)
                self.ListObsDataFileParams.append([LocalObsDataFileName, LocalListModelFiles, BestModelID])

        # Debug:
        # print ("self.ListObsDataFileNames = ", self.ListObsDataFileNames)
        # print ("self.ListObsDataFileParams = ", self.ListObsDataFileParams)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## if no obs. data file was found, use first file in the list
        if (not self.FoundObsDataFileFlag):
            self.CurrentFITSFileName = self.JobDirFileListing[0]
            self.ComboIndex = 0
        else:
            FirstFITSCubeParam = self.ListObsDataFileParams[0]
            self.CurrentFITSFileName = FirstFITSCubeParam[0]
            LocalListModelFiles = FirstFITSCubeParam[1]
            BestModelID = FirstFITSCubeParam[2]
            self.CurrentModelFITSFileName = LocalListModelFiles[BestModelID]
            self.SpectrumObsFlag = True
            self.SpectrumModFlag = True

        # Debug:
        # print ("ObsXMLFileName = ", ObsXMLFileName)
        # print ("FirstObsDataFileName = ", FirstObsDataFileName)
        # print ("self.JobDirFileListing = ", self.JobDirFileListing)
        # print ("self.CurrentFITSFileName = ", self.CurrentFITSFileName)
        # print ("self.CurrentModelFITSFileName = ", self.CurrentModelFITSFileName)


        ## define list for combo box
        self.ComboList = []
        for LocalFileName in self.JobDirFileListing:
            self.ComboList.append(os.path.basename(LocalFileName))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## import first FITS cubes
        print ("Import FITS files .. ", end = "", flush = True)


        ## import obs data cube
        CubeDict = FITSImport(self.CurrentFITSFileName, mask = self.CurrentModelFITSFileName)
        self.FITSImageFlag = CubeDict['FITSImageFlag']
        if (self.FITSImageFlag):
            self.FITSImage = CubeDict['Image']
        else:
            self.FITSObsCube = CubeDict['Cube']
            self.FrequencyAxis = CubeDict['FrequencyAxis']
            self.FrequencyAxisUnit = CubeDict['FrequencyAxisUnit']
            self.MapFreqMin = CubeDict['MapFreqMin']
            self.MapFreqMax = CubeDict['MapFreqMax']
            self.MapFreqStep = CubeDict['MapFreqStep']
            self.MapCurrFreq = self.MapFreqMin
        self.DecAxis = CubeDict['DecAxis']
        self.DecAxisUNIT = CubeDict['DecAxisUNIT']
        self.RAAxis = CubeDict['RAAxis']
        self.RAAxisUNIT = CubeDict['RAAxisUNIT']
        self.DecAxisDelta = CubeDict['DecAxisDelta']
        self.yPixelID = CubeDict['yPixelID']
        self.RAAxisDelta = CubeDict['RAAxisDelta']
        self.xPixelID = CubeDict['xPixelID']
        self.UnitCube = CubeDict['UnitCube']
        self.xPixelCoord = self.RAAxis[self.xPixelID]
        self.yPixelCoord = self.DecAxis[self.yPixelID]


        ## import spectrum
        self.ObsSpectrum = self.FITSPixelSpecImport(self.FITSObsCube)
        if (self.FoundObsDataFileFlag):
            CubeDict = FITSImport(self.CurrentModelFITSFileName)
            self.FITSImageFlag = CubeDict['FITSImageFlag']
            self.FITSModelCube = CubeDict['Cube']
            self.FITSModelCube.allow_huge_operations = True

            # Debug:
            # print ("self.CurrentModelFITSFileName = ", self.CurrentModelFITSFileName)
            # print ("self.FITSModelCube = ", self.FITSModelCube)


        ## we're done
        # print ("done!")


        ##================================================================================================================================================
        ## create plot window for spectrum and map


        ## initialize main widget
        self._main = QtWidgets.QWidget()
        self.setCentralWidget(self._main)


        ## prepare plot for FITS cube or image
        self.figMap = pylab.figure(figsize = (15, 10))
        self.figMap.clear()
        self.figMap.subplots_adjust(bottom = 0.09, top = 0.966, right = 0.981, left = 0.14, hspace = 0.0, wspace = 0.0)
        self.PlotMap = pylab.subplot(1, 1, 1, projection = self.FITSObsCube[0, :, :].wcs)
        self.canvasMap = FigureCanvas(self.figMap)
        self.canvasMap.draw()
        self.toolbarMap = NavigationToolbar(self.canvasMap, self)
        self.canvasMap.draw()


        ## prepare plot for spectrum and toolbar
        self.figSpectrum = pylab.figure(figsize = (15, 10))
        self.figSpectrum.clear()
        self.figSpectrum.subplots_adjust(bottom = 0.12, top = 0.88, right = 0.98, left = 0.15, hspace = 0.0, wspace = 0.0)
        self.PlotSpec = pylab.subplot(1, 1, 1)
        self.canvasSpectrum = FigureCanvas(self.figSpectrum)
        self.canvasSpectrum.draw()
        self.toolbarSpectrum = NavigationToolbar(self.canvasSpectrum, self)
        self.canvasSpectrum.draw()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define elements


        ## add line edit element for entering min. freq.
        self.LabelChannel = QtWidgets.QLabel("Spec. axis:")
        self.LabelChannel.setFixedWidth(80)

        self.LineEditChannel = QtWidgets.QLineEdit()
        self.LineEditChannel.setObjectName("EditChannel")
        self.LineEditChannel.setFixedWidth(350)
        self.LineEditChannel.setToolTip('Enter frequency.')
        self.LineEditChannel.setValidator(QtGui.QDoubleValidator(self.MapFreqMin, self.MapFreqMax, 3))
        self.LineEditChannel.editingFinished.connect(self.LineEditWidget)
        self.LineEditChannel.setEnabled(True)
        self.LineEditChannel.setFixedWidth(250)
        self.LineEditChannel.setText("{:.3f}".format(self.MapCurrFreq))


        ## add slider element for entering min. freq.
        self.SliderChannel = QtWidgets.QSlider()
        self.SliderChannel.setOrientation(QtCore.Qt.Horizontal)
        self.SliderChannel.setMinimum(0)
        self.SliderChannel.setMaximum(len(self.FrequencyAxis) - 1)
        self.SliderChannel.setTickInterval(1)
        try:
            self.SliderChannel.setValue(self.MapCurrFreq)
        except:
            self.SliderChannel.setValue(1)
        self.SliderChannel.setObjectName("SliderChannel")
        self.SliderChannel.setFixedWidth(350)
        self.SliderChannel.setToolTip('Enter frequency.')
        self.SliderChannel.valueChanged.connect(self.SliderWidget)
        self.SliderChannel.setEnabled(True)


        ## add line edit element for entering min. freq.
        self.LabelComboRange = QtWidgets.QLabel("Select file:")
        self.ComboRangeSelect = QtWidgets.QComboBox()
        self.ComboRangeSelect.setObjectName("ComboRange")
        self.ComboRangeSelect.setToolTip('Select file:')
        self.ComboRangeSelect.currentIndexChanged.connect(self.ComboWidget)
        self.ComboRangeSelect.setFixedWidth(450)
        self.ComboRangeSelect.setEnabled(True)
        self.ComboRangeSelect.addItems(self.ComboList)
        self.ComboRangeSelect.setCurrentIndex(self.ComboIndex)


        ## add button "|<"
        self.buttonFSC = QtWidgets.QPushButton('|<')
        self.buttonFSC.setShortcut("0")
        self.buttonFSC.setToolTip('first spectral channel')
        self.buttonFSC.setObjectName("SpecChannelFitst")
        self.buttonFSC.setFixedWidth(80)
        self.buttonFSC.clicked.connect(self.ButtonWidget)


        ## add button "<"
        self.buttonPSC = QtWidgets.QPushButton('<')
        self.buttonPSC.setShortcut("-")
        self.buttonPSC.setToolTip('previous spectral channel')
        self.buttonPSC.setObjectName("SpecChannelm1")
        self.buttonPSC.setFixedWidth(80)
        self.buttonPSC.clicked.connect(self.ButtonWidget)


        ## add button ">"
        self.buttonNSC = QtWidgets.QPushButton('>')
        self.buttonNSC.setShortcut("+")
        self.buttonNSC.setToolTip('next spectral channel')
        self.buttonNSC.setObjectName("SpecChannelp1")
        self.buttonNSC.setFixedWidth(80)
        self.buttonNSC.clicked.connect(self.ButtonWidget)


        ## add button ">|"
        self.buttonLSC = QtWidgets.QPushButton('>|')
        self.buttonLSC.setShortcut("9")
        self.buttonLSC.setToolTip('last spectral channel')
        self.buttonLSC.setObjectName("SpecChannelLast")
        self.buttonLSC.setFixedWidth(80)
        self.buttonLSC.clicked.connect(self.ButtonWidget)


        ## add button "Cancel"
        self.buttonCancel = QtWidgets.QPushButton('Cancel')
        self.buttonCancel.setShortcut("ESC")
        self.buttonCancel.setToolTip('Cancel')
        self.buttonCancel.setObjectName("ButtonCancel")
        self.buttonCancel.clicked.connect(self.ButtonWidget)


        ## add button "OK"
        self.buttonOK = QtWidgets.QPushButton('OK')
        self.buttonOK.setShortcut("q")
        self.buttonOK.setToolTip('Quit GUI.')
        self.buttonOK.setObjectName("ButtonOK")
        self.buttonOK.clicked.connect(self.ButtonWidget)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## connect events to current plot
        self.connectSpecPlot()
        self.connectMap()


        ##================================================================================================================================================
        ## set layout and window title
        layout = QtWidgets.QGridLayout()
        layout.addWidget(self.canvasMap, 0, 0, 3, 5)
        layout.addWidget(self.toolbarMap, 3, 0, 1, 5)
        layout.addWidget(self.LabelChannel, 4, 0)
        layout.addWidget(self.LineEditChannel, 4, 1, 1, 3)
        layout.addWidget(self.SliderChannel, 4, 4, 1, 1)
        layout.addWidget(self.canvasSpectrum, 0, 5, 3, 5)
        layout.addWidget(self.toolbarSpectrum, 3, 5, 1, 5)
        layout.addWidget(self.LabelComboRange, 4, 5)
        layout.addWidget(self.ComboRangeSelect, 5, 5, 1, 3)
        layout.addWidget(self.buttonFSC, 5, 0)
        layout.addWidget(self.buttonPSC, 5, 1)
        layout.addWidget(self.buttonNSC, 5, 2)
        layout.addWidget(self.buttonLSC, 5, 3)
        layout.addWidget(self.buttonCancel, 5, 8)
        layout.addWidget(self.buttonOK, 5, 9)
        self._main.setLayout(layout)


        ## set title of window
        PureJobDir = self.JobDir.split("/")
        PureJobDir = PureJobDir[-2]
        LocalTitelString = "Results of myXCLASSMapFit run " + chr(34) + PureJobDir.strip() + chr(34)
        self.setWindowTitle(LocalTitelString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## plot spectra
        self.lockElements = False
        self.InitFlag = True
        self.replotGUI()
        self.InitFlag = False


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## show spectra
        self.show()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## redefine return-press event
    def keyPressEvent(self, event):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("event = ", event)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## redefine return-press event
        if (event.key() == QtCore.Qt.Key_Return):
            self.returnPressed.emit(event)
        else:
            event.accept()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle button event of widget
    def ButtonWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## is element locked
        if (self.lockElements):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()
        name = sender.text()

        # Debug:
        # print ("objectName = ", objectName)
        # print ("name = ", name)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## identify calling button


        # cancel GUI
        if (objectName == "ButtonCancel"):
            self.ok = 1
            self.disconnect()


        # quit GUI
        elif (objectName == "ButtonOK"):
            self.disconnect()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## (only for FITS map): control spectral channel


        # go to first spectral channel
        elif (objectName == "SpecChannelFitst"):
            self.ChannelIndex = 0


            ## update GUI
            self.replotGUI()


        # go to first spectral channel
        elif (objectName == "SpecChannelm1"):
            self.ChannelIndex = max(0, self.ChannelIndex - 1)


            ## update GUI
            self.replotGUI()


        # go to first spectral channel
        elif (objectName == "SpecChannelp1"):
            self.ChannelIndex = min(len(self.FrequencyAxis) - 1, self.ChannelIndex + 1)


            ## update GUI
            self.replotGUI()


        # go to first spectral channel
        elif (objectName == "SpecChannelLast"):
            self.ChannelIndex = len(self.FrequencyAxis) - 1


            ## update GUI
            self.replotGUI()


        ## return to main GUI
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle combo box event of widget
    def ComboWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is element locked
        if (self.lockElements):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()

        # Debug:
        # print ("objectName = ", objectName)


        ## get index of selected frequency range
        LocalComboIndex = max(0, self.ComboRangeSelect.currentIndex())
        if (self.ComboIndex != LocalComboIndex):
            self.ComboIndex = LocalComboIndex
            self.CurrentFITSFileName = self.JobDirFileListing[LocalComboIndex].strip()

            # Debug:
            # print ("\n\n\n\nself.ComboIndex = ", self.ComboIndex)
            # print ("self.CurrentFITSFileName = ", self.CurrentFITSFileName)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## check, if selected file is a obs. data file
            if (self.CurrentFITSFileName in self.ListObsDataFileNames):
                i = self.ListObsDataFileNames.index(self.CurrentFITSFileName)
                FirstFITSCubeParam = self.ListObsDataFileParams[i]
                self.CurrentFITSFileName = FirstFITSCubeParam[0]
                LocalListModelFiles = FirstFITSCubeParam[1]
                BestModelID = FirstFITSCubeParam[2]
                self.CurrentModelFITSFileName = LocalListModelFiles[BestModelID]

                # Debug:
                # print ("BestModelID = ", BestModelID)
                # print ("self.CurrentModelFITSFileName = ", self.CurrentModelFITSFileName)


                ## import obs data cube
                CubeDict = FITSImport(self.CurrentFITSFileName, mask = self.CurrentModelFITSFileName)
                try:
                    self.FITSImageFlag = CubeDict['FITSImageFlag']
                except:
                    return
                if (self.FITSImageFlag):
                    self.FITSImage = CubeDict['Image']
                else:
                    self.FITSObsCube = CubeDict['Cube']
                    self.FITSObsCube.allow_huge_operations = True
                    self.FrequencyAxis = CubeDict['FrequencyAxis']
                    self.FrequencyAxisUnit = CubeDict['FrequencyAxisUnit']
                    self.MapFreqMin = CubeDict['MapFreqMin']
                    self.MapFreqMax = CubeDict['MapFreqMax']
                    self.MapFreqStep = CubeDict['MapFreqStep']
                    self.MapCurrFreq = self.MapFreqMin
                self.DecAxis = CubeDict['DecAxis']
                self.DecAxisUNIT = CubeDict['DecAxisUNIT']
                self.RAAxis = CubeDict['RAAxis']
                self.RAAxisUNIT = CubeDict['RAAxisUNIT']
                self.DecAxisDelta = CubeDict['DecAxisDelta']
                self.yPixelID = CubeDict['yPixelID']
                self.RAAxisDelta = CubeDict['RAAxisDelta']
                self.xPixelID = CubeDict['xPixelID']
                self.UnitCube = CubeDict['UnitCube']
                self.xPixelCoord = self.RAAxis[self.xPixelID]
                self.yPixelCoord = self.DecAxis[self.yPixelID]
                self.LogFlag = False
                self.SpectrumObsFlag = True
                self.SpectrumModFlag = True
                self.SpectrumChi2Flag = False


                ## import spectrum
                self.ObsSpectrum = self.FITSPixelSpecImport(self.FITSObsCube)
                try:
                    self.FITSImageFlag = CubeDict['FITSImageFlag']
                except:
                    return
                if (self.FoundObsDataFileFlag):
                    CubeDict = FITSImport(self.CurrentModelFITSFileName)
                    self.FITSImageFlag = CubeDict['FITSImageFlag']
                    self.FITSModelCube = CubeDict['Cube']
                    self.FITSModelCube.allow_huge_operations = True


                ## define colormap
                self.MapCMAP = self.MapCMAPDefault
                self.ParameterName = None


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## continue here, if file is not a obs. data cube
            else:


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## check, if selected file describes a FITS image
                LowerCurrentFITSFileName = self.CurrentFITSFileName.lower()
                CubeDict = FITSImport(self.CurrentFITSFileName)
                try:
                    self.FITSImageFlag = CubeDict['FITSImageFlag']
                except:
                    return
                if (self.FITSImageFlag):
                    self.FITSImage = CubeDict['Image']
                    self.DecAxis = CubeDict['DecAxis']
                    self.DecAxisUNIT = CubeDict['DecAxisUNIT']
                    self.RAAxis = CubeDict['RAAxis']
                    self.RAAxisUNIT = CubeDict['RAAxisUNIT']
                    self.DecAxisDelta = CubeDict['DecAxisDelta']
                    self.yPixelID = CubeDict['yPixelID']
                    self.RAAxisDelta = CubeDict['RAAxisDelta']
                    self.xPixelID = CubeDict['xPixelID']
                    self.UnitCube = CubeDict['UnitCube']
                    self.xPixelCoord = self.RAAxis[self.xPixelID]
                    self.yPixelCoord = self.DecAxis[self.yPixelID]


                    ## depending on the parameter described by the map chose color map
                    ParameterName = os.path.basename(self.CurrentFITSFileName).strip()
                    if (ParameterName.startswith("BestResult___chi2values")):
                        self.ParameterName = "chi2"
                    else:
                        ParameterName = ParameterName.replace("BestResult___parameter__", "")
                        ParameterName = ParameterName.split("___")
                        self.ParameterName = ParameterName[0].strip()


                    ## chose colormap depending on parameter name
                    if (self.ParameterName in ["T_rot"]):
                        self.MapCMAP = "hot"
                        self.LogFlag = False
                    elif (self.ParameterName in ["V_off","V_width_Gauss"]):
                        self.MapCMAP = "cool"
                        self.LogFlag = False
                    elif (self.ParameterName in ["N_tot"]):
                        self.MapCMAP = "winter"
                        self.LogFlag = True
                    elif (self.ParameterName in ["source-size"]):
                        self.MapCMAP = "copper"
                        self.LogFlag = False
                    elif (self.ParameterName in ["chi2"]):
                        self.MapCMAP = "gray"
                        self.LogFlag = False
                    else:
                        self.MapCMAP = "gray"
                        self.LogFlag = False


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## check, if selected file describes a model or a chi2 cube
                elif (LowerCurrentFITSFileName.endswith(".chi2.out.fits")):


                    ## import obs data cube
                    CubeDict = FITSImport(self.CurrentFITSFileName)
                    try:
                        self.FITSImageFlag = CubeDict['FITSImageFlag']
                    except:
                        return
                    if (not self.FITSImageFlag):
                        self.FITSChi2Cube = CubeDict['Cube']
                        self.FITSChi2Cube.allow_huge_operations = True
                        self.FrequencyAxis = CubeDict['FrequencyAxis']
                        self.FrequencyAxisUnit = CubeDict['FrequencyAxisUnit']
                        self.MapFreqMin = CubeDict['MapFreqMin']
                        self.MapFreqMax = CubeDict['MapFreqMax']
                        self.MapFreqStep = CubeDict['MapFreqStep']
                        self.MapCurrFreq = self.MapFreqMin
                        self.DecAxis = CubeDict['DecAxis']
                        self.DecAxisUNIT = CubeDict['DecAxisUNIT']
                        self.RAAxis = CubeDict['RAAxis']
                        self.RAAxisUNIT = CubeDict['RAAxisUNIT']
                        self.DecAxisDelta = CubeDict['DecAxisDelta']
                        self.yPixelID = CubeDict['yPixelID']
                        self.RAAxisDelta = CubeDict['RAAxisDelta']
                        self.xPixelID = CubeDict['xPixelID']
                        self.UnitCube = CubeDict['UnitCube']
                        self.xPixelCoord = self.RAAxis[self.xPixelID]
                        self.yPixelCoord = self.DecAxis[self.yPixelID]
                        self.LogFlag = False
                        self.SpectrumObsFlag = False
                        self.SpectrumModFlag = False
                        self.SpectrumChi2Flag = True


                        ## define colormap
                        self.MapCMAP = self.MapCMAPDefault


                ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                ## the remaining file must be a cube describing a model spectrum
                else:


                    ## import obs data cube
                    CubeDict = FITSImport(self.CurrentFITSFileName)
                    try:
                        self.FITSImageFlag = CubeDict['FITSImageFlag']
                    except:
                        return
                    if (not self.FITSImageFlag):
                        self.FITSModelCube = CubeDict['Cube']
                        self.FITSModelCube.allow_huge_operations = True
                        self.FrequencyAxis = CubeDict['FrequencyAxis']
                        self.FrequencyAxisUnit = CubeDict['FrequencyAxisUnit']
                        self.MapFreqMin = CubeDict['MapFreqMin']
                        self.MapFreqMax = CubeDict['MapFreqMax']
                        self.MapFreqStep = CubeDict['MapFreqStep']
                        self.MapCurrFreq = self.MapFreqMin
                        self.DecAxis = CubeDict['DecAxis']
                        self.DecAxisUNIT = CubeDict['DecAxisUNIT']
                        self.RAAxis = CubeDict['RAAxis']
                        self.RAAxisUNIT = CubeDict['RAAxisUNIT']
                        self.DecAxisDelta = CubeDict['DecAxisDelta']
                        self.yPixelID = CubeDict['yPixelID']
                        self.RAAxisDelta = CubeDict['RAAxisDelta']
                        self.xPixelID = CubeDict['xPixelID']
                        self.UnitCube = CubeDict['UnitCube']
                        self.xPixelCoord = self.RAAxis[self.xPixelID]
                        self.yPixelCoord = self.DecAxis[self.yPixelID]
                        self.LogFlag = False
                        self.SpectrumObsFlag = True
                        self.SpectrumModFlag = True
                        self.SpectrumChi2Flag = False


                        ## find corresponding obs. data file
                        ## self.ListObsDataFileParams = [LocalObsDataFileName, LocalListModelFiles, BestModelID]
                        CurrObsDataFileName = ""
                        for LocalObsDataFileParams in self.ListObsDataFileParams:
                            LocalObsDataFileName = LocalObsDataFileParams[0]
                            LocalListModelFiles = LocalObsDataFileParams[1]
                            for LocalModelFiles in LocalListModelFiles:
                                if (LocalModelFiles == self.CurrentFITSFileName):
                                    CurrObsDataFileName = LocalObsDataFileName
                                    break
                            if (CurrObsDataFileName != ""):
                                break
                        if (CurrObsDataFileName != ""):
                            CubeDict = FITSImport(CurrObsDataFileName, mask = self.CurrentFITSFileName)
                            self.FITSImageFlag = CubeDict['FITSImageFlag']
                            if (not self.FITSImageFlag):
                                self.FITSObsCube = CubeDict['Cube']
                                self.FITSObsCube.allow_huge_operations = True


                        ## define colormap
                        self.MapCMAP = self.MapCMAPDefault


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## (de-)activate map elements for parameter maps
            if (self.FITSImageFlag):
                self.LabelChannel.setEnabled(False)
                self.LineEditChannel.setEnabled(False)
                self.SliderChannel.setEnabled(False)
                self.buttonFSC.setEnabled(False)
                self.buttonPSC.setEnabled(False)
                self.buttonNSC.setEnabled(False)
                self.buttonLSC.setEnabled(False)
            else:
                self.LabelChannel.setEnabled(True)
                self.LineEditChannel.setEnabled(True)
                self.SliderChannel.setEnabled(True)
                self.buttonFSC.setEnabled(True)
                self.buttonPSC.setEnabled(True)
                self.buttonNSC.setEnabled(True)
                self.buttonLSC.setEnabled(True)


            ## update GUI
            self.replotGUI()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle line edit event of widget
    def LineEditWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is element locked
        if (self.lockElements):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name and flag of radio button
        sender = self.sender()
        objectName = sender.objectName()
        val = str(sender.text())
        self.CurrCursorPos = sender.cursorPosition()

        # Debug:
        # print ("objectName = ", objectName)
        # print ("val = ", val)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if value is number which is greater or equal zero
        IsNumber = True
        try:
            i = int(val)
        except ValueError:
            try:
                i = float(val)
            except ValueError:
                IsNumber = False
        if (not IsNumber):
            if (str(IsNumber).strip() == ""):
                return
        if (IsNumber):                                                                      ## continue if input is a number
            val = float(val)


            ## get channel index
            if (objectName == "EditChannel"):
                self.MapCurrFreq = val
                self.ChannelIndex = max(0, (numpy.abs(self.FrequencyAxis - val)).argmin() - 1)
                self.RefreshMapOnlyFlag = True
                self.replotGUI()                                                            ## replot spectrum
                self.RefreshMapOnlyFlag = False

        # Debug:
        # print ("val = ", val)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## handle slider event of widget
    def SliderWidget(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## is element locked
        if (self.lockElements):
            return


        ## get name of slider
        sender = self.sender()
        objectName = sender.objectName()                                                    ## get name of object
        val = sender.value()                                                                ## get value from slider
        if (objectName == "SliderChannel"):
            self.ChannelIndex = val
            self.MapCurrFreq = self.FrequencyAxis[val]

            # Debug:
            # print ("self.MapCurrFreq = ", self.MapCurrFreq)
            # print ("self.ChannelIndex = ", self.ChannelIndex)
            # print ("val = ", val)
            # print ("numpy.abs(self.FrequencyAxis - val) = ", numpy.abs(self.FrequencyAxis - val))
            # print ("(numpy.abs(self.FrequencyAxis - val)).argmin() - 1 = ", (numpy.abs(self.FrequencyAxis - val)).argmin() - 1)


            ## replot spectrum
            self.RefreshMapOnlyFlag = True
            self.replotGUI()                                                                ## replot spectrum
            self.RefreshMapOnlyFlag = False


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## connect all the stored connection ids
    def connectSpecPlot(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## define events
        self.cidquitSpectrum = self.figSpectrum.canvas.mpl_connect('close_event', self.dispose)
        self.cidpressSpectrum = self.figSpectrum.canvas.mpl_connect('button_press_event', self.onclickSpectrum)
        self.cidmotionSpectrum = self.figSpectrum.canvas.mpl_connect('motion_notify_event', self.MouseMoveSpectrum)
        self.cidkeySpectrum = self.figSpectrum.canvas.mpl_connect('key_press_event', self.OnKeySpectrum)


        ## use tight layout
        # self.figSpectrum.tight_layout()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## connect events to
    def connectMap(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## define events
        self.cidquitMap = self.figMap.canvas.mpl_connect('close_event', self.dispose)
        self.cidpressMap = self.figMap.canvas.mpl_connect('button_press_event', self.onclickMap)
        self.cidmotionMap = self.figMap.canvas.mpl_connect('motion_notify_event', self.MouseMoveMap)
        # self.cidkeyMap = self.figMap.canvas.mpl_connect('key_press_event', self.OnKeyMap)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## quit gui
    def dispose(self, event):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("event = ", event)


        ## stop loop
        self.disconnect()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## disconnect all the stored connection ids and close gui
    def disconnect(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## disconnect events of map plot
        self.canvasMap.mpl_disconnect(self.cidquitMap)
        self.canvasMap.mpl_disconnect(self.cidpressMap)
        self.canvasMap.mpl_disconnect(self.cidmotionMap)
        # self.canvasMap.mpl_disconnect(self.cidkeyMap)


        ## disconnect events of spectrum plot
        self.canvasSpectrum.mpl_disconnect(self.cidquitSpectrum)
        self.canvasSpectrum.mpl_disconnect(self.cidpressSpectrum)
        self.canvasSpectrum.mpl_disconnect(self.cidmotionSpectrum)
        self.canvasSpectrum.mpl_disconnect(self.cidkeySpectrum)
        self.close()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## replot window
    def replotGUI(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("\n\nself.LowestFreq = ", self.LowestFreq)
        # print ("self.HighestFreq = ", self.HighestFreq)
        # print ("self.GlobalTbg = ", self.GlobalTbg)
        # print ("self.GlobalTSlope = ", self.GlobalTSlope)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## lock element
        self.lockElements = True


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create map plot

        # Debug:
        # print ("self.NumDimFitsFile = ", self.NumDimFitsFile)
        # print ("self.ChannelIndex = ", self.ChannelIndex)
        # print ("max(int(self.ChannelIndex - 1) , 0) = ", max(int(self.ChannelIndex - 1) , 0))


        ## get FITS image at given position and add image to figure #.set_data
        LocalLogFlag = False
        if (self.FITSImageFlag):
            LocalLogFlag = self.LogFlag
            LocalFITSImage = self.FITSImage[:, :]
        else:
            LocalFITSImage = self.FITSObsCube[max(int(self.ChannelIndex - 1) , 0), :, :]
        if (LocalLogFlag):
            im = self.PlotMap.imshow(LocalFITSImage.value, cmap = self.MapCMAP, origin = 'lower', norm = LogNorm())
        else:
            im = self.PlotMap.imshow(LocalFITSImage.value, cmap = self.MapCMAP, origin = 'lower')


        ## add some object only once
        if (self.OnceFlag):


            ## add colorbar
            self.colorbar = self.figMap.colorbar(im)


            ## plot lines for cursor cursor
            if (self.ParameterName in ["chi2", "source-size", "T_rot"] or self.ParameterName is None):
                LocalCursorMap = "green"
            else:
                LocalCursorMap = "black"
            self.xPosMap = self.PlotMap.axvline(self.xCursorPixelID, linewidth = 1, color = LocalCursorMap, zorder = 10)
            self.yPosMap = self.PlotMap.axhline(self.yCursorPixelID, linewidth = 1, color = LocalCursorMap, zorder = 10)


            ## we're leaving the "once" area
            self.OnceFlag = False
        else:
            self.colorbar.update_normal(im)

        # Debug:
        # print ("\n\nself.RAAxis = ", self.RAAxis)
        # print ("self.DecAxis = ", self.DecAxis)


        ## labels, grid (?), and title
        # self.PlotMap.grid()
        self.PlotMap.set_xlabel("RA ({:s})".format(self.RAAxisUNIT))
        self.PlotMap.set_ylabel("Dec ({:s})".format(self.DecAxisUNIT))
        LocalFITSFileName = os.path.basename(self.CurrentFITSFileName)
        LocalTitelString = "FITS file: {:s}".format(chr(34) + LocalFITSFileName + chr(34))
        # self.PlotMap.set_title(LocalTitelString)



        ## update map plot of gui
        self.canvasMap.draw()


        ## use tight layout
        # self.figMap.tight_layout()


        ## if channel index is changed stop here
        if (self.RefreshMapOnlyFlag):
            MapCurrFreqString = "{:.2f}".format(self.MapCurrFreq)
            self.LineEditChannel.setText("{:<s}".format(MapCurrFreqString))
            self.SliderChannel.setValue(self.ChannelIndex)
            self.lockElements = False
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create spectrum plot
        if (not self.LockCursorPosFlag):
            self.PlotSpec.clear()


            ## check, if range defined correctly
            f1 = self.LowestFreq
            f2 = self.HighestFreq
            self.LowestFreq = min(f1, f2)
            self.HighestFreq = max(f1, f2)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## set title
            ix = max(0, self.xPixelID)
            ix = min(len(self.RAAxis) - 1, self.xPixelID)
            iy = max(0, self.yPixelID)
            iy = min(len(self.DecAxis) - 1, self.yPixelID)
            if (self.RAAxisUNIT in ["deg"]):
                xCoord = "RA:  {:s}".format(Angle(self.RAAxis[ix], u.Unit(self.RAAxisUNIT)).to_string(unit = u.degree, sep=':'))
            else:
                xCoord = "RA:  {:.4f} {:s}".format(self.RAAxis[ix], self.RAAxisUNIT)
            if (self.DecAxisUNIT in ["deg"]):
                yCoord = "DEC: {:s}".format(Angle(self.DecAxis[iy], u.Unit(self.DecAxisUNIT)).to_string(unit = u.degree, sep=':'))
            else:
                yCoord = "DEC: {:.4f} {:s}".format(self.DecAxis[iy], self.DecAxisUNIT)
            LocalTitelString = "Spectra at \nposition = ({:s}, {:s}) .. ".format(xCoord, yCoord)
            self.PlotSpec.set_title(LocalTitelString)


            ## define grids and axis labels
            self.PlotSpec.grid(True)
            self.PlotSpec.set_ylabel(r"Intensity ({:s})".format(self.UnitCube))
            self.PlotSpec.set_xlabel("Spectral axis ({:s})".format(self.FrequencyAxisUnit))


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## add obs. data
            AllObsMinInt = numpy.nan
            AllObsMaxInt = numpy.nan
            if (self.SpectrumObsFlag):
                velo, dec, ra = self.FITSObsCube.world[:]
                LocalDecAxis = dec[0, :, 0].to(u.deg).value
                MinLocalDecAxis = numpy.nanmin(LocalDecAxis)
                MaxLocalDecAxis = numpy.nanmax(LocalDecAxis)
                LocalRAAxis = ra[0, 0, :].to(u.deg).value
                MinLocalRAAxis = numpy.nanmin(LocalRAAxis)
                MaxLocalRAAxis = numpy.nanmax(LocalRAAxis)
                if (MinLocalRAAxis <= self.xPixelCoord and self.xPixelCoord <= MaxLocalRAAxis \
                    and MinLocalDecAxis <= self.yPixelCoord and self.yPixelCoord <= MaxLocalDecAxis):
                    LocalyPixelID = max(0, (numpy.abs(LocalDecAxis[:] - self.yPixelCoord)).argmin())
                    LocalxPixelID = max(0, (numpy.abs(LocalRAAxis[:] - self.xPixelCoord)).argmin())
                    LocalObsSpectrum = self.FITSObsCube[:, LocalyPixelID, LocalxPixelID].value
                    self.PlotSpec.plot(self.FrequencyAxis[:], LocalObsSpectrum[:], '-', color = 'blue', linewidth = 2.0, drawstyle = 'steps-mid', \
                                    label = "obs. data")
                    AllObsMinInt = numpy.nanmin(LocalObsSpectrum[:])
                    AllObsMaxInt = numpy.nanmax(LocalObsSpectrum[:])


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## add model spectrum
            AllModelMinInt = numpy.nan
            AllModelMaxInt = numpy.nan
            if (self.SpectrumModFlag):
                velo, dec, ra = self.FITSModelCube.world[:]
                LocalDecAxis = dec[0, :, 0].to(u.deg).value
                MinLocalDecAxis = numpy.nanmin(LocalDecAxis)
                MaxLocalDecAxis = numpy.nanmax(LocalDecAxis)
                LocalRAAxis = ra[0, 0, :].to(u.deg).value
                MinLocalRAAxis = numpy.nanmin(LocalRAAxis)
                MaxLocalRAAxis = numpy.nanmax(LocalRAAxis)
                if (MinLocalRAAxis <= self.xPixelCoord and self.xPixelCoord <= MaxLocalRAAxis \
                    and MinLocalDecAxis <= self.yPixelCoord and self.yPixelCoord <= MaxLocalDecAxis):
                    LocalyPixelID = max(0, (numpy.abs(LocalDecAxis[:] - self.yPixelCoord)).argmin())
                    LocalxPixelID = max(0, (numpy.abs(LocalRAAxis[:] - self.xPixelCoord)).argmin())
                    LocalModelSpectrum = self.FITSModelCube[:, LocalyPixelID, LocalxPixelID].value
                    self.PlotSpec.plot(self.FrequencyAxis[:], LocalModelSpectrum[:], '-', color = 'red', linewidth = 2.0, label = "fit")
                    AllModelMinInt = numpy.nanmin(LocalModelSpectrum[:])
                    AllModelMaxInt = numpy.nanmax(LocalModelSpectrum[:])


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## add chi2 spectrum
            AllChi2MinInt = numpy.nan
            AllChi2MaxInt = numpy.nan
            if (self.SpectrumChi2Flag):
                velo, dec, ra = self.FITSChi2Cube.world[:]
                LocalDecAxis = dec[0, :, 0].to(u.deg).value
                MinLocalDecAxis = numpy.nanmin(LocalDecAxis)
                MaxLocalDecAxis = numpy.nanmax(LocalDecAxis)
                LocalRAAxis = ra[0, 0, :].to(u.deg).value
                MinLocalRAAxis = numpy.nanmin(LocalRAAxis)
                MaxLocalRAAxis = numpy.nanmax(LocalRAAxis)
                if (MinLocalRAAxis <= self.xPixelCoord and self.xPixelCoord <= MaxLocalRAAxis \
                    and MinLocalDecAxis <= self.yPixelCoord and self.yPixelCoord <= MaxLocalDecAxis):
                    LocalyPixelID = max(0, (numpy.abs(LocalDecAxis[:] - self.yPixelCoord)).argmin())
                    LocalxPixelID = max(0, (numpy.abs(LocalRAAxis[:] - self.xPixelCoord)).argmin())
                    LocalChi2Spectrum = self.FITSChi2Cube[:, LocalyPixelID, LocalxPixelID].value
                    self.PlotSpec.plot(self.FrequencyAxis[:], LocalChi2Spectrum[:], '-', color = 'black', linewidth = 2.0, label = r"$\chi^2$")
                    AllChi2MinInt = numpy.nanmin(LocalChi2Spectrum[:])
                    AllChi2MaxInt = numpy.nanmax(LocalChi2Spectrum[:])


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## set limits of spectral axis
            if (self.InitFlag):
                self.AllMinInt = min(AllObsMinInt, AllModelMinInt, AllChi2MinInt)
                self.AllMaxInt = max(AllObsMaxInt, AllModelMaxInt, AllChi2MaxInt)
                offset = abs(self.AllMaxInt - self.AllMinInt) * 0.05
                # if (self.AllMinInt - offset < self.AllMaxInt + offset):
                #     self.PlotSpec.set_ylim(self.AllMinInt - offset, self.AllMaxInt + offset)
                self.PlotSpec.set_xlim(self.MapFreqMin, self.MapFreqMax)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## plot lines for cursor cursor
            # self.yPosSpectrum = self.PlotSpec.axhline(color = 'k')                          ## the horiz line
            # self.xPosSpectrum = self.PlotSpec.axvline(color = 'k')                          ## the vert line


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## reformat labeling
            if (self.FrequencyAxisUnit in ["Hz"]):
                self.PlotSpec.xaxis.set_major_formatter(pylab.matplotlib.ticker.FormatStrFormatter('%.3e'))
            else:
                self.PlotSpec.xaxis.set_major_formatter(pylab.matplotlib.ticker.FormatStrFormatter('%.1f'))


            ## add legend
            self.PlotSpec.legend()


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## update spectrum plot of gui
            self.canvasSpectrum.draw()


            ## use tight layout
            # self.figSpectrum.tight_layout()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update values in lineEdit boxes
        MapCurrFreqString = "{:.2f}".format(self.MapCurrFreq)
        self.LineEditChannel.setText("{:<s}".format(MapCurrFreqString))
        self.SliderChannel.setValue(self.ChannelIndex)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## unlock elements
        self.lockElements = False


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what happens on key event
    def OnKeySpectrum(self, event):
        """

    input parameters:
    -----------------

        - event:            evenet object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ('you pressed', event.key, event.xdata, event.ydata)


        ## define event on "-" key:  remove selected frequency range
        if (event.key == "-"):
            x, y = event.xdata, event.ydata                                                 ## get position of mouse cursor


            ## update GUI
            self.replotGUI()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what to do when mouse is moved in spectrum plot
    def MouseMoveSpectrum(self, eventSpec):
        """

    input parameters:
    -----------------

        - eventSpec:         evenet object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("eventSpec = ", eventSpec)


        ## check, if mouse cursor position is in allowed range
        if (not eventSpec.inaxes):
            return


        ## get position of mouse cursor
        x, y = eventSpec.xdata, eventSpec.ydata


        ## update the line positions
        # self.yPosSpectrum.set_ydata(y )
        # self.xPosSpectrum.set_xdata(x )
        # self.txt.set_text( 'Freq.={:1.2f} MHz, Int.={:1.2f} K'.format(x, y) )
        # self.canvasSpectrum.draw()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what to do when mouse is moved in map plot
    def MouseMoveMap(self, eventMap):
        """

    input parameters:
    -----------------

        - eventMap:            evenet object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("eventMap = ", eventMap)


        ## get position of mouse cursor
        x, y = eventMap.xdata, eventMap.ydata

        # Debug:
        # print ("y, x = ", y, x, eventMap.inaxes)


        # ## update the line positions
        if (eventMap.inaxes):
            self.yPosMap.set_ydata(y)
            self.xPosMap.set_xdata(x)
            self.xCursorPixelID = int(round(x))
            self.yCursorPixelID = int(round(y))


        ## update GUI
        self.canvasMap.draw()


        ## use tight layout
        # self.figMap.tight_layout()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## refresh GUI
        if (x is not None and y is not None):
            self.yPixelID = self.yCursorPixelID
            self.xPixelID = self.xCursorPixelID
            self.replotGUI()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get parameter
    def getParam(self):
        """

    input parameters:
    -----------------

        - None


    output parameters:
    ------------------

        - None
        """


        ## make pixel coordinates available
        if (self.ListSpecCoord == []):
            self.ListSpecCoord = [[self.xPixelID, self.yPixelID]]


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what happens on mouse click (NOT defined at the moment)
    def onclickSpectrum(self, event):
        """

    input parameters:
    -----------------

        - event:            evenet object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ('events', event.key, event.xdata, event.ydata, event.inaxes, event.button, event.dblclick)


        ## check mouse buttons
        # if (event.button == 1):
        #    print ("You pressed the left mouse button!")
        # elif (event.button == 2):
        #    print ("You pressed the middle mouse button!")
        # elif (event.button == 3):
        #    print ("You pressed the right mouse button!")


        ## check, if mouse cursor position is in allowed range
        if (not event.inaxes):
            return


        ## define event on left mouse button pressed
        if (event.dblclick and (event.button == 1 or event.button == 3)):
            x, y = event.xdata, event.ydata                                                 ## get position of mouse cursor

            # Debug:
            # print ("\n\nLocalRangeID = ", LocalRangeID)
            # print ("NearestMinFreqRangeID = ", NearestMinFreqRangeID)
            # print ("NearestMaxFreqRangeID = ", NearestMaxFreqRangeID)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## refresh GUI
            self.replotGUI()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what happens on mouse click in map plot
    def onclickMap(self, event):
        """

    input parameters:
    -----------------

        - event:            evenet object


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ('button={:d}, x={:d}, y={:d}, xdata={:.2f}, ydata={:.2f}'.format(event.button, event.x, event.y, event.xdata, event.ydata))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if mouse cursor position is in allowed range
        if (not event.inaxes):
            return


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set title
        x, y = event.xdata, event.ydata                                                     ## get position of mouse cursor


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define event on left mouse button pressed    ( and event.dblclick)
        if (event.button == 1):


            ## get cursor position
            self.xCursorPixelID = int(round(x))
            self.yCursorPixelID = int(round(y))

            # Debug:
            # print ("self.xCursorPixelID = ", self.xCursorPixelID)
            # print ("self.yCursorPixelID = ", self.yCursorPixelID)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## change lock flag and mark position
            if (self.LockCursorPosFlag):
                self.LockCursorPosFlag = False
                self.xPosFixedMap.remove()
                self.yPosFixedMap.remove()
            else:
                self.LockCursorPosFlag = True
                self.xFixedCursorPixelID = self.xCursorPixelID
                self.yFixedCursorPixelID = self.yCursorPixelID


                ## indicate fixed position
                if (self.LockCursorPosFlag and self.xFixedCursorPixelID is not None and self.yFixedCursorPixelID is not None):
                    if (self.ParameterName in ["chi2", "source-size"]):
                        LocalCursorMap = "red"
                    else:
                        LocalCursorMap = "red"
                    self.xPosFixedMap = self.PlotMap.axvline(self.xFixedCursorPixelID, linewidth = 2, color = LocalCursorMap, linestyle = '--')
                    self.yPosFixedMap = self.PlotMap.axhline(self.yFixedCursorPixelID, linewidth = 2, color = LocalCursorMap, linestyle = '--')


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## replot map plot
            self.InitFlag = True
            self.replotGUI()
            self.InitFlag = False


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## on single click with right mouse button average over selected spectra
        if (event.button == 3):
            pass

        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import pixel spectrum from fits file
    def FITSPixelSpecImport(self, LocalCube):
        """

    input parameters:
    -----------------

        - LocalCube:                    local cube


    output parameters:
    ------------------

        - LocalSpectrum                 local spectrum
        """

        # Debug:
        # print ("LocalCube = ", LocalCube)
        # print ("self.FrequencyAxis = ", self.FrequencyAxis)


        ## extract spectrum at current position
        LocalSpectrum = LocalCube[:, self.yPixelID, self.xPixelID].value
        LocalSpectrum = numpy.vstack((self.FrequencyAxis, LocalSpectrum)).T

        # Debug:
        # print ("LocalSpectrum[:, 0] = ", LocalSpectrum[:, 0])
        # print ("LocalSpectrum[:, 1] = ", LocalSpectrum[:, 1])


        ## we're done
        return LocalSpectrum
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## start GUI to inspect results of myXCLASSMapFit function
def StartPlottingGUI(JobDir):
    """

input parameters:
-----------------

    - JobDir:                   name of molecule


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("JobDir = ", JobDir)


    ## start GUI
    qapp = QtWidgets.QApplication.instance()
    if not qapp:
        qapp = QtWidgets.QApplication(sys.argv)
    app = PlottingGUI(JobDir)
    app.show()
    app.activateWindow()
    app.raise_()
    qapp.exec()
    qapp.quit()


    ## we're done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## determine interpolation object for partition function
def GetPartitionFunctionFunc(MolName, dbList):
    """

input parameters:
-----------------

    - MolName:                  name of molecule

    - dbList:                   path and name of database file


output parameters:
------------------

    - QTFunc:                   interpolation function object of partition function
    """

    # Debug:
    # print ("MolName = ", MolName)
    # print ("dbList = ", dbList)


    ## initialize return parameter
    QT = 1.0


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define query string


    ## set names of columns
    NameOfPartFuncTable = "partitionfunctions"
    ColumnNameForNamePartFunc = 'PF_Name'
    ColumnNamesPartFunc = ['PF_1_072', 'PF_1_148', 'PF_1_230', 'PF_1_318', 'PF_1_413', 'PF_1_514', 'PF_1_622', 'PF_1_738', 'PF_1_862', 'PF_1_995', \
                           'PF_2_138', 'PF_2_291', 'PF_2_455', 'PF_2_630', 'PF_2_725', 'PF_2_818', 'PF_3_020', 'PF_3_236', 'PF_3_467', 'PF_3_715', \
                           'PF_3_981', 'PF_4_266', 'PF_4_571', 'PF_4_898', 'PF_5_000', 'PF_5_248', 'PF_5_623', 'PF_6_026', 'PF_6_457', 'PF_6_918', \
                           'PF_7_413', 'PF_7_943', 'PF_8_511', 'PF_9_120', 'PF_9_375', 'PF_9_772', 'PF_10_471', 'PF_11_220', 'PF_12_023', 'PF_12_882', \
                           'PF_13_804', 'PF_14_791', 'PF_15_849', 'PF_16_982', 'PF_18_197', 'PF_18_750', 'PF_19_498', 'PF_20_893', 'PF_22_387', \
                           'PF_23_988', 'PF_25_704', 'PF_27_542', 'PF_29_512', 'PF_31_623', 'PF_33_884', 'PF_36_308', 'PF_37_500', 'PF_38_905', \
                           'PF_41_687', 'PF_44_668', 'PF_47_863', 'PF_51_286', 'PF_54_954', 'PF_58_884', 'PF_63_096', 'PF_67_608', 'PF_72_444', \
                           'PF_75_000', 'PF_77_625', 'PF_83_176', 'PF_89_125', 'PF_95_499', 'PF_102_329', 'PF_109_648', 'PF_117_490', 'PF_125_893', \
                           'PF_134_896', 'PF_144_544', 'PF_150_000', 'PF_154_882', 'PF_165_959', 'PF_177_828', 'PF_190_546', 'PF_204_174', 'PF_218_776', \
                           'PF_225_000', 'PF_234_423', 'PF_251_189', 'PF_269_153', 'PF_288_403', 'PF_300_000', 'PF_309_030', 'PF_331_131', 'PF_354_813', \
                           'PF_380_189', 'PF_407_380', 'PF_436_516', 'PF_467_735', 'PF_500_000', 'PF_501_187', 'PF_537_032', 'PF_575_440', 'PF_616_595', \
                           'PF_660_693', 'PF_707_946', 'PF_758_578', 'PF_812_831', 'PF_870_964', 'PF_933_254', 'PF_1000_000']

    ## set query string
    query_string = "SELECT"
    query_string += " {:s}".format(ColumnNameForNamePartFunc)                               ## position 1: molecule name
    for local_temp in ColumnNamesPartFunc:
        query_string += ", {:s}".format(local_temp)                                         ## position 2 - 111: temperature
    query_string += " FROM {:s} WHERE (".format(NameOfPartFuncTable)
    query_string += " {:s} = {:s}{:s}{:s})".format(ColumnNameForNamePartFunc, chr(34), MolName, chr(34))

    # Debug:
    # print ("query_string = ", query_string)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get partition function values from database
    dbFile = dbList[0]
    try:
        conn = sqlite3.connect(dbFile)
    except sqlite3.Error as e:
        print ("\n\nError in subroutine myXCLASSMapFit.GetPartitionFunctionFunc:")
        print ("\t\tCan not connect to sqlite3 databse {:s}.".format(dbFile))
        print ("\t\tError: {:d}: {:s}".format(e.args[0], e.args[1]))
        sys.exit(1)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read data from database
    cursor = conn.cursor()
    cursor.execute(query_string)
    conn.commit()
    rows = cursor.fetchall()
    conn.close()

    # Debug:
    # print ("\nQueryString = ", QueryString)
    # print ("rows = ", rows)


    ## check, if at least one entry was found
    if (len(rows) == 0):
        return None


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## extract partition function values
    TempArray = [1.072, 1.148, 1.23, 1.318, 1.413, 1.514, 1.622, 1.738, 1.862, 1.995, 2.138, 2.291, 2.455, 2.63, 2.725, 2.818, 3.02, 3.236, 3.467, \
                 3.715, 3.981, 4.266, 4.571, 4.898, 5, 5.248, 5.623, 6.026, 6.457, 6.918, 7.413, 7.943, 8.511, 9.12, 9.375, 9.772, 10.471, 11.22, \
                 12.023, 12.882, 13.804, 14.791, 15.849, 16.982, 18.197, 18.75, 19.498, 20.893, 22.387, 23.988, 25.704, 27.542, 29.512, 31.623, \
                 33.884, 36.308, 37.5, 38.905, 41.687, 44.668, 47.863, 51.286, 54.954, 58.884, 63.096, 67.608, 72.444, 75, 77.625, 83.176, 89.125, \
                 95.499, 102.329, 109.648, 117.49, 125.893, 134.896, 144.544, 150, 154.882, 165.959, 177.828, 190.546, 204.174, 218.776, 225, \
                 234.423, 251.189, 269.153, 288.403, 300, 309.03, 331.131, 354.813, 380.189, 407.38, 436.516, 467.735, 500, 501.187, 537.032, \
                 575.44, 616.595, 660.693, 707.946, 758.578, 812.831, 870.964, 933.254, 1000.0]
    QTArray = rows[0][1:]

    # Debug:
    # print ("TempArray = ", TempArray)
    # print ("len(TempArray) = ", len(TempArray))
    # print ("len(QTArray) = ", len(QTArray))


    ## interpolate QT array
    try:
        QTFunc = interp1d(TempArray, QTArray, fill_value = 'extrapolate')
    except:
        QTFunc = 1.0

    # Debug:
    # print ("QTFunc = ", QTFunc)


    ## define return parameter
    return QTFunc
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check given string for boolean
##
def CheckBool(OptionValue):
    """

input parameters:
-----------------

    - OptionValue:          string to analyze


output parameters:
------------------

    - OutFlag:              boolean value
    """

    # Debug:
    # print ("OptionValue = ", OptionValue)


    ## initialize return parameter
    OutFlag = False


    ## define list of known "true" phrases
    ListOfKnownTruePhrases = ['1', 't', 'true', 'y', 'yes', 'yeah']


    ## define return parameter
    OptionValue = str(OptionValue).strip()
    OptionValue = OptionValue.lower()
    if (OptionValue in ListOfKnownTruePhrases):
        OutFlag = True


    ## we're done
    return OutFlag
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check given string for number
##
def CheckNumber(OptionValue):
    """

input parameters:
-----------------

    - OptionValue:          string to analyze


output parameters:
------------------

    - OutFlag:              boolean value
    """

    # Debug:
    # print ("OptionValue = ", OptionValue)


    ## initialize return parameter
    try:
        float(str(OptionValue))
        OutFlag = True
    except ValueError:
        OutFlag = False


    ## we're done
    return OutFlag
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## import and analyze obs. data file(s) and extract parameters from obs. xml file
##
def AnaylzeObsXMLFile(ObsXMLFileName, RegionFileName, FITSImageMaskFileName, NameOfFunction, GlobalThreshold = None, \
                      TransFlag = False, TransDict = None, FastFitFlag = False, FullFitFlag = False, RemoveContinuumFlag = False, \
                      daskFlag = False, EstimateContinuumFlag = False, EstimateConNumEntireCubeFlag = True, EstimateConNumParts = 20, \
                      NumberCoresEstCont = 1, CompRegions = None, FITSImageMask = None):
    """

input parameters:
-----------------

    - ObsXMLFileName:                   path and name of obs. xml file

    - RegionFileName:                   path and name of file describing region(s)

    - FITSImageMaskFileName:            path and name of file describing FITS image used for pixel selection

    - NameOfFunction:                   name of calling function

    - GlobalThreshold:                  (optional) global thresholds used for each frequency range (default: None)

    - TransFlag:                        (optional) flag indicating that molecular parameters are stored as well
                                        (default: False)

    - TransDict:                        (optinal) dictionary containing some parameters used for import of
                                        molecular parameters (default: None)
                                        TransDict['MolNameList']
                                        TransDict['SQLParamArray']

    - FastFitFlag:                      (optional) flag indicating usage of fast-fitting method (default: False)

    - FullFitFlag:                      (optional) flag indicating usage of full-fitting method (default: False)

    - RemoveContinuumFlag:              (optional) flag indicating removing of continuum (default: False)

    - daskFlag:                         (optional) flag indicating usage of dask package (default: False)

    - EstimateContinuumFlag:            (optional) flag indicating continuum estimation (default: False)

    - EstimateConNumEntireCubeFlag:     (optional, only used for continuum estimation) flag indicating continuum
                                        estimation for entire frequency range (default: True)

    - EstimateConNumParts:              (optional, only used for continuum estimation) number of subparts
                                        used for non-constant continuum estimation (default: 20)

    - NumberCoresEstCont:               (optional, only used for continuum estimation) number of cores
                                        used for estimation (default: 1)

    - CompRegions:                      (optional) region object (default: None)

    - FITSImageMask:                    (optional) FITS image used for masking (default: None)


output parameters:
------------------

    - ParamDict:                        python dictionary containing parameters
    """

    # Debug:
    # print ("ObsXMLFileName = ", ObsXMLFileName)
    # print ("RegionFileName = ", RegionFileName)
    # print ("FITSImageMaskFileName = ", FITSImageMaskFileName)
    # print ("NameOfFunction = ", NameOfFunction)
    # print ("GlobalThreshold = ", GlobalThreshold)
    # print ("TransFlag = ", TransFlag)
    # print ("TransDict = ", TransDict)
    # print ("FastFitFlag = ", FastFitFlag)
    # print ("FullFitFlag = ", FullFitFlag)
    # print ("RemoveContinuumFlag = ", RemoveContinuumFlag)
    # print ("daskFlag = ", daskFlag)
    # print ("EstimateContinuumFlag = ", EstimateContinuumFlag)
    # print ("EstimateConNumEntireCubeFlag = ", EstimateConNumEntireCubeFlag)
    # print ("EstimateConNumParts = ", EstimateConNumParts)
    # print ("NumberCoresEstCont = ", NumberCoresEstCont)
    # print ("CompRegions = ", CompRegions)
    # print ("FITSImageMask = ", FITSImageMask)


    ## initialize return parameter
    ParamDict = {}


    ##------------------------------------------------------------------------------------------------------------------------------------------------
    ## import ds9 region file
    if (RegionFileName != ""):
        region_list = Regions.read(RegionFileName, format = 'ds9')


        ## prepare region if only one region was selected
        if (len(region_list) == 1):
            CompRegions = copy.deepcopy(region_list)


        ## prepare region if many regions were selected
        else:
            # for region in region_list:
            #     print ("\n\nregion = ", region)
            #     if (CompRegions is None):
            #         CompRegions = copy.deepcopy(region)
            #     else:
            #         CompRegions = CompRegions | region

            #     # Debug:
            #     # print ("\n\nregion = ", region)


            ## prepare comp. variable for spectral cube
            CompRegions = copy.deepcopy(region_list)
            # CompRegions = [CompRegions]

    # Debug:
    # print ("CompRegions = ", CompRegions)


    ##------------------------------------------------------------------------------------------------------------------------------------------------
    ## import FITS image used for masking
    if (FITSImageMaskFileName != ""):
        FITSImageMaskFileName = FITSImageMaskFileName.strip()
        FITSImageMaskValues = Subimage_from_regions(FITSImageMaskFileName, CompRegions)
        FITSImageMask = FITSImageMaskValues
        FITSImageMaskValues = FITSImageMaskValues.value
    else:
        FITSImageMaskValues = None

    # Debug:
    # print ("FITSImageMask.shape = ", FITSImageMask.shape)
    # print ("FITSImageMask = ", FITSImageMask)
    # print ("FITSImageMaskValues = ", FITSImageMaskValues)


    ##------------------------------------------------------------------------------------------------------------------------------------------------
    ## obs. xml file


    ## import parameters from obs. xml file
    LocalObsXMLParameter = task_LineIdentification.GetObsXMLParameters(ObsXMLFileName)


    ## get obs. xml file parameters from dictionary
    ExpFileList = LocalObsXMLParameter['ExpFileList']
    # ImportFilterList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "ImportFilter")
    NumberExpRangesList = LocalObsXMLParameter['NumberExpRangesList']
    FreqMinList = LocalObsXMLParameter['FreqMinList']
    FreqMaxList = LocalObsXMLParameter['FreqMaxList']
    FreqStepList = LocalObsXMLParameter['FreqStepList']
    t_back_flagList = LocalObsXMLParameter['t_back_flagList']
    tBackList = LocalObsXMLParameter['tBackList']
    tSlopeList = LocalObsXMLParameter['tSlopeList']
    N_HList = LocalObsXMLParameter['N_HList']
    beta_dustList = LocalObsXMLParameter['beta_dustList']
    kappa_1300List = LocalObsXMLParameter['kappa_1300List']
    DustFileNameList = LocalObsXMLParameter['DustFileNameList']
    BackgroundFileNameList = LocalObsXMLParameter['BackgroundFileNameList']
    ContPhenFuncIDList = LocalObsXMLParameter['ContPhenFuncIDList']
    ContPhenFuncParam1List = LocalObsXMLParameter['ContPhenFuncParam1List']
    ContPhenFuncParam2List = LocalObsXMLParameter['ContPhenFuncParam2List']
    ContPhenFuncParam3List = LocalObsXMLParameter['ContPhenFuncParam3List']
    ContPhenFuncParam4List = LocalObsXMLParameter['ContPhenFuncParam4List']
    ContPhenFuncParam5List = LocalObsXMLParameter['ContPhenFuncParam5List']
    NoiseList = LocalObsXMLParameter['NoiseList']
    SmoothList = LocalObsXMLParameter['SmoothList']
    ListOfThresholds = LocalObsXMLParameter['ThresholdList']
    GlobalvLSRList = LocalObsXMLParameter['GlobalvLSRList']
    TelescopeSizeList = LocalObsXMLParameter['TelescopeSizeList']
    BMINList = LocalObsXMLParameter['BMINList']
    BMAJList = LocalObsXMLParameter['BMAJList']
    BPAList = LocalObsXMLParameter['BPAList']
    Inter_FlagList = LocalObsXMLParameter['Inter_FlagList']
    RedshiftList = LocalObsXMLParameter['RedshiftList']
    ErrorYList = LocalObsXMLParameter['ErrorYList']
    NumberHeaderLinesList = LocalObsXMLParameter['NumberHeaderLinesList']
    SeparatorColumnsList = LocalObsXMLParameter['SeparatorColumnsList']
    IsoTableFileNameList = LocalObsXMLParameter['IsoTableFileNameList']
    dbList = LocalObsXMLParameter['dbList']
    NumModelPixelXXList = LocalObsXMLParameter['NumModelPixelXXList']
    NumModelPixelYYList = LocalObsXMLParameter['NumModelPixelYYList']
    LocalOverlapFlagList = LocalObsXMLParameter['LocalOverlapFlagList']
    NoSubBeamFlagList = LocalObsXMLParameter['NoSubBeamFlagList']
    EmAbsPATHList = LocalObsXMLParameter['EmAbsPATHList']


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## analyze some parameters


    ## get path and name of iso ratio file
    IsoRatioFileName = ""
    if (len(IsoTableFileNameList) > 0):
        IsoRatioFileName = IsoTableFileNameList[0]


    ## set local-overlap flag
    LocalOverlapFlag = 0
    if (len(LocalOverlapFlagList) > 0):
        LocalOverlapFlag = CheckBool(LocalOverlapFlagList[0])
        if (FastFitFlag):
            if (LocalOverlapFlag):
                LocalOverlapFlag = 1
            else:
                LocalOverlapFlag = 0


    ## analyze usage of emission and absorption functions
    LocalEmAbsPATH = None
    if (EmAbsPATHList != []):
        print ("Import emission and absorption functions from files ..")
        LocalEmAbsPATH = EmAbsPATHList[0]
        EmsAbsFileInterDict = task_LineIdentification.GetEmAbsFunc(LocalEmAbsPATH)
    else:
        LocalEmAbsPATH = None
        EmsAbsFileInterDict = None


    ## get number of grid pixels along x-axis
    if (NumModelPixelXXList == []):
        NumModelPixelXX = 100
    else:
        NumModelPixelXX = NumModelPixelXXList[0]


    ## get number of grid pixels along y-axis
    if (NumModelPixelYYList == []):
        NumModelPixelYY = 100
    else:
        NumModelPixelYY = NumModelPixelYYList[0]


    ## get sub-beam flag
    if (NoSubBeamFlagList == []):
        NoSubBeamFlag = True
    else:
        NoSubBeamFlag = NoSubBeamFlagList[0]
        NoSubBeamFlag = CheckBool(NoSubBeamFlag)


    ## get path and name of database file
    dbFileName = dbList[0]

    # Debug:
    # print ("ExpFileList = ", ExpFileList)
    # print ("ImportFilterList = ", ImportFilterList)
    # print ("NumberExpRangesList = ", NumberExpRangesList)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("FreqStepList = ", FreqStepList)
    # print ("t_back_flagList = ", t_back_flagList)
    # print ("tBackList = ", tBackList)
    # print ("tSlopeList = ", tSlopeList)
    # print ("N_HList = ", N_HList)
    # print ("beta_dustList = ", beta_dustList)
    # print ("kappa_1300List = ", kappa_1300List)
    # print ("DustFileNameList = ", DustFileNameList)
    # print ("BackgroundFileNameList = ", BackgroundFileNameList)
    # print ("ContPhenFuncIDList = ", ContPhenFuncIDList)
    # print ("ContPhenFuncParam1List = ", ContPhenFuncParam1List)
    # print ("ContPhenFuncParam2List = ", ContPhenFuncParam2List)
    # print ("ContPhenFuncParam3List = ", ContPhenFuncParam3List)
    # print ("ContPhenFuncParam4List = ", ContPhenFuncParam4List)
    # print ("ContPhenFuncParam5List = ", ContPhenFuncParam5List)
    # print ("NoiseList = ", NoiseList)
    # print ("SmoothList = ", SmoothList)
    # print ("TelescopeSizeList = ", TelescopeSizeList)
    # print ("BMINList = ", BMINList)
    # print ("BMAJList = ", BMAJList)
    # print ("BPAList = ", BPAList)
    # print ("Inter_FlagList 0 ", Inter_FlagList)
    # print ("GlobalvLSRList = ", GlobalvLSRList)
    # print ("RedshiftList = ", RedshiftList)
    # print ("NumberHeaderLinesList = ", NumberHeaderLinesList)
    # print ("SeparatorColumnsList = ", SeparatorColumnsList)
    # print ("dbList = ", dbList)
    # print ("NumModelPixelXXList = ", NumModelPixelXXList)
    # print ("NumModelPixelYYList = ", NumModelPixelYYList)
    # print ("LocalOverlapFlagList = ", LocalOverlapFlagList)
    # print ("NoSubBeamFlagList = ", NoSubBeamFlagList)
    # print ("EmAbsPATHList = ", EmAbsPATHList)
    # print ("LocalEmAbsPATH = ", LocalEmAbsPATH)
    # print ("dbFileName = ", dbFileName)


    ##------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze obs. data files and determine v_lsr corrected frequency ranges
    NumExpDataFiles = len(ExpFileList)                                                      ## get number of obs. data files
    ListObsDataFormats = []
    PixelMask = None
    DECAxis = None
    RAAxis = None
    ListOfCubes = []
    ListOfFullCubes = []
    ListOfSpectralAxis = []
    ListOfFullSpectralAxis = []
    OrigUnits = []
    OrigRestFreq = []
    ParameterMapHeader = None
    ParameterMapWCS = None
    LocalFreqMinList = []
    LocalFreqMaxList = []
    for ObsDataFileIndex in range(NumExpDataFiles):                                         ## loop over all obs. data files


        ## get parameters from xml file for current range
        ObsDataFileName = ExpFileList[ObsDataFileIndex][1]
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                       NumberRangeListIn = NumberExpRangesList, \
                                                                       GlobalvLSRListIn = GlobalvLSRList, \
                                                                       Redshift_ListIn = RedshiftList, \
                                                                       NumberHeaderLinesListIn = NumberHeaderLinesList, \
                                                                       SeparatorColumnsListIn = SeparatorColumnsList)
        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']            ## get number of frequency ranges for current obs. data file
        GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
        if (GlobalvLSR is None):
            GlobalvLSR = 0.0
        Redshift = ObsXMLParameterDictFile['Redshift']
        if (Redshift is None):
            Redshift = 0.0
        LocalNumberHeaderLines = ObsXMLParameterDictFile['NumberHeaderLines']
        if (LocalNumberHeaderLines is None):
            LocalNumberHeaderLines = 0
        LocalSeparatorColumns = ObsXMLParameterDictFile['SeparatorColumns']


        ## print what you do
        print ("\n\n\tAnalyze obs. data file ({:d}/{:d}):".format((ObsDataFileIndex + 1), NumExpDataFiles))


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## import obs. data file


        ## import FITS cube
        FormatObsDataFile = ""
        LowerObsDataFileName = ObsDataFileName.lower()
        if (LowerObsDataFileName.endswith(".fits")):


            ## import FITS file
            print ("\t\tImport FITS data file .. ", end = "", flush = True)
            CubeDict = FITSImport(ObsDataFileName, daskFlag = daskFlag)
            print ("done!")
            LocalFITSImageFlag = CubeDict['FITSImageFlag']


            ## check, if data file is a cube and not an image
            if (LocalFITSImageFlag):
                print ("\n\n\t\tWARNING:  The given obs. data file")
                print ("\t\t          {:s}".format(ObsDataFileName))
                print ("\t\t          describes a FITS image and not a FITS cube!")
                print ("\t\t          Please correct input and")
                print ("\t\t          start the {:s} function again!".format(NameOfFunction))
                print ("\n\n\t\t          Abort {:s} function now!".format(NameOfFunction))
                sys.exit(0)


            ## everything is fine, the obs. data file is a cube
            FormatObsDataFile = "fits"
            ListObsDataFormats.append(FormatObsDataFile)
            LocalCube = CubeDict['Cube']
            LocalCube.allow_huge_operations = True
            OrigIntUnit = CubeDict['UnitCube']
            OrigspecUnit = CubeDict['FrequencyAxisUnit']
            # FrequencyAxis = CubeDict['FrequencyAxis']
            OrigUnits.append([OrigIntUnit, OrigspecUnit])

            # Debug:
            # print ("OrigIntUnit = ", OrigIntUnit)
            # print ("OrigspecUnit = ", OrigspecUnit)


        ## import ASCII file
        else:
            FormatObsDataFile = "ascii"
            ListObsDataFormats.append(FormatObsDataFile)
            LocalNumberHeaderLines = int(LocalNumberHeaderLines)
            if (LocalSeparatorColumns.strip() == ""):
                ImportObsDataASCII = numpy.loadtxt(ObsDataFileName, skiprows = LocalNumberHeaderLines)
            else:
                ImportObsDataASCII = numpy.loadtxt(ObsDataFileName, skiprows = LocalNumberHeaderLines, delimiter = LocalSeparatorColumns)
            ListOfFullCubes.append(ImportObsDataASCII)


            ## define units etc.
            OrigIntUnit = "K"
            OrigspecUnit = "MHz"
            OrigUnits.append([OrigIntUnit, OrigspecUnit])
            DECAxis = [0.0]
            RAAxis = [0.0]
            OrigRestFreq.append(0.0)
            ParameterMapHeader = None
            ParameterMapWCS = None
            PixelMask = [[1]]
            PixelMask = numpy.asarray(PixelMask)

            # Debug:
            # print ("OrigIntUnit = ", OrigIntUnit)
            # print ("OrigspecUnit = ", OrigspecUnit)
            # print ("DECAxis = ", DECAxis)
            # print ("RAAxis = ", RAAxis)
            # print ("PixelMask = ", PixelMask)


            ## get min and max of spectral axis (in MHz)
            FullFreqMin = numpy.nanmin(ImportObsDataASCII[:, 0])
            FullFreqMax = numpy.nanmax(ImportObsDataASCII[:, 0])
            ListOfFullSpectralAxis.append([FullFreqMin, FullFreqMax])

            # Debug:
            # print ("FullFreqMin = ", FullFreqMin)
            # print ("FullFreqMax = ", FullFreqMax)


        ##************************************************************************************************************************************************
        ## FITS file only: apply regions and mask image
        if (FormatObsDataFile == "fits"):


            ## apply region file
            if (CompRegions is not None):
                print ("\t\tApply region to FITS file .. ", end = "", flush = True)
                # LocalCube = LocalCube.subcube_from_regions(CompRegions)
                LocalCube = local_subcube_from_regions(LocalCube, CompRegions)
                print ("done!")


            ## apply FITS image to mask
            if (FITSImageMask is not None):
                print ("\t\tApply mask defined by image to FITS file .. ", end = "", flush = True)
                decImage, raImage = FITSImageMask.world[:]


                ## get limits of ra axis
                RAAxis = raImage[0, :]
                xlo = numpy.nanmin(RAAxis.value) * RAAxis.unit
                xhi = numpy.nanmax(RAAxis.value) * RAAxis.unit


                ## get limits of dec axis
                DECAxis = decImage[:, 0]
                ylo = numpy.nanmin(DECAxis.value) * DECAxis.unit
                yhi = numpy.nanmax(DECAxis.value) * DECAxis.unit


                ## get limits of spectral axis
                LocalFrequencyAxis = LocalCube.spectral_axis.value
                LocalFrequencyAxisUnit = LocalCube.spectral_axis.unit
                LocalMapFreqMin = numpy.nanmin(LocalFrequencyAxis)
                LocalMapFreqMax = numpy.nanmax(LocalFrequencyAxis)
                zlo = LocalMapFreqMin * LocalFrequencyAxisUnit
                zhi = LocalMapFreqMax * LocalFrequencyAxisUnit

                # Debug:
                # print ("xlo = ", xlo)
                # print ("xhi = ", xhi)
                # print ("ylo = ", ylo)
                # print ("yhi = ", yhi)
                # print ("zlo = ", zlo)
                # print ("zhi = ", zhi)
                # print ("RAAxis = ", RAAxis)
                # print ("DECAxis = ", DECAxis)
                # print ("LocalFrequencyAxis = ", LocalFrequencyAxis)
                # print ("LocalCube = ", LocalCube)


                ## shrink cube to masking FITS image
                LocalCube = LocalCube.subcube(xlo = xlo, xhi = xhi,
                                              ylo = ylo, yhi = yhi,
                                              zlo = zlo, zhi = zhi)
                ## apply mask
                # FITSImageMask = numpy.isnan(FITSImageMask)
                # print ("LocalCube = ", LocalCube)


                ## we're done
                print ("done!")

                # Debug:
                # print ("FITSImageMask = ", FITSImageMask)
                # print ("NumberNonNaNPixels = ", numpy.count_nonzero(~numpy.isnan(FITSImageMask)))


            ## initialize mask describing region
            ## Here the first two lines are necessary to handle cubes, which have some NaN along their spectral axis within the selected ds9 region(s)
            if (ObsDataFileIndex == 0):
                PixelMask = LocalCube[:2, :, :]
                PixelMask = PixelMask.with_fill_value(0.0)
                if (FITSImageMask is not None):
                    PixelMask = numpy.where(numpy.isnan(FITSImageMask), False, True)
                else:
                    PixelMask = PixelMask[0, :, :]
                    PixelMask = numpy.where(numpy.isnan(PixelMask), False, True)

                # Debug:
                # print ("PixelMask = ", PixelMask)


                ## check, if mask map is empty
                if (PixelMask.shape[1] == 0 and PixelMask.shape[2] == 0):
                    print ("\n\n\t\tWARNING:  After applying the selected mask(s) to the current FITS cube")
                    print ("\t\t          {:s}".format(ObsDataFileName))
                    print ("\t\t          no pixel is selected to fit!")
                    print ("\t\t          Please check, the selected ds9 region file and / or the defined FITS image for masking")
                    print ("\t\t          and start the {:s} function again!".format(NameOfFunction))
                    print ("\n\n\t\t          Abort {:s} function now!".format(NameOfFunction))
                    sys.exit(0)


            ## check, if size of obs. data FITS cubes have the same number of pixel in both spatial directions
            else:
                if (PixelMask.shape[0] != LocalCube[0, :, :].shape[0] or PixelMask.shape[1] != LocalCube[0, :, :].shape[1]):
                    print ("\n\n\t\tWARNING:  The current obs. data FITS cube {:s}".format(ObsDataFileName))
                    print ("\t\t          has a different number of pixels along the dec-direction and / or ra-direction")
                    print ("\t\t          compared to other cubes!")
                    print ("\t\t          Shape of other obs. data cube(s): ", PixelMask.shape)
                    print ("\t\t          Shape of this obs. data cube:     ", LocalCube[0, :, :].shape)
                    print ("\n\n")

            # Debug:
            # print ("PixelMask = ", PixelMask)
            # print ("NumberNonNaNPixels = ", numpy.count_nonzero(PixelMask))
            # sys.exit(0)


            ## store RA and DEC axis for the first data cube
            if (ObsDataFileIndex == 0):
                velo, dec, ra = LocalCube.world[:]
                DECAxis = dec[0, :, 0]
                RAAxis = ra[0, 0, :]


            ##********************************************************************************************************************************************
            ## convert units


            ## convert spectral axis to MHz, if necessary
            restfreqList = None
            if (not OrigspecUnit in ["MHz", "mhz"]):
                print ("\t\tConvert spectral axis to MHz .. ", end = "", flush = True)
                if (OrigspecUnit in ["m / s", "m/s", "M/S", "M / S", "km / s", "km/s", "KM/S", "KM / S"]):
                    Cube1Header = LocalCube.header
                    try:
                        restfreq = Cube1Header['RESTFREQ']
                    except:
                        restfreq = Cube1Header['RESTFRQ']
                    velConv = "optical"
                    try:
                        ctype3 = Cube1Header['CTYPE3']
                    except:
                        ctype3 = None
                    if (ctype3 is not None):
                        ctype3 = ctype3.strip()
                        ctype3 = ctype3.lower()
                        if (ctype3 in ["vrad"]):
                            velConv = "radio"
                    restfreqList = [restfreq, velConv]
                    LocalCube = LocalCube.with_spectral_unit(u.MHz, velocity_convention = velConv, rest_value = restfreq * u.Hz)
                else:
                    LocalCube = LocalCube.with_spectral_unit(u.MHz)
                print ("done!")
            OrigRestFreq.append(restfreqList)
            ListOfFullCubes.append(LocalCube)

            # Debug:
            # print ("\nOrigIntUnit = ", OrigIntUnit)
            # print ("OrigspecUnit = ", OrigspecUnit)
            # print ("restfreqList = ", restfreqList)
            # print ("LocalCube.spectral_axis = ", LocalCube.spectral_axis)


            ##*-------------------------------------------------------------------------------------------------------------------------------------------
            ## get spatial coordinates and header from moment 1 map
            if (ObsDataFileIndex == 0):
                m1 = LocalCube[:1, :, :].moment(order = 1)
                ParameterMapHeader = copy.deepcopy(m1.header)
                # ParameterMapWCS = copy.deepcopy(m1.wcs)
                ParameterMapWCS = WCS(ParameterMapHeader)

            # Debug:
            # print ("ParameterMapHeader = ", ParameterMapHeader)
            # print ("ParameterMapWCS = ", ParameterMapWCS)


            ##********************************************************************************************************************************************
            ## get min and max of spectral axis (in MHz)
            FullFreqMin = numpy.nanmin(LocalCube.spectral_axis.value)
            FullFreqMax = numpy.nanmax(LocalCube.spectral_axis.value)
            ListOfFullSpectralAxis.append([FullFreqMin, FullFreqMax])


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## get v_lsr corrected frequency ranges
        for RangeIndex in range(NumberFrequencyRanges):                                     ## loop over all range definitions in the whole xml file


            ## get parameters for current range
            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                            FreqMinListIn = FreqMinList, \
                                                                            FreqMaxListIn = FreqMaxList)
            FreqMin = ObsXMLParameterDictRange['FreqMin']
            FreqMax = ObsXMLParameterDictRange['FreqMax']
            if (FreqMin is None or FreqMax is None):
                FreqMin = FullFreqMin
                FreqMax = FullFreqMax
            else:
                FreqMin = max(FullFreqMin, FreqMin)
                FreqMax = min(FullFreqMax, FreqMax)

            # Debug:
            # print ("\n\nObsDataFileIndex, RangeIndex = ", ObsDataFileIndex, RangeIndex)
            # print ("FreqMin = ", FreqMin)
            # print ("FreqMax = ", FreqMax)
            # print ("GlobalvLSR = ", GlobalvLSR)


            ## shift min. and max. frequency back to original value for each range by v_LSR
            NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
            NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
            ShiftedFreqMin = min(NewFreqMin, NewFreqMax)
            ShiftedFreqMax = max(NewFreqMin, NewFreqMax)
            LocalFreqMinList.append(ShiftedFreqMin)
            LocalFreqMaxList.append(ShiftedFreqMax)

    # Debug:
    # print ("ListOfCubes = ", ListOfCubes)
    # print ("ListOfFullCubes = ", ListOfFullCubes)
    # print ("ListOfSpectralAxis = ", ListOfSpectralAxis)
    # print ("ListOfFullSpectralAxis = ", ListOfFullSpectralAxis)
    # print ("LocalFreqMinList = " , LocalFreqMinList)
    # print ("LocalFreqMaxList = " , LocalFreqMaxList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get a list of all transition frequencies in all selected frequency ranges
    ## SQLParamArray = [MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL]
    if (TransFlag):
        MolNameList = TransDict['MolNameList']
        SQLParamArray = TransDict['SQLParamArray']
        MaxNumTransitionsSQL = SQLParamArray[1]


        ## get transition frequencies from database file
        print ("\n\tGet molecular parameters .. ", end = "", flush = True)
        GlobalTransFreqList, GlobalDBParamList = task_LineIdentification.GetTransFreq(MolNameList, LocalFreqMinList, \
                                                                                      LocalFreqMaxList, SQLParamArray, \
                                                                                      dbFileName, MaxNumTransitionsSQL, \
                                                                                      MinDistTransFreq = -100.0)
        GlobalTransFreqNPList = numpy.asarray(GlobalTransFreqList)
        print ("done!")

        # Debug:
        # print ("\n\n")
        # print ("MolNameList = ", MolNameList)
        # print ("GlobalTransFreqNPList = ", GlobalTransFreqNPList)
        # print ("GlobalDBParamList = ", GlobalDBParamList)
        # print ("LocalFreqMinList = ", LocalFreqMinList)
        # print ("LocalFreqMaxList = ", LocalFreqMaxList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## determine root velocities for each frequency range
    LocalRangeCounter = (-1)
    RangeParam = []
    ThresholdPixelMask = numpy.zeros((PixelMask.shape[0], PixelMask.shape[1]), dtype = bool)
    for ObsDataFileIndex in range(len(ExpFileList)):
        ObsDataFileName = ExpFileList[ObsDataFileIndex][1]


        ## print what you do
        print ("\n\n\tGet xml parameter for obs. data file ({:d}/{:d}):".format((ObsDataFileIndex + 1), NumExpDataFiles))

        # Debug:
        # print ("ExpFileList = ", ExpFileList)
        # print ("FreqMinList = ", FreqMinList)


        ## get masked FITS cube, allow huge operations and get limits of spectral axis
        FormatObsDataFile = ListObsDataFormats[ObsDataFileIndex]
        if (FormatObsDataFile == "fits"):
            LocalCube = ListOfFullCubes[ObsDataFileIndex]
            LocalCube.allow_huge_operations = True
        elif (FormatObsDataFile == "ascii"):
            LocalASCIIData = ListOfFullCubes[ObsDataFileIndex]
            ThresholdPixelMask = [[True]]
            ThresholdPixelMask = numpy.asarray(ThresholdPixelMask)
        FullFreqMin = ListOfFullSpectralAxis[ObsDataFileIndex][0]
        FullFreqMax = ListOfFullSpectralAxis[ObsDataFileIndex][1]
        OrigIntUnit = OrigUnits[ObsDataFileIndex][0]
        OrigspecUnit = OrigUnits[ObsDataFileIndex][1]

        # Debug:
        # print ("\n\nObsDataFileIndex = ", ObsDataFileIndex)
        # print ("FullFreqMin, FullFreqMax = ", FullFreqMin, FullFreqMax)
        # print ("LocalCube = ", LocalCube)
        # print ("CompRegions = ", CompRegions)
        # print ("FITSImageMask = ", FITSImageMask)
        # print ("OrigIntUnit = ", OrigIntUnit)
        # print ("OrigspecUnit = ", OrigspecUnit)


        ##************************************************************************************************************************************************
        ## estimate continuum level using the entire cube / spectrum
        GlobaltBack = None
        GlobaltSlope = None
        if (EstimateContinuumFlag and EstimateConNumEntireCubeFlag):


            ## estimate continuum of cube
            if (FormatObsDataFile == "fits"):
                if (not OrigIntUnit in ["K"]):
                    print ("\t\tConvert intensities to Kelvin .. ", end = "", flush = True)
                    LocalCubeConverted = LocalCube.to(u.K)
                    print ("done!")
                    GlobaltBack, GlobaltSlope = EstimateContinuumCube(LocalCubeConverted, NumParts = EstimateConNumParts, \
                                                                      RefFreq = numpy.nanmin(LocalCubeConverted.spectral_axis.value))
                else:
                    GlobaltBack, GlobaltSlope = EstimateContinuumCube(LocalCube, NumParts = EstimateConNumParts, \
                                                                      RefFreq = numpy.nanmin(LocalCube.spectral_axis.value))


            ## estimate continuum of single spectrum
            elif (FormatObsDataFile == "ascii"):
                DummyData, GlobaltBack, GlobaltSlope = task_LineIdentification.EstimateContinuum(LocalASCIIData)


        ##************************************************************************************************************************************************
        ## get parameters for current range
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                       NumberRangeListIn = NumberExpRangesList, \
                                                                       TelescopeSizeListIn = TelescopeSizeList, \
                                                                       BMIN_ListIn = BMINList, \
                                                                       BMAJ_ListIn = BMAJList, \
                                                                       BPA_ListIn = BPAList, \
                                                                       InterFlagListIn = Inter_FlagList, \
                                                                       GlobalvLSRListIn = GlobalvLSRList, \
                                                                       Redshift_ListIn = RedshiftList, \
                                                                       ErrorYFlagListIn = ErrorYList)
        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
        InterFlag = ObsXMLParameterDictFile['InterFlag']
        InterFlag = CheckBool(InterFlag)
        if (InterFlag):
            InterFlag = 1
        else:
            InterFlag = 0
        TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
        if (TelescopeSize is None and FastFitFlag and (not FullFitFlag)):
            TelescopeSize = 0.0
        BMIN = ObsXMLParameterDictFile['BMIN']
        if (BMIN is None and FastFitFlag and (not FullFitFlag)):
            BMIN = 0.0
        BMAJ = ObsXMLParameterDictFile['BMAJ']
        if (BMAJ is None and FastFitFlag and (not FullFitFlag)):
            BMAJ = 0.0
        BPA = ObsXMLParameterDictFile['BPA']
        if (BPA is None and FastFitFlag and (not FullFitFlag)):
            BPA = 0.0
        if ((TelescopeSize == 0.0 and BMIN == 0.0 and BMAJ == 0.0) \
            or (TelescopeSize is None and BMIN is None and BMAJ is None) \
            or (TelescopeSize == 0.0 and BMIN is None and BMAJ is None)):
            if (FormatObsDataFile == "fits"):
                BMIN = LocalCube.beam.major.to(u.arcsec).value
                BMAJ = LocalCube.beam.minor.to(u.arcsec).value
                BPA = LocalCube.beam.pa.value
                if (FastFitFlag):
                    TelescopeSize = (BMIN + BMAJ) / 2.0
                    InterFlag = 1
        GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
        if (GlobalvLSR is None):
            GlobalvLSR = 0.0
        Redshift = ObsXMLParameterDictFile['Redshift']
        if (Redshift is None and FastFitFlag and (not FullFitFlag)):
            Redshift = 0.0
        ErrorY = ObsXMLParameterDictFile['ErrorY']
        ErrorY = CheckBool(ErrorY)

        # Debug:
        # print ("FormatObsDataFile = ", FormatObsDataFile)
        # print ("NumberFrequencyRanges = ", NumberFrequencyRanges)
        # print ("InterFlag = ", InterFlag)
        # print ("TelescopeSize = ", TelescopeSize)
        # print ("BMIN = ", BMIN)
        # print ("BMAJ = ", BMAJ)
        # print ("BPA = ", BPA)
        # print ("GlobalvLSR = ", GlobalvLSR)
        # print ("Redshift = ", Redshift)
        # print ("ErrorY = ", ErrorY)
        # sys.exit(0)


        ## get range parameters
        for RangeIndex in range(NumberFrequencyRanges):                                     ## loop over all range definitions in the whole xml file


            ## print what you do
            print ("\r\t\tAnalyze frequency range ({:d}/{:d}) .. ".format((RangeIndex + 1), NumberFrequencyRanges), end = "", flush = True)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get parameters for current range
            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                            FreqMinListIn = FreqMinList, \
                                                                            FreqMaxListIn = FreqMaxList, \
                                                                            FreqStepListIn = FreqStepList, \
                                                                            tBackFlagListIn = t_back_flagList, \
                                                                            tBackListIn = tBackList, \
                                                                            tSlopeListIn = tSlopeList, \
                                                                            N_HListIn = N_HList, \
                                                                            beta_dustListIn = beta_dustList, \
                                                                            kappa_1300ListIn = kappa_1300List, \
                                                                            DustFileNameListIn = DustFileNameList, \
                                                                            BackgroundFileNameListIn = BackgroundFileNameList, \
                                                                            ContPhenFuncID_ListIn = ContPhenFuncIDList, \
                                                                            ContPhenFuncParam1_ListIn = ContPhenFuncParam1List, \
                                                                            ContPhenFuncParam2_ListIn = ContPhenFuncParam2List, \
                                                                            ContPhenFuncParam3_ListIn = ContPhenFuncParam3List, \
                                                                            ContPhenFuncParam4_ListIn = ContPhenFuncParam4List, \
                                                                            ContPhenFuncParam5_ListIn = ContPhenFuncParam5List, \
                                                                            NoiseListIn = NoiseList, \
                                                                            SmoothValueListIn = SmoothList, \
                                                                            ThresholdListIn = ListOfThresholds)
            LocalRangeCounter += 1
            FreqMin = ObsXMLParameterDictRange['FreqMin']
            FreqMax = ObsXMLParameterDictRange['FreqMax']
            if (FreqMin is None or FreqMax is None):
                FreqMin = FullFreqMin
                FreqMax = FullFreqMax
            else:
                FreqMin = max(FullFreqMin, FreqMin)
                FreqMax = min(FullFreqMax, FreqMax)

            # Debug:
            # Contents = task_ListDatabase.ListDatabaseCore(FreqMin = FreqMin, FreqMax = FreqMax)
            # print ("\n\nFreqMin, FreqMax = ", FreqMin, FreqMax)
            # print ("Contents = ", Contents)

            FreqStep = ObsXMLParameterDictRange['FreqStep']
            t_back_flag = ObsXMLParameterDictRange['t_back_flag']
            t_back_flag = CheckBool(t_back_flag)
            nH_flag = True
            N_H = ObsXMLParameterDictRange['N_H']
            beta_dust = ObsXMLParameterDictRange['beta_dust']
            kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
            if (N_H is None or beta_dust is None or kappa_1300 is None):
                nH_flag = False
                N_H = 0.0
                beta_dust = 0.0
                kappa_1300 = 0.0
            else:
                kappa_1300 = kappa_1300 * (2.0 * 1.66e-24 / 100.0)
            DustFileName = ObsXMLParameterDictRange['DustFileName']
            BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
            LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
            if (LocalContPhenFuncID is None and FastFitFlag and (not FullFitFlag)):
                LocalContPhenFuncID = 0.0
            LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
            if (LocalContPhenFuncParam1 is None and FastFitFlag and (not FullFitFlag)):
                LocalContPhenFuncParam1 = 0.0
            LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
            if (LocalContPhenFuncParam2 is None and FastFitFlag and (not FullFitFlag)):
                LocalContPhenFuncParam2 = 0.0
            LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
            if (LocalContPhenFuncParam3 is None and FastFitFlag and (not FullFitFlag)):
                LocalContPhenFuncParam3 = 0.0
            LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
            if (LocalContPhenFuncParam4 is None and FastFitFlag and (not FullFitFlag)):
                LocalContPhenFuncParam4 = 0.0
            LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']
            if (LocalContPhenFuncParam5 is None and FastFitFlag and (not FullFitFlag)):
                LocalContPhenFuncParam5 = 0.0
            # NoiseLevel = ObsXMLParameterDictRange['NoiseLevel']
            LocalSmoothValue = ObsXMLParameterDictRange['SmoothValue']
            LocalThreshold = ObsXMLParameterDictRange['Threshold']
            if (GlobalThreshold is not None and LocalThreshold is None):
                LocalThreshold = GlobalThreshold

            # Debug:
            # print ("\n\n\n\n\n\n\n\n\n\n\n")
            # print ("MolfitsFileName = ", MolfitsFileName)
            # print ("ObsDataFileName = ", ObsDataFileName)
            # print ("ObsDataFileIndex, RangeIndex = ", ObsDataFileIndex, RangeIndex)
            # print ("FreqMin = ", FreqMin)
            # print ("FreqMax = ", FreqMax)
            # print ("t_back_flag = ", t_back_flag)
            # print ("tBack = ", tBack)
            # print ("tSlope = ", tSlope)
            # print ("LocalSmoothValue = ", LocalSmoothValue)
            # print ("AllowSmoothFlag = ", AllowSmoothFlag)
            # print ("BackgroundFileName = ", BackgroundFileName)
            # print ("DustFileName = ", DustFileName)


            ## shift min. and max. frequency back to original value for each range by v_LSR
            NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
            NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
            ShiftedFreqMin = min(NewFreqMin, NewFreqMax)
            ShiftedFreqMax = max(NewFreqMin, NewFreqMax)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get transition frequencies and other molecule parameter from global parameter list
            TransFreqList = []
            DBParamList = []
            if (TransFlag):
                idx = numpy.where((GlobalTransFreqNPList >= ShiftedFreqMin) & (GlobalTransFreqNPList <= ShiftedFreqMax))
                TransFreqIndexList = list(idx[0])
                for mm in TransFreqIndexList:
                    TransFreqList.append(GlobalTransFreqList[mm])
                    LocalTransitionParameter = GlobalDBParamList[mm]
                    LocalMoleculeName = LocalTransitionParameter[4]
                    try:
                        i = MolNameList.index(LocalMoleculeName)
                    except:
                        i = (-1)
                    if (i > (-1)):
                        LocalTransitionParameter[4] = i
                        DBParamList.append(LocalTransitionParameter)

                    # Debug:
                    # print ("\nTransFreq = ", TransFreqList[mm])
                    # print ("EinsteinA = ", DBParamList[mm][1])
                    # print ("ElowMin   = ", DBParamList[mm][2])
                    # print ("gup       = ", DBParamList[mm][3])
                    # print ("Name      = ", DBParamList[mm][4])

            # Debug:
            # print ("TransFreqList = ", TransFreqList)
            # print ("DBParamList = ", DBParamList)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## prepare cube for the current range


            ## extracting slab corresponding with current freqeuncy range
            if (FormatObsDataFile == "fits"):
                LocalSubCube = LocalCube
                if (FreqMin is None):
                    FreqMin = numpy.nanmin(LocalSubCube.spectral_axis.value)
                if (FreqMax is None):
                    FreqMax = numpy.nanmax(LocalSubCube.spectral_axis.value)
                LocalSubCube = LocalSubCube.spectral_slab(FreqMin * u.MHz, FreqMax * u.MHz)

                # Debug:
                # print ("LocalRangeCounter, FreqMin, FreqMax = ", LocalRangeCounter, FreqMin, FreqMax, LocalSubCube.spectral_axis)


            ## extracting slab from single spectrum
            elif (FormatObsDataFile == "ascii"):
                LocalSubASCIIData = copy.deepcopy(LocalASCIIData)
                if (FreqMin is not None and FreqMax is not None):
                    FreqMinIndex, FreqMaxIndex = task_myXCLASSFit.GetArrayPart(LocalASCIIData[:, 0], FreqMin, FreqMax)
                    LocalSubASCIIData = LocalASCIIData[FreqMinIndex:FreqMaxIndex]


            ##********************************************************************************************************************************************
            ## FITS file only: apply threshold (if threshold is NOT in Kelvin)
            if (FormatObsDataFile == "fits"):


                ## determine n-sigma level
                LocalThresholdString = str(LocalThreshold).lower()
                if (LocalThresholdString.find("sigma") > (-1)):
                    LocalThresholdString = LocalThresholdString.replace("-sigma", "")
                    LocalThresholdString = LocalThresholdString.replace("sigma", "").strip()
                    try:
                        LocalThresholdValue = float(LocalThresholdString)
                    except:
                        LocalThresholdValue = None
                    if (LocalThresholdValue is not None):
                        noisemap = LocalThresholdValue * LocalSubCube.std(axis = 0)
                        SigmaMask = LocalSubCube > noisemap
                        SigmaMask = numpy.where(numpy.isnan(SigmaMask), False, True)
                        ThresholdPixelMask = numpy.logical_or(ThresholdPixelMask, SigmaMask)


                ## apply threshold
                elif (LocalThreshold is not None):
                    try:
                        # ThresholdMask = LocalSubCube > LocalThreshold
                        ThresholdMask = numpy.where(LocalSubCube.max(axis = 0) >= LocalThreshold, True, False)
                    except:
                        ThresholdMask = numpy.where(LocalSubCube.max(axis = 0) >= LocalThreshold * OrigIntUnit, True, False)
                    ThresholdPixelMask = numpy.logical_or(ThresholdPixelMask, ThresholdMask)


                ## no threshold is defined, include all pixels
                else:
                    ThresholdMask = numpy.ones((PixelMask.shape[0], PixelMask.shape[1]), dtype = bool)
                    ThresholdPixelMask = numpy.logical_or(ThresholdPixelMask, ThresholdMask)

                # Debug:
                # print ("\n\nLocalThreshold = ", LocalThreshold)
                # print ("OrigIntUnit = ", OrigIntUnit)
                # print ("LocalSubCube = ", LocalSubCube)
                # print ("ThresholdPixelMask.shape = ", ThresholdPixelMask.shape)
                # print ("ThresholdPixelMask = ", ThresholdPixelMask)
                # print ("ThresholdMask = ", ThresholdMask)
                # print ("LocalSubCube.max(axis = 0) = ", LocalSubCube.max(axis = 0))
                # sys.exit(0)


                ##****************************************************************************************************************************************
                ## FITS file only: convert intensity to Kelvin, if necessary
                if (not OrigIntUnit in ["K"]):
                    print ("\n\t\tConvert intensities to Kelvin .. ", end = "", flush = True)
                    LocalSubCube = LocalSubCube.to(u.K)
                    print ("done!")


            ##********************************************************************************************************************************************
            ## estimate continuum level
            if (EstimateContinuumFlag):
                tBack = None
                tSlope = None
                if (EstimateConNumEntireCubeFlag):
                    TBackImage = GlobaltBack
                    TSlopeImage = GlobaltSlope
                else:
                    if (FormatObsDataFile == "fits"):
                        TBackImage, TSlopeImage = EstimateContinuumCube(LocalSubCube, NumParts = EstimateConNumParts)
                    elif (FormatObsDataFile == "ascii"):
                        DummyData, TBackImage, TSlopeImage = task_LineIdentification.EstimateContinuum(LocalSubASCIIData, \
                                                                                                       NumParts = EstimateConNumParts)
            else:
                TBackImage = None
                TSlopeImage = None
                tBack = ObsXMLParameterDictRange['tBack']
                if (tBack is None and FastFitFlag and (not FullFitFlag)):
                    tBack = 0.0
                tSlope = ObsXMLParameterDictRange['tSlope']
                if (tSlope is None and FastFitFlag and (not FullFitFlag)):
                    tSlope = 0.0


            ##********************************************************************************************************************************************
            ## smooth cube
            if (LocalSmoothValue is not None):
                if (FormatObsDataFile == "fits"):
                    LocalKernel = Gaussian1DKernel(LocalSmoothValue)
                    LocalSubCube = LocalSubCube.spectral_smooth(LocalKernel)
                elif (FormatObsDataFile == "ascii"):
                    LocalSubASCIIData[:, 1] = convolve1d(LocalSubASCIIData[:, 1])


            ##********************************************************************************************************************************************
            ## store modified cube and spectral axis
            if (FormatObsDataFile == "fits"):
                ListOfCubes.append(LocalSubCube)
                LocalSpectralAxis = LocalSubCube.spectral_axis
                ListOfSpectralAxis.append(LocalSpectralAxis)
            elif (FormatObsDataFile == "ascii"):
                ListOfCubes.append(LocalSubASCIIData)
                LocalSpectralAxis = LocalSubASCIIData[:, 0] * u.MHz
                ListOfSpectralAxis.append(LocalSpectralAxis)

            # Debug:
            # print ("LocalSubCube = ", LocalSubCube)
            # print ("FormatObsDataFile = ", FormatObsDataFile)
            # print ("LocalRangeCounter = ", LocalRangeCounter)
            # print ("LocalSpectralAxis = ", LocalSpectralAxis)


            ##********************************************************************************************************************************************
            ## get file describing background spectra, which corresponds to the current range
            LocalBackgroundCube = None
            if (BackgroundFileName is not None):
                if (BackgroundFileName != ""):
                    if (FormatObsDataFile == "fits"):
                        LocalBackgroundCube = GetHelperCubes(BackgroundFileName, LocalSubCube, PixelMask, "background spectra")
                    elif (FormatObsDataFile == "ascii"):
                        BackgroundSpectrum = numpy.loadtxt(BackgroundFileName)
                        if (FreqMin is not None and FreqMax is not None):
                            FreqMinIndex, FreqMaxIndex = task_myXCLASSFit.GetArrayPart(BackgroundSpectrum[:, 0], FreqMin, FreqMax)
                            if (abs(FreqMinIndex - FreqMaxIndex) < 1):
                                LocalBackgroundFunc = interp1d(BackgroundSpectrum[:, 0], \
                                                               BackgroundSpectrum[:, 1], \
                                                               fill_value = 'extrapolate')
                            else:
                                LocalBackgroundFunc = interp1d(BackgroundSpectrum[FreqMinIndex:FreqMaxIndex, 0], \
                                                               BackgroundSpectrum[FreqMinIndex:FreqMaxIndex, 1], \
                                                               fill_value = 'extrapolate')
                            LocalBackgroundCube = LocalBackgroundFunc(LocalSubASCIIData[:, 0])

            # Debug:
            # print ("BackgroundFileName = ", BackgroundFileName)
            # print ("LocalBackgroundCube = ", LocalBackgroundCube)
            # sys.exit(0)


            ##********************************************************************************************************************************************
            ## get file describing dust opacity, which corresponds to the current range
            LocalDustCube = None
            if (DustFileName is not None):
                if (DustFileName != ""):
                    if (FormatObsDataFile == "fits"):
                        LocalDustCube = GetHelperCubes(DustFileName, LocalSubCube, PixelMask, "dust opacity")
                    elif (FormatObsDataFile == "ascii"):
                        DustSpectrum = numpy.loadtxt(DustFileName)
                        if (FreqMin is not None and FreqMax is not None):
                            FreqMinIndex, FreqMaxIndex = task_myXCLASSFit.GetArrayPart(DustSpectrum[:, 0], FreqMin, FreqMax)

                            # Debug:
                            # print ("\n\nlen(DustSpectrum[:, 0]) = ", len(DustSpectrum[:, 0]))
                            # print ("FreqMinIndex, FreqMaxIndex = ", FreqMinIndex, FreqMaxIndex)

                            if (abs(FreqMinIndex - FreqMaxIndex) < 1):
                                LocalDustFunc = interp1d(DustSpectrum[:, 0], \
                                                         DustSpectrum[:, 1], \
                                                         fill_value = 'extrapolate')
                                                        #  bounds_error = False,
                                                        #  fill_value = (DustSpectrum[0, 1], DustSpectrum[-1, 1]))
                            else:
                                LocalDustFunc = interp1d(DustSpectrum[FreqMinIndex:FreqMaxIndex, 0], \
                                                         DustSpectrum[FreqMinIndex:FreqMaxIndex, 1], \
                                                         fill_value = 'extrapolate')
                            LocalDustCube = LocalDustFunc(LocalSubASCIIData[:, 0])

            # Debug:
            # print ("DustFileName = ", DustFileName)
            # print ("LocalDustCube = ", LocalDustCube)


            ##********************************************************************************************************************************************
            ## store obs. xml parameter for current TFW
            ObsXMLParameterList = {}
            ObsXMLParameterList['FITSFileName'] = ObsDataFileName
            ObsXMLParameterList['FormatObsDataFile'] = FormatObsDataFile
            ObsXMLParameterList['RemoveContinuumFlag'] = RemoveContinuumFlag
            ObsXMLParameterList['LocalOverlapFlag'] = LocalOverlapFlag
            ObsXMLParameterList['ObsDataFileIndex'] = ObsDataFileIndex
            ObsXMLParameterList['NumberFrequencyRanges'] = NumberFrequencyRanges
            ObsXMLParameterList['TelescopeSize'] = TelescopeSize
            ObsXMLParameterList['BMIN'] = BMIN
            ObsXMLParameterList['BMAJ'] = BMAJ
            ObsXMLParameterList['BPA'] = BPA
            ObsXMLParameterList['InterFlag'] = InterFlag
            ObsXMLParameterList['GlobalvLSR'] = GlobalvLSR
            ObsXMLParameterList['Redshift'] = Redshift
            ObsXMLParameterList['ErrorY'] = ErrorY
            ObsXMLParameterList['RangeIndex'] = RangeIndex
            ObsXMLParameterList['FreqMin'] = FreqMin
            ObsXMLParameterList['FreqMax'] = FreqMax
            # ObsXMLParameterList['ShiftedFreqMin'] = ShiftedFreqMin
            # ObsXMLParameterList['ShiftedFreqMax'] = ShiftedFreqMax
            ObsXMLParameterList['FreqStep'] = FreqStep
            ObsXMLParameterList['t_back_flag'] = t_back_flag
            ObsXMLParameterList['tBack'] = tBack
            ObsXMLParameterList['tSlope'] = tSlope
            ObsXMLParameterList['TBackImage'] = TBackImage
            ObsXMLParameterList['TSlopeImage'] = TSlopeImage
            ObsXMLParameterList['N_H'] = N_H
            ObsXMLParameterList['beta_dust'] = beta_dust
            ObsXMLParameterList['kappa_1300'] = kappa_1300
            ObsXMLParameterList['ContPhenFuncID'] = LocalContPhenFuncID
            ObsXMLParameterList['ContPhenFuncParam1'] = LocalContPhenFuncParam1
            ObsXMLParameterList['ContPhenFuncParam2'] = LocalContPhenFuncParam2
            ObsXMLParameterList['ContPhenFuncParam3'] = LocalContPhenFuncParam3
            ObsXMLParameterList['ContPhenFuncParam4'] = LocalContPhenFuncParam4
            ObsXMLParameterList['ContPhenFuncParam5'] = LocalContPhenFuncParam5
            ObsXMLParameterList['SmoothValue'] = LocalSmoothValue
            ObsXMLParameterList['EmAbsPATH'] = EmsAbsFileInterDict
            ObsXMLParameterList['NumModelPixelXX'] = NumModelPixelXX
            ObsXMLParameterList['NumModelPixelYY'] = NumModelPixelYY
            ObsXMLParameterList['NoSubBeamFlag'] = NoSubBeamFlag
            ObsXMLParameterList['DBParamList'] = numpy.asarray(DBParamList)
            ObsXMLParameterList['BackgroundCube'] = LocalBackgroundCube
            ObsXMLParameterList['DustCube'] = LocalDustCube
            ObsXMLParameterList['LocalSpectralAxis'] = LocalSpectralAxis
            if (FormatObsDataFile == "fits"):
                ObsXMLParameterList['FITSHeader'] = LocalSubCube.header
                ObsXMLParameterList['FITSWCS'] = LocalSubCube.wcs
            else:
                ObsXMLParameterList['FITSHeader'] = None
                ObsXMLParameterList['FITSWCS'] = None
            ObsXMLParameterList['IsoRatioFileName'] = IsoRatioFileName
            ObsXMLParameterList['dbFileName'] = dbFileName
            if ((LocalSpectralAxis.value[1] - LocalSpectralAxis.value[0]) < 0.0):
                ReverseFlag = True
            else:
                ReverseFlag = False
            ObsXMLParameterList['ReverseFlag'] = ReverseFlag


            ## store range parameters
            RangeParam.append(ObsXMLParameterList)
            # print ("done!")

    # Debug:
    # print ("RangeParam = ", RangeParam)
    # print ("PixelMask.shape = ", PixelMask.shape)
    # numpy.savetxt(myXCLASSMapFitJobDir + "mask.txt", PixelMask)
    # print ("PixelMask: \n")
    # [print (str(PixelMask[i, :]).replace("True", "*").replace("False", ".")) for i in range(len(PixelMask))]


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## combine boolean arrays indicating which pixels are fitted or not
    PixelMask = numpy.logical_and(PixelMask, ThresholdPixelMask)

    # Debug:
    # print ("\n\nPixelMask = ", PixelMask)
    # print ("ThresholdPixelMask = ", ThresholdPixelMask)
    # print ("NumberNonNaNPixels = ", numpy.count_nonzero(PixelMask))
    # sys.exit(0)


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## get dimensions of pixel matrix
    NumYPixels = PixelMask.shape[0]
    NumXPixels = PixelMask.shape[1]
    # TotalNumPixel = NumYPixels * NumXPixels
    TotalNumPixel = numpy.count_nonzero(PixelMask)
    ListOfSelectedPixels = numpy.transpose((PixelMask).nonzero())

    # Debug:
    # print ("len(ListOfSelectedPixels) = ", len(ListOfSelectedPixels))
    # sys.exit(0)


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## define directory
    ParamDict['DataFileFormat'] = FormatObsDataFile
    ParamDict['CompRegions'] = CompRegions
    ParamDict['FITSImageMask'] = FITSImageMask
    ParamDict['FITSImageMaskValues'] = FITSImageMaskValues
    ParamDict['ExpFileList'] = ExpFileList
    ParamDict['dbFileName'] = dbFileName
    ParamDict['ListOfCubes'] = ListOfCubes
    ParamDict['ListOfFullCubes'] = ListOfFullCubes
    ParamDict['ListOfSpectralAxis'] = ListOfSpectralAxis
    ParamDict['ListOfFullSpectralAxis'] = ListOfFullSpectralAxis
    ParamDict['OrigRestFreq'] = OrigRestFreq
    ParamDict['OrigUnits'] = OrigUnits
    ParamDict['RangeParam'] = RangeParam
    ParamDict['PixelMask'] = PixelMask
    ParamDict['ParameterMapHeader'] = ParameterMapHeader
    ParamDict['ParameterMapWCS'] = ParameterMapWCS
    ParamDict['NumYPixels'] = NumYPixels
    ParamDict['NumXPixels'] = NumXPixels
    ParamDict['TotalNumPixel'] = TotalNumPixel
    ParamDict['ListOfSelectedPixels'] = ListOfSelectedPixels
    ParamDict['DECAxis'] = DECAxis
    ParamDict['RAAxis'] = RAAxis


    ## we're done
    return ParamDict
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## local version of the spectral_cube function "subcube_from_regions"
##
def local_subcube_from_regions(cube, region_list, minimize = False):
    """
    Extract a masked subcube from a list of ``regions.Region`` object
    (only functions on celestial dimensions)

    Parameters
    ----------
    cube:   spectral_cube object
    region_list: ``regions.Region`` list
        The region(s) to extract
    minimize : bool
        Run :meth:`~SpectralCube.minimal_subcube`.  This is mostly redundant, since the
        bounding box of the region is already used, but it will sometimes
        slice off a one-pixel rind depending on the details of the region
        shape.  If minimize is disabled, there will potentially be a ring
        of NaN values around the outside.
    """

    # Debug:
    # print ("\ncube = ", cube)
    # print ("region_list = ", region_list)
    # print ("minimize = ", minimize)


    ## import regions package
    import regions


    ## convert every region to a `regions.PixelRegion` object.
    regs = []
    for x in region_list:
        if isinstance(x, regions.SkyRegion):
            regs.append(x.to_pixel(cube.wcs.celestial))
        elif isinstance(x, regions.PixelRegion):
            regs.append(x)
        else:
            raise TypeError("'{}' should be `regions.Region` object".format(x))


    ## list of regions are converted to a `regions.CompoundPixelRegion` object.
    compound_region = local_regionlist_to_single_region(regs)


    ## compound mask of all the regions.
    mask = compound_region.to_mask()
    xlo, xhi, ylo, yhi = mask.bbox.ixmin, mask.bbox.ixmax, mask.bbox.iymin, mask.bbox.iymax


    # Negative indices will do bad things, like wrap around the cube
    # If xhi/yhi are negative, there is not overlap
    if (xhi < 0) or (yhi < 0):
        raise ValueError("Region is outside of cube.")
    if xlo < 0:
        xlo = 0
    if ylo < 0:
        ylo = 0

    # Debug:
    # print ("xlo = ", xlo)
    # print ("ylo = ", ylo)
    # print ("xhi = ", xhi)
    # print ("yhi = ", yhi)
    # print ("cube = ", cube)


    # If None, then the whole spectral range of the cube is selected.
    subcube = cube.subcube(xlo = xlo, ylo = ylo, xhi = xhi, yhi = yhi)

    # Debug:
    # print ("subcube = ", subcube)


    shp = cube.shape[1:]
    _, slices_small = mask.get_overlap_slices(shp)

    maskarray = numpy.zeros(subcube.shape[1:], dtype = 'bool')

    # Debug:
    # print ("\n\nmaskarray = ", maskarray)
    # print ("mask = ", mask)
    # print ("slices_small = ", slices_small)


    maskarray[:] = mask.data[slices_small]


    ## create boolean mask
    BAM = BooleanArrayMask(maskarray, subcube.wcs, shape = subcube.shape)
    masked_subcube = subcube.with_mask(BAM)


    # by using ceil / floor above, we potentially introduced a NaN buffer
    # that we can now crop out
    if minimize:
        return masked_subcube.minimal_subcube(spatial_only=True)
    else:
        return masked_subcube
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## recursively merge a region list into a single compound region, required by function "local_subcube_from_regions"
##
def local_regionlist_to_single_region(region_list):

    import regions
    import operator

    if len(region_list) == 1:
        return region_list[0]
    left = local_regionlist_to_single_region(region_list[:len(region_list)//2])
    right = local_regionlist_to_single_region(region_list[len(region_list)//2:])
    return regions.CompoundPixelRegion(left, right, operator.or_)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## extracts a subcube from a cube using a masking cube
##
def SubCubeFromCube(LocalCube, MaskCube):
    """

input parameters:
-----------------

    - LocalCube:                        unmodified cube

    - MaskCube:                         masking cube



output parameters:
------------------

    - LocalCube:                       new subcube
    """

    # Debug:
    # print ("\nLocalCube = ", LocalCube)
    # print ("MaskCube = ", MaskCube)


    ## get world coordinates
    f, dec, ra = MaskCube.world[:]


    ## get limits of ra axis
    RAAxis = ra[0, 0, :]
    xlo = numpy.nanmin(RAAxis.value) * RAAxis.unit
    xhi = numpy.nanmax(RAAxis.value) * RAAxis.unit


    ## get limits of dec axis
    DECAxis = dec[0, :, 0]
    ylo = numpy.nanmin(DECAxis.value) * DECAxis.unit
    yhi = numpy.nanmax(DECAxis.value) * DECAxis.unit


    ## get limits of spectral axis
    LocalFrequencyAxis = MaskCube.spectral_axis.value
    LocalFrequencyAxisUnit = MaskCube.spectral_axis.unit
    LocalMapFreqMin = numpy.nanmin(LocalFrequencyAxis)
    LocalMapFreqMax = numpy.nanmax(LocalFrequencyAxis)
    zlo = LocalMapFreqMin * LocalFrequencyAxisUnit
    zhi = LocalMapFreqMax * LocalFrequencyAxisUnit

    # Debug:
    # print ("\nxlo = ", xlo)
    # print ("xhi = ", xhi)
    # print ("ylo = ", ylo)
    # print ("yhi = ", yhi)
    # print ("zlo = ", zlo)
    # print ("zhi = ", zhi)


    ## shrink cube to masking FITS image
    LocalCube = LocalCube.subcube(xlo = xlo, xhi = xhi,
                                  ylo = ylo, yhi = yhi,
                                  zlo = zlo, zhi = zhi)

    # Debug:
    # print ("LocalCube = ", LocalCube)


    ## we're done
    return LocalCube
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## extracts a subimage from an image using ds9 region and return an Spectral_Cube object
##
def Subimage_from_regions(FITSFileName, CompRegions):
    """

input parameters:
-----------------

    - FITSFileName:                     path and name of FITS image

    - CompRegions:                      combined ds9 region



output parameters:
------------------

    - LocalImage:                       reduced FITS image object
    """

    # Debug:
    # print (FITSFileName = ", FITSFileName)
    # print (CompRegions = ", CompRegions)


    ## import FITS image
    LocalImage = fits.getdata(FITSFileName, ext = 0)


    ## in order to apply SpectralCube function to select region, convert FITS image to cube
    NumX = LocalImage.shape[0]
    NumY = LocalImage.shape[1]
    ImageCube = numpy.ones((1, NumX, NumY))
    ImageCube[0, :, :] = LocalImage[:, :]
    LocalImageFile = fits.open(FITSFileName)
    ImageCubeHeader = LocalImageFile[0].header
    ImageCubeHeader['NAXIS'] = 1
    ImageCubeHeader['NAXIS3'] = '1'
    ImageCubeHeader['CRPIX3'] = '1'
    ImageCubeHeader['CRVAL3'] = '1'
    ImageCubeHeader['CDELT3'] = '1'
    ImageCubeHeader['CTYPE3'] = 'FREQ'
    ImageCubeWCS =  WCS(ImageCubeHeader)
    ImageCubeSC = SpectralCube(data = ImageCube, wcs = ImageCubeWCS)
    if (CompRegions is not None):
        ImageCubeSC = ImageCubeSC.subcube_from_regions(CompRegions)
    LocalImage = ImageCubeSC[0, :, :]

    # Debug:
    # print ("LocalImage = ", LocalImage)


    ## create header for FITS image from masking cube
    # m1 = MaskCube[:1, :, :].moment(order = 1)
    # ParameterMapWCS = copy.deepcopy(m1.wcs)
    # MasImageObject = Slice(value = MaskCube[0, :, :].value, wcs = ParameterMapWCS)


    # ## use spectral cube function "reproject" to reproject cube spatially
    # MaskingCubeHeader = MasImageObject.header
    # LocalImage = LocalImage.reproject(MaskingCubeHeader, order = 'bilinear')


    ## we're done
    return LocalImage
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## import fits file
def FITSImport(LocalFITSFileName, mask = None, MaskCubeFlag = False, daskFlag = False):
    """

input parameters:
-----------------

    - LocalFITSFileName:        path and name of FITS file

    - mask:                     (optional) path and name of FITS cube used for masking (default: None)

    - MaskCubeFlag:             (optional) flag indicating that mask describes a cube instead of the file name

    - daskFlag:                 (optional) flag indicate using of dask (default: False)


output parameters:
------------------

    - CubeDict:                 dictionary conataining parameters
    """

    # Debug:
    # print ("LocalFITSFileName = ", LocalFITSFileName)
    # print ("mask = ", mask)
    # print ("MaskCubeFlag = ", MaskCubeFlag)
    # print ("daskFlag = ", daskFlag)


    ## initialize return parameter
    CubeDict = {}


    ## check, if selected file is a CASA FITS file
    LocalObsDataFileNameLower = LocalFITSFileName.lower()
    if (LocalObsDataFileNameLower.endswith(".image") or LocalObsDataFileNameLower.endswith(".tt0") or LocalObsDataFileNameLower.endswith(".psf")):
        CASACubeFlag = True
    else:
        CASACubeFlag = False


    ## import FITS file
    LocalFITSImageFlag = False
    ImportErrorFlag = False
    if (CASACubeFlag):
        try:
            LocalCube = SpectralCube.read(LocalFITSFileName, format = 'casa_image', use_dask = daskFlag)
            LocalFITSImageFlag = False
        except:
            try:
                LocalImage = fits.getdata(LocalFITSFileName, ext = 0)
                LocalFITSImageFlag = True
            except:
                ImportErrorFlag = True
    else:
        try:
            LocalCube = SpectralCube.read(LocalFITSFileName, use_dask = daskFlag)
            LocalFITSImageFlag = False
        except:
            try:
                LocalImage = fits.getdata(LocalFITSFileName, ext = 0)
                LocalFITSImageFlag = True
            except:
                ImportErrorFlag = True

    # Debug:
    # print ("LocalFITSFileName = ", LocalFITSFileName)
    # print ("LocalFITSImageFlag = ", LocalFITSImageFlag)
    # print ("ImportErrorFlag = ", ImportErrorFlag)


    ## if file can not be imported raise error
    if (ImportErrorFlag):
        print ("\n\nCan not import selected FITS file {:s}!\n\n".format(LocalFITSFileName))
        return CubeDict


    ## import mask
    if (mask is not None):
        if (MaskCubeFlag):
            MaskCube = mask
        else:
            MaskCube = SpectralCube.read(mask, use_dask = daskFlag)
        MaskSpectralAxis = MaskCube.spectral_axis


        ## get world coordinates
        velo, dec, ra = MaskCube.world[:]
        ra = ra.to(u.deg)
        dec = dec.to(u.deg)
        # MaskDecAxis = dec[0, :, 0].value
        # MaskRAAxis = ra[0, 0, :].value

        # Debug:
        # print ("\nMaskSpectralAxis = ", MaskSpectralAxis)
        # print ("MaskDecAxis = ", MaskDecAxis)
        # print ("MaskRAAxis = ", MaskRAAxis)


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## analyze FITS image
    if (LocalFITSImageFlag):


        ## hack to get the FITS image as a Spectral Cube object
        LocalImage = fits.getdata(LocalFITSFileName, ext = 0)
        LocalImageFile = fits.open(LocalFITSFileName)
        ImageCubeHeader = LocalImageFile[0].header
        # try:
        #     del ImageCubeHeader['CROTA3']
        # except:
        #     pass
        # try:
        #     del ImageCubeHeader['CROTA4']
        # except:
        #     pass
        ImageCubeWCS = WCS(ImageCubeHeader)
        LocalImage = Slice(value = LocalImage, wcs = ImageCubeWCS)

        # Debug:
        # print ("LocalFITSFileName = ", LocalFITSFileName)
        # print ("ImageCubeHeader = ", ImageCubeHeader)
        # print ("ImageCubeWCS = ", ImageCubeWCS)


        ##********************************************************************************************************************************************
        ## apply mask and interpolate FITS image to make sure, that new FITS image is defined exactly at the coordinates of the masking cube/image
        if (mask is not None):


            ## create header for FITS image from masking cube
            m1 = MaskCube[:1, :, :].moment(order = 1)
            ParameterMapWCS = copy.deepcopy(m1.wcs)
            MasImageObject = Slice(value = MaskCube[0, :, :].value, wcs = ParameterMapWCS)


            ## use spectral cube function "reproject" to reproject cube spatially
            MaskingCubeHeader = MasImageObject.header
            LocalImage = LocalImage.reproject(MaskingCubeHeader, order = 'bilinear')

        # Debug:
        # print ("LocalImage.shape = ", LocalImage.shape)
        # print ("\n\ndec = ", dec)
        # print ("ra = ", ra)


        ## get world coordinats and define parameters for RA and DEC axis
        dec, ra = LocalImage.world[:, :]
        UnitCube = LocalImage.unit
        LocalDecAxis = dec[:, 0].value
        LocalDecAxisUNIT = dec[:, 0].unit
        LocalRAAxis = ra[0, :].value
        LocalRAAxisUNIT = ra[0, :].unit

        # Debug:
        # print ("\n\ndec = ", dec)
        # print ("ra = ", ra)
        # print ("LocalDecAxis = ", LocalDecAxis)
        # print ("LocalDecAxisUNIT = ", LocalDecAxisUNIT)
        # print ("LocalRAAxis = ", LocalRAAxis)
        # print ("LocalRAAxisUNIT = ", LocalRAAxisUNIT)


        ## store cube
        CubeDict['Image'] = LocalImage


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## analye FITS cube
    else:


        ## allow huge operations
        LocalCube.allow_huge_operations = True

        # Debug:
        # print ("LocalCube = ", LocalCube)
        # print ("LocalCube.shape = ", LocalCube.shape)
        # print ("LocalCube.shape[0] = ", LocalCube.shape[0])
        # print ("LocalCube.spectral_axis = ", LocalCube.spectral_axis)
        # print ("MaskCube = ", MaskCube)
        # print ("mask = ", mask)
        # LocalCube.write("orig.fits", overwrite = True)


        ##************************************************************************************************************************************************
        ## apply mask and interpolate FITS cube to make sure, that new FITS cube is defined exactly at the coordinates of the masking image
        if (mask is not None):


            ## reproject cube spectrally
            if (not numpy.array_equal(MaskCube.spectral_axis.value, LocalCube.spectral_axis.value, equal_nan = True)):
                LocalCube = LocalCube.spectral_interpolate(MaskSpectralAxis, suppress_smooth_warning = True, fill_value = 0.0)


            ## extract subcube with size of masking cube
            # LocalCube = SubCubeFromCube(LocalCube, MaskCube)


            ## TO-DO: check, if "LocalCube" and "MaskingCubeHeader" have the same dimensions
            ##        if not, make masking cube have the same dimesions
            ## spectral-cube:   (stokes?, n_spectral, n_y, n_x)

            LocalCubeHeader = LocalCube.header
            LocalCubeHeaderNAXIS = int(LocalCubeHeader['NAXIS'])
            MaskingCubeHeader = MaskCube.header
            MaskingCubeHeaderNAXIS = int(LocalCubeHeader['NAXIS'])
            # if (LocalCubeHeaderNAXIS != MaskingCubeHeaderNAXIS):
            #     if (LocalCubeHeaderNAXIS == 3 and MaskingCubeHeaderNAXIS == 4):
            #         print ()

            # Debug:
            # print ("\n\nLocalCubeHeaderNAXIS = ", LocalCubeHeaderNAXIS)
            # print ("MaskingCubeHeaderNAXIS  = ", MaskingCubeHeaderNAXIS)


            ## use spectral cube function "reproject" to reproject cube spatially
            ## The error message
            ## AttributeError: SpectralCoord instance has no attribute 'in_observer_velocity_frame'
            ## is fixed in astropy version 4.3 or later
            try:
                print ("\n\tReproject cube .. ", end = "", flush = True)
                LocalCube = LocalCube.reproject(MaskingCubeHeader, order = 'bilinear', use_memmap = False, filled = False)
                print ("done!")
            except:
                print ("\n\n\nError in XCLASS package, function task_myXCLASSMapFit.FITSImport:")
                print ("\tCan not reproject FITS cube!")
                print ("\n\tMaybe the coordinates of the selected FITS cube are not quite correct!\n\n")
                # print ("\tLocalCube = ", LocalCube)
                # print ("\tMaskCube = ", MaskCube)


            ## hack to avoid error message
            ## AttributeError: Can't pickle local object 'FITSWCSAPIMixin._get_components_and_classes.<locals>.value_from_spectralcoord'
            TempFileName = "temp.fits"
            LocalCube.write(TempFileName)
            LocalCube = SpectralCube.read(TempFileName, use_dask = daskFlag)
            cmdString = "rm -rf " + TempFileName
            os.system(cmdString)

            # Debug:
            # LocalCube.write("cube.fits", overwrite = True)
            # MaskCube.write("mask.fits", overwrite = True)
            # sys.exit(0)


        ## store cube
        CubeDict['Cube'] = LocalCube


        ## deactivate silencing warnings
        warnings.filterwarnings(action = 'ignore', category = SpectralCubeWarning, append = True)
        warnings.filterwarnings(action = 'ignore', category = WCSMismatchWarning, append = True)
        warnings.filterwarnings(action = 'ignore', category = SliceWarning, append = True)
        warnings.filterwarnings(action = 'ignore', category = StokesWarning, append = True)
        warnings.simplefilter('ignore')


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define parameters for freq. axis


        ## get world coordinats
        velo, dec, ra = LocalCube.world[:]

        # Debug:
        # print ("LocalCube = ", LocalCube)
        # print ("LocalCube.spectral_axis = ", LocalCube.spectral_axis)


        ## get lowest and highest frequency point
        UnitCube = LocalCube.unit
        LocalFrequencyAxis = LocalCube.spectral_axis.value
        LocalFrequencyAxisUnit = LocalCube.spectral_axis.unit
        LocalMapFreqMin = numpy.nanmin(LocalFrequencyAxis)
        LocalMapFreqMax = numpy.nanmax(LocalFrequencyAxis)
        try:
            LocalMapFreqStep = LocalFrequencyAxis[1] - LocalFrequencyAxis[0]
        except:
            LocalMapFreqStep = None

        # Debug:
        # print ("LocalFrequencyAxis = ", LocalFrequencyAxis)
        # print ("LocalMapFreqMin = ", LocalMapFreqMin)
        # print ("LocalMapFreqMax = ", LocalMapFreqMax)
        # print ("LocalMapFreqStep = ", LocalMapFreqStep)


        ## store frequency parameters
        CubeDict['FrequencyAxis'] = LocalFrequencyAxis
        CubeDict['FrequencyAxisUnit'] = LocalFrequencyAxisUnit
        CubeDict['MapFreqMin'] = LocalMapFreqMin
        CubeDict['MapFreqMax'] = LocalMapFreqMax
        CubeDict['MapFreqStep'] = LocalMapFreqStep


        ## define parameters for RA axis
        LocalDecAxis = dec[0, :, 0].value
        LocalDecAxisUNIT = dec[0, :, 0].unit
        LocalRAAxis = ra[0, 0, :].value
        LocalRAAxisUNIT = ra[0, 0, :].unit

        # Debug:
        # print ("\n\nLocalDecAxis = ", LocalDecAxis)
        # print ("LocalDecAxisUNIT = ", LocalDecAxisUNIT)
        # print ("LocalRAAxis = ", LocalRAAxis)
        # print ("LocalRAAxisUNIT = ", LocalRAAxisUNIT)


    ## define parameters for Dec axis
    try:
        LocalDecAxisDelta = LocalDecAxis[1] - LocalDecAxis[0]
    except:
        LocalDecAxisDelta = None
    LocalyPixelID = int(len(LocalDecAxis) / 2.0)

    # Debug:
    # print ("LocalDecAxisDelta = ", LocalDecAxisDelta)
    # print ("LocalyPixelID = ", LocalyPixelID)


    ## define parameters for RA axis
    try:
        LocalRAAxisDelta = LocalRAAxis[1] - LocalRAAxis[0]
    except:
        LocalRAAxisDelta = None
    LocalxPixelID = int(len(LocalRAAxis) / 2.0)

    # Debug:
    # print ("LocalRAAxisDelta = ", LocalRAAxisDelta)
    # print ("LocalxPixelID = ", LocalxPixelID)


    ## store frequency parameters
    CubeDict['UnitCube'] = UnitCube
    CubeDict['DecAxis'] = LocalDecAxis
    CubeDict['DecAxisUNIT'] = LocalDecAxisUNIT
    CubeDict['RAAxis'] = LocalRAAxis
    CubeDict['RAAxisUNIT'] = LocalRAAxisUNIT
    CubeDict['DecAxisDelta'] = LocalDecAxisDelta
    CubeDict['RAAxisDelta'] = LocalRAAxisDelta
    CubeDict['yPixelID'] = LocalyPixelID
    CubeDict['xPixelID'] = LocalxPixelID
    CubeDict['FITSImageFlag'] = LocalFITSImageFlag


    ## we're done
    return CubeDict
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## import additional cubes, e.g. cubes describing the background spectum. If FITS images are provided instead of FITS cubes, convert images to cubes
##
def GetHelperCubes(AddFITSFileName, MaskingCube, PixelMask, NameOfParameter, daskFlag = False):
    """

input parameters:
-----------------

    - AddFITSFileName:      path and name of additional FITS file

    - MaskingCube:          cube used for maskinig

    - PixelMask:            mask describing pixels which are fitted

    - NameOfParameter:      name of parameter, which is described by the additional FITS cube

    - daskFlag:             (optional) flag indicating usage of dask


output parameters:
------------------

    - LocalAddFITSCube:     add cube (reduced to selected region(s), pixel(s), and to current frequency range)
    """

    # Debug:
    # print ("AddFITSFileName = ", AddFITSFileName)
    # print ("MaskingCube = ", MaskingCube)
    # print ("PixelMask = ", PixelMask)
    # print ("NameOfParameter = ", NameOfParameter)
    # print ("daskFlag = ", daskFlag)


    ## initialize return parameter
    LocalAddFITSCube = None


    ## check, if file name is defined
    if (AddFITSFileName is None):
        return LocalAddFITSCube


    ## get header and wcs from masking cube
    LocalAddFITSCubeHeader = MaskingCube.header
    LocalAddFITSCubeHeaderCopy = LocalAddFITSCubeHeader.copy()
    LocalAddFITSCubeWCS = MaskingCube.wcs


    ## import obs data cube
    CubeDict = FITSImport(AddFITSFileName, mask = MaskingCube, MaskCubeFlag = True, daskFlag = daskFlag)
    LocalFITSImageFlag = CubeDict['FITSImageFlag']


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## if additional FITS file name describes an image, convert image to cube
    if (LocalFITSImageFlag):


        ## get image
        LocalFITSImage = CubeDict['Image']

        # Debug:
        # print ("LocalFITSImage = ", LocalFITSImage)


        ## convert FITS image to FITS cube


        ## using MaskingCube.world[:]
        # velo, dec, ra = MaskingCube.world[:]
        # ra = ra.to(u.deg)
        # dec = dec.to(u.deg)
        # MaskDecAxis = dec[0, :, 0].value
        # MaskRAAxis = ra[0, 0, :].value
        # MaskSpectralAxis = MaskingCube.spectral_axis
        # NumSpecChannels = len(MaskSpectralAxis.value)
        # NewCubeArray = numpy.ones((NumSpecChannels, len(MaskDecAxis), len(MaskRAAxis)))
        # for FreqID in range(NumSpecChannels):
        #     NewCubeArray[FreqID, :, :] = LocalFITSImage[:, :]


        ## using faster method ??
        NumRAAxis = MaskingCube.shape[2]
        NumDecAxis = MaskingCube.shape[1]
        NumSpecChannels = MaskingCube.shape[0]
        NewCubeArray = numpy.zeros((NumSpecChannels, NumDecAxis, NumRAAxis))
        for FreqID in range(NumSpecChannels):
            NewCubeArray[FreqID, :, :] = LocalFITSImage[:, :]


        ## creating a SpectralCube instance
        LocalAddFITSCube = SpectralCube(data = NewCubeArray, wcs = LocalAddFITSCubeWCS, header = LocalAddFITSCubeHeaderCopy, use_dask = daskFlag)

        # Debug:
        # print ("\n\nLocalAddFITSCube = ", LocalAddFITSCube)
        # print ("MaskingCube = ", MaskingCube)


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## continue here, if additional FITS file name describes a cube
    else:


        ## get cube
        LocalAddFITSCube = CubeDict['Cube']

        # Debug:
        # print ("\n\nLocalAddFITSCube = ", LocalAddFITSCube)
        # print ("MaskingCube = ", MaskingCube)
        # print ("MaskSpectralAxis = ", MaskSpectralAxis)
        # print ("LocalAddFITSCube.spectral_axis = ", LocalAddFITSCube.spectral_axis)


    ## convert units
    LocalAddFITSCube.allow_huge_operations = True
    if (LocalAddFITSCube.unit == ""):
        if (NameOfParameter in ["background spectra"]):
            LocalAddFITSCube = LocalAddFITSCube * u.K
        elif (NameOfParameter in ["dust opacity"]):
            LocalAddFITSCube = LocalAddFITSCube * u.Unit("")

    # Debug:
    # print ("LocalAddFITSCube = ", LocalAddFITSCube)


    ## we're done
    return LocalAddFITSCube
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## estimate continuum level for all pixels of given FITS cube
##
def EstimateContinuumCube(LocalCube, NumParts = 20, NumberCores = None, RefFreq = None):
    """

input parameters:
-----------------

    - LocalCube:            FITS cube

    - NumParts:             number of parts used to subdivide the selected data set

    - NumberCores:          (optional) number of cores (default: 1)
                            (used for later use)

    - RefFreq:              (optional) lowest frequency for T_Back (default: None)


output parameters:
------------------

    - TBackImge:            FITS image describing background temperature for each pixel

    - TSlopeImage:          FITS image describing the corresponding temperature slope for each pixel
    """

    # Debug:
    # print ("LocalCube = ", LocalCube)
    # print ("NumParts = ", NumParts)
    # print ("NumberCores = ", NumberCores)
    # print ("RefFreq = ", RefFreq)


    ## initialize return parameter
    TBackImage = None
    TSlopeImage = None


    ## convert unit to Kelvin
    if (not LocalCube.unit in ["K"]):
        LocalCube = LocalCube.to(u.K)


    ## apply functino from LineID to whole cube
    OutputMatrix = LocalCube.apply_function_parallel_spectral(EstimateContinuumPixel, NumParts = NumParts, \
                                                              SpectralAxis = LocalCube.spectral_axis.value, \
                                                              RefFreq = RefFreq, \
                                                              num_cores = NumberCores)

    ## get parameters from output matrix
    TBackImage = OutputMatrix[0, :, :].value
    TSlopeImage = OutputMatrix[1, :, :].value

    # Debug:
    # print ("TBackImage[:, :] = ", TBackImage[:, :])
    # print ("TSlopeImage[:, :] = ", TSlopeImage[:, :])


    ## we're done
    return TBackImage, TSlopeImage
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## continuum function
##
def ContinuumFunction(nu, Tbg, TSlope):
    """

input parameters:
-----------------

    - nu:                   renormalized frequency array (=\frac{\nu}{\nu_min})

    - Tbg:                  background temperature

    - TSlope:               temperature slope


output parameters:
------------------

    - None
    """

    # Debug:
    # print "nu = ", nu
    # print "Tbg = ", Tbg
    # print "TSlope = ", TSlope


    ## return to main program
    return Tbg * (nu)**TSlope
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## estimate continuum for each pixel
def EstimateContinuumPixel(DataArray, Method = "statcont", NumParts = 20, SpectralAxis = None, RefFreq = None):
    """

input parameters:
-----------------

    - DataArray:        data array

    - Method:           (optional) method used to estimate continuum, (default: "statcont")

    - NumParts:         (optional) number of subclasses, (default: 20)

    - SpectralAxis:     (optional) spectral axis (default: None)

    - RefFreq:          (optional) reference frequency for T_Back, (default: None)


output parameters:
------------------

    - NewTBack:         new background temperature

    - NewTSlope:        new temperature slope
    """

    # Debug:
    # print ("\nDataArray = ", DataArray)
    # print ("numpy.shape(DataArray) = ", numpy.shape(DataArray))
    # print ("Method = ", Method)
    # print ("NumParts = ", NumParts)
    # print ("SpectralAxis = ", SpectralAxis)
    # print ("RefFreq = ", RefFreq)


    ## initialize return parameter
    NewTBack = None
    NewTSlope = None


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## use statcont to remove continuum
    if (Method in ["statcont"]):


        ## try to import statcont
        STATCONTAvailableFlag = True
        try:
            from xclass.external.statcont import cont_finding
        except:
            STATCONTAvailableFlag = False

        # Debug:
        # print ("STATCONTAvailableFlag = ", STATCONTAvailableFlag)


        ## remove continuum using statcont
        if (STATCONTAvailableFlag):


            ## compute rms
            rms = task_LineIdentification.ComputeRMS(DataArray[:])

            # Debug:
            # print ("rms = ", rms)


            ## split spectrum into smaller parts
            l = len(DataArray)
            q = float(l) / float(NumParts)
            if (q == float(int(q))):
                SplittedLocalData = numpy.array_split(DataArray, NumParts)
            else:
                ll = int(q) * NumParts
                SplittedLocalData = numpy.array_split(DataArray[:ll], NumParts)

            # Debug:
            # print ("l = ", l)
            # print ("NumParts = ", NumParts)
            # print ("q = ", q)


            ## apply statcont to each part
            nuList = []
            contLevelList = []
            for LocalPartData in SplittedLocalData:


                ## write the intensity of a given pixel for all the channels into the array flux and the frequencies in the array freqs
                rms_noise = rms
                freq_axis = 0
                freq = SpectralAxis
                if (len(freq) > 0):
                    MinFreq = numpy.nanmin(freq)
                    MaxFreq = numpy.nanmax(freq)
                    CenterFreq = MinFreq + ((MaxFreq - MinFreq) / 2.0)
                    flux = LocalPartData[:]
                    nuList.append(CenterFreq)


                    ## call sigma clipping subroutine
                    # sigmaclip_flux_prev, sigmaclip_flux, sigmaclip_noise, filtered_data = statcont.cont_finding.c_sigmaclip(flux, rms_noise, \
                    #                                                                                                         freq_axis)
                    sigmaclip_flux_prev, sigmaclip_flux, sigmaclip_noise, filtered_data = cont_finding.c_sigmaclip(flux, rms_noise, freq_axis)
                    contLevelList.append(sigmaclip_flux)


            ## fit background function to list of continuum levels
            contLevelList = numpy.asarray(contLevelList)
            nuList = numpy.asarray(nuList)
            nuMin = numpy.nanmin(nuList)
            Tbg = DataArray[0]
            TSlope = 0.0
            p0 = [Tbg, TSlope]
            p0 = numpy.asarray(p0)
            p0Bounds = ((0.0, -100.0), (1.e4, 100.0))
            try:
                popt, _ = curve_fit(ContinuumFunction, nuList / nuMin, contLevelList, p0 = p0, bounds = p0Bounds)
            except:
                popt = [0.0, 0.0]

            # Debug:
            # print ("popt = ", popt)


            ## define array of fit function
            if (popt is not None):
                NewTBack = popt[0]
                NewTSlope = popt[1]
                if (RefFreq is None):
                    RefFreq = numpy.nanmin(SpectralAxis)


                ## compute background temperature for new lower frequency
                NewTBack = NewTBack * (RefFreq / nuMin)**NewTSlope

    # Debug:
    # print ("NewTBack = ", NewTBack)
    # print ("NewTSlope = ", NewTSlope)


    ## prepare output array
    OutputArray = DataArray * 0.0
    OutputArray[0] = NewTBack
    OutputArray[1] = NewTSlope


    ## we're done
    return OutputArray
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## converts Jansky/Beam in Kelvin and vise versa for elliptical Gaussian beams
##
def JanskyPerBeamInKelvin(diameter, BMAJ, BMIN, InterFlag, frequency):
    """

input parameters:
-----------------

    - diameter:             telescope diameter (in m for InterFlag = False, in arcesc for InterFlag = True)

    - BMAJ:                 beam major axis length (in arcsec)

    - BMIN:                 beam minor axis length (in arcsec)

    - InterFlag:            flag indicating single dish (False) or interferometric (True) data

    - frequency:            frequency (in MHz)


output parameters:
------------------

    - ConvertJyPerBeam2Kelvin:     Jansky/Beam to Kelvin prefactor

    - ConvertKelvin2JyPerBeam:     Kelvin to Jansky/Beam prefactor
    """

    # Debug:
    # print ("\n\ndiameter = ", diameter)
    # print ("BMAJ = ", BMAJ)
    # print ("BMIN = ", BMIN)
    # print ("InterFlag = ", InterFlag)
    # print ("frequency = ", frequency)


    ## define some constants
    c = 2.99792458e8                                                                        ## speed of light (in m/s)
    k = 1.3806488e-23                                                                       ## Boltzmann constan (in J/K)
    wavelength = c / (frequency * 1.e6)                                                     ## determine wavelength (in m)


    ## calculate beam size in radian
    InterFlag = CheckBool(InterFlag)
    if (InterFlag):
        if (BMAJ is not None and BMIN is not None):                                         ## elliptical, two-dimensional Gaussian beam
            beam_radian_min = (BMIN / 3600.0) * (numpy.pi / 180.0)                          ## convert arcsec to degree to radians
            beam_radian_maj = (BMAJ / 3600.0) * (numpy.pi / 180.0)                          ## convert arcsec to degree to radians
        else:                                                                               ## circle shaped two-dimensional Gaussian beam
            beam_radian_min = (diameter / 3600.0) * (numpy.pi / 180.0)                      ## convert arcsec to degree to radians
            beam_radian_maj = beam_radian_min
    else:
        beam_radian_min = 1.22 * wavelength / diameter                                      ## determine telescope beam FWHM size
                                                                                            ## using diffraction limit (in radian)
        beam_radian_maj = beam_radian_min


    ## calculate solid angle of the beam
    Omega = beam_radian_min * beam_radian_maj * numpy.pi / (4.0 * numpy.log(2.0))


    ## calculate conversion factors
    ## Dividing the flux F (in Jy) by Omega gives the flux density or intensity, which is converted to brightness temperature by applying the
    ## Rayleigh-Jeans approximation of the Planck function
    ConvertJyPerBeam2Kelvin = (wavelength**2 / (2.0 * k)) * 1.0 / Omega * 1.e-26            ## convert Jansky/Beam to Kelvin prefactor
                                                                                            ## factor 1.e-26 caused by 1 Jy = 1.e-26 J / (sec * Hz * m^2)
    ConvertKelvin2JyPerBeam = 1.0 / ConvertJyPerBeam2Kelvin                                 ## convert Kelvin to Jansky/Beam prefactor

    # Debug:
    # print ("ConvertJyPerBeam2Kelvin = {:12.6f}".format(ConvertJyPerBeam2Kelvin))
    # print ("ConvertKelvin2JyPerBeam = {:12.6f}".format(ConvertKelvin2JyPerBeam))


    ## define return variable
    return (ConvertJyPerBeam2Kelvin, ConvertKelvin2JyPerBeam)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert cube to Kelvin
##
def ConvertToKelvin(LocalPath, LocalCube):
    """

input parameters:
-----------------

    - LocalPath:                current working directory

        - LocalCube:                cube to convert


output parameters:
------------------

    - LocalCubeConverted:       cube with converted intensity
    """

    # Debug:
    # print ("LocalCube = ", LocalCube)


    ## print what you do
    print ("\t\tConvert intensities to Kelvin .. ", end = "", flush = True)


    ## get size of cube
    SizeLocalCube = 1
    for i in range(len(LocalCube.shape)):
        k *= LocalCube.shape[i]
    SizeLocalCube *= 32                                                                     ## assume 32 bit represenation
    SizeLocalCube *= 1e-9                                                                   ## convert byte to GB

    # Debug:
    # print ("SizeLocalCube = ", SizeLocalCube)


    ## for small cubes (<= 5GB), use the astropy method
    LocalCubeConverted = None
    if (SizeLocalCube <= 5.0):
        LocalCubeConverted = LocalCube.to(u.K)


    ## use the following method for very large cubes (> 5GB)
    ## taken from https://spectral-cube.readthedocs.io/en/latest/big_data.html
    else:
        # shutil.copy('file.fits', 'newfile.fits')
        LocalTempFileName = "newfile.fits"
        outfh = fits.open(LocalTempFileName, mode = 'update')
        factors = LocalCube.jtok_factors()
        for index,(slice,factor) in enumerate(zip(LocalCube,factors)):
            outfh[0].data[index] = slice * factor
            outfh.flush()                                                                   ## write the data to disk
        outfh[0].header['BUNIT'] = 'K'
        outfh.flush()


        ## read in the converted cube
        if os.path.isfile(LocalTempFileName):
            LocalCubeConverted = SpectralCube.read(LocalTempFileName)
            os.remove(LocalTempFileName)                                                    ## remove temp-file


    ## we're done
    print ("done!")


    ## define return variable
    return LocalCubeConverted
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## reads in file that contains cluster definition
##
def ReadClusterdefFile(clusterdefFileName, CurrentJobDir):
    """

input parameters:
-----------------

    - clusterdefFileName:   path and name of the clusterdef file

    - CurrentJobDir:        current job directory


output parameters:
------------------

    - TotalNumberCores:     total number of cores of all machines defined in the clusterdef file

    - ServerList:           list of server parameters

    - ClusterFlag:          flag indicating a cluster or smp calculation
    """

    # Debug:
    # print ("clusterdefFileName = ", clusterdefFileName, os.path.exists(clusterdefFileName))
    # print ("os.path.exists(clusterdefFileName) = ", os.path.exists(clusterdefFileName))
    # print ("CurrentJobDir = ", CurrentJobDir)


    ## initialize output variables
    ClusterFlag = False
    TotalNumberCores = 1
    Server = "localhost"
    NumCores = 1
    ServerPath = ""
    ServerList = [[Server, NumCores, ServerPath]]


    ## continue here, if a clusterdef file is defined
    if (clusterdefFileName is not None and clusterdefFileName != "" and os.path.exists(clusterdefFileName)):
        print ("Analyze cluster file ..", end = ' ')
        ErrorFlag = False


        ## readin clusterdef file
        clusterdefFile = open(clusterdefFileName)
        clusterdefFileContents = clusterdefFile.readlines()
        clusterdefFile.close()


        ## analyze cluster file
        ServerList = []
        TotalNumberCores = 0
        for lines in clusterdefFileContents:                                                ## loop over all lines of clusterdef file
            pureline = lines.strip()                                                        ## remove leading and trailing blanks

            # Debug:
            # print ("pureline = ", pureline)


            ## remove comments
            w = pureline.find("#")                                                          ## are there comments in the current line ?
            if (w == 0):                                                                    ## ignore lines which contains only comments
                pureline = ""                                                               ## if line is only a comment clear line
            elif (w > 0):                                                                   ## is there a comment in the current line ?
                pureline = pureline[:w]                                                     ## remove comments
            pureline = pureline.strip()

            # Debug:
            # print ("pureline = ", pureline)


            ## if remaining line (without comments) is not empty, get server parameters
            if (pureline != ""):                                                            ## ignore empty lines


                ## split current line
                SplittedLine = pureline.split(",")

                # Debug:
                # print ("SplittedLine = ", SplittedLine)


                ## import socket package
                import socket


                ## get name of current machine (system hostname)
                try:
                    SystemHostname = socket.gethostname()
                except ImportError:
                    SystemHostname = "localhost"


                ## get server
                Server = "localhost"
                NumCores = 1
                ServerPath = ""
                i = len(SplittedLine)
                ServerErrorFlag = False
                if (i > 0):
                    Server = SplittedLine[0].strip()
                    Server = Server.replace(chr(34), "")
                    ClusterFlag = True
                    if (Server.lower() != "localhost" and Server != SystemHostname.strip()):


                        ## test, if defined server is reachable
                        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
                        try:
                            s.connect((Server, 22))
                            ClusterFlag = True
                        except socket.error as e:
                            if (not ErrorFlag):
                                print ("\n\n")
                            print ("\tWARNING:    Server " + Server + " is not reachable!")
                            ErrorFlag = True
                            ServerErrorFlag = True
                        s.close()


                    ## check, defined number of cores
                    if (i > 1):
                        NumCores = int(SplittedLine[1].strip())
                        TotalNumberCores += NumCores
                        if (i > 2):
                            ServerPath = SplittedLine[2].strip()
                if (not ServerErrorFlag):
                    ServerList.append([Server, NumCores, ServerPath])


        ## check ServerList
        if (ServerList == []):
            Server = "localhost"
            NumCores = 1
            TotalNumberCores = 1
            ServerPath = ""
            ServerList.append([Server, NumCores, ServerPath])


        ## done
        if (not ErrorFlag):
            print ("done!")

    # Debug:
    # print ("\tTotalNumberCores = ", TotalNumberCores)
    # print ("\n\n")
    # print ("\tServerList = ", ServerList)


    ## define return variables
    return (TotalNumberCores, ServerList, ClusterFlag)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## core myXCLASSMapFit function
##
def myXCLASSMapFitCore(MolfitsFileName = "", MolfitFileName = None, \
                       ObsXMLFileName = "", ObsDataFileName = None, experimentalData = None, \
                       FastFitFlag = True, FullFitFlag = False, ChannelIntegrationFlag = False, \
                       AlgorithmXMLFile = "", AlgorithmXMLFileName = "", NumberIteration = 50, NumberProcessors = 4, \
                       clusterdef = None, daskFlag = False, ParallelMethod = "process", \
                       regionFileName = "", FITSImageMaskFileName = "", Threshold = None, \
                       EstimateContinuumFlag = False, EstimateConNumParts = 20, EstimateConNumEntireCubeFlag = True, \
                       UsePreviousResults = None, DataSmoothValue = 0.0, \
                       ParameterMapDir = None, ParamMapIterations = 1, ParamSmoothMethodParam = 1.0, ParamSmoothMethod = "uniform", \
                       FreqMin = None, FreqMax = None, TelescopeSize = None, BMIN = None, BMAJ = None, BPA = None, Redshift = None, \
                       Inter_Flag = False,  \
                       t_back_flag = True, tBack = 0.0, tSlope = 0.0, \
                       nH_flag = None, N_H = 0.0, beta_dust = 0.0, kappa_1300 = 0.0, DustFileName = "", \
                       BackgroundFileName = "", \
                       ContPhenFuncID = None, ContPhenFuncParam1 = None, ContPhenFuncParam2 = None, \
                       ContPhenFuncParam3 = None, ContPhenFuncParam4 = None, ContPhenFuncParam5 = None, \
                       iso_flag = False, IsoTableFileName = "", \
                       NumModelPixelXX = 100, NumModelPixelYY = 100,
                       LocalOverlapFlag = False, NoSubBeamFlag = False, \
                       EmAbsPATH = None,  \
                       RestFreq = 0.0, vLSR = 0.0, \
                       ZipDirFlag = False,  \
                       DoParameterEstimationFlag = False, InternalParameterList = {}, \
                       JobDir = None, NameOfFunction = None):
    """

This function provides a simplified interface for MAGIX using the myXCLASS program for a
map. The function starts MAGIX using the Levenberg-Marquardt algorithm to fit
observational data with the myXCLASS program. The user has to specify the max. number of
iterations, the observational data (i.e. the path and the name of a FITS file
containing the data cube), and the path and name of the molfit file.

Please note, the FITS data cube has to be given in the World Coordinate System (WCS)
described in Greisen, E. W., and Calabretta, M. R., A & A, 395, 1061-1075, 2002,
Calabretta, M. R., and Greisen, E. W., A & A, 395, 1077-1122, 2002, and
Greisen, E. W., Calabretta, M. R., Valdes, F. G., and Allen, S. L., A & A, 446,
747-771, 2006.

For each run the myXCLASSMapFit function creates a so-called job directory located in
the run directory ("path-of-myXCLASS-CASA-Interface/run/myXCLASSMapFit/") where all
files created by the myXCLASSMapFit function are stored in. The name of this job
directory is made up of four components: The first part of the name consists of the
phrase "job_" whereas the second part describes the date (day, month, year), the third
part the time stamp (hours, minutes, seconds) of the function execution. The last part
describes a so-called job ID which is composed of the so-called PID followed by a
four digit random integer number to create a really unambiguous job number, e.g.
"path-to-myXCLASS-Interface/run/myXCLASSMapFit/job__25-07-2013__12-02-03__189644932/"

Additionally, the myXCLASSMapFit function creates a subdirectory within this job
directory called "Pixel-Fits/" which contains furhter subdirectories for each pixel,
where the name of the subdirectory is made up of the values (in degrees) for the right
ascension and for the declination, separately, e.g. "83.65498__-5.24258". All output
files, created by MAGIX for this pixel are stored in this subdirectory.

At the end of the whole fit procedure, the myXCLASSMapFit function creates FITS images
for each free parameter, where each pixel correspond to the value of the free parameter
taken from the best fit for this pixel. Additionally, the myXCLASSMapFit function
creates one FITS image, where each pixel corresponds to the chi^2 value of the best
fit for this pixel.



input parameters:
-----------------

    - MolfitsFileName:              path and name of the extended molfit file, (default: "").

    - MolfitFileName:               (optional) alias of MolfitsFileName, (default: None).

    - ObsXMLFileName:               (optional) path and name of obs. xml file
                                    (recommended procedure), (default: "")

    - ObsDataFileName:              (optional) path and name of a FITS data cube,  (default: None).
                                    (Please note, without using an obs. xml file, it is not
                                    possible to fit more than one data cube, simultaneously.)

                                    If the intensities of a data cube are given in Jy/beam, the
                                    myXCLASSMapFit function converts the intensities to Kelvin
                                    using the following expression

                                    .. math:: T_{\rm mb} (\nu) = \frac{c^2 \cdot S_\nu}
                                                                      {2 \, k_B \, \nu^2 \, \Omega}
                                                                 \cdot 10^{-26},

                                    where :math:`S_\nu` describes the intensity at a certain
                                    frequency :math:`\nu` in Jy/beam and :math:`c` and :math:`k_B`
                                    represents the speed of light and the Boltzmann constant,
                                    respectively. Additionally, :math:`\Omega` indicates the solid
                                    angle of the beam and can be expressed under the assumption that
                                    the beam can be approximated by a two-dimensional, elliptical
                                    Gaussian function :math:`G(\alpha, \beta)`

                                    .. math:: \Omega = \iint G(\alpha, \beta) \, d\alpha \, d \beta
                                                     = {\rm BMAJ} \cdot {\rm BMIN}
                                                       \cdot \frac{\pi}{4 \, \ln(2)},

                                    where BMAJ and BMIN indicate the FWHM of the beam major and
                                    minor axis length in arcsec.

                                    Note, the name of the data file has to end with ".fits"

    - experimentalData:             (optional) alias of ObsDataFileName, (default: None)

    - FastFitFlag:                  (optional) flag indicating fast fitting (default: True)

    - FullFitFlag:                  (optional) full fit flag (default: False)

    - ChannelIntegrationFlag:       (optional, for full fitting method only) flag indicating channel
                                    integration (default: False)

    - AlgorithmXMLFileName:         (optional) path and name of an algorithm xml-file defining
                                    settings for an algorithm or algorithm chain has to be given. (A
                                    relative path has to be defined relative to the current
                                    working directory!), (default: "")

    - AlgorithmXMLFile:             (optional) alias for AlgorithmXMLFileName

    - NumberIteration:              (only necessary if no algorithm xml and no cluster file
                                    is given) max. number of iterations (default: 50).

    - NumberProcessors:             (only necessary if no algorithm xml and no cluster file
                                    is given) number of cores (default: 4)

    - clusterdef:                   (optional) file that contains cluster definition, (default: None).

    - ParallelMethod:               (optional) method used for parallelization ("pool",
                                    "process", "sc", "dask"), (default: "process")

    - daskFlag:                     (optional) flag indicating usage of dask (default: False)

    - regionFileName:               (optional) path and name of a ds9 region file (default: "")

    - FITSImageMaskFileName:        (optional) path and name of FITS image used for masking (default: "")

    - Threshold:                    (optional) defines a threshold for a pixel. If the spectrum of a pixel
                                    has a intensity lower than the value defined by this parameter
                                    the pixel is not fitted (ignored), (default: None)

    - EstimateContinuumFlag:        (optional) flag indicating continuum estimation (default: False)

    - EstimateConNumParts:          (optional) number of sub parts used for continuum estimation
                                    (default: 20)

    - EstimateConNumEntireCubeFlag: (optional) flag indicating that continuum estimation is done
                                    for the entire spectral range (default: True)

    - UsePreviousResults:           (optional) defined if the molfit file (described by the paramter
                                    MolfitsFileName) is used as initial guess for all pixels (False)
                                    or if the result from a previous pixel is used.

                                    This input parameter is no longer used!

    - DataSmoothValue:              (optional) smooth factor (=0, no smoothing is performed) used for
                                    smoothing obs. data (default: 0.0).

    - ParameterMapDir:              (optional) path of subdirectory containing FITS images for
                                    different molfit parameters, (default: None)

    - ParamMapIterations:           (optional) number of iterations to smooth parameter maps,
                                    (default: 1, i.e. no smoothing)

    - ParamSmoothMethodParam:       (optional) parameter used for smoothing, (default: 1.0)

    - ParamSmoothMethod:            (optional) method used for smoothing, (default: "uniform")

    - JobDir:                       (optional) job directory, if None, new directory is created (default: None)

    - NameOfFunction:               (optional) name of calling function (default: None)

    The following parameters are needed, if parameter ObsDataFileName is used instead of
    parameter ObsXMLFileName:

    - FreqMin:                      start frequency of simulated spectrum (in MHz), (default: 0).

                                    Please note, if no start frequency is given, or if start frequency is
                                    lower or equal to the end frequency, the myXCLASSMapFit function will
                                    use the max. frequency range defined in the FITS header. Additionally,
                                    the stepsize, i.e. the difference between two frequencies is taken
                                    from the FITS header.

    - FreqMax:                      end frequency of simulated spectrum (in MHz), (default: 0).

    - vLSR:                         local standard of rest velocity (in km/s), (default: 0).

    - TelescopeSize:                for single dish observations (Inter_Flag = False): TelescopeSize
                                    describes the size of telescope (in m), (default: 1); for
                                    interferometric observations (Inter_Flag = True): TelescopeSize
                                    describes the interferometric beam FWHM size (in arcsec),
                                    (default: 1).

    - Inter_Flag:                   defines, if single dish ("False") or interferometric
                                    observations ("True") are described, (default: "False").

    - BMIN:                         (optional) Beam minor axis length (default: None)

    - BMAJ:                         (optional) Beam major axis length (default: None)

    - BPA:                          (optional) Beam position angle (default: None)

    - Redshift:                     (optional) red shift (dimensionless), (default: None)

    - t_back_flag:                  ("True"/"False"), defines, if the user defined background
                                    temperature Tbg and temperature slope Tslope given
                                    by the input parameters tBack and tslope describe the
                                    continuum contribution completely (t_back_flag = "True")
                                    or not (t_back_flag = "False") (default: "True").

    - tBack:                        background temperature (in K), (default: 0).

    - tslope:                       temperature slope (dimensionless), (default: 0).

    - N_H:                          (optional) Hydrogen column density (in cm^{-2}), (default: None).

    - beta_dust:                    (optional) beta for dust (dimensionless), (default: None).

    - kappa_1300:                   (optional) kappa (cm^2 g^{-1}), (default: None).

    - DustFileName:                 (optional) path and name of file describing the optical depth
                                    of dust (default: "").

    - BackgroundFileName:           (optional) path and name of file describing background
                                    (default: "").

    - ContPhenFuncID:               (optional) describes which phenomenological function is used
                                    to describe the continuum (default: None).

    - ContPhenFuncParam1:           (optional) first parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam2:           (optional) second parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam3:           (optional) third parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam4:           (optional) fourth parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam5:           (optional) fifth parameter for phenomenological function
                                    describing the continuum (default: None).

    - iso_flag:                     use isotopologues ("True"/"False"). If iso_flag is set to
                                    "False" the ratios are all set to 1 (default: "False").

    - IsoTableFileName:             path and name of an ASCII file including the iso ratios
                                    between certain molecules. The so-called "iso ratio file"
                                    defines the iso ratios between molecules. The ASCII file
                                    consists of three columns, where the first two columns
                                    indicates the molecules, respectively. The third column
                                    defines the ratio for both molecules. The columns are
                                    separated by blanks or tabs. So, the names of the
                                    molecules must not contain blanks.

                                    The myXCLASSFit function offers the possibility to
                                    optimize the ratios between isotopologues as well. For
                                    that purpose, the user has to add two additional columns
                                    on the right indicating the lower and the upper limit
                                    of a certain ratio, respectively. For more informations
                                    please look at the description of the iso ratio file given
                                    for the myXCLASS function.

                                    If the lower and upper limit are equal or if the lower
                                    limit is higher than the upper limit, the ratio is kept
                                    constant and is not optimized by the myXCLASSFit function.

                                    If no path and name is specified (default), the so-called
                                    iso-flag is set to "false".

                                    NOTE, if the parameter "IsoTableFileName" defines a relative
                                    path, the path has to be defined relative to the current
                                    working directory!

    - NumModelPixelXX:              (optional) used for sub-beam modeling, describes the number
                                    of pixels used in x-direction (default: None).

    - NumModelPixelYY:              (optional) used for sub-beam modeling, describes the number
                                    of pixels used in y-direction (default: None).

    - LocalOverlapFlag:             (optional) flag indicates if local-overlap description is
                                    used or not (default: None).

    - NoSubBeamFlag:                (optional) prevent sub beam description (default: False).

    - EmAbsPATH:                    (optional) path containing files describing functions
                                    emission / absorption (default: None)

    - RestFreq:                     rest frequency in MHz (default: 0). (If this parameter is
                                    set to zero, the intensity is plotted against frequency (in MHz)
                                    otherwise against velocity (in km/s).

    - ZipDirFlag:                   (optional) flag indicating if subdirectories "Pixel-Fits" and
                                    "Pixel-Plots" are zipped

    - DoParameterEstimationFlag:    (optional) do parameter estimation for each molecule in given
                                    molfit file, (default: False)

    - InternalParameterList:        (optional) list of internal parameters, (default: {})


output parameters:
------------------

    - JobDir:                       absolute path of the job directory created for the current run.



Example using a molfit file in the old format and an ASCII file containing the observational data:
-------------------------------------------------------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSMapFit/CH3OH.molfit"
AlgorithmXMLFileName = ""
experimentalData = "demo/myXCLASSMapFit/Orion.methanol.cbc.contsub.image.fits"
regionFileName = "demo/myXCLASSMapFit/ds9__test.reg"
Threshold = 0.0
vLSR = 0.0
TelescopeSize = 3.5
Inter_Flag = False
t_back_flag = True
tBack = 0.95
tSlope = 0.0
nH_flag = True
N_H = 3.0E+24
beta_dust = 2.0
kappa_1300 = 0.02
iso_flag = True
IsoTableFileName = "demo/myXCLASSMapFit/iso_names.txt"
RestFreq = 0.0
clusterdef = "demo/myXCLASSMapFit/clusterdef.txt"
JobDir = myXCLASSMapFit()
    """

    # Debug:
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("MolfitFileName = ", MolfitFileName)
    # print ("ObsXMLFileName = ", ObsXMLFileName)
    # print ("ObsDataFileName = ", ObsDataFileName)
    # print ("FastFitFlag = ", FastFitFlag)
    # print ("clusterdef = ", clusterdef)
    # print ("daskFlag = ", daskFlag)
    # print ("ParallelMethod = ", ParallelMethod)
    # print ("AlgorithmXMLFileName = ", AlgorithmXMLFileName)
    # print ("AlgorithmXMLFile = ", AlgorithmXMLFile)
    # print ("regionFileName = ", regionFileName)
    # print ("FITSImageMaskFileName = ", FITSImageMaskFileName)
    # print ("Threshold = ", Threshold)
    # print ("FullFitFlag = ", FullFitFlag)
    # print ("ChannelIntegrationFlag = ", ChannelIntegrationFlag)
    # print ("EstimateContinuumFlag = ", EstimateContinuumFlag)
    # print ("EstimateConNumParts = ", EstimateConNumParts)
    # print ("EstimateConNumEntireCubeFlag = ", EstimateConNumEntireCubeFlag)
    # print ("UsePreviousResults = ", UsePreviousResults)
    # print ("DataSmoothValue = ", DataSmoothValue)
    # print ("ParameterMapDir = ", ParameterMapDir)
    # print ("ParamMapIterations = ", ParamMapIterations)
    # print ("ParamSmoothMethodParam = ", ParamSmoothMethodParam)
    # print ("ParamSmoothMethod = ", ParamSmoothMethod)
    # print ("JobDir = ", JobDir)
    # print ("NameOfFunction = ", NameOfFunction)
    # print ("experimentalData = ", experimentalData)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("TelescopeSize = ", TelescopeSize)
    # print ("BMIN = ", BMIN)
    # print ("BMAJ = ", BMAJ)
    # print ("BPA = ", BPA)
    # print ("vLSR = ", vLSR)
    # print ("Redshift = ", Redshift)
    # print ("Inter_Flag = ", Inter_Flag)
    # print ("t_back_flag = ", t_back_flag)
    # print ("tBack = ", tBack)
    # print ("tSlope = ", tSlope)
    # print ("nH_flag = ", nH_flag)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("iso_flag = ", iso_flag)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)
    # print ("EmAbsPATH = ", EmAbsPATH)
    # print ("RestFreq = ", RestFreq)
    # print ("ZipDirFlag = ", ZipDirFlag)
    # print ("DoParameterEstimationFlag = ", DoParameterEstimationFlag)
    # print ("InternalParameterList = ", InternalParameterList)


    ##====================================================================================================================================================
    ## reset output variable
    myXCLASSMapFitJobDir = ""


    ##====================================================================================================================================================
    ## initialize some parameters


    ## define name of function
    if (NameOfFunction is None):
        NameOfFunction = "myXCLASSMapFit"


    ## get current working directory
    CurrentDir = os.getcwd() + "/"


    ## get default database file
    DefaultDBFileName = task_myXCLASS.GetDefaultDBFile()


    ## define print flag
    printflag = True


    ##====================================================================================================================================================
    ## create run directory for the current function call
    XCLASSRootDir = task_myXCLASS.GetXCLASSRootDir()                                        ## get absolute path of XCLASS root directory
    MAGIXrootDir = task_myXCLASS.GetMAGIXRootDir()                                          ## get absolute path of MAGIX directory
    myXCLASSrootDir = task_myXCLASS.GetmyXCLASSRootDir()                                    ## get absolute path of myXCLASS root directory
    LocalPrintFlag = True
    if (JobDir is None):
        myXCLASSMapFitJobDir = task_MAGIX.CreateRunDirectory(NameOfFunction, myXCLASSrootDir, LocalPrintFlag)
        myXCLASSMapFitJobDir = os.path.normpath(myXCLASSMapFitJobDir) + "/"
    else:
        myXCLASSMapFitJobDir = JobDir

    # Debug:
    # print ("MAGIXrootDir = ", MAGIXrootDir)
    # print ("myXCLASSMapFitJobDir = ", myXCLASSMapFitJobDir)
    # sys.exit(0)


    ##====================================================================================================================================================
    ## set up environment for MAGIX
    task_MAGIX.SetMAGIXEnvironment(MAGIXrootDir)

    # Debug:
    # print ("sys.path = ", sys.path)


    ## extend sys.path variable to include path for new packages
    # status = task_LineIdentification.ExtendPackage(XCLASSRootDir, NameOfFunction)
    # if (status != 0):
    #     return myXCLASSMapFitJobDir


    ## extend sys.path variable to include path for new packages
    # NewPathList = [XCLASSRootDir + "xclass/src/interface/bin/",
    #                XCLASSRootDir + "xclass/addons/MAGIX/lite/"]
    # task_myXCLASS.ExtendSysPath(NewPathList)


    # ## check if additional packages can be imported
    # if (FullFitFlag):
    #     PacakgeList = [["xclassinterface", "interfact to xclass"]]
    #     ErrorList = ""
    #     ErrorCounter = 0
    #     for LocalImport in PacakgeList:
    #         LocalPackage = LocalImport[0]
    #         LocalName = LocalImport[1]
    #         exec("from xclass import " + LocalPackage)
    #         try:
    #             exec("from xclass import " + LocalPackage)
    #         except:
    #             ErrorCounter += 1
    #             if (ErrorCounter > 1):
    #                 ErrorList += " and "
    #             ErrorList += LocalName
    #     if (ErrorList != ""):
    #         status = 1
    #         print ("\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
    #         print ("\tCan not import " + ErrorList + " package(s)" + "!")
    #         print ("\n\tPlease check your XCLASS installation and try again!")
    #         sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check use previous result flag
    if (UsePreviousResults is not None):
        print ("\n\n\t\tWARNING:  The parameter " + chr(34) + "UsePreviousResults" + chr(34) + " is no longer used!\n\n")


    ##====================================================================================================================================================
    ## get alias for keyword "MolfitFileName"
    MolfitsFileName = MolfitsFileName.strip()
    if (MolfitsFileName == "" and MolfitFileName is not None):
        MolfitsFileName = MolfitFileName.strip()


    ##====================================================================================================================================================
    ## analyze input parameter regarding obs. data


    ## print warning messages for deprecate input paramters
    if (nH_flag is not None):
        warnings.warn('nH_flag is going to be deprecated', DeprecationWarning, stacklevel = 2)


    ## analyze input parameters and create obs. xml file if neccessary
    LocalObsXMLFileName = task_myXCLASSFit.CheckObsInput(ObsXMLFileName, \
                                                         ObsDataFileName, \
                                                         experimentalData, \
                                                         NameOfFunction, \
                                                         CurrentDir, \
                                                         myXCLASSMapFitJobDir, \
                                                         FreqMin = FreqMin, \
                                                         FreqMax = FreqMax, \
                                                         t_back_flag = t_back_flag, \
                                                         tBack = tBack, \
                                                         tSlope = tSlope, \
                                                         N_H = N_H, \
                                                         beta_dust = beta_dust, \
                                                         kappa_1300 = kappa_1300, \
                                                         DustFileName = DustFileName, \
                                                         BackgroundFileName = BackgroundFileName, \
                                                         ContPhenFuncID = ContPhenFuncID, \
                                                         ContPhenFuncParam1 = ContPhenFuncParam1, \
                                                         ContPhenFuncParam2 = ContPhenFuncParam2, \
                                                         ContPhenFuncParam3 = ContPhenFuncParam3, \
                                                         ContPhenFuncParam4 = ContPhenFuncParam4, \
                                                         ContPhenFuncParam5 = ContPhenFuncParam5, \
                                                         TelescopeSize = TelescopeSize, \
                                                         BMIN = BMIN, \
                                                         BMAJ = BMAJ, \
                                                         BPA = BPA, \
                                                         Inter_Flag = Inter_Flag, \
                                                         vLSR = vLSR, \
                                                         Redshift = Redshift, \
                                                         DBFileName = DefaultDBFileName, \
                                                         NumModelPixelXX = NumModelPixelXX, \
                                                         NumModelPixelYY = NumModelPixelYY, \
                                                         LocalOverlapFlag = LocalOverlapFlag, \
                                                         NoSubBeamFlag = NoSubBeamFlag, \
                                                         EmAbsPATH = EmAbsPATH, \
                                                         iso_flag = iso_flag, \
                                                         IsoTableFileName = IsoTableFileName, \
                                                         PrintFlag = True, \
                                                         CopyFlag = False)
    ## did an error occurr?
    if (LocalObsXMLFileName is None):
        return myXCLASSMapFitJobDir


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## adjust paths in obs. xml file
    LocalObsXMLFileName, IsoRatioFileName, dbFileName = task_myXCLASSFit.AdjustObsXMLFile(NameOfFunction, LocalObsXMLFileName, CurrentDir, \
                                                                                          myXCLASSMapFitJobDir, printflag)
    if (LocalObsXMLFileName == ""):                                                         ## did an error occur
        return myXCLASSMapFitJobDir

    # Debug:
    # print ("\n\nLocalObsXMLFileName = ", LocalObsXMLFileName)
    # print ("IsoRatioFileName = ", IsoRatioFileName)
    # print ("dbFileName = ", dbFileName)


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## rename obs. xml file in the current job directory to "obs.xml"
    LocalObsXMLFileNameOLD = LocalObsXMLFileName
    LocalObsXMLFileName = myXCLASSMapFitJobDir + "obs.xml"
    if (LocalObsXMLFileNameOLD != LocalObsXMLFileName):
        cmdString = "mv " + LocalObsXMLFileNameOLD + " " + LocalObsXMLFileName
        os.system(cmdString)


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## CASA image file to fits format, apply regions to each FITS cube, convert intensities to Kelvin and spectral axes to MHz,
    ## smooth spectral axes, and apply thresholds


    ## get path and name of obs. data files from obs. xml file
    ObsDataFileNameList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "FileNamesExpFiles")


    ## import FITS file, if necessary convert image file
    NewObsDataFileNameList = []
    for LocalObsDataFile in ObsDataFileNameList:
        ObsDataFileID = LocalObsDataFile[0]
        LocalObsDataFileName = LocalObsDataFile[1].strip()


        ## make path of obs. data file absolute, if necessary
        if (not LocalObsDataFileName.startswith("/")):
            LocalObsDataFileName = CurrentDir + LocalObsDataFileName

        # Debug:
        # print ("LocalObsDataFileName = ", LocalObsDataFileName)


        ## store obs data file id and name
        NewObsDataFileNameList.append([ObsDataFileID, LocalObsDataFileName])


    ## write converted obs. data file names to obs. xml file
    task_MAGIX.WriteXMLtagNEW(LocalObsXMLFileName, "FileNamesExpFiles", NewObsDataFileNameList)


    ##====================================================================================================================================================
    ## check algorithm xml file
    AlgorithmXMLFile = AlgorithmXMLFile.strip()
    if (AlgorithmXMLFile != ""):
        AlgorithmXMLFileName = AlgorithmXMLFile
    else:
        AlgorithmXMLFileName = AlgorithmXMLFileName.strip()
    if (AlgorithmXMLFileName != ""):


        ## if neccessary, make relative path absolute
        if (AlgorithmXMLFileName[0] != "/"):
            AlgorithmXMLFileName = CurrentDir + AlgorithmXMLFileName


        ## check if path and name of the observational file exsits
        if not(os.path.exists(AlgorithmXMLFileName) or os.path.isfile(AlgorithmXMLFileName)):
            print ("\n\n\t Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:")
            print ("\t\t The given path and name of the algorithm xml-file " + chr(34) + AlgorithmXMLFileName + chr(34))
            print ("\t\t does not exsist!")
            print ("\n\t\t Please enter a valid path and name of an algorithm xml-file and restart XCLASS!")
            return myXCLASSMapFitJobDir


    ##====================================================================================================================================================
    ## analyze molfits file


    ## get some parameters from obs. xml file
    ObsXMLParameter = task_LineIdentification.GetObsXMLParameters(LocalObsXMLFileName)
    ExpFileList = ObsXMLParameter['ExpFileList']                                            ## get path and name(s) of obs. data file(s)
    NumberSpectra = int(len(ExpFileList))                                                   ## get number of obs. data file
    NumberExpRangesList = ObsXMLParameter["NumberExpRangesList"]                            ## get number of frequency ranges
    FreqMinList = ObsXMLParameter["FreqMinList"]                                            ## get min. freq. values for each freq. range
    FreqMaxList = ObsXMLParameter["FreqMaxList"]                                            ## get max. freq. values for each freq. range
    GlobalvLSRList = ObsXMLParameter["GlobalvLSRList"]                                      ## get global v_LSR
    RedshiftList = ObsXMLParameter["RedshiftList"]                                          ## get redshift
    dbList = ObsXMLParameter["dbList"]                                                      ## get path and name of used database file

    # Debug:
    # print ("ExpFileList = ", ExpFileList)
    # print ("NumberSpectra = ", NumberSpectra)
    # print ("NumberExpRangesList = ", NumberExpRangesList)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("ListOfTSlope = ", ListOfTSlope)
    # print ("Threshold = ", Threshold)
    # print ("GlobalvLSRList = ", GlobalvLSRList)
    # print ("RedshiftList = ", RedshiftList)
    # print ("SizeTelescopePerSpectrum = ", SizeTelescopePerSpectrum)
    # print ("ListBMAJPerSpectrum = ", ListBMAJPerSpectrum)
    # print ("ListBMINPerSpectrum = ", ListBMINPerSpectrum)
    # print ("InterFlagPerSpectrum = ", InterFlagPerSpectrum)
    # print ("dbFileName = ", dbFileName)
    # sys.exit(0)


    ## analyze molfit parameter: check path and name of the molfit file and copy molfit file to job directory
    MolfitsFileName = MolfitsFileName.strip()
    ok, dbFilename, LocalMolfitsFileName, PureNameMolfitsFile = task_myXCLASSFit.CheckMolfitFile(MolfitsFileName, NameOfFunction, CurrentDir, \
                                                                                                 myXCLASSMapFitJobDir, FreqMinList, FreqMaxList, \
                                                                                                 GlobalvLSRList, RedshiftList, dbList)
    if (ok == 1):                                                                           ## an error occurred
        return myXCLASSMapFitJobDir

    # Debug:
    # print ("ok = ", ok)
    # print ("LocalMolfitsFileName = ", LocalMolfitsFileName)
    # print ("PureNameMolfitsFile = ", PureNameMolfitsFile)


    ##====================================================================================================================================================
    ## iso ratio file:


    ## get iso ratio file name
    IsoFlag = False
    IsoFlagList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "Isotopologues")
    if (IsoFlagList == []):
        IsoFlagList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "iso_flag")
    if (len(IsoFlagList) > 0):
        LocalIsoFlag = CheckBool(IsoFlagList[0])
        if (LocalIsoFlag):
            IsoFlag = True
    NewIsoRatioFileName = ""
    IsoRatioFileName = ""
    if (IsoFlag):
        IsoRatioFileNameList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "IsoTableFileName")
        if (IsoRatioFileNameList != []):
            IsoRatioFileName = IsoRatioFileNameList[0].strip()
            if (IsoRatioFileName != ""):

                # Debug:
                # print ("\n\nIsoRatioFileName = ", IsoRatioFileName)
                # print ("NameOfFunction = ", NameOfFunction)
                # print ("CurrentDir = ", CurrentDir)
                # print ("FreqMinList = ", FreqMinList)
                # print ("FreqMaxList = ", FreqMaxList)
                # print ("GlobalvLSRList = ", GlobalvLSRList)
                # print ("RedshiftList = ", RedshiftList)
                # print ("dbList = ", dbList)


                ## analyze iso ratio file parameter: check path and name of iso ratio file and copy new formatted iso ratio file to job directory
                ok, NewIsoRatioFileName = task_myXCLASSFit.CheckIsoRatioFile(IsoRatioFileName, NameOfFunction, CurrentDir, myXCLASSMapFitJobDir, \
                                                                             FreqMinList, FreqMaxList, GlobalvLSRList, RedshiftList, dbList, \
                                                                             LocalMolfitsFileName)
                if (ok == 1):                                                               ## an error occurred
                    return myXCLASSMapFitJobDir
                else:
                    task_MAGIX.WriteXMLtagNEW(LocalObsXMLFileName, "IsoTableFileName", [NewIsoRatioFileName])

    # Debug:
    # print ("NewIsoRatioFileName = ", NewIsoRatioFileName)


    ##====================================================================================================================================================
    ## analyze cluster file
    ClusterNumberProcessors, ClusterServerList, ClusterFlag = ReadClusterdefFile(clusterdef, myXCLASSMapFitJobDir)


    ##====================================================================================================================================================
    ## make sure, that for "full" fitting, fast fitting is always selected as well
    if ((not FastFitFlag) and FullFitFlag):
        FastFitFlag = True


    ##====================================================================================================================================================
    ## store parameters to parameter dictionary
    ParameterDict = {}
    ParameterDict['NameOfFunction'] = NameOfFunction
    ParameterDict['CurrentDir'] = CurrentDir
    ParameterDict['myXCLASSMapFitJobDir'] = myXCLASSMapFitJobDir
    ParameterDict['ParallelMethod'] = ParallelMethod
    ParameterDict['FastFitFlag'] = FastFitFlag
    ParameterDict['FullFitFlag'] = FullFitFlag
    ParameterDict['daskFlag'] = daskFlag
    ParameterDict['EstimateContinuumFlag'] = EstimateContinuumFlag
    ParameterDict['EstimateConNumParts'] = EstimateConNumParts
    ParameterDict['EstimateConNumEntireCubeFlag'] = EstimateConNumEntireCubeFlag
    ParameterDict['ParameterMapDir'] = ParameterMapDir
    ParameterDict['ParamMapIterations'] = ParamMapIterations
    ParameterDict['ParamSmoothMethodParam'] = ParamSmoothMethodParam
    ParameterDict['ParamSmoothMethod'] = ParamSmoothMethod
    ParameterDict['RegionFileName'] = regionFileName
    ParameterDict['FITSImageMaskFileName'] = FITSImageMaskFileName
    ParameterDict['Threshold'] = Threshold
    ParameterDict['ChannelIntegrationFlag'] = ChannelIntegrationFlag
    ParameterDict['dbList'] = dbList
    ParameterDict['LocalObsXMLFileName'] = LocalObsXMLFileName
    ParameterDict['ClusterServerList'] = ClusterServerList
    ParameterDict['AlgorithmXMLFileName'] = AlgorithmXMLFileName
    ParameterDict['NumberProcessors'] = NumberProcessors
    ParameterDict['NumberIteration'] = NumberIteration
    ParameterDict['LocalMolfitsFileName'] = LocalMolfitsFileName
    ParameterDict['DoParameterEstimationFlag'] = DoParameterEstimationFlag
    ParameterDict['InternalParameterList'] = InternalParameterList
    ParameterDict['IsoFlag'] = IsoFlag
    ParameterDict['IsoRatioFileName'] = NewIsoRatioFileName

    # Debug:
    # print ("ParameterDict = ", ParameterDict)
    # sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## for parameter estimation only: extract molfit parameters
    if (DoParameterEstimationFlag):


        ## get sql parameters from molfit file
        MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(LocalMolfitsFileName)
        SQLParamArray = [MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL]


        ## prepare call of parameter estimation algorithm
        AllMoleculesInMolfitFile, OverallAllParameters, AllMolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(LocalMolfitsFileName)
        SelectedMolecules = []
        MoleculeMolfitFileContent = []
        for MoleculeIndex in range(len(AllMoleculesInMolfitFile)):                          ## loop over all molecules
            LocalMolName = AllMoleculesInMolfitFile[MoleculeIndex]                          ## get current molecule
            if (not LocalMolName.startswith("cont-")):
                SelectedMolecules.append(LocalMolName)


                ## store local molfit file
                LocalMoleculeMolfitFileContent = []
                MolfitHeaderLinesList = task_myXCLASS.WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)
                for line in MolfitHeaderLinesList:
                    LocalMoleculeMolfitFileContent.append(line)
                LocalMolfitFileContent = AllMolfitFileForEachMolecule[MoleculeIndex]
                for line in LocalMolfitFileContent:
                    LocalMoleculeMolfitFileContent.append(line)


                ## store file file
                MoleculeMolfitFileContent.append([LocalMolName, task_LineIdentification.MoleculeFileName(LocalMolName), LocalMoleculeMolfitFileContent])


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## determine all molcules (and their molecular data) which have at least one transition within the user-defined frequency ranges


        ## extract min and max. frequency and global v_lsr of each freq. range
        ListOfAllFreqMin = []
        ListOfAllFreqMax = []
        ListOfAllGlobalvLSR = []
        ListOfAllRedshift = []
        for ObsDataFileIndex in range(NumberSpectra):                                       ## loop over all spectra in the experimentalData directory
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberRangeListIn = NumberExpRangesList, \
                                                                           GlobalvLSRListIn = GlobalvLSRList, Redshift_ListIn = RedshiftList)
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']        ## get number of frequency ranges for current obs. data file
            GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
            if (GlobalvLSR is None):
                GlobalvLSR = 0.0
            Redshift = ObsXMLParameterDictFile['Redshift']
            if (Redshift is None):
                Redshift = 0.0
            for RangeIndex in range(NumberFrequencyRanges):                                 ## loop over all range definitions in the whole xml file
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = FreqMinList, \
                                                                                FreqMaxListIn = FreqMaxList)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None):
                    ListOfAllFreqMin.append(FreqMin)
                    FreqMax = ObsXMLParameterDictRange['FreqMax']
                    ListOfAllFreqMax.append(FreqMax)
                    ListOfAllGlobalvLSR.append(GlobalvLSR)
                    ListOfAllRedshift.append(Redshift)


        ## get molecular parameters from database
        Isotopologues  = []
        IsoMolecule = []
        OutputArray = task_LineIdentification.ReadDatabase(dbFileName, ListOfAllFreqMin, ListOfAllFreqMax, ListOfAllGlobalvLSR, ListOfAllRedshift, \
                                                           SelectedMolecules, Isotopologues, IsoMolecule, MinNumTransitionsSQL, MaxNumTransitionsSQL, \
                                                           MaxElowSQL, MingASQL, OrderTransSQL)
        ok = OutputArray[0]
        TotalCounterMol = OutputArray[1]
        if (ok == 1 or TotalCounterMol == 0):                                               ## no molecule is found within range
            print ("\n\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
            print ("\tFound no molecule with transitions within the given frequency range(s)!\n\n")
            print ("\n\tAbort LineID function!\n\n\n")
            return myXCLASSMapFitJobDir
        else:
            LineCatalog = OutputArray[4]
            InternalParameterList['LineCatalog'] = LineCatalog


        ## add some MapFit parameters to internal parameter list
        InternalParameterList['MapFitMAGIXrootDir'] = MAGIXrootDir
        InternalParameterList['MapFitSQLParamArray'] = SQLParamArray
        InternalParameterList['MapFitSmoothValue'] = DataSmoothValue
        InternalParameterList['MapFitMoleculeMolfitFileContent'] = MoleculeMolfitFileContent
        InternalParameterList['MapFitPureNameMolfitsFile'] = PureNameMolfitsFile
        InternalParameterList['MapFitClusterFlag'] = ClusterFlag


    ##====================================================================================================================================================
    ## fast fitting
    FastMapFitClass = FastMapFit(ParameterDict)


    ##====================================================================================================================================================
    ## compress subdirectories
    if (ZipDirFlag):
        cmdString = "gzip -r " + myXCLASSMapFitJobDir + "*/"
        os.system(cmdString)


    ##====================================================================================================================================================
    ## print some informations to screen
    print ("\n\nPlease note, all files created by the current fit process are stored in the")
    print ("following working directory of the {:s} function:\n".format(NameOfFunction))
    print ("{:s}!\n\n".format(chr(34) + myXCLASSMapFitJobDir + chr(34)))



    ##====================================================================================================================================================
    ## define return variables
    return myXCLASSMapFitJobDir
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Simplified CASA interface for MAGIX with myXCLASS program to fit a complete data cube instead of a single spectrum
##
def myXCLASSMapFit(NumberIteration, AlgorithmXMLFile, MolfitsFileName, experimentalData, regionFileName, UsePreviousResults, \
                   Threshold, FreqMin, FreqMax, TelescopeSize, BMIN, BMAJ, BPA, Inter_Flag, Redshift, t_back_flag, tBack, tslope, \
                   BackgroundFileName, N_H, beta_dust, kappa_1300, DustFileName, \
                   ContPhenFuncID, ContPhenFuncParam1, ContPhenFuncParam2, ContPhenFuncParam3, ContPhenFuncParam4, ContPhenFuncParam5, \
                   iso_flag, IsoTableFileName, NumModelPixelXX, NumModelPixelYY, LocalOverlapFlag, NoSubBeamFlag, \
                   RestFreq, vLSR, clusterdef, ParamMapIterations, ParamSmoothMethodParam, ParamSmoothMethod, PlotPixelSpectraFlag, \
                   DataSmoothValue, ParameterMapDir):
    """

This function provides a simplified interface for MAGIX using the myXCLASS program for a
map. The function starts MAGIX using the Levenberg-Marquardt algorithm to fit
observational data with the myXCLASS program. The user has to specify the max. number of
iterations, the observational data (i.e. the path and the name of a FITS file
containing the data cube), and the path and name of the molfit file.

Please note, the FITS data cube has to be given in the World Coordinate System (WCS)
described in Greisen, E. W., and Calabretta, M. R., A & A, 395, 1061-1075, 2002,
Calabretta, M. R., and Greisen, E. W., A & A, 395, 1077-1122, 2002, and
Greisen, E. W., Calabretta, M. R., Valdes, F. G., and Allen, S. L., A & A, 446,
747-771, 2006.

For each run the myXCLASSMapFit function creates a so-called job directory located in
the run directory ("path-of-myXCLASS-CASA-Interface/run/myXCLASSMapFit/") where all
files created by the myXCLASSMapFit function are stored in. The name of this job
directory is made up of four components: The first part of the name consists of the
phrase "job_" whereas the second part describes the date (day, month, year), the third
part the time stamp (hours, minutes, seconds) of the function execution. The last part
describes a so-called job ID which is composed of the so-called PID followed by a
four digit random integer number to create a really unambiguous job number, e.g.
"path-to-myXCLASS-Interface/run/myXCLASSMapFit/job__25-07-2013__12-02-03__189644932/"

Additionally, the myXCLASSMapFit function creates a subdirectory within this job
directory called "Pixel-Fits/" which contains furhter subdirectories for each pixel,
where the name of the subdirectory is made up of the values (in degrees) for the right
ascension and for the declination, separately, e.g. "83.65498__-5.24258". All output
files, created by MAGIX for this pixel are stored in this subdirectory.

At the end of the whole fit procedure, the myXCLASSMapFit function creates FITS images
for each free parameter, where each pixel correspond to the value of the free parameter
taken from the best fit for this pixel. Additionally, the myXCLASSMapFit function
creates one FITS image, where each pixel corresponds to the chi^2 value of the best
fit for this pixel.



input parameters:
-----------------

    - NumberIteration:              max. number of iterations (default: 50).

    - AlgorithmXMLFile:             only necessary, if the user wants to use another fit
                                    algorithm (than Levenberg-Marquardt) for fitting. Therefore,
                                    the path and name of a MAGXI xml-file defining settings
                                    for an algorithm or algorithm chain has to be given. (A
                                    relative path has to be defined relative to the current
                                    working directory!)

                                    NOTE, if the user specify a xml file, the number of
                                    iterations given by the parameter "NumberIteration" is
                                    ignored. The number of iteration is then given by the xml
                                    file. In order to use the implemented fit algorithm
                                    (Levenberg-Marquardt) clear the AlgorithmXMLFile parameter,
                                    i.e. AlgorithmXMLFile = "", and define the max. number of
                                    iterations by using parameter "NumberIteration".

    - MolfitsFileName:              path and name of the extended molfit file, including the
                                    source size, the rotation temperature the column density,
                                    the velocity width, the velocity offset and the flag
                                    indicating if the current component is considered for
                                    core "c" or foreground "f". Depending on the values of
                                    "t_back_flag" and "nH_flag", the molfit file has to
                                    include the columns for the background temperature, the
                                    temperature slope, the hydrogen column density, for beta
                                    and for kappa for each component as well.

                                    In contrast to the format of the molfit file the extended
                                    molfit file required by the myXCLASSMapFit function contains
                                    one (three) additional column(s) for each parameter of
                                    each component.

                                    For a detailed description of the extended molfit file
                                    required by the myXCLASSMapFit function, see the manual.

                                    NOTE, a relative path has to be defined relative to the
                                    current working directory!

    - experimentalData:             This parameter offers two different possibility to send
                                    the observational data to the myXCLASSMapFit function:

                                    - the parameter experimentalData defines the path and
                                      name of and observational xml-file suitable for MAGIX.
                                      Here, the XML file contains the path and the name of
                                      each FITS or CASA image (IMAGE) file. The tag
                                      <ImportFilter> has to be included in the XML file. The
                                      content will be automatically set by the myXCLASSMapFit
                                      function.

                                      Please note, if more than one data cube is specified
                                      in the xml file, the data cubes must describe the same
                                      map, i.e. the right ascension and the declination has
                                      to be identical!  The different data cubes must be differ
                                      only in the frequency/velocity axis, i.e. it is possible
                                      to specify different units for the frequency axis for
                                      different data cubes.

                                    - the parameter experimentalData defines the path and
                                      name of a FITS or CASA image (IMAGE) file called
                                      "observational data file" containing the data cube. Please
                                      note, the myXCLASSMapFit function assumes, that the first
                                      three axes describe the right ascension, the declination,
                                      and the frequency, respectively. If the frequencies are
                                      not given in MHz, the FITS header has to contain the
                                      definition of the CUNIT3 command word. For velocities, the
                                      header has to contain the command word RESTFRQ as well.

                                    NOTE, if the parameter experimentalData defines a relative
                                    path, the path has to be defined relative to the current
                                    working directory!

    - regionFileName:               the so-called region file (defined by the ds9 program or the casa
                                    viewer). At the moment the shape of the region has to be
                                    rectangular. (Other formats and other shapes will be available soon).

    - UsePreviousResults:           defined if the molfit file (described by the paramter
                                    MolfitsFileName) is used as initial guess for all pixels (False)
                                    or if the result from a previous pixel is used.

                                    This input parameter is no longer used!

    - ParamMapIterations:           number of iterations to smooth parameter maps, (default: 1, i.e. no smoothing)

    - ParamSmoothMethodParam:       parameter used for smoothing, (default: 1.0)

    - ParamSmoothMethod:            method used for smoothing, (default: "uniform")

    - PlotPixelSpectraFlag:         (NO LONGER USED) flag indicating if pixel spectra are created or not (default: True)

    - DataSmoothValue:              smooth factor (=0, no smoothing is performed) used for smoothing obs. data (default: 0.0).

    - ParameterMapDir:              path of subdirectory containing FITS images for different molfit parameters, (default: None)


    The following parameters are needed, if the parameter experimentalData does NOT
    describe the path and name of a MAGIX xml-file:

    - Threshold:                    defines a threshold for a pixel. If the spectrum of a pixel
                                    has a intensity lower than the value defined by this parameter
                                    the pixel is not fitted (ignored), (default: 0)

    - FreqMin:                      start frequency of simulated spectrum (in MHz), (default: 0).

                                    Please note, if no start frequency is given, or if start frequency is
                                    lower or equal to the end frequency, the myXCLASSMapFit function will
                                    use the max. frequency range defined in the FITS header. Additionally,
                                    the stepsize, i.e. the difference between two frequencies is taken
                                    from the FITS header.

    - FreqMax:                      end frequency of simulated spectrum (in MHz), (default: 0).

    - vLSR:                         local standard of rest velocity (in km/s), (default: 0).

    - TelescopeSize:                for single dish observations (Inter_Flag = False): TelescopeSize
                                    describes the size of telescope (in m), (default: 1); for
                                    interferometric observations (Inter_Flag = True): TelescopeSize
                                    describes the interferometric beam FWHM size (in arcsec),
                                    (default: 1).

    - BMIN:                         Beam minor axis length (default: None)

    - BMAJ:                         Beam major axis length (default: None)

    - BPA:                          Beam position angle (default: None)

    - Inter_Flag:                   defines, if single dish ("False") or interferometric
                                    observations ("True") are described, (default: "False").

    - Redshift:                     red shift (dimensionless), (default: None)

    - t_back_flag:                  ("True"/"False"), defines, if the user defined background
                                    temperature Tbg and temperature slope Tslope given
                                    by the input parameters tBack and tslope describe the
                                    continuum contribution completely (t_back_flag = "True")
                                    or not (t_back_flag = "False") (default: "True").

    - tBack:                        background temperature (in K), (default: 0).

    - tslope:                       temperature slope (dimensionless), (default: 0).

    - BackgroundFileName:           path and name of background file (default: "").

    - N_H:                          Hydrogen column density (in cm^{-2}), (default: None).

    - beta_dust:                    beta for dust (dimensionless), (default: None).

    - kappa_1300:                   kappa (cm^2 g^{-1}), (default: None).

    - DustFileName:                 path and name of file describing the optical depth
                                    of dust (default: "").

    - ContPhenFuncID:               describes which phenomenological function is used
                                    to describe the continuum (default: None).

    - ContPhenFuncParam1:           first parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam2:           second parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam3:           third parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam4:           fourth parameter for phenomenological function
                                    describing the continuum (default: None).

    - ContPhenFuncParam5:           fifth parameter for phenomenological function
                                    describing the continuum (default: None).

    - iso_flag:                     use isotopologues ("True"/"False"). If iso_flag is set to
                                    "False" the ratios are all set to 1 (default: "False").

    - IsoTableFileName:             path and name of an ASCII file including the iso ratios
                                    between certain molecules. The so-called "iso ratio file"
                                    defines the iso ratios between molecules. The ASCII file
                                    consists of three columns, where the first two columns
                                    indicates the molecules, respectively. The third column
                                    defines the ratio for both molecules. The columns are
                                    separated by blanks or tabs. So, the names of the
                                    molecules must not contain blanks.

                                    The myXCLASSFit function offers the possibility to
                                    optimize the ratios between isotopologues as well. For
                                    that purpose, the user has to add two additional columns
                                    on the right indicating the lower and the upper limit
                                    of a certain ratio, respectively. For more informations
                                    please look at the description of the iso ratio file given
                                    for the myXCLASS function.

                                    If the lower and upper limit are equal or if the lower
                                    limit is higher than the upper limit, the ratio is kept
                                    constant and is not optimized by the myXCLASSFit function.

                                    If no path and name is specified (default), the so-called
                                    iso-flag is set to "false".

                                    NOTE, if the parameter "IsoTableFileName" defines a relative
                                    path, the path has to be defined relative to the current
                                    working directory!

    - NumModelPixelXX:              used for sub-beam modeling, describes the number
                                    of pixels used in x-direction (default: 100).

    - NumModelPixelYY:              used for sub-beam modeling, describes the number
                                    of pixels used in y-direction (default: 100).

    - LocalOverlapFlag:             flag indicates if local-overlap description is
                                    used or not, (default: False).

    - NoSubBeamFlag:                do not use sub-beam description (default: True).

    - RestFreq:                     rest frequency in MHz (default: 0). (If this parameter is
                                    set to zero, the intensity is plotted against frequency (in MHz)
                                    otherwise against velocity (in km/s).


output parameters:
------------------

    - JobDir:                       absolute path of the job directory created for the current run.



Example using a molfit file in the old format and an ASCII file containing the observational data:
-------------------------------------------------------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSMapFit/CH3OH.molfit"
AlgorithmXMLFile = ""
experimentalData = "demo/myXCLASSMapFit/Orion.methanol.cbc.contsub.image.fits"
regionFileName = "demo/myXCLASSMapFit/ds9__test.reg"
Threshold = 0.0
vLSR = 0.0
TelescopeSize = 3.5
Inter_Flag = False
t_back_flag = True
tBack = 0.95
tslope = 0.0
nH_flag = True
N_H = 3.0E+24
beta_dust = 2.0
kappa_1300 = 0.02
iso_flag = True
IsoTableFileName = "demo/myXCLASSMapFit/iso_names.txt"
RestFreq = 0.0
clusterdef = "demo/myXCLASSMapFit/clusterdef.txt"
JobDir = myXCLASSMapFit()
    """

    # Debug:
    # print ("NumberIteration = ", NumberIteration)
    # print ("AlgorithmXMLFile = ", AlgorithmXMLFile)
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("experimentalData = ", experimentalData)
    # print ("regionFileName = ", regionFileName)
    # print ("UsePreviousResults = ", UsePreviousResults)
    # print ("Threshold = ", Threshold)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("TelescopeSize = ", TelescopeSize)
    # print ("BMIN = ", BMIN)
    # print ("BMAJ = ", BMAJ)
    # print ("BPA = ", BPA)
    # print ("Inter_Flag = ", Inter_Flag)
    # print ("Redshift = ", Redshift)
    # print ("t_back_flag = ", t_back_flag)
    # print ("tBack = ", tBack)
    # print ("tslope = ", tslope)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("iso_flag = ", iso_flag)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)
    # print ("RestFreq = ", RestFreq)
    # print ("vLSR = ", vLSR)
    # print ("clusterdef = ", clusterdef)
    # print ("ParamMapIterations = ", ParamMapIterations)
    # print ("ParamSmoothMethodParam = ", ParamSmoothMethodParam)
    # print ("ParamSmoothMethod = ", ParamSmoothMethod)
    # print ("PlotPixelSpectraFlag = ", PlotPixelSpectraFlag)
    # print ("DataSmoothValue = ", DataSmoothValue)
    # print ("ParameterMapDir = ", ParameterMapDir)


    ## interpret input variable "experimentalData"
    ObsXMLFileName = ""
    ObsDataFileName = None
    if (type(experimentalData).__name__ == 'str'):
        if (experimentalData.endswith(".xml")):                                             ## parameter defines obs. xml file
            ObsXMLFileName = experimentalData.strip()                                       ## remove leading and tailing blanks
        else:
            ObsDataFileName = experimentalData.strip()
    else:
        ObsDataFileName = experimentalData


    ## set some input parameters
    FastFitFlag = False                                                                     ## use 'normal' fitting
    FullFitFlag = False                                                                     ## procedure
    EmAbsPATH = None
    nH_flag = False
    ZipDirFlag = True
    DoParameterEstimationFlag = False
    InternalParameterList = {}


    ## call myXCLASSFit core function
    myXCLASSMapFitJobDir = myXCLASSMapFitCore(MolfitsFileName = MolfitsFileName,
                                              ObsXMLFileName = ObsXMLFileName,
                                              ObsDataFileName = ObsDataFileName,
                                              FastFitFlag = FastFitFlag,
                                              FullFitFlag = FullFitFlag,
                                              clusterdef = clusterdef,
                                              NumberIteration = NumberIteration,
                                              AlgorithmXMLFile = AlgorithmXMLFile,
                                              regionFileName = regionFileName,
                                              Threshold = Threshold,
                                              DataSmoothValue = DataSmoothValue,
                                              UsePreviousResults = UsePreviousResults,
                                              FreqMin = FreqMin,
                                              FreqMax = FreqMax,
                                              TelescopeSize = TelescopeSize,
                                              BMIN = BMIN,
                                              BMAJ = BMAJ,
                                              BPA = BPA,
                                              Redshift = Redshift,
                                              Inter_Flag = Inter_Flag,
                                              t_back_flag = t_back_flag,
                                              tBack = tBack,
                                              tSlope = tslope,
                                              BackgroundFileName = BackgroundFileName,
                                              nH_flag = nH_flag,
                                              N_H = N_H,
                                              beta_dust = beta_dust,
                                              kappa_1300 = kappa_1300,
                                              DustFileName = DustFileName,
                                              ContPhenFuncID = ContPhenFuncID,
                                              ContPhenFuncParam1 = ContPhenFuncParam1,
                                              ContPhenFuncParam2 = ContPhenFuncParam2,
                                              ContPhenFuncParam3 = ContPhenFuncParam3,
                                              ContPhenFuncParam4 = ContPhenFuncParam4,
                                              ContPhenFuncParam5 = ContPhenFuncParam5,
                                              iso_flag = iso_flag,
                                              IsoTableFileName = IsoTableFileName,
                                              NumModelPixelXX = NumModelPixelXX,
                                              NumModelPixelYY = NumModelPixelYY,
                                              LocalOverlapFlag = LocalOverlapFlag,
                                              NoSubBeamFlag = NoSubBeamFlag,
                                              EmAbsPATH = EmAbsPATH,
                                              RestFreq = RestFreq,
                                              vLSR = vLSR,
                                              ParameterMapDir = ParameterMapDir,
                                              ParamMapIterations = ParamMapIterations,
                                              ParamSmoothMethodParam = ParamSmoothMethodParam,
                                              ParamSmoothMethod = ParamSmoothMethod,
                                              PlotPixelSpectraFlag = PlotPixelSpectraFlag,
                                              ZipDirFlag = ZipDirFlag,
                                              DoParameterEstimationFlag = DoParameterEstimationFlag,
                                              InternalParameterList = InternalParameterList)


    ##====================================================================================================================================================
    ## define return variables
    return myXCLASSMapFitJobDir
##--------------------------------------------------------------------------------------------------------------------------------------------------------

