#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module contains all subroutines to perform the myXCLASSFit function, a simplified interface to MAGIX using myXCLASS.
##  Copyright (C) 2013 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function CheckMoleculesInDatabase:                            determine molecules in database
##      - function CheckMolecules:                                      check molecules used in molfit and iso ratio file
##      - function GetSplatTranslationTable:                            get translation table from given database file
##      - function FindIsotopologues:                                   find isotopologues to molecules described in molfit file
##      - function MolfitTranslateSplatalogueNames:                     analyze molecule names in given molfit file
##      - function IsoRatioTranslateSplatalogueNames:                   analyze molecule names in given iso ratio file
##      - function CheckMolfitFile:                                     check given molfit file
##      - function CheckIsoRatioFile:                                   check given iso ratio file
##      - function CheckObsInput:                                       check user input parameter "experimentalData"
##      - function CheckDustFile:                                       check given dust file
##      - function CheckBackgroundFile:                                 check given background file
##      - function AdjustObsXMLFile:                                    adjust paths in an obs. xml file
##      - function CreateExpXMLFile:                                    creates the experimental xml file in the current job directory
##      - function CreateLMControlXMLFile:                              creates the LM control xml file in the current job directory
##      - function CreateIOControlXMLFile:                              creates the io-control xml file in the current job directory
##      - function ReadInOutputFiles:                                   reads in output files
##      - function GetArrayPart:                                        extract part of array between selected limits
##      - function ItertiveFitting.__init__:                            (class ItertiveFitting) initialize class
##      - function ItertiveFitting.GetFinalResults:                     (class ItertiveFitting) get final results
##      - function ItertiveFitting.CallmyXCLASSFitFunction:             (class ItertiveFitting) call myXCLASSFit function
##      - function ItertiveFitting.StartClusterParallelization:         (class ItertiveFitting) start different MAGIX instances via the
##                                                                                              command line on the cluster
##      - function ItertiveFitting.FastCompBackground:                  (class ItertiveFitting) compute background spectra in parallel
##      - function ItertiveFitting.DoMoleculeFit:                       (class ItertiveFitting) perform molecule fits for ith molecule
##      - function ItertiveFitting.UpdatePerMoleculeFit:                (class ItertiveFitting) perform molecule fits one by one
##      - function ItertiveFitting.SeparateFits:                        (class ItertiveFitting) perform single molecule fits
##      - function myXCLASSFitCore:                                     core myXCLASSFit function
##      - function myXCLASSFit:                                         Simplified CASA interface for MAGIX with myXCLASS program
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2013-07-25      initial version
##  T. Moeller      2014-04-09      improved version
##  T. Moeller      2020-01-03      porting to python 3.x
##  T. Moeller      2020-03-20      adding class 'ItertiveFitting'
##  T. Moeller      2023-07-01      adding non-MAGIX optimization
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import copy                                                                                 ## import copy package
if (not 'matplotlib' in sys.modules):
    import matplotlib                                                                       ## import matplotlib package
    matplotlib.use("Agg")                                                                   ## avoid display error
from . import task_MAGIX                                                                    ## import package MAGIX
from . import task_myXCLASS                                                                 ## import package myXCLASS
from . import task_myXCLASSMapFit                                                           ## import package task_myXCLASSMapFit
from . import task_myXCLASSPlot                                                             ## import package myXCLASSPlot
from . import task_LineIdentification                                                       ## import package task_LineIdentification
from . import task_ListDatabase                                                             ## import package task_ListDatabase
import sqlite3                                                                              ## import sqlite3 package
import multiprocessing                                                                      ## import multiprocessing package
import pickle                                                                               ## import pickle package
from warnings import warn
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## determine molecules in database
##
def CheckMoleculesInDatabase(ListOfMolecules, PrintFlag, FreqMinList, FreqMaxList, dbList, MinNumTransitionsSQL, \
                             MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL):
    """

input parameters:
-----------------

    - ListOfMolecules:          list of molecules which are analyzed

    - PrintFlag:                flag indicating if messages are printed to screen

    - FreqMinList:              list of min. frequencies of all ranges (if list is empty, i.e. FreqMinList == [], table "partitionfunctions"
                                is used without taken frequency ranges into account, for FreqMinList != [], table "transitions" is used.)

    - FreqMaxList:              list of max. frequencies of all ranges

    - dbList:                   path and name of user defined database file

    - MinNumTransitionsSQL:     min. number of transitions for SQL query

    - MaxNumTransitionsSQL:     max. number of transitions for SQL query

    - MaxElowSQL:               max. lower energy of transitions for SQL query

    - MingASQL:                 min. intensity (gA) of transitions for SQL query

    - OrderTransSQL:            order of transitions for SQL query


output parameters:
------------------

    - ListOfIncludedMolecules:  list of included molecules

    - dbFile:                   path and name of used database file
    """

    # Debug:
    # print ("\n\nListOfMolecules = ", ListOfMolecules)
    # print ("PrintFlag = ", PrintFlag)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("dbList = ", dbList)
    # print ("MinNumTransitionsSQL = ", MinNumTransitionsSQL)
    # print ("MaxNumTransitionsSQL = ", MaxNumTransitionsSQL)
    # print ("MaxElowSQL = ", MaxElowSQL)
    # print ("MingASQL = ", MingASQL)
    # print ("OrderTransSQL = ", OrderTransSQL)
    # tic = time.perf_counter()


    ## initialize return parameter
    ListOfIncludedMolecules = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check, if molecule names are included in database
    if (PrintFlag):
        print ("\n\nCheck, if names of molecules in molfit file are included in XCLASS database .. ", end = "", flush = True)


    ## get path of default database file
    dbFilename = task_myXCLASS.GetDefaultDBFile()


    ## get path and name of database file
    if (dbList == [] or dbList == [""]):
        dbFile = dbFilename
    else:
        dbFile = dbList[0]

    # Debug:
    # print ("dbFilename = ", dbFilename)
    # print ("dbList = ", dbList)
    # print ("dbFile = ", dbFile)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## connect to database
    connOK = True
    try:
        conn = sqlite3.connect(dbFile)
    except sqlite3.Error as e:
        print ("\n\nError in function task_myXCLASSFit.CheckMoleculesInDatabase:")
        print ("\t Can not connect to sqlite3 database {:s}.".format(dbFile))
        print ("\t Error: {:d}: {:s}".format(e.args[0], e.args[1]))
        print ("\n\t Can not check names of molecules.")
        print ("\n\n")
        connOK = False


    ## remove double entries and continuum definitions from list of molecules
    MoleculeNames = sorted(set(ListOfMolecules))
    MoleculeNames[:] = [x for x in MoleculeNames if not x.lower().startswith("cont-")]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## send query string to database
    if (connOK and len(MoleculeNames) > 0):
        cursor = conn.cursor()
        if (PrintFlag):
            print ("done!")


        ## start loop over both tables
        # TableList = ["partitionfunctions", "transitions"]
        if (FreqMinList != []):
            TableList = ["transitions"]
        else:
            TableList = ["partitionfunctions"]


        ## start database query
        for table in TableList:


            ##============================================================================================================================================
            ## get name of molecules from table partitionfunctions
            if (table == "partitionfunctions"):


                ## define query string
                query_string = "SELECT PF_Name FROM " + table + " where ("
                for counter, molecule in enumerate(MoleculeNames):
                    if (counter > 0):
                        query_string += " or "
                    query_string += "PF_Name = " + chr(34) + molecule.strip() + chr(34)
                query_string += ")"


                ## read from database
                cursor = conn.cursor()
                cursor.execute(query_string)
                rows = cursor.fetchall()
                dbContents = []
                for name in rows:
                    FormatName = name[0]
                    FormatName = FormatName.strip()
                    FormatName = "{:s}".format(FormatName)
                    dbContents.append(FormatName)
                dbContents = sorted(set(dbContents))                                        ## remove double entries
                ListOfIncludedMolecules = copy.deepcopy(dbContents)


            ##============================================================================================================================================
            ## get name of molecules from table partitionfunctions
            elif (table == "transitions"):


                ## define order of transitions
                if (OrderTransSQL == 1):                                                    ## order by lower energy
                    OrderString = "T_EnergyLower"
                elif (OrderTransSQL == 2):                                                  ## order by gA
                    OrderString = "T_UpperStateDegeneracy * T_EinsteinA"
                else:                                                                       ## order by frequency
                    OrderString = "T_Frequency"


                ## define numpy array describing frequency ranges
                RangeList = []
                for RangeIndex in range(len(FreqMinList)):
                    FreqMin = FreqMinList[RangeIndex][2]
                    FreqMax = FreqMaxList[RangeIndex][2]
                    RangeList.append([FreqMin, FreqMax])
                RangeList = numpy.asarray(RangeList)
                nuMin = numpy.nanmin(RangeList[:, 0])
                nuMax = numpy.nanmax(RangeList[:, 1])


                ## define query string additions
                queryAddString = ""
                if (MaxElowSQL > 0.0):
                    h = MaxElowSQL / 1.42879
                    queryAddString += " and T_EnergyLower <= " + str(h)
                if (MingASQL > 0.0):
                    queryAddString += " and (T_UpperStateDegeneracy * T_EinsteinA) >= " + str(MingASQL)


                ## determine number of transitions for each molecule
                for counter, molecule in enumerate(MoleculeNames):
                    StrippedMoleculeName = molecule.strip()
                    LowerStrippedMoleculeName = StrippedMoleculeName.lower()

                    # Debug:
                    # print ("\nStrippedMoleculeName = ", StrippedMoleculeName)


                    ## define query string
                    ## a query string including all frequency ranges is sometimes not possible due to expression tree limit, i.e.:
                    ## "sqlite3.OperationalError: Expression tree is too large (maximum depth 1000)"
                    query_string = "SELECT T_Name FROM " + table + " where (T_Name = " + chr(34) + StrippedMoleculeName + chr(34) + " and ("
                    for RangeIndex in range(len(FreqMinList)):
                        FreqMin = FreqMinList[RangeIndex][2]
                        FreqMax = FreqMaxList[RangeIndex][2]
                        if (RangeIndex > 0):
                            query_string += " or "
                        query_string += "(T_Frequency >= " + str(FreqMin) + " and T_Frequency <= " + str(FreqMax) + ")"
                    if (not (LowerStrippedMoleculeName.startswith("rrl"))):
                        query_string += ")" + queryAddString + ") ORDER BY " + OrderString
                    else:
                        query_string += "))"


                    ##====================================================================================================================================
                    ## get content from sqlite database


                    ## first, try the normal query including all frequency ranges
                    cursor = conn.cursor()
                    NumberTransitions = 0
                    ExpressionTreeTooLargeFlag = False
                    try:
                        cursor.execute(query_string)
                    except:
                        ExpressionTreeTooLargeFlag = True
                    if (not ExpressionTreeTooLargeFlag):
                        rows = cursor.fetchall()
                        NumberTransitions = len(rows)


                    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                    ## continue here, if expression tree is too large (maximum depth 1000)!
                    else:


                        ## define reduced query string with only one big range
                        query_string = "SELECT T_Name, T_Frequency FROM " + table + " where (T_Name = " + chr(34) + StrippedMoleculeName + chr(34)
                        query_string += " and ((T_Frequency >= {:.3f} and T_Frequency <= {:.3f})".format(nuMin, nuMax)
                        if (not (LowerStrippedMoleculeName.startswith("rrl"))):
                            query_string += ")" + queryAddString + ") ORDER BY " + OrderString
                        else:
                            query_string += "))"

                        # Debug:
                        # print ("query_string = ", query_string)


                        ## get content from sqlite database
                        cursor = conn.cursor()
                        cursor.execute(query_string)
                        rows = cursor.fetchall()
                        rows = numpy.asarray(rows)
                        if (len(rows) > 0):
                            vtAraray = rows[:, 1].astype(numpy.float)


                            ## range test for each transition frequency using numpy's meshgrid
                            x11, x21 = numpy.meshgrid(vtAraray, RangeList[:, 0], sparse = True)
                            x12, x22 = numpy.meshgrid(vtAraray, RangeList[:, 1], sparse = True)
                            mask1 = (x11 >= x21)
                            mask2 = (x12 <= x22)
                            mask12 = numpy.logical_and(mask1, mask2)
                            mask = numpy.sum(mask12, axis = 0)
                            NumberTransitions = len(mask[mask > 0])

                            # Debug:
                            # for vtID, vt in enumerate(vtAraray):
                            #     for LocalRangeID, LocalRange in enumerate(RangeList[:, 0]):
                            #         print ("vt, FreqMin, FreqMax,  = ", vt, LocalRange, RangeList[LocalRangeID, 1], mask12[LocalRangeID, vtID])
                            #     print ("mask[:, vtID] = ", mask12[:, vtID], mask[vtID])


                    ##====================================================================================================================================
                    ## check if number of transitions within freq. ranges are above user-given threshold
                    if (NumberTransitions >= MinNumTransitionsSQL):
                        ListOfIncludedMolecules.append(MoleculeNames[counter])


        ## close connection to database
        if (PrintFlag):
            print ("\n\n")
        conn.close()

    # Debug:
    # print ("ListOfIncludedMolecules = ", ListOfIncludedMolecules, "\n\n")
    # toc = time.perf_counter()
    # print (f"Timing for function CheckMoleculesInDatabase: {toc - tic:0.4f} sec.")


    ## define return parameter
    return (ListOfIncludedMolecules, dbFile)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check molecules used in molfit and iso ratio file
##
def CheckMolecules(MolfitsFileName, IsoRatioFileName, FreqMinList, FreqMaxList, dbList):
    """

input parameters:
-----------------

    - MolfitsFileName:      path and name of molfit file

    - IsoRatioFileName:     path and name of iso ratio file

    - FreqMinList:          list of min. frequencies of all ranges

    - FreqMaxList:          list of max. frequencies of all ranges

    - dbList:               path and name of database file


output parameters:
------------------

    - FoundIsoRatioMolecules:   list of iso ratio molecules which were found in database
    """

    # Debug:
    # print ("\n\nMolfitsFileName = ", MolfitsFileName)
    # print ("IsoRatioFileName = ", IsoRatioFileName)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("dbList = ", dbList)


    ## define return parameters
    FoundIsoRatioMolecules = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze given molfit file


    ## get sql parameters from molfit file
    MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(MolfitsFileName)


    ## analyze given molfit file
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(MolfitsFileName)

    # Debug:
    # print ("MoleculesInMolfitFile = ", MoleculesInMolfitFile)


    ## check, if molecules defined in the molfit file are included in the current database
    PrintFlag = False
    FoundMolfitMolecules, dbFile = CheckMoleculesInDatabase(MoleculesInMolfitFile, PrintFlag, FreqMinList, FreqMaxList, dbList, MinNumTransitionsSQL, \
                                                            MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze given iso-ratio file
    NewLocalIsoRatioFileName = ""
    IsoRatioTable, Isotopologues, IsoMolecule = task_myXCLASS.ImportIsoRatioFile(IsoRatioFileName, NewLocalIsoRatioFileName)

    # Debug:
    # print ("len(IsoRatioTable) = ", len(IsoRatioTable))
    # print ("IsoRatioTable = ", IsoRatioTable)
    # print ("len(Isotopologues) = ", len(Isotopologues))
    # print ("Isotopologues = ", Isotopologues)
    # print ("IsoMolecule = ", IsoMolecule)


    ## select only those isotopologues whose iso master molecule is included in the molfit file
    SelectIsoRatioTable = []
    SelectedIsotopologues = []
    SelectedIsoMolecule = []
    GlobalIsoList = []
    for LineIndex in range(len(Isotopologues)):
        LocalIsoMolecule = IsoMolecule[LineIndex]
        LocalIsotopologues = Isotopologues[LineIndex]
        if (LocalIsoMolecule in FoundMolfitMolecules or LocalIsotopologues.find("GLOBAL__") > (-1)):


            ## check, if molecules defined in the molfit file are included in the current database
            if (LocalIsotopologues.find("GLOBAL__") == (-1)):
                FoundIsoRatioMolecules, dbFile = CheckMoleculesInDatabase([LocalIsotopologues], PrintFlag, FreqMinList, FreqMaxList, dbList, \
                                                                          MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, \
                                                                          OrderTransSQL)
                if (FoundIsoRatioMolecules != []):
                    SelectIsoRatioTable.append(IsoRatioTable[LineIndex])
                    SelectedIsotopologues.append(Isotopologues[LineIndex])
                    SelectedIsoMolecule.append(LocalIsoMolecule)
            else:
                SelectIsoRatioTable.append(IsoRatioTable[LineIndex])
                GlobalIsoList.append(LocalIsotopologues.replace("GLOBAL__", ""))

    # Debug:
    # print ("SelectIsoRatioTable = ", SelectIsoRatioTable)
    # print ("SelectedIsotopologues = ", SelectedIsotopologues)
    # print ("SelectedIsoMolecule = ", SelectedIsoMolecule)


    ## check, if globally defined isotopologue ratios are really needed
    if (GlobalIsoList != []):
        UnusedGlobalIsoList = []
        for GlobalIso in GlobalIsoList:                                                     ## loop over all global iso definitions
            IncludedFlag = False
            for LocalIsotopologue in SelectedIsotopologues:                                 ## loop over all user defined isotopologues
                if (LocalIsotopologue.find(GlobalIso) > (-1)):                              ## is globally defined iso used
                    IncludedFlag = True
                    break
            if (not IncludedFlag):                                                          ## if globally defined isotopologue not used?
                UnusedGlobalIsoList.append(GlobalIso)


        ## remove unused globally defined isotopologues from iso ratio file
        if (UnusedGlobalIsoList != []):
            OldSelectIsoRatioTable = copy.deepcopy(SelectIsoRatioTable)
            SelectIsoRatioTable = []
            for IsoLine in OldSelectIsoRatioTable:
                PureIsoLine = IsoLine[0].strip()
                FoundFlag = False
                for UnusedGlobalIsoRatio in UnusedGlobalIsoList:
                    if (PureIsoLine == "GLOBAL__" + UnusedGlobalIsoRatio):
                        FoundFlag = True
                        break
                if (not FoundFlag):
                    SelectIsoRatioTable.append(IsoLine)

    # Debug:
    # print ("SelectIsoRatioTable = ", SelectIsoRatioTable)


    ## define return parameter
    return SelectIsoRatioTable
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get translation table from given database file
##
def GetSplatTranslationTable(LocalDBFileName):
    """

input parameters:
-----------------

    - LocalDBFileName:      path and name of database file


output parameters:
------------------

    - SplatFlag:            flag indicating if translation table is included in given database file

    - LocalMoleculeNameTranslationTable:        translation table
    """

    # Debug:
    # print ("LocalDBFileName = ", LocalDBFileName)


    ## initialize return parameter
    SplatFlag = False
    LocalMoleculeNameTranslationTable = []


    ## check, if database file was defined
    LocalDBFileName = LocalDBFileName.strip()
    if (LocalDBFileName != ""):


        ## connect to database
        connOK = True
        try:
            conn = sqlite3.connect(LocalDBFileName)
        except sqlite3.Error as e:
            print ("\n\nError in function task_myXCLASSFit.GetSplatTranslationTable:")
            print ("\t\t Can not connect to sqlite3 database {:s}.".format(LocalDBFileName))
            print ("\t\t Error: {:d}: {:s}".format(e.args[0], e.args[1]))
            print ("\n\t\t Can not check names of molecules.")
            print ("\n\n")
            connOK = False
        if (connOK):
            cursor = conn.cursor()


            ## check, if translation table is included
            query_string = "SELECT count(PF_SPLAT_NAME) FROM partitionfunctions"
            try:
                cursor.execute(query_string)
                SplatFlag = True
            except sqlite3.OperationalError:
                SplatFlag = False

            # Debug:
            # print ("\n\nSplatFlag = ", SplatFlag)


            ## get Splatalogue and corresponding VAMDC names
            if (SplatFlag):
                query_string = "SELECT PF_Name, PF_SPLAT_NAME FROM partitionfunctions ORDER BY PF_Name"
                cursor.execute(query_string)
                rows = cursor.fetchall()
                for name in rows:


                    ## get VAMDC name
                    VAMDCName = name[0] #.encode('ascii', 'ignore')
                    VAMDCName = VAMDCName.strip()
                    #VAMDCName = "{:s}".format(VAMDCName)


                    ## get Splatalogue name
                    SplatName = name[1]
                    if (SplatName is not None):
                        SplatName = SplatName   #.encode('ascii', 'ignore')
                        SplatName = SplatName.strip()
                    #SplatName = "{:s}".format(SplatName)

                    # Debug:
                    # print ("\nname = ", name)
                    # print ("VAMDCName = ", VAMDCName)
                    # print ("SplatName = ", SplatName)


                    ## store both names
                    LocalMoleculeNameTranslationTable.append([VAMDCName, SplatName])

        # Debug:
        # print ("SplatFlag = ", SplatFlag)
        # print ("LocalMoleculeNameTranslationTable = ", LocalMoleculeNameTranslationTable)


    ## define return parameter
    return (SplatFlag, LocalMoleculeNameTranslationTable)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## find isotopologues
##
def FindIsotopologues(NewIsoRatioFileName, ObsXMLFileName, MoleculesInMolfitFile = None, MolfitsFileName = "", IsoRatioFileName = "", \
                      printflag = True):
    """

input parameters:
-----------------

    - NewIsoRatioFileName:      path and name of new iso ratio file

    - ObsXMLFileName:           path and name of obs. xml-file

    - MoleculesInMolfitFile:    (optional) list of molecules (default: None)

    - MolfitsFileName:          (optional) path and name of molfit file (default: "")

    - IsoRatioFileName:         (optional) path and name of iso-ratio file (default: "")

    - printflag:                (optional) flag for screen output (default: True)


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("NewIsoRatioFileName = ", NewIsoRatioFileName)
    # print ("ObsXMLFileName = ", ObsXMLFileName)
    # print ("MoleculesInMolfitFile = ", MoleculesInMolfitFile)
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("IsoRatioFileName = ", IsoRatioFileName)
    # print ("printflag = ", printflag)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define list of phrases indicating isotopologues
    #                atom:  ratio:  isotop.:                                low:  high:
    IndicatorList = [["H",     50, ["D", "T"],                             [30.0, 1.e4]], \
                     ["Li",     1, ["-6-Li",  "6-Li",  "Li-6-",  "Li-6" ], [ 1.0,  1.0]], \
                     ["B",      1, ["-10-B",  "10-B",  "B-10-",  "B-10" ], [ 1.0,  1.0]], \
                     ["C",      1, ["-12-C",  "12-C",  "C-12-",  "C-12" ], [ 1.0,  1.0]], \
                     ["C",      1, ["-13-C",  "13-C",  "C-13-",  "C-13" ], [ 1.0,  1.0]], \
                     ["N",      1, ["-14-N",  "14-N",  "N-14-",  "N-14" ], [ 1.0,  1.0]], \
                     ["N",      1, ["-15-N",  "15-N",  "N-15-",  "N-15" ], [ 1.0,  1.0]], \
                     ["O",      1, ["-16-O",  "16-O",  "O-16-",  "O-16" ], [ 1.0,  1.0]], \
                     ["O",      1, ["-17-O",  "17-O",  "O-17-",  "O-17" ], [ 1.0,  1.0]], \
                     ["O",      1, ["-18-O",  "18-O",  "O-18-",  "O-18" ], [ 1.0,  1.0]], \
                     ["Al",     1, ["-27-Al", "27-Al", "Al-27-", "Al-27"], [ 1.0,  1.0]], \
                     ["Si",     1, ["-28-Si", "28-Si", "Si-28-", "Si-28"], [ 1.0,  1.0]], \
                     ["Si",     1, ["-29-Si", "29-Si", "Si-29-", "Si-29"], [ 1.0,  1.0]], \
                     ["Si",     1, ["-30-Si", "30-Si", "Si-30-", "Si-30"], [ 1.0,  1.0]], \
                     ["Ne",     1, ["-22-Ne", "22-Ne", "Ne-22-", "Ne-22"], [ 1.0,  1.0]], \
                     ["S",      1, ["-33-S",  "33-S",  "S-33-",  "S-33" ], [ 1.0,  1.0]], \
                     ["S",      1, ["-34-S",  "34-S",  "S-34-",  "S-34" ], [ 1.0,  1.0]], \
                     ["S",      1, ["-36-S",  "36-S",  "S-36-",  "S-36" ], [ 1.0,  1.0]], \
                     ["Cl",     1, ["-35-Cl", "35-Cl", "Cl-35-", "Cl-35"], [ 1.0,  1.0]], \
                     ["Cl",     1, ["-37-Cl", "37-Cl", "Cl-37-", "Cl-37"], [ 1.0,  1.0]], \
                     ["K",      1, ["-41-K",  "41-K",  "K-41-",  "K-41" ], [ 1.0,  1.0]], \
                     ["Ti",     1, ["-48-Ti", "48-Ti", "Ti-48-", "Ti-48"], [ 1.0,  1.0]], \
                     ["Ti",     1, ["-50-Ti", "50-Ti", "Ti-50-", "Ti-50"], [ 1.0,  1.0]], \
                     ["Fe",     1, ["-56-Fe", "56-Fe", "Fe-56-", "Fe-56"], [ 1.0,  1.0]], \
                     ["Ni",     1, ["-58-Ni", "58-Ni", "Ni-58-", "Ni-58"], [ 1.0,  1.0]], \
                     ["Cu",     1, ["-63-Cu", "63-Cu", "Cu-63-", "Cu-63"], [ 1.0,  1.0]], \
                     ["Cu",     1, ["-65-Cu", "65-Cu", "Cu-65-", "Cu-65"], [ 1.0,  1.0]], \
                     ["Zn",     1, ["-64-Zn", "64-Zn", "Zn-64-", "Zn-64"], [ 1.0,  1.0]], \
                     ["Zn",     1, ["-66-Zn", "66-Zn", "Zn-66-", "Zn-66"], [ 1.0,  1.0]], \
                     ["Zn",     1, ["-68-Zn", "68-Zn", "Zn-68-", "Zn-68"], [ 1.0,  1.0]], \
                     ["Br",     1, ["-79-Br", "79-Br", "Br-79-", "Br-79"], [ 1.0,  1.0]], \
                     ["Se",     1, ["-80-Se", "80-Se", "Se-80-", "Se-80"], [ 1.0,  1.0]], \
                     ["Br",     1, ["-81-Br", "81-Br", "Br-81-", "Br-81"], [ 1.0,  1.0]], \
                     ["Br",     1, ["-88-Sr", "88-Sr", "Sr-88-", "Sr-88"], [ 1.0,  1.0]]]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze given molfit file
    if (MoleculesInMolfitFile is None):
        if (printflag):
            print ("\n\nAnalyze molfit file ..", flush = True)
        MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(MolfitsFileName)
        # MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(MolfitsFileName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in parameters for each freq. range
    FreqMinList = task_MAGIX.GetXMLtag(ObsXMLFileName, "MinExpRange")
    FreqMaxList = task_MAGIX.GetXMLtag(ObsXMLFileName, "MaxExpRange")
    dbList = task_MAGIX.GetXMLtag(ObsXMLFileName, "dbFilename")
    if (dbList == []):
        dbFileName = task_myXCLASS.GetDefaultDBFile()
        dbList = [dbFileName]
    dbFileName = dbList[0]

    # Debug:
    # print ("dbFileName = ", dbFileName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze given iso-ratio file
    IsoRatioTable = []


    ## if no external iso ratio file is given, get iso ratio file from obs. xml file
    if (IsoRatioFileName == ""):
        IsoFlag = False
        IsotopologuesList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "iso_flag")
        if (IsotopologuesList == []):
            IsotopologuesList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "Isotopologues")
        if (len(IsotopologuesList) > 0):
            IsoFlag = IsotopologuesList[0].lower()
            IsoFlag = task_myXCLASSMapFit.CheckBool(IsoFlag)
        if (IsoFlag):
            IsoTableFileNameList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "IsoTableFileName")
            IsoRatioFileName = IsoTableFileNameList[0]


    ## analyze iso-ratio file
    if (IsoRatioFileName != ""):
        if (printflag):
            print ("Analyze iso-ratio file ..")
        NewLocalIsoRatioFileName = ""
        IsoRatioTable, Isotopologues, IsoMolecule = task_myXCLASS.ImportIsoRatioFile(IsoRatioFileName, NewLocalIsoRatioFileName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get molecules within given frequency ranges
    ListOfMolecules = []
    for RangeID in range(len(FreqMinList)):
        FreqMin = float(FreqMinList[RangeID][1])
        FreqMax = float(FreqMaxList[RangeID][1])


        ## print what you do
        if (printflag):
            print ("Get all molecules from database in range " + str(RangeID + 1) + ":  " + str(FreqMin) + " - " + str(FreqMax) + " MHz ..", \
                    end = ' ', flush = True)


        ## read database
        OutputDevice = "quiet"
        Contents = task_ListDatabase.ListDatabaseCore(FreqMin = FreqMin, FreqMax = FreqMax, OutputDevice = "quiet", dbFilename = dbFileName)
        for line in Contents:
            StrippedLine = line.strip()
            SplittedLine = StrippedLine.split()
            LocalMolfitMoleculeName = SplittedLine[0].strip()
            LocalMolfitMoleculeName = "{:s}".format(LocalMolfitMoleculeName)
            if (not LocalMolfitMoleculeName in ListOfMolecules):
                ListOfMolecules.append(LocalMolfitMoleculeName)


        ## print what you do
        if (printflag):
            print ("done!", flush = True)

    # Debug:
    # print ("ListOfMolecules = ", ListOfMolecules)
    # sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## loop over all molecules in the molfit file
    ListOfIsoMaster = []
    ListOfIsotopologues = []
    for MolfitMoleculeName in MoleculesInMolfitFile:


        ## print what you do
        if (printflag):
            print ("\nSearch for isotopologues for " + MolfitMoleculeName + " ..", end = ' ', flush = True)


        ## extract pure name of molecule
        StrippedMolfitMoleculeName = MolfitMoleculeName.strip()
        i = StrippedMolfitMoleculeName.find(";")
        if (i > (-1)):
            PureMolfitMoleculeName = StrippedMolfitMoleculeName[:i]
        else:
            PureMolfitMoleculeName = StrippedMolfitMoleculeName


        ## search for isotopologues and vibrational excited molecules
        for AllMolecule in ListOfMolecules:
            StrippedAllMolecule = AllMolecule.strip()
            i = StrippedAllMolecule.find(";")
            if (i > (-1)):
                PureAllMolecule = StrippedAllMolecule[:i]
            else:
                PureAllMolecule = StrippedAllMolecule


            ## search for non-isotopologues
            if (StrippedAllMolecule != StrippedMolfitMoleculeName and (not StrippedAllMolecule.endswith("#1")) \
                and (not StrippedAllMolecule.endswith("#2")) and (not StrippedAllMolecule.endswith("hyp1"))):
                if (PureMolfitMoleculeName == PureAllMolecule):
                    ListOfIsotopologues.append([StrippedAllMolecule, StrippedMolfitMoleculeName, 1.0, 1.0, 1.0])
                    ListOfIsoMaster.append(StrippedMolfitMoleculeName)


                ## search for isotopologues
                else:
                    IsoLimits = [1.0, 1.0]
                    LocalAllMolecule = AllMolecule
                    LocalIsoRatio = 1.0
                    for IndicatorString in IndicatorList:
                        MasterIon = IndicatorString[0]
                        IsoRatio = float(IndicatorString[1])
                        IsoMolList = IndicatorString[2]
                        for IsoMol in IsoMolList:
                            if (LocalAllMolecule.find(IsoMol) > (-1)):
                                LocalIsoRatio = LocalIsoRatio * IsoRatio
                                LocalAllMolecule = LocalAllMolecule.replace(IsoMol, MasterIon)
                                LocalIsoLimits = IndicatorString[3]
                                if (LocalIsoLimits[1] > 1.0):
                                    IsoLimits = [LocalIsoLimits[0], LocalIsoLimits[1]]
                    i = LocalAllMolecule.find(";")
                    if (i > (-1)):
                        LocalAllMolecule = LocalAllMolecule[:i]
                    if (LocalAllMolecule.strip() == PureMolfitMoleculeName):
                        ListOfIsotopologues.append([AllMolecule, StrippedMolfitMoleculeName, LocalIsoRatio, IsoLimits[0], IsoLimits[1]])
                        ListOfIsoMaster.append(StrippedMolfitMoleculeName)
                    else:
                        ModLocalAllMolecule = LocalAllMolecule.strip()
                        ModLocalAllMolecule = ModLocalAllMolecule.replace("OO", "O2")
                        ModLocalAllMolecule = ModLocalAllMolecule.replace("HD", "H2")
                        ModLocalAllMolecule = ModLocalAllMolecule.replace("DH", "H2")
                        ModLocalAllMolecule = ModLocalAllMolecule.replace("HH", "H2")
                        if (ModLocalAllMolecule == PureMolfitMoleculeName):
                            ListOfIsotopologues.append([AllMolecule, StrippedMolfitMoleculeName, LocalIsoRatio, IsoLimits[0], IsoLimits[1]])
                            ListOfIsoMaster.append(StrippedMolfitMoleculeName)


        ## print what you do
        if (printflag):
            print ("done!", flush = True)


    ## compare with given iso ratio file
    if (IsoRatioFileName != ""):
        NotIncludedList = []
        for line in ListOfIsotopologues:
            IsoMasterNew = line[0].strip()
            IsotopologueNew = line[1].strip()
            IncludedFlag = False
            for OldLine in IsoRatioTable:
                IsoMasterOld = OldLine[0].strip()
                IsotopologueOld = OldLine[1].strip()
                if (IsoMasterNew == IsoMasterOld):
                    if (IsotopologueNew == IsotopologueOld):
                        IncludedFlag = True
                        break
            if (not IncludedFlag):
                NotIncludedList.append(line)


        ## find not included entries
        FurtherMolList = []
        for OldLine in IsoRatioTable:
            IsoMasterOld = OldLine[0].strip()
            IsotopologueOld = OldLine[1].strip()
            IncludedFlag = False
            for line in ListOfIsotopologues:
                IsoMasterNew = line[0].strip()
                IsotopologueNew = line[1].strip()
                if (IsoMasterNew == IsoMasterOld):
                    if (IsotopologueNew == IsotopologueOld):
                        IncludedFlag = True
                        break
            if (not IncludedFlag):
                FurtherMolList.append(OldLine)


    ## define new iso ratio file
    NewIsoRatioFile = open(NewIsoRatioFileName, "w")
    NewLine = "%-45s %-45s %25s %25s %25s" % ("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
    NewIsoRatioFile.write(NewLine + "\n")
    ListOfAlreadyUsedIsotopologues = []
    KnownPairs = []
    for line in IsoRatioTable:
        LocalIsotopologue = line[0].strip()
        LocalIsoMaster = line[1].strip()
        LocalPair = [LocalIsotopologue, LocalIsoMaster]
        if ((not LocalIsotopologue in ListOfAlreadyUsedIsotopologues) and (not LocalIsotopologue in ListOfIsoMaster) and (not LocalPair in KnownPairs)):
            ListOfAlreadyUsedIsotopologues.append(LocalIsotopologue)
            KnownPairs.append(LocalPair)
            hstring = "%-45s %-45s %25.2f %25.2f %25.2f" % (LocalIsotopologue, line[1], float(line[2]), float(line[3]), float(line[4]))
            NewIsoRatioFile.write(hstring + "\n")
    for line in ListOfIsotopologues:
        LocalIsotopologue = line[0].strip()
        LocalIsoMaster = line[1].strip()
        LocalPair = [LocalIsotopologue, LocalIsoMaster]
        if ((not LocalIsotopologue in ListOfAlreadyUsedIsotopologues) and (not LocalIsotopologue in ListOfIsoMaster) and (not LocalPair in KnownPairs)):
            ListOfAlreadyUsedIsotopologues.append(LocalIsotopologue)
            KnownPairs.append(LocalPair)
            if (LocalIsotopologue.endswith("A")):
                LocalIsotopologue = "%1  " + LocalIsotopologue
            elif (LocalIsotopologue.endswith("E")):
                LocalIsotopologue = "%2  " + LocalIsotopologue
            elif (LocalIsotopologue.endswith("ortho") and (not line[1].endswith("ortho")) and (not line[1].endswith("para"))):
                LocalIsotopologue = "%3  " + LocalIsotopologue
            elif (LocalIsotopologue.endswith("para") and (not line[1].endswith("ortho")) and (not line[1].endswith("para"))):
                LocalIsotopologue = "%4  " + LocalIsotopologue
            hstring = "%-45s %-45s %25.2f %25.2f %25.2f" % (LocalIsotopologue, line[1], line[2], line[3], line[4])
            NewIsoRatioFile.write(hstring + "\n")
    NewIsoRatioFile.close()


    ## define return parameters
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## analyze molecule names in given molfit file
##
def MolfitTranslateSplatalogueNames(LocalMolfitsFileName, LocalDBFileName, printFlag):
    """

input parameters:
-----------------

    - LocalMolfitsFileName: path and name of molfit file

    - LocalDBFileName:      path and name of database file

    - printFlag:            flag for screen output


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("LocalMolfitsFileName = ", LocalMolfitsFileName)
    # print ("LocalDBFileName = ", LocalDBFileName)
    # print ("printFlag = ", printFlag)


    ## check, if database file was defined
    LocalDBFileName = LocalDBFileName.strip()
    if (LocalDBFileName != ""):


        ## import molfit file
        LocalMolfitsFile = open(LocalMolfitsFileName)
        MolfitFileContent = LocalMolfitsFile.readlines()
        LocalMolfitsFile.close()


        ##================================================================================================================================================
        ## get Splatalgoue translation table from database file
        SplatFlag, LocalMoleculeNameTranslationTable = GetSplatTranslationTable(LocalDBFileName)
        if (SplatFlag):
            ListVAMDCNames, ListSplatNames = list(zip(*LocalMoleculeNameTranslationTable))
        else:
            ListVAMDCNames = copy.deepcopy(LocalMoleculeNameTranslationTable)
            ListSplatNames = []

        # Debug:
        # print ("\n\nSplatFlag = ", SplatFlag)
        # print ("LocalMoleculeNameTranslationTable = ", LocalMoleculeNameTranslationTable)
        # print ("ListVAMDCNames = ", ListVAMDCNames)
        # print ("ListSplatNames = ", ListSplatNames)


        ##================================================================================================================================================
        ## split final molfit file in N small molfit files containing only one molecule
        NewMolfitsFile = open(LocalMolfitsFileName, 'w')
        ListOfReplacedNames = []
        for line in MolfitFileContent:                                                      ## loop over all lines of the molfit file
            StrippedLine = line.strip()                                                     ## remove leading and trailing blanks


            ## remove comments
            i = StrippedLine.find("%")
            if (i > (-1)):
                StrippedLine = StrippedLine[:i].strip()


            ## analyze current line of molfit file
            if (StrippedLine != ""):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## check if current line contains a molecule name from VAMDC or Splatalogue
                if (SplatFlag):                                                             ## check if translation table is defined
                    FoundNameFlag = False
                    for TranslationName in LocalMoleculeNameTranslationTable:               ## loop over all entries in translation table
                        LocalStrippedLine = StrippedLine                                    ## make a copy of current line


                        ## check VAMDC name
                        VAMDCName = TranslationName[0]                                      ## get VAMDC name
                        VAMDCIndex = LocalStrippedLine.find(VAMDCName)                      ## check, if VAMDC name is included in current line


                        ## check Splatalogue name
                        if (VAMDCIndex == (-1)):                                            ## we didn't find a VAMDC name in the current line
                            SplatName = TranslationName[1]                                  ## get Splatalogue name
                            if (str(SplatName) != "None" and SplatName != ""):              ## check, if Splatalogue name is not empty
                                SplatIndex = LocalStrippedLine.find(SplatName)              ## check, if Splatalogue name is included in current line
                                if (SplatIndex > (-1)):                                     ## we've found a Splatalogue name in the current line
                                    LocalStrippedLine = LocalStrippedLine.replace(SplatName, VAMDCName) ## replace Splatalogue with VAMDC name


                                    ## check if name is identified
                                    LocalSplittedLine = LocalStrippedLine.split()           ## split current line into columns
                                    if (len(LocalSplittedLine) == 2):                       ## check if current line has two columns
                                        if (LocalSplittedLine[0].strip() in ListVAMDCNames):
                                            StrippedLine = LocalStrippedLine                ## make a copy of current line
                                            FoundNameFlag = True                            ## set found flag
                                            ListOfReplacedNames.append([SplatName, VAMDCName])
                                            break                                           ## continue with normal analysis


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## analyze line
                SplittedLine = StrippedLine.split()                                         ## split current line into columns
                NumberElementsLine = len(SplittedLine)                                      ## determine number of columns
                if (NumberElementsLine == 2):                                               ## two columns current line: a molecule name is defined
                    NewMolfitsFile.write(StrippedLine + "\n")
                else:
                    NewMolfitsFile.write(line)
            else:
                NewMolfitsFile.write(line)


        ## close file
        NewMolfitsFile.close()


    ## print all replaced molecule names
    if (printFlag and ListOfReplacedNames != []):
        print ("\n\nFound Splatalogue name(s) in molfit file. Replaced the following name(s):")
        for LocalReplacedNames in ListOfReplacedNames:
            print ("\t{:s}{:s}{:s} -> {:s}{:s}{:s}".format(chr(34), LocalReplacedNames[0], chr(34), chr(34), LocalReplacedNames[1], chr(34)))
        print ("\n")


    ## define return parameter
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## analyze molecule names in given iso ratio file
##
def IsoRatioTranslateSplatalogueNames(LocalIsoRatioFileName, LocalDBFileName, printFlag):
    """

input parameters:
-----------------

    - LocalIsoRatioFileName: path and name of iso ratio file

    - LocalDBFileName:      path and name of database file

    - printFlag:            flag for screen output


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("LocalIsoRatioFileName = ", LocalIsoRatioFileName)
    # print ("LocalDBFileName = ", LocalDBFileName)
    # print ("printFlag = ", printFlag)


    ## check, if database file was defined
    LocalDBFileName = LocalDBFileName.strip()
    if (LocalDBFileName != ""):


        ##================================================================================================================================================
        ## import iso ratio file
        LocalIsoRatioFile = open(LocalIsoRatioFileName)
        LocalIsoRatioFileContent = LocalIsoRatioFile.readlines()
        LocalIsoRatioFile.close()


        ##================================================================================================================================================
        ## get Splatalgoue translation table from database file
        SplatFlag, LocalMoleculeNameTranslationTable = GetSplatTranslationTable(LocalDBFileName)
        if (SplatFlag):
            ListVAMDCNames, ListSplatNames = list(zip(*LocalMoleculeNameTranslationTable))
        else:
            ListVAMDCNames = copy.deepcopy(LocalMoleculeNameTranslationTable)
            ListSplatNames = []

        # Debug:
        # print ("SplatFlag = ", SplatFlag)
        # print ("LocalMoleculeNameTranslationTable = ", LocalMoleculeNameTranslationTable)
        # print ("ListVAMDCNames = ", ListVAMDCNames)
        # print ("ListSplatNames = ", ListSplatNames)


        ##================================================================================================================================================
        ## analyze iso ratio file
        NewIsoRatioFile = open(LocalIsoRatioFileName, 'w')
        ListOfReplacedNames = []
        for line in LocalIsoRatioFileContent:                                               ## loop over all lines of the iso ratio file
            StrippedLine = line.strip()                                                     ## remove leading and trailing blanks


            ## remove comments
            i = StrippedLine.find("%")
            if (i > (-1)):
                StrippedLine = StrippedLine[:i].strip()


            ## analyze current line of iso ratio file
            if (StrippedLine != ""):

                # Debug:
                # print ("\n\n\n\nStrippedLine = >>", StrippedLine, "<<")


                ## replace tab by blank character
                StrippedLine = StrippedLine.replace("\t", " ")


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## check if current line contains a molecule name from VAMDC or Splatalogue
                if (SplatFlag):                                                             ## check if translation table is defined
                    FoundNameFlag = False
                    LocalReplacementList = []
                    LocalSplittedLine = StrippedLine.split("  ")                            ## make a copy of current line
                    NewLine = ""
                    for LocalElement in LocalSplittedLine:
                        LocalElementStripped = LocalElement.strip()

                        # Debug:
                        # print ("LocalElementStripped = ", LocalElementStripped)

                        if (LocalElementStripped != ""):
                            if (LocalElementStripped in ListSplatNames):                    ## does current element describe a Splatalogue name?
                                SplatIndex = ListSplatNames.index(LocalElementStripped)     ## check, if Splatalogue name is included in current line
                                if (SplatIndex > (-1)):                                     ## we've found a Splatalogue name in the current line
                                    VAMDCName = ListVAMDCNames[SplatIndex]
                                    LocalReplacementList.append([LocalElementStripped, VAMDCName])
                                    LocalElementStripped = LocalElementStripped.replace(LocalElementStripped, VAMDCName)
                                NewLine += "  {:<45s}".format(LocalElementStripped.strip())

                            elif (LocalElementStripped in ListVAMDCNames):
                                NewLine += "  {:<45s}".format(LocalElementStripped.strip())
                            else:
                                NewLine += "  " + LocalElementStripped.strip()
                    if (NewLine != ""):
                        NewIsoRatioFile.write(NewLine + "\n")

                    # Debug:
                    # print ("\n\nStrippedLine = >>" + StrippedLine + "<<")
                    # print ("   NewLine = >>" + NewLine + "<<")
                else:
                    NewIsoRatioFile.write(line)
            else:
                NewIsoRatioFile.write(line)


        ## close file
        NewIsoRatioFile.close()


    ## print all replaced molecule names
    if (printFlag and ListOfReplacedNames != []):
        print ("\n\nFound Splatalogue name(s) in iso ratio file. Replaced the following name(s):")
        for LocalReplacedNames in ListOfReplacedNames:
            print ("\t{:s}{:s}{:s} -> {:s}{:s}{:s}".format(chr(34), LocalReplacedNames[0], chr(34), chr(34), LocalReplacedNames[1], chr(34)))
        print ("\n")


    ## define return parameter
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check given molfit file
##
def CheckMolfitFile(MolfitsFileName, NameOfFunction, CurrentDir, CurrentJobDir, FreqMinList, FreqMaxList, GlobalvLSRList, RedshiftList, dbList):
    """

input parameters:
-----------------

    - MolfitsFileName:      path and name of molfit file

    - NameOfFunction:       name of the calling function

    - CurrentDir:           current directory

    - CurrentJobDir:        MAGIX job directory

    - FreqMinList:          list of min. frequencies of all ranges

    - FreqMaxList:          list of max. frequencies of all ranges

    - GlobalvLSRList:       list of v_LSR for each obs. data file

    - RedshiftList:         list of redshifts for each obs. data file

    - dbList:               path and name of user defined database file


output parameters:
------------------

    - ok                    status variable

    - dbFile:               path and name of database file

    - NewMolfitsFileName:   the absolute path and name of the molfit file within the MAGIX job directory

    - PureNameMolfitsFile:  the name of the molfit file without path
    """

    # Debug:
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("NameOfFunction = ", NameOfFunction)
    # print ("CurrentDir = ", CurrentDir)
    # print ("CurrentJobDir = ", CurrentJobDir)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("GlobalvLSRList = ", GlobalvLSRList)
    # print ("RedshiftList = ", RedshiftList)
    # print ("dbList = ", dbList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return variables
    ok = 0
    dbFile = ""
    NewMolfitsFileName = ""
    PureNameMolfitsFile = ""


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check existence of molfit file and copy molfit file to current working directory
    LocalPrintFlag = False
    NameOfFile = "molfit"
    NewMolfitsFileName = task_myXCLASS.CheckNCopy(MolfitsFileName, NameOfFunction, NameOfFile, CurrentDir, CurrentJobDir, LocalPrintFlag)
    if (NewMolfitsFileName == ""):                                                          ## did an error occurred
        ok = 1
        return (ok, dbFile, NewMolfitsFileName, PureNameMolfitsFile)
    else:
        PureNameMolfitsFile = NewMolfitsFileName.split("/")
        PureNameMolfitsFile = PureNameMolfitsFile[-1].strip()

    # Debug:
    # print ("\n\n\n>>NewMolfitsFileName = ", NewMolfitsFileName)
    # print ("PureNameMolfitsFile = ", PureNameMolfitsFile)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check molecules in database (replace Splatalogue names)
    dbFileName = dbList[0]
    LocalPrintFlag = True
    MolfitTranslateSplatalogueNames(NewMolfitsFileName, dbFileName, LocalPrintFlag)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## determine doppler shifted min. and max. frequencies
    DopplerFreqMinList = []
    DopplerFreqMaxList = []
    for rangeID in range(len(FreqMinList)):                                                 ## loop over all range definitions in the whole xml file
        DataFileIndex = int(FreqMinList[rangeID][0])                                        ## get exp data file index for each range definition
        RangeIndex = int(FreqMinList[rangeID][1])                                           ## get range index
        FreqMin = float(FreqMinList[rangeID][2])                                            ## continue and get min. freq.
        FreqMax = float(FreqMaxList[rangeID][2])                                            ## and max. freq.
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(DataFileIndex, RangeIndex, GlobalvLSRListIn = GlobalvLSRList, \
                                                                        Redshift_ListIn = RedshiftList)
        GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
        if (GlobalvLSR is None):
            GlobalvLSR = 0.0
        Redshift = ObsXMLParameterDictFile['Redshift']
        if (Redshift is None):
            Redshift = 0.0


        ## apply doppler shift for v_LSR != 0.0
        NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
        NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
        DopplerFreqMinList.append([DataFileIndex, RangeIndex, min(NewFreqMin, NewFreqMax)])
        DopplerFreqMaxList.append([DataFileIndex, RangeIndex, max(NewFreqMin, NewFreqMax)])

    # Debug:
    # print ("\n\nFreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("GlobalvLSRList = ", GlobalvLSRList)
    # print (">>DopplerFreqMinList = ", DopplerFreqMinList)
    # print ("->DopplerFreqMaxList = ", DopplerFreqMaxList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze molfit file


    ## print what you do
    print ("Analyze molfit file .. ", end = "", flush = True)


    ## get sql parameters from molfit file
    MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(NewMolfitsFileName)

    # Debug:
    # print ("\n\nMinNumTransitionsSQL = ", MinNumTransitionsSQL)
    # print ("MaxNumTransitionsSQL = ", MaxNumTransitionsSQL)
    # print ("MaxElowSQL = ", MaxElowSQL)
    # print ("MingASQL = ", MingASQL)
    # print ("OrderTransSQL = ", OrderTransSQL)
    # print ("dbFile = ", dbFile)


    ## analyze molfit file using myXCLASS routine
    MolfitMoleculeNames, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(NewMolfitsFileName)
    SortedListOfMolfitMolecules = sorted(set(MolfitMoleculeNames))

    # Debug:
    # print ("SortedListOfMolfitMolecules = ", SortedListOfMolfitMolecules)


    ## we're done
    print ("done!")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check molecules in database
    LocalPrintFlag = True
    DBMoleculeNames, dbFile = CheckMoleculesInDatabase(MolfitMoleculeNames, LocalPrintFlag, DopplerFreqMinList, DopplerFreqMaxList, dbList, \
                                                       MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)
    SortedListOfDBMoleculeNames = sorted(set(DBMoleculeNames))

    # Debug:
    # print ("DBMoleculeNames = ", DBMoleculeNames)
    # print ("dbFile = ", dbFile)
    # print ("SortedListOfDBMoleculeNames = ", SortedListOfDBMoleculeNames)


    ## print error message, if both lists are not identical
    DiffMolList = []
    for LocalMolfitMol in SortedListOfMolfitMolecules:
        LowerLocalMolfitMol = LocalMolfitMol.lower()
        if (not LowerLocalMolfitMol.startswith("cont-")):
            if (not LocalMolfitMol in SortedListOfDBMoleculeNames):
                DiffMolList.append(LocalMolfitMol)
    if (DiffMolList != []):
        print ("\n\n\t Error in XCLASS package, subroutine task_myXCLASSFit.CheckMolfitFile:")
        print ("\n\t\t The following molecule(s) defined in the molfit file has (have) no transitions within the given frequency ranges:\n")
        for LocalDiffMol in DiffMolList:
            print ("\t\t\t{:s}".format(LocalDiffMol.strip()))
        print ("\n\n\t\t Please correct the molfit file and restart function!")
        print ("\n\n")
        #ok = 1
        #return (ok, dbFile, NewMolfitsFileName, PureNameMolfitsFile)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write new molfit file including only those molecules which fulfills the SQL conditions
    WriteMoleculeFlag = False
    NewMolfitsFile = open(NewMolfitsFileName, 'w')
    MolfitHeaderLinesList = task_myXCLASS.WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)
    for line in MolfitHeaderLinesList:
        NewMolfitsFile.write(line)
    for MolID, LocalMolfitFileForEachMolecule in enumerate(MolfitFileForEachMolecule):
        LocalMoleculeNames = MolfitMoleculeNames[MolID]
        LowerLocalMoleculeNames = LocalMoleculeNames.lower()
        if ((LocalMoleculeNames in DBMoleculeNames) or LowerLocalMoleculeNames.startswith("cont-")):
            WriteMoleculeFlag = True
            for line in LocalMolfitFileForEachMolecule:
                NewMolfitsFile.write(line)
    NewMolfitsFile.close()

    # Debug:
    # print ("\n\nok = ", ok)
    # print ("dbFile = ", dbFile)
    # print ("NewMolfitsFileName = ", NewMolfitsFileName)
    # print ("PureNameMolfitsFile = ", PureNameMolfitsFile)
    #    os.system("ls -lah " + CurrentJobDir)


    ## define return variables
    return (ok, dbFile, NewMolfitsFileName, PureNameMolfitsFile)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check given iso ratio file
##
def CheckIsoRatioFile(IsoRatioFileName, NameOfFunction, CurrentDir, CurrentJobDir, FreqMinList, FreqMaxList, GlobalvLSRList, RedshiftList, dbList, \
                      LocalMolfitsFileName):
    """

input parameters:
-----------------

    - IsoRatioFileName:     path and name of iso ratio file

    - NameOfFunction:       name of the calling function

    - CurrentDir:           current directory

    - CurrentJobDir:        MAGIX job directory

    - FreqMinList:          list of min. frequencies of all ranges

    - FreqMaxList:          list of max. frequencies of all ranges

    - GlobalvLSRList:       list of v_LSR for each obs. data file

    - RedshiftList:         list of redshifts for each obs. data file

    - dbList:               path and name of user defined database file

    - LocalMolfitsFileName: path and name of molfit file


output parameters:
------------------

    - ok                    status variable

    - NewIsoRatioFileName:  absolute path and name of new iso ratio file within the current job directory
    """

    # Debug:
    # print ("IsoRatioFileName = ", IsoRatioFileName)
    # print ("NameOfFunction = ", NameOfFunction)
    # print ("CurrentDir = ", CurrentDir)
    # print ("CurrentJobDir = ", CurrentJobDir)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("GlobalvLSRList = ", GlobalvLSRList)
    # print ("RedshiftList = ", RedshiftList)
    # print ("dbList = ", dbList)
    # print ("LocalMolfitsFileName = ", LocalMolfitsFileName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return variables
    ok = 0
    dbFile = ""
    NewIsoRatioFileName = ""


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check existence of iso ratio file and copy iso ratio file to current working directory
    LocalPrintFlag = False
    NameOfFile = "iso ratio"
    NewIsoRatioFileName = task_myXCLASS.CheckNCopy(IsoRatioFileName, NameOfFunction, NameOfFile, CurrentDir, CurrentJobDir, LocalPrintFlag)
    if (NewIsoRatioFileName == ""):                                                         ## did an error occurred
        ok = 1
        return (ok, NewIsoRatioFileName)

    # Debug:
    # print ("NewIsoRatioFileName = ", NewIsoRatioFileName)
    #    os.system("more " + NewIsoRatioFileName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check molecules in database (replace Splatalogue names)
    LocalDBFileName = dbList[0]
    if (NameOfFunction in ["myXCLASSMapFit", "myXCLASSMapRedoFit", "LineIdentification", "LineID"]):
        LocalPrintFlag = False
    else:
        LocalPrintFlag = True
    IsoRatioTranslateSplatalogueNames(NewIsoRatioFileName, LocalDBFileName, LocalPrintFlag)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze iso ratio file
    LocalNewIsoRatioFileName = ""
    IsoRatioTable, Isotopologues, IsoMolecule = task_myXCLASS.ImportIsoRatioFile(NewIsoRatioFileName, LocalNewIsoRatioFileName)

    # Debug:
    # print ("\nIsoRatioTable = ", IsoRatioTable)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze molfit file
    LocalMolfitsFileName = LocalMolfitsFileName.strip()
    MoleculesInMolfitFile = []


    ## get molfit parameters
    if (LocalMolfitsFileName != ""):
        MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(LocalMolfitsFileName)


    ## get SQL parameters
    MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(LocalMolfitsFileName)

    # Debug:
    # print ("MoleculesInMolfitFile = ", MoleculesInMolfitFile)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if new file name is defined
    NewIsoRatioFile = open(NewIsoRatioFileName, 'w')                                    ## open file


    ## define header line
    NewLine = "{:<45s} {:<45s} {:25s} {:25s} {:25s}       {:s}".format("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:", \
                                                                       "additional file:")
    NewIsoRatioFile.write(NewLine + "\n")


    ## write new formatted iso ratio file
    for IsoLine in IsoRatioTable:
        LocalIsotopologue = IsoLine[0].strip()
        LocalIsotopologueLower = LocalIsotopologue.lower()
        LocalIsoMaster = IsoLine[1].strip()
        if ((LocalIsoMaster in MoleculesInMolfitFile) or MoleculesInMolfitFile == [] or LocalIsotopologueLower.startswith("global__")):
            LocalIsoRatio = IsoLine[2].strip()
            LocalIsoLowLimit = ""
            LocalIsoUpLimit = ""
            if (len(IsoLine) >= 5):
                LocalIsoLowLimit = IsoLine[3].strip()
                LocalIsoUpLimit = IsoLine[4].strip()


            ## write new line of iso ratio file
            NewLine = "{:<45s} {:<45s} {:25s} {:25s} {:25s}".format(LocalIsotopologue, LocalIsoMaster, LocalIsoRatio, LocalIsoLowLimit, \
                                                                    LocalIsoUpLimit)
            NewIsoRatioFile.write(NewLine + "\n")
    NewIsoRatioFile.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## determine doppler shifted min. and max. frequencies
    DopplerFreqMinList = []
    DopplerFreqMaxList = []
    for rangeID in range(len(FreqMinList)):                                                 ## loop over all range definitions in the whole xml file
        DataFileIndex = int(FreqMinList[rangeID][0])                                        ## get exp data file index for each range definition
        RangeIndex = int(FreqMinList[rangeID][1])                                           ## get range index
        FreqMin = float(FreqMinList[rangeID][2])                                            ## continue and get min. freq.
        FreqMax = float(FreqMaxList[rangeID][2])                                            ## and max. freq.
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(DataFileIndex, RangeIndex, GlobalvLSRListIn = GlobalvLSRList, \
                                                                       Redshift_ListIn = RedshiftList)
        GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
        if (GlobalvLSR is None):
            GlobalvLSR = 0.0
        Redshift = ObsXMLParameterDictFile['Redshift']
        if (Redshift is None):
            Redshift = 0.0


        ## apply doppler shift for v_LSR != 0.0
        NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
        NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
        DopplerFreqMinList.append([DataFileIndex, RangeIndex, min(NewFreqMin, NewFreqMax)])
        DopplerFreqMaxList.append([DataFileIndex, RangeIndex, max(NewFreqMin, NewFreqMax)])

    # Debug:
    # print ("DopplerFreqMinList = ", DopplerFreqMinList)
    # print ("DopplerFreqMaxList = ", DopplerFreqMaxList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check, if molecules described in iso ratio file are contained in used database file
    LocalPrintFlag = False
    dbList = []
    ListOfFoundMolecules = []
    NotFoundIsotopologuesList = []
    for LineIndex in range(len(Isotopologues)):
        LocalIsoMolecule = IsoMolecule[LineIndex]
        if (LocalIsoMolecule in ListOfFoundMolecules):


            ## check, if molecules defined in the iso ratio file are included in the current database
            LocalIsotopologues = Isotopologues[LineIndex]
            if (LocalIsotopologues.find("GLOBAL__") == (-1)):
                FoundIsoRatioMolecules, dbFile = CheckMoleculesInDatabase([LocalIsotopologues], LocalPrintFlag, DopplerFreqMinList, \
                                                                           DopplerFreqMaxList, dbList, MinNumTransitionsSQL, \
                                                                           MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)
                if (FoundIsoRatioMolecules == []):
                    NotFoundIsotopologuesList.append(LocalIsotopologues)
                else:
                    ListOfFoundMolecules.append(LocalIsotopologues)


    ## print error message, if both lists are not identical
    NotFoundIsotopologuesList = []
    if (NotFoundIsotopologuesList != []):
        print ("\n\n\t Error in XCLASS package, subroutine task_myXCLASSFit.CheckIsoRatioFile:")
        print ("\n\t\t The following molecule(s) defined in the iso ratio file are not included in the database:\n")
        for LocalDiffMol in NotFoundIsotopologuesList:
            print ("\t\t\t{:s}".format(LocalDiffMol.strip()))
        print ("\n\n\t\t Please correct the iso ratio file and restart function!")
        print ("\n\n")
        ok = 1
        return (ok, NewIsoRatioFileName)

    # Debug:
    # print (" ")
    # print ("LocalIsoTableFileName = ", LocalIsoTableFileName)
    # print ("IsoMolecule = ", IsoMolecule)


    ## define return variables
    return (ok, NewIsoRatioFileName)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check user input parameter "experimentalData"
##
def CheckObsInput(ObsXMLFileName, \
                  ObsDataFileName, \
                  experimentalData, \
                  NameOfFunction, \
                  CurrentDir, \
                  JobDir, \
                  Noise = None, \
                  Threshold = None, \
                  FreqMin = None, \
                  FreqMax = None, \
                  t_back_flag = True, \
                  tBack = None,
                  tSlope = None, \
                  N_H = None, \
                  beta_dust = None, \
                  kappa_1300 = None, \
                  DustFileName = None, \
                  BackgroundFileName = None, \
                  ContPhenFuncID = None, \
                  ContPhenFuncParam1 = None, \
                  ContPhenFuncParam2 = None, \
                  ContPhenFuncParam3 = None, \
                  ContPhenFuncParam4 = None, \
                  ContPhenFuncParam5 = None, \
                  TelescopeSize = None, \
                  BMIN = None, \
                  BMAJ = None, \
                  BPA = None, \
                  Inter_Flag = None, \
                  vLSR = None, \
                  Redshift = None, \
                  DBFileName = None, \
                  NumModelPixelXX = None, \
                  NumModelPixelYY = None, \
                  LocalOverlapFlag = None, \
                  NoSubBeamFlag = None, \
                  EmAbsPATH = None, \
                  iso_flag = None,
                  IsoTableFileName = None, \
                  PrintFlag = True, \
                  CopyFlag = True):
    """

input parameters:
-----------------

    - ObsXMLFileName:       input parameter describing path and name of obs. xml file

    - ObsDataFileName:      input parameter describing path and name of obs. data file

    - experimentalData:     alias for ObsDataFileName

    - NameOfFunction:       name of function

    - CurrentDir:           current working directory

    - JobDir:               current job directory

    - Noise:                (optional) noise level, (default: None)

    - Threshold:            (optional) local threshold, (default: None)

    - FreqMin:              (optional) min. frequency, (default: None)

    - FreqMax:              (optional) max. frequency, (default: None)

    - t_back_flag:          (optional) defines, if background temperature and slope
                            describe continuum completely, (default: None)

    - tBack:                background temperature, (default: None)

    - tSlope:               temperature slope, (default: None)

    - N_H:                  hydrogen column density, (default: None)

    - beta_dust:            spectral index for dust, (default: None)

    - kappa_1300:           kappa for dust, (default: None)

    - DustFileName:         path and name of dust file, (default: None)

    - BackgroundFileName:   path and name of background file, (default: None)

    - ContPhenFuncID:       describes which phenomenologicalal function is used to
                            describe the continuum, (default: None)

    - ContPhenFuncParam1:   first parameter for phenomenological function
                            describing the continuum, (default: None)

    - ContPhenFuncParam2:   second parameter for phenomenological function
                            describing the continuum, (default: None)

    - ContPhenFuncParam3:   third parameter for phenomenological function
                            describing the continuum, (default: None)

    - ContPhenFuncParam4:   fourth parameter for phenomenological function
                            describing the continuum, (default: None)

    - ContPhenFuncParam5:   fifth parameter for phenomenological function
                            describing the continuum, (default: None)

    - TelescopeSize:        size of the telescope, (default: None)

    - BMIN:                 telescope parameter BMIN, (default: None)

    - BMAJ:                 telescope parameter BMAJ, (default: None)

    - BPA:                  telescope parameter BPA, (default: None)

    - Inter_Flag:           defines if observational data is single
                            dish ("False") or interferrometric data ("True"),
                            (default: None)

    - vLSR:                 local standard of rest v_LSR, (default: None)

    - Redshift:             red shift, (default: None)

    - DBFileName:           path and name of default database file,
                            (default: None)

    - NumModelPixelXX:      number of points along x-axis for sub-beam
                            description, (default: None)

    - NumModelPixelYY:      number of points along y-axis for sub-beam
                            description, (default: None)

    - LocalOverlapFlag:     local overlap flag, (default: None)

    - NoSubBeamFlag:        no sub-beam flag, (default: None)

    - EmAbsPATH:            path to emission and absorption functions,
                            (default: None)

    - iso_flag:             (optional) flag for usage of iso ratio file,
                            (default: None)

    - IsoTableFileName:     (optional) path and name of iso ratio file,
                            (default: None)

    - PrintFlag:            (optional) local print flag, (default: True)

    - CopyFlag:             (optional) flag indicating, that obs. data
                            file(s) are copied to current job directory,
                            (default: True)


output parameters:
------------------

    - NewObsXMLFileName:    new path and name of obs. xml file
    """

    # Debug:
    # print ("\n\nObsXMLFileName = ", ObsXMLFileName)
    # print ("ObsDataFileName = ", ObsDataFileName)
    # print ("experimentalData = ", experimentalData)
    # print ("NameOfFunction = ", NameOfFunction)
    # print ("CurrentDir = ", CurrentDir)
    # print ("JobDir = ", JobDir)
    # print ("tBack = ", tBack)
    # print ("tSlope = ", tSlope)
    # print ("N_H, beta_dust = ", N_H, beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("TelescopeSize = ", TelescopeSize)
    # print ("BMIN = ", BMIN)
    # print ("BMAJ = ", BMAJ)
    # print ("BPA = ", BPA)
    # print ("Inter_Flag = ", Inter_Flag)
    # print ("vLSR = ", vLSR)
    # print ("Redshift = ", Redshift)
    # print ("DBFileName = ", DBFileName)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)
    # print ("EmAbsPATH = ", EmAbsPATH)
    # print ("t_back_flag = ", t_back_flag)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("Noise = ", Noise)
    # print ("Threshold = ", Threshold)
    # print ("iso_flag = ", iso_flag)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("PrintFlag = ", PrintFlag)
    # print ("CopyFlag = ", CopyFlag)


    ## analyze input parameter "ObsXMLFileName"
    ObsXMLFileName = ObsXMLFileName.strip()
    NewObsXMLFileName = None
    if (ObsXMLFileName == "" and experimentalData is not None):
        if (type(experimentalData).__name__ == 'str'):
            if (experimentalData.lower().endswith(".xml")):
                ObsXMLFileName = experimentalData.strip()
    if (ObsXMLFileName != ""):
        if (not os.path.exists(ObsXMLFileName)):
            print ("\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
            print ("\tThe given path and name of the obs. xml file")
            print ("\t{:s}".format(chr(34) + ObsXMLFileName + chr(34)))
            print ("\tdoes not exist!")
            print ("\n\tPlease enter an existing path and name of an obs. xml file and start again!")
        else:
            NewObsXMLFileName = ObsXMLFileName


    ## analyze input parameters "ObsDataFileName", "experimentalData"
    else:
        if (ObsDataFileName is None):
            ObsDataFileName = experimentalData
        if (ObsDataFileName is None):
            i = [chr(34) + "ObsDataFileName" + chr(34),
                 chr(34) + "experimentalData" + chr(34)]
            print ("\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
            print ("\tThe given path and name of the obs. xml file")
            print ("\t{:s}".format(chr(34) + ObsXMLFileName + chr(34)))
            print ("\tis not defined and the parameters ({:s})".format(", ".join(i)))
            print ("\tdescribing the obs. data are also not defined!")
            print ("\n\tPlease check the call of the function and start again!")
        else:


            ## print what you do
            if (PrintFlag):
                print ("Analyze parameters in obs. xml file ..", flush = True)


            ## parameter "ObsDataFileName" describes path and name of obs. data file
            FitsFlag = False
            if (type(ObsDataFileName).__name__ == 'str'):                                   ## is ObsDataFileName a string ?
                ObsDataFileName = ObsDataFileName.strip()                                   ## remove leading and tailing blanks


                ## check existence of obs. data file and copy obs. data file to current working directory
                if (CopyFlag):
                    LocalPrintFlag = False
                    NameOfFile = "obs. data"
                    ASCIIDataFileName = task_myXCLASS.CheckNCopy(ObsDataFileName, NameOfFunction, NameOfFile, CurrentDir, JobDir, LocalPrintFlag)
                    if (ASCIIDataFileName == ""):                                           ## did an error occurred
                        return ObsXMLFileName

                    # Debug:
                    # print ("ASCIIDataFileName = ", ASCIIDataFileName)


                ## do not copy obs. data file and make path absolute
                else:
                    ASCIIDataFileName = ObsDataFileName
                    if (not ASCIIDataFileName.startswith("/")):
                        ASCIIDataFileName = CurrentDir + "/" + ASCIIDataFileName
                        ASCIIDataFileName = os.path.normpath(ASCIIDataFileName) + "/"


                ## import obs. data file
                LowerASCIIDataFileName = ASCIIDataFileName.lower()
                LowerASCIIDataFileName = LowerASCIIDataFileName.strip()
                if (LowerASCIIDataFileName.endswith(".fits")):
                    FitsFlag = True
                    from spectral_cube import SpectralCube                                  ## import spectral_cube package
                    import astropy.units as u
                    LocalCube = SpectralCube.read(ASCIIDataFileName)
                    LocalCube.allow_huge_operations = True
                    LocalCube = LocalCube.with_spectral_unit(u.MHz)
                else:
                    if (PrintFlag):
                        print ("Import ASCII data from ", ASCIIDataFileName, "..", end = ' ')
                    ObsDataFileName = numpy.loadtxt(ASCIIDataFileName, skiprows = 0)
                    if (PrintFlag):
                        print ("done!")


            ## is "ObsDataFileName" an array or a list
            elif (type(ObsDataFileName).__name__ == 'ndarray' or type(ObsDataFileName).__name__ == 'list'):
                if (type(ObsDataFileName).__name__ == 'list'):
                    ObsDataFileName = numpy.asarray(ObsDataFileName)
                if (len(ObsDataFileName[0]) != 2):                                         ## check, if array is a 2D array
                    print ("\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
                    print ("\tThe given parameter for the experimental data is an array, but it is not a 2D array containing frequencies and intensities!")
                    print ("\n\tPlease correct your input and redo {:s} function call!".format(NameOfFunction))
                    return ObsXMLFileName
                else:                                                                       ## write obs. data to temp. file
                    if (PrintFlag):
                        print ("\rWrite ASCII data to temp file ..", end = ' ')
                    ASCIIDataFileName = JobDir + "ASCIIdata.dat"
                    numpy.savetxt(ASCIIDataFileName, ObsDataFileName, delimiter = '\t')
                    if (PrintFlag):
                        print ("done!")


            ## determine min. and max. frequency and step size of frequency
            if (FreqMin is None):
                if (FitsFlag):
                    FreqMin = min(LocalCube.spectral_axis.value)
                else:
                    FreqMin = numpy.nanmin(ObsDataFileName[:,0])
            if (FreqMax is None):
                if (FitsFlag):
                    FreqMax = max(LocalCube.spectral_axis.value)
                else:
                    FreqMax = numpy.nanmax(ObsDataFileName[:,0])


            ## get number of data points
            if (FitsFlag):
                number_points = len(LocalCube.spectral_axis.value)
            else:
                number_points = len(ObsDataFileName[:,0])
            if (number_points <= 1):
                print ("\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
                print ("\tThe given experimental data is an array, but it contains only one data point!")
                print ("\n\tPlease correct your input and redo {:s} function call!".format(NameOfFunction))
                return ObsXMLFileName


            ## define min., max., and step frequency
            NumberRange = [[0, 1]]
            FreqStep = (FreqMax - FreqMin) / (number_points - 1)
            Threshold = [[0, 0, None]]
            NoiseLevel = [[0, 0, Noise]]
            ErrorYFlag = False
            NumberHeaderLines = 0
            SeparatorColumns = ""
            if (LocalOverlapFlag is None):
                LocalOverlapFlagList = [False]
            else:
                LocalOverlapFlagList = [LocalOverlapFlag]
            if (NoSubBeamFlag is None):
                NoSubBeamFlagList = [False]
            else:
                NoSubBeamFlagList = [NoSubBeamFlag]
            if (EmAbsPATH is None):
                EmAbsPATHList = [None]
            else:
                EmAbsPATH = EmAbsPATH.strip()
                if (not EmAbsPATH.startswith("/")):                                             ## make path absolute if given path is relative
                    EmAbsPATH = CurrentDir + EmAbsPATH
                EmAbsPATHList = [EmAbsPATH]
            NumModelPixelXXList = [NumModelPixelXX]
            NumModelPixelYYList = [NumModelPixelYY]
            nH_flag = False


            ## create obs. xml-file
            if (PrintFlag):
                print ("Creating obs. xml file .. ", end = "")
            NewObsXMLFileName = JobDir + "exp.xml"
            CreateExpXMLFile(JobDir, ASCIIDataFileName, NumberRange, FreqMin, FreqMax, FreqStep, t_back_flag, tBack, tSlope, \
                             nH_flag, N_H, beta_dust, kappa_1300, DustFileName, BackgroundFileName, \
                             ContPhenFuncID, ContPhenFuncParam1, ContPhenFuncParam2, ContPhenFuncParam3, \
                             ContPhenFuncParam4, ContPhenFuncParam5, Threshold, NoiseLevel, TelescopeSize, BMIN, BMAJ, BPA, Inter_Flag, \
                             vLSR, Redshift, ErrorYFlag, NumberHeaderLines, SeparatorColumns, iso_flag, IsoTableFileName, \
                             DBFileName, NumModelPixelXXList, NumModelPixelYYList, LocalOverlapFlagList, \
                             NoSubBeamFlagList, EmAbsPATHList)
            if (PrintFlag):
                print ("done!")


    ## define return parameter
    return NewObsXMLFileName
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check given dust file
##
def CheckDustFile(DustFileName, NameOfFunction, CurrentDir, CurrentJobDir, FreqMinList, FreqMaxList):
    """

input parameters:
-----------------

    - DustFileName:         path and name of dust file

    - NameOfFunction:       name of the calling function

    - CurrentDir:           current directory

    - CurrentJobDir:        MAGIX job directory

    - FreqMinList:          list of min. frequencies of all ranges

    - FreqMaxList:          list of max. frequencies of all ranges


output parameters:
------------------

    - ok                    status variable

    - NewDustFileNameList:  absolute path and name of new dust file(s) within the current job directory
    """

    # Debug:
    # print ("DustFileName = ", DustFileName)
    # print ("NameOfFunction = ", NameOfFunction)
    # print ("CurrentDir = ", CurrentDir)
    # print ("CurrentJobDir = ", CurrentJobDir)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return variables
    ok = 0
    NewDustFileNameList = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check dust file(s)
    ErrorDustFileName = []
    for j in range(len(DustFileName)):                                                      ## loop over all dust file names
        LocalDustFileObsID = int(DustFileName[j][0])                                        ## get obs. data file index for current dust file
        LocalDustFileRageID = DustFileName[j][1]                                            ## get range index of current dust file
        LocalDustFileName = DustFileName[j][2]                                              ## get path and name of current dust file
        NameOfFile = "dust"
        LocalPrintFlag = False


        ## copy dust file to local directory
        LowerLocalDustFileName = LocalDustFileName.lower()
        if (LowerLocalDustFileName != "" and (not LowerLocalDustFileName.endswith(".fits"))):
            NewDustFileName = task_myXCLASS.CheckNCopy(LocalDustFileName, NameOfFunction, NameOfFile, CurrentDir, CurrentJobDir, LocalPrintFlag)
            if (NewDustFileName == ""):                                                     ## did an error occurred
                ErrorDustFileName.append(LocalDustFileName.strip())
        else:
            NewDustFileName = LocalDustFileName


        ## add path and name of dust file to list
        NewDustFileName = NewDustFileName.strip()
        NewDustFileNameList.append([LocalDustFileObsID, LocalDustFileRageID, NewDustFileName])


        ## check dust file
        LowerNewDustFileName = NewDustFileName.lower()
        if (LowerNewDustFileName != "" and (not LowerNewDustFileName.endswith(".fits"))):


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## check, if current dust file describes complete frequency range


            ## import dust file
            LocalDustFileContentUnsorted = numpy.loadtxt(NewDustFileName, skiprows = 0)


            ## make sure, that dust file is described in ascending frequency order
            LocalDustFileContent = LocalDustFileContentUnsorted[LocalDustFileContentUnsorted[:, 0].argsort()]
            if (not numpy.array_equal(LocalDustFileContent, LocalDustFileContentUnsorted)):
                numpy.savetxt(NewDustFileName, LocalDustFileContent, delimiter = '\t')


            ## get min. and max. frequency
            MinFreqDustFile = numpy.nanmin(LocalDustFileContent[:, 0])
            MaxFreqDustFile = numpy.nanmax(LocalDustFileContent[:, 0])


            ## check, if all frequency ranges are covered by the dust file
            for iii in range(len(FreqMinList)):                                             ## loop over all ranges
                DataFileIndex = int(FreqMinList[iii][0])                                    ## get exp data file index for each range definition
                RangeIndex = int(FreqMinList[iii][1])                                       ## get range index
                if (DataFileIndex == LocalDustFileObsID and LocalDustFileRageID == RangeIndex):
                                                                                            ## are obs. data file and range id identical
                    FreqMin = float(FreqMinList[iii][2])
                    FreqMax = float(FreqMaxList[iii][2])
                    if (FreqMin < MinFreqDustFile or MaxFreqDustFile < FreqMax):
                        print ("\n\n\t Warning: The dust file {:s}".format(NewDustFileName))
                        print ("\t          of obs. data file {:d} and frequency range {:d}".format(LocalDustFileObsID + 1, RangeIndex + 1))
                        print ("\t          does not describe the complete frequency range")
                        print ("\t          from {:.5f} to {:.5f} MHz".format(FreqMin, FreqMax))
                        print ("\n\t          Extrapolation is used.")
                        print ("\n\n")


    ## are paths and names of all dust files well defined?
    if (ErrorDustFileName != []):
        print ("\n\n\nError in XCLASS package, function {:s}, subroutine CheckDustFile:".format(NameOfFunction))
        print ("\tThe path(s) and name(s) of the following dust file(s) defined in the obs. xml file is (are) incomplete:")
        for LocalErrorFileName in ErrorDustFileName:
            print ("\t\t{:s}{:s}{:s}".format(chr(34), LocalErrorFileName, chr(34)))
        print ("\n\tPlease note, if a relative path is specified, the path has to be defined relative to the")
        print ("\tcurrent working directory!")
        print ("\n\tPlease correct the definitions in the obs. xml file and redo function call!")
        ok = 1


    ## define return variables
    return (ok, NewDustFileNameList)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check given background file
##
def CheckBackgroundFile(BackgroundFileName, NameOfFunction, CurrentDir, CurrentJobDir, FreqMinList, FreqMaxList, CopyBackgroundFileFlag = False):
    """

input parameters:
-----------------

    - BackgroundFileName:           path and name of background file

    - NameOfFunction:               name of the calling function

    - CurrentDir:                   current directory

    - CurrentJobDir:                MAGIX job directory

    - FreqMinList:                  list of min. frequencies of all ranges

    - FreqMaxList:                  list of max. frequencies of all ranges

    - CopyBackgroundFileFlag:       (optional) flag indicating if background file is copied to local directory, (default: False)


output parameters:
------------------

    - ok                            status variable

    - NewBackgroundFileNameList:    absolute path and name of new background file(s) within the current job directory
    """

    # Debug:
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("NameOfFunction = ", NameOfFunction)
    # print ("CurrentDir = ", CurrentDir)
    # print ("CurrentJobDir = ", CurrentJobDir)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("CopyBackgroundFileFlag = ", CopyBackgroundFileFlag)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return variables
    ok = 0
    NewBackgroundFileNameList = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check background file(s)
    ErrorBackgroundFileName = []
    for j in range(len(BackgroundFileName)):                                                ## loop over all Background file names
        LocalBackgroundFileObsID = int(BackgroundFileName[j][0])                            ## get obs. data file index for current Background file
        LocalBackgroundFileRageID = BackgroundFileName[j][1]                                ## get range index of current Background file
        LocalBackgroundFileName = BackgroundFileName[j][2]                                  ## get path and name of current Background file
        NameOfFile = "background"
        LocalPrintFlag = False


        ## check, if background file exists
        if (not os.path.exists(LocalBackgroundFileName)):
            if (os.path.exists(LocalBackgroundFileName + ".gz")):
                LocalBackgroundFileName += ".gz"


        ## copy background file to local directory
        if (CopyBackgroundFileFlag):
           LowerLocalBackgroundFileName = LocalBackgroundFileName.lower()
           if (LowerLocalBackgroundFileName != "" and (not LowerLocalBackgroundFileName.endswith(".fits"))):
               NewBackgroundFileName = task_myXCLASS.CheckNCopy(LocalBackgroundFileName, NameOfFunction, NameOfFile, CurrentDir, \
                                                                CurrentJobDir, LocalPrintFlag)
               if (NewBackgroundFileName == ""):                                            ## did an error occurred
                   ErrorBackgroundFileName.append(LocalBackgroundFileName.strip())
           else:
               NewBackgroundFileName = LocalBackgroundFileName
        else:
            NewBackgroundFileName = LocalBackgroundFileName


        ## add path and name of background file to list
        NewBackgroundFileName = NewBackgroundFileName.strip()
        NewBackgroundFileNameList.append([LocalBackgroundFileObsID, LocalBackgroundFileRageID, NewBackgroundFileName])


        ## check background file
        LowerNewBackgroundFileName = NewBackgroundFileName.lower()
        if (NewBackgroundFileName != "" and (not LowerNewBackgroundFileName.endswith(".fits"))):    ## check, if Background file is given


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## check, if current Background file describes complete frequency range


            ## import Background file
            LocalBackgroundFileContentUnsorted = numpy.loadtxt(NewBackgroundFileName, skiprows = 0)


            ## make sure, that background file is described in ascending frequency order
            LocalBackgroundFileContent = LocalBackgroundFileContentUnsorted[LocalBackgroundFileContentUnsorted[:, 0].argsort()]
            if (not numpy.array_equal(LocalBackgroundFileContentUnsorted, LocalBackgroundFileContent)):
                numpy.savetxt(NewBackgroundFileName, LocalBackgroundFileContent, delimiter = '\t')


            ## get min. and max. frequency
            MinFreqBackgroundFile = numpy.nanmin(LocalBackgroundFileContent[:, 0])
            MaxFreqBackgroundFile = numpy.nanmax(LocalBackgroundFileContent[:, 0])


            ## check, if all frequency ranges are covered by the background file
            for iii in range(len(FreqMinList)):                                         ## loop over all ranges
                DataFileIndex = int(FreqMinList[iii][0])                                ## get exp data file index for each range definition
                RangeIndex = int(FreqMinList[iii][1])                                   ## get range index
                if (DataFileIndex == LocalBackgroundFileObsID and LocalBackgroundFileRageID == RangeIndex):
                                                                                        ## are obs. data file and range id identical
                    FreqMin = float(FreqMinList[iii][2])
                    FreqMax = float(FreqMaxList[iii][2])
                    if (FreqMin < MinFreqBackgroundFile or MaxFreqBackgroundFile < FreqMax):
                        print ("\n\n\t Warning: The Background file")
                        print ("\t          {:s}".format(chr(34) + NewBackgroundFileName + chr(34)))
                        print ("\t          of obs. data file {:d} and frequency range {:d}".format(LocalBackgroundFileObsID + 1, RangeIndex + 1))
                        print ("\t          does not describe the complete frequency range")
                        print ("\t          from {:.3f} MHz to {:.3f} MHz\n".format(FreqMin, FreqMax))
                        print ("\t          Lowest frequency (background file):  {:.3f} MHz,".format(MinFreqBackgroundFile))
                        print ("\t          Highest frequency (background file): {:.3f} MHz.\n".format(MaxFreqBackgroundFile))
                        print ("\t          Extrapolation is used.\n\n")


    ## are paths and names of all Background files well defined?
    if (ErrorBackgroundFileName != []):
        print ("\n\n\nError in XCLASS package, function {:s}, subroutine CheckBackgroundFile:".format(NameOfFunction))
        print ("\tThe path(s) and name(s) of the following Background file(s) defined in the obs. xml file is (are) incomplete:")
        for LocalErrorFileName in ErrorBackgroundFileName:
            print ("\t\t{:s}{:s}{:s}".format(chr(34), LocalErrorFileName, chr(34)))
        print ("\n\tPlease note, if a relative path is specified, the path has to be defined relative to the")
        print ("\tcurrent working directory!")
        print ("\n\tPlease correct the definitions in the obs. xml file and redo function call!")
        ok = 1


    ## define return variables
    return (ok, NewBackgroundFileNameList)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## adjust paths in an obs. xml file
##
def AdjustObsXMLFile(NameOfFunction, ObsXMLFileName, CurrentDir, CurrentJobDir, printflag, WriteNewObsXMLFileFlag = True):
    """

input parameters:
-----------------

    - NameOfFunction:           name of calling function

    - ObsXMLFileName:           path and name of obs. xml file

    - CurrentDir:               current root directory

    - CurrentJobDir:            path of the current job directory

    - printflag:                print flag

    - WriteNewObsXMLFileFlag:   (optional) create a well-formatted obs. xml file (default: True)


output parameters:
------------------

    - NewObsXMLFileName:        path and name of modified (adjusted) obs. xml file

    - IsoRatioFileName:         path and name of iso ratio file

    - dbFileName:               path and name of used database file

    """

    # Debug:
    # print ("NameOfFunction = ", NameOfFunction)
    # print ("ObsXMLFileName = ", ObsXMLFileName)
    # print ("CurrentDir = ", CurrentDir)
    # print ("CurrentJobDir = ", CurrentJobDir)
    # print ("printflag = ", printflag)
    # print ("WriteNewObsXMLFileFlag = ", WriteNewObsXMLFileFlag)


    ## initialize return parameter
    NewObsXMLFileName = ""
    IsoRatioFileName = ""
    dbFileName = ""


    ##====================================================================================================================================================
    ## check existence of obs. xml file and copy obs. xml file to current working directory
    NameOfFile = "obs. xml"
    NewObsXMLFileName = task_myXCLASS.CheckNCopy(ObsXMLFileName, NameOfFunction, NameOfFile, CurrentDir, CurrentJobDir, printflag)
    if (NewObsXMLFileName == ""):                                                           ## did an error occurred
        return (NewObsXMLFileName, IsoRatioFileName, dbFileName)

    # Debug:
    # print ("LocalObsXMLFileName = ", LocalObsXMLFileName)


    ##====================================================================================================================================================
    ## check existence of obs. data files and copy obs. data file to current working directory
    ObsDataFileNames = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "FileNamesExpFiles")
    NumObsDataFiles = len(ObsDataFileNames)
    NewObsDataFileNameList = []
    ErrorObsDataFileNames = []
    for LocalObsDataFileID, LocalObsDataFile in enumerate(ObsDataFileNames):                ## loop over all obs. data file names
        LocalObsDataFileName = LocalObsDataFile[1].strip()
        if (LocalObsDataFileName != "" and NameOfFunction != "myXCLASSMapFit" and NameOfFunction != "myXCLASSMapRedoFit"):
            NameOfFile = "obs. data"
            LocalPrintFlag = False
            NewObsDataFileName = task_myXCLASS.CheckNCopy(LocalObsDataFileName, NameOfFunction, NameOfFile, CurrentDir, CurrentJobDir, LocalPrintFlag)
            if (NewObsDataFileName == ""):                                                  ## did an error occurred
                ErrorObsDataFileNames.append(LocalObsDataFileName.strip())
            NewObsDataFileNameList.append([LocalObsDataFileID, NewObsDataFileName])         ## save name(s) in list


    ## are paths and names of all obs. data files well defined?
    if (ErrorObsDataFileNames != []):
        print ("\n\n\nError in XCLASS package, function {:s}, subroutine AdjustObsXMLFile:".format(NameOfFunction))
        print ("\tThe path(s) and name(s) of the following obs. data file(s) defined in the obs. xml file is (are) incomplete:")
        for LocalErrorFileName in ErrorObsDataFileNames:
            print ("\t\t{:s}{:s}{:s}".format(chr(34), LocalErrorFileName, chr(34)))
        print ("\n\tPlease note, if a relative path is specified, the path has to be defined relative to the")
        print ("\tcurrent working directory!")
        print ("\n\tPlease correct the definitions in the obs. xml file and redo function call!")
        NewObsXMLFileName = ""
        return (NewObsXMLFileName, IsoRatioFileName, dbFileName)


    ## update path definitions of obs. data file in obs. xml file
    if (NewObsDataFileNameList != []):
        task_MAGIX.WriteXMLtagNEW(NewObsXMLFileName, "FileNamesExpFiles", NewObsDataFileNameList)
    else:
        NewObsDataFileNameList = copy.deepcopy(ObsDataFileNames)

    # Debug:
    # print ("ObsDataFileNames = ", ObsDataFileNames)
    # print ("NewObsDataFileNameList = ", NewObsDataFileNameList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get path and name of user defined database file otherwise use default database file
    dbList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "dbFilename")
    if (len(dbList) > 0):
        LocalDBFileName = dbList[0].strip()
        if (LocalDBFileName.strip() != ""):
            if (LocalDBFileName[0] != "/"):                                                 ## make relative path absolute
                LocalDBFileName = CurrentDir + LocalDBFileName
            if not(os.path.exists(LocalDBFileName) or os.path.isfile(LocalDBFileName)):
                print ("\n\nError in XCLASS package, function task_myXCLASSFit.myXCLASSFit:")
                print ("\tThe given path and name of the database file " + LocalDBFileName)
                print ("\tdoes not exist!")
                print ("\n\tPlease note, if a relative path is specified, the path has to be defined relative to the")
                print ("\tcurrent working directory!")
                print ("\n\tPlease enter an existing path and name of a database file and restart function call!")
                NewObsXMLFileName = ""
                return (NewObsXMLFileName, IsoRatioFileName, dbFileName)


            ## modify experimental xml-file in MAGIX temp directory
            dbList = [LocalDBFileName]
        else:
            dbList = []
    else:
        dbList = []
    if (dbList == [] or dbList == [""]):
        DefaultDBFileName = task_myXCLASS.GetDefaultDBFile()
        dbList = [DefaultDBFileName]
    dbFileName = dbList[0]
    task_MAGIX.WriteXMLtagNEW(NewObsXMLFileName, "dbFilename", dbList)

    # Debug:
    # print ("dbList = ", dbList)


    ##====================================================================================================================================================
    ## check settings for global v_LSR parameter
    GlobalvLSRList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "GlobalvLSR")
    FreqMinList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "MinExpRange")
    FreqMaxList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "MaxExpRange")


    ##====================================================================================================================================================
    ## adjust path of dust file(s)
    DustFileName = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "DustFileName")
    if (DustFileName != []):


        ## check and adjust dust file(s)
        ok, NewDustFileNameList = CheckDustFile(DustFileName, NameOfFunction, CurrentDir, CurrentJobDir, FreqMinList, FreqMaxList)
        if (ok == 1):
            return (NewObsXMLFileName, IsoRatioFileName, dbFileName)


        ## update path definitions of dust file(s) in obs. xml file
        task_MAGIX.WriteXMLtagNEW(NewObsXMLFileName, "DustFileName", NewDustFileNameList)

        # Debug:
        # print ("NewDustFileNameList = ", NewDustFileNameList)


    ##====================================================================================================================================================
    ## adjust path of Background file(s)
    BackgroundFileName = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "BackgroundFileName")
    if (BackgroundFileName != []):


        ## check and adjust Background file(s)
        ok, NewBackgroundFileNameList = CheckBackgroundFile(BackgroundFileName, NameOfFunction, CurrentDir, CurrentJobDir, \
                                                            FreqMinList, FreqMaxList, CopyBackgroundFileFlag = True)
        if (ok == 1):
            return (NewObsXMLFileName, IsoRatioFileName, dbFileName)


        ## update path definitions of Background file(s) in obs. xml file
        task_MAGIX.WriteXMLtagNEW(NewObsXMLFileName, "BackgroundFileName", NewBackgroundFileNameList)

        # Debug:
        # print ("NewBackgroundFileNameList = ", NewBackgroundFileNameList)


    ##====================================================================================================================================================
    ## make path of files describing emission and absorption functions absolute
    EmAbsPATHList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "EmAbsPATH")
    if (EmAbsPATHList != []):
        EmAbsPATH = EmAbsPATHList[0].strip()
        if (not EmAbsPATH.startswith("/")):
            EmAbsPATH = CurrentDir + EmAbsPATH


            ## update path definitions of ems/abs file in obs. xml file
            EmAbsPATHList = [EmAbsPATH]
            task_MAGIX.WriteXMLtagNEW(NewObsXMLFileName, "EmAbsPATH", EmAbsPATHList)


    ##====================================================================================================================================================
    ## adjust path of iso ratio file


    ## get flag for iso file
    IsoFlag = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "Isotopologues")
    if (IsoFlag == []):
        IsoFlag = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "iso_flag")
    if (len(IsoFlag) > 0):
        IsoFlag = task_myXCLASSMapFit.CheckBool(IsoFlag[0])
    else:
        IsoFlag = False

    # Debug:
    # print ("\n\n\n\nIsoFlag = ", IsoFlag)


    ## get name of iso ratio file, check existence and copy to current working directory
    if (IsoFlag):
        LocalIsoFileName = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "IsoTableFileName")
        NameOfFile = "iso ratio"
        LocalPrintFlag = False
        NewIsoFileName = task_myXCLASS.CheckNCopy(LocalIsoFileName[0], NameOfFunction, NameOfFile, CurrentDir, CurrentJobDir, LocalPrintFlag)
        if (NewIsoFileName == ""):                                                          ## did an error occurred
            return (NewObsXMLFileName, IsoRatioFileName, dbFileName)


        ## modify experimental xml-file in MAGIX temp directory
        IsoRatioFileName = NewIsoFileName
        NewIsoFileName = [NewIsoFileName]
        task_MAGIX.WriteXMLtagNEW(NewObsXMLFileName, "IsoTableFileName", NewIsoFileName)

        # Debug:
        # print ("NewIsoFileName = ", NewIsoFileName)


    ## create new well formatted obs. xml file
    if (WriteNewObsXMLFileFlag):
        NumberRangeList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "NumberExpRanges")
        FreqStepList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "StepFrequency")
        ListOft_back_flag = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "t_back_flag")
        ListOfTBack = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "BackgroundTemperature")
        ListOfTSlope = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "TemperatureSlope")
        ListOfnH_flag = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "nH_flag")
        ListOfN_HList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "HydrogenColumnDensity")
        ListOfbeta_dustList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "DustBeta")
        ListOfkappa_1300List = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "Kappa")
        ListOfDustFileName = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "DustFileName")
        ListOfBackgroundFileName = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "BackgroundFileName")
        ContPhenFuncIDList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "ContPhenFuncID")
        ContPhenFuncParam1List = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "ContPhenFuncParam1")
        ContPhenFuncParam2List = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "ContPhenFuncParam2")
        ContPhenFuncParam3List = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "ContPhenFuncParam3")
        ContPhenFuncParam4List = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "ContPhenFuncParam4")
        ContPhenFuncParam5List = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "ContPhenFuncParam5")
        ListOfThresholds = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "Threshold")
        ListOfNoiseLevel = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "NoiseLevel")
        SizeTelescopePerSpectrum = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "TelescopeSize")
        BMINList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "BMIN")
        BMAJList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "BMAJ")
        BPAList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "BPA")
        RedshiftList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "Redshift")
        InterFlagPerSpectrum = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "Inter_Flag")
        ErrorYFlagList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "ErrorY")
        NumberHeaderLinesList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "NumberHeaderLines")
        SeparatorColumnsList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "SeparatorColumns")
        LocalIsoFileName = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "IsoTableFileName")
        NumModelPixelXXList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "NumModelPixelXX")
        NumModelPixelYYList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "NumModelPixelYY")
        LocalOverlapFlagList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "LocalOverlap_Flag")
        NoSubBeamFlagList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "NoSubBeam_Flag")
        EmAbsPATHList = task_MAGIX.GetXMLtagNEW(NewObsXMLFileName, "EmAbsPATH")


        ## create well formatted obs. xml file
        CreateExpXMLFile(NewObsXMLFileName, NewObsDataFileNameList, NumberRangeList, FreqMinList, FreqMaxList, FreqStepList, ListOft_back_flag, \
                         ListOfTBack, ListOfTSlope, ListOfnH_flag, ListOfN_HList, ListOfbeta_dustList, ListOfkappa_1300List, ListOfDustFileName, \
                         ListOfBackgroundFileName, ContPhenFuncIDList, \
                         ContPhenFuncParam1List, ContPhenFuncParam2List, ContPhenFuncParam3List, ContPhenFuncParam4List, ContPhenFuncParam5List, \
                         ListOfThresholds, ListOfNoiseLevel, SizeTelescopePerSpectrum, BMINList, BMAJList, BPAList, InterFlagPerSpectrum, \
                         GlobalvLSRList, RedshiftList, ErrorYFlagList, NumberHeaderLinesList, SeparatorColumnsList, IsoFlag, LocalIsoFileName, \
                         dbList, NumModelPixelXXList, NumModelPixelYYList, LocalOverlapFlagList, NoSubBeamFlagList, \
                         EmAbsPATHList)


    ## finished
    return (NewObsXMLFileName, IsoRatioFileName, dbFileName)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


#---------------------------------------------------------------------------------------------------------------------------------------------------------
##
## creates the experimental xml file in the current job directory
##
def CreateExpXMLFile(JobDir, ASCIIDataFileName, NumberRange, FreqMin, FreqMax, FreqStep, t_back_flag, tBack, tSlope, nH_flag, N_H, beta_dust, \
                     kappa_1300, DustFileName, BackgroundFileName, ContPhenFuncID, \
                     ContPhenFuncParam1, ContPhenFuncParam2, ContPhenFuncParam3, ContPhenFuncParam4, ContPhenFuncParam5, Threshold, NoiseLevel, \
                     TelescopeSize, BMIN, BMAJ, BPA, Inter_Flag, vLSR, Redshift, ErrorYFlag, NumberHeaderLines, SeparatorColumns, iso_flag, \
                     IsoTableFileName, dbFilename, NumModelPixelXX, NumModelPixelYY, LocalOverlapFlagList, NoSubBeamFlagList, \
                     EmAbsPATHList):
    """

input parameters:
-----------------

    - JobDir:               path of job directory

    - ASCIIDataFileName:    path and name of experimental ASCII file

    - NumberRange:          number of frequency ranges

    - FreqMin:              first frequency

    - FreqMax:              last frequency

    - FreqStep:             frequency step size

    - t_back_flag:          defines, if background temperature and slope describe continuum completely

    - tBack:                background temperature

    - tSlope:               temperature slope

    - nH_flag:              defines, if Hydrogen column density etc. is given by the xml file

    - N_H:                  Hydrogen column density

    - beta_dust:            beta for dust

    - kappa_1300:           kappa

    - DustFileName:         path and name of file describing dust contribution

    - BackgroundFileName:   path and name of file describing background

    - ContPhenFuncID:       function ID used for phenomenological description

    - ContPhenFuncParam1:   parameter 1 for phenomenological description

    - ContPhenFuncParam2:   parameter 2 for phenomenological description

    - ContPhenFuncParam3:   parameter 3 for phenomenological description

    - ContPhenFuncParam4:   parameter 4 for phenomenological description

    - ContPhenFuncParam5:   parameter 5 for phenomenological description

    - TelescopeSize:        size of the telescope

    - BMIN:                 telescope parameter BMIN

    - BMAJ:                 telescope parameter BMAJ

    - BPA:                  telescope parameter BPA

    - vLSR:                 local standard of rest v_LSR

    - Redshift:             redshift

    - Threshold:            threshold

    - NoiseLevel:           noise level

    - ErrorYFlag:           flag for weighting

    - NumberHeaderLines:    number of header lines

    - Inter_Flag:           defines if observational data is single dish ("False") or interferrometric data ("True")

    - iso_flag:             use isotopologues ("True"/"False"). If iso_flag is set to "False" the ratios are all set to 1 (default: "False")

    - IsoTableFileName:     name of iso ratio file

    - dbFilename:           path and name of the sqlite3 database file

    - NumModelPixelXX:      number of pixel for sub-beam modeling along x-direction

    - NumModelPixelYY:      number of pixel for sub-beam modeling along y-direction

    - LocalOverlapFlagList: flag indicating if local-overlap description is used

    - NoSubBeamFlagList:    flag indicating if sub-beam description is prevented or not

    - EmAbsPATHList:        path of files describing emission and absorption functions


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("\n\nJobDir = ", JobDir)
    # print ("ASCIIDataFileName = ", ASCIIDataFileName)
    # print ("NumberRange = ", NumberRange)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("FreqStep = ", FreqStep)
    # print ("t_back_flag = ", t_back_flag)
    # print ("tBack = ", tBack)
    # print ("tSlope = ", tSlope)
    # print ("nH_flag = ", nH_flag)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("Threshold = ", Threshold)
    # print ("NoiseLevel = ", NoiseLevel)
    # print ("TelescopeSize = ", TelescopeSize)
    # print ("BMIN = ", BMIN)
    # print ("BMAJ = ", BMAJ)
    # print ("BPA = ", BPA)
    # print ("vLSR = ", vLSR)
    # print ("Redshift = ", Redshift)
    # print ("Inter_Flag = ", Inter_Flag)
    # print ("ErrorYFlag = ", ErrorYFlag)
    # print ("NumberHeaderLines = ", NumberHeaderLines)
    # print ("SeparatorColumns = ", SeparatorColumns)
    # print ("iso_flag = ", iso_flag)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("dbFilename = ", dbFilename)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlagList = ", LocalOverlapFlagList)
    # print ("NoSubBeamFlagList = ", NoSubBeamFlagList)
    # print ("EmAbsPATHList = ", EmAbsPATHList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## prepare parameters


    ## define list of obs. data file names
    if (type(ASCIIDataFileName) in (tuple, list)):
        LocalObsDataFileNameList = copy.deepcopy(ASCIIDataFileName)
    else:
        LocalObsDataFileNameList = [[0, ASCIIDataFileName]]


    ## define list of range numbers
    if (type(NumberRange) in (tuple, list)):
        LocalNumberRangeList = copy.deepcopy(NumberRange)
    else:
        LocalNumberRangeList = [[0, 0, NumberRange]]


    ## define list of min. frequencies
    if (type(FreqMin) in (tuple, list)):
        LocalFreqMinList = copy.deepcopy(FreqMin)
    else:
        LocalFreqMinList = [[0, 0, FreqMin]]


    ## define list of max. frequencies
    if (type(FreqMax) in (tuple, list)):
        LocalFreqMaxList = copy.deepcopy(FreqMax)
    else:
        LocalFreqMaxList = [[0, 0, FreqMax]]


    ## define list of step frequencies
    if (type(FreqStep) in (tuple, list)):
        LocalFreqStepList = copy.deepcopy(FreqStep)
    else:
        LocalFreqStepList = [[0, 0, FreqStep]]


    ## define list of t_back_flag
    if (t_back_flag is None):
        t_back_flag = True
    if (type(t_back_flag) in (tuple, list)):
        Localt_back_flagList = copy.deepcopy(t_back_flag)
    else:
        Localt_back_flagList = [[0, 0, t_back_flag]]


    ## define list of background temperatures
    if (type(tBack) in (tuple, list)):
        LocaltBackList = copy.deepcopy(tBack)
    else:
        LocaltBackList = [[0, 0, tBack]]


    ## define list of temperatures slopes
    if (type(tSlope) in (tuple, list)):
        LocaltSlopeList = copy.deepcopy(tSlope)
    else:
        LocaltSlopeList = [[0, 0, tSlope]]


    ## define list of nH_flag
    if (nH_flag is None):
        nH_flag = False
    if (type(nH_flag) in (tuple, list)):
        LocalnH_flagList = copy.deepcopy(nH_flag)
    else:
        LocalnH_flagList = [[0, 0, nH_flag]]


    ## define list of N_H   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< has to be redefined !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    if (type(N_H) in (tuple, list)):
        LocalN_HList = copy.deepcopy(N_H)
    else:
        LocalN_HList = [[0, 0, N_H]]


    ## define list of beta_dust
    if (type(beta_dust) in (tuple, list)):
        Localbeta_dustList = copy.deepcopy(beta_dust)
    else:
        Localbeta_dustList = [[0, 0, beta_dust]]


    ## define list of kappa_1300
    if (type(kappa_1300) in (tuple, list)):
        Localkappa_1300List = copy.deepcopy(kappa_1300)
    else:
        Localkappa_1300List = [[0, 0, kappa_1300]]


    ## define list of Dust file name
    if (type(DustFileName) in (tuple, list)):
        LocalDustFileNameList = copy.deepcopy(DustFileName)
    else:
        LocalDustFileNameList = [[0, 0, DustFileName]]


    ## define list of background file name
    if (type(BackgroundFileName) in (tuple, list)):
        LocalBackgroundFileNameList = copy.deepcopy(BackgroundFileName)
    else:
        LocalBackgroundFileNameList = [[0, 0, BackgroundFileName]]


    ## define list of function IDs for phenomenological continuum
    if (type(ContPhenFuncID) in (tuple, list)):
        LocalContPhenFuncIDList = copy.deepcopy(ContPhenFuncID)
    else:
        LocalContPhenFuncIDList = [[0, 0, ContPhenFuncID]]


    ## define list of first parameters for phenomenological continuum
    if (type(ContPhenFuncParam1) in (tuple, list)):
        LocalContPhenFuncParam1List = copy.deepcopy(ContPhenFuncParam1)
    else:
        LocalContPhenFuncParam1List = [[0, 0, ContPhenFuncParam1]]


    ## define list of second parameters for phenomenological continuum
    if (type(ContPhenFuncParam2) in (tuple, list)):
        LocalContPhenFuncParam2List = copy.deepcopy(ContPhenFuncParam2)
    else:
        LocalContPhenFuncParam2List = [[0, 0, ContPhenFuncParam2]]


    ## define list of third parameters for phenomenological continuum
    if (type(ContPhenFuncParam3) in (tuple, list)):
        LocalContPhenFuncParam3List = copy.deepcopy(ContPhenFuncParam3)
    else:
        LocalContPhenFuncParam3List = [[0, 0, ContPhenFuncParam3]]


    ## define list of fourth parameters for phenomenological continuum
    if (type(ContPhenFuncParam4) in (tuple, list)):
        LocalContPhenFuncParam4List = copy.deepcopy(ContPhenFuncParam4)
    else:
        LocalContPhenFuncParam4List = [[0, 0, ContPhenFuncParam4]]


    ## define list of fifth parameters for phenomenological continuum
    if (type(ContPhenFuncParam5) in (tuple, list)):
        LocalContPhenFuncParam5List = copy.deepcopy(ContPhenFuncParam5)
    else:
        LocalContPhenFuncParam5List = [[0, 0, ContPhenFuncParam5]]


    ## define list of thresholds
    if (type(Threshold) in (tuple, list)):
        LocalThresholdList = copy.deepcopy(Threshold)
    else:
        LocalThresholdList = [[0, 0, Threshold]]


    ## define list of noise levels
    if (type(NoiseLevel) in (tuple, list)):
        LocalNoiseLevelList = copy.deepcopy(NoiseLevel)
    else:
        LocalNoiseLevelList = [[0, 0, NoiseLevel]]


    ## define list of v_LSR
    if (type(vLSR) in (tuple, list)):
        LocalvLSRList = copy.deepcopy(vLSR)
    else:
        LocalvLSRList = [[0, vLSR]]


    ## define list of telescope sizes
    if (type(TelescopeSize) in (tuple, list)):
        LocalTelescopeSizeList = copy.deepcopy(TelescopeSize)
    else:
        LocalTelescopeSizeList = [[0, TelescopeSize]]


    ## define list of BMIN
    if (type(BMIN) in (tuple, list)):
        LocalBMINList = copy.deepcopy(BMIN)
    else:
        LocalBMINList = [[0, BMIN]]


    ## define list of BMAJ
    if (type(BMAJ) in (tuple, list)):
        LocalBMAJList = copy.deepcopy(BMAJ)
    else:
        LocalBMAJList = [[0, BMAJ]]


    ## define list of BPA
    if (type(BPA) in (tuple, list)):
        LocalBPAList = copy.deepcopy(BPA)
    else:
        LocalBPAList = [[0, BPA]]


    ## define list of Redshift
    if (type(Redshift) in (tuple, list)):
        LocalRedshiftList = copy.deepcopy(Redshift)
    else:
        LocalRedshiftList = [[0, Redshift]]


    ## define list of Inter_Flag
    if (type(Inter_Flag) in (tuple, list)):
        LocalInter_FlagList = copy.deepcopy(Inter_Flag)
    else:
        LocalInter_FlagList = [[0, Inter_Flag]]


    ## define list of ErrorYFlag
    if (ErrorYFlag is None):
        ErrorYFlag = False
    if (type(ErrorYFlag) in (tuple, list)):
        LocalErrorYFlagList = copy.deepcopy(ErrorYFlag)
    else:
        LocalErrorYFlagList = [[0, ErrorYFlag]]


    ## define list of NumberHeaderLines
    if (NumberHeaderLines is None):
        NumberHeaderLines = ""
    if (type(NumberHeaderLines) in (tuple, list)):
        LocalNumberHeaderLinesList = copy.deepcopy(NumberHeaderLines)
    else:
        LocalNumberHeaderLinesList = [[0, NumberHeaderLines]]


    ## define list of SeparatorColumns
    if (SeparatorColumns is None):
        SeparatorColumns = ""
    if (type(SeparatorColumns) in (tuple, list)):
        LocalSeparatorColumnsList = copy.deepcopy(SeparatorColumns)
    else:
        LocalSeparatorColumnsList = [[0, SeparatorColumns]]


    ## check, if IsoTableFileName is a list
    if (type(IsoTableFileName) in (tuple, list)):
        if (IsoTableFileName != []):
            IsoTableFileName = IsoTableFileName[0].strip()
        else:
            IsoTableFileName = ""


    ## check, if dbFilename is a list
    if (type(dbFilename) in (tuple, list)):
        if (dbFilename != []):
            dbFilename = dbFilename[0].strip()
        else:
            dbFilename = ""

    # Debug:
    # print ("\n\nLocalObsDataFileNameList = ", LocalObsDataFileNameList)
    # print ("LocalTelescopeSizeList = ", LocalTelescopeSizeList)
    # print ("LocalBMINList = ", LocalBMINList)
    # print ("LocalBMAJList = ", LocalBMAJList)
    # print ("LocalBPAList = ", LocalBPAList)
    # print ("LocalRedshiftList = ", LocalRedshiftList)
    # print ("LocalvLSRList = ", LocalvLSRList)
    # print ("LocalInter_FlagList = ", LocalInter_FlagList)
    # print ("LocalErrorYFlagList = ", LocalErrorYFlagList)
    # print ("LocalNumberHeaderLinesList = ", LocalNumberHeaderLinesList)
    # print ("LocalSeparatorColumnsList = ", LocalSeparatorColumnsList)
    # print ("LocalNumberRangeList = ", LocalNumberRangeList)
    # print ("LocalFreqMinList = ", LocalFreqMinList)
    # print ("LocalFreqMaxList = ", LocalFreqMaxList)
    # print ("LocalFreqStepList = ", LocalFreqStepList)
    # print ("Localt_back_flagList = ", Localt_back_flagList)
    # print ("LocaltBackList = ", LocaltBackList)
    # print ("LocaltSlopeList = ", LocaltSlopeList)
    # print ("LocalnH_flagList = ", LocalnH_flagList)
    # print ("LocalN_HList = ", LocalN_HList)
    # print ("Localbeta_dustList = ", Localbeta_dustList)
    # print ("Localkappa_1300List = ", Localkappa_1300List)
    # print ("LocalDustFileNameList = ", LocalDustFileNameList)
    # print ("LocalBackgroundFileNameList =", LocalBackgroundFileNameList)
    # print ("LocalContPhenFuncIDList = ", LocalContPhenFuncIDList)
    # print ("LocalContPhenFuncParam1List = ", LocalContPhenFuncParam1List)
    # print ("LocalContPhenFuncParam2List = ", LocalContPhenFuncParam2List)
    # print ("LocalContPhenFuncParam3List = ", LocalContPhenFuncParam3List)
    # print ("LocalContPhenFuncParam4List = ", LocalContPhenFuncParam4List)
    # print ("LocalContPhenFuncParam5List = ", LocalContPhenFuncParam5List)
    # print ("LocalThresholdList = ", LocalThresholdList)
    # print ("LocalNoiseLevelList = ", LocalNoiseLevelList)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("dbFilename = ", dbFilename)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write new obs. xml file


    ## define name of new obs. xml file
    if (JobDir.endswith(".xml")):
        NewObsDataFileName = JobDir
    else:
        NewObsDataFileName = JobDir + "exp.xml"


    ## open new obs. xml file and add header lines
    NewObsXMLFile = open(NewObsDataFileName, 'w')
    NewObsXMLFile.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
    NewObsXMLFile.write("<ExpFiles>\n")


    ## set number of obs. data files
    NewObsXMLFile.write("\n")
    NewObsXMLFile.write("\n")
    NewObsXMLFile.write("    <!-- set number of obs. data files -->\n")
    NewObsXMLFile.write("    <NumberExpFiles>" + str(len(LocalObsDataFileNameList)) + "</NumberExpFiles>\n")


    ## loop over all obs. data files
    for LocalObsDataFileNameID, LocalObsDataFileName in enumerate(LocalObsDataFileNameList):


        ## define path and name of ith obs. data file
        StrippedLocalObsDataFileName = LocalObsDataFileName[1].strip()
        NewObsXMLFile.write("\n")
        NewObsXMLFile.write("\n")
        star = "*" * (len(StrippedLocalObsDataFileName) + 47 - 13)
        NewObsXMLFile.write("    <!-- " + star + " -->\n")
        NewObsXMLFile.write("    <!-- parameters for import of obs. data file " + str(LocalObsDataFileNameID + 1) + " -->\n")
        NewObsXMLFile.write("    <file>\n")
        NewObsXMLFile.write("\n")
        NewObsXMLFile.write("\n")
        NewObsXMLFile.write("        <!-- define path and name of obs. data file -->\n")
        NewObsXMLFile.write("        <FileNamesExpFiles>" + StrippedLocalObsDataFileName + "</FileNamesExpFiles>\n")


        ## define import filter
        NewObsXMLFile.write("\n")
        NewObsXMLFile.write("\n")
        NewObsXMLFile.write("        <!-- define kind of obs. data file -->\n")
        LowerStrippedLocalObsDataFileName = StrippedLocalObsDataFileName.lower()
        if (LowerStrippedLocalObsDataFileName.endswith(".fits")):
            NewObsXMLFile.write("        <ImportFilter>xclassFITS</ImportFilter>\n")
        else:
            NewObsXMLFile.write("        <ImportFilter>xclassASCII</ImportFilter>\n")


        ## define number of ranges for current obs. data file
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(LocalObsDataFileNameID, RangeIndex, NumberRangeListIn = LocalNumberRangeList)
        LocalNumRange = ObsXMLParameterDictFile['NumberFrequencyRanges']
        NewObsXMLFile.write("\n")
        NewObsXMLFile.write("\n")
        NewObsXMLFile.write("        <!-- define number of frequency ranges -->\n")
        NewObsXMLFile.write("        <NumberExpRanges>" + str(LocalNumRange) + "</NumberExpRanges>\n")


        ## loop over range parameters
        for RangeIndex in range(LocalNumRange):


            ## get parameters for current range
            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(LocalObsDataFileNameID, RangeIndex, FreqMinListIn = LocalFreqMinList, \
                                                                            FreqMaxListIn = LocalFreqMaxList, FreqStepListIn = LocalFreqStepList, \
                                                                            tBackFlagListIn = Localt_back_flagList, tBackListIn = LocaltBackList, \
                                                                            tSlopeListIn = LocaltSlopeList, nHFlagListIn = LocalnH_flagList, \
                                                                            N_HListIn = LocalN_HList, beta_dustListIn = Localbeta_dustList, \
                                                                            kappa_1300ListIn = Localkappa_1300List, \
                                                                            ContPhenFuncID_ListIn = LocalContPhenFuncIDList, \
                                                                            ContPhenFuncParam1_ListIn = LocalContPhenFuncParam1List, \
                                                                            ContPhenFuncParam2_ListIn = LocalContPhenFuncParam2List, \
                                                                            ContPhenFuncParam3_ListIn = LocalContPhenFuncParam3List, \
                                                                            ContPhenFuncParam4_ListIn = LocalContPhenFuncParam4List, \
                                                                            ContPhenFuncParam5_ListIn = LocalContPhenFuncParam5List, \
                                                                            ThresholdListIn = LocalThresholdList, NoiseListIn = LocalNoiseLevelList, \
                                                                            DustFileNameListIn = LocalDustFileNameList, \
                                                                            BackgroundFileNameListIn = LocalBackgroundFileNameList)
            LocalFreqMin = ObsXMLParameterDictRange['FreqMin']
            if (LocalFreqMin is not None):
                LocalFreqMax = ObsXMLParameterDictRange['FreqMax']
                LocalFreqStep = ObsXMLParameterDictRange['FreqStep']
                t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                LocaltBack = ObsXMLParameterDictRange['tBack']
                LocaltSlope = ObsXMLParameterDictRange['tSlope']
                nH_flag = ObsXMLParameterDictRange['nH_flag']
                LocalN_H = ObsXMLParameterDictRange['N_H']
                Localbeta_dust = ObsXMLParameterDictRange['beta_dust']
                Localkappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                LocalDustFileName = ObsXMLParameterDictRange['DustFileName']
                LocalBackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']
                LocalNoiseLevel = ObsXMLParameterDictRange['NoiseLevel']
                LocalThreshold = ObsXMLParameterDictRange['Threshold']

                # Debug:
                # print ("LocalFreqMin = ", LocalFreqMin)
                # print ("LocalFreqMax = ", LocalFreqMax)
                # print ("LocalFreqStep = ", LocalFreqStep)
                # print ("t_back_flag = ", t_back_flag)
                # print ("LocaltBack = ", LocaltBack)
                # print ("LocaltSlope = ", LocaltSlope)
                # print ("nH_flag = ", nH_flag)
                # print ("LocalN_H = ", LocalN_H)
                # print ("Localbeta_dust = ", Localbeta_dust)
                # print ("Localkappa_1300 = ", Localkappa_1300)
                # print ("LocalDustFileName = ", LocalDustFileName)
                # print ("LocalBackgroundFileName = ", LocalBackgroundFileName)
                # print ("LocalContPhenFuncID = ", LocalContPhenFuncID)
                # print ("LocalContPhenFuncParam1 = ", LocalContPhenFuncParam1)
                # print ("LocalContPhenFuncParam2 = ", LocalContPhenFuncParam2)
                # print ("LocalContPhenFuncParam3 = ", LocalContPhenFuncParam3)
                # print ("LocalContPhenFuncParam4 = ", LocalContPhenFuncParam4)
                # print ("LocalContPhenFuncParam5 = ", LocalContPhenFuncParam5)
                # print ("LocalNoiseLevel = ", LocalNoiseLevel)
                # print ("LocalThreshold = ", LocalThreshold)


                ## define min., max., and step frequency
                NewObsXMLFile.write("\n\n        <!-- define parameters for frequency range " + str(RangeIndex + 1) + " -->\n")
                NewObsXMLFile.write("        <FrequencyRange>\n")
                NewObsXMLFile.write("\n\n            <!-- define min., max., and step frequency -->\n")
                NewObsXMLFile.write("            <MinExpRange>" + str(LocalFreqMin) + "</MinExpRange>\n")
                NewObsXMLFile.write("            <MaxExpRange>" + str(LocalFreqMax) + "</MaxExpRange>\n")
                NewObsXMLFile.write("            <StepFrequency>" + str(LocalFreqStep) + "</StepFrequency>\n")


                ## define background temperature and slope
                if (LocaltBack is not None or LocaltSlope is not None):
                    NewObsXMLFile.write("\n\n            <!-- define background temperature and slope -->\n")
                    t_back_flag = task_myXCLASSMapFit.CheckBool(t_back_flag)
                    if (t_back_flag):
                        NewObsXMLFile.write("            <t_back_flag>True</t_back_flag>\n")
                    else:
                        NewObsXMLFile.write("            <t_back_flag>False</t_back_flag>\n")
                    NewObsXMLFile.write("            <BackgroundTemperature>" + str(LocaltBack) + "</BackgroundTemperature>\n")
                    NewObsXMLFile.write("            <TemperatureSlope>" + str(LocaltSlope) + "</TemperatureSlope>\n")


                ## define path and name of background file
                if (LocalBackgroundFileName is not None):
                    LocalBackgroundFileName = LocalBackgroundFileName.strip()
                    if (LocalBackgroundFileName != ""):
                        NewObsXMLFile.write("\n\n            <!-- define path and name of file describing background -->\n")
                        NewObsXMLFile.write("            <BackgroundFileName>" + LocalBackgroundFileName + "</BackgroundFileName>\n")


                ## define dust parameters
                if (LocalN_H is not None or Localbeta_dust is not None or Localkappa_1300 is not None):
                    NewObsXMLFile.write("\n\n            <!-- define hydrogen column density, beta, and kappa -->\n")
                    if (LocalN_H is not None):
                        NewObsXMLFile.write("            <HydrogenColumnDensity>" + str(LocalN_H) + "</HydrogenColumnDensity>\n")
                    if (Localbeta_dust is not None):
                        NewObsXMLFile.write("            <DustBeta>" + str(Localbeta_dust) + "</DustBeta>\n")
                    if (Localkappa_1300 is not None):
                        NewObsXMLFile.write("            <Kappa>" + str(Localkappa_1300) + "</Kappa>\n")
                if (LocalDustFileName is not None):
                    LocalDustFileName = LocalDustFileName.strip()
                    if (LocalDustFileName != ""):
                        NewObsXMLFile.write("\n\n            <!-- define path and name of file describing dust contribution -->\n")
                        NewObsXMLFile.write("            <DustFileName>" + LocalDustFileName + "</DustFileName>\n")


                ## define parameters for phenomenological continuum description
                if (LocalContPhenFuncID is not None):
                    NewObsXMLFile.write("\n\n            <!-- define parameters for phenomenological continuum description -->\n")
                    NewObsXMLFile.write("            <ContPhenFuncID>" + str(LocalContPhenFuncID) + "</ContPhenFuncID>\n")
                    if (LocalContPhenFuncParam1 is not None):
                        NewObsXMLFile.write("            <ContPhenFuncParam1>" + str(LocalContPhenFuncParam1) + "</ContPhenFuncParam1>\n")
                    if (LocalContPhenFuncParam2 is not None):
                        NewObsXMLFile.write("            <ContPhenFuncParam2>" + str(LocalContPhenFuncParam2) + "</ContPhenFuncParam2>\n")
                    if (LocalContPhenFuncParam3 is not None):
                        NewObsXMLFile.write("            <ContPhenFuncParam3>" + str(LocalContPhenFuncParam3) + "</ContPhenFuncParam3>\n")
                    if (LocalContPhenFuncParam4 is not None):
                        NewObsXMLFile.write("            <ContPhenFuncParam4>" + str(LocalContPhenFuncParam4) + "</ContPhenFuncParam4>\n")
                    if (LocalContPhenFuncParam5 is not None):
                        NewObsXMLFile.write("            <ContPhenFuncParam5>" + str(LocalContPhenFuncParam5) + "</ContPhenFuncParam5>\n")


                ## define additional parameters
                if (LocalNoiseLevel is not None):
                    NewObsXMLFile.write("\n\n            <!-- define noise level -->\n")
                    NewObsXMLFile.write("            <NoiseLevel>" + str(LocalNoiseLevel) + "</NoiseLevel>\n")
                if (LocalThreshold is not None):
                    NewObsXMLFile.write("\n\n            <!-- define threshold -->\n")
                    NewObsXMLFile.write("            <Threshold>" + str(LocalThreshold) + "</Threshold>\n")
                NewObsXMLFile.write("        </FrequencyRange>\n")


        ## get parameters for current range
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(LocalObsDataFileNameID, RangeIndex, TelescopeSizeListIn = LocalTelescopeSizeList, \
                                                                       BMIN_ListIn = LocalBMINList, BMAJ_ListIn = LocalBMAJList, \
                                                                       BPA_ListIn = LocalBPAList, GlobalvLSRListIn = LocalvLSRList, \
                                                                       Redshift_ListIn = LocalRedshiftList, InterFlagListIn = LocalInter_FlagList, \
                                                                       ErrorYFlagListIn = LocalErrorYFlagList, \
                                                                       NumberHeaderLinesListIn = LocalNumberHeaderLinesList, \
                                                                       SeparatorColumnsListIn = LocalSeparatorColumnsList)
        LocalTelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
        LocalBMIN = ObsXMLParameterDictFile['BMIN']
        LocalBMAJ = ObsXMLParameterDictFile['BMAJ']
        LocalBPA = ObsXMLParameterDictFile['BPA']
        LocalGlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
        LocalRedshift = ObsXMLParameterDictFile['Redshift']
        LocalInterFlag = ObsXMLParameterDictFile['InterFlag']
        LocalErrorY = ObsXMLParameterDictFile['ErrorY']
        LocalNumberHeaderLines = ObsXMLParameterDictFile['NumberHeaderLines']
        LocalSeparatorColumns = ObsXMLParameterDictFile['SeparatorColumns']

        # Debug:
        # print ("\n\nLocalTelescopeSize = ", LocalTelescopeSize)
        # print ("LocalBMIN = ", LocalBMIN)
        # print ("LocalBMAJ = ", LocalBMAJ)
        # print ("LocalBPA = ", LocalBPA)
        # print ("LocalGlobalvLSR = ", LocalGlobalvLSR)
        # print ("LocalRedshift = ", LocalRedshift)
        # print ("LocalInterFlag = ", LocalInterFlag)
        # print ("LocalErrorY = ", LocalErrorY)
        # print ("LocalNumberHeaderLines = ", LocalNumberHeaderLines)
        # print ("LocalSeparatorColumns = ", LocalSeparatorColumns)


        ## define size of telescope for circular beams
        if (LocalTelescopeSize is not None):
            NewObsXMLFile.write("\n\n        <!-- define size of telescope -->\n")
            NewObsXMLFile.write("        <TelescopeSize>" + str(LocalTelescopeSize) + "</TelescopeSize>\n")


        ## define BMIN parameters
        if (LocalBMIN is not None):
            NewObsXMLFile.write("\n\n        <!-- define BMIN parameters -->\n")
            NewObsXMLFile.write("        <BMIN>" + str(LocalBMIN) + "</BMIN>\n")


        ## define BMAJ parameters
        if (LocalBMAJ is not None):
            NewObsXMLFile.write("\n\n        <!-- define BMAJ parameters -->\n")
            NewObsXMLFile.write("        <BMAJ>" + str(LocalBMAJ) + "</BMAJ>\n")


        ## define BPA parameters
        if (LocalBPA is not None):
            NewObsXMLFile.write("\n\n        <!-- define BPA parameters -->\n")
            NewObsXMLFile.write("        <BPA>" + str(LocalBPA) + "</BPA>\n")


        ## define global v_lsr
        NewObsXMLFile.write("\n\n        <!-- define local standard of rest (vLSR) -->\n")
        if (LocalGlobalvLSR == "" or LocalGlobalvLSR is None):
            LocalGlobalvLSR = 0.0
        NewObsXMLFile.write("        <GlobalvLSR>" + str(LocalGlobalvLSR) + "</GlobalvLSR>\n")


        ## define Redshift parameters
        if (LocalRedshift is not None):
            NewObsXMLFile.write("\n\n        <!-- define redshift -->\n")
            NewObsXMLFile.write("        <Redshift>" + str(LocalRedshift) + "</Redshift>\n")


        ## define interferometer flag
        NewObsXMLFile.write("\n\n        <!-- define if interferrometric observation is modeled -->\n")
        if (LocalInterFlag is None):
            LocalInterFlag = False
        else:
            LocalInterFlag = task_myXCLASSMapFit.CheckBool(LocalInterFlag)
        if (LocalInterFlag):
            NewObsXMLFile.write("        <Inter_Flag>True</Inter_Flag>\n")
        else:
            NewObsXMLFile.write("        <Inter_Flag>False</Inter_Flag>\n")


        ## define parameters for obs. data file import
        if (not LowerStrippedLocalObsDataFileName.endswith(".fits")):
            NewObsXMLFile.write("\n\n        <!-- define parameters for ASCII file import -->\n")
            LocalErrorY = task_myXCLASSMapFit.CheckBool(LocalErrorY)
            if (LocalErrorY):
                NewObsXMLFile.write("        <ErrorY>True</ErrorY>\n")
            else:
                NewObsXMLFile.write("        <ErrorY>False</ErrorY>\n")
            if (LocalNumberHeaderLines == ""):
                NewObsXMLFile.write("        <NumberHeaderLines>0</NumberHeaderLines>\n")
            else:
                NewObsXMLFile.write("        <NumberHeaderLines>" + str(LocalNumberHeaderLines) + "</NumberHeaderLines>\n")
            if (LocalSeparatorColumns == ""):
                NewObsXMLFile.write("        <SeparatorColumns> </SeparatorColumns>\n")
            else:
                NewObsXMLFile.write("        <SeparatorColumns>" + LocalSeparatorColumns + "</SeparatorColumns>\n")
        NewObsXMLFile.write("    </file>\n")


    ## define iso flag and ratio file
    star = "*" * (len(IsoTableFileName) + 41 - 13)
    NewObsXMLFile.write("\n\n    <!-- " + star + " -->\n")
    NewObsXMLFile.write("    <!-- parameters for isotopologues -->\n")
    iso_flag = task_myXCLASSMapFit.CheckBool(iso_flag)
    if (IsoTableFileName.strip() != "" and iso_flag):
        NewObsXMLFile.write("    <iso_flag>True</iso_flag>\n")
        NewObsXMLFile.write("    <IsoTableFileName>" + IsoTableFileName + "</IsoTableFileName>\n")
    else:
        NewObsXMLFile.write("    <iso_flag>False</iso_flag>\n")
        NewObsXMLFile.write("    <IsoTableFileName></IsoTableFileName>\n")


    ## define path and name of database file
    if (dbFilename != "" and dbFilename is not None):
        star = "*" * (len(dbFilename) + 29 - 13)
        NewObsXMLFile.write("\n\n    <!-- " + star + " -->\n")
        NewObsXMLFile.write("    <!-- define path and name of database file -->\n")
        NewObsXMLFile.write("    <dbFilename>" + dbFilename + "</dbFilename>\n")


    ## define number of model pixels in x-and y-direction
    if (NumModelPixelXX is not None and NumModelPixelYY is not None):
        if (NumModelPixelXX != [] and NumModelPixelYY != []):
            if (NumModelPixelXX[0] is not None and NumModelPixelYY[0] is not None):
                star = "*" * 54
                NewObsXMLFile.write("\n\n    <!-- " + star + " -->\n")
                NewObsXMLFile.write("    <!-- define number of model pixels along x- and y-direction -->\n")
                NewObsXMLFile.write("    <NumModelPixelXX>" + str(NumModelPixelXX[0]) + "</NumModelPixelXX>\n")
                NewObsXMLFile.write("    <NumModelPixelYY>" + str(NumModelPixelYY[0]) + "</NumModelPixelYY>\n")


    ## add local-overlap flag
    LocalOverlapFlag = None
    if (LocalOverlapFlagList is not None):
        if (LocalOverlapFlagList != []):
            if (LocalOverlapFlagList[0] is not None):
                LocalOverlapFlag = LocalOverlapFlagList[0]
    else:
        LocalOverlapFlag = False
    if (LocalOverlapFlag is not None):
        star = "*" * 54
        NewObsXMLFile.write("\n\n    <!-- " + star + " -->\n")
        NewObsXMLFile.write("    <!-- take local-overlap into account -->\n")
        NewObsXMLFile.write("    <LocalOverlap_Flag>" + str(LocalOverlapFlag) + "</LocalOverlap_Flag>\n")


    ## add no-sub-beam flag
    NoSubBeamFlag = None
    if (NoSubBeamFlagList is not None):
        if (NoSubBeamFlagList != []):
            if (NoSubBeamFlagList[0] is not None):
                NoSubBeamFlag = NoSubBeamFlagList[0]
    else:
        NoSubBeamFlag = False
    if (NoSubBeamFlag is not None):
        star = "*" * 54
        NewObsXMLFile.write("\n\n    <!-- " + star + " -->\n")
        NewObsXMLFile.write("    <!-- prevent sub-beam description -->\n")
        NewObsXMLFile.write("    <NoSubBeam_Flag>" + str(NoSubBeamFlag) + "</NoSubBeam_Flag>\n")


    ## add path of files describing emission and absorption functions
    if (EmAbsPATHList is not None):
        if (EmAbsPATHList != []):
            if (EmAbsPATHList[0] is not None):
                star = "*" * 65
                NewObsXMLFile.write("\n\n    <!-- " + star + " -->\n")
                NewObsXMLFile.write("    <!-- define path of files describing emission and absorption functions -->\n")
                NewObsXMLFile.write("    <EmAbsPATH>" + str(EmAbsPATHList[0]) + "</EmAbsPATH>\n")


    ## close file
    NewObsXMLFile.write("</ExpFiles>\n")
    NewObsXMLFile.write("\n")
    NewObsXMLFile.close()


    ## we're done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## creates the LM control xml file in the current job directory
##
def CreateLMControlXMLFile(JobDir, NumberIteration, Optimizer = "magix"):
    """

input parameters:
-----------------

    - JobDir:               path of job directory

    - NumberIteration:      max. number of iterations

    - Optimizer:            (optional) used optimizer, (default: "magix")


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("JobDir = ", JobDir)
    # print ("NumberIteration = ", NumberIteration)
    # print ("Optimizer = ", Optimizer)


    ## define content of algorithm file
    LMcontrol = open(JobDir + "algorithm_control.xml", 'w')
    LMcontrol.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
    LMcontrol.write("<FitControl>\n")
    LMcontrol.write("    <!-- settings for fit process -->\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("    <!-- set number of used algorithms -->\n")
    LMcontrol.write("    <NumberOfFitAlgorithms>1</NumberOfFitAlgorithms>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("    <algorithm>\n")
    if (Optimizer in ["magix"]):
        LMcontrol.write("        <FitAlgorithm>Levenberg-Marquardt</FitAlgorithm>\n")
    else:
        LMcontrol.write("        <FitAlgorithm>trf</FitAlgorithm>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    if (Optimizer in ["magix"]):
        LMcontrol.write("        <!-- define method used for Levenberg-Marquardt -->\n")
        LMcontrol.write("        <MethodLM>nr</MethodLM>\n")
        LMcontrol.write("\n")
        LMcontrol.write("\n")
    LMcontrol.write("        <!-- define value of the variation (used for the calculation of the gradient (of chi^2 function)) in percent/100 -->\n")
    LMcontrol.write("        <VariationValue>1.e-3</VariationValue>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- set max. number of iterations -->\n")
    LMcontrol.write("        <number_iterations>" + str(NumberIteration) + "</number_iterations>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- set max. number of processors -->\n")
    LMcontrol.write("        <NumberProcessors>8</NumberProcessors>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- define path and name of host file -->\n")
    LMcontrol.write("        <MPIHostFileName> </MPIHostFileName>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- settings for chi^2 -->\n")
    LMcontrol.write("        <limit_of_chi2>1e-7</limit_of_chi2>\n")
    LMcontrol.write("        <RenormalizedChi2>yes</RenormalizedChi2>\n")
    LMcontrol.write("        <DeterminationChi2>default</DeterminationChi2>\n")
    LMcontrol.write("        <SaveChi2>yes</SaveChi2>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- set plot options -->\n")
    LMcontrol.write("        <PlotAxisX>Frequency [MHz]</PlotAxisX>\n")
    LMcontrol.write("        <PlotAxisY>T_mb [K]</PlotAxisY>\n")
    LMcontrol.write("        <PlotIteration>no</PlotIteration>\n")
    LMcontrol.write("    </algorithm>\n")
    LMcontrol.write("</FitControl>\n")
    LMcontrol.close()


    ## we're done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## creates the io-control xml file in the current job directory
##
def CreateIOControlXMLFile(JobDir, MAGIXExpXML, MolfitsFileName, MAGIXrootDir):
    """

input parameters:
-----------------

    - JobDir:               path of job directory

    - MAGIXExpXML:          path and name of an experimental xml file

    - MolfitsFileName:      path and name of a molfits file

    - MAGIXrootDir:         path of MAGIX root directory


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("JobDir = ", JobDir)
    # print ("MAGIXExpXML = ", MAGIXExpXML)
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("MAGIXrootDir = ", MAGIXrootDir)


    ## define content of i/o file
    iocontrol = open(JobDir + "io_control.xml", 'w')
    iocontrol.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
    iocontrol.write("<ioControl>\n")
    iocontrol.write("    <title>io control</title>\n")
    iocontrol.write("    <description>This xml-file contains the paths and the names of all files which are used by MAGIX</description>\n")
    iocontrol.write("    <PathFilename>\n")
    iocontrol.write("\n")
    iocontrol.write("        <experimental_data>\n")
    if (MAGIXExpXML != ""):
        iocontrol.write("            <filename>" + MAGIXExpXML + "</filename>\n")
    else:
        iocontrol.write("            <filename>" + JobDir + "exp.xml</filename>\n")
    iocontrol.write("            <description>path and name of the experimental file</description>\n")
    iocontrol.write("        </experimental_data>\n")
    iocontrol.write("\n")
    iocontrol.write("        <parameter_file>\n")
    iocontrol.write("            <filename>" + MolfitsFileName + "</filename>\n")
    iocontrol.write("            <description>path and name of the xml-file including the start values of each parameter</description>\n")
    iocontrol.write("        </parameter_file>\n")
    iocontrol.write("\n")
    iocontrol.write("        <fit_control>\n")
    iocontrol.write("            <filename>" + JobDir + "algorithm_control.xml</filename>\n")
    iocontrol.write("            <description>path and name of the xml file controlling the fitting process</description>\n")
    iocontrol.write("        </fit_control>\n")
    iocontrol.write("\n")
    iocontrol.write("        <fit_log>\n")
    iocontrol.write("            <filename>" + JobDir + "fit.log</filename>\n")
    iocontrol.write("            <description>path and name of log file describing the fitting process</description>\n")
    iocontrol.write("        </fit_log>\n")
    iocontrol.write("\n")
    iocontrol.write("        <model_description>\n")
    iocontrol.write("            <filename>" + MAGIXrootDir + "Fit-Functions/myXCLASS/xml/myNewXCLASS.xml</filename>\n")
    iocontrol.write("            <description>path and name of the xml-description of the input/output file of the fit function module</description>\n")
    iocontrol.write("        </model_description>\n")
    iocontrol.write("\n")
    iocontrol.write("    </PathFilename>\n")
    iocontrol.write("</ioControl>\n")
    iocontrol.write("\n")
    iocontrol.close()


    ## we're done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## reads in output files
##
def ReadInOutputFiles(JobDir, NameOfFunction, ConvertLogLinFlag, RestFreq, vLSR, printInfoFlag, ObsXMLFileName = None):
    """

input parameters:
-----------------

    - JobDir:               path of job directory

    - NameOfFunction:       name of current CASA function

    - ConvertLogLinFlag:    defines if column densities in the fitted molfit file(s) are converted back to linear values

    - RestFreq:             rest frequency in MHz

    - vLSR:                 velocity (local standard of rest) in km/s

    - printInfoFlag:        (True/False) print information to screen

    - ObsXMLFileName:       (optional) path and name of used obs. xml file (default: None)


output parameters:
------------------

    - OutputDict:           dictionary containing results of fit procedure:

                            - OutputDict["JobDir"]:                          path of job directory
                            - OutputDict["MolfitFileNames"]:                 list of molfit file names
                            - OutputDict["MolfitFilesContent"]:              list of molfit file content
                            - OutputDict["IsoRatioFileNames"]:               list of iso ratio file names
                            - OutputDict["IsoRatioFilesContent"]:            list of iso ratio file content
                            - OutputDict["ListOfSpectraFileNames"]:          list of spectra file names
                            - OutputDict["ListOfSpectraFilesContent"]:       list of optimized spectra
                            - OutputDict["BestMolfitFileName"]:              best molfit file name
                            - OutputDict["BestMolfitFileContent"]:           best molfit file content
                            - OutputDict["BestIsoRatioFileName"]:            best iso ratio file name
                            - OutputDict["BestIsoRatioFileContent"]:         best iso ratio file content
                            - OutputDict["BestSpectraFileNames"]:            path and name of file describing spectra
                            - OutputDict["BestSpectraFilesContent"]:         spectra of best fit
    """

    # Debug:
    # print ("JobDir = ", JobDir)
    # print ("NameOfFunction = ", NameOfFunction)
    # print ("ConvertLogLinFlag = ", ConvertLogLinFlag)
    # print ("RestFreq = ", RestFreq)
    # print ("vLSR = ", vLSR)
    # print ("printInfoFlag = ", printInfoFlag)
    # print ("ObsXMLFileName = ", ObsXMLFileName)


    ## initialize return parameters
    OutputDict = {}
    ListOfMolfitFileNames = []                                                              ## initialize list of molfit file names
    ListMolfitFileContent = []                                                              ## initialize list of molfit file content
    OptimizedIsoRatioFileNameList = []                                                      ## initialize list of iso ratio file names
    OptimizedIsoRatioFileContentList = []                                                   ## initialize list of iso ratio file content
    ListOfSpectraFileNames = []                                                             ## initialize list of spectra file names
    ListOfSpectraFilesContent = []                                                          ## initialize list of optimized spectra
    BestMolfitFileName = ""                                                                 ## best molfit file name
    BestMolfitFileContent = []                                                              ## best molfit file content
    BestIsoRatioFileName = ""                                                               ## best iso ratio file name
    BestIsoRatioFileContent = []                                                            ## best iso ratio file content
    BestSpectraFileNames = ""                                                               ## path and name of file describing spectra
    BestSpectraFilesContent = []                                                            ## spectra of best fit


    ## determine number of new output files
    listing = os.listdir(JobDir)
    listing.sort()
    NumOutFiles = 0                                                                         ## determine number of output files
    for files in listing:
        if (files.endswith(".out.molfit")):
            NumOutFiles += 1
    if (NumOutFiles == 0):                                                                  ## check if new input files exist
        print ("\n\nError in XCLASS package, function " + NameOfFunction + ":")
        print ("\tNo path and name of the molfit file is defined!")
        print ("\n\tPlease enter an existing path and name of the molfit file and redo MAGIX call!")
    else:


        ## get name of iso ratio file
        ObsDataFileNamesIN = []
        IsoRatioFileNameIN = None
        if (ObsXMLFileName is not None):


            ## get path and name of obs. data files
            ObsDataFileNamesList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, 'FileNamesExpFiles')
            for element in ObsDataFileNamesList:
                ObsDataFileName = os.path.basename(element[1])
                ObsDataFileNamesIN.append(ObsDataFileName)


            ## get name of iso ratio file
            IsoFlag = False
            IsoFlagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "Isotopologues")
            if (IsoFlagList == []):
                IsoFlagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "iso_flag")
            if (len(IsoFlagList) > 0):
                LocalIsoFlag = task_myXCLASSMapFit.CheckBool(IsoFlagList[0])
                if (LocalIsoFlag):
                    IsoFlag = True
            IsoRatioFileName = ""
            if (IsoFlag):
                IsoRatioFileNameList = task_MAGIX.GetXMLtagNEW(ObsXMLFileName, "IsoTableFileName")
                if (IsoRatioFileNameList != []):
                    IsoRatioFileName = IsoRatioFileNameList[0].strip()
                    if (IsoRatioFileName != ""):
                        IsoRatioFileName = os.path.basename(IsoRatioFileName)               ## we need the name of the file only
                        IsoRatioFileNameIN, fileExtension = os.path.splitext(IsoRatioFileName)

        # Debug:
        # print ("IsoRatioFileNameIN = ", IsoRatioFileNameIN)
        # print ("ObsDataFileNamesIN = ", ObsDataFileNamesIN)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## change column densities back to linear scale and find iso ratio file
        count_files = 0
        ErrorInstanceXMLFileAll = []
        ErrorMolfitFileAll = []
        BestChi2Value = 1.e99
        FileNameBestChi2 = ""
        for files in listing:


            ## search for optimized molfit files, ending with ".out.molfit"
            if (files.endswith(".out.molfit")):
                OptimizedMolfitFileName = JobDir + files
                ListOfMolfitFileNames.append(OptimizedMolfitFileName)
                count_files += 1

                # Debug:
                # print ("NewInputFile = ", NewInputFile)
                # os.system("more " + NewInputFile)


                ## convert column and hydrogen column density (if given for each component) back to linear scale
                if (ConvertLogLinFlag):
                    LogLinearFlag = "linear"
                    task_myXCLASS.ConvertNtotToLogScale(OptimizedMolfitFileName, LogLinearFlag, ConverSQLElow = True)


                ## read in new output files
                f = open(OptimizedMolfitFileName)
                OptimizedMolfitFileContent = f.readlines()
                f.close()
                ListMolfitFileContent.append(OptimizedMolfitFileContent)


                ## save molfit files corresponding to error estimation algorithm
                if (files.find("ErrorEstim") > (-1)):
                    ErrorMolfitFileAll.append(files)


            ## reformat iso ratio files
            elif (files.startswith("myXClass_isoratios") and files.endswith(".out.input")):
                NewLocalIsoRatioFileName = JobDir + files
                OptimizedIsoRatioFileNameList.append(NewLocalIsoRatioFileName)
            elif (IsoRatioFileNameIN is not None):
                if (files.startswith(IsoRatioFileNameIN)):
                    NewLocalIsoRatioFileName = JobDir + files
                    OptimizedIsoRatioFileNameList.append(NewLocalIsoRatioFileName)


            ## get all fitted instance xml-file for error estimation
            if (files.startswith("parameters.ErrorEstim") and files.endswith(".out.xml")):
                ErrorInstanceXMLFileAll.append(files)


            ## get the lowest chi^2 value stored in log.chi2 file
            if (files.endswith(".log.chi2")):                                               ## we're interested only in the log.chi2 files
                Chi2LogFileName = JobDir + files                                            ## open log.chi2 file
                Chi2LogFileContent = numpy.loadtxt(Chi2LogFileName, skiprows = 1, usecols = (1))
                try:
                    LowestChi2Index = numpy.argmin(Chi2LogFileContent[:])
                    LowestChi2Value = Chi2LogFileContent[LowestChi2Index, 1]
                except:
                    LowestChi2Value = numpy.nan
                if (not numpy.isnan(LowestChi2Value)):
                    if (LowestChi2Value <= BestChi2Value):                                  ## check, if current chi2 value is the best one
                        BestChi2Value = LowestChi2Value                                     ## if yes, save chi2 value
                        FileNameBestChi2 = files                                            ## and corresponding file name

                    # Debug:
                    # print ("chi2Value = ", chi2Value)

        # Debug:
        # print ("\n\n\n\n\n\n\n\nErrorMolfitFileAll = ", ErrorMolfitFileAll)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## reformat some optimized input files


        ## reformat iso ratio files
        OptimizedIsoRatioFileContentList = []
        for NewLocalIsoRatioFileName in OptimizedIsoRatioFileNameList:
            IsoRatioTable, Isotopologues, IsoMolecule = task_myXCLASS.ImportIsoRatioFile(NewLocalIsoRatioFileName, NewLocalIsoRatioFileName)
            OptimizedIsoRatioFileContentList.append(IsoRatioTable)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## get phrase identifying the used algorithm for the best chi2 value
        BestAlgorithm = ""
        BestAlgorithm = FileNameBestChi2.replace(".log.chi2", "")
        BestAlgorithm = BestAlgorithm.replace("fit__", "")


        ## get molfit file and content which corresponds to the lowest chi^2 value
        for LocalMolfitFileNameID, LocalMolfitFileName in enumerate(ListOfMolfitFileNames):
            PureMolfitFileName = os.path.basename(LocalMolfitFileName)
            i = PureMolfitFileName.find(BestAlgorithm)
            if (i > (-1)):
                BestMolfitFileName = LocalMolfitFileName
                BestMolfitFileContent = ListMolfitFileContent[LocalMolfitFileNameID]


        ## get iso ratio file and content which corresponds to the lowest chi^2 value
        for LocalIsoRatioFileNameID, LocalIsoRatioFileName in enumerate(OptimizedIsoRatioFileNameList):
            PureIsoRatioFileName = os.path.basename(LocalIsoRatioFileName)
            i = PureIsoRatioFileName.find(BestAlgorithm)
            if (i > (-1)):
                BestIsoRatioFileName = LocalIsoRatioFileName
                BestIsoRatioFileContent = OptimizedIsoRatioFileContentList[LocalIsoRatioFileNameID]


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## create error molfit file
        if (ErrorInstanceXMLFileAll != []):
            for InstanceXMLFile in ErrorInstanceXMLFileAll:


                ## determine file name addon for current xml-file
                AddOn = ""
                i = InstanceXMLFile.find(".")
                if (i > (-1)):
                    AddOn = InstanceXMLFile[i + 1:]
                    i = AddOn.rfind(".out.xml")
                    if (i > (-1)):
                        AddOn = AddOn[:i].strip()

                # Debug:
                # print ("InstanceXMLFile = ", InstanceXMLFile)
                # print ("AddOn = ", AddOn)


                ## find molfit file with this add on
                if (AddOn.strip() != ""):
                    for mol in ErrorMolfitFileAll:
                        molMod = mol.replace("_MCMC__error-values","")
                        molMod = molMod.replace("_Fisher__error-values","")
                        molMod = molMod.replace("_INS__error-values","")
                        molMod = molMod.replace("_UltraNest__error-values","")
                        i = molMod.find(AddOn)
                        if (i > (-1)):

                            # Debug:
                            # print ("mol = ", mol)


                            ## create error molfit file
                            task_myXCLASS.CreateErrorMolfitFile(JobDir + InstanceXMLFile, JobDir + mol, printInfoFlag)
                            break


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read in output file(s)
        if (printInfoFlag == "true"):
            print ("\n\t Read in synthetic spectra from file(s) ..")


        ## determine number of output files
        NumOutFiles = 0
        for files in listing:
            if (files.endswith(".out.dat")):
                NumOutFiles += 1


        ## read in model function values
        count_files = 0


	## continue here, if obs. xml file is defined
        if (len(ObsDataFileNamesIN) > 0):
            for LocalObsDataFileNamesIN in ObsDataFileNamesIN:
                LocalSynthFileName = LocalObsDataFileNamesIN
                if (LocalSynthFileName.endswith(".dat")):
                    LocalSynthFileName = LocalSynthFileName.replace(".dat", "")
                for element in listing:
                    if (element.startswith(LocalSynthFileName) and element.endswith(".out.dat") \
                        and (not element.endswith(".chi2.out.dat"))):
                        SpectraFileName = JobDir + element

                        # Debug:
                        # print ("SpectraFileName = ", SpectraFileName)


                        ## read in optimized spectra function values
                        try:
                            LocalOptimizedSpectrum = numpy.loadtxt(SpectraFileName, skiprows = 0)
                        except:
                            LocalOptimizedSpectrum = None
                        if (LocalOptimizedSpectrum is not None):
                            count_files += 1
                            ListOfSpectraFileNames.append(SpectraFileName)

                            # Debug:
                            # print ("LocalOptimizedSpectrum = ", LocalOptimizedSpectrum)


                            ## add velocity axis if RestFreq != 0.0
                            if (RestFreq != 0.0):
                                if (printInfoFlag == "true"):
                                    print ("\tAdd velocity axis .. ", end = "")
                                FreqData = LocalOptimizedSpectrum[:, 0]
                                VelocityData = task_myXCLASS.ChangeToVelocitiy(FreqData, RestFreq, vLSR)


                                ## add velocity axis to LocalOptimizedSpectrum array
                                NewmyXCLASSspectrum = numpy.zeros((len(LocalOptimizedSpectrum), 3), dtype = numpy.float32)
                                NewmyXCLASSspectrum[:, 0] = LocalOptimizedSpectrum[:, 0]
                                NewmyXCLASSspectrum[:, 1] = VelocityData[:]
                                NewmyXCLASSspectrum[:, 2] = LocalOptimizedSpectrum[:, 1]
                                LocalOptimizedSpectrum = NewmyXCLASSspectrum


                            ## append current array with model function values to array OptimizedSpectra
                            ListOfSpectraFilesContent.append(LocalOptimizedSpectrum)


	## continue here, if no obs. xml file is defined
        else:
            for element in listing:
                if (element.endswith(".out.dat") and (not element.endswith(".chi2.out.dat"))):
                    SpectraFileName = JobDir + element

                    # Debug:
                    # print ("SpectraFileName = ", SpectraFileName)


                    ## read in optimized spectra function values
                    try:
                        LocalOptimizedSpectrum = numpy.loadtxt(SpectraFileName, skiprows = 0)
                    except:
                        LocalOptimizedSpectrum = None
                    if (LocalOptimizedSpectrum is not None):
                        count_files += 1
                        ListOfSpectraFileNames.append(SpectraFileName)

                        # Debug:
                        # print ("LocalOptimizedSpectrum = ", LocalOptimizedSpectrum)


                        ## add velocity axis if RestFreq != 0.0
                        if (RestFreq != 0.0):
                            if (printInfoFlag == "true"):
                                print ("\tAdd velocity axis .. ", end = "")
                            FreqData = LocalOptimizedSpectrum[:, 0]
                            VelocityData = task_myXCLASS.ChangeToVelocitiy(FreqData, RestFreq, vLSR)


                            ## add velocity axis to LocalOptimizedSpectrum array
                            NewmyXCLASSspectrum = numpy.zeros((len(LocalOptimizedSpectrum), 3), dtype = numpy.float32)
                            NewmyXCLASSspectrum[:, 0] = LocalOptimizedSpectrum[:, 0]
                            NewmyXCLASSspectrum[:, 1] = VelocityData[:]
                            NewmyXCLASSspectrum[:, 2] = LocalOptimizedSpectrum[:, 1]
                            LocalOptimizedSpectrum = NewmyXCLASSspectrum


                        ## append current array with model function values to array OptimizedSpectra
                        ListOfSpectraFilesContent.append(LocalOptimizedSpectrum)


        ## reduce dimension of output array if there is only one output file
        if (count_files == 1):
            ListOfSpectraFilesContent = ListOfSpectraFilesContent[0]
            BestSpectraFilesContent = [copy.deepcopy(ListOfSpectraFilesContent)]

            # Debug:
            # print ("ListOfSpectraFilesContent = ", ListOfSpectraFilesContent)


        ## get spectra of best plot
        else:
            BestSpectraFilesContent = []
            for LocalSpectraFileNameID, LocalSpectraFileName in enumerate(ListOfSpectraFileNames):
                PureSpectraFileName = os.path.basename(LocalSpectraFileName)
                i = PureSpectraFileName.find(BestAlgorithm)
                if (i > (-1)):
                    BestSpectraFileNames = LocalSpectraFileName
                    BestSpectraFilesContent.append(ListOfSpectraFilesContent[LocalSpectraFileNameID])


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## save parameters for output dictionary
    OutputDict["JobDir"] = JobDir
    OutputDict["MolfitFileNames"] = ListOfMolfitFileNames                               ## store list of molfit file names
    OutputDict["MolfitFilesContent"] = ListMolfitFileContent                            ## store list of molfit file content
    OutputDict["IsoRatioFileNames"] = OptimizedIsoRatioFileNameList                     ## store list of iso ratio file names
    OutputDict["IsoRatioFilesContent"] = OptimizedIsoRatioFileContentList               ## store list of iso ratio file content
    OutputDict["ListOfSpectraFileNames"] = ListOfSpectraFileNames                       ## store list of spectra file names
    OutputDict["ListOfSpectraFilesContent"] = ListOfSpectraFilesContent                 ## store list of optimized spectra
    OutputDict["BestMolfitFileName"] = BestMolfitFileName                               ## store best molfit file name
    OutputDict["BestMolfitFileContent"] = BestMolfitFileContent                         ## store best molfit file content
    OutputDict["BestIsoRatioFileName"] = BestIsoRatioFileName                           ## store best iso ratio file name
    OutputDict["BestIsoRatioFileContent"] = BestIsoRatioFileContent                     ## store best iso ratio file content
    OutputDict["BestSpectraFileNames"] = BestSpectraFileNames                           ## path and name of file describing spectra
    OutputDict["BestSpectraFilesContent"] = BestSpectraFilesContent                     ## spectra of best fit


    ## we're done
    return OutputDict
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## extract part of array between selected limits
##
def GetArrayPart(AxisArray, LowerLimit, UpperLimit):
    """

input parameters:
-----------------

    - AxisArray:                    array describing axis for selection

    - LowerLimit:                   lower limit

    - UpperLimit:                   upper limit



output parameters:
------------------

    - LowerIndex:                   index of lower limit

    - UpperIndex:                   index of upper limit
    """

    # Debug:
    # print ("AxisArray = ", AxisArray)
    # print ("LowerLimit = ", LowerLimit)
    # print ("UpperLimit = ", UpperLimit)


    ## restrict modeled spectra array to selected velocity range
    AxisArray = numpy.asarray(AxisArray)
    li = max(0, (numpy.abs(AxisArray - LowerLimit)).argmin())
    ui = min(len(AxisArray[:]) - 1, (numpy.abs(AxisArray - UpperLimit)).argmin())
    LowerIndex = min(li, ui)
    UpperIndex = max(li, ui)

    # Debug:
    # print ("LowerIndex = ", LowerIndex)
    # print ("UpperIndex = ", UpperIndex)


    ## declare return variable
    return (LowerIndex, UpperIndex)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## class for iterative fitting
##
class ItertiveFitting(object):


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize class 'ItertiveFitting'
    def __init__(self, UserMolfitsFileName, UserObsXMLFilename, UserAlgXMLFile, MoleculejobDir, \
                 ClusterFileName = "", NumberProcessors = 1, ServerList = [], ClusterFlag = False, AdditionMolfitFileContentFileName = "", \
                 RangevelLowLimit = -100.0, RangevelUpLimit = 100.0, IncludeAllMolFlag = True, FirstMoleculeOnlyFlag = False, \
                 LimitRangesFlag = False, dryrunFlag = False, RedRangeFlag = False, SetNewRangeFlag = False, LocalMinColumnDensityAbs = 1.e12, \
                 LocalMinColumnDensityEmis = 1.e12, SourceLow = 100.0, SourceUp = 100.0, TLow = 50.0, TUp = 50.0, NLow = 3.0, NUp = 3.0, \
                 VwidthLow = 5.0, VwidthUp = 5.0, vOffLow = 5.0, vOffUp = 5.0, DoFinalPlotFlag = False, UpdateFlag = False, NumRep = 1, \
                 PlotNum = 32, RemoveContFlag = False, AddMoleculeNamesToPlotFlag = False, backgroundFlag = True, BackgroundDir = "", \
                 EmsAbsFlag = False, FastCompBackFlag = False, DoMoleculePlotFlag = True, AddCompFileName = ""):
        """

    input parameters:
    -----------------

        - UserMolfitsFileName:                  path and name of molfit file

        - UserObsXMLFilename:                   path and name of obs. xml file

        - UserAlgXMLFile:                       path and name of algorithm xml file

        - MoleculejobDir:                       path containing

        - ClusterFileName:                      (optional) path and name of file describing cluster information (default: "")

        - NumberProcessors:                     (optional) total number of available cores (default: 1)

        - ServerList:                           (optional) list of all available nodes (default: [])

        - ClusterFlag:                          (optional) flag indicating usage of cluster (default: False)

        - AdditionMolfitFileContentFileName:    (optional) path and name of file describing additional information for molfit file (default: "")

        - RangevelLowLimit:                     (optional) lower velocity limit (default: -100.0)

        - RangevelUpLimit:                      (optional) upper velocity limit (default: 100.0)

        - IncludeAllMolFlag:                    (optional) flag to include all molecules in each single molecule fit (default: True)

        - FirstMoleculeOnlyFlag:                (optional) flag for taking only first mol. in given molfit file (default: False)

        - LimitRangesFlag:                      (optional) flag for limit parameter ranges in molfit file (default: False)

        - dryrunFlag:                           (optional) flag for executing script without fitting (default: False)

        - RedRangeFlag:                         (optional) flag for reducing frequency ranges in obs. xml file for each molecule (default: False)

        - SetNewRangeFlag:                      (optional) flag for allowing modifications on parameter ranges in molfit file (default: False)

        - LocalMinColumnDensityAbs:             (optional) minimal column density for absorption components (default: 1.e12)

        - LocalMinColumnDensityEmis:            (optional) minimal column density for emission components (default: 1.e12)

        - SourceLow:                            (optional) relative lower limit for source size (default: 100.0)

        - SourceUp:                             (optional) relative upper limit for source size (default: 100.0)

        - TLow:                                 (optional) relative lower limit for temperature (default: 50.0)

        - TUp:                                  (optional) relative upper limit for temperature (default: 50.0)

        - NLow:                                 (optional) relative lower limit for column density (default: 3.0)

        - NUp:                                  (optional) relative upper limit for column density (default: 3.0)

        - VwidthLow:                            (optional) relative lower limit for velocity width (default: 5.0)

        - VwidthUp:                             (optional) relative upper limit for velocity width (default: 5.0)

        - vOffLow:                              (optional) relative lower limit for velocity offset (default: 5.0)

        - vOffUp:                               (optional) relative upper limit for velocity offset (default: 5.0)

        - DoFinalPlotFlag:                      (optional) flag for final plot with all optimized molecules (default: False)

        - UpdateFlag:                           (optional) flag for using new optimized molfit parameter for further molecules
                                                           (only in serial mode, i.e. without clustering) (default: False)

        - NumRep:                               (optional) number of repetitions / iterations (default: 1)

        - PlotNum:                              (optional) number of subplots (default: 32)

        - RemoveContFlag:                       (optional) flag indicating subtraction of continuum (default: False)

        - AddMoleculeNamesToPlotFlag:           (optional) flag for adding molecule names to plot (default: False)

        - backgroundFlag:                       (optional) background flag (default: True)

        - BackgroundDir:                        (optional) directory for background (default: "")

        - EmsAbsFlag:                           (optional) flag for computing emission / absorption function (default: False)

        - FastCompBackFlag:                     (optional) flag for fast computing of emission / absorption function at the
                                                           beginning of each iteration for all molecules (default: False)

        - DoMoleculePlotFlag:                   (optional) flag for plotting all transitions for each molecule (default: True)

        - AddCompFileName:                      (optional) path and name of file describing additional components for each molecule molfit file


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("UserMolfitsFileName = ", UserMolfitsFileName)
        # print ("UserObsXMLFilename = ", UserObsXMLFilename)
        # print ("UserAlgXMLFile = ", UserAlgXMLFile)
        # print ("MoleculejobDir = ", MoleculejobDir)
        # print ("ClusterFileName = ", ClusterFileName)
        # print ("NumberProcessors = ", NumberProcessors)
        # print ("ServerList = ", ServerList)
        # print ("ClusterFlag = ", ClusterFlag)
        # print ("AdditionMolfitFileContentFileName = ", AdditionMolfitFileContentFileName)
        # print ("RangevelLowLimit = ", RangevelLowLimit)
        # print ("RangevelUpLimit = ", RangevelUpLimit)
        # print ("IncludeAllMolFlag = ", IncludeAllMolFlag)
        # print ("FirstMoleculeOnlyFlag = ", FirstMoleculeOnlyFlag)
        # print ("LimitRangesFlag = ", LimitRangesFlag)
        # print ("dryrunFlag = ", dryrunFlag)
        # print ("RedRangeFlag = ", RedRangeFlag)
        # print ("SetNewRangeFlag = ", SetNewRangeFlag)
        # print ("LocalMinColumnDensityAbs = ", LocalMinColumnDensityAbs)
        # print ("LocalMinColumnDensityEmis = ", LocalMinColumnDensityEmis)
        # print ("SourceLow = ", SourceLow)
        # print ("SourceUp = ", SourceUp)
        # print ("TLow = ", TLow)
        # print ("TUp = ", TUp)
        # print ("NLow = ", NLow)
        # print ("NUp = ", NUp)
        # print ("VwidthLow = ", VwidthLow)
        # print ("VwidthUp = ", VwidthUp)
        # print ("vOffLow = ", vOffLow)
        # print ("vOffUp = ", vOffUp)
        # print ("DoFinalPlotFlag = ", DoFinalPlotFlag)
        # print ("UpdateFlag = ", UpdateFlag)
        # print ("NumRep = ", NumRep)
        # print ("PlotNum = ", PlotNum)
        # print ("RemoveContFlag = ", RemoveContFlag)
        # print ("AddMoleculeNamesToPlotFlag = ", AddMoleculeNamesToPlotFlag)
        # print ("backgroundFlag = ", backgroundFlag)
        # print ("BackgroundDir = ", BackgroundDir)
        # print ("EmsAbsFlag = ", EmsAbsFlag)
        # print ("FastCompBackFlag = ", FastCompBackFlag)
        # print ("DoMoleculePlotFlag = ", DoMoleculePlotFlag)
        # print ("AddCompFileName = ", AddCompFileName)


        ## define some internal parameters
        self.DoMoleculePlotFlag = DoMoleculePlotFlag                                        ## set plot flag


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get cluster information
        ClusterFileName = ClusterFileName.strip()
        if (ClusterFileName != ""):
            NumberProcessors, ServerList, ClusterFlag = task_myXCLASSMapFit.ReadClusterdefFile(ClusterFileName, MoleculejobDir)
            OldServerList = ServerList
            ServerList = []
            for server in OldServerList:                                                    ## loop over all server in the cluster
                Node = server[0]                                                            ## get name of node
                NumCores = server[1]                                                        ## get number of cores
                RootPath = server[2].strip()                                                ## get path of xclass interface
                if (RootPath != ""):                                                        ## if path of xclass interface is defined
                    RootPath = RootPath + "/addons/MAGIX/"                                  ## define path of MAGIX
                for core in range(NumCores):                                                ## loop over all cores of the current node
                    ServerList.append([Node, NumCores, RootPath])

        # Debug:
        # print ("NumberProcessors = ", NumberProcessors)
        # print ("ServerList = ", ServerList)
        # print ("ClusterFlag = ", ClusterFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define ranges
        NewParameterRanges = [[SourceLow, SourceUp],
                              [TLow, TUp],
                              [NLow, NUp],
                              [VwidthLow, VwidthUp],
                              [vOffLow, vOffUp]]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## import additional components if defined
        self.AddCompContent = []
        AddCompFileName = AddCompFileName.strip()
        if (AddCompFileName != ""):
            AddCompFile = open(AddCompFileName)
            AddCompFileContent = AddCompFile.readlines()
            AddCompFile.close()
            for line in AddCompFileContent:
                StrippedLine = line.strip()
                if (StrippedLine != ""):
                    self.AddCompContent.append(line)

        # Debug:
        # print ("AddCompFileName = ", AddCompFileName)
        # print ("self.AddCompContent = ", self.AddCompContent)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## start iterative fitting
        self.SeparateFits(MoleculejobDir, UserMolfitsFileName, UserObsXMLFilename, UserAlgXMLFile, RangevelLowLimit, \
                          RangevelUpLimit, IncludeAllMolFlag, FirstMoleculeOnlyFlag, NewParameterRanges, LimitRangesFlag, \
                          dryrunFlag, RedRangeFlag, LocalMinColumnDensityAbs, LocalMinColumnDensityEmis, SetNewRangeFlag, \
                          DoFinalPlotFlag, NumRep, UpdateFlag, PlotNum, RemoveContFlag, AddMoleculeNamesToPlotFlag, EmsAbsFlag, \
                          backgroundFlag, BackgroundDir, NumberProcessors, ServerList, ClusterFlag, AdditionMolfitFileContentFileName, \
                          FastCompBackFlag)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## get final results
    ##
    def GetFinalResults(self):
        """

    input parameters:
    -----------------

        - None
        """


        ## we're done
        return self.FinalmolfitFileContent, self.FinalIsoRatioFileContents
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## call myXCLASSFit function
    ##
    def CallmyXCLASSFitFunction(self, AlgorithmXMLFileName, MolfitsFileName, LocalObsXMLFileName, NoSubBeamFlag, LocalWorkDir, InputDict):
        """

    input parameters:
    -----------------

        - AlgorithmXMLFileName: path and name of algorithm xml file

        - MolfitsFileName:      path and name of molfit file

        - LocalObsXMLFileName:  path and name of algorithm xml file

        - NoSubBeamFlag:        no sub-beam flag

        - LocalWorkDir:         path and name of current working directory

        - InputDict:            parameter directory


    output parameters:
    ------------------

        - OptimizedInputFile:   the contents of the molfit file containing the
                                parameters for the best fit.

        - OptimizedSpectra:     the model function values for each data point, which
                                correspond to the best fit.

        - JobDir:               absolute path of the job directory created for the current run.
        """

        # Debug:
        # print ("AlgorithmXMLFileName = ", AlgorithmXMLFileName)
        # print ("MolfitsFileName = ", MolfitsFileName)
        # print ("LocalObsXMLFileName = ", LocalObsXMLFileName)
        # print ("NoSubBeamFlag = ", NoSubBeamFlag)
        # print ("LocalWorkDir = ", LocalWorkDir)
        # print ("InputDict = ", InputDict)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize return parameter
        OptimizedInputFile = []
        OptimizedSpectra = []
        myXCLASSFitJobDir = ""


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## prepare call of myXCLASSFit function using functions of myXCLASSFit package


        ## define some internal parameters
        printflag = True
        NameOfFunction = "myXCLASSFit"
        CurrentDir = LocalWorkDir
        myXCLASSFitJobDir = LocalWorkDir
        RestFreq = 0.0
        vLSR = 0.0


        ## adjust paths in obs. xml file
        LocalObsXMLFileName, IsoRatioFileName, dbFileName = AdjustObsXMLFile(NameOfFunction, LocalObsXMLFileName, CurrentDir, \
                                                                             myXCLASSFitJobDir, printflag)
        if (LocalObsXMLFileName == ""):                                                     ## did an error occurred
            return (OptimizedInputFile, OptimizedSpectra, myXCLASSFitJobDir)

        # Debug:
        # print ("\n\nLocalObsXMLFileName = ", LocalObsXMLFileName)
        # print ("IsoRatioFileName = ", IsoRatioFileName)
        # print ("dbFileName = ", dbFileName)


        ## get all min. and max. frequencies for each range
        FreqMinList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "MinExpRange")
        FreqMaxList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "MaxExpRange")
        GlobalvLSRList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "GlobalvLSR")
        RedshiftList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "Redshift")


        ## get path and name of used database file
        dbList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "dbFilename")
        if (dbList == []):
            DefaultDBFileName = task_myXCLASS.GetDefaultDBFile()
            dbList = [DefaultDBFileName]


        ##================================================================================================================================================
        ## molfits file:
        MolfitsFileName = MolfitsFileName.strip()


        ## analyze molfit parameter: check path and name of the molfit file and copy molfit file to job directory
        ok, dbFilename, LocalMolfitsFileName, PureNameMolfitsFile = CheckMolfitFile(MolfitsFileName, NameOfFunction, CurrentDir, \
                                                                                    myXCLASSFitJobDir, FreqMinList, FreqMaxList, \
                                                                                    GlobalvLSRList, RedshiftList, dbList)
        if (ok == 1):                                                                       ## an error occurred
            return (OptimizedInputFile, OptimizedSpectra, myXCLASSFitJobDir)

        # Debug:
        # print ("ok = ", ok)
        # print ("LocalMolfitsFileName = ", LocalMolfitsFileName)
        # print ("PureNameMolfitsFile = ", PureNameMolfitsFile)


        ##================================================================================================================================================
        ## check molecules in iso ratio file (if defined)
        if (IsoRatioFileName != ""):


            ## analyze iso ratio file parameter: check path and name of the iso ratio file and copy new formatted iso ratio file to job directory
            ok, NewIsoRatioFileName = CheckIsoRatioFile(IsoRatioFileName, NameOfFunction, CurrentDir, myXCLASSFitJobDir, FreqMinList, \
                                                        FreqMaxList, GlobalvLSRList, RedshiftList, dbList, LocalMolfitsFileName)
            if (ok == 1):                                                                   ## an error occurred
                return (OptimizedInputFile, OptimizedSpectra, myXCLASSFitJobDir)


        ##================================================================================================================================================
        ## construct fit.xml file, if no path for another algorithm xml file is given


        ## adjust algorithm xml file (if defined)
        AlgorithmXMLFileName = AlgorithmXMLFileName.strip()
        if (AlgorithmXMLFileName != ""):


            ## check existence of algorithm xml file and copy algorithm xml file to current working directory
            LocalPrintFlag = False
            NameOfFile = "algorithm xml"
            AlgorithmXMLFileName = task_myXCLASS.CheckNCopy(AlgorithmXMLFileName, NameOfFunction, NameOfFile, CurrentDir, \
                                                            myXCLASSFitJobDir, LocalPrintFlag)
            if (AlgorithmXMLFileName == ""):                                                ## did an error occurred
                return (OptimizedInputFile, OptimizedSpectra, myXCLASSFitJobDir)

            # Debug:
            # print ("AlgorithmXMLFileName = ", AlgorithmXMLFileName)


            ## rename algorithm xml
            NewAlgorithmXMLFileName = myXCLASSFitJobDir + "algorithm_control.xml"
            if (AlgorithmXMLFileName != NewAlgorithmXMLFileName):
                cmdString = "mv " + AlgorithmXMLFileName + " " + NewAlgorithmXMLFileName
                os.system(cmdString)


        ##================================================================================================================================================
        ## construct io_control.xml file
        print ("Creating io_control file ..", end = "")
        MAGIXrootDir = task_myXCLASS.GetMAGIXRootDir()
        CreateIOControlXMLFile(myXCLASSFitJobDir, LocalObsXMLFileName, LocalMolfitsFileName, MAGIXrootDir)
        print ("done!")


        ##================================================================================================================================================
        ## compute on other node?
        DoSSHFlag = False
        ClusterFlag = InputDict['ClusterFlag']
        LocalNode = "localhost"
        LocalNodePath = ""
        if (ClusterFlag):
            LocalNode = InputDict['LocalNode']
            if (LocalNode != "localhost"):
                DoSSHFlag = True
                LocalNodePath = InputDict['LocalNodePath'].strip()                           ## get XCLASS root directory on other node
                if (LocalNodePath == ""):
                    LocalNodePath = MAGIXrootDir
                else:
                    LocalNodePath = LocalNodePath + "addons/MAGIX/"

        # Debug:
        # print ("\n\nDoSSHFlag = ", DoSSHFlag)
        # print ("ClusterFlag = ", ClusterFlag)
        # print ("LocalNode = ", LocalNode)
        # print ("LocalNodePath = ", LocalNodePath)
        # print ("MAGIXrootDir = ", MAGIXrootDir)
        # print ("CurrentDir = ", CurrentDir)


        ##================================================================================================================================================
        ## start MAGIX
        print ("\n\n\nStart MAGIX ..")
        task_MAGIX.SetMAGIXEnvironment(MAGIXrootDir)
        if (DoSSHFlag):
            cmdString = "ssh " + LocalNode + " " + chr(34) + "cd " + LocalNodePath + "; "
            cmdString += "nohup python3 magix_start.py --plotsaveonly --model=myxclass "
            cmdString += CurrentDir + "io_control.xml >> " + CurrentDir + "screen-output__MAGIX.out"
            cmdString += chr(34)
        else:
            cmdString = "cd " + MAGIXrootDir+ "; "
            cmdString += "python3 magix_start.py --plotsaveonly --model=myxclass "
            cmdString += CurrentDir + "io_control.xml >> " + CurrentDir + "screen-output__MAGIX.out"
        os.system(cmdString)                                                                ## start MAGIX

        # Debug:
        # print ("cmdString = ", cmdString)


        ## we're done
        return myXCLASSFitJobDir
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start different MAGIX instances via the command line on the cluster
    def StartClusterParallelization(self, NodeList, ClusterFlag, ParameterDict, LocalCompBackFlag):
        """

    input parameters:
    -----------------

        - NodeList:             list of server parameters

        - ClusterFlag:          cluster of smp calculation

        - ParameterDict:        parameter dictionary

        - LocalCompBackFlag:    compute emission and absorption functions


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("NodeList = ", NodeList)
        # print ("ClusterFlag = ", ClusterFlag)
        # print ("ParameterDict = ", ParameterDict)
        # print ("LocalCompBackFlag = ", LocalCompBackFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get some parameters from dictionary
        NumberOfMolecules = ParameterDict['NumberOfMolecules']
        NumberProcessors = ParameterDict['NumberProcessors']

        # Debug:
        # print ("\n\nNumberOfMolecules = ", NumberOfMolecules)
        # print ("NumberProcessors = ", NumberProcessors)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## perform single molecule fits
        for k in range(0, NumberOfMolecules, NumberProcessors):
            NumLocalWorker = min((NumberOfMolecules - k), NumberProcessors)

            # Debug:
            # print ("NumLocalWorker = ", NumLocalWorker)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## prepare start of workers
            thread_list = []
            for i in range(NumLocalWorker):                                                 ## loop over all workers
                MoleculeIndex = k + i

                # Debug:
                # print ("\tMoleculeIndex = ", MoleculeIndex)
                # print ("k, i, NumberProcessors, NumberOfMolecules, NumLocalWorker = ", k, i, NumberProcessors, NumberOfMolecules, NumLocalWorker)


                ## extract some parameters
                InputDict = copy.deepcopy(ParameterDict)
                AllParameters = InputDict['AllParameters']
                MoleculesInMolfitFile = InputDict['MoleculesInMolfitFile']
                LocalMolfitParameters = AllParameters[MoleculeIndex]
                LocalMoleculeName = MoleculesInMolfitFile[MoleculeIndex].strip()
                LowerLocalMoleculeName = LocalMoleculeName.lower()
                if (not LowerLocalMoleculeName.startswith("cont")):
                    InputDict['MoleculeIndex'] = MoleculeIndex
                    InputDict['LocalMolfitParameters'] = LocalMolfitParameters
                    InputDict['LocalMoleculeName'] = LocalMoleculeName


                    ## get node parameters
                    LocalNode = "localhost"
                    LocalNodePath = ""
                    if (ClusterFlag):
                        LocalNodeParam = NodeList[i]
                        LocalNode = LocalNodeParam[0]
                        try:
                            LocalNodePath = LocalNodeParam[2]
                        except:
                            LocalNodePath = ""
                    InputDict['LocalNode'] = LocalNode
                    InputDict['LocalNodePath'] = LocalNodePath


                    ## Set up the thread
                    if (LocalCompBackFlag):
                        t = multiprocessing.Process(target = self.FastCompBackground, args = (InputDict,))
                    else:
                        t = multiprocessing.Process(target = self.DoMoleculeFit, args = (InputDict,))


                    ## Sticks the thread in a list so that it remains accessible
                    thread_list.append(t)


            # Start all threads (workers)
            for thread in thread_list:
                thread.start()


            # This blocks the calling thread until the thread whose join() method is called is terminated.
            # From http://docs.python.org/2/library/threading.html#thread-objects
            for thread in thread_list:
                thread.join()


        ## declare return variable
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## compute background spectra in parallel
    ##
    def FastCompBackground(self, InputDict, BackgroundSpectrumFlag = False):
        """

    input parameters:
    -----------------

        - InputDict:                     parameter dictionary

        - BackgroundSpectrumFlag:       (optional) store spectrum (default: False)


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("InputDict = ", InputDict)
        # print ("BackgroundSpectrumFlag = ", BackgroundSpectrumFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get parameters from dictionary
        NumberExpRangesList = InputDict['NumberExpRangesList']
        FreqMinList = InputDict['FreqMinList']
        FreqMaxList = InputDict['FreqMaxList']
        FreqStepList = InputDict['FreqStepList']
        t_back_flagList = InputDict['t_back_flagList']
        tBackList = InputDict['tBackList']
        tSlopeList = InputDict['tSlopeList']
        N_HList = InputDict['N_HList']
        beta_dustList = InputDict['beta_dustList']
        kappa_1300List = InputDict['kappa_1300List']
        DustFileNameList = InputDict['DustFileNameList']
        ContPhenFuncIDList = InputDict['ContPhenFuncIDList']
        ContPhenFuncParam1List = InputDict['ContPhenFuncParam1List']
        ContPhenFuncParam2List = InputDict['ContPhenFuncParam2List']
        ContPhenFuncParam3List = InputDict['ContPhenFuncParam3List']
        ContPhenFuncParam4List = InputDict['ContPhenFuncParam4List']
        ContPhenFuncParam5List = InputDict['ContPhenFuncParam5List']
        TelescopeSizeList = InputDict['TelescopeSizeList']
        BMINList = InputDict['BMINList']
        BMAJList = InputDict['BMAJList']
        BPAList = InputDict['BPAList']
        Inter_FlagList = InputDict['Inter_FlagList']
        GlobalvLSRList = InputDict['GlobalvLSRList']
        RedshiftList = InputDict['RedshiftList']
        IsoTableFileName = InputDict['IsoTableFileName']
        IsoFlag = InputDict['IsoFlag']
        dbFilename = InputDict['dbFile']
        NumModelPixelXX = InputDict['NumModelPixelXX']
        NumModelPixelYY = InputDict['NumModelPixelYY']
        LocalOverlapFlag = InputDict['LocalOverlapFlag']
        ResultPath = InputDict['ResultPath']
        MolfitFileForEachMolecule = InputDict['MolfitFileForEachMolecule']
        MolfitHeaderLinesList = InputDict['MolfitHeaderLinesList']
        SQLParamArray = InputDict['SQLParamArray']
        NumberOfMolecules = InputDict['NumberOfMolecules']
        NumberProcessors = InputDict['NumberProcessors']
        LocalPrintFlag = InputDict['LocalPrintFlag']
        MoleculeIndex = InputDict['MoleculeIndex']
        LocalMoleculeName = InputDict['LocalMoleculeName']
        AdditionMolfitFileContent = InputDict['AdditionMolfitFileContent']


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print what you do
        if (LocalPrintFlag):
            print ("\tAnalyze (" + str(MoleculeIndex + 1) + "/" + str(NumberOfMolecules) + ") molecule: "+ chr(34) + LocalMoleculeName + chr(34))


        ## create a subdirectory for single molecule fit
        LocalMoleculeFileName = task_LineIdentification.MoleculeFileName(LocalMoleculeName)
        LocalFitDir = ResultPath + "emission+absorptions/" + LocalMoleculeFileName + "/"
        if (not os.path.exists(LocalFitDir)):
            cmdString = "rm -rf " + LocalFitDir + "; "
            cmdString += "mkdir -p " + LocalFitDir + "; "
            os.system(cmdString)


        ## redirect screen output to file
        if (NumberProcessors > 1):
            ScreenOutFileName = LocalFitDir + "screen-output__" + str(os.getpid()) + ".out"
            sys.stdout = open(ScreenOutFileName, "a", buffering = 1)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## construct molfit file
        BackgroundMolfitFileContent = []


        ## add molfit parameters of current molecule file
        for line in MolfitFileForEachMolecule[MoleculeIndex]:
            if (line.strip() != ""):
                BackgroundMolfitFileContent.append(line)


        ## add additional component(s) to molfit file
        for AddLine in self.AddCompContent:
            if (AddLine.strip() != ""):
                BackgroundMolfitFileContent.append(AddLine)


        ## add additional content to molfit file
        for AddLine in AdditionMolfitFileContent:
            if (AddLine.strip() != ""):
                BackgroundMolfitFileContent.append(AddLine)


        ## remove components with column densities below given thresholds
        BackgroundMolfitFileName = LocalFitDir + LocalMoleculeFileName + "__background.molfit"
        LocalMinColumnDensityEmis = 1.0
        LocalMinColumnDensityAbs = 1.0
        task_LineIdentification.CreateMolfitFileInNewFormat(BackgroundMolfitFileName, [BackgroundMolfitFileContent], LocalMinColumnDensityAbs, \
                                                            LocalMinColumnDensityEmis, SQLParameters = SQLParamArray)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## call myXCLASS function to compute background file


        ## check if at least on transition is within the given frequency ranges
        for element in TelescopeSizeList:                                                   ## loop over all obs. data files
            ObsDataFileIndex = element[0]


            ## get number of header lines and character separating columns from obs. xml file
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                           TelescopeSizeListIn = TelescopeSizeList, \
                                                                           BMIN_ListIn = BMINList, \
                                                                           BMAJ_ListIn = BMAJList, \
                                                                           BPA_ListIn = BPAList, \
                                                                           GlobalvLSRListIn = GlobalvLSRList, \
                                                                           Redshift_ListIn = RedshiftList, \
                                                                           InterFlagListIn = Inter_FlagList, \
                                                                           NumberRangeListIn = NumberExpRangesList)
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
            TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
            BMIN = ObsXMLParameterDictFile['BMIN']
            BMAJ = ObsXMLParameterDictFile['BMAJ']
            BPA = ObsXMLParameterDictFile['BPA']
            InterFlag = ObsXMLParameterDictFile['InterFlag']
            GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
            if (GlobalvLSR is None):
                GlobalvLSR = 0.0
            Redshift = ObsXMLParameterDictFile['Redshift']

            # Debug:
            # print ("\nNumberFrequencyRanges = ", NumberFrequencyRanges)
            # print ("TelescopeSize = ", TelescopeSize)
            # print ("BMIN = ", BMIN)
            # print ("BMAJ = ", BMAJ)
            # print ("BPA = ", BPA)
            # print ("InterFlag = ", InterFlag)
            # print ("GlobalvLSR = ", GlobalvLSR)
            # print ("Redshift = ", Redshift)


            ## analyze ranges
            for RangeIndex in range(NumberFrequencyRanges):                                    ## loop over all range definitions in the whole xml file


                ## get parameter for current frequency range
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = FreqMinList, \
                                                                                FreqMaxListIn = FreqMaxList, \
                                                                                FreqStepListIn = FreqStepList, \
                                                                                tBackFlagListIn = t_back_flagList, \
                                                                                tBackListIn = tBackList, \
                                                                                tSlopeListIn = tSlopeList, N_HListIn = N_HList, \
                                                                                beta_dustListIn = beta_dustList, \
                                                                                kappa_1300ListIn = kappa_1300List, \
                                                                                DustFileNameListIn = DustFileNameList, \
                                                                                ContPhenFuncID_ListIn = ContPhenFuncIDList, \
                                                                                ContPhenFuncParam1_ListIn = ContPhenFuncParam1List, \
                                                                                ContPhenFuncParam2_ListIn = ContPhenFuncParam2List, \
                                                                                ContPhenFuncParam3_ListIn = ContPhenFuncParam3List, \
                                                                                ContPhenFuncParam4_ListIn = ContPhenFuncParam4List, \
                                                                                ContPhenFuncParam5_ListIn = ContPhenFuncParam5List)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None):
                    FreqMax = ObsXMLParameterDictRange['FreqMax']
                    FreqStep = ObsXMLParameterDictRange['FreqStep']
                    t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                    tBack = ObsXMLParameterDictRange['tBack']
                    tSlope = ObsXMLParameterDictRange['tSlope']
                    N_H = ObsXMLParameterDictRange['N_H']
                    beta_dust = ObsXMLParameterDictRange['beta_dust']
                    kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                    if (N_H is None or beta_dust is None or kappa_1300 is None):
                        N_H = 0.0
                        beta_dust = 0.0
                        kappa_1300 = 0.0
                    DustFileName = ObsXMLParameterDictRange['DustFileName']
                    if (DustFileName is None):
                        DustFileName = ""
                    try:
                        BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                    except:
                        BackgroundFileName = None
                    if (BackgroundFileName is None):
                        BackgroundFileName = ""
                    LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                    LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                    LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                    LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                    LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                    LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']

                    # Debug:
                    # print ("FreqMin = ", FreqMin)
                    # print ("FreqMax = ", FreqMax)
                    # print ("FreqStep = ", FreqStep)
                    # print ("t_back_flag = ", t_back_flag)
                    # print ("tBack = ", tBack)
                    # print ("tSlope = ", tSlope)
                    # print ("N_H = ", N_H)
                    # print ("beta_dust = ", beta_dust)
                    # print ("kappa_1300 = ", kappa_1300)
                    # print ("DustFileName = ", DustFileName)
                    # print ("BackgroundFileName = ", BackgroundFileName)
                    # print ("LocalContPhenFuncID = ", LocalContPhenFuncID)
                    # print ("LocalContPhenFuncParam1 = ", LocalContPhenFuncParam1)
                    # print ("LocalContPhenFuncParam2 = ", LocalContPhenFuncParam2)
                    # print ("LocalContPhenFuncParam3 = ", LocalContPhenFuncParam3)
                    # print ("LocalContPhenFuncParam4 = ", LocalContPhenFuncParam4)
                    # print ("LocalContPhenFuncParam5 = ", LocalContPhenFuncParam5)


                    ## print what you do
                    if (LocalPrintFlag):
                        PrintString = "\n\t\t\tCompute background file for obs. data file %i and range %i .." % (ObsDataFileIndex + 1, RangeIndex + 1)
                        print (PrintString, end = ' ', flush = True)


                    ## call myXCLASS function
                    printFlag = False
                    NoSubBeamFlag = True
                    EmAbsFlag = True
                    modeldata, log, TransEnergies, IntOpt, JobDir = task_myXCLASS.myXCLASSCore(FreqMin = FreqMin, \
                                                                                               FreqMax = FreqMax, \
                                                                                               FreqStep = FreqStep, \
                                                                                               MolfitsFileName = BackgroundMolfitFileName, \
                                                                                               iso_flag = IsoFlag, \
                                                                                               IsoTableFileName = IsoTableFileName, \
                                                                                               TelescopeSize = TelescopeSize, \
                                                                                               BMIN = BMIN, \
                                                                                               BMAJ = BMAJ, \
                                                                                               BPA = BPA, \
                                                                                               vLSR = GlobalvLSR, \
                                                                                               Redshift = Redshift, \
                                                                                               Inter_Flag = InterFlag, \
                                                                                               t_back_flag = t_back_flag, \
                                                                                               tBack = tBack, \
                                                                                               tSlope = tSlope, \
                                                                                               BackgroundFileName = BackgroundFileName, \
                                                                                               N_H = N_H, \
                                                                                               beta_dust = beta_dust, \
                                                                                               kappa_1300 = kappa_1300, \
                                                                                               DustFileName = DustFileName,\
                                                                                               ContPhenFuncID = LocalContPhenFuncID, \
                                                                                               ContPhenFuncParam1 = LocalContPhenFuncParam1, \
                                                                                               ContPhenFuncParam2 = LocalContPhenFuncParam2, \
                                                                                               ContPhenFuncParam3 = LocalContPhenFuncParam3, \
                                                                                               ContPhenFuncParam4 = LocalContPhenFuncParam4, \
                                                                                               ContPhenFuncParam5 = LocalContPhenFuncParam5, \
                                                                                               NumModelPixelXX = NumModelPixelXX, \
                                                                                               NumModelPixelYY = NumModelPixelYY, \
                                                                                               LocalOverlapFlag = LocalOverlapFlag, \
                                                                                               NoSubBeamFlag = NoSubBeamFlag, \
                                                                                               dbFileName = dbFilename, \
                                                                                               EmAbsFlag = EmAbsFlag, \
                                                                                               printFlag = printFlag)
                                                                                              # SpectrumOnlyFlag = True)
                    ## continue here, if spectrum was created
                    if (modeldata != []):


                        ## store synthetic spectrum as background file
                        if (BackgroundSpectrumFlag):
                            LocalBackgroundFileName = LocalFitDir + "Background-File__" + str(ObsDataFileIndex + 1) + "_-_" + str(RangeIndex + 1)
                            LocalBackgroundFileName += ".dat"
                            numpy.savetxt(LocalBackgroundFileName, modeldata, delimiter = '\t')


                        ## store emission and absorption function
                        if (EmAbsFlag):
                            DirListing = os.listdir(JobDir)
                            for LocalFileName in DirListing:
                                LowerLocalFileName = LocalFileName.lower()
                                if (LowerLocalFileName.startswith("emission_absorption____distance__")):
                                    LocalEmAbsFileName = JobDir + "/" + LocalFileName
                                    NewLocalEmAbsFileName = LocalFileName.replace("emission_absorption____distance__",
                                                                                  "emission_absorption____" + str(ObsDataFileIndex + 1) + "_-_" \
                                                                                                            + str(RangeIndex + 1) + "__distance__")
                                    cmdString = "mv " + LocalEmAbsFileName + " " + LocalFitDir + NewLocalEmAbsFileName
                                    os.system(cmdString)


                    ## remove myXCLASS job directory
                    LocalScreenFileName = LocalFitDir + "screen__myXCLASS__" + str(ObsDataFileIndex + 1) + "_-_" + str(RangeIndex + 1) + ".out"
                    cmdString = "mv " + JobDir + "/screen.out " + LocalScreenFileName + "; "
                    cmdString += "rm -rf " + JobDir
                    os.system(cmdString)


        ## declare return variable
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## perform molecule fits for ith molecule
    ##
    def DoMoleculeFit(self, InputDict):
        """

    input parameters:
    -----------------

        - InputDict:                     parameter dictionary



    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("InputDict = ", InputDict)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get parameters from dictionary
        ExpFileList = InputDict['ExpFileList']
        NumberExpRangesList = InputDict['NumberExpRangesList']
        FreqMinList = InputDict['FreqMinList']
        FreqMaxList = InputDict['FreqMaxList']
        FreqStepList = InputDict['FreqStepList']
        t_back_flagList = InputDict['t_back_flagList']
        tBackList = InputDict['tBackList']
        tSlopeList = InputDict['tSlopeList']
        N_HList = InputDict['N_HList']
        beta_dustList = InputDict['beta_dustList']
        kappa_1300List = InputDict['kappa_1300List']
        DustFileNameList = InputDict['DustFileNameList']
        BackgroundFileNameList = InputDict['BackgroundFileNameList']
        ContPhenFuncIDList = InputDict['ContPhenFuncIDList']
        ContPhenFuncParam1List = InputDict['ContPhenFuncParam1List']
        ContPhenFuncParam2List = InputDict['ContPhenFuncParam2List']
        ContPhenFuncParam3List = InputDict['ContPhenFuncParam3List']
        ContPhenFuncParam4List = InputDict['ContPhenFuncParam4List']
        ContPhenFuncParam5List = InputDict['ContPhenFuncParam5List']
        NoiseList = InputDict['NoiseList']
        SmoothList = InputDict['SmoothList']
        TelescopeSizeList = InputDict['TelescopeSizeList']
        BMINList = InputDict['BMINList']
        BMAJList = InputDict['BMAJList']
        BPAList = InputDict['BPAList']
        Inter_FlagList = InputDict['Inter_FlagList']
        GlobalvLSRList = InputDict['GlobalvLSRList']
        RedshiftList = InputDict['RedshiftList']
        ErrorYList = InputDict['ErrorYList']
        NumberHeaderLinesList = InputDict['NumberHeaderLinesList']
        SeparatorColumnsList = InputDict['SeparatorColumnsList']
        IsotopologuesList = InputDict['IsotopologuesList']
        IsoTableFileNameList = InputDict['IsoTableFileNameList']
        IsoTableFileName = InputDict['IsoTableFileName']
        IsoFlag = InputDict['IsoFlag']
        OrigIsoRatioFileContent = InputDict['OrigIsoRatioFileContent']
        dbList = InputDict['dbList']
        dbFilename = InputDict['dbFile']
        NumModelPixelXXList = InputDict['NumModelPixelXXList']
        NumModelPixelXX = InputDict['NumModelPixelXX']
        NumModelPixelYYList = InputDict['NumModelPixelYYList']
        NumModelPixelYY = InputDict['NumModelPixelYY']
        LocalOverlapFlagList = InputDict['LocalOverlapFlagList']
        LocalOverlapFlag = InputDict['LocalOverlapFlag']
        MaxNumTransitionsSQL = InputDict['MaxNumTransitionsSQL']
        UnmodifiedExpXML = InputDict['UnmodifiedExpXML']
        LocalAlgorithmXMLFileName = InputDict['LocalAlgorithmXMLFileName']
        RangevelLowLimit = InputDict['RangevelLowLimit']
        RangevelUpLimit = InputDict['RangevelUpLimit']
        IncludeAllMolFlag = InputDict['IncludeAllMolFlag']
        NewParameterRanges = InputDict['NewParameterRanges']
        LimitRangesFlag = InputDict['LimitRangesFlag']
        AdditionMolfitFileContent = InputDict['AdditionMolfitFileContent']
        dryrun = InputDict['dryrun']
        RedRangeFlag = InputDict['RedRangeFlag']
        LocalMinColumnDensityAbs = InputDict['LocalMinColumnDensityAbs']
        LocalMinColumnDensityEmis = InputDict['LocalMinColumnDensityEmis']
        SetNewRangeFlag = InputDict['SetNewRangeFlag']
        UpdateFlag = InputDict['UpdateFlag']
        EmsAbsFlag = InputDict['EmsAbsFlag']
        backgroundFlag = InputDict['backgroundFlag']
        BackgroundDir = InputDict['BackgroundDir']
        DoFinalPlotFlag = InputDict['DoFinalPlotFlag']
        PlotNum = InputDict['PlotNum']
        RemoveContFlag = InputDict['RemoveContFlag']
        AddMoleculeNamesToPlotFlag = InputDict['AddMoleculeNamesToPlotFlag']
        RepID = InputDict['RepID']
        ResultPath = InputDict['ResultPath']
        MolfitFileForEachMolecule = InputDict['MolfitFileForEachMolecule']
        AllParameters = InputDict['AllParameters']
        MoleculesInMolfitFile = InputDict['MoleculesInMolfitFile']
        ContInMolfitFileFlag = InputDict['ContInMolfitFileFlag']
        SQLParamArray = InputDict['SQLParamArray']
        MaxElowSQL = InputDict['MaxElowSQL']
        MolfitHeaderLinesList = InputDict['MolfitHeaderLinesList']
        NumberOfMolecules = InputDict['NumberOfMolecules']
        NumberProcessors = InputDict['NumberProcessors']
        LocalPrintFlag = InputDict['LocalPrintFlag']
        MoleculeIndex = InputDict['MoleculeIndex']
        LocalMolfitParameters = InputDict['LocalMolfitParameters']
        LocalMoleculeName = InputDict['LocalMoleculeName']
        EmAbsPATHList = InputDict['EmAbsPATHList']
        FastCompBackFlag = InputDict['FastCompBackFlag']


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## fit molecule


        ## check, if current molecule should be optimized or not
        FitMoleculeFlag = False
        for line in LocalMolfitParameters:
            for col in line:
                flag = col[1]
                if (flag == "1"):
                    FitMoleculeFlag = True
                    break


        ## start fitting of current molecule if at least one parameter is free
        NewExpXMLFileName = ""
        if (not FitMoleculeFlag):


            ## store results as pickle file
            LocalDict = {}
            LocalDict['MoleculeName'] = LocalMoleculeName
            LocalDict['BestMolfitFile'] = MolfitFileForEachMolecule[MoleculeIndex]
            LocalDict['IsoRatioFileContent'] = copy.deepcopy(self.FinalIsoRatioFileContents)
            LocalPickelName = ResultPath + "molecule__" + str(MoleculeIndex + 1) + ".pickle"
            pickle.dump(LocalDict, open(LocalPickelName, 'wb'))
        else:


            ## print what you do
            if (LocalPrintFlag):
                print ("\tAnalyze (" + str(MoleculeIndex + 1) + "/" + str(NumberOfMolecules) + ") molecule: "+ chr(34) + LocalMoleculeName + chr(34))


            ## create a subdirectory for single molecule fit
            LocalMoleculeFileName = task_LineIdentification.MoleculeFileName(LocalMoleculeName)
            LocalFitDir = ResultPath + "molecule-fits/" + LocalMoleculeFileName + "/"
            cmdString = "rm -rf " + LocalFitDir + "; "
            cmdString += "mkdir -p " + LocalFitDir + "; "
            if (self.DoMoleculePlotFlag):
                LocalPlotDir = LocalFitDir + "plot/"
                cmdString += "rm -rf " + LocalPlotDir + "; "
                cmdString += "mkdir -p " + LocalPlotDir + "; "
            os.system(cmdString)


            ## redirect screen output to file
            if (NumberProcessors > 1):
                ScreenOutFileName = LocalFitDir + "screen-output__" + str(os.getpid()) + ".out"
                sys.stdout = open(ScreenOutFileName, "a", buffering = 10)
                # sys.stderr = open(ScreenOutFileName, "a", buffering = 0)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## create obs. xml file for current molecule with reduced frequency ranges around transition frequencies
            if (RedRangeFlag):
                NewExpXMLFileName = LocalFitDir + "exp.xml"
                MaxWidth = 100.0
                NoSubBeamFlag = False
                task_LineIdentification.CreateXMLFile(LocalMoleculeName, RangevelLowLimit, RangevelUpLimit, MaxWidth, NewExpXMLFileName, \
                                                      ExpFileList, NumberExpRangesList, FreqMinList, FreqMaxList, FreqStepList, t_back_flagList, \
                                                      tBackList, tSlopeList, N_HList, beta_dustList, kappa_1300List, DustFileNameList, \
                                                      BackgroundFileNameList, ContPhenFuncIDList, ContPhenFuncParam1List, ContPhenFuncParam2List, \
                                                      ContPhenFuncParam3List, ContPhenFuncParam4List, ContPhenFuncParam5List, NoiseList, \
                                                      SmoothList, TelescopeSizeList, BMINList, BMAJList, BPAList, Inter_FlagList, GlobalvLSRList, \
                                                      RedshiftList, ErrorYList, NumberHeaderLinesList, SeparatorColumnsList, IsotopologuesList, \
                                                      IsoTableFileNameList, dbList, NumModelPixelXXList, NumModelPixelYYList, \
                                                      LocalOverlapFlagList, SQLParamArray, MaxNumTransitionsSQL, [NoSubBeamFlag], EmAbsPATHList)
            else:
                NewExpXMLFileName = LocalFitDir + "exp.xml"
                cmdString = "cp " + UnmodifiedExpXML + " " + NewExpXMLFileName
                os.system(cmdString)

                # Debug:
                # print (">>UnmodifiedExpXML = ", UnmodifiedExpXML)
                # print (">>NewExpXMLFileName = ", NewExpXMLFileName)


            ## get path and names of observational data files from shrinked xml file
            NewExpFileList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "FileNamesExpFiles")
            NewNumberExpFiles = len(NewExpFileList)
            ListNumberExpRanges = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "NumberExpRanges")
            NewFreqMinList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "MinExpRange")
            NewFreqMaxList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "MaxExpRange")
            NewFreqStepList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "StepFrequency")
            Newt_back_flagList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "t_back_flag")
            NewtBackList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "BackgroundTemperature")
            NewtSlopeList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "TemperatureSlope")
            NewN_HList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "HydrogenColumnDensity")
            Newbeta_dustList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "DustBeta")
            Newkappa_1300List = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "Kappa")
            NewDustFileNameList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "DustFileName")
            NewBackgroundFileNameList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "BackgroundFileName")
            NewContPhenFuncIDList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "ContPhenFuncID")
            NewContPhenFuncParam1List = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "ContPhenFuncParam1")
            NewContPhenFuncParam2List = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "ContPhenFuncParam2")
            NewContPhenFuncParam3List = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "ContPhenFuncParam3")
            NewContPhenFuncParam4List = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "ContPhenFuncParam4")
            NewContPhenFuncParam5List = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "ContPhenFuncParam5")
            NewNoiseList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "NoiseLevel")
            NewTelescopeSizeList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "TelescopeSize")
            NewBMINList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "BMIN")
            NewBMAJList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "BMAJ")
            NewBPAList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "BPA")
            NewInter_FlagList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "Inter_Flag")
            NewGlobalvLSRList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "GlobalvLSR")
            NewRedshiftList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "Redshift")
            NewIsotopologuesList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "iso_flag")
            if (NewIsotopologuesList == []):
                NewIsotopologuesList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "Isotopologues")
            NewIsoTableFileNameList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "IsoTableFileName")
            NewErrorYFlagList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "ErrorY")
            NewNumberHeaderLinesList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "NumberHeaderLines")
            NewSeparatorColumnsList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "SeparatorColumns")
            EmAbsPATHList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "EmAbsPATH")
            counterPNG = 0

            # Debug:
            # print ("\n\n>>NewExpXMLFileName = ", NewExpXMLFileName)
            # print ("NewtBackList = ", NewtBackList)
            # print ("NewtSlopeList = ", NewtSlopeList)
            # print ("RedRangeFlag = ", RedRangeFlag)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## construct molfit file
            NameOfCurrentMolfitFile = LocalFitDir + LocalMoleculeFileName + ".molfit"
            CurrentMolfitFile = open(NameOfCurrentMolfitFile, 'w')
            for line in MolfitHeaderLinesList:
                CurrentMolfitFile.write(line)
            AddFlag = False
            AllTransFreqList = []


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## add additional content
            if (len(AdditionMolfitFileContent) > 0 and (not EmsAbsFlag) and (not backgroundFlag) and BackgroundDir == ""):
                for line in AdditionMolfitFileContent:
                    CurrentMolfitFile.write(line)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## open new molfit file for background file computation
            backgroundMoleculeFlag = False
            LocalBackgroundFileList = []
            FastCompEmsAbsFileList = {}
            if (backgroundFlag or EmsAbsFlag):


                ## check if no global continuum is defined
                AllTBackFlag = False
                for ObsDataFileIndex in range(len(NewExpFileList)):                         ## loop over all obs. data files
                    RangeIndex = (-1)
                    ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                   NumberRangeListIn = ListNumberExpRanges)
                    NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
                    for RangeIndex in range(NumberFrequencyRanges):                         ## loop over all range definitions in the whole xml file
                        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                        tBackFlagListIn = Newt_back_flagList)
                        t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                        t_back_flag = task_myXCLASSMapFit.CheckBool(t_back_flag)
                        if (t_back_flag):
                            AllTBackFlag = True
                            break

                # Debug:
                # print ("AllTBackFlag = ", AllTBackFlag)


                ## import files describing emission / absorption files
                if (FastCompBackFlag):
                    LocalEmsAbsPath = ResultPath + "emission+absorptions/"
                    StoredMoleculesDirList = os.listdir(LocalEmsAbsPath)
                    for LoalMolDir in StoredMoleculesDirList:
                        # if ((not LoalMolDir.startswith("RRL-")) and (not AllTBackFlag)):
                        if (AllTBackFlag or (not LoalMolDir.startswith("RRL-")) ):
                            LocalMolDirAbsPath = LocalEmsAbsPath + "/" + LoalMolDir + "/"
                            StoredMoleculesDirList = os.listdir(LocalMolDirAbsPath)
                            FileNameList = []
                            for LocalFileName in StoredMoleculesDirList:
                                if (LocalFileName.startswith("emission_absorption____")):
                                    FileNameList.append(LocalMolDirAbsPath + LocalFileName)
                            FastCompEmsAbsFileList[LoalMolDir] = FileNameList

                    # Debug:
                    # print ("FastCompEmsAbsFileList = ", FastCompEmsAbsFileList)


                ## determine list of background files
                elif (BackgroundDir != ""):
                    BackgroundMolfitFile = None
                    LocalBackgroundDir = BackgroundDir + "/" + LocalMoleculeFileName + "/"
                    LocalBackgroundDir = os.path.normpath(LocalBackgroundDir) + "/"
                    listing = os.listdir(LocalBackgroundDir)
                    LocalBackgroundFileList = []
                    for LocalFile in listing:
                        if (LocalFile.startswith("Background-File__")):
                            LocalBackgroundFileList.append(LocalBackgroundDir + LocalFile)


                ## open molfit file
                else:
                    BackgroundMolfitFileName = LocalFitDir + LocalMoleculeFileName + "__background.molfit"
                    BackgroundMolfitFile = open(BackgroundMolfitFileName, 'w')
                    for line in MolfitHeaderLinesList:
                        BackgroundMolfitFile.write(line)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get transition frequencies within reduced frequency ranges for each molecule in the molfit file
            for MoleculeIndex2, molfitFile in enumerate(MolfitFileForEachMolecule):
                LocalMoleculeName2 = MoleculesInMolfitFile[MoleculeIndex2]


                ## check if at least one transition is within the given frequency ranges
                LocalElowMax = MaxElowSQL
                ShiftedFreqMinList = []
                ShiftedFreqMaxList = []
                for ObsDataFileIndex in range(len(NewExpFileList)):                         ## loop over all obs. data files


                    ## get number of header lines and character separating columns from obs. xml file
                    RangeIndex = (-1)
                    ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                   NumberRangeListIn = ListNumberExpRanges, \
                                                                                   GlobalvLSRListIn = NewGlobalvLSRList, \
                                                                                   Redshift_ListIn = NewRedshiftList)
                    NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
                    GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
                    if (GlobalvLSR is None):
                        GlobalvLSR = 0.0
                    Redshift = ObsXMLParameterDictFile['Redshift']
                    if (Redshift is None):
                        Redshift = 0.0


                    ## get parameters for current frequency range
                    for RangeIndex in range(NumberFrequencyRanges):                         ## loop over all range definitions in the whole xml file


                        ## get parameter for current frequency range
                        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                        FreqMinListIn = NewFreqMinList, \
                                                                                        FreqMaxListIn = NewFreqMaxList)
                        FreqMin = ObsXMLParameterDictRange['FreqMin']
                        if (FreqMin is not None):
                            FreqMax = ObsXMLParameterDictRange['FreqMax']


                            ## shift min. and max. frequency of each range by v_LSR
                            NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift)
                            NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift)
                            ShiftedFreqMin = min(NewFreqMin, NewFreqMax)
                            ShiftedFreqMax = max(NewFreqMin, NewFreqMax)
                            ShiftedFreqMinList.append(FreqMin)
                            ShiftedFreqMaxList.append(FreqMax)


                ## get parameters for each transition
                LocalMaxNumTransInFit = 0
                TransFreqList, DBParamList = task_LineIdentification.GetTransFreq(LocalMoleculeName2, ShiftedFreqMinList, ShiftedFreqMaxList, \
                                                                                  LocalElowMax, dbFilename, LocalMaxNumTransInFit)
                AllTransFreqList.append(TransFreqList)


            ## read in results from previous molecule fits and create dictionary
            if (UpdateFlag):
                SMFResultsList = {}
                listing = os.listdir(ResultPath)
                for LocalFile in listing:
                    if (LocalFile.endswith(".pickle")):
                        LocalDict = pickle.load(open(ResultPath + LocalFile, "rb"))
                        DictMolName = LocalDict['MoleculeName']
                        DictBestMolfitFile = LocalDict['BestMolfitFile']
                        DictIsoRatioFileContent = LocalDict['IsoRatioFileContent']
                        SMFResultsList[DictMolName] = [DictBestMolfitFile, DictIsoRatioFileContent]


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## construct molfit file
            MoleculeIncludedFlag = False
            for MoleculeIndex2, molfitFile in enumerate(MolfitFileForEachMolecule):
                LocalMoleculeName2 = MoleculesInMolfitFile[MoleculeIndex2]
                LowerLocalMoleculeName2 = LocalMoleculeName2.lower()
                TransFreqList = AllTransFreqList[MoleculeIndex2]


                ## check, if current molecule is added to current molfit file
                ContinueFlag = False
                if (LowerLocalMoleculeName2.startswith("cont") or (LowerLocalMoleculeName2.startswith("rrl") and MoleculeIndex != MoleculeIndex2)):
                    ContinueFlag = True
                elif (len(TransFreqList) > 0):
                    if (FastCompBackFlag):
                        if (MoleculeIndex == MoleculeIndex2):
                            ContinueFlag = True
                    else:
                        ContinueFlag = True
                if (ContinueFlag):
                    LocalMolfitFileContent = copy.deepcopy(molfitFile)


                    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                    ## get previous optimized molfit parameters
                    if (UpdateFlag):


                        ## check, if molecule was optimized before
                        if (LocalMoleculeName2 in SMFResultsList):
                            LocalMolfitFileContent = SMFResultsList[LocalMoleculeName2][0]
                            LocalIsoRatioFileContent = SMFResultsList[LocalMoleculeName2][1]
                            if (LocalIsoRatioFileContent != [] and IsoFlag):


                                ##************************************************************************************************************************
                                ## update local iso ratio file


                                ## create new content of iso ratio file
                                OldOrigIsoRatioFileContent = copy.deepcopy(OrigIsoRatioFileContent)
                                OrigIsoRatioFileContent = []


                                ## add all lines of iso ratio file which do not belong to the previous optimized molecule
                                for line in OldOrigIsoRatioFileContent:
                                    i = line.find("%")
                                    if (i > (-1)):
                                        StrippedLine = line[:i].strip()
                                    else:
                                        StrippedLine = line.strip()
                                    if (StrippedLine != ""):
                                        SplittedLine = StrippedLine.split()
                                        LocalIsotopologue = SplittedLine[0].strip()
                                        LocalIsoMaster = SplittedLine[1].strip()


                                        ## check if current line of iso ratio file is not included in optimized iso ratio content
                                        AddLocalLineFlag = True
                                        for NewLine in LocalIsoRatioFileContent:
                                            i = NewLine.find("%")
                                            if (i > (-1)):
                                                NewStrippedLine = NewLine[:i].strip()
                                            else:
                                                NewStrippedLine = NewLine.strip()
                                            if (NewStrippedLine != ""):
                                                NewSplittedLine = NewStrippedLine.split()
                                                NewIsotopologue = NewSplittedLine[0].strip()
                                                NewIsoMaster = NewSplittedLine[1].strip()
                                                if (LocalIsotopologue == NewIsotopologue and LocalIsoMaster == NewIsoMaster):
                                                    AddLocalLineFlag = False                ## current ratio was optimized in the previous iteration
                                                    break
                                        if (AddLocalLineFlag):
                                            OrigIsoRatioFileContent.append(line)


                                ## add content of optimized iso ratio file of previous molecule
                                for line in LocalIsoRatioFileContent:
                                    OrigIsoRatioFileContent.append(line)


                                ## write new iso ratio file and update obs. xml file
                                LocalIsoFileName = LocalFitDir + LocalMoleculeFileName + "__iso-ratio.dat"
                                LocalIsoFile = open(LocalIsoFileName, 'w')
                                for line in OrigIsoRatioFileContent:
                                    LocalIsoFile.write(line)
                                LocalIsoFile.close()
                                task_MAGIX.WriteXMLtagNEW(NewExpXMLFileName, "IsoTableFileName", [LocalIsoFileName])


                    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                    ## do not update molfit and iso ratio file
                    else:
                        LocalMolfitFileContent = copy.deepcopy(molfitFile)


                    ## construct molfit file for current molecule
                    MolfitFileForEachMolecule[MoleculeIndex2] = copy.deepcopy(LocalMolfitFileContent)
                    for line in LocalMolfitFileContent:
                        i = line.find("%")
                        if (i > (-1)):
                            StrippedLine = line[:i].strip()
                        else:
                            StrippedLine = line.strip()
                        SplittedLine = StrippedLine.split()
                        if (MoleculeIndex == MoleculeIndex2):


                            ## add molecule name and number of components
                            if (len(SplittedLine) == 2 or len(NewParameterRanges) != 5):
                                AddFlag = True
                                MoleculeIncludedFlag = True
                                NumComp = int(SplittedLine[1]) + 1
                                NewLine = SplittedLine[0] + "  {:d}\n".format(NumComp + len(self.AddCompContent))
                                CurrentMolfitFile.write(NewLine)


                                ## add additional content
                                for AddLine in self.AddCompContent:
                                    CurrentMolfitFile.write(AddLine)


                            ## add component description
                            elif (len(SplittedLine) > 2 and len(NewParameterRanges) == 5 and AddFlag):


                                ## set new ranges
                                AddFlag = False
                                NewLine = ""
                                LocalMolfitParameters = AllParameters[MoleculeIndex]
                                for LocalLine in LocalMolfitParameters:

                                    # Debug:
                                    # print ("LocalLine = ", LocalLine)


                                    LocalvOff = 0.0
                                    FitFlag = False
                                    NewComp = " "
                                    for col in LocalLine:
                                        Name = col[0]
                                        flag = col[1]
                                        lowlimit = min(float(col[2]), float(col[3]))
                                        uplimit = max(float(col[2]), float(col[3]))
                                        element = col[4]
                                        AddFlag = False

                                        # Debug:
                                        # print ("col = ", col)
                                        # print ("Name = ", Name)


                                        ## get fit flag
                                        if (Name.endswith("_flag") or Name.endswith("_FitFlag")):
                                            FitFlag = element.strip().lower()


                                        ## ignore limits
                                        elif (Name.endswith("_lowlimit") or Name.endswith("_uplimit")):
                                            AddFlag = False


                                        ## store values for source size
                                        elif (Name in ["source_size", "source_radius", "source_center_x", "source_center_y"]):
                                            value = float(element)
                                            if (SetNewRangeFlag):
                                                if (LimitRangesFlag):
                                                    lowlimit = max(value - NewParameterRanges[0][0], lowlimit)
                                                    uplimit =  min(value + NewParameterRanges[0][1], uplimit)
                                                else:
                                                    lowlimit = max(1.0, value - NewParameterRanges[0][0])
                                                    uplimit = value + NewParameterRanges[0][1]
                                            AddFlag = True


                                        ## store values for T_rot
                                        elif (Name in ["T_rot", "T_e", "T_dOff", "T_cont_dust" , "T_Back", "T_dSlope"]):
                                            value = float(element)
                                            if (SetNewRangeFlag):
                                                if (LimitRangesFlag):
                                                    lowlimit = max(value - NewParameterRanges[1][0], lowlimit)
                                                    uplimit =  min(value + NewParameterRanges[1][1], uplimit)
                                                else:
                                                    lowlimit = max(1.072, value - NewParameterRanges[1][0])
                                                    uplimit = value + NewParameterRanges[1][1]
                                            AddFlag = True


                                        ## store values for N_tot, EM_RRL, N_e, nHcolumn_cont_dust, nHcolumn
                                        elif (Name in ["N_tot", "nHcolumn_cont_dust", "nHcolumn", "EM_RRL", \
                                                       "kappa_cont_dust", "kappa"]):
                                            value = float(element)
                                            if (SetNewRangeFlag):
                                                if (LimitRangesFlag):
                                                    lowlimit = max(10.0**(numpy.log10(value) - NewParameterRanges[2][0]), lowlimit)
                                                    uplimit =  min(10.0**(numpy.log10(value) + NewParameterRanges[2][1]), uplimit)
                                                else:
                                                    if (Name.startswith("N_tot")):
                                                        MinLow = 1.e8
                                                        MaxUp = 1.e24
                                                    elif (Name.startswith("EM_RRL")):
                                                        MinLow = 1.e1
                                                        MaxUp = 1.e20
                                                    elif (Name.startswith("N_e") or Name.startswith("nHcolumn_cont_dust") \
                                                        or Name.startswith("nHcolumn")):
                                                        MinLow = 1.e1
                                                        MaxUp = 1.e50
                                                    lowlimit = max(MinLow, 10.0**(numpy.log10(value) - NewParameterRanges[2][0]))
                                                    uplimit = min(MaxUp, 10.0**(numpy.log10(value) + NewParameterRanges[2][1]))
                                            AddFlag = False
                                            NewComp += "    %1s  %10.2e  %10.2e   %15.5e" % (FitFlag, lowlimit, uplimit, value)


                                        ## store values for V_width
                                        elif (Name in ["V_width_Gauss"]):
                                            value = float(element)
                                            if (SetNewRangeFlag):
                                                if (LimitRangesFlag):
                                                    lowlimit = max(value - NewParameterRanges[3][0], lowlimit)
                                                    uplimit =  min(value + NewParameterRanges[3][1], uplimit)
                                                else:
                                                    lowlimit = max(0.1, value - NewParameterRanges[3][0])
                                                    uplimit = value + NewParameterRanges[3][1]
                                            AddFlag = True


                                        ## store values for V_off
                                        elif (Name == "V_off"):
                                            value = float(element)
                                            LocalvOff = value
                                            if (SetNewRangeFlag):
                                                if (LimitRangesFlag):
                                                    lowlimit = max(value - NewParameterRanges[4][0], lowlimit)
                                                    uplimit =  min(value + NewParameterRanges[4][1], uplimit)
                                                else:
                                                    lowlimit = value - NewParameterRanges[4][0]
                                                    uplimit = value + NewParameterRanges[4][1]
                                            AddFlag = True


                                        ## store values
                                        elif (Name in ["beta_cont_dust", "beta", "ContFuncID_param_1", "ContFuncID_param_2", \
                                                       "ContFuncID_param_3", "ContFuncID_param_4", "ContFuncID_param_5"]):
                                            value = float(element)
                                            LocalvOff = value
                                            if (SetNewRangeFlag):
                                                if (LimitRangesFlag):
                                                    lowlimit = max(value - NewParameterRanges[4][0], lowlimit)
                                                    uplimit =  min(value + NewParameterRanges[4][1], uplimit)
                                                else:
                                                    lowlimit = value - NewParameterRanges[4][0]
                                                    uplimit = value + NewParameterRanges[4][1]
                                            AddFlag = True


                                        ## store values for EA-Flag
                                        elif (Name in ["LayerDistance", "ContFuncID_phen", "CFFlag", "keyword"]):
                                            AddFlag = False
                                            NewComp += "    " + element


                                        ## create new line
                                        if (AddFlag):
                                            NewComp += "    %s  %10.2f  %10.2f   %15.5f" % (FitFlag, lowlimit, uplimit, value)


                                    ## add line break
                                    if (LocalvOff < RangevelLowLimit or RangevelUpLimit < LocalvOff):
                                        NewComp = NewComp.replace(" y ", " n ")
                                    NewLine += NewComp + "\n"


                                ## check, if vel. offset is within given vel. range and write new line to file
                                MoleculeIncludedFlag = True
                                CurrentMolfitFile.write(NewLine)


                        ## include other molecules
                        else:
                            if (IncludeAllMolFlag):
                                if (len(SplittedLine) > 2):
                                    if (MoleculeIndex != MoleculeIndex2):
                                        line = line.replace(" y ", " n ")
                                if (not FastCompBackFlag and (backgroundFlag or EmsAbsFlag)):
                                    backgroundMoleculeFlag = True
                                    LocalStrippedLine = line.strip()
                                    if ((not LocalStrippedLine.startswith("%")) and BackgroundDir == ""):
                                        BackgroundMolfitFile.write(line)
                                else:
                                    MoleculeIncludedFlag = True
                                    CurrentMolfitFile.write(line)


            ## close molfit file(s)
            CurrentMolfitFile.close()
            if (not FastCompBackFlag and (backgroundFlag or EmsAbsFlag) and BackgroundDir == ""):
                BackgroundMolfitFile.close()


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call myXCLASS function to compute background file
            if (len(LocalBackgroundFileList) > 0 or FastCompBackFlag \
                or ((backgroundFlag or EmsAbsFlag) and MoleculeIncludedFlag and backgroundMoleculeFlag)):


                ## check if at least on transition is within the given frequency ranges
                LocalBackgroundFileNameList = []
                LocaltBackList = []
                LocaltSlopeList = []
                LocalDustKappaList = []
                NewEmAbsPATH = []
                for ObsDataFileIndex in range(NewNumberExpFiles):                           ## loop over all obs. data files


                    ## get number of header lines and character separating columns from obs. xml file
                    RangeIndex = (-1)
                    ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                   TelescopeSizeListIn = NewTelescopeSizeList, \
                                                                                   BMIN_ListIn = NewBMINList, \
                                                                                   BMAJ_ListIn = NewBMAJList, \
                                                                                   BPA_ListIn = NewBPAList, \
                                                                                   GlobalvLSRListIn = NewGlobalvLSRList, \
                                                                                   Redshift_ListIn = NewRedshiftList, \
                                                                                   InterFlagListIn = NewInter_FlagList, \
                                                                                   NumberRangeListIn = ListNumberExpRanges)
                    NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
                    TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
                    BMIN = ObsXMLParameterDictFile['BMIN']
                    BMAJ = ObsXMLParameterDictFile['BMAJ']
                    BPA = ObsXMLParameterDictFile['BPA']
                    InterFlag = ObsXMLParameterDictFile['InterFlag']
                    GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
                    if (GlobalvLSR is None):
                        GlobalvLSR = 0.0
                    Redshift = ObsXMLParameterDictFile['Redshift']

                    # Debug:
                    # print ("\nNumberFrequencyRanges = ", NumberFrequencyRanges)
                    # print ("TelescopeSize = ", TelescopeSize)
                    # print ("BMIN = ", BMIN)
                    # print ("BMAJ = ", BMAJ)
                    # print ("BPA = ", BPA)
                    # print ("InterFlag = ", InterFlag)
                    # print ("GlobalvLSR = ", GlobalvLSR)
                    # print ("Redshift = ", Redshift)


                    ## analyze ranges
                    for RangeIndex in range(NumberFrequencyRanges):                         ## loop over all range definitions in the whole xml file


                        ## get parameter for current frequency range
                        ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = NewFreqMinList, \
                                                                                        FreqMaxListIn = NewFreqMaxList, \
                                                                                        FreqStepListIn = NewFreqStepList, \
                                                                                        tBackFlagListIn = Newt_back_flagList, \
                                                                                        tBackListIn = NewtBackList, \
                                                                                        tSlopeListIn = NewtSlopeList, N_HListIn = NewN_HList, \
                                                                                        beta_dustListIn = Newbeta_dustList, \
                                                                                        kappa_1300ListIn = Newkappa_1300List, \
                                                                                        DustFileNameListIn = NewDustFileNameList, \
                                                                                        BackgroundFileNameListIn = NewBackgroundFileNameList, \
                                                                                        ContPhenFuncID_ListIn = NewContPhenFuncIDList, \
                                                                                        ContPhenFuncParam1_ListIn = NewContPhenFuncParam1List, \
                                                                                        ContPhenFuncParam2_ListIn = NewContPhenFuncParam2List, \
                                                                                        ContPhenFuncParam3_ListIn = NewContPhenFuncParam3List, \
                                                                                        ContPhenFuncParam4_ListIn = NewContPhenFuncParam4List, \
                                                                                        ContPhenFuncParam5_ListIn = NewContPhenFuncParam5List)
                        FreqMin = ObsXMLParameterDictRange['FreqMin']
                        if (FreqMin is not None):
                            FreqMax = ObsXMLParameterDictRange['FreqMax']
                            FreqStep = ObsXMLParameterDictRange['FreqStep']
                            t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                            tBack = ObsXMLParameterDictRange['tBack']
                            tSlope = ObsXMLParameterDictRange['tSlope']
                            N_H = ObsXMLParameterDictRange['N_H']
                            beta_dust = ObsXMLParameterDictRange['beta_dust']
                            kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                            if (N_H is None or beta_dust is None or kappa_1300 is None):
                                N_H = 0.0
                                beta_dust = 0.0
                                kappa_1300 = 0.0
                            DustFileName = ObsXMLParameterDictRange['DustFileName']
                            if (DustFileName is None):
                                DustFileName = ""
                            BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                            if (BackgroundFileName is None):
                                BackgroundFileName = ""
                            LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                            LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                            LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                            LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                            LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                            LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']

                            # Debug:
                            # print ("FreqMin = ", FreqMin)
                            # print ("FreqMax = ", FreqMax)
                            # print ("FreqStep = ", FreqStep)
                            # print ("t_back_flag = ", t_back_flag)
                            # print ("tBack = ", tBack)
                            # print ("tSlope = ", tSlope)
                            # print ("N_H = ", N_H)
                            # print ("beta_dust = ", beta_dust)
                            # print ("kappa_1300 = ", kappa_1300)
                            # print ("DustFileName = ", DustFileName)
                            # print ("BackgroundFileName = ", BackgroundFileName)
                            # print ("LocalContPhenFuncID = ", LocalContPhenFuncID)
                            # print ("LocalContPhenFuncParam1 = ", LocalContPhenFuncParam1)
                            # print ("LocalContPhenFuncParam2 = ", LocalContPhenFuncParam2)
                            # print ("LocalContPhenFuncParam3 = ", LocalContPhenFuncParam3)
                            # print ("LocalContPhenFuncParam4 = ", LocalContPhenFuncParam4)
                            # print ("LocalContPhenFuncParam5 = ", LocalContPhenFuncParam5)


                            ## define new background parameters
                            if (ContInMolfitFileFlag):
                                LocaltBackList.append([ObsDataFileIndex, RangeIndex, 0.0])
                                LocaltSlopeList.append([ObsDataFileIndex, RangeIndex, 0.0])
                            else:
                                LocaltBackList.append([ObsDataFileIndex, RangeIndex, tBack])
                                LocaltSlopeList.append([ObsDataFileIndex, RangeIndex, tSlope])


                            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                            ## import emission/absorption functions
                            LocalFoundBackgroundFileFlag = False
                            if (FastCompBackFlag):
                                NewEmAbsPATH = [LocalFitDir]


                                ## get list of files for current molecule
                                try:
                                    ListOfEmsAbsFuncCurrentMolecule = FastCompEmsAbsFileList[LocalMoleculeFileName]
                                except:
                                    ListOfEmsAbsFuncCurrentMolecule = None

                                # Debug:
                                # print ("\n\nLocalMoleculeFileName = ", LocalMoleculeFileName)
                                # print ("ListOfEmsAbsFuncCurrentMolecule = ", ListOfEmsAbsFuncCurrentMolecule)


                                if (ListOfEmsAbsFuncCurrentMolecule is not None):
                                    for LocalEmsAbsMolPATH in ListOfEmsAbsFuncCurrentMolecule:
                                        LocalEmsAbsMolFileName = os.path.basename(LocalEmsAbsMolPATH)
                                        FileNameBegin = "emission_absorption____" + str(ObsDataFileIndex + 1) + "_-_" + str(RangeIndex + 1) + "__distance__"

                                        # Debug:
                                        # print ("LocalEmsAbsMolFileName = ", LocalEmsAbsMolFileName)
                                        # print ("FileNameBegin = ", FileNameBegin)


                                        if (LocalEmsAbsMolFileName.startswith(FileNameBegin)):


                                            EmsAbsArray = None
                                            EmsAbsHeader = None
                                            for LocalEmsAbsMol in list(FastCompEmsAbsFileList.keys()):
                                                if (LocalEmsAbsMol != LocalMoleculeFileName):

                                                    # Debug:
                                                    # print (">>>>LocalEmsAbsMol = ", LocalEmsAbsMol)


                                                    for LocalEmsAbsMolPATH2 in FastCompEmsAbsFileList[LocalEmsAbsMol]:
                                                        LocalEmsAbsMolFileName2 = os.path.basename(LocalEmsAbsMolPATH2)
                                                        if (LocalEmsAbsMolFileName2 == LocalEmsAbsMolFileName):

                                                            # Debug:
                                                            # print ("LocalEmsAbsMolFileName2 = ", LocalEmsAbsMolFileName2)


                                                            ## import emission / absorption function
                                                            LocalEmsAbsArray = numpy.loadtxt(LocalEmsAbsMolPATH2, skiprows = 3)


                                                            ## import header (if not already imported)
                                                            if (EmsAbsHeader is None):
                                                                EmsAbsHeader = ""
                                                                EmsAbsHeaderFile = open(LocalEmsAbsMolPATH2)
                                                                EmsAbsHeader += EmsAbsHeaderFile.readline()
                                                                EmsAbsHeader += EmsAbsHeaderFile.readline()
                                                                EmsAbsHeader += EmsAbsHeaderFile.readline()
                                                                EmsAbsHeaderFile.close()
                                                                EmsAbsHeader = EmsAbsHeader[:-1]    ## remove last "\n"

                                                                # Debug:
                                                                # print ("EmsAbsHeader = ", EmsAbsHeader)


                                                            ## add up emission and absorption functions
                                                            if (EmsAbsArray is None):
                                                                EmsAbsArray = copy.deepcopy(LocalEmsAbsArray)
                                                            else:


                                                                ## check, if frequency axis is identical
                                                                if ((EmsAbsArray[:, 0] == LocalEmsAbsArray[:, 0]).all()):
                                                                    EmsAbsArray[:, 1] = numpy.add(EmsAbsArray[:, 1], LocalEmsAbsArray[:, 1])
                                                                    EmsAbsArray[:, 2] = numpy.add(EmsAbsArray[:, 2], LocalEmsAbsArray[:, 2])
                                                                else:
                                                                    print ("\n\nError in function task_myXCLASSFit.DoMoleculeFit:")
                                                                    print ("\t Frequency axis of current ems./abs. file does not correspond to curent frequency axis")
                                                                    print ("\n\t Ignore file.")
                                                                    print ("\n\t Current ems./abs. file: {:s}".format(LocalEmsAbsMolPATH2))
                                                                    print ("\t Min. freq. axis of file: {:.3e} MHz".format(numpy.nanmin(LocalEmsAbsArray[:, 0])))
                                                                    print ("\t Max. freq. axis of file: {:.3e} MHz".format(numpy.nanmax(LocalEmsAbsArray[:, 0])))
                                                                    print ("\n\t Min. freq. axis: {:.3e} MHz".format(numpy.nanmin(EmsAbsArray[:, 0])))
                                                                    print ("\t Max. freq. axis: {:.3e} MHz".format(numpy.nanmax(EmsAbsArray[:, 0])))
                                                                    print ("\n\n")


                                            ## store summed emission and absorption functions to file
                                            if (EmsAbsArray is not None):
                                                numpy.savetxt(LocalFitDir + LocalEmsAbsMolFileName, EmsAbsArray, header = EmsAbsHeader, comments = "")
                                                LocalFoundBackgroundFileFlag = True
                                                # break


                            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                            ## read background file
                            elif (LocalBackgroundFileList != []):
                                MaxAxis = []
                                NewLocalBackgroundFileName = ""
                                for LocalBackgroundFileName in LocalBackgroundFileList:
                                    LocalBackgroundFile = numpy.loadtxt(LocalBackgroundFileName)
                                    FreqAxis = LocalBackgroundFile[:, 0]
                                    kkk = FreqAxis[(FreqMin <= FreqAxis[:]) & (FreqAxis[:] <= FreqMax)]
                                    if (len(kkk) > len(MaxAxis)):
                                        MaxAxis = copy.deepcopy(kkk)
                                        NewLocalBackgroundFileName = LocalBackgroundFileName
                                if (len(MaxAxis) > 0 and NewLocalBackgroundFileName != ""):
                                    LocalBackgroundFileNameList.append([ObsDataFileIndex, RangeIndex, NewLocalBackgroundFileName])
                                    LocalFoundBackgroundFileFlag = True


                            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                            ## compute background spectrum
                            else:
                                LocalFoundBackgroundFileFlag = True


                                ## print what you do
                                if (LocalPrintFlag):
                                    PrintString = "\n\t\t\tCompute background file for obs. data file %i and range %i .." % (ObsDataFileIndex + 1, \
                                                                                                                             RangeIndex + 1)
                                    print (PrintString, end = "", flush = True)


                                ## call myXCLASS function
                                printFlag = False
                                NoSubBeamFlag = True

                                modeldata, log, TransEnergies, IntOpt, JobDir = task_myXCLASS.myXCLASSCore(FreqMin = FreqMin, \
                                                                                                           FreqMax = FreqMax, \
                                                                                                           FreqStep = FreqStep, \
                                                                                                           MolfitsFileName = BackgroundMolfitFileName, \
                                                                                                           iso_flag = IsoFlag, \
                                                                                                           IsoTableFileName = IsoTableFileName, \
                                                                                                           TelescopeSize = TelescopeSize, \
                                                                                                           BMIN = BMIN, \
                                                                                                           BMAJ = BMAJ, \
                                                                                                           BPA = BPA, \
                                                                                                           vLSR = GlobalvLSR, \
                                                                                                           Redshift = Redshift, \
                                                                                                           Inter_Flag = InterFlag, \
                                                                                                           t_back_flag = t_back_flag, \
                                                                                                           tBack = tBack, \
                                                                                                           tSlope = tSlope, \
                                                                                                           BackgroundFileName = BackgroundFileName, \
                                                                                                           N_H = N_H, \
                                                                                                           beta_dust = beta_dust, \
                                                                                                           kappa_1300 = kappa_1300, \
                                                                                                           DustFileName = DustFileName,\
                                                                                                           ContPhenFuncID = LocalContPhenFuncID, \
                                                                                                           ContPhenFuncParam1 = LocalContPhenFuncParam1, \
                                                                                                           ContPhenFuncParam2 = LocalContPhenFuncParam2, \
                                                                                                           ContPhenFuncParam3 = LocalContPhenFuncParam3, \
                                                                                                           ContPhenFuncParam4 = LocalContPhenFuncParam4, \
                                                                                                           ContPhenFuncParam5 = LocalContPhenFuncParam5, \
                                                                                                           NumModelPixelXX = NumModelPixelXX, \
                                                                                                           NumModelPixelYY = NumModelPixelYY, \
                                                                                                           LocalOverlapFlag = LocalOverlapFlag, \
                                                                                                           NoSubBeamFlag = NoSubBeamFlag, \
                                                                                                           dbFileName = dbFilename, \
                                                                                                           EmAbsFlag = EmsAbsFlag, \
                                                                                                           printFlag = printFlag)

                                ## store synthetic spectrum as background file
                                if (modeldata != []):
                                    if (backgroundFlag):
                                        LocalBackgroundFileName = LocalFitDir + "Background-File__" + str(ObsDataFileIndex + 1)
                                        LocalBackgroundFileName += "_-_" + str(RangeIndex + 1) + ".dat"
                                        LocalBackgroundFileNameList.append([ObsDataFileIndex, RangeIndex, LocalBackgroundFileName])
                                        numpy.savetxt(LocalBackgroundFileName, modeldata, delimiter = '\t')


                                    ## store emission and absorption function
                                    if (EmsAbsFlag):
                                        NewEmAbsPATH = [LocalFitDir]
                                        DirListing = os.listdir(JobDir)
                                        for LocalFileName in DirListing:
                                            LowerLocalFileName = LocalFileName.lower()
                                            if (LowerLocalFileName.startswith("emission_absorption____distance__")):
                                                LocalEmAbsFileName = JobDir + "/" + LocalFileName
                                                NewLocalEmAbsFileName = LocalFileName.replace("emission_absorption____distance__",
                                                                                              "emission_absorption____" + str(ObsDataFileIndex + 1) \
                                                                                              + "_-_" + str(RangeIndex + 1) + "__distance__")
                                                cmdString = "mv " + LocalEmAbsFileName + " " + LocalFitDir + NewLocalEmAbsFileName
                                                os.system(cmdString)


                                ## remove myXCLASS job directory
                                LocalScreenFileName = LocalFitDir + "screen__myXCLASS__"
                                LocalScreenFileName += str(ObsDataFileIndex + 1) + "_-_" + str(RangeIndex + 1) + ".out"
                                cmdString = "cp " + JobDir + "/screen.out " + LocalScreenFileName + "; "
                                cmdString += "rm -rf " + JobDir
                                os.system(cmdString)


                                ## we're done
                                if (LocalPrintFlag):
                                    print ("done!", flush = True)


                            ## define kappa for dust
                            if (LocalFoundBackgroundFileFlag and ContInMolfitFileFlag):
                                LocalDustKappaList.append([ObsDataFileIndex, RangeIndex, 0.0])
                            else:
                                LocalDustKappaList.append([ObsDataFileIndex, RangeIndex, kappa_1300])


                ## update obs. xml file
                if (NewExpXMLFileName.strip() != "" and (LocalBackgroundFileNameList != [] or NewEmAbsPATH != [])):
                    if (LocalBackgroundFileNameList != []):
                        task_MAGIX.WriteXMLtagNEW(NewExpXMLFileName, "BackgroundFileName", LocalBackgroundFileNameList)
                    task_MAGIX.WriteXMLtagNEW(NewExpXMLFileName, "BackgroundTemperature", LocaltBackList)
                    task_MAGIX.WriteXMLtagNEW(NewExpXMLFileName, "TemperatureSlope", LocaltSlopeList)
                    task_MAGIX.WriteXMLtagNEW(NewExpXMLFileName, "Kappa", LocalDustKappaList)
                    if (NewEmAbsPATH != []):
                        task_MAGIX.WriteXMLtagNEW(NewExpXMLFileName, "EmAbsPATH", NewEmAbsPATH)

                # Debug:
                # print (">>LocalBackgroundFileNameList = ", LocalBackgroundFileNameList)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## start myXCLASSFit function
            if (not dryrun and MoleculeIncludedFlag):


                ## print what you do
                if (LocalPrintFlag):
                    print ("\n\t\tFit molecule:", flush = True)


                ## update path and name of iso-ratio file
                NewIsoRatioFileName = InputDict['NewIsoRatioFileName']
                if (RepID > 0 and NewIsoRatioFileName != ""):
                    task_MAGIX.WriteXMLtagNEW(NewExpXMLFileName, "IsoTableFileName", [NewIsoRatioFileName])


                ## call myXCLASSFit function
                NoSubBeamFlag = True
                JobDir = self.CallmyXCLASSFitFunction(LocalAlgorithmXMLFileName, NameOfCurrentMolfitFile, NewExpXMLFileName, \
                                                      NoSubBeamFlag, LocalFitDir, InputDict)
                ConvertLogLinFlag = "false"
                LocaltBackList = []
                LocaltSlopeList = []
                BestMolfitFile, IsoRatioFileContent, modeldata = task_LineIdentification.GetBestResult(JobDir, ConvertLogLinFlag, NewExpFileList, \
                                                                                                       ListNumberExpRanges, NewFreqMinList, \
                                                                                                       NewFreqMaxList, LocaltBackList, \
                                                                                                       LocaltSlopeList)
                ## store results as pickle file
                LocalDict = {}
                LocalDict['MoleculeName'] = LocalMoleculeName
                LocalDict['BestMolfitFile'] = BestMolfitFile
                LocalDict['IsoRatioFileContent'] = IsoRatioFileContent
                LocalPickelName = ResultPath + "molecule__" + str(MoleculeIndex + 1) + ".pickle"
                pickle.dump(LocalDict, open(LocalPickelName, 'wb'))


                ## copy contents of current job directory to local directory
                cmdString = "mv " + JobDir + "/* " + LocalFitDir + "; "
                cmdString += "rm -rf " + JobDir + "; "
                # cmdString += "rm -f " + LocalFitDir  + "/final_plot.*.png"
                # os.system(cmdString)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## create new fine plots
                if (self.DoMoleculePlotFlag):                                               ## create plot for each transition


                    ## print what you do
                    if (LocalPrintFlag):
                        print ("\t\tCreate new fine plots .. ", end = "", flush = True)


                    ## get path and names of new background files
                    NewBackgroundFileNameList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "BackgroundFileName")


                    ## analyze each frequency range
                    CounterTransWindow = (-1)
                    for ObsDataFileIndex in range(len(NewExpFileList)):                     ## loop over all obs. data files
                        LocalExpFileName = NewExpFileList[ObsDataFileIndex][1]
                        RestFreq = 0.0
                        vLSR = 0.0


                        ## get number of header lines and character separating columns from obs. xml file
                        RangeIndex = (-1)
                        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                    NumberHeaderLinesListIn = NewNumberHeaderLinesList, \
                                                                                    SeparatorColumnsListIn = NewSeparatorColumnsList, \
                                                                                    TelescopeSizeListIn = NewTelescopeSizeList, \
                                                                                    BMIN_ListIn = NewBMINList, BMAJ_ListIn = NewBMAJList, \
                                                                                    BPA_ListIn = NewBPAList, GlobalvLSRListIn = NewGlobalvLSRList, \
                                                                                    Redshift_ListIn = NewRedshiftList, \
                                                                                    InterFlagListIn = NewInter_FlagList, \
                                                                                    NumberRangeListIn = ListNumberExpRanges)
                        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
                        GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
                        if (GlobalvLSR is None):
                            GlobalvLSR = 0.0
                        NumberHeaderLines = ObsXMLParameterDictFile['NumberHeaderLines']
                        if (NumberHeaderLines is None):
                            NumberHeaderLines = 0
                        SeparatorColumns = ObsXMLParameterDictFile['SeparatorColumns']
                        if (SeparatorColumns is None):
                            SeparatorColumns = ""
                        TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
                        BMIN = ObsXMLParameterDictFile['BMIN']
                        BMAJ = ObsXMLParameterDictFile['BMAJ']
                        BPA = ObsXMLParameterDictFile['BPA']
                        InterFlag = ObsXMLParameterDictFile['InterFlag']
                        GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
                        if (GlobalvLSR is None):
                            GlobalvLSR = 0.0
                        Redshift = ObsXMLParameterDictFile['Redshift']


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## import exp data
                        if (SeparatorColumns.strip() == ""):
                            ImportObsData = numpy.loadtxt(LocalExpFileName, skiprows = NumberHeaderLines)
                        else:
                            ImportObsData = numpy.loadtxt(LocalExpFileName, skiprows = NumberHeaderLines, delimiter = SeparatorColumns)


                        ## get parameter for current spectrum
                        t_back_flag = True
                        nH_flag = True


                        ## get parameters for current frequency range
                        for RangeIndex in range(NumberFrequencyRanges):                     ## loop over all range definitions in the whole xml file


                            ## get parameter for current frequency range
                            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                            FreqMinListIn = NewFreqMinList, \
                                                                                            FreqMaxListIn = NewFreqMaxList, \
                                                                                            FreqStepListIn = NewFreqStepList,  \
                                                                                            tBackFlagListIn = Newt_back_flagList, \
                                                                                            tBackListIn = NewtBackList, \
                                                                                            tSlopeListIn = NewtSlopeList, \
                                                                                            BackgroundFileNameListIn = NewBackgroundFileNameList, \
                                                                                            N_HListIn = NewN_HList, \
                                                                                            beta_dustListIn = Newbeta_dustList, \
                                                                                            kappa_1300ListIn = Newkappa_1300List, \
                                                                                            NoiseListIn = NewNoiseList)
                            FreqMin = ObsXMLParameterDictRange['FreqMin']
                            if (FreqMin is not None):
                                CounterTransWindow += 1
                                FreqMax = ObsXMLParameterDictRange['FreqMax']
                                FreqStep = ObsXMLParameterDictRange['FreqStep']
                                t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                                tBack = ObsXMLParameterDictRange['tBack']
                                tSlope = ObsXMLParameterDictRange['tSlope']
                                nH_flag = True
                                N_H = ObsXMLParameterDictRange['N_H']
                                beta_dust = ObsXMLParameterDictRange['beta_dust']
                                kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                                if (N_H is None or beta_dust is None or kappa_1300 is None):
                                    nH_flag = False
                                    N_H = 0.0
                                    beta_dust = 0.0
                                    kappa_1300 = 0.0
                                Noise = ObsXMLParameterDictRange['NoiseLevel']
                                if (Noise is None):
                                    Noise = 0.0
                                BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                                if (BackgroundFileName is None):
                                    BackgroundFileName = ""

                                # Debug:
                                # print ("MolfitsFileName = ", MolfitsFileName)
                                # print ("LocalExpFileName = ", LocalExpFileName)
                                # print ("FreqMin = ", FreqMin)
                                # print ("FreqMax = ", FreqMax)
                                # print ("FreqStep = ", FreqStep)
                                # print ("t_back_flag = ", t_back_flag)
                                # print ("tBack = ", tBack)
                                # print ("tSlope = ", tSlope)
                                # print ("nH_flag = ", nH_flag)
                                # print ("N_H = ", N_H)
                                # print ("beta_dust = ", beta_dust)
                                # print ("kappa_1300 = ", kappa_1300)
                                # print ("molfit file = ", MolfitsFileName)
                                # print ("TelescopeSize = ", TelescopeSize)
                                # print ("Inter_Flag = ", Inter_Flag)
                                # print ("IsoTableFileName = ", IsoTableFileName)
                                # print ("NumberHeaderLines = ", NumberHeaderLines)
                                # print ("SeparatorColumns = ", SeparatorColumns)
                                # print ("\n\n======================================================================================================")


                                ## determine min and max. freq. index of obs. data
                                MinFreqIndexExp, MaxFreqIndexExp = GetArrayPart(ImportObsData[:, 0], FreqMin, FreqMax)


                                ## define local copy of obs. data
                                NewASCIIdata = numpy.zeros((len(ImportObsData), 3), dtype = numpy.float32)
                                NewASCIIdata[:, 0] = ImportObsData[:, 0]                    ## frequencies
                                NewASCIIdata[:, 1] = 0.0                                    ## velocities (will be added later)
                                NewASCIIdata[:, 2] = ImportObsData[:, 1]                    ## intensities
                                #NewASCIIdata[:, 3] = 0.0                                   ## weights


                                ##------------------------------------------------------------------------------------------------------------------------
                                ## determine model function values for current frequency range
                                #    if (NewNumberExpFiles== 1):
                                #        LocalModelValues = modeldata[0]
                                #    else:
                                #        LocalModelValues = modeldata[CounterTransWindow]
                                LocalModelValues = modeldata[CounterTransWindow]

                                # Debug:
                                # print ("LocalModelValues = ", LocalModelValues)
                                # print ("\n\nlen(modeldata) = ", len(modeldata))
                                # print ("len(LocalModelValues) = ", len(LocalModelValues))
                                # print ("NewNumberExpFiles = ", NewNumberExpFiles)


                                ## determine min and max. freq. index of modeled data and copy extracted data to local copy
                                In1, In2 = GetArrayPart(LocalModelValues[:, 0], FreqMin, FreqMax)
                                ModelTransWindow = LocalModelValues[In1:In2, :]


                                ##------------------------------------------------------------------------------------------------------------------------
                                ## get transition frequencies and other molecule parameter from data base
                                f1 = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
                                f2 = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
                                ShiftedFreqMin = min(f1, f2)
                                ShiftedFreqMax = max(f1, f2)
                                TransFreqList, DBParamList = task_LineIdentification.GetTransFreq(LocalMoleculeName, ShiftedFreqMin, ShiftedFreqMax, \
                                                                                                SQLParamArray, dbFilename, MaxNumTransitionsSQL)
                                # Debug:
                                # print ("\n\nLocalMoleculeName = ", LocalMoleculeName)
                                # print ("FreqMin = ", FreqMin)
                                # print ("FreqMax = ", FreqMax)
                                # print ("TransFreqList = ", TransFreqList)
                                # print ("ModelTransWindow = ", ModelTransWindow)


                                ##------------------------------------------------------------------------------------------------------------------------
                                ## construct continuum array
                                if (BackgroundFileName == ""):
                                    ContinuumArray = numpy.zeros((len(ImportObsData[:, 0]), 2), dtype = numpy.float32)
                                    for mm in range(len(ModelTransWindow[:, 0])):
                                        ContinuumArray[mm, 0] = ModelTransWindow[mm, 0]
                                        if (tBack != 0.0):
                                            ContinuumArray[mm, 1] = abs(tBack) * (ContinuumArray[mm, 0] / FreqMin)**tSlope
                                        else:
                                            ContinuumArray[mm, 1] = 0.0
                                else:
                                    ContinuumArray = numpy.loadtxt(BackgroundFileName)


                                ##------------------------------------------------------------------------------------------------------------------------
                                ## analyze each frequency window
                                if (TransFreqList != []):                                   ## continue here, if at least one trans. freq. is found
                                    for TransIndex in range(len(TransFreqList)):            ## loop over all transition frequencies
                                        TransFreq = task_myXCLASS.ConvertFreq(TransFreqList[TransIndex], GlobalvLSR, z = Redshift)
                                        counterPNG += 1                                     ## increase counter for plots
                                        RestFreq = TransFreq                                ## define rest frequency
                                        vLSR = 0.0                                          ## current trans. freq. is at v = 0

                                        # Debug:
                                        # print ("\n\n\n\n")
                                        # print ("FreqMin = ", FreqMin)
                                        # print ("FreqMax = ", FreqMax)
                                        # print ("TransIndex = ", TransIndex)
                                        # print ("TransFreq = ", TransFreq)
                                        # print ("EinsteinA = ", DBParamList[TransIndex][1])
                                        # print ("ElowMin   = ", DBParamList[TransIndex][2])
                                        # print ("gup       = ", DBParamList[TransIndex][3])


                                        ## add new velocity axis to ImportObsData array


                                        ## create velocity axis and determine indices for lower and upper velocity limit and their corresponding frequency
                                        VelocityData = task_myXCLASS.ChangeToVelocitiy(ImportObsData[:, 0], RestFreq, vLSR)
                                        NewASCIIdata[:, 1] = VelocityData[:]
                                        vi1, vi2 = GetArrayPart(VelocityData, RangevelLowLimit, RangevelUpLimit)


                                        ## restrict obs. data and continuum array to selected velocity range
                                        ObsDataTransWindow = NewASCIIdata[vi1:vi2, :]
                                        LocalContinuumArray = ContinuumArray[vi1:vi2, :]


                                        ## restrict modeled spectra array to selected velocity range
                                        f1 = min(NewASCIIdata[vi1, 0], NewASCIIdata[vi2, 0])
                                        f2 = max(NewASCIIdata[vi1, 0], NewASCIIdata[vi2, 0])
                                        fi1, fi2 = GetArrayPart(ModelTransWindow[:, 0], f1, f2)
                                        LocalModelTransWindow = ModelTransWindow[fi1:fi2, :]


                                        ##----------------------------------------------------------------------------------------------------------------
                                        ## create plot
                                        RawDataPlotFlag = False
                                        TransitionParameter = DBParamList[TransIndex]
                                        # NoiseLevel = Noise + tBack
                                        NoiseLevel = 0.0
                                        Localv00 = 0.0
                                        task_LineIdentification.PlotTranstionWindow(RawDataPlotFlag, counterPNG, TransFreq, TransitionParameter, \
                                                                                    NoiseLevel, ObsDataTransWindow, LocalModelTransWindow, LocalPlotDir, \
                                                                                    LocalMoleculeName, velLowLimit = RangevelLowLimit, \
                                                                                    velUpLimit = RangevelUpLimit, vComp = Localv00, \
                                                                                    ContinuumArray = LocalContinuumArray)
                ## we're done
                if (LocalPrintFlag):
                    print ("done!", flush = True)


        ## declare return variable
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## perform molecule fits one by one
    ##
    def UpdatePerMoleculeFit(self, InputDict):
        """

    input parameters:
    -----------------

        - InputDict:                     parameter dictionary



    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("InputDict = ", InputDict)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get some parameters from dictionary
        NumberOfMolecules = InputDict['NumberOfMolecules']
        AllParameters = InputDict['AllParameters']
        MoleculesInMolfitFile = InputDict['MoleculesInMolfitFile']

        # Debug:
        # print ("NumberOfMolecules = ", NumberOfMolecules)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## perform single molecule fits
        for MoleculeIndex in range(NumberOfMolecules):
            LocalMolfitParameters = AllParameters[MoleculeIndex]
            LocalMoleculeName = MoleculesInMolfitFile[MoleculeIndex].strip()
            LowerLocalMoleculeName = LocalMoleculeName.lower()
            if (not LowerLocalMoleculeName.startswith("cont")):
                InputDict['MoleculeIndex'] = MoleculeIndex
                InputDict['LocalMolfitParameters'] = LocalMolfitParameters
                InputDict['LocalMoleculeName'] = LocalMoleculeName


                ## do molecule fit
                self.DoMoleculeFit(InputDict)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## perform single molecule fits
    ##
    def SeparateFits(self, LocalPath, MolfitsFileName, UnmodifiedExpXML, LocalAlgorithmXMLFileName, RangevelLowLimit, RangevelUpLimit, \
                     IncludeAllMolFlag, FirstMoleculeOnlyFlag, NewParameterRanges, LimitRangesFlag, dryrun, RedRangeFlag, LocalMinColumnDensityAbs, \
                     LocalMinColumnDensityEmis, SetNewRangeFlag, DoFinalPlotFlag, NumRep, UpdateFlag, PlotNum, RemoveContFlag, \
                     AddMoleculeNamesToPlotFlag, EmsAbsFlag, backgroundFlag, BackgroundDir, NumberProcessors, NodeList, ClusterFlag, \
                     AdditionMolfitFileContentFileName, FastCompBackFlag):
        """

    input parameters:
    -----------------

        - LocalPath:                            current path

        - MolfitsFileName:                      path and name of molfit file

        - UnmodifiedExpXML:                     path and name of obs. xml-file

        - LocalAlgorithmXMLFileName:            path and name of algorithm xml-file

        - RangevelLowLimit:                     lower velocity limit for fit range

        - RangevelUpLimit:                      upper velocity limit for fit range

        - IncludeAllMolFlag:                    include all molecules flag

        - FirstMoleculeOnlyFlag:                fit only first molecule in molfit file

        - NewParameterRanges:                   list of new ranges

        - LimitRangesFlag:                      flag indicates if modified ranges should stay inside original defined ranges or not

        - dryrun:                               dryrun flag

        - RedRangeFlag:                         reduced frequency range flag

        - LocalMinColumnDensityAbs:             min. column density for foreground components

        - LocalMinColumnDensityEmis:            min. column density for core components

        - SetNewRangeFlag:                      new range flag

        - DoFinalPlotFlag:                      flag for final plot with all molecules

        - NumRep:                               number of repetitions

        - UpdateFlag:                           use optimized molfit parameters for all further molecule fits

        - PlotNum:                              number of subplot

        - RemoveContFlag:                       remove continuum

        - AddMoleculeNamesToPlotFlag:           add names of molecules to plots

        - EmsAbsFlag:                           flag for computing emission / absorption function

        - backgroundFlag:                       background flag

        - BackgroundDir:                        directory containing background files for different molecules

        - NumberProcessors:                     total number of threads

        - NodeList:                             list of nodes

        - ClusterFlag:                          flag for using a cluster of nodes

        - AdditionMolfitFileContentFileName:    path and name of file describing additional information for molfit file

        - FastCompBackFlag:                     flag for fast computing of emission / absorption function at the
                                                beginning of each iteration for all molecules



    output parameters:
    ------------------

        - FinalmolfitFileContent:               content of molfit file

        - FinalIsoRatioFileContents:            content of iso ratio file
        """

        # Debug:
        # print ("LocalPath = ", LocalPath)
        # print ("MolfitsFileName = ", MolfitsFileName)
        # print ("UnmodifiedExpXML = ", UnmodifiedExpXML)
        # print ("LocalAlgorithmXMLFileName = ", LocalAlgorithmXMLFileName)
        # print ("RangevelLowLimit = ", RangevelLowLimit)
        # print ("RangevelUpLimit = ", RangevelUpLimit)
        # print ("IncludeAllMolFlag = ", IncludeAllMolFlag)
        # print ("FirstMoleculeOnlyFlag = ", FirstMoleculeOnlyFlag)
        # print ("NewParameterRanges = ", NewParameterRanges)
        # print ("LimitRangesFlag = ", LimitRangesFlag)
        # print ("dryrun = ", dryrun)
        # print ("RedRangeFlag = ", RedRangeFlag)
        # print ("LocalMinColumnDensityAbs = ", LocalMinColumnDensityAbs)
        # print ("LocalMinColumnDensityEmis = ", LocalMinColumnDensityEmis)
        # print ("SetNewRangeFlag = ", SetNewRangeFlag)
        # print ("DoFinalPlotFlag = ", DoFinalPlotFlag)
        # print ("NumRep = ", NumRep)
        # print ("UpdateFlag = ", UpdateFlag)
        # print ("PlotNum = ", PlotNum)
        # print ("RemoveContFlag = ", RemoveContFlag)
        # print ("AddMoleculeNamesToPlotFlag = ", AddMoleculeNamesToPlotFlag)
        # print ("EmsAbsFlag = ", EmsAbsFlag)
        # print ("backgroundFlag = ", backgroundFlag)
        # print ("BackgroundDir = ", BackgroundDir)
        # print ("NumberProcessors = ", NumberProcessors)
        # print ("NodeList = ", NodeList)
        # print ("ClusterFlag = ", ClusterFlag)
        # print ("AdditionMolfitFileContentFileName = ", AdditionMolfitFileContentFileName)
        # print ("FastCompBackFlag = ", FastCompBackFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## initialize return parameters
        self.FinalmolfitFileContent = []
        self.FinalIsoRatioFileContents = []


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read in parameters for each freq. range for the current molecule
        ExpFileList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "FileNamesExpFiles")
        NumberExpRangesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumberExpRanges")
        FreqMinList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "MinExpRange")
        FreqMaxList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "MaxExpRange")
        FreqStepList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "StepFrequency")
        t_back_flagList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "t_back_flag")
        tBackList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BackgroundTemperature")
        tSlopeList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "TemperatureSlope")
        N_HList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "HydrogenColumnDensity")
        beta_dustList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "DustBeta")
        kappa_1300List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Kappa")
        DustFileNameList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "DustFileName")
        BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BackgroundFileName")
        ContPhenFuncIDList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncID")
        ContPhenFuncParam1List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam1")
        ContPhenFuncParam2List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam2")
        ContPhenFuncParam3List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam3")
        ContPhenFuncParam4List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam4")
        ContPhenFuncParam5List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam5")
        NoiseList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NoiseLevel")
        SmoothList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "SmoothValue")
        TelescopeSizeList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "TelescopeSize")
        BMINList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BMIN")
        BMAJList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BMAJ")
        BPAList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BPA")
        Inter_FlagList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Inter_Flag")
        GlobalvLSRList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "GlobalvLSR")
        RedshiftList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Redshift")
        ErrorYList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ErrorY")
        NumberHeaderLinesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumberHeaderLines")
        SeparatorColumnsList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "SeparatorColumns")
        IsotopologuesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "iso_flag")
        if (IsotopologuesList == []):
            IsotopologuesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Isotopologues")
        if (len(IsotopologuesList) > 0):
            IsoFlag = IsotopologuesList[0].lower()
            IsoFlag = task_myXCLASSMapFit.CheckBool(IsoFlag)
        IsoTableFileNameList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "IsoTableFileName")
        if (IsoTableFileNameList == []):
            IsoTableFileName = None
            OrigIsoRatioFileContent = []
        else:
            IsoTableFileName = IsoTableFileNameList[0].strip()
            if (IsoTableFileName != ""):
                IsoRatioFile = open(IsoTableFileName)
                OrigIsoRatioFileContent = IsoRatioFile.readlines()
                IsoRatioFile.close()
            else:
                OrigIsoRatioFileContent = []
        dbList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "dbFilename")
        if (dbList == []):
            dbFile = task_myXCLASS.GetDefaultDBFile()
            dbList = [dbFile]
        dbFile = dbList[0]
        NumModelPixelXXList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumModelPixelXX")
        if (NumModelPixelXXList == []):
            NumModelPixelXX = None
        else:
            NumModelPixelXX = NumModelPixelXXList[0]
        NumModelPixelYYList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumModelPixelYY")
        if (NumModelPixelYYList == []):
            NumModelPixelYY = None
        else:
            NumModelPixelYY = NumModelPixelYYList[0]
        LocalOverlapFlagList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "LocalOverlap_Flag")
        if (LocalOverlapFlagList == []):
            LocalOverlapFlag = None
        else:
            LocalOverlapFlag = LocalOverlapFlagList[0]
        EmAbsPATHList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "EmAbsPATHList")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze file describing additional content
        AdditionMolfitFileContent = []
        if (AdditionMolfitFileContentFileName.strip() != ""):
            AdditionMolfitFileContentFile = open(AdditionMolfitFileContentFileName)
            AdditionMolfitFileContent = AdditionMolfitFileContentFile.readlines()
            AdditionMolfitFileContentFile.close()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create dictionary containing obs. xml parameters and many more
        InputDict = {}


        ## store obs. xml file parameters
        InputDict['ExpFileList'] = ExpFileList
        InputDict['NumberExpRangesList'] = NumberExpRangesList
        InputDict['FreqMinList'] = FreqMinList
        InputDict['FreqMaxList'] = FreqMaxList
        InputDict['FreqStepList'] = FreqStepList
        InputDict['t_back_flagList'] = t_back_flagList
        InputDict['tBackList'] = tBackList
        InputDict['tSlopeList'] = tSlopeList
        InputDict['N_HList'] = N_HList
        InputDict['beta_dustList'] = beta_dustList
        InputDict['kappa_1300List'] = kappa_1300List
        InputDict['DustFileNameList'] = DustFileNameList
        InputDict['BackgroundFileNameList'] = BackgroundFileNameList
        InputDict['ContPhenFuncIDList'] = ContPhenFuncIDList
        InputDict['ContPhenFuncParam1List'] = ContPhenFuncParam1List
        InputDict['ContPhenFuncParam2List'] = ContPhenFuncParam2List
        InputDict['ContPhenFuncParam3List'] = ContPhenFuncParam3List
        InputDict['ContPhenFuncParam4List'] = ContPhenFuncParam4List
        InputDict['ContPhenFuncParam5List'] = ContPhenFuncParam5List
        InputDict['NoiseList'] = NoiseList
        InputDict['SmoothList'] = SmoothList
        InputDict['TelescopeSizeList'] = TelescopeSizeList
        InputDict['BMINList'] = BMINList
        InputDict['BMAJList'] = BMAJList
        InputDict['BPAList'] = BPAList
        InputDict['Inter_FlagList'] = Inter_FlagList
        InputDict['GlobalvLSRList'] = GlobalvLSRList
        InputDict['RedshiftList'] = RedshiftList
        InputDict['ErrorYList'] = ErrorYList
        InputDict['NumberHeaderLinesList'] = NumberHeaderLinesList
        InputDict['SeparatorColumnsList'] = SeparatorColumnsList
        InputDict['IsotopologuesList'] = IsotopologuesList
        InputDict['IsoTableFileNameList'] = IsoTableFileNameList
        InputDict['IsoTableFileName'] = IsoTableFileName
        InputDict['IsoFlag'] = IsoFlag
        InputDict['OrigIsoRatioFileContent'] = OrigIsoRatioFileContent
        InputDict['dbList'] = dbList
        InputDict['dbFile'] = dbFile
        InputDict['NumModelPixelXXList'] = NumModelPixelXXList
        InputDict['NumModelPixelXX'] = NumModelPixelXX
        InputDict['NumModelPixelYYList'] = NumModelPixelYYList
        InputDict['NumModelPixelYY'] = NumModelPixelYY
        InputDict['LocalOverlapFlagList'] = LocalOverlapFlagList
        InputDict['LocalOverlapFlag'] = LocalOverlapFlag
        InputDict['EmAbsPATHList'] = EmAbsPATHList


        ## store additional parameters
        InputDict['UnmodifiedExpXML'] = UnmodifiedExpXML
        InputDict['LocalAlgorithmXMLFileName'] = LocalAlgorithmXMLFileName
        InputDict['RangevelLowLimit'] = RangevelLowLimit
        InputDict['RangevelUpLimit'] = RangevelUpLimit
        InputDict['IncludeAllMolFlag'] = IncludeAllMolFlag
        InputDict['NewParameterRanges'] = NewParameterRanges
        InputDict['LimitRangesFlag'] = LimitRangesFlag
        InputDict['dryrun'] = dryrun
        InputDict['RedRangeFlag'] = RedRangeFlag
        InputDict['LocalMinColumnDensityAbs'] = LocalMinColumnDensityAbs
        InputDict['LocalMinColumnDensityEmis'] = LocalMinColumnDensityEmis
        InputDict['SetNewRangeFlag'] = SetNewRangeFlag
        InputDict['UpdateFlag'] = UpdateFlag
        InputDict['EmsAbsFlag'] = EmsAbsFlag
        InputDict['FastCompBackFlag'] = FastCompBackFlag
        InputDict['backgroundFlag'] = backgroundFlag
        InputDict['BackgroundDir'] = BackgroundDir
        InputDict['NewIsoRatioFileName'] = ""
        InputDict['AdditionMolfitFileContent'] = AdditionMolfitFileContent


        ## parameters for plotting
        InputDict['DoFinalPlotFlag'] = DoFinalPlotFlag
        InputDict['PlotNum'] = PlotNum
        InputDict['RemoveContFlag'] = RemoveContFlag
        InputDict['AddMoleculeNamesToPlotFlag'] = AddMoleculeNamesToPlotFlag

        # Debug:
        # print ("InputDict = ", InputDict)


        ## clean up previous directories
        ResultPath = LocalPath + "Separate-Molecule-Fits/"
        cmdString = "rm -rf " + ResultPath
        os.system(cmdString)


        ##================================================================================================================================================
        ## start iterations
        NewIsoRatioFileName = ""
        self.FinalIsoRatioFileContents = copy.deepcopy(OrigIsoRatioFileContent)
        for RepID in range(NumRep):
            InputDict['RepID'] = RepID


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## create plot subdirectory
            ResultPath = LocalPath + "Separate-Molecule-Fits__" + str(RepID + 1) + "/"
            cmdString = "rm -rf " + ResultPath + "; "
            cmdString += "mkdir " + ResultPath
            os.system(cmdString)
            InputDict['ResultPath'] = ResultPath


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## analyze given molfit file
            if (self.FinalmolfitFileContent == []):
                MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(MolfitsFileName)
            else:
                MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(self.FinalmolfitFileContent)
            InputDict['MolfitFileForEachMolecule'] = MolfitFileForEachMolecule
            InputDict['AllParameters'] = AllParameters
            InputDict['MoleculesInMolfitFile'] = MoleculesInMolfitFile

            # Debug:
            # print ("\nMolfitsFileName = ", MolfitsFileName)
            # print ("MoleculesInMolfitFile = ", MoleculesInMolfitFile)


            ## check, if continuum is described in molfit file
            ContInMolfitFileFlag = False
            for LocalMolfitMol in MoleculesInMolfitFile:
                LowerLocalMolfitMol = LocalMolfitMol.lower()
                if (LowerLocalMolfitMol.startswith("cont-")):
                    ContInMolfitFileFlag = True
                    break
            InputDict['ContInMolfitFileFlag'] = ContInMolfitFileFlag


            ## get sql parameters
            if (self.FinalmolfitFileContent == []):
                MinNumTransSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(MolfitsFileName)
            else:
                MinNumTransSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(self.FinalmolfitFileContent)
            SQLParamArray = [MinNumTransSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL]
            MolfitHeaderLinesList = task_myXCLASS.WriteSQLParameter(MinNumTransSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)
            InputDict['SQLParamArray'] = SQLParamArray
            InputDict['MaxElowSQL'] = MaxElowSQL
            InputDict['MaxNumTransitionsSQL'] = MaxNumTransitionsSQL
            InputDict['MolfitHeaderLinesList'] = MolfitHeaderLinesList

            # Debug:
            # print ("SQLParamArray = ", SQLParamArray)
            # print ("MolfitHeaderLinesList = ", MolfitHeaderLinesList)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## define number of molecules
            if (FirstMoleculeOnlyFlag):
                NumberOfMolecules = 1
            else:
                NumberOfMolecules = len(MoleculesInMolfitFile)
            InputDict['NumberOfMolecules'] = NumberOfMolecules


            ## use cluster?
            if (NumberProcessors > 1 and (NumberOfMolecules == 1 or FirstMoleculeOnlyFlag)):
                NumberProcessors = 1
                ClusterFlag = False
            InputDict['NumberProcessors'] = NumberProcessors
            InputDict['ClusterFlag'] = ClusterFlag
            InputDict['LocalNode'] = "localhost"
            InputDict['LocalNodePath'] = ""

            # Debug:
            # print ("NumberOfMolecules = ", NumberOfMolecules)
            # print ("NumberProcessors = ", NumberProcessors)
            # print ("NodeList = ", NodeList)
            # print ("ClusterFlag = ", ClusterFlag)
            # sys.exit(0)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## fit molecules


            ## if update flag selected perform molecule fits one by one
            if (UpdateFlag or NumberProcessors == 1):
                LocalPrintFlag = True
                InputDict['LocalPrintFlag'] = LocalPrintFlag
                self.UpdatePerMoleculeFit(InputDict)


            ## if update flag is not selected perform molecule fits in parallel
            else:
                LocalPrintFlag = True
                InputDict['LocalPrintFlag'] = LocalPrintFlag


                ## do a fast computation of the emission and absorption functions
                if (FastCompBackFlag):
                    self.StartClusterParallelization(NodeList, ClusterFlag, InputDict, FastCompBackFlag)


                ## fit molecules
                self.StartClusterParallelization(NodeList, ClusterFlag, InputDict, False)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## create overall molfit file containing fit results of all molecules
            SMFResultsList = {}
            if (not dryrun):


                ## read in results from molecule fits
                listing = os.listdir(ResultPath)
                ListOfMolecules = []
                for LocalFile in listing:
                    if (LocalFile.endswith(".pickle")):
                        LocalDict = pickle.load(open(ResultPath + LocalFile, "rb"))
                        DictMolName = LocalDict['MoleculeName']
                        DictBestMolfitFile = LocalDict['BestMolfitFile']
                        DictIsoRatioFileContent = LocalDict['IsoRatioFileContent']
                        ListOfMolecules.append(DictMolName)
                        SMFResultsList[DictMolName] = [DictBestMolfitFile, DictIsoRatioFileContent]


                ## remove pickle file
                cmdString = "rm -rf " + ResultPath + "*.pickle"
                os.system(cmdString)


                ## open molfit file containing all optimized molfit and add SQL parameter definitions
                if (SMFResultsList != {}):
                    NewOverallMolfitFileName = ResultPath + "New_Single-Molecule-Fits.molfit"
                    NewOverallMolfitFile = open(NewOverallMolfitFileName, 'w')
                    for line in MolfitHeaderLinesList:
                        NewOverallMolfitFile.write(line)


                    ## add additional content
                    if (AdditionMolfitFileContent != []):
                        for line in AdditionMolfitFileContent:
                            NewOverallMolfitFile.write(line)


                    ## add molecule content
                    ListKnownCont = []
                    for LocalMol in ListOfMolecules:
                        LocalMolfitFileContent = SMFResultsList[LocalMol][0]
                        SMFMoleculesInMolfitFile, SMFAllParameters, SMFMolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(LocalMolfitFileContent)
                        for SMFMoleculeIndex, SMFMolecule in enumerate(SMFMoleculesInMolfitFile):
                            LowerSMFMolecule = SMFMolecule.lower()
                            AddFlag = True
                            if (LowerSMFMolecule.startswith("cont") or LowerSMFMolecule.startswith("rrl")):
                                if (not SMFMolecule in ListKnownCont):
                                    ListKnownCont.append(SMFMolecule)
                                else:
                                    AddFlag = False
                            if (AddFlag):
                                for line in SMFMolfitFileForEachMolecule[SMFMoleculeIndex]:
                                    NewOverallMolfitFile.write(line)
                    NewOverallMolfitFile.close()


                    ## analyze new overall molfit file content
                    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(NewOverallMolfitFileName)


                    ## remove components with column densities below given thresholds
                    task_LineIdentification.CreateMolfitFileInNewFormat(NewOverallMolfitFileName, MolfitFileForEachMolecule, LocalMinColumnDensityAbs, \
                                                                        LocalMinColumnDensityEmis)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## write new iso ratio file
                NewIsoRatioFileName = ""
                if (IsoFlag):


                    ## create final iso ratio file
                    NewIsoRatioFileContents = []
                    NewRatios = []
                    for LocalMol in ListOfMolecules:
                        LocalIsoRatioFileContent = SMFResultsList[LocalMol][1]
                        for line in LocalIsoRatioFileContent:
                            StrippedLine = line.strip()
                            SplittedLine = StrippedLine.split()
                            LocalIso = SplittedLine[0].strip()
                            LocalMaster = SplittedLine[1].strip()
                            if (not ([LocalIso, LocalMaster] in NewRatios)):
                                NewRatios.append([LocalIso, LocalMaster])
                                NewIsoRatioFileContents.append(line)


                    ## write content to file
                    if (NewIsoRatioFileContents != []):
                        NewIsoRatioFileName = ResultPath + "New_Iso-Ratio-File.dat"
                        InputDict['NewIsoRatioFileName'] = NewIsoRatioFileName
                        NewIsoRatioFile = open(NewIsoRatioFileName, 'w')
                        for line in NewIsoRatioFileContents:
                            NewIsoRatioFile.write(line)
                        NewIsoRatioFile.close()
                        self.FinalIsoRatioFileContents = copy.deepcopy(NewIsoRatioFileContents)


                ## import molfit file
                NewOverallMolfitFile = open(NewOverallMolfitFileName)
                NewOverallMolfitFileContent = NewOverallMolfitFile.readlines()
                NewOverallMolfitFile.close()
                self.FinalmolfitFileContent = copy.deepcopy(NewOverallMolfitFileContent)

                # Debug:
                # print ("self.FinalmolfitFileContent = ", self.FinalmolfitFileContent)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## do final plot
                if (DoFinalPlotFlag and SMFResultsList != {}):


                    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                    ## remove sql definitions in overall-molfit file


                    ## create final subdirectory
                    BasePath = ResultPath + "final-plot/"
                    cmdString = "mkdir -p " + BasePath
                    os.system(cmdString)


                    ## write new molfit file and remove sql definitions
                    LocalOverallMolfitFileName = ResultPath + "Overall-Molecule-Fit.molfit"
                    NewOverallMolfitFile = open(LocalOverallMolfitFileName, 'w')
                    for line in NewOverallMolfitFileContent:
                        i = line.find("%")
                        if (i > (-1)):
                            StrippedLine = line[:i]
                        else:
                            StrippedLine = line
                        if (StrippedLine != ""):
                            NewOverallMolfitFile.write(StrippedLine)
                    NewOverallMolfitFile.close()


                    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                    ## initialize plot class
                    DefaultDB = task_myXCLASS.GetDefaultDBFile()
                    AdditionalFiles = []
                    JobDir = ""
                    PlotOption = ""
                    LocalPlotClass = task_myXCLASSPlot.PlotClass(LocalOverallMolfitFileName, UnmodifiedExpXML, PlotNum, RemoveContFlag, \
                                                                 AddMoleculeNamesToPlotFlag, DefaultDB, AdditionalFiles, JobDir, PlotOption, \
                                                                 NewIsoRatioFileName, BasePath)
                    ## call plot function
                    LocalPlotClass.PlotmyXCLASS()


                    ## plot contributions of each molecule
                    # LocalPlotClass.PlotMoleculeContributions()


                    ## remove temp molfit file
                    cmdString = "rm -rf " + LocalOverallMolfitFileName
                    os.system(cmdString)


            ##============================================================================================================================================
            ## move output directory and define new molfit and obs. xml file
            #if (NumRep > 1):


                ## move new molfit file
                #MolfitsFileName = NewOverallMolfitFileName
                # if (NewIsoRatioFileName != ""):
                #     s


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## rename output directory if only no repetition is defined
        if (NumRep == 1):
            CurrDir = LocalPath + "Separate-Molecule-Fits/"
            NewDir = LocalPath + "Separate-Molecule-Fits__1/"
            cmdString = "mv " + NewDir + " " + CurrDir
            os.system(cmdString)
        else:
            CurrDir = LocalPath + "Separate-Molecule-Fits/"
            NewDir = LocalPath + "Separate-Molecule-Fits__*/"
            cmdString = "mkdir -p " + CurrDir + "; "
            cmdString += "mv " + NewDir + " " + CurrDir
            os.system(cmdString)


        ## define return parameters
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Simplified CASA interface for MAGIX with myXCLASS program
##
def myXCLASSFitCore(MolfitsFileName = "", MolfitFileName = None, \
                    ObsXMLFileName = "", ObsDataFileName = None, experimentalData = None, observationalData = None, \
                    NumberIteration = 10, AlgorithmXMLFileName = "", Optimizer = "magix", FastFlag = False, \
                    ChannelIntegrationFlag = True, \
                    TelescopeSize = None, BMIN = None, BMAJ = None, BPA = None, Redshift = 0.0, Inter_Flag = False, \
                    t_back_flag = True, tBack = 0.0, tSlope = 0.0, BackgroundFileName = "", \
                    nH_flag = None, N_H = 0.0, beta_dust = 0.0, kappa_1300 = 0.0, DustFileName = "", \
                    ContPhenFuncID = None, ContPhenFuncParam1 = None, ContPhenFuncParam2 = None, \
                    ContPhenFuncParam3 = None, ContPhenFuncParam4 = None, ContPhenFuncParam5 = None, \
                    iso_flag = False, IsoTableFileName = "", EmAbsPATH = None, \
                    NoSubBeamFlag = None, NumModelPixelXX = 100, NumModelPixelYY = 100, LocalOverlapFlag = None, \
                    RestFreq = 0.0, vLSR = 0.0, DictOutFlag = False, DicOutFlag = None):
    """

Simplified interface for MAGIX with myXCLASS program:
-----------------------------------------------------


This function provides a simplified interface for MAGIX using the myXCLASS program. The
function starts MAGIX using the Levenberg-Marquardt algorithm to fit experimental data
with the myXCLASS program. The user has to specify the max. number of iterations, the
experimental data (or path and name of the experimental xml-file), and the path and
name of the molfit file.

For each run the myXCLASSFit function creates a so-called job directory located in
the run directory ("path-of-myXCLASS-CASA-Interface/run/myXCLASSFit/") where all
files created by the myXCLASSFit function are stored in. The name of this job
directory is made up of four components: The first part of the name consists of the
phrase "job_" whereas the second part describes the date (day, month, year), the third
part the time stamp (hours, minutes, seconds) of the function execution. The last
part describes a so-called job ID which is composed of the so-called PID followed
by a four digit random integer number to create a really unambiguous job number, e.g.
"path-to-myXCLASS-Interface/run/myXCLASSFit/job__25-07-2013__12-02-03__189644932/"

Before the fit procedure starts, the function copies the molfits, the experimental data,
and xml-files to the myXCLASSFit job directory. The path(s) defined in the observational
xml-file are adjusted so that these paths point to the current myXCLASSFit job directory.

Please note, that the original experimental xml- and data files are not modified. Only the
copy of the experimental xml file located in the myXCLASSFit job directory is modified!

If no experimental xml-file is defined, the myXCLASSFit function creates an experimental
xml-file containing the settings for the different parameters in the myXCLASSFit job
directory. Additionally, the function creates an ASCII file located in the myXCLASSFit
job directory containing the experimental data given by the parameter "experimentalData".
If the parameter "experimentalData" defines the path and the name of an experimental
data file, then the data file is copied to the myXCLASSFit job directory as well.
The path and the name of this ASCII file is pasted in the created experimental xml file.
Furthermore, the function creates automatically the algorithm and i/o control xml files,
need by MAGIX, so that the user does not need to edit any xml-file.


input parameters:
-----------------

    - MolfitsFileName:      path and name of the extended molfit file, (default: "").

    - MolfitFileName:       (optional) alias of MolfitsFileName, (default: None).

    - NumberIteration:      max. number of iterations (default: 50).

    - AlgorithmXMLFileName: only necessary, if the user wants to use another fit
                            algorithm (than Levenberg-Marquardt) for fitting. Therefore,
                            the path and name of a MAGIX xml-file defining settings
                            for an algorithm or algorithm chain has to be given. (A
                            relative path has to be defined relative to the current
                            working directory!)

                            NOTE, if the user specify a xml file, the number of
                            iterations given by the parameter "NumberIteration" is
                            ignored. The number of iteration is then given by the xml
                            file. In order to use the implemented fit algorithm
                            (Levenberg-Marquardt) clear the AlgorithmXMLFileName parameter,
                            i.e. AlgorithmXMLFileName = "", and define the max. number of
                            iterations by using parameter "NumberIteration".

    - Optimizer:            (optional) package used for optimization ('magix' or 'scipy'),
                            (default: 'magix')

    - FastFlag:             (optional) fast flag for scipy optimizer (default: False)

    - ChannelIntegrationFlag: (optional, 'scipy' only) flag for using channel integration,  (default: True)

    - DictOutFlag:          (optional) collect all return parameters in a dictionary (default: False)

    - DicOutFlag:           (optional) alias for DictOutFlag (default: None)

    - ObsXMLFileName:       (optional) path and name of obs. xml file
                            (default: "")

    - ObsDataFileName:      (optional) path and name of an ASCII file called observational
                            data file, where the first column describe the frequency (in MHz)
                            and the second column the brightness temperature (in Kelvin),
                            (default: None)

    - experimentalData:     (optional) alias for ObsDataFileName (default: None)

    - observationalData:    (optional) alias for ObsDataFileName (default: None)

    The following parameters are needed, if parameter ObsDataFileName is used instead of
    parameter ObsXMLFileName:

    - vLSR:                 local standard of rest velocity (in km/s), (default 0).

    - TelescopeSize:        for single dish observations (Inter_Flag = False): TelescopeSize
                            describes the size of telescope (in m), (default: 1); for
                            interferometric observations (Inter_Flag = True): TelescopeSize
                            describes the interferometric beam FWHM size (in arcsec),
                            (default: 1).

    - Inter_Flag:           defines, if single dish ("False") or interferometric
                            observations ("True") are described, (default: "False").

    - BMIN:                 (optional) Beam minor axis length (default: None)

    - BMAJ:                 (optional) Beam major axis length (default: None)

    - BPA:                  (optional) Beam position angle (default: None)

    - Redshift:             (optional) red shift (dimensionless), (default: None)

    - t_back_flag:          ("True"/"False"), defines, if the user defined background
                            temperature Tbg and temperature slope Tslope given
                            by the input parameters tBack and tSlope describe the
                            continuum contribution completely (t_back_flag = "True")
                            or not (t_back_flag = "False") (default: "True").

    - tBack:                background temperature (in K), (default: 0).

    - tSlope:               temperature slope (dimensionless), (default: 0).

    - N_H:                  (optional) Hydrogen column density (in cm^{-2}), (default: None).

    - beta_dust:            (optional) beta for dust (dimensionless), (default: None).

    - kappa_1300:           (optional) kappa (cm^2 g^{-1}), (default: None).

    - DustFileName:         (optional) path and name of file describing the optical depth
                            of dust (default: "").

    - BackgroundFileName:   (optional) path and name of file describing background
                            (default: "").

    - ContPhenFuncID:       (optional) describes which phenomenological function is used
                            to describe the continuum (default: None).

    - ContPhenFuncParam1:   (optional) first parameter for phenomenological function
                            describing the continuum (default: None).

    - ContPhenFuncParam2:   (optional) second parameter for phenomenological function
                            describing the continuum (default: None).

    - ContPhenFuncParam3:   (optional) third parameter for phenomenological function
                            describing the continuum (default: None).

    - ContPhenFuncParam4:   (optional) fourth parameter for phenomenological function
                            describing the continuum (default: None).

    - ContPhenFuncParam5:   (optional) fifth parameter for phenomenological function
                            describing the continuum (default: None).

    - iso_flag:             use isotopologues ("True"/"False"). If iso_flag is set to
                            "False" the ratios are all set to 1 (default: "False").

    - IsoTableFileName:     path and name of an ASCII file including the iso ratios
                            between certain molecules. The so-called "iso ratio file"
                            defines the iso ratios between molecules. The ASCII file
                            consists of three columns, where the first two columns
                            indicates the molecules, respectively. The third column
                            defines the ratio for both molecules. The columns are
                            separated by blanks or tabs. So, the names of the
                            molecules must not contain blanks.

                            The myXCLASSFit function offers the possibility to
                            optimize the ratios between isotopologues as well. For
                            that purpose, the user has to add two additional columns
                            on the right indicating the lower and the upper limit
                            of a certain ratio, respectively. For more information
                            please look at the description of the iso ratio file given
                            for the myXCLASS function.

                            If the lower and upper limit are equal or if the lower
                            limit is higher than the upper limit, the ratio is kept
                            constant and is not optimized by the myXCLASSFit function.

                            NOTE, if the parameter "IsoTableFileName" defines a relative
                            path, the path has to be defined relative to the current
                            working directory!

    - NumModelPixelXX:      (optional) used for sub-beam modeling, describes the number
                            of pixels used in x-direction (default: None).

    - NumModelPixelYY:      (optional) used for sub-beam modeling, describes the number
                            of pixels used in y-direction (default: None).

    - LocalOverlapFlag:     (optional) flag indicates if local-overlap description is
                            used or not (default: None).

    - NoSubBeamFlag:        (optional) prevent sub beam description (default: None).

    - EmAbsPATH:            (optional) path containing files describing functions
                            emission / absorption (default: None)

    The following parameter is needed, to add a velocity axis to the output
    array OptimizedSpectra, see below:

    - RestFreq:             rest frequency in MHz (default: 0). (If this parameter is
                            set to zero, the intensity is plotted against frequency (in MHz)
                            otherwise against velocity (in km/s).


Output parameters:
------------------

    - OptimizedInputFile:   the contents of the molfit file containing the
                            parameters for the best fit.

    - OptimizedSpectra:     the model function values for each data point, which
                            correspond to the best fit.

    - JobDir:               absolute path of the job directory created for the current run.


    if DictOutFlag is set to True, the return parameter is

    - OutDict:              python dictionary containing the following items:

                            - OutputDict["JobDir"]:                         path of job directory
                            - OutputDict["MolfitFileNames"]:                list of molfit file names
                            - OutputDict["MolfitFilesContent"]:             list of molfit file content
                            - OutputDict["IsoRatioFileNames"]:              list of iso ratio file names
                            - OutputDict["IsoRatioFilesContent"]:           list of iso ratio file content
                            - OutputDict["ListOfSpectraFileNames"]:         list of spectra file names
                            - OutputDict["ListOfSpectraFilesContent"]:      list of optimized spectra
                            - OutputDict["BestMolfitFileName"]:             best molfit file name
                            - OutputDict["BestMolfitFileContent"]:          best molfit file content
                            - OutputDict["BestIsoRatioFileName"]:           best iso ratio file name
                            - OutputDict["BestIsoRatioFileContent"]:        best iso ratio file content
                            - OutputDict["BestSpectraFileNames"]:           path and name of file describing spectra
                            - OutputDict["BestSpectraFilesContent"]:        spectra of best fit


Note, the user is free to define different names for the output parameters.



Example using a molfit file in the old format and an ASCII file containing the experimental data:
-------------------------------------------------------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSFit/CH3OH__old.molfit"
experimentalData = "demo/myXCLASSFit/band1b.dat"
vLSR = 0.0
TelescopeSize = 3.5
Inter_Flag = False
t_back_flag = True
tBack = 1.1
tslope = 0.0
nH_flag = True
N_H = 3.0E+24
beta_dust = 2.0
kappa_1300 = 0.02
iso_flag = False
IsoTableFileName = ""
RestFreq = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()


Example NGC6334 with MAGIX xml files (old format):
--------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSFit/CH3OH__old.molfit"
experimentalData = "demo/myXCLASSFit/observation.xml"
RestFreq = 0.0
vLSR = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()


Example NGC6334 with MAGIX xml files (new format):
--------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSFit/CH3OH__new.molfit"
experimentalData = "demo/myXCLASSFit/observation.xml"
RestFreq = 0.0
vLSR = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()


Example NGC6334 with MAGIX xml files (other algorithm):
-------------------------------------------------------

MolfitsFileName = "demo/myXCLASSFit/CH3OH__new.molfit"
experimentalData = "demo/myXCLASSFit/observation.xml"
AlgorithmXMLFile = "demo/myXCLASSFit/algorithm-settings.xml"
RestFreq = 0.0
vLSR = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()

    """

    # Debug:
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("MolfitFileName = ", MolfitFileName)
    # print ("ObsXMLFileName = ", ObsXMLFileName)
    # print ("ObsDataFileName = ", ObsDataFileName)
    # print ("experimentalData = ", experimentalData)
    # print ("observationalData = ", observationalData)
    # print ("NumberIteration = ", NumberIteration)
    # print ("AlgorithmXMLFile = ", AlgorithmXMLFile)
    # print ("Optimizer = ", Optimizer)
    # print ("FastFlag = ", FastFlag)
    # print ("TelescopeSize = ", TelescopeSize)
    # print ("BMIN = ", BMIN)
    # print ("BMAJ = ", BMAJ)
    # print ("BPA = ", BPA)
    # print ("Redshift = ", Redshift)
    # print ("Inter_Flag = ", Inter_Flag)
    # print ("t_back_flag = ", t_back_flag)
    # print ("tBack = ", tBack)
    # print ("tSlope = ", tSlope)
    # print ("nH_flag = ", nH_flag)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("iso_flag = ", iso_flag)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)
    # print ("EmAbsPATH = ", EmAbsPATH)
    # print ("RestFreq = ", RestFreq)
    # print ("vLSR = ", vLSR)
    # print ("ChannelIntegrationFlag = ", ChannelIntegrationFlag)
    # print ("DictOutFlag = ", DictOutFlag)
    # print ("DicOutFlag = ", DicOutFlag)


    ##====================================================================================================================================================
    ## initialize function


    ## define name of function
    NameOfFunction = "myXCLASSFit"


    ## define print flag
    printflag = True


    ## get current working directory
    CurrentDir = os.getcwd() + "/"


    ## get default database file
    DefaultDBFileName = task_myXCLASS.GetDefaultDBFile()


    ## initialize return parameters
    OutputDict = {}
    OptimizedInputFile = []
    OptimizedSpectra = []
    myXCLASSFitJobDir = ""


    ##====================================================================================================================================================
    ## create run directory for the current function call
    myXCLASSrootDir = task_myXCLASS.GetmyXCLASSRootDir()                                    ## get absolute path of myXCLASS root directory
    myXCLASSFitJobDir = task_MAGIX.CreateRunDirectory(NameOfFunction, myXCLASSrootDir, printflag)

    # Debug:
    # print ("myXCLASSrootDir = ", myXCLASSrootDir)
    # print ("myXCLASSFitJobDir = ", myXCLASSFitJobDir)


    ##====================================================================================================================================================
    ## get alias for keyword "MolfitFileName"
    MolfitsFileName = MolfitsFileName.strip()
    if (MolfitsFileName == "" and MolfitFileName is not None):
        MolfitsFileName = MolfitFileName.strip()


    ##====================================================================================================================================================
    ## get alias for DictOutFlag
    if (DicOutFlag is not None):
        DictOutFlag = DicOutFlag

    # Debug:
    # print ('DictOutFlag = ', DictOutFlag)


    ##====================================================================================================================================================
    ## check optimizer keyword


    ## get NoMAGIX flag from init file
    OutDict = task_myXCLASS.GetXCLASSSettings()
    NoMAGIXFlag = OutDict['NoMAGIXFlag']
    if (NoMAGIXFlag):
        Optimizer = "scipy"


    ## analyze optimizer key word
    Optimizer = Optimizer.strip()
    Optimizer = Optimizer.lower()
    if (not Optimizer in ["magix", "scipy"]):
        print ("\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
        print ("\n\t\t The selected optimizer is unknown!\n")
        print ("\n\n\t\t Please use {:s} or {:s} and restart function!".format(chr(34) + "magix" + chr(34), chr(34) + "scipy" + chr(34)))
        print ("\n\n")
        sys.exit(0)

    # Debug:
    # print ("Optimizer = ", Optimizer)


    ##====================================================================================================================================================
    ## analyze input parameter regarding obs. data


    ## print warning messages for deprecate input parameters
    if (observationalData is not None):
        warn('observationalData is going to be deprecated', DeprecationWarning, stacklevel = 2)
    if (nH_flag is not None):
        warn('nH_flag is going to be deprecated', DeprecationWarning, stacklevel = 2)


    ## analyze input parameters and create obs. xml file if necessary
    LocalObsXMLFileName = CheckObsInput(ObsXMLFileName, \
                                        ObsDataFileName, \
                                        experimentalData, \
                                        NameOfFunction, \
                                        CurrentDir, \
                                        myXCLASSFitJobDir, \
                                        t_back_flag = t_back_flag, \
                                        tBack = tBack,
                                        tSlope = tSlope, \
                                        N_H = N_H, \
                                        beta_dust = beta_dust, \
                                        kappa_1300 = kappa_1300, \
                                        DustFileName = DustFileName, \
                                        BackgroundFileName = BackgroundFileName, \
                                        ContPhenFuncID = ContPhenFuncID, \
                                        ContPhenFuncParam1 = ContPhenFuncParam1, \
                                        ContPhenFuncParam2 = ContPhenFuncParam2, \
                                        ContPhenFuncParam3 = ContPhenFuncParam3, \
                                        ContPhenFuncParam4 = ContPhenFuncParam4, \
                                        ContPhenFuncParam5 = ContPhenFuncParam5, \
                                        TelescopeSize = TelescopeSize, \
                                        BMIN = BMIN, \
                                        BMAJ = BMAJ, \
                                        BPA = BPA, \
                                        Inter_Flag = Inter_Flag, \
                                        vLSR = vLSR, \
                                        Redshift = Redshift, \
                                        DBFileName = DefaultDBFileName, \
                                        NumModelPixelXX = NumModelPixelXX, \
                                        NumModelPixelYY = NumModelPixelYY, \
                                        LocalOverlapFlag = LocalOverlapFlag, \
                                        NoSubBeamFlag = NoSubBeamFlag, \
                                        EmAbsPATH = EmAbsPATH, \
                                        iso_flag = iso_flag, \
                                        IsoTableFileName = IsoTableFileName, \
                                        PrintFlag = True, \
                                        CopyFlag = True)
    ## did an error occur ?
    if (LocalObsXMLFileName is None):
        if (DictOutFlag):
            return OutputDict
        else:
            return (OptimizedInputFile, OptimizedSpectra, myXCLASSFitJobDir)


    ##====================================================================================================================================================
    ## if no algorithm xml file is defined, create one
    AlgorithmXMLFileName = AlgorithmXMLFileName.strip()
    if (AlgorithmXMLFileName == ""):
        print ("Creating fit.xml file ..", end = ' ')
        CreateLMControlXMLFile(myXCLASSFitJobDir, NumberIteration, Optimizer = Optimizer)
        print ("done!")
        AlgorithmXMLFileName = myXCLASSFitJobDir + "algorithm_control.xml"


    ##====================================================================================================================================================
    ## start optimization using routines from the myXCLASSMapFit function
    if (Optimizer in ["scipy"]):
        FullFitFlag = not FastFlag                                                          ## set full flag
        JobDir = task_myXCLASSMapFit.myXCLASSMapFitCore(MolfitsFileName = MolfitsFileName, \
                                                        ObsXMLFileName = ObsXMLFileName, \
                                                        AlgorithmXMLFile = AlgorithmXMLFileName, \
                                                        ChannelIntegrationFlag = ChannelIntegrationFlag, \
                                                        regionFileName = "", \
                                                        clusterdef = "", \
                                                        FastFitFlag = True, \
                                                        FullFitFlag = FullFitFlag, \
                                                        JobDir = myXCLASSFitJobDir, \
                                                        NameOfFunction = NameOfFunction)
        LocalObsXMLFileName = JobDir + "obs.xml"
        ok = 0


    ##====================================================================================================================================================
    ## prepare application of MAGIX and start MAGIX
    elif (Optimizer in ["magix"]):


        ## get absolute path of MAGIX directory
        MAGIXrootDir = task_myXCLASS.GetMAGIXRootDir()

        # Debug:
        # print ("MAGIXrootDir = ", MAGIXrootDir)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## adjust paths in obs. xml file
        LocalObsXMLFileName, IsoRatioFileName, dbFileName = AdjustObsXMLFile(NameOfFunction, LocalObsXMLFileName, CurrentDir, \
                                                                             myXCLASSFitJobDir, printflag)
        if (LocalObsXMLFileName == ""):                                                     ## did an error occurred
            if (DictOutFlag):
                return OutputDict
            else:
                return (OptimizedInputFile, OptimizedSpectra, myXCLASSFitJobDir)

        # Debug:
        # print ("\n\nLocalObsXMLFileName = ", LocalObsXMLFileName)
        # print ("IsoRatioFileName = ", IsoRatioFileName)
        # print ("dbFileName = ", dbFileName)


        ## get all min. and max. frequencies for each range
        FreqMinList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "MinExpRange")
        FreqMaxList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "MaxExpRange")
        GlobalvLSRList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "GlobalvLSR")
        RedshiftList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "Redshift")


        ## get path and name of used database file
        dbList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "dbFilename")
        if (dbList == []):
            dbList = [DefaultDBFileName]
        dbFileName = dbList[0]

        # Debug:
        # print ("FreqMinList = ", FreqMinList)
        # print ("FreqMaxList = ", FreqMaxList)
        # print ("dbFileName = ", dbFileName)


        ## update local-overlap flag, if defined
        if (LocalOverlapFlag is not None):
            task_MAGIX.WriteXMLtagNEW(LocalObsXMLFileName, "LocalOverlap_Flag", [LocalOverlapFlag])
        if (NoSubBeamFlag is not None):
            task_MAGIX.WriteXMLtagNEW(LocalObsXMLFileName, "NoSubBeam_Flag", [NoSubBeamFlag])
        if (EmAbsPATH is not None):
            task_MAGIX.WriteXMLtagNEW(LocalObsXMLFileName, "EmAbsPATH", [EmAbsPATH])


        ##================================================================================================================================================
        ## molfits file:


        ## get alias for keyword "MolfitFileName"
        MolfitsFileName = MolfitsFileName.strip()
        if (MolfitsFileName == "" and MolfitFileName is not None):
            MolfitsFileName = MolfitFileName.strip()


        ## analyze molfit parameter: check path and name of the molfit file and copy molfit file to job directory
        ok, dbFilename, LocalMolfitsFileName, PureNameMolfitsFile = CheckMolfitFile(MolfitsFileName, NameOfFunction, CurrentDir, myXCLASSFitJobDir, \
                                                                                    FreqMinList, FreqMaxList, GlobalvLSRList, RedshiftList, dbList)
        if (ok == 1):                                                                       ## an error occurred
            if (DictOutFlag):
                return OutputDict
            else:
                return (OptimizedInputFile, OptimizedSpectra, myXCLASSFitJobDir)

        # Debug:
        # print ("ok = ", ok)
        # print ("LocalMolfitsFileName = ", LocalMolfitsFileName)
        # print ("PureNameMolfitsFile = ", PureNameMolfitsFile)


        ##================================================================================================================================================
        ## check molecules in iso ratio file (if defined)


        ## get iso flag
        IsoFlag = False
        IsotopologuesList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "iso_flag")
        if (IsotopologuesList == []):
            IsotopologuesList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "Isotopologues")
        if (len(IsotopologuesList) > 0):
            IsoFlag = IsotopologuesList[0].lower()
            IsoFlag = task_myXCLASSMapFit.CheckBool(IsoFlag)


        ## get path and name of new iso ratio file
        NewIsoRatioFileName = ""
        if (IsoRatioFileName != ""):


            ## analyze iso ratio file parameter: check path and name of the iso ratio file and copy new formatted iso ratio file to job directory
            ok, NewIsoRatioFileName = CheckIsoRatioFile(IsoRatioFileName, NameOfFunction, CurrentDir, myXCLASSFitJobDir, FreqMinList, FreqMaxList, \
                                                        GlobalvLSRList, RedshiftList, dbList, LocalMolfitsFileName)
            if (ok == 1):                                                                   ## an error occurred
                if (DictOutFlag):
                    return OutputDict
                else:
                    return (OptimizedInputFile, OptimizedSpectra, myXCLASSFitJobDir)


        ##================================================================================================================================================
        ## construct fit.xml file, if no path for another algorithm xml file is given


        ## adjust algorithm xml file (if defined)
        Optimizer = Optimizer.lower()


        ## check existence of algorithm xml file and copy algorithm xml file to current working directory
        LocalPrintFlag = False
        NameOfFile = "algorithm xml"
        AlgorithmXMLFileName = task_myXCLASS.CheckNCopy(AlgorithmXMLFileName, NameOfFunction, NameOfFile, \
                                                        CurrentDir, myXCLASSFitJobDir, LocalPrintFlag)
        if (AlgorithmXMLFileName == ""):                                                ## did an error occurred
            if (DictOutFlag):
                return OutputDict
            else:
                return (OptimizedInputFile, OptimizedSpectra, myXCLASSFitJobDir)

        # Debug:
        # print ("AlgorithmXMLFileName = ", AlgorithmXMLFileName)


        ## for MAGIX: rename algorithm xml
        if (Optimizer in ["magix"]):
            task_myXCLASS.NormNCopyMove(AlgorithmXMLFileName, myXCLASSFitJobDir + "algorithm_control.xml", copyFlag = False)


        ##================================================================================================================================================
        ## construct io_control.xml file
        print ("Creating io_control file ..", end = ' ')
        CreateIOControlXMLFile(myXCLASSFitJobDir, LocalObsXMLFileName, LocalMolfitsFileName, MAGIXrootDir)
        print ("done!")


        ##================================================================================================================================================
        ## start optimization


        ## set environment
        try:
            OMPStacksize = os.environ["OMP_STACKSIZE"]
        except:
            OMPStacksize = ""
        if (OMPStacksize == ""):
            os.environ["OMP_STACKSIZE"] = '6999M'
        task_MAGIX.SetMAGIXEnvironment(MAGIXrootDir)


        ##================================================================================================================================================
        ## start MAGIX
        print ("\n\nStart MAGIX ..\n")
        MAGIXOption = "model=myxclass"
        ok = task_MAGIX.StartMAGIX(MAGIXrootDir, MAGIXOption, myXCLASSFitJobDir + "io_control.xml")


    ##====================================================================================================================================================
    ## read in output files
    if (ok != 0):
        print ("\n\n\n\t Program MAGIX aborted!\n\n")
    else:


        ## read in input file
        print ("\n\nRead in optimized input file(s) ..")
        if (Optimizer in ["magix"]):
            ConvertLogLinFlag = True
        else:
            ConvertLogLinFlag = False
        printInfoFlag = "true"
        OutputDict = ReadInOutputFiles(myXCLASSFitJobDir, NameOfFunction, ConvertLogLinFlag, RestFreq, vLSR, \
                                      printInfoFlag, ObsXMLFileName = LocalObsXMLFileName)
        ListMolfitFileContent = OutputDict["MolfitFilesContent"]
        ListOfSpectraFilesContent = OutputDict["ListOfSpectraFilesContent"]

        # Debug:
        # print ("ListMolfitFileContent = ", ListMolfitFileContent)


        ##================================================================================================================================================
        ## print some informations to screen
        print ("\n\nPlease note, all files created by the current fit process (new molfit file, log-files etc.) are stored in the")
        print ("myXCLASSFit working directory " + chr(34) + myXCLASSFitJobDir + chr(34) + "!\n\n")


    ##====================================================================================================================================================
    ## define return variables
    if (DictOutFlag):
        return OutputDict
    else:
        return (ListMolfitFileContent, ListOfSpectraFilesContent, myXCLASSFitJobDir)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Simplified CASA interface for MAGIX with myXCLASS program
##
def myXCLASSFit(NumberIteration, AlgorithmXMLFile, MolfitsFileName, experimentalData, TelescopeSize, BMIN, BMAJ, BPA, Inter_Flag, Redshift, \
                t_back_flag, tBack, tslope, BackgroundFileName, N_H, beta_dust, kappa_1300, DustFileName, \
                ContPhenFuncID, ContPhenFuncParam1, ContPhenFuncParam2, ContPhenFuncParam3, ContPhenFuncParam4, ContPhenFuncParam5, \
                iso_flag, IsoTableFileName, NumModelPixelXX, NumModelPixelYY, LocalOverlapFlag, NoSubBeamFlag, RestFreq, vLSR):
    """

Simplified CASA interface for MAGIX with myXCLASS program:
----------------------------------------------------------


This function provides a simplified interface for MAGIX using the myXCLASS program. The
function starts MAGIX using the Levenberg-Marquardt algorithm to fit experimental data
with the myXCLASS program. The user has to specify the max. number of iterations, the
experimental data (or path and name of the experimental xml-file), and the path and
name of the molfit file.

For each run the myXCLASSFit function creates a so-called job directory located in
the run directory ("path-of-myXCLASS-CASA-Interface/run/myXCLASSFit/") where all
files created by the myXCLASSFit function are stored in. The name of this job
directory is made up of four components: The first part of the name consists of the
phrase "job_" whereas the second part describes the date (day, month, year), the third
part the time stamp (hours, minutes, seconds) of the function execution. The last
part describes a so-called job ID which is composed of the so-called PID followed
by a four digit random integer number to create a really unambiguous job number, e.g.
"path-to-myXCLASS-Interface/run/myXCLASSFit/job__25-07-2013__12-02-03__189644932/"

Before the fit procedure starts, the function copies the molfits, the experimental data,
and xml-files to the myXCLASSFit job directory. The path(s) defined in the observational
xml-file are adjusted so that these paths point to the current myXCLASSFit job directory.

Please note, that the original experimental xml- and data files are not modified. Only the
copy of the experimental xml file located in the myXCLASSFit job directory is modified!

If no experimental xml-file is defined, the myXCLASSFit function creates an experimental
xml-file containing the settings for the different parameters in the myXCLASSFit job
directory. Additionally, the function creates an ASCII file located in the myXCLASSFit
job directory containing the experimental data given by the parameter "experimentalData".
If the parameter "experimentalData" defines the path and the name of an experimental
data file, then the data file is copied to the myXCLASSFit job directory as well.
The path and the name of this ASCII file is pasted in the created experimental xml file.
Furthermore, the function creates automatically the algorithm and i/o control xml files,
need by MAGIX, so that the user does not need to edit any xml-file.


input parameters:
-----------------

    - NumberIteration:          max. number of iterations (default: 50).

    - AlgorithmXMLFileName:     only necessary, if the user wants to use another fit
                                algorithm (than Levenberg-Marquardt) for fitting. Therefore,
                                the path and name of a MAGIX xml-file defining settings
                                for an algorithm or algorithm chain has to be given. (A
                                relative path has to be defined relative to the current
                                working directory!)

                                NOTE, if the user specify a xml file, the number of
                                iterations given by the parameter "NumberIteration" is
                                ignored. The number of iteration is then given by the xml
                                file. In order to use the implemented fit algorithm
                                (Levenberg-Marquardt) clear the AlgorithmXMLFileName parameter,
                                i.e. AlgorithmXMLFileName = "", and define the max. number of
                                iterations by using parameter "NumberIteration".

    - MolfitsFileName:          path and name of the extended molfit file, (default: "").

                                In contrast to the format of the molfit file the extended
                                molfit file required by the myXCLASSFit function contains
                                one (three) additional column(s) for each parameter of
                                each component.

                                For a detailed description of the extended molfit file
                                required by the myXCLASSFit function, see the manual.

                                NOTE, a relative path has to be defined relative to the
                                current working directory!

    - experimentalData:         This parameter offers two different possibility to send
                                the experimental data to the myXCLASSFit function:

                                - the parameter experimentalData defines the path and
                                  name of and experimental xml-file suitable for MAGIX.

                                - the parameter experimentalData defines the path and
                                  name of and ASCII file called experimental data file,
                                  where the first column describe the frequency (in MHz)
                                  and the second column the beam temperature (intensity).

                                NOTE, if the parameter experimentalData defines a relative
                                path, the path has to be defined relative to the current
                                working directory!

    The following parameters are needed, if the parameter experimentalData does NOT
    describe the path and name of a MAGIX xml-file:

    - vLSR:                     local standard of rest velocity (in km/s), (default 0).

    - TelescopeSize:            for single dish observations (Inter_Flag = False): TelescopeSize
                                describes the size of telescope (in m), (default: 1); for
                                interferometric observations (Inter_Flag = True): TelescopeSize
                                describes the interferometric beam FWHM size (in arcsec),
                                (default: 1).

    - Inter_Flag:               defines, if single dish ("False") or interferometric
                                observations ("True") are described, (default: "False").

    - BMIN:                     Beam minor axis length (default: None)

    - BMAJ:                     Beam major axis length (default: None)

    - BPA:                      Beam position angle (default: None)

    - Redshift:                 red shift (dimensionless), (default: None)

    - t_back_flag:              ("True"/"False"), defines, if the user defined background
                                temperature Tbg and temperature slope Tslope given
                                by the input parameters tBack and tSlope describe the
                                continuum contribution completely (t_back_flag = "True")
                                or not (t_back_flag = "False") (default: "True").

    - tBack:                    background temperature (in K), (default: 0).

    - tSlope:                   temperature slope (dimensionless), (default: 0).

    - BackgroundFileName:       path and name of background file (default: "").

    - N_H:                      Hydrogen column density (in cm^{-2}), (default: None).

    - beta_dust:                beta for dust (dimensionless), (default: None).

    - kappa_1300:               kappa (cm^2 g^{-1}), (default: None).

    - DustFileName:             path and name of file describing the optical depth
                                of dust (default: "").

    - ContPhenFuncID:           describes which phenomenological function is used
                                to describe the continuum (default: None).

    - ContPhenFuncParam1:       first parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam2:       second parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam3:       third parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam4:       fourth parameter for phenomenological function
                                describing the continuum (default: None).

    - ContPhenFuncParam5:       fifth parameter for phenomenological function
                                describing the continuum (default: None).

    - iso_flag:                 use isotopologues ("True"/"False"). If iso_flag is set to
                                "False" the ratios are all set to 1 (default: "False").

    - IsoTableFileName:         path and name of an ASCII file including the iso ratios
                                between certain molecules. The so-called "iso ratio file"
                                defines the iso ratios between molecules. The ASCII file
                                consists of three columns, where the first two columns
                                indicates the molecules, respectively. The third column
                                defines the ratio for both molecules. The columns are
                                separated by blanks or tabs. So, the names of the
                                molecules must not contain blanks.

                                The myXCLASSFit function offers the possibility to
                                optimize the ratios between isotopologues as well. For
                                that purpose, the user has to add two additional columns
                                on the right indicating the lower and the upper limit
                                of a certain ratio, respectively. For more information
                                please look at the description of the iso ratio file given
                                for the myXCLASS function.

                                If the lower and upper limit are equal or if the lower
                                limit is higher than the upper limit, the ratio is kept
                                constant and is not optimized by the myXCLASSFit function.

                                NOTE, if the parameter "IsoTableFileName" defines a relative
                                path, the path has to be defined relative to the current
                                working directory!

    - NumModelPixelXX:          used for sub-beam modeling, describes the number
                                of pixels used in x-direction (default: None).

    - NumModelPixelYY:          used for sub-beam modeling, describes the number
                                of pixels used in y-direction (default: None).

    - LocalOverlapFlag:         flag indicates if local-overlap description is
                                used or not (default: False).

    - NoSubBeamFlag:            do not use sub-beam description (default: True).

    The following parameter is needed, to add a velocity axis to the output
    array OptimizedSpectra, see below:

    - RestFreq:                 rest frequency in MHz (default: 0). (If this parameter is
                                set to zero, the intensity is plotted against frequency (in MHz)
                                otherwise against velocity (in km/s).


Output parameters:
------------------

    - OptimizedInputFile :      the contents of the molfit file containing the
                                parameters for the best fit.

    - OptimizedSpectra:         the model function values for each data point, which
                                correspond to the best fit.

    - JobDir:                   absolute path of the job directory created for the current run.



Example using a molfit file in the old format and an ASCII file containing the experimental data:
-------------------------------------------------------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSFit/CH3OH__old.molfit"
experimentalData = "demo/myXCLASSFit/band1b.dat"
vLSR = 0.0
TelescopeSize = 3.5
Inter_Flag = False
t_back_flag = True
tBack = 1.1
tslope = 0.0
nH_flag = True
N_H = 3.0E+24
beta_dust = 2.0
kappa_1300 = 0.02
iso_flag = False
IsoTableFileName = ""
RestFreq = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()


Example NGC6334 with MAGIX xml files (old format):
--------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSFit/CH3OH__old.molfit"
experimentalData = "demo/myXCLASSFit/observation.xml"
RestFreq = 0.0
vLSR = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()


Example NGC6334 with MAGIX xml files (new format):
--------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSFit/CH3OH__new.molfit"
experimentalData = "demo/myXCLASSFit/observation.xml"
RestFreq = 0.0
vLSR = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()


Example NGC6334 with MAGIX xml files (other algorithm):
-------------------------------------------------------

MolfitsFileName = "demo/myXCLASSFit/CH3OH__new.molfit"
experimentalData = "demo/myXCLASSFit/observation.xml"
AlgorithmXMLFile = "demo/myXCLASSFit/algorithm-settings.xml"
RestFreq = 0.0
vLSR = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()

    """

    # Debug:
    # print ("NumberIteration = ", NumberIteration)
    # print ("AlgorithmXMLFile = ", AlgorithmXMLFile)
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("experimentalData = ", experimentalData)
    # print ("TelescopeSize = ", TelescopeSize)
    # print ("BMIN = ", BMIN)
    # print ("BMAJ = ", BMAJ)
    # print ("BPA = ", BPA)
    # print ("Inter_Flag = ", Inter_Flag)
    # print ("Redshift = ", Redshift)
    # print ("t_back_flag = ", t_back_flag)
    # print ("tBack = ", tBack)
    # print ("tslope = ", tslope)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("iso_flag = ", iso_flag)
    # print ("IsoTableFileName = ", IsoTableFileName)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)
    # print ("RestFreq = ", RestFreq)
    # print ("vLSR = ", vLSR)


    ## interpret input variable "experimentalData"
    ObsXMLFileName = ""
    ObsDataFileName = None
    if (type(experimentalData).__name__ == 'str'):
        if (experimentalData.endswith(".xml")):                                             ## parameter defines obs. xml file
            ObsXMLFileName = experimentalData.strip()                                       ## remove leading and tailing blanks
        else:
            ObsDataFileName = experimentalData.strip()
    else:
        ObsDataFileName = experimentalData


    ## call myXCLASSFit core function
    nH_flag = False
    EmAbsPATH = None
    OptimizedInputFile, OptimizedSpectra, myXCLASSFitJobDir = myXCLASSFitCore(NumberIteration = NumberIteration, \
                                                                              AlgorithmXMLFileName = AlgorithmXMLFile, \
                                                                              MolfitsFileName = MolfitsFileName, \
                                                                              ObsXMLFileName = ObsXMLFileName, \
                                                                              ObsDataFileName = ObsDataFileName, \
                                                                              iso_flag = iso_flag, \
                                                                              IsoTableFileName = IsoTableFileName, \
                                                                              TelescopeSize = TelescopeSize, \
                                                                              BMIN = BMIN, BMAJ = BMAJ, BPA = BPA, \
                                                                              Redshift = Redshift, Inter_Flag = Inter_Flag, \
                                                                              t_back_flag = t_back_flag, \
                                                                              tBack = tBack, tSlope = tslope, \
                                                                              BackgroundFileName = BackgroundFileName, \
                                                                              nH_flag = nH_flag, N_H = N_H, beta_dust = beta_dust, \
                                                                              kappa_1300 = kappa_1300, DustFileName = DustFileName, \
                                                                              ContPhenFuncID = ContPhenFuncID, \
                                                                              ContPhenFuncParam1 = ContPhenFuncParam1, \
                                                                              ContPhenFuncParam2 = ContPhenFuncParam2, \
                                                                              ContPhenFuncParam3 = ContPhenFuncParam3, \
                                                                              ContPhenFuncParam4 = ContPhenFuncParam4, \
                                                                              ContPhenFuncParam5 = ContPhenFuncParam5, \
                                                                              NumModelPixelXX = NumModelPixelXX, \
                                                                              NumModelPixelYY = NumModelPixelYY, LocalOverlapFlag = LocalOverlapFlag, \
                                                                              NoSubBeamFlag = NoSubBeamFlag, EmAbsPATH = EmAbsPATH, \
                                                                              RestFreq = RestFreq, vLSR = vLSR)


    ##====================================================================================================================================================
    ## define return variables
    return (OptimizedInputFile, OptimizedSpectra, myXCLASSFitJobDir)
##--------------------------------------------------------------------------------------------------------------------------------------------------------

