#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module function updates the XCLASS sqlite3 database
##  Copyright (C) 2012 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function DefineRRLEntries:                    define RRL entries
##      - function AddRRLEntries2DB:                    add RRL entries to given database file
##      - function CheckDB:                             check, if database file contains already RRL entries
##      - function dlProgress:                          function updates the XCLASS sqlite3 database
##      - function UpdateDatabase:                      function updates the XCLASS sqlite3 database
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2013-07-25      initial version
##  C. Endres       2013-09-06      add Christians download package
##  T. Moeller      2018-07-04      add RRL entries
##  T. Moeller      2020-01-03      porting to python 3.x
##  T. Moeller      2022-04-27      add versioning
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import urllib.request                                                                       ## import package for handling url's
import urllib.parse                                                                         ## import package for handling url's
import urllib.error                                                                         ## import package for handling url's
import time                                                                                 ## import package date and time
import sqlite3                                                                              ## import sqlite3 package
from datetime import datetime                                                               ## import datetime package
from . import task_myXCLASS                                                                 ## import package myXCLASS
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define RRL entries
##
def DefineRRLEntries(RRLMaxN = 900, RRLMaxDeltaN = 6):
    """

input parameters:
-----------------

    - RRLMaxN:                      (optional) max. main quantum number n for which RRLs are computed, (default: 900)

    - RRLMaxDeltaN:                 (optional) max. \Delta n for which RRLs are computed, (default: 6)


output parameters:
------------------

    - Contents:                     array containing information

    - DeltaNList:                   names for different \Delta n values
    """

    # Debug:
    # print ("RRLMaxN = ", RRLMaxN)
    # print ("RRLMaxDeltaN = ", RRLMaxDeltaN)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    Contents = []
    DeltaNList = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define constants
    me = 9.10938291e-28                                                                     ## electron mass [g]
    h = 6.62606957e-27                                                                      ## Planck constant [erg s]
    hbar = h / (2.0 * numpy.pi)                                                             ## reduced Planck constant
    e = 4.8e-10                                                                             ## electron charge [esu]
    c = 2.997924580e10                                                                      ## speed of light in vacuum [cm / s]
    ergTOeV = 6.2415e11                                                                     ## convert erg to eV
    eVTOcm1 = 8065.73                                                                       ## convert eV to cm-1


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create content without ASCII file


    ## define list of atoms
    ListOfElements = ["H", "He", "C", "N", "O", "S"]


    ## define corresponding masses (currently not used)
    # MassList = [1.007825035, 4.00260324, 12.000000, 14.003074, 15.994915, 31.97207070]


    ## define corresponding Rydberg constants
    RydbergList = [109677.576, 109722.27, 109732.30, 109733.01, 109733.55, 109735.43]


    ## define names for different \Delta n values
    ## \Delta n =       1,      2,       3,       4,         5,      6,     7,       8,      9,      10
    DeltaNList = ["alpha", "beta", "gamma", "delta", "epsilon", "zeta", "eta", "theta", "iota", "kappa"]


    ## define parameters (linear values) for Menzel's approx. for each \Delta n value
    ## \Delta n =       "alpha",       "beta",      "gamma",      "delta",    "epsilon",       "zeta",        "eta",      "theta",       "iota",      "kappa"
    MDeltaNList = [1.907749e-01, 2.633210e-02, 8.105620e-03, 3.491680e-03, 1.811850e-03, 1.058470e-03, 6.712780e-04, 4.521900e-04, 3.190010e-04, 2.334070e-04]


    ## define parameters (as log10 values) for Menzel's approx. for each \Delta n value (currently not used)
    ## \Delta n =          "alpha",     "beta",    "gamma",    "delta",  "epsilon",     "zeta",      "eta",    "theta",     "iota",    "kappa"
    # Mlog10DeltaNList = [-0.7194788, -1.5795145, -2.0912138, -2.4569656, -2.7418778, -2.9753214, -3.1730976, -3.3446790, -3.4962080, -3.6318861]


    ## define max. delta n for which RRLs are computed
    LocalMaxDeltaN = min(RRLMaxDeltaN, len(DeltaNList))


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## compute RRLs
    for LocalElementID, LocalElement in enumerate(ListOfElements):                          ## loop over atoms
        for LocalDeltaN in range(LocalMaxDeltaN):                                           ## loop over Delta n values
            for LocalN in range(1, RRLMaxN + 1):                                            ## loop over main quantum numbers n


                ## compute E_low
                ELow = -(me * e**4) / (2.0 * hbar**2 * LocalN**2) * ergTOeV * eVTOcm1       ## lower energy [cm-1]


                ## compute transition frequency (in MHz)
                RM = RydbergList[LocalElementID]                                            ## get Rydberg constant
                Freq = RM * c * ((1.0 / LocalN**2) - (1.0 / (LocalN + (LocalDeltaN + 1))**2))   ## transition frequency [Hz]
                Freq = Freq * 1.e-6                                                         ## convert transition frequency to MHz


                ## compute oscillator strength using Menzel's approximation
                n2f = LocalN * MDeltaNList[LocalDeltaN] * (1.0 + 1.5 * (LocalDeltaN + 1) / LocalN) * LocalN**2


                ## compute degree of degeneracy
                gup = 2.0 * LocalN**2


                ## store computed values
                Contents.append([LocalElement, LocalN, DeltaNList[LocalDeltaN], float(Freq), float(n2f), float(ELow), int(float(gup))])

    # Debug:
    # print ("Contents = ", Contents)


    ## define return parameter
    return (Contents, DeltaNList)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## add RRL entries to given database file
##
def AddRRLEntries2DB(dbFileName, RRLMaxN, RRLMaxDeltaN):
    """

input parameters:
-----------------

    - dbFileName:                   path and name of database file

    - RRLMaxN:                      max. main quantum number n for which RRLs are computed

    - RRLMaxDeltaN:                 max. \Delta n for which RRLs are computed


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("dbFileName = ", dbFileName)
    # print ("RRLMaxN = ", RRLMaxN)
    # print ("RRLMaxDeltaN = ", RRLMaxDeltaN)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define entries for RRL
    Contents, DeltaNList = DefineRRLEntries(RRLMaxN = RRLMaxN, RRLMaxDeltaN = RRLMaxDeltaN)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## connect to database
    try:
        conn = sqlite3.connect(dbFileName)
        # conn = sqlite3.connect(dbFileName, timeout = 10)

        # Debug:
        # print (" ")
        # print ("Connection to sqlite3 database {:s} established.".format(dbFileName))
    except sqlite3.Error as e:
        print ("\nCan not connect to sqlite3 database {:s}.".format(dbFileName))
        print ("Error: {:d}: {:s}".format(e.args[0], e.args[1]))
        sys.exit(1)


    ## from connection, get cursor object. The cursor is used to traverse the records from the result set.
    cursor = conn.cursor()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define names of columns


    ## table partitionfunctions
    NameOfPFTable = "partitionfunctions"
    ColumnNameForNamePartFunc = 'PF_Name'
    ColumnNamesPartFunc = ['PF_1_072', 'PF_1_148', 'PF_1_230', 'PF_1_318', 'PF_1_413', 'PF_1_514', 'PF_1_622', 'PF_1_738', 'PF_1_862', 'PF_1_995', \
                           'PF_2_138', 'PF_2_291', 'PF_2_455', 'PF_2_630', 'PF_2_725', 'PF_2_818', 'PF_3_020', 'PF_3_236', 'PF_3_467', 'PF_3_715', \
                           'PF_3_981', 'PF_4_266', 'PF_4_571', 'PF_4_898', 'PF_5_000', 'PF_5_248', 'PF_5_623', 'PF_6_026', 'PF_6_457', 'PF_6_918', \
                           'PF_7_413', 'PF_7_943', 'PF_8_511', 'PF_9_120', 'PF_9_375', 'PF_9_772', 'PF_10_471', 'PF_11_220', 'PF_12_023', 'PF_12_882', \
                           'PF_13_804', 'PF_14_791', 'PF_15_849', 'PF_16_982', 'PF_18_197', 'PF_18_750', 'PF_19_498', 'PF_20_893', 'PF_22_387', \
                           'PF_23_988', 'PF_25_704', 'PF_27_542', 'PF_29_512', 'PF_31_623', 'PF_33_884', 'PF_36_308', 'PF_37_500', 'PF_38_905', \
                           'PF_41_687', 'PF_44_668', 'PF_47_863', 'PF_51_286', 'PF_54_954', 'PF_58_884', 'PF_63_096', 'PF_67_608', 'PF_72_444', \
                           'PF_75_000', 'PF_77_625', 'PF_83_176', 'PF_89_125', 'PF_95_499', 'PF_102_329', 'PF_109_648', 'PF_117_490', 'PF_125_893', \
                           'PF_134_896', 'PF_144_544', 'PF_150_000', 'PF_154_882', 'PF_165_959', 'PF_177_828', 'PF_190_546', 'PF_204_174', 'PF_218_776', \
                           'PF_225_000', 'PF_234_423', 'PF_251_189', 'PF_269_153', 'PF_288_403', 'PF_300_000', 'PF_309_030', 'PF_331_131', 'PF_354_813', \
                           'PF_380_189', 'PF_407_380', 'PF_436_516', 'PF_467_735', 'PF_500_000', 'PF_501_187', 'PF_537_032', 'PF_575_440', 'PF_616_595', \
                           'PF_660_693', 'PF_707_946', 'PF_758_578', 'PF_812_831', 'PF_870_964', 'PF_933_254', 'PF_1000_000']


    ## table transitions
    NameOfRadTransTable = "transitions"
    ColumnNameForNameTransitions = "T_Name"
    ColumnNameForFreqTransitions = "T_Frequency"
    ColumnNameForIntTransitions = "T_Intensity"
    ColumnNameForEinsteinATransitions = "T_EinsteinA"
    ColumnNameForFreqErrTransitions = "T_Uncertainty"
    ColumnNameForELowTransitions = "T_EnergyLower"
    ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## add entries
    NumEntries = len(Contents)
    ElementsWithPF = []
    for ContentLineID, ContentLine in enumerate(Contents):
        Element = "RRL-" + ContentLine[0]
        n = ContentLine[1]
        DeltaN = ContentLine[2]
        DeltaNID = DeltaNList.index(DeltaN) + 1
        Freq = ContentLine[3]
        EinsteinA = ContentLine[4]
        ELow = ContentLine[5]
        gup = ContentLine[6]
        Date = datetime.now()

        # Debug:
        # print ("\n\nElement = ", Element)
        # print ("n = ", n)
        # print ("DeltaN = ", DeltaN)
        # print ("Freq = ", Freq)
        # print ("EinsteinA = ", EinsteinA)
        # print ("ELow = ", ELow)
        # print ("gup = ", gup)
        # print ("Date = ", Date)


        ## print what you do
        print ("\rAdd RRL entry (" + str(ContentLineID + 1) + "/" + str(NumEntries) + ") to database file ..                       ", end = " ", flush = True)


        ## define query string
        if (not Element in ElementsWithPF):


            ## delete old entry for current molecule
            query_string = "DELETE FROM " + NameOfPFTable + " WHERE PF_Name=" + chr(34) + Element + chr(34)
            cursor.execute(query_string)

            # Debug:
            # print ("\n\nquery_string = ", query_string)


            ## define query string
            query_string = "INSERT INTO " + NameOfPFTable + " VALUES("
            query_string += chr(34) + Element + chr(34) + ", "                              ## add name
            query_string += chr(34) + " "  + chr(34) + ", "                                 ## PF_VamdcSpeciesID
            query_string += chr(34) + " "  + chr(34) + ", "                                 ## PF_SpeciesID
            query_string += chr(34) + " "  + chr(34) + ", "                                 ## PF_NuclearSpinIsomer
            query_string += chr(34) + " "  + chr(34) + ", "                                 ## PF_HFS
            for i in range(len(ColumnNamesPartFunc)):
                query_string += "1.0, "                                                     ## partition function values for each temperature
            query_string += chr(34) + " "  + chr(34) + ", "                                 ## PF_ResourceID
            query_string += chr(34) + " "  + chr(34) + ", "                                 ## PF_URL
            query_string += chr(34) + " "  + chr(34) + ", "                                 ## PF_Comment
            query_string += chr(34) + str(Date) + chr(34) + ", "                            ## date
            query_string += chr(34) + " "  + chr(34) + ", "                                 ## Splat_ID
            query_string += chr(34) + Element + chr(34) + ");"                              ## add Splatalogue name
            ElementsWithPF.append(Element)

            # Debug:
            # print ("query_string = ", query_string)


            ## add entry to table partitionfunctions
            cursor.execute(query_string)


            ## delete old entry for current molecule
            query_string = "DELETE FROM " + NameOfRadTransTable + " WHERE T_Name=" + chr(34) + Element + chr(34)
            cursor.execute(query_string)

            # Debug:
            # print ("\n\nquery_string = ", query_string)


        ## define query string
        query_string = "INSERT INTO " + NameOfRadTransTable + " VALUES("
        query_string += chr(34) + Element + chr(34) + ", "                                  ## add name
        query_string += str(Freq) + ", "                                                    ## T_Frequency
        query_string += "0.0, "                                                             ## T_Intensity
        query_string += "{:.4e}, ".format(EinsteinA)                                        ## T_EinsteinA
        query_string += "0.0, "                                                             ## T_Uncertainty
        query_string += "{:.4e}, ".format(ELow)                                             ## T_EnergyLower
        query_string += str(int(gup)) + ", "                                                ## T_UpperStateDegeneracy
        query_string += chr(34) + " "  + chr(34) + ", "                                     ## T_NuclearSpinIsomer
        query_string += chr(34) + " "  + chr(34) + ", "                                     ## T_HFS
        query_string += chr(34) + " "  + chr(34) + ", "                                     ## T_Case
        query_string += chr(34) + "n_u = " + str(n + DeltaNID)  + chr(34) + ", "            ## T_UpperStateQuantumNumbers
        query_string += chr(34) + "n_l = " + str(n)  + chr(34) + ");"                       ## T_LowerStateQuantumNumbers

        # Debug:
        # print ("query_string = ", query_string, "   ", DeltaN)


        ## add entry to table partitionfunctions
        cursor.execute(query_string)
        conn.commit()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## send query string to database
    conn.close()


    ## we're done
    print ("\rAdding RRL entries to database file .. done!                                                                                ", flush = True)


    ## define return parameters
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check, if database file contains already RRL entries
##
def CheckDB(dbFileName):
    """

input parameters:
-----------------

    - dbFileName:                   path and name of database file


output parameters:
------------------

    - IncludeFlag:                  indicates if RRL-H entries are already included
    """

    # Debug:
    # print ("dbFileName = ", dbFileName)


    ## initialize return parameter
    IncludeFlag = False


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## connect to database
    try:
        conn = sqlite3.connect(dbFileName)

        # Debug:
        # print (" ")
        # print ("Connection to sqlite3 database {:s} established.".format(dbFileName)
    except sqlite3.Error as e:
        print ("\nCan not connect to sqlite3 database {:s}.".format(dbFileName))
        print ("Error: {:d}: {:s}".format(e.args[0], e.args[1]))
        sys.exit(1)


    ## from connection, get cursor object. The cursor is used to traverse the records from the result set.
    cursor = conn.cursor()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define query string
    QueryString = "select * from partitionfunctions where PF_Name=" + chr(34) + "RRL-H" + chr(34)

    # Debug:
    # print ("QueryString = ", QueryString)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get content
    cursor.execute(QueryString)
    conn.commit()
    rows = cursor.fetchall()
    if (rows != []):
        IncludeFlag = True


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## send query string to database
    conn.close()


    ## define return parameter
    return IncludeFlag
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## This function updates the XCLASS sqlite3 database
##
def dlProgress(count, blockSize, totalSize):
    """

input parameters:
-----------------

    - count:                counter

    - blockSize:            size of block

    - totalSize:            total size



output parameters:
------------------

    - None
    """

    # Debug:
    # print ("count = ", count)
    # print ("blockSize = ", blockSize)
    # print ("totalSize = ", totalSize)


    percent = int(count * blockSize * 100 / totalSize)
    # sys.stdout.write("\rDownloading sqlite3 database file for XCLASS from CDMS Server .. {:2d} %".format(percent), flush = True)
    print ("\rDownloading sqlite3 database file for XCLASS from CDMS Server .. {:2d} %".format(percent), end = "", flush = True)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## This function updates the XCLASS sqlite3 database
##
def UpdateDatabase(DBUpdateNew, ForceRRLAdd = False, RRLMaxN = 900, RRLMaxDeltaN = 6):
    """

This function updates the XCLASS database file located in the subdirectory "Database".

The XCLASS interface uses a sqlite3 database which contains two tables:

The values for the partition function for many molecules are saved in table "partitionfunctions". For each molecule the following data
are saved:

column 1:       name of molecule
column 6 - 115: the partition function for the temperatures (all in K):
                1.072, 1.148, 1.23, 1.318, 1.413, 1.514, 1.622, 1.738, 1.862, 1.995, 2.138, 2.291, 2.455, 2.63, 2.725, 2.818, 3.02, 3.236,
                3.467, 3.715, 3.981, 4.266, 4.571, 4.898, 5, 5.248, 5.623, 6.026, 6.457, 6.918, 7.413, 7.943, 8.511, 9.12, 9.375, 9.772,
                10.471, 11.22, 12.023, 12.882, 13.804, 14.791, 15.849, 16.982, 18.197, 18.75, 19.498, 20.893, 22.387, 23.988, 25.704, 27.542,
                29.512, 31.623, 33.884, 36.308, 37.5, 38.905, 41.687, 44.668, 47.863, 51.286, 54.954, 58.884, 63.096, 67.608, 72.444, 75, 77.625,
                83.176, 89.125, 95.499, 102.329, 109.648, 117.49, 125.893, 134.896, 144.544, 150, 154.882, 165.959, 177.828, 190.546, 204.174,
                218.776, 225, 234.423, 251.189, 269.153, 288.403, 300, 309.03, 331.131, 354.813, 380.189, 407.38, 436.516, 467.735, 500, 501.187,
                537.032, 575.44, 616.595, 660.693, 707.946, 758.578, 812.831, 870.964, 933.254, 1000

Additionally, the data for many radiative transitions are stored in table "transitions". Here, the
following data are saved for each radiative transition:

column 1*:   name of molecule
column 2*:   Frequency (in MHz)
column 3:    Intensity (in nm2 MHz)
column 4*:   Einstein A coefficient
column 5*:   Uncertainty of frequency (in MHz)
column 6*:   Energy_Lower (in cm^(-1))
column 7*:   upper state degeneracy
column 8:    nuclear spin isomer
column 9:    HFS
column 10:   not used at the moment
column 11:   upper state quantum numbers
column 12:   lower state quantum numbers

Please note, if you add private entries to the database, please make sure, that the partition function is given for all temperatures described above
and that all entries/columns in table "transitions" which are marked with a "*" sign are defined as well!

During the update process, the old sqlite database file is renamed whereat the file name "cdms_sqlite.db" is extended by the current date and time. For
example, the XCLASS interface renames this file to "cdms_sqlite__16-09-2013__10-40-15.db" and then downloads/updates the latest version of the database from
the CDMS server. So, the database which is used for a previous simulated spectra is not removed.



input parameters:
-----------------

    - DBUpdateNew:          flag indicating, if a complete new database file is downloaded from the CDMS server ("new") or the existing database
                            file is updated ("update").

    - ForceRRLAdd:          (optional) flag for forcing the addition and calculation of RRLs (default: False)

    - RRLMaxN:              (optional) max. main quantum number n for which RRLs are computed, (default: 900)

    - RRLMaxDeltaN:         (optional) max. \Delta n for which RRLs are computed, (default: 6)



output parameters:
------------------

    - None


Example 1:
----------

DBUpdateNew = "new"
UpdateDatabase()


Example 2:
----------

DBUpdateNew = "update"
UpdateDatabase()

    """

    # Debug:
    # print ("DBUpdateNew = ", DBUpdateNew)
    # print ("ForceRRLAdd = ", ForceRRLAdd)
    # print ("RRLMaxN = ", RRLMaxN)
    # print ("RRLMaxDeltaN = ", RRLMaxDeltaN)


    ## get path of default database file
    dbFileName = task_myXCLASS.GetDefaultDBFile()


    ## get path of database directory
    i = dbFileName.rfind("/")
    if (i > (-1)):
        dbPath = dbFileName[:i]
        dbFile = dbFileName[i + 1:]


    ##====================================================================================================================================================
    ## download the sqlite file from the CDMS server and store the file within the directory "Database"
    DBUpdateNew = DBUpdateNew.strip()
    DBUpdateNew = DBUpdateNew.lower()
    if (DBUpdateNew in ["new", "beta", "official"]):


        ## download the latest version of the sqlite3 database file from the cdms server
        if (DBUpdateNew in ["new", "beta"]):
            urlDB = "https://cdms.astro.uni-koeln.de/cdms/portal/cdms_lite.db.gz"           ## define url
        elif (DBUpdateNew in ["official"]):
            # urlDB = "https://cdms.astro.uni-koeln.de/cdms/portal/cdms_lite__official.db.gz" ## define url
            urlDB = "https://cdms.astro.uni-koeln.de/static/cdms/cdms_lite__official.db.gz" ## define url


        ## rename the current sqlite3 file and
        print ("\nRename the current sqlite3 file ..", end = " ", flush = True)
        lt = time.localtime()
        cmd_string = "cd " + dbPath                                                         ## change to database directory
        cmd_string += "; mv " + dbFileName + " " + dbPath + "/cdms_sqlite__" + time.strftime("%d-%m-%Y", lt) + "__" + time.strftime("%H-%M-%S", lt) + ".old.db"
        os.system(cmd_string)                                                               ## execute command string
        print ("done!")


        ## download latest version of database
        global ZipFileName                                                                  ## global variable to be used in dlProgress
        ZipFileName = dbPath + "/cdms_lite.db.gz"
        urllib.request.urlretrieve(urlDB, ZipFileName, reporthook = dlProgress)
        sys.stdout.write("\rDownloading sqlite3 database file for XCLASS from CDMS Server .. done!                        ")


        ## unzip file
        print ("\nUnzip the downloaded database file ..", end = " ", flush = True)
        cmd_string = "cd " + dbPath                                                         ## change to database directory
        cmd_string += "; gunzip cdms_lite.db.gz"                                            ## unzip database file
        cmd_string += "; mv cdms_lite.db " + dbFileName                                     ## make a copy of the old database file
        os.system(cmd_string)                                                               ## execute command string
        print ("done!")



    ##====================================================================================================================================================
    ## add path of update package to sys.path variable
    else:


        ## print what you do
        print ("\nUpdating the sqlite3 database file {:s}.\n".format(dbFile))
        lt = time.localtime()
        datestring = time.strftime("Start updating at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)
        print (str(datestring))
        print (" ", flush = True)


        ## make a copy of the old database file
        print ("\nMake a copy of the old sqlite3 file ..", end = " ", flush = True)
        lt = time.localtime()
        cmd_string = "cd " + dbPath + "; "                                                  ## change to database directory
        cmd_string += "cp " + dbFileName + " " + dbPath + "/cdms_sqlite__" + time.strftime("%d-%m-%Y", lt) + "__" + time.strftime("%H-%M-%S", lt) + ".old.db"
        os.system(cmd_string)                                                               ## execute command string
        print ("done!")


        ## define path of program package
        i = dbPath.rfind("/")
        if (i > (-1)):
            UpadatePackagePath = dbPath[:i]
        else:
            UpadatePackagePath = dbPath
        UpadatePackagePath = UpadatePackagePath + "/xclass/external/vamdclib/vamdclib/"
        task_myXCLASS.ExtendSysPath([UpadatePackagePath])


        ## try to import update package
        try:
            from external.vamdclib.vamdclib import database
        except ImportError:
            print ("\n\nError in subroutine UpdateDatabase:")
            print ("\n\n\tCan not import update package. Please use another update option!\n\n")
            return


        ## update database
        db = database.Database(dbFileName)
        db.update_database()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check, if RRL-H entries are already included
    IncludeFlag = CheckDB(dbFileName)
    if ((not IncludeFlag) or ForceRRLAdd):                                                  ## if not, add RRL entries
        AddRRLEntries2DB(dbFileName, RRLMaxN, RRLMaxDeltaN)


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------

