#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module reads in entries from the sqlite3 database
##  Copyright (C) 2012 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function LatexMoleculeName:                   translate name of molecule in database to latex name
##      - function ListDatabaseCore:                    function reads in entries from the sqlite3 database
##      - function ListDatabase:                        call for standard interface
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2013-07-25      initial version
##  T. Moeller      2020-01-03      porting to python 3.x
##  T. Moeller      2020-10-29      add translation table for latex names
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import copy                                                                                 ## import copy package
import sqlite3                                                                              ## import sqlite3 package
from . import task_myXCLASS                                                                 ## import package myXCLASS
from . import task_myXCLASSFit                                                              ## import package myXCLASSFit
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## translate name of molecule in database to latex name
##
def LatexMoleculeName(DBMoleculeName):
    """

input parameters:
-----------------


    - DBMoleculeName:       name of molecule in database


output parameters:
------------------

    - LatexMoleculeName:    latex name of molecule
    """

    # Debug:
    # print ("DBMoleculeName = ", DBMoleculeName)


    ## define translation table between name of molecule in database and latex name
    TranslationNameList = [['CONT-Dust', "CONT-Dust"],
                           ["(CH3)2C-13-HCN;v=0;", "(CH$_3$)$_2^{13}$CHCN"],
                           ["(O-18)2;v=0;", "($^{18}$O)$_2$"],
                           ["(Z)-HC2CHCHCN;v=0;", "(Z)-HC$_2$CHCHCN"],
                           ["(c-C3H2)CH2;v=0;", "(c-C$_3$H$_2$)CH$_2$"],
                           ["(c-C3H2)CH2;v=0;ortho", "ortho-(c-C$_3$H$_2$)CH$_2$"],
                           ["(c-C3H2)CH2;v=0;para", "para-(c-C$_3$H$_2$)CH$_2$"],
                           ["12C", "$^{12}$C"],
                           ["12C+", "$^{12}$C$^+$"],
                           ["13C", "$^{13}$C"],
                           ["13C#1", "$^{13}$C"],
                           ["13C+", "$^{13}$C$^+$"],
                           ["14N", "$^{14}$N"],
                           ["14N", "$^{14}$N"],
                           ["14N+", "$^{14}$N$^+$"],
                           ["16O", "$^{16}$O"],
                           ["28Si", "$^{28}$Si"],
                           ["28Si+", "$^{28}$Si$^+$"],
                           ["2D", "$^{2}$D"],
                           ["32S", "$^{32}$S"],
                           ["56Fe", "$^{56}$Fe"],
                           ["56Fe+", "$^{56}$Fe$^+$"],
                           ["AA-n-C4H9CN;v=0;", "AA-n-C$_4$H$_9$CN"],
                           ["AG-n-C4H9CN;v=0;", "AG-n-C$_4$H$_9$CN"],
                           ["Al-26-Cl-35;v=0;", "$^{26}$Al$^{35}$Cl"],
                           ["Al-26-Cl-35;v=1;", "$^{26}$Al$^{35}$Clv=1"],
                           ["Al-26-Cl-35;v=2;", "$^{26}$Al$^{35}$Clv=2"],
                           ["Al-26-Cl-37;v=0;", "$^{26}$Al$^{37}$Cl"],
                           ["Al-26-Cl-37;v=1;", "$^{26}$Al$^{37}$Clv=1"],
                           ["Al-26-Cl-37;v=2;", "$^{26}$Al$^{37}$Clv=2"],
                           ["Al-26-F;v=0;", "$^{26}$AlF"],
                           ["Al-26-F;v=1;", "$^{26}$AlF, v=1"],
                           ["Al-26-F;v=2;", "$^{26}$AlF, v=2"],
                           ["Al-26-O;v=0;", "$^{26}$AlO"],
                           ["Al-26-O;v=0;hyp1", "$^{26}$AlO"],
                           ["Al-26-O;v=1;", "$^{26}$AlO, v=1"],
                           ["Al-26-O;v=1;hyp1", "$^{26}$AlO, v=1"],
                           ["Al-26-O;v=2;", "$^{26}$AlO, v=2"],
                           ["Al-26-O;v=2;hyp1", "$^{26}$AlO, v=2"],
                           ["AlC2;v=0;hyp1", "AlC$_2$"],
                           ["AlC3N;v=0;", "AlC$_3$N"],
                           ["AlCCH;v=0;", "AlCCH"],
                           ["AlCCH;v=0;hyp1", "AlCCH"],
                           ["AlCN;v=0;", "AlCN"],
                           ["AlCN;v=0;hyp1", "AlCN"],
                           ["AlCl-35;v=0;", "Al$^{35}$Cl"],
                           ["AlCl-35;v=10;", "Al$^{35}$Cl, v=10"],
                           ["AlCl-35;v=1;", "Al$^{35}$Cl, v=1"],
                           ["AlCl-35;v=2;", "Al$^{35}$Cl, v=2"],
                           ["AlCl-35;v=3;", "Al$^{35}$Cl, v=3"],
                           ["AlCl-35;v=4;", "Al$^{35}$Cl, v=4"],
                           ["AlCl-35;v=5;", "Al$^{35}$Cl, v=5"],
                           ["AlCl-35;v=6;", "Al$^{35}$Cl, v=6"],
                           ["AlCl-35;v=7;", "Al$^{35}$Cl, v=7"],
                           ["AlCl-35;v=8;", "Al$^{35}$Cl, v=8"],
                           ["AlCl-35;v=9;", "Al$^{35}$Cl, v=9"],
                           ["AlCl-36;v=0;", "Al$^{36}$Cl"],
                           ["AlCl-36;v=1;", "Al$^{36}$Cl, v=1"],
                           ["AlCl-36;v=2;", "Al$^{36}$Cl, v=2"],
                           ["AlCl-37;v=0;", "Al$^{37}$Cl"],
                           ["AlCl-37;v=10;", "Al$^{37}$Cl, v=10"],
                           ["AlCl-37;v=1;", "Al$^{37}$Cl, v=1"],
                           ["AlCl-37;v=2;", "Al$^{37}$Cl, v=2"],
                           ["AlCl-37;v=3;", "Al$^{37}$Cl, v=3"],
                           ["AlCl-37;v=4;", "Al$^{37}$Cl, v=4"],
                           ["AlCl-37;v=5;", "Al$^{37}$Cl, v=5"],
                           ["AlCl-37;v=6;", "Al$^{37}$Cl, v=6"],
                           ["AlCl-37;v=7;", "Al$^{37}$Cl, v=7"],
                           ["AlCl-37;v=8;", "Al$^{37}$Cl, v=8"],
                           ["AlCl-37;v=9;", "Al$^{37}$Cl, v=9"],
                           ["AlF;v=0;", "AlF"],
                           ["AlF;v=1;", "AlF, v=1"],
                           ["AlF;v=2;", "AlF, v=2"],
                           ["AlF;v=3;", "AlF, v=3"],
                           ["AlF;v=4;", "AlF, v=4"],
                           ["AlF;v=5;", "AlF, v=5"],
                           ["AlH;v=0;", "AlH"],
                           ["AlNC;v=0;", "AlNC"],
                           ["AlO-17;v=0;", "Al$^{17}$O"],
                           ["AlO-17;v=0;hyp2", "Al$^{17}$O"],
                           ["AlO-17;v=1;", "Al$^{17}$O, v=1"],
                           ["AlO-17;v=1;hyp2", "Al$^{17}$O, v=1"],
                           ["AlO-17;v=2;", "Al$^{17}$O, v=2"],
                           ["AlO-17;v=2;hyp2", "Al$^{17}$O, v=2"],
                           ["AlO-18;v=0;", "Al$^{18}$O"],
                           ["AlO-18;v=0;hyp1", "Al$^{18}$O"],
                           ["AlO-18;v=1;", "Al$^{18}$O, v=1"],
                           ["AlO-18;v=1;hyp1", "Al$^{18}$O, v=1"],
                           ["AlO-18;v=2;", "Al$^{18}$O, v=2"],
                           ["AlO-18;v=2;hyp1", "Al$^{18}$O, v=2"],
                           ["AlO;v=0;", "AlO"],
                           ["AlO;v=0;#1", "AlO"],
                           ["AlO;v=0;#2", "AlO"],
                           ["AlO;v=0;hyp1", "AlO"],
                           ["AlO;v=1;", "AlO, v=1"],
                           ["AlO;v=1;hyp1", "AlO, v=1"],
                           ["AlO;v=2;", "AlO, v=2"],
                           ["AlO;v=2;hyp1", "AlO, v=2"],
                           ["AlO;v=3;", "AlO, v=3"],
                           ["AlO;v=3;hyp1", "AlO, v=3"],
                           ["AlO;v=4;", "AlO, v=4"],
                           ["AlO;v=4;hyp1", "AlO, v=4"],
                           ["AlO;v=5;", "AlO, v=5"],
                           ["AlO;v=5;hyp1", "AlO, v=5"],
                           ["AlOD;v=0;", "AlOD"],
                           ["AlOD;v=0;#1", "AlOD"],
                           ["AlOH;v=0;", "AlOH"],
                           ["AlOH;v=0;#1", "AlOH"],
                           ["AlS;v=0;", "AlS"],
                           ["AlSH;v=0;", "AlSH"],
                           ["Ar-36-H+;v=0;", "$^{36}$ArH$^+$"],
                           ["Ar-38-H+;v=0;", "$^{38}$ArH$^+$"],
                           ["ArD+;v=0;", "ArD$^+$"],
                           ["ArH+;v=0;", "ArH$^+$"],
                           ["B-10-O;v=0;", "$^{10}$BO"],
                           ["BH;v=0;", "BH"],
                           ["BO;v=0;hyp1", "BO"],
                           ["Br-79-O;v=0;", "$^{79}$BrO"],
                           ["Br-79-O;v=0;#1", "$^{79}$BrO"],
                           ["Br-79-O;v=1;", "$^{79}$BrO, v=1"],
                           ["Br-79-O;v=1;#1", "$^{79}$BrO, v=1"],
                           ["Br-81-O;v=0;", "$^{81}$BrO"],
                           ["Br-81-O;v=0;#1", "$^{81}$BrO"],
                           ["Br-81-O;v=1;", "$^{81}$BrO, v=1"],
                           ["Br-81-O;v=1;#1", "$^{81}$BrO, v=1"],
                           ["C-13-C4H;v=0;hyp2", "$^{13}$CC$_4$H"],
                           ["C-13-C4S;v=0;", "$^{13}$CC$_4$S"],
                           ["C-13-C5H;v=0;", "$^{13}$CC$_5$H"],
                           ["C-13-CC-13-S;v=0;", "$^{13}$CC$^{13}$CS"],
                           ["C-13-CCCH;v=0;hyp2", "$^{13}$CCCCH"],
                           ["C-13-CCH;v4=1;", "$^{13}$CCCH, v$_4$=1"],
                           ["C-13-CCH;v=0;", "$^{13}$CCCH"],
                           ["C-13-CCN;v=0;", "$^{13}$CCCN"],
                           ["C-13-CCO;v=0;", "$^{13}$CCCO"],
                           ["C-13-CCO;v=0;#1", "$^{13}$CCCO"],
                           ["C-13-CCO;v=0;#2", "$^{13}$CCCO"],
                           ["C-13-CCS-34;v=0;", "$^{13}$CCC$^{34}$S"],
                           ["C-13-CCS;v=0;", "$^{13}$CCCS"],
                           ["C-13-CH;v=0;", "$^{13}$CCH"],
                           ["C-13-CP;v=0;", "$^{13}$CCP"],
                           ["C-13-D+;v=0;", "$^{13}$CD$^+$"],
                           ["C-13-F+;v=0;", "$^{13}$CF$^+$"],
                           ["C-13-F+;v=1;", "$^{13}$CF$^+$, v=1"],
                           ["C-13-H+;v=0;", "$^{13}$CH$^+$"],
                           ["C-13-H+;v=1-0;", "$^{13}$CH$^+$, v=1-0"],
                           ["C-13-H2(OH)CHO;v=0;", "$^{13}$CH$_2$(OH)CHO"],
                           ["C-13-H2CHCN;v=0;", "$^{13}$CH$_2$CHCN"],
                           ["C-13-H2CHCN;v=0;#1", "$^{13}$CH$_2$CHCN"],
                           ["C-13-H2DCCH;v=0;", "$^{13}$CH$_2$DCCH"],
                           ["C-13-H2NH;v=0;", "$^{13}$CH$_2$NH"],
                           ["C-13-H2NH;v=0;#1", "$^{13}$CH$_2$NH"],
                           ["C-13-H3(CH3)CHCN;v=0;", "$^{13}$CH$_3$(CH$_3$)CHCN"],
                           ["C-13-H3C-13-CH;v=0;", "$^{13}$CH$_3 \\! ^{13}$CCH"],
                           ["C-13-H3C-13-H2C-13-N;v=0;", "$^{13}$CH$_3 \\! ^{13}$CH$_2 \\! ^{13}$CN"],
                           ["C-13-H3C-13-H2CN;v=0;", "$^{13}$CH$_3 \\! ^{13}$CH$_2$CN"],
                           ["C-13-H3C-13-N;v=0;", "$^{13}$CH$_3 \\! ^{13}$CN"],
                           ["C-13-H3C-13-N;v=0;#1", "$^{13}$CH$_3 \\! ^{13}$CN"],
                           ["C-13-H3C-13-N;v=0;#2", "$^{13}$CH$_3 \\! ^{13}$CN"],
                           ["C-13-H3C-13-N;v=0;A", "$^{13}$CH$_3 \\! ^{13}$CN A"],
                           ["C-13-H3C-13-N;v=0;E", "$^{13}$CH$_3 \\! ^{13}$CN E"],
                           ["C-13-H3C-13-N;v=0;hyp1", "$^{13}$CH$_3 \\! ^{13}$CN"],
                           ["C-13-H3C-13-N;v=0;hyp1#1", "$^{13}$CH$_3 \\! ^{13}$CN"],
                           ["C-13-H3CC-13-H;v=0;", "$^{13}$CH$_3$C$^{13}$CH"],
                           ["C-13-H3CCD;v=0;", "$^{13}$CH$_3$CCD"],
                           ["C-13-H3CCH;v=0;", "$^{13}$CH$_3$CCH"],
                           ["C-13-H3CH2C-13-N;v=0;", "$^{13}$CH$_3$CH$_2 \\! ^{13}$CN"],
                           ["C-13-H3CH2CN;v=0;", "$^{13}$CH$_3$CH$_2$CN"],
                           ["C-13-H3CH2CN;v=0;#1", "$^{13}$CH$_3$CH$_2$CN"],
                           ["C-13-H3CH2CN;v=0;#2", "$^{13}$CH$_3$CH$_2$CN"],
                           ["C-13-H3CHO;v15=1;", "$^{13}$CH$_3$CHO, v$_{15}$=1"],
                           ["C-13-H3CHO;v=0;", "$^{13}$CH$_3$CHO"],
                           ["C-13-H3CN;v=0;", "$^{13}$CH$_3$CN"],
                           ["C-13-H3CN;v=0;A", "$^{13}$CH$_3$CN A"],
                           ["C-13-H3CN;v=0;E", "$^{13}$CH$_3$CN E"],
                           ["C-13-H3CN;v=0;hyp1", "$^{13}$CH$_3$CN"],
                           ["C-13-H3CN;v=0;hyp1;A", "$^{13}$CH$_3$CN A"],
                           ["C-13-H3CN;v=0;hyp1;E", "$^{13}$CH$_3$CN E"],
                           ["C-13-H3D;v=0;", "$^{13}$CH$_3$D"],
                           ["C-13-H3D;v=0;#1", "$^{13}$CH$_3$D"],
                           ["C-13-H3D;v=0;#2", "$^{13}$CH$_3$D"],
                           ["C-13-H3OH;v12=1;", "$^{13}$CH$_3$OH, v$_{12}$=1"],
                           ["C-13-H3OH;v12=1;A", "$^{13}$CH$_3$OH, v$_{12}$=1 A"],
                           ["C-13-H3OH;v12=1;E", "$^{13}$CH$_3$OH, v$_{12}$=1 E"],
                           ["C-13-H3OH;v=0;", "$^{13}$CH$_3$OH"],
                           ["C-13-H3OH;v=0;A", "$^{13}$CH$_3$OH A"],
                           ["C-13-H3OH;v=0;E", "$^{13}$CH$_3$OH E"],
                           ["C-13-H3SH;v12=1;", "$^{13}$CH$_3$SH, v$_{12}$=1"],
                           ["C-13-H3SH;v12=2;", "$^{13}$CH$_3$SH, v$_{12}$=2"],
                           ["C-13-H3SH;v=0;", "$^{13}$CH$_3$SH"],
                           ["C-13-H;v=0;", "$^{13}$CH"],
                           ["C-13-H;v=0;#1", "$^{13}$CH"],
                           ["C-13-N-15;v=0;", "$^{13}$C$^{15}$N"],
                           ["C-13-N;v=0;", "$^{13}$CN"],
                           ["C-13-N;v=0;#1", "$^{13}$CN"],
                           ["C-13-N;v=0;#2", "$^{13}$CN"],
                           ["C-13-O+;v=0;", "$^{13}$CO$^+$"],
                           ["C-13-O-17;v=0;", "$^{13}$C$^{17}$O"],
                           ["C-13-O-18+;v=0;", "$^{13}$C$^{18}$O$^+$"],
                           ["C-13-O-18;v=0;", "$^{13}$C$^{18}$O"],
                           ["C-13-O;v=0;", "$^{13}$CO"],
                           ["C-13-O;v=0;#1", "$^{13}$CO"],
                           ["C-13-O;v=0;#2", "$^{13}$CO"],
                           ["C-13-O;v=0;hyp1", "$^{13}$CO"],
                           ["C-13-S-33;v=0;", "$^{13}$C$^{33}$S"],
                           ["C-13-S-34;v=0;", "$^{13}$C$^{34}$S"],
                           ["C-13-S-36;v=0;", "$^{13}$C$^{36}$S"],
                           ["C-13-S;v=0;", "$^{13}$CS"],
                           ["C-13-S;v=0;#1", "$^{13}$CS"],
                           ["C-13-S;v=0;#2", "$^{13}$CS"],
                           ["C-13-S;v=1-0;", "$^{13}$CS, v=1-0"],
                           ["C-13-S;v=1;", "$^{13}$CS, v=1"],
                           ["C-14-H+;v=0;", "$^{14}$CH$^+$"],
                           ["C-34-S;v=1-0;", "C$^{34}$S, v=1-0"],
                           ["C10H;v=0;", "C$_{10}$H"],
                           ["C2C-13-C2H;v=0;hyp2", "C$_2 \\! ^{13}$CC$_2$H"],
                           ["C2C-13-C2S;v=0;", "C$_2 \\! ^{13}$CC$_2$S"],
                           ["C2C-13-C3H;v=0;", "C$_2 \\! ^{13}$CC$_3$H"],
                           ["C2Cl-37;v=0;", "C$_2 \\! ^{37}$Cl"],
                           ["C2Cl;v=0;", "C$_2$Cl"],
                           ["C2H-;v=0;", "C$_2$H$^-$"],
                           ["C2H2;(v4,v5)=(0,1)-(0,0);", "C$_2$H$_2$, (v$_4$,v$_5$)=(0,1)-(0,0)"],
                           ["C2H2;(v4,v5)=(0,1)-(0,0);#1", "C$_2$H$_2$, (v$_4$,v$_5$)=(0,1)-(0,0)"],
                           ["C2H2;(v4,v5)=(0,1)-(0,0);ortho", "ortho-C$_2$H$_2$, (v$_4$,v$_5$)=(0,1)-(0,0)"],
                           ["C2H2;(v4,v5)=(0,1)-(0,0);para", "para-C$_2$H$_2$, (v$_4$,v$_5$)=(0,1)-(0,0)"],
                           ["C2H2;(v4,v5)=(1,0)-(0,1);", "C$_2$H$_2$, (v$_4$,v$_5$)=(1,0)-(0,1)"],
                           ["C2H2;(v4,v5)=(1,0)-(0,1);#1", "C$_2$H$_2$, (v$_4$,v$_5$)=(1,0)-(0,1)"],
                           ["C2H2;(v4,v5)=(1,0)-(0,1);#2", "C$_2$H$_2$, (v$_4$,v$_5$)=(1,0)-(0,1)"],
                           ["C2H2;(v4,v5)=(1,0)-(0,1);ortho", "ortho-C$_2$H$_2$, (v$_4$,v$_5$)=(1,0)-(0,1)"],
                           ["C2H2;(v4,v5)=(1,0)-(0,1);ortho#1", "C$_2$H$_2$, (v$_4$,v$_5$)=(1,0)-(0,1)"],
                           ["C2H2;(v4,v5)=(1,0)-(0,1);para", "para-C$_2$H$_2$, (v$_4$,v$_5$)=(1,0)-(0,1)"],
                           ["C2H2;(v4,v5)=(1,0)-(0,1);para#1", "C$_2$H$_2$, (v$_4$,v$_5$)=(1,0)-(0,1)"],
                           ["C2H2;(v4,v5)=(1,1)-(0,0);", "C$_2$H$_2$, (v$_4$,v$_5$)=(1,1)-(0,0)"],
                           ["C2H2;(v4,v5)=(1,1)-(0,0);#1", "C$_2$H$_2$, (v$_4$,v$_5$)=(1,1)-(0,0)"],
                           ["C2H2;(v4,v5)=(1,1)-(0,0);ortho", "ortho-C$_2$H$_2$, (v$_4$,v$_5$)=(1,1)-(0,0)"],
                           ["C2H2;(v4,v5)=(1,1)-(0,0);para", "para-C$_2$H$_2$, (v$_4$,v$_5$)=(1,1)-(0,0)"],
                           ["C2H2;(v4,v5)=(1,1)-(1,0);", "C$_2$H$_2$, (v$_4$,v$_5$)=(1,1)-(1,0)"],
                           ["C2H2;(v4,v5)=(1,1)-(1,0);#1", "C$_2$H$_2$, (v$_4$,v$_5$)=(1,1)-(1,0)"],
                           ["C2H2;(v4,v5)=(1,1)-(1,0);ortho", "ortho-C$_2$H$_2$, (v$_4$,v$_5$)=(1,1)-(1,0)"],
                           ["C2H2;(v4,v5)=(1,1)-(1,0);para", "para-C$_2$H$_2$, (v$_4$,v$_5$)=(1,1)-(1,0)"],
                           ["C2H2;(v4,v5)=(2,0)-(0,1);", "C$_2$H$_2$, (v$_4$,v$_5$)=(2,0)-(0,1)"],
                           ["C2H2;(v4,v5)=(2,0)-(0,1);#1", "C$_2$H$_2$, (v$_4$,v$_5$)=(2,0)-(0,1)"],
                           ["C2H2;(v4,v5)=(2,0)-(0,1);ortho", "ortho-C$_2$H$_2$, (v$_4$,v$_5$)=(2,0)-(0,1)"],
                           ["C2H2;(v4,v5)=(2,0)-(0,1);para", "para-C$_2$H$_2$, (v$_4$,v$_5$)=(2,0)-(0,1)"],
                           ["C2H2;(v4,v5)=(2,0)-(1,1);", "C$_2$H$_2$, (v$_4$,v$_5$)=(2,0)-(1,1)"],
                           ["C2H2;(v4,v5)=(2,0)-(1,1);#1", "C$_2$H$_2$, (v$_4$,v$_5$)=(2,0)-(1,1)"],
                           ["C2H2;(v4,v5)=(2,0)-(1,1);ortho", "ortho-C$_2$H$_2$, (v$_4$,v$_5$)=(2,0)-(1,1)"],
                           ["C2H2;(v4,v5)=(2,0)-(1,1);para", "para-C$_2$H$_2$, (v$_4$,v$_5$)=(2,0)-(1,1)"],
                           ["C2H2O2;v=0;", "C$_2$H$_2$O$_2$"],
                           ["C2H2O2;v=0;ortho", "ortho-C$_2$H$_2$O$_2$"],
                           ["C2H2O2;v=0;para", "para-C$_2$H$_2$O$_2$"],
                           ["C2H3+;v=0;", "C$_2$H$_3 \\! ^+$"],
                           ["C2H3+;v=0;ortho", "ortho-C$_2$H$_3 \\! ^+$"],
                           ["C2H3+;v=0;para", "para-C$_2$H$_3 \\! ^+$"],
                           ["C2H3C3N;v=0;", "C$_2$H$_3$C$_3$N"],
                           ["C2H3CHO;v18=1;", "C$_2$H$_3$CHO, v$_{18}$=1"],
                           ["C2H3CHO;v18=1;#1", "C$_2$H$_3$CHO, v$_{18}$=1"],
                           ["C2H3CHO;v=0;", "C$_2$H$_3$CHO"],
                           ["C2H3CHO;v=0;#1", "C$_2$H$_3$CHO"],
                           ["C2H3CHO;v=0;#2", "C$_2$H$_3$CHO"],
                           ["C2H3CN;(v11,v15)=(1,0)-(0,0);", "C$_2$H$_3$CN, (v$_{11}$,v$_{15}$)=(1,0)-(0,0)"],
                           ["C2H3CN;(v11,v15)=(1,0)-(0,0);#1", "C$_2$H$_3$CN, (v$_{11}$,v$_{15}$)=(1,0)-(0,0)"],
                           ["C2H3CN;v10=1;private", "C$_2$H$_3$CN, v$_{10}$=1"],
                           ["C2H3CN;v10=1;v11=1;private", "C$_2$H$_3$CN, v$_{10}$=1, v$_{11}$=1"],
                           ["C2H3CN;v11=1;", "C$_2$H$_3$CN, v$_{11}$=1"],
                           ["C2H3CN;v11=1;#1", "C$_2$H$_3$CN, v$_{11}$=1"],
                           ["C2H3CN;v11=1;private", "C$_2$H$_3$CN, v$_{11}$=1"],
                           ["C2H3CN;v11=1;v15=1;private", "C$_2$H$_3$CN, v$_{11}$=1, v$_{15}$=1"],
                           ["C2H3CN;v11=2;private", "C$_2$H$_3$CN, v$_{11}$=2"],
                           ["C2H3CN;v11=2;v15=1;private", "C$_2$H$_3$CN, v$_{11}$=2, v$_{15}$=1"],
                           ["C2H3CN;v11=3;private", "C$_2$H$_3$CN, v$_{11}$=3"],
                           ["C2H3CN;v14=1;private", "C$_2$H$_3$CN, v$_{14}$=1"],
                           ["C2H3CN;v15=1;", "C$_2$H$_3$CN, v$_{15}$=1"],
                           ["C2H3CN;v15=1;#1", "C$_2$H$_3$CN, v$_{15}$=1"],
                           ["C2H3CN;v15=1;private", "C$_2$H$_3$CN, v$_{15}$=1"],
                           ["C2H3CN;v15=2;private", "C$_2$H$_3$CN, v$_{15}$=2"],
                           ["C2H3CN;v=0;", "C$_2$H$_3$CN"],
                           ["C2H3CN;v=0;#1", "C$_2$H$_3$CN"],
                           ["C2H3CN;v=0;#2", "C$_2$H$_3$CN"],
                           ["C2H3CN;v=0;private", "C$_2$H$_3$CN"],
                           ["C2H3D;v=0;", "C$_2$H$_3$D"],
                           ["C2H3NC;v=0;", "C$_2$H$_3$NC"],
                           ["C2H3NC;v=0;#1", "C$_2$H$_3$NC"],
                           ["C2H3NH2;v=0;", "C$_2$H$_3$NH$_2$"],
                           ["C2H3NH2;v=0;#1", "C$_2$H$_3$NH$_2$"],
                           ["C2H3OCHO;v=0;", "C$_2$H$_3$OCHO"],
                           ["C2H5C-13-N;v=0;", "C$_2$H$_5 \\! ^{13}$CN"],
                           ["C2H5CCH;v=0;", "C$_2$H$_5$CCH"],
                           ["C2H5CHCNCH3;v=0;", "C$_2$H$_5$CHCNCH$_3$"],
                           ["C2H5CN-15;v=0;", "C$_2$H$_5$C$^{15}$N"],
                           ["C2H5CN-15;v=0;#1", "C$_2$H$_5$C$^{15}$N"],
                           ["C2H5CN-15;v=0;#2", "C$_2$H$_5$C$^{15}$N"],
                           ["C2H5CN;v12=1;", "C$_2$H$_5$CN, v$_{12}$=1"],
                           ["C2H5CN;v12=1;private", "C$_2$H$_5$CN, v$_{12}$=1"],
                           ["C2H5CN;v13+v21=1;private", "C$_2$H$_5$CN, v$_{13}$, v$_{21}$=1"],
                           ["C2H5CN;v20=1;", "C$_2$H$_5$CN, v$_{20}$=1"],
                           ["C2H5CN;v20=1;private", "C$_2$H$_5$CN, v$_{20}$=1"],
                           ["C2H5CN;v=0;", "C$_2$H$_5$CN"],
                           ["C2H5CN;v=0;#1", "C$_2$H$_5$CN"],
                           ["C2H5CN;v=0;#3", "C$_2$H$_5$CN"],
                           ["C2H5CN;v=0;hyp1", "C$_2$H$_5$CN"],
                           ["C2H5CN;v=0;private", "C$_2$H$_5$CN"],
                           ["C2H5NC;v=0;", "C$_2$H$_5$NC"],
                           ["C2H5NCO;v=0;", "C$_2$H$_5$NCO"],
                           ["C2H5NH2;v=0;", "C$_2$H$_5$NH$_2$"],
                           ["C2H5OCHO;v=0;", "C$_2$H$_5$OCHO"],
                           ["C2H5OH;v=0;#1", "C$_2$H$_5$OH"],
                           ["C2N;v=0;", "C$_2$N"],
                           ["C2O;v=0;", "C$_2$O"],
                           ["C2O;v=0;#1", "C$_2$O"],
                           ["C3;(v1,v2,v3)=(0,1,0)-(0,0,0);", "C$_3$, (v$_1$,v$_2$,v$_3$)=(0,1,0)-(0,0,0)"],
                           ["C3C-13-C2H;v=0;", "C$_3 \\! ^{13}$CC$_2$H"],
                           ["C3C-13-CH;v=0;hyp2", "C$_3 \\! ^{13}$CCH"],
                           ["C3C-13-CS;v=0;", "C$_3 \\! ^{13}$CCS"],
                           ["C3CHCH2;v=0;", "C$_3$CHCH$_2$"],
                           ["C3Cl;v=0;", "C$_3$Cl"],
                           ["C3D;v4=1-0;", "C$_3$D, v$_4$=1-0"],
                           ["C3D;v4=1-0;hyp1", "C$_3$D, v$_4$=1-0"],
                           ["C3D;v4=1;", "C$_3$D, v$_4$=1"],
                           ["C3D;v4=1;hyp1", "C$_3$D, v$_4$=1"],
                           ["C3D;v=0;", "C$_3$D"],
                           ["C3D;v=0;hyp1", "C$_3$D"],
                           ["C3F;v=0;", "C$_3$F"],
                           ["C3H+;v=0;", "C$_3$H$^+$"],
                           ["C3H6O2;v=0;", "C$_3$H$_6$O$_2$"],
                           ["C3H6O2;v=0;#1", "C$_3$H$_6$O$_2$"],
                           ["C3H7CN;v=0;", "C$_3$H$_7$CN"],
                           ["C3H7CN;v=0;#1", "C$_3$H$_7$CN"],
                           ["C3H8;v14=1;", "C$_3$H$_8$, v$_{14}$=1"],
                           ["C3H8;v14=1;#1", "C$_3$H$_8$, v$_{14}$=1"],
                           ["C3H8;v27=1;", "C$_3$H$_8$, v$_{27}$=1"],
                           ["C3H8;v27=1;#1", "C$_3$H$_8$, v$_{27}$=1"],
                           ["C3H8;v=0;", "C$_3$H$_8$"],
                           ["C3H8;v=0;#1", "C$_3$H$_8$"],
                           ["C3H;v4=1-0;", "C$_3$H, v$_4$=1-0"],
                           ["C3H;v4=1;", "C$_3$H, v$_4$=1"],
                           ["C3H;v=0;", "C$_3$H"],
                           ["C3N-;v=0;", "C$_3$N$^-$"],
                           ["C3O2;v7=1-0;", "C$_3$O$_2$, v$_7$=1-0"],
                           ["C3O3H6;v28=1;", "C$_3$O$_3$H$_6$, v$_{28}$=1"],
                           ["C3O3H6;v28=1;#1", "C$_3$O$_3$H$_6$, v$_{28}$=1"],
                           ["C3O3H6;v29=1;", "C$_3$O$_3$H$_6$, v$_{29}$=1"],
                           ["C3O3H6;v29=1;#1", "C$_3$O$_3$H$_6$, v$_{29}$=1"],
                           ["C3O3H6;v30=1;", "C$_3$O$_3$H$_6$, v$_{30}$=1"],
                           ["C3O3H6;v30=1;#1", "C$_3$O$_3$H$_6$, v$_{30}$=1"],
                           ["C3O3H6;v=0;", "C$_3$O$_3$H$_6$"],
                           ["C3O3H6;v=0;#1", "C$_3$O$_3$H$_6$"],
                           ["C3S;v5=1;", "C$_3$S, v$_5$=1"],
                           ["C4C-13-CH;v=0;", "C$_4 \\! ^{13}$CCH"],
                           ["C4C-13-H;v=0;hyp2", "C$_4 \\! ^{13}$CH"],
                           ["C4C-13-S;v=0;", "C$_4 \\! ^{13}$CS"],
                           ["C4D-;v=0;", "C$_4$D$^-$"],
                           ["C4H-;v=0;", "C$_4$H$^-$"],
                           ["C4N;v=0;", "C$_4$N"],
                           ["C4N;v=0;hyp1", "C$_4$N"],
                           ["C4O;v=0;", "C$_4$O"],
                           ["C4O;v=0;#1", "C$_4$O"],
                           ["C4S;v=0;", "C$_4$S"],
                           ["C4Si;v=0;", "C$_4$Si"],
                           ["C4Si;v=0;#1", "C$_4$Si"],
                           ["C5C-13-H;v=0;", "C$_5 \\! ^{13}$CH"],
                           ["C5D;v=0;", "C$_5$D"],
                           ["C5D;v=0;#1", "C$_5$D"],
                           ["C5D;v=0;#2", "C$_5$D"],
                           ["C5H;v=0;hyp1", "C$_5$H"],
                           ["C5N-;v=0;", "C$_5$N$^-$"],
                           ["C5N;v=0;", "C$_5$N"],
                           ["C5O;v=0;", "C$_5$O"],
                           ["C5O;v=0;#1", "C$_5$O"],
                           ["C5S-34;v=0;", "C$_5 \\! ^{34}$S"],
                           ["C5S-34;v=0;#1", "C$_5 \\! ^{34}$S"],
                           ["C5S-34;v=0;#2", "C$_5 \\! ^{34}$S"],
                           ["C5S;v=0;", "C$_5$S"],
                           ["C5S;v=0;#1", "C$_5$S"],
                           ["C5S;v=0;#2", "C$_5$S"],
                           ["C6D;v=0;", "C$_6$D"],
                           ["C6D;v=0;hyp1", "C$_6$D"],
                           ["C6H-;v=0;", "C$_6$H$^-$"],
                           ["C6H;v=0;", "C$_6$H"],
                           ["C6H;v=0;hyp1", "C$_6$H"],
                           ["C6O;v=0;", "C$_6$O"],
                           ["C6O;v=0;#1", "C$_6$O"],
                           ["C6S;v=0;", "C$_6$S"],
                           ["C7H;v=0;", "C$_7$H"],
                           ["C7O;v=0;", "C$_7$O"],
                           ["C7O;v=0;#1", "C$_7$O"],
                           ["C7S;v=0;", "C$_7$S"],
                           ["C8H-;v=0;", "C$_8$H$^-$"],
                           ["C8H;v=0;", "C$_8$H"],
                           ["C8O;v=0;", "C$_8$O"],
                           ["C8O;v=0;#1", "C$_8$O"],
                           ["C9H;v=0;", "C$_9$H"],
                           ["C9O;v=0;", "C$_9$O"],
                           ["C9O;v=0;#1", "C$_9$O"],
                           ["C9S;v=0;", "C$_9$S"],
                           ["CC-13-C-13-S;v=0;", "C$^{13}$C$^{13}$CS"],
                           ["CC-13-C3H;v=0;hyp2", "C$^{13}$CC$_3$H"],
                           ["CC-13-C3S;v=0;", "C$^{13}$CC$_3$S"],
                           ["CC-13-C4H;v=0;", "C$^{13}$CC$_4$H"],
                           ["CC-13-CCH;v=0;hyp2", "C$^{13}$CCCH"],
                           ["CC-13-CH;v4=1;", "C$^{13}$CCH, v$_4$=1"],
                           ["CC-13-CH;v=0;", "C$^{13}$CCH"],
                           ["CC-13-CN;v=0;", "C$^{13}$CCN"],
                           ["CC-13-CO;v=0;", "C$^{13}$CCO"],
                           ["CC-13-CO;v=0;#1", "C$^{13}$CCO"],
                           ["CC-13-CO;v=0;#2", "C$^{13}$CCO"],
                           ["CC-13-CS-34;v=0;", "C$^{13}$CC$^{34}$S"],
                           ["CC-13-H;v=0;", "C$^{13}$CH"],
                           ["CC-13-P;v=0;", "C$^{13}$CP"],
                           ["CC-13CS;v=0;", "CC$^{13}$CS"],
                           ["CCC-13-CH;v=0;hyp2", "CC$^{13}$CCH"],
                           ["CCC-13-H;v4=1;", "CC$^{13}$CH, v$_4$=1"],
                           ["CCC-13-H;v=0;", "CC$^{13}$CH"],
                           ["CCC-13-N;v=0;", "CC$^{13}$CN"],
                           ["CCC-13-O;v=0;", "CC$^{13}$CO"],
                           ["CCC-13-O;v=0;#1", "CC$^{13}$CO"],
                           ["CCC-13-O;v=0;#2", "CC$^{13}$CO"],
                           ["CCC-13-S-34;v=0;", "CC$^{13}$C$^{34}$S"],
                           ["CCC-13-S;v=0;", "CC$^{13}$CS"],
                           ["CCCC-13-H;v=0;hyp2", "CCC$^{13}$CH"],
                           ["CCCCD;v=0;", "CCCCD"],
                           ["CCCCD;v=0;hyp1", "CCCCD"],
                           ["CCCCH;v5=1;hyp1", "CCCCH, v$_5$=1"],
                           ["CCCCH;v6=1,v7=1;hyp1", "CCCCH, v$_6$=1,v7=1"],
                           ["CCCCH;v6=1;hyp1", "CCCCH, v$_6$=1"],
                           ["CCCCH;v7=1;hyp1", "CCCCH, v$_7$=1"],
                           ["CCCCH;v7=2;hyp1", "CCCCH, v$_7$=2"],
                           ["CCCCH;v7=2;hyp1#1", "CCCCH, v$_7$=2"],
                           ["CCCCH;v=0;hyp1", "CCCCH"],
                           ["CCCN-15;v=0;", "CCC$^{15}$N"],
                           ["CCCN;v5=1;", "CCCN, v$_5$=1"],
                           ["CCCN;v=0;", "CCCN"],
                           ["CCCN;v=0;#1", "CCCN"],
                           ["CCCN;v=0;#2", "CCCN"],
                           ["CCCO-18;v=0;", "CCC$^{18}$O"],
                           ["CCCO-18;v=0;#1", "CCC$^{18}$O"],
                           ["CCCO-18;v=0;#2", "CCC$^{18}$O"],
                           ["CCCO;v5=1;", "CCCO, v$_5$=1"],
                           ["CCCO;v=0;", "CCCO"],
                           ["CCCO;v=0;#1", "CCCO"],
                           ["CCCO;v=0;#2", "CCCO"],
                           ["CCCS-34;v=0;", "CCC$^{34}$S"],
                           ["CCCS-34;v=0;#1", "CCC$^{34}$S"],
                           ["CCCS-34;v=0;#2", "CCC$^{34}$S"],
                           ["CCCS;v=0;", "CCCS"],
                           ["CCCS;v=0;#1", "CCCS"],
                           ["CCCS;v=0;#2", "CCCS"],
                           ["CCD;v=0;", "CCD"],
                           ["CCH;(v1,v2,v3)=(0,0,1)-(0,0,0);", "CCH, (v$_1$,v$_2$,v$_3$)=(0,0,1)-(0,0,0)"],
                           ["CCH;(v1,v2,v3)=(0,1,0)-(0,0,0);", "CCH, (v$_1$,v$_2$,v$_3$)=(0,1,0)-(0,0,0)"],
                           ["CCH;(v1,v2,v3)=(0,1,1)-(0,0,0);", "CCH, (v$_1$,v$_2$,v$_3$)=(0,1,1)-(0,0,0)"],
                           ["CCH;(v1,v2,v3)=(0,5,0)-(0,0,0);", "CCH, (v$_1$,v$_2$,v$_3$)=(0,5,0)-(0,0,0)"],
                           ["CCH;v2=1;", "CCH, v$_2$=1"],
                           ["CCH;v2=2;", "CCH, v$_2$=2"],
                           ["CCH;v3=1;", "CCH, v$_3$=1"],
                           ["CCH;v=0;", "CCH"],
                           ["CCP;v=0;", "CCP"],
                           ["CCS;v=0;", "CCS"],
                           ["CD+;v=0;", "CD$^+$"],
                           ["CD+;v=1-0;", "CD$^+$, v=1-0"],
                           ["CD3CN;v=0;", "CD$_3$CN"],
                           ["CD;v=0;", "CD"],
                           ["CD;v=0;#1", "CD"],
                           ["CF+;v=0;", "CF$^+$"],
                           ["CF+;v=1;", "CF$^+$, v=1"],
                           ["CF;v=0;", "CF"],
                           ["CH+;v=0;", "CH$^+$"],
                           ["CH+;v=0;#1", "CH$^+$"],
                           ["CH+;v=0;#2", "CH$^+$"],
                           ["CH+;v=1-0;", "CH$^+$, v=1-0"],
                           ["CH+;v=2-0;", "CH$^+$, v=2-0"],
                           ["CH2(OD)CHO;v=0;", "CH$_2$(OD)CHO"],
                           ["CH2(OH)C-13-HO;v=0;", "CH$_2$(OH)$^{13}$CHO"],
                           ["CH2(OH)CDO;v=0;", "CH$_2$(OH)CDO"],
                           ["CH2(OH)CHO;v=0;", "CH$_2$(OH)CHO"],
                           ["CH2;v=0;", "CH$_2$"],
                           ["CH2;v=0;ortho", "ortho-CH$_2$"],
                           ["CH2;v=0;para", "para-CH$_2$"],
                           ["CH2C-13-HCN;v=0;", "CH$_2 \\! ^{13}$CHCN"],
                           ["CH2C-13-HCN;v=0;#1", "CH$_2 \\! ^{13}$CHCN"],
                           ["CH2CDCN;v=0;", "CH$_2$CDCN"],
                           ["CH2CDCN;v=0;#1", "CH$_2$CDCN"],
                           ["CH2CHC-13-N;v=0;", "CH$_2$CH$^{13}$CN"],
                           ["CH2CHC-13-N;v=0;#1", "CH$_2$CH$^{13}$CN"],
                           ["CH2CHC-15-N;v=0;", "CH$_2$CHC$^{15}$N"],
                           ["CH2CHC-15-N;v=0;#1", "CH$_2$CHC$^{15}$N"],
                           ["CH2CHCNH+;v=0;", "CH$_2$CHCNH$^+$"],
                           ["CH2D+;v=0;", "CH$_2$D$^+$"],
                           ["CH2D+;v=0;ortho", "ortho-CH$_2$D$^+$"],
                           ["CH2D+;v=0;para", "para-CH$_2$D$^+$"],
                           ["CH2DC-13-CH;v=0;", "CH$_2$D$^{13}$CCH"],
                           ["CH2DCC-13-H;v=0;", "CH$_2$DC$^{13}$CH"],
                           ["CH2DCCH;v=0;", "CH$_2$DCCH"],
                           ["CH2DCH2CN-a;v=0;", "CH$_2$DCH$_2$CN-a"],
                           ["CH2DCH2CN-a;v=0;#1", "CH$_2$DCH$_2$CN-a"],
                           ["CH2DCH2CN-s;v=0;", "CH$_2$DCH$_2$CN-s"],
                           ["CH2DCH2CN-s;v=0;#1", "CH$_2$DCH$_2$CN-s"],
                           ["CH2DCHO;v=0;", "CH$_2$DCHO"],
                           ["CH2DCN;v=0;", "CH$_2$DCN"],
                           ["CH2DCN;v=0;#1", "CH$_2$DCN"],
                           ["CH2DCN;v=0;#2", "CH$_2$DCN"],
                           ["CH2DCl-35;v=0;hyp1", "CH$_2$D$^{35}$Cl"],
                           ["CH2DOH;v=0;", "CH$_2$DOH"],
                           ["CH2DOH;v=0;#1", "CH$_2$DOH"],
                           ["CH2DipCH2CN;v=0;", "CH$_2$DipCH$_2$CN"],
                           ["CH2DoopCH2CN;v=0;", "CH$_2$DoopCH$_2$CN"],
                           ["CH2F2;v4=1;", "CH$_2$F$_2$, v$_4$=1"],
                           ["CH2F2;v4=1;#1", "CH$_2$F$_2$, v$_4$=1"],
                           ["CH2F2;v4=1;ortho", "ortho-CH$_2$F$_2$, v$_4$=1"],
                           ["CH2F2;v4=1;para", "para-CH$_2$F$_2$, v$_4$=1"],
                           ["CH2F2;v=0;", "CH$_2$F$_2$"],
                           ["CH2F2;v=0;#1", "CH$_2$F$_2$"],
                           ["CH2F2;v=0;ortho", "ortho-CH$_2$F$_2$"],
                           ["CH2F2;v=0;para", "para-CH$_2$F$_2$"],
                           ["CH2N-15-H;v=0;", "CH$_2 \\! ^{15}$N-H"],
                           ["CH2N-15-H;v=0;#1", "CH$_2 \\! ^{15}$N-H"],
                           ["CH2NH;v=0;", "CH$_2$NH"],
                           ["CH2NH;v=0;#1", "CH$_2$NH"],
                           ["CH2OHCOOH;v=0;", "CH$_2$OHCOOH"],
                           ["CH3C-13-C-13-H;v=0;", "CH$_3 \\! ^{13}$C$^{13}$CH"],
                           ["CH3C-13-CD;v=0;", "CH$_3 \\! ^{13}$CCD"],
                           ["CH3C-13-CH;v=0;", "CH$_3 \\! ^{13}$CCH"],
                           ["CH3C-13-H2C-13-N;v=0;", "CH$_3 \\! ^{13}$CH$_2 \\! ^{13}$CN"],
                           ["CH3C-13-H2CN;v=0;", "CH$_3 \\! ^{13}$CH$_2$CN"],
                           ["CH3C-13-H2CN;v=0;#1", "CH$_3 \\! ^{13}$CH$_2$CN"],
                           ["CH3C-13-H2CN;v=0;#2", "CH$_3 \\! ^{13}$CH$_2$CN"],
                           ["CH3C-13-HO;v15=1;", "CH$_3 \\! ^{13}$CHO, v$_{15}$=1"],
                           ["CH3C-13-HO;v=0;", "CH$_3 \\! ^{13}$CHO"],
                           ["CH3C-13-N;v=0;", "CH$_3 \\! ^{13}$CN"],
                           ["CH3C-13-N;v=0;#1", "CH$_3 \\! ^{13}$CN"],
                           ["CH3C-13-N;v=0;#2", "CH$_3 \\! ^{13}$CN"],
                           ["CH3C-13-N;v=0;hyp1", "CH$_3 \\! ^{13}$CN"],
                           ["CH3C3N;v=0;", "CH$_3$C$_3$N"],
                           ["CH3C4H;v=0;", "CH$_3$C$_4$H"],
                           ["CH3C4H;v=0;ortho", "ortho-CH$_3$C$_4$H"],
                           ["CH3C4H;v=0;para", "para-CH$_3$C$_4$H"],
                           ["CH3C5N;v=0;", "CH$_3$C$_5$N"],
                           ["CH3C6H;v=0;", "CH$_3$C$_6$H"],
                           ["CH3C6H;v=0;ortho", "ortho-CH$_3$C$_6$H"],
                           ["CH3C6H;v=0;para", "para-CH$_3$C$_6$H"],
                           ["CH3C7N;v=0;", "CH$_3$C$_7$N"],
                           ["CH3C7N;v=0;ortho", "ortho-CH$_3$C$_7$N"],
                           ["CH3C7N;v=0;para", "para-CH$_3$C$_7$N"],
                           ["CH3C8H;v=0;", "CH$_3$C$_8$H"],
                           ["CH3C8H;v=0;ortho", "ortho-CH$_3$C$_8$H"],
                           ["CH3C8H;v=0;para", "para-CH$_3$C$_8$H"],
                           ["CH3C9N;v=0;", "CH$_3$C$_9$N"],
                           ["CH3C9N;v=0;ortho", "ortho-CH$_3$C$_9$N"],
                           ["CH3C9N;v=0;para", "para-CH$_3$C$_9$N"],
                           ["CH3CC-13-D;v=0;", "CH$_3$C$^{13}$CD"],
                           ["CH3CC-13-H;v=0;", "CH$_3$C$^{13}$CH"],
                           ["CH3CCD;v=0;", "CH$_3$CCD"],
                           ["CH3CCH;(v9,v10)=(0,2)-(0,0);", "CH$_3$CCH, (v$_9$, v$_{10}$)=(0,2)-(0,0)"],
                           ["CH3CCH;(v9,v10)=(0,2)-(0,0);E", "CH$_3$CCH, (v$_9$, v$_{10}$)=(0,2)-(0,0) E"],
                           ["CH3CCH;(v9,v10)=(1,0)-(0,0);", "CH$_3$CCH, (v$_9$, v$_{10}$)=(1,0)-(0,0)"],
                           ["CH3CCH;(v9,v10)=(1,0)-(0,0);A", "CH$_3$CCH, (v$_9$, v$_{10}$)=(1,0)-(0,0) A"],
                           ["CH3CCH;(v9,v10)=(1,0)-(0,0);E", "CH$_3$CCH, (v$_9$, v$_{10}$)=(1,0)-(0,0) E"],
                           ["CH3CCH;v10=1-0;", "CH$_3$CCH, v$_{10}$=1-0"],
                           ["CH3CCH;v10=1-0;A", "CH$_3$CCH, v$_{10}$=1-0 A"],
                           ["CH3CCH;v10=1-0;E", "CH$_3$CCH, v$_{10}$=1-0 E"],
                           ["CH3CCH;v10=1;", "CH$_3$CCH, v$_{10}$=1"],
                           ["CH3CCH;v10=1;A", "CH$_3$CCH, v$_{10}$=1 A"],
                           ["CH3CCH;v10=1;E", "CH$_3$CCH, v$_{10}$=1 E"],
                           ["CH3CCH;v10=2-1;", "CH$_3$CCH, v$_{10}$=2-1"],
                           ["CH3CCH;v10=2-1;A", "CH$_3$CCH, v$_{10}$=2-1 A"],
                           ["CH3CCH;v10=2-1;E", "CH$_3$CCH, v$_{10}$=2-1 E"],
                           ["CH3CCH;v10=2;", "CH$_3$CCH, v$_{10}$=2"],
                           ["CH3CCH;v10=2;E", "CH$_3$CCH, v$_{10}$=2 E"],
                           ["CH3CCH;v=0;#1", "CH$_3$CCH"],
                           ["CH3CCH;v=0;#2", "CH$_3$CCH"],
                           ["CH3CCH;v=0;A", "CH$_3$CCH A"],
                           ["CH3CCH;v=0;E", "CH$_3$CCH E"],
                           ["CH3CCH;v=0;E#1", "CH$_3$CCH E"],
                           ["CH3CCNC;v=0;", "CH$_3$CCNC"],
                           ["CH3CCNC;v=0;ortho", "ortho-CH$_3$CCNC"],
                           ["CH3CCNC;v=0;para", "para-CH$_3$CCNC"],
                           ["CH3CDO;v15=1-0;", "CH$_3$CDO, v$_{15}$=1-0"],
                           ["CH3CDO;v15=1-0;A", "CH$_3$CDO, v$_{15}$=1-0 A"],
                           ["CH3CDO;v15=1-0;E", "CH$_3$CDO, v$_{15}$=1-0 E"],
                           ["CH3CDO;v15=1;", "CH$_3$CDO, v$_{15}$=1"],
                           ["CH3CDO;v15=1;A", "CH$_3$CDO, v$_{15}$=1 A"],
                           ["CH3CDO;v15=1;E", "CH$_3$CDO, v$_{15}$=1 E"],
                           ["CH3CDO;v=0;", "CH$_3$CDO"],
                           ["CH3CDO;v=0;A", "CH$_3$CDO A"],
                           ["CH3CDO;v=0;E", "CH$_3$CDO E"],
                           ["CH3CH(NH2)CN;v=0;", "CH$_3$CH(NH$_2$)CN"],
                           ["CH3CH2C-13-N;v=0;", "CH$_3$CH$_2 \\! ^{13}$CN"],
                           ["CH3CH2C-13-N;v=0;#1", "CH$_3$CH$_2 \\! ^{13}$CN"],
                           ["CH3CHDCN;v=0;", "CH$_3$CHDCN"],
                           ["CH3CHNH2COOH;v=0;", "CH$_3$CHNH$_2$COOH"],
                           ["CH3CHNH2COOH;v=0;#1", "CH$_3$CHNH$_2$COOH"],
                           ["CH3CHNH2COOH;v=0;hyp1", "CH$_3$CHNH$_2$COOH"],
                           ["CH3CHO;v15=1-0;", "CH$_3$CHO, v$_{15}$=1-0"],
                           ["CH3CHO;v15=1-0;#1", "CH$_3$CHO, v$_{15}$=1-0"],
                           ["CH3CHO;v15=1-0;A", "CH$_3$CHO, v$_{15}$=1-0 A"],
                           ["CH3CHO;v15=1-0;A#1", "CH$_3$CHO, v$_{15}$=1-0 A"],
                           ["CH3CHO;v15=1-0;E", "CH$_3$CHO, v$_{15}$=1-0 E"],
                           ["CH3CHO;v15=1-0;E#1", "CH$_3$CHO, v$_{15}$=1-0 E"],
                           ["CH3CHO;v15=1;", "CH$_3$CHO, v$_{15}$=1"],
                           ["CH3CHO;v15=1;#1", "CH$_3$CHO, v$_{15}$=1"],
                           ["CH3CHO;v15=1;A", "CH$_3$CHO, v$_{15}$=1 A"],
                           ["CH3CHO;v15=1;A#1", "CH$_3$CHO, v$_{15}$=1 A"],
                           ["CH3CHO;v15=1;E", "CH$_3$CHO, v$_{15}$=1 E"],
                           ["CH3CHO;v15=1;E#1", "CH$_3$CHO, v$_{15}$=1 E"],
                           ["CH3CHO;v15=2-0;", "CH$_3$CHO, v$_{15}$=2-0"],
                           ["CH3CHO;v15=2-0;#1", "CH$_3$CHO, v$_{15}$=2-0"],
                           ["CH3CHO;v15=2-0;A", "CH$_3$CHO, v$_{15}$=2-0 A"],
                           ["CH3CHO;v15=2-0;A#1", "CH$_3$CHO, v$_{15}$=2-0 A"],
                           ["CH3CHO;v15=2-0;E", "CH$_3$CHO, v$_{15}$=2-0 E"],
                           ["CH3CHO;v15=2-0;E#1", "CH$_3$CHO, v$_{15}$=2-0 E"],
                           ["CH3CHO;v15=2-1;", "CH$_3$CHO, v$_{15}$=2-1"],
                           ["CH3CHO;v15=2-1;#1", "CH$_3$CHO, v$_{15}$=2-1"],
                           ["CH3CHO;v15=2-1;A", "CH$_3$CHO, v$_{15}$=2-1 A"],
                           ["CH3CHO;v15=2-1;A#1", "CH$_3$CHO, v$_{15}$=2-1 A"],
                           ["CH3CHO;v15=2-1;E", "CH$_3$CHO, v$_{15}$=2-1 E"],
                           ["CH3CHO;v15=2-1;E#1", "CH$_3$CHO, v$_{15}$=2-1 E"],
                           ["CH3CHO;v15=2;", "CH$_3$CHO, v$_{15}$=2"],
                           ["CH3CHO;v15=2;#1", "CH$_3$CHO, v$_{15}$=2"],
                           ["CH3CHO;v15=2;A", "CH$_3$CHO, v$_{15}$=2 A"],
                           ["CH3CHO;v15=2;A#1", "CH$_3$CHO, v$_{15}$=2 A"],
                           ["CH3CHO;v15=2;E", "CH$_3$CHO, v$_{15}$=2 E"],
                           ["CH3CHO;v15=2;E#1", "CH$_3$CHO, v$_{15}$=2 E"],
                           ["CH3CHO;v=0;", "CH$_3$CHO"],
                           ["CH3CHO;v=0;#1", "CH$_3$CHO"],
                           ["CH3CHO;v=0;A", "CH$_3$CHO A"],
                           ["CH3CHO;v=0;A#1", "CH$_3$CHO A"],
                           ["CH3CHO;v=0;E", "CH$_3$CHO E"],
                           ["CH3CHO;v=0;E#1", "CH$_3$CHO E"],
                           ["CH3CHOHCHO;v=0;", "CH$_3$CHOHCHO"],
                           ["CH3CN-15;v=0;", "CH$_3$C$^{15}$N"],
                           ["CH3CN-15;v=0;#1", "CH$_3$C$^{15}$N"],
                           ["CH3CN-15;v=0;#2", "CH$_3$C$^{15}$N"],
                           ["CH3CN-15;v=0;A", "CH$_3$C$^{15}$N A"],
                           ["CH3CN-15;v=0;E", "CH$_3$C$^{15}$N E"],
                           ["CH3CN;(v8,v4,v7)=(1,0,0)-(0,0,0);#1", "CH$_3$CN, (v$_8$,v$_4$,v$_7$)=(1,0,0)-(0,0,0)"],
                           ["CH3CN;(v8,v4,v7)=(2,0,0)-(0,0,0);", "CH$_3$CN, (v$_8$,v$_4$,v$_7$)=(2,0,0)-(0,0,0)"],
                           ["CH3CN;(v8,v4,v7)=(2,0,0)-(1,0,0);", "CH$_3$CN, (v$_8$,v$_4$,v$_7$)=(2,0,0)-(1,0,0)"],
                           ["CH3CN;(v8,v4,v7)=(2,0,0)-(1,0,0);#1", "CH$_3$CN, (v$_8$,v$_4$,v$_7$)=(2,0,0)-(1,0,0)"],
                           ["CH3CN;v8=1;", "CH$_3$CN, v$_8$=1"],
                           ["CH3CN;v8=1;#1", "CH$_3$CN, v$_8$=1"],
                           ["CH3CN;v8=2;", "CH$_3$CN, v$_8$=2"],
                           ["CH3CN;v=0;", "CH$_3$CN"],
                           ["CH3CN;v=0;#1", "CH$_3$CN"],
                           ["CH3CN;v=0;#2", "CH$_3$CN"],
                           ["CH3CNH+;v=0;", "CH$_3$CNH$^+$"],
                           ["CH3CNH+;v=0;A", "CH$_3$CNH$^+$ A"],
                           ["CH3CNH+;v=0;E", "CH$_3$CNH$^+$ E"],
                           ["CH3CNO;v=0;", "CH$_3$CNO"],
                           ["CH3COCH3;v=0;", "CH$_3$COCH$_3$"],
                           ["CH3COCH3;v=0;#1", "CH$_3$COCH$_3$"],
                           ["CH3COOH;v18=1-0;", "CH$_3$COOH, v$_{18}$=1-0"],
                           ["CH3COOH;v18=1;", "CH$_3$COOH, v$_{18}$=1"],
                           ["CH3COOH;v18=2-0;", "CH$_3$COOH, v$_{18}$=2-0"],
                           ["CH3COOH;v18=2-1;", "CH$_3$COOH, v$_{18}$=2-1"],
                           ["CH3COOH;v18=2;", "CH$_3$COOH, v$_{18}$=2"],
                           ["CH3COOH;v=0;", "CH$_3$COOH"],
                           ["CH3CP;v=0;", "CH$_3$CP"],
                           ["CH3CP;v=0;ortho", "ortho-CH$_3$CP"],
                           ["CH3CP;v=0;para", "para-CH$_3$CP"],
                           ["CH3Cl-35;v=0;", "CH$_3 \\! ^{35}$Cl"],
                           ["CH3Cl-35;v=0;#1", "CH$_3 \\! ^{35}$Cl"],
                           ["CH3Cl-37;v=0;", "CH$_3 \\! ^{37}$Cl"],
                           ["CH3Cl-37;v=0;#1", "CH$_3 \\! ^{37}$Cl"],
                           ["CH3Cl-37;v=0;ortho", "ortho-CH$_3 \\! ^{37}$Cl"],
                           ["CH3Cl-37;v=0;para", "para-CH$_3 \\! ^{37}$Cl"],
                           ["CH3D;v=0;", "CH$_3$D"],
                           ["CH3NC;v=0;#1", "CH$_3$NC"],
                           ["CH3NCO;v=0;", "CH$_3$NCO"],
                           ["CH3NCO;v=0;#1", "CH$_3$NCO"],
                           ["CH3NH2;v=0;", "CH$_3$NH$_2$"],
                           ["CH3NH2;v=0;#1", "CH$_3$NH$_2$"],
                           ["CH3O-18-H;v12=1-0;", "CH$_3 \\! ^{18}$OH, v$_{12}$=1-0"],
                           ["CH3O-18-H;v12=1-0;A", "CH$_3 \\! ^{18}$OH, v$_{12}$=1-0 A"],
                           ["CH3O-18-H;v12=1-0;E", "CH$_3 \\! ^{18}$OH, v$_{12}$=1-0 E"],
                           ["CH3O-18-H;v12=1;", "CH$_3 \\! ^{18}$OH, v$_{12}$=1"],
                           ["CH3O-18-H;v12=1;A", "CH$_3 \\! ^{18}$OH, v$_{12}$=1 A"],
                           ["CH3O-18-H;v12=1;E", "CH$_3 \\! ^{18}$OH, v$_{12}$=1 E"],
                           ["CH3O-18-H;v12=2-0;", "CH$_3 \\! ^{18}$OH, v$_{12}$=2-0"],
                           ["CH3O-18-H;v12=2-0;A", "CH$_3 \\! ^{18}$OH, v$_{12}$=2-0 A"],
                           ["CH3O-18-H;v12=2-0;E", "CH$_3 \\! ^{18}$OH, v$_{12}$=2-0 E"],
                           ["CH3O-18-H;v12=2-1;", "CH$_3 \\! ^{18}$OH, v$_{12}$=2-1"],
                           ["CH3O-18-H;v12=2-1;A", "CH$_3 \\! ^{18}$OH, v$_{12}$=2-1 A"],
                           ["CH3O-18-H;v12=2-1;E", "CH$_3 \\! ^{18}$OH, v$_{12}$=2-1 E"],
                           ["CH3O-18-H;v12=2;", "CH$_3 \\! ^{18}$OH, v$_{12}$=2"],
                           ["CH3O-18-H;v12=2;A", "CH$_3 \\! ^{18}$OH, v$_{12}$=2 A"],
                           ["CH3O-18-H;v12=2;E", "CH$_3 \\! ^{18}$OH, v$_{12}$=2 E"],
                           ["CH3O-18-H;v=0;", "CH$_3 \\! ^{18}$OH"],
                           ["CH3O-18-H;v=0;A", "CH$_3 \\! ^{18}$OH A"],
                           ["CH3O-18-H;v=0;E", "CH$_3 \\! ^{18}$OH E"],
                           ["CH3O;v=0;", "CH$_3$O"],
                           ["CH3O;v=0;#1", "CH$_3$O"],
                           ["CH3OC-13-HO;v18=1;", "CH$_3$O$^{13}$CHO, v$_{18}$=1"],
                           ["CH3OC-13-HO;v=0;", "CH$_3$O$^{13}$CHO"],
                           ["CH3OCH2CHO;v=0;", "CH$_3$OCH$_2$CHO"],
                           ["CH3OCH2OH;v=0;private", "CH$_3$OCH$_2$OH"],
                           ["CH3OCH3;v=0;", "CH$_3$OCH$_3$"],
                           ["CH3OCH3;v=0;#1", "CH$_3$OCH$_3$"],
                           ["CH3OCH3;v=0;#2", "CH$_3$OCH$_3$"],
                           ["CH3OCHO;v18=1-0;", "CH$_3$OCHO, v$_{18}$=1-0"],
                           ["CH3OCHO;v18=1-0;#1", "CH$_3$OCHO, v$_{18}$=1-0"],
                           ["CH3OCHO;v18=1;", "CH$_3$OCHO, v$_{18}$=1"],
                           ["CH3OCHO;v18=1;#1", "CH$_3$OCHO, v$_{18}$=1"],
                           ["CH3OCHO;v=0;", "CH$_3$OCHO"],
                           ["CH3OCHO;v=0;#1", "CH$_3$OCHO"],
                           ["CH3OCN;v=0;", "CH$_3$OCN"],
                           ["CH3OH;v12=1-0;", "CH$_3$OH, v$_{12}$=1-0"],
                           ["CH3OH;v12=1-0;#1", "CH$_3$OH, v$_{12}$=1-0"],
                           ["CH3OH;v12=1-0;#2", "CH$_3$OH, v$_{12}$=1-0"],
                           ["CH3OH;v12=1-0;A", "CH$_3$OH, v$_{12}$=1-0 A"],
                           ["CH3OH;v12=1-0;E", "CH$_3$OH, v$_{12}$=1-0 E"],
                           ["CH3OH;v12=1;", "CH$_3$OH, v$_{12}$=1"],
                           ["CH3OH;v12=1;#1", "CH$_3$OH, v$_{12}$=1"],
                           ["CH3OH;v12=1;#2", "CH$_3$OH, v$_{12}$=1"],
                           ["CH3OH;v12=1;A", "CH$_3$OH, v$_{12}$=1 A"],
                           ["CH3OH;v12=1;E", "CH$_3$OH, v$_{12}$=1 E"],
                           ["CH3OH;v12=2-0;", "CH$_3$OH, v$_{12}$=2-0"],
                           ["CH3OH;v12=2-0;#1", "CH$_3$OH, v$_{12}$=2-0"],
                           ["CH3OH;v12=2-0;#2", "CH$_3$OH, v$_{12}$=2-0"],
                           ["CH3OH;v12=2-0;A", "CH$_3$OH, v$_{12}$=2-0 A"],
                           ["CH3OH;v12=2-0;E", "CH$_3$OH, v$_{12}$=2-0 E"],
                           ["CH3OH;v12=2-1;", "CH$_3$OH, v$_{12}$=2-1"],
                           ["CH3OH;v12=2-1;#1", "CH$_3$OH, v$_{12}$=2-1"],
                           ["CH3OH;v12=2-1;#2", "CH$_3$OH, v$_{12}$=2-1"],
                           ["CH3OH;v12=2-1;A", "CH$_3$OH, v$_{12}$=2-1 A"],
                           ["CH3OH;v12=2-1;E", "CH$_3$OH, v$_{12}$=2-1 E"],
                           ["CH3OH;v12=2;", "CH$_3$OH, v$_{12}$=2"],
                           ["CH3OH;v12=2;#1", "CH$_3$OH, v$_{12}$=2"],
                           ["CH3OH;v12=2;#2", "CH$_3$OH, v$_{12}$=2"],
                           ["CH3OH;v12=2;A", "CH$_3$OH, v$_{12}$=2 A"],
                           ["CH3OH;v12=2;E", "CH$_3$OH, v$_{12}$=2 E"],
                           ["CH3OH;v=0;", "CH$_3$OH"],
                           ["CH3OH;v=0;#1", "CH$_3$OH"],
                           ["CH3OH;v=0;#2", "CH$_3$OH"],
                           ["CH3OH;v=0;A", "CH$_3$OH A"],
                           ["CH3OH;v=0;E", "CH$_3$OH E"],
                           ["CH3OOH;v=0;", "CH$_3$OOH"],
                           ["CH3OOH;v=0;#1", "CH$_3$OOH"],
                           ["CH3S-34-H;v12=1;", "CH$_3 \\! ^{34}$SH, v$_{12}$=1"],
                           ["CH3S-34-H;v12=2;", "CH$_3 \\! ^{34}$SH, v$_{12}$=2"],
                           ["CH3S-34-H;v=0;", "CH$_3 \\! ^{34}$SH"],
                           ["CH3SD;v12=1;", "CH$_3$SD, v$_{12}$=1"],
                           ["CH3SD;v12=2;", "CH$_3$SD, v$_{12}$=2"],
                           ["CH3SD;v=0;", "CH$_3$SD"],
                           ["CH;v=0;", "CH"],
                           ["CH;v=0;#1", "CH"],
                           ["CH;v=0;#2", "CH"],
                           ["CH;v=1;", "CH, v=1"],
                           ["CH;v=1;#1", "CH, v=1"],
                           ["CHD2+;v=0;", "CHD$_2 \\! ^+$"],
                           ["CHD2+;v=0;ortho", "ortho-CHD$_2 \\! ^+$"],
                           ["CHD2+;v=0;para", "para-CHD$_2 \\! ^+$"],
                           ["CHD2CN;v=0;", "CHD$_2$CN"],
                           ["CHOCHOHCH2OH;v=0;", "CHOCHOHCH$_2$OH"],
                           ["CHOCOOH;v=0;", "CHOCOOH"],
                           ["CN-15;v=0;", "C$^{15}$N"],
                           ["CN-;v=0;", "CN$^-$"],
                           ["CN;v=0;", "CN"],
                           ["CN;v=0;#1", "CN"],
                           ["CN;v=0;#2", "CN"],
                           ["CN;v=1;", "CN, v=1"],
                           ["CN;v=1;#1", "CN, v=1"],
                           ["CN;v=1;#2", "CN, v=1"],
                           ["CNCN;v=0;", "CNCN"],
                           ["CNCN;v=0;#1", "CNCN"],
                           ["CO+;v=0;", "CO$^+$"],
                           ["CO+;v=0;#1", "CO$^+$"],
                           ["CO+;v=0;#2", "CO$^+$"],
                           ["CO+;v=1;", "CO$^+$, v=1"],
                           ["CO-17;v=0;", "C$^{17}$O"],
                           ["CO-17;v=0;#1", "C$^{17}$O"],
                           ["CO-17;v=0;hyp1", "C$^{17}$O"],
                           ["CO-18+;v=0;", "C$^{18}$O$^+$"],
                           ["CO-18;v=0;", "C$^{18}$O"],
                           ["CO-18;v=0;#1", "C$^{18}$O"],
                           ["CO-18;v=0;#2", "C$^{18}$O"],
                           ["CO;v=0;", "CO"],
                           ["CO;v=0;#1", "CO"],
                           ["CO;v=0;#2", "CO"],
                           ["CO;v=1;", "CO, v=1"],
                           ["CO;v=2;", "CO, v=2"],
                           ["CO;v=3;", "CO, v=3"],
                           ["CP;v=0;", "CP"],
                           ["CP;v=0;#1", "CP"],
                           ["CP;v=0;#2", "CP"],
                           ["CS+;v=0;", "CS$^+$"],
                           ["CS-33;v=0;", "C$^{33}$S"],
                           ["CS-33;v=0;hyp1", "C$^{33}$S"],
                           ["CS-33;v=1;", "C$^{33}$S, v=1"],
                           ["CS-33;v=1;hyp1", "C$^{33}$S, v=1"],
                           ["CS-34;v=0;", "C$^{34}$S"],
                           ["CS-34;v=1;", "C$^{34}$S, v=1"],
                           ["CS-36;v=0;", "C$^{36}$S"],
                           ["CS-36;v=80;", "C$^{36}$S, v=80"],
                           ["CS;v=0;", "CS"],
                           ["CS;v=1-0;", "CS, v=1-0"],
                           ["CS;v=1;", "CS, v=1"],
                           ["CS;v=2-0;", "CS, v=2-0"],
                           ["CS;v=2-1;", "CS, v=2-1"],
                           ["CS;v=2;", "CS, v=2"],
                           ["CS;v=3;", "CS, v=3"],
                           ["CS;v=4;", "CS, v=4"],
                           ["CT+;v=0;", "CT$^+$"],
                           ["Ca-40-D;v=0;", "$^{40}$CaD"],
                           ["Ca-40-D;v=0;#1", "$^{40}$CaD"],
                           ["Ca-40-H;v=0;", "$^{40}$CaH"],
                           ["Ca-40-H;v=0;#1", "$^{40}$CaH"],
                           ["CaC;v=0;", "CaC"],
                           ["CaCCH;v=0;", "CaCCH"],
                           ["CaCH3;v=0;", "CaCH$_3$"],
                           ["CaCH3;v=0;#1", "CaCH$_3$"],
                           ["CaCH3;v=0;ortho", "ortho-CaCH$_3$"],
                           ["CaCH3;v=0;para", "para-CaCH$_3$"],
                           ["CaCl;v=0;", "CaCl"],
                           ["CaF;v=0;", "CaF"],
                           ["CaNC;v=0;", "CaNC"],
                           ["CaNC;v=0;#1", "CaNC"],
                           ["CaNC;v=0;#2", "CaNC"],
                           ["CaO;v=0;", "CaO"],
                           ["CaO;v=1;", "CaO, v=1"],
                           ["CaOH;v=0;", "CaOH"],
                           ["CaOH;v=0;hyp1", "CaOH"],
                           ["CaS;v=0;", "CaS"],
                           ["CaS;v=0;#1", "CaS"],
                           ["CaS;v=0;#2", "CaS"],
                           ["CaS;v=1;", "CaS, v=1"],
                           ["CaS;v=1;#1", "CaS, v=1"],
                           ["CaS;v=1;#2", "CaS, v=1"],
                           ["CaSH;v=0;", "CaSH"],
                           ["Cl-35-NO2;v=0;", "$^{35}$ClNO$_2$"],
                           ["Cl-35-NO2;v=0;#1", "$^{35}$ClNO$_2$"],
                           ["Cl-35-ONO2;v=0;", "$^{35}$ClONO$_2$"],
                           ["Cl-35-ONO2;v=0;#1", "$^{35}$ClONO$_2$"],
                           ["Cl-37-O;v=0;", "$^{37}$ClO"],
                           ["Cl-37-O;v=0;#1", "$^{37}$ClO"],
                           ["Cl-37-O;v=1;", "$^{37}$ClO, v=1"],
                           ["Cl-37-O;v=1;#1", "$^{37}$ClO, v=1"],
                           ["Cl-37-ONO2;v=0;", "$^{37}$ClONO$_2$"],
                           ["Cl-37-ONO2;v=0;#1", "$^{37}$ClONO$_2$"],
                           ["Cl-37-OOCl;v=0;", "$^{37}$ClOOCl"],
                           ["Cl-37-OOCl;v=0;#1", "$^{37}$ClOOCl"],
                           ["ClClO2;v=0;", "ClClO$_2$"],
                           ["ClClO2;v=0;#1", "ClClO$_2$"],
                           ["ClO;v=0;", "ClO"],
                           ["ClO;v=0;#1", "ClO"],
                           ["ClO;v=1;", "ClO, v=1"],
                           ["ClO;v=1;#1", "ClO, v=1"],
                           ["ClOOCl;v=0;", "ClOOCl"],
                           ["ClOOCl;v=0;#1", "ClOOCl"],
                           ["CoC;v=0;", "CoC"],
                           ["Cr-52-C;v=0;", "$^{52}$CrC"],
                           ["Cr-52-Cl-37;v=0;", "$^{52}$Cr$^{37}$Cl"],
                           ["Cr-52-Cl-37;v=0;hyp1", "$^{52}$Cr$^{37}$Cl"],
                           ["Cr-52-Cl;v=0;", "$^{52}$CrCl"],
                           ["Cr-52-Cl;v=0;hyp1", "$^{52}$CrCl"],
                           ["Cr-52-F;v=0;", "$^{52}$CrF"],
                           ["Cr-52-F;v=0;hyp1", "$^{52}$CrF"],
                           ["Cr-52-H;v=0;hyp1", "$^{52}$CrH"],
                           ["Cr-52-N;v=0;", "$^{52}$CrN"],
                           ["Cr-52-N;v=0;hyp1", "$^{52}$CrN"],
                           ["Cr-52-O;v=0;", "$^{52}$CrO"],
                           ["Cu-63-H;v=0;", "$^{63}CuH"],
                           ["Cu-65-H;v=0;", "$^{65}CuH"],
                           ["D2C-13-O;v=0;", "D$_2 \\! ^{13}$CO"],
                           ["D2C-13-O;v=0;#1", "D$_2 \\! ^{13}$CO"],
                           ["D2C-13-O;v=0;ortho", "ortho-D$_2 \\! ^{13}$CO"],
                           ["D2C-13-O;v=0;ortho#1", "D$_2 \\! ^{13}$CO"],
                           ["D2C-13-O;v=0;para", "para-D$_2 \\! ^{13}$CO"],
                           ["D2C-13-O;v=0;para#1", "D$_2 \\! ^{13}$CO"],
                           ["D2CCO;v=0;", "D$_2$CCO"],
                           ["D2CCO;v=0;ortho", "ortho-D$_2$CCO"],
                           ["D2CCO;v=0;para", "para-D$_2$CCO"],
                           ["D2CO-18;v=0;", "D$_2$C$^{18}$O"],
                           ["D2CO-18;v=0;ortho", "ortho-D$_2$C$^{18}$O"],
                           ["D2CO-18;v=0;para", "para-D$_2$C$^{18}$O"],
                           ["D2CO;v=0;", "D$_2$CO"],
                           ["D2CO;v=0;#1", "D$_2$CO"],
                           ["D2CO;v=0;#2", "D$_2$CO"],
                           ["D2CO;v=0;ortho", "ortho-D$_2$CO"],
                           ["D2CO;v=0;para", "para-D$_2$CO"],
                           ["D2CS;v=0;", "D$_2$CS"],
                           ["D2CS;v=0;ortho", "ortho-D$_2$CS"],
                           ["D2CS;v=0;para", "para-D$_2$CS"],
                           ["D2NCH2CN;v=0;", "D$_2$NCH$_2$CN"],
                           ["D2O;v=0;", "D$_2$O"],
                           ["D2O;v=0;#1", "D$_2$O"],
                           ["D2O;v=0;#2", "D$_2$O"],
                           ["D2O;v=0;ortho", "ortho-D$_2$O"],
                           ["D2O;v=0;para", "para-D$_2$O"],
                           ["D2S-34;v=0;", "D$_2 \\! ^{34}$S"],
                           ["D2S-34;v=0;ortho", "ortho-D$_2 \\! ^{34}$S"],
                           ["D2S-34;v=0;para", "para-D$_2 \\! ^{34}$S"],
                           ["D2S;v=0;", "D$_2$S"],
                           ["D2S;v=0;ortho", "ortho-D$_2$S"],
                           ["D2S;v=0;para", "para-D$_2$S"],
                           ["DC(O)NH2;v=0;", "DC(O)NH$_2$"],
                           ["DC(O)NH2;v=0;hyp1", "DC(O)NH$_2$"],
                           ["DC-13-CCN;v=0;", "D$^{13}$CCCN"],
                           ["DC-13-N-15;v=0;", "D$^{13}$C$^{15}$N"],
                           ["DC-13-N;v=0;", "D$^{13}$CN"],
                           ["DC-13-N;v=0;hyp1", "D$^{13}$CN"],
                           ["DC-13-O+;v=0;", "D$^{13}$CO$^+$"],
                           ["DC-13-O+;v=0;#1", "D$^{13}$CO$^+$"],
                           ["DC-13-O+;v=0;#2", "D$^{13}$CO$^+$"],
                           ["DC9N;v=0;", "DC$_9$N"],
                           ["DC9N;v=0;hyp1", "DC$_9$N"],
                           ["DCC-13-CN;v=0;", "DC$^{13}$CCN"],
                           ["DCCC-13-N;v=0;", "DCC$^{13}$CN"],
                           ["DCCCCCCCN;v=0;", "DCCCCCCCN"],
                           ["DCCCCCN;v=0;", "DCCCCCN"],
                           ["DCCCHO;v=0;", "DCCCHO"],
                           ["DCCCN-15;v=0;", "DCCC$^{15}$N"],
                           ["DCCCN;v7=1;", "DCCCN, v$_7$=1"],
                           ["DCCCN;v=0;", "DCCCN"],
                           ["DCCCN;v=0;#1", "DCCCN"],
                           ["DCCCN;v=0;#2", "DCCCN"],
                           ["DCCD;(v4,v5)=(0,1)-(0,0);", "DCCD, (v$_4$,v$_5$)=(0,1)-(0,0)"],
                           ["DCCD;(v4,v5)=(0,1)-(0,0);#1", "DCCD, (v$_4$,v$_5$)=(0,1)-(0,0)"],
                           ["DCCD;(v4,v5)=(0,1)-(0,0);ortho", "ortho-DCCD, (v$_4$,v$_5$)=(0,1)-(0,0)"],
                           ["DCCD;(v4,v5)=(0,1)-(0,0);para", "para-DCCD, (v$_4$,v$_5$)=(0,1)-(0,0)"],
                           ["DCCD;(v4,v5)=(0,2)-(0,1);", "DCCD, (v$_4$,v$_5$)=(0,2)-(0,1)"],
                           ["DCCD;(v4,v5)=(0,2)-(0,1);#1", "DCCD, (v$_4$,v$_5$)=(0,2)-(0,1)"],
                           ["DCCD;(v4,v5)=(0,2)-(0,1);ortho", "ortho-DCCD, (v$_4$,v$_5$)=(0,2)-(0,1)"],
                           ["DCCD;(v4,v5)=(0,2)-(0,1);para", "para-DCCD, (v$_4$,v$_5$)=(0,2)-(0,1)"],
                           ["DCCD;(v4,v5)=(0,3)-(0,2);", "DCCD, (v$_4$,v$_5$)=(0,3)-(0,2)"],
                           ["DCCD;(v4,v5)=(0,3)-(0,2);#1", "DCCD, (v$_4$,v$_5$)=(0,3)-(0,2)"],
                           ["DCCD;(v4,v5)=(0,3)-(0,2);ortho", "ortho-DCCD, (v$_4$,v$_5$)=(0,3)-(0,2)"],
                           ["DCCD;(v4,v5)=(0,3)-(0,2);para", "para-DCCD, (v$_4$,v$_5$)=(0,3)-(0,2)"],
                           ["DCCD;(v4,v5)=(1,0)-(0,1);", "DCCD, (v$_4$,v$_5$)=(1,0)-(0,1)"],
                           ["DCCD;(v4,v5)=(1,0)-(0,1);#1", "DCCD, (v$_4$,v$_5$)=(1,0)-(0,1)"],
                           ["DCCD;(v4,v5)=(1,0)-(0,1);ortho", "ortho-DCCD, (v$_4$,v$_5$)=(1,0)-(0,1)"],
                           ["DCCD;(v4,v5)=(1,0)-(0,1);para", "para-DCCD, (v$_4$,v$_5$)=(1,0)-(0,1)"],
                           ["DCCD;(v4,v5)=(1,1)-(0,2);", "DCCD, (v$_4$,v$_5$)=(1,1)-(0,2)"],
                           ["DCCD;(v4,v5)=(1,1)-(0,2);#1", "DCCD, (v$_4$,v$_5$)=(1,1)-(0,2)"],
                           ["DCCD;(v4,v5)=(1,1)-(0,2);ortho", "ortho-DCCD, (v$_4$,v$_5$)=(1,1)-(0,2)"],
                           ["DCCD;(v4,v5)=(1,1)-(0,2);para", "para-DCCD, (v$_4$,v$_5$)=(1,1)-(0,2)"],
                           ["DCCD;(v4,v5)=(1,1)-(1,0);", "DCCD, (v$_4$,v$_5$)=(1,1)-(1,0)"],
                           ["DCCD;(v4,v5)=(1,1)-(1,0);#1", "DCCD, (v$_4$,v$_5$)=(1,1)-(1,0)"],
                           ["DCCD;(v4,v5)=(1,1)-(1,0);ortho", "ortho-DCCD, (v$_4$,v$_5$)=(1,1)-(1,0)"],
                           ["DCCD;(v4,v5)=(1,1)-(1,0);para", "para-DCCD, (v$_4$,v$_5$)=(1,1)-(1,0)"],
                           ["DCCD;(v4,v5)=(2,0)-(0,1);", "DCCD, (v$_4$,v$_5$)=(2,0)-(0,1)"],
                           ["DCCD;(v4,v5)=(2,0)-(0,1);#1", "DCCD, (v$_4$,v$_5$)=(2,0)-(0,1)"],
                           ["DCCD;(v4,v5)=(2,0)-(0,1);ortho", "ortho-DCCD, (v$_4$,v$_5$)=(2,0)-(0,1)"],
                           ["DCCD;(v4,v5)=(2,0)-(0,1);para", "para-DCCD, (v$_4$,v$_5$)=(2,0)-(0,1)"],
                           ["DCCD;(v4,v5)=(2,0)-(0,3);", "DCCD, (v$_4$,v$_5$)=(2,0)-(0,3)"],
                           ["DCCD;(v4,v5)=(2,0)-(0,3);#1", "DCCD, (v$_4$,v$_5$)=(2,0)-(0,3)"],
                           ["DCCD;(v4,v5)=(2,0)-(0,3);ortho", "ortho-DCCD, (v$_4$,v$_5$)=(2,0)-(0,3)"],
                           ["DCCD;(v4,v5)=(2,0)-(0,3);para", "para-DCCD, (v$_4$,v$_5$)=(2,0)-(0,3)"],
                           ["DCCD;(v4,v5)=(2,0)-(1,1);", "DCCD, (v$_4$,v$_5$)=(2,0)-(1,1)"],
                           ["DCCD;(v4,v5)=(2,0)-(1,1);#1", "DCCD, (v$_4$,v$_5$)=(2,0)-(1,1)"],
                           ["DCCD;(v4,v5)=(2,0)-(1,1);ortho", "ortho-DCCD, (v$_4$,v$_5$)=(2,0)-(1,1)"],
                           ["DCCD;(v4,v5)=(2,0)-(1,1);para", "para-DCCD, (v$_4$,v$_5$)=(2,0)-(1,1)"],
                           ["DCCD;(v4,v5)=(2,1)-(0,2);", "DCCD, (v$_4$,v$_5$)=(2,1)-(0,2)"],
                           ["DCCD;(v4,v5)=(2,1)-(0,2);#1", "DCCD, (v$_4$,v$_5$)=(2,1)-(0,2)"],
                           ["DCCD;(v4,v5)=(2,1)-(0,2);ortho", "ortho-DCCD, (v$_4$,v$_5$)=(2,1)-(0,2)"],
                           ["DCCD;(v4,v5)=(2,1)-(0,2);para", "para-DCCD, (v$_4$,v$_5$)=(2,1)-(0,2)"],
                           ["DCCD;(v4,v5)=(2,1)-(2,0);", "DCCD, (v$_4$,v$_5$)=(2,1)-(2,0)"],
                           ["DCCD;(v4,v5)=(2,1)-(2,0);#1", "DCCD, (v$_4$,v$_5$)=(2,1)-(2,0)"],
                           ["DCCD;(v4,v5)=(2,1)-(2,0);ortho", "ortho-DCCD, (v$_4$,v$_5$)=(2,1)-(2,0)"],
                           ["DCCD;(v4,v5)=(2,1)-(2,0);para", "para-DCCD, (v$_4$,v$_5$)=(2,1)-(2,0)"],
                           ["DCCN;v=0;", "DCCN"],
                           ["DCCO;v=0;", "DCCO"],
                           ["DCN-15;v=0;", "DC$^{15}$N"],
                           ["DCN;v2=1;", "DCN, v$_2$=1"],
                           ["DCN;v2=1;#1", "DCN, v$_2$=1"],
                           ["DCN;v=0;", "DCN"],
                           ["DCN;v=0;hyp1", "DCN"],
                           ["DCNO;v=0;", "DCNO"],
                           ["DCO+;v1=1;", "DCO$^+$, v$_1$=1"],
                           ["DCO+;v1=1;#1", "DCO$^+$, v$_1$=1"],
                           ["DCO+;v2=1;", "DCO$^+$, v$_2$=1"],
                           ["DCO+;v2=1;#1", "DCO$^+$, v$_2$=1"],
                           ["DCO+;v3=1;", "DCO$^+$, v$_3$=1"],
                           ["DCO+;v3=1;#1", "DCO$^+$, v$_3$=1"],
                           ["DCO+;v=0;", "DCO$^+$"],
                           ["DCO+;v=0;#1", "DCO$^+$"],
                           ["DCO+;v=0;#2", "DCO$^+$"],
                           ["DCO-18+;v=0;", "DC$^{18}$O$^+$"],
                           ["DCOOH;v=0;", "DCOOH"],
                           ["DCOOH;v=0;#1", "DCOOH"],
                           ["DCP;v=0;", "DCP"],
                           ["DCP;v=0;#1", "DCP"],
                           ["DCP;v=0;#2", "DCP"],
                           ["DCS+;v=0;", "DCS$^+$"],
                           ["DCl-37;v=0;", "D$^{37}$Cl"],
                           ["DCl-37;v=0;#1", "D$^{37}$Cl"],
                           ["DCl;v=0;", "DCl"],
                           ["DCl;v=0;#1", "DCl"],
                           ["DF;v=0;", "DF"],
                           ["DF;v=0;#1", "DF"],
                           ["DN2+;v=0;", "DN$_2^+$"],
                           ["DNC;v=0;", "DNC"],
                           ["DNCCC;v=0;", "DNCCC"],
                           ["DNCCC;v=0;#1", "DNCCC"],
                           ["DNCO;v=0;", "DNCO"],
                           ["DNCO;v=0;#1", "DNCO"],
                           ["DNCS;v=0;", "DNCS"],
                           ["DNCS;v=0;#1", "DNCS"],
                           ["DNO3;v=0;", "DNO$_3$"],
                           ["DNO3;v=0;#1", "DNO$_3$"],
                           ["DNO;v=0;", "DNO"],
                           ["DNO;v=0;#1", "DNO"],
                           ["DOC+;v=0;", "DOC$^+$"],
                           ["DOC+;v=0;#1", "DOC$^+$"],
                           ["DOC+;v=0;#2", "DOC$^+$"],
                           ["DOCO+;v=0;", "DOCO$^+$"],
                           ["DS2;v=0;", "DS$_2$"],
                           ["DSCN;v=0;", "DSCN"],
                           ["DSCN;v=0;hyp2", "DSCN"],
                           ["DSCO+;v=0;", "DSCO$^+$"],
                           ["E-HC2CHCHCN;v=0;", "E-HC$_2$CHCHCN"],
                           ["E-HNCHCN;v=0;", "E-HNCHCN"],
                           ["Fe-54-O;v=0;", "$^{54}$FeO"],
                           ["Fe-54-O;v=0;#1", "$^{54}$FeO"],
                           ["Fe-54-O;v=0;#1", "$^{54}$FeO"],
                           ["Fe-56-C;v=0;", "$^{56}$FeC"],
                           ["Fe-56-N;v=0;", "$^{56}$FeN"],
                           ["Fe-56-O;v=0;", "$^{56}$FeO"],
                           ["Fe-56-O;v=0;", "$^{56}$FeO"],
                           ["Fe-56-O;v=0;", "$^{56}$FeO"],
                           ["FeCO;v=0;", "FeCO"],
                           ["G'Gg'gg'-CH2(OH)CH(OH)CH2OH;v=0;", "G'Gg'gg'-CH$_2$(OH)CH(OH)CH$_2$OH"],
                           ["GA-n-C4H9CN;v=0;", "GA-n-C$_4$H$_9$CN"],
                           ["GGag'g'-CH2(OH)CH(OH)CH2OH;v=0;", "GGag'g'-CH$_2$(OH)CH(OH)CH$_2$OH"],
                           ["Ga-n-C3H7OH;v=0;", "Ga-n-C$_3$H$_7$OH"],
                           ["H2C(CN)2;v=0;", "H$_2$C(CN)2"],
                           ["H2C-13-CHCN;v=0;", "H$_2 \\! ^{13}$CCHCN"],
                           ["H2C-13-CO;v=0;", "H$_2 \\! ^{13}$CCO"],
                           ["H2C-13-CO;v=0;ortho", "ortho-H$_2 \\! ^{13}$CCO"],
                           ["H2C-13-CO;v=0;para", "para-H$_2 \\! ^{13}$CCO"],
                           ["H2C-13-NH;v=0;", "H$_2 \\! ^{13}$CNH"],
                           ["H2C-13-NH;v=0;hyp1", "H$_2 \\! ^{13}$CNH"],
                           ["H2C-13-O;v=0;", "H$_2 \\! ^{13}$CO"],
                           ["H2C-13-O;v=0;#1", "H$_2 \\! ^{13}$CO"],
                           ["H2C-13-O;v=0;#2", "H$_2 \\! ^{13}$CO"],
                           ["H2C-13-O;v=0;ortho", "ortho-H$_2 \\! ^{13}$CO"],
                           ["H2C-13-O;v=0;para", "para-H$_2 \\! ^{13}$CO"],
                           ["H2C-13-OH+;v=0;", "H$_2 \\! ^{13}$COH$^+$"],
                           ["H2C-13-S-34;v=0;", "H$_2 \\! ^{13}$C$^{34}$S"],
                           ["H2C-13-S-34;v=0;ortho", "ortho-H$_2 \\! ^{13}$C$^{34}$S"],
                           ["H2C-13-S-34;v=0;para", "para-H$_2 \\! ^{13}$C$^{34}$S"],
                           ["H2C-13-S;v=0;", "H$_2 \\! ^{13}$CS"],
                           ["H2C-13-S;v=0;#1", "H$_2 \\! ^{13}$CS"],
                           ["H2C-13-S;v=0;#2", "H$_2 \\! ^{13}$CS"],
                           ["H2C-13-S;v=0;ortho", "ortho-H$_2 \\! ^{13}$CS"],
                           ["H2C-13-S;v=0;ortho#1", "H$_2 \\! ^{13}$CS"],
                           ["H2C-13-S;v=0;ortho#2", "H$_2 \\! ^{13}$CS"],
                           ["H2C-13-S;v=0;para", "para-H$_2 \\! ^{13}$CS"],
                           ["H2C-13-S;v=0;para#1", "H$_2 \\! ^{13}$CS"],
                           ["H2C-13-S;v=0;para#2", "H$_2 \\! ^{13}$CS"],
                           ["H2C2S;v=0;", "H$_2$C$_2$S"],
                           ["H2C2S;v=0;ortho", "ortho-H$_2$C$_2$S"],
                           ["H2C2S;v=0;para", "para-H$_2$C$_2$S"],
                           ["H2C2Si;v=0;", "H$_2$C$_2$Si"],
                           ["H2C2Si;v=0;ortho", "ortho-H$_2$C$_2$Si"],
                           ["H2C2Si;v=0;para", "para-H$_2$C$_2$Si"],
                           ["H2C3H;v=0;", "H$_2$C$_3$H"],
                           ["H2C3H;v=0;ortho", "ortho-H$_2$C$_3$H"],
                           ["H2C3H;v=0;para", "para-H$_2$C$_3$H"],
                           ["H2C3S;v=0;", "H$_2$C$_3$S"],
                           ["H2C3S;v=0;ortho", "ortho-H$_2$C$_3$S"],
                           ["H2C3S;v=0;para", "para-H$_2$C$_3$S"],
                           ["H2C4N;v=0;", "H$_2$C$_4$N"],
                           ["H2C4N;v=0;ortho", "ortho-H$_2$C$_4$N"],
                           ["H2C4N;v=0;para", "para-H$_2$C$_4$N"],
                           ["H2C4O;v=0;", "H$_2$C$_4$O"],
                           ["H2C4O;v=0;ortho", "ortho-H$_2$C$_4$O"],
                           ["H2C4O;v=0;para", "para-H$_2$C$_4$O"],
                           ["H2C4S;v=0;", "H$_2$C$_4$S"],
                           ["H2C4S;v=0;ortho", "ortho-H$_2$C$_4$S"],
                           ["H2C4S;v=0;para", "para-H$_2$C$_4$S"],
                           ["H2C5S;v=0;", "H$_2$C$_5$S"],
                           ["H2C5S;v=0;ortho", "ortho-H$_2$C$_5$S"],
                           ["H2C5S;v=0;para", "para-H$_2$C$_5$S"],
                           ["H2CC-13-HCN;v=0;", "H$_2$C$^{13}$CHCN"],
                           ["H2CC-13-O;v=0;", "H$_2$C$^{13}$CO"],
                           ["H2CC-13-O;v=0;ortho", "ortho-H$_2$C$^{13}$CO"],
                           ["H2CC-13-O;v=0;para", "para-H$_2$C$^{13}$CO"],
                           ["H2CCCHCN;v=0;", "H$_2$CCCHCN"],
                           ["H2CCHC-13-N;v=0;", "H$_2$CCH$^{13}$CN"],
                           ["H2CCHCCH;v=0;", "H$_2$CCHCCH"],
                           ["H2CCHCN-15;v=0;", "H$_2$CCHC$^{15}$N"],
                           ["H2CCN;v=0;", "H$_2$CCN"],
                           ["H2CCN;v=0;hyp2", "H$_2$CCN"],
                           ["H2CCN;v=0;ortho", "ortho-H$_2$CCN"],
                           ["H2CCN;v=0;para", "para-H$_2$CCN"],
                           ["H2CCNH;v=0;", "H$_2$CCNH"],
                           ["H2CCO-18;v=0;", "H$_2$CC$^{18}$O"],
                           ["H2CCO-18;v=0;ortho", "ortho-H$_2$CC$^{18}$O"],
                           ["H2CCO-18;v=0;para", "para-H$_2$CC$^{18}$O"],
                           ["H2CCO;v=0;", "H$_2$CCO"],
                           ["H2CCO;v=0;ortho", "ortho-H$_2$CCO"],
                           ["H2CCO;v=0;para", "para-H$_2$CCO"],
                           ["H2CN-15-H;v=0;", "H$_2$C$^{15}$NH"],
                           ["H2CN;v=0;", "H$_2$CN"],
                           ["H2CN;v=0;ortho", "ortho-H$_2$CN"],
                           ["H2CN;v=0;para", "para-H$_2$CN"],
                           ["H2CNCN;v=0;", "H$_2$CNCN"],
                           ["H2CNCN;v=0;hyp2", "H$_2$CNCN"],
                           ["H2CND;v=0;", "H$_2$CND"],
                           ["H2CND;v=0;hyp1", "H$_2$CND"],
                           ["H2CNH2+;v=0;", "H$_2$CNH$_2 \\! ^+$"],
                           ["H2CNH2+;v=0;ortho", "ortho-H$_2$CNH$_2 \\! ^+$"],
                           ["H2CNH2+;v=0;para", "para-H$_2$CNH$_2 \\! ^+$"],
                           ["H2CNH;v=0;", "H$_2$CNH"],
                           ["H2CO-17;v=0;", "H$_2$C$^{17}$O"],
                           ["H2CO-17;v=0;hyp1", "H$_2$C$^{17}$O"],
                           ["H2CO-17;v=0;ortho", "ortho-H$_2$C$^{17}$O"],
                           ["H2CO-17;v=0;para", "para-H$_2$C$^{17}$O"],
                           ["H2CO-18;v=0;", "H$_2$C$^{18}$O"],
                           ["H2CO-18;v=0;#1", "H$_2$C$^{18}$O"],
                           ["H2CO-18;v=0;#2", "H$_2$C$^{18}$O"],
                           ["H2CO-18;v=0;hyp2", "H$_2$C$^{18}$O"],
                           ["H2CO-18;v=0;ortho", "ortho-H$_2$C$^{18}$O"],
                           ["H2CO-18;v=0;para", "para-H$_2$C$^{18}$O"],
                           ["H2CO;v=0;", "H$_2$CO"],
                           ["H2CO;v=0;#1", "H$_2$CO"],
                           ["H2CO;v=0;#2", "H$_2$CO"],
                           ["H2CO;v=0;hyp2", "H$_2$CO"],
                           ["H2CO;v=0;ortho", "ortho-H$_2$CO"],
                           ["H2CO;v=0;para", "para-H$_2$CO"],
                           ["H2COH+;v=0;", "H$_2$COH$^+$"],
                           ["H2CP;v=0;", "H$_2$CP"],
                           ["H2CP;v=0;ortho", "ortho-H$_2$CP"],
                           ["H2CP;v=0;para", "para-H$_2$CP"],
                           ["H2CS-33;v=0;", "H$_2$C$^{33}$S"],
                           ["H2CS-33;v=0;hyp1", "H$_2$C$^{33}$S"],
                           ["H2CS-33;v=0;hyp1;ortho", "ortho-H$_2$C$^{33}$S"],
                           ["H2CS-33;v=0;hyp1;para", "para-H$_2$C$^{33}$S"],
                           ["H2CS-34;v=0;", "H$_2$C$^{34}$S"],
                           ["H2CS-34;v=0;#1", "H$_2$C$^{34}$S"],
                           ["H2CS-34;v=0;#2", "H$_2$C$^{34}$S"],
                           ["H2CS-34;v=0;ortho", "ortho-H$_2$C$^{34}$S"],
                           ["H2CS-34;v=0;ortho#1", "H$_2$C$^{34}$S"],
                           ["H2CS-34;v=0;ortho#2", "H$_2$C$^{34}$S"],
                           ["H2CS-34;v=0;para", "para-H$_2$C$^{34}$S"],
                           ["H2CS-34;v=0;para#1", "H$_2$C$^{34}$S"],
                           ["H2CS-34;v=0;para#2", "H$_2$C$^{34}$S"],
                           ["H2CS-36;v=0;", "H$_2$C$^{36}$S"],
                           ["H2CS-36;v=0;ortho", "ortho-H$_2$C$^{36}$S"],
                           ["H2CS-36;v=0;para", "para-H$_2$C$^{36}$S"],
                           ["H2CS;v=0;", "H$_2$CS"],
                           ["H2CS;v=0;#1", "H$_2$CS"],
                           ["H2CS;v=0;#2", "H$_2$CS"],
                           ["H2CS;v=0;hyp2", "H$_2$CS"],
                           ["H2CS;v=0;ortho", "ortho-H$_2$CS"],
                           ["H2CS;v=0;ortho#1", "H$_2$CS"],
                           ["H2CS;v=0;ortho#2", "H$_2$CS"],
                           ["H2CS;v=0;para", "para-H$_2$CS"],
                           ["H2CS;v=0;para#1", "H$_2$CS"],
                           ["H2CS;v=0;para#2", "H$_2$CS"],
                           ["H2CSi;v=0;", "H$_2$CSi"],
                           ["H2CSi;v=0;ortho", "ortho-H$_2$CSi"],
                           ["H2CSi;v=0;para", "para-H$_2$CSi"],
                           ["H2CSiH2;v=0;", "H$_2$CSiH$_2$"],
                           ["H2CSiH2;v=0;ortho", "ortho-H$_2$CSiH$_2$"],
                           ["H2CSiH2;v=0;para", "para-H$_2$CSiH$_2$"],
                           ["H2Cl+;v=0;", "H$_2$Cl$^+$"],
                           ["H2Cl+;v=0;#1", "H$_2$Cl$^+$"],
                           ["H2Cl+;v=0;ortho", "ortho-H$_2$Cl$^+$"],
                           ["H2Cl+;v=0;ortho#1", "H$_2$Cl$^+$"],
                           ["H2Cl+;v=0;para", "para-H$_2$Cl$^+$"],
                           ["H2Cl+;v=0;para#1", "H$_2$Cl$^+$"],
                           ["H2D+;v=0;", "H$_2$D$^+$"],
                           ["H2D+;v=0;ortho", "ortho-H$_2$D$^+$"],
                           ["H2D+;v=0;para", "para-H$_2$D$^+$"],
                           ["H2DO+;v=0;", "H$_2$DO$^+$"],
                           ["H2DO+;v=0;ortho", "ortho-H$_2$DO$^+$"],
                           ["H2DO+;v=0;para", "para-H$_2$DO$^+$"],
                           ["H2F+;v=0;", "H$_2$F$^+$"],
                           ["H2F+;v=0;ortho", "ortho-H$_2$F$^+$"],
                           ["H2F+;v=0;para", "para-H$_2$F$^+$"],
                           ["H2N-15-CN;v=0;", "H$_2 \\! ^{15}$N-CN"],
                           ["H2N-15-CN;v=0;ortho", "ortho-H$_2 \\! ^{15}$N-CN"],
                           ["H2N-15-CN;v=0;para", "para-H$_2 \\! ^{15}$N-CN"],
                           ["H2NC(O)NH2;v=0;", "H$_2$NC(O)NH$_2$"],
                           ["H2NC-13-H2CN;v=0;", "H$_2$N$^{13}$CH$_2$CN"],
                           ["H2NC-13-N;v=0;", "H$_2$N$^{13}$CN"],
                           ["H2NC-13-N;v=0;ortho", "ortho-H$_2$N$^{13}$CN"],
                           ["H2NC-13-N;v=0;para", "para-H$_2$N$^{13}$CN"],
                           ["H2NCH2C-13-N;v=0;", "H$_2$NCH$_2 \\! ^{13}$CN"],
                           ["H2NCH2CN;v=0;", "H$_2$NCH$_2$CN"],
                           ["H2NCH2CN;v=0;#1", "H$_2$NCH$_2$CN"],
                           ["H2NCH2CN;v=0;#1", "H$_2$NCH$_2$CN"],
                           ["H2NCH2CN;v=0;hyp2", "H$_2$NCH$_2$CN"],
                           ["H2NCH2COOH,II;v=0;", "H$_2$NCH$_2$COOH,II"],
                           ["H2NCH2COOH,II;v=0;hyp1", "H$_2$NCH$_2$COOH,II"],
                           ["H2NCH2COOH;v=0;", "H$_2$NCH$_2$COOH"],
                           ["H2NCN-15;v=0;", "H$_2$NC$^{15}$N"],
                           ["H2NCN-15;v=0;ortho", "ortho-H$_2$NC$^{15}$N"],
                           ["H2NCN-15;v=0;para", "para-H$_2$NC$^{15}$N"],
                           ["H2NCO+;v=0;", "H$_2$NCO$^+$"],
                           ["H2NCO+;v=0;hyp1", "H$_2$NCO$^+$"],
                           ["H2NCO+;v=0;hyp1;ortho", "ortho-H$_2$NCO$^+$"],
                           ["H2NCO+;v=0;hyp1;para", "para-H$_2$NCO$^+$"],
                           ["H2NCO+;v=0;ortho", "ortho-H$_2$NCO$^+$"],
                           ["H2NCO+;v=0;para", "para-H$_2$NCO$^+$"],
                           ["H2NCO2CH3;v24=1-0;", "H$_2$NCO$_2$CH$_3$, v$_{24}$=1-0"],
                           ["H2NCO2CH3;v24=1-0;#1", "H$_2$NCO$_2$CH$_3$, v$_{24}$=1-0"],
                           ["H2NCO2CH3;v24=1;", "H$_2$NCO$_2$CH$_3$, v$_{24}$=1"],
                           ["H2NCO2CH3;v24=1;#1", "H$_2$NCO$_2$CH$_3$, v$_{24}$=1"],
                           ["H2NCO2CH3;v=0;", "H$_2$NCO$_2$CH$_3$"],
                           ["H2NCO2CH3;v=0;#1", "H$_2$NCO$_2$CH$_3$"],
                           ["H2NNC;v=0;", "H$_2$NNC"],
                           ["H2NNC;v=0;ortho", "ortho-H$_2$NNC"],
                           ["H2NNC;v=0;para", "para-H$_2$NNC"],
                           ["H2O+;v=0;hyp2", "H$_2$O$^+$"],
                           ["H2O+;v=0;hyp2;", "H$_2$O$^+$"],
                           ["H2O+;v=0;hyp2;ortho", "ortho-H$_2$O$^+$"],
                           ["H2O+;v=0;hyp2;para", "para-H$_2$O$^+$"],
                           ["H2O-17;v=0;", "H$_2 \\! ^{17}$O"],
                           ["H2O-17;v=0;#1", "H$_2 \\! ^{17}$O"],
                           ["H2O-17;v=0;ortho", "ortho-H$_2 \\! ^{17}$O"],
                           ["H2O-17;v=0;ortho#1", "H$_2 \\! ^{17}$O"],
                           ["H2O-17;v=0;para", "para-H$_2 \\! ^{17}$O"],
                           ["H2O-17;v=0;para#1", "H$_2 \\! ^{17}$O"],
                           ["H2O-18;v2=1;", "H$_2 \\! ^{18}$O, v$_2$=1"],
                           ["H2O-18;v2=1;#1", "H$_2 \\! ^{18}$O, v$_2$=1"],
                           ["H2O-18;v2=1;ortho", "ortho-H$_2 \\! ^{18}$O, v$_2$=1"],
                           ["H2O-18;v2=1;ortho#1", "H$_2 \\! ^{18}$O, v$_2$=1"],
                           ["H2O-18;v2=1;para", "para-H$_2 \\! ^{18}$O, v$_2$=1"],
                           ["H2O-18;v2=1;para#1", "H$_2 \\! ^{18}$O, v$_2$=1"],
                           ["H2O-18;v=0;", "H$_2 \\! ^{18}$O"],
                           ["H2O-18;v=0;#1", "H$_2 \\! ^{18}$O"],
                           ["H2O-18;v=0;ortho", "ortho-H$_2 \\! ^{18}$O"],
                           ["H2O-18;v=0;ortho#1", "H$_2 \\! ^{18}$O"],
                           ["H2O-18;v=0;para", "para-H$_2 \\! ^{18}$O"],
                           ["H2O-18;v=0;para#1", "H$_2 \\! ^{18}$O"],
                           ["H2O2;v4=1-0;", "H$_2$O$_2$, v$_4$=1-0"],
                           ["H2O2;v4=1-0;#1", "H$_2$O$_2$, v$_4$=1-0"],
                           ["H2O2;v4=1-0;ortho", "ortho-H$_2$O$_2$, v$_4$=1-0"],
                           ["H2O2;v4=1-0;para", "para-H$_2$O$_2$, v$_4$=1-0"],
                           ["H2O2;v4=1;", "H$_2$O$_2$, v$_4$=1"],
                           ["H2O2;v4=1;#1", "H$_2$O$_2$, v$_4$=1"],
                           ["H2O2;v4=1;ortho", "ortho-H$_2$O$_2$, v$_4$=1"],
                           ["H2O2;v4=1;para", "para-H$_2$O$_2$, v$_4$=1"],
                           ["H2O2;v4=2-0;", "H$_2$O$_2$, v$_4$=2-0"],
                           ["H2O2;v4=2-0;#1", "H$_2$O$_2$, v$_4$=2-0"],
                           ["H2O2;v4=2-0;ortho", "ortho-H$_2$O$_2$, v$_4$=2-0"],
                           ["H2O2;v4=2-0;para", "para-H$_2$O$_2$, v$_4$=2-0"],
                           ["H2O2;v4=2-1;", "H$_2$O$_2$, v$_4$=2-1"],
                           ["H2O2;v4=2-1;#1", "H$_2$O$_2$, v$_4$=2-1"],
                           ["H2O2;v4=2-1;ortho", "ortho-H$_2$O$_2$, v$_4$=2-1"],
                           ["H2O2;v4=2-1;para", "para-H$_2$O$_2$, v$_4$=2-1"],
                           ["H2O2;v=0;", "H$_2$O$_2$"],
                           ["H2O2;v=0;#1", "H$_2$O$_2$"],
                           ["H2O2;v=0;ortho", "ortho-H$_2$O$_2$"],
                           ["H2O2;v=0;para", "para-H$_2$O$_2$"],
                           ["H2O;(v1,v2,v3)=(0,2,0)-(0,0,1);", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(0,2,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(0,2,0)-(0,0,1);#1", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(0,2,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(0,2,0)-(0,0,1);ortho", "ortho-H$_2$O, (v$_1$,v$_2$,v$_3$)=(0,2,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(0,2,0)-(0,0,1);ortho#1", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(0,2,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(0,2,0)-(0,0,1);para", "para-H$_2$O, (v$_1$,v$_2$,v$_3$)=(0,2,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(0,2,0)-(0,0,1);para#1", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(0,2,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,0,1);", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,0,1);#1", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,0,1);ortho", "ortho-H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,0,1);ortho#1", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,0,1);para", "para-H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,0,1);para#1", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,0,1)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,2,0);", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,2,0)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,2,0);#1", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,2,0)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,2,0);ortho", "ortho-H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,2,0)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,2,0);ortho#1", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,2,0)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,2,0);para", "para-H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,2,0)"],
                           ["H2O;(v1,v2,v3)=(1,0,0)-(0,2,0);para#1", "H$_2$O, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,2,0)"],
                           ["H2O;v1=1;", "H$_2$O, v$_1$=1"],
                           ["H2O;v1=1;#1", "H$_2$O, v$_1$=1"],
                           ["H2O;v1=1;ortho", "ortho-H$_2$O, v$_1$=1"],
                           ["H2O;v1=1;ortho#1", "H$_2$O, v$_1$=1"],
                           ["H2O;v1=1;para", "para-H$_2$O, v$_1$=1"],
                           ["H2O;v1=1;para#1", "H$_2$O, v$_1$=1"],
                           ["H2O;v2=1;", "H$_2$O, v$_2$=1"],
                           ["H2O;v2=1;#1", "H$_2$O, v$_2$=1"],
                           ["H2O;v2=1;ortho", "ortho-H$_2$O, v$_2$=1"],
                           ["H2O;v2=1;ortho#1", "H$_2$O, v$_2$=1"],
                           ["H2O;v2=1;para", "para-H$_2$O, v$_2$=1"],
                           ["H2O;v2=1;para#1", "H$_2$O, v$_2$=1"],
                           ["H2O;v2=2;", "H$_2$O, v$_2$=2"],
                           ["H2O;v2=2;#1", "H$_2$O, v$_2$=2"],
                           ["H2O;v2=2;ortho", "ortho-H$_2$O, v$_2$=2"],
                           ["H2O;v2=2;ortho#1", "H$_2$O, v$_2$=2"],
                           ["H2O;v2=2;para", "para-H$_2$O, v$_2$=2"],
                           ["H2O;v2=2;para#1", "H$_2$O, v$_2$=2"],
                           ["H2O;v3=1;", "H$_2$O, v$_3$=1"],
                           ["H2O;v3=1;#1", "H$_2$O, v$_3$=1"],
                           ["H2O;v3=1;ortho", "ortho-H$_2$O, v$_3$=1"],
                           ["H2O;v3=1;ortho#1", "H$_2$O, v$_3$=1"],
                           ["H2O;v3=1;para", "para-H$_2$O, v$_3$=1"],
                           ["H2O;v3=1;para#1", "H$_2$O, v$_3$=1"],
                           ["H2O;v=0;", "H$_2$O"],
                           ["H2O;v=0;#1", "H$_2$O"],
                           ["H2O;v=0;ortho", "ortho-H$_2$O"],
                           ["H2O;v=0;ortho#1", "H$_2$O"],
                           ["H2O;v=0;para", "para-H$_2$O"],
                           ["H2O;v=0;para#1", "H$_2$O"],
                           ["H2PCN;v=0;", "H$_2$PCN"],
                           ["H2S-33;v=0;", "H$_2 \\! ^{33}$S"],
                           ["H2S-33;v=0;ortho", "ortho-H$_2 \\! ^{33}$S"],
                           ["H2S-33;v=0;para", "para-H$_2 \\! ^{33}$S"],
                           ["H2S-34;v=0;", "H$_2 \\! ^{34}$S"],
                           ["H2S-34;v=0;ortho", "ortho-H$_2 \\! ^{34}$S"],
                           ["H2S-34;v=0;para", "para-H$_2 \\! ^{34}$S"],
                           ["H2S2;v=0;", "H$_2$S2"],
                           ["H2S2;v=0;ortho", "ortho-H$_2$S2"],
                           ["H2S2;v=0;para", "para-H$_2$S2"],
                           ["H2S;v=0;", "H$_2$S"],
                           ["H2S;v=0;ortho", "ortho-H$_2$S"],
                           ["H2S;v=0;para", "para-H$_2$S"],
                           ["H2SO4;v=0;", "H$_2$SO$_4$"],
                           ["H2SO4;v=0;#1", "H$_2$SO$_4$"],
                           ["H2SO4;v=0;ortho", "ortho-H$_2$SO$_4$"],
                           ["H2SO4;v=0;para", "para-H$_2$SO$_4$"],
                           ["H2SiO;v=0;", "H$_2$SiO"],
                           ["H2SiO;v=0;ortho", "ortho-H$_2$SiO"],
                           ["H2SiO;v=0;para", "para-H$_2$SiO"],
                           ["H2SiS;v=0;", "H$_2$SiS"],
                           ["H2SiS;v=0;ortho", "ortho-H$_2$SiS"],
                           ["H2SiS;v=0;para", "para-H$_2$SiS"],
                           ["H3CCHCH2O;v=0;", "H$_3$CCHCH$_2$O"],
                           ["H3O+;(v1,v2,v3,v4)=(0,0,0,1)-(0,0,0,0);", "H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,0,0,1)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(0,0,0,1)-(0,0,0,0);#1", "H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,0,0,1)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(0,0,0,1)-(0,0,0,0);ortho", "ortho-H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,0,0,1)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(0,0,0,1)-(0,0,0,0);para", "para-H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,0,0,1)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(0,0,1,0)-(0,0,0,0);", "H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,0,1,0)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(0,0,1,0)-(0,0,0,0);#1", "H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,0,1,0)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(0,0,1,0)-(0,0,0,0);ortho", "ortho-H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,0,1,0)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(0,0,1,0)-(0,0,0,0);para", "para-H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,0,1,0)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(0,1,0,0)-(0,0,0,0);", "H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,1,0,0)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(0,1,0,0)-(0,0,0,0);#1", "H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,1,0,0)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(0,1,0,0)-(0,0,0,0);ortho", "ortho-H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,1,0,0)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(0,1,0,0)-(0,0,0,0);para", "para-H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(0,1,0,0)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(1,0,0,0)-(0,0,0,0);", "H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(1,0,0,0)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(1,0,0,0)-(0,0,0,0);#1", "H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(1,0,0,0)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(1,0,0,0)-(0,0,0,0);ortho", "ortho-H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(1,0,0,0)-(0,0,0,0)"],
                           ["H3O+;(v1,v2,v3,v4)=(1,0,0,0)-(0,0,0,0);para", "para-H$_3$O$^+$, (v$_1$,v$_2$,v$_3$,v$_4$)=(1,0,0,0)-(0,0,0,0)"],
                           ["H3O+;v2=1;", "H$_3$O$^+$, v$_2$=1"],
                           ["H3O+;v2=1;#1", "H$_3$O$^+$, v$_2$=1"],
                           ["H3O+;v2=1;ortho", "ortho-H$_3$O$^+$, v$_2$=1"],
                           ["H3O+;v2=1;ortho#1", "H$_3$O$^+$, v$_2$=1"],
                           ["H3O+;v2=1;para", "para-H$_3$O$^+$, v$_2$=1"],
                           ["H3O+;v2=1;para#1", "H$_3$O$^+$, v$_2$=1"],
                           ["H3O+;v=0;", "H$_3$O$^+$"],
                           ["H3O+;v=0;#1", "H$_3$O$^+$"],
                           ["H3O+;v=0;#2", "H$_3$O$^+$"],
                           ["H3O+;v=0;ortho", "ortho-H$_3$O$^+$"],
                           ["H3O+;v=0;ortho#1", "H$_3$O$^+$"],
                           ["H3O+;v=0;ortho#2", "H$_3$O$^+$"],
                           ["H3O+;v=0;para", "para-H$_3$O$^+$"],
                           ["H3O+;v=0;para#1", "H$_3$O$^+$"],
                           ["H3O+;v=0;para#2", "H$_3$O$^+$"],
                           ["HB-10-O;v=0;", "H$^{10}$BO"],
                           ["HBO;v=0;", "HBO"],
                           ["HBr-79;v=0;", "H$^{79}$Br"],
                           ["HBr-79;v=0;#1", "H$^{79}$Br"],
                           ["HBr-81;v=0;", "H$^{81}$Br"],
                           ["HBr-81;v=0;#1", "H$^{81}$Br"],
                           ["HC(O)CN;v=0;", "HC(O)CN"],
                           ["HC(O)N-15-H2;v=0;", "HC(O)$^{15}$NH$_2$"],
                           ["HC(O)NH2;v12=1;", "HC(O)NH$_2$, v$_{12}$=1"],
                           ["HC(O)NH2;v=0;", "HC(O)NH$_2$"],
                           ["HC(O)NH2;v=0;hyp1", "HC(O)NH$_2$"],
                           ["HC(O-18)NH2;v=0;", "HC($^{18}$O)NH$_2$"],
                           ["HC-13-(O)NH2;v=0;", "H$^{13}$C(O)NH$_2$"],
                           ["HC-13-C-13-CN;v=0;", "H$^{13}$C$^{13}$CCN"],
                           ["HC-13-C8N;v=0;", "H$^{13}$CC$_8$N"],
                           ["HC-13-C8N;v=0;hyp1", "H$^{13}$CC$_8$N"],
                           ["HC-13-CC-13-N;v=0;", "H$^{13}$CC$^{13}$CN"],
                           ["HC-13-CCCCCCN;v=0;", "H$^{13}$CCCCCCCN"],
                           ["HC-13-CCCCN;v=0;", "H$^{13}$CCCCCN"],
                           ["HC-13-CCHO;v=0;", "H$^{13}$CCCHO"],
                           ["HC-13-CCN;v5=1;", "H$^{13}$CCCN, v$_5$=1"],
                           ["HC-13-CCN;v6=1,v7=1;", "H$^{13}$CCCN, v$_6$=1,v$_7$=1"],
                           ["HC-13-CCN;v6=1;", "H$^{13}$CCCN, v$_6$=1"],
                           ["HC-13-CCN;v7=1;", "H$^{13}$CCCN, v$_7$=1"],
                           ["HC-13-CCN;v7=2;", "H$^{13}$CCCN, v$_7$=2"],
                           ["HC-13-CCN;v7=3;", "H$^{13}$CCCN, v$_7$=3"],
                           ["HC-13-CCN;v=0;", "H$^{13}$CCCN"],
                           ["HC-13-CCN;v=0;#1", "H$^{13}$CCCN"],
                           ["HC-13-CCN;v=0;#2", "H$^{13}$CCCN"],
                           ["HC-13-CN;v=0;", "H$^{13}$CCN"],
                           ["HC-13-N-15;v2=1;", "H$^{13}$C$^{15}$N, v$_2$=1"],
                           ["HC-13-N-15;v=0;", "H$^{13}$C$^{15}$N"],
                           ["HC-13-N;v=0;", "H$^{13}$CN"],
                           ["HC-13-N;v=0;hyp1", "H$^{13}$CN"],
                           ["HC-13-N;v=0;hyp3", "H$^{13}$CN hyp3"],
                           ["HC-13-NH+;v=0;", "H$^{13}$CNH$^+$"],
                           ["HC-13-NO;v=0;", "H$^{13}$CNO"],
                           ["HC-13-O+;v=0;", "H$^{13}$CO$^+$"],
                           ["HC-13-O+;v=0;#1", "H$^{13}$CO$^+$"],
                           ["HC-13-O+;v=0;#2", "H$^{13}$CO$^+$"],
                           ["HC-13-OOH;v=0;", "H$^{13}$COOH"],
                           ["HC-13-OOH;v=0;#1", "H$^{13}$COOH"],
                           ["HC-13-P;v=0;", "H$^{13}$CP"],
                           ["HC-13-S+;v=0;", "H$^{13}$CS$^+$"],
                           ["HC11N;v=0;", "HC$_{11}$N"],
                           ["HC11N;v=0;#1", "HC$_{11}$N"],
                           ["HC11N;v=0;#2", "HC$_{11}$N"],
                           ["HC13N;v=0;", "HC$_{13}$N"],
                           ["HC15N-15-O;v=0;", "HC$_{15}^{15}$N-O"],
                           ["HC2C-13-C6N;v=0;", "HC$_2 \\! ^{13}$CC$_6$N"],
                           ["HC2C-13-C6N;v=0;hyp1", "HC$_2 \\! ^{13}$CC$_6$N"],
                           ["HC2CHS;v=0;", "HC$_2$CHS"],
                           ["HC3C-13-C5N;v=0;", "HC$_3 \\! ^{13}$CC$_5$N"],
                           ["HC3C-13-C5N;v=0;hyp1", "HC$_3 \\! ^{13}$CC$_5$N"],
                           ["HC3N;(v5,v7)=(1,1)-(0,4);", "HC$_3$N, (v$_5$,v$_7$)=(1,1)-(0,4)"],
                           ["HC3N;v1=1-0;", "HC$_3$N, v$_1$=1-0"],
                           ["HC3N;v2=1-0;", "HC$_3$N, v$_2$=1-0"],
                           ["HC3N;v3=1-0;", "HC$_3$N, v$_3$=1-0"],
                           ["HC3N;v5=1,v7=1;", "HC$_3$N, v$_5$=1,v$_7$=1"],
                           ["HC3N;v6=2;", "HC$_3$N, v$_6$=2"],
                           ["HC3N;v7=4;", "HC$_3$N, v$_7$=4"],
                           ["HC3NH+;v=0;", "HC$_3$NH$^+$"],
                           ["HC3P;v=0;", "HC$_3$P"],
                           ["HC3S;v=0;", "HC$_3$S"],
                           ["HC3S;v=0;hyp1", "HC$_3$S"],
                           ["HC4C-13-C4N;v=0;", "HC$_4 \\! ^{13}$CC$_4$N"],
                           ["HC4C-13-C4N;v=0;hyp1", "HC$_4 \\! ^{13}$CC$_4$N"],
                           ["HC4CHO;v=0;", "HC$_4$CHO"],
                           ["HC4CHS;v=0;", "HC$_4$CHS"],
                           ["HC4NC;v=0;", "HC$_4$NC"],
                           ["HC4O;v=0;hyp1", "HC$_4$O"],
                           ["HC4S;v=0;hyp1", "HC$_4$S"],
                           ["HC5C-13-C3N;v=0;", "HC$_5 \\! ^{13}$CC$_3$N"],
                           ["HC5C-13-C3N;v=0;hyp1", "HC$_5 \\! ^{13}$CC$_3$N"],
                           ["HC5N;v10=1,v11=1;", "HC$_5$N, v$_{10}$=1,v$_{11}$=1"],
                           ["HC5N;v11=5;", "HC$_5$N, v$_{11}$=5"],
                           ["HC5O;v=0;hyp1", "HC$_5$O"],
                           ["HC5S;v=0;hyp1", "HC$_5$S"],
                           ["HC6CHO;v=0;", "HC$_6$CHO"],
                           ["HC6O;v=0;hyp1", "HC$_6$O"],
                           ["HC7C-13-CN;v=0;", "HC$_7 \\! ^{13}$CCN"],
                           ["HC7C-13-CN;v=0;hyp1", "HC$_7 \\! ^{13}$CCN"],
                           ["HC7O;v=0;hyp1", "HC$_7$O"],
                           ["HC7S;v=0;hyp1", "HC$_7$S"],
                           ["HC8C-13-N;v=0;", "HC$_8 \\! ^{13}$CN"],
                           ["HC8C-13-N;v=0;hyp1", "HC$_8 \\! ^{13}$CN"],
                           ["HC9N-15;v=0;", "HC$_9 \\! ^{15}$N"],
                           ["HC9N;v=0;", "HC$_9$N"],
                           ["HC9N;v=0;hyp1", "HC$_9$N"],
                           ["HCC-13-C-13-N;v=0;", "HC$^{13}$C$^{13}$CN"],
                           ["HCC-13-C7N;v=0;", "HC$^{13}$CC$_7$N"],
                           ["HCC-13-C7N;v=0;hyp1", "HC$^{13}$CC$_7$N"],
                           ["HCC-13-CCCCCN;v=0;", "HC$^{13}$CCCCCCN"],
                           ["HCC-13-CCCN;v=0;", "HC$^{13}$CCCCN"],
                           ["HCC-13-CHO;v=0;", "HC$^{13}$CCHO"],
                           ["HCC-13-CN-15;v=0;", "HC$^{13}$CC$^{15}$N"],
                           ["HCC-13-CN;(v5,v7)=(1,0)-(0,3);", "HC$^{13}$CCN, (v$_5$,v$_7$)=(1,0)-(0,3)"],
                           ["HCC-13-CN;v5=1;", "HC$^{13}$CCN, v$_5$=1"],
                           ["HCC-13-CN;v6=1,v7=1;", "HC$^{13}$CCN, v$_6$=1,v$_7$=1"],
                           ["HCC-13-CN;v6=1;", "HC$^{13}$CCN, v$_6$=1"],
                           ["HCC-13-CN;v7=1;", "HC$^{13}$CCN, v$_7$=1"],
                           ["HCC-13-CN;v7=2;", "HC$^{13}$CCN, v$_7$=2"],
                           ["HCC-13-CN;v7=3;", "HC$^{13}$CCN, v$_7$=3"],
                           ["HCC-13-CN;v=0;", "HC$^{13}$CCN"],
                           ["HCC-13-CN;v=0;#1", "HC$^{13}$CCN"],
                           ["HCC-13-CN;v=0;#2", "HC$^{13}$CCN"],
                           ["HCC-13-N;v=0;", "HC$^{13}$CN"],
                           ["HCCC-13-CCCCN;v=0;", "HCC$^{13}$CCCCCN"],
                           ["HCCC-13-CCN;v=0;", "HCC$^{13}$CCCN"],
                           ["HCCC-13-HO;v=0;", "HCC$^{13}$CHO"],
                           ["HCCC-13-N;(v5,v7)=(1,0)-(0,3);", "HCC$^{13}$CN, (v$_5$,v$_7$)=(1,0)-(0,3)"],
                           ["HCCC-13-N;v4=1,v7=1;", "HCC$^{13}$CN, v$_4$=1,v$_7$=1"],
                           ["HCCC-13-N;v4=1;", "HCC$^{13}$CN, v$_4$=1"],
                           ["HCCC-13-N;v5=1;", "HCC$^{13}$CN, v$_5$=1"],
                           ["HCCC-13-N;v6=1,v7=1;", "HCC$^{13}$CN, v$_6$=1,v$_7$=1"],
                           ["HCCC-13-N;v6=1;", "HCC$^{13}$CN, v$_6$=1"],
                           ["HCCC-13-N;v7=1;", "HCC$^{13}$CN, v$_7$=1"],
                           ["HCCC-13-N;v7=2;", "HCC$^{13}$CN, v$_7$=2"],
                           ["HCCC-13-N;v7=3;", "HCC$^{13}$CN, v$_7$=3"],
                           ["HCCC-13-N;v=0;", "HCC$^{13}$CN"],
                           ["HCCC-13-N;v=0;#1", "HCC$^{13}$CN"],
                           ["HCCC-13-N;v=0;#2", "HCC$^{13}$CN"],
                           ["HCCCC-13-CCCN;v=0;", "HCCC$^{13}$CCCCN"],
                           ["HCCCC-13-CN;v=0;", "HCCC$^{13}$CCN"],
                           ["HCCCCC-13-CCN;v=0;", "HCCCC$^{13}$CCCN"],
                           ["HCCCCC-13-N;v=0;", "HCCCC$^{13}$CN"],
                           ["HCCCCCC-13-CN;v=0;", "HCCCCC$^{13}$CCN"],
                           ["HCCCCCC-13-CN;v=0;hyp1", "HCCCCC$^{13}$CCN"],
                           ["HCCCCCCC-13-N;v=0;", "HCCCCCC$^{13}$CN"],
                           ["HCCCCCCC-13-N;v=0;hyp1", "HCCCCCC$^{13}$CN"],
                           ["HCCCCCCCN-15;v=0;", "HCCCCCCC$^{15}$N"],
                           ["HCCCCCCCN;v15=1;", "HCCCCCCCN, v$_{15}$=1"],
                           ["HCCCCCCCN;v15=2;", "HCCCCCCCN, v$_{15}$=2"],
                           ["HCCCCCCCN;v=0;", "HCCCCCCCN"],
                           ["HCCCCCCCN;v=0;#1", "HCCCCCCCN"],
                           ["HCCCCCCCN;v=0;#2", "HCCCCCCCN"],
                           ["HCCCCCCCN;v=0;hyp1", "HCCCCCCCN"],
                           ["HCCCCCN-15;v=0;", "HCCCCC$^{15}$N"],
                           ["HCCCCCN;v10=1;", "HCCCCCN, v$_{10}$=1"],
                           ["HCCCCCN;v11=1;", "HCCCCCN, v$_{11}$=1"],
                           ["HCCCCCN;v11=2;", "HCCCCCN, v$_{11}$=2"],
                           ["HCCCCCN;v11=3;", "HCCCCCN, v$_{11}$=3"],
                           ["HCCCCCN;v11=4;", "HCCCCCN, v$_{11}$=4"],
                           ["HCCCCCN;v7=1-0;", "HCCCCCN, v$_7$=1-0"],
                           ["HCCCCCN;v9=1;", "HCCCCCN, v$_9$=1"],
                           ["HCCCCCN;v=0;", "HCCCCCN"],
                           ["HCCCCCN;v=0;hyp1", "HCCCCCN"],
                           ["HCCCDO;v=0;", "HCCCDO"],
                           ["HCCCH2CN;v=0;", "HCCCH$_2$CN"],
                           ["HCCCH2NH2;v=0;", "HCCCH$_2$NH$_2$"],
                           ["HCCCH2NH2;v=0;hyp1", "HCCCH$_2$NH$_2$"],
                           ["HCCCH2OD;v=0;", "HCCCH$_2$OD"],
                           ["HCCCH2OD;v=0;#1", "HCCCH$_2$OD"],
                           ["HCCCH2OH;v=0;", "HCCCH$_2$OH"],
                           ["HCCCH2OH;v=0;#1", "HCCCH$_2$OH"],
                           ["HCCCHO;v=0;", "HCCCHO"],
                           ["HCCCN-15;v7=1;", "HCCC$^{15}$N, v$_7$=1"],
                           ["HCCCN-15;v=0;", "HCCC$^{15}$N"],
                           ["HCCCN-15;v=0;#1", "HCCC$^{15}$N"],
                           ["HCCCN-15;v=0;#2", "HCCC$^{15}$N"],
                           ["HCCCN;(v4,v5,v7)=(1,0,2)-(0,2,0);", "HCCCN, (v$_4$,v$_5$,v$_7$)=(1,0,2)-(0,2,0)"],
                           ["HCCCN;v2=1;", "HCCCN, v$_2$=1"],
                           ["HCCCN;v3=1;", "HCCCN, v$_3$=1"],
                           ["HCCCN;v4=1,v7=1;", "HCCCN, v$_4$=1,v$_7$=1"],
                           ["HCCCN;v4=1,v7=2;", "HCCCN, v$_4$=1,v$_7$=2"],
                           ["HCCCN;v4=1;", "HCCCN, v$_4$=1"],
                           ["HCCCN;v5=1;", "HCCCN, v$_5$=1"],
                           ["HCCCN;v5=2;", "HCCCN, v$_5$=2"],
                           ["HCCCN;v6=1,v7=1;", "HCCCN, v$_6$=1,v$_7$=1"],
                           ["HCCCN;v6=1;", "HCCCN, v$_6$=1"],
                           ["HCCCN;v7=1;", "HCCCN, v$_7$=1"],
                           ["HCCCN;v7=2;", "HCCCN, v$_7$=2"],
                           ["HCCCN;v7=3;", "HCCCN, v$_7$=3"],
                           ["HCCCN;v=0;", "HCCCN"],
                           ["HCCCN;v=0;#1", "HCCCN"],
                           ["HCCCN;v=0;#2", "HCCCN"],
                           ["HCCD;v=0;", "HCCD"],
                           ["HCCN-15;v=0;", "HCC$^{15}$N"],
                           ["HCCN;v=0;", "HCCN"],
                           ["HCCN;v=0;hyp1", "HCCN"],
                           ["HCCNC;v5=1;", "HCCNC, v$_5$=1"],
                           ["HCCNC;v5=1;#1", "HCCNC, v$_5$=1"],
                           ["HCCNC;v6=1;", "HCCNC, v$_6$=1"],
                           ["HCCNC;v6=1;#1", "HCCNC, v$_6$=1"],
                           ["HCCNC;v7=1;", "HCCNC, v$_7$=1"],
                           ["HCCNC;v7=1;#1", "HCCNC, v$_7$=1"],
                           ["HCCNC;v=0;", "HCCNC"],
                           ["HCCNC;v=0;#1", "HCCNC"],
                           ["HCCNSi;v=0;", "HCCNSi"],
                           ["HCCO;v=0;hyp1", "HCCO"],
                           ["HCCP;v=0;", "HCCP"],
                           ["HCCS;v=0;", "HCCS"],
                           ["HCCS;v=0;hyp1", "HCCS"],
                           ["HCCSH;v=0;", "HCCSH"],
                           ["HCN-15;v1=1;", "HC$^{15}$N, v$_1$=1"],
                           ["HCN-15;v2=1;", "HC$^{15}$N, v$_2$=1"],
                           ["HCN-15;v2=2;", "HC$^{15}$N, v$_2$=2"],
                           ["HCN-15;v2=3;", "HC$^{15}$N, v$_2$=3"],
                           ["HCN-15;v3=1;", "HC$^{15}$N, v$_3$=1"],
                           ["HCN-15;v=0;", "HC$^{15}$N"],
                           ["HCN;v1=1;", "HCN, v$_1$=1"],
                           ["HCN;v1=1;#1", "HCN, v$_1$=1"],
                           ["HCN;v2=1;", "HCN, v$_2$=1"],
                           ["HCN;v2=1;#1", "HCN, v$_2$=1"],
                           ["HCN;v2=1;hyp1", "HCN, v$_2$=1"],
                           ["HCN;v2=1;hyp1#1", "HCN, v$_2$=1"],
                           ["HCN;v2=2;", "HCN, v$_2$=2"],
                           ["HCN;v2=2;#1", "HCN, v$_2$=2"],
                           ["HCN;v2=2;hyp1", "HCN, v$_2$=2"],
                           ["HCN;v2=2;hyp1#1", "HCN, v$_2$=2"],
                           ["HCN;v2=3;", "HCN, v$_2$=3"],
                           ["HCN;v2=3;#1", "HCN, v$_2$=3"],
                           ["HCN;v3=1;", "HCN, v$_3$=1"],
                           ["HCN;v3=1;#1", "HCN, v$_3$=1"],
                           ["HCN;v=0;", "HCN"],
                           ["HCN;v=0;#1", "HCN"],
                           ["HCN;v=0;hyp1", "HCN"],
                           ["HCN;v=0;hyp1#1", "HCN"],
                           ["HCND+;v=0;", "HCND$^+$"],
                           ["HCNH+;v=0;", "HCNH$^+$"],
                           ["HCNO;v=0;", "HCNO"],
                           ["HCNO;v=0;hyp1", "HCNO"],
                           ["HCNS;v=0;", "HCNS"],
                           ["HCNS;v=0;hyp1", "HCNS"],
                           ["HCO+;v2=1-0;", "HCO$^+$, v$_2$=1-0"],
                           ["HCO+;v2=1-0;#1", "HCO$^+$, v$_2$=1-0"],
                           ["HCO+;v2=1-0;#2", "HCO$^+$, v$_2$=1-0"],
                           ["HCO+;v2=1;", "HCO$^+$, v$_2$=1"],
                           ["HCO+;v2=2-0;", "HCO$^+$, v$_2$=2-0"],
                           ["HCO+;v2=2-0;#1", "HCO$^+$, v$_2$=2-0"],
                           ["HCO+;v=0;", "HCO$^+$"],
                           ["HCO+;v=0;#1", "HCO$^+$"],
                           ["HCO+;v=0;#2", "HCO$^+$"],
                           ["HCO-17+;v=0;", "HC$^{17}$O$^+$"],
                           ["HCO-18+;v=0;", "HC$^{18}$O$^+$"],
                           ["HCO-18+;v=0;#1", "HC$^{18}$O$^+$"],
                           ["HCO-18+;v=0;#2", "HC$^{18}$O$^+$"],
                           ["HCO;v=0;", "HCO"],
                           ["HCOCH2OH;v12=1;", "HCOCH$_2$OH, v$_{12}$=1"],
                           ["HCOCH2OH;v12=1;#1", "HCOCH$_2$OH, v$_{12}$=1"],
                           ["HCOCH2OH;v17=1;", "HCOCH$_2$OH, v$_{17}$=1"],
                           ["HCOCH2OH;v17=1;#1", "HCOCH$_2$OH, v$_{17}$=1"],
                           ["HCOCH2OH;v18=1;", "HCOCH$_2$OH, v$_{18}$=1"],
                           ["HCOCH2OH;v18=1;#1", "HCOCH$_2$OH, v$_{18}$=1"],
                           ["HCOCH2OH;v=0;", "HCOCH$_2$OH"],
                           ["HCOCH2OH;v=0;#1", "HCOCH$_2$OH"],
                           ["HCOOD;v=0;", "HCOOD"],
                           ["HCOOD;v=0;#1", "HCOOD"],
                           ["HCOOH;v=0;", "HCOOH"],
                           ["HCOOH;v=0;#1", "HCOOH"],
                           ["HCP;v2=1;", "HCP, v$_2$=1"],
                           ["HCP;v=0;", "HCP"],
                           ["HCP;v=0;#1", "HCP"],
                           ["HCP;v=0;#2", "HCP"],
                           ["HCS+;v=0;", "HCS$^+$"],
                           ["HCS+;v=0;#1", "HCS$^+$"],
                           ["HCS+;v=0;#2", "HCS$^+$"],
                           ["HCS-34+;v=0;", "HC$^{34}$S$^+$"],
                           ["HCS;v=0;", "HCS"],
                           ["HCl-35-CO;v=0;", "H$^{35}$ClCO"],
                           ["HCl-35-CO;v=0;#1", "H$^{35}$ClCO"],
                           ["HCl-37-CO;v=0;", "H$^{37}$ClCO"],
                           ["HCl-37-CO;v=0;#1", "H$^{37}$ClCO"],
                           ["HCl-37;v=0;", "H$^{37}$Cl"],
                           ["HCl-37;v=0;#1", "H$^{37}$Cl"],
                           ["HCl;v=0;", "HCl"],
                           ["HCl;v=0;#1", "HCl"],
                           ["HD2+;v=0;", "HD$_2 \\! ^+$"],
                           ["HD2+;v=0;ortho", "ortho-HD$_2 \\! ^+$"],
                           ["HD2+;v=0;para", "para-HD$_2 \\! ^+$"],
                           ["HD;v=0;", "HD"],
                           ["HD;v=0;#1", "HD"],
                           ["HD;v=0;#2", "HD"],
                           ["HD;v=1;", "HD, v=1"],
                           ["HDC-13-O;v=0;", "HD$^{13}$CO"],
                           ["HDCCO;v=0;", "HDCCO"],
                           ["HDCO-18;v=0;", "HDC$^{18}$O"],
                           ["HDCO;v=0;", "HDCO"],
                           ["HDCO;v=0;#1", "HDCO"],
                           ["HDCO;v=0;#2", "HDCO"],
                           ["HDCS;v=0;", "HDCS"],
                           ["HDNCH2CN;v=0;", "HDNCH$_2$CN"],
                           ["HDNCN;v=0;", "HDNCN"],
                           ["HDO-18;v=0;", "HD$^{18}$O"],
                           ["HDO-18;v=0;#1", "HD$^{18}$O"],
                           ["HDO;v=0;", "HDO"],
                           ["HDO;v=0;#1", "HDO"],
                           ["HDS-34;v=0;", "HD$^{34}$S"],
                           ["HDS;v=0;", "HDS"],
                           ["HF;v=0;", "HF"],
                           ["HF;v=0;#1", "HF"],
                           ["HFCO;v5=1;", "HFCO, v$_5$=1"],
                           ["HFCO;v5=1;#1", "HFCO, v$_5$=1"],
                           ["HFCO;v=0;", "HFCO"],
                           ["HFCO;v=0;#1", "HFCO"],
                           ["HMgNC;v=0;", "HMgNC"],
                           ["HMgNC;v=0;hyp1", "HMgNC"],
                           ["HN-15-C;v=0;", "H$^{15}$NC"],
                           ["HN-15-C;v=0;#1", "H$^{15}$NC"],
                           ["HN-15-CO;v=0;", "H$^{15}$NCO"],
                           ["HN-15-CO;v=0;#1", "H$^{15}$NCO"],
                           ["HN-15-CS;v=0;", "H$^{15}$NCS"],
                           ["HN-15-O3;v=0;", "H$^{15}$NO$_3$"],
                           ["HN-15-O3;v=0;#1", "H$^{15}$NO$_3$"],
                           ["HN2O+;v=0;", "HN$_2$O$^+$"],
                           ["HNC-13-O;v=0;", "HN$^{13}$CO"],
                           ["HNC-13-O;v=0;#1", "HN$^{13}$CO"],
                           ["HNC-13-S;v=0;", "HN$^{13}$CS"],
                           ["HNC-13;v=0;", "HN$^{13}$C"],
                           ["HNC3;v=0;", "HNC$_3$"],
                           ["HNC3;v=0;hyp1", "HNC$_3$"],
                           ["HNC;v2=1;", "HNC, v$_2$=1"],
                           ["HNC;v=0;", "HNC"],
                           ["HNCCC;v=0;", "HNCCC"],
                           ["HNCCC;v=0;#1", "HNCCC"],
                           ["HNCCN+;v=0;", "HNCCN$^+$"],
                           ["HNCCN+;v=0;#1", "HNCCN$^+$"],
                           ["HNCN;v=0;", "HNCN"],
                           ["HNCNH;v=0;", "HNCNH"],
                           ["HNCO-18;v=0;", "HNC$^{18}$O"],
                           ["HNCO-18;v=0;#1", "HNC$^{18}$O"],
                           ["HNCO;v=0;", "HNCO"],
                           ["HNCS-34;v=0;", "HNC$^{34}$S"],
                           ["HNCS;v=0;", "HNCS"],
                           ["HNCS;v=0;#1", "HNCS"],
                           ["HNO3;(v9,v5)=(2,0)-(0,1);", "HNO$_3$, (v$_9$,v$_5$)=(2,0)-(0,1)"],
                           ["HNO3;(v9,v5)=(2,0)-(0,1);#1", "HNO$_3$, (v$_9$,v$_5$)=(2,0)-(0,1)"],
                           ["HNO3;v5=1;", "HNO$_3$, v$_5$=1"],
                           ["HNO3;v5=1;#1", "HNO$_3$, v$_5$=1"],
                           ["HNO3;v6=1;", "HNO$_3$, v$_6$=1"],
                           ["HNO3;v8=1;", "HNO$_3$, v$_8$=1"],
                           ["HNO3;v8=1;#1", "HNO$_3$, v$_8$=1"],
                           ["HNO3;v9=1;", "HNO$_3$, v$_9$=1"],
                           ["HNO3;v9=1;#1", "HNO$_3$, v$_9$=1"],
                           ["HNO3;v9=2;", "HNO$_3$, v$_9$=2"],
                           ["HNO3;v9=2;#1", "HNO$_3$, v$_9$=2"],
                           ["HNO3;v=0;", "HNO$_3$"],
                           ["HNO3;v=0;#1", "HNO$_3$"],
                           ["HNO;v=0;", "HNO"],
                           ["HNO;v=0;#1", "HNO"],
                           ["HNSi;v=0;", "HNSi"],
                           ["HO-18-C+;v=0;", "H$^{18}$OC$^+$"],
                           ["HO-18-C+;v=0;#1", "H$^{18}$OC$^+$"],
                           ["HO-18-NO2;v=0;", "H$^{18}$ONO$_2$"],
                           ["HO-18-NO2;v=0;#1", "H$^{18}$ONO$_2$"],
                           ["HO-18;v=0;", "H$^{18}$O"],
                           ["HO-18;v=0;#1", "H$^{18}$O"],
                           ["HO2;v=0;", "HO$_2$"],
                           ["HO2;v=0;#1", "HO$_2$"],
                           ["HO3;v=0;", "HO$_3$"],
                           ["HO3;v=0;#1", "HO$_3$"],
                           ["HOBr-79;v=0;", "HO$^{79}$Br"],
                           ["HOBr-79;v=0;#1", "HO$^{79}$Br"],
                           ["HOBr-81;v=0;", "HO$^{81}$Br"],
                           ["HOBr-81;v=0;#1", "HO$^{81}$Br"],
                           ["HOC+;v2=1;", "HOC$^+$, v$_2$=1"],
                           ["HOC+;v=0;", "HOC$^+$"],
                           ["HOC+;v=0;#1", "HOC$^+$"],
                           ["HOC+;v=0;#2", "HOC$^+$"],
                           ["HOC-13+;v=0;", "HO$^{13}$C$^+$"],
                           ["HOC-13+;v=0;#1", "HO$^{13}$C$^+$"],
                           ["HOC-13-O+;v=0;", "HO$^{13}$CO$^+$"],
                           ["HOCH2CN;v=0;", "HOCH$_2$CN"],
                           ["HOCHCHCHO;v=0;", "HOCHCHCHO"],
                           ["HOCHCHCHO;v=0;ortho", "ortho-HOCHCHCHO"],
                           ["HOCHCHCHO;v=0;para", "para-HOCHCHCHO"],
                           ["HOCN;v=0;", "HOCN"],
                           ["HOCO+;v=0;", "HOCO$^+$"],
                           ["HOCS+;v=0;", "HOCS$^+$"],
                           ["HOCl-37;v=0;", "HO$^{37}$Cl"],
                           ["HOCl-37;v=0;#1", "HO$^{37}$Cl"],
                           ["HOCl;v=0;", "HOCl"],
                           ["HOCl;v=0;#1", "HOCl"],
                           ["HOF;v=0;", "HOF"],
                           ["HOF;v=0;#1", "HOF"],
                           ["HON2+;v=0;", "HON$_2 \\! ^+$"],
                           ["HONC;v=0;", "HONC"],
                           ["HONO-18-O;v=0;", "HON$^{18}$OO"],
                           ["HONO-18-O;v=0;#1", "HON$^{18}$OO"],
                           ["HONO;v=0;", "HONO"],
                           ["HONO;v=0;#1", "HONO"],
                           ["HONOO-18;v=0;", "HONO$^{18}$O"],
                           ["HONOO-18;v=0;#1", "HONO$^{18}$O"],
                           ["HOONO2;v12=1;", "HOONO$_2$, v$_{12}$=1"],
                           ["HOONO2;v12=1;#1", "HOONO$_2$, v$_{12}$=1"],
                           ["HOONO2;v=0;", "HOONO$_2$"],
                           ["HOONO2;v=0;#1", "HOONO$_2$"],
                           ["HOONO;v=0;", "HOONO"],
                           ["HOONO;v=0;#1", "HOONO"],
                           ["HPO;v=0;", "HPO"],
                           ["HS-34-CN;v=0;", "H$^{34}$SCN"],
                           ["HS-34-CN;v=0;hyp1", "H$^{34}$SCN"],
                           ["HS2;v=0;", "HS2"],
                           ["HSC-13-N;v=0;", "HS$^{13}$CN"],
                           ["HSC-13-N;v=0;hyp1", "HS$^{13}$CN"],
                           ["HSC;v=0;", "HSC"],
                           ["HSCH2CN;v=0;", "HSCH$_2$CN"],
                           ["HSCN-15;v=0;", "HSC$^{15}$N"],
                           ["HSCN;v=0;", "HSCN"],
                           ["HSCO+;v=0;", "HSCO$^+$"],
                           ["HSNC;v=0;", "HSNC"],
                           ["HSNC;v=0;hyp1", "HSNC"],
                           ["HSO;v=0;hyp1", "HSO"],
                           ["HSiCN;v=0;", "HSiCN"],
                           ["HSiCN;v=0;hyp1", "HSiCN"],
                           ["HSiNC;v=0;", "HSiNC"],
                           ["HSiNC;v=0;hyp1", "HSiNC"],
                           ["HSiO;v=0;hyp1", "HSiO"],
                           ["HSiS;v=0;", "HSiS"],
                           ["HeH+;v=0;", "HeH$^+$"],
                           ["IO;v=0;", "IO"],
                           ["IO;v=0;#1", "IO"],
                           ["IO;v=1;", "IO, v=1"],
                           ["IO;v=1;#1", "IO, v=1"],
                           ["IO;v=2;", "IO, v=2"],
                           ["IO;v=2;#1", "IO, v=2"],
                           ["IO;v=3;", "IO, v=3"],
                           ["IO;v=3;#1", "IO, v=3"],
                           ["IO;v=4;", "IO, v=4"],
                           ["IO;v=4;#1", "IO, v=4"],
                           ["K-39-Cl-35;v=0;", "$^{39}$K$^{35}$Cl"],
                           ["K-39-Cl-35;v=10;", "$^{39}$K$^{35}$Cl, v=10"],
                           ["K-39-Cl-35;v=11;", "$^{39}$K$^{35}$Cl, v=11"],
                           ["K-39-Cl-35;v=12;", "$^{39}$K$^{35}$Cl, v=12"],
                           ["K-39-Cl-35;v=13;", "$^{39}$K$^{35}$Cl, v=13"],
                           ["K-39-Cl-35;v=14;", "$^{39}$K$^{35}$Cl, v=14"],
                           ["K-39-Cl-35;v=15;", "$^{39}$K$^{35}$Cl, v=15"],
                           ["K-39-Cl-35;v=1;", "$^{39}$K$^{35}$Cl, v=1"],
                           ["K-39-Cl-35;v=2;", "$^{39}$K$^{35}$Cl, v=2"],
                           ["K-39-Cl-35;v=3;", "$^{39}$K$^{35}$Cl, v=3"],
                           ["K-39-Cl-35;v=4;", "$^{39}$K$^{35}$Cl, v=4"],
                           ["K-39-Cl-35;v=5;", "$^{39}$K$^{35}$Cl, v=5"],
                           ["K-39-Cl-35;v=6;", "$^{39}$K$^{35}$Cl, v=6"],
                           ["K-39-Cl-35;v=7;", "$^{39}$K$^{35}$Cl, v=7"],
                           ["K-39-Cl-35;v=8;", "$^{39}$K$^{35}$Cl, v=8"],
                           ["K-39-Cl-35;v=9;", "$^{39}$K$^{35}$Cl, v=9"],
                           ["K-39-Cl-36;v=0;", "$^{39}$K$^{36}$Cl"],
                           ["K-39-Cl-36;v=1;", "$^{39}$K$^{36}$Cl, v=1"],
                           ["K-39-Cl-36;v=2;", "$^{39}$K$^{36}$Cl, v=2"],
                           ["K-39-Cl-36;v=3;", "$^{39}$K$^{36}$Cl, v=3"],
                           ["K-39-Cl-36;v=4;", "$^{39}$K$^{36}$Cl, v=4"],
                           ["K-39-Cl-36;v=5;", "$^{39}$K$^{36}$Cl, v=5"],
                           ["K-39-Cl-37;v=0;", "$^{39}$K$^{37}$Cl"],
                           ["K-39-Cl-37;v=10;", "$^{39}$K$^{37}$Cl, v=10"],
                           ["K-39-Cl-37;v=11;", "$^{39}$K$^{37}$Cl, v=11"],
                           ["K-39-Cl-37;v=12;", "$^{39}$K$^{37}$Cl, v=12"],
                           ["K-39-Cl-37;v=13;", "$^{39}$K$^{37}$Cl, v=13"],
                           ["K-39-Cl-37;v=14;", "$^{39}$K$^{37}$Cl, v=14"],
                           ["K-39-Cl-37;v=15;", "$^{39}$K$^{37}$Cl, v=15"],
                           ["K-39-Cl-37;v=1;", "$^{39}$K$^{37}$Cl, v=1"],
                           ["K-39-Cl-37;v=2;", "$^{39}$K$^{37}$Cl, v=2"],
                           ["K-39-Cl-37;v=3;", "$^{39}$K$^{37}$Cl, v=3"],
                           ["K-39-Cl-37;v=4;", "$^{39}$K$^{37}$Cl, v=4"],
                           ["K-39-Cl-37;v=5;", "$^{39}$K$^{37}$Cl, v=5"],
                           ["K-39-Cl-37;v=6;", "$^{39}$K$^{37}$Cl, v=6"],
                           ["K-39-Cl-37;v=7;", "$^{39}$K$^{37}$Cl, v=7"],
                           ["K-39-Cl-37;v=8;", "$^{39}$K$^{37}$Cl, v=8"],
                           ["K-39-Cl-37;v=9;", "$^{39}$K$^{37}$Cl, v=9"],
                           ["K-40-Cl-35;v=0;", "$^{40}$K$^{35}$Cl"],
                           ["K-40-Cl-35;v=1;", "$^{40}$K$^{35}$Cl, v=1"],
                           ["K-40-Cl-35;v=2;", "$^{40}$K$^{35}$Cl, v=2"],
                           ["K-40-Cl-35;v=3;", "$^{40}$K$^{35}$Cl, v=3"],
                           ["K-40-Cl-35;v=4;", "$^{40}$K$^{35}$Cl, v=4"],
                           ["K-40-Cl-35;v=5;", "$^{40}$K$^{35}$Cl, v=5"],
                           ["K-40-Cl-37;v=0;", "$^{40}$K$^{37}$Cl"],
                           ["K-40-Cl-37;v=1;", "$^{40}$K$^{37}$Cl, v=1"],
                           ["K-40-Cl-37;v=2;", "$^{40}$K$^{37}$Cl, v=2"],
                           ["K-40-Cl-37;v=3;", "$^{40}$K$^{37}$Cl, v=3"],
                           ["K-40-Cl-37;v=4;", "$^{40}$K$^{37}$Cl, v=4"],
                           ["K-40-Cl-37;v=5;", "$^{40}$K$^{37}$Cl, v=5"],
                           ["K-41-Cl-35;v=0;", "$^{41}$K-$^{35}$Cl"],
                           ["K-41-Cl-35;v=10;", "$^{41}$K-$^{35}$Cl, v=10"],
                           ["K-41-Cl-35;v=11;", "$^{41}$K-$^{35}$Cl, v=11"],
                           ["K-41-Cl-35;v=12;", "$^{41}$K-$^{35}$Cl, v=12"],
                           ["K-41-Cl-35;v=13;", "$^{41}$K-$^{35}$Cl, v=13"],
                           ["K-41-Cl-35;v=14;", "$^{41}$K-$^{35}$Cl, v=14"],
                           ["K-41-Cl-35;v=15;", "$^{41}$K-$^{35}$Cl, v=15"],
                           ["K-41-Cl-35;v=1;", "$^{41}$K-$^{35}$Cl, v=1"],
                           ["K-41-Cl-35;v=2;", "$^{41}$K-$^{35}$Cl, v=2"],
                           ["K-41-Cl-35;v=3;", "$^{41}$K-$^{35}$Cl, v=3"],
                           ["K-41-Cl-35;v=4;", "$^{41}$K-$^{35}$Cl, v=4"],
                           ["K-41-Cl-35;v=5;", "$^{41}$K-$^{35}$Cl, v=5"],
                           ["K-41-Cl-35;v=6;", "$^{41}$K-$^{35}$Cl, v=6"],
                           ["K-41-Cl-35;v=7;", "$^{41}$K-$^{35}$Cl, v=7"],
                           ["K-41-Cl-35;v=8;", "$^{41}$K-$^{35}$Cl, v=8"],
                           ["K-41-Cl-35;v=9;", "$^{41}$K-$^{35}$Cl, v=9"],
                           ["K-41-Cl-36;v=0;", "$^{41}$K$^{36}$Cl"],
                           ["K-41-Cl-36;v=1;", "$^{41}$K$^{36}$Cl, v=1"],
                           ["K-41-Cl-36;v=2;", "$^{41}$K$^{36}$Cl, v=2"],
                           ["K-41-Cl-36;v=3;", "$^{41}$K$^{36}$Cl, v=3"],
                           ["K-41-Cl-36;v=4;", "$^{41}$K$^{36}$Cl, v=4"],
                           ["K-41-Cl-36;v=5;", "$^{41}$K$^{36}$Cl, v=5"],
                           ["K-41-Cl-37;v=0;", "$^{41}$K-$^{37}$Cl"],
                           ["K-41-Cl-37;v=10;", "$^{41}$K-$^{37}$Cl, v=10"],
                           ["K-41-Cl-37;v=11;", "$^{41}$K-$^{37}$Cl, v=11"],
                           ["K-41-Cl-37;v=12;", "$^{41}$K-$^{37}$Cl, v=12"],
                           ["K-41-Cl-37;v=13;", "$^{41}$K-$^{37}$Cl, v=13"],
                           ["K-41-Cl-37;v=14;", "$^{41}$K-$^{37}$Cl, v=14"],
                           ["K-41-Cl-37;v=15;", "$^{41}$K-$^{37}$Cl, v=15"],
                           ["K-41-Cl-37;v=1;", "$^{41}$K-$^{37}$Cl, v=1"],
                           ["K-41-Cl-37;v=2;", "$^{41}$K-$^{37}$Cl, v=2"],
                           ["K-41-Cl-37;v=3;", "$^{41}$K-$^{37}$Cl, v=3"],
                           ["K-41-Cl-37;v=4;", "$^{41}$K-$^{37}$Cl, v=4"],
                           ["K-41-Cl-37;v=5;", "$^{41}$K-$^{37}$Cl, v=5"],
                           ["K-41-Cl-37;v=6;", "$^{41}$K-$^{37}$Cl, v=6"],
                           ["K-41-Cl-37;v=7;", "$^{41}$K-$^{37}$Cl, v=7"],
                           ["K-41-Cl-37;v=8;", "$^{41}$K-$^{37}$Cl, v=8"],
                           ["K-41-Cl-37;v=9;", "$^{41}$K-$^{37}$Cl, v=9"],
                           ["K-41-H;v=0;", "$^{41}$K-H"],
                           ["K-41-H;v=0;#1", "$^{41}$K-H"],
                           ["K-41-H;v=1;", "$^{41}$K-H, v=1"],
                           ["K-41-H;v=1;#1", "$^{41}$K-H, v=1"],
                           ["K-41-OH;v=0;", "$^{41}$K-OH"],
                           ["KC;v=0;", "KC"],
                           ["KCCH;v=0;", "KCCH"],
                           ["KCH;v=0;", "KCH"],
                           ["KCN,KNC;v=0;", "KCN,KNC"],
                           ["KCl-35;v=0;", "K$^{35}$Cl"],
                           ["KCl-35;v=0;#1", "K$^{35}$Cl"],
                           ["KCl-37;v=0;", "K$^{37}$Cl"],
                           ["KCl-37;v=0;#1", "K$^{37}$Cl"],
                           ["KF;v=0;", "KF"],
                           ["KF;v=1;", "KF, v=1"],
                           ["KH;v=0;", "KH"],
                           ["KOH;v=0;", "KOH"],
                           ["KS;v=0;", "KS"],
                           ["KSH;v=0;", "KSH"],
                           ["Li-6-D;v=0;", "$^{6}$LiD"],
                           ["Li-6-D;v=0;#1", "$^{6}$LiD"],
                           ["Li-6-D;v=1;", "$^{6}$LiD, v=1"],
                           ["Li-6-D;v=1;#1", "$^{6}$LiD, v=1"],
                           ["Li-6-H;v=0;", "$^{6}$LiH"],
                           ["Li-6-H;v=0;#1", "$^{6}$LiH"],
                           ["Li-6-H;v=1;", "$^{6}$LiH, v=1"],
                           ["Li-6-H;v=1;#1", "$^{6}$LiH, v=1"],
                           ["LiCCH;v=0;", "LiCCH"],
                           ["LiD;v=0;", "LiD"],
                           ["LiD;v=0;#1", "LiD"],
                           ["LiD;v=1;", "LiD, v=1"],
                           ["LiD;v=1;#1", "LiD, v=1"],
                           ["LiH;v=0;", "LiH"],
                           ["LiH;v=0;#1", "LiH"],
                           ["LiH;v=1;", "LiH, v=1"],
                           ["LiH;v=1;#1", "LiH, v=1"],
                           ["LiOH;v=0;", "LiOH"],
                           ["MgC3N;v=0;", "MgC$_3$N"],
                           ["MgC4H;v=0;", "MgC$_4$H"],
                           ["MgCCH;v=0;", "MgCCH"],
                           ["MgCN;v=0;", "MgCN"],
                           ["MgCN;v=0;#1", "MgCN"],
                           ["MgCl;v=0;", "MgCl"],
                           ["MgF;v=0;", "MgF"],
                           ["MgH;v=0;", "MgH"],
                           ["MgNC;v2=1;", "MgNC, v$_2$=1"],
                           ["MgNC;v=0;", "MgNC"],
                           ["MgNC;v=0;#1", "MgNC"],
                           ["MgNC;v=0;#2", "MgNC"],
                           ["MgO;v=0;", "MgO"],
                           ["MgO;v=0;#1", "MgO"],
                           ["MgO;v=1;", "MgO, v=1"],
                           ["MgO;v=1;#1", "MgO, v=1"],
                           ["MgOH;v=0;", "MgOH"],
                           ["MgOH;v=0;hyp1", "MgOH"],
                           ["MgS;v=0;", "MgS"],
                           ["MgS;v=0;#1", "MgS"],
                           ["MgSH;v=0;", "MgSH"],
                           ["MnO;v=0;", "MnO"],
                           ["MnO;v=0;#1", "MnO"],
                           ["N-15-H2D;v=0;", "$^{15}$NH$_2$D"],
                           ["N-15-H2D;v=0;ortho", "ortho-$^{15}$NH$_2$D"],
                           ["N-15-H2D;v=0;para", "para-$^{15}$NH$_2$D"],
                           ["N-15-H3;v=0;", "$^{15}$NH$_3$"],
                           ["N-15-H3;v=0;#1", "$^{15}$NH$_3$"],
                           ["N-15-H3;v=0;#2", "$^{15}$NH$_3$"],
                           ["N-15-H3;v=0;ortho", "ortho-$^{15}$NH$_3$"],
                           ["N-15-H3;v=0;ortho#1", "$^{15}$NH$_3$"],
                           ["N-15-H3;v=0;ortho#2", "$^{15}$NH$_3$"],
                           ["N-15-H3;v=0;para", "para-$^{15}$NH$_3$"],
                           ["N-15-H3;v=0;para#1", "$^{15}$NH$_3$"],
                           ["N-15-H3;v=0;para#2", "$^{15}$NH$_3$"],
                           ["N-15-H;v=0;hyp2", "$^{15}$NH"],
                           ["N-15-HD2;v=0;", "$^{15}$NHD$_2$"],
                           ["N-15-HD2;v=0;ortho", "ortho-$^{15}$NHD$_2$"],
                           ["N-15-HD2;v=0;para", "para-$^{15}$NHD$_2$"],
                           ["N-15-N-15-D+;v=0;", "$^{15}$N$^{15}$ND$^+$"],
                           ["N-15-N-15-H+;v=0;", "$^{15}$N$^{15}$NH$^+$"],
                           ["N-15-ND+;v=0;", "$^{15}$NND$^+$"],
                           ["N-15-NH+;v=0;", "$^{15}$NNH$^+$"],
                           ["N-15-NO;v=0;", "$^{15}$NNO"],
                           ["N-15-NO;v=0;#1", "$^{15}$NNO"],
                           ["N-15-O-17;v=0;hyp2", "$^{15}$N$^{17}$O"],
                           ["N-15-O;v=0;hyp1", "$^{15}$NO"],
                           ["N-15-S-34;v=0;", "$^{15}$N$^{34}$S"],
                           ["N-15-S;v=0;", "$^{15}$NS"],
                           ["N-36-S;v=0;", "N$^{36}$S"],
                           ["N2H+;v2=1-0;", "N$_2$H$^+$, v$_2$=1-0"],
                           ["N2H+;v2=1;", "N$_2$H$^+$, v$_2$=1"],
                           ["N2H+;v=0;", "N$_2$H$^+$"],
                           ["N2H+;v=0;hyp2", "N$_2$H$^+$"],
                           ["N2O-18;v=0;", "N$_2$$^{18}$O"],
                           ["N2O-18;v=0;#1", "N$_2$$^{18}$O"],
                           ["N2O;v2=1;", "N$_2$O, v$_2$=1"],
                           ["N2O;v2=1;#1", "N$_2$O, v$_2$=1"],
                           ["N2O;v2=2;", "N$_2$O, v$_2$=2"],
                           ["N2O;v2=2;#1", "N$_2$O, v$_2$=2"],
                           ["N2O;v=0;", "N$_2$O"],
                           ["N2O;v=0;#1", "N$_2$O"],
                           ["NC-13-CNH+;v=0;", "N$^{13}$CCNH$^+$"],
                           ["NC2P;v=0;", "NC$_2$P"],
                           ["NC3NC;v=0;", "NC$_3$NC"],
                           ["NCC(O)NH2;v=0;", "NCC(O)NH$_2$"],
                           ["NCC-13-NH+;v=0;", "NC$^{13}$CNH$^+$"],
                           ["NCCND+;v=0;", "NCCND$^+$"],
                           ["NCCNH+;v=0;", "NCCNH$^+$"],
                           ["NCCNH+;v=0;hyp2", "NCCNH$^+$"],
                           ["NCHCCO;v=0;", "NCHCCO"],
                           ["NCHCCO;v=0;hyp1", "NCHCCO"],
                           ["NCNSi;v=0;", "NCNSi"],
                           ["NCO-;v=0;", "NCO$^-$"],
                           ["NCO;v=0;", "NCO"],
                           ["NCS;v=0;", "NCS"],
                           ["ND3;v=0;", "ND$_3$"],
                           ["ND;v=0;", "ND"],
                           ["NH2;v=0;", "NH$_2$"],
                           ["NH2;v=0;ortho", "ortho-NH$_2$"],
                           ["NH2;v=0;para", "para-NH$_2$"],
                           ["NH2CH2CH2CN;v=0;", "NH$_2$CH$_2$CH$_2$CN"],
                           ["NH2CH2CH2CN;v=0;#1", "NH$_2$CH$_2$CH$_2$CN"],
                           ["NH2CH2CH2OH;v25=1,v27=1;", "NH$_2$CH$_2$CH$_2$OH, v$_{25}$=1,v$_{27}$=1"],
                           ["NH2CH2CH2OH;v25=1,v27=1;#1", "NH$_2$CH$_2$CH$_2$OH, v$_{25}$=1,v$_{27}$=1"],
                           ["NH2CH2CH2OH;v25=1;", "NH$_2$CH$_2$CH$_2$OH, v$_{25}$=1"],
                           ["NH2CH2CH2OH;v25=1;#1", "NH$_2$CH$_2$CH$_2$OH, v$_{25}$=1"],
                           ["NH2CH2CH2OH;v26=1,v27=1;", "NH$_2$CH$_2$CH$_2$OH, v$_{26}$=1,v$_{27}$=1"],
                           ["NH2CH2CH2OH;v26=1,v27=1;#1", "NH$_2$CH$_2$CH$_2$OH, v$_{26}$=1,v$_{27}$=1"],
                           ["NH2CH2CH2OH;v26=1;", "NH$_2$CH$_2$CH$_2$OH, v$_{26}$=1"],
                           ["NH2CH2CH2OH;v26=1;#1", "NH$_2$CH$_2$CH$_2$OH, v$_{26}$=1"],
                           ["NH2CH2CH2OH;v27=1;", "NH$_2$CH$_2$CH$_2$OH, v$_{27}$=1"],
                           ["NH2CH2CH2OH;v27=1;#1", "NH$_2$CH$_2$CH$_2$OH, v$_{27}$=1"],
                           ["NH2CH2CH2OH;v27=2;", "NH$_2$CH$_2$CH$_2$OH, v$_{27}$=2"],
                           ["NH2CH2CH2OH;v27=2;#1", "NH$_2$CH$_2$CH$_2$OH, v$_{27}$=2"],
                           ["NH2CH2CH2OH;v=0;", "NH$_2$CH$_2$CH$_2$OH"],
                           ["NH2CH2CH2OH;v=0;#1", "NH$_2$CH$_2$CH$_2$OH"],
                           ["NH2CN;v=0;", "NH$_2$CN"],
                           ["NH2CN;v=0;#1", "NH$_2$CN"],
                           ["NH2CN;v=0;ortho", "ortho-NH$_2$CN"],
                           ["NH2CN;v=0;para", "para-NH$_2$CN"],
                           ["NH2D;v=0;", "NH$_2$D"],
                           ["NH2D;v=0;#1", "NH$_2$D"],
                           ["NH2D;v=0;#2", "NH$_2$D"],
                           ["NH2D;v=0;hyp1", "NH$_2$D"],
                           ["NH2D;v=0;ortho", "ortho-NH$_2$D"],
                           ["NH2D;v=0;para", "para-NH$_2$D"],
                           ["NH2OH;v=0;", "NH$_2$OH"],
                           ["NH3;v2=1-0;", "NH$_3$, v$_2$=1-0"],
                           ["NH3;v2=1-0;#1", "NH$_3$, v$_2$=1-0"],
                           ["NH3;v2=1-0;ortho", "ortho-NH$_3$, v$_2$=1-0"],
                           ["NH3;v2=1-0;ortho#1", "NH$_3$, v$_2$=1-0"],
                           ["NH3;v2=1-0;para", "para-NH$_3$, v$_2$=1-0"],
                           ["NH3;v2=1-0;para#1", "NH$_3$, v$_2$=1-0"],
                           ["NH3;v2=1;", "NH$_3$, v$_2$=1"],
                           ["NH3;v2=1;#1", "NH$_3$, v$_2$=1"],
                           ["NH3;v2=1;ortho", "ortho-NH$_3$, v$_2$=1"],
                           ["NH3;v2=1;ortho#1", "NH$_3$, v$_2$=1"],
                           ["NH3;v2=1;para", "para-NH$_3$, v$_2$=1"],
                           ["NH3;v2=1;para#1", "NH$_3$, v$_2$=1"],
                           ["NH3;v=0;", "NH$_3$"],
                           ["NH3;v=0;#1", "NH$_3$"],
                           ["NH3;v=0;ortho", "ortho-NH$_3$"],
                           ["NH3;v=0;ortho#1", "NH$_3$"],
                           ["NH3;v=0;para", "para-NH$_3$"],
                           ["NH3;v=0;para#1", "NH$_3$"],
                           ["NH3D+;v=0;", "NH$_3$D$^+$"],
                           ["NH3D+;v=0;A", "NH$_3$D$^+$ A"],
                           ["NH3D+;v=0;E", "NH$_3$D$^+$ E"],
                           ["NH;v=0;", "NH"],
                           ["NH;v=0;#1", "NH"],
                           ["NH;v=0;#2", "NH"],
                           ["NHD2;v=0;", "NHD$_2$"],
                           ["NHD2;v=0;hyp1", "NHD$_2$"],
                           ["NHD2;v=0;hyp1;ortho", "ortho-NHD$_2$"],
                           ["NHD2;v=0;hyp1;para", "para-NHD$_2$"],
                           ["NHD2;v=0;ortho", "ortho-NHD$_2$"],
                           ["NHD2;v=0;para", "para-NHD$_2$"],
                           ["NN-15-D+;v=0;", "N$^{15}$ND$^+$"],
                           ["NN-15-H+;v=0;", "N$^{15}$NH$^+$"],
                           ["NN-15-O;v=0;", "N$^{15}$NO"],
                           ["NN-15-O;v=0;#1", "N$^{15}$NO"],
                           ["NO+;v=0;", "NO$^+$"],
                           ["NO+;v=0;hyp1", "NO$^+$"],
                           ["NO+;v=1;", "NO$^+$, v=1"],
                           ["NO-17;v=0;hyp2", "N$^{17}$O"],
                           ["NO-18;v=0;hyp1", "N$^{18}$O"],
                           ["NO2;v=0;", "NO$_2$"],
                           ["NO2;v=0;#1", "NO$_2$"],
                           ["NO;v=0;hyp1", "NO"],
                           ["NO;v=1;hyp1", "NO, v=1"],
                           ["NS+;v=0;", "NS$^+$"],
                           ["NS+;v=0;hyp1", "NS$^+$"],
                           ["NS-33;v=0;", "N$^{33}$S"],
                           ["NS-34;v=0;", "N$^{34}$S"],
                           ["NS;v=0;hyp1", "NS"],
                           ["NS;v=1-0;", "NS, v=1-0"],
                           ["NS;v=1;", "NS, v=1"],
                           ["NaC-13-N,NaNC-13;v=0;", "Na$^{13}$CN,NaN$^{13}$C"],
                           ["NaC;v=0;", "NaC"],
                           ["NaCCH;v=0;", "NaCCH"],
                           ["NaCH;v=0;", "NaCH"],
                           ["NaCN,NaNC;v=0;", "NaCN,NaNC"],
                           ["NaCl-35;v=0;", "Na$^{35}$Cl"],
                           ["NaCl-35;v=10;", "Na$^{35}$Cl, v=10"],
                           ["NaCl-35;v=11;", "Na$^{35}$Cl, v=11"],
                           ["NaCl-35;v=12;", "Na$^{35}$Cl, v=12"],
                           ["NaCl-35;v=13;", "Na$^{35}$Cl, v=13"],
                           ["NaCl-35;v=14;", "Na$^{35}$Cl, v=14"],
                           ["NaCl-35;v=15;", "Na$^{35}$Cl, v=15"],
                           ["NaCl-35;v=1;", "Na$^{35}$Cl, v=1"],
                           ["NaCl-35;v=2;", "Na$^{35}$Cl, v=2"],
                           ["NaCl-35;v=3;", "Na$^{35}$Cl, v=3"],
                           ["NaCl-35;v=4;", "Na$^{35}$Cl, v=4"],
                           ["NaCl-35;v=5;", "Na$^{35}$Cl, v=5"],
                           ["NaCl-35;v=6;", "Na$^{35}$Cl, v=6"],
                           ["NaCl-35;v=7;", "Na$^{35}$Cl, v=7"],
                           ["NaCl-35;v=8;", "Na$^{35}$Cl, v=8"],
                           ["NaCl-35;v=9;", "Na$^{35}$Cl, v=9"],
                           ["NaCl-36;v=0;", "Na$^{36}$Cl"],
                           ["NaCl-36;v=1;", "Na$^{36}$Cl, v=1"],
                           ["NaCl-36;v=2;", "Na$^{36}$Cl, v=2"],
                           ["NaCl-36;v=3;", "Na$^{36}$Cl, v=3"],
                           ["NaCl-36;v=4;", "Na$^{36}$Cl, v=4"],
                           ["NaCl-36;v=5;", "Na$^{36}$Cl, v=5"],
                           ["NaCl-37;v=0;", "Na$^{37}$Cl"],
                           ["NaCl-37;v=0;#1", "Na$^{37}$Cl"],
                           ["NaCl-37;v=0;#2", "Na$^{37}$Cl"],
                           ["NaCl-37;v=10;", "Na$^{37}$Cl, v=10"],
                           ["NaCl-37;v=11;", "Na$^{37}$Cl, v=11"],
                           ["NaCl-37;v=12;", "Na$^{37}$Cl, v=12"],
                           ["NaCl-37;v=13;", "Na$^{37}$Cl, v=13"],
                           ["NaCl-37;v=14;", "Na$^{37}$Cl, v=14"],
                           ["NaCl-37;v=15;", "Na$^{37}$Cl, v=15"],
                           ["NaCl-37;v=1;", "Na$^{37}$Cl, v=1"],
                           ["NaCl-37;v=2;", "Na$^{37}$Cl, v=2"],
                           ["NaCl-37;v=3;", "Na$^{37}$Cl, v=3"],
                           ["NaCl-37;v=4;", "Na$^{37}$Cl, v=4"],
                           ["NaCl-37;v=5;", "Na$^{37}$Cl, v=5"],
                           ["NaCl-37;v=6;", "Na$^{37}$Cl, v=6"],
                           ["NaCl-37;v=7;", "Na$^{37}$Cl, v=7"],
                           ["NaCl-37;v=8;", "Na$^{37}$Cl, v=8"],
                           ["NaCl-37;v=9;", "Na$^{37}$Cl, v=9"],
                           ["NaCl;v=0;", "NaCl"],
                           ["NaCl;v=0;#1", "NaCl"],
                           ["NaF;v=0;", "NaF"],
                           ["NaF;v=1;", "NaF, v=1"],
                           ["NaH;v=0;", "NaH"],
                           ["NaH;v=0;#1", "NaH"],
                           ["NaH;v=0;#2", "NaH"],
                           ["NaH;v=1;", "NaH, v=1"],
                           ["NaH;v=1;#1", "NaH, v=1"],
                           ["NaO;v=0;", "NaO"],
                           ["NaO;v=0;#1", "NaO"],
                           ["NaO;v=1;", "NaO, v=1"],
                           ["NaO;v=1;#1", "NaO, v=1"],
                           ["NaO;v=2;", "NaO, v=2"],
                           ["NaO;v=2;#1", "NaO, v=2"],
                           ["NaO;v=3;", "NaO, v=3"],
                           ["NaO;v=3;#1", "NaO, v=3"],
                           ["NaOH;v=0;", "NaOH"],
                           ["NaOH;v=0;#1", "NaOH"],
                           ["NaOH;v=0;#2", "NaOH"],
                           ["NaOH;v=0;hyp1", "NaOH"],
                           ["NaS;v=0;", "NaS"],
                           ["NaSH;v=0;", "NaSH"],
                           ["Ne-22-H+;v=0;", "$^{22}$NeH$^+$"],
                           ["NeD+;v=0;", "NeD$^+$"],
                           ["NeH+;v=0;", "NeH$^+$"],
                           ["Ni-58-C;v=0;", "$^{58}$NiC"],
                           ["Ni-58-O;v=0;", "$^{58}$NiO"],
                           ["NiCO;v=0;", "NiCO"],
                           ["O-17-CO;v=0;", "$^{17}$OCO"],
                           ["O-17-CO;v=0;#1", "$^{17}$OCO"],
                           ["O-17-CS;v=0;", "$^{17}$OCS"],
                           ["O-17-CS;v=0;hyp1", "$^{17}$OCS"],
                           ["O-17-O;v=0;", "$^{17}$OO"],
                           ["O-17-O;v=0;#1", "$^{17}$OO"],
                           ["O-18-C-13-S;v=0;", "$^{18}$O$^{13}$CS"],
                           ["O-18-CO;v=0;", "$^{18}$OCO"],
                           ["O-18-CO;v=0;#1", "$^{18}$OCO"],
                           ["O-18-CS-34;v=0;", "$^{18}$OC$^{34}$S"],
                           ["O-18-CS;v=0;", "$^{18}$OCS"],
                           ["O-18-H+;v=0;", "$^{18}$OH$^+$"],
                           ["O-18-H-;v=0;", "$^{18}$OH$^-$"],
                           ["O2;v=0;", "O$_2$"],
                           ["O2;v=0;#1", "O$_2$"],
                           ["O2;v=1;", "O$_2$, v=1"],
                           ["O2;v=1;#1", "O$_2$, v=1"],
                           ["O2O-17;v=0;", "O$_2 \\! ^{17}$O"],
                           ["O2O-17;v=0;#1", "O$_2 \\! ^{17}$O"],
                           ["O2O-18;v2=1;", "O$_2 \\! ^{18}$O, v$_2$=1"],
                           ["O2O-18;v2=1;#1", "O$_2 \\! ^{18}$O, v$_2$=1"],
                           ["O2O-18;v=0;", "O$_2 \\! ^{18}$O"],
                           ["O2O-18;v=0;#1", "O$_2 \\! ^{18}$O"],
                           ["O3;(v1,v2,v3)=(1,0,0)-(0,0,1);", "O$_3$, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,0,1)"],
                           ["O3;(v1,v2,v3)=(1,0,0)-(0,0,1);#1", "O$_3$, (v$_1$,v$_2$,v$_3$)=(1,0,0)-(0,0,1)"],
                           ["O3;(v1,v2,v3)=(1,1,0)-(0,1,1);", "O$_3$, (v$_1$,v$_2$,v$_3$)=(1,1,0)-(0,1,1)"],
                           ["O3;(v1,v2,v3)=(1,1,0)-(0,1,1);#1", "O$_3$, (v$_1$,v$_2$,v$_3$)=(1,1,0)-(0,1,1)"],
                           ["O3;v1=1,v2=1;", "O$_3$, v$_1$=1,v2=1"],
                           ["O3;v1=1,v2=1;#1", "O$_3$, v$_1$=1,v2=1"],
                           ["O3;v1=1;", "O$_3$, v$_1$=1"],
                           ["O3;v1=1;#1", "O$_3$, v$_1$=1"],
                           ["O3;v2=1,v3=1;", "O$_3$, v$_2$=1,v3=1"],
                           ["O3;v2=1,v3=1;#1", "O$_3$, v$_2$=1,v3=1"],
                           ["O3;v2=1;", "O$_3$, v$_2$=1"],
                           ["O3;v2=1;#1", "O$_3$, v$_2$=1"],
                           ["O3;v2=2;", "O$_3$, v$_2$=2"],
                           ["O3;v2=2;#1", "O$_3$, v$_2$=2"],
                           ["O3;v3=1;", "O$_3$, v$_3$=1"],
                           ["O3;v3=1;#1", "O$_3$, v$_3$=1"],
                           ["O3;v=0;", "O$_3$"],
                           ["O3;v=0;#1", "O$_3$"],
                           ["OBr-81-O;v=0;", "O$^{81}$BrO"],
                           ["OBr-81-O;v=0;#1", "O$^{81}$BrO"],
                           ["OC(CN)2;v=0;", "OC(CN)$_2$"],
                           ["OC-13-S-33;v=0;", "O$^{13}$C$^{33}$S"],
                           ["OC-13-S-34;v=0;", "O$^{13}$C$^{34}$S"],
                           ["OC-13-S;v=0;", "O$^{13}$CS"],
                           ["OC3S;v=0;", "OC$_3$S"],
                           ["OCS-33;v=0;", "OC$^{33}$S"],
                           ["OCS-34;v=0;", "OC$^{34}$S"],
                           ["OCS-36;v=0;", "OC$^{36}$S"],
                           ["OCS;v2=1;", "OCS, v$_2$=1"],
                           ["OCS;v=0;", "OCS"],
                           ["OCl-35-O;v=0;", "O$^{35}$ClO"],
                           ["OCl-35-O;v=0;#1", "O$^{35}$ClO"],
                           ["OCl-37-O;v=0;", "O$^{37}$ClO"],
                           ["OCl-37-O;v=0;#1", "O$^{37}$ClO"],
                           ["OD+;v=0;", "OD$^+$"],
                           ["OD-;v=0;", "OD$^-$"],
                           ["OD;v=0;", "OD"],
                           ["OD;v=0;#1", "OD"],
                           ["OH+;v=0;hyp1", "OH$^+$"],
                           ["OH-;v=0;", "OH$^-$"],
                           ["OH;v=0;", "OH"],
                           ["OH;v=0;#1", "OH"],
                           ["OH;v=1;", "OH, v=1"],
                           ["OH;v=1;#1", "OH, v=1"],
                           ["OH;v=2;", "OH, v=2"],
                           ["OH;v=2;#1", "OH, v=2"],
                           ["ONCN;v=0;", "ONCN"],
                           ["OO-17-O;v=0;", "O$^{17}$OO"],
                           ["OO-17-O;v=0;#1", "O$^{17}$OO"],
                           ["OO-18-O;v2=1;", "O$^{18}$OO, v$_2$=1"],
                           ["OO-18-O;v2=1;#1", "O$^{18}$OO, v$_2$=1"],
                           ["OO-18-O;v=0;", "O$^{18}$OO"],
                           ["OO-18-O;v=0;#1", "O$^{18}$OO"],
                           ["OO-18;v=0;", "O$^{18}$O"],
                           ["OSiS;v=0;", "OSiS"],
                           ["PH2;v=0;", "PH$_2$"],
                           ["PH2;v=0;ortho", "ortho-PH$_2$"],
                           ["PH2;v=0;para", "para-PH$_2$"],
                           ["PH3;v=0;", "PH$_3$"],
                           ["PH3;v=0;ortho", "ortho-PH$_3$"],
                           ["PH3;v=0;para", "para-PH$_3$"],
                           ["PH;v=0;", "PH"],
                           ["PN-15;v=0;", "P$^{15}$N"],
                           ["PN-15;v=1;", "P$^{15}$N, v=1"],
                           ["PN;v=0;", "PN"],
                           ["PN;v=0;hyp1", "PN"],
                           ["PN;v=1;", "PN, v=1"],
                           ["PN;v=2;", "PN, v=2"],
                           ["PN;v=3;", "PN, v=3"],
                           ["PN;v=4;", "PN, v=4"],
                           ["PN;v=5;", "PN, v=5"],
                           ["PNO;v=0;", "PNO"],
                           ["PO+;v=0;", "PO$^+$"],
                           ["PO+;v=0;#1", "PO$^+$"],
                           ["PO+;v=1;", "PO$^+$, v=1"],
                           ["PO+;v=1;#1", "PO$^+$, v=1"],
                           ["PO+;v=2;", "PO$^+$, v=2"],
                           ["PO+;v=2;#1", "PO$^+$, v=2"],
                           ["PO+;v=3;", "PO$^+$, v=3"],
                           ["PO+;v=3;#1", "PO$^+$, v=3"],
                           ["PO+;v=4;", "PO$^+$, v=4"],
                           ["PO+;v=4;#1", "PO$^+$, v=4"],
                           ["PO2;v=0;", "PO$_2$"],
                           ["PO2;v=0;#1", "PO$_2$"],
                           ["PO;v=0;hyp1", "PO"],
                           ["PO;v=1;hyp1", "PO, v=1"],
                           ["PO;v=2;hyp1", "PO, v=2"],
                           ["PO;v=3;hyp1", "PO, v=3"],
                           ["PO;v=4;hyp1", "PO, v=4"],
                           ["PO;v=5;hyp1", "PO, v=5"],
                           ["PS;v=1-0;", "PS, v=1-0"],
                           ["PS;v=1-0;#1", "PS, v=1-0"],
                           ["RRL-C", "RRL-C"],
                           ["RRL-H", "RRL-H"],
                           ["RRL-He", "RRL-He"],
                           ["RRL-N", "RRL-N"],
                           ["RRL-O", "RRL-O"],
                           ["RRL-S", "RRL-S"],
                           ["S-33-O2;v=0;", "$^{33}$SO$_2$"],
                           ["S-33-O2;v=0;#1", "$^{33}$SO$_2$"],
                           ["S-33-O2;v=0;#2", "$^{33}$SO$_2$"],
                           ["S-33-O;v=0;", "$^{33}$SO"],
                           ["S-34-H+;v=0;hyp1", "$^{34}$SH$^+$"],
                           ["S-34-O2;v=0;", "$^{34}$SO$_2$"],
                           ["S-34-O2;v=0;#1", "$^{34}$SO$_2$"],
                           ["S-34-O2;v=0;#2", "$^{34}$SO$_2$"],
                           ["S-34-O;v=0;", "$^{34}$SO"],
                           ["S-36-O;v=0;", "$^{36}$SO"],
                           ["S2;v=0;", "S$_2$"],
                           ["S2;v=0;#1", "S$_2$"],
                           ["S2O;v=0;", "S$_2$O"],
                           ["S3;v=0;", "S$_3$"],
                           ["S3;v=0;#1", "S$_3$"],
                           ["S4;v=0;", "S$_4$"],
                           ["S4;v=0;#1", "S$_4$"],
                           ["SD;v=0;", "SD"],
                           ["SD;v=0;#1", "SD"],
                           ["SH+;v=0;hyp1", "SH$^+$"],
                           ["SH-;v=0;", "SH$^-$"],
                           ["SH;v=0;", "SH"],
                           ["SH;v=0;#1", "SH"],
                           ["SH;v=0;hyp1", "SH"],
                           ["SH;v=1;", "SH, v=1"],
                           ["SH;v=1;#1", "SH, v=1"],
                           ["SH;v=1;hyp1", "SH, v=1"],
                           ["SO+;v=0;", "SO$^+$"],
                           ["SO+;v=0;#1", "SO$^+$"],
                           ["SO-16-O-18;v=0;", "SO$^{18}$O"],
                           ["SO-17;v=0;", "S$^{17}$O"],
                           ["SO-18;v=0;", "S$^{18}$O"],
                           ["SO2;(v1,v2,v3)=(0,1,0)-(0,0,0);", "SO$_2$, (v$_1$,v$_2$,v$_3$)=(0,1,0)-(0,0,0)"],
                           ["SO2;v2=1;", "SO$_2$, v$_2$=1"],
                           ["SO2;v=0;", "SO$_2$"],
                           ["SO;v=0;", "SO"],
                           ["SO;v=0;#2", "SO"],
                           ["SO;v=1;", "SO, v=1"],
                           ["SO;v=1;#2", "SO, v=1"],
                           ["SOO-17;v=0;", "SO$^{17}$O"],
                           ["SOO-18;v=0;", "SO$^{18}$O"],
                           ["SOO-18;v=0;#1", "SO$^{18}$O"],
                           ["ScCl-37;v=0;", "Sc$^{37}$Cl"],
                           ["ScCl;v=0;", "ScCl"],
                           ["ScF;v=0;", "ScF"],
                           ["ScO;v=0;", "ScO"],
                           ["ScO;v=0;hyp1", "ScO"],
                           ["ScS;v=0;", "ScS"],
                           ["Se-80-O2;v=0;", "$^{80}$SeO$_2$"],
                           ["Se-80-O2;v=0;#1", "$^{80}$SeO$_2$"],
                           ["Si-29-C4;v=0;", "$^{29}$SiC$_4$"],
                           ["Si-29-CC;v=0;", "$^{29}$SiCC"],
                           ["Si-29-H3CN;v=0;", "$^{29}$SiH$_3$CN"],
                           ["Si-29-H3CN;v=0;A", "$^{29}$SiH$_3$CN A"],
                           ["Si-29-H3CN;v=0;E", "$^{29}$SiH$_3$CN E"],
                           ["Si-29-O-17;v=0;", "$^{29}$Si$^{17}$O"],
                           ["Si-29-O-17;v=0;hyp2", "$^{29}$Si$^{17}$O"],
                           ["Si-29-O-17;v=1;", "$^{29}$Si$^{17}$O, v=1"],
                           ["Si-29-O-17;v=1;hyp2", "$^{29}$Si$^{17}$O, v=1"],
                           ["Si-29-O-17;v=2;", "$^{29}$Si$^{17}$O, v=2"],
                           ["Si-29-O-17;v=2;hyp2", "$^{29}$Si$^{17}$O, v=2"],
                           ["Si-29-O-18;v=0;", "$^{29}$Si$^{18}$O"],
                           ["Si-29-O-18;v=0;hyp1", "$^{29}$Si$^{18}$O"],
                           ["Si-29-O-18;v=1;", "$^{29}$Si$^{18}$O, v=1"],
                           ["Si-29-O-18;v=2;", "$^{29}$Si$^{18}$O, v=2"],
                           ["Si-29-O-18;v=3;", "$^{29}$Si$^{18}$O, v=3"],
                           ["Si-29-O;v=0;#1", "$^{29}$SiO"],
                           ["Si-29-O;v=0;hyp1", "$^{29}$SiO"],
                           ["Si-29-O;v=1;", "$^{29}$SiO, v=1"],
                           ["Si-29-O;v=2;", "$^{29}$SiO, v=2"],
                           ["Si-29-O;v=3;", "$^{29}$SiO, v=3"],
                           ["Si-29-O;v=4;", "$^{29}$SiO, v=4"],
                           ["Si-29-O;v=5;", "$^{29}$SiO, v=5"],
                           ["Si-29-O;v=6;", "$^{29}$SiO, v=6"],
                           ["Si-29-S-33;v=0;", "$^{29}$Si$^{33}$S"],
                           ["Si-29-S-34;v=0;", "$^{29}$Si$^{34}$S"],
                           ["Si-29-S-34;v=1;", "$^{29}$Si$^{34}$S, v=1"],
                           ["Si-29-S-36;v=0;", "$^{29}$Si$^{36}$S"],
                           ["Si-29-S;v=0;", "$^{29}$SiS"],
                           ["Si-29-S;v=1-0;", "$^{29}$SiS, v=1-0"],
                           ["Si-29-S;v=10;", "$^{29}$SiS, v=10"],
                           ["Si-29-S;v=11;", "$^{29}$SiS, v=11"],
                           ["Si-29-S;v=12;", "$^{29}$SiS, v=12"],
                           ["Si-29-S;v=1;", "$^{29}$SiS, v=1"],
                           ["Si-29-S;v=2;", "$^{29}$SiS, v=2"],
                           ["Si-29-S;v=3;", "$^{29}$SiS, v=3"],
                           ["Si-29-S;v=4;", "$^{29}$SiS, v=4"],
                           ["Si-29-S;v=5;", "$^{29}$SiS, v=5"],
                           ["Si-29-S;v=6;", "$^{29}$SiS, v=6"],
                           ["Si-29-S;v=7;", "$^{29}$SiS, v=7"],
                           ["Si-29-S;v=8;", "$^{29}$SiS, v=8"],
                           ["Si-29-S;v=9;", "$^{29}$SiS, v=9"],
                           ["Si-30-C4;v=0;", "$^{30}$SiC$_4$"],
                           ["Si-30-CC;v=0;", "$^{30}$SiCC"],
                           ["Si-30-H3CN;v=0;", "$^{30}$SiH$_3$CN"],
                           ["Si-30-H3CN;v=0;A", "$^{30}$SiH$_3$CN A"],
                           ["Si-30-H3CN;v=0;E", "$^{30}$SiH$_3$CN E"],
                           ["Si-30-O-17;v=0;", "$^{30}$Si$^{17}$O"],
                           ["Si-30-O-17;v=0;hyp1", "$^{30}$Si$^{17}$O"],
                           ["Si-30-O-17;v=1;", "$^{30}$Si$^{17}$O, v=1"],
                           ["Si-30-O-17;v=1;hyp1", "$^{30}$Si$^{17}$O, v=1"],
                           ["Si-30-O-17;v=2;", "$^{30}$Si$^{17}$O, v=2"],
                           ["Si-30-O-17;v=2;hyp1", "$^{30}$Si$^{17}$O, v=2"],
                           ["Si-30-O-18;v=0;", "$^{30}$Si$^{18}$O"],
                           ["Si-30-O-18;v=1,v=1;", "$^{30}$Si$^{18}$O, v=1,v=1"],
                           ["Si-30-O-18;v=2,v=2;", "$^{30}$Si$^{18}$O, v=2,v=2"],
                           ["Si-30-O-18;v=3,v=3;", "$^{30}$Si$^{18}$O, v=3,v=3"],
                           ["Si-30-O;v=0;#1", "$^{30}$SiO"],
                           ["Si-30-O;v=0;hyp1", "$^{30}$SiO"],
                           ["Si-30-O;v=1;", "$^{30}$SiO, v=1"],
                           ["Si-30-O;v=2;", "$^{30}$SiO, v=2"],
                           ["Si-30-O;v=3;", "$^{30}$SiO, v=3"],
                           ["Si-30-O;v=4;", "$^{30}$SiO, v=4"],
                           ["Si-30-O;v=5;", "$^{30}$SiO, v=5"],
                           ["Si-30-O;v=6;", "$^{30}$SiO, v=6"],
                           ["Si-30-S-33;v=0;", "$^{30}$Si$^{33}$S"],
                           ["Si-30-S-34;v=0;", "$^{30}$Si$^{34}$S"],
                           ["Si-30-S-34;v=1;", "$^{30}$Si$^{34}$S, v=1"],
                           ["Si-30-S-36;v=0;", "$^{30}$Si$^{36}$S"],
                           ["Si-30-S;v=0;", "$^{30}$SiS"],
                           ["Si-30-S;v=1-0;", "$^{30}$SiS, v=1-0"],
                           ["Si-30-S;v=10;", "$^{30}$SiS, v=10"],
                           ["Si-30-S;v=11;", "$^{30}$SiS, v=11"],
                           ["Si-30-S;v=12;", "$^{30}$SiS, v=12"],
                           ["Si-30-S;v=1;", "$^{30}$SiS, v=1"],
                           ["Si-30-S;v=2;", "$^{30}$SiS, v=2"],
                           ["Si-30-S;v=3;", "$^{30}$SiS, v=3"],
                           ["Si-30-S;v=4;", "$^{30}$SiS, v=4"],
                           ["Si-30-S;v=5;", "$^{30}$SiS, v=5"],
                           ["Si-30-S;v=6;", "$^{30}$SiS, v=6"],
                           ["Si-30-S;v=7;", "$^{30}$SiS, v=7"],
                           ["Si-30-S;v=8;", "$^{30}$SiS, v=8"],
                           ["Si-30-S;v=9;", "$^{30}$SiS, v=9"],
                           ["Si2C;v=0;", "Si$_2$C"],
                           ["Si2H2;v=0;", "Si$_2$H$_2$"],
                           ["Si2H2;v=0;#1", "Si$_2$H$_2$"],
                           ["Si2H2;v=0;ortho", "ortho-Si$_2$H$_2$"],
                           ["Si2H2;v=0;para", "para-Si$_2$H$_2$"],
                           ["SiC-13-C3;v=0;", "Si$^{13}$CC$_3$"],
                           ["SiC-13-C;v=0;", "Si$^{13}$CC"],
                           ["SiC-13;v=0;", "Si$^{13}$C"],
                           ["SiC-13;v=0;#1", "Si$^{13}$C"],
                           ["SiC2C-13-C;v=0;", "SiC$_2 \\! ^{13}$CC"],
                           ["SiC2N;v=0;hyp1", "SiC$_2$N"],
                           ["SiC3C-13;v=0;", "SiC$_3 \\! ^{13}$C"],
                           ["SiC3H;v=0;hyp1", "SiC$_3$H"],
                           ["SiC4;v=0;", "SiC$_4$"],
                           ["SiC5;v=0;", "SiC$_5$"],
                           ["SiC6;v=0;", "SiC$_6$"],
                           ["SiC;v=0;", "SiC"],
                           ["SiC;v=1;", "SiC, v=1"],
                           ["SiC;v=1;#1", "SiC, v=1"],
                           ["SiCC-13-C2;v=0;", "SiC$^{13}$CC$_2$"],
                           ["SiCC;v3=1;", "SiCC, v$_3$=1"],
                           ["SiCC;v3=2;", "SiCC, v$_3}$=2"],
                           ["SiCC;v=0;", "SiCC"],
                           ["SiCC;v=0;#1", "SiCC"],
                           ["SiCC;v=0;#2", "SiCC"],
                           ["SiCCH;v=0;", "SiCCH"],
                           ["SiCN;v=0;hyp1", "SiCN"],
                           ["SiH+;v=0;", "SiH$^+$"],
                           ["SiH+;v=1-0;", "SiH$^+$, v=1-0"],
                           ["SiH3C-13-N;v=0;", "SiH$_3 \\! ^{13}$CN"],
                           ["SiH3C-13-N;v=0;A", "SiH$_3 \\! ^{13}$CN A"],
                           ["SiH3C-13-N;v=0;E", "SiH$_3 \\! ^{13}$CN E"],
                           ["SiH3CCH;v=0;", "SiH$_3$CCH"],
                           ["SiH3CCH;v=0;ortho", "ortho-SiH$_3$CCH"],
                           ["SiH3CCH;v=0;para", "para-SiH$_3$CCH"],
                           ["SiH3CN;v=0;", "SiH$_3$CN"],
                           ["SiH3CN;v=0;A", "SiH$_3$CN A"],
                           ["SiH3CN;v=0;E", "SiH$_3$CN E"],
                           ["SiH;v=0;", "SiH"],
                           ["SiN;v=0;", "SiN"],
                           ["SiNC;v=0;", "SiNC"],
                           ["SiO-17;v=0;", "Si$^{17}$O"],
                           ["SiO-17;v=0;hyp1", "Si$^{17}$O"],
                           ["SiO-17;v=1;", "Si$^{17}$O, v=1"],
                           ["SiO-17;v=1;hyp1", "Si$^{17}$O, v=1"],
                           ["SiO-17;v=2;", "Si$^{17}$O, v=2"],
                           ["SiO-17;v=2;hyp1", "Si$^{17}$O, v=2"],
                           ["SiO-17;v=3;", "Si$^{17}$O, v=3"],
                           ["SiO-17;v=3;hyp1", "Si$^{17}$O, v=3"],
                           ["SiO-17;v=4;", "Si$^{17}$O, v=4"],
                           ["SiO-17;v=4;hyp1", "Si$^{17}$O, v=4"],
                           ["SiO-18;v=0;", "Si$^{18}$O"],
                           ["SiO-18;v=1;", "Si$^{18}$O, v=1"],
                           ["SiO-18;v=2;", "Si$^{18}$O, v=2"],
                           ["SiO-18;v=3;", "Si$^{18}$O, v=3"],
                           ["SiO-18;v=4;", "Si$^{18}$O, v=4"],
                           ["SiO-18;v=5;", "Si$^{18}$O, v=5"],
                           ["SiO;v=0;#1", "SiO"],
                           ["SiO;v=10;", "SiO, v=10"],
                           ["SiO;v=1;", "SiO, v=1"],
                           ["SiO;v=2;", "SiO, v=2"],
                           ["SiO;v=3;", "SiO, v=3"],
                           ["SiO;v=4;", "SiO, v=4"],
                           ["SiO;v=5;", "SiO, v=5"],
                           ["SiO;v=6;", "SiO, v=6"],
                           ["SiO;v=7;", "SiO, v=7"],
                           ["SiO;v=8;", "SiO, v=8"],
                           ["SiO;v=9;", "SiO, v=9"],
                           ["SiS-33;v=0;", "Si$^{33}$S"],
                           ["SiS-33;v=0;hyp1", "Si$^{33}$S"],
                           ["SiS-33;v=1;", "Si$^{33}$S, v=1"],
                           ["SiS-33;v=1;hyp1", "Si$^{33}$S, v=1"],
                           ["SiS-33;v=2;", "Si$^{33}$S, v=2"],
                           ["SiS-33;v=2;hyp1", "Si$^{33}$S, v=2"],
                           ["SiS-33;v=3;", "Si$^{33}$S, v=3"],
                           ["SiS-33;v=3;hyp1", "Si$^{33}$S, v=3"],
                           ["SiS-33;v=4;", "Si$^{33}$S, v=4"],
                           ["SiS-33;v=4;hyp1", "Si$^{33}$S, v=4"],
                           ["SiS-33;v=5;", "Si$^{33}$S, v=5"],
                           ["SiS-33;v=5;hyp1", "Si$^{33}$S, v=5"],
                           ["SiS-33;v=6;", "Si$^{33}$S, v=6"],
                           ["SiS-33;v=6;hyp1", "Si$^{33}$S, v=6"],
                           ["SiS-33;v=7;", "Si$^{33}$S, v=7"],
                           ["SiS-33;v=7;hyp1", "Si$^{33}$S, v=7"],
                           ["SiS-33;v=8;", "Si$^{33}$S, v=8"],
                           ["SiS-33;v=8;hyp1", "Si$^{33}$S, v=8"],
                           ["SiS-33;v=9;", "Si$^{33}$S, v=9"],
                           ["SiS-33;v=9;hyp1", "Si$^{33}$S, v=9"],
                           ["SiS-34;v=0;", "Si$^{34}$S"],
                           ["SiS-34;v=1-0;", "Si$^{34}$S, v=1-0"],
                           ["SiS-34;v=10;", "Si$^{34}$S, v=10"],
                           ["SiS-34;v=11;", "Si$^{34}$S, v=11"],
                           ["SiS-34;v=12;", "Si$^{34}$S, v=12"],
                           ["SiS-34;v=1;", "Si$^{34}$S, v=1"],
                           ["SiS-34;v=2;", "Si$^{34}$S, v=2"],
                           ["SiS-34;v=3;", "Si$^{34}$S, v=3"],
                           ["SiS-34;v=4;", "Si$^{34}$S, v=4"],
                           ["SiS-34;v=5;", "Si$^{34}$S, v=5"],
                           ["SiS-34;v=6;", "Si$^{34}$S, v=6"],
                           ["SiS-34;v=7;", "Si$^{34}$S, v=7"],
                           ["SiS-34;v=8;", "Si$^{34}$S, v=8"],
                           ["SiS-34;v=9;", "Si$^{34}$S, v=9"],
                           ["SiS-36;v=0;", "Si$^{36}$S"],
                           ["SiS;v=0;", "SiS"],
                           ["SiS;v=1-0;", "SiS, v=1-0"],
                           ["SiS;v=10;", "SiS, v=10"],
                           ["SiS;v=11;", "SiS, v=11"],
                           ["SiS;v=12;", "SiS, v=12"],
                           ["SiS;v=13;", "SiS, v=13"],
                           ["SiS;v=14;", "SiS, v=14"],
                           ["SiS;v=15;", "SiS, v=15"],
                           ["SiS;v=16;", "SiS, v=16"],
                           ["SiS;v=17;", "SiS, v=17"],
                           ["SiS;v=18;", "SiS, v=18"],
                           ["SiS;v=19;", "SiS, v=19"],
                           ["SiS;v=1;", "SiS, v=1"],
                           ["SiS;v=2-0;", "SiS, v=2-0"],
                           ["SiS;v=2-1;", "SiS, v=2-1"],
                           ["SiS;v=20;", "SiS, v=20"],
                           ["SiS;v=2;", "SiS, v=2"],
                           ["SiS;v=3;", "SiS, v=3"],
                           ["SiS;v=4;", "SiS, v=4"],
                           ["SiS;v=5;", "SiS, v=5"],
                           ["SiS;v=6;", "SiS, v=6"],
                           ["SiS;v=7;", "SiS, v=7"],
                           ["SiS;v=8;", "SiS, v=8"],
                           ["SiS;v=9;", "SiS, v=9"],
                           ["Sr-88-D;v=0;", "$^{88}$SrD"],
                           ["Sr-88-D;v=0;#1", "$^{88}$SrD"],
                           ["Sr-88-D;v=1;", "$^{88}$SrD, v=1"],
                           ["Sr-88-D;v=1;#1", "$^{88}$SrD, v=1"],
                           ["Sr-88-H;v=0;", "$^{88}$SrH"],
                           ["Sr-88-H;v=0;#1", "$^{88}$SrH"],
                           ["Sr-88-H;v=1;", "$^{88}$SrH, v=1"],
                           ["Sr-88-H;v=1;#1", "$^{88}$SrH, v=1"],
                           ["Ti-46-O2;v=0;", "$^{46}$TiO$_2$"],
                           ["Ti-46-S;v=0;", "$^{46}$TiS"],
                           ["Ti-48-F;v=0;hyp1", "$^{48}$TiF"],
                           ["Ti-48-N;v=0;", "$^{48}$TiN"],
                           ["Ti-48-O;v=0;", "$^{48}$TiO"],
                           ["Ti-48-S;v=0;", "$^{48}$TiS"],
                           ["Ti-50-O2;v=0;", "$^{50}$TiO$_2$"],
                           ["Ti-50-S;v=0;", "$^{50}$TiS"],
                           ["TiO2;v=0;", "TiO$_2$"],
                           ["YO;v=0;", "YO"],
                           ["YS;v=0;", "YS"],
                           ["Z-HNCHCN;v=0;", "Z-HNCHCN"],
                           ["Zn-64-H;v=0;", "$^{64}$ZnH"],
                           ["Zn-66-H;v=0;", "$^{66}$ZnH"],
                           ["Zn-68-H;v=0;", "$^{68}$ZnH"],
                           ["ZnO;v=0;", "ZnO"],
                           ["ZnO;v=0;#1", "ZnO"],
                           ["ZnO;v=1;", "ZnO, v=1"],
                           ["ZnO;v=1;#1", "ZnO, v=1"],
                           ["ZnO;v=2;", "ZnO, v=2"],
                           ["ZnO;v=2;#1", "ZnO, v=2"],
                           ["a'GG'g-CH2OHCH2CH2OH;v=0;", "a'GG'g-CH$_2$OHCH$_2$CH$_2$OH"],
                           ["a-(CH3)2CHCH2CN;v=0;", "a-(CH$_3$)$_2$CHCH$_2$CN"],
                           ["a-C-13-H3CH2OH;v=0;", "a-$^{13}$CH$_3$CH$_2$OH"],
                           ["a-C2H5SH;v=0;", "a-C$_2$H$_5$SH"],
                           ["a-CH3C-13-H2OH;v=0;", "a-CH$_3 \\! ^{13}$CH$_2$OH"],
                           ["a-CH3CH2OD;v=0;", "a-CH$_3$CH$_2$OD"],
                           ["a-CH3CHDOH;v=0;", "a-CH$_3$CHDOH"],
                           ["a-H2C=CHOH;v=0;", "a-H$_2$C=CHOH"],
                           ["a-H2CCHSH;v=0;", "a-H$_2$CCHSH"],
                           ["a-a-CH2DCH2OH;v=0;", "a-a-CH$_2$DCH$_2$OH"],
                           ["a-i-C3H7OH;v=0;", "a-i-C$_3$H$_7$OH"],
                           ["aGg'-(CH2OH)2;v=0;", "aGg'-(CH$_2$OH)2"],
                           ["aGg'-CH3CHOHCH2OH;v=0;", "aGg'-CH$_3$CHOHCH$_2$OH"],
                           ["aa-(C2H5)2O;v=0;", "aa-(C$_2$H$_5$)2O"],
                           ["ag-(C2H5)2O;v=0;", "ag-(C$_2$H$_5$)2O"],
                           ["c-C-13-C2H2;v=0;", "c-$^{13}$CC$_2$H$_2$"],
                           ["c-C-13-C2H2;v=0;hyp1", "c-$^{13}$CC$_2$H$_2$"],
                           ["c-C-13-C2H2;v=0;ortho", "ortho-c-$^{13}$CC$_2$H$_2$"],
                           ["c-C-13-C2H2;v=0;para", "para-c-$^{13}$CC$_2$H$_2$"],
                           ["c-C-13-C2HD;v=0;", "c-$^{13}$CC$_2$HD"],
                           ["c-C-13-C2HD;v=0;hyp2", "c-$^{13}$CC$_2$HD"],
                           ["c-C-13-CCH;v=0;", "c-$^{13}$CCCH"],
                           ["c-C-13-CCH;v=0;#1", "c-$^{13}$CCCH"],
                           ["c-C2H3(O)CN;v=0;", "c-C$_2$H$_3$(O)CN"],
                           ["c-C2H4NH;v=0;", "c-C$_2$H$_4$NH"],
                           ["c-C2H4O-18;v=0;", "c-C$_2$H$_4 \\! ^{18}$O"],
                           ["c-C2H4O-18;v=0;ortho", "ortho-c-C$_2$H$_4 \\! ^{18}$O"],
                           ["c-C2H4O-18;v=0;para", "para-c-C$_2$H$_4 \\! ^{18}$O"],
                           ["c-C2H4O;v=0;", "c-C$_2$H$_4$O"],
                           ["c-C2H4O;v=0;ortho", "ortho-c-C$_2$H$_4$O"],
                           ["c-C2H4O;v=0;para", "para-c-C$_2$H$_4$O"],
                           ["c-C2H4S;v=0;", "c-C$_2$H$_4$S"],
                           ["c-C2H4S;v=0;ortho", "ortho-c-C$_2$H$_4$S"],
                           ["c-C2H4S;v=0;para", "para-c-C$_2$H$_4$S"],
                           ["c-C3D2;v=0;", "c-C$_3$D$_2$"],
                           ["c-C3D2;v=0;hyp2", "c-C$_3$D$_2$"],
                           ["c-C3D2;v=0;ortho", "ortho-c-C$_3$D$_2$"],
                           ["c-C3D2;v=0;para", "para-c-C$_3$D$_2$"],
                           ["c-C3D;v=0;", "c-C$_3$D"],
                           ["c-C3D;v=0;#1", "c-C$_3$D"],
                           ["c-C3H2;v=0;", "c-C$_3$H$_2$"],
                           ["c-C3H2;v=0;ortho", "ortho-c-C$_3$H$_2$"],
                           ["c-C3H2;v=0;para", "para-c-C$_3$H$_2$"],
                           ["c-C3H2;v2=1;", "c-C$_3$H$_2$, v$_2$=1"],
                           ["c-C3H2;v3=1;", "c-C$_3$H$_2$, v$_3$=1"],
                           ["c-C3H2;v5=1;", "c-C$_3$H$_2$, v$_5$=1"],
                           ["c-C3H2;v6=1;", "c-C$_3$H$_2$, v$_6$=1"],
                           ["c-C3H4;v=0;", "c-C$_3$H$_4$"],
                           ["c-C3H4;v=0;ortho", "ortho-c-C$_3$H$_4$"],
                           ["c-C3H4;v=0;para", "para-c-C$_3$H$_4$"],
                           ["c-C3H5CN;v=0;", "c-C$_3$H$_5$CN"],
                           ["c-C3H6O;v=0;", "c-C$_3$H$_6$O"],
                           ["c-C3H6O;v=0;ortho", "ortho-c-C$_3$H$_6$O"],
                           ["c-C3H6O;v=0;para", "para-c-C$_3$H$_6$O"],
                           ["c-C3H;v=0;", "c-C$_3$H"],
                           ["c-C3H;v=0;#1", "c-C$_3$H"],
                           ["c-C3HCN;v=0;", "c-C$_3$HCN"],
                           ["c-C3HD;v=0;", "c-C$_3$HD"],
                           ["c-C3HD;v=0;hyp1", "c-C$_3$HD"],
                           ["c-C4H4NH;v=0;", "c-C$_4$H$_4$NH"],
                           ["c-C4H4NH;v=0;ortho", "ortho-c-C$_4$H$_4$NH"],
                           ["c-C4H4NH;v=0;para", "para-c-C$_4$H$_4$NH"],
                           ["c-C4H4O2;v=0;", "c-C$_4$H$_4$O$_2$"],
                           ["c-C4H4O;v=0;", "c-C$_4$H$_4$O"],
                           ["c-C4H4O;v=0;ortho", "ortho-c-C$_4$H$_4$O"],
                           ["c-C4H4O;v=0;para", "para-c-C$_4$H$_4$O"],
                           ["c-C5H5N;v=0;", "c-C$_5$H$_5$N"],
                           ["c-C5H5N;v=0;ortho", "ortho-c-C$_5$H$_5$N"],
                           ["c-C5H5N;v=0;para", "para-c-C$_5$H$_5$N"],
                           ["c-C6H4;v=0;", "c-C$_6$H$_4$"],
                           ["c-C6H4;v=0;ortho", "ortho-c-C$_6$H$_4$"],
                           ["c-C6H4;v=0;para", "para-c-C$_6$H$_4$"],
                           ["c-C6H5CCH;v=0;", "c-C$_6$H$_5$CCH"],
                           ["c-C6H5CCH;v=0;ortho", "ortho-c-C$_6$H$_5$CCH"],
                           ["c-C6H5CCH;v=0;para", "para-c-C$_6$H$_5$CCH"],
                           ["c-C6H5CHO;v=0;", "c-C$_6$H$_5$CHO"],
                           ["c-C6H5CN;v=0;", "c-C$_6$H$_5$CN"],
                           ["c-C6H5CN;v=0;ortho", "ortho-c-C$_6$H$_5$CN"],
                           ["c-C6H5CN;v=0;para", "para-c-C$_6$H$_5$CN"],
                           ["c-C6H5OCH3;v=0;", "c-C$_6$H$_5$OCH$_3$"],
                           ["c-C6H5OH;v=0;", "c-C$_6$H$_5$OH"],
                           ["c-CC-13-CHD;v=0;", "c-C$^{13}$CCHD"],
                           ["c-CC-13-CHD;v=0;hyp2", "c-C$^{13}$CCHD"],
                           ["c-CC-13-H4O;v=0;", "c-C$^{13}$CH$_4$O"],
                           ["c-CCC-13-H2;v=0;", "c-CC$^{13}$CH$_2$"],
                           ["c-CCC-13-H2;v=0;hyp1", "c-CC$^{13}$CH$_2$"],
                           ["c-CCC-13-H;v=0;", "c-CC$^{13}$CH"],
                           ["c-CCC-13-H;v=0;#1", "c-CC$^{13}$CH"],
                           ["c-CCC-13-HD;v=0;", "c-CC$^{13}$CHD"],
                           ["c-CCC-13-HD;v=0;hyp2", "c-CC$^{13}$CHD"],
                           ["c-H2C2Si;v=0;", "c-H$_2$C$_2$Si"],
                           ["c-H2C2Si;v=0;ortho", "ortho-c-H$_2$C$_2$Si"],
                           ["c-H2C2Si;v=0;para", "para-c-H$_2$C$_2$Si"],
                           ["c-H2C3O;v=0;", "c-H$_2$C$_3$O"],
                           ["c-H2C3O;v=0;ortho", "ortho-c-H$_2$C$_3$O"],
                           ["c-H2C3O;v=0;para", "para-c-H$_2$C$_3$O"],
                           ["c-HC(O)SH;v=0;", "c-HC(O)SH"],
                           ["c-HC(S)SH;v=0;", "c-HC(S)SH"],
                           ["c-HCOOH;v=0;", "c-HCOOH"],
                           ["cis-DCOOH;v=0;", "cis-DCOOH"],
                           ["cis-DCOOH;v=0;#1", "cis-DCOOH"],
                           ["cis-HC(O)NHD;v=0;", "cis-HC(O)NHD"],
                           ["cis-HC(O)NHD;v=0;hyp1", "cis-HC(O)NHD"],
                           ["cis-HC-13-OOH;v=0;", "cis-H$^{13}$COOH"],
                           ["cis-HC-13-OOH;v=0;#1", "cis-H$^{13}$COOH"],
                           ["cis-HCOOD;v=0;", "cis-HCOOD"],
                           ["cis-HCOOD;v=0;#1", "cis-HCOOD"],
                           ["cis-HOCO;v=0;", "cis-HOCO"],
                           ["cis-HOSO+;v=0;", "cis-HOSO$^+$"],
                           ["g'Ga-CH3CHOHCH2OH;v=0;", "g'Ga-CH$_3$CHOHCH$_2$OH"],
                           ["g'Gg-CH3CHOHCH2OH;v=0;", "g'Gg-CH$_3$CHOHCH$_2$OH"],
                           ["g-(CH3)2CHCH2CN;v=0;", "g-(CH$_3$)2CHCH$_2$CN"],
                           ["g-C2H5CHO;v=0;", "g-C$_2$H$_5$CHO"],
                           ["g-C2H5S-34-H;v=0;", "g-C$_2$H$_5 \\! ^{34}$SH"],
                           ["g-C2H5SH;v=0;", "g-C$_2$H$_5$SH"],
                           ["g-i-C3H7OH;v=0;", "g-i-C$_3$H$_7$OH"],
                           ["gG'g'-CH3CHOHCH2OH;v=0;", "gG'g'-CH$_3$CHOHCH$_2$OH"],
                           ["gGg'-(CH2OH)2;v=0;", "gGg'-(CH$_2$OH)2"],
                           ["i-C3H7C-13-N;v=0;", "i-C$_3$H$_7 \\! ^{13}$CN"],
                           ["i-C3H7CN;v=0;", "i-C$_3$H$_7$CN"],
                           ["i-C3H7CN;v=0;#1", "i-C$_3$H$_7$CN"],
                           ["i-C3H7CN;v=0;hyp1", "i-C$_3$H$_7$CN"],
                           ["l-C-13-C2H2;v=0;", "l-$^{13}$CC$_2$H$_2$"],
                           ["l-C-13-C2H2;v=0;ortho", "ortho-l-$^{13}$CC$_2$H$_2$"],
                           ["l-C-13-C2H2;v=0;para", "para-l-$^{13}$CC$_2$H$_2$"],
                           ["l-C-13-C3H2;v=0;", "l-$^{13}$CC$_3$H$_2$"],
                           ["l-C-13-C3H2;v=0;ortho", "ortho-l-$^{13}$CC$_3$H$_2$"],
                           ["l-C-13-C3H2;v=0;para", "para-l-$^{13}$CC$_3$H$_2$"],
                           ["l-C2C-13-CH2;v=0;", "l-C$_2 \\! ^{13}$CCH$_2$"],
                           ["l-C2C-13-CH2;v=0;ortho", "ortho-l-C$_2 \\! ^{13}$CCH$_2$"],
                           ["l-C2C-13-CH2;v=0;para", "para-l-C$_2 \\! ^{13}$CCH$_2$"],
                           ["l-C2C-13-H2;v=0;", "l-C$_2 \\! ^{13}$CH$_2$"],
                           ["l-C2C-13-H2;v=0;ortho", "ortho-l-C$_2 \\! ^{13}$CH$_2$"],
                           ["l-C2C-13-H2;v=0;para", "para-l-C$_2 \\! ^{13}$CH$_2$"],
                           ["l-C3C-13-H2;v=0;", "l-C$_3 \\! ^{13}$CH$_2$"],
                           ["l-C3C-13-H2;v=0;ortho", "ortho-l-C$_3 \\! ^{13}$CH$_2$"],
                           ["l-C3C-13-H2;v=0;para", "para-l-C$_3 \\! ^{13}$CH$_2$"],
                           ["l-C3D2;v=0;", "l-C$_3$D$_2$"],
                           ["l-C3D2;v=0;ortho", "ortho-l-C$_3$D$_2$"],
                           ["l-C3D2;v=0;para", "para-l-C$_3$D$_2$"],
                           ["l-C3H2;v=0;", "l-C$_3$H$_2$"],
                           ["l-C3H2;v=0;ortho", "ortho-l-C$_3$H$_2$"],
                           ["l-C3H2;v=0;para", "para-l-C$_3$H$_2$"],
                           ["l-C3HD;v=0;", "l-C$_3$HD"],
                           ["l-C4H2;v=0;", "l-C$_4$H$_2$"],
                           ["l-C4H2;v=0;ortho", "ortho-l-C$_4$H$_2$"],
                           ["l-C4H2;v=0;para", "para-l-C$_4$H$_2$"],
                           ["l-C4HD;v=0;", "l-C$_4$HD"],
                           ["l-C5H2;v=0;", "l-C$_5$H$_2$"],
                           ["l-C5H2;v=0;ortho", "ortho-l-C$_5$H$_2$"],
                           ["l-C5H2;v=0;para", "para-l-C$_5$H$_2$"],
                           ["l-C6H2;v=0;", "l-C$_6$H$_2$"],
                           ["l-C6H2;v=0;ortho", "ortho-l-C$_6$H$_2$"],
                           ["l-C6H2;v=0;para", "para-l-C$_6$H$_2$"],
                           ["l-C7H2;v=0;", "l-C$_7$H$_2$"],
                           ["l-C7H2;v=0;ortho", "ortho-l-C$_7$H$_2$"],
                           ["l-C7H2;v=0;para", "para-l-C$_7$H$_2$"],
                           ["l-CC-13-C2H2;v=0;", "l-C$^{13}$CC$_2$H$_2$"],
                           ["l-CC-13-C2H2;v=0;ortho", "ortho-l-C$^{13}$CC$_2$H$_2$"],
                           ["l-CC-13-C2H2;v=0;para", "para-l-C$^{13}$CC$_2$H$_2$"],
                           ["l-CC-13-CH2;v=0;", "l-C$^{13}$CCH$_2$"],
                           ["l-CC-13-CH2;v=0;ortho", "ortho-l-C$^{13}$CCH$_2$"],
                           ["l-CC-13-CH2;v=0;para", "para-l-C$^{13}$CCH$_2$"],
                           ["l-H2C3O;v=0;", "l-H$_2$C$_3$O"],
                           ["l-H2C3O;v=0;ortho", "ortho-l-H$_2$C$_3$O"],
                           ["l-H2C3O;v=0;para", "para-l-H$_2$C$_3$O"],
                           ["l-HC4N;v=0;", "l-HC$_4$N"],
                           ["l-HC6N;v=0;hyp1", "l-HC$_6$N"],
                           ["l-HC6N;v=0;hyp2", "l-HC$_6$N"],
                           ["l-SiCCC;v=0;", "l-SiCCC"],
                           ["o-c-SiCCC;v=0;", "o-c-SiCCC"],
                           ["p-c-SiC3;v=0;", "p-c-SiC$_3$"],
                           ["s-C2H5CHO;v23=1;", "s-C$_2$H$_5$CHO, v$_{23}$=1"],
                           ["s-C2H5CHO;v24=1;", "s-C$_2$H$_5$CHO, v$_{24}$=1"],
                           ["s-C2H5CHO;v=0;", "s-C$_2$H$_5$CHO"],
                           ["s-H2C=CHOH;v=0;", "s-H$_2$C=CHOH"],
                           ["s-H2CCHSH;v=0;", "s-H$_2$CCHSH"],
                           ["s-a-CH2DCH2OH;v=0;", "s-a-CH$_2$DCH$_2$OH"],
                           ["s-cis-H2CCHCOOH;v=0;", "s-cis-H$_2$CCHCOOH"],
                           ["s-trans-H2CCHCOOH;v=0;", "s-trans-H$_2$CCHCOOH"],
                           ["t-C4H9CN;v=0;", "t-C$_4$H$_9$CN"],
                           ["t-HC(O)SH;v=0;", "t-HC(O)SH"],
                           ["t-HC(S)SH;v=0;", "t-HC(S)SH"],
                           ["t-HC-13-OOH;v=0;", "t-H$^{13}$COOH"],
                           ["t-HC3O;v=0;", "t-HC$_3$O"],
                           ["t-HCOOH;v=0;", "t-HCOOH"],
                           ["trans-HC(O)NHD;v=0;", "trans-HC(O)NHD"],
                           ["trans-HC(O)NHD;v=0;hyp1",  "trans-HC(O)NHD"]]


    ## convert translation table to numpy array
    TranslationNameList = numpy.asarray(TranslationNameList)


    ## get latex version of molecule name
    i, j = numpy.where(TranslationNameList == DBMoleculeName)
    if (len(i) > 0):
        LatexMoleculeName = TranslationNameList[i[0], 1]
    else:
        LatexMoleculeName = DBMoleculeName

    # Debug:
    # print ("LatexMoleculeName = ", LatexMoleculeName)


    ## define return variables
    return LatexMoleculeName
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## This function reads in entries from the sqlite3 database
##
def ListDatabaseCore(FreqMin = 1.0, FreqMax = 1.e8, ElowMin = 0.0, ElowMax = 1.e6, SelectMolecule = [], OutputDevice = "", dbFilename = None):
    """

input parameters:
-----------------


    - FreqMin:              minimum frequency (in MHz) (default: 0) for transitions in the table "Transitions"

    - FreqMax:              maximum frequency (in MHz) (default: 1.e8) for transitions in the table "Transitions"

    - ElowMin:              minimum for the lower energy (in K) (default: 0) in the "Transitions" table

    - ElowMax:              maximum for the lower energy (in K) (default: 1.e6) in the "Transitions" table

    - SelectMolecule:       a (python) list containing the names of all molecules which should be considered
                            or a file name including the molecules which should be selected.

                            Note, if the parameter defines a relative path, this path has to be defined
                            relative to the current working directory!

    - OutputDevice:         path and name of the file where the output is written to. If no file name is
                            defined (None), the contents of the database is written to the screen (default: None).

                            Note, if the parameter defines a relative path, this path has to be defined
                            relative to the current working directory!

    - dbFilename:           path and name of used database file (default: None)


output parameters:
------------------

    - Contents:             contents of the database (as an python array, e.g. contents[0] contains
                            the entries for the first molecule within the frequency/energy range)

                            Note, the user is free to define a different name for the output parameter.
    """

    # Debug:
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("ElowMin = ", ElowMin)
    # print ("ElowMax = ", ElowMax)
    # print ("SelectMolecule = ", SelectMolecule)
    # print ("OutputDevice = ", OutputDevice)
    # print ("dbFilename = ", dbFilename)


    ##====================================================================================================================================================
    ## check input parameters
    printFlag = "true"
    OutputDevice = OutputDevice.strip()
    if (OutputDevice == "quiet"):
        printFlag = "false"

    # Debug:
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("SelectMolecule = ", SelectMolecule)


    ## test, if SelectMolecule is a list, otherwise try to use SelectMolecule as filename
    if (printFlag != "false"):
        print (" ")
        print (" ")
        print ("Analyze selected molecules ..", flush = True)
    ListOfSelectedMolecules = []
    if (len(SelectMolecule) > 0 and SelectMolecule != ["None"]):
        for entry in SelectMolecule:                                                        ## loop over all entries of SelectMolecule list
            entryStrip = entry.strip()                                                      ## remove leading and trailing blanks


            ## check, if current entry is a file
            if (os.path.isfile(entryStrip)):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## if file is a molfit file, get names of molecules from molfit file
                if (entryStrip.endswith("molfit")):

                    # Debug:
                    # print ("Selection string is a molfit file!")


                    ## read in contents of molfit file
                    f = open(entryStrip)
                    MolfitContents = f.readlines()                                          ## read in contents of molfit file
                    f.close()


                    ## get names of molecules
                    for lines in MolfitContents:                                            ## loop over all lines of molfit file
                        CurrentLine = lines.strip()                                         ## remove leading and trailing blanks

                        # Debug:
                        # print ("CurrentLine = ", CurrentLine)

                        ## check for comments
                        w = CurrentLine.find("%")                                           ## are there comments in the current line ?
                        if (w == 0):                                                        ## ignore lines which contains only comments
                            CurrentLine = ""                                                ## if line is only a comment clear line
                        elif (w > 0):                                                       ## is there a comment in the current line ?
                            CurrentLine = CurrentLine[:w]                                   ## remove comments
                        CurrentLine = CurrentLine.strip()

                        # Debug:
                        # print ("CurrentLine = ", CurrentLine)


                        ## analyze, if the current line contains the name of a molecule
                        if (CurrentLine != ""):                                             ## ignore empty lines
                            SplitLines = CurrentLine.split()                                ## split current line into columns

                            # Debug:
                            # print ("SplitLines = ", SplitLines)


                            ## contains the last column an integer number?
                            helpstring = SplitLines[-1].strip()
                            is_int = "false"
                            if (helpstring.isdigit()):                                      ## continue, if string is an integer number
                                is_int = "true"


                            ## If yes, then the current line contains the name of a molecule
                            if (is_int == "true"):                                          ## if last entry in a line is an integer number, then
                                MolName = ""                                                ## the current line contains the name of a molecule
                                for col in SplitLines[:-1]:                                 ## get whole line without integer number at the end
                                    MolName += col                                          ## construct entry for SelectMolecule

                                # Debug:
                                # print ("MolName = ", MolName)

                                ListOfSelectedMolecules.append(MolName)                     ## construct final array containing all molecule names


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## read names of molecules from ASCII file
                else:

                    # Debug:
                    # print ("Selection string is a ASCII file!")


                    ## read in whole ASCII file
                    f = open(entryStrip)
                    AllLines = f.readlines()
                    f.close()


                    ## append lines in ASCII file to selection list
                    for line in AllLines:
                        ListOfSelectedMolecules.append(line)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## continue here, if current entry is not a filename
            else:
                ListOfSelectedMolecules.append(entryStrip)


    ##====================================================================================================================================================
    ## translate Splatalogue into VAMDC names


    ## get path of default database file
    if (dbFilename is None):
        dbFilename = task_myXCLASS.GetDefaultDBFile()


    ## get Splatalgoue translation table from database file
    SplatFlag = False
    ListVAMDCNames = []
    ListSplatNames = []
    if (ListOfSelectedMolecules != []):
        SplatFlag, LocalMoleculeNameTranslationTable = task_myXCLASSFit.GetSplatTranslationTable(dbFilename)
        ListVAMDCNames, ListSplatNames = list(zip(*LocalMoleculeNameTranslationTable))


        ## translate molecule names in ListOfSelectedMolecules
        NewListOfSelectedMolecules = []
        ListOfReplacedNames = []
        for LocalSelectedMolecules in ListOfSelectedMolecules:
            if (LocalSelectedMolecules in ListSplatNames):
                i = ListSplatNames.index(LocalSelectedMolecules)
                NewMol = ListVAMDCNames[i]
                ListOfReplacedNames.append([LocalSelectedMolecules, NewMol])
            else:
                NewMol = LocalSelectedMolecules
            NewListOfSelectedMolecules.append(NewMol)
        ListOfSelectedMolecules = copy.deepcopy(NewListOfSelectedMolecules)


        ## print all replaced molecule names
        if (ListOfReplacedNames != []):
            print ("\n\n\tFound Splatalogue name(s) in selected molecules. Replaced the following name(s):")
            for LocalReplacedNames in ListOfReplacedNames:
                print ("\t\t{:s}{:s}{:s} -> {:s}{:s}{:s}".format(chr(34), LocalReplacedNames[0], chr(34), chr(34), LocalReplacedNames[1], chr(34)))
            print ("\n")

    # Debug:
    # print ("SplatFlag = ", SplatFlag)
    # print ("ListVAMDCNames = ", ListVAMDCNames)
    # print ("ListSplatNames = ", ListSplatNames)
    # print ("ListOfSelectedMolecules = ", ListOfSelectedMolecules)


    ##====================================================================================================================================================
    ## connect to the sqlite3 database
    if (printFlag != "false"):
        print ("\n\nReading data from sqlite3 database ..", end = ' ', flush = True)


    ## connect to database
    try:
        conn = sqlite3.connect(dbFilename)

        # Debug:
        # print (" ")
        # print ("Connection to sqlite3 database {:s} established.".format(dbFilename))
    except sqlite3.Error as e:
        print (" ")
        print ("Can not connect to sqlite3 database {:s}.".format(dbFilename))
        print ("Error: {:d}: {:s}".format(e.args[0], e.args[1]))
        sys.exit(1)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get entries from database


    ## define query string
    ##
    ## get names of columns with
    ##      import sqlite3
    ##      connection = sqlite3.connect('cdms_sqlite.db')
    ##      cursor = connection.execute('select * from partitionfunctions')
    ##      names = list(map(lambda x: x[0], cursor.description))
    ##
    NameOfRadTransTable = "transitions"
    ColumnNameForNameTransitions = "T_Name"
    ColumnNameForFreqTransitions = "T_Frequency"
    ColumnNameForIntTransitions = "T_Intensity"
    ColumnNameForEinsteinATransitions = "T_EinsteinA"
    ColumnNameForFreqErrTransitions = "T_Uncertainty"
    ColumnNameForELowTransitions = "T_EnergyLower"
    ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
    ColumnNameForQNUpLabelTransitions = "T_UpperStateQuantumNumbers"
    ColumnNameForQNLowLabelTransitions = "T_LowerStateQuantumNumbers"
    ColumnNameForURLTransitions = "T_URL"

    query_string = "SELECT "
    query_string += ColumnNameForNameTransitions + ", "                                     ## position 1: molecule name
    query_string += ColumnNameForFreqTransitions + ", "                                     ## position 2: frequency
    query_string += ColumnNameForIntTransitions + ", "                                      ## position 3: intensity
    query_string += ColumnNameForEinsteinATransitions + ", "                                ## position 4: Einstein A
    query_string += ColumnNameForFreqErrTransitions + ", "                                  ## position 5: Error frequency
    query_string += ColumnNameForELowTransitions + ", "                                     ## position 6: E_low
    query_string += ColumnNameForgUpTransitions + ", "                                      ## position 7: upper state degeneracy
    query_string += ColumnNameForQNUpLabelTransitions + ", "                                ## position 8: quantum number label for upper state
    query_string += ColumnNameForQNLowLabelTransitions                                      ## position 9: quantum number label for lower state
    # query_string += ColumnNameForURLTransitions                                             ## position 10: URL
    query_string += " FROM " + NameOfRadTransTable
    query_string += " WHERE (" + ColumnNameForFreqTransitions + " >= " + str(FreqMin) + " and " + ColumnNameForFreqTransitions + " <= " + str(FreqMax)
    h = ElowMin / 1.42879
    query_string += " and " + ColumnNameForELowTransitions + " >= " + str(h)
    h = ElowMax / 1.42879
    query_string += " and " + ColumnNameForELowTransitions + " <= " + str(h)
    if (len(ListOfSelectedMolecules) > 0 and SelectMolecule != ["None"] and SelectMolecule != [""]):
        query_string += " and ("
        counter = 0
        for molecule in ListOfSelectedMolecules:


            ## check, if name of molecule contains invalid character
            i = molecule.find(chr(34))
            if (i > (-1)):
                molecule.translate(None, chr(34))

            counter += 1
            if (counter > 1):
                query_string += " or "
            query_string += ColumnNameForNameTransitions + " = " + chr(34) + molecule.strip() + chr(34)
        query_string += ")"
    query_string += ")"
    query_string += " ORDER by " + ColumnNameForFreqTransitions

    # Debug:
    # print (" ")
    # print ("SelectMolecule = ", SelectMolecule)
    # print ("ListOfSelectedMolecules = ", ListOfSelectedMolecules)
    # print ("query_string = ", query_string)
    # print ("len(query_string) = ", len(query_string))


    ## read data from database
    cursor = conn.cursor()
    cursor.execute(query_string)
    rows = cursor.fetchall()
    conn.close()


    ## finished
    if (printFlag != "false"):
        print ("done!")


    ## open file if file name is defined
    file_flag = False
    formatted_line = "{:25s}".format("Name:")
    formatted_line += "{:37s}".format("quantum number upper state:")
    formatted_line += "{:37s}".format("quantum number lower state:")
    formatted_line += "{:18s}".format("Frequency [MHz]:")
    formatted_line += "{:24s}".format("Error Frequency [MHz]:")
    formatted_line += "{:27s}".format("Einstein A coefficient:")
    #formatted_line += "{:27s}".format("log(Intensity) [nm2 MHz]:")
    formatted_line += "{:20s}".format("Energy_low [K]:")
    formatted_line += "  {:23s}".format("upper state degeneracy:")
    if (OutputDevice != "" and OutputDevice is not None and printFlag == "true"):
        file_flag = True
        out = open(OutputDevice, 'w')
        out.write(formatted_line + "\n")
        if (printFlag != "false"):
            print ("Reading contents of database .. ", end = "", flush = True)
    else:
        if (printFlag != "false"):
            print ("\n\nContents of the database between {:s} MHz and {:s} MHz:\n".format(str(FreqMin), str(FreqMax)))
            print (formatted_line)


    ## store entries in output variable Contents
    Contents = []
    col_counter = 0
    for row in rows:                                                                        ## loop over all entries of the database
        col_counter += 1
        # Contents.append(row)

        # Debug:
        # print ("col_counter, row = ", col_counter, row)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define formatted line
        ## position 1: molecule name
        ## position 2: frequency
        ## position 3: intensity
        ## position 4: Einstein A
        ## position 5: Error frequency
        ## position 6: E_low
        ## position 7: upper state degeneracy
        ## position 8: quantum number label for upper state
        ## position 9: quantum number label for lower state
        ## position 10: URL
        formatted_line = ""
        name = ""
        freq = ""
        unfreq = ""
        intensity = ""
        EinsteinA = ""
        elow = ""
        gUp = ""
        QNUpLabel = ""
        QNLowLabel = ""
        DBurl = ""
        element_counter = 0
        for elements in row:
            element_counter += 1


            ## get name of molecule
            if (element_counter == 1):
                name = elements.strip()
                if (OutputDevice == "quiet"):
                    name = '{:<45s}'.format(str(name))
                else:
                    name = '{:25s}'.format(name[:25])                                       ## limit name of molecule to 25 characters

                # Debug:
                # print ("name = ", name)


            ## get frequency (in MHz)
            elif (element_counter == 2):
                freqVal = '{:17s}'.format(str(elements))
                if (freqVal.strip() != "None"):
                    freqVal = float(freqVal)
                    freq = '{:17.5f}'.format(freqVal)
                else:
                    freq = freqVal

                # Debug:
                # print ("freq = ", freq)


            ## get intensity
            elif (element_counter == 3):
                intensityVal = '{:26s}'.format(str(elements))
                if (intensityVal.strip() != "None"):
                    intensityVal = float(intensityVal)
                    intensity = '{:26.9e}'.format(intensityVal)
                else:
                    intensity = intensityVal

                # Debug:
                # print ("intensity = ", intensity)


            ## get Einstein A coefficient
            elif (element_counter == 4):
                EinsteinAVal = '{:26s}'.format(str(elements))
                if (EinsteinAVal.strip() != "None"):
                    EinsteinAVal = float(EinsteinAVal)
                    EinsteinA = '{:26.3e}'.format(EinsteinAVal)
                else:
                    EinsteinA = EinsteinAVal

                # Debug:
                # print ("EinsteinA = ", EinsteinA)


            ## get uncertainty of frequency (in MHz)
            elif (element_counter == 5):
                unfreqVal = '{:23s}'.format(str(elements))
                if (unfreqVal.strip() != "None"):
                    unfreqVal = float(unfreqVal)
                    unfreq = '{:23.4e}'.format(unfreqVal)
                else:
                    unfreq = unfreqVal

                # Debug:
                # print ("unfreq = ", unfreq)


            ## get E_lower (in cm-1)
            elif (element_counter == 6):
                elowVal = '{:19s}'.format(str(elements))
                if (elowVal.strip() != "None"):
                    elowVal = float(elowVal) * 1.438769                                     ## convert cm-1 to K
                    if (abs(elowVal) > 0.01 and abs(elowVal) < 10000):
                        elow = '{:19.3f}'.format(elowVal)
                    else:
                        elow = '{:19.3e}'.format(elowVal)
                else:
                    elow = elowVal

                # Debug:
                # print ("elow = ", elow)


            ## get upper state degeneracy
            elif (element_counter == 7):
                gUpVal = '{:23s}'.format(str(elements))
                if (gUpVal.strip() != "None"):
                    gUpVal = float(gUpVal)
                    gUp = '{:25.1e}'.format(gUpVal)
                else:
                    gUp = gUpVal

                # Debug:
                # print ("gUp = ", gUp)


            ## get quantum number label for upper state
            elif (element_counter == 8):
                try:
                    lab = elements
                    lab = lab.replace("ElecStateLabel = ", "").strip()
                    lab = lab.replace(" ", "").strip()                                      ## remove blank characters
                    if (len(lab) > 34):
                        lab = lab[:31] + "..."
                    QNUpLabel = '  {:34s}'.format(lab)
                except TypeError:
                    QNUpLabel = "     "

                # Debug:
                # print ("QNUpLabel = ", QNUpLabel)


            ## get quantum number label for lower state
            elif (element_counter == 9):
                try:
                    lab = elements
                    lab = lab.replace("ElecStateLabel = ", "").strip()
                    lab = lab.replace(" ", "").strip()                                      ## remove blank characters
                    if (len(lab) > 34):
                        lab = lab[:31] + "..."
                    QNLowLabel = '  {:34s}'.format(lab)
                except TypeError:
                    QNLowLabel = "     "

                # Debug:
                # print ("QNLowLabel = ", QNLowLabel)


            ## get URL
            # elif (element_counter == 10):
            #     try:
            #         DBurl = '{:s}'.format(elements)
            #     except TypeError:
            #         DBurl = "     "

            # Debug:
            # print ("DBurl = ", DBurl)


        ## construct line for output
        formatted_line = " ".join([name, QNUpLabel, QNLowLabel, freq, unfreq, EinsteinA, elow, gUp, DBurl])
        Contents.append(formatted_line)

        # Debug:
        # print ("formatted_line = ", formatted_line)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## if file name is defined
        if (file_flag):
            out.write(formatted_line + "\n")
        else:
            if (printFlag != "false"):
                print (formatted_line)


    ## close file, if file name is defined
    if (file_flag):
        out.close()
        if (printFlag != "false"):
            print ("done!")


    ##====================================================================================================================================================
    ## define return variables
    return Contents
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## call for standard interface
##
def ListDatabase(FreqMin, FreqMax, ElowMin, ElowMax, SelectMolecule, OutputDevice):
    """

This function reads in entries from the sqlite database table "Transitions" and prints out the contents
to the screen or file (defined by the input parameter OutputDevice).

The user can limit the output by defining a minimum and maximum for the frequency (or for
the lower energy) for the transitions.


input parameters:
-----------------


    - FreqMin:              minimum frequency (in MHz) (default: 0) for transitions in the table "Transitions"

    - FreqMax:              maximum frequency (in MHz) (default: 1.e8) for transitions in the table "Transitions"

    - ElowMin:              minimum for the lower energy (in K) (default: 0) in the "Transitions" table

    - ElowMax:              maximum for the lower energy (in K) (default: 1.e6) in the "Transitions" table

    - SelectMolecule:       a (python) list containing the names of all molecules which should be considered
                            or a file name including the molecules which should be selected.

                            Note, if the parameter defines a relative path, this path has to be defined
                            relative to the current working directory!

    - OutputDevice:         path and name of the file where the output is written to. If no file name is
                            defined (None), the contents of the database is written to the screen.

                            Note, if the parameter defines a relative path, this path has to be defined
                            relative to the current working directory!


output parameters:
------------------

    - Contents:             contents of the database (as an python array, e.g. contents[0] contains
                            the entries for the first molecule within the frequency/energy range)

                            Note, the user is free to define a different name for the output parameter.


Example 1:
----------

FreqMin = 20000.0
FreqMax = 20100.0
ElowMin = 100.0
ElowMax = 1000.0
SelectMolecule = "demo/ListDatabase/molecules.txt"
OutputDevice = ""
Contents = ListDatabase()


Example 2:
----------

FreqMin = 20000.0
FreqMax = 20100.0
ElowMin = 100.0
ElowMax = 2000.0
SelectMolecule = []
OutputDevice = ""
Contents = ListDatabase()
    """

    # Debug:
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("ElowMin = ", ElowMin)
    # print ("ElowMax = ", ElowMax)
    # print ("SelectMolecule = ", SelectMolecule)
    # print ("OutputDevice = ", OutputDevice)


    ##====================================================================================================================================================
    ## call ListDatabase core function
    Contents = ListDatabaseCore(FreqMin = FreqMin, FreqMax = FreqMax, ElowMin = ElowMin, ElowMax = ElowMax, SelectMolecule = SelectMolecule, \
                                OutputDevice = OutputDevice)

    # Debug:
    # print ("Contents = ", Contents)


    ##====================================================================================================================================================
    ## define return variables
    return Contents
##--------------------------------------------------------------------------------------------------------------------------------------------------------

