#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module identifies lines in the given frequency range
##  Copyright (C) 2012 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function gaussian_kde.__init__:                               initialize class gaussian_kde
##      - function gaussian_kde.evaluate:                               evaluate the estimated pdf on a set of points.
##      - function gaussian_kde.scotts_factor:                          scotts_factor
##      - function gaussian_kde.silverman_factor:                       silverman_factor
##      - function gaussian_kde.set_bandwidth:                          compute the estimator bandwidth with given method.
##      - function gaussian_kde._compute_covariance:                    computes the covariance matrix for each Gaussian kernel using covariance_factor().
##      - function GetEmAbsFunc:                                        get emission / absorption function
##      - function ContinuumFunction:                                   continuum function
##      - function EstimateContinuum:                                   estimate continuum
##      - function GetKnownMolecules:                                   get list of all known molecules
##      - function ComputeRMS:                                          compute rms of given array
##      - function GetObsXMLParameters:                                 get obs. xml file parameters
##      - function GetDicKey:                                           get key value of dictionary
##      - function ImportASCIIFile:                                     import ASCII file
##      - function GetBackgroundFileNames:                              get background file names
##      - function FindBackgroundFileName4Range:                        find background file name for given frequency range
##      - function MoleculeFileName:                                    get file name of a molecule
##      - function GetTransFreq:                                        get all transition energies from database
##      - function DefineNewFreqRanges:                                 define new frequency ranges
##      - function CreateXMLFile:                                       create new observational xml file for each molecule
##      - function ShrinkFreqRanges:                                    shrink freq ranges defined in the exp. xml file to reduce computational effort
##      - function AdjustExpXMLFile:                                    adjust paths in an MAGIX xml file
##      - function ReadDatabase:                                        get info. for all molecules between FreqMin and FreqMax from sqlite3 database
##      - function SmoothObsData:                                       smooth experimental data for current frequency window
##      - function GetPartitionFunction:                                calculate partition function for T = T_rot
##------- parameter estimation algorithm: ----------------------------------------------------------------------------------------------------------------
##      - function CreateSummedPlots:                                   create plots for total KDE and summed spectra
##      - function PlotTranstionWindow:                                 plot a single transition window
##      - function CreateIOControlXMLFile:                              creates the io-control xml file in the current job directory
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##      - function CreateMolfitFile:                                    creates molfit file for the current molecule, and writes create molfits file to
##                                                                      current molecule subdirectory
##      - function GetBestResult:                                       determines the best fit result for a molecule
##      - function CheckContribution:                                   check, if molecule contributes to current spectrum or not
##      - function SingleMoleculeFit:                                   start a single molecule fit
##      - function StartWorker:                                         starts worker for cluster fits via command line
##      - function StrongLinesFits:                                     start single molecule fits related to strong lines
##      - function StartClusterFits:                                    start different molecule fits on the cluster
##      - function CreateMolfitFileInNewFormat:                         create a new molfit file in the new format
##      - function ExtendPackage:                                       include local python packages
##      - function LineIdentificationCore:                              core LineIdentification function
##      - function LineIdentification:                                  function identifies the lines in the given frequency range
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2014-05-08      initial version
##  T. Moeller      2015-04-14      improved version (add new method: redefine freq. ranges for single molecule fits)
##  T. Moeller      2015-2017       other improvements (add KDE option, weighting of transitions, bug-fixes etc.)
##  T. Moeller      2018-05-14      add automated gaussian decomposition
##  T. Moeller      2018-06-26      reorganization, cleaning, introduce class MolfitParameterEstimation
##  T. Moeller      2020-01-03      porting to python 3.x
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import copy                                                                                 ## import copy package
import time                                                                                 ## import time package
if (not 'matplotlib' in sys.modules):
    import matplotlib                                                                       ## import matplotlib package
    matplotlib.use("Agg")                                                                   ## avoid display error
else:
    import matplotlib                                                                       ## import matplotlib package
import pylab                                                                                ## import pylab package
from scipy import signal                                                                    ## import signal from scipy package
from scipy import stats                                                                     ## import stats from scipy package
from scipy.spatial.distance import cdist                                                    ## import cdist from scipy package
from scipy.optimize import curve_fit                                                        ## import curve_fit from scipy package
from scipy.interpolate import interp1d                                                      ## import interp1d from scipy package
from . import task_MAGIX                                                                    ## import package MAGIX
from . import task_myXCLASS                                                                 ## import package myXCLASS
from . import task_myXCLASSFit                                                              ## import package myXCLASSFit
from . import task_myXCLASSMapFit                                                           ## import package myXCLASSMapFit
import sqlite3                                                                              ## import sqlite3 package
import multiprocessing                                                                      ## import multiprocessing package
import pickle                                                                               ## import pickle package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define class for weighted KDE
## taken from "https://gist.github.com/tillahoffmann/f844bce2ec264c1c8cb5"
##
class gaussian_kde(object):
    """Representation of a kernel-density estimate using Gaussian kernels.

    Kernel density estimation is a way to estimate the probability density
    function (PDF) of a random variable in a non-parametric way.
    `gaussian_kde` works for both uni-variate and multi-variate data.   It
    includes automatic bandwidth determination.  The estimation works best for
    a unimodal distribution; bimodal or multi-modal distributions tend to be
    oversmoothed.

    Parameters
    ----------
    dataset : array_like
        Datapoints to estimate from. In case of univariate data this is a 1-D
        array, otherwise a 2-D array with shape (# of dims, # of data).
    bw_method : str, scalar or callable, optional
        The method used to calculate the estimator bandwidth.  This can be
        'scott', 'silverman', a scalar constant or a callable.  If a scalar,
        this will be used directly as `kde.factor`.  If a callable, it should
        take a `gaussian_kde` instance as only parameter and return a scalar.
        If None (default), 'scott' is used.  See Notes for more details.
    weights : array_like, shape (n, ), optional, default: None
        An array of weights, of the same shape as `x`.  Each value in `x`
        only contributes its associated weight towards the bin count
        (instead of 1).

    Attributes
    ----------
    dataset : ndarray
        The dataset with which `gaussian_kde` was initialized.
    d : int
        Number of dimensions.
    n : int
        Number of datapoints.
    neff : float
        Effective sample size using Kish's approximation.
    factor : float
        The bandwidth factor, obtained from `kde.covariance_factor`, with which
        the covariance matrix is multiplied.
    covariance : ndarray
        The covariance matrix of `dataset`, scaled by the calculated bandwidth
        (`kde.factor`).
    inv_cov : ndarray
        The inverse of `covariance`.

    Methods
    -------
    kde.evaluate(points) : ndarray
        Evaluate the estimated pdf on a provided set of points.
    kde(points) : ndarray
        Same as kde.evaluate(points)
    kde.pdf(points) : ndarray
        Alias for ``kde.evaluate(points)``.
    kde.set_bandwidth(bw_method='scott') : None
        Computes the bandwidth, i.e. the coefficient that multiplies the data
        covariance matrix to obtain the kernel covariance matrix.
        .. versionadded:: 0.11.0
    kde.covariance_factor : float
        Computes the coefficient (`kde.factor`) that multiplies the data
        covariance matrix to obtain the kernel covariance matrix.
        The default is `scotts_factor`.  A subclass can overwrite this method
        to provide a different method, or set it through a call to
        `kde.set_bandwidth`.

    Notes
    -----
    Bandwidth selection strongly influences the estimate obtained from the KDE
    (much more so than the actual shape of the kernel).  Bandwidth selection
    can be done by a "rule of thumb", by cross-validation, by "plug-in
    methods" or by other means; see [3]_, [4]_ for reviews.  `gaussian_kde`
    uses a rule of thumb, the default is Scott's Rule.

    Scott's Rule [1]_, implemented as `scotts_factor`, is::

        n**(-1./(d+4)),

    with ``n`` the number of data points and ``d`` the number of dimensions.
    Silverman's Rule [2]_, implemented as `silverman_factor`, is::

        (n * (d + 2) / 4.)**(-1. / (d + 4)).

    Good general descriptions of kernel density estimation can be found in [1]_
    and [2]_, the mathematics for this multi-dimensional implementation can be
    found in [1]_.

    References
    ----------
    .. [1] D.W. Scott, "Multivariate Density Estimation: Theory, Practice, and
           Visualization", John Wiley & Sons, New York, Chicester, 1992.
    .. [2] B.W. Silverman, "Density Estimation for Statistics and Data
           Analysis", Vol. 26, Monographs on Statistics and Applied Probability,
           Chapman and Hall, London, 1986.
    .. [3] B.A. Turlach, "Bandwidth Selection in Kernel Density Estimation: A
           Review", CORE and Institut de Statistique, Vol. 19, pp. 1-33, 1993.
    .. [4] D.M. Bashtannyk and R.J. Hyndman, "Bandwidth selection for kernel
           conditional density estimation", Computational Statistics & Data
           Analysis, Vol. 36, pp. 279-298, 2001.

    Examples
    --------
    Generate some random two-dimensional data:

    >>> from scipy import stats
    >>> def measure(n):
    >>>     "Measurement model, return two coupled measurements."
    >>>     m1 = numpy.random.normal(size=n)
    >>>     m2 = numpy.random.normal(scale=0.5, size=n)
    >>>     return m1+m2, m1-m2

    >>> m1, m2 = measure(2000)
    >>> xmin = m1.min()
    >>> xmax = m1.max()
    >>> ymin = m2.min()
    >>> ymax = m2.max()

    Perform a kernel density estimate on the data:

    >>> X, Y = numpy.mgrid[xmin:xmax:100j, ymin:ymax:100j]
    >>> positions = numpy.vstack([X.ravel(), Y.ravel()])
    >>> values = numpy.vstack([m1, m2])
    >>> kernel = stats.gaussian_kde(values)
    >>> Z = numpy.reshape(kernel(positions).T, X.shape)

    Plot the results:

    >>> import matplotlib.pyplot as plt
    >>> fig = plt.figure()
    >>> ax = fig.add_subplot(111)
    >>> ax.imshow(numpy.rot90(Z), cmap=plt.cm.gist_earth_r,
    ...           extent=[xmin, xmax, ymin, ymax])
    >>> ax.plot(m1, m2, 'k.', markersize=2)
    >>> ax.set_xlim([xmin, xmax])
    >>> ax.set_ylim([ymin, ymax])
    >>> plt.show()

    """


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize
    def __init__(self, dataset, bw_method = None, weights = None):

        # Debug:
        # print ("dataset = ", dataset)
        # print ("bw_method = ", bw_method)
        # print ("weights = ", weights)


        self.dataset = numpy.atleast_2d(dataset)
        if not self.dataset.size > 1:
            raise ValueError("`dataset` input should have multiple elements.")
        self.d, self.n = self.dataset.shape

        if weights is not None:
            self.weights = weights / numpy.sum(weights)
        else:
            self.weights = numpy.ones(self.n) / self.n

        # Compute the effective sample size
        # http://surveyanalysis.org/wiki/Design_Effects_and_Effective_Sample_Size#Kish.27s_approximate_formula_for_computing_effective_sample_size
        self.neff = 1.0 / numpy.sum(self.weights ** 2)

        self.set_bandwidth(bw_method=bw_method)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Evaluate the estimated pdf on a set of points.
    def evaluate(self, points):
        """

        Parameters
        ----------
        points : (# of dimensions, # of points)-array
            Alternatively, a (# of dimensions,) vector can be passed in and
            treated as a single point.

        Returns
        -------
        values : (# of points,)-array
            The values at each point.

        Raises
        ------
        ValueError : if the dimensionality of the input points is different than
                     the dimensionality of the KDE.
        """

        # Debug:
        # print ("points = ", points)


        points = numpy.atleast_2d(points)
        d, m = points.shape
        if d != self.d:
            if d == 1 and m == self.d:


                # points was passed in as a row vector
                points = numpy.reshape(points, (self.d, 1))
                m = 1
            else:
                msg = "points have dimension {:s}, dataset has dimension {:s}".format(d, self.d)
                raise ValueError(msg)


        # compute the normalised residuals
        chi2 = cdist(points.T, self.dataset.T, 'mahalanobis', VI=self.inv_cov) ** 2


        # compute the pdf
        result = numpy.sum(numpy.exp(-.5 * chi2) * self.weights, axis=1) / self._norm_factor


        ## we're done
        return result
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##  ??
    __call__ = evaluate


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## scotts_factor
    def scotts_factor(self):
        return numpy.power(self.neff, -1./(self.d+4))
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## silverman_factor
    def silverman_factor(self):
        return numpy.power(self.neff*(self.d+2.0)/4.0, -1./(self.d+4))
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##  Default method to calculate bandwidth, can be overwritten by subclass
    covariance_factor = scotts_factor


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Compute the estimator bandwidth with given method.
    def set_bandwidth(self, bw_method=None):
        """
        The new bandwidth calculated after a call to `set_bandwidth` is used
        for subsequent evaluations of the estimated density.

        Parameters
        ----------
        bw_method : str, scalar or callable, optional
            The method used to calculate the estimator bandwidth.  This can be
            'scott', 'silverman', a scalar constant or a callable.  If a
            scalar, this will be used directly as `kde.factor`.  If a callable,
            it should take a `gaussian_kde` instance as only parameter and
            return a scalar.  If None (default), nothing happens; the current
            `kde.covariance_factor` method is kept.

        Notes
        -----
        .. versionadded:: 0.11

        Examples
        --------
        >>> x1 = numpy.array([-7, -5, 1, 4, 5.])
        >>> kde = stats.gaussian_kde(x1)
        >>> xs = numpy.linspace(-10, 10, num=50)
        >>> y1 = kde(xs)
        >>> kde.set_bandwidth(bw_method='silverman')
        >>> y2 = kde(xs)
        >>> kde.set_bandwidth(bw_method=kde.factor / 3.)
        >>> y3 = kde(xs)

        >>> fig = plt.figure()
        >>> ax = fig.add_subplot(111)
        >>> ax.plot(x1, numpy.ones(x1.shape) / (4. * x1.size), 'bo',
        ...         label='Data points (rescaled)')
        >>> ax.plot(xs, y1, label='Scott (default)')
        >>> ax.plot(xs, y2, label='Silverman')
        >>> ax.plot(xs, y3, label='Const (1/3 * Silverman)')
        >>> ax.legend()
        >>> plt.show()
        """

        # Debug:
        # print ("bw_method = ", bw_method)


        if bw_method is None:
            pass
        elif bw_method == 'scott':
            self.covariance_factor = self.scotts_factor
        elif bw_method == 'silverman':
            self.covariance_factor = self.silverman_factor
        elif numpy.isscalar(bw_method):     # and not isinstance(bw_method, string_types):
            self._bw_method = 'use constant'
            self.covariance_factor = lambda: bw_method
        elif callable(bw_method):
            self._bw_method = bw_method
            self.covariance_factor = lambda: self._bw_method(self)
        else:
            msg = "`bw_method` should be 'scott', 'silverman', a scalar " \
                  "or a callable."
            raise ValueError(msg)

        self._compute_covariance()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Computes the covariance matrix for each Gaussian kernel using covariance_factor().
    def _compute_covariance(self):


        self.factor = self.covariance_factor()


        # Cache covariance and inverse covariance of the data
        if not hasattr(self, '_data_inv_cov'):


            # Compute the mean and residuals
            _mean = numpy.sum(self.weights * self.dataset, axis=1)
            _residual = (self.dataset - _mean[:, None])


            # Compute the biased covariance
            self._data_covariance = numpy.atleast_2d(numpy.dot(_residual * self.weights, _residual.T))


            # Correct for bias (http://en.wikipedia.org/wiki/Weighted_arithmetic_mean#Weighted_sample_covariance)
            self._data_covariance /= (1 - numpy.sum(self.weights ** 2))
            self._data_inv_cov = numpy.linalg.inv(self._data_covariance)

        self.covariance = self._data_covariance * self.factor**2
        self.inv_cov = self._data_inv_cov / self.factor**2
        self._norm_factor = numpy.sqrt(numpy.linalg.det(2*numpy.pi*self.covariance)) #* self.n


        ## we're done
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## get emission / absorption function
def GetEmAbsFunc(LocalEmAbsPATH):
    """

input parameters:
-----------------

    - LocalEmAbsPATH:               path containing ASCII files describing emission / absorption functions


output parameters:
------------------

    - EmsAbsFileInterDict:          directory containing the interpolation function objects for each distance
    """

    # Debug:
    # print ("LocalEmAbsPATH = ", LocalEmAbsPATH)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## is only one molecule contained in current molfit file
    EmsAbsFileContentDir = {}
    listing = os.listdir(LocalEmAbsPATH)
    for LocalFileName in listing:
        LowerLocalFileName = LocalFileName.lower()


        ## check, if emission / absorption files are included in the given directory
        if (LowerLocalFileName.startswith("emission_absorption____")):
            LocalEmAbsFileName = LocalEmAbsPATH + "/" + LocalFileName


            ## determine distance from file name
            Distance = LowerLocalFileName.split("__")
            Distance = Distance[-1]
            Distance = Distance.replace(".dat", "")
            Distance = Distance.replace(".gz", "")
            DistanceNum = float(Distance)


            ## import content of ASCII file and store to python dictionary
            LocalEmsAbsFileContent = numpy.loadtxt(LocalEmAbsFileName, skiprows = 3)
            try:
                l = EmsAbsFileContentDir[Distance]
                EmsAbsFileContentDir[Distance] = numpy.concatenate((l, LocalEmsAbsFileContent), axis = 0)
            except:
                EmsAbsFileContentDir[Distance] = LocalEmsAbsFileContent

    # Debug:
    # print ("EmsAbsFileContentDir = ", EmsAbsFileContentDir)
    # print ("EmsAbsFileContentDir.keys() = ", EmsAbsFileContentDir.keys())


    ## sort merged arrays
    EmsAbsFileInterDict = {}
    for LocalDistance in list(EmsAbsFileContentDir.keys()):
        LocalArray = EmsAbsFileContentDir[LocalDistance]
        LocalArray = LocalArray[LocalArray[:, 0].argsort()]

        # Debug:
        #    numpy.savetxt(LocalDistance + ".dat", LocalArray)


        ## interpolate emission and absorption functions
        EmInterpolFunc = interp1d(LocalArray[:, 0], LocalArray[:, 1], bounds_error = False, fill_value = "extrapolate")
        AbsInterpolFunc = interp1d(LocalArray[:, 0], LocalArray[:, 2], bounds_error = False, fill_value = "extrapolate")
        EmsAbsFileInterDict[LocalDistance] = [EmInterpolFunc, AbsInterpolFunc]

    # Debug:
    # print ("EmsAbsFileInterDict = ", EmsAbsFileInterDict)


    ## we're done
    return EmsAbsFileInterDict
##----------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## continuum function
##
def ContinuumFunction(nu, Tbg, TSlope, nuMin):
    """

input parameters:
-----------------

    - nu:                   frequency

    - Tbg:                  background temperature

    - TSlope:               temperature slope

    - nuMin:                min. frequency


output parameters:
------------------

    - None
    """

    # Debug:
    # print "nu = ", nu
    # print "Tbg = ", Tbg
    # print "TSlope = ", TSlope
    # print "nuMin = ", nuMin


    ## return to main program
    return Tbg * (nu / nuMin)**TSlope
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## estimate continuum
def EstimateContinuum(DataArray, Method = "statcont", NumParts = 20):
    """

input parameters:
-----------------

    - DataArray:        data array

    - Method:           (optional) method used to estimate continuum, (default: "statcont")

    - NumParts:         (optional) number of subclasses, (default: 20)


output parameters:
------------------

    - DataNoContArray:  data array without continuum

    - NewTBack:         new background temperature

    - NewTSlope:        new temperature slope
    """

    # Debug:
    # print ("\nDataArray = ", DataArray)
    # print ("numpy.shape(DataArray) = ", numpy.shape(DataArray))
    # print ("Method = ", Method)
    # print ("NumParts = ", NumParts)


    ## initialize return parameter
    DataNoContArray = numpy.zeros((len(DataArray[:, 0])), dtype = numpy.float32)
    NewTBack = None
    NewTSlope = None


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## use statcont to remove continuum
    if (Method in ["statcont"]):


        ## try to import statcont
        STATCONTAvailableFlag = True
        try:
            # import statcont
            from external.statcont import cont_finding
        except:
            STATCONTAvailableFlag = False

        # Debug:
        # print ("STATCONTAvailableFlag = ", STATCONTAvailableFlag)


        ## remove continuum using statcont
        if (STATCONTAvailableFlag):


            ## compute rms
            rms = ComputeRMS(DataArray[:, 1])

            # Debug:
            # print ("rms = ", rms)


            ## split spectrum into smaller parts
            l = len(DataArray)
            q = float(l) / float(NumParts)
            if (q == float(int(q))):
                SplittedLocalData = numpy.vsplit(DataArray, NumParts)
            else:
                ll = int(q) * NumParts
                SplittedLocalData = numpy.vsplit(DataArray[:ll, :], NumParts)

            # Debug:
            # print ("l = ", l)
            # print ("NumParts = ", NumParts)
            # print ("q = ", q)


            ## apply statcont to each part
            nuList = []
            contLevelList = []
            for LocalPartData in SplittedLocalData:


                ## write the intensity of a given pixel for all the channels into the array flux and the frequencies in the array freqs
                rms_noise = rms
                freq_axis = 0
                freq = LocalPartData[:, 0]
                if (len(freq) > 0):
                    MinFreq = numpy.nanmin(freq)
                    MaxFreq = numpy.nanmax(freq)
                    CenterFreq = MinFreq + ((MaxFreq - MinFreq) / 2.0)
                    flux = LocalPartData[:, 1]
                    nuList.append(CenterFreq)


                    ## call sigma clipping subroutine
                    # sigmaclip_flux_prev, sigmaclip_flux, sigmaclip_noise, filtered_data = statcont.cont_finding.c_sigmaclip(flux, rms_noise, \
                    #                                                                                                         freq_axis)
                    sigmaclip_flux_prev, sigmaclip_flux, sigmaclip_noise, filtered_data = cont_finding.c_sigmaclip(flux, rms_noise,freq_axis)
                    contLevelList.append(sigmaclip_flux)


            ## fit background function to list of continuum levels
            contLevelList = numpy.asarray(contLevelList)
            nuList = numpy.asarray(nuList)
            Tbg = DataArray[0, 1]
            TSlope = 0.0
            nuMin= DataArray[0, 0]
            p0 = [Tbg, TSlope, nuMin]
            p0 = numpy.asarray(p0)
            try:
                popt, pcov = curve_fit(ContinuumFunction, nuList, contLevelList, p0)
            except:
                popt = None

            # Debug:
            # print ("popt = ", popt)


            ## define array of fit function
            if (popt is not None):
                NewTBack = popt[0]
                NewTSlope = popt[1]
                DataNoContArray[:] = DataArray[:, 1] - ContinuumFunction(DataArray[:, 0], *popt)

    # Debug:
    # print ("DataNoContArray = ", DataNoContArray)


    ## we're done
    return DataNoContArray, NewTBack, NewTSlope
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get list of all known molecules
##
def GetKnownMolecules():

    """

input parameters:
-----------------

    - None


output parameters:
------------------

    - ListOfKnownMolecules:             list of all known molecules
    """


    ## define list of known molecules
    ListOfKnownMolecules = [## one atom
                            "12C", "14N", "16O", "28Si", "2D", "32S", "56Fe", "56Fe+",

                            ## two atom molecules
                            "AlCl;v=0;", "AlF;v=0;", "AlO;v=0;", "ArH+;v=0;", "C2;v=0;", "CF+;v=0;", "CH;v=0;", "CH+;v=0;", "CN;v=0;", "CN-;v=0;",
                            "CO;v=0;", "CO+;v=0;", "CP;v=0;", "CS;v=0;", "FeO;v=0;", "H2;v=0;", "HCl;v=0;", "HCl+;v=0;", "HD;v=0;", "HF;v=0;",
                            "KCl;v=0;", "N2;v=0;", "NaCl;v=0;", "NH;v=0;", "NO;v=0;", "NO+;v=0;", "NS;v=0;", "NS+;v=0;", "O2;v=0;", "OH;v=0;",
                            "OH+;v=0;", "PN;v=0;", "PO;v=0;", "SH;v=0;", "SH+;v=0;", "SiC;v=0;", "SiH;v=0;", "SiN;v=0;", "SiO;v=0;", "SiS;v=0;",
                            "SO;v=0;", "SO+;v=0;", "TiO;v=0;",

                            ## three atom molecules
                            "AlNC;v=0;", "AlOH;v=0;", "C2H;v=0;", "C2O;v=0;", "C2S;v=0;", "C3;v=0;", "CCN;v=0;", "CCP;v=0;", "CH2;v=0;",
                            "CO2;v=0;", "c-SiC2;v=0;", "FeCN;v=0;", "H2Cl+;v=0;", "H2O;v=0;", "H2O+;v=0;", "H2S;v=0;", "H3+;v=0;", "HCN;v=0;",
                            "HCO;v=0;", "HCO+;v=0;", "HCP;v=0;", "HCS;v=0;", "HCS+;v=0;", "HNC;v=0;", "HNO;v=0;", "HO2;v=0;", "HOC+;v=0;",
                            "HS2;v=0;", "HSC;v=0;", "KCN;v=0;", "MgCN;v=0;", "MgNC;v=0;", "N2H+;v=0;", "N2O;v=0;", "NaCN;v=0;", "NCO;v=0;",
                            "NH2;v=0;", "OCS;v=0;", "S2H;v=0;", "Si2C;v=0;", "SiC2;v=0;", "SiCN;v=0;", "SiCSi;v=0;", "SiNC;v=0;", "SO2;v=0;",
                            "TiO2;v=0;",

                            ## four atom molecules
                            "C2H2;v=0;", "C3N-;v=0;", "C3N;v=0;", "C3O;v=0;", "C3S;v=0;", "c-C3H;v=0;", "CH3;v=0;", "CNCN;v=0;", "H2CN;v=0;",
                            "H2CO;v=0;", "H2CS;v=0;", "H2O2;v=0;", "H3O+;v=0;", "HC2N;v=0;", "HCCN;v=0;", "HCCO;v=0;", "HCNH+;v=0;", "HCNO;v=0;",
                            "HMgNC;v=0;", "HNCO;v=0;", "HNCS;v=0;", "HOCN;v=0;", "HOCO+;v=0;", "HOOH;v=0;", "HSCN;v=0;", "l-C3H;v=0;", "l-C3H+;v=0;",
                            "NH3;v=0;", "PH3;v=0;", "SiC3;v=0;",

                            ## five atom molecules
                            "C4H-;v=0;", "C4H;v=0;", "C4Si;v=0;", "C5;v=0;", "c-C3H2;v=0;", "CH2CN;v=0;", "CH2NH;v=0;", "CH3Cl;v=0;", "CH3O;v=0;",
                            "CH4;v=0;", "CNCHO;v=0;", "H2C2O;v=0;", "H2CCN;v=0;", "H2CNH;v=0;", "H2CCC;v=0;", "H2CCO;v=0;", "H2COH+;v=0;",
                            "H2NCN;v=0;", "H2NCO+;v=0;", "HC2NC;v=0;", "HC3N;v=0;", "HCCNC;v=0;", "HC(O)CN;v=0;", "HCOOH;v=0;",
                            "HNC3;v=0;", "HNCCC;v=0;", "HNCNH;v=0;", "l-C3H2;v=0;", "NCCNH+;v=0;", "NH2CN;v=0;", "NH4+;v=0;", "NH3D+;v=0;",
                            "SiC4;v=0;", "SiH4;v=0;",

                            ## six atom molecules
                            "C2H4;v=0;", "C5H;v=0;", "C5N-;v=0;", "C5N;v=0;", "C5S;v=0;", "c-H2C3O;v=0;", "CH2CNH;v=0;", "CH3CN;v=0;", "CH3NC;v=0;",
                            "CH3OH;v=0;", "CH3SH;v=0;", "H2C4;v=0;", "HC2CHO;v=0;", "H2CCNH;v=0;", "HC3NH+;v=0;", "HC4H;v=0;", "HC4N;v=0;",
                            "HNCHCN;v=0;", "NH2CHO;v=0;", "SiH3CN;v=0;",

                            ## seven atom molecules
                            "C2H3CN;v=0;", "C6H-;v=0;", "C6H;v=0;", "c-C2H4O;v=0;", "CH2CHCN;v=0;", "CH2CHOH;v=0;", "CH3CCH;v=0;", "CH3C2H;v=0;",
                            "CH3CHO;v=0;", "CH3NCO;v=0;", "CH3NH2;v=0;", "H2CCHOH;v=0;", "HC5N;v=0;", "HC5O;v=0;", "HOCH2CN;v=0;"

                            ## eight atom molecules
                            "C6H2;v=0;", "C7H;v=0;", "CH2CCHCN;v=0;", "CH2CHCHO;v=0;", "CH2OHCHO;v=0;", "CH3C3N;v=0;", "CH3CHNH;v=0;", "CH3COOH;v=0;",
                            "CH3SiH3;v=0;", "H2C6;v=0;", "H2NCH2CN;v=0;", "HC6H;v=0;", "HCOOCH3;v=0;", "HC(O)OCH3;v=0;", "NH2CH2CN;v=0;",
                            "l-HC6H;v=0;",

                            ## nine atom molecules
                            "C2H5CN;v=0;", "C3H6;v=0;", "C8H-;v=0;", "C8H;v=0;", "CH2CHCH3;v=0;", "CH3C4H;v=0;",  "CH3CH2CN;v=0;", "CH3CH2OH;v=0;",
                            "CH3CH2SH;v=0;", "CH3CONH2;v=0;", "CH3OCH3;v=0;", "HC7N;v=0;", "HC7O;v=0;", "(CH3)2O;v=0;", "CH3C(O)NH2;v=0;",
                            "CH3NHCHO;v=0;",

                            ## ten atom molecules
                            "(CH3)2CO;v=0;", "CH3C5N;v=0;", "CH3CH2CHO;v=0;", "CH3CHCH2O;v=0;", "CH3OCH2OH;v=0;", "HOCH2CH2OH;v=0;",
                            "(CH2OH)2;v=0;",

                            ## eleven atom molecules
                            "C2H5OCHO;v=0;", "CH3C6H;v=0;", "CH3COOCH3;v=0;", "HC9N;v=0;", "CH3OC(O)CH3;v=0;",

                            ## twelve atom molecules
                            "c-C6H6;v=0;", "C6H6;v=0;", "n-C3H7CN;v=0;", "i-C3H7CN;v=0;", "C2H5OCH3;v=0;",

                            ## thirteen atom molecules
                            "c-C6H5CN;v=0;",

                            ## fullerene molecules
                            "C60;v=0;", "C60+;v=0;", "C70;v=0;"]


    ## define return parameter
    return ListOfKnownMolecules
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## compute rms of given array
##
def ComputeRMS(IntensityArray):

    """

input parameters:
-----------------

    - IntensityArray:           array describing intensities


output parameters:
------------------

    - rms:                      rms for IntensityArray
    """

    # Debug:
    # print ("IntensityArray = ", IntensityArray)


    ## compute rms
    rms = numpy.sqrt(numpy.mean(IntensityArray[:]**2))

    # Debug:
    # print ("rms = ", rms)


    ## define return parameter
    return rms
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get obs. xml file parameters
##
def GetObsXMLParameters(LocalObsXMLFileName):

    """

input parameters:
-----------------

    - LocalObsXMLFileName:      path and name of obs. xml file


output parameters:
------------------

    - ObsXMLParameter:          dictionary containing obs. xml file parameters
    """

    # Debug:
    # print ("\n\nLocalObsXMLFileName = ", LocalObsXMLFileName)


    ## read in some parameters from observational xml file
    ExpFileList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "FileNamesExpFiles")
    NumberExpRangesList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "NumberExpRanges")
    FreqMinList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "MinExpRange")
    FreqMaxList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "MaxExpRange")
    FreqStepList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "StepFrequency")
    t_back_flagList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "t_back_flag")
    tBackList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "BackgroundTemperature")
    tSlopeList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "TemperatureSlope")
    N_HList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "HydrogenColumnDensity")
    beta_dustList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "DustBeta")
    kappa_1300List = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "Kappa")
    DustFileNameList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "DustFileName")
    BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "BackgroundFileName")
    ContPhenFuncIDList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "ContPhenFuncID")
    ContPhenFuncParam1List = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "ContPhenFuncParam1")
    ContPhenFuncParam2List = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "ContPhenFuncParam2")
    ContPhenFuncParam3List = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "ContPhenFuncParam3")
    ContPhenFuncParam4List = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "ContPhenFuncParam4")
    ContPhenFuncParam5List = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "ContPhenFuncParam5")
    NoiseList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "NoiseLevel")
    SmoothList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "SmoothValue")
    ThresholdList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "Threshold")
    TelescopeSizeList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "TelescopeSize")
    BMINList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "BMIN")
    BMAJList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "BMAJ")
    BPAList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "BPA")
    Inter_FlagList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "Inter_Flag")
    GlobalvLSRList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "GlobalvLSR")
    RedshiftList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "Redshift")
    ErrorYList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "ErrorY")
    NumberHeaderLinesList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "NumberHeaderLines")
    SeparatorColumnsList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "SeparatorColumns")
    IsotopologuesList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "iso_flag")
    if (IsotopologuesList == []):
        IsotopologuesList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "Isotopologues")
    IsoTableFileNameList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "IsoTableFileName")
    dbList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "dbFilename")
    if (dbList == []):
        dbFileName = task_myXCLASS.GetDefaultDBFile()
        dbList = [dbFileName]
    NumModelPixelXXList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "NumModelPixelXX")
    NumModelPixelYYList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "NumModelPixelYY")
    LocalOverlapFlagList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "LocalOverlap_Flag")
    NoSubBeamFlagList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "NoSubBeam_Flag")
    EmAbsPATHList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "EmAbsPATH")

    # Debug:
    # print ("ExpFileList = ", ExpFileList)
    # print ("NumberExpRangesList = ", NumberExpRangesList)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("FreqStepList = ", FreqStepList)
    # print ("t_back_flagList = ", t_back_flagList)
    # print ("tBackList = ", tBackList)
    # print ("tSlopeList = ", tSlopeList)
    # print ("N_HList = ", N_HList)
    # print ("beta_dustList = ", beta_dustList)
    # print ("kappa_1300List = ", kappa_1300List)
    # print ("DustFileNameList = ", DustFileNameList)
    # print ("BackgroundFileNameList = ", BackgroundFileNameList)
    # print ("ContPhenFuncIDList = ", ContPhenFuncIDList)
    # print ("ContPhenFuncParam1List = ", ContPhenFuncParam1List)
    # print ("ContPhenFuncParam2List = ", ContPhenFuncParam2List)
    # print ("ContPhenFuncParam3List = ", ContPhenFuncParam3List)
    # print ("ContPhenFuncParam4List = ", ContPhenFuncParam4List)
    # print ("ContPhenFuncParam5List = ", ContPhenFuncParam5List)
    # print ("NoiseList = ", NoiseList)
    # print ("SmoothList = ", SmoothList)
    # print ("ThresholdList = ", ThresholdList)
    # print ("GlobalvLSRList = ", GlobalvLSRList)
    # print ("TelescopeSizeList = ", TelescopeSizeList)
    # print ("BMINList = ", BMINList)
    # print ("BMAJList = ", BMAJList)
    # print ("BPAList = ", BPAList)
    # print ("Inter_FlagList = ", Inter_FlagList)
    # print ("RedshiftList = ", RedshiftList)
    # print ("ErrorYList = ", ErrorYList)
    # print ("NumberHeaderLinesList = ", NumberHeaderLinesList)
    # print ("SeparatorColumnsList = ", SeparatorColumnsList)
    # print ("IsotopologuesList = ", IsotopologuesList)
    # print ("dbList = ", dbList)
    # print ("NumModelPixelXXList = ", NumModelPixelXXList)
    # print ("NumModelPixelYYList = ", NumModelPixelYYList)
    # print ("LocalOverlapFlagList = ", LocalOverlapFlagList)
    # print ("NoSubBeamFlagList = ", NoSubBeamFlagList)
    # print ("EmAbsPATHList = ", EmAbsPATHList)


    ## deactivate iso ratio file usage for single molecule fits
    IsotopologuesList = task_MAGIX.GetXMLtagNEW(LocalObsXMLFileName, "iso_flag")
    # IsotopologuesList = ["n"]


    ## store obs. xml file
    ObsXMLParameter = {}
    ObsXMLParameter['ExpFileList'] = ExpFileList
    ObsXMLParameter['NumberExpRangesList'] = NumberExpRangesList
    ObsXMLParameter['FreqMinList'] = FreqMinList
    ObsXMLParameter['FreqMaxList'] = FreqMaxList
    ObsXMLParameter['FreqStepList'] = FreqStepList
    ObsXMLParameter['t_back_flagList'] = t_back_flagList
    ObsXMLParameter['tBackList'] = tBackList
    ObsXMLParameter['tSlopeList'] = tSlopeList
    ObsXMLParameter['N_HList'] = N_HList
    ObsXMLParameter['beta_dustList'] = beta_dustList
    ObsXMLParameter['kappa_1300List'] = kappa_1300List
    ObsXMLParameter['DustFileNameList'] = DustFileNameList
    ObsXMLParameter['BackgroundFileNameList'] = BackgroundFileNameList
    ObsXMLParameter['ContPhenFuncIDList'] = ContPhenFuncIDList
    ObsXMLParameter['ContPhenFuncParam1List'] = ContPhenFuncParam1List
    ObsXMLParameter['ContPhenFuncParam2List'] = ContPhenFuncParam2List
    ObsXMLParameter['ContPhenFuncParam3List'] = ContPhenFuncParam3List
    ObsXMLParameter['ContPhenFuncParam4List'] = ContPhenFuncParam4List
    ObsXMLParameter['ContPhenFuncParam5List'] = ContPhenFuncParam5List
    ObsXMLParameter['NoiseList'] = NoiseList
    ObsXMLParameter['SmoothList'] = SmoothList
    ObsXMLParameter['ThresholdList'] = ThresholdList
    ObsXMLParameter['GlobalvLSRList'] = GlobalvLSRList
    ObsXMLParameter['TelescopeSizeList'] = TelescopeSizeList
    ObsXMLParameter['BMINList'] = BMINList
    ObsXMLParameter['BMAJList'] = BMAJList
    ObsXMLParameter['BPAList'] = BPAList
    ObsXMLParameter['Inter_FlagList'] = Inter_FlagList
    ObsXMLParameter['RedshiftList'] = RedshiftList
    ObsXMLParameter['ErrorYList'] = ErrorYList
    ObsXMLParameter['NumberHeaderLinesList'] = NumberHeaderLinesList
    ObsXMLParameter['SeparatorColumnsList'] = SeparatorColumnsList
    ObsXMLParameter['IsotopologuesList'] = IsotopologuesList
    ObsXMLParameter['IsoTableFileNameList'] = IsoTableFileNameList
    ObsXMLParameter['dbList'] = dbList
    ObsXMLParameter['NumModelPixelXXList'] = NumModelPixelXXList
    ObsXMLParameter['NumModelPixelYYList'] = NumModelPixelYYList
    ObsXMLParameter['LocalOverlapFlagList'] = LocalOverlapFlagList
    ObsXMLParameter['NoSubBeamFlagList'] = NoSubBeamFlagList
    ObsXMLParameter['EmAbsPATHList'] = EmAbsPATHList

    # Debug:
    # print ("ObsXMLParameter = ", ObsXMLParameter)


    ## we're done
    return ObsXMLParameter
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get key value of dictionary
##
def GetDicKey(LocalDictionary, LocalKey, ErrorValue):
    """

input parameters:
-----------------

    - LocalDictionary:          local dictionary

    - LocalKey:                 local key

    - ErrorValue:               error value


output parameters:
------------------

    - KeyValue:                 key value
    """

    # Debug:
    # print ("\n\nLocalDictionary = ", LocalDictionary)
    # print ("LocalKey = ", LocalKey)
    # print ("ErrorValue = ", ErrorValue)


    KeyValue = 0.0
    if (LocalDictionary is None or LocalKey is None):
        KeyValue = ErrorValue
    else:
        if (LocalKey in LocalDictionary):
            KeyValue = LocalDictionary[LocalKey]
            if (type(KeyValue) == list):                                                    ## check, if input parameter is a python list
                pass
            elif (type(KeyValue) == numpy.ndarray):
                if (KeyValue.any() is None):
                    KeyValue = ErrorValue
            else:
                if (KeyValue is None):
                    KeyValue = ErrorValue
        else:
            KeyValue = ErrorValue

        # Debug:
        # print ("KeyValue = ", KeyValue)


    ## we're done
    return KeyValue
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## import ASCII file
def ImportASCIIFile(ASCIIFileName):
    """

input parameters:
-----------------

    - ASCIIFileName:            path and name of ASCII file


output parameters:
------------------

    - InterpolASCIIFunc:        interpolation function to compute function on a new frequency axis
    """

    # Debug:
    # print ("ASCIIFileName = ", ASCIIFileName)


    ## initialize return variable
    InterpolASCIIFunc = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check path and name of ASCII file
    if (ASCIIFileName is None):
        ASCIIFileName = ""
    if (ASCIIFileName.strip() == ""):
        return InterpolASCIIFunc
    if (not ASCIIFileName.startswith("/")):
        LocalPath = os.getcwd() + "/"
        ASCIIFileName = LocalPath + ASCIIFileName
    if (not os.path.exists(ASCIIFileName)):
        return InterpolASCIIFunc


    ##------------------------------------------------------------------------------------------------------------------------------------------------
    ## import ASCII file
    FileContent = numpy.loadtxt(ASCIIFileName)


    ##------------------------------------------------------------------------------------------------------------------------------------------------
    ## get ASCII values at frequency points
    InterpolASCIIFunc = interp1d(FileContent[:, 0], FileContent[:, 1], fill_value = 'extrapolate')

    # Debug:
    # print ("InterpolASCIIFunc = ", InterpolASCIIFunc)


    ## we're done
    return InterpolASCIIFunc
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## get background file names
def GetBackgroundFileNames(LocalInternalParameterList, LocalMolNameFileName):
    """

input parameters:
-----------------

    - LocalInternalParameterList:   dictionary of internal parameters

    - LocalMolNameFileName:         name of current molecule subdirectory


output parameters:
------------------

    - LocalBackgroundFileList:      list of file names describing background spectra
    """

    # Debug:
    # print ("LocalInternalParameterList = ", LocalInternalParameterList)
    # print ("LocalMolNameFileName = ", LocalMolNameFileName)


    ## initialize return variable
    LocalBackgroundFileList = []


    ## get path of subdirectory containing background files
    BackgroundFilePath = GetDicKey(LocalInternalParameterList, 'BackgroundFilePath', "")
    BackgroundFilePath = BackgroundFilePath.strip()

    # Debug:
    # print ("BackgroundFilePath = ", BackgroundFilePath)


    ## get default molecule for background files
    BackgroundFileDefault = GetDicKey(LocalInternalParameterList, 'BackgroundFileDefault', "")
    BackgroundFileDefault = BackgroundFileDefault.strip()

    # Debug:
    # print ("BackgroundFileDefault = ", BackgroundFileDefault)


    ## get file names
    if (BackgroundFilePath != ""):
        LocalBackgroundDir = BackgroundFilePath + "/" + LocalMolNameFileName + "/"
        LocalBackgroundDir = os.path.normpath(LocalBackgroundDir) + "/"
        if (not os.path.isdir(LocalBackgroundDir) and BackgroundFileDefault != ""):
            LocalBackgroundDir = BackgroundFilePath + "/" + BackgroundFileDefault + "/"
            LocalBackgroundDir = os.path.normpath(LocalBackgroundDir) + "/"

        if (os.path.isdir(LocalBackgroundDir)):
            listing = os.listdir(LocalBackgroundDir)
            for LocalFile in listing:
                if (LocalFile.startswith("Background-File__")):
                    LocalBackgroundFileList.append(LocalBackgroundDir + LocalFile)

    # Debug:
    # print ("LocalBackgroundFileList = ", LocalBackgroundFileList)


    ## we're done
    return LocalBackgroundFileList
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## find background file name for given frequency range
def FindBackgroundFileName4Range(LocalBackgroundFileList, FreqMin, FreqMax):
    """

input parameters:
-----------------

    - LocalBackgroundFileList:      list of background file paths

    - FreqMin:                      min. frequency of current range

    - FreqMax:                      max. frequency of current range


output parameters:
------------------

    - NewLocalBackgroundFileName:   path and name of background file
    """

    # Debug:
    # print ("LocalBackgroundFileList = ", LocalBackgroundFileList)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)


    ## initialize return variable
    NewLocalBackgroundFileName = ""
    if (LocalBackgroundFileList == []):
        return NewLocalBackgroundFileName


    ## identify file
    MaxAxis = []
    for LocalBackgroundFileName in LocalBackgroundFileList:
        LocalBackgroundFile = numpy.loadtxt(LocalBackgroundFileName)
        FreqAxis = LocalBackgroundFile[:, 0]
        kkk = FreqAxis[(FreqMin <= FreqAxis[:]) & (FreqAxis[:] <= FreqMax)]
        if (len(kkk) > len(MaxAxis)):
            MaxAxis = copy.deepcopy(kkk)
            NewLocalBackgroundFileName = LocalBackgroundFileName
    if (len(MaxAxis) == 0):
        NewLocalBackgroundFileName = ""

    # Debug:
    # print ("NewLocalBackgroundFileName = ", NewLocalBackgroundFileName)


    ## we're done
    return NewLocalBackgroundFileName
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## define updated list of background file names
def DefineListBgFileNames(InternalParameterList, MolNameFileName, NumExpDataFiles, NumberExpRangesList, FreqMinList, FreqMaxList, BackgroundFileNameList):
    """

input parameters:
-----------------

    - InternalParameterList:        dictionary of internal parameters

    - MolNameFileName:              name of current molecule

    - NumExpDataFiles:              number of obs. data files

    - NumberExpRangesList:          list of range numbers for each obs. data files

    - FreqMinList:                  list of min. frequency of current range

    - FreqMaxList:                  list of max. frequency of current range

    - BackgroundFileNameList:       list of background file names


output parameters:
------------------

    - NewBackgroundFileNameList:   list of paths and names of background files
    """

    # Debug:
    # print ("InternalParameterList = ", InternalParameterList)
    # print ("MolNameFileName = ", MolNameFileName)
    # print ("NumExpDataFiles = ", NumExpDataFiles)
    # print ("NumberExpRangesList = ", NumberExpRangesList)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("BackgroundFileNameList = ", BackgroundFileNameList)


    ## initialize return parameter
    NewBackgroundFileNameList = []


    ## check if path of background files is defined
    LocalBackgroundFileList = GetBackgroundFileNames(InternalParameterList, MolNameFileName)


    ## define new list of background files
    for ObsDataFileIndex in range(NumExpDataFiles):                                         ## loop over all exp. data files


        ## get number of header lines and character separating columns from obs. xml file
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberRangeListIn = NumberExpRangesList)
        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']            ## get number of frequency ranges for current obs. data file


        ## get parameters for current frequency range
        for RangeIndex in range(NumberFrequencyRanges):                                     ## loop over all range definitions in the whole xml file


            ## get parameter for current frequency range
            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                            FreqMinListIn = FreqMinList, \
                                                                            FreqMaxListIn = FreqMaxList, \
                                                                            BackgroundFileNameListIn = BackgroundFileNameList)
            FreqMin = ObsXMLParameterDictRange['FreqMin']
            if (FreqMin is not None):
                FreqMax = ObsXMLParameterDictRange['FreqMax']
                BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                if (BackgroundFileName is None):
                    BackgroundFileName = FindBackgroundFileName4Range(LocalBackgroundFileList, FreqMin, FreqMax)
                NewBackgroundFileNameList.append([ObsDataFileIndex, RangeIndex, BackgroundFileName])


    ## we're done
    return NewBackgroundFileNameList
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get file name of a molecule
##
def MoleculeFileName(molecule):
    """

input parameters:
-----------------

    - molecule:             name of molecule


output parameters:
------------------

    - MoleculeFileName:     corresponding molecule file name
    """

    # Debug:
    # print ("molecule = ", molecule)


    ## initialize output parameter
    MoleculeFileName = ""


    ## remove not allowed characters from the molecule name
    MoleculeFileName = molecule
    MoleculeFileName = MoleculeFileName.replace(";", "_")
    MoleculeFileName = MoleculeFileName.replace(",", "_")
    MoleculeFileName = MoleculeFileName.replace("'", "_")
    MoleculeFileName = MoleculeFileName.replace("(", "_")
    MoleculeFileName = MoleculeFileName.replace(")", "_")
    MoleculeFileName = MoleculeFileName.replace("#", "_")


    ## define return parameter
    return MoleculeFileName
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get all transition energies from database
##
def GetTransFreq(MolNameList, FreqMin, FreqMax, SQLParamArray, dbFile, MaxNumTransitionsSQL, IncludeQNFlag = False, MinDistTransFreq = 2.0):
    """

input parameters:
-----------------

    - MolNameList:              name of molecule

    - FreqMin:                  min. freq. of range

    - FreqMax:                  max. freq. of range

    - SQLParamArray:            array containing sql parameters

    - dbFile:                   path and name of database file

    - MaxNumTransitionsSQL:     max. number of transitions

    - IncludeQNFlag:            (optional) include quantum number as well (default: False)

    - MinDistTransFreq:         (optional) minimal distance between two transition freq. (in MHz), (default: 2.0)


output parameters:
------------------

    - TransFreqList:            list containing the transition frequencies

    - DBParamList:              list containing the all important transition parameters for a each transition within FreqMin and FreqMax

    """

    # Debug:
    # print ("MolNameList = ", MolNameList)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("SQLParamArray = ", SQLParamArray)
    # print ("\n\ndbFile = ", dbFile)
    # print ("MaxNumTransitionsSQL = ", MaxNumTransitionsSQL)
    # print ("IncludeQNFlag = ", IncludeQNFlag)
    # print ("MinDistTransFreq = ", MinDistTransFreq)


    ## define internal parameters
    ElowMin = 0.0                                                                           ## lower limit for E_low (in K) for query string


    ## initialize return parameters
    TransFreqList = []
    DBParamList = []


    ## check, if database file is defined
    if (dbFile.strip() == ""):
        print ("\n\nError in function LineIdentification.GetTransFreq:")
        print ("\t\tThere is no database file defined!")
        print ("\n\tP A N I C !!!!")
        return (TransFreqList, DBParamList)


    ## check if one or more frequency ranges are defined
    if (isinstance(FreqMin, (float, int))):                                                 ## check if FreqMin parameter is a list or float/int
        FreqMinList = [FreqMin]
        FreqMaxList = [FreqMax]
    else:
        FreqMinList = copy.deepcopy(FreqMin)
        FreqMaxList = copy.deepcopy(FreqMax)
    if (len(FreqMinList) != len(FreqMaxList)):
        print ("\nError in function LineIdentification.GetTransFreq:")
        print ("\t\tThe length of the frequency range definitions are not equal!")
        print ("\t\tFreqMinList = ", FreqMinList)
        print ("\t\tFreqMaxList = ", FreqMaxList)


    ## check if at least one frequency range is selected
    if (len(FreqMinList) == 0):
        print ("\nError in function LineIdentification.GetTransFreq:")
        print ("\t\tThere is no frequency range defined!")
        print ("\t\tFreqMinList = ", FreqMinList)
        print ("\t\tFreqMaxList = ", FreqMaxList)
        return (TransFreqList, DBParamList)


    ## get sql parameters (if 'SQLParamArray' is not a list assume, that SQLParamArray defines upper limit for lower energy
    if (not isinstance(SQLParamArray, list)):
        MinNumTransitionsSQL = 1
        MaxNumTransitionsSQL = 0
        MaxElowSQL = SQLParamArray
        MingASQL = 0.0
        OrderTransSQL = 1
    else:
        MinNumTransitionsSQL = SQLParamArray[0]
        MaxNumTransitionsSQL = SQLParamArray[1]
        MaxElowSQL = SQLParamArray[2]
        MingASQL = SQLParamArray[3]
        OrderTransSQL = SQLParamArray[4]


    ## establish connection to database
    try:
        conn = sqlite3.connect(dbFile)
    except sqlite3.Error as e:
        print ("\nError in function LineIdentification.GetTransFreq:")
        print ("\t\tCan not connect to sqlite3 database {:s}.".format(dbFile))
        print ("\t\tError: {:d}: {:s}".format(e.args[0], e.args[1]))
        sys.exit(1)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get entries from database
    if (not isinstance(MolNameList, list)):
        MolNameList = [MolNameList]


    ## check, if name of molecule contains invalid character
    #    MolName = LocalMolName
    #    i = MolName.find(chr(34))
    #    if (i > (-1)):
    #        MolName.translate(None, chr(34))


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define query string


    ## define names of columns and table
    NameOfRadTransTable = "transitions"
    ColumnNameForNameTransitions = "T_Name"
    ColumnNameForFreqTransitions = "T_Frequency"
    ColumnNameForIntTransitions = "T_Intensity"
    ColumnNameForEinsteinATransitions = "T_EinsteinA"
    ColumnNameForFreqErrTransitions = "T_Uncertainty"
    ColumnNameForELowTransitions = "T_EnergyLower"
    ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
    ColumnNameForQNUpLabelTransitions = "T_UpperStateQuantumNumbers"
    ColumnNameForQNLowLabelTransitions = "T_LowerStateQuantumNumbers"
    ColumnNameForURLTransitions = "T_URL"


    ## define order of transitions
    if (OrderTransSQL == 1):                                                                ## order by lower energy
        OrderString = ColumnNameForELowTransitions
    elif (OrderTransSQL == 2):                                                              ## order by gA
        OrderString = ColumnNameForgUpTransitions + " * " + ColumnNameForEinsteinATransitions
    elif (OrderTransSQL == 3):                                                              ## order by gA
        OrderString = "(" + ColumnNameForgUpTransitions + " * " + ColumnNameForEinsteinATransitions + ") / ("
        OrderString += "NULLIF(" + ColumnNameForELowTransitions + ", 0) * NULLIF(" + ColumnNameForELowTransitions + ", 0) )"
        #OrderString += "POWER(ISNULL(NULLIF(" + ColumnNameForELowTransitions + ", 0), 1), 2))"
    else:                                                                                   ## order by frequency
        OrderString = ColumnNameForFreqTransitions


    ## start building query
    if (MaxNumTransitionsSQL > 0 and MaxNumTransitionsSQL != 1.e99):
        query_string = "SELECT * FROM (SELECT "
    else:
        query_string = "SELECT "
    query_string += ColumnNameForNameTransitions + ", "                                     ## position 1: molecule name
    query_string += ColumnNameForFreqTransitions + ", "                                     ## position 2: frequency
    query_string += ColumnNameForIntTransitions + ", "                                      ## position 3: intensity
    query_string += ColumnNameForEinsteinATransitions + ", "                                ## position 4: Einstein A
    query_string += ColumnNameForFreqErrTransitions + ", "                                  ## position 5: Error frequency
    query_string += ColumnNameForELowTransitions + ", "                                     ## position 6: E_low
    query_string += ColumnNameForgUpTransitions + ", "                                      ## position 7: upper state degeneracy
    query_string += ColumnNameForQNUpLabelTransitions + ", "                                ## position 8: quantum number label for upper state
    query_string += ColumnNameForQNLowLabelTransitions                                      ## position 9: quantum number label for lower state
    query_string += " FROM " + NameOfRadTransTable + " WHERE (("                            ## define table


    ## select frequency ranges
    for RangeID in range(len(FreqMinList)):                                                 ## loop over all ranges defined by FreqMin and FreqMax
        LocalFreqMin = FreqMinList[RangeID]                                                 ## get min. frequency of current range
        LocalFreqMax = FreqMaxList[RangeID]                                                 ## get max. frequency of current range
        if (RangeID > 0):
            query_string += " or "
        query_string += ColumnNameForFreqTransitions + " >= " + str(LocalFreqMin) + " and " + ColumnNameForFreqTransitions + " <= " + str(LocalFreqMax)
    query_string += ") and ("


    ## select molecules
    for LocalMolNameID, LocalMolName in enumerate(MolNameList):
        MolName = LocalMolName
        i = MolName.find(chr(34))
        if (i > (-1)):
            MolName.translate(None, chr(34))
        if (len(MolNameList) > 1 and LocalMolNameID > 0):
            query_string += " or "
        query_string += ColumnNameForNameTransitions + " = " + chr(34) + MolName.strip() + chr(34)
    query_string += ")"


    ## limits for lower energy and minimal gA
    if (MaxElowSQL > 0.0):
        h = MaxElowSQL / 1.42879
        query_string += " and " + ColumnNameForELowTransitions + " <= " + str(h)
    if (MingASQL > 0.0):
        query_string += " and (" + ColumnNameForgUpTransitions + " * " + ColumnNameForEinsteinATransitions + ") >= " + str(MingASQL)
    query_string += ")"


    # query_string += ColumnNameForNameTransitions + " = " + chr(34) + MolName.strip() + chr(34) + "))"
    if (MaxNumTransitionsSQL > 0 and MaxNumTransitionsSQL != 1.e99):
        query_string += " ORDER BY " + OrderString + " DESC limit "
        query_string += str(MaxNumTransitionsSQL) + ")"

    # Debug:
    # print ("\n\n\n")
    # print ("MolNameList = ", MolNameList)
    # print ("dbFile = ", dbFile)
    # print ("query_string = >>" + query_string + "<<")
    # print ("len(query_string) = ", len(query_string))
    #    sys.exit(0)


    ## read data from database
    cursor = conn.cursor()
    try:
        cursor.execute(query_string)
    except:
        print ("\n\n\n")
        print ("query_string = ", query_string)
        sys.exit(0)
    rows = cursor.fetchall()


    ## store entries in output variable Contents
    LocalCounterTransitions = 0
    for row in rows:                                                                        ## loop over all entries of the database


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define formatted line
        ## position 0: molecule name
        ## position 1: frequency
        ## position 2: intensity
        ## position 3: Einstein A
        ## position 4: Error frequency
        ## position 5: E_low
        ## position 6: upper state degeneracy
        ## position 7: quantum number label for upper state
        ## position 8: quantum number label for lower state
        ## position 9: URL
        FreqString = ""
        EinsteinAString = ""
        ElowString = ""
        gUpString = ""
        for element_counter, elements in enumerate(row):                                    ## loop over all elements in a row


            ## get name of molecule
            if (element_counter == 0):
                LocalMoleculeName = "{:s}".format(str(elements))

                # Debug:
                # print ("LocalMoleculeName = ", LocalMoleculeName)


            ## get transition frequency (in MHz)
            elif (element_counter == 1):
                freqVal = '{:17s}'.format(str(elements))
                if (freqVal.strip() != "None"):
                    freqVal = float(freqVal)
                    FreqString = '{:17.5f}'.format(freqVal)
                else:
                    FreqString = freqVal
                TransFreq = float(FreqString)

                # Debug:
                # print ("FreqString = ", FreqString)


            ## get Einstein A coefficient
            elif (element_counter == 3):
                EinsteinAVal = '{:26s}'.format(str(elements))
                if (EinsteinAVal.strip() != "None"):
                    EinsteinAVal = float(EinsteinAVal)
                    EinsteinAString = '{:26.3e}'.format(EinsteinAVal)
                else:
                    EinsteinAString = EinsteinAVal
                EinsteinA = float(EinsteinAString)

                # Debug:
                # print ("EinsteinAString = ", EinsteinAString)


            ## get E_lower (in cm-1)
            elif (element_counter == 5):
                elowVal = '{:19s}'.format(str(elements))
                if (elowVal.strip() != "None"):
                    elowVal = float(elowVal) * 1.438769                                     ## convert cm-1 to K
                    if (abs(elowVal) > 0.01 and abs(elowVal) < 10000):
                        ElowString = '{:19.3f}'.format(elowVal)
                    else:
                        ElowString = '{:19.3e}'.format(elowVal)
                else:
                    ElowString = elowVal
                Elow = float(ElowString)

                # Debug:
                # print ("ElowString = ", ElowString)


            ## get upper state degeneracy
            elif (element_counter == 6):
                gUpVal = '{:23s}'.format(str(elements))
                if (gUpVal.strip() != "None"):
                    gUpVal = float(gUpVal)
                    gUpString = '{:25.1e}'.format(gUpVal)
                else:
                    gUpString = gUpVal
                gup = float(gUpString)

                # Debug:
                # print ("gUpString = ", gUpString)


            ## get quantum number label for upper state
            elif (element_counter == 7 and IncludeQNFlag):
                QNUpperState = '{:s}'.format(str(elements))

                # Debug:
                # print ("QNUpperState = ", QNUpperState)


            ## get quantum number label for lower state
            elif (element_counter == 8 and IncludeQNFlag):
                QNLowerState = '{:s}'.format(str(elements))

                # Debug:
                # print ("QNLowerState = ", QNLowerState)


        ## check, if transition frequencies have enough distance
        IncludeFlag = False
        if (TransFreqList == []):
            IncludeFlag = True
        else:
            NPTransFreqList = numpy.asarray(TransFreqList)
            TransFreqDistance = numpy.nanmin(numpy.abs(NPTransFreqList - TransFreq))
            if (TransFreqDistance > MinDistTransFreq or MaxNumTransitionsSQL == 1.e99 or MaxNumTransitionsSQL == 0.0):
                IncludeFlag = True
            ## to do: if two transition frequencies are located to close together, special handling is necessary
            # else:


        ## store molecular parameters
        if (IncludeFlag):
            TransFreqList.append(TransFreq)
            if (IncludeQNFlag):
                DBParamList.append([TransFreq, EinsteinA, Elow, gup, LocalMoleculeName, QNUpperState, QNLowerState])
            else:
                DBParamList.append([TransFreq, EinsteinA, Elow, gup, LocalMoleculeName])

    # Debug:
    # print ("\n\nMolName = ", MolName)
    # print ("LocalFreqMin, LocalFreqMax = ", LocalFreqMin, LocalFreqMax)
    # print ("ElowMin, SQLParamArray = ", ElowMin, SQLParamArray)
    # print (">>>>>>>>>>>>>>>>>>>>>>>>>>TransFreqList = ", TransFreqList)
    # print ("rows = ", rows)
    # print ("\n\n\n")
    #    sys.exit(0)


    ## close connection to database
    conn.close()


    ## define return parameter
    return (TransFreqList, DBParamList)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define new frequency ranges
##
##
## to do: extend function to handle list of molecules MolName -> [Mol1, Mol2, ...]
##
##
def DefineNewFreqRanges(LowestDataPoint, HighestDataPoint, MolName, FreqMin, FreqMax, FreqStep, t_back_flag, tBack, tSlope, nH_flag, N_H, beta_dust, \
                        kappa_1300, DustFileName, BackgroundFileName, ContPhenFuncID, \
                        ContPhenFuncParam1, ContPhenFuncParam2, ContPhenFuncParam3, ContPhenFuncParam4, ContPhenFuncParam5, Noise, velLowLimit, \
                        velUpLimit, MaxWidth, SQLParamArray, dbFile, LocalMaxNumTransInFit, FreqMinOrig, TransFreqList = None, MinimalDist = 2.0):
    """

input parameters:
-----------------

    - LowestDataPoint:          lowest frequency of the current exp. data file

    - HighestDataPoint:         highest frequency of the current exp. data file

    - MolName:                  name of current molecule

    - FreqMin:                  lowest frequency for current frequency range

    - FreqMax:                  highest frequency for current frequency range

    - FreqStep:                 step size for current frequency range

    - t_back_flag:              flag indicating if continuum is described completely by tBack and tSlope

    - tBack:                    background temperature for current frequency range

    - tSlope:                   temperature slope for current frequency range

    - nH_flag:                  flag indicating if dust parameters are defined globally or not

    - N_H:                      hydrogen column density for current frequency range

    - beta_dust:                spectral index for current frequency range

    - kappa_1300:               kappa for current frequency range

    - DustFileName:             path and name of file describing dust contribution

    - BackgroundFileName:       path and name of file describing background

    - ContPhenFuncID:           function ID used for phenomenological description

    - ContPhenFuncParam1:       parameter 1 for phenomenological description

    - ContPhenFuncParam2:       parameter 2 for phenomenological description

    - ContPhenFuncParam3:       parameter 3 for phenomenological description

    - ContPhenFuncParam4:       parameter 4 for phenomenological description

    - ContPhenFuncParam5:       parameter 5 for phenomenological description

    - Noise:                    noise level for current frequency range

    - velLowLimit:              mini. lower limit of velocity offset for current molecule

    - velUpLimit:               max. upper limit of velocity offset for current molecule

    - MaxWidth:                 max. velocity width for current molecule

    - SQLParamArray:            array containing sql parameters

    - dbFile:                   path and name of database file

    - LocalMaxNumTransInFit:    local max. number of transitions which are taken into account

    - FreqMinOrig:              unshifted lower frequency

    - TransFreqList:            (optional) list of transition frequencies (Default: None)

    - MinimalDist:              (optional) minimal distance between two transition frequencies as multiple of frequency step (Default: 2.0)


output parameters:
------------------

    - NewFreqRanges:            list containing all parameters suitable for new freq. ranges

    """

    # Debug:
    # print ("LowestDataPoint = ", LowestDataPoint)
    # print ("HighestDataPoint = ", HighestDataPoint)
    # print ("MolName = ", MolName)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("FreqStep = ", FreqStep)
    # print ("t_back_flag = ", t_back_flag)
    # print ("tBack = ", tBack)
    # print ("tSlope = ", tSlope)
    # print ("nH_flag = ", nH_flag)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("Noise = ", Noise)
    # print ("velLowLimit = ", velLowLimit)
    # print ("velUpLimit, ", velUpLimit)
    # print ("MaxWidth = ", MaxWidth)
    # print ("TransFreqList = ", TransFreqList)
    # print ("MinimalDist = ", MinimalDist)


    ## reset output parameter
    NewFreqRanges = []


    ## define speed of light (m/s) and (km/s)
    LineWidthNr = 1.0
    SimSize = int((FreqMax - FreqMin)/FreqStep) + 1


    ## get transition frequencies in current frequency range
    if (TransFreqList is None):
        TransFreqList, DBParamList = GetTransFreq(MolName, FreqMin, FreqMax, SQLParamArray, dbFile, LocalMaxNumTransInFit)
        TransFreqList.sort()

    # Debug:
    # print ("\n\n\n****************************************************")
    # print ("MolName = ", MolName)
    # print ("FreqMin, FreqMax = ", FreqMin, FreqMax)
    # print ("SQLParamArray = ", SQLParamArray)
    # print ("dbFile = ", dbFile)
    # print ("LocalMaxNumTransInFit = ", LocalMaxNumTransInFit)
    # print ("TransFreqList = ", TransFreqList)
    # print (TransFreqList != [])
    # print ("len(TransFreqList) = ", len(TransFreqList), "\n\n")


    ## determine ranges
    if (len(TransFreqList) > 0):


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine new frequency ranges
        FreqRangeList = []
        MultiplicityOfWidth = 1.0                                                           ## set multiplicity of half of velocity width
        for TransIndex in range(len(TransFreqList)):                                        ## loop over all transition frequencies
            if (TransIndex <= LocalMaxNumTransInFit):
                TransFreq = TransFreqList[TransIndex]


                ## get min/max of lower/upper limits
                v1 = velLowLimit - MultiplicityOfWidth * (MaxWidth / 2.0)
                v2 = velUpLimit + MultiplicityOfWidth * (MaxWidth / 2.0)
                r1 = task_myXCLASS.ConvertFreq(TransFreq, v1)
                r2 = task_myXCLASS.ConvertFreq(TransFreq, v2)
                LowFreq = max(min(r1, r2), FreqMin)
                HighFreq = min(max(r1, r2), FreqMax)

                # Debug:
                # print ("\nTransFreq = ", TransFreq)
                # print ("velLowLimit = ", velLowLimit)
                # print ("velUpLimit = ", velUpLimit)
                # print ("v1 = ", v1)
                # print ("v2 = ", v2)
                # print ("LowFreq = ", LowFreq)
                # print ("HighFreq = ", HighFreq)


                ## make sure, that non-shifted transition frequency is always included in new freq. range
                if (TransFreq < LowFreq):
                    LowFreq = max((TransFreq - 10.0), FreqMin)
                if (TransFreq > HighFreq):
                    HighFreq = min((TransFreq + 10.0), FreqMax)


                ## check, if new frequency ranges are within frequency range defined in exp. data file
                if (LowestDataPoint > LowFreq):
                    LowFreq = LowestDataPoint
                if (HighestDataPoint < HighFreq):
                    HighFreq = HighestDataPoint
                if (LowFreq >= HighFreq):
                    AddNewRangeFlag = False
                else:

                    # Debug:
                    # print ("MaxWidth = ", MaxWidth)
                    # print ("LowFreq = ", LowFreq)
                    # print ("HighFreq = ", HighFreq)
                    # print ("\n\n\n")


                    ## check, if part of current freq. range is already covered by a previous range
                    AddNewRangeFlag = True
                    for i in range(len(FreqRangeList)):
                        lf = FreqRangeList[i][0]
                        uf = FreqRangeList[i][1]


                        ## check case:  overlapping range definitions
                        ##  lf|                 |uf
                        ##          LowFreq|            |HighFreq
                        if (LowFreq <= uf and uf <= HighFreq):
                            FreqRangeList[i][1] = HighFreq
                            AddNewRangeFlag = False
                            break


                        ## check case:  overlapping range definitions
                        ##  lf|                             |uf
                        ##          LowFreq|    |HighFreq
                        elif (lf <= LowFreq and HighFreq <= uf):
                            AddNewRangeFlag = False
                            break


                        ## check case:  overlapping range definitions
                        ##          lf|                     |uf
                        ##  LowFreq|            |HighFreq
                        elif (LowFreq <= lf and HighFreq <= uf):
                            FreqRangeList[i][0] = LowFreq
                            AddNewRangeFlag = False
                            break


                ## add new range if range is not covered before
                if (AddNewRangeFlag):
                    if (abs(HighFreq - LowFreq) >= MinimalDist * FreqStep):
                        FreqRangeList.append([LowFreq, HighFreq, TransFreq])

                        # Debug:
                        # print ("TransFreq, LowFreq, HighFreq = ", TransFreq, LowFreq, HighFreq)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## construct new ranges
        for NewRange in FreqRangeList:
            LowFreq = NewRange[0]
            HighFreq = NewRange[1]
            TransFreq = NewRange[2]

            # Debug:
            # print ("(HighFreq - LowFreq) / FreqStep = ", (HighFreq - LowFreq) / FreqStep)


            ## determine new background temperature
            NewTBack = tBack * (LowFreq / FreqMinOrig)**tSlope


            ## add parameters to NewFreqRanges list
            NewFreqRanges.append([str(LowFreq), str(HighFreq), str(FreqStep), str(t_back_flag), str(NewTBack), str(tSlope), str(nH_flag), str(N_H), \
                                  str(beta_dust), str(kappa_1300), DustFileName, BackgroundFileName, \
                                  str(ContPhenFuncID), str(ContPhenFuncParam1), str(ContPhenFuncParam2), \
                                  str(ContPhenFuncParam3), str(ContPhenFuncParam4), str(ContPhenFuncParam5), str(Noise), str(TransFreq)])
    # Debug:
    # print ("len(NewFreqRanges) = ", len(NewFreqRanges))
    # print ("NewFreqRanges = ", NewFreqRanges)


    ## define return parameter
    return NewFreqRanges
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## create new observational xml file for each molecule
##
##
## to do: extend function to handle list of molecules MolName -> [Mol1, Mol2, ...]
##
##
def CreateXMLFile(MolName, velLowLimit, velUpLimit, MaxWidth, ExpXML, ExpFileList, NumberExpRangesList, FreqMinList, FreqMaxList, FreqStepList, \
                  t_back_flagList, tBackList, tSlopeList, N_HList, beta_dustList, kappa_1300List, DustFileNameList, BackgroundFileNameList, \
                  ContPhenFuncIDList, ContPhenFuncParam1List, ContPhenFuncParam2List, \
                  ContPhenFuncParam3List, ContPhenFuncParam4List, ContPhenFuncParam5List, NoiseList, SmoothList, TelescopeSizeList, BMINList, BMAJList, \
                  BPAList, InterFlagList, GlobalvLSRList, RedshiftList, ErrorYList, NumberHeaderLinesList, SeparatorColumnsList, IsotopologuesList, \
                  IsoTableFileNameList, dbList, NumModelPixelXXList, NumModelPixelYYList, LocalOverlapFlagList, SQLParamArray, \
                  LocalMaxNumTransInFit, NoSubBeamFlagList, EmAbsPATHList):
    """

input parameters:
-----------------

    - MolName:                  name of current molecule

    - velLowLimit:              lower velocity limit

    - velUpLimit:               upper velocity limit

    - MaxWidth:                 max. velocity width

    - ExpXML:                   path and name of exp xml file

    - ExpFileList:              list of observational data files

    - NumberExpRangesList:      list of frequency range numbers

    - FreqMinList:              list of lowest frequency for each range

    - FreqMaxList:              list of highest frequency for each range

    - FreqStepList:             list of step sizes for each range

    - t_back_flagList:          list of all T_back flags

    - tBackList:                list of background temperatures

    - tSlopeList:               list of temperature slopes

    - N_HList:                  list of hydrogen column densities

    - beta_dustList:            list of spectral indices

    - kappa_1300List:           list of kappa values

    - DustFileNameList:         list of paths and names of dust file(s)

    - BackgroundFileNameList:   list of paths and names of background file(s)

    - ContPhenFuncIDList:       list of function ids for phenomenological continuum description

    - ContPhenFuncParam1List:   list of first parameters for phenomenological continuum description

    - ContPhenFuncParam2List:   list of second parameters for phenomenological continuum description

    - ContPhenFuncParam3List:   list of third parameters for phenomenological continuum description

    - ContPhenFuncParam4List:   list of fourth parameters for phenomenological continuum description

    - ContPhenFuncParam5List:   list of fifth parameters for phenomenological continuum description

    - TelescopeSizeList:        list of telescope sizes

    - BMINList:                 list of telescope parameters BMIN

    - BMAJList:                 list of telescope parameters BMAJ

    - BPAList:                  list of telescope parameters BPA

    - InterFlagList:            list of interferometer flags

    - GlobalvLSRList:           list of vLSR values

    - RedshiftList:             list of redshifts

    - NoiseList:                list of noise levels

    - SmoothList:               list of smooth values

    - ErrorYList:               list error flags

    - NumberHeaderLinesList:    list of numbers of header lines

    - SeparatorColumnsList:     list of separator characters

    - IsotopologuesList:        iso flag

    - IsoTableFileNameList:     path and name of iso flag

    - dbList:                   path and name of database file

    - NumModelPixelXXList:      number of pixel for sub-beam modeling along x-direction

    - NumModelPixelYYList:      number of pixel for sub-beam modeling along y-direction

    - LocalOverlapFlagList:     flag indicating if local-overlap description is used

    - NoSubBeamFlagList:        flag for preventing sub-beam description

    - EmAbsPATHList:            path of files describing emission and absorption functions

    - SQLParamArray:            array containing sql parameters

    - LocalMaxNumTransInFit:    local max. number of transitions which are taken into account


output parameters:
------------------

    - None

    """


    # Debug:
    # print ("\n\n\nMolName = ", MolName)
    # print ("velLowLimit = ", velLowLimit)
    # print ("velUpLimit = ", velUpLimit)
    # print ("MaxWidth = ", MaxWidth)
    # print ("ExpXML = ", ExpXML)
    # print ("ExpFileList = ", ExpFileList)
    # print ("NumberExpRangesList = ", NumberExpRangesList)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("FreqStepList = ", FreqStepList)
    # print ("t_back_flagList = ", t_back_flagList)
    # print ("tBackList = ", tBackList)
    # print ("tSlopeList = ", tSlopeList)
    # print ("N_HList = ", N_HList)
    # print ("beta_dustList = ", beta_dustList)
    # print ("kappa_1300List = ", kappa_1300List)
    # print ("DustFileNameList = ", DustFileNameList)
    # print ("BackgroundFileNameList = ", BackgroundFileNameList)
    # print ("ContPhenFuncIDList = ", ContPhenFuncIDList)
    # print ("ContPhenFuncParam1List = ", ContPhenFuncParam1List)
    # print ("ContPhenFuncParam2List = ", ContPhenFuncParam2List)
    # print ("ContPhenFuncParam3List = ", ContPhenFuncParam3List)
    # print ("ContPhenFuncParam4List = ", ContPhenFuncParam4List)
    # print ("ContPhenFuncParam5List = ", ContPhenFuncParam5List)
    # print ("TelescopeSizeList = ", TelescopeSizeList)
    # print ("BMINList = ", BMINList)
    # print ("BMAJList = ", BMAJList)
    # print ("BPAList = ", BPAList)
    # print ("InterFlagList = ", InterFlagList)
    # print ("GlobalvLSRList = ", GlobalvLSRList)
    # print ("RedshiftList = ", RedshiftList)
    # print ("NoiseList = ", NoiseList)
    # print ("SmoothList = ", SmoothList)
    # print ("ErrorYList = ", ErrorYList)
    # print ("NumberHeaderLinesList = ", NumberHeaderLinesList)
    # print ("SeparatorColumnsList = ", SeparatorColumnsList)
    # print ("IsotopologuesList = ", IsotopologuesList)
    # print ("IsoTableFileNameList = ", IsoTableFileNameList)
    # print ("dbList = ", dbList)
    # print ("NumModelPixelXXList = ", NumModelPixelXXList)
    # print ("NumModelPixelYYList = ", NumModelPixelYYList)
    # print ("LocalOverlapFlagList = ", LocalOverlapFlagList)
    # print ("NoSubBeamFlagList = ", NoSubBeamFlagList)
    # print ("EmAbsPATHList = ", EmAbsPATHList)
    # print ("SQLParamArray = ", SQLParamArray)
    # print ("LocalMaxNumTransInFit = ", LocalMaxNumTransInFit)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get a list of all transition frequencies in all selected frequency ranges


    ## define list of min and max. frequencies of each range
    dbFile = dbList[0]
    NumExpDataFiles = len(ExpFileList)                                                      ## get number of obs. data files
    ShiftedFreqMinList = []
    ShiftedFreqMaxList = []
    for ObsDataFileIndex in range(NumExpDataFiles):                                         ## loop over all exp. data files


        ## get number of header lines and character separating columns from obs. xml file
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberRangeListIn = NumberExpRangesList, \
                                                                       GlobalvLSRListIn = GlobalvLSRList, Redshift_ListIn = RedshiftList)
        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']            ## get number of frequency ranges for current obs. data file
        GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
        if (GlobalvLSR is None):
            GlobalvLSR = 0.0
        Redshift = ObsXMLParameterDictFile['Redshift']
        if (Redshift is None):
            Redshift = 0.0


        ## shift frequency ranges
        for RangeIndex in range(NumberFrequencyRanges):                                     ## loop over all frequency ranges


            ## get parameter for current frequency range
            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = FreqMinList, \
                                                                            FreqMaxListIn = FreqMaxList)
            FreqMin = ObsXMLParameterDictRange['FreqMin']
            if (FreqMin is not None):
                FreqMax = ObsXMLParameterDictRange['FreqMax']


                ## shift min. and max. frequency of each range by v_LSR or redshift z
                NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
                NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
                LocalFreqMin = min(NewFreqMin, NewFreqMax)
                LocalFreqMax = max(NewFreqMin, NewFreqMax)
                ShiftedFreqMinList.append(LocalFreqMin)
                ShiftedFreqMaxList.append(LocalFreqMax)


    ## get isotopologues (if defined) for current molecule
    LocalListOfMolecules = [MolName]
    IsoFlag = IsotopologuesList[0].lower()
    IsoFlag = task_myXCLASSMapFit.CheckBool(IsoFlag)
    if (IsoFlag):
        IsoFileName = IsoTableFileNameList[0]


        ## read in iso ratio file
        NewLocalIsoRatioFileName = ""
        IsoRatioTable, Isotopologues, IsoMolecule = task_myXCLASS.ImportIsoRatioFile(IsoFileName, NewLocalIsoRatioFileName)


        ## get all isotopologues which correspond to the current molecule
        for IsoIndex in range(len(IsoRatioTable)):
            IsoRatioTableLine = IsoRatioTable[IsoIndex]
            IsoMaster = IsoRatioTableLine[1].strip()
            if (IsoMaster == MolName):
                LocalIsotopologue = IsoRatioTableLine[0].strip()
                if (not LocalIsotopologue in LocalListOfMolecules):
                    LocalListOfMolecules.append(LocalIsotopologue)


    ## get all transition parameters for main molecule within selected ranges
    TransFreqList, DBParamList = GetTransFreq(LocalListOfMolecules, ShiftedFreqMinList, ShiftedFreqMaxList, SQLParamArray, dbFile, LocalMaxNumTransInFit)
    TransFreqNPList = numpy.asarray(TransFreqList)

    # Debug:
    # print ("\n\nLocalMaxNumTransInFit = ", LocalMaxNumTransInFit)
    # print ("TransFreqList = ", TransFreqList)
    # print ("DBParamList = ", DBParamList)
    # print ("TransFreqNPList = ", TransFreqNPList)
    # print ("ShiftedFreqMinList = ", ShiftedFreqMinList)
    # print ("ShiftedFreqMaxList = ", ShiftedFreqMaxList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create new xml file for current xml file


    ## open xml file and write header
    NewExpXMLFile = open(ExpXML, 'w')
    NewExpXMLFile.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
    NewExpXMLFile.write("<ExpFiles>\n\n\n")
    NewExpXMLFile.write("    <!-- define number of observation files -->\n")


    ## write parameters for each observational data file
    CounterTransitions = abs(LocalMaxNumTransInFit)
    EffectiveObsDataFileCounter = 0
    LinesForXMLFile = []
    for ObsDataFileIndex in range(NumExpDataFiles):                                         ## loop over all obs. data files
        helpString = "*" * 141
        LinesForObsFile = ["\n\n    <!-- " + helpString + " -->\n"]
        LinesForObsFile.append("    <!-- define parameters for observation file " + chr(34)
                                     + os.path.basename(ExpFileList[ObsDataFileIndex][1]) + chr(34) + " -->\n")
        LinesForObsFile.append("    <file>\n")
        LinesForObsFile.append("        <FileNamesExpFiles>" + ExpFileList[ObsDataFileIndex][1] + "</FileNamesExpFiles>\n")
        LinesForObsFile.append("\n\n        <!-- define import filter -->\n")
        LinesForObsFile.append("        <ImportFilter>xclassASCII</ImportFilter>\n")


        ## get number of header lines and character separating columns from obs. xml file
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberRangeListIn = NumberExpRangesList, \
                                                                       TelescopeSizeListIn = TelescopeSizeList, BMIN_ListIn = BMINList, \
                                                                       BMAJ_ListIn = BMAJList, BPA_ListIn = BPAList, InterFlagListIn = InterFlagList, \
                                                                       GlobalvLSRListIn = GlobalvLSRList, Redshift_ListIn = RedshiftList, \
                                                                       ErrorYFlagListIn = ErrorYList, NumberHeaderLinesListIn = NumberHeaderLinesList, \
                                                                       SeparatorColumnsListIn = SeparatorColumnsList)
        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']             ## get number of frequency ranges for current obs. data file
        TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
        LocalBMIN = ObsXMLParameterDictFile['BMIN']
        LocalBMAJ = ObsXMLParameterDictFile['BMAJ']
        LocalBPA = ObsXMLParameterDictFile['BPA']
        InterFlag = ObsXMLParameterDictFile['InterFlag']
        GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
        if (GlobalvLSR is None):
            GlobalvLSR = 0.0
        LocalRedshift = ObsXMLParameterDictFile['Redshift']
        if (LocalRedshift is None):
            LocalRedshift = 0.0
        ErrorY = ObsXMLParameterDictFile['ErrorY']
        NumberHeaderLines = ObsXMLParameterDictFile['NumberHeaderLines']
        LocalNumberHeaderLines = NumberHeaderLines
        SeparatorColumns = ObsXMLParameterDictFile['SeparatorColumns']


        ## determine lowest and highest data point
        if (LocalNumberHeaderLines is None):
            LocalNumberHeaderLines = 0
        data = numpy.loadtxt(ExpFileList[ObsDataFileIndex][1], skiprows = int(LocalNumberHeaderLines))
        LowestDataPoint = min(data[:,0])
        HighestDataPoint = max(data[:,0])

        # Debug:
        # print ("LowestDataPoint = ", LowestDataPoint)
        # print ("HighestDataPoint = ", HighestDataPoint)


        ## determine new frequency range definitions and write to xml file
        TotalNumberFreqRanges = 0
        for RangeIndex in range(NumberFrequencyRanges):                                     ## loop over all frequency ranges


            ## get parameter for current frequency range
            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = FreqMinList, \
                                                                            FreqMaxListIn = FreqMaxList, FreqStepListIn = FreqStepList, \
                                                                            tBackFlagListIn = t_back_flagList, tBackListIn = tBackList, \
                                                                            tSlopeListIn = tSlopeList, N_HListIn = N_HList, \
                                                                            beta_dustListIn = beta_dustList, kappa_1300ListIn = kappa_1300List, \
                                                                            DustFileNameListIn = DustFileNameList, \
                                                                            BackgroundFileNameListIn = BackgroundFileNameList, \
                                                                            ContPhenFuncID_ListIn = ContPhenFuncIDList, \
                                                                            ContPhenFuncParam1_ListIn = ContPhenFuncParam1List, \
                                                                            ContPhenFuncParam2_ListIn = ContPhenFuncParam2List, \
                                                                            ContPhenFuncParam3_ListIn = ContPhenFuncParam3List, \
                                                                            ContPhenFuncParam4_ListIn = ContPhenFuncParam4List, \
                                                                            ContPhenFuncParam5_ListIn = ContPhenFuncParam5List, \
                                                                            NoiseListIn = NoiseList, SmoothValueListIn = SmoothList)
            FreqMin = ObsXMLParameterDictRange['FreqMin']
            if (FreqMin is not None):
                FreqMax = ObsXMLParameterDictRange['FreqMax']
                FreqStep = ObsXMLParameterDictRange['FreqStep']
                t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                tBack = ObsXMLParameterDictRange['tBack']
                tSlope = ObsXMLParameterDictRange['tSlope']
                nH_flag = True
                N_H = ObsXMLParameterDictRange['N_H']
                beta_dust = ObsXMLParameterDictRange['beta_dust']
                kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                if (N_H is None or beta_dust is None or kappa_1300 is None):
                    nH_flag = False
                    N_H = 0.0
                    beta_dust = 0.0
                    kappa_1300 = 0.0
                DustFileName = ObsXMLParameterDictRange['DustFileName']
                BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']
                Noise = ObsXMLParameterDictRange['NoiseLevel']
                if (Noise is None):
                    Noise = 0.0
                SmoothValue = ObsXMLParameterDictRange['SmoothValue']


                ## shift min. and max. frequency of each range by v_LSR or redshift z
                FreqMinOrig = FreqMin
                NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = LocalRedshift, backTrafo = True)
                NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = LocalRedshift, backTrafo = True)
                ShiftedFreqMin = min(NewFreqMin, NewFreqMax)
                ShiftedFreqMax = max(NewFreqMin, NewFreqMax)


                ## shift min. and max. frequency of obs. data file by v_LSR or redshift z
                LocalLowestDataPoint1 = task_myXCLASS.ConvertFreq(LowestDataPoint, GlobalvLSR, z = LocalRedshift, backTrafo = True)
                LocalHighestDataPoint1 = task_myXCLASS.ConvertFreq(HighestDataPoint, GlobalvLSR, z = LocalRedshift, backTrafo = True)
                LocalLowestDataPoint = min(LocalLowestDataPoint1, LocalHighestDataPoint1)
                LocalHighestDataPoint = max(LocalLowestDataPoint1, LocalHighestDataPoint1)


                ## determine number of transition frequencies
                idx = numpy.where((TransFreqNPList >= ShiftedFreqMin) & (TransFreqNPList <= ShiftedFreqMax))
                CounterTransitions = len(TransFreqNPList[idx])
                if (CounterTransitions > 0):


                    ## define new parameter ranges
                    dbFile = dbList[0]
                    NewFreqRanges = DefineNewFreqRanges(LocalLowestDataPoint, LocalHighestDataPoint, LocalListOfMolecules, ShiftedFreqMin, \
                                                        ShiftedFreqMax, FreqStep, t_back_flag, tBack, tSlope, nH_flag, N_H, beta_dust, kappa_1300, \
                                                        DustFileName, BackgroundFileName, LocalContPhenFuncID, LocalContPhenFuncParam1, LocalContPhenFuncParam2, \
                                                        LocalContPhenFuncParam3, LocalContPhenFuncParam4, LocalContPhenFuncParam5, \
                                                        Noise, velLowLimit, velUpLimit, MaxWidth, SQLParamArray, dbFile, CounterTransitions, FreqMinOrig)
                    NumberFreqRanges = len(NewFreqRanges)
                    TotalNumberFreqRanges += NumberFreqRanges

                    # Debug:
                    # print ("\n\n>>>FreqMin = ", FreqMin)
                    # print (">>>FreqMax = ", FreqMax)
                    # print (">>>MolName = ", MolName)
                    # print (">>>ShiftedFreqMin = ", ShiftedFreqMin)
                    # print (">>>ShiftedFreqMax = ", ShiftedFreqMax)
                    # print (">>>SQLParamArray = ", SQLParamArray)
                    # print (">>>NumberFreqRanges = ", NumberFreqRanges)
                    # print (">>>NewFreqRanges = ", NewFreqRanges)


                    ## write parameters for new freq range(s) to xml file
                    for FreqRange in range(NumberFreqRanges):
                        MinExpRange = float(NewFreqRanges[FreqRange][0])
                        MaxExpRange = float(NewFreqRanges[FreqRange][1])
                        StepFrequency = NewFreqRanges[FreqRange][2]
                        t_back_flag = NewFreqRanges[FreqRange][3]
                        BackgroundTemperature = NewFreqRanges[FreqRange][4]
                        TemperatureSlope = NewFreqRanges[FreqRange][5]
                        nH_flag = NewFreqRanges[FreqRange][6]
                        HydrogenColumnDensity = NewFreqRanges[FreqRange][7]
                        DustBeta = NewFreqRanges[FreqRange][8]
                        Kappa = NewFreqRanges[FreqRange][9]
                        DustFileName = NewFreqRanges[FreqRange][10]
                        BackgroundFileName = NewFreqRanges[FreqRange][11]
                        ContPhenFuncID = NewFreqRanges[FreqRange][12]
                        ContPhenFuncParam1 = NewFreqRanges[FreqRange][13]
                        ContPhenFuncParam2 = NewFreqRanges[FreqRange][14]
                        ContPhenFuncParam3 = NewFreqRanges[FreqRange][15]
                        ContPhenFuncParam4 = NewFreqRanges[FreqRange][16]
                        ContPhenFuncParam5 = NewFreqRanges[FreqRange][17]
                        NoiseLevel = NewFreqRanges[FreqRange][18]

                        # Debug:
                        # print ("\n\nFreqRange = ", FreqRange)
                        # print ("MinExpRange = ", MinExpRange)
                        # print ("MaxExpRange = ", MaxExpRange)
                        # print ("GlobalvLSR = ", GlobalvLSR)


                        ## shift min. and max. frequency of each range back by v_LSR
                        f1 = task_myXCLASS.ConvertFreq(MinExpRange, GlobalvLSR, z = Redshift)
                        f2 = task_myXCLASS.ConvertFreq(MaxExpRange, GlobalvLSR, z = Redshift)
                        MinExpRange = min(f1, f2)
                        MaxExpRange = max(f1, f2)


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## write parameter for current range to xml file
                        LinesForObsFile.append("\n\n        <!-- define parameters for each data range -->\n")
                        LinesForObsFile.append("        <FrequencyRange>\n")
                        LinesForObsFile.append("            <MinExpRange>" + str(MinExpRange) + "</MinExpRange>\n")
                        LinesForObsFile.append("            <MaxExpRange>" + str(MaxExpRange) + "</MaxExpRange>\n")
                        LinesForObsFile.append("            <StepFrequency>" + str(StepFrequency) + "</StepFrequency>\n")


                        ## define parameters for background continuum
                        LinesForObsFile.append("\n\n            <!-- define background temperature and temperature slope -->\n")
                        LinesForObsFile.append("            <t_back_flag>" + t_back_flag + "</t_back_flag>\n")
                        LinesForObsFile.append("            <BackgroundTemperature>" + str(BackgroundTemperature) + "</BackgroundTemperature>\n")
                        LinesForObsFile.append("            <TemperatureSlope>" + str(TemperatureSlope) + "</TemperatureSlope>\n")
                        if (BackgroundFileName is not None):
                            BackgroundFileName = BackgroundFileName.strip()
                            if (BackgroundFileName != ""):
                                LinesForObsFile.append("\n\n            <!-- define path and name of background file -->\n")
                                LinesForObsFile.append("            <BackgroundFileName>" + BackgroundFileName + "</BackgroundFileName>\n")


                        ## define parameters for dust continuum
                        if (nH_flag.lower() == "true"):
                            LinesForObsFile.append("\n\n            <!-- define hydrogen column density, beta for dust, and kappa -->\n")
                            LinesForObsFile.append("            <HydrogenColumnDensity>" + str(HydrogenColumnDensity) + "</HydrogenColumnDensity>\n")
                            LinesForObsFile.append("            <DustBeta>" + str(DustBeta) + "</DustBeta>\n")
                            LinesForObsFile.append("            <Kappa>" + str(Kappa) + "</Kappa>\n")
                        if (DustFileName is not None):
                            DustFileName = DustFileName.strip()
                            if (DustFileName != ""):
                                LinesForObsFile.append("\n\n            <!-- define path and name of dust file -->\n")
                                LinesForObsFile.append("            <DustFileName>" + DustFileName + "</DustFileName>\n")


                        ## define function ID for phenomenological continuum description
                        if (ContPhenFuncID != "None"):
                            LinesForObsFile.append("\n\n            <!-- define parameters for phenomenological continuum description -->\n")
                            LinesForObsFile.append("            <ContPhenFuncID>" + ContPhenFuncID + "</ContPhenFuncID>\n")
                            if (ContPhenFuncParam1 != "None"):
                                LinesForObsFile.append("            <ContPhenFuncParam1>" + ContPhenFuncParam1 + "</ContPhenFuncParam1>\n")
                            if (ContPhenFuncParam2 != "None"):
                                LinesForObsFile.append("            <ContPhenFuncParam2>" + ContPhenFuncParam2 + "</ContPhenFuncParam2>\n")
                            if (ContPhenFuncParam3 != "None"):
                                LinesForObsFile.append("            <ContPhenFuncParam3>" + ContPhenFuncParam3 + "</ContPhenFuncParam3>\n")
                            if (ContPhenFuncParam4 != "None"):
                                LinesForObsFile.append("            <ContPhenFuncParam4>" + ContPhenFuncParam4 + "</ContPhenFuncParam4>\n")
                            if (ContPhenFuncParam5 != "None"):
                                LinesForObsFile.append("            <ContPhenFuncParam5>" + ContPhenFuncParam5 + "</ContPhenFuncParam5>\n")


                        ## define noise level
                        LinesForObsFile.append("\n\n            <!-- define noise level for current frequency range -->\n")
                        LinesForObsFile.append("            <NoiseLevel>" + str(NoiseLevel) + "</NoiseLevel>\n")


                        ## define smooth values
                        if (SmoothValue is not None):
                            LinesForObsFile.append("\n\n            <!-- define smooth value for current frequency range -->\n")
                            LinesForObsFile.append("            <SmoothValue>" + str(SmoothValue) + "</SmoothValue>\n")
                        LinesForObsFile.append("        </FrequencyRange>\n")


        ## add number of frequency ranges
        LinesForObsFile.append("\n\n        <!-- define number of data ranges -->\n")
        LinesForObsFile.append("        <NumberExpRanges>" + str(TotalNumberFreqRanges) + "</NumberExpRanges>\n")


        ## write rest settings of frequency range definition
        if (TelescopeSize is not None):
            LinesForObsFile.append("\n\n        <!-- define size of telescope -->\n")
            LinesForObsFile.append("        <TelescopeSize>" + str(TelescopeSize) + "</TelescopeSize>\n")
        if (LocalBMIN is not None):
            LinesForObsFile.append("\n\n        <!-- define BMIN parameters -->\n")
            LinesForObsFile.append("        <BMIN>" + str(LocalBMIN) + "</BMIN>\n")
        if (LocalBMAJ is not None):
            LinesForObsFile.append("\n\n        <!-- define BMAJ parameters -->\n")
            LinesForObsFile.append("        <BMAJ>" + str(LocalBMAJ) + "</BMAJ>\n")
        if (LocalBPA is not None):
            LinesForObsFile.append("\n\n        <!-- define BPA parameters -->\n")
            LinesForObsFile.append("        <BPA>" + str(LocalBPA) + "</BPA>\n")
        LinesForObsFile.append("\n\n        <!-- define if interferrometric observation is modeled -->\n")
        LinesForObsFile.append("        <Inter_Flag>" + str(InterFlag) + "</Inter_Flag>\n")
        LinesForObsFile.append("\n\n        <!-- define local standard of rest (vLSR) -->\n")
        LinesForObsFile.append("        <GlobalvLSR>" + str(GlobalvLSR) + "</GlobalvLSR>\n")
        if (LocalRedshift is not None):
            LinesForObsFile.append("\n\n        <!-- define redshift parameter -->\n")
            LinesForObsFile.append("        <Redshift>" + str(LocalRedshift) + "</Redshift>\n")


        ## write parameters for import of obs. data file
        LinesForObsFile.append("\n\n        <!-- define parameters for ASCII file import -->\n")
        LinesForObsFile.append("        <ErrorY>" + str(ErrorY) + "</ErrorY>\n")
        LinesForObsFile.append("        <NumberHeaderLines>" + str(NumberHeaderLines) + "</NumberHeaderLines>\n")
        LinesForObsFile.append("        <SeparatorColumns>" + SeparatorColumns + "</SeparatorColumns>\n")
        LinesForObsFile.append("    </file>\n")


        ## write parameters for current obs. data file to global output array, if at least one frequency array is defined
        if (TotalNumberFreqRanges > 0):
            EffectiveObsDataFileCounter += 1
            for line in LinesForObsFile:
                LinesForXMLFile.append(line)


    ## write effective total number of obs. data file and parameters for obs. data file(s) to xml file
    NewExpXMLFile.write("    <NumberExpFiles>" + str(EffectiveObsDataFileCounter)  + "</NumberExpFiles>\n")
    for line in LinesForXMLFile:
        NewExpXMLFile.write(line)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write rest of xml file


    ## define iso flag and ratio file
    if (IsoTableFileNameList != []):
        star = "*" * (len(IsoTableFileNameList[0].strip()) + 41 - 13)
    else:
        star = "*" * 7
    NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
    NewExpXMLFile.write("    <!-- parameters for isotopologues -->\n")
    iso_flag = str(IsotopologuesList[0])
    iso_flag = task_myXCLASSMapFit.CheckBool(iso_flag)
    if (iso_flag):
        NewExpXMLFile.write("    <iso_flag>True</iso_flag>\n")
        NewExpXMLFile.write("    <IsoTableFileName>" + IsoTableFileNameList[0].strip() + "</IsoTableFileName>\n")
    else:
        NewExpXMLFile.write("    <iso_flag>False</iso_flag>\n")
        NewExpXMLFile.write("    <IsoTableFileName></IsoTableFileName>\n")


    ## define path and name of database file
    if (dbList != []):
        dbFilename = dbList[0].strip()
        if (dbFilename != ""):
            star = "*" * (len(dbFilename) + 29 - 13)
            NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
            NewExpXMLFile.write("    <!-- define path and name of database file -->\n")
            NewExpXMLFile.write("    <dbFilename>" + dbFilename + "</dbFilename>\n")


    ## define number of model pixels
    if (NumModelPixelXXList is not None and NumModelPixelYYList is not None):
        if (NumModelPixelXXList != [] and NumModelPixelYYList != []):
            if (NumModelPixelXXList[0] is not None and NumModelPixelYYList[0]):
                star = "*" * 54
                NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
                NewExpXMLFile.write("    <!-- define number of model pixels along x- and y-direction -->\n")
                NewExpXMLFile.write("    <NumModelPixelXX>" + str(NumModelPixelXXList[0]) + "</NumModelPixelXX>\n")
                NewExpXMLFile.write("    <NumModelPixelYY>" + str(NumModelPixelYYList[0]) + "</NumModelPixelYY>\n")


    ## add local-overlap flag
    if (LocalOverlapFlagList is not None):
        if (LocalOverlapFlagList != []):
            if (LocalOverlapFlagList[0] is not None):
                star = "*" * 54
                NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
                NewExpXMLFile.write("    <!-- take local-overlap into account -->\n")
                NewExpXMLFile.write("    <LocalOverlap_Flag>" + str(LocalOverlapFlagList[0]) + "</LocalOverlap_Flag>\n")


    ## add no-sub-beam flag
    if (NoSubBeamFlagList is not None):
        if (NoSubBeamFlagList != []):
            if (NoSubBeamFlagList[0] is not None):
                star = "*" * 54
                NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
                NewExpXMLFile.write("    <!-- do not use sub-beam description -->\n")
                NewExpXMLFile.write("    <NoSubBeam_Flag>" + str(NoSubBeamFlagList[0]) + "</NoSubBeam_Flag>\n")


    ## add no-sub-beam flag
    if (EmAbsPATHList is not None):
        if (EmAbsPATHList != []):
            if (EmAbsPATHList[0] is not None):
                star = "*" * 65
                NewExpXMLFile.write("\n\n    <!-- " + star + " -->\n")
                NewExpXMLFile.write("    <!-- define path of files describing emission and absorption functions -->\n")
                NewExpXMLFile.write("    <EmAbsPATH>" + str(EmAbsPATHList[0]) + "</EmAbsPATH>\n")


    ## write footer of obs. xml file
    NewExpXMLFile.write("</ExpFiles>\n")
    NewExpXMLFile.close()


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## shrink freq ranges defined in the exp. xml file to reduce computational effort
##
def ShrinkFreqRanges(molecule, MolDirName, ExpXML, MolfitsFileName, SQLParamArray):
    """

input parameters:
-----------------

    - molecule:                 name of the current molecule

    - MolDirName:               the corresponding file name

    - ExpXML:                   path and name of exp. xml file

    - MolfitsFileName:          path and name of molfits file

    - SQLParamArray:            array containing sql parameters


output parameters:
------------------

    - None

    """

    # Debug:
    # print ("\n\n\n\nmolecule = ", molecule)
    # print ("MolDirName = ", MolDirName)
    # print ("ExpXML = ", ExpXML)
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("SQLParamArray = ", SQLParamArray)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in some parameters from observational xml file
    ExpFileList = task_MAGIX.GetXMLtagNEW(ExpXML, "FileNamesExpFiles")
    NumberExpRangesList = task_MAGIX.GetXMLtagNEW(ExpXML, "NumberExpRanges")
    FreqMinList = task_MAGIX.GetXMLtagNEW(ExpXML, "MinExpRange")
    FreqMaxList = task_MAGIX.GetXMLtagNEW(ExpXML, "MaxExpRange")
    FreqStepList = task_MAGIX.GetXMLtagNEW(ExpXML, "StepFrequency")
    t_back_flagList = task_MAGIX.GetXMLtagNEW(ExpXML, "t_back_flag")
    tBackList = task_MAGIX.GetXMLtagNEW(ExpXML, "BackgroundTemperature")
    tSlopeList = task_MAGIX.GetXMLtagNEW(ExpXML, "TemperatureSlope")
    N_HList = task_MAGIX.GetXMLtagNEW(ExpXML, "HydrogenColumnDensity")
    beta_dustList = task_MAGIX.GetXMLtagNEW(ExpXML, "DustBeta")
    kappa_1300List = task_MAGIX.GetXMLtagNEW(ExpXML, "Kappa")
    DustFileNameList = task_MAGIX.GetXMLtagNEW(ExpXML, "DustFileName")
    BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(ExpXML, "BackgroundFileName")
    ContPhenFuncIDList = task_MAGIX.GetXMLtagNEW(ExpXML, "ContPhenFuncID")
    ContPhenFuncParam1List = task_MAGIX.GetXMLtagNEW(ExpXML, "ContPhenFuncParam1")
    ContPhenFuncParam2List = task_MAGIX.GetXMLtagNEW(ExpXML, "ContPhenFuncParam2")
    ContPhenFuncParam3List = task_MAGIX.GetXMLtagNEW(ExpXML, "ContPhenFuncParam3")
    ContPhenFuncParam4List = task_MAGIX.GetXMLtagNEW(ExpXML, "ContPhenFuncParam4")
    ContPhenFuncParam5List = task_MAGIX.GetXMLtagNEW(ExpXML, "ContPhenFuncParam5")
    TelescopeSizeList = task_MAGIX.GetXMLtagNEW(ExpXML, "TelescopeSize")
    BMINList = task_MAGIX.GetXMLtagNEW(ExpXML, "BMIN")
    BMAJList = task_MAGIX.GetXMLtagNEW(ExpXML, "BMAJ")
    BPAList = task_MAGIX.GetXMLtagNEW(ExpXML, "BPA")
    InterFlagList = task_MAGIX.GetXMLtagNEW(ExpXML, "Inter_Flag")
    GlobalvLSRList = task_MAGIX.GetXMLtagNEW(ExpXML, "GlobalvLSR")
    RedshiftList = task_MAGIX.GetXMLtagNEW(ExpXML, "Redshift")
    NoiseList = task_MAGIX.GetXMLtagNEW(ExpXML, "NoiseLevel")
    SmoothList = task_MAGIX.GetXMLtagNEW(ExpXML, "SmoothValue")
    ErrorYList = task_MAGIX.GetXMLtagNEW(ExpXML, "ErrorY")
    NumberHeaderLinesList = task_MAGIX.GetXMLtagNEW(ExpXML, "NumberHeaderLines")
    SeparatorColumnsList = task_MAGIX.GetXMLtagNEW(ExpXML, "SeparatorColumns")
    IsotopologuesList = task_MAGIX.GetXMLtagNEW(ExpXML, "Isotopologues")
    if (IsotopologuesList == []):
        IsotopologuesList = task_MAGIX.GetXMLtagNEW(ExpXML, "iso_flag")
    IsoTableFileNameList = task_MAGIX.GetXMLtagNEW(ExpXML, "IsoTableFileName")
    dbList = task_MAGIX.GetXMLtagNEW(ExpXML, "dbFilename")
    NumModelPixelXXList = task_MAGIX.GetXMLtagNEW(ExpXML, "NumModelPixelXX")
    NumModelPixelYYList = task_MAGIX.GetXMLtagNEW(ExpXML, "NumModelPixelYY")
    LocalOverlapFlagList = task_MAGIX.GetXMLtagNEW(ExpXML, "LocalOverlap_Flag")
    NoSubBeamFlagList = task_MAGIX.GetXMLtagNEW(ExpXML, "NoSubBeam_Flag")
    EmAbsPATHList = task_MAGIX.GetXMLtagNEW(ExpXML, "EmAbsPATH")

    # Debug:
    # print ("ExpFileList = ", ExpFileList)
    # print ("NumberExpRangesList = ", NumberExpRangesList)
    # print ("FreqMinList = ", FreqMinList)
    # print ("FreqMaxList = ", FreqMaxList)
    # print ("FreqStepList = ", FreqStepList)
    # print ("t_back_flagList = ", t_back_flagList)
    # print ("tBackList = ", tBackList)
    # print ("tSlopeList = ", tSlopeList)
    # print ("N_HList = ", N_HList)
    # print ("beta_dustList = ", beta_dustList)
    # print ("kappa_1300List = ", kappa_1300List)
    # print ("DustFileNameList = ", DustFileNameList)
    # print ("BackgroundFileNameList = ", BackgroundFileNameList)
    # print ("ContPhenFuncIDList = ", ContPhenFuncIDList)
    # print ("ContPhenFuncParam1List = ", ContPhenFuncParam1List)
    # print ("ContPhenFuncParam2List = ", ContPhenFuncParam2List)
    # print ("ContPhenFuncParam3List = ", ContPhenFuncParam3List)
    # print ("ContPhenFuncParam4List = ", ContPhenFuncParam4List)
    # print ("ContPhenFuncParam5List = ", ContPhenFuncParam5List)
    # print ("NoiseList = ", NoiseList)
    # print ("SmoothList = ", SmoothList)
    # print ("TelescopeSizeList = ", TelescopeSizeList)
    # print ("BMINList = ", BMINList)
    # print ("BMAJList = ", BMAJList)
    # print ("BPAList = ", BPAList)
    # print ("InterFlagList = ", InterFlagList)
    # print ("GlobalvLSRList = ", GlobalvLSRList)
    # print ("RedshiftList = ", RedshiftList)
    # print ("ErrorYList = ", ErrorYList)
    # print ("NumberHeaderLinesList = ", NumberHeaderLinesList)
    # print ("SeparatorColumnsList = ", SeparatorColumnsList)
    # print ("IsotopologuesList = ", IsotopologuesList)
    # print ("IsoTableFileNameList = ", IsoTableFileNameList)
    # print ("dbList = ", dbList)
    # print ("NumModelPixelXXList = ", NumModelPixelXXList)
    # print ("NumModelPixelYYList = ", NumModelPixelYYList)
    # print ("LocalOverlapFlagList = ", LocalOverlapFlagList)
    # print ("NoSubBeamFlagList = ", NoSubBeamFlagList)
    # print ("EmAbsPATHList = ", EmAbsPATHList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get parameters from molfit file
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(MolfitsFileName)
    MolName = MoleculesInMolfitFile[0]
    LocalMolfitParameters = AllParameters[0]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get velocity offset and width
    MaxWidth = -1.e99
    velLowLimit = 1.e99
    velUpLimit = -1.e99
    for line in LocalMolfitParameters:

        # Debug:
        # print ("line = ", line)

        for col in line:
            Name = col[0]
            flag = col[1]
            lowlimit = col[2]
            uplimit = col[3]
            element = col[4]

            # Debug:
            # print ("col = ", col)


            ## determine minimal lower and max. upper range for velocity
            if (Name == "V_off"):
                lowlimit = float(lowlimit)
                uplimit = float(uplimit)
                if (lowlimit < velLowLimit):
                    velLowLimit = lowlimit
                if (uplimit > velUpLimit):
                    velUpLimit = uplimit


            ## determine minimal an max. velocity offset
            elif (Name == "V_width_Gauss"):
                element = float(element)
                if (MaxWidth < element):
                    MaxWidth = element

    # Debug:
    # print ("\n\n\nmolecule = ", molecule)
    # print ("velLowLimit = ", velLowLimit)
    # print ("velUpLimit = ", velUpLimit)
    # print ("MaxWidth = ", MaxWidth)
    # print ("FreqMinList, FreqMaxList = ", FreqMinList, FreqMaxList)
    # print ("GlobalvLSRList = ", GlobalvLSRList)
    # print ("SQLParamArray = ", SQLParamArray)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create new xml file for current molecule with shrinked frequency ranges
    LocalMaxNumTransInFit = 1.e99
    CreateXMLFile(molecule, velLowLimit, velUpLimit, MaxWidth, ExpXML, ExpFileList, NumberExpRangesList, FreqMinList, FreqMaxList, FreqStepList, \
                  t_back_flagList, tBackList, tSlopeList, N_HList, beta_dustList, kappa_1300List, DustFileNameList, BackgroundFileNameList, \
                  ContPhenFuncIDList, ContPhenFuncParam1List, ContPhenFuncParam2List, \
                  ContPhenFuncParam3List, ContPhenFuncParam4List, ContPhenFuncParam5List, NoiseList, SmoothList, TelescopeSizeList, BMINList, BMAJList, \
                  BPAList, InterFlagList, GlobalvLSRList, RedshiftList, ErrorYList, NumberHeaderLinesList, SeparatorColumnsList, IsotopologuesList, \
                  IsoTableFileNameList, dbList, NumModelPixelXXList, NumModelPixelYYList, LocalOverlapFlagList, SQLParamArray, \
                  LocalMaxNumTransInFit, NoSubBeamFlagList, EmAbsPATHList)

    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## adjust paths in an MAGIX xml file
##
def AdjustExpXMLFile(MAGIXExpXML, JobDir, CurrentDir, dbDefaultFilename):
    """

input parameters:
-----------------

    - MAGIXExpXML:              path and name of MAGIX xml file

    - JobDir:                   path of the current job directory

    - CurrentDir:               current root directory

    - dbDefaultFilename:        path and name of default database file


output parameters:
------------------

    - None
    """

    # Debug:
    # print ('MAGIXExpXML = ', MAGIXExpXML)
    # print ('JobDir = ', JobDir)
    # print ('CurrentDir = ', CurrentDir)
    # print ('dbDefaultFilename = ', dbDefaultFilename)


    ##====================================================================================================================================================
    ## analyze experimentalData parameter: check if input parameter defines path and name of a file
    print ("Adjust MAGIX XML file ..", end=' ')


    ## copy xml file to temp directory
    task_myXCLASS.NormNCopyMove(MAGIXExpXML, JobDir + "exp.xml", copyFlag = True)
    MAGIXExpXML = JobDir + "exp.xml"

    # Debug:
    # print ('MAGIXExpXML = ', MAGIXExpXML)


    ## copy the files defined in the xml-file to the current MAGIX working directory
    ExpFiles = []
    ExpFilesOrig = task_MAGIX.GetXMLtagNEW(MAGIXExpXML, "FileNamesExpFiles")
    for filename in ExpFilesOrig:
        if (filename[1][0] != "/"):                                                         ## make relative path absolute
            filename = CurrentDir + filename[1]
        else:
            filename = filename[1]
        if not (os.path.exists(filename) or os.path.isfile(filename)):
            print ("\n\n\nError in XCLASS package, function LineID, subroutine AdjustExpXMLFile:")
            print ("\tThe given path and name of the experimental data file " + filename)
            print ("\tdefined in the experimental data xml-file does not exist!")
            print ("\n\tPlease note, if a relative path is specified, the path has to be defined relative to the")
            print ("\tcurrent working directory!")
            print ("\n\tPlease enter an existing path and name and redo LineIdentification function call!")
            return
        ExpFiles.append(filename)                                                           ## save name(s) in list

    # Debug:
    # print ('ExpFiles = ', ExpFiles)


    ## get flag for iso file
    IsoFlag = task_MAGIX.GetXMLtagNEW(MAGIXExpXML, "Isotopologues")
    if (IsoFlag == []):
        IsoFlag = task_MAGIX.GetXMLtagNEW(MAGIXExpXML, "iso_flag")
    if (len(IsoFlag) > 0):
        IsoFlag = IsoFlag[0].lower()
        IsoFlag = task_myXCLASSMapFit.CheckBool(IsoFlag)
        if (IsoFlag):
            IsoFlag = True
        else:
            IsoFlag = False
    else:
        IsoFlag = False

    # Debug:
    # print ('IsoFlag = ', IsoFlag)


    ## get name of iso file
    if (IsoFlag):
        IsoFile = task_MAGIX.GetXMLtagNEW(MAGIXExpXML, "IsoTableFileName")

        # Debug:
        # print ('IsoFile = ', IsoFile)

        if (len(IsoFile) > 0):
            IsoFile = IsoFile[0]
            if (IsoFile.strip() != ""):
                if (IsoFile[0] != "/"):                                                     ## make relative path absolute
                    IsoFile = CurrentDir + IsoFile
                if not (os.path.exists(IsoFile) or os.path.isfile(IsoFile)):
                    print ("\n\n\nError in XCLASS package, function LineID, subroutine AdjustExpXMLFile:")
                    print ("\tThe given path and name of the iso file " + IsoFile)
                    print ("\tdoes not exist!")
                    print ("\n\tPlease note, if a relative path is specified, the path has to be defined relative to the")
                    print ("\tcurrent working directory!")
                    print ("\n\tPlease enter an existing path and name of iso file and redo LineIdentification function call!")
                    return
                command_string = "cp " + IsoFile + " " + JobDir                             ## copy obs. data files to temp directory
                os.system(command_string)
                i = IsoFile.rfind("/")
                if (i < 0):
                    i = 0
                IsoFile = JobDir + IsoFile[i + 1:]                                          ## define new iso file name
                IsoFile = [IsoFile]

                # Debug:
                # print ('IsoFile = ', IsoFile)


                ## modify experimental xml-file in MAGIX temp directory
                task_MAGIX.WriteXMLtagNEW(MAGIXExpXML, "IsoTableFileName", IsoFile)


    ## copy experimental data file(s) to MAGIX temp directory
    NewExpFiles = []                                                                        ## list with modified data file names
    for filesID, files in enumerate(ExpFiles):
        files = files.strip()
        if (files[0] != "/"):
            files = CurrentDir + files

        command_string = "cp " + files + " " + JobDir                                       ## copy obs. data files to temp directory
        os.system(command_string)
        i = files.rfind("/")
        if (i < 0):
            i = 0
        NewExpFiles.append([filesID, JobDir + files[i + 1:]])                               ## define new data file names


    ## modify experimental xml-file in MAGIX temp directory
    task_MAGIX.WriteXMLtagNEW(MAGIXExpXML, "FileNamesExpFiles", NewExpFiles)

    # Debug:
    # print ("NewExpFiles = ", NewExpFiles)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get name of database file
    PathFileNameDB = task_MAGIX.GetXMLtagNEW(MAGIXExpXML, "dbFilename")

    # Debug:
    # print ('PathFileNameDB = ', PathFileNameDB)

    if (len(PathFileNameDB) > 0):
        PathFileNameDB = PathFileNameDB[0].strip()
        if (PathFileNameDB.strip() != ""):
            if (PathFileNameDB[0] != "/"):                                                  ## make relative path absolute
                PathFileNameDB = CurrentDir + PathFileNameDB
            if not (os.path.exists(PathFileNameDB) or os.path.isfile(PathFileNameDB)):
                print ("\n\n\nError in XCLASS package, function LineID, subroutine AdjustExpXMLFile:")
                print ("\tThe given path and name of the database file " + PathFileNameDB)
                print ("\tdoes not exist!")
                print ("\n\tPlease note, if a relative path is specified, the path has to be defined relative to the")
                print ("\tcurrent working directory!")
                print ("\n\tPlease enter an existing path and name of iso file and redo LineIdentification function call!")
                return

            # Debug:
            # print ('PathFileNameDB = ', PathFileNameDB)


            ## modify experimental xml-file in MAGIX temp directory
            PathFileNameDB = [PathFileNameDB]
        else:
            PathFileNameDB = [dbDefaultFilename]
    else:
        PathFileNameDB = [dbDefaultFilename]
    task_MAGIX.WriteXMLtagNEW(MAGIXExpXML, "dbFilename", PathFileNameDB)


    ## we're done
    print ("done!")


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get information for all molecules between FreqMin and FreqMax from sqlite3 database
##
def ReadDatabase(PathFileNameDB, ListOfAllFreqMin, ListOfAllFreqMax, ListOfAllGlobalvLSR, ListOfAllRedshift, SelectedMolecules, IsotopologuesList, \
                 IsoMoleculeList, MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL):
    """

input parameters:
-----------------

    - PathFileNameDB:           path and name of a user defined sqlite3 database file (if empty, use default file)

    - ListOfAllFreqMin:         list of minimum frequencies

    - ListOfAllFreqMax:         list of maximum frequencies

    - ListOfAllGlobalvLSR:      list of global vLSR velocities

    - ListOfAllRedshift:        list of redshifts

    - SelectedMolecules:        (optional) python list containing names of molecules

    - IsotopologuesList:        list containing the isotopologues

    - IsoMoleculeList:          list containing the corresponding molecules

    - MinNumTransitionsSQL:     min. number of transitions for SQL query

    - MaxNumTransitionsSQL:     max. number of transitions for SQL query

    - MaxElowSQL:               max. lower energy of transitions for SQL query

    - MingASQL:                 min. intensity (gA) of transitions for SQL query

    - OrderTransSQL:            order of transitions for SQL query


output parameters:
------------------

    - ok:                       status variable

    - CounterMolecules:         number of molecules in given frequency range

    - MoleculeData:             all information about molecules between FreqMin and FreqMax from sqlite3 database

    - ListTotalNumTransPerMol:  updated list containing the names and the number of transitions for each molecule (not defined, for later updates)

    - LineCatalog:              parameters of all transitions within given frequency ranges

    """

    # Debug:
    # print ("PathFileNameDB = ", PathFileNameDB)
    # print ("ListOfAllFreqMin = ", ListOfAllFreqMin)
    # print ("ListOfAllFreqMax = ", ListOfAllFreqMax)
    # print ("ListOfAllGlobalvLSR = ", ListOfAllGlobalvLSR)
    # print ("ListOfAllRedshift = ", ListOfAllRedshift)
    # print ("SelectedMolecules = ", SelectedMolecules)
    # print ("IsotopologuesList = ", IsotopologuesList)
    # print ("IsoMoleculeList = ", IsoMoleculeList)
    # print ("MinNumTransitionsSQL = ", MinNumTransitionsSQL)
    # print ("MaxNumTransitionsSQL = ", MaxNumTransitionsSQL)
    # print ("MaxElowSQL = ", MaxElowSQL)
    # print ("MingASQL = ", MingASQL)
    # print ("OrderTransSQL = ", OrderTransSQL)


    ## reset output variables
    ok = 0
    CounterMolecules = 0
    MoleculeData = []
    ListTotalNumTransPerMol = []
    LineCatalog = []
    OutputArray = [ok, CounterMolecules, MoleculeData, ListTotalNumTransPerMol, LineCatalog]


    ## define some parameters for database
    NameOfRadTransTable = "transitions"
    ColumnNameForNameTransitions = "T_Name"
    ColumnNameForFreqTransitions = "T_Frequency"
    ColumnNameForIntTransitions = "T_Intensity"
    ColumnNameForEinsteinATransitions = "T_EinsteinA"
    ColumnNameForFreqErrTransitions = "T_Uncertainty"
    ColumnNameForELowTransitions = "T_EnergyLower"
    ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
    NameOfPartFuncTable = "partitionfunctions"
    NumberOfTemperatures = 110
    ColumnNameForNamePartFunc = 'PF_Name'
    ColumnNamesPartFunc = ['PF_1_072', 'PF_1_148', 'PF_1_230', 'PF_1_318', 'PF_1_413', 'PF_1_514', 'PF_1_622', 'PF_1_738', 'PF_1_862', 'PF_1_995', \
                           'PF_2_138', 'PF_2_291', 'PF_2_455', 'PF_2_630', 'PF_2_725', 'PF_2_818', 'PF_3_020', 'PF_3_236', 'PF_3_467', 'PF_3_715', \
                           'PF_3_981', 'PF_4_266', 'PF_4_571', 'PF_4_898', 'PF_5_000', 'PF_5_248', 'PF_5_623', 'PF_6_026', 'PF_6_457', 'PF_6_918', \
                           'PF_7_413', 'PF_7_943', 'PF_8_511', 'PF_9_120', 'PF_9_375', 'PF_9_772', 'PF_10_471', 'PF_11_220', 'PF_12_023', 'PF_12_882', \
                           'PF_13_804', 'PF_14_791', 'PF_15_849', 'PF_16_982', 'PF_18_197', 'PF_18_750', 'PF_19_498', 'PF_20_893', 'PF_22_387', \
                           'PF_23_988', 'PF_25_704', 'PF_27_542', 'PF_29_512', 'PF_31_623', 'PF_33_884', 'PF_36_308', 'PF_37_500', 'PF_38_905', \
                           'PF_41_687', 'PF_44_668', 'PF_47_863', 'PF_51_286', 'PF_54_954', 'PF_58_884', 'PF_63_096', 'PF_67_608', 'PF_72_444', \
                           'PF_75_000', 'PF_77_625', 'PF_83_176', 'PF_89_125', 'PF_95_499', 'PF_102_329', 'PF_109_648', 'PF_117_490', 'PF_125_893', \
                           'PF_134_896', 'PF_144_544', 'PF_150_000', 'PF_154_882', 'PF_165_959', 'PF_177_828', 'PF_190_546', 'PF_204_174', 'PF_218_776', \
                           'PF_225_000', 'PF_234_423', 'PF_251_189', 'PF_269_153', 'PF_288_403', 'PF_300_000', 'PF_309_030', 'PF_331_131', 'PF_354_813', \
                           'PF_380_189', 'PF_407_380', 'PF_436_516', 'PF_467_735', 'PF_500_000', 'PF_501_187', 'PF_537_032', 'PF_575_440', 'PF_616_595', \
                           'PF_660_693', 'PF_707_946', 'PF_758_578', 'PF_812_831', 'PF_870_964', 'PF_933_254', 'PF_1000_000']


    ## print what you do
    print ("\n\nReading data from sqlite3 database .. ", end = "", flush = True)


    ## connect to the sqlite3 database
    try:
        conn = sqlite3.connect(PathFileNameDB)
    except sqlite3.Error as e:
        print ("\nError in function LineIdentification.ReadDatabase:")
        print ("\t\tCan not connect to sqlite3 database {:s}.".format(PathFileNameDB))
        print ("\t\tError: {:d}: {:s}".format(e.args[0], e.args[1]))
        ok = 1
        return OutputArray


    ## get cursor
    cursor = conn.cursor()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get all molecule names within frequency range


    ## define order of transitions
    if (OrderTransSQL == 1):                                                    ## order by lower energy
        OrderString = "{:s}".format(ColumnNameForELowTransitions)
    elif (OrderTransSQL == 2):                                                  ## order by gA
        OrderString = "{:s} * {:s}".format(ColumnNameForgUpTransitions, ColumnNameForEinsteinATransitions)
    else:                                                                       ## order by frequency
        OrderString = "{:s}".format(ColumnNameForFreqTransitions)


    ## construct query string for transition table
    # query_string = "SELECT " + ColumnNameForNameTransitions + " FROM " + NameOfRadTransTable + " WHERE ("
    query_string = "SELECT"
    query_string += " {:s},".format(ColumnNameForNameTransitions)               ## position 1: molecule name
    query_string += " {:s},".format(ColumnNameForFreqTransitions)               ## position 2: transition frequency
    query_string += " {:s},".format(ColumnNameForIntTransitions)                ## position 3: intensity
    query_string += " {:s},".format(ColumnNameForEinsteinATransitions)          ## position 4: EinsteinA
    query_string += " {:s},".format(ColumnNameForFreqErrTransitions)            ## position 5: unvertainty
    query_string += " {:s},".format(ColumnNameForELowTransitions)               ## position 6: E_low
    query_string += " {:s}".format(ColumnNameForgUpTransitions)                 ## position 7: upper state degeneracy
    query_string += " FROM {:s} WHERE (".format(NameOfRadTransTable)
    NumberRanges = len(ListOfAllFreqMin)
    for RangeIndex in range(NumberRanges):
        vLSR = ListOfAllGlobalvLSR[RangeIndex]
        FreqMin = ListOfAllFreqMin[RangeIndex]
        FreqMax = ListOfAllFreqMax[RangeIndex]
        Redshift = ListOfAllRedshift[RangeIndex]


        ## shift min. and max. frequency of each range by v_LSR
        NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, vLSR, z = Redshift, backTrafo = True)
        NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, vLSR, z = Redshift, backTrafo = True)
        FreqMin = min(NewFreqMin, NewFreqMax)
        FreqMax = max(NewFreqMin, NewFreqMax)


        ## define sql query
        query_string += "(" + ColumnNameForFreqTransitions + " >= " + str(FreqMin)
        query_string += " AND " + ColumnNameForFreqTransitions + " <= " + str(FreqMax)
        query_string += ")"
        if (RangeIndex < (NumberRanges - 1)):
            query_string += " OR "
    if (MaxElowSQL > 0.0):
        query_string += " AND " + ColumnNameForELowTransitions + " <= " + str(MaxElowSQL / 1.42879)
    if (MingASQL > 0.0):
        query_string += " and (" + ColumnNameForgUpTransitions + " * "+ ColumnNameForEinsteinATransitions + ") >= " + str(MingASQL)
    query_string += ") ORDER by " + OrderString

    # Debug:
    # print ("\n\nquery_string = ", query_string)


    ## read data for all molecules from table RadTrans
    cursor.execute(query_string)
    rows = cursor.fetchall()

    # Debug:
    # print ("\n\n\n\nrows = ", rows)
    # print ("rows[0] = ", rows[0])
    # print ("rows[1] = ", rows[1])


    ## check, if list is empty, i.e. no molecule is within frequency range
    if (len(rows) == []):
        conn.close()
        ok = 1
        print ("\n\n\nError in XCLASS package, function LineID, subroutine ReadDatabase:")
        print ("\tThere is no molecule data in the database within the given frequency range!")
        print ("\n\tPlease enlarge the frequency range and redo LineIdentification function call!")
        return OutputArray


    ## remove duplicates in list with names of molecules
    ListMoleculeNames = []
    LineCatalog = []
    for MolName in rows:
        name = MolName[0]


        ## extract transition parameters
        LocalIncludeFlag = True
        if (SelectedMolecules != [] and SelectedMolecules != [None] and SelectedMolecules is not None):
            if (not name in SelectedMolecules):
                LocalIncludeFlag = False
        if (LocalIncludeFlag):
            TransFreq = float(MolName[1])
            EinsteinA = float(MolName[3])
            ELow = float(MolName[5]) * 1.42879
            gup = float(MolName[6])
            LocalWeight = ((gup * EinsteinA) / (max(1.0, ELow)**2))
            LineCatalog.append([TransFreq, EinsteinA, ELow, gup, LocalWeight, name])

        # Debug:
        # print ("LineCatalog[-1] = ", LineCatalog[-1])


        ## store molecule names
        if not name in ListMoleculeNames:
            if (name[-2:] == "#1" and (not name in SelectedMolecules)):
                namered = name[:-2]
                if not name in ListMoleculeNames:
                    if not namered in ListMoleculeNames:
                        ListMoleculeNames.append(name)
            elif (name[-2:] == "#2" and (not name in SelectedMolecules)):
                namered = name[:-2]
                if not name in ListMoleculeNames:
                    if not namered in ListMoleculeNames:
                        ListMoleculeNames.append(name)
            else:
                ListMoleculeNames.append(name)

                # Debug:
                # print ("\n\nname = ", name)
                # print ("MolName[1:] = ", MolName[1:])


    ## we're done
    print ("done!", flush = True)


    ## sort and convert LineCatalog list to numpy array
    LineCatalogNP = sorted(LineCatalog, key=lambda x:x[0])
    LineCatalogNP = numpy.asarray(LineCatalogNP)
    TransFreqList = LineCatalogNP[:, 0].astype(float)
    LineCatalog = [TransFreqList, LineCatalogNP]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get data for each molecule separately
    print ("Analyze molecular data .. ", end = "", flush = True)

    # Debug:
    # print ("SelectedMolecules = ", SelectedMolecules)


    ## exclude / consider only molecules
    MoleculesInFreqeuncyRange = []
    FinalListMoleculeNames = []
    ExcludedMolecules = []
    for MolName in ListMoleculeNames:                                                       ## loop over all molecules within the frequency range

        # Debug:
        # print ("MolName = ", MolName)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check molecules and read in data
        checkPartitionFuncTableFlag = True
        NoNoneEntry = "true"
        ConsiderMoleculeFlag = "true"
        PrintWarningFlag = "minimal"


        ## check, if some molecules should be excluded or included
        if (SelectedMolecules != [] and SelectedMolecules != [None] and SelectedMolecules is not None):
            OnlyExcludeMoleculesFlag = "true"
            for mol in SelectedMolecules:
                StrippedMoleculeName = mol.strip()


                ## should molecule NOT be considered
                if (StrippedMoleculeName[:2] == "--"):
                    StrippedMoleculeName = StrippedMoleculeName[2:]
                    if (StrippedMoleculeName == MolName):
                        ConsiderMoleculeFlag = "false"
                        break
                else:
                    OnlyExcludeMoleculesFlag = "false"
                    break


            ## consider only those molecules defined in list SelectedMolecules
            if (ConsiderMoleculeFlag == "true" and OnlyExcludeMoleculesFlag == "false"):
                if not MolName in SelectedMolecules:
                    ConsiderMoleculeFlag = "false"

        # Debug:
        # print ("\n\nMolName = ", MolName)
        # print ("type(MolName).__name__ = ", type(MolName).__name__)
        # print ("checkPartitionFuncTableFlag = ", checkPartitionFuncTableFlag)
        # print ("ConsiderMoleculeFlag = ", ConsiderMoleculeFlag)


        ## check, if table partitionfunctions contain data for all molecules as well
        if (checkPartitionFuncTableFlag and type(MolName).__name__ != 'float' and ConsiderMoleculeFlag == "true"):


            ## construct query string for transition table
            query_string = "SELECT"
            query_string += " {:s}".format(ColumnNameForNamePartFunc)                   ## position 1: molecule name
            for local_temp in ColumnNamesPartFunc:
                query_string += ", {:s}".format(local_temp)                             ## position 2 - 111: temperature
            query_string += " FROM {:s} WHERE (".format(NameOfPartFuncTable)
            query_string += " {:s} = {:s}{:s}{:s})".format(ColumnNameForNamePartFunc, chr(34), MolName, chr(34))

            # Debug:
            # print ("\n\nquery_string = ", query_string)


            ## read data for all molecules from table RadTrans
            cursor.execute(query_string)
            rows = cursor.fetchall()
            DoNotIncludeFlag = "false"
            try:
                i = rows[0]                                                                 ## check, if rows variable is not empty
            except IndexError:
                NoNoneEntry = "false"
                DoNotIncludeFlag = "true"

            # Debug:
            # print ("rows = ", rows)
            # print ("NoNoneEntry = ", NoNoneEntry)
            # print ("DoNotIncludeFlag = ", DoNotIncludeFlag)


            ## do further checks
            if (DoNotIncludeFlag != "true"):
                if (rows == []):
                    NoNoneEntry = "false"
                    DoNotIncludeFlag = "true"

                elif (len(rows[0]) < 110):
                    NoNoneEntry = "false"
                    DoNotIncludeFlag = "true"

                else:
                    HelpList = rows[0][1:]


                ## continue here, if rows is not empty and has (more than) 110 columns
                if (DoNotIncludeFlag != "true"):

                    # Debug:
                    # print ("\n\nrows = ", rows)
                    # print ("rows[0] = ", rows[0])
                    # print ("rows[0][1:] = ", rows[0][1:])


                    ## check if partition function is given for the current molecule MolName
                    if (HelpList.count(None) > 0):
                        NoNoneEntry = "false"
                        if (PrintWarningFlag == "true"):
                            print ("\n\nWARNING in XCLASS package, subroutine LineIdentification:")
                            print ("\tThere is no partition function given for molecule " + MolName)
                            print ("\tIgnore molecule " + MolName + " in the current line identification.")
                            print ("\n\tPartition function values = ", HelpList)
                        DoNotIncludeFlag = "true"


        ## continue here, if partition function for the current molecule is complete
        if (NoNoneEntry == "true" and type(MolName).__name__ != 'float' and ConsiderMoleculeFlag == "true"):
            CounterMolecules += 1
            MoleculesInFreqeuncyRange.append(MolName)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get all molecule names within frequency range


            ## create entry for MoleculeData
            FinalListMoleculeNames.append(MolName)
            helpList = [MolName]
            MoleculeData.append(helpList)


        ## if not included, add current molecule to list of excluded molecules
        if (NoNoneEntry != "true" or type(MolName).__name__ == 'float' or ConsiderMoleculeFlag != "true"):
            ExcludedMolecules.append(MolName)


    ## we're done
    conn.close()
    # print ("done!")

    # Debug:
    # print ("MoleculeData[0] = ", MoleculeData[0])
    # print ("MoleculesInFreqeuncyRange = ", MoleculesInFreqeuncyRange)
    # print ("\n\n\n")
    # print ("IsotopologuesList = ", IsotopologuesList)
    # print ("IsoMoleculeList = ", IsoMoleculeList)
    #    for mol in FinalListMoleculeNames:
    #        print "===>>>" + mol + "<<<==="


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## is an iso ratio file defined? Extend SelectedMolecules list
    if (IsotopologuesList != []):                                                           ## is an iso ratio file defined?
        for LocalIsotopologue in IsotopologuesList:                                         ## loop over all isotopologues in the iso ratio file
            Isotopologue = LocalIsotopologue.strip()
            if (Isotopologue in FinalListMoleculeNames):                                    ## is an isotopologue located in the given spectra?
                i = IsotopologuesList.index(Isotopologue)                                   ## get index of current isotopologue
                                                                                            ## (we consider only the first occurrence of an isotopologue
                                                                                            ##  in the iso ratio file)
                IsoMolecule = IsoMoleculeList[i]                                            ## to get the corresponding iso molecule
                if (IsoMolecule in FinalListMoleculeNames):                                 ## check, if iso molecule is already considered
                    i = FinalListMoleculeNames.index(Isotopologue)                          ## get position of isotopologue in molecule list
                                                                                            ## delete isotopologue from list of mol. in the given spectra
                    FinalListMoleculeNames = FinalListMoleculeNames[:i] + FinalListMoleculeNames[i + 1:]
                    MoleculeData = MoleculeData[:i] + MoleculeData[i + 1:]
                    # ListTotalNumTransPerMol = ListTotalNumTransPerMol[:i] + ListTotalNumTransPerMol[i + 1:]
                    ExcludedMolecules.append(Isotopologue)


    ## define new numbers of molecules
    CounterMolecules = len(FinalListMoleculeNames)

    # Debug:
    # print ("CounterMolecules = ", CounterMolecules)
    # print ("FinalListMoleculeNames = ", FinalListMoleculeNames)
    # print ("\n\n\n")
    #    for mol in FinalListMoleculeNames:
    #        print "+++>>>", mol, "<<<+++"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print list of identified molecules
    print ("\nNumber of molecules in the freq. range: ", CounterMolecules)
    if (ExcludedMolecules != []):
        for mol in ExcludedMolecules:
            print ("excluded molecule = ", mol)


    ## define return values
    OutputArray = [ok, CounterMolecules, MoleculeData, ListTotalNumTransPerMol, LineCatalog]
    return OutputArray
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## smooth experimental data for current frequency window
def SmoothObsData(x, window_len, window = 'hanning'):
    """
    taken from "http://wiki.scipy.org/Cookbook/SignalSmooth" and "http://stackoverflow.com/questions/5515720/python-smooth-time-series-data"


    smooth the data using a window with requested size.

    This method is based on the convolution of a scaled window with the signal.
    The signal is prepared by introducing reflected copies of the signal
    (with the window size) in both ends so that transient parts are minimized
    in the begining and end part of the output signal.


input parameters:
-----------------

    - x:                        the input signal

    - window_len:               the dimension of the smoothing window; should be an odd integer

    - window:                   the type of window from 'savgol', 'flat', 'hanning', 'hamming', 'bartlett', 'blackman' flat window will produce a moving
                                average smoothing.

output parameters:
------------------

    - y:                        the smoothed signal


    example:

    t = linspace(-2,2,0.1)
    x = sin(t) + randn(len(t)) * 0.1
    y = smooth(x)

    see also:

    numpy.hanning, numpy.hamming, numpy.bartlett, numpy.blackman, numpy.convolve
    scipy.signal.lfilter

    TODO: the window parameter could be the window itself if an array instead of a string
    NOTE: length(output) != length(input), to correct this: return y[(window_len/2-1):-(window_len/2)] instead of just y.
    """

    # Debug:
    # print ("x = ", x)
    # print ("window_len = ", window_len)
    # print ("window = ", window)


    ## check input parameter
    if (x.ndim != 1):
        print ("smooth only accepts 1 dimension arrays.")
    if (x.size < window_len):
        window_len = min(window_len, len(x))
    if (window_len < 3):
        return x

    # Debug:
    # print ("window = ", window)


    ## apply a Savitzky-Golay filter to an array.
    window_len = int(window_len)
    if  (window == "savgol"):
        if (window_len % 2 == 0):
            window_len -= 1
        y = signal.savgol_filter(x, window_len, 1, mode="constant")

        # Debug:
        # print ("window_len = ", window_len)


    elif (window in ["flat", "hanning", "hamming", "bartlett", "blackman"]):


        ## Translates slice objects to concatenation along the first axis.
        s = numpy.r_[2 * x[0] - x[window_len-1::-1], x, 2 * x[-1] - x[-1:-window_len:-1]]
        if (window == 'flat'):                                                                  ## moving average
            w = numpy.ones(window_len, 'd')
        else:
            w = eval('numpy.' + window + '(window_len)')


        ## convolve
        y = numpy.convolve(w/w.sum(), s, mode = 'same')


    ## define return parameter
    return y[window_len:-window_len+1]
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## create plots for total KDE and summed spectra
def CreateSummedPlots(LineIDJobDir):
    """

input parameters:
-----------------

    - LineIDJobDir:             path of current job directory



output parameters:
------------------

    - None

    """

    # Debug:
    # print ("LineIDJobDir = ", LineIDJobDir)


    ## import dictionaries from pickle files
    SummaryDir = LineIDJobDir + "combined-molecules/"
    listing = os.listdir(SummaryDir)


    ## import dictionaries from pickle files
    velList = []
    ListOfVelEM = []
    ListOfVelABS = []
    SummedSummeSpectra = []
    SummedWeightedSummeSpectra = []
    SummedSpectraCounter = 0
    for LocalFile in listing:                                                               ## loop over files


        ## import emission velocities
        if (LocalFile.startswith("em-vel__")):
            ListEmissionVelocities = pickle.load(open(SummaryDir + LocalFile))
            for ParamDict in ListEmissionVelocities:
                v0 = ParamDict['v0']                                                         ## get velocity
                ListOfVelEM.append(v0)                                                      ## append velocity


        ## import absorption velocities
        elif (LocalFile.startswith("abs-vel__")):
            ListAbsorptionVelocities = pickle.load(open(SummaryDir + LocalFile))
            for ParamDict in ListAbsorptionVelocities:
                v0 = ParamDict['v0']                                                         ## get velocity
                ListOfVelABS.append(v0)                                                     ## append velocity


        ## import summed spectra
        elif (LocalFile.startswith("sum-int__")):
            SummedSpectraCounter += 1
            SummedSpectra = pickle.load(open(SummaryDir + LocalFile))
            velList = SummedSpectra[0]
            SummedIntensities = SummedSpectra[1]
            if (SummedSummeSpectra == []):
                SummedSummeSpectra = copy.deepcopy(SummedIntensities)
            else:
                SummedSummeSpectra += SummedIntensities
            SummedWeightedIntensities = SummedSpectra[2]
            if (SummedWeightedSummeSpectra == []):
                SummedWeightedSummeSpectra = copy.deepcopy(SummedWeightedIntensities)
            else:
                SummedWeightedSummeSpectra += SummedWeightedIntensities


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create plot with summed sum-spectra
    if (SummedSummeSpectra != []):
        VelBin = abs(velList[1] - velList[0])
        velLowLimit = numpy.nanmin(velList)
        velUpLimit = numpy.nanmax(velList)
        fig, layer = pylab.subplots()
        fig.set_size_inches(15.0, 10.0)
        layer.grid(True)
        layer.set_ylabel(r"Brightness Temperature [K]")
        layer.set_xlabel("velocity [km/s]")
        layer.set_title("Summed intensities of all molecules,  Number of summed spectra: " + str(SummedSpectraCounter))
        i = (1.0 / SummedSpectraCounter)
        layer.plot(velList, i * SummedSummeSpectra, '-', color = "blue", linewidth = 2.0, label = "all summed spectra")
        layer.plot(velList, i * SummedWeightedSummeSpectra, '-', color = "red", linewidth = 2.0, label = "all summed weighted spectra")
        layer.legend()
        layer.set_xlim(velLowLimit, velUpLimit)
        LocalFileName = SummaryDir + "All-summed-spectra" + "___" + str(velLowLimit) + "_-_" + str(velUpLimit) + ".png"
        fig.savefig(LocalFileName, dpi = 300)
        pylab.close(fig)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create KDE plot
    for TypeFlag in range(2):                                                               ## consider emission and absorption "peaks" separately


        ## get current list of roots
        if (TypeFlag == 0):
            TypeFlagString = "Em"
            CFFlagVal = "c"
            ListOfRoots = copy.deepcopy(ListOfVelEM)
        else:
            TypeFlagString = "Abs"
            CFFlagVal = "f"
            ListOfRoots = copy.deepcopy(ListOfVelABS)
        ListOfRoots = numpy.asarray(ListOfRoots)
        ListOfRoots = numpy.nan_to_num(ListOfRoots)
        if (len(ListOfRoots) > 0):
            MinRoot = min(ListOfRoots)
            MaxRoot = max(ListOfRoots)


            ## determine number of bins
            if (abs(abs(MinRoot) - abs(MaxRoot)) < 1.e-6):
                NumBins = 2
            else:
                NumBins = int((abs(velLowLimit) + abs(velUpLimit)) / VelBin)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## compute (weighted) KDE using scipy routine
            if (len(ListOfRoots) > 1):
                KDEFactor = float(VelBin) / float(NumBins)
                ErrorFlag = False
                gkde = stats.gaussian_kde(ListOfRoots, bw_method = KDEFactor)


                ## define stepsize for sampling of KDE
                if (abs(abs(MinRoot) - abs(MaxRoot)) < 1.e-6):
                    KDEStep = 2
                else:
                    KDEStep = int((abs(velLowLimit) + abs(velUpLimit)) / (VelBin * 0.1))

                # Debug:
                # print ("KDEStep = ", KDEStep)


                ## compute density function for the estimated density function
                xKDE = numpy.linspace(velLowLimit, velUpLimit, KDEStep)
                yKDE = gkde.evaluate(xKDE)
                yKDE = numpy.nan_to_num(yKDE)

                # Debug:
                # print ("\nUseWeightedKDE = ", UseWeightedKDE)
                # print ("xKDE = ", xKDE)
                # print ("yKDE = ", yKDE)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## determine 3 sigma level assuming that the underlying density function is unimodal and gaussian.
                Hist3Sigma = yKDE.mean() + 3.0 * yKDE.std()
                Hist3SigmaString = "{:.4e}".format(Hist3Sigma)

                # Debug:
                # print ("Hist3Sigma = ", Hist3Sigma)
            else:
                xKDE = [ListOfRoots[0]]
                yKDE = [1.0]
                Hist3Sigma = 0.0
                Hist3SigmaString = "{:.4e}".format(Hist3Sigma)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## plot kde
            if (len(ListOfRoots) > 1):
                fig, layer = pylab.subplots()
                fig.set_size_inches(15.0, 10.0)
                layer.grid(True)
                layer.set_ylabel(r"frequency")
                layer.set_xlabel("velocity [km/s]")


                ## add title
                if (TypeFlag == 0):
                    layer.set_title(r"all molecules: Gaussian kernel density of central velocities of  EMISSION  peaks, 3 $\sigma$ = " \
                                    + Hist3SigmaString)
                else:
                    layer.set_title(r"all molecules: Gaussian kernel density of central velocities of  ABSORPTION  peaks, 3 $\sigma$ = " \
                                    + Hist3SigmaString)


                ## add histogram as well
                Hist1, Bins1 = numpy.histogram(ListOfRoots, bins = NumBins, density = True)
                SelectedHistArray = numpy.asarray(Hist1)                                    ## convert list of histogram to numpy array
                width = 0.8 * (Bins1[1] - Bins1[0])
                center = (Bins1[:-1] + Bins1[1:]) / 2
                layer.bar(center, SelectedHistArray, align = 'center', width = width, label = "histogram")


                ## add KDE to plot
                layer.plot(xKDE, yKDE, color = "black", linewidth = 2, linestyle = '-', label = "KDE")


                ## add 3sigma level as horizontal line to plot
                layer.axhline(y = Hist3Sigma, color = 'green', linewidth = 2, linestyle = '-')


                ## set x-range limits and add text to 3sigma line
                layer.set_xlim(velLowLimit, velUpLimit)
                layer.text(min(velLowLimit * 0.98, velUpLimit), (Hist3Sigma * 0.98), r'3 $\sigma$', color='green')
                layer.set_ylim(0.0, max(numpy.nanmax(yKDE), Hist3Sigma) * 1.07)
                layer.legend()


                ## create subdirectory for KDE plots and save plot to file
                OutputFileName = SummaryDir + "All-molecules___kde__" + TypeFlagString
                fig.savefig(OutputFileName + "__" + str(velLowLimit) + "_-_" + str(velUpLimit) + ".png", dpi=300)
                pylab.close(fig)


    ## remove pickle files
    cmdString = "rm -rf " + SummaryDir + "*.pickle"
    os.system(cmdString)


    ## define return parameter
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## plot a single transition window
def PlotTranstionWindow(RawDataPlotFlag, counterPNG, TransFreq, TransitionParameter, NoiseLevel, ObsDataTransWindow, ModeledSpectra, OutputDir, \
                        MolNameFileName, WeigthingRange = [], ContinuumArray = [], Deriv1st = [], Deriv2nd = [], IntOptical = [], velLowLimit = None, \
                        velUpLimit = None, vComp = None, LineCatalog = None):
    """

input parameters:
-----------------

    - TransFreq:                transition frequency

    - ObsDataTransWindow:       observational data within current window

    - RawDataPlotFlag:          flag indicating if obs. data of transition window is plotted only

    - ModeledSpectra:           modeled spectrum (from myXCLASS)

    - OutputDir:                ouptut directory

    - MolNameFileName:          name of current molecule

    - counterPNG:               index variable

    - TransitionParameter:      transition parameters

    - NoiseLevel:               user defined noise level

    optional parameters:

    - Deriv1st:                 1st derivative of observational data

    - Deriv2nd:                 2nd derivative of observational data

    - WeigthingRange:           range for weighting

    - ContinuumArray:           array contain continuum

    - IntOptical:               modeled components (from myXCLASS)

    - velLowLimit:              lower limit of velocity

    - velUpLimit:               upper limit of velocity

    - vComp:                    current velocity

    - LineCatalog:              catalog of other molecule transitions



output parameters:
------------------

    - None
    """

    # Debug:
    # print ("RawDataPlotFlag = ", RawDataPlotFlag)
    # print ("counterPNG = ", counterPNG)
    # print ("TransFreq = ", TransFreq)
    # print ("TransitionParameter = ", TransitionParameter)
    # print ("NoiseLevel = ", NoiseLevel)
    # print ("ObsDataTransWindow = ", ObsDataTransWindow)
    # print ("ModeledSpectra = ", ModeledSpectra)
    # print ("OutputDir = ", OutputDir)
    # print ("MolNameFileName = ", MolNameFileName)
    # print ("WeigthingRange = ", WeigthingRange)
    # print ("ContinuumArray = ", ContinuumArray)
    # print ("Deriv1st = ", Deriv1st)
    # print ("Deriv2nd = ", Deriv2nd)
    # print ("IntOptical = ", IntOptical)
    # print ("velLowLimit = ", velLowLimit)
    # print ("velUpLimit = ", velUpLimit)
    # print ("vComp = ", vComp)
    # print ("LineCatalog = " , LineCatalog)


    ## define some constants
    cms = 299792458.0
    ckms = cms * 1.e-3
    FreqMin = numpy.nanmin(ObsDataTransWindow[:, 0])
    FreqMax = numpy.nanmax(ObsDataTransWindow[:, 0])
    yMinObs = numpy.nanmin(ObsDataTransWindow[:, 2])
    yMaxObs = numpy.nanmax(ObsDataTransWindow[:, 2])
    yMinMod = None
    yMaxMod = None


    ## define styles for lines
    OwnLineStyle = ["g-", "b-", "c-", "m-", "g--", "b--", "c--", "m--", "g-.", "b-.", "c-.", "m-.", "g..", "b..", "c..", "m.."]


    ## initialize plot environment
    fig, layer = pylab.subplots()
    fig.set_size_inches(15.0, 10.0)
    layer.grid(True)
    layer.xaxis.set_major_formatter(pylab.FormatStrFormatter('%.5e'))
    layer.set_ylabel(r"Brightness Temperature [K]")
    layer.set_xlabel("Rest Frequency [MHz]")
    TransFreqString = "{:.2f}".format(TransFreq)
    EinsteinA = TransitionParameter[1]
    EinsteinAString = "{:.2e}".format(EinsteinA)
    ElowMin = max(1.0, TransitionParameter[2])
    ElowMinString = "{:.2f}".format(ElowMin)
    gup = TransitionParameter[3]
    LocalStrength = ((gup * EinsteinA) / (max(1.0, ElowMin)**2))
    LocalStrengthDBString = "{:.3e}".format(LocalStrength)
    TitleString = ""
    if (vComp is not None):
        TitleString = "v$_{comp}$ = {:.2f} km s$^{-1}$, \n ".format(vComp)
    TitleString += "$\\nu_{Trans}$ = " + TransFreqString + " MHz,  "
    TitleString += "Einstein A = " + EinsteinAString + " s$^{-1}$,  "
    TitleString += "g$_{up}$ = " + str(gup) + ",  "
    TitleString += "E$_{low}$ = " + ElowMinString + " K,  "
    TitleString += "strength = (g$_{up}$ * Einstein A) / (E$_{low}^2$) = " + LocalStrengthDBString



    ## determine min. and max. velocities
    RestFreq = TransFreq
    vLSR = 0.0
    f1 = FreqMin
    f2 = FreqMax
    if (velLowLimit is not None and velUpLimit is not None):
        vi = task_myXCLASS.ConvertFreq(TransFreq, velLowLimit)
        vf = task_myXCLASS.ConvertFreq(TransFreq, velUpLimit)
        vl = min(vi, vf)
        vu = max(vi, vf)
        fl = min(f1, f2)
        fu = max(f1, f2)
        fi = max(fl, vl)
        ff = min(fu, vu)
        v1 = velLowLimit
        v2 = velUpLimit
    else:
        v1 = vLSR + (ckms / RestFreq) * (RestFreq - f1)
        v2 = vLSR + (ckms / RestFreq) * (RestFreq - f2)
        fi = f1
        ff = f2


    ## add transitions of other molecules
    ## LineCatalog = [TransFreq, EinsteinA, ELow, gup, LocalWeight, name]
    StrengthRatio = 1.0
    if (LineCatalog is not None):
        TransFreqList = LineCatalog[:, 0].astype(float)
        ElowList = LineCatalog[:, 2].astype(float)
        StrengthList = LineCatalog[:, 4].astype(float)
        tf1 = max(0, (numpy.abs(TransFreqList[:] - fi)).argmin())
        tf2 = min(len(TransFreqList[:]) - 1, (numpy.abs(TransFreqList[:] - ff)).argmin())
        MinTFIndex = min(tf1, tf2)
        MaxTFIndex = max(tf1, tf2)
        LocalTransFreqList = TransFreqList[MinTFIndex:MaxTFIndex]
        if (len(LocalTransFreqList) > 1):
            LocalStrengthList = StrengthList[MinTFIndex:MaxTFIndex]
            LocalElowList = ElowList[MinTFIndex:MaxTFIndex]
            LocalMolNameList = LineCatalog[MinTFIndex:MaxTFIndex, 5]



            ## define ratio of line strengths
            ReduedStrengthList = LocalStrengthList[numpy.where(LocalStrengthList != LocalStrength)]
            if (len(ReduedStrengthList) > 0):
                MaxStrength = numpy.nanmax(ReduedStrengthList)
                if (abs(MaxStrength) > 1.e-25):
                    StrengthRatio = (LocalStrength / MaxStrength)
            TitleString += ", strength-ratio = {:.3e}".format(StrengthRatio)


            ## add additional axis for line strength
            ax3 = layer.twinx()
            bar_width = abs(ObsDataTransWindow[1, 0] - ObsDataTransWindow[0, 0])
            ax3.bar(LocalTransFreqList, LocalStrengthList, bar_width, alpha = 0.3, color = 'green', align = 'center')
            ax3.bar((TransFreq), (LocalStrength), (bar_width * 2.0), alpha = 0.6, color = 'orange', align = 'center')
            ax3.set_xlim(fi, ff)
            ax3.set_ylim(numpy.nanmin(LocalStrengthList), numpy.nanmax(LocalStrengthList))
            ax3.tick_params('y', colors='green')
            ax3.set_ylabel(r'transition strength = (g$_{up}$ * Einstein A) / (E$_{low}^2$)', color = 'green')
            ax3.yaxis.set_major_formatter(pylab.matplotlib.ticker.FormatStrFormatter('%.2e'))
            ax3.axhline(y = LocalStrength, color = 'orange', alpha = 0.5, linewidth = 1, linestyle = '--')
            # ax3.set_yscale('log')                                                         ## activate for log plot

            # Debug:
            # print ("\nLocalTransFreqList = ", LocalTransFreqList)
            # print ("LocalStrengthList = ", LocalStrengthList)
            # print ("fi, ff = ", fi, ff)


            ## add name of molecule
            for LocalTransFreqID, LocalTransFreq in enumerate(LocalTransFreqList):
                CurrentStrength = LocalStrengthList[LocalTransFreqID]
                CurrentElow = LocalElowList[LocalTransFreqID]
                # if (CurrentStrength > LocalStrength and CurrentElow <= 5.0 * ElowMin):
                CurrentMolName = LocalMolNameList[LocalTransFreqID]
                ax3.text((LocalTransFreq + 0.5 * bar_width), CurrentStrength, CurrentMolName, horizontalalignment = 'right', \
                          verticalalignment = 'top', rotation = 90.0, alpha = 0.3, clip_on = True)


    ## add title
    layer.set_title(TitleString, y = 1.08)


    ## mark weighted areas
    if (WeigthingRange != []):
        MinFreqLocal = 1.e99
        MaxFreqLocal = -1.e99
        f1 = WeigthingRange[0]
        f2 = WeigthingRange[1]
        layer.axvspan(min(f1, f2), max(f1, f2), alpha = 0.5, color = 'grey')


    ## add observational data to plot
    if (vComp is None):
        layer.axvline(x = TransFreq, color = 'blue', linewidth = 3, linestyle = '-')
    else:
        nu0 = task_myXCLASS.ConvertFreq(TransFreq, vComp)
        layer.axvline(x = nu0, color = 'blue', linewidth = 3, linestyle = '-')
    if (len(ObsDataTransWindow[:, 0]) > 2):
        try:
            f1 = max(0, (numpy.abs(ObsDataTransWindow[:, 0] - fi)).argmin() - 1)
            f2 = min(len(ObsDataTransWindow[:, 0]) - 1, (numpy.abs(ObsDataTransWindow[:, 0] - ff)).argmin() + 1)
            i1 = min(f1, f2)
            i2 = max(f1, f2)
            layer.plot(ObsDataTransWindow[i1:i2, 0], ObsDataTransWindow[i1:i2, 2], '-', color='black', linewidth=3.0, label='obs. data', \
                       drawstyle='steps-mid')
            yMinMod = numpy.nanmin(ObsDataTransWindow[i1:i2, 2])
            yMaxMod = numpy.nanmax(ObsDataTransWindow[i1:i2, 2])
        except RuntimeError:
            print ("\n\n\n\n\n")
            print ("len(ObsDataTransWindow[:, 0]) = ", len(ObsDataTransWindow[:, 0]))
            print ("ObsDataTransWindow[:, 0] = ", ObsDataTransWindow[:, 0])
            print ("ObsDataTransWindow[:, 1] = ", ObsDataTransWindow[:, 1])
            print ("\n\n\n\n\n")


    ## add first and second derivatives to plot
    if (Deriv1st != []):
        layer.plot(Deriv1st[:, 0], Deriv1st[:, 1], '-', color = 'blue', linewidth = 3.0, label = '1st deriv. of data')
    if (Deriv2nd != []):
        layer.plot(Deriv2nd[:, 0], Deriv2nd[:, 1], '-', color = 'green', linewidth = 3.0, label = '2nd deriv. data')


    ## add modeled spectrum to plot
    if (not RawDataPlotFlag):
        f1 = max(0, (numpy.abs(ModeledSpectra[:, 0] - fi)).argmin() - 1)
        f2 = min(len(ModeledSpectra[:, 0]) - 1, (numpy.abs(ModeledSpectra[:, 0] - ff)).argmin() + 1)
        k1 = min(f1, f2)
        k2 = max(f1, f2)
        yMinMod = min(yMinMod, numpy.nanmin(ModeledSpectra[k1:k2, 1]))
        yMaxMod = max(yMaxMod, numpy.nanmax(ModeledSpectra[k1:k2, 1]))
        if (len(ModeledSpectra[:, 0]) > 2):
            try:
                layer.plot(ModeledSpectra[k1:k2, 0], ModeledSpectra[k1:k2, 1], '-', color = 'red', linewidth = 3.0, label = 'fit')
            except RuntimeError:
                print ("\n\n\n\n\n")
                print ("len(ModeledSpectra[:, 0]) = ", len(ModeledSpectra[:, 0]))
                print ("ModeledSpectra[:, 0] = ", ModeledSpectra[:, 0])
                print ("ModeledSpectra[:, 1] = ", ModeledSpectra[:, 1])
                print ("\n\n\n\n\n")


        ## add continuum array and mark noise band
        if (len(ContinuumArray) > 0):
            x = ContinuumArray[:, 0]
            y = ContinuumArray[:, 1]
            if (NoiseLevel > 0.0):
                layer.plot(x, y + NoiseLevel, '--', color = 'blue', linewidth = 1.0)
                layer.plot(x, y - NoiseLevel, '--', color = 'blue', linewidth = 1.0)
                layer.fill_between(x, y + NoiseLevel, y - NoiseLevel, facecolor = 'blue', alpha = 0.3)
            else:
                layer.plot(x, y, '--', color = 'blue', linewidth = 2.0)
        elif (NoiseLevel > 0.0):
            layer.axhspan(-NoiseLevel, NoiseLevel, alpha = 0.3, color = 'blue')


        ## plot intensities for each component and add continuum to component intensities
        if (IntOptical != []):
            if (len(IntOptical[0][:]) > 1):
                NumComp = len(IntOptical[0][:])
                for IndexComp in range(NumComp):
                    LengthInt = len(IntOptical[0][0][2])
                    IntComp = numpy.zeros((LengthInt, 2), dtype=numpy.float32)
                    FreqMinInt = IntOptical[0][0][2][0, 0]


                    ## to do:  what happened to molfit files including definitions for tback and tSlope (not globally)
                    for ii in range(LengthInt):
                        IntComp[ii, 0] = IntOptical[0][IndexComp][2][ii, 0]
                        IntComp[ii, 1] = IntOptical[0][IndexComp][2][ii, 1]     # + abs(tBack) * (IntComp[ii, 0] / FreqMinInt)**tSlope
                    layer.plot(IntComp[:, 0], IntComp[:, 1], OwnLineStyle[IndexComp], linewidth=1.5, label = 'int. comp. ' + str(IndexComp + 1))

    # Debug:
    # print ("\n\n")
    # print ("TransFreq = ", TransFreq)
    # print ("RestFreq = ", RestFreq)
    # print ("f1, v1 = ", f1, v1)
    # print ("f2, v2 = ", f2, v2)


    ## add noise level
    if (NoiseLevel >= 0.0 and RawDataPlotFlag):
        layer.axhline(y = NoiseLevel, color = 'blue', linewidth = 1, linestyle = '--')
        layer.axhline(y = -NoiseLevel, color = 'blue', linewidth = 1, linestyle = '--')

    # Debug:
    #    if (abs(TransFreq - 345448) < 5.0):
    #        print "\n\nTransFreq = ", TransFreq
    #        print "fi = ", fi
    #        print "ff = ", ff
    #        print "yMinMod = ", yMinMod
    #        print "yMaxMod = ", yMaxMod
    #        print "ObsDataTransWindow[i1:i2, 2] = ", ObsDataTransWindow[i1:i2, 2]
    #        print "ModeledSpectra[k1:k2, 1] = ", ModeledSpectra[k1:k2, 1]


    ## restrict x- and y-axis
    layer.set_xlim(fi, ff)
    if (yMinMod is None or yMaxMod is None):
        layer.set_ylim(yMinMod, yMaxMod * 1.07)


    ## add velocity axis
    try:
        ax2 = layer.twiny()
        vLow = min(v1, v2)
        vUp = max(v1, v2)
        if (FreqMin < FreqMax):
            ax2.set_xlim(vUp, vLow)
        else:
            ax2.set_xlim(vLow, vUp)
        ax2.set_xlabel("velocity [km/s]")
        ax2.axvline(x = vLSR, color = 'black', linewidth = 1, linestyle = '-')
        ax2.xaxis.set_major_formatter(pylab.matplotlib.ticker.FormatStrFormatter('%.2f'))
        if (yMinMod is None or yMaxMod is None):
            ax2.set_ylim(yMinMod, yMaxMod * 1.07)
    except RuntimeError:
        print ("\n\n\n\n\n")
        print ("\tWARNING:  RuntimeError in subroutine PlotTranstionWindow!")
        print ("\t\tv1 = ", v1)
        print ("\t\tv2 = ", v2)
        print ("\n\n\n\n\n")


    ## add legend
    layer.xaxis.set_major_formatter(pylab.matplotlib.ticker.FormatStrFormatter('%.2f'))
    layer.legend()


    ## save figure
    if (RawDataPlotFlag):
        OutputDirLocal = OutputDir + "TransistionWindow/"
        if (not os.path.exists(OutputDirLocal)):
            os.system("mkdir -p " + OutputDirLocal)
    else:
        OutputDirLocal = OutputDir
    vtString = "{:.4f}".format(TransFreq)
    #    OutputFileName = OutputDirLocal + MolNameFileName + "___" + str(FreqMin) + "_-_" + str(FreqMax) + "_MHz____vt_=_" + vtString + "__" \
    #                     + str(counterPNG)
    OutputFileName = OutputDirLocal + MolNameFileName + "____strength_=_" + LocalStrengthDBString + "____vt_=_" + vtString + "_MHz__" \
                     + str(counterPNG)
    fig.savefig(OutputFileName + ".png", dpi=300)
    pylab.close(fig)

    # Debug:
    # print ("OutputFileName = ", OutputFileName)


    ## save modeled spectrum
    if (RawDataPlotFlag):
        numpy.savetxt(OutputFileName + ".dat", ObsDataTransWindow[:, 1:], fmt = '%1.10e')


    ## we've finish
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## calculate partition function for T = T_rot
def GetPartitionFunction(MolName, Trot, dbList):
    """

input parameters:
-----------------

    - MolName:                  name of molecule

    - Trot:                     rotation temperature

    - dbList:                   path and name of database file


output parameters:
------------------

    - QT:                       partition function for T = Trot
    """

    # Debug:
    # print ("MolName = ", MolName)
    # print ("Trot = ", Trot)
    # print ("dbList = ", dbList)


    ## initialize return parameter
    QT = 1.0


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define query string


    ## set names of columns
    NameOfPartFuncTable = "partitionfunctions"
    ColumnNameForNamePartFunc = 'PF_Name'
    ColumnNamesPartFunc = ['PF_1_072', 'PF_1_148', 'PF_1_230', 'PF_1_318', 'PF_1_413', 'PF_1_514', 'PF_1_622', 'PF_1_738', 'PF_1_862', 'PF_1_995', \
                           'PF_2_138', 'PF_2_291', 'PF_2_455', 'PF_2_630', 'PF_2_725', 'PF_2_818', 'PF_3_020', 'PF_3_236', 'PF_3_467', 'PF_3_715', \
                           'PF_3_981', 'PF_4_266', 'PF_4_571', 'PF_4_898', 'PF_5_000', 'PF_5_248', 'PF_5_623', 'PF_6_026', 'PF_6_457', 'PF_6_918', \
                           'PF_7_413', 'PF_7_943', 'PF_8_511', 'PF_9_120', 'PF_9_375', 'PF_9_772', 'PF_10_471', 'PF_11_220', 'PF_12_023', 'PF_12_882', \
                           'PF_13_804', 'PF_14_791', 'PF_15_849', 'PF_16_982', 'PF_18_197', 'PF_18_750', 'PF_19_498', 'PF_20_893', 'PF_22_387', \
                           'PF_23_988', 'PF_25_704', 'PF_27_542', 'PF_29_512', 'PF_31_623', 'PF_33_884', 'PF_36_308', 'PF_37_500', 'PF_38_905', \
                           'PF_41_687', 'PF_44_668', 'PF_47_863', 'PF_51_286', 'PF_54_954', 'PF_58_884', 'PF_63_096', 'PF_67_608', 'PF_72_444', \
                           'PF_75_000', 'PF_77_625', 'PF_83_176', 'PF_89_125', 'PF_95_499', 'PF_102_329', 'PF_109_648', 'PF_117_490', 'PF_125_893', \
                           'PF_134_896', 'PF_144_544', 'PF_150_000', 'PF_154_882', 'PF_165_959', 'PF_177_828', 'PF_190_546', 'PF_204_174', 'PF_218_776', \
                           'PF_225_000', 'PF_234_423', 'PF_251_189', 'PF_269_153', 'PF_288_403', 'PF_300_000', 'PF_309_030', 'PF_331_131', 'PF_354_813', \
                           'PF_380_189', 'PF_407_380', 'PF_436_516', 'PF_467_735', 'PF_500_000', 'PF_501_187', 'PF_537_032', 'PF_575_440', 'PF_616_595', \
                           'PF_660_693', 'PF_707_946', 'PF_758_578', 'PF_812_831', 'PF_870_964', 'PF_933_254', 'PF_1000_000']

    ## set query string
    query_string = "SELECT"
    query_string += " {:s}".format(ColumnNameForNamePartFunc)                               ## position 1: molecule name
    for local_temp in ColumnNamesPartFunc:
        query_string += ", {:s}".format(local_temp)                                         ## position 2 - 111: temperature
    query_string += " FROM {:s} WHERE (".format(NameOfPartFuncTable)
    query_string += " {:s} = {:s}{:s}{:s})".format(ColumnNameForNamePartFunc, chr(34), MolName, chr(34))

    # Debug:
    # print ("query_string = ", query_string)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get partition function values from database
    dbFile = dbList[0]
    try:
        conn = sqlite3.connect(dbFile)
    except sqlite3.Error as e:
        print ("\n\nError in subroutine LineIdentification.GetPartitionFunction:")
        print ("\t\tCan not connect to sqlite3 database {:s}.".format(dbFile))
        print ("\t\tError: {:d}: {:s}".format(e.args[0], e.args[1]))
        sys.exit(1)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read data from database
    cursor = conn.cursor()
    cursor.execute(query_string)
    conn.commit()
    rows = cursor.fetchall()
    conn.close()

    # Debug:
    # print ("rows = ", rows)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## extract partition function values
    TempArray = [1.072, 1.148, 1.23, 1.318, 1.413, 1.514, 1.622, 1.738, 1.862, 1.995, 2.138, 2.291, 2.455, 2.63, 2.725, 2.818, 3.02, 3.236, 3.467, \
                 3.715, 3.981, 4.266, 4.571, 4.898, 5, 5.248, 5.623, 6.026, 6.457, 6.918, 7.413, 7.943, 8.511, 9.12, 9.375, 9.772, 10.471, 11.22, \
                 12.023, 12.882, 13.804, 14.791, 15.849, 16.982, 18.197, 18.75, 19.498, 20.893, 22.387, 23.988, 25.704, 27.542, 29.512, 31.623, \
                 33.884, 36.308, 37.5, 38.905, 41.687, 44.668, 47.863, 51.286, 54.954, 58.884, 63.096, 67.608, 72.444, 75, 77.625, 83.176, 89.125, \
                 95.499, 102.329, 109.648, 117.49, 125.893, 134.896, 144.544, 150, 154.882, 165.959, 177.828, 190.546, 204.174, 218.776, 225, \
                 234.423, 251.189, 269.153, 288.403, 300, 309.03, 331.131, 354.813, 380.189, 407.38, 436.516, 467.735, 500, 501.187, 537.032, \
                 575.44, 616.595, 660.693, 707.946, 758.578, 812.831, 870.964, 933.254, 1000.0]
    QTArray = rows[0][1:]

    # Debug:
    # print ("TempArray = ", TempArray)
    # print ("len(TempArray) = ", len(TempArray))
    # print ("len(QTArray) = ", len(QTArray))


    ## interpolate QT array to get partition function at T = T_rot
    try:
        QT = numpy.interp(Trot, TempArray, QTArray)
    except:
        QT = 1.0

    # Debug:
    # print ("\nTrot = ", Trot)
    # print ("QT = ", QT)


    ## define return parameter
    return QT
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## creates the io-control xml file in the current job directory
##
def CreateIOControlXMLFile(JobDir, MAGIXExpXML, MolfitsFileName, MAGIXrootDir):
    """

input parameters:
-----------------

    - JobDir:               path of job directory

    - MAGIXExpXML:          path and name of an experimental xml file

    - MolfitsFileName:      path and name of a molfits file

    - MAGIXrootDir:         path of MAGIX root directory


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("JobDir = ", JobDir)
    # print ("MAGIXExpXML = ", MAGIXExpXML)
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("MAGIXrootDir = ", MAGIXrootDir)


    ## define i/o xml file
    iocontrol = open(JobDir + "io_control.xml", 'w')
    iocontrol.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
    iocontrol.write("<ioControl>\n")
    iocontrol.write("    <title>io control</title>\n")
    iocontrol.write("    <description>This xml-file contains the paths and the names of all files used by MAGIX during the fit process</description>\n")
    iocontrol.write("    <PathFilename>\n")
    iocontrol.write("\n")
    iocontrol.write("        <experimental_data>\n")
    if (MAGIXExpXML != ""):
        iocontrol.write("            <filename>" + MAGIXExpXML + "</filename>\n")
    else:
        iocontrol.write("            <filename>" + JobDir + "exp.xml</filename>\n")
    iocontrol.write("            <description>path and name of the experimental file</description>\n")
    iocontrol.write("        </experimental_data>\n")
    iocontrol.write("\n")
    iocontrol.write("        <parameter_file>\n")
    iocontrol.write("            <filename>" + MolfitsFileName + "</filename>\n")
    iocontrol.write("            <description>path and name of the xml-file including the start values of each parameter</description>\n")
    iocontrol.write("        </parameter_file>\n")
    iocontrol.write("\n")
    iocontrol.write("        <fit_control>\n")
    iocontrol.write("            <filename>" + JobDir + "algorithm_control.xml</filename>\n")
    iocontrol.write("            <description>path and name of the xml file controlling the fitting process</description>\n")
    iocontrol.write("        </fit_control>\n")
    iocontrol.write("\n")
    iocontrol.write("        <fit_log>\n")
    iocontrol.write("            <filename>" + JobDir + "fit.log</filename>\n")
    iocontrol.write("            <description>path and name of log file describing the fitting process</description>\n")
    iocontrol.write("        </fit_log>\n")
    iocontrol.write("\n")
    iocontrol.write("        <model_description>\n")
    iocontrol.write("            <filename>" + MAGIXrootDir + "Fit-Functions/myXCLASS/xml/myNewXCLASS__LineID.xml</filename>\n")
    iocontrol.write("            <description>path and name of the xml-description of the input/output file of the fit function module</description>\n")
    iocontrol.write("        </model_description>\n")
    iocontrol.write("\n")
    iocontrol.write("    </PathFilename>\n")
    iocontrol.write("</ioControl>\n")
    iocontrol.write("\n")
    iocontrol.close()


    ## we're done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## creates molfit file for the current molecule, and writes created molfits file to current molecule subdirectory
##
def CreateMolfitFile(CurrentDir, myXCLASSRootDirectory, MoleculejobDir, MolfitsFileName, MoleculeName, SourceName, DefaultMolfitFile):
    """

input parameters:
-----------------

    - CurrentDir:               current directory

    - myXCLASSRootDirectory:    XCLASS interface root directory

    - MoleculejobDir:           working directory for current molecule

    - MolfitsFileName:          path and name of molfits file

    - MoleculeName:             full name of current molecule

    - SourceName:               name of source

    - DefaultMolfitFile:        path and name of a default molfit file used for molecules which are not described by source templates


output parameters:
------------------

    - DefaultMolfitUsedFlag:    flag indicating if default molfit file was used


    """
    # Debug:
    # print ("CurrentDir = ", CurrentDir)
    # print ("myXCLASSRootDirectory = ", myXCLASSRootDirectory)
    # print ("MoleculejobDir = ", MoleculejobDir)
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("MoleculeName = ", MoleculeName)
    # print ("SourceName = ", SourceName)
    # print ("DefaultMolfitFile = ", DefaultMolfitFile)


    ## define some internal parameters
    MolfFitFileCreatedFlag = False                                                          ## no molfit file is created
    DefaultMolfitUsedFlag = True
    SourceName = SourceName.strip()
    DefaultMolfitFile = DefaultMolfitFile.strip()

    # Debug:
    # print ("SourceName = ", SourceName)
    # print ("DefaultMolfitFile = ", DefaultMolfitFile)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## test if template directory already exists
    MolfitFileNotCreatedFlag = True
    if (SourceName != ""):


        ## define template directory
        # TemplateDirectory = myXCLASSRootDirectory + "???????????/Templates/"
        # SourceMolfitFile = TemplateDirectory + SourceName + ".molfit"

        # Debug:
        # print ("TemplateDirectory = ", TemplateDirectory)


        ## check if template for the current source already exists
        if (SourceName[0] != "/"):                                                          ## make relative path absolute
            SourceMolfitFile = CurrentDir + SourceName
        else:
            SourceMolfitFile = SourceName
        if (not SourceMolfitFile.endswith(".molfit")):
            SourceMolfitFile = SourceMolfitFile + ".molfit"
        if (os.path.isfile(SourceMolfitFile)):                                              ## check, if template for current molecule and source exists

            # Debug:
            # print ("SourceMolfitFile = ", SourceMolfitFile)


            ## get all parameters from source molfit file
            MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(SourceMolfitFile)


            ## create new molfit file in new format
            CurrentMoleculeFlag = "false"
            NewMolfitFileContent = []
            for MoleculeIndex in range(len(MoleculesInMolfitFile)):
                LocalMolName = MoleculesInMolfitFile[MoleculeIndex]
                if (LocalMolName.strip() == MoleculeName.strip()):
                    CurrentMoleculeFlag = "true"
                    LocalMolfitParameters = AllParameters[MoleculeIndex]


                    ## create final contribution to all further molfit files of identified strong molecule
                    NewLine = MoleculeName.strip() + "    " + str(len(LocalMolfitParameters))
                    NewMolfitFileContent.append(NewLine)
                    for line in LocalMolfitParameters:

                        # Debug:
                        # print ("line = ", line)


                        ## create new formatted line
                        NewLine = task_myXCLASS.WriteLineMolfitFile(line)


                        ## append new line to NewMolfitFileContent list
                        NewMolfitFileContent.append(NewLine)


            ## write part of template which describes the current molecule to current working directory
            if (NewMolfitFileContent != []):
                MolfitFileNotCreatedFlag = False
                NewMolfitFile = open(MolfitsFileName, 'w')
                NewMolfitFile.write("% Number of molecules =     1\n")                      ## write first line
                for line in NewMolfitFileContent:
                    NewMolfitFile.write(line + "\n")
                NewMolfitFile.close()
                MolfFitFileCreatedFlag = True
                DefaultMolfitUsedFlag = False
                return DefaultMolfitUsedFlag


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check, if default molfit file is defined
    if ((not MolfFitFileCreatedFlag) and DefaultMolfitFile != "" and MolfitFileNotCreatedFlag):
        DefaultMolfitUsedFlag = False
        if (DefaultMolfitFile.endswith(".molfit")):                                         ## check, if variable 'DefaultMolfitFile' ends with ".molfit"
            MolfFitFileCreatedFlag = True


            ## if necessary, make relative path absolute
            if (DefaultMolfitFile[0] != "/"):
                filename = CurrentDir + DefaultMolfitFile
            else:
                filename = DefaultMolfitFile


            ## copy default molfit file to current working directory
            task_myXCLASS.NormNCopyMove(filename, MolfitsFileName, copyFlag = True)


            ## get all parameters from source molfit file
            MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(MolfitsFileName)

            # Debug:
            # print ("MoleculesInMolfitFile = ", MoleculesInMolfitFile)
            # print ("AllParameters = ", AllParameters)


            ## create new molfit file in new format
            NewMolfitFileContent = []
            for MoleculeIndex in range(len(MoleculesInMolfitFile)):
                LocalMolName = MoleculesInMolfitFile[MoleculeIndex]
                LocalMolfitParameters = AllParameters[MoleculeIndex]


                ## create final contribution to all further molfit files of identified strong molecule
                if (MoleculeIndex == 0):
                    NewLine = MoleculeName.strip() + "    " + str(len(LocalMolfitParameters))
                else:
                    NewLine = LocalMolName.strip() + "    " + str(len(LocalMolfitParameters))
                NewMolfitFileContent.append(NewLine)
                for line in LocalMolfitParameters:

                    # Debug:
                    # print ("\n--line = ", line)


                    ## create new formatted line
                    NewLine = task_myXCLASS.WriteLineMolfitFile(line)


                    ## append new line to NewMolfitFileContent list
                    NewMolfitFileContent.append(NewLine)


            ## write part of template which describes the current molecule to current working directory
            if (NewMolfitFileContent != []):
                NewMolfitFile = open(MolfitsFileName, 'w')
                # NewMolfitFile.write("% Number of molecules =     " + str(len(LocalMolfitParameters)) + "\n")
                for line in NewMolfitFileContent:
                    NewMolfitFile.write(line + "\n")
                NewMolfitFile.close()
                MolfFitFileCreatedFlag = True
                DefaultMolfitUsedFlag = True
                return DefaultMolfitUsedFlag
            return DefaultMolfitUsedFlag


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if no default molfit file is defined, create a new one (has to be done!!!!!!!!)
    if (not MolfFitFileCreatedFlag):
        print ("")

    # Debug:
    # print ("DefaultMolfitUsedFlag = ", DefaultMolfitUsedFlag)


    ## finished
    return DefaultMolfitUsedFlag
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## determines the best fit result for a molecule
##
def GetBestResult(JobDir, ConvertLogLinFlag, ListOfSpectraNames, NumberExpRangesList, ListOfAllFreqMin, \
                  ListOfAllFreqMax, tBackList, tSlopeList, SQLParameters = None, \
                  LocalMinColumnDensityAbs = 1.0, LocalMinColumnDensityEmis = 1.0):
    """

input parameters:
-----------------

    - JobDir:                       path of job directory

    - ConvertLogLinFlag:            defines if column densities in the fitted molfit file(s) are converted back to linear values

    - ListOfSpectraNames:           list of exp. data file names

    - NumberExpRangesList:          list of frequency numbers

    - ListOfAllFreqMin:             list containing all min. frequencies of all frequency ranges

    - ListOfAllFreqMax:             list containing all max. frequencies of all frequency ranges

    - tBackList:                    list containing the background temperatures for all frequency ranges

    - tSlopeList:                   list containing the temperature slopes for all frequency ranges

    - SQLParameters:                (optional): SQL parameters

    - LocalMinColumnDensityAbs:     (optional): min. column density of absorption component

    - LocalMinColumnDensityEmis:    (optional): min. column density of emission component


output parameters:
------------------

    - BestMolfitFileContent:        list of all molfit files

    - IsoRatioFileContent:          optimized iso ratio file

    - ListSubtractedModelSpectra:   list of all function values for all molfit files

    """

    # Debug:
    # print ("JobDir = ", JobDir)
    # print ("ConvertLogLinFlag = ", ConvertLogLinFlag)
    # print ("ListOfSpectraNames = ", ListOfSpectraNames)
    # print ("NumberExpRangesList = ", NumberExpRangesList)
    # print ("ListOfAllFreqMin = ", ListOfAllFreqMin)
    # print ("ListOfAllFreqMax = ", ListOfAllFreqMax)
    # print ("tBackList = ", tBackList)
    # print ("tSlopeList = ", tSlopeList)
    # print ("SQLParameters = ", SQLParameters)
    # print ("LocalMinColumnDensityAbs = ", LocalMinColumnDensityAbs)
    # print ("LocalMinColumnDensityEmis = ", LocalMinColumnDensityEmis)


    ## initialize return parameters
    BestMolfitFileContent = []
    IsoRatioFileContent = []
    ListSubtractedModelSpectra = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get all filenames in the current job directory
    listing = os.listdir(JobDir)

    # Debug:
    # print ("\n\n\nlisting = ", listing)


    ## get number of chi2 log files
    NumOutFiles = 0                                                                         ## determine number of output files
    NumMolfitFiles = 0
    for files in listing:
        if (files.endswith(".log.chi2")):
            NumOutFiles += 1
        if (files.endswith(".out.molfit")):
            NumMolfitFiles += 1
    if (NumOutFiles == 0):                                                                  ## check if new input files exist
        print ("\n\nError in XCLASS package, function LineID, subroutine GetBestResult:")
        print ("\tCan not find a chi2 log file!")
        print ("\n\tJobDir = ", JobDir)
        print ("\tlisting = ", listing)
        print ("\n\n\n")
        return (BestMolfitFileContent, IsoRatioFileContent, ListSubtractedModelSpectra)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## determine the chi2 log file, which corresponds to the best fit
    BestChi2Value = 1.e99
    FileNameBestChi2 = ""
    for files in listing:                                                                   ## loop over all files in the current job directory
        if (files.endswith(".log.chi2")):                                                   ## we're interested only in the log.chi2 files
            Chi2LogFile = open(JobDir + files)                                              ## open log.chi2 file
            dummyline = Chi2LogFile.readline()                                              ## read in first line as dummy
            LogLine = Chi2LogFile.readline()                                                ## read second line with best chi2 value
            Chi2LogFile.close()                                                             ## close log file
            LogLine = LogLine.strip()                                                       ## remove blanks
            SplittedLine = LogLine.split()                                                  ## split line into columns
            try:
                chi2Value = float(SplittedLine[1])                                          ## get chi2 value
            except IndexError:
                chi2Value = numpy.nan
            if (not numpy.isnan(chi2Value)):
                if (chi2Value <= BestChi2Value):                                            ## check, if current chi2 value is the best one
                    BestChi2Value = chi2Value                                               ## if yes, save chi2 value
                    FileNameBestChi2 = files                                                ## and corresponding file name

                # Debug:
                # print ("chi2Value = ", chi2Value)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get phrase identifying the used algorithm for the best chi2 value
    BestAlgorithm = ""
    BestAlgorithm = FileNameBestChi2.replace(".log.chi2", "")
    BestAlgorithm = BestAlgorithm.replace("fit__", "")

    # Debug:
    # print ("BestAlgorithm = >>{:s}<<".format(BestAlgorithm))


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## find molfit file
    SingleAlgFlag = False
    OptimizedMolfitFileNameList = []
    for files in listing:                                                                   ## loop over all files in the current job directory
        if (files.endswith(".out.molfit")):
            OptimizedMolfitFileNameList.append(files)
    if (len(OptimizedMolfitFileNameList) == 1):
        SingleAlgFlag = True
    for files in OptimizedMolfitFileNameList:
        i = files.find(BestAlgorithm)
        if (i > (-1) or SingleAlgFlag):


            ## convert column and hydrogen column density (if given for each component) back to linear scale
            if (ConvertLogLinFlag == "true"):
                LogLinearFlag = "linear"
                task_myXCLASS.ConvertNtotToLogScale(JobDir + files, LogLinearFlag, ConverSQLElow = True)


            ## read in molfit file and check, if fitted parameter values are within ranges
            LocalMolfitFileName = JobDir + files
            MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(LocalMolfitFileName)

            # Debug:
            # print ("\n\nLocalMolfitFileName = ", LocalMolfitFileName)
            # print ("MolfitFileForEachMolecule = ", MolfitFileForEachMolecule)


            ## analyze molfit file content
            CreateMolfitFileInNewFormat(LocalMolfitFileName, MolfitFileForEachMolecule, LocalMinColumnDensityAbs, LocalMinColumnDensityEmis, \
                                        SQLParameters = SQLParameters)


            ## read in new output files
            MolfitFile = open(LocalMolfitFileName)
            BestMolfitFileContent = MolfitFile.readlines()
            MolfitFile.close()

            # Debug:
            # print ("\n\nfiles = ", files)
            # print ("BestMolfitFileContent = ", BestMolfitFileContent)
            break


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## find optimized iso ratio file
    for files in listing:                                                                   ## loop over all files in the current job directory
        if (files.startswith("myXClass_isoratios") and files.endswith(".out.input")):
            i = files.find(BestAlgorithm)
            if (i > (-1)):
                IsoRatioFile = open(JobDir + files)
                IsoRatioFileContent = IsoRatioFile.readlines()
                IsoRatioFile.close()
                break


    ## special handling for LM
    if (NumMolfitFiles == 1 and BestAlgorithm.find("LM") > (-1)):
        BestAlgorithm = "LM.out"

    # Debug:
    # print ("BestAlgorithm = ", BestAlgorithm)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## find xml file which corresponds to molfit file
    #    for files in listing:                                                                   ## loop over all files in the current job directory
    #        if (files.endswith(".out.xml")):                                                    ## we're interested only in the molfit files
    #            i = files.find(BestAlgorithm)
    #            if (i > (-1)):
    #                ListOfParamXMLFiles.append(files)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## find model data file
    if (ListOfAllFreqMin != []):
        for ObsDataFileIndex in range(len(ListOfSpectraNames)):                             ## loop over all frequency ranges


            ## analyze exp. data file name, we only want the name, not the path
            ExpDataFileName = ListOfSpectraNames[ObsDataFileIndex][1]                       ## get path and name of corresponding exp. data file
            i = ExpDataFileName.rfind("/")
            if (i > (-1)):
                ExpDataFileName = ExpDataFileName[i + 1:]
                if (ExpDataFileName.endswith(".dat")):
                    ExpDataFileName = ExpDataFileName.replace(".dat", "")
                elif (ExpDataFileName.endswith(".fits")):
                    ExpDataFileName = ExpDataFileName.replace(".fits", "")

            # Debug:
            # print ("\n\n\n\n\n\n\n\n\n\n\n")
            # print ("ExpDataFileName = ", ExpDataFileName)
            # print ("BestAlgorithm = ", BestAlgorithm)


            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            ## find exp. data file in the current single molecule fit directory
            ImportFlag = False
            for files in listing:                                                           ## loop over all files in the current job directory

                # Debug:
                # print ("files = ", files)


                ## file found ?
                if (files == ExpDataFileName + ".out.dat" and SingleAlgFlag):
                    ImportFlag = True
                elif (files.startswith(ExpDataFileName) and (files.endswith(BestAlgorithm) or files.endswith(".out.dat"))):
                    i = files.find(BestAlgorithm)
                    if (i > (-1)):
                        ImportFlag = True


                ## read in model function from file
                if (ImportFlag):
                    SpectrumAll = numpy.loadtxt(JobDir + files, skiprows = 0)
                    ImportFlag = True
                    break

            # Debug:
            # print ("ImportFlag = ", ImportFlag)
            # print ("SpectrumAll = ", SpectrumAll)


            ## get parameters for current range
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberRangeListIn = NumberExpRangesList)
            NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']         ## get number of frequency ranges for current obs. data file

            # Debug:
            # print ("NumberFrequencyRanges = ", NumberFrequencyRanges)


            ## get parameters for current range
            for RangeIndex in range(NumberFrequencyRanges):                                 ## loop over all range definitions in the whole xml file


                ## get parameters for current range
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = ListOfAllFreqMin, \
                                                                                FreqMaxListIn = ListOfAllFreqMax, tBackListIn = tBackList, \
                                                                                tSlopeListIn = tSlopeList)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None and ImportFlag):
                    FreqMax = ObsXMLParameterDictRange['FreqMax']
                    tBack = ObsXMLParameterDictRange['tBack']
                    tSlope = ObsXMLParameterDictRange['tSlope']

                    # Debug:
                    # print ("\n\n\n\n\n\n\n\n\n\n\n")
                    # print ("FreqMin = ", FreqMin)
                    # print ("FreqMax = ", FreqMax)
                    # print ("tBack = ", tBack)
                    # print ("tSlope = ", tSlope)


                    ## get range indices for exp. data
                    LocalCopyArray = SpectrumAll[:, 0]
                    i1 = max(0, (numpy.abs(LocalCopyArray - FreqMin)).argmin() - 1)
                    i2 = min(len(LocalCopyArray) - 1, (numpy.abs(LocalCopyArray - FreqMax)).argmin() + 1)
                    FreqMinIndex = min(i1, i2)
                    FreqMaxIndex = max(i1, i2)

                    # Debug:
                    # print ("FreqMinIndex = ", FreqMinIndex)
                    # print ("FreqMaxIndex = ", FreqMaxIndex)


                    ## subtract continuum
                    SubtractedModelSpectra = SpectrumAll[FreqMinIndex:FreqMaxIndex, :]
                    if (tBack is not None and tBack != 0.0 and tSlope is not None):
                        for i in range(len(SubtractedModelSpectra[:, 1])):                  ## loop over all points in the spectrum
                           SubtractedModelSpectra[i, 1] = SubtractedModelSpectra[i, 1] - (abs(tBack) * (SubtractedModelSpectra[i, 0] / FreqMin)**tSlope)

                    # Debug:
                    # print ("len(SubtractedModelSpectra) = ", len(SubtractedModelSpectra))
                    # print ("SubtractedModelSpectra = ", SubtractedModelSpectra)


                    ## store subtracted spectrum
                    ListSubtractedModelSpectra.append(SubtractedModelSpectra)

    # Debug:
    # print ("\n\n\nBestMolfitFileContent = ", BestMolfitFileContent)
    # print ("IsoRatioFileContent = ", IsoRatioFileContent)
    # print ("ListSubtractedModelSpectra = ", ListSubtractedModelSpectra)


    ## define output variables
    return (BestMolfitFileContent, IsoRatioFileContent, ListSubtractedModelSpectra)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check, if molecule contributes to current spectrum or not
##
def CheckContribution(molecule, CurrentMolfitFileName, NewExpFileList, NumberExpRangesList, NewFreqMinList, NewFreqMaxList, NewGlobalvLSRList, \
                      NewRedshiftList, ListSubtractedSpectra, ListSubtractedModelSpectra, MaxOverestimationHeight, LineIDJobDir, NewtBackList, \
                      NewtSlopeList, NewNoiseList, ListOfSpectraNames, ListOfSpectra, StrongMolContribution, CheckEmissionAbsorptionFlag, \
                      DecisionMethod, Tolerance, dbFile, SQLParamArray):
    """

input parameters:
-----------------

    - molecule:                     name of current molecule

    - CurrentMolfitFileName:        path and name of current molfit file

    - NewExpFileList:               list containing all paths and names of all observational data files

    - NumberExpRangesList:          number of frequency ranges

    - NewFreqMinList:               min. freq. for each frequency range

    - NewFreqMaxList:               max. freq. for each frequency range

    - NewGlobalvLSRList:            global v_LSR for each obs. data file

    - NewRedshiftList:              list of redshifts

    - ListSubtractedSpectra:        the observed spectrum (continuum subtracted)

    - ListSubtractedModelSpectra:   the modeled spectrum (continuum subtracted)

    - MaxOverestimationHeight:      overestimation limit

    - LineIDJobDir:                 current line ID job directory

    - NewtBackList:                 list containing the background temperatures for all frequency ranges

    - NewtSlopeList:                list containing the temperature slopes for all frequency ranges

    - NewNoiseList:                 list with all noise levels for all frequency ranges

    - ListOfSpectraNames:           list containing the names of the exp. data file names for all frequency range

    - ListOfSpectra:                list of all observed spectra

    - StrongMolContribution:        contribution of strong molecule(s)

    - CheckEmissionAbsorptionFlag:  check (true/false), if a peak in modeled spectrum has the same type as corresponding peak in observational data

    - DecisionMethod:               method which is used

    - Tolerance:                    tolerance factor for method "significant"

    - dbFile:                       path and file of database file

    - SQLParamArray:                array containing sql parameters


output parameters:
------------------

    - ContributionFlag:             flag indicating if current molecule contributes or not

    - LocalResultsFile:             contents for result file

    """

    # Debug:
    # print ("molecule = ", molecule)
    # print ("CurrentMolfitFileName = ", CurrentMolfitFileName)
    # print ("NewExpFileList = ", NewExpFileList)
    # print ("NumberExpRangesList = ", NumberExpRangesList)
    # print ("NewFreqMinList = ", NewFreqMinList)
    # print ("NewFreqMaxList = ", NewFreqMaxList)
    # print ("NewGlobalvLSRList = ", NewGlobalvLSRList)
    # print ("NewRedshiftList = ", NewRedshiftList)
    # print ("ListSubtractedSpectra = ", ListSubtractedSpectra)
    # print ("ListSubtractedModelSpectra = ", ListSubtractedModelSpectra)
    # print ("MaxOverestimationHeight = ", MaxOverestimationHeight)
    # print ("LineIDJobDir = ", LineIDJobDir)
    # print ("NewtBackList = ", NewtBackList)
    # print ("NewtSlopeList = ", NewtSlopeList)
    # print ("NewNoiseList = ", NewNoiseList)
    # print ("ListOfSpectraNames = ", ListOfSpectraNames)
    # print ("ListOfSpectra = ", ListOfSpectra)
    # print ("StrongMolContribution = ", StrongMolContribution)
    # print ("CheckEmissionAbsorptionFlag = ", CheckEmissionAbsorptionFlag)
    # print ("DecisionMethod = ", DecisionMethod)
    # print ("Tolerance = ", Tolerance)
    # print ("dbFile = ", dbFile)
    # print ("SQLParamArray = ", SQLParamArray)


    ## define numerical zero
    NumericalZero = 1.e-6


    ## initialize output variable
    ContributionFlag = "true"
    LocalResultsFile = []
    if (len(ListSubtractedModelSpectra) == 0):
        ContributionFlag = "false"
        return (ContributionFlag, LocalResultsFile)


    ## read all velocity offsets from molfit file
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(CurrentMolfitFileName)
    VelocityOffsets = []
    for MoleculeIndex in range(len(MoleculesInMolfitFile)):
        LocalMolName = MoleculesInMolfitFile[MoleculeIndex]
        if (LocalMolName.strip() == molecule.strip()):
            LocalMolfitParameters = AllParameters[MoleculeIndex]


            ## create final contribution to all further molfit files of identified strong molecule
            for line in LocalMolfitParameters:
                for col in line:
                    Name = col[0]
                    element = col[4]
                    if (Name == "V_off"):
                        VelocityOffsets.append(float(element))


    ## loop over all modeled spectra
    EmptySpectrumCounter = 0
    CounterPeak = 0                                                                         ## reset counter for overall peaks
    CounterAcceptedPeaks = 0                                                                ## reset counter for accepted peaks
    CounterOverestimatedPeaks = 0                                                           ## reset counter for overestimated peaks
    CounterFreqRange = (-1)
    for ObsDataFileIndex in range(len(NewExpFileList)):                                     ## loop over all obs. data files
        NewExpFileName = NewExpFileList[ObsDataFileIndex][1]
        NewSpectraName = os.path.basename(NewExpFileName)                                   ## get name without path

        # Debug:
        # print ("\n\nNewExpFileName = ", NewExpFileName)
        # print ("NewSpectraName = ", NewSpectraName)


        ## write name of observational data file to log file
        helpstring = "-" * 150
        LocalResultsFile.append("\n\n\t" + helpstring + "\n")
        LocalNewSpectraName = NewExpFileName.split("/")
        LocalNewSpectraName = LocalNewSpectraName[-1]
        LocalResultsFile.append("\tObs. data file " + str(ObsDataFileIndex + 1) + ": " + chr(34) + LocalNewSpectraName + chr(34) + "\n")


        ## determine corresponding observed unmodified spectrum
        SpectraName = ""
        NewASCIIdata = []
        OldSpectraIndex = 0
        for LocalSpectraIndex in range(len(ListOfSpectraNames)):
            if (NewSpectraName == os.path.basename(ListOfSpectraNames[LocalSpectraIndex][1])):
                OldSpectraIndex = LocalSpectraIndex
                NewASCIIdata = ListOfSpectra[LocalSpectraIndex]
                SubtractedSpectrumLocal = ListSubtractedSpectra[LocalSpectraIndex]


                ## get name of exp. data file of current frequency range
                SpectraName = ListOfSpectraNames[LocalSpectraIndex][1]
                i = SpectraName.rfind("/")
                if (i > (-1)):
                    SpectraName = SpectraName[i + 1:]
                OrigSpectraName = os.path.basename(SpectraName)
                SpectraName = os.path.splitext(OrigSpectraName)[0]
                break

        # Debug:
        # print ("OldSpectraIndex = ", OldSpectraIndex)
        # print ("SpectraName = ", SpectraName)


        ## get parameters for current range
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, GlobalvLSRListIn = NewGlobalvLSRList, \
                                                                       NumberRangeListIn = NumberExpRangesList, Redshift_ListIn = NewRedshiftList)
        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
        GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
        if (GlobalvLSR is None):
            GlobalvLSR = 0.0
        Redshift = ObsXMLParameterDictFile['Redshift']
        if (Redshift is None):
            Redshift = 0.0


        ## subtract strong molecule contribution
        FreqArray = NewASCIIdata[:, 0]
        for RangeIndex in range(NumberFrequencyRanges):


            ## get parameters for current range
            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = NewFreqMinList, \
                                                                            FreqMaxListIn = NewFreqMaxList, tBackListIn = NewtBackList, \
                                                                            tSlopeListIn = NewtSlopeList, NoiseListIn = NewNoiseList)
            FreqMin = ObsXMLParameterDictRange['FreqMin']
            if (FreqMin is not None):
                CounterFreqRange += 1
                FreqMax = ObsXMLParameterDictRange['FreqMax']
                tBack = ObsXMLParameterDictRange['tBack']
                tSlope = ObsXMLParameterDictRange['tSlope']
                Noise = ObsXMLParameterDictRange['NoiseLevel']
                if (Noise is None):
                    Noise = 0.0

                # Debug:
                # print ("CounterFreqRange = ", CounterFreqRange)
                # print ("ListSubtractedModelSpectra = ", ListSubtractedModelSpectra)


                ## get modeled spectrum
                CurrentFreqRangeModelValues = ListSubtractedModelSpectra[CounterFreqRange]
                ModelFreqMinIndex = max(0, (numpy.abs(CurrentFreqRangeModelValues[:, 0] - FreqMin)).argmin() - 1)
                ModelFreqMaxIndex = min(len(CurrentFreqRangeModelValues[:, 0]) - 1, (numpy.abs(CurrentFreqRangeModelValues[:, 0] - FreqMax)).argmin() + 1)
                In1 = min(ModelFreqMinIndex, ModelFreqMaxIndex)
                In2 = max(ModelFreqMinIndex, ModelFreqMaxIndex)
                CurrentFreqRangeModelValues = CurrentFreqRangeModelValues[In1:In2, :]


                ## add strong molecule contribution to modeled spectra
                if (StrongMolContribution != []):
                    StrongMolModeledSpectra = StrongMolContribution[1]


                    ## get index for FreqMin and FreqMax
                    AllLocalStrongMolModeledSpectra = StrongMolModeledSpectra[OldSpectraIndex]
                    StrongFreqMinIndex = max(0, (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMin)).argmin() - 1)
                    StrongFreqMaxIndex = min(len(AllLocalStrongMolModeledSpectra[:, 0]) - 1, \
                                             (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMax)).argmin() + 1)
                    In1 = min(StrongFreqMinIndex, StrongFreqMaxIndex)
                    In2 = max(StrongFreqMinIndex, StrongFreqMaxIndex)
                    LocalStrongMolModeledSpectra = AllLocalStrongMolModeledSpectra[In1:In2, :]


                    ## add strong molecule contribution and remove continuum contribution
                    for i in range(min(len(LocalStrongMolModeledSpectra[:, 1]), len(CurrentFreqRangeModelValues[:, 1]))):
                        CurrentFreqRangeModelValues[i, 1] = CurrentFreqRangeModelValues[i, 1] + LocalStrongMolModeledSpectra[i, 1]

                # Debug:
                # print ("RangeIndex = ", RangeIndex)
                # print ("CurrentFreqRangeModelValues = ", CurrentFreqRangeModelValues)


                ## shift min. and max. frequency by v_lsr or redshift
                f1 = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
                f2 = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
                ShiftedFreqMin = min(f1, f2)
                ShiftedFreqMax = max(f1, f2)


                ## get transition frequencies in current frequency range from database
                LocalMaxNumTransInFit = 90000
                TransFreqList, DBParamList = GetTransFreq(molecule, ShiftedFreqMin, ShiftedFreqMax, SQLParamArray, dbFile, LocalMaxNumTransInFit)

                # Debug:
                # print ("\n\nFreqMin, FreqMax = ", FreqMin, FreqMax)
                # print ("TransFreqList = ", TransFreqList)
                # print ("DBParamList = ", DBParamList)


                ## define all doppler-shifted transition frequencies
                DopplerShiftedTransFreq = []
                if (TransFreqList != []):
                    for LocalTransFreq in TransFreqList:
                        TransFreq = task_myXCLASS.ConvertFreq(LocalTransFreq, GlobalvLSR, z = Redshift)
                        for VelOff in VelocityOffsets:
                            vt = task_myXCLASS.ConvertFreq(TransFreq, VelOff)
                            if vt not in DopplerShiftedTransFreq:
                                DopplerShiftedTransFreq.append(vt)
                    DopplerShiftedTransFreq.sort()


                ## write some information to log file
                helpstringLocal = "-" * 146
                LocalResultsFile.append("\n\t\t" + helpstringLocal + "\n")
                LocalResultsFile.append("\t\tFrequency range " + str(CounterFreqRange + 1) + ":\n\n")
                LocalResultsFile.append("\t\t\tMin. Frequency = " + str(FreqMin) + "\n")
                LocalResultsFile.append("\t\t\tMax. Frequency = " + str(FreqMax) + "\n")
                LocalResultsFile.append("\t\t\tNoise level = " + str(Noise) + "\n\n")


                ## check, if modeled spectrum is identical to zero
                if (numpy.nanmax(abs(CurrentFreqRangeModelValues[:, 1])) <= Noise):
                    EmptySpectrumCounter += 1
                    LocalResultsFile.append("\n\t\t\tFound no peak above the noise level!\n")

                    if (EmptySpectrumCounter == len(ListSubtractedModelSpectra)):
                        LocalResultsFile.append("\n\n\t\tModeled spectrum is zero!\n")
                        LocalResultsFile.append("\n\n\t\tMolecule " + molecule + " is not included!\n\n")
                        ContributionFlag = "false"
                        return (ContributionFlag, LocalResultsFile)
                else:
                    # Debug:
                    # print ("DopplerShiftedTransFreq = ", DopplerShiftedTransFreq)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## determine the intensities of observed and modeled spectra at doppler-shifted transition frequencies
                    number_points = len(CurrentFreqRangeModelValues[:, 1])                  ## define number of data points
                    AllPeakIndices = []
                    for vt in DopplerShiftedTransFreq:                                      ## loop over all doppler-shifted transition freq.


                        ## get index of central peak in modeled spectrum
                        LocalFreqArray = CurrentFreqRangeModelValues[:, 0]
                        ModelPeakIndex = min(len(LocalFreqArray) - 1, max(0, (numpy.abs(LocalFreqArray - vt)).argmin()))


                        ## get index of central peak in modeled spectrum
                        LocalFreqArray = SubtractedSpectrumLocal[:, 0]
                        ObsPeakIndex = min(len(LocalFreqArray) - 1, max(0, (numpy.abs(LocalFreqArray - vt)).argmin()))


                        ## if current index is not already analyzed before, continue ..
                        if ModelPeakIndex not in AllPeakIndices:
                            AllPeakIndices.append(ModelPeakIndex)

                            # Debug:
                            # print ("\n\nvt = ", vt)
                            # print ("ModelPeakIndex = ", ModelPeakIndex)
                            # print ("abs(CurrentFreqRangeModelValues[ModelPeakIndex, 1]) = ", abs(CurrentFreqRangeModelValues[ModelPeakIndex, 1]))
                            # print ("Noise = ", Noise)


                            ## print some status, why molecule is not included
                            if (abs(CurrentFreqRangeModelValues[ModelPeakIndex, 1]) >= Noise):
                                CounterPeak += 1                                            ## increase counter for all peaks
                                LocalResultsFile.append("\n\t\t\tPeak of modeled spectrum at: " + str(CurrentFreqRangeModelValues[ModelPeakIndex, 0]) \
                                                        + " MHz\n")
                                LocalResultsFile.append("\t\t\t\tHeight of modeled spectrum at this frequency (continuum subtracted): " \
                                                        + str(CurrentFreqRangeModelValues[ModelPeakIndex, 1]) + " K\n")
                                LocalResultsFile.append("\t\t\t\tHeight of observed data at this frequency (continuum subtracted): " \
                                                        + str(SubtractedSpectrumLocal[ObsPeakIndex, 1]) + " K\n")
                                LocalResultsFile.append("\t\t\t\tDifference between model and exp. data at this frequency: " \
                                                        + str(CurrentFreqRangeModelValues[ModelPeakIndex, 1] - SubtractedSpectrumLocal[ObsPeakIndex, 1]) \
                                                        + " K\n")
                                if (abs(SubtractedSpectrumLocal[ObsPeakIndex, 1]) < NumericalZero):
                                    LocalResultsFile.append("\t\t\t\tFraction of height: infinity\n")
                                else:
                                    FractionOfHeight = (CurrentFreqRangeModelValues[ModelPeakIndex, 1] / SubtractedSpectrumLocal[ObsPeakIndex, 1]) * 100.0
                                    LocalResultsFile.append("\t\t\t\tFraction of height: " + str(abs(FractionOfHeight)) + " %\n")


                                ## check, if modeled and observed spectrum describe same type of peak (both emission or both absorption)
                                if ((CurrentFreqRangeModelValues[ModelPeakIndex, 1] * SubtractedSpectrumLocal[ObsPeakIndex, 1]) < 0.0 \
                                    and CheckEmissionAbsorptionFlag):
                                    LocalResultsFile.append("\n")
                                    if (CurrentFreqRangeModelValues[ModelPeakIndex, 1] < 0.0):  ## model spectrum describes absorption at current freq.
                                        line = "\t\t\tModeled spectrum at this frequency describes absorption whereas obs. spectrum describes emission "
                                        line += "or vis versa!"
                                    else:
                                        line = "\t\t\tModeled spectrum at this frequency describes emission whereas obs. spectrum describes absorption "
                                        line += "or vis versa!"
                                    LocalResultsFile.append(line + "\n")
                                    if (DecisionMethod == "strict"):
                                        LocalResultsFile.append("\n\t\t\t\t\tMolecule " + molecule + " is not included!\n")
                                        ContributionFlag = "false"
                                        break
                                    elif (DecisionMethod == "significant"):
                                        CounterOverestimatedPeaks += 1


                                ##------------------------------------------------------------------------------------------------------------------------
                                ## check, if max. peak height of modeled peak is within user defined range above observed peak height
                                elif (abs((CurrentFreqRangeModelValues[ModelPeakIndex, 1] / SubtractedSpectrumLocal[ObsPeakIndex, 1])) \
                                      > (1.0 + MaxOverestimationHeight / 100.0)):
                                    LocalResultsFile.append("\n")
                                    LocalResultsFile.append("\t\t\tFraction of heights is   O U T   O F   allowed range of " \
                                                            + str(100.0 + MaxOverestimationHeight) + " %\n")
                                    if (DecisionMethod == "strict"):
                                        LocalResultsFile.append("\n\t\t\t\t\tMolecule " + molecule + " is not included!\n")
                                        ContributionFlag = "false"
                                        break
                                    elif (DecisionMethod == "significant"):
                                        CounterOverestimatedPeaks += 1


                                ##------------------------------------------------------------------------------------------------------------------------
                                ## peak is within user defined range
                                else:
                                    CounterAcceptedPeaks += 1
                                    LocalResultsFile.append("\n\t\t\t\tFraction of heights is   W I T H I N   allowed range of " \
                                                            + str(100.0 + MaxOverestimationHeight) + " %\n")
                                LocalResultsFile.append("\n")


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## abort loop if mismatch is observed
                if (ContributionFlag == "false"):                                           ## abort if an absolute mismatch is observed
                    break


    ## if no absolute abort criterion is specified ..
    if (ContributionFlag != "false" and CounterPeak > 0):
        if (DecisionMethod == "significant"):                                               ## the number of accepted peaks has to be significant
                                                                                            ## greater than the number of overestimated peaks
            helpstring2 = "-" * 150
            LocalResultsFile.append("\n\t" + helpstring2 + "\n")
            LocalResultsFile.append("\tSummary:\n\n")
            fraction = (float(CounterOverestimatedPeaks) / float(CounterPeak)) * 100.0      ## define fraction of overestimated peaks
            line = "\t\tFraction (number_overestimated_peaks / total_number_all_peaks) =  (" + str(CounterOverestimatedPeaks) + "/" + str(CounterPeak)
            line += ") = " + str(fraction)
            if (fraction > float(Tolerance)):                                               ## is fraction greater than Tolerance % abort!
                line += "% > " + str(Tolerance) + "% !\n"
                LocalResultsFile.append(line)
                LocalResultsFile.append("\n\t\tTo many overestimated peaks!\n")
                LocalResultsFile.append("\n\n\tMolecule " + molecule + " is not included!\n")
                ContributionFlag = "false"
            else:
                line += "% <= " + str(Tolerance) + "% !\n"
                LocalResultsFile.append(line)
                LocalResultsFile.append("\n\n\tMolecule " + molecule + " is identified!\n")


    ## check, if at least one peak was found
    if (CounterPeak == 0):
        LocalResultsFile.append("\n\n\t" + helpstring + "\n")
        LocalResultsFile.append("\tFound no peak in all modeled spectra!\n")
        LocalResultsFile.append("\n\n\tMolecule " + molecule + " is not included!\n")
        ContributionFlag = "false"
    LocalResultsFile.append("\n\n")


    ## define return variables
    return (ContributionFlag, LocalResultsFile)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start a single molecule fit
##
def SingleMoleculeFit(ClusterFlag, lock, Node, molecule, StrongMolContribution, PathXCLASSInterface, LineIDJobDir, MoleculejobDirList, \
                      LocalMoleculeIndex, ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, \
                      IdentifiedMoleculeFile, ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, \
                      ListMolDirName, ObsXMLFileNameOrig, DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, \
                      SQLParamArray, DecisionMethod, Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit, InternalParameterList):
    """

input parameters:
-----------------

    - ClusterFlag:                  flag indicating if a fit is done on a cluster or not

    - lock:                         lock object used for threading package

    - Node:                         current node

    - molecule:                     current molecule

    - StrongMolContribution:        contribution of strong (highly abundant) molecules

    - PathXCLASSInterface:          path of XCLASS interface

    - LineIDJobDir:                 job directory for current LineID run

    - MoleculejobDirList:           current single molecule fit directory

    - LocalMoleculeIndex:           local molecule index

    - ListOfSpectraNames:           list of all spectrum (observational data) files

    - ListOfSpectra:                list of all observed spectra

    - ListSubtractedSpectra:        list of all continuum subtracted observational spectra

    - MaxOverestimationHeight:      overestimation factor

    - ResultsFile:                  final result file

    - IdentifiedMoleculeFile:       file for intermediate identified molecules

    - ListNoiseLevels:              list of all noise levels for all frequency ranges of all spectra

    - CheckEmissionAbsorptionFlag:  flag to consider emission and absorption separately

    - IdentifiedMoleculesDir:       path of the directory containing the intermediate identified molecules of the current LineID run

    - NotIdentifiedMoleculesDir:    path of the directory containing the non-identified molecules of the current LineID run

    - ListMolDirName:               list of modified molecule names (used for naming of directories)

    - ObsXMLFileNameOrig:           name of MAGIX exp. xml file

    - DefaultMolfitUsedFlagList:    defines, if a default molfit file was used or not (already takes EstimateParamFlag into account)

    - AlgorithmXMLFileEstParam:     path and name of algorithm xml file (used for parameter estimation algorithm)

    - AlgorithmXMLFileISO:          path and name of algorithm xml file (used for fit of iso ratios)

    - VelBin:                       velocity resolution for histogram analysis

    - SQLParamArray:                array containing sql parameters

    - DecisionMethod:               method which is used

    - Tolerance:                    tolerance factor for method "significant"

    - TypeSelection:                exclude type of lines (only emission or only absorption or all types)

    - SmoothValue:                  smooth factor (=0, no smoothing is performed)

    - MaxNumTransInFit:             max. number of transitions which are fitted in parameter estimation algorithm

    - InternalParameterList:        list of internal parameters


output parameters:
------------------

    - ContributionFlag:             flag indicting if current molecule contributes or not

    - MoleculeResultName:           path and name of corresponding molfit file

    """

    # Debug:
    # print ("ClusterFlag = ", ClusterFlag)
    # print ("lock = ", lock)
    # print ("Node = ", Node)
    # print ("molecule = ", molecule)
    # print ("StrongMolContribution = ", StrongMolContribution)
    # print ("PathXCLASSInterface = ", PathXCLASSInterface)
    # print ("LineIDJobDir = ", LineIDJobDir)
    # print ("MoleculejobDirList = ", MoleculejobDirList)
    # print ("LocalMoleculeIndex = ", LocalMoleculeIndex)
    # print ("ListOfSpectraNames = ", ListOfSpectraNames)
    # print ("ListOfSpectra = ", ListOfSpectra)
    # print ("ListSubtractedSpectra = ", ListSubtractedSpectra)
    # print ("MaxOverestimationHeight = ", MaxOverestimationHeight)
    # print ("ResultsFile = ", ResultsFile)
    # print ("IdentifiedMoleculeFile = ", IdentifiedMoleculeFile)
    # print ("ListNoiseLevels = ", ListNoiseLevels)
    # print ("CheckEmissionAbsorptionFlag = ", CheckEmissionAbsorptionFlag)
    # print ("IdentifiedMoleculesDir = ", IdentifiedMoleculesDir)
    # print ("NotIdentifiedMoleculesDir = ", NotIdentifiedMoleculesDir)
    # print ("ListMolDirName = ", ListMolDirName)
    # print ("ObsXMLFileNameOrig = ", ObsXMLFileNameOrig)
    # print ("DefaultMolfitUsedFlagList = ", DefaultMolfitUsedFlagList)
    # print ("AlgorithmXMLFileEstParam = ", AlgorithmXMLFileEstParam)
    # print ("AlgorithmXMLFileISO = ", AlgorithmXMLFileISO)
    # print ("VelBin = ", VelBin)
    # print ("SQLParamArray = ", SQLParamArray)
    # print ("DecisionMethod = ", DecisionMethod)
    # print ("Tolerance = ", Tolerance)
    # print ("TypeSelection = ", TypeSelection)
    # print ("SmoothValue = ", SmoothValue)
    # print ("MaxNumTransInFit = ", MaxNumTransInFit)


    ## initialize output parameters
    ContributionFlag = "false"
    MoleculeResultName = ""


    ## for testing
    DoFinalSMFitFlag = True


    ## define name of current molecule where all not allowed characters were removed from the molecule name
    MolDirName = MoleculeFileName(molecule)


    ## deactivate iso ratio file for single molecule file
    CurrentExpXMLFile = LineIDJobDir + "/" + MoleculejobDirList[LocalMoleculeIndex] + "exp.xml"
    OrigIsotopologuesList = task_MAGIX.GetXMLtagNEW(CurrentExpXMLFile, "iso_flag")
    if (OrigIsotopologuesList == []):
        OrigIsotopologuesList = task_MAGIX.GetXMLtagNEW(CurrentExpXMLFile, "Isotopologues")
    OrigIsoTableFileNameList = task_MAGIX.GetXMLtagNEW(CurrentExpXMLFile, "IsoTableFileName")
    task_MAGIX.WriteXMLtagNEW(CurrentExpXMLFile, "iso_flag", ["n"])
    # task_MAGIX.WriteXMLtagNEW(CurrentExpXMLFile, "Isotopologues", ["n"])

    # Debug:
    # print ("\n\n\n\n")
    # print ("OrigIsotopologuesList = ", OrigIsotopologuesList)
    # print ("OrigIsoTableFileNameList = ", OrigIsoTableFileNameList)
    # print ("molecule = ", molecule)
    # print ("NewExpXMLFileName = ", NewExpXMLFileName)
    #    sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## use parameter estimation algorithm to create molfit file
    MoleculejobDir = LineIDJobDir + "/" + MoleculejobDirList[LocalMoleculeIndex] + "/"
    MoleculejobDir = os.path.normpath(MoleculejobDir) + "/"
    CurrentMolfitFileName = os.path.normpath(MoleculejobDir + MolDirName + ".molfit")
    MAGIXrootDir = os.path.normpath(PathXCLASSInterface) + "/"
    ParamEstOKFlag = "fine"                                                                 ## initialize parameter estimation flag
    velLowerLimit = -100.0
    velUpperLimit = 100.0
    TWFSpectraList = {}


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if everything is fine, continue here
    if (ParamEstOKFlag == "fine"):


        ## read in some parameters from observational xml file
        UnmodifiedExpXML = CurrentExpXMLFile
        ExpFileList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "FileNamesExpFiles")
        NumberExpRangesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumberExpRanges")
        FreqMinList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "MinExpRange")
        FreqMaxList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "MaxExpRange")
        FreqStepList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "StepFrequency")
        t_back_flagList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "t_back_flag")
        tBackList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BackgroundTemperature")
        tSlopeList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "TemperatureSlope")
        N_HList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "HydrogenColumnDensity")
        beta_dustList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "DustBeta")
        kappa_1300List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Kappa")
        DustFileNameList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "DustFileName")
        BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BackgroundFileName")
        ContPhenFuncIDList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncID")
        ContPhenFuncParam1List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam1")
        ContPhenFuncParam2List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam2")
        ContPhenFuncParam3List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam3")
        ContPhenFuncParam4List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam4")
        ContPhenFuncParam5List = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ContPhenFuncParam5")
        NoiseList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NoiseLevel")
        SmoothList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "SmoothValue")
        TelescopeSizeList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "TelescopeSize")
        BMINList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BMIN")
        BMAJList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BMAJ")
        BPAList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "BPA")
        Inter_FlagList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Inter_Flag")
        GlobalvLSRList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "GlobalvLSR")
        RedshiftList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Redshift")
        ErrorYList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "ErrorY")
        NumberHeaderLinesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumberHeaderLines")
        SeparatorColumnsList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "SeparatorColumns")
        IsotopologuesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "iso_flag")
        if (IsotopologuesList == []):
            IsotopologuesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "Isotopologues")
        IsoTableFileNameList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "IsoTableFileName")
        dbList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "dbFilename")
        if (dbList == []):
            dbFile = MAGIXrootDir + "../../"
            dbFile = os.path.normpath(dbFile) + "/Database/cdms_sqlite.db"
            dbList = [dbFile]
        dbFile = dbList[0]
        NumModelPixelXXList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumModelPixelXX")
        NumModelPixelYYList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NumModelPixelYY")
        LocalOverlapFlagList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "LocalOverlap_Flag")
        NoSubBeamFlagList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "NoSubBeam_Flag")
        EmAbsPATHList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXML, "EmAbsPATH")

        # Debug:
        # print ("\n\nExpFileList = ", ExpFileList)
        # print ("len(ExpFileList) = ", len(ExpFileList))
        # print ("NumberExpRangesList = ", NumberExpRangesList)
        # print ("FreqMinList = ", FreqMinList)
        # print ("FreqMaxList = ", FreqMaxList)
        # print ("FreqStepList = ", FreqStepList)
        # print ("t_back_flagList = ", t_back_flagList)
        # print ("tBackList = ", tBackList)
        # print ("tSlopeList = ", tSlopeList)
        # print ("N_HList = ", N_HList)
        # print ("beta_dustList = ", beta_dustList)
        # print ("kappa_1300List = ", kappa_1300List)
        # print ("DustFileNameList = ", DustFileNameList)
        # print ("BackgroundFileNameList = ", BackgroundFileNameList)
        # print ("ContPhenFuncIDList = ", ContPhenFuncIDList)
        # print ("ContPhenFuncParam1List = ", ContPhenFuncParam1List)
        # print ("ContPhenFuncParam2List = ", ContPhenFuncParam2List)
        # print ("ContPhenFuncParam3List = ", ContPhenFuncParam3List)
        # print ("ContPhenFuncParam4List = ", ContPhenFuncParam4List)
        # print ("ContPhenFuncParam5List = ", ContPhenFuncParam5List)
        # print ("NoiseList = ", NoiseList)
        # print ("SmoothList = ", SmoothList)
        # print ("TelescopeSizeList = ", TelescopeSizeList)
        # print ("BMINList = ", BMINList)
        # print ("BMAJList = ", BMAJList)
        # print ("BPAList = ", BPAList)
        # print ("Inter_FlagList = ", Inter_FlagList)
        # print ("GlobalvLSRList = ", GlobalvLSRList)
        # print ("RedshiftList = ", RedshiftList)
        # print ("ErrorYList = ", ErrorYList)
        # print ("NumberHeaderLinesList = ", NumberHeaderLinesList)
        # print ("SeparatorColumnsList = ", SeparatorColumnsList)
        # print ("IsotopologuesList = ", IsotopologuesList)
        # print ("IsoTableFileNameList = ", IsoTableFileNameList)
        # print ("dbList = ", dbList)
        # print ("NumModelPixelXXList = ", NumModelPixelXXList)
        # print ("NumModelPixelYYList = ", NumModelPixelYYList)
        # print ("LocalOverlapFlagList = ", LocalOverlapFlagList)
        # print ("NoSubBeamFlagList = ", NoSubBeamFlagList)
        # print ("EmAbsPATHList = ", EmAbsPATHList)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## update list of background files
        NumExpDataFiles = len(ExpFileList)
        NewBackgroundFileNameList = DefineListBgFileNames(InternalParameterList, MolDirName, NumExpDataFiles, \
                                                          NumberExpRangesList, FreqMinList, \
                                                          FreqMaxList, BackgroundFileNameList)

        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## shrink frequency ranges


        ## create xml file for each molecule
        NewExpXMLFileName = CurrentExpXMLFile
        velLowLimit = -100.0
        velUpLimit = 100.0
        MaxWidth = 100.0
        LocalMaxNumTransInFit = 1.e99
        CreateXMLFile(molecule, velLowLimit, velUpLimit, MaxWidth, NewExpXMLFileName, ExpFileList, NumberExpRangesList, FreqMinList, FreqMaxList, \
                      FreqStepList, t_back_flagList, tBackList, tSlopeList, N_HList, beta_dustList, kappa_1300List, DustFileNameList, \
                      NewBackgroundFileNameList, ContPhenFuncIDList, \
                      ContPhenFuncParam1List, ContPhenFuncParam2List, ContPhenFuncParam3List, ContPhenFuncParam4List, ContPhenFuncParam5List, NoiseList, \
                      SmoothList, TelescopeSizeList, BMINList, BMAJList, BPAList, Inter_FlagList, GlobalvLSRList, RedshiftList, ErrorYList, \
                      NumberHeaderLinesList, SeparatorColumnsList, IsotopologuesList, IsoTableFileNameList, dbList, \
                      NumModelPixelXXList, NumModelPixelYYList, LocalOverlapFlagList, SQLParamArray, LocalMaxNumTransInFit, NoSubBeamFlagList, \
                      EmAbsPATHList)


        ## get path and names of observational data files from shrinked xml file
        NewExpFileList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "FileNamesExpFiles")
        NumberExpRangesList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "NumberExpRanges")
        NewFreqMinList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "MinExpRange")
        NewFreqMaxList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "MaxExpRange")
        NewGlobalvLSRList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "GlobalvLSR")
        NewRedshiftList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "Redshift")
        NewtBackList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "BackgroundTemperature")
        NewtSlopeList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "TemperatureSlope")
        NewNoiseList = task_MAGIX.GetXMLtagNEW(NewExpXMLFileName, "NoiseLevel")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## subtract strong molecule contribution from observational data
        if (StrongMolContribution != []):
            StrongMolModeledSpectra = StrongMolContribution[1]                              ## get modeled spectra of strong molecules


            ## modify ListSubtractedSpectra or add strong molecule spectra to modeled spectra
            NewListNumberHeaderLines = []
            NewListSeparatorColumns = []
            ListSubtractedStrongMolModelSpectra = []
            for ObsDataFileIndex in range(len(NewExpFileList)):
                NewExpFileName = NewExpFileList[ObsDataFileIndex][1]
                NewSpectraName = os.path.basename(NewExpFileName)
                NewListNumberHeaderLines.append([ObsDataFileIndex, "0"])
                NewListSeparatorColumns.append([ObsDataFileIndex, " "])


                ## determine corresponding observed spectrum
                OldSpectraIndex = 0
                for LocalSpectraIndex in range(len(ListOfSpectraNames)):
                    if (NewSpectraName == os.path.basename(ListOfSpectraNames[LocalSpectraIndex][1])):
                        NewASCIIdata = ListOfSpectra[LocalSpectraIndex]
                        OldSpectraIndex = LocalSpectraIndex
                        break


                ## get parameters for current range
                RangeIndex = (-1)
                ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberRangeListIn = NumberExpRangesList)
                NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']     ## get number of frequency ranges for current obs. data file


                ## subtract strong molecule contribution
                FreqArray = NewASCIIdata[:, 0]
                IntArray = NewASCIIdata[:, 1]
                for RangeIndex in range(NumberFrequencyRanges):


                    ## get parameters for current range
                    ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = NewFreqMinList, \
                                                                                    FreqMaxListIn = NewFreqMaxList, tBackListIn = NewtBackList, \
                                                                                    tSlopeListIn = NewtSlopeList)
                    FreqMin = ObsXMLParameterDictRange['FreqMin']
                    if (FreqMin is not None):
                        FreqMax = ObsXMLParameterDictRange['FreqMax']
                        tBack = ObsXMLParameterDictRange['tBack']
                        tSlope = ObsXMLParameterDictRange['tSlope']

                        ## global v_LSR not need here
                        #    try:
                        #        GlobalvLSR = float(NewGlobalvLSRList[RangeIndex])
                        #    except IndexError:
                        #        GlobalvLSR = 0.0


                        ## get index for FreqMin and FreqMax of observational data spectrum
                        FreqMinIndex = max(0, (numpy.abs(FreqArray - FreqMin)).argmin() - 1)
                        FreqMaxIndex = min(len(FreqArray) - 1, (numpy.abs(FreqArray - FreqMax)).argmin() + 1)


                        ## get index for FreqMin and FreqMax of modeled (strong molecule) spectrum
                        AllLocalStrongMolModeledSpectra = StrongMolModeledSpectra[OldSpectraIndex]
                        StrongFreqMinIndex = max(0, (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMin)).argmin() - 1)
                        StrongFreqMaxIndex = min(len(AllLocalStrongMolModeledSpectra[:, 0]) - 1, \
                                                 (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMax)).argmin() + 1)
                        In1 = min(StrongFreqMinIndex, StrongFreqMaxIndex)
                        In2 = max(StrongFreqMinIndex, StrongFreqMaxIndex)
                        LocalStrongMolModeledSpectra = AllLocalStrongMolModeledSpectra[In1:In2, :]


                        ## subtract strong molecule contribution from observational data
                        for kkk in range(min(len(IntArray), len(LocalStrongMolModeledSpectra[:, 0]))):
                            NewASCIIdata[FreqMinIndex + kkk, 1] = IntArray[FreqMinIndex + kkk] - LocalStrongMolModeledSpectra[kkk, 1]


                ## save modified observational data array to file
                numpy.savetxt(NewExpFileName, NewASCIIdata, delimiter='\t')


                ## update number of header lines and separator characters
                task_MAGIX.WriteXMLtagNEW(NewExpXMLFileName, "NumberHeaderLines", NewListNumberHeaderLines)
                task_MAGIX.WriteXMLtagNEW(NewExpXMLFileName, "SeparatorColumns", NewListSeparatorColumns)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print what you do
        print ("\rStart single molecule fit for " + chr(34) + molecule + chr(34) + " on node: " + Node + "                ")

        # Debug:
        # print ("\n\n\nClusterFlag = ", ClusterFlag)
        # print ("Node = ", Node)
        # print ("PathXCLASSInterface = ", PathXCLASSInterface)
        # print ("MoleculejobDir = ", MoleculejobDir)


        ## get NoMAGIX flag from init file
        OutDict = task_myXCLASS.GetXCLASSSettings()
        NoMAGIXFlag = OutDict['NoMAGIXFlag']
        if (NoMAGIXFlag):
            Optimizer = "scipy"
        else:
            Optimizer = "magix"
        Optimizer = "scipy"

        # Debug:
        # print ("NoMAGIXFlag = ", NoMAGIXFlag)
        # print ("Optimizer = ", Optimizer)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## start fit


        ## no parallel computing is possible
        if (ClusterFlag == "false"):


            ## use magix
            if (Optimizer == "magix"):
                cmdString = "cd " + PathXCLASSInterface + "; python3 magix_start.py --quiet --model=myxclass "
                cmdString += MoleculejobDir + "io_control.xml "
                # cmdString += MoleculejobDir + "io_control.xml >> " + MoleculejobDir + "screen-output__magix.out"
                os.system(cmdString)


            ## use scipy
            else:
                ChannelIntegrationFlag = True
                NameOfFunction = "LineIdentification"
                JobDir = task_myXCLASSMapFit.myXCLASSMapFitCore(MolfitsFileName = CurrentMolfitFileName, \
                                                                ObsXMLFileName = NewExpXMLFileName, \
                                                                AlgorithmXMLFile = AlgorithmXMLFileEstParam, \
                                                                ChannelIntegrationFlag = ChannelIntegrationFlag, \
                                                                regionFileName = "", \
                                                                clusterdef = "", \
                                                                FastFitFlag = True, \
                                                                FullFitFlag = True, \
                                                                JobDir = MoleculejobDir, \
                                                                NameOfFunction = NameOfFunction)
                NewExpXMLFileName = JobDir + "obs.xml"


                ## move results of MapFit run to current directory
                # cmdString = "mv {:s} {:s}".format(JobDir, MoleculejobDir)
                # os.system(cmdString)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## (MAGIX only:) start MAGIX run on node via ssh
        ## ssh charger 'cd ~/magma; nohup nice /usr/bin/magma magma.script > /var/tmp/magma.out' &
        else:
            cmdString = "ssh " + Node + " " + chr(34) + "cd " + PathXCLASSInterface + "; "
            cmdString += "nohup python3 magix_start.py --plotsaveonly --model=myxclass "
            cmdString += MoleculejobDir + "io_control.xml >> " + MoleculejobDir + "screen-output__magix.out"
            cmdString += chr(34)
            os.system(cmdString)

        # Debug:
        # print ("\n\n\n\ncmdString = ", cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read in MAGIX result
        NameOfFunction = "LineIdentification"
        if (Optimizer == "scipy"):
            ConvertLogLinFlag = "false"
        else:
            ConvertLogLinFlag = "true"
        printInfoFlag = "false"
        LocalJobDir = LineIDJobDir + MoleculejobDirList[LocalMoleculeIndex]
        MolfitFile, IsoRatioFileContent, ListSubtractedModelSpectra = GetBestResult(LocalJobDir, ConvertLogLinFlag, \
                                                                                    NewExpFileList, NumberExpRangesList, \
                                                                                    NewFreqMinList, NewFreqMaxList, \
                                                                                    NewtBackList, NewtSlopeList, \
                                                                                    SQLParameters = SQLParamArray)
        # Debug:
        # print ("MolfitFile = ", MolfitFile)
        # print ("IsoRatioFileContent = ", IsoRatioFileContent)
        # print ("ListSubtractedModelSpectra = ", ListSubtractedModelSpectra)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if molecule contributes to current spectrum or not
        ContributionFlag, LocalResultsFile = CheckContribution(molecule, MolfitFile, NewExpFileList, NumberExpRangesList, NewFreqMinList, \
                                                               NewFreqMaxList, NewGlobalvLSRList, NewRedshiftList, ListSubtractedSpectra, \
                                                               ListSubtractedModelSpectra, MaxOverestimationHeight, LineIDJobDir, NewtBackList, \
                                                               NewtSlopeList, NewNoiseList, ListOfSpectraNames, ListOfSpectra, StrongMolContribution, \
                                                               CheckEmissionAbsorptionFlag, DecisionMethod, Tolerance, dbFile, SQLParamArray)
        ## read in intermediate result file
        helpString = "=" * 154
        ResultsFile.write(helpString + "\n")
        ResultsFile.write("Results of single-molecule fit for molecule: " + chr(34) + molecule + chr(34) + ":\n")
        for line in LocalResultsFile:
            ResultsFile.write(line)
            ResultsFile.flush()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## if molecule contributes, save molfit file, add molecule to the list of identified molecules and fit iso ratios (if required)
        if (ContributionFlag == "true"):

            # Debug:
            # print ("\n\n\nMolfitFile = ", MolfitFile)


            ## write best molfit file to subdirectory IdentifiedMoleculesDir and create corresponding plot
            MoleculeResultName = IdentifiedMoleculesDir + ListMolDirName[LocalMoleculeIndex]
            BestMolfitFile = open(MoleculeResultName + ".out.molfit", 'w')
            for line in MolfitFile:
                BestMolfitFile.write(line)
            BestMolfitFile.close()


            ## write message to output files
            helpString = "-" * 154
            IdentifiedMoleculeFile.write(helpString + "\n")
            IdentifiedMoleculeFile.write("Molecule " + chr(34) + molecule + chr(34) + " was identified!\n")
            IdentifiedMoleculeFile.flush()

            # Debug:
            # print ("\n\n\n\n\n")
            # print ("LocalMoleculeIndex = ", LocalMoleculeIndex)
            # print ("DefaultMolfitUsedFlagList[LocalMoleculeIndex] = ", DefaultMolfitUsedFlagList[LocalMoleculeIndex])
            # print ("OrigIsotopologuesList = ", OrigIsotopologuesList)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## fit iso ratios


            ## check, if an iso ratio file was defined
            if (len(OrigIsotopologuesList) > 0):
                IsoFlag = OrigIsotopologuesList[0].lower()
                IsoFlag = task_myXCLASSMapFit.CheckBool(IsoFlag)
                if (IsoFlag):
                    IsoFileName = OrigIsoTableFileNameList[0]


                    ## read in iso ratio file
                    NewLocalIsoRatioFileName = ""
                    IsoRatioTable, Isotopologues, IsoMolecule = task_myXCLASS.ImportIsoRatioFile(IsoFileName, NewLocalIsoRatioFileName)


                    ## search for isotopologues for the current molecule
                    NewIsoRatioFile = []
                    FitRatiosFlag = False
                    for IsoIndex in range(len(IsoRatioTable)):
                        IsoRatioTableLine = IsoRatioTable[IsoIndex]
                        IsoMaster = IsoRatioTableLine[1].strip()
                        if (IsoMaster == molecule):
                            LocalIsotopologue = IsoRatioTableLine[0].strip()


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## get all transition frequencies


                            ## determine doppler shifted frequency ranges
                            TransitionOccurFlag = False
                            for ObsDataFileIndex in range(len(NewExpFileList)):


                                ## get parameters for current range
                                RangeIndex = (-1)
                                ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                               GlobalvLSRListIn = NewGlobalvLSRList, \
                                                                                               Redshift_ListIn = RedshiftList, \
                                                                                               NumberRangeListIn = NumberExpRangesList)
                                NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
                                GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
                                if (GlobalvLSR is None):
                                    GlobalvLSR = 0.0
                                Redshift = ObsXMLParameterDictFile['Redshift']
                                if (Redshift is None):
                                    Redshift = 0.0


                                ## check, if isotopologues has at least one transition within the given frequency ranges
                                for RangeIndex in range(NumberFrequencyRanges):


                                    ## get parameters for current range
                                    ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                                                    FreqMinListIn = NewFreqMinList, \
                                                                                                    FreqMaxListIn = NewFreqMaxList, \
                                                                                                    tBackListIn = NewtBackList, \
                                                                                                    tSlopeListIn = NewtSlopeList)
                                    FreqMin = ObsXMLParameterDictRange['FreqMin']
                                    if (FreqMin is not None):
                                        FreqMax = ObsXMLParameterDictRange['FreqMax']
                                        tBack = ObsXMLParameterDictRange['tBack']
                                        tSlope = ObsXMLParameterDictRange['tSlope']


                                        ## shift min. and max. frequency of each range by v_LSR
                                        NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
                                        NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
                                        FreqMin = min(NewFreqMin, NewFreqMax)
                                        FreqMax = max(NewFreqMin, NewFreqMax)


                                        ## get transition frequencies
                                        LocalMaxNumTransInFit = 0
                                        TransFreqList, DBParamList = GetTransFreq(LocalIsotopologue, FreqMin, FreqMax, SQLParamArray, \
                                                                                  dbFile, LocalMaxNumTransInFit)
                                        if (TransFreqList != []):
                                            TransitionOccurFlag = True
                                            break


                            ## append setting for current isotopologue to new iso ratio file
                            if (TransitionOccurFlag):
                                RatioValue = float(IsoRatioTableLine[2])
                                if (len(IsoRatioTableLine) == 3):
                                    LowerLimitIsoRatio = 1.0
                                    UpperLimitIsoRatio = 1.0
                                    NewIsoRatioFileLine = "{:<40s}     {:<40s}     {:.15e}\n".format(LocalIsotopologue, IsoMaster, RatioValue)
                                elif (len(IsoRatioTableLine) == 4):
                                    LowerLimitIsoRatio = 1.0
                                    UpperLimitIsoRatio = 1.0
                                    NewIsoRatioFileLine = "{:<40s}     {:<40s}     {:.15e} \n".format(LocalIsotopologue, IsoMaster, RatioValue)
                                elif (len(IsoRatioTableLine) >= 5):
                                    LowerLimitIsoRatio = float(IsoRatioTableLine[3])
                                    UpperLimitIsoRatio = float(IsoRatioTableLine[4])
                                    NewIsoRatioFileLine = "{:<40s}     {:<40s}     {:.15e}     {:.15e}     {:.15e}\n".format(LocalIsotopologue, \
                                                                                                                             IsoMaster, RatioValue, \
                                                                                                                             LowerLimitIsoRatio, \
                                                                                                                             UpperLimitIsoRatio)
                                NewIsoRatioFile.append(NewIsoRatioFileLine)
                                if (LowerLimitIsoRatio != UpperLimitIsoRatio):
                                    FitRatiosFlag = True

                    # Debug:
                    # print ("\n\n\n")
                    # print ("FitRatiosFlag = ", FitRatiosFlag)
                    # print ("NewIsoRatioFile = ", NewIsoRatioFile)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## check, if iso ratios have to be fitted
                    if (FitRatiosFlag and AlgorithmXMLFileISO != ""):


                        ## create subdirectory for fitting
                        IsoRatioFitDir = LocalJobDir + "Iso-Ratio-Fit/"
                        if (not os.path.exists(IsoRatioFitDir)):
                            os.system("mkdir -p " + IsoRatioFitDir)


                        ## write molfit file to fit directory and fix all molfit parameters
                        IsoMolfitFileName = IsoRatioFitDir + "smf.molfit"
                        IsoMolfitFile = open(IsoMolfitFileName, 'w')
                        for line in MolfitFile:
                            StrippedLine = line.strip()
                            if (not StrippedLine.startswith("%") and StrippedLine != ""):
                                SplittedLine = StrippedLine.split()
                                if (len(SplittedLine) > 2):
                                    line = line.lower()
                                    line = line.replace("y", "n")
                            IsoMolfitFile.write(line)
                        IsoMolfitFile.close()


                        ## write iso ratio file containing only the isotopologues of the current molecule
                        IsoRatioFileName = IsoRatioFitDir + "iso-ratio.dat"
                        IsoRatioFile = open(IsoRatioFileName, 'w')
                        for line in NewIsoRatioFile:
                            IsoRatioFile.write(line)
                        IsoRatioFile.close()


                        ## copy algorithm xml file to local fit directory
                        task_myXCLASS.NormNCopyMove(AlgorithmXMLFileISO, IsoRatioFitDir + "algorithm_control.xml", copyFlag = True)


                        ## update paths of obs. xml file
                        IsoExpXML = IsoRatioFitDir + "exp.xml"
                        dbDefaultFilename = MAGIXrootDir + "../../"
                        dbDefaultFilename = os.path.normpath(dbDefaultFilename) + "/Database/cdms_sqlite.db"
                        AdjustExpXMLFile(NewExpXMLFileName, IsoRatioFitDir, LocalJobDir, dbDefaultFilename)
                        dbList = task_MAGIX.GetXMLtagNEW(IsoExpXML, "dbFilename")
                        if (dbList == []):
                            dbList = [dbDefaultFilename]
                        task_MAGIX.WriteXMLtagNEW(IsoExpXML, "IsoTableFileName", [IsoRatioFileName])
                        task_MAGIX.WriteXMLtagNEW(IsoExpXML, "iso_flag", ["True"])


                        ## prepare other xml files for application of myXCLASSFit function
                        # task_myXCLASSFit.CreateIOControlXMLFile(IsoRatioFitDir, IsoExpXML, IsoMolfitFileName, MAGIXrootDir)
                        CreateIOControlXMLFile(IsoRatioFitDir, IsoExpXML, IsoMolfitFileName, MAGIXrootDir)


                        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        ## start MAGIX run locally
                        if (ClusterFlag == "false"):


                            ## use magix
                            if (Optimizer == "magix"):
                                cmdString = "cd " + MAGIXrootDir + "; python3 magix_start.py --quiet --model=myxclass "
                                cmdString += IsoRatioFitDir + "io_control.xml >> " + IsoRatioFitDir + "screen-output__magix.out"
                                # cmdString += IsoRatioFitDir + "io_control.xml"
                                os.system(cmdString)


                            ## use scipy
                            else:
                                ChannelIntegrationFlag = True
                                NameOfFunction = "LineIdentification"
                                LocalAlgorithmXMLFileName = IsoRatioFitDir + "algorithm_control.xml"
                                JobDir = task_myXCLASSMapFit.myXCLASSMapFitCore(MolfitsFileName = IsoMolfitFileName, \
                                                                                ObsXMLFileName = IsoExpXML, \
                                                                                AlgorithmXMLFile = LocalAlgorithmXMLFileName, \
                                                                                ChannelIntegrationFlag = ChannelIntegrationFlag, \
                                                                                regionFileName = "", \
                                                                                clusterdef = "", \
                                                                                FastFitFlag = True, \
                                                                                FullFitFlag = True, \
                                                                                JobDir = IsoRatioFitDir, \
                                                                                NameOfFunction = NameOfFunction)
                                NewExpXMLFileName = JobDir + "obs.xml"


                                ## move results of MapFit run to current directory
                                # cmdString = "mv {:s} {:s}".format(JobDir, IsoRatioFitDir)
                                # os.system(cmdString)


                        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        ## start MAGIX on node
                        else:
                            cmdString = "ssh " + Node + " " + chr(34) + "cd " + MAGIXrootDir + "; "
                            cmdString += "nohup python3 magix_start.py --noplot --model=myxclass "
                            cmdString += IsoRatioFitDir + "io_control.xml >> " + IsoRatioFitDir + "screen-output__magix.out"
                            # cmdString += IsoRatioFitDir + "io_control.xml"
                            cmdString += chr(34)
                            os.system(cmdString)


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## read in molfit file which correspond to the best fit
                        if (Optimizer == "scipy"):
                            ConvertLogLinFlag = "false"
                        else:
                            ConvertLogLinFlag = "true"
                        NewExpFileList = task_MAGIX.GetXMLtagNEW(IsoExpXML, "FileNamesExpFiles")
                        ListNumberExpRanges = task_MAGIX.GetXMLtagNEW(IsoExpXML, "NumberExpRanges")
                        NewFreqMinList = task_MAGIX.GetXMLtagNEW(IsoExpXML, "MinExpRange")
                        NewFreqMaxList = task_MAGIX.GetXMLtagNEW(IsoExpXML, "MaxExpRange")
                        NewtBackList = None
                        BestMolfitFile, IsoRatioFileContent, ListSubtractedModelSpectra = GetBestResult(IsoRatioFitDir, ConvertLogLinFlag, \
                                                                                                        NewExpFileList, ListNumberExpRanges, \
                                                                                                        NewFreqMinList, NewFreqMaxList, NewtBackList, \
                                                                                                        NewtSlopeList, SQLParameters = SQLParamArray)

                        ##================================================================================================================================
                        ## perform a final single molecule fit to adjust molfit parameters and iso ratio
                        if (DoFinalSMFitFlag):


                            ## create subdirectory for fitting
                            FinalSMFitDir = LocalJobDir + "final/"
                            if (not os.path.exists(FinalSMFitDir)):
                                os.system("mkdir -p " + FinalSMFitDir)


                            ## write molfit file to fit directory and fix all molfit parameters
                            FinalSMFMolfitFileName = FinalSMFitDir + "Final_SMF.molfit"
                            FinalSMFMolfitFile = open(FinalSMFMolfitFileName, 'w')
                            for line in MolfitFile:
                                StrippedLine = line.strip()
                                if (StrippedLine != ""):
                                    if (not StrippedLine.startswith("%")):
                                        SplittedLine = StrippedLine.split()
                                        if (len(SplittedLine) > 2):
                                            line = line.lower()
                                            line = line.replace("n", "y")                   ## free all molfit parameters
                                FinalSMFMolfitFile.write(line)
                            FinalSMFMolfitFile.close()


                            ## write iso ratio file to final directory
                            NewIsoRatioFileName = FinalSMFitDir + ListMolDirName[LocalMoleculeIndex]
                            NewIsoRatioFile = open(NewIsoRatioFileName + ".iso", 'w')
                            CountLines = 0
                            for line in IsoRatioFileContent:
                                CountLines += 1
                                if (CountLines > 1):
                                    NewIsoRatioFile.write(line)
                            NewIsoRatioFile.close()


                            ## copy algorithm xml file to local fit directory
                            task_myXCLASS.NormNCopyMove(LocalJobDir + "/algorithm_control.xml", FinalSMFitDir + "algorithm_control.xml", copyFlag = True)


                            ## update paths of obs. xml file
                            FinalSMFitExpXML = FinalSMFitDir + "exp.xml"
                            dbDefaultFilename = MAGIXrootDir + "../../"
                            dbDefaultFilename = os.path.normpath(dbDefaultFilename) + "/Database/cdms_sqlite.db"
                            AdjustExpXMLFile(NewExpXMLFileName, FinalSMFitDir, LocalJobDir, dbDefaultFilename)
                            dbList = task_MAGIX.GetXMLtagNEW(FinalSMFitExpXML, "dbFilename")
                            if (dbList == []):
                                dbList = [dbDefaultFilename]
                            task_MAGIX.WriteXMLtagNEW(FinalSMFitExpXML, "IsoTableFileName", [NewIsoRatioFileName])
                            task_MAGIX.WriteXMLtagNEW(FinalSMFitExpXML, "iso_flag", ["True"])


                            ## prepare other xml files for application of myXCLASSFit function
                            task_myXCLASSFit.CreateIOControlXMLFile(FinalSMFitDir, FinalSMFitExpXML, FinalSMFMolfitFileName, MAGIXrootDir)
                            # CreateIOControlXMLFile(FinalSMFitDir, FinalSMFitExpXML, FinalSMFMolfitFileName, MAGIXrootDir)


                            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                            ## start MAGIX locally
                            if (ClusterFlag == "false"):


                                ## use magix
                                if (Optimizer == "magix"):
                                    cmdString = "cd " + MAGIXrootDir + "; python3 magix_start.py --quiet --model=myxclass "
                                    cmdString += FinalSMFitDir + "io_control.xml >> " + FinalSMFitDir + "screen-output__magix.out"
                                    # cmdString += FinalSMFitDir + "io_control.xml"
                                    os.system(cmdString)


                                ## use scipy
                                else:
                                    ChannelIntegrationFlag = True
                                    NameOfFunction = "LineIdentification"
                                    LocalAlgorithmXMLFileName = FinalSMFitDir + "algorithm_control.xml"
                                    JobDir = task_myXCLASSMapFit.myXCLASSMapFitCore(MolfitsFileName = FinalSMFMolfitFileName, \
                                                                                    ObsXMLFileName = FinalSMFitExpXML, \
                                                                                    AlgorithmXMLFile = LocalAlgorithmXMLFileName, \
                                                                                    ChannelIntegrationFlag = ChannelIntegrationFlag, \
                                                                                    regionFileName = "", \
                                                                                    clusterdef = "", \
                                                                                    FastFitFlag = True, \
                                                                                    FullFitFlag = True, \
                                                                                    JobDir = FinalSMFitDir, \
                                                                                    NameOfFunction = NameOfFunction)
                                    NewExpXMLFileName = JobDir + "obs.xml"


                                    ## move results of MapFit run to current directory
                                    # cmdString = "mv {:s} {:s}".format(JobDir, IsoRatioFitDir)
                                    # os.system(cmdString)


                            ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                            ## start MAGIX on node
                            else:
                                cmdString = "ssh " + Node + " " + chr(34) + "cd " + MAGIXrootDir + "; "
                                cmdString += "nohup python3 magix_start.py --noplot --model=myxclass "
                                cmdString += FinalSMFitDir + "io_control.xml >> " + FinalSMFitDir + "screen-output__magix.out"
                                # cmdString += FinalSMFitDir + "io_control.xml"
                                cmdString += chr(34)
                                os.system(cmdString)


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## read in molfit file which correspond to the best fit
                            if (Optimizer == "scipy"):
                                ConvertLogLinFlag = "false"
                            else:
                                ConvertLogLinFlag = "true"
                            NewExpFileList = task_MAGIX.GetXMLtagNEW(FinalSMFitExpXML, "FileNamesExpFiles")
                            ListNumberExpRanges = task_MAGIX.GetXMLtagNEW(FinalSMFitExpXML, "NumberExpRanges")
                            NewFreqMinList = task_MAGIX.GetXMLtagNEW(FinalSMFitExpXML, "MinExpRange")
                            NewFreqMaxList = task_MAGIX.GetXMLtagNEW(FinalSMFitExpXML, "MaxExpRange")
                            NewtBackList = None
                            BestMolfitFile, IsoRatioFileContent, ListSubtractedModelSpectra = GetBestResult(FinalSMFitDir, ConvertLogLinFlag, \
                                                                                                            NewExpFileList, ListNumberExpRanges, \
                                                                                                            NewFreqMinList, NewFreqMaxList, \
                                                                                                            NewtBackList, NewtSlopeList, \
                                                                                                            SQLParameters = SQLParamArray)

                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## write optimized molfit and iso ratio file to "identified molecule" directory


                            ## write optimized molfit file to "identified molecule" directory
                            MoleculeResultName = IdentifiedMoleculesDir + ListMolDirName[LocalMoleculeIndex]
                            FinalBestMolfitFile = open(MoleculeResultName + ".out.molfit", 'w')
                            for line in BestMolfitFile:
                                FinalBestMolfitFile.write(line)
                            FinalBestMolfitFile.close()


                            ## write optimized iso ratio file to "identified molecule" directory
                            NewIsoRatioFileName = IdentifiedMoleculesDir + ListMolDirName[LocalMoleculeIndex]
                            NewIsoRatioFile = open(NewIsoRatioFileName + ".out.iso", 'w')
                            CountLines = 0
                            for line in IsoRatioFileContent:
                                CountLines += 1
                                if (CountLines > 1):
                                    NewIsoRatioFile.write(line)
                            NewIsoRatioFile.close()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create a plot of the current molecule and store the plot into the identified or not-identified directory


        ## create subdirectory
        if (ContributionFlag == "true"):
            LocalPlotDir = IdentifiedMoleculesDir + MolDirName + "/"
        else:
            LocalPlotDir = NotIdentifiedMoleculesDir + MolDirName + "/"
        cmdString = "mkdir -p " + LocalPlotDir
        os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create plots
        if (TWFSpectraList != {}):
            counterPNG = 0
            for TFWParameterDict in TWFSpectraList:                                              ## loop over all transition frequency windows
                counterPNG += 1


                ## get parameters for current transition frequency window
                CurrentFrequencyAxis = TFWParameterDict['Frequency']
                CurrentVelocityAxis = TFWParameterDict['Velocity']
                CurrentIntensity = TFWParameterDict['Intensity']
                CurrentContSubtracted = TFWParameterDict['ContSubtracted']
                ContinuumArray = TFWParameterDict['Continuum']
                VelListInt = TFWParameterDict['VelListInt']
                TransFreq = TFWParameterDict['TransFreq']
                EinsteinA = TFWParameterDict['EinsteinA']
                ElowMin = TFWParameterDict['ElowMin']
                gup = TFWParameterDict['gup']
                GaussianParameters = TFWParameterDict['GaussianParameters']
                LocalLineCatalog = TFWParameterDict['LineCatalog']
                LocalLineCatalog = numpy.asarray(LocalLineCatalog)


                ## get obs. xml file
                ObsXMLParameterList = TFWParameterDict['ObsXMLParameterList']
                GlobalvLSR = GetDicKey(ObsXMLParameterList, 'GlobalvLSR', 0.0)
                NoiseLevel = GetDicKey(ObsXMLParameterList, 'NoiseLevel', 0.0)
                LocalRangeCounter = ObsXMLParameterList['LocalRangeCounter']


                ## get modeled spectrum
                of1 = numpy.nanmin(CurrentFrequencyAxis)
                of2 = numpy.nanmax(CurrentFrequencyAxis)
                LocalFoundFlag = False
                for LocalModelSpectrum in ListSubtractedModelSpectra:
                    mf1 = numpy.nanmin(LocalModelSpectrum[:, 0])
                    mf2 = numpy.nanmax(LocalModelSpectrum[:, 0])
                    if ((mf1 <= of1 and of1 <= mf2) or (mf1 <= of2 and of2 <= mf2)):
                        LocalFoundFlag = True
                        if ((mf1 <= of1 and of1 <= mf2) and (mf1 <= of2 and of2 <= mf2)):
                            break
                if (not LocalFoundFlag):
                    print ("\n\n\nWARNING: Can not find modeled spectrum for current TWF spectrum!")
                    print ("\n\tFrequency axis of obs. TWF spectrum = ", CurrentFrequencyAxis)
                    print ("\tFrequency axis of modeled TWF spectrum:")
                    for LocalModelSpectrum in ListSubtractedModelSpectra:
                        print ("\t\tLocalModelSpectrum[:, 0] = ", LocalModelSpectrum[:, 0])


                ## add strong molecule contribution to modeled spectra
                if (StrongMolContribution != []):
                    LocalFoundFlag = False
                    for LocalStrongModelSpectrum in StrongMolModeledSpectra:
                        mf1 = numpy.nanmin(LocalStrongModelSpectrum[:, 0])
                        mf2 = numpy.nanmax(LocalStrongModelSpectrum[:, 0])
                        if ((mf1 <= of1 and of1 <= mf2) or (mf1 <= of2 and of2 <= mf2)):
                            LocalFoundFlag = True
                            if ((mf1 <= of1 and of1 <= mf2) and (mf1 <= of2 and of2 <= mf2)):
                                break
                    if (not LocalFoundFlag):
                        print ("\n\n\nWARNING: Can not find modeled spectrum for current TWF spectrum!")
                        print ("\n\tFrequency axis of obs. TWF spectrum = ", CurrentFrequencyAxis)
                        print ("\tFrequency axis of modeled TWF spectrum:")
                        for LocalStrongModelSpectrum in StrongMolModeledSpectra:
                            print ("\t\tLocalModelSpectrum[:, 0] = ", LocalStrongModelSpectrum[:, 0])
                    else:
                        LocalModelSpectrum[:, 1] += LocalStrongModelSpectrum[:, 1]


                ## add continuum
                # if (ContinuumArray != []):
                #     i1 = max(0, (numpy.abs(LocalModelSpectrum[:, 0] - of1)).argmin())
                #     i2 = min(len(LocalModelSpectrum[:, 0]) - 1, (numpy.abs(LocalModelSpectrum[:, 0] - of2)).argmin())
                #     In1 = min(i1, i2)
                #     In2 = max(i1, i2)
                #     LocalModelSpectrum[In1:In2, 1] += ContinuumArray[1:In2 - In1 + 1]


                ## create local obs. data and continuum array
                ObsDataTransWindow = numpy.zeros((len(CurrentContSubtracted), 3), dtype=numpy.float32)
                ObsDataTransWindow[:, 0] = CurrentFrequencyAxis[:]
                ObsDataTransWindow[:, 1] = CurrentVelocityAxis[:]
                ObsDataTransWindow[:, 2] = CurrentContSubtracted[:]

                # Debug:
                # print ("\nLocalRangeCounter = ", LocalRangeCounter)
                # print ("CurrentFrequencyAxis = ", CurrentFrequencyAxis)
                # print ("LocalModelSpectrum[:, 0] = ", LocalModelSpectrum[:, 0])
                # print ("ListSubtractedModelSpectra[0][:, 0] = ", ListSubtractedModelSpectra[0][:, 0])


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## create plot
                RawDataPlotFlag = False
                TransitionParameter = [TransFreq, EinsteinA, ElowMin, gup]
                Localv00 = None
                PlotTranstionWindow(RawDataPlotFlag, counterPNG, TransFreq, TransitionParameter, NoiseLevel, ObsDataTransWindow, LocalModelSpectrum, \
                                    LocalPlotDir, MolDirName, velLowLimit = velLowerLimit, velUpLimit = velUpperLimit, vComp = Localv00, \
                                    LineCatalog = LocalLineCatalog)



        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## use old plotting method to create final TWF plots
        else:
            CounterFreqRange = (-1)
            counterPNG = 0
            for ObsDataFileIndex in range(len(NewExpFileList)):
                NewExpFileName = NewExpFileList[ObsDataFileIndex][1]
                NewSpectraName = os.path.basename(NewExpFileName)

                # Debug:
                # print ("\n\nNewExpFileName = ", NewExpFileName)
                # print ("NewSpectraName = ", NewSpectraName)


                ## determine corresponding observed unmodified spectrum
                # SpectraName = ""
                OldSpectraIndex = 0
                for LocalSpectraIndex in range(len(ListOfSpectraNames)):
                    if (NewSpectraName == os.path.basename(ListOfSpectraNames[LocalSpectraIndex][1])):
                        OldSpectraIndex = LocalSpectraIndex
                        NewASCIIdata = ListOfSpectra[LocalSpectraIndex]
                        SubtractedSpectrumLocal = ListSubtractedSpectra[LocalSpectraIndex]


                        ## get name of exp. data file of current frequency range
                        SpectraName = ListOfSpectraNames[LocalSpectraIndex][1]
                        i = SpectraName.rfind("/")
                        if (i > (-1)):
                            SpectraName = SpectraName[i + 1:]
                        SpectraName = os.path.basename(SpectraName)
                        break

                # Debug:
                # print ("OldSpectraIndex = ", OldSpectraIndex)
                # print ("SpectraName = ", SpectraName)
                # print ("SubtractedSpectrumLocal = ", SubtractedSpectrumLocal)


                ## get parameters for current range
                RangeIndex = (-1)
                ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, GlobalvLSRListIn = NewGlobalvLSRList, \
                                                                               NumberRangeListIn = NumberExpRangesList, Redshift_ListIn = RedshiftList)
                NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']
                GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
                if (GlobalvLSR is None):
                    GlobalvLSR = 0.0
                Redshift = ObsXMLParameterDictFile['Redshift']
                if (Redshift is None):
                    Redshift = 0.0


                ## subtract strong molecule contribution
                FreqArray = NewASCIIdata[:, 0]
                for RangeIndex in range(NumberFrequencyRanges):


                    ## get parameters for current range
                    ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = NewFreqMinList, \
                                                                                    FreqMaxListIn = NewFreqMaxList, tBackListIn = NewtBackList, \
                                                                                    tSlopeListIn = NewtSlopeList, NoiseListIn = NewNoiseList)
                    FreqMin = ObsXMLParameterDictRange['FreqMin']
                    if (FreqMin is not None):
                        CounterFreqRange += 1
                        FreqMax = ObsXMLParameterDictRange['FreqMax']
                        tBack = ObsXMLParameterDictRange['tBack']
                        tSlope = ObsXMLParameterDictRange['tSlope']
                        Noise = ObsXMLParameterDictRange['NoiseLevel']
                        if (Noise is None):
                            Noise = 0.0

                        # Debug:
                        # print ("\n\n\nRangeIndex = ", RangeIndex)
                        # print ("FreqMin = ", FreqMin)
                        # print ("FreqMax = ", FreqMax)
                        # print ("CounterFreqRange = ", CounterFreqRange)
                        # print ("ListSubtractedModelSpectra = ", ListSubtractedModelSpectra)


                        ## get modeled spectrum
                        CurrentFreqRangeModelValues = ListSubtractedModelSpectra[CounterFreqRange]
                        ModelFreqMinIndex = max(0, (numpy.abs(CurrentFreqRangeModelValues[:, 0] - FreqMin)).argmin() - 1)
                        ModelFreqMaxIndex = min(len(CurrentFreqRangeModelValues[:, 0]) - 1, \
                                                (numpy.abs(CurrentFreqRangeModelValues[:, 0] - FreqMax)).argmin() + 1)
                        In1 = min(ModelFreqMinIndex, ModelFreqMaxIndex)
                        In2 = max(ModelFreqMinIndex, ModelFreqMaxIndex)
                        CurrentFreqRangeModelValues = CurrentFreqRangeModelValues[In1:In2, :]

                        # Debug:
                        # print ("SpectraName = ", SpectraName)
                        # print ("CurrentFreqRangeModelValues = ", CurrentFreqRangeModelValues)


                        ## add strong molecule contribution to modeled spectra
                        if (StrongMolContribution != []):
                            AllLocalStrongMolModeledSpectra = StrongMolModeledSpectra[OldSpectraIndex]
                            StrongFreqMinIndex = max(0, (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMin)).argmin() - 1)
                            StrongFreqMaxIndex = min(len(AllLocalStrongMolModeledSpectra[:, 0]) - 1, \
                                                     (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMax)).argmin() + 1)
                            In1 = min(StrongFreqMinIndex, StrongFreqMaxIndex)
                            In2 = max(StrongFreqMinIndex, StrongFreqMaxIndex)
                            LocalStrongMolModeledSpectra = AllLocalStrongMolModeledSpectra[In1:In2, :]


                            ## subtract strong molecule contribution from observational data
                            for kkk in range(min(len(CurrentFreqRangeModelValues[:, 1]), len(LocalStrongMolModeledSpectra[:, 1]))):
                                CurrentFreqRangeModelValues[kkk, 1] = CurrentFreqRangeModelValues[kkk, 1] + LocalStrongMolModeledSpectra[kkk, 1]

                        # Debug:
                        # print ("ObsDataFileIndex = ", ObsDataFileIndex)
                        # print ("RangeIndex = ", RangeIndex)
                        # print ("CurrentFreqRangeModelValues = ", CurrentFreqRangeModelValues)
                        # print ("ListSubtractedModelSpectra[CounterFreqRange] = ", ListSubtractedModelSpectra[CounterFreqRange])


                        ## plot window only, if modeled spectrum has a length > 1
                        if (len(CurrentFreqRangeModelValues[:, 0]) > 1):


                            ## get observation data for current freq. range
                            ymin1 = numpy.nanmin(CurrentFreqRangeModelValues[:, 1])
                            ymax1 = numpy.nanmax(CurrentFreqRangeModelValues[:, 1])


                            ## get range indices for exp. data
                            LocalCopyArray = SubtractedSpectrumLocal[:, 0]
                            FreqMinIndexLocal = max(0, (numpy.abs(LocalCopyArray - FreqMin)).argmin() - 1)
                            FreqMaxIndexLocal = min(len(LocalCopyArray) - 1, (numpy.abs(LocalCopyArray - FreqMax)).argmin() + 1)


                            ##****************************************************************************************************************************
                            ## check, that indices describe frequencies within given frequency range
                            #    nuMin = SubtractedSpectrumLocal[FreqMinIndexLocal, 0]
                            #    if (nuMin < FreqMin):
                            #        FreqMinIndexLocal += 1
                            #        nuMin = SubtractedSpectrumLocal[FreqMinIndexLocal, 0]
                            #        if (nuMin < FreqMin):                                           ## data seemed to be in reversed order
                            #            FreqMinIndexLocal = FreqMinIndexLocal - 2
                            #            FreqMinIndexLocal = max(0, FreqMinIndexLocal)
                            #        FreqMinIndexLocal = max(0, FreqMinIndexLocal)
                            #    nuMax = SubtractedSpectrumLocal[FreqMaxIndexLocal, 0]
                            #    if (nuMax > FreqMax):
                            #        FreqMaxIndexLocal -= 1
                            #        nuMax = SubtractedSpectrumLocal[FreqMaxIndexLocal, 0]
                            #        if (nuMax > FreqMax):
                            #            FreqMaxIndexLocal = FreqMaxIndexLocal + 2
                            #        FreqMaxIndexLocal = min(len(LocalCopyArray), FreqMaxIndexLocal)


                            ## cut out data within frequency range
                            xminIndex = min(FreqMinIndexLocal, FreqMaxIndexLocal)
                            xmaxIndex = max(FreqMinIndexLocal, FreqMaxIndexLocal)
                            SubtractedSpectrum = SubtractedSpectrumLocal[xminIndex:xmaxIndex, :]

                            # Debug:
                            # print ("FreqMin = ", FreqMin)
                            # print ("FreqMax = ", FreqMax)
                            # print ("FreqMinIndexLocal = ", FreqMinIndexLocal)
                            # print ("FreqMaxIndexLocal = ", FreqMaxIndexLocal)
                            # print ("xminIndex = ", xminIndex)
                            # print ("xmaxIndex = ", xmaxIndex)
                            # print ("LocalCopyArray = ", LocalCopyArray)
                            # print ("SubtractedSpectrum = ", SubtractedSpectrum)


                            ## mark non-doppler shifted transition frequencies
                            f1 = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
                            f2 = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
                            ShiftedFreqMin = min(f1, f2)
                            ShiftedFreqMax = max(f1, f2)
                            LocalMaxNumTransInFit = 0
                            TransFreqList, DBParamList = GetTransFreq(molecule, ShiftedFreqMin, ShiftedFreqMax, SQLParamArray, dbFile, \
                                                                      LocalMaxNumTransInFit)
                            if (TransFreqList != []):


                                ## plot frequency window
                                for TransIndex in range(len(TransFreqList)):                ## loop over all transition frequencies
                                    counterPNG += 1                                         ## increase counter for plots


                                    ## get trans. frequency and shift back by v_LSR to coincide with obs. data freq. range
                                    TransFreq = task_myXCLASS.ConvertFreq(TransFreqList[TransIndex], GlobalvLSR, z = Redshift)
                                    RestFreq = TransFreq                                    ## define rest frequency
                                    vLSR = 0.0                                              ## current trans. freq. is at v = 0

                                    # Debug:
                                    # print ("\n\n\n\n")
                                    # print ("FreqMin = ", FreqMin)
                                    # print ("FreqMax = ", FreqMax)
                                    # print ("TransIndex = ", TransIndex)
                                    # print ("TransFreq = ", TransFreq)
                                    # print ("EinsteinA = ", DBParamList[TransIndex][1])
                                    # print ("ElowMin   = ", DBParamList[TransIndex][2])
                                    # print ("gup       = ", DBParamList[TransIndex][3])


                                    ## add new velocity axis to ImportExpData array
                                    VelocityData = task_myXCLASS.ChangeToVelocitiy(SubtractedSpectrumLocal[:, 0], RestFreq, vLSR)
                                    ObsDataTransWindow = numpy.zeros((len(SubtractedSpectrumLocal), 4), dtype = numpy.float32)
                                    ObsDataTransWindow[:, 0] = SubtractedSpectrumLocal[:, 0]
                                    ObsDataTransWindow[:, 1] = numpy.asarray(VelocityData[:])
                                    ObsDataTransWindow[:, 2] = SubtractedSpectrumLocal[:, -1]


                                    ##--------------------------------------------------------------------------------------------------------------------
                                    ## create plot
                                    RawDataPlotFlag = False
                                    v1 = task_myXCLASS.ConvertFreq(TransFreq, velLowerLimit)
                                    v2 = task_myXCLASS.ConvertFreq(TransFreq, velUpperLimit)
                                    vv1 = min(v1, v2)
                                    vv2 = max(v1, v2)
                                    v1FreqIndex = max(0, (numpy.abs(ObsDataTransWindow[:, 0] - vv1)).argmin() - 1)
                                    v2FreqIndex = min(len(ObsDataTransWindow[:, 0]) - 1, (numpy.abs(ObsDataTransWindow[:, 0] - vv2)).argmin() + 1)
                                    In1 = min(v1FreqIndex, v2FreqIndex)
                                    In2 = max(v1FreqIndex, v2FreqIndex)
                                    PlotTranstionWindow(RawDataPlotFlag, counterPNG, TransFreq, DBParamList[TransIndex], Noise, \
                                                        ObsDataTransWindow[In1:In2, :], ListSubtractedModelSpectra[CounterFreqRange], \
                                                        LocalPlotDir, ListMolDirName[LocalMoleculeIndex], \
                                                        velLowLimit = velLowerLimit, velUpLimit = velUpperLimit)


    ## define return variables
    return (ContributionFlag, MoleculeResultName)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## starts worker via command line and analyze results
##
def StartWorker(ThreadID, ServerList, ListOfMolecules, StrongMolContribution, myXCLASSrootDir, RelativeMAGIXDir, LineIDJobDir, MoleculejobDirList, \
                ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, ListNoiseLevels, \
                CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, ListMolDirName, ObsXMLFileNameOrig, \
                DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, SQLParamArray, DecisionMethod, Tolerance, \
                TypeSelection, SmoothValue, MaxNumTransInFit, InternalParameterList):
    """

input parameters:
-----------------

    - ThreadID:                     current thread id

    - ServerList:                   list of all server in the cluster and the corresponding settings

    - ListOfMolecules:              list of molecules

    - StrongMolContribution:        contribution of strong (highly abundant) molecules

    - myXCLASSrootDir:              root directory of the XCLASS package

    - RelativeMAGIXDir:             relative MAGIX directory

    - LineIDJobDir:                 job directory for current LineID run

    - MoleculejobDirList:           current single molecule fit directory

    - ListOfSpectraNames:           list of all spectrum (observational data) files

    - ListOfSpectra:                list of all observed spectra

    - ListSubtractedSpectra:        list of all continuum subtracted observational spectra

    - MaxOverestimationHeight:      overestimation factor

    - ResultsFile:                  final result file

    - IdentifiedMoleculeFile:       file for intermediate identified molecules

    - ListNoiseLevels:              list of all noise levels for all frequency ranges of all spectra

    - CheckEmissionAbsorptionFlag:  flag to consider emission and absorption separately

    - IdentifiedMoleculesDir:       path of the directory containing the intermediate identified molecules of the current LineID run

    - NotIdentifiedMoleculesDir:    path of the directory containing the non-identified molecules of the current LineID run

    - ListMolDirName:               list of modified molecule names (used for naming of directories)

    - ObsXMLFileNameOrig:           name of MAGIX exp. xml file

    - DefaultMolfitUsedFlagList:    defines, if a default molfit file was used or not (already takes EstimateParamFlag into account)

    - AlgorithmXMLFileEstParam:     path and name of algorithm xml file (used for parameter estimation algorithm)

    - AlgorithmXMLFileISO:          path and name of algorithm xml file (used for fit of iso ratios)

    - VelBin:                       velocity resolution for histogram analysis

    - SQLParamArray:                array containing sql parameters

    - DecisionMethod:               method which is used

    - Tolerance:                    tolerance factor for method "significant"

    - TypeSelection:                exclude type of lines (only emission or only absorption or all types)

    - InternalParameterList:        list of internal parameters


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("ThreadID = ", ThreadID)
    # print ("ThreadID, ServerList = ", ThreadID, ServerList)
    # print ("ListOfMolecules = ", ListOfMolecules)
    # print ("RelativeMAGIXDir = ", RelativeMAGIXDir)
    # print ("ConvertLogLinFlag = ", ConvertLogLinFlag)
    # print ("ListSubtractedSpectra = ", ListSubtractedSpectra)
    # print ("ListSubtractedModelSpectra = ", ListSubtractedModelSpectra)
    # print ("MaxOverestimationHeight = ", MaxOverestimationHeight)
    # print ("ResultsFile = ", ResultsFile)
    # print ("ListNoiseLevels = ", ListNoiseLevels)
    # print ("ListOfSpectraNames = ", ListOfSpectraNames)
    # print ("CheckEmissionAbsorptionFlag = ", CheckEmissionAbsorptionFlag)
    # print ("IdentifiedMoleculesDir = ", IdentifiedMoleculesDir)
    # print ("ListMolDirName = ", ListMolDirName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## increase global molecule counter for all threads
    lock = ""
    InitFlag = "true"
    NumMol = len(ListOfMolecules)
    NumWorkers = len(ServerList)
    for LocalMoleculeIndex in range(ThreadID, NumMol, NumWorkers):
        molecule = ListOfMolecules[LocalMoleculeIndex]

        # Debug:
        # print ("\n\n  LocalMoleculeIndex = ", LocalMoleculeIndex)


        ## print what you do
        print ("\nPerform single molecule fit for " + molecule + "..", flush = True)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get some parameters for each node in the cluster
        server = ServerList[ThreadID]                                                       ## get cluster settings for the server ThreadID
        Node = server[0]                                                                    ## get name of node
        NumCores = server[1]                                                                ## get number of cores
        RootPath = server[2].strip()                                                        ## get path of xclass interface
        if (RootPath != ""):                                                                ## if path of xclass interface is defined ..
            RootPath = RootPath + "/addons/MAGIX/"                                          ## .. define path of MAGIX


        ## define path of the xclass interface on the different nodes
        if (RootPath.strip() == ""):
            PathXCLASSInterface = task_myXCLASS.GetMAGIXRootDir()
        else:
            PathXCLASSInterface = RootPath


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define cluster flag


        ## get NoMAGIX flag from init file
        OutDict = task_myXCLASS.GetXCLASSSettings()
        NoMAGIXFlag = OutDict['NoMAGIXFlag']
        if (NoMAGIXFlag):
            Optimizer = "scipy"
        else:
            Optimizer = "magix"

        # Debug:
        # print ("Optimizer = ", Optimizer)


        ## set cluster flag
        ClusterFlag = "true"
        if (Optimizer == "scipy" or (len(ServerList) == 1 and Node == "localhost")):
            ClusterFlag = "false"


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## redirect screen output to file
        MolDirName = MoleculeFileName(molecule)
        if (ClusterFlag == "true"):
            sys.stdout = open(LineIDJobDir + "/" + MoleculejobDirList[LocalMoleculeIndex] + "/screen__" + MolDirName + "__" + str(os.getpid()) + ".out", \
                              "a", buffering = 1)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## do single molecule fit
        ContributionFlag, MoleculeResultName = SingleMoleculeFit(ClusterFlag, lock, Node, molecule, StrongMolContribution, PathXCLASSInterface, \
                                                                 LineIDJobDir, MoleculejobDirList, LocalMoleculeIndex, ListOfSpectraNames, \
                                                                 ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, \
                                                                 IdentifiedMoleculeFile, ListNoiseLevels, CheckEmissionAbsorptionFlag, \
                                                                 IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, ListMolDirName, ObsXMLFileNameOrig, \
                                                                 DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, \
                                                                 SQLParamArray, DecisionMethod, Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit, \
                                                                 InternalParameterList)

        # Debug:
        # print ("ContributionFlag = ", ContributionFlag)
        # print ("MoleculeResultName = ", MoleculeResultName)
        # sys.exit(0)


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start single molecule fits related to strong lines
##
def StrongLinesFits(StrongMoleculeList, ListOfMolecules, myXCLASSRootDirectory, MAGIXrootDir, LineIDJobDir, MoleculejobDirList, \
                    ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, \
                    ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, ListMolDirName, \
                    ObsXMLFileNameOrig, UnmodifiedExpXMLFilename, DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, \
                    VelBin, SQLParamArray, DecisionMethod, Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit, InternalParameterList):
    """

input parameters:
-----------------

    - StrongMoleculeList:           list of all strong molecules, i.e. molecules with a high abundance

    - ListOfMolecules:              list of all molecules

    - myXCLASSRootDirectory:        root directory of the XCLASS package

    - MAGIXrootDir:                 MAGIX root directory

    - LineIDJobDir:                 job directory for current LineID run

    - MoleculejobDirList:           current single molecule fit directory

    - ListOfSpectraNames:           list of all spectrum (observational data) files

    - ListOfSpectra:                list of all observed spectra

    - ListSubtractedSpectra:        list of all continuum subtracted observational spectra

    - MaxOverestimationHeight:      overestimation factor

    - ResultsFile:                  final result file

    - IdentifiedMoleculeFile:       file for intermediate molecules

    - ListNoiseLevels:              list of all noise levels for all frequency ranges of all spectra

    - CheckEmissionAbsorptionFlag:  flag to consider emission and absorption separately

    - IdentifiedMoleculesDir:       path of the directory containing the intermediate identified molecules of the current LineID run

    - NotIdentifiedMoleculesDir:    path of the directory containing the non-identified molecules of the current LineID run

    - ListMolDirName:               list of modified molecule names (used for naming of directories)

    - ObsXMLFileNameOrig:           name of MAGIX exp. xml file

    - UnmodifiedExpXMLFilename:     path and name of unmodified observational xml file

    - DefaultMolfitUsedFlagList:    defines, if a default molfit file was used or not (already takes EstimateParamFlag into account)

    - AlgorithmXMLFileEstParam:     path and name of algorithm xml file (used for parameter estimation algorithm)

    - AlgorithmXMLFileISO:          path and name of algorithm xml file (used for fit of iso ratios)

    - VelBin:                       velocity resolution for histogram analysis

    - SQLParamArray:                array containing sql parameters

    - DecisionMethod:               method which is used

    - Tolerance:                    tolerance factor for method "significant"

    - TypeSelection:                exclude type of lines (only emission or only absorption or all types)

    - SmoothValue:                  smooth factor (=0, no smoothing is performed)

    - MaxNumTransInFit:             max. number of transitions which are fitted in parameter estimation algorithm

    - InternalParameterList:        list of internal parameters


output parameters:
------------------

    - StrongMolContribution:        contribution of all identified strong molecules to each single molecule fit:

                                    - StrongMolContribution[0]:   molfit parameters

                                    - StrongMolContribution[1]:   corresponding modeled spectra

    - NewListOfMolecules:           correct list of molecules
    """

    # Debug:
    # print ("StrongMoleculeList = ", StrongMoleculeList)
    # print ("ListOfMolecules = ", ListOfMolecules)
    # print ("myXCLASSRootDirectory = ", myXCLASSRootDirectory)
    # print ("MAGIXrootDir = ", MAGIXrootDir)
    # print ("LineIDJobDir = ", LineIDJobDir)
    # print ("MoleculejobDirList = ", MoleculejobDirList)
    # print ("ListOfSpectraNames = ", ListOfSpectraNames)
    # print ("ListOfSpectra = ", ListOfSpectra)
    # print ("ListSubtractedSpectra = ", ListSubtractedSpectra)
    # print ("MaxOverestimationHeight = ", MaxOverestimationHeight)
    # print ("ResultsFile = ", ResultsFile)
    # print ("IdentifiedMoleculeFile = ", IdentifiedMoleculeFile)
    # print ("ListNoiseLevels = ", ListNoiseLevels)
    # print ("CheckEmissionAbsorptionFlag = ", CheckEmissionAbsorptionFlag)
    # print ("IdentifiedMoleculesDir = ", IdentifiedMoleculesDir)
    # print ("NotIdentifiedMoleculesDir = ", NotIdentifiedMoleculesDir)
    # print ("ListMolDirName = ", ListMolDirName)
    # print ("ObsXMLFileNameOrig = ", ObsXMLFileNameOrig)
    # print ("UnmodifiedExpXMLFilename = ", UnmodifiedExpXMLFilename)
    # print ("DefaultMolfitUsedFlagList = ", DefaultMolfitUsedFlagList)
    # print ("AlgorithmXMLFileEstParam = ", AlgorithmXMLFileEstParam)
    # print ("AlgorithmXMLFileISO = ", AlgorithmXMLFileISO)
    # print ("VelBin = ", VelBin)
    # print ("SQLParamArray = ", SQLParamArray)
    # print ("DecisionMethod = ", DecisionMethod)
    # print ("Tolerance = ", Tolerance)
    # print ("TypeSelection = ", TypeSelection)
    # print ("SmoothValue = ", SmoothValue)
    # print ("MaxNumTransInFit = ", MaxNumTransInFit)
    # print ("InternalParameterList = ", InternalParameterList)


    ## initialize output parameters
    StrongMolContribution = []
    NewListOfMolecules = []
    ModelSpectra = []


    ## check, if no strong molecule is defined
    if (StrongMoleculeList == []):
        return (StrongMolContribution, NewListOfMolecules)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## do single molecule fit for each strong molecule
    ClusterFlag = "false"
    PathXCLASSInterface = task_myXCLASS.GetMAGIXRootDir()
    Node = "localhost"
    lock = ""
    LocalMoleculeIndex = 1
    StrongMolMolfit = []
    StrongMolContributionLocal = []
    CopyListOfMolecules = ListOfMolecules
    for StrongMolecule in StrongMoleculeList:

        # Debug:
        # print ("StrongMolecule = ", StrongMolecule)
        # print ("ClusterFlag = ", ClusterFlag)
        # print ("Node = ", Node)
        # print ("PathXCLASSInterface = ", PathXCLASSInterface)
        # print ("LineIDJobDir = ", LineIDJobDir)
        # print ("MoleculejobDirList = ", MoleculejobDirList)
        # print ("LocalMoleculeIndex = ", LocalMoleculeIndex)
        # print ("ListOfSpectraNames = ", ListOfSpectraNames)
        # print ("ListSubtractedSpectra = ", ListSubtractedSpectra)
        # print ("CheckEmissionAbsorptionFlag = ", CheckEmissionAbsorptionFlag)
        # print ("IdentifiedMoleculesDir = ", IdentifiedMoleculesDir)
        # print ("NotIdentifiedMoleculesDir = ", NotIdentifiedMoleculesDir)
        # print ("ListMolDirName = ", ListMolDirName)
        # print ("ObsXMLFileNameOrig = ", ObsXMLFileNameOrig)


        ## check if path of background files is defined
        CurrentMolecule = MoleculeFileName(StrongMolecule)
        LocalBackgroundFileList = GetBackgroundFileNames(InternalParameterList, CurrentMolecule)


        ## determine LocalMoleculeIndex
        for i in range(len(CopyListOfMolecules)):
            if (CopyListOfMolecules[i].strip() == StrongMolecule.strip()):
                LocalMoleculeIndex = i
                break


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## do single molecule fit
        ContributionFlag, MolfitsFileName = SingleMoleculeFit(ClusterFlag, lock, Node, StrongMolecule, StrongMolContributionLocal, PathXCLASSInterface, \
                                                              LineIDJobDir, MoleculejobDirList, LocalMoleculeIndex, ListOfSpectraNames, ListOfSpectra, \
                                                              ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, \
                                                              ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, \
                                                              NotIdentifiedMoleculesDir, ListMolDirName, ObsXMLFileNameOrig, DefaultMolfitUsedFlagList, \
                                                              AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, SQLParamArray, DecisionMethod, \
                                                              Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit, InternalParameterList)
        MolfitsFileName += ".out.molfit"
        StrongMolMolfit.append(MolfitsFileName)

        # Debug:
        # print ("ContributionFlag = ", ContributionFlag)
        # print ("MolfitsFileName = ", MolfitsFileName)


        ## remove an identified molecule from list of all molecules
        CopyListOfMolecules = ListOfMolecules
        ListOfMolecules = []
        for mol in CopyListOfMolecules:
            if (mol.strip() != StrongMolecule.strip()):
                ListOfMolecules.append(mol)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## if a strong molecule was identified
        if (ContributionFlag == "true"):


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## determine modeled spectra


            ## prepare molfit file
            LocalMolfitFile = open(MolfitsFileName)
            LocalMolfitFileContent = LocalMolfitFile.readlines()
            LocalMolfitFile.close()
            LocalMolfitsFileName = LineIDJobDir + "/strong_molecule__all.molfit"
            # LocalMolfitsFileName = MolfitsFileName
            NewLocalMolfitFile = open(LocalMolfitsFileName, 'w')
            for line in LocalMolfitFileContent:
                StrippedLine = line.strip()
                if (not StrippedLine.startswith("%") and StrippedLine != ""):
                    SplittedLine = StrippedLine.split()
                    if (len(SplittedLine) > 2):
                        line = line.lower()
                        line = line.replace("y", "n")                                       ## we don't want to fit, we just use the model values
                NewLocalMolfitFile.write(line)
            NewLocalMolfitFile.close()


            ## prepare call of myXCLASSFit function
            LocalNumberIteration = 1
            LocalRestFreq = 0.0
            LocalvLSR = 0.0


            ## start myXCLASSFit function
            newmolfit, modeldata, JobDir = task_myXCLASSFit.myXCLASSFit(NumberIteration = LocalNumberIteration, MolfitsFileName = LocalMolfitsFileName, \
                                                                        experimentalData = UnmodifiedExpXMLFilename, RestFreq = LocalRestFreq, \
                                                                        vLSR = LocalvLSR)

            ## get names of obs. data files and some other parameters
            ExpFileList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXMLFilename, "FileNamesExpFiles")
            NumberHeaderLinesList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXMLFilename, "NumberHeaderLines")
            SeparatorColumnsList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXMLFilename, "SeparatorColumns")
            UnmodifiedNumFreqRangeList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXMLFilename, "NumberExpRanges")
            UnmodifiedFreqMinList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXMLFilename, "MinExpRange")
            UnmodifiedFreqMaxList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXMLFilename, "MaxExpRange")
            # UnmodifiedGlobalvLSRList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXMLFilename, "GlobalvLSR")
            UnmodifiedtBackList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXMLFilename, "BackgroundTemperature")
            UnmodifiedtSlopeList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXMLFilename, "TemperatureSlope")
            UnmodifiedBackgroundFileNameList = task_MAGIX.GetXMLtagNEW(UnmodifiedExpXMLFilename, "BackgroundFileName")


            ## read in modeled spectra
            modeldata = []
            for ObsDataFileIndex in range(len(ExpFileList)):


                ## get parameters for current obs. data file
                RangeIndex = (-1)
                ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, \
                                                                               NumberRangeListIn = UnmodifiedNumFreqRangeList, \
                                                                               NumberHeaderLinesListIn = NumberHeaderLinesList, \
                                                                               SeparatorColumnsListIn = SeparatorColumnsList)
                NumFreqRange = ObsXMLParameterDictFile['NumberFrequencyRanges']
                NumberHeaderLines = ObsXMLParameterDictFile['NumberHeaderLines']
                SeparatorColumns = ObsXMLParameterDictFile['SeparatorColumns']


                ## define name of modeled spectra
                LocalObsFileName = ExpFileList[ObsDataFileIndex][1]
                LocalObsFileName = os.path.basename(LocalObsFileName)
                if (LocalObsFileName.endswith(".dat")):
                    LocalObsFileName = LocalObsFileName.replace(".dat", ".LM.out.dat")
                elif (LocalObsFileName.endswith(".fits")):
                    LocalObsFileName = LocalObsFileName.replace(".fits", ".LM.out.fits")


                ## import exp data
                if (SeparatorColumns.strip() == ""):
                    ImportExpData = numpy.loadtxt(JobDir + LocalObsFileName, skiprows = NumberHeaderLines)
                else:
                    ImportExpData = numpy.loadtxt(JobDir + LocalObsFileName, skiprows = NumberHeaderLines, delimiter = SeparatorColumns)


                ## remove continuum from modeled (strong molecule) spectra
                FreqArray = ImportExpData[:, 0]
                for RangeIndex in range(NumFreqRange):


                    ## get parameters for current obs. data file
                    ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = UnmodifiedFreqMinList, \
                                                                                    FreqMaxListIn = UnmodifiedFreqMaxList, \
                                                                                    tBackListIn = UnmodifiedtBackList, \
                                                                                    tSlopeListIn = UnmodifiedtSlopeList, \
                                                                                    BackgroundFileNameListIn = UnmodifiedBackgroundFileNameList)
                    FreqMin = ObsXMLParameterDictRange['FreqMin']
                    if (FreqMin is not None):
                        FreqMax = ObsXMLParameterDictRange['FreqMax']
                        tBack = ObsXMLParameterDictRange['tBack']
                        tSlope = ObsXMLParameterDictRange['tSlope']
                        BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                        if (BackgroundFileName is None):
                            BackgroundFileName = FindBackgroundFileName4Range(LocalBackgroundFileList, FreqMin, FreqMax)


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## subtract continuum, (IMPORTANT: v_LSR is not relevant here)


                        ## subtract background spectrum
                        if (BackgroundFileName != ""):
                            BackgroundFile = ImportASCIIFile(BackgroundFileName)            ## import interpolation object
                            LocalBackgroundSpectrum = BackgroundFile(FreqArray)             ## compute background spectrum at given frequencies
                            for i in range(len(ImportExpData[:, 1])):                       ## subtract background spectrum
                                ImportExpData[i, 1] = ImportExpData[i, 1] - LocalBackgroundSpectrum[i]


                        ## subtract phenom. continuum description from obs. data
                        if (tBack is not None and tBack != 0.0 and tSlope is not None):
                            LowFreq = numpy.nanmin(FreqArray)
                            NewTBack = tBack * (LowFreq / FreqMin)**tSlope                  ## determine new background temperature
                            for i in range(len(ImportExpData[:, 1])):
                               ImportExpData[i, 1] = ImportExpData[i, 1] - (abs(NewTBack) * (ImportExpData[i, 0] / LowFreq)**tSlope)


                ## store modeled spectra
                modeldata.append(ImportExpData)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## remove job directory for last myXCLASSFit run and modified molfit file
            cmdString = "rm -rf " + JobDir + " " + LocalMolfitsFileName
            os.system(cmdString)


            ## add modeled spectra to modeled spectra of other strong molecules
            if (ModelSpectra == []):
                ModelSpectra = modeldata
            else:
                for SpectraIndex in range(len(modeldata)):
                    ModelSpectra[SpectraIndex][:, 1] = ModelSpectra[SpectraIndex][:, 1] + modeldata[SpectraIndex][:, 1]


            ## store results to return parameter
            StrongMolContributionLocal = ModelSpectra


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define further return parameters
    NewListOfMolecules = ListOfMolecules
    StrongMolContribution = [StrongMolMolfit, StrongMolContributionLocal]
    return (StrongMolContribution, NewListOfMolecules)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start different molecule fits on the cluster
##
def StartClusterFits(ServerList, ListOfMolecules, StrongMolContribution, myXCLASSRootDirectory, MAGIXrootDir, LineIDJobDir, MoleculejobDirList, \
                     ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, \
                     ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, ListMolDirName, \
                     ObsXMLFileNameOrig, DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, SQLParamArray, \
                     DecisionMethod, Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit, InternalParameterList):
    """

input parameters:
-----------------

    - ServerList:                   list of all server in the cluster and the corresponding settings

    - ListOfMolecules:              list of molecules

    - StrongMolContribution:        contribution of strong (highly abundant) molecules

    - myXCLASSRootDirectory:        root directory of the XCLASS package

    - MAGIXrootDir:                 MAGIX root directory

    - LineIDJobDir:                 job directory for current LineID run

    - MoleculejobDirList:           current single molecule fit directory

    - ListOfSpectraNames:           list of all spectrum (observational data) files

    - ListOfSpectra:                list of all observed spectra

    - ListSubtractedSpectra:        list of all continuum subtracted observational spectra

    - MaxOverestimationHeight:      overestimation factor

    - ResultsFile:                  final result file

    - IdentifiedMoleculeFile:       file for intermediate molecules

    - ListNoiseLevels:              list of all noise levels for all frequency ranges of all spectra

    - CheckEmissionAbsorptionFlag:  flag to consider emission and absorption separately

    - IdentifiedMoleculesDir:       path of the directory containing the intermediate identified molecules of the current LineID run

    - NotIdentifiedMoleculesDir:    path of the directory containing the non-identified molecules of the current LineID run

    - ListMolDirName:               list of modified molecule names (used for naming of directories)

    - ObsXMLFileNameOrig:           name of MAGIX exp. xml file

    - DefaultMolfitUsedFlagList:    defines, if a default molfit file was used or not (already takes EstimateParamFlag into account)

    - AlgorithmXMLFileEstParam:     path and name of algorithm xml file (used for parameter estimation algorithm)

    - AlgorithmXMLFileISO:          path and name of algorithm xml file (used for fit of iso ratios)

    - VelBin:                       velocity resolution for histogram analysis

    - SQLParamArray:                array containing sql parameters

    - DecisionMethod:               method which is used

    - Tolerance:                    tolerance factor for method "significant"

    - TypeSelection:                exclude type of lines (only emission or only absorption or all types)

    - SmoothValue:                  smooth factor (=0, no smoothing is performed)

    - MaxNumTransInFit:             max. number of transitions which are fitted in parameter estimation algorithm

    - InternalParameterList:        list of internal parameters


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("CurrentDir = ", CurrentDir)
    # print ("ServerList = ", ServerList)
    # print ("ListOfMolecules = ", ListOfMolecules)
    # print ("StrongMolContribution = ", StrongMolContribution)
    # print ("myXCLASSRootDirectory = ", myXCLASSRootDirectory)
    # print ("MAGIXrootDir = ", MAGIXrootDir)
    # print ("LineIDJobDir = ", LineIDJobDir)
    # print ("MoleculejobDirList = ", MoleculejobDirList)
    # print ("ListOfSpectraNames = ", ListOfSpectraNames)
    # print ("ListOfSpectra = ", ListOfSpectra)
    # print ("ListSubtractedSpectra = ", ListSubtractedSpectra)
    # print ("MaxOverestimationHeight = ", MaxOverestimationHeight)
    # print ("ResultsFile = ", ResultsFile)
    # print ("IdentifiedMoleculeFile = ", IdentifiedMoleculeFile)
    # print ("ListNoiseLevels = ", ListNoiseLevels)
    # print ("CheckEmissionAbsorptionFlag = ", CheckEmissionAbsorptionFlag)
    # print ("IdentifiedMoleculesDir = ", IdentifiedMoleculesDir)
    # print ("NotIdentifiedMoleculesDir = ", NotIdentifiedMoleculesDir)
    # print ("ListMolDirName = ", ListMolDirName)
    # print ("ObsXMLFileNameOrig = ", ObsXMLFileNameOrig)
    # print ("DefaultMolfitUsedFlagList = ", DefaultMolfitUsedFlagList)
    # print ("AlgorithmXMLFileEstParam = ", AlgorithmXMLFileEstParam)
    # print ("AlgorithmXMLFileISO = ", AlgorithmXMLFileISO)
    # print ("VelBin = ", VelBin)
    # print ("SQLParamArray = ", SQLParamArray)
    # print ("DecisionMethod = ", DecisionMethod)
    # print ("Tolerance = ", Tolerance)
    # print ("TypeSelection = ", TypeSelection)
    # print ("SmoothValue = ", SmoothValue)
    # print ("MaxNumTransInFit = ", MaxNumTransInFit)
    # print ("InternalParameterList = ", InternalParameterList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start single molecule fits
    BlankString = 140 * " "
    print ("\r{:s}".format(BlankString), end = "")
    RelativeMAGIXDir = MAGIXrootDir.replace(myXCLASSRootDirectory, "")


    ## print what you do
    NumWorkers = len(ServerList)
    # if (NumWorkers == 1):
    #     print "\rPerform single molecule fit for ",
    # else:
    #     print "\rPerform single molecule fits for ",


    # Debug:
    # print ("\n\nServerList = ", ServerList)
    # print ("ListOfMolecules = ", ListOfMolecules)
    # print ("StrongMolContribution = ", StrongMolContribution)
    # print ("myXCLASSRootDirectory = ", myXCLASSRootDirectory)
    # print ("RelativeMAGIXDir = ", RelativeMAGIXDir)
    # print ("LineIDJobDir = ", LineIDJobDir)
    # print ("MoleculejobDirList = ", MoleculejobDirList)
    # print ("ListOfSpectraNames = ", ListOfSpectraNames)
    # print ("ListSubtractedSpectra = ", ListSubtractedSpectra)
    # print ("MaxOverestimationHeight = ", MaxOverestimationHeight)
    # print ("ResultsFile = ", ResultsFile)
    # print ("IdentifiedMoleculeFile = ", IdentifiedMoleculeFile)
    # print ("ListNoiseLevels = ", ListNoiseLevels)
    # print ("CheckEmissionAbsorptionFlag = ", CheckEmissionAbsorptionFlag)
    # print ("IdentifiedMoleculesDir = ", IdentifiedMoleculesDir)
    # print ("NotIdentifiedMoleculesDir = ", NotIdentifiedMoleculesDir)
    # print ("ListMolDirName = ", ListMolDirName)
    # print ("ObsXMLFileNameOrig = ", ObsXMLFileNameOrig)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## prepare start of workers
    thread_list = []
    for i in range(len(ServerList)):                                                        ## loop over all workers


        ## Set up the thread
        t = multiprocessing.Process(target = StartWorker, args = (i, ServerList, ListOfMolecules, StrongMolContribution, myXCLASSRootDirectory, \
                                                                  RelativeMAGIXDir, LineIDJobDir, MoleculejobDirList, ListOfSpectraNames, ListOfSpectra, \
                                                                  ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, \
                                                                  ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, \
                                                                  NotIdentifiedMoleculesDir, ListMolDirName, ObsXMLFileNameOrig, DefaultMolfitUsedFlagList, \
                                                                  AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, SQLParamArray, DecisionMethod, \
                                                                  Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit, InternalParameterList))


        ## Sticks the thread in a list so that it remains accessible
        thread_list.append(t)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    # Start all threads (workers)
    for thread in thread_list:
       thread.start()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    # This blocks the calling thread until the thread whose join() method is called is terminated.
    # From http://docs.python.org/2/library/threading.html#thread-objects
    for thread in thread_list:
       thread.join()


    # Debug:
    # sys.exit(0)
    ok = 0


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## create a new molfit file in the new format
##
def CreateMolfitFileInNewFormat(MolfitsFileName, SingleMolMolfitFiles, MinColumnDensityAbs, MinColumnDensityEmis, SQLParameters = None, \
                                FixParameterFlag = False):
    """

input parameters:
-----------------

    - MolfitsFileName:          path and name of molfit file

    - SingleMolMolfitFiles:     list containing all molfit files of all intermediate identified molecules

    - MinColumnDensityAbs:      minimal column density for an absorption component

    - MinColumnDensityEmis:     minimal column density for an emission component

    - SQLParameters:            (optional) SQL parameters

    - FixParameterFlag:         (optional) fix molfit parameters


output parameters:
------------------

    - None
    """

    # Debug:
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("SingleMolMolfitFiles = ", SingleMolMolfitFiles)
    # print ("MinColumnDensityAbs = ", MinColumnDensityAbs)
    # print ("MinColumnDensityEmis = ", MinColumnDensityEmis)
    # print ("SQLParameters = ", SQLParameters)
    # print ("FixParameterFlag = ", FixParameterFlag)


    ## get sql parameters from molfit file
    if (SQLParameters is None):
        MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(MolfitsFileName)
    else:
        MinNumTransitionsSQL = SQLParameters[0]
        MaxNumTransitionsSQL = SQLParameters[1]
        MaxElowSQL = SQLParameters[2]
        MingASQL = SQLParameters[3]
        OrderTransSQL = SQLParameters[4]

    # Debug:
    # print ("SQLParameters = ", SQLParameters)
    # print ("MolfitsFileName = ", MolfitsFileName)
    # print ("MinNumTransitionsSQL = ", MinNumTransitionsSQL)
    # print ("MaxNumTransitionsSQL = ", MaxNumTransitionsSQL)
    # print ("MaxElowSQL = ", MaxElowSQL)
    # print ("MingASQL = ", MingASQL)
    # print ("OrderTransSQL = ", OrderTransSQL)


    ## analyze single molecule files and remove components with column densities lower than user defined threshold
    AllMolfitsFile = open(MolfitsFileName, "w")


    ## write sql parameters
    MolfitHeaderLinesList = task_myXCLASS.WriteSQLParameter(MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)
    for line in MolfitHeaderLinesList:
        AllMolfitsFile.write(line)


    ## write molfit file
    AllMolfitsFileContens = []
    MoleculeDefFlag = "false"
    EmptyFlag = "false"
    CurrentCompMol = []
    for molfitFile in SingleMolMolfitFiles:


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze single molecule molfit file using myXCLASS routine
        MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(molfitFile)


        ## write parameter to new molfit file
        for MoleculeIndex in range(len(MoleculesInMolfitFile)):                             ## loop over all molecules in the molfit file
            LocalMolfitParameters = AllParameters[MoleculeIndex]                            ## get the parameters describing the current molecule


            ## check, if molecule is already included in the overall molfit file
            AlreadyIncludedFlag = False
            if (MoleculeIndex > 0):
                for ii in range(0, MoleculeIndex):
                    if (MoleculesInMolfitFile[ii] == MoleculesInMolfitFile[MoleculeIndex]):
                        AlreadyIncludedFlag = True
                        break
            if (not AlreadyIncludedFlag):                                                   ## continue here, if molecule is not already included


                ## write parameter for current molecule to xml file
                NewSingleMolMolfitFileContent = []
                for line in LocalMolfitParameters:                                          ## loop over all components of the current molecule


                    ## write each parameter for current molecule to xml file
                    MolfitFileFormatVersionLocal = ""
                    EmissionFlag = "T"                                                      ## define default setting for EmissionFlag
                    tdFlag = "F"                                                            ## define default setting for tdFlag
                    nHFlag = "F"                                                            ## define default setting for nHFlag
                    EAFlag = "f"
                    ColumnDensity = (-1.0)
                    NewLine = ""                                                            ## initialize new line
                    for param in line:                                                      ## loop over all parameters

                        # Debug:
                        # print ("param = ", param)


                        ParameterName = param[0].strip()                                    ## get name of parameter
                        LimitValues = param[1]                                              ## get fit-flag of parameter
                        lowlimit = param[2]                                                 ## get lower limit
                        uplimit = param[3]                                                  ## get upper limit
                        ParameterValue = param[4]                                           ## get value of parameter

                        # Debug:
                        # print ("param = ", param)


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## define molfit file format for each line
                        if (ParameterName == "MolfitFileFormatVersion"):
                            MolfitFileFormatVersionLocal = ParameterValue[:3]               ## get parameter indicating molfit format
                            EmissionFlag = ParameterValue[3:4]                              ## get parameter indicating if a absorption component
                                                                                            ## definition contains no source size setting
                            tdFlag = ParameterValue[6:7]                                    ## get parameter tbFlag indicating if background temperature
                                                                                            ## etc is given in molfit file
                            nHFlag = ParameterValue[9:10]                                   ## write parameter nHFlag indicating if column density etc
                                                                                            ## is given in molfit file
                        else:


                            ## get column density
                            if (ParameterName == "N_tot" or ParameterName == "EM_RRL" \
                                or ParameterName == "nHcolumn_cont_dust" or ParameterName == "nHcolumn"):
                                ColumnDensity = float(ParameterValue)


                            ## get absorption/emission flag
                            elif (ParameterName == "CFFlag" or ParameterName == "keyword"):
                                EAFlag = ParameterValue.lower().strip()


                            ##============================================================================================================================
                            ## construct new line of current single molecule molfit file


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## molfit file is in OLD format
                            LowerLimit = lowlimit
                            UpperLimit = uplimit
                            if (MolfitFileFormatVersionLocal == "OLD"):                     ## check, if current line has to be converted from OLD to NEW
                                                                                            ## format
                                if (ParameterName.endswith("_FitFlag")):
                                    ParameterValue = float(ParameterValue)
                                    if (ParameterValue == 0.0 or FixParameterFlag):         ## no range is defined, parameter is kept constant
                                        element = " n "
                                        lowlimit = 0.0
                                        uplimit = 0.0
                                    else:                                                   ## a range is defined, determine limits for parameter
                                        element = " y "
                                        lowlimit, uplimit = task_myXCLASS.GetParameterLimits(ParameterName, ParameterValue, LimitValues)
                                elif (ParameterName != "LayerDistance" and ParameterName != "ContFuncID_phen" and ParameterName != "CFFlag" \
                                    and ParameterName != "keyword"):


                                    ## define lower and upper limit
                                    l1 = lowlimit
                                    u1 = uplimit
                                    lowlimit = min(l1, u1)
                                    uplimit = max(l1, u1)


                                    ## check, if parameter value is located within parameter ranges defined by lower and upper limits
                                    value = float(ParameterValue)
                                    if (value <= lowlimit):
                                        if (value < 0.0):
                                            lowlimit = value * 1.15
                                        elif (value == 0.0):
                                            lowlimit = 0.0
                                        else:
                                            lowlimit = value * 0.85
                                    elif (value >= uplimit):
                                        if (value < 0.0):
                                            uplimit = value * 0.85
                                        elif (value == 0.0):
                                            uplimit = 0.0
                                        else:
                                            uplimit = value * 1.15


                                    ## define part of molfit file
                                    if (ParameterName.startswith("N_tot") or ParameterName.startswith("EM_RRL") \
                                        or ParameterName.startswith("nHcolumn_cont_dust") \
                                        or ParameterName.startswith("nHcolumn")):
                                        element += "{:12.3e} {:12.3e} {:20.10e}".format(lowlimit, uplimit, value)
                                    else:
                                        element += "{:12.3f} {:12.3f} {:20.10e}".format(lowlimit, uplimit, value)
                                else:
                                    element = ParameterValue


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## molfit file is in NEW format
                            else:
                                if (ParameterName == "LayerDistance" or ParameterName == "ContFuncID_phen" or ParameterName == "keyword" \
                                    or ParameterName == "CFFlag" or ParameterName.endswith("_flag") ):
                                    element = ParameterValue


                                    ## ++++++++++++++++++++++++++++++++++++++++++++++ TEST ++++++++++++++++++++++++++++++++++++++++++++++
                                    #    if (ParameterName.endswith("_flag") and (not ParameterName.startswith("source_size"))
                                    #        and MolfitsFileName.endswith("all.molfit")):
                                    #        element = "y"
                                    ##***************************************************************************************************

                                    if (ParameterName.endswith("_flag") and FixParameterFlag):
                                        element = "n"


                                elif (ParameterName.endswith("_uplimit")):
                                    UpperLimit = float(ParameterValue)
                                    element = ""
                                elif (ParameterName.endswith("_lowlimit")):
                                    LowerLimit = float(ParameterValue)
                                    element = ""
                                else:


                                    ## define lower and upper limit
                                    l1 = LowerLimit
                                    u1 = UpperLimit
                                    LowerLimit = min(l1, u1)
                                    UpperLimit = max(l1, u1)


                                    ## check, if parameter value is located within parameter ranges defined by lower and upper limits
                                    value = float(ParameterValue)
                                    if (value <= LowerLimit):
                                        if (value < 0.0):
                                            LowerLimit = value * 1.15
                                        elif (value == 0.0):
                                            LowerLimit = 0.0
                                        else:
                                            LowerLimit = value * 0.85
                                    elif (value >= UpperLimit):
                                        if (value < 0.0):
                                            UpperLimit = value * 0.85
                                        elif (value == 0.0):
                                            UpperLimit = 0.0
                                        else:
                                            UpperLimit = value * 1.15


                                    ## define part of molfit file
                                    if (ParameterName.startswith("N_tot") or ParameterName.startswith("EM_RRL") \
                                        or ParameterName.startswith("nHcolumn_cont_dust") or ParameterName.startswith("nHcolumn")):
                                        element = "{:12.3e} {:12.3e} {:20.10e}".format(LowerLimit, UpperLimit, value)
                                    else:
                                        element = "{:12.3f} {:12.3f} {:20.10e}".format(LowerLimit, UpperLimit, value)


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## create new line
                            NewLine += " " + element


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## if column density is above user defined limit
                    if (EAFlag == "f"):
                        if (ColumnDensity >= MinColumnDensityAbs or ColumnDensity < 0.0):
                            # AddBlanks = 51 * " "
                            # NewLine = AddBlanks + NewLine
                            NewSingleMolMolfitFileContent.append(NewLine)
                    else:
                        if (ColumnDensity >= MinColumnDensityEmis or ColumnDensity < 0.0):
                            NewSingleMolMolfitFileContent.append(NewLine)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## consider only molecules with at least one component
                if (len(NewSingleMolMolfitFileContent) > 0):


                    ## write modified contents of single molecule molfit file to overall molfit file
                    NewLine = MoleculesInMolfitFile[MoleculeIndex] + "   " + str(len(NewSingleMolMolfitFileContent)) + "\n"
                    AllMolfitsFile.write(NewLine)                                           ## write new line to file
                    for line in NewSingleMolMolfitFileContent:                              ## loop over all lines of the current single molecule file
                        AllMolfitsFile.write(line + "\n")                                   ## write line to new overall molfit file


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## close channel for overall molfit file
    AllMolfitsFile.write("\n\n")
    AllMolfitsFile.close()


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## include local python packages
##
def ExtendPackage(XCLASSRootDir, NameOfFunction):
    """

input parameters:
-----------------

    - XCLASSRootDir:            path and name of XCLASS root directory

    - NameOfFunction:           name of function


output parameters:
------------------

    - status:                   status variable
    """

    # Debug:
    # print ("XCLASSRootDir = ", XCLASSRootDir)
    # print ("NameOfFunction = ", NameOfFunction)


    ## initialize status variable
    status = 0


    ## check if additional packages can be imported
    ErrorList = ""
    ErrorCounter = 0
    try:
        from .lib import xclasslite
    except:
        ErrorCounter += 1
        if (ErrorCounter > 1):
            ErrorList += " and "
        ErrorList += "xclasslite"
    try:
        from .external.statcont import cont_finding
    except:
        ErrorCounter += 1
        if (ErrorCounter > 1):
            ErrorList += " and "
        ErrorList += "cont_finding"
    if (ErrorList != ""):
        status = 1
        print ("\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
        print ("\tCan not import " + ErrorList + " package(s)" + "!")
        print ("\n\tPlease check your XCLASS installation and try again!")


    ## we're done
    return status
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Line identification routine
##
def LineIdentificationCore(ObsXMLFileName = "",
                           ObsDataFileName = None,
                           experimentalData = None,
                           Noise = 2.0, MaxOverestimationHeight = 10.0, SourceName = "", DefaultMolfitFile = "", EstimateParamFlag = True, \
                           VelBin = 2.0, ElowMax = 1.e99, Tolerance = 10.0, TypeSelection = "all", SmoothValue = 0.0, MaxNumTransInFit = 1e99, \
                           SelectedMolecules = [], StrongMoleculeList = [], MinColumnDensityEmis = 1.e8, MinColumnDensityAbs = 1.e8, \
                           NumberIteration = 50, AlgorithmXMLFileEstParam = "", AlgorithmXMLFileISO = "", AlgorithmXMLFileSMF = "", \
                           AlgorithmXMLFileOverAll = "", vLSR = 0.0, TelescopeSize = None, BMIN = None, BMAJ = None, \
                           BPA = None, Redshift = 0.0, Inter_Flag = False, tBack = None, tSlope = None, N_H = None, beta_dust = None, \
                           kappa_1300 = None, DustFileName = "", BackgroundFileName = "", \
                           ContPhenFuncID = None, ContPhenFuncParam1 = None, ContPhenFuncParam2 = None, \
                           ContPhenFuncParam3 = None, ContPhenFuncParam4 = None, ContPhenFuncParam5 = None, \
                           NumModelPixelXX = None, NumModelPixelYY = None, LocalOverlapFlag = None, NoSubBeamFlag = False, EmAbsPATH = None, \
                           clusterdef = "", InternalParameterList = {}):
    """

The function starts the line identification routine and returns the identified lines with the corresponding best fit.


input parameters:
-----------------

    - Noise:                    defines noise level, (default: 2.0).

    - MaxOverestimationHeight:  defines the overestimation limit, (default: 200.0).

    - SourceName:               name of source molfit file, (default: "").

    - DefaultMolfitFile:        path and name of a default molfit file used for molecules
                                which are not described by the source templates, (default: "").

    - EstimateParamFlag:        defines, if the calculation parameters for a single molecule
                                fit are estimated directly from the given observational data
                                (default: False). Parameter estimation is used for all single
                                molecule fits, which use the default molfit file.

    - VelBin:                   (only used for parameter estimation, i.e. EstimateParamFlag = True):
                                velocity resolution for histogram analysis (default: 2.0).

    - ElowMax:                  (only used for parameter estimation, i.e. EstimateParamFlag = True):
                                upper limit of E_low (K) for transitions (default: 1.e99).

    - Tolerance:                fraction (%) of overestimated lines (default: 10.0).

    - TypeSelection:            (only used for parameter estimation, i.e. EstimateParamFlag = True):
                                select type of transition ("all": emission and absorption lines
                                are analyzed, "em": only emission lines are considered, "abs": only
                                absorption lines are taken into account. (default: "all").

    - SmoothValue:              smooth factor (=0, no smoothing is performed) used for smoothing
                                obs. data (default: 0.0).

    - MaxNumTransInFit:         max. number of transitions which are fitted in parameter estimation
                                algorithm (default: 1e99).

    - SelectedMolecules:        a (python) list containing the names of all molecules which
                                should be considered or a file name including the molecules
                                which should be selected.

                                Note, if the parameter defines a relative path, this path has
                                to be defined relative to the current working directory!

    - StrongMoleculeList:       list of strong (highly abundant) molecules which has to be
                                fitted before the other single molecule fits are started

    - MinColumnDensityEmis:     min. column density (for emission lines) of an optimized
                                component to be included in the overall molfit file (default: 1.e8).

    - MinColumnDensityAbs:      min. column density (for absorption lines) of an optimized
                                component to be included in the overall molfit file (default: 1.e8).

    - NumberIteration:          max. number of iterations (default: 50).

    - AlgorithmXMLFileEstParam: (only used for parameter estimation, i.e.
                                EstimateParamFlag = True): path and name of a MAGIX xml-file
                                including definitions for an algorithm or algorithm chain. This
                                xml file is used to estimate the molfit parameters of each velocity
                                component.
                                (A relative path has to be defined relative to the
                                current working directory!)

    - AlgorithmXMLFileISO:      (only used if iso ratio file is defined): path and name of a MAGIX
                                xml-file including definitions for an algorithm or algorithm chain.
                                This xml file is used to fit the iso ratios associated with an
                                intermediate identified molecule.

    - AlgorithmXMLFileSMF:      path and name of a MAGIX xml-file including definitions
                                for an algorithm or algorithm chain has to be given. This
                                xml file is used for the each single molecule fit. (A
                                relative path has to be defined relative to the current
                                working directory!)

                                NOTE, if the user specify a xml file, the number of
                                iterations given by the parameter "NumberIteration" is
                                ignored. The number of iteration is then given by the xml
                                file. In order to use the implemented fit algorithm
                                (Levenberg-Marquardt) clear the AlgorithmXMLFileSMF parameter,
                                i.e. AlgorithmXMLFileSMF = "", and define the max. number of
                                iterations by using parameter "NumberIteration".

    - AlgorithmXMLFileOverAll:  only necessary, if the user wants to use another fit
                                algorithm (than Levenberg-Marquardt) for fitting the final
                                molfit file including all identified molecules. Therefore,
                                the path and name of a MAGIX xml-file defining settings
                                for an algorithm or algorithm chain has to be given. (A
                                relative path has to be defined relative to the current
                                working directory!)

                                NOTE, if the user specify a xml file, the number of
                                iterations given by the parameter "NumberIteration"
                                is ignored. The number of iteration is then given by the
                                xml file. In order to use the implemented fit algorithm
                                (Levenberg-Marquardt) clear the AlgorithmXMLFileOverAll
                                parameter, i.e. AlgorithmXMLFileOverAll = "", and define
                                the max. number of iterations by using parameter
                                "NumberIteration".

    - ObsXMLFileName:           (optional) path and name of obs. xml file
                                (default: "")

    - ObsDataFileName:          (optional) path and name of an ASCII file called observational
                                data file, where the first column describe the frequency (in MHz)
                                and the second column the brightness temperature (in Kelvin),
                                (default: None)

    - experimentalData:         (optional) alias for ObsDataFileName (default: None)

    The following parameters are needed, if the parameter ObsDataFileName does NOT
    describe the path and name of a MAGIX xml-file:

    - vLSR:                     local standard of rest v_LSR (in km/s), (default: 0).

    - TelescopeSize:            size of telescope (in m), (default: 1).

    - BMIN:                     (optional) Beam minor axis length (default: None)

    - BMAJ:                     (optional) Beam major axis length (default: None)

    - BPA:                      (optional) Beam position angle (default: None)

    - Redshift:                 (optional) red shift (default 0.0).

    - Inter_Flag:               defines, if single dish ("False") or interferometric
                                observations ("True") are described, (default: "False").

    - tBack:                    background temperature (in K), (default: 0).

    - tSlope:                   temperature slope (dimensionless), (default: 0).

    - N_H:                      Hydrogen column density (in cm^{-2}), (default: not set).

    - beta_dust:                beta for dust (dimensionless), (default: not set).

    - kappa_1300:               kappa (cm^2 g^{-1}), (default: not set).

    - DustFileName:             path and name of file describing dust contribution
                                (default: "").

    - BackgroundFileName:       path and name of file describing background
                                (default: "").

    - ContPhenFuncID:           (optional) describes which phenomenological function is used
                                to describe the continuum (default: not set).

    - ContPhenFuncParam1:       (optional) first parameter for phenomenological function
                                describing the continuum (default: not set).

    - ContPhenFuncParam2:       (optional) second parameter for phenomenological function
                                describing the continuum (default: not set).

    - ContPhenFuncParam3:       (optional) third parameter for phenomenological function
                                describing the continuum (default: not set).

    - ContPhenFuncParam4:       (optional) fourth parameter for phenomenological function
                                describing the continuum (default: not set).

    - ContPhenFuncParam5:       (optional) fifth parameter for phenomenological function
                                describing the continuum (default: not set).

    - NumModelPixelXX:          (optional) used for sub-beam modeling, describes the number
                                of pixels used in x-direction (default: not set).

    - NumModelPixelYY:          (optional) used for sub-beam modeling, describes the number
                                of pixels used in y-direction (default: not set).

    - LocalOverlapFlag:         (optional) flag indicates if local-overlap description is
                                used or not (default: not set).

    - NoSubBeamFlag:            (optional) do not use sub-beam description (default: False).

    - EmAbsPATH:                (optional) path containing files describing functions
                                emission / absorption (default: None)

    - clusterdef:               file that contains cluster definition.


output parameters:
------------------

    - IdentifiedLines:          contains all information about the identified lines
                                (as an python array, e.g. IdentifiedLines[0] contains
                                the entries for the first molecule within the
                                frequency range)

    - JobDir:                   absolute path of the job directory created for the current run.



Example:
--------

Noise = 2.0
MaxOverestimationHeight = 200.0
SourceName = ""
DefaultMolfitFile = "demo/LineIdentification/default.molfit"
EstimateParamFlag = True
VelBin = 2.0
ElowMax = 1000.0
Tolerance = 10.0
TypeSelection = "all"
SmoothValue = 100.0
MaxNumTransInFit = 2000.0
SelectedMolecules = []
MinColumnDensityEmis = 0.0
MinColumnDensityAbs = 0.0
NumberIteration = 10
AlgorithmXMLFileEstParam = ""
AlgorithmXMLFileISO = ""
AlgorithmXMLFileSMF = ""
AlgorithmXMLFileOverAll = ""
experimentalData = "demo/LineIdentification/sma1_part-A.dat"
vLSR = 0.0
TelescopeSize = 200
Inter_Flag = False
tBack = 0.0
tSlope = 0.0
N_H = 3.0E+24
beta_dust = 2.0
kappa_1300 = 0.42
clusterdef = "demo/LineIdentification/clusterdef.txt"
IdentifiedLines, JobDir = LineIdentification()
    """

    # Debug:
    # print ("Noise = ", Noise)
    # print ("MaxOverestimationHeight = ", MaxOverestimationHeight)
    # print ("SourceName = ", SourceName)
    # print ("DefaultMolfitFile = ", DefaultMolfitFile)
    # print ("EstimateParamFlag = ", EstimateParamFlag)
    # print ("VelBin = ", VelBin)
    # print ("ElowMax = ", ElowMax)
    # print ("Tolerance = ", Tolerance)
    # print ("TypeSelection = ", TypeSelection)
    # print ("SmoothValue = ", SmoothValue)
    # print ("MaxNumTransInFit = ", MaxNumTransInFit)
    # print ("SelectedMolecules = ", SelectedMolecules)
    # print ("StrongMoleculeList = ", StrongMoleculeList)
    # print ("MinColumnDensityEmis = ", MinColumnDensityEmis)
    # print ("MinColumnDensityAbs = ", MinColumnDensityAbs)
    # print ("NumberIteration = ", NumberIteration)
    # print ("AlgorithmXMLFileEstParam = ", AlgorithmXMLFileEstParam)
    # print ("AlgorithmXMLFileISO = ", AlgorithmXMLFileISO)
    # print ("AlgorithmXMLFileSMF = ", AlgorithmXMLFileSMF)
    # print ("AlgorithmXMLFileOverAll = ", AlgorithmXMLFileOverAll)
    # print ("ObsXMLFileName = ", ObsXMLFileName)
    # print ("ObsDataFileName = ", ObsDataFileName)
    # print ("experimentalData = ", experimentalData)
    # print ("vLSR = ", vLSR)
    # print ("TelescopeSize = ", TelescopeSize)
    # print ("BMIN = ", BMIN)
    # print ("BMAJ = ", BMAJ)
    # print ("BPA = ", BPA)
    # print ("Redshift = ", Redshift)
    # print ("Inter_Flag = ", Inter_Flag)
    # print ("tBack = ", tBack)
    # print ("tSlope = ", tSlope)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("DustFileName = ", DustFileName)
    # print ("BackgroundFileName = ", BackgroundFileName)
    # print ("ContPhenFuncID = ", ContPhenFuncID)
    # print ("ContPhenFuncParam1 = ", ContPhenFuncParam1)
    # print ("ContPhenFuncParam2 = ", ContPhenFuncParam2)
    # print ("ContPhenFuncParam3 = ", ContPhenFuncParam3)
    # print ("ContPhenFuncParam4 = ", ContPhenFuncParam4)
    # print ("ContPhenFuncParam5 = ", ContPhenFuncParam5)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)
    # print ("EmAbsPATH = ", EmAbsPATH)
    # print ("clusterdef = ", clusterdef)
    # print ("InternalParameterList = ", InternalParameterList)


    ##====================================================================================================================================================
    ## to do:
    ##
    ##  - check, if source file contains Splatalogue names
    ##
    ##
    ##
    ##
    ##
    ##====================================================================================================================================================


    ##====================================================================================================================================================
    ## initialize function


    ## define name of function
    NameOfFunction = "LineIdentification"


    ## get current working directory
    CurrentDir = os.getcwd() + "/"


    ## define internal parameter
    CheckEmissionAbsorptionFlag = True                                                      ## check for emission/absorption flag
    RestFreq = 0.0                                                                          ## set frequency of rest to zero
    LineIDMethod = ""                                                                       ## define method for component detection
    DecisionMethod = "significant"
    # DecisionMethod = "strict"


    ## get path and name of default database file
    DefaultDBFileName = task_myXCLASS.GetDefaultDBFile()


    ##====================================================================================================================================================
    ## reset output variable
    BestMolfitFile = []
    JobDir = ""


    ##====================================================================================================================================================
    ## create run directory for the current function call
    myXCLASSRootDirectory = task_myXCLASS.GetmyXCLASSRootDir()                              ## get absolute path of myXCLASS root directory
    MAGIXrootDir = task_myXCLASS.GetMAGIXRootDir()                                          ## get absolute path of MAGIX directory
    LocalPrintFlag = True
    LineIDJobDir = task_MAGIX.CreateRunDirectory(NameOfFunction, myXCLASSRootDirectory, LocalPrintFlag)
    JobDir = LineIDJobDir

    # Debug:
    # print ("MAGIXrootDir = ", MAGIXrootDir)
    # print ("LineIDJobDir = ", LineIDJobDir)


    ##====================================================================================================================================================
    ## set up environment for MAGIX
    task_MAGIX.SetMAGIXEnvironment(MAGIXrootDir)


    ## extend sys.path variable to include path for new packages
    XCLASSRootDir = task_myXCLASS.GetXCLASSRootDir()
    # NewPathList = [XCLASSRootDir + "xclass/external/ModifiedGausspy/version_1.0.0/", \
    #                XCLASSRootDir + "xclass/src/lite/bin/"]
    # task_myXCLASS.ExtendSysPath(NewPathList)


    ##====================================================================================================================================================
    ## check user-defined parameters
    NoOverallFitFlag = GetDicKey(InternalParameterList, 'NoOverallFitFlag', False)          ## get max. overestimation factor for an identified component


    ## check, min. column densities for core and foreground components
    if (MinColumnDensityEmis < 0.0 or MinColumnDensityAbs < 0.0):
        print ("\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
        print ("\tThe user-defined min. column density for", end=' ')
        if (MinColumnDensityEmis < 0.0 and MinColumnDensityAbs >= 0.0):
            print ("core components", end=' ')
        elif (MinColumnDensityEmis >= 0.0 and MinColumnDensityAbs < 0.0):
            print ("foreground components", end=' ')
        elif (MinColumnDensityEmis < 0.0 and MinColumnDensityAbs < 0.0):
            print ("core and foreground components", end=' ')
        print ("is less than zero.")
        print ("\n\tMinColumnDensityEmis = ", MinColumnDensityEmis)
        print ("\tMinColumnDensityAbs = ", MinColumnDensityAbs)
        print ("\n\tPlease correct the definition(s) and restart {:s} function!\n".format(NameOfFunction))
        return (BestMolfitFile, JobDir)
    else:
        Min_Ntot_Emis = GetDicKey(InternalParameterList, 'Min_Ntot_Emis', 0.0)              ## get minimal column density for emission components
        if (Min_Ntot_Emis == 0.0):                                                          ## if value is not defined, use MinColumnDensityEmis
            InternalParameterList['Min_Ntot_Emis'] = MinColumnDensityEmis
        Min_Ntot_Abs = GetDicKey(InternalParameterList, 'Min_Ntot_Abs', 0.0)                ## get minimal column density for absorption components
        if (Min_Ntot_Abs == 0.0):                                                           ## if value is not defined, use MinColumnDensityAbs
            InternalParameterList['Min_Ntot_Abs'] = MinColumnDensityAbs
        OverEstim = GetDicKey(InternalParameterList, 'OverEstim', 0.0)                      ## get max. overestimation factor for an identified component
        if (OverEstim == 0.0):                                                              ## if factor is not defined, use MaxOverestimationHeight
            InternalParameterList['OverEstim'] = MaxOverestimationHeight / 100.0


    ## remove leading and trailing blanks in path definitions
    AlgorithmXMLFileEstParam = AlgorithmXMLFileEstParam.strip()
    AlgorithmXMLFileISO = AlgorithmXMLFileISO.strip()
    AlgorithmXMLFileSMF = AlgorithmXMLFileSMF.strip()
    AlgorithmXMLFileOverAll = AlgorithmXMLFileOverAll.strip()


    ##====================================================================================================================================================
    ## analyze input parameter regarding obs. data


    ## analyze input parameters and create obs. xml file if necessary
    ObsXMLFileNameOrig = task_myXCLASSFit.CheckObsInput(ObsXMLFileName, \
                                                        ObsDataFileName, \
                                                        experimentalData, \
                                                        NameOfFunction, \
                                                        CurrentDir, \
                                                        LineIDJobDir, \
                                                        Noise = Noise, \
                                                        tBack = tBack, \
                                                        tSlope = tSlope, \
                                                        N_H = N_H, \
                                                        beta_dust = beta_dust, \
                                                        kappa_1300 = kappa_1300, \
                                                        DustFileName = DustFileName, \
                                                        BackgroundFileName = BackgroundFileName, \
                                                        ContPhenFuncID = ContPhenFuncID, \
                                                        ContPhenFuncParam1 = ContPhenFuncParam1, \
                                                        ContPhenFuncParam2 = ContPhenFuncParam2, \
                                                        ContPhenFuncParam3 = ContPhenFuncParam3, \
                                                        ContPhenFuncParam4 = ContPhenFuncParam4, \
                                                        ContPhenFuncParam5 = ContPhenFuncParam5, \
                                                        TelescopeSize = TelescopeSize, \
                                                        BMIN = BMIN, \
                                                        BMAJ = BMAJ, \
                                                        BPA = BPA, \
                                                        Inter_Flag = Inter_Flag, \
                                                        vLSR = vLSR, \
                                                        Redshift = Redshift, \
                                                        DBFileName = DefaultDBFileName, \
                                                        NumModelPixelXX = NumModelPixelXX, \
                                                        NumModelPixelYY = NumModelPixelYY, \
                                                        LocalOverlapFlag = LocalOverlapFlag, \
                                                        NoSubBeamFlag = NoSubBeamFlag, \
                                                        EmAbsPATH = EmAbsPATH, \
                                                        iso_flag = True, \
                                                        IsoTableFileName = "", \
                                                        PrintFlag = True, \
                                                        CopyFlag = False)
    ## did an error occur ?
    if (ObsXMLFileNameOrig is None):
        return (BestMolfitFile, LineIDJobDir)


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## adjust paths in obs. xml file
    LocalPrintFlag = False
    ObsXMLFileNameOrig, IsoRatioFileName, dbFileName = task_myXCLASSFit.AdjustObsXMLFile(NameOfFunction, ObsXMLFileNameOrig, CurrentDir, \
                                                                                         LineIDJobDir, LocalPrintFlag)
    if (ObsXMLFileNameOrig == ""):                                                          ## did an error occurred
        return BestMolfitFile, LineIDJobDir

    # Debug:
    # print ("\n\nLocalObsXMLFileName = ", LocalObsXMLFileName)
    # print ("IsoRatioFileName = ", IsoRatioFileName)
    # print ("dbFileName = ", dbFileName)


    ##====================================================================================================================================================
    ## extract parameters from obs. xml file


    ## read parameters
    NumberExpFiles = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "NumberExpFiles")
    NumberExpFiles = int(NumberExpFiles[0])
    ListOfSpectraNames = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "FileNamesExpFiles")
    NumberExpRanges = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "NumberExpRanges")
    ListFreqMin = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "MinExpRange")
    ListFreqMax = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "MaxExpRange")
    ListFreqStep = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "StepFrequency")
    tBackList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "BackgroundTemperature")
    tSlopeList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "TemperatureSlope")
    BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "BackgroundFileName")
    nHList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "HydrogenColumnDensity")
    betaList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "DustBeta")
    kappaList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "Kappa")
    NoiseLevel = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "NoiseLevel")
    GlobalvLSRList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "GlobalvLSR")
    TelescopeSizeList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "TelescopeSize")
    RedshiftList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "Redshift")
    InterFlagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "Inter_Flag")
    NumberHeaderLinesXML = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "NumberHeaderLines")
    SeparatorColumnsXML = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "SeparatorColumns")

    # Debug:
    # print ("NumberExpFiles = ", NumberExpFiles)
    # print ("ListOfSpectraNames = ", ListOfSpectraNames)
    # print ("NumberExpRanges = ", NumberExpRanges)
    # print ("ListFreqMin = ", ListFreqMin)
    # print ("ListFreqMax = ", ListFreqMax)
    # print ("ListFreqStep = ", ListFreqStep)
    # print ("tBackList = ", tBackList)
    # print ("tSlopeList = ", tSlopeList)
    # print ("BackgroundFileNameList = ", BackgroundFileNameList)
    # print ("nHList = ", nHList)
    # print ("betaList = ", betaList)
    # print ("kappaList = ", kappaList)
    # print ("NoiseLevelList = ", NoiseLevelList)
    # print ("GlobalvLSRList = ", GlobalvLSRList)
    # print ("RedshiftList = ", RedshiftList)
    # print ("TelescopeSizeList = ", TelescopeSizeList)
    # print ("InterFlagList = ", InterFlagList)
    # print ("NumberHeaderLinesXML = ", NumberHeaderLinesXML)
    # print ("SeparatorColumnsXML = ", SeparatorColumnsXML)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define path and name of used database file and determine Splatalogue translation table
    ListPathFileNameDB = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "dbFilename")
    if (len(ListPathFileNameDB) > 0):
        PathFileNameDB = ListPathFileNameDB[0].strip()
        if (PathFileNameDB.strip() != ""):
            if (PathFileNameDB[0] != "/"):                                              ## make relative path absolute
                PathFileNameDB = CurrentDir + PathFileNameDB
            if not(os.path.exists(PathFileNameDB) or os.path.isfile(PathFileNameDB)):
                print ("\n\n\tError in XCLASS package, function {:s}:".format(NameOfFunction))
                print ("\t\tThe given path and name of the database file " + PathFileNameDB)
                print ("\t\tdoes not exist!")
                print ("\n\t\tPlease note, if a relative path is specified, the path has to be defined relative to the")
                print ("\t\tcurrent working directory!")
                print ("\n\t\tPlease enter an existing path and name of iso file and redo LineIdentification function call!")
                return (BestMolfitFile, LineIDJobDir)
    else:
        PathFileNameDB = DefaultDBFileName
    dbList = [PathFileNameDB]


    ## get Splatalgoue translation table from database file
    SplatFlag, LocalMoleculeNameTranslationTable = task_myXCLASSFit.GetSplatTranslationTable(dbList[0])
    if (SplatFlag):
        ListVAMDCNames, ListSplatNames = list(zip(*LocalMoleculeNameTranslationTable))
    else:
        ListVAMDCNames = copy.deepcopy(LocalMoleculeNameTranslationTable)
        ListSplatNames = []

    # Debug:
    # print ("ListPathFileNameDB = ", ListPathFileNameDB)
    # print ("len(ListPathFileNameDB) = ", len(ListPathFileNameDB))
    # print ("PathFileNameDB = ", PathFileNameDB)
    # print ("SplatFlag = ", SplatFlag)
    # print ("ListVAMDCNames = ", ListVAMDCNames)
    # print ("ListSplatNames = ", ListSplatNames)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define flag indicating if iso ratio file is used and determine isotopologues defined in iso ratio file


    ## get flag for usage of iso ratio file
    IsoFlag = False
    IsoFlagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "Isotopologues")
    if (IsoFlagList == []):
        IsoFlagList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "iso_flag")
    if (len(IsoFlagList) > 0):
        LocalIsoFlag = IsoFlagList[0].lower()
        LocalIsoFlag = task_myXCLASSMapFit.CheckBool(LocalIsoFlag)
        if (LocalIsoFlag):
            IsoFlag = True


    ## get path and name of iso ratio file (if usage is indicated)
    IsoRatioTable = []
    Isotopologues  = []
    IsoMolecule = []
    IsoRatioFileName = ""
    if (IsoFlag):
        IsoRatioFileNameList = task_MAGIX.GetXMLtagNEW(ObsXMLFileNameOrig, "IsoTableFileName")
        if (IsoRatioFileNameList != []):
            IsoRatioFileName = IsoRatioFileNameList[0].strip()
            if (IsoRatioFileName != ""):


                ## analyze iso ratio file parameter: check path and name of the iso ratio file and copy new formatted iso ratio file to job directory
                ok, IsoRatioFileName = task_myXCLASSFit.CheckIsoRatioFile(IsoRatioFileName, NameOfFunction, CurrentDir, LineIDJobDir, \
                                                                          ListFreqMin, ListFreqMax, GlobalvLSRList, RedshiftList, dbList, "")
                if (ok == 1):                                                               ## an error occurred
                    return (BestMolfitFile, LineIDJobDir)


                ## get all isotopologues
                NewLocalIsoRatioFileName = ""
                IsoRatioTable, Isotopologues, IsoMolecule = task_myXCLASS.ImportIsoRatioFile(IsoRatioFileName, NewLocalIsoRatioFileName)

    # Debug:
    # print ("IsoFlag = ", IsoFlag)
    # print ("IsoRatioFileName = ", IsoRatioFileName)
    # print ("IsoRatioTable = ", IsoRatioTable)
    # print ("Isotopologues = ", Isotopologues)
    # print ("IsoMolecule = ", IsoMolecule)


    ##====================================================================================================================================================
    ## analyze default molfit file: translate Splatalogue names and get sql parameters
    DefaultMolfitFile = DefaultMolfitFile.strip()
    MinNumTransitionsSQL = 1
    MaxNumTransitionsSQL = 1.e99
    MaxElowSQL = 1.e6
    MingASQL = 1.e-99
    OrderTransSQL = 1
    if (DefaultMolfitFile != ""):


        ## if necessary, make relative path absolute
        if (DefaultMolfitFile[0] != "/"):
            LocalDefaultMolfitFile = CurrentDir + DefaultMolfitFile
        else:
            LocalDefaultMolfitFile = DefaultMolfitFile


        ## check existence of obs. data file and copy obs. data file to current working directory
        LocalPrintFlag = False
        NameOfFile = "default molfit"
        LocalDefaultMolfitFile = task_myXCLASS.CheckNCopy(DefaultMolfitFile, NameOfFunction, NameOfFile, CurrentDir, LineIDJobDir, LocalPrintFlag)
        if (LocalDefaultMolfitFile == ""):                                                  ## did an error occurred
            return (BestMolfitFile, LineIDJobDir)
        else:
            DefaultMolfitFile = LocalDefaultMolfitFile

        # Debug:
        # print ("LocalDefaultMolfitFile = ", LocalDefaultMolfitFile)


        ## check molecules in database (replace Splatalogue names)
        dbFileName = dbList[0]
        LocalPrintFlag = True
        task_myXCLASSFit.MolfitTranslateSplatalogueNames(DefaultMolfitFile, dbFileName, LocalPrintFlag)


        ## get sql parameters
        MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL = task_myXCLASS.GetSQLParameter(LocalDefaultMolfitFile)
        MaxNumTransInFit = MaxNumTransitionsSQL


    ## define SQL parameter array
    SQLParamArray = [MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL]

    # Debug:
    # print ("MinNumTransitionsSQL = ", MinNumTransitionsSQL)
    # print ("MaxNumTransitionsSQL = ", MaxNumTransitionsSQL)
    # print ("MaxElowSQL = ", MaxElowSQL)
    # print ("MingASQL = ", MingASQL)
    # print ("OrderTransSQL = ", OrderTransSQL)
    # print ("SQLParamArray = ", SQLParamArray)


    ##====================================================================================================================================================
    ## check, if Splatalogue names are used in SelectedMolecules and StrongMoleculeList
    if (SplatFlag):


        ## search for Splatalogue names in list 'SelectedMolecules'
        NewSelectedMolecules = []
        for LocalSelectedMolecules in SelectedMolecules:
            if (LocalSelectedMolecules in ListVAMDCNames):
                NewSelectedMolecules.append(LocalSelectedMolecules)
            elif (LocalSelectedMolecules in ListSplatNames):
                i = ListSplatNames.index(LocalSelectedMolecules)
                NewSelectedMolecules.append(ListVAMDCNames[i])
        SelectedMolecules = copy.deepcopy(NewSelectedMolecules)


        ## search for Splatalogue names in list 'StrongMoleculeList'
        NewStrongMoleculeList = []
        for LocalStrongMolecule in StrongMoleculeList:
            if (not LocalStrongMolecule in ListVAMDCNames):
                NewStrongMoleculeList.append(LocalStrongMolecule)
            elif (LocalStrongMolecule in ListSplatNames):
                i = ListSplatNames.index(LocalStrongMolecule)
                NewStrongMoleculeList.append(ListVAMDCNames[i])
        StrongMoleculeList = copy.deepcopy(NewStrongMoleculeList)

        # Debug:
        # print ("SelectedMolecules = ", SelectedMolecules)
        # print ("StrongMoleculeList = ", StrongMoleculeList)


    ##====================================================================================================================================================
    ## import spectra from obs. data file(s)
    ListOfSpectra = []
    ListSubtractedSpectra = []
    ListNoiseLevels = []
    ListOfAllFreqMin = []
    ListOfAllFreqMax = []
    ListOfAllFreqStep = []
    ListOfAllGlobalvLSR = []
    ListOfAllRedshift = []
    TotalCounterMol = 0
    for ObsDataFileIndex in range(NumberExpFiles):                                          ## loop over all spectra in the experimentalData directory


        ## get path and name of obs. data file
        LocalObsDataFileName = ListOfSpectraNames[ObsDataFileIndex][1]


        ## get number of header lines and character separating columns from obs. xml file
        RangeIndex = (-1)
        ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberHeaderLinesListIn = NumberHeaderLinesXML, \
                                                                       SeparatorColumnsListIn = SeparatorColumnsXML, GlobalvLSRListIn = GlobalvLSRList, \
                                                                       NumberRangeListIn = NumberExpRanges, Redshift_ListIn = RedshiftList)
        NumberHeaderLines = ObsXMLParameterDictFile['NumberHeaderLines']
        SeparatorColumns = ObsXMLParameterDictFile['SeparatorColumns']
        GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
        if (GlobalvLSR is None):
            GlobalvLSR = 0.0
        Redshift = ObsXMLParameterDictFile['Redshift']
        if (Redshift is None):
            Redshift = 0.0


        ## import obs. data file
        if (SeparatorColumns.strip() == ""):
            ImportObsData = numpy.loadtxt(LocalObsDataFileName, skiprows = NumberHeaderLines)
        else:
            ImportObsData = numpy.loadtxt(LocalObsDataFileName, skiprows = NumberHeaderLines, delimiter = SeparatorColumns)


        ## check length of import obs. data file
        if (len(ImportObsData[:,0]) == 1):                                                  ## data has just one data point
            print ("\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
            print ("\tThe given experimental data is an array, but it contains only one data point!")
            print ("\n\tPlease correct your input and redo LineIdentification function call!")
            return (BestMolfitFile, LineIDJobDir)


        ## make a copy of obs. spectrum array and add further column to avoid multiple continuum subtraction
        ObsSpectrumLocal = numpy.zeros((len(ImportObsData), 3))
        ObsSpectrumLocal[:, 0] = ImportObsData[:, 0]
        ObsSpectrumLocal[:, 1] = ImportObsData[:, 1]
        ObsSpectrumLocal[:, 2] = 0.0
        LocalLowestDataFreq = min(ImportObsData[:, 0])
        LocalHighestDataFreq = max(ImportObsData[:, 0])
        FreqArray = ImportObsData[:, 0]
        ListOfSpectra.append(ObsSpectrumLocal)

        # Debug:
        # print ("len(ListFreqMin) = ", len(ListFreqMin))
        # print ("ObsDataFileIndex = ", ObsDataFileIndex)
        # print ("RangeIndex = ", RangeIndex)
        # print ("ListFreqMin = ", ListFreqMin)
        # print ("ListFreqMax = ", ListFreqMax)
        # print ("ListFreqStep = ", ListFreqStep)
        # print ("tBackList = ", tBackList)
        # print ("tSlopeList = ", tSlopeList)
        # print ("NoiseLevel = ", NoiseLevel)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get parameters for each frequency range
        NumberFrequencyRanges = ObsXMLParameterDictFile['NumberFrequencyRanges']             ## get number of frequency ranges for current obs. data file
        for RangeIndex in range(NumberFrequencyRanges):                                     ## loop over all range definitions in the whole xml file


            ## get parameter for current frequency range
            ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = ListFreqMin, \
                                                                            FreqMaxListIn = ListFreqMax, FreqStepListIn = ListFreqStep, \
                                                                            tBackListIn = tBackList, tSlopeListIn = tSlopeList, \
                                                                            BackgroundFileNameListIn = BackgroundFileNameList, \
                                                                            NoiseListIn = NoiseLevel)
            FreqMin = ObsXMLParameterDictRange['FreqMin']
            if (FreqMin is not None):
                ListOfAllFreqMin.append(FreqMin)
                FreqMax = ObsXMLParameterDictRange['FreqMax']
                ListOfAllFreqMax.append(FreqMax)
                FreqStep = ObsXMLParameterDictRange['FreqStep']
                ListOfAllFreqStep.append(FreqStep)
                tBack = ObsXMLParameterDictRange['tBack']
                tSlope = ObsXMLParameterDictRange['tSlope']
                BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                if (BackgroundFileName is None):
                    BackgroundFileName = ""
                BackgroundFileName = BackgroundFileName.strip()
                LocalNoiseLevel = ObsXMLParameterDictRange['NoiseLevel']
                if (LocalNoiseLevel is None):
                    LocalNoiseLevel = 0.0
                ListNoiseLevels.append(LocalNoiseLevel)
                ListOfAllGlobalvLSR.append(GlobalvLSR)
                ListOfAllRedshift.append(Redshift)

                # Debug:
                # print ("\n\n\nObsDataFileIndex = ", ObsDataFileIndex)
                # print ("RangeIndex = ", RangeIndex)
                # print ("FreqMin = ", FreqMin)
                # print ("FreqMax = ", FreqMax)
                # print ("FreqStep = ", FreqStep)
                # print ("tBack = ", tBack)
                # print ("tSlope = ", tSlope)
                # print ("LocalNoiseLevel = ", LocalNoiseLevel)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## subtract continuum, (IMPORTANT: v_LSR is not relevant here)


                ## subtract background spectrum
                if (BackgroundFileName != ""):
                    BackgroundFile = ImportASCIIFile(BackgroundFileName)                    ## import interpolation object
                    LocalBackgroundSpectrum = BackgroundFile(FreqArray)                     ## compute background spectrum at given frequencies
                    FreqMinIndex = max(0, (numpy.abs(FreqArray - FreqMin)).argmin() - 1)
                    FreqMaxIndex = min(len(FreqArray) - 1, (numpy.abs(FreqArray - FreqMax)).argmin() + 1)
                    for i in range(FreqMinIndex, FreqMaxIndex):                             ## subtract background spectrum
                        if (ObsSpectrumLocal[i, 2] == 0.0):
                            ObsSpectrumLocal[i, 2] = 1.0                                    ## avoid multiple continuum subtraction
                            ObsSpectrumLocal[i, 1] = ObsSpectrumLocal[i, 1] - LocalBackgroundSpectrum[i]


                ## subtract phenom. continuum description from obs. data
                elif (tBack is not None and tBack != 0.0 and tSlope is not None):                   ## continue, if background temperature is unequal zero
                    FreqMinIndex = max(0, (numpy.abs(FreqArray - FreqMin)).argmin() - 1)
                    FreqMaxIndex = min(len(FreqArray) - 1, (numpy.abs(FreqArray - FreqMax)).argmin() + 1)
                    for i in range(FreqMinIndex, FreqMaxIndex):
                        if (ObsSpectrumLocal[i, 2] == 0.0):
                            ObsSpectrumLocal[i, 2] = 1.0                                    ## avoid multiple continuum subtraction
                            ObsSpectrumLocal[i, 1] = ObsSpectrumLocal[i, 1] - (abs(tBack) * (ObsSpectrumLocal[i, 0] / FreqMin)**tSlope)


        ## store continuum subtracted spectra
        ImportObsData[:, 0] = ObsSpectrumLocal[:, 0]
        ImportObsData[:, 1] = ObsSpectrumLocal[:, 1]
        ListSubtractedSpectra.append(ImportObsData)

    # Debug:
    # print ("ListOfAllFreqMin = ", ListOfAllFreqMin)
    # print ("ListOfAllFreqMax = ", ListOfAllFreqMax)


    ##====================================================================================================================================================
    ## determine all molecules for single molecule fits


    ## determine all molecules (and their molecular data) which have at least one transition within the user-defined frequency ranges
    OutputArray = ReadDatabase(PathFileNameDB, ListOfAllFreqMin, ListOfAllFreqMax, ListOfAllGlobalvLSR, ListOfAllRedshift, SelectedMolecules, \
                               Isotopologues, IsoMolecule, MinNumTransitionsSQL, MaxNumTransitionsSQL, MaxElowSQL, MingASQL, OrderTransSQL)
    ok = OutputArray[0]
    TotalCounterMol = OutputArray[1]
    if (ok == 1 or TotalCounterMol == 0):                                                   ## no molecule is found within range
        print ("\n\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
        print ("\tFound no molecule with transitions within the given frequency range(s)!\n\n")
        print ("\n\tAbort LineID function!\n\n\n")
        return (BestMolfitFile, LineIDJobDir)
    else:
        MoleculeData = OutputArray[2]
        ListTotalNumTransPerMol = OutputArray[3]
        LineCatalog = OutputArray[4]
        InternalParameterList['LineCatalog'] = LineCatalog


    ## create list with molecules without duplicates
    ListOfMolecules = []
    for LocalMoleculeData in MoleculeData:
        LocalMoleculeName = LocalMoleculeData[0]
        if (LocalMoleculeName not in ListOfMolecules):
            ListOfMolecules.append(LocalMoleculeName)
    ListOfMolecules.sort()                                                                  ## sort list of molecule names


    ## print list of all molecules with the total number of transitions within the selected range(s)
    # print ("\n\nConsider the following molecules (with total number of transitions):")
    #    ListTotalNumTransPerMol = sorted(ListTotalNumTransPerMol, key=lambda l:l[1], reverse=True)
    # print ("\n\n{:<40s}  Total number of transitions:".format("Molecule:"))
    #    for entry in ListTotalNumTransPerMol:
    #        print "{:<40s}  {:5d}".format(entry[0], entry[1])
    # print ("\n\n")

    # Debug:
    # print ("\n\n\n\n\n\n\n\n\n\n")
    #    for mol in ListOfMolecules:
    #        print mol
    #    sys.exit(0)


    ##====================================================================================================================================================
    ## prepare single molecule fits


    ## create directory for single molecule fits
    print ("\n\nCreating directory for single molecule fits ..", end=' ')
    LineIDSingleMoleculeFitsDir = LineIDJobDir + "single-molecule_fits/"
    command_string = "mkdir -p " + LineIDSingleMoleculeFitsDir
    os.system(command_string)
    # print ("done!")


    ## create subdirectory for identified molecules
    IdentifiedMoleculesDir = LineIDJobDir + "Intermediate_identified_molecules/"
    command_string = "mkdir -p " + IdentifiedMoleculesDir
    os.system(command_string)


    ## create subdirectory for non-identified molecules
    NotIdentifiedMoleculesDir = LineIDJobDir + "Not_identified_molecules/"
    command_string = "mkdir -p " + NotIdentifiedMoleculesDir
    os.system(command_string)


    ## create subdirectory for velocities and summed spectra
    SummaryDir = LineIDJobDir + "combined-molecules/"
    command_string = "mkdir -p " + SummaryDir
    os.system(command_string)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze clusterdef file
    NumberProcessors, ServerList, ClusterFlag = task_myXCLASSMapFit.ReadClusterdefFile(clusterdef, LineIDJobDir)
    if (NumberProcessors > 1 and len(ListOfMolecules) == 1):
        NumberProcessors = 1
        ClusterFlag = "false"


    ## modify ServerList, so that ServerList has as many entries as workers, i.e. one entry with settings for each worker
    OldServerList = ServerList
    ServerList = []
    for server in OldServerList:                                                            ## loop over all server in the cluster
        Node = server[0]                                                                    ## get name of node
        NumCores = server[1]                                                                ## get number of cores
        RootPath = server[2].strip()                                                        ## get path of xclass interface
        if (RootPath != ""):                                                                ## if path of xclass interface is defined ..
            RootPath = RootPath + "/addons/MAGIX/"                                          ## .. define path of MAGIX
        for core in range(NumCores):                                                        ## loop over all cores of the current node
            ServerList.append([Node, NumCores, RootPath])


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## open file for including the results of the single molecule fits
    ResultsFile = open(LineIDJobDir + "LineID.log", 'w')
    ResultsFile.write("Results of the line identification:\n")
    lt = time.localtime()
    ResultsFile.write("\n\nStart LineID at date: " + time.strftime("%d-%m-%Y", lt) + ",  time: " + time.strftime("%H:%M:%S", lt) + "\n")
    ResultsFile.write("\n\n\nCurrent directory = " + CurrentDir + "\n")
    ResultsFile.write("\n\n\nInput parameter:\n\n")
    ResultsFile.write("SourceName = " + SourceName + "\n")
    ResultsFile.write("DefaultMolfitFile = " + DefaultMolfitFile + "\n")
    if (AlgorithmXMLFileSMF == ""):
        ResultsFile.write("NumberIteration = " + str(NumberIteration) + "\n")
    else:
        ResultsFile.write("AlgorithmXMLFileSMF = " + AlgorithmXMLFileSMF + "\n")
        ResultsFile.write("AlgorithmXMLFileOverAll = " + AlgorithmXMLFileOverAll + "\n")
    ResultsFile.write("\n\n\nConsider the following molecules:\n\n")
    i = 0
    for molecule in ListOfMolecules:
        i += 1
        ResultsFile.write("Molecule {:d}:   {:s}\n".format(i, molecule))
    ResultsFile.write("\n\n")
    ResultsFile.flush()

    # Debug:
    #    ResultsFile.write("Noise = " + str(Noise) + "\n")
    #    ResultsFile.write("FreqMin = " + str(FreqMin) + "\n")
    #    ResultsFile.write("FreqMax = " + str(FreqMax) + "\n")
    #    ResultsFile.write("FreqStep = " + str(FreqStep) + "\n")
    #    ResultsFile.write("TelescopeSize = " + str(TelescopeSize) + "\n")
    #    ResultsFile.write("InterFlag = " + str(InterFlag) + "\n")
    #    ResultsFile.write("tBack = " + str(tBack) + "\n")
    #    ResultsFile.write("tSlope = " + str(tSlope) + "\n")


    ## open file for identified molecules
    IdentifiedMoleculeFile = open(LineIDJobDir + "Identified_Molecules.dat", 'w')
    IdentifiedMoleculeFile.write("Identified molecules:\n")
    lt = time.localtime()
    IdentifiedMoleculeFile.write("\n\nStart LineID at date: " + time.strftime("%d-%m-%Y", lt) + ",  time: " + time.strftime("%H:%M:%S", lt) \
                                 + "\n\n\n\n")
    IdentifiedMoleculeFile.flush()
    print ("\n")


    ##====================================================================================================================================================
    ## start single molecule fits
    ListMolDirName = []
    MoleculejobDirList = []
    DefaultMolfitUsedFlagList = []
    NumberOfSMF = len(ListOfMolecules)
    for counterMolecule, molecule in enumerate(ListOfMolecules):                            ## loop over all molecules in the freq. range

        # Debug:
        # print ("counterMolecule, molecule = ", counterMolecule, molecule)


        ## print what you do
        print ("\rPrepare single molecule fit (" + str(counterMolecule + 1) + "/" + str(NumberOfSMF) + ") for molecule:  " + molecule \
                                                + " ..                                                                         ", end = ' ')

        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define name of subdirectory for current molecule, remove not allowed characters from the molecule name
        MolDirName = MoleculeFileName(molecule)
        ListMolDirName.append(MolDirName)


        ## create single molecule fit directory
        MoleculejobDir = LineIDSingleMoleculeFitsDir + MolDirName + "/"
        RelativeMoleculejobDir = MoleculejobDir.replace(LineIDJobDir, "")
        MoleculejobDirList.append(RelativeMoleculejobDir)
        command_string = "mkdir -p " + MoleculejobDir
        os.system(command_string)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## adjust paths in obs. xml file
        #        LocalPrintFlag = False
        #        LocalObsXMLFileName, IsoRatioFileName, dbFileName = task_myXCLASSFit.AdjustObsXMLFile(NameOfFunction, ObsXMLFileNameOrig, CurrentDir, \
        #                                                                                              MoleculejobDir, LocalPrintFlag)
        #        if (LocalObsXMLFileName == ""):                                                     ## did an error occurred
        #            return (BestMolfitFile, LineIDJobDir)

        #        # Debug:
        #        print "\n\nLocalObsXMLFileName = ", LocalObsXMLFileName
        #        print "IsoRatioFileName = ", IsoRatioFileName
        #        print "dbFileName = ", dbFileName


        #        ## copy created xml file
        #        UnmodifiedExpXMLFilename = MoleculejobDir + "exp.xml"
        #        if (LocalObsXMLFileName != UnmodifiedExpXMLFilename):
        #            cmdString = "mv " + LocalObsXMLFileName + " " + UnmodifiedExpXMLFilename
        #            os.system(cmdString)


        dbDefaultFilename = task_myXCLASS.GetDefaultDBFile()
        UnmodifiedExpXMLFilename = MoleculejobDir + "exp.xml"
        AdjustExpXMLFile(ObsXMLFileNameOrig, MoleculejobDir, CurrentDir, dbDefaultFilename)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create molfit file for the current molecule
        MolfitsFileName = MoleculejobDir + MolDirName + ".molfit"
        DefaultMolfitUsedFlag = CreateMolfitFile(CurrentDir, myXCLASSRootDirectory, MoleculejobDir, MolfitsFileName, molecule, SourceName, \
                                                 DefaultMolfitFile)

        ## check EstimateParamFlag
        if (not EstimateParamFlag):
            DefaultMolfitUsedFlag = False
        DefaultMolfitUsedFlagList.append(DefaultMolfitUsedFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## shrink freq. ranges to reduce computational effort
        if (not DefaultMolfitUsedFlag):
            ShrinkFreqRanges(molecule, MolDirName, UnmodifiedExpXMLFilename, MolfitsFileName, SQLParamArray)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## creates fit control xml-file
        AlgorithmXMLFileLocal = MoleculejobDir + "algorithm_control.xml"
        AlgorithmXMLFileSMF = AlgorithmXMLFileSMF.strip()
        if (AlgorithmXMLFileSMF != ""):


            ## check existence of algorithm xml file and copy algorithm xml file to current working directory
            LocalPrintFlag = False
            NameOfFile = "single-molecule-fit algorithm xml"
            LocalAlgorithmXMLFileSMF = task_myXCLASS.CheckNCopy(AlgorithmXMLFileSMF, NameOfFunction, NameOfFile, CurrentDir, MoleculejobDir, \
                                                                LocalPrintFlag)
            if (LocalAlgorithmXMLFileSMF == ""):                                            ## did an error occurred
                return (BestMolfitFile, LineIDJobDir)

            # Debug:
            # print ("LocalAlgorithmXMLFileSMF = ", LocalAlgorithmXMLFileSMF)


            ## rename algorithm xml
            task_myXCLASS.NormNCopyMove(LocalAlgorithmXMLFileSMF, AlgorithmXMLFileLocal, copyFlag = False)
        else:
            task_myXCLASSFit.CreateLMControlXMLFile(MoleculejobDir, NumberIteration)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## creates i/o control xml file
        task_myXCLASSFit.CreateIOControlXMLFile(MoleculejobDir, UnmodifiedExpXMLFilename, MolfitsFileName, MAGIXrootDir)

    # Debug:
    # print ("DefaultMolfitUsedFlagList = ", DefaultMolfitUsedFlagList)
    # print ("\n\nStrongMoleculeList = ", StrongMoleculeList)
    # print ("ListOfMolecules = ", ListOfMolecules)
    # print ("\n\n\n\n\n\n")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## do single molecule fits for strong line first
    StrongMolContribution, NewListOfMolecules = StrongLinesFits(StrongMoleculeList, ListOfMolecules, myXCLASSRootDirectory, MAGIXrootDir, LineIDJobDir, \
                                                                MoleculejobDirList, ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, \
                                                                MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, ListNoiseLevels, \
                                                                CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, \
                                                                ListMolDirName, ObsXMLFileNameOrig, UnmodifiedExpXMLFilename, \
                                                                DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, \
                                                                SQLParamArray, DecisionMethod, Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit, \
                                                                InternalParameterList)
    if (NewListOfMolecules != []):
        ListOfMolecules = NewListOfMolecules

    # Debug:
    # print ("\n\n\n\n\n\nListOfMolecules = ", ListOfMolecules)
    # print ("NewListOfMolecules = ", NewListOfMolecules)
    # print ("\n\n\n\n\n\n")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start different molecule fits on the cluster
    StartClusterFits(ServerList, ListOfMolecules, StrongMolContribution, myXCLASSRootDirectory, MAGIXrootDir, LineIDJobDir, MoleculejobDirList, \
                     ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, \
                     ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, ListMolDirName, \
                     ObsXMLFileNameOrig, DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, SQLParamArray, \
                     DecisionMethod, Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit, InternalParameterList)


    ## close output files
    lt = time.localtime()
    ResultsFile.write("\n\nEnd LineID at date: " + time.strftime("%d-%m-%Y", lt) + ",  time: " + time.strftime("%H:%M:%S", lt) + "\n")
    ResultsFile.close()
    IdentifiedMoleculeFile.write("\n\nEnd LineID at date: " + time.strftime("%d-%m-%Y", lt) + ",  time: " + time.strftime("%H:%M:%S", lt) + "\n")
    IdentifiedMoleculeFile.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create plots for total KDE and summed spectra
    CreateSummedPlots(LineIDJobDir)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in all molfit files of intermediate identified molecules
    listing = os.listdir(IdentifiedMoleculesDir)
    SingleMolMolfitFiles = []
    SingleMolIsoRatioFiles = []
    StrongMolFileList = []
    StrongIsoFileList = []


    ## make sure, that strong molecules (if defined) are read first
    if (StrongMoleculeList != []):
        for StrongMol in StrongMoleculeList:
            StrongMolFileName = MoleculeFileName(StrongMol)                                 ## get file name of current molecule
            for files in listing:


                ## molfit files
                if (files == StrongMolFileName + ".out.molfit"):
                    StrongMolFileList.append(files)


                    ## read in molfit file and add to list of all single molecule
                    LocalMolfitFile = open(IdentifiedMoleculesDir + "/" + files)
                    LocalMolfitFileContent = LocalMolfitFile.readlines()
                    LocalMolfitFile.close()
                    SingleMolMolfitFiles.append(LocalMolfitFileContent)


                ## iso ratio files
                if (files == StrongMolFileName + ".out.iso"):
                    StrongIsoFileList.append(files)


                    ## read in iso ratio file and add to list of all single molecule
                    LocalIsoRatioFile = open(IdentifiedMoleculesDir + "/" + files)
                    LocalIsoRatioFileContent = LocalIsoRatioFile.readlines()
                    LocalIsoRatioFile.close()
                    for line in LocalIsoRatioFileContent:
                        SingleMolIsoRatioFiles.append(line)


    ## read molfit files for all other molecules
    for files in listing:
        if (files.endswith(".out.molfit") or files.endswith(".out.iso")):


            ## do not add strong molecules again
            AddFlag = "true"
            if (StrongMoleculeList != []):
                if (files in StrongMolFileList):
                    AddFlag = "false"


            ## read in molfit/iso file and add to list of all single molecule
            if (AddFlag == "true"):
                if (files.endswith(".out.molfit")):
                    LocalMolfitFile = open(IdentifiedMoleculesDir + "/" + files)
                    LocalMolfitFileContent = LocalMolfitFile.readlines()
                    LocalMolfitFile.close()
                    SingleMolMolfitFiles.append(LocalMolfitFileContent)
                elif (files.endswith(".out.iso")):
                    LocalIsoRatioFile = open(IdentifiedMoleculesDir + "/" + files)
                    LocalIsoRatioFileContent = LocalIsoRatioFile.readlines()
                    LocalIsoRatioFile.close()
                    for line in LocalIsoRatioFileContent:
                        SingleMolIsoRatioFiles.append(line)


    ## check, if molecules are identified
    if (len(SingleMolMolfitFiles) == 0):
        print ("\n\nError in XCLASS package, function {:s}:".format(NameOfFunction))
        print ("\tCan not find a molecules in the given spectrum/spectra!")
        print ("\n\tModify list of included/excluded molecules and redo call of function LineIdentification!")
        return (BestMolfitFile, JobDir)


    ##====================================================================================================================================================
    ## start final fit with all identified molecules


    ## prepare MAGIX call
    LineSting = "=" * 173
    print ("\n\n\n\n" + LineSting)
    print ("Prepare final fit with all identified molecules ..")


    ## define name of subdirectory for current molecule, remove not allowed characters from the molecule name
    MolDirName = "all"


    ## create working directory
    print ("\tCreate working directory .. ", end = "")
    MoleculejobDir = LineIDJobDir + MolDirName + "/"
    command_string = "mkdir -p " + MoleculejobDir
    os.system(command_string)
    print ("done!")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write merged iso ratio files to working directory
    if (SingleMolIsoRatioFiles != []):
        IsoFileName = MoleculejobDir + MolDirName + ".iso"
        NewIsoFile = open(IsoFileName, 'w')
        for line in SingleMolIsoRatioFiles:
            NewIsoFile.write(line)
        NewIsoFile.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create molfit file for all identified molecules
    print ("\tCreate molfit file ..", end=' ')
    MolfitsFileName = MoleculejobDir + MolDirName + ".molfit"
    LocalSQLParamArray = SQLParamArray
    LocalSQLParamArray = None
    LocalFixParameterFlag = GetDicKey(InternalParameterList, 'FixParameterFlag', False)
    CreateMolfitFileInNewFormat(MolfitsFileName, SingleMolMolfitFiles, MinColumnDensityAbs, MinColumnDensityEmis, SQLParameters = LocalSQLParamArray, \
                                FixParameterFlag = LocalFixParameterFlag)
    print ("done!")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## adjust path of obs. xml file


    ## adjust paths in obs. xml file
    LocalPrintFlag = False
    LocalObsXMLFileName, IsoRatioFileName, dbFileName = task_myXCLASSFit.AdjustObsXMLFile(NameOfFunction, ObsXMLFileNameOrig, CurrentDir, \
                                                                                          MoleculejobDir, LocalPrintFlag)
    if (LocalObsXMLFileName == ""):                                                         ## did an error occurred
        return (BestMolfitFile, LineIDJobDir)


    ## rename adjusted obs. xml file
    NewLocalObsXMLFileName = MoleculejobDir + "exp.xml"
    task_myXCLASS.NormNCopyMove(LocalObsXMLFileName, NewLocalObsXMLFileName, copyFlag = False)

    # Debug:
    # print ("\n\n\nLocalObsXMLFileName,  = ", LocalObsXMLFileName)
    # print ("IsoRatioFileName = ", IsoRatioFileName)
    # print ("dbFileName = ", dbFileName)
    # print ("NewLocalObsXMLFileName = ", NewLocalObsXMLFileName)


    ##====================================================================================================================================================
    ## check molecules in iso ratio file (if defined)
    #    if (IsoRatioFileName != ""):
    #        FreqMinList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "MinExpRange")
    #        FreqMaxList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "MaxExpRange")
    #        GlobalvLSRList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "GlobalvLSR")
    #        RedshiftList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "Redshift")


    #        ## analyze iso ratio file parameter: check path and name of the iso ratio file and copy new formatted iso ratio file to job directory
    #        ok, NewIsoRatioFileName = task_myXCLASSFit.CheckIsoRatioFile(IsoRatioFileName, NameOfFunction, CurrentDir, MoleculejobDir, FreqMinList, \
    #                                                                     FreqMaxList, GlobalvLSRList, RedshiftList, dbList, MolfitsFileName)
    #        if (ok == 1):                                                                       ## an error occurred
    #            return (BestMolfitFile, LineIDJobDir)

    #        # Debug:
    #        print "FreqMinList = ", FreqMinList
    #        print "FreqMaxList = ", FreqMaxList
    #        print "GlobalvLSRList = ", GlobalvLSRList
    #        print "RedshiftList = ", RedshiftList
    #        print "IsoRatioFileName = ", IsoRatioFileName
    #        print "MoleculejobDir = ", MoleculejobDir
    #        print "NewIsoRatioFileName = ", NewIsoRatioFileName


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## creates fit control xml-file
    AlgorithmXMLFileOverAll = AlgorithmXMLFileOverAll.strip()
    if (LocalFixParameterFlag):
        print ("\tCreating algorithm xml file ..", end=' ')
        task_myXCLASSFit.CreateLMControlXMLFile(MoleculejobDir, 1)
        print ("done!")
    elif (AlgorithmXMLFileOverAll != ""):


        ## check existence of algorithm xml file and copy algorithm xml file to current working directory
        LocalPrintFlag = False
        NameOfFile = "single-molecule-fit algorithm xml"
        LocalAlgorithmXMLFileOverAll = task_myXCLASS.CheckNCopy(AlgorithmXMLFileOverAll, NameOfFunction, NameOfFile, CurrentDir, MoleculejobDir, \
                                                                LocalPrintFlag)
        if (LocalAlgorithmXMLFileOverAll == ""):                                                 ## did an error occurred
            return (BestMolfitFile, LineIDJobDir)

        # Debug:
        # print ("LocalAlgorithmXMLFileOverAll = ", LocalAlgorithmXMLFileOverAll)


        ## rename algorithm xml
        AlgorithmXMLFileOverAll = MoleculejobDir + "algorithm_control.xml"
        task_myXCLASS.NormNCopyMove(LocalAlgorithmXMLFileOverAll, AlgorithmXMLFileOverAll, copyFlag = False)
    else:
        print ("\tCreating algorithm xml file .. ", end = "")
        task_myXCLASSFit.CreateLMControlXMLFile(MoleculejobDir, NumberIteration)
        print ("done!")


    ## creates i/o control xml file
    print ("\tCreating i/o control xml file .. ", end = "")
    task_myXCLASSFit.CreateIOControlXMLFile(MoleculejobDir, NewLocalObsXMLFileName, MolfitsFileName, MAGIXrootDir)
    print ("done!")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## update path and name of iso ratio file and set iso_flag in experimental xml file
    # if (SingleMolIsoRatioFiles != []):
    #     task_MAGIX.WriteXMLtagNEW(NewLocalObsXMLFileName, "iso_flag", ["y"])
    #     task_MAGIX.WriteXMLtagNEW(NewLocalObsXMLFileName, "Isotopologues", ["y"])
    #     task_MAGIX.WriteXMLtagNEW(NewLocalObsXMLFileName, "IsoTableFileName", [IsoFileName])


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start myXCLASSFit
    if (not NoOverallFitFlag):
        print ("\n\nStart MAGIX ..\n")
        MAGIXOption = "plotsaveonly, model=myxclass"
        ok = task_MAGIX.StartMAGIX(MAGIXrootDir, MAGIXOption, MoleculejobDir + "io_control.xml")


    ##------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in parameters for each freq. range for the current molecule
    ListNumberExpRanges = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "NumberExpRanges")
    ListOfAllFreqMin = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "MinExpRange")
    ListOfAllFreqMax = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "MaxExpRange")
    FreqStepList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "StepFrequency")
    t_back_flagList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "t_back_flag")
    tBackList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "BackgroundTemperature")
    tSlopeList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "TemperatureSlope")
    N_HList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "HydrogenColumnDensity")
    beta_dustList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "DustBeta")
    kappa_1300List = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "Kappa")
    DustFileNameList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "DustFileName")
    BackgroundFileNameList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "BackgroundFileName")
    ContPhenFuncIDList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "ContPhenFuncID")
    ContPhenFuncParam1List = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "ContPhenFuncParam1")
    ContPhenFuncParam2List = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "ContPhenFuncParam2")
    ContPhenFuncParam3List = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "ContPhenFuncParam3")
    ContPhenFuncParam4List = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "ContPhenFuncParam4")
    ContPhenFuncParam5List = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "ContPhenFuncParam5")
    GlobalvLSRList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "GlobalvLSR")
    TelescopeSizeList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "TelescopeSize")
    BMINList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "BMIN")
    BMAJList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "BMAJ")
    BPAList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "BPA")
    RedshiftList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "Redshift")
    InterFlagList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "Inter_Flag")
    NoiseList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "NoiseLevel")
    NumberHeaderLinesXML = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "NumberHeaderLines")
    SeparatorColumnsXML = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "SeparatorColumns")
    NumModelPixelXX = None
    NumModelPixelXXList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "NumModelPixelXX")
    if (NumModelPixelXXList != []):
        NumModelPixelXX = NumModelPixelXXList[0]
    NumModelPixelYY = None
    NumModelPixelYYList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "NumModelPixelYY")
    if (NumModelPixelYYList != []):
        NumModelPixelYY = NumModelPixelYYList[0]
    LocalOverlapFlag = None
    LocalOverlapFlagYList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "LocalOverlap_Flag")
    if (LocalOverlapFlagYList != []):
        LocalOverlapFlag = LocalOverlapFlagYList[0]
    NoSubBeamFlag = None
    NoSubBeamFlagList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "NoSubBeam_Flag")
    if (NoSubBeamFlagList != []):
        NoSubBeamFlag = NoSubBeamFlagList[0]
    EmAbsPATH = None
    EmAbsPATHList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "EmAbsPATH")
    if (EmAbsPATHList != []):
        EmAbsPATH = EmAbsPATHList[0]

    OrigIsoFlag = False
    IsoFlagList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "Isotopologues")
    if (IsoFlagList == []):
        IsoFlagList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "iso_flag")
    if (len(IsoFlagList) > 0):
        LocalIsoFlag = IsoFlagList[0].lower()
        LocalIsoFlag = task_myXCLASSMapFit.CheckBool(LocalIsoFlag)
        if (LocalIsoFlag):
            OrigIsoFlag = True
    IsoTableFileNameList = task_MAGIX.GetXMLtagNEW(NewLocalObsXMLFileName, "IsoTableFileName")
    if (IsoTableFileNameList != []):
        IsoTableFileName = IsoTableFileNameList[0]
    else:
        IsoTableFileName = ""


    ## read in molfit file which correspond to the best fit (variable 'ListSubtractedModelSpectra' is not used)
    if (not NoOverallFitFlag):
        print ("Read in molfit file which correspond to the best fit .. ", end = "")
        ConvertLogLinFlag = "true"
        ListOfAllFreqMinCopy = ListOfAllFreqMin
        ListOfAllFreqMin = []                                                               ## clear list, because we don't want the model values
        BestMolfitFile, IsoRatioFileContent, ListSubtractedModelSpectra = GetBestResult(MoleculejobDir, ConvertLogLinFlag, ListOfSpectraNames, \
                                                                                        ListNumberExpRanges, ListOfAllFreqMin, ListOfAllFreqMax, \
                                                                                        tBackList, tSlopeList, SQLParameters = SQLParamArray)
        ListOfAllFreqMin = ListOfAllFreqMinCopy
        print ("done!")
    else:
        MolfitsFile = open(MolfitsFileName)
        BestMolfitFile = MolfitsFile.readlines()
        MolfitsFile.close()
        #    IsoRatioFile = open(NewIsoRatioFileName)
        #    IsoRatioFileContent = IsoRatioFile.readlines()
        #    IsoRatioFile.close()


    ##====================================================================================================================================================
    ## determine the spectra of each identified molecule


    ## split final molfit file in N small molfit files containing only one molecule
    MoleculesInOverallMolfitFile = []
    MolfitFileForEachMolecule = []
    CounterMolecules = 0
    LinesOfMolfitFile = []
    for CounterLines, line in enumerate(BestMolfitFile):
        StrippedLine = line.strip()
        i = StrippedLine.find("%")
        if (i > (-1)):
            StrippedLine = StrippedLine[:i].strip()
        if (StrippedLine != ""):
            SplittedLine = StrippedLine.split()
            if (len(SplittedLine) == 2):
                if (CounterMolecules > 0):
                    MolfitFileForEachMolecule.append(LinesOfMolfitFile)
                CounterMolecules += 1
                LinesOfMolfitFile = []
                LinesOfMolfitFile.append(line)
                MoleculesInOverallMolfitFile.append(SplittedLine[0])
            else:
                LinesOfMolfitFile.append(line)
    if (CounterMolecules > 0):
        MolfitFileForEachMolecule.append(LinesOfMolfitFile)


    ## add complete molfit file to MolfitFileForEachMolecule
    MolfitFileForEachMolecule.append(BestMolfitFile)
    MoleculesInOverallMolfitFile.append("all identified molecules")

    # Debug:
    # print ("\n\n\n\n\n\n\n\n\n")
    # print ("BestMolfitFile = ", BestMolfitFile)
    # print ("MolfitFileForEachMolecule = ", MolfitFileForEachMolecule)
    # print ("MoleculesInOverallMolfitFile = ", MoleculesInOverallMolfitFile)
    # print ("\n\n\n\n\n\n\n\n\n")


    ## create new subdirectories for each molecule and exp. data file within the directory where the final fit was done
    print ("\nCreate working directory for determine contributions of each identified molecule ..", end=' ')
    IdentifiedMoleculesSubDirectories = MoleculejobDir + "final_fit/"
    command_string = "mkdir -p " + IdentifiedMoleculesSubDirectories
    os.system(command_string)
    print ("done!")


    ##----------------------------------------------------------------------------------------------------------------------------------------
    ## loop over all molecules
    PlotNum = GetDicKey(InternalParameterList, 'PlotNum', 10)
    DPIResolution = GetDicKey(InternalParameterList, 'DPIResolution', 100.0)
    for counterMolecules, molfitFile in enumerate(MolfitFileForEachMolecule):


        ## print what you do
        if (counterMolecules == 0):
            print ("\n\n\n")
        print ("Determine contribution of molecule " + MoleculesInOverallMolfitFile[counterMolecules] + "..")


        ## define current molecule name
        if (counterMolecules < len(MolfitFileForEachMolecule) - 1):
            CurrentMolecule = MoleculeFileName(MoleculesInOverallMolfitFile[counterMolecules])
        else:
            CurrentMolecule = "all"


        ## create working directory for current molecule
        SubDirectoryForCurrentMolecule = IdentifiedMoleculesSubDirectories + CurrentMolecule + "/"
        command_string = "mkdir -p " + SubDirectoryForCurrentMolecule
        os.system(command_string)


        ## check if path of background files is defined
        LocalBackgroundFileList = GetBackgroundFileNames(InternalParameterList, CurrentMolecule)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine the intensities for each molecule
        for ObsDataFileIndex in range(len(ListOfSpectra)):                                  ## loop over all spectra in the experimentalData directory


            ## get number of header lines and character separating columns etc. from obs. xml file
            RangeIndex = (-1)
            ObsXMLParameterDictFile = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, NumberHeaderLinesListIn = NumberHeaderLinesXML, \
                                                                           SeparatorColumnsListIn = SeparatorColumnsXML, \
                                                                           TelescopeSizeListIn = TelescopeSizeList, BMIN_ListIn = BMINList, \
                                                                           BMAJ_ListIn = BMAJList, BPA_ListIn = BPAList, \
                                                                           GlobalvLSRListIn = GlobalvLSRList, \
                                                                           Redshift_ListIn = RedshiftList, InterFlagListIn = InterFlagList, \
                                                                           NumberRangeListIn = ListNumberExpRanges)
            NumberHeaderLines = ObsXMLParameterDictFile['NumberHeaderLines']
            SeparatorColumns = ObsXMLParameterDictFile['SeparatorColumns']
            TelescopeSize = ObsXMLParameterDictFile['TelescopeSize']
            BMIN = ObsXMLParameterDictFile['BMIN']
            BMAJ = ObsXMLParameterDictFile['BMAJ']
            BPA = ObsXMLParameterDictFile['BPA']
            InterFlag = ObsXMLParameterDictFile['InterFlag']
            GlobalvLSR = ObsXMLParameterDictFile['GlobalvLSR']
            if (GlobalvLSR is None):
                GlobalvLSR = 0.0
            Redshift = ObsXMLParameterDictFile['Redshift']
            if (Redshift is None):
                Redshift = 0.0


            ## get path and name of corresponding data file
            SpectraName = ListOfSpectraNames[ObsDataFileIndex][1].strip()
            i = SpectraName.rfind("/")
            if (i > (-1)):
                SpectraName = SpectraName[i+1:]


            ## get parameter for current spectrum
            t_back_flag = True
            nH_flag = True
            #if (SingleMolIsoRatioFiles != [] or (CurrentMolecule == "all" and OrigIsoFlag)):
            if (SingleMolIsoRatioFiles != [] or OrigIsoFlag):
                iso_flag = True
            else:
                iso_flag = False
                IsoTableFileName = ""
            NumberProcessors = 1
            myXCLASSrootDir = task_myXCLASS.GetmyXCLASSRootDir()


            ## get obs. data
            LocalObsSpectrum = ListSubtractedSpectra[ObsDataFileIndex]
            LowestDataPoint = min(LocalObsSpectrum[:,0])
            HighestDataPoint = max(LocalObsSpectrum[:,0])
            FreqArray = LocalObsSpectrum[:, 0]

            # Debug:
            # print ("LowestDataPoint = ", LowestDataPoint)
            # print ("HighestDataPoint = ", HighestDataPoint)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## create figure
            WidthSpectrum = (HighestDataPoint - LowestDataPoint) / float(PlotNum)
            fig, layers = pylab.subplots(PlotNum, 1, figsize = (17, PlotNum * 10))
            # fig.clear()
            pylab.subplots_adjust(bottom = 0.005, top = 0.995, right = 0.98, left = 0.07, hspace = 0.1, wspace = 0.2)


            ## switch backend
            if ((PlotNum * 10 * DPIResolution) > 32768):
                plotBackend = "svg"
                pylab.switch_backend('{:s}'.format(plotBackend))
            else:
                plotBackend = "png"


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## add observational data for each subplot
            ListIntObsMinMax = []
            for PlotID in range(PlotNum):


                ## add title
                if (PlotID == 0):
                    TitleStringPart = "Contribution of " + MoleculesInOverallMolfitFile[counterMolecules]
                    layers[PlotID].set_title(TitleStringPart)


                ## initialize current subplot
                layers[PlotID].grid(True)
                layers[PlotID].set_ylabel(r"Brightness Temperature [K]")
                layers[PlotID].set_xlabel("Rest Frequency [MHz]")


                ## define boundaries
                x1min = LowestDataPoint + PlotID * WidthSpectrum
                x1max = LowestDataPoint + (PlotID + 1) * WidthSpectrum


                ## get min and max indices
                x1minIndex = max(0, (numpy.abs(LocalObsSpectrum[:, 0] - x1min)).argmin() - 1)
                x1maxIndex = min(len(LocalObsSpectrum[:, 0]) - 1, (numpy.abs(LocalObsSpectrum[:, 0] - x1max)).argmin() + 1)


                ## get frequencies and intensities
                FreqObsArray = LocalObsSpectrum[x1minIndex:x1maxIndex, 0]
                IntObsArray = LocalObsSpectrum[x1minIndex:x1maxIndex, 1]
                IntObsMin = numpy.nanmin(IntObsArray)
                IntObsMax = numpy.nanmax(IntObsArray)
                ListIntObsMinMax.append([IntObsMin, IntObsMax])


                ## add plot to layer
                layers[PlotID].plot(FreqObsArray, IntObsArray, '-', color='black', linewidth=3.0, label = 'exp. data', drawstyle='steps-mid')

                # Debug:
                # print ("\n\nx1min, x1max = ", x1min, x1max)


            ## analyze ranges
            for RangeIndex in range(NumberFrequencyRanges):                                 ## loop over all frequency ranges


                ## get parameter for current frequency range
                ObsXMLParameterDictRange = task_myXCLASS.GetObsXMLFileParameters(ObsDataFileIndex, RangeIndex, FreqMinListIn = ListOfAllFreqMin, \
                                                                                FreqMaxListIn = ListOfAllFreqMax, FreqStepListIn = FreqStepList, \
                                                                                tBackFlagListIn = t_back_flagList, tBackListIn = tBackList, \
                                                                                tSlopeListIn = tSlopeList, N_HListIn = N_HList, \
                                                                                beta_dustListIn = beta_dustList, kappa_1300ListIn = kappa_1300List, \
                                                                                DustFileNameListIn = DustFileNameList, \
                                                                                BackgroundFileNameListIn = BackgroundFileNameList, \
                                                                                ContPhenFuncID_ListIn = ContPhenFuncIDList, \
                                                                                ContPhenFuncParam1_ListIn = ContPhenFuncParam1List, \
                                                                                ContPhenFuncParam2_ListIn = ContPhenFuncParam2List, \
                                                                                ContPhenFuncParam3_ListIn = ContPhenFuncParam3List, \
                                                                                ContPhenFuncParam4_ListIn = ContPhenFuncParam4List, \
                                                                                ContPhenFuncParam5_ListIn = ContPhenFuncParam5List, \
                                                                                NoiseListIn = NoiseList)
                FreqMin = ObsXMLParameterDictRange['FreqMin']
                if (FreqMin is not None):
                    FreqMinOrig = FreqMin
                    FreqMax = ObsXMLParameterDictRange['FreqMax']
                    FreqStep = ObsXMLParameterDictRange['FreqStep']
                    t_back_flag = ObsXMLParameterDictRange['t_back_flag']
                    tBack = ObsXMLParameterDictRange['tBack']
                    tSlope = ObsXMLParameterDictRange['tSlope']
                    nH_flag = True
                    N_H = ObsXMLParameterDictRange['N_H']
                    beta_dust = ObsXMLParameterDictRange['beta_dust']
                    kappa_1300 = ObsXMLParameterDictRange['kappa_1300']
                    if (N_H is None or beta_dust is None or kappa_1300 is None):
                        nH_flag = False
                        N_H = 0.0
                        beta_dust = 0.0
                        kappa_1300 = 0.0
                    DustFileName = ObsXMLParameterDictRange['DustFileName']
                    if (DustFileName is None):
                        DustFileName = ""
                    BackgroundFileName = ObsXMLParameterDictRange['BackgroundFileName']
                    if (BackgroundFileName is None):
                        BackgroundFileName = FindBackgroundFileName4Range(LocalBackgroundFileList, FreqMin, FreqMax)
                    BackgroundFileName = BackgroundFileName.strip()
                    LocalContPhenFuncID = ObsXMLParameterDictRange['ContPhenFuncID']
                    LocalContPhenFuncParam1 = ObsXMLParameterDictRange['ContPhenFuncParam1']
                    LocalContPhenFuncParam2 = ObsXMLParameterDictRange['ContPhenFuncParam2']
                    LocalContPhenFuncParam3 = ObsXMLParameterDictRange['ContPhenFuncParam3']
                    LocalContPhenFuncParam4 = ObsXMLParameterDictRange['ContPhenFuncParam4']
                    LocalContPhenFuncParam5 = ObsXMLParameterDictRange['ContPhenFuncParam5']
                    Noise = ObsXMLParameterDictRange['NoiseLevel']
                    if (Noise is None):
                        Noise = 0.0


                    ## determine doppler shifted min. and max. frequencies
                    NewFreqMin = task_myXCLASS.ConvertFreq(FreqMin, GlobalvLSR, z = Redshift, backTrafo = True)
                    NewFreqMax = task_myXCLASS.ConvertFreq(FreqMax, GlobalvLSR, z = Redshift, backTrafo = True)
                    ShiftedFreqMin = min(NewFreqMin, NewFreqMax)
                    ShiftedFreqMax = max(NewFreqMin, NewFreqMax)


                    ## determine doppler shifted min. and max. frequencies of obs. data file
                    LocalLowestDataPoint1 = task_myXCLASS.ConvertFreq(LowestDataPoint, GlobalvLSR, z = Redshift, backTrafo = True)
                    LocalHighestDataPoint1 = task_myXCLASS.ConvertFreq(HighestDataPoint, GlobalvLSR, z = Redshift, backTrafo = True)
                    ShiftedLowestDataPoint = min(LocalLowestDataPoint1, LocalHighestDataPoint1)
                    ShiftedHighestDataPoint = max(LocalLowestDataPoint1, LocalHighestDataPoint1)

                    # Debug:
                    # print ("\n\n\n\n\n\n\n\n\n\n\n")
                    # print ("ObsDataFileIndex, RangeIndex = ", ObsDataFileIndex, RangeIndex)
                    # print ("FreqMin = ", FreqMin)
                    # print ("FreqMax = ", FreqMax)
                    # print ("tBack = ", tBack)
                    # print ("tSlope = ", tSlope)
                    # print ("N_H = ", N_H)
                    # print ("beta_dust = ", beta_dust)
                    # print ("kappa_1300 = ", kappa_1300)
                    # print ("DustFileName = ", DustFileName)
                    # print ("BackgroundFileName = ", BackgroundFileName)
                    # print ("LocalContPhenFuncID = ", LocalContPhenFuncID)
                    # print ("LocalContPhenFuncParam1 = ", LocalContPhenFuncParam1)
                    # print ("LocalContPhenFuncParam2 = ", LocalContPhenFuncParam2)
                    # print ("LocalContPhenFuncParam3 = ", LocalContPhenFuncParam3)
                    # print ("LocalContPhenFuncParam4 = ", LocalContPhenFuncParam4)
                    # print ("LocalContPhenFuncParam5 = ", LocalContPhenFuncParam5)
                    # print ("Noise = ", Noise)


                    ## get name of current molecule
                    TransFreqList = []
                    if (counterMolecules < len(MolfitFileForEachMolecule) - 1):
                        LocalMaxNumTransInFit = 0
                        TransFreqList, DBParamList = GetTransFreq(MoleculesInOverallMolfitFile[counterMolecules], ShiftedFreqMin, ShiftedFreqMax, \
                                                                  SQLParamArray, PathFileNameDB, LocalMaxNumTransInFit)

                    # Debug:
                    # print ("molfitFile = ", molfitFile)
                    # print ("CurrentMolecule = ", CurrentMolecule)
                    # print ("MoleculesInOverallMolfitFile[counterMolecules] = ", MoleculesInOverallMolfitFile[counterMolecules])
                    # print ("TransFreqList = ", TransFreqList)


                    ## consider only molecules which have transitions
                    if (TransFreqList != [] or CurrentMolecule == "all"):


                        ## write molfit file to the subdirectory of the current molecule
                        NameOfCurrentMolfitFile = SubDirectoryForCurrentMolecule + CurrentMolecule + ".molfit"
                        CurrentMolfitFile = open(NameOfCurrentMolfitFile, 'w')
                        for line in molfitFile:
                            CurrentMolfitFile.write(line)
                        CurrentMolfitFile.close()


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## write iso ratio file to current subdirectory
                        if (iso_flag and (not NoOverallFitFlag)):
                            IsoTableFileName = SubDirectoryForCurrentMolecule + CurrentMolecule + ".iso"
                            IsoTableFile = open(IsoTableFileName, 'w')
                            for line in IsoRatioFileContent:
                                IsoTableFile.write(line)
                            IsoTableFile.close()


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## convert molfit to xml file
                        LocalPrintFlag = "false"
                        InstanceXMLFileName = SubDirectoryForCurrentMolecule + "parameter.xml"
                        ok = task_myXCLASS.ConversionMolfit2XML(LocalPrintFlag, NameOfCurrentMolfitFile, InstanceXMLFileName)
                        if (iso_flag):
                            ok = task_myXCLASS.GetIsoTableParameters(LocalPrintFlag, IsoTableFileName, InstanceXMLFileName)


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## prepare call of myXCLASS
                        LocalPrintFlag = False
                        SpectrumOnlyFlag = True
                        task_myXCLASS.CallmyXCLASS(myXCLASSrootDir, SubDirectoryForCurrentMolecule, FreqMin, FreqMax, FreqStep, TelescopeSize, \
                                                   InterFlag, t_back_flag, tBack, tSlope, nH_flag, N_H, beta_dust, kappa_1300, DustFileName, \
                                                   GlobalvLSR, iso_flag, InstanceXMLFileName, dbFileName, LocalPrintFlag, SpectrumOnlyFlag, \
                                                   NumModelPixelXX, NumModelPixelYY, LocalOverlapFlag, BMIN, BMAJ, BPA, Redshift, \
                                                   LocalContPhenFuncID, LocalContPhenFuncParam1, LocalContPhenFuncParam2, LocalContPhenFuncParam3, \
                                                   LocalContPhenFuncParam4, LocalContPhenFuncParam5, NoSubBeamFlag = NoSubBeamFlag, \
                                                   EmAbsPATH = EmAbsPATH, BackgroundFileName = BackgroundFileName)


                        ##================================================================================================================================
                        ## read in calculated spectrum
                        myXCLASSoutputFilename = SubDirectoryForCurrentMolecule + "xclass_spectrum_output.dat"
                        ImportModeledSpectrumFlag = True
                        try:
                            modeldata = numpy.loadtxt(myXCLASSoutputFilename, skiprows = 0)
                            ModFreqArray = modeldata[:, 0]
                            if (abs(numpy.nanmin(modeldata[:, 1])) == 0.0 and abs(numpy.nanmax(modeldata[:, 1])) == 0.0): ## empty spectrum?
                                ImportModeledSpectrumFlag = False
                        except IOError as err:
                            ImportModeledSpectrumFlag = False

                        # Debug:
                        # print ("CurrentMolecule, ImportModeledSpectrumFlag = ", CurrentMolecule, ImportModeledSpectrumFlag)


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## do we have an non-empty spectrum?
                        if (ImportModeledSpectrumFlag):


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## subtract continuum, (IMPORTANT: v_LSR is not relevant here)


                            ## subtract background spectrum
                            if (BackgroundFileName != ""):
                                BackgroundFile = ImportASCIIFile(BackgroundFileName)        ## import interpolation object
                                LocalBackgroundSpectrum = BackgroundFile(ModFreqArray)      ## compute background spectrum at given frequencies
                                for i in range(len(modeldata[:, 1])):                       ## subtract background spectrum
                                    modeldata[i, 1] = modeldata[i, 1] - LocalBackgroundSpectrum[i]


                            ## subtract phenom. continuum description from obs. data
                            elif (tBack is not None and tBack != 0.0 and tSlope is not None):       ## tBack or tSlope are unequal to zero.
                                for i in range(len(modeldata[:, 1])):                       ## loop over all points in the spectrum
                                    modeldata[i, 1] = modeldata[i, 1] - (abs(tBack) * (modeldata[i, 0] / FreqMin)**tSlope)


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## define extended spectral name
                            ExtendedSpectraName = SpectraName + "__" + str(FreqMin) + "_-_" + str(FreqMax) + "_MHz"


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## read in transition energies from file
                            myXCLASSTransFilename = SubDirectoryForCurrentMolecule + "transition_energies.dat"
                            cmdString = ""
                            try:
                                f = open(myXCLASSTransFilename)
                                contents = f.readlines()
                                f.close()
                                myXCLASSTrans = []
                                for line in contents:
                                    StrippedLine = line.strip()
                                    i = StrippedLine.find("%")
                                    if (i > (-1)):
                                        StrippedLine = StrippedLine[:i].strip()
                                    if (StrippedLine != ""):
                                        SplittedLine = StrippedLine.split()
                                        myXCLASSTrans.append(float(SplittedLine[0]))
                                    cmdString += "mv " + myXCLASSTransFilename + " " + SubDirectoryForCurrentMolecule \
                                                                                        + "transition_energies___" + ExtendedSpectraName + ".dat; "
                            except IOError as err:
                                myXCLASSTrans = []


                            ## rename output files
                            cmdString += "mv " + SubDirectoryForCurrentMolecule + "xclass_spectrum.log " + SubDirectoryForCurrentMolecule \
                                               + "xclass___" + ExtendedSpectraName + "___" + CurrentMolecule + ".log; "
                            cmdString += "mv " + SubDirectoryForCurrentMolecule + "xclass_spectrum_output.dat " + SubDirectoryForCurrentMolecule \
                                               + "modeled_spectrum___" + ExtendedSpectraName + "___" + CurrentMolecule + ".dat"
                            os.system(cmdString)


                            ## add modeled spectra to figure
                            for PlotID in range(PlotNum):


                                ## define boundaries
                                x1min = LowestDataPoint + PlotID * WidthSpectrum
                                x1max = LowestDataPoint + (PlotID + 1) * WidthSpectrum


                                ## get frequencies and intensities
                                FreqModArray = modeldata[:, 0]
                                IntModArray = modeldata[:, 1]
                                IntModMin = numpy.nanmin(IntModArray)
                                IntModMax = numpy.nanmax(IntModArray)


                                ## add vertical lines indicating transition frequencies to plot
                                if (myXCLASSTrans != [] and CurrentMolecule != "all"):
                                    for lines in myXCLASSTrans:
                                        layers[PlotID].axvline(x = lines, color='green', linewidth=1, linestyle='--')


                                ## add noise level to plot
                                layers[PlotID].axhspan(-Noise, Noise, alpha = 0.3, color = 'blue')


                                ## add plot to layer
                                layers[PlotID].plot(FreqModArray, IntModArray, '-', color='red', linewidth=2.0, label = 'fit')


                                ## get index for NewLocalFreqMin and NewLocalFreqMax
                                ObsFreqMinIndex = max(0, (numpy.abs(LocalObsSpectrum[:, 0] - x1min)).argmin() - 1)
                                ObsFreqMaxIndex = min(len(LocalObsSpectrum[:, 0]) - 1, (numpy.abs(LocalObsSpectrum[:, 0] - x1max)).argmin() + 1)
                                ObsIn1 = min(ObsFreqMinIndex, ObsFreqMaxIndex)
                                ObsIn2 = max(ObsFreqMinIndex, ObsFreqMaxIndex)


                                ModFreqMinIndex = max(0, (numpy.abs(FreqModArray[:] - x1min)).argmin() - 1)
                                ModFreqMaxIndex = min(len(FreqModArray[:]) - 1, (numpy.abs(FreqModArray[:] - x1max)).argmin() + 1)
                                ModIn1 = min(ModFreqMinIndex, ModFreqMaxIndex)
                                ModIn2 = max(ModFreqMinIndex, ModFreqMaxIndex)


                                ## determine min. and max. values for current plot
                                y1Min = numpy.nanmin(LocalObsSpectrum[ObsIn1:ObsIn2, 1])
                                y2Min = numpy.nanmin(IntModArray[ModIn1:ModIn2])
                                yMin = min(y1Min, y2Min, -Noise)
                                yMin = yMin - 0.07 * abs(yMin)
                                y1Max = numpy.nanmax(LocalObsSpectrum[ObsIn1:ObsIn2, 1])
                                y2Max = numpy.nanmax(IntModArray[ModIn1:ModIn2])
                                yMax = max(y1Max, y2Max, Noise)
                                yMax = yMax + 0.07 * abs(yMax)
                                if (yMin == 0.0):
                                    yMin = -yMax * 0.07
                                layers[PlotID].set_ylim(yMin, yMax)
                                layers[PlotID].set_xlim(x1min, x1max)


            ## save plot to file
            pngFile = SubDirectoryForCurrentMolecule + CurrentMolecule + "__" + SpectraName.replace(".dat", "")
            pylab.savefig(pngFile + ".png", dpi = 100)
            pylab.draw()
            pylab.close(fig)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## remove copy of unmodified xml file and default molfit file
    os.system("rm -rf " + ObsXMLFileNameOrig + " " + DefaultMolfitFile)


    ##====================================================================================================================================================
    ## define return variable
    return (BestMolfitFile, JobDir)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Line identification routine
##
def LineIdentification(Noise, MaxOverestimationHeight, SourceName, DefaultMolfitFile, Tolerance, SelectedMolecules, StrongMoleculeList, \
                       MinColumnDensityEmis, MinColumnDensityAbs, NumberIteration, AlgorithmXMLFileSMF, AlgorithmXMLFileOverAll, experimentalData, \
                       vLSR, TelescopeSize, Inter_Flag, tBack, tSlope, N_H, beta_dust, kappa_1300, NumModelPixelXX, NumModelPixelYY, \
                       LocalOverlapFlag, NoSubBeamFlag, clusterdef):
    """

The function starts the line identification routine and returns the identified lines with the corresponding best fit.


input parameters:
-----------------

    - Noise:                        defines noise level, (default: 2.0).

    - MaxOverestimationHeight:      defines the overestimation limit

    - SourceName:                   name of source molfit file, (default: "").

    - DefaultMolfitFile:            path and name of a default molfit file used for molecules
                                    which are not described by the source templates, (default: "").

    - Tolerance:                    fraction (%) of overestimated lines (default: 10.0).

    - SelectedMolecules:            a (python) list containing the names of all molecules which
                                    should be considered or a file name including the molecules
                                    which should be selected.

                                    Note, if the parameter defines a relative path, this path has
                                    to be defined relative to the current working directory!

    - StrongMoleculeList:           list of strong (highly abundant) molecules which has to be
                                    fitted before the other single molecule fits are started

    - MinColumnDensityEmis:         min. column density (for emission lines) of an optimized
                                    component to be included in the overall molfit file (default: 0).

    - MinColumnDensityAbs:          min. column density (for absorption lines) of an optimized
                                    component to be included in the overall molfit file (default: 0).

    - NumberIteration:              max. number of iterations (default: 50).

    - AlgorithmXMLFileSMF:          path and name of a MAGIX xml-file including definitions
                                    for an algorithm or algorithm chain has to be given. This
                                    xml file is used for the each single molecule fit. (A
                                    relative path has to be defined relative to the current
                                    working directory!)

                                    NOTE, if the user specify a xml file, the number of
                                    iterations given by the parameter "NumberIteration" is
                                    ignored. The number of iteration is then given by the xml
                                    file. In order to use the implemented fit algorithm
                                    (Levenberg-Marquardt) clear the AlgorithmXMLFileSMF parameter,
                                    i.e. AlgorithmXMLFileSMF = "", and define the max. number of
                                    iterations by using parameter "NumberIteration".

    - AlgorithmXMLFileOverAll:      only necessary, if the user wants to use another fit
                                    algorithm (than Levenberg-Marquardt) for fitting the final
                                    molfit file including all identified molecules. Therefore,
                                    the path and name of a MAGIX xml-file defining settings
                                    for an algorithm or algorithm chain has to be given. (A
                                    relative path has to be defined relative to the current
                                    working directory!)

                                    NOTE, if the user specify a xml file, the number of
                                    iterations given by the parameter "NumberIteration"
                                    is ignored. The number of iteration is then given by the
                                    xml file. In order to use the implemented fit algorithm
                                    (Levenberg-Marquardt) clear the AlgorithmXMLFileOverAll
                                    parameter, i.e. AlgorithmXMLFileOverAll = "", and define
                                    the max. number of iterations by using parameter
                                    "NumberIteration".

    - experimentalData:             This parameter offers two different possibility to send
                                    the experimental data to the line identification function:

                                    - the parameter experimentalData defines the path and
                                      name of and experimental xml-file suitable for MAGIX.

                                    - the parameter experimentalData defines the path and
                                      name of and ASCII file called experimental data file,
                                      where the first column describe the frequency (in MHz)
                                      and the second column the beam temperature (intensity).

                                    NOTE, if the parameter experimentalData defines a relative
                                    path, the path has to be defined relative to the current
                                    working directory!

    The following parameters are needed, if the parameter experimentalData does NOT
    describe the path and name of a MAGIX xml-file:

    - vLSR:                         local standard of rest v_LSR (in km/s), (default: 0).

    - TelescopeSize:                size of telescope (in m), (default: 1).

    - Inter_Flag:                   defines, if single dish ("False") or interferometric
                                    observations ("True") are described, (default: "False").

    - tBack:                        background temperature (in K), (default: 0).

    - tSlope:                       temperature slope (dimensionless), (default: 0).

    - N_H:                          Hydrogen column density (in cm^{-2}), (default: 0).

    - beta_dust:                    beta for dust (dimensionless), (default: 0).

    - kappa_1300:                   kappa (cm^2 g^{-1}), (default: 0).

    - NumModelPixelXX:              used for sub-beam modeling, describes the number
                                    of pixels used in x-direction (default: 100).

    - NumModelPixelYY:              used for sub-beam modeling, describes the number
                                    of pixels used in y-direction (default: 100).

    - LocalOverlapFlag:             flag indicates if local-overlap description is
                                    used or not (default: False).

    - NoSubBeamFlag:                do not use sub-beam description (default: True).

    - clusterdef:                   file that contains cluster definition.


output parameters:
------------------

    - IdentifiedLines:              contains all information about the identified lines
                                    (as an python array, e.g. IdentifiedLines[0] contains
                                    the entries for the first molecule within the
                                    frequency range)

    - JobDir:                       absolute path of the job directory created for the current run.



Example:
--------

Noise = 2.0
MaxOverestimationHeight = 200.0
SourceName = ""
DefaultMolfitFile = "demo/LineIdentification/default.molfit"
Tolerance = 10.0
MaxNumTransInFit = 2000.0
SelectedMolecules = []
MinColumnDensityEmis = 0.0
MinColumnDensityAbs = 0.0
NumberIteration = 10
AlgorithmXMLFileSMF = ""
AlgorithmXMLFileOverAll = ""
experimentalData = "demo/LineIdentification/sma1_part-A.dat"
vLSR = 0.0
TelescopeSize = 200
Inter_Flag = False
tBack = 0.0
tSlope = 0.0
N_H = 3.0E+24
beta_dust = 2.0
kappa_1300 = 0.42
clusterdef = "demo/LineIdentification/clusterdef.txt"
IdentifiedLines, JobDir = LineIdentification()
    """

    # Debug:
    # print ("Noise = ", Noise)
    # print ("MaxOverestimationHeight = ", MaxOverestimationHeight)
    # print ("SourceName = ", SourceName)
    # print ("DefaultMolfitFile = ", DefaultMolfitFile)
    # print ("Tolerance = ", Tolerance)
    # print ("SelectedMolecules = ", SelectedMolecules)
    # print ("StrongMoleculeList = ", StrongMoleculeList)
    # print ("MinColumnDensityEmis = ", MinColumnDensityEmis)
    # print ("MinColumnDensityAbs = ", MinColumnDensityAbs)
    # print ("NumberIteration = ", NumberIteration)
    # print ("AlgorithmXMLFileSMF = ", AlgorithmXMLFileSMF)
    # print ("AlgorithmXMLFileOverAll = ", AlgorithmXMLFileOverAll)
    # print ("experimentalData = ", experimentalData)
    # print ("vLSR = ", vLSR)
    # print ("TelescopeSize = ", TelescopeSize)
    # print ("Inter_Flag = ", Inter_Flag)
    # print ("tBack = ", tBack)
    # print ("tSlope = ", tSlope)
    # print ("N_H = ", N_H)
    # print ("beta_dust = ", beta_dust)
    # print ("kappa_1300 = ", kappa_1300)
    # print ("NumModelPixelXX = ", NumModelPixelXX)
    # print ("NumModelPixelYY = ", NumModelPixelYY)
    # print ("LocalOverlapFlag = ", LocalOverlapFlag)
    # print ("NoSubBeamFlag = ", NoSubBeamFlag)
    # print ("clusterdef = ", clusterdef)


    ## interpret input variable "experimentalData"
    ObsXMLFileName = ""
    ObsDataFileName = None
    if (type(experimentalData).__name__ == 'str'):
        if (experimentalData.endswith(".xml")):                                             ## parameter defines obs. xml file
            ObsXMLFileName = experimentalData.strip()                                       ## remove leading and tailing blanks
        else:
            ObsDataFileName = experimentalData.strip()
    else:
        ObsDataFileName = experimentalData


    ## for later releases
    BMIN = None
    BMAJ = None
    BPA = None
    Redshift = None
    DustFileName = None
    BackgroundFileName = None
    ContPhenFuncID = None
    ContPhenFuncParam1 = None
    ContPhenFuncParam2 = None
    ContPhenFuncParam3 = None
    ContPhenFuncParam4 = None
    ContPhenFuncParam5 = None


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define default settings for parameter "InternalParameterList"
    InternalParameterList = {}


    ## define method for peak analysis
    # InternalParameterList['PeakFindingMethod'] = "bruteforce"
    InternalParameterList['PeakFindingMethod'] = "gausspy_plot"


    ## set gp.parameters
    InternalParameterList['gp.Level'] = "two"
    InternalParameterList['gp.alpha1'] = 0.01
    InternalParameterList['gp.alpha2'] = 0.01
    InternalParameterList['gp.snr_thresh'] = 1.e-4
    InternalParameterList['gp.snr2_thresh'] = 1.e-4
    InternalParameterList['gp.debug'] = False
    InternalParameterList['gp.res'] = 2.0


    ## set gp.parameters for summed spectrum
    InternalParameterList['gp.alpha1.sum'] = 0.01
    InternalParameterList['gp.alpha2.sum'] = 0.01
    InternalParameterList['gp.snr_thresh.sum'] = 1.e-4
    InternalParameterList['gp.snr2_thresh.sum'] = 1.e-4
    InternalParameterList['gp.res.sum'] = 2.0


    ## define kind of KDE analysis
    # InternalParameterList['KDEHist'] = "stack"
    InternalParameterList['KDEHist'] = "kde"


    ## define KDE weighting options
    InternalParameterList['UseWeightedKDE'] = ""                                                ## do not weight KDE
    # InternalParameterList['UseWeightedKDE'] = "trans"                                           ## weight KDE with gA / E_low**2
    # InternalParameterList['UseWeightedKDE'] = "sum"                                             ## weight KDE with summed spectrum
    # InternalParameterList['UseWeightedKDE'] = "weighted-sum"                                    ## weight KDE with weighted summed spectrum


    ## define min. number of identified transitions per vel. comp. (values less than 1 indicate procentage)
    InternalParameterList['MinCounts'] = 0.15


    ## settings for temperature and column density estimation
    InternalParameterList['UseRotationalDiagramFlag'] = False
    InternalParameterList['ParamEstimMethod'] = "parameter-grid"
    InternalParameterList['Trot_grid'] = numpy.linspace(1.073, 1000, 100, endpoint = True)
    InternalParameterList['Ntot_grid'] = [8.0, 12.0, 12.5, 13.0, 13.5, 14.0, 14.5, 15.0, 15.5, 16.0, 16.5, 17.0, 17.5, 18.0, 18.5, 19.0, \
                                          19.5, 20.0, 20.5, 21.0]
    InternalParameterList['MolfitParameterDebugFlag'] = False
    InternalParameterList['SelectedFrequenciesOnlyFlag'] = False
    InternalParameterList['FitParamFlag'] = ["Trot", "Ntot", "vwidth"]                      ## use "[]" to fix all parameters for each component fit or
                                                                                            ## ["Trot", "Ntot", "vwidth", "voff"] to free individual
                                                                                            ## param. or ["all"] to fit T_rot, N_tot, v_width and v_off
    InternalParameterList['ConsiderPrevCompFlag'] = True                                    ## for component fits, take previous components into account
    InternalParameterList['AnalyzePeakFlag'] = False                                        ## use intensity at identified vel. offset and ignore
                                                                                            ## Gaussian amplitutes
    InternalParameterList['FixParameterFlag'] = False                                       ## fix molfit parameter
    InternalParameterList['NumTempGradMult'] = 1                                            ## define number of additional temp. components.
    InternalParameterList['NoOverallFitFlag'] = False                                       ## skip final overall fit ?


    ## plot settings
    InternalParameterList['PlotFlag'] = "all"
    InternalParameterList['PlotOffset'] = 0.0
    InternalParameterList['PlotNum'] = 32


    ## general debug settings
    # InternalParameterList['NoFitFlag'] = False
    # InternalParameterList['AllowSmoothFlag'] = False



    ##====================================================================================================================================================
    ## initialize working variables
    BestMolfitFile, JobDir = LineIdentificationCore(Noise = Noise, MaxOverestimationHeight = MaxOverestimationHeight, SourceName = SourceName, \
                                                    DefaultMolfitFile = DefaultMolfitFile, Tolerance = Tolerance, SelectedMolecules = SelectedMolecules, \
                                                    StrongMoleculeList = StrongMoleculeList, MinColumnDensityEmis = MinColumnDensityEmis, \
                                                    MinColumnDensityAbs = MinColumnDensityAbs, NumberIteration = NumberIteration, \
                                                    AlgorithmXMLFileSMF = AlgorithmXMLFileSMF, AlgorithmXMLFileOverAll = AlgorithmXMLFileOverAll, \
                                                    ObsXMLFileName = ObsXMLFileName, \
                                                    ObsDataFileName = ObsDataFileName, \
                                                    vLSR = vLSR, TelescopeSize = TelescopeSize, BMIN = BMIN, \
                                                    BMAJ = BMAJ, BPA = BPA, Redshift = Redshift, Inter_Flag = Inter_Flag, tBack = tBack, \
                                                    tSlope = tSlope, N_H = N_H, beta_dust = beta_dust, kappa_1300 = kappa_1300, \
                                                    DustFileName = DustFileName, BackgroundFileName = BackgroundFileName, \
                                                    ContPhenFuncID = ContPhenFuncID, ContPhenFuncParam1 = ContPhenFuncParam1, \
                                                    ContPhenFuncParam2 = ContPhenFuncParam2, ContPhenFuncParam3 = ContPhenFuncParam3, \
                                                    ContPhenFuncParam4 = ContPhenFuncParam4, ContPhenFuncParam5 = ContPhenFuncParam5, \
                                                    NumModelPixelXX = NumModelPixelXX, \
                                                    NumModelPixelYY = NumModelPixelYY, LocalOverlapFlag = LocalOverlapFlag, \
                                                    NoSubBeamFlag = NoSubBeamFlag, clusterdef = clusterdef, InternalParameterList = InternalParameterList)


    ##====================================================================================================================================================
    ## define return variable
    return (BestMolfitFile, JobDir)
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
