#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module reads in entries from the sqlite3 database around a selected frequency
##  Copyright (C) 2012 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function ListDatabase:                        function reads in entries from the sqlite3 database
##      - function Cursor.__init__:                     initialize cursor class
##      - function Cursor.connect:                      connect all the stored connection ids
##      - function Cursor.disconnect:                   disconnect all the stored connection ids
##      - function Cursor.on_key:                       (only for debugging) on key event
##      - function Cursor.set_xvalues:                  used to update span positions
##      - function Cursor.mouse_move:                   define what to do when mouse is moved
##      - function Cursor.onclick:                      print out information from database on mouse click
##      - function GetTransitionsCore:                  function reads in entries from the sqlite3 database around a selected frequency
##      - function GetTransitions:                      starter for GetTransitionsCore function
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  T. Moeller      2013-07-25      initial version
##  T. Moeller      2017-03-05      add model data parameter
##  T. Moeller      2020-01-03      porting to python 3.x
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import sqlite3                                                                              ## import sqlite3 package
if (not 'matplotlib' in sys.modules):
    import matplotlib                                                                       ## import matplotlib package
    # matplotlib.use("Agg")                                                                   ## avoid display error
import pylab                                                                                ## import pylab package
from . import task_MAGIX                                                                    ## import MAGIX function from XCLASS package
from . import task_myXCLASS                                                                 ## import package myXCLASS
from . import task_myXCLASSFit                                                              ## import package myXCLASSFit
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define routines for mouse handling
##
class Cursor:


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize GUI
    def __init__(self, fig, ax, conn, ListOfSelectedMolecules, FrequencyWidth, ElowMin, ElowMax, LowerLimitExpX, UpperLimitExpX, FreqStep, DBFile, \
                 SplatFlag, ListVAMDCNames, ListSplatNames):
        """

    input parameters:
    -----------------

        - fig:                      figure object

        - ax:                       layer object

        - conn:                     database object

        - ListOfSelectedMolecules:  list of selected molecules

        - FrequencyWidth:           frequency width

        - ElowMin:                  min. lower energy

        - ElowMax:                  max. lower energy

        - LowerLimitExpX:           lower limit of obs. data

        - UpperLimitExpX:           upper limit of obs. data

        - FreqStep:                 frequency step

        - DBFile:                   path and name of database file

        - SplatFlag:                flag indicating if Splatalogue translation table is included in current database file

        - ListVAMDCNames:           list of VAMDC names

        - ListSplatNames:           list of Splatalogue names


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("fig = ", fig)
        # print ("ax = ", ax)
        # print ("conn = ", conn)
        # print ("ListOfSelectedMolecules = ", ListOfSelectedMolecules)
        # print ("FrequencyWidth = ", FrequencyWidth)
        # print ("ElowMin = ", ElowMin)
        # print ("ElowMax = ", ElowMax)
        # print ("LowerLimitExpX = ", LowerLimitExpX)
        # print ("UpperLimitExpX = ", UpperLimitExpX)
        # print ("FreqStep = ", FreqStep)
        # print ("DBFile = ", DBFile)
        # print ("SplatFlag = ", SplatFlag)
        # print ("ListVAMDCNames = ", ListVAMDCNames)
        # print ("ListSplatNames = ", ListSplatNames)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## initialize GUI parameters and objects
        self.fig = fig                                                                      ## figure object
        self.ax = ax                                                                        ## layer object
        self.conn = conn                                                                    ## database object
        self.ListOfSelectedMolecules = ListOfSelectedMolecules                              ## list of selected molecules
        self.FrequencyWidth = FrequencyWidth                                                ## frequency step
        self.ElowMin = ElowMin / 1.42879                                                    ## get min. lower energy and convert K to cm-1
        self.ElowMax = ElowMax / 1.42879                                                    ## get max. lower energy and convert K to cm-1
        self.LowerLimitExpX = LowerLimitExpX                                                ## lower limit of obs. data
        self.UpperLimitExpX = UpperLimitExpX                                                ## upper limit of obs. data
        self.FreqStep = FreqStep                                                            ## frequency step
        self.DBFile = DBFile                                                                ## get path and name of database file
        self.OrderIndex = 1                                                                 ## define order index of database output
        self.OrderText = "E_low"                                                            ## define text for display
        self.GUIJobDir = ""                                                                 ## initialize job directory
        self.ly = ax.axhline(color = 'b', linestyle = ":")                                  ## horiz., blue, dotted line object of mouse cursor
        self.lx = ax.axvline(color = 'b')                                                   ## vert., blue line object of mouse cursor
        self.x0 = LowerLimitExpX + (UpperLimitExpX - LowerLimitExpX) / 2.0                  ## define initial mouse cursor position
        self.xLeft = self.x0 + FrequencyWidth                                               ## left edge of initial frequency range
        self.xRight = self.x0 - FrequencyWidth                                              ## right edge of initial frequency range
        self.spanx = ax.axvspan(self.xLeft, self.xRight, alpha = 0.2, color = 'grey')       ## span object
        self.txt = ax.text( 0.5, 0.9, '', transform = ax.transAxes)                         ## text location in axes coords
        self.ax.set_xlim(self.LowerLimitExpX, self.UpperLimitExpX)                          ## set lower and upper limits of layer object
        self.SplatFlag = SplatFlag                                                          ## flag indicating if Splatalogue translation table is avail.
        self.ListVAMDCNames = ListVAMDCNames                                                ## list of VAMDC names
        self.ListSplatNames = ListSplatNames                                                ## list of Splatalogue names
        # self.ax.xaxis.set_major_formatter(matplotlib.ticker.FormatStrFormatter('%.4e'))


        ## translate molecule names in ListOfSelectedMolecules
        if (ListOfSelectedMolecules != []):
            self.ListOfSelectedMolecules = []
            ListOfReplacedNames = []
            for LocalSelectedMolecules in ListOfSelectedMolecules:
                if (LocalSelectedMolecules in ListSplatNames):
                    i = ListSplatNames.index(LocalSelectedMolecules)
                    NewMol = ListVAMDCNames[i]
                    ListOfReplacedNames.append([LocalSelectedMolecules, NewMol])
                else:
                    NewMol = LocalSelectedMolecules
                self.ListOfSelectedMolecules.append(NewMol)


            ## print all replaced molecule names
            if (ListOfReplacedNames != []):
                print ("\n\nFound Splatalogue name(s) in selected molecules. Replaced the following name(s):")
                for LocalReplacedNames in ListOfReplacedNames:
                    print ("\t{:s}{:s}{:s} -> {:s}{:s}{:s}".format(chr(34), LocalReplacedNames[0], chr(34), chr(34), LocalReplacedNames[1], chr(34)))
                print ("\n")

        # Debug:
        # print ("\n\n ListOfSelectedMolecules = ", ListOfSelectedMolecules)
        # print ("self.ListOfSelectedMolecules = ", self.ListOfSelectedMolecules)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## connect all the stored connection ids
    def connect(self):
        self.cidpress = self.ax.figure.canvas.mpl_connect('button_press_event', self.onclick)
        self.cidmotion = self.ax.figure.canvas.mpl_connect('motion_notify_event', self.mouse_move)
        self.cidkey = self.ax.figure.canvas.mpl_connect('key_press_event', self.on_key)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## disconnect all the stored connection ids
    def disconnect(self):
        self.ax.figure.canvas.mpl_disconnect(self.cidpress)
        self.ax.figure.canvas.mpl_disconnect(self.cidmotion)
        self.ax.figure.canvas.mpl_disconnect(self.cidkey)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (only for debugging): on key event
    def on_key(self, event):
        """

    input parameters:
    -----------------

        - event:                    ??


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ('you pressed', event.key, event.xdata, event.ydata)


        ## increase / decrease frequency range by self.FreqStep when "+" or "-" is pressed
        UpdateFlag = False
        if (event.key == "+" or event.key == "-"):


            ## increase frequency range by self.FreqStep when "+" is pressed
            if (event.key == "+" and self.FrequencyWidth < (self.UpperLimitExpX - self.LowerLimitExpX)):
                self.FrequencyWidth += abs(self.FreqStep)
                UpdateFlag = True


            ## decrease frequency range by self.FreqStep when "-" is pressed
            elif (event.key == "-" and self.FrequencyWidth > self.FreqStep):
                self.FrequencyWidth -= abs(self.FreqStep)
                UpdateFlag = True


        ## change order of database entries: order by frequencies when "0" is pressed
        elif (event.key == "0"):
            self.OrderIndex = 0
            self.OrderText = "Freq."
            UpdateFlag = True


        ## change order of database entries: order by E_low when "1" is pressed
        elif (event.key == "1"):
            self.OrderIndex = 1
            self.OrderText = "E_low"
            UpdateFlag = True


        ## change order of database entries: order by gA when "2" is pressed
        elif (event.key == "2"):
            self.OrderIndex = 2
            self.OrderText = "Einstein A"
            UpdateFlag = True


        ## change order of database entries: order by gA when "3" is pressed
        elif (event.key == "3"):
            self.OrderIndex = 3
            self.OrderText = "gA"
            UpdateFlag = True


        ## change order of database entries: order by gA/E_low^2 when "4" is pressed
        elif (event.key == "4"):
            self.OrderIndex = 4
            self.OrderText = "gA/E_low**2"
            UpdateFlag = True


        ## update figure
        if (UpdateFlag):
            if (event.inaxes):
                x = self.lx.get_xdata()
                if (type(x) == list):
                    x = x[0]
                y = self.ly.get_ydata()
                if (type(y) == list):
                    y = y[0]
                self.set_xvalues(self.spanx, x - self.FrequencyWidth, x + self.FrequencyWidth)
                self.txt.set_text('Freq.={:1.2f} MHz, Int.={:1.2f}, Width={:.2f}, db-order: {:s}'.format(x, y, self.FrequencyWidth, self.OrderText))
                self.ax.patch.figure.canvas.draw()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set x values for span box
    ## taken from http://stackoverflow.com/questions/35551903/update-location-of-axvspan-with-matplotlib
    def set_xvalues(self, polygon, x0, x1):
        """

    input parameters:
    -----------------

        - polygon:                  ??

        - x0:                       ??

        - x1:                       ??


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("polygon = ", polygon)
        # print ("x0 = ", x0)
        # print ("x1 = ", x1)


        _ndarray = polygon.get_xy()
        _ndarray[:, 0] = [x0, x0, x1, x1, x0]
        polygon.set_xy(_ndarray)


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what to do when mouse is moved
    def mouse_move(self, event):
        """

    input parameters:
    -----------------

        - event:                    ??


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ("event = ", event)


        if not event.inaxes: return
        x, y = event.xdata, event.ydata


        ## update the positions of mouse cursor
        self.ly.set_ydata(y )
        self.lx.set_xdata(x )


        ## update position of span indicating frequency range
        self.set_xvalues(self.spanx, x - self.FrequencyWidth, x + self.FrequencyWidth)


        ## update text wihin figure
        self.txt.set_text('Freq.={:1.2f} MHz, Int.={:1.2f}, Width={:.2f}, db-order: {:s}'.format(x, y, self.FrequencyWidth, self.OrderText))


        ## update plot
        self.ax.patch.figure.canvas.draw()


        ## we're done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print out information from database on mouse click
    def onclick(self, event):
        """

    input parameters:
    -----------------

        - event:                    ??


    output parameters:
    ------------------

        - None
        """

        # Debug:
        # print ('you pressed', event.key, event.button, event.xdata, event.ydata)


        ## check, if right button is pressed
        # if (event.key != "escape"): return
        if (event.button != 3): return


        ## print what you do
        print ("\n\nPlease wait ..")


        ## save plot if escape key is pressed
        OutputDevice = ""
        if (event.key == "escape"):


            ## create job directory for current run
            if (self.GUIJobDir == ""):
                LocalPrintFlag = False
                myXCLASSDir = task_myXCLASS.GetmyXCLASSRootDir()
                self.GUIJobDir = task_MAGIX.CreateRunDirectory("GetTransitions", myXCLASSDir, LocalPrintFlag)

                # Debug:
                # print ("\n\nmyXCLASSDir = ", myXCLASSDir)
                # print ("self.GUIJobDir = ", self.GUIJobDir)


            ## save figure
            LocalFileName = self.GUIJobDir + "Selected-freq__" + str(event.xdata) + "_MHz.png"
            OutputDevice = self.GUIJobDir + "Selected-freq__" + str(event.xdata) + "_MHz___molecules-in-database.dat"
            self.fig.savefig(LocalFileName, dpi=300)


        ## define frequency interval form mouse position
        FreqIntMin = event.xdata - self.FrequencyWidth
        FreqIntMax = event.xdata + self.FrequencyWidth


        ## initialize some parameter
        NameOfRadTransTable = "transitions"
        ColumnNameForNameTransitions = "T_Name"
        ColumnNameForFreqTransitions = "T_Frequency"
        ColumnNameForIntTransitions = "T_Intensity"
        ColumnNameForEinsteinATransitions = "T_EinsteinA"
        ColumnNameForFreqErrTransitions = "T_Uncertainty"
        ColumnNameForELowTransitions = "T_EnergyLower"
        ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
        ColumnNameForQNUpLabelTransitions = "T_UpperStateQuantumNumbers"
        ColumnNameForQNLowLabelTransitions = "T_LowerStateQuantumNumbers"
        ColumnNameForURLTransitions = "T_URL"


        ## define query string
        ##
        ## get names of columns with
        ##      import sqlite3
        ##      connection = sqlite3.connect('cdms_sqlite.db')
        ##      cursor = connection.execute('select * from partitionfunctions')
        ##      names = list(map(lambda x: x[0], cursor.description))
        ##
        query_string = "SELECT "
        query_string += ColumnNameForNameTransitions + ", "                                 ## position 1: molecule name
        query_string += ColumnNameForFreqTransitions + ", "                                 ## position 2: frequency
        query_string += ColumnNameForIntTransitions + ", "                                  ## position 3: intensity
        query_string += ColumnNameForEinsteinATransitions + ", "                            ## position 4: Einstein A
        query_string += ColumnNameForFreqErrTransitions + ", "                              ## position 5: Error frequency
        query_string += ColumnNameForELowTransitions + ", "                                 ## position 6: E_low
        query_string += ColumnNameForgUpTransitions + ", "                                  ## position 7: upper state degeneracy
        query_string += ColumnNameForQNUpLabelTransitions + ", "                            ## position 8: quantum number label for upper state
        query_string += ColumnNameForQNLowLabelTransitions                                  ## position 9: quantum number label for lower state
        # query_string += ColumnNameForURLTransitions                                       ## position 10: URL
        query_string += " FROM " + NameOfRadTransTable
        query_string += " WHERE (" + ColumnNameForFreqTransitions + " >= " + str(FreqIntMin)
        query_string += " and " + ColumnNameForFreqTransitions + " <= " + str(FreqIntMax)
        if (self.ElowMin < self.ElowMax):
            query_string += " and " + ColumnNameForELowTransitions + " >= " + str(self.ElowMin)
            query_string += " and " + ColumnNameForELowTransitions + " <= " + str(self.ElowMax)
        if (len(self.ListOfSelectedMolecules) > 0):
            if (self.ListOfSelectedMolecules[0] != ["None"]):
                query_string += " and ("
                counter = 0
                for molecule in self.ListOfSelectedMolecules:
                    counter += 1
                    if (counter > 1):
                        query_string += " or "
                    query_string += ColumnNameForNameTransitions + " = " + chr(34) + molecule.strip() + chr(34)
                query_string += ")"
        query_string += ")"
        if (self.OrderIndex == 0):
            query_string += " ORDER by " + ColumnNameForFreqTransitions
        elif (self.OrderIndex == 1):
            query_string += " ORDER by " + ColumnNameForELowTransitions
        elif (self.OrderIndex == 2):
            query_string += " ORDER by " + ColumnNameForEinsteinATransitions
        elif (self.OrderIndex == 3):
            query_string += " ORDER by (" + ColumnNameForgUpTransitions + " * " + ColumnNameForEinsteinATransitions + ")"
        elif (self.OrderIndex == 4):
            query_string += " ORDER by ((" + ColumnNameForgUpTransitions + " * " + ColumnNameForEinsteinATransitions + ") / ("
            query_string += ColumnNameForELowTransitions + " * " + ColumnNameForELowTransitions + "))"

        # Debug:
        # print ("\n\nquery_string = ", query_string)
        # print ("len(query_string) = ", len(query_string))


        ## read data from database
        cursor = self.conn.cursor()
        cursor.execute(query_string)
        rows = cursor.fetchall()


        ## open file if file name is defined
        file_flag = False
        formatted_line = "{:25s}".format("Name:")
        formatted_line += "{:37s}".format("quantum number upper state:")
        formatted_line += "{:37s}".format("quantum number lower state:")
        formatted_line += "{:18s}".format("Frequency [MHz]:")
        formatted_line += "{:24s}".format("Error Frequency [MHz]:")
        formatted_line += "{:27s}".format("Einstein A coefficient:")
        # formatted_line += "{:27s}".format("log(Intensity) [nm2 MHz]:")
        formatted_line += "{:20s}".format("Energy_low [K]:")
        if (OutputDevice.strip() != "" and OutputDevice is not None):
            file_flag = True
            out = open(OutputDevice.strip(), 'w')
            out.write(formatted_line + "\n")
        if (len(rows) > 0):
            print ('\n\nContents of the database between {:s} MHz and {:s} MHz:\n'.format(str(FreqIntMin), str(FreqIntMax)))
            print (formatted_line)
        else:
            print ('\n\nNo entries in the database between {:s} MHz and {:s} MHz:\n' .format(str(FreqIntMin), str(FreqIntMax)))
            return


        ## store entries in output variable Contents
        Contents = []
        col_counter = 0
        for row in rows:                                                                    ## loop over all entries of the database
            col_counter += 1
            Contents.append(row)

            # Debug:
            # print ("col_counter, row = ", col_counter, row)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## define formatted line
            ## position 1: molecule name
            ## position 2: frequency
            ## position 3: intensity
            ## position 4: Einstein A
            ## position 5: Error frequency
            ## position 6: E_low
            ## position 7: upper state degeneracy
            ## position 8: quantum number label for upper state
            ## position 9: quantum number label for lower state
            ## position 10: URL
            formatted_line = ""
            name = ""
            freq = ""
            unfreq = ""
            intensity = ""
            EinsteinA = ""
            elow = ""
            QNUpLabel = ""
            QNLowLabel = ""
            DBurl = ""
            for element_counter, elements in enumerate(row):                                ## loop over all elements in current row
                if (elements is not None):                                                  ## check, if element is not None


                    ## get name of molecule
                    if (element_counter == 0):
                        name = elements.strip()
                        name = "{:25s}".format(name[:25])                                   ## name of molecule has max. 25 characters


                    ## get frequency (in MHz)
                    elif (element_counter == 1):
                        freqVal = "{:17s}".format(str(elements))
                        try:
                            freq = "{:17.5f}".format(float(freqVal))
                        except:
                            freq = freqVal

                        # Debug:
                        # print ("freq = ", freq)


                    ## get intensity
                    elif (element_counter == 2):
                        intensityVal = "{:26s}".format(str(elements))
                        try:
                            intensity = "{:26.9e}".format(float(intensityVal))
                        except:
                            intensity = intensityVal

                        # Debug:
                        # print ("intensity = ", intensity)


                    ## get Einstein A coefficient
                    elif (element_counter == 3):
                        EinsteinAVal = "{:26s}".format(str(elements))
                        try:
                            EinsteinA = "{:26.3e}".format(float(EinsteinAVal))
                        except:
                            EinsteinA = EinsteinAVal

                        # Debug:
                        # print ("EinsteinA = ", EinsteinA)


                    ## get uncertainty of frequency (in MHz)
                    elif (element_counter == 4):
                        unfreqVal = "{:23s}".format(str(elements))
                        try:
                            unfreq = "{:23.4e}".format(float(unfreqVal))
                        except:
                            unfreq = unfreqVal

                        # Debug:
                        # print ("unfreq = ", unfreq)


                    ## get E_lower (in cm-1)
                    elif (element_counter == 5):
                        elowVal = "{:19s}".format(str(elements))
                        try:
                            elowVal = float(elowVal) * 1.42879                              ## convert cm-1 to K
                            if (abs(elowVal) > 0.01 and abs(elowVal) < 10000):
                                elow = "{:19.3f}".format(elowVal)
                            else:
                                elow = "{:19.3e}".format(elowVal)
                        except:
                            elow = elowVal

                        # Debug:
                        # print ("elow = ", elow)


                    ## get quantum number label for upper state
                    elif (element_counter == 7):
                        try:
                            lab = elements.replace("ElecStateLabel = ", "").strip()
                            if (len(lab) > 34):
                                lab = lab[:31] + "..."
                            QNUpLabel = "  {:34s}".format(lab)
                        except TypeError:
                            QNUpLabel = "     "

                        # Debug:
                        # print ("QNUpLabel = ", QNUpLabel)


                    ## get quantum number label for lower state
                    elif (element_counter == 8):
                        try:
                            lab = elements.replace("ElecStateLabel = ", "").strip()
                            if (len(lab) > 34):
                                lab = lab[:31] + "..."
                            QNLowLabel = "  {:34s}".format(lab)
                        except TypeError:
                            QNLowLabel = "     "

                        # Debug:
                        # print ("QNLowLabel = ", QNLowLabel)


                    ## get URL
                    #elif (element_counter == 10):
                    #    try:
                    #        DBurl = "{:s}}".format(elements)
                    #    except TypeError:
                    #        DBurl = "     "

                        # Debug:
                        # print ("DBurl = ", DBurl)


            ## construct line for output
            formatted_line = name + " " + QNUpLabel + " " + QNLowLabel + " " + freq + " " + unfreq + " " + EinsteinA + " " + elow + " " + DBurl

            # Debug:
            # print ("formatted_line = ", formatted_line)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## if file name is defined
            if (file_flag):
                out.write(formatted_line + "\n")
            print (formatted_line)


        ## close file, if file name is defined
        if (file_flag):
            out.close()


        ## do some screen formatting
        print ("\n\n\n")


        ## finished
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## This function reads in entries from the sqlite3 database around a selected frequency
##
def GetTransitionsCore(obsdata = None, FreqMin = None, FreqMax = None, SelectMolecule = [], \
                       FrequencyWidth = 5, ElowMin = 0, ElowMax = 1.e9, modeldata = None):
    """

input parameters:
-----------------


    - obsdata:              2D numpy array containing the observational data, (default: None)

                            Note, the data has to be given as a function of frequency (in MHz).

    - FreqMin:              minimum frequency (in MHz) for plot window, (default: None)

    - FreqMax:              maximum frequency (in MHz) for plot window, (default: None)

    - SelectMolecule:       a (python) list containing the names of all molecules which should be considered
                            or an (ASCII) file name including the molecules which should be considered,
                            (default: []).

                            If this parameter is left empty, all molecules in the database within the given
                            range are considered!

                            Note, if the parameter defines a relative path, this path has to be defined
                            relative to the current working directory!

    - FrequencyWidth:       defines the width of the frequency interval in MHz (selected frequency
                            +/- "FrequencyWidth" where the line information are printed out to the screen.
                            (default: 5).

    - ElowMin:              minimum for lower energy (in K) in the "Transitions" table, (default: 0)

    - ElowMax:              maximum for lower energy (in K) in the "Transitions" table, (default: 1.e9)

    - modeldata:            2D numpy array containing the modeled data, (default: None)

                            Note, the data has to be given as a function of frequency (in MHz).



Example 1:
----------

FileName = "demo/LoadASCIIFile/ASCII.dat"
NumHeaderLines = 0
obsdata = LoadASCIIFile()

FreqMin = 4.92100000e+05
FreqMax = 4.92200000e+05
SelectMolecule = "demo/ListDatabase/molecules.txt"
FrequencyWidth = 5.0
ElowMin = 100.0
ElowMax = 1000.0
GetTransitions()


Example 2:
----------

FileName = "demo/LoadASCIIFile/ASCII.dat"
NumHeaderLines = 0
obsdata = LoadASCIIFile()

FreqMin = 4.92100000e+05
FreqMax = 4.92200000e+05
SelectMolecule = ["HNO3;v5=1;", "C2H5CN-15;v=0;"]
FrequencyWidth = 5.0
ElowMin = 100.0
ElowMax = 1000.0
GetTransitions()
    """

    # Debug:
    # print ("obsdata = ", obsdata)
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("SelectMolecule = ", SelectMolecule)
    # print ("FrequencyWidth = ", FrequencyWidth)
    # print ("ElowMin = ", ElowMin)
    # print ("ElowMax = ", ElowMax)
    # print ("modeldata = ", modeldata)


    ##====================================================================================================================================================
    ## analyze obsdata parameter
    PlotobsdataFlag = True
    if (obsdata is None):
        PlotobsdataFlag = False
    else:
        if (type(obsdata).__name__ == 'list'):
            obsdata_flag = "list"
        else:
            obsdata_flag = "array"
        if (len(obsdata) < 3):
            PlotobsdataFlag = False
    if (not PlotobsdataFlag):
        print ("Error in XCLASS package, function GetTransitions:")
        print (" ")
        print ("  Please specify a input parameter named 'obsdata' containing observational data as a 2D python list or array!")
        print (" ")
        print ("  Please correct parameter 'obsdata' and start GetTransitionsCore again.")
        print (" ")
        print (" ")
        return


    ##====================================================================================================================================================
    ## analyze xlimit parameter
    xlimits_flag = False
    if (FreqMin is None or FreqMax is None):
        FreqMin = numpy.nanmin(obsdata[:, 0])
        FreqMax = numpy.nanmax(obsdata[:, 0])

        print ("FreqMin = ", FreqMin)
        print ("FreqMax = ", FreqMax)


    elif (FreqMin != 0.0 or FreqMax != 0.0):
        if (FreqMin != FreqMax):
            if (FreqMin < FreqMax):
                xlimits_flag = True
            else:
                if (xlimits_flag):
                    print ("Error in XCLASS package, function GetTransitions:")
                    print (" ")
                    print ("  The given upper limit of the frequency is smaller than the lower limit!")
                    print (" ")
                    print ("  Please correct the frequency range definition and start myXCLASSPlot again.")
                    print (" ")
                    print (" ")
                    return

    # Debug:
    # print ("xlimits_flag = ", xlimits_flag)


    ##====================================================================================================================================================
    ## analyze modeldata parameter
    if (modeldata is not None):
        PlotModelDataFlag = True
    else:
        PlotModelDataFlag = True
        if (modeldata is None):
            PlotModelDataFlag = False
        elif (len(modeldata) < 3):
            PlotModelDataFlag = False
        else:
            modeldata = numpy.asarray(modeldata)


    ##====================================================================================================================================================
    ## test, if SelectMolecule is a list, otherwise try to use SelectMolecule as filename
    print ("\n\nAnalyze selected molecules ..", flush = True)
    ListOfSelectedMolecules = []
    if (type(SelectMolecule).__name__ != 'list'):                                           ## is SelectMolecule a list ?
        SelectMolecule = [SelectMolecule]
    if (len(SelectMolecule) > 0):
        for entry in SelectMolecule:                                                        ## loop over all entries of SelectMolecule list
            if (entry is not None):
                entryStrip = entry.strip()                                                  ## remove leading and trailing blanks
                if (entryStrip != ""):

                    # Debug:
                    # print ("entryStrip = ", entryStrip)


                    ## check, if current entry is a file
                    if (os.path.isfile(entryStrip)):


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## if file is a molfit file, get names of molecules from molfit file
                        if (entryStrip.endswith(".molfit")):

                            # Debug:
                            # print ("Selection string is a molfit file!")


                            ## read in contents of molfit file
                            f = open(entryStrip)
                            MolfitContents = f.readlines()                                  ## read in contents of molfit file
                            f.close()


                            ## get names of molecules
                            for lines in MolfitContents:                                    ## loop over all lines of molfit file
                                CurrentLine = lines.strip()                                 ## remove leading and trailing blanks

                                # Debug:
                                # print ("CurrentLine = ", CurrentLine)


                                ## check for comments
                                w = CurrentLine.find("%")                                   ## are there comments in the current line ?
                                if (w == 0):                                                ## ignore lines which contains only comments
                                    CurrentLine = ""                                        ## if line is only a comment clear line
                                elif (w > 0):                                               ## is there a comment in the current line ?
                                    CurrentLine = CurrentLine[:w]                           ## remove comments

                                # Debug:
                                # print ("CurrentLine = ", CurrentLine)


                                ## analyze, if the current line contains the name of a molecule
                                if (CurrentLine != ""):                                     ## ignore empty lines
                                    SplitLines = CurrentLine.split()                        ## split current line into columns

                                    # Debug:
                                    # print ("SplitLines = ", SplitLines)


                                    ## contains the last column an integer number?
                                    helpstring = SplitLines[-1].strip()
                                    is_int = "false"
                                    if (helpstring.isdigit()):                              ## continue, if string is an integer number
                                        is_int = "true"


                                    ## If yes, then the current line contains the name of a molecule
                                    if (is_int == "true"):                                  ## if last entry in a line is an integer number, then
                                        MolName = ""                                        ## the current line contains the name of a molecule
                                        for col in SplitLines[:-1]:                         ## get whole line without integer number at the end
                                            MolName += col                                  ## construct entry for SelectMolecule

                                        # Debug:
                                        # print ("MolName = ", MolName)

                                        ListOfSelectedMolecules.append(MolName)             ## construct final array containing all molecule names


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## read names of molecules from ASCII file
                        else:

                            # Debug:
                            # print ("Selection string is a ASCII file!")


                            ## read in whole ASCII file
                            f = open(entryStrip)
                            AllLines = f.readlines()
                            f.close()


                            ## append lines in ASCII file to selection list
                            for line in AllLines:
                                ListOfSelectedMolecules.append(line)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## continue here, if current entry is not a filename
                    else:
                        ListOfSelectedMolecules.append(entryStrip)

    # Debug:
    # print ("ListOfSelectedMolecules = ", ListOfSelectedMolecules)


    ##====================================================================================================================================================
    ## get path of default database file
    dbFilename = task_myXCLASS.GetDefaultDBFile()


    ##====================================================================================================================================================
    ## get Splatalgoue translation table from database file
    SplatFlag = False
    ListVAMDCNames = []
    ListSplatNames = []
    if (ListOfSelectedMolecules != []):
        SplatFlag, LocalMoleculeNameTranslationTable = task_myXCLASSFit.GetSplatTranslationTable(dbFilename)
        ListVAMDCNames, ListSplatNames = list(zip(*LocalMoleculeNameTranslationTable))

    # Debug:
    # print ("ListPathFileNameDB = ", ListPathFileNameDB)
    # print ("len(ListPathFileNameDB) = ", len(ListPathFileNameDB))
    # print ("PathFileNameDB = ", PathFileNameDB)
    # print ("SplatFlag = ", SplatFlag)
    # print ("ListVAMDCNames = ", ListVAMDCNames)
    # print ("ListSplatNames = ", ListSplatNames)


    ##====================================================================================================================================================
    ## connect to the sqlite3 database


    ## connect to database
    try:
        conn = sqlite3.connect(dbFilename)
        print (" ")
        print ("Connection to sqlite3 database {:s} established.".format(dbFilename))
        print (" ")
        print (" ")
    except sqlite3.Error as e:
        print (" ")
        print ("Can not connect to sqlite3 database {:s}.".format(dbFilename))
        print ("Error: {:d}: {:s}".format(e.args[0], e.args[1]))
        sys.exit(1)


    ## print some help to screen
    print ("Press right mouse button to select central frequency.")
    print ("Close plot window to exit GetTransition function.\n")


    ##====================================================================================================================================================
    ## define plotting variables, clear and create plot


    ## define plotting variables
    if (obsdata_flag == "list"):
        expX = [(li[0]) for li in obsdata]
    else:
        expX = obsdata[:,0]
    if (obsdata_flag == "list"):
        expY = [(li[1]) for li in obsdata]
    else:
        expY = obsdata[:,1]


    ## clear plot window
    import matplotlib
    matplotlib.use('Qt5Agg')
    fig = pylab.figure(figsize = (17, 6))
    fig.clear()


    ## create plot window and define axes labels
    if (PlotModelDataFlag):
        pylab.subplots_adjust(bottom = 0.08, top = 0.85, right = 0.98, left = 0.05, hspace = 0.1, wspace = 0.2)
    else:
        pylab.subplots_adjust(bottom = 0.08, top = 0.95, right = 0.98, left = 0.05, hspace = 0.1, wspace = 0.2)
    ax = pylab.subplot(1, 1, 1)
    if (PlotModelDataFlag):
        ax.set_title("Press right mouse button to select central frequency", y = 1.09)
    else:
        ax.set_title("Press right mouse button to select central frequency")
    ax.grid(True)
    ax.set_ylabel(r"Brightness Temperature [K]")
    ax.set_xlabel("Frequency [MHz]")


    ## set Limits
    LowestVal = min(s for s in expX)
    if (not xlimits_flag):
        LowerLimitExpX = LowestVal
    else:
        if (FreqMin < LowestVal):
            LowerLimitExpX = LowestVal
        else:
            LowerLimitExpX = FreqMin
    HighestVal = max(s for s in expX)
    if (not xlimits_flag):
        UpperLimitExpX = HighestVal
    else:
        if (FreqMax > HighestVal):
            UpperLimitExpX = HighestVal
        else:
            UpperLimitExpX = FreqMax

    # Debug:
    # print ('LowerLimitExpX={:19.2f}, UpperLimitExpX={:19.2f}'.format(LowerLimitExpX, UpperLimitExpX))
    # print ('expX[0]={:19.2f}, expX[-1]={:19.2f}'.format(expX[0], expX[-1]))
    # print ('LowestVal={:19.2f}, HighestVal={:19.2f}'.format(LowestVal, HighestVal))


    ## add data to plot window
    ax.plot(expX, expY, '-', color='black', linewidth=2.0, label = 'obs. data', drawstyle='steps-mid')


    ## if modeldata parameter is defined, add model data to plot
    if (PlotModelDataFlag):
        ax.plot(modeldata[:, 0], modeldata[:, -1], '-', color='red', linewidth=2.0, label = 'fit')


    ## set x-range limits
    ax.set_xlim(LowerLimitExpX, UpperLimitExpX)


    ## add legend, if modeldata variable is defined
    if (PlotModelDataFlag):
        ax.legend(bbox_to_anchor=(0., 1.02, 1., .102), loc = 3, ncol = 2, mode = "expand", borderaxespad = 0.)


    ## handle mouse input
    FreqStep = expX[1] - expX[0]
    fig.tight_layout()
    cursor = Cursor(fig, ax, conn, ListOfSelectedMolecules, FrequencyWidth, ElowMin, ElowMax, LowerLimitExpX, UpperLimitExpX, FreqStep, dbFilename, \
                    SplatFlag, ListVAMDCNames, ListSplatNames)
    cursor.connect()
    pylab.show(block = True)
    cursor.disconnect()
    conn.close()                                                                            ## close connection to database


    ##====================================================================================================================================================
    ## define return variables
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------



##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## This function reads in entries from the sqlite3 database around a selected frequency
##
def GetTransitions(obsdata, FreqMin, FreqMax, SelectMolecule, FrequencyWidth, ElowMin, ElowMax, modeldata):
    """

Therefore, the function plots the observational data (contained in the input parameter "obsdata") from frequency
"FreqMin" to "FreqMax". By using the mouse cursor and pressing the mouse button and the "Escape-key"
simultaneously, the user selects a frequency and gets all entries for the corresponding transitions around
the selected frequency from the table "Transitions". The frequency interval is defined by the selected
frequency +/- the input parameter "FrequencyWidth".

The information are printed out to the screen.


Bug:

Please note, due to a bug in the matplotlib package version 0.99 and lower, the following error message is printed out to the screen

    SEVERE  GetTransitions::::casa  An error occurred running task GetTransitions.

You can ignore this message. The function still works!


Please note, you have to close the plotting window, to stop the function.



input parameters:
-----------------


    - obsdata:              2D numpy array containing the observational data

                            Note, the data has to be given as a function of frequency (in MHz).

    - FreqMin:              minimum frequency (in MHz) (default: 0) for plot window

    - FreqMax:              maximum frequency (in MHz) (default: 1.e8) for plot window

    - SelectMolecule:       a (python) list containing the names of all molecules which should be considered
                            or an (ASCII) file name including the molecules which should be considered.

                            If this parameter is left empty, all molecules in the database within the given
                            range are considered!

                            Note, if the parameter defines a relative path, this path has to be defined
                            relative to the current working directory!

    - FrequencyWidth:       defines the width of the frequency interval in MHz (selected frequency
                            +/- "FrequencyWidth" where the line information are printed out to the screen.
                            (default: 5).

    - ElowMin:              minimum for lower energy (in K) (default: 0) in the "Transitions" table

    - ElowMax:              maximum for lower energy (in K) (default: 1.e6) in the "Transitions" table

    - modeldata:            2D numpy array containing the modeled data

                            Note, the data has to be given as a function of frequency (in MHz).



Example 1:
----------

FileName = "demo/LoadASCIIFile/ASCII.dat"
NumHeaderLines = 0
obsdata = LoadASCIIFile()

FreqMin = 4.92100000e+05
FreqMax = 4.92200000e+05
SelectMolecule = "demo/ListDatabase/molecules.txt"
FrequencyWidth = 5.0
ElowMin = 100.0
ElowMax = 1000.0
GetTransitions()


Example 2:
----------

FileName = "demo/LoadASCIIFile/ASCII.dat"
NumHeaderLines = 0
obsdata = LoadASCIIFile()

FreqMin = 4.92100000e+05
FreqMax = 4.92200000e+05
SelectMolecule = ["HNO3;v5=1;", "C2H5CN-15;v=0;"]
FrequencyWidth = 5.0
ElowMin = 100.0
ElowMax = 1000.0
GetTransitions()
    """

    # Debug:
    # print ("FreqMin = ", FreqMin)
    # print ("FreqMax = ", FreqMax)
    # print ("SelectMolecule = ", SelectMolecule)
    # print ("FrequencyWidth = ", FrequencyWidth)
    # print ("ElowMin = ", ElowMin)
    # print ("ElowMax = ", ElowMax)
    # print ("modeldata = ", modeldata)


    ## call GetTransitionsCore function
    GetTransitionsCore(obsdata = obsdata, FreqMin = FreqMin, FreqMax = FreqMax, SelectMolecule = SelectMolecule,
                       FrequencyWidth = FrequencyWidth, ElowMin = ElowMin, ElowMax = ElowMax, modeldata = modeldata)


    ##====================================================================================================================================================
    ## define return variables
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------
