#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  emcee package
##
##  Copyright (C) 2012 - 2013, Dan Foreman-Mackey & contributors.
##
##  http://dan.iel.fm/emcee/current/
##
##
##
##  The following subroutines and functions are included in this module:
##
##      Corner package:
##
##      - subroutine corner:                            Make a *sick* corner plot showing the projections of a data set in a multi-dimensional space.
##                                                      kwargs are passed to hist2d() or used for `matplotlib` styling.
##      - subroutine quantile:                          like numpy.percentile
##      - subroutine hist2d:                            plot a 2-D histogram of samples.
##
##      MCMC credible regions (taken from http://bebi103.caltech.edu/2015/tutorials/l06_credible_regions.html):
##
##      - subroutine hpd:                               Returns highest probability density region given by a set of samples.
##      - subroutine hpdPyMC:                           Calculate highest posterior density (HPD) of array for given alpha. The HPD is the minimum
##                                                      width Bayesian credible interval (BCI).
##      - subroutine calc_min_interval:                 Internal method to determine the minimum interval of a given width.
##
##
##
##  Versions of the program:
##
##  Who                                 When            What
##
##  Dan Foreman-Mackey & contributors   2012 - 2014     initial version
##  T. Moeller                          2016-05-19      improved documentation, modifications for XCLASS
##  F. Eupen                            2019-06-05      include official bug-fix (typo)
##  T. Moeller                          2020-01-02      porting to python 3, minor improvments
##
##
##
##  License:
##
##    The MIT License (MIT)
##
##    Copyright (c) 2010 - 2013 Daniel Foreman-Mackey & contributors.
##
##    Permission is hereby granted, free of charge, to any person obtaining a copy
##    of this software and associated documentation files (the "Software"), to deal
##    in the Software without restriction, including without limitation the rights
##    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
##    copies of the Software, and to permit persons to whom the Software is
##    furnished to do so, subject to the following conditions:
##
##    The above copyright notice and this permission notice shall be included in all
##    copies or substantial portions of the Software.
##
##    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
##    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
##    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
##    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
##    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
##    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
##    SOFTWARE.
##
##********************************************************************************************************************************************************


##******************************************************************** load packages *********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy as np                                                                          ## load numpy package
import logging                                                                              ## load python package logging
import matplotlib.pyplot as pl                                                              ## load python package matplotlib.pyplot
from matplotlib.ticker import MaxNLocator                                                   ## load python package matplotlib.MaxNLocator
from matplotlib.colors import LinearSegmentedColormap, colorConverter                       ## load python package matplotlib.LinearSegmentedColormap
from matplotlib.ticker import ScalarFormatter                                               ## load python package ScalarFormatter
try:
    from scipy.ndimage import gaussian_filter                                               ## load python package gaussian_filter
except ImportError:
    gaussian_filter = None
##********************************************************************************************************************************************************


##********************************************************************************************************************************************************
## corner package
def corner(xs, bins = 20, range = None, weights = None, color = "k", smooth = None, smooth1d = None, labels = None, label_kwargs = None, \
           show_titles = False, title_fmt = ".2f", title_kwargs = None, truths = None, truth_color = "#4682b4", scale_hist = False, \
           quantiles = None, ErrorTyp = "hpd", verbose = False, fig = None, max_n_ticks = 5, top_ticks = False, use_math_text = False, \
           useOffset = False, hist_kwargs = None, **hist2d_kwargs):
    """
    Make a *sick* corner plot showing the projections of a data set in a
    multi-dimensional space. kwargs are passed to hist2d() or used for
    `matplotlib` styling.

    taken from
    https://pypi.python.org/pypi/corner, call May 22th 2016

    Parameters
    ----------
    xs : array_like (nsamples, ndim)
        The samples. This should be a 1- or 2-dimensional array. For a 1-D
        array this results in a simple histogram. For a 2-D array, the zeroth
        axis is the list of samples and the next axis are the dimensions of
        the space.

    bins : int or array_like (ndim,) (optional)
        The number of bins to use in histograms, either as a fixed value for
        all dimensions, or as a list of integers for each dimension.

    weights : array_like (nsamples,)
        The weight of each sample. If `None` (default), samples are given
        equal weight.

    color : str (optional)
        A ``matplotlib`` style color for all histograms.

    smooth, smooth1d : float (optional)
       The standard deviation for Gaussian kernel passed to
       `scipy.ndimage.gaussian_filter` to smooth the 2-D and 1-D histograms
       respectively. If `None` (default), no smoothing is applied.

    labels : iterable (ndim,) (optional)
        A list of names for the dimensions. If a ``xs`` is a
        ``pandas.DataFrame``, labels will default to column names.

    label_kwargs : dict (optional)
        Any extra keyword arguments to send to the `set_xlabel` and
        `set_ylabel` methods.

    show_titles : bool (optional)
        Displays a title above each 1-D histogram showing the 0.5 quantile
        with the upper and lower errors supplied by the quantiles argument.

    title_fmt : string (optional)
        The format string for the quantiles given in titles. If you explicitly
        set ``show_titles=True`` and ``title_fmt=None``, the labels will be
        shown as the titles. (default: ``.2f``)

    title_kwargs : dict (optional)
        Any extra keyword arguments to send to the `set_title` command.

    range : iterable (ndim,) (optional)
        A list where each element is either a length 2 tuple containing
        lower and upper bounds or a float in range (0., 1.)
        giving the fraction of samples to include in bounds, e.g.,
        [(0.,10.), (1.,5), 0.999, etc.].
        If a fraction, the bounds are chosen to be equal-tailed.

    truths : iterable (ndim,) (optional)
        A list of reference values to indicate on the plots.  Individual
        values can be omitted by using ``None``.

    truth_color : str (optional)
        A ``matplotlib`` style color for the ``truths`` makers.

    scale_hist : bool (optional)
        Should the 1-D histograms be scaled in such a way that the zero line
        is visible?

    quantiles : iterable (optional)
        A list of fractional quantiles to show on the 1-D histograms as
        vertical dashed lines.

    ErrorTyp: str (optional)
        defines the type of error (``gauss``, ``quantile``, ``hpd``)

    verbose : bool (optional)
        If true, print the values of the computed quantiles.

    plot_contours : bool (optional)
        Draw contours for dense regions of the plot.

    use_math_text : bool (optional)
        If true, then axis tick labels for very large or small exponents will
        be displayed as powers of 10 rather than using `e`.

    useOffset : bool (optional)
        If true, the offset is calculated as needed. If false, no offset is
        used. If a numeric value, it sets the offset. The formatter default
        is rcParams["axes.formatter.useoffset"] (default: False).

    max_n_ticks: int (optional)
        Maximum number of ticks to try to use

    top_ticks : bool (optional)
        If true, label the top ticks of each axis

    fig : matplotlib.Figure (optional)
        Overplot onto the provided figure object.

    hist_kwargs : dict (optional)
        Any extra keyword arguments to send to the 1-D histogram plots.

    **hist2d_kwargs : (optional)
        Any remaining keyword arguments are sent to `corner.hist2d` to generate
        the 2-D histogram plots.
    """

    # Debug:
    # print ("xs = ", xs)
    # print ("bins = ", bins)
    # print ("range = ", range)
    # print ("weights = ", weights)
    # print ("color = ", color)
    # print ("smooth = ", smooth)
    # print ("smooth1d = ", smooth1d)
    # print ("labels = ", labels)
    # print ("label_kwargs = ", label_kwargs)
    # print ("show_titles = ", show_titles)
    # print ("title_fmt = ", title_fmt)
    # print ("title_kwargs = ", title_kwargs)
    # print ("truths = ", truths)
    # print ("truth_color = ", truth_color)
    # print ("scale_hist = ", scale_hist)
    # print ("quantiles = ", quantiles)
    # print ("ErrorTyp = ", ErrorTyp)
    # print ("verbose = ", verbose)
    # print ("fig = ", fig)
    # print ("max_n_ticks = ", max_n_ticks)
    # print ("top_ticks = ", top_ticks)
    # print ("use_math_text = ", use_math_text)
    # print ("useOffset = ", useOffset)
    # print ("hist_kwargs = ", hist_kwargs)


    if quantiles is None:
        quantiles = []
    if title_kwargs is None:
        title_kwargs = dict()
    if label_kwargs is None:
        label_kwargs = dict()


    ## Try filling in labels from pandas.DataFrame columns.
    if labels is None:
        try:
            labels = xs.columns
        except AttributeError:
            pass


    ## Deal with 1D sample lists.
    xs = np.atleast_1d(xs)
    if len(xs.shape) == 1:
        xs = np.atleast_2d(xs)
    else:
        assert len(xs.shape) == 2, "The input sample array must be 1- or 2-D."
        xs = xs.T
    assert xs.shape[0] <= xs.shape[1], "I don't believe that you want more dimensions than samples!"


    ## Parse the weight array.
    if weights is not None:
        weights = np.asarray(weights)
        if weights.ndim != 1:
            raise ValueError("Weights must be 1-D")
        if xs.shape[1] != weights.shape[0]:
            raise ValueError("Lengths of weights must match number of samples")


    ## Parse the parameter ranges.
    if range is None:
        if "extents" in hist2d_kwargs:
            logging.warn("Deprecated keyword argument 'extents'. Use 'range' instead.")
            range = hist2d_kwargs.pop("extents")
        else:
            range = [[x.min(), x.max()] for x in xs]


            ## Check for parameters that never change.
            m = np.array([e[0] == e[1] for e in range], dtype = bool)
            # if np.any(m):
            #     raise ValueError(("It looks like the parameter(s) in column(s) {0} have no dynamic range. "
            #                       "Please provide a `range` argument.")
            #                      .format(", ".join(map("{0}".format, np.arange(len(m))[m]))))

    else:
        ## If any of the extents are percentiles, convert them to ranges. Also make sure it's a normal list.
        range = list(range)
        for i, _ in enumerate(range):
            try:
                emin, emax = range[i]
            except TypeError:
                q = [0.5 - 0.5*range[i], 0.5 + 0.5*range[i]]
                range[i] = quantile(xs[i], q, weights = weights)


    if len(range) != xs.shape[0]:
        raise ValueError("Dimension mismatch between samples and range")


    ## Parse the bin specifications.
    try:
        bins = [int(bins) for _ in range]
    except TypeError:
        if len(bins) != len(range):
            raise ValueError("Dimension mismatch between bins and range")


    ## Some magic numbers for pretty axis layout.
    K = len(xs)
    factor = 2.0                                                                            ## size of one side of one panel
    lbdim = 0.5 * factor                                                                    ## size of left/bottom margin
    trdim = 0.2 * factor                                                                    ## size of top/right margin
    whspace = 0.05                                                                          ## w/hspace size
    plotdim = factor * K + factor * (K - 1.) * whspace
    dim = lbdim + plotdim + trdim


    ## Create a new figure if one wasn't provided.
    if fig is None:
        fig, axes = pl.subplots(K, K, figsize = (dim, dim))
    else:
        try:
            axes = np.array(fig.axes).reshape((K, K))
        except:
            raise ValueError("Provided figure has {0} axes, but data has dimensions K={1}".format(len(fig.axes), K))


    ## Format the figure.
    lb = lbdim / dim
    tr = (lbdim + plotdim) / dim
    fig.subplots_adjust(left = lb, bottom = lb, right = tr, top = tr, wspace = whspace, hspace = whspace)


    ## Set up the default histogram keywords.
    if hist_kwargs is None:
        hist_kwargs = dict()
    hist_kwargs["color"] = hist_kwargs.get("color", color)
    if smooth1d is None:
        hist_kwargs["histtype"] = hist_kwargs.get("histtype", "step")
    for i, x in enumerate(xs):


        ## Deal with masked arrays.
        if hasattr(x, "compressed"):
            x = x.compressed()


        if np.shape(xs)[0] == 1:
            ax = axes
        else:
            ax = axes[i, i]


        ## Plot the histograms.
        if smooth1d is None:
            n, _, _ = ax.hist(x, bins = bins[i], weights = weights, range = np.sort(range[i]), **hist_kwargs)
        else:
            if gaussian_filter is None:
                raise ImportError("Please install scipy for smoothing")
            n, b = np.histogram(x, bins = bins[i], weights = weights, range = np.sort(range[i]))
            n = gaussian_filter(n, smooth1d)
            x0 = np.array(list(zip(b[:-1], b[1:]))).flatten()
            y0 = np.array(list(zip(n, n))).flatten()
            ax.plot(x0, y0, **hist_kwargs)
        if truths is not None and truths[i] is not None:
            ax.axvline(truths[i], color=truth_color)


        ## Plot quantiles if wanted.
        if len(quantiles[i]) > 0:
            if (ErrorTyp != "quantile"):
                for q in quantiles[i]:
                    ax.axvline(q, ls="dashed", color=color)
            else:
                qvalues = quantile(x, quantiles[i], weights=weights)
                for q in qvalues:
                    ax.axvline(q, ls="dashed", color=color)
                if verbose:
                    print ("Quantiles:")
                    print ([item for item in zip(quantiles[i], qvalues)])


        if show_titles:
            title = None
            if title_fmt is not None:


                ## Compute the quantiles for the title. This might redo unneeded computation but who cares.
                if (ErrorTyp == "quantile"):
                    qLeft, qMiddle, qRight = quantile(x, quantiles[i], weights = weights)
                else:
                    qLeft = quantiles[i][0]
                    qMiddle = quantiles[i][1]
                    qRight = quantiles[i][2]
                q_m, q_p = qMiddle - qLeft, qRight - qMiddle


                ## Format the quantile display.
                fmt = "{{0:{0}}}".format(title_fmt).format
                title = r"${{{0}}}_{{-{1}}}^{{+{2}}}$"
                title = title.format(fmt(qMiddle), fmt(q_m), fmt(q_p))


                ## Add in the column name if it's given.
                if labels is not None:
                    title = "{0}:\n{1}".format(labels[i], title)


            elif labels is not None:
                title = "{0}".format(labels[i])


            if title is not None:
                ax.set_title(title, **title_kwargs)


        ## Set up the axes.
        ax.set_xlim(range[i])
        if scale_hist:
            maxn = np.max(n)
            ax.set_ylim(-0.1 * maxn, 1.1 * maxn)
        else:
            ax.set_ylim(0, 1.1 * np.max(n))
        ax.set_yticklabels([])
        ax.xaxis.set_major_locator(MaxNLocator(max_n_ticks, prune = "lower"))
        if i < K - 1:
            if top_ticks:
                ax.xaxis.set_ticks_position("top")
                [l.set_rotation(45) for l in ax.get_xticklabels()]
            else:
                ax.set_xticklabels([])
        else:
            [l.set_rotation(45) for l in ax.get_xticklabels()]
            if labels is not None:
                ax.set_xlabel(labels[i], **label_kwargs)
                ax.xaxis.set_label_coords(0.5, -0.3)


            ## use MathText for axes ticks
            ax.xaxis.set_major_formatter(
                ScalarFormatter(useMathText = use_math_text, useOffset = useOffset))


        for j, y in enumerate(xs):
            if np.shape(xs)[0] == 1:
                ax = axes
            else:
                ax = axes[i, j]
            if j > i:
                ax.set_frame_on(False)
                ax.set_xticks([])
                ax.set_yticks([])
                continue
            elif j == i:
                continue


            ## Deal with masked arrays.
            if hasattr(y, "compressed"):
                y = y.compressed()
            hist2d(y, x, ax = ax, range = [range[j], range[i]], weights = weights, color = color, \
                   smooth = smooth, bins = [bins[j], bins[i]], **hist2d_kwargs)

            if truths is not None:
                if truths[i] is not None and truths[j] is not None:
                    ax.plot(truths[j], truths[i], "s", color = truth_color)
                if truths[j] is not None:
                    ax.axvline(truths[j], color = truth_color)
                if truths[i] is not None:
                    ax.axhline(truths[i], color = truth_color)

            ax.xaxis.set_major_locator(MaxNLocator(max_n_ticks, prune = "lower"))
            ax.yaxis.set_major_locator(MaxNLocator(max_n_ticks, prune = "lower"))

            if i < K - 1:
                ax.set_xticklabels([])
            else:
                [l.set_rotation(45) for l in ax.get_xticklabels()]
                if labels is not None:
                    ax.set_xlabel(labels[j], **label_kwargs)
                    ax.xaxis.set_label_coords(0.5, -0.3)


                ## use MathText for axes ticks
                ax.xaxis.set_major_formatter(ScalarFormatter(useMathText = use_math_text, useOffset = useOffset))


            if j > 0:
                ax.set_yticklabels([])
            else:
                [l.set_rotation(45) for l in ax.get_yticklabels()]
                if labels is not None:
                    ax.set_ylabel(labels[i], **label_kwargs)
                    ax.yaxis.set_label_coords(-0.3, 0.5)


                ## use MathText for axes ticks
                ax.yaxis.set_major_formatter(ScalarFormatter(useMathText = use_math_text, useOffset = useOffset))


    ## define return parameter
    return fig
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Like numpy.percentile
def quantile(x, q, weights = None):
    """
    Compute sample quantiles with support for weighted samples.

    Note
    ----
    When ``weights`` is ``None``, this method simply calls numpy's percentile
    function with the values of ``q`` multiplied by 100.

    Parameters
    ----------
    x : array_like[nsamples,]
       The samples.

    q : array_like[nquantiles,]
       The list of quantiles to compute. These should all be in the range
       ``[0, 1]``.

    weights : Optional[array_like[nsamples,]]
        An optional weight corresponding to each sample. These

    Returns
    -------
    quantiles : array_like[nquantiles,]
        The sample quantiles computed at ``q``.

    Raises
    ------
    ValueError
        For invalid quantiles; ``q`` not in ``[0, 1]`` or dimension mismatch
        between ``x`` and ``weights``.
    """

    # Debug:
    # print ("x = ", x)
    # print ("q = ", q)
    # print ("weights = ", weights)


    x = np.atleast_1d(x)
    q = np.atleast_1d(q)

    if np.any(q < 0.0) or np.any(q > 1.0):
        raise ValueError("Quantiles must be between 0 and 1")

    if weights is None:
        return np.percentile(x, 100.0 * q)
    else:
        weights = np.atleast_1d(weights)
        if len(x) != len(weights):
            raise ValueError("Dimension mismatch: len(weights) != len(x)")
        idx = np.argsort(x)
        sw = weights[idx]
        cdf = np.cumsum(sw)[:-1]
        cdf /= cdf[-1]
        cdf = np.append(0, cdf)
        return np.interp(q, cdf, x[idx]).tolist()


    ## we're done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Plot a 2-D histogram of samples.
def hist2d(x, y, bins = 20, range = None, weights = None, levels = None, smooth = None, ax = None, color = None, plot_datapoints = True, \
           plot_density = True, plot_contours = True, no_fill_contours = False, fill_contours = False, contour_kwargs = None, contourf_kwargs = None, \
           data_kwargs = None, **kwargs):
    """

    Parameters
    ----------
    x : array_like[nsamples,]
       The samples.

    y : array_like[nsamples,]
       The samples.

    levels : array_like
        The contour levels to draw.

    ax : matplotlib.Axes (optional)
        A axes instance on which to add the 2-D histogram.

    plot_datapoints : bool (optional)
        Draw the individual data points.

    plot_density : bool (optional)
        Draw the density colormap.

    plot_contours : bool (optional)
        Draw the contours.

    no_fill_contours : bool (optional)
        Add no filling at all to the contours (unlike setting
        ``fill_contours=False``, which still adds a white fill at the densest
        points).

    fill_contours : bool (optional)
        Fill the contours.

    contour_kwargs : dict (optional)
        Any additional keyword arguments to pass to the `contour` method.

    contourf_kwargs : dict (optional)
        Any additional keyword arguments to pass to the `contourf` method.

    data_kwargs : dict (optional)
        Any additional keyword arguments to pass to the `plot` method when
        adding the individual data points.
    """

    # Debug:
    # print ("x = ", x)
    # print ("y = ", y)
    # print ("bins = ", bins)
    # print ("range = ", range)
    # print ("weights = ", weights)
    # print ("levels = ", levels)
    # print ("smooth = ", smooth)
    # print ("ax = ", ax)
    # print ("color = ", color)
    # print ("plot_datapoints = ", plot_datapoints)
    # print ("plot_density = ", plot_density)
    # print ("plot_contours = ", plot_contours)
    # print ("no_fill_contours = ", no_fill_contours)
    # print ("fill_contours = ", fill_contours)
    # print ("contour_kwargs = ", contour_kwargs)
    # print ("contourf_kwargs = ", contourf_kwargs)
    # print ("data_kwargs = ", data_kwargs)


    if ax is None:
        ax = pl.gca()


    ## Set the default range based on the data range if not provided.
    if range is None:
        if "extent" in kwargs:
            logging.warn("Deprecated keyword argument 'extent'. "
                         "Use 'range' instead.")
            range = kwargs["extent"]
        else:
            range = [[x.min(), x.max()], [y.min(), y.max()]]


    ## Set up the default plotting arguments.
    if color is None:
        color = "k"


    ## Choose the default "sigma" contour levels.
    if levels is None:
        levels = 1.0 - np.exp(-0.5 * np.arange(0.5, 2.1, 0.5) ** 2)


    ## This is the color map for the density plot, over-plotted to indicate the density of the points near the center.
    density_cmap = LinearSegmentedColormap.from_list("density_cmap", [color, (1, 1, 1, 0)])


    ## This color map is used to hide the points at the high density areas.
    white_cmap = LinearSegmentedColormap.from_list("white_cmap", [(1, 1, 1), (1, 1, 1)], N=2)


    ## This "color map" is the list of colors for the contour levels if the contours are filled.
    rgba_color = colorConverter.to_rgba(color)
    contour_cmap = [list(rgba_color) for l in levels] + [rgba_color]
    for i, l in enumerate(levels):
        contour_cmap[i][-1] *= float(i) / (len(levels)+1)


    ## We'll make the 2D histogram to directly estimate the density.
    try:
        H, X, Y = np.histogram2d(x.flatten(), y.flatten(), bins=bins, range=list(map(np.sort, range)), weights=weights)
    except ValueError:
        raise ValueError("It looks like at least one of your sample columns "
                         "have no dynamic range. You could try using the "
                         "'range' argument.")

    if smooth is not None:
        if gaussian_filter is None:
            raise ImportError("Please install scipy for smoothing")
        H = gaussian_filter(H, smooth)


    ## Compute the density levels.
    Hflat = H.flatten()
    inds = np.argsort(Hflat)[::-1]
    Hflat = Hflat[inds]
    sm = np.cumsum(Hflat)
    sm /= sm[-1]
    V = np.empty(len(levels))
    for i, v0 in enumerate(levels):
        try:
            V[i] = Hflat[sm <= v0][-1]
        except:
            V[i] = Hflat[0]
    V.sort()
    m = np.diff(V) == 0
    if np.any(m):
        logging.warning("Too few points to create valid contours")
    while np.any(m):
        V[np.where(m)[0][0]] *= 1.0 - 1e-4
        m = np.diff(V) == 0
    V.sort()


    ## Compute the bin centers.
    X1, Y1 = 0.5 * (X[1:] + X[:-1]), 0.5 * (Y[1:] + Y[:-1])


    ## Extend the array for the sake of the contours at the plot edges.
    H2 = H.min() + np.zeros((H.shape[0] + 4, H.shape[1] + 4))
    H2[2:-2, 2:-2] = H
    H2[2:-2, 1] = H[:, 0]
    H2[2:-2, -2] = H[:, -1]
    H2[1, 2:-2] = H[0]
    H2[-2, 2:-2] = H[-1]
    H2[1, 1] = H[0, 0]
    H2[1, -2] = H[0, -1]
    H2[-2, 1] = H[-1, 0]
    H2[-2, -2] = H[-1, -1]
    X2 = np.concatenate([X1[0] + np.array([-2, -1]) * np.diff(X1[:2]), X1, X1[-1] + np.array([1, 2]) * np.diff(X1[-2:]), ])
    Y2 = np.concatenate([Y1[0] + np.array([-2, -1]) * np.diff(Y1[:2]), Y1, Y1[-1] + np.array([1, 2]) * np.diff(Y1[-2:]), ])

    if plot_datapoints:
        if data_kwargs is None:
            data_kwargs = dict()
        data_kwargs["color"] = data_kwargs.get("color", color)
        data_kwargs["ms"] = data_kwargs.get("ms", 2.0)
        data_kwargs["mec"] = data_kwargs.get("mec", "none")
        data_kwargs["alpha"] = data_kwargs.get("alpha", 0.1)
        ax.plot(x, y, "o", zorder=-1, rasterized=True, **data_kwargs)


    ## Plot the base fill to hide the densest data points.
    if ((plot_contours or plot_density) and not no_fill_contours):
        try:
            ax.contourf(X2, Y2, H2.T, [V.min(), H.max()], cmap = white_cmap, antialiased = False)
        except:
            pass

    if plot_contours and fill_contours:
        if contourf_kwargs is None:
            contourf_kwargs = dict()
        contourf_kwargs["colors"] = contourf_kwargs.get("colors", contour_cmap)
        contourf_kwargs["antialiased"] = contourf_kwargs.get("antialiased", False)
        ax.contourf(X2, Y2, H2.T, np.concatenate([[0], V, [H.max()*(1+1e-4)]]), **contourf_kwargs)


    ## Plot the density map. This can't be plotted at the same time as the contour fills.
    elif plot_density:
        ax.pcolor(X, Y, H.max() - H.T, cmap=density_cmap)


    ## Plot the contour edge colors.
    if plot_contours:
        if contour_kwargs is None:
            contour_kwargs = dict()
        contour_kwargs["colors"] = contour_kwargs.get("colors", color)
        ax.contour(X2, Y2, H2.T, V, **contour_kwargs)

    ax.set_xlim(range[0])
    ax.set_ylim(range[1])


    ## we're done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Returns highest probability density region given by a set of samples.
def hpd(trace, mass_frac) :
    """

    taken from
    http://bebi103.caltech.edu/2015/tutorials/l06_credible_regions.html

    Computation of the HPD is a little trickier. The function below will compute the HPD interval. The idea is that we rank-order the MCMC trace.
    We know that the number of samples that are included in the HPD is 0.95 times the total number of MCMC sample. We then consider all intervals
    that contain that many samples and find the shortest one.


    Parameters
    ----------
    trace : array
        1D array of MCMC samples for a single variable
    mass_frac : float with 0 < mass_frac <= 1
        The fraction of the probability to be included in
        the HPD.  For example, `massfrac` = 0.95 gives a
        95% HPD.

    Returns
    -------
    output : array, shape (2,)
        The bounds of the HPD
    """

    # Debug:
    # print ("trace = ", trace)
    # print ("mass_frac = ", mass_frac)


    ## Get sorted list
    d = np.sort(np.copy(trace))

    # Debug:
    # print ("d = ", d)
    # print ("d[0] = ", d[0])


    ## Number of total samples taken
    n = len(trace)

    # Debug:
    # print ("n = ", n)


    ## Get number of samples that should be included in HPD
    n_samples = np.floor(mass_frac * n).astype(int)

    # Debug:
    # print ("n_samples = ", n_samples)


    ## Get width (in units of data) of all intervals with n_samples samples
    int_width = d[n_samples:] - d[:n-n_samples]

    # Debug:
    # print ("int_width = ", int_width)


    ## Pick out minimal interval
    min_int = np.argmin(int_width)
    dMin = max(0, min(min_int, (n - 1)))
    dMax = max(0, min(min_int + n_samples, (n - 1)))

    # Debug:
    # print ("min_int = ", min_int)
    # print ("dMin = ", dMin)
    # print ("dMax = ", dMax)


    ## Return interval
    return np.array([d[dMin], d[dMax]])
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Calculate highest posterior density (HPD) of array for given alpha. The HPD is the minimum width Bayesian credible interval (BCI).
def hpdPyMC(y, alpha):
    """

    taken from:
    https://github.com/pymc-devs/pymc/blob/master/pymc/utils.py


    Arguments:
      x : Numpy array
          An array containing MCMC samples
      alpha : float
          Desired probability of type I error
    """

    # Debug:
    # print ("y = ", y)
    # print ("alpha = ", alpha)


    ## Make a copy of trace
    x = y.copy()

    # Debug:
    # print ("x = ", x)
    # print ("x.ndim = ", x.ndim)


    ## For multivariate node
    if (x.ndim > 1):
        from numpy import (sqrt, ndarray, asmatrix, array, prod,
                           asarray, atleast_1d, iterable, linspace, diff,
                           around, log10, zeros, arange, digitize, apply_along_axis,
                           concatenate, bincount, sort, hsplit, argsort, inf, shape,
                           ndim, swapaxes, ravel, diag, cov, transpose as tr)


        ## Transpose first, then sort
        tx = tr(x, list(range(x.ndim)[1:]) + [0])
        dims = shape(tx)


        ## Container list for intervals
        intervals = np.resize(0.0, (2,) + dims[:-1])
        for index in make_indices(dims[:-1]):
            try:
                index = tuple(index)
            except TypeError:
                pass


            ## Sort trace
            sx = sort(tx[index])


            ## Append to list
            intervals[0][index], intervals[1][index] = calc_min_interval(sx, alpha)


        ## Transpose back before returning
        return array(intervals)

    else:
        ## Sort univariate node
        sx = sort(x)


        ## Return interval
        return array(calc_min_interval(sx, alpha))
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Internal method to determine the minimum interval of a given width. Assumes that x is sorted numpy array.
def calc_min_interval(x, alpha):
    """

    taken from:
    https://github.com/pymc-devs/pymc/blob/master/pymc/utils.py


    """

    # Debug:
    # print ("x = ", x)
    # print ("alpha = ", alpha)


    n = len(x)
    cred_mass = 1.0 - alpha


    interval_idx_inc = int(np.floor(cred_mass * n))
    n_intervals = n - interval_idx_inc
    interval_width = x[interval_idx_inc:] - x[:n_intervals]

    if len(interval_width) == 0:
        print ('Too few elements for interval calculation')
        return [None, None]

    min_idx = np.argmin(interval_width)
    hdi_min = x[min_idx]
    hdi_max = x[min_idx + interval_idx_inc]


    ## define return parameter
    return [hdi_min, hdi_max]
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Generates complete set of indices for given dimensions
def make_indices(dimensions):
    """

    taken from:
    https://github.com/pymc-devs/pymc/blob/master/pymc/utils.py


    """

    # Debug:
    # print ("dimensions = ", dimensions)


    level = len(dimensions)

    if level == 1:
        return list(range(dimensions[0]))

    indices = [[]]

    while level:

        _indices = []

        for j in range(dimensions[level - 1]):

            _indices += [[j] + i for i in indices]

        indices = _indices

        level -= 1

    try:
        return [tuple(i) for i in indices]
    except TypeError:


        ## define return parameter
        return indices
##--------------------------------------------------------------------------------------------------------------------------------------------------------

