#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module controls the whole program and calls the other modules.
##  Furthermore it manages the communication between the subpackages.
##  Copyright (C) 2009 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - subroutine ReadParameter:                     read xml-file containing the start values for the fit process
##      - subroutine read_control:                      Loads i/o control file, which contains the path and the file name of several files. Furthermore,
##                                                      the existence of the declared files are checked.
##      - subroutine main                               calls the other modules and manages the communication between these modules
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    2009-06-22   initial version
##  T. Moeller    2012-01-16   improve documentation of source code
##  T. Moeller    2020-01-02   porting to python 3.x, minor improvements
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************

"""
Main program:
-------------

The module "MainProgram" is the central module of MAGIX, because it controls
the whole program and calls the other modules. Furthermore it manages the
communication between the other modules.

MAGIX requires a so-called i/o control file which is located in the same
directory where the "magix_start.py" file is. If you want to use an i/o
control file located somewhere else, enter the path and the file name of the
i/o control file in the following of the magix_start command.

For example: "./magix_start.py test/io_control.xml"

This command starts the MAGIX program and uses the i/o control file located
in the test directory with file name "i/o_control.xml".

The i/o control file is loaded by the subroutine "read_control" and must
contain the path and the file names of

- the xml-file necessary for the import of the experimental data,
- the xml-file containing the fit parameters,
- the control file containing settings for the fit procedure,
- the log-file used during the fit process,
- the xml-file describing the input file being used by the external program


The i/o control file must contain the following tags:

- "experimental_data":  Path and name of the xml-file containing import
                        settings for the experimental file(s). If the user
                        specify the path and file name of a dat or fits file
                        default settings for the import are used.
- "parameter_file":     Path and name of the xml-file containing the
                        model parameters. (File name has to end with ".xml").
- "fit_control":        Path and name of the fit control file containing
                        the settings for the fit process. (File name has to end
                        with ".fit").
- "fit_log":            Path and name of the fit log file, which contains
                        all information of each step of the fit process. (File
                        name has to end with ".log").
- "model_description":  Path and name of the xml-file describing the input
                        file of the used external program, which determines the
                        fit function


Example for the i/o control file:
---------------------------------


<?xml version="1.0" encoding="ISO-8859-1"?>
<ioControl>
    <title>io control</title>
    <description>This xml-file contains the paths and the file names of all working files which are used by MAGIX during the fit process</description>
    <PathFilename>

        <experimental_data>
            <filename>test/conv_Drude-Lorentz-model/One_OscillatorRefFit_R.xml</filename>
            <description>path and file name of the experimental file</description>
        </experimental_data>

        <parameter_file>
            <filename>test/conv_Drude-Lorentz-model/parameters.xml</filename>
            <description>path and file name of the xml-file including the start values of each parameter</description>
        </parameter_file>

        <fit_control>
            <filename>test/conv_Drude-Lorentz-model/Levenberg-Marquardt_Parameters.xml</filename>
            <description>path and file name of the xml file controlling the fitting process</description>
        </fit_control>

        <fit_log>
            <filename>test/conv_Drude-Lorentz-model/Levenberg-Marquardt/fit.log</filename>
            <description>path and file name of log file describing the fitting process</description>
        </fit_log>

        <model_description>
            <filename>Fit-Functions/Drude-Lorentz_conv/xml/Conventional_Drude-Lorentz.xml</filename>
            <description>path and file name of the xml-description of the input/output file of the fit function module</description>
        </model_description>

    </PathFilename>
</ioControl>


IMPORTANT:
----------

- MAGIX distinguishes between upper and lower case of the tags!

- the description tags are not read by MAGIX and can contain arbitrary information

- MAGIX will check the existence of the files given by the command words
  "experimental_data", "parameter_file" and "fit_control" and "model_description"
  and will abort if one of these files do not exist. The file given by "fit_log" need
  not to exist but the existence of the path will be checked by MAGIX. A non-existence
  of the path will abort MAGIX, too. Within the fit process a log-file with file name
  given by "fit_log" will be created.


The subroutine "main" calls the following modules:

- LoadExpFile, for loading the experimental data,
- GetParameterSet for getting the model parameters,
- FittingEngine for optimizing the model parameters by using several fit
  algorithms.
- PlotData, for primitive plotting of the experimental data together with
  the fit function
"""


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import os                                                                                   ## import os package
import os.path                                                                              ## load package for file management
import sys                                                                                  ## import sys package
from xclass.addons.MAGIX.Modules.python import XMLPackage
from xclass.addons.MAGIX.Modules.python import LoadExpFile
from xclass.addons.MAGIX.Modules.python import GetParameterSet
from xclass.addons.MAGIX.Modules.python import FittingEngine
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Loads i/o control file, which contains the path and the file name of several files. Furthermore, the existence of the declared files are checked.
##
def read_control(printflag, filename_control):
    """
    input parameters:       filename_control:           path and file name of the i/o control file
                            printflag:                  print flag

    output paramters:       ok:                         status of subroutine: ok = 0 everything is ok, != 0: an error occurs
                            experimental_data:          path and file name of the xml-file containing import settings for the experimental
                                                        file(s)
                            parameter_file:             path and file name of the xml-file containing the fit parameters
                            parameter_file_orig:        copy of the string including the path and file name of the file containing the
                                                        fit parameters. This copy is necessary to detect the orginal format of the model
                                                        input file.
                            fit_control:                path and file name of the control file containing settings for the fit procedure
                            fit_log:                    path and file name of the log-file used during the fit process
                            XmlFileInOutput:            path and file name of the xml-description of the input/output file

    working variables:      XmlFile:                    contains the hole content of the xml-file
                            i:                          index for remark character '#'
                            ii:                         counter variable
    """

    # Debug:
    # print ("printflag = ", printflag)
    # print ("filename_control = ", filename_control)


    ## initialize return parameter
    ok = 0
    from xclass import task_myXCLASS                                                        ## import package task_myXCLASS


    ## get filename
    if not (os.path.exists(filename_control)):
        print ("\n\t  Error in subroutine read_control() !")
        print ("\t      The file ", filename_control)
        print ("\t      does not exists!\n")
        ok = 1
        sys.exit(0)


    ## read xml-description of the INPUT file is the defined file a xml-file ?
    if not filename_control.endswith(".xml"):
        print ("\n\t Error in subroutine read_control()!")
        print ("\t     The selected file is not a xml-file!")
        print ("\t     Please select a .xml and restart the whole program.\n")
        ok = 1
        sys.exit(0)


    ## print what you do ..
    if (printflag != "false"):
        print ("\nImport i/o control file:")
        print ("\t Open i/o control file: " + chr(34) + filename_control + chr(34))
        print ("\t Import settings .. ", end = "")


    ## open xml-file file and save content of file to variable XmlFile
    IOFile = XMLPackage.XMLRoutines(filename_control)


    ## reset variables
    experimental_data = " "
    parameter_file = " "
    fit_control = " "
    fit_log = " "
    XmlFileInOutput = " "
    filenamelist = []


    # get filname of experimental_data
    XMLString = "./PathFilename/experimental_data/filename"
    experimental_data = IOFile.GetSingleTagValue(XMLString)
    filenamelist.append(experimental_data)


    # get file name of parameter_file
    XMLString = "./PathFilename/parameter_file/filename"
    parameter_file = IOFile.GetSingleTagValue(XMLString)
    filenamelist.append(parameter_file)


    # get filename of fit_control
    XMLString = "./PathFilename/fit_control/filename"
    fit_control = IOFile.GetSingleTagValue(XMLString)
    filenamelist.append(fit_control)


    # get filename of fit_log
    XMLString = "./PathFilename/fit_log/filename"
    fit_log = IOFile.GetSingleTagValue(XMLString)
    filenamelist.append(fit_log)


    # get path and filename of XmlFileInOutput
    XMLString = "./PathFilename/model_description/filename"
    XmlFileInOutput = IOFile.GetSingleTagValue(XMLString)
    filenamelist.append(XmlFileInOutput)


    if (printflag != "false"):
        print ("done!\n")


    ## transform molfit-file to xml-file (myXCLASS is required)
    parameter_file_orig = parameter_file.strip()                                            ## make a copy of the original parameter file name
    if (parameter_file.endswith(".molfit")):


        ## start transformation function "task_myXCLASS.ConversionMolfit2XML"
        ii = parameter_file.rfind("/")                                                      ## get path of experimental data
        if (ii != -1):
            xmlFileName = parameter_file[:ii+1] + "parameters.xml"
        else:
            xmlFileName = "parameters.xml"
        ok = task_myXCLASS.ConversionMolfit2XML(printflag, parameter_file, xmlFileName)


        ## set new experimental data path and file name
        ii = parameter_file.rfind("/")                                                      ## get path of experimental data
        if (ii != -1):
            parameter_file = parameter_file[:ii+1] + "parameters.xml"
        else:
            parameter_file = "parameters.xml"


    ## print some informations on screen
    if (printflag != "false"):
        print ("\t Path and name of observation file:   {0:s}".format(chr(34) + experimental_data + chr(34)))
        print ("\t Path and name of parameter file:     {0:s}".format(chr(34) + parameter_file + chr(34)))
        print ("\t Path and name of fit control file:   {0:s}".format(chr(34) + fit_control + chr(34)))
        print ("\t Path and name of log file:           {0:s}".format(chr(34) + fit_log + chr(34)))
        print ("\t Path and name of registration file:  {0:s}".format(chr(34) + XmlFileInOutput + chr(34)))
        print ("\n", flush = True)


    ## testing path and filenames
    ii = 0
    for filename in filenamelist:
        ii += 1
        if (ii != 4):
            if not(os.path.exists(filename)):
                print (" ")
                print ("\t Error in io_control.xml file!")
                print ("\t      The file ", filename)
                print ("\t      does not exists!")
                print (" ")
                ok = 1
        else:
            j = filename.rfind('/')
            if (j != -1):
                PathLog = filename[0:j+1]
                if not(os.path.exists(PathLog)):
                    print (" ")
                    print ("\t Error in io_control.xml file!")
                    print ("\t      The path ", PathLog)
                    print ("\t      does not exists!")
                    print (" ")
                    ok = 1
            else:
                PathLog = filename.strip()


        ## check, if files has correct file extension
        if (ii == 1):
            if not filename.endswith(".xml"):
                if not filename.endswith(".txt"):
                    if not filename.endswith(".dat"):
                        if not filename.endswith(".fits"):
                            if not filename.endswith(".cso"):
                                print (" ")
                                print ("\t Error in io_control.xml file!")
                                print ("\t     The selected experimental file ",filename)
                                print ("\t     is neither a xml nor a dat nor a fits-file!")
                                print ("\t     Please select a .xml, a .dat, a .cso or a .fits-file!")
                                print (" ")
                                ok = 1

        if (ii == 2):
            if not filename.endswith(".xml"):
                if not filename.endswith(".molfit"):
                    print (" ")
                    print ("\t Error in io_control.xml file!")
                    print ("\t     The selected file ",filename)
                    print ("\t     is neither a xml-file nor a molfit-file!")
                    print ("\t     Please select a .xml-file!")
                    print (" ")
                    ok = 1

        elif (ii == 3):
            if not filename.endswith(".xml"):
                print (" ")
                print ("\t Error in io_control.xml file!")
                print ("\t     The selected file ",filename)
                print ("\t     is not a xml-file!")
                print ("\t     Please select a .xml-file!")
                print (" ")
                ok = 1


    ## define output parameters
    return ok, experimental_data, parameter_file, parameter_file_orig, fit_control, fit_log, XmlFileInOutput
##--------------------------------------------------------------------------------------------------------------------------------------------------------



##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## call the other modules and manages the communication between these modules
##
def main(printflag, plotflag, debugflag, modelflag, filename_control, jobID, MAGIXrootDirectory, LibDir):
    """
    input parameters:       printflag:                  flag for printing messages to the screen
                            plotflag:                   flag for plotting graphs
                            debugflag:                  flag for debugging call of external model program
                            modelflag:                  flag indicaing the usage of MAGIX packages optimized for a certain model
                            filename_control:           path and file name of the i/o control file
                            jobID:                      number to identify the current job
                            MAGIXrootDirectory:         MAGIX root directory
                            LibDir:                     path to binaries of optimization algorithm

    output paramters:       ok:                         status of the program

    working variables:      experimental_data:          array containing the experimental data
                            parameter_file:             path and file name of the xml-file containing the model parameter
                            parameter_file_orig:        copy of the variable including the path and the name of the file
                                                        containing the start values of the model parameters
                            fit_control:                path and file name of the fit control file
                            fit_log:                    path and file name of the log-file used by the fit-process
                            XmlFileInput:               xml-description of the model input file
                            XmlFileOutput:              xml-description of the model output file
                            i:                          loop counter
                            ExpInputFormat:             ending of the filename of the experimental file(s)
                            NumberExpFiles:             number of experimental files
                            LengthExpRange:             length of each experimental file
                            ExpDataX:                   array containing the "x"-columns of each experimental file
                            ExpDataY:                   array containing the "y"-columns of each experimental file
                            ExpDataError:               array containing the "error"-columns of each experimental file
                            NumberParameter:            total number of model parameters
                            parameter_set:              valus, fit-flags, upper and lower limit of the model parameters
                            FitParameterName:           names of the model parameters
                            ColumnY:                    number of "y" columns of each experimental file
                            OptimizedParameterSet:      optimized parameter set
                            FitFunctionValues:          values of the fit function at the "x"-column points
                            XmlParameterFileOut:        path and file name of the xml-file containing the new values and errors of the
                                                        optimized model parameters
                            InputFileIn:                name of the input file of the external program, which is created at the end
                                                        of the fit process and includes the optimized parameter
                            InputFileOut:               name of the input file of the external program, which is created at the end
                                                        of the fit process and includes the optimized parameter; ends with ".molfit"
                            command_string:             string being used for renaming the input file of the external program, which
                                                        is created at the end of the fit process and includes the optimized parameter
    """

    # Debug
    # print ("printflag = ", printflag)
    # print ("plotflag = ", plotflag)
    # print ("debugflag = ", debugflag)
    # print ("modelflag = ", modelflag)
    # print ("filename_control = ", filename_control)
    # print ("jobID = ", jobID)
    # print ("MAGIXrootDirectory = ", MAGIXrootDirectory)
    # print ("LibDir = ", LibDir)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## extend sys.path variable
    XCLASSBaseDir = MAGIXrootDirectory + "../../"
    XCLASSBaseDir = os.path.normpath(XCLASSBaseDir) + "/"
    NewModulesPaths = XCLASSBaseDir + "build_tasks"
    if (not NewModulesPaths in sys.path):
        sys.path.append(NewModulesPaths)
    from xclass import task_myXCLASS                                                        ## import package task_myXCLASS


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read control file for i/o traffic
    ok, experimental_data, parameter_file, parameter_file_orig, fit_control, fit_log, XmlFileInOutput = read_control(printflag, filename_control)
    if (ok != 0):
        return ok


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read experimental data from file using settings for the import declared in the xml-file
    ## initialize class LoadExpFile.LoadExp
    ClassLoadExp = LoadExpFile.LoadExp(printflag)


    ## load xml-file containing setting for import of experimental data
    ok, ExpInputFormat, NumExpRanges, MinExpRange, MaxExpRange, ImportParameterSet = ClassLoadExp.xml(experimental_data, printflag, XmlFileInOutput)
    if (ok != 0):                                                                           ## does an error occur while importing xml-file
        if (ok == 1):                                                                       ## a heavy error occur --> abort program
            return ok


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ## ONLY FOR myXCLASS: special treatment of the iso-file
    if (parameter_file_orig.endswith(".molfit")):
        IsoTableFileName = ""
        IsoFlag = ""
        try:
            i = ImportParameterSet[0].index("MAGIXImportIsotopologues")
        except:
            i = (-1)
        if (i > (-1)):
            IsoFlag = ImportParameterSet[1][i]
            if (IsoFlag == 1.0):
                try:
                    i = ImportParameterSet[0].index("MAGIXImportIsoTableFileName")
                except ValueError:
                    i = (-1)
                if (i > (-1)):
                    IsoTableFileName = ImportParameterSet[1][i]

        # Debug:
        # print ("IsoFlag = ", IsoFlag)
        # print ("\n\nIsoTableFileName = ", IsoTableFileName)
        # print ("parameter_file = ", parameter_file)
        # print ("parameter_file_orig = ", parameter_file_orig)
        # print ("ImportParameterSet = ", ImportParameterSet, "\n\n")


        ok = task_myXCLASS.GetIsoTableParameters(printflag, IsoTableFileName, parameter_file)
    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


    ## load experimental data using several import functions depending on the file format
    ok, NumberExpFiles, LengthExpRange, ExpDataX, ExpDataY, ExpDataError = ClassLoadExp.LoadFile(printflag)
    if (ok != 0):
        return ok


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read fit parameter from xml file
    ok, NumberParameter, parameter_set, FitParameterName = GetParameterSet.ReadParameter(printflag, parameter_file)
    if (ok != 0):
        return ok


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start fitting engine by calling FittigEngine module
    ok, PlotOption, OptimizedParameterSet, FitFunctionValues, Chi2SaveFlag, Chi2Values, \
    LengthExpRange, ColumnY, ExpDataX, ExpDataY, ExpDataError = FittingEngine.start(printflag, plotflag, debugflag, modelflag, jobID, fit_control, \
                                                                                    fit_log, parameter_file, XmlFileInOutput, NumberParameter, \
                                                                                    parameter_set, FitParameterName, NumberExpFiles, LengthExpRange, \
                                                                                    ExpDataX, ExpDataY, ExpDataError, ClassLoadExp, ExpInputFormat, \
                                                                                    NumExpRanges, MinExpRange, MaxExpRange, ImportParameterSet, \
                                                                                    parameter_file_orig, MAGIXrootDirectory, LibDir)
    if (ok != 0):
        return ok


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## finished program!
    if (printflag != "false"):
        print ("Program finished!")


    ## define return parameters
    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------

