#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module reads and writes the model parameter to a xml-file.
##  Copyright (C) 2009 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - subroutine ReadParameter:                     import xml-file containing the start values of model parameters for the fit process
##      - subroutine TrueFalse:                         function return true of false, if val == 0 or val != 0
##      - subroutine WriteParameter:                    export parameters to xml file using lxml routines
##      - subroutine WriteParameterString:              export parameters to xml file using direct ASCII method
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    2009-06-22   initial version
##  T. Moeller    2012-01-16   improve documentation of source code
##  T. Moeller    2020-01-02   porting to python 3.x, minor improvements
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************

"""
Read model parameter:
---------------------

The package "GetParameterSet" reads and writes the model parameter to a
xml-file. The names of the model parameters, etc. have to confirm with the
registration file valid for the according model. The names of the model
parameters must not include the characters "[", "]", and "," (except [[]]).

The tags <uplimit> </uplimit> and <lowlimit> </lowlimit> indicate the upper
and the lower limit of the model parameters, respectively.


Example xml-file (parameter file):
----------------------------------

<?xml version="1.0" encoding="ISO-8859-1"?>
<ModelParameters>
    <NumberParameters>2</NumberParameters>
    <Parameter fit="false">
        <name>EpsilonInfinity</name>
        <value>3.0</value>
        <error> </error>
        <lowlimit>0</lowlimit>
        <uplimit>+inf</uplimit>
    </Parameter>
    <Parameter fit="false">
        <name>NumberOscillators</name>
        <value>1</value>
        <error> </error>
        <lowlimit>0</lowlimit>
        <uplimit>+inf</uplimit>
    </Parameter>
    <Parameter fit="false">
        <name>EigenFrequency</name>
        <value>300.0</value>
        <error> </error>
        <lowlimit>0</lowlimit>
        <uplimit>+inf</uplimit>
    </Parameter>
    <Parameter fit="true">
        <name>PlasmaFrequency</name>
        <value>800.0</value>
        <error> </error>
        <lowlimit>0</lowlimit>
        <uplimit>+inf</uplimit>
    </Parameter>
    <Parameter fit="false">
        <name>Damping</name>
        <value>10.0</value>
        <error> </error>
        <lowlimit>0</lowlimit>
        <uplimit>+inf</uplimit>
    </Parameter>
</ModelParameters>


IMPORTANT:

- The descriptions of each parameter are bracket between the
  <Parameter></Parameter> tags. These tags have to occur as many times as
  defined in the tag <NumberParameters></NumberParameters>.


- In order to include a parameter into the fitting process set the
  attribute of the tags <Parameter></Parameter> to "true".
  For example: If you want to optimize the value of the Parameter
               "EpsilonInfinity" set the attribute to "true"

               <Parameter fit="true">
                   <name>EpsilonInfinity</name>
                   <value>3.0</value>
                   <error> </error>
                   <lowlimit>0</lowlimit>
                   <uplimit>+inf</uplimit>
               </Parameter>

               If the value of this parameter should be fixed set
               the attribute to "false":
               <Parameter fit="false">
                   ...

- The tag <error> </error> must occur for every parameter and must have a
  non-empty value because the value of this tag is replaced by the error
  of the optimized parameter.
  Note, do not remove these tags!


- The number of model parameters defined in this file must be identical with
  the number of parameters defined in the registration file (including all
  replications of lines and parameters).


- The names of the model parameter defined in the tags <name></name>
  must be identical with the definitions of the names of the model
  parameter defined in the registration file. Otherwise the program
  stops!
"""


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
from xclass.addons.MAGIX.Modules.python import XMLPackage
from xclass.addons.MAGIX.Modules.python import LoadExpFile
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## import xml-file containing the start values of model parameters for the fit process
##
def ReadParameter(print_flag, ParameterXMLFileName):
    """
    input parameters:       print_flag:                 print flag
                            ParameterXMLFileName:       path and file name of the xml file containing the start values

    output parameters:      ok:                         status of reading
                            FitParametersNumber:        total number of model parameters
                            FitParameters:              values, fit flags, lower and upper limit of each model parameter
                            FitParameterName:           names of the model parameters
    """

    # Debug:
    # print ("print_flag = ", print_flag)
    # print ("ParameterXMLFileName = ", ParameterXMLFileName)


    ## set status parameter, reset FitParameter array and reset FitParameterName array
    ok = 0
    FitParametersNumber = 0
    FitParameters = 0
    FitParameterName = " "


    ## read xml-description of the INPUT file
    ## is the defined file a xml-file ?
    if (not ParameterXMLFileName.endswith(".xml")):
        print ("\n\t Error in subroutine GetParameterSet.ParameterXMLFileName!")
        print ("\t\t The selected file is not a xml-file!")
        print ("\t\t Please select a .xml and restart the whole program.\n")
        ok = 1
        return ok, FitParametersNumber, FitParameters, FitParameterName


    ## print what you do ..
    if (print_flag != "false"):
        print ("\nImport parameter xml-file:")
        print ("\t Open parameter xml-file: " + chr(34) + ParameterXMLFileName + chr(34))
        print ("\t Import settings .. ", end = "")


    # ## check, if instance xml file contains empty tags
    # LocalXMLFile = open(ParameterXMLFileName)
    # LocalXMLFileContents = LocalXMLFile.readlines()
    # LocalXMLFile.close()
    # NewLocalXMLFile = open(ParameterXMLFileName, 'w')
    # for line in LocalXMLFileContents:
    #     if (line.find("<error></error>") > (-1)):
    #         NewLine = line.replace("<error></error>", "<error> </error>")
    #     elif (line.find("<lowlimit></lowlimit>") > (-1)):
    #         NewLine = line.replace("<lowlimit></lowlimit>", "<lowlimit> </lowlimit>")
    #     elif (line.find("<uplimit></uplimit>") > (-1)):
    #         NewLine = line.replace("<uplimit></uplimit>", "<uplimit> </uplimit>")
    #     else:
    #         NewLine = line
    #     NewLocalXMLFile.write(NewLine)
    # NewLocalXMLFile.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## open xml-file file and save content of file to variable XmlFile
    #    XmlFile = MAGIXxmlPACKAGE.XmlObject(fileName = ParameterXMLFileName)
    ParamFileIn = XMLPackage.XMLRoutines(ParameterXMLFileName)


    ## define start string
    XMLStartStringV1 = "./"
    XMLStartStringV2 = "./Section/SubSection/"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get number of model parameters
    #    FitParametersNumber = XmlFile.ModelParameters.NumberParameters.getValue()
    TagString = "NumberParameters"
    FitParametersNumber = ParamFileIn.GetSingleObsTagValue(XMLStartStringV1, XMLStartStringV2, TagString, FormatString = "int", DefaultValue = 0)

    # Debug:
    # print ("FitParametersNumber = ", FitParametersNumber)


    ## get list of all tags and values from xml file
    ## in a previous version, the function "GetSingleObsTagValue" was used to extract the parameters from the xml file.
    ## But using this function is much too slow, so now we use the GetTags function instead.
    TagList, FullElementList = ParamFileIn.GetTags()


    ## get settings for each model parameter
    RawNameList = []
    FitParameterName = []
    FitParameters = []
    ParameterValues = []
    ParameterFitFlag = []
    ParameterLowLimit = []
    ParameterUpLimit = []
    i = (-1)
    fitflag = 0
    # for i in range(FitParametersNumber):                                                    ## loop over all parameters
    for element in FullElementList:
        LocalName = str(element[0])
        LocalAttrib = element[1]
        LocalValue = element[2]

        # Debug:
        # print ("\nLocalName = ", LocalName)
        # print ("LocalAttrib = ", LocalAttrib)
        # print ("LocalValue = ", LocalValue)


        ## get number of parameter
        if (LocalName in ["Parameter"]):
            i += 1
            fitflagRAW = LocalAttrib["fit"].lower()
            fitflag = LoadExpFile.CheckBool(fitflagRAW)
            if (fitflag):
                fitflag = 1
            else:
                fitflag = 0

            # Debug:
            # print ("fitflagRAW = ", fitflagRAW)
            # print ("fitflag = ", fitflag)


        ## we are interested in parameters only
        if (i >= 0):


            ## define stat string
            # ParamXMLStartStringV1 = XMLStartStringV1 + "Parameter[" + str(i + 1) + "]"
            # ParamXMLStartStringV2 = XMLStartStringV2 + "Parameter[" + str(i + 1) + "]"


            ##------------------------------------------------------------------------------------------------------------------------------------------------
            ## read parameter name
            # TagString = ""
            # FitFlagDict = ParamFileIn.GetSingleObsTagAttrib(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString)


            ##------------------------------------------------------------------------------------------------------------------------------------------------
            ## get name of parameter
            # TagString = "name"
            # name = ParamFileIn.GetSingleObsTagValue(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString)
            if (LocalName == "name"):
                name = str(LocalValue)


                ## test if name include one of the following commands (XValue, NumberXValue, CurrentExpFile, CurrentYColumn)
                if (name.find("XValue") != -1):
                    print ("\n\t Error in subroutine GetParameterSet.ParameterXMLFileName!")
                    print ("\t\t The selected parameter name: ",name)
                    print ("\t\t contains the command XValue!")
                    print ("\t\t Please select a another name for the parameter and restart the whole program.")
                    print (" ")
                    ok = 1
                    return ok, FitParametersNumber, FitParameters, FitParameterName

                elif (name.find("NumberXValue") != -1):
                    print ("\n\t Error in subroutine GetParameterSet.ParameterXMLFileName!")
                    print ("\t\t The selected parameter name: ",name)
                    print ("\t\t contains the command NumberXValue!")
                    print ("\t\t Please select a another name for the parameter and restart the whole program.")
                    print (" ")
                    ok = 1
                    return ok, FitParametersNumber, FitParameters, FitParameterName

                elif (name.find("CurrentExpFile") != -1):
                    print ("\n\t Error in subroutine GetParameterSet.ParameterXMLFileName!")
                    print ("\t\t The selected parameter name: ",name)
                    print ("\t\t contains the command CurrentExpFile!")
                    print ("\t\t Please select a another name for the parameter and restart the whole program.")
                    print (" ")
                    ok = 1
                    return ok, FitParametersNumber, FitParameters, FitParameterName

                elif (name.find("CurrentYColumn") != -1):
                    print ("\n\t Error in subroutine GetParameterSet.ParameterXMLFileName!")
                    print ("\t\t The selected parameter name: ",name)
                    print ("\t\t contains the command CurrentYColumn!")
                    print ("\t\t Please select a another name for the parameter and restart the whole program.")
                    print (" ")
                    ok = 1
                    return ok, FitParametersNumber, FitParameters, FitParameterName


                # make each parameter name unique
                RawNameList.append(name)
                if (name != ""):
                    counter = RawNameList.count(name)
                    if (name.rfind("[[]]") == (-1)):
                        name = name + "[[" + str(counter) + "]]"
                FitParameterName.append(name)
                ParameterFitFlag.append(fitflag)

                # Debug:
                # print ("i, name = ", i, name)
                # print ("fitflag = ", fitflag)


            ##------------------------------------------------------------------------------------------------------------------------------------------------
            ## read lower limit
            # TagString = "lowlimit"
            # lowlimit = ParamFileIn.GetSingleObsTagValue(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString, FormatString = "lower", DefaultValue = "")
            if (LocalName == "lowlimit"):
                lowlimit = LocalValue
                if (lowlimit == ""):
                    lowlimit = 0
                elif (lowlimit == "+inf" or lowlimit == "inf"):
                    lowlimit = 1e99
                elif (lowlimit == "-inf"):
                    lowlimit = -1e99
                ParameterLowLimit.append(lowlimit)

                # Debug:
                # print ("lowlimit = ", lowlimit)


            ##------------------------------------------------------------------------------------------------------------------------------------------------
            ## read upper limit
            # TagString = "uplimit"
            # uplimit = ParamFileIn.GetSingleObsTagValue(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString, FormatString = "lower", DefaultValue = "")
            if (LocalName == "uplimit"):
                uplimit = LocalValue
                if (uplimit == ""):
                    uplimit = 0
                elif (uplimit == "+inf" or uplimit == "inf"):
                    uplimit = 1e99
                elif (uplimit == "-inf"):
                    uplimit = -1e99
                ParameterUpLimit.append(uplimit)

                # Debug:
                # print ("uplimit = ", uplimit)


            ##------------------------------------------------------------------------------------------------------------------------------------------------
            ## read parameter value
            # TagString = "value"
            # value = ParamFileIn.GetSingleObsTagValue(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString)
            if (LocalName == "value"):
                value = LocalValue
                if (fitflag == 1):
                    try:
                        value = value.strip()
                    except AttributeError as err:
                        pass
                    if (value == ""):
                        value = (float(uplimit) - float(lowlimit)) / 2.0
                ParameterValues.append(value)

                # Debug:
                # print ("value = ", value)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## construct FitParameters array
    FitParameters.append(ParameterValues)
    FitParameters.append(ParameterFitFlag)
    FitParameters.append(ParameterLowLimit)
    FitParameters.append(ParameterUpLimit)


    ## print: everything is done
    if (print_flag != "false"):
        print ("done!")
        print ("\n\t Number of fit parameters:   ", FitParametersNumber)
        # print ("\t Names of fit parameters:    ", FitParameterName)
        # print ("\t Parameter settings:         ", FitParameters)
        print ("")


    # Debug:
    # for i in range(FitParametersNumber):
    #     print ("\n", FitParameterName[i])
    #     print (FitParameters[0][i])
    #     print (FitParameters[1][i])
    #     print (FitParameters[2][i])
    #     print (FitParameters[3][i])
    # sys.exit(0)


    ## define return parameters
    return ok, FitParametersNumber, FitParameters, FitParameterName
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## function return true of false, if val == 0 or val != 0
##
def TrueFalse(val):
    """
    input parameters:       val:                        input variable indicating error flag

    output parameters:       TrueFalseString:            string = "true" or "false"
    """

    # Debug:
    # print ("val = ", val)


    if (val == 0):
        TrueFalseString = "false"
    else:
        TrueFalseString = "true"


    ## define return parameter
    return TrueFalseString
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## export parameters to xml file using lxml routines
##
def WriteParameter(print_flag, ParameterXMLFileName, FitParametersNumber, FitParameters, FitParameterName):
    """
    input parameters:       print_flag:                 print flag
                            ParameterXMLFileName:       path and file name of the xml file containing the start values
                            FitParameters:              values, fit flags, lower and upper limit of each model parameter
                            FitParameterName:           names of the model parameters


    output parameters:       ok:                         status of reading


    working parameters:     i:                          loop counter
                            XmlFile:                    contents of the xml-file
                            ParamName:                  name of ith parameter
    """

    # Debug:
    # print ("print_flag = ", print_flag)
    # print ("ParameterXMLFileName = ", ParameterXMLFileName)
    # print ("FitParametersNumber = ", FitParametersNumber)
    # print ("FitParameters = ", FitParameters)
    # print ("FitParameterName = ", FitParameterName)


    ## set status parameter, reset FitParameter array and reset FitParameterName array
    ok = 0


    ## read xml-description of the INPUT file is the defined file a xml-file ?
    if (not ParameterXMLFileName.endswith(".xml")):
        print ("\n\t Error in subroutine GetParameterSet.ParameterXMLFileName!")
        print ("\t\t The selected file is not a xml-file!")
        print ("\t\t Please select a .xml and restart the whole program.\n")
        ok = 1
        return ok


    ## print what you do ..
    if (print_flag != "false"):
        print ("Write optimized model parameter to file.")
        print ("\t Open xml-file: " + ParameterXMLFileName)
        print ("\t Writing file .. ", end = "")


    ## open xml-file file and save content of file to variable XmlFile
    # XmlFile = MAGIXxmlPACKAGE.XmlObject(fileName = ParameterXMLFileName)
    ParamFileOut = XMLPackage.XMLRoutines(ParameterXMLFileName)


    ## define start string
    XMLStartStringV1 = "./"
    XMLStartStringV2 = "./Section/SubSection/"


    ## set values of the optimized parameters
    countFreeParam = 0
    for i in range(FitParametersNumber):


        ## define stat string
        ParamXMLStartStringV1 = XMLStartStringV1 + "Parameter[" + str(i + 1) + "]"
        ParamXMLStartStringV2 = XMLStartStringV2 + "Parameter[" + str(i + 1) + "]"


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write attribute
        # XmlFile.ModelParameters.Parameter[i].setAttribute("fit", TrueFalse(FitParameters[1][i]))
        # TagString = ""
        # LocalTagValue = TrueFalse(FitParameters[1][i])
        # ParamFileOut.SetSingleObsTagAttrib(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString, LocalTagValue)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## remove squared brackets at the end of the parameter name
        ParamName = FitParameterName[i]
        j = ParamName.find("[[]]")
        if (j == (-1)):
            j = ParamName.find("[[")
            if (j != (-1)):
                ParamName = FitParameterName[i][:j]

        # Debug:
        # print ("\n\ni, ParamName = ", i, ParamName)
        # print ("FitParameters[0][i] = ", FitParameters[0][i])
        # print ("FitParameters[1][i] = ", FitParameters[1][i])
        # print ("FitParameters[2][i] = ", FitParameters[2][i])
        # print ("FitParameters[3][i] = ", FitParameters[3][i])


        ## write name of parameter to xml-file
        # XmlFile.ModelParameters.Parameter[i].name.setValue(ParamName)
        # TagString = "/name"
        # ParamFileOut.SetSingleObsTagValue(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString, LocalTagValue, WriteToFile = False)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write parameter value
        # XmlFile.ModelParameters.Parameter.value.setValue(FitParameters[0][i])
        TagString = "/value"
        try:
            LocalTagValue = "{:.15e}".format(float(FitParameters[0][i]))
        except:
            LocalTagValue = str(FitParameters[0][i])
        ParamFileOut.SetSingleObsTagValue(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString, LocalTagValue, WriteToFile = False)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write error value
        TagString = "/error"
        if (len(FitParameters) == 6):
            if (type(FitParameters[5][i]).__name__=='list'):                                ## is entry list or number
                countFreeParam += 1
                param_num = "parameter-number = {:d};".format(int(countFreeParam))
                err_left_str = "  error_left = {:.4e};".format(float(FitParameters[5][i][0]))
                err_right_str = "  error_right = {:.4e};".format(float(FitParameters[5][i][1]))
                tag_content = param_num + err_left_str + err_right_str
                if (len(FitParameters[5][i]) > 2):
                    mean_val_str = "  mean_value = {:.4e};".format(float(FitParameters[5][i][2]))
                    err_mean_str = "  standard_dev = {:.4e};".format(float(FitParameters[5][i][3]))
                    all_logZ_str = "  log(evidence) = {:.4e}".format(float(FitParameters[5][i][4]))
                    tag_content += mean_val_str + err_mean_str + all_logZ_str
                # XmlFile.ModelParameters.Parameter[i].error.setValue(tag_content)
                ParamFileOut.SetSingleObsTagValue(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString, tag_content, WriteToFile = False)
            else:
                # XmlFile.ModelParameters.Parameter[i].error.setValue(str(FitParameters[5][i]))
                LocalTagValue = str(FitParameters[5][i])
                ParamFileOut.SetSingleObsTagValue(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString, LocalTagValue, WriteToFile = False)
        else:
            #    XmlFile.ModelParameters.Parameter[i].error.setValue(FitParameters[4][i])
            LocalTagValue = str(FitParameters[4][i])
            ParamFileOut.SetSingleObsTagValue(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString, LocalTagValue, WriteToFile = False)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write low limit value
        # XmlFile.ModelParameters.Parameter.lowlimit.setValue(FitParameters[2][i])
        TagString = "/lowlimit"
        try:
            LocalTagValue = "{:.15e}".format(float(FitParameters[2][i]))
        except:
            LocalTagValue = str(FitParameters[2][i])
        ParamFileOut.SetSingleObsTagValue(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString, LocalTagValue, WriteToFile = False)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write upper limit value
        # XmlFile.ModelParameters.Parameter.uplimit.setValue(FitParameters[3][i])
        TagString = "/uplimit"
        try:
            LocalTagValue = "{:.15e}".format(float(FitParameters[3][i]))
        except:
            LocalTagValue = str(FitParameters[3][i])
        WriteToFile = False
        if (i == (FitParametersNumber - 1)):
            WriteToFile = True
        ParamFileOut.SetSingleObsTagValue(ParamXMLStartStringV1, ParamXMLStartStringV2, TagString, LocalTagValue, WriteToFile = WriteToFile)


    ## print that you have finished
    if (print_flag != "false"):
        print ("done !")


    ## define return parameter
    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## export parameters to xml file using direct ASCII method
##
def WriteParameterString(print_flag, ParameterXMLFileName, FitParametersNumber, FitParameters, FitParameterName):
    """
    input parameters:       print_flag:                 print flag
                            ParameterXMLFileName:       path and file name of the xml file containing the start values
                            FitParameters:              values, fit flags, lower and upper limit of each model parameter
                            FitParameterName:           names of the model parameters


    output parameters:       ok:                         status of reading


    working parameters:     i:                          loop counter
                            XmlFile:                    contents of the xml-file
                            ParamName:                  name of ith parameter
    """

    # Debug:
    # print ("print_flag = ", print_flag)
    # print ("ParameterXMLFileName = ", ParameterXMLFileName)
    # print ("FitParametersNumber = ", FitParametersNumber)
    # print ("FitParameters = ", FitParameters)
    # print ("FitParameterName = ", FitParameterName)


    ## set status parameter, reset FitParameter array and reset FitParameterName array
    ok = 0


    ## read xml-description of the INPUT file is the defined file a xml-file ?
    if (not ParameterXMLFileName.endswith(".xml")):
        print ("\n\t Error in subroutine GetParameterSet.ParameterXMLFileName!")
        print ("\t\t The selected file is not a xml-file!")
        print ("\t\t Please select a .xml and restart the whole program.\n")
        ok = 1
        return ok


    ## print what you do ..
    if (print_flag != "false"):
        print ("Write optimized model parameter to file.")
        print ("\t Open xml-file: " + ParameterXMLFileName)
        print ("\t Writing file .. ", end = "")


    ## print that you have finished
    if (print_flag != "false"):
        print ("done !")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## open new xml file and write header
    NewXMLFile = open(ParameterXMLFileName, 'w')
    NewXMLFile.write("<?xml version='1.0' encoding='UTF-8'?>\n")
    NewXMLFile.write("<ModelParameters>\n\n\n")
    NewXMLFile.write("    <!-- define number of parameters -->\n")
    NewXMLFile.write("    <NumberParameters>{:d}</NumberParameters>\n".format(FitParametersNumber))


    ## write tags for each parameter
    countFreeParam = 0
    for i in range(FitParametersNumber):


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## remove squared brackets at the end of the parameter name
        ParamName = FitParameterName[i]
        j = ParamName.find("[[]]")
        if (j == (-1)):
            j = ParamName.find("[[")
            if (j != (-1)):
                ParamName = FitParameterName[i][:j]

        # Debug:
        # print ("\n\ni, ParamName = ", i, ParamName)
        # print ("FitParameters[0][i] = ", FitParameters[0][i])
        # print ("FitParameters[1][i] = ", FitParameters[1][i])
        # print ("FitParameters[2][i] = ", FitParameters[2][i])
        # print ("FitParameters[3][i] = ", FitParameters[3][i])


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add attribute of parameter
        NewXMLFile.write("\n\n    <!-- define parameter {:d} -->\n".format(i + 1))
        LocalTagValue = TrueFalse(FitParameters[1][i])
        NewXMLFile.write("    <Parameter fit=" + chr(34) + LocalTagValue + chr(34) + ">\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add name of parameter
        NewXMLFile.write("        <name>{:s}</name>\n".format(ParamName))


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add value of parameter
        try:
            LocalTagValue = "{:.15e}".format(float(FitParameters[0][i]))
        except:
            LocalTagValue = str(FitParameters[0][i])
        NewXMLFile.write("        <value>" + LocalTagValue + "</value>\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add error of parameter
        if (len(FitParameters) == 6):
            if (type(FitParameters[5][i]).__name__=='list'):                                ## is entry list or number
                countFreeParam += 1
                param_num = "parameter-number = {:d};".format(int(countFreeParam))
                err_left_str = "  error_left = {:.4e};".format(float(FitParameters[5][i][0]))
                err_right_str = "  error_right = {:.4e};".format(float(FitParameters[5][i][1]))
                LocalTagValue = param_num + err_left_str + err_right_str
                if (len(FitParameters[5][i]) > 2):
                    mean_val_str = "  mean_value = {:.4e};".format(float(FitParameters[5][i][2]))
                    err_mean_str = "  standard_dev = {:.4e};".format(float(FitParameters[5][i][3]))
                    all_logZ_str = "  log(evidence) = {:.4e}".format(float(FitParameters[5][i][4]))
                    LocalTagValue += mean_val_str + err_mean_str + all_logZ_str
            else:
                LocalTagValue = str(FitParameters[5][i])
        else:
            LocalTagValue = str(FitParameters[4][i])
        NewXMLFile.write("        <error>" + LocalTagValue + "</error>\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add lower limit of parameter
        try:
            LocalTagValue = "{:.15e}".format(float(FitParameters[2][i]))
        except:
            LocalTagValue = str(FitParameters[2][i])
        NewXMLFile.write("        <lowlimit>" + LocalTagValue + "</lowlimit>\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## add upper limit of parameter
        try:
            LocalTagValue = "{:.15e}".format(float(FitParameters[3][i]))
        except:
            LocalTagValue = str(FitParameters[3][i])
        NewXMLFile.write("        <uplimit>" + LocalTagValue + "</uplimit>\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## finalize parameter description
        NewXMLFile.write("    </Parameter>\n")


    ## we're done
    NewXMLFile.write("</ModelParameters>\n\n")
    NewXMLFile.close()


    ## define return parameter
    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------

