!*********************************************************************************************************************************************************
!> Package: PSO algorithm
!>
!>
!>  This module contains the subroutines for particle swarm algorithm
!>  Copyright (C) 2009 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - Module PSOVariables:                      Module contains global variables and subroutines for all PSO algorithm
!>      - subroutine check_position:                cheking of pos. and func. calc.: if the point lies outside of parametric space, than function = 1e+09
!>      - subroutine check_parameters:              checks if parameter is within range and correct value if parameter is out of range
!>      - subroutine stop_crit:                     stopping criterion for iteration
!>      - subroutine sort_swarm:                    sortirung particles swarm
!>      - subroutine hyb_sim_mod:                   implementation of Nelder-Mead Simplex Algorithm
!>      - subroutine hyb_pso_mod:                   implementation of Modified Particle Swarm Optimization Algorithm
!>      - subroutine CallPSO:                       call the different pso subroutines
!>      - Module Algorithm:                         module contains the main subroutine used to start the different versions of the PSO algorithm
!>      - subroutine MainAlg:                       main subroutine which starts the PSO algorithm
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    2009-06-09  Initial version
!>  T. Moeller    2012-01-13  Updated version
!>  T. Moeller    2014-08-20  myXCLASS (model) optimized version
!>  T. Moeller    2014-09-01  modified and restructured for GPU
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: PSOVariables
!>
!>         Module contains global variables and subroutines for PSO algorithm
!>
!>
!> \author Thomas Moeller
!>
!> \date 2010-06-09
!>
Module PSOVariables

    use FunctionCalling
    use Model

    implicit none
    integer :: iterat, MaxPSOIter                                                           !< current and max. iteration number
    real*8 :: chi2lim                                                                       !< lower limit of chi**2 normalized to the total number of
                                                                                            !< all data points
    real*8, allocatable, dimension(:) :: a                                                  !< array containing all parameters
    real*8, allocatable, dimension(:) :: OptimizedParameterLowLimit                         !< lower limit for each free parameter
    real*8, allocatable, dimension(:) :: OptimizedParameterUpperLimit                       !< upper limit for each free parameter
    logical, allocatable, dimension(:) :: ia                                                !< array indicating the parameter which should be optimized

    contains


        !*************************************************************************************************************************************************
        !> subroutine: check_position
        !>
        !> Cheking of position and functions calculation: if the point lies outside of parametric space, than function = 1e+09
        !>
        !> input variables:     nfit:                   number of parameters which should be optimized
        !>                      ma:                     total number of all parameter
        !>                      xira:                   array containing all parameters which should be optimized
        !>                      NumFile:                number of input files
        !>                      MaxL:                   max. total length
        !>                      MaxCol:                 max. number of columns
        !>
        !> output variables:    funcValue:              contains the value of chisq on exit
        !>                                              (If one parameter is out ouf range chisq is set to 1e+9)
        !>
        !> \author Irina Bernst, Thomas Moeller
        !>
        !> \date 2010-06-09
        !>
        subroutine check_position(funcValue, nfit, ma, xira, NumFile, MaxL, MaxCol)

            use Variables
            use FunctionCalling

            implicit none
            integer :: nfit                                                                 !< (= effective parameter number) number of model
            integer :: ma                                                                   !< total number of all parameters
            integer :: i                                                                    !< loop variable
            integer :: NumFile, MaxL, MaxCol                                                !< working variables
            real*8 :: funcValue                                                             !< contains the value of chisq on exit
            real*8, dimension(1) :: chi2ValuesVector                                        !< here only one chi2 value
            real*8, dimension(nfit) :: xira                                                 !< current parameter set (only optimized parameter)
            real*8, dimension(1, nfit) :: xiraVec                                           !< copy of current parameter set
            logical :: OutOfRangeFlag                                                       !< flag for parameter is out of range


            !< are all parameter values within the given ranges
            OutOfRangeFlag = .false.
            Do i = 1, nfit
                if (xira(i) < OptimizedParameterLowLimit(i) .or. OptimizedParameterUpperLimit(i) < xira(i)) then
                    OutOfRangeFlag = .true.
                    exit
                endif
            end Do


            !< are all parameter values within the given ranges?
            funcValue = 0.d0
            if (.not. OutOfRangeFlag) then


                !< determine chi2 values
                chi2ValuesVector = 0.d0
                xiraVec(1, :) = xira(:)
                call ModelCalcChiFunctionGeneral(ma, ia, paramset(1, :), 1, nfit, NumFile, MaxL, MaxCol, xiraVec, chi2ValuesVector)
                funcValue = chi2ValuesVector(1)
            else
                funcValue = 1.d+99
            endif
            return
        end subroutine check_position


        !*************************************************************************************************************************************************
        !> subroutine: check_parameters
        !>
        !> Checks if parameter is within range and correct value if parameter is out of range
        !>
        !> input variables:     param:                  parameter which is checked
        !>                      low_limit:              lower limit of parameter param
        !>                      upper_limit:            upper limit of parameter param
        !>
        !> output variables:    param:                  contains corrected value of param if param is out of range param
        !>
        !> \author Thomas Moeller
        !>
        !> \date 19.08.2010
        !>
        subroutine check_parameters(param, low_limit, upper_limit)

            use Variables
            use FunctionCalling

            implicit none
            real*8 :: param                                                                 !< parameter which is checked
            real*8 :: low_limit                                                             !< lower limit of parameter param
            real*8 :: upper_limit                                                           !< upper limit of parameter param
            real*8 :: range_size                                                            !< size of range
            real*8, parameter :: reduction_coeff = 1.d-3                                    !< determine the range around the limit border

            if (param < low_limit.or.upper_limit < param) then
                range_size = upper_limit - low_limit
                if (param < low_limit) then
                    param = RandomWithLimits(low_limit, low_limit + range_size * reduction_coeff)
                else
                    param = RandomWithLimits(upper_limit - range_size * reduction_coeff, upper_limit)
                endif
            endif
            return
        end subroutine check_parameters


        !*************************************************************************************************************************************************
        !> subroutine: stop_crit
        !>
        !> Stopping criterion
        !>
        !> input variables:     in_swarm:               array containing positions of each particle
        !>                      i11:                    index
        !>                      nfit:                   number of parameters which should be optimized
        !>                      n_particles:            number of particles
        !>
        !> output variables:    metka:                  contains the value of chisq on exit
        !>                                              (If one parameter is out ouf range chisq is set to 1e+9)
        !>
        !> subroutine requires: OptimizedParameterLowLimit:     lower limit of the ith optimized parameter
        !>                      chilm:                  value of chisq where iteration stops
        !>                      MaxPSOIter:             max. number of iterations
        !>
        !> \author Irina Bernst, Thomas Moeller
        !>
        !> \date 2010-06-09
        !>
        subroutine stop_crit(in_swarm, metka, nfit, n_particles, i11, fun_min)

            use Variables
            use FunctionCalling

            implicit none
            integer :: i, j                                                                 !< loop variables
            integer :: i11                                                                  !< index
            integer :: metka                                                                !< contains the value of chisq on exit
            integer :: nfit                                                                 !< number of parameters which should be optimized
            integer :: n_particles                                                          !< number of particles
            real*8 :: sum1, sumj                                                            !< working variables
            real*8 :: ddelta                                                                !< working variable
            real*8 :: fun_min                                                               !< working variable
            real*8, dimension(nfit) :: comp                                                 !< working variable
            real*8, dimension(n_particles, nfit) :: in_swarm                                !< array containing positions of each particle

            comp = 0.d0
            sum1 = 0.d0
            Do i = 1, nfit
                sum1 = sum1 + OptimizedParameterLowLimit(i)**2.0
                sumj = 0.d0
                Do j = 1, (nfit + 1)
                    if (j /= i11) then
                        sumj = sumj + (in_swarm(j,i) - OptimizedParameterLowLimit(i))**2.0
                    endif
                end Do
                comp(i) = (sumj/(nfit))**0.5
            end Do
            sum1 = (sum1/nfit)**0.5
            ddelta = DMAX1(1.d0,sum1)

            sum1 = comp(1)
            Do i = 1, nfit
                if (sum1 <= comp(i)) then
                    sum1 = comp(i)
                endif
            end Do
            sum1 = sum1/ddelta

            metka = 0
            if (fun_min <= chi2lim .or. iterat >= MaxPSOIter) then
                metka = 1
            else
                metka = 0
            endif
            return
        end subroutine stop_crit


        !*************************************************************************************************************************************************
        !> subroutine: sort_swarm
        !>
        !> Sortirung particles swarm
        !>
        !> input variables:     in_swarm:               array containing the parameter which should be optimized
        !>                      fun_swarm:              the corresponding value of the model function
        !>                      NumFile:                number of input files
        !>                      MaxL:                   max. total length
        !>                      MaxCol:                 max. number of columns
        !>                      nfit:                   number of parameters which should be optimized
        !>                      n_particles:            number of particles
        !>                      ma:                     total number of all parameter
        !>
        !> output parameters:   in_swarm1:              array containing the parameter which should be optimized
        !>                      fun_swarm:              the corresponding value of the model function
        !>
        !> subroutine requires: OptimizedParameterLowLimit:     lower limit of the ith optimized parameter
        !>
        !> \author Irina Bernst and Thomas Moeller
        !>
        !> \date 2010-06-09
        !>
        subroutine sort_swarm(in_swarm1, in_swarm, fun_swarm, NumFile, MaxL, MaxCol, nfit, n_particles, ma)

            use Variables
            use FunctionCalling

            implicit none
            integer :: i, j, k                                                              !< loop variables
            integer :: nfit                                                                 !< number of parameters which should be optimized
            integer :: ma                                                                   !< total number of all parameter
            integer :: n_particles                                                          !< number of particles
            integer :: NumFile                                                              !< number of input files
            integer :: MaxL                                                                 !< max. total length
            integer :: MaxCol                                                               !< max. number of columns
            real*8, dimension(n_particles) :: chi2ValuesVector                              !< chi2 value vector
            real*8, dimension(n_particles) :: fun_swarm1, fun_swarm                         !< the corresponding value of the model function
            real*8, dimension(n_particles, nfit) :: in_swarm, in_swarm1                     !< array containing the parameter which should be optimized


            !< print what you do!!
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
                print '(11x,"Sorting particles swarm ..",$)'
            endif


            !< if calculation reduction is chosen, read chi**2 file
            if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then
                call CheckCalculatedParameterSets(nfit, 1)
            endif


            !< print what you do ..
            if (printflag) then
                print '(A,11x,"Sorting particles swarm ..                    ",A,$)', char(13), char(13)
            endif


            !< determine chi2 values
            chi2ValuesVector = 0.d0
            call ModelCalcChiFunctionGeneral(ma, ia, a, n_particles, nfit, NumFile, MaxL, MaxCol, in_swarm, chi2ValuesVector)
            fun_swarm(:) = chi2ValuesVector(:)
            fun_swarm1(:) = chi2ValuesVector(:)


            !< update CurrentNumberLinesCalcReduction variable
            CurrentNumberLinesCalcReduction = NumberLinesChi2


            !< sort array fun_swarm1
            call sort(n_particles, fun_swarm1)


            !< rearange in_swarm1
            Do i = 1, n_particles
                Do j = 1, n_particles
                    if (fun_swarm(j) == fun_swarm1(i)) then
                        Do k = 1, nfit
                            in_swarm1(i, k) = in_swarm(j, k)
                        end Do
                    endif
                end Do
            end Do

            ! Debug:
            ! print*,' '
            ! print*,'fun_swarm = ',fun_swarm
            ! print*,'fun_swarm1 = ',fun_swarm1
            ! print*,' '
            ! print*,'in_swarm(1,:) = ',in_swarm(1,:)
            ! print*,'in_swarm1(1,:) = ',in_swarm1(1,:)
            ! stop

            !< copy to output array
            fun_swarm = fun_swarm1


            !< clear screen massage
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
            endif


            !< return to main program
            return
        end subroutine sort_swarm


        !*************************************************************************************************************************************************
        !> subroutine: hyb_sim_mod
        !>
        !> Implementation of Nelder-Mead Simplex Algorithm
        !>
        !> (For details, http://en.wikipedia.org/wiki/Nelder%E2%80%93Mead_method)
        !>
        !> input variables:     in_simpex:              array containing the parameter which should be optimized
        !>                      fun_simplex:            the corresponding value of the model function
        !>                      NumFile:                number of input files
        !>                      MaxL:                   max. total length
        !>                      MaxCol:                 max. number of columns
        !>                      nfit:                   number of parameters which should be optimized
        !>                      n_particles:            number of particles
        !>                      ma:                     total number of all parameter
        !>
        !> output parameters:   in_simpex:              array containing the parameter which should be optimized
        !>
        !> subroutine requires: OptimizedParameterLowLimit:     lower limit of the ith optimized parameter
        !>                      nfit:                   number of parameters which should be optimized
        !>                      chilm:                  value of chisq where iteration stops
        !>                      MaxPSOIter:             max. number of iterations
        !>
        !> \author Irina Bernst and Thomas Moeller
        !>
        !> \date 2010-06-09
        !>
        subroutine hyb_sim_mod(in_simplex, fun_simplex, NumFile, MaxL, MaxCol, nfit, ma)

            use Variables
            use FunctionCalling

            implicit none
            integer :: i,j                                                                  !< loop variables
            integer :: ihigh                                                                !< index of the worst point
            integer :: NumFile                                                              !< Number of observation files
            integer :: MaxL                                                                 !< max. number of lines
            integer :: MaxCol                                                               !< max. number of columns
            integer :: nfit                                                                 !< number of free parameters
            integer :: ma                                                                   !< total number of parameters

            real*8, parameter :: coef_alpha = 1.5d0                                         !< Reflection coefficient
            real*8, parameter :: coef_gamma = 2.75d0                                        !< Expansion coefficient
            real*8, parameter :: coef_theta = 2.0d0                                         !< Second expansion coefficient
            real*8, parameter :: coef_beta = 0.75d0                                         !< Contraction coefficient
            real*8 :: coef_delta                                                            !< Shrinkage coefficient
            real*8 :: sumi                                                                  !< working variable for determination of center of gravity
            real*8 :: fP_high                                                               !< chi**2 for the worst point
            real*8 :: fP_sechi                                                              !< chi**2 for the second worst point
            real*8 :: fP_low                                                                !< chi**2 for the best point
            real*8 :: fP_cent                                                               !< chi**2 for the center of gravity point
            real*8 :: fP_refl                                                               !< chi**2 for the reflection point
            real*8 :: fP_exp                                                                !< chi**2 for the expansion point
            real*8 :: fP_secexp                                                             !< chi**2 for contraction point
            real*8 :: fP_cont                                                               !< chi**2 for contraction point
            real*8, dimension(nfit) :: P_high                                               !< position of the worst point
            real*8, dimension(nfit) :: P_sechi                                              !< position of the second worst point
            real*8, dimension(nfit) :: P_low                                                !< position of the best point
            real*8, dimension(nfit) :: P_cent                                               !< position of the center of gravity
            real*8, dimension(nfit) :: P_refl                                               !< position of the reflection point
            real*8, dimension(nfit) :: P_exp                                                !< position of the expansion point
            real*8, dimension(nfit) :: P_secexp                                             !< position of the second expansion point
            real*8, dimension(nfit) :: P_cont                                               !< position of the contraction point
            real*8, dimension(nfit+1) :: fun_simplex                                        !< values of chi**2 for each point in the vertex
            real*8, dimension(nfit+1,nfit) :: in_simplex                                    !< array with all positions of the vertex


            coef_delta = 0.5d0                                                              !< define Shrinkage (reduction) coefficient
            ihigh = nfit + 1                                                                !< define index of worst point
            P_low = in_simplex(1,:)                                                         !< best parameter set (point)
            fP_low = fun_simplex(1)                                                         !< chi**2 of the best parameter set
            P_high = in_simplex(nfit+1,:)                                                   !< worst parameter set
            fP_high = fun_simplex(nfit+1)                                                   !< chi**2 of the worst point x_{nfit + 1}
            P_sechi = in_simplex(nfit,:)                                                    !< second worst parameter set
            fP_sechi = fun_simplex(nfit)                                                    !< chi**2 of the second worst parameter set

            ! Debug:
            ! Do i=1,nfit+1
            !     print*,'///>',in_simplex(i,:),fun_simplex(i)
            ! end Do
            ! stop


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< Step 3:  Reflection
            !< f(x_{1}) = fP_low        (best point)
            !< f(x_{r}) = fP_refl       (reflection point)
            !< f(x_{nfit}) = fP_sechi   (second worst point)
            P_cent = 0.d0
            P_refl = 0.d0
            Do i = 1, nfit                                                                  !< loop over all parameters


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< Step 2:  calculate x_0, the center of gravity
                !< calculate x_0, the center of gravity of all points except x_{nfit + 1}, x_0 = P_cent(i)
                sumi = 0.d0
                Do j = 1, nfit
                    sumi = sumi + in_simplex(j, i)
                end Do
                P_cent(i) = sumi / nfit                                                     !< determine center of gravity x_0


                !< check, if new parameter is within limits
                if (P_cent(i) < OptimizedParameterLowLimit(i)) then
                    P_cent(i) = OptimizedParameterLowLimit(i)
                elseif (P_cent(i) > OptimizedParameterUpperLimit(i)) then
                    P_cent(i) = OptimizedParameterUpperLimit(i)
                endif


                !< Compute reflected point x_r = x_0 + \alpha (x_0 - x_{nfit+1}), x_r = P_refl(i), x_0 = P_cent(i)
                P_refl(i) = (1.d0 + coef_alpha) * P_cent(i) - coef_alpha * P_high(i)        !< P_refl(i) = x_r


                !< check, if new parameter is within limits
                if (P_refl(i) < OptimizedParameterLowLimit(i)) then
                    P_refl(i) = OptimizedParameterLowLimit(i)
                elseif (P_refl(i) > OptimizedParameterUpperLimit(i)) then
                    P_refl(i) = OptimizedParameterUpperLimit(i)
                endif
            end Do


            !< print what you do and calculate chi**2 for parameter set P_cent (fP_cent = f(x_0))
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
                print '(A,11x,"Calculate Reflection (1/2) ..",A,$)',char(13),char(13)
            endif
            fP_cent = 0.d0
            if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then     !< if calculation reduction is chosen, read chi**2 file
                call CheckCalculatedParameterSets(nfit, 1)
            endif
            call check_position(fP_cent, nfit, ma, P_cent, NumFile, MaxL, MaxCol)
            CurrentNumberLinesCalcReduction = NumberLinesChi2                               !< update CurrentNumberLinesCalcReduction variable


            !< print what you do and calculate chi**2 for parameter set P_refl (fP_cent = f(x_r))
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
                print '(A,11x,"Calculate Reflection (2/2) ..",A,$)',char(13),char(13)
            endif
            fP_refl = 0.d0
            if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then     !< if calculation reduction is chosen, read chi**2 file
                call CheckCalculatedParameterSets(nfit, 1)
            endif
            call check_position(fP_refl, nfit, ma, P_refl, NumFile, MaxL, MaxCol)
            CurrentNumberLinesCalcReduction = NumberLinesChi2                               !< update CurrentNumberLinesCalcReduction variable

            ! Debug:
            ! print*,'P_cent,fP_cent = ',P_cent, fP_cent
            ! print*,'P_refl,fP_refl = ',P_refl, fP_refl
            ! stop


            !< If the reflected point is better than the second worst, but not better than the best, i.e.: f(x_{1}) <= f(x_{r}) < f(x_{nfit}),
            !< then obtain a new simplex by replacing the worst point x_{nfit + 1} with the reflected point x_r.
            if (fP_refl >= fP_low.and.fP_refl <= fP_sechi) then                             !< test if f(x_{1}) <= f(x_{r}) .and. f(x_{r}) < f(x_{nfit}
                P_high(:) = P_refl(:)                                                       !< update P_high array
                in_simplex(ihigh,:) = P_refl(:)                                             !< replace worst point x_{nfit+1} with the reflected point x_r
                fP_high = fP_refl                                                           !< and the corresponding value of chi**2
                fun_simplex(ihigh) = fP_high                                                !< update fun_simplex array
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< Step 4:  Expansion
            !< If the reflected point is the best point so far, f(x_{r}) < f(x_{1}),
            P_exp = 0.d0
            if (fP_refl < fP_low) then


                !< compute the expanded point x_{e} = x_{0} + \gamma (x_{0} - x_{nfit + 1}), x_{e} = P_exp, x_{r} = P_refl,
                !< the corresponding chi**2 and print what you do
                if (printflag) then
                    print '(A,120(" "),A,$)',char(13),char(13)
                    print '(A,11x,"Calculate Expansion ..",A,$)',char(13),char(13)
                endif
                Do i = 1,nfit
                    P_exp(i) = coef_gamma * P_refl(i) + (1.d0 - coef_gamma) * P_cent(i)
                !   P_exp(i) = P_cent(i) + coef_gamma * (P_refl(i) - P_cent(i))             !< formula taken from wikipedia


                    !< check, if new parameter is within limits
                    if (P_exp(i) < OptimizedParameterLowLimit(i)) then
                        P_exp(i) = OptimizedParameterLowLimit(i)
                    elseif (P_exp(i) > OptimizedParameterUpperLimit(i)) then
                        P_exp(i) = OptimizedParameterUpperLimit(i)
                    endif
                end Do
                fP_exp = 0.d0
                if (UseCalculationReduction.and.CurrentNumberLinesCalcReduction > 0) then   !< if calculation reduction is chosen, read chi**2 file
                    call CheckCalculatedParameterSets(nfit, 1)
                endif
                call check_position(fP_exp, nfit, ma, P_exp, NumFile, MaxL, MaxCol)
                CurrentNumberLinesCalcReduction = NumberLinesChi2                           !< update CurrentNumberLinesCalcReduction variable

                ! Debug:
                ! print*,'P_exp,fP_exp = ',P_exp, fP_exp


                !< check if the expanded point is better than the reflected point, f(x_{e}) < f(x_{1})
                ! if (fP_exp < fP_refl) then                                                !< correct value
                if (fP_exp < fP_low) then


                    !< repeat computation of the expanded point,
                    !< the corresponding chi**2 and print what you do
                    if (printflag) then
                        print '(A,120(" "),A,$)',char(13),char(13)
                        print '(A,11x,"Calculate Second Expansion ..",A,$)',char(13),char(13)
                    endif
                    P_secexp = 0.d0
                    Do i = 1, nfit
                        P_secexp(i) = coef_theta * P_exp(i) + (1.d0 - coef_theta) * P_cent(i)


                        !< check, if new parameter is within limits
                        if (P_secexp(i) < OptimizedParameterLowLimit(i)) then
                            P_secexp(i) = OptimizedParameterLowLimit(i)
                        elseif (P_secexp(i) > OptimizedParameterUpperLimit(i)) then
                            P_secexp(i) = OptimizedParameterUpperLimit(i)
                        endif
                    end Do
                    fP_secexp = 0.d0
                    if (UseCalculationReduction.and.CurrentNumberLinesCalcReduction > 0) then   !< if calculation reduction is chosen, read chi**2 file
                        call CheckCalculatedParameterSets(nfit, 1)
                    endif
                    call check_position(fP_secexp, nfit, ma, P_secexp, NumFile, MaxL, MaxCol)
                    CurrentNumberLinesCalcReduction = NumberLinesChi2                       !< update CurrentNumberLinesCalcReduction variable

                    ! Debug:
                    ! print*,'P_secexp,fP_secexp = ',P_secexp, fP_secexp


                    if (fP_secexp < fP_low) then                                            !< obtain a new simplex by replacing the second worst
                        P_high(:) = P_secexp(:)                                             !< point x_{nfit} with the second expanded point x_{secexp}
                        fP_high = fP_secexp
                        in_simplex(ihigh,:) = P_secexp(:)
                        fun_simplex(ihigh) = fP_secexp                                      !< update fun_simplex array
                    else                                                                    !< obtain a new simplex by replacing the worst point
                        P_high(:) = P_exp(:)                                                !< x_{nfit + 1} with the expanded point x_{e}
                        fP_high = fP_exp                                                    !< update fP_high
                        in_simplex(ihigh,:) = P_exp(:)                                      !< update in_simplex array
                        fun_simplex(ihigh) = fP_exp                                         !< update fun_simplex array
                    endif
                else
                    P_high(:) = P_refl(:)
                    fP_high = fP_refl                                                       !< update fP_high
                    in_simplex(ihigh,:) = P_refl(:)                                         !< update in_simplex array with reflection point
                    fun_simplex(ihigh) = fP_refl                                            !< update fun_simplex array
                endif
            endif

            ! Debug:
            ! Do i=1,nfit+1
            !     print*,'++++>',in_simplex(i,:),fun_simplex(i)
            ! end Do


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< Step 5:  Contraction
            !< is it certain that f(x_{r}) >= f(x_{n})
            if (fP_refl > fP_sechi) then
                if (fP_refl <= fP_high) then
                    P_high(:) = P_refl(:)
                    fP_high = fP_refl
                    in_simplex(ihigh,:) = P_refl(:)
                    fun_simplex(ihigh) = fP_high
                endif


                !< Compute contracted point x_{c} = x_{nfit + 1} + \rho(x_{0} - x_{nfit + 1}), \rho = coef_beta
                !< the corresponding chi**2 and print what you do
                if (printflag) then
                    print '(A,120(" "),A,$)',char(13),char(13)
                    print '(A,11x,"Calculate Contraction ..",A,$)',char(13),char(13)
                endif
                P_cont = 0.d0
                Do i = 1, nfit
                    P_cont(i) = coef_beta * P_high(i) + (1.d0 - coef_beta) * P_cent(i)


                    !< check, if new parameter is within limits
                    if (P_cont(i) < OptimizedParameterLowLimit(i)) then
                        P_cont(i) = OptimizedParameterLowLimit(i)
                    elseif (P_cont(i) > OptimizedParameterUpperLimit(i)) then
                        P_cont(i) = OptimizedParameterUpperLimit(i)
                    endif
                end Do
                fP_cont = 0.d0
                if (UseCalculationReduction.and.CurrentNumberLinesCalcReduction > 0) then   !< if calculation reduction is chosen, read chi**2 file
                    call CheckCalculatedParameterSets(nfit, 1)
                endif
                call check_position(fP_cont, nfit, ma, P_cont, NumFile, MaxL, MaxCol)
                CurrentNumberLinesCalcReduction = NumberLinesChi2                           !< update CurrentNumberLinesCalcReduction variable

                ! Debug:
                ! print*,'P_cont,fP_cont = ',P_cont, fP_cont


                !< If the contracted point is better than the worst point, i.e. f(x_{c}) < f(x_{nfit + 1})
                if (fP_cont < fP_high) then
                    P_high(:) = P_cont(:)                                                   !< obtain a new simplex by replacing the worst point
                    fP_high = fP_cont                                                       !< x_{nfit + 1} with the contracted point x_c
                    in_simplex(nfit+1,:) = P_cont(:)                                        !< update in_simplex array
                    fun_simplex(nfit+1) = fP_high                                           !< update fun_simplex array
                else


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< Step 6:  Reduction
                    !< For all but the best point, replace the point with x_{i} = x_{1} + \sigma(x_{i} - x_{1}) for all i \in \{2, \dots, nfit + 1\}
                    Do i = 1, nfit
                        Do j = 2, (nfit + 1)
                            in_simplex(j,i) = coef_delta * in_simplex(j,i) + (1.d0 - coef_delta) * P_low(i)
                        end Do
                    end Do
                endif
            endif

            ! Debug:
            ! Do i=1,nfit+1
            !     print*,'***>',in_simplex(i,:),fun_simplex(i)
            ! end Do
            ! print*,'###################################################################'

            if (fP_cont < fP_low) then
            !    P_low(:) = P_cont(:)                                                !< update position of best point
            !    fP_low = fP_cont                                                    !< update corresponding chi**2 value
            !    in_simplex(1,:) = P_cont(:)                                         !< update in_simplex array
            !    fun_simplex(1) = fP_cont                                            !< update fun_simplex array
            endif

            return
        end subroutine hyb_sim_mod


        !*************************************************************************************************************************************************
        !> subroutine: hyb_pso_mod
        !>
        !> Implementation of Modified Particle Swarm Optimization Algorithm
        !>
        !> input variables:     X:                      array containing the parameter which should be optimized
        !>                      fitness_X:              the corresponding value of the model function
        !>                      NumFile:                number of input files
        !>                      MaxL:                   max. total length
        !>                      MaxCol:                 max. number of columns
        !>                      nfit:                   number of parameters which should be optimized
        !>                      n_particles:            number of particles
        !>                      ma:                     total number of all parameter
        !>
        !> output parameters:   X:                      array containing the optimized parameter
        !>                      fitness_X:              the corresponding value of the model function
        !>
        !> subroutine requires: OptimizedParameterLowLimit:     lower limit of the ith optimized parameter
        !>                      nfit:                   number of parameters which should be optimized
        !>                      chilm:                  value of chisq where iteration stops
        !>                      MaxPSOIter:             max. number of iterations
        !>
        !> \author Irina Bernst and Thomas Moeller
        !>
        !> \date 2010-06-09
        !>
        subroutine hyb_pso_mod(X, fitness_X, NumFile, MaxL, MaxCol, nfit, ma)

            use Variables
            use FunctionCalling

            implicit none

            !< Coefficients and constants
            integer :: nfit                                                                 !< number of free parameters
            integer :: ma                                                                   !< total number of all parameters
            integer :: metka_p                                                              !< used for stop creteria
            integer :: iterat_p                                                             !< current iteration number
            integer :: i,j, i4, k                                                           !<
            integer :: ii, jj, ch11, i22                                                    !<
            integer :: NumFile                                                              !< number of input files
            integer :: MaxL                                                                 !< max. total length
            integer :: MaxCol                                                               !< max. number of columns belonging to the "y-column"
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            real*8, parameter :: coef_lambda = 0.85d0                                       !<
            real*8, parameter :: delta_fit = 1.d0                                           !<
            real*8 :: sigma_pso                                                             !<
            real*8 :: expl, i5                                                              !<
            real*8 :: w1, ww, C1, C2, R1, R2                                                !<
            real*8 :: fitness_gbest, fitness_gbest0                                         !<
            real*8, dimension(2 * nfit, nfit) :: X, X1                                      !<
            real*8, dimension(2 * nfit) :: fitness_X                                        !<
            real*8, dimension(2 * nfit, nfit) :: V                                          !<
            real*8, dimension(nfit) :: fitness_lbest                                        !< array containing values of all free parameters
            real*8, dimension(2 * nfit) :: fitness_X1                                       !<
            real*8, dimension(2 * nfit, nfit) :: xx1Array                                   !<
            real*8, dimension(nfit) :: X_gbest                                              !< array containing values of all free parameters
            real*8, dimension(nfit) :: X0_gbest                                             !< array containing values of all free parameters
            real*8, dimension(nfit, nfit) :: X_lbest                                        !<
            real*8, dimension(5, nfit) :: Xnew_gbest                                        !<
            real*8, dimension(5) :: fitness_newgbest                                        !<
            real*8, allocatable, dimension(:) :: chi2ValuesVector                           !< chi2 value vector
            character(len=10) :: Number1, Number2, Number3                                  !< help strings for converting number to string


            !< allocate memory for variable chi2ValuesVector
            if (allocated(chi2ValuesVector)) then
                deallocate(chi2ValuesVector, stat = dealloc_status)
                if (dealloc_status /= 0) then                                               !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine hyb_pso_mod:")')
                    write(logchannel,'(2x,"Can not deallocate variable chi2ValuesVector.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine hyb_pso_mod:")'
                    print '(2x,"Can not deallocate variable chi2ValuesVector.")',
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("dealloc_status = ",I4)', dealloc_status
                    print '(" ")'
                    stop
                endif
            endif
            allocate(chi2ValuesVector(max0(5, 2 * nfit)), stat = alloc_status)
            if (alloc_status /= 0) then                                                     !< is all ok?
                write(logchannel,*)
                write(logchannel,'("Error in subroutine hyb_pso_mod:")')
                write(logchannel,'(2x,"Can not allocate variable chi2ValuesVector.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,*)
                write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')
                close(logchannel)

                print '(" ")'
                print '("Error in subroutine hyb_pso_mod:")'
                print '(2x,"Can not allocate variable chi2ValuesVector.")',
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("dealloc_status = ",I4)', dealloc_status
                print '(" ")'
                stop
            endif
            chi2ValuesVector = 0.d0


            !< initialize some variables
            X0_gbest = X(1, :)
            fitness_gbest0 = fitness_X(1)
            X_gbest = X(1, :)
            fitness_gbest = fitness_X(1)
            expl = fitness_gbest * 0.1d0


            !< Loop until convergence, in this example a finite number of iterations chosen
            sigma_pso = 1.d0
            V = 0.d0
            metka_p = 0
            iterat_p = 0
            Do While (metka_p == 0)
                iterat_p = iterat_p + 1
                X1 = 0.d0
                ii = 0
                X_lbest = 0.d0
                Do While (ii < nfit)
                    Do jj = 1, (2 * nfit), 2
                        ii = ii + 1
                        fitness_lbest(ii) = fitness_X(jj)
                        X_lbest(ii,:) = X(jj,:)
                    end Do
                end Do


                !< print what you do
                if (printflag) then
                    write(Number1, '(I10)') iterat_p
                    print '(A,120(" "),A,$)',char(13),char(13)
                    print '(A,11x,"Iter.: ",A,", Calculate Mutation heuristic ..",A,$)',char(13), trim(adjustl(Number1)), char(13)
                endif


                !< Mutation heuristic
                Xnew_gbest = 0.d0
                Do i = 1, 5
                    Do j = 1, nfit
                        Xnew_gbest(i, j) = X_gbest(j) + RandomWithLimits(0.d0, sigma_pso)
                        if (Xnew_gbest(i, j) < OptimizedParameterLowLimit(j)) then
                            Xnew_gbest(i, j) = OptimizedParameterLowLimit(j)
                        elseif (Xnew_gbest(i, j) > OptimizedParameterUpperLimit(j)) then
                            Xnew_gbest(i, j) = OptimizedParameterUpperLimit(j)
                        endif
                    end Do
                end Do


                !< if calculation reduction is chosen, read chi**2 file
                if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then
                    call CheckCalculatedParameterSets(nfit, ParallelizationFlag)
                endif

                ! Debug:
                ! Do i = 1,5
                !     print*,'=>',i,Xnew_gbest(i,:)
                !  end Do
                ! print*,'--------------------------------'
                ! stop


                !< print what you do
                if (printflag) then
                    print '(A,11x,"Iter.: ",A,", Calculate Mutation heuristic ..                    ",A,$)',char(13), trim(adjustl(Number1)), char(13)
                endif


                !< determine chi2 values
                chi2ValuesVector = 0.d0
                call ModelCalcChiFunctionGeneral(ma, ia, paramset(1, :), 5, nfit, NumFile, MaxL, MaxCol, Xnew_gbest, chi2ValuesVector(1:5))
                fitness_newgbest(1:5) = chi2ValuesVector(1:5)


                !< update CurrentNumberLinesCalcReduction variable
                CurrentNumberLinesCalcReduction = NumberLinesChi2

                ! Debug:
                ! Do i = 1, 5
                !     print*,'>',Xnew_gbest(i,:),fitness_newgbest(i)
                ! end Do
                ! print*,'####################################################'


                !< update sigma_pso
                ch11 = 0
                i22 = 2
                Do i = 1, 5
                    if (fitness_newgbest(i) < fitness_gbest) then
                        ch11 = ch11 + 1
                    endif
                    if (i /= 1) then
                        if (fitness_newgbest(i22) < fitness_newgbest(i22 - 1)) then
                            i22 = i
                        endif
                    endif
                end Do
                if (ch11 > 2) then
                    sigma_pso = sigma_pso * (1.d0/coef_lambda)
                endif
                if (ch11 < 2) then
                    sigma_pso = sigma_pso * coef_lambda
                endif
                if (ch11 /= 0 .and. fitness_gbest > fitness_newgbest(i22)) then
                    fitness_gbest = fitness_newgbest(i22)
                    X_gbest = Xnew_gbest(i22,:)
                endif


                !< Update the particle velocity and position
                w1 = RandomWithLimits(0.d0, 1.d0)
                ww = 0.5d0 + w1/2.d0
                C1 = 0.6d0
                C2 = 1.6d0
                i4 = 1


                !< print what you do
                if (printflag) then
                    print '(A,120(" "),A,$)',char(13),char(13)
                    print '(A,11x,"Iter.: ",A,", Calculate particle velocity and position ..",A,$)',char(13), trim(adjustl(Number1)), char(13)
                endif


                xx1Array = 0.d0
                Do i = 1, (2 * nfit)
                    i5 = i/2.d0
                    if (i /= 1 .and. (i5 - int(i5)) /= 0.d0) then
                        i4 = i4 + 1
                    endif
                    if (i4 > nfit) then
                        i4 = nfit - 1
                    endif
                    Do j = 1, nfit
                        R1 = RandomWithLimits(0.d0, 1.d0)
                        R2 = RandomWithLimits(0.d0, 1.d0)
                        if (V(i, j) /= 0.d0) then
                            metka_p = 1
                        endif

                        ! Debug:
                        ! print*,'>>=>1:=',ww * V(i,j), ww,V(i,j)
                        ! print*,'>>=>2:=', C1 * R1 * (X_lbest(i4,j) - X(i,j)), C1,R1,X_lbest(i4,j), X(i,j)
                        ! print*,'>>=>3:=', C2 * R2 * (X_gbest(j) - X(i,j)), C2, R2, X_gbest(j), X(i,j)

                        V(i, j) = (ww * V(i, j) + C1 * R1 * (X_lbest(i4, j) - X(i, j)) + C2 * R2 * (X_gbest(j) - X(i, j)))
                        X(i, j) = X(i, j) + V(i, j)


                        !< check, if new parameter is within limits
                        if (X(i, j) < OptimizedParameterLowLimit(j)) then
                            X(i, j) = OptimizedParameterLowLimit(j)
                        elseif (X(i, j) > OptimizedParameterUpperLimit(j)) then
                            X(i, j) = OptimizedParameterUpperLimit(j)
                        endif
                    end Do
                    xx1Array(i, :) = X(i, :)


                    !< print what you do
                    if (printflag) then
                        write(Number2, '(I10)') i
                        write(Number3, '(I10)') (2 * nfit)
                        print '(A,120(" "),A,$)',char(13),char(13)
                        print '(A,11x,"Iter.: ",A,", Calculate particle velocity and position  (",A,"/",A,") ..",A,$)',char(13), trim(adjustl(Number1)), &
                                                                                                  trim(adjustl(Number2)), trim(adjustl(Number3)), char(13)
                    endif
                end Do
                if (printflag) then
                    write(Number2, '(I10)') (2 * nfit)                                      !< convert number n to string for output
                endif


                !< if calculation reduction is chosen, read chi**2 file
                if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then
                    call CheckCalculatedParameterSets(nfit, ParallelizationFlag)
                endif

                ! Debug:
                !  Do i = 1,2*nfit
                !      print*,'->',i,xx1Array(i,:)
                !  end Do
                !  print*,'#############################################################################'


                !< print what you do ..
                if (printflag) then
                    print '(A,120(" "),A,$)', char(13), char(13)
                    print '(A,11x,"Check position  ..                    ",A,$)', char(13), char(13)
                endif


                !< determine chi2 values
                chi2ValuesVector = 0.d0
                call ModelCalcChiFunctionGeneral(ma, ia, paramset(1, :), (2 * nfit), nfit, NumFile, MaxL, MaxCol, xx1Array, chi2ValuesVector(1:2*nfit))
                fitness_X = chi2ValuesVector(1:2*nfit)
                fitness_X1 = chi2ValuesVector(1:2*nfit)


                !< update CurrentNumberLinesCalcReduction variable
                CurrentNumberLinesCalcReduction = NumberLinesChi2


                !< print what you do
                if (printflag) then
                    print '(A,120(" "),A,$)',char(13),char(13)
                endif


                !< sort fitnesses
                call sort(2 * nfit, fitness_X)

                ! Debug:
                ! Do i = 1, 2*nfit
                !     print*,'>',i,fitness_X(i)
                ! end Do
                ! print*,'################################################'


                !< get parameter sets for the different fitnesses
                Do i = 1, (2 * nfit)
                    Do j = 1, (2 * nfit)
                        if (fitness_X1(j) == fitness_X(i)) then
                            Do k = 1, nfit
                                X1(i,k) = X(j,k)
                            end Do
                        endif
                     end Do
                end Do
                X = X1
                X_gbest = X(1,:)
                fitness_gbest = fitness_X(1)


                !< if loop does not converge
                if (iterat_p > max(MaxPSOIter, 10)) then
                    exit
                endif
            end Do
            return
        end subroutine hyb_pso_mod


        !*************************************************************************************************************************************************
        !> subroutine: CallPSO
        !>
        !> input variables:     nfit:                   number of parameters which should be optimized
        !>                      ma:                     total number of all parameter
        !>                      n_particles:            number of particles
        !>                      counter:                number of best sites
        !>                      colx:                   number of columns belonging to the x-column
        !>                      NumFile:                number of input files
        !>                      MaxL:                   max. total length
        !>                      MaxCol:                 max. number of columns
        !>                      PlotIteration:          dummy argument
        !>                      PlotType:               kind of plotting
        !>                      xAxisLabel:             label for x-axis
        !>                      yAxisLabel:             label for y-axis
        !>                      zAxisLabel:             label for z-axis
        !>
        !> output variables:    counter:                number of best sites
        !>                      Parameterset:           paramsets for the best sites
        !>
        !> \author Thomas Moeller
        !>
        !> \date 22.06.2010
        !>
        subroutine CallPSO(nfit, ma, n_particles, counter, Parameterset, colx, NumFile, MaxL, MaxCol, PlotIteration, PlotType, xAxisLabel, &
                           yAxisLabel, zAxisLabel, fitlog)
            !< subroutine to call the different pso subroutines

            use Variables
            use FunctionCalling

            implicit none
            integer :: i, j, k                                                              !< loop variables
            integer :: nfit                                                                 !< number of free parameters
            integer :: ma                                                                   !< total number of all parameters
            integer :: counter                                                              !< counts the number of parameter sets
            integer :: colx                                                                 !< max. number of columns belonging to "x-column"
            integer :: NumFile                                                              !< number of exp. files
            integer :: MaxL                                                                 !< max. number of lines
            integer :: MaxCol                                                               !< max. number of columns belonging to "y-column"
            integer :: n_particles                                                          !< number of particles
            integer :: i1, i2, i3, i11, metka                                               !< working variables
            integer :: NumInputFiles                                                        !< needed for loop over input files
            integer :: PlotIteration                                                        !< flag for plotting the model function for each step
            integer :: PlotType                                                             !< get type of plot
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            real*8 :: l, r1, r2                                                             !<
            real*8 :: fun_min, dummy                                                        !<
            real*8, dimension(n_particles, nfit) :: in_swarm, in_swarm1                     !<
            real*8, dimension(nfit + 1, nfit) :: in_simplex                                 !<
            real*8, dimension(nfit + 1) :: fun_simplex                                      !<
            real*8, dimension(2 * nfit, nfit) :: in_pso                                     !<
            real*8, dimension(2 * nfit) :: fun_pso                                          !<
            real*8, dimension(nfit) :: x0                                                   !< Starting point
            real*8, dimension(nfit) :: Xmin                                                 !< Result
            real*8, dimension(nfit) :: xx                                                   !<
            real*8, dimension(nfit) :: delxx                                                !<
            real*8, dimension(colx) :: posdatexp                                            !< array for point within observation data
            real*8, dimension(ma) :: acopy                                                  !< copy of total parameter list
            real*8, dimension(n_particles) :: fun_swarm                                     !< values of chi**2 for each particle
            real*8, dimension(counter, ma) :: Parameterset                                  !< array containing the parameter set which fullfil the
                                                                                            !< quality creteria
            character(len=25) :: LongNumber1, LongNumber2                                   !< working variables
            character(len=100) :: HelpString                                                !< used for number to string conversion
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=512) :: LongHelpString1                                           !< help string
            character(len=512) :: WorkingDirectory1, WorkingDirectory2                      !< path and fi
            character(len=5196) :: ListParamFormated                                        !< string for screen output of parameter set
            character(len=8192) :: fitlog                                                   !< path of log files
            logical :: IntegerTrue                                                          !<
            logical :: InitPlotFlag                                                         !<


            !< initialize ran1
            idum = (-1)
            call ran1(dummy, idum)


            !< Coefficient and constant
            ! sigma = 1.d0   !<--- not necessary


            !< define OptimizedParameter range and determine starting point
            OptimizedParameterLowLimit = 0.d0
            OptimizedParameterUpperLimit = 0.d0
            x0 = 0.d0
            j = 0
            Do i = 1, parameternumber
                if (ia(i)) then
                    j = j + 1
                    OptimizedParameterLowLimit(j) = paramset(3, i)
                    OptimizedParameterUpperLimit(j) = paramset(4, i)
                    x0(j) = RandomWithLimits(OptimizedParameterLowLimit(j), OptimizedParameterUpperLimit(j))
                endif
            end Do


            !< Initial Swarm (Simplex + 2*N particles)
            in_swarm = 0.d0
            l = 1.d0
            r1 = l * (((nfit + 1.d0)**0.5 + nfit - 1.d0)/(nfit * (2.d0**0.5)))
            r2 = l * (((nfit + 1.d0)**0.5 - 1.d0)/(nfit * (2.d0**0.5)))
            in_swarm(1, :) = x0(:)

            ! Debug:
            ! print*,' '
            ! print*,'nfit = ',nfit
            ! print*,'n_particles = ',n_particles


            !< ??
            Do i = 1, nfit
                delxx(i) = (OptimizedParameterUpperLimit(i) - OptimizedParameterLowLimit(i)) / 2.d0
                Do j = 2, (nfit + 1)
                    if (i == (j - 1)) then
                        in_swarm(j, i) = x0(i) + r1
                    else
                        in_swarm(j, i) = x0(i) + r2
                    endif

                    ! Debug:
                    ! print*,' '
                    ! print*,'j,i = ',j,',',i
                    ! print*,'in_swarm(j,i) = ',in_swarm(j,i)
                    ! print*,' r1,r2 = ',r1,r2

                end Do
                i1 = 0
                i2 = 0
                i3 = 0
                Do k = (nfit + 1), n_particles
                    in_swarm(k, :) = x0(:)
                    if (k <= (2 * nfit)) then
                        i1 = i1 + 1
                        in_swarm(k, i1) = in_swarm(k, i1) + delxx(i)


                        !< check bounds
                        call check_parameters(in_swarm(k, i1), OptimizedParameterLowLimit(i1), OptimizedParameterUpperLimit(i1))

                        ! Debug:
                        ! print*,' '
                        ! print*,'k, i1 = ',k,',', i1
                        ! print*,'in_swarm(k, i1) = ',in_swarm(k, i1)
                        ! print*,'delxx(i)       = ',delxx(i)
                        ! print*,'x0(:)          = ',x0(:)

                    endif
                    if (k >= (2 * nfit + 1) .and. k < n_particles) then
                        i2 = i2 + 1
                        in_swarm(k, i2) = in_swarm(k, i2) - delxx(i)


                        !< check bounds
                        call check_parameters(in_swarm(k, i2), OptimizedParameterLowLimit(i2), OptimizedParameterUpperLimit(i2))

                        ! Debug:
                        ! print*,' '
                        ! print*,'k,i2 = ',k,',',i2
                        ! print*,'in_swarm(k, i2) = ',in_swarm(k, i2)

                    endif
                    if (k == n_particles) then
                        i3 = i3 + 1
                        in_swarm(k, :) = in_swarm(k, i3) + x0(i3)
                    endif
                end Do
            end Do


            !< check, if new parameters are within limits
            Do i = 1, nfit
                Do k = 1, n_particles
                    if (in_swarm(k, i) < OptimizedParameterLowLimit(i)) then
                        in_swarm(k, i) = OptimizedParameterLowLimit(i)
                    elseif (in_swarm(k, i) > OptimizedParameterUpperLimit(i)) then
                        in_swarm(k, i) = OptimizedParameterUpperLimit(i)
                    endif
                end Do
            end Do

            ! Debug:
            ! print*,'>>', nfit, n_particles
            ! Do k = 1, n_particles
            !     print*,'>', k, in_swarm(k, :)
            ! end Do
            ! stop


            !< print what you do
            if (printflag) then
                print '(11x,"Number of particles = ",I10)', n_particles
                print '(" ")'
                print '(" ")'
                print '(11x,"Iteration:",20x,"chi^2:",5x,"Parameter:")'
                print '(11x,"Initialize model function ..",20(" "),A1,$ )',char(13)
            endif
            write(logchannel,'(11x,"Number of particles = ",I10)') n_particles
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')
            write(logchannel,'(11x,"Iteration:",20x,"chi^2:",5x,"Parameter:")')


            !< sort algorithm
            in_swarm1 = 0.d0
            fun_swarm = 0.d0
            call sort_swarm(in_swarm1, in_swarm, fun_swarm, NumFile, MaxL, MaxCol, nfit, n_particles, ma)
            in_swarm = in_swarm1


            !< Algorithm NM-PSO
            iterat = -1
            metka = 0
            Do While (metka == 0)
                iterat = iterat + 1


                !< Modified Simplex
                in_simplex(1:nfit + 1, 1:nfit) = in_swarm1(1:(nfit + 1), 1:nfit)
                fun_simplex(1:nfit + 1) = fun_swarm(1:(nfit + 1))

                ! Debug:
                ! Do i = 1, n_particles
                !     print*,'==>',fun_swarm(i),in_swarm1(i,:)
                ! end Do
                ! print*,'----------------------------------------------------------------'


                !< call Nelder-Mead Simplex Algorithm
                call hyb_sim_mod(in_simplex, fun_simplex, NumFile, MaxL, MaxCol, nfit, ma)
                in_swarm1(1:nfit + 1, 1:nfit) = in_simplex(1:nfit + 1, 1:nfit)

                ! Debug:
                ! Do i = 1, nfit+1
                !     print*,'==>',in_swarm1(i,:)
                ! end Do
                ! print*,'>',(nfit+1),n_particles, n_particles - (nfit+1)
                ! print*,'>',((2*nfit)+1), ((2*nfit)+1) - 1


                !< Modified PSO
                in_pso(1:(2 * nfit), 1:nfit) = in_swarm1((nfit + 2):n_particles, 1:nfit)
                fun_pso(1:(2 *nfit)) = fun_swarm((nfit + 2):n_particles)


                !< call Modified Particle Swarm Optimization Algorithm
                call hyb_pso_mod(in_pso, fun_pso, NumFile, MaxL, MaxCol, nfit, ma)
                in_swarm1((nfit + 2):n_particles, 1:nfit) = in_pso(1:(2 * nfit), 1:nfit)
                in_swarm = in_swarm1

                ! Debug:
                ! Do i = 1, n_particles
                !     print*,'-->',in_swarm(i, :)
                ! end Do


                !< check, if new parameters are within limits
                Do i = 1, nfit
                    Do k = 1, n_particles
                        if (in_swarm(k, i) < OptimizedParameterLowLimit(i)) then
                            in_swarm(k, i) = OptimizedParameterLowLimit(i)
                        elseif (in_swarm(k, i) > OptimizedParameterUpperLimit(i)) then
                            in_swarm(k, i) = OptimizedParameterUpperLimit(i)
                        endif
                    end Do
                end Do


                !< sort algorithm
                in_swarm1 = 0.d0
                fun_swarm = 0.d0
                call sort_swarm(in_swarm1, in_swarm, fun_swarm, NumFile, MaxL, MaxCol, nfit, n_particles, ma)


                !< is iteration > 1
                if (iterat > 0) then
                    ! fun_min = minval(fun_swarm)
                    fun_min = BestSitesParamSet(1, 1)
                    i11 = 0
                    Do i = 1, n_particles
                        if (fun_min == fun_swarm(i)) then
                            i11 = i
                            Xmin(:) = in_swarm1(i,:)
                            exit
                        endif
                    end Do
                    if (i11 == 0) then                                                      !< if overall best site is not one of the current particle
                        i11 = 1                                                             !< set first particle to the best position (parameter set)
                        in_swarm1(i11,:) = BestSitesParamSet(1, 2:)
                        Xmin(:) = in_swarm1(i11,:)
                    endif


                    !< call stop subroutine
                    call stop_crit(in_swarm, metka, nfit, n_particles, i11, fun_min)


                    !< print what you do!
                    k = 0
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            a(j) = BestSitesParamSet(1, k + 1)
                        endif
                    end Do


                    !< build list with fit parameters
                    k = 0
                    ListParamFormated = ""
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                            if (IntegerTrue) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(j))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(a(j))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(j)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) a(j)
                                endif
                            endif
                            if (k == 1) then
                                ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                            else
                                ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                            endif
                        endif
                    end Do


                    !< print status of iteration process ..
                    if (printflag) print '(11x,I10,ES26.15,5x,A)',iterat, fun_min, trim(adjustl(ListParamFormated))
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("  ")')
                    write(paramchannel,'(123("*"))')
                    write(paramchannel,'("Iteration: ",I5,",  chi^2 = ",ES25.15)') iterat, fun_min
                    write(logchannel,'(11x,I10,ES26.15,5x,A)') iterat, fun_min, trim(adjustl(ListParamFormated))


                    !< write actual parameters to files
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("Parameters: ",A)') trim(adjustl(ListParamFormated))
                    write(paramchannel,'(123("-"))')
                    write(paramchannel,'("  ")')


                    !< save current experimental x point of the first experimental file to variable posdatexp
                    posdatexp(1:colx) = 0.d0


                    !< call subroutine to write current values of the parameter to file
                    write(paramchannel,'("-",61(" -"))')
                    Do NumInputFiles = 1, NumberInputFiles
                        write(paramchannel,'("Input-File ",I5,":  , file: ",A)') NumInputFiles, trim(adjustl(FitFktInput(NumInputFiles)))
                        write(paramchannel,'("  ")')
                        write(paramchannel,'("-start_input-file",106("-"))')
                        call WriteParameter(paramchannel, .true., colx, posdatexp, parameternumber, a, NumInputFiles)
                        write(paramchannel,'("-end_input-file",108("-"))')
                    end Do


                    !< plot experimental data, model function, and chi**2
                    if (PlotIteration == 0) then
                        if (iterat == 1) then
                            InitPlotFlag = .true.
                        else
                            InitPlotFlag = .false.
                        endif
                        call PlotFitFunction(InitPlotFlag, xAxisLabel, yAxisLabel, zAxisLabel)
                    endif
                endif


                !< print reason for stop of iteration
                if (iterat > 0) then
                    write(paramchannel,'("  ")')
                    write(paramchannel,'(123("="))')
                    write(paramchannel,'("  ")')
                    if (fun_min <= chi2lim) then
                        write(LongNumber1,'(ES25.15)') fun_min
                        write(LongNumber2,'(ES25.15)') chi2lim
                        if (printflag) then
                            print '(" ")'
                            print '(11x,"Iteration stopped. chi^2 (=", A, ") dropped below limit = ", A)', trim(adjustl(LongNumber1)), &
                                                                                                          trim(adjustl(LongNumber2))
                        endif
                        write(logchannel,'("  ")')
                        write(logchannel,'(11x,"Iteration stopped. chi^2 (=", A, ") dropped below limit = ", A)') trim(adjustl(LongNumber1)), &
                                                                                                                  trim(adjustl(LongNumber2))

                    elseif (iterat >= MaxPSOIter) then
                        if (printflag) then
                            print '(" ")'
                            print '(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ", I6)', MaxPSOIter
                        endif
                        write(logchannel,'("  ")')
                        write(logchannel,'(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ", I6)') MaxPSOIter
                    endif
                endif
            end Do


            !< determine the number of best sites
            j = 0
            Do i = 1, QualityLimit
                if (BestSitesParamSet(i, 1) /= 1.d99) then
                    j = j + 1
                endif
            end Do
            QualityLimit = j


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< select best sites
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(11x,"Best results:")'
            endif
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')
            write(logchannel,'(11x,"Best results:")')


            !< write best results to screen and to log-file
            Do i = 1, QualityLimit
                acopy = a
                fun_min = BestSitesParamSet(i, 1)
                xx = BestSitesParamSet(i, 2:)


                !< build list with fit parameters
                k = 0
                ListParamFormated = ""
                Do j = 1, parameternumber
                    if (ia(j)) then
                        k = k + 1
                        acopy(j) = xx(k)

                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                        if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                            .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(acopy(j))
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) int(acopy(j))
                            endif
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) acopy(j)
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) acopy(j)
                            endif
                        endif
                        if (k == 1) then
                            ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                        else
                            ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                        endif
                    endif
                end Do
                Parameterset(i, :) = acopy(:)


                !< print best results to screen and to log-file
                if (printflag) then
                    print '(13x,"Site number: ",I4,", chi**2 = ",ES25.15,",  Parameterset = ",A)', i, fun_min, trim(adjustl(ListParamFormated))
                endif
                write(logchannel,'(13x,"Site number: ",I4,", chi**2 = ",ES25.15,",  Parameterset = ",A)') i, fun_min, trim(adjustl(ListParamFormated))
            end Do


            !< clear screen massage
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
            endif


            !< write model functions values and chi2 values for good and best sites to scratch file
            write(LongHelpString1,'(I30)') JobID                                            !< write JobID to string
            WorkingDirectory1 = trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(LongHelpString1)) // "/" // "FitFunction.dat"
            WorkingDirectory2 = trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(LongHelpString1)) // "/" // "Chi2Values.dat"
            return
        end subroutine CallPSO
end Module PSOVariables
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: Algorithm
!>
!>         Module contains the main subroutine used to start the different versions of the PSO algorithm
!>
!>
!> \author Thomas Moeller
!>
!> \date 01.09.2014
!>
Module Algorithm

    use Variables
    use PSOVariables

    implicit none

    contains


        !*************************************************************************************************************************************************
        !> subroutine: MainAlg
        !>
        !> main subroutine which starts the PSO algorithm
        !>
        !>
        !> input variables:         printflagNum:           flag for screen output 1 (=yes) or 0 (=no)
        !>                          LastAlgorithmNum:       number of last algorithm
        !>                          chilm:                  user defined abort criteria for chi**2
        !>                          NumberOfFitAlgorithms:  total number of all algorithms in the chain
        !>                          numiter:                max. number of iterations
        !>                          PSOCounter:             counts number of calls
        !>                          DeterminationChi2:      method being used for the determination of chi^2
        !>                          PlotIteration:          plot model function for each iteration set 1(=yes) or 0(=no)
        !>                          PlotTypeOrg:            get type of plot
        !>                          fitlog:                 path for log-file containing the current values of chi**2
        !>                          NumberInputFilesorg:    number of input files for the external model program
        !>                          NumberOutputFilesOrg:   number of output files for the external model program
        !>                          ParallelizationFlagorg: contains the number of processors used for parallelization
        !>                          JobIDorg:               job identification number
        !>                          MaxInputLinesOrg:       max number of lines in an input file
        !>                          MaxParameterOrg:        max number of parameters in a line of an input file
        !>                          RenormalizedChi2Org:    flag for using renormalized chi**2
        !>                          currentpathorg:         path of the working directory
        !>                          FitParameterNameLocal:  array containing the names of the model parameters
        !>                          FitParameterValueLocal: array containing the values of the model parameters as string
        !>                          CalculationMethodOrg:   method of computation (at once or point-to-point)
        !>                          xAxisLabel:             label of the x-axis (for plot)
        !>                          yAxisLabel:             label of the y-axis (for plot)
        !>                          zAxisLabel:             label of the z-axis (for plot)
        !>                          PathStartScriptOrg:     path and name of the start script for calling model function
        !>                          ExeCommandStartScriptOrg:   command for calling model function
        !>                          parametersetorg:        the complete set of paramters (incl. flags and limits)
        !>                          expdataxorg:            array containing the experimental x side
        !>                          expdatayorg:            array containing the experimental y side
        !>                          expdataerrororg:        array containing the experimental error of the y side
        !>                          NumberRangesOrg:        number of y-columns for each experimental file
        !>                          MinRangeOrg:            array containing the minimal exp. ranges
        !>                          MaxRangeOrg:            array containing the maximal exp. ranges
        !>                          NumberXColumnsOrg:      number of x-columns for each experimental file
        !>                          NumberYColumnsOrg:      number of y-columns for each experimental file
        !>                          lengthexpdataorg:       number of lines in experimental data
        !>                          MaxRangeNumber:         max. number of ranges
        !>                          NumFileOrg:             number of experimental files
        !>                          MaxLengthOrg:           max length of experimental data
        !>                          MaxColXOrg:             number of columns concerning to the experimental x side
        !>                          MaxColYOrg:             number of columns concerning to the experimental y side
        !>                          parameternum:           number of model parameter
        !>                          SortFortranNum:         sort chi^2 log file by fortran
        !>
        !> output variables:        calstatus:              status flag of calculation (= 0: all ok)
        !>                          FitFunctionOut:         values of the model function at the calculated points
        !>                          Chi2Values:             values of the chi^2 function at the calculated points
        !>                          FinalParameterSet:      the complete set of paramters (incl. flags and limits)
        !>
        subroutine MainAlg(printflagNum, LastAlgorithmNum, calstatus, FitFunctionOut, Chi2Values, chilm, NumberOfFitAlgorithms, numiter, PSOCounter, &
                           ParamSetCounter, GeneralAlgorithmSettings, DeterminationChi2, PlotIteration, PlotType, fitlog, NumberInputFilesorg, &
                           NumberOutputFilesOrg, ParallelizationFlagorg, JobIDorg, MaxInputLinesOrg, MaxParameterOrg, RenormalizedChi2Org, &
                           currentpathorg, FitParameterNameLocal, FitParameterValueLocal, CalculationMethodOrg, xAxisLabel, yAxisLabel, zAxisLabel, &
                           PathStartScriptOrg, ExeCommandStartScriptOrg, parametersetorg, FinalParameterSet, expdataxorg, expdatayorg, expdataerrororg, &
                           NumberRangesOrg, MinRangeOrg, MaxRangeOrg, NumberXColumnsOrg, NumberYColumnsOrg, lengthexpdataorg, MaxRangeNumber, &
                           NumFileOrg, MaxLengthOrg, MaxColXOrg, MaxColYOrg, parameternum, SortFortranNum)


            implicit none
            ! ********** input variables **********
            integer :: parameternum                                                         !< number of model parameter
            integer :: NumberOfFitAlgorithms                                                !< total number of all algorithms in the chain
            integer :: numiter                                                              !< max. number of iterations
            integer :: NumFileOrg                                                           !< number of experimental files
            integer, dimension(NumFileOrg) :: lengthexpdataorg                              !< number of lines in experimental data
            integer, dimension(NumFileOrg) :: NumberXColumnsOrg                             !< number of x-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberYColumnsOrg                             !< number of y-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberRangesOrg                               !< number of y-columns for each experimental file
            integer :: MaxColXOrg                                                           !< number of columns concerning to the experimental x side
            integer :: MaxColYOrg                                                           !< number of columns concerning to the experimental y side
            integer :: MaxLengthOrg                                                         !< max length of experimental data
            integer :: printflagNum                                                         !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: LastAlgorithmNum                                                     !< number of last algorithm
            integer :: SortFortranNum                                                       !< flag indicating if chi2 log file is sorted by fortran
                                                                                            !< yes (=1) or not (=0)
            integer :: DeterminationChi2                                                    !< method being used for the determination of chi^2
            integer :: PlotIteration                                                        !< plot model func. for each iteration set 1(=yes) or 0(=no)
            integer :: PlotTypeOrg                                                          !< get type of plot
            integer :: NumberInputFilesorg                                                  !< number of input files for the external model program
            integer :: NumberOutputFilesOrg                                                 !< number of output files for the external model program
            integer :: ParallelizationFlagorg                                               !< contains the number of processors used for parallelization
            integer :: JobIDorg                                                             !< job identification number
            integer :: MaxInputLinesOrg                                                     !< max number of lines in an input file
            integer :: MaxParameterOrg                                                      !< max number of parameters in a line of an input file
            integer :: RenormalizedChi2Org                                                  !< flag for using renormalized chi**2
            integer :: PSOCounter                                                           !< counts number of calls
            integer :: ParamSetCounter                                                      !< number of best sites
            integer :: NumberParticles                                                      !< number of particles
            integer :: MaxRangeNumber                                                       !< max. number of ranges
            real*8 :: chilm                                                                 !< user defined abort criteria for chi**2
            real*8, dimension(15) :: GeneralAlgorithmSettings                               !< special algorithm settings
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColXOrg) :: expdataxorg          !< array containing the experimental x side
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColYOrg) :: expdatayorg          !< array containing the experimental y side
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColYOrg) :: expdataerrororg      !< array containing the experimental error of the y side
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MinRangeOrg        !< array containing the minimal exp. ranges
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MaxRangeOrg        !< array containing the maximal exp. ranges
            character(len=8192) :: fitlog                                                   !< path for log-file containing the current values of chi**2
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=20) :: CalculationMethodOrg                                       !< method of computation (at once or point-to-point)
            character(len=8192) :: PathStartScriptOrg                                       !< path and name of the start script for calling model func.
            character(len=8192) :: ExeCommandStartScriptOrg                                 !< command for calling model function
            character(len=8192) :: currentpathorg                                           !< path of the working directory
            character(len=512), dimension(parameternum) :: FitParameterNameLocal            !< array containing the names of the model parameters
            character(len=512), dimension(parameternum) :: FitParameterValueLocal           !< array containing the values of the model parameters as


            ! ********** in/output variables **********
            real*8, dimension(4, parameternum) :: parametersetorg                           !< the non-optimized (initial parameter set)


            ! ********** output variables **********
            integer :: calstatus                                                            !< the following line is necessary for f2py
            real*8, dimension(ParamSetCounter, parameternum) :: FinalParameterSet           !< array containing the optimized parameter set
            real*8, dimension(ParamSetCounter, NumFileOrg, MaxLengthOrg, MaxColYOrg) :: FitFunctionOut !< values of the model func. at the calculated pts.
            real*8, dimension(ParamSetCounter, NumFileOrg, MaxLengthOrg, MaxColYOrg) :: Chi2Values     !< values of the model func. at the calculated pts.


            ! ********** working variabels **********
            integer :: i, j, k, ii, jj                                                      !< working variables
            integer :: nfit, ma, n_particles, MaxLength, PlotType
            integer :: ok                                                                   !< status of calculation
            integer :: actualiteration                                                      !< contains the current iteration within the iteration loop
            integer :: flag                                                                 !< working variable used within the iteration loop
            integer :: NumInputFiles                                                        !< need for loop over input files
            integer :: allocstatus, deallocstatus                                           !< working variables for allocation/deallocation
            integer, dimension(8) :: VALUES                                                 !< value for the date_and_time subroutine
            real*8, dimension(1) :: chi2ValuesVector                                        !< here only one chi2 value

            character(len=512) :: fitlogparam                                               !< path for log-file containing the current parameter values
            character(len=512) :: fitlogChi2                                                !< path for log-file containing chi**2 and the corresponding
                                                                                            !< parameter values
            character(len=8) :: DATE                                                        !< variable for the date_and_time subroutine
            character(len=10) :: TIME                                                       !< variable for the date_and_time subroutine
            character(len=5) :: ZONE                                                        !< variable for the date_and_time subroutine
            character(len=10) :: Number1                                                    !< variable for number to string converting
            character(len=100) :: helpString                                                !< help string
            character(len=8192) :: SiteExt, NumExt, BaseDir, FuncCallExt                    !< working variables for final input file name


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< set print flag and last algorithm flag
            calstatus = 0                                                                   !< set calculation status to 0 = everything is ok
            if (printflagNum == 1) then                                                     !< set printflag
                printflag = .true.
            else
                printflag = .false.
            endif
            if (LastAlgorithmNum == 1) then
                LastAlgorithmFlag = .true.
            else
                LastAlgorithmFlag = .false.
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy contents of some input variables to module variables
            NumberExpFiles = NumFileOrg                                                     !< copy number of experimental files to global variable
            currentpath = trim(adjustl(currentpathorg))                                     !< copy path of working directory to module variable without
                                                                                            !< trailing and leading blanks
            MaxColX = MaxColXOrg                                                            !< copy number of columns of the experimental x data to
                                                                                            !< module variable
            MaxColY = MaxColYOrg                                                            !< copy number of columns of the experimental y data to
                                                                                            !< module variable
            MaxLength = MaxLengthOrg                                                        !< copy max. number of lines
            MaxExpLength = MaxLength                                                        !< copy max. number of exp. data points
            MaxNumberRanges = MaxRangeNumber                                                !< copy of max. number of data ranges in a exp. data file
            parameternumber = parameternum                                                  !< copy input variable containing the number of parameters
                                                                                            !< to module variable
            DetChi2 = DeterminationChi2                                                     !< copy method of chi**2 determination
            NumberInputFiles = NumberInputFilesorg                                          !< copy number of input files for the external program to
                                                                                            !< global variable
            NumberOutputFiles = NumberOutputFilesOrg                                        !< copy number of output files for the external program to
                                                                                            !< global variable
            ParallelizationFlag = ParallelizationFlagorg                                    !< copy number of used processors to global variable
            JobID = JobIDorg                                                                !< copy job-ID number to global variable
            PlotType = PlotTypeOrg                                                          !< copy flag for plotting
            MaxInputLines = MaxInputLinesOrg                                                !< copy max number of input lines in an input file
            MaxParameter = MaxParameterOrg                                                  !< copy max number of parameters in a line of an input file
            MaxPSOIter = numiter                                                            !< copy max. number of iteration
            RenormalizedChi2 = .true.                                                       !< define flag for using renormalized chi**2
            if (RenormalizedChi2Org /= 1) then
                RenormalizedChi2 = .false.
            endif
            PlotIterationFlag = .false.
            if (PlotIteration == 0) PlotIterationFlag = .true.


            !< get special algorithm settings
            NumberParticles = int(GeneralAlgorithmSettings(3))
            QualityLimit = ParamSetCounter                                                  !< which positions should be used

            ! Debug:
            !    print*,'PlotIteration = ', PlotIteration
            !    print*,'PlotIterationFlag = ', PlotIterationFlag
            !    print*,'ParamSetCounter = ', ParamSetCounter
            !    print*,'NumberParticles = ', NumberParticles
            !    print*,'PathStartScriptOrg = ',trim(PathStartScriptOrg)
            !    print*,'ExeCommandStartScriptOrg = ',trim(ExeCommandStartScriptOrg)
            !    print*,'FitFktInputOrg = ',trim(FitFktInputOrg)
            !    print*,'MaxNumberParameter = ',MaxNumberParameter
            !    print*,'NumFileOrg = ',NumFileOrg
            !    print*,'MaxColXOrg = ',MaxColXOrg
            !    print*,'MaxColYOrg = ',MaxColYOrg
            !    Do i=1,NumFileOrg
            !        print*,'    Experimental file: i = ',i
            !        print*,'    lengthexpdataorg(i) = ',lengthexpdataorg(i)
            !        print*,'    NumberYColumnsOrg(i) = ',NumberYColumnsOrg(i)
            !        print*,'    expdataxorg(i,1:5,1) = ',expdataxorg(i,1:5,1)
            !        print*,'    expdatayorg(i,1:5,1) = ',expdatayorg(i,1:5,1)
            !        print*,'    expdataerrororg(i,1:5,1) = ',expdataerrororg(i,1:5,1)
            !    end Do
            !    print*,'chilm = ',chilm
            !    print*,'numiter = ',numiter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'currentpathorg = ',trim(adjustl(currentpathorg))
            !    print*,'len(FitParameterNameOrg) = ',len(FitParameterNameOrg
            !    print*,'FitParameterNameOrg = ',FitParameterNameOrg
            !    print*,"parametersetorg(1,:) = ",parametersetorg(1,:)
            !    print*,"parametersetorg(2,:) = ",parametersetorg(2,:)
            !    print*,"parametersetorg(3,:) = ",parametersetorg(3,:)
            !    print*,"parametersetorg(4,:) = ",parametersetorg(4,:)
            !    print*,"RenormalizedChi2 = ",RenormalizedChi2
            !    return


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< set temp-directory
            TempDirectory = " "
            CALL GetEnv('MAGIXTempDirectory', TempDirectory)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< modify name of log file
            i = index(fitlog, "/", back = .true.)
            j = index(fitlog, ".", back = .true.)
            Number1 = "          "
            write(Number1,'(I10)') PSOCounter
            if (j > i) then
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__PSO__call_" // trim(adjustl(Number1)) // trim(adjustl(fitlog(j:)))
                else
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__PSO" // trim(adjustl(fitlog(j:)))
                endif
            else
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog)) // "__PSO__call_" // trim(adjustl(Number1)) // ".log"
                else
                    fitlog = trim(adjustl(fitlog)) // "__PSO.log"
                endif
            endif

            ! Debug:
            !   print*,'>',trim(adjustl(fitlog)),'<'


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open log file and write header
            open(logchannel,file = trim(fitlog), status='replace')
            write(logchannel,'(" ")')
            write(logchannel,'("log-file for Particle-Swarm algorithm:")')
            write(logchannel,'(38("-"))')
            write(logchannel,'(" ")')


            !< get current local time and date and write to log-file
            call date_and_time(DATE, TIME, ZONE, VALUES)
            write(logchannel,'(" ")')
            write(logchannel,'("algorithm starts at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                                  TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open log-file for the parameter and write header
            fitlogparam = trim(fitlog) // ".param"
            open(paramchannel,file = trim(fitlogparam), status='replace')
            write(paramchannel,'(" ")')
            write(paramchannel,'("log-file containing the actual values of the parameters used in the Particle-Swarm algorithm:")')
            write(paramchannel,'(93("-"))')
            write(paramchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open file containing the values of chi**2 and the corresponding values of the parameters
            WriteChi2Flag = .true.
            NumberLinesChi2 = 0
            fitlogChi2 = trim(fitlog) // ".chi2"
            open(Chi2Channel,file = trim(fitlogChi2), status='replace')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate if necessary and print error message if necessay
            if (allocated(expdatax)) then
                deallocate(expdatax, expdatay, expdatae, lengthexpdata, NumberXColumns, NumberYColumns, FirstPointExpData, LastPointExpData, &
                           NumberRanges, MinRange, MaxRange, ExpData_reversed_flag, stat = deallocstatus)
                if (deallocstatus /= 0) then                                                    !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine MainAlg:")')
                    write(logchannel,'(2x,"Can not deallocate variables expdatax etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine MainAlg:")'
                    print '(2x,"Can not deallocate variables expdatax etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for variables, clear content of the variables and print error message if necessay
            allocate(expdatax(NumberExpFiles, MaxLength, MaxColX), expdatay(NumberExpFiles, MaxLength, MaxColY), &
                     expdatae(NumberExpFiles, MaxLength, MaxColY), lengthexpdata(NumberExpFiles), NumberXColumns(NumberExpFiles), &
                     NumberYColumns(NumberExpFiles), FirstPointExpData(NumberExpFiles, MaxColX), LastPointExpData(NumberExpFiles, MaxColX), &
                     NumberRanges(NumberExpFiles), MinRange(NumberExpFiles, MaxRangeNumber, MaxColX), MaxRange(NumberExpFiles, MaxRangeNumber, MaxColX), &
                     ExpData_reversed_flag(NumberExpFiles), stat = allocstatus)
            if (allocstatus /= 0) then                                                      !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not allocate variables expdatax etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not allocate variables expdatax,expdatay etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            MaxRangeNumber = MaxRangeNumber - 1                                             !< get real value
            expdatax = 0.d0
            expdatay = 0.d0
            expdatae = 0.d0
            lengthexpdata = 0
            NumberXColumns = 0
            NumberYColumns = 0
            FirstPointExpData = 1.d99
            LastPointExpData = -1.d99
            NumberRanges = 0
            MinRange = 0.d0
            MaxRange = 0.d0
            ExpData_reversed_flag = .false.


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy input variables to module variabels
            expdatax(:,:,:) = expdataxorg(:,:,:)
            expdatay(:,:,:) = expdatayorg(:,:,:)
            expdatae(:,:,:) = expdataerrororg(:,:,:)
            lengthexpdata = lengthexpdataorg
            NumberXColumns = NumberXColumnsOrg
            NumberYColumns = NumberYColumnsOrg
            CalculationMethod = CalculationMethodOrg
            PathStartScript = PathStartScriptOrg
            ExeCommandStartScript = ExeCommandStartScriptOrg
            NumberRanges = NumberRangesOrg
            MinRange = MinRangeOrg
            MaxRange = MaxRangeOrg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine first and last point of each exp. data file
            Do i = 1, NumberExpFiles
                Do j = 1, lengthexpdata(i)
                    ii = 0
                    jj = 0
                    Do k = 1,NumberXColumns(i)
                        if (expdatax(i, j, k) <= FirstPointExpData(i, k)) then
                            ii = ii + 1
                        endif
                        if (expdatax(i, j, k) >= LastPointExpData(i, k)) then
                            jj = jj + 1
                        endif
                    end Do
                    if (ii == NumberXColumns(i)) then
                        FirstPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                    if (jj == NumberXColumns(i)) then
                        LastPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                end Do


                !< check output file starts with the highest x-column value interchange FirstPointOutputFile and LastPointOutputFile
                ii = 0
                Do k = 1, NumberXColumns(i)
                    if (expdatax(i, 1, k) >= expdatax(i, lengthexpdata(i), k)) then
                        ii = ii + 1
                    endif
                end Do
                if (ii == NumberXColumns(i)) then
                    ExpData_reversed_flag(i) = .true.
                endif

                ! Debug:
                ! print*,' '
                ! print*,'File = ',i
                ! print*,'FirstPointExpData(i, 1:NumberXColumns(i)) = ', FirstPointExpData(i, 1:NumberXColumns(i))
                ! print*,'LastPointExpData(i, 1:NumberXColumns(i)) = ', LastPointExpData(i, 1:NumberXColumns(i))
                ! print*,'##########################################'
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy number of parameters required for the model function to working variable
            nfit = sum(parametersetorg(2, :))                                               !< number of parameters which should be optimized
            ma = parameternumber
            Gradientflag = .false.                                                          !< we do not need the gradient of the function here
            UseCalculationReduction = .true.                                                !< activate calculation reduction
            CurrentNumberLinesCalcReduction = 0                                             !< reset number of lines


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< Initial population
            n_particles = (3 * nfit + 1)
            if (NumberParticles > n_particles) then
                n_particles = NumberParticles
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate/allocate memory for some working variables, clear contents of the variables and print error message if necessary
            if (allocated(paramset)) then
                deallocate(paramset, FitParameterName, FitParameterValue, ia, a, OptimizedParameterLowLimit, OptimizedParameterUpperLimit, &
                           AtOnceFunction, ModelFunction, chisqValues, BestSitesParamSet, BestSitesModelValues, BestSitesChi2Values, ConverterInfit, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    write(logchannel,*)
                    write(logchannel,'(11x,"Error in subroutine MainAlg:")')
                    write(logchannel,'(13x,"Can not deallocate variables paramset etc.")')
                    write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'(13x,"deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'(13x,"Program aborted!")')

                    print '(" ")'
                    print '(11x,"Error in subroutine MainAlg:")'
                    print '(13x,"Can not deallocate variables paramset etc.")'
                    print '(13x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '(13x,"deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif
            allocate(paramset(4, parameternumber), OptimizedParameterLowLimit(nfit), OptimizedParameterUpperLimit(nfit), ConverterInfit(nfit), &
                     ia(parameternumber), FitParameterName(parameternumber), FitParameterValue(parameternumber), a(parameternumber), &
                     ModelFunction(1, NumberExpFiles, MaxColY, MaxLength), chisqValues(0:0), &
                     AtOnceFunction(0:ParallelizationFlag - 1, NumberExpFiles, MaxColY, MaxLength), &
                     BestSitesParamSet(ParamSetCounter, nfit + 1), BestSitesModelValues(ParamSetCounter, NumberExpFiles, MaxLength, MaxColY), &
                     BestSitesChi2Values(ParamSetCounter, NumberExpFiles, MaxLength, MaxColY), stat = allocstatus)
            if (allocstatus /= 0) then
                write(logchannel,'(" ")')
                write(logchannel,'(11x,"Error in subroutine MainAlg:")')
                write(logchannel,'(13x,"Can not allocate variables paramset etc.")')
                write(logchannel,'(13x,"Please PSOVariablesclose all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'(13x,"allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'(13x,"Program aborted!")')

                print '(" ")'
                print '(11x,"Error in subroutine MainAlg:")'
                print '(13x,"Can not allocate variables paramset etc.")'
                print '(13x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '(13x,"allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            ia = .false.
            ConverterInfit = 0
            FitParameterName = FitParameterNameLocal
            FitParameterValue = FitParameterValueLocal
            OptimizedParameterLowLimit = 0.d0
            OptimizedParameterUpperLimit = 0.d0
            ModelFunction = 0.d0
            AtOnceFunction = 0.d0
            chisqValues = 0.d0
            BestSitesParamSet = 0.d0
            BestSitesParamSet(:, 1) = 1.d99
            BestSitesModelValues = 0.d0
            BestSitesChi2Values = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy parameter set to module variable
            paramset = parametersetorg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write registration mask for the fit model
            if (printflag) print '(9x, "Writing registration mask for the fit model .. ", $)'
            call RegistrationMask(ok)
            if (ok /= 0) then
                return
            endif
            if (printflag) print '("done!")'


            !< define ia variable
            ConverterInfit = 0
            ia = .false.
            k = 0
            Do i = 1, parameternumber
                if (paramset(2, i) == 1) then
                    k = k + 1
                    ia(i) = .true.
                    ConverterInfit(k) = i
                endif
            end Do
            NumberFreeParameter = int(sum(paramset(2, :)))                                  !< determine number of free parameter


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize model program
            call ModelInit


            !< write on the screen what you Do ..
            if (printflag) then
                print '(" ")'
                print '(" ")'
                write(Number1,'(I10)') JobID                                                !< write JobID to string
                print '(9x,"Temporary files are stored in: ",A)', trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(Number1)) // "/"
                print '(" ")'
                print '(" ")'
                print '(9x,"Start Particle-Swarm algorithm (", A, " version) ..")', trim(adjustl(ParallelizationMethod))
                print '(" ")'
            endif
            write(logchannel,'(11x,"Start Particle-Swarm algorithm (", A, " version) ..")') trim(adjustl(ParallelizationMethod))
            write(logchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< lower limit of chi^2 for stopping condition normalized to the number of calculation points
            chi2lim = 0.d0
            if (RenormalizedChi2) then
                Do i = 1, NumberExpFiles
                    chi2lim = chi2lim + (NumberYColumns(i) * lengthexpdata(i) - parameternumber) * dabs(chilm)
                end Do
                chi2lim = dabs(chi2lim)
                write(logchannel,'(11x,"Renormalized limit for chi^2 = ", ES25.15)') chi2lim
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Renormalized limit for chi^2 = ",ES25.15)', chi2lim
                    print '(" ")'
                endif
            else
                chi2lim = dabs(chilm)
                write(logchannel,'(11x,"Limit for chi^2 = ", ES25.15)') chi2lim
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Limit for chi^2 = ",ES25.15)', chi2lim
                    print '(" ")'
                endif
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< start iteration ..
            Gradientflag = .false.                                                          !< we do not need the gradient of the function here
            flag = 0                                                                        !< set flag variable to 0
            actualiteration = 0                                                             !< set working variable containing the current iteration
                                                                                            !< number to 0
            a = paramset(1,1:parameternumber)                                               !< copy the values of the parameters to another array


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< call subroutine to call pso algorithm
            call CallPSO(nfit, ma, n_particles, ParamSetCounter, FinalParameterSet, MaxColX, NumberExpFiles, MaxLength, MaxColY, PlotIteration, &
                         PlotType, xAxisLabel, yAxisLabel, zAxisLabel, fitlog)
            ParamSetCounter = QualityLimit
            FitFunctionOut = 0.d0
            Chi2Values = 0.d0
            Do i  = 1, QualityLimit


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< determine model function values if plot for each iteration option is not selected
                if (.not. PlotIterationFlag .and. DontStoreModelFuncValuesFlag) then
                    chi2ValuesVector = 0.d0
                    call ModelCalcChiFunctionGeneral(parameternumber, ia, paramset(1, :), 1, nfit, NumFileOrg, MaxColY, MaxLength, &
                                                     BestSitesParamSet(i, 2:), chi2ValuesVector)
                endif
                Do j = 1, NumFileOrg
                    Do k = 1, MaxLength
                        FitFunctionOut(i, j, k, 1:MaxColY) = BestSitesModelValues(i, j, k, 1:MaxColY)
                        Chi2Values(i, j, k, 1:MaxColY) = BestSitesChi2Values(i, j, k, 1:MaxColY)
                    end Do
                end Do
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< close log-files
            if (SortFortranNum == 1) call SortChi2File(nfit, parameternumber, ia, a)
            close(Chi2Channel)
            close(paramchannel)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< write final parameter sets for each site to final input files


            !< define base directory, i.e. path where final input file is written to
            k = index(fitlog, '/', back = .true.)
            if (k == 0) then
                BaseDir = ""
            else
                BaseDir = trim(adjustl(fitlog(:k)))
            endif


            !< define file name extension for number of algorithm call
            write(Number1,'(I10)') PSOCounter
            if (NumberOfFitAlgorithms > 1) then
                FuncCallExt = "__PSO__call_" //trim(adjustl(Number1))
            else
                FuncCallExt = "__PSO"
            endif

            ! Debug:
            !    print*,"NumberOfFitAlgorithms = ", NumberOfFitAlgorithms
            !    print*,"FuncCallExt = ", trim(adjustl(FuncCallExt))


            !< write files
            Do i  = 1, QualityLimit                                                         !< loop over all sites


                !< create site extension for file name
                write(helpString,'(I5)') i
                SiteExt = "__site_" // trim(adjustl(helpString)) // ".out"


                !< write parameter sets to file
                Do NumInputFiles = 1, NumberInputFiles                                      !< loop over all input files
                    NumExt = trim(adjustl(FitFktInput(NumInputFiles)))
                    k = index(NumExt, '/', back = .true.)
                    if (k > 0) then                                                         !< we only need the file name
                        NumExt = trim(adjustl(NumExt(k:)))
                    endif
                    j = index(NumExt, '.', back = .true.)
                    if (j > 1) then                                                         !< remove file name extension
                        NumExt = trim(adjustl(NumExt(:j - 1)))
                    endif

                    ! Debug:
                    ! print*,"Site = ", i
                    ! print*,"Nr. Final input file = ", NumInputFiles
                    ! print*,"Final input file = ", trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt)) &
                    !                               // trim(adjustl(SiteExt)) // ".input"


                    !< write parameter sets to file
                    open(235, file = trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt)) // trim(adjustl(SiteExt)) // ".input")
                    call WriteParameter(235, .false., MaxColX, expdataxorg, parameternumber, FinalParameterSet(i, :), NumInputFiles)
                    close(235)
                end Do
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write end of log-file and print message to screen
            call date_and_time(DATE, TIME, ZONE, VALUES)
            write(logchannel,*)
            write(logchannel,'("algorithm ends at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                              TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel,'(" ")')
            write(logchannel,'(150("-"))')

            ! Debug:
            !    print*,'lengthexpdata = ',lengthexpdata
            !    print*,'MaxColX = ',MaxColX
            !    print*,'MaxColY = ',MaxColY
            !    print*,'expdataxorg(1:5,1) = ',expdataxorg(1:5,1)
            !    print*,'expdatayorg(1:5,1) = ',expdatayorg(1:5,1)
            !    print*,'expdataerrororg(1:5,1) = ',expdataerrororg(1:5,1)
            !    print*,'chilm = ',chilm
            !    print*,'numiter = ',numiter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'model = ',trim(model)
            !    print*,'currentpathorg = ',currentpathorg
            !    print*,'parametersetorg(1,:) = ',parametersetorg(1,:)
            !    print*,'parametersetorg(2,:) = ',parametersetorg(2,:)
            !    print*,'parametersetorg(3,:) = ',parametersetorg(3,:)
            !    print*,'parametersetorg(4,:) = ',parametersetorg(4,:)


            !< send plot program the kill signal
            if (PlotIteration == 0) then
                call system("kill -9 " // trim(adjustl(PlotPID)) // " 1>/dev/null 2>&1")
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< free memory of general pso variables
            if (allocated(ia)) deallocate(ia, stat = deallocstatus)
            if (allocated(a)) deallocate(a, stat = deallocstatus)
            if (allocated(OptimizedParameterLowLimit)) deallocate(OptimizedParameterLowLimit, stat = deallocstatus)
            if (allocated(OptimizedParameterUpperLimit)) deallocate(OptimizedParameterUpperLimit, stat = deallocstatus)


            !< free memory of model variables
            call ModelParamFree(deallocstatus)
            if (deallocstatus /= 0) then
                write(logchannel,*)
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not deallocate expdatax etc.")')
                write(logchannel,*)
                write(logchannel,'("deallocstatus = ", I4)') deallocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not deallocate variables expdatax etc.")'
                print '(" ")'
                print '("deallocstatus = ", I4)', deallocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< close log file
            close(logchannel)
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(9x,"Finished Particle-Swarm algorithm!")'
                print '(" ")'
                print '(" ")'
            endif


            !< we're done
            return
        end subroutine MainAlg
end Module Algorithm
!*********************************************************************************************************************************************************


