!*********************************************************************************************************************************************************
!> Module: NestedSamplingVariables
!>
!>
!>  This module contains the subroutines for Nested Sampling
!>  Copyright (C) 2009 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - Module NestedSamplingVariables:           Module contains global variables and subroutines for Nested-Sampling algorithm
!>      - subroutine loglike_nest:                  function for optimization
!>      - subroutine explore:                  	    evolve object within likelihood constraint
!>      - subroutine plus:                          logarithmic addition log(exp(x)+exp(y))
!>      - subroutine ndim_nest:                     implementation of John Skilling's Nested Sampling algorithm for computing the normalizing constant
!>                                                  of a probability distribution
!>      - subroutine CallNestedSampling:            calls nested sampling subroutine
!>      - Module Algorithm:                         module contains the main subroutine used to start the different versions of the PSO algorithm
!>      - subroutine MainAlg:                       main subroutine which starts the PSO algorithm
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  I. Bernst     2010-05-03  Initial version (python)
!>  T. Moeller    2010-02-07  translation to fortran, parallelization using OpenMP
!>  T. Moeller    2014-08-20  myXCLASS (model) optimized version
!>  T. Moeller    2014-08-31  myXCLASS (model) optimized version for GPU
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: NestedSamplingVariables
!>
!>         Module contains global variables and subroutines for Nested-Sampling algorithm
!>
!>
!> \author Thomas Moeller
!>
!> \date 2010-06-09
!>
Module NestedSamplingVariables

    use Model

    implicit none
    integer :: MaxNSIter                                                                    !< current and max. iteration number
    real*8 :: DBL_MAX                                                                       !< max. value of double precision
    real*8 :: chilim                                                                        !< lower limit of chi**2 normalized to the total number of
                                                                                            !< all data points
    real*8 :: minmaxTrans                                                                   !< transformation value for min to max search
    real*8 :: ObjectlogL                                                                    !< class Object: logLikelihood = ln Prob(data | position)
    real*8 :: ObjectlogWt                                                                   !< class Object: log(Weight), adding to SUM(Wt) = Evidence Z
    real*8, allocatable, dimension(:) :: ObjectMinval                                       !< class Object: lower limits of parametric space
    real*8, allocatable, dimension(:) :: ObjectMaxval                                       !< class Object: upper limits of parametric space
    real*8, allocatable, dimension(:) :: ObjectPrior                                        !< class Object: a prior volume for ObjectPosition
    real*8, allocatable, dimension(:) :: ObjectPosition                                     !< class Object: a set of parameters values (multidimensional)
    real*8, allocatable, dimension(:) :: a                                                  !< values of all parameters incl. of non-optimized parameters
    real*8, allocatable, dimension(:) :: OptParamLowLimit                                   !< lower limit for each free parameter
    real*8, allocatable, dimension(:) :: OptParamUpperLimit                                 !< upper limit for each free parameter

    logical, allocatable, dimension(:) :: ia                                                !< array indicating the parameter which should be optimized

    contains


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !>
        !> class Object:
        !>
        !>------------------------------------------------------------------------------------------------------------------------------------------------


        !*************************************************************************************************************************************************
        !> subroutine: loglike_nest
        !>
        !> input variables:     nfit:               number of parameters which should be optimized
        !>                      ma:                 total number of all parameter
        !>                      a:                  values of all parameter
        !>                      colx:               number of columns belonging to the x-column
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>
        !> output variables:    ObjMinval:          lower limits of each optimized parameter
        !>                      ObjMaxval:          upper limits of each optimized parameter
        !>                      ObjPrior:           prior
        !>                      ObjPosition:        values of optimized parameters
        !>                      ObjlogL:            likelihood
        !>
        !> \author Irina Bernst, translation (python to fortran) by Thomas Moeller
        !>
        !> \date 02.07.2010
        !>
        subroutine loglike_nest(ObjMinval, ObjMaxval, ObjPrior, ObjPosition, ObjlogL, nfit)
    	    !< Function for optimization

            use Variables
            use FunctionCalling

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: i                                                                    !< loop variable
            real*8 :: ObjlogL                                                               !< likelihood
            real*8, dimension(nfit) :: ObjPrior                                             !< prior
            real*8, dimension(nfit) :: ObjPosition                                          !< values of optimized parameters
            real*8, dimension(nfit) :: ObjMinval                                            !< lower limits of each optimized parameter
            real*8, dimension(nfit) :: ObjMaxval                                            !< upper limits of each optimized parameter


            !< copy lower and upper limits
            ObjMinval = ObjectMinval
            ObjMaxval = ObjectMaxval


            !< define prior, position
            ObjlogL = 0.d0
            ObjPrior = 0.d0
            ObjPosition = 0.d0
            Do i = 1, nfit


                !< define prior
                ObjPrior(i) = RandomWithLimits(0.d0, 1.d0)


                !< define position and check if new parameter is within limits
                ObjPosition(i) = ObjMinval(i) + (ObjectMaxval(i) - ObjMinval(i)) * ObjPrior(i)
                if (ObjPosition(i) < OptParamLowLimit(i)) then
                    ObjPosition(i) = OptParamLowLimit(i)
                elseif (ObjPosition(i) > OptParamUpperLimit(i)) then
                    ObjPosition(i) = OptParamUpperLimit(i)
                endif
            end Do
            return
        end subroutine loglike_nest


        !*************************************************************************************************************************************************
        !> subroutine: explore
        !>
        !> input variables:     retMinval:          lower limits of each optimized parameter
        !>                      retMaxval:          upper limits of each optimized parameter
        !>                      retPrior:           prior
        !>                      retPosition:        values of optimized parameters
        !>                      retlogL:            likelihood
        !>                      nfit:               number of parameters which should be optimized
        !>                      ma:                 total number of all parameter
        !>                      a:                  values of all parameter
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>                      logLstar:           d
        !>
        !> output variables:    retMinval:          lower limits of each optimized parameter
        !>                      retMaxval:          upper limits of each optimized parameter
        !>                      retPrior:           prior
        !>                      retPosition:        values of optimized parameters
        !>                      retlogL:            likelihood
        !>
        !> \author Irina Bernst, translation (python to fortran) by Thomas Moeller
        !>
        !> \date 02.07.2010
        !>
        subroutine explore(nfit, retMinval, retMaxval, retPrior, retPosition, retlogL, logLstar, ma, NumFile, MaxL, MaxCol)
    	    !< Evolve object within likelihood constraint

            use Variables
            use FunctionCalling

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: i, n                                                                 !< loop variable
            integer :: ma                                                                   !< total number of all parameter
            integer :: NumFile                                                              !< number of input files
            integer :: MaxL                                                                 !< max. total length
            integer :: MaxCol                                                               !< max. number of columns
            real*8 :: step                                                                  !< Initial guess suitable step-size in (0,1)
            real*8 :: accept                                                                !< MCMC acceptances
            real*8 :: reject                                                                !< MCMC rejections
            real*8 :: logLstar                                                              !< d
            real*8 :: retlogL                                                               !< likelihood
            real*8 :: TrylogL                                                               !< (for exploration): likelihood
            real*8, dimension(nfit) :: TryPosition                                          !< current position vector
            real*8, dimension(nfit) :: TryMinval                                            !< (for exploration): lower limits of each optimized parameter
            real*8, dimension(nfit) :: TryMaxval                                            !< (for exploration): upper limits of each optimized parameter
            real*8, dimension(nfit) :: TryPrior                                             !< (for exploration): prior
            real*8, dimension(nfit) :: retPosition                                          !< values of optimized parameters
            real*8, dimension(nfit) :: retMinval                                            !< lower limits of each optimized parameter
            real*8, dimension(nfit) :: retMaxval                                            !< upper limits of each optimized parameter
            real*8, dimension(nfit) :: retPrior                                             !< prior
            real*8, dimension(1, nfit) :: paramVec                                          !< copy of param. vector
            real*8, dimension(1) :: chi2ValuesVector                                        !< vector for only one chi2 value
            character(len=10) :: Number1                                                    !< working variable for number to string conversion


            step = 0.1d0
            accept = 0.d0
            reject = 0.d0
            Do n = 1, 10                                                                    !< pre-judged number of steps
                if (printflag) then
                    write(Number1, '(I10)') n
                    print '(A,11x,"Determine step (",A,"/10) ..                   ",A,$)', char(13), trim(adjustl(Number1)), char(13)
                endif


                !< Trial object
                TryMaxval(:) = ObjectMaxval(:)
                TryMinval(:) = ObjectMinval(:)


                !< define prior, position
                TryPrior = 0.d0
                TryPosition = 0.d0
                Do i = 1, nfit


                    !< define prior
                    TryPrior(i) = retPrior(i) + step * (2.d0 * RandomWithLimits(0.d0, 1.d0) - 1.d0)
                    TryPrior(i) = (TryPrior(i) - floor(TryPrior(i)))


                    !< define position
                    TryPosition(i) = retMinval(i) + (retMaxval(i) - retMinval(i)) * TryPrior(i)


                    !< check, if new parameter is within limits
                    if (TryPosition(i) < OptParamLowLimit(i)) then
                        TryPosition(i) = OptParamLowLimit(i)
                    elseif (TryPosition(i) > OptParamUpperLimit(i)) then
                        TryPosition(i) = OptParamUpperLimit(i)
                    endif
                end Do


                !< check previos calculated parameter sets
                if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then !< if calculation reduction is chosen, read chi**2 file
                    call CheckCalculatedParameterSets(nfit, 1)
                endif


                !< determine Likelihood constraint for each object
                chi2ValuesVector = 0.d0
                paramVec(1, :) = TryPosition(:)
                call ModelCalcChiFunctionGeneral(ma, ia, paramset(1, :), 1, nfit, NumFile, MaxL, MaxCol, paramVec, chi2ValuesVector)
                TrylogL = (minmaxTrans - chi2ValuesVector(1))


                !< update CurrentNumberLinesCalcReduction variable
                CurrentNumberLinesCalcReduction = NumberLinesChi2


                !< Accept if and only if within hard likelihood constraint
                if (TrylogL > logLstar) then
                    retMaxval = TryMaxval
                    retMinval = TryMinval
                    retPrior = TryPrior
                    retPosition = TryPosition
                    retlogL = TrylogL

                    accept = accept + 1
                else
                    reject = reject + 1
                endif


                !< Refine step-size to let acceptance ratio converge around 50%
                if (accept > reject) then
                    step = step * dexp(1.d0 / accept)
                endif
                if (accept < reject) then
                    step = step / dexp(1.d0 / reject)
                endif
            end Do


            !< clear screen massage
            if (printflag) then
                print '(A,80(" "),A,$)',char(13),char(13)
            endif

            return
        end subroutine explore


        !*************************************************************************************************************************************************
        !> subroutine: plus
        !>
        !> input variables:     x:                  arg1
        !>                      y:                  arg2
        !>
        !> output variables:    value:              return value
        !>
        !> \author Irina Bernst, translation (python to fortran) by Thomas Moeller
        !>
        !> \date 02.07.2010
        !>
        subroutine plus(value, x, y)
	        !< logarithmic addition log(exp(x)+exp(y))

            implicit none
            real*8 :: value                                                                 !< return value
            real*8 :: x                                                                     !< arg1
            real*8 :: y                                                                     !< arg2

            if (x > y) then
               value = x + dlog(1.d0 + dexp(y - x))
            else
               value = y + dlog(1.d0 + dexp(x - y))
            endif

            ! Debug:
            !    print*,' '
            !    print*,'x = ',x
            !    print*,'y = ',y
            !    print*,'x-y = ',x-y
            !    print*,'y-x = ',y-x
            !    print*,'dlog(1.d0 + dexp(y - x)) = ',dlog(1.d0 + dexp(y - x))
            !    print*,'dlog(1.d0 + dexp(x - y)) = ',dlog(1.d0 + dexp(x - y))
            !    print*,'value = ',value
            !    print*,'########################################################'

            return
        end subroutine plus


        !*************************************************************************************************************************************************
        !> subroutine: ndim_nest
        !>
        !> input variables:     n:                  number of objects
        !>                      nfit:               number of free parameters
        !>                      ma:                 total number of parameters
        !>                      colx:               number of columns in experimental x data
        !>                      NumFile:            Number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      SampleslogL:        ?
        !>                      SampleslogWt:       ?
        !>                      SamplesPosition:    ?
        !>                      SamplesPrior:       ?
        !>                      SamplesMinval:      ?
        !>                      SamplesMaxval:      ?
        !>                      ni:                 number of iterations
        !>                      logZ:               ln(Evidence Z, initially 0)
        !>                      sdev_logZ:          standers deviation of log(Z)
        !>                      H:                  information
        !>                      sdev_H:             ?
        !>                      PlotIteration:      flag for plotting the model function for each step
        !>                      PlotType  :         get type of plot
        !>                      xAxisLabel:         label of the x-axis (for plot)
        !>                      yAxisLabel:         label of the y-axis (for plot)
        !>                      zAxisLabel:         label of the z-axis (for plot)
        !>                      fitlog:             path for log files
        !>
        !> output variables:    value:              return value
        !>
        !> \author Irina Bernst, translation (python to fortran) by Thomas Moeller
        !>
        !> \date 02.07.2010
        !>
        subroutine ndim_nest(n, nfit, ma, colx, NumFile, MaxL, MaxCol, SampleslogL, SampleslogWt, SamplesPosition, SamplesPrior, SamplesMinval, &
                            SamplesMaxval, ni, logZ, sdev_logZ, H, sdev_H, PlotIteration, PlotType, xAxisLabel, yAxisLabel, zAxisLabel, fitlog)
	        !<
	        !< This is an implementation of John Skilling's Nested Sampling algorithm for computing the normalizing constant of a probability
	        !< distribution (usually the posterior in Bayesian inference).
	        !<
	        !< The return value is a dictionary with the following entries:
	        !< "samples"
	        !< "num_iterations"
	        !< "logZ"
	        !< "logZ_sdev"
	        !< "info_nats"
	        !< "info_sdev"

            use Variables
            use FunctionCalling

            implicit none
            integer :: n                                                                    !< number of objects
            integer :: i, j, k                                                              !< loop variable
            integer :: nest                                                                 !< Nested sampling iteration count
            integer :: ni                                                                   !< number of iterations
            integer :: worst                                                                !< Worst object
            integer :: copy                                                                 !< Duplicated object
            integer :: SamplesCounter                                                       !< counter for Samples array
            integer :: nfit                                                                 !< number of free parameters
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of x-columns
            integer :: NumFile                                                              !< max number of input file
            integer :: MaxL                                                                 !< max number of length
            integer :: MaxCol                                                               !< max number of columns
            integer :: NumInputFiles                                                        !< needed for loop over input files
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            integer :: PlotIteration                                                        !< flag for plotting the model function for each step
            integer :: PlotType                                                             !< get type of plot
            real*8 :: H                                                                     !< information
            real*8 :: logwidth                                                              !< ln(width in prior mass)
            real*8 :: logLstar                                                              !< ln(Likelihood constraint)
            real*8 :: logZ                                                                  !< ln(Evidence Z, initially 0)
            real*8 :: logZnew                                                               !< Updated logZ
            real*8 :: sdev_H                                                                !< standers deviation of H
            real*8 :: sdev_logZ                                                             !< standers deviation of log(Z)
            real*8, dimension(colx) :: posdatexp                                            !< working variable
            real*8, dimension(ma) :: currentparmval                                         !< working variable
            real*8, dimension(n) :: ObjlogL                                                 !< (worst object): logL
            real*8, dimension(n) :: ObjlogWt                                                !< (worst object): logWt (weighted logL ?)
            real*8, dimension(n,nfit) :: ObjPosition                                        !< (worst object): Position
            real*8, dimension(n,nfit) :: ObjPrior                                           !< (worst object): Prior
            real*8, dimension(n,nfit) :: ObjMinval                                          !< (worst object): Minval
            real*8, dimension(n,nfit) :: ObjMaxval                                          !< (worst object): Maxval
            real*8, dimension(ni) :: SampleslogL                                            !< (Posterior Samples): logL
            real*8, dimension(ni) :: SampleslogWt                                           !< (Posterior Samples): logWt (weighted logL ?)
            real*8, dimension(ni,nfit) :: SamplesPosition                                   !< (Posterior Samples): Position
            real*8, dimension(ni,nfit) :: SamplesPrior                                      !< (Posterior Samples): Prior
            real*8, dimension(ni,nfit) :: SamplesMinval                                     !< (Posterior Samples): Minval
            real*8, dimension(ni,nfit) :: SamplesMaxval                                     !< (Posterior Samples): Maxval
            real*8, dimension(n) :: chi2ValuesVector                                        !< chi2 value vector
            character(len=100) :: HelpString                                                !< help string for writing parameter in correct format
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=5196) :: ListParamFormated                                        !< list of all formated free parameters
            character(len=8192) :: fitlog                                                   !< path for log files
            character(len=10) :: Number1, Number2                                           !< help strings for converting number to string
            character(len=25) :: LongNumber1, LongNumber2                                   !< working variables
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical :: InitPlotFlag                                                         !< flag for saving plot to file


            !< print what you do
            if (printflag) then
                print '(11x,"Iteration:",20x,"chi^2:",5x,"Parameter:")'
                print '(11x,"Initialize model function ..",20(" "),A1,$ )',char(13)
            endif
            write(logchannel,'(11x,"Iteration:",20x,"chi^2:",5x,"Parameter:")')


            !< initalization
            H = 0.d0                                                                        !< Information, initially 0
            logZ = -DBL_MAX                                                                 !< ln(Evidence Z, initially 0)


            !< Set prior objects
            ObjPosition = 0.d0
            Do i = 1, n
                call loglike_nest(ObjMinval(i,:), ObjMaxval(i,:), ObjPrior(i,:), ObjPosition(i,:), ObjlogL(i), nfit)
            end Do
            if (printflag) then
                write(Number2, '(I10)') n                                                   !< convert number n to string for output
                print '(A,120(" "),A,$)',char(13),char(13)
            endif


            !< if calculation reduction is chosen, read chi**2 file
            if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then
                call CheckCalculatedParameterSets(nfit, 1)
            endif


            !< print what you do ..
            if (printflag) then
                print '(A,11x,"Initialize object ..                    ",A,$)', char(13), char(13)
            endif


            !< determine Likelihood constraint for each object
            chi2ValuesVector = 0.d0
            call ModelCalcChiFunctionGeneral(ma, ia, paramset(1, :), n, nfit, NumFile, MaxL, MaxCol, ObjPosition, chi2ValuesVector)
            Do i = 1, n
                ObjlogL(i) = (minmaxTrans - chi2ValuesVector(i))
            end Do


            !< update CurrentNumberLinesCalcReduction variable
            CurrentNumberLinesCalcReduction = NumberLinesChi2


            !< clear screen massage
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
            endif


            !< Outermost interval of prior mass
            logwidth = dlog(1.d0 - dexp(-1.d0 / n))


            !< NESTED SAMPLING LOOP_______________________________________________________________________________________________________________________
            SamplesCounter = 0
            write(Number2, '(I10)') MaxNSIter
            Do nest = 1, MaxNSIter
                if (printflag) then
                    write(Number1, '(I10)') nest
                    print '(A,120(" "),A,$)',char(13),char(13)
                    print '(A,11x,"Determine iteration step (",A,"/",A,") ..                    ",A,$)',char(13), trim(adjustl(Number1)), &
                                                                                                       trim(adjustl(Number2)), char(13)
                endif


                !< Worst object in collection, with Weight = width * Likelihood
                worst = 1
                Do i = 2, n
                    if (ObjlogL(i) < ObjlogL(worst)) then
                       worst = i
                    endif
                end Do
                ObjlogWt(worst) = logwidth + ObjlogL(worst)


                !< Update Evidence Z and Information H
                call plus(logZnew, logZ, ObjlogWt(worst))
                H = dexp(ObjlogWt(worst) - logZnew) * ObjlogL(worst) + dexp(logZ - logZnew) * (H + logZ) - logZnew
                logZ = logZnew


                !< Posterior Samples (optional)
                SamplesCounter = SamplesCounter + 1
                SampleslogL(SamplesCounter) = ObjlogL(worst)
                SampleslogWt(SamplesCounter) = ObjlogWt(worst)
                SamplesPosition(SamplesCounter,:) = ObjPosition(worst,:)
                SamplesPrior(SamplesCounter,:) = ObjPrior(worst,:)
                SamplesMinval(SamplesCounter,:) = ObjMinval(worst,:)
                SamplesMaxval(SamplesCounter,:) = ObjMaxval(worst,:)


                !< Posterior Samples (optional)
                ! best = 1
                ! Do i = 2,n
                !     if (ObjlogL(i) > ObjlogL(best)) then
                !        best = i
                !     endif
                ! end Do
                ! SamplesCounter = SamplesCounter + 1
                ! SampleslogL(SamplesCounter) = ObjlogL(best)
                ! SampleslogWt(SamplesCounter) = ObjlogWt(best)
                ! SamplesPosition(SamplesCounter,:) = ObjPosition(best,:)
                ! SamplesPrior(SamplesCounter,:) = ObjPrior(best,:)
                ! SamplesMinval(SamplesCounter,:) = ObjMinval(best,:)
                ! SamplesMaxval(SamplesCounter,:) = ObjMaxval(best,:)


                !< Kill worst object in favour of copy of different survivor
                logLstar = ObjlogL(worst)                                                   !< new likelihood constraint
                if (n > 1) then                                                             !< don't kill if n is only 1
                    copy = 0
                    Do
                        copy = int(n * RandomWithLimits(0.d0, 1.d0))                        !< force 0 <= copy < n
                        if (copy /= worst) exit
                    end Do
                    if (copy > n.or.copy < 1) then
                        copy = int(RandomWithLimits(1.d0, dfloat(n)))
                    endif
                    ObjlogL(worst) = ObjlogL(copy)                                          !< overwrite worst object: logL
                    ObjlogWt(worst) = ObjlogWt(copy)                                        !< overwrite worst object: logWt
                    ObjPosition(worst,:) = ObjPosition(copy,:)                              !< overwrite worst object: Position
                    ObjPrior(worst,:) = ObjPrior(copy,:)                                    !< overwrite worst object: Prior
                    ObjMinval(worst,:) = ObjMinval(copy,:)                                  !< overwrite worst object: Minval
                    ObjMaxval(worst,:) = ObjMaxval(copy,:)                                  !< overwrite worst object: Maxval
                endif


                !< Evolve copied object within constraint
                call explore(nfit, ObjMinval(worst,:), ObjMaxval(worst,:), ObjPrior(worst,:), ObjPosition(worst,:), ObjlogL(worst), logLstar, ma, &
                             NumFile, MaxL, MaxCol)


                !< Shrink interval
                logwidth = logwidth - (1.d0 / n)


                !< print information about current iteration
                if (nest > 0) then


                    !< update a array
                    i = 0
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            i = i + 1
                            a(j) = BestSitesParamSet(1, i + 1)
                        endif
                    end Do


                    !< build list with fit parameters
                    k = 0
                    ListParamFormated = ""
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                            if (IntegerTrue) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(j))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(a(j))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(j)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) a(j)
                                endif
                            endif
                            if (k == 1) then
                                ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                            else
                                ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                            endif
                        endif
                    end Do
                    currentparmval = a


                    !< print status of iteration process ..
                    if (printflag) then
                        print '(11x,I10,ES26.15,5x,A)',nest, (BestSitesParamSet(1, 1)), trim(adjustl(ListParamFormated))
                    endif
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("  ")')
                    write(paramchannel,'(123("*"))')
                    write(paramchannel,'("Iteration: ",I5,",  chi^2 = ",ES25.15)') nest, (BestSitesParamSet(1, 1))
                    write(logchannel,'(11x,I10,ES26.15,5x,A)') nest, (BestSitesParamSet(1, 1)), trim(adjustl(ListParamFormated))


                    !< write actual parameters to files
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("Parameters: ",A)') trim(adjustl(ListParamFormated))
                    write(paramchannel,'(123("-"))')
                    write(paramchannel,'("  ")')


                    !< save current experimental x point of the first experimental file to variable posdatexp
                    posdatexp(1:colx) = 0.d0


                    !< call subroutine to write current values of the parameter to file
                    write(paramchannel,'("-",61(" -"))')
                    Do NumInputFiles = 1, NumberInputFiles
                        write(paramchannel,'("Input-File ",I5,":  , file: ",A)') NumInputFiles, trim(adjustl(FitFktInput(NumInputFiles)))
                        write(paramchannel,'("  ")')
                        write(paramchannel,'("-start_input-file",106("-"))')
                        call WriteParameter(paramchannel, .true., colx, posdatexp, parameternumber, a, NumInputFiles)
                        write(paramchannel,'("-end_input-file",108("-"))')
                    end Do


                    !< plot experimental data, model function, and chi**2
                    if (PlotIteration == 0) then
                        if (nest == 1) then
                            InitPlotFlag = .true.
                        else
                            InitPlotFlag = .false.
                        endif
                        call PlotFitFunction(InitPlotFlag, xAxisLabel, yAxisLabel, zAxisLabel)
                    endif
                endif
                if ((BestSitesParamSet(1, 1)) < chilim) then
                    exit
                endif
            end Do


            !< print reason for stop of iteration to screen and to log-file
            write(paramchannel,'("  ")')
            write(paramchannel,'(123("="))')
            write(paramchannel,'("  ")')
            if ((BestSitesParamSet(1, 1)) < chilim) then
                write(LongNumber1,'(ES25.15)') (BestSitesParamSet(1, 1))
                write(LongNumber2,'(ES25.15)') chilim
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)', trim(adjustl(LongNumber1)), trim(adjustl(LongNumber2))
                endif
                write(logchannel,'("  ")')
                write(logchannel,'(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)') trim(adjustl(LongNumber1)), trim(adjustl(LongNumber2))

            else
                if (printflag) then
                    print '(" ")'
                    print '(11x, "Iteration stopped. Number of iterations is equal to max. number of iterations = ", I6)', MaxNSIter
                endif
                write(logchannel, '("  ")')
                write(logchannel, '(11x, "Iteration stopped. Number of iterations is equal to max. number of iterations = ", I6)') MaxNSIter
            endif


            !< clear screen massage
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
            endif


            !< Exit with evidence Z, information H, and optional posterior Samples
            sdev_H = H/dlog(2.d0)
            sdev_logZ = dsqrt(H/n)
            return
        end subroutine ndim_nest


        !*************************************************************************************************************************************************
        !> subroutine: CallNestedSampling
        !>
        !> input variables:     ni:                 number of iterations
        !>                      nfit:               number of parameters which should be optimized
        !>                      numberObjects:      number of objects
        !>                      ma:                 total number of all parameter
        !>                      currentparmval:     values of all parameter
        !>                      colx:               number of columns belonging to the x-column
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>                      PlotIteration:      dummy argument
        !>                      PlotType:           kind of plotting
        !>                      xAxisLabel:         label for x-axis
        !>                      yAxisLabel:         label for y-axis
        !>                      ThreadNumber:       number of the current thread (used for parallelization)
        !>
        !> output variables:    counter:            number of best sites
        !>                      Parameterset:       parameter sets for best sites
        !>
        !> \author Thomas Moeller
        !>
        !> \date 22.06.2010
        !>
        subroutine CallNestedSampling(ni, nfit, numberObjects, ma, counter, Parameterset, colx, NumFile, MaxL, MaxCol, PlotIteration, PlotType, &
                                      xAxisLabel, yAxisLabel, zAxisLabel, fitlog)
            !< calls nested sampling subroutine

            use Variables
            use FunctionCalling

            implicit none
            integer :: i, j, k, ni                                                          !< loop variables
            integer :: numberObjects                                                        !< number of objects
            integer :: nfit                                                                 !< number of free parameters
            integer :: ma                                                                   !< total number of parameters
            integer :: counter                                                              !< counter for limit determination
            integer :: colx                                                                 !< number of x-columns
            integer :: NumFile                                                              !< max number of input file
            integer :: MaxL                                                                 !< max number of length
            integer :: MaxCol                                                               !< max number of columns
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            integer :: PlotIteration                                                        !< flag for plotting the model function for each step
            integer :: PlotType                                                             !< get type of plot
            real*8 :: fun_min, logZ, logZ_sdev                                              !< working variables
            real*8 :: H                                                                     !< information
            real*8 :: H_sdev                                                                !< standers deviation of H
            real*8 :: dummy                                                                 !< needed for initialization of random subroutine
            real*8, dimension(1) :: chi2ValuesVector                                        !< here only one chi2 value
            real*8, dimension(ni) :: SampleslogL                                            !< (Posterior Samples): logL
            real*8, dimension(ni) :: SampleslogWt                                           !< (Posterior Samples): logWt (weighted logL ?)
            real*8, dimension(ni,nfit) :: SamplesPosition                                   !< (Posterior Samples): Position
            real*8, dimension(ni,nfit) :: SamplesPrior                                      !< (Posterior Samples): Prior
            real*8, dimension(ni,nfit) :: SamplesMinval                                     !< (Posterior Samples): Minval
            real*8, dimension(ni,nfit) :: SamplesMaxval                                     !< (Posterior Samples): Maxval
            real*8, dimension(nfit) :: param                                                !< working variable
            real*8, dimension(ma) :: acopy                                                  !< copy of total parameter list
            real*8, dimension(counter, ma) :: Parameterset                                  !< array containing the parameter set which fullfil the
                                                                                            !< quality creteria
            character(len=100) :: HelpString                                                !< used for number to string conversion
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=5196) :: ListParamFormated                                        !< string for printing free parameters
            character(len=8192) :: fitlog                                                   !< path for log files
            character(len=512) :: LongHelpString1                                           !< help string
            character(len=512) :: WorkingDirectory1, WorkingDirectory2                      !< path and file names for scratch files

            logical :: IntegerTrue                                                          !< flag for indicating integer numbers


            !< initialize ran1
            idum = (-1)
            call ran1(dummy, idum)


            !< initialize values
            DBL_MAX = 1.d300


            !< define OptimizedParameter range and determine starting point
            OptParamLowLimit = 0.d0
            OptParamUpperLimit = 0.d0
            j = 0
            Do i = 1, parameternumber
                if (ia(i)) then
                    j = j + 1
                    OptParamLowLimit(j) = paramset(3,i)
                    OptParamUpperLimit(j) = paramset(4,i)
                endif
            end Do
            ObjectMinval = OptParamLowLimit
            ObjectMaxval = OptParamUpperLimit


            !< call ndim_nest subroutines
            call ndim_nest(numberObjects, nfit, ma, colx, NumFile, MaxL, MaxCol, SampleslogL, SampleslogWt, SamplesPosition, SamplesPrior, &
                           SamplesMinval, SamplesMaxval, ni, logZ, logZ_sdev, H, H_sdev, PlotIteration, PlotType, xAxisLabel, yAxisLabel, &
                           zAxisLabel, fitlog)


            !< print what you do
            if (printflag) then
                print '(" ")'
                print '(11x,"Evidence: ln(Z) = ",F20.10," +/- ",F20.10)', (minmaxTrans - logZ), logZ_sdev
            endif


            !< determine the number of best sites
            j = 0
            Do i = 1, QualityLimit
                if (BestSitesParamSet(i, 1) /= -1.d99) then
                    j = j + 1
                endif
            end Do
            QualityLimit = j


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< select best sites
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(11x,"Best results:")'
            endif
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')
            write(logchannel,'(11x,"Best results:")')


            !< write best results to screen and to log-file
            Do i = 1, QualityLimit
                acopy = a
                fun_min = -BestSitesParamSet(i, 1)
                param = BestSitesParamSet(i, 2:)


                !< build list with fit parameters
                k = 0
                ListParamFormated = ""
                Do j = 1, parameternumber
                    if (ia(j)) then
                        k = k + 1
                        acopy(j) = param(k)

                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                        if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                            .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(acopy(j))
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) int(acopy(j))
                            endif
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) acopy(j)
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) acopy(j)
                            endif
                        endif
                        if (k == 1) then
                            ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                        else
                            ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                        endif
                    endif
                end Do
                Parameterset(i, :) = acopy(:)


                !< print best results to screen and to log-file
                if (printflag) then
                    print '(13x,"Site number: ",I4,", chi^2 = ",ES25.15,",  Parameterset = ",A)', i, (minmaxTrans - fun_min), &
                                                                                                  trim(adjustl(ListParamFormated))
                endif
                write(logchannel,'(13x,"Site number: ",I4,", chi^2 = ",ES25.15,",  Parameterset = ",A)') i, (minmaxTrans - fun_min), &
                                                                                                         trim(adjustl(ListParamFormated))
            end Do


            !< clear screen massage
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
            endif


            !< write SampleFitFunctionOut and SampleChi2Values to scratch file
            write(LongHelpString1,'(I30)') JobID                                            !< write JobID to string
            WorkingDirectory1 = trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(LongHelpString1)) // "/" // "FitFunction.dat"
            WorkingDirectory2 = trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(LongHelpString1)) // "/" // "Chi2Values.dat"
            open(scratchChannel1,file = trim(adjustl(WorkingDirectory1)))
            write(scratchChannel1,*) QualityLimit
            write(scratchChannel1,*) NumFile
            write(scratchChannel1,*) MaxL
            write(scratchChannel1,*) MaxCol
            open(scratchChannel2,file = trim(adjustl(WorkingDirectory2)))
            Do i = 1, QualityLimit


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< determine model function values if plot for each iteration option is not selected
                if (.not. PlotIterationFlag .and. DontStoreModelFuncValuesFlag) then
                    PlotIterationFlag = .true.
                    chi2ValuesVector = 0.d0
                    call ModelCalcChiFunctionGeneral(parameternumber, ia, paramset(1, :), 1, nfit, NumFile, MaxCol, MaxL, &
                                                     BestSitesParamSet(i, 2:), chi2ValuesVector)
                    PlotIterationFlag = .false.
                endif
                Do j = 1, NumFile
                    Do k = 1, MaxL
                        write(scratchChannel1,*) BestSitesModelValues(i, j, k, 1:MaxCol)
                        write(scratchChannel2,*) BestSitesChi2Values(i, j, k, 1:MaxCol)
                    end Do
                end Do
            end Do
            close(scratchChannel1)
            close(scratchChannel2)
            return
        end subroutine CallNestedSampling
end Module NestedSamplingVariables
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: Algorithm
!>
!>         Module contains the main subroutine used to start the different versions of the NestedSampling algorithm
!>
!>
!> \author Thomas Moeller
!>
!> \date 2014-08-31
!>
Module Algorithm

    use Variables
    use NestedSamplingVariables

    implicit none

    contains


        !*************************************************************************************************************************************************
        !> subroutine: MainAlg
        !>
        !> main subroutine which starts the nested sampling algorithm
        !>
        !>
        !> input variables:         printflagNum:           flag for screen output 1 (=yes) or 0 (=no)
        !>                          LastAlgorithmNum:       number of last algorithm
        !>                          ParamSetCounter:        number of best sites
        !>                          FinalParameterSet:      array containing the parameter set with SampleslogL(i) >= logZ
        !>                          chilm:                  user defined abort criteria for chi**2
        !>                          numiter:                max. number of iterations
        !>                          NestedSamplingCounter:  counts number of calls
        !>                          DeterminationChi2:      method being used for the determination of chi^2
        !>                          PlotIterationOrg:       plot model function for each iteration set 1(=yes) or 0(=no)
        !>                          PlotTypeOrg:            get type of plot
        !>                          fitlog:                 path for log-file containing the current values of chi**2
        !>                          NumberInputFilesorg:    number of input files for the external model program
        !>                          NumberOutputFilesOrg:   number of output files for the external model program
        !>                          ParallelizationFlagorg: contains the number of processors used for parallelization
        !>                          JobIDorg:               job identification number
        !>                          MaxInputLinesOrg:       max number of lines in an input file
        !>                          MaxParameterOrg:        max number of parameters in a line of an input file
        !>                          RenormalizedChi2Org:    flag for using renormalized chi**2
        !>                          currentpathorg:         path of the working directory
        !>                          FitParameterNameOrg:    array containing the names of the model parameters
        !>                          FitParameterValueLineOrg:   array containing the values of the model parameters as string
        !>                          CalculationMethodOrg:   method of computation (at once or point-to-point)
        !>                          xAxisLabel:             label of the x-axis (for plot)
        !>                          yAxisLabel:             label of the y-axis (for plot)
        !>                          zAxisLabel:             label of the z-axis (for plot)
        !>                          PathStartScriptOrg:     path and name of the start script for calling model function
        !>                          ExeCommandStartScriptOrg:   command for calling model function
        !>                          parametersetorg:        the complete set of paramters (incl. flags and limits)
        !>                          expdataxorg:            array containing the experimental x side
        !>                          expdatayorg:            array containing the experimental y side
        !>                          expdataerrororg:        array containing the experimental error of the y side
        !>                          NumberRangesOrg:        number of y-columns for each experimental file
        !>                          MinRangeOrg:            array containing the minimal exp. ranges
        !>                          MaxRangeOrg:            array containing the maximal exp. ranges
        !>                          NumberXColumnsOrg:      number of x-columns for each experimental file
        !>                          NumberYColumnsOrg:      number of y-columns for each experimental file
        !>                          lengthexpdataorg:       number of lines in experimental data
        !>                          MaxRangeNumber:         max. number of ranges
        !>                          NumFileOrg:             number of experimental files
        !>                          MaxLengthOrg:           max length of experimental data
        !>                          MaxColXOrg:             number of columns concerning to the experimental x side
        !>                          MaxColYOrg:             number of columns concerning to the experimental y side
        !>                          parameternum:           number of model parameter
        !>                          SortFortranNum:         sort chi^2 log file by fortran
        !>
        !> output variables:        calstatus:              status flag of calculation (= 0: all ok)
        !>                          ParamSetCounter:        number of best sites
        !>                          FinalParameterSet:      array containing the parameter set with SampleslogL(i) >= logZ
        !>
        subroutine MainAlg(printflagNum, LastAlgorithmNum, calstatus, FitFunctionOut, Chi2Values, chilm, NumberOfFitAlgorithms, numiter, &
                           NestedSamplingCounter, ParamSetCounter, GeneralAlgorithmSettings, DeterminationChi2, PlotIteration, PlotType, fitlog, &
                           NumberInputFilesorg, NumberOutputFilesOrg, ParallelizationFlagorg, JobIDorg, MaxInputLinesOrg, MaxParameterOrg, &
                           RenormalizedChi2Org, currentpathorg, FitParameterNameLocal, FitParameterValueLocal, CalculationMethodOrg, xAxisLabel, &
                           yAxisLabel, zAxisLabel, PathStartScriptOrg, ExeCommandStartScriptOrg, parametersetorg, FinalParameterSet, expdataxorg, &
                           expdatayorg, expdataerrororg, NumberRangesOrg, MinRangeOrg, MaxRangeOrg, NumberXColumnsOrg, NumberYColumnsOrg, &
                           lengthexpdataorg, MaxRangeNumber, NumFileOrg, MaxLengthOrg, MaxColXOrg, MaxColYOrg, parameternum, SortFortranNum)


            implicit none
            ! ********** input variables **********
            integer :: parameternum                                                         !< number of model parameter
            integer :: NumberOfFitAlgorithms                                                !< total number of all algorithms in the chain
            integer :: numiter                                                              !< max. number of iterations
            integer :: NumFileOrg                                                           !< number of experimental files
            integer, dimension(NumFileOrg) :: lengthexpdataorg                              !< number of lines in experimental data
            integer, dimension(NumFileOrg) :: NumberXColumnsOrg                             !< number of x-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberYColumnsOrg                             !< number of y-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberRangesOrg                               !< number of y-columns for each experimental file
            integer :: MaxColXOrg                                                           !< number of columns concerning to the experimental x side
            integer :: MaxColYOrg                                                           !< number of columns concerning to the experimental y side
            integer :: MaxLengthOrg                                                         !< max length of experimental data
            integer :: printflagNum                                                         !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: LastAlgorithmNum                                                     !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: SortFortranNum                                                       !< flag indicating if chi2 log file is sorted by fortran
                                                                                            !< yes (=1) or not (=0)
            integer :: DeterminationChi2                                                    !< method being used for the determination of chi^2
            integer :: PlotIteration                                                        !< plot model func. for each iteration set 1 (=yes) or 0 (=no)
            integer :: PlotTypeOrg                                                          !< get type of plot
            integer :: NumberInputFilesorg                                                  !< number of input files for the external model program
            integer :: NumberOutputFilesOrg                                                 !< number of output files for the external model program
            integer :: ParallelizationFlagorg                                               !< contains the number of processors used for parallelization
            integer :: JobIDorg                                                             !< job identification number
            integer :: MaxInputLinesOrg                                                     !< max number of lines in an input file
            integer :: MaxParameterOrg                                                      !< max number of parameters in a line of an input file
            integer :: RenormalizedChi2Org                                                  !< flag for using renormalized chi**2
            integer :: numberObjectsorg                                                     !< number of objects
            integer :: NestedSamplingCounter                                                !< counts the number of calls
            integer :: MaxRangeNumber                                                       !< max. number of ranges
            integer :: ParamSetCounter                                                      !< which positions should be used
            real*8 :: chilm                                                                 !< user defined abort criteria for chi**2
            real*8, dimension(15) :: GeneralAlgorithmSettings                               !< special algorithm settings
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColXOrg) :: expdataxorg          !< array containing the experimental x side
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColYOrg) :: expdatayorg          !< array containing the experimental y side
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColYOrg) :: expdataerrororg      !< array containing the experimental error of the y side
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MinRangeOrg        !< array containing the minimal exp. ranges
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MaxRangeOrg        !< array containing the maximal exp. ranges
            character(len=8192) :: fitlog                                                   !< path for log-file containing the current values of chi**2
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=20) :: CalculationMethodOrg                                       !< method of computation
            character(len=8192) :: PathStartScriptOrg                                       !< command for calling model function
            character(len=8192) :: ExeCommandStartScriptOrg                                 !< command for calling model function
            character(len=8192) :: currentpathorg                                           !< path of the working directory
            character(len=512), dimension(parameternum) :: FitParameterNameLocal            !< array containing the names of the model parameters
            character(len=512), dimension(parameternum) :: FitParameterValueLocal           !< array containing the values of the model parameters as


            ! ********** in/output variables **********
            real*8, dimension(4, parameternum) :: parametersetorg                           !< the non-optimized (initial parameter set)


            ! ********** output variables **********
            integer :: calstatus                                                            !< the following line is necessary for f2py
            real*8, dimension(ParamSetCounter, parameternum) :: FinalParameterSet           !< array containing the optimized parameter set
            real*8, dimension(ParamSetCounter, NumFileOrg, MaxLengthOrg, MaxColYOrg) :: FitFunctionOut !< values of the model func. at the calculated pts.
            real*8, dimension(ParamSetCounter, NumFileOrg, MaxLengthOrg, MaxColYOrg) :: Chi2Values     !< values of the model func. at the calculated pts.


            !<********** working variabels **********
            integer :: i, j, k, ii, jj, NumInputFiles                                       !< working variables
            integer :: nfit, ma, MaxLength, PlotType                                        !< working variables
            integer :: MaxObj                                                               !<
            integer :: ok                                                                   !< status of calculation
            integer :: actualiteration                                                      !< contains the current iteration within the iteration loop
            integer :: flag                                                                 !< working variable used within the iteration loop
            integer :: numberObjects                                                        !< number of objects
            integer :: allocstatus, deallocstatus                                           !< working variables for allocation/deallocation
            integer, dimension(8) :: VALUES                                                 !< value for the date_and_time subroutine
            real*8, dimension(1) :: chi2ValuesVector                                        !< here only one chi2 value
            real*8, allocatable, dimension(:) :: currentparmval                             !< array containing the current parameter set within the
                                                                                            !< iteration loop
            character(len=512) :: fitlogparam                                               !< path for log-file containing the current parameter values
            character(len=512) :: fitlogChi2                                                !< path for log-file containing chi**2 and the corresponding
                                                                                            !< parameter values
            character(len=8) :: DATE                                                        !< variable for the date_and_time subroutine
            character(len=10) :: TIME                                                       !< variable for the date_and_time subroutine
            character(len=5) :: ZONE                                                        !< variable for the date_and_time subroutine
            character(len=10) :: Number1                                                    !< variable for number to string converting
            character(len=100) :: helpString                                                !<
            character(len=8192) :: SiteExt, NumExt, BaseDir, FuncCallExt                    !< working variables for final input file name


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< set print flag and last algorithm flag
            calstatus = 0                                                                   !< set calculation status to 0 = everything is ok
            if (printflagNum == 1) then                                                     !< set printflag
                printflag = .true.
            else
                printflag = .false.
            endif
            if (LastAlgorithmNum == 1) then
                LastAlgorithmFlag = .true.
            else
                LastAlgorithmFlag = .false.
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< copy contents of some input variables to module variables
            NumberExpFiles = NumFileOrg                                                     !< copy number of experimental files to global variable
            currentpath = trim(adjustl(currentpathorg))                                     !< copy path of working directory to module variable without
                                                                                            !< trailing and leading blanks
            MaxColX = MaxColXOrg                                                            !< copy number of columns of the experimental x data to
                                                                                            !< module variable
            MaxColY = MaxColYOrg                                                            !< copy number of columns of the experimental y data to
                                                                                            !< module variable
            MaxLength = MaxLengthOrg                                                        !< copy max. number of lines
            MaxExpLength = MaxLength                                                        !< copy max. number of exp. data points
            MaxNumberRanges = MaxRangeNumber                                                !< copy of max. number of data ranges in a exp. data file
            parameternumber = parameternum                                                  !< copy input variable containing the number of parameters
                                                                                            !< to module variable
            DetChi2 = DeterminationChi2                                                     !< copy method of chi**2 determination
            NumberInputFiles = NumberInputFilesorg                                          !< copy number of input files for the external program to
                                                                                            !< global variable
            NumberOutputFiles = NumberOutputFilesOrg                                        !< copy number of output files for the external program to
                                                                                            !< global variable
            ParallelizationFlag = ParallelizationFlagorg                                    !< copy number of used processors to global variable
            JobID = JobIDorg                                                                !< copy job-ID number to global variable
            PlotType = PlotTypeOrg                                                          !< copy flag for type of plotting
            MaxInputLines = MaxInputLinesOrg                                                !< copy max number of input lines in an input file
            MaxParameter = MaxParameterOrg                                                  !< copy max number of parameters in a line of an input file
            MaxNSIter = numiter                                                             !< copy max number of iterations
            RenormalizedChi2 = .true.                                                       !< define flag for using renormalized chi**2
            if (RenormalizedChi2Org /= 1) then
                RenormalizedChi2 = .false.
            endif
            PlotIterationFlag = .false.
            if (PlotIteration == 0) PlotIterationFlag = .true.


            !< get special algorithm settings
            numberObjectsorg = int(GeneralAlgorithmSettings(3))
            numberObjects = numberObjectsorg                                                !< copy number of objects
            minmaxTrans = 0                                                                 !< number to invert the model function
            QualityLimit = ParamSetCounter                                                  !< which positions should be used

            ! Debug:
            !    print*,'PathStartScriptOrg = ',trim(PathStartScriptOrg)
            !    print*,'ExeCommandStartScriptOrg = ',trim(ExeCommandStartScriptOrg)
            !    print*,'FitFktInputOrg = ',trim(FitFktInputOrg)
            !    print*,'MaxNumberParameter = ',MaxNumberParameter
            !    print*,'NumFileOrg = ',NumFileOrg
            !    print*,'MaxColXOrg = ',MaxColXOrg
            !    print*,'MaxColYOrg = ',MaxColYOrg
            !    Do i=1,NumFileOrg
            !        print*,'    Experimental file: i = ',i
            !        print*,'    lengthexpdataorg(i) = ',lengthexpdataorg(i)
            !        print*,'    NumberYColumnsOrg(i) = ',NumberYColumnsOrg(i)
            !        print*,'    expdataxorg(i,1:5,1) = ',expdataxorg(i,1:5,1)
            !        print*,'    expdatayorg(i,1:5,1) = ',expdatayorg(i,1:5,1)
            !        print*,'    expdataerrororg(i,1:5,1) = ',expdataerrororg(i,1:5,1)
            !    end Do
            !    print*,'chilm = ',chilm
            !    print*,'numiter = ',numiter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'currentpathorg = ',trim(adjustl(currentpathorg))
            !    print*,'len(FitParameterNameOrg) = ',len(FitParameterNameOrg
            !    print*,'FitParameterNameOrg = ',FitParameterNameOrg
            !    print*,'FitParameterValueLineOrg = ',FitParameterValueLineOrg
            !    print*,"parametersetorg(1,:) = ",parametersetorg(1,:)
            !    print*,"parametersetorg(2,:) = ",parametersetorg(2,:)
            !    print*,"parametersetorg(3,:) = ",parametersetorg(3,:)
            !    print*,"parametersetorg(4,:) = ",parametersetorg(4,:)
            !    print*,"RenormalizedChi2 = ",RenormalizedChi2
            !    return


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< set temp-directory
            TempDirectory = " "
            CALL GetEnv('MAGIXTempDirectory', TempDirectory)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< modify name of log file
            i = index(fitlog,"/",back = .true.)
            j = index(fitlog,".",back = .true.)
            Number1 = "          "
            write(Number1,'(I10)') NestedSamplingCounter
            if (j > i) then
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__NS__call_" // trim(adjustl(Number1)) // trim(adjustl(fitlog(j:)))
                else
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__NS" // trim(adjustl(fitlog(j:)))
                endif
            else
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog)) // "__NS__call_" // trim(adjustl(Number1)) // ".log"
                else
                    fitlog = trim(adjustl(fitlog)) // "__NS.log"
                endif
            endif

            ! Debug:
            !   print*,'>',trim(adjustl(fitlog)),'<'


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open log file and write header
            open(logchannel,file = trim(fitlog), status='replace')
            write(logchannel,'(" ")')
            write(logchannel,'("log-file for Nested-Sampling algorithm:")')
            write(logchannel,'(39("-"))')
            write(logchannel,'(" ")')
            call date_and_time(DATE, TIME, ZONE, VALUES)                                    !< get current local time and date and write to log-file
            write(logchannel,'(" ")')
            write(logchannel,'("algorithm starts at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                                TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open log-file for the parameter and write header
            fitlogparam = trim(fitlog)//".param"
            open(paramchannel,file = trim(fitlogparam), status='replace')
            write(paramchannel,'(" ")')
            write(paramchannel,'("log-file containing the actual values of the parameters used in the Particle-Swarm algorithm:")')
            write(paramchannel,'(93("-"))')
            write(paramchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open file containing the values of chi**2 and the corresponding values of the parameters
            WriteChi2Flag = .true.
            NumberLinesChi2 = 0
            fitlogChi2 = trim(fitlog)//".chi2"
            open(Chi2Channel,file = trim(fitlogChi2), status='replace')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate if necessary and print error message if necessay
            if (allocated(expdatax)) then
                deallocate(expdatax, expdatay, expdatae, lengthexpdata, NumberXColumns, NumberYColumns, FirstPointExpData, LastPointExpData, &
                           NumberRanges, MinRange, MaxRange, ExpData_reversed_flag, stat = deallocstatus)
                if (deallocstatus /= 0) then                                                !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine MainAlg:")')
                    write(logchannel,'(2x,"Can not deallocate variables expdatax etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine MainAlg:")'
                    print '(2x,"Can not deallocate variables expdatax etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for variables, clear content of the variables and print error message if necessay
            allocate(expdatax(NumberExpFiles, MaxLength, MaxColX), expdatay(NumberExpFiles, MaxLength, MaxColY), &
                     expdatae(NumberExpFiles, MaxLength, MaxColY), lengthexpdata(NumberExpFiles), NumberXColumns(NumberExpFiles), &
                     NumberYColumns(NumberExpFiles), FirstPointExpData(NumberExpFiles, MaxColX), LastPointExpData(NumberExpFiles, MaxColX), &
                     NumberRanges(NumberExpFiles), MinRange(NumberExpFiles, MaxRangeNumber, MaxColX), MaxRange(NumberExpFiles, MaxRangeNumber, MaxColX), &
                     ExpData_reversed_flag(NumberExpFiles), stat = allocstatus)
            if (allocstatus /= 0) then                                                      !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not allocate variables expdatax etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not allocate variables expdatax,expdatay etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            MaxRangeNumber = MaxRangeNumber - 1                                             !< get real value
            expdatax = 0.d0
            expdatay = 0.d0
            expdatae = 0.d0
            lengthexpdata = 0
            NumberXColumns = 0
            NumberYColumns = 0
            FirstPointExpData = 1.d99
            LastPointExpData = -1.d99
            NumberRanges = 0
            MinRange = 0.d0
            MaxRange = 0.d0
            ExpData_reversed_flag = .false.


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy input variables to module variabels
            expdatax(:,:,:) = expdataxorg(:,:,:)
            expdatay(:,:,:) = expdatayorg(:,:,:)
            expdatae(:,:,:) = expdataerrororg(:,:,:)
            lengthexpdata = lengthexpdataorg
            NumberXColumns = NumberXColumnsOrg
            NumberYColumns = NumberYColumnsOrg
            CalculationMethod = CalculationMethodOrg
            PathStartScript = PathStartScriptOrg
            ExeCommandStartScript = ExeCommandStartScriptOrg
            NumberRanges = NumberRangesOrg
            MinRange = MinRangeOrg
            MaxRange = MaxRangeOrg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine first and last point of each exp. data file
            Do i = 1, NumberExpFiles
                Do j = 1, lengthexpdata(i)
                    ii = 0
                    jj = 0
                    Do k = 1,NumberXColumns(i)
                        if (expdatax(i, j, k) <= FirstPointExpData(i, k)) then
                            ii = ii + 1
                        endif
                        if (expdatax(i, j, k) >= LastPointExpData(i, k)) then
                            jj = jj + 1
                        endif
                    end Do
                    if (ii == NumberXColumns(i)) then
                        FirstPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                    if (jj == NumberXColumns(i)) then
                        LastPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                end Do


                !< check output file starts with the highest x-column value interchange FirstPointOutputFile and LastPointOutputFile
                ii = 0
                Do k = 1, NumberXColumns(i)
                    if (expdatax(i, 1, k) >= expdatax(i, lengthexpdata(i), k)) then
                        ii = ii + 1
                    endif
                end Do
                if (ii == NumberXColumns(i)) then
                    ExpData_reversed_flag(i) = .true.
                endif

                ! Debug:
                ! print*,' '
                ! print*,'File = ',i
                ! print*,'FirstPointExpData(i, 1:NumberXColumns(i)) = ', FirstPointExpData(i, 1:NumberXColumns(i))
                ! print*,'LastPointExpData(i, 1:NumberXColumns(i)) = ', LastPointExpData(i, 1:NumberXColumns(i))
                ! print*,'##########################################'
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy number of parameters required for the model function to working variable
            nfit = sum(parametersetorg(2,:))                                                !< number of parameters which should be optimized
            ma = parameternumber
            Gradientflag = .false.                                                          !< we do not need the gradient of the function here
            UseCalculationReduction = .true.                                                !< activate calculation reduction
            CurrentNumberLinesCalcReduction = 0                                             !< reset number of lines


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate/allocate memory for some working variables, clear contents of the variables and print error message if necessary
            if (allocated(paramset)) then
                deallocate(paramset, currentparmval, FitParameterName, FitParameterValue, ia, a, OptParamLowLimit, OptParamUpperLimit, &
                           ModelFunction, chisqValues, ObjectMinval, ObjectMaxval, ObjectPrior, ObjectPosition, BestSitesParamSet, AtOnceFunction, &
                           BestSitesModelValues, BestSitesChi2Values, ConverterInfit, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine MainAlg:")')
                    write(logchannel,'(2x,"Can not deallocate variables paramset etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine MainAlg:")'
                    print '(2x,"Can not deallocate variables paramset etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif
            MaxObj = numberObjects
            if (numberObjects < MaxNSIter) then
                MaxObj = MaxNSIter
            endif

            allocate(paramset(4, parameternumber), currentparmval(parameternumber), OptParamLowLimit(nfit), OptParamUpperLimit(nfit), &
                     ia(parameternumber), FitParameterName(parameternumber), FitParameterValue(parameternumber), a(parameternumber), &
                     ModelFunction(1, NumberExpFiles, MaxColY, MaxLength), chisqValues(0:0), ConverterInfit(nfit), &
                     AtOnceFunction(0:ParallelizationFlag - 1, NumberExpFiles, MaxColY, MaxLength), &
                     ObjectMinval(nfit), ObjectMaxval(nfit), ObjectPrior(nfit), ObjectPosition(nfit), BestSitesParamSet(QualityLimit, nfit + 1), &
                     BestSitesModelValues(QualityLimit, NumberExpFiles, MaxLength, MaxColY), &
                     BestSitesChi2Values(QualityLimit, NumberExpFiles, MaxLength, MaxColY), stat = allocstatus)
            if (allocstatus /= 0) then
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not allocate variables paramset etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not allocate variables paramset etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            ia = .false.
            FitParameterName = FitParameterNameLocal
            FitParameterValue = FitParameterValueLocal
            OptParamLowLimit = 0.d0
            OptParamUpperLimit = 0.d0
            currentparmval = 0.d0
            ModelFunction = 0.d0
            AtOnceFunction = 0.d0
            chisqValues = 0.d0
            ConverterInfit = 0
            BestSitesParamSet = 0.d0
            BestSitesParamSet(:,1) = 1.d99
            BestSitesModelValues = 0.d0
            BestSitesChi2Values = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy parameter set to module variable
            paramset = parametersetorg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write registration mask for the fit model
            if (printflag) print '(9x, "Writing registration mask for the fit model .. ", $)'
            call RegistrationMask(ok)
            if (ok /= 0) then
                return
            endif
            if (printflag) print '("done!")'


            !< define ia variable
            ConverterInfit = 0
            ia = .false.
            k = 0
            Do i = 1, parameternumber
                if (paramset(2, i) == 1) then
                    k = k + 1
                    ia(i) = .true.
                    ConverterInfit(k) = i
                endif
            end Do
            NumberFreeParameter = int(sum(paramset(2, :)))                                  !< determine number of free parameter


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize model program
            call ModelInit


            !< write on the screen what you Do ..
            if (printflag) then
                print '(" ")'
                print '(" ")'
                write(Number1,'(I10)') JobID                                                !< write JobID to string
                print '(9x,"Temporary files are stored in: ",A)', trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(Number1)) // "/"
                print '(" ")'
                print '(" ")'
                print '(9x,"Start Nested-Sampling algorithm (", A, " version) ..")', trim(adjustl(ParallelizationMethod))
                print '(" ")'
            endif
            write(logchannel,'(11x,"Start Nested-Sampling algorithm (", A, " version) ..")') trim(adjustl(ParallelizationMethod))
            write(logchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< lower limit of chi^2 for stopping condition normalized to the number of calculation points
            chilim = 0.d0
            if (RenormalizedChi2) then
                Do i = 1, NumberExpFiles
                    chilim = chilim + (NumberYColumns(i) * lengthexpdata(i) - parameternumber) * dabs(chilm)
                end Do
                chilim = dabs(chilim)
                write(logchannel,'(11x,"Renormalized limit for chi^2 = ", ES25.15)') chilim
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Renormalized limit for chi^2 = ",ES25.15)', chilim
                    print '(" ")'
                endif
            else
                chilim = dabs(chilm)
                write(logchannel,'(11x,"Limit for chi^2 = ", ES25.15)') chilim
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Limit for chi^2 = ",ES25.15)', chilim
                    print '(" ")'
                endif
            endif


            !< write number of chromosomes to screen and log-file
            if (printflag) then
                print '(11x,"Number of objects = ",I10)', numberObjects
                print '(" ")'
                print '(" ")'
            endif
            write(logchannel,'(11x,"Number of objects = ",I10)') numberObjects
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< start iteration ..
            Gradientflag = .false.                                                          !< we do not need the gradient of the function here
            flag = 0                                                                        !< set flag variable to 0
            actualiteration = 0                                                             !< set working variable containing the current
                                                                                            !< iteration number to 0
            currentparmval = paramset(1,1:parameternumber)                                  !< copy the values of the parameters to another array
            a = currentparmval


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< call subroutine to call bees algorithm
            call CallNestedSampling(MaxNSIter, nfit, numberObjects, ma, ParamSetCounter, FinalParameterSet, MaxColX, NumberExpFiles, MaxLength, MaxColY, &
                                    PlotIteration, PlotType, xAxisLabel, yAxisLabel, zAxisLabel, fitlog)
            ParamSetCounter = QualityLimit                                                  !< set total number of best sites
            FitFunctionOut = 0.d0
            Chi2Values = 0.d0
            Do i  = 1, QualityLimit


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< determine model function values if plot for each iteration option is not selected
                if (.not. PlotIterationFlag .and. DontStoreModelFuncValuesFlag) then
                    PlotIterationFlag = .true.
                    chi2ValuesVector = 0.d0
                    call ModelCalcChiFunctionGeneral(parameternumber, ia, paramset(1, :), 1, nfit, NumFileOrg, MaxColY, MaxLength, &
                                                     BestSitesParamSet(i, 2:), chi2ValuesVector)
                    PlotIterationFlag = .false.
                endif
                Do j = 1, NumFileOrg
                    Do k = 1, MaxLength
                        FitFunctionOut(i, j, k, 1:MaxColY) = BestSitesModelValues(i, j, k, 1:MaxColY)
                        Chi2Values(i, j, k, 1:MaxColY) = BestSitesChi2Values(i, j, k, 1:MaxColY)
                    end Do
                end Do
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< close log-files
            if (SortFortranNum == 1) call SortChi2File(nfit, parameternumber, ia, a)
            close(Chi2Channel)
            close(paramchannel)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< write final parameter sets for each site to final input files


            !< define base directory, i.e. path where final input file is written to
            k = index(fitlog, '/', back = .true.)
            if (k == 0) then
                BaseDir = ""
            else
                BaseDir = trim(adjustl(fitlog(:k)))
            endif


            !< define file name extension for number of algorithm call
            write(Number1,'(I10)') NestedSamplingCounter
            if (NumberOfFitAlgorithms > 1) then
                FuncCallExt = "__NS__call_" //trim(adjustl(Number1))
            else
                FuncCallExt = "__NS"
            endif

            ! Debug:
            !    print*,"NumberOfFitAlgorithms = ", NumberOfFitAlgorithms
            !    print*,"FuncCallExt = ", trim(adjustl(FuncCallExt))


            !< write files
            Do i  = 1, QualityLimit                                                         !< loop over all sites


                !< create site extension for file name
                write(helpString,'(I5)') i
                SiteExt = "__site_" // trim(adjustl(helpString)) // ".out"


                !< write parameter sets to file
                Do NumInputFiles = 1, NumberInputFiles                                      !< loop over all input files
                    NumExt = trim(adjustl(FitFktInput(NumInputFiles)))
                    k = index(NumExt, '/', back = .true.)
                    if (k > 0) then                                                         !< we only need the file name
                        NumExt = trim(adjustl(NumExt(k:)))
                    endif
                    j = index(NumExt, '.', back = .true.)
                    if (j > 1) then                                                         !< remove file name extension
                        NumExt = trim(adjustl(NumExt(:j - 1)))
                    endif

                    ! Debug:
                    ! print*,"Site = ", i
                    ! print*,"Nr. Final input file = ", NumInputFiles
                    ! print*,"Final input file = ", trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt))) &
                    !                               // trim(adjustl(SiteExt) // ".input"


                    !< write parameter sets to file
                    open(235,file = trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt)) // trim(adjustl(SiteExt)) // ".input")
                    call WriteParameter(235, .false., MaxColX, expdataxorg, parameternumber, FinalParameterSet(i, :), NumInputFiles)
                    close(235)
                end Do
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write end of log-file and print message to screen
            call date_and_time(DATE, TIME, ZONE, VALUES)
            write(logchannel,*)
            write(logchannel,'("algorithm ends at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                                  TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel,'(" ")')
            write(logchannel,'(150("-"))')

            ! Debug:
            !    print*,'lengthexpdata = ',lengthexpdata
            !    print*,'MaxColX = ',MaxColX
            !    print*,'MaxColY = ',MaxColY
            !    print*,'expdataxorg(1:5,1) = ',expdataxorg(1:5,1)
            !    print*,'expdatayorg(1:5,1) = ',expdatayorg(1:5,1)
            !    print*,'expdataerrororg(1:5,1) = ',expdataerrororg(1:5,1)
            !    print*,'chilm = ',chilm
            !    print*,'numiter = ',numiter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'currentpathorg = ',currentpathorg
            !    print*,'parametersetorg(1,:) = ',parametersetorg(1,:)
            !    print*,'parametersetorg(2,:) = ',parametersetorg(2,:)
            !    print*,'parametersetorg(3,:) = ',parametersetorg(3,:)
            !    print*,'parametersetorg(4,:) = ',parametersetorg(4,:)


            !< send plot program the kill signal
            if (PlotIteration == 0) then
                call system("kill -9 " // trim(adjustl(PlotPID)) // " 1>/dev/null 2>&1")
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< free memory of general nested sampling variables
            if (allocated(currentparmval)) deallocate(currentparmval, stat = deallocstatus)
            if (allocated(ia)) deallocate(ia, stat = deallocstatus)
            if (allocated(a)) deallocate(a, stat = deallocstatus)
            if (allocated(OptParamLowLimit)) deallocate(OptParamLowLimit, stat = deallocstatus)
            if (allocated(OptParamUpperLimit)) deallocate(OptParamUpperLimit, stat = deallocstatus)
            if (allocated(ObjectMinval)) deallocate(ObjectMinval, stat = deallocstatus)
            if (allocated(ObjectMaxval)) deallocate(ObjectMaxval, stat = deallocstatus)
            if (allocated(ObjectPrior)) deallocate(ObjectPrior, stat = deallocstatus)
            if (allocated(ObjectPosition)) deallocate(ObjectPosition, stat = deallocstatus)


            !< free memory of model variables
            call ModelParamFree(deallocstatus)
            if (deallocstatus /= 0) then
                write(logchannel,*)
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not deallocate expdatax etc.")')
                write(logchannel,*)
                write(logchannel,'("deallocstatus = ", I4)') deallocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not deallocate variables expdatax etc.")'
                print '(" ")'
                print '("deallocstatus = ", I4)', deallocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< close log file
            close(logchannel)
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(9x,"Finished Nested-Sampling algorithm!")'
                print '(" ")'
                print '(" ")'
            endif


            !< we're done
            return
        end subroutine MainAlg
end Module Algorithm
!*********************************************************************************************************************************************************


