#!/usr/bin/env python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module contains the subroutines for Interval Nested Sampling algorithm for global optimization (MINIMIZATION!) in multidimensional space
##  Copyright (C) 2009 - 2024  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - subroutine ModelFunctionCallClass.__init__:   initialize class ModelFunctionCallClass
##      - subroutine ModelFunctionCallClass.logLhood:   call of model-function
##      - subroutine ModelFunctionCallClass.GetBestResult:          get best result
##      - subroutine ModelFunctionCallClass.GetBestFunctionValues:  get function values of best result
##      - subroutine Interval_Nested_Sampling:          main subroutine for Interval Nested Sampling algorithm algorithm
##      - subroutine check_point_paral:                 Checking of point from the list Vectors (evaluated or not yet): if -yes- that don't evaluate
##                                                      (function value from the list -Points-)
##      - subroutine my_hist1:                          Create histigramms for each parameter after Error estimation using matplotlib
##      - subroutine del_chi2:                          Calculation of quantiles of chi2-distribution on dependence of the number of free parameter
##                                                      (np) for 1-sigma, 2-sigma and 3-sigma contours
##      - subroutine error_info_new:                    Get information about error for each parameter --> (0:num_par)
##      - subroutine bisection:                         Bisection of parametric space (par_space) for k-coordinate direction
##      - subroutine interval_division:                 Division of num_par-dimension parameter space to subdomains with bisection
##      - subroutine calc_mid_point:                    Calculation of center point of the interval <par_space>
##      - subroutine coord_choice:                      Choosing of coordinate for subdivision of parametric space (par_space): Rule A
##      - subroutine inclusion_function:                Calculation of inclusion function with use of interval arithmetics rules: return inclusion
##                                                      - interval F1 = (minimal value , maximal value)
##      - subroutine volume_box:                        Calculation of volume of interval box and return ratio (volume_box/whole_volume)
##      - subroutine calc_cen_point:                    Calculation of center point of the interval par_space
##      - subroutine interval_method:                   Interval analysis for part of space - par_space
##      - subroutine Nlive:                             Function for calculation the initial number of objects and max_iter of NS (if these parameters
##                                                      are defined by default)
##      - subroutine explore:                           Evolve object within function constraint contents
##      - subroutine plus(x,y):                         logarithmic addition log(exp(x)+exp(y))
##      - subroutine ndim_nest:                         This is an implementation of John Skilling's Nested Sampling algorithm for computing the
##                                                      normalizing constant of a probability distribution (usually the posterior in Bayesian inference).
##      - subroutine loglike_nest:                      Initial List of working objects
##
##
##
##  Versions of the program:
##
##  Who             When            What
##
##  I. Bernst       2011-05-13      Working version
##  T. Moeller      2011-06-08      combination with ModelFunctionCall package
##  I. Bernst       2011-09-08      Update
##  I. Bernst       2011-11-29      Update
##  T. Moeller      2012-01-16      improve documentation of source code
##  T. Moeller      2020-01-02      porting to python 3, minor improvments
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##******************************************************************** load packages *********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import math                                                                                 ## load math package
import numpy                                                                                ## load numpy package
import random                                                                               ## load random package
import copy                                                                                 ## import random copy
import sys                                                                                  ## load sys package
import time                                                                                 ## load time package
import os                                                                                   ## load os package
import matplotlib                                                                           ## load python package for plotting matplotlib
import pylab                                                                                ## load python package for plotting pylab
from xclass.addons.MAGIX.Modules.python import FittingEngine                                ## import package containing the fitting engine
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## This class contains all required parameters for the ModelFunctionCall package
class ModelFunctionCallClass:


    ##****************************************************************************************************************************************************
    ## initialize class variables
    def __init__(self, printflagNumIn, GeneralAlgorithmSettingsIn, LastAlgorithmNumIn, DeterminationChi2In, PlotIterationIn, \
                 PlotTypeIn, fit_logIn, NumberInputFilesIn, NumberOutputFilesIn, ParallelizationFlagIn, JobIDIn, MaxInputLinesIn, MaxParameterIn, \
                 RenormalizedChi2In, currentpathIn, CalculationMethodIn, xAxisLabelIn, yAxisLabelIn, zAxisLabelIn, PathStartScriptIn, \
                 ExeCommandStartScriptIn, parameter_setIn, ExpDataXIn, ExpDataYIn, ExpDataErrorIn, NumberRangesIn, MinRangeIn, MaxRangeIn, \
                 NumberXColumnsIn, NumberYColumnsIn, LengthExpRangeIn, MaxRangeNumberIn, NumberExpFilesIn, MaxLengthIn, MaxColXIn, MaxColYIn, \
                 NumberParameterIn, MPIFlagIn, MAGIXrootDirectoryIn, JobDirIn, SpecialAlgorithmSettingsIn, StarterExecutable, Chi2Channel, \
                 logchannel, paramchannel, INSBestSiteCounter, SortFortranNum):
        self.printflagNum = printflagNumIn
        self.GeneralAlgorithmSettings = GeneralAlgorithmSettingsIn
        self.LastAlgorithmNum = LastAlgorithmNumIn
        self.DeterminationChi2 = DeterminationChi2In
        self.PlotIteration = PlotIterationIn
        self.PlotType = PlotTypeIn
        self.fitlog = fit_logIn
        self.NumberInputFiles = NumberInputFilesIn
        self.NumberOutputFiles = NumberOutputFilesIn
        self.ParallelizationFlag = ParallelizationFlagIn
        self.JobID = JobIDIn
        self.MaxInputLines = MaxInputLinesIn
        self.MaxParameter = MaxParameterIn
        self.RenormalizedChi2 = RenormalizedChi2In
        self.currentpath = currentpathIn
        self.CalculationMethod = CalculationMethodIn
        self.xAxisLabel = xAxisLabelIn
        self.yAxisLabel = yAxisLabelIn
        self.zAxisLabel = zAxisLabelIn
        self.PathStartScript = PathStartScriptIn
        self.ExeCommandStartScript = ExeCommandStartScriptIn
        self.parameter_set = parameter_setIn
        self.ExpDataX = ExpDataXIn
        self.ExpDataY = ExpDataYIn
        self.ExpDataError = ExpDataErrorIn
        self.NumberRanges = NumberRangesIn
        self.MinRange = MinRangeIn
        self.MaxRange = MaxRangeIn
        self.NumberXColumns = NumberXColumnsIn
        self.NumberYColumns = NumberYColumnsIn
        self.LengthExpRange = LengthExpRangeIn
        self.MaxRangeNumber = MaxRangeNumberIn
        self.NumberExpFiles = NumberExpFilesIn
        self.MaxLength = MaxLengthIn
        self.MaxColX = MaxColXIn
        self.MaxColY = MaxColYIn
        self.NumberParameter = NumberParameterIn
        self.MPIFlag = MPIFlagIn
        self.MAGIXrootDirectory = MAGIXrootDirectoryIn
        self.JobDir = JobDirIn
        self.SpecialAlgorithmSettings = SpecialAlgorithmSettingsIn
        self.StarterExecutable = StarterExecutable
        self.Chi2Channel = Chi2Channel
        self.logchannel = logchannel
        self.paramchannel = paramchannel
        self.INSBestSiteCounter = int(INSBestSiteCounter)
        self.BestChi2 = numpy.ones(self.INSBestSiteCounter) * 1.e99
        self.BestFunctionValues = []
        self.BestChi2Values = []
        self.BestParameter = []
        self.BestInputFiles = []
        for i in range(self.INSBestSiteCounter):
            self.BestParameter.append(0.0)
            self.BestFunctionValues.append(0.0)
            self.BestChi2Values.append(0.0)
            self.BestInputFiles.append(0.0)
        self.BestLogLine = ""
        self.FuncCallCounter = 0
        self.TempDir = str(os.environ.get('MAGIXTempDirectory',''))
        self.CurrIter = (-1)                                                            ## turn initialization flag on
        self.SortFortranNum = SortFortranNum


        ## we're done
        return
    ##------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## call of logLhood
    def logLhood(self, number_param_set, parameter_vector, i7, nvec):
        """
        input variables:    number_param_set:       total number of parameter
                            parameter_vector:       parameter vector
                            i7:                     current number of function call
                            nvec:                   total number of function call

        output variables:   chi2value:              value of chi^2
        """

        # Debug:
        # print ("number_param_set = ", number_param_set)
        # print ("parameter_vector = ", parameter_vector)
        # print ("i7 = ", i7)
        # print ("nvec = ", nvec)


        ## reset output value
        chi2value = 0.0


        ## update parameter_set array
        if (number_param_set == 1):
            dummyVector = parameter_vector
            parameter_vector = []
            parameter_vector.append(dummyVector)
        parameter_vector_set = self.parameter_set
        for k in range(number_param_set):
            j = (-1)
            for i in range(self.NumberParameter):
                if (self.parameter_set[1][i] == 1):
                    j += 1
                    parameter_vector_set[k][i] = parameter_vector[k][j]


        ## write new parameter vectors to file
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        NewParamFile = open(WorkingDirectory.strip() + "new-parameters.dat", 'w')
        for k in range(number_param_set):
            j = (-1)
            line = ""
            for i in range(self.NumberParameter):
                if (self.parameter_set[1][i] == 1):
                    j += 1
                    line += "   " + str(parameter_vector[k][j])
                else:
                    line += "   " + str(self.parameter_set[0][i])
            NewParamFile.write(line + "\n")
        NewParamFile.close()

        # Debug:
        # print ("ModelFunctionCall.startcall.__doc__ = ", ModelFunctionCall.startcall.__doc__)
        # print ("self.printflagNum = ", self.printflagNum)
        # print ("number_param_set = ", number_param_set)
        # print ("self.DeterminationChi2 = ", self.DeterminationChi2)
        # print ("self.PlotIteration = ",  self.PlotIteration)
        # print ("self.PlotType = ", self.PlotType)
        # print ("self.NumberInputFiles = ", self.NumberInputFiles)
        # print ("self.NumberOutputFiles = ", self.NumberOutputFiles)
        # print ("self.ParallelizationFlag = ", self.ParallelizationFlag)
        # print ("self.JobID = ", self.JobID)
        # print ("self.MaxInputLines = ", self.MaxInputLines)
        # print ("self.MaxParameter = ", self.MaxParameter)
        # print ("self.RenormalizedChi2 = ", self.RenormalizedChi2)


        ## define dummy arguments
        chilm = 1.0
        NumberOfFitAlgorithms = 1
        AlgCounter = 1


        ## call model function package
        ParamSetCounter = number_param_set
        numiter = self.CurrIter
        NameOfAlgorithm = "Model-Function_Call"
        calstatus, FitFunctionOut, Chi2Values, parmCopy, FinalParameterSet = FittingEngine.StartAlgPackage(NameOfAlgorithm, self.StarterExecutable, \
                                                                                        self.MPIFlag, self.MAGIXrootDirectory, self.JobDir, \
                                                                                        self.JobID, self.NumberExpFiles, self.MaxLength, \
                                                                                        self.MaxColX, self.MaxColY, self.MaxRangeNumber, \
                                                                                        self.NumberParameter, ParamSetCounter, \
                                                                                        self.LastAlgorithmNum, numiter, self.ParallelizationFlag, \
                                                                                        self.DeterminationChi2, self.PlotIteration, self.PlotType, \
                                                                                        self.RenormalizedChi2, AlgCounter, chilm, \
                                                                                        self.GeneralAlgorithmSettings, \
                                                                                        self.SpecialAlgorithmSettings, parameter_vector_set, \
                                                                                        self.xAxisLabel, self.yAxisLabel, self.zAxisLabel, \
                                                                                        self.SortFortranNum)
        self.CurrIter = 0                                                                   ## turn initialization flag off


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get calculated chi2 values from temp file
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        ResultFile = open(WorkingDirectory.strip() + "chi2Values.dat", 'r')
        ResultFileContents = ResultFile.readlines()
        ResultFile.close()
        ValuesOfChi2 = []
        for val in ResultFileContents:
            ValuesOfChi2.append(float(val))

        # Debug:
        # print ("ValuesOfChi2 = ", ValuesOfChi2)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check if parameter vector is within given range
        for i in range(number_param_set):                                                  ## loop over parameter sets
            k = (-1)
            for j in range(self.NumberParameter):                                          ## loop over all parameter
                if (self.parameter_set[1][j] == 1):
                    k += 1
                    value = float(parameter_vector[i][k])
                    if (value < float(self.parameter_set[2][j]) or value > float(self.parameter_set[3][j])):
                        ValuesOfChi2[i] = numpy.inf
                        FitFunctionOut[i] = 0.e0
                        Chi2Values = 0.e0


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define complete list of chi2 values
        AllChi2Values = []
        RealNumParamVec = len(ValuesOfChi2)
        ParamVector = []
        for k, chi2value in enumerate(ValuesOfChi2):
            if (chi2value != numpy.inf):
                lll = [x for x in parameter_vector[k]]
                if (not (lll in ParamVector)):
                    AllChi2Values.append([chi2value, k])
                    ParamVector.append(lll)
        for k, chi2value in enumerate(self.BestChi2):
            AllChi2Values.append([chi2value, RealNumParamVec + k])

        # Debug:
        # print ("\nAllChi2Values = ", AllChi2Values)


        ## sort list of chi2 values
        AllChi2Values = sorted(AllChi2Values, key=lambda l:l[0])

        # Debug:
        # print ("Sorted: AllChi2Values = ", AllChi2Values)


        ## make a copy of the current best site variables
        OldBestParameter = copy.deepcopy(self.BestParameter)
        OldBestFunctionValues = copy.deepcopy(self.BestFunctionValues)
        OldBestChi2Values = copy.deepcopy(self.BestChi2Values)
        OldBestInputFiles = copy.deepcopy(self.BestInputFiles)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store parameter vectors and function values for best sites
        for i in range(self.INSBestSiteCounter):
            LocalChi2Value = AllChi2Values[i][0]
            LocalChi2Index = AllChi2Values[i][1]

            # Debug:
            # print ("\nLocalChi2Value = ", LocalChi2Value)
            # print ("LocalChi2Index = ", LocalChi2Index)
            # print ("number_param_set = ", RealNumParamVec)
            # print ("LocalChi2Index - RealNumParamVec = ", (LocalChi2Index - RealNumParamVec))


            ## identify best chi2 values
            if (LocalChi2Index < RealNumParamVec):
                self.BestChi2[i] = LocalChi2Value
                self.BestParameter[i] = copy.deepcopy(parameter_vector[LocalChi2Index])
                self.BestFunctionValues[i] = copy.deepcopy(FitFunctionOut[LocalChi2Index])
                self.BestChi2Values[i] = copy.deepcopy(Chi2Values[LocalChi2Index])


                ## get corresponding line with formatted parameters
                if (i == 0):
                    f = open(WorkingDirectory.strip() + "log_lines__single_call__" + str(LocalChi2Index + 1) + ".dat", 'r')
                    self.BestLogLine = f.readline()
                    f.close()
                    self.BestLogLine = self.BestLogLine[0:len(self.BestLogLine)-1]


                ## write best model function values to temp file
                for j in range(self.NumberExpFiles):
                    f = open(WorkingDirectory.strip() + "best_model_function_values_call__" + str(i + 1) + "__" + str(j + 1) + ".dat", 'w')
                    for lines in self.BestFunctionValues[i][j]:
                        f.write(str(lines[0]) + "\n")
                    f.close()


                ## write chi2 values to temp file
                for j in range(self.NumberExpFiles):
                    f = open(WorkingDirectory.strip() + "best_chi2_values_call__" + str(i + 1) + "__" + str(j + 1) + ".dat", 'w')
                    for lines in self.BestChi2Values[i][j]:
                        f.write(str(lines[0]) + "\n")
                    f.close()


                ## get corresponding contents of the input file(s)
                f = open(WorkingDirectory.strip() + "log_input-files__single_call__" + str(LocalChi2Index + 1) + ".dat", 'r')
                fcontentLocal = f.readlines()
                f.close()
                self.BestInputFiles[i] = fcontentLocal
            else:
                k = LocalChi2Index - RealNumParamVec
                self.BestChi2[i] = LocalChi2Value
                self.BestParameter[i] = copy.deepcopy(OldBestParameter[k])
                self.BestFunctionValues[i] = copy.deepcopy(OldBestFunctionValues[k])
                self.BestChi2Values[i] = copy.deepcopy(OldBestChi2Values[k])
                self.BestInputFiles[i] = copy.deepcopy(OldBestInputFiles[k])


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## append content of chi2-log file to total list of all calculated chi^2 values
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        for i in range(number_param_set):
            if (ValuesOfChi2[i] != numpy.inf):
                f = open(WorkingDirectory.strip() + "log_chi2_single_call.dat", 'r')
                contentChi2LogFile = f.readlines()
                for line in contentChi2LogFile:
                    self.Chi2Channel.write(line)
                    self.Chi2Channel.flush()
                f.close()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print what you do
        self.FuncCallCounter += 1                                                           ## increase counter for model function calls
        if (self.printflagNum == 1):
            if (self.FuncCallCounter % 21 == 1 and self.FuncCallCounter > 1):
                command_string = "echo -en \"\r                                                                                                  \r\""
                os.system(command_string)
            else:
                command_string = "echo -n \".\""
                os.system(command_string)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define return variable
        if (number_param_set == 1):
            chi2value = ValuesOfChi2[0]
        else:
            chi2value = ValuesOfChi2


        ## define return parameter
        return chi2value
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get best result
    def GetBestResult(self):
        """
        input variables:    None

        output variables:   self.BestChi2:          best chi^2 value
                            self.BestParameter:     best parameter vector
                            self.BestLogLine:       corresponding log line
                            self.BestInputFiles:    corrsponding input file
        """


        ## define return parameters
        return self.BestChi2, self.BestParameter, self.BestLogLine, self.BestInputFiles
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get function values of best result
    def GetBestFunctionValues(self):
        """
        input variables:    None

        output variables:   None
        """


        ## define return parameters
        return self.BestParameter, self.BestFunctionValues, self.BestChi2Values
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Interval Nested Sampling Algorithm for global optimization in multidimensional space
def Interval_Nested_Sampling(Internal_param, GeneralAlgorithmSettings, printflagNum, LastAlgorithmNum, ParamSetCounter, chilm, numiter, INSCounter, \
                             DeterminationChi2, PlotIteration, PlotType, fit_log, NumberInputFiles, NumberOutputFiles, ParallelizationFlag, JobID, \
                             MaxInputLines, MaxParameter, RenormalizedChi2, currentpath, CalculationMethod, xAxisLabel, yAxisLabel, zAxisLabel, \
                             PathStartScript, ExeCommandStartScript, parameter_set, ExpDataX, ExpDataY, ExpDataError, NumberRanges, MinRange, MaxRange, \
                             NumberXColumns, NumberYColumns, LengthExpRange, MaxRangeNumber, NumberExpFiles, MaxLength, MaxColX, MaxColY, \
                             NumberParameter, plotflag, modelflag, MPIFlag, MAGIXrootDirectory, JobDir, SortFortranNum):
    """
    input variables:            Internal_param:         algorithm user setttings
                                GeneralAlgorithmSettings:   special algorithm settings
                                printflagNum:           flag for screen output 1 (=yes) or 0 (=no)
                                LastAlgorithmNum:       number of last algorithm
                                ParamSetCounter:        number of best sites
                                chilm:                  user defined abort criteria for chi**2
                                numiter:                max. number of iterations
                                INSCounter:             counts number of calls
                                DeterminationChi2:      method being used for the determination of chi^2
                                PlotIteration:          plot model function for each iteration set 1(=yes) or 0(=no)
                                PlotType:               get type of plot
                                fit_log:                path for log-file containing the current values of chi**2
                                NumberInputFiles:       number of input files for the external model program
                                NumberOutputFiles:      number of output files for the external model program
                                ParallelizationFlag:    contains the number of processors used for parallelization
                                JobID:                  job identification number
                                MaxInputLines:          max number of lines in an input file
                                MaxParameter:           max number of parameters in a line of an input file
                                RenormalizedChi2:       flag for using renormalized chi**2
                                currentpath:            path of the working directory
                                CalculationMethod:      method of computation (at once or point-to-point)
                                xAxisLabel:             label of the x-axis (for plot)
                                yAxisLabel:             label of the y-axis (for plot)
                                zAxisLabel:             label of the z-axis (for plot)
                                PathStartScript:        path and name of the start script for calling model function
                                ExeCommandStartScript:  command for calling model function
                                parameter_set:          the complete set of paramters (incl. flags and limits)
                                ExpDataX:               array containing the experimental x side
                                ExpDataY:               array containing the experimental y side
                                ExpDataError:           array containing the experimental error of the y side
                                NumberRanges:           number of ranges
                                MinRange:               array containing lower limit of each range for each file
                                MaxRange:               array containing upper limit of each range for each file
                                NumberXColumns:         number x-columns
                                NumberYColumns:         number y-columns
                                LengthExpRange:         total number of data points
                                MaxRangeNumber:         max number of ranges over all files
                                NumberExpFiles:         number of exp. data files
                                MaxLength:              max. length of experimental data
                                MaxColX:                number of columns concerning to the experimental x side
                                MaxColY:                number of columns concerning to the experimental y side
                                NumberParameter:        number of model parameter
                                plotflag:               flag for plotting histograms
                                modelflag:              flag to use optimized MAGIX packages for a certain model or not
                                MPIFlag:                mpi flag
                                MAGIXrootDirectory:     MAGIX root directory
                                JobDir:                 job directory
                                SortFortranNum:         flag for sorting chi2 log file

    output variables:           calstatus:              status flag of calculation (= 0: all ok)
                                FitFunctionValues:      values of the model function of the last function call
                                Chi2Values:             corresponding chi^2 values
                                parameter_set:          compelete parameter set
    """

    # Debug:
    # print ("Internal_param = ", Internal_param)
    # print ("GeneralAlgorithmSettings = ", GeneralAlgorithmSettings)
    # print ("printflagNum = ", printflagNum)
    # print ("LastAlgorithmNum = ", LastAlgorithmNum)
    # print ("ParamSetCounter = ", ParamSetCounter)
    # print ("chilm = ", chilm)
    # print ("numiter = ", numiter)
    # print ("INSCounter = ", INSCounter)
    # print ("DeterminationChi2 = ", DeterminationChi2)
    # print ("PlotIteration = ", PlotIteration)
    # print ("PlotType = ", PlotType)
    # print ("fit_log = ", fit_log)
    # print ("NumberInputFiles = ", NumberInputFiles)
    # print ("NumberOutputFiles = ", NumberOutputFiles)
    # print ("ParallelizationFlag = ", ParallelizationFlag)
    # print ("JobID = ", JobID)
    # print ("MaxInputLines = ", MaxInputLines)
    # print ("MaxParameter = ", MaxParameter)
    # print ("RenormalizedChi2 = ", RenormalizedChi2)
    # print ("currentpath = ", currentpath)
    # print ("CalculationMethod = ", CalculationMethod)
    # print ("xAxisLabel = ", xAxisLabel)
    # print ("yAxisLabel = ", yAxisLabel)
    # print ("zAxisLabel = ", zAxisLabel)
    # print ("PathStartScript = ", PathStartScript)
    # print ("ExeCommandStartScript = ", ExeCommandStartScript)
    # print ("parameter_set = ", parameter_set)
    # print ("ExpDataX = ", ExpDataX)
    # print ("ExpDataY = ", ExpDataY)
    # print ("ExpDataError = ", ExpDataError)
    # print ("NumberRanges = ", NumberRanges)
    # print ("MinRange = ", MinRange)
    # print ("MaxRange = ", MaxRange)
    # print ("NumberXColumns = ", NumberXColumns)
    # print ("NumberYColumns = ", NumberYColumns)
    # print ("LengthExpRange = ", LengthExpRange)
    # print ("MaxRangeNumber = ", MaxRangeNumber)
    # print ("NumberExpFiles = ", NumberExpFiles)
    # print ("MaxLength = ", MaxLength)
    # print ("MaxColX = ", MaxColX)
    # print ("MaxColY = ", MaxColY)
    # print ("NumberParameter = ", NumberParameter)
    # print ("plotflag = ", plotflag)
    # print ("modelflag = ", modelflag)
    # print ("MPIFlag = ", MPIFlag)
    # print ("MAGIXrootDirectory = ", MAGIXrootDirectory)
    # print ("JobDir = ", JobDir)
    # print ("SortFortranNum = ", SortFortranNum)


    ##====================================================================================================================================================
    ##
    ##  subroutine for Interval nested sampling algorithm
    ##
    ##====================================================================================================================================================


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Checking of point from the list Vectors (evaluated or not yet): if -yes- that don't evaluate (function value from the list -Points-)
    def check_point_paral(Vectors):
        """
        input variables:        Vectors:                list of parameter vectors of dimension (num_par+1, nvec)
                                                            nvec --> number of parameter vectors;
                                                            without parallelization nvec=1 --> only 1 parameter vector in the list <Vectors>

                                                            for each vectors:
                                                            (0:num_par) --> parameter values
                                                            (num_par+1)--> function values (in input is 0.0)

        output variables:       Vectors:                list of parameter vectors of dimension (num_par+1,nvec)
                                                            nvec --> number of parameter vectors;
                                                            (0:num_par) --> parameter values
                                                            (num_par+1)--> calculated function values
        """

        # Debug:
        # print ("Vectors = ", Vectors)


        ## Points - list of already evaluated points
        np = len(Points)                                                                    ## number points, for which we have already calculated
                                                                                            ## function values
        nvec = len(Vectors)                                                                 ## number of points in the list <Vectors>
        Points_paral = []                                                                   ## form a list for parallel calculation of optimization
                                                                                            ## function
        # Debug:
        # print ("nvec = ", nvec


        for i7 in range(nvec):
            punkt1 = Vectors[i7]
            ii = (-1)
            for i in range(np):
                pp = Points[i]
                mi = 0
                for j in range(num_par):

                    # Debug:
                    # if (numpy.isnan(pp[j])):
                    #     print ("i7, i, punkt1, pp = ", i7, i, punkt1, pp)


                    if (punkt1[j] == pp[j]):
                        mi += 1
                if (mi == num_par):
                    ii = i
                    punkt1[num_par] = pp[num_par]
                    Vectors[i7][num_par] = punkt1[num_par]


            if (ii < 0):
                Points_paral.append(punkt1)


                ### without parallelization to use:
                punkt1[num_par] = modelClass.logLhood(1, punkt1[0:num_par], i7, nvec)
                Vectors[i7][num_par] = punkt1[num_par]
                Points.append(punkt1)


        ## for further parallelization to use list of parameter vectors <Points_paral> and than update list <Vectors> (see above) !!!
        return (Vectors)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Create histigramms for each parameter after Error estimation using matplotlib
    def my_hist1(para, pud, mp_val, CurrParamIndex, TotalNumFreeParam, fit_log, INSCounter, plotflag):
        """
        input variables:        para:                   set of parameter values after ErrorEstimator
                                pud:                    parameter space for creating of histogram
                                mp_val:                 parameter value at the point of minimum after first optimization procedure (LM, SA ets.)
                                CurrParamIndex:         current index of free parameter
                                TotalNumFreeParam:      total number of free parameter
                                fit_log:                path and name of log-files
                                INSCounter:             counter for algorithm calling
                                plotflag:               flag for plotting

        output variables:       none
        """

        # Debug:
        # print ("para = ", para)
        # print ("pud = ", pud)
        # print ("mp_val = ", mp_val)
        # print ("CurrParamIndex = ", CurrParamIndex)
        # print ("TotalNumFreeParam = ", TotalNumFreeParam)
        # print ("fit_log = ", fit_log)
        # print ("INSCounter = ", INSCounter)
        # print ("plotflag = ", plotflag)


        ## check plotflag
        if (plotflag == "false"):
            return


        ## chose type of histograms
        PlotType = "alone"


        ## define variables
        num_interval = 30                                                                   ## set number of intervals to 30 (why 30?)
        npoint = len(para)                                                                  ## get number of parameters


        ## sort array para containing parameter values
        para.sort()


        ## draw all histograms in one plot
        if (PlotType == "alltogether"):


            ## create plot for all histogramms
            if (CurrParamIndex == 0):
                pylab.subplots_adjust(hspace = 0.3, wspace=0.2, left=0.14, bottom=0.10, right=0.95, top=0.93)
                ax1 = pylab.subplot((TotalNumFreeParam + 1),1,1)
            else:
                ax1 = pylab.subplot(TotalNumFreeParam,1,(CurrParamIndex + 1))
            pylab.clf()


            ## create the histogram for the current free parameter
            # n, bins, patches = ax1.hist(para, num_interval, normed = 1, facecolor = 'green', alpha = 0.75, label = str(CurrParamIndex + 1))
            n, bins, patches = ax1.hist(para, num_interval, color = 'green', label = str(CurrParamIndex + 1))

            # Debug:
            # print ("n = ", n)
            # print ("bins = ", bins)
            # print ("patches = ", patches)


            ## define label for y-axis
            y1lab = 'weight of \n param ' + str(CurrParamIndex + 1)
            ax1.set_ylabel(y1lab)


            ## set range of x-axis and add grid
            ax1.set_xlim(pud[0], pud[1])
            ax1.grid(True)


            ## if last free parameter is reached add label to x-axis and save histograms to file
            if (CurrParamIndex == (TotalNumFreeParam - 1)):
                ax1.set_xlabel('Parameter range')                                           ## add label to x-axis


                ## save histograms to file
                fit_log = fit_log.strip()
                i = fit_log.rfind("/")
                if (i > (-1)):
                    path_to_save_png = fit_log[:i + 1]
                else:
                    path_to_save_png = ""
                if (plotflag == "true"):
                    pylab.savefig(path_to_save_png + "INS__Histograms_of_all-free-parameter__call_" + str(abs(INSCounter)) + ".png", format='png')
                    pylab.show()
                elif (plotflag == "saveonly"):
                    pylab.savefig(path_to_save_png + "INS__Histograms_of_all-free-parameter__call_" + str(abs(INSCounter)))


            ## construct ascii file for histogram data
            histascii_file = open(path_to_save_png + "INS__Histograms_of_free-parameter_parm__" + str(CurrParamIndex + 1) \
                                                   + "__call_" + str(abs(INSCounter)) + ".dat", 'w')
            count = -1
            for i in n:
                count += 1
                histascii_file.write(str(bins[count]) + "   " + str(i) + "\n")
            histascii_file.close()


        ## draw each histogram in a separate file
        elif (PlotType == "alone"):


            ## create plot for all histogramms
            ax1 = pylab.subplot(1, 1, 1)
            pylab.clf()


            ## create the histogram for the current free parameter
            #    n, bins, patches = pylab.hist(para, num_interval, normed = 1, facecolor = 'green', alpha = 0.75, \
            #                                  label = 'parameter ' + str(CurrParamIndex + 1))
            n, bins, patches = pylab.hist(para, num_interval, color = 'green', label = 'parameter ' + str(CurrParamIndex + 1))

            # Debug:
            # print ("n = ", n)
            # print ("bins = ", bins)
            # print ("patches = ", patches)


            ## define label for y-axis
            y1lab = 'propotional weight'
            pylab.ylabel(y1lab)


            ## set range of x-axis and add grid
            pylab.xlim(pud[0], pud[1])
            pylab.grid(True)


            ## add label to x-axis and add legend
            pylab.xlabel('Parameter range')
            pylab.legend(fancybox = False, shadow = False, loc = 1)


            ## save histograms to file
            fit_log = fit_log.strip()
            i = fit_log.rfind("/")
            if (i > (-1)):
                path_to_save_png = fit_log[:i + 1]
            else:
                path_to_save_png = ""
            if (plotflag == "true"):
                pylab.savefig(path_to_save_png + "INS__Histograms_of_free-parameter_parm_" + str(CurrParamIndex + 1) + "__call_" \
                                               + str(abs(INSCounter)) + ".png", format='png')
                pylab.show()
            elif (plotflag == "saveonly"):
                pylab.savefig(path_to_save_png + "INS__Histograms_of_free-parameter_parm_" + str(CurrParamIndex + 1) + "__call_" + str(abs(INSCounter)) )


            ## construct ascii file for histogram data
            histascii_file = open(path_to_save_png + "INS__Histograms_of_free-parameter_parm_" + str(CurrParamIndex + 1) + "__call_" \
                                                   + str(abs(INSCounter)) + ".dat", 'w')
            count = -1
            for i in n:
                count += 1
                histascii_file.write(str(bins[count]) + "   " + str(i) + "\n")
            histascii_file.close()


        ## the histogram need not be shown to the user, but saved in different files
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Calculation of quantiles of chi2-distribution on dependence of the number of free parameter (np) for 1-sigma, 2-sigma and 3-sigma contours
    def del_chi2(np):
        """
        input variables:        np:                     number of free parameter

        output variables:       delta_chi2:             set of quantiles of chi2-distribution for 1-sigma (68.3%), 2-sigma(95.5%) and 3-sigma(99.7%)
                                                        contours

        Source: "http://en.wikipedia.org/wiki/68-95-99.7_rule", call 18.11.2011
        """

        # Debug:
        # print ("np = ", np)


        alfa = [0.683, 0.955, 0.997]                                                        ## probabilities for definition of the confidence levels
        delta_chi2 = numpy.array(list(range(3)), dtype = 'float')
        for i in range(3):
            a1 = numpy.log(1.0/(1.0 - alfa[i]))
            a2 = (a1 - 0.16)**0.4274
            d = 2.0637 * a2 - 1.5774
            d0 = 2**0.5
            A = d * d0
            d1 = d * d - 1.0
            B = (2.0 * d1)/3.0
            d2 = d * d - 7
            C = (d * d2)/(9.0 * d0)
            d3 = 6.0 * d**4 + 14 * d * d - 32
            D = d3/405.0
            d4 = 9 * d**4 + 256 * d * d - 433
            E = d * d4/(4860 * d0)
            np0 = np**0.5
            delta_chi2[i] = np + A * np0 + B + (C/np0) + (D/np) + E/(np * np0)


        ## define return parameters
        return (delta_chi2)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Get information about error for each parameter --> (0:num_par)
    def error_info_new(param, Paramset, numpo):
        """
        input variables:        param:                  point of minimum (after INS algorithm)
                                Paramset:               set of parameter values with function values and proportional weights
                                numpo:                  number of vectors in <Paramset>

        output variables:       left                    error from the left side of parameter value at the point of minimum
                                right                   error from the right side of parameter value at the point of minimum
                                mean_val:               vector of mean values for parameters
                                dev:                    vector of standard deviations
        """

        # Debug:
        # print ("param = ", param)
        # print ("Paramset = ", Paramset)
        # print ("numpo = ", numpo)


        ## define variables num_par
        num_par = len(param) - 1


        ## calculation of chi2-contours (1sigma, 2sigma, 3sigma)
        delta_chi2 = del_chi2 (1)
        param_err = numpy.array(list(range(numpo*(num_par+3))), dtype = 'float')
        param_err.shape = (numpo, (num_par+3))
        param_prom = numpy.array(list(range(num_par+3)), dtype = 'float')

        # param_err: (0:num_par) - parameter values, (num_par) - function value,
        #            (num_par+1) - weights, (num_par+2) - distance from param

        for j in range(numpo):
            dist = 0
            for k in range(num_par):
                dist = dist + (Paramset[j,k] - param[k])**2
            param_err[j,0:(num_par+2)] = Paramset[j,0:(num_par+2)]
            param_err[j,num_par+2] = dist
        for j in range(numpo):
            for k in range(j,(numpo)):
                if (param_err[k,num_par+2] < param_err[j,num_par+2]):
                    param_prom[:] = param_err[j,:]
                    param_err[j,:] = param_err[k,:]
                    param_err[k,:] = param_prom[:]


        ## Calculate left and right errors from information about mean and stdev
        left = numpy.array(list(range(num_par)), dtype = 'float')
        right = numpy.array(list(range(num_par)), dtype = 'float')
        left[:] = 0.0
        right[:] = 0.0


        ## Calculate mean value and standard deviations for each parameter
        mean_val = numpy.array(list(range(num_par)), dtype = 'float')
        dev = numpy.array(list(range(num_par)), dtype = 'float')
        mean_val[:] = 0.0
        dev[:] = 0.0


        ## form list <Min_points> for error estimation; get all points with delta_chi2 < 3sigma contour
        cont = 0
        ncont = 2
        nconira = 0
        while (cont == 0):
            Min_points = []
            mett_contur = 0
            ii = 0
            while (mett_contur == 0 and ii < numpo):
                if (param_err[ii,num_par] <= delta_chi2[ncont]):
                    Min_points.append(param_err[ii,:])
                    ii += 1
                else:
                    mett_contur = 1
            if (len(Min_points) > 3):
                cont = 1
            else:
                if (ncont < 2 and nconira < 2):
                    delta_chi2[ncont] = 2.0 * delta_chi2[ncont]
                    nconira += 1
                else:
                    print ("\n\n\t   WARNING: The statistical sample for error estimation is too small.")
                    print ("\t            Increase the number of iterations and")
                    print ("\t            restart the MAGIX.\n")

                    # Debug:
                    # print ("left = ", left)
                    # print ("right = ", right)
                    # print ("mean_val = ", mean_val)
                    # print ("dev = ", dev)

                    return (left, right, mean_val, dev)


        ## define how many points we have for error estimation
        nminpo = len(Min_points)


        ## calculate mean value and standard deviation for each parameter
        for j in range(num_par):
            sum_wes = 0.0
            mean_val[j]=0.0
            for i in range(nminpo):
                mean_val[j] += Min_points[i][j] * Min_points[i][num_par+1]
                sum_wes += Min_points[i][num_par+1]


            ## define mean_val --> mean value for each parameter from the list <Min_points>
            if (sum_wes != 0.0):
                mean_val[j] = mean_val[j]/sum_wes


            ## calculate standard deviation  for the sample points from the list <Min_points>
            for i in range(nminpo):
                dev[j] += (Min_points[i][j] - mean_val[j])**2.0


            ## check the number of points for error estimation
            if ((nminpo-1) != 0):
                dev[j] = (dev[j]/(nminpo - 1))**0.5
            else:
                dev[j] = (dev[j])**0.5


            ## Left and right errors
            ## if mean value is equal parameter value at the point of minimum
            if (mean_val[j] == param[j]):
                left[j] = dev[j]
                right[j] = dev[j]
            if ((mean_val[j] >= 0.0 and param[j] >= 0.0) or (mean_val[j] < 0.0 and param[j] < 0.0)):
                rast = abs(mean_val[j] - param[j])
            if ((mean_val[j] >= 0.0 and param[j] < 0.0) or (mean_val[j] < 0.0 and param[j] >= 0.0)):
                rast = abs(mean_val[j]) + abs(param[j])


            ## if mean value is from left side of the parameter value at the point of minimum
            if (mean_val[j] < param[j]):
                if (dev[j] > rast):
                    right[j] = dev[j] - rast
                    left[j] = dev[j] + rast
                else:
                    right[j] = 0.0
                    left[j] = 2.0 * dev[j]


            ## if mean value is from right side of the parameter value at the point of minimum
            if (mean_val[j] > param[j]):
                if (dev[j] > rast):
                    right[j] = dev[j] + rast
                    left[j] = dev[j] - rast
                else:
                    left[j] = 0.0
                    right[j] = 2.0 * dev[j]


            ## check if (min-error_left; min+error_right) lies in the parameter space
            #if ((min_point[i] - left[i]) < param_up_down[i, 0]):                      ## check if Error_left is smaller than lower limit
            #    left[i] = abs(min_point[i] - param_up_down[i, 0])
            #if ((min_point[i] + right[i]) > param_up_down[i, 1]):                     ## check if Error_right is larger than upper limit
            #    right[i] = abs(param_up_down[i, 1] - min_point[i])

        #print mean_val
        #print dev
        #a33=input()

        ## define return (output) variables
        return (left, right, mean_val, dev)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Bisection of parametric space (par_space) for k-coordinate direction
    def bisection(k, par_space):
        """
        input variables:        k:                      direction of bisection
                                par_space:              parental interval box in form (lower_limit, upper_limit)

        output variables:       par_space1:             interval box1 after bisection in form (lower_limit, upper_limit)
                                par_space2:             interval box2 after bisection in form (lower_limit, upper_limit)
        """

        # Debug:
        # print ("k = ", k)
        # print ("par_space = ", par_space)


        np = len(par_space)                                                                 ## Dimension of space
        k = int(k)
        par_space1 = numpy.array(list(range(np * 2)), dtype = 'float')
        par_space1.shape = (np, 2)
        par_space2 = numpy.array(list(range(np * 2)), dtype = 'float')
        par_space2.shape = (np, 2)
        bi_point = (par_space[k, 1] + par_space[k, 0]) / 2.0
        for i in range(np):
            par_space1[i,0] = par_space[i,0]
            par_space2[i,1] = par_space[i,1]
            if (i == k):
               par_space1[i,1] = bi_point
               par_space2[i,0] = bi_point
            else:
               par_space1[i,1] = par_space[i,1]
               par_space2[i,0] = par_space[i,0]


        ## define return parameters
        return (par_space1, par_space2)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Division of num_par-dimension parameter space to subdomains with bisection
    ## If num_par>5 then bisection on the five the broad coordinate directions !
    def interval_division(num_par, param_up_down):
        """
        input variables:        num_par:                ??
                                param_up_down:          ??

        output variables:       L:                      ??
        """

        # Debug:
        # print ("num_par = ", num_par)
        # print ("param_up_down = ", param_up_down)


        ## if dimension of parameter space < 5, then bisect the parameter space in each parameters direction
        L = []
        if (num_par < 5):
            L1 = []
            pud1, pud2 = bisection (0,param_up_down)
            L1.append(pud1)
            L1.append(pud2)
            j = 1
            ll2 = 0
            if (num_par==1): L=L1
            while (j < num_par):
                  ll1 = len(L1)
                  if (j == num_par - 1):
                     for i in range(ll2,ll1):
                         par_space = L1[i]
                         pud1, pud2 = bisection(j,par_space)
                         L.append(pud1)
                         L.append(pud2)
                     j += 1
                  else:
                     for i in range(ll2,ll1):
                         par_space = L1[i]
                         pud1, pud2 = bisection(j,par_space)
                         L1.append(pud1)
                         L1.append(pud2)
                     j += 1
                     ll2 = ll1
        else:
            ## if dimension of parameter space > 5, then bisect the parameter space in five the widest coordinate directions
            wid = numpy.array(list(range(num_par)),dtype = 'float')
            coord_dir = numpy.array(list(range(num_par)), dtype = 'float')
            for i in range(num_par):
                wid[i] = abs(param_up_down[i, 1] - param_up_down[i, 0])

            for i in range(num_par):
                for j in range(i,num_par):
                    if (wid[i] < wid[j]):
                        prom = coord_dir[i]
                        coord_dir[i] = coord_dir[j]
                        coord_dir[j] = prom
                        prom = wid[i]
                        wid[i] = wid[j]
                        wid[j] = prom


            L1 =[]
            pud1, pud2 = bisection(coord_dir[0], param_up_down)
            L1.append(pud1)
            L1.append(pud2)
            j = 1
            ll2 = 0
            while (j < 5):
                  ll1 = len(L1)
                  if (j == 4):
                     for i in range(ll2,ll1):
                         par_space = L1[i]
                         pud1, pud2 = bisection(coord_dir[j], par_space)
                         L.append(pud1)
                         L.append(pud2)
                     j += 1
                  else:
                     for i in range(ll2,ll1):
                         par_space = L1[i]
                         pud1, pud2 = bisection(coord_dir[j], par_space)
                         L1.append(pud1)
                         L1.append(pud2)
                     j += 1
                     ll2 = ll1


        ## define return parameter
        return(L)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Choosing of coordinate for subdivision of parametric space (par_space): Rule A
    def coord_choice(par_space):
        """
        input variables:        par_space:              interval in form (lower_limit, upper_limit)

        output variables:       direction:              ??
        """

        # Debug:
        # print ("par_space = ", par_space)


        np = len(par_space)                                                                 ## Dimension of space
        dist = numpy.array(list(range(np)), dtype = 'float')
        direction = 0
        for i in range(np):
            dist[i] = abs(par_space[i,1] - par_space[i,0])

        for i in range(np):
            if (dist[i] == max(dist)):
               direction = i

        return (direction)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Calculation of inclusion function with use of interval arithmetics rules: return inclusion - interval F1 = (minimal value , maximal value)
    def inclusion_function(V1, direc, mid, mid1):
        """
        input variables:        V1:                     initial interval box
                                mid:                    center point of interval box
                                mid1:                   center point of parental interval box
                                direc:                  direction of bisection

        output variables:       F1:                     inclusion function in form of the interval F1 = (F1_min, F1_max)
        """

        # Debug:
        # print ("V1 = ", V1)
        # print ("direc = ", direc)
        # print ("mid = ", mid)
        # print ("mid1 = ", mid1)


        aa = V1[direc,0]
        bb = V1[direc,1]
        aa = aa - mid[direc]
        bb = bb - mid[direc]
        s1 = (mid1[num_par] - mid[num_par])/(mid1[direc] - mid[direc])
        s1_aa = s1 * aa
        s1_bb = s1 * bb
        F1 = [0, 0]
        F1[0] = mid1[num_par] + s1_aa
        F1[1] = mid1[num_par] + s1_bb
        if (F1[0] > F1[1]):
           cc = F1[0]
           F1[0] = F1[1]
           F1[1] = cc


        ## define return parameter
        return (F1)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Calculation of volume of interval box and return ratio (volume_box/whole_volume)
    def volume_box(box, pud):
        """
        input variables:        box:                    current interval box in form (lower_limit, upper_limit)
                                pud11:                  the whole parameter space in form (lower_limit, upper_limit)

        output variables:       volume_ratio:           ration of the volume of the current interval box to the whole volume of parameter space
        """

        # Debug:
        # print ("box = ", box)
        # print ("pud = ", pud)


        whole_volume = 1.0
        volume_box = 1.0
        for j in range(num_par):
            volume_box = volume_box * abs(box[j,1] - box[j,0])
            whole_volume = whole_volume * abs(pud[j,1] - pud[j,0])

        volume_ratio = volume_box/whole_volume


        ## define return parameter
        return (volume_ratio)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Calculation of center point of the interval par_space
    def calc_cen_point(par_space):
        """
        input variables:        par_space:              current interval box in form (lower_limit, upper_limit)
                                nvar:                   number of the variable parameter (from 0 to num_par)
                                min_point:              point of minimum after first optimization algorithm (LM, SA ets.)

        output variables:       cen_point:              center point of the current interval box <par_space>
        """

        # Debug:
        # print ("par_space = ", par_space)


        cen_point = numpy.array(list(range(num_par + 1)), dtype = 'float')
        for i in range(num_par):
            cen_point[i] = (par_space[i,1] + par_space[i,0])/2.0
        cen_point[num_par] = 0.0


        ## define return parameter
        return (cen_point)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Interval analysis for part of space - par_space
    def interval_method(par_space):
        """
        1. Choice of direction for division - def coord_choice
        2. Division of par_space on this direction - def bisection
        3. Finding of inclusion function in accordance with interval arithmetic rules

        input variables:        par_space:              current interval box

        output variables:       cen_point:              center point od the current interval box <par_space>
                                V1:                     first interval box after bisection of <par_space>
                                cen_point1:             center point of first interval box
                                V2:                     second interval box after bisection of <par_space>
                                cen_point2:             center point of second interval box
                                F1:                     inclusion function of V1 in form F1 = (F1_min, F1_max)
                                F2:                     inclusion function of V2 in form F2 = (F2_min, F2_max)
        """

        # Debug:
        # print ("par_space = ", par_space)


        ## calculate center point of <par_space>
        Centers = []
        cen_point = calc_cen_point(par_space)
        Centers.append(cen_point)


        ## choose of coordinate direction
        direc = coord_choice(par_space)


        ##bisect <par_space> in the direction: two new interval boxes V1 and V2
        V1, V2 = bisection(direc, par_space)


        ## calculate center point of V1
        cen_point1 = calc_cen_point(V1)
        Centers.append(cen_point1)


        ## calculate center point of V2
        cen_point2 = calc_cen_point(V2)
        Centers.append(cen_point2)


        ## check if we calculate for these points function values, if not yet: calculate the optimization function values
        Centers = check_point_paral(Centers)
        cen_point = Centers[0]
        cen_point1 = Centers[1]
        cen_point2 = Centers[2]


        ## calculate inclusion functions for interval boxes V1 and V2
        F1 = inclusion_function(V1, direc, cen_point, cen_point1)
        F2 = inclusion_function(V2, direc, cen_point, cen_point2)


        ## output results: center points for intervals, new intervals boxes and inclusion functions
        return (cen_point, V1, cen_point1, V2, cen_point2, F1, F2)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Function for calculation the initial number of objects and max_iter of NS (if these parameters are defined by default)
    def Nlive(num_par, pud):
        """
        input variables:        num_par:                ??
                                pud:                    ??

        output variables:       nob:                    ??
                                max_iter:               ??
        """

        # Debug:
        # print ("num_par = ", num_par)
        # print ("pud = ", pud)


        max_iter = 50                                                                       ## default number of NS-iterations
        if (num_par < 5):
            nob = 3 * num_par + 10
        else:
            nob = 3 * num_par
            mn = num_par - 5
            max_iter = max_iter + mn*5
            if (max_iter > 500):                                                            ## max_iter is less then 500 only (by default)
                max_iter = 500

        if (nob > 32):                                                                      ## the number of working objects is more then 32 (by default)
            nob = 32


        ## calculate index <par_lim> for various parameter spaces
        for i in range(num_par):
            if (pud[i,0] > 0.0 and pud[i,1] > 0.0):
               par_lim = pud[i,1]/pud[i,0]
            if (pud[i,0] == 0.0 and pud[i,1] > 0.0):
               par_lim = pud[i,1]
            if (pud[i,0] < 0.0 and pud[i,1] == 0.0):
               par_lim = abs(pud[i,0])
            if (pud[i,0] < 0.0 and pud[i,1] < 0.0):
               par_lim = pud[i,1]/pud[i,0]
            if (pud[i,0] < 0.0 and pud[i,1] > 0.0):
               par_lim = abs(pud[i,1] - pud[i,0])


            ## if parameter space is big
            if (par_lim >= 100.0 and par_lim < 1000.0) :
               #nob = nob + 30
               max_iter += 30


            ## if parameter space is very big
            if (par_lim >= 1000.0):
               #nob = nob + 50
               max_iter += 100


        ## it is probable values for definition of values for NS-proccess
        ## I specify these values from a number of numerical experiments
        return (nob, max_iter)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Evolve object within function constraint contents
    def explore(Obj, logLstar, Best, stop_crit):
        """
        input variables:        Obj:                    Object being evolved
                                logLstar:               best function value (for the moment)
                                Best:                   list of best intervals
                                stop_crit:              terminating criterion

        output variables:       ret:                    dictionary with best Object
                                stop_crit:              termination criteria of inclusion function (not used in ErrorEstimator!)
        """

        # Debug:
        # print ("Obj = ", Obj)
        # print ("logLstar = ", logLstar)
        # print ("Best = ", Best)
        # print ("stop_crit = ", stop_crit)


        ret = Object()
        ret.__dict__ = Obj.__dict__.copy()

        step = 0.1                                                                          ## Initial guess suitable step-size in (0,1)
        accept = 0                                                                          ## MCMC acceptances
        reject = 0                                                                          ## MCMC rejections
        Try = Object()                                                                      ## Trial object
        for m in range(2):                                                                  ## pre-judged number of steps
            L1 = []


            ## Trial object
            Try.prior = ret.prior + step * (2.0 * uniform() - 1.0)
            Try.prior = Try.prior - math.floor(Try.prior)
            if (len(L00) != 0):
                LL = L00[0]
                del L00[0]
            else:
                if (len(Best) != 0):
                    LL = Best[0]
                    del Best[0]


            ## -------------------------------------------------------------------------------------------------------------------------------------------
            ## Interval method for investigation of parameter space ##############
            ## <in_param> - list with internal parameters of INS algorithm:
            ## the list contents vol_bound and delta_incl
            if (Internal_param[1] != 0):                                                    ## internal parameters are defined by user
               vol_bound = Internal_param[1]
            else:                                                                           ## internal parameters are defined by default
               vol_bound = 0.1 * (1.0 - ((num_par - 0.75)/num_par)**0.5)
            if (Internal_param[2] != 0):                                                    ## internal parameters are defined by user
               delta_incl= Internal_param[2]
            else:                                                                           ## internal parameters are defined by default
               delta_incl = 0.001


            ## define <volume_ratio>
            volume_ratio = volume_box(LL[0], param_up_down)


            ## if <volume_ratio> is less then <vol_bound>, then I add the best interval box to the list <Best> and both of new interval boxes to the
            ## working list <LL>
            if (volume_ratio < vol_bound):


                ## apply interval method to one interval from the working list <LL>
                cen, V1, cen1, V2, cen2, F1, F2 = interval_method(LL[0])


                ## if function value new interval box V1 is less than box V2 and minimal value of inclusion function of V1 is better that for V2:
                ## add the best box V1 to the working list and to the list <Best>
                if (cen1[num_par] <= cen2[num_par] and F1[0] <= F2[0]):
                    L1.append(V1)
                    L1.append(cen1)
                    L00.append(L1)
                    Best.append(L1)


                    ## choosie a best point for Try.position
                    if (m == 0):
                        Try.position = [cen1[i] for i in range (num_par) ]
                        Try.logL = cen1[num_par]
                    if (m > 0 and Try.logL > cen1[num_par]):
                        Try.position = [cen1[i] for i in range (num_par) ]
                        Try.logL = cen1[num_par]


                    ## termination criterion
                    if (abs(F1[1] - F1[0]) < delta_incl):
                       stop_crit = 1


                if (cen1[num_par] > cen2[num_par] and F1[0] >= F2[0]):
                    L1.append(V2)
                    L1.append(cen2)
                    L00.append(L1)
                    Best.append(L1)


                    ## choosie a best point for Try.position
                    if (m == 0):
                        Try.position = [cen2[i] for i in range (num_par) ]
                        Try.logL = cen2[num_par]
                    if (m > 0 and Try.logL > cen2[num_par]):
                        Try.position = [cen2[i] for i in range (num_par) ]
                        Try.logL = cen2[num_par]
                    if (abs(F2[1] - F2[0]) < delta_incl):
                       stop_crit = 1
            else:


                ## if <volume_ratio> is more then <vol_bound>, then I add both of new interval boxes to the working list <LL> only
                cen, V1, cen1, V2, cen2, F1, F2 = interval_method(LL[0])
                if (cen1[num_par] == cen2[num_par]):
                    L1.append(V1)
                    L1.append(cen1)
                    L00.append(L1)


                    ## choosie a best point for Try.position
                    if (m == 0):
                        Try.position = [cen1[i] for i in range (num_par) ]
                        Try.logL = cen1[num_par]
                    if (m > 0 and Try.logL > cen1[num_par]):
                        Try.position = [cen1[i] for i in range (num_par) ]
                        Try.logL = cen1[num_par]
                    L2 = []
                    L2.append(V2)
                    L2.append(cen2)
                    L00.append(L2)


                if (cen1[num_par] < cen2[num_par]):
                    L1.append(V1)
                    L1.append(cen1)
                    L00.append(L1)


                    ## choosie a best point for Try.position
                    if (m == 0):
                        Try.position = [cen1[i] for i in range (num_par) ]
                        Try.logL = cen1[num_par]
                    if (m > 0 and Try.logL > cen1[num_par]):
                        Try.position = [cen1[i] for i in range (num_par) ]
                        Try.logL = cen1[num_par]
                    L2 = []
                    L2.append(V2)
                    L2.append(cen2)
                    L00.append(L2)


                if (cen1[num_par] > cen2[num_par]):
                    L1.append(V2)
                    L1.append(cen2)
                    L00.append(L1)


                    ## choosie a best point for Try.position
                    if (m == 0):
                        Try.position = [cen2[i] for i in range (num_par) ]
                        Try.logL = cen2[num_par]
                    if (m > 0 and Try.logL > cen2[num_par]):
                        Try.position = [cen2[i] for i in range (num_par) ]
                        Try.logL = cen2[num_par]
                    L2 = []
                    L2.append(V1)
                    L2.append(cen1)
                    L00.append(L2)


            ## ---------------------------------------------------------------------------
            ## Accept if and only if within hard likelihood constraint
            if (Try.logL < logLstar):
                 ret.__dict__ = Try.__dict__.copy()
                 accept += 1
            else:
                 reject += 1


            ## Refine step-size to let acceptance ratio converge around 50%
            if (accept > reject):
                step *= numpy.exp(1.0 / accept)
            if (accept < reject):
                step /= numpy.exp(1.0 / reject)


        ## define return parameters
        return (ret, stop_crit)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## logarithmic addition log(exp(x)+exp(y))
    def plus(x, y):
        """
        input variables:        x:                      floating point
                                y:                      floating point

        output variables:       log(exp(x)+exp(y))
        """

        # Debug:
        # print ("x = ", x)
        # print ("y = ", y)


        ## define return parameter
        if (x > y):
           return x + numpy.log(1 + numpy.exp(y - x))
        else:
           return y + numpy.log(1 + numpy.exp(x - y))
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## This is an implementation of John Skilling's Nested Sampling algorithm for computing the normalizing constant of a probability distribution
    ## (usually the posterior in Bayesian inference).
    def ndim_nest(n, explore, param_up_down, printflag, paramchannel, logchannel):
        """
        The return value is a dictionary with the following entries:

        input variables:        n:                      number of objects
                                explore:                subroutine for searching of the next best point in parameter space (uses the interval method)
                                param_up_down:          parameter space
                                printflag:              flag for screen output
                                paramchannel:           number of channel for parameter log-file
                                logchannel:             number of channel for log-file

        output variables:       samples:                a set of the rejected points from nested sampling process;
                                                            for these points we have proportional weights
                                num_iterations:         number of iterations of nested sampling process
                                logZ:                   logarithm of the Bayesian evidence
                                logZ_sdev:              standers deviation of log(Z)
                                Best:                   list with interval boxes
                                InputFile:              content of final input files
        """

        # Debug:
        # print ("n = ", n)
        # print ("explore = ", explore)
        # print ("param_up_down = ", param_up_down)
        # print ("printflag = ", printflag)
        # print ("paramchannel = ", paramchannel)
        # print ("logchannel = ", logchannel)


        ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        ## Initial List of working objects
        def loglike_nest(i):
            """
            input variables:    i:                      number of the object

            output variables:   Obj:                    interval box from the working list <LL>
            """

            # Debug:
            # print ("i = ", i)


            Obj = Object()
            Obj.prior = random.uniform(0,1)


            ## choose allready calculated <mid_point> - center points for intervals after initial bisection
            if (i == 0):
                Obj.position = [ mid_point0[j] for j in range(num_par) ]
                #Obj.logL = mid_point0[num_par]
                Obj.logL = 0.0

                # Debug:
                # for j in xrange(num_par):
                #     if (numpy.isnan(mid_point0[j])):
                #         print ("i,j,mid_point0[j] = ", i,j,mid_point0[j])


            ## if the number of points in Objects is not enough: add other points from the list <Points>
            if (i != 0 and i < len(Points)):
                LL = Points[i]
                Obj.position = [ LL[j] for j in range(num_par) ]
                #Obj.logL = LL[num_par]
                Obj.logL = 0.0

                # Debug:
                # for j in xrange(num_par):
                #     if (numpy.isnan(LL[j])):
                #         print ("i,j,LL[j] = ", i,j,LL[j])


            ## if the number of points in Objects is not enough: define next points with random number generator
            if (i >= len(Points)):
                Obj.prior = random.uniform(0,1)
                Obj.position = [(param_up_down[i,0] + (param_up_down[i,1] - param_up_down[i,0]) * Obj.prior)  for i in range(num_par) ]
                Obj.logL = 0.0
                #prom_point = check_point(Obj.position)
                #Obj.logL = prom_point[num_par]

                # Debug:
                # for j in xrange (num_par):
                #     val = (param_up_down[i,0] + (param_up_down[i,1] - param_up_down[i,0]) * Obj.prior)
                #     if (numpy.isnan(val)):
                #         print ("i,j, param_up_down[i,0], param_up_down[i,1], param_up_down[i,0], Obj.prior = ", \
                #                       i,j,param_up_down[i,0], param_up_down[i,1], param_up_down[i,0], Obj.prior)


            ## define return parameter
            return(Obj)


        ##================================================================================================================================================
        Obj = []                                                                            ## Collection of n objects
        Samples = []                                                                        ## Objects stored for posterior results
        logwidth = None                                                                     ## ln(width in prior mass)
        logLstar = None                                                                     ## ln(Likelihood constraint)
        H    = 0.0                                                                          ## Information, initially 0
        logZ =-DBL_MAX                                                                      ## ln(Evidence Z, initially 0)
        logZnew = None                                                                      ## Updated logZ
        copy = None                                                                         ## Duplicated object
        worst = None                                                                        ## Worst object
        nest = None                                                                         ## Nested sampling iteration count
        Prior_vectors = []
        bestinput = ""


        ## Set prior objects
        for i in range(n):
            Obj.append(loglike_nest(i))

            # Debug:
            # if (numpy.isnan(loglike_nest(i))):
            # print ("i,loglike_nest(i) = ",i,loglike_nest(i))


            prom_ira = numpy.array(list(range(num_par + 1)), dtype = 'float')
            for jj in range(num_par):
                prom_ira[jj] = Obj[i].position[jj]

                # Debug:
                # if (numpy.isnan(Obj[i].position[jj])):
                # print ("i, jj, Obj[i].position[jj], Obj[i].logL = ", i, jj, Obj[i].position[jj], Obj[i].logL)


            prom_ira[num_par] = Obj[i].logL
            Prior_vectors.append(prom_ira)

        Prior_vectors = check_point_paral(Prior_vectors)

        for ii in range(n):
            prom_ira = Prior_vectors[ii]
            Obj[ii].logL = prom_ira[num_par]

            # Debug:
            # if (numpy.isnan(prom_ira[num_par])):
            #     print ("ii, num_par, prom_ira[num_par] = ", ii, num_par, prom_ira[num_par])


        ## Outermost interval of prior mass
        logwidth = numpy.log(1.0 - numpy.exp(-1.0 / n))


        ## NESTED SAMPLING LOOP __________________________________________________________________________________________________________________________
        nest = 0
        Best = []
        stop_crit = 0
        chisq = 1.e99
        glavmet = 0
        while (nest < max_iter and chisq > chilim):


            ## determine worst index
            worst = 0
            for i in range(1,n):


                ## bug fix:
                if (Obj[i].logL == None):
                    Obj[i].logL = 0
                    Obj[i].position = [0.0 for j in range(num_par)]

                # Debug:
                # if (numpy.isnan(Obj[i].logL)):
                # print ("i, Obj[i].logL, Obj[i].position[:] = ", i, Obj[i].logL, Obj[i], Obj[i].position[:])


                ## determine worst index
                if (Obj[i].logL > Obj[worst].logL):
                   worst = i


            ## bug fix
            if (n == 0 and Obj[0].logL == None):
                Obj[0].logL = 0
                Obj[0].position = [0.0 for j in range(num_par)]


            ## calculate contour for the worst object (with big optimization function value)
            try:
                Obj[worst].logWt = logwidth - Obj[worst].logL

                # Debug:
                # if (numpy.isnan(Obj[worst].logWt)):
                #     print ("logwidth - Obj[worst].logL = ", logwidth - Obj[worst].logL)


            except TypeError as err:
                print ("\n\nError in subroutine Interval_Nested_Sampling:\n")
                print ("\t logwidth = ", logwidth)
                print ("\t Obj[worst].logL = ", Obj[worst].logL)
                print ("\t n = ", n)
                for i in range(1, n):
                   print ("\t i,Obj[i].logL = ",i,Obj[i].logL)
                sys.exit(0)


            ## Update Evidence Z and Information H
            logZnew = plus(logZ, Obj[worst].logWt)
            H = numpy.exp(Obj[worst].logWt - logZnew) * Obj[worst].logL + numpy.exp(logZ - logZnew) * (H + logZ) - logZnew
            logZ = logZnew


            ## Posterior Samples (optional)
            Samples.append(Obj[worst])


            ## Kill worst object in favour of copy of different survivor
            if (n > 1):                                                                     ## don't kill if n is only 1
               while True:
                     copy = int(n * uniform()) % n                                          ## force 0 <= copy < n
                     if (copy != worst):
                        break
            logLstar = Obj[worst].logL                                                      ## new likelihood constraint
            Obj[worst] = Obj[copy]                                                          ## overwrite worst object


            ## Evolve copied object within constraint
            updated, stop_crit = explore(Obj[worst], logLstar, Best, stop_crit)


            ## if <stop_crit> is reached and number of iteration <nest> is enough --> the loop is ended
            if (stop_crit == 1 and nest >= 5):
                glavmet = 1


            ## clean the working list <L00> each 5 iterations
            iira = (nest/5.0) - int(nest/5.0)
            if (nest != 0 and iira == 0):


                ## Clearning of the main list <L00>:
                new_len = int(len(L00)/2.0)
                nl = len(L00)
                for i in range(nl):
                    for j in range(i,nl):
                        L11 = []
                        if (L00[i][1][num_par] > L00[j][1][num_par]):
                            L11 = L00[i]
                            L00[i] = L00[j]
                            L00[j] = L11
                i = new_len + 1
                ir = len(L00) - i
                for i5 in range(ir):
                    del L00[i]


            ## print what you do
            bestchi, bestparam, bestline, bestinput = modelClass.GetBestResult()
            chisq = bestchi[0]
            if (printflag == "true"):
                print ("\r                %5d%s" % (nest + 1, bestline))


            ## wirte information to param-log file
            paramchannel.write("\n")
            paramchannel.write("\n")
            SepString = 122 * "="
            paramchannel.write(SepString + "\n")
            outputstring = "Iteration: %5d,  chi^2 = %25.15e" % (nest + 1, chisq)
            paramchannel.write(outputstring + "\n")
            paramchannel.write("\n")
            paramchannel.write("\n")
            paramchannel.write("\n")
            paramchannel.write("Parameters: " + str(bestparam[0]) + "\n")
            SepString = 122 * "-"
            paramchannel.write(SepString + "\n")
            paramchannel.write("\n")
            paramchannel.flush()


            ## write contents of input file(s) to param-log file
            for lineinput in bestinput[0]:
                paramchannel.write(lineinput)
                paramchannel.flush()


            ## wirte information to log file
            outputstring = "                %5d%s" % (nest + 1, bestline)
            logchannel.write(outputstring + "\n")
            logchannel.flush()
            assert(updated != None)                                                         ## Make sure explore didn't update in-place
            Obj[worst] = updated

            # Debug:
            # if (numpy.isnan(updated)):
            #   print ("updated = ", updated)


            ## Shrink interval
            logwidth -= 1.0 / n
            nest += 1


        ## Exit with evidence Z, information H, and optional posterior Samples
        sdev_H = H / numpy.log(2.0)
        if (H > 0):
            sdev_logZ = math.sqrt(H / n)
        else:
            sdev_logZ = 0.0
        nest0 = nest - 1
        if (nest0 < 1):
            nest0 = 1


        ## print reason for stop of iteration to screen and to log-file
        bestchi, bestparam, bestline, bestinput = modelClass.GetBestResult()
        paramchannel.write("\n")
        paramchannel.write("==========================================================================================================================\n")
        paramchannel.write("\n")
        paramchannel.flush()
        if (bestchi[0] < chilim):
            outputstring = "           Iteration stopped. chi^2 (=%25.15e) dropped below limit = %25.15e" % (bestchi[0], chilim)
            if (printflag == "true"):
                print ("\n" + outputstring)
            logchannel.write("\n")
            logchannel.write(outputstring + "\n")
            logchannel.flush()
        elif (len(L00) == 0):
            if (printflag == "true"):
                print ("\n           Iteration stopped. len(L00) == 0")
            logchannel.write("\n          Iteration stopped. len(L00) == 0\n")
            logchannel.flush()
        elif (nest >= max_iter):
            if (printflag == "true"):
                print ("\n           Iteration stopped. Number of iterations is equal to max. number of iterations = %6d" % max_iter)
            logchannel.write("\n           Iteration stopped. Number of iterations is equal to max. number of iterations = " + str(max_iter) + "\n")
            logchannel.flush()
        else:
            if (printflag == "true"):
                print ("\n           Iteration stopped because inclusion function is constant.")
            logchannel.write("\n          Iteration stopped because inclusion function is constant.\n")
            logchannel.flush()


        ## define return parameters
        return {"samples":Samples, "num_iterations":(nest0), "logZ":logZ, "logZ_sdev":sdev_logZ, "Best":Best, "InputFile":bestinput}
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    class Object:
          def __init__(self):
              self.prior = None                                                             ## a prior volume for self.position (multidimensional)
              self.position = None                                                          ## a set of parameters values (multidimensional)
              self.logL = None                                                              ## logLikelihood = ln Prob(data | position)
              self.logWt = None                                                             ## log(Weight), adding to SUM(Wt) = Evidence Z
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##====================================================================================================================================================
    ##
    ## Main-Routine:                    #################### Interval Nested Sampling algorithm ########################
    ##
    ##====================================================================================================================================================


    ## define printflag
    printflag = "true"
    if (printflagNum == 1):
        printflag = "true"
    else:
        printflag = "false"
    printflagNum = 2


    ## define num_par and param_up_down
    for i in range(NumberParameter):
        parameter_set[1][i] = float(parameter_set[1][i])
    num_par = int(sum(parameter_set[1]))                                                    ## determine num_par
    param_up_down = numpy.zeros( (num_par, 2), dtype = 'float')                             ## reset param_up_down
    j = (-1)
    for i in range(NumberParameter):
        if (parameter_set[1][i] == 1):
            j += 1
            param_up_down[j, 0] = parameter_set[2][i]
            param_up_down[j, 1] = parameter_set[3][i]


    ## print what you do
    if (printflag == "true"):
        print ("\n         temporary files are stored in: " + str(os.environ.get('MAGIXTempDirectory','')).strip() + "job_" + str(JobID).strip() + "/")
        print ("\n\n         Start Interval-Nested-Sampling algorithm ..")


    ## modify file names for log-files
    fit_log = fit_log.strip()
    i = fit_log.rfind("/")
    j = fit_log.rfind(".")
    if (j > i):
        fit_log = fit_log[:j] + "_INS__call_" + str(abs(INSCounter)).strip() + ".log"
    else:
        fit_log = fit_log + "_INS__call_" + str(abs(INSCounter)).strip() + ".log"

    # Debug:
    # print ("fit_log = ", fit_log)


    ## open param-log file
    fitlogparam = fit_log.strip() + ".param"                                                ## define file name for param-log file
    paramchannel = open(fitlogparam.strip(), 'w')
    paramchannel.write("\n")
    LogString = "log-file containing the actual values of the parameters used in the Interval-Nested-Sampling algorithm:"
    paramchannel.write(LogString + "\n")
    LogString = "-" * len(LogString)
    paramchannel.write(LogString + "\n")
    paramchannel.write("\n")
    paramchannel.flush()


    ## open log file and get current time and date
    fitlogparam = fit_log.strip()                                                           ## define file name for param-log file
    logchannel = open(fitlogparam.strip(), 'w')
    logchannel.write("\n")
    LogString = "log-file for Interval-Nested-Sampling algorithm:"
    logchannel.write(LogString + "\n")
    LogString = "-" * len(LogString)
    logchannel.write(LogString + "\n")
    logchannel.write("\n")
    logchannel.write("\n")


    ## get local time
    lt = time.localtime()
    datestring = time.strftime("algorithm starts at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)
    logchannel.write(str(datestring) + "\n")
    logchannel.write("\n")
    logchannel.write("\n")
    logchannel.flush()


    ## open log file and get current time and date
    fitlogChi2 = fit_log.strip() + ".chi2"                                                  ## define file name for param-log file
    Chi2Channel = open(fitlogChi2.strip(), 'w')


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import MAGIX package ModelFunctionCall ..


    ## define name of MAGIX executable
    if (modelflag == "true" or modelflag.lower() == "myxclass"):                            ## .. optimized for myXCLASS model
        StarterExecutable = "Starter__myXCLASS.exe"
    else:                                                                                   ## .. default package
        StarterExecutable = "Starter.exe"


    ## define parameters for class
    SpecialAlgorithmSettings = Internal_param
    INSBestSiteCounter = Internal_param[3]                                                  ## get number of best sites
    modelClass = ModelFunctionCallClass(printflagNum, GeneralAlgorithmSettings, LastAlgorithmNum, DeterminationChi2, PlotIteration, PlotType, fit_log, \
                                        NumberInputFiles, NumberOutputFiles, ParallelizationFlag, JobID, MaxInputLines, MaxParameter, RenormalizedChi2, \
                                        currentpath, CalculationMethod, xAxisLabel, yAxisLabel, \
                                        zAxisLabel, PathStartScript, ExeCommandStartScript, parameter_set, ExpDataX, ExpDataY, ExpDataError, \
                                        NumberRanges, MinRange, MaxRange, NumberXColumns, NumberYColumns, LengthExpRange, MaxRangeNumber, \
                                        NumberExpFiles, MaxLength, MaxColX, MaxColY, NumberParameter, MPIFlag, MAGIXrootDirectory, JobDir, \
                                        SpecialAlgorithmSettings, StarterExecutable, Chi2Channel, logchannel, paramchannel, INSBestSiteCounter, \
                                        SortFortranNum)


    ## lower limit of chi^2 for stopping condition normalized to the number of calculation points
    chilim = 0.E0
    if (RenormalizedChi2 == 1):
        for i in range(NumberExpFiles):
            chilim = chilim + (NumberYColumns[i] * LengthExpRange[i] - num_par) * abs(chilm)
        chilim = abs(chilim)
        screen_out = "           Renormalized limit for chi^2 = %25.15e" % chilim
        logchannel.write(screen_out + "\n")
        logchannel.write("\n\n")
        logchannel.flush()
        if (printflag == "true"):
            print ("\n")
            print (screen_out)
            print (" ")
    else:
        chilim = abs(chilm)
        screen_out = "           Limit for chi^2 = %25.15e" % chilim
        logchannel.write(screen_out + "\n")
        logchannel.write("\n\n")
        logchannel.flush()
        if (printflag == "true"):
            print ("\n")
            print (screen_out)
            print (" ")


    ## define working variables
    L0 = []
    Points = []


    ## do initial interval division in the parameter space <param_up_down>
    L0 = interval_division(num_par, param_up_down)
    Init_centers = []
    mid_point0 = calc_cen_point(param_up_down)                                              ## mid_point for the whole parameter space
    Init_centers.append(mid_point0)
    n_list = len(L0)
    n, max_iter = Nlive(num_par, param_up_down)


    ## if internal parameter of INS algorithm are difined by user
    max_iter = numiter


    ## print what you do
    if (printflag == "true"):
        print ("\n           Max. number of iterations = ",max_iter)
        print ("\n")


    ## print what you do
    if (printflag == "true"):
        print ("           Iteration:                    chi^2:     Parameter:")
        logchannel.write("           Iteration:                    chi^2:     Parameter:\n")
        logchannel.flush()
        command_string = "echo -n \"           Initialize model function ..\""
        os.system(command_string)


    ## Center points for initial list of intervals L0
    L00 = []
    for i in range(n_list):
        mid_point = calc_cen_point(L0[i])
        Init_centers.append(mid_point)


    ## Checking of point from the list Vectors (evaluated or not yet): if -yes- that don't evaluate
    Init_centers = check_point_paral(Init_centers)


    ## define first mid_point of the whole parameter space
    mid_point0 = Init_centers[0]


    ## Form list L00 for using in explore()
    for i in range(n_list):
        L01 = []
        L01.append(L0[i])
        L01.append(Init_centers[i+1])
        L00.append(L01)
    DBL_MAX = 1e300                                                                         ## ??


    ## uniform random number distrubution
    uniform = random.random


    ## call <ndim_nest> subroutine which performs the nested-sampling steps
    Best = []
    results = ndim_nest(n, explore, param_up_down, printflag, paramchannel, logchannel)
    ni = results['num_iterations']                                                          ## get number of iterations
    samples = results['samples']                                                            ## get parameter vectors and function values
    logZ = results['logZ']                                                                  ## get logZ
    Best = results['Best']                                                                  ## get best of interval boxes
    Bestinput = results['InputFile']

    # Debug:
    # print ("Number of INS-iterations = ", ni)
    # a33=input()


    ## Definition of first set of parameter vectors with proportional weights
    Paramset1 = numpy.array(list(range(ni * (num_par + 2))), dtype = 'float')
    Paramset1.shape = (ni, (num_par + 2))
    min_prob = 1e+03
    max_prob = 0.0


    ## calculate proportional weights, positions and function values in <Paramset1>
    for i in range(ni):
        w = numpy.exp(samples[i].logWt - logZ);                                                   ## Proportional weight
        Paramset1[i,0:num_par] = samples[i].position
        Paramset1[i,num_par] = samples[i].logL
        Paramset1[i,num_par+1] = w


        ## find maximal proportional weight
        if (max_prob < w):
            max_prob = w
            ffmax = Paramset1[i,num_par]
        else:
            ffmax = 1.0


    ## define minimal proportional weight
    min_prob = Paramset1[0,num_par+1]
    ffmin = Paramset1[0,num_par]


    ## Calculation of weights for all best points from the list <Points> (linear approximation of existing weights)
    aa5 = max_prob - min_prob
    bb5 = ffmax - ffmin
    if (bb5 != 0.0):
        kk5 = aa5/bb5
    else:
        kk5 = aa5
    cc5 = min_prob - kk5 * ffmin


    ## add calculated points from the list <Points>
    npp = len(Points)
    Best_Points = []
    for i in range(npp):
        if (Points[i][num_par] <= abs(logZ)):
           Best_Points.append(Points[i])


    ## Form a second set of parameter vectors, function values and weights for each point
    npp = len(Best_Points)
    npoint = npp + ni


    ## form a new set <Paramset2> with number of points <npoint>
    Paramset2 = numpy.array(list(range(npoint * (num_par + 2))), dtype = 'float')
    Paramset2.shape = (npoint, (num_par + 2))
    min_paramset = numpy.array(list(range(num_par + 1)), dtype = 'float')
    min_paramset[0:num_par] = 0.0
    min_paramset[num_par] = 1.e+10
    ii = 0
    for i in range(npoint):
        if (i < ni):
            Paramset2[i,:] = Paramset1[i,:]
        else:
            Param = Best_Points[ii]
            for j in range(num_par+1):
                Paramset2[i,j] = Param[j]
            Paramset2[i,j+1] = kk5 * Param[num_par] + cc5
            ii += 1


        ## check for NaN
        for j in range(num_par+1):
            if (numpy.isnan(Paramset2[i,j])):
                Paramset2[i,j] = 0.0


        ## define min_paramset
        if (min_paramset[num_par] > Paramset2[i,num_par]):
            min_paramset[0:(num_par+1)] = Paramset2[i,0:(num_par+1)]


    ## Sort and cleaning of the list <Best>
    Best_box = []
    nnn = len(Best)


    ## Choosing of the best interval (mid_point < abs(logZ))
    for ii in range(nnn):
        if Best[ii][1][num_par] < abs(logZ):
           Best_box.append(Best[ii])
    if (len(Best_box) == 0):
        ii = 0
        while (ii < nnn):
            metira = 0
            for j in range(num_par):
                if (min_paramset[j] >= Best[ii][0][j,0] and min_paramset[j] <= Best[ii][0][j,1]):
                    metira += 1
            if (metira == num_par):
                Best_box.append(Best[ii])
                ii = nnn + 1
            else:
                ii += 1
        Best = []
        if (Best_box!=[]):
            Best.append(Best_box[0])
    else:                                                                                   ## Sort Best_box and remove repeated interval boxes
       nl = len(Best_box)
       for i in range(nl):
           for j in range(i,nl):
               L11 = []
               if (Best_box[i][1][num_par] > Best_box[j][1][num_par]):
                   L11 = Best_box[i]
                   Best_box[i] = Best_box[j]
                   Best_box[j] = L11
       Best = []
       metka = 0
       L12 = Best_box
       i = 0
       while (metka == 0):
           nl = len(L12)
           L11 = []
           L12 = []
           L11.append(Best_box[i])
           for j in range(i,nl):
               metira1 = 0
               metira2 = 0
               metira3 = 0
               metira4 = 0
               for k in range(num_par):
                   if (Best_box[i][1][k] >= Best_box[j][0][k,0] and Best_box[i][1][k] <= Best_box[j][0][k,1]): metira1 += 1
                   if (Best_box[j][1][k] >= Best_box[i][0][k,0] and Best_box[j][1][k] <= Best_box[i][0][k,1]): metira2 += 1
                   if (Best_box[j][0][k,0] >= Best_box[i][0][k,0] and Best_box[j][0][k,0] <= Best_box[i][0][k,1]): metira3 += 1
                   if (Best_box[j][0][k,1] >= Best_box[i][0][k,0] and Best_box[j][0][k,1] <= Best_box[i][0][k,1]): metira4 += 1

               if (metira1 == num_par or metira2 == num_par or metira3 == num_par or metira4 == num_par):
                   L11.append(Best_box[j])
               else:
                   L12.append(Best_box[j])
               if (metira1 == num_par or metira2 == num_par or metira3 == num_par or metira4 == num_par):
                   L11.append(Best_box[j])
               else:
                   L12.append(Best_box[j])

           if (len(L12) == 0):
               Best.append(L11[0])
               metka = 1
           else:
               Best_box = []
               Best_box = L12
               Best.append(L11[0])


    ## Arrays for results
    ## point of the minimum after INS algorithm
    prob_minim = numpy.array(list(range(num_par + 1)), dtype = 'float')
    nnn = len(Best)


    ## error left anf error right
    error_left = numpy.array(list(range(nnn * num_par)), dtype = 'float')
    error_left.shape = (nnn,num_par)
    error_right = numpy.array(list(range(nnn * num_par)), dtype = 'float')
    error_right.shape = (nnn,num_par)


    ## mean values for parameters
    minima = numpy.array(list(range(nnn * num_par)), dtype = 'float')
    minima.shape = (nnn,num_par)


    ## standard deviations for parameters
    stdev=numpy.array(list(range(nnn * num_par)), dtype = 'float')
    stdev.shape = (nnn,num_par)
    global_flag = -1
    for i in range(nnn):


        ## Point of more probable minimum
        prob_minim[:] = Best[i][1][:]
        if (prob_minim[num_par] == min_paramset[num_par]):
            global_flag = i


        ## Call subroutine <error_info_new> for getting of information about errors
        ## minima - set of other probable minima points
        ## stdev  - standard deviations from mean values
        error_left[i,:], error_right[i,:], minima[i,:], stdev[i,:] = error_info_new(prob_minim, Paramset2, npoint)


    # print some information to log file and screen
    logline1 = "           Number of function evaluations: " + str(len(Points))
    logline2 = "           Evidence = " + str(logZ)
    logchannel.write("\n")
    logchannel.write(logline1 + "\n")
    logchannel.write(logline2 + "\n")
    logchannel.write("\n")
    logchannel.write("\n")
    logchannel.flush()
    if (printflag == "true"):
        print (logline1)
        print (logline2)
        print ("\n")
        print ("\n")



    ## Result information about probable minima and error estimation
    results_box = numpy.array(list(range(num_par * 2)), dtype = 'float')
    results_box.shape = (num_par,2)
    if (global_flag < 0):                                                            ## if best point of minimum is not from list <Best>
        min_global = numpy.array(list(range(num_par)), dtype = 'float')
        stdev_global = numpy.array(list(range(num_par)), dtype = 'float')
        err_left = numpy.array(list(range(num_par)), dtype = 'float')
        err_right = numpy.array(list(range(num_par)), dtype = 'float')
        prob_minim[:] = min_paramset[:]


        ## Call subroutine <error_info_new> for getting informations about errors
        ## minima - set of other probable minima points
        ## stdev  - standard deviations from mean values
        err_left, err_right, min_global, stdev_global = error_info_new(prob_minim, Paramset2, npoint)


        ## screen output
        logline1 = "           | Free parameter | Point of minimum  |   Error_left    |   Error_right   | Mean parameter value |  Standard deviation  |"
        logline2 = "           ------------------------------------------------------------------------------------------------------------------------"
        logchannel.write("\n")
        logchannel.write(logline1 + "\n")
        logchannel.write(logline2 + "\n")

        # Debug:
        # print ("min_paramset = ", min_paramset)
        # print ("abs(err_left) = ", abs(err_left))
        # print ("abs(err_right) = ", abs(err_right))
        # print ("min_global = ", min_global)
        # print ("stdev_global = ", stdev_global)


        if (printflag == "true"):
            print ("\n")
            print (logline1)
            print (logline2)

        for j1 in range(num_par):
            logline1 = "           |     % 6i     |% 18.8e |% 16.8e |% 16.8e |% 21.8e |% 21.8e |"      \
                        % ((j1 + 1), min_paramset[j1], abs(err_left[j1]), abs(err_right[j1]), min_global[j1], stdev_global[j1])
            logchannel.write(logline1 + "\n")
            if (printflag == "true"):
                print (logline1)

        logline2 = "           ------------------------------------------------------------------------------------------------------------------------"
        logchannel.write(logline2 + "\n")
        logchannel.write("\n")
        logchannel.write("\n")
        logchannel.flush()
        if (printflag == "true"):
            print (logline2)
            print (" ")
            print (" ")

        for j1 in range(num_par):
            results_box[j1,0] = min_paramset[j1] - abs(err_left[j1])
            results_box[j1,1] = min_paramset[j1] + abs(err_right[j1])

    else:                                                                            ## if best point of minimum is from best intervals (list <Best>)
        i7 = global_flag
        logline1 = "           | Free parameter | Point of minimum  |   Error_left    |   Error_right   | Mean parameter value |  Standard deviation  |"
        logline2 = "           ------------------------------------------------------------------------------------------------------------------------"
        logchannel.write(logline1 + "\n")
        logchannel.write(logline2 + "\n")
        if (printflag == "true"):
            print (logline1)
            print (logline2)
        for j1 in range(num_par):
            logline1 = "           |     % 6i     |% 18.8e |% 16.8e |% 16.8e |% 21.8e |% 21.8e |"      \
                        % ((j1 + 1), min_paramset[j1], abs(error_left[i7,j1]), abs(error_right[i7,j1]), minima[i7,j1], stdev[i7,j1])
            logchannel.write(logline1 + "\n")
            if (printflag == "true"):
                print (logline1)
        logline2 = "           ------------------------------------------------------------------------------------------------------------------------"
        logchannel.write(logline2 + "\n")
        logchannel.write("\n")
        logchannel.write("\n")
        logchannel.flush()
        if (printflag == "true"):
            print (logline2)
            print (" ")
            print (" ")

        for j1 in range(num_par):
            results_box[j1,0] = min_paramset[j1] - abs(error_left[i7,j1])
            results_box[j1,1] = min_paramset[j1] + abs(error_right[i7,j1])



    ## Formation of list <Best_box> for output results
    Best_box = []
    for i1 in range(len(Best)):
        mett = 0
        for j1 in range(num_par):
            if (Best[i1][1][j1] >= results_box[j1,0] and Best[i1][1][j1] <= results_box[j1,1]): mett += 1
        L15 = []
        if (mett < num_par):
            L15.append(Best[i1][0])
            L15.append(Best[i1][1])
            L15.append(minima[i1,:])
            L15.append(stdev[i1,:])
            Best_box.append(L15)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print result
    if (len(Best_box) > 0):
        logline1 = "           Other best interval boxes (probable minima): " + str(len(Best_box))
        logline2 = "           Nr.:  | Parameter | Best interval boxes         |   Center point    | Mean parameter value | Standard deviation  | Function value"
        logline3 = "           ---------------------------------------------------------------------------------------------------------------------------------"
        logchannel.write("\n")
        logchannel.write(logline1 + "\n")
        logchannel.write("\n")
        logchannel.write(logline2 + "\n")
        logchannel.write(logline3 + "\n")
        if (printflag == "true"):
            print ("\n")
            print (logline1)
            print ("\n")
            print (logline2)
            print (logline3)


        for i1 in range(len(Best_box)):
            for j1 in range(num_par):
                a1 = Best_box[i1][0][j1,0]
                a2 = Best_box[i1][0][j1,1]

                if (j1 == 0):
                   logline1 = "           %4d  |  % 6i   | [% 10.4e, % 10.4e]  | % 10.4e       |  % 11.4e         |  % 10.4e        |  % 10.4e" \
                                % ((i1 + 1), (j1 + 1), a1,a2,Best_box[i1][1][j1],Best_box[i1][2][j1],Best_box[i1][3][j1],Best_box[i1][1][num_par])
                else:
                   logline1 = "                 |  % 6i   | [% 10.4e, % 10.4e]  | % 10.4e       |  % 11.4e         |  % 10.4e        |" \
                            % ((j1 + 1), a1,a2,Best_box[i1][1][j1],Best_box[i1][2][j1],Best_box[i1][3][j1],)
                logchannel.write(logline1 + "\n")
                if (printflag == "true"):
                    print (logline1)

            logline1 = "           ---------------------------------------------------------------------------------------------------------------------------------"
            logchannel.write(logline1 + "\n")
            if (printflag == "true"):
                print (logline1)
        logchannel.write("\n")
        logchannel.write("\n")
        logchannel.flush()
        if (printflag == "true"):
            print ("\n")
            print ("\n")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Building of histogramms for parameter values (for the whole parameter space!)
    parud = numpy.array(list(range(2)), dtype='float')
    for i in range(num_par):
        parud[0] = param_up_down[i,0]
        parud[1] = param_up_down[i,1]


        ## call subroutine for histogramms (with saving!)
        if (plotflag == "true" or plotflag == "saveonly"):
            my_hist1(Paramset2[:,i], parud, min_paramset[i], i, num_par, fit_log, INSCounter, plotflag)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## close param-log file
    Chi2Channel.close()
    paramchannel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## sort log.chi^2 file
    Chi2Channel = open(fitlogChi2.strip(), 'r')
    contens = Chi2Channel.readlines()                                                       ## read in contents of input file
    Chi2Contents = []
    for inputline in contens:                                                               ## loop over all lines of the input file
        line = inputline.split()
        Currentline = []
        count = 0
        for elements in line:
            count += 1
            if (count == 1):
                elements = float(elements)
            Currentline.append(elements)

        Chi2Contents.append(Currentline)
    Chi2Channel.close()


    ## sort belonging to chi^2 values
    sortchi2 = sorted(Chi2Contents)

    # Debug:
    # print (Chi2Contents[0])
    # print (sortchi2[0])


    ## write sorted chi^2 parameter vectors
    Chi2Channel = open(fitlogChi2.strip(), 'w')
    Chi2Channel.write("    Nr.:                     chi**2:         parameters:\n")
    lastline = ""
    countLine = 0
    for inputline in sortchi2:
        count = 0
        formatline = ""
        for elements in inputline:
            count += 1
            if (count == 1):
                formatline = '%27.15e        ' % float(elements)
            else:
                k = len(elements)
                if (k < 20):
                    formatelements = ' %20s ' % elements
                else:
                    formatelements = " " + elements + " "
                formatline += formatelements
        if (lastline != formatline):
            lastline = formatline
            countLine += 1
            countLineString = '%8d' % countLine
            Chi2Channel.write(countLineString + " " + str(formatline) + "\n")
    Chi2Channel.close()


    ## close log file
    logchannel.write("\n")
    lt = time.localtime()
    datestring = time.strftime("algorithm ends at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)
    logchannel.write(str(datestring) + "\n")
    logchannel.write("\n")
    SepString = 149 * "-"
    logchannel.write(SepString + "\n")
    logchannel.flush()
    logchannel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create final input files


    ## define path where final input files are written to
    fit_log = fit_log.strip()
    i = fit_log.rfind("/")
    if (i > (-1)):
        BaseDir = fit_log[:i]
    else:
        BaseDir = ""


    ## extract informations from Bestinput variable
    NameInputFile = ""
    WriteFlag = False
    for LocalBestSite in range(INSBestSiteCounter):
        for line in Bestinput[LocalBestSite]:
            StrippedLine = line.strip()


            ## get name of input file
            if (StrippedLine.startswith("Input-File ")):
                k = StrippedLine.find(":  , file:")
                if (k > (-1)):
                    NameInputFile = StrippedLine[k + 10:].strip()


                    ## modify input file name
                    j = NameInputFile.rfind(".")
                    if (j > (-1)):
                        NameInputFile = NameInputFile[:j] + "__INS__call_" + str(abs(INSCounter))
                    else:
                        NameInputFile = NameInputFile + "__INS__call_" + str(abs(INSCounter))
                    if (INSBestSiteCounter > 1):
                        NameInputFile += "__site_" + str(LocalBestSite + 1)
                    NameInputFile += ".out.input"


                    ## open channel for final input file
                    NameInputFile = BaseDir + "/" + NameInputFile
                    InputFile = open(NameInputFile, 'w')


            ## identify begin of contents of final input file(s)
            elif (StrippedLine.startswith("-start_input-file-")):
                WriteFlag = True


            ## identify end of contents of final input file(s)
            elif (StrippedLine.startswith("-end_input-file-")):
                try:
                    InputFile.close()
                except AttributeError:
                    NameInputFile = ""
                WriteFlag = False


            ## write contents of final input file(s) to current input file
            elif (WriteFlag == True and NameInputFile != ""):
                InputFile.write(line)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get best function parameters
    FinalParameter, FitFunctionAll, Chi2ValuesAll = modelClass.GetBestFunctionValues()


    ## define new parameter set
    FinalParameterSet = []
    for LocalBestSite in range(INSBestSiteCounter):
        j = (-1)
        LocalParameterSet = []
        for i in range(NumberParameter):
            if (parameter_set[1][i] == 1):
                j += 1
                LocalParameterSet.append(FinalParameter[LocalBestSite][j])
            else:
                LocalParameterSet.append(parameter_set[0][i])
        FinalParameterSet.append(LocalParameterSet)

    # Debug:
    # print ("NumberParameter = ", NumberParameter)
    # print ("FinalParameter = ", FinalParameter)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define return variables
    calstatus = 0
    return (calstatus, FinalParameterSet, FitFunctionAll, Chi2ValuesAll)
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------

