!*********************************************************************************************************************************************************
!> Package: Genetic algorithm
!>
!>
!>  This module contains the subroutines for genetic algorithm
!>  Copyright (C) 2009 - 2024  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - Module GenVariables:                      Module contains global variables and subroutines for Genetic algorithm
!>      - subroutine check_parameters:              Chek if parameter is within range and correct value if parameter is out of range
!>      - subroutine analysis_GA:                   analysis of GA work: are used next criteria:
!>      - subroutine opfun_calc:                    parallel calculations of optimization function's values
!>      - subroutine sort_population:               sortirung population on value of optimization function
!>      - subroutine Random_selection:              random selection for choosing of parental couples from population
!>      - subroutine Elit_selection:                elite selection for choosing of parental couples from population
!>      - subroutine Tournament_selection:          Tournament selection for choosing of parental couples from population
!>      - subroutine Inbreeding:                    Inbreeding for choosing of parental couples from population
!>      - subroutine Outbreeding:                   outbreeding for choosing of parental couples from population
!>      - subroutine distance:                      calculation of distance between two chromosomes: chrom1 and chrom2
!>      - subroutine discrete_crossover:            discrete crossover: each child has genes from parents genes
!>      - subroutine simple_crossover:              simple crossover - each child has genes from parents genes
!>      - subroutine SBX_crossover:                 simulated Binary crossover - each child has genes from parents genes
!>      - subroutine Mutation:                      each mutant has uniform()*genes from parents genes
!>      - subroutine Mutation1:                     each mutant is random from distribution near boundaries of parametric space
!>      - subroutine Mutation2:                     each mutant is random from distribution near boundaries of parametric space
!>      - subroutine check_chromosomes:             checking of the chromosomes, if two or more chromosomes are the same than to take a new chromosomes
!>                                                  from popul-array
!>      - subroutine all_distances:                 calculation of distance between ALL chromosomes and finding of maximal distance to check a
!>                                                  convergence of the population
!>      - subroutine CallGenAlg:                    calls the different subroutines of the genetic algorithm
!>      - Module Algorithm:                         module contains the main subroutine used to start the different versions of the PSO algorithm
!>      - subroutine MainAlg:                       main subroutine which starts the PSO algorithm
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  I. Bernst     2009-09-24  Initial version (python)
!>  T. Moeller    2010-06-22  Initial version (fortran)
!>  T. Moeller    2010-12-15  modification of parameter output
!>  T. Moeller    2012-01-16  Updated version
!>  T. Moeller    2014-08-20  myXCLASS (model) optimized version
!>  T. Moeller    2014-08-30  modified and restructured for GPU
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: GenVariables
!>
!>         Module contains global variables and subroutines for Genetic algorithm
!>
!>
!> \author Thomas Moeller
!>
!> \date 2010-06-09
!>
Module GenVariables

    use FunctionCalling
    use Model

    implicit none
    integer :: num_iter, MaxGeneticIter                                                     !< current and max. iteration number

    real*8 :: chilim                                                                        !< lower limit of chi**2 normalized to the total number of
                                                                                            !< all data points
    real*8, allocatable, dimension(:) :: a                                                  !< array containing all parameters
    real*8, allocatable, dimension(:) :: OptimizedParameterLowLimit                         !< lower limit for each free parameter
    real*8, allocatable, dimension(:) :: OptimizedParameterUpperLimit                       !< upper limit for each free parameter

    logical :: RangeFlag                                                                    !< flag indicating if new ranges are used
    logical, allocatable, dimension(:) :: ia                                                !< array indicating the parameter which should be optimized

    contains


        !*************************************************************************************************************************************************
        !> subroutine: check_parameters
        !>
        !> Chek if parameter is within range and correct value if parameter is out of range
        !>
        !> input variables:     param:              parameter which is checked
        !>                      low_limit:          lower limit of parameter param
        !>                      upper_limit:        upper limit of parameter param
        !>
        !> output variables:    param:              contains corrected value of param if param is out of range param
        !>
        !> \author Thomas Moeller
        !>
        !> \date 19.08.2010
        !>
        subroutine check_parameters(param, low_limit, upper_limit)

            use Variables
            use FunctionCalling

            implicit none
            real*8 :: param                                                                 !< parameter which is checked
            real*8 :: low_limit                                                             !< lower limit of parameter param
            real*8 :: upper_limit                                                           !< upper limit of parameter param
            real*8 :: range_size                                                            !< size of range
            real*8, parameter :: reduction_coeff = 1.d-3                                    !< determine the range around the limit border


            if (param < low_limit .or. upper_limit < param) then
                range_size = upper_limit - low_limit
                if (param < low_limit) then
                    param = RandomWithLimits(low_limit, low_limit + range_size * reduction_coeff)
                else
                    param = RandomWithLimits(upper_limit - range_size * reduction_coeff, upper_limit)
                endif
            endif
            return
        end subroutine check_parameters


        !*************************************************************************************************************************************************
        !> subroutine: analysis_GA
        !>
        !> Analysis of GA work: are used next criteria:
        !>
        !>
        !> input variables:     Nch:                number of chromosomes
        !>                      max_dist :          maximal distance between chromosomes in population
        !>                      nfit:               number of parameters which should be optimized
        !>                      chromosomes:        array of the chromosomes
        !>
        !> output variables:    zira:               If 0 - there is not a new range of parametric space
        !>                                          If 1 - there is a new range of parametric space (there is a convergence of population to global
        !>                                                 solution)
        !>                      new_range:          contains array with new range of parametric space
        !>                                          (If there is a convergence of population)
        !>
        !>
        !> \author Irina Bernst
        !>
        !> \date 28.09.2010
        !>
        subroutine analysis_GA(chromosomes, Nch, max_dist, nfit, new_range, zira)

            use Variables

            implicit none
            integer :: i, j                                                                 !< loop variables
            integer :: zira                                                                 !< If 0 - there is not a new range of parametric space
                                                                                            !< If 1 - there is a new range of parametric space (there
                                                                                            !<        is a convergence of population to global solution)
            integer :: nfit                                                                 !< number of parameters which should be optimized
            integer :: Nch                                                                  !< number of chromosomes
            real*8 :: max_dist                                                              !< maximal distance between chromosomes in population
            real*8 :: plow, pupp, pmid                                                      !< working variables
            real*8, dimension(Nch, nfit + 1) :: chromosomes                                 !< array of the chromosomes
            real*8, dimension(nfit, 2) :: new_range                                         !< contains array with new range of parametric space


            new_range = 0.d0
            if (max_dist > 90.d0) then
                new_range(:,1) = 1.d99
                new_range(:,2) = -1.d99
                zira = 1
                Do i = 1, Nch
                    Do j = 1, nfit
                        if (new_range(j, 1) > chromosomes(i, j)) then
                            new_range(j, 1) = chromosomes(i, j)
                        endif
                        if (new_range(j, 2) < chromosomes(i, j)) then
                            new_range(j, 2) = chromosomes(i, j)
                        endif
                    end Do
                end Do
            else
                zira = 0
            endif

            if (zira == 0) then
                new_range(:, 1) = OptimizedParameterLowLimit(:)
                new_range(:, 2) = OptimizedParameterUpperLimit(:)
            else
                if (new_range(1, 1) /= 0.d0 .and. new_range(1, 2) /= 0.d0) then
                    Do j = 1, nfit

                        ! Debug:
                        ! print*, new_range(j, 1), new_range(j, 2)

                        plow = abs(new_range(j, 2) - OptimizedParameterLowLimit(j))
                        pupp = abs(new_range(j, 1) - OptimizedParameterUpperLimit(j))
                        pmid = abs(new_range(j, 2) - new_range(j, 1))

                        if (pmid < plow .and. pmid < pupp) then
                            if (plow < pupp) then
                                new_range(j, 1) = OptimizedParameterLowLimit(j)
                            else
                                new_range(j, 2) = OptimizedParameterUpperLimit(j)
                            endif
                        endif

                        if (plow == pupp) then
                            new_range(j, 1) = OptimizedParameterLowLimit(j)
                            new_range(j, 2) = OptimizedParameterUpperLimit(j)
                        endif
                    end Do
                endif
            endif
            return
        end subroutine analysis_GA


        !*************************************************************************************************************************************************
        !> subroutine: opfun_calc
        !>
        !> Parallel calculations of optimization function's values
        !>
        !> input variables:     objects:                        array containing the parameter which should be optimized; last column - function value
        !>                      NumFile:                        number of input files
        !>                      MaxL:                           max. total length
        !>                      MaxCol:                         max. number of columns
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Nob:                            number of objects
        !>                      ma:                             total number of all parameter
        !>
        !> output parameters:   objects:                        array containing the parameter which should be optimized
        !>                                                      + last column contents the corresponding value of the model function
        !>
        !> \author Irina Bernst and Thomas Moeller
        !>
        !> \date 24.09.2010
        !>
        subroutine opfun_calc(NumFile, MaxL, MaxCol, nfit, Nob, ma, objects)

            use Variables
            use FunctionCalling

            implicit none
            integer :: nfit                                                                 !< number of parameters which should be optimized
            integer :: ma                                                                   !< total number of all parameter
            integer :: Nob                                                                  !< number of objects
            integer :: i                                                                    !< loop variable
            integer :: NumFile                                                              !< number of input files
            integer :: MaxL                                                                 !< max. total length
            integer :: MaxCol                                                               !< max. number of columns
            real*8,dimension(Nob) :: chi2ValuesVector                                       !< chi2 value vector
            real*8, dimension(Nob, nfit + 1) :: objects                                     !< array containing the parameter which should be optimized
                                                                                            !< + last column contents the corresponding value of the
                                                                                            !< model function
            ! Debug:
            ! print*,'>>',objects(1,:)
            ! print*,'>>',objects(20,:)
            ! print*, Nob, nfit
            ! stop

            !< print what you do!!
            if (printflag) then
                print '(A, 120(" "), A, $)',char(13),char(13)
                print '(11x, "Calculation of optimization function value ..", $)'
            endif


            !< if calculation reduction is chosen, read chi**2 file
            if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then
                call CheckCalculatedParameterSets(nfit, 1)
            endif


            !< print what you do ..
            if (printflag) then
                print '(A, 11x, "Calculation of optimization function values ..                    ", A, $)', char(13), char(13)
            endif


            !< determine chi2 values for all parameter vectors
            chi2ValuesVector = 0.d0
            call ModelCalcChiFunctionGeneral(ma, ia, a, Nob, nfit, NumFile, MaxL, MaxCol, objects, chi2ValuesVector)
            Do i = 1, Nob
                objects(i, nfit + 1) = chi2ValuesVector(i)
            end Do


            CurrentNumberLinesCalcReduction = NumberLinesChi2                               !< update CurrentNumberLinesCalcReduction variable

            ! Debug:
            !   print*,'>>',objects(1,:)
            !   print*,'>>',objects(20,:)
            !   stop

            return                                                                          !< return to main program
        end subroutine opfun_calc


        !*************************************************************************************************************************************************
        !> subroutine: sort_population
        !>
        !> Sortirung population on value of optimization function (above --> minimal value; last value = maximal value)
        !>
        !> input variables:     popul:                  array containing the parameter which should be optimized and values of optimization function
        !>                      colx:                   number of columns belonging to the x-column
        !>                      NumFile:                number of input files
        !>                      MaxL:                   max. total length
        !>                      MaxCol:                 max. number of columns
        !>                      nfit:                   number of parameters which should be optimized
        !>                      Npopul:                 number of chromosomes in population
        !>
        !> output parameters:   popul:                  sorted array popul
        !>
        !> \author Irina Bernst and Thomas Moeller
        !>
        !> \date 27.09.2010
        !>
        subroutine sort_population(popul, nfit, Npopul)

            use Variables
            use FunctionCalling

            implicit none
            integer :: nfit                                                                 !< number of parameters which should be optimized
            integer :: Npopul                                                               !< number of chromosomes in population
            integer :: i, j, jj                                                             !< loop variables
            integer :: k                                                                    !< working variable
            real*8 :: xx, populat                                                           !< working variables
            real*8, dimension(Npopul, nfit + 1) :: popul                                    !< array containing the parameter which should be optimized
                                                                                            !< and values of optimization function

            k = 1
            Do i = 1, Npopul
                xx = popul(i, nfit + 1)
                Do j = i, Npopul
                    if (xx >= popul(j, nfit + 1)) then
                        xx = popul(j, nfit + 1)
                        k = j
                    endif
                end Do
                Do jj = 1, (nfit + 1)
                    populat = popul(i, jj)
                    popul(i, jj) = popul(k, jj)
                    popul(k, jj) = populat
                end Do
            enddo
            return
        end subroutine sort_population


        !*************************************************************************************************************************************************
        !> subroutine: Random_selection
        !>
        !> Random selection for choosing of parental couples from population (easy with random generator;
        !< there is a possibility that one chromosome will two or more times parental chromosome!)
        !< Output: parent = array of parental couples
        !>
        !> input variables:     chromosomes:                    array of chromosomes
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Nch:                            number of chromosomes
        !>                      Npar:                           number of parental couples
        !>
        !> output parameters:   parent1:                        array of parental couples
        !<
        !> \author Irina Bernst
        !>
        !> \date 27.09.2010
        !>
        subroutine Random_selection(chromosomes, nfit, Nch, Npar, parent1)

            use Variables
            use FunctionCalling

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: Nch                                                                  !< number of chromosomes
            integer :: Npar                                                                 !< number of parental couples
            integer :: ran_num                                                              !< random number of chromosome
            integer :: i                                                                    !< loop variable
            real*8 :: p1, p2                                                                !< working variables
            real*8, dimension(Npar, nfit + 1) :: parent1                                    !< array of parental couples
            real*8, dimension(Nch, nfit + 1) :: chromosomes                                 !< array of chromosomes

            Do i = 1, Npar
                p1 = 1
                p2 = float(Nch)
                ran_num = int(RandomWithLimits(p1, p2))
                parent1(i,:) = chromosomes(ran_num, :)
            end Do
            return
        end subroutine Random_selection


        !*************************************************************************************************************************************************
        !> subroutine: Elit_selection
        !>
        !> Elit selection for choosing of parental couples from population
        !< Beste first Npar from chromosomes-array
        !< Output: parent = array of parental couples
        !>
        !> input variables:     chromosomes:                    array of chromosomes
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Nch:                            number of chromosomes
        !>                      Npar:                           number of parental couples
        !>
        !> output parameters:   parent1:                        array of parental couples
        !<
        !> \author Irina Bernst, Thomas Möller
        !>
        !> \date 27.09.2010
        !>
        subroutine Elit_selection(chromosomes, nfit, Nch, Npar, parent1)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: Nch                                                                  !< number of chromosomes
            integer :: Npar                                                                 !< number of parental couples
            real*8, dimension(Npar, nfit + 1) :: parent1                                    !< array of parental couples
            real*8, dimension(Nch, nfit + 1) :: chromosomes                                 !< array of chromosomes

            if (Npar > Nch) then
                print*,' Error in subroutine Elit_selection:'
                print*,'   The upper bound of array parent1 (Npar) is larger than the upper bound'
                print*,'   of the array chromosomes (Nch)!'
                print*,' '
                print*,'   Npar = ', Npar
                print*,'   Nch = ', Nch
                print*,' '
                stop 'Program stopped!'
            else
                parent1(1:Npar, :)=chromosomes(1:Npar, :)
            endif
            return
        end subroutine Elit_selection


        !*************************************************************************************************************************************************
        !> subroutine: Tournament_selection
        !>
        !> Tournament selection for choosing of parental couples from population
        !< Random choosing of two chromosomes, is choosed the best and this comes to parent
        !< Output: parent = array of parental couples
        !>
        !> input variables:     chromosomes:                    array of chromosomes
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Nch:                            number of chromosomes
        !>                      Npar:                           number of parental couples
        !>
        !> output parameters:   parent1:                        array of parental couples
        !<
        !> \author Irina Bernst
        !>
        !> \date 27.09.2010
        !>
        subroutine Tournament_selection(chromosomes, nfit, Nch, Npar, parent1)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: Nch                                                                  !< number of chromosomes
            integer :: Npar                                                                 !< number of parental couples
            integer :: nr1                                                                  !< working variable
            integer :: nr2                                                                  !< working variable
            integer :: i                                                                    !< loop variable
            real*8 :: p1, p2                                                                !< working variables
            real*8, dimension(Npar, nfit + 1) :: parent1                                    !< array of parental couples
            real*8, dimension(Nch, nfit + 1) :: chromosomes                                 !< array of chromosomes

            p1 = 1
            p2 = float(Nch)
            Do i = 1, Npar
                nr1 = int(RandomWithLimits(p1, p2))
                nr2 = int(RandomWithLimits(p1, p2))
                if (chromosomes(nr1, nfit + 1) < chromosomes(nr2, nfit + 1)) then
                    parent1(i,:) = chromosomes(nr1,:)
                else
                    parent1(i,:) = chromosomes(nr2,:)
                endif
            end Do
            return
        end subroutine Tournament_selection


        !*************************************************************************************************************************************************
        !> subroutine: Inbreeding
        !>
        !> Inbreeding for choosing of parental couples from population
        !< Sorts chromosomes on genes and chooses neighbour chromosomes
        !< Output: parent = array of parental couples
        !>
        !> input variables:     chromosomes:                    array of chromosomes
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Nch:                            number of chromosomes
        !>                      Npar:                           number of parental couples
        !>
        !> output parameters:   parent1:                        array of parental couples
        !<
        !> \author Irina Bernst
        !>
        !> \date 27.09.2010
        !>
        subroutine Inbreeding(chromosomes, nfit, Nch, Npar, parent1)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: Nch                                                                  !< number of chromosomes
            integer :: Npar                                                                 !< number of parental couples
            integer :: nr1                                                                  !< working variable
            integer :: nr2, nr3, np                                                         !< working variables
            real*8 :: p1, p2                                                                !< working variables
            real*8, dimension(Npar, nfit + 1) :: parent1                                    !< array of parental couples
            real*8, dimension(Nch, nfit + 1) :: chromosomes                                 !< array of chromosomes
            real*8, dimension(2) :: dist_chrom1, dist_chrom2, dist_chrom3                   !< distances between two chromosmes

            p1 = 1
            p2 = float(Nch)
            np = 1
            Do While (np < Npar)

                nr1 = int(RandomWithLimits(p1, p2))
                nr2 = int(RandomWithLimits(p1, p2))
                nr3 = int(RandomWithLimits(p1, p2))

                call distance(dist_chrom1, chromosomes(nr1,:), chromosomes(nr2,:), nfit)
                call distance(dist_chrom2, chromosomes(nr2,:), chromosomes(nr3,:), nfit)
                call distance(dist_chrom3, chromosomes(nr3,:), chromosomes(nr1,:), nfit)

                ! Debug:
                ! print*,'<<<< Inbreeding:'
                ! print*,chromosomes(nr1,:)
                ! print*,chromosomes(nr2,:)
                ! print*,chromosomes(nr3,:)
                ! print*,dist_chrom1(1),dist_chrom2(1),dist_chrom3(1)

                if (dist_chrom1(1) < dist_chrom2(1).and.dist_chrom1(1) < dist_chrom3(1)) then
                    parent1(np,:) = chromosomes(nr1,:)
                    parent1(np + 1,:) = chromosomes(nr2,:)
                    np = np + 2
                endif

                if (dist_chrom2(1) < dist_chrom1(1).and.dist_chrom2(1) < dist_chrom3(1)) then
                    parent1(np,:) = chromosomes(nr2,:)
                    parent1(np+1,:) = chromosomes(nr3,:)
                    np = np + 2
                endif
                if (dist_chrom3(1) < dist_chrom2(1).and.dist_chrom3(1) < dist_chrom1(1)) then
                    parent1(np,:) = chromosomes(nr3,:)
                    parent1(np + 1,:) = chromosomes(nr1,:)
                    np = np + 2
                endif
            end Do
            return
        end subroutine Inbreeding


        !*************************************************************************************************************************************************
        !> subroutine: Outbreeding
        !>
        !> Outbreeding for choosing of parental couples from population
        !< Sorts chromosomes on genes and chooses far chromosomes
        !< Output: parent = array of parental couples
        !>
        !> input variables:     chromosomes:                    array of chromosomes
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Nch:                            number of chromosomes
        !>                      Npar:                           number of parental couples
        !>
        !> output parameters:   parent1:                        array of parental couples
        !<
        !> \author Irina Bernst
        !>
        !> \date 27.09.2010
        !>
        subroutine Outbreeding(chromosomes, nfit, Nch, Npar, parent1)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: Nch                                                                  !< number of chromosomes
            integer :: Npar                                                                 !< number of parental couples
            integer :: nr1, nr2, nr3, np                                                    !< working variables
            real*8 :: p1, p2                                                                !< working variables
            real*8, dimension(Npar, nfit + 1) :: parent1                                    !< array of parental couples
            real*8, dimension(Nch, nfit + 1) :: chromosomes                                 !< array of chromosomes
            real*8, dimension(2) :: dist_chrom1, dist_chrom2, dist_chrom3                   !< distances between two chromosmes

            p1 = 1
            p2 = float(Nch)
            np = 1

            Do While (np < Npar)

                nr1 = int(RandomWithLimits(p1, p2))
                nr2 = int(RandomWithLimits(p1, p2))
                nr3 = int(RandomWithLimits(p1, p2))

                call distance(dist_chrom1, chromosomes(nr1,:), chromosomes(nr2,:), nfit)
                call distance(dist_chrom2, chromosomes(nr2,:), chromosomes(nr3,:), nfit)
                call distance(dist_chrom3, chromosomes(nr3,:), chromosomes(nr1,:), nfit)

                ! Debug:
                ! print*,'<<<< Outbreeding:'
                ! print*,nr1,nr2,nr3
                ! print*,chromosomes(nr1,:)
                ! print*,chromosomes(nr2,:)
                ! print*,chromosomes(nr3,:)
                ! print*,dist_chrom1(1), dist_chrom2(1), dist_chrom3(1)

                if (dist_chrom1(1) > dist_chrom2(1) .and. dist_chrom1(1) > dist_chrom3(1)) then
                    parent1(np,:) = chromosomes(nr1,:)
                    parent1(np + 1,:) = chromosomes(nr2,:)
                    np = np + 2
                endif

                if (dist_chrom2(1) > dist_chrom1(1) .and. dist_chrom2(1) > dist_chrom3(1)) then
                    parent1(np,:) = chromosomes(nr2,:)
                    parent1(np + 1,:) = chromosomes(nr3,:)
                    np = np + 2
                endif
                if (dist_chrom3(1) > dist_chrom2(1) .and. dist_chrom3(1) > dist_chrom1(1)) then
                    parent1(np,:) = chromosomes(nr3,:)
                    parent1(np + 1,:) = chromosomes(nr1,:)
                    np = np  + 2
                endif
            end Do
            return
        end subroutine Outbreeding


        !*************************************************************************************************************************************************
        !> subroutine: distance
        !>
        !> Calculation of distance between two chromosomes: chrom1 and chrom2
        !<            dist_chrom(1) - distance between two chromosomes
        !<            dist_chrom(2) - if 0 then chromosomes are really very close to each other in parameters space
        !<                            if more than 0 then may be chromosomes are simmetrically in parameters space
        !<
        !< Output: dist_chrom = array(2)
        !>
        !> input variables:     chrom1:                         first chromosome
        !>                      chrom2:                         second chromosome
        !>                      nfit:                           number of parameters which should be optimized
        !>
        !> output parameters:   dist_chrom:                     array(2) with distance between two chromosomes
        !<
        !> \author Irina Bernst
        !>
        !> \date 27.09.2010
        !>
        subroutine distance(dist_chrom, chrom1, chrom2, nfit)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: i                                                                    !< loop variable
            real*8, dimension(nfit + 1) :: chrom1, chrom2                                   !< arrays of chromosomes
            real*8, dimension(2) :: dist_chrom                                              !< array with distance and


            dist_chrom(1) = 0.d0
            dist_chrom(2) = 0.d0

            ! Debug:
            ! print*,"<<< Distances"
            ! print*,chrom1
            ! print*,chrom2

            Do i = 1, nfit
                dist_chrom(1) = dist_chrom(1) + (chrom1(i)-chrom2(i))**2.0
                if ((chrom1(i) <= 0.d0 .and. chrom2(i) >= 0.d0) .or. (chrom1(i) >= 0.d0 .and. chrom2(i) <= 0.d0)) then
                    dist_chrom(2) = dist_chrom(2) + 1
                endif
            end Do

            return
        end subroutine distance


        !*************************************************************************************************************************************************
        !> subroutine: discrete_crossover
        !>
        !> Discrete crossover: each child has genes from parents genes (ramdomly from intervals parents genes)
        !> Number of children chromosomes == number of parental chromosomes
        !> Output: children - array of children chromosomes
        !>
        !> input variables:     parent:                         array of parental chromosomes
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Npar:                           number of parental couples
        !>
        !> output parameters:   children1:                      array of children chromosomes
        !<
        !> \author Irina Bernst
        !>
        !> \date 27.09.2010
        !>
        subroutine discrete_crossover(parent, nfit, Npar, children1)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: Npar                                                                 !< number of parental couples
            integer :: i, j                                                                 !< loop variables
            real*8, dimension(Npar, nfit + 1) :: parent, children1                          !< array of parental and children chromosomes

            ! Debug:
            ! print*,'<<<< Discrete crossover'
            ! print*,parent(1,:)
            ! print*,parent(2,:)

            i = 1
            Do While (i <= (Npar - 1))

                Do j = 1, nfit
                   if (parent(i, j) < parent(i + 1, j)) then
                       children1(i, j) = RandomWithLimits(parent(i, j), parent(i + 1, j))
                       children1(i + 1, j) = RandomWithLimits(parent(i, j), parent(i + 1, j))
                   else
                       children1(i, j) = RandomWithLimits(parent(i+1,j),parent(i, j))
                       children1(i + 1, j) = RandomWithLimits(parent(i+1,j),parent(i, j))
                   endif
                   if (parent(i, j) == parent(i + 1, j)) then
                       children1(i,j) = parent(i, j)
                       children1(i + 1,j) = RandomWithLimits(OptimizedParameterLowLimit(j), OptimizedParameterUpperLimit(j))
                   endif
                end Do

                children1(i, nfit + 1) = 0.d0
                children1(i + 1, nfit + 1) = 0.d0
                i = i + 2
            end Do

            return
        end subroutine discrete_crossover


        !*************************************************************************************************************************************************
        !> subroutine: simple_crossover
        !>
        !> Simple crossover - each child has genes from parents genes:
        !> k = int(nfit*uniform())
        !> child1=(genes_parent1[0:k], genes_parent2[k:num_par]
        !> child2=(genes_parent2[0:k], genes_parent1[k:num_par]
        !> Output: children - array of children chromosomes
        !>
        !> input variables:     parent:                         array of parental chromosomes
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Npar:                           number of parental couples
        !>
        !> output parameters:   children1:                      array of children chromosomes
        !<
        !> \author Irina Bernst
        !>
        !> \date 27.09.2010
        !>
        subroutine simple_crossover(parent, nfit, Npar, children1)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: Npar                                                                 !< number of parental couples
            integer :: i, j                                                                 !< loop variables
            integer :: kk                                                                   !< working variable
            real*8 :: p1, p2                                                                !< working variables
            real*8, dimension(Npar, nfit + 1) :: parent, children1                          !< array of parental and children chromosomes

            i = 1
            Do While (i <= Npar - 1)
                p1 = 1
                p2 = float(Npar)
                kk = int(RandomWithLimits(p1, p2))
                Do j = 1, nfit
                    if (j < kk) then
                        children1(i, j) = parent(i, j)
                        children1(i + 1, j) = parent(i + 1, j)
                    else
                        children1(i, j) = parent(i + 1, j)
                        children1(i + 1, j) = parent(i, j)
                    endif
                end Do
                children1(i, nfit + 1) = 0.d0
                children1(i + 1, nfit + 1) = 0.d0
                i = i + 2
            end Do
            return
        end subroutine simple_crossover


        !*************************************************************************************************************************************************
        !> subroutine: SBX_crossover
        !>
        !> Simulated Binary crossover - each child has genes from parents genes:
        !> ncr = [2-5] - parameter of crossover
        !> u(0,1) = uniform()
        !>
        !> Output: children - array of children chromosomes
        !>
        !> input variables:     parent:                         array of parental chromosomes
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Npar:                           number of parental couples
        !>
        !> output parameters:   children1:                      array of children chromosomes
        !<
        !> \author Irina Bernst
        !>
        !> \date 27.09.2010
        !>
        subroutine SBX_crossover(parent, nfit, Npar, children1)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: Npar                                                                 !< number of parental couples
            integer :: i, j                                                                 !< loop variables
            integer :: ncr                                                                  !< working variable
            real*8 :: xx                                                                    !< working variable
            real*8 :: u, beta                                                               !< help variables
            real*8 :: p1, p2                                                                !< working variables
            real*8, dimension(Npar, nfit + 1) :: parent, children1                          !< array of parental and children chromosomes

            ncr = 5
            i = 1
            Do While (i <= Npar - 1)
                p1 = 0
                p2 = 1
                u = RandomWithLimits(p1, p2)
                if (u <= 0.5d0) then
                    beta = (2.d0 * u)**(1.d0/(ncr + 1))
                else
                    beta = (1.d0/(2.d0 * (1.d0 - u)))**(1.d0/(ncr + 1))
                endif

                Do j = 1, nfit
                   children1(i, j) = 0.5d0 * (parent(i, j) * (1.d0 - beta) + parent(i + 1, j)*(1.d0 + beta))
                   children1(i + 1, j) = 0.5d0 * (parent(i + 1, j) * (1.d0 - beta) + parent(i, j) * (1.d0 + beta))

                   xx = children1(i, j)
                   call check_parameters(children1(i, j), OptimizedParameterLowLimit(j), OptimizedParameterUpperLimit(j))
                   if (xx /= children1(i, j)) then
                      ncr = 2
                   endif

                   xx = children1(i + 1, j)
                   call check_parameters(children1(i + 1, j), OptimizedParameterLowLimit(j), OptimizedParameterUpperLimit(j))
                   if (xx /= children1(i + 1, j)) then
                      ncr = 2
                   endif
                end Do
                children1(i, nfit + 1) = 0.d0
                children1(i + 1, nfit + 1) = 0.d0
                i = i + 2
            end Do
            return
        end subroutine SBX_crossover


        !*************************************************************************************************************************************************
        !> subroutine: Mutation
        !>
        !> each mutant has uniform()*genes from parents genes;
        !>
        !> Output: mutant - massiv of mutanted chromosomes
        !>
        !> input variables:     children:                       array of parental chromosomes
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Npar:                           number of parental couples
        !>                      Nmut:                           number of mutants
        !>
        !> output parameters:   mutant1:                        array of mutanted chromosomes
        !<
        !> \author Irina Bernst
        !>
        !> \date 28.09.2010
        !>
        subroutine Mutation(children, nfit, Npar, Nmut, mutant1)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: Npar                                                                 !< number of parental couples
            integer :: Nmut                                                                 !< number of mutants
            integer :: i, k1, k2                                                            !< working variables
            real*8 :: p1, p2, p3                                                            !< working variables
            real*8, dimension(Npar, nfit + 1) :: children                                   !< array of parental and children chromosomes
            real*8, dimension(Nmut, nfit + 1) :: mutant1                                    !< array of mutanted chromosomes

            Do i = 1, Nmut
                p1 = 1
                p2 = float(Npar)
                p3 = float(nfit)
                k1 = int(RandomWithLimits(p1, p2))
                k2 = int(RandomWithLimits(p1, p3))

                mutant1(i, 1:k2) = children(k1, 1:k2)
                mutant1(i, k2) = RandomWithLimits(OptimizedParameterLowLimit(k2), OptimizedParameterUpperLimit(k2))
                mutant1(i,(k2 + 1):nfit) = children(k1, (k2 + 1):nfit)
            end Do
            return
        end subroutine Mutation


        !*************************************************************************************************************************************************
        !> subroutine: Mutation1
        !>
        !> each mutant is random from distribution near boundaries of parametric space;
        !>
        !> Output: mutant - massiv of mutanted chromosomes
        !>
        !> input variables:     nfit:                           number of parameters which should be optimized
        !>                      Npar:                           number of parental couples
        !>                      Nmut:                           number of mutants
        !>
        !> output parameters:   mutant1:                        array of mutanted chromosomes
        !<
        !> \author Irina Bernst
        !>
        !> \date 02.11.2010
        !>
        subroutine Mutation1(nfit, Nmut, mutant1)

            use Variables

            implicit none
            integer :: nfit, nira1                                                          !< number of free parameters
            integer :: Nmut                                                                 !< number of mutants
            integer :: i, j                                                                 !< loop variables
            real*8 :: p0, p1, p2                                                            !< working variables
            real*8, dimension(Nmut, nfit + 1) :: mutant1                                    !< array of mutanted chromosomes

            Do i = 1, Nmut
                Do j = 1, nfit
                    p0 = (OptimizedParameterUpperLimit(j) - OptimizedParameterLowLimit(j)) * 0.05d0
                    p1 = 1
                    p2 = 100
                    nira1 = int(RandomWithLimits(p1, p2))
                    if (nira1 < 50) then
                        mutant1(i, j) = RandomWithLimits(OptimizedParameterLowLimit(j), OptimizedParameterLowLimit(j) + p0)
                    else
                        mutant1(i, j) = RandomWithLimits(OptimizedParameterUpperLimit(j) - p0, OptimizedParameterUpperLimit(j))
                    endif
                end Do
            end Do
            return
        end subroutine Mutation1


        !*************************************************************************************************************************************************
        !> subroutine: Mutation2
        !>
        !> each mutant is random from distribution near boundaries of parametric space;
        !>
        !> Output: mutant - massiv of mutanted chromosomes
        !>
        !> input variables:     nfit:                           number of parameters which should be optimized
        !>                      pmin:                           ?
        !>                      Nmut:                           number of mutants
        !>
        !> output parameters:   mutant1:                        array of mutanted chromosomes
        !<
        !> \author Irina Bernst
        !>
        !> \date 02.11.2010
        !>
        subroutine Mutation2(nfit, pmin, Nmut, mutant1)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: Nmut                                                                 !< number of mutants
            integer :: i, j                                                                 !< loop variables
            real*8 :: p1, p2                                                                !< working variables
            real*8, dimension(nfit) :: pmin                                                 !< ?
            real*8, dimension(Nmut, nfit + 1) :: mutant1                                    !< array of mutanted chromosomes

            Do i = 1, Nmut
                Do j = 1, nfit
                    p1 = pmin(j) - pmin(j) * 0.1d0
                    if (p1 < OptimizedParameterLowLimit(j)) then
                        p1 = OptimizedParameterLowLimit(j)
                    endif
                    p2 = pmin(j) + pmin(j) * 0.1d0
                    if (p2 > OptimizedParameterUpperLimit(j)) then
                        p2 = OptimizedParameterUpperLimit(j)
                    endif
                    mutant1(i, j) = RandomWithLimits(p1, p2)
                end Do
            end Do
            return
        end subroutine Mutation2


        !*************************************************************************************************************************************************
        !> subroutine: check_chromosomes
        !>
        !> Checking of the chromosomes, if two or more chromosomes are the same than to take a new chromosomes from popul-array
        !>
        !> input variables:     chromo:                         array containing the parameter which should be optimized and values of optimization func.
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Nchro:                          initial number of chromosomes in population
        !>                      Nchro1:                         final number of chromosomes in population
        !>                      ma:                             total number of all parameter
        !>
        !> subroutine requires: chromo1:                        modified array chromo
        !>
        !> \author Irina Bernst
        !>
        !> \date 28.10.2010
        !>
        subroutine check_chromosomes(chromo, nfit, Nchro, Nchro1, chromo1)

            use Variables
            use FunctionCalling

            implicit none
            integer :: nfit                                                                 !< number of parameters which should be optimized
            integer :: Nchro                                                                !< initial number of chromosomes in population
            integer :: Nchro1                                                               !< final number of chromosomes in population
            integer :: i, m                                                                 !< loop variables
            integer :: i1, i2, meti1, meti2, j, metj                                        !< working variables
            !real*8 :: a33                                                                  !< for Debugging
            real*8, dimension(Nchro, nfit+1) :: chromo                                      !< array containing the parameter which should be optimized
                                                                                            !< and values of optimization func.
            real*8, dimension(Nchro, nfit+1) :: chromo1                                     !< modified array chromo

            i1 = 1
            i2 = 1
            Nchro1 = Nchro
            chromo1 = 0.d0
            Do i = 1, Nchro

                ! Debug:
                ! print*, chromo(i,:)

                meti1 = 0
                meti2 = 0
                Do j = i, Nchro
                    metj = 0

                    ! Debug:
                    ! print*, chromo(i,:)
                    Do m = 1, nfit

                        ! Debug:
                        ! print*,'<<<', chromo(i,m), chromo(j,m)
                        if (chromo(i, m) == chromo(j, m)) then
                            metj = metj + 1
                        endif
                    end Do

                    ! Debug:
                    ! print*, metj
                    ! read*, a33
                    if (metj /= nfit) then
                        meti1 = meti1 + 1
                    else
                        meti2 = meti2 + 1
                    endif
                end Do
                if (meti1 == Nchro - i) then
                    chromo1(i1,:) = chromo(i,:)

                    ! Debug:
                    ! print*, i1, chromo1(i1,:)
                    i1 = i1 + 1
                endif
                if (meti2 > 1) then
                   i2 = i2 + 1
                endif
            end Do
            Nchro1 = i1 - 1


            !< check if parameters are within limits
            Do i = 1, Nchro
                Do j = 1, nfit
                    if (chromo1(i, j) < OptimizedParameterLowLimit(j)) then
                        chromo1(i, j) = OptimizedParameterLowLimit(j)
                    elseif (chromo1(i, j) > OptimizedParameterUpperLimit(j)) then
                        chromo1(i, j) = OptimizedParameterUpperLimit(j)
                    endif
                end Do
            end Do

            ! Debug:
            ! Do i=1,Nchro1
            !    print*, chromo1(i,:)
            ! end Do
            ! read*, a33
            return
        end subroutine check_chromosomes


        !*************************************************************************************************************************************************
        !> subroutine: all_distances
        !>
        !> Calculation of distance between ALL chromosomes and finding of maximal distance to check a convergence of the population
        !>
        !> Output: max_d - maximal distance between chromosomes in population
        !>
        !> input variables:     chromosomes:                    array of chromosomes
        !>                      nfit:                           number of parameters which should be optimized
        !>                      Nch:                            number of chromosomes
        !>
        !> output parameters:   max_d:                          maximal distance between chromosomes in population
        !<
        !> \author Irina Bernst
        !>
        !> \date 28.09.2010
        !>
        subroutine all_distances(chromosomes, nfit, Nch, max_d)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: Nch                                                                  !< number of parental couples
            integer :: i, ii                                                                !< loop variables
            real*8 :: max_d                                                                 !< maximal distance between chromosomes in population
            real*8, dimension(2) :: dist_chrom                                              !< distance between chromosomes
            real*8, dimension(Nch, nfit + 1) :: chromosomes                                 !< array of chromosomes

            i = 0
            max_d = 0.d0
            Do i = 1, Nch
                Do ii = 1, Nch
                     if (i /= ii) then
                         call distance(dist_chrom, chromosomes(i,:), chromosomes(ii,:), nfit)
                         if (dist_chrom(1) > max_d) then
                             max_d = dist_chrom(1)
                         endif
                     endif
                end Do
            end Do
            return
        end subroutine all_distances


        !*************************************************************************************************************************************************
        !> subroutine: CallGenAlg
        !>
        !> calls the different subroutines of the genetic algorithm
        !>
        !> input variables:     max_iter:           max. number of iterations
        !>                      nfit:               number of parameters which should be optimized
        !>                      ma:                 total number of all parameter
        !>                      Nch:                number of chromosomes
        !>                      Npar:               number of parental couples
        !>                      counter:            number of best sites
        !>                      colx:               number of columns belonging to the x-column
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>                      PlotIteration:      dummy argument
        !>                      PlotType:           kind of plotting
        !>                      xAxisLabel:         label for x-axis
        !>                      yAxisLabel:         label for y-axis
        !>                      zAxisLabel:         label for z-axis
        !>
        !> output variables:    chromosomes:        results of calculations after max_iter iterations (default max_iter=30)
        !>                                          -> array of parameters vectors (Nch)
        !>                      new_range:          new parametric space after genetic algorithm: dimension (nfit,2)
        !>                                          -> (:,1)-lower limits; (:,2)-upper limits
        !>
        !>                      Parallel computations of optimization function values ONLY in subroutine --opfun_calc-- !!!
        !>
        !> \author Thomas Moeller and Irina Bernst
        !>
        !> \date 24.09.2010
        !>
        subroutine CallGenAlg(max_iter, nfit, ma, Nch, Npar, Nmut, Npop, counter, Parameterset, colx, NumFile, MaxL, MaxCol, new_range_full, &
                              PlotIteration, PlotType, xAxisLabel, yAxisLabel, zAxisLabel, fitlog)


            implicit none
            integer :: i, j, k                                                              !< loop variables
            integer :: max_iter                                                             !< max. number of iterations
            integer :: nfit                                                                 !< number of free parameters
            integer :: ma                                                                   !< total number of all parameters
            integer :: counter                                                              !< counts the number of parameter sets
            integer :: LoopCounter                                                          !< bug fix
            integer :: colx                                                                 !< max. number of columns belonging to "x-column"
            integer :: NumFile                                                              !< number of exp. files
            integer :: MaxL                                                                 !< max. number of lines
            integer :: MaxCol                                                               !< max. number of columns belonging to "y-column"
            integer :: Nch                                                                  !< number of chromosomes in population
            integer :: Npar, Nchil                                                          !< number of parental couples, number of children after
                                                                                            !< checking
            integer :: Nmut                                                                 !< number of mutants
            integer :: Npop, Npop0                                                          !< number of chromosomes in population (next generation)
            integer :: fun_calls                                                            !< number of functions calls in GA
            integer :: metira, num_int                                                      !< ?
            integer :: nira1, nira2                                                         !< ?
            integer :: ira11, ira12, ira13, ira14, ira15, ira21, ira22, ira23               !< ?
            integer :: alloc_status                                                         !< status variable for allocation of memory
            integer :: i1, i2                                                               !< ?
            integer :: NumInputFiles                                                        !< needed for loop over input files
            integer :: PlotIteration                                                        !< flag for plotting the model function for each step
            integer :: PlotType                                                             !< get type of plot
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            integer :: allocstatus, deallocstatus                                           !< used for allocation/deallocation of memory
            real*8 :: opfun_min, opfun_max, opfun_sum                                       !< ?
            real*8 :: opfun_min_old, opfun_max_old, opfun_sum_old                           !< ?
            real*8 :: opfun_min_old0, opfun_max_old0, opfun_sum_old0                        !< ?
            real*8 :: improve_opfun_min, improve_opfun_max, improve_opfun_sum               !< ?
            real*8 :: improve_max_dist                                                      !< ?
            real*8 :: dummy                                                                 !< ?
            real*8 :: fun_min, fun_min_pred                                                 !< ?
            real*8 :: p1, p2, rat, re_Nch, re_nfit, delta_int                               !< ?
            real*8 :: LocalParam                                                            !< working variable for limit check
            real*8 :: LowLimit, UpperLimit                                                  !< lower and upper limit of the random number interval
            real*8 :: coef_mut                                                              !< coefficient of mutations (from 0.0 to 0.3; default is 0.2)
            real*8 :: rod_par, intervals, a1, a2                                            !< coefficient for number of parental couples
                                                                                            !< (from 0.5 to 0.9; default is 0.8)
            ! real*8 :: a33                                                                 !< DEBUGGING !!!
            real*8, dimension(2) :: dist_chrom                                              !< ?
            real*8, dimension(nfit) :: xx                                                   !< help array
            real*8, allocatable, dimension(:,:) :: chromosomes                              !< working array of chromosomes
            real*8, allocatable, dimension(:,:) :: old_chromosomes                          !< copy of chromosomes
            real*8, allocatable, dimension(:,:) :: parent                                   !< working array of parent-chromosomes
            real*8, allocatable, dimension(:,:) :: children, children1                      !< working array of child-chromosomes
            real*8, allocatable, dimension(:,:) :: mutant                                   !< working array of mutant-chromosomes
            real*8, allocatable, dimension(:,:) :: population, population1                  !< working array of population-chromosomes
            real*8, dimension(max_iter + 1) :: max_distance                                 !< array of max distances between chromosomes for each iter.
            real*8, dimension(nfit) :: Xmin, Xmin_pred                                      !< ?
            real*8, dimension(colx) :: posdatexp                                            !< array for point within observation data
            real*8, dimension(ma) :: acopy                                                  !< copy of total parameter list
            real*8, dimension(nfit, 2) :: new_range                                         !< array with new parametric space
            real*8, dimension(ma, 2) :: new_range_full                                      !< ?
            real*8, dimension(counter, ma) :: Parameterset                                  !< array containing the parameter set which fullfil the

            character(len=25) :: LongNumber1, LongNumber2                                   !< working variables
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=100) :: HelpString                                                !< string for number to string conversion
            character(len=5196) :: ListParamFormated                                        !< string for screen output of parameter set
            character(len=8192) :: fitlog                                                   !< path for log files

            logical :: IntegerTrue                                                          !< flag indicating if a number is an integer
            logical :: InitPlotFlag                                                         !< flag for saving plot for each iteration
            logical :: all_zero_flag                                                        !< flag for zero chromosomes


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< Genetic Algorithm loop
            !< Default number of the chromosomes
            if (Nch == 0) then                                                              !< use automatic definition for number of chromosomes
                Nch = 10
                if (nfit > 4 .and. nfit <= 10) then
                   Nch = int(nfit * 3)
                endif
                if (nfit > 10 .and. nfit <= 100) then
                   Nch = int(nfit * 2)
                endif
                if (nfit > 100) then
                   Nch = int(nfit / 2)
                endif
            endif


            !< write number of chromosomes to screen and log-file
            if (printflag) then
                print '(11x,"Number of chromosomes = ",I10)',Nch
                print '(" ")'
            endif
            write(logchannel,'(11x,"Number of chromosomes = ",I10)') Nch
            write(logchannel,'("  ")')


            !< allocate memory
            if (allocated(chromosomes)) then
                deallocate(chromosomes, old_chromosomes, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    write(logchannel,*)
                    write(logchannel,'(11x,"Error in subroutine CallGenAlg:")')
                    write(logchannel,'(13x,"Can not deallocate variable chromosomes etc.")')
                    write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'(13x,"deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'(11x,"Program aborted!")')

                    print '(" ")'
                    print '(11x,"Error in subroutine CallGenAlg:")'
                    print '(13x,"Can not deallocate variable chromosomes etc.")'
                    print '(13x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '(13x,"deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop '           Program aborted!'
                endif
            endif
            allocate(chromosomes(Nch, nfit + 1), old_chromosomes(Nch, nfit + 1), stat = allocstatus)
            if (allocstatus /= 0) then
                write(logchannel,*)
                write(logchannel,'(11x,"Error in subroutine CallGenAlg:")')
                write(logchannel,'(13x,"Can not allocate variable chromosomes etc.")')
                write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                write(logchannel,*)
                write(logchannel,'(13x,"allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'(11x,"Program aborted!")')

                print '(" ")'
                print '(11x,"Error in subroutine CallGenAlg:")'
                print '(13x,"Can not allocate variable chromosomes etc.")'
                print '(13x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '(13x,"allocstatus = ",I4)', allocstatus
                print '(" ")'
                stop '           Program aborted!'
            endif
            chromosomes = 0.d0
            old_chromosomes = 0.d0


            !< write on the screen what you do ..
            if (printflag) then
                print '(11x, "Iteration:", 20x, "chisq:", 5x, "Parameter:")'
                print '(11x, "Initialize model function ..", 20(" "), A1, $ )', char(13)
            endif
            write(logchannel,'(11x, "Iteration:", 20x, "chisq:", 5x, "Parameter:")')


            !< Parental couples
            rod_par = 0.8d0
            Npar = int(rod_par * Nch)
            if ((Npar/2) /= int(Npar/2)) then
               Npar = Npar - 1
            endif

            ! Debug:
            ! print*,'<<< Npar=', Npar


            !< Mutants
            coef_mut = 0.5d0
            Nmut = int(Nch * coef_mut)
            Npop0 = Nch + Npar + Nmut

            ! Debug:
            ! print*,'>>>> ', Npop0


            !< allocate memory for arrays parent, childer, mutant, population
            if (allocated(parent)) then
                deallocate(parent, children, children1, mutant, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    write(logchannel,*)
                    write(logchannel,'(11x,"Error in subroutine CallGenAlg:")')
                    write(logchannel,'(13x,"Can not deallocate variables parent etc.")')
                    write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'(13x,"deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'(11x,"Program aborted!")')

                    print '(" ")'
                    print '(11x,"Error in subroutine CallGenAlg:")'
                    print '(13x,"Can not deallocates variables parent etc.")'
                    print '(13x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '(13x,"deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop '           Program aborted!'
                endif
            endif
            allocate(parent(Npar, nfit + 1), children(Npar, nfit + 1), children1(Npar, nfit + 1), mutant(Nmut, nfit + 1), stat = alloc_status)
            if (alloc_status /= 0) then
                write(logchannel,'(" ")')
                write(logchannel,'(11x,"Error in subroutine CallGenAlg:")')
                write(logchannel,'(13x,"Can not allocate variables parent etc.")')
                write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'(13x,"allocstatus = ",I4)') alloc_status
                write(logchannel,'(" ")')
                write(logchannel,'(11x,"Program aborted!")')

                print '(" ")'
                print '(11x,"Error in subroutine CallGenAlg:")'
                print '(13x,"Can not allocate variables parent etc.")'
                print '(13x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '(13x,"allocstatus = ",I4)',alloc_status
                print '(" ")'
                stop ' Program aborted!'
            endif
            parent = 0.d0
            children = 0.d0
            mutant = 0.d0
            metira = 0

            opfun_max = 0.d0
            opfun_min = 1d10
            opfun_sum = 0.d0


            !< initialize ran1
            ! call random_seed()
            idum = (-1)
            call ran1(dummy, idum)


            !< define OptimizedParameter range and determine starting set of the chromosomes
            OptimizedParameterLowLimit = 0.d0
            OptimizedParameterUpperLimit = 0.d0
            j = 0
            Do i = 1, parameternumber
                if (ia(i)) then
                    j = j + 1
                    OptimizedParameterLowLimit(j) = paramset(3, i)
                    OptimizedParameterUpperLimit(j) = paramset(4, i)
                endif
            end Do


            !< initialize parameter vector
            chromosomes = 0.d0

            ! Debug:
            ! Do i = 1, Nch
            !    Do j = 1, nfit
            !       chromosomes(i,j) = RandomWithLimits(OptimizedParameterLowLimit(j), OptimizedParameterUpperLimit(j))
            !    end Do
            !    chromosomes(i,nfit+1)=0.d0
            ! end Do


            re_Nch = float(Nch)
            re_nfit = float(nfit)
            a1 = log10(re_Nch)
            a2 = log10(re_nfit)
            intervals = a1 / a2
            num_int = 0

            ! Debug:
            ! print*,intervals
            ! stop

            Do j = 1, nfit
                a1 = OptimizedParameterLowLimit(j)
                delta_int = (OptimizedParameterUpperLimit(j) - OptimizedParameterLowLimit(j))/intervals

                ! Debug:
                ! print*,j, OptimizedParameterLowLimit(j), a1+delta_int, a1+2*delta_int, a1 + 3 * delta_int
                ! read*, a33

                Do i = 1, Nch
                    if (a1 + (num_int + 1) * delta_int > OptimizedParameterUpperLimit(j)) then
                        a2 = OptimizedParameterUpperLimit(j)
                    else
                        a2 = a1 + (num_int + 1) * delta_int
                    endif

                    ! Debug:
                    ! print*,j, a1+num_int*delta_int, a2
                    ! read*, a33

                    chromosomes(i,j) = RandomWithLimits(a1 + num_int * delta_int, a2)
                    num_int = num_int + 1
                    if (num_int > intervals) then
                        num_int = 0
                    endif
                end Do
            end Do
            chromosomes(Nch, nfit + 1) = 0.d0

            ! Debug:
            ! print*, '>>>Chromosomes'
            ! Do i = 1, Nch
            !    print*,'>>',chromosomes(i,:)
            ! end Do
            ! read*, a33


            !< Calculation of optimization function for chromosomes -----> opfun = chromosomes(1:Nch, nfit + 1)
            call opfun_calc(NumFile, MaxL, MaxCol, nfit, Nch, ma, chromosomes(1:Nch, :))


            !< Sorting of chromosomes on function values -> minimal value: above
            call sort_population(chromosomes(1:Nch,:), nfit, Nch)

            ! Debug:
            ! Do i = 1, Nch
            !     print*,'>>',chromosomes(i,:)
            ! end Do
            ! read*, a33

            ira11 = 0
            ira12 = 0
            ira13 = 0
            ira14 = 0
            ira15 = 0

            ira21 = 0
            ira22 = 0
            ira23 = 0

            fun_calls = Nch

            opfun_max_old0 = opfun_max
            opfun_min_old0 = opfun_min
            opfun_sum_old0 = opfun_sum

            num_iter = 1
            fun_min = 1.d99
            Do While (num_iter <= max_iter .and. fun_min > chilim)
                if (num_iter /= 0) then
                    opfun_max_old = opfun_max
                    opfun_min_old = opfun_min
                    opfun_sum_old = opfun_sum

                    opfun_max = 0.d0
                    opfun_min = 1.d10
                    opfun_sum = 0.d0

                    Do i = 1, Nch
                       if (opfun_max < chromosomes(i, nfit + 1)) then
                           opfun_max = chromosomes(i, nfit + 1)
                       endif
                       if (opfun_min > chromosomes(i, nfit + 1)) then
                           opfun_min = chromosomes(i, nfit + 1)
                       endif
                       opfun_sum = opfun_sum + chromosomes(i, nfit + 1)
                    end Do
                    improve_opfun_max = 100.d0 - (opfun_max * 100.d0 / opfun_max_old0)
                    improve_opfun_min = 100.d0 - (opfun_min * 100.d0 / opfun_min_old0)
                    improve_opfun_sum = 100.d0 - (opfun_sum * 100.d0 / opfun_sum_old0)
                endif


                !< Now are algorithms for selection of parental population
                parent = 0.d0
                p1 = 0
                p2 = 100
                nira1 = int(RandomWithLimits(p1, p2))

                ! Debug:
                ! call ran1(dummy, idum)
                ! nira1 = 100 * dummy
                ! print*,'<<< nira1= ', nira1
                ! nira1 = 90

                if (nira1 <= 20) then

                    ! Debug:
                    ! print*, 'Selection: random'

                    call Random_selection(chromosomes(1:Nch,:), nfit, Nch, Npar, parent)
                    ira11 = ira11 + 1

                    ! Debug:
                    ! print*,'<<< Parent :'
                    ! Do i = 1, Npar
                    !     print*,parent(i,:)
                    ! end Do
                    ! stop
                endif
                if (nira1 > 20.and.nira1 <= 40) then

                    ! Debug:
                    ! print*, 'Selection: elite'

                    call Elit_selection(chromosomes(1:Nch,:), nfit, Nch, Npar, parent)
                    ira12 = ira12 + 1

                    ! Debug:
                    ! print*,'<<< Parent :'
                    ! Do i = 1, Npar
                    !     print*,parent(i,:)
                    ! end Do
                    ! stop
                endif
                if (nira1 > 40.and.nira1 <= 60) then

                    ! Debug:
                    ! print*, 'Selection: tournament'

                    call Tournament_selection(chromosomes(1:Nch,:), nfit, Nch, Npar, parent)
                    ira13 = ira13 + 1

                    ! Debug:
                    ! print*,'<<< Parent :'
                    ! Do i = 1, Npar
                    !     print*,parent(i,:)
                    ! end Do
                    ! stop
                endif
                if (nira1 > 60.and.nira1 <= 65) then

                    ! Debug:
                    ! print*, 'Selection: inbreeding'

                    call Inbreeding (chromosomes(1:Nch,:), nfit, Nch, Npar, parent)
                    ira14 = ira14 + 1

                    ! Debug:
                    ! print*,'<<< Parent :'
                    ! Do i = 1, Npar
                    !     print*,parent(i,:)
                    ! end Do
                    ! stop
                endif
                if (nira1 > 65) then

                    ! Debug:
                    ! print*, 'Selection: outbreeding'

                    call Outbreeding(chromosomes(1:Nch,:), nfit, Nch, Npar, parent)
                    ira15 = ira15 + 1

                    ! Debug:
                    ! print*,'<<< Parent :'
                    ! Do i = 1, Npar
                    !     print*, parent(i,:)
                    ! end Do
                    ! stop
                endif
                parent(Npar,:) = chromosomes(1,:)

                ! Debug:
                ! print*,'<<< Parent :'
                ! Do i = 1, Npar
                !     print*, parent(i,:)
                ! end Do
                ! read*, a33


                !< Now are algorithms for crossover
                children = 0.d0
                p1 = 0
                p2 = 100
                nira2 = int(RandomWithLimits(p1, p2))

                ! Debug:
                ! call ran1(dummy, idum)
                ! nira1 = 100 * dummy
                ! print*,'<<< nira2= ', nira2
                ! stop
                ! nira2=50

                if (nira2 <= 20) then

                    ! Debug:
                    ! print*, 'Reproduction: discrete crossover'

                    call discrete_crossover(parent(1:Npar,:), nfit, Npar, children)
                    ira21 = ira21 + 1

                    ! Debug:
                    ! print*,'<<< Children :'
                    ! Do i = 1, Npar
                    !     print*, children(i,:)
                    ! end Do
                    ! stop
                endif
                if (nira2 > 20.and.nira2 <= 40) then

                    ! Debug:
                    ! print*, 'Reproduction: simple crossover'

                    call simple_crossover(parent(1:Npar,:), nfit, Npar, children)
                    ira22 = ira22 + 1

                    ! Debug:
                    ! print*,'<<< Children :'
                    ! Do i = 1, Npar
                    !     print*, children(i,:)
                    ! end Do
                    ! stop
                endif
                if (nira2 > 40) then

                    ! Debug:
                    ! print*, 'Reproduction: SBX-crossover'

                    call SBX_crossover(parent(1:Npar,:), nfit, Npar, children)
                    ira23 = ira23 + 1

                    ! Debug:
                    ! print*,'<<< Children :'
                    ! Do i = 1, Npar
                    !     print*, children(i,:)
                    ! end Do
                    ! stop
                endif

                ! Debug:
                ! print*,'<<< Children :', Npar
                ! Do i = 1, Npar
                !    print*, children(i,:)
                ! end Do


                !< Checking of children
                children1 = 0.d0
                call check_chromosomes(children(1:Npar,:), nfit, Npar, Nchil, children1)

                ! Debug:
                ! print*, '>>>Children1 :', Nchil
                ! Do i = 1, Nchil
                !    print*,'>>',children1(i,:)
                ! end Do


                !< Calculation of optimization function for children chromosomes -----> opfun = children(1:Npar, nfit + 1)
                call opfun_calc(NumFile, MaxL, MaxCol, nfit, Nchil, ma, children1(1:Nchil,:))

                ! Debug:
                ! print*,'<<< Children with optimization function:'
                ! Do i = 1, Nchil
                !     print*, children1(i,:)
                ! end Do

                fun_calls = fun_calls + Nchil
                call Mutation(children1(1:Nchil,:), nfit, Nchil, Nmut, mutant)
                call opfun_calc(NumFile, MaxL, MaxCol, nfit, Nmut, ma, mutant(1:Nmut,:))
                fun_calls = fun_calls + Nmut


                !< population = chromosomes + children + mutant -----> next generation of the chromosomes
                Npop = Nch + Nchil + Nmut

                ! Debug:
                ! print*,'>>>> Npop = ', Npop

                if (allocated(population)) then
                    deallocate(population, population1, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        write(logchannel,*)
                        write(logchannel,'(11x,"Error in subroutine CallGenAlg:")')
                        write(logchannel,'(13x,"Can not deallocate variables population etc.")')
                        write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'(13x,"deallocstatus = ",I4)') deallocstatus
                        write(logchannel,'(" ")')
                        write(logchannel,'(11x,"Program aborted!")')

                        print '(" ")'
                        print '(11x,"Error in subroutine CallGenAlg:")'
                        print '(13x,"Can not deallocates variables population etc.")'
                        print '(13x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '(13x,"deallocstatus = ",I4)',deallocstatus
                        print '(" ")'
                        stop '           Program aborted!'
                    endif
                endif
                allocate(population(Npop, nfit + 1), population1(Npop, nfit + 1), stat = alloc_status)
                if (alloc_status /= 0) then
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine CallGenAlg:")')
                    write(logchannel,'(2x,"Can not allocate variables population etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("allocstatus = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine CallGenAlg:")'
                    print '(2x,"Can not allocate variables population etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("allocstatus = ",I4)',alloc_status
                    print '(" ")'
                    stop ' Program aborted!'
                endif
                population  = 0.d0
                population1 = 0.d0

                i1 = 1
                i2 = 1
                Do i = 1, Npop
                    if (i <= Nch) then
                        population(i,:) = chromosomes(i,:)

                        ! Debug:
                        ! if (num_iter > 1) then
                        !     print*,i
                        !     print*,population1(i,:)
                        ! endif
                    endif
                    if (i > Nch.and.i <= Nch + Nchil) then
                        population(i,:) = children1(i1,:)
                        i1 = i1 + 1
                    endif
                    if (i > Nch + Nchil) then
                        population(i,:) = mutant(i2,:)
                        i2 = i2 + 1
                    endif

                    ! Debug:
                    ! print*,i
                    ! print*,population(i,:)
                end Do

                ! Debug:
                ! print*,Npop
                ! print*,'<<< Population = Chromosomes + Children + Mutants:'
                ! Do i = 1, Npop
                !     print*, population(i,:)
                ! end Do


                !< Sorting in array population on value of optimization function --> population(:,nfit+1): above minimal value
                call sort_population(population(1:Npop,:), nfit, Npop)

                ! Debug:
                ! print*,'<<< Population after sorting:'
                ! print*, Npop
                ! Do i = 1, Npop
                !     print*, population(i,:)
                ! end Do
                ! num_iter = num_iter + 1


                !< Selection and Truncation for next generation
                call all_distances(chromosomes(1:Nch,:), nfit, Nch, max_distance(num_iter))

                ! Debug:
                ! print*,'<<<< Number of iteration = ', num_iter
                ! print*,'<<<< Max distance between chromosomes improved on (%): ', improve_max_dist
                ! print*,'<<<< Maximal value of ortimization function : ', opfun_max
                ! print*,'<<<< Minimal value of ortimization function : ', opfun_min
                ! print*,'*** Number function calls = ', fun_calls
                ! print*,'**************************************************************'
                ! print*,'Number of used methods'
                ! print*,'*** Selection: random ', ira11
                ! print*,'*** Selection: elite  ', ira12
                ! print*,'*** Selection: tournament ', ira13
                ! print*,'*** Selection: inbreeding', ira14
                ! print*,'*** Selection: outbreeding ', ira15
                ! print*,'*** Reproduction: discrete crossover ',ira21
                ! print*,'*** Reproduction: simple crossover ',ira22
                ! print*,'*** Reproduction: SBX-crossover ',ira23
                ! print*,'**************************************************************'
                ! population1 = 0.d0
                ! call check_chromosomes(population(1:Npop,1:(nfit+1)), nfit, Npop, Npop1, population1)
                ! print*,'<<< Population after checking:'
                ! print*, Npop1
                ! Do i = 1, Npop1
                !     print*, population1(i,:)
                ! end Do
                ! read*, a33

                old_chromosomes = chromosomes
                chromosomes = 0.d0
                i1 = 1
                i2 = 1
                rat = 0.01d0
                LoopCounter = 0
                Do While (i2 <= Nch .and. LoopCounter < 100)
                    LoopCounter = LoopCounter + 1
                    if (i1 <= Npop) then

                        ! Debug:
                        ! print*, i2, chromosomes(i2,:)
                        ! read*, a33

                        call distance(dist_chrom, population(i1,:), population(i1 + 1,:), nfit)
                        if (dist_chrom(1) == 0) then
                            i1 = i1 + 1

                            ! Debug:
                            ! print*, i1, population(i1,:)
                            ! read*, a33

                            if (i1 == (Npop - 1) .and. i2 < Nch) then
                                i1 = 2
                            endif
                        else
                            !if (dist_chrom(1) > max_distance(num_iter) * rat.and.dist_chrom(2) /= 0.d0) then
                            if (dist_chrom(1) > max_distance(num_iter) * rat) then
                                chromosomes(i2,:) = population(i1,:)
                                if (i1 < Npop .and. i2 < Nch) then
                                    chromosomes(i2 + 1,:) = population(i1 + 1,:)
                                endif
                                i1 = i1 + 2
                                i2 = i2 + 2
                            else
                                chromosomes(i2,:) = population(i1,:)
                                i1 = i1 + 2
                                i2 = i2 + 1
                            endif
                        endif
                        if (i1 == (Npop - 1) .and. i2 < Nch) then
                            rat = rat * 0.1d0

                            ! Debug:
                            ! if (rat <= 1e-04) then
                            !     rat = 1e-03
                            ! endif
                            ! print*,'*********<<<<< rat =', rat

                            i1 = 2
                        endif
                    endif
                end Do


                !< test if upper loop was truncated
                if (LoopCounter >= 100) then
                    Do i = 1, Nch
                        all_zero_flag = .true.
                        Do j = 1, (nfit  + 1)
                            if (chromosomes(i,j) /= 0.d0) then
                                all_zero_flag = .false.
                                exit
                            endif
                        end Do
                        if (all_zero_flag) then
                            chromosomes(i,:) = old_chromosomes(i,:)
                        endif
                    end Do
                endif

                ! Debug:
                ! print*,'<<< Chromosomes of new generation:'
                ! Do i = 1, Nch
                !    print*, chromosomes(i,:)
                ! end Do
                ! read*, a33

                call sort_population(chromosomes(1:Nch,:), nfit, Nch)

                ! Debug:
                ! print*,'<<< New generation of the chromosomes:'
                ! Do i = 1, Nch
                !     print*, chromosomes(i,:)
                ! end Do
                ! read*, a33

                if (num_iter == 1) then
                    Xmin(:) = chromosomes(1, 1:nfit)
                    fun_min = chromosomes(1, nfit + 1)
                else

                    ! Debug:
                    ! print*, Xmin_pred
                    ! print*, Xmin
                    ! read*, a33

                    Xmin_pred(:) = Xmin(:)
                    fun_min_pred = fun_min
                endif
                if (num_iter > 1) then
                    improve_max_dist = 100.d0 - (max_distance(num_iter) * 100.d0 / max_distance(1))
                    Xmin(:) = chromosomes(1, 1:nfit)                                        !< get best parameter set
                    fun_min = chromosomes(1, nfit + 1)                                      !< get corresponding chi2 value


                    !< determine new ranges
                    call analysis_GA(chromosomes(1:Nch,:), Nch, improve_max_dist, nfit, new_range, metira)
                endif


                !< print what you do!
                if (num_iter > 0) then
                    k = 0
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            a(j) = BestSitesParamSet(1, k + 1)
                        endif
                    end Do


                    !< build list with fit parameters
                    k = 0
                    ListParamFormated = ""
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                            if (IntegerTrue) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(j))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(a(j))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(j)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) a(j)
                                endif
                            endif
                            if (k == 1) then
                                ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                            else
                                ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                            endif
                        endif
                    end Do


                    !< print status of iteration process .
                    if (printflag) then
                        print '(11x,I10,ES26.15,5x,A)', num_iter, BestSitesParamSet(1, 1), trim(adjustl(ListParamFormated))
                    endif
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("  ")')
                    write(paramchannel,'(123("*"))')
                    write(paramchannel,'("Iteration: ",I5,",  chi^2 = ",ES25.15)') num_iter, BestSitesParamSet(1, 1)
                    write(logchannel,'(11x,I10,ES26.15,5x,A)') num_iter, BestSitesParamSet(1, 1), trim(adjustl(ListParamFormated))


                    !< write actual parameters to files
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("Parameters: ",A)') trim(adjustl(ListParamFormated))
                    write(paramchannel,'(123("-"))')
                    write(paramchannel,'("  ")')


                    !< save current experimental x point of the first experimental file to variable posdatexp
                    posdatexp(1:colx) = 0.d0


                    !< call subroutine to write current values of the parameter to file
                    write(paramchannel,'("-",61(" -"))')
                    Do NumInputFiles = 1, NumberInputFiles
                        write(paramchannel,'("Input-File ",I5,":  , file: ",A)') NumInputFiles, trim(adjustl(FitFktInput(NumInputFiles)))
                        write(paramchannel,'("  ")')
                        write(paramchannel,'("-start_input-file",106("-"))')
                        call WriteParameter(paramchannel, .true., colx, posdatexp, parameternumber, a, NumInputFiles)
                        write(paramchannel,'("-end_input-file",108("-"))')
                    end Do


                    !< plot experimental data, model function, and chi**2
                    if (PlotIteration == 0) then
                        if (num_iter == 1) then
                            InitPlotFlag = .true.
                        else
                            InitPlotFlag = .false.
                        endif
                        call PlotFitFunction(InitPlotFlag, xAxisLabel, yAxisLabel, zAxisLabel)
                    endif
                endif

                ! Debug:
                ! print*,'<<<< Number of iteration: ',num_iter

                num_iter = num_iter + 1
            end Do


            !< print reason for stop of iteration
            write(paramchannel,'("  ")')
            write(paramchannel,'(123("="))')
            write(paramchannel,'("  ")')
            if (BestSitesParamSet(1, 1) <= chilim) then
                write(LongNumber1,'(ES25.15)') BestSitesParamSet(1, 1)
                write(LongNumber2,'(ES25.15)') chilim
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)', trim(adjustl(LongNumber1)), &
                                                                                                            trim(adjustl(LongNumber2))
                endif
                write(logchannel,'("  ")')
                write(logchannel,'(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)') trim(adjustl(LongNumber1)), &
                                                                                                            trim(adjustl(LongNumber2))

            elseif (num_iter >= max_iter) then
                if (printflag) then
                    print '(" ")'
                    print '(11x, "Iteration stopped. Number of iterations is equal to max. number of iterations = ", I6)', MaxGeneticIter
                endif
                write(logchannel,'("  ")')
                write(logchannel,'(11x, "Iteration stopped. Number of iterations is equal to max. number of iterations = ", I6)') MaxGeneticIter
            endif

            ! Debug:
            !    print*, '********************************************************************************'
            !    print*,'*** Number of function calls = ', fun_calls
            !    print*, '*** New range of parametric space :'
            !    print*, new_range(:,1)
            !    print*, new_range(:,2)
            !    print*, '********************************************************************************'


            !< determine the number of best sites
            j = 0
            Do i = 1, QualityLimit
                if (BestSitesParamSet(i, 1) /= 1.d99) then
                    j = j + 1
                endif
            end Do
            QualityLimit = j


            !< define final range array
            new_range_full = 0.d0
            k = 0
            Do j = 1, ma
                if (ia(j)) then
                    k = k + 1
                    LowLimit = new_range(k, 1)
                    UpperLimit = new_range(k, 2)

                    ! LowLimit = paramset(3, j)
                    ! UpperLimit = paramset(4, j)
                else
                    LowLimit = paramset(3, j)
                    UpperLimit = paramset(4, j)
                endif


                !< check under and overflow
                if (abs(LowLimit) > HUGE(0.d0)) then
                    LowLimit = sign(HUGE(0.d0), LowLimit)
                endif
                if (abs(LowLimit) < TINY(0.d0)) then
                    LowLimit = 0.d0     !sign(TINY(0.d0), LowLimit)
                endif
                if (abs(UpperLimit) > HUGE(0.d0)) then
                    UpperLimit = sign(HUGE(0.d0), UpperLimit)
                endif
                if (abs(UpperLimit) < TINY(0.d0)) then
                    UpperLimit = 0.d0   !sign(TINY(0.d0), UpperLimit)
                endif


                !< store new range
                new_range_full(j, 1) = DMIN1(LowLimit, UpperLimit)
                new_range_full(j, 2) = DMAX1(LowLimit, UpperLimit)


                !< check, if new range includes best sites
                Do i = 1, QualityLimit
                    xx = BestSitesParamSet(i, 2:)
                    if (ia(j)) then
                        LocalParam = xx(k)
                    else
                        LocalParam = a(j)
                    endif
                    if (LocalParam < new_range_full(j, 1)) then
                        new_range_full(j, 1) = LocalParam
                    endif
                    if (LocalParam > new_range_full(j, 2)) then
                        new_range_full(j, 2) = LocalParam
                    endif
                end Do
            end Do


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< select best sites
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(11x,"Best results:")'
            endif
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')
            write(logchannel,'(11x,"Best results:")')


            !< write best results to screen and to log-file
            Do i = 1, QualityLimit
                acopy = a
                fun_min = BestSitesParamSet(i, 1)
                xx = BestSitesParamSet(i, 2:)


                !< build list with fit parameters
                k = 0
                ListParamFormated = ""
                Do j = 1, parameternumber
                    if (ia(j)) then
                        k = k + 1
                        acopy(j) = xx(k)

                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                        if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                            .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(acopy(j))
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) int(acopy(j))
                            endif
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) acopy(j)
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) acopy(j)
                            endif
                        endif
                        if (k == 1) then
                            ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                        else
                            ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                        endif
                    endif
                end Do
                Parameterset(i, :) = acopy(:)


                !< print best results to screen and to log-file
                if (printflag) then
                    print '(13x,"Site number: ",I4,", chi^2 = ",ES25.15,",  Parameterset = ",A)', i, fun_min, trim(adjustl(ListParamFormated))
                endif
                write(logchannel,'(13x,"Site number: ",I4,", chi^2 = ",ES25.15,",  Parameterset = ",A)') i, fun_min, trim(adjustl(ListParamFormated))


                !< create list for lower limits
                if (RangeFlag) then
                    if (QualityLimit == 1) then
                        if (printflag) then
                            print '(" ")'
                            print '(" ")'
                            print '(11x,"New limits for the optimized parameters:")'
                        endif
                        write(logchannel,'(" ")')
                        write(logchannel,'(" ")')
                        write(logchannel,'(11x,"New limits for the optimized parameters:")')
                    endif
                    k = 0
                    ListParamFormated = ""
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                            if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(new_range_full(j, 1))
                                if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                    write(HelpString, *) int(new_range_full(j, 1))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) new_range_full(j, 1)
                                if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                    write(HelpString, *) new_range_full(j, 1)
                                endif
                            endif
                            if (k == 1) then
                                ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                            else
                                ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                            endif
                        endif
                    end Do
                    if (QualityLimit == 1) then
                        if (printflag) then
                            print '(13x,"Lower limits:  ",A)', trim(adjustl(ListParamFormated))
                        endif
                        write(logchannel,'(13x,"Lower limits:  ",A)') trim(adjustl(ListParamFormated))
                    else
                        if (printflag) then
                            print '(32x,"Lower limits:  ",A)', trim(adjustl(ListParamFormated))
                        endif
                        write(logchannel,'(32x,"Lower limits:  ",A)') trim(adjustl(ListParamFormated))
                    endif


                    !< create list for upper limits
                    k = 0
                    ListParamFormated = ""
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                            if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(new_range_full(j, 2))
                                if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                    write(HelpString, *) int(new_range_full(j, 2))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) new_range_full(j, 2)
                                if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                    write(HelpString, *) new_range_full(j, 2)
                                endif
                            endif
                            if (k == 1) then
                                ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                            else
                                ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                            endif
                        endif
                    end Do
                    if (QualityLimit == 1) then
                        if (printflag) then
                            print '(13x,"Upper limits:  ",A)', trim(adjustl(ListParamFormated))
                        endif
                        write(logchannel,'(13x,"Upper limits:  ",A)') trim(adjustl(ListParamFormated))
                    else
                        if (printflag) then
                            print '(32x,"Upper limits:  ",A)', trim(adjustl(ListParamFormated))
                            print '(" ")'
                        endif
                        write(logchannel,'(32x,"Upper limits:  ",A)') trim(adjustl(ListParamFormated))
                        write(logchannel,'(" ")')
                    endif
                endif
            end Do


            !< clear screen massage
            if (printflag) then
                print '(A, 120(" "), A, $)',char(13), char(13)
            endif
            return
        end subroutine CallGenAlg
end Module GenVariables
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: Algorithm
!>
!>         Module contains the main subroutine used to start the different versions of the Genetic algorithm
!>
!>
!> \author Thomas Moeller
!>
!> \date 2014-08-31
!>
Module Algorithm

    use Variables
    use GenVariables

    implicit none

    contains


        !*************************************************************************************************************************************************
        !> subroutine: MainAlg
        !>
        !> main subroutine which starts the genetic algorithm
        !>
        !>
        !> input variables:         printflagNum:           flag for screen output 1 (=yes) or 0 (=no)
        !>                          LastAlgorithmNum:       number of last algorithm
        !>                          ParamSetCounter:        number of best sites
        !>                          FinalParameterSet:      array containing the parameter set with SampleslogL(i) >= logZ
        !>                          chilm:                  user defined abort criteria for chi**2
        !>                          numiter:                max. number of iterations
        !>                          GenCounter:             counts number of calls
        !>                          NumChrom:               number of chromosomes
        !>                          DeterminationChi2:      method being used for the determination of chi^2
        !>                          PlotIterationOrg:       plot model function for each iteration set 1(=yes) or 0(=no)
        !>                          PlotTypeOrg:            get type of plot
        !>                          fitlog:                 path for log-file containing the current values of chi**2
        !>                          NumberInputFilesorg:    number of input files for the external model program
        !>                          NumberOutputFilesOrg:   number of output files for the external model program
        !>                          ParallelizationFlagorg: contains the number of processors used for parallelization
        !>                          JobIDorg:               job identification number
        !>                          MaxInputLinesOrg:       max number of lines in an input file
        !>                          MaxParameterOrg:        max number of parameters in a line of an input file
        !>                          RenormalizedChi2Org:    flag for using renormalized chi**2
        !>                          currentpathorg:         path of the working directory
        !>                          FitParameterNameOrg:    array containing the names of the model parameters
        !>                          FitParameterValueLineOrg:   array containing the values of the model parameters as string
        !>                          CalculationMethodOrg:   method of computation (at once or point-to-point)
        !>                          xAxisLabel:             label of the x-axis (for plot)
        !>                          yAxisLabel:             label of the y-axis (for plot)
        !>                          zAxisLabel:             label of the z-axis (for plot)
        !>                          PathStartScriptOrg:     path and name of the start script for calling model function
        !>                          ExeCommandStartScriptOrg:   command for calling model function
        !>                          parametersetorg:        the complete set of paramters (incl. flags and limits)
        !>                          expdataxorg:            array containing the experimental x side
        !>                          expdatayorg:            array containing the experimental y side
        !>                          expdataerrororg:        array containing the experimental error of the y side
        !>                          NumberRangesOrg:        number of y-columns for each experimental file
        !>                          MinRangeOrg:            array containing the minimal exp. ranges
        !>                          MaxRangeOrg:            array containing the maximal exp. ranges
        !>                          NumberXColumnsOrg:      number of x-columns for each experimental file
        !>                          NumberYColumnsOrg:      number of y-columns for each experimental file
        !>                          lengthexpdataorg:       number of lines in experimental data
        !>                          MaxRangeNumber:         max. number of ranges
        !>                          NumFileOrg:             number of experimental files
        !>                          MaxLengthOrg:           max length of experimental data
        !>                          MaxColXOrg:             number of columns concerning to the experimental x side
        !>                          MaxColYOrg:             number of columns concerning to the experimental y side
        !>                          parameternum:           number of model parameter
        !>                          SortFortranNum:         sort chi^2 log file by fortran
        !>
        !> output variables:        calstatus:              status flag of calculation (= 0: all ok)
        !>                          FitFunctionOut:         values of the model function at the calculated points
        !>                          Chi2Values:             values of the chi^2 function at the calculated points
        !>                          FinalParameterSet:      the complete set of paramters (incl. flags and limits)
        !>
        subroutine MainAlg(printflagNum, LastAlgorithmNum, calstatus, FitFunctionOut, Chi2Values, chilm, NumberOfFitAlgorithms, max_iter, GenCounter, &
                           ParamSetCounter, GeneralAlgorithmSettings, DeterminationChi2, PlotIteration, PlotType, fitlog, NumberInputFilesorg, &
                           NumberOutputFilesOrg, ParallelizationFlagorg, JobIDorg, MaxInputLinesOrg, MaxParameterOrg, RenormalizedChi2Org, &
                           currentpathorg, FitParameterNameLocal, FitParameterValueLocal, CalculationMethodOrg, xAxisLabel, yAxisLabel, zAxisLabel, &
                           PathStartScriptOrg, ExeCommandStartScriptOrg, parametersetorg, FinalParameterSet, expdataxorg, expdatayorg, expdataerrororg, &
                           NumberRangesOrg, MinRangeOrg, MaxRangeOrg, NumberXColumnsOrg, NumberYColumnsOrg, lengthexpdataorg, MaxRangeNumber, &
                           NumFileOrg, MaxLengthOrg, MaxColXOrg, MaxColYOrg, parameternum, SortFortranNum)


            implicit none
            ! ********** input variables **********
            integer :: parameternum                                                         !< number of model parameter
            integer :: NumberOfFitAlgorithms                                                !< total number of all algorithms in the chain
            integer :: max_iter                                                             !< max. number of iterations
            integer :: NumFileOrg                                                           !< number of experimental files
            integer, dimension(NumFileOrg) :: lengthexpdataorg                              !< number of lines in experimental data
            integer, dimension(NumFileOrg) :: NumberXColumnsOrg                             !< number of x-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberYColumnsOrg                             !< number of y-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberRangesOrg                               !< number of y-columns for each experimental file
            integer :: MaxColXOrg                                                           !< number of columns concerning to the experimental x side
            integer :: MaxColYOrg                                                           !< number of columns concerning to the experimental y side
            integer :: MaxLengthOrg                                                         !< max length of experimental data
            integer :: printflagNum                                                         !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: LastAlgorithmNum                                                     !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: SortFortranNum                                                       !< flag indicating if chi2 log file is sorted by fortran
                                                                                            !< yes (=1) or not (=0)
            integer :: DeterminationChi2                                                    !< method being used for the determination of chi^2
            integer :: PlotIteration                                                        !< plot model func. for each iteration set 1 (=yes) or 0 (=no)
            integer :: PlotTypeOrg                                                          !< get type of plot
            integer :: NumberInputFilesorg                                                  !< number of input files for the external model program
            integer :: NumberOutputFilesOrg                                                 !< number of output files for the external model program
            integer :: ParallelizationFlagorg                                               !< contains the number of processors used for parallelization
            integer :: JobIDorg                                                             !< job identification number
            integer :: MaxInputLinesOrg                                                     !< max number of lines in an input file
            integer :: MaxParameterOrg                                                      !< max number of parameters in a line of an input file
            integer :: RenormalizedChi2Org                                                  !< flag for using renormalized chi**2
            integer :: GenCounter                                                           !< counts number of calls
            integer :: NumChrom                                                             !< number of chromosomes (user defined)
            integer :: MaxRangeNumber                                                       !< max. number of ranges
            integer :: ParamSetCounter                                                      !< which positions should be used
            real*8 :: chilm                                                                 !< user defined abort criteria for chi**2
            real*8, dimension(15) :: GeneralAlgorithmSettings                               !< special algorithm settings
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColXOrg) :: expdataxorg          !< array containing the experimental x side
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColYOrg) :: expdatayorg          !< array containing the experimental y side
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColYOrg) :: expdataerrororg      !< array containing the experimental error of the y side
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MinRangeOrg        !< array containing the minimal exp. ranges
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MaxRangeOrg        !< array containing the maximal exp. ranges
            character(len=8192) :: fitlog                                                   !< path for log-file containing the current values of chi**2
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=20) :: CalculationMethodOrg                                       !< method of computation
            character(len=8192) :: PathStartScriptOrg                                       !< command for calling model function
            character(len=8192) :: ExeCommandStartScriptOrg                                 !< command for calling model function
            character(len=8192) :: currentpathorg                                           !< path of the working directory
            character(len=512), dimension(parameternum) :: FitParameterNameLocal            !< array containing the names of the model parameters
            character(len=512), dimension(parameternum) :: FitParameterValueLocal           !< array containing the values of the model parameters as


            ! ********** in/output variables **********
            real*8, dimension(4, parameternum) :: parametersetorg                           !< the non-optimized (initial parameter set)


            ! ********** output variables **********
            integer :: calstatus                                                            !< the following line is necessary for f2py
            real*8, dimension(ParamSetCounter, parameternum) :: FinalParameterSet           !< array containing the optimized parameter set
            real*8, dimension(ParamSetCounter, NumFileOrg, MaxLengthOrg, MaxColYOrg) :: FitFunctionOut !< values of the model func. at the calculated pts.
            real*8, dimension(ParamSetCounter, NumFileOrg, MaxLengthOrg, MaxColYOrg) :: Chi2Values     !< values of the model func. at the calculated pts.


            ! ********** working variabels **********
            integer :: i, j, k, ii, jj                                                      !< working variables
            integer :: nfit, ma, MaxLength, PlotType
            integer :: Nch, Nmut, Npar, Npop
            integer :: ok                                                                   !< status of calculation
            integer :: actualiteration                                                      !< contains the current iteration within the iteration loop
            integer :: flag                                                                 !< working variable used within the iteration loop
            integer :: NumInputFiles                                                        !< need for loop over input files
            integer :: allocstatus, deallocstatus                                           !< working variables for allocation/deallocation
            integer, dimension(8) :: VALUES                                                 !< value for the date_and_time subroutine
            real*8, dimension(1) :: chi2ValuesVector                                        !< here only one chi2 value
            real*8, dimension(parameternum, 2) :: new_range_full
            character(len=512) :: fitlogparam                                               !< path for log-file containing the current parameter values
            character(len=512) :: fitlogChi2                                                !< path for log-file containing chi**2 and the corresponding
                                                                                            !< parameter values
            character(len=8) :: DATE                                                        !< variable for the date_and_time subroutine
            character(len=10) :: TIME                                                       !< variable for the date_and_time subroutine
            character(len=5) :: ZONE                                                        !< variable for the date_and_time subroutine
            character(len=10) :: Number1                                                    !< variable for number to string converting
            character(len=100) :: helpString                                                !< help string
            character(len=8192) :: SiteExt, NumExt, BaseDir, FuncCallExt                    !< working variables for final input file name


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< set print flag and last algorithm flag
            calstatus = 0                                                                   !< set calculation status to 0 = everything is ok
            if (printflagNum == 1) then                                                     !< set printflag
                printflag = .true.
            else
                printflag = .false.
            endif
            if (LastAlgorithmNum == 1) then
                LastAlgorithmFlag = .true.
            else
                LastAlgorithmFlag = .false.
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy contents of some input variables to module variables
            NumberExpFiles = NumFileOrg                                                     !< copy number of experimental files to global variable
            currentpath = trim(adjustl(currentpathorg))                                     !< copy path of working directory to module variable without
                                                                                            !< trailing and leading blanks
            MaxColX = MaxColXOrg                                                            !< copy number of columns of the experimental x data to
                                                                                            !< module variable
            MaxColY = MaxColYOrg                                                            !< copy number of columns of the experimental y data to
                                                                                            !< module variable
            MaxLength = MaxLengthOrg                                                        !< copy max. number of lines
            MaxExpLength = MaxLength                                                        !< copy max. number of exp. data points
            MaxNumberRanges = MaxRangeNumber                                                !< copy of max. number of data ranges in a exp. data file
            parameternumber = parameternum                                                  !< copy input variable containing the number of parameters
                                                                                            !< to module variable
            DetChi2 = DeterminationChi2                                                     !< copy method of chi**2 determination
            NumberInputFiles = NumberInputFilesorg                                          !< copy number of input files for the external program to
                                                                                            !< global variable
            NumberOutputFiles = NumberOutputFilesOrg                                        !< copy number of output files for the external program to
                                                                                            !< global variable
            ParallelizationFlag = ParallelizationFlagorg                                    !< copy number of used processors to global variable
            PlotType = PlotTypeOrg                                                          !< copy flag for plotting
            MaxInputLines = MaxInputLinesOrg                                                !< copy max number of input lines in an input file
            MaxParameter = MaxParameterOrg                                                  !< copy max number of parameters in a line of an input file
            MaxGeneticIter = max_iter                                                       !< copy max. number of iteration
            JobID = JobIDorg                                                                !< copy job-ID number to global variable
            QualityLimit = abs(ParamSetCounter)                                             !< which positions should be used
            RenormalizedChi2 = .true.                                                       !< define flag for using renormalized chi**2
            if (RenormalizedChi2Org /= 1) then
                RenormalizedChi2 = .false.
            endif
            PlotIterationFlag = .false.
            if (PlotIteration == 0) PlotIterationFlag = .true.


            !< get special algorithm settings
            NumChrom = int(GeneralAlgorithmSettings(3))
            Nch = NumChrom                                                                  !< copy number of chromosomes (user defined)
            if (int(GeneralAlgorithmSettings(2)) == 1) then
                RangeFlag = .true.
            else
                RangeFlag = .false.
            endif

            ! Debug:
            !    print*,'NumChrom = ', NumChrom
            !    print*,'GeneralAlgorithmSettings = ', GeneralAlgorithmSettings
            !    print*,'PathStartScriptOrg = ',trim(PathStartScriptOrg)
            !    print*,'ExeCommandStartScriptOrg = ',trim(ExeCommandStartScriptOrg)
            !    print*,'FitFktInputOrg = ',trim(FitFktInputOrg)
            !    print*,'MaxNumberParameter = ',MaxNumberParameter
            !    print*,'NumFileOrg = ',NumFileOrg
            !    print*,'MaxColXOrg = ',MaxColXOrg
            !    print*,'MaxColYOrg = ',MaxColYOrg
            !    Do i=1,NumFileOrg
            !        print*,'    Experimental file: i = ',i
            !        print*,'    lengthexpdataorg(i) = ',lengthexpdataorg(i)
            !        print*,'    NumberYColumnsOrg(i) = ',NumberYColumnsOrg(i)
            !        print*,'    expdataxorg(i,1:5,1) = ',expdataxorg(i,1:5,1)
            !        print*,'    expdatayorg(i,1:5,1) = ',expdatayorg(i,1:5,1)
            !        print*,'    expdataerrororg(i,1:5,1) = ',expdataerrororg(i,1:5,1)
            !    end Do
            !    print*,'chilm = ',chilm
            !    print*,'max_iter = ',max_iter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'currentpathorg = ',trim(adjustl(currentpathorg))
            !    print*,'len(FitParameterNameOrg) = ',len(FitParameterNameOrg
            !    print*,'FitParameterNameOrg = ',FitParameterNameOrg
            !    print*,'FitParameterValueLineOrg = ',FitParameterValueLineOrg
            !    print*,"parametersetorg(1,:) = ",parametersetorg(1,:)
            !    print*,"parametersetorg(2,:) = ",parametersetorg(2,:)
            !    print*,"parametersetorg(3,:) = ",parametersetorg(3,:)
            !    print*,"parametersetorg(4,:) = ",parametersetorg(4,:)
            !    print*,"RenormalizedChi2 = ",RenormalizedChi2
            !    return


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< get standard magix working directory
            ! stat = getcwd(StandardWorkingDirectory)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< set temp-directory
            TempDirectory = " "
            CALL GetEnv('MAGIXTempDirectory', TempDirectory)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< modify name of log file
            i = index(fitlog,"/",back = .true.)
            j = index(fitlog,".",back = .true.)
            Number1 = "          "
            write(Number1,'(I10)') abs(GenCounter)
            if (j > i) then
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__Genetic__call_" // trim(adjustl(Number1)) // trim(adjustl(fitlog(j:)))
                else
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__Genetic" // trim(adjustl(fitlog(j:)))
                endif
            else
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog)) // "__Genetic__call_" // trim(adjustl(Number1)) // ".log"
                else
                    fitlog = trim(adjustl(fitlog)) // "__Genetic.log"
                endif
            endif

            ! Debug:
            !   print*,'>',trim(adjustl(fitlog)),'<'


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open log file and write header
            open(logchannel, file = trim(fitlog), status='replace')
            write(logchannel,'(" ")')
            write(logchannel,'("log-file for Genetic algorithm:")')
            write(logchannel,'(31("-"))')
            write(logchannel,'(" ")')

            !< get current local time and date and write to log-file
            call date_and_time(DATE, TIME, ZONE, VALUES)
            write(logchannel,'(" ")')
            write(logchannel,'("algorithm starts at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                                  TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open log-file for the parameter and write header
            fitlogparam = trim(fitlog)//".param"
            open(paramchannel, file = trim(fitlogparam), status='replace')
            write(paramchannel,'(" ")')
            write(paramchannel,'("log-file containing the actual values of the parameters used in the Genetic algorithm:")')
            write(paramchannel,'(86("-"))')
            write(paramchannel,'(" ")')


            !< call subroutine to write current values of the parameter to file
            write(paramchannel,'("  ")')
            write(paramchannel,'("  ")')
            write(paramchannel,'("Input file with start values of all parameters:")')
            write(paramchannel,'(46("-"))')
            write(paramchannel,'("  ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open file containing the values of chi**2 and the corresponding values of the parameters
            WriteChi2Flag = .true.
            NumberLinesChi2 = 0
            fitlogChi2 = trim(fitlog) // ".chi2"
            open(Chi2Channel, file = trim(fitlogChi2), status='replace')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate if necessary and print error message if necessay
            if (allocated(expdatax)) then
                deallocate(expdatax, expdatay, expdatae, lengthexpdata, NumberXColumns, NumberYColumns, FirstPointExpData, LastPointExpData, &
                           NumberRanges, MinRange, MaxRange, ExpData_reversed_flag, stat = deallocstatus)
                if (deallocstatus /= 0) then                                                    !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine MainAlg:")')
                    write(logchannel,'(2x,"Can not deallocate variables expdatax etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine MainAlg:")'
                    print '(2x,"Can not deallocate variables expdatax etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for variables, clear content of the variables and print error message if necessay
            allocate(expdatax(NumberExpFiles, MaxLength, MaxColX), expdatay(NumberExpFiles, MaxLength, MaxColY), &
                     expdatae(NumberExpFiles, MaxLength, MaxColY), lengthexpdata(NumberExpFiles), NumberXColumns(NumberExpFiles), &
                     NumberYColumns(NumberExpFiles), FirstPointExpData(NumberExpFiles, MaxColX), LastPointExpData(NumberExpFiles, MaxColX), &
                     NumberRanges(NumberExpFiles), MinRange(NumberExpFiles, MaxRangeNumber, MaxColX), MaxRange(NumberExpFiles, MaxRangeNumber, MaxColX), &
                     ExpData_reversed_flag(NumberExpFiles), stat = allocstatus)
            if (allocstatus /= 0) then                                                      !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not allocate variables expdatax etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not allocate variables expdatax,expdatay etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            MaxRangeNumber = MaxRangeNumber - 1                                             !< get real value
            expdatax = 0.d0
            expdatay = 0.d0
            expdatae = 0.d0
            lengthexpdata = 0
            NumberXColumns = 0
            NumberYColumns = 0
            FirstPointExpData = 1.d99
            LastPointExpData = -1.d99
            NumberRanges = 0
            MinRange = 0.d0
            MaxRange = 0.d0
            ExpData_reversed_flag = .false.



            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy input variables to module variabels
            expdatax(:,:,:) = expdataxorg(:,:,:)
            expdatay(:,:,:) = expdatayorg(:,:,:)
            expdatae(:,:,:) = expdataerrororg(:,:,:)
            lengthexpdata = lengthexpdataorg
            NumberXColumns = NumberXColumnsOrg
            NumberYColumns = NumberYColumnsOrg
            CalculationMethod = CalculationMethodOrg
            PathStartScript = PathStartScriptOrg
            ExeCommandStartScript = ExeCommandStartScriptOrg
            NumberRanges = NumberRangesOrg
            MinRange = MinRangeOrg
            MaxRange = MaxRangeOrg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine first and last point of each exp. data file
            Do i = 1, NumberExpFiles
                Do j = 1, lengthexpdata(i)
                    ii = 0
                    jj = 0
                    Do k = 1,NumberXColumns(i)
                        if (expdatax(i, j, k) <= FirstPointExpData(i, k)) then
                            ii = ii + 1
                        endif
                        if (expdatax(i, j, k) >= LastPointExpData(i, k)) then
                            jj = jj + 1
                        endif
                    end Do
                    if (ii == NumberXColumns(i)) then
                        FirstPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                    if (jj == NumberXColumns(i)) then
                        LastPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                end Do


                !< check output file starts with the highest x-column value interchange FirstPointOutputFile and LastPointOutputFile
                ii = 0
                Do k = 1, NumberXColumns(i)
                    if (expdatax(i, 1, k) >= expdatax(i, lengthexpdata(i), k)) then
                        ii = ii + 1
                    endif
                end Do
                if (ii == NumberXColumns(i)) then
                    ExpData_reversed_flag(i) = .true.
                endif

                ! Debug:
                ! print*,' '
                ! print*,'File = ',i
                ! print*,'FirstPointExpData(i, 1:NumberXColumns(i)) = ', FirstPointExpData(i, 1:NumberXColumns(i))
                ! print*,'LastPointExpData(i, 1:NumberXColumns(i)) = ', LastPointExpData(i, 1:NumberXColumns(i))
                ! print*,'##########################################'
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy number of parameters required for the model function to working variable
            nfit = sum(parametersetorg(2,:))                                                !< number of parameters which should be optimized
            ma = parameternumber
            Gradientflag = .false.                                                          !< we do not need the gradient of the function here
            UseCalculationReduction = .true.                                                !< activate calculation reduction
            CurrentNumberLinesCalcReduction = 0                                             !< reset number of lines


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< dimension of the arrays 'ModelFunction'
            i = 0
            Do k = 1, NumberExpFiles
                i = i + (NumberYColumns(k) * lengthexpdata(k))
            end Do
            j = nfit


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate/allocate memory for some working variables, clear contents of the variables and print error message if necessary
            if (allocated(paramset)) then
                deallocate(paramset, FitParameterName, FitParameterValue, ia, a, OptimizedParameterLowLimit, OptimizedParameterUpperLimit, &
                           ModelFunction, chisqValues, BestSitesParamSet, BestSitesModelValues, BestSitesChi2Values, ConverterInfit, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine MainAlg:")')
                    write(logchannel,'(2x,"Can not deallocate variables paramset etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine MainAlg:")'
                    print '(2x,"Can not deallocate variables paramset etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif
            allocate(paramset(4, parameternumber), OptimizedParameterLowLimit(nfit), OptimizedParameterUpperLimit(nfit), ConverterInfit(nfit), &
                     ia(parameternumber), FitParameterName(parameternumber), FitParameterValue(parameternumber), a(parameternumber), &
                     ModelFunction(1, NumberExpFiles, MaxColY, MaxLength), chisqValues(0:0), &
                     AtOnceFunction(0:ParallelizationFlag - 1, NumberExpFiles, MaxColY, MaxLength), &
                     BestSitesParamSet(QualityLimit, nfit + 1), BestSitesModelValues(QualityLimit, NumberExpFiles, MaxLength, MaxColY), &
                     BestSitesChi2Values(QualityLimit, NumberExpFiles, MaxLength, MaxColY), stat = allocstatus)
            if (allocstatus /= 0) then
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not allocate variables paramset etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not allocate variables paramset etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            ia = .false.
            FitParameterName = FitParameterNameLocal
            FitParameterValue = FitParameterValueLocal
            OptimizedParameterLowLimit = 0.d0
            OptimizedParameterUpperLimit = 0.d0
            ModelFunction = 0.d0
            chisqValues = 0.d0
            ConverterInfit = 0
            AtOnceFunction = 0.d0
            BestSitesParamSet = 0.d0
            BestSitesParamSet(:, 1) = 1.d99
            BestSitesModelValues = 0.d0
            BestSitesChi2Values = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy parameter set to module variable
            paramset = parametersetorg

            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write registration mask for the fit model
            if (printflag) print '(9x, "Writing registration mask for the fit model .. ", $)'
            call RegistrationMask(ok)
            if (ok /= 0) then
                return
            endif
            if (printflag) print '("done!")'


            !< define ia variable
            ConverterInfit = 0
            ia = .false.
            k = 0
            Do i = 1, parameternumber
                if (paramset(2, i) == 1) then
                    k = k + 1
                    ia(i) = .true.
                    ConverterInfit(k) = i
                endif
            end Do
            NumberFreeParameter = int(sum(paramset(2, :)))                                  !< determine number of free parameter


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize model program
            call ModelInit


            !< write on the screen what you Do ..
            if (printflag) then
                print '(" ")'
                print '(" ")'
                write(Number1,'(I10)') JobID                                                !< write JobID to string
                print '(9x,"Temporary files are stored in: ",A)', trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(Number1)) // "/"
                print '(" ")'
                print '(" ")'
                print '(9x,"Start Genetic algorithm (", A, " version) ..")', trim(adjustl(ParallelizationMethod))
                print '(" ")'
            endif
            write(logchannel,'(11x,"Start Genetic algorithm (", A, " version) ..")') trim(adjustl(ParallelizationMethod))
            write(logchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< lower limit of chi^2 for stopping condition normalized to the number of calculation points
            chilim = 0.d0
            if (RenormalizedChi2) then
                Do i = 1, NumberExpFiles
                    chilim = chilim + (NumberYColumns(i) * lengthexpdata(i) - parameternumber) * dabs(chilm)
                end Do
                chilim = dabs(chilim)
                write(logchannel,'(11x,"Renormalized limit for chi^2 = ", ES25.15)') chilim
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(11x,"Renormalized limit for chi^2 = ",ES25.15)', chilim
                    print '(" ")'
                endif
            else
                chilim = dabs(chilm)
                write(logchannel,'(11x,"Limit for chi^2 = ", ES25.15)') chilim
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(11x,"Limit for chi^2 = ",ES25.15)', chilim
                    print '(" ")'
                endif
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< start iteration ..
            Gradientflag = .false.                                                          !< we do not need the gradient of the function here
            flag = 0                                                                        !< set flag variable to 0
            actualiteration = 0                                                             !< set working variable containing the current iteration
                                                                                            !< number to 0
            a = paramset(1, 1:parameternumber)                                              !< copy the values of the parameters to another array
            new_range_full = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< call subroutine to call Genetic algorithm
            call CallGenAlg(max_iter, nfit, ma, Nch, Npar, Nmut, Npop, ParamSetCounter, FinalParameterset, MaxColx, NumberExpFiles, MaxLength, MaxColY, &
                            new_range_full, PlotIteration, PlotType, xAxisLabel, yAxisLabel, zAxisLabel, fitlog)
            !    ParamSetCounter = QualityLimit
            FitFunctionOut = 0.d0
            Chi2Values = 0.d0
            Do i  = 1, QualityLimit


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< determine model function values if plot for each iteration option is not selected
                if (.not. PlotIterationFlag .and. DontStoreModelFuncValuesFlag) then
                    PlotIterationFlag = .true.
                    chi2ValuesVector = 0.d0
                    call ModelCalcChiFunctionGeneral(parameternumber, ia, paramset(1, :), 1, nfit, NumFileOrg, MaxColY, MaxLength, &
                                                     BestSitesParamSet(i, 2:), chi2ValuesVector)
                    PlotIterationFlag = .false.
                endif
                Do j = 1, NumFileOrg
                    Do k = 1, MaxLength
                        FitFunctionOut(i, j, k, 1:MaxColY) = BestSitesModelValues(i, j, k, 1:MaxColY)
                        Chi2Values(i, j, k, 1:MaxColY) = BestSitesChi2Values(i, j, k, 1:MaxColY)
                    end Do
                end Do
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< close log-files
            if (SortFortranNum == 1) call SortChi2File(nfit, parameternumber, ia, a)
            close(Chi2Channel)
            close(paramchannel)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< write final parameter sets for each site to final input files


            !< define base directory, i.e. path where final input file is written to
            k = index(fitlog, '/', back = .true.)
            if (k == 0) then
                BaseDir = ""
            else
                BaseDir = trim(adjustl(fitlog(:k)))
            endif


            !< define file name extension for number of algorithm call
            write(Number1,'(I10)') abs(GenCounter)
            if (NumberOfFitAlgorithms > 1) then
                FuncCallExt = "__Genetic__call_" //trim(adjustl(Number1))
            else
                FuncCallExt = "__Genetic"
            endif

            ! Debug:
            !    print*,"NumberOfFitAlgorithms = ", NumberOfFitAlgorithms
            !    print*,"FuncCallExt = ", trim(adjustl(FuncCallExt))


            !< write files
            Do i  = 1, QualityLimit                                                         !< loop over all sites


                !< create site extension for file name
                if (QualityLimit > 1) then
                    write(helpString,'(I5)') i
                    SiteExt = "__site_" // trim(adjustl(helpString)) // ".out"
                else
                    SiteExt = ".out"
                endif


                !< write parameter sets to file
                Do NumInputFiles = 1, NumberInputFiles                                      !< loop over all input files
                    NumExt = trim(adjustl(FitFktInput(NumInputFiles)))
                    k = index(NumExt, '/', back = .true.)
                    if (k > 0) then                                                         !< we only need the file name
                        NumExt = trim(adjustl(NumExt(k:)))
                    endif
                    j = index(NumExt, '.', back = .true.)
                    if (j > 1) then                                                         !< remove file name extension
                        NumExt = trim(adjustl(NumExt(:j - 1)))
                    endif

                    ! Debug:
                    ! print*,"Site = ", i
                    ! print*,"Nr. Final input file = ", NumInputFiles
                    ! print*,"Final input file = ", trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt)) &
                    !                               // trim(adjustl(SiteExt)) // ".input"


                    !< write parameter sets to file
                    open(235,file = trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt)) // trim(adjustl(SiteExt)) // ".input")
                    call WriteParameter(235, .false., MaxColX, expdataxorg, parameternumber, FinalParameterSet(i, :), NumInputFiles)
                    close(235)
                end Do
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write end of log-file and print message to screen
            call date_and_time(DATE, TIME, ZONE, VALUES)
            write(logchannel,*)
            write(logchannel,'("algorithm ends at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                              TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel,'(" ")')
            write(logchannel,'(150("-"))')

            ! Debug:
            !    print*,'lengthexpdata = ',lengthexpdata
            !    print*,'MaxColX = ',MaxColX
            !    print*,'MaxColY = ',MaxColY
            !    print*,'expdataxorg(1:5,1) = ',expdataxorg(1:5,1)
            !    print*,'expdatayorg(1:5,1) = ',expdatayorg(1:5,1)
            !    print*,'expdataerrororg(1:5,1) = ',expdataerrororg(1:5,1)
            !    print*,'chilm = ',chilm
            !    print*,'max_iter = ',max_iter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'model = ',trim(model)
            !    print*,'currentpathorg = ',currentpathorg
            !    print*,'parametersetorg(1,:) = ',parametersetorg(1,:)
            !    print*,'parametersetorg(2,:) = ',parametersetorg(2,:)
            !    print*,'parametersetorg(3,:) = ',parametersetorg(3,:)
            !    print*,'parametersetorg(4,:) = ',parametersetorg(4,:)


            !< send plot program the kill signal
            !    if (PlotIteration == 0) then
            !        call system("kill -9 " // trim(adjustl(PlotPID)) // " 1>/dev/null 2>&1")
            !    endif


            !< store new parameter limits to output parameter
            k = 0
            Do j = 1, ma
                paramset(3, j) = new_range_full(j, 1)
                paramset(4, j) = new_range_full(j, 2)
            end Do
            parametersetorg = paramset


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< free memory of general genetic variables
            if (allocated(ia)) deallocate(ia, stat = deallocstatus)
            if (allocated(a)) deallocate(a, stat = deallocstatus)
            if (allocated(OptimizedParameterLowLimit)) deallocate(OptimizedParameterLowLimit, stat = deallocstatus)
            if (allocated(OptimizedParameterUpperLimit)) deallocate(OptimizedParameterUpperLimit, stat = deallocstatus)


            !< free memory of model variables
            call ModelParamFree(deallocstatus)
            if (deallocstatus /= 0) then
                write(logchannel,*)
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not deallocate expdatax etc.")')
                write(logchannel,*)
                write(logchannel,'("deallocstatus = ", I4)') deallocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not deallocate variables expdatax etc.")'
                print '(" ")'
                print '("deallocstatus = ", I4)', deallocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< close log file
            close(logchannel)
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(9x,"Finished Genetic algorithm!")'
                print '(" ")'
                print '(" ")'
            endif


            !< we're done
            return
        end subroutine MainAlg
end Module Algorithm
!*********************************************************************************************************************************************************


