!*********************************************************************************************************************************************************
!> Module: OpticalProperties
!>
!>
!>  This module contains the subroutines for calculating the epsilon tensor using the generalized Drude-Lorentz model
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine reflectance:                   calculate the reflectance within the ac-plane
!>
!>
!> Copyright (C) 2009 - 2024
!>
!> I. Physikalisches Institut, Universitaet zu Koeln
!>
!> Produced for the CATS project and MnWO4 Paper
!>
!>
!> fortran module containing the subroutine for "generalized Drude-Lorentz model"
!>
!> Who           When        What
!>
!> T. Moeller    2009-07-07  Initial version
!>
!*********************************************************************************************************************************************************
Module OpticalProperties

    implicit none
    integer :: number_osc                                                                   !< number of osc. for the generalized Drude-Lorentz model
    real*8 :: eps_inf_xx                                                                    !< epsilon_\infty for epsilon_xx matrix element
    real*8 :: eps_inf_xz                                                                    !< epsilon_\infty for epsilon_xz matrix element
    real*8 :: eps_inf_zz                                                                    !< epsilon_\infty for epsilon_zz matrix element
    real*8 :: InfallAngle
    real*8 :: pi
    real*8 :: RotAngle
    real*8, allocatable, dimension(:) :: w0, wp, G, theta                                   !< eigenfrequencies, plasma frequencies, damping, angle
                                                                                            !< for the generalized Drude-Lorentz model

    contains


        !>************************************************************************************************************************************************
        !> subroutine: reflectance
        !>
        !> calculate the reflectance within the ac-plane
        !>
        !>
        !> input variables:     w:                  frequency
        !>
        !>                      angle:              polarization angle
        !>
        !> output variables:    value:              calculated reflectance
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine reflectance(w, value)

            implicit none
            integer :: i, k                                                                 !< loop variables
            real*8 :: w                                                                     !< frequency
            real*8 :: value                                                                 !< output variable: calculated reflectance
            real*8 :: theta_rad                                                             !< theta angle in radians
            real*8, dimension(2, 2) :: RotMatrix                                            !< rotation matrix
            real*8, dimension(2, 2) :: InverseRotMatrix                                     !< inverse of rotation matrix
            complex(8) :: epsxx, epsxz, epszz                                               !< components of epsilon tensor
            complex(8) :: DrudeLorentz                                                      !< working variable: contribution of ith oscillator
            complex(8) :: const                                                             !< working variable: used for Koch formula
            complex(8), dimension(2, 2) :: epsilon_tensor                                   !< epsilon tensor

            value = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate matrix elements of the epsilon tensor
            epsxx = eps_inf_xx
            epsxz = eps_inf_xz
            epszz = eps_inf_zz
            k = 4
            Do i = 1, number_osc                                                            !< loop over all oscillators
                DrudeLorentz = ((wp(i)**2) / (w0(i)**2 - w**2 - dcmplx(0.d0,1.d0) * G(i) * w))
                theta_rad = theta(i) * (pi/180.d0)                                          !< convert degree to rad
                epsxx = epsxx + DrudeLorentz * (dcos(theta_rad)**2)
                epsxz = epsxz + DrudeLorentz * (dsin(theta_rad) * dcos(theta_rad))
                epszz = epszz + DrudeLorentz * (dsin(theta_rad)**2)
                k = k + 4
            end Do


            !< define epsilon tensor
            epsilon_tensor = 0.d0
            epsilon_tensor(1,1) = epsxx
            epsilon_tensor(1,2) = epsxz
            epsilon_tensor(2,1) = epsxz
            epsilon_tensor(2,2) = epszz

            ! Debug:
            ! print*,'w = ', w
            ! print*,'epsilon_tensor(1, 1) = ', epsilon_tensor(1, 1)
            ! print*,'epsilon_tensor(1, 2) = ', epsilon_tensor(1, 2)
            ! print*,'epsilon_tensor(2, 1) = ', epsilon_tensor(2, 1)
            ! print*,'epsilon_tensor(2, 2) = ', epsilon_tensor(2, 2)


            !< rotate basis of epsilon tensor
            RotMatrix(1, 1) = dcos(RotAngle)
            RotMatrix(1, 2) = -dsin(RotAngle)
            RotMatrix(2, 1) = dsin(RotAngle)
            RotMatrix(2, 2) = dcos(RotAngle)
            InverseRotMatrix(1, 1) = dcos(RotAngle)
            InverseRotMatrix(1, 2) = dsin(RotAngle)
            InverseRotMatrix(2, 1) = -dsin(RotAngle)
            InverseRotMatrix(2, 2) = dcos(RotAngle)
            epsilon_tensor = matmul(matmul(RotMatrix, epsilon_tensor), InverseRotMatrix)

            epsxx = epsilon_tensor(1, 1)
            epsxz = epsilon_tensor(1, 2)
            epszz = epsilon_tensor(2, 2)


            !< define constant
            const = cdsqrt(epsxx * epszz - epsxz**2)

            ! Debug:
            ! print*,'RotAngle = ', RotAngle
            ! print*,'RotAngle [°] = ', RotAngle * 180.d0/pi
            ! print*,'InfallAngle = ', InfallAngle
            ! print*,'InfallAngle [°] = ', InfallAngle * 180.d0/pi
            ! print*,'epsxx = ', epsxx
            ! print*,'epsxz = ', epsxz
            ! print*,'epszz = ', epszz
            ! print*,'const = ', const


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate Koch model (eq. 23, Chem. Phys. 3, 362 (1974)
            value = cdabs((dcos(InfallAngle) - cdsqrt(epszz - dsin(InfallAngle)**2) * 1.d0/(const)) &
                    / (dcos(InfallAngle) + cdsqrt(epszz - dsin(InfallAngle)**2) * 1.d0/(const)))**2

            if (value > 1.d0) value = 1.d0 / value

            ! Debug:
            ! print*,'value = ', value
            ! stop
            return
        end subroutine reflectance
end Module OpticalProperties
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!>
!>
!> Main Program
!>
!>
program DrudeLorentzGeneral
    !< calculation of the refelctivity using the generalized Drude-Lorentz model

    use OpticalProperties

    implicit none
    integer :: i                                                                            !< loop variables
    integer :: NumberXValues                                                                !< number exp. data points
    integer :: alloc_status, dealloc_status                                                 !< status variables for (de-)allocation
    real*8 :: w                                                                             !< working variable: current frequency w
    real*8 :: refl                                                                          !< working variable: final reflectance for frequency w


    pi = 4.d0 * datan(1.d0)                                                                 !< pi


    !< open files
    open(21,file = "in.txt")                                                                !< open parameter file
    open(22,file = "DataIn.dat")                                                            !< open experimental-point file
    open(23,file = "FitFunctionValues.dat")                                                 !< open file for fit function values


    !< read parameter from file
    read(21,*) NumberXValues                                                                !< read number of exp. data points
    read(21,*) eps_inf_xx                                                                   !< read epsilon_xx
    read(21,*) eps_inf_xz                                                                   !< read epsilon_xz
    read(21,*) eps_inf_zz                                                                   !< read epsilon_zz
    read(21,*) number_osc                                                                   !< read number of oscillators


    ! deallocate/allocate memory
    if (allocated(w0)) then
        deallocate(w0, wp, G, theta, stat = dealloc_status)
        if (dealloc_status /= 0) then
            print '(" ")'
            print '(2x,"Error in module DrudeLorentzGeneral:")'
            print '(4x,"Cannot deallocate variable w0, wp, G, theta.")'
            print '(" ")'
            stop '  Program aborted!'
        endif
    endif
    allocate(w0(number_osc), wp(number_osc), G(number_osc), theta(number_osc), stat = alloc_status)
    if (alloc_status /= 0) then
        print '(" ")'
        print '(2x,"Error in module DrudeLorentzGeneral:")'
        print '(4x,"Cannot allocate variable w0, wp, G, theta.")'
        print '(" ")'
        stop '  Program aborted!'
    endif
    w0 = 0.d0
    wp = 0.d0
    G = 0.d0
    theta = 0.d0
    InfallAngle = 0.d0


    !< read parameters for each oscillators from file
    Do i = 1, number_osc                                                                    !< loop over all oscillators
        read(21,*) w0(i), wp(i), G(i), theta(i)
    end Do
    read(21,*) RotAngle
    RotAngle = RotAngle * (pi/180.d0)
    read(21,*) InfallAngle
    InfallAngle = InfallAngle * (pi/180.d0)
    close(21, status = 'delete')                                                            !< close and delete parameter file

    ! Debug:
    !    print*,'DrudeLorentzGeneral:'
    !    print*,'w = ',w
    !    print*,'RotAngle = ',RotAngle
    !    print*,'eps_inf_xx = ',eps_inf_xx
    !    print*,'eps_inf_xz = ',eps_inf_xz
    !    print*,'eps_inf_zz = ',eps_inf_zz
    !    print*,'number_osc = ',number_osc
    !    print*,'w0(1:number_osc) = ',w0(1:number_osc)
    !    print*,'wp(1:number_osc) = ',wp(1:number_osc)
    !    print*,'G(1:number_osc) = ',G(1:number_osc)
    !    print*,'theta(1:number_osc) = ',theta(1:number_osc)
    !    print*,'phi(1:number_osc) = ',phi(1:number_osc)
    !    print*,' '


    !< calculate optical properties at any frequency w
    Do i = 1, NumberXValues                                                                 !< loop over all frequency points
        read(22,*) w                                                                        !< read frequency and polarization angle form file


        !< call subroutine for determine reflectance
        call reflectance(w, refl)                                                           !< calculate reflectance

        !< write fit function value to file
        write(23,*) refl                                                                    !< write reflection to output file
    end Do

    !< close files
    close(22, status = 'delete')                                                            !< close and delete experimental-point file
    close(23)                                                                               !< close output file
end program DrudeLorentzGeneral
!---------------------------------------------------------------------------------------------------------------------------------------------------------

