#!/usr/bin/python3
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  generate spectrum for LineID function tutorial
##
##
##  Usage:
##
##      python3 XCLASS-tutorial___LineID__generator.py
##
##
##  Command linie parameter:
##
##      - None
##
##
##
##
##  Copyright (C) 2023
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function main:                                main program
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    2023-10-31   initial version
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## for python 2 usage
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import matplotlib                                                                           ## import matplotlib package
matplotlib.rcParams['backend'] = 'Agg'                                                      ## disable GUI window
import pylab                                                                                ## import pylab
import warnings                                                                             ## import warnings package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##-----------------------------------------------------------------------------
##
## start main program
##
if __name__ == '__main__':


    ## get path of current directory
    LocalPath = os.getcwd() + "/"


    # import XCLASS packages
    from xclass import task_myXCLASS
    from xclass import task_LineIdentification


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## call myXCLASS function


    ## set path and name of molfit file
    MolfitsFileName = LocalPath + "files/tutorial-lineid__generation.molfit"


    ## set path and name of obs. xml file
    ObsXMLFileName = LocalPath + "../files/tutorial-lineid__obs.xml"


    ## add noise?
    NoiseAmplitude = 0.05


    ## plot?
    DoPlotFlag = True


    ## call myXCLASS function
    printFlag = True
    modeldata, log, TransEnergies, IntOpt, JobDir = task_myXCLASS.myXCLASSCore(MolfitsFileName = MolfitsFileName, \
                                                                               ObsXMLFileName = ObsXMLFileName, \
                                                                               printFlag = printFlag)


    ## add noise to spectrum
    if (NoiseAmplitude > 0.0):
        NumDataPoints = len(modeldata[:, 0])
        rng = numpy.random.default_rng()
        std = NoiseAmplitude * numpy.std(modeldata[:, 1])
        noise = rng.standard_normal(NumDataPoints) * std
        modeldata[:, 1] += noise[:]


    ## save spectrum
    numpy.savetxt(LocalPath + "../files/data/SyntheticData.dat", modeldata)


    ##====================================================================================================================================================
    ##
    ## FOR TESTING: Create a plot
    ##
    ##====================================================================================================================================================
    if (DoPlotFlag):
        print ("\n\nCreate a plot ..\n\n")


        ## create plot subdirectory
        ResultPath = LocalPath + "plot/"
        os.system("rm -rf " + ResultPath + "; mkdir " + ResultPath)


        ## create figure
        modeldata = [modeldata]
        fig = pylab.figure(figsize = (15, 10))
        fig.clear()
        pylab.subplots_adjust(hspace = 0.2, wspace = 0.2)
        for LocalSpectrumID, LocalSpectrum in enumerate(modeldata):
            layer = pylab.subplot(len(modeldata), 1, LocalSpectrumID + 1)
            layer.grid(True)
            if (LocalSpectrumID == 0):
                layer.set_title("Generated function for LineID tutorial")
            layer.set_ylabel(r"Brightness Temperature [K]")
            layer.set_xlabel("Rest Frequency [MHz]")
            layer.plot(LocalSpectrum[:, 0], LocalSpectrum[:, 1], '-', color = 'black', linewidth = 2.0)
            FreqMin = numpy.min(LocalSpectrum[:, 0])
            FreqMax = numpy.max(LocalSpectrum[:, 0])
            layer.set_xlim(FreqMin, FreqMax)
            layer.xaxis.set_major_formatter(matplotlib.ticker.FormatStrFormatter('%.5e'))
        pylab.tight_layout()
        pylab.savefig(ResultPath + "/myXCLASS____generated-spectrum.png", dpi = 300)
        pylab.draw()
        pylab.close(fig)


        ## create figure
        for LocalSpectrumID, LocalSpectrumRaw in enumerate(IntOpt[0]):
            MolName = LocalSpectrumRaw[0]
            CompID = LocalSpectrumRaw[1]
            LocalSpectrum = LocalSpectrumRaw[2]
            fig = pylab.figure(figsize = (15, 10))
            fig.clear()
            pylab.subplots_adjust(hspace = 0.2, wspace = 0.2)
            layer = pylab.subplot(1, 1, 1)
            layer.grid(True)
            layer.set_title("Generated function for LineID tutorial: {:s}, comp. {:d}".format(MolName, CompID))
            layer.set_ylabel(r"Brightness Temperature [K]")
            layer.set_xlabel("Rest Frequency [MHz]")
            layer.plot(LocalSpectrum[:, 0], LocalSpectrum[:, 1], '-', color = 'black', linewidth = 2.0)
            FreqMin = numpy.min(LocalSpectrum[:, 0])
            FreqMax = numpy.max(LocalSpectrum[:, 0])
            layer.set_xlim(FreqMin, FreqMax)
            layer.xaxis.set_major_formatter(matplotlib.ticker.FormatStrFormatter('%.5e'))
            pylab.tight_layout()
            MolNameFile = task_LineIdentification.MoleculeFileName(MolName)
            pylab.savefig(ResultPath + "/myXCLASS____contributions__{:s}__comp_{:d}.png".format(MolNameFile, CompID), dpi = 300)
            pylab.draw()
            pylab.close(fig)


## finished!
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
